"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_secretsmanager_1 = require("monocdk/aws-secretsmanager");
const monocdk_1 = require("monocdk");
const nag_rules_1 = require("../../nag-rules");
/**
 * Secrets have automatic rotation scheduled
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_secretsmanager_1.CfnSecret) {
        const secretLogicalId = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.ref);
        const secretTargetAttachmentLogicalIds = Array();
        const cfnSecretTargetAttachments = Array();
        const cfnRotationSchedules = Array();
        for (const child of monocdk_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_secretsmanager_1.CfnSecretTargetAttachment) {
                cfnSecretTargetAttachments.push(child);
            }
            else if (child instanceof aws_secretsmanager_1.CfnRotationSchedule) {
                cfnRotationSchedules.push(child);
            }
        }
        if (cfnRotationSchedules.length === 0) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        let found = false;
        for (const child of cfnSecretTargetAttachments) {
            const attachmentLogicalId = getMatchingSecretTargetAttachment(child, secretLogicalId);
            if (attachmentLogicalId) {
                secretTargetAttachmentLogicalIds.push(attachmentLogicalId);
            }
        }
        for (const child of cfnRotationSchedules) {
            if (isMatchingRotationSchedule(child, secretLogicalId, secretTargetAttachmentLogicalIds)) {
                found = true;
                break;
            }
        }
        if (!found) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function to check whether a given Secret Target Attachment is associated with the given secret.
 * @param node The CfnTargetAttachment to check.
 * @param secretLogicalId The Cfn Logical ID of the secret.
 * Returns the Logical ID if the attachment if is associated with the secret, otherwise and empty string.
 */
function getMatchingSecretTargetAttachment(node, secretLogicalId) {
    const resourceSecretId = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.secretId);
    if (secretLogicalId === resourceSecretId) {
        return nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.ref);
    }
    return '';
}
/**
 * Helper function to check whether a given Rotation Schedule is associated with the given secret.
 * @param node The CfnRotationSchedule to check.
 * @param secretLogicalId The Cfn Logical ID of the secret.
 * @param secretTargetAttachmentLogicalIds The Cfn Logical IDs of any Secret Target Attachments associated with the given secret.
 * Returns whether the CfnRotationSchedule is associated with the given secret.
 */
function isMatchingRotationSchedule(node, secretLogicalId, secretTargetAttachmentLogicalIds) {
    const resourceSecretId = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.secretId);
    if (secretLogicalId === resourceSecretId ||
        secretTargetAttachmentLogicalIds.includes(resourceSecretId)) {
        if (monocdk_1.Stack.of(node).resolve(node.hostedRotationLambda) === undefined &&
            monocdk_1.Stack.of(node).resolve(node.rotationLambdaArn) === undefined) {
            return false;
        }
        const rotationRules = monocdk_1.Stack.of(node).resolve(node.rotationRules);
        if (rotationRules !== undefined) {
            const automaticallyAfterDays = monocdk_1.Stack.of(node).resolve(rotationRules.automaticallyAfterDays);
            if (automaticallyAfterDays !== undefined) {
                return true;
            }
        }
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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