"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NagSuppressionHelper = void 0;
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
class NagSuppressionHelper {
    static toCfnFormat(suppression) {
        const { appliesTo, ...result } = suppression;
        if (appliesTo) {
            result.applies_to = appliesTo;
        }
        return result;
    }
    static toApiFormat(suppression) {
        const { applies_to, ...result } = suppression;
        if (applies_to) {
            result.appliesTo = applies_to;
        }
        return result;
    }
    static addRulesToMetadata(metadata, rules) {
        const { rules_to_suppress } = metadata !== null && metadata !== void 0 ? metadata : {};
        const serialisedRules = [
            ...(rules_to_suppress !== null && rules_to_suppress !== void 0 ? rules_to_suppress : []).map((r) => JSON.stringify(r)),
            ...rules
                .map(NagSuppressionHelper.toCfnFormat)
                .map((r) => JSON.stringify(r)),
        ];
        const deduplicatedRules = Array.from(new Set(serialisedRules));
        return { rules_to_suppress: deduplicatedRules.map((r) => JSON.parse(r)) };
    }
    static getSuppressions(node) {
        var _a, _b, _c, _d, _e;
        const resourceIgnores = (_b = (_a = node.getMetadata('cdk_nag')) === null || _a === void 0 ? void 0 : _a.rules_to_suppress) !== null && _b !== void 0 ? _b : [];
        const stackIgnores = (_e = (_d = (_c = monocdk_1.Stack.of(node).templateOptions.metadata) === null || _c === void 0 ? void 0 : _c.cdk_nag) === null || _d === void 0 ? void 0 : _d.rules_to_suppress) !== null && _e !== void 0 ? _e : [];
        const result = [...resourceIgnores, ...stackIgnores].map(NagSuppressionHelper.toApiFormat);
        NagSuppressionHelper.assertSuppressionsAreValid(node.node.id, result);
        return result;
    }
    static assertSuppressionsAreValid(id, suppressions) {
        const errors = suppressions
            .map(NagSuppressionHelper.getSuppressionFormatError)
            .filter((errorMessage) => !!errorMessage);
        if (errors.length) {
            throw Error(`${id}: ${errors.join('')}\nSee https://github.com/cdklabs/cdk-nag#suppressing-a-rule for information on suppressing a rule.`);
        }
    }
    static doesApply(suppression, ruleId, findingId) {
        if (ruleId !== suppression.id) {
            return false;
        }
        if (!suppression.appliesTo) {
            // the rule is not granular so it always applies
            return true;
        }
        if (findingId &&
            suppression.appliesTo.some((appliesTo) => {
                if (typeof appliesTo === 'string') {
                    return appliesTo === findingId;
                }
                else {
                    const regex = NagSuppressionHelper.toRegEx(appliesTo.regex);
                    return regex.test(findingId);
                }
            })) {
            return true;
        }
        return false;
    }
    static getSuppressionFormatError(suppression) {
        var _a;
        let errors = '';
        const finding = suppression.id.match(/\[.*\]/);
        if (finding) {
            errors += `The suppression 'id' contains a finding '${finding}. A finding must be suppressed using 'applies_to'.`;
        }
        if (suppression.reason.length < 10) {
            errors +=
                "The suppression must have a 'reason' of 10 characters or more.";
        }
        ((_a = suppression.appliesTo) !== null && _a !== void 0 ? _a : []).forEach((appliesTo) => {
            if (typeof appliesTo !== 'string') {
                try {
                    NagSuppressionHelper.toRegEx(appliesTo.regex);
                }
                catch (err) {
                    errors += err.message;
                }
            }
        });
        return errors
            ? `\n\tError(s) detected in suppression with 'id' ${suppression.id}. ${errors}`
            : '';
    }
    static toRegEx(s) {
        try {
            // verify that the regex is correctly formatted
            const m = s.match(/\/(.*)\/(.*)?/);
            if (!m) {
                throw new Error();
            }
            return new RegExp(m[1], m[2] || '');
        }
        catch {
            throw new Error(`Invalid regular expression [${s}]`);
        }
    }
}
exports.NagSuppressionHelper = NagSuppressionHelper;
//# sourceMappingURL=data:application/json;base64,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