"""BlueLamp専門エージェント群"""

import os
from typing import List, Dict, Any, Optional
from core.agents.orchestrator_agent.orchestrator_agent import OrchestratorAgent
from core.agents.state.state import State
from core.events.action import AgentFinishAction
from core.utils.prompt import PromptManager
from extensions.portal.portal_prompt_manager import PortalPromptManager
from core.logger import openhands_logger as logger


class BlueLampBaseAgent(OrchestratorAgent):
    """BlueLamp専門エージェントの基底クラス"""

    VERSION = '1.0'

    def __init__(self, llm, config):
        """BlueLampエージェント用の初期化（system_prompt_filenameを保護）"""
        # system_prompt_filenameを一時保存
        original_filename = getattr(config, 'system_prompt_filename', None)
        
        # 親クラスの初期化を実行（これによりorchestratorに上書きされる）
        super().__init__(llm, config)
        
        # 元のsystem_prompt_filenameを復元
        if original_filename and original_filename != 'orchestrator':
            config.system_prompt_filename = original_filename
            # プロンプトマネージャーをリセットして新しいfilenameで再作成
            self._prompt_manager = None
            logger.info(f"BlueLampAgent system_prompt_filename restored to: {original_filename}")

    @property
    def prompt_manager(self) -> PromptManager:
        """BlueLampエージェント用のPortalプロンプトマネージャーを返す"""
        if self._prompt_manager is None:
            # BlueLampエージェントのプロンプトディレクトリを使用（フォールバック用）
            prompt_dir = os.path.join(os.path.dirname(__file__), 'prompts')
            
            
            # PortalPromptManagerを使用（Portal優先、ローカルフォールバック）
            self._prompt_manager = PortalPromptManager(
                prompt_dir=prompt_dir,
                system_prompt_filename=self.config.system_prompt_filename,
                enable_portal=True  # Portal連携を有効化
            )
            
            logger.info(f"PortalPromptManager initialized for {self.config.system_prompt_filename}")
        return self._prompt_manager

    def _get_tools(self) -> List[Any]:
        """委譲ツールを除外したツールリストを返す"""
        tools = super()._get_tools()
        # 委譲関連ツールを除外（delegate_で始まるツール名）
        filtered_tools = []
        tool_names = []
        
        for tool in tools:
            # ChatCompletionToolParamオブジェクトの場合
            if hasattr(tool, 'function') and hasattr(tool.function, 'name'):  # type: ignore
                tool_name = tool.function.name  # type: ignore
                tool_names.append(tool_name)
                if not tool_name.startswith('delegate_'):
                    filtered_tools.append(tool)
            # dict形式の場合
            elif isinstance(tool, dict) and 'function' in tool:
                tool_name = tool.get('function', {}).get('name', '')
                tool_names.append(tool_name)
                if not tool_name.startswith('delegate_'):
                    filtered_tools.append(tool)
            else:
                # その他のツールはそのまま含める
                filtered_tools.append(tool)
        
        # デバッグ用：委譲先エージェントが利用可能なツール一覧をログ出力
        logger.debug(f"🔧 [TOOLS DEBUG] {self.__class__.__name__} available tools: {tool_names}")
        
        # finishツールの詳細をログ出力
        for tool in filtered_tools:
            if hasattr(tool, 'function') and hasattr(tool.function, 'name'):  # type: ignore
                if tool.function.name == 'finish':  # type: ignore
                    logger.debug(f"🏁 [FINISH TOOL DEBUG] Description: {tool.function.description}")  # type: ignore
                    logger.debug(f"🏁 [FINISH TOOL DEBUG] Parameters: {tool.function.parameters}")  # type: ignore
        
        return filtered_tools


# 16個の専門エージェント定義
class RequirementsEngineer(BlueLampBaseAgent):
    """★1 要件定義エンジニア: プロジェクトの要件定義書を作成"""
    def __init__(self, llm: Any, config: Any = None):
        if config is None:
            from core.config import AgentConfig
            config = AgentConfig()
        config.system_prompt_filename = 'requirements_engineer'
        super().__init__(llm, config)


class UIUXDesigner(BlueLampBaseAgent):
    """★2 UI/UXデザイナー: ユーザーインターフェースとエクスペリエンスを設計"""
    def __init__(self, llm: Any, config: Any = None):
        if config is None:
            from core.config import AgentConfig
            config = AgentConfig()
        config.system_prompt_filename = 'uiux_designer'
        super().__init__(llm, config)


class DataModelingEngineer(BlueLampBaseAgent):
    """★3 データモデリングエンジニア: データベース設計とER図作成"""
    def __init__(self, llm: Any, config: Any = None):
        if config is None:
            from core.config import AgentConfig
            config = AgentConfig()
        config.system_prompt_filename = 'data_modeling_engineer'
        super().__init__(llm, config)


class SystemArchitect(BlueLampBaseAgent):
    """★4 システムアーキテクト: システム全体のアーキテクチャ設計"""
    def __init__(self, llm: Any, config: Any = None):
        if config is None:
            from core.config import AgentConfig
            config = AgentConfig()
        config.system_prompt_filename = 'system_architect'
        super().__init__(llm, config)


class ImplementationConsultant(BlueLampBaseAgent):
    """★5 実装計画コンサルタント: 実装戦略と技術選定"""
    def __init__(self, llm: Any, config: Any = None):
        if config is None:
            from core.config import AgentConfig
            config = AgentConfig()
        config.system_prompt_filename = 'implementation_consultant'
        super().__init__(llm, config)


class EnvironmentSetup(BlueLampBaseAgent):
    """★6 環境構築: 開発環境とインフラ構築"""
    def __init__(self, llm: Any, config: Any = None):
        if config is None:
            from core.config import AgentConfig
            config = AgentConfig()
        config.system_prompt_filename = 'environment_setup'
        super().__init__(llm, config)


class PrototypeImplementation(BlueLampBaseAgent):
    """★7 プロトタイプ実装: 迅速なプロトタイプ作成と概念実証"""
    def __init__(self, llm: Any, config: Any = None):
        if config is None:
            from core.config import AgentConfig
            config = AgentConfig()
        config.system_prompt_filename = 'prototype_implementation'
        super().__init__(llm, config)


class ImplementationAgent(BlueLampBaseAgent):
    """★8 実装エージェント: Backend実装、統合テスト、Frontend UI、API統合"""
    def __init__(self, llm: Any, config: Any = None):
        if config is None:
            from core.config import AgentConfig
            config = AgentConfig()
        config.system_prompt_filename = 'implementation_agent'
        super().__init__(llm, config)


class DebugAgent(BlueLampBaseAgent):
    """★09 デバッグエージェント v3.0: バグ修正とエラー解決"""
    def __init__(self, llm: Any, config: Any = None):
        if config is None:
            from core.config import AgentConfig
            config = AgentConfig()
        config.system_prompt_filename = 'debug_agent'
        super().__init__(llm, config)


class DeploySpecialist(BlueLampBaseAgent):
    """#10 デプロイスペシャリスト: デプロイメントとインフラ管理"""
    def __init__(self, llm: Any, config: Any = None):
        if config is None:
            from core.config import AgentConfig
            config = AgentConfig()
        config.system_prompt_filename = 'deploy_specialist'
        super().__init__(llm, config)


# ExpansionOrchestrator は ExtensionManagerAgent に統一されました
# 重複を避けるため削除


class PageCreator(BlueLampBaseAgent):
    """★12 新ページ作成エージェント: 新規ページとコンポーネント作成"""
    def __init__(self, llm: Any, config: Any = None):
        if config is None:
            from core.config import AgentConfig
            config = AgentConfig()
        config.system_prompt_filename = 'page_creator'
        super().__init__(llm, config)


class RefactoringEngineer(BlueLampBaseAgent):
    """#13 コード徹底除去専門リファクタリングエージェント v1.0: コード改善計画策定と実装"""
    def __init__(self, llm: Any, config: Any = None):
        if config is None:
            from core.config import AgentConfig
            config = AgentConfig()
        config.system_prompt_filename = 'refactoring_engineer'
        super().__init__(llm, config)


class AIFriendlinessDiagnostic(BlueLampBaseAgent):
    """#14 Universal AI-Friendliness診断プロンプト v3.0: AI親和性診断と改善提案"""
    def __init__(self, llm: Any, config: Any = None):
        if config is None:
            from core.config import AgentConfig
            config = AgentConfig()
        config.system_prompt_filename = 'ai_friendliness_diagnostic'
        super().__init__(llm, config)


