#!/usr/bin/env python
# -*-coding: utf-8 -*-

# ----------------------------------------------------------------------------------------------------------------------

# Name:		pysills_app.py
# Author:	Maximilian A. Beeskow
# Version:	v1.0.82
# Date:		24.09.2025

# ----------------------------------------------------------------------------------------------------------------------

## MODULES
# external
import os, pathlib, sys, re, datetime, csv, string, math, webbrowser, time
import numpy as np
import pandas as pd
pd.options.mode.chained_assignment = None
import scipy.io
import matplotlib as mpl
import matplotlib.pyplot as plt
from matplotlib import colors
from matplotlib.backends.backend_tkagg import (FigureCanvasTkAgg, NavigationToolbar2Tk)
from matplotlib.figure import Figure
import tkinter as tk
from tkinter import filedialog, ttk, font
import random as rd
# internal
try:
    from pysills.pypitzer.Pitzer.models import FluidPitzer
except:
    from pypitzer.Pitzer.models import FluidPitzer

try:
    from pysills.modules.chemistry import PeriodicSystem
    from pysills.modules.chemistry import PeriodicSystemOfElements as PSE
    from pysills.modules.data import Data
    from pysills.modules.data_reduction import DataExtraction as DE
    from pysills.modules.data_reduction import IntensityQuantification as IQ
    from pysills.modules.data_reduction import SensitivityQuantification as SQ
    from pysills.modules.essential_functions import Essentials as ES
    from pysills.modules.essential_functions import EssentialsSRM as ESRM
    from pysills.modules.gui_elements import SimpleElements as SE
    from pysills.modules.spike_elimination import GrubbsTestSILLS, OutlierDetection
except:
    from modules.chemistry import PeriodicSystem
    from modules.chemistry import PeriodicSystemOfElements as PSE
    from modules.data import Data
    from modules.data_reduction import DataExtraction as DE
    from modules.data_reduction import IntensityQuantification as IQ
    from modules.data_reduction import SensitivityQuantification as SQ
    from modules.essential_functions import Essentials as ES
    from modules.essential_functions import EssentialsSRM as ESRM
    from modules.gui_elements import SimpleElements as SE
    from modules.spike_elimination import GrubbsTestSILLS, OutlierDetection

# import subprocess
import string

###############
### PySILLS ###
###############
class PySILLS(tk.Frame):
    #
    def __init__(
            self, parent, var_screen_width, var_screen_height, var_path=None, restart=False, last_used_directory=None,
            last_used_directory_input=None):
        tk.Frame.__init__(self, parent)
        var_screen_width = var_screen_width
        var_screen_height = var_screen_height
        var_scaling_horizontal = round(1920/var_screen_width, 2)
        var_scaling_vertical = round(1080/var_screen_height, 2)
        if var_scaling_horizontal == 1.0 and var_scaling_vertical == 1.0:
            var_scaling = 1.3

        ## Current version
        self.str_version_number = "1.0.82"
        self.val_version = self.str_version_number + " - 24.09.2025"

        ## Colors
        self.green_dark = "#282D28"
        self.green_medium = "#616D61"
        self.green_light = "#CFD3CF"
        green_dict = {"Dark": self.green_dark, "Medium": self.green_medium, "Light": self.green_light}
        self.red_dark = "#E76F51"
        self.red_medium = "#F1A896"
        self.red_light = "#FDF0ED"
        red_dict = {"Dark": self.red_dark, "Medium": self.red_medium, "Light": self.red_light}
        self.yellow_dark = "#E9C46A"
        self.yellow_medium = "#F3DFAE"
        self.yellow_light = "#FDFAF2"
        yellow_dict = {"Dark": self.yellow_dark, "Medium": self.yellow_medium, "Light": self.yellow_light}
        self.blue_dark = "#5B828E"
        self.blue_medium = "#8CA7AF"
        self.blue_light = "#CDD9DD"
        blue_dict = {"Dark": self.blue_dark, "Medium": self.blue_medium, "Light": self.blue_light}
        self.brown_dark = "#AC7E62"
        self.brown_medium = "#C4A491"
        self.brown_light = "#EEE5DF"
        brown_dict = {"Dark": self.brown_dark, "Medium": self.brown_medium, "Light": self.brown_light}
        self.slate_grey_dark = "#6E7894"
        self.slate_grey_medium = "#9AA1B4"
        self.slate_grey_light = "#E2E4EA"
        slategrey_dict = {"Dark": self.slate_grey_dark, "Medium": self.slate_grey_medium,
                          "Light": self.slate_grey_light}

        self.bg_colors = {
            "BG Window": "#2C2C2C", "Very Dark": "#3C3C3C", "Dark": "#676767", "Medium": "#909090", "Light": "#BABABA",
            "Very Light": "#E3E3E3", "Dark Font": "#292929", "Light Font": "#F7F7F7", "White": "#FFFFFF",
            "Black": "#000000", "Accent": "#E76F51"}

        self.standard_dark = {
            "BG Window": "#2C2C2C", "Very Dark": "#3C3C3C", "Dark": "#676767", "Medium": "#909090", "Button": "#BABABA",
            "Very Light": "#E3E3E3", "Dark Font": "#2C2C2C", "Light Font": "#E3E3E3", "White": "#FFFFFF",
            "Black": "#000000", "Accent": "#E76F51"}
        self.standard_light = {
            "BG Window": "#E3E3E3", "Very Dark": "#BABABA", "Dark": "#909090", "Medium": "#676767", "Light": "#BABABA",
            "Very Light": "#E3E3E3", "Dark Font": "#E3E3E3", "Light Font": "#E3E3E3", "White": "#FFFFFF",
            "Black": "#000000", "Accent": "#E76F51"}
        self.green_dark = {
            "BG Window": "#323A25", "Very Dark": "#505D3C", "Dark": "#606F49", "Medium": "#676767", "Light": "#CDD5BF",
            "Very Light": "#FAFBF9", "Dark Font": "#323A25", "Light Font": "#FAFBF9", "White": "#FFFFFF",
            "Black": "#000000", "Accent": "#E76F51"}
        self.boho_theme = {
            "BG Window": "#72574f", "Very Dark": "#a3573a", "Dark": "#e5af9e", "Medium": "#e7b7a7", "Light": "#f2d7ce",
            "Very Light": "#f9efeb", "Dark Font": "#2d231f", "Light Font": "#f9efeb", "White": "#FFFFFF",
            "Black": "#000000", "Accent": "#B15C4D"}

        self.accent_color = self.bg_colors["Accent"]
        self.colors_ma = {"Very Dark": "#2F3E46", "Dark": "#354F52", "Medium": "#52796F", "Light": "#84A98C",
                          "Very Light": "#CAD2C5", "Dark Font": "#182320", "Light Font": "#F2F6F5"}
        self.colors_fi = {"Very Dark": "#722F1C", "Dark": "#A04228", "Medium": "#D36A4D", "Light": "#E4A694",
                          "Very Light": "#F6E1DB", "Dark Font": "#411B10", "Light Font": "#FCF4F2"}
        self.colors_mi = {"Very Dark": "#013a63", "Dark": "#014f86", "Medium": "#2c7da0", "Light": "#61a5c2",
                          "Very Light": "#a9d6e5", "Dark Font": "#0D242E", "Light Font": "#EEF7FA"}
        self.colors_intervals = {"BG": "#7F4F24", "SIG": "#414833", "MAT": "#414833", "INCL": "#F4D35E",
                                 "BG LB": "#D8CABD", "SIG LB": "#C6C8C1", "MAT LB": "#C6C8C1", "INCL LB": "#FBF1CE"}

        self.sign_red = "#E84258"
        self.sign_yellow = "#FFDE00"
        self.sign_green = "#B0D8A4"
        sign_dict = {"Green": self.sign_green, "Red": self.sign_red, "Yellow": self.sign_yellow}

        style = ttk.Style()
        style.theme_use("default")
        style.configure(
            "Vertical.TScrollbar", bg=self.bg_colors["Very Light"], bordercolor=self.bg_colors["Very Dark"],
            arrowcolor=self.bg_colors["Dark"])
        style.configure(
            "Horizontal.TScrollbar", bg=self.bg_colors["Very Light"], bordercolor=self.bg_colors["Very Dark"],
            arrowcolor=self.bg_colors["Dark"])
        ## Constants
        self.list_std = []
        self.list_smpl = []
        self.chemistry_data = {
            "O": 15.999, "Na": 22.990, "Mg": 24.305, "Al": 26.982, "Si": 28.085, "P": 30.974, "K": 39.098, "Ca": 40.078,
            "Ti": 47.867, "Cr": 51.996, "Mn": 54.938, "Fe": 55.845, "Ga": 69.723, "Ge": 72.630, "Zr": 91.224,
            "Ba": 137.33, "B": 10.81, "Ag": 107.87, "As": 74.922, "Li": 6.94, "Rb": 85.468, "Cs": 132.91, "Sr": 87.62,
            "Sc": 44.956, "Y": 88.906, "Hf": 178.49, "V": 50.942, "Nb": 92.906, "Ta": 180.95, "Mo": 95.962, "W": 183.84,
            "Tc": 98.906, "Re": 186.21, "Ru": 101.07, "Os": 190.23, "Co": 58.933, "Rh": 102.91, "Ir": 192.22,
            "Ni": 58.693, "Pd": 106.42, "Pt": 195.08, "Cu": 63.546, "Au": 196.97, "Zn": 65.38, "Cd": 112.41,
            "Hg": 200.59, "In": 114.82, "Tl": 204.38, "C": 12.011, "Sn": 118.71, "Pb": 207.2, "N": 14.007, "Sb": 121.76,
            "Bi": 208.98, "S": 32.06, "Se": 78.96, "Te": 127.60, "Po": 209.98, "Cl": 35.45, "Br": 79.904, "I": 126.90,
            "At": 210.99, "La": 138.91, "Ce": 140.12, "Pr": 140.91, "Nd": 144.24, "Pm": 146.92, "Sm": 150.36,
            "Eu": 151.96, "Gd": 157.25, "Tb": 158.93, "Dy": 162.50, "Ho": 164.93, "Er": 167.26, "Tm": 168.93,
            "Yb": 173.05, "Lu": 174.97, "Ac": 227.03, "Th": 232.04, "Pa": 231.04, "U": 238.05, "Be": 9.0122,
            "F": 18.998, "H": 1.008}
        self.chemistry_data_sills = {
            "O": 16.000, "Na": 22.990, "Mg": 24.300, "Al": 26.980, "Si": 28.090, "P": 30.970, "K": 39.100, "Ca": 40.080,
            "Ti": 47.870, "Cr": 52.000, "Mn": 54.940, "Fe": 55.850, "Ga": 69.720, "Ge": 72.610, "Zr": 91.220,
            "Ba": 137.300}
        self.chemistry_data_oxides = {
            "SiO2": 60.083, "Al2O3": 101.961, "Fe2O3": 159.687, "FeO": 71.844, "Na2O": 61.979, "TiO2": 79.865,
            "MnO": 70.937, "Mn2O3": 157.873, "SnO": 134.709, "Li2O": 29.879, "Ga2O3": 187.443, "B2O3": 69.617,
            "BeO": 25.0112, "GeO2": 104.628, "CaO": 56.077, "Rb2O": 186.935, "AgO": 123.869, "As2O3": 197.841,
            "Au2O": 409.939, "BaO": 153.32, "Br2O": 175.807, "Cl2O": 86.899, "Cs2O": 281.819, "CuO": 79.545,
            "PbO": 223.199, "SO3": 80.057, "Sb2O3": 291.517, "SrO": 103.619, "WO3": 231.837, "ZnO": 81.379,
            "MgO": 40.304, "K2O": 55.097, "SnO2": 150.708, "Ag2O": 231.739, "Bi2O5": 497.955, "CO2": 44.009,
            "CdO": 128.409, "Ce2O3": 328.237, "CeO2": 172.118, "CoO": 74.932, "Cr2O3": 151.989, "Dy2O3": 372.997,
            "Er2O3": 382.517, "Eu2O3": 351.917, "Gd2O3": 362.497, "HfO2": 404.977, "HgO": 216.589, "Ho2O3": 377.857,
            "In2O3": 277.637, "IrO": 208.219, "La2O3": 325.817, "Lu2O3": 397.937, "MnO2": 86.936, "MoO3": 143.959,
            "N2O5": 108.009, "Nb2O5": 265.807, "Nd2O3": 336.477, "NiO": 74.692, "OsO": 206.229, "P2O5": 141.943,
            "PbO2": 239.198, "PdO": 122.419, "Pr2O3": 329.817, "Pr6O11": 1021.449, "PtO": 211.079, "ReO": 202.209,
            "RhO": 118.909, "RuO": 117.069, "SO4": 96.056, "Sb2O5": 323.515, "Sc2O3": 137.909, "SeO3": 126.957,
            "Sm2O3": 348.717, "Ta2O5": 441.895, "Tb2O3": 365.857, "Tb4O7": 747.713, "TeO3": 175.597, "ThO2": 264.038,
            "Tl2O3": 456.757, "Tm2O3": 385.857, "UO2": 270.048, "UO3": 286.047, "U3O8": 842.142, "V2O5": 181.879,
            "Y2O3": 225.809, "Yb2O3": 394.097, "ZrO2": 123.222, "I2O4": 317.796, "I2O5": 333.795, "I4O9": 651.591,
            "I2O": 269.799, "Ni2O3": 165.383, "Co2O3": 165.863, "CrO": 67.995}

        self.conversion_factors = {
            "SiO2": round((self.chemistry_data["Si"]/self.chemistry_data_oxides["SiO2"])**(-1), 4),
            "Al2O3": round((2*self.chemistry_data["Al"]/self.chemistry_data_oxides["Al2O3"])**(-1), 4),
            "Fe2O3": round((2*self.chemistry_data["Fe"]/self.chemistry_data_oxides["Fe2O3"])**(-1), 4),
            "FeO": round((self.chemistry_data["Fe"]/self.chemistry_data_oxides["FeO"])**(-1), 4),
            "Na2O": round((2*self.chemistry_data["Na"]/self.chemistry_data_oxides["Na2O"])**(-1), 4),
            "TiO2": round((self.chemistry_data["Ti"]/self.chemistry_data_oxides["TiO2"])**(-1), 4),
            "MnO": round((self.chemistry_data["Mn"]/self.chemistry_data_oxides["MnO"])**(-1), 4),
            "Mn2O3": round((2*self.chemistry_data["Mn"]/self.chemistry_data_oxides["Mn2O3"])**(-1), 4),
            "SnO": round((self.chemistry_data["Sn"]/self.chemistry_data_oxides["SnO"])**(-1), 4),
            "Li2O": round((2*self.chemistry_data["Li"]/self.chemistry_data_oxides["Li2O"])**(-1), 4),
            "Ga2O3": round((2*self.chemistry_data["Ga"]/self.chemistry_data_oxides["Ga2O3"])**(-1), 4),
            "B2O3": round((2*self.chemistry_data["B"]/self.chemistry_data_oxides["B2O3"])**(-1), 4),
            "BeO": round((self.chemistry_data["Be"]/self.chemistry_data_oxides["BeO"])**(-1), 4),
            "GeO2": round((self.chemistry_data["Ge"]/self.chemistry_data_oxides["GeO2"])**(-1), 4),
            "CaO": round((self.chemistry_data["Ca"]/self.chemistry_data_oxides["CaO"])**(-1), 4),
            "Rb2O": round((2*self.chemistry_data["Rb"]/self.chemistry_data_oxides["Rb2O"])**(-1), 4),
            "AgO": round((self.chemistry_data["Ag"]/self.chemistry_data_oxides["AgO"])**(-1), 4),
            "As2O3": round((2*self.chemistry_data["As"]/self.chemistry_data_oxides["As2O3"])**(-1), 4),
            "Au2O": round((2*self.chemistry_data["Au"]/self.chemistry_data_oxides["Au2O"])**(-1), 4),
            "BaO": round((self.chemistry_data["Ba"]/self.chemistry_data_oxides["BaO"])**(-1), 4),
            "Br2O": round((2*self.chemistry_data["Br"]/self.chemistry_data_oxides["Br2O"])**(-1), 4),
            "Cl2O": round((2*self.chemistry_data["Cl"]/self.chemistry_data_oxides["Cl2O"])**(-1), 4),
            "Cs2O": round((2*self.chemistry_data["Cs"]/self.chemistry_data_oxides["Cs2O"])**(-1), 4),
            "CuO": round((self.chemistry_data["Cu"]/self.chemistry_data_oxides["CuO"])**(-1), 4),
            "PbO": round((self.chemistry_data["Pb"]/self.chemistry_data_oxides["PbO"])**(-1), 4),
            "SO3": round((self.chemistry_data["S"]/self.chemistry_data_oxides["SO3"])**(-1), 4),
            "Sb2O3": round((2*self.chemistry_data["Sb"]/self.chemistry_data_oxides["Sb2O3"])**(-1), 4),
            "SrO": round((self.chemistry_data["Sr"]/self.chemistry_data_oxides["SrO"])**(-1), 4),
            "WO3": round((self.chemistry_data["W"]/self.chemistry_data_oxides["WO3"])**(-1), 4),
            "ZnO": round((self.chemistry_data["Zn"]/self.chemistry_data_oxides["ZnO"])**(-1), 4),
            "MgO": round((self.chemistry_data["Mg"]/self.chemistry_data_oxides["MgO"])**(-1), 4),
            "K2O": round((2*self.chemistry_data["K"]/self.chemistry_data_oxides["K2O"])**(-1), 4),
            "SnO2": round((self.chemistry_data["Sn"]/self.chemistry_data_oxides["SnO2"])**(-1), 4),
            "Ag2O": round((2*self.chemistry_data["Ag"]/self.chemistry_data_oxides["Ag2O"])**(-1), 4),
            "Bi2O5": round((2*self.chemistry_data["Bi"]/self.chemistry_data_oxides["Bi2O5"])**(-1), 4),
            "CO2": round((self.chemistry_data["C"]/self.chemistry_data_oxides["CO2"])**(-1), 4),
            "CdO": round((self.chemistry_data["Cd"]/self.chemistry_data_oxides["CdO"])**(-1), 4),
            "Ce2O3": round((2*self.chemistry_data["Ce"]/self.chemistry_data_oxides["Ce2O3"])**(-1), 4),
            "CeO2": round((self.chemistry_data["Ce"]/self.chemistry_data_oxides["CeO2"])**(-1), 4),
            "CoO": round((self.chemistry_data["Co"]/self.chemistry_data_oxides["CoO"])**(-1), 4),
            "CrO": round((self.chemistry_data["Cr"]/self.chemistry_data_oxides["CrO"])**(-1), 4),
            "Cr2O3": round((2*self.chemistry_data["Cr"]/self.chemistry_data_oxides["Cr2O3"])**(-1), 4),
            "Dy2O3": round((2*self.chemistry_data["Dy"]/self.chemistry_data_oxides["Dy2O3"])**(-1), 4),
            "Er2O3": round((2*self.chemistry_data["Er"]/self.chemistry_data_oxides["Er2O3"])**(-1), 4),
            "Eu2O3": round((2*self.chemistry_data["Eu"]/self.chemistry_data_oxides["Eu2O3"])**(-1), 4),
            "Gd2O3": round((2*self.chemistry_data["Gd"]/self.chemistry_data_oxides["Gd2O3"])**(-1), 4),
            "HfO2": round((self.chemistry_data["Hf"]/self.chemistry_data_oxides["HfO2"])**(-1), 4),
            "HgO": round((self.chemistry_data["Hg"]/self.chemistry_data_oxides["HgO"])**(-1), 4),
            "Ho2O3": round((2*self.chemistry_data["Ho"]/self.chemistry_data_oxides["Ho2O3"])**(-1), 4),
            "In2O3": round((2*self.chemistry_data["In"]/self.chemistry_data_oxides["In2O3"])**(-1), 4),
            "IrO": round((self.chemistry_data["Ir"]/self.chemistry_data_oxides["IrO"])**(-1), 4),
            "La2O3": round((2*self.chemistry_data["La"]/self.chemistry_data_oxides["La2O3"])**(-1), 4),
            "Lu2O3": round((2*self.chemistry_data["Lu"]/self.chemistry_data_oxides["Lu2O3"])**(-1), 4),
            "MnO2": round((self.chemistry_data["Mn"]/self.chemistry_data_oxides["MnO2"])**(-1), 4),
            "MoO3": round((self.chemistry_data["Mo"]/self.chemistry_data_oxides["MoO3"])**(-1), 4),
            "N2O5": round((2*self.chemistry_data["N"]/self.chemistry_data_oxides["N2O5"])**(-1), 4),
            "Nb2O5": round((2*self.chemistry_data["Nb"]/self.chemistry_data_oxides["Nb2O5"])**(-1), 4),
            "Nd2O3": round((2*self.chemistry_data["Nd"]/self.chemistry_data_oxides["Nd2O3"])**(-1), 4),
            "NiO": round((self.chemistry_data["Ni"]/self.chemistry_data_oxides["NiO"])**(-1), 4),
            "OsO": round((self.chemistry_data["Os"]/self.chemistry_data_oxides["OsO"])**(-1), 4),
            "P2O5": round((2*self.chemistry_data["P"]/self.chemistry_data_oxides["P2O5"])**(-1), 4),
            "PbO2": round((self.chemistry_data["Pb"]/self.chemistry_data_oxides["PbO2"])**(-1), 4),
            "PdO": round((self.chemistry_data["Pd"]/self.chemistry_data_oxides["PdO"])**(-1), 4),
            "Pr2O3": round((2*self.chemistry_data["Pr"]/self.chemistry_data_oxides["Pr2O3"])**(-1), 4),
            "Pr6O11": round((6*self.chemistry_data["Pr"]/self.chemistry_data_oxides["Pr6O11"])**(-1), 4),
            "PtO": round((self.chemistry_data["Pt"]/self.chemistry_data_oxides["PtO"])**(-1), 4),
            "ReO": round((self.chemistry_data["Re"]/self.chemistry_data_oxides["ReO"])**(-1), 4),
            "RhO": round((self.chemistry_data["Rh"]/self.chemistry_data_oxides["RhO"])**(-1), 4),
            "RuO": round((self.chemistry_data["Ru"]/self.chemistry_data_oxides["RuO"])**(-1), 4),
            "SO4": round((self.chemistry_data["S"]/self.chemistry_data_oxides["SO4"])**(-1), 4),
            "Sb2O5": round((2*self.chemistry_data["Sb"]/self.chemistry_data_oxides["Sb2O5"])**(-1), 4),
            "Sc2O3": round((2*self.chemistry_data["Sc"]/self.chemistry_data_oxides["Sc2O3"])**(-1), 4),
            "SeO3": round((self.chemistry_data["Se"]/self.chemistry_data_oxides["SeO3"])**(-1), 4),
            "Sm2O3": round((2*self.chemistry_data["Sm"]/self.chemistry_data_oxides["Sm2O3"])**(-1), 4),
            "Ta2O5": round((2*self.chemistry_data["Ta"]/self.chemistry_data_oxides["Ta2O5"])**(-1), 4),
            "Tb2O3": round((2*self.chemistry_data["Tb"]/self.chemistry_data_oxides["Tb2O3"])**(-1), 4),
            "Tb4O7": round((4*self.chemistry_data["Tb"]/self.chemistry_data_oxides["Tb4O7"])**(-1), 4),
            "TeO3": round((self.chemistry_data["Te"]/self.chemistry_data_oxides["TeO3"])**(-1), 4),
            "ThO2": round((self.chemistry_data["Th"]/self.chemistry_data_oxides["ThO2"])**(-1), 4),
            "Tl2O3": round((2*self.chemistry_data["Tl"]/self.chemistry_data_oxides["Tl2O3"])**(-1), 4),
            "Tm2O3": round((2*self.chemistry_data["Tm"]/self.chemistry_data_oxides["Tm2O3"])**(-1), 4),
            "UO2": round((self.chemistry_data["U"]/self.chemistry_data_oxides["UO2"])**(-1), 4),
            "UO3": round((self.chemistry_data["U"]/self.chemistry_data_oxides["UO3"])**(-1), 4),
            "U3O8": round((3*self.chemistry_data["U"]/self.chemistry_data_oxides["U3O8"])**(-1), 4),
            "V2O5": round((2*self.chemistry_data["V"]/self.chemistry_data_oxides["V2O5"])**(-1), 4),
            "Y2O3": round((2*self.chemistry_data["Y"]/self.chemistry_data_oxides["Y2O3"])**(-1), 4),
            "Yb2O3": round((2*self.chemistry_data["Yb"]/self.chemistry_data_oxides["Yb2O3"])**(-1), 4),
            "ZrO2": round((self.chemistry_data["Zr"]/self.chemistry_data_oxides["ZrO2"])**(-1), 4),
            "I2O4": round((2*self.chemistry_data["I"]/self.chemistry_data_oxides["I2O4"])**(-1), 4),
            "I2O5": round((2*self.chemistry_data["I"]/self.chemistry_data_oxides["I2O5"])**(-1), 4),
            "I4O9": round((4*self.chemistry_data["I"]/self.chemistry_data_oxides["I4O9"])**(-1), 4),
            "I2O": round((2*self.chemistry_data["I"]/self.chemistry_data_oxides["I2O"])**(-1), 4),
            "Co2O3": round((2*self.chemistry_data["Co"]/self.chemistry_data_oxides["Co2O3"])**(-1), 4),
            "Ni2O3": round((2*self.chemistry_data["Ni"]/self.chemistry_data_oxides["Ni2O3"])**(-1), 4)}

        self.chemistry_oxides_sorted = {
            "H": ["H2O"], "Li": ["Li2O"], "Be": ["BeO"], "B": ["B2O3"], "C": ["CO", "CO2"],
            "N": ["NO", "N2O3", "NO2", "N2O5"], "Na": ["Na2O"], "Mg": ["MgO"], "Al": ["Al2O3"], "Si": ["SiO2"],
            "P": ["P2O3", "P2O5"], "S": ["SO3"], "Cl": ["Cl2O"],
            "K": ["K2O"], "Ca": ["CaO"], "Sc": ["Sc2O3"], "Ti": ["TiO2"], "V": ["V2O5"],
            "Cr": ["CrO", "Cr2O3", "CrO3"], "Mn": ["MnO", "Mn2O3"],
            "Fe": ["FeO", "Fe2O3"], "Co": ["Co2O3"], "Ni": ["NiO", "Ni2O3"], "Cu": ["Cu2O", "CuO"],
            "Zn": ["ZnO"], "Ga": ["Ga2O3"], "Ge": ["GeO2"], "As": ["As2O3", "As2O5"], "Se": ["SeO2", "SiO3"],
            "Br": ["Br2O"], "Kr": ["KrO"], "Rb": ["Rb2O"], "Sr": ["SrO"], "Y": ["Y2O3"],
            "Zr": ["ZrO2"], "Nb": ["Nb2O3", "Nb2O5"], "Mo": ["MoO", "Mo2O3", "MoO2", "Mo2O5", "MoO3"], "Tc": ["Tc2O7"],
            "Ru": ["RuO", "Ru2O3", "RuO2", "RuO3", "RuO4"], "Rh": ["Rh2O", "RhO", "Rh2O3", "RhO2", "Rh2O5"],
            "Pd": ["PdO", "PdO2"], "Ag": ["Ag2O", "AgO"], "Cd": ["CdO"], "In": ["In2O3"], "Sn": ["SnO", "SnO2"],
            "Sb": ["Sb2O3", "Sb2O5"], "Te": ["TeO2", "TeO3"], "I": ["I2O"],
            "Xe": ["XeO", "XeO2", "XeO3"], "Cs": ["Cs2O"], "Ba": ["BaO"], "La": ["La2O3"], "Ce": ["Ce2O3", "CeO2"],
            "Pr": ["Pr2O3", "PrO2"], "Nd": ["Nd2O3"], "Pm": ["Pm2O3"], "Sm": ["SmO", "Sm2O3"], "Eu": ["EuO", "Eu2O3"],
            "Gd": ["Gd2O3"], "Tb": ["Tb2O3", "TbO2"], "Dy": ["Dy2O3"], "Ho": ["Ho2O3"], "Er": ["Er2O3"],
            "Tm": ["TmO", "Tm2O3"], "Yb": ["YbO", "Yb2O3"], "Lu": ["Lu2O3"], "Hf": ["HfO2"], "Ta": ["Ta2O5"],
            "W": ["WO", "WO2O3", "WO2", "W2O5", "WO3"], "Re": ["ReO", "ReO2", "ReO3", "Re2O7"],
            "Os": ["OsO", "Os2O3", "OsO2", "OsO3", "OsO4"], "Ir": ["Ir2O", "IrO", "Ir2O3", "IrO2", "IrO3"],
            "Pt": ["PtO", "PtO2"], "Au": ["Au2O", "Au2O3"], "Hg": ["Hg2O", "HgO"], "Tl": ["Tl2O", "Tl2O3"],
            "Pb": ["PbO", "PbO2"], "Bi": ["Bi2O3", "B2O5"], "Po": ["PoO", "PoO2", "PoO3"],
            "At": ["At2O", "At2O3", "At2O5", "At2O7"], "Rn": ["RnO"], "Fr": ["Fr2O"], "Ra": ["RaO"], "Ac": ["Ac2O3"],
            "Th": ["ThO2"], "Pa": ["PaO2", "Pa2O5"], "U": ["U2O3", "UO2", "U2O5", "UO3"],
            "Np": ["Np2O3", "NpO2", "Np2O5", "NpO3"], "Pu": ["Pu2O3", "PuO2", "Pu2O5", "PuO3"],
            "Am": ["Am2O3", "AmO2", "Am2O5", "AmO3"], "Cm": ["Cm2O3", "CmO2"], "Bk": ["Bk2O3", "BkO2"],
            "Cf": ["Cf2O3", "CfO2"], "Es": ["Es2O3"], "Fm": ["Fm2O3"], "Md": ["Md2O3"], "No": ["NoO", "No2O3"],
            "Lr": ["Lr2O3"]}

        # General Settings
        self.parent = parent
        # self.parent.tk.call("tk", "scaling", var_scaling)
        self.parent.title("PySILLS - LA-ICP-MS data reduction")
        var_geometry = ""
        var_window_width = int(440)
        var_geometry += str(var_window_width)
        var_geometry += "x"
        var_window_height = int(950)
        var_geometry += str(var_window_height)
        var_geometry += "+0+0"
        self.parent.geometry(var_geometry)
        self.parent.resizable(False, False)

        self.list_alphabet = list(string.ascii_uppercase)

        var_os = sys.platform
        self.var_os = var_os
        # ['GTK3Agg', 'GTK3Cairo', 'MacOSX', 'nbAgg', 'Qt4Agg', 'Qt4Cairo', 'Qt5Agg', 'Qt5Cairo', 'TkAgg', 'TkCairo',
        # 'WebAgg', 'WX', 'WXAgg', 'WXCairo', 'agg', 'cairo', 'pdf', 'pgf', 'ps', 'svg', 'template']

        if var_os == "linux":
            self.defaultFont = font.nametofont("TkDefaultFont")
            default_font = font.nametofont("TkDefaultFont")
            default_font.configure(family="Ubuntu", size=10, weight=font.NORMAL)
            self.parent.option_add("*Font", default_font)
            mpl.use("TkAgg")
            self.fontsize_normal = 10
        elif var_os == "darwin":
            self.defaultFont = font.nametofont("TkDefaultFont")
            default_font = font.nametofont("TkDefaultFont")
            self.parent.option_add("*Font", default_font)
            mpl.use("MacOSX")
            self.fontsize_normal = 10
        else:
            self.defaultFont = font.nametofont("TkDefaultFont")
            default_font = font.nametofont("TkDefaultFont")
            self.parent.option_add("*Font", default_font)
            mpl.use("TkAgg")
            self.fontsize_normal = 10

        ## Data Container
        self.container_elements = {}
        self.gui_elements = {}
        self.container_gui = {}
        self.window_created = {}
        menu_list = ["main", "SRM", "ma_setting", "ma_datareduction", "ma_dataexploration", "ma_datareduction_files",
                     "fi_setting", "fi_datareduction_isotopes", "fi_datareduction_files", "fi_datareduction",
                     "fi_dataexploration", "mi_setting", "mi_datareduction",
                     "mi_dataexploration", "plotting", "PSE", "salt_correction", "fi_method_setting",
                     "mi_method_setting", "se_method_setting", "dwell_times", "general_settings"]
        gui_categories = ["Label", "Button", "Option Menu", "Entry", "Frame", "Radiobutton", "Checkbox", "Listbox",
                          "Canvas", "Treeview"]
        for menu in menu_list:
            self.container_elements[menu] = {}
            self.container_elements[menu]["Label"] = []
            self.container_elements[menu]["Button"] = []
            self.container_elements[menu]["Option Menu"] = []
            self.container_elements[menu]["Entry"] = []
            self.container_elements[menu]["Frame"] = []
            self.container_elements[menu]["Radiobutton"] = []
            self.container_elements[menu]["Checkbox"] = []
            self.container_elements[menu]["Listbox"] = []
            self.container_elements[menu]["Canvas"] = []
            self.container_elements[menu]["Treeview"] = []
            self.container_gui[menu] = {}
            self.window_created[menu] = False

            self.gui_elements[menu] = {}
            for gui_category in gui_categories:
                self.gui_elements[menu][gui_category] = {}
                self.gui_elements[menu][gui_category]["General"] = []
                self.gui_elements[menu][gui_category]["Specific"] = []
                self.container_gui[menu][gui_category] = {}
                self.container_gui[menu][gui_category]["General"] = []
                self.container_gui[menu][gui_category]["Specific"] = []

        ## SUBWINDOWS
        self.gui_subwindows = {}
        main_categories = ["Mineral Analysis", "Fluid Inclusions", "Melt Inclusions"]
        sub_categories = ["Check SRM", "Check IS", "Check Intervals", "Check Files"]
        gui_elements = ["Frame", "Label", "Button", "Entry", "Checkbox", "Radiobutton", "Option Menu", "Listbox",
                        "Treeview"]
        priorities = ["Permanent", "Temporary"]

        for main_category in main_categories:
            self.gui_subwindows[main_category] = {}
            for sub_category in sub_categories:
                self.gui_subwindows[main_category][sub_category] = {}
                for gui_element in gui_elements:
                    self.gui_subwindows[main_category][sub_category][gui_element] = {}
                    for priority in priorities:
                        self.gui_subwindows[main_category][sub_category][gui_element][priority] = []

        ## Container (Variables)
        categories = ["main"]
        subcategories = ["Label", "Entry", "Radiobutton", "Checkbox"]

        self.helper_option_menus = {}
        self.container_listbox_files = {"STD": None, "SMPL": None, "STD Manager": None, "SMPL Manager": None}

        self.container_var = {}
        for category in categories:
            self.container_var[category] = {}
            for subcategory in subcategories:
                self.container_var[category][subcategory] = []
        self.container_var["SRM"] = {"default": [tk.StringVar(), tk.StringVar()]}
        for variable in self.container_var["SRM"]["default"]:
            variable.set("Select SRM")
        self.container_var["isotopes"] = {"default": tk.StringVar()}
        self.container_var["acquisition times"] = {"STD": {}, "SMPL": {}}
        self.container_var["mineral"] = tk.StringVar()
        self.container_var["mineral"].set("Select Mineral")
        self.container_var["srm_window"] = {}
        self.container_var["srm_window"]["Option Menu"] = {
            "SRM": tk.StringVar(), "Mineral": tk.StringVar(), "Entry": {}}
        self.container_var["ma_setting"] = {}
        self.container_var["ma_datareduction_isotopes"] = {}
        self.container_var["ma_datareduction_files"] = {}
        self.container_var["fi_setting"] = []
        self.container_var["mi_setting"] = []
        self.container_var["fi_setting"] = {}
        self.container_var["fi_datareduction_isotopes"] = {}
        self.container_var["fi_datareduction_files"] = {}
        self.container_var["mi_setting"] = {}
        self.container_var["salt_correction"] = {}
        self.container_var["salt_correction"]["Checkboxes"] = {}
        self.container_var["salt_correction"]["Salinity"] = tk.StringVar()
        self.container_var["salt_correction"]["Salinity"].set("Set salinity")
        self.container_var["salt_correction"]["Concentration"] = {}
        self.container_var["dwell_times"] = {}
        self.container_var["dwell_times"]["Entry"] = {}
        self.container_var["dwell_times"]["Entry"]["Default"] = tk.StringVar()
        self.container_var["dwell_times"]["Entry"]["Default"].set("0.002")
        self.container_var["IS STD Default"] = tk.StringVar()
        self.container_var["IS STD Default"].set("0.0")
        self.container_var["IS SMPL Default"] = tk.StringVar()
        self.container_var["IS SMPL Default"].set("0.0")
        self.container_var["x-y diagram"] = {"x": tk.StringVar(), "y": tk.StringVar(), "z": tk.StringVar()}
        self.container_var["x-y diagram"]["x"].set("Select x")
        self.container_var["x-y diagram"]["y"].set("Select y")
        self.container_var["x-y diagram"]["z"].set("Select z")
        self.container_var["a(TiO2,Rt)"] = tk.StringVar()
        self.container_var["a(TiO2,Rt)"].set("1.0")
        self.container_var["stepwise focus"] = tk.IntVar()
        self.container_var["stepwise focus"].set(1)
        self.container_var["stepwise parameter"] = tk.IntVar()
        self.container_var["stepwise parameter"].set(0)
        self.container_var["stepwise visualization"] = tk.IntVar()
        self.container_var["stepwise visualization"].set(0)
        self.container_var["check INCL"] = tk.BooleanVar()
        self.container_var["check INCL"].set(False)
        self.container_var["Python path"] = tk.StringVar()
        self.container_var["Python path"].set("Set python path")
        self.container_var["PySILLS path"] = tk.StringVar()
        self.container_var["PySILLS path"].set("Set PySILLS path")
        self.container_var["PySILLS script"] = tk.StringVar()
        self.container_var["PySILLS script"].set("Set PySILLS path")

        self.copied_file = False
        self.helper_salt_composition = {}
        self.charge_balance_check = {}
        self.counter_calculation_runs = 0
        self.molality_based_quantification = tk.BooleanVar()
        self.molality_based_quantification.set(False)
        # Quantification Method
        self.container_var["Quantification Mineral"] = {"Method": tk.StringVar()}
        self.container_var["Quantification Mineral"]["Method"].set("Internal Standard")

        # 100 wt.% Oxides
        self.container_var["Oxides Quantification"] = {"Major": {}, "Minor": {}, "Ratios": {}, "Total Amounts": {}}
        self.container_var["Oxides Quantification INCL"] = {"Major": {}, "Minor": {}, "Ratios": {}, "Total Amounts": {}}
        list_major_oxides = [
            "SiO2", "TiO2", "Al2O3", "FeO", "Fe2O3", "MnO", "Mn2O3", "MgO", "CaO", "Na2O", "K2O", "P2O5", "SO3"]
        list_industrial_metal_oxides = [
            "CrO", "Cr2O3", "NiO", "Ni2O3", "ZnO", "CuO", "PbO", "PbO2", "SnO2", "WO3", "MoO2", "MoO3", "V2O5", "ZrO2",
            "Nb2O5", "HfO2", "Ta2O5", "Co2O3", "ThO2", "UO2"]
        list_precious_metals = ["AgO", "PdO", "PtO", "Au2O", "OsO", "RuO", "IrO", "RhO"]
        list_rareearth_metals = [
            "Ce2O3", "Nd2O3", "La2O3", "Y2O3", "Sc2O3", "Pr2O3", "Pr6O11", "Sm2O3", "Gd2O3", "Dy2O3", "Er2O3", "Yb2O3",
            "Eu2O3", "Ho2O3", "Tb2O3", "Tb4O7", "Lu2O3", "Tm2O3"]
        list_other_elements = [
            "Li2O", "Ga2O3", "B2O3", "BeO", "GeO2", "As2O3", "Sb2O3", "BaO", "SrO", "Cl2O", "Br2O", "I2O", "Rb2O",
            "Cs2O", "Fr2O",]
        list_oxideratios = ["Fe-Ratio", "Mn-Ratio", "Pb-Ratio", "Pr-Ratio", "Tb-Ratio"]
        for oxide in list_major_oxides:
            self.container_var["Oxides Quantification"]["Major"][oxide] = tk.IntVar()
            self.container_var["Oxides Quantification"]["Major"][oxide].set(0)
            self.container_var["Oxides Quantification INCL"]["Major"][oxide] = tk.IntVar()
            self.container_var["Oxides Quantification INCL"]["Major"][oxide].set(0)
        for oxide in list_industrial_metal_oxides:
            self.container_var["Oxides Quantification"]["Minor"][oxide] = tk.IntVar()
            self.container_var["Oxides Quantification"]["Minor"][oxide].set(0)
            self.container_var["Oxides Quantification INCL"]["Minor"][oxide] = tk.IntVar()
            self.container_var["Oxides Quantification INCL"]["Minor"][oxide].set(0)
        for oxide in list_precious_metals:
            self.container_var["Oxides Quantification"]["Minor"][oxide] = tk.IntVar()
            self.container_var["Oxides Quantification"]["Minor"][oxide].set(0)
            self.container_var["Oxides Quantification INCL"]["Minor"][oxide] = tk.IntVar()
            self.container_var["Oxides Quantification INCL"]["Minor"][oxide].set(0)
        for oxide in list_rareearth_metals:
            self.container_var["Oxides Quantification"]["Minor"][oxide] = tk.IntVar()
            self.container_var["Oxides Quantification"]["Minor"][oxide].set(0)
            self.container_var["Oxides Quantification INCL"]["Minor"][oxide] = tk.IntVar()
            self.container_var["Oxides Quantification INCL"]["Minor"][oxide].set(0)
        for oxide in list_other_elements:
            self.container_var["Oxides Quantification"]["Minor"][oxide] = tk.IntVar()
            self.container_var["Oxides Quantification"]["Minor"][oxide].set(0)
            self.container_var["Oxides Quantification INCL"]["Minor"][oxide] = tk.IntVar()
            self.container_var["Oxides Quantification INCL"]["Minor"][oxide].set(0)
        for ratio in list_oxideratios:
            if "Fe" in ratio:
                value_default = "0.0"
            else:
                value_default = "1.0"

            self.container_var["Oxides Quantification"]["Ratios"][ratio] = tk.StringVar()
            self.container_var["Oxides Quantification"]["Ratios"][ratio].set(value_default)
            self.container_var["Oxides Quantification INCL"]["Ratios"][ratio] = tk.StringVar()
            self.container_var["Oxides Quantification INCL"]["Ratios"][ratio].set(value_default)

        # Detailed Data Analysis
        self.container_var["Detailed Data Analysis"] = {
            "Filename STD": tk.StringVar(), "Filename SMPL": tk.StringVar(), "Datatype": tk.IntVar(),
            "Focus": tk.IntVar(), "Intensity Results": tk.StringVar(), "Sensitivity Results": tk.StringVar(),
            "Concentration Results": tk.StringVar()}
        self.container_var["Detailed Data Analysis"]["Filename STD"].set("Select Standard File")
        self.container_var["Detailed Data Analysis"]["Filename SMPL"].set("Select Sample File")
        self.container_var["Detailed Data Analysis"]["Datatype"].set(0)
        self.container_var["Detailed Data Analysis"]["Focus"].set(1)
        self.container_var["Detailed Data Analysis"]["Intensity Results"].set("Select Parameter")
        self.container_var["Detailed Data Analysis"]["Sensitivity Results"].set("Select Parameter")
        self.container_var["Detailed Data Analysis"]["Concentration Results"].set("Select Parameter")

        self.container_var["General Settings"] = {
            "Language": tk.StringVar(), "Default Author": tk.StringVar(), "Default SRM": tk.StringVar(),
            "Colormap": tk.StringVar(), "Line width": tk.StringVar(), "File type": tk.StringVar(),
            "Delimiter": tk.StringVar(), "Default IS MA": tk.StringVar(), "Default IS FI": tk.StringVar(),
            "Default IS MI": tk.StringVar(), "Accuracy Intensity": tk.IntVar(), "Accuracy Sensitivity": tk.IntVar(),
            "Accuracy Concentration": tk.IntVar(), "Sensitivity Drift": tk.IntVar(), "LOD Selection": tk.IntVar(),
            "Desired Average": tk.IntVar(), "Interval Processing": tk.IntVar(), "BG Offset Start": tk.IntVar(),
            "BG Offset End": tk.IntVar(), "MAT Offset Start": tk.IntVar(), "MAT Offset End": tk.IntVar(),
            "Calculation Accuracy": tk.IntVar(), "Color scheme": tk.StringVar(), "Screen resolution": tk.StringVar(),
            "Project type": tk.StringVar()}
        self.container_var["General Settings"]["Screen resolution"].set("1920x1080")
        self.container_var["General Settings"]["Language"].set("English")
        self.container_var["General Settings"]["Color scheme"].set("Dark scheme")
        self.container_var["General Settings"]["Default Author"].set("J. Doe")
        self.container_var["General Settings"]["Default SRM"].set("Select SRM")
        self.container_var["General Settings"]["Colormap"].set("turbo")
        self.container_var["General Settings"]["Line width"].set("1.0")
        self.container_var["General Settings"]["File type"].set("*.csv")
        self.container_var["General Settings"]["Project type"].set("*.proj")
        self.container_var["General Settings"]["Delimiter"].set("semicolon")
        self.container_var["General Settings"]["Default IS MA"].set("Select IS")
        self.container_var["General Settings"]["Default IS FI"].set("Select IS")
        self.container_var["General Settings"]["Default IS MI"].set("Select IS")
        self.container_var["General Settings"]["Accuracy Intensity"].set(6)
        self.container_var["General Settings"]["Accuracy Sensitivity"].set(4)
        self.container_var["General Settings"]["Accuracy Concentration"].set(4)
        self.container_var["General Settings"]["Sensitivity Drift"].set(0)
        self.container_var["General Settings"]["LOD Selection"].set(0)
        self.container_var["General Settings"]["Desired Average"].set(1)
        self.container_var["General Settings"]["Interval Processing"].set(1)
        self.container_var["General Settings"]["BG Offset Start"].set(15)
        self.container_var["General Settings"]["BG Offset End"].set(15)
        self.container_var["General Settings"]["MAT Offset Start"].set(15)
        self.container_var["General Settings"]["MAT Offset End"].set(15)
        self.container_var["General Settings"]["Calculation Accuracy"].set(1)

        self.container_var["Plotting"] = {"MA": {"Quickview": {}, "Time-Signal": {}, "Time-Ratio": {}},
                                          "FI": {"Quickview": {}, "Time-Signal": {}, "Time-Ratio": {}},
                                          "MI": {"Quickview": {}, "Time-Signal": {}, "Time-Ratio": {}},
                                          "INCL": {"Quickview": {}, "Time-Signal": {}, "Time-Ratio": {}}}
        self.container_var["Subwindows"] = {"MA": {}, "FI": {}, "MI": {}, "INCL": {}}

        self.container_var["Spike Elimination Check"] = {"RB Value STD": tk.IntVar(), "RB Value SMPL": tk.IntVar()}
        self.container_var["Spike Elimination Check"]["RB Value STD"].set(1)
        self.container_var["Spike Elimination Check"]["RB Value SMPL"].set(1)
        self.container_var["Geothermometry"] = {"Titanium in Quartz": tk.IntVar()}
        self.container_var["Geothermometry"]["Titanium in Quartz"].set(0)

        self.container_var["Spike Elimination"] = {
            "STD": {"State": False}, "SMPL": {"State": False}, "Threshold": {}, "Alpha": tk.StringVar()}
        self.container_var["Spike Elimination"]["Alpha"].set("0.05")

        self.container_var["Initialization"] = {"STD": False, "SMPL": False, "ISOTOPES": False}

        self.container_icpms = {"name": None, "skipheader": 1, "skipfooter": 0, "timestamp": 0}
        self.container_var["ICP-MS Info"] = {"name": tk.StringVar(), "skipheader": tk.IntVar(),
                                             "skipfooter": tk.IntVar(), "timestamp": tk.IntVar()}
        self.container_var["ICP-MS Info"]["name"].set("Unknown ICP-MS")
        self.container_var["ICP-MS Info"]["skipheader"].set(1)
        self.container_var["ICP-MS Info"]["skipfooter"].set(0)
        self.container_var["ICP-MS Info"]["timestamp"].set(0)

        self.container_var["Halter2002"] = {
            "Dimension a": tk.StringVar(), "Dimension b": tk.StringVar(), "Rho(Host)": tk.StringVar(),
            "Rho(Incl)": tk.StringVar(), "Laser Radius": tk.StringVar(), "Name": tk.StringVar()}
        self.container_var["Halter2002"]["Dimension a"].set("50.0")
        self.container_var["Halter2002"]["Dimension b"].set("50.0")
        self.container_var["Halter2002"]["Rho(Host)"].set("2700.0")
        self.container_var["Halter2002"]["Rho(Incl)"].set("1200.0")
        self.container_var["Halter2002"]["Laser Radius"].set("75.0")
        self.container_var["Halter2002"]["Name"].set("Select isotope")

        self.container_var["Borisova2021"] = {
            "Radius Inclusion": tk.StringVar(), "Radius Host": tk.StringVar(), "Rho(Host)": tk.StringVar(),
            "Rho(Incl)": tk.StringVar(), "Name": tk.StringVar()}
        self.container_var["Borisova2021"]["Radius Inclusion"].set("50.0")
        self.container_var["Borisova2021"]["Radius Host"].set("75.0")
        self.container_var["Borisova2021"]["Rho(Host)"].set("2700.0")
        self.container_var["Borisova2021"]["Rho(Incl)"].set("1200.0")
        self.container_var["Borisova2021"]["Name"].set("Select isotope")

        self.file_system_need_update = True
        self.pypitzer_performed = False
        self.initialization_run_std = True
        self.initialization_run_smpl = True
        self.var_calculation = False
        self.bool_incl_is_massbalance = False
        self.bool_incl_is_chargebalance = False
        self.bool_incl_is_pypitzer = False
        self.bool_incl_is_custom = False
        self.bool_incl_is_external = False
        self.str_incl_is_custom_external = "Unknown"

        self.bool_matrixonlytracer = False
        self.bool_secondinternalstandard = False
        self.bool_halter2002 = False
        self.bool_borisova2021 = False

        self.container_flags = {"STD": {"Initialization": False}, "SMPL": {"Initialization": False}}

        self.list_std_changed = False
        self.list_smpl_changed = False
        self.list_std_previous = []
        self.list_smpl_previous = []

        self.container_spike_values = {}

        self.old_file = False
        self.without_pypitzer = False

        self.counter_fast_track_std = 0
        self.counter_fast_track_smpl = 0

        self.container_spikes = {}

        self.autodetection_bg = False
        self.autodetection_sig = False

        self.init_fi_massbalance = False
        self.init_fi_chargebalance = False

        self.container_checkup = {"STD": [], "SMPL": [], "Counter STD": 0, "Counter SMPL": 0}

        self.initial_spike_elimination_std = False
        self.initial_spike_elimination_smpl = False

        ## ALL
        self.container_intensity = {"STD": {"RAW": {}, "SMOOTHED": {}}, "SMPL": {"RAW": {}, "SMOOTHED": {}}}
        self.container_intensity_mix = {"STD": {"RAW": {}, "SMOOTHED": {}}, "SMPL": {"RAW": {}, "SMOOTHED": {}}}
        self.container_intensity_corrected = {"STD": {"RAW": {}, "SMOOTHED": {}}, "SMPL": {"RAW": {}, "SMOOTHED": {}}}
        self.container_intensity_ratio = {"STD": {"RAW": {}, "SMOOTHED": {}}, "SMPL": {"RAW": {}, "SMOOTHED": {}}}
        self.container_normalized_sensitivity = {"STD": {"RAW": {}, "SMOOTHED": {}},
                                                 "SMPL": {"RAW": {}, "SMOOTHED": {}}}
        self.container_analytical_sensitivity = {"STD": {"RAW": {}, "SMOOTHED": {}},
                                                 "SMPL": {"RAW": {}, "SMOOTHED": {}}}
        self.container_rsf = {"STD": {"RAW": {}, "SMOOTHED": {}}, "SMPL": {"RAW": {}, "SMOOTHED": {}}}
        self.container_concentration = {"STD": {"RAW": {}, "SMOOTHED": {}}, "SMPL": {"RAW": {}, "SMOOTHED": {}}}
        self.container_mixed_concentration = {"SMPL": {"RAW": {}, "SMOOTHED": {}}}
        self.container_mixed_concentration_error = {"SMPL": {"RAW": {}, "SMOOTHED": {}}}
        self.container_concentration_ratio = {"STD": {"RAW": {}, "SMOOTHED": {}}, "SMPL": {"RAW": {}, "SMOOTHED": {}}}
        self.container_lob = {"STD": {"RAW": {}, "SMOOTHED": {}}, "SMPL": {"RAW": {}, "SMOOTHED": {}}}
        self.container_lod = {"STD": {"RAW": {}, "SMOOTHED": {}}, "SMPL": {"RAW": {}, "SMOOTHED": {}}}
        self.container_loq = {"STD": {"RAW": {}, "SMOOTHED": {}}, "SMPL": {"RAW": {}, "SMOOTHED": {}}}
        self.container_mixed_concentration_ratio = {"SMPL": {"RAW": {}, "SMOOTHED": {}}}
        self.container_mixing_ratio = {"SMPL": {"RAW": {}, "SMOOTHED": {}}}

        self.container_signs = {"STD": {}, "SMPL": {}}
        self.rb_report = tk.IntVar()
        self.rb_report.set(0)

        self.helper_var_rb = {
            "File setup": {"QA section": tk.IntVar(), "Section": tk.IntVar(), "Analysis mode": tk.IntVar(),
                           "Interval limits": tk.IntVar()},
            "Mass/Charge balance": {"File selection": tk.IntVar()}}
        self.helper_var_rb["File setup"]["QA section"].set(1)
        self.helper_var_rb["File setup"]["Section"].set(0)
        self.helper_var_rb["File setup"]["Analysis mode"].set(0)
        self.helper_var_rb["File setup"]["Interval limits"].set(0)
        self.helper_var_rb["Mass/Charge balance"]["File selection"].set(0)

        self.helper_var_entr = {
            "File setup": {"Start": tk.StringVar(), "End": tk.StringVar()},
            "Mass/Charge balance": {"Salinity": tk.StringVar(), "Concentration": tk.StringVar()}}
        self.helper_var_entr["File setup"]["Start"].set("Set start value")
        self.helper_var_entr["File setup"]["End"].set("Set end value")
        self.helper_var_entr["Mass/Charge balance"]["Salinity"].set("0.0")
        self.helper_var_entr["Mass/Charge balance"]["Concentration"].set("0.0")

        self.helper_var_opt = {
            "File setup": {"Reference isotope 1": tk.StringVar(), "Reference isotope 2": tk.StringVar(),
                           "Parameter": tk.StringVar()},
            "Mass/Charge balance": {"Assemblage": tk.StringVar(), "File": tk.StringVar(), "IS": tk.StringVar()}}
        self.helper_var_opt["File setup"]["Reference isotope 1"].set("None")
        self.helper_var_opt["File setup"]["Reference isotope 2"].set("None")
        self.helper_var_opt["File setup"]["Parameter"].set("None")
        self.helper_var_opt["Mass/Charge balance"]["Assemblage"].set("Select assemblage")
        self.helper_var_opt["Mass/Charge balance"]["File"].set("Select file")
        self.helper_var_opt["Mass/Charge balance"]["IS"].set("Select isotope")

        self.helper_var_cb = {"Mass/Charge balance": {"Molality consideration": tk.IntVar()}}
        self.helper_var_cb["Mass/Charge balance"]["Molality consideration"].set(0)

        self.helper_lists = {"Considered chlorides": {}, "Salinity": {}, "Assemblages": {}}

        ## LANGUAGE SUPPORT
        self.language_dict = {
            "Select Mode": {
                "English": "Select Mode", "German": "Modusauswahl", "Chinese": "选择模式",
                "Greek": "Επιλέξτε Λειτουργία",
                "Russian": "Выбор режима", "Spanish": "Seleccionar modo", "Italian": "Selezionare la modalità"},
            "Standard Files": {"English": "Standard Files", "German": "Standardmessungen", "Chinese": "标准文件"},
            "Sample Files": {"English": "Sample Files", "German": "Probenmessungen", "Chinese": "样本文件"},
            "ICP-MS file setup": {"English": "ICP-MS file setup", "German": "ICP-MS Dateikonfiguration"},
            "Define ICP-MS": {"English": "Define ICP-MS", "German": "ICP-MS einstellen"},
            "Project": {"English": "Project", "German": "Projekt"},
            "Inclusions": {"English": "Inclusion Analysis", "German": "Einschlüsse"},
            "Add": {"English": "Add", "German": "Hinzufügen"},
            "Isotope": {"English": "Isotope", "German": "Isotop"},
            "Set salinity": {"English": "Set salinity", "German": "Salinität eingeben"},
            "Interval setup": {"English": "Interval setup", "German": "Zeintfenster Einstellungen"},
            "Remove interval": {"English": "Remove interval", "German": "Zeitfenster entfernen"},
            "Remove": {"English": "Remove", "German": "Entfernen"},
            "Remove sample intervals": {"English": "Remove sample intervals", "German": "Probenintervalle entfernen"},
            "Remove inclusion intervals": {
                "English": "Remove inclusion intervals", "German": "Einschlussintervalle entfernen"},
            "Remove matrix intervals": {"English": "Remove matrix intervals", "German": "Matrixintervalle entfernen"},
            "Update": {"English": "Update", "German": "Aktualisieren"},
            "Error course": {"English": "Error course", "German": "Fehlerverlauf"},
            "Chart analysis": {"English": "Chart analysis", "German": "Chartanalyse", "Chinese": "图表分析"},
            "Parallelism": {"English": "Parallelism", "German": "Parallelität"},
            "No selection": {"English": "No selection", "German": "Keine Auswahl"},
            "Background interval": {"English": "Background interval", "German": "Zeitfenster Untergrund"},
            "Sample interval": {"English": "Sample interval", "German": "Zeitfenster Probe"},
            "Matrix interval": {"English": "Matrix interval", "German": "Zeitfenster Matrix"},
            "Inclusion interval": {"English": "Inclusion interval", "German": "Zeitfenster Einschluss"},
            "Standard file": {"English": "Standard file", "German": "Standardmessung"},
            "Sample file": {"English": "Sample file", "German": "Probenmessung"},
            "Confirm all": {"English": "Confirm all", "German": "Alles bestätigen"},
            "Confirm": {"English": "Confirm", "German": "Bestätigen"},
            "Extend left": {"English": "Move left", "German": "Links verschieben"},
            "Extend right": {"English": "Move right", "German": "Rechts verschieben"},
            "Measured isotopes": {"English": "Measured isotopes", "German": "Gemessene Isotope"},
            "Display options": {"English": "Display options", "German": "Anzeigeoptionen"},
            "Analysis mode": {"English": "Analysis mode", "German": "Analysemodus"},
            "Show all": {"English": "Show all", "German": "Alles anzeigen"},
            "Hide all": {"English": "Hide all", "German": "Alles ausblenden"},
            "Show only RAW": {"English": "Show only RAW", "German": "RAW anzeigen"},
            "Hide only RAW": {"English": "Hide only RAW", "German": "RAW ausblenden"},
            "Time series view": {"English": "Time series view", "German": "Zeitreihe Messung"},
            "Time ratios view": {"English": "Time ratios view", "German": "Zeitreihe Verhältnisse"},
            "Time signal view": {"English": "Time signal view", "German": "Zeit-Signal Diagramm"},
            "Spectral data view": {"English": "Spectral data view", "German": "Datenspektrum"},
            "Box plot data view": {"English": "Box plot data view", "German": "Kastengrafik"},
            "Quick analysis": {"English": "Quick analysis", "German": "Schnellanalyse"},
            "Stepwise analysis": {"English": "Stepwise analysis", "German": "Schrittweise Analyse"},
            "Additional diagrams": {"English": "Additional diagrams", "German": "Weitere Diagramme"},
            "Copy": {"English": "Copy", "German": "Kopieren"},
            "Delete": {"English": "Delete", "German": "Löschen"},
            "Rename": {"English": "Rename", "German": "Umbenennen"},
            "Data analysis": {"English": "Data analysis", "German": "Datenanalyse"},
            "Interval options": {"English": "Interval options", "German": "Intervalleinstellungen"},
            "Reference isotope": {"English": "Reference isotope", "German": "Referenzisotop"},
            "Additional analysis": {"English": "Additional analysis", "German": "Weitere Analysen"},
            "Time-Signal Diagram": {"English": "Time-signal diagram", "German": "Zeit-Signal Diagramm"},
            "New Project": {"English": "New Project", "German": "Neues Projekt"},
            "Load Project": {"English": "Open Project", "German": "Projekt laden"},
            "Save Project": {"English": "Save Project", "German": "Projekt speichern"},
            "General Settings": {"English": "Main Settings", "German": "Haupteinstellungen"},
            "About": {"English": "About PySILLS", "German": "Über PySILLS"},
            "Python path": {"English": "Python path", "German": "Python Pfad"},
            "PySILLS path": {"English": "PySILLS path", "German": "PySILLS Pfad"},
            "Before": {"English": "Previous", "German": "Zurück"},
            "Project file": {"English": "Project file", "German": "Projektdatei", "Chinese": "项目文件"},
            "Previous": {"English": "Previous", "German": "Zurück"},
            "Next": {"English": "Next", "German": "Weiter"},
            "Quit": {"English": "Quit", "German": "Beenden"},
            "Manager": {"English": "Manager", "German": "Manager"},
            "Setup": {"English": "Setup", "German": "Einstellungen"},
            "Select ICP-MS": {"English": "Select ICP-MS", "German": "ICP-MS auswählen"},
            "Mineral Analysis": {"English": "Mineral Analysis", "German": "Minerale"},
            "Settings": {"English": "Settings", "German": "Einstellungen"},
            "Results": {"English": "Results", "German": "Ergebnisse"},
            "Extras": {"English": "Extras", "German": "Extras"},
            "Mass balance": {"English": "Mass balance", "German": "Massenbilanz"},
            "Charge balance": {"English": "Charge balance", "German": "Ladungsbilanz"},
            "Fluid Inclusions": {"English": "Fluid Inclusions", "German": "Flüssigkeitseinschlüsse"},
            "Geometric approach": {"English": "Geometric approach", "German": "Geometrischer Ansatz"},
            "External quantification support": {
                "English": "External quantification", "German": "Externe Quantifizierung"},
            "Custom data": {"English": "Custom data", "German": "Manuelle Daten"},
            "Melt Inclusions": {"English": "Melt Inclusions", "German": "Schmelzeinschlüsse"},
            "Development": {"English": "Development", "German": "Entwicklung"},
            "Additional Information": {"English": "Additional Information", "German": "Weitere Informationen"},
            "Citation": {"English": "Citing PySILLS", "German": "PySILLS zitieren"},
            "Matrix Settings": {"English": "Matrix Settings", "German": "Einstellungen - Matrix"},
            "References": {"English": "References", "German": "Quellen"},
            "Project Information": {"English": "Project Information", "German": "Projektinformationen"},
            "Calculation Accuracy": {"English": "Calculation Accuracy", "German": "Numerische Genauigkeit"},
            "Sensitivity Drift": {"English": "Sensitivity Drift", "German": "Sensitivitätsverschiebung"},
            "Limit of Detection": {"English": "Limit of Detection", "German": "Nachweisgrenze"},
            "Default ICP-MS": {"English": "Default ICP-MS", "German": "Voreinstellung ICP-MS"},
            "Offset Automatic Interval Detection": {
                "English": "Offset Automatic Interval Detection",
                "German": "Versatz der automatischen Intervalerkennung"},
            "Colormap": {"English": "Colormap", "German": "Farbpalette"},
            "Report File": {"English": "Report File", "German": "Ausgabedatei"},
            "Plotting Setup": {"English": "Plotting Setup", "German": "Diagrammeinstellungen"},
            "Language Selection": {"English": "Language Selection", "German": "Sprachauswahl"},
            "Data Processing": {"English": "Data Processing", "German": "Datenverarbeitung"},
            "MINERAL ANALYSIS - Setup": {
                "English": "Mineral analysis - Setup", "German": "Mineralanalyse - Einstellungen"},
            "FLUID INCLUSION ANALYSIS - Setup": {
                "English": "Fluid inclusion analysis - Setup",
                "German": "Flüssigkeitseinschlussanalyse - Einstellungen"},
            "INCLUSION ANALYSIS - Setup": {
                "English": "Inclusion analysis - Setup", "German": "Einschlussanalyse - Einstellungen"},
            "MELT INCLUSION ANALYSIS - Setup": {
                "English": "Melt inclusion analysis - Setup",
                "German": "Schmelzeinschlussanalyse - Einstellungen"},
            "Author": {"English": "Author", "German": "Autor"},
            "Select last solid": {"English": "Select last solid", "German": "Letzten Festkörper auswählen"},
            "Source ID": {"English": "Source ID", "German": "Proben ID"},
            "Standard Reference Material (SRM)": {
                "English": "Standard Reference Material", "German": "Standard/Referenzmaterial"},
            "Isotopes": {"English": "Isotopes", "German": "Isotope"},
            "Dwell times (in s)": {"English": "Dwell times (in s)", "German": "Haltezeiten (in s)"},
            "Sample Settings": {"English": "Sample Settings", "German": "Probeneinstellungen"},
            "Dwell Time Settings": {"English": "Dwell Time Settings", "German": "Einstellungen - Haltezeiten"},
            "Default Time Window (Background)": {
                "English": "Default Time Interval (Background)", "German": "Einstellungen Zeitintervall (Untergrund)"},
            "Default Time Window (Matrix)": {
                "English": "Default Time Interval (Matrix)", "German": "Einstellungen - Zeitintervall (Matrix)"},
            "Default Time Window (Sample)": {
                "English": "Default Time Interval (Sample)", "German": "Einstellungen - Zeitintervall (Probe)"},
            "Check-Up - Standard Reference Material": {
                "English": "Check-Up - Standard Reference Material",
                "German": "Kontrolle - Standard/Referenzmaterial (SRM)"},
            "SRM values": {"English": "SRM values", "German": "SRM Werte"},
            "Name": {"English": "Name", "German": "Name"},
            "Category": {"English": "Category", "German": "Kategorie"},
            "Primary isotope": {"English": "Isotope Nr. 1", "German": "Isotop Nr. 1"},
            "Secondary isotope": {"English": "Isotope Nr. 2", "German": "Isotop Nr. 2"},
            "Intensity results": {"English": "Results - Intensity", "German": "Ergebnisse - Intensität"},
            "Sensitivity results": {"English": "Results - Sensitivity", "German": "Ergebnisse - Sensitivität"},
            "Composition results": {"English": "Results - Composition", "German": "Ergebnisse - Zusammensetzung"},
            "ID": {"English": "ID", "German": "ID"},
            "SRM": {"English": "SRM", "German": "SRM"},
            "Concentration (ppm)": {"English": "Concentration (ppm)", "German": "Konzentration (ppm)"},
            "Element": {"English": "Element", "German": "Element"},
            "Start": {"English": "Start", "German": "Start"},
            "End": {"English": "End", "German": "Ende"},
            "Run": {"English": "Run", "German": "Ausführen"},
            "N(isotopes)": {"English": "N(isotopes)", "German": "N(Isotope)"},
            "t(start)": {"English": "t(start)", "German": "t(Start)"},
            "t(end)": {"English": "t(end)", "German": "t(Ende)"},
            "t(day)": {"English": "t(day)", "German": "t(Tag)"},
            "date": {"English": "date", "German": "Datum"},
            "Spike Elimination": {"English": "Spike Elimination", "German": "Einstellungen - Ausreißer"},
            "Calculation Method": {"English": "Calculation Method", "German": "Berechnungsmethode"},
            "Check-Up": {"English": "Check-Up", "German": "Kontrolle"},
            "Internal Standard": {"English": "Internal Standard", "German": "Interner Standard"},
            "Exclude inclusion": {"English": "Exclude inclusion", "German": "Inklusion ausschließen"},
            "Check inclusion": {"English": "Check only inclusion", "German": "Nur Inklusion überprüfen"},
            "Quantification Method": {"English": "Quantification Method", "German": "Quantifizierungsmethode"},
            "Significance level": {"English": "Significance level", "German": "Signifikanzniveau"},
            "Threshold": {"English": "Threshold", "German": "Schwellenwert"},
            "Remove all": {"English": "Remove all", "German": "Alles entfernen"},
            "Apply to all": {"English": "Apply to all", "German": "Auf alles anwenden"},
            "Check": {"English": "Check", "German": "Überprüfen"},
            "Calculation Intervals": {"English": "Calculation intervals", "German": "Zeitintervalle"},
            "Acquisition Times": {"English": "Acquisition Times", "German": "Messzeiten"},
            "Acquisition Time": {"English": "Acquisition time", "German": "Messzeit"},
            "Imported Files": {"English": "Imported Files", "German": "Importierte Dateien"},
            "Line of time stamp": {"English": "Line of time stamp", "German": "Zeitstempel in Zeile"},
            "Skip header": {"English": "Skip header", "German": "Überspringe ... Kopfzeilen"},
            "Skip footer": {"English": "Skip footer", "German": "Überspringe ... Fußzeilen"},
            "Save ICP-MS setup": {"English": "Save ICP-MS setup", "German": "ICP-MS Daten speichern"},
            "100 wt.% oxides": {"English": "100 wt.% oxides", "German": "100 Gew.-% Oxide"},
            "Auto-Detection": {"English": "Auto-Detection", "German": "Auto-Erkennung"},
            "Set start time": {"English": "Set start time", "German": "Startzeit setzen"},
            "Set end time": {"English": "Set end time", "German": "Endzeit setzen"},
            "Select oxide": {"English": "Select oxide", "German": "Oxid auswählen"},
            "Select isotope": {"English": "Select isotope", "German": "Isotop auswählen"},
            "Select element": {"English": "Select element", "German": "Element auswählen"},
            "Select file": {"English": "Select file", "German": "Datei auswählen"},
            "Select all": {"English": "Select all", "German": "Alles auswählen"},
            "Select": {"English": "Select", "German": "Auswählen"},
            "Alkali metals": {"English": "Alkali metals", "German": "Alkalimetalle"},
            "Alkaline earth metals": {"English": "Alkaline earth metals", "German": "Erdalkalimetalle"},
            "Transition metals": {"English": "Transition metals", "German": "Übergangsmetalle"},
            "Other metals": {"English": "Other metals", "German": "Andere Metalle"},
            "Metalloids": {"English": "Metalloids", "German": "Halbmetalle"},
            "Lanthanides": {"English": "Lanthanides", "German": "Lanthanoide"},
            "Actinides": {"English": "Actinides", "German": "Actinoide"},
            "Non-metals": {"English": "Non-metals", "German": "Nichtmetalle"},
            "Halogens": {"English": "Halogens", "German": "Halogene"},
            "Noble gases": {"English": "Noble gases", "German": "Edelgase"},
            "Select all oxides": {"English": "Select all oxides", "German": "Alle Oxide auswählen"},
            "Deselect all oxides": {"English": "Deselect all oxides", "German": "Alle Oxide abwählen"},
            "Rock-forming elements": {"English": "Rock-forming elements", "German": "Gesteinsbildende Elemente"},
            "Fix quantification": {"English": "Fix quantification", "German": "Quantifizierung reparieren"},
            "Calculate again": {"English": "Calculate again", "German": "Berechnung wiederholen"},
            "Molality-based quantification": {
                "English": "Molality-based quantification", "German": "Quantifizierung gemäß Molalität"},
            "Guess the composition": {"English": "Guess the composition", "German": "Zusammensetzung schätzen"},
            "Setup - Oxide ratios": {"English": "Setup - Oxide ratios", "German": "Einstellungen - Oxidverhältnisse"},
            "Setup - Reference element": {
                "English": "Setup - Reference element", "German": "Einstellungen - Referenzelement"},
            "Setup - Oxide selection": {"English": "Setup - Oxide selection", "German": "Einstellungen - Oxidauswahl"},
            "Run 100 wt.% oxides calculation for": {
                "English": "Run 100 wt.% oxides calculation for", "German": "Berechnung 100 Gew.% Oxide ausführen für"},
            "Setup - Files": {"English": "Setup - Files", "German": "Einstellungen - Dateien"},
            "Mineral quantification": {"English": "Mineral quantification", "German": "Mineralquantifizierung"},
            "Inclusion quantification": {"English": "Inclusion quantification", "German": "Einschlussquantifizierung"},
            "Assemblage Settings": {"English": "Assemblage settings", "German": "Einstellungen - Assemblage"},
            "Isotope Dwell Times": {"English": "Isotope Dwell Times", "German": "Haltezeiten je Isotop"},
            "Inclusion Settings": {"English": "Inclusion Settings", "German": "Einstellungen - Einschlüsse"},
            "Internal Standard Setup": {
                "English": "Internal Standard Setup", "German": "Einstellungen interner Standard"},
            "Oxide stoichiometry": {"English": "Oxide stoichiometry", "German": "Oxide Stöchiometrie"},
            "Default settings": {"English": "Default settings", "German": "Voreinstellungen"},
            "Composition": {"English": "Composition", "German": "Zusammensetzung"},
            "Cations": {"English": "Cations", "German": "Kationen"},
            "Anions": {"English": "Anions", "German": "Anionen"},
            "Melting temperature of the last solid": {
                "English": "Melting temperature of the last solid",
                "German": "Schmelztemperatur des letzten Festkörpers"},
            "Melting temperature": {"English": "Melting temperature", "German": "Schmelztemperatur"},
            "Last solid": {"English": "Last solid", "German": "Letzter Festkörper"},
            "Select IS": {"English": "Select IS", "German": "IS auswählen"},
            "Select mineral": {"English": "Select mineral", "German": "Mineral auswählen"},
            "Element Calculation": {"English": "Element Calculation", "German": "Elementberechnung"},
            "Experimental Data": {"English": "Experimental Data", "German": "Experimentelle Daten"},
            "Import external data": {"English": "Import external data", "German": "Externe Daten importieren"},
            "LA-ICP-MS data export": {"English": "LA-ICP-MS data export", "German": "LA-ICP-MS Daten exportieren"},
            "Oxide Setup": {"English": "Oxide Setup", "German": "Oxideinstellungen"},
            "Import data": {"English": "Import data", "German": "Daten importieren"},
            "Application launcher": {"English": "Application launcher", "German": "Startdatei"},
            "Dwell times": {"English": "Dwell times", "German": "Haltezeiten"},
            "Default author": {"English": "Default author", "German": "Voreinstellung (Name)"},
            "Default SRM": {"English": "Default SRM", "German": "Voreinstellung (SRM)"},
            "Median": {"English": "Median", "German": "Median"},
            "Standard deviation": {"English": "Standard deviation", "German": "Standardabweichung"},
            "Charge": {"English": "Charge", "German": "Ladung"},
            "charged": {"English": "charged", "German": "geladen"},
            "Screen resolution": {"English": "Screen resolution", "German": "Bildschirmauflösung"},
            "Arithmetic mean": {"English": "Arithmetic mean", "German": "Arithmetisches Mittel"},
            "Average of every interval": {
                "English": "Average of every interval", "German": "Durchschnitt pro Interval"},
            "Stack all intervals into one": {
                "English": "Stack all intervals into one", "German": "Alle Intervalle stappeln"},
            "File type": {"English": "File type", "German": "Dateiendung"},
            "Data type": {"English": "Data type", "German": "Datentyp"},
            "Data selection": {"English": "Data selection", "German": "Datenauswahl"},
            "Assemblage": {"English": "Assemblage", "German": "Assemblage"},
            "Value": {"English": "Value", "German": "Wert"},
            "Select standard file": {"English": "Select standard file", "German": "Standardmessung auswählen"},
            "Select sample file": {"English": "Select sample file", "German": "Probenmessung auswählen"},
            "Select parameter": {"English": "Select parameter", "German": "Parameter auswählen"},
            "Original data": {"English": "Original data", "German": "Originaldaten"},
            "Smoothed data": {"English": "Smoothed data", "German": "Korrigierte Daten"},
            "Current data": {"English": "Current data", "German": "Aktuelle Daten"},
            "Current outlier": {"English": "Current outlier", "German": "Aktueller Ausreißer"},
            "Scatter plot": {"English": "Scatter plot", "German": "Streudiagramm"},
            "Histogram plot": {"English": "Histogram plot", "German": "Häufigkeitsdiagramm"},
            "Diagram selection": {"English": "Diagram selection", "German": "Diagrammauswahl"},
            "Plotting setup": {"English": "Plotting setup", "German": "Ploteinstellungen"},
            "Section selection": {"English": "Phase selection", "German": "Phasenauswahl"},
            "Select section": {"English": "Select section", "German": "Abschnitt auswählen"},
            "Parameter selection": {"English": "Select parameter", "German": "Parameter auswählen"},
            "Color coding": {"English": "Color coding", "German": "Farbkodierung"},
            "Isotope selection": {"English": "Isotope selection", "German": "Isotopauswahl"},
            "Outlier selection": {"English": "Outlier selection", "German": "Ausreißerauswahl"},
            "Outlier correction": {"English": "Outlier correction", "German": "Ausreißerkorrektur"},
            "Number of outliers": {"English": "Number of outliers", "German": "Anzahl Ausreißer"},
            "File selection": {"English": "File selection", "German": "Dateiauswahl"},
            "Original value": {"English": "Original value", "German": "Originalwert"},
            "Smoothed value": {"English": "Smoothed value", "German": "Korrekturwert"},
            "Current value": {"English": "Current value", "German": "Aktueller Wert"},
            "Value selection": {"English": "Value selection", "German": "Wert auswählen"},
            "Outlier check": {"English": "Outlier check", "German": "Ausreißerüberprüfung"},
            "Signal intensity": {"English": "Signal intensity", "German": "Signalintensität"},
            "Experiment time": {"English": "Experiment time", "German": "Experimentzeit"},
            "Delimiter": {"English": "Delimiter", "German": "Trennzeichen"},
            "Sample": {"English": "Sample", "German": "Probe"},
            "of": {"English": "of", "German": "von"},
            "with": {"English": "with", "German": "mit"},
            "Up": {"English": "Up", "German": "Hoch"},
            "Down": {"English": "Down", "German": "Runter"},
            "without": {"English": "without", "German": "ohne"},
            "Densities": {"English": "Densities", "German": "Dichten"},
            "Dimensions": {"English": "Dimensions", "German": "Abmessungen"},
            "Inclusion intensity": {"English": "Inclusion intensity", "German": "Einschlussintensität"},
            "Inclusion dimensions": {"English": "Inclusion dimensions", "German": "Einschlussgröße"},
            "Inclusion concentration": {"English": "Inclusion concentration", "German": "Einschlusskonzentration"},
            "Length of": {"English": "Length of", "German": "Länge der"},
            "semi-axis": {"English": "semi-axis", "German": "Halbachse"},
            "Ablation radius": {"English": "Ablation radius", "German": "Ablationsradius"},
            "Inclusion radius": {"English": "Inclusion radius", "German": "Einschlussradius"},
            "Inclusion density": {"English": "Inclusion density", "German": "Einschlussdichte"},
            "Matrix density": {"English": "Matrix density", "German": "Matrixdichte"},
            "elements": {"English": "elements", "German": "Elemente"},
            "element ratios": {"English": "element ratios", "German": "Elementverhältnisse"},
            "oxides": {"English": "oxides", "German": "Oxide"},
            "oxide ratios": {"English": "oxide ratios", "German": "Oxidverhältnisse"},
            "with respect to": {"English": "with respect to", "German": "bezogen auf"},
            "x-y diagram": {"English": "X-Y diagram", "German": "X-Y-Diagramm"},
            "Ternary diagram": {"English": "Ternary diagram", "German": "Dreiecksdiagramm"},
            "Geothermometry": {"English": "Geothermometry", "German": "Geothermometrie"},
            "Matrix": {"English": "Matrix", "German": "Matrix"},
            "Inclusion": {"English": "Inclusion", "German": "Einschluss"},
            "Additional Settings": {"English": "Additional settings", "German": "Weitere Einstellungen"},
            "Select x": {"English": "Select x", "German": "x auswählen"},
            "Select y": {"English": "Select y", "German": "y auswählen"},
            "Select z": {"English": "Select z", "German": "z auswählen"},
            "Mixed signal": {"English": "Mixed signal", "German": "Gemischtes Signal"},
            "Concentration": {"English": "Concentration", "German": "Concentration"},
            "Concentration ratio": {"English": "Concentration ratio", "German": "Konzentrationsverhältnis"},
            "Limit of detection": {"English": "Limit of detection", "German": "Nachweisgrenze"},
            "Mixed intensity": {"English": "Mixed intensity", "German": "Gemischte Intensität"},
            "Mixed concentration": {"English": "Mixed concentration", "German": "Gemischte Konzentration"},
            "Mixing ratio": {"English": "Mixing ratio", "German": "Mischungsverhältnis"},
            "Intensity": {"English": "Intensity", "German": "Intensität"},
            "Intensity ratio": {"English": "Intensity ratio", "German": "Intensitätsverhältnis"},
            "Analytical sensitivity": {"English": "Analytical sensitivity", "German": "Analytische Sensitivität"},
            "Normalized sensitivity": {"English": "Normalized sensitivity", "German": "Normalisierte Sensitivität"},
            "Relative sensitivity factor": {
                "English": "Relative sensitivity factor", "German": "Relativer Sensitivitätsfaktor"},
            "Export results": {"English": "Export results", "German": "Ergebnisse exportieren"},
            "Further data analysis": {"English": "Further data analysis", "German": "Weitere Datenanalyse"},
            "Sensitivity drift": {"English": "Sensitivity drift", "German": "Sensitivitätsverschiebung"},
            "Internal standard": {"English": "Internal standard", "German": "Interner Standard"},
            "Sorting by": {"English": "Sorting by", "German": "Sortierung nach"},
            "All in one file": {"English": "All in one file", "German": "Alles in eine Datei"},
            "Background": {"English": "Background", "German": "Untergrund"},
            "Main signal": {"English": "Main signal", "German": "Hauptsignal"},
            "Line width": {"English": "Line width", "German": "Linienbreite"},
            "Save settings": {"English": "Save settings", "German": "Einstellungen speichern"},
            "Create launcher": {"English": "Create launcher", "German": "Startdatei erstellen"},
            "Calculation formula from ...": {
                "English": "Calculation formula from ...", "German": "Berechnung basierend auf ..."},
            "PySILLS color scheme": {"English": "PySILLS color scheme", "German": "PySILLS Farbschema"},
            "Default value": {"English": "Default value", "German": "Voreinstellung"},
            "Composition Setup": {"English": "Composition Setup", "German": "Zusammensetzung\n definieren"},
            "Mineral stoichiometry": {"English": "Mineral stoichiometry", "German": "Minerale Stöchiometrie"},
            "PySILLS subtitle": {
                "English": "Major, minor and trace element analysis of minerals, fluid and melt inclusions",
                "German":
                    "Haupt-, Neben- und Spurenelementanalysie von Mineralen, Flüssigkeits- und Schmelzeinschlüssen"},
            "File-specific Setup": {"English": "File-specific Setup", "German": "Datei-spezifische\n Einstellungen"}}

        self.var_language = self.container_var["General Settings"]["Language"].get()

        ## MINERAL ANALYSIS
        str_var_01 = self.language_dict["Set start time"][self.var_language]
        str_var_02 = self.language_dict["Set end time"][self.var_language]
        str_var_03 = self.language_dict["Select IS"][self.var_language]
        str_var_04 = self.language_dict["Select oxide"][self.var_language]
        str_var_05 = self.language_dict["Select element"][self.var_language]
        str_var_06 = self.language_dict["Select mineral"][self.var_language]

        self.container_var["settings"] = {
            "Time BG Start": tk.StringVar(), "Time BG End": tk.StringVar(), "Time SIG Start": tk.StringVar(),
            "Time SIG End": tk.StringVar(), "Author": tk.StringVar(), "Source ID": tk.StringVar(),
            "IS Concentration": tk.StringVar(), "SE Alpha": tk.StringVar()}
        self.container_var["settings"]["Time BG Start"].set(str_var_01)
        self.container_var["settings"]["Time BG End"].set(str_var_02)
        self.container_var["settings"]["Time SIG Start"].set(str_var_01)
        self.container_var["settings"]["Time SIG End"].set(str_var_02)
        self.container_var["settings"]["Source ID"].set("RUR01")
        self.container_var["settings"]["IS Concentration"].set("0.0")
        self.container_var["settings"]["SE Alpha"].set("0.05")

        self.container_var["ma_setting"]["Data Type Plot"] = {"STD": {}, "SMPL": {}}
        self.container_var["ma_setting"]["Analyse Mode Plot"] = {"STD": {}, "SMPL": {}}
        self.container_var["ma_setting"]["Display RAW"] = {"STD": {}, "SMPL": {}}
        self.container_var["ma_setting"]["Display SMOOTHED"] = {"STD": {}, "SMPL": {}}
        self.container_var["ma_setting"]["Time-Signal Lines"] = {"STD": {}, "SMPL": {}}
        self.container_var["ma_setting"]["Time-Ratio Lines"] = {"STD": {}, "SMPL": {}}
        self.container_var["ma_setting"]["Checkboxes Isotope Diagram"] = {"STD": {}, "SMPL": {}}
        self.container_var["ma_setting"]["Calculation Interval"] = {"STD": {}, "SMPL": {}}
        self.container_var["ma_setting"]["Calculation Interval Visibility"] = {"STD": {}, "SMPL": {}}
        self.container_var["ma_setting"]["Default IS STD"] = tk.StringVar()
        self.container_var["ma_setting"]["Default IS STD"].set(str_var_03)
        self.container_var["ma_setting"]["Default IS SMPL"] = tk.StringVar()
        self.container_var["ma_setting"]["Default IS SMPL"].set(str_var_03)
        self.container_var["ma_setting"]["Time BG Start"] = tk.StringVar()
        self.container_var["ma_setting"]["Time BG Start"].set(str_var_01)
        self.container_var["ma_setting"]["Time BG End"] = tk.StringVar()
        self.container_var["ma_setting"]["Time BG End"].set(str_var_02)
        self.container_var["ma_setting"]["Time MAT Start"] = tk.StringVar()
        self.container_var["ma_setting"]["Time MAT Start"].set(str_var_01)
        self.container_var["ma_setting"]["Time MAT End"] = tk.StringVar()
        self.container_var["ma_setting"]["Time MAT End"].set(str_var_02)
        self.container_var["ma_setting"]["Author"] = tk.StringVar()
        self.container_var["ma_setting"]["Source ID"] = tk.StringVar()
        self.container_var["ma_setting"]["Source ID"].set("RUR01")
        self.container_var["ma_setting"]["Oxide"] = tk.StringVar()
        self.container_var["ma_setting"]["Oxide"].set(str_var_04)
        self.container_var["ma_setting"]["Oxide Concentration"] = tk.StringVar()
        self.container_var["ma_setting"]["Oxide Concentration"].set("100")
        self.container_var["ma_setting"]["Element"] = tk.StringVar()
        self.container_var["ma_setting"]["Element"].set(str_var_05)
        self.container_var["ma_setting"]["Element Concentration"] = tk.StringVar()
        self.container_var["ma_setting"]["Element Concentration"].set("1000000")
        self.container_var["ma_setting"]["Mineral"] = tk.StringVar()
        self.container_var["ma_setting"]["Mineral"].set(str_var_06)
        self.container_var["ma_setting"]["Mineral Concentration"] = tk.StringVar()
        self.container_var["ma_setting"]["Mineral Concentration"].set("100")
        self.container_var["ma_setting"]["SE Alpha"] = tk.StringVar()
        self.container_var["ma_setting"]["SE Alpha"].set("0.05")
        self.container_var["ma_setting"]["SE Threshold"] = tk.StringVar()
        self.container_var["ma_setting"]["SE Threshold"].set("5000")
        self.container_var["ma_setting"]["Host Setup Selection"] = tk.IntVar()
        self.container_var["ma_setting"]["Host Setup Selection"].set(1)
        self.container_var["ma_setting"]["Quantification Method"] = tk.IntVar()
        self.container_var["ma_setting"]["Quantification Method"].set(1)
        self.container_var["ma_setting"]["Time-Signal Checker"] = tk.IntVar()
        self.container_var["ma_setting"]["Time-Signal Checker"].set(1)

        self.container_var["ma_datareduction_isotopes"]["File Type"] = tk.IntVar()  # e.g. Sample files
        self.container_var["ma_datareduction_isotopes"]["File Type"].set(1)
        self.container_var["ma_datareduction_isotopes"]["Data Type"] = tk.IntVar()  # e.g. RAW data
        self.container_var["ma_datareduction_isotopes"]["Data Type"].set(0)
        self.container_var["ma_datareduction_isotopes"]["Focus"] = tk.IntVar()  # e.g. Matrix
        self.container_var["ma_datareduction_isotopes"]["Focus"].set(0)
        self.container_var["ma_datareduction_isotopes"]["Result Category"] = tk.IntVar()  # e.g. Concentration
        self.container_var["ma_datareduction_isotopes"]["Result Category"].set(0)
        self.container_var["ma_datareduction_isotopes"]["Display Option"] = tk.IntVar()  # e.g. Table mode
        self.container_var["ma_datareduction_isotopes"]["Display Option"].set(0)

        self.container_var["ma_datareduction_files"]["File Type"] = tk.IntVar()  # e.g. Sample files
        self.container_var["ma_datareduction_files"]["File Type"].set(1)
        self.container_var["ma_datareduction_files"]["Data Type"] = tk.IntVar()  # e.g. RAW data
        self.container_var["ma_datareduction_files"]["Data Type"].set(0)
        self.container_var["ma_datareduction_files"]["Focus"] = tk.IntVar()  # e.g. Matrix
        self.container_var["ma_datareduction_files"]["Focus"].set(0)
        self.container_var["ma_datareduction_files"]["Result Category"] = tk.IntVar()  # e.g. Concentration
        self.container_var["ma_datareduction_files"]["Result Category"].set(0)
        self.container_var["ma_datareduction_files"]["Display Option"] = tk.IntVar()  # e.g. Table mode
        self.container_var["ma_datareduction_files"]["Display Option"].set(0)

        self.container_var["ma_datareduction"] = {}
        self.container_var["ma_dataexploration"] = {}
        self.container_var["ma_datareduction"]["Option ID"] = tk.StringVar()
        self.container_var["ma_datareduction"]["Option ID"].set("Select Assemblage")
        self.container_var["ma_datareduction"]["Option IS"] = tk.StringVar()
        self.container_var["ma_datareduction"]["Option Drift"] = tk.StringVar()

        self.container_var["ma_dataexploration"]["Exploration Mode"] = tk.IntVar()
        self.container_var["ma_dataexploration"]["Exploration Mode"].set(0)
        self.container_var["ma_dataexploration"]["STD File"] = tk.StringVar()
        self.container_var["ma_dataexploration"]["STD File"].set("Select File")
        self.container_var["ma_dataexploration"]["SMPL File"] = tk.StringVar()
        self.container_var["ma_dataexploration"]["SMPL File"].set("Select File")
        self.container_var["ma_dataexploration"]["Isotope"] = tk.StringVar()
        self.container_var["ma_dataexploration"]["Isotope"].set("Select Isotope")

        self.var_mode_ma = False
        self.var_init_ma_setting = False
        self.var_init_ma_datareduction = False
        self.var_init_fi_datareduction = False
        self.var_init_mi_datareduction = False
        self.var_init_ma_dataexploration = False

        if restart == False:
            self.last_used_directory = os.getcwd()
            self.last_used_directory_input = os.getcwd()
        else:
            self.last_used_directory = last_used_directory
            self.last_used_directory_input = last_used_directory_input

        ## FLUID/MELT INCLUSION ANALYSIS
        keys = ["fi_setting", "mi_setting"]
        for key_setting in keys:
            self.container_var[key_setting]["Time BG Start"] = tk.StringVar()
            self.container_var[key_setting]["Time BG Start"].set(str_var_01)
            self.container_var[key_setting]["Time BG End"] = tk.StringVar()
            self.container_var[key_setting]["Time BG End"].set(str_var_02)
            self.container_var[key_setting]["Time MAT Start"] = tk.StringVar()
            self.container_var[key_setting]["Time MAT Start"].set(str_var_01)
            self.container_var[key_setting]["Time MAT End"] = tk.StringVar()
            self.container_var[key_setting]["Time MAT End"].set(str_var_02)
            self.container_var[key_setting]["Time INCL Start"] = tk.StringVar()
            self.container_var[key_setting]["Time INCL Start"].set(str_var_01)
            self.container_var[key_setting]["Time INCL End"] = tk.StringVar()
            self.container_var[key_setting]["Time INCL End"].set(str_var_02)
            self.container_var[key_setting]["Author"] = tk.StringVar()
            self.container_var[key_setting]["Source ID"] = tk.StringVar()
            self.container_var[key_setting]["Source ID"].set("RUR01")
            self.container_var[key_setting]["Method"] = tk.StringVar()
            self.container_var[key_setting]["Method"].set("Select Method")
            self.container_var[key_setting]["Host Setup Selection"] = tk.IntVar()
            self.container_var[key_setting]["Host Setup Selection"].set(1)
            self.container_var[key_setting]["Inclusion Setup Selection"] = tk.IntVar()
            self.container_var[key_setting]["Inclusion Setup Selection"].set(1)
            self.container_var[key_setting]["Inclusion Setup Option"] = tk.StringVar()
            self.container_var[key_setting]["Inclusion Setup Option"].set("Mass Balance")
            self.container_var[key_setting]["Last solid"] = tk.StringVar()
            self.container_var[key_setting]["Last solid"].set("Select last solid")

            self.helper_filesetup_lines = {}

            if key_setting == "mi_setting":
                self.container_var[key_setting]["Inclusion Setup Option"].set("100 wt.% Oxides")
            else:
                self.container_var[key_setting]["Inclusion Setup Option"].set("Mass Balance")

            self.container_var[key_setting]["Quantification Method"] = tk.IntVar()
            self.container_var[key_setting]["Quantification Method"].set(1)
            self.container_var[key_setting]["Quantification Method Option"] = tk.StringVar()
            self.container_var[key_setting]["Quantification Method Option"].set("Matrix-only Tracer (SILLS)")
            self.container_var[key_setting]["IS MAT Default"] = tk.StringVar()
            self.container_var[key_setting]["IS MAT Default"].set(str_var_03)
            self.container_var[key_setting]["IS MAT Default Concentration"] = tk.StringVar()
            self.container_var[key_setting]["IS MAT Default Concentration"].set("1000000")
            self.container_var[key_setting]["IS INCL Default"] = tk.StringVar()
            self.container_var[key_setting]["IS INCL Default"].set(str_var_03)
            self.container_var[key_setting]["IS INCL Default Concentration"] = tk.StringVar()
            self.container_var[key_setting]["IS INCL Default Concentration"].set("1000000")
            self.container_var[key_setting]["Oxide"] = tk.StringVar()
            self.container_var[key_setting]["Oxide"].set(str_var_04)
            self.container_var[key_setting]["Oxide Concentration"] = tk.StringVar()
            self.container_var[key_setting]["Oxide Concentration"].set("100")
            self.container_var[key_setting]["Oxide Inclusion"] = tk.StringVar()
            self.container_var[key_setting]["Oxide Inclusion"].set(str_var_04)
            self.container_var[key_setting]["Oxide Inclusion Concentration"] = tk.StringVar()
            self.container_var[key_setting]["Oxide Inclusion Concentration"].set("100")
            self.container_var[key_setting]["Element"] = tk.StringVar()
            self.container_var[key_setting]["Element"].set("Select Element")
            self.container_var[key_setting]["Element Concentration"] = tk.StringVar()
            self.container_var[key_setting]["Element Concentration"].set("100")
            self.container_var[key_setting]["Sulfide"] = tk.StringVar()
            self.container_var[key_setting]["Sulfide"].set("Select Sulfide")
            self.container_var[key_setting]["Sulfide Concentration"] = tk.StringVar()
            self.container_var[key_setting]["Sulfide Concentration"].set("100")
            self.container_var[key_setting]["Halide"] = tk.StringVar()
            self.container_var[key_setting]["Halide"].set("Select Halide")
            self.container_var[key_setting]["Halide Concentration"] = tk.StringVar()
            self.container_var[key_setting]["Halide Concentration"].set("100")
            self.container_var[key_setting]["Mineral"] = tk.StringVar()
            self.container_var[key_setting]["Mineral"].set("Select mineral")
            self.container_var[key_setting]["Mineral Concentration"] = tk.StringVar()
            self.container_var[key_setting]["Mineral Concentration"].set("100")
            self.container_var[key_setting]["Host Only"] = tk.StringVar()
            self.container_var[key_setting]["Host Only"].set("Select Isotope")
            self.container_var[key_setting]["2nd Internal"] = tk.StringVar()
            self.container_var[key_setting]["2nd Internal"].set("Select Isotope")
            self.container_var[key_setting]["2nd Internal Concentration Default"] = tk.StringVar()
            self.container_var[key_setting]["2nd Internal Concentration Default"].set("0")
            self.container_var[key_setting]["Matrix-Only Concentration Default"] = tk.StringVar()
            self.container_var[key_setting]["Matrix-Only Concentration Default"].set("0")
            self.container_var[key_setting]["Matrix Amount Default"] = tk.StringVar()
            self.container_var[key_setting]["Matrix Amount Default"].set("100")
            self.container_var[key_setting]["IS Selection"] = tk.IntVar()
            self.container_var[key_setting]["IS Selection"].set(1)
            self.container_var[key_setting]["Spike Elimination Inclusion"] = tk.IntVar()
            self.container_var[key_setting]["Spike Elimination Inclusion"].set(2)
            self.container_var[key_setting]["Check Inclusion Exclusion"] = tk.BooleanVar()
            self.container_var[key_setting]["Check Inclusion Exclusion"].set(True)
            self.container_var[key_setting]["SE Alpha"] = tk.StringVar()
            self.container_var[key_setting]["SE Alpha"].set("0.05")
            self.container_var[key_setting]["SE Threshold"] = tk.StringVar()
            self.container_var[key_setting]["SE Threshold"].set("5000")
            self.container_var[key_setting]["Inclusion Plugin"] = {
                "Intensity BG": tk.IntVar(), "Intensity MAT": tk.IntVar(), "Intensity MIX": tk.IntVar(),
                "Intensity INCL": tk.IntVar(), "Analytical Sensitivity": tk.IntVar(), "Concentration SRM": tk.IntVar()}
            self.container_var[key_setting]["Time-Signal Checker"] = tk.IntVar()
            self.container_var[key_setting]["Time-Signal Checker"].set(1)
            self.container_var[key_setting]["Inclusion Intensity Calculation"] = tk.IntVar()
            self.container_var[key_setting]["Inclusion Intensity Calculation"].set(0)
            self.container_var[key_setting]["Inclusion Concentration Calculation"] = tk.IntVar()
            self.container_var[key_setting]["Inclusion Concentration Calculation"].set(0)

            for key, item in self.container_var[key_setting]["Inclusion Plugin"].items():
                item.set(1)

            str_opt_01 = self.language_dict["Set salinity"][self.var_language]

            self.molar_masses_compounds = {}
            self.container_var[key_setting]["Salt Correction"] = {
                "Chlorides": {}, "Carbonates": {}, "Sulfates": {}, "Salinity": tk.StringVar(),
                "Default Salinity": tk.StringVar(), "Default Concentration": tk.StringVar(),
                "Default IS": tk.StringVar(), "Salinity SMPL": {}}
            self.container_var[key_setting]["Salt Correction"]["Salinity"].set("0.0")
            self.container_var[key_setting]["Salt Correction"]["Default Salinity"].set(str_opt_01 + " (%)")
            self.container_var[key_setting]["Salt Correction"]["Default Concentration"].set("0.0")
            self.container_var[key_setting]["Salt Correction"]["Default IS"].set(str_var_03)
            salt_chlorides = ["NaCl", "KCl", "CaCl2", "MgCl2", "LiCl", "FeCl2"]
            salt_carbonates = ["Na2CO3", "K2CO3", "NaHCO3"]
            salt_sulfates = ["Na2SO4"]
            for salt in salt_chlorides:
                self.container_var[key_setting]["Salt Correction"]["Chlorides"][salt] = {
                    "State": tk.IntVar(), "Weight": tk.StringVar()}
                if salt == "NaCl":
                    self.container_var[key_setting]["Salt Correction"]["Chlorides"][salt]["State"].set(1)
                    self.container_var[key_setting]["Salt Correction"]["Chlorides"][salt]["Weight"].set("1.0")
                else:
                    self.container_var[key_setting]["Salt Correction"]["Chlorides"][salt]["State"].set(0)
                    self.container_var[key_setting]["Salt Correction"]["Chlorides"][salt]["Weight"].set("1.0")
            for salt in salt_carbonates:
                self.container_var[key_setting]["Salt Correction"]["Carbonates"][salt] = {
                    "State": tk.IntVar(), "Weight": tk.StringVar()}
                self.container_var[key_setting]["Salt Correction"]["Carbonates"][salt]["State"].set(0)
                self.container_var[key_setting]["Salt Correction"]["Carbonates"][salt]["Weight"].set("1.0")
            for salt in salt_sulfates:
                self.container_var[key_setting]["Salt Correction"]["Sulfates"][salt] = {
                    "State": tk.IntVar(), "Weight": tk.StringVar()}
                self.container_var[key_setting]["Salt Correction"]["Sulfates"][salt]["State"].set(0)
                self.container_var[key_setting]["Salt Correction"]["Sulfates"][salt]["Weight"].set("1.0")
            self.container_var[key_setting]["Data Type Plot"] = {"STD": {}, "SMPL": {}}
            self.container_var[key_setting]["Analyse Mode Plot"] = {"STD": {}, "SMPL": {}}
            self.container_var[key_setting]["Display RAW"] = {"STD": {}, "SMPL": {}}
            self.container_var[key_setting]["Display SMOOTHED"] = {"STD": {}, "SMPL": {}}
            self.container_var[key_setting]["Time-Signal Lines"] = {"STD": {}, "SMPL": {}}
            self.container_var[key_setting]["Time-Ratio Lines"] = {"STD": {}, "SMPL": {}}
            self.container_var[key_setting]["Checkboxes Isotope Diagram"] = {"STD": {}, "SMPL": {}}
            self.container_var[key_setting]["Calculation Interval"] = {"STD": {}, "SMPL": {}}
            self.container_var[key_setting]["Calculation Interval Visibility"] = {"STD": {}, "SMPL": {}}

        self.oxide_calculation_mat = tk.IntVar()
        self.oxide_calculation_incl = tk.IntVar()
        self.oxide_calculation_mat.set(0)
        self.oxide_calculation_incl.set(0)

        self.projectname = None

        self.container_var["fi_datareduction_isotopes"]["File Type"] = tk.IntVar()  # e.g. Sample files
        self.container_var["fi_datareduction_isotopes"]["File Type"].set(1)
        self.container_var["fi_datareduction_isotopes"]["Data Type"] = tk.IntVar()  # e.g. RAW data
        self.container_var["fi_datareduction_isotopes"]["Data Type"].set(0)
        self.container_var["fi_datareduction_isotopes"]["Focus"] = tk.IntVar()  # e.g. Inclusion
        self.container_var["fi_datareduction_isotopes"]["Focus"].set(1)
        self.container_var["fi_datareduction_isotopes"]["Result Category"] = tk.IntVar()  # e.g. Concentration
        self.container_var["fi_datareduction_isotopes"]["Result Category"].set(0)
        self.container_var["fi_datareduction_isotopes"]["Display Option"] = tk.IntVar()  # e.g. Table mode
        self.container_var["fi_datareduction_isotopes"]["Display Option"].set(0)
        self.container_var["fi_datareduction_isotopes"]["Isotope Left Plot"] = tk.StringVar()  # e.g. Si29
        self.container_var["fi_datareduction_isotopes"]["Isotope Left Plot"].set("Select Isotope")
        self.container_var["fi_datareduction_isotopes"]["Isotope 1 Right Plot"] = tk.StringVar()  # e.g. Si29
        self.container_var["fi_datareduction_isotopes"]["Isotope 1 Right Plot"].set("Select Isotope")
        self.container_var["fi_datareduction_isotopes"]["Isotope 2 Right Plot"] = tk.StringVar()  # e.g. Si29
        self.container_var["fi_datareduction_isotopes"]["Isotope 2 Right Plot"].set("Select Isotope")
        self.container_var["fi_datareduction_isotopes"]["Sensitivity Diagrams"] = {
            "Left": {"FIG": None, "CANVAS": None, "AX": None}, "Right": {"FIG": None, "CANVAS": None, "AX": None}}
        self.container_var["fi_datareduction_isotopes"]["Sensitivity Plots"] = {
            "Sensitivity Drift": {"FIG": None, "CANVAS": None, "AX": None},
            "Sensitivity Drift Change": {"FIG": None, "CANVAS": None, "AX": None},
            "Sensitivity Comparison": {"FIG": None, "CANVAS": None, "AX": None},
            "Histograms": {"FIG": None, "CANVAS": None, "AX": None},
            "Box Plots": {"FIG": None, "CANVAS": None, "AX": None}}

        self.container_var["fi_datareduction_files"]["File Type"] = tk.IntVar()  # e.g. Sample files
        self.container_var["fi_datareduction_files"]["File Type"].set(1)
        self.container_var["fi_datareduction_files"]["Data Type"] = tk.IntVar()  # e.g. RAW data
        self.container_var["fi_datareduction_files"]["Data Type"].set(0)
        self.container_var["fi_datareduction_files"]["Focus"] = tk.IntVar()  # e.g. Inclusion
        self.container_var["fi_datareduction_files"]["Focus"].set(1)
        self.container_var["fi_datareduction_files"]["Result Category"] = tk.IntVar()  # e.g. Concentration
        self.container_var["fi_datareduction_files"]["Result Category"].set(0)
        self.container_var["fi_datareduction_files"]["Display Option"] = tk.IntVar()  # e.g. Table mode
        self.container_var["fi_datareduction_files"]["Display Option"].set(0)
        self.container_var["fi_datareduction_files"]["Isotope Left Plot"] = tk.StringVar()  # e.g. Si29
        self.container_var["fi_datareduction_files"]["Isotope Left Plot"].set("Select Isotope")
        self.container_var["fi_datareduction_files"]["Isotope 1 Right Plot"] = tk.StringVar()  # e.g. Si29
        self.container_var["fi_datareduction_files"]["Isotope 1 Right Plot"].set("Select Isotope")
        self.container_var["fi_datareduction_files"]["Isotope 2 Right Plot"] = tk.StringVar()  # e.g. Si29
        self.container_var["fi_datareduction_files"]["Isotope 2 Right Plot"].set("Select Isotope")
        self.container_var["fi_datareduction_files"]["Sensitivity Diagrams"] = {
            "Left": {"FIG": None, "CANVAS": None, "AX": None}, "Right": {"FIG": None, "CANVAS": None, "AX": None}}

        self.container_var["fi_datareduction"] = {}
        self.container_var["fi_dataexploration"] = {}
        self.var_mode_fi = False

        self.container_intervals = {}

        self.container_var["mi_setting"]["Time BG Start"] = tk.StringVar()
        self.container_var["mi_setting"]["Time BG Start"].set("Set start time")
        self.container_var["mi_setting"]["Time BG End"] = tk.StringVar()
        self.container_var["mi_setting"]["Time BG End"].set("Set end time")
        self.container_var["mi_setting"]["Time MAT Start"] = tk.StringVar()
        self.container_var["mi_setting"]["Time MAT Start"].set("Set start time")
        self.container_var["mi_setting"]["Time MAT End"] = tk.StringVar()
        self.container_var["mi_setting"]["Time MAT End"].set("Set end time")
        self.container_var["mi_setting"]["Time INCL Start"] = tk.StringVar()
        self.container_var["mi_setting"]["Time INCL Start"].set("Set start time")
        self.container_var["mi_setting"]["Time INCL End"] = tk.StringVar()
        self.container_var["mi_setting"]["Time INCL End"].set("Set end time")
        self.container_var["mi_setting"]["Author"] = tk.StringVar()
        self.container_var["mi_setting"]["Source ID"] = tk.StringVar()
        self.container_var["mi_setting"]["Source ID"].set("RUR01")
        self.container_var["mi_datareduction"] = {}
        self.container_var["mi_dataexploration"] = {}
        self.container_var["ma_datareduction"]["Radiobutton"] = [
            tk.IntVar(), tk.IntVar(), tk.IntVar(), tk.IntVar(), tk.IntVar()]

        for var_rb in self.container_var["ma_datareduction"]["Radiobutton"]:
            var_rb.set(0)

        self.var_mode_mi = False

        self.container_var["Radiobuttons"] = {"Quick Plot File": tk.IntVar()}
        self.container_var["Radiobuttons"]["Quick Plot File"].set(1)
        self.container_var["Checkboxes"] = {
            "Quick Plot File BG": tk.IntVar(), "Quick Plot File MAT": tk.IntVar(), "Quick Plot File INCL": tk.IntVar()}
        for key_cb in ["Quick Plot File BG", "Quick Plot File MAT", "Quick Plot File INCL"]:
            self.container_var["Checkboxes"][key_cb].set(1)
        self.container_var["mineralchemistry"] = []
        self.container_var["plotting"] = {}
        self.container_var["IS"] = {}
        self.container_var["IS"]["Default STD"] = tk.StringVar()
        self.container_var["IS"]["Default STD"].set(str_var_03)
        self.container_var["IS"]["Default SMPL"] = tk.StringVar()
        self.container_var["IS"]["Default SMPL"].set(str_var_03)
        self.container_var["IS"]["Default SMPL Concentration"] = tk.StringVar()
        self.container_var["IS"]["Default SMPL Concentration"].set("1000000")
        self.container_var["ID"] = {}
        self.container_var["ID"]["Default SMPL"] = tk.StringVar()
        self.container_var["ID"]["Default SMPL"].set("A")
        self.container_var["ID"]["Results Isotopes"] = tk.StringVar()
        self.container_var["ID"]["Results Isotopes"].set("A")
        self.container_var["ID"]["Results Files"] = tk.StringVar()
        self.container_var["ID"]["Results Files"].set("A")

        for type in ["STD", "SMPL"]:
            if type not in self.container_var:
                self.container_var[type] = {}
                self.container_var[type]["Project manager"] = tk.IntVar()
                self.container_var[type]["Project manager"].set(0)

        self.container_var["LASER"] = tk.StringVar()
        self.container_var["LASER"].set("Argon")
        self.container_var["charge"] = {}
        self.container_var["Gas Energy"] = tk.StringVar()
        self.container_var["Gas Energy"].set("15.760")
        self.container_var["Spike Elimination Method"] = tk.StringVar()
        self.container_var["Spike Elimination Method"].set("Grubbs test")
        self.list_se_methods = ["Grubbs test", "Whisker analysis", "Z-score analysis"]
        self.list_isotopes = []
        self.srm_actual = {}
        self.container_files = {}
        self.container_files["STD"] = {}
        self.container_files["SRM"] = {}
        self.container_files["SMPL"] = {}
        self.container_optionmenu = {}
        self.container_optionmenu["STD"] = {}
        self.container_optionmenu["SMPL"] = {}
        self.container_optionmenu["ISOTOPES"] = {}
        self.container_measurements = {}
        categories_01 = ["RAW", "SELECTED", "EDITED"]

        for category_01 in categories_01:
            self.container_measurements[category_01] = {}

        self.mineral_chem = {}
        self.time_ratio_data = {}

        self.last_rb_setup = {}

        self.container_helper = {}
        self.container_helper["STD"] = {}
        self.container_helper["SMPL"] = {}
        self.container_helper["positions"] = {}
        self.container_helper["positions"]["BG STD"] = {}
        self.container_helper["positions"]["SIG STD"] = {}
        self.container_helper["positions"]["MAT STD"] = {}
        self.container_helper["positions"]["INCL STD"] = {}
        self.container_helper["positions"]["SPK STD"] = {}
        self.container_helper["positions"]["BG SMPL"] = {}
        self.container_helper["positions"]["SIG SMPL"] = {}
        self.container_helper["positions"]["MAT SMPL"] = {}
        self.container_helper["positions"]["INCL SMPL"] = {}
        self.container_helper["positions"]["SPK SMPL"] = {}
        self.container_helper["positions"]["BG"] = {}
        self.container_helper["positions"]["SIG"] = {}
        self.container_helper["positions"]["SPK"] = {}
        self.container_helper["positions"]["MAT"] = {}
        self.container_helper["positions"]["INCL"] = {}
        self.container_helper["indices"] = {}
        self.container_helper["limits BG"] = {}
        self.container_helper["limits SIG"] = {}
        self.container_helper["limits MAT"] = {}
        self.container_helper["limits INCL"] = {}
        self.container_helper["limits SPK"] = {}
        self.container_helper["limits BG Ratio"] = {}
        self.container_helper["limits SIG Ratio"] = {}
        self.container_helper["limits MAT Ratio"] = {}
        self.container_helper["limits INCL Ratio"] = {}
        self.container_helper["limits SPK Ratio"] = {}
        self.container_helper["Default BG"] = {}
        self.container_helper["Default BG"]["Times"] = [0, 0]
        self.container_helper["Default BG"]["Positions"] = [0, 0]
        self.container_helper["Default SIG"] = {}
        self.container_helper["Default SIG"]["Times"] = [0, 0]
        self.container_helper["Default SIG"]["Positions"] = [0, 0]

        self.container_lists = {}

        self.container_lists["Colors PySILLS"] = {}
        categories = ["Dark", "Medium", "Light", "Sign"]
        subcategories = ["Green", "Red", "Blue", "Yellow", "Brown", "Slate Grey"]

        for category in categories:
            self.container_lists["Colors PySILLS"][category] = {}
            for subcategory in subcategories:
                if category != "Sign":
                    if subcategory == "Green":
                        self.container_lists["Colors PySILLS"][category][subcategory] = green_dict[category]
                    elif subcategory == "Red":
                        self.container_lists["Colors PySILLS"][category][subcategory] = red_dict[category]
                    elif subcategory == "Blue":
                        self.container_lists["Colors PySILLS"][category][subcategory] = blue_dict[category]
                    elif subcategory == "Yellow":
                        self.container_lists["Colors PySILLS"][category][subcategory] = yellow_dict[category]
                    elif subcategory == "Brown":
                        self.container_lists["Colors PySILLS"][category][subcategory] = brown_dict[category]
                    elif subcategory == "Slate Grey":
                        self.container_lists["Colors PySILLS"][category][subcategory] = slategrey_dict[category]
                else:
                    if subcategory in ["Red", "Yellow", "Green"]:
                        self.container_lists["Colors PySILLS"][category][subcategory] = sign_dict[subcategory]

        self.container_lists["Selected Cations"] = []
        self.container_lists["Selected Anions"] = []
        self.container_lists["Possible Cations"] = []
        self.container_lists["Possible Anions"] = []
        self.container_lists["Salt Chemistry"] = {}
        self.container_lists["Selected Salts"] = []
        self.container_lists["SRM Data"] = {}
        self.container_lists["SRM"] = []
        self.container_lists["SRM Library"] = []
        self.container_lists["ICPMS Library"] = []
        self.container_lists["SRM Files"] = {}
        self.container_lists["SRM Isotopes"] = {}
        self.container_lists["IS"] = []
        self.container_lists["ID"] = []
        self.container_lists["Possible IS"] = []
        self.container_lists["ID Files"] = {}
        self.container_lists["STD"] = {}
        self.container_lists["STD"]["Long"] = []
        self.container_lists["STD"]["Short"] = []
        self.container_lists["STD"]["Old"] = []
        self.container_lists["STD"]["SRM"] = {}
        self.container_lists["SMPL"] = {}
        self.container_lists["SMPL"]["Long"] = []
        self.container_lists["SMPL"]["Short"] = []
        self.container_lists["SMPL"]["Old"] = []
        self.container_lists["ISOTOPES"] = []
        self.container_lists["Measured Isotopes"] = {}
        self.container_lists["Measured Isotopes"]["All"] = []
        self.container_lists["Possible Oxides"] = {}
        self.container_lists["Possible Oxides"]["All"] = []
        self.container_lists["Measured Elements"] = {}
        self.container_lists["Measured Elements"]["All"] = []
        self.container_lists["Acquisition Times Delta"] = {}
        self.container_lists["Analytical Sensitivity Regression"] = {}
        self.container_lists["Analytical Sensitivity Regression RAW"] = {}
        self.container_lists["Analytical Sensitivity Regression SMOOTHED"] = {}
        self.container_lists["Possible IS STD"] = []
        self.container_lists["Possible IS SMPL"] = []
        self.container_lists["Plugins FI"] = {}  # FI = Fluid Inclusion
        self.container_lists["Plugins FI"]["Names"] = []
        self.container_lists["Plugins FI"]["Files"] = []
        self.container_lists["Plugins MI"] = {}  # MI = Melt Inclusion
        self.container_lists["Plugins MI"]["Names"] = []
        self.container_lists["Plugins MI"]["Files"] = []
        self.container_lists["Plugins SE"] = {}  # SE = Spike Elimination
        self.container_lists["Plugins SE"]["Names"] = []
        self.container_lists["Plugins SE"]["Files"] = []

        self.helper_checkbuttons = {"Isotopes": {}, "On": [], "Off": []}
        self.temp_checkbuttons_pypitzer = {}

        self.container_lists["Oxides"] = []
        self.container_lists["Sulfides"] = [
            "FeS2", "ZnS" "PbS", "Ag2S", "Na2S", "MoS2", "CdS", "SeS2", "BaS", "BeS", "CoS", "Cu2S", "CuS", "FeS",
            "Fe2S3", "Hg2S", "HgS", "MnS", "NiS", "Tl2S", "SnS", "SnS2"]
        self.container_lists["Halides"] = [
            "NaCl", "KCl", "KI", "LiCl", "CuCl2", "AgCl", "CaCl2", "ClF", "CH3Br", "CHI3", "HCl", "HBr"]
        self.container_lists["Elements"] = []
        self.container_lists["Selected Oxides"] = {"All": [], "Conversion Factor": {"SiO2", 2.1392}}

        self.maximum_amounts = {}

        for oxide in self.conversion_factors.keys():
            key = re.search(r"(\D+)(\d*)(\D+)(\d*)", oxide)
            element_cation = key.group(1)

            if key.group(2) != "":
                index_cation = int(key.group(2))
            else:
                index_cation = 1

            if key.group(4) != "":
                index_anion = int(key.group(4))
            else:
                index_anion = 1

            molar_mass_cation = self.chemistry_data[element_cation]
            molar_mass_anion = 15.999
            molar_mass_oxide = index_cation*molar_mass_cation + index_anion*molar_mass_anion
            amount_cation_ppm = round((index_cation*molar_mass_cation)/molar_mass_oxide*10**6, 4)

            if element_cation in self.maximum_amounts:
                if amount_cation_ppm > self.maximum_amounts[element_cation]:
                    self.maximum_amounts[element_cation] = amount_cation_ppm
            else:
                self.maximum_amounts[element_cation] = amount_cation_ppm

        self.mineral_chemistry = {}
        list_minerals = ["Quartz", "Chloro-Apatite", "Fluor-Apatite", "Hydroxy-Apatite"]
        for mineral in list_minerals:
            self.mineral_chemistry[mineral] = {}
            if mineral == "Quartz":  # SiO2
                mass_si = self.chemistry_data["Si"]
                mass_o = self.chemistry_data["O"]
                mass_total = 1*mass_si + 2*mass_o
                w_si = 1*mass_si/mass_total
                self.mineral_chemistry[mineral]["Si"] = round(w_si, 10)
            elif mineral == "Chloro-Apatite":  # Ca5 (PO4)3 Cl
                mass_ca = self.chemistry_data["Ca"]
                mass_p = self.chemistry_data["P"]
                mass_cl = self.chemistry_data["Cl"]
                mass_o = self.chemistry_data["O"]
                mass_total = 5*mass_ca + 3*(1*mass_p + 4*mass_o) + 1*mass_cl
                w_ca = 5*mass_ca/mass_total
                w_p = 3*mass_p/mass_total
                w_cl = 1*mass_cl/mass_total
                self.mineral_chemistry[mineral]["Ca"] = round(w_ca, 10)
                self.mineral_chemistry[mineral]["P"] = round(w_p, 10)
                self.mineral_chemistry[mineral]["Cl"] = round(w_cl, 10)
            elif mineral == "Fluor-Apatite":  # Ca5 (PO4)3 F
                mass_ca = self.chemistry_data["Ca"]
                mass_p = self.chemistry_data["P"]
                mass_f = self.chemistry_data["F"]
                mass_o = self.chemistry_data["O"]
                mass_total = 5*mass_ca + 3*(1*mass_p + 4*mass_o) + 1*mass_f
                w_ca = 5*mass_ca/mass_total
                w_p = 3*mass_p/mass_total
                self.mineral_chemistry[mineral]["Ca"] = round(w_ca, 10)
                self.mineral_chemistry[mineral]["P"] = round(w_p, 10)
            elif mineral == "Hydroxy-Apatite":  # Ca5 (PO4)3 (OH)
                mass_ca = self.chemistry_data["Ca"]
                mass_p = self.chemistry_data["P"]
                mass_h = self.chemistry_data["H"]
                mass_o = self.chemistry_data["O"]
                mass_total = 5*mass_ca + 3*(1*mass_p + 4*mass_o) + 1*(mass_h + mass_o)
                w_ca = 5*mass_ca/mass_total
                w_p = 3*mass_p/mass_total
                self.mineral_chemistry[mineral]["Ca"] = round(w_ca, 10)
                self.mineral_chemistry[mineral]["P"] = round(w_p, 10)

        self.container_results = {}
        self.container_results["STD"] = {}
        self.container_results["STD"]["RAW"] = {}
        self.container_results["STD"]["SMOOTHED"] = {}
        self.container_results["SMPL"] = {}
        self.container_results["SMPL"]["RAW"] = {}
        self.container_results["SMPL"]["SMOOTHED"] = {}
        self.xi_std_time = {}
        self.xi_regr = None

        self.demo_view = False

        self.fast_track_std = False
        self.fast_track_smpl = False

        self.file_loaded = False

        self.spikes_isotopes = {}

        self.diagrams_setup = {}
        categories_01 = ["STD", "SMPL"]

        for category_01 in categories_01:
            self.diagrams_setup[category_01] = {}

        self.container_diagrams = {}
        self.container_listboxes = {}
        categories_01 = ["STD", "SMPL"]

        for category_01 in categories_01:
            self.container_diagrams[category_01] = {}
            self.container_listboxes[category_01] = {}
            self.spikes_isotopes[category_01] = {}

        # Calculation Report
        self.container_report = {}
        categories_01 = ["Total STD", "Total SMPL"]
        categories_02 = ["intensity bg", "intensity sig corr", "intensity ratio", "sensitivity", "concentration", "RSF",
                         "LOD"]
        for category_01 in categories_01:
            self.container_report[category_01] = {}
            self.container_report[category_01]["Mean"] = {}
            self.container_report[category_01]["Error"] = {}
            for category_02 in categories_02:
                self.container_report[category_01]["Mean"][category_02] = {}
                self.container_report[category_01]["Error"][category_02] = {}
                self.container_report[category_01]["Mean"][category_02]["filename"] = category_01
                self.container_report[category_01]["Error"][category_02]["filename"] = category_01

        self.list_srm = np.array(
            [["NIST 606"], ["NIST 610"], ["NIST 610 (GeoReM)"], ["NIST 610 (Spandler)"], ["NIST 611"],
             ["NIST 611 (GeoReM)"], ["NIST 612"], ["NIST 612 (GeoReM)"], ["NIST 613"], ["NIST 613 (GeoReM)"],
             ["NIST 614"], ["NIST 614 (GeoReM)"], ["NIST 615"], ["NIST 615 (GeoReM)"], ["NIST 616"],
             ["NIST 616 (GeoReM)"], ["NIST 617"], ["NIST 617 (GeoReM)"], ["USGS BCR-2G (GeoReM)"],
             ["USGS GSD-1G (GeoReM)"], ["USGS GSE-1G (GeoReM)"], ["B6"], ["Durango Apatite"], ["Scapolite 17"],
             ["BAM-376"], ["BCR-2G"], ["BL-Q"], ["Br-Glass"], ["GSD-1G (GeoReM)"], ["GSE-1G (GeoReM)"], ["GSE-2G"],
             ["HAL-O"], ["K-Br"], ["MACS-3"], ["Po 724"], ["STDGL-2B2"], ["OU-6 (GeoReM)"]])[:, 0]

        self.path_pysills = os.path.dirname(os.path.realpath(sys.argv[0]))
        self.path_pysills_main = self.path_pysills

        if "/pysills" in self.path_pysills_main:
            self.path_pysills_main = self.path_pysills_main.replace("/pysills", "")
        elif "/tests" in self.path_pysills_main:
            self.path_pysills_main = self.path_pysills_main.replace("/tests", "")
        elif "/bin" in self.path_pysills_main:
            self.path_pysills_main = os.path.abspath(__file__)
            self.path_pysills_main = self.path_pysills_main.replace("/pysills_app.py", "")
        elif "Scripts" in self.path_pysills_main:
            self.path_pysills_main = os.path.abspath(__file__)
            self.path_pysills_main = self.path_pysills_main.replace("pysills_app.py", "")

        helper_srm_library = []

        if "pysills" in self.path_pysills_main:
            self.path_pysills_main = self.path_pysills_main.replace("pysills", "")

        if "\\" in self.path_pysills_main:
            self.path_pysills_main = self.path_pysills_main[:-1]

        folder_path = os.path.join(self.path_pysills_main, "pysills", "lib", "srm")
        folder_path_translations = os.path.join(self.path_pysills_main, "pysills", "lib", "translations", "")
        helper_srm_library = os.listdir(folder_path)

        for language in ["Chinese"]:
            if language == "Chinese":
                var_ending = ".txt"
            else:
                var_ending = ".csv"

            path_language = folder_path_translations + language + var_ending

            if language == "Chinese":
                var_encoding = "utf-8"
            else:
                var_encoding = "utf-8"
            with open(path_language, mode="r", newline="", encoding=var_encoding) as file:
                reader = csv.reader(file)
                for row in reader:
                    if language == "Chinese":
                        key = row[0].split("\t")[0]
                        translation = row[0].split("\t")[-1]
                        self.language_dict[key][language] = translation

        helper_srm_library.remove("__init__.py")

        try:
            helper_srm_library.remove(".DS_Store")
        except:
            pass

        helper_srm_library.sort()

        for var_srm in helper_srm_library:
            if ".csv" in var_srm:
                var_srm_new = var_srm.replace("_", " ")
                var_srm_new = var_srm_new.replace(".csv", "")
                if "GeoReM" in var_srm_new:
                    var_srm_new = var_srm_new.replace("GeoReM", "(GeoReM)")
                if "Spandler" in var_srm_new:
                    var_srm_new = var_srm_new.replace("Spandler", "(Spandler)")
                if "BAM 376" in var_srm_new:
                    var_srm_new = var_srm_new.replace("BAM 376", "BAM-376")
                if "BCR 2G" in var_srm_new:
                    var_srm_new = var_srm_new.replace("BCR 2G", "BCR-2G")
                if "BL Q" in var_srm_new:
                    var_srm_new = var_srm_new.replace("BL Q", "BL-Q")
                if "Br Glass" in var_srm_new:
                    var_srm_new = var_srm_new.replace("Br Glass", "Br-Glass")
                if " 1G" in var_srm_new:
                    var_srm_new = var_srm_new.replace(" 1G", "-1G")
                if " 2G" in var_srm_new:
                    var_srm_new = var_srm_new.replace(" 2G", "-2G")
                if " 2B2" in var_srm_new:
                    var_srm_new = var_srm_new.replace(" 2B2", "-2B2")
                if "HAL O" in var_srm_new:
                    var_srm_new = var_srm_new.replace("HAL O", "HAL-O")
                if "K Br" in var_srm_new:
                    var_srm_new = var_srm_new.replace("K Br", "K-Br")
                if "MACS 3" in var_srm_new:
                    var_srm_new = var_srm_new.replace("MACS 3", "MACS-3")
                if "BCR2G" in var_srm_new:
                    var_srm_new = var_srm_new.replace("BCR2G", "BCR-2G")
                if "GSD1G" in var_srm_new:
                    var_srm_new = var_srm_new.replace("GSD1G", "GSD-1G")
                if "GSE1G" in var_srm_new:
                    var_srm_new = var_srm_new.replace("GSE1G", "GSE-1G")
                if "OU 6" in var_srm_new:
                    var_srm_new = var_srm_new.replace("OU 6", "OU-6")

                self.container_lists["SRM Library"].append(var_srm_new)

        helper_icpms_library = []

        if "site-packages" in self.path_pysills_main:
            folder_path = os.path.join(self.path_pysills_main, "pysills", "lib", "icpms")
            helper_icpms_library = os.listdir(folder_path)
        else:
            try:
                folder_path = os.path.join(self.path_pysills_main, "lib", "icpms")
                helper_icpms_library = os.listdir(folder_path)
            except:
                folder_path = os.path.join(self.path_pysills_main, "pysills", "lib", "icpms")
                helper_icpms_library = os.listdir(folder_path)

        if "__init__.py" in helper_icpms_library:
            helper_icpms_library.remove("__init__.py")
        if ".DS_Store" in helper_icpms_library:
            helper_icpms_library.remove(".DS_Store")

        try:
            helper_icpms_library.remove(".DS_Store")
        except:
            pass

        helper_icpms_library.sort()

        for var_icpms in helper_icpms_library:
            if ".csv" in var_icpms:
                var_icpms_new = var_icpms.replace("_", " ")
                var_icpms_new = var_icpms_new.replace(".csv", "")
                self.container_lists["ICPMS Library"].append(var_icpms_new)

        self.srm_window_checker = tk.IntVar()
        self.srm_window_checker.set(0)
        self.previous_std_list = []
        self.list_pse = PSE().get_element_names()
        self.ionization_energies = {}
        self.ionization_energies["First"] = {}
        self.ionization_energies["Second"] = {}

        for element in self.list_pse:
            data_element = PeriodicSystem(name=element).get_data()
            if len(data_element) == 13:
                self.ionization_energies["First"][element] = data_element[-2]
                self.ionization_energies["Second"][element] = data_element[-1]
            elif len(data_element) == 12:
                self.ionization_energies["First"][element] = data_element[-1]
                self.ionization_energies["Second"][element] = 0.0

        self.container_settings = {}
        categories_01 = ["MA", "FI", "MI"]  # MA=Mineral Analysis, FI=Fluid Inclusions, MI=Melt Inclusions
        categories_02 = ["Start BG", "End BG", "Start SIG", "End SIG", "Start MAT", "End MAT", "Start INCL", "End INCL"]
        categories_03 = ["Deviation", "Threshold", "Author", "Source ID"]

        for category_01 in categories_01:
            self.container_settings[category_01] = {}
            for category_02 in categories_02:
                if category_01 == "MA" and category_02 in ["Start MAT", "End MAT", "Start INCL", "End INCL"]:
                    pass
                elif category_01 in ["FI", "MI"] and category_02 in ["Start SIG", "End SIG"]:
                    pass
                else:
                    self.container_settings[category_01][category_02] = tk.StringVar()
            for category_03 in categories_03:
                self.container_settings[category_01][category_03] = tk.StringVar()

        self.mineral_list = ["Quartz", "Calcite", "Fluorite", "Apatite-Cl", "Apatite-F", "Apatite-OH", "Forsterite",
                             "Fayalite", "Tephroite", "Albite", "Anorthite", "Orthoclase", "Microcline", "Zircon",
                             "Enargite", "Pyrite", "Chalcopyrite", "Bornite", "Arsenopyrite", "Sphalerite", "Galena",
                             "Molybdenite", "Hematite", "Magnetite", "Chromite", "Gahnite", "Meionite", "Marialite",
                             "Strontianite", "Titanite", "Aegirine", "Diopside", "Hedenbergite", "Ferrosilite",
                             "Enstatite", "Monazite-La", "Monazite-Ce", "Monazite-Pr", "Monazite-Nd", "Monazite-Sm",
                             "Monazite-Eu", "Monazite-Gd", "Monazite-Th", "Barite", "Ilvaite"]
        self.mineral_list.sort()
        self.container_lists["Minerals"] = self.mineral_list

        self.calculate_mineral_chemistry()
        self.create_srm_data_list()

        ## USER SETTINGS
        project_path_prew = self.path_pysills[:]
        project_path = project_path_prew.strip("/pysills")

        self.var_opt_icp = tk.StringVar()
        str_opt_icpms = self.language_dict["Select ICP-MS"][self.var_language]
        self.var_opt_icp.set(str_opt_icpms)

        try:
            file_usersettings = open(self.path_pysills_main + str("/user_settings.txt"), "r")
            for index, file_data in enumerate(file_usersettings):
                file_data_splitted = file_data.split(";")

                try:
                    self.container_var["General Settings"][file_data_splitted[0]].set(file_data_splitted[1])
                except:
                    pass

                if "Dwell Times" in file_data_splitted:
                    self.container_var["dwell_times"]["Entry"]["Default"].set(file_data_splitted[1])

                if "Default ICP-MS" in file_data_splitted:
                    self.var_opt_icp.set(file_data_splitted[1])

                    str_icpms = file_data_splitted[1]
                    var_instrument = str_icpms.replace(" ", "_")
                    var_instrument_raw = str_icpms

                    if var_instrument_raw != "Select ICP-MS":
                        try:
                            file_long = self.path_pysills_main + str("/lib/icpms/" + var_instrument + ".csv")
                            file_content = open(file_long)
                        except:
                            file_long = self.path_pysills_main + str("/pysills/lib/icpms/" + var_instrument + ".csv")
                            file_content = open(file_long)

                        for index, line in enumerate(file_content):
                            line_parts = line.split(",")
                            if "\n" in line_parts[-1]:
                                line_parts[-1] = line_parts[-1].replace("\n", "")
                            try:
                                self.container_icpms[line_parts[0]] = int(line_parts[-1])
                            except:
                                self.container_icpms[line_parts[0]] = line_parts[-1]
        except:
            print("There is no user_settings file!")

        # Screen resolution
        str_screen_resolution = self.container_var["General Settings"]["Screen resolution"].get()
        if str_screen_resolution == "1920x1080":
            self.row_height = 24
            self.column_width = 21

            if var_os == "linux":
                self.font_settings = {
                    "Header": "sans 12 bold", "Elements": "sans 10 bold", "Small": "sans 7 bold", "Options": "sans 10",
                    "Table": "sans 10", "Tiny": "sans 7 bold"}
            elif var_os == "darwin":
                self.font_settings = {
                    "Header": "sans 16 bold", "Elements": "sans 14 bold", "Small": "sans 10 bold", "Options": "sans 14",
                    "Table": "sans 14", "Tiny": "sans 10 bold"}
            else:
                self.font_settings = {
                    "Header": "sans 12 bold", "Elements": "sans 10 bold", "Small": "sans 8 bold", "Options": "sans 10",
                    "Table": "sans 10", "Tiny": "sans 8 bold"}
        elif str_screen_resolution == "1280x720":
            self.row_height = 16
            self.column_width = 14

            if var_os == "linux":
                self.font_settings = {
                    "Header": "sans 8 bold", "Elements": "sans 7 bold", "Small": "sans 5 bold", "Options": "sans 7",
                    "Table": "sans 7", "Tiny": "sans 5 bold"}
            elif var_os == "darwin":
                self.font_settings = {
                    "Header": "sans 11 bold", "Elements": "sans 9 bold", "Small": "sans 7 bold", "Options": "sans 9",
                    "Table": "sans 9", "Tiny": "sans 5 bold"}
            else:
                self.font_settings = {
                    "Header": "sans 8 bold", "Elements": "sans 7 bold", "Small": "sans 5 bold", "Options": "sans 7",
                    "Table": "sans 7", "Tiny": "sans 5 bold"}
        elif str_screen_resolution == "3840x2160":
            self.row_height = 32
            self.column_width = 28

            if var_os == "linux":
                self.font_settings = {
                    "Header": "sans 16 bold", "Elements": "sans 14 bold", "Small": "sans 14 bold", "Options": "sans 14",
                    "Table": "sans 12", "Tiny": "sans 12 bold"}
            elif var_os == "darwin":
                self.font_settings = {
                    "Header": "sans 18 bold", "Elements": "sans 16 bold", "Small": "sans 14 bold", "Options": "sans 16",
                    "Table": "sans 16", "Tiny": "sans 12 bold"}
            else:
                self.font_settings = {
                    "Header": "sans 16 bold", "Elements": "sans 14 bold", "Small": "sans 14 bold", "Options": "sans 14",
                    "Table": "sans 12", "Tiny": "sans 12 bold"}

        str_screen_resolution = self.container_var["General Settings"]["Screen resolution"].get()

        var_ncol_ma_settings = 65
        var_ncol_fi_settings = 67
        var_ncol_mi_settings = 63

        if str_screen_resolution == "1920x1080":    # Full HD
            if var_os == "linux":
                var_ncol_ma_settings = 66
                var_ncol_fi_settings = 68
                var_ncol_mi_settings = 64
            elif var_os == "darwin":
                var_ncol_ma_settings = 65
                var_ncol_fi_settings = 67
                var_ncol_mi_settings = 63
            else:
                var_ncol_ma_settings = 65
                var_ncol_fi_settings = 67
                var_ncol_mi_settings = 63
        elif str_screen_resolution == "1280x720":   # HD ready
            var_ncol_ma_settings = 70
            var_ncol_fi_settings = 72
            var_ncol_mi_settings = 68
        elif str_screen_resolution == "3840x2160":  # 4K
            var_ncol_ma_settings = 65
            var_ncol_fi_settings = 67
            var_ncol_mi_settings = 63

        self.window_dimensions = {
            "Main window": [33, 21], "MA main settings": [38, var_ncol_ma_settings],
            "FI main settings": [40, var_ncol_fi_settings], "MI main settings": [40, var_ncol_mi_settings],
            "ICP-MS setup": [8, 17], "Quick plot": [35, 70], "Spike elimination threshold": [24, 15],
            "Check-up oxides": [29, 83], "Check-up IS": [16, 42], "General settings": [23, 35],
            "Check-up SRM": [23, 32], "Check-up intervals": [32, 54], "Check-up acquisition times": [33, 26],
            "Check-up files": [32, 50], "MA Extras": [8, 25], "FI Extras": [9, 34], "MI Extras": [8, 34],
            "Geothermometry": [16, 40], "Diagram xy": [24, 50], "Diagram halogen ratios": [24, 40],
            "Matrix quantification": [5, 24], "Mineral quantification": [18, 40], "Stepwise analysis": [32, 55],
            "MA specific file": [34, 60], "MA datareduction files": [25, 63],
            "Detailed analysis": [32, 45], "About PySILLS": [22, 32], "Calculation report": [9, 13],
            "Sensitivity drift": [32, 70], "FI datareduction files": [29, 63], "Setup Halter": [24, 34],
            "Setup Borisova": [22, 34], "Setup matrix-only tracer": [15, 41], "Setup 2nd IS": [15, 36],
            "FI specific file": [34, 68], "Setup PyPitzer": [20, 60], "Setup mass balance": [24, 55],
            "Setup external calculation": [11, 33], "Custom spike check": [24, 45], "Popup window error": [12, 20],
            "Popup window progress": [4, 20], "Popup window progress datareduction": [16, 20],
            "Project manager": [16, 50], "Quick Plot Extra": [34, 46], "Specific file setup": [38, 75]}

        row_min = self.row_height
        self.n_rows = 33
        window_height = int(row_min*self.n_rows)
        column_min = self.column_width

        self.n_columns = 21
        window_width = int(column_min*self.n_columns)
        var_geometry = str(window_width) + "x" + str(window_height) + "+0+0"
        self.parent.geometry(var_geometry)

        for x in range(self.n_columns):
            tk.Grid.columnconfigure(self.parent, x, weight=1)
        for y in range(self.n_rows):
            tk.Grid.rowconfigure(self.parent, y, weight=1)

        # Rows
        for i in range(0, self.n_rows):
            self.parent.grid_rowconfigure(i, minsize=row_min)
        # Columns
        for i in range(0, self.n_columns):
            self.parent.grid_columnconfigure(i, minsize=column_min)

        ################################################################################################################

        ## FRAMES

        frame_01 = tk.Frame(self.parent, bg=self.bg_colors["BG Window"], borderwidth=0, highlightthickness=0,
                            relief=tk.FLAT)
        frame_01.grid(row=3, column=0, rowspan=42, columnspan=22, sticky="nesw")
        frame_02 = tk.Frame(self.parent, bg=self.red_dark, borderwidth=0, highlightthickness=0)
        frame_02.grid(row=2, column=0, rowspan=1, columnspan=22, sticky="nesw")

        # ## USER SETTINGS
        str_color_scheme = self.container_var["General Settings"]["Color scheme"].get()
        if str_color_scheme == "Dark scheme":
            self.bg_colors = {
                "BG Window": "#2C2C2C", "Very Dark": "#3C3C3C", "Dark": "#676767", "Medium": "#909090",
                "Light": "#BABABA", "Very Light": "#E3E3E3", "Dark Font": "#292929", "Light Font": "#F7F7F7",
                "White": "#FFFFFF", "Black": "#000000", "Accent": "#E76F51",
                "BG Interval": 0, "MAT Interval": 0, "INCL Interval": 0,
                "BG Light": 0, "MAT Light": 0, "INCL Light": 0}
        elif str_color_scheme == "Light scheme":
            self.bg_colors = {
                "BG Window": "#cccccc", "Very Dark": "#91A1B1", "Dark": "#BAC4CE", "Medium": "#CFD6DD",
                "Light": "#e5e5e5", "Very Light": "#f2f2f2", "Dark Font": "#2C2C2C", "Light Font": "#2C2C2C",
                "White": "#FFFFFF", "Black": "#000000", "Accent": "#bd7966"}
        elif str_color_scheme == "Boho theme 1":
            self.bg_colors = {
                "BG Window": "#463F3A", "Very Dark": "#a3573a", "Dark": "#e5af9e", "Medium": "#e7b7a7",
                "Light": "#BCB8B1", "Very Light": "#F4F3EE", "Dark Font": "#463F3A", "Light Font": "#F4F3EE",
                "White": "#FFFFFF", "Black": "#000000", "Accent": "#E0AFA0"}
        elif str_color_scheme == "Synthwave theme":
            self.bg_colors = {
                "BG Window": "#1C1C3E", "Very Dark": "#3C3C3C", "Dark": "#676767", "Medium": "#909090",
                "Light": "#72419C", "Very Light": "#D3FCFF", "Dark Font": "#1C1C3E", "Light Font": "#F7F7F7",
                "White": "#FFFFFF", "Black": "#000000", "Accent": "#FF49DB"}
        elif str_color_scheme == "Gunmetal theme":
            self.bg_colors = {
                "BG Window": "#223843", "Light": "#B5BEC7", "Very Light": "#EFF1F3", "Dark Font": "#223843",
                "Light Font": "#EFF1F3", "White": "#FFFFFF", "Black": "#000000", "Accent": "#D77A61"}
        elif str_color_scheme == "Dark Jungle":
            self.bg_colors = {
                "BG Window": "#242C1D", "Light": "#C5D0B9", "Very Light": "#F0F3EE", "Dark Font": "#242C1D",
                "Light Font": "#F0F3EE", "White": "#FFFFFF", "Black": "#000000", "Accent": "#582F0E"}
        elif str_color_scheme == "Noble Room":
            self.bg_colors = {
                "BG Window": "#4B3F32", "Light": "#C3B5A5", "Very Light": "#E1DAD2", "Dark Font": "#4B3F32",
                "Light Font": "#E1DAD2", "White": "#FFFFFF", "Black": "#000000", "Accent": "#22333B"}

        ## GUI
        self.pysills_mode = None
        self.demo_mode = True

        self.place_main_window_static_elements()

        ## Initialization
        self.quick_plot_figure_std = Figure(
            figsize=(10, 5), dpi=150, tight_layout=True, facecolor=self.bg_colors["Very Light"])
        self.quick_plot_figure_smpl = Figure(
            figsize=(10, 5), dpi=150, tight_layout=True, facecolor=self.bg_colors["Very Light"])
        self.lb_std.bind("<Double-1>", lambda event, var_filetype="STD": self.quick_plot_file(var_filetype, event))
        self.lb_smpl.bind("<Double-1>", lambda event, var_filetype="SMPL": self.quick_plot_file(var_filetype, event))
        self.select_experiment(var_rb=self.var_rb_mode)

    def place_main_window_static_elements(self):
        # Initial constants
        start_row = 3
        start_column = 0
        n_rows_header = 2
        common_n_rows = 1
        common_n_columns = 10
        n_columns_button = 3
        font_color_dark = self.bg_colors["Dark Font"]
        font_color_light = self.bg_colors["Light Font"]
        background_color_header = self.bg_colors["BG Window"]
        background_color_elements = self.bg_colors["Light"]
        background_color_listbox = self.bg_colors["Very Light"]
        accent_color = self.bg_colors["Accent"]
        font_header = self.font_settings["Header"]
        font_element = self.font_settings["Elements"] # "sans 10 bold"
        font_option = self.font_settings["Options"]
        font_small = self.font_settings["Small"]
        font_tiny = self.font_settings["Tiny"]
        self.var_language = self.container_var["General Settings"]["Language"].get()
        self.update_variables_initial_values()
        self.parent["bg"] = self.bg_colors["BG Window"]

        ## Logo
        try:
            try:
                if self.var_os in ["darwin", "linux"]:
                    if r"/pysills" in self.path_pysills_main:
                        pysills_logo = tk.PhotoImage(file=self.path_pysills_main + str(r"/lib/images/PySILLS_Logo.png"))
                    else:
                        pysills_logo = tk.PhotoImage(
                            file=self.path_pysills_main + str("/pysills/lib/images/PySILLS_Logo.png"))
                else:
                    if r"\pysills" in self.path_pysills_main:
                        pysills_logo = tk.PhotoImage(file=self.path_pysills_main + str(r"\lib\images\PySILLS_Logo.png"))
                    else:
                        pysills_logo = tk.PhotoImage(
                            file=self.path_pysills_main + str(r"\pysills\lib\images\PySILLS_Logo.png"))
            except:
                if self.var_os in ["darwin", "linux"]:
                    pysills_logo = tk.PhotoImage(file=self.path_pysills + str(
                        r"/pysills/lib/images/PySILLS_Logo.png"))
                else:
                    pysills_logo = tk.PhotoImage(file=self.path_pysills + str(
                        r"\pysills\lib/images\PySILLS_Logo.png"))

            pysills_logo = pysills_logo.subsample(1, 1)
            img = tk.Label(self.parent, image=pysills_logo, bg="#2C2C2C")
            img.image = pysills_logo
            img.grid(
                row=start_row - 3, column=start_column, rowspan=n_rows_header, columnspan=common_n_columns + 11,
                sticky="nesw")
        except:
            self.parent.bell()
            print("There is a problem with the PySILLS header image.")

        ## Icon
        try:
            try:
                if self.var_os in ["darwin", "linux"]:
                    if r"/pysills" in self.path_pysills_main:
                        pysills_icon = tk.PhotoImage(file=self.path_pysills_main + str(r"/lib/images/PySILLS_Icon.png"))
                    else:
                        pysills_icon = tk.PhotoImage(
                            file=self.path_pysills_main + str(r"/pysills/lib/images/PySILLS_Icon.png"))
                else:
                    if r"\pysills" in self.path_pysills_main:
                        pysills_icon = tk.PhotoImage(file=self.path_pysills_main + str(r"\lib\images\PySILLS_Icon.png"))
                    else:
                        pysills_icon = tk.PhotoImage(
                            file=self.path_pysills_main + str(r"\pysills\lib\images\PySILLS_Icon.png"))
            except:
                if self.var_os in ["darwin", "linux"]:
                    pysills_icon = tk.PhotoImage(file=self.path_pysills + str(r"/pysills/lib/images/PySILLS_Icon.png"))
                else:
                    pysills_icon = tk.PhotoImage(file=self.path_pysills + str(r"\pysills\lib\images\PySILLS_Icon.png"))

            self.parent.iconphoto(False, pysills_icon)
        except:
            self.parent.bell()
            print("There is a problem with the PySILLS icon.")

        # FRAMES
        frm_00 = SE(
            parent=self.parent, row_id=start_row, column_id=start_column, n_rows=self.n_rows - 3,
            n_columns=common_n_columns + 11, fg=font_color_light, bg=background_color_header).create_frame(
            relief=tk.FLAT)
        frm_01 = SE(
            parent=self.parent, row_id=start_row + 2, column_id=start_column + 11, n_rows=n_rows_header + 4,
            n_columns=common_n_columns, fg=font_color_light, bg=background_color_elements).create_frame(relief=tk.FLAT)

        # LABELS
        var_lbl_00 = self.language_dict["Project"][self.var_language]
        var_lbl_01 = self.language_dict["Select Mode"][self.var_language]
        var_lbl_02 = self.language_dict["Standard Files"][self.var_language]
        var_lbl_03 = self.language_dict["Sample Files"][self.var_language]
        var_lbl_04c = self.language_dict["Define ICP-MS"][self.var_language]
        var_lbl_dev = self.language_dict["PySILLS subtitle"][self.var_language]

        lbl_00 = SE(
            parent=self.parent, row_id=start_row, column_id=start_column, n_rows=n_rows_header,
            n_columns=common_n_columns, fg=font_color_light, bg=background_color_header).create_simple_label(
            text=var_lbl_00, relief=tk.FLAT, fontsize=font_header)
        lbl_01 = SE(
            parent=self.parent, row_id=start_row, column_id=start_column + 11, n_rows=n_rows_header,
            n_columns=common_n_columns, fg=font_color_light, bg=background_color_header).create_simple_label(
            text=var_lbl_01, relief=tk.FLAT, fontsize=font_header)
        lbl_02 = SE(
            parent=self.parent, row_id=start_row + 13, column_id=start_column, n_rows=n_rows_header,
            n_columns=common_n_columns, fg=font_color_light, bg=background_color_header).create_simple_label(
            text=var_lbl_02, relief=tk.FLAT, fontsize=font_header)
        lbl_03 = SE(
            parent=self.parent, row_id=start_row + 13, column_id=start_column + 11, n_rows=n_rows_header,
            n_columns=common_n_columns, fg=font_color_light, bg=background_color_header).create_simple_label(
            text=var_lbl_03, relief=tk.FLAT, fontsize=font_header)
        lbl_04c = SE(
            parent=self.parent, row_id=start_row + 10, column_id=start_column, n_rows=common_n_rows,
            n_columns=common_n_columns, fg=font_color_light, bg=background_color_header).create_simple_label(
            text=var_lbl_04c, relief=tk.FLAT, fontsize=font_header)
        lbl_version = SE(
            parent=self.parent, row_id=start_row + 29, column_id=start_column, n_rows=common_n_rows,
            n_columns=common_n_columns + 11, fg=font_color_light, bg=background_color_header).create_simple_label(
            text="Version: " + self.val_version, relief=tk.FLAT, fontsize=font_tiny)
        lbl_dev = SE(
            parent=self.parent, row_id=start_row - 1, column_id=start_column, n_rows=common_n_rows,
            n_columns=common_n_columns + 11, fg=font_color_light, bg=accent_color).create_simple_label(
            text=var_lbl_dev, relief=tk.FLAT, fontsize=font_tiny, sticky="news")

        # LISTBOXES
        self.lb_std = SE(
            parent=self.parent, row_id=start_row + 17, column_id=start_column, n_rows=12,
            n_columns=common_n_columns, fg=font_color_dark, bg=background_color_listbox).create_simple_listbox()
        self.lb_smpl = SE(
            parent=self.parent, row_id=start_row + 17, column_id=start_column + 11, n_rows=12,
            n_columns=common_n_columns, fg=font_color_dark, bg=background_color_listbox).create_simple_listbox()

        self.lb_std.configure(font=font_option)
        self.lb_smpl.configure(font=font_option)
        self.container_listbox_files["STD"] = self.lb_std
        self.container_listbox_files["SMPL"] = self.lb_smpl

        # RADIOBUTTONS
        if self.var_language == "English":
            #list_mode = ["Mineral Analysis", "Fluid Inclusions", "Melt Inclusions", "Inclusion Analysis"]
            list_mode = ["Mineral Analysis", "Inclusion Analysis"]
        elif self.var_language == "German":
            #list_mode = ["Minerale", "Flüssigkeitseinschlüsse", "Schmelzeinschlüsse", "Einschlüsse"]
            list_mode = ["Minerale", "Einschlüsse"]
        elif self.var_language == "Chinese":
            str_01 = self.language_dict["Mineral Analysis"][self.var_language]
            str_02 = self.language_dict["Fluid Inclusions"][self.var_language]
            str_03 = self.language_dict["Melt Inclusions"][self.var_language]
            str_04 = "包含分析"
            #list_mode = [str_01, str_02, str_03, str_04]
            list_mode = [str_01, str_04]

        self.var_rb_mode = tk.IntVar()
        for index, mode in enumerate(list_mode):
            rb_mode = SE(
                parent=self.parent, row_id=start_row + 2 + index, column_id=start_column + 11, n_rows=common_n_rows,
                n_columns=common_n_columns, fg=font_color_dark, bg=background_color_elements).create_radiobutton(
                var_rb=self.var_rb_mode, value_rb=index, color_bg=background_color_elements, fg=font_color_dark,
                text=mode, sticky="NESW", relief=tk.FLAT, font=font_element, command=lambda var_rb=self.var_rb_mode:
                self.select_experiment(var_rb))
            if index == 0:
                self.rb_ma = rb_mode
            if mode in ["Report Analysis"]:
                rb_mode.configure(state="disabled")

            self.gui_elements["main"]["Radiobutton"]["General"].append(rb_mode)

        # BUTTONS
        var_btn_01 = self.language_dict["Add"][self.var_language]
        var_btn_02 = self.language_dict["Copy"][self.var_language]
        var_btn_03 = self.language_dict["Delete"][self.var_language]
        var_btn_04 = self.language_dict["New Project"][self.var_language]
        var_btn_05 = self.language_dict["Load Project"][self.var_language]
        var_btn_06 = self.language_dict["Save Project"][self.var_language]
        var_btn_07 = self.language_dict["General Settings"][self.var_language]
        var_btn_08 = self.language_dict["About"][self.var_language]
        var_btn_09 = self.language_dict["Quit"][self.var_language]
        var_btn_10 = self.language_dict["Setup"][self.var_language]
        var_btn_11 = self.language_dict["Manager"][self.var_language]

        btn_01 = SE(
            parent=self.parent, row_id=start_row + 15, column_id=start_column, n_rows=common_n_rows,
            n_columns=n_columns_button + 2, fg=font_color_dark,
            bg=background_color_elements).create_simple_button(
            text=var_btn_01, bg_active=accent_color, fg_active=font_color_dark, command=lambda datatype="STD":
            self.open_csv(datatype))
        btn_02 = SE(
            parent=self.parent, row_id=start_row + 16, column_id=start_column, n_rows=common_n_rows,
            n_columns=n_columns_button + 2, fg=font_color_dark,
            bg=background_color_elements).create_simple_button(
            text=var_btn_02, bg_active=accent_color, fg_active=font_color_dark, command=lambda filetype="STD":
            self.copy_file(filetype))
        btn_03 = SE(
            parent=self.parent, row_id=start_row + 15, column_id=n_columns_button + 2, n_rows=common_n_rows,
            n_columns=n_columns_button + 2, fg=font_color_dark,
            bg=background_color_elements).create_simple_button(
            text=var_btn_03, bg_active=accent_color, fg_active=font_color_dark,
            command=lambda var_lb=self.lb_std, var_list=self.list_std: self.delete_csv(var_lb, var_list))
        btn_11_std = SE(
            parent=self.parent, row_id=start_row + 16, column_id=n_columns_button + 2, n_rows=common_n_rows,
            n_columns=n_columns_button + 2, fg=font_color_dark,
            bg=background_color_elements).create_simple_button(
            text=var_btn_11, bg_active=accent_color, fg_active=font_color_dark,
            command=lambda type="STD": self.project_manager(type))

        btn_04 = SE(
            parent=self.parent, row_id=start_row + 15, column_id=11, n_rows=common_n_rows,
            n_columns=n_columns_button + 2, fg=font_color_dark,
            bg=background_color_elements).create_simple_button(
            text=var_btn_01, bg_active=accent_color, fg_active=font_color_dark, command=lambda datatype="SMPL":
            self.open_csv(datatype))
        btn_05 = SE(
            parent=self.parent, row_id=start_row + 16, column_id=11, n_rows=common_n_rows,
            n_columns=n_columns_button + 2, fg=font_color_dark,
            bg=background_color_elements).create_simple_button(
            text=var_btn_02, bg_active=accent_color, fg_active=font_color_dark, command=lambda filetype="SMPL":
            self.copy_file(filetype))
        btn_06 = SE(
            parent=self.parent, row_id=start_row + 15, column_id=5*n_columns_button + 1, n_rows=common_n_rows,
            n_columns=n_columns_button + 2, fg=font_color_dark,
            bg=background_color_elements).create_simple_button(
            text=var_btn_03, bg_active=accent_color, fg_active=font_color_dark,
            command=lambda var_lb=self.lb_smpl, var_list=self.list_smpl: self.delete_csv(var_lb, var_list))
        btn_11_smpl = SE(
            parent=self.parent, row_id=start_row + 16, column_id=5*n_columns_button + 1, n_rows=common_n_rows,
            n_columns=n_columns_button + 2, fg=font_color_dark,
            bg=background_color_elements).create_simple_button(
            text=var_btn_11, bg_active=accent_color, fg_active=font_color_dark,
            command=lambda type="SMPL": self.project_manager(type))

        btn_07 = SE(
            parent=self.parent, row_id=start_row + 2, column_id=start_column, n_rows=common_n_rows,
            n_columns=common_n_columns, fg=font_color_dark, bg=background_color_elements).create_simple_button(
            text=var_btn_04, bg_active=accent_color, fg_active=font_color_dark, command=self.restart_pysills)
        btn_08 = SE(
            parent=self.parent, row_id=start_row + 3, column_id=start_column, n_rows=common_n_rows,
            n_columns=common_n_columns, fg=font_color_dark, bg=background_color_elements).create_simple_button(
            text=var_btn_05, bg_active=accent_color, fg_active=font_color_dark, command=self.open_project)
        self.btn_save_project = SE(
            parent=self.parent, row_id=start_row + 4, column_id=start_column, n_rows=common_n_rows,
            n_columns=common_n_columns, fg=font_color_dark, bg=background_color_elements).create_simple_button(
            text=var_btn_06, bg_active=accent_color, fg_active=font_color_dark, command=self.save_project)
        self.btn_save_project.configure(state="disabled")
        btn_09 = SE(
            parent=self.parent, row_id=start_row + 5, column_id=start_column, n_rows=common_n_rows,
            n_columns=common_n_columns, fg=font_color_dark, bg=background_color_elements).create_simple_button(
            text=var_btn_07, bg_active=accent_color, fg_active=font_color_dark,
            command=self.subwindow_general_settings)
        btn_about = SE(
            parent=self.parent, row_id=start_row + 6, column_id=start_column, n_rows=common_n_rows,
            n_columns=common_n_columns, fg=font_color_dark, bg=background_color_elements).create_simple_button(
            text=var_btn_08, bg_active=accent_color, fg_active=font_color_dark, command=self.about_pysills)
        btn_10 = SE(
            parent=self.parent, row_id=start_row + 7, column_id=start_column, n_rows=common_n_rows,
            n_columns=common_n_columns, fg=font_color_dark, bg=background_color_elements).create_simple_button(
            text=var_btn_09, bg_active=accent_color, fg_active=font_color_dark, command=self.close_pysills)
        btn_icp = SE(
            parent=self.parent, row_id=start_row + 12, column_id=start_column, n_rows=common_n_rows,
            n_columns=common_n_columns, fg=font_color_dark, bg=background_color_elements).create_simple_button(
            text=var_btn_10, bg_active=accent_color, fg_active=font_color_dark, command=self.define_icp_ms_import_setup)

        btn_01.configure(font=font_element)
        btn_02.configure(font=font_element)
        btn_03.configure(font=font_element)
        btn_04.configure(font=font_element)
        btn_05.configure(font=font_element)
        btn_06.configure(font=font_element)
        btn_07.configure(font=font_element)
        btn_08.configure(font=font_element)
        btn_09.configure(font=font_element)
        btn_10.configure(font=font_element)
        btn_11_std.configure(font=font_element)
        btn_11_smpl.configure(font=font_element)
        self.btn_save_project.configure(font=font_element)
        btn_about.configure(font=font_element)
        btn_icp.configure(font=font_element)

        # OPTION MENUS
        opt_icp = SE(
            parent=self.parent, row_id=start_row + 11, column_id=start_column, n_rows=common_n_rows,
            n_columns=common_n_columns, fg=font_color_dark, bg=background_color_elements).create_simple_optionmenu(
            var_opt=self.var_opt_icp, var_default=self.var_opt_icp.get(),
            var_list=self.container_lists["ICPMS Library"], fg_active=font_color_light, bg_active=accent_color,
            command=lambda var_opt=self.var_opt_icp: self.select_icp_ms(var_opt))
        opt_icp.configure(font=font_option)

    def close_pysills(self):
        self.parent.quit()
        self.parent.destroy()
        exit()

    def define_icp_ms_import_setup(self):
        """Window for the ICP-MS file setup."""
        font_color_dark = self.bg_colors["Dark Font"]
        font_color_light = self.bg_colors["Light Font"]
        background_color_header = self.bg_colors["BG Window"]
        background_color_elements = self.bg_colors["Light"]
        background_color_listbox = self.bg_colors["Very Light"]
        accent_color = self.bg_colors["Accent"]
        font_header = self.font_settings["Header"]
        font_element = self.font_settings["Elements"] # "sans 10 bold"
        font_option = self.font_settings["Options"]

        str_title = self.language_dict["ICP-MS file setup"][self.var_language]

        ## Window Settings
        row_min = self.row_height
        column_min = self.column_width
        n_rows = self.window_dimensions["ICP-MS setup"][0]
        n_columns = self.window_dimensions["ICP-MS setup"][1]

        window_width = int(n_columns*self.column_width)
        window_height = int(n_rows*self.row_height)

        var_geometry = str(window_width) + "x" + str(window_height) + "+" + str(0) + "+" + str(0)

        self.subwindow_icpms_setup = tk.Toplevel(self.parent)
        self.subwindow_icpms_setup.title(str_title)
        self.subwindow_icpms_setup.geometry(var_geometry)
        self.subwindow_icpms_setup.resizable(False, False)
        self.subwindow_icpms_setup["bg"] = background_color_header

        for x in range(n_columns):
            tk.Grid.columnconfigure(self.subwindow_icpms_setup, x, weight=1)
        for y in range(n_rows):
            tk.Grid.rowconfigure(self.subwindow_icpms_setup, y, weight=1)

        # Rows
        for i in range(0, n_rows):
            self.subwindow_icpms_setup.grid_rowconfigure(i, minsize=row_min)
        # Columns
        for i in range(0, n_columns):
            self.subwindow_icpms_setup.grid_columnconfigure(i, minsize=column_min)

        var_row_start = 0
        var_column_start = 0
        var_header_n = 16
        int_category_n = 8

        # LABELS
        str_lbl_01 = self.language_dict["Name"][self.var_language]
        str_lbl_02 = self.language_dict["Line of time stamp"][self.var_language]
        str_lbl_03 = self.language_dict["Skip header"][self.var_language]
        str_lbl_04 = self.language_dict["Skip footer"][self.var_language]
        str_lbl_05 = self.language_dict["Save settings"][self.var_language]
        str_lbl_06 = self.language_dict["Save ICP-MS setup"][self.var_language]

        lbl_01 = SE(
            parent=self.subwindow_icpms_setup, row_id=var_row_start, column_id=var_column_start, n_rows=1,
            n_columns=var_header_n, fg=font_color_light, bg=background_color_header).create_simple_label(
            text=str_title, relief=tk.FLAT, fontsize=font_element)
        lbl_001 = SE(
            parent=self.subwindow_icpms_setup, row_id=var_row_start + 1, column_id=var_column_start, n_rows=1,
            n_columns=int_category_n, fg=font_color_dark, bg=background_color_elements).create_simple_label(
            text=str_lbl_01, relief=tk.FLAT, fontsize=font_element)
        lbl_001 = SE(
            parent=self.subwindow_icpms_setup, row_id=var_row_start + 2, column_id=var_column_start, n_rows=1,
            n_columns=int_category_n, fg=font_color_dark, bg=background_color_elements).create_simple_label(
            text=str_lbl_02, relief=tk.FLAT, fontsize=font_element)
        lbl_001 = SE(
            parent=self.subwindow_icpms_setup, row_id=var_row_start + 3, column_id=var_column_start, n_rows=1,
            n_columns=int_category_n, fg=font_color_dark, bg=background_color_elements).create_simple_label(
            text=str_lbl_03, relief=tk.FLAT, fontsize=font_element)
        lbl_001 = SE(
            parent=self.subwindow_icpms_setup, row_id=var_row_start + 4, column_id=var_column_start, n_rows=1,
            n_columns=int_category_n, fg=font_color_dark, bg=background_color_elements).create_simple_label(
            text=str_lbl_04, relief=tk.FLAT, fontsize=font_element)

        # ENTRIES
        current_val_name = self.container_var["ICP-MS Info"]["name"].get()
        current_val_timestamp = self.container_var["ICP-MS Info"]["timestamp"].get()
        current_val_skipheader = self.container_var["ICP-MS Info"]["skipheader"].get()
        current_val_skipfooter = self.container_var["ICP-MS Info"]["skipfooter"].get()

        entr_002 = SE(
            parent=self.subwindow_icpms_setup, row_id=var_row_start + 1, column_id=var_column_start + int_category_n,
            n_rows=1, n_columns=(var_header_n - int_category_n), fg=font_color_dark,
            bg=self.bg_colors["White"]).create_simple_entry(
            var=self.container_var["ICP-MS Info"]["name"], text_default=current_val_name)
        entr_002 = SE(
            parent=self.subwindow_icpms_setup, row_id=var_row_start + 2, column_id=var_column_start + int_category_n,
            n_rows=1, n_columns=(var_header_n - int_category_n), fg=font_color_dark,
            bg=self.bg_colors["White"]).create_simple_entry(
            var=self.container_var["ICP-MS Info"]["timestamp"], text_default=current_val_timestamp)
        entr_002 = SE(
            parent=self.subwindow_icpms_setup, row_id=var_row_start + 3, column_id=var_column_start + int_category_n,
            n_rows=1, n_columns=(var_header_n - int_category_n), fg=font_color_dark,
            bg=self.bg_colors["White"]).create_simple_entry(
            var=self.container_var["ICP-MS Info"]["skipheader"], text_default=current_val_skipheader)
        entr_002 = SE(
            parent=self.subwindow_icpms_setup, row_id=var_row_start + 4, column_id=var_column_start + int_category_n,
            n_rows=1, n_columns=(var_header_n - int_category_n), fg=font_color_dark,
            bg=self.bg_colors["White"]).create_simple_entry(
            var=self.container_var["ICP-MS Info"]["skipfooter"], text_default=current_val_skipfooter)

        # BUTTONS
        if self.var_os == "darwin":
            font_color_accent = font_color_dark
        else:
            font_color_accent = font_color_light

        btn_02 = SE(
            parent=self.subwindow_icpms_setup, row_id=var_row_start + 6,
            column_id=var_column_start + (var_header_n - int_category_n), n_rows=1,
            n_columns=(var_header_n - int_category_n), fg=font_color_accent,
            bg=accent_color).create_simple_button(
            text=str_lbl_05, bg_active=accent_color, fg_active=font_color_accent,
            command=self.confirm_icpms_settings)
        btn_002 = SE(
            parent=self.subwindow_icpms_setup, row_id=var_row_start + 6,
            column_id=var_column_start, n_rows=1,
            n_columns=(var_header_n - int_category_n), fg=font_color_accent,
            bg=accent_color).create_simple_button(
            text=str_lbl_06, bg_active=accent_color, fg_active=font_color_accent,
            command=self.save_icpms_settings)

        btn_02.configure(font=font_element)
        btn_002.configure(font=font_element)

    def confirm_icpms_settings(self):
        self.var_opt_icp.set(self.container_var["ICP-MS Info"]["name"].get())
        self.container_icpms["timestamp"] = self.container_var["ICP-MS Info"]["timestamp"].get()
        self.container_icpms["skipheader"] = self.container_var["ICP-MS Info"]["skipheader"].get()
        self.container_icpms["skipfooter"] = self.container_var["ICP-MS Info"]["skipfooter"].get()

    def save_icpms_settings(self):
        str_icpms_name = self.container_var["ICP-MS Info"]["name"].get()
        int_icpms_skipheader = self.container_var["ICP-MS Info"]["skipheader"].get()
        int_icpms_skipfooter = self.container_var["ICP-MS Info"]["skipfooter"].get()
        int_icpms_timestamp = self.container_var["ICP-MS Info"]["timestamp"].get()

        filename_export = filedialog.asksaveasfile(
            mode="w", initialfile=str_icpms_name, defaultextension=".csv", filetypes=[("csv", "*.csv")])
        filename_export = filename_export.name

        with open(filename_export, "w") as file_content:
            file_content.write("name" + "," + str(str_icpms_name) + "\n")
            file_content.write("skipheader" + "," + str(int_icpms_skipheader) + "\n")
            file_content.write("skipfooter" + "," + str(int_icpms_skipfooter) + "\n")
            file_content.write("timestamp" + "," + str(int_icpms_timestamp) + "\n")

    def select_icp_ms(self, var_opt):
        path = os.getcwd()
        path = self.path_pysills
        var_instrument_raw = "Select ICP-MS"

        if self.file_loaded == False:
            if self.demo_mode:
                try:
                    var_instrument_raw = var_opt.get()
                except:
                    var_instrument_raw = var_opt

                var_instrument = var_instrument_raw.replace(" ", "_")
            else:
                var_instrument = var_opt.replace(" ", "_")
                if var_instrument != "Select_ICP-MS":
                    var_instrument_raw = var_opt
        else:
            var_instrument_raw = var_opt.get()
            if var_instrument_raw != "Select ICP-MS":
                var_instrument = var_instrument_raw.replace(" ", "_")

        if var_instrument_raw != "Select ICP-MS":
            try:
                file_long = self.path_pysills_main + str("/lib/icpms/" + var_instrument + ".csv")
                file_content = open(file_long)
            except:
                file_long = self.path_pysills_main + str("/pysills/lib/icpms/" + var_instrument + ".csv")
                file_content = open(file_long)

            for index, line in enumerate(file_content):
                line_parts = line.split(",")
                if "\n" in line_parts[-1]:
                    line_parts[-1] = line_parts[-1].replace("\n", "")
                try:
                    self.container_icpms[line_parts[0]] = int(line_parts[-1])
                except:
                    self.container_icpms[line_parts[0]] = line_parts[-1]

    def copy_file(self, filetype):
        if self.pysills_mode == "MA":
            info_key = "ma_setting"
        elif self.pysills_mode in ["FI", "INCL"]:
            info_key = "fi_setting"
        elif self.pysills_mode == "MI":
            info_key = "mi_setting"

        if filetype == "STD":
            var_lb = self.lb_std
            var_list = self.list_std
        elif filetype == "SMPL":
            var_lb = self.lb_smpl
            var_list = self.list_smpl

        item = var_lb.curselection()
        var_file_long = var_list[item[0]]
        var_file_long_parts = var_file_long.split(".")
        var_file_extension = var_file_long_parts[-1]
        if "_copy" in var_file_long_parts[0]:
            if "_copy" == var_file_long_parts[0][-5:]:
                str_added = "_copy2"
                file_base = var_file_long_parts[0][:-5]
            else:
                number_current = var_file_long_parts[0][-5:][-1]
                new_number = int(number_current) + 1
                str_added = "_copy" + str(new_number)
                file_base = var_file_long_parts[0][:-6]
        else:
            str_added = "_copy"
            file_base = var_file_long_parts[0]
        var_file_long_copy = file_base + str_added + "." + var_file_extension

        file_parts = var_file_long.split("/")
        file_short_original = file_parts[-1]
        file_isotopes_original = self.container_lists["Measured Isotopes"][file_short_original]

        problem_present = False
        try:
            with open(var_file_long, "r") as file:
                content = file.read()

            with open(var_file_long_copy, "w") as copied_file:
                copied_file.write(content)
        except:
            problem_present = True
            print("File cannot be read.")

        if var_file_long_copy not in var_list:
            var_list.append(var_file_long_copy)
            file_parts_copy = var_file_long_copy.split("/")
            var_file_short_copy = file_parts_copy[-1]
            var_lb.insert(tk.END, var_file_short_copy)

            self.container_lists["Measured Isotopes"][var_file_short_copy] = file_isotopes_original
            file_isotopes = self.container_lists["Measured Isotopes"][var_file_short_copy]

            self.add_needed_variables_for_later_added_files(
                filename_long=var_file_long_copy, filename_short=var_file_short_copy, filetype=filetype,
                file_isotopes=file_isotopes)

            var_skipheader = self.container_icpms["skipheader"]
            var_skipfooter = self.container_icpms["skipfooter"]
            var_timestamp = self.container_icpms["timestamp"]
            var_icpms = self.container_icpms["name"]

            if problem_present == False:
                dates, times = Data(filename=var_file_long_copy).import_as_list(
                    skip_header=var_skipheader, skip_footer=var_skipfooter, timestamp=var_timestamp, icpms=var_icpms)
            else:
                try:
                    dates, times = Data(filename=var_file_long).import_as_list(
                        skip_header=var_skipheader, skip_footer=var_skipfooter, timestamp=var_timestamp,
                        icpms=var_icpms)
                except:
                    dates = 0
                    times_input = self.container_var["acquisition times"][filetype][file_short_original].get()
                    times_input_parts = times_input.split(":")
                    times = [[times_input_parts[0], times_input_parts[1], times_input_parts[2]]]

            t_start = datetime.timedelta(hours=int(times[0][0]), minutes=int(times[0][1]), seconds=int(times[0][2]))

            if var_file_short_copy not in self.container_var["acquisition times"][filetype]:
                self.container_var["acquisition times"][filetype][var_file_short_copy] = tk.StringVar()
                self.container_var["acquisition times"][filetype][var_file_short_copy].set(
                    times[0][0] + ":" + times[0][1] + ":" + times[0][2])

            if var_file_short_copy not in self.container_measurements["RAW"]:
                self.container_measurements["RAW"][var_file_short_copy] = {}

            for key, item in self.container_measurements["RAW"][file_short_original].items():
                self.container_measurements["RAW"][var_file_short_copy][key] = item

            df_data = self.container_measurements["Dataframe"][file_short_original]
            self.container_measurements["Dataframe"][var_file_short_copy] = df_data

            if file_short_original in self.container_spikes:
                self.container_spikes[var_file_short_copy] = self.container_spikes[file_short_original].copy()
            else:
                self.container_spikes[var_file_short_copy] = {}

            self.copied_file = True

            try:
                if self.pysills_mode == "MA":
                    self.subwindow_ma_settings.destroy()
                    self.ma_settings()
                elif self.pysills_mode in ["FI", "INCL"]:
                    self.subwindow_fi_settings.destroy()
                    self.fi_settings()
                elif self.pysills_mode == "MI":
                    self.subwindow_mi_settings.destroy()
                    self.mi_settings()
            except:
                if self.pysills_mode == "MA":
                    self.ma_settings()
                elif self.pysills_mode in ["FI", "INCL"]:
                    self.fi_settings()
                elif self.pysills_mode == "MI":
                    self.mi_settings()
        else:
            file_parts_copy = var_file_long_copy.split("/")
            var_file_short_copy = file_parts_copy[-1]

            var_file_long_parts = var_file_long_copy.split(".")
            var_file_extension = var_file_long_parts[-1]

            if "_copy" in var_file_long_parts[0]:
                if "_copy" == var_file_long_parts[0][-5:]:
                    str_added = "_copy2"
                    file_base = var_file_long_parts[0][:-5]
                else:
                    number_current = var_file_long_parts[0][-5:][-1]
                    new_number = int(number_current) + 1
                    str_added = "_copy" + str(new_number)
                    file_base = var_file_long_parts[0][:-6]
            else:
                str_added = "_copy"
                file_base = var_file_long_parts[0]

            var_file_long_copy = file_base + str_added + "." + var_file_extension
            file_parts_copy = var_file_long_copy.split("/")
            var_file_short_copy = file_parts_copy[-1]
            var_list.append(var_file_long_copy)
            var_lb.insert(tk.END, var_file_short_copy)

            self.container_lists["Measured Isotopes"][var_file_short_copy] = file_isotopes_original
            file_isotopes = self.container_lists["Measured Isotopes"][var_file_short_copy]

            self.add_needed_variables_for_later_added_files(
                filename_long=var_file_long_copy, filename_short=var_file_short_copy, filetype=filetype,
                file_isotopes=file_isotopes)

            var_skipheader = self.container_icpms["skipheader"]
            var_skipfooter = self.container_icpms["skipfooter"]
            var_timestamp = self.container_icpms["timestamp"]
            var_icpms = self.container_icpms["name"]

            if problem_present == False:
                dates, times = Data(filename=var_file_long_copy).import_as_list(
                    skip_header=var_skipheader, skip_footer=var_skipfooter, timestamp=var_timestamp, icpms=var_icpms)
            else:
                try:
                    dates, times = Data(filename=var_file_long).import_as_list(
                        skip_header=var_skipheader, skip_footer=var_skipfooter, timestamp=var_timestamp,
                        icpms=var_icpms)
                except:
                    dates = 0
                    times_input = self.container_var["acquisition times"][filetype][file_short_original].get()
                    times_input_parts = times_input.split(":")
                    times = [[times_input_parts[0], times_input_parts[1], times_input_parts[2]]]

            t_start = datetime.timedelta(hours=int(times[0][0]), minutes=int(times[0][1]), seconds=int(times[0][2]))

            if var_file_short_copy not in self.container_var["acquisition times"][filetype]:
                self.container_var["acquisition times"][filetype][var_file_short_copy] = tk.StringVar()
                self.container_var["acquisition times"][filetype][var_file_short_copy].set(
                    times[0][0] + ":" + times[0][1] + ":" + times[0][2])

            if var_file_short_copy not in self.container_measurements["RAW"]:
                self.container_measurements["RAW"][var_file_short_copy] = {}

            for key, item in self.container_measurements["RAW"][file_short_original].items():
                self.container_measurements["RAW"][var_file_short_copy][key] = item

            df_data = self.container_measurements["Dataframe"][file_short_original]
            self.container_measurements["Dataframe"][var_file_short_copy] = df_data

            if file_short_original in self.container_spikes:
                self.container_spikes[var_file_short_copy] = self.container_spikes[file_short_original].copy()
            else:
                self.container_spikes[var_file_short_copy] = {}

            self.copied_file = True

            try:
                if self.pysills_mode == "MA":
                    self.subwindow_ma_settings.destroy()
                    self.ma_settings()
                elif self.pysills_mode in ["FI", "INCL"]:
                    self.subwindow_fi_settings.destroy()
                    self.fi_settings()
                elif self.pysills_mode == "MI":
                    self.subwindow_mi_settings.destroy()
                    self.mi_settings()
            except:
                if self.pysills_mode == "MA":
                    self.ma_settings()
                elif self.pysills_mode in ["FI", "INCL"]:
                    self.fi_settings()
                elif self.pysills_mode == "MI":
                    self.mi_settings()

    def add_needed_variables_for_later_added_files(self, filename_long, filename_short, filetype, file_isotopes):
        if self.pysills_mode == "MA":
            info_key = "ma_setting"
        elif self.pysills_mode in ["FI", "INCL"]:
            info_key = "fi_setting"
        elif self.pysills_mode == "MI":
            info_key = "mi_setting"

        if filename_long not in self.container_lists[filetype]["Long"]:
            self.container_lists[filetype]["Long"].append(filename_long)
            self.container_lists[filetype]["Short"].append(filename_short)

        self.container_var[info_key]["Data Type Plot"][filetype][filename_short] = tk.IntVar()
        self.container_var[info_key]["Data Type Plot"][filetype][filename_short].set(0)
        self.container_var[info_key]["Analyse Mode Plot"][filetype][filename_short] = tk.IntVar()
        self.container_var[info_key]["Analyse Mode Plot"][filetype][filename_short].set(0)
        self.container_var[info_key]["Display RAW"][filetype][filename_short] = {}
        self.container_var[info_key]["Display SMOOTHED"][filetype][filename_short] = {}

        if filename_short not in self.container_var["ma_setting"]["Time-Signal Lines"][filetype]:
            self.container_var[info_key]["Time-Signal Lines"][filetype][filename_short] = {}
            self.container_var[info_key]["Time-Ratio Lines"][filetype][filename_short] = {}
            self.container_var[info_key]["Checkboxes Isotope Diagram"][filetype][filename_short] = {}
            self.container_var[info_key]["Calculation Interval"][filetype][
                filename_short] = tk.IntVar()
            self.container_var[info_key]["Calculation Interval"][filetype][filename_short].set(3)
            self.container_var[info_key]["Calculation Interval Visibility"][filetype][
                filename_short] = {}

        for isotope in file_isotopes:
            self.build_checkbutton_isotope_visibility(
                var_mode=info_key, var_filetype=filetype, var_filename_short=filename_short,
                var_isotope=isotope)

            self.container_var[info_key]["Time-Signal Lines"][filetype][filename_short][isotope] = {
                "RAW": None, "SMOOTHED": None}
            self.container_var[info_key]["Time-Ratio Lines"][filetype][filename_short][isotope] = {
                "RAW": None, "SMOOTHED": None}
            self.container_var[info_key]["Checkboxes Isotope Diagram"][filetype][filename_short][
                isotope] = {"RAW": None, "SMOOTHED": None}

        self.container_var["Plotting"][self.pysills_mode]["Quickview"] = {"Canvas": None, "Toolbar": None}
        self.container_var["Plotting"][self.pysills_mode]["Time-Signal"] = {"Canvas": None, "Toolbar": None}
        self.container_var["Plotting"][self.pysills_mode]["Time-Ratio"] = {"Canvas": None, "Toolbar": None}

    def select_experiment(self, var_rb):
        font_color_dark = self.bg_colors["Dark Font"]
        font_color_light = self.bg_colors["Light Font"]
        background_color_header = self.bg_colors["BG Window"]
        background_color_elements = self.bg_colors["Light"]
        background_color_listbox = self.bg_colors["Very Light"]
        accent_color = self.bg_colors["Accent"]
        font_header = self.font_settings["Header"]
        font_elements = self.font_settings["Elements"]
        font_small = self.font_settings["Small"]

        start_row = 11
        start_column = 11

        str_btn_01 = self.language_dict["Settings"][self.var_language]
        str_btn_02 = self.language_dict["Results"][self.var_language]
        str_btn_03 = self.language_dict["Extras"][self.var_language]

        if var_rb.get() == 0:  # Mineral Analysis
            self.pysills_mode = "MA"
            ## Cleaning
            for gui_category in ["Label", "Button"]:
                if len(self.gui_elements["main"][gui_category]["Specific"]) > 0:
                    for gui_item in self.gui_elements["main"][gui_category]["Specific"]:
                        gui_item.grid_remove()
                    self.gui_elements["main"][gui_category]["Specific"].clear()

            ## Labels
            str_lbl_01 = self.language_dict["Mineral Analysis"][self.var_language]
            lb_01 = SE(
                parent=self.parent, row_id=start_row, column_id=start_column, n_rows=2, n_columns=10,
                fg=font_color_light, bg=background_color_header).create_simple_label(
                text=str_lbl_01, relief=tk.FLAT, fontsize=font_header)

            self.gui_elements["main"]["Label"]["Specific"].append(lb_01)

            ## Buttons
            btn_01 = SE(
                parent=self.parent, row_id=start_row + 2, column_id=start_column, n_rows=1, n_columns=10,
                fg=font_color_dark, bg=background_color_elements).create_simple_button(
                text=str_btn_01, bg_active=accent_color, fg_active=font_color_light,
                command=self.ma_settings)
            btn_02 = SE(
                parent=self.parent, row_id=start_row + 3, column_id=start_column, n_rows=1, n_columns=10,
                fg=font_color_dark, bg=background_color_elements).create_simple_button(
                text=str_btn_02, bg_active=accent_color, fg_active=font_color_light,
                command=self.ma_datareduction_files)
            btn_03 = SE(
                parent=self.parent, row_id=start_row + 4, column_id=start_column, n_rows=1, n_columns=10,
                fg=font_color_dark, bg=background_color_elements).create_simple_button(
                text=str_btn_03, bg_active=accent_color, fg_active=font_color_light,
                command=self.ma_extras)

            self.gui_elements["main"]["Button"]["Specific"].extend([btn_01, btn_02, btn_03])

        elif var_rb.get() == 3:  # Fluid Inclusions (1)
            self.pysills_mode = "FI"
            ## Cleaning
            for gui_category in ["Label", "Button"]:
                if len(self.gui_elements["main"][gui_category]["Specific"]) > 0:
                    for gui_item in self.gui_elements["main"][gui_category]["Specific"]:
                        gui_item.grid_remove()
                    self.gui_elements["main"][gui_category]["Specific"].clear()

            ## Labels
            str_lbl_01 = self.language_dict["Fluid Inclusions"][self.var_language]
            if self.var_language == "German":
                str_lbl_01 = "Flüssigkeits-\neinschlüsse"
            lb_01 = SE(
                parent=self.parent, row_id=start_row, column_id=start_column, n_rows=2, n_columns=10,
                fg=self.bg_colors["Light Font"], bg=self.bg_colors["BG Window"]).create_simple_label(
                text=str_lbl_01, relief=tk.FLAT, fontsize=font_header)

            self.gui_elements["main"]["Label"]["Specific"].append(lb_01)

            ## Buttons
            btn_01 = SE(
                parent=self.parent, row_id=start_row + 2, column_id=start_column, n_rows=1, n_columns=10,
                fg=font_color_dark, bg=background_color_elements).create_simple_button(
                text=str_btn_01, bg_active=accent_color, fg_active=font_color_light,
                command=self.fi_settings)
            btn_02 = SE(
                parent=self.parent, row_id=start_row + 3, column_id=start_column, n_rows=1, n_columns=10,
                fg=font_color_dark, bg=background_color_elements).create_simple_button(
                text=str_btn_02, bg_active=accent_color, fg_active=font_color_light,
                command=self.fi_datareduction_files)
            btn_03 = SE(
                parent=self.parent, row_id=start_row + 4, column_id=start_column, n_rows=1, n_columns=10,
                fg=font_color_dark, bg=background_color_elements).create_simple_button(
                text=str_btn_03, bg_active=accent_color, fg_active=font_color_light,
                command=lambda init=True: self.fi_extras(init))

            self.gui_elements["main"]["Button"]["Specific"].extend([btn_01, btn_02, btn_03])

        elif var_rb.get() == 2:  # Melt Inclusions
            self.pysills_mode = "MI"
            ## Cleaning
            for gui_category in ["Label", "Button"]:
                if len(self.gui_elements["main"][gui_category]["Specific"]) > 0:
                    for gui_item in self.gui_elements["main"][gui_category]["Specific"]:
                        gui_item.grid_remove()
                    self.gui_elements["main"][gui_category]["Specific"].clear()

            ## Labels
            str_lbl_01 = self.language_dict["Melt Inclusions"][self.var_language]
            lb_01 = SE(
                parent=self.parent, row_id=start_row, column_id=start_column, n_rows=2, n_columns=10,
                fg=self.bg_colors["Light Font"], bg=self.bg_colors["BG Window"]).create_simple_label(
                text=str_lbl_01, relief=tk.FLAT, fontsize=font_header)

            self.gui_elements["main"]["Label"]["Specific"].append(lb_01)

            ## Buttons
            btn_01 = SE(
                parent=self.parent, row_id=start_row + 2, column_id=start_column, n_rows=1, n_columns=10,
                fg=font_color_dark, bg=background_color_elements).create_simple_button(
                text=str_btn_01, bg_active=accent_color, fg_active=font_color_light,
                command=self.mi_settings)
            btn_02 = SE(
                parent=self.parent, row_id=start_row + 3, column_id=start_column, n_rows=1, n_columns=10,
                fg=font_color_dark, bg=background_color_elements).create_simple_button(
                text=str_btn_02, bg_active=accent_color, fg_active=font_color_light,
                command=self.fi_datareduction_files)
            btn_03 = SE(
                parent=self.parent, row_id=start_row + 4, column_id=start_column, n_rows=1, n_columns=10,
                fg=font_color_dark, bg=background_color_elements).create_simple_button(
                text=str_btn_03, bg_active=accent_color, fg_active=font_color_light,
                command=lambda init=True: self.mi_extras(init))

            self.gui_elements["main"]["Button"]["Specific"].extend([btn_01, btn_02, btn_03])

        elif var_rb.get() == 1:  # Inclusion Analysis (3)
            self.pysills_mode = "INCL"
            ## Cleaning
            for gui_category in ["Label", "Button"]:
                if len(self.gui_elements["main"][gui_category]["Specific"]) > 0:
                    for gui_item in self.gui_elements["main"][gui_category]["Specific"]:
                        gui_item.grid_remove()
                    self.gui_elements["main"][gui_category]["Specific"].clear()

            ## Labels
            str_lbl_01 = self.language_dict["Inclusions"][self.var_language]
            lb_01 = SE(
                parent=self.parent, row_id=start_row, column_id=start_column, n_rows=2, n_columns=10,
                fg=self.bg_colors["Light Font"], bg=self.bg_colors["BG Window"]).create_simple_label(
                text=str_lbl_01, relief=tk.FLAT, fontsize=font_header)

            self.gui_elements["main"]["Label"]["Specific"].append(lb_01)

            ## Buttons
            btn_01 = SE(
                parent=self.parent, row_id=start_row + 2, column_id=start_column, n_rows=1, n_columns=10,
                fg=font_color_dark, bg=background_color_elements).create_simple_button(
                text=str_btn_01, bg_active=accent_color, fg_active=font_color_light,
                command=self.fi_settings)
            btn_02 = SE(
                parent=self.parent, row_id=start_row + 3, column_id=start_column, n_rows=1, n_columns=10,
                fg=font_color_dark, bg=background_color_elements).create_simple_button(
                text=str_btn_02, bg_active=accent_color, fg_active=font_color_light,
                command=self.fi_datareduction_files)
            btn_03 = SE(
                parent=self.parent, row_id=start_row + 4, column_id=start_column, n_rows=1, n_columns=10,
                fg=font_color_dark, bg=background_color_elements).create_simple_button(
                text=str_btn_03, bg_active=accent_color, fg_active=font_color_light,
                command=lambda init=True: self.fi_extras(init))

            self.gui_elements["main"]["Button"]["Specific"].extend([btn_01, btn_02, btn_03])

        elif var_rb.get() == 4:  # Report Analysis
            self.pysills_mode = "OA"
            ## Cleaning
            for gui_category in ["Label", "Button"]:
                if len(self.gui_elements["main"][gui_category]["Specific"]) > 0:
                    for gui_item in self.gui_elements["main"][gui_category]["Specific"]:
                        gui_item.grid_remove()
                    self.gui_elements["main"][gui_category]["Specific"].clear()

            ## Labels
            lb_01 = SE(
                parent=self.parent, row_id=start_row, column_id=start_column, n_rows=2, n_columns=10,
                fg=self.bg_colors["Light Font"], bg=self.bg_colors["BG Window"]).create_simple_label(
                text="Report Analysis", relief=tk.FLAT, fontsize=font_header)

            self.gui_elements["main"]["Label"]["Specific"].append(lb_01)

            ## Buttons
            btn_01 = SE(
                parent=self.parent, row_id=start_row + 2, column_id=start_column, n_rows=1, n_columns=10,
                fg=font_color_dark, bg=background_color_elements).create_simple_button(
                text=str_btn_01, bg_active=accent_color, fg_active=font_color_light)
            btn_02 = SE(
                parent=self.parent, row_id=start_row + 3, column_id=start_column, n_rows=1, n_columns=10,
                fg=font_color_dark, bg=background_color_elements).create_simple_button(
                text=str_btn_02, bg_active=accent_color, fg_active=font_color_light)
            btn_03 = SE(
                parent=self.parent, row_id=start_row + 4, column_id=start_column, n_rows=1, n_columns=10,
                fg=font_color_dark, bg=background_color_elements).create_simple_button(
                text=str_btn_03, bg_active=accent_color, fg_active=font_color_light,
                command=self.ma_datareduction_files)
            btn_03.configure(state="disabled")

            self.gui_elements["main"]["Button"]["Specific"].extend([btn_01, btn_02, btn_03])

        btn_01.configure(font=font_elements)
        btn_02.configure(font=font_elements)
        btn_03.configure(font=font_elements)

    def build_srm_database(self):
        for key, item in self.container_var["SRM"].items():
            if key != "default":
                if key in self.container_lists["Measured Isotopes"]["All"]:
                    self.container_lists["SRM Isotopes"][key] = item.get()
                else:
                    key_short = key.split("/")[-1]
                    var_srm_file = self.container_var["STD"][key]["SRM"].get()
                    self.container_lists["SRM Files"][key_short] = var_srm_file

        self.collect_srm_data()

    def collect_srm_data(self):
        helper_list = []
        for var_srm in self.container_lists["SRM Files"].values():
            if var_srm not in helper_list and var_srm != "Select SRM":
                helper_list.append(var_srm)

        if len(helper_list) > 0:
            for var_srm in helper_list:
                self.fill_srm_values(var_srm=var_srm)

    def load_data_as_dataframe(self, filename_short, filename_long, with_isotopes=False):
        if self.file_loaded == False:
            if self.container_icpms["name"] != None:
                var_skipheader = self.container_icpms["skipheader"]
                var_skipfooter = self.container_icpms["skipfooter"]
                df_data = DE(filename_long=filename_long).get_measurements(
                    delimiter=",", skip_header=var_skipheader, skip_footer=var_skipfooter)
            else:
                df_data = DE(filename_long=filename_long).get_measurements(
                    delimiter=",", skip_header=3, skip_footer=1)
        else:
            df_data = self.container_measurements["Dataframe"][filename_short]

        times = DE().get_times(dataframe=df_data)

        if with_isotopes == True:
            isotopes = DE().get_isotopes(dataframe=df_data)

        if with_isotopes == True:
            return df_data, times, isotopes
        else:
            return df_data, times

    def build_container_measurements(self, filetype, filename_short):
        index_filename = self.container_lists[filetype]["Short"].index(filename_short)
        filename_long = self.container_lists[filetype]["Long"][index_filename]

        self.container_measurements["RAW"][filename_short] = {}
        self.container_measurements["EDITED"][filename_short] = {}
        self.container_measurements["SELECTED"][filename_short] = {}
        self.container_measurements["SELECTED"][filename_short]["RAW"] = {}
        self.container_measurements["SELECTED"][filename_short]["SMOOTHED"] = {}

        df_data, times = self.load_data_as_dataframe(filename_short=filename_short, filename_long=filename_long)

        self.container_measurements["RAW"][filename_short]["Time"] = times.tolist()
        self.container_measurements["EDITED"][filename_short]["Time"] = times.tolist()
        self.container_measurements["SELECTED"][filename_short]["Time"] = times.tolist()

        file_isotopes = self.container_lists["Measured Isotopes"][filename_short]
        for isotope in file_isotopes:
            self.container_measurements["RAW"][filename_short][isotope] = df_data[isotope].tolist()
            self.container_measurements["EDITED"][filename_short][isotope] = {}
            self.container_measurements["SELECTED"][filename_short]["RAW"][isotope] = {}
            self.container_measurements["SELECTED"][filename_short]["SMOOTHED"][isotope] = {}
            self.container_measurements["EDITED"][filename_short][isotope]["BG"] = []
            self.container_measurements["SELECTED"][filename_short]["RAW"][isotope]["BG"] = []
            self.container_measurements["SELECTED"][filename_short]["SMOOTHED"][isotope]["BG"] = []

            if self.pysills_mode == "MA":
                self.container_measurements["EDITED"][filename_short][isotope]["MAT"] = []
                self.container_measurements["SELECTED"][filename_short]["RAW"][isotope]["MAT"] = []
                self.container_measurements["SELECTED"][filename_short]["SMOOTHED"][isotope]["MAT"] = []
                self.container_measurements["SELECTED"][filename_short]["SMOOTHED"][isotope]["MAT"] = []
            elif self.pysills_mode in ["FI", "MI", "INCL"]:
                self.container_measurements["EDITED"][filename_short][isotope]["MAT"] = []
                self.container_measurements["SELECTED"][filename_short]["RAW"][isotope]["MAT"] = []
                self.container_measurements["SELECTED"][filename_short]["SMOOTHED"][isotope]["MAT"] = []
                self.container_measurements["SELECTED"][filename_short]["SMOOTHED"][isotope]["MAT"] = []
                self.container_measurements["EDITED"][filename_short][isotope]["INCL"] = []
                self.container_measurements["SELECTED"][filename_short]["RAW"][isotope]["INCL"] = []
                self.container_measurements["SELECTED"][filename_short]["SMOOTHED"][isotope]["INCL"] = []
                self.container_measurements["SELECTED"][filename_short]["SMOOTHED"][isotope]["INCL"] = []

    def spike_elimination_all(self, filetype, algorithm, mode="MA", calculation=False):
        if self.pysills_mode == "MA":
            key_setting = "ma_setting"
        elif self.pysills_mode in ["FI", "INCL"]:
            key_setting = "fi_setting"
        elif self.pysills_mode == "MI":
            key_setting = "mi_setting"

        self.var_calculation = calculation

        if algorithm == "Grubbs":
            if filetype == "STD":
                for filename_long in self.list_std:
                    parts = filename_long.split("/")
                    filename_short = parts[-1]
                    #
                    self.container_measurements["RAW"][filename_short] = {}
                    self.container_measurements["EDITED"][filename_short] = {}
                    self.container_measurements["SELECTED"][filename_short] = {}
                    self.container_measurements["SELECTED"][filename_short]["RAW"] = {}
                    self.container_measurements["SELECTED"][filename_short]["SMOOTHED"] = {}
                    #
                    if self.diagrams_setup[filetype][filename_short]["FIG"] == None:
                        self.diagrams_setup[filetype][filename_short]["FIG"] = Figure(
                            figsize=(10, 5), facecolor=self.green_light)
                        self.diagrams_setup[filetype][filename_short]["AX"] = \
                            self.diagrams_setup[filetype][filename_short][
                                "FIG"].add_subplot()
                    #
                    if self.diagrams_setup[filetype][filename_short]["FIG_RATIO"] == None:
                        self.diagrams_setup[filetype][filename_short]["FIG_RATIO"] = Figure(
                            figsize=(10, 5), facecolor=self.green_light)
                        self.diagrams_setup[filetype][filename_short]["AX_RATIO"] = \
                            self.diagrams_setup[filetype][filename_short][
                                "FIG_RATIO"].add_subplot()

                    if self.file_loaded == False:
                        if self.container_icpms["name"] != None:
                            var_skipheader = self.container_icpms["skipheader"]
                            var_skipfooter = self.container_icpms["skipfooter"]
                            df_data = DE(filename_long=filename_long).get_measurements(
                                delimiter=",", skip_header=var_skipheader, skip_footer=var_skipfooter)
                        else:
                            df_data = DE(filename_long=filename_long).get_measurements(
                                delimiter=",", skip_header=3, skip_footer=1)
                    else:
                        df_data = self.container_measurements["Dataframe"][filename_short]

                    times = DE().get_times(dataframe=df_data)
                    start_time = times.iloc[0]
                    end_time = times.iloc[-1]
                    start_index = times[times == start_time].index[0]
                    end_index = times[times == end_time].index[0]
                    spk_id = 1
                    #
                    box_spk = self.diagrams_setup[filetype][filename_short]["AX"].axvspan(
                        start_time, end_time, alpha=0.125, color=self.yellow_dark)
                    box_spk_ratio = self.diagrams_setup[filetype][filename_short]["AX_RATIO"].axvspan(
                        start_time, end_time, alpha=0.125, color=self.yellow_dark)
                    #
                    isotope = "".join(self.container_lists["ISOTOPES"])
                    #
                    self.container_measurements["RAW"][filename_short]["Time"] = times.tolist()
                    self.container_measurements["EDITED"][filename_short]["Time"] = times.tolist()
                    self.container_measurements["SELECTED"][filename_short]["Time"] = times.tolist()

                    df_isotopes = self.container_lists["Measured Isotopes"][filename_short]
                    for isotope in df_isotopes:
                        self.container_measurements["RAW"][filename_short][isotope] = df_data[isotope].tolist()
                        self.container_measurements["EDITED"][filename_short][isotope] = {}
                        self.container_measurements["SELECTED"][filename_short]["RAW"][isotope] = {}
                        self.container_measurements["SELECTED"][filename_short]["SMOOTHED"][isotope] = {}
                        #
                        self.container_measurements["EDITED"][filename_short][isotope]["BG"] = []
                        self.container_measurements["SELECTED"][filename_short]["RAW"][isotope]["BG"] = []
                        self.container_measurements["SELECTED"][filename_short]["SMOOTHED"][isotope]["BG"] = []
                        if mode == "MA":
                            self.container_measurements["EDITED"][filename_short][isotope]["MAT"] = []
                            self.container_measurements["SELECTED"][filename_short]["RAW"][isotope]["MAT"] = []
                            self.container_measurements["SELECTED"][filename_short]["SMOOTHED"][isotope]["MAT"] = []
                            self.container_measurements["SELECTED"][filename_short]["SMOOTHED"][isotope]["MAT"] = []
                            #
                            var_settings = "ma_setting"
                        elif mode in ["FI", "MI"]:
                            self.container_measurements["EDITED"][filename_short][isotope]["MAT"] = []
                            self.container_measurements["SELECTED"][filename_short]["RAW"][isotope]["MAT"] = []
                            self.container_measurements["SELECTED"][filename_short]["SMOOTHED"][isotope]["MAT"] = []
                            self.container_measurements["SELECTED"][filename_short]["SMOOTHED"][isotope]["MAT"] = []
                            self.container_measurements["EDITED"][filename_short][isotope]["INCL"] = []
                            self.container_measurements["SELECTED"][filename_short]["RAW"][isotope]["INCL"] = []
                            self.container_measurements["SELECTED"][filename_short]["SMOOTHED"][isotope]["INCL"] = []
                            self.container_measurements["SELECTED"][filename_short]["SMOOTHED"][isotope]["INCL"] = []
                            if mode == "FI":
                                var_settings = "fi_setting"
                            elif mode == "MI":
                                var_settings = "mi_setting"

                        self.spikes_isotopes["STD"][filename_short][isotope] = []
                        self.spikes_isotopes["STD"][filename_short][isotope].append([start_index, end_index])

                self.do_spike_elimination_all_grubbs(filetype=filetype)
                self.fast_track_std = True
                self.counter_fast_track_std += 1
                self.container_var["Spike Elimination"]["STD"]["State"] = True
                #
                self.frm_spk_std.config(background=self.sign_green, bd=1)
                #
            elif filetype == "SMPL":
                for filename_long in self.list_smpl:
                    parts = filename_long.split("/")
                    filename_short = parts[-1]
                    #
                    self.container_measurements["RAW"][filename_short] = {}
                    self.container_measurements["EDITED"][filename_short] = {}
                    self.container_measurements["SELECTED"][filename_short] = {}
                    self.container_measurements["SELECTED"][filename_short]["RAW"] = {}
                    self.container_measurements["SELECTED"][filename_short]["SMOOTHED"] = {}
                    #
                    if self.diagrams_setup[filetype][filename_short]["FIG"] == None:
                        self.diagrams_setup[filetype][filename_short]["FIG"] = Figure(
                            figsize=(10, 5), facecolor=self.green_light)
                        self.diagrams_setup[filetype][filename_short]["AX"] = \
                            self.diagrams_setup[filetype][filename_short][
                                "FIG"].add_subplot()
                    #
                    if self.diagrams_setup[filetype][filename_short]["FIG_RATIO"] == None:
                        self.diagrams_setup[filetype][filename_short]["FIG_RATIO"] = Figure(
                            figsize=(10, 5), facecolor=self.green_light)
                        self.diagrams_setup[filetype][filename_short]["AX_RATIO"] = \
                            self.diagrams_setup[filetype][filename_short][
                                "FIG_RATIO"].add_subplot()

                    if self.file_loaded == False:
                        if self.container_icpms["name"] != None:
                            var_skipheader = self.container_icpms["skipheader"]
                            var_skipfooter = self.container_icpms["skipfooter"]
                            df_data = DE(filename_long=filename_long).get_measurements(
                                delimiter=",", skip_header=var_skipheader, skip_footer=var_skipfooter)
                        else:
                            df_data = DE(filename_long=filename_long).get_measurements(
                                delimiter=",", skip_header=3, skip_footer=1)
                    else:
                        try:
                            df_data = self.container_measurements["Dataframe"][filename_short]
                        except:
                            if "_copy" in filename_short:
                                filename_short_original = filename_short.replace("_copy", "")
                                filename_short = filename_short_original
                                df_data = self.container_measurements["Dataframe"][filename_short]
                            else:
                                df_data = self.container_measurements["Dataframe"][filename_short]

                    list_names = list(df_data.columns.values)
                    list_names.pop(0)
                    df_isotopes = list_names
                    times = DE().get_times(dataframe=df_data)
                    start_time = times.iloc[0]
                    end_time = times.iloc[-1]
                    start_index = times[times == start_time].index[0]
                    end_index = times[times == end_time].index[0]
                    spk_id = 1
                    #
                    box_spk = self.diagrams_setup[filetype][filename_short]["AX"].axvspan(
                        start_time, end_time, alpha=0.125, color=self.yellow_dark)
                    box_spk_ratio = self.diagrams_setup[filetype][filename_short]["AX_RATIO"].axvspan(
                        start_time, end_time, alpha=0.125, color=self.yellow_dark)
                    #
                    isotope = "".join(self.container_lists["ISOTOPES"])
                    #
                    self.container_measurements["RAW"][filename_short]["Time"] = times.tolist()
                    self.container_measurements["EDITED"][filename_short]["Time"] = times.tolist()
                    self.container_measurements["SELECTED"][filename_short]["Time"] = times.tolist()

                    # for isotope in self.container_lists["ISOTOPES"]:
                    for isotope in df_isotopes:
                        self.container_measurements["RAW"][filename_short][isotope] = df_data[isotope].tolist()
                        self.container_measurements["EDITED"][filename_short][isotope] = {}
                        self.container_measurements["SELECTED"][filename_short]["RAW"][isotope] = {}
                        self.container_measurements["SELECTED"][filename_short]["SMOOTHED"][isotope] = {}
                        #
                        self.container_measurements["EDITED"][filename_short][isotope]["BG"] = []
                        self.container_measurements["SELECTED"][filename_short]["RAW"][isotope]["BG"] = []
                        self.container_measurements["SELECTED"][filename_short]["SMOOTHED"][isotope]["BG"] = []
                        if mode == "MA":
                            self.container_measurements["EDITED"][filename_short][isotope]["MAT"] = []
                            self.container_measurements["SELECTED"][filename_short]["RAW"][isotope]["MAT"] = []
                            self.container_measurements["SELECTED"][filename_short]["SMOOTHED"][isotope]["MAT"] = []
                            self.container_measurements["SELECTED"][filename_short]["SMOOTHED"][isotope]["MAT"] = []
                            #
                            var_settings = "ma_setting"
                        elif mode in ["FI", "MI"]:
                            self.container_measurements["EDITED"][filename_short][isotope]["MAT"] = []
                            self.container_measurements["SELECTED"][filename_short]["RAW"][isotope]["MAT"] = []
                            self.container_measurements["SELECTED"][filename_short]["SMOOTHED"][isotope]["MAT"] = []
                            self.container_measurements["SELECTED"][filename_short]["SMOOTHED"][isotope]["MAT"] = []
                            self.container_measurements["EDITED"][filename_short][isotope]["INCL"] = []
                            self.container_measurements["SELECTED"][filename_short]["RAW"][isotope]["INCL"] = []
                            self.container_measurements["SELECTED"][filename_short]["SMOOTHED"][isotope][
                                "INCL"] = []
                            self.container_measurements["SELECTED"][filename_short]["SMOOTHED"][isotope][
                                "INCL"] = []
                            if mode == "FI":
                                var_settings = "fi_setting"
                            elif mode == "MI":
                                var_settings = "mi_setting"
                        #
                        self.spikes_isotopes["SMPL"][filename_short][isotope] = []
                        self.spikes_isotopes["SMPL"][filename_short][isotope].append([start_index, end_index])

                self.do_spike_elimination_all_grubbs(filetype=filetype)
                self.fast_track_smpl = True
                self.counter_fast_track_smpl += 1
                self.container_var["Spike Elimination"]["SMPL"]["State"] = True
                #
                if self.pysills_mode == "MA":
                    self.frm_spk_smpl.config(background=self.sign_green, bd=1)
                elif self.pysills_mode in ["FI", "INCL"]:
                    if self.container_var[key_setting]["Spike Elimination Inclusion"].get() == 1:
                        self.frm_spk_smpl.config(background=self.sign_green, bd=1)
                    elif self.container_var[key_setting]["Spike Elimination Inclusion"].get() == 2:
                        if len(self.container_helper["SMPL"][filename_short]["INCL"]["Content"]) == 0:
                            self.frm_spk_smpl.config(background=self.sign_yellow, bd=1)
                        else:
                            self.frm_spk_smpl.config(background=self.sign_green, bd=1)

        if filetype == "STD":
            self.btn_09e2.configure(state="normal")
            self.initial_spike_elimination_std = True
        else:
            self.btn_09f2.configure(state="normal")
            self.initial_spike_elimination_smpl = True

    def change_carrier_gas(self, var_opt):
        if var_opt == "Helium":
            try:
                self.var_entr_10.set("24.587")
                var_energy = self.var_entr_10.get()
            except:
                self.container_var["Gas Energy"].set("24.587")
                var_energy = self.container_var["Gas Energy"].get()
            #
            for isotope in self.container_lists["ISOTOPES"]:
                key_element = re.search(r"(\D+)(\d+)", isotope)
                element = key_element.group(1)
                if float(var_energy) >= float(self.ionization_energies["First"][element]) \
                        and float(var_energy) >= float(self.ionization_energies["Second"][element]):
                    fg_color = self.accent_color
                    bg_color = self.bg_colors["Very Light"]
                    self.container_var["charge"][isotope]["textvar"].set("2+ charged")
                    self.container_var["charge"][isotope]["labelvar"].config(bg=bg_color, fg=fg_color)
                else:
                    fg_color = self.bg_colors["Dark Font"]
                    bg_color = self.bg_colors["Very Light"]
                    self.container_var["charge"][isotope]["textvar"].set("1+ charged")
                    self.container_var["charge"][isotope]["labelvar"].config(bg=bg_color, fg=fg_color)
            #
        elif var_opt == "Neon":
            try:
                self.var_entr_10.set("21.565")
                var_energy = self.var_entr_10.get()
            except:
                self.container_var["Gas Energy"].set("21.565")
                var_energy = self.container_var["Gas Energy"].get()
            #
            for isotope in self.container_lists["ISOTOPES"]:
                key_element = re.search(r"(\D+)(\d+)", isotope)
                element = key_element.group(1)
                if float(var_energy) >= float(self.ionization_energies["First"][element]) \
                        and float(var_energy) >= float(self.ionization_energies["Second"][element]):
                    fg_color = self.accent_color
                    bg_color = self.bg_colors["Very Light"]
                    self.container_var["charge"][isotope]["textvar"].set("2+ charged")
                    self.container_var["charge"][isotope]["labelvar"].config(bg=bg_color, fg=fg_color)
                else:
                    fg_color = self.bg_colors["Very Dark"]
                    bg_color = self.bg_colors["Very Light"]
                    self.container_var["charge"][isotope]["textvar"].set("1+ charged")
                    self.container_var["charge"][isotope]["labelvar"].config(bg=bg_color, fg=fg_color)
        elif var_opt == "Argon":
            try:
                self.var_entr_10.set("15.760")
                var_energy = self.var_entr_10.get()
            except:
                self.container_var["Gas Energy"].set("15.760")
                var_energy = self.container_var["Gas Energy"].get()
            #
            for isotope in self.container_lists["ISOTOPES"]:
                key_element = re.search(r"(\D+)(\d+)", isotope)
                element = key_element.group(1)
                if float(var_energy) >= float(self.ionization_energies["First"][element]) \
                        and float(var_energy) >= float(self.ionization_energies["Second"][element]):
                    fg_color = self.accent_color
                    bg_color = self.bg_colors["Very Light"]
                    self.container_var["charge"][isotope]["textvar"].set("2+ charged")
                    self.container_var["charge"][isotope]["labelvar"].config(bg=bg_color, fg=fg_color)
                else:
                    fg_color = self.bg_colors["Very Dark"]
                    bg_color = self.bg_colors["Very Light"]
                    self.container_var["charge"][isotope]["textvar"].set("1+ charged")
                    self.container_var["charge"][isotope]["labelvar"].config(bg=bg_color, fg=fg_color)
        elif var_opt == "Krypton":
            try:
                self.var_entr_10.set("14.000")
                var_energy = self.var_entr_10.get()
            except:
                self.container_var["Gas Energy"].set("14.000")
                var_energy = self.container_var["Gas Energy"].get()
            #
            for isotope in self.container_lists["ISOTOPES"]:
                key_element = re.search(r"(\D+)(\d+)", isotope)
                element = key_element.group(1)
                if float(var_energy) >= float(self.ionization_energies["First"][element]) \
                        and float(var_energy) >= float(self.ionization_energies["Second"][element]):
                    fg_color = self.accent_color
                    bg_color = self.bg_colors["Very Light"]
                    self.container_var["charge"][isotope]["textvar"].set("2+ charged")
                    self.container_var["charge"][isotope]["labelvar"].config(bg=bg_color, fg=fg_color)
                else:
                    fg_color = self.bg_colors["Very Dark"]
                    bg_color = self.bg_colors["Very Light"]
                    self.container_var["charge"][isotope]["textvar"].set("1+ charged")
                    self.container_var["charge"][isotope]["labelvar"].config(bg=bg_color, fg=fg_color)
        elif var_opt == "Xenon":
            try:
                self.var_entr_10.set("12.130")
                var_energy = self.var_entr_10.get()
            except:
                self.container_var["Gas Energy"].set("12.130")
                var_energy = self.container_var["Gas Energy"].get()
            #
            for isotope in self.container_lists["ISOTOPES"]:
                key_element = re.search(r"(\D+)(\d+)", isotope)
                element = key_element.group(1)
                if float(var_energy) >= float(self.ionization_energies["First"][element]) \
                        and float(var_energy) >= float(self.ionization_energies["Second"][element]):
                    fg_color = self.accent_color
                    bg_color = self.bg_colors["Very Light"]
                    self.container_var["charge"][isotope]["textvar"].set("2+ charged")
                    self.container_var["charge"][isotope]["labelvar"].config(bg=bg_color, fg=fg_color)
                else:
                    fg_color = self.bg_colors["Very Dark"]
                    bg_color = self.bg_colors["Very Light"]
                    self.container_var["charge"][isotope]["textvar"].set("1+ charged")
                    self.container_var["charge"][isotope]["labelvar"].config(bg=bg_color, fg=fg_color)
        elif var_opt == "Radon":
            try:
                self.var_entr_10.set("10.749")
                var_energy = self.var_entr_10.get()
            except:
                self.container_var["Gas Energy"].set("10.749")
                var_energy = self.container_var["Gas Energy"].get()
            #
            for isotope in self.container_lists["ISOTOPES"]:
                key_element = re.search(r"(\D+)(\d+)", isotope)
                element = key_element.group(1)
                if float(var_energy) >= float(self.ionization_energies["First"][element]) \
                        and float(var_energy) >= float(self.ionization_energies["Second"][element]):
                    fg_color = self.accent_color
                    bg_color = self.bg_colors["Very Light"]
                    self.container_var["charge"][isotope]["textvar"].set("2+ charged")
                    self.container_var["charge"][isotope]["labelvar"].config(bg=bg_color, fg=fg_color)
                else:
                    fg_color = self.bg_colors["Very Dark"]
                    bg_color = self.bg_colors["Very Light"]
                    self.container_var["charge"][isotope]["textvar"].set("1+ charged")
                    self.container_var["charge"][isotope]["labelvar"].config(bg=bg_color, fg=fg_color)

    def change_std_is(self, element, file):
        self.container_var["STD"][file]["IS"].set(element)
        parts = file.split("/")
        self.container_files["STD"][parts[-1]]["IS"].set(element)

    #
    def change_std_is_default(self, element, mineral=None):
        self.container_var["IS"]["Default STD"].set(element)
        for file in self.list_std:
            self.container_var["STD"][file]["IS"].set(element)
            parts = file.split("/")
            self.container_files["STD"][parts[-1]]["IS"].set(element)
        #
        if element in self.container_lists["ISOTOPES"]:
            key_is = re.search(r"(\D+)(\d+)", element)
            element_is = key_is.group(1)
        if mineral != None and mineral != "Select Mineral":
            if element in self.mineral_chem[mineral]:
                # self.var_entr_09.set(self.mineral_chem[mineral][element])
                self.container_var["settings"]["IS Concentration"].set(self.mineral_chem[mineral][element])
            elif element_is in self.mineral_chem[mineral]:
                # self.var_entr_09.set(self.mineral_chem[mineral][element_is])
                self.container_var["settings"]["IS Concentration"].set(self.mineral_chem[mineral][element_is])
            else:
                # self.var_entr_09.set(0.0)
                self.container_var["settings"]["IS Concentration"].set(0.0)
            #
            if self.container_var["IS"]["Default SMPL"].get() == "Select IS":
                self.change_smpl_is_default(element=element, mineral=mineral)
            #
        else:
            warning_is = tk.Toplevel(self.parent)
            warning_is.title("Warning")
            warning_is.geometry("250x100+0+0")
            warning_is.resizable(False, False)
            warning_is["bg"] = self.green_light
            #
            window_width = 250
            window_height = 100
            row_min = 25
            n_rows = int(window_height/row_min)
            column_min = 20
            n_columns = int(window_width/column_min)
            #
            for x in range(n_columns):
                tk.Grid.columnconfigure(warning_is, x, weight=1)
            for y in range(n_rows):
                tk.Grid.rowconfigure(warning_is, y, weight=1)
            #
            # Rows
            for i in range(0, n_rows):
                warning_is.grid_rowconfigure(i, minsize=row_min)
            # Columns
            for i in range(0, n_columns):
                warning_is.grid_columnconfigure(i, minsize=column_min)
            #
            lbl_01 = SE(
                parent=warning_is, row_id=0, column_id=0, n_rows=n_rows, n_columns=n_columns, fg=self.green_dark,
                bg=self.red_dark).create_simple_label(
                text="Please load your internal\n standard concentration data\n or select a mineral", relief=tk.GROOVE,
                fontsize="sans 10 bold")
            #
            self.container_var["IS"]["Default STD"].set("Select IS")

    #
    def change_smpl_is(self, element, file, mineral=None):
        self.container_var["SMPL"][file]["IS"].set(element)
        if mineral != None:
            parts = file.split("/")
            key_i = re.search(r"(\D+)(\d+)", element)
            var_element = key_i.group(1)
            self.container_files["SMPL"][parts[-1]]["IS Concentration"].set(self.mineral_chem[mineral][var_element])
            #
            self.container_var["SMPL"][file]["IS Data"]["IS"].set(element)
            self.container_var["SMPL"][file]["IS Data"]["Concentration"].set(
                self.mineral_chem[mineral][element])
            #

    #
    def change_smpl_is_default(self, element, mineral=None):
        self.container_var["IS"]["Default SMPL"].set(element)
        for file in self.list_smpl:
            self.container_var["SMPL"][file]["IS"].set(element)
        #
        if element in self.container_lists["ISOTOPES"]:
            key_is = re.search(r"(\D+)(\d+)", element)
            element_is = key_is.group(1)
        if mineral != None and mineral != "Select Mineral":
            if element in self.mineral_chem[mineral]:
                # self.var_entr_09.set(self.mineral_chem[mineral][element])
                self.container_var["settings"]["IS Concentration"].set(self.mineral_chem[mineral][element])
                for file in self.list_smpl:
                    parts = file.split("/")
                    self.container_files["SMPL"][parts[-1]]["IS"].set(element)
                    self.container_files["SMPL"][parts[-1]]["IS Concentration"].set(self.mineral_chem[mineral][element])
                    #
                    self.container_var["SMPL"][file]["IS Data"]["IS"].set(element)
                    self.container_var["SMPL"][file]["IS Data"]["Concentration"].set(
                        self.mineral_chem[mineral][element])
                    #
            elif element_is in self.mineral_chem[mineral]:
                # self.var_entr_09.set(self.mineral_chem[mineral][element_is])
                self.container_var["settings"]["IS Concentration"].set(self.mineral_chem[mineral][element_is])
                for file in self.list_smpl:
                    parts = file.split("/")
                    self.container_files["SMPL"][parts[-1]]["IS"].set(element)
                    self.container_files["SMPL"][parts[-1]]["IS Concentration"].set(
                        self.mineral_chem[mineral][element_is])
                    #
                    self.container_var["SMPL"][file]["IS Data"]["IS"].set(element)
                    self.container_var["SMPL"][file]["IS Data"]["Concentration"].set(
                        self.mineral_chem[mineral][element_is])
                    #
            #
            if self.container_var["IS"]["Default SMPL"].get() == "Select IS":
                self.change_std_is_default(element=element, mineral=mineral)
        else:
            warning_is = tk.Toplevel(self.parent)
            warning_is.title("Warning")
            warning_is.geometry("250x100+0+0")
            warning_is.resizable(False, False)
            warning_is["bg"] = self.green_light
            #
            window_width = 250
            window_height = 100
            row_min = 25
            n_rows = int(window_height/row_min)
            column_min = 20
            n_columns = int(window_width/column_min)
            #
            for x in range(n_columns):
                tk.Grid.columnconfigure(warning_is, x, weight=1)
            for y in range(n_rows):
                tk.Grid.rowconfigure(warning_is, y, weight=1)
            #
            # Rows
            for i in range(0, n_rows):
                warning_is.grid_rowconfigure(i, minsize=row_min)
            # Columns
            for i in range(0, n_columns):
                warning_is.grid_columnconfigure(i, minsize=column_min)
            #
            lbl_01 = SE(
                parent=warning_is, row_id=0, column_id=0, n_rows=n_rows, n_columns=n_columns, fg=self.green_dark,
                bg=self.red_dark).create_simple_label(
                text="Please load your internal\n standard concentration data\n or select a mineral", relief=tk.GROOVE,
                fontsize="sans 10 bold")
            #
            self.container_var["IS"]["Default SMPL"].set("Select IS")

    def change_srm_default(self, var_srm, key="STD"):
        if key == "STD":
            for file in self.list_std:
                parts = file.split("/")
                self.container_var["STD"][file]["SRM"].set(var_srm)
                self.container_files["STD"][parts[-1]]["SRM"].set(var_srm)
                try:
                    self.container_var["SRM"][file].set(var_srm)
                except:
                    print(file, self.container_var["SRM"][file])
            if self.container_var["SRM"]["default"][1].get() == "Select SRM":
                self.container_var["SRM"]["default"][1].set(var_srm)
                for isotope in self.container_lists["ISOTOPES"]:
                    self.container_files["SRM"][isotope].set(var_srm)

                    try:
                        self.container_var["SRM"][isotope].set(var_srm)
                    except:
                        print(isotope, self.container_var["SRM"][isotope])
        elif key == "isotope":
            for isotope in self.container_lists["ISOTOPES"]:
                self.container_files["SRM"][isotope].set(var_srm)

                try:
                    self.container_var["SRM"][isotope].set(var_srm)
                except:
                    print(isotope, self.container_var["SRM"][isotope])
            if self.container_var["SRM"]["default"][0].get() == "Select SRM":
                self.container_var["SRM"]["default"][0].set(var_srm)
                for file in self.list_std:
                    parts = file.split("/")
                    self.container_var["STD"][file]["SRM"].set(var_srm)
                    self.container_files["STD"][parts[-1]]["SRM"].set(var_srm)
                    try:
                        self.container_var["SRM"][file].set(var_srm)
                    except:
                        print(file, self.container_var["SRM"][file])

        self.fill_srm_values(var_srm=var_srm)

    def quick_plot_file(self, var_filetype, event):
        if var_filetype == "STD":
            click_id = self.lb_std.curselection()
            var_fig = self.quick_plot_figure_std
        elif var_filetype == "STD Manager":
            click_id = self.lb_std_manager.curselection()
            var_fig = self.quick_plot_figure_std
            var_filetype = "STD"
        elif var_filetype == "SMPL":
            click_id = self.lb_smpl.curselection()
            var_fig = self.quick_plot_figure_smpl
        elif var_filetype == "SMPL Manager":
            click_id = self.lb_smpl_manager.curselection()
            var_fig = self.quick_plot_figure_smpl
            var_filetype = "SMPL"

        click_id = click_id[0]
        file_long = self.container_lists[var_filetype]["Long"][click_id]

        self.build_quick_plot_window(filetype=var_filetype, filename_long=file_long)

    def previous_quick_plot_file(self, filetype, filename_long, subwindow):
        index_filename = self.container_lists[filetype]["Long"].index(filename_long)

        if index_filename > 0:
            filename_long = self.container_lists[filetype]["Long"][index_filename - 1]
        else:
            filename_long = self.container_lists[filetype]["Long"][-1]

        subwindow.destroy()

        self.build_quick_plot_window(filetype=filetype, filename_long=filename_long)

    def next_quick_plot_file(self, filetype, filename_long, subwindow):
        index_filename = self.container_lists[filetype]["Long"].index(filename_long)

        if index_filename < len(self.container_lists[filetype]["Long"]) - 1:
            filename_long = self.container_lists[filetype]["Long"][index_filename + 1]
        else:
            filename_long = self.container_lists[filetype]["Long"][0]

        subwindow.destroy()

        self.build_quick_plot_window(filetype=filetype, filename_long=filename_long)

    def build_quick_plot_window(self, filetype, filename_long):
        var_filetype = filetype
        file_long = filename_long
        file_short = filename_long.split("/")[-1]
        file_isotopes = self.container_lists["Measured Isotopes"][file_short]

        font_color_dark = self.bg_colors["Dark Font"]
        font_color_light = self.bg_colors["Light Font"]
        background_color_dark = self.bg_colors["BG Window"]
        background_color_elements = self.bg_colors["Light"]
        background_color_light = self.bg_colors["Very Light"]
        accent_color = self.bg_colors["Accent"]  # self.accent_color
        font_header = self.font_settings["Header"]
        font_elements = self.font_settings["Elements"]
        font_option = self.font_settings["Options"]
        font_table = self.font_settings["Table"]

        ## Window Settings
        row_min = self.row_height
        column_min = self.column_width
        n_rows = self.window_dimensions["Quick plot"][0]
        n_columns = self.window_dimensions["Quick plot"][1]

        window_width = int(n_columns*self.column_width)
        window_height = int(n_rows*self.row_height)

        var_geometry = str(window_width) + "x" + str(window_height) + "+" + str(0) + "+" + str(0)

        subwindow_quickplotter = tk.Toplevel(self.parent)
        subwindow_quickplotter.title("Quick Plotter")
        subwindow_quickplotter.geometry(var_geometry)
        subwindow_quickplotter.resizable(False, False)
        subwindow_quickplotter["bg"] = self.bg_colors["Very Light"]

        for x in range(n_columns):
            tk.Grid.columnconfigure(subwindow_quickplotter, x, weight=1)
        for y in range(n_rows):
            tk.Grid.rowconfigure(subwindow_quickplotter, y, weight=1)

        # Rows
        for i in range(0, n_rows):
            subwindow_quickplotter.grid_rowconfigure(i, minsize=row_min)
        # Columns
        for i in range(0, n_columns):
            subwindow_quickplotter.grid_columnconfigure(i, minsize=column_min)

        ###########################################################

        row_start = 0
        column_start = 0
        n_rows_isotopes = n_rows - 15
        n_columns_isotopes = 8
        n_rows_diagram = n_rows_isotopes + 2
        n_columns_diagram = int(0.6*n_columns)
        n_columns_statistics = n_columns - n_columns_isotopes - n_columns_diagram
        row_display_opt = n_rows_isotopes + 1
        row_interval_opt = row_display_opt + 4
        n_columns_sections = int(n_columns_diagram/3)

        ## TRANSLATIONS
        str_var_01 = self.language_dict["Time-Signal Diagram"][self.var_language]
        str_var_02 = self.language_dict["Isotopes"][self.var_language]
        str_var_03 = self.language_dict["Data analysis"][self.var_language]
        str_var_04 = self.language_dict["Display options"][self.var_language]
        str_var_05 = self.language_dict["Interval options"][self.var_language]
        str_var_06 = self.language_dict["Background"][self.var_language]
        str_var_07 = self.language_dict["Matrix"][self.var_language]
        str_var_08 = self.language_dict["Inclusion"][self.var_language]
        str_var_09 = self.language_dict["Reference isotope"][self.var_language]
        str_var_10 = self.language_dict["Internal standard"][self.var_language]
        str_var_11 = self.language_dict["No selection"][self.var_language]
        str_var_12 = self.language_dict["Show all"][self.var_language]
        str_var_13 = self.language_dict["Hide all"][self.var_language]
        str_var_14 = self.language_dict["Remove interval"][self.var_language]
        str_var_15 = self.language_dict["Previous"][self.var_language]
        str_var_16 = self.language_dict["Next"][self.var_language]
        str_var_17 = self.language_dict["Additional analysis"][self.var_language]
        str_var_18 = self.language_dict["Sample"][self.var_language]

        ## LABELS
        lbl_01 = SE(
            parent=subwindow_quickplotter, row_id=row_start, column_id=column_start + n_columns_isotopes, n_rows=1,
            n_columns=n_columns_diagram, fg=self.bg_colors["Light Font"],
            bg=self.bg_colors["BG Window"]).create_simple_label(
            text=str_var_01, relief=tk.FLAT, fontsize=font_header, anchor=tk.W)
        lbl_02 = SE(
            parent=subwindow_quickplotter, row_id=row_start, column_id=column_start, n_rows=1,
            n_columns=n_columns_isotopes, fg=self.bg_colors["Light Font"],
            bg=self.bg_colors["BG Window"]).create_simple_label(
            text=str_var_02, relief=tk.FLAT, fontsize=font_header, anchor=tk.W)
        lbl_03 = SE(
            parent=subwindow_quickplotter, row_id=row_start,
            column_id=column_start + n_columns_isotopes + n_columns_diagram, n_rows=1, n_columns=n_columns_statistics,
            fg=self.bg_colors["Light Font"], bg=self.bg_colors["BG Window"]).create_simple_label(
            text=str_var_03, relief=tk.FLAT, fontsize=font_header, anchor=tk.W)
        lbl_04 = SE(
            parent=subwindow_quickplotter, row_id=row_display_opt, column_id=column_start, n_rows=1,
            n_columns=n_columns_isotopes, fg=self.bg_colors["Light Font"],
            bg=self.bg_colors["BG Window"]).create_simple_label(
            text=str_var_04, relief=tk.FLAT, fontsize=font_header, anchor=tk.W)
        lbl_05 = SE(
            parent=subwindow_quickplotter, row_id=row_interval_opt, column_id=column_start, n_rows=1,
            n_columns=n_columns_isotopes, fg=self.bg_colors["Light Font"],
            bg=self.bg_colors["BG Window"]).create_simple_label(
            text=str_var_05, relief=tk.FLAT, fontsize=font_header, anchor=tk.W)
        lbl_06 = SE(
            parent=subwindow_quickplotter, row_id=n_rows - 3, column_id=column_start, n_rows=1,
            n_columns=n_columns_isotopes, fg=self.bg_colors["Light Font"],
            bg=self.bg_colors["BG Window"]).create_simple_label(
            text="", relief=tk.FLAT, fontsize=font_header, anchor=tk.W)
        lbl_07 = SE(
            parent=subwindow_quickplotter, row_id=row_interval_opt, column_id=column_start + n_columns_isotopes,
            n_rows=1, n_columns=n_columns_sections, fg=self.colors_intervals["BG LB"],
            bg=self.colors_intervals["BG"]).create_simple_label(
            text=str_var_06, relief=tk.FLAT, fontsize=font_header, anchor=tk.W)
        lbl_08 = SE(
            parent=subwindow_quickplotter, row_id=row_interval_opt,
            column_id=column_start + n_columns_isotopes + n_columns_sections, n_rows=1, n_columns=n_columns_sections,
            fg=self.colors_intervals["MAT LB"], bg=self.colors_intervals["MAT"]).create_simple_label(
            text=str_var_07, relief=tk.FLAT, fontsize=font_header, anchor=tk.W)
        lbl_09 = SE(
            parent=subwindow_quickplotter, row_id=row_interval_opt,
            column_id=column_start + n_columns_isotopes + 2*n_columns_sections, n_rows=1, n_columns=n_columns_sections,
            fg=self.colors_intervals["INCL LB"], bg=self.colors_intervals["INCL"]).create_simple_label(
            text=str_var_08, relief=tk.FLAT, fontsize=font_header, anchor=tk.W)

        if var_filetype == "STD":
            lbl_10 = SE(
                parent=subwindow_quickplotter, row_id=row_start + 1,
                column_id=column_start + n_columns_isotopes + n_columns_diagram, n_rows=1,
                n_columns=int(n_columns_statistics/2), fg=self.bg_colors["Dark Font"],
                bg=self.bg_colors["Very Light"]).create_simple_label(
                text=str_var_09, relief=tk.FLAT, fontsize=font_elements, anchor=tk.W)
        elif var_filetype == "SMPL":
            lbl_10 = SE(
                parent=subwindow_quickplotter, row_id=row_start + 1,
                column_id=column_start + n_columns_isotopes + n_columns_diagram, n_rows=1,
                n_columns=int(n_columns_statistics/2), fg=self.bg_colors["Dark Font"],
                bg=self.bg_colors["Very Light"]).create_simple_label(
                text=str_var_10, relief=tk.FLAT, fontsize=font_elements, anchor=tk.W)

        ## OPTION MENU
        self.qpl_opt_is = tk.StringVar()
        self.qpl_opt_is.set("Select isotope")

        opt_is = SE(
            parent=subwindow_quickplotter, row_id=row_start + 1,
            column_id=column_start + n_columns_isotopes + n_columns_diagram + int(n_columns_statistics/2), n_rows=1,
            n_columns=int(n_columns_statistics/2), fg=font_color_dark,
            bg=background_color_elements).create_option_isotope(
            var_iso=self.qpl_opt_is, option_list=file_isotopes, text_set=self.qpl_opt_is.get(),
            fg_active=font_color_light, bg_active=accent_color)

        ## RADIOBUTTONS
        rb_01 = SE(
            parent=subwindow_quickplotter, row_id=row_interval_opt + 2, column_id=column_start, n_rows=1,
            n_columns=n_columns_isotopes, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_radiobutton(
            var_rb=self.container_var["Radiobuttons"]["Quick Plot File"], value_rb=0, color_bg=self.bg_colors["Light"],
            fg=self.bg_colors["Dark Font"], text=str_var_06, sticky="nesw", relief=tk.FLAT,
            command=lambda var_type=var_filetype, var_file_short=file_short, list_isotopes=file_isotopes, section="BG":
            self.fill_tv_quick_plot_file(var_type, var_file_short, list_isotopes, section))
        rb_02 = SE(
            parent=subwindow_quickplotter, row_id=row_interval_opt + 3, column_id=column_start, n_rows=1,
            n_columns=n_columns_isotopes, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_radiobutton(
            var_rb=self.container_var["Radiobuttons"]["Quick Plot File"], value_rb=1, color_bg=self.bg_colors["Light"],
            fg=self.bg_colors["Dark Font"], text=str_var_07, sticky="nesw", relief=tk.FLAT,
            command=lambda var_type=var_filetype, var_file_short=file_short, list_isotopes=file_isotopes, section="MAT":
            self.fill_tv_quick_plot_file(var_type, var_file_short, list_isotopes, section))
        rb_03 = SE(
            parent=subwindow_quickplotter, row_id=row_interval_opt + 4, column_id=column_start, n_rows=1,
            n_columns=n_columns_isotopes, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_radiobutton(
            var_rb=self.container_var["Radiobuttons"]["Quick Plot File"], value_rb=2, color_bg=self.bg_colors["Light"],
            fg=self.bg_colors["Dark Font"], text=str_var_08, sticky="nesw", relief=tk.FLAT,
            command=lambda var_type=var_filetype, var_file_short=file_short, list_isotopes=file_isotopes,
                           section="INCL":
            self.fill_tv_quick_plot_file(var_type, var_file_short, list_isotopes, section))
        rb_04 = SE(
            parent=subwindow_quickplotter, row_id=row_interval_opt + 1, column_id=column_start, n_rows=1,
            n_columns=n_columns_isotopes, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_radiobutton(
            var_rb=self.container_var["Radiobuttons"]["Quick Plot File"], value_rb=3, color_bg=self.bg_colors["Light"],
            fg=self.bg_colors["Dark Font"], text=str_var_11, sticky="nesw", relief=tk.FLAT)

        if self.file_loaded == False:
            if self.container_icpms["name"] != None:
                var_skipheader = self.container_icpms["skipheader"]
                var_skipfooter = self.container_icpms["skipfooter"]
                df_data = DE(filename_long=file_long).get_measurements(
                    delimiter=",", skip_header=var_skipheader, skip_footer=var_skipfooter)
            else:
                df_data = DE(filename_long=file_long).get_measurements(
                    delimiter=",", skip_header=3, skip_footer=1)
        else:
            if file_short not in self.container_measurements["Dataframe"] and "_copy" in file_short:
                file_short_original = file_short.replace("_copy", "")
                df_data = self.container_measurements["Dataframe"][file_short_original]
            else:
                df_data = self.container_measurements["Dataframe"][file_short]

        if file_short not in self.container_measurements["Dataframe"]:
            self.container_measurements["Dataframe"][file_short] = df_data

        list_keys = list(df_data.columns.values)
        del list_keys[0]
        dataset_time = list(df_data.iloc[:, 0])
        self.dataset_time = dataset_time
        x_max = max(dataset_time)

        if len(file_isotopes) == 0:
            self.container_lists["ISOTOPES"] = list_keys
            self.define_isotope_colors()

        icp_measurements = np.array([[df_data[isotope] for isotope in file_isotopes]])
        y_max = np.amax(icp_measurements)

        ## ISOTOPE LIST
        frm_isotopes = SE(
            parent=subwindow_quickplotter, row_id=row_start + 1, column_id=column_start, n_rows=n_rows_isotopes,
            n_columns=n_columns_isotopes, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Very Light"]).create_frame(relief=tk.FLAT)
        vsb = ttk.Scrollbar(master=frm_isotopes, orient="vertical")
        text = tk.Text(
            master=frm_isotopes, width=30, height=15, yscrollcommand=vsb.set, bg=self.bg_colors["Very Light"])
        vsb.config(command=text.yview)
        vsb.pack(side="right", fill="y")
        text.pack(side="left", fill="both", expand=True)

        self.temp_checkbuttons = {}
        for index, isotope in enumerate(file_isotopes):
            self.temp_checkbuttons[isotope] = tk.IntVar(value=1)

            cb_i = tk.Checkbutton(
                master=frm_isotopes, text=isotope, fg=self.bg_colors["Dark Font"], bg=self.bg_colors["Very Light"],
                variable=self.temp_checkbuttons[isotope],
                command=lambda var_isotope=isotope, var_file_short=file_short:
                self.show_line(var_isotope, var_file_short))

            text.window_create("end", window=cb_i)
            text.insert("end", "\n")
            cb_i.select()
        text.configure(state="disabled")

        ## DIAGRAMS
        var_fig = Figure(figsize=(10, 5), tight_layout=True, facecolor=self.bg_colors["Very Light"])
        var_ax = var_fig.add_subplot(label=np.random.uniform())

        self.temp_lines = {}

        var_lw = float(self.container_var["General Settings"]["Line width"].get())
        if var_lw < 0.25:
            var_lw = 0.25
        elif var_lw > 2.5:
            var_lw = 2.5

        try:
            for isotope in file_isotopes:
                ln = var_ax.plot(
                    dataset_time, df_data[isotope], label=isotope, color=self.isotope_colors[isotope],
                    linewidth=var_lw, visible=True)
                self.temp_lines[isotope] = ln
        except:
            self.define_temporary_colors(filename_short=file_short)
            for isotope in file_isotopes:
                ln = var_ax.plot(
                    dataset_time, df_data[isotope], label=isotope, color=self.isotope_colors_temporary[isotope],
                    linewidth=var_lw, visible=True)
                self.temp_lines[isotope] = ln

        var_ax.grid(True)
        var_ax.set_yscale("log")
        var_ax.set_xlim(left=0, right=x_max)
        var_ax.set_xticks(np.arange(0, x_max, 20))
        var_ax.set_ylim(top=1.5*y_max)
        var_ax.grid(which="major", linestyle="-", linewidth=1)
        var_ax.minorticks_on()
        var_ax.grid(which="minor", linestyle=":", linewidth=0.5, alpha=0.75)
        var_ax.set_axisbelow(True)
        var_ax.set_title(file_short, fontsize=9)
        var_ax.set_xlabel("Time (s)", labelpad=0.5, fontsize=8)
        var_ax.set_ylabel("Signal (cps)", labelpad=0.5, fontsize=8)
        var_ax.xaxis.set_tick_params(labelsize=8)
        var_ax.yaxis.set_tick_params(labelsize=8)
        self.var_ax = var_ax

        # Canvas
        self.var_canvas_qpl = FigureCanvasTkAgg(var_fig, master=subwindow_quickplotter)
        self.var_canvas_qpl.get_tk_widget().grid(
            row=row_start + 1, column=column_start + n_columns_isotopes, rowspan=n_rows_diagram,
            columnspan=n_columns_diagram, sticky="nesw")
        # Toolbar
        var_toolbarFrame = tk.Frame(master=subwindow_quickplotter)
        var_toolbarFrame.grid(
            row=n_rows_diagram + 1, column=column_start + n_columns_isotopes, rowspan=2, columnspan=n_columns_diagram,
            sticky="ew")
        var_toolbar = NavigationToolbar2Tk(self.var_canvas_qpl, var_toolbarFrame)
        var_toolbar.config(
            bg=self.bg_colors["Very Light"], highlightthickness=0, highlightbackground=self.bg_colors["Very Light"],
            highlightcolor=self.bg_colors["Dark Font"], bd=0)
        var_toolbar._message_label.config(
            background=self.bg_colors["Very Light"], fg=self.bg_colors["Dark Font"], font="sans 12")
        var_toolbar.winfo_children()[-2].config(background=self.bg_colors["Very Light"], fg=self.bg_colors["Dark Font"])

        self.container_var["Plotting"][self.pysills_mode]["Quickview"][file_short] = {
            "Figure": var_fig, "Canvas": self.var_canvas_qpl, "Toolbar": var_toolbar, "Axes": var_ax}

        self.var_canvas_qpl.mpl_connect(
            "button_press_event", lambda event, var_type=filetype, var_file_short=file_short:
            self.add_interval_to_diagram_qpl(var_type, var_file_short, event))

        ## BUTTONS
        btn_01 = SE(
            parent=subwindow_quickplotter, row_id=row_display_opt + 1, column_id=column_start, n_rows=1,
            n_columns=n_columns_isotopes, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_simple_button(
            text=str_var_12, bg_active=self.accent_color, fg_active=self.bg_colors["Dark Font"],
            command=lambda var_file_short=file_short: self.show_all_lines(var_file_short))
        btn_02 = SE(
            parent=subwindow_quickplotter, row_id=row_display_opt + 2,
            column_id=column_start, n_rows=1, n_columns=n_columns_isotopes,
            fg=self.bg_colors["Dark Font"], bg=self.bg_colors["Light"]).create_simple_button(
            text=str_var_13, bg_active=self.accent_color, fg_active=self.bg_colors["Dark Font"],
            command=lambda var_file_short=file_short: self.hide_all_lines(var_file_short))
        btn_06 = SE(
            parent=subwindow_quickplotter, row_id=row_display_opt + 3,
            column_id=column_start, n_rows=1, n_columns=n_columns_isotopes,
            fg=self.bg_colors["Dark Font"], bg=self.bg_colors["Light"]).create_simple_button(
            text=str_var_17, bg_active=self.accent_color, fg_active=self.bg_colors["Dark Font"],
            command=lambda var_type=var_filetype, var_file_short=file_short:
            self.qpl_additional_diagrams(var_type, var_file_short))
        btn_03 = SE(
            parent=subwindow_quickplotter, row_id=n_rows - 2,
            column_id=column_start, n_rows=2, n_columns=int(n_columns_isotopes/2),
            fg=self.bg_colors["Dark Font"], bg=self.bg_colors["Light"]).create_simple_button(
            text=str_var_15, bg_active=self.accent_color, fg_active=self.bg_colors["Dark Font"],
            command=lambda filetype=var_filetype, filename_long=file_long, subwindow=subwindow_quickplotter:
            self.previous_quick_plot_file(filetype, filename_long, subwindow))
        btn_04 = SE(
            parent=subwindow_quickplotter, row_id=n_rows - 2,
            column_id=column_start + int(n_columns_isotopes/2), n_rows=2, n_columns=int(n_columns_isotopes/2),
            fg=self.bg_colors["Dark Font"], bg=self.bg_colors["Light"]).create_simple_button(
            text=str_var_16, bg_active=self.accent_color, fg_active=self.bg_colors["Dark Font"],
            command=lambda filetype=var_filetype, filename_long=file_long, subwindow=subwindow_quickplotter:
            self.next_quick_plot_file(filetype, filename_long, subwindow))
        btn_05 = SE(
            parent=subwindow_quickplotter, row_id=row_interval_opt + 5,
            column_id=column_start, n_rows=2, n_columns=n_columns_isotopes,
            fg=self.bg_colors["Dark Font"], bg=self.bg_colors["Light"]).create_simple_button(
            text=str_var_14, bg_active=self.accent_color, fg_active=self.bg_colors["Dark Font"],
            command=lambda var_type=var_filetype, var_file_short=file_short:
            self.remove_interval_qpl(var_type, var_file_short))

        btn_01.configure(font=font_elements)
        btn_02.configure(font=font_elements)
        btn_03.configure(font=font_elements)
        btn_04.configure(font=font_elements)
        btn_05.configure(font=font_elements)
        btn_06.configure(font=font_elements)

        ## CHECKBOXES
        self.qpl_cb_bg = tk.IntVar()
        self.qpl_cb_mat = tk.IntVar()
        self.qpl_cb_incl = tk.IntVar()
        self.qpl_cb_bg.set(1)
        self.qpl_cb_mat.set(1)
        self.qpl_cb_incl.set(1)

        cb_01 = SE(
            parent=subwindow_quickplotter, row_id=row_interval_opt,
            column_id=column_start + n_columns_isotopes + (n_columns_sections - 1), n_rows=1, n_columns=1,
            fg=self.bg_colors["Dark Font"], bg=self.colors_intervals["BG"]).create_simple_checkbox(
            var_cb=self.qpl_cb_bg, text="", set_sticky="nesw", own_color=True,
            command=lambda key="Background", var_type=var_filetype, var_file_short=file_short:
            self.show_or_hide_interval(key, var_type, var_file_short))
        cb_02 = SE(
            parent=subwindow_quickplotter, row_id=row_interval_opt,
            column_id=column_start + n_columns_isotopes + n_columns_sections + (n_columns_sections - 1), n_rows=1,
            n_columns=1, fg=self.bg_colors["Dark Font"], bg=self.colors_intervals["MAT"]).create_simple_checkbox(
            var_cb=self.qpl_cb_mat, text="", set_sticky="nesw", own_color=True,
            command=lambda key="Matrix", var_type=var_filetype, var_file_short=file_short:
            self.show_or_hide_interval(key, var_type, var_file_short))
        cb_03 = SE(
            parent=subwindow_quickplotter, row_id=row_interval_opt,
            column_id=column_start + n_columns_isotopes + 2*n_columns_sections + (n_columns_sections - 1), n_rows=1,
            n_columns=1, fg=self.bg_colors["Dark Font"], bg=self.colors_intervals["INCL"]).create_simple_checkbox(
            var_cb=self.qpl_cb_incl, text="", set_sticky="nesw", own_color=True,
            command=lambda key="Inclusion", var_type=var_filetype, var_file_short=file_short:
            self.show_or_hide_interval(key, var_type, var_file_short))

        cb_01.configure(font=font_elements, onvalue=1, offvalue=0)
        cb_02.configure(font=font_elements, onvalue=1, offvalue=0)
        cb_03.configure(font=font_elements, onvalue=1, offvalue=0)

        ## TREEVIEW
        self.tv_quick_file_statistics = SE(
            parent=subwindow_quickplotter, row_id=row_start + 2,
            column_id=column_start + n_columns_isotopes + n_columns_diagram, n_rows=n_rows - 2,
            n_columns=n_columns_statistics, fg=font_color_dark, bg=self.bg_colors["White"]).create_treeview(
            n_categories=4, text_n=["Isotope", "\u03bc", "\u03c3", "I(i)/I(IS)"],
            width_n=["88", "110", "110", "110"], individual=True)

        style = ttk.Style()
        style.configure("Treeview", font=font_table)
        style.configure("Treeview.Heading", font=font_elements)

        ## LISTBOXES
        self.lb_quick_plot_file_bg = SE(
            parent=subwindow_quickplotter, row_id=row_interval_opt + 1, column_id=column_start + n_columns_isotopes,
            n_rows=n_rows - 26, n_columns=n_columns_sections, fg=self.bg_colors["Dark Font"],
            bg=self.colors_intervals["BG LB"]).create_simple_listbox()
        self.lb_quick_plot_file_mat = SE(
            parent=subwindow_quickplotter, row_id=row_interval_opt + 1,
            column_id=column_start + n_columns_isotopes + n_columns_sections, n_rows=n_rows - 26,
            n_columns=n_columns_sections, fg=self.bg_colors["Dark Font"],
            bg=self.colors_intervals["MAT LB"]).create_simple_listbox()
        self.lb_quick_plot_file_incl = SE(
            parent=subwindow_quickplotter, row_id=row_interval_opt + 1,
            column_id=column_start + n_columns_isotopes + 2*n_columns_sections, n_rows=n_rows - 26,
            n_columns=n_columns_sections, fg=self.bg_colors["Dark Font"],
            bg=self.colors_intervals["INCL LB"]).create_simple_listbox()

        ## INITIALIZATION
        self.helper_intervals = {"BG": [], "MAT": [], "INCL": []}

        if var_filetype not in self.container_helper:
            self.container_helper[var_filetype] = {}
            if file_short not in self.container_helper[var_filetype]:
                self.container_helper[var_filetype][file_short] = {}
                if "AXES" not in self.container_helper[var_filetype][file_short]:
                    self.container_helper[var_filetype][file_short]["AXES"] = {
                        "Quick Plot": var_ax, "Canvas QPL": self.var_canvas_qpl}
        else:
            if file_short not in self.container_helper[var_filetype]:
                self.build_container_helper(mode=var_filetype)

            for key in ["BG", "MAT", "INCL"]:
                if key not in self.container_helper[var_filetype][file_short]:
                    self.container_helper[var_filetype][file_short][key] = {"ID": 0, "Content": {}, "Indices": []}

            if "Quick Plot" not in self.container_helper[var_filetype][file_short]["AXES"]:
                self.container_helper[var_filetype][file_short]["AXES"] = {
                    "Quick Plot": var_ax, "Canvas QPL": self.var_canvas_qpl}

        if self.pysills_mode == "MA":
            lbl_08.configure(text=str_var_18)
            rb_02.configure(text=str_var_18)
            rb_03.configure(state="disabled")
            cb_03.configure(state="disabled")
            self.qpl_cb_incl.set(0)
            self.lb_quick_plot_file_incl.configure(state="disabled")

        self.fill_tv_quick_plot_file(
            var_type=var_filetype, var_file_short=file_short, list_isotopes=file_isotopes, init=True)
        self.check_for_intervals_qpl(filetype=var_filetype, filename_short=file_short)

    def qpl_additional_diagrams(self, var_type, var_file_short):
        list_isotopes = self.container_lists["Measured Isotopes"][var_file_short]
        df_data = self.container_measurements["Dataframe"][var_file_short]
        list_keys = list_isotopes.copy()
        list_keys.insert(0, "None")
        list_keys.insert(1, "Time")

        font_color_dark = self.bg_colors["Dark Font"]
        font_color_light = self.bg_colors["Light Font"]
        background_color_dark = self.bg_colors["BG Window"]
        background_color_elements = self.bg_colors["Light"]
        background_color_light = self.bg_colors["Very Light"]
        accent_color = self.bg_colors["Accent"]  # self.accent_color
        font_header = self.font_settings["Header"]
        font_elements = self.font_settings["Elements"]
        font_option = self.font_settings["Options"]
        font_table = self.font_settings["Table"]

        ## Window Settings
        row_min = self.row_height
        column_min = self.column_width
        n_rows = self.window_dimensions["Quick Plot Extra"][0]
        n_columns = self.window_dimensions["Quick Plot Extra"][1]

        window_width = int(n_columns*self.column_width)
        window_height = int(n_rows*self.row_height)

        var_geometry = str(window_width) + "x" + str(window_height) + "+" + str(0) + "+" + str(0)

        subwindow_qpl_extra = tk.Toplevel(self.parent)
        subwindow_qpl_extra.title("Quick Plotter")
        subwindow_qpl_extra.geometry(var_geometry)
        subwindow_qpl_extra.resizable(False, False)
        subwindow_qpl_extra["bg"] = background_color_light

        for x in range(n_columns):
            tk.Grid.columnconfigure(subwindow_qpl_extra, x, weight=1)
        for y in range(n_rows):
            tk.Grid.rowconfigure(subwindow_qpl_extra, y, weight=1)

        # Rows
        for i in range(0, n_rows):
            subwindow_qpl_extra.grid_rowconfigure(i, minsize=row_min)
        # Columns
        for i in range(0, n_columns):
            subwindow_qpl_extra.grid_columnconfigure(i, minsize=column_min)

        ###########################################################

        row_start = 0
        column_start = 0
        n_rows_navigation = n_rows
        n_colums_navigation = int(1/3*n_columns)
        n_columns_plot = n_columns - n_colums_navigation
        rows_tv = n_rows - 12
        n_rows_tv = 12

        ## TRANSLATIONS
        str_var_01 = self.language_dict["Plotting setup"][self.var_language]
        str_var_02 = self.language_dict["Isotope selection"][self.var_language]
        str_var_03 = self.language_dict["Section selection"][self.var_language]
        str_var_04 = self.language_dict["Color coding"][self.var_language]
        str_var_05 = self.language_dict["Diagram selection"][self.var_language]
        str_var_06 = self.language_dict["Data analysis"][self.var_language]
        str_var_07 = self.language_dict["Background"][self.var_language]
        str_var_08 = self.language_dict["Matrix"][self.var_language]
        str_var_09 = self.language_dict["Inclusion"][self.var_language]
        str_var_10 = self.language_dict["Scatter plot"][self.var_language]
        str_var_11 = self.language_dict["Histogram plot"][self.var_language]
        str_var_12 = self.language_dict["Sample"][self.var_language]

        ## FRAMES
        frm_01 = SE(
            parent=subwindow_qpl_extra, row_id=row_start, column_id=column_start, n_rows=n_rows_navigation,
            n_columns=n_colums_navigation, fg=font_color_dark, bg=background_color_dark).create_frame(relief=tk.FLAT)

        ## LABELS
        lbl_01 = SE(
            parent=subwindow_qpl_extra, row_id=row_start, column_id=column_start, n_rows=1,
            n_columns=n_colums_navigation, fg=font_color_light, bg=background_color_dark).create_simple_label(
            text=str_var_01, relief=tk.FLAT, fontsize=font_header, anchor=tk.W)
        lbl_02 = SE(
            parent=subwindow_qpl_extra, row_id=row_start + 1, column_id=column_start, n_rows=1,
            n_columns=n_colums_navigation, fg=font_color_light, bg=background_color_dark).create_simple_label(
            text=str_var_02, relief=tk.FLAT, fontsize=font_elements, anchor=tk.W)
        lbl_02a = SE(
            parent=subwindow_qpl_extra, row_id=row_start + 2, column_id=column_start, n_rows=1,
            n_columns=2, fg=font_color_light, bg=background_color_dark).create_simple_label(
            text="x", relief=tk.FLAT, fontsize=font_elements, anchor=tk.W)
        lbl_02b = SE(
            parent=subwindow_qpl_extra, row_id=row_start + 3, column_id=column_start, n_rows=1,
            n_columns=2, fg=font_color_light, bg=background_color_dark).create_simple_label(
            text="y", relief=tk.FLAT, fontsize=font_elements, anchor=tk.W)
        lbl_03 = SE(
            parent=subwindow_qpl_extra, row_id=row_start + 4, column_id=column_start, n_rows=1,
            n_columns=n_colums_navigation, fg=font_color_light, bg=background_color_dark).create_simple_label(
            text=str_var_03, relief=tk.FLAT, fontsize=font_elements, anchor=tk.W)
        lbl_04 = SE(
            parent=subwindow_qpl_extra, row_id=row_start + 8, column_id=column_start, n_rows=1,
            n_columns=n_colums_navigation, fg=font_color_light, bg=background_color_dark).create_simple_label(
            text=str_var_04, relief=tk.FLAT, fontsize=font_elements, anchor=tk.W)
        lbl_04a = SE(
            parent=subwindow_qpl_extra, row_id=row_start + 9, column_id=column_start, n_rows=1,
            n_columns=2, fg=font_color_light, bg=background_color_dark).create_simple_label(
            text="z", relief=tk.FLAT, fontsize=font_elements, anchor=tk.W)
        lbl_05 = SE(
            parent=subwindow_qpl_extra, row_id=row_start + 10, column_id=column_start, n_rows=1,
            n_columns=n_colums_navigation, fg=font_color_light, bg=background_color_dark).create_simple_label(
            text=str_var_05, relief=tk.FLAT, fontsize=font_elements, anchor=tk.W)
        lbl_06 = SE(
            parent=subwindow_qpl_extra, row_id=row_start, column_id=column_start + n_colums_navigation, n_rows=1,
            n_columns=n_columns_plot, fg=font_color_light, bg=background_color_dark).create_simple_label(
            text="", relief=tk.FLAT, fontsize=font_header, anchor=tk.W)
        lbl_07 = SE(
            parent=subwindow_qpl_extra, row_id=rows_tv - 1, column_id=column_start + n_colums_navigation, n_rows=1,
            n_columns=n_columns_plot, fg=font_color_light, bg=background_color_dark).create_simple_label(
            text=str_var_06, relief=tk.FLAT, fontsize=font_header, anchor=tk.W)

        self.lbl_qpl_extra_06 = lbl_06

        ## RADIOBUTTONS
        self.var_rb_qpl_extra_01 = tk.IntVar()
        self.var_rb_qpl_extra_01.set(1)
        self.var_rb_qpl_extra_02 = tk.IntVar()
        self.var_rb_qpl_extra_02.set(0)

        var_filetype = var_type
        var_filename_short = var_file_short
        var_list_isotopes = list_isotopes
        var_row_start = row_start
        var_column_start = column_start
        var_df_data = df_data

        rb_01 = SE(
            parent=subwindow_qpl_extra, row_id=row_start + 5, column_id=column_start, n_rows=1,
            n_columns=n_colums_navigation, fg=font_color_dark,
            bg=background_color_elements).create_radiobutton(
            var_rb=self.var_rb_qpl_extra_01, value_rb=0, color_bg=background_color_elements,
            fg=font_color_dark, text=str_var_07, sticky="nesw", relief=tk.FLAT,
            command=lambda var_type=var_filetype, var_file_short=var_filename_short, list_isotopes=var_list_isotopes,
                           section="BG":
            self.fill_tv_qpl_extra(var_type, var_file_short, list_isotopes, section))
        rb_02 = SE(
            parent=subwindow_qpl_extra, row_id=row_start + 6, column_id=column_start, n_rows=1,
            n_columns=n_colums_navigation, fg=font_color_dark,
            bg=background_color_elements).create_radiobutton(
            var_rb=self.var_rb_qpl_extra_01, value_rb=1, color_bg=background_color_elements,
            fg=font_color_dark, text=str_var_08, sticky="nesw", relief=tk.FLAT,
            command=lambda var_type=var_filetype, var_file_short=var_filename_short, list_isotopes=var_list_isotopes,
                           section="MAT":
            self.fill_tv_qpl_extra(var_type, var_file_short, list_isotopes, section))
        rb_03 = SE(
            parent=subwindow_qpl_extra, row_id=row_start + 7, column_id=column_start, n_rows=1,
            n_columns=n_colums_navigation, fg=font_color_dark,
            bg=background_color_elements).create_radiobutton(
            var_rb=self.var_rb_qpl_extra_01, value_rb=2, color_bg=background_color_elements,
            fg=font_color_dark, text=str_var_09, sticky="nesw", relief=tk.FLAT,
            command=lambda var_type=var_filetype, var_file_short=var_filename_short, list_isotopes=var_list_isotopes,
                           section="INCL":
            self.fill_tv_qpl_extra(var_type, var_file_short, list_isotopes, section))
        rb_04 = SE(
            parent=subwindow_qpl_extra, row_id=row_start + 11, column_id=column_start, n_rows=1,
            n_columns=n_colums_navigation, fg=font_color_dark,
            bg=background_color_elements).create_radiobutton(
            var_rb=self.var_rb_qpl_extra_02, value_rb=0, color_bg=background_color_elements,
            fg=font_color_dark, text=str_var_10, sticky="nesw", relief=tk.FLAT,
            command=lambda filetype=var_filetype, filename_short=var_filename_short, df_data=var_df_data,
                           subwindow=subwindow_qpl_extra, row_start=var_row_start + 1,
                           column_start=var_column_start + n_colums_navigation, n_rows_diagram=n_rows - 14,
                           n_columns_diagram=n_columns_plot, init=False:
            self.create_qpl_extra_diagram(filetype, filename_short, df_data, subwindow, row_start, column_start,
                                          n_rows_diagram, n_columns_diagram, init))
        rb_05 = SE(
            parent=subwindow_qpl_extra, row_id=row_start + 12, column_id=column_start, n_rows=1,
            n_columns=n_colums_navigation, fg=font_color_dark,
            bg=background_color_elements).create_radiobutton(
            var_rb=self.var_rb_qpl_extra_02, value_rb=1, color_bg=background_color_elements,
            fg=font_color_dark, text=str_var_11, sticky="nesw", relief=tk.FLAT,
            command=lambda filetype=var_filetype, filename_short=var_filename_short, df_data=var_df_data,
                           subwindow=subwindow_qpl_extra, row_start=var_row_start + 1,
                           column_start=var_column_start + n_colums_navigation, n_rows_diagram=n_rows - 14,
                           n_columns_diagram=n_columns_plot, init=False:
            self.create_qpl_extra_diagram(filetype, filename_short, df_data, subwindow, row_start, column_start,
                                          n_rows_diagram, n_columns_diagram, init))

        #rb_05.configure(state="disabled")

        if self.pysills_mode == "MA":
            rb_02.configure(text=str_var_12)
            rb_03.configure(state="disabled")

        ## OPTION MENUS
        self.var_opt_qpl_extra_01 = tk.StringVar()
        self.var_opt_qpl_extra_01.set("Select isotope")
        self.var_opt_qpl_extra_02 = tk.StringVar()
        self.var_opt_qpl_extra_02.set("Select isotope")
        self.var_opt_qpl_extra_03 = tk.StringVar()
        self.var_opt_qpl_extra_03.set(list_keys[0])

        opt_01 = SE(
            parent=subwindow_qpl_extra, row_id=row_start + 2, column_id=column_start + 2, n_rows=1,
            n_columns=n_colums_navigation - 2, fg=font_color_dark, bg=background_color_elements).create_option_isotope(
            var_iso=self.var_opt_qpl_extra_01, option_list=list_isotopes, text_set=self.var_opt_qpl_extra_01.get(),
            fg_active=font_color_light, bg_active=accent_color,
            command=lambda filetype=var_filetype, filename_short=var_filename_short, df_data=var_df_data,
                           subwindow=subwindow_qpl_extra, row_start=var_row_start + 1,
                           column_start=var_column_start + n_colums_navigation, n_rows_diagram=n_rows - 14,
                           n_columns_diagram=n_columns_plot, init=False:
            self.create_qpl_extra_diagram(filetype, filename_short, df_data, subwindow, row_start, column_start,
                                          n_rows_diagram, n_columns_diagram, init))
        opt_02 = SE(
            parent=subwindow_qpl_extra, row_id=row_start + 3, column_id=column_start + 2, n_rows=1,
            n_columns=n_colums_navigation - 2, fg=font_color_dark, bg=background_color_elements).create_option_isotope(
            var_iso=self.var_opt_qpl_extra_02, option_list=list_isotopes, text_set=self.var_opt_qpl_extra_02.get(),
            fg_active=font_color_light, bg_active=accent_color,
            command=lambda filetype=var_type, filename_short=var_file_short, df_data=var_df_data,
            subwindow=subwindow_qpl_extra, row_start=var_row_start + 1,
                           column_start=var_column_start + n_colums_navigation, n_rows_diagram=n_rows - 14,
                           n_columns_diagram=n_columns_plot:
            self.create_qpl_scatter_plot(filetype, filename_short, df_data, subwindow, row_start, column_start,
                                         n_rows_diagram, n_columns_diagram))
        opt_03 = SE(
            parent=subwindow_qpl_extra, row_id=row_start + 9, column_id=column_start + 2, n_rows=1,
            n_columns=n_colums_navigation - 2, fg=font_color_dark, bg=background_color_elements).create_option_isotope(
            var_iso=self.var_opt_qpl_extra_03, option_list=list_keys, text_set=self.var_opt_qpl_extra_03.get(),
            fg_active=font_color_light, bg_active=accent_color,
            command=lambda filetype=var_type, filename_short=var_file_short, df_data=var_df_data,
            subwindow=subwindow_qpl_extra, row_start=var_row_start + 1,
                           column_start=var_column_start + n_colums_navigation, n_rows_diagram=n_rows - 14,
                           n_columns_diagram=n_columns_plot:
            self.create_qpl_scatter_plot(filetype, filename_short, df_data, subwindow, row_start, column_start,
                                         n_rows_diagram, n_columns_diagram))

        ## TREEVIEWS
        list_widths = 6*[112]
        list_widths[0] = 80

        self.tv_qpl_extra = SE(
            parent=subwindow_qpl_extra, row_id=rows_tv, column_id=column_start + n_colums_navigation, n_rows=n_rows_tv,
            n_columns=n_columns_plot, fg=font_color_dark, bg=self.bg_colors["White"]).create_treeview(
            n_categories=6, text_n=["Isotope", "Minimum", "Maximum", "\u03bc", "\u03c3", "I(i)/I(IS)"],
            width_n=list_widths, individual=True)

        style = ttk.Style()
        style.configure("Treeview", font=font_table)
        style.configure("Treeview.Heading", font=font_elements)

        ## INITIALIZATION
        self.fill_tv_qpl_extra(
            var_type=var_type, var_file_short=var_file_short, list_isotopes=list_isotopes, init=True)
        self.create_qpl_scatter_plot(
            filetype=var_type, filename_short=var_file_short, df_data=df_data, subwindow=subwindow_qpl_extra,
            row_start=row_start + 1, column_start=column_start + n_colums_navigation, n_rows_diagram=n_rows - 14,
            n_columns_diagram=n_columns_plot, init=True)

        n_intervals = 0
        for section in ["BG", "MAT", "INCL"]:
            n_intervals += len(self.container_helper[var_type][var_file_short][section]["Indices"])

        if n_intervals > 0:
            if len(self.container_helper[var_type][var_file_short]["BG"]["Indices"]) == 0:
                rb_01.configure(state="disabled")
            elif len(self.container_helper[var_type][var_file_short]["MAT"]["Indices"]) == 0:
                rb_02.configure(state="disabled")
            elif len(self.container_helper[var_type][var_file_short]["INCL"]["Indices"]) == 0:
                rb_03.configure(state="disabled")
        else:
            rb_01.configure(state="disabled")
            rb_02.configure(state="disabled")
            rb_03.configure(state="disabled")
            rb_04.configure(state="disabled")
            rb_05.configure(state="disabled")

            opt_01.configure(state="disabled")
            opt_02.configure(state="disabled")
            opt_03.configure(state="disabled")

    def create_qpl_extra_diagram(self, filetype, filename_short, df_data, subwindow, row_start, column_start,
                                  n_rows_diagram, n_columns_diagram, init=False):
        if self.var_rb_qpl_extra_02.get() == 0: # Scatter
            self.create_qpl_scatter_plot(
                filetype, filename_short, df_data, subwindow, row_start, column_start, n_rows_diagram,
                n_columns_diagram, init)
        elif self.var_rb_qpl_extra_02.get() == 1: # Histogram
            self.create_qpl_histogram_plot(
                filetype, filename_short, df_data, subwindow, row_start, column_start, n_rows_diagram,
                n_columns_diagram, init)

    def create_qpl_histogram_plot(self, filetype, filename_short, df_data, subwindow, row_start, column_start,
                                  n_rows_diagram, n_columns_diagram, init=False):
        # Data preparation
        list_isotopes = list(df_data.keys())[1:]
        if filename_short in self.container_lists["STD"]["Short"]:
            filetype = "STD"
        elif filename_short in self.container_lists["SMPL"]["Short"]:
            filetype = "SMPL"

        if init == False:
            var_x = self.var_opt_qpl_extra_01.get()
            var_y = self.var_opt_qpl_extra_02.get()
            var_z = self.var_opt_qpl_extra_03.get()
        else:
            var_x = list_isotopes[0]
            var_y = list_isotopes[1]
            var_z = list_isotopes[2]

        if self.var_rb_qpl_extra_01.get() == 0:
            section = "BG"
        elif self.var_rb_qpl_extra_01.get() == 1:
            section = "MAT"
        elif self.var_rb_qpl_extra_01.get() == 2:
            section = "INCL"

        var_section = section

        if (len(self.container_helper[filetype][filename_short][section]["Indices"]) > 0 and
                "Select isotope" not in [var_x, var_y]):
            condensed_intervals = IQ(dataframe=None).combine_all_intervals(
                interval_set=self.container_helper[filetype][filename_short][section]["Content"])
            helper_data = {}

            helper_data[var_x] = []
            helper_data[var_y] = []
            for index, interval in condensed_intervals.items():
                start_index = interval[0]
                end_index = interval[1] + 1
                dataset_x = df_data[var_x][start_index:end_index]
                dataset_y = df_data[var_y][start_index:end_index]
                if var_z != "None" and var_z not in [var_x, var_y]:
                    if var_z not in helper_data:
                        helper_data[var_z] = []

                    dataset_z = df_data[var_z][start_index:end_index]
                    helper_data[var_z].extend(list(dataset_z))

                helper_data[var_x].extend(list(dataset_x))
                helper_data[var_y].extend(list(dataset_y))

            data_x = helper_data[var_x]
            data_y = helper_data[var_y]

            min_x = min(data_x)
            max_x = max(data_x)
            min_y = min(data_y)
            max_y = max(data_y)
            lim_x_min, lim_x_max = self.round_to_the_next_natural_limit(minimum=min_x, maximum=max_x)
            lim_y_min, lim_y_max = self.round_to_the_next_natural_limit(minimum=min_y, maximum=max_y)

            # Diagram
            var_fig = Figure(figsize=(10, 5), tight_layout=True, facecolor=self.bg_colors["Light"])
            var_ax = var_fig.add_subplot(label=np.random.uniform())

            n_bins = 30
            var_ax.hist(data_x, bins=n_bins, color=self.accent_color, edgecolor="black")

            var_ax.grid(which="major", linestyle="-", linewidth=1)
            var_ax.minorticks_on()
            var_ax.grid(which="minor", linestyle=":", linewidth=0.5, alpha=0.75)
            var_ax.set_axisbelow(True)
            var_ax.set_title(filename_short, fontsize=9)
            var_ax.set_xlabel("Signal " + var_x + " (cps)", labelpad=0.5, fontsize="x-small")
            var_ax.set_ylabel("Frequency", labelpad=0.5, fontsize="x-small")
            var_ax.xaxis.set_tick_params(labelsize="x-small")
            var_ax.yaxis.set_tick_params(labelsize="x-small")

            self.var_canvas_qpl_extra = FigureCanvasTkAgg(var_fig, master=subwindow)
            self.var_canvas_qpl_extra.get_tk_widget().grid(
                row=row_start, column=column_start, rowspan=n_rows_diagram, columnspan=n_columns_diagram, sticky="nesw")

            # TREEVIEW UPDATE
        self.fill_tv_qpl_extra(
            var_type=filetype, var_file_short=filename_short, list_isotopes=list_isotopes, section=var_section,
            isotope_is=var_x)

    def create_qpl_scatter_plot(
            self, filetype, filename_short, df_data, subwindow, row_start, column_start, n_rows_diagram,
            n_columns_diagram, init=False):
        # Data preparation
        list_isotopes = list(df_data.keys())[1:]
        if filename_short in self.container_lists["STD"]["Short"]:
            filetype = "STD"
        elif filename_short in self.container_lists["SMPL"]["Short"]:
            filetype = "SMPL"

        if init == False:
            var_x = self.var_opt_qpl_extra_01.get()
            var_y = self.var_opt_qpl_extra_02.get()
            var_z = self.var_opt_qpl_extra_03.get()
        else:
            var_x = list_isotopes[0]
            var_y = list_isotopes[1]
            var_z = list_isotopes[2]

        if self.var_rb_qpl_extra_01.get() == 0:
            section = "BG"
        elif self.var_rb_qpl_extra_01.get() == 1:
            section = "MAT"
        elif self.var_rb_qpl_extra_01.get() == 2:
            section = "INCL"

        var_section = section

        if (len(self.container_helper[filetype][filename_short][section]["Indices"]) > 0 and
                "Select isotope" not in [var_x, var_y]):
            condensed_intervals = IQ(dataframe=None).combine_all_intervals(
                interval_set=self.container_helper[filetype][filename_short][section]["Content"])
            helper_data = {}

            helper_data[var_x] = []
            helper_data[var_y] = []
            for index, interval in condensed_intervals.items():
                start_index = interval[0]
                end_index = interval[1] + 1
                dataset_x = df_data[var_x][start_index:end_index]
                dataset_y = df_data[var_y][start_index:end_index]
                if var_z != "None" and var_z not in [var_x, var_y]:
                    if "Time" not in var_z:
                        pass
                    else:
                        if "Time" not in list(df_data.keys()):
                            if list(df_data.keys())[0] not in list_isotopes:
                                var_z = list(df_data.keys())[0]

                    if var_z not in helper_data:
                        helper_data[var_z] = []

                    dataset_z = df_data[var_z][start_index:end_index]
                    helper_data[var_z].extend(list(dataset_z))

                helper_data[var_x].extend(list(dataset_x))
                helper_data[var_y].extend(list(dataset_y))

            data_x = helper_data[var_x]
            data_y = helper_data[var_y]

            min_x = min(data_x)
            max_x = max(data_x)
            min_y = min(data_y)
            max_y = max(data_y)
            lim_x_min, lim_x_max = self.round_to_the_next_natural_limit(minimum=min_x, maximum=max_x)
            lim_y_min, lim_y_max = self.round_to_the_next_natural_limit(minimum=min_y, maximum=max_y)

            # Diagram
            var_fig = Figure(figsize=(10, 5), tight_layout=True, facecolor=self.bg_colors["Light"])
            var_ax = var_fig.add_subplot(label=np.random.uniform())

            if var_z == "None":
                sct = var_ax.scatter(data_x, data_y, edgecolor="black")
            else:
                data_z = helper_data[var_z]

                min_z = min(data_z)
                max_z = max(data_z)
                sct = var_ax.scatter(data_x, data_y, c=data_z, edgecolor="black")
                cbar = var_fig.colorbar(sct, ax=var_ax)
                cbar.set_ticks(np.linspace(min_z, max_z, 6, endpoint=True))
                cbar.ax.tick_params(labelsize="x-small")
                if var_z != "Time" and "Time" not in var_z:
                    cbar.set_label("Signal " + var_z + " (cps)", fontsize=8)
                else:
                    cbar.set_label("Time (s)", fontsize="x-small")

            var_ax.grid(True)
            var_ax.set_xlim(left=lim_x_min, right=lim_x_max)
            var_ax.set_xticks(np.linspace(lim_x_min, lim_x_max, 6, endpoint=True))
            var_ax.set_ylim(bottom=lim_y_min, top=lim_y_max)
            var_ax.set_yticks(np.linspace(lim_y_min, lim_y_max, 6, endpoint=True))
            var_ax.grid(which="major", linestyle="-", linewidth=1)
            var_ax.minorticks_on()
            var_ax.grid(which="minor", linestyle=":", linewidth=0.5, alpha=0.75)
            var_ax.set_axisbelow(True)
            var_ax.set_title(filename_short, fontsize=9)
            var_ax.set_xlabel("Signal " + var_x + " (cps)", labelpad=0.5, fontsize="x-small")
            var_ax.set_ylabel("Signal " + var_y + " (cps)", labelpad=0.5, fontsize="x-small")
            var_ax.xaxis.set_tick_params(labelsize="x-small")
            var_ax.yaxis.set_tick_params(labelsize="x-small")

            self.var_canvas_qpl_extra = FigureCanvasTkAgg(var_fig, master=subwindow)
            self.var_canvas_qpl_extra.get_tk_widget().grid(
                row=row_start, column=column_start, rowspan=n_rows_diagram, columnspan=n_columns_diagram, sticky="nesw")

        # TREEVIEW UPDATE
        self.fill_tv_qpl_extra(
            var_type=filetype, var_file_short=filename_short, list_isotopes=list_isotopes, section=var_section,
            isotope_is=var_x)

    def round_to_the_next_natural_limit(self, minimum, maximum):
        data_range = maximum - minimum
        scale_order = np.floor(np.log10(data_range))
        rounding_factor = 10**scale_order

        lim_min = np.floor(minimum/rounding_factor)*rounding_factor

        if lim_min == minimum:
            lim_min = lim_min
        else:
            lim_min = rounding_factor*np.floor(minimum/rounding_factor)

        lim_max = np.ceil(maximum/rounding_factor)*rounding_factor
        if lim_max == maximum:
            lim_max = lim_max
        else:
            lim_max = rounding_factor*np.ceil(maximum/rounding_factor)

        return lim_min, lim_max

    def fill_tv_qpl_extra(self, var_type, var_file_short, list_isotopes, section="MAT", init=False, isotope_is=None):
        if init == True:
            for isotope in list_isotopes:
                helper_values = [isotope, "---", "---", "---", "---", "---"]
                self.tv_qpl_extra.insert("", tk.END, values=helper_values)
        else:
            if len(self.tv_qpl_extra.get_children()) > 0:
                for item in self.tv_qpl_extra.get_children():
                    self.tv_qpl_extra.delete(item)

            if len(self.container_helper[var_type][var_file_short][section]["Indices"]) > 0:
                df_data  = self.container_measurements["Dataframe"][var_file_short]
                condensed_intervals = IQ(dataframe=None).combine_all_intervals(
                    interval_set=self.container_helper[var_type][var_file_short][section]["Content"])

                if isotope_is == None:
                    reference_is = self.var_opt_qpl_extra_01.get()
                else:
                    reference_is = isotope_is
                if reference_is != "Select isotope":
                    isotope_is = reference_is
                    for isotope in list_isotopes:
                        helper = {"Min": [], "Max": [], "Mean": [], "Standard deviation": [], "Ratio": []}
                        for index, interval in condensed_intervals.items():
                            start_index = interval[0]
                            end_index = interval[1] + 1
                            dataset_i = df_data[isotope][start_index:end_index]
                            dataset_is = df_data[isotope_is][start_index:end_index]
                            helper["Min"].append(min(dataset_i))
                            helper["Max"].append(max(dataset_i))
                            helper["Mean"].append(np.mean(dataset_i))
                            helper["Standard deviation"].append(np.std(dataset_i, ddof=1))
                            helper["Ratio"].append(np.mean(dataset_i)/np.mean(dataset_is))
                        helper_values = [
                            isotope, round(np.mean(helper["Min"]), 2), round(np.mean(helper["Max"]), 2),
                            round(np.mean(helper["Mean"]), 2), round(np.mean(helper["Standard deviation"]), 2),
                            "{:0.2e}".format(np.mean(helper["Ratio"]))]
                        self.tv_qpl_extra.insert("", tk.END, values=helper_values)
                else:
                    for isotope in list_isotopes:
                        helper_values = [isotope, "---", "---", "---", "---", "---"]
                        self.tv_qpl_extra.insert("", tk.END, values=helper_values)

            else:
                for isotope in list_isotopes:
                    helper_values = [isotope, "---", "---", "---", "---", "---"]
                    self.tv_qpl_extra.insert("", tk.END, values=helper_values)

    def add_interval_to_diagram_qpl(self, var_type, var_file_short, event):
        filename_index = self.container_lists[var_type]["Short"].index(var_file_short)
        var_file_long = self.container_lists[var_type]["Long"][filename_index]

        if self.container_var["Radiobuttons"]["Quick Plot File"].get() == 0:    # BG
            var_key = "BG"
            var_color = self.colors_intervals[var_key]
        elif self.container_var["Radiobuttons"]["Quick Plot File"].get() == 1:  # MAT
            var_key = "MAT"
            var_color = self.colors_intervals[var_key]
        elif self.container_var["Radiobuttons"]["Quick Plot File"].get() == 2:  # INCL
            var_key = "INCL"
            var_color = self.colors_intervals[var_key]

        if self.container_var["Radiobuttons"]["Quick Plot File"].get() in [0, 1, 2]:
            x_nearest = min(self.dataset_time, key=lambda x: abs(x - event.xdata))

            if len(self.helper_intervals[var_key]) < 1:
                x_id = self.dataset_time.index(x_nearest)
                self.helper_intervals[var_key].append([x_nearest, x_id])
            else:
                x_id = self.dataset_time.index(x_nearest)
                self.helper_intervals[var_key].append([x_nearest, x_id])
                key_id = self.container_helper[var_type][var_file_short][var_key]["ID"] + 1
                time_0_pre = self.helper_intervals[var_key][0][0]
                time_1_pre = self.helper_intervals[var_key][1][0]
                index_0_pre = self.helper_intervals[var_key][0][1]
                index_1_pre = self.helper_intervals[var_key][1][1]

                if time_1_pre < time_0_pre:
                    time_0 = time_1_pre
                    time_1 = time_0_pre
                    index_0 = index_1_pre
                    index_1 = index_0_pre
                else:
                    time_0 = time_0_pre
                    time_1 = time_1_pre
                    index_0 = index_0_pre
                    index_1 = index_1_pre

                box_key = self.var_ax.axvspan(time_0, time_1, alpha=0.35, color=var_color)
                self.var_canvas_qpl.draw()

                self.container_helper[var_type][var_file_short][var_key]["Content"][key_id] = {
                    "Times": [time_0, time_1], "Indices": [index_0, index_1], "Object": box_key}

                if var_key == "BG":
                    var_lb = self.lb_quick_plot_file_bg
                elif var_key == "MAT":
                    var_lb = self.lb_quick_plot_file_mat
                elif var_key == "INCL":
                    var_lb = self.lb_quick_plot_file_incl

                var_lb.insert(tk.END, var_key + str(key_id) + " [" + str(time_0) + "-" + str(time_1) + "]")

                self.helper_intervals[var_key].clear()
                self.container_helper[var_type][var_file_short][var_key]["ID"] = key_id
                self.container_helper[var_type][var_file_short][var_key]["Indices"].append(key_id)

                self.var_canvas_qpl.draw()
        if var_type == "STD":
            if len(self.container_helper[var_type][var_file_short]["BG"]["Content"]) + \
                    len(self.container_helper[var_type][var_file_short]["MAT"]["Content"]) < 2:
                if var_file_long in self.container_var[var_type]:
                    self.container_var[var_type][var_file_long]["Frame"].config(background=self.sign_yellow, bd=1)
                    self.container_var[var_type][var_file_long]["Sign Color"].set(self.sign_yellow)
            else:
                if len(self.container_helper[var_type][var_file_short]["BG"]["Content"]) > 0 and \
                        len(self.container_helper[var_type][var_file_short]["MAT"]["Content"]) > 0:
                    if var_file_long in self.container_var[var_type]:
                        self.container_var[var_type][var_file_long]["Frame"].config(background=self.sign_green, bd=1)
                        self.container_var[var_type][var_file_long]["Sign Color"].set(self.sign_green)
                else:
                    if var_file_long in self.container_var[var_type]:
                        self.container_var[var_type][var_file_long]["Frame"].config(background=self.sign_yellow, bd=1)
                        self.container_var[var_type][var_file_long]["Sign Color"].set(self.sign_yellow)
        else:
            if len(self.container_helper[var_type][var_file_short]["BG"]["Content"]) + \
                    len(self.container_helper[var_type][var_file_short]["MAT"]["Content"]) + \
                    len(self.container_helper[var_type][var_file_short]["INCL"]["Content"]) < 3:
                if var_file_long in self.container_var[var_type]:
                    self.container_var[var_type][var_file_long]["Frame"].config(background=self.sign_yellow, bd=1)
                    self.container_var[var_type][var_file_long]["Sign Color"].set(self.sign_yellow)
            else:
                if len(self.container_helper[var_type][var_file_short]["BG"]["Content"]) > 0 and \
                        len(self.container_helper[var_type][var_file_short]["MAT"]["Content"]) > 0 and \
                        len(self.container_helper[var_type][var_file_short]["INCL"]["Content"]) > 0:
                    if var_file_long in self.container_var[var_type]:
                        self.container_var[var_type][var_file_long]["Frame"].config(background=self.sign_green, bd=1)
                        self.container_var[var_type][var_file_long]["Sign Color"].set(self.sign_green)
                else:
                    if var_file_long in self.container_var[var_type]:
                        self.container_var[var_type][var_file_long]["Frame"].config(background=self.sign_yellow, bd=1)
                        self.container_var[var_type][var_file_long]["Sign Color"].set(self.sign_yellow)

    def remove_interval_qpl(self, var_type, var_file_short):
        if self.container_var["Radiobuttons"]["Quick Plot File"].get() == 0:  # BG
            var_key = "BG"
            var_lb = self.lb_quick_plot_file_bg
        elif self.container_var["Radiobuttons"]["Quick Plot File"].get() == 1:  # MAT
            var_key = "MAT"
            var_lb = self.lb_quick_plot_file_mat
        elif self.container_var["Radiobuttons"]["Quick Plot File"].get() == 2:  # INCL
            var_key = "INCL"
            var_lb = self.lb_quick_plot_file_incl

        item = var_lb.curselection()[0]
        value = var_lb.get(item)
        value_parts = value.split(" ")
        key_id = re.search(r"(\D+)(\d+)", value_parts[0])
        var_id = int(key_id.group(2))

        self.container_helper[var_type][var_file_short][var_key]["Indices"].remove(var_id)
        var_lb.delete(tk.ANCHOR)
        self.container_helper[var_type][var_file_short][var_key]["Content"][var_id]["Object"].set_visible(False)
        del self.container_helper[var_type][var_file_short][var_key]["Content"][var_id]

        self.var_canvas_qpl.draw()

    def fill_tv_quick_plot_file(self, var_type, var_file_short, list_isotopes, section="MAT", init=False):
        if init == True:
            for isotope in list_isotopes:
                helper_values = [isotope, "---", "---", "---"]
                self.tv_quick_file_statistics.insert("", tk.END, values=helper_values)
        else:
            if len(self.tv_quick_file_statistics.get_children()) > 0:
                for item in self.tv_quick_file_statistics.get_children():
                    self.tv_quick_file_statistics.delete(item)

            if len(self.container_helper[var_type][var_file_short][section]["Indices"]) > 0:
                df_data  = self.container_measurements["Dataframe"][var_file_short]
                condensed_intervals = IQ(dataframe=None).combine_all_intervals(
                    interval_set=self.container_helper[var_type][var_file_short][section]["Content"])

                if self.qpl_opt_is.get() != "Select isotope":
                    isotope_is = self.qpl_opt_is.get()
                    for isotope in list_isotopes:
                        helper = {"Mean": [], "Standard deviation": [], "Ratio": []}
                        for index, interval in condensed_intervals.items():
                            start_index = interval[0]
                            end_index = interval[1] + 1
                            dataset_i = df_data[isotope][start_index:end_index]
                            dataset_is = df_data[isotope_is][start_index:end_index]
                            helper["Mean"].append(np.mean(dataset_i))
                            helper["Standard deviation"].append(np.std(dataset_i, ddof=1))
                            helper["Ratio"].append(np.mean(dataset_i)/np.mean(dataset_is))
                        helper_values = [
                            isotope, round(np.mean(helper["Mean"]), 2), round(np.mean(helper["Standard deviation"]), 2),
                            "{:0.2e}".format(np.mean(helper["Ratio"]))]
                        self.tv_quick_file_statistics.insert("", tk.END, values=helper_values)
                else:
                    for isotope in list_isotopes:
                        helper_values = [isotope, "---", "---", "---"]
                        self.tv_quick_file_statistics.insert("", tk.END, values=helper_values)

            else:
                for isotope in list_isotopes:
                    helper_values = [isotope, "---", "---", "---"]
                    self.tv_quick_file_statistics.insert("", tk.END, values=helper_values)

    def check_for_intervals_qpl(self, filetype, filename_short):
        if self.pysills_mode == "MA":
            list_sections = ["BG", "MAT"]
        else:
            list_sections = ["BG", "MAT", "INCL"]

        for section in list_sections:
            if len(self.container_helper[filetype][filename_short][section]["Indices"]) > 0:
                if section == "BG":
                    var_lb = self.lb_quick_plot_file_bg
                elif section == "MAT":
                    var_lb = self.lb_quick_plot_file_mat
                elif section == "INCL":
                    var_lb = self.lb_quick_plot_file_incl

                var_color = self.colors_intervals[section]

                for key_id, container in self.container_helper[filetype][filename_short][section]["Content"].items():
                    list_times = container["Times"]
                    time_0 = list_times[0]
                    time_1 = list_times[1]
                    var_lb.insert(tk.END, section + str(key_id) + " [" + str(time_0) + "-" + str(time_1) + "]")

                    box = self.var_ax.axvspan(time_0, time_1, alpha=0.35, color=var_color)
                    container["Object"] = box

        self.var_canvas_qpl.draw()

    def show_or_hide_interval(self, key, var_type, var_file_short):
        if key == "Background":
            var_key = "BG"
            state_var = self.qpl_cb_bg.get()
            if state_var == 1:
                state = True
            elif state_var == 0:
                state = False
        elif key in ["Matrix", "Sample"]:
            var_key = "MAT"
            state_var = self.qpl_cb_mat.get()
            if state_var == 1:
                state = True
            elif state_var == 0:
                state = False
        elif key == "Inclusion":
            var_key = "INCL"
            state_var = self.qpl_cb_incl.get()
            if state_var == 1:
                state = True
            elif state_var == 0:
                state = False

        for id, container in self.container_helper[var_type][var_file_short][var_key]["Content"].items():
            obj_interval = container["Object"]
            obj_interval.set_visible(state)

        self.var_canvas_qpl.draw()

    def show_line(self, var_isotope, var_file_short):
        if self.temp_checkbuttons[var_isotope].get() == 1:
            self.temp_lines[var_isotope][0].set_visible(True)
        else:
            self.temp_lines[var_isotope][0].set_visible(False)
        #
        self.container_var["Plotting"][self.pysills_mode]["Quickview"][var_file_short]["Canvas"].draw()

    def show_all_lines(self, var_file_short):
        file_isotopes = self.container_lists["Measured Isotopes"][var_file_short]
        for isotope in file_isotopes:
            self.temp_checkbuttons[isotope].set(1)
            self.temp_lines[isotope][0].set_visible(True)

        self.container_var["Plotting"][self.pysills_mode]["Quickview"][var_file_short]["Canvas"].draw()

    def hide_all_lines(self, var_file_short):
        file_isotopes = self.container_lists["Measured Isotopes"][var_file_short]
        for isotope in file_isotopes:
            self.temp_checkbuttons[isotope].set(0)
            self.temp_lines[isotope][0].set_visible(False)

        self.container_var["Plotting"][self.pysills_mode]["Quickview"][var_file_short]["Canvas"].draw()

    def place_srm_values(self, var_srm, header_col, default=False):
        lbl_srm_03 = SE(parent=self.parent, row_id=0, column_id=header_col, n_rows=1, n_columns=42, fg=self.green_light,
                        bg=self.green_dark).create_simple_label(text=str(var_srm) + " - Element Concentrations (ppm)",
                                                                relief=tk.GROOVE, fontsize="sans 10 bold")
        self.container_elements["SRM"]["Label"].append(lbl_srm_03)

        if default:
            for file in self.list_std:
                parts = file.split("/")
                self.container_var["SRM"][file].set(var_srm)
                self.container_files["STD"][parts[-1]]["SRM"].set(var_srm)

        try:
            ESRM().place_srm_values(srm_name=var_srm, srm_dict=self.srm_actual)
        except:
            self.srm_actual[var_srm] = {}
            ESRM().place_srm_values(srm_name=var_srm, srm_dict=self.srm_actual)

        for element in self.list_pse:
            if element in self.srm_actual[var_srm]:
                self.container_var["SRM"][element].set(self.srm_actual[var_srm][element])
            else:
                self.container_var["SRM"][element].set(0.0)

    #
    def fill_srm_values(self, var_srm):
        if var_srm not in self.srm_actual:
            self.srm_actual[var_srm] = {}
            ESRM().place_srm_values(srm_name=var_srm, srm_dict=self.srm_actual)

    #
    def calculate_mineral_chemistry(self):
        #
        M_H = 1.008
        M_C = 12.011
        M_O = 15.999
        M_F = 18.998
        M_Na = 22.990
        M_Mg = 24.305
        M_Al = 26.982
        M_Si = 28.085
        M_P = 30.974
        M_S = 32.059
        M_Cl = 35.450
        M_K = 39.098
        M_Ca = 40.078
        M_Ti = 47.867
        M_Cr = 51.996
        M_Mn = 54.938
        M_Fe = 55.845
        M_Cu = 63.546
        M_Zn = 65.382
        M_As = 74.922
        M_Sr = 87.620
        M_Zr = 91.224
        M_Mo = 95.950
        M_Ba = 137.33
        M_La = 138.91
        M_Ce = 140.12
        M_Pr = 140.91
        M_Nd = 144.24
        M_Sm = 150.360
        M_Eu = 151.960
        M_Gd = 157.25
        M_Pb = 207.200
        M_Th = 232.04
        #
        # self.mineral_chem = {}
        for mineral in self.mineral_list:
            self.mineral_chem[mineral] = {}
            if mineral == "Quartz":  # SiO2
                M_Qz = M_Si + 2*M_O
                w_Si = (M_Si)/(M_Qz)*1000000
                self.mineral_chem[mineral]["Si"] = w_Si
            elif mineral == "Arsenopyrite":  # FeAsS
                M_Apy = M_Fe + M_As + M_S
                w_S = (M_S)/(M_Apy)*1000000
                w_Fe = (M_Fe)/(M_Apy)*1000000
                w_As = (M_As)/(M_Apy)*1000000
                self.mineral_chem[mineral]["S"] = w_S
                self.mineral_chem[mineral]["Fe"] = w_Fe
                self.mineral_chem[mineral]["As"] = w_As
            elif mineral == "Barite":  # BaSO4
                M_Brt = M_Ba + M_S + 4*M_O
                w_O = (4*M_O)/(M_Brt)*1000000
                w_S = (M_S)/(M_Brt)*1000000
                w_Ba = (M_Ba)/(M_Brt)*1000000
                self.mineral_chem[mineral]["O"] = w_O
                self.mineral_chem[mineral]["S"] = w_S
                self.mineral_chem[mineral]["Ba"] = w_Ba
            elif mineral == "Ilvaite":  # CaFe3(SiO4)2(OH)
                M_Ilvt = M_Ca + 3*M_Fe + 2*(M_Si + 4*M_O) + (M_O + M_H)
                w_H = (M_H)/(M_Ilvt)*1000000
                w_O = (9*M_O)/(M_Ilvt)*1000000
                w_Si = (2*M_Si)/(M_Ilvt)*1000000
                w_Ca = (M_Ca)/(M_Ilvt)*1000000
                w_Fe = (3*M_Fe)/(M_Ilvt)*1000000
                self.mineral_chem[mineral]["H"] = w_H
                self.mineral_chem[mineral]["O"] = w_O
                self.mineral_chem[mineral]["Si"] = w_Si
                self.mineral_chem[mineral]["Ca"] = w_Ca
                self.mineral_chem[mineral]["Fe"] = w_Fe
            elif mineral == "Bornite":  # Cu5FeS4
                M_Bn = 5*M_Cu + M_Fe + 4*M_S
                w_S = (4*M_S)/(M_Bn)*1000000
                w_Fe = (M_Fe)/(M_Bn)*1000000
                w_Cu = (5*M_Cu)/(M_Bn)*1000000
                self.mineral_chem[mineral]["S"] = w_S
                self.mineral_chem[mineral]["Fe"] = w_Fe
                self.mineral_chem[mineral]["Cu"] = w_Cu
            elif mineral == "Calcite":  # CaCO3
                M_Cal = M_Ca + M_C + 3*M_O
                w_C = (M_C)/(M_Cal)*1000000
                w_Ca = (M_Ca)/(M_Cal)*1000000
                self.mineral_chem[mineral]["C"] = w_C
                self.mineral_chem[mineral]["Ca"] = w_Ca
            elif mineral == "Chalcopyrite":  # CuFeS2
                M_Ccp = M_Cu + M_Fe + 2*M_S
                w_S = (2*M_S)/(M_Ccp)*1000000
                w_Fe = (M_Fe)/(M_Ccp)*1000000
                w_Cu = (M_Cu)/(M_Ccp)*1000000
                self.mineral_chem[mineral]["S"] = w_S
                self.mineral_chem[mineral]["Fe"] = w_Fe
                self.mineral_chem[mineral]["Cu"] = w_Cu
            elif mineral == "Chromite":  # FeCr2O4
                M = M_Fe + 2*M_Cr + 4*M_O
                w_O = (4*M_O)/(M)*1000000
                w_Cr = (2*M_Cr)/(M)*1000000
                w_Fe = (M_Fe)/(M)*1000000
                self.mineral_chem[mineral]["O"] = w_O
                self.mineral_chem[mineral]["Cr"] = w_Cr
                self.mineral_chem[mineral]["Fe"] = w_Fe
            elif mineral == "Enargite":  # Cu3AsS4
                M_En = 3*M_Cu + M_As + 4*M_S
                w_S = (4*M_S)/(M_En)*1000000
                w_Cu = (3*M_Cu)/(M_En)*1000000
                w_As = (M_As)/(M_En)*1000000
                self.mineral_chem[mineral]["S"] = w_S
                self.mineral_chem[mineral]["Cu"] = w_Cu
                self.mineral_chem[mineral]["As"] = w_As
            elif mineral == "Fluorite":  # CaF2
                M_Fl = M_Ca + 2*M_F
                w_F = (2*M_F)/(M_Fl)*1000000
                w_Ca = (M_Ca)/(M_Fl)*1000000
                self.mineral_chem[mineral]["F"] = w_F
                self.mineral_chem[mineral]["Ca"] = w_Ca
            elif mineral == "Apatite-Cl":  # Ca5(PO4)3Cl
                M_Ap = 5*M_Ca + 3*(M_P + 4*M_O) + M_Cl
                w_P = (3*M_P)/(M_Ap)*1000000
                w_Cl = (M_Cl)/(M_Ap)*1000000
                w_Ca = (5*M_Ca)/(M_Ap)*1000000
                self.mineral_chem[mineral]["P"] = w_P
                self.mineral_chem[mineral]["Cl"] = w_Cl
                self.mineral_chem[mineral]["Ca"] = w_Ca
            elif mineral == "Apatite-F":  # Ca5(PO4)3F
                M_Ap = 5*M_Ca + 3*(M_P + 4*M_O) + M_F
                w_F = (M_F)/(M_Ap)*1000000
                w_P = (3*M_P)/(M_Ap)*1000000
                w_Ca = (5*M_Ca)/(M_Ap)*1000000
                self.mineral_chem[mineral]["F"] = w_F
                self.mineral_chem[mineral]["P"] = w_P
                self.mineral_chem[mineral]["Ca"] = w_Ca
            elif mineral == "Apatite-OH":  # Ca5(PO4)3OH
                M_Ap = 5*M_Ca + 3*(M_P + 4*M_O) + (M_O + M_H)
                w_H = (M_H)/(M_Ap)*1000000
                w_P = (3*M_P)/(M_Ap)*1000000
                w_Ca = (5*M_Ca)/(M_Ap)*1000000
                self.mineral_chem[mineral]["H"] = w_H
                self.mineral_chem[mineral]["P"] = w_P
                self.mineral_chem[mineral]["Ca"] = w_Ca
            elif mineral == "Forsterite":  # Mg2SiO4
                M_Ol = 2*M_Mg + M_Si + 4*M_O
                w_Mg = (2*M_Mg)/(M_Ol)*1000000
                w_Si = (M_Si)/(M_Ol)*1000000
                self.mineral_chem[mineral]["Mg"] = w_Mg
                self.mineral_chem[mineral]["Si"] = w_Si
            elif mineral == "Fayalite":  # Fe2SiO4
                M_Ol = 2*M_Fe + M_Si + 4*M_O
                w_Si = (M_Si)/(M_Ol)*1000000
                w_Fe = (2*M_Fe)/(M_Ol)*1000000
                self.mineral_chem[mineral]["Si"] = w_Si
                self.mineral_chem[mineral]["Fe"] = w_Fe
            elif mineral == "Gahnite":  # ZnAl2O4
                M = M_Zn + 2*M_Al + 4*M_O
                w_O = (4*M_O)/(M)*1000000
                w_Al = (2*M_Al)/(M)*1000000
                w_Zn = (M_Zn)/(M)*1000000
                self.mineral_chem[mineral]["O"] = w_O
                self.mineral_chem[mineral]["Al"] = w_Al
                self.mineral_chem[mineral]["Zn"] = w_Zn
            elif mineral == "Galena":  # PbS
                M_Gn = M_Pb + M_S
                w_S = (M_S)/(M_Gn)*1000000
                w_Pb = (M_Pb)/(M_Gn)*1000000
                self.mineral_chem[mineral]["S"] = w_S
                self.mineral_chem[mineral]["Pb"] = w_Pb
            elif mineral == "Hematite":  # Fe2O3
                M = 2*M_Fe + 3*M_O
                w_O = (3*M_O)/(M)*1000000
                w_Fe = (2*M_Fe)/(M)*1000000
                self.mineral_chem[mineral]["O"] = w_O
                self.mineral_chem[mineral]["Fe"] = w_Fe
            elif mineral == "Tephroite":  # Mn2SiO4
                M_Ol = 2*M_Mn + M_Si + 4*M_O
                w_Si = (M_Si)/(M_Ol)*1000000
                w_Mn = (2*M_Mn)/(M_Ol)*1000000
                self.mineral_chem[mineral]["Si"] = w_Si
                self.mineral_chem[mineral]["Mn"] = w_Mn
            elif mineral == "Albite":  # NaAlSi3O8
                M_Ab = M_Na + M_Al + 3*M_Si + 8*M_O
                w_Na = (M_Na)/(M_Ab)*1000000
                w_Al = (M_Al)/(M_Ab)*1000000
                w_Si = (3*M_Si)/(M_Ab)*1000000
                self.mineral_chem[mineral]["Na"] = w_Na
                self.mineral_chem[mineral]["Al"] = w_Al
                self.mineral_chem[mineral]["Si"] = w_Si
            elif mineral == "Anorthite":  # CaAl2Si2O8
                M_An = M_Ca + 2*(M_Al + M_Si) + 8*M_O
                w_Al = (2*M_Al)/(M_An)*1000000
                w_Si = (2*M_Si)/(M_An)*1000000
                w_Ca = (M_Ca)/(M_An)*1000000
                self.mineral_chem[mineral]["Al"] = w_Al
                self.mineral_chem[mineral]["Si"] = w_Si
                self.mineral_chem[mineral]["Ca"] = w_Ca
            elif mineral == "Magnetite":  # Fe3O4
                M = 3*M_Fe + 4*M_O
                w_O = (4*M_O)/(M)*1000000
                w_Fe = (3*M_Fe)/(M)*1000000
                self.mineral_chem[mineral]["O"] = w_O
                self.mineral_chem[mineral]["Fe"] = w_Fe
            elif mineral == "Molybdenite":  # MoS2
                M = M_Mo + 2*M_S
                w_S = (2*M_S)/(M)*1000000
                w_Mo = (M_Mo)/(M)*1000000
                self.mineral_chem[mineral]["S"] = w_S
                self.mineral_chem[mineral]["Mo"] = w_Mo
            elif mineral == "Pyrite":  # FeS2
                M_Py = M_Fe + 2*M_S
                w_S = (2*M_S)/(M_Py)*1000000
                w_Fe = (M_Fe)/(M_Py)*1000000
                self.mineral_chem[mineral]["S"] = w_S
                self.mineral_chem[mineral]["Fe"] = w_Fe
            elif mineral in ["Orthoclase", "Microcline"]:  # KAlSi3O8
                M_OrMc = M_K + M_Al + 3*M_Si + 8*M_O
                w_Al = (M_Al)/(M_OrMc)*1000000
                w_Si = (3*M_Si)/(M_OrMc)*1000000
                w_K = (M_K)/(M_OrMc)*1000000
                self.mineral_chem[mineral]["Al"] = w_Al
                self.mineral_chem[mineral]["Si"] = w_Si
                self.mineral_chem[mineral]["K"] = w_K
            elif mineral == "Sphalerite":  # ZnS
                M_Sp = M_Zn + M_S
                w_S = (M_S)/(M_Sp)*1000000
                w_Zn = (M_Zn)/(M_Sp)*1000000
                self.mineral_chem[mineral]["S"] = w_S
                self.mineral_chem[mineral]["Zn"] = w_Zn
            elif mineral == "Zircon":  # ZrSiO4
                M_Zrn = M_Zr + (M_Si + 4*M_O)
                w_Si = (M_Si)/(M_Zrn)*1000000
                w_Zr = (M_Zr)/(M_Zrn)*1000000
                self.mineral_chem[mineral]["Si"] = w_Si
                self.mineral_chem[mineral]["Zr"] = w_Zr
            elif mineral == "Meionite":  # Ca4 Al6 Si6 O24 CO3
                M = 4*M_Ca + 6*M_Al + 6*M_Si + 24*M_O + M_C + 3*M_O
                w_C = (M_C)/(M)*1000000
                w_Al = (6*M_Al)/(M)*1000000
                w_Si = (6*M_Si)/(M)*1000000
                w_Ca = (4*M_Ca)/(M)*1000000
                self.mineral_chem[mineral]["C"] = w_C
                self.mineral_chem[mineral]["Al"] = w_Al
                self.mineral_chem[mineral]["Si"] = w_Si
                self.mineral_chem[mineral]["Ca"] = w_Ca
            elif mineral == "Marialite":  # Na4 Al3 Si9 O24 Cl
                M = 4*M_Na + 3*M_Al + 9*M_Si + 24*M_O + M_Cl
                w_Na = (4*M_Na)/(M)*1000000
                w_Al = (3*M_Al)/(M)*1000000
                w_Si = (9*M_Si)/(M)*1000000
                w_Cl = (M_Cl)/(M)*1000000
                self.mineral_chem[mineral]["Na"] = w_Na
                self.mineral_chem[mineral]["Al"] = w_Al
                self.mineral_chem[mineral]["Si"] = w_Si
                self.mineral_chem[mineral]["Cl"] = w_Cl
            elif mineral == "Strontianite":  # Sr C O3
                M = M_Sr + M_C + 3*M_O
                w_C = (M_C)/(M)*1000000
                w_O = (3*M_O)/(M)*1000000
                w_Sr = (M_Sr)/(M)*1000000
                self.mineral_chem[mineral]["C"] = w_C
                self.mineral_chem[mineral]["O"] = w_O
                self.mineral_chem[mineral]["Sr"] = w_Sr
            elif mineral == "Titanite":  # Ca Ti Si O5
                M = M_Ca + M_Ti + M_Si + 5*M_O
                w_O = (5*M_O)/(M)*1000000
                w_Si = (M_Si)/(M)*1000000
                w_Ca = (M_Ca)/(M)*1000000
                w_Ti = (M_Ti)/(M)*1000000
                self.mineral_chem[mineral]["O"] = w_O
                self.mineral_chem[mineral]["Si"] = w_Si
                self.mineral_chem[mineral]["Ca"] = w_Ca
                self.mineral_chem[mineral]["Ti"] = w_Ti
            elif mineral == "Aegirine":  # Na Fe Si2 O6
                M = M_Na + M_Fe + 2*M_Si + 6*M_O
                w_O = (6*M_O)/(M)*1000000
                w_Na = (M_Na)/(M)*1000000
                w_Si = (2*M_Si)/(M)*1000000
                w_Fe = (M_Fe)/(M)*1000000
                self.mineral_chem[mineral]["O"] = w_O
                self.mineral_chem[mineral]["Na"] = w_Na
                self.mineral_chem[mineral]["Si"] = w_Si
                self.mineral_chem[mineral]["Fe"] = w_Fe
            elif mineral == "Diopside":  # Mg Ca Si2 O6
                M = M_Mg + M_Ca + 2*M_Si + 6*M_O
                w_O = (6*M_O)/(M)*1000000
                w_Mg = (M_Mg)/(M)*1000000
                w_Si = (2*M_Si)/(M)*1000000
                w_Ca = (M_Ca)/(M)*1000000
                self.mineral_chem[mineral]["O"] = w_O
                self.mineral_chem[mineral]["Mg"] = w_Mg
                self.mineral_chem[mineral]["Si"] = w_Si
                self.mineral_chem[mineral]["Ca"] = w_Ca
            elif mineral == "Hedenbergite":  # Fe Ca Si2 O6
                M = M_Fe + M_Ca + 2*M_Si + 6*M_O
                w_O = (6*M_O)/(M)*1000000
                w_Si = (2*M_Si)/(M)*1000000
                w_Ca = (M_Ca)/(M)*1000000
                w_Fe = (M_Fe)/(M)*1000000
                self.mineral_chem[mineral]["O"] = w_O
                self.mineral_chem[mineral]["Si"] = w_Si
                self.mineral_chem[mineral]["Ca"] = w_Ca
                self.mineral_chem[mineral]["Fe"] = w_Fe
            elif mineral == "Ferrosilite":  # Fe2 Si2 O6
                M = 2*M_Fe + 2*M_Si + 6*M_O
                w_O = (6*M_O)/(M)*1000000
                w_Si = (2*M_Si)/(M)*1000000
                w_Fe = (2*M_Fe)/(M)*1000000
                self.mineral_chem[mineral]["O"] = w_O
                self.mineral_chem[mineral]["Si"] = w_Si
                self.mineral_chem[mineral]["Fe"] = w_Fe
            elif mineral == "Enstatite":  # Mg2 Si2 O6
                M = 2*M_Mg + 2*M_Si + 6*M_O
                w_O = (6*M_O)/(M)*1000000
                w_Mg = (2*M_Mg)/(M)*1000000
                w_Si = (2*M_Si)/(M)*1000000
                self.mineral_chem[mineral]["O"] = w_O
                self.mineral_chem[mineral]["Mg"] = w_Mg
                self.mineral_chem[mineral]["Si"] = w_Si
            elif mineral == "Monazite-La":  # La P O4
                M = M_La + M_P + 4*M_O
                w_O = (4*M_O)/(M)*1000000
                w_P = (M_P)/(M)*1000000
                w_La = (M_La)/(M)*1000000
                self.mineral_chem[mineral]["O"] = w_O
                self.mineral_chem[mineral]["P"] = w_P
                self.mineral_chem[mineral]["La"] = w_La
            elif mineral == "Monazite-Ce":  # Ce P O4
                M = M_Ce + M_P + 4*M_O
                w_O = (4*M_O)/(M)*1000000
                w_P = (M_P)/(M)*1000000
                w_Ce = (M_Ce)/(M)*1000000
                self.mineral_chem[mineral]["O"] = w_O
                self.mineral_chem[mineral]["P"] = w_P
                self.mineral_chem[mineral]["Ce"] = w_Ce
            elif mineral == "Monazite-Pr":  # Pr P O4
                M = M_Pr + M_P + 4*M_O
                w_O = (4*M_O)/(M)*1000000
                w_P = (M_P)/(M)*1000000
                w_Pr = (M_Pr)/(M)*1000000
                self.mineral_chem[mineral]["O"] = w_O
                self.mineral_chem[mineral]["P"] = w_P
                self.mineral_chem[mineral]["Pr"] = w_Pr
            elif mineral == "Monazite-Nd":  # Nd P O4
                M = M_Nd + M_P + 4*M_O
                w_O = (4*M_O)/(M)*1000000
                w_P = (M_P)/(M)*1000000
                w_Nd = (M_Nd)/(M)*1000000
                self.mineral_chem[mineral]["O"] = w_O
                self.mineral_chem[mineral]["P"] = w_P
                self.mineral_chem[mineral]["Nd"] = w_Nd
            elif mineral == "Monazite-Sm":  # Sm P O4
                M = M_Sm + M_P + 4*M_O
                w_O = (4*M_O)/(M)*1000000
                w_P = (M_P)/(M)*1000000
                w_Sm = (M_Sm)/(M)*1000000
                self.mineral_chem[mineral]["O"] = w_O
                self.mineral_chem[mineral]["P"] = w_P
                self.mineral_chem[mineral]["Sm"] = w_Sm
            elif mineral == "Monazite-Eu":  # Eu P O4
                M = M_Eu + M_P + 4*M_O
                w_O = (4*M_O)/(M)*1000000
                w_P = (M_P)/(M)*1000000
                w_Eu = (M_Eu)/(M)*1000000
                self.mineral_chem[mineral]["O"] = w_O
                self.mineral_chem[mineral]["P"] = w_P
                self.mineral_chem[mineral]["Eu"] = w_Eu
            elif mineral == "Monazite-Gd":  # Gd P O4
                M = M_Gd + M_P + 4*M_O
                w_O = (4*M_O)/(M)*1000000
                w_P = (M_P)/(M)*1000000
                w_Gd = (M_Gd)/(M)*1000000
                self.mineral_chem[mineral]["O"] = w_O
                self.mineral_chem[mineral]["P"] = w_P
                self.mineral_chem[mineral]["Gd"] = w_Gd
            elif mineral == "Monazite-Th":  # Th P O4
                M = M_Th + M_P + 4*M_O
                w_O = (4*M_O)/(M)*1000000
                w_P = (M_P)/(M)*1000000
                w_Th = (M_Th)/(M)*1000000
                self.mineral_chem[mineral]["O"] = w_O
                self.mineral_chem[mineral]["P"] = w_P
                self.mineral_chem[mineral]["Th"] = w_Th
            elif mineral == "Select Mineral":
                pass
            #
            for element, value in self.mineral_chem[mineral].items():
                self.mineral_chem[mineral][element] = round(value, 4)

    #
    def place_mineral_values(self, var_min, header_col):
        #
        lbl_srm_03 = SE(parent=self.parent, row_id=0, column_id=header_col, n_rows=1, n_columns=42, fg=self.green_light,
                        bg=self.green_dark).create_simple_label(text=str(var_min) + " - Element Concentrations (ppm)",
                                                                relief=tk.GROOVE, fontsize="sans 10 bold")
        self.container_elements["SRM"]["Label"].append(lbl_srm_03)
        #
        self.srm_actual[var_min] = {}
        M_H = 1.008
        M_C = 12.011
        M_O = 15.999
        M_F = 18.998
        M_Na = 22.990
        M_Mg = 24.305
        M_Al = 26.982
        M_Si = 28.085
        M_P = 30.974
        M_Cl = 35.450
        M_K = 39.098
        M_Ca = 40.078
        M_Mn = 54.938
        M_Fe = 55.845
        M_Zr = 91.224
        M_Ba = 137.33
        #
        if var_min == "Quartz":  # SiO2
            M_Qz = M_Si + 2*M_O
            w_Si = (M_Si)/(M_Qz)*1000000
            self.srm_actual[var_min]["Si"] = w_Si
        elif var_min == "Arsenopyrite":
            self.srm_actual[var_min]["S"] = self.mineral_chem[var_min]["S"]
            self.srm_actual[var_min]["Fe"] = self.mineral_chem[var_min]["Fe"]
            self.srm_actual[var_min]["As"] = self.mineral_chem[var_min]["As"]
        elif var_min == "Barite":  # BaSO4
            self.srm_actual[var_min]["S"] = self.mineral_chem[var_min]["S"]
            self.srm_actual[var_min]["Ba"] = self.mineral_chem[var_min]["Ba"]
        elif var_min == "Ilvaite":  # CaFe3(SiO4)2(OH)
            self.srm_actual[var_min]["H"] = self.mineral_chem[var_min]["H"]
            self.srm_actual[var_min]["Si"] = self.mineral_chem[var_min]["Si"]
            self.srm_actual[var_min]["Ca"] = self.mineral_chem[var_min]["Ca"]
            self.srm_actual[var_min]["Fe"] = self.mineral_chem[var_min]["Fe"]
        elif var_min == "Bornite":
            self.srm_actual[var_min]["S"] = self.mineral_chem[var_min]["S"]
            self.srm_actual[var_min]["Fe"] = self.mineral_chem[var_min]["Fe"]
            self.srm_actual[var_min]["Cu"] = self.mineral_chem[var_min]["Cu"]
        elif var_min == "Calcite":  # CaCO3
            M_Cal = M_Ca + M_C + 3*M_O
            w_C = (M_C)/(M_Cal)*1000000
            w_Ca = (M_Ca)/(M_Cal)*1000000
            self.srm_actual[var_min]["C"] = w_C
            self.srm_actual[var_min]["Ca"] = w_Ca
        elif var_min == "Chalcopyrite":
            self.srm_actual[var_min]["S"] = self.mineral_chem[var_min]["S"]
            self.srm_actual[var_min]["Fe"] = self.mineral_chem[var_min]["Fe"]
            self.srm_actual[var_min]["Cu"] = self.mineral_chem[var_min]["Cu"]
        elif var_min == "Chromite":
            self.srm_actual[var_min]["Cr"] = self.mineral_chem[var_min]["Cr"]
            self.srm_actual[var_min]["Fe"] = self.mineral_chem[var_min]["Fe"]
        elif var_min == "Sphalerite":
            self.srm_actual[var_min]["S"] = self.mineral_chem[var_min]["S"]
            self.srm_actual[var_min]["Zn"] = self.mineral_chem[var_min]["Zn"]
        elif var_min == "Enargite":
            self.srm_actual[var_min]["S"] = self.mineral_chem[var_min]["S"]
            self.srm_actual[var_min]["Cu"] = self.mineral_chem[var_min]["Cu"]
            self.srm_actual[var_min]["As"] = self.mineral_chem[var_min]["As"]
        elif var_min == "Fluorite":  # CaF2
            M_Fl = M_Ca + 2*M_F
            w_F = (2*M_F)/(M_Fl)*1000000
            w_Ca = (M_Ca)/(M_Fl)*1000000
            self.srm_actual[var_min]["F"] = w_F
            self.srm_actual[var_min]["Ca"] = w_Ca
        elif var_min == "Hematite":
            self.srm_actual[var_min]["Fe"] = self.mineral_chem[var_min]["Fe"]
        elif var_min == "Galena":
            self.srm_actual[var_min]["S"] = self.mineral_chem[var_min]["S"]
            self.srm_actual[var_min]["Pb"] = self.mineral_chem[var_min]["Pb"]
        elif var_min == "Gahnite":
            self.srm_actual[var_min]["Al"] = self.mineral_chem[var_min]["Al"]
            self.srm_actual[var_min]["Zn"] = self.mineral_chem[var_min]["Zn"]
        elif var_min == "Magnetite":
            self.srm_actual[var_min]["Fe"] = self.mineral_chem[var_min]["Fe"]
        elif var_min == "Molybdenite":
            self.srm_actual[var_min]["S"] = self.mineral_chem[var_min]["S"]
            self.srm_actual[var_min]["Mo"] = self.mineral_chem[var_min]["Mo"]
        elif var_min == "Pyrite":
            self.srm_actual[var_min]["S"] = self.mineral_chem[var_min]["S"]
            self.srm_actual[var_min]["Fe"] = self.mineral_chem[var_min]["Fe"]
        elif var_min == "Apatite-Cl":  # Ca5(PO4)3Cl
            M_Ap = 5*M_Ca + 3*(M_P + 4*M_O) + M_Cl
            w_P = (3*M_P)/(M_Ap)*1000000
            w_Cl = (M_Cl)/(M_Ap)*1000000
            w_Ca = (5*M_Ca)/(M_Ap)*1000000
            self.srm_actual[var_min]["P"] = w_P
            self.srm_actual[var_min]["Cl"] = w_Cl
            self.srm_actual[var_min]["Ca"] = w_Ca
        elif var_min == "Apatite-F":  # Ca5(PO4)3F
            M_Ap = 5*M_Ca + 3*(M_P + 4*M_O) + M_F
            w_F = (M_F)/(M_Ap)*1000000
            w_P = (3*M_P)/(M_Ap)*1000000
            w_Ca = (5*M_Ca)/(M_Ap)*1000000
            self.srm_actual[var_min]["F"] = w_F
            self.srm_actual[var_min]["P"] = w_P
            self.srm_actual[var_min]["Ca"] = w_Ca
        elif var_min == "Apatite-OH":  # Ca5(PO4)3OH
            M_Ap = 5*M_Ca + 3*(M_P + 4*M_O) + (M_O + M_H)
            w_H = (M_H)/(M_Ap)*1000000
            w_P = (3*M_P)/(M_Ap)*1000000
            w_Ca = (5*M_Ca)/(M_Ap)*1000000
            self.srm_actual[var_min]["H"] = w_H
            self.srm_actual[var_min]["P"] = w_P
            self.srm_actual[var_min]["Ca"] = w_Ca
        elif var_min == "Forsterite":  # Mg2SiO4
            M_Ol = 2*M_Mg + M_Si + 4*M_O
            w_Mg = (2*M_Mg)/(M_Ol)*1000000
            w_Si = (M_Si)/(M_Ol)*1000000
            self.srm_actual[var_min]["Mg"] = w_Mg
            self.srm_actual[var_min]["Si"] = w_Si
        elif var_min == "Fayalite":  # Fe2SiO4
            M_Ol = 2*M_Fe + M_Si + 4*M_O
            w_Si = (M_Si)/(M_Ol)*1000000
            w_Fe = (2*M_Fe)/(M_Ol)*1000000
            self.srm_actual[var_min]["Si"] = w_Si
            self.srm_actual[var_min]["Fe"] = w_Fe
        elif var_min == "Tephroite":  # Mn2SiO4
            M_Ol = 2*M_Mn + M_Si + 4*M_O
            w_Si = (M_Si)/(M_Ol)*1000000
            w_Mn = (2*M_Mn)/(M_Ol)*1000000
            self.srm_actual[var_min]["Si"] = w_Si
            self.srm_actual[var_min]["Mn"] = w_Mn
        elif var_min == "Albite":  # NaAlSi3O8
            M_Ab = M_Na + M_Al + 3*M_Si + 8*M_O
            w_Na = (M_Na)/(M_Ab)*1000000
            w_Al = (M_Al)/(M_Ab)*1000000
            w_Si = (3*M_Si)/(M_Ab)*1000000
            self.srm_actual[var_min]["Na"] = w_Na
            self.srm_actual[var_min]["Al"] = w_Al
            self.srm_actual[var_min]["Si"] = w_Si
        elif var_min == "Anorthite":  # CaAl2Si2O8
            M_An = M_Ca + 2*(M_Al + M_Si) + 8*M_O
            w_Al = (2*M_Al)/(M_An)*1000000
            w_Si = (2*M_Si)/(M_An)*1000000
            w_Ca = (M_Ca)/(M_An)*1000000
            self.srm_actual[var_min]["Al"] = w_Al
            self.srm_actual[var_min]["Si"] = w_Si
            self.srm_actual[var_min]["Ca"] = w_Ca
        elif var_min in ["Orthoclase", "Microcline"]:  # KAlSi3O8
            M_OrMc = M_K + M_Al + 3*M_Si + 8*M_O
            w_Al = (M_Al)/(M_OrMc)*1000000
            w_Si = (3*M_Si)/(M_OrMc)*1000000
            w_K = (M_K)/(M_OrMc)*1000000
            self.srm_actual[var_min]["Al"] = w_Al
            self.srm_actual[var_min]["Si"] = w_Si
            self.srm_actual[var_min]["K"] = w_K
        elif var_min == "Zircon":  # ZrSiO4
            M_Zrn = M_Zr + (M_Si + 4*M_O)
            w_Si = (M_Si)/(M_Zrn)*1000000
            w_Zr = (M_Zr)/(M_Zrn)*1000000
            self.srm_actual[var_min]["Si"] = w_Si
            self.srm_actual[var_min]["Zr"] = w_Zr
        elif var_min == "Meionite":
            self.srm_actual[var_min]["C"] = self.mineral_chem[var_min]["C"]
            self.srm_actual[var_min]["Al"] = self.mineral_chem[var_min]["Al"]
            self.srm_actual[var_min]["Si"] = self.mineral_chem[var_min]["Si"]
            self.srm_actual[var_min]["Ca"] = self.mineral_chem[var_min]["Ca"]
        elif var_min == "Marialite":
            self.srm_actual[var_min]["Na"] = self.mineral_chem[var_min]["Na"]
            self.srm_actual[var_min]["Al"] = self.mineral_chem[var_min]["Al"]
            self.srm_actual[var_min]["Si"] = self.mineral_chem[var_min]["Si"]
            self.srm_actual[var_min]["Cl"] = self.mineral_chem[var_min]["Cl"]
        elif var_min == "Strontianite":
            self.srm_actual[var_min]["C"] = self.mineral_chem[var_min]["C"]
            self.srm_actual[var_min]["Sr"] = self.mineral_chem[var_min]["Sr"]
        elif var_min == "Titanite":
            self.srm_actual[var_min]["Si"] = self.mineral_chem[var_min]["Si"]
            self.srm_actual[var_min]["Ca"] = self.mineral_chem[var_min]["Ca"]
            self.srm_actual[var_min]["Ti"] = self.mineral_chem[var_min]["Ti"]
        elif var_min == "Aegirine":
            self.srm_actual[var_min]["Na"] = self.mineral_chem[var_min]["Na"]
            self.srm_actual[var_min]["Si"] = self.mineral_chem[var_min]["Si"]
            self.srm_actual[var_min]["Fe"] = self.mineral_chem[var_min]["Fe"]
        elif var_min == "Diopside":
            self.srm_actual[var_min]["Mg"] = self.mineral_chem[var_min]["Mg"]
            self.srm_actual[var_min]["Si"] = self.mineral_chem[var_min]["Si"]
            self.srm_actual[var_min]["Ca"] = self.mineral_chem[var_min]["Ca"]
        elif var_min == "Hedenbergite":
            self.srm_actual[var_min]["Si"] = self.mineral_chem[var_min]["Si"]
            self.srm_actual[var_min]["Ca"] = self.mineral_chem[var_min]["Ca"]
            self.srm_actual[var_min]["Fe"] = self.mineral_chem[var_min]["Fe"]
        elif var_min == "Ferrosilite":
            self.srm_actual[var_min]["Si"] = self.mineral_chem[var_min]["Si"]
            self.srm_actual[var_min]["Fe"] = self.mineral_chem[var_min]["Fe"]
        elif var_min == "Enstatite":
            self.srm_actual[var_min]["Mg"] = self.mineral_chem[var_min]["Mg"]
            self.srm_actual[var_min]["Si"] = self.mineral_chem[var_min]["Si"]
        elif var_min == "Monazite-La":
            self.srm_actual[var_min]["P"] = self.mineral_chem[var_min]["P"]
            self.srm_actual[var_min]["La"] = self.mineral_chem[var_min]["La"]
        elif var_min == "Monazite-Ce":
            self.srm_actual[var_min]["P"] = self.mineral_chem[var_min]["P"]
            self.srm_actual[var_min]["Ce"] = self.mineral_chem[var_min]["Ce"]
        elif var_min == "Monazite-Pr":
            self.srm_actual[var_min]["P"] = self.mineral_chem[var_min]["P"]
            self.srm_actual[var_min]["Pr"] = self.mineral_chem[var_min]["Pr"]
        elif var_min == "Monazite-Nd":
            self.srm_actual[var_min]["P"] = self.mineral_chem[var_min]["P"]
            self.srm_actual[var_min]["Nd"] = self.mineral_chem[var_min]["Nd"]
        elif var_min == "Monazite-Sm":
            self.srm_actual[var_min]["P"] = self.mineral_chem[var_min]["P"]
            self.srm_actual[var_min]["Sm"] = self.mineral_chem[var_min]["Sm"]
        elif var_min == "Monazite-Eu":
            self.srm_actual[var_min]["P"] = self.mineral_chem[var_min]["P"]
            self.srm_actual[var_min]["Eu"] = self.mineral_chem[var_min]["Eu"]
        elif var_min == "Monazite-Gd":
            self.srm_actual[var_min]["P"] = self.mineral_chem[var_min]["P"]
            self.srm_actual[var_min]["Gd"] = self.mineral_chem[var_min]["Gd"]
        elif var_min == "Monazite-Th":
            self.srm_actual[var_min]["P"] = self.mineral_chem[var_min]["P"]
            self.srm_actual[var_min]["Th"] = self.mineral_chem[var_min]["Th"]
        #
        for element in self.list_pse:
            if element in self.srm_actual[var_min]:
                self.container_var["SRM"][element].set(self.srm_actual[var_min][element])
            else:
                self.container_var["SRM"][element].set(0.0)

    def select_mineral_is(self, var_min, fluidinclusion=False):
        self.srm_actual[var_min] = {}

        if var_min in self.mineral_chem:
            for key, value in self.mineral_chem[var_min].items():
                self.srm_actual[var_min][key] = value

            if fluidinclusion == False:
                if self.file_loaded == False:
                    self.container_var["isotopes"]["default"].set("Select IS")
                    self.container_var["settings"]["IS Concentration"].set(0.0)
            if len(self.container_var["mineralchemistry"]) > 0:
                self.container_var["mineralchemistry"].clear()
            self.container_var["mineralchemistry"].extend(list(self.mineral_chem[var_min].keys()))
            self.container_var["mineralchemistry"].sort()

            possible_is = []
            for element in self.container_var["mineralchemistry"]:
                for isotope in self.container_lists["ISOTOPES"]:
                    key = re.search(r"(\D+)(\d+)", isotope)
                    if element == key.group(1):
                        possible_is.append(isotope)

            if fluidinclusion:
                list_fluidchemistry = ["H", "Na", "Mg", "Ca", "K", "Cl", "F", "Br", "I", "At"]
                for element in list_fluidchemistry:
                    for isotope in self.container_lists["ISOTOPES"]:
                        key = re.search(r"(\D+)(\d+)", isotope)
                        if element == key.group(1):
                            possible_is.append(isotope)

            if fluidinclusion == False:
                self.opt_is_std_def["menu"].delete(0, "end")
                self.opt_is_smpl_def["menu"].delete(0, "end")

                for index, isotope in enumerate(possible_is):
                    for file_long in self.container_lists["STD"]["Long"]:
                        if index == 0:
                            self.container_optionmenu["STD"][file_long]["menu"].delete(0, "end")

                        self.container_optionmenu["STD"][file_long]["menu"].add_command(
                            label=isotope, command=lambda element=isotope, file=file_long:
                            self.change_std_is(element, file))

                        if self.file_loaded is False:
                            self.container_var["STD"][file_long]["IS"].set("Select IS")
                    for file_long in self.container_lists["SMPL"]["Long"]:
                        if index == 0:
                            self.container_optionmenu["SMPL"][file_long]["menu"].delete(0, "end")

                        self.container_optionmenu["SMPL"][file_long]["menu"].add_command(
                            label=isotope,
                            command=lambda element=isotope, file=file_long, mineral=self.container_var["mineral"].get():
                            self.change_smpl_is(element, file, mineral))

                        if self.file_loaded is False:
                            self.container_var["SMPL"][file_long]["IS"].set("Select IS")

                    self.opt_is_std_def["menu"].add_command(
                        label=isotope, command=lambda element=isotope, mineral=var_min:
                        self.change_std_is_default(element, mineral))
                    self.opt_is_smpl_def["menu"].add_command(
                        label=isotope, command=lambda element=isotope, mineral=var_min:
                        self.change_smpl_is_default(element, mineral))

            self.container_lists["Possible IS"].clear()
            self.container_lists["Possible IS"].extend(possible_is)

            self.mineral_chem["Unknown"] = {}
            for element in self.container_var["mineralchemistry"]:
                for isotope in self.container_lists["ISOTOPES"]:
                    key = re.search(r"(\D+)(\d+)", isotope)
                    if element == key.group(1):
                        self.mineral_chem["Unknown"][element] = self.srm_actual[var_min][element]
                        self.mineral_chem["Unknown"][isotope] = self.srm_actual[var_min][element]

    def onclick(self, var, filename, ratio_mode, event, spikes=False):
        if spikes == False:
            if var.get() in [1, 2, 3, 4, 5]:
                if len(self.container_helper["positions"][filename]) == 2 and len(
                        self.container_helper["indices"][filename]) == 2:
                    self.container_helper["positions"][filename].clear()
                    self.container_helper["indices"][filename].clear()
                #
                x_nearest = min(self.times, key=lambda x: abs(x - event.xdata))
                self.container_helper["positions"][filename].append(x_nearest)
                self.container_helper["indices"][filename].append(self.times[self.times == x_nearest].index[0])
                #
                if len(self.container_helper["positions"][filename]) + len(
                        self.container_helper["indices"][filename]) == 4:
                    if var.get() == 1:  # BG
                        if self.file_type == "STD":
                            if len(self.container_helper["positions"]["BG STD"][filename]) > 0:
                                self.bg_id = self.container_helper["positions"]["BG STD"][filename][-1][4]
                        elif self.file_type == "SMPL":
                            if len(self.container_helper["positions"]["BG SMPL"][filename]) > 0:
                                self.bg_id = self.container_helper["positions"]["BG SMPL"][filename][-1][4]
                        #
                        self.bg_id += 1
                        self.bg_idlist.append(self.bg_id)
                        self.container_helper["limits BG"][self.file]["ID"].append(self.bg_id)
                        self.container_helper["limits BG"][self.file]["type"].append("custom")
                        self.container_helper["positions"]["BG"][filename].append(
                            [round(self.container_helper["positions"][filename][0], 4),
                             round(self.container_helper["positions"][filename][1], 4)])
                        self.container_listboxes[self.file_type][filename]["BG"][0].insert(
                            tk.END, "BG" + str(self.bg_id) + " [" + str(
                                self.container_helper["positions"][filename][0]) + "-" +
                                    str(self.container_helper["positions"][filename][1]) + "]")
                        #
                        if ratio_mode == False:
                            box_bg = self.ax.axvspan(
                                self.container_helper["positions"][filename][0],
                                self.container_helper["positions"][filename][1], alpha=0.25, color=self.blue_dark)
                            box_bg_ratio = self.ax_ratio.axvspan(
                                self.container_helper["positions"][filename][0],
                                self.container_helper["positions"][filename][1], alpha=0.25, color=self.blue_dark)
                            #
                            self.container_helper["limits BG"][self.file][str(self.bg_id)] = box_bg
                            self.container_helper["limits BG Ratio"][self.file][str(self.bg_id)] = box_bg_ratio
                            #
                            self.canvas.draw()
                            try:
                                self.canvas_ratio.draw()
                            except:
                                pass
                        else:
                            box_bg = self.ax.axvspan(
                                self.container_helper["positions"][filename][0],
                                self.container_helper["positions"][filename][1], alpha=0.25, color=self.blue_dark)
                            box_bg_ratio = self.ax_ratio.axvspan(
                                self.container_helper["positions"][filename][0],
                                self.container_helper["positions"][filename][1], alpha=0.25, color=self.blue_dark)
                            #
                            self.container_helper["limits BG"][self.file][str(self.bg_id)] = box_bg
                            self.container_helper["limits BG Ratio"][self.file][str(self.bg_id)] = box_bg_ratio
                            #
                            self.canvas.draw()
                            self.canvas_ratio.draw()
                        #
                        self.indices_bg = self.container_helper["indices"][filename]
                        #
                        self.container_files[self.file_type][self.filename_short]["BG"][self.bg_id] = {}
                        self.container_files[self.file_type][self.filename_short]["BG"][self.bg_id]["Times"] = [
                            self.container_helper["positions"][filename][0],
                            self.container_helper["positions"][filename][1]]
                        self.container_files[self.file_type][self.filename_short]["BG"][self.bg_id]["Positions"] = [
                            self.container_helper["indices"][filename][0],
                            self.container_helper["indices"][filename][1]]
                        self.container_files[self.file_type][self.filename_short]["BG"][self.bg_id]["Box"] = box_bg
                        #
                        if self.file_type == "STD":
                            self.container_helper["STD"][filename]["BG"][self.bg_id] = {
                                "Times": [self.container_helper["positions"][filename][0],
                                          self.container_helper["positions"][filename][1]],
                                "Positions": [self.container_helper["indices"][filename][0],
                                              self.container_helper["indices"][filename][1]],
                                "Object": [box_bg, box_bg_ratio]}
                            self.container_helper["positions"]["BG STD"][filename].append(
                                [self.container_helper["positions"][filename][0],
                                 self.container_helper["positions"][filename][1],
                                 self.container_helper["indices"][filename][0],
                                 self.container_helper["indices"][filename][1],
                                 self.bg_id])
                        elif self.file_type == "SMPL":
                            self.container_helper["SMPL"][filename]["BG"][self.bg_id] = {
                                "Times": [self.container_helper["positions"][filename][0],
                                          self.container_helper["positions"][filename][1]],
                                "Positions": [self.container_helper["indices"][filename][0],
                                              self.container_helper["indices"][filename][1]],
                                "Object": [box_bg, box_bg_ratio]}
                            self.container_helper["positions"]["BG SMPL"][filename].append(
                                [self.container_helper["positions"][filename][0],
                                 self.container_helper["positions"][filename][1],
                                 self.container_helper["indices"][filename][0],
                                 self.container_helper["indices"][filename][1],
                                 self.bg_id])
                    elif var.get() == 2:  # SIG
                        if self.file_type == "STD":
                            if len(self.container_helper["positions"]["SIG STD"][filename]) > 0:
                                self.sig_id = self.container_helper["positions"]["SIG STD"][filename][-1][4]
                        elif self.file_type == "SMPL":
                            if len(self.container_helper["positions"]["SIG SMPL"][filename]) > 0:
                                self.sig_id = self.container_helper["positions"]["SIG SMPL"][filename][-1][4]
                        self.sig_id += 1
                        self.sig_idlist.append(self.sig_id)
                        self.container_helper["limits SIG"][self.file]["ID"].append(self.sig_id)
                        self.container_helper["limits SIG"][self.file]["type"].append("custom")
                        self.container_helper["positions"]["SIG"][filename].append(
                            [round(self.container_helper["positions"][filename][0], 4),
                             round(self.container_helper["positions"][filename][1], 4)])
                        self.container_listboxes[self.file_type][filename]["SIG"][0].insert(
                            tk.END,
                            "SIG" + str(self.sig_id) + " [" + str(self.container_helper["positions"][filename][0]) + "-"
                            + str(self.container_helper["positions"][filename][1]) + "]")
                        #
                        if ratio_mode == False:
                            box_sig = self.ax.axvspan(
                                self.container_helper["positions"][filename][0],
                                self.container_helper["positions"][filename][1], alpha=0.25, color=self.brown_dark)
                            self.container_helper["limits SIG"][self.file][str(self.sig_id)] = box_sig
                            self.canvas.draw()
                            #
                            # box_sig_ratio = self.ax_ratio.axvspan(
                            #     self.container_helper["positions"][filename][0],
                            #     self.container_helper["positions"][filename][1], alpha=0.25, color=self.brown_dark)
                            # self.container_helper["limits SIG Ratio"][self.file][str(self.sig_id)] = box_sig_ratio
                            # self.canvas_ratio.draw()
                        else:
                            box_sig_ratio = self.ax_ratio.axvspan(
                                self.container_helper["positions"][filename][0],
                                self.container_helper["positions"][filename][1], alpha=0.25, color=self.brown_dark)
                            box_sig = self.ax.axvspan(
                                self.container_helper["positions"][filename][0],
                                self.container_helper["positions"][filename][1], alpha=0.25, color=self.brown_dark)
                            #
                            self.container_helper["limits SIG Ratio"][self.file][str(self.sig_id)] = box_sig_ratio
                            self.container_helper["limits SIG"][self.file][str(self.sig_id)] = box_sig
                            #
                            self.canvas_ratio.draw()
                            self.canvas.draw()
                        #
                        self.indices_sig = self.container_helper["indices"][filename]
                        #
                        self.container_files[self.file_type][self.filename_short]["SIG"][self.sig_id] = {}
                        self.container_files[self.file_type][self.filename_short]["SIG"][self.sig_id]["Times"] = [
                            self.container_helper["positions"][filename][0],
                            self.container_helper["positions"][filename][1]]
                        self.container_files[self.file_type][self.filename_short]["SIG"][self.sig_id]["Positions"] = [
                            self.container_helper["indices"][filename][0],
                            self.container_helper["indices"][filename][1]]
                        self.container_files[self.file_type][self.filename_short]["SIG"][self.sig_id]["Box"] = box_sig
                        #
                        if self.file_type == "STD":
                            self.container_helper["STD"][filename]["SIG"][self.sig_id] = {
                                "Times": [self.container_helper["positions"][filename][0],
                                          self.container_helper["positions"][filename][1]],
                                "Positions": [self.container_helper["indices"][filename][0],
                                              self.container_helper["indices"][filename][1]],
                                "Object": [box_sig, box_sig_ratio]}
                            self.container_helper["positions"]["SIG STD"][filename].append(
                                [self.container_helper["positions"][filename][0],
                                 self.container_helper["positions"][filename][1],
                                 self.container_helper["indices"][filename][0],
                                 self.container_helper["indices"][filename][1],
                                 self.sig_id])
                        elif self.file_type == "SMPL":
                            self.container_helper["SMPL"][filename]["SIG"][self.sig_id] = {
                                "Times": [self.container_helper["positions"][filename][0],
                                          self.container_helper["positions"][filename][1]],
                                "Positions": [self.container_helper["indices"][filename][0],
                                              self.container_helper["indices"][filename][1]],
                                "Object": [box_sig, box_sig_ratio]}
                            self.container_helper["positions"]["SIG SMPL"][filename].append(
                                [self.container_helper["positions"][filename][0],
                                 self.container_helper["positions"][filename][1],
                                 self.container_helper["indices"][filename][0],
                                 self.container_helper["indices"][filename][1],
                                 self.sig_id])
                    elif var.get() == 3:  # SPK
                        isotope_list = []
                        for isotope in self.container_lists["ISOTOPES"]:
                            if isotope not in self.container_helper[self.file_type][filename]["SPK"]:
                                self.container_helper[self.file_type][filename]["SPK"][isotope] = {}
                            if self.container_var["plotting"][isotope][1].get() == 1:
                                isotope_list.append(isotope)
                        for isotope in isotope_list:
                            if len(self.container_helper[self.file_type][filename]["SPK"][isotope]) == 0:
                                self.spk_id = 1
                            else:
                                self.spk_id = int(
                                    len(self.container_helper[self.file_type][filename]["SPK"][isotope]) + 1)
                        isotope = "".join(isotope_list)
                        self.spk_idlist.append(self.spk_id)
                        self.container_helper["limits SPK"][self.file]["ID"].append(self.spk_id)
                        self.container_helper["limits SPK"][self.file]["type"].append("custom")
                        self.container_helper["limits SPK"][self.file]["info"].append([isotope, self.spk_id])
                        self.container_helper["positions"]["SPK"][filename].append(
                            [round(self.container_helper["positions"][filename][0], 4),
                             round(self.container_helper["positions"][filename][1], 4)])
                        #
                        if len(isotope_list) > 1:
                            color_var = self.yellow_dark
                        else:
                            color_var = self.isotope_colors[isotope]
                        self.container_listboxes[self.file_type][filename]["SPK"][0].insert(
                            tk.END, "[" + ", ".join(isotope_list) + "] #" + str(self.spk_id) + " [" + str(
                                self.container_helper["positions"][filename][0]) + "-" + str(
                                self.container_helper["positions"][filename][1]) + "]")
                        #
                        if ratio_mode == False:
                            box_spk = self.ax.axvspan(
                                self.container_helper["positions"][filename][0],
                                self.container_helper["positions"][filename][1], alpha=0.125, color=color_var)
                            box_spk_ratio = self.ax_ratio.axvspan(
                                self.container_helper["positions"][filename][0],
                                self.container_helper["positions"][filename][1], alpha=0.125, color=color_var)
                            #
                            self.container_helper["limits SPK"][self.file][str(self.spk_id)] = box_spk
                            self.container_helper["limits SPK Ratio"][self.file][str(self.spk_id)] = box_spk_ratio
                            #
                            self.canvas_ratio.draw()
                            self.canvas.draw()
                        else:
                            box_spk_ratio = self.ax_ratio.axvspan(
                                self.container_helper["positions"][filename][0],
                                self.container_helper["positions"][filename][1], alpha=0.125, color=color_var)
                            box_spk = self.ax.axvspan(
                                self.container_helper["positions"][filename][0],
                                self.container_helper["positions"][filename][1], alpha=0.125, color=color_var)
                            #
                            self.container_helper["limits SPK Ratio"][self.file][str(self.spk_id)] = box_spk_ratio
                            self.container_helper["limits SPK"][self.file][str(self.spk_id)] = box_spk
                            #
                            self.canvas_ratio.draw()
                            self.canvas.draw()
                        #
                        self.container_files[self.file_type][self.filename_short]["SPK"][self.spk_id] = {}
                        for isotope in isotope_list:
                            if isotope not in self.spikes_isotopes[self.file_type][filename]:
                                self.spikes_isotopes[self.file_type][filename][isotope] = []
                                self.spikes_isotopes[self.file_type][filename][isotope].append(
                                    [self.container_helper["indices"][filename][0],
                                     self.container_helper["indices"][filename][1]])
                            else:
                                self.spikes_isotopes[self.file_type][filename][isotope].append(
                                    [self.container_helper["indices"][filename][0],
                                     self.container_helper["indices"][filename][1]])
                            #
                            self.container_files[self.file_type][self.filename_short]["SPK"][self.spk_id][
                                "Isotope"] = isotope
                            self.container_files[self.file_type][self.filename_short]["SPK"][self.spk_id][
                                "Times"] = [
                                self.container_helper["positions"][filename][0],
                                self.container_helper["positions"][filename][1]]
                            self.container_files[self.file_type][self.filename_short]["SPK"][self.spk_id][
                                "Positions"] = [
                                self.container_helper["indices"][filename][0],
                                self.container_helper["indices"][filename][1]]
                            self.container_files[self.file_type][self.filename_short]["SPK"][self.spk_id][
                                "Box"] = box_spk
                            #
                            if self.file_type == "STD":
                                self.container_helper["STD"][filename]["SPK"][isotope][self.spk_id] = {
                                    "Times": [self.container_helper["positions"][filename][0],
                                              self.container_helper["positions"][filename][1]],
                                    "Positions": [self.container_helper["indices"][filename][0],
                                                  self.container_helper["indices"][filename][1]],
                                    "Object": [box_spk, box_spk_ratio]}
                            elif self.file_type == "SMPL":
                                self.container_helper["SMPL"][filename]["SPK"][isotope][self.spk_id] = {
                                    "Times": [self.container_helper["positions"][filename][0],
                                              self.container_helper["positions"][filename][1]],
                                    "Positions": [self.container_helper["indices"][filename][0],
                                                  self.container_helper["indices"][filename][1]],
                                    "Object": [box_spk, box_spk_ratio]}
                    #
                    elif var.get() == 4:  # MAT
                        if self.file_type == "STD":
                            if len(self.container_helper["positions"]["MAT STD"][filename]) > 0:
                                self.mat_id = self.container_helper["positions"]["MAT STD"][filename][-1][4]
                        elif self.file_type == "SMPL":
                            if len(self.container_helper["positions"]["MAT SMPL"][filename]) > 0:
                                self.mat_id = self.container_helper["positions"]["MAT SMPL"][filename][-1][4]
                        self.mat_id += 1
                        self.mat_idlist.append(self.mat_id)
                        self.container_helper["limits MAT"][self.file]["ID"].append(self.mat_id)
                        self.container_helper["limits MAT"][self.file]["type"].append("custom")
                        self.container_helper["positions"]["MAT"][filename].append(
                            [round(self.container_helper["positions"][filename][0], 4),
                             round(self.container_helper["positions"][filename][1], 4)])
                        self.container_listboxes[self.file_type][filename]["MAT"][0].insert(
                            tk.END,
                            "MAT" + str(self.mat_id) + " [" + str(self.container_helper["positions"][filename][0]) + "-"
                            + str(self.container_helper["positions"][filename][1]) + "]")
                        #
                        # if ratio_mode == False:
                        #     box_mat = self.ax.axvspan(
                        #         self.container_helper["positions"][filename][0],
                        #         self.container_helper["positions"][filename][1], alpha=0.25, color=self.brown_dark)
                        #     self.container_helper["limits MAT"][self.file][str(self.mat_id)] = box_mat
                        #     #
                        #     self.canvas.draw()
                        # else:
                        #     box_mat = self.ax_ratio.axvspan(
                        #         self.container_helper["positions"][filename][0],
                        #         self.container_helper["positions"][filename][1], alpha=0.25, color=self.brown_dark)
                        #     self.container_helper["limits MAT Ratio"][self.file][str(self.mat_id)] = box_mat
                        #     #
                        #     self.canvas_ratio.draw()
                        #
                        box_mat = self.ax.axvspan(
                            self.container_helper["positions"][filename][0],
                            self.container_helper["positions"][filename][1], alpha=0.25, color=self.brown_dark)
                        box_mat_ratio = self.ax_ratio.axvspan(
                            self.container_helper["positions"][filename][0],
                            self.container_helper["positions"][filename][1], alpha=0.25, color=self.brown_dark)
                        self.container_helper["limits MAT"][self.file][str(self.mat_id)] = box_mat
                        self.container_helper["limits MAT Ratio"][self.file][str(self.mat_id)] = box_mat_ratio
                        #
                        self.canvas_ratio.draw()
                        self.canvas.draw()
                        #
                        self.indices_mat = self.container_helper["indices"][filename]
                        #
                        self.container_files[self.file_type][self.filename_short]["MAT"][self.mat_id] = {}
                        self.container_files[self.file_type][self.filename_short]["MAT"][self.mat_id]["Times"] = [
                            self.container_helper["positions"][filename][0],
                            self.container_helper["positions"][filename][1]]
                        self.container_files[self.file_type][self.filename_short]["MAT"][self.mat_id]["Positions"] = [
                            self.container_helper["indices"][filename][0],
                            self.container_helper["indices"][filename][1]]
                        self.container_files[self.file_type][self.filename_short]["MAT"][self.mat_id]["Box"] = box_mat
                        #
                        if self.file_type == "STD":
                            self.container_helper["STD"][filename]["MAT"][self.mat_id] = {
                                "Times": [self.container_helper["positions"][filename][0],
                                          self.container_helper["positions"][filename][1]],
                                "Positions": [self.container_helper["indices"][filename][0],
                                              self.container_helper["indices"][filename][1]],
                                "Object": [box_mat, box_mat_ratio]}
                            self.container_helper["positions"]["MAT STD"][filename].append(
                                [self.container_helper["positions"][filename][0],
                                 self.container_helper["positions"][filename][1],
                                 self.container_helper["indices"][filename][0],
                                 self.container_helper["indices"][filename][1],
                                 self.mat_id])
                        elif self.file_type == "SMPL":
                            self.container_helper["SMPL"][filename]["MAT"][self.mat_id] = {
                                "Times": [self.container_helper["positions"][filename][0],
                                          self.container_helper["positions"][filename][1]],
                                "Positions": [self.container_helper["indices"][filename][0],
                                              self.container_helper["indices"][filename][1]],
                                "Object": [box_mat, box_mat_ratio]}
                            self.container_helper["positions"]["MAT SMPL"][filename].append(
                                [self.container_helper["positions"][filename][0],
                                 self.container_helper["positions"][filename][1],
                                 self.container_helper["indices"][filename][0],
                                 self.container_helper["indices"][filename][1],
                                 self.mat_id])
                    #
                    elif var.get() == 5:  # INCL
                        if self.file_type == "STD":
                            if len(self.container_helper["positions"]["INCL STD"][filename]) > 0:
                                self.incl_id = self.container_helper["positions"]["INCL STD"][filename][-1][4]
                        elif self.file_type == "SMPL":
                            if len(self.container_helper["positions"]["INCL SMPL"][filename]) > 0:
                                self.incl_id = self.container_helper["positions"]["INCL SMPL"][filename][-1][4]
                        self.incl_id += 1
                        self.incl_idlist.append(self.incl_id)
                        self.container_helper["limits INCL"][self.file]["ID"].append(self.incl_id)
                        self.container_helper["limits INCL"][self.file]["type"].append("custom")
                        self.container_helper["positions"]["INCL"][filename].append(
                            [round(self.container_helper["positions"][filename][0], 4),
                             round(self.container_helper["positions"][filename][1], 4)])
                        self.container_listboxes[self.file_type][filename]["INCL"][0].insert(
                            tk.END, "INCL" + str(self.incl_id) + " [" + str(
                                self.container_helper["positions"][filename][0]) + "-"
                                    + str(self.container_helper["positions"][filename][1]) + "]")
                        #
                        if ratio_mode == False:
                            box_incl = self.ax.axvspan(
                                self.container_helper["positions"][filename][0],
                                self.container_helper["positions"][filename][1], alpha=0.25, color=self.slate_grey_dark)
                            self.container_helper["limits INCL"][self.file][str(self.incl_id)] = box_incl
                            #
                            self.canvas.draw()
                        else:
                            box_incl = self.ax_ratio.axvspan(
                                self.container_helper["positions"][filename][0],
                                self.container_helper["positions"][filename][1], alpha=0.25, color=self.slate_grey_dark)
                            self.container_helper["limits INCL Ratio"][self.file][str(self.incl_id)] = box_incl
                            #
                            self.canvas_ratio.draw()
                        #
                        self.indices_incl = self.container_helper["indices"][filename]
                        #
                        self.container_files[self.file_type][self.filename_short]["INCL"][self.incl_id] = {}
                        self.container_files[self.file_type][self.filename_short]["INCL"][self.incl_id]["Times"] = [
                            self.container_helper["positions"][filename][0],
                            self.container_helper["positions"][filename][1]]
                        self.container_files[self.file_type][self.filename_short]["INCL"][self.incl_id][
                            "Positions"] = [
                            self.container_helper["indices"][filename][0],
                            self.container_helper["indices"][filename][1]]
                        self.container_files[self.file_type][self.filename_short]["INCL"][self.incl_id][
                            "Box"] = box_incl
                        #
                        if self.file_type == "STD":
                            self.container_helper["STD"][filename]["INCL"][self.incl_id] = {
                                "Times": [self.container_helper["positions"][filename][0],
                                          self.container_helper["positions"][filename][1]],
                                "Positions": [self.container_helper["indices"][filename][0],
                                              self.container_helper["indices"][filename][1]],
                                "Object": box_incl}
                            self.container_helper["positions"]["INCL STD"][filename].append(
                                [self.container_helper["positions"][filename][0],
                                 self.container_helper["positions"][filename][1],
                                 self.container_helper["indices"][filename][0],
                                 self.container_helper["indices"][filename][1],
                                 self.incl_id])
                        elif self.file_type == "SMPL":
                            self.container_helper["SMPL"][filename]["INCL"][self.incl_id] = {
                                "Times": [self.container_helper["positions"][filename][0],
                                          self.container_helper["positions"][filename][1]],
                                "Positions": [self.container_helper["indices"][filename][0],
                                              self.container_helper["indices"][filename][1]],
                                "Object": box_incl}
                            self.container_helper["positions"]["INCL SMPL"][filename].append(
                                [self.container_helper["positions"][filename][0],
                                 self.container_helper["positions"][filename][1],
                                 self.container_helper["indices"][filename][0],
                                 self.container_helper["indices"][filename][1],
                                 self.incl_id])
                    #
                    elif var.get() == 0:
                        pass
            else:
                pass
            #
        else:
            if var.get() == 1:
                if len(self.container_helper["positions"][filename]) == 2 and len(
                        self.container_helper["indices"][filename]) == 2:
                    self.container_helper["positions"][filename].clear()
                    self.container_helper["indices"][filename].clear()
                #
                x_nearest = min(self.times, key=lambda x: abs(x - event.xdata))
                self.container_helper["positions"][filename].append(x_nearest)
                self.container_helper["indices"][filename].append(self.times[self.times == x_nearest].index[0])
                #
                if len(self.container_helper["positions"][filename]) + len(
                        self.container_helper["indices"][filename]) == 4:
                    if var.get() == 1:
                        self.se_id += 1
                        self.se_idlist.append(self.se_id)
                        self.positions_se.append([round(self.container_helper["positions"][filename][0], 4),
                                                  round(self.container_helper["positions"][filename][1], 4)])
                        self.lb_se.insert(tk.END, "Spikes" + str(self.se_id) + " [" + str(
                            self.container_helper["positions"][filename][0]) + "-" +
                                          str(self.container_helper["positions"][filename][1]) + "]" + " [" + str(
                            self.container_helper["indices"][filename][0]) + "-" +
                                          str(self.container_helper["indices"][filename][1]) + "]")
                        box_se = self.ax.axvspan(self.container_helper["positions"][filename][0],
                                                 self.container_helper["positions"][filename][1], alpha=0.25,
                                                 color="#fff6a4")
                        self.limits_se[str(self.se_id)] = box_se
                        self.canvas_se.draw()
                    elif var.get() == 0:
                        pass
            else:
                pass

    def update_variables_initial_values(self):
        str_var_01 = self.language_dict["Set start time"][self.var_language]
        str_var_02 = self.language_dict["Set end time"][self.var_language]
        str_var_03 = self.language_dict["Select IS"][self.var_language]
        str_var_04 = self.language_dict["Select oxide"][self.var_language]
        str_var_05 = self.language_dict["Select element"][self.var_language]
        str_var_06 = self.language_dict["Select mineral"][self.var_language]

        self.container_var["settings"]["Time BG Start"].set(str_var_01)
        self.container_var["settings"]["Time BG End"].set(str_var_02)
        self.container_var["settings"]["Time SIG Start"].set(str_var_01)
        self.container_var["settings"]["Time SIG End"].set(str_var_02)
        self.container_var["ma_setting"]["Time BG Start"].set(str_var_01)
        self.container_var["ma_setting"]["Time BG End"].set(str_var_02)
        self.container_var["ma_setting"]["Time MAT Start"].set(str_var_01)
        self.container_var["ma_setting"]["Time MAT End"].set(str_var_02)

        self.container_var["ma_setting"]["Default IS STD"].set(str_var_03)
        self.container_var["ma_setting"]["Default IS SMPL"].set(str_var_03)
        self.container_var["IS"]["Default STD"].set(str_var_03)
        self.container_var["IS"]["Default SMPL"].set(str_var_03)
        self.container_var["General Settings"]["Default IS MA"].set(str_var_03)
        self.container_var["General Settings"]["Default IS FI"].set(str_var_03)
        self.container_var["General Settings"]["Default IS MI"].set(str_var_03)
        self.container_var["IS"]["Default SMPL"].set(str_var_03)

        self.container_var["ma_setting"]["Oxide"].set(str_var_04)

        self.container_var["ma_setting"]["Element"].set(str_var_05)

        self.container_var["ma_setting"]["Mineral"].set(str_var_06)

        keys = ["fi_setting", "mi_setting"]
        for key_setting in keys:
            self.container_var[key_setting]["Time BG Start"].set(str_var_01)
            self.container_var[key_setting]["Time BG End"].set(str_var_02)
            self.container_var[key_setting]["Time MAT Start"].set(str_var_01)
            self.container_var[key_setting]["Time MAT End"].set(str_var_02)
            self.container_var[key_setting]["Time INCL Start"].set(str_var_01)
            self.container_var[key_setting]["Time INCL End"].set(str_var_02)

            self.container_var[key_setting]["IS MAT Default"].set(str_var_03)
            self.container_var[key_setting]["IS INCL Default"].set(str_var_03)
            self.container_var[key_setting]["Salt Correction"]["Default IS"].set(str_var_03)

            self.container_var[key_setting]["Oxide"].set(str_var_04)
            self.container_var[key_setting]["Oxide Inclusion"].set(str_var_04)

    def do_spike_elimination_all_grubbs(self, filetype, spike_elimination_performed=True):
        if self.pysills_mode == "MA":
            key_setting = "ma_setting"
        elif self.pysills_mode in ["FI", "INCL"]:
            key_setting = "fi_setting"
        elif self.pysills_mode == "MI":
            key_setting = "mi_setting"

        if "Selection" not in self.container_spikes:
            self.container_spikes["Selection"] = {}

        if self.pysills_mode == "MA":
            var_alpha = float(self.container_var[key_setting]["SE Alpha"].get())
            var_threshold = int(self.container_var[key_setting]["SE Threshold"].get())
            if self.container_var["Spike Elimination Method"].get() in ["Grubbs-Test (SILLS)", "Grubbs test"]:
                var_method = 0
            elif self.container_var["Spike Elimination Method"].get() == "Grubbs-Test":
                var_method = 1
            elif self.container_var["Spike Elimination Method"].get() in ["PySILLS Spike Finder", "Whisker analysis"]:
                var_method = 2
            elif self.container_var["Spike Elimination Method"].get() in ["Z-score analysis"]:
                var_method = 3
        elif self.pysills_mode in ["FI", "INCL"]:
            var_alpha = float(self.container_var[key_setting]["SE Alpha"].get())
            var_threshold = int(self.container_var[key_setting]["SE Threshold"].get())
            if self.container_var["Spike Elimination Method"].get() in ["Grubbs-Test (SILLS)", "Grubbs test"]:
                var_method = 0
            elif self.container_var["Spike Elimination Method"].get() == "Grubbs-Test":
                var_method = 1
            elif self.container_var["Spike Elimination Method"].get() in ["PySILLS Spike Finder", "Whisker analysis"]:
                var_method = 2
            elif self.container_var["Spike Elimination Method"].get() in ["Z-score analysis"]:
                var_method = 3
        elif self.pysills_mode == "MI":
            var_alpha = float(self.container_var[key_setting]["SE Alpha"].get())
            var_threshold = int(self.container_var[key_setting]["SE Threshold"].get())
            if self.container_var["Spike Elimination Method"].get() in ["Grubbs-Test (SILLS)", "Grubbs test"]:
                var_method = 0
            elif self.container_var["Spike Elimination Method"].get() == "Grubbs-Test":
                var_method = 1
            elif self.container_var["Spike Elimination Method"].get() in ["PySILLS Spike Finder", "Whisker analysis"]:
                var_method = 2
            elif self.container_var["Spike Elimination Method"].get() in ["Z-score analysis"]:
                var_method = 3

        if filetype == "STD":
            subwindow_progressbar_spike_elimination, prgbar_spk = self.create_progress_bar_spike_elimination()
            n_files = len(self.container_lists["STD"]["Short"])
            n_steps = 0

            for index, file_std in enumerate(self.container_lists["STD"]["Short"]):
                file_isotopes = self.container_lists["Measured Isotopes"][file_std]
                n_steps += len(file_isotopes)

            stepwidth = round(100/n_steps, 4)
            stepwidth_files = round(100/n_files, 4)
            current_step = 0
            self.update_progress(
                parent=subwindow_progressbar_spike_elimination, variable=prgbar_spk, value=current_step)
            self.lbl_prg_spk.configure(text="Spike detection started!", anchor=tk.W)
            for index, file_std in enumerate(self.container_lists["STD"]["Short"]):
                current_step = index*stepwidth_files
                file_long = self.container_lists[filetype]["Long"][index]
                self.lbl_prg_spk.configure(text=file_std)
                if self.container_var[filetype][file_long]["Checkbox"].get() == 1:
                    isotopes_spiked_list = [*self.spikes_isotopes[filetype][file_std]]
                    corrected_isotopes = []
                    not_corrected_isotopes = []
                    self.container_spikes[file_std] = {}

                    if file_std not in self.container_spikes["Selection"]:
                        self.container_spikes["Selection"][file_std] = {}

                    file_isotopes = self.container_lists["Measured Isotopes"][file_std]
                    for isotope in self.container_lists["Measured Isotopes"]["All"]:
                        if isotope in file_isotopes:
                            if bool(self.spikes_isotopes[filetype][file_std]):
                                for isotope_spiked, intervals in self.spikes_isotopes[filetype][file_std].items():

                                    if isotope_spiked not in self.container_spikes["Selection"][file_std]:
                                        self.container_spikes["Selection"][file_std][isotope_spiked] = {}

                                    if isotope in isotopes_spiked_list:
                                        if isotope not in corrected_isotopes:
                                            corrected_isotopes.append(isotope)
                                            spike_intervals = np.array(intervals)
                                            merged_intervals = ES(variable=spike_intervals).merge_times()
                                            for interval in merged_intervals:
                                                dataset_raw = self.container_measurements["RAW"][file_std][isotope][
                                                              interval[0]:interval[1]]
                                                dataset_complete = self.container_measurements["RAW"][file_std][isotope]
                                                dataset_complete_all = self.container_measurements["RAW"][file_std]
                                                var_threshold = int(float(
                                                    self.container_var["Spike Elimination"]["Threshold"][
                                                        isotope].get()))

                                                if var_threshold < 0:
                                                    var_threshold = abs(var_threshold)

                                                self.container_var["Spike Elimination"]["Threshold"][isotope].set(
                                                    var_threshold)

                                                if spike_elimination_performed == True:
                                                    time_start = datetime.datetime.now()
                                                    if var_method == 0:
                                                        data_smoothed, indices_outl = GrubbsTestSILLS(
                                                            raw_data=dataset_raw, alpha=var_alpha,
                                                            threshold=var_threshold, start_index=interval[0],
                                                            dataset_complete=dataset_complete).determine_outlier()
                                                    elif var_method == 1:
                                                        data_smoothed, indices_outl = ES(
                                                            variable=dataset_raw).do_grubbs_test(
                                                            alpha=var_alpha, dataset_complete=dataset_complete,
                                                            threshold=var_threshold)
                                                    elif var_method == 2:
                                                        data_smoothed, indices_outl = OutlierDetection(
                                                            raw_data=dataset_raw, alpha=var_alpha,
                                                            threshold=var_threshold, isotope=isotope,
                                                            dataset_complete=dataset_complete).find_outlier()
                                                    elif var_method == 3:
                                                        data_smoothed, indices_outl = GrubbsTestSILLS(
                                                            raw_data=dataset_raw, alpha=var_alpha,
                                                            threshold=var_threshold, start_index=interval[0],
                                                            dataset_complete=dataset_complete).outlier_elimination_zsore()

                                                    time_end = datetime.datetime.now()
                                                    time_delta = (time_end - time_start)*1000
                                                else:
                                                    data_smoothed = dataset_raw
                                                    indices_outl = []

                                                data_improved = data_smoothed.copy()
                                                self.container_measurements["EDITED"][file_std][isotope][
                                                    "Uncut"] = data_smoothed

                                                self.container_spikes[file_std][isotope] = {
                                                    "Data RAW": self.container_measurements["RAW"][file_std][isotope],
                                                    "Data SMOOTHED": data_smoothed, "Data IMPROVED": data_improved,
                                                    "Indices": indices_outl,
                                                    "Times": self.container_measurements["SELECTED"][file_std]["Time"]}

                                            for var_index in indices_outl:
                                                self.container_spikes["Selection"][file_std][isotope][
                                                    var_index] = data_smoothed[var_index]

                                        else:
                                            pass
                                    else:
                                        if isotope not in not_corrected_isotopes:
                                            not_corrected_isotopes.append(isotope)
                                            self.container_measurements["EDITED"][file_std][isotope]["Uncut"] = \
                                                self.container_measurements["RAW"][file_std][isotope]
                                        else:
                                            pass
                            else:
                                if isotope not in not_corrected_isotopes:
                                    not_corrected_isotopes.append(isotope)
                                    if "Uncut" not in self.container_measurements["EDITED"][file_std][isotope]:
                                        self.container_measurements["EDITED"][file_std][isotope]["Uncut"] = None

                                    if file_std not in self.container_measurements["RAW"]:
                                        self.build_container_measurements(filetype=filetype, filename_short=file_std)

                                    self.container_measurements["EDITED"][file_std][isotope]["Uncut"] = (
                                        self.container_measurements["RAW"][file_std][isotope])

                                    self.container_spikes[file_std][isotope] = {
                                        "Data RAW": self.container_measurements["RAW"][file_std][isotope],
                                        "Data SMOOTHED": self.container_measurements["RAW"][file_std][isotope],
                                        "Data IMPROVED": self.container_measurements["RAW"][file_std][isotope],
                                        "Indices": [],
                                        "Times": self.container_measurements["SELECTED"][file_std]["Time"]}
                                else:
                                    pass

                        current_step += stepwidth
                        self.update_progress(
                            parent=subwindow_progressbar_spike_elimination, variable=prgbar_spk, value=current_step)
                        self.lbl_prg_spk.configure(
                            text=file_std + " : " + isotope + " - " + str(round(current_step, 2)) + " %", anchor=tk.W)

            # Fill container_spike_values
            self.helper_fill_container_spike_values(mode=filetype)

            current_step += stepwidth
            if current_step >= 100:
                self.lbl_prg_spk.configure(text="Spike detection finished!", anchor=tk.W)
                subwindow_progressbar_spike_elimination.destroy()
        elif filetype == "SMPL":
            list_times = {}
            subwindow_progressbar_spike_elimination, prgbar_spk = self.create_progress_bar_spike_elimination()
            n_files = len(self.container_lists["SMPL"]["Short"])
            n_steps = 0

            for index, file_smpl in enumerate(self.container_lists["SMPL"]["Short"]):
                file_isotopes = self.container_lists["Measured Isotopes"][file_smpl]
                n_steps += len(file_isotopes)

            stepwidth = round(100/n_steps, 4)
            stepwidth_files = round(100/n_files, 4)
            current_step = 0
            self.update_progress(
                parent=subwindow_progressbar_spike_elimination, variable=prgbar_spk, value=current_step)
            self.lbl_prg_spk.configure(text="Spike detection started!", anchor=tk.W)

            for index, file_smpl in enumerate(self.container_lists["SMPL"]["Short"]):
                current_step = index*stepwidth_files
                list_times[file_smpl] = []
                filename_long = self.container_lists["SMPL"]["Long"][index]
                self.lbl_prg_spk.configure(text=file_smpl)
                if self.container_var[filetype][filename_long]["Checkbox"].get() == 1:
                    isotopes_spiked_list = [*self.spikes_isotopes[filetype][file_smpl]]
                    corrected_isotopes = []
                    not_corrected_isotopes = []
                    self.container_spikes[file_smpl] = {}
                    if len(isotopes_spiked_list) == 0 and "_copy" in file_smpl:
                        file_smpl_original = file_smpl.replace("_copy", "")
                        isotopes_spiked_list = [*self.spikes_isotopes[filetype][file_smpl_original]]
                    if file_smpl not in self.container_spikes["Selection"]:
                        self.container_spikes["Selection"][file_smpl] = {}

                    if self.file_loaded == False:
                        if self.container_icpms["name"] != None:
                            var_skipheader = self.container_icpms["skipheader"]
                            var_skipfooter = self.container_icpms["skipfooter"]
                            df_data = DE(filename_long=filename_long).get_measurements(
                                delimiter=",", skip_header=var_skipheader, skip_footer=var_skipfooter)
                        else:
                            df_data = DE(filename_long=filename_long).get_measurements(
                                delimiter=",", skip_header=3, skip_footer=1)
                    else:
                        try:
                            df_data = self.container_measurements["Dataframe"][file_smpl]
                        except:
                            if "_copy" in file_smpl:
                                file_smpl_original = file_smpl.replace("_copy", "")
                                file_smpl = file_smpl_original
                                df_data = self.container_measurements["Dataframe"][file_smpl]
                            else:
                                df_data = self.container_measurements["Dataframe"][file_smpl]

                    list_names = list(df_data.columns.values)
                    list_names.pop(0)
                    df_isotopes = list_names

                    for isotope in self.container_lists["Measured Isotopes"]["All"]:
                        if isotope in df_isotopes:
                            if bool(self.spikes_isotopes[filetype][file_smpl]) == False and "_copy" in file_smpl:
                                file_smpl_original = file_smpl.replace("_copy", "")
                                self.spikes_isotopes[filetype][file_smpl] = self.spikes_isotopes[filetype][
                                    file_smpl_original]
                            if bool(self.spikes_isotopes[filetype][file_smpl]):
                                for isotope_spiked, intervals in self.spikes_isotopes[filetype][file_smpl].items():
                                    if isotope_spiked not in self.container_spikes["Selection"][file_smpl]:
                                        self.container_spikes["Selection"][file_smpl][isotope_spiked] = {}

                                    if isotope in isotopes_spiked_list:
                                        if isotope not in corrected_isotopes:
                                            corrected_isotopes.append(isotope)
                                            spike_intervals = np.array(intervals)
                                            merged_intervals = ES(variable=spike_intervals).merge_times()
                                            for interval in merged_intervals:
                                                if (isotope not in self.container_measurements["RAW"][file_smpl] and
                                                        "_copy" in file_smpl):
                                                    file_smpl_original = file_smpl.replace("_copy", "")
                                                    dataset_raw = self.container_measurements["RAW"][
                                                                      file_smpl_original][isotope][
                                                                  interval[0]:interval[1]]
                                                    self.container_measurements["RAW"][file_smpl][
                                                        isotope] = self.container_measurements["RAW"][
                                                        file_smpl_original][isotope]
                                                else:
                                                    dataset_raw = self.container_measurements["RAW"][file_smpl][
                                                                      isotope][interval[0]:interval[1]]

                                                dataset_complete = self.container_measurements["RAW"][file_smpl][
                                                    isotope]
                                                dataset_complete_all = self.container_measurements["RAW"][file_smpl]
                                                var_threshold = int(float(
                                                    self.container_var["Spike Elimination"]["Threshold"][
                                                        isotope].get()))

                                                if var_threshold < 0:
                                                    var_threshold = abs(var_threshold)

                                                self.container_var["Spike Elimination"]["Threshold"][isotope].set(
                                                    var_threshold)

                                                if spike_elimination_performed == True:
                                                    time_start = datetime.datetime.now()
                                                    if var_method == 0:
                                                        data_smoothed, indices_outl = GrubbsTestSILLS(
                                                            raw_data=dataset_raw, alpha=var_alpha,
                                                            threshold=var_threshold, start_index=interval[0],
                                                            dataset_complete=dataset_complete).determine_outlier()
                                                    elif var_method == 1:
                                                        data_smoothed, indices_outl = ES(
                                                            variable=dataset_raw).do_grubbs_test(
                                                            alpha=var_alpha, dataset_complete=dataset_complete,
                                                            threshold=var_threshold)
                                                    elif var_method == 2:
                                                        data_smoothed, indices_outl = OutlierDetection(
                                                            raw_data=dataset_raw, alpha=var_alpha,
                                                            threshold=var_threshold, isotope=isotope,
                                                            dataset_complete=dataset_complete).find_outlier()
                                                    elif var_method == 3:
                                                        data_smoothed, indices_outl = GrubbsTestSILLS(
                                                            raw_data=dataset_raw, alpha=var_alpha,
                                                            threshold=var_threshold, start_index=interval[0],
                                                            dataset_complete=dataset_complete).outlier_elimination_zsore()

                                                    time_end = datetime.datetime.now()
                                                    time_delta = (time_end - time_start)*1000
                                                    list_times[file_smpl].append(time_delta.total_seconds())
                                                else:
                                                    data_smoothed = dataset_raw
                                                    indices_outl = []

                                                data_improved = data_smoothed.copy()
                                                if self.pysills_mode in ["FI", "MI", "INCL"]:
                                                    if self.container_var[key_setting][
                                                        "Spike Elimination Inclusion"].get() == 2:
                                                        length_incl_datasets = len(
                                                            self.container_helper["SMPL"][file_smpl]["INCL"]["Content"])
                                                        if length_incl_datasets > 0:
                                                            for key, items in self.container_helper["SMPL"][file_smpl][
                                                                "INCL"]["Content"].items():
                                                                var_indices = items["Indices"]
                                                                lower_limit = var_indices[0]
                                                                upper_limit = var_indices[1] + 1
                                                                data_raw = self.container_measurements["RAW"][
                                                                    file_smpl][isotope]
                                                                if len(indices_outl) > 0:
                                                                    for index_outl in indices_outl:
                                                                        if (index_outl >= lower_limit
                                                                                and upper_limit >= index_outl):
                                                                            value_raw = data_raw[index_outl]
                                                                            value_smoothed = data_smoothed[index_outl]
                                                                            data_improved[index_outl] = value_raw
                                                                            value_improved = data_improved[index_outl]

                                                if (isotope not in self.container_measurements["EDITED"][file_smpl] and
                                                        "_copy" in file_smpl):
                                                    self.container_measurements["EDITED"][file_smpl][isotope] = {
                                                        "Uncut": None}
                                                    self.container_measurements["EDITED"][file_smpl][isotope][
                                                        "Uncut"] = data_smoothed
                                                else:
                                                    self.container_measurements["EDITED"][file_smpl][isotope][
                                                        "Uncut"] = data_smoothed

                                                if ("Time" not in self.container_measurements["SELECTED"][file_smpl] and
                                                        "_copy" in file_smpl):
                                                    self.container_measurements["SELECTED"][file_smpl][
                                                        "Time"] = self.container_measurements["SELECTED"][
                                                        file_smpl_original]["Time"]

                                                self.container_spikes[file_smpl][isotope] = {
                                                    "Data RAW": self.container_measurements["RAW"][file_smpl][isotope],
                                                    "Data SMOOTHED": data_smoothed, "Data IMPROVED": data_improved,
                                                    "Indices": indices_outl,
                                                    "Times": self.container_measurements["SELECTED"][file_smpl]["Time"]}

                                            for var_index in indices_outl:
                                                self.container_spikes["Selection"][file_smpl][isotope][
                                                    var_index] = data_smoothed[var_index]
                                        else:
                                            pass
                                    else:
                                        if isotope not in not_corrected_isotopes:
                                            not_corrected_isotopes.append(isotope)
                                            self.container_measurements["EDITED"][file_smpl][isotope]["Uncut"] = \
                                                self.container_measurements["RAW"][file_smpl][isotope]
                                        else:
                                            pass
                            else:
                                if isotope not in not_corrected_isotopes:
                                    not_corrected_isotopes.append(isotope)

                                    if "Uncut" not in self.container_measurements["EDITED"][file_smpl][isotope]:
                                        self.container_measurements["EDITED"][file_smpl][isotope]["Uncut"] = None

                                    if file_smpl not in self.container_measurements["RAW"]:
                                        self.build_container_measurements(filetype=filetype, filename_short=file_smpl)

                                    self.container_measurements["EDITED"][file_smpl][isotope]["Uncut"] = \
                                        self.container_measurements["RAW"][file_smpl][isotope]

                                    self.container_spikes[file_smpl][isotope] = {
                                        "Data RAW": self.container_measurements["RAW"][file_smpl][isotope],
                                        "Data SMOOTHED": self.container_measurements["RAW"][file_smpl][isotope],
                                        "Data IMPROVED": self.container_measurements["RAW"][file_smpl][isotope],
                                        "Indices": [],
                                        "Times": self.container_measurements["SELECTED"][file_smpl]["Time"]}
                                else:
                                    pass

                        current_step += stepwidth
                        self.update_progress(
                            parent=subwindow_progressbar_spike_elimination, variable=prgbar_spk, value=current_step)
                        self.lbl_prg_spk.configure(
                            text=file_smpl + " : " + isotope + " - " + str(round(current_step, 2)) + " %", anchor=tk.W)

            self.helper_fill_container_spike_values(mode=filetype)

            current_step += stepwidth
            if current_step >= 100:
                self.lbl_prg_spk.configure(text="Spike detection finished!", anchor=tk.W)
                subwindow_progressbar_spike_elimination.destroy()

    def single_spike_elimination(self, var_filetype, var_filename_short, var_spike_elimination_performed=True):
        if self.pysills_mode == "MA":
            key_setting = "ma_setting"
        elif self.pysills_mode in ["FI", "INCL"]:
            key_setting = "fi_setting"
        elif self.pysills_mode == "MI":
            key_setting = "mi_setting"

        if self.pysills_mode == "MA":
            var_alpha = float(self.container_var[key_setting]["SE Alpha"].get())
            var_threshold = int(self.container_var[key_setting]["SE Threshold"].get())
            if self.container_var["Spike Elimination Method"].get() in ["Grubbs-Test (SILLS)", "Grubbs test"]:
                var_method = 0
            elif self.container_var["Spike Elimination Method"].get() == "Grubbs-Test":
                var_method = 1
            elif self.container_var["Spike Elimination Method"].get() in ["PySILLS Spike Finder", "Whisker analysis"]:
                var_method = 2
        elif self.pysills_mode in ["FI", "INCL"]:
            var_alpha = float(self.container_var[key_setting]["SE Alpha"].get())
            var_threshold = int(self.container_var[key_setting]["SE Threshold"].get())
            if self.container_var["Spike Elimination Method"].get() in ["Grubbs-Test (SILLS)", "Grubbs test"]:
                var_method = 0
            elif self.container_var["Spike Elimination Method"].get() == "Grubbs-Test":
                var_method = 1
            elif self.container_var["Spike Elimination Method"].get() in ["PySILLS Spike Finder", "Whisker analysis"]:
                var_method = 2
        elif self.pysills_mode == "MI":
            var_alpha = float(self.container_var[key_setting]["SE Alpha"].get())
            var_threshold = int(self.container_var[key_setting]["SE Threshold"].get())
            if self.container_var["Spike Elimination Method"].get() in ["Grubbs-Test (SILLS)", "Grubbs test"]:
                var_method = 0
            elif self.container_var["Spike Elimination Method"].get() == "Grubbs-Test":
                var_method = 1
            elif self.container_var["Spike Elimination Method"].get() in ["PySILLS Spike Finder", "Whisker analysis"]:
                var_method = 2

        list_times = {}
        subwindow_progressbar_spike_elimination, prgbar_spk = self.create_progress_bar_spike_elimination()
        n_files = len(self.container_lists[var_filetype]["Short"])
        n_isotopes = len(self.container_lists["Measured Isotopes"]["All"])
        n_steps = n_files*n_isotopes
        stepwidth = round(100/n_steps, 2)
        current_step = 0
        self.update_progress(parent=subwindow_progressbar_spike_elimination, variable=prgbar_spk, value=current_step)
        self.lbl_prg_spk.configure(text="Spike detection started!", anchor=tk.W)

        list_times[var_filename_short] = []
        index_filename = self.container_lists[var_filetype]["Short"].index(var_filename_short)
        filename_long = self.container_lists[var_filetype]["Long"][index_filename]
        self.lbl_prg_spk.configure(text=var_filename_short)
        if self.container_var[var_filetype][filename_long]["Checkbox"].get() == 1:
            isotopes_spiked_list = [*self.spikes_isotopes[var_filetype][var_filename_short]]
            corrected_isotopes = []
            not_corrected_isotopes = []
            self.container_spikes[var_filename_short] = {}
            if len(isotopes_spiked_list) == 0 and "_copy" in var_filename_short:
                var_filename_short_original = var_filename_short.replace("_copy", "")
                isotopes_spiked_list = [*self.spikes_isotopes[var_filetype][var_filename_short_original]]
            if var_filename_short not in self.container_spikes["Selection"]:
                self.container_spikes["Selection"][var_filename_short] = {}

            if self.file_loaded == False:
                if self.container_icpms["name"] != None:
                    var_skipheader = self.container_icpms["skipheader"]
                    var_skipfooter = self.container_icpms["skipfooter"]
                    df_data = DE(filename_long=filename_long).get_measurements(
                        delimiter=",", skip_header=var_skipheader, skip_footer=var_skipfooter)
                else:
                    df_data = DE(filename_long=filename_long).get_measurements(
                        delimiter=",", skip_header=3, skip_footer=1)
            else:
                try:
                    df_data = self.container_measurements["Dataframe"][var_filename_short]
                except:
                    if "_copy" in var_filename_short:
                        var_filename_short_original = var_filename_short.replace("_copy", "")
                        var_filename_short = var_filename_short_original
                        df_data = self.container_measurements["Dataframe"][var_filename_short]
                    else:
                        df_data = self.container_measurements["Dataframe"][var_filename_short]

            list_names = list(df_data.columns.values)
            list_names.pop(0)
            df_isotopes = list_names

            for isotope in df_isotopes:
                if bool(self.spikes_isotopes[var_filetype][var_filename_short]) == False and "_copy" in var_filename_short:
                    var_filename_short_original = var_filename_short.replace("_copy", "")
                    self.spikes_isotopes[var_filetype][var_filename_short] = self.spikes_isotopes[var_filetype][
                        var_filename_short_original]
                if bool(self.spikes_isotopes[var_filetype][var_filename_short]):
                    for isotope_spiked, intervals in self.spikes_isotopes[var_filetype][var_filename_short].items():
                        if isotope_spiked not in self.container_spikes["Selection"][var_filename_short]:
                            self.container_spikes["Selection"][var_filename_short][isotope_spiked] = {}

                        if isotope in isotopes_spiked_list:
                            if isotope not in corrected_isotopes:
                                corrected_isotopes.append(isotope)
                                spike_intervals = np.array(intervals)
                                merged_intervals = ES(variable=spike_intervals).merge_times()
                                for interval in merged_intervals:
                                    if (isotope not in self.container_measurements["RAW"][var_filename_short] and
                                            "_copy" in var_filename_short):
                                        var_filename_short_original = var_filename_short.replace("_copy", "")
                                        dataset_raw = self.container_measurements["RAW"][var_filename_short_original][
                                                          isotope][interval[0]:interval[1]]
                                        self.container_measurements["RAW"][var_filename_short][
                                            isotope] = self.container_measurements["RAW"][var_filename_short_original][
                                            isotope]
                                    else:
                                        dataset_raw = self.container_measurements["RAW"][var_filename_short][isotope][
                                                      interval[0]:interval[1]]

                                    dataset_complete = self.container_measurements["RAW"][var_filename_short][isotope]
                                    dataset_complete_all = self.container_measurements["RAW"][var_filename_short]

                                    if var_spike_elimination_performed == True:
                                        time_start = datetime.datetime.now()
                                        if var_method == 0:
                                            data_smoothed, indices_outl = GrubbsTestSILLS(
                                                raw_data=dataset_raw, alpha=var_alpha, threshold=var_threshold,
                                                start_index=interval[0],
                                                dataset_complete=dataset_complete).determine_outlier()
                                        elif var_method == 1:
                                            data_smoothed, indices_outl = ES(
                                                variable=dataset_raw).do_grubbs_test(
                                                alpha=var_alpha, dataset_complete=dataset_complete,
                                                threshold=var_threshold)
                                        elif var_method == 2:
                                            data_smoothed, indices_outl = OutlierDetection(
                                                raw_data=dataset_raw, alpha=var_alpha, threshold=var_threshold,
                                                isotope=isotope,
                                                dataset_complete=dataset_complete).find_outlier()
                                        time_end = datetime.datetime.now()
                                        time_delta = (time_end - time_start)*1000
                                        list_times[var_filename_short].append(time_delta.total_seconds())
                                    else:
                                        data_smoothed = dataset_raw
                                        indices_outl = []

                                    data_improved = data_smoothed.copy()
                                    if self.pysills_mode in ["FI", "MI", "INCL"]:
                                        if self.container_var[key_setting][
                                            "Spike Elimination Inclusion"].get() == 2 and var_filetype == "SMPL":
                                            length_incl_datasets = len(
                                                self.container_helper["SMPL"][var_filename_short]["INCL"]["Content"])
                                            if length_incl_datasets > 0:
                                                for key, items in self.container_helper["SMPL"][var_filename_short][
                                                    "INCL"]["Content"].items():
                                                    var_indices = items["Indices"]
                                                    lower_limit = var_indices[0]
                                                    upper_limit = var_indices[1] + 1
                                                    data_raw = self.container_measurements["RAW"][var_filename_short][
                                                        isotope]
                                                    if len(indices_outl) > 0:
                                                        for index_outl in indices_outl:
                                                            if (index_outl >= lower_limit
                                                                    and upper_limit >= index_outl):
                                                                value_raw = data_raw[index_outl]
                                                                value_smoothed = data_smoothed[index_outl]
                                                                data_improved[index_outl] = value_raw
                                                                value_improved = data_improved[index_outl]

                                    if (isotope not in self.container_measurements["EDITED"][var_filename_short] and
                                            "_copy" in var_filename_short):
                                        self.container_measurements["EDITED"][var_filename_short][isotope] = {
                                            "Uncut": None}
                                        self.container_measurements["EDITED"][var_filename_short][isotope][
                                            "Uncut"] = data_smoothed
                                    else:
                                        self.container_measurements["EDITED"][var_filename_short][isotope][
                                            "Uncut"] = data_smoothed

                                    if ("Time" not in self.container_measurements["SELECTED"][var_filename_short] and
                                            "_copy" in var_filename_short):
                                        self.container_measurements["SELECTED"][var_filename_short][
                                            "Time"] = self.container_measurements["SELECTED"][
                                            var_filename_short_original]["Time"]

                                    self.container_spikes[var_filename_short][isotope] = {
                                        "Data RAW": self.container_measurements["RAW"][var_filename_short][isotope],
                                        "Data SMOOTHED": data_smoothed, "Data IMPROVED": data_improved,
                                        "Indices": indices_outl,
                                        "Times": self.container_measurements["SELECTED"][var_filename_short]["Time"]}

                                for var_index in indices_outl:
                                    self.container_spikes["Selection"][var_filename_short][isotope][
                                        var_index] = data_smoothed[var_index]
                            else:
                                pass
                        else:
                            if isotope not in not_corrected_isotopes:
                                not_corrected_isotopes.append(isotope)
                                self.container_measurements["EDITED"][var_filename_short][isotope]["Uncut"] = \
                                    self.container_measurements["RAW"][var_filename_short][isotope]
                            else:
                                pass
                else:
                    if isotope not in not_corrected_isotopes:
                        not_corrected_isotopes.append(isotope)

                        if "Uncut" not in self.container_measurements["EDITED"][var_filename_short][isotope]:
                            self.container_measurements["EDITED"][var_filename_short][isotope]["Uncut"] = None

                        if var_filename_short not in self.container_measurements["RAW"]:
                            self.build_container_measurements(filetype=var_filetype, filename_short=var_filename_short)

                        self.container_measurements["EDITED"][var_filename_short][isotope]["Uncut"] = \
                            self.container_measurements["RAW"][var_filename_short][isotope]

                        self.container_spikes[var_filename_short][isotope] = {
                            "Data RAW": self.container_measurements["RAW"][var_filename_short][isotope],
                            "Data SMOOTHED": self.container_measurements["RAW"][var_filename_short][isotope],
                            "Data IMPROVED": self.container_measurements["RAW"][var_filename_short][isotope],
                            "Indices": [],
                            "Times": self.container_measurements["SELECTED"][var_filename_short]["Time"]}
                    else:
                        pass

                current_step += stepwidth
                self.update_progress(
                    parent=subwindow_progressbar_spike_elimination, variable=prgbar_spk, value=current_step)
                self.lbl_prg_spk.configure(
                    text=var_filename_short + " : " + isotope + " - " + str(round(current_step, 1)) + " %", anchor=tk.W)

    def ma_export_calculation_report(self):
        header = ["filename", "ID"]
        n_decimals_concentration = 5
        n_decimals_intensity = 5
        n_decimals_sensitivity = 5
        #
        # Concentration
        report_concentration = {}
        report_concentration["Total STD"] = {}
        report_concentration["Total SMPL"] = {}
        # 1-Sigma-Concentration
        report_concentration_sigma = {}
        report_concentration_sigma["Total STD"] = {}
        report_concentration_sigma["Total SMPL"] = {}
        # Concentration Ratio
        report_concentration_ratio = {}
        report_concentration_ratio["Total STD"] = {}
        report_concentration_ratio["Total SMPL"] = {}
        # Limit of Blank
        report_lob = {}
        report_lob["Total STD"] = {}
        report_lob["Total SMPL"] = {}
        # Limit of Detection
        report_lod = {}
        report_lod["Total STD"] = {}
        report_lod["Total SMPL"] = {}
        # Limit of Quantification
        report_loq = {}
        report_loq["Total STD"] = {}
        report_loq["Total SMPL"] = {}
        # Intensity
        report_intensity = {}
        report_intensity["Total STD"] = {}
        report_intensity["Total SMPL"] = {}
        # 1-Sigma-Intensity
        report_intensity_sigma = {}
        report_intensity_sigma["Total STD"] = {}
        report_intensity_sigma["Total SMPL"] = {}
        # Intensity Ratio
        report_intensity_ratio = {}
        report_intensity_ratio["Total STD"] = {}
        report_intensity_ratio["Total SMPL"] = {}
        # Analytical Sensitivity
        report_analytical_sensitivity = {}
        report_analytical_sensitivity["Total STD"] = {}
        report_analytical_sensitivity["Total SMPL"] = {}
        # Normalized Sensitivity
        report_normalized_sensitivity = {}
        report_normalized_sensitivity["Total STD"] = {}
        report_normalized_sensitivity["Total SMPL"] = {}
        # Relative Sensitivity Factor
        report_rsf = {}
        report_rsf["Total STD"] = {}
        report_rsf["Total SMPL"] = {}

        for var_filetype in ["STD", "SMPL"]:
            var_key = "Total " + str(var_filetype)

            report_concentration[var_filetype] = {}
            report_concentration[var_key] = {}
            report_concentration[var_key]["filename"] = "All Files"

            report_concentration_sigma[var_filetype] = {}
            report_concentration_sigma[var_key] = {}
            report_concentration_sigma[var_key]["filename"] = "All Files"

            report_concentration_ratio[var_filetype] = {}
            report_concentration_ratio[var_key] = {}
            report_concentration_ratio[var_key]["filename"] = "All Files"

            report_lob[var_filetype] = {}
            report_lob[var_key] = {}
            report_lob[var_key]["filename"] = "All Files"

            report_lod[var_filetype] = {}
            report_lod[var_key] = {}
            report_lod[var_key]["filename"] = "All Files"

            report_loq[var_filetype] = {}
            report_loq[var_key] = {}
            report_loq[var_key]["filename"] = "All Files"

            report_intensity[var_filetype] = {}
            report_intensity[var_key] = {}
            report_intensity[var_key]["filename"] = "All Files"

            report_intensity_sigma[var_filetype] = {}
            report_intensity_sigma[var_key] = {}
            report_intensity_sigma[var_key]["filename"] = "All Files"

            report_intensity_ratio[var_filetype] = {}
            report_intensity_ratio[var_key] = {}
            report_intensity_ratio[var_key]["filename"] = "All Files"

            report_analytical_sensitivity[var_filetype] = {}
            report_analytical_sensitivity[var_key] = {}
            report_analytical_sensitivity[var_key]["filename"] = "All Files"

            report_normalized_sensitivity[var_filetype] = {}
            report_normalized_sensitivity[var_key] = {}
            report_normalized_sensitivity[var_key]["filename"] = "All Files"

            report_rsf[var_filetype] = {}
            report_rsf[var_key] = {}
            report_rsf[var_key]["filename"] = "All Files"

            for var_datatype in ["SMOOTHED", "RAW"]:
                report_concentration[var_filetype][var_datatype] = {}
                report_concentration_sigma[var_filetype][var_datatype] = {}
                report_concentration_ratio[var_filetype][var_datatype] = {}
                report_lob[var_filetype][var_datatype] = {}
                report_lod[var_filetype][var_datatype] = {}
                report_loq[var_filetype][var_datatype] = {}

                report_intensity[var_filetype][var_datatype] = {}
                report_intensity_sigma[var_filetype][var_datatype] = {}
                report_intensity_ratio[var_filetype][var_datatype] = {}

                report_analytical_sensitivity[var_filetype][var_datatype] = {}
                report_normalized_sensitivity[var_filetype][var_datatype] = {}
                report_rsf[var_filetype][var_datatype] = {}

                for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                    file_long = self.container_lists[var_filetype]["Long"][index]
                    ## Compositional Results
                    report_concentration[var_filetype][var_datatype][file_short] = {}
                    report_concentration[var_filetype][var_datatype][file_short]["filename"] = file_short
                    report_concentration_sigma[var_filetype][var_datatype][file_short] = {}
                    report_concentration_sigma[var_filetype][var_datatype][file_short]["filename"] = file_short
                    report_concentration_ratio[var_filetype][var_datatype][file_short] = {}
                    report_concentration_ratio[var_filetype][var_datatype][file_short]["filename"] = file_short
                    report_lob[var_filetype][var_datatype][file_short] = {}
                    report_lob[var_filetype][var_datatype][file_short]["filename"] = file_short
                    report_lod[var_filetype][var_datatype][file_short] = {}
                    report_lod[var_filetype][var_datatype][file_short]["filename"] = file_short
                    report_loq[var_filetype][var_datatype][file_short] = {}
                    report_loq[var_filetype][var_datatype][file_short]["filename"] = file_short
                    if var_filetype == "STD":
                        report_concentration[var_filetype][var_datatype][file_short]["ID"] = "---"
                        report_concentration_ratio[var_filetype][var_datatype][file_short]["ID"] = "---"
                        report_lob[var_filetype][var_datatype][file_short]["ID"] = "---"
                        report_lod[var_filetype][var_datatype][file_short]["ID"] = "---"
                        report_loq[var_filetype][var_datatype][file_short]["ID"] = "---"
                    else:
                        report_concentration[var_filetype][var_datatype][file_short]["ID"] = self.container_var[
                            var_filetype][file_long]["ID"].get()
                        report_concentration_ratio[var_filetype][var_datatype][file_short]["ID"] = self.container_var[
                            var_filetype][file_long]["ID"].get()
                        report_lob[var_filetype][var_datatype][file_short]["ID"] = self.container_var[var_filetype][
                            file_long]["ID"].get()
                        report_lod[var_filetype][var_datatype][file_short]["ID"] = self.container_var[var_filetype][
                            file_long]["ID"].get()
                        report_loq[var_filetype][var_datatype][file_short]["ID"] = self.container_var[var_filetype][
                            file_long]["ID"].get()

                    ## Intensity Results
                    report_intensity[var_filetype][var_datatype][file_short] = {}
                    report_intensity[var_filetype][var_datatype][file_short]["filename"] = file_short
                    report_intensity_sigma[var_filetype][var_datatype][file_short] = {}
                    report_intensity_sigma[var_filetype][var_datatype][file_short]["filename"] = file_short
                    report_intensity_ratio[var_filetype][var_datatype][file_short] = {}
                    report_intensity_ratio[var_filetype][var_datatype][file_short]["filename"] = file_short
                    if var_filetype == "STD":
                        report_intensity[var_filetype][var_datatype][file_short]["ID"] = "---"
                        report_intensity_ratio[var_filetype][var_datatype][file_short]["ID"] = "---"
                    else:
                        report_intensity[var_filetype][var_datatype][file_short]["ID"] = self.container_var[
                            var_filetype][file_long]["ID"].get()
                        report_intensity_ratio[var_filetype][var_datatype][file_short]["ID"] = self.container_var[
                            var_filetype][file_long]["ID"].get()
                    ## Sensitivity Results
                    report_analytical_sensitivity[var_filetype][var_datatype][file_short] = {}
                    report_analytical_sensitivity[var_filetype][var_datatype][file_short]["filename"] = file_short
                    report_normalized_sensitivity[var_filetype][var_datatype][file_short] = {}
                    report_normalized_sensitivity[var_filetype][var_datatype][file_short]["filename"] = file_short
                    report_rsf[var_filetype][var_datatype][file_short] = {}
                    report_rsf[var_filetype][var_datatype][file_short]["filename"] = file_short
                    if var_filetype == "STD":
                        report_analytical_sensitivity[var_filetype][var_datatype][file_short]["ID"] = "---"
                        report_normalized_sensitivity[var_filetype][var_datatype][file_short]["ID"] = "---"
                        report_rsf[var_filetype][var_datatype][file_short]["ID"] = "---"
                    else:
                        report_analytical_sensitivity[var_filetype][var_datatype][file_short][
                            "ID"] = self.container_var[var_filetype][file_long]["ID"].get()
                        report_normalized_sensitivity[var_filetype][var_datatype][file_short][
                            "ID"] = self.container_var[var_filetype][file_long]["ID"].get()
                        report_rsf[var_filetype][var_datatype][file_short]["ID"] = self.container_var[var_filetype][
                            file_long]["ID"].get()

                    for isotope in self.container_lists["ISOTOPES"]:
                        ## Compositional Results
                        # Concentration
                        if var_filetype == "SMPL":
                            value_i = self.container_concentration[var_filetype][var_datatype][file_short]["MAT"][
                                isotope]
                            value_lod_i = self.container_lod[var_filetype][var_datatype][file_short]["MAT"][isotope]
                        else:
                            var_srm_i = self.container_var["SRM"][isotope].get()
                            var_srm_file = self.container_var["STD"][file_long]["SRM"].get()

                            if var_srm_i == var_srm_file:
                                value_i = self.container_concentration[var_filetype][var_datatype][file_short]["MAT"][
                                    isotope]
                                value_lod_i = self.container_lod[var_filetype][var_datatype][file_short]["MAT"][isotope]
                            else:
                                value_i = None

                        if value_i != None:
                            if value_i >= value_lod_i:
                                report_concentration[var_filetype][var_datatype][file_short][isotope] = round(
                                    value_i, n_decimals_concentration)
                            else:
                                report_concentration[var_filetype][var_datatype][file_short][isotope] = round(
                                    -value_lod_i, n_decimals_concentration)
                        else:
                            report_concentration[var_filetype][var_datatype][file_short][isotope] = "---"

                        # 1-Sigma-Concentration
                        value_i = self.container_concentration[var_filetype][var_datatype][file_short]["1 SIGMA MAT"][
                            isotope]
                        report_concentration_sigma[var_filetype][var_datatype][file_short][isotope] = round(
                            value_i, n_decimals_concentration)

                        # Concentration Ratio
                        value_i = self.container_concentration_ratio[var_filetype][var_datatype][file_short]["MAT"][
                            isotope]
                        try:
                            report_concentration_ratio[var_filetype][var_datatype][file_short][
                                isotope] = "{:0.5e}".format(value_i)
                        except:
                            report_concentration_ratio[var_filetype][var_datatype][file_short][isotope] = "---"

                        # Limit of Blank
                        value_i = self.container_lob[var_filetype][var_datatype][file_short]["MAT"][isotope]

                        if value_i != None:
                            report_lob[var_filetype][var_datatype][file_short][isotope] = round(
                                value_i, n_decimals_concentration)
                        else:
                            report_lob[var_filetype][var_datatype][file_short][isotope] = "---"

                        # Limit of Detection
                        value_i = self.container_lod[var_filetype][var_datatype][file_short]["MAT"][isotope]

                        if value_i != None:
                            report_lod[var_filetype][var_datatype][file_short][isotope] = round(
                                value_i, n_decimals_concentration)
                        else:
                            report_lod[var_filetype][var_datatype][file_short][isotope] = "---"

                        # Limit of Quantification
                        value_i = self.container_loq[var_filetype][var_datatype][file_short]["MAT"][isotope]

                        if value_i != None:
                            report_loq[var_filetype][var_datatype][file_short][isotope] = round(
                                value_i, n_decimals_concentration)
                        else:
                            report_loq[var_filetype][var_datatype][file_short][isotope] = "---"

                        ## Intensity Results
                        # Intensity
                        if var_filetype == "SMPL":
                            value_i = self.container_intensity_corrected[var_filetype][var_datatype][file_short]["MAT"][
                                isotope]
                        else:
                            var_srm_i = self.container_var["SRM"][isotope].get()
                            var_srm_file = self.container_var["STD"][file_long]["SRM"].get()
                            if var_srm_i == var_srm_file:
                                value_i = self.container_intensity_corrected[var_filetype][var_datatype][file_short][
                                    "MAT"][isotope]
                            else:
                                value_i = None
                        if value_i != None:
                            report_intensity[var_filetype][var_datatype][file_short][isotope] = round(
                                value_i, n_decimals_intensity)
                        else:
                            report_intensity[var_filetype][var_datatype][file_short][isotope] = "---"

                        # 1-Sigma-Intensity
                        value_i = self.container_intensity[var_filetype][var_datatype][file_short]["1 SIGMA MAT"][
                            isotope]
                        report_intensity_sigma[var_filetype][var_datatype][file_short][isotope] = round(
                            value_i, n_decimals_intensity)

                        # Intensity Ratio
                        value_i = self.container_intensity_ratio[var_filetype][var_datatype][file_short]["MAT"][isotope]
                        try:
                            report_intensity_ratio[var_filetype][var_datatype][file_short][isotope] = "{:0.5e}".format(
                                value_i)
                        except:
                            report_intensity_ratio[var_filetype][var_datatype][file_short][isotope] = "---"

                        ## Sensitivity Results
                        # Analytical Sensitivity
                        value_i = self.container_analytical_sensitivity[var_filetype][var_datatype][file_short]["MAT"][
                            isotope]
                        try:
                            report_analytical_sensitivity[var_filetype][var_datatype][file_short][isotope] = round(
                                value_i, n_decimals_sensitivity)
                        except:
                            report_analytical_sensitivity[var_filetype][var_datatype][file_short][isotope] = "---"

                        # Normalized Sensitivity
                        if var_filetype == "SMPL":
                            value_i = self.container_normalized_sensitivity[var_filetype][var_datatype][file_short][
                                "MAT"][isotope]
                        else:
                            var_srm_i = self.container_var["SRM"][isotope].get()
                            var_srm_file = self.container_var["STD"][file_long]["SRM"].get()
                            if var_srm_i == var_srm_file:
                                value_i = self.container_normalized_sensitivity[var_filetype][var_datatype][file_short][
                                    "MAT"][isotope]
                            else:
                                value_i = None
                        if value_i != None:
                            report_normalized_sensitivity[var_filetype][var_datatype][file_short][isotope] = round(
                                value_i, n_decimals_sensitivity)
                        else:
                            report_normalized_sensitivity[var_filetype][var_datatype][file_short][isotope] = "---"

                        # Relative Sensitivity Factor
                        value_i = self.container_rsf[var_filetype][var_datatype][file_short]["MAT"][isotope]
                        try:
                            report_rsf[var_filetype][var_datatype][file_short][isotope] = round(
                                value_i, n_decimals_sensitivity)
                        except:
                            report_rsf[var_filetype][var_datatype][file_short][isotope] = "---"

        for isotope in self.container_lists["ISOTOPES"]:
            header.append(isotope)

        var_file_extension_pre = self.container_var["General Settings"]["File type"].get()
        if var_file_extension_pre == "*.csv":
            var_file_extension = ".csv"
        elif var_file_extension_pre == "*.txt":
            var_file_extension = ".txt"

        var_delimiter_pre = self.container_var["General Settings"]["Delimiter"].get()
        if var_delimiter_pre == "comma":
            var_delimiter = ","
        elif var_delimiter_pre == "semicolon":
            var_delimiter = ";"

        if self.rb_report.get() == 0:  # All in one
            self.ma_export_report_0(
                report_concentration, report_concentration_sigma, report_concentration_ratio, report_lob, report_lod,
                report_loq,
                report_intensity, report_intensity_sigma, report_intensity_ratio, report_analytical_sensitivity,
                report_normalized_sensitivity, report_rsf, var_file_extension, var_delimiter, header)
        elif self.rb_report.get() == 1:  # STD vs. SMPL
            self.ma_export_report_1(
                report_concentration, report_concentration_sigma, report_concentration_ratio, report_lob, report_lod,
                report_loq,
                report_intensity, report_intensity_sigma, report_intensity_ratio, report_analytical_sensitivity,
                report_normalized_sensitivity, report_rsf, var_file_extension, var_delimiter, header)
        elif self.rb_report.get() == 2:  # RAW vs. SMOOTHED
            self.ma_export_report_2(
                report_concentration, report_concentration_sigma, report_concentration_ratio, report_lob, report_lod,
                report_loq,
                report_intensity, report_intensity_sigma, report_intensity_ratio, report_analytical_sensitivity,
                report_normalized_sensitivity, report_rsf, var_file_extension, var_delimiter, header)

    def ma_export_report_0(
            self, report_concentration, report_concentration_sigma, report_concentration_ratio, report_lob, report_lod,
            report_loq, report_intensity, report_intensity_sigma, report_intensity_ratio, report_analytical_sensitivity,
            report_normalized_sensitivity, report_rsf, var_file_extension, var_delimiter, header):
        export_file = filedialog.asksaveasfile(mode="w", defaultextension=var_file_extension)
        filename = export_file.name

        with open(filename, "w", newline="") as report_file:
            writer = csv.DictWriter(report_file, fieldnames=header, delimiter=var_delimiter)
            report_file.write("CALCULATION REPORT\n")
            report_file.write("\n")
            report_file.write("AUTHOR:;" + str(self.container_var["ma_setting"]["Author"].get()) + "\n")
            report_file.write("SOURCE ID:;" + str(self.container_var["ma_setting"]["Source ID"].get()) + "\n")
            report_file.write("\n")
            for var_datatype in ["SMOOTHED", "RAW"]:
                report_file.write("DATA TYPE:;" + str(var_datatype) + str(" DATA") + "\n")
                report_file.write("\n")
                for var_filetype in ["SMPL", "STD"]:
                    if var_filetype == "SMPL":
                        report_file.write("SAMPLE FILES\n")
                    elif var_filetype == "STD":
                        report_file.write("STANDARD FILES\n")
                    var_key = "Total " + str(var_filetype)
                    report_file.write("\n")

                    ## COMPOSITIONAL ANALYSIS
                    report_file.write("COMPOSITIONAL ANALYSIS\n")

                    report_file.write("Concentration\n")  # Concentration
                    report_file.write("(ppm)\n")
                    writer.writeheader()
                    for file_short in self.container_lists[var_filetype]["Short"]:
                        writer.writerow(report_concentration[var_filetype][var_datatype][file_short])
                    report_file.write("\n")

                    report_file.write("1-Sigma-Concentration\n")  # 1-Sigma-Concentration
                    report_file.write("(ppm)\n")
                    writer.writeheader()
                    for file_short in self.container_lists[var_filetype]["Short"]:
                        writer.writerow(report_concentration_sigma[var_filetype][var_datatype][file_short])
                    report_file.write("\n")

                    if var_filetype == "SMPL":
                        report_file.write("Concentration Ratio\n")  # Concentration Ratio
                        report_file.write("(1)\n")
                        writer.writeheader()
                        for file_short in self.container_lists[var_filetype]["Short"]:
                            writer.writerow(report_concentration_ratio[var_filetype][var_datatype][file_short])
                        report_file.write("\n")

                        report_file.write("Limit of Blank\n")  # Limit of Blank
                        report_file.write("(ppm)\n")
                        writer.writeheader()
                        for file_short in self.container_lists[var_filetype]["Short"]:
                            writer.writerow(report_lob[var_filetype][var_datatype][file_short])
                        report_file.write("\n")

                        report_file.write("Limit of Detection\n")  # Limit of Detection
                        report_file.write("(ppm)\n")
                        writer.writeheader()
                        for file_short in self.container_lists[var_filetype]["Short"]:
                            writer.writerow(report_lod[var_filetype][var_datatype][file_short])
                        report_file.write("\n")

                        report_file.write("Limit of Quantification\n")  # Limit of Quantification
                        report_file.write("(ppm)\n")
                        writer.writeheader()
                        for file_short in self.container_lists[var_filetype]["Short"]:
                            writer.writerow(report_loq[var_filetype][var_datatype][file_short])
                        report_file.write("\n")

                    report_file.write("INTENSITY ANALYSIS\n")

                    report_file.write("Intensity (Sample)\n")  # Intensity
                    report_file.write("(cps)\n")
                    writer.writeheader()
                    for file_short in self.container_lists[var_filetype]["Short"]:
                        writer.writerow(report_intensity[var_filetype][var_datatype][file_short])
                    report_file.write("\n")

                    report_file.write("1-Sigma-Intensity (Sample)\n")  # 1-Sigma-Intensity
                    report_file.write("(cps)\n")
                    writer.writeheader()
                    for file_short in self.container_lists[var_filetype]["Short"]:
                        writer.writerow(report_intensity_sigma[var_filetype][var_datatype][file_short])
                    report_file.write("\n")

                    if var_filetype == "SMPL":
                        report_file.write("Intensity Ratio (Sample)\n")  # Intensity Ratio
                        report_file.write("(1)\n")
                        writer.writeheader()
                        for file_short in self.container_lists[var_filetype]["Short"]:
                            writer.writerow(report_intensity_ratio[var_filetype][var_datatype][file_short])
                        report_file.write("\n")

                    report_file.write("SENSITIVITY ANALYSIS\n")

                    report_file.write("Analytical Sensitivity\n")  # Analytical Sensitivity
                    report_file.write("(1)\n")
                    writer.writeheader()
                    for file_short in self.container_lists[var_filetype]["Short"]:
                        writer.writerow(report_analytical_sensitivity[var_filetype][var_datatype][file_short])
                    report_file.write("\n")

                    report_file.write("Normalized Sensitivity\n")  # Normalized Sensitivity
                    report_file.write("(cps)/(ppm)\n")
                    writer.writeheader()
                    for file_short in self.container_lists[var_filetype]["Short"]:
                        writer.writerow(report_normalized_sensitivity[var_filetype][var_datatype][file_short])
                    report_file.write("\n")

                    if var_filetype == "SMPL":
                        report_file.write("Relative Sensitivity Factor\n")  # Relative Sensitivity Factor
                        report_file.write("(1)\n")
                        writer.writeheader()
                        for file_short in self.container_lists[var_filetype]["Short"]:
                            writer.writerow(report_rsf[var_filetype][var_datatype][file_short])
                        report_file.write("\n")

    def ma_export_report_1(
            self, report_concentration, report_concentration_sigma, report_concentration_ratio, report_lob, report_lod,
            report_loq, report_intensity, report_intensity_sigma, report_intensity_ratio, report_analytical_sensitivity,
            report_normalized_sensitivity, report_rsf, var_file_extension, var_delimiter, header):
        export_file = filedialog.asksaveasfile(mode="w", defaultextension=var_file_extension)
        filename_base = export_file.name
        filename_base_parts = filename_base.split(".")
        filename_std = filename_base_parts[0] + "_std" + var_file_extension
        filename_smpl = filename_base_parts[0] + "_smpl" + var_file_extension

        # Standard Files
        with open(filename_std, "w", newline="") as report_file_std:
            writer = csv.DictWriter(report_file_std, fieldnames=header, delimiter=var_delimiter)
            report_file_std.write("CALCULATION REPORT\n")
            report_file_std.write("\n")
            report_file_std.write("AUTHOR:;" + str(self.container_var["ma_setting"]["Author"].get()) + "\n")
            report_file_std.write("SOURCE ID:;" + str(self.container_var["ma_setting"]["Source ID"].get()) + "\n")
            report_file_std.write("\n")
            for var_datatype in ["SMOOTHED", "RAW"]:
                report_file_std.write("DATA TYPE:;" + str(var_datatype) + str(" DATA") + "\n")
                report_file_std.write("\n")
                for var_filetype in ["STD"]:
                    if var_filetype == "SMPL":
                        report_file_std.write("SAMPLE FILES\n")
                    elif var_filetype == "STD":
                        report_file_std.write("STANDARD FILES\n")
                    var_key = "Total " + str(var_filetype)
                    report_file_std.write("\n")

                    ## COMPOSITIONAL ANALYSIS
                    report_file_std.write("COMPOSITIONAL ANALYSIS\n")

                    report_file_std.write("Concentration\n")  # Concentration
                    report_file_std.write("(ppm)\n")
                    writer.writeheader()
                    for file_short in self.container_lists[var_filetype]["Short"]:
                        writer.writerow(report_concentration[var_filetype][var_datatype][file_short])
                    report_file_std.write("\n")

                    report_file_std.write("1-Sigma-Concentration\n")  # 1-Sigma-Concentration
                    report_file_std.write("(ppm)\n")
                    writer.writeheader()
                    for file_short in self.container_lists[var_filetype]["Short"]:
                        writer.writerow(report_concentration_sigma[var_filetype][var_datatype][file_short])
                    report_file_std.write("\n")

                    if var_filetype == "SMPL":
                        report_file_std.write("Concentration Ratio\n")  # Concentration Ratio
                        report_file_std.write("(1)\n")
                        writer.writeheader()
                        for file_short in self.container_lists[var_filetype]["Short"]:
                            writer.writerow(report_concentration_ratio[var_filetype][var_datatype][file_short])
                        report_file_std.write("\n")

                        report_file_std.write("Limit of Blank\n")  # Limit of Blank
                        report_file_std.write("(ppm)\n")
                        writer.writeheader()
                        for file_short in self.container_lists[var_filetype]["Short"]:
                            writer.writerow(report_lob[var_filetype][var_datatype][file_short])
                        report_file_std.write("\n")

                        report_file_std.write("Limit of Detection\n")  # Limit of Detection
                        report_file_std.write("(ppm)\n")
                        writer.writeheader()
                        for file_short in self.container_lists[var_filetype]["Short"]:
                            writer.writerow(report_lod[var_filetype][var_datatype][file_short])
                        report_file_std.write("\n")

                        report_file_std.write("Limit of Quantification\n")  # Limit of Quantification
                        report_file_std.write("(ppm)\n")
                        writer.writeheader()
                        for file_short in self.container_lists[var_filetype]["Short"]:
                            writer.writerow(report_loq[var_filetype][var_datatype][file_short])
                        report_file_std.write("\n")

                    report_file_std.write("INTENSITY ANALYSIS\n")

                    report_file_std.write("Intensity (Sample)\n")  # Intensity
                    report_file_std.write("(cps)\n")
                    writer.writeheader()
                    for file_short in self.container_lists[var_filetype]["Short"]:
                        writer.writerow(report_intensity[var_filetype][var_datatype][file_short])
                    report_file_std.write("\n")

                    report_file_std.write("1-Sigma-Intensity (Sample)\n")  # 1-Sigma-Intensity
                    report_file_std.write("(cps)\n")
                    writer.writeheader()
                    for file_short in self.container_lists[var_filetype]["Short"]:
                        writer.writerow(report_intensity_sigma[var_filetype][var_datatype][file_short])
                    report_file_std.write("\n")

                    if var_filetype == "SMPL":
                        report_file_std.write("Intensity Ratio (Sample)\n")  # Intensity Ratio
                        report_file_std.write("(1)\n")
                        writer.writeheader()
                        for file_short in self.container_lists[var_filetype]["Short"]:
                            writer.writerow(report_intensity_ratio[var_filetype][var_datatype][file_short])
                        report_file_std.write("\n")

                    report_file_std.write("SENSITIVITY ANALYSIS\n")

                    report_file_std.write("Analytical Sensitivity\n")  # Analytical Sensitivity
                    report_file_std.write("(1)\n")
                    writer.writeheader()
                    for file_short in self.container_lists[var_filetype]["Short"]:
                        writer.writerow(report_analytical_sensitivity[var_filetype][var_datatype][file_short])
                    report_file_std.write("\n")

                    report_file_std.write("Normalized Sensitivity\n")  # Normalized Sensitivity
                    report_file_std.write("(cps)/(ppm)\n")
                    writer.writeheader()
                    for file_short in self.container_lists[var_filetype]["Short"]:
                        writer.writerow(report_normalized_sensitivity[var_filetype][var_datatype][file_short])
                    report_file_std.write("\n")

                    if var_filetype == "SMPL":
                        report_file_std.write("Relative Sensitivity Factor\n")  # Relative Sensitivity Factor
                        report_file_std.write("(1)\n")
                        writer.writeheader()
                        for file_short in self.container_lists[var_filetype]["Short"]:
                            writer.writerow(report_rsf[var_filetype][var_datatype][file_short])
                        report_file_std.write("\n")
        # Sample Files
        with open(filename_smpl, "w", newline="") as report_file_smpl:
            writer = csv.DictWriter(report_file_smpl, fieldnames=header, delimiter=var_delimiter)
            report_file_smpl.write("CALCULATION REPORT\n")
            report_file_smpl.write("\n")
            report_file_smpl.write("AUTHOR:;" + str(self.container_var["ma_setting"]["Author"].get()) + "\n")
            report_file_smpl.write("SOURCE ID:;" + str(self.container_var["ma_setting"]["Source ID"].get()) + "\n")
            report_file_smpl.write("\n")
            for var_datatype in ["SMOOTHED", "RAW"]:
                report_file_smpl.write("DATA TYPE:;" + str(var_datatype) + str(" DATA") + "\n")
                report_file_smpl.write("\n")
                for var_filetype in ["SMPL"]:
                    if var_filetype == "SMPL":
                        report_file_smpl.write("SAMPLE FILES\n")
                    elif var_filetype == "STD":
                        report_file_smpl.write("STANDARD FILES\n")
                    var_key = "Total " + str(var_filetype)
                    report_file_smpl.write("\n")

                    ## COMPOSITIONAL ANALYSIS
                    report_file_smpl.write("COMPOSITIONAL ANALYSIS\n")

                    report_file_smpl.write("Concentration\n")  # Concentration
                    report_file_smpl.write("(ppm)\n")
                    writer.writeheader()
                    for file_short in self.container_lists[var_filetype]["Short"]:
                        writer.writerow(report_concentration[var_filetype][var_datatype][file_short])
                    report_file_smpl.write("\n")

                    report_file_smpl.write("1-SigmaConcentration\n")  # 1-Sigma-Concentration
                    report_file_smpl.write("(ppm)\n")
                    writer.writeheader()
                    for file_short in self.container_lists[var_filetype]["Short"]:
                        writer.writerow(report_intensity_sigma[var_filetype][var_datatype][file_short])
                    report_file_smpl.write("\n")

                    if var_filetype == "SMPL":
                        report_file_smpl.write("Concentration Ratio\n")  # Concentration Ratio
                        report_file_smpl.write("(1)\n")
                        writer.writeheader()
                        for file_short in self.container_lists[var_filetype]["Short"]:
                            writer.writerow(report_concentration_ratio[var_filetype][var_datatype][file_short])
                        report_file_smpl.write("\n")

                        report_file_smpl.write("Limit of Blank\n")  # Limit of Blank
                        report_file_smpl.write("(ppm)\n")
                        writer.writeheader()
                        for file_short in self.container_lists[var_filetype]["Short"]:
                            writer.writerow(report_lob[var_filetype][var_datatype][file_short])
                        report_file_smpl.write("\n")

                        report_file_smpl.write("Limit of Detection\n")  # Limit of Detection
                        report_file_smpl.write("(ppm)\n")
                        writer.writeheader()
                        for file_short in self.container_lists[var_filetype]["Short"]:
                            writer.writerow(report_lod[var_filetype][var_datatype][file_short])
                        report_file_smpl.write("\n")

                        report_file_smpl.write("Limit of Quantification\n")  # Limit of Quantification
                        report_file_smpl.write("(ppm)\n")
                        writer.writeheader()
                        for file_short in self.container_lists[var_filetype]["Short"]:
                            writer.writerow(report_loq[var_filetype][var_datatype][file_short])
                        report_file_smpl.write("\n")

                    report_file_smpl.write("INTENSITY ANALYSIS\n")

                    report_file_smpl.write("Intensity (Sample)\n")  # Intensity
                    report_file_smpl.write("(cps)\n")
                    writer.writeheader()
                    for file_short in self.container_lists[var_filetype]["Short"]:
                        writer.writerow(report_intensity[var_filetype][var_datatype][file_short])
                    report_file_smpl.write("\n")

                    report_file_smpl.write("1-Sigma-Intensity (Sample)\n")  # 1-Sigma-Intensity
                    report_file_smpl.write("(cps)\n")
                    writer.writeheader()
                    for file_short in self.container_lists[var_filetype]["Short"]:
                        writer.writerow(report_intensity_sigma[var_filetype][var_datatype][file_short])
                    report_file_smpl.write("\n")

                    report_file_smpl.write("1-Sigma-Intensity (Sample)\n")  # 1-Sigma-Intensity
                    report_file_smpl.write("(cps)\n")
                    writer.writeheader()
                    for file_short in self.container_lists[var_filetype]["Short"]:
                        writer.writerow(report_intensity[var_filetype][var_datatype][file_short])
                    report_file_smpl.write("\n")

                    if var_filetype == "SMPL":
                        report_file_smpl.write("Intensity Ratio (Sample)\n")  # Intensity Ratio
                        report_file_smpl.write("(1)\n")
                        writer.writeheader()
                        for file_short in self.container_lists[var_filetype]["Short"]:
                            writer.writerow(report_intensity_ratio[var_filetype][var_datatype][file_short])
                        report_file_smpl.write("\n")

                    report_file_smpl.write("SENSITIVITY ANALYSIS\n")

                    report_file_smpl.write("Analytical Sensitivity\n")  # Analytical Sensitivity
                    report_file_smpl.write("(1)\n")
                    writer.writeheader()
                    for file_short in self.container_lists[var_filetype]["Short"]:
                        writer.writerow(report_analytical_sensitivity[var_filetype][var_datatype][file_short])
                    report_file_smpl.write("\n")

                    report_file_smpl.write("Normalized Sensitivity\n")  # Normalized Sensitivity
                    report_file_smpl.write("(cps)/(ppm)\n")
                    writer.writeheader()
                    for file_short in self.container_lists[var_filetype]["Short"]:
                        writer.writerow(report_normalized_sensitivity[var_filetype][var_datatype][file_short])
                    report_file_smpl.write("\n")

                    if var_filetype == "SMPL":
                        report_file_smpl.write("Relative Sensitivity Factor\n")  # Relative Sensitivity Factor
                        report_file_smpl.write("(1)\n")
                        writer.writeheader()
                        for file_short in self.container_lists[var_filetype]["Short"]:
                            writer.writerow(report_rsf[var_filetype][var_datatype][file_short])
                        report_file_smpl.write("\n")

        if os.path.exists(filename_base):
            os.remove(filename_base)
        else:
            print("The file does not exist!")

    def ma_export_report_2(
            self, report_concentration, report_concentration_sigma, report_concentration_ratio, report_lob, report_lod,
            report_loq, report_intensity, report_intensity_sigma, report_intensity_ratio, report_analytical_sensitivity,
            report_normalized_sensitivity, report_rsf, var_file_extension, var_delimiter, header):
        export_file = filedialog.asksaveasfile(mode="w", defaultextension=var_file_extension)
        filename_base = export_file.name
        filename_base_parts = filename_base.split(".")
        filename_raw = filename_base_parts[0] + "_raw" + var_file_extension
        filename_smoothed = filename_base_parts[0] + "_smoothed" + var_file_extension

        # RAW Data
        with open(filename_raw, "w", newline="") as report_file_raw:
            writer = csv.DictWriter(report_file_raw, fieldnames=header, delimiter=var_delimiter)
            report_file_raw.write("CALCULATION REPORT\n")
            report_file_raw.write("\n")
            report_file_raw.write("AUTHOR:;" + str(self.container_var["ma_setting"]["Author"].get()) + "\n")
            report_file_raw.write("SOURCE ID:;" + str(self.container_var["ma_setting"]["Source ID"].get()) + "\n")
            report_file_raw.write("\n")
            for var_datatype in ["RAW"]:
                report_file_raw.write("DATA TYPE:;" + str(var_datatype) + str(" DATA") + "\n")
                report_file_raw.write("\n")
                for var_filetype in ["SMPL", "STD"]:
                    if var_filetype == "SMPL":
                        report_file_raw.write("SAMPLE FILES\n")
                    elif var_filetype == "STD":
                        report_file_raw.write("STANDARD FILES\n")
                    var_key = "Total " + str(var_filetype)
                    report_file_raw.write("\n")

                    ## COMPOSITIONAL ANALYSIS
                    report_file_raw.write("COMPOSITIONAL ANALYSIS\n")

                    report_file_raw.write("Concentration\n")  # Concentration
                    report_file_raw.write("(ppm)\n")
                    writer.writeheader()
                    for file_short in self.container_lists[var_filetype]["Short"]:
                        writer.writerow(report_concentration[var_filetype][var_datatype][file_short])
                    report_file_raw.write("\n")

                    report_file_raw.write("1-Sigma-Concentration\n")  # 1-Sigma-Concentration
                    report_file_raw.write("(ppm)\n")
                    writer.writeheader()
                    for file_short in self.container_lists[var_filetype]["Short"]:
                        writer.writerow(report_concentration_sigma[var_filetype][var_datatype][file_short])
                    report_file_raw.write("\n")

                    if var_filetype == "SMPL":
                        report_file_raw.write("Concentration Ratio\n")  # Concentration Ratio
                        report_file_raw.write("(1)\n")
                        writer.writeheader()
                        for file_short in self.container_lists[var_filetype]["Short"]:
                            writer.writerow(report_concentration_ratio[var_filetype][var_datatype][file_short])
                        report_file_raw.write("\n")

                        report_file_raw.write("Limit of Blank\n")  # Limit of Blank
                        report_file_raw.write("(ppm)\n")
                        writer.writeheader()
                        for file_short in self.container_lists[var_filetype]["Short"]:
                            writer.writerow(report_lob[var_filetype][var_datatype][file_short])
                        report_file_raw.write("\n")

                        report_file_raw.write("Limit of Detection\n")  # Limit of Detection
                        report_file_raw.write("(ppm)\n")
                        writer.writeheader()
                        for file_short in self.container_lists[var_filetype]["Short"]:
                            writer.writerow(report_lod[var_filetype][var_datatype][file_short])
                        report_file_raw.write("\n")

                        report_file_raw.write("Limit of Quantification\n")  # Limit of Quantification
                        report_file_raw.write("(ppm)\n")
                        writer.writeheader()
                        for file_short in self.container_lists[var_filetype]["Short"]:
                            writer.writerow(report_loq[var_filetype][var_datatype][file_short])
                        report_file_raw.write("\n")

                    report_file_raw.write("INTENSITY ANALYSIS\n")

                    report_file_raw.write("Intensity (Sample)\n")  # Intensity
                    report_file_raw.write("(cps)\n")
                    writer.writeheader()
                    for file_short in self.container_lists[var_filetype]["Short"]:
                        writer.writerow(report_intensity[var_filetype][var_datatype][file_short])
                    report_file_raw.write("\n")

                    report_file_raw.write("1-Sigma-Intensity (Sample)\n")  # 1-Sigma-Intensity
                    report_file_raw.write("(cps)\n")
                    writer.writeheader()
                    for file_short in self.container_lists[var_filetype]["Short"]:
                        writer.writerow(report_intensity_sigma[var_filetype][var_datatype][file_short])
                    report_file_raw.write("\n")

                    if var_filetype == "SMPL":
                        report_file_raw.write("Intensity Ratio (Sample)\n")  # Intensity Ratio
                        report_file_raw.write("(1)\n")
                        writer.writeheader()
                        for file_short in self.container_lists[var_filetype]["Short"]:
                            writer.writerow(report_intensity_ratio[var_filetype][var_datatype][file_short])
                        report_file_raw.write("\n")

                    report_file_raw.write("SENSITIVITY ANALYSIS\n")

                    report_file_raw.write("Analytical Sensitivity\n")  # Analytical Sensitivity
                    report_file_raw.write("(1)\n")
                    writer.writeheader()
                    for file_short in self.container_lists[var_filetype]["Short"]:
                        writer.writerow(report_analytical_sensitivity[var_filetype][var_datatype][file_short])
                    report_file_raw.write("\n")

                    report_file_raw.write("Normalized Sensitivity\n")  # Normalized Sensitivity
                    report_file_raw.write("(cps)/(ppm)\n")
                    writer.writeheader()
                    for file_short in self.container_lists[var_filetype]["Short"]:
                        writer.writerow(report_normalized_sensitivity[var_filetype][var_datatype][file_short])
                    report_file_raw.write("\n")

                    if var_filetype == "SMPL":
                        report_file_raw.write("Relative Sensitivity Factor\n")  # Relative Sensitivity Factor
                        report_file_raw.write("(1)\n")
                        writer.writeheader()
                        for file_short in self.container_lists[var_filetype]["Short"]:
                            writer.writerow(report_rsf[var_filetype][var_datatype][file_short])
                        report_file_raw.write("\n")
        # SMOOTHED Data
        with open(filename_smoothed, "w", newline="") as report_file_smoothed:
            writer = csv.DictWriter(report_file_smoothed, fieldnames=header, delimiter=var_delimiter)
            report_file_smoothed.write("CALCULATION REPORT\n")
            report_file_smoothed.write("\n")
            report_file_smoothed.write("AUTHOR:;" + str(self.container_var["ma_setting"]["Author"].get()) + "\n")
            report_file_smoothed.write("SOURCE ID:;" + str(self.container_var["ma_setting"]["Source ID"].get()) + "\n")
            report_file_smoothed.write("\n")
            for var_datatype in ["SMOOTHED"]:
                report_file_smoothed.write("DATA TYPE:;" + str(var_datatype) + str(" DATA") + "\n")
                report_file_smoothed.write("\n")
                for var_filetype in ["SMPL", "STD"]:
                    if var_filetype == "SMPL":
                        report_file_smoothed.write("SAMPLE FILES\n")
                    elif var_filetype == "STD":
                        report_file_smoothed.write("STANDARD FILES\n")
                    var_key = "Total " + str(var_filetype)
                    report_file_smoothed.write("\n")

                    ## COMPOSITIONAL ANALYSIS
                    report_file_smoothed.write("COMPOSITIONAL ANALYSIS\n")

                    report_file_smoothed.write("Concentration\n")  # Concentration
                    report_file_smoothed.write("(ppm)\n")
                    writer.writeheader()
                    for file_short in self.container_lists[var_filetype]["Short"]:
                        writer.writerow(report_concentration[var_filetype][var_datatype][file_short])
                    report_file_smoothed.write("\n")

                    if var_filetype == "SMPL":
                        report_file_smoothed.write("Concentration Ratio\n")  # Concentration Ratio
                        report_file_smoothed.write("(1)\n")
                        writer.writeheader()
                        for file_short in self.container_lists[var_filetype]["Short"]:
                            writer.writerow(report_concentration_ratio[var_filetype][var_datatype][file_short])
                        report_file_smoothed.write("\n")

                        report_file_smoothed.write("Limit of Blank\n")  # Limit of Blank
                        report_file_smoothed.write("(ppm)\n")
                        writer.writeheader()
                        for file_short in self.container_lists[var_filetype]["Short"]:
                            writer.writerow(report_lob[var_filetype][var_datatype][file_short])
                        report_file_smoothed.write("\n")

                        report_file_smoothed.write("Limit of Detection\n")  # Limit of Detection
                        report_file_smoothed.write("(ppm)\n")
                        writer.writeheader()
                        for file_short in self.container_lists[var_filetype]["Short"]:
                            writer.writerow(report_lod[var_filetype][var_datatype][file_short])
                        report_file_smoothed.write("\n")

                        report_file_smoothed.write("Limit of Quantification\n")  # Limit of Quantification
                        report_file_smoothed.write("(ppm)\n")
                        writer.writeheader()
                        for file_short in self.container_lists[var_filetype]["Short"]:
                            writer.writerow(report_loq[var_filetype][var_datatype][file_short])
                        report_file_smoothed.write("\n")

                    report_file_smoothed.write("INTENSITY ANALYSIS\n")

                    report_file_smoothed.write("Intensity (Sample)\n")  # Intensity
                    report_file_smoothed.write("(cps)\n")
                    writer.writeheader()
                    for file_short in self.container_lists[var_filetype]["Short"]:
                        writer.writerow(report_intensity[var_filetype][var_datatype][file_short])
                    report_file_smoothed.write("\n")

                    if var_filetype == "SMPL":
                        report_file_smoothed.write("Intensity Ratio (Sample)\n")  # Intensity Ratio
                        report_file_smoothed.write("(1)\n")
                        writer.writeheader()
                        for file_short in self.container_lists[var_filetype]["Short"]:
                            writer.writerow(report_intensity_ratio[var_filetype][var_datatype][file_short])
                        report_file_smoothed.write("\n")

                    report_file_smoothed.write("SENSITIVITY ANALYSIS\n")

                    report_file_smoothed.write("Analytical Sensitivity\n")  # Analytical Sensitivity
                    report_file_smoothed.write("(1)\n")
                    writer.writeheader()
                    for file_short in self.container_lists[var_filetype]["Short"]:
                        writer.writerow(report_analytical_sensitivity[var_filetype][var_datatype][file_short])
                    report_file_smoothed.write("\n")

                    report_file_smoothed.write("Normalized Sensitivity\n")  # Normalized Sensitivity
                    report_file_smoothed.write("(cps)/(ppm)\n")
                    writer.writeheader()
                    for file_short in self.container_lists[var_filetype]["Short"]:
                        writer.writerow(report_normalized_sensitivity[var_filetype][var_datatype][file_short])
                    report_file_smoothed.write("\n")

                    if var_filetype == "SMPL":
                        report_file_smoothed.write("Relative Sensitivity Factor\n")  # Relative Sensitivity Factor
                        report_file_smoothed.write("(1)\n")
                        writer.writeheader()
                        for file_short in self.container_lists[var_filetype]["Short"]:
                            writer.writerow(report_rsf[var_filetype][var_datatype][file_short])
                        report_file_smoothed.write("\n")

        if os.path.exists(filename_base):
            os.remove(filename_base)
        else:
            print("The file does not exist!")

    def fi_export_calculation_report(self):
        if self.pysills_mode in ["FI", "INCL"]:
            key_setting = "fi_setting"
        elif self.pysills_mode == "MI":
            key_setting = "mi_setting"

        header = ["filename", "ID"]

        # Concentration Inclusion
        report_concentration_incl = {}
        report_concentration_incl_1_sigma = {}
        report_concentration_incl["Total STD"] = {}
        report_concentration_incl["Total SMPL"] = {}
        # Concentration Matrix
        report_concentration_mat = {}
        report_concentration_mat_1_sigma = {}
        report_concentration_mat["Total STD"] = {}
        report_concentration_mat["Total SMPL"] = {}
        # Mixed Concentration
        report_concentration_mix = {}
        report_concentration_mix_1_sigma = {}
        report_concentration_mix["Total STD"] = {}
        report_concentration_mix["Total SMPL"] = {}
        # Limit of Blank
        report_lob_incl = {}
        report_lob_incl["Total STD"] = {}
        report_lob_incl["Total SMPL"] = {}
        report_lob_mat = {}
        report_lob_mat["Total STD"] = {}
        report_lob_mat["Total SMPL"] = {}
        # Limit of Detection
        report_lod_incl = {}
        report_lod_incl["Total STD"] = {}
        report_lod_incl["Total SMPL"] = {}
        report_lod_mat = {}
        report_lod_mat["Total STD"] = {}
        report_lod_mat["Total SMPL"] = {}
        # Limit of Quantification
        report_loq_incl = {}
        report_loq_incl["Total STD"] = {}
        report_loq_incl["Total SMPL"] = {}
        report_loq_mat = {}
        report_loq_mat["Total STD"] = {}
        report_loq_mat["Total SMPL"] = {}
        # Mixing ratio
        report_mixingratio_a = {}
        report_mixingratio_a["Total STD"] = {}
        report_mixingratio_a["Total SMPL"] = {}
        report_mixingratio_x = {}
        report_mixingratio_x["Total STD"] = {}
        report_mixingratio_x["Total SMPL"] = {}
        # Intensity Inclusion
        report_intensity_incl = {}
        report_intensity_incl["Total STD"] = {}
        report_intensity_incl["Total SMPL"] = {}
        # Intensity Matrix
        report_intensity_mat = {}
        report_intensity_mat["Total STD"] = {}
        report_intensity_mat["Total SMPL"] = {}
        # Intensity Background
        report_intensity_bg = {}
        report_intensity_bg["Total STD"] = {}
        report_intensity_bg["Total SMPL"] = {}
        # Intensity Mix
        report_intensity_mix = {}
        report_intensity_mix["Total STD"] = {}
        report_intensity_mix["Total SMPL"] = {}
        # Analytical Sensitivity
        report_analytical_sensitivity = {}
        report_analytical_sensitivity["Total STD"] = {}
        report_analytical_sensitivity["Total SMPL"] = {}
        # Normalized Sensitivity
        report_normalized_sensitivity = {}
        report_normalized_sensitivity["Total STD"] = {}
        report_normalized_sensitivity["Total SMPL"] = {}
        # Relative Sensitivity Factor
        report_rsf = {}
        report_rsf["Total STD"] = {}
        report_rsf["Total SMPL"] = {}

        for var_filetype in ["STD", "SMPL"]:
            var_key = "Total " + str(var_filetype)

            report_concentration_incl[var_filetype] = {}
            report_concentration_incl[var_key] = {}
            report_concentration_incl_1_sigma[var_filetype] = {}
            report_concentration_incl_1_sigma[var_key] = {}
            report_concentration_incl[var_key]["filename"] = "All Files"

            report_concentration_mat[var_filetype] = {}
            report_concentration_mat[var_key] = {}
            report_concentration_mat_1_sigma[var_filetype] = {}
            report_concentration_mat_1_sigma[var_key] = {}
            report_concentration_mat[var_key]["filename"] = "All Files"

            report_concentration_mix[var_filetype] = {}
            report_concentration_mix[var_key] = {}
            report_concentration_mix_1_sigma[var_filetype] = {}
            report_concentration_mix_1_sigma[var_key] = {}
            report_concentration_mix[var_key]["filename"] = "All Files"

            report_lob_incl[var_filetype] = {}
            report_lob_incl[var_key] = {}
            report_lob_incl[var_key]["filename"] = "All Files"

            report_lob_mat[var_filetype] = {}
            report_lob_mat[var_key] = {}
            report_lob_mat[var_key]["filename"] = "All Files"

            report_lod_incl[var_filetype] = {}
            report_lod_incl[var_key] = {}
            report_lod_incl[var_key]["filename"] = "All Files"

            report_lod_mat[var_filetype] = {}
            report_lod_mat[var_key] = {}
            report_lod_mat[var_key]["filename"] = "All Files"

            report_loq_incl[var_filetype] = {}
            report_loq_incl[var_key] = {}
            report_loq_incl[var_key]["filename"] = "All Files"

            report_loq_mat[var_filetype] = {}
            report_loq_mat[var_key] = {}
            report_loq_mat[var_key]["filename"] = "All Files"

            report_mixingratio_a[var_filetype] = {}
            report_mixingratio_a[var_key] = {}
            report_mixingratio_a[var_key]["filename"] = "All Files"

            report_mixingratio_x[var_filetype] = {}
            report_mixingratio_x[var_key] = {}
            report_mixingratio_x[var_key]["filename"] = "All Files"

            report_intensity_incl[var_filetype] = {}
            report_intensity_incl[var_key] = {}
            report_intensity_incl[var_key]["filename"] = "All Files"

            report_intensity_mat[var_filetype] = {}
            report_intensity_mat[var_key] = {}
            report_intensity_mat[var_key]["filename"] = "All Files"

            report_intensity_bg[var_filetype] = {}
            report_intensity_bg[var_key] = {}
            report_intensity_bg[var_key]["filename"] = "All Files"

            report_intensity_mix[var_filetype] = {}
            report_intensity_mix[var_key] = {}
            report_intensity_mix[var_key]["filename"] = "All Files"

            report_analytical_sensitivity[var_filetype] = {}
            report_analytical_sensitivity[var_key] = {}
            report_analytical_sensitivity[var_key]["filename"] = "All Files"

            report_normalized_sensitivity[var_filetype] = {}
            report_normalized_sensitivity[var_key] = {}
            report_normalized_sensitivity[var_key]["filename"] = "All Files"

            report_rsf[var_filetype] = {}
            report_rsf[var_key] = {}
            report_rsf[var_key]["filename"] = "All Files"

            for var_datatype in ["SMOOTHED", "RAW"]:
                report_concentration_incl[var_filetype][var_datatype] = {}
                report_concentration_incl_1_sigma[var_filetype][var_datatype] = {}
                report_concentration_mat[var_filetype][var_datatype] = {}
                report_concentration_mat_1_sigma[var_filetype][var_datatype] = {}
                report_concentration_mix[var_filetype][var_datatype] = {}
                report_concentration_mix_1_sigma[var_filetype][var_datatype] = {}
                report_lob_incl[var_filetype][var_datatype] = {}
                report_lob_mat[var_filetype][var_datatype] = {}
                report_lod_incl[var_filetype][var_datatype] = {}
                report_lod_mat[var_filetype][var_datatype] = {}
                report_loq_incl[var_filetype][var_datatype] = {}
                report_loq_mat[var_filetype][var_datatype] = {}
                report_mixingratio_a[var_filetype][var_datatype] = {}
                report_mixingratio_x[var_filetype][var_datatype] = {}

                report_intensity_incl[var_filetype][var_datatype] = {}
                report_intensity_mat[var_filetype][var_datatype] = {}
                report_intensity_bg[var_filetype][var_datatype] = {}
                report_intensity_mix[var_filetype][var_datatype] = {}

                report_analytical_sensitivity[var_filetype][var_datatype] = {}
                report_normalized_sensitivity[var_filetype][var_datatype] = {}
                report_rsf[var_filetype][var_datatype] = {}

                for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                    file_long = self.container_lists[var_filetype]["Long"][index]
                    file_isotopes = self.container_lists["Measured Isotopes"][file_short]

                    if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                        report_concentration_incl[var_filetype][var_datatype][file_short] = {}
                        report_concentration_incl[var_filetype][var_datatype][file_short]["filename"] = file_short
                        report_concentration_incl_1_sigma[var_filetype][var_datatype][file_short] = {}
                        report_concentration_incl_1_sigma[var_filetype][var_datatype][file_short]["filename"] = file_short
                        report_concentration_mat[var_filetype][var_datatype][file_short] = {}
                        report_concentration_mat[var_filetype][var_datatype][file_short]["filename"] = file_short
                        report_concentration_mat_1_sigma[var_filetype][var_datatype][file_short] = {}
                        report_concentration_mat_1_sigma[var_filetype][var_datatype][file_short]["filename"] = file_short
                        report_concentration_mix[var_filetype][var_datatype][file_short] = {}
                        report_concentration_mix[var_filetype][var_datatype][file_short]["filename"] = file_short
                        report_concentration_mix_1_sigma[var_filetype][var_datatype][file_short] = {}
                        report_concentration_mix_1_sigma[var_filetype][var_datatype][file_short]["filename"] = file_short
                        report_lob_incl[var_filetype][var_datatype][file_short] = {}
                        report_lob_incl[var_filetype][var_datatype][file_short]["filename"] = file_short
                        report_lob_mat[var_filetype][var_datatype][file_short] = {}
                        report_lob_mat[var_filetype][var_datatype][file_short]["filename"] = file_short
                        report_lod_incl[var_filetype][var_datatype][file_short] = {}
                        report_lod_incl[var_filetype][var_datatype][file_short]["filename"] = file_short
                        report_lod_mat[var_filetype][var_datatype][file_short] = {}
                        report_lod_mat[var_filetype][var_datatype][file_short]["filename"] = file_short
                        report_loq_incl[var_filetype][var_datatype][file_short] = {}
                        report_loq_incl[var_filetype][var_datatype][file_short]["filename"] = file_short
                        report_loq_mat[var_filetype][var_datatype][file_short] = {}
                        report_loq_mat[var_filetype][var_datatype][file_short]["filename"] = file_short
                        report_mixingratio_a[var_filetype][var_datatype][file_short] = {}
                        report_mixingratio_a[var_filetype][var_datatype][file_short]["filename"] = file_short
                        report_mixingratio_x[var_filetype][var_datatype][file_short] = {}
                        report_mixingratio_x[var_filetype][var_datatype][file_short]["filename"] = file_short

                        if var_filetype == "STD":
                            report_concentration_incl[var_filetype][var_datatype][file_short]["ID"] = "---"
                            report_concentration_incl_1_sigma[var_filetype][var_datatype][file_short]["ID"] = "---"
                            report_concentration_mat[var_filetype][var_datatype][file_short]["ID"] = "---"
                            report_concentration_mat_1_sigma[var_filetype][var_datatype][file_short]["ID"] = "---"
                            report_concentration_mix[var_filetype][var_datatype][file_short]["ID"] = "---"
                            report_concentration_mix_1_sigma[var_filetype][var_datatype][file_short]["ID"] = "---"
                            report_lob_incl[var_filetype][var_datatype][file_short]["ID"] = "---"
                            report_lob_mat[var_filetype][var_datatype][file_short]["ID"] = "---"
                            report_lod_incl[var_filetype][var_datatype][file_short]["ID"] = "---"
                            report_lod_mat[var_filetype][var_datatype][file_short]["ID"] = "---"
                            report_loq_incl[var_filetype][var_datatype][file_short]["ID"] = "---"
                            report_loq_mat[var_filetype][var_datatype][file_short]["ID"] = "---"
                            report_mixingratio_a[var_filetype][var_datatype][file_short]["ID"] = "---"
                            report_mixingratio_x[var_filetype][var_datatype][file_short]["ID"] = "---"
                        else:
                            report_concentration_incl[var_filetype][var_datatype][file_short]["ID"] = self.container_var[
                                var_filetype][file_long]["ID"].get()
                            report_concentration_incl_1_sigma[var_filetype][var_datatype][file_short][
                                "ID"] = self.container_var[var_filetype][file_long]["ID"].get()
                            report_concentration_mat[var_filetype][var_datatype][file_short]["ID"] = self.container_var[
                                var_filetype][file_long]["ID"].get()
                            report_concentration_mat_1_sigma[var_filetype][var_datatype][file_short][
                                "ID"] = self.container_var[var_filetype][file_long]["ID"].get()
                            report_concentration_mix[var_filetype][var_datatype][file_short]["ID"] = self.container_var[
                                var_filetype][file_long]["ID"].get()
                            report_concentration_mix_1_sigma[var_filetype][var_datatype][file_short][
                                "ID"] = self.container_var[var_filetype][file_long]["ID"].get()
                            report_lob_incl[var_filetype][var_datatype][file_short]["ID"] = self.container_var[
                                var_filetype][file_long]["ID"].get()
                            report_lob_mat[var_filetype][var_datatype][file_short]["ID"] = self.container_var[
                                var_filetype][file_long]["ID"].get()
                            report_lod_incl[var_filetype][var_datatype][file_short]["ID"] = self.container_var[
                                var_filetype][file_long]["ID"].get()
                            report_lod_mat[var_filetype][var_datatype][file_short]["ID"] = self.container_var[
                                var_filetype][file_long]["ID"].get()
                            report_loq_incl[var_filetype][var_datatype][file_short]["ID"] = self.container_var[
                                var_filetype][file_long]["ID"].get()
                            report_loq_mat[var_filetype][var_datatype][file_short]["ID"] = self.container_var[
                                var_filetype][file_long]["ID"].get()
                            report_mixingratio_a[var_filetype][var_datatype][file_short]["ID"] = self.container_var[
                                var_filetype][file_long]["ID"].get()
                            report_mixingratio_x[var_filetype][var_datatype][file_short]["ID"] = self.container_var[
                                var_filetype][file_long]["ID"].get()

                        report_intensity_incl[var_filetype][var_datatype][file_short] = {}
                        report_intensity_incl[var_filetype][var_datatype][file_short]["filename"] = file_short
                        report_intensity_mat[var_filetype][var_datatype][file_short] = {}
                        report_intensity_mat[var_filetype][var_datatype][file_short]["filename"] = file_short
                        report_intensity_bg[var_filetype][var_datatype][file_short] = {}
                        report_intensity_bg[var_filetype][var_datatype][file_short]["filename"] = file_short
                        report_intensity_mix[var_filetype][var_datatype][file_short] = {}
                        report_intensity_mix[var_filetype][var_datatype][file_short]["filename"] = file_short

                        if var_filetype == "STD":
                            report_intensity_incl[var_filetype][var_datatype][file_short]["ID"] = "---"
                            report_intensity_mat[var_filetype][var_datatype][file_short]["ID"] = "---"
                            report_intensity_bg[var_filetype][var_datatype][file_short]["ID"] = "---"
                            report_intensity_mix[var_filetype][var_datatype][file_short]["ID"] = "---"
                        else:
                            report_intensity_incl[var_filetype][var_datatype][file_short]["ID"] = self.container_var[
                                var_filetype][file_long]["ID"].get()
                            report_intensity_mat[var_filetype][var_datatype][file_short]["ID"] = self.container_var[
                                var_filetype][file_long]["ID"].get()
                            report_intensity_bg[var_filetype][var_datatype][file_short]["ID"] = self.container_var[
                                var_filetype][file_long]["ID"].get()
                            report_intensity_mix[var_filetype][var_datatype][file_short]["ID"] = self.container_var[
                                var_filetype][file_long]["ID"].get()

                        report_analytical_sensitivity[var_filetype][var_datatype][file_short] = {}
                        report_analytical_sensitivity[var_filetype][var_datatype][file_short]["filename"] = file_short
                        report_normalized_sensitivity[var_filetype][var_datatype][file_short] = {}
                        report_normalized_sensitivity[var_filetype][var_datatype][file_short]["filename"] = file_short
                        report_rsf[var_filetype][var_datatype][file_short] = {}
                        report_rsf[var_filetype][var_datatype][file_short]["filename"] = file_short

                        if var_filetype == "STD":
                            report_analytical_sensitivity[var_filetype][var_datatype][file_short]["ID"] = "---"
                            report_normalized_sensitivity[var_filetype][var_datatype][file_short]["ID"] = "---"
                            report_rsf[var_filetype][var_datatype][file_short]["ID"] = "---"
                        else:
                            report_analytical_sensitivity[var_filetype][var_datatype][file_short][
                                "ID"] = self.container_var[var_filetype][file_long]["ID"].get()
                            report_normalized_sensitivity[var_filetype][var_datatype][file_short][
                                "ID"] = self.container_var[var_filetype][file_long]["ID"].get()
                            report_rsf[var_filetype][var_datatype][file_short]["ID"] = self.container_var[var_filetype][
                                file_long]["ID"].get()

                        for isotope in file_isotopes:
                            ## COMPOSITIONAL ANALYSIS
                            # Concentration Inclusion
                            if var_filetype == "SMPL":
                                value_i = self.container_concentration[var_filetype][var_datatype][file_short]["INCL"][
                                    isotope]
                                lod_i = self.container_lod[var_filetype][var_datatype][file_short]["INCL"][isotope]
                                value_sigma_i = self.container_concentration[var_filetype][var_datatype][file_short][
                                    "1 SIGMA INCL"][isotope]
                            else:
                                value_i = 0.0
                                lod_i = 0.0
                                value_sigma_i = 0.0

                            n_digits = 5

                            if value_i >= lod_i:
                                value_i = value_i
                            else:
                                #value_i *= -1
                                value_i = -lod_i

                            report_concentration_incl[var_filetype][var_datatype][file_short][isotope] = round(
                                value_i, n_digits)
                            report_concentration_incl_1_sigma[var_filetype][var_datatype][file_short][isotope] = round(
                                value_sigma_i, n_digits)

                            # Concentration Matrix
                            if var_filetype == "STD":
                                var_srm_i = self.container_var["SRM"][isotope].get()
                                var_srm_file = self.container_var["STD"][file_long]["SRM"].get()
                            value_i = self.container_concentration[var_filetype][var_datatype][file_short]["MAT"][
                                isotope]
                            lod_i = self.container_lod[var_filetype][var_datatype][file_short]["MAT"][isotope]
                            n_digits = 5

                            if value_i >= lod_i:
                                value_i = value_i
                            else:
                                #value_i *= -1
                                value_i = -lod_i

                            if var_filetype == "SMPL":
                                value_sigma_i = self.container_concentration[var_filetype][var_datatype][file_short][
                                    "1 SIGMA MAT"][isotope]
                                report_concentration_mat[var_filetype][var_datatype][file_short][isotope] = round(
                                    value_i, n_digits)
                                report_concentration_mat_1_sigma[var_filetype][var_datatype][file_short][
                                    isotope] = round(value_sigma_i, n_digits)

                            else:
                                if var_srm_i == var_srm_file:
                                    report_concentration_mat[var_filetype][var_datatype][file_short][isotope] = round(
                                        value_i, n_digits)
                                else:
                                    report_concentration_mat[var_filetype][var_datatype][file_short][
                                        isotope] = "undefined"

                            # Concentration Mixed
                            if var_filetype == "SMPL":
                                value_i = self.container_mixed_concentration[var_filetype][var_datatype][file_short][
                                    isotope]
                            else:
                                value_i = 0.0

                            n_digits = 5
                            report_concentration_mix[var_filetype][var_datatype][file_short][isotope] = round(
                                value_i, n_digits)

                            # Limit of Blank (Inclusion)
                            if var_filetype == "SMPL":
                                value_i = self.container_lob[var_filetype][var_datatype][file_short]["INCL"][isotope]
                            else:
                                value_i = 0.0

                            n_digits = 5
                            report_lob_incl[var_filetype][var_datatype][file_short][isotope] = round(
                                value_i, n_digits)

                            # Limit of Detection (Inclusion)
                            if var_filetype == "SMPL":
                                value_i = self.container_lod[var_filetype][var_datatype][file_short]["INCL"][isotope]
                            else:
                                value_i = 0.0

                            n_digits = 5
                            report_lod_incl[var_filetype][var_datatype][file_short][isotope] = round(
                                value_i, n_digits)

                            # Limit of Quantification (Inclusion)
                            if var_filetype == "SMPL":
                                value_i = self.container_loq[var_filetype][var_datatype][file_short]["INCL"][isotope]
                            else:
                                value_i = 0.0

                            n_digits = 5
                            report_loq_incl[var_filetype][var_datatype][file_short][isotope] = round(
                                value_i, n_digits)

                            # Limit of Blank (Matrix)
                            value_i = self.container_lob[var_filetype][var_datatype][file_short]["MAT"][isotope]
                            n_digits = 5
                            report_lob_mat[var_filetype][var_datatype][file_short][isotope] = round(
                                value_i, n_digits)

                            # Limit of Detection (Matrix)
                            value_i = self.container_lod[var_filetype][var_datatype][file_short]["MAT"][isotope]
                            n_digits = 5
                            report_lod_mat[var_filetype][var_datatype][file_short][isotope] = round(
                                value_i, n_digits)

                            # Limit of Quantification (Matrix)
                            value_i = self.container_loq[var_filetype][var_datatype][file_short]["MAT"][isotope]
                            n_digits = 5
                            report_loq_mat[var_filetype][var_datatype][file_short][isotope] = round(
                                value_i, n_digits)

                            # Mixing Ratio (Factor a)
                            if var_filetype == "SMPL":
                                value_i = self.container_mixed_concentration_ratio[var_filetype][var_datatype][
                                    file_short][isotope]
                            else:
                                value_i = 0.0

                            report_mixingratio_a[var_filetype][var_datatype][file_short][isotope] = "{:0.5e}".format(
                                value_i)

                            # Mixing Ratio (Factor x)
                            if var_filetype == "SMPL":
                                value_i = self.container_mixing_ratio[var_filetype][var_datatype][file_short][isotope]
                            else:
                                value_i = 0.0

                            report_mixingratio_x[var_filetype][var_datatype][file_short][isotope] = "{:0.5e}".format(
                                value_i)

                            ## INTENSITY ANALYSIS
                            # Intensity Inclusion
                            if var_filetype == "SMPL":
                                value_i = self.container_intensity_corrected[var_filetype][var_datatype][file_short][
                                    "INCL"][isotope]
                            else:
                                value_i = 0.0

                            n_digits = 5
                            report_intensity_incl[var_filetype][var_datatype][file_short][isotope] = round(
                                value_i, n_digits)

                            # Intensity Matrix
                            value_i = self.container_intensity_corrected[var_filetype][var_datatype][file_short]["MAT"][
                                isotope]
                            n_digits = 5
                            report_intensity_mat[var_filetype][var_datatype][file_short][isotope] = round(
                                value_i, n_digits)

                            # Intensity Background
                            value_i = self.container_intensity_corrected[var_filetype][var_datatype][file_short]["BG"][
                                isotope]
                            n_digits = 5
                            report_intensity_bg[var_filetype][var_datatype][file_short][isotope] = round(
                                value_i, n_digits)

                            # Intensity Mixed
                            if var_filetype == "SMPL":
                                value_i = self.container_intensity_mix[var_filetype][var_datatype][file_short][isotope]
                            else:
                                value_i = 0.0

                            n_digits = 5
                            report_intensity_mix[var_filetype][var_datatype][file_short][isotope] = round(
                                value_i, n_digits)

                            ## SENSITIVITY ANALYSIS
                            # Analytical Sensitivity
                            value_i = self.container_analytical_sensitivity[var_filetype][var_datatype][file_short]["MAT"][
                                isotope]

                            if var_filetype == "STD" and index == 0:
                                helper_std = {}
                                helper_std[isotope] = []
                                var_srm_i = self.container_var["SRM"][isotope].get()
                                for index, filename_short in enumerate(self.container_lists["STD"]["Short"]):
                                    filename_long = self.container_lists["STD"]["Long"][index]
                                    var_srm_file = self.container_var["STD"][filename_long]["SRM"].get()
                                    if var_srm_file == var_srm_i:
                                        value_i = self.container_analytical_sensitivity[var_filetype][var_datatype][
                                            filename_short]["MAT"][isotope]
                                        helper_std[isotope].append(value_i)

                            n_digits = 5

                            if value_i != None:
                                report_analytical_sensitivity[var_filetype][var_datatype][file_short][isotope] = round(
                                    value_i, n_digits)
                            else:
                                report_analytical_sensitivity[var_filetype][var_datatype][file_short][isotope] = "undefined"

                            # Normalized Sensitivity
                            value_i = self.container_normalized_sensitivity[var_filetype][var_datatype][file_short]["MAT"][
                                isotope]
                            n_digits = 5
                            report_normalized_sensitivity[var_filetype][var_datatype][file_short][isotope] = round(
                                value_i, n_digits)

                            # Relative Sensitivity Factor
                            value_i = self.container_rsf[var_filetype][var_datatype][file_short]["MAT"][isotope]
                            report_rsf[var_filetype][var_datatype][file_short][isotope] = "{:0.5e}".format(value_i)

        for isotope in self.container_lists["ISOTOPES"]:
            header.append(isotope)

        var_file_extension_pre = self.container_var["General Settings"]["File type"].get()
        if var_file_extension_pre == "*.csv":
            var_file_extension = ".csv"
        elif var_file_extension_pre == "*.txt":
            var_file_extension = ".txt"

        var_delimiter_pre = self.container_var["General Settings"]["Delimiter"].get()
        if var_delimiter_pre == "comma":
            var_delimiter = ","
        elif var_delimiter_pre == "semicolon":
            var_delimiter = ";"

        if self.rb_report.get() == 0:  # All in one
            self.fi_export_report_0(
                report_concentration_incl, report_concentration_incl_1_sigma, report_concentration_mat,
                report_concentration_mat_1_sigma, report_concentration_mix, report_mixingratio_a,
                report_mixingratio_x, report_lob_incl, report_lob_mat, report_lod_incl, report_lod_mat, report_loq_incl,
                report_loq_mat, report_intensity_incl, report_intensity_mat, report_intensity_bg, report_intensity_mix,
                report_analytical_sensitivity, report_normalized_sensitivity, report_rsf, var_file_extension,
                var_delimiter, header)
        elif self.rb_report.get() == 1:  # STD vs. SMPL
            self.fi_export_report_1(
                report_concentration_incl, report_concentration_incl_1_sigma, report_concentration_mat,
                report_concentration_mat_1_sigma, report_concentration_mix, report_mixingratio_a,
                report_mixingratio_x, report_lob_incl, report_lob_mat, report_lod_incl, report_lod_mat, report_loq_incl,
                report_loq_mat, report_intensity_incl, report_intensity_mat, report_intensity_bg, report_intensity_mix,
                report_analytical_sensitivity, report_normalized_sensitivity, report_rsf, var_file_extension,
                var_delimiter, header)
        elif self.rb_report.get() == 2:  # RAW vs. SMOOTHED
            self.fi_export_report_2(
                report_concentration_incl, report_concentration_incl_1_sigma, report_concentration_mat,
                report_concentration_mat_1_sigma, report_concentration_mix, report_mixingratio_a,
                report_mixingratio_x, report_lob_incl, report_lob_mat, report_lod_incl, report_lod_mat, report_loq_incl,
                report_loq_mat, report_intensity_incl, report_intensity_mat, report_intensity_bg, report_intensity_mix,
                report_analytical_sensitivity, report_normalized_sensitivity, report_rsf, var_file_extension,
                var_delimiter, header)

    def fi_export_report_0(
            self, report_concentration_incl, report_concentration_incl_1_sigma, report_concentration_mat,
            report_concentration_mat_1_sigma, report_concentration_mix, report_mixingratio_a,
            report_mixingratio_x, report_lob_incl, report_lob_mat, report_lod_incl, report_lod_mat, report_loq_incl,
            report_loq_mat, report_intensity_incl, report_intensity_mat, report_intensity_bg, report_intensity_mix,
            report_analytical_sensitivity, report_normalized_sensitivity, report_rsf, var_file_extension,
            var_delimiter, header):
        export_file = filedialog.asksaveasfile(mode="w", defaultextension=var_file_extension)
        filename = export_file.name

        with open(filename, "w", newline="") as report_file:
            writer = csv.DictWriter(report_file, fieldnames=header, delimiter=var_delimiter)
            report_file.write("CALCULATION REPORT\n")
            report_file.write("\n")
            report_file.write("AUTHOR:;" + str(self.container_var["ma_setting"]["Author"].get()) + "\n")
            report_file.write("SOURCE ID:;" + str(self.container_var["ma_setting"]["Source ID"].get()) + "\n")
            report_file.write("\n")

            for var_datatype in ["SMOOTHED", "RAW"]:
                report_file.write("DATA TYPE:;" + str(var_datatype) + str(" DATA") + "\n")
                report_file.write("\n")

                for var_filetype in ["SMPL", "STD"]:
                    if var_filetype == "SMPL":
                        report_file.write("SAMPLE FILES\n")
                    elif var_filetype == "STD":
                        report_file.write("STANDARD FILES\n")

                    var_key = "Total " + str(var_filetype)
                    report_file.write("\n")

                    ## COMPOSITIONAL ANALYSIS
                    report_file.write("COMPOSITIONAL ANALYSIS\n")
                    if var_filetype == "SMPL":
                        report_file.write("Composition (Inclusion)\n")  # Concentration Inclusion
                        report_file.write("(ppm)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_concentration_incl[var_filetype][var_datatype][file_short])

                        report_file.write("\n")

                        report_file.write("1-Sigma Composition (Inclusion)\n")  # Concentration Inclusion (1-Sigma)
                        report_file.write("(ppm)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_concentration_incl_1_sigma[var_filetype][var_datatype][
                                                    file_short])

                        report_file.write("\n")

                        report_file.write("Limit of Blank (Inclusion)\n")  # Limit of Blank (Inclusion)
                        report_file.write("(ppm)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_lob_incl[var_filetype][var_datatype][file_short])

                        report_file.write("\n")

                        report_file.write("Limit of Detection (Inclusion)\n")  # Limit of Detection (Inclusion)
                        report_file.write("(ppm)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_lod_incl[var_filetype][var_datatype][file_short])

                        report_file.write("\n")

                        report_file.write("Limit of Quantification (Inclusion)\n")  # Limit of Quantification (Inclusion)
                        report_file.write("(ppm)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_loq_incl[var_filetype][var_datatype][file_short])

                        report_file.write("\n")

                    report_file.write("Composition (Matrix)\n")  # Concentration Matrix
                    report_file.write("(ppm)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_concentration_mat[var_filetype][var_datatype][file_short])

                    report_file.write("\n")

                    if var_filetype == "SMPL":
                        report_file.write("1-Sigma Composition (Matrix)\n")  # Concentration Matrix (1-Sigma)
                        report_file.write("(ppm)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_concentration_mat_1_sigma[var_filetype][var_datatype][
                                                    file_short])

                    report_file.write("\n")

                    report_file.write("Limit of Blank (Matrix)\n")  # Limit of Blank (Matrix)
                    report_file.write("(ppm)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_lob_mat[var_filetype][var_datatype][file_short])

                    report_file.write("\n")

                    report_file.write("Limit of Detection (Matrix)\n")  # Limit of Detection (Matrix)
                    report_file.write("(ppm)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_lod_mat[var_filetype][var_datatype][file_short])

                    report_file.write("\n")

                    report_file.write("Limit of Quantification (Matrix)\n")  # Limit of Quantification (Matrix)
                    report_file.write("(ppm)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_loq_mat[var_filetype][var_datatype][file_short])

                    report_file.write("\n")

                    if var_filetype == "SMPL":
                        report_file.write("Composition (Mixed)\n")  # Concentration Mixed
                        report_file.write("(ppm)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_concentration_mix[var_filetype][var_datatype][file_short])

                        report_file.write("\n")

                    if var_filetype == "SMPL":
                        report_file.write("Mixing Ratio a\n")  # Mixing Ratio a
                        report_file.write("(1)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_mixingratio_a[var_filetype][var_datatype][file_short])

                        report_file.write("\n")

                        report_file.write("Mixing Ratio x\n")  # Mixing Ratio x
                        report_file.write("(1)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_mixingratio_x[var_filetype][var_datatype][file_short])

                        report_file.write("\n")

                    report_file.write("INTENSITY ANALYSIS\n")
                    if var_filetype == "SMPL":
                        report_file.write("Intensity (Inclusion)\n")  # Intensity Inclusion
                        report_file.write("(cps)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_intensity_incl[var_filetype][var_datatype][file_short])

                        report_file.write("\n")

                    report_file.write("Intensity (Matrix)\n")  # Intensity Matrix
                    report_file.write("(cps)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_intensity_mat[var_filetype][var_datatype][file_short])

                    report_file.write("\n")

                    report_file.write("Intensity (Background)\n")  # Intensity Background
                    report_file.write("(cps)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_intensity_bg[var_filetype][var_datatype][file_short])

                    report_file.write("\n")

                    if var_filetype == "SMPL":
                        report_file.write("Intensity (Mixed)\n")  # Intensity Mixed
                        report_file.write("(cps)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_intensity_mix[var_filetype][var_datatype][file_short])

                        report_file.write("\n")

                    report_file.write("SENSITIVITY ANALYSIS\n")
                    report_file.write("Analytical Sensitivity\n")  # Analytical Sensitivity
                    report_file.write("(1)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_analytical_sensitivity[var_filetype][var_datatype][file_short])

                    report_file.write("\n")

                    report_file.write("Normalized Sensitivity\n")  # Normalized Sensitivity
                    report_file.write("(cps)/(ppm)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_normalized_sensitivity[var_filetype][var_datatype][file_short])

                    report_file.write("\n")

                    report_file.write("Relative Sensitivity Factor\n")  # Relative Sensitivity Factor
                    report_file.write("(1)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_rsf[var_filetype][var_datatype][file_short])

                    report_file.write("\n")

    def fi_export_report_1(
            self, report_concentration_incl, report_concentration_incl_1_sigma, report_concentration_mat,
            report_concentration_mat_1_sigma, report_concentration_mix, report_mixingratio_a,
            report_mixingratio_x, report_lob_incl, report_lob_mat, report_lod_incl, report_lod_mat, report_loq_incl,
            report_loq_mat, report_intensity_incl, report_intensity_mat, report_intensity_bg, report_intensity_mix,
            report_analytical_sensitivity, report_normalized_sensitivity, report_rsf, var_file_extension,
            var_delimiter, header):
        export_file = filedialog.asksaveasfile(mode="w", defaultextension=var_file_extension)
        filename_base = export_file.name
        filename_base_parts = filename_base.split(".")
        filename_std = filename_base_parts[0] + "_std" + var_file_extension
        filename_smpl = filename_base_parts[0] + "_smpl" + var_file_extension
        export_file.close()

        # Standard Files
        with open(filename_std, "w", newline="") as report_file_std:
            writer = csv.DictWriter(report_file_std, fieldnames=header, delimiter=var_delimiter)
            report_file_std.write("CALCULATION REPORT\n")
            report_file_std.write("\n")
            report_file_std.write("AUTHOR:;" + str(self.container_var["fi_setting"]["Author"].get()) + "\n")
            report_file_std.write("SOURCE ID:;" + str(self.container_var["fi_setting"]["Source ID"].get()) + "\n")
            report_file_std.write("\n")
            for var_datatype in ["SMOOTHED", "RAW"]:
                report_file_std.write("DATA TYPE:;" + str(var_datatype) + str(" DATA") + "\n")
                report_file_std.write("\n")
                for var_filetype in ["STD"]:
                    report_file_std.write("STANDARD FILES\n")
                    var_key = "Total " + str(var_filetype)
                    report_file_std.write("\n")

                    ## COMPOSITIONAL ANALYSIS
                    report_file_std.write("COMPOSITIONAL ANALYSIS\n")
                    report_file_std.write("Composition (Matrix)\n")  # Concentration Matrix
                    report_file_std.write("(ppm)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_concentration_mat[var_filetype][var_datatype][file_short])

                    report_file_std.write("\n")

                    report_file_std.write("Limit of Blank (Matrix)\n")  # Limit of Blank (Matrix)
                    report_file_std.write("(ppm)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_lob_mat[var_filetype][var_datatype][file_short])

                    report_file_std.write("\n")

                    report_file_std.write("Limit of Detection (Matrix)\n")  # Limit of Detection (Matrix)
                    report_file_std.write("(ppm)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_lod_mat[var_filetype][var_datatype][file_short])

                    report_file_std.write("\n")

                    report_file_std.write("Limit of Quantification (Matrix)\n")  # Limit of Quantification (Matrix)
                    report_file_std.write("(ppm)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_loq_mat[var_filetype][var_datatype][file_short])

                    report_file_std.write("\n")

                    report_file_std.write("INTENSITY ANALYSIS\n")
                    report_file_std.write("Intensity (Matrix)\n")  # Intensity Matrix
                    report_file_std.write("(cps)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_intensity_mat[var_filetype][var_datatype][file_short])

                    report_file_std.write("\n")

                    report_file_std.write("Intensity (Background)\n")  # Intensity Background
                    report_file_std.write("(cps)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_intensity_bg[var_filetype][var_datatype][file_short])

                    report_file_std.write("\n")

                    report_file_std.write("SENSITIVITY ANALYSIS\n")
                    report_file_std.write("Analytical Sensitivity\n")  # Analytical Sensitivity
                    report_file_std.write("(1)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_analytical_sensitivity[var_filetype][var_datatype][file_short])

                    report_file_std.write("\n")

                    report_file_std.write("Normalized Sensitivity\n")  # Normalized Sensitivity
                    report_file_std.write("(cps)/(ppm)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_normalized_sensitivity[var_filetype][var_datatype][file_short])

                    report_file_std.write("\n")

                    report_file_std.write("Relative Sensitivity Factor\n")  # Relative Sensitivity Factor
                    report_file_std.write("(1)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_rsf[var_filetype][var_datatype][file_short])

                    report_file_std.write("\n")

        # Sample Files
        with open(filename_smpl, "w", newline="") as report_file_smpl:
            writer = csv.DictWriter(report_file_smpl, fieldnames=header, delimiter=var_delimiter)
            report_file_smpl.write("CALCULATION REPORT\n")
            report_file_smpl.write("\n")
            report_file_smpl.write("AUTHOR:;" + str(self.container_var["fi_setting"]["Author"].get()) + "\n")
            report_file_smpl.write("SOURCE ID:;" + str(self.container_var["fi_setting"]["Source ID"].get()) + "\n")
            report_file_smpl.write("\n")
            for var_datatype in ["SMOOTHED", "RAW"]:
                report_file_smpl.write("DATA TYPE:;" + str(var_datatype) + str(" DATA") + "\n")
                report_file_smpl.write("\n")
                for var_filetype in ["SMPL"]:
                    if var_filetype == "SMPL":
                        report_file_smpl.write("SAMPLE FILES\n")
                    elif var_filetype == "STD":
                        report_file_smpl.write("STANDARD FILES\n")
                    var_key = "Total " + str(var_filetype)
                    report_file_smpl.write("\n")

                    ## COMPOSITIONAL ANALYSIS
                    report_file_smpl.write("COMPOSITIONAL ANALYSIS\n")
                    if var_filetype == "SMPL":
                        report_file_smpl.write("Composition (Inclusion)\n")  # Concentration Inclusion
                        report_file_smpl.write("(ppm)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_concentration_incl[var_filetype][var_datatype][file_short])

                        report_file_smpl.write("\n")

                        report_file_smpl.write("1-Sigma Composition (Inclusion)\n")  # Concentration Inclusion (1-Sigma)
                        report_file_smpl.write("(ppm)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_concentration_incl_1_sigma[var_filetype][var_datatype][
                                                    file_short])

                        report_file_smpl.write("\n")

                        report_file_smpl.write("Limit of Blank (Inclusion)\n")  # Limit of Blank (Inclusion)
                        report_file_smpl.write("(ppm)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_lob_incl[var_filetype][var_datatype][file_short])

                        report_file_smpl.write("\n")

                        report_file_smpl.write("Limit of Detection (Inclusion)\n")  # Limit of Detection (Inclusion)
                        report_file_smpl.write("(ppm)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_lod_incl[var_filetype][var_datatype][file_short])

                        report_file_smpl.write("\n")

                        report_file_smpl.write("Limit of Quantification (Inclusion)\n")  # Limit of Quantification (Inclusion)
                        report_file_smpl.write("(ppm)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_loq_incl[var_filetype][var_datatype][file_short])

                        report_file_smpl.write("\n")

                    report_file_smpl.write("Composition (Matrix)\n")  # Concentration Matrix
                    report_file_smpl.write("(ppm)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_concentration_mat[var_filetype][var_datatype][file_short])

                    report_file_smpl.write("\n")

                    report_file_smpl.write("1-Sigma Composition (Matrix)\n")  # Concentration Matrix (1-Sigma)
                    report_file_smpl.write("(ppm)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_concentration_mat_1_sigma[var_filetype][var_datatype][file_short])

                    report_file_smpl.write("\n")

                    report_file_smpl.write("Limit of Blank (Matrix)\n")  # Limit of Blank (Matrix)
                    report_file_smpl.write("(ppm)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_lob_mat[var_filetype][var_datatype][file_short])

                    report_file_smpl.write("\n")

                    report_file_smpl.write("Limit of Detection (Matrix)\n")  # Limit of Detection (Matrix)
                    report_file_smpl.write("(ppm)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_lod_mat[var_filetype][var_datatype][file_short])

                    report_file_smpl.write("\n")

                    report_file_smpl.write("Limit of Quantification (Matrix)\n")  # Limit of Quantification (Matrix)
                    report_file_smpl.write("(ppm)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_loq_mat[var_filetype][var_datatype][file_short])

                    report_file_smpl.write("\n")

                    if var_filetype == "SMPL":
                        report_file_smpl.write("Composition (Mixed)\n")  # Concentration Mixed
                        report_file_smpl.write("(ppm)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_concentration_mix[var_filetype][var_datatype][file_short])

                        report_file_smpl.write("\n")

                    if var_filetype == "SMPL":
                        report_file_smpl.write("Mixing Ratio a\n")  # Mixing Ratio a
                        report_file_smpl.write("(1)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_mixingratio_a[var_filetype][var_datatype][file_short])

                        report_file_smpl.write("\n")

                        report_file_smpl.write("Mixing Ratio x\n")  # Mixing Ratio x
                        report_file_smpl.write("(1)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_mixingratio_x[var_filetype][var_datatype][file_short])

                        report_file_smpl.write("\n")

                    report_file_smpl.write("INTENSITY ANALYSIS\n")
                    if var_filetype == "SMPL":
                        report_file_smpl.write("Intensity (Inclusion)\n")  # Intensity Inclusion
                        report_file_smpl.write("(cps)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_intensity_incl[var_filetype][var_datatype][file_short])

                        report_file_smpl.write("\n")

                    report_file_smpl.write("Intensity (Matrix)\n")  # Intensity Matrix
                    report_file_smpl.write("(cps)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_intensity_mat[var_filetype][var_datatype][file_short])

                    report_file_smpl.write("\n")

                    report_file_smpl.write("Intensity (Background)\n")  # Intensity Background
                    report_file_smpl.write("(cps)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_intensity_bg[var_filetype][var_datatype][file_short])

                    report_file_smpl.write("\n")

                    if var_filetype == "SMPL":
                        report_file_smpl.write("Intensity (Mixed)\n")  # Intensity Mixed
                        report_file_smpl.write("(cps)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_intensity_mix[var_filetype][var_datatype][file_short])

                        report_file_smpl.write("\n")

                    report_file_smpl.write("SENSITIVITY ANALYSIS\n")
                    report_file_smpl.write("Analytical Sensitivity\n")  # Analytical Sensitivity
                    report_file_smpl.write("(1)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_analytical_sensitivity[var_filetype][var_datatype][file_short])

                    report_file_smpl.write("\n")

                    report_file_smpl.write("Normalized Sensitivity\n")  # Normalized Sensitivity
                    report_file_smpl.write("(cps)/(ppm)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_normalized_sensitivity[var_filetype][var_datatype][file_short])

                    report_file_smpl.write("\n")

                    report_file_smpl.write("Relative Sensitivity Factor\n")  # Relative Sensitivity Factor
                    report_file_smpl.write("(1)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_rsf[var_filetype][var_datatype][file_short])

                    report_file_smpl.write("\n")

        if os.path.exists(filename_base):
            os.remove(filename_base)
        else:
            print("The file does not exist!")

    def fi_export_report_2(
            self, report_concentration_incl, report_concentration_incl_1_sigma, report_concentration_mat,
            report_concentration_mat_1_sigma, report_concentration_mix, report_mixingratio_a,
            report_mixingratio_x, report_lob_incl, report_lob_mat, report_lod_incl, report_lod_mat, report_loq_incl,
            report_loq_mat, report_intensity_incl, report_intensity_mat, report_intensity_bg, report_intensity_mix,
            report_analytical_sensitivity, report_normalized_sensitivity, report_rsf, var_file_extension,
            var_delimiter, header):
        export_file = filedialog.asksaveasfile(mode="w", defaultextension=var_file_extension)
        filename_base = export_file.name
        filename_base_parts = filename_base.split(".")
        filename_raw = filename_base_parts[0] + "_raw" + var_file_extension
        filename_smoothed = filename_base_parts[0] + "_smoothed" + var_file_extension
        export_file.close()

        # RAW Data
        with open(filename_raw, "w", newline="") as report_file_raw:
            writer = csv.DictWriter(report_file_raw, fieldnames=header, delimiter=var_delimiter)
            report_file_raw.write("CALCULATION REPORT\n")
            report_file_raw.write("\n")
            report_file_raw.write("AUTHOR:;" + str(self.container_var["fi_setting"]["Author"].get()) + "\n")
            report_file_raw.write("SOURCE ID:;" + str(self.container_var["fi_setting"]["Source ID"].get()) + "\n")
            report_file_raw.write("\n")
            for var_datatype in ["RAW"]:
                report_file_raw.write("DATA TYPE:;" + str(var_datatype) + str(" DATA") + "\n")
                report_file_raw.write("\n")
                for var_filetype in ["SMPL", "STD"]:
                    if var_filetype == "SMPL":
                        report_file_raw.write("SAMPLE FILES\n")
                    elif var_filetype == "STD":
                        report_file_raw.write("STANDARD FILES\n")
                    var_key = "Total " + str(var_filetype)
                    report_file_raw.write("\n")

                    ## COMPOSITIONAL ANALYSIS
                    report_file_raw.write("COMPOSITIONAL ANALYSIS\n")
                    if var_filetype == "SMPL":
                        report_file_raw.write("Composition (Inclusion)\n")  # Concentration Inclusion
                        report_file_raw.write("(ppm)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_concentration_incl[var_filetype][var_datatype][file_short])

                        report_file_raw.write("\n")

                        report_file_raw.write("1-Sigma Composition (Inclusion)\n")  # Concentration Inclusion (1-Sigma)
                        report_file_raw.write("(ppm)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_concentration_incl_1_sigma[var_filetype][var_datatype][file_short])

                        report_file_raw.write("\n")

                        report_file_raw.write("Limit of Blank (Inclusion)\n")  # Limit of Blank (Inclusion)
                        report_file_raw.write("(ppm)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_lob_incl[var_filetype][var_datatype][file_short])

                        report_file_raw.write("\n")

                        report_file_raw.write("Limit of Detection (Inclusion)\n")  # Limit of Detection (Inclusion)
                        report_file_raw.write("(ppm)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_lod_incl[var_filetype][var_datatype][file_short])

                        report_file_raw.write("\n")

                        report_file_raw.write("Limit of Quantification (Inclusion)\n")  # Limit of Quantification (Inclusion)
                        report_file_raw.write("(ppm)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_loq_incl[var_filetype][var_datatype][file_short])

                        report_file_raw.write("\n")

                    report_file_raw.write("Composition (Matrix)\n")  # Concentration Matrix
                    report_file_raw.write("(ppm)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_concentration_mat[var_filetype][var_datatype][file_short])

                    report_file_raw.write("\n")

                    if var_filetype == "SMPL":
                        report_file_raw.write("1-Sigma Composition (Matrix)\n")  # Concentration Matrix (1-Sigma)
                        report_file_raw.write("(ppm)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_concentration_mat_1_sigma[var_filetype][var_datatype][file_short])

                        report_file_raw.write("\n")

                    report_file_raw.write("Limit of Blank (Matrix)\n")  # Limit of Blank (Matrix)
                    report_file_raw.write("(ppm)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_lob_mat[var_filetype][var_datatype][file_short])

                    report_file_raw.write("\n")

                    report_file_raw.write("Limit of Detection (Matrix)\n")  # Limit of Detection (Matrix)
                    report_file_raw.write("(ppm)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_lod_mat[var_filetype][var_datatype][file_short])

                    report_file_raw.write("\n")

                    report_file_raw.write("Limit of Quantification (Matrix)\n")  # Limit of Quantification (Matrix)
                    report_file_raw.write("(ppm)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_loq_mat[var_filetype][var_datatype][file_short])

                    report_file_raw.write("\n")

                    if var_filetype == "SMPL":
                        report_file_raw.write("Composition (Mixed)\n")  # Concentration Mixed
                        report_file_raw.write("(ppm)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_concentration_mix[var_filetype][var_datatype][file_short])

                        report_file_raw.write("\n")

                    if var_filetype == "SMPL":
                        report_file_raw.write("Mixing Ratio a\n")  # Mixing Ratio a
                        report_file_raw.write("(1)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_mixingratio_a[var_filetype][var_datatype][file_short])

                        report_file_raw.write("\n")

                        report_file_raw.write("Mixing Ratio x\n")  # Mixing Ratio x
                        report_file_raw.write("(1)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_mixingratio_x[var_filetype][var_datatype][file_short])

                        report_file_raw.write("\n")

                    report_file_raw.write("INTENSITY ANALYSIS\n")
                    if var_filetype == "SMPL":
                        report_file_raw.write("Intensity (Inclusion)\n")  # Intensity Inclusion
                        report_file_raw.write("(cps)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_intensity_incl[var_filetype][var_datatype][file_short])

                        report_file_raw.write("\n")

                    report_file_raw.write("Intensity (Matrix)\n")  # Intensity Matrix
                    report_file_raw.write("(cps)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_intensity_mat[var_filetype][var_datatype][file_short])

                    report_file_raw.write("\n")

                    report_file_raw.write("Intensity (Background)\n")  # Intensity Background
                    report_file_raw.write("(cps)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_intensity_bg[var_filetype][var_datatype][file_short])

                    report_file_raw.write("\n")

                    if var_filetype == "SMPL":
                        report_file_raw.write("Intensity (Mixed)\n")  # Intensity Mixed
                        report_file_raw.write("(cps)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_intensity_mix[var_filetype][var_datatype][file_short])

                        report_file_raw.write("\n")

                    report_file_raw.write("SENSITIVITY ANALYSIS\n")
                    report_file_raw.write("Analytical Sensitivity\n")  # Analytical Sensitivity
                    report_file_raw.write("(1)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_analytical_sensitivity[var_filetype][var_datatype][file_short])

                    report_file_raw.write("\n")

                    report_file_raw.write("Normalized Sensitivity\n")  # Normalized Sensitivity
                    report_file_raw.write("(cps)/(ppm)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_normalized_sensitivity[var_filetype][var_datatype][file_short])

                    report_file_raw.write("\n")

                    report_file_raw.write("Relative Sensitivity Factor\n")  # Relative Sensitivity Factor
                    report_file_raw.write("(1)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_rsf[var_filetype][var_datatype][file_short])

                    report_file_raw.write("\n")

        # SMOOTHED Data
        with open(filename_smoothed, "w", newline="") as report_file_smoothed:
            writer = csv.DictWriter(report_file_smoothed, fieldnames=header, delimiter=var_delimiter)
            report_file_smoothed.write("CALCULATION REPORT\n")
            report_file_smoothed.write("\n")
            report_file_smoothed.write("AUTHOR:;" + str(self.container_var["ma_setting"]["Author"].get()) + "\n")
            report_file_smoothed.write("SOURCE ID:;" + str(self.container_var["ma_setting"]["Source ID"].get()) + "\n")
            report_file_smoothed.write("\n")
            for var_datatype in ["SMOOTHED"]:
                report_file_smoothed.write("DATA TYPE:;" + str(var_datatype) + str(" DATA") + "\n")
                report_file_smoothed.write("\n")
                for var_filetype in ["SMPL", "STD"]:
                    if var_filetype == "SMPL":
                        report_file_smoothed.write("SAMPLE FILES\n")
                    elif var_filetype == "STD":
                        report_file_smoothed.write("STANDARD FILES\n")
                    var_key = "Total " + str(var_filetype)
                    report_file_smoothed.write("\n")

                    ## COMPOSITIONAL ANALYSIS
                    report_file_smoothed.write("COMPOSITIONAL ANALYSIS\n")
                    if var_filetype == "SMPL":
                        report_file_smoothed.write("Composition (Inclusion)\n")  # Concentration Inclusion
                        report_file_smoothed.write("(ppm)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_concentration_incl[var_filetype][var_datatype][file_short])

                        report_file_smoothed.write("\n")

                        report_file_smoothed.write("1-Sigma Composition (Inclusion)\n")  # Concentration Inclusion (1-Sigma)
                        report_file_smoothed.write("(ppm)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_concentration_incl_1_sigma[var_filetype][var_datatype][file_short])

                        report_file_smoothed.write("\n")

                        report_file_smoothed.write("Limit of Blank (Inclusion)\n")  # Limit of Blank (Inclusion)
                        report_file_smoothed.write("(ppm)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_lob_incl[var_filetype][var_datatype][file_short])

                        report_file_smoothed.write("\n")

                        report_file_smoothed.write("Limit of Detection (Inclusion)\n")  # Limit of Detection (Inclusion)
                        report_file_smoothed.write("(ppm)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_lod_incl[var_filetype][var_datatype][file_short])

                        report_file_smoothed.write("\n")

                        report_file_smoothed.write("Limit of Quantification (Inclusion)\n")  # Limit of Quantification (Inclusion)
                        report_file_smoothed.write("(ppm)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_loq_incl[var_filetype][var_datatype][file_short])

                        report_file_smoothed.write("\n")

                    report_file_smoothed.write("Composition (Matrix)\n")  # Concentration Matrix
                    report_file_smoothed.write("(ppm)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_concentration_mat[var_filetype][var_datatype][file_short])

                    report_file_smoothed.write("\n")

                    if var_filetype == "SMPL":
                        report_file_smoothed.write("1-Sigma Composition (Matrix)\n")  # Concentration Matrix (1-Sigma)
                        report_file_smoothed.write("(ppm)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_concentration_mat_1_sigma[var_filetype][var_datatype][file_short])

                        report_file_smoothed.write("\n")

                    report_file_smoothed.write("Limit of Blank (Matrix)\n")  # Limit of Blank (Matrix)
                    report_file_smoothed.write("(ppm)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_lob_mat[var_filetype][var_datatype][file_short])

                    report_file_smoothed.write("\n")

                    report_file_smoothed.write("Limit of Detection (Matrix)\n")  # Limit of Detection (Matrix)
                    report_file_smoothed.write("(ppm)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_lod_mat[var_filetype][var_datatype][file_short])

                    report_file_smoothed.write("\n")

                    report_file_smoothed.write("Limit of Quantification (Matrix)\n")  # Limit of Quantification (Matrix)
                    report_file_smoothed.write("(ppm)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_loq_mat[var_filetype][var_datatype][file_short])

                    report_file_smoothed.write("\n")

                    if var_filetype == "SMPL":
                        report_file_smoothed.write("Composition (Mixed)\n")  # Concentration Mixed
                        report_file_smoothed.write("(ppm)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_concentration_mix[var_filetype][var_datatype][file_short])

                        report_file_smoothed.write("\n")

                    if var_filetype == "SMPL":
                        report_file_smoothed.write("Mixing Ratio a\n")  # Mixing Ratio a
                        report_file_smoothed.write("(1)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_mixingratio_a[var_filetype][var_datatype][file_short])

                        report_file_smoothed.write("\n")

                        report_file_smoothed.write("Mixing Ratio x\n")  # Mixing Ratio x
                        report_file_smoothed.write("(1)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_mixingratio_x[var_filetype][var_datatype][file_short])

                        report_file_smoothed.write("\n")

                    report_file_smoothed.write("INTENSITY ANALYSIS\n")
                    if var_filetype == "SMPL":
                        report_file_smoothed.write("Intensity (Inclusion)\n")  # Intensity Inclusion
                        report_file_smoothed.write("(cps)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_intensity_incl[var_filetype][var_datatype][file_short])

                        report_file_smoothed.write("\n")

                    report_file_smoothed.write("Intensity (Matrix)\n")  # Intensity Matrix
                    report_file_smoothed.write("(cps)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_intensity_mat[var_filetype][var_datatype][file_short])

                    report_file_smoothed.write("\n")

                    report_file_smoothed.write("Intensity (Background)\n")  # Intensity Background
                    report_file_smoothed.write("(cps)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_intensity_bg[var_filetype][var_datatype][file_short])

                    report_file_smoothed.write("\n")

                    if var_filetype == "SMPL":
                        report_file_smoothed.write("Intensity (Mixed)\n")  # Intensity Mixed
                        report_file_smoothed.write("(cps)\n")
                        writer.writeheader()

                        for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                            file_long = self.container_lists[var_filetype]["Long"][index]
                            if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                                writer.writerow(report_intensity_mix[var_filetype][var_datatype][file_short])

                        report_file_smoothed.write("\n")

                    report_file_smoothed.write("SENSITIVITY ANALYSIS\n")
                    report_file_smoothed.write("Analytical Sensitivity\n")  # Analytical Sensitivity
                    report_file_smoothed.write("(1)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_analytical_sensitivity[var_filetype][var_datatype][file_short])

                    report_file_smoothed.write("\n")

                    report_file_smoothed.write("Normalized Sensitivity\n")  # Normalized Sensitivity
                    report_file_smoothed.write("(cps)/(ppm)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_normalized_sensitivity[var_filetype][var_datatype][file_short])

                    report_file_smoothed.write("\n")

                    report_file_smoothed.write("Relative Sensitivity Factor\n")  # Relative Sensitivity Factor
                    report_file_smoothed.write("(1)\n")
                    writer.writeheader()

                    for index, file_short in enumerate(self.container_lists[var_filetype]["Short"]):
                        file_long = self.container_lists[var_filetype]["Long"][index]
                        if self.container_var[var_filetype][file_long]["Checkbox"].get() == 1:
                            writer.writerow(report_rsf[var_filetype][var_datatype][file_short])

                    report_file_smoothed.write("\n")

        if os.path.exists(filename_base):
            os.remove(filename_base)
        else:
            print("The file does not exist!")

    def save_project(self):
        file_ending_pre = self.container_var["General Settings"]["Project type"].get()
        file_ending = file_ending_pre[1:]

        if "csv" in file_ending:
            list_file_endings = [("CSV files *.csv", "*.csv"), ("Project files *.proj", "*.proj")]
        elif "proj" in file_ending:
            list_file_endings = [("Project files *.proj", "*.proj"), ("CSV files *.csv", "*.csv")]

        save_file = filedialog.asksaveasfile(
            mode="w", defaultextension=file_ending, title="Save as", filetypes=list_file_endings)

        # Save information about the project
        self.save_project_information_in_file(save_file=save_file)
        # Save information about 'Standard Files Setup'
        self.save_standard_file_information_in_file(save_file=save_file)
        # Save information about 'Sample Files Setup'
        self.save_sample_file_information_in_file(save_file=save_file)
        # Save information about 'Measured Isotopes'
        self.save_measured_isotopes_in_file(save_file=save_file)

        if self.pysills_mode in ["FI", "MI", "INCL"]:
            # Save information about 'Inclusion Setup'
            self.save_inclusion_information_in_file(save_file=save_file)
            if self.pysills_mode in ["FI", "INCL"]:
                # Save information about 'PyPitzer'
                self.save_pypitzer_settings_in_file(save_file=save_file)

            # Save information about 'Quantification Setup (Matrix-Only Tracer)'
            self.save_quantification_method_matrix_only_in_file(save_file=save_file)
            # Save information about 'Quantification Setup (Second Internal Standard)'
            self.save_quantification_method_second_internal_in_file(save_file=save_file)
            if self.pysills_mode in ["FI", "INCL"]:
                # Save information about 'Geometric Approach (Halter2002)'
                self.save_quantification_method_halter2002(save_file=save_file)
                # Save information about 'Geometric Approach (Borisova2021)'
                self.save_quantification_method_borisova2021(save_file=save_file)

        # Save information about 'Sample/Matrix Setup'
        self.save_mineralphase_information_in_file(save_file=save_file)
        # Save information about 'Dwell Time Settings'
        self.save_dwell_time_information_in_file(save_file=save_file)
        # Save information about 'Interval Settings'
        self.save_interval_information_in_file(save_file=save_file)
        # Save information about 'Spike Elimination'
        self.save_spike_elimination_information_in_file(save_file=save_file)
        # Save information about the experimental input data
        self.save_experimental_data_in_file(save_file=save_file)
        # END
        save_file.write("END")
        save_file.close()

    def save_project_information_in_file(self, save_file):
        if self.pysills_mode == "MA":
            info_mode = "Mineral Analysis"
            info_key = "ma_setting"
        elif self.pysills_mode in "FI":
            info_mode = "Fluid Inclusion Analysis"
            info_key = "fi_setting"
        elif self.pysills_mode == "MI":
            info_mode = "Melt Inclusion Analysis"
            info_key = "mi_setting"
        elif self.pysills_mode == "INCL":
            info_mode = "Inclusion Analysis"
            info_key = "fi_setting"

        save_file.write("PROJECT INFORMATION" + "\n")
        save_file.write(str(info_mode) + "\n")

        info_author = self.container_var[info_key]["Author"].get()
        info_source = self.container_var[info_key]["Source ID"].get()
        info_carrier_gas = self.container_var["LASER"].get()
        info_icpms = self.var_opt_icp.get()

        str_proj = str(info_author) + ";" + str(info_source) + ";" + str(info_carrier_gas) + ";" + str(info_icpms)
        str_proj += "\n"

        save_file.write(str_proj)
        save_file.write("\n")

    def save_standard_file_information_in_file(self, save_file):
        save_file.write("STANDARD FILES" + "\n")

        for index, filename_short in enumerate(self.container_lists["STD"]["Short"]):
            filename_long = self.container_lists["STD"]["Long"][index]

            if filename_long in self.container_var["STD"]:
                info_srm = self.container_var["STD"][filename_long]["SRM"].get()
                info_cb_state = self.container_var["STD"][filename_long]["Checkbox"].get()
                info_sign_color = self.container_var["STD"][filename_long]["Sign Color"].get()
                info_acquisition = self.container_var["acquisition times"]["STD"][filename_short].get()
            else:
                info_srm = "Select SRM"
                info_cb_state = 1
                info_sign_color = self.sign_red
                info_acquisition = "00:00:00"

            str_std = (str(filename_short) + ";" + str(info_srm) + ";" + str(info_cb_state) + ";" +
                       str(info_sign_color) + ";" + str(info_acquisition))

            str_std += "\n"
            save_file.write(str_std)
        save_file.write("\n")

    def save_sample_file_information_in_file(self, save_file):
        save_file.write("SAMPLE FILES" + "\n")

        for index, filename_short in enumerate(self.container_lists["SMPL"]["Short"]):
            filename_long = self.container_lists["SMPL"]["Long"][index]
            if filename_long in self.container_var["SMPL"]:
                info_is = self.container_var["SMPL"][filename_long]["IS Data"]["IS"].get()
                info_assemblage = self.container_var["SMPL"][filename_long]["ID"].get()
                info_cb_state = self.container_var["SMPL"][filename_long]["Checkbox"].get()
                info_sign_color = self.container_var["SMPL"][filename_long]["Sign Color"].get()
                info_acquisition = self.container_var["acquisition times"]["SMPL"][filename_short].get()
                info_oxide = self.container_var["SMPL"][filename_long]["Matrix Setup"]["Oxide"]["Name"].get()
                info_totalamount = self.container_var["Oxides Quantification"]["Total Amounts"][filename_short].get()
                info_totalamount_incl = self.container_var["Oxides Quantification INCL"]["Total Amounts"][
                    filename_short].get()
            else:
                info_is = "Select IS"
                info_assemblage = "A"
                info_cb_state = 1
                info_sign_color = self.sign_red
                info_acquisition = "00:00:00"
                info_oxide = "Select Oxide"
                info_totalamount = "100.0"
                info_totalamount_incl = "100.0"

            str_smpl = (str(filename_short) + ";" + str(info_is) + ";" + str(info_assemblage) + ";" + str(info_cb_state)
                        + ";" + str(info_sign_color) + ";" + str(info_acquisition) + ";" + str(info_oxide) + ";"
                        + str(info_totalamount) + ";" + str(info_totalamount_incl))

            str_smpl += "\n"
            save_file.write(str_smpl)
        save_file.write("\n")

    def save_measured_isotopes_in_file(self, save_file):
        save_file.write("ISOTOPES" + "\n")

        for isotope in self.container_lists["Measured Isotopes"]["All"]:
            if isotope.isdigit():
                pass
            else:
                info_srm = self.container_var["SRM"][isotope].get()

                str_iso = str(isotope) + ";" + str(info_srm)
                str_iso += "\n"

                save_file.write(str_iso)

        save_file.write("\n")

        if len(self.container_lists["Selected Oxides"]["All"]) > 0:
            for oxide in self.container_lists["Selected Oxides"]["All"]:
                str_oxide = str(oxide)
                str_oxide += "\n"

                save_file.write(str_oxide)

            save_file.write("\n")

    def save_inclusion_information_in_file(self, save_file):
        if self.pysills_mode == "MA":
            key_setting = "ma_setting"
        elif self.pysills_mode in ["FI", "INCL"]:
            key_setting = "fi_setting"
        elif self.pysills_mode == "MI":
            key_setting = "mi_setting"

        save_file.write("INCLUSION SETTINGS" + "\n")

        info_method = self.container_var[key_setting]["Inclusion Setup Selection"].get()

        str_incl = str("Method") + ";" + str(info_method) + "\n"
        save_file.write(str_incl)
        for index, filename_short in enumerate(self.container_lists["SMPL"]["Short"]):
            filename_long = self.container_lists["SMPL"]["Long"][index]
            info_is = self.container_var["SMPL"][filename_long]["IS Data"]["IS"].get()
            info_concentration = self.container_var["SMPL"][filename_long]["IS Data"]["Concentration"].get()
            if filename_short in self.helper_lists["Salinity"]:
                info_salinity = self.helper_lists["Salinity"][filename_short]
            else:
                info_salinity = 0.0
            if filename_short in self.helper_lists["Considered chlorides"]:
                info_chlorides = self.helper_lists["Considered chlorides"][filename_short]
            else:
                info_chlorides = []

            str_incl = (str(filename_short) + ";" + str(info_is) + ";" + str(info_concentration) + ";"
                        + str(info_salinity) + ";" + str(info_chlorides))

            str_incl += "\n"
            save_file.write(str_incl)

        save_file.write("\n")

    def save_pypitzer_settings_in_file(self, save_file):
        save_file.write("PYPITZER SETTINGS" + "\n")

        str_cations = "Cations" + ";"
        str_anions = "Anions" + ";"
        str_isotopes = "Isotopes" + ";"

        for index, cation in enumerate(self.container_lists["Selected Cations"]):
            if index < len(self.container_lists["Selected Cations"]) - 1:
                str_cations += cation + ";"
            else:
                str_cations += cation + "\n"

        for index, anion in enumerate(self.container_lists["Selected Anions"]):
            if index < len(self.container_lists["Selected Anions"]) - 1:
                str_anions += anion + ";"
            else:
                str_anions += anion + "\n"

        for index, isotope in enumerate(list(self.helper_checkbuttons["Isotopes"].keys())):
            if self.helper_checkbuttons["Isotopes"][isotope].get() == 1:
                if index < len(list(self.helper_checkbuttons["Isotopes"].keys())) - 1:
                    str_isotopes += isotope + ";"
                else:
                    str_isotopes += isotope + "\n"

        str_isotopes = str_isotopes[:-1]
        str_isotopes += "\n"

        save_file.write(str_cations)
        save_file.write(str_anions)
        save_file.write(str_isotopes)

        for index, filename_smpl_long in enumerate(self.container_lists["SMPL"]["Long"]):
            filename_smpl_short = self.container_lists["SMPL"]["Short"][index]
            try:
                var_last_compound = self.container_var["SMPL"][filename_smpl_long]["Last compound"].get()
                var_melting_temperature = self.container_var["SMPL"][filename_smpl_long]["Melting temperature"].get()
            except:
                var_last_compound = "Undefined"
                var_melting_temperature = "Undefined"
            str_file = filename_smpl_short + ";" + var_last_compound + ";" + var_melting_temperature + "\n"

            save_file.write(str_file)

        save_file.write("\n")

    def save_quantification_method_matrix_only_in_file(self, save_file):
        if self.pysills_mode == "MA":
            key_setting = "ma_setting"
        elif self.pysills_mode in ["FI", "INCL"]:
            key_setting = "fi_setting"
        elif self.pysills_mode == "MI":
            key_setting = "mi_setting"

        save_file.write("QUANTIFICATION SETTINGS (MATRIX-ONLY TRACER)" + "\n")

        info_method = self.container_var[key_setting]["Quantification Method"].get()
        info_method_incl = self.container_var[key_setting]["Inclusion Intensity Calculation"].get()

        str_method = str("Method") + ";" + str(info_method) + ";" + str(info_method_incl) + "\n"
        save_file.write(str_method)
        for index, filename_short in enumerate(self.container_lists["SMPL"]["Short"]):
            filename_long = self.container_lists["SMPL"]["Long"][index]
            try:
                info_amount = self.container_var["SMPL"][filename_long]["Host Only Tracer"]["Amount"].get()
                info_matrix = self.container_var["SMPL"][filename_long]["Host Only Tracer"]["Matrix"].get()
                info_isotope = self.container_var["SMPL"][filename_long]["Host Only Tracer"]["Name"].get()
                info_concentration = self.container_var["SMPL"][filename_long]["Host Only Tracer"]["Value"].get()
            except:
                info_amount = 0.0
                info_matrix = "Select Matrix"
                info_isotope = "Select Isotope"
                info_concentration = 0.0

            str_method = (str(filename_short) + ";" + str(info_amount) + ";" + str(info_matrix) + ";"
                          + str(info_isotope) + ";" + str(info_concentration) + "\n")

            save_file.write(str_method)

        save_file.write("\n")

    def save_quantification_method_second_internal_in_file(self, save_file):
        if self.pysills_mode == "MA":
            key_setting = "ma_setting"
        elif self.pysills_mode in ["FI", "INCL"]:
            key_setting = "fi_setting"
        elif self.pysills_mode == "MI":
            key_setting = "mi_setting"

        save_file.write("QUANTIFICATION SETTINGS (SECOND INTERNAL STANDARD)" + "\n")

        info_method = self.container_var[key_setting]["Quantification Method"].get()
        info_method_incl = self.container_var[key_setting]["Inclusion Intensity Calculation"].get()

        str_method = str("Method") + ";" + str(info_method) + ";" + str(info_method_incl) + "\n"
        save_file.write(str_method)
        for index, filename_short in enumerate(self.container_lists["SMPL"]["Short"]):
            filename_long = self.container_lists["SMPL"]["Long"][index]
            try:
                info_isotope = self.container_var["SMPL"][filename_long]["Second Internal Standard"]["Name"].get()
                info_concentration = self.container_var["SMPL"][filename_long]["Second Internal Standard"][
                    "Value"].get()
            except:
                info_isotope = "Select Isotope"
                info_concentration = 0.0
            str_method = str(filename_short) + ";" + str(info_isotope) + ";" + str(info_concentration) + "\n"
            save_file.write(str_method)

        save_file.write("\n")

    def save_quantification_method_halter2002(self, save_file):
        if self.pysills_mode == "MA":
            key_setting = "ma_setting"
        elif self.pysills_mode in ["FI", "INCL"]:
            key_setting = "fi_setting"
        elif self.pysills_mode == "MI":
            key_setting = "mi_setting"

        save_file.write("QUANTIFICATION SETTINGS (HALTER2002)" + "\n")
        info_method = self.container_var[key_setting]["Inclusion Intensity Calculation"].get()
        str_method = str("Method") + ";" + str(info_method) + "\n"
        save_file.write(str_method)

        for index, filename_short in enumerate(self.container_lists["SMPL"]["Short"]):
            filename_long = self.container_lists["SMPL"]["Long"][index]

            try:
                val_a_i = self.container_var["SMPL"][filename_long]["Halter2002"]["a"].get()
                val_b_i = self.container_var["SMPL"][filename_long]["Halter2002"]["b"].get()
                val_rho_host_i = self.container_var["SMPL"][filename_long]["Halter2002"]["rho(host)"].get()
                val_rho_incl_i = self.container_var["SMPL"][filename_long]["Halter2002"]["rho(incl)"].get()
                val_r_i = self.container_var["SMPL"][filename_long]["Halter2002"]["R"].get()
            except:
                val_a_i = "Undefined"
                val_b_i = "Undefined"
                val_rho_host_i = "Undefined"
                val_rho_incl_i = "Undefined"
                val_r_i = "Undefined"

            str_i = (str(filename_short) + ";" + str(val_a_i) + ";" + str(val_b_i) + ";" + str(val_rho_host_i) +
                     ";" + str(val_rho_incl_i) + ";" + str(val_r_i) + "\n")
            save_file.write(str_i)

        save_file.write("\n")

    def save_quantification_method_borisova2021(self, save_file):
        if self.pysills_mode == "MA":
            key_setting = "ma_setting"
        elif self.pysills_mode in ["FI", "INCL"]:
            key_setting = "fi_setting"
        elif self.pysills_mode == "MI":
            key_setting = "mi_setting"

        save_file.write("QUANTIFICATION SETTINGS (BORISOVA2021)" + "\n")
        info_method = self.container_var[key_setting]["Inclusion Intensity Calculation"].get()
        str_method = str("Method") + ";" + str(info_method) + "\n"
        save_file.write(str_method)

        for index, filename_short in enumerate(self.container_lists["SMPL"]["Short"]):
            filename_long = self.container_lists["SMPL"]["Long"][index]

            try:
                val_r_host_i = self.container_var["SMPL"][filename_long]["Borisova2021"]["R(host)"].get()
                val_r_incl_i = self.container_var["SMPL"][filename_long]["Borisova2021"]["R(incl)"].get()
                val_rho_host_i = self.container_var["SMPL"][filename_long]["Borisova2021"]["rho(host)"].get()
                val_rho_incl_i = self.container_var["SMPL"][filename_long]["Borisova2021"]["rho(incl)"].get()
            except:
                val_r_host_i = "Undefined"
                val_r_incl_i = "Undefined"
                val_rho_host_i = "Undefined"
                val_rho_incl_i = "Undefined"

            str_i = (str(filename_short) + ";" + str(val_r_host_i) + ";" + str(val_r_incl_i) + ";" +
                     str(val_rho_host_i) + ";" + str(val_rho_incl_i) + "\n")
            save_file.write(str_i)

        save_file.write("\n")

    def save_mineralphase_information_in_file(self, save_file):
        if self.pysills_mode == "MA":
            info_title = "SAMPLE SETTINGS"
        elif self.pysills_mode in ["FI", "INCL"]:
            info_title = "MATRIX SETTINGS"
        elif self.pysills_mode == "MI":
            info_title = "MATRIX SETTINGS"

        save_file.write(str(info_title) + "\n")
        for index, filename_short in enumerate(self.container_lists["SMPL"]["Short"]):
            filename_long = self.container_lists["SMPL"]["Long"][index]
            if filename_long in self.container_var["SMPL"]:
                if self.pysills_mode == "MA":
                    info_is = self.container_var["SMPL"][filename_long]["IS Data"]["IS"].get()
                    info_concentration = self.container_var["SMPL"][filename_long]["IS Data"]["Concentration"].get()
                else:
                    info_is = self.container_var["SMPL"][filename_long]["Matrix Setup"]["IS"]["Name"].get()
                    info_concentration = self.container_var["SMPL"][filename_long]["Matrix Setup"]["IS"][
                        "Concentration"].get()
            else:
                info_is = "Select IS"
                info_concentration = "1000000"

            str_smpl_is = str(filename_short) + ";" + str(info_is) + ";" + str(info_concentration) + "\n"

            save_file.write(str_smpl_is)

        save_file.write("\n")

    def save_dwell_time_information_in_file(self, save_file):
        save_file.write("DWELL TIME SETTINGS" + "\n")

        for isotope in self.container_lists["Measured Isotopes"]["All"]:
            if isotope.isdigit():
                pass
            else:
                info_dwell = self.container_var["dwell_times"]["Entry"][isotope].get()
                str_dwell = str(isotope) + ";" + str(info_dwell) + "\n"
                save_file.write(str_dwell)

        save_file.write("\n")

    def save_interval_information_in_file(self, save_file):
        save_file.write("INTERVAL SETTINGS" + "\n")

        for index, filename_short in enumerate(self.container_lists["STD"]["Short"]):
            str_intervals = str(filename_short) + ";" + "STD" + "\n"
            if filename_short in self.container_helper["STD"]:
                for key, item in self.container_helper["STD"][filename_short]["BG"]["Content"].items():
                    info_id = key
                    info_times = item["Times"]
                    info_indices = item["Indices"]

                    str_intervals += "BG" + ";" + str(info_id) + ";" + str(info_times) + ";" + str(info_indices) + "\n"

                for key, item in self.container_helper["STD"][filename_short]["MAT"]["Content"].items():
                    info_id = key
                    info_times = item["Times"]
                    info_indices = item["Indices"]

                    str_intervals += "MAT" + ";" + str(info_id) + ";" + str(info_times) + ";" + str(info_indices) + "\n"

                save_file.write(str_intervals)

        for index, filename_short in enumerate(self.container_lists["SMPL"]["Short"]):
            str_intervals = str(filename_short) + ";" + "SMPL" + "\n"

            if filename_short in self.container_helper["SMPL"]:
                for key, item in self.container_helper["SMPL"][filename_short]["BG"]["Content"].items():
                    info_id = key
                    info_times = item["Times"]
                    info_indices = item["Indices"]

                    str_intervals += "BG" + ";" + str(info_id) + ";" + str(info_times) + ";" + str(info_indices) + "\n"

                for key, item in self.container_helper["SMPL"][filename_short]["MAT"]["Content"].items():
                    info_id = key
                    info_times = item["Times"]
                    info_indices = item["Indices"]

                    str_intervals += "MAT" + ";" + str(info_id) + ";" + str(info_times) + ";" + str(info_indices) + "\n"

                if self.pysills_mode in ["FI", "MI", "INCL"]:
                    for key, item in self.container_helper["SMPL"][filename_short]["INCL"]["Content"].items():
                        info_id = key
                        info_times = item["Times"]
                        info_indices = item["Indices"]

                        str_intervals += "INCL" + ";" + str(info_id) + ";" + str(info_times) + ";" + str(
                            info_indices) + "\n"

                save_file.write(str_intervals)

        save_file.write("\n")

    def save_spike_elimination_information_in_file(self, save_file):
        if self.pysills_mode == "MA":
            mode_key = "ma_setting"
        elif self.pysills_mode in ["FI", "INCL"]:
            mode_key = "fi_setting"
        elif self.pysills_mode == "MI":
            mode_key = "mi_setting"

        save_file.write("SPIKE ELIMINATION" + "\n")

        info_std_state = self.container_var["Spike Elimination"]["STD"]["State"]
        info_smpl_state = self.container_var["Spike Elimination"]["SMPL"]["State"]
        info_method = self.container_var["Spike Elimination Method"].get()
        info_alpha = self.container_var[mode_key]["SE Alpha"].get()
        info_threshold = self.container_var[mode_key]["SE Threshold"].get()

        if self.pysills_mode == "MA":
            str_spike = (str("STD") + ";" + str(info_std_state) + ";" + str("SMPL") + ";" + str(info_smpl_state) + ";"
                         + str(info_method) + ";" + str(info_alpha) + ";" + str(info_threshold) + "\n")
        else:
            info_inclusion_consideration = self.container_var[mode_key]["Spike Elimination Inclusion"].get()

            str_spike = (str("STD") + ";" + str(info_std_state) + ";" + str("SMPL") + ";" + str(info_smpl_state) + ";"
                         + str(info_inclusion_consideration) + ";" + str(info_method) + ";" + str(info_alpha) + ";"
                         + str(info_threshold) + "\n")

        save_file.write(str_spike)

        # for filetype in ["STD", "SMPL"]:
        #     for file_short in self.container_lists[filetype]["Short"]:
        #         df_isotopes = self.container_lists["Measured Isotopes"][file_short]
        #         if file_short in self.container_spike_values:
        #             for isotope in df_isotopes:
        #                 print(isotope)
        #                 print(len(self.container_spike_values[file_short][isotope]["Save"]),
        #                       self.container_spike_values[file_short][isotope]["Save"])

        for var_file, dataset_isotopes in self.container_spike_values.items():
            str_spike_file = var_file + "\n"
            save_file.write(str_spike_file)
            for var_isotope, dataset_values in dataset_isotopes.items():
                str_spike_isotope = str(var_isotope) + ";"
                for var_id, val_id in dataset_values["Save"].items():
                    str_spike_isotope += str(var_id) + ";" + str(val_id) + ";"
                str_spike_isotope = str_spike_isotope[:-1]
                str_spike_isotope += "\n"
                save_file.write(str_spike_isotope)

        save_file.write("\n")

    def save_experimental_data_in_file(self, save_file):
        save_file.write("EXPERIMENTAL DATA" + "\n")

        for filetype in ["STD", "SMPL"]:
            if filetype == "STD":
                save_file.write("Standard Files" + "\n")
            else:
                save_file.write("Sample Files" + "\n")
            for index, filename_short in enumerate(self.container_lists[filetype]["Short"]):
                if filename_short in self.container_var["acquisition times"][filetype]:
                    info_acquisition = self.container_var["acquisition times"][filetype][filename_short].get()
                    filename_long = self.container_lists[filetype]["Long"][index]
                    save_file.write(str(filename_short) + "\n")
                    save_file.write("Acquisition" + ";" + str(info_acquisition) + "\n")
                    file_header = "Time" + ";"
                    file_isotopes = self.container_lists["Measured Isotopes"][filename_short]
                    for isotope in file_isotopes:
                        file_header += str(isotope) + ";"
                    save_file.write(str(file_header) + "\n")
                    
                    if self.file_loaded == False:
                        if self.container_icpms["name"] != None:
                            var_skipheader = self.container_icpms["skipheader"]
                            var_skipfooter = self.container_icpms["skipfooter"]
                            df_data = DE(filename_long=filename_long).get_measurements(
                                delimiter=",", skip_header=var_skipheader, skip_footer=var_skipfooter)
                        else:
                            df_data = DE(filename_long=filename_long).get_measurements(
                                delimiter=",", skip_header=3, skip_footer=1)
                    else:
                        df_data = self.container_measurements["Dataframe"][filename_short]

                    dataset_time = list(DE().get_times(dataframe=df_data))

                    for index_line, time_value in enumerate(dataset_time):
                        file_content_line = str(time_value) + ";"
                        for isotope in file_isotopes:
                            file_content_line += str(df_data[isotope][index_line]) + ";"
                        save_file.write(str(file_content_line) + "\n")
                    save_file.write("\n")

    def open_project_part_01(self, key_setting, index_container, loaded_lines):
        ## PROJECT INFORMATION
        for i in range(index_container["PROJECT INFORMATION"] + 2, index_container["STANDARD FILES"] - 1):
            line_std = str(loaded_lines[i].strip())
            splitted_lines = line_std.split(";")

            if len(splitted_lines[0]) != 0:
                if len(splitted_lines) == 1:
                    splitted_lines = line_std.split(",")

                self.container_var[key_setting]["Author"].set(splitted_lines[0])

                if len(splitted_lines) > 1:
                    self.container_var[key_setting]["Source ID"].set(splitted_lines[1])
                    self.container_var["LASER"].set(splitted_lines[2])
                    try:
                        self.var_opt_icp.set(splitted_lines[3])
                    except:
                        self.var_opt_icp.set("Select ICP-MS")

    def open_project_part_02(self, key_setting, index_container, loaded_lines):
        ## STANDARD FILES
        for i in range(index_container["STANDARD FILES"] + 1, index_container["SAMPLE FILES"] - 1):
            line_std = str(loaded_lines[i].strip())
            splitted_lines = line_std.split(";")

            if len(splitted_lines[0]) != 0:
                if len(splitted_lines) == 1:
                    splitted_lines = line_std.split(",")

                var_file_long = splitted_lines[0]
                var_file_short = splitted_lines[0].split("/")[-1]

                self.lbl_prg_spk.configure(text=str(20) +" % - " + var_file_short, anchor=tk.W)

                self.list_std.append(var_file_long)
                self.container_lists["STD"]["Long"].append(var_file_long)
                self.container_lists["STD"]["Short"].append(var_file_short)

                self.container_spikes[var_file_short] = {}

                self.container_files["STD"][var_file_short] = {"SRM": tk.StringVar()}
                self.container_var["STD"][var_file_long] = {
                    "IS Data": {"IS": tk.StringVar(), "Concentration": tk.StringVar()},
                    "Checkbox": tk.IntVar(), "Sign Color": tk.StringVar(), "SRM": tk.StringVar()}

                self.container_var["acquisition times"]["STD"][var_file_short] = tk.StringVar()

                if self.old_file == True:
                    df_exmpl, file_info = self.find_icpms_data_in_file(filename_long=var_file_long, filetype="STD",
                                                                       only_time_check=True)

                self.lb_std.insert(tk.END, str(var_file_short))
                self.container_var["STD"][var_file_long]["SRM"].set(splitted_lines[1])
                self.container_var["STD"][var_file_long]["Checkbox"].set(splitted_lines[2])

                try:
                    self.container_var["STD"][var_file_long]["Sign Color"].set(splitted_lines[3])
                except:
                    self.container_var["STD"][var_file_long]["Sign Color"].set(self.sign_red)

                if len(splitted_lines) > 4:
                    self.container_var["acquisition times"]["STD"][var_file_short].set(splitted_lines[4])

                analysis_mode = key_setting[:2]
                if analysis_mode == "ma":
                    self.ma_current_file_std = self.list_std[0]
                elif analysis_mode == "fi":
                    self.fi_current_file_std = self.list_std[0]
                elif analysis_mode == "mi":
                    self.mi_current_file_std = self.list_std[0]

    def open_project_part_03(self, key_setting, index_container, loaded_lines):
        ## SAMPLE FILES
        analysis_mode = key_setting[:2]
        for i in range(index_container["SAMPLE FILES"] + 1, index_container["ISOTOPES"] - 1):
            line_smpl = str(loaded_lines[i].strip())
            splitted_lines = line_smpl.split(";")

            if len(splitted_lines[0]) != 0:
                if len(splitted_lines) == 1:
                    splitted_lines = line_smpl.split(",")

                var_file_long = splitted_lines[0]
                var_file_short = splitted_lines[0].split("/")[-1]

                self.lbl_prg_spk.configure(text=str(30) +" % - " + var_file_short, anchor=tk.W)

                self.container_spikes[var_file_short] = {}

                self.container_var["SMPL"][var_file_long] = {
                    "IS Data": {"IS": tk.StringVar(), "Concentration": tk.StringVar()},
                    "Checkbox": tk.IntVar(), "ID": tk.StringVar(), "Sign Color": tk.StringVar()}
                self.container_var["SMPL"][var_file_long]["Matrix Setup"] = {
                    "IS": {"Name": tk.StringVar(), "Concentration": tk.StringVar()},
                    "Oxide": {"Name": tk.StringVar(), "Concentration": tk.StringVar()},
                    "Element": {"Name": tk.StringVar(), "Concentration": tk.StringVar()}}

                self.container_var["acquisition times"]["SMPL"][var_file_short] = tk.StringVar()
                self.container_var["Oxides Quantification"]["Total Amounts"][var_file_short] = tk.StringVar()
                self.container_var["Oxides Quantification INCL"]["Total Amounts"][var_file_short] = tk.StringVar()

                self.lb_smpl.insert(tk.END, str(var_file_short))
                self.list_smpl.append(var_file_long)
                self.container_lists["SMPL"]["Long"].append(var_file_long)
                self.container_lists["SMPL"]["Short"].append(var_file_short)
                self.container_var["SMPL"][var_file_long]["Checkbox"].get()

                if self.old_file == True:
                    df_exmpl, file_info = self.find_icpms_data_in_file(filename_long=var_file_long, filetype="SMPL",
                                                                       only_time_check=True)

                if analysis_mode == "ma":
                    self.container_var["SMPL"][var_file_long]["Matrix Setup"]["IS"]["Name"].set("Select IS")
                    self.container_var["SMPL"][var_file_long]["Matrix Setup"]["IS"]["Concentration"].set("0.0")
                    self.container_var["SMPL"][var_file_long]["Matrix Setup"]["Oxide"]["Name"].set("Select Oxide")
                    self.container_var["SMPL"][var_file_long]["Matrix Setup"]["Oxide"]["Concentration"].set("100.0")
                    self.container_var["SMPL"][var_file_long]["Matrix Setup"]["Element"]["Name"].set("Select Element")
                    self.container_var["SMPL"][var_file_long]["Matrix Setup"]["Element"]["Concentration"].set("100.0")
                elif analysis_mode in ["fi", "mi"]:
                    self.container_var["SMPL"][var_file_long]["IS Data"] = {
                        "IS": tk.StringVar(), "Concentration": tk.StringVar(),
                        "RAW": {"IS": tk.StringVar(), "Concentration": tk.StringVar()},
                        "SMOOTHED": {"IS": tk.StringVar(), "Concentration": tk.StringVar()}}

                    self.container_var["SMPL"][var_file_long]["IS Data"]["IS"].set("Select IS")
                    self.container_var["SMPL"][var_file_long]["IS Data"]["Concentration"].set("0.0")
                    self.container_var["SMPL"][var_file_long]["IS Data"]["RAW"]["IS"].set("Select IS")
                    self.container_var["SMPL"][var_file_long]["IS Data"]["RAW"]["Concentration"].set("0.0")
                    self.container_var["SMPL"][var_file_long]["IS Data"]["SMOOTHED"]["IS"].set("Select IS")
                    self.container_var["SMPL"][var_file_long]["IS Data"]["SMOOTHED"]["Concentration"].set("0.0")

                self.container_var["SMPL"][var_file_long]["IS Data"]["IS"].set(splitted_lines[1])

                if analysis_mode == "ma":
                    self.ma_current_file_smpl = self.list_smpl[0]
                    self.container_var["SMPL"][var_file_long]["Matrix Setup"]["IS"]["Name"].set(splitted_lines[1])
                elif analysis_mode == "fi":
                    self.fi_current_file_smpl = self.list_smpl[0]
                elif analysis_mode == "mi":
                    self.mi_current_file_smpl = self.list_smpl[0]

                self.container_var["SMPL"][var_file_long]["ID"].set(splitted_lines[2])
                self.container_var["SMPL"][var_file_long]["Checkbox"].set(splitted_lines[3])
                self.container_var["SMPL"][var_file_long]["Sign Color"].set(splitted_lines[4])

                if len(splitted_lines) > 5:
                    self.container_var["acquisition times"]["SMPL"][var_file_short].set(splitted_lines[5])

                    if len(splitted_lines[6]) > 0:
                        self.container_var["SMPL"][var_file_long]["Matrix Setup"]["Oxide"]["Name"].set(
                            splitted_lines[6])
                    else:
                        self.container_var["SMPL"][var_file_long]["Matrix Setup"]["Oxide"]["Name"].set("Select Oxide")

                    if len(splitted_lines[7]) > 0:
                        self.container_var["Oxides Quantification"]["Total Amounts"][var_file_short].set(
                            splitted_lines[7])
                    else:
                        self.container_var["Oxides Quantification"]["Total Amounts"][var_file_short].set("100.0")

                    if len(splitted_lines) == 9:
                        if len(splitted_lines[8]) > 0:
                            self.container_var["Oxides Quantification INCL"]["Total Amounts"][var_file_short].set(
                                splitted_lines[8])
                        else:
                            self.container_var["Oxides Quantification INCL"]["Total Amounts"][var_file_short].set(
                                "100.0")
                    else:
                        self.container_var["Oxides Quantification INCL"]["Total Amounts"][var_file_short].set("100.0")

    def open_project_part_04(self, key_setting, index_container, loaded_lines):
        ## ISOTOPES
        analysis_mode = key_setting[:2]

        if analysis_mode == "ma":
            title_next = "SAMPLE SETTINGS"
        else:
            title_next = "INCLUSION SETTINGS"

        for i in range(index_container["ISOTOPES"] + 1, index_container[title_next] - 1):
            line_std = str(loaded_lines[i].strip())
            splitted_lines = line_std.split(";")

            if len(splitted_lines[0]) != 0:
                if len(splitted_lines) == 1:
                    splitted_lines = line_std.split(",")

                if len(splitted_lines) > 1:
                    isotope = splitted_lines[0]
                    self.container_var["SRM"][isotope] = tk.StringVar()
                    self.container_lists["ISOTOPES"].append(isotope)
                    self.container_var["SRM"][isotope].set(splitted_lines[1])
                else:
                    oxide = splitted_lines[0]
                    if len(oxide) > 0:
                        self.container_lists["Selected Oxides"]["All"].append(oxide)

    def open_project_part_05(self, key_setting, index_container, loaded_lines):
        ## SAMPLE/MATRIX SETTINGS
        analysis_mode = key_setting[:2]

        if analysis_mode == "ma":
            title_0 = "SAMPLE SETTINGS"
            title_1 = "DWELL TIME SETTINGS"
        elif analysis_mode == "fi":
            title_0 = "INCLUSION SETTINGS"

            if self.without_pypitzer == False:
                title_1 = "PYPITZER SETTINGS"
            else:
                title_1 = "QUANTIFICATION SETTINGS (MATRIX-ONLY TRACER)"
        elif analysis_mode == "mi":
            title_0 = "INCLUSION SETTINGS"
            title_1 = "QUANTIFICATION SETTINGS (MATRIX-ONLY TRACER)"

        for i in range(index_container[title_0] + 1, index_container[title_1] - 1):
            line_std = str(loaded_lines[i].strip())
            splitted_lines = line_std.split(";")

            if len(splitted_lines[0]) != 0:
            #if len(splitted_lines) > 1:
                if len(splitted_lines) == 1:
                    splitted_lines = line_std.split(",")

                if len(splitted_lines) == 2:
                    self.container_var[key_setting]["Inclusion Setup Selection"] = tk.IntVar()
                    self.container_var[key_setting]["Inclusion Setup Selection"].set(splitted_lines[1])

                if len(splitted_lines) > 2:
                    info_file = splitted_lines[0]
                    info_file_short = info_file.split("/")[-1]
                    info_isotope = splitted_lines[1]
                    info_concentration = splitted_lines[2]

                    self.container_var["SMPL"][info_file]["IS Data"]["IS"].set(info_isotope)
                    self.container_var["SMPL"][info_file]["IS Data"]["Concentration"].set(info_concentration)
                    self.container_var["SMPL"][info_file]["Matrix Setup"]["IS"]["Name"].set(info_isotope)
                    self.container_var["SMPL"][info_file]["Matrix Setup"]["IS"]["Concentration"].set(info_concentration)

                    if analysis_mode == "fi":
                        info_salinity = splitted_lines[3]

                        self.container_var[key_setting]["Salt Correction"]["Salinity SMPL"][
                            info_file_short] = tk.StringVar()
                        self.container_var[key_setting]["Salt Correction"]["Salinity SMPL"][
                            info_file_short].set(info_salinity)
                        self.helper_lists["Salinity"][info_file_short] = info_salinity

                        if len(splitted_lines) == 5:
                            clean_content = splitted_lines[4].replace("'", "").replace("[", "").replace("]", "").split(
                                ", ")
                            self.helper_lists["Considered chlorides"][info_file_short] = list(clean_content)

    def open_project_part_06(self, key_setting, index_container, loaded_lines):
        ## DWELL TIME SETTINGS
        for i in range(index_container["DWELL TIME SETTINGS"] + 1, index_container["INTERVAL SETTINGS"] - 1):
            line_std = str(loaded_lines[i].strip())
            splitted_lines = line_std.split(";")

            if len(splitted_lines[0]) != 0:
            #if len(splitted_lines) > 1:
                if len(splitted_lines) == 1:
                    splitted_lines = line_std.split(",")

                isotope = splitted_lines[0]
                self.container_var["dwell_times"]["Entry"][isotope] = tk.StringVar()
                self.container_var["dwell_times"]["Entry"][isotope].set(splitted_lines[1])

    def open_project_part_07(self, key_setting, index_container, loaded_lines):
        ## INTERVAL SETTINGS
        analysis_mode = key_setting[:2]
        for i in range(index_container["INTERVAL SETTINGS"] + 1, index_container["SPIKE ELIMINATION"] - 1):
            line_std = str(loaded_lines[i].strip())
            splitted_lines = line_std.split(";")

            if len(splitted_lines[0]) != 0:
            #if len(splitted_lines) > 1:
                if len(splitted_lines) == 1:
                    splitted_lines = [element for element in csv.reader(splitted_lines, delimiter=",")][0]

                if splitted_lines[1] in ["STD", "SMPL"]:
                    var_filetype = splitted_lines[1]
                    var_file_long = splitted_lines[0]
                    var_file_short = splitted_lines[0].split("/")[-1]

                    if analysis_mode == "ma":
                        self.container_helper[var_filetype][var_file_short] = {
                            "BG": {"Content": {}, "Indices": []},
                            "MAT": {"Content": {}, "Indices": []}}
                    else:
                        self.container_helper[var_filetype][var_file_short] = {
                            "BG": {"Content": {}, "Indices": []},
                            "MAT": {"Content": {}, "Indices": []},
                            "INCL": {"Content": {}, "Indices": []}}
                else:
                    if splitted_lines[0] == "BG":
                        var_id = int(splitted_lines[1])
                        var_times = splitted_lines[2]
                        try:
                            var_indices = splitted_lines[3]

                            key_times = re.search(r"\[(\d+\.\d+)(\,\s+)(\d+\.\d+)\]", var_times)
                            key_indices = re.search(r"\[(\d+)(\,\s+)(\d+)\]", var_indices)
                            helper_times = [float(key_times.group(1)), float(key_times.group(3))]
                            helper_indices = [int(key_indices.group(1)), int(key_indices.group(3))]
                            helper_times.sort()
                            helper_indices.sort()
                            self.container_helper[var_filetype][var_file_short]["BG"]["ID"] = var_id
                            self.container_helper[var_filetype][var_file_short]["BG"]["Content"][var_id] = {}
                            self.container_helper[var_filetype][var_file_short]["BG"]["Content"][var_id][
                                "Times"] = helper_times
                            self.container_helper[var_filetype][var_file_short]["BG"]["Content"][var_id][
                                "Indices"] = helper_indices

                            self.container_helper[var_filetype][var_file_short]["BG"]["Indices"].append(var_id)
                        except:
                            print("File:", var_file_short)
                            print(
                                "There is a problem with the background interval data. It seems that they are damaged.")
                    elif splitted_lines[0] == "MAT":
                        var_id = int(splitted_lines[1])
                        var_times = splitted_lines[2]
                        try:
                            var_indices = splitted_lines[3]

                            key_times = re.search(r"\[(\d+\.\d+)(\,\s+)(\d+\.\d+)\]", var_times)
                            key_indices = re.search(r"\[(\d+)(\,\s+)(\d+)\]", var_indices)

                            helper_times = [float(key_times.group(1)), float(key_times.group(3))]
                            helper_indices = [int(key_indices.group(1)), int(key_indices.group(3))]
                            helper_times.sort()
                            helper_indices.sort()
                            self.container_helper[var_filetype][var_file_short]["MAT"]["ID"] = var_id
                            self.container_helper[var_filetype][var_file_short]["MAT"]["Content"][var_id] = {}
                            self.container_helper[var_filetype][var_file_short]["MAT"]["Content"][var_id][
                                "Times"] = helper_times
                            self.container_helper[var_filetype][var_file_short]["MAT"]["Content"][var_id][
                                "Indices"] = helper_indices

                            self.container_helper[var_filetype][var_file_short]["MAT"]["Indices"].append(var_id)
                        except:
                            print("File:", var_file_short)
                            print("There is a problem with the matrix interval data. It seems that they are damaged.")
                    elif splitted_lines[0] == "INCL":
                        var_id = int(splitted_lines[1])
                        var_times = splitted_lines[2]
                        try:
                            var_indices = splitted_lines[3]

                            key_times = re.search(r"\[(\d+\.\d+)(\,\s+)(\d+\.\d+)\]", var_times)
                            key_indices = re.search(r"\[(\d+)(\,\s+)(\d+)\]", var_indices)
                            helper_times = [float(key_times.group(1)), float(key_times.group(3))]
                            helper_indices = [int(key_indices.group(1)), int(key_indices.group(3))]
                            helper_times.sort()
                            helper_indices.sort()
                            self.container_helper[var_filetype][var_file_short]["INCL"]["ID"] = var_id
                            self.container_helper[var_filetype][var_file_short]["INCL"]["Content"][var_id] = {}
                            self.container_helper[var_filetype][var_file_short]["INCL"]["Content"][var_id][
                                "Times"] = helper_times
                            self.container_helper[var_filetype][var_file_short]["INCL"]["Content"][var_id][
                                "Indices"] = helper_indices

                            self.container_helper[var_filetype][var_file_short]["INCL"]["Indices"].append(var_id)
                        except:
                            print("File:", var_file_short)
                            print(
                                "There is a problem with the inclusion interval data. It seems that they are damaged.")
                if splitted_lines[0] == "BG":
                    self.container_helper[var_filetype][var_file_short]["BG"]["ID"] = len(
                        self.container_helper[var_filetype][var_file_short]["BG"]["Content"])
                elif splitted_lines[0] == "MAT":
                    self.container_helper[var_filetype][var_file_short]["MAT"]["ID"] = len(
                        self.container_helper[var_filetype][var_file_short]["MAT"]["Content"])
                elif splitted_lines[0] == "INCL":
                    self.container_helper[var_filetype][var_file_short]["INCL"]["ID"] = len(
                        self.container_helper[var_filetype][var_file_short]["INCL"]["Content"])

    def open_project_part_08(self, key_setting, index_container, loaded_lines):
        ## SPIKE ELIMINATION
        analysis_mode = key_setting[:2]
        index = 0
        if self.old_file:
            final_line = index_container["END"] - 1
        else:
            final_line = index_container["EXPERIMENTAL DATA"] - 1

        for i in range(index_container["SPIKE ELIMINATION"] + 1, final_line):
            line_std = str(loaded_lines[i].strip())
            splitted_lines = line_std.split(";")
            if len(splitted_lines[0]) != 0:
                if len(splitted_lines) == 1:
                    splitted_lines = line_std.split(",")

                if index == 0:
                    self.container_var["Spike Elimination"]["STD"]["State"] = bool(splitted_lines[1])
                    self.container_var["Spike Elimination"]["SMPL"]["State"] = bool(splitted_lines[3])

                    if analysis_mode == "ma":
                        if splitted_lines[4] in ["Grubbs-Test", "Grubbs-Test (SILLS)"]:
                            splitted_lines[4] = "Grubbs test"

                        self.container_var["Spike Elimination Method"].set(splitted_lines[4])
                        self.container_var[key_setting]["SE Alpha"].set(splitted_lines[5])
                        self.container_var[key_setting]["SE Threshold"].set(int(splitted_lines[6]))
                    else:
                        if splitted_lines[5] in ["Grubbs-Test", "Grubbs-Test (SILLS)"]:
                            splitted_lines[5] = "Grubbs test"

                        if len(splitted_lines) == 8:
                            self.container_var[key_setting]["Spike Elimination Inclusion"].set(splitted_lines[4])
                            self.container_var["Spike Elimination Method"].set(splitted_lines[5])
                            self.container_var[key_setting]["SE Alpha"].set(splitted_lines[6])
                            self.container_var[key_setting]["SE Threshold"].set(int(splitted_lines[7]))
                        else:
                            self.container_var["Spike Elimination Method"].set(splitted_lines[4])
                            self.container_var[key_setting]["SE Alpha"].set(splitted_lines[5])
                            self.container_var[key_setting]["SE Threshold"].set(int(splitted_lines[6]))

                    index += 1
                else:
                    if len(splitted_lines) == 1:
                        var_file = splitted_lines[0]
                        if var_file not in self.container_spike_values:
                            self.container_spike_values[var_file] = {}
                    if len(splitted_lines) > 1:
                        var_isotope = splitted_lines[0]
                        list_values = splitted_lines[1:]
                        if var_isotope not in self.container_spike_values[var_file]:
                            self.container_spike_values[var_file][var_isotope] = {
                                "RAW": [], "SMOOTHED": [], "Current": [], "Save": {}}

                        for var_index in range(0, len(list_values), 2):
                            var_id = int(list_values[var_index])
                            val_id = float(list_values[var_index + 1])
                            self.container_spike_values[var_file][var_isotope]["Save"][var_id] = val_id

    def open_project_part_09(self, key_setting, index_container, loaded_lines, filename):
        ## EXPERIMENTAL DATA
        if self.old_file == False:
            helper_indices = {}
            for index, i in enumerate(range(index_container["EXPERIMENTAL DATA"] + 1, index_container["END"] - 1)):
                line_std = str(loaded_lines[i].strip())
                splitted_lines = line_std.split(";")

                if len(splitted_lines) == 1:
                    splitted_lines = line_std.split(",")

                if len(splitted_lines) == 1:
                    if splitted_lines[0] not in ["Standard Files", "Sample Files", ""]:
                        if splitted_lines[0] not in helper_indices:
                            helper_indices[splitted_lines[0]] = [i + 2]
                        if index > 1:
                            helper_indices[last_file].append(i - 2)
                        last_file = splitted_lines[0]

            if index > 1:
                helper_indices[last_file].append(i)

            for key, data in helper_indices.items():
                filename_short = key

                if filename_short in self.container_lists["STD"]["Short"]:
                    str_filetype = "STD"
                    index_file = self.container_lists["STD"]["Short"].index(filename_short)
                    str_filename_long = self.container_lists["STD"]["Long"][index_file]
                elif filename_short in self.container_lists["SMPL"]["Short"]:
                    str_filetype = "SMPL"
                    index_file = self.container_lists["SMPL"]["Short"].index(filename_short)
                    str_filename_long = self.container_lists["SMPL"]["Long"][index_file]
                else:
                    str_filetype = "unknown"

                header_names = ["Time"]
                header_names.extend(self.container_lists["ISOTOPES"])

                try:
                    dataframe = pd.read_csv(
                        filename, sep=";", header=0, skiprows=data[0], nrows=data[1] - data[0],
                        usecols=header_names, engine="python")
                except:
                    dataframe = pd.read_csv(
                        filename, sep=",", header=0, skiprows=data[0], nrows=data[1] - data[0],
                        usecols=header_names, engine="python")

                dataframe_blank = dataframe.loc[dataframe.isnull().all(1)]
                if len(dataframe_blank) > 0:
                    first_blank_index = dataframe_blank.index[0]
                    dataframe = dataframe[:first_blank_index]
                var_columns = dataframe.columns

                for column in var_columns:
                    if (list(dataframe[var_columns[0]])[-1] == ",,"
                            or list(dataframe[var_columns[0]])[-1] == ";;"):
                        dataframe[column] = dataframe[column][:-1].astype(float)
                    else:
                        dataframe[column] = dataframe[column].astype(float)
                df_isotopes = DE().get_isotopes(dataframe=dataframe)

                times = DE().get_times(dataframe=dataframe)
                self.container_lists["Measured Isotopes"][key] = df_isotopes

                if key not in self.container_lists["Measured Elements"]:
                    self.container_lists["Measured Elements"][key] = {}

                for isotope in df_isotopes:
                    key_element = re.search(r"(\D+)(\d+)", isotope)
                    element = key_element.group(1)
                    if element not in self.container_lists["Measured Elements"][key]:
                        self.container_lists["Measured Elements"][key][element] = [isotope]
                    else:
                        if isotope not in self.container_lists["Measured Elements"][key][element]:
                            self.container_lists["Measured Elements"][key][element].append(isotope)
                    if element not in self.container_lists["Measured Elements"]["All"]:
                        self.container_lists["Measured Elements"]["All"].append(element)

                if "Dataframe" not in self.container_measurements:
                    self.container_measurements["Dataframe"] = {}
                if key not in self.container_measurements["Dataframe"]:
                    self.container_measurements["Dataframe"][key] = dataframe

                if key not in self.container_measurements["RAW"]:
                    self.container_measurements["RAW"][key] = {}
                    self.container_measurements["EDITED"][key] = {}
                    self.container_measurements["SELECTED"][key] = {}
                self.container_measurements["RAW"][key]["Time"] = times.tolist()
                self.container_measurements["EDITED"][key]["Time"] = times.tolist()
                self.container_measurements["SELECTED"][key]["Time"] = times.tolist()

                for isotope in df_isotopes:
                    self.container_measurements["RAW"][key][isotope] = dataframe[isotope].tolist()
                    self.container_measurements["EDITED"][key][isotope] = {}

                    if "RAW" not in self.container_measurements["SELECTED"][key]:
                        self.container_measurements["SELECTED"][key]["RAW"] = {}

                    if "SMOOTHED" not in self.container_measurements["SELECTED"][key]:
                        self.container_measurements["SELECTED"][key]["SMOOTHED"] = {}

                    self.container_measurements["SELECTED"][key]["RAW"][isotope] = {}
                    self.container_measurements["SELECTED"][key]["SMOOTHED"][isotope] = {}

                    data_raw = self.container_measurements["RAW"][filename_short][isotope]
                    data_smoothed = data_raw.copy()
                    data_improved = data_raw.copy()
                    data_indices = []
                    data_times = self.container_measurements["SELECTED"][filename_short]["Time"]

                    if filename_short in self.container_spike_values:
                        if isotope in self.container_spike_values[filename_short]:
                            for index, value in self.container_spike_values[filename_short][isotope]["Save"].items():
                                if index < len(data_improved):
                                    data_indices.append(index)
                                    data_improved[index] = value
                                else:
                                    print("ATTENTION - - -")
                                    print(filename_short, isotope, index, value)
                                    print(len(data_improved))
                                    data_indices.append(index)
                                    data_improved[index] = value

                    self.container_spikes[filename_short][isotope] = {
                        "Data RAW": data_raw, "Data SMOOTHED": data_smoothed, "Data IMPROVED": data_improved,
                        "Indices": data_indices, "Times": data_times}

    def build_file_data_09(self, key, df_isotopes, dataframe, times, dataframe_smoothed=None, with_smoothed=None):
        self.container_lists["Measured Isotopes"][key] = df_isotopes

        if key not in self.container_lists["Measured Elements"]:
            self.container_lists["Measured Elements"][key] = {}

        for isotope in df_isotopes:
            key_element = re.search(r"(\D+)(\d+)", isotope)
            element = key_element.group(1)
            if element not in self.container_lists["Measured Elements"][key]:
                self.container_lists["Measured Elements"][key][element] = [isotope]
            else:
                if isotope not in self.container_lists["Measured Elements"][key][element]:
                    self.container_lists["Measured Elements"][key][element].append(isotope)
            if element not in self.container_lists["Measured Elements"]["All"]:
                self.container_lists["Measured Elements"]["All"].append(element)

        if "Dataframe" not in self.container_measurements:
            self.container_measurements["Dataframe"] = {}
        if key not in self.container_measurements["Dataframe"]:
            self.container_measurements["Dataframe"][key] = dataframe

        if key not in self.container_measurements["RAW"]:
            self.container_measurements["RAW"][key] = {}
            self.container_measurements["EDITED"][key] = {}
            self.container_measurements["SELECTED"][key] = {}

        self.container_measurements["RAW"][key]["Time"] = times
        self.container_measurements["EDITED"][key]["Time"] = times
        self.container_measurements["SELECTED"][key]["Time"] = times

        for isotope in df_isotopes:
            self.container_measurements["RAW"][key][isotope] = dataframe[isotope].tolist()

            if with_smoothed != None:
                self.container_measurements["EDITED"][key][isotope] = dataframe_smoothed[isotope].tolist()
            else:
                self.container_measurements["EDITED"][key][isotope] = {}

            if "RAW" not in self.container_measurements["SELECTED"][key]:
                self.container_measurements["SELECTED"][key]["RAW"] = {}

            if "SMOOTHED" not in self.container_measurements["SELECTED"][key]:
                self.container_measurements["SELECTED"][key]["SMOOTHED"] = {}

            self.container_measurements["SELECTED"][key]["RAW"][isotope] = {}
            self.container_measurements["SELECTED"][key]["SMOOTHED"][isotope] = {}

            data_raw = self.container_measurements["RAW"][key][isotope]

            if with_smoothed != None:
                data_smoothed = self.container_measurements["EDITED"][key][isotope]
                data_improved = data_smoothed.copy()
            else:
                data_smoothed = data_raw.copy()
                data_improved = data_raw.copy()

            data_indices = []
            data_times = self.container_measurements["SELECTED"][key]["Time"]

            if key not in self.container_spike_values:
                self.container_spike_values[key] = {}
            if key not in self.container_spikes:
                self.container_spikes[key] = {}

            if key in self.container_spike_values:
                if isotope in self.container_spike_values[key]:
                    for index, value in self.container_spike_values[key][isotope]["Save"].items():
                        data_indices.append(index)
                        data_improved[index] = value
                else:
                    self.container_spike_values[key][isotope] = {"RAW": [], "SMOOTHED": [], "Current": [], "Save": {}}

            self.container_spikes[key][isotope] = {
                "Data RAW": data_raw, "Data SMOOTHED": data_smoothed, "Data IMPROVED": data_improved,
                "Indices": data_indices, "Times": data_times}

    def build_intervals(self, filetype, filename, focus, list_time, start_i, end_i, id):
        index_start_i = list_time.index(start_i)
        index_end_i = list_time.index(end_i)
        int_id = id + 1

        if len(self.container_helper[filetype][filename][focus]["Indices"]) > 0:
            int_id = self.container_helper[filetype][filename][focus]["Indices"][-1] + 1

        helper_times = [float(start_i), float(end_i)]
        helper_indices = [int(index_start_i), int(index_end_i)]
        helper_times.sort()
        helper_indices.sort()

        self.container_helper[filetype][filename][focus]["ID"] = int_id
        self.container_helper[filetype][filename][focus]["Content"][int_id] = {}
        self.container_helper[filetype][filename][focus]["Content"][int_id][
            "Times"] = helper_times
        self.container_helper[filetype][filename][focus]["Content"][int_id][
            "Indices"] = helper_indices
        self.container_helper[filetype][filename][focus]["Indices"].append(int_id)

    def load_sills_file(self, filename):
        data_sills = scipy.io.loadmat(filename)
        list_srm = []

        if "SRM" in data_sills:
            helper_srm = {}
            for index, data_srm in enumerate(data_sills["SRM"]):
                str_srm = str(data_srm[0][4][0])
                list_srm.append(str_srm)
                helper_srm[str_srm] = {}
                container_values = data_srm[0][0]
                container_elements = data_srm[0][1]
                for index2, element_pre in enumerate(container_elements):
                    element = str(element_pre[0][0])
                    value = float(container_values[index2][0])
                    helper_srm[str_srm][element] = value

        # if "A" in data_sills:
        #     for index, data in enumerate(data_sills["A"]):
        #         print(index, data)

        if "STD" in data_sills:
            helper_std = {}
            helper_std_smoothed = {}
            for index, data_std in enumerate(data_sills["STD"]):
                for index2, data_file in enumerate(data_std):
                    str_std = str(data_file[3][0][0][0][0])
                    helper_std[str_std] = {}
                    helper_std_smoothed[str_std] = {}
                    list_isotopes = []
                    list_time = []

                    if "xl" in str_std:
                        str_acquisition_pre = str(data_file[3][0][0][1][0])
                    elif "csv" in str_std:
                        str_acquisition_pre = str(data_file[1][2][0][0])
                    else:
                        str_acquisition_pre = str(data_file[3][0][0][1][0])

                    self.container_spikes[str_std] = {}
                    self.container_helper["STD"][str_std] = {"BG": {"Content": {}, "Indices": []},
                                                             "MAT": {"Content": {}, "Indices": []},
                                                             "INCL": {"Content": {}, "Indices": []}}
                    self.container_files["STD"][str_std] = {"SRM": tk.StringVar()}
                    self.container_var["STD"][str_std] = {
                        "IS Data": {"IS": tk.StringVar(), "Concentration": tk.StringVar()},
                        "Checkbox": tk.IntVar(), "Sign Color": tk.StringVar(), "SRM": tk.StringVar()}

                    if "Acquired" in str_acquisition_pre:
                        key_start = re.search(
                            r"Acquired\s+\:\s+(\d+)\/(\d+)\/(\d+)\s+(\d+)\:(\d+)\:(\d+)( using Batch )(\w+)",
                            str_acquisition_pre)
                        if key_start:
                            str_acquisition = [
                                str(key_start.group(4)), str(key_start.group(5)), str(key_start.group(6))]
                            self.container_var["acquisition times"]["STD"][str_std] = tk.StringVar()
                            self.container_var["acquisition times"]["STD"][str_std].set(str_acquisition[0] + ":"
                                                                                        + str_acquisition[1] + ":"
                                                                                        + str_acquisition[2])
                    else:
                        if "xl" in str_std:
                            key_time = re.search(r"(\d+)\-(\w+)\-(\d+)\s+(\d+)\:(\d+)\:(\d+)",
                                                 str_acquisition_pre)
                            if key_time:
                                str_acquisition = [
                                    str(key_time.group(4)), str(key_time.group(5)), str(key_time.group(6))]

                                self.container_var["acquisition times"]["STD"][str_std] = tk.StringVar()
                                self.container_var["acquisition times"]["STD"][str_std].set(str_acquisition[0] + ":"
                                                                                            + str_acquisition[1] + ":"
                                                                                            + str_acquisition[2])

                    for i, item in enumerate(data_file[2][0]):
                        if i > 0:
                            isotope = str(item[0])
                            list_isotopes.append(isotope)
                            helper_std[str_std][isotope] = []
                            helper_std_smoothed[str_std][isotope] = []
                            if isotope not in self.container_var["dwell_times"]["Entry"]:
                                self.container_var["dwell_times"]["Entry"][isotope] = tk.StringVar()
                                self.container_var["dwell_times"]["Entry"][isotope].set("0.002")

                    for index3, data in enumerate(data_file):
                        if index3 == 0:
                            number_measurements = len(data)
                            list_indices_measurements = []
                        if len(data) == number_measurements:
                            list_indices_measurements.append(index3)
                        #print(index3, len(data), data)
                        if index3 == 0: # Corrected values
                            for line, values in enumerate(data):
                                value_time = float(values[0])
                                list_time.append(value_time)
                                for j, value in enumerate(values):
                                    if j > 0:
                                        isotope = list_isotopes[j - 1]
                                        helper_std_smoothed[str_std][isotope].append(float(value))
                        elif index3 == 26:   # Original values
                            for line, values in enumerate(data):
                                for j, value in enumerate(values):
                                    isotope = list_isotopes[j]
                                    helper_std[str_std][isotope].append(float(value))
                        elif index3 == 5:    # Standard Reference Material (SRM)
                            index_srm = data[0][0] - 1
                            self.container_files["STD"][str_std]["SRM"].set(list_srm[index_srm])
                            self.container_var["STD"][str_std]["SRM"].set(list_srm[index_srm])
                        elif index3 == 6:    # Background interval
                            for i, interval in enumerate(data):
                                start_i = interval[0]
                                end_i = interval[1]
                                self.build_intervals(filetype="STD", filename=str_std, focus="BG", list_time=list_time,
                                                     start_i=start_i, end_i=end_i, id=i)
                        elif index3 == 7:    # Matrix interval
                            for i, interval in enumerate(data):
                                start_i = interval[0]
                                end_i = interval[1]
                                self.build_intervals(filetype="STD", filename=str_std, focus="MAT", list_time=list_time,
                                                     start_i=start_i, end_i=end_i, id=i)
                        elif index3 == 38:    # SRM data
                            helper_srm_i = {}
                            unique_srm = []
                            self.srm_isotopes = {}

                            for i, srm_value in enumerate(data[0]):
                                isotope = list_isotopes[i]
                                helper_srm_i[isotope] = float(srm_value)

                            for isotope, value in helper_srm_i.items():
                                if value not in unique_srm:
                                    unique_srm.append(value)

                            for srm, srm_dataset in helper_srm.items():
                                data_srm_i = list(srm_dataset.values())
                                common_data_i = list(set(data_srm_i).intersection(unique_srm))

                                if unique_srm.sort() == common_data_i.sort():
                                    # self.container_files["STD"][str_std]["SRM"].set(srm)
                                    # self.container_var["STD"][str_std]["SRM"].set(srm)
                                    self.container_var["SRM"]["default"][0].set(srm)
                                    self.container_var["SRM"]["default"][1].set(srm)

                                    if srm not in self.srm_actual:
                                        self.srm_actual[srm] = {}

                                    for isotope, value in helper_srm_i.items():
                                        self.srm_isotopes[isotope] = {}
                                        key_element = re.search(r"(\D+)(\d+)", isotope)
                                        element = key_element.group(1)

                                        if element not in self.container_var["SRM"]:
                                            self.container_var["SRM"][element] = tk.StringVar()
                                        if isotope not in self.container_var["SRM"]:
                                            self.container_var["SRM"][isotope] = tk.StringVar()
                                        if element not in self.container_var["STD"]:
                                            self.container_var["STD"][element] = {"SRM": tk.StringVar()}

                                        self.container_var["STD"][element]["SRM"].set(srm)
                                        self.container_var["SRM"][isotope].set(srm)
                                        self.srm_isotopes[isotope]["SRM"] = srm

                                        if element in helper_srm[srm]:
                                            self.srm_actual[srm][element] = helper_srm[srm][element]
                                        else:
                                            self.srm_actual[srm][element] = 0.0

                                        if element in self.srm_actual[srm]:
                                            self.srm_isotopes[isotope]["Concentration"] = self.srm_actual[srm][element]
                                        else:
                                            self.srm_isotopes[isotope]["Concentration"] = 0.0

                    helper_std[str_std]["Time"] = list_time
                    helper_std_smoothed[str_std]["Time"] = list_time

                    self.container_var["STD"][str_std]["Checkbox"].set(1)
                    self.container_var["STD"][str_std]["Sign Color"].set(self.sign_yellow)
                    self.lb_std.insert(tk.END, str(str_std))
                    self.list_std.append(str_std)
                    self.container_lists["STD"]["Long"].append(str_std)
                    self.container_lists["STD"]["Short"].append(str_std)

                    # Build pandas dataframe
                    column_names = ["Time"]
                    column_names.extend(list_isotopes)
                    data_columns = [list_time]
                    data_columns_smoothed = [list_time]

                    for isotope in list_isotopes:
                        data_columns.append(helper_std[str_std][isotope])
                        data_columns_smoothed.append(helper_std_smoothed[str_std][isotope])

                    data_dict = {col: values for col, values in zip(column_names, data_columns)}
                    data_dict_smoothed = {col: values for col, values in zip(column_names, data_columns_smoothed)}

                    try:
                        df_std = pd.DataFrame(data_dict)
                        df_std_smoothed = pd.DataFrame(data_dict_smoothed)
                    except:
                        for index, data_std in enumerate(data_sills["STD"]):
                            for index2, data_file in enumerate(data_std):
                                if str_std == str(data_file[3][0][0][0][0]):
                                    helper_std[str_std] = {}
                                    helper_std_smoothed[str_std] = {}
                                    list_time = []
                                    for index3, data in enumerate(data_file):
                                        if index3 in list_indices_measurements:
                                            if index3 == 0:  # Corrected values
                                                for line, values in enumerate(data):
                                                    value_time = float(values[0])
                                                    list_time.append(value_time)
                                                    for j, value in enumerate(values):
                                                        if j > 0:
                                                            isotope = list_isotopes[j - 1]
                                                            if isotope not in helper_std_smoothed[str_std]:
                                                                helper_std_smoothed[str_std][isotope] = []
                                                            helper_std_smoothed[str_std][isotope].append(float(value))
                                            if index3 == list_indices_measurements[-1]:  # Original values
                                                if len(list_indices_measurements) > 1:
                                                    if len(data[0]) == len(list_isotopes):
                                                        for line, values in enumerate(data):
                                                            for j, value in enumerate(values):
                                                                isotope = list_isotopes[j]
                                                                if isotope not in helper_std[str_std]:
                                                                    helper_std[str_std][isotope] = []
                                                                helper_std[str_std][isotope].append(float(value))
                                                    else:
                                                        for index3, data in enumerate(data_file):
                                                            if index3 == list_indices_measurements[-2]:
                                                                if len(data[0]) == len(list_isotopes):
                                                                    for line, values in enumerate(data):
                                                                        for j, value in enumerate(values):
                                                                            isotope = list_isotopes[j]
                                                                            if isotope not in helper_std[str_std]:
                                                                                helper_std[str_std][isotope] = []
                                                                            helper_std[str_std][isotope].append(
                                                                                float(value))
                                                else:
                                                    for line, values in enumerate(data):
                                                        for j, value in enumerate(values):
                                                            if j > 0:
                                                                isotope = list_isotopes[j - 1]
                                                                if isotope not in helper_std[str_std]:
                                                                    helper_std[str_std][isotope] = []
                                                                helper_std[str_std][isotope].append(float(value))

                                    data_dict = {"Time": list_time}
                                    data_dict_smoothed = {"Time": list_time}
                                    for key, values in helper_std.items():
                                        for isotope, data in values.items():
                                            data_dict[isotope] = data
                                    for key, values in helper_std_smoothed.items():
                                        for isotope, data in values.items():
                                            data_dict_smoothed[isotope] = data

                                    df_std = pd.DataFrame(data_dict)
                                    df_std_smoothed = pd.DataFrame(data_dict_smoothed)

                    self.build_file_data_09(key=str_std, df_isotopes=list_isotopes, dataframe=df_std, times=list_time,
                                            dataframe_smoothed=df_std_smoothed, with_smoothed=False)
                    self.container_var["Spike Elimination"]["STD"]["State"] = True

            helper_srm_2 = {}

            if len(self.srm_actual) > 0:
                for srm, dataset in self.srm_actual.items():
                    helper_srm_2[srm] = 0
                    for isotope, value in dataset.items():
                        if value == 0:
                            helper_srm_2[srm] += 1
            else:
                helper_srm_i = {}
                unique_srm = []
                self.srm_isotopes = {}

                for srm, dataset in helper_srm.items():
                    for isotope in list_isotopes:
                        key_element = re.search(r"(\D+)(\d+)", isotope)
                        element = key_element.group(1)
                        if element in dataset:
                            helper_srm_i[isotope] = float(dataset[element])

                for isotope, value in helper_srm_i.items():
                    if value not in unique_srm:
                        unique_srm.append(value)

                for srm, srm_dataset in helper_srm.items():
                    data_srm_i = list(srm_dataset.values())
                    common_data_i = list(set(data_srm_i).intersection(unique_srm))

                    if unique_srm.sort() == common_data_i.sort():
                        self.container_var["SRM"]["default"][0].set(srm)
                        self.container_var["SRM"]["default"][1].set(srm)

                        if srm not in self.srm_actual:
                            self.srm_actual[srm] = {}

                        for isotope, value in helper_srm_i.items():
                            self.srm_isotopes[isotope] = {}
                            key_element = re.search(r"(\D+)(\d+)", isotope)
                            element = key_element.group(1)

                            if element not in self.container_var["SRM"]:
                                self.container_var["SRM"][element] = tk.StringVar()
                            if isotope not in self.container_var["SRM"]:
                                self.container_var["SRM"][isotope] = tk.StringVar()
                            if element not in self.container_var["STD"]:
                                self.container_var["STD"][element] = {"SRM": tk.StringVar()}

                            self.container_var["STD"][element]["SRM"].set(srm)
                            self.container_var["SRM"][isotope].set(srm)
                            self.srm_isotopes[isotope]["SRM"] = srm

                            if element in helper_srm[srm]:
                                self.srm_actual[srm][element] = helper_srm[srm][element]
                            else:
                                self.srm_actual[srm][element] = 0.0

                            if element in self.srm_actual[srm]:
                                self.srm_isotopes[isotope]["Concentration"] = self.srm_actual[srm][element]
                            else:
                                self.srm_isotopes[isotope]["Concentration"] = 0.0

                for srm, dataset in self.srm_actual.items():
                    helper_srm_2[srm] = 0
                    for isotope, value in dataset.items():
                        if value == 0:
                            helper_srm_2[srm] += 1

            if len(helper_srm_2) > 1:
                for isotope in list_isotopes:
                    srm = list_srm[1]
                    key_element = re.search(r"(\D+)(\d+)", isotope)
                    element = key_element.group(1)
                    self.container_var["STD"][element]["SRM"].set(srm)
                    self.container_var["SRM"][isotope].set(srm)
                    self.srm_isotopes[isotope]["SRM"] = srm

                    if element in self.srm_actual[srm]:
                        self.srm_isotopes[isotope]["Concentration"] = self.srm_actual[srm][element]
                    else:
                        self.srm_isotopes[isotope]["Concentration"] = 0.0

                    if element in ["Cl", "Br", "I"]:
                        for srm in list_srm:
                            if "Sca" in srm or "sca" in srm:
                                self.container_var["STD"][element]["SRM"].set(srm)
                                self.container_var["SRM"][isotope].set(srm)
                                self.srm_isotopes[isotope]["SRM"] = srm

                                if element in self.srm_actual[srm]:
                                    self.srm_isotopes[isotope]["Concentration"] = self.srm_actual[srm][element]
                                else:
                                    self.srm_isotopes[isotope]["Concentration"] = 0.0

        if "UNK" in data_sills:
            helper_smpl = {}
            helper_smpl_smoothed = {}
            self.mode_ma = True
            self.mode_2ndis = False
            for index, data_smpl in enumerate(data_sills["UNK"]):
                for index2, data_file in enumerate(data_smpl):
                    str_smpl = str(data_file[3][0][0][0][0])
                    helper_smpl[str_smpl] = {}
                    helper_smpl_smoothed[str_smpl] = {}
                    list_isotopes = []
                    list_time = []

                    if "xl" in str_std:
                        str_acquisition_pre = str(data_file[3][0][0][1][0])
                    elif "csv" in str_std:
                        str_acquisition_pre = str(data_file[1][2][0][0])
                    else:
                        str_acquisition_pre = str(data_file[3][0][0][1][0])

                    self.container_helper["SMPL"][str_smpl] = {"BG": {"Content": {}, "Indices": []},
                                                               "MAT": {"Content": {}, "Indices": []},
                                                               "INCL": {"Content": {}, "Indices": []}}
                    self.container_var["SMPL"][str_smpl] = {
                        "IS Data": {"IS": tk.StringVar(), "Concentration": tk.StringVar()},
                        "Checkbox": tk.IntVar(), "ID": tk.StringVar(), "Sign Color": tk.StringVar()}
                    self.container_var["SMPL"][str_smpl]["Matrix Setup"] = {
                        "IS": {"Name": tk.StringVar(), "Concentration": tk.StringVar()},
                        "Oxide": {"Name": tk.StringVar(), "Concentration": tk.StringVar()},
                        "Element": {"Name": tk.StringVar(), "Concentration": tk.StringVar()}}
                    self.container_var["SMPL"][str_smpl]["Host Only Tracer"] = {
                        "Name": tk.StringVar(), "Value": tk.StringVar(), "Matrix": tk.StringVar(),
                        "Amount": tk.StringVar()}
                    self.container_var["SMPL"][str_smpl]["Second Internal Standard"] = {
                        "Name": tk.StringVar(), "Value": tk.StringVar()}

                    if "Acquired" in str_acquisition_pre:
                        key_start = re.search(
                            r"Acquired\s+\:\s+(\d+)\/(\d+)\/(\d+)\s+(\d+)\:(\d+)\:(\d+)( using Batch )(\w+)",
                            str_acquisition_pre)
                        if key_start:
                            str_acquisition = [
                                str(key_start.group(4)), str(key_start.group(5)), str(key_start.group(6))]
                            self.container_var["acquisition times"]["SMPL"][str_smpl] = tk.StringVar()
                            self.container_var["acquisition times"]["SMPL"][str_smpl].set(str_acquisition[0] + ":"
                                                                                          + str_acquisition[1] + ":"
                                                                                          + str_acquisition[2])
                    else:
                        if "xl" in str_smpl:
                            key_time = re.search(r"(\d+)\-(\w+)\-(\d+)\s+(\d+)\:(\d+)\:(\d+)", str_acquisition_pre)
                            if key_time:
                                str_acquisition = [
                                    str(key_time.group(4)), str(key_time.group(5)), str(key_time.group(6))]

                                self.container_var["acquisition times"]["SMPL"][str_smpl] = tk.StringVar()
                                self.container_var["acquisition times"]["SMPL"][str_smpl].set(str_acquisition[0] + ":"
                                                                                            + str_acquisition[1] + ":"
                                                                                            + str_acquisition[2])

                    for i, item in enumerate(data_file[2][0]):
                        if i > 0:
                            isotope = str(item[0])
                            list_isotopes.append(isotope)
                            helper_smpl[str_smpl][isotope] = []
                            helper_smpl_smoothed[str_smpl][isotope] = []

                    for index3, data in enumerate(data_file):
                        if index3 == 0:
                            number_measurements = len(data)
                            list_indices_measurements = []
                        if len(data) == number_measurements:
                            list_indices_measurements.append(index3)
                        #print(index3, len(data), data)
                        if index3 == 0: # Corrected values
                            for line, values in enumerate(data):
                                value_time = float(values[0])
                                list_time.append(value_time)
                                for j, value in enumerate(values):
                                    if j > 0:
                                        isotope = list_isotopes[j - 1]
                                        helper_smpl_smoothed[str_smpl][isotope].append(float(value))
                        if index3 == 73: # Original values
                            for line, values in enumerate(data):
                                for j, value in enumerate(values):
                                    isotope = list_isotopes[j]
                                    helper_smpl[str_smpl][isotope].append(float(value))
                        elif index3 == 6:    # Background interval (MA/FI/MI)
                            for i, interval in enumerate(data):
                                start_i = interval[0]
                                end_i = interval[1]
                                self.build_intervals(filetype="SMPL", filename=str_smpl, focus="BG",
                                                     list_time=list_time, start_i=start_i, end_i=end_i, id=i)
                        elif index3 == 7 and len(data) > 0: # Matrix interval (FI/MI)
                            for i, interval in enumerate(data):
                                start_i = interval[0]
                                end_i = interval[1]
                                self.build_intervals(filetype="SMPL", filename=str_smpl, focus="MAT",
                                                     list_time=list_time, start_i=start_i, end_i=end_i, id=i)
                                self.mode_ma = False
                        elif index3 == 8 and len(data) > 0: # Matrix interval (FI/MI)
                            for i, interval in enumerate(data):
                                start_i = interval[0]
                                end_i = interval[1]
                                self.build_intervals(filetype="SMPL", filename=str_smpl, focus="MAT",
                                                     list_time=list_time, start_i=start_i, end_i=end_i, id=i)
                                self.mode_ma = False
                        elif index3 == 10:    # Matrix interval (MA) / Inclusion interval (FI/MI)
                            for i, interval in enumerate(data):
                                start_i = interval[0]
                                end_i = interval[1]

                                if self.mode_ma == True:
                                    self.build_intervals(filetype="SMPL", filename=str_smpl, focus="MAT",
                                                         list_time=list_time, start_i=start_i, end_i=end_i, id=i)
                                elif self.mode_ma == False:
                                    self.pysills_mode = "FIMI"
                                    self.build_intervals(filetype="SMPL", filename=str_smpl, focus="INCL",
                                                         list_time=list_time, start_i=start_i, end_i=end_i, id=i)
                        elif index3 == 26 and len(data) > 0:  # IS (Matrix)
                            index_isotope = data[0][0] - 1
                            if self.mode_ma == False:
                                self.container_var["SMPL"][str_smpl]["Matrix Setup"]["IS"]["Name"].set(
                                    list_isotopes[index_isotope])
                            else:
                                self.container_var["SMPL"][str_smpl]["IS Data"]["IS"].set(list_isotopes[index_isotope])
                        elif index3 == 29 and len(data) > 0 and self.mode_ma == False:    # Concentration (Matrix)
                            for i, value in enumerate(data[0]):
                                self.container_var["SMPL"][str_smpl]["Matrix Setup"]["IS"]["Concentration"].set(value)
                        elif index3 == 30 and len(data) > 0 and self.mode_ma == False:    # Amount oxide (Matrix)
                            try:
                                amount_oxide = float(data[0][0])
                            except:
                                amount_oxide = 0.0

                            self.container_var["SMPL"][str_smpl]["Matrix Setup"]["Oxide"]["Concentration"].set(
                                amount_oxide)
                        elif index3 == 31 and len(data) > 0:    # Oxide (Matrix)
                            index_oxide = data[0][0] - 1
                            list_oxides = ["SiO2", "TiO2", "Fe2O3", "FeO", "MnO", "MgO", "CaO", "Na2O", "K2O"]
                            self.container_var["SMPL"][str_smpl]["Matrix Setup"]["Oxide"]["Name"].set(
                                list_oxides[index_oxide])
                        elif index3 == 33 and len(data) > 0 and self.mode_ma == True:    # Amount oxide (Sample)
                            if len(data[0]) > 1:
                                amount_oxide = float(data[0][0])
                            else:
                                amount_oxide = 0.0
                            self.container_var["SMPL"][str_smpl]["Matrix Setup"]["Oxide"]["Concentration"].set(
                                amount_oxide)
                            key = re.search(r"(\D+)(\d*)(\D+)(\d*)", list_oxides[index_oxide])
                            element_cation = key.group(1)

                            if len(key.group(2)) > 0:
                                n_cation = int(key.group(2))
                            else:
                                n_cation = 1

                            value_is = round((n_cation*self.chemistry_data[element_cation]/self.chemistry_data_oxides[
                                list_oxides[index_oxide]])*10**6, 4)

                            if self.mode_ma == True:
                                self.container_var["SMPL"][str_smpl]["IS Data"]["Concentration"].set(value_is)
                        elif index3 == 37 and len(data) > 0 and self.mode_ma == False:    # IS (Sample/Inclusion)
                            index_isotope = data[0][0] - 1
                            self.container_var["SMPL"][str_smpl]["IS Data"]["IS"].set(list_isotopes[index_isotope])
                        elif index3 == 40 and len(data) > 0:    # Concentration (Sample/Inclusion)
                            for i, value in enumerate(data[0]):
                                self.container_var["SMPL"][str_smpl]["IS Data"]["Concentration"].set(value)
                        elif index3 == 44 and len(data) > 0 and self.mode_ma == False:  # 2nd IS
                            index_isotope = data[0][0] - 1
                            self.container_var["SMPL"][str_smpl]["Second Internal Standard"]["Name"].set(
                                list_isotopes[index_isotope])
                        elif index3 == 47 and len(data) > 0 and self.mode_ma == False:  # 2nd IS concentration
                            try:
                                value = data[0][0]
                            except:
                                value = 0.0

                            self.container_var["SMPL"][str_smpl]["Second Internal Standard"]["Value"].set(value)
                            self.mode_2ndis = True

                            for key_setting in ["fi_setting", "mi_setting"]:
                                self.container_var[key_setting]["Quantification Method Option"].set(
                                    "Second Internal Standard (SILLS)")
                        elif index3 == 60 and len(data) > 0 and self.mode_ma == False:  # Matrix-only tracer (MOT)
                            index_isotope = data[0][0] - 1
                            self.container_var["SMPL"][str_smpl]["Host Only Tracer"]["Name"].set(
                                list_isotopes[index_isotope])
                            self.container_var["SMPL"][str_smpl]["Host Only Tracer"]["Value"].set("0.0")

                            try:
                                self.container_var["SMPL"][str_smpl]["Host Only Tracer"]["Amount"].set(amount_oxide)
                                self.container_var["SMPL"][str_smpl]["Host Only Tracer"]["Matrix"].set(
                                    list_oxides[index_oxide])
                            except:
                                print("No oxide amount found.")

                            if self.mode_2ndis == False:
                                for key_setting in ["fi_setting", "mi_setting"]:
                                    self.container_var[key_setting]["Quantification Method Option"].set(
                                        "Matrix-only Tracer (SILLS)")

                    helper_smpl[str_smpl]["Time"] = list_time
                    helper_smpl_smoothed[str_smpl]["Time"] = list_time

                    self.container_var["SMPL"][str_smpl]["ID"].set("A")
                    self.container_var["SMPL"][str_smpl]["Checkbox"].set(1)
                    self.container_var["SMPL"][str_smpl]["Sign Color"].set(self.sign_yellow)
                    self.lb_smpl.insert(tk.END, str(str_smpl))
                    self.list_smpl.append(str_smpl)
                    self.container_lists["SMPL"]["Long"].append(str_smpl)
                    self.container_lists["SMPL"]["Short"].append(str_smpl)

                    # Build pandas dataframe
                    column_names = ["Time"]
                    column_names.extend(list_isotopes)
                    data_columns = [list_time]
                    data_columns_smoothed = [list_time]

                    for isotope in list_isotopes:
                        data_columns.append(helper_smpl[str_smpl][isotope])
                        data_columns_smoothed.append(helper_smpl_smoothed[str_smpl][isotope])

                    data_dict = {col: values for col, values in zip(column_names, data_columns)}
                    data_dict_smoothed = {col: values for col, values in zip(column_names, data_columns_smoothed)}

                    try:
                        df_smpl = pd.DataFrame(data_dict)
                        df_smpl_smoothed = pd.DataFrame(data_dict_smoothed)
                    except:
                        for index, data_smpl in enumerate(data_sills["UNK"]):
                            for index2, data_file in enumerate(data_smpl):
                                if str_smpl == str(data_file[3][0][0][0][0]):
                                    helper_smpl[str_smpl] = {}
                                    helper_smpl_smoothed[str_smpl] = {}
                                    list_time = []
                                    for index3, data in enumerate(data_file):
                                        if index3 in list_indices_measurements:
                                            if index3 == 0:  # Corrected values
                                                for line, values in enumerate(data):
                                                    value_time = float(values[0])
                                                    list_time.append(value_time)
                                                    for j, value in enumerate(values):
                                                        if j > 0:
                                                            isotope = list_isotopes[j - 1]
                                                            if isotope not in helper_smpl_smoothed[str_smpl]:
                                                                helper_smpl_smoothed[str_smpl][isotope] = []
                                                            helper_smpl_smoothed[str_smpl][isotope].append(float(value))
                                            if index3 == list_indices_measurements[-1]:  # Original values
                                                if len(list_indices_measurements) > 1:
                                                    if len(data[0]) == len(list_isotopes):
                                                        for line, values in enumerate(data):
                                                            for j, value in enumerate(values):
                                                                isotope = list_isotopes[j]
                                                                if isotope not in helper_smpl[str_smpl]:
                                                                    helper_smpl[str_smpl][isotope] = []
                                                                helper_smpl[str_smpl][isotope].append(float(value))
                                                    else:
                                                        for index3, data in enumerate(data_file):
                                                            if index3 == list_indices_measurements[-2]:
                                                                if len(data[0]) == len(list_isotopes):
                                                                    for line, values in enumerate(data):
                                                                        for j, value in enumerate(values):
                                                                            isotope = list_isotopes[j]
                                                                            if isotope not in helper_smpl[str_smpl]:
                                                                                helper_smpl[str_smpl][isotope] = []
                                                                            helper_smpl[str_smpl][isotope].append(
                                                                                float(value))
                                                else:
                                                    for line, values in enumerate(data):
                                                        for j, value in enumerate(values):
                                                            if j > 0:
                                                                isotope = list_isotopes[j - 1]
                                                                if isotope not in helper_smpl[str_smpl]:
                                                                    helper_smpl[str_smpl][isotope] = []
                                                                helper_smpl[str_smpl][isotope].append(float(value))

                                    data_dict = {"Time": list_time}
                                    data_dict_smoothed = {"Time": list_time}
                                    for key, values in helper_smpl.items():
                                        for isotope, data in values.items():
                                            data_dict[isotope] = data
                                    for key, values in helper_smpl_smoothed.items():
                                        for isotope, data in values.items():
                                            data_dict_smoothed[isotope] = data

                                    df_smpl = pd.DataFrame(data_dict)
                                    df_smpl_smoothed = pd.DataFrame(data_dict_smoothed)

                    self.build_file_data_09(key=str_smpl, df_isotopes=list_isotopes, dataframe=df_smpl, times=list_time,
                                            dataframe_smoothed=df_smpl_smoothed, with_smoothed=False)
                    self.container_var["Spike Elimination"]["SMPL"]["State"] = True

    def open_project(self):
        if len(self.container_lists["Measured Isotopes"]["All"]) > 0:
            print("We are sorry but it is currently necessary to restart PySILLS before another project can be opened. "
                  "This issue will be fixed in the future.")
            self.parent.bell()
        else:
            subwindow_progressbar, prgbar = self.create_progress_bar_spike_elimination()
            current_step = 0
            self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
            self.lbl_prg_spk.configure(text="Opening project started!", anchor=tk.W)

            file_ending_pre = self.container_var["General Settings"]["Project type"].get()
            file_ending = file_ending_pre[1:]

            if "csv" in file_ending:
                list_file_endings = [("CSV files *.csv", "*.csv"), ("Project files *.proj", "*.proj"),
                                     ("SILLS files *.mat", "*.mat"), ("Text files *.txt", "*.txt")]
            elif "proj" in file_ending:
                list_file_endings = [("Project files *.proj", "*.proj"), ("CSV files *.csv", "*.csv"),
                                     ("SILLS files *.mat", "*.mat"), ("Text files *.txt", "*.txt")]

            filename = filedialog.askopenfilename(
                title="Open as", defaultextension=file_ending,
                filetypes=list_file_endings,
                initialdir=self.last_used_directory)

            self.projectname = filename.split("/")[-1]
            split_filename = filename.split("/")
            self.last_used_directory = '/'.join(split_filename[:-1])
            self.parent.title("PySILLS - " + str(self.projectname))

            n_commas = 0
            n_semicolons = 0

            if ".mat" in filename:
                #print("Filename:", filename)
                self.load_sills_file(filename=filename)
                mat_data = scipy.io.loadmat(filename)
                #print(mat_data.keys())

                current_step = 100
                self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
                self.lbl_prg_spk.configure(text="Opening project finished!", anchor=tk.W)

                if current_step >= 100:
                    self.lbl_prg_spk.configure(text="Opening project finished!", anchor=tk.W)
                    subwindow_progressbar.destroy()

                # Initialization
                self.file_loaded = True
                self.demo_mode = False

                if self.old_file == True and self.var_opt_icp.get() == "Select ICP-MS":
                    self.container_icpms["name"] = None
                else:
                    self.select_icp_ms(var_opt=self.var_opt_icp)

                if self.pysills_mode == "MA":
                    self.ma_settings()
                elif self.pysills_mode == "FIMI":
                    self.rb_ma.configure(state="disabled")
                    print("Please select the correct analysis mode (Fluid/Melt inclusion analysis).")
            else:
                try:
                    file_loaded = open(str(filename), "r")
                    loaded_lines = file_loaded.readlines()
                    n_lines = len(loaded_lines)
                    for i in range(10):
                        if i == 2:
                            n_commas += loaded_lines[i].count(",")
                            n_semicolons += loaded_lines[i].count(";")
                            if n_commas > 0 and n_semicolons == 0:
                                delimiter = ","
                            elif n_commas == 0 and n_semicolons > 0:
                                delimiter = ";"
                            elif n_commas + n_semicolons == 0:
                                n_commas += loaded_lines[4].count(",")
                                n_semicolons += loaded_lines[4].count(";")
                                if n_commas > 0 and n_semicolons == 0:
                                    delimiter = ","
                                elif n_commas == 0 and n_semicolons > 0:
                                    delimiter = ";"

                    if delimiter == "," and loaded_lines[2].count(",") > 3:
                        if str("\n") in loaded_lines[0]:
                            cleaned_lines = [sub.replace(str("\n"), "") for sub in loaded_lines]
                            cleaned_lines_final = [sub.rstrip(",") for sub in cleaned_lines]
                            loaded_lines = cleaned_lines_final
                    else:
                        if ",,\n" in loaded_lines:
                            loaded_lines = [sub.replace(",,\n", "") for sub in loaded_lines]
                        elif ";;\n" in loaded_lines:
                            loaded_lines = [sub.replace(";;\n", "") for sub in loaded_lines]

                    n_settings = 0
                    strings = ["PROJECT INFORMATION", "STANDARD FILES"]
                    index_container = {}

                    while n_settings < len(strings):
                        index_container[strings[n_settings]] = 0
                        index = 0
                        flag = 0
                        for line in open(str(filename), "r"):
                            if strings[n_settings] in line:
                                flag = 1
                                break
                            else:
                                index += 1
                        if flag == 0:
                            pass
                        else:
                            index_container[strings[n_settings]] += index
                            n_settings += 1

                    for i in range(index_container["PROJECT INFORMATION"] + 1, index_container["STANDARD FILES"] - 1):
                        line_std = str(loaded_lines[i].strip())
                        splitted_line = line_std.split(";")

                        var_mode = splitted_line[0]
                        if var_mode == "":
                            line_std = str(loaded_lines[2].strip())
                            splitted_line = line_std.split(";")
                            var_mode = splitted_line[0]

                        if var_mode == "Mineral Analysis":
                            key_setting = "ma_setting"
                            self.pysills_mode = "MA"
                            self.var_rb_mode.set(0)
                        elif var_mode == "Fluid Inclusion Analysis":
                            key_setting = "fi_setting"
                            #self.pysills_mode = "FI"
                            self.pysills_mode = "INCL"
                            #self.var_rb_mode.set(1)
                            self.var_rb_mode.set(1)
                        elif var_mode == "Melt Inclusion Analysis":
                            key_setting = "mi_setting"
                            #self.pysills_mode = "MI"
                            self.pysills_mode = "INCL"
                            #self.var_rb_mode.set(2)
                            self.var_rb_mode.set(1)
                        elif var_mode == "Inclusion Analysis":
                            key_setting = "fi_setting"
                            self.pysills_mode = "INCL"
                            self.var_rb_mode.set(1)
                        break
                    self.select_experiment(var_rb=self.var_rb_mode)

                    n_settings = 0
                    if "EXPERIMENTAL DATA\n" in loaded_lines or "EXPERIMENTAL DATA" in loaded_lines:
                        if self.pysills_mode == "MA":
                            strings = ["PROJECT INFORMATION", "STANDARD FILES", "SAMPLE FILES", "ISOTOPES",
                                       "SAMPLE SETTINGS", "DWELL TIME SETTINGS", "INTERVAL SETTINGS",
                                       "SPIKE ELIMINATION", "EXPERIMENTAL DATA", "END"]
                        else:
                            if ("PYPITZER SETTINGS\n" in loaded_lines and "QUANTIFICATION SETTINGS (HALTER2002)\n"
                                    in loaded_lines and "QUANTIFICATION SETTINGS (BORISOVA2021)\n" in loaded_lines):
                                strings = ["PROJECT INFORMATION", "STANDARD FILES", "SAMPLE FILES", "ISOTOPES",
                                           "INCLUSION SETTINGS", "PYPITZER SETTINGS",
                                           "QUANTIFICATION SETTINGS (MATRIX-ONLY TRACER)",
                                           "QUANTIFICATION SETTINGS (SECOND INTERNAL STANDARD)",
                                           "QUANTIFICATION SETTINGS (HALTER2002)",
                                           "QUANTIFICATION SETTINGS (BORISOVA2021)", "MATRIX SETTINGS",
                                           "DWELL TIME SETTINGS", "INTERVAL SETTINGS", "SPIKE ELIMINATION",
                                           "EXPERIMENTAL DATA", "END"]
                            elif ("PYPITZER SETTINGS" in loaded_lines and "QUANTIFICATION SETTINGS (HALTER2002)"
                                    in loaded_lines and "QUANTIFICATION SETTINGS (BORISOVA2021)" in loaded_lines):
                                strings = ["PROJECT INFORMATION", "STANDARD FILES", "SAMPLE FILES", "ISOTOPES",
                                           "INCLUSION SETTINGS", "PYPITZER SETTINGS",
                                           "QUANTIFICATION SETTINGS (MATRIX-ONLY TRACER)",
                                           "QUANTIFICATION SETTINGS (SECOND INTERNAL STANDARD)",
                                           "QUANTIFICATION SETTINGS (HALTER2002)",
                                           "QUANTIFICATION SETTINGS (BORISOVA2021)", "MATRIX SETTINGS",
                                           "DWELL TIME SETTINGS", "INTERVAL SETTINGS", "SPIKE ELIMINATION",
                                           "EXPERIMENTAL DATA", "END"]
                            else:
                                strings = ["PROJECT INFORMATION", "STANDARD FILES", "SAMPLE FILES", "ISOTOPES",
                                           "INCLUSION SETTINGS", "QUANTIFICATION SETTINGS (MATRIX-ONLY TRACER)",
                                           "QUANTIFICATION SETTINGS (SECOND INTERNAL STANDARD)", "MATRIX SETTINGS",
                                           "DWELL TIME SETTINGS", "INTERVAL SETTINGS", "SPIKE ELIMINATION",
                                           "EXPERIMENTAL DATA", "END"]
                                self.without_pypitzer = True
                    else:
                        if self.pysills_mode == "MA":
                            strings = ["PROJECT INFORMATION", "STANDARD FILES", "SAMPLE FILES", "ISOTOPES",
                                       "SAMPLE SETTINGS", "DWELL TIME SETTINGS", "INTERVAL SETTINGS",
                                       "SPIKE ELIMINATION", "END"]
                        else:
                            strings = ["PROJECT INFORMATION", "STANDARD FILES", "SAMPLE FILES", "ISOTOPES",
                                       "INCLUSION SETTINGS", "PYPITZER SETTINGS",
                                       "QUANTIFICATION SETTINGS (MATRIX-ONLY TRACER)",
                                       "QUANTIFICATION SETTINGS (SECOND INTERNAL STANDARD)",
                                       "QUANTIFICATION SETTINGS (HALTER2002)", "QUANTIFICATION SETTINGS (BORISOVA2021)",
                                       "MATRIX SETTINGS", "DWELL TIME SETTINGS", "INTERVAL SETTINGS",
                                       "SPIKE ELIMINATION", "END"]
                        self.old_file = True

                    index_container = {}
                    while n_settings < len(strings):
                        index_container[strings[n_settings]] = 0
                        index = 0
                        flag = 0
                        for line in open(str(filename), "r"):
                            if strings[n_settings] in line:
                                flag = 1
                                break
                            else:
                                index += 1
                        if flag == 0:
                            pass
                        else:
                            index_container[strings[n_settings]] += index
                            n_settings += 1

                    time_start = datetime.datetime.now()

                    current_step = 10
                    self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
                    self.lbl_prg_spk.configure(text="Initialization", anchor=tk.W)

                    if self.pysills_mode == "MA":
                        self.open_project_part_01(
                            key_setting=key_setting, index_container=index_container, loaded_lines=loaded_lines)
                        current_step = 20
                        self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
                        self.lbl_prg_spk.configure(text=str(current_step) +" %", anchor=tk.W)
                        ## STANDARD FILES
                        self.open_project_part_02(
                            key_setting=key_setting, index_container=index_container, loaded_lines=loaded_lines)
                        current_step = 30
                        self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
                        self.lbl_prg_spk.configure(text=str(current_step) +" %", anchor=tk.W)
                        ## SAMPLE FILES
                        self.open_project_part_03(
                            key_setting=key_setting, index_container=index_container, loaded_lines=loaded_lines)
                        current_step = 40
                        self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
                        self.lbl_prg_spk.configure(text=str(current_step) +" %", anchor=tk.W)
                        ## ISOTOPES
                        self.open_project_part_04(
                            key_setting=key_setting, index_container=index_container, loaded_lines=loaded_lines)
                        current_step = 50
                        self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
                        self.lbl_prg_spk.configure(text=str(current_step) +" %", anchor=tk.W)
                        ## SAMPLE/MATRIX SETTINGS
                        self.open_project_part_05(
                            key_setting=key_setting, index_container=index_container, loaded_lines=loaded_lines)
                        current_step = 60
                        self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
                        self.lbl_prg_spk.configure(text=str(current_step) +" %", anchor=tk.W)
                        ## DWELL TIME SETTINGS
                        self.open_project_part_06(
                            key_setting=key_setting, index_container=index_container, loaded_lines=loaded_lines)
                        current_step = 70
                        self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
                        self.lbl_prg_spk.configure(text=str(current_step) +" %", anchor=tk.W)
                        ## INTERVAL SETTINGS
                        self.open_project_part_07(
                            key_setting=key_setting, index_container=index_container, loaded_lines=loaded_lines)
                        current_step = 80
                        self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
                        self.lbl_prg_spk.configure(text=str(current_step) +" %", anchor=tk.W)
                        ## SPIKE ELIMINATION
                        self.open_project_part_08(
                            key_setting=key_setting, index_container=index_container, loaded_lines=loaded_lines)
                        current_step = 90
                        self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
                        self.lbl_prg_spk.configure(text=str(current_step) +" %", anchor=tk.W)
                        ## EXPERIMENTAL DATA
                        self.open_project_part_09(
                            key_setting=key_setting, index_container=index_container, loaded_lines=loaded_lines,
                            filename=filename)
                        current_step = 100
                        self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
                        self.lbl_prg_spk.configure(text="Opening project finished!", anchor=tk.W)

                        if current_step >= 100:
                            self.lbl_prg_spk.configure(text="Opening project finished!", anchor=tk.W)
                            subwindow_progressbar.destroy()
                    elif self.pysills_mode in ["FI", "INCL"]:
                        ## PROJECT INFORMATION
                        self.open_project_part_01(
                            key_setting=key_setting, index_container=index_container, loaded_lines=loaded_lines)
                        current_step = 20
                        self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
                        self.lbl_prg_spk.configure(text=str(current_step) + " %", anchor=tk.W)
                        ## STANDARD FILES
                        self.open_project_part_02(
                            key_setting=key_setting, index_container=index_container, loaded_lines=loaded_lines)
                        current_step = 30
                        self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
                        self.lbl_prg_spk.configure(text=str(current_step) + " %", anchor=tk.W)
                        ## SAMPLE FILES
                        self.open_project_part_03(
                            key_setting=key_setting, index_container=index_container, loaded_lines=loaded_lines)
                        current_step = 40
                        self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
                        self.lbl_prg_spk.configure(text=str(current_step) + " %", anchor=tk.W)
                        ## ISOTOPES
                        self.open_project_part_04(
                            key_setting=key_setting, index_container=index_container, loaded_lines=loaded_lines)
                        current_step = 50
                        self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
                        self.lbl_prg_spk.configure(text=str(current_step) + " %", anchor=tk.W)
                        ## INCLUSION SETTINGS
                        self.open_project_part_05(
                            key_setting=key_setting, index_container=index_container, loaded_lines=loaded_lines)
                        current_step = 60
                        self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
                        self.lbl_prg_spk.configure(text=str(current_step) + " %", anchor=tk.W)
                        ## PYPITZER SETTINGS
                        if self.without_pypitzer == False:
                            for i in range(index_container["PYPITZER SETTINGS"] + 1,
                                           index_container["QUANTIFICATION SETTINGS (MATRIX-ONLY TRACER)"] - 1):
                                line_data = str(loaded_lines[i].strip())
                                splitted_data = line_data.split(";")

                                if len(splitted_data) == 1:
                                    splitted_data = line_data.split(",")

                                if splitted_data[0] == "Cations":
                                    self.container_lists["Selected Cations"].extend(splitted_data[1:])
                                elif splitted_data[0] == "Anions":
                                    self.container_lists["Selected Anions"].extend(splitted_data[1:])
                                elif splitted_data[0] == "Isotopes":
                                    for isotope in splitted_data[1:]:
                                        self.helper_checkbuttons["Isotopes"][isotope] = tk.IntVar()
                                        self.helper_checkbuttons["Isotopes"][isotope].set(1)
                                elif splitted_data[0] in self.container_lists["SMPL"]["Short"]:
                                    for filename_smpl_long in self.container_lists["SMPL"]["Long"]:
                                        var_last_compound = splitted_data[1]
                                        var_melting_temperature = splitted_data[2]

                                        self.container_var["SMPL"][filename_smpl_long]["Last compound"] = tk.StringVar()
                                        self.container_var["SMPL"][filename_smpl_long][
                                            "Melting temperature"] = tk.StringVar()
                                        self.container_var["SMPL"][filename_smpl_long]["Last compound"].set(
                                            var_last_compound)
                                        self.container_var["SMPL"][filename_smpl_long]["Melting temperature"].set(
                                            var_melting_temperature)
                        current_step = 63
                        self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
                        self.lbl_prg_spk.configure(text=str(current_step) + " %", anchor=tk.W)
                        ## QUANTIFICATION SETTINGS (MATRIX-ONLY TRACER)
                        index = 0
                        for i in range(index_container["QUANTIFICATION SETTINGS (MATRIX-ONLY TRACER)"] + 1,
                                       index_container["QUANTIFICATION SETTINGS (SECOND INTERNAL STANDARD)"] - 1):
                            line_data = str(loaded_lines[i].strip())
                            splitted_data = line_data.split(";")

                            if len(splitted_data) == 1:
                                splitted_data = line_data.split(",")

                            if index == 0:
                                self.container_var[key_setting]["Quantification Method"] = tk.IntVar()
                                self.container_var[key_setting]["Quantification Method"].set(splitted_data[1])

                                if "Inclusion Intensity Calculation" not in self.container_var[key_setting]:
                                    self.container_var[key_setting]["Inclusion Intensity Calculation"] = tk.IntVar()
                                if len(splitted_data) == 3:
                                    self.container_var[key_setting]["Inclusion Intensity Calculation"].set(
                                        int(splitted_data[2]))
                                else:
                                    self.container_var[key_setting]["Inclusion Intensity Calculation"].set(0)
                            else:
                                info_file = splitted_data[0]
                                info_amount = splitted_data[1]
                                info_matrix = splitted_data[2]
                                info_isotope = splitted_data[3]
                                info_concentration = splitted_data[4]

                                self.container_var["SMPL"][info_file]["Host Only Tracer"] = {
                                    "Name": tk.StringVar(), "Value": tk.StringVar(), "Matrix": tk.StringVar(),
                                    "Amount": tk.StringVar()}
                                self.container_var["SMPL"][info_file]["Host Only Tracer"]["Amount"].set(info_amount)
                                self.container_var["SMPL"][info_file]["Host Only Tracer"]["Matrix"].set(info_matrix)
                                self.container_var["SMPL"][info_file]["Host Only Tracer"]["Name"].set(info_isotope)
                                self.container_var["SMPL"][info_file]["Host Only Tracer"]["Value"].set(
                                    info_concentration)

                            index += 1

                        current_step = 66
                        self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
                        self.lbl_prg_spk.configure(text=str(current_step) + " %", anchor=tk.W)
                        ## QUANTIFICATION SETTINGS (SECOND INTERNAL STANDARD)
                        index = 0
                        if self.without_pypitzer == False:
                            keyword = "QUANTIFICATION SETTINGS (HALTER2002)"
                        else:
                            keyword = "MATRIX SETTINGS"

                        for i in range(index_container["QUANTIFICATION SETTINGS (SECOND INTERNAL STANDARD)"] + 1,
                                       index_container[keyword] - 1):
                            line_data = str(loaded_lines[i].strip())
                            splitted_data = line_data.split(";")

                            if len(splitted_data) == 1:
                                splitted_data = line_data.split(",")

                            if index == 0:
                                self.container_var[key_setting]["Quantification Method"] = tk.IntVar()
                                self.container_var[key_setting]["Quantification Method"].set(splitted_data[1])

                                if "Inclusion Intensity Calculation" not in self.container_var[key_setting]:
                                    self.container_var[key_setting]["Inclusion Intensity Calculation"] = tk.IntVar()
                                if len(splitted_data) == 3:
                                    self.container_var[key_setting]["Inclusion Intensity Calculation"].set(
                                        int(splitted_data[2]))
                                else:
                                    self.container_var[key_setting]["Inclusion Intensity Calculation"].set(0)
                            else:
                                if len(splitted_data) == 3:
                                    info_file = splitted_data[0]
                                    info_isotope = splitted_data[1]
                                    info_concentration = splitted_data[2]

                                    self.container_var["SMPL"][info_file]["Second Internal Standard"] = {
                                        "Name": tk.StringVar(), "Value": tk.StringVar()}
                                    self.container_var["SMPL"][info_file]["Second Internal Standard"]["Name"].set(
                                        info_isotope)
                                    self.container_var["SMPL"][info_file]["Second Internal Standard"]["Value"].set(
                                        info_concentration)

                            index += 1

                        current_step = 69
                        self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
                        self.lbl_prg_spk.configure(text=str(current_step) + " %", anchor=tk.W)
                        ## QUANTIFICATION SETTINGS (HALTER2002)
                        index = 0
                        if self.without_pypitzer == False:
                            for i in range(index_container["QUANTIFICATION SETTINGS (HALTER2002)"] + 1,
                                           index_container["QUANTIFICATION SETTINGS (BORISOVA2021)"] - 1):
                                line_data = str(loaded_lines[i].strip())
                                splitted_data = line_data.split(";")

                                if len(splitted_data) == 1:
                                    splitted_data = line_data.split(",")

                                if index == 0:
                                    val_method = int(splitted_data[1])
                                    if "Inclusion Intensity Calculation" not in self.container_var[key_setting]:
                                        self.container_var[key_setting]["Inclusion Intensity Calculation"] = tk.IntVar()
                                    self.container_var[key_setting]["Inclusion Intensity Calculation"].set(val_method)
                                else:
                                    filename_short = splitted_data[0]
                                    index_filename = self.container_lists["SMPL"]["Short"].index(filename_short)
                                    filename_long = self.container_lists["SMPL"]["Long"][index_filename]
                                    val_a_i = splitted_data[1]
                                    val_b_i = splitted_data[2]
                                    val_rho_host_i = splitted_data[3]
                                    val_rho_incl_i = splitted_data[4]
                                    val_r_i = splitted_data[5]
                                    if "Halter2002" not in self.container_var["SMPL"][filename_long]:
                                        self.container_var["SMPL"][filename_long]["Halter2002"] = {
                                            "a": tk.StringVar(), "b": tk.StringVar(), "rho(host)": tk.StringVar(),
                                            "rho(incl)": tk.StringVar(), "R": tk.StringVar()}
                                    self.container_var["SMPL"][filename_long]["Halter2002"]["a"].set(val_a_i)
                                    self.container_var["SMPL"][filename_long]["Halter2002"]["b"].set(val_b_i)
                                    self.container_var["SMPL"][filename_long]["Halter2002"]["rho(host)"].set(
                                        val_rho_host_i)
                                    self.container_var["SMPL"][filename_long]["Halter2002"]["rho(incl)"].set(
                                        val_rho_incl_i)
                                    self.container_var["SMPL"][filename_long]["Halter2002"]["R"].set(val_r_i)

                                index += 1

                        current_step = 72
                        self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
                        self.lbl_prg_spk.configure(text=str(current_step) + " %", anchor=tk.W)
                        ## QUANTIFICATION SETTINGS (BORISOVA2021)
                        index = 0
                        if self.without_pypitzer == False:
                            for i in range(index_container["QUANTIFICATION SETTINGS (BORISOVA2021)"] + 1,
                                           index_container["MATRIX SETTINGS"] - 1):
                                line_data = str(loaded_lines[i].strip())
                                splitted_data = line_data.split(";")

                                if len(splitted_data) == 1:
                                    splitted_data = line_data.split(",")

                                if index == 0:
                                    val_method = int(splitted_data[1])
                                    if "Inclusion Intensity Calculation" not in self.container_var[key_setting]:
                                        self.container_var[key_setting]["Inclusion Intensity Calculation"] = tk.IntVar()
                                    self.container_var[key_setting]["Inclusion Intensity Calculation"].set(val_method)
                                else:
                                    filename_short = splitted_data[0]
                                    index_filename = self.container_lists["SMPL"]["Short"].index(filename_short)
                                    filename_long = self.container_lists["SMPL"]["Long"][index_filename]
                                    val_r_host_i = splitted_data[1]
                                    val_r_incl_i = splitted_data[2]
                                    val_rho_host_i = splitted_data[3]
                                    val_rho_incl_i = splitted_data[4]
                                    if "Borisova2021" not in self.container_var["SMPL"][filename_long]:
                                        self.container_var["SMPL"][filename_long]["Borisova2021"] = {
                                            "R(host)": tk.StringVar(), "R(incl)": tk.StringVar(),
                                            "rho(host)": tk.StringVar(), "rho(incl)": tk.StringVar()}
                                    self.container_var["SMPL"][filename_long]["Borisova2021"]["R(host)"].set(
                                        val_r_host_i)
                                    self.container_var["SMPL"][filename_long]["Borisova2021"]["R(incl)"].set(
                                        val_r_incl_i)
                                    self.container_var["SMPL"][filename_long]["Borisova2021"]["rho(host)"].set(
                                        val_rho_host_i)
                                    self.container_var["SMPL"][filename_long]["Borisova2021"]["rho(incl)"].set(
                                        val_rho_incl_i)

                                index += 1

                        current_step = 75
                        self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
                        self.lbl_prg_spk.configure(text=str(current_step) + " %", anchor=tk.W)
                        ## MATRIX SETTINGS
                        for i in range(index_container["MATRIX SETTINGS"] + 1,
                                       index_container["DWELL TIME SETTINGS"] - 1):
                            line_std = str(loaded_lines[i].strip())
                            splitted_std = line_std.split(";")

                            if len(splitted_std) == 1:
                                splitted_std = line_std.split(",")

                            info_file = splitted_std[0]
                            info_isotope = splitted_std[1]
                            info_concentration = splitted_std[2]

                            if "Matrix Setup" not in self.container_var["SMPL"][info_file]:
                                self.container_var["SMPL"][info_file]["Matrix Setup"] = {
                                    "IS": {"Name": tk.StringVar(), "Concentration": tk.StringVar()},
                                    "Oxide": {"Name": tk.StringVar(), "Concentration": tk.StringVar()},
                                    "Element": {"Name": tk.StringVar(), "Concentration": tk.StringVar()}}
                            self.container_var["SMPL"][info_file]["Matrix Setup"]["IS"]["Name"].set(info_isotope)
                            self.container_var["SMPL"][info_file]["Matrix Setup"]["IS"]["Concentration"].set(
                                info_concentration)

                        current_step = 78
                        self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
                        self.lbl_prg_spk.configure(text=str(current_step) + " %", anchor=tk.W)
                        ## DWELL TIME SETTINGS
                        self.open_project_part_06(
                            key_setting=key_setting, index_container=index_container, loaded_lines=loaded_lines)
                        current_step = 81
                        self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
                        self.lbl_prg_spk.configure(text=str(current_step) + " %", anchor=tk.W)
                        ## INTERVAL SETTINGS
                        self.open_project_part_07(
                            key_setting=key_setting, index_container=index_container, loaded_lines=loaded_lines)
                        current_step = 84
                        self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
                        self.lbl_prg_spk.configure(text=str(current_step) + " %", anchor=tk.W)
                        ## SPIKE ELIMINATION
                        self.open_project_part_08(
                            key_setting=key_setting, index_container=index_container, loaded_lines=loaded_lines)
                        current_step = 87
                        self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
                        self.lbl_prg_spk.configure(text=str(current_step) + " %", anchor=tk.W)
                        ## EXPERIMENTAL DATA
                        self.open_project_part_09(
                            key_setting=key_setting, index_container=index_container, loaded_lines=loaded_lines,
                            filename=filename)
                        current_step = 100
                        self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
                        self.lbl_prg_spk.configure(text="Opening project finished!", anchor=tk.W)

                        if current_step >= 100:
                            self.lbl_prg_spk.configure(text="Opening project finished!", anchor=tk.W)
                            subwindow_progressbar.destroy()
                    elif self.pysills_mode == "MI":
                        ## PROJECT INFORMATION
                        self.open_project_part_01(
                            key_setting=key_setting, index_container=index_container, loaded_lines=loaded_lines)
                        current_step = 20
                        self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
                        self.lbl_prg_spk.configure(text=str(current_step) + " %", anchor=tk.W)
                        ## STANDARD FILES
                        self.open_project_part_02(
                            key_setting=key_setting, index_container=index_container, loaded_lines=loaded_lines)
                        current_step = 30
                        self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
                        self.lbl_prg_spk.configure(text=str(current_step) + " %", anchor=tk.W)
                        ## SAMPLE FILES
                        self.open_project_part_03(
                            key_setting=key_setting, index_container=index_container, loaded_lines=loaded_lines)
                        current_step = 40
                        self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
                        self.lbl_prg_spk.configure(text=str(current_step) + " %", anchor=tk.W)
                        ## ISOTOPES
                        self.open_project_part_04(
                            key_setting=key_setting, index_container=index_container, loaded_lines=loaded_lines)
                        current_step = 50
                        self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
                        self.lbl_prg_spk.configure(text=str(current_step) + " %", anchor=tk.W)
                        ## INCLUSION SETTINGS
                        self.open_project_part_05(
                            key_setting=key_setting, index_container=index_container, loaded_lines=loaded_lines)
                        current_step = 60
                        self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
                        self.lbl_prg_spk.configure(text=str(current_step) + " %", anchor=tk.W)
                        ## PYPITZER SETTINGS
                        if self.without_pypitzer == False:
                            for i in range(index_container["PYPITZER SETTINGS"] + 1,
                                           index_container["QUANTIFICATION SETTINGS (MATRIX-ONLY TRACER)"] - 1):
                                line_data = str(loaded_lines[i].strip())
                                splitted_data = line_data.split(";")

                                if len(splitted_data) == 1:
                                    splitted_data = line_data.split(",")

                                if splitted_data[0] == "Cations":
                                    self.container_lists["Selected Cations"].extend(splitted_data[1:])
                                elif splitted_data[0] == "Anions":
                                    self.container_lists["Selected Anions"].extend(splitted_data[1:])
                                elif splitted_data[0] == "Isotopes":
                                    for isotope in splitted_data[1:]:
                                        self.helper_checkbuttons["Isotopes"][isotope] = tk.IntVar()
                                        self.helper_checkbuttons["Isotopes"][isotope].set(1)
                                elif splitted_data[0] in self.container_lists["SMPL"]["Short"]:
                                    for filename_smpl_long in self.container_lists["SMPL"]["Long"]:
                                        var_last_compound = splitted_data[1]
                                        var_melting_temperature = splitted_data[2]

                                        self.container_var["SMPL"][filename_smpl_long]["Last compound"] = tk.StringVar()
                                        self.container_var["SMPL"][filename_smpl_long][
                                            "Melting temperature"] = tk.StringVar()
                                        self.container_var["SMPL"][filename_smpl_long]["Last compound"].set(
                                            var_last_compound)
                                        self.container_var["SMPL"][filename_smpl_long]["Melting temperature"].set(
                                            var_melting_temperature)

                        current_step = 63
                        self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
                        self.lbl_prg_spk.configure(text=str(current_step) + " %", anchor=tk.W)
                        ## QUANTIFICATION SETTINGS (MATRIX-ONLY TRACER)
                        index = 0
                        for i in range(index_container["QUANTIFICATION SETTINGS (MATRIX-ONLY TRACER)"] + 1,
                                       index_container["QUANTIFICATION SETTINGS (SECOND INTERNAL STANDARD)"] - 1):
                            line_data = str(loaded_lines[i].strip())
                            splitted_data = line_data.split(";")

                            if len(splitted_data) == 1:
                                splitted_data = line_data.split(",")

                            if index == 0:
                                self.container_var[key_setting]["Quantification Method"] = tk.IntVar()
                                self.container_var[key_setting]["Quantification Method"].set(splitted_data[1])

                                if "Inclusion Intensity Calculation" not in self.container_var[key_setting]:
                                    self.container_var[key_setting]["Inclusion Intensity Calculation"] = tk.IntVar()
                                if len(splitted_data) == 3:
                                    self.container_var[key_setting]["Inclusion Intensity Calculation"].set(
                                        int(splitted_data[2]))
                                else:
                                    self.container_var[key_setting]["Inclusion Intensity Calculation"].set(0)
                            else:
                                info_file = splitted_data[0]
                                info_amount = splitted_data[1]
                                info_matrix = splitted_data[2]
                                info_isotope = splitted_data[3]
                                info_concentration = splitted_data[4]

                                self.container_var["SMPL"][info_file]["Host Only Tracer"] = {
                                    "Name": tk.StringVar(), "Value": tk.StringVar(), "Matrix": tk.StringVar(),
                                    "Amount": tk.StringVar()}
                                self.container_var["SMPL"][info_file]["Host Only Tracer"]["Amount"].set(info_amount)
                                self.container_var["SMPL"][info_file]["Host Only Tracer"]["Matrix"].set(info_matrix)
                                self.container_var["SMPL"][info_file]["Host Only Tracer"]["Name"].set(info_isotope)
                                self.container_var["SMPL"][info_file]["Host Only Tracer"]["Value"].set(
                                    info_concentration)

                            index += 1

                        current_step = 66
                        self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
                        self.lbl_prg_spk.configure(text=str(current_step) + " %", anchor=tk.W)
                        ## QUANTIFICATION SETTINGS (SECOND INTERNAL STANDARD)
                        index = 0
                        if self.without_pypitzer == False:
                            keyword = "QUANTIFICATION SETTINGS (HALTER2002)"
                        else:
                            keyword = "MATRIX SETTINGS"

                        for i in range(index_container["QUANTIFICATION SETTINGS (SECOND INTERNAL STANDARD)"] + 1,
                                       index_container[keyword] - 1):
                            line_data = str(loaded_lines[i].strip())
                            splitted_data = line_data.split(";")

                            if len(splitted_data) == 1:
                                splitted_data = line_data.split(",")

                            if index == 0:
                                self.container_var[key_setting]["Quantification Method"] = tk.IntVar()
                                self.container_var[key_setting]["Quantification Method"].set(splitted_data[1])

                                if "Inclusion Intensity Calculation" not in self.container_var[key_setting]:
                                    self.container_var[key_setting]["Inclusion Intensity Calculation"] = tk.IntVar()
                                if len(splitted_data) == 3:
                                    self.container_var[key_setting]["Inclusion Intensity Calculation"].set(
                                        int(splitted_data[2]))
                                else:
                                    self.container_var[key_setting]["Inclusion Intensity Calculation"].set(0)
                            else:
                                info_file = splitted_data[0]
                                info_isotope = splitted_data[1]
                                info_concentration = splitted_data[2]

                                self.container_var["SMPL"][info_file]["Second Internal Standard"] = {
                                    "Name": tk.StringVar(), "Value": tk.StringVar()}
                                self.container_var["SMPL"][info_file]["Second Internal Standard"]["Name"].set(
                                    info_isotope)
                                self.container_var["SMPL"][info_file]["Second Internal Standard"]["Value"].set(
                                    info_concentration)

                            index += 1

                        current_step = 69
                        self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
                        self.lbl_prg_spk.configure(text=str(current_step) + " %", anchor=tk.W)
                        ## QUANTIFICATION SETTINGS (HALTER2002)
                        index = 0
                        if self.without_pypitzer == False:
                            for i in range(index_container["QUANTIFICATION SETTINGS (HALTER2002)"] + 1,
                                           index_container["QUANTIFICATION SETTINGS (BORISOVA2021)"] - 1):
                                line_data = str(loaded_lines[i].strip())
                                splitted_data = line_data.split(";")

                                if len(splitted_data) == 1:
                                    splitted_data = line_data.split(",")

                                if index == 0:
                                    val_method = int(splitted_data[1])
                                    if "Inclusion Intensity Calculation" not in self.container_var[key_setting]:
                                        self.container_var[key_setting]["Inclusion Intensity Calculation"] = tk.IntVar()
                                    self.container_var[key_setting]["Inclusion Intensity Calculation"].set(val_method)
                                else:
                                    filename_short = splitted_data[0]
                                    index_filename = self.container_lists["SMPL"]["Short"].index(filename_short)
                                    filename_long = self.container_lists["SMPL"]["Long"][index_filename]
                                    val_a_i = splitted_data[1]
                                    val_b_i = splitted_data[2]
                                    val_rho_host_i = splitted_data[3]
                                    val_rho_incl_i = splitted_data[4]
                                    val_r_i = splitted_data[5]
                                    if "Halter2002" not in self.container_var["SMPL"][filename_long]:
                                        self.container_var["SMPL"][filename_long]["Halter2002"] = {
                                            "a": tk.StringVar(), "b": tk.StringVar(), "rho(host)": tk.StringVar(),
                                            "rho(incl)": tk.StringVar(), "R": tk.StringVar()}
                                    self.container_var["SMPL"][filename_long]["Halter2002"]["a"].set(val_a_i)
                                    self.container_var["SMPL"][filename_long]["Halter2002"]["b"].set(val_b_i)
                                    self.container_var["SMPL"][filename_long]["Halter2002"]["rho(host)"].set(
                                        val_rho_host_i)
                                    self.container_var["SMPL"][filename_long]["Halter2002"]["rho(incl)"].set(
                                        val_rho_incl_i)
                                    self.container_var["SMPL"][filename_long]["Halter2002"]["R"].set(val_r_i)

                                index += 1

                        current_step = 72
                        self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
                        self.lbl_prg_spk.configure(text=str(current_step) + " %", anchor=tk.W)
                        ## QUANTIFICATION SETTINGS (BORISOVA2021)
                        index = 0
                        if self.without_pypitzer == False:
                            for i in range(index_container["QUANTIFICATION SETTINGS (BORISOVA2021)"] + 1,
                                           index_container["MATRIX SETTINGS"] - 1):
                                line_data = str(loaded_lines[i].strip())
                                splitted_data = line_data.split(";")

                                if len(splitted_data) == 1:
                                    splitted_data = line_data.split(",")

                                if index == 0:
                                    val_method = int(splitted_data[1])
                                    if "Inclusion Intensity Calculation" not in self.container_var[key_setting]:
                                        self.container_var[key_setting]["Inclusion Intensity Calculation"] = tk.IntVar()
                                    self.container_var[key_setting]["Inclusion Intensity Calculation"].set(val_method)
                                else:
                                    filename_short = splitted_data[0]
                                    index_filename = self.container_lists["SMPL"]["Short"].index(filename_short)
                                    filename_long = self.container_lists["SMPL"]["Long"][index_filename]
                                    val_r_host_i = splitted_data[1]
                                    val_r_incl_i = splitted_data[2]
                                    val_rho_host_i = splitted_data[3]
                                    val_rho_incl_i = splitted_data[4]
                                    if "Borisova2021" not in self.container_var["SMPL"][filename_long]:
                                        self.container_var["SMPL"][filename_long]["Borisova2021"] = {
                                            "R(host)": tk.StringVar(), "R(incl)": tk.StringVar(),
                                            "rho(host)": tk.StringVar(), "rho(incl)": tk.StringVar()}
                                    self.container_var["SMPL"][filename_long]["Borisova2021"]["R(host)"].set(
                                        val_r_host_i)
                                    self.container_var["SMPL"][filename_long]["Borisova2021"]["R(incl)"].set(
                                        val_r_incl_i)
                                    self.container_var["SMPL"][filename_long]["Borisova2021"]["rho(host)"].set(
                                        val_rho_host_i)
                                    self.container_var["SMPL"][filename_long]["Borisova2021"]["rho(incl)"].set(
                                        val_rho_incl_i)

                                index += 1

                        current_step = 75
                        self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
                        self.lbl_prg_spk.configure(text=str(current_step) + " %", anchor=tk.W)
                        ## MATRIX SETTINGS
                        for i in range(index_container["MATRIX SETTINGS"] + 1,
                                       index_container["DWELL TIME SETTINGS"] - 1):
                            line_std = str(loaded_lines[i].strip())
                            splitted_std = line_std.split(";")

                            if len(splitted_data) == 1:
                                splitted_data = line_data.split(",")

                            info_file = splitted_std[0]
                            info_isotope = splitted_std[1]
                            info_concentration = splitted_std[2]

                            if "Matrix Setup" not in self.container_var["SMPL"][info_file]:
                                self.container_var["SMPL"][info_file]["Matrix Setup"] = {
                                    "IS": {"Name": tk.StringVar(), "Concentration": tk.StringVar()},
                                    "Oxide": {"Name": tk.StringVar(), "Concentration": tk.StringVar()},
                                    "Element": {"Name": tk.StringVar(), "Concentration": tk.StringVar()}}
                            self.container_var["SMPL"][info_file]["Matrix Setup"]["IS"]["Name"].set(info_isotope)
                            self.container_var["SMPL"][info_file]["Matrix Setup"]["IS"]["Concentration"].set(
                                info_concentration)

                        current_step = 78
                        self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
                        self.lbl_prg_spk.configure(text=str(current_step) + " %", anchor=tk.W)
                        ## DWELL TIME SETTINGS
                        self.open_project_part_06(
                            key_setting=key_setting, index_container=index_container, loaded_lines=loaded_lines)
                        current_step = 81
                        self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
                        self.lbl_prg_spk.configure(text=str(current_step) + " %", anchor=tk.W)
                        ## INTERVAL SETTINGS
                        self.open_project_part_07(
                            key_setting=key_setting, index_container=index_container, loaded_lines=loaded_lines)
                        current_step = 84
                        self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
                        self.lbl_prg_spk.configure(text=str(current_step) + " %", anchor=tk.W)
                        ## SPIKE ELIMINATION
                        self.open_project_part_08(
                            key_setting=key_setting, index_container=index_container, loaded_lines=loaded_lines)
                        current_step = 87
                        self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
                        self.lbl_prg_spk.configure(text=str(current_step) + " %", anchor=tk.W)
                        ## EXPERIMENTAL DATA
                        self.open_project_part_09(
                            key_setting=key_setting, index_container=index_container, loaded_lines=loaded_lines,
                            filename=filename)
                        current_step = 100
                        self.update_progress(parent=subwindow_progressbar, variable=prgbar, value=current_step)
                        self.lbl_prg_spk.configure(text="Opening project finished!", anchor=tk.W)

                        if current_step >= 100:
                            self.lbl_prg_spk.configure(text="Opening project finished!", anchor=tk.W)
                            subwindow_progressbar.destroy()

                    # Initialization
                    time_start = datetime.datetime.now()
                    self.file_loaded = True
                    self.demo_mode = False

                    if self.old_file == True and self.var_opt_icp.get() == "Select ICP-MS":
                        self.container_icpms["name"] = None
                    else:
                        self.select_icp_ms(var_opt=self.var_opt_icp)

                    if self.pysills_mode == "MA":
                        self.ma_settings()
                    elif self.pysills_mode in ["FI", "INCL"]:
                        self.fi_settings()
                    elif self.pysills_mode == "MI":
                        self.mi_settings()

                    time_end = datetime.datetime.now()
                    time_delta = (time_end - time_start)*1000
                    print(f"Process time (opening project - part 'Initialization'):", time_delta.total_seconds(), "ms")
                except FileNotFoundError:
                    print("File not found!")

    def convert_true_false_string(self, var_str):
        d = {"True": True, "False": False}
        return d.get(var_str, var_str)

    def delete_csv(self, var_lb, var_list):
        if var_lb == self.lb_std:
            var_list = self.list_std
        elif var_lb == self.lb_smpl:
            var_list = self.list_smpl
        item = var_lb.curselection()
        var_list.remove(var_list[item[0]])
        var_lb.delete(tk.ANCHOR)
        #
        if var_lb == self.lb_std:
            del self.container_lists["STD"]["Long"][item[0]]
            del self.container_lists["STD"]["Short"][item[0]]
        elif var_lb == self.lb_smpl:
            del self.container_lists["SMPL"]["Long"][item[0]]
            del self.container_lists["SMPL"]["Short"][item[0]]

    def restart_pysills(self):
        self.parent.destroy()
        root = tk.Tk()
        screen_width = root.winfo_screenwidth()
        screen_height = root.winfo_screenheight()
        PySILLS(
            parent=root, var_screen_width=screen_width, var_screen_height=screen_height, restart=True,
            last_used_directory=self.last_used_directory, last_used_directory_input=self.last_used_directory_input)
        root.mainloop()

    ####################
    ## DATA PROCESSING #
    ####################


    def find_icpms_data_in_file(self, filename_long, filetype=None, only_time_check=False):
        str_filename_long = filename_long
        str_filename_short = str_filename_long.split("/")[-1]
        potential_isotopes = ["Na23", "Al27", "Si29", "P31", "S32", "S33", "K39", "Ca43", "Ca44", "Fe57", "Pb208",
                              "U238"]
        list_indices = []
        var_timestamp = None

        if only_time_check == True:
            df = None
            file_info = None
            condition = False
            with open(str_filename_long, "r", encoding="utf-8", errors="ignore") as file:
                for lineno, line in enumerate(file, start=0):
                    if condition:
                        break
                    for isotope in potential_isotopes:
                        if condition:
                            break
                        if "Acquired" in line:
                            key_start = re.search(
                                r"Acquired\s+\:\s+(\d+)\/(\d+)\/(\d+)\s+(\d+)\:(\d+)\:(\d+)( using Batch )(\w+)", line)
                            if key_start:
                                str_acquisition = [
                                    str(key_start.group(4)), str(key_start.group(5)), str(key_start.group(6))]
                                if filetype != None:
                                    self.container_var["acquisition times"][filetype][str_filename_short] = tk.StringVar()
                                    self.container_var["acquisition times"][filetype][str_filename_short].set(
                                        str_acquisition[0] + ":" + str_acquisition[1] + ":" + str_acquisition[2])
                            var_timestamp = lineno
                            condition = True
                        if isotope in line:
                            list_indices.append(lineno)
                            condition = True
        else:
            with open(str_filename_long, "r", encoding="utf-8", errors="ignore") as file:
                for lineno, line in enumerate(file, start=0):
                    for isotope in potential_isotopes:
                        if isotope in line:
                            list_indices.append(lineno)
                        if "Acquired" in line:
                            key_start = re.search(
                                r"Acquired\s+\:\s+(\d+)\/(\d+)\/(\d+)\s+(\d+)\:(\d+)\:(\d+)( using Batch )(\w+)", line)
                            if key_start:
                                str_acquisition = [
                                    str(key_start.group(4)), str(key_start.group(5)), str(key_start.group(6))]
                                if filetype != None:
                                    self.container_var["acquisition times"][filetype][str_filename_short] = tk.StringVar()
                                    self.container_var["acquisition times"][filetype][str_filename_short].set(
                                        str_acquisition[0] + ":" + str_acquisition[1] + ":" + str_acquisition[2])
                            var_timestamp = lineno

        # list_indices = []
        #
        # try:
        #     file = pd.read_csv(str_filename_long, sep=",", engine="python")
        # except:
        #     file = pd.read_csv(str_filename_long, sep=";", engine="python")
        #
        # for isotope in potential_isotopes:
        #     a = file[file.columns[file.dtypes == "object"]].apply(lambda x: " ".join(x), axis=1).str.contains(isotope)
        #     matches = file.index[a]
        #     if len(matches) > 0:
        #         list_indices.append(matches[0])
        #
        # var_skipheader = int(list_indices[0] + 1)
        if only_time_check == False:
            var_skipheader = int(list_indices[0])
            var_skipfooter = 0

            for skip_footer in range(10):
                try:
                    df = DE(filename_long=str_filename_long).get_measurements(
                        delimiter=",", skip_header=var_skipheader, skip_footer=skip_footer)
                except:
                    var_skipfooter += 1

            df = DE(filename_long=str_filename_long).get_measurements(
                        delimiter=",", skip_header=var_skipheader, skip_footer=var_skipfooter)
            file_info = {"skipheader": var_skipheader, "skipfooter": var_skipfooter, "delimiter": ",",
                         "timestamp": var_timestamp}

        return df, file_info

    def get_delimiter(self, var_line):
        n_commas = 0
        n_semicolons = 0

        n_commas += var_line.count(",")
        n_semicolons += var_line.count(";")

        if n_commas > 0 and n_semicolons == 0:
            delimiter = ","
        elif n_commas == 0 and n_semicolons > 0:
            delimiter = ";"

        return delimiter

    def open_csv(self, datatype):
        if datatype == "STD":
            if "Default_STD_01.csv" in self.list_std:
                self.list_std.clear()
            var_list = self.list_std
            var_listbox = self.lb_std
        elif datatype == "SMPL":
            if "Default_SMPL_01.csv" in self.list_smpl:
                self.list_smpl.clear()
            var_list = self.list_smpl
            var_listbox = self.lb_smpl

        filename = filedialog.askopenfilenames(
            parent=self.parent,
            filetypes=(("LA-ICP-MS files", "*.csv *.FIN2 *.xl *.txt"), ("csv files", "*.csv"), ("FIN2 files", "*.FIN2"),
                       ("xl files", "*.xl"), ("txt files", "*.txt"), ("all files", "*.*")),
            initialdir=self.last_used_directory_input)

        split_filename = filename[0].split("/")

        self.last_used_directory_input = '/'.join(split_filename[:-1])

        for i in filename:
            if i not in var_list:
                var_list.append(i)
                file_parts = i.split("/")
                var_listbox.insert(tk.END, file_parts[-1])
                str_filename_long = i
                str_filename_short = file_parts[-1]

                if self.container_icpms["name"] != None:
                    var_skipheader = self.container_icpms["skipheader"]
                    var_skipfooter = self.container_icpms["skipfooter"]
                    df_exmpl = DE(filename_long=str_filename_long).get_measurements(
                        delimiter=",", skip_header=var_skipheader, skip_footer=var_skipfooter)
                else:
                    df_exmpl, file_info = self.find_icpms_data_in_file(filename_long=str_filename_long)
                    var_skipheader = file_info["skipheader"]
                    var_skipfooter = file_info["skipfooter"]
                    self.container_icpms["skipheader"] = file_info["skipheader"]
                    self.container_icpms["skipfooter"] = file_info["skipfooter"]

                    if var_skipheader == 3 and var_skipfooter == 1:
                        try:
                            dates, times = Data(filename=str_filename_long).import_as_list(
                                skip_header=3, skip_footer=1, timestamp=2, icpms="Agilent 7900s")
                            var_timestamp = 2
                            self.container_icpms["timestamp"] = var_timestamp
                            self.container_icpms["name"] = "Agilent 7900s"
                        except:
                            self.container_icpms["name"] = "Undefined ICP-MS"
                    elif var_skipheader == 3 and var_skipfooter == 0:
                        try:
                            dates, times = Data(filename=str_filename_long).import_as_list(
                                skip_header=3, skip_footer=1, timestamp=2, icpms="Agilent 7900s")
                            var_timestamp = 2
                            self.container_icpms["timestamp"] = var_timestamp
                            self.container_icpms["name"] = "Agilent 7900s"
                        except:
                            self.container_icpms["name"] = "Undefined ICP-MS"
                    else:
                        self.container_icpms["name"] = "Undefined ICP-MS"

                    self.var_opt_icp.set(self.container_icpms["name"])

                if "Dataframe" not in self.container_measurements:
                    self.container_measurements["Dataframe"] = {}
                if str_filename_short not in self.container_measurements["Dataframe"]:
                    self.container_measurements["Dataframe"][str_filename_short] = df_exmpl

                self.times = DE().get_times(dataframe=df_exmpl)
                df_isotopes = DE().get_isotopes(dataframe=df_exmpl)
                self.container_lists["Measured Isotopes"][str_filename_short] = df_isotopes
                file_isotopes = df_isotopes

                self.add_needed_variables_for_later_added_files(
                    filename_long=str_filename_long, filename_short=str_filename_short, filetype=datatype,
                    file_isotopes=file_isotopes)

                if (self.container_icpms["name"] != None and
                        self.container_icpms["name"] not in ["PerkinElmer Syngistix"]):
                    var_skipheader = self.container_icpms["skipheader"]
                    var_skipfooter = self.container_icpms["skipfooter"]
                    var_timestamp = self.container_icpms["timestamp"]
                    var_icpms = self.container_icpms["name"]
                    dates, times = Data(filename=str_filename_long).import_as_list(
                        skip_header=var_skipheader, skip_footer=var_skipfooter, timestamp=var_timestamp,
                        icpms=var_icpms)
                elif self.container_icpms["name"] == "PerkinElmer Syngistix":
                    create_time = os.path.getctime(str_filename_long)
                    create_date = datetime.datetime.fromtimestamp(create_time)
                    times = [[str(create_date.hour), str(create_date.minute), str(create_date.second)]]
                    dates = [str(create_date.year), str(create_date.month), str(create_date.day)]
                else:
                    dates, times = Data(filename=str_filename_long).import_as_list(
                        skip_header=3, skip_footer=1, timestamp=2,
                        icpms="Agilent 7900s")

                if str_filename_short not in self.container_var["acquisition times"][datatype]:
                    self.container_var["acquisition times"][datatype][str_filename_short] = tk.StringVar()
                    self.container_var["acquisition times"][datatype][str_filename_short].set(
                        times[0][0] + ":" + times[0][1] + ":" + times[0][2])

        self.demo_mode = False

    def create_dwell_time_window(self):
        font_color_dark = self.bg_colors["Dark Font"]
        font_color_light = self.bg_colors["Light Font"]
        background_color_dark = self.bg_colors["BG Window"]
        background_color_elements = self.bg_colors["Light"]
        background_color_light = self.bg_colors["Very Light"]
        accent_color = self.bg_colors["Accent"]
        font_header = self.font_settings["Header"]
        font_elements = self.font_settings["Elements"]
        font_option = self.font_settings["Options"]
        font_table = self.font_settings["Table"]

        str_title = self.language_dict["Dwell Time Settings"][self.var_language]
        str_lbl_01 = self.language_dict["Dwell times (in s)"][self.var_language]
        str_lbl_02 = self.language_dict["Default settings"][self.var_language]
        ## Window Settings
        window_dwell = tk.Toplevel(self.parent)
        window_dwell.title(str_title)
        window_dwell.geometry("300x450+0+0")
        window_dwell.resizable(False, False)
        window_dwell["bg"] = background_color_dark

        window_width = 300
        window_height = 450
        row_min = 25
        n_rows = int(window_height/row_min)
        column_min = 20
        n_columns = int(window_width/column_min)

        for x in range(n_columns):
            tk.Grid.columnconfigure(window_dwell, x, weight=1)
        for y in range(n_rows):
            tk.Grid.rowconfigure(window_dwell, y, weight=1)

        # Rows
        for i in range(0, n_rows):
            window_dwell.grid_rowconfigure(i, minsize=row_min)
        # Columns
        for i in range(0, n_columns):
            window_dwell.grid_columnconfigure(i, minsize=column_min)

        ## Labels
        lbl_01 = SE(
            parent=window_dwell, row_id=0, column_id=0, n_rows=1, n_columns=14,
            fg=font_color_light, bg=background_color_dark).create_simple_label(
            text=str_lbl_01, relief=tk.FLAT, fontsize="sans 10 bold")
        lbl_02 = SE(
            parent=window_dwell, row_id=1, column_id=0, n_rows=1, n_columns=7, fg=font_color_dark,
            bg=background_color_elements).create_simple_label(
            text=str_lbl_02, relief=tk.FLAT, fontsize="sans 10 bold")

        self.container_elements["dwell_times"]["Label"].extend([lbl_01, lbl_02])

        if self.container_var["dwell_times"]["Entry"]["Default"].get() != "0.002":
            var_text = self.container_var["dwell_times"]["Entry"]["Default"].get()
        else:
            var_text = "0.002"

        entr_dwell = SE(
            parent=window_dwell, row_id=1, column_id=7, n_rows=1, n_columns=7,
            fg=font_color_dark, bg=self.bg_colors["White"]).create_simple_entry(
            var=self.container_var["dwell_times"]["Entry"]["Default"], text_default=var_text,
            command=lambda event, var_isotope=None, mode="Default":
            self.change_dwell_times(var_isotope, mode, event))

        self.container_elements["dwell_times"]["Entry"].append(entr_dwell)

        ## TREEVIEWS
        frm_dwell = SE(
            parent=window_dwell, row_id=2, column_id=0, n_rows=15, n_columns=14, fg=font_color_dark,
            bg=background_color_light).create_frame()
        vsb_dwell = ttk.Scrollbar(master=frm_dwell, orient="vertical")
        text_dwell = tk.Text(
            master=frm_dwell, width=30, height=25, yscrollcommand=vsb_dwell.set, bg=background_color_light)
        vsb_dwell.config(command=text_dwell.yview)
        vsb_dwell.pack(side="right", fill="y")
        text_dwell.pack(side="left", fill="both", expand=True)

        for var_isotope in self.container_lists["Measured Isotopes"]["All"]:
            lbl_i = tk.Label(
                frm_dwell, text=var_isotope, bg=background_color_light, fg=font_color_dark)
            text_dwell.window_create("end", window=lbl_i)
            text_dwell.insert("end", "\t")

            entr_i = tk.Entry(
                frm_dwell, textvariable=self.container_var["dwell_times"]["Entry"][var_isotope],
                fg=font_color_dark, bg=self.bg_colors["White"], highlightthickness=0,
                highlightbackground=background_color_light, width=12)
            text_dwell.window_create("insert", window=entr_i)
            text_dwell.insert("end", "\n")

            if var_isotope not in self.container_var["Spike Elimination"]["Threshold"]:
                self.container_var["Spike Elimination"]["Threshold"][var_isotope] = tk.StringVar()

    def change_dwell_times(self, var_isotope, mode, event):
        if mode == "Default":
            value = self.container_var["dwell_times"]["Entry"]["Default"].get()
            for isotope in self.container_lists["ISOTOPES"]:
                self.container_var["dwell_times"]["Entry"][isotope].set(value)
        elif mode == "Specific":
            value = self.container_var["dwell_times"]["Entry"][var_isotope].get()
            self.container_var["dwell_times"]["Entry"][var_isotope].set(value)

    def create_spike_elimination_threshold_window(self):
        """Check-up window to control the spike elimination threshold setup."""
        # Colors
        font_color_dark = self.bg_colors["Dark Font"]
        font_color_light = self.bg_colors["Light Font"]
        background_color_dark = self.bg_colors["BG Window"]
        background_color_elements = self.bg_colors["Light"]
        background_color_light = self.bg_colors["Very Light"]
        accent_color = self.bg_colors["Accent"]  # self.accent_color
        font_header = self.font_settings["Header"]
        font_elements = self.font_settings["Elements"]
        font_option = self.font_settings["Options"]
        font_table = self.font_settings["Table"]

        ## Window Settings
        row_min = self.row_height
        column_min = self.column_width
        n_rows = self.window_dimensions["Spike elimination threshold"][0]
        n_columns = self.window_dimensions["Spike elimination threshold"][1]

        window_width = int(n_columns*self.column_width)
        window_height = int(n_rows*self.row_height)

        var_geometry = str(window_width) + "x" + str(window_height) + "+" + str(0) + "+" + str(0)

        subwindow_se_threshold = tk.Toplevel(self.parent)
        subwindow_se_threshold.title("Spike Elimination")
        subwindow_se_threshold.geometry(var_geometry)
        subwindow_se_threshold.resizable(False, False)
        subwindow_se_threshold["bg"] = background_color_dark

        for x in range(n_columns):
            tk.Grid.columnconfigure(subwindow_se_threshold, x, weight=1)
        for y in range(n_rows):
            tk.Grid.rowconfigure(subwindow_se_threshold, y, weight=1)

        # Rows
        for i in range(0, n_rows):
            subwindow_se_threshold.grid_rowconfigure(i, minsize=row_min)
        # Columns
        for i in range(0, n_columns):
            subwindow_se_threshold.grid_columnconfigure(i, minsize=column_min)

        if self.pysills_mode == "MA":
            var_setting_key = "ma_setting"
            var_threshold = self.container_var[var_setting_key]["SE Threshold"]
        elif self.pysills_mode in ["FI", "INCL"]:
            var_setting_key = "fi_setting"
            var_threshold = self.container_var[var_setting_key]["SE Threshold"]
        elif self.pysills_mode == "MI":
            var_setting_key = "mi_setting"
            var_threshold = self.container_var[var_setting_key]["SE Threshold"]

        var_row_start = 0
        var_column_start = 0
        var_header_n = 14
        int_category_n = 8

        ## LABELS
        lbl_01 = SE(
            parent=subwindow_se_threshold, row_id=var_row_start, column_id=var_column_start, n_rows=1,
            n_columns=var_header_n, fg=font_color_light, bg=background_color_dark).create_simple_label(
            text="Spike Elimination - Threshold Setup", relief=tk.FLAT, fontsize="sans 10 bold", anchor=tk.W)
        lbl_01 = SE(
            parent=subwindow_se_threshold, row_id=var_row_start + 1, column_id=var_column_start, n_rows=1,
            n_columns=int_category_n, fg=font_color_dark, bg=background_color_elements).create_simple_label(
            text="Default threshold value", relief=tk.FLAT, fontsize="sans 10 bold", anchor=tk.W)
        lbl_02 = SE(
            parent=subwindow_se_threshold, row_id=var_row_start + 2, column_id=var_column_start, n_rows=1,
            n_columns=var_header_n, fg=font_color_light,
            bg=self.bg_colors["BG Window"]).create_simple_label(
            text="Isotope-specific threshold value", relief=tk.FLAT, fontsize="sans 10 bold", anchor=tk.W)

        ## ENTRIES
        entr_01 = SE(
            parent=subwindow_se_threshold, row_id=var_row_start + 1, column_id=int_category_n, n_rows=1,
            n_columns=var_header_n - int_category_n, fg=font_color_dark,
            bg=self.bg_colors["White"]).create_simple_entry(var=var_threshold, text_default=var_threshold.get())
        entr_01.bind( "<Return>", lambda event, variable=var_threshold: self.update_threshold_value(variable, event))

        ## BUTTONS
        if self.var_os == "darwin":
            font_color_accent = font_color_dark
        else:
            font_color_accent = font_color_light

        btn_03 = SE(
            parent=subwindow_se_threshold, row_id=n_rows - 2, column_id=var_header_n - int_category_n, n_rows=1,
            n_columns=int_category_n, fg=font_color_accent, bg=accent_color).create_simple_button(
            text="Update values", bg_active=accent_color, fg_active=font_color_accent,
            command=lambda mode="specific": self.calculate_threshold_spike_elimination(mode))

        ## TREEVIEWS
        frm_se_threshold = SE(
            parent=subwindow_se_threshold, row_id=var_row_start + 3, column_id=var_column_start, n_rows=n_rows - 5,
            n_columns=var_header_n, fg=font_color_dark, bg=background_color_light).create_frame()
        vsb_se_threshold = ttk.Scrollbar(master=frm_se_threshold, orient="vertical")
        text_se_threshold = tk.Text(
            master=frm_se_threshold, width=30, height=25, yscrollcommand=vsb_se_threshold.set,
            bg=background_color_light)
        vsb_se_threshold.config(command=text_se_threshold.yview)
        vsb_se_threshold.pack(side="right", fill="y")
        text_se_threshold.pack(side="left", fill="both", expand=True)

        for index, var_isotope in enumerate(self.container_lists["Measured Isotopes"]["All"]):
            if index == 0:
                lbl_i = tk.Label(
                    frm_se_threshold, text="Isotope" + "\t" + "Dwell time" + "\t" + "Threshold",
                    bg=background_color_light, fg=font_color_dark)
                text_se_threshold.window_create("end", window=lbl_i)
                text_se_threshold.insert("end", "\n")

                lbl_i = tk.Label(
                    frm_se_threshold, text=var_isotope, bg=background_color_light, fg=font_color_dark)
                text_se_threshold.window_create("end", window=lbl_i)
                text_se_threshold.insert("end", "\t")

                if var_isotope not in self.container_var["Spike Elimination"]["Threshold"]:
                    self.container_var["Spike Elimination"]["Threshold"][var_isotope] = tk.StringVar()
                    self.container_var["Spike Elimination"]["Threshold"][var_isotope].set(var_threshold.get())

                entr_i = tk.Entry(
                    frm_se_threshold, textvariable=self.container_var["dwell_times"]["Entry"][var_isotope],
                    fg=font_color_dark, bg=self.bg_colors["White"], highlightthickness=0,
                    highlightbackground=background_color_light, width=8)
                text_se_threshold.window_create("insert", window=entr_i)
                text_se_threshold.insert("end", "\t")

                entr_i = tk.Entry(
                    frm_se_threshold, textvariable=self.container_var["Spike Elimination"]["Threshold"][var_isotope],
                    fg=font_color_dark, bg=self.bg_colors["White"], highlightthickness=0,
                    highlightbackground=background_color_light, width=8)
                text_se_threshold.window_create("insert", window=entr_i)
                text_se_threshold.insert("end", "\n")
            else:
                lbl_i = tk.Label(
                    frm_se_threshold, text=var_isotope, bg=background_color_light, fg=font_color_dark)
                text_se_threshold.window_create("end", window=lbl_i)
                text_se_threshold.insert("end", "\t")

                if var_isotope not in self.container_var["Spike Elimination"]["Threshold"]:
                    self.container_var["Spike Elimination"]["Threshold"][var_isotope] = tk.StringVar()
                    self.container_var["Spike Elimination"]["Threshold"][var_isotope].set(var_threshold.get())

                entr_i = tk.Entry(
                    frm_se_threshold, textvariable=self.container_var["dwell_times"]["Entry"][var_isotope],
                    fg=font_color_dark, bg=self.bg_colors["White"], highlightthickness=0,
                    highlightbackground=background_color_light, width=8)
                text_se_threshold.window_create("insert", window=entr_i)
                text_se_threshold.insert("end", "\t")

                entr_i = tk.Entry(
                    frm_se_threshold, textvariable=self.container_var["Spike Elimination"]["Threshold"][var_isotope],
                    fg=font_color_dark, bg=self.bg_colors["White"], highlightthickness=0,
                    highlightbackground=background_color_light, width=8)
                text_se_threshold.window_create("insert", window=entr_i)
                text_se_threshold.insert("end", "\n")

    def update_threshold_value(self, variable, event):
        val_threshold_default = int(variable.get())

        if val_threshold_default < 0:
            val_threshold_default = abs(val_threshold_default)

        variable.set(val_threshold_default)

        for isotope in self.container_lists["Measured Isotopes"]["All"]:
            self.container_var["Spike Elimination"]["Threshold"][isotope].set(val_threshold_default)
            val_dwell_time = 10/val_threshold_default
            self.container_var["dwell_times"]["Entry"][isotope].set(val_dwell_time)

    def detect_signal_interval(self, mode="BG"):
        if self.pysills_mode == "MA":
            key_setting = "ma_setting"
        elif self.pysills_mode in ["FI", "INCL"]:
            key_setting = "fi_setting"
        elif self.pysills_mode == "MI":
            key_setting = "mi_setting"

        data_interval_detection = {}
        data_calc = {}

        list_files = []
        list_files.extend(self.container_lists["STD"]["Long"])
        if self.pysills_mode == "MA" or mode == "BG":
            list_files.extend(self.container_lists["SMPL"]["Long"])

        for file_long in list_files:
            file_short = file_long.split("/")[-1]
            if self.file_loaded == False:
                if self.container_icpms["name"] != None:
                    var_skipheader = self.container_icpms["skipheader"]
                    var_skipfooter = self.container_icpms["skipfooter"]
                    df_data = DE(filename_long=file_long).get_measurements(
                        delimiter=",", skip_header=var_skipheader, skip_footer=var_skipfooter)
                else:
                    df_data = DE(filename_long=file_long).get_measurements(
                        delimiter=",", skip_header=3, skip_footer=1)
                dataset_time = list(DE().get_times(dataframe=df_data))
            else:
                if "Time" not in self.container_measurements["RAW"][file_short] and "_copy" in file_short:
                    file_short_original = file_short.replace("_copy", "")
                    self.container_measurements["RAW"][file_short]["Time"] = self.container_measurements["RAW"][
                        file_short_original]["Time"]
                dataset_time = self.container_measurements["RAW"][file_short]["Time"]
                df_data = self.container_measurements["RAW"][file_short]

            condition = False
            index_isotope = 0
            if file_long in self.container_lists["STD"]["Long"]:
                var_srm_file = self.container_var["STD"][file_long]["SRM"].get()
                index = self.container_lists["STD"]["Long"].index(file_long)
                file_short = self.container_lists["STD"]["Short"][index]

                for element, value in sorted(
                        self.srm_actual[var_srm_file].items(), key=lambda item: item[1], reverse=True):
                    if element in self.container_lists["Measured Elements"][file_short]:
                        isotope_is = self.container_lists["Measured Elements"][file_short][element][0]
                    break
            else:
                isotope_is = self.container_var["SMPL"][file_long]["IS Data"]["IS"].get()

            if isotope_is == "Select IS":
                file_dummy = file_long
                file_dummy_short = file_dummy.split("/")[-1]
                if self.file_loaded == False:
                    if self.container_icpms["name"] != None:
                        var_skipheader = self.container_icpms["skipheader"]
                        var_skipfooter = self.container_icpms["skipfooter"]
                        df_data_dummy = DE(filename_long=file_dummy).get_measurements(
                            delimiter=",", skip_header=var_skipheader, skip_footer=var_skipfooter)
                    else:
                        df_data_dummy = DE(filename_long=file_dummy).get_measurements(
                            delimiter=",", skip_header=3, skip_footer=1)
                else:
                    df_data_dummy = self.container_measurements["RAW"][file_dummy_short]

                max_values = {"Last": 0}
                for isotope in list(df_data_dummy.keys())[1:]:
                    dataset_isotope = list(df_data[isotope])
                    if self.container_var["General Settings"]["Desired Average"].get() == 1:
                        if np.mean(dataset_isotope) > max_values["Last"]:
                            max_values.clear()
                            max_values[isotope] = np.mean(dataset_isotope)
                            max_values["Last"] = np.mean(dataset_isotope)
                    else:
                        if np.median(dataset_isotope) > max_values["Last"]:
                            max_values.clear()
                            max_values[isotope] = np.median(dataset_isotope)
                            max_values["Last"] = np.median(dataset_isotope)

                del max_values["Last"]
                isotope_is = list(max_values.keys())[0]
            else:
                max_values = {"Last": 0}

                for isotope in list(df_data.keys())[1:]:
                    dataset_isotope = list(df_data[isotope])
                    if self.container_var["General Settings"]["Desired Average"].get() == 1:
                        if np.mean(dataset_isotope) > max_values["Last"]:
                            max_values.clear()
                            max_values[isotope] = np.mean(dataset_isotope)
                            max_values["Last"] = np.mean(dataset_isotope)
                    else:
                        if np.median(dataset_isotope) > max_values["Last"]:
                            max_values.clear()
                            max_values[isotope] = np.median(dataset_isotope)
                            max_values["Last"] = np.median(dataset_isotope)

                del max_values["Last"]
                isotope_is = list(max_values.keys())[0]

            n_loops = 0
            while condition == False:
                n = 0
                if file_short not in data_interval_detection:
                    data_interval_detection[file_short] = {"BG": {"Start": None, "End": None, "Index": None},
                                                           "MAT": {"Start": None, "End": None, "Index": None}}
                    data_calc[file_short] = {"Time": [], "Cumulative Average": [], "Change": [], "Ratio": []}

                dataset_isotope = list(df_data[isotope_is])
                condition_2 = False
                index_2 = 10
                while condition_2 == False:
                    if self.container_var["General Settings"]["Desired Average"].get() == 1:
                        mean_first10 = np.mean(dataset_isotope[:index_2])
                    else:
                        mean_first10 = np.median(dataset_isotope[:index_2])

                    if mean_first10 > 0:
                        condition_2 = True
                    else:
                        index_2 += 1

                for index, value_time in enumerate(dataset_time):
                    data_calc[file_short]["Time"].append(value_time)

                    if index > 1:
                        if self.container_var["General Settings"]["Desired Average"].get() == 1:
                            value_cumulavg = np.mean(dataset_isotope[:index + 1])
                        else:
                            value_cumulavg = np.median(dataset_isotope[:index + 1])
                        data_calc[file_short]["Cumulative Average"].append(value_cumulavg)
                        value_change = value_cumulavg - data_calc[file_short]["Cumulative Average"][-2]
                        data_calc[file_short]["Change"].append(value_change)
                        value_ratio = value_cumulavg/mean_first10
                        data_calc[file_short]["Ratio"].append(value_ratio)
                    elif index == 1:
                        if self.container_var["General Settings"]["Desired Average"].get() == 1:
                            value_cumulavg = np.mean(dataset_isotope[:index])
                        else:
                            value_cumulavg = np.median(dataset_isotope[:index])
                        data_calc[file_short]["Cumulative Average"].append(value_cumulavg)
                        value_change = value_cumulavg - data_calc[file_short]["Cumulative Average"][0]
                        data_calc[file_short]["Change"].append(value_change)
                        value_ratio = value_cumulavg/mean_first10
                        data_calc[file_short]["Ratio"].append(value_ratio)
                    elif index == 0:
                        if self.container_var["General Settings"]["Desired Average"].get() == 1:
                            value_cumulavg = np.mean(dataset_isotope[0])
                        else:
                            value_cumulavg = np.median(dataset_isotope[0])
                        data_calc[file_short]["Cumulative Average"].append(value_cumulavg)
                        value_change = 0.0
                        data_calc[file_short]["Change"].append(value_change)
                        value_ratio = value_cumulavg/mean_first10
                        data_calc[file_short]["Ratio"].append(value_ratio)

                    if value_ratio > 20 and data_interval_detection[file_short]["BG"]["End"] == None:
                        offset_bg_start = self.container_var["General Settings"]["BG Offset Start"].get()
                        offset_bg_end = self.container_var["General Settings"]["BG Offset End"].get()
                        offset_mat_start = self.container_var["General Settings"]["MAT Offset Start"].get()

                        data_interval_detection[file_short]["BG"]["Start"] = dataset_time[offset_bg_start]

                        index_time = dataset_time.index(value_time)
                        data_interval_detection[file_short]["BG"]["End"] = dataset_time[index_time - offset_bg_end]
                        data_interval_detection[file_short]["MAT"]["Start"] = dataset_time[
                            index_time + offset_mat_start]

                        index_start = dataset_time.index(dataset_time[offset_bg_start])
                        index_end = dataset_time.index(dataset_time[index_time - offset_bg_end])
                        data_interval_detection[file_short]["BG"]["Index"] = [index_start, index_end]

                        index_start_mat = dataset_time.index(dataset_time[index_time + offset_mat_start])

                    if value_change < 0 and index > 0.5*len(dataset_time) \
                            and data_interval_detection[file_short]["MAT"]["End"] == None:
                        offset_mat_end = self.container_var["General Settings"]["MAT Offset End"].get()

                        index_time = dataset_time.index(value_time)
                        data_interval_detection[file_short]["MAT"]["End"] = dataset_time[index_time - offset_mat_end]

                        index_end = dataset_time.index(dataset_time[index_time - offset_mat_end])
                        data_interval_detection[file_short]["MAT"]["Index"] = [index_start_mat, index_end]

                for key1, value1 in data_interval_detection[file_short].items():
                    if key1 == mode:
                        for key2, value2 in value1.items():
                            if value2 != None:
                                n += 1

                if n == 3:
                    condition = True
                else:
                    index_isotope += 1

                n_loops += 1

                if n_loops == 100:
                    break

        if mode == "BG":
            if self.pysills_mode == "MA":
                self.container_var[key_setting]["Time BG Start"].set("auto-detection used")
                self.container_var[key_setting]["Time BG End"].set("auto-detection used")
            else:
                if self.pysills_mode in ["FI", "INCL"]:
                    self.container_var[key_setting]["Time BG Start"].set("auto-detection used")
                    self.container_var[key_setting]["Time BG End"].set("auto-detection used")
                else:
                    if n_loops == 100:
                        self.container_var[key_setting]["Time BG Start"].set("auto-detection not successful")
                        self.container_var[key_setting]["Time BG End"].set("auto-detection not successful")
                    else:
                        self.container_var[key_setting]["Time BG Start"].set("auto-detection used")
                        self.container_var[key_setting]["Time BG End"].set("auto-detection used")

            self.autodetection_bg = True
        elif mode == "MAT":
            if self.pysills_mode == "MA":
                self.container_var[key_setting]["Time MAT Start"].set("auto-detection used")
                self.container_var[key_setting]["Time MAT End"].set("auto-detection used")
            elif self.pysills_mode in ["FI", "INCL"]:
                self.container_var[key_setting]["Time MAT Start"].set("auto-detection used")
                self.container_var[key_setting]["Time MAT End"].set("auto-detection used")
            elif self.pysills_mode == "MI":
                self.container_var[key_setting]["Time MAT Start"].set("auto-detection used")
                self.container_var[key_setting]["Time MAT End"].set("auto-detection used")

            self.autodetection_sig = True

        if n_loops < 1000:
            for filename, dataset in data_interval_detection.items():
                if filename in self.container_lists["STD"]["Short"]:
                    filetype = "STD"

                    if self.pysills_mode == "MA":
                        self.temp_lines_checkup2[filetype][filename] = 0
                elif filename in self.container_lists["SMPL"]["Short"]:
                    filetype = "SMPL"

                    if self.pysills_mode == "MA":
                        self.temp_lines_checkup2[filetype][filename] = 0

                var_index = self.container_lists[filetype]["Short"].index(filename)
                filename_long = self.container_lists[filetype]["Long"][var_index]

                t_start = dataset[mode]["Start"]
                t_end = dataset[mode]["End"]
                index_start = dataset[mode]["Index"][0]
                index_end = dataset[mode]["Index"][1]

                self.container_helper[filetype][filename][mode]["Content"][1] = {
                    "Times": [t_start, t_end], "Indices": [index_start, index_end], "Object": None}
                self.container_helper[filetype][filename][mode]["ID"] += 1
                self.container_helper[filetype][filename][mode]["Indices"].append(1)

                self.container_var[filetype][filename_long]["Frame"].config(background=self.sign_yellow, bd=1)
                self.container_var[filetype][filename_long]["Sign Color"].set(self.sign_yellow)

                if self.pysills_mode == "MA":
                    self.temp_lines_checkup2[filetype][filename] = 0
                    self.show_time_signal_diagram_checker(var_setting_key=key_setting)
                elif self.pysills_mode in ["FI", "INCL"]:
                    self.temp_lines_checkup2[filetype][filename] = 0
                    self.show_time_signal_diagram_checker(var_setting_key=key_setting)
                elif self.pysills_mode == "MI":
                    self.temp_lines_checkup2[filetype][filename] = 0
                    self.show_time_signal_diagram_checker(var_setting_key=key_setting)

    def checkup_oxides(self):
        """Check-up window to control the 100 wt.% oxides setup."""
        # Colors
        font_color_dark = self.bg_colors["Dark Font"]
        font_color_light = self.bg_colors["Light Font"]
        background_color_dark = self.bg_colors["BG Window"]
        background_color_elements = self.bg_colors["Light"]
        background_color_light = self.bg_colors["Very Light"]
        accent_color = self.bg_colors["Accent"]  # self.accent_color
        font_header = self.font_settings["Header"]
        font_elements = self.font_settings["Elements"]
        font_option = self.font_settings["Options"]
        font_table = self.font_settings["Table"]

        ## Window Settings
        row_min = self.row_height
        column_min = self.column_width
        n_rows = self.window_dimensions["Check-up oxides"][0]
        n_columns = self.window_dimensions["Check-up oxides"][1]

        window_width = int(n_columns*self.column_width)
        window_height = int(n_rows*self.row_height)

        var_geometry = str(window_width) + "x" + str(window_height) + "+" + str(0) + "+" + str(0)

        subwindow_checkup_oxides = tk.Toplevel(self.parent)
        subwindow_checkup_oxides.title("Check-Up - 100 wt.% oxides")
        subwindow_checkup_oxides.geometry(var_geometry)
        subwindow_checkup_oxides.resizable(False, False)
        subwindow_checkup_oxides["bg"] = background_color_dark

        for x in range(n_columns):
            tk.Grid.columnconfigure(subwindow_checkup_oxides, x, weight=1)
        for y in range(n_rows):
            tk.Grid.rowconfigure(subwindow_checkup_oxides, y, weight=1)

        # Rows
        for i in range(0, n_rows):
            subwindow_checkup_oxides.grid_rowconfigure(i, minsize=row_min)
        # Columns
        for i in range(0, n_columns):
            subwindow_checkup_oxides.grid_columnconfigure(i, minsize=column_min)

        var_row_start = 0
        var_column_start = 0
        var_header_n = 10
        int_category_n = 8

        ## FRAMES
        frm_01 = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start + 1, column_id=var_column_start, n_rows=12,
            n_columns=var_header_n, fg=font_color_dark, bg=background_color_light).create_frame()
        frm_02 = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start + 1, column_id=var_header_n, n_rows=12,
            n_columns=var_header_n, fg=font_color_dark, bg=background_color_light).create_frame()
        frm_03 = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start + 1, column_id=2*var_header_n, n_rows=12,
            n_columns=var_header_n, fg=font_color_dark, bg=background_color_light).create_frame()
        frm_04 = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start + 1, column_id=3*var_header_n, n_rows=12,
            n_columns=var_header_n, fg=font_color_dark, bg=background_color_light).create_frame()
        frm_05 = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start + 1, column_id=4*var_header_n, n_rows=12,
            n_columns=var_header_n, fg=font_color_dark, bg=background_color_light).create_frame()
        frm_06 = SE(
            parent=subwindow_checkup_oxides, row_id=15, column_id=var_column_start, n_rows=12,
            n_columns=var_header_n, fg=font_color_dark, bg=background_color_light).create_frame()
        frm_07 = SE(
            parent=subwindow_checkup_oxides, row_id=15, column_id=var_header_n, n_rows=12,
            n_columns=var_header_n, fg=font_color_dark, bg=background_color_light).create_frame()
        frm_08 = SE(
            parent=subwindow_checkup_oxides, row_id=15, column_id=2*var_header_n, n_rows=12,
            n_columns=var_header_n, fg=font_color_dark, bg=background_color_light).create_frame()
        frm_09 = SE(
            parent=subwindow_checkup_oxides, row_id=15, column_id=3*var_header_n, n_rows=12,
            n_columns=var_header_n, fg=font_color_dark, bg=background_color_light).create_frame()
        frm_10 = SE(
            parent=subwindow_checkup_oxides, row_id=15, column_id=4*var_header_n, n_rows=12,
            n_columns=var_header_n, fg=font_color_dark, bg=background_color_light).create_frame()
        frm_11 = SE(
            parent=subwindow_checkup_oxides, row_id=13, column_id=var_column_start, n_rows=1,
            n_columns=5*var_header_n, fg=font_color_dark, bg=background_color_elements).create_frame(
            relief=tk.FLAT)
        frm_12 = SE(
            parent=subwindow_checkup_oxides, row_id=27, column_id=var_column_start, n_rows=1,
            n_columns=5*var_header_n, fg=font_color_dark, bg=background_color_elements).create_frame(
            relief=tk.FLAT)
        frm_13 = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start + 15, column_id=5*var_header_n + 1, n_rows=3,
            n_columns=int(1.5*var_header_n), fg=font_color_dark, bg=background_color_elements).create_frame(
            relief=tk.FLAT)
        frm_14 = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start + 2, column_id=5*var_header_n + 1, n_rows=2,
            n_columns=int(1.5*var_header_n), fg=font_color_dark, bg=background_color_elements).create_frame(
            relief=tk.FLAT)
        frm_15 = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start + 5, column_id=5*var_header_n + 1, n_rows=2,
            n_columns=int(1.5*var_header_n), fg=font_color_dark, bg=background_color_elements).create_frame(
            relief=tk.FLAT)
        frm_16 = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start + 9, column_id=5*var_header_n + 1, n_rows=2,
            n_columns=int(1.5*var_header_n), fg=font_color_dark, bg=background_color_elements).create_frame(
            relief=tk.FLAT)
        frm_17 = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start + 2,
            column_id=5*var_header_n + int(1.5*var_header_n) + 2, n_rows=11, n_columns=int(1.5*var_header_n),
            fg=font_color_dark, bg=background_color_elements).create_frame(relief=tk.FLAT)
        frm_18 = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start + 15,
            column_id=5*var_header_n + int(1.5*var_header_n) + 2, n_rows=11, n_columns=int(1.5*var_header_n),
            fg=font_color_dark, bg=background_color_elements).create_frame(relief=tk.FLAT)

        ## LABELS
        str_lbl_01 = self.language_dict["Select isotope"][self.var_language]
        str_lbl_02 = self.language_dict["Sample"][self.var_language]
        str_lbl_03 = self.language_dict["Matrix"][self.var_language]
        str_lbl_04 = self.language_dict["Inclusion"][self.var_language]
        str_lbl_05 = self.language_dict["Select all"][self.var_language]
        str_lbl_06 = self.language_dict["Run"][self.var_language]
        str_lbl_07 = self.language_dict["Select"][self.var_language]
        str_lbl_08 = self.language_dict["Setup - Oxide ratios"][self.var_language]
        str_lbl_09 = self.language_dict["Setup - Reference element"][self.var_language]
        str_lbl_10 = self.language_dict["Setup - Oxide selection"][self.var_language]
        str_lbl_11 = self.language_dict["Run 100 wt.% oxides calculation for"][self.var_language]
        str_lbl_12 = self.language_dict["Mineral quantification"][self.var_language]
        str_lbl_13 = self.language_dict["Inclusion quantification"][self.var_language]
        str_lbl_14 = self.language_dict["Select all oxides"][self.var_language]
        str_lbl_15 = self.language_dict["Deselect all oxides"][self.var_language]
        str_lbl_16 = self.language_dict["Rock-forming elements"][self.var_language]
        str_lbl_17 = self.language_dict["Guess the composition"][self.var_language]
        str_lbl_18 = self.language_dict["Setup - Files"][self.var_language]
        str_lbl_19 = self.language_dict["Alkali metals"][self.var_language]
        str_lbl_20 = self.language_dict["Alkaline earth metals"][self.var_language]
        str_lbl_21 = self.language_dict["Transition metals"][self.var_language]
        str_lbl_22 = self.language_dict["Other metals"][self.var_language]
        str_lbl_23 = self.language_dict["Metalloids"][self.var_language]
        str_lbl_24 = self.language_dict["Lanthanides"][self.var_language]
        str_lbl_25 = self.language_dict["Actinides"][self.var_language]
        str_lbl_26 = self.language_dict["Non-metals"][self.var_language]
        str_lbl_27 = self.language_dict["Halogens"][self.var_language]
        str_lbl_28 = self.language_dict["Noble gases"][self.var_language]

        lbl_01 = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start, column_id=var_column_start, n_rows=1,
            n_columns=var_header_n, fg=font_color_light,
            bg=background_color_dark).create_simple_label(
            text=str_lbl_19, relief=tk.FLAT, fontsize="sans 12 bold")
        lbl_02 = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start, column_id=var_header_n, n_rows=1,
            n_columns=var_header_n, fg=font_color_light,
            bg=background_color_dark).create_simple_label(
            text=str_lbl_20, relief=tk.FLAT, fontsize="sans 12 bold")
        lbl_03 = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start, column_id=2*var_header_n, n_rows=1,
            n_columns=var_header_n, fg=font_color_light,
            bg=background_color_dark).create_simple_label(
            text=str_lbl_21, relief=tk.FLAT, fontsize="sans 12 bold")
        lbl_04 = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start, column_id=3*var_header_n, n_rows=1,
            n_columns=var_header_n, fg=font_color_light,
            bg=background_color_dark).create_simple_label(
            text=str_lbl_22, relief=tk.FLAT, fontsize="sans 12 bold")
        lbl_05 = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start, column_id=4*var_header_n, n_rows=1,
            n_columns=var_header_n, fg=font_color_light,
            bg=background_color_dark).create_simple_label(
            text=str_lbl_23, relief=tk.FLAT, fontsize="sans 12 bold")
        lbl_06 = SE(
            parent=subwindow_checkup_oxides, row_id=14, column_id=var_column_start, n_rows=1,
            n_columns=var_header_n, fg=font_color_light,
            bg=background_color_dark).create_simple_label(
            text=str_lbl_24, relief=tk.FLAT, fontsize="sans 12 bold")
        lbl_07 = SE(
            parent=subwindow_checkup_oxides, row_id=14, column_id=var_header_n, n_rows=1,
            n_columns=var_header_n, fg=font_color_light,
            bg=background_color_dark).create_simple_label(
            text=str_lbl_25, relief=tk.FLAT, fontsize="sans 12 bold")
        lbl_08 = SE(
            parent=subwindow_checkup_oxides, row_id=14, column_id=2*var_header_n, n_rows=1,
            n_columns=var_header_n, fg=font_color_light,
            bg=background_color_dark).create_simple_label(
            text=str_lbl_26, relief=tk.FLAT, fontsize="sans 12 bold")
        lbl_09 = SE(
            parent=subwindow_checkup_oxides, row_id=14, column_id=3*var_header_n, n_rows=1,
            n_columns=var_header_n, fg=font_color_light,
            bg=background_color_dark).create_simple_label(
            text=str_lbl_27, relief=tk.FLAT, fontsize="sans 12 bold")
        lbl_10 = SE(
            parent=subwindow_checkup_oxides, row_id=14, column_id=4*var_header_n, n_rows=1,
            n_columns=var_header_n, fg=font_color_light,
            bg=background_color_dark).create_simple_label(
            text=str_lbl_28, relief=tk.FLAT, fontsize="sans 12 bold")
        lbl_11 = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start, column_id=5*var_header_n + 1, n_rows=1,
            n_columns=int(1.5*var_header_n), fg=font_color_light,
            bg=background_color_dark).create_simple_label(
            text=str_lbl_08, relief=tk.FLAT, fontsize="sans 12 bold", anchor=tk.W)
        lbl_11a = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start + 1, column_id=5*var_header_n + 1, n_rows=1,
            n_columns=int_category_n, fg=font_color_light,
            bg=background_color_dark).create_simple_label(
            text=str_lbl_02 + "/"+ str_lbl_03, relief=tk.FLAT, fontsize="sans 10 bold", anchor=tk.W)
        lbl_11b = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start + 2, column_id=5*var_header_n + 1, n_rows=1,
            n_columns=int_category_n, fg=font_color_dark,
            bg=background_color_elements).create_simple_label(
            text="FeO/(FeO + Fe2O3)", relief=tk.FLAT, fontsize="sans 10 bold", anchor=tk.W)
        lbl_11c = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start + 3, column_id=5*var_header_n + 1, n_rows=1,
            n_columns=int_category_n, fg=font_color_dark,
            bg=background_color_elements).create_simple_label(
            text="MnO/(MnO + Mn2O3)", relief=tk.FLAT, fontsize="sans 10 bold", anchor=tk.W)
        lbl_11d = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start + 4, column_id=5*var_header_n + 1, n_rows=1,
            n_columns=int_category_n, fg=font_color_light,
            bg=background_color_dark).create_simple_label(
            text=str_lbl_04, relief=tk.FLAT, fontsize="sans 10 bold", anchor=tk.W)
        lbl_11e = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start + 5, column_id=5*var_header_n + 1, n_rows=1,
            n_columns=int_category_n, fg=font_color_dark,
            bg=background_color_elements).create_simple_label(
            text="FeO/(FeO + Fe2O3)", relief=tk.FLAT, fontsize="sans 10 bold", anchor=tk.W)
        lbl_11f = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start + 6, column_id=5*var_header_n + 1, n_rows=1,
            n_columns=int_category_n, fg=font_color_dark,
            bg=background_color_elements).create_simple_label(
            text="MnO/(MnO + Mn2O3)", relief=tk.FLAT, fontsize="sans 10 bold", anchor=tk.W)
        lbl_12 = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start + 8, column_id=5*var_header_n + 1, n_rows=1,
            n_columns=int(1.5*var_header_n), fg=font_color_light,
            bg=background_color_dark).create_simple_label(
            text=str_lbl_09, relief=tk.FLAT, fontsize="sans 12 bold", anchor=tk.W)
        lbl_12a = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start + 9, column_id=5*var_header_n + 1, n_rows=1,
            n_columns=int_category_n, fg=font_color_dark,
            bg=background_color_elements).create_simple_label(
            text=str_lbl_12, relief=tk.FLAT, fontsize="sans 10 bold", anchor=tk.W)
        lbl_12b = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start + 10, column_id=5*var_header_n + 1, n_rows=1,
            n_columns=int_category_n, fg=font_color_dark,
            bg=background_color_elements).create_simple_label(
            text=str_lbl_13, relief=tk.FLAT, fontsize="sans 10 bold", anchor=tk.W)
        lbl_13 = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start + 12, column_id=5*var_header_n + 1, n_rows=1,
            n_columns=int(1.5*var_header_n), fg=font_color_light,
            bg=background_color_dark).create_simple_label(
            text=str_lbl_10, relief=tk.FLAT, fontsize="sans 12 bold", anchor=tk.W)
        lbl_13a = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start + 13, column_id=5*var_header_n + 1, n_rows=1,
            n_columns=int_category_n, fg=font_color_dark,
            bg=background_color_elements).create_simple_label(
            text=str_lbl_14, relief=tk.FLAT, fontsize="sans 10 bold", anchor=tk.W)
        lbl_13b = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start + 14, column_id=5*var_header_n + 1, n_rows=1,
            n_columns=int_category_n, fg=font_color_dark,
            bg=background_color_elements).create_simple_label(
            text=str_lbl_15, relief=tk.FLAT, fontsize="sans 10 bold", anchor=tk.W)
        lbl_13c = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start + 15, column_id=5*var_header_n + 1, n_rows=1,
            n_columns=int_category_n, fg=font_color_dark,
            bg=background_color_elements).create_simple_label(
            text=str_lbl_16, relief=tk.FLAT, fontsize="sans 10 bold", anchor=tk.W)
        lbl_13d = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start + 16, column_id=5*var_header_n + 1, n_rows=1,
            n_columns=int_category_n, fg=font_color_dark,
            bg=background_color_elements).create_simple_label(
            text=str_lbl_17, relief=tk.FLAT, fontsize="sans 10 bold", anchor=tk.W)
        lbl_14 = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start, column_id=5*var_header_n + int(1.5*var_header_n) + 2,
            n_rows=1, n_columns=int(1.5*var_header_n), fg=font_color_light,
            bg=background_color_dark).create_simple_label(
            text=str_lbl_18, relief=tk.FLAT, fontsize="sans 12 bold", anchor=tk.W)
        lbl_14a = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start + 1,
            column_id=5*var_header_n + int(1.5*var_header_n) + 2, n_rows=1, n_columns=int_category_n,
            fg=font_color_light, bg=background_color_dark).create_simple_label(
            text=str_lbl_02 + "/"+ str_lbl_03, relief=tk.FLAT, fontsize="sans 10 bold", anchor=tk.W)
        lbl_14b = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start + 14,
            column_id=5*var_header_n + int(1.5*var_header_n) + 2, n_rows=1, n_columns=int_category_n,
            fg=font_color_light, bg=background_color_dark).create_simple_label(
            text=str_lbl_04, relief=tk.FLAT, fontsize="sans 10 bold", anchor=tk.W)
        lbl_15 = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start + 17, column_id=5*var_header_n + 1, n_rows=1,
            n_columns=int(1.5*var_header_n), fg=font_color_light,
            bg=background_color_dark).create_simple_label(
            text=str_lbl_11, relief=tk.FLAT, fontsize="sans 12 bold", anchor=tk.W)

        ## BUTTONS
        btn_01 = SE(
            parent=subwindow_checkup_oxides, row_id=13, column_id=int(0.5*var_header_n), n_rows=1,
            n_columns=int(0.5*var_header_n), fg=font_color_dark,
            bg=background_color_elements).create_simple_button(
            text=str_lbl_05, bg_active=accent_color, fg_active=font_color_light)
        btn_02 = SE(
            parent=subwindow_checkup_oxides, row_id=13, column_id=int(1.5*var_header_n), n_rows=1,
            n_columns=int(0.5*var_header_n), fg=font_color_dark,
            bg=background_color_elements).create_simple_button(
            text=str_lbl_05, bg_active=accent_color, fg_active=font_color_light)
        btn_03 = SE(
            parent=subwindow_checkup_oxides, row_id=13, column_id=int(2.5*var_header_n), n_rows=1,
            n_columns=int(0.5*var_header_n), fg=font_color_dark,
            bg=background_color_elements).create_simple_button(
            text=str_lbl_05, bg_active=accent_color, fg_active=font_color_light)
        btn_04 = SE(
            parent=subwindow_checkup_oxides, row_id=13, column_id=int(3.5*var_header_n), n_rows=1,
            n_columns=int(0.5*var_header_n), fg=font_color_dark,
            bg=background_color_elements).create_simple_button(
            text=str_lbl_05, bg_active=accent_color, fg_active=font_color_light)
        btn_05 = SE(
            parent=subwindow_checkup_oxides, row_id=13, column_id=int(4.5*var_header_n), n_rows=1,
            n_columns=int(0.5*var_header_n), fg=font_color_dark,
            bg=background_color_elements).create_simple_button(
            text=str_lbl_05, bg_active=accent_color, fg_active=font_color_light)
        btn_06 = SE(
            parent=subwindow_checkup_oxides, row_id=27, column_id=int(0.5*var_header_n), n_rows=1,
            n_columns=int(0.5*var_header_n), fg=font_color_dark,
            bg=background_color_elements).create_simple_button(
            text=str_lbl_05, bg_active=accent_color, fg_active=font_color_light)
        btn_07 = SE(
            parent=subwindow_checkup_oxides, row_id=27, column_id=int(1.5*var_header_n), n_rows=1,
            n_columns=int(0.5*var_header_n), fg=font_color_dark,
            bg=background_color_elements).create_simple_button(
            text=str_lbl_05, bg_active=accent_color, fg_active=font_color_light)
        btn_08 = SE(
            parent=subwindow_checkup_oxides, row_id=27, column_id=int(2.5*var_header_n), n_rows=1,
            n_columns=int(0.5*var_header_n), fg=font_color_dark,
            bg=background_color_elements).create_simple_button(
            text=str_lbl_05, bg_active=accent_color, fg_active=font_color_light)
        btn_09 = SE(
            parent=subwindow_checkup_oxides, row_id=27, column_id=int(3.5*var_header_n), n_rows=1,
            n_columns=int(0.5*var_header_n), fg=font_color_dark,
            bg=background_color_elements).create_simple_button(
            text=str_lbl_05, bg_active=accent_color, fg_active=font_color_light)
        btn_10 = SE(
            parent=subwindow_checkup_oxides, row_id=27, column_id=int(4.5*var_header_n), n_rows=1,
            n_columns=int(0.5*var_header_n), fg=font_color_dark,
            bg=background_color_elements).create_simple_button(
            text=str_lbl_05, bg_active=accent_color, fg_active=font_color_light)
        btn_13a = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start + 13, column_id=5*var_header_n + 1 + int_category_n,
            n_rows=1, n_columns=int(0.75*var_header_n), fg=font_color_dark,
            bg=background_color_elements).create_simple_button(
            text=str_lbl_06, bg_active=accent_color, fg_active=font_color_light,
            command=self.select_all_oxides)
        btn_13b = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start + 14, column_id=5*var_header_n + 1 + int_category_n,
            n_rows=1, n_columns=int(0.75*var_header_n), fg=font_color_dark,
            bg=background_color_elements).create_simple_button(
            text=str_lbl_06, bg_active=accent_color, fg_active=font_color_light,
            command=self.deselect_all_oxides)
        btn_13c = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start + 15, column_id=5*var_header_n + 1 + int_category_n,
            n_rows=1, n_columns=int(0.75*var_header_n), fg=font_color_dark,
            bg=background_color_elements).create_simple_button(
            text=str_lbl_07, bg_active=accent_color, fg_active=font_color_light,
            command=self.select_main_rockforming_elements)
        btn_13d = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start + 16, column_id=5*var_header_n + 1 + int_category_n,
            n_rows=1, n_columns=int(0.75*var_header_n), fg=font_color_dark,
            bg=background_color_elements).create_simple_button(
            text=str_lbl_06, bg_active=accent_color, fg_active=font_color_light)
        btn_13d.configure(state="disabled")

        ## ENTRIES
        var_entr_mat_fe = self.container_var["Oxides Quantification"]["Ratios"]["Fe-Ratio"]
        var_entr_mat_mn = self.container_var["Oxides Quantification"]["Ratios"]["Mn-Ratio"]
        var_entr_incl_fe = self.container_var["Oxides Quantification INCL"]["Ratios"]["Fe-Ratio"]
        var_entr_incl_mn = self.container_var["Oxides Quantification INCL"]["Ratios"]["Mn-Ratio"]

        entr_mat_fe = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start + 2, column_id=5*var_header_n + 1 + int_category_n,
            n_rows=1, n_columns=int(0.75*var_header_n), fg=font_color_dark,
            bg=self.bg_colors["White"]).create_simple_entry(
            var=var_entr_mat_fe, text_default=var_entr_mat_fe.get())
        entr_mat_mn = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start + 3, column_id=5*var_header_n + 1 + int_category_n,
            n_rows=1, n_columns=int(0.75*var_header_n), fg=font_color_dark,
            bg=self.bg_colors["White"]).create_simple_entry(
            var=var_entr_mat_mn, text_default=var_entr_mat_mn.get())

        entr_incl_fe = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start + 5, column_id=5*var_header_n + 1 + int_category_n,
            n_rows=1, n_columns=int(0.75*var_header_n), fg=font_color_dark,
            bg=self.bg_colors["White"]).create_simple_entry(
            var=var_entr_incl_fe, text_default=var_entr_incl_fe.get())
        entr_incl_mn = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start + 6, column_id=5*var_header_n + 1 + int_category_n,
            n_rows=1, n_columns=int(0.75*var_header_n), fg=font_color_dark,
            bg=self.bg_colors["White"]).create_simple_entry(
            var=var_entr_incl_mn, text_default=var_entr_incl_mn.get())

        if self.pysills_mode in ["MA", "FI"]:
            entr_incl_fe.configure(state="disabled")
            entr_incl_mn.configure(state="disabled")

        ## TREEVIEWS
        list_mat_isotopes = self.container_lists["Measured Isotopes"]["All"].copy()
        list_incl_isotopes = self.container_lists["Measured Isotopes"]["All"].copy()
        # File setup (matrix/sample)
        vsb_17 = ttk.Scrollbar(master=frm_17, orient="vertical")
        text_17 = tk.Text(
            master=frm_17, width=30, height=25, yscrollcommand=vsb_17.set, bg=background_color_light)
        vsb_17.config(command=text_17.yview)
        vsb_17.pack(side="right", fill="y")
        text_17.pack(side="left", fill="both", expand=True)

        for index, filename_short in enumerate(self.container_lists["SMPL"]["Short"]):
            filename_long = self.container_lists["SMPL"]["Long"][index]
            lbl_i = tk.Label(frm_17, text=filename_short, bg=background_color_light,
                             fg=font_color_dark)
            text_17.window_create("end", window=lbl_i)
            text_17.insert("end", "\t")

            if filename_short not in self.container_var["Oxides Quantification"]["Total Amounts"]:
                self.container_var["Oxides Quantification"]["Total Amounts"][filename_short] = tk.StringVar()
                self.container_var["Oxides Quantification"]["Total Amounts"][filename_short].set("100.0")

            entr_i = tk.Entry(
                frm_17, textvariable=self.container_var["Oxides Quantification"]["Total Amounts"][filename_short],
                fg=font_color_dark, bg=self.bg_colors["White"], highlightthickness=0,
                highlightbackground=background_color_light, width=6)
            text_17.window_create("insert", window=entr_i)
            text_17.insert("end", "\t")

            if self.pysills_mode == "MA":
                var_opt_ref_i = self.container_var["SMPL"][filename_long]["IS Data"]["IS"]
            else:
                var_opt_ref_i = self.container_var["SMPL"][filename_long]["Matrix Setup"]["IS"]["Name"]

            opt_ref_i = tk.OptionMenu(frm_17, var_opt_ref_i, *list_mat_isotopes)
            opt_ref_i["menu"].config(
                fg=font_color_dark, bg=background_color_elements,
                activeforeground=font_color_light, activebackground=accent_color)
            opt_ref_i.config(
                bg=background_color_elements, fg=font_color_dark,
                activeforeground=font_color_light, activebackground=accent_color, highlightthickness=0)
            text_17.window_create("end", window=opt_ref_i)
            text_17.insert("end", " \n")

        # File setup (inclusion)
        vsb_18 = ttk.Scrollbar(master=frm_18, orient="vertical")
        text_18 = tk.Text(
            master=frm_18, width=30, height=25, yscrollcommand=vsb_18.set, bg=background_color_light)
        vsb_18.config(command=text_18.yview)
        vsb_18.pack(side="right", fill="y")
        text_18.pack(side="left", fill="both", expand=True)

        if self.pysills_mode in ["MI", "INCL"]:
            for index, filename_short in enumerate(self.container_lists["SMPL"]["Short"]):
                filename_long = self.container_lists["SMPL"]["Long"][index]
                lbl_i = tk.Label(frm_18, text=filename_short, bg=background_color_light,
                                 fg=font_color_dark)
                text_18.window_create("end", window=lbl_i)
                text_18.insert("end", "\t")

                if filename_short not in self.container_var["Oxides Quantification INCL"]["Total Amounts"]:
                    self.container_var["Oxides Quantification INCL"]["Total Amounts"][filename_short] = tk.StringVar()
                    self.container_var["Oxides Quantification INCL"]["Total Amounts"][filename_short].set("100.0")

                entr_i = tk.Entry(
                    frm_18, textvariable=self.container_var["Oxides Quantification INCL"]["Total Amounts"][
                        filename_short], fg=font_color_dark, bg=self.bg_colors["White"],
                    highlightthickness=0, highlightbackground=background_color_light, width=6)

                text_18.window_create("insert", window=entr_i)
                text_18.insert("end", "\t")

                var_opt_ref_i = self.container_var["SMPL"][filename_long]["IS Data"]["IS"]

                opt_ref_i = tk.OptionMenu(frm_18, var_opt_ref_i, *list_mat_isotopes)
                opt_ref_i["menu"].config(
                    fg=font_color_dark, bg=background_color_elements,
                    activeforeground=font_color_light, activebackground=accent_color)
                opt_ref_i.config(
                    bg=background_color_elements, fg=font_color_dark,
                    activeforeground=font_color_light, activebackground=accent_color,
                    highlightthickness=0)
                text_18.window_create("end", window=opt_ref_i)
                text_18.insert("end", " \n")

        list_major_oxides = [
            "SiO2", "Al2O3", "FeO", "Fe2O3", "CaO", "Na2O", "MgO", "K2O", "TiO2", "P2O5", "MnO", "Mn2O3", "SO3"]
        # Alkali metals
        vsb_01 = ttk.Scrollbar(master=frm_01, orient="vertical")
        text_01 = tk.Text(
            master=frm_01, width=30, height=25, yscrollcommand=vsb_01.set, bg=background_color_light)
        vsb_01.config(command=text_01.yview)
        vsb_01.pack(side="right", fill="y")
        text_01.pack(side="left", fill="both", expand=True)
        counter_01 = 0
        for index, element in enumerate(self.container_lists["Measured Elements"]["All"]):
            if element in ["Li", "Na", "K", "Rb", "Cs", "Fr"]:
                for oxide in self.chemistry_oxides_sorted[element]:
                    if oxide in list_major_oxides:
                        var_cb = self.container_var["Oxides Quantification"]["Major"][oxide]
                    else:
                        var_cb = self.container_var["Oxides Quantification"]["Minor"][oxide]

                    cb_i = tk.Checkbutton(
                        master=frm_01, text=oxide, fg=font_color_dark,
                        bg=background_color_light, variable=var_cb,
                            command=lambda var_checkbox=var_cb, key_oxide=oxide:
                            self.select_oxide2(var_checkbox, key_oxide))
                    text_01.window_create("end", window=cb_i)
                    text_01.insert("end", "\n")
                    counter_01 += 1
                    if element not in self.container_lists["Possible Oxides"]:
                        self.container_lists["Possible Oxides"][element] = []

                    if oxide not in self.container_lists["Possible Oxides"]["All"]:
                        self.container_lists["Possible Oxides"]["All"].append(oxide)
                    if oxide not in self.container_lists["Possible Oxides"][element]:
                        self.container_lists["Possible Oxides"][element].append(oxide)

        # Alkaline earth metals
        vsb_02 = ttk.Scrollbar(master=frm_02, orient="vertical")
        text_02 = tk.Text(
            master=frm_02, width=30, height=25, yscrollcommand=vsb_02.set, bg=background_color_light)
        vsb_02.config(command=text_02.yview)
        vsb_02.pack(side="right", fill="y")
        text_02.pack(side="left", fill="both", expand=True)
        counter_02 = 0
        for index, element in enumerate(self.container_lists["Measured Elements"]["All"]):
            if element in ["Be", "Mg", "Ca", "Sr", "Ba", "Ra"]:
                for oxide in self.chemistry_oxides_sorted[element]:
                    if oxide in list_major_oxides:
                        var_cb = self.container_var["Oxides Quantification"]["Major"][oxide]
                    else:
                        var_cb = self.container_var["Oxides Quantification"]["Minor"][oxide]

                    cb_i = tk.Checkbutton(
                        master=frm_02, text=oxide, fg=font_color_dark,
                        bg=background_color_light, variable=var_cb,
                            command=lambda var_checkbox=var_cb, key_oxide=oxide:
                            self.select_oxide2(var_checkbox, key_oxide))
                    text_02.window_create("end", window=cb_i)
                    text_02.insert("end", "\n")
                    counter_02 += 1
                    if element not in self.container_lists["Possible Oxides"]:
                        self.container_lists["Possible Oxides"][element] = []

                    if oxide not in self.container_lists["Possible Oxides"]["All"]:
                        self.container_lists["Possible Oxides"]["All"].append(oxide)
                    if oxide not in self.container_lists["Possible Oxides"][element]:
                        self.container_lists["Possible Oxides"][element].append(oxide)

        # Transition metals
        vsb_03 = ttk.Scrollbar(master=frm_03, orient="vertical")
        text_03 = tk.Text(
            master=frm_03, width=30, height=25, yscrollcommand=vsb_03.set, bg=background_color_light)
        vsb_03.config(command=text_03.yview)
        vsb_03.pack(side="right", fill="y")
        text_03.pack(side="left", fill="both", expand=True)
        counter_03 = 0
        for index, element in enumerate(self.container_lists["Measured Elements"]["All"]):
            if element in ["Sc", "Ti", "V", "Cr", "Mn", "Fe", "Co", "Ni", "Cu", "Zn", "Y", "Zr", "Nb", "Mo", "Tc", "Ru",
                           "Rh", "Pd", "Ag", "Cd", "Hf", "Ta", "W", "Re", "Os", "Ir", "Pt", "Au", "Hg", "Rf", "Db",
                           "Sg", "Bh", "Hs", "Mt", "Ds", "Rg", "Cn"]:
                for oxide in self.chemistry_oxides_sorted[element]:
                    bool_problem = False
                    if oxide in list_major_oxides:
                        var_cb = self.container_var["Oxides Quantification"]["Major"][oxide]
                    else:
                        if oxide in self.container_var["Oxides Quantification"]["Minor"]:
                            var_cb = self.container_var["Oxides Quantification"]["Minor"][oxide]
                        else:
                            bool_problem = True

                    if bool_problem == False:
                        cb_i = tk.Checkbutton(
                            master=frm_03, text=oxide, fg=font_color_dark,
                            bg=background_color_light, variable=var_cb,
                            command=lambda var_checkbox=var_cb, key_oxide=oxide:
                            self.select_oxide2(var_checkbox, key_oxide))
                        text_03.window_create("end", window=cb_i)
                        text_03.insert("end", "\n")
                        counter_03 += 1
                        if element not in self.container_lists["Possible Oxides"]:
                            self.container_lists["Possible Oxides"][element] = []

                        if oxide not in self.container_lists["Possible Oxides"]["All"]:
                            self.container_lists["Possible Oxides"]["All"].append(oxide)
                        if oxide not in self.container_lists["Possible Oxides"][element]:
                            self.container_lists["Possible Oxides"][element].append(oxide)

        # Other metals
        vsb_04 = ttk.Scrollbar(master=frm_04, orient="vertical")
        text_04 = tk.Text(
            master=frm_04, width=30, height=25, yscrollcommand=vsb_04.set, bg=background_color_light)
        vsb_04.config(command=text_04.yview)
        vsb_04.pack(side="right", fill="y")
        text_04.pack(side="left", fill="both", expand=True)
        counter_04 = 0
        for index, element in enumerate(self.container_lists["Measured Elements"]["All"]):
            if element in ["Al", "Ga", "In", "Sn", "Tl", "Pb", "Bi", "Nh", "Fl", "Mc", "Lv"]:
                for oxide in self.chemistry_oxides_sorted[element]:
                    bool_problem = False
                    if oxide in list_major_oxides:
                        var_cb = self.container_var["Oxides Quantification"]["Major"][oxide]
                    else:
                        if oxide in self.container_var["Oxides Quantification"]["Minor"]:
                            var_cb = self.container_var["Oxides Quantification"]["Minor"][oxide]
                        else:
                            bool_problem = True

                    if bool_problem == False:
                        cb_i = tk.Checkbutton(
                            master=frm_04, text=oxide, fg=font_color_dark,
                            bg=background_color_light, variable=var_cb,
                            command=lambda var_checkbox=var_cb, key_oxide=oxide:
                            self.select_oxide2(var_checkbox, key_oxide))
                        text_04.window_create("end", window=cb_i)
                        text_04.insert("end", "\n")
                        counter_04 += 1
                        if element not in self.container_lists["Possible Oxides"]:
                            self.container_lists["Possible Oxides"][element] = []

                        if oxide not in self.container_lists["Possible Oxides"]["All"]:
                            self.container_lists["Possible Oxides"]["All"].append(oxide)
                        if oxide not in self.container_lists["Possible Oxides"][element]:
                            self.container_lists["Possible Oxides"][element].append(oxide)

        # Metalloids
        vsb_05 = ttk.Scrollbar(master=frm_05, orient="vertical")
        text_05 = tk.Text(
            master=frm_05, width=30, height=25, yscrollcommand=vsb_05.set, bg=background_color_light)
        vsb_05.config(command=text_05.yview)
        vsb_05.pack(side="right", fill="y")
        text_05.pack(side="left", fill="both", expand=True)
        counter_05 = 0
        for index, element in enumerate(self.container_lists["Measured Elements"]["All"]):
            if element in ["B", "Si", "Ge", "As", "Sb", "Te", "Po"]:
                for oxide in self.chemistry_oxides_sorted[element]:
                    bool_problem = False
                    if oxide in list_major_oxides:
                        var_cb = self.container_var["Oxides Quantification"]["Major"][oxide]
                    else:
                        if oxide in self.container_var["Oxides Quantification"]["Minor"]:
                            var_cb = self.container_var["Oxides Quantification"]["Minor"][oxide]
                        else:
                            bool_problem = True

                    if bool_problem == False:
                        cb_i = tk.Checkbutton(
                            master=frm_05, text=oxide, fg=font_color_dark,
                            bg=background_color_light, variable=var_cb,
                            command=lambda var_checkbox=var_cb, key_oxide=oxide:
                            self.select_oxide2(var_checkbox, key_oxide))
                        text_05.window_create("end", window=cb_i)
                        text_05.insert("end", "\n")
                        counter_05 += 1
                        if element not in self.container_lists["Possible Oxides"]:
                            self.container_lists["Possible Oxides"][element] = []

                        if oxide not in self.container_lists["Possible Oxides"]["All"]:
                            self.container_lists["Possible Oxides"]["All"].append(oxide)
                        if oxide not in self.container_lists["Possible Oxides"][element]:
                            self.container_lists["Possible Oxides"][element].append(oxide)

        # Lanthanides
        vsb_06 = ttk.Scrollbar(master=frm_06, orient="vertical")
        text_06 = tk.Text(
            master=frm_06, width=30, height=25, yscrollcommand=vsb_06.set, bg=background_color_light)
        vsb_06.config(command=text_06.yview)
        vsb_06.pack(side="right", fill="y")
        text_06.pack(side="left", fill="both", expand=True)
        counter_06 = 0
        for index, element in enumerate(self.container_lists["Measured Elements"]["All"]):
            if element in ["La", "Ce", "Pr", "Nd", "Pm", "Sm", "Eu", "Gd", "Tb", "Dy", "Ho", "Er", "Tm", "Yb", "Lu"]:
                for oxide in self.chemistry_oxides_sorted[element]:
                    bool_problem = False
                    if oxide in list_major_oxides:
                        var_cb = self.container_var["Oxides Quantification"]["Major"][oxide]
                    else:
                        if oxide in self.container_var["Oxides Quantification"]["Minor"]:
                            var_cb = self.container_var["Oxides Quantification"]["Minor"][oxide]
                        else:
                            bool_problem = True

                    if bool_problem == False:
                        cb_i = tk.Checkbutton(
                            master=frm_06, text=oxide, fg=font_color_dark,
                            bg=background_color_light, variable=var_cb,
                            command=lambda var_checkbox=var_cb, key_oxide=oxide:
                            self.select_oxide2(var_checkbox, key_oxide))
                        text_06.window_create("end", window=cb_i)
                        text_06.insert("end", "\n")
                        counter_06 += 1
                        if element not in self.container_lists["Possible Oxides"]:
                            self.container_lists["Possible Oxides"][element] = []

                        if oxide not in self.container_lists["Possible Oxides"]["All"]:
                            self.container_lists["Possible Oxides"]["All"].append(oxide)
                        if oxide not in self.container_lists["Possible Oxides"][element]:
                            self.container_lists["Possible Oxides"][element].append(oxide)

        # Actinides
        vsb_07 = ttk.Scrollbar(master=frm_07, orient="vertical")
        text_07 = tk.Text(
            master=frm_07, width=30, height=25, yscrollcommand=vsb_07.set, bg=background_color_light)
        vsb_07.config(command=text_07.yview)
        vsb_07.pack(side="right", fill="y")
        text_07.pack(side="left", fill="both", expand=True)
        counter_07 = 0
        for index, element in enumerate(self.container_lists["Measured Elements"]["All"]):
            if element in ["Ac", "Th", "Pa", "U", "Np", "Pu", "Am", "Cm", "Bk", "Cf", "Es", "Fm", "Md", "No", "Lr"]:
                for oxide in self.chemistry_oxides_sorted[element]:
                    bool_problem = False
                    if oxide in list_major_oxides:
                        var_cb = self.container_var["Oxides Quantification"]["Major"][oxide]
                    else:
                        if oxide in self.container_var["Oxides Quantification"]["Minor"]:
                            var_cb = self.container_var["Oxides Quantification"]["Minor"][oxide]
                        else:
                            bool_problem = True

                    if bool_problem == False:
                        cb_i = tk.Checkbutton(
                            master=frm_07, text=oxide, fg=font_color_dark,
                            bg=background_color_light, variable=var_cb,
                            command=lambda var_checkbox=var_cb, key_oxide=oxide:
                            self.select_oxide2(var_checkbox, key_oxide))
                        text_07.window_create("end", window=cb_i)
                        text_07.insert("end", "\n")
                        counter_07 += 1
                        if element not in self.container_lists["Possible Oxides"]:
                            self.container_lists["Possible Oxides"][element] = []

                        if oxide not in self.container_lists["Possible Oxides"]["All"]:
                            self.container_lists["Possible Oxides"]["All"].append(oxide)
                        if oxide not in self.container_lists["Possible Oxides"][element]:
                            self.container_lists["Possible Oxides"][element].append(oxide)

        # Non-metals
        vsb_08 = ttk.Scrollbar(master=frm_08, orient="vertical")
        text_08 = tk.Text(
            master=frm_08, width=30, height=25, yscrollcommand=vsb_08.set, bg=background_color_light)
        vsb_08.config(command=text_08.yview)
        vsb_08.pack(side="right", fill="y")
        text_08.pack(side="left", fill="both", expand=True)
        counter_08 = 0
        for index, element in enumerate(self.container_lists["Measured Elements"]["All"]):
            if element in ["H", "C", "N", "O", "P", "S", "Se"]:
                for oxide in self.chemistry_oxides_sorted[element]:
                    bool_problem = False
                    if oxide in list_major_oxides:
                        var_cb = self.container_var["Oxides Quantification"]["Major"][oxide]
                    else:
                        if oxide in self.container_var["Oxides Quantification"]["Minor"]:
                            var_cb = self.container_var["Oxides Quantification"]["Minor"][oxide]
                        else:
                            bool_problem = True

                    if bool_problem == False:
                        cb_i = tk.Checkbutton(
                            master=frm_08, text=oxide, fg=font_color_dark,
                            bg=background_color_light, variable=var_cb,
                            command=lambda var_checkbox=var_cb, key_oxide=oxide:
                            self.select_oxide2(var_checkbox, key_oxide))
                        text_08.window_create("end", window=cb_i)
                        text_08.insert("end", "\n")
                        counter_08 += 1
                        if element not in self.container_lists["Possible Oxides"]:
                            self.container_lists["Possible Oxides"][element] = []

                        if oxide not in self.container_lists["Possible Oxides"]["All"]:
                            self.container_lists["Possible Oxides"]["All"].append(oxide)
                        if oxide not in self.container_lists["Possible Oxides"][element]:
                            self.container_lists["Possible Oxides"][element].append(oxide)

        # Halogens
        vsb_09 = ttk.Scrollbar(master=frm_09, orient="vertical")
        text_09 = tk.Text(
            master=frm_09, width=30, height=25, yscrollcommand=vsb_09.set, bg=background_color_light)
        vsb_09.config(command=text_09.yview)
        vsb_09.pack(side="right", fill="y")
        text_09.pack(side="left", fill="both", expand=True)
        counter_09 = 0
        for index, element in enumerate(self.container_lists["Measured Elements"]["All"]):
            if element in ["F", "Cl", "Br", "I", "At", "Ts"]:
                for oxide in self.chemistry_oxides_sorted[element]:
                    bool_problem = False
                    if oxide in list_major_oxides:
                        var_cb = self.container_var["Oxides Quantification"]["Major"][oxide]
                    else:
                        if oxide in self.container_var["Oxides Quantification"]["Minor"]:
                            var_cb = self.container_var["Oxides Quantification"]["Minor"][oxide]
                        else:
                            bool_problem = True

                    if bool_problem == False:
                        cb_i = tk.Checkbutton(
                            master=frm_09, text=oxide, fg=font_color_dark,
                            bg=background_color_light, variable=var_cb,
                            command=lambda var_checkbox=var_cb, key_oxide=oxide:
                            self.select_oxide2(var_checkbox, key_oxide))
                        text_09.window_create("end", window=cb_i)
                        text_09.insert("end", "\n")
                        counter_09 += 1
                        if element not in self.container_lists["Possible Oxides"]:
                            self.container_lists["Possible Oxides"][element] = []

                        if oxide not in self.container_lists["Possible Oxides"]["All"]:
                            self.container_lists["Possible Oxides"]["All"].append(oxide)
                        if oxide not in self.container_lists["Possible Oxides"][element]:
                            self.container_lists["Possible Oxides"][element].append(oxide)

        # Noble gases
        vsb_10 = ttk.Scrollbar(master=frm_10, orient="vertical")
        text_10 = tk.Text(
            master=frm_10, width=30, height=25, yscrollcommand=vsb_10.set, bg=background_color_light)
        vsb_10.config(command=text_10.yview)
        vsb_10.pack(side="right", fill="y")
        text_10.pack(side="left", fill="both", expand=True)
        counter_10 = 0
        for index, element in enumerate(self.container_lists["Measured Elements"]["All"]):
            if element in ["He", "Ne", "Ar", "Kr", "Xe", "Rn", "Og"]:
                for oxide in self.chemistry_oxides_sorted[element]:
                    bool_problem = False
                    if oxide in list_major_oxides:
                        var_cb = self.container_var["Oxides Quantification"]["Major"][oxide]
                    else:
                        if oxide in self.container_var["Oxides Quantification"]["Minor"]:
                            var_cb = self.container_var["Oxides Quantification"]["Minor"][oxide]
                        else:
                            bool_problem = True

                    if bool_problem == False:
                        cb_i = tk.Checkbutton(
                            master=frm_10, text=oxide, fg=font_color_dark,
                            bg=background_color_light, variable=var_cb,
                            command=lambda var_checkbox=var_cb, key_oxide=oxide:
                            self.select_oxide2(var_checkbox, key_oxide))
                        text_10.window_create("end", window=cb_i)
                        text_10.insert("end", "\n")
                        counter_10 += 1
                        if element not in self.container_lists["Possible Oxides"]:
                            self.container_lists["Possible Oxides"][element] = []

                        if oxide not in self.container_lists["Possible Oxides"]["All"]:
                            self.container_lists["Possible Oxides"]["All"].append(oxide)
                        if oxide not in self.container_lists["Possible Oxides"][element]:
                            self.container_lists["Possible Oxides"][element].append(oxide)

        if counter_01 == 0:
            btn_01.configure(state="disabled")
        if counter_02 == 0:
            btn_02.configure(state="disabled")
        if counter_03 == 0:
            btn_03.configure(state="disabled")
        if counter_04 == 0:
            btn_04.configure(state="disabled")
        if counter_05 == 0:
            btn_05.configure(state="disabled")
        if counter_06 == 0:
            btn_06.configure(state="disabled")
        if counter_07 == 0:
            btn_07.configure(state="disabled")
        if counter_08 == 0:
            btn_08.configure(state="disabled")
        if counter_09 == 0:
            btn_09.configure(state="disabled")
        if counter_10 == 0:
            btn_10.configure(state="disabled")

        ## OPTION MENUS
        var_opt_mat_isotope = tk.StringVar()
        var_opt_mat_isotope.set(str_lbl_01)
        var_opt_incl_isotope = tk.StringVar()
        var_opt_incl_isotope.set(str_lbl_01)

        opt_mat_isotope = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start + 9, column_id=5*var_header_n + 1 + int_category_n,
            n_rows=1, n_columns=int(0.75*var_header_n), fg=font_color_dark,
            bg=background_color_elements).create_option_isotope(
            var_iso=var_opt_mat_isotope, option_list=list_mat_isotopes, text_set=var_opt_mat_isotope.get(),
            fg_active=font_color_light, bg_active=accent_color,
            command=lambda var_opt=var_opt_incl_isotope, mode="MAT": self.select_reference_isotope(var_opt, mode))
        opt_mat_isotope["menu"].config(
            fg=font_color_dark, bg=background_color_elements,
            activeforeground=font_color_light,
            activebackground=accent_color)
        opt_mat_isotope.config(
            fg=font_color_dark, bg=background_color_elements,
            activeforeground=font_color_light,
            activebackground=accent_color, highlightthickness=0)

        opt_incl_isotope = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start + 10, column_id=5*var_header_n + 1 + int_category_n,
            n_rows=1, n_columns=int(0.75*var_header_n), fg=font_color_dark,
            bg=background_color_elements).create_option_isotope(
            var_iso=var_opt_incl_isotope, option_list=list_incl_isotopes, text_set=var_opt_incl_isotope.get(),
            fg_active=font_color_light, bg_active=accent_color,
            command=lambda var_opt=var_opt_incl_isotope, mode="INCL": self.select_reference_isotope(var_opt, mode))
        opt_incl_isotope["menu"].config(
            fg=font_color_dark, bg=background_color_elements,
            activeforeground=font_color_light,
            activebackground=accent_color)
        opt_incl_isotope.config(
            fg=font_color_dark, bg=background_color_elements,
            activeforeground=font_color_light,
            activebackground=accent_color, highlightthickness=0)

        if self.pysills_mode in ["MA", "FI"]:
            opt_incl_isotope.configure(state="disabled")

        ## CHECKBOXES
        cb_01 = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start + 18, column_id=5*var_header_n + 1,
            fg=font_color_dark, n_rows=1, n_columns=int(1.5*var_header_n),
            bg=background_color_elements).create_simple_checkbox(
            var_cb=self.oxide_calculation_mat, text=str_lbl_02 + "/"+ str_lbl_03, set_sticky="nesw", own_color=True)
        cb_02 = SE(
            parent=subwindow_checkup_oxides, row_id=var_row_start + 19, column_id=5*var_header_n + 1,
            fg=font_color_dark, n_rows=1, n_columns=int(1.5*var_header_n),
            bg=background_color_elements).create_simple_checkbox(
            var_cb=self.oxide_calculation_incl, text=str_lbl_04, set_sticky="nesw", own_color=True)

        if self.pysills_mode in ["MA", "FI"]:
            cb_02.configure(state="disabled")

    def select_oxide2(self, var_checkbox, key_oxide):
        if var_checkbox.get() == 1:
            if key_oxide not in self.container_lists["Selected Oxides"]["All"]:
                self.container_lists["Selected Oxides"]["All"].append(key_oxide)
        else:
            if key_oxide in self.container_lists["Selected Oxides"]["All"]:
                self.container_lists["Selected Oxides"]["All"].remove(key_oxide)

    def select_main_rockforming_elements(self):
        list_oxides = ["SiO2", "TiO2", "Al2O3", "Fe2O3", "FeO", "Mn2O3", "MnO", "MgO", "CaO", "Na2O", "K2O", "P2O5"]
        self.container_lists["Selected Oxides"]["All"].clear()
        for oxide, variable in self.container_var["Oxides Quantification"]["Major"].items():
            if oxide in list_oxides and oxide in self.container_lists["Possible Oxides"]["All"]:
                variable.set(1)
                if oxide not in self.container_lists["Selected Oxides"]["All"]:
                    self.container_lists["Selected Oxides"]["All"].append(oxide)
        for oxide, variable in self.container_var["Oxides Quantification"]["Minor"].items():
            if oxide in list_oxides and oxide in self.container_lists["Possible Oxides"]["All"]:
                variable.set(1)
                if oxide not in self.container_lists["Selected Oxides"]["All"]:
                    self.container_lists["Selected Oxides"]["All"].append(oxide)

    def select_reference_isotope(self, var_opt, mode="MAT"):
        for index, filename_long in enumerate(self.container_lists["SMPL"]["Long"]):
            if mode == "MAT":
                if self.pysills_mode == "MA":
                    self.container_var["SMPL"][filename_long]["IS Data"]["IS"].set(var_opt)
                else:
                    self.container_var["SMPL"][filename_long]["Matrix Setup"]["IS"]["Name"].set(var_opt)
            else:
                self.container_var["SMPL"][filename_long]["IS Data"]["IS"].set(var_opt)

    def select_all_oxides(self):
        self.container_lists["Selected Oxides"]["All"].clear()
        for oxide, variable in self.container_var["Oxides Quantification"]["Major"].items():
            if oxide in self.container_lists["Possible Oxides"]["All"]:
                variable.set(1)
                if oxide not in self.container_lists["Selected Oxides"]["All"]:
                    self.container_lists["Selected Oxides"]["All"].append(oxide)
        for oxide, variable in self.container_var["Oxides Quantification"]["Minor"].items():
            if oxide in self.container_lists["Possible Oxides"]["All"]:
                variable.set(1)
                if oxide not in self.container_lists["Selected Oxides"]["All"]:
                    self.container_lists["Selected Oxides"]["All"].append(oxide)

    def deselect_all_oxides(self):
        self.container_lists["Selected Oxides"]["All"].clear()
        for oxide, variable in self.container_var["Oxides Quantification"]["Major"].items():
            if oxide in self.container_lists["Possible Oxides"]["All"]:
                variable.set(0)
        for oxide, variable in self.container_var["Oxides Quantification"]["Minor"].items():
            if oxide in self.container_lists["Possible Oxides"]["All"]:
                variable.set(0)

    def checkup_internal_standard(self):
        """Check-up window to control the internal standard setup."""
        # Colors
        font_color_dark = self.bg_colors["Dark Font"]
        font_color_light = self.bg_colors["Light Font"]
        background_color_dark = self.bg_colors["BG Window"]
        background_color_elements = self.bg_colors["Light"]
        background_color_light = self.bg_colors["Very Light"]
        accent_color = self.bg_colors["Accent"]  # self.accent_color
        font_header = self.font_settings["Header"]
        font_elements = self.font_settings["Elements"]
        font_option = self.font_settings["Options"]
        font_table = self.font_settings["Table"]

        str_lbl_01 = self.language_dict["Check-Up"][self.var_language]
        str_lbl_02 = self.language_dict["Internal standard"][self.var_language]

        ## Window Settings
        row_min = self.row_height
        column_min = self.column_width
        n_rows = self.window_dimensions["Check-up IS"][0]
        n_columns = self.window_dimensions["Check-up IS"][1]

        window_width = int(n_columns*self.column_width)
        window_height = int(n_rows*self.row_height)

        var_geometry = str(window_width) + "x" + str(window_height) + "+" + str(0) + "+" + str(0)

        subwindow_checkup_is = tk.Toplevel(self.parent)
        subwindow_checkup_is.title(str_lbl_01 + " - " + str_lbl_02)
        subwindow_checkup_is.geometry(var_geometry)
        subwindow_checkup_is.resizable(False, False)
        subwindow_checkup_is["bg"] = background_color_dark

        for x in range(n_columns):
            tk.Grid.columnconfigure(subwindow_checkup_is, x, weight=1)
        for y in range(n_rows):
            tk.Grid.rowconfigure(subwindow_checkup_is, y, weight=1)

        # Rows
        for i in range(0, n_rows):
            subwindow_checkup_is.grid_rowconfigure(i, minsize=row_min)
        # Columns
        for i in range(0, n_columns):
            subwindow_checkup_is.grid_columnconfigure(i, minsize=column_min)

        var_row_start = 0
        var_column_start = 0
        var_header_n = 20
        int_category_n = 6

        ## LABELS
        str_lbl_03 = self.language_dict["Sample"][self.var_language]
        str_lbl_04 = self.language_dict["Matrix"][self.var_language]
        str_lbl_05 = self.language_dict["Inclusion"][self.var_language]
        str_lbl_06 = self.language_dict["Select isotope"][self.var_language]

        lbl_01 = SE(
            parent=subwindow_checkup_is, row_id=var_row_start, column_id=var_column_start, n_rows=1,
            n_columns=var_header_n, fg=font_color_light,
            bg=background_color_dark).create_simple_label(
            text=str_lbl_02 + " (" + str_lbl_03 + "/" + str_lbl_04 + ")", relief=tk.FLAT, fontsize="sans 10 bold")
        lbl_02 = SE(
            parent=subwindow_checkup_is, row_id=var_row_start, column_id=var_header_n + 1, n_rows=1,
            n_columns=var_header_n, fg=font_color_light,
            bg=background_color_dark).create_simple_label(
            text=str_lbl_02 + " (" + str_lbl_05 + ")", relief=tk.FLAT, fontsize="sans 10 bold")

        ## TREEVIEWS
        list_is_mat = self.container_lists["Measured Isotopes"]["All"]
        list_is_incl = self.container_lists["Measured Isotopes"]["All"]

        frm_mat = SE(
            parent=subwindow_checkup_is, row_id=var_row_start + 1, column_id=var_column_start, n_rows=n_rows - 3,
            n_columns=var_header_n, fg=font_color_dark, bg=background_color_light).create_frame()
        vsb_mat = ttk.Scrollbar(master=frm_mat, orient="vertical")
        text_mat = tk.Text(
            master=frm_mat, width=30, height=25, yscrollcommand=vsb_mat.set, bg=background_color_light)
        vsb_mat.config(command=text_mat.yview)
        vsb_mat.pack(side="right", fill="y")
        text_mat.pack(side="left", fill="both", expand=True)

        for index, filename_short in enumerate(self.container_lists["SMPL"]["Short"]):
            filename_long = self.container_lists["SMPL"]["Long"][index]
            lbl_i = tk.Label(
                frm_mat, text=filename_short, bg=background_color_light, fg=font_color_dark)
            text_mat.window_create("end", window=lbl_i)
            text_mat.insert("end", "\t")

            if self.pysills_mode == "MA":
                var_opt_is = self.container_var["SMPL"][filename_long]["IS Data"]["IS"]
                var_entr_is = self.container_var["SMPL"][filename_long]["IS Data"]["Concentration"]
            else:
                var_opt_is = self.container_var["SMPL"][filename_long]["Matrix Setup"]["IS"]["Name"]
                var_entr_is = self.container_var["SMPL"][filename_long]["Matrix Setup"]["IS"]["Concentration"]

            opt_is_i = tk.OptionMenu(
                frm_mat, var_opt_is, *list_is_mat)
            opt_is_i["menu"].config(
                fg=font_color_dark, bg=background_color_elements,
                activeforeground=font_color_light, activebackground=accent_color)
            opt_is_i.config(
                bg=background_color_elements, fg=font_color_dark, activebackground=accent_color,
                activeforeground=font_color_light, highlightthickness=0)
            text_mat.window_create("end", window=opt_is_i)
            text_mat.insert("end", "\t")

            var_opt_is_mat_i = tk.StringVar()
            var_entr_is_mat_i = tk.StringVar()
            var_opt_is_mat_i.set("Select isotope")
            var_entr_is_mat_i.set("0.0")

            entr_i = tk.Entry(
                frm_mat, textvariable=var_entr_is, fg=font_color_dark, bg=self.bg_colors["White"],
                highlightthickness=0, highlightbackground=background_color_light, width=12)
            text_mat.window_create("insert", window=entr_i)
            text_mat.insert("end", "\n")

        frm_incl = SE(
            parent=subwindow_checkup_is, row_id=var_row_start + 1, column_id=var_header_n + 1, n_rows=n_rows - 3,
            n_columns=var_header_n, fg=font_color_dark, bg=background_color_light).create_frame()
        vsb_incl = ttk.Scrollbar(master=frm_incl, orient="vertical")
        text_incl = tk.Text(
            master=frm_incl, width=30, height=25, yscrollcommand=vsb_incl.set, bg=background_color_light)
        vsb_incl.config(command=text_incl.yview)
        vsb_incl.pack(side="right", fill="y")
        text_incl.pack(side="left", fill="both", expand=True)

        if self.pysills_mode != "MA":
            for index, filename_short in enumerate(self.container_lists["SMPL"]["Short"]):
                filename_long = self.container_lists["SMPL"]["Long"][index]
                lbl_i = tk.Label(
                    frm_incl, text=filename_short, bg=background_color_light, fg=font_color_dark)
                text_incl.window_create("end", window=lbl_i)
                text_incl.insert("end", "\t")

                var_opt_is_incl_i = tk.StringVar()
                var_entr_is_incl_i = tk.StringVar()
                var_opt_is_incl_i.set(str_lbl_06)
                var_entr_is_incl_i.set("0.0")

                opt_is_i = tk.OptionMenu(
                    frm_incl, self.container_var["SMPL"][filename_long]["IS Data"]["IS"], *list_is_incl)
                opt_is_i["menu"].config(
                    fg=font_color_dark, bg=background_color_elements,
                    activeforeground=font_color_light, activebackground=accent_color)
                opt_is_i.config(
                    bg=background_color_elements, fg=font_color_dark, activebackground=accent_color,
                    activeforeground=font_color_light, highlightthickness=0)
                text_incl.window_create("end", window=opt_is_i)
                text_incl.insert("end", "\t")

                entr_i = tk.Entry(
                    frm_incl, textvariable=self.container_var["SMPL"][filename_long]["IS Data"]["Concentration"],
                    fg=font_color_dark, bg=self.bg_colors["White"], highlightthickness=0,
                    highlightbackground=background_color_light, width=12)
                text_incl.window_create("insert", window=entr_i)
                text_incl.insert("end", "\n")

        ## OPTION MENUS
        self.var_opt_is_mat_default = tk.StringVar()
        self.var_opt_is_incl_default = tk.StringVar()
        self.var_opt_is_mat_default.set(str_lbl_06)
        self.var_opt_is_incl_default.set(str_lbl_06)

        opt_is_mat = SE(
            parent=subwindow_checkup_is, row_id=n_rows - 2, column_id=var_header_n - 2*int_category_n, n_rows=1,
            n_columns=int_category_n, fg=font_color_dark, bg=background_color_elements).create_option_isotope(
            var_iso=self.var_opt_is_mat_default, option_list=list_is_mat, text_set=self.var_opt_is_mat_default.get(),
            fg_active=font_color_light, bg_active=accent_color,
            command=lambda var_opt=self.var_opt_is_mat_default: self.change_default_is(var_opt))
        opt_is_mat["menu"].config(
            fg=font_color_dark, bg=background_color_elements,
            activeforeground=font_color_light,
            activebackground=accent_color)
        opt_is_mat.config(
            fg=font_color_dark, bg=background_color_elements,
            activeforeground=font_color_light,
            activebackground=accent_color, highlightthickness=0)

        if self.pysills_mode != "MA":
            opt_is_incl = SE(
                parent=subwindow_checkup_is, row_id=n_rows - 2, column_id=2*var_header_n + 1 - 2*int_category_n,
                n_rows=1, n_columns=int_category_n, fg=font_color_dark,
                bg=background_color_elements).create_option_isotope(
                var_iso=self.var_opt_is_incl_default, option_list=list_is_incl,
                text_set=self.var_opt_is_incl_default.get(), fg_active=font_color_light,
                bg_active=accent_color,
                command=lambda var_opt=self.var_opt_is_mat_default, mode="INCL": self.change_default_is(var_opt, mode))
            opt_is_incl["menu"].config(
                fg=font_color_dark, bg=background_color_elements,
                activeforeground=font_color_light,
                activebackground=accent_color)
            opt_is_incl.config(
                fg=font_color_dark, bg=background_color_elements,
                activeforeground=font_color_light,
                activebackground=accent_color, highlightthickness=0)

        ## ENTRIES
        self.var_entr_is_mat_default = tk.StringVar()
        self.var_entr_is_incl_default = tk.StringVar()
        self.var_entr_is_mat_default.set("0.0")
        self.var_entr_is_incl_default.set("0.0")

        entr_is_mat = SE(
            parent=subwindow_checkup_is, row_id=n_rows - 2, column_id=var_header_n - int_category_n, n_rows=1,
            n_columns=int_category_n, fg=font_color_dark,
            bg=self.bg_colors["White"]).create_simple_entry(
            var=self.var_entr_is_mat_default, text_default=self.var_entr_is_mat_default.get())
        entr_is_mat.bind(
            "<Return>", lambda event, var_entr=self.var_entr_is_mat_default, mode="MAT":
            self.change_default_is_value(var_entr, mode, event))

        if self.pysills_mode != "MA":
            entr_is_incl = SE(
                parent=subwindow_checkup_is, row_id=n_rows - 2, column_id=2*var_header_n + 1 - int_category_n, n_rows=1,
                n_columns=int_category_n, fg=font_color_dark,
                bg=self.bg_colors["White"]).create_simple_entry(
                var=self.var_entr_is_incl_default, text_default=self.var_entr_is_incl_default.get())
            entr_is_incl.bind(
                "<Return>", lambda event, var_entr=self.var_entr_is_incl_default, mode="INCL":
                self.change_default_is_value(var_entr, mode, event))

        ## INITIALIZATION
        try:
            self.srm_isotopes
        except:
            self.srm_isotopes = {}
            for isotope in self.container_lists["Measured Isotopes"]["All"]:
                self.srm_isotopes[isotope] = {}
                var_srm = self.container_var["SRM"][isotope].get()
                key_element = re.search(r"(\D+)(\d+)", isotope)
                var_element = key_element.group(1)
                self.srm_isotopes[isotope]["SRM"] = var_srm
                if var_element in self.srm_actual[var_srm]:
                    self.srm_isotopes[isotope]["Concentration"] = self.srm_actual[var_srm][var_element]
                else:
                    self.srm_isotopes[isotope]["Concentration"] = 0.0

    def change_default_is(self, var_opt, mode="MAT"):
        if mode == "MAT":
            for index, filename_long in enumerate(self.container_lists["SMPL"]["Long"]):
                if self.pysills_mode == "MA":
                    self.container_var["SMPL"][filename_long]["IS Data"]["IS"].set(var_opt)
                else:
                    self.container_var["SMPL"][filename_long]["Matrix Setup"]["IS"]["Name"].set(var_opt)
        else:
            for index, filename_long in enumerate(self.container_lists["SMPL"]["Long"]):
                self.container_var["SMPL"][filename_long]["IS Data"]["IS"].set(var_opt)

    def change_default_is_value(self, var_entr, mode, event):
        val_entr = var_entr.get()
        if mode == "MAT":
            for index, filename_long in enumerate(self.container_lists["SMPL"]["Long"]):
                if self.pysills_mode == "MA":
                    self.container_var["SMPL"][filename_long]["IS Data"]["Concentration"].set(val_entr)
                else:
                    self.container_var["SMPL"][filename_long]["Matrix Setup"]["IS"]["Concentration"].set(val_entr)
        else:
            for index, filename_long in enumerate(self.container_lists["SMPL"]["Long"]):
                self.container_var["SMPL"][filename_long]["IS Data"]["Concentration"].set(val_entr)

    def change_is_value_default(self, var_entr, event):
        for file_smpl in self.container_lists["SMPL"]["Long"]:
            self.container_var["SMPL"][file_smpl]["IS Data"]["Concentration"].set(var_entr.get())

    def change_is_setup(self, var_is, var_file=None, mode="Default"):
        if mode == "Default":
            try:
                value_is = self.mineral_chem["Unknown"][var_is]
            except:
                value_is = 0.0

            self.container_var["IS SMPL Default"].set(value_is)

            for file in self.container_lists["SMPL"]["Long"]:
                self.container_var["SMPL"][file]["IS Data"]["IS"].set(var_is)
                self.container_var["SMPL"][file]["IS Data"]["Concentration"].set(value_is)

                file_short = file.split("/")[-1]
                if file_short not in self.container_files["SMPL"]:
                    self.container_files["SMPL"][file_short] = {
                        "IS": tk.StringVar(), "IS Concentration": tk.StringVar()}
                    self.container_files["SMPL"][file_short]["IS"].set(var_is)
                    self.container_files["SMPL"][file_short]["IS Concentration"].set(value_is)
                else:
                    self.container_files["SMPL"][file_short]["IS"].set(var_is)
                    self.container_files["SMPL"][file_short]["IS Concentration"].set(value_is)

        else:
            try:
                value_is = self.mineral_chem["Unknown"][var_is]
            except:
                value_is = 0.0

            self.container_var["SMPL"][var_file]["IS Data"]["IS"].set(var_is)
            self.container_var["SMPL"][var_file]["IS Data"]["Concentration"].set(value_is)

            file_short = var_file.split("/")[-1]

            if file_short not in self.container_files["SMPL"]:
                self.container_files["SMPL"][file_short] = {
                    "IS": tk.StringVar(), "IS Concentration": tk.StringVar()}
                self.container_files["SMPL"][file_short]["IS"].set(var_is)
                self.container_files["SMPL"][file_short]["IS Concentration"].set(value_is)
            else:
                self.container_files["SMPL"][file_short]["IS"].set(var_is)
                self.container_files["SMPL"][file_short]["IS Concentration"].set(value_is)

    # def srm_window(self):
    #     ## Window Settings
    #     window_width = 860
    #     window_height = 625
    #     var_geometry = str(window_width) + "x" + str(window_height) + "+" + str(0) + "+" + str(0)
    #
    #     window_srm = tk.Toplevel(self.parent)
    #     window_srm.title("SRM and Mineral Composition Data")
    #     window_srm.geometry(var_geometry)
    #     window_srm.resizable(False, False)
    #     window_srm["bg"] = self.green_light
    #
    #     row_min = 25
    #     n_rows = int(window_height/row_min)
    #     column_min = 20
    #     n_columns = int(window_width/column_min)
    #
    #     for x in range(n_columns):
    #         tk.Grid.columnconfigure(window_srm, x, weight=1)
    #     for y in range(n_rows):
    #         tk.Grid.rowconfigure(window_srm, y, weight=1)
    #
    #     # Rows
    #     for i in range(0, n_rows):
    #         window_srm.grid_rowconfigure(i, minsize=row_min)
    #     # Columns
    #     for i in range(0, n_columns):
    #         window_srm.grid_columnconfigure(i, minsize=column_min)
    #
    #     ###########################################################
    #
    #     ## Frames
    #     frm_01 = SE(
    #         parent=window_srm, row_id=0, column_id=0, n_rows=n_rows, n_columns=9,
    #         fg=self.green_light, bg=self.green_dark).create_frame(relief=tk.FLAT)
    #
    #     ## Labels
    #     lbl_01 = SE(
    #         parent=window_srm, row_id=0, column_id=0, n_rows=1, n_columns=9, fg=self.green_light,
    #         bg=self.green_dark).create_simple_label(
    #         text="Select SRM", relief=tk.GROOVE, fontsize="sans 10 bold")
    #     lbl_02 = SE(
    #         parent=window_srm, row_id=2, column_id=0, n_rows=1, n_columns=9, fg=self.green_light,
    #         bg=self.green_dark).create_simple_label(
    #         text="Select Mineral", relief=tk.GROOVE, fontsize="sans 10 bold")
    #     lbl_03 = SE(
    #         parent=window_srm, row_id=0, column_id=10, n_rows=1, n_columns=32, fg=self.green_light,
    #         bg=self.green_dark).create_simple_label(
    #         text="Periodic Table of the Elements", relief=tk.GROOVE, fontsize="sans 10 bold")
    #     lbl_04 = SE(
    #         parent=window_srm, row_id=n_rows - 4, column_id=0, n_rows=4, n_columns=9, fg=self.green_dark,
    #         bg=self.red_dark).create_simple_label(
    #         text="CAUTION!\n The mineral composition\n data reflects only the\n ideal composition of\n pure endmembers",
    #         relief=tk.GROOVE, fontsize="sans 10 bold")
    #
    #     ## Option Menues
    #     list_srm = np.sort(self.list_srm)
    #     opt_srm = SE(
    #         parent=window_srm, row_id=1, column_id=0, n_rows=1, n_columns=9, fg=self.green_dark,
    #         bg=self.green_medium).create_simple_optionmenu(
    #         var_opt=self.container_var["srm_window"]["Option Menu"]["SRM"], var_default="Select SRM",
    #         var_list=list_srm, fg_active=self.green_dark, bg_active=self.red_dark,
    #         command=lambda var_opt=self.container_var["srm_window"]["Option Menu"]["SRM"]:
    #         self.change_option_srm_window(var_opt))
    #
    #     list_minerals = np.sort(self.mineral_list)
    #     opt_mineral = SE(
    #         parent=window_srm, row_id=3, column_id=0, n_rows=1, n_columns=9, fg=self.green_dark,
    #         bg=self.green_medium).create_simple_optionmenu(
    #         var_opt=self.container_var["srm_window"]["Option Menu"]["Mineral"], var_default="Select Mineral",
    #         var_list=list_minerals, fg_active=self.green_dark, bg_active=self.red_dark,
    #         command=lambda var_opt=self.container_var["srm_window"]["Option Menu"]["Mineral"]:
    #         self.change_option_srm_window(var_opt))
    #
    #     ## Labels and Entries
    #     start_column_pse = 10
    #     for index, element in enumerate(self.list_pse):
    #         if 0 <= index < 23:
    #             lbl_i = SE(
    #                 parent=window_srm, row_id=index + 1, column_id=start_column_pse, n_rows=1, n_columns=3,
    #                 fg=self.green_light, bg=self.green_medium).create_simple_label(
    #                 text=str(index + 1) + " " + element, relief=tk.GROOVE, fontsize="sans 10 bold", anchor="w")
    #
    #             if element not in self.container_var["srm_window"]["Entry"]:
    #                 self.container_var["srm_window"]["Entry"][element] = tk.StringVar()
    #
    #             entr_i = SE(
    #                 parent=window_srm, row_id=index + 1, column_id=start_column_pse + 3, n_rows=1, n_columns=5,
    #                 fg=self.green_light, bg=self.green_dark).create_simple_entry(
    #                 var=self.container_var["srm_window"]["Entry"][element], text_default="0.0", command=None)
    #         elif 23 <= index < 46:
    #             lbl_i = SE(
    #                 parent=window_srm, row_id=index + 1 - 23, column_id=start_column_pse + 8, n_rows=1, n_columns=3,
    #                 fg=self.green_light, bg=self.green_medium).create_simple_label(
    #                 text=str(index + 1) + " " + element, relief=tk.GROOVE, fontsize="sans 10 bold", anchor="w")
    #
    #             if element not in self.container_var["srm_window"]["Entry"]:
    #                 self.container_var["srm_window"]["Entry"][element] = tk.StringVar()
    #
    #             entr_i = SE(
    #                 parent=window_srm, row_id=index + 1 - 23, column_id=start_column_pse + 11, n_rows=1, n_columns=5,
    #                 fg=self.green_light, bg=self.green_dark).create_simple_entry(
    #                 var=self.container_var["srm_window"]["Entry"][element], text_default="0.0", command=None)
    #         elif 46 <= index < 69:
    #             lbl_i = SE(
    #                 parent=window_srm, row_id=index + 1 - 46, column_id=start_column_pse + 16, n_rows=1, n_columns=3,
    #                 fg=self.green_light, bg=self.green_medium).create_simple_label(
    #                 text=str(index + 1) + " " + element, relief=tk.GROOVE, fontsize="sans 10 bold", anchor="w")
    #
    #             if element not in self.container_var["srm_window"]["Entry"]:
    #                 self.container_var["srm_window"]["Entry"][element] = tk.StringVar()
    #
    #             entr_i = SE(
    #                 parent=window_srm, row_id=index + 1 - 46, column_id=start_column_pse + 19, n_rows=1, n_columns=5,
    #                 fg=self.green_light, bg=self.green_dark).create_simple_entry(
    #                 var=self.container_var["srm_window"]["Entry"][element], text_default="0.0", command=None)
    #         elif 69 <= index < 92:
    #             lbl_i = SE(
    #                 parent=window_srm, row_id=index + 1 - 69, column_id=start_column_pse + 24, n_rows=1, n_columns=3,
    #                 fg=self.green_light, bg=self.green_medium).create_simple_label(
    #                 text=str(index + 1) + " " + element, relief=tk.GROOVE, fontsize="sans 10 bold", anchor="w")
    #
    #             if element not in self.container_var["srm_window"]["Entry"]:
    #                 self.container_var["srm_window"]["Entry"][element] = tk.StringVar()
    #
    #             entr_i = SE(
    #                 parent=window_srm, row_id=index + 1 - 69, column_id=start_column_pse + 27, n_rows=1, n_columns=5,
    #                 fg=self.green_light, bg=self.green_dark).create_simple_entry(
    #                 var=self.container_var["srm_window"]["Entry"][element], text_default="0.0", command=None)

    def change_option_srm_window(self, var_opt):
        self.srm_values = {}
        self.srm_values[var_opt] = {}
        #
        if var_opt in self.list_srm:
            self.container_var["srm_window"]["Option Menu"]["Mineral"].set("Select Mineral")
            #
            ESRM().place_srm_values(srm_name=var_opt, srm_dict=self.srm_values)
            #
            for element in self.list_pse:
                if element in self.srm_values[var_opt].keys():
                    self.container_var["srm_window"]["Entry"][element].set(self.srm_values[var_opt][element])
                else:
                    self.container_var["srm_window"]["Entry"][element].set("0.0")
            #
        elif var_opt in self.mineral_list:
            self.container_var["srm_window"]["Option Menu"]["SRM"].set("Select SRM")
            #
            for element in self.list_pse:
                if element in self.mineral_chem[var_opt].keys():
                    self.container_var["srm_window"]["Entry"][element].set(self.mineral_chem[var_opt][element])
                else:
                    self.container_var["srm_window"]["Entry"][element].set("0.0")

    def create_srm_data_list(self):
        for var_srm in self.list_srm:
            self.container_lists["SRM Data"][var_srm] = {}
            ESRM().place_srm_values(srm_name=var_srm, srm_dict=self.container_lists["SRM Data"])

        for var_srm in self.list_srm:
            data_srm = self.container_lists["SRM Data"][var_srm]
            for element in self.list_pse:
                if element not in self.container_lists["SRM Data"]:
                    self.container_lists["SRM Data"][element] = {}

                if element in data_srm:
                    self.container_lists["SRM Data"][element][var_srm] = data_srm[element]

    def subwindow_general_settings(self):
        # Colors
        font_color_dark = self.bg_colors["Dark Font"]
        font_color_light = self.bg_colors["Light Font"]
        background_color_dark = self.bg_colors["BG Window"]
        background_color_elements = self.bg_colors["Light"]
        background_color_light = self.bg_colors["Very Light"]
        accent_color = self.bg_colors["Accent"]  # self.accent_color
        font_header = self.font_settings["Header"]
        font_element = self.font_settings["Elements"]
        font_option = self.font_settings["Options"]
        font_table = self.font_settings["Table"]

        var_title = self.language_dict["General Settings"][self.var_language]

        ## Window Settings
        row_min = self.row_height
        column_min = self.column_width
        n_rows = self.window_dimensions["General settings"][0]
        n_columns = self.window_dimensions["General settings"][1]

        window_width = int(n_columns*self.column_width)
        window_height = int(n_rows*self.row_height)

        var_geometry = str(window_width) + "x" + str(window_height) + "+" + str(0) + "+" + str(0)

        subwindow_generalsettings = tk.Toplevel(self.parent)
        subwindow_generalsettings.title(var_title)
        subwindow_generalsettings.geometry(var_geometry)
        subwindow_generalsettings.resizable(False, False)
        subwindow_generalsettings["bg"] = background_color_dark

        for x in range(n_columns):
            tk.Grid.columnconfigure(subwindow_generalsettings, x, weight=1)
        for y in range(n_rows):
            tk.Grid.rowconfigure(subwindow_generalsettings, y, weight=1)

        # Rows
        for i in range(0, n_rows):
            subwindow_generalsettings.grid_rowconfigure(i, minsize=row_min)
        # Columns
        for i in range(0, n_columns):
            subwindow_generalsettings.grid_columnconfigure(i, minsize=column_min)

        ###########################################################
        start_row = 0
        start_column = 0

        ## Labels
        str_lbl_01 = self.language_dict["Project Information"][self.var_language]
        str_lbl_02 = self.language_dict["Calculation Accuracy"][self.var_language]
        str_lbl_03 = self.language_dict["Sensitivity Drift"][self.var_language]
        str_lbl_04 = self.language_dict["Limit of Detection"][self.var_language]
        str_lbl_05 = self.language_dict["Offset Automatic Interval Detection"][self.var_language]
        str_lbl_06 = self.language_dict["Colormap"][self.var_language]
        str_lbl_07 = self.language_dict["Report File"][self.var_language]
        str_lbl_08 = self.language_dict["Plotting Setup"][self.var_language]
        str_lbl_09 = self.language_dict["Language Selection"][self.var_language]
        str_lbl_10 = self.language_dict["Data Processing"][self.var_language]
        str_lbl_11 = self.language_dict["Dwell times"][self.var_language]
        str_lbl_12 = self.language_dict["Default value"][self.var_language]
        str_lbl_13 = self.language_dict["Application launcher"][self.var_language]
        str_lbl_14 = self.language_dict["PySILLS color scheme"][self.var_language]
        str_lbl_15 = self.language_dict["Default author"][self.var_language]
        str_lbl_16 = self.language_dict["Default SRM"][self.var_language]
        str_lbl_17 = self.language_dict["Calculation formula from ..."][self.var_language]
        str_lbl_18 = self.language_dict["Background"][self.var_language]
        str_lbl_19 = self.language_dict["Main signal"][self.var_language]
        str_lbl_20 = self.language_dict["File type"][self.var_language]
        str_lbl_21 = self.language_dict["Delimiter"][self.var_language]
        str_lbl_22 = self.language_dict["Line width"][self.var_language]
        str_lbl_23 = self.language_dict["Save settings"][self.var_language]
        str_lbl_24 = self.language_dict["Create launcher"][self.var_language]
        str_lbl_25 = self.language_dict["Arithmetic mean"][self.var_language]
        str_lbl_26 = self.language_dict["Average of every interval"][self.var_language]
        str_lbl_27 = self.language_dict["Stack all intervals into one"][self.var_language]
        str_lbl_28 = self.language_dict["Median"][self.var_language]
        str_lbl_29 = self.language_dict["Screen resolution"][self.var_language]
        str_lbl_30 = self.language_dict["Python path"][self.var_language]
        str_lbl_31 = self.language_dict["PySILLS path"][self.var_language]
        str_lbl_32 = self.language_dict["Default ICP-MS"][self.var_language]
        str_lbl_33 = self.language_dict["Project file"][self.var_language]

        lbl_01 = SE(
            parent=subwindow_generalsettings, row_id=2, column_id=start_column, n_rows=1, n_columns=10,
            fg=font_color_light, bg=background_color_dark).create_simple_label(
            text=str_lbl_01, relief=tk.FLAT, fontsize=font_element)
        lbl_02 = SE(
            parent=subwindow_generalsettings, row_id=5, column_id=start_column, n_rows=2, n_columns=10,
            fg=font_color_light, bg=background_color_dark).create_simple_label(
            text="Standard Reference\n Material (SRM)", relief=tk.FLAT, fontsize=font_element)
        lbl_13 = SE(
            parent=subwindow_generalsettings, row_id=0, column_id=25, n_rows=1, n_columns=9,
            fg=font_color_light, bg=background_color_dark).create_simple_label(
            text=str_lbl_13, relief=tk.FLAT, fontsize=font_element)
        lbl_14 = SE(
            parent=subwindow_generalsettings, row_id=2, column_id=25, n_rows=1, n_columns=9,
            fg=font_color_light, bg=background_color_dark).create_simple_label(
            text=str_lbl_14, relief=tk.FLAT, fontsize=font_element)
        lbl_06 = SE(
            parent=subwindow_generalsettings, row_id=0, column_id=11, n_rows=1, n_columns=13,
            fg=font_color_light, bg=background_color_dark).create_simple_label(
            text=str_lbl_04, relief=tk.FLAT, fontsize=font_element)
        lbl_07 = SE(
            parent=subwindow_generalsettings, row_id=5, column_id=11, n_rows=2, n_columns=13,
            fg=font_color_light, bg=background_color_dark).create_simple_label(
            text=str_lbl_05, relief=tk.FLAT, fontsize=font_element)
        lbl_08 = SE(
            parent=subwindow_generalsettings, row_id=9, column_id=start_column, n_rows=1, n_columns=10,
            fg=font_color_light, bg=background_color_dark).create_simple_label(
            text=str_lbl_06, relief=tk.FLAT, fontsize=font_element)
        lbl_09 = SE(
            parent=subwindow_generalsettings, row_id=18, column_id=start_column, n_rows=1, n_columns=10,
            fg=font_color_light, bg=background_color_dark).create_simple_label(
            text=str_lbl_07, relief=tk.FLAT, fontsize=font_element)
        lbl_10 = SE(
            parent=subwindow_generalsettings, row_id=11, column_id=start_column, n_rows=1, n_columns=10,
            fg=font_color_light, bg=background_color_dark).create_simple_label(
            text=str_lbl_08, relief=tk.FLAT, fontsize=font_element)
        lbl_11 = SE(
            parent=subwindow_generalsettings, row_id=0, column_id=start_column, n_rows=1, n_columns=10,
            fg=font_color_light, bg=background_color_dark).create_simple_label(
            text=str_lbl_09, relief=tk.FLAT, fontsize=font_element)
        lbl_11 = SE(
            parent=subwindow_generalsettings, row_id=13, column_id=start_column, n_rows=1, n_columns=10,
            fg=font_color_light, bg=background_color_dark).create_simple_label(
            text=str_lbl_10, relief=tk.FLAT, fontsize=font_element)
        lbl_11 = SE(
            parent=subwindow_generalsettings, row_id=9, column_id=11, n_rows=1, n_columns=13,
            fg=font_color_light, bg=background_color_dark).create_simple_label(
            text=str_lbl_11, relief=tk.FLAT, fontsize=font_element)
        lbl_15 = SE(
            parent=subwindow_generalsettings, row_id=4, column_id=25, n_rows=1, n_columns=9,
            fg=font_color_light, bg=background_color_dark).create_simple_label(
            text=str_lbl_29, relief=tk.FLAT, fontsize=font_element)
        lbl_30 = SE(
            parent=subwindow_generalsettings, row_id=11, column_id=11, n_rows=1, n_columns=13,
            fg=font_color_light, bg=background_color_dark).create_simple_label(
            text=str_lbl_30, relief=tk.FLAT, fontsize=font_element)
        lbl_31 = SE(
            parent=subwindow_generalsettings, row_id=13, column_id=11, n_rows=1, n_columns=13,
            fg=font_color_light, bg=background_color_dark).create_simple_label(
            text=str_lbl_31, relief=tk.FLAT, fontsize=font_element)
        lbl_09 = SE(
            parent=subwindow_generalsettings, row_id=18, column_id=11, n_rows=1, n_columns=13,
            fg=font_color_light, bg=background_color_dark).create_simple_label(
            text=str_lbl_32, relief=tk.FLAT, fontsize=font_element)
        lbl_33 = SE(
            parent=subwindow_generalsettings, row_id=21, column_id=start_column, n_rows=1, n_columns=5,
            fg=font_color_dark, bg=background_color_elements).create_simple_label(
            text=str_lbl_33, relief=tk.FLAT, fontsize=font_element)

        self.gui_elements["general_settings"]["Label"]["General"].extend(
            [lbl_01, lbl_02, lbl_06, lbl_07, lbl_08, lbl_09, lbl_10, lbl_11])

        lbl_11a = SE(
            parent=subwindow_generalsettings, row_id=10, column_id=11, n_rows=1, n_columns=7,
            fg=font_color_dark, bg=background_color_elements).create_simple_label(
            text=str_lbl_12, relief=tk.FLAT, fontsize=font_element)
        lbl_01a = SE(
            parent=subwindow_generalsettings, row_id=3, column_id=start_column, n_rows=1, n_columns=10,
            fg=font_color_dark, bg=background_color_elements).create_simple_label(
            text=str_lbl_15, relief=tk.FLAT, fontsize=font_element)
        lbl_02a = SE(
            parent=subwindow_generalsettings, row_id=7, column_id=start_column, n_rows=1, n_columns=10,
            fg=font_color_dark, bg=background_color_elements).create_simple_label(
            text=str_lbl_16, relief=tk.FLAT, fontsize=font_element)
        lbl_06a = SE(
            parent=subwindow_generalsettings, row_id=1, column_id=11, n_rows=1, n_columns=13,
            fg=font_color_dark, bg=background_color_elements).create_simple_label(
            text=str_lbl_17, relief=tk.FLAT, fontsize=font_element, anchor=tk.W)
        lbl_07a = SE(
            parent=subwindow_generalsettings, row_id=7, column_id=11, n_rows=1, n_columns=7,
            fg=font_color_dark, bg=background_color_elements).create_simple_label(
            text=str_lbl_18, relief=tk.FLAT, fontsize=font_element)
        lbl_07b = SE(
            parent=subwindow_generalsettings, row_id=8, column_id=11, n_rows=1, n_columns=7,
            fg=font_color_dark, bg=background_color_elements).create_simple_label(
            text=str_lbl_19, relief=tk.FLAT, fontsize=font_element)
        lbl_09a = SE(
            parent=subwindow_generalsettings, row_id=19, column_id=start_column, n_rows=1, n_columns=5,
            fg=font_color_dark, bg=background_color_elements).create_simple_label(
            text=str_lbl_20, relief=tk.FLAT, fontsize=font_element)
        lbl_09b = SE(
            parent=subwindow_generalsettings, row_id=20, column_id=start_column, n_rows=1, n_columns=5,
            fg=font_color_dark, bg=background_color_elements).create_simple_label(
            text=str_lbl_21, relief=tk.FLAT, fontsize=font_element)
        lbl_10a = SE(
            parent=subwindow_generalsettings, row_id=12, column_id=start_column, n_rows=1, n_columns=5,
            fg=font_color_dark, bg=background_color_elements).create_simple_label(
            text=str_lbl_22, relief=tk.FLAT, fontsize=font_element)

        self.gui_elements["general_settings"]["Label"]["General"].extend(
            [lbl_01a, lbl_02a, lbl_06a, lbl_07a, lbl_07b, lbl_09a, lbl_09b, lbl_10a])

        ## Entries
        entr_11a = SE(
            parent=subwindow_generalsettings, row_id=10, column_id=18, n_rows=1, n_columns=6,
            fg=font_color_dark, bg=self.bg_colors["White"]).create_simple_entry(
            var=self.container_var["dwell_times"]["Entry"]["Default"],
            text_default=self.container_var["dwell_times"]["Entry"]["Default"].get(), command=None)
        entr_01a = SE(
            parent=subwindow_generalsettings, row_id=4, column_id=start_column, n_rows=1, n_columns=10,
            fg=font_color_dark, bg=self.bg_colors["White"]).create_simple_entry(
            var=self.container_var["General Settings"]["Default Author"],
            text_default=self.container_var["General Settings"]["Default Author"].get(), command=None)
        entr_07a = SE(
            parent=subwindow_generalsettings, row_id=7, column_id=18, n_rows=1, n_columns=3,
            fg=font_color_dark, bg=self.bg_colors["White"]).create_simple_entry(
            var=self.container_var["General Settings"]["BG Offset Start"],
            text_default=self.container_var["General Settings"]["BG Offset Start"].get(), command=None)
        entr_07b = SE(
            parent=subwindow_generalsettings, row_id=7, column_id=21, n_rows=1, n_columns=3,
            fg=font_color_dark, bg=self.bg_colors["White"]).create_simple_entry(
            var=self.container_var["General Settings"]["BG Offset End"],
            text_default=self.container_var["General Settings"]["BG Offset End"].get(), command=None)
        entr_07c = SE(
            parent=subwindow_generalsettings, row_id=8, column_id=18, n_rows=1, n_columns=3,
            fg=font_color_dark, bg=self.bg_colors["White"]).create_simple_entry(
            var=self.container_var["General Settings"]["MAT Offset Start"],
            text_default=self.container_var["General Settings"]["MAT Offset Start"].get(), command=None)
        entr_07d = SE(
            parent=subwindow_generalsettings, row_id=8, column_id=21, n_rows=1, n_columns=3,
            fg=font_color_dark, bg=self.bg_colors["White"]).create_simple_entry(
            var=self.container_var["General Settings"]["MAT Offset End"],
            text_default=self.container_var["General Settings"]["MAT Offset End"].get(), command=None)
        entr_10a = SE(
            parent=subwindow_generalsettings, row_id=12, column_id=5, n_rows=1, n_columns=5,
            fg=font_color_dark, bg=self.bg_colors["White"]).create_simple_entry(
            var=self.container_var["General Settings"]["Line width"],
            text_default=self.container_var["General Settings"]["Line width"].get(),
            command=lambda event, var_entr=self.container_var["General Settings"]["Line width"]:
            self.check_lw_value(var_entr, event))
        entr_30 = SE(
            parent=subwindow_generalsettings, row_id=12, column_id=11, n_rows=1, n_columns=13,
            fg=font_color_dark, bg=self.bg_colors["White"]).create_simple_entry(
            var=self.container_var["Python path"], text_default=self.container_var["Python path"].get(), command=None)
        entr_31 = SE(
            parent=subwindow_generalsettings, row_id=14, column_id=11, n_rows=1, n_columns=13,
            fg=font_color_dark, bg=self.bg_colors["White"]).create_simple_entry(
            var=self.container_var["PySILLS path"], text_default=self.container_var["PySILLS path"].get(), command=None)
        entr_32 = SE(
            parent=subwindow_generalsettings, row_id=15, column_id=11, n_rows=1, n_columns=13,
            fg=font_color_dark, bg=self.bg_colors["White"]).create_simple_entry(
            var=self.container_var["PySILLS script"], text_default=self.container_var["PySILLS script"].get(),
            command=None)

        entr_11a.configure(font=font_element)
        entr_01a.configure(font=font_element)
        entr_07a.configure(font=font_element)
        entr_07b.configure(font=font_element)
        entr_07c.configure(font=font_element)
        entr_07d.configure(font=font_element)
        entr_10a.configure(font=font_element)
        entr_30.configure(font=font_element)
        entr_31.configure(font=font_element)
        entr_32.configure(font=font_element)

        self.gui_elements["general_settings"]["Entry"]["General"].extend(
            [entr_01a, entr_07a, entr_07b, entr_07c, entr_07d, entr_10a])

        ## Option Menus
        list_srm = list(np.sort(self.list_srm))
        list_srm.insert(0, "Select SRM")
        list_colormaps = [
            "turbo", "rainbow", "gist_rainbow", "jet", "nipy_spectral", "gnuplot", "gist_earth", "ocean", "hsv",
            "seismic", "coolwarm", "Spectral", "copper", "hot", "cool", "viridis", "plasma", "inferno", "magma",
            "cividis", "brg"]
        list_languages = ["English", "German", "Spanish", "Italian", "French", "Chinese", "Greek", "Russian"]
        list_colorschemes = [
            "Dark scheme", "Light scheme", "Boho theme 1", "Synthwave theme", "Gunmetal theme",
            "Dark Jungle", "Noble Room"]
        list_screenresolutions = ["3840x2160", "1920x1080", "1280x720"]
        list_colormaps.sort()
        list_filetypes = ["*.csv", "*.txt"]
        list_filetypes.sort()
        list_delimiter = ["comma", "semicolon"]
        list_delimiter.sort()
        list_languages.sort()
        list_projecttypes = ["*.proj", "*.csv"]
        list_projecttypes.sort()

        opt_srm = SE(
            parent=subwindow_generalsettings, row_id=8, column_id=start_column, n_rows=1, n_columns=10,
            fg=font_color_dark, bg=background_color_elements).create_simple_optionmenu(
            var_opt=self.container_var["General Settings"]["Default SRM"],
            var_default=self.container_var["General Settings"]["Default SRM"].get(), var_list=list_srm,
            fg_active=font_color_light, bg_active=accent_color)
        opt_colormaps = SE(
            parent=subwindow_generalsettings, row_id=10, column_id=start_column, n_rows=1, n_columns=10,
            fg=font_color_dark, bg=background_color_elements).create_simple_optionmenu(
            var_opt=self.container_var["General Settings"]["Colormap"],
            var_default=self.container_var["General Settings"]["Colormap"].get(), var_list=list_colormaps,
            fg_active=font_color_light, bg_active=accent_color,
            command=lambda var_opt=self.container_var["General Settings"]["Colormap"]: self.change_colormap(var_opt))
        opt_filetype = SE(
            parent=subwindow_generalsettings, row_id=19, column_id=5, n_rows=1, n_columns=5,
            fg=font_color_dark, bg=background_color_elements).create_simple_optionmenu(
            var_opt=self.container_var["General Settings"]["File type"],
            var_default=self.container_var["General Settings"]["File type"].get(), var_list=list_filetypes,
            fg_active=font_color_light, bg_active=accent_color,
            command=lambda var_opt=self.container_var["General Settings"]["File type"]: self.change_filetype(var_opt))
        opt_delimiter = SE(
            parent=subwindow_generalsettings, row_id=20, column_id=5, n_rows=1, n_columns=5,
            fg=font_color_dark, bg=background_color_elements).create_simple_optionmenu(
            var_opt=self.container_var["General Settings"]["Delimiter"],
            var_default=self.container_var["General Settings"]["Delimiter"].get(), var_list=list_delimiter,
            fg_active=font_color_light, bg_active=accent_color,
            command=lambda var_opt=self.container_var["General Settings"]["Delimiter"]: self.change_delimiter(var_opt))
        opt_language = SE(
            parent=subwindow_generalsettings, row_id=1, column_id=start_column, n_rows=1, n_columns=10,
            fg=font_color_dark, bg=background_color_elements).create_simple_optionmenu(
            var_opt=self.container_var["General Settings"]["Language"],
            var_default=self.container_var["General Settings"]["Language"].get(), var_list=list_languages,
            fg_active=font_color_light, bg_active=accent_color)
        opt_icpms_default = SE(
            parent=subwindow_generalsettings, row_id=19, column_id=11, n_rows=1, n_columns=13,
            fg=font_color_dark, bg=background_color_elements).create_simple_optionmenu(
            var_opt=self.var_opt_icp, var_default=self.var_opt_icp.get(),
            var_list=self.container_lists["ICPMS Library"], fg_active=font_color_light, bg_active=accent_color)
        opt_projecttype = SE(
            parent=subwindow_generalsettings, row_id=21, column_id=5, n_rows=1, n_columns=5,
            fg=font_color_dark, bg=background_color_elements).create_simple_optionmenu(
            var_opt=self.container_var["General Settings"]["Project type"],
            var_default=self.container_var["General Settings"]["Project type"].get(), var_list=list_projecttypes,
            fg_active=font_color_light, bg_active=accent_color)

        # opt_language["menu"].entryconfig("German", state="disable")
        opt_language["menu"].entryconfig("Italian", state="disable")
        opt_language["menu"].entryconfig("Spanish", state="disable")
        opt_language["menu"].entryconfig("French", state="disable")
        #opt_language["menu"].entryconfig("Chinese", state="disable")
        opt_language["menu"].entryconfig("Greek", state="disable")
        opt_language["menu"].entryconfig("Russian", state="disable")

        opt_srm.configure(font=font_option)
        opt_colormaps.configure(font=font_option)
        opt_filetype.configure(font=font_option)
        opt_delimiter.configure(font=font_option)
        opt_language.configure(font=font_option)

        opt_colorscheme = SE(
            parent=subwindow_generalsettings, row_id=3, column_id=25, n_rows=1, n_columns=9,
            fg=font_color_dark, bg=background_color_elements).create_simple_optionmenu(
            var_opt=self.container_var["General Settings"]["Color scheme"],
            var_default=self.container_var["General Settings"]["Color scheme"].get(), var_list=list_colorschemes,
            fg_active=font_color_light, bg_active=accent_color)
        opt_screenresolution = SE(
            parent=subwindow_generalsettings, row_id=5, column_id=25, n_rows=1, n_columns=9,
            fg=font_color_dark, bg=background_color_elements).create_simple_optionmenu(
            var_opt=self.container_var["General Settings"]["Screen resolution"],
            var_default=self.container_var["General Settings"]["Screen resolution"].get(),
            var_list=list_screenresolutions, fg_active=font_color_light, bg_active=accent_color)

        opt_colorscheme.configure(font=font_option)
        opt_screenresolution.configure(font=font_option)

        self.gui_elements["general_settings"]["Option Menu"]["General"].extend(
            [opt_srm, opt_colormaps, opt_filetype, opt_delimiter, opt_language])

        ## Radiobuttons
        rb_06a = SE(
            parent=subwindow_generalsettings, row_id=2, column_id=11, n_rows=1, n_columns=13,
            fg=font_color_dark, bg=background_color_elements).create_radiobutton(
            var_rb=self.container_var["General Settings"]["LOD Selection"], value_rb=0,
            color_bg=background_color_elements, fg=font_color_dark, text="Pettke et al. (2012)",
            sticky="nesw", relief=tk.FLAT, font="sans 10 bold")
        rb_06b = SE(
            parent=subwindow_generalsettings, row_id=3, column_id=11, n_rows=1, n_columns=13,
            fg=font_color_dark, bg=background_color_elements).create_radiobutton(
            var_rb=self.container_var["General Settings"]["LOD Selection"], value_rb=1,
            color_bg=background_color_elements, fg=font_color_dark, text="Longerich et al. (1996)",
            sticky="nesw", relief=tk.FLAT, font="sans 10 bold")

        rb_06c = SE(
            parent=subwindow_generalsettings, row_id=14, column_id=start_column, n_rows=1, n_columns=10,
            fg=font_color_dark, bg=background_color_elements).create_radiobutton(
            var_rb=self.container_var["General Settings"]["Desired Average"], value_rb=1,
            color_bg=background_color_elements, fg=font_color_dark, text=str_lbl_25,
            sticky="nesw", relief=tk.FLAT, font="sans 10 bold")
        rb_06d = SE(
            parent=subwindow_generalsettings, row_id=15, column_id=start_column, n_rows=1, n_columns=10,
            fg=font_color_dark, bg=background_color_elements).create_radiobutton(
            var_rb=self.container_var["General Settings"]["Desired Average"], value_rb=2,
            color_bg=background_color_elements, fg=font_color_dark, text=str_lbl_28,
            sticky="nesw", relief=tk.FLAT, font="sans 10 bold")

        rb_07a = SE(
            parent=subwindow_generalsettings, row_id=16, column_id=start_column, n_rows=1, n_columns=10,
            fg=font_color_dark, bg=background_color_elements).create_radiobutton(
            var_rb=self.container_var["General Settings"]["Interval Processing"], value_rb=1,
            color_bg=background_color_elements, fg=font_color_dark, text=str_lbl_26,
            sticky="nesw", relief=tk.FLAT, font="sans 10 bold")
        rb_07b = SE(
            parent=subwindow_generalsettings, row_id=17, column_id=start_column, n_rows=1, n_columns=10,
            fg=font_color_dark, bg=background_color_elements).create_radiobutton(
            var_rb=self.container_var["General Settings"]["Interval Processing"], value_rb=2,
            color_bg=background_color_elements, fg=font_color_dark, text=str_lbl_27,
            sticky="nesw", relief=tk.FLAT, font="sans 10 bold")
        rb_07a.configure(state="disabled")
        rb_07b.configure(state="disabled")
        self.gui_elements["general_settings"]["Radiobutton"]["General"].extend(
            [rb_06a, rb_06b])

        rb_06a.configure(font=font_element)
        rb_06b.configure(font=font_element)
        rb_06c.configure(font=font_element)
        rb_06d.configure(font=font_element)
        rb_07a.configure(font=font_element)
        rb_07b.configure(font=font_element)

        ## Buttons
        if self.var_os == "darwin":
            font_color_accent = font_color_dark
        else:
            font_color_accent = font_color_light

        btn_01 = SE(
            parent=subwindow_generalsettings, row_id=19, column_id=25, n_rows=2, n_columns=9,
            fg=font_color_accent, bg=accent_color).create_simple_button(
            text=str_lbl_23, bg_active=accent_color, fg_active=font_color_accent,
            command=self.confirm_general_settings)
        btn_13 = SE(
            parent=subwindow_generalsettings, row_id=1, column_id=25, n_rows=1, n_columns=9,
            fg=font_color_dark, bg=background_color_elements).create_simple_button(
            text=str_lbl_24, bg_active=accent_color, fg_active=font_color_accent,
            command=self.create_starter_file)

        btn_01.configure(font=font_element)
        btn_13.configure(font=font_element)

        self.gui_elements["general_settings"]["Button"]["General"].extend([btn_01])

        self.check_lw_value(var_entr=self.container_var["General Settings"]["Line width"], event="")

        # INITIALIZATION
        self.get_current_path_python()
        self.get_current_path_pysills_folder()
        self.get_current_path_pysills_script()

    def check_lw_value(self, var_entr, event):
        if float(var_entr.get()) < 0:
            var_entr.set("0.5")
        elif float(var_entr.get()) > 2:
            var_entr.set("2.0")

    def change_colormap(self, var_opt):
        self.container_var["General Settings"]["Colormap"].set(var_opt)

    def change_filetype(self, var_opt):
        self.container_var["General Settings"]["File type"].set(var_opt)

    def change_delimiter(self, var_opt):
        self.container_var["General Settings"]["Delimiter"].set(var_opt)

    def confirm_general_settings(self):
        path_pysills = os.path.dirname(os.path.realpath(__file__))
        path_pysills = self.path_pysills
        filename = os.path.join(self.path_pysills_main, "user_settings.txt")
        with open(filename, "w") as file_settings:
            file_settings.write("GENERAL SETTINGS" + ";\n")
            for key, value in self.container_var["General Settings"].items():
                str_key = str(key) + ";" + str(value.get()) + ";\n"
                file_settings.write(str_key)
            str_key = str("Dwell Times") + ";" + str(self.container_var["dwell_times"]["Entry"]["Default"].get()) + ";\n"
            file_settings.write(str_key)
            str_key = str("Default ICP-MS") + ";" + str(self.var_opt_icp.get()) + ";\n"
            file_settings.write(str_key)

    def check_srm_settings(self):
        # Colors
        font_color_dark = self.bg_colors["Dark Font"]
        font_color_light = self.bg_colors["Light Font"]
        background_color_dark = self.bg_colors["BG Window"]
        background_color_elements = self.bg_colors["Light"]
        background_color_light = self.bg_colors["Very Light"]
        accent_color = self.bg_colors["Accent"]  # self.accent_color
        font_header = self.font_settings["Header"]
        font_elements = self.font_settings["Elements"]
        font_option = self.font_settings["Options"]
        font_table = self.font_settings["Table"]

        str_title = self.language_dict["Check-Up - Standard Reference Material"][self.var_language]
        str_lbl_01 = self.language_dict["Standard Files"][self.var_language]
        str_lbl_02 = self.language_dict["Isotopes"][self.var_language]
        str_lbl_03 = self.language_dict["SRM values"][self.var_language]
        str_lbl_04 = self.language_dict["Name"][self.var_language]
        str_lbl_05 = self.language_dict["SRM"][self.var_language]
        str_lbl_06 = self.language_dict["Concentration (ppm)"][self.var_language]
        str_lbl_07 = self.language_dict["Element"][self.var_language]

        ## Window Settings
        row_min = self.row_height
        column_min = self.column_width
        n_rows = self.window_dimensions["Check-up SRM"][0]
        n_columns = self.window_dimensions["Check-up SRM"][1]

        window_width = int(n_columns*self.column_width)
        window_height = int(n_rows*self.row_height)

        var_geometry = str(window_width) + "x" + str(window_height) + "+" + str(0) + "+" + str(0)

        subwindow_srm_checkup = tk.Toplevel(self.parent)
        subwindow_srm_checkup.title(str_title)
        subwindow_srm_checkup.geometry(var_geometry)
        subwindow_srm_checkup.resizable(False, False)
        subwindow_srm_checkup["bg"] = background_color_dark

        for x in range(n_columns):
            tk.Grid.columnconfigure(subwindow_srm_checkup, x, weight=1)
        for y in range(n_rows):
            tk.Grid.rowconfigure(subwindow_srm_checkup, y, weight=1)

        # Rows
        for i in range(0, n_rows):
            subwindow_srm_checkup.grid_rowconfigure(i, minsize=row_min)
        # Columns
        for i in range(0, n_columns):
            subwindow_srm_checkup.grid_columnconfigure(i, minsize=column_min)

        ##########################################################

        start_column = 0
        start_row = 0

        ## LABELS
        lbl_std = SE(
            parent=subwindow_srm_checkup, row_id=start_row, column_id=start_column, n_rows=1, n_columns=17,
            fg=font_color_light, bg=background_color_dark).create_simple_label(
            text=str_lbl_01, relief=tk.FLAT, fontsize="sans 10 bold")
        lbl_smpl = SE(
            parent=subwindow_srm_checkup, row_id=start_row + 11, column_id=start_column, n_rows=1, n_columns=17,
            fg=font_color_light, bg=background_color_dark).create_simple_label(
            text=str_lbl_02, relief=tk.FLAT, fontsize="sans 10 bold")
        lbl_smpl = SE(
            parent=subwindow_srm_checkup, row_id=start_row, column_id=start_column + 18, n_rows=1, n_columns=13,
            fg=font_color_light, bg=background_color_dark).create_simple_label(
            text=str_lbl_03, relief=tk.FLAT, fontsize="sans 10 bold")

        ## TREEVIEW
        tv_std = SE(
            parent=subwindow_srm_checkup, row_id=start_row + 1, column_id=start_column, n_rows=10, n_columns=17,
            fg=font_color_dark, bg=background_color_light).create_treeview(
            n_categories=2, text_n=[str_lbl_04, str_lbl_05], width_n=["120", "220"], individual=True)

        list_srm = ["Select SRM"]
        for index, var_file in enumerate(self.container_lists["STD"]["Long"]):
            parts = var_file.split("/")
            file_std = parts[-1]
            var_srm_file_i = self.container_var["STD"][var_file]["SRM"].get()
            #
            if var_srm_file_i not in list_srm:
                list_srm.append(var_srm_file_i)
            #
            entry_std = [file_std, var_srm_file_i]
            #
            tv_std.insert("", tk.END, values=entry_std)

        if len(list_srm) > 1:
            list_srm.remove("Select SRM")

        tv_isotope = SE(
            parent=subwindow_srm_checkup, row_id=start_row + 12, column_id=start_column, n_rows=10, n_columns=17,
            fg=font_color_dark, bg=background_color_light).create_treeview(
            n_categories=3, text_n=[str_lbl_04, str_lbl_05, str_lbl_06], width_n=["70", "130", "140"],
            individual=True)

        for index, isotope in enumerate(self.container_lists["ISOTOPES"]):
            var_srm_i = self.container_var["SRM"][isotope].get()

            key_element = re.search(r"(\D+)(\d+)", isotope)
            element = key_element.group(1)
            if element in self.srm_actual[var_srm_i]:
                var_isotope_concentration = self.srm_actual[var_srm_i][element]
            else:
                var_isotope_concentration = 0.0

            entry_isotope = [isotope, var_srm_i, var_isotope_concentration]

            tv_isotope.insert("", tk.END, values=entry_isotope)

        tv_srm = SE(
            parent=subwindow_srm_checkup, row_id=start_row + 2, column_id=start_column + 18, n_rows=20, n_columns=13,
            fg=font_color_dark, bg=background_color_light).create_treeview(
            n_categories=2, text_n=[str_lbl_07, str_lbl_06], width_n=["100", "160"], individual=True)

        ## Option Menus
        var_opt_srm = tk.StringVar()
        if self.container_var["SRM"]["default"][0].get() != "Select SRM":
            var_opt_srm.set(self.container_var["SRM"]["default"][0].get())
        else:
            var_opt_srm.set("Select SRM")

        opt_srm = SE(
            parent=subwindow_srm_checkup, row_id=start_row + 1, column_id=start_column + 18, n_rows=1, n_columns=13,
            fg=font_color_dark, bg=background_color_elements).create_simple_optionmenu(
            var_opt=var_opt_srm, var_default=var_opt_srm.get(), var_list=list_srm,
            fg_active=font_color_light, bg_active=accent_color,
            command=lambda var_opt=var_opt_srm, var_tv=tv_srm: self.change_srm_checkup(var_opt, var_tv))

        ## INITIALIZATION
        self.change_srm_checkup(var_opt=var_opt_srm.get(), var_tv=tv_srm)

    def change_srm_checkup(self, var_opt, var_tv):
        if len(var_tv.get_children()) > 0:
            for item in var_tv.get_children():
                var_tv.delete(item)

        if var_opt != "Select SRM":
            if var_opt in self.srm_actual:
                for element in np.sort(list(self.srm_actual[var_opt].keys())):
                    entry_isotope = [element, self.srm_actual[var_opt][element]]
                    var_tv.insert("", tk.END, values=entry_isotope)
            else:
                for str_srm in np.sort(list(self.srm_actual.keys())):
                    for element in np.sort(list(self.srm_actual[str_srm].keys())):
                        entry_isotope = [element, self.srm_actual[var_opt][element]]
                        var_tv.insert("", tk.END, values=entry_isotope)

    def check_interval_settings(self):
        # Colors
        font_color_dark = self.bg_colors["Dark Font"]
        font_color_light = self.bg_colors["Light Font"]
        background_color_dark = self.bg_colors["BG Window"]
        background_color_elements = self.bg_colors["Light"]
        background_color_light = self.bg_colors["Very Light"]
        accent_color = self.bg_colors["Accent"]  # self.accent_color
        font_header = self.font_settings["Header"]
        font_elements = self.font_settings["Elements"]
        font_option = self.font_settings["Options"]
        font_table = self.font_settings["Table"]

        str_lbl_01 = self.language_dict["Check-Up"][self.var_language]
        str_lbl_02 = self.language_dict["Calculation Intervals"][self.var_language]

        ## Window Settings
        row_min = self.row_height
        column_min = self.column_width
        n_rows = self.window_dimensions["Check-up intervals"][0]
        n_columns = self.window_dimensions["Check-up intervals"][1]

        window_width = int(n_columns*self.column_width)
        window_height = int(n_rows*self.row_height)

        var_geometry = str(window_width) + "x" + str(window_height) + "+" + str(0) + "+" + str(0)

        subwindow_intervals = tk.Toplevel(self.parent)
        subwindow_intervals.title(str_lbl_01 + " - " + str_lbl_02)
        subwindow_intervals.geometry(var_geometry)
        subwindow_intervals.resizable(False, False)
        subwindow_intervals["bg"] = background_color_dark

        for x in range(n_columns):
            tk.Grid.columnconfigure(subwindow_intervals, x, weight=1)
        for y in range(n_rows):
            tk.Grid.rowconfigure(subwindow_intervals, y, weight=1)

        # Rows
        for i in range(0, n_rows):
            subwindow_intervals.grid_rowconfigure(i, minsize=row_min)
        # Columns
        for i in range(0, n_columns):
            subwindow_intervals.grid_columnconfigure(i, minsize=column_min)

        str_lbl_03 = self.language_dict["Name"][self.var_language]
        str_lbl_04 = self.language_dict["Category"][self.var_language]
        str_lbl_05 = self.language_dict["ID"][self.var_language]
        str_lbl_06 = self.language_dict["Start"][self.var_language]
        str_lbl_07 = self.language_dict["End"][self.var_language]

        if self.pysills_mode == "MA":
            var_categories = [str_lbl_03, str_lbl_04, str_lbl_05, str_lbl_06, str_lbl_07]
            var_widths = ["160", "90", "90", "90", "90"]
        elif self.pysills_mode in ["FI", "INCL"]:
            var_categories = [str_lbl_03, str_lbl_04, str_lbl_05, str_lbl_06, str_lbl_07]
            var_widths = ["160", "90", "90", "90", "90"]
        elif self.pysills_mode == "MI":
            var_categories = [str_lbl_03, str_lbl_04, str_lbl_05, str_lbl_06, str_lbl_07]
            var_widths = ["160", "90", "90", "90", "90"]

        ################################################################################################################

        start_column = 0
        start_row = 0

        ## FRAMES
        frm_00 = SE(
            parent=subwindow_intervals, row_id=start_row, column_id=start_column + 26,
            n_rows=n_rows, n_columns=n_columns - 26, fg=font_color_dark,
            bg=background_color_light).create_frame(relief=tk.FLAT)

        ## FIGURES
        fig_checkup_std = Figure(figsize=(10, 5), tight_layout=True, facecolor=background_color_light)
        fig_checkup_smpl = Figure(figsize=(10, 5), tight_layout=True, facecolor=background_color_light)

        canvas_std = FigureCanvasTkAgg(fig_checkup_std, master=subwindow_intervals)
        canvas_std.get_tk_widget().grid(
            row=start_row + 1, column=start_column + 26, rowspan=15, columnspan=27, sticky="nesw")
        canvas_smpl = FigureCanvasTkAgg(fig_checkup_smpl, master=subwindow_intervals)
        canvas_smpl.get_tk_widget().grid(
            row=start_row + 17, column=start_column + 26, rowspan=15, columnspan=27, sticky="nesw")

        self.container_checkup["STD"].append(canvas_std)
        self.container_checkup["SMPL"].append(canvas_smpl)
        self.temp_figure_checkup = {"STD": fig_checkup_std, "SMPL": fig_checkup_smpl}
        self.temp_canvas_checkup = {"STD": canvas_std, "SMPL": canvas_smpl}
        self.temp_axes_checkup = {"STD": {}, "SMPL": {}}
        self.temp_lines_checkup = {"STD": {}, "SMPL": {}}

        ## LABELS
        str_lbl_03 = self.language_dict["Standard Files"][self.var_language]
        str_lbl_04 = self.language_dict["Sample Files"][self.var_language]
        str_lbl_05 = self.language_dict["Before"][self.var_language]
        str_lbl_06 = self.language_dict["Next"][self.var_language]
        str_lbl_07 = self.language_dict["Setup"][self.var_language]

        lbl_std = SE(
            parent=subwindow_intervals, row_id=start_row, column_id=start_column, n_rows=1, n_columns=26,
            fg=font_color_light, bg=background_color_dark).create_simple_label(
            text=str_lbl_03, relief=tk.FLAT, fontsize="sans 10 bold")
        lbl_smpl = SE(
            parent=subwindow_intervals, row_id=start_row + 16, column_id=start_column, n_rows=1, n_columns=26,
            fg=font_color_light, bg=background_color_dark).create_simple_label(
            text=str_lbl_04, relief=tk.FLAT, fontsize="sans 10 bold")

        ## BUTTONS
        btn_std_01 = SE(
            parent=subwindow_intervals, row_id=start_row, column_id=start_column + 26, n_rows=1, n_columns=6,
            fg=font_color_dark, bg=background_color_elements).create_simple_button(
            text=str_lbl_05, bg_active=accent_color, fg_active=font_color_light,
            command=lambda filetype="STD", mode="Back", var_fig=fig_checkup_std:
            self.change_graph_checkup(filetype, mode, var_fig))
        btn_std_02 = SE(
            parent=subwindow_intervals, row_id=start_row, column_id=start_column + 32, n_rows=1, n_columns=6,
            fg=font_color_dark, bg=background_color_elements).create_simple_button(
            text=str_lbl_06, bg_active=accent_color, fg_active=font_color_light,
            command=lambda filetype="STD", var_fig=fig_checkup_std: self.change_graph_checkup(filetype, var_fig))

        if self.pysills_mode == "MA":
            btn_std_03 = SE(
                parent=subwindow_intervals, row_id=start_row, column_id=start_column + 38, n_rows=1, n_columns=6,
                fg=font_color_dark, bg=background_color_elements).create_simple_button(
                text=str_lbl_07, bg_active=accent_color, fg_active=font_color_light,
                command=lambda var_filename_long=self.ma_current_file_std, var_filetype="STD", checkup_mode=True:
                self.ma_check_specific_file(var_filename_long, var_filetype, checkup_mode))
        elif self.pysills_mode in ["FI", "INCL"]:
            btn_std_03 = SE(
                parent=subwindow_intervals, row_id=start_row, column_id=start_column + 38, n_rows=1, n_columns=6,
                fg=font_color_dark, bg=background_color_elements).create_simple_button(
                text=str_lbl_07, bg_active=accent_color, fg_active=font_color_light,
                command=lambda var_file=self.fi_current_file_std, var_type="STD", checkup_mode=True:
                self.fi_check_specific_file(var_file, var_type, checkup_mode))
        elif self.pysills_mode == "MI":
            btn_std_03 = SE(
                parent=subwindow_intervals, row_id=start_row, column_id=start_column + 38, n_rows=1, n_columns=6,
                fg=font_color_dark, bg=background_color_elements).create_simple_button(
                text=str_lbl_07, bg_active=accent_color, fg_active=font_color_light,
                command=lambda var_file=self.mi_current_file_std, var_type="STD", checkup_mode=True:
                self.mi_check_specific_file(var_file, var_type, checkup_mode))

        btn_smpl_01 = SE(
            parent=subwindow_intervals, row_id=start_row + 16, column_id=start_column + 26, n_rows=1, n_columns=6,
            fg=font_color_dark, bg=background_color_elements).create_simple_button(
            text=str_lbl_05, bg_active=accent_color, fg_active=font_color_light,
            command=lambda filetype="SMPL", mode="Back", var_fig=fig_checkup_smpl:
            self.change_graph_checkup(filetype, mode, var_fig))
        btn_smpl_02 = SE(
            parent=subwindow_intervals, row_id=start_row + 16, column_id=start_column + 32, n_rows=1, n_columns=6,
            fg=font_color_dark, bg=background_color_elements).create_simple_button(
            text=str_lbl_06, bg_active=accent_color, fg_active=font_color_light,
            command=lambda filetype="SMPL", var_fig=fig_checkup_smpl: self.change_graph_checkup(filetype, var_fig))
        if self.pysills_mode == "MA":
            btn_smpl_03 = SE(
                parent=subwindow_intervals, row_id=start_row + 16, column_id=start_column + 38, n_rows=1, n_columns=6,
                fg=font_color_dark, bg=background_color_elements).create_simple_button(
                text=str_lbl_07, bg_active=accent_color, fg_active=font_color_light,
                command=lambda var_filename_long=self.ma_current_file_smpl, var_filetype="SMPL", checkup_mode=True:
                self.ma_check_specific_file(var_filename_long, var_filetype, checkup_mode))
        elif self.pysills_mode in ["FI", "INCL"]:
            btn_smpl_03 = SE(
                parent=subwindow_intervals, row_id=start_row + 16, column_id=start_column + 38, n_rows=1, n_columns=6,
                fg=font_color_dark, bg=background_color_elements).create_simple_button(
                text=str_lbl_07, bg_active=accent_color, fg_active=font_color_light,
                command=lambda var_file=self.fi_current_file_smpl, var_type="SMPL", checkup_mode=True:
                self.fi_check_specific_file(var_file, var_type, checkup_mode))
        elif self.pysills_mode == "MI":
            btn_smpl_03 = SE(
                parent=subwindow_intervals, row_id=start_row + 16, column_id=start_column + 38, n_rows=1, n_columns=6,
                fg=font_color_dark, bg=background_color_elements).create_simple_button(
                text=str_lbl_07, bg_active=accent_color, fg_active=font_color_light,
                command=lambda var_file=self.mi_current_file_smpl, var_type="SMPL", checkup_mode=True:
                self.mi_check_specific_file(var_file, var_type, checkup_mode))

        ## TREEVIEW
        tv_std = SE(
            parent=subwindow_intervals, row_id=start_row + 1, column_id=start_column, n_rows=15, n_columns=26,
            fg=font_color_dark, bg=self.bg_colors["White"]).create_treeview(
            n_categories=len(var_categories), text_n=var_categories, width_n=var_widths, individual=True)

        if self.pysills_mode == "MA":
            for file_std in self.container_lists["STD"]["Short"]:
                for category in ["BG", "MAT"]:
                    if len(self.container_helper["STD"][file_std][category]["Content"]) > 0:
                        for var_id, var_content in self.container_helper["STD"][file_std][category]["Content"].items():
                            if category == "BG":
                                times = var_content["Times"]
                            else:
                                times = var_content["Times"]

                            entry_std = [file_std, category, var_id, times[0], times[1]]
                            tv_std.insert("", tk.END, values=entry_std)
                    else:
                        var_id = 0
                        times = ["undefined", "undefined"]

                        entry_std = [file_std, category, var_id, times[0], times[1]]
                        tv_std.insert("", tk.END, values=entry_std)

                self.temp_axes_checkup["STD"][file_std] = fig_checkup_std.add_subplot(label=file_std)
                self.temp_lines_checkup["STD"][file_std] = 0
        else:
            for file_std in self.container_lists["STD"]["Short"]:
                for category in ["BG", "MAT"]:
                    if len(self.container_helper["STD"][file_std][category]["Content"]) > 0:
                        for var_id, var_content in self.container_helper["STD"][file_std][category]["Content"].items():
                            if category == "BG":
                                times = var_content["Times"]
                            else:
                                times = var_content["Times"]

                            entry_std = [file_std, category, var_id, times[0], times[1]]
                            tv_std.insert("", tk.END, values=entry_std)
                    else:
                        var_id = 0
                        times = ["undefined", "undefined"]

                        entry_std = [file_std, category, var_id, times[0], times[1]]
                        tv_std.insert("", tk.END, values=entry_std)

                self.temp_axes_checkup["STD"][file_std] = fig_checkup_std.add_subplot(label=file_std)
                self.temp_lines_checkup["STD"][file_std] = 0

        tv_smpl = SE(
            parent=subwindow_intervals, row_id=start_row + 17, column_id=start_column, n_rows=15, n_columns=26,
            fg=font_color_dark, bg=self.bg_colors["White"]).create_treeview(
            n_categories=len(var_categories), text_n=var_categories, width_n=var_widths, individual=True)

        if self.pysills_mode == "MA":
            for file_smpl in self.container_lists["SMPL"]["Short"]:
                for category in ["BG", "MAT"]:
                    if len(self.container_helper["SMPL"][file_smpl][category]["Content"]) > 0:
                        for var_id, var_content in self.container_helper["SMPL"][file_smpl][category][
                            "Content"].items():
                            if category == "BG":
                                times = var_content["Times"]
                            else:
                                times = var_content["Times"]

                            entry_smpl = [file_smpl, category, var_id, times[0], times[1]]
                            tv_smpl.insert("", tk.END, values=entry_smpl)
                    else:
                        var_id = 0
                        times = ["undefined", "undefined"]

                        entry_smpl = [file_smpl, category, var_id, times[0], times[1]]
                        tv_smpl.insert("", tk.END, values=entry_smpl)

                self.temp_axes_checkup["SMPL"][file_smpl] = fig_checkup_smpl.add_subplot(label=file_smpl)
                self.temp_lines_checkup["SMPL"][file_smpl] = 0
        else:
            for file_smpl in self.container_lists["SMPL"]["Short"]:
                for category in ["BG", "MAT", "INCL"]:
                    if len(self.container_helper["SMPL"][file_smpl][category]["Content"]) > 0:
                        for var_id, var_content in self.container_helper["SMPL"][file_smpl][category][
                            "Content"].items():
                            if category == "BG":
                                times = var_content["Times"]
                            else:
                                times = var_content["Times"]

                            entry_smpl = [file_smpl, category, var_id, times[0], times[1]]
                            tv_smpl.insert("", tk.END, values=entry_smpl)
                    else:
                        var_id = 0
                        times = ["undefined", "undefined"]

                        entry_smpl = [file_smpl, category, var_id, times[0], times[1]]
                        tv_smpl.insert("", tk.END, values=entry_smpl)

                self.temp_axes_checkup["SMPL"][file_smpl] = fig_checkup_smpl.add_subplot(label=file_smpl)
                self.temp_lines_checkup["SMPL"][file_smpl] = 0

        ## INITIALIZATION
        self.change_graph_checkup(filetype="STD", initialization=True, var_fig=fig_checkup_std)
        self.change_graph_checkup(filetype="SMPL", initialization=True, var_fig=fig_checkup_smpl)

    def check_acquisition_times(self):
        # Colors
        font_color_dark = self.bg_colors["Dark Font"]
        font_color_light = self.bg_colors["Light Font"]
        background_color_dark = self.bg_colors["BG Window"]
        background_color_elements = self.bg_colors["Light"]
        background_color_light = self.bg_colors["Very Light"]
        accent_color = self.bg_colors["Accent"]  # self.accent_color
        font_header = self.font_settings["Header"]
        font_elements = self.font_settings["Elements"]
        font_option = self.font_settings["Options"]
        font_table = self.font_settings["Table"]

        str_lbl_01 = self.language_dict["Check-Up"][self.var_language]
        str_lbl_02 = self.language_dict["Acquisition Times"][self.var_language]

        ## Window Settings
        row_min = self.row_height
        column_min = self.column_width
        n_rows = self.window_dimensions["Check-up acquisition times"][0]
        n_columns = self.window_dimensions["Check-up acquisition times"][1]

        window_width = int(n_columns*self.column_width)
        window_height = int(n_rows*self.row_height)

        var_geometry = str(window_width) + "x" + str(window_height) + "+" + str(0) + "+" + str(0)

        subwindow_aquisition_times = tk.Toplevel(self.parent)
        subwindow_aquisition_times.title(str_lbl_01 + " - " + str_lbl_02)
        subwindow_aquisition_times.geometry(var_geometry)
        subwindow_aquisition_times.resizable(False, False)
        subwindow_aquisition_times["bg"] = background_color_dark

        for x in range(n_columns):
            tk.Grid.columnconfigure(subwindow_aquisition_times, x, weight=1)
        for y in range(n_rows):
            tk.Grid.rowconfigure(subwindow_aquisition_times, y, weight=1)

        # Rows
        for i in range(0, n_rows):
            subwindow_aquisition_times.grid_rowconfigure(i, minsize=row_min)
        # Columns
        for i in range(0, n_columns):
            subwindow_aquisition_times.grid_columnconfigure(i, minsize=column_min)

        start_column = 0
        start_row = 0

        ## LABELS
        str_lbl_03 = self.language_dict["Standard Files"][self.var_language]
        str_lbl_04 = self.language_dict["Sample Files"][self.var_language]

        lbl_std = SE(
            parent=subwindow_aquisition_times, row_id=start_row, column_id=start_column, n_rows=1, n_columns=16,
            fg=font_color_light, bg=background_color_dark).create_simple_label(
            text=str_lbl_03, relief=tk.FLAT, fontsize="sans 10 bold")
        lbl_smpl = SE(
            parent=subwindow_aquisition_times, row_id=start_row + 16, column_id=start_column, n_rows=1, n_columns=16,
            fg=font_color_light, bg=background_color_dark).create_simple_label(
            text=str_lbl_04, relief=tk.FLAT, fontsize="sans 10 bold")

        ## ENTRY
        self.var_entr_std_time = tk.StringVar()
        self.var_entr_std_time.set("00:00:00")

        entr_std = SE(
            parent=subwindow_aquisition_times, row_id=start_row + 1, column_id=start_column + 17, n_rows=2,
            n_columns=8, fg=font_color_dark, bg=self.bg_colors["White"]).create_simple_entry(
            var=self.var_entr_std_time, text_default=self.var_entr_std_time.get(),
            command=lambda event, filetype="STD": self.change_value_acquisition(filetype, event))

        self.var_entr_smpl_time = tk.StringVar()
        self.var_entr_smpl_time.set("00:00:00")

        entr_smpl = SE(
            parent=subwindow_aquisition_times, row_id=start_row + 17, column_id=start_column + 17, n_rows=2,
            n_columns=8, fg=font_color_dark, bg=self.bg_colors["White"]).create_simple_entry(
            var=self.var_entr_smpl_time, text_default=self.var_entr_smpl_time.get(),
            command=lambda event, filetype="SMPL": self.change_value_acquisition(filetype, event))

        ## TREEVIEWS
        str_lbl_05 = self.language_dict["Name"][self.var_language]
        str_lbl_06 = self.language_dict["Acquisition Time"][self.var_language]

        self.tv_std = SE(
            parent=subwindow_aquisition_times, row_id=start_row + 1, column_id=start_column, n_rows=14, n_columns=15,
            fg=font_color_dark, bg=background_color_light).create_treeview(
            n_categories=2, text_n=[str_lbl_05, str_lbl_06],
            width_n=["150", "150"], individual=True)

        scb_v = ttk.Scrollbar(subwindow_aquisition_times, orient="vertical")
        scb_h = ttk.Scrollbar(subwindow_aquisition_times, orient="horizontal")
        self.tv_std.configure(xscrollcommand=scb_h.set, yscrollcommand=scb_v.set)
        scb_v.config(command=self.tv_std.yview)
        scb_h.config(command=self.tv_std.xview)
        scb_v.grid(row=start_row + 1, column=start_column + 15, rowspan=14, columnspan=1, sticky="ns")
        scb_h.grid(row=start_row + 15, column=start_column, rowspan=1, columnspan=15, sticky="ew")

        for index, var_file in enumerate(self.container_lists["STD"]["Long"]):
            parts = var_file.split("/")
            file_std = parts[-1]
            var_file_short = file_std

            var_skipheader = self.container_icpms["skipheader"]
            var_skipfooter = self.container_icpms["skipfooter"]
            var_timestamp = self.container_icpms["timestamp"]
            var_icpms = self.container_icpms["name"]

            if self.file_loaded == False:
                dates, times = Data(filename=var_file).import_as_list(
                    skip_header=var_skipheader, skip_footer=var_skipfooter, timestamp=var_timestamp, icpms=var_icpms)
            else:
                var_time = self.container_var["acquisition times"]["STD"][var_file_short].get()
                parts_time = var_time.split(":")
                times = parts_time

            if index == 0:
                if self.file_loaded == False:
                    t_start_0 = datetime.timedelta(
                        hours=int(times[0][0]), minutes=int(times[0][1]), seconds=int(times[0][2]))
                    t_start = datetime.timedelta(
                        hours=int(times[0][0]), minutes=int(times[0][1]), seconds=int(times[0][2]))
                else:
                    t_start_0 = datetime.timedelta(
                        hours=int(times[0]), minutes=int(times[1]), seconds=int(times[2]))
                    t_start = datetime.timedelta(
                        hours=int(times[0]), minutes=int(times[1]), seconds=int(times[2]))
            else:
                if self.file_loaded == False:
                    t_start = datetime.timedelta(
                        hours=int(times[0][0]), minutes=int(times[0][1]), seconds=int(times[0][2]))
                else:
                    t_start = datetime.timedelta(
                        hours=int(times[0]), minutes=int(times[1]), seconds=int(times[2]))

            t_delta_0 = (t_start - t_start_0).total_seconds()
            self.container_lists["Acquisition Times Delta"][file_std] = t_delta_0

            if file_std not in self.container_var["acquisition times"]["STD"]:
                self.container_var["acquisition times"]["STD"][file_std] = tk.StringVar()
                self.container_var["acquisition times"]["STD"][file_std].set(
                    times[0][0] + ":" + times[0][1] + ":" + times[0][2])

            entry_std = [file_std, self.container_var["acquisition times"]["STD"][file_std].get()]
            self.tv_std.insert("", tk.END, values=entry_std)

        self.tv_smpl = SE(
            parent=subwindow_aquisition_times, row_id=start_row + 17, column_id=start_column, n_rows=14, n_columns=15,
            fg=font_color_dark, bg=background_color_light).create_treeview(
            n_categories=2, text_n=[str_lbl_05, str_lbl_06],
            width_n=["150", "150"], individual=True)

        scb_v = ttk.Scrollbar(subwindow_aquisition_times, orient="vertical")
        scb_h = ttk.Scrollbar(subwindow_aquisition_times, orient="horizontal")
        self.tv_smpl.configure(xscrollcommand=scb_h.set, yscrollcommand=scb_v.set)
        scb_v.config(command=self.tv_smpl.yview)
        scb_h.config(command=self.tv_smpl.xview)
        scb_v.grid(row=start_row + 17, column=start_column + 15, rowspan=14, columnspan=1, sticky="ns")
        scb_h.grid(row=start_row + 31, column=start_column, rowspan=1, columnspan=15, sticky="ew")

        for index, var_file in enumerate(self.container_lists["SMPL"]["Long"]):
            parts = var_file.split("/")
            file_smpl = parts[-1]
            var_file_short = file_smpl

            var_skipheader = self.container_icpms["skipheader"]
            var_skipfooter = self.container_icpms["skipfooter"]
            var_timestamp = self.container_icpms["timestamp"]
            var_icpms = self.container_icpms["name"]

            if self.file_loaded == False:
                dates, times = Data(filename=var_file).import_as_list(
                    skip_header=var_skipheader, skip_footer=var_skipfooter, timestamp=var_timestamp, icpms=var_icpms)
            else:
                var_time = self.container_var["acquisition times"]["SMPL"][var_file_short].get()
                parts_time = var_time.split(":")
                times = parts_time

            if self.file_loaded == False:
                t_start = datetime.timedelta(hours=int(times[0][0]), minutes=int(times[0][1]), seconds=int(times[0][2]))
            else:
                t_start = datetime.timedelta(hours=int(times[0]), minutes=int(times[1]), seconds=int(times[2]))
            t_delta_0 = (t_start - t_start_0).total_seconds()
            self.container_lists["Acquisition Times Delta"][file_smpl] = t_delta_0

            if file_smpl not in self.container_var["acquisition times"]["SMPL"]:
                self.container_var["acquisition times"]["SMPL"][file_smpl] = tk.StringVar()
                if self.file_loaded == False:
                    self.container_var["acquisition times"]["SMPL"][file_smpl].set(
                        times[0][0] + ":" + times[0][1] + ":" + times[0][2])
                else:
                    self.container_var["acquisition times"]["SMPL"][file_smpl].set(
                        times[0] + ":" + times[1] + ":" + times[2])

            entry_smpl = [file_smpl, self.container_var["acquisition times"]["SMPL"][file_smpl].get()]
            self.tv_smpl.insert("", tk.END, values=entry_smpl)

    def calculate_acquisition_time_deltas(self):
        for index, var_file in enumerate(self.container_lists["STD"]["Long"]):
            parts = var_file.split("/")
            file_std = parts[-1]
            var_skipheader = self.container_icpms["skipheader"]
            var_skipfooter = self.container_icpms["skipfooter"]
            var_timestamp = self.container_icpms["timestamp"]
            var_icpms = self.container_icpms["name"]

            if self.file_loaded == False:
                dates, times = Data(filename=var_file).import_as_list(
                    skip_header=var_skipheader, skip_footer=var_skipfooter, timestamp=var_timestamp, icpms=var_icpms)
            else:
                var_time = self.container_var["acquisition times"]["STD"][file_std].get()
                parts_time = var_time.split(":")
                times = parts_time

            if times[0] == "unknown" and self.file_loaded == True:
                dates, times = Data(filename=var_file).import_as_list(
                    skip_header=var_skipheader, skip_footer=var_skipfooter, timestamp=var_timestamp, icpms=var_icpms)

            if len(times) == 2:
                data_times = times[0]
            else:
                data_times = times

            if index == 0:
                if self.container_var["General Settings"]["Sensitivity Drift"].get() == 0:
                    if self.container_var["General Settings"]["Calculation Accuracy"].get() == 1:
                        t_start_0 = datetime.timedelta(
                            hours=int(data_times[0]), minutes=int(data_times[1]), seconds=int(data_times[2]))
                    else:
                        t_start_0 = datetime.timedelta(
                            hours=int(data_times[0]), minutes=int(data_times[1]), seconds=int("00"))
                else:
                    t_start_0 = int(data_times[0]) + int(data_times[1])/60

            if self.container_var["General Settings"]["Sensitivity Drift"].get() == 0:
                if self.container_var["General Settings"]["Calculation Accuracy"].get() == 1:
                    t_start = datetime.timedelta(
                        hours=int(data_times[0]), minutes=int(data_times[1]), seconds=int(data_times[2]))
                else:
                    t_start = datetime.timedelta(
                        hours=int(data_times[0]), minutes=int(data_times[1]), seconds=int("00"))
            else:
                t_start = int(data_times[0]) + int(data_times[1])/60

            if self.container_var["General Settings"]["Sensitivity Drift"].get() == 0:
                t_delta_0 = (t_start - t_start_0).total_seconds()
            else:
                t_delta_0 = t_start - t_start_0

            self.container_lists["Acquisition Times Delta"][file_std] = t_delta_0

        for index, var_file in enumerate(self.container_lists["SMPL"]["Long"]):
            parts = var_file.split("/")
            file_smpl = parts[-1]
            var_skipheader = self.container_icpms["skipheader"]
            var_skipfooter = self.container_icpms["skipfooter"]
            var_timestamp = self.container_icpms["timestamp"]
            var_icpms = self.container_icpms["name"]

            if self.file_loaded == False:
                dates, times = Data(filename=var_file).import_as_list(
                    skip_header=var_skipheader, skip_footer=var_skipfooter, timestamp=var_timestamp, icpms=var_icpms)
            else:
                var_time = self.container_var["acquisition times"]["SMPL"][file_smpl].get()
                parts_time = var_time.split(":")
                times = parts_time

            if times[0] == "unknown" and self.file_loaded == True:
                dates, times = Data(filename=var_file).import_as_list(
                    skip_header=var_skipheader, skip_footer=var_skipfooter, timestamp=var_timestamp, icpms=var_icpms)

            if len(times) == 2:
                data_times = times[0]
            else:
                data_times = times

            if self.container_var["General Settings"]["Sensitivity Drift"].get() == 0:
                if self.container_var["General Settings"]["Calculation Accuracy"].get() == 1:
                    t_start = datetime.timedelta(
                        hours=int(data_times[0]), minutes=int(data_times[1]), seconds=int(data_times[2]))
                else:
                    t_start = datetime.timedelta(
                        hours=int(data_times[0]), minutes=int(data_times[1]), seconds=int("00"))
            else:
                t_start = int(data_times[0]) + int(data_times[1])/60

            if self.container_var["General Settings"]["Sensitivity Drift"].get() == 0:
                t_delta_0 = (t_start - t_start_0).total_seconds()
            else:
                t_delta_0 = t_start - t_start_0

            self.container_lists["Acquisition Times Delta"][file_smpl] = t_delta_0

    def check_imported_files(self):
        # Colors
        font_color_dark = self.bg_colors["Dark Font"]
        font_color_light = self.bg_colors["Light Font"]
        background_color_dark = self.bg_colors["BG Window"]
        background_color_elements = self.bg_colors["Light"]
        background_color_light = self.bg_colors["Very Light"]
        accent_color = self.bg_colors["Accent"]  # self.accent_color
        font_header = self.font_settings["Header"]
        font_elements = self.font_settings["Elements"]
        font_option = self.font_settings["Options"]
        font_table = self.font_settings["Table"]

        str_lbl_01 = self.language_dict["Check-Up"][self.var_language]
        str_lbl_02 = self.language_dict["Imported Files"][self.var_language]

        ## Window Settings
        row_min = self.row_height
        column_min = self.column_width
        n_rows = self.window_dimensions["Check-up files"][0]
        n_columns = self.window_dimensions["Check-up files"][1]

        window_width = int(n_columns*self.column_width)
        window_height = int(n_rows*self.row_height)

        var_geometry = str(window_width) + "x" + str(window_height) + "+" + str(0) + "+" + str(0)

        subwindow_imported_files = tk.Toplevel(self.parent)
        subwindow_imported_files.title(str_lbl_01 + " - " + str_lbl_02)
        subwindow_imported_files.geometry(var_geometry)
        subwindow_imported_files.resizable(False, False)
        subwindow_imported_files["bg"] = background_color_dark

        for x in range(n_columns):
            tk.Grid.columnconfigure(subwindow_imported_files, x, weight=1)
        for y in range(n_rows):
            tk.Grid.rowconfigure(subwindow_imported_files, y, weight=1)

        # Rows
        for i in range(0, n_rows):
            subwindow_imported_files.grid_rowconfigure(i, minsize=row_min)
        # Columns
        for i in range(0, n_columns):
            subwindow_imported_files.grid_columnconfigure(i, minsize=column_min)

        ###########################################################

        start_column = 0
        start_row = 0

        ## FRAMES
        frm_00 = SE(
            parent=subwindow_imported_files, row_id=start_row, column_id=start_column + 26,
            n_rows=n_rows, n_columns=n_columns - 26, fg=font_color_dark,
            bg=background_color_light).create_frame(relief=tk.FLAT)

        ## LABELS
        str_lbl_03 = self.language_dict["Standard Files"][self.var_language]
        str_lbl_04 = self.language_dict["Sample Files"][self.var_language]

        lbl_std = SE(
            parent=subwindow_imported_files, row_id=start_row, column_id=start_column, n_rows=1, n_columns=26,
            fg=font_color_light, bg=background_color_dark).create_simple_label(
            text=str_lbl_03, relief=tk.FLAT, fontsize="sans 10 bold")
        lbl_smpl = SE(
            parent=subwindow_imported_files, row_id=start_row + 16, column_id=start_column, n_rows=1, n_columns=26,
            fg=font_color_light, bg=background_color_dark).create_simple_label(
            text=str_lbl_04, relief=tk.FLAT, fontsize="sans 10 bold")

        ## FIGURES
        fig_checkup_std = Figure(figsize=(10, 5), tight_layout=True, facecolor=background_color_light)
        fig_checkup_smpl = Figure(figsize=(10, 5), tight_layout=True, facecolor=background_color_light)

        canvas_std = FigureCanvasTkAgg(fig_checkup_std, master=subwindow_imported_files)
        canvas_std.get_tk_widget().grid(
            row=start_row + 1, column=start_column + 26, rowspan=15, columnspan=28, sticky="nesw")
        canvas_smpl = FigureCanvasTkAgg(fig_checkup_smpl, master=subwindow_imported_files)
        canvas_smpl.get_tk_widget().grid(
            row=start_row + 17, column=start_column + 26, rowspan=15, columnspan=28, sticky="nesw")

        self.container_checkup["STD"].append(canvas_std)
        self.container_checkup["SMPL"].append(canvas_smpl)
        self.temp_figure_checkup = {"STD": fig_checkup_std, "SMPL": fig_checkup_smpl}
        self.temp_canvas_checkup = {"STD": canvas_std, "SMPL": canvas_smpl}
        self.temp_axes_checkup = {"STD": {}, "SMPL": {}}
        self.temp_lines_checkup = {"STD": {}, "SMPL": {}}

        ## BUTTONS
        str_lbl_05 = self.language_dict["Before"][self.var_language]
        str_lbl_06 = self.language_dict["Next"][self.var_language]

        btn_std_01 = SE(
            parent=subwindow_imported_files, row_id=start_row, column_id=start_column + 26, n_rows=1, n_columns=6,
            fg=font_color_dark, bg=background_color_elements).create_simple_button(
            text=str_lbl_05, bg_active=accent_color, fg_active=font_color_light,
            command=lambda filetype="STD", mode="Back", var_fig=fig_checkup_std:
            self.change_graph_checkup(filetype, mode, var_fig))
        btn_std_02 = SE(
            parent=subwindow_imported_files, row_id=start_row, column_id=start_column + 32, n_rows=1, n_columns=6,
            fg=font_color_dark, bg=background_color_elements).create_simple_button(
            text=str_lbl_06, bg_active=accent_color, fg_active=font_color_light,
            command=lambda filetype="STD", var_fig=fig_checkup_std: self.change_graph_checkup(filetype, var_fig))
        btn_smpl_01 = SE(
            parent=subwindow_imported_files, row_id=start_row + 16, column_id=start_column + 26, n_rows=1, n_columns=6,
            fg=font_color_dark, bg=background_color_elements).create_simple_button(
            text=str_lbl_05, bg_active=accent_color, fg_active=font_color_light,
            command=lambda filetype="SMPL", mode="Back", var_fig=fig_checkup_smpl:
            self.change_graph_checkup(filetype, mode, var_fig))
        btn_smpl_02 = SE(
            parent=subwindow_imported_files, row_id=start_row + 16, column_id=start_column + 32, n_rows=1, n_columns=6,
            fg=font_color_dark, bg=background_color_elements).create_simple_button(
            text=str_lbl_06, bg_active=accent_color, fg_active=font_color_light,
            command=lambda filetype="SMPL", var_fig=fig_checkup_smpl: self.change_graph_checkup(filetype, var_fig))

        ## TREEVIEW
        str_lbl_07 = self.language_dict["Name"][self.var_language]
        str_lbl_08 = self.language_dict["N(isotopes)"][self.var_language]
        str_lbl_09 = self.language_dict["t(start)"][self.var_language]
        str_lbl_10 = self.language_dict["t(end)"][self.var_language]
        str_lbl_11 = self.language_dict["t(day)"][self.var_language]
        str_lbl_12 = self.language_dict["date"][self.var_language]

        tv_std = SE(
            parent=subwindow_imported_files, row_id=start_row + 1, column_id=start_column, n_rows=15, n_columns=26,
            fg=font_color_dark, bg=self.bg_colors["White"]).create_treeview(
            n_categories=6, text_n=[str_lbl_07, str_lbl_08, str_lbl_09, str_lbl_10, str_lbl_11, str_lbl_12],
            width_n=["120", "80", "80", "80", "80", "80"], individual=True)

        for index, var_file in enumerate(self.container_lists["STD"]["Long"]):
            parts = var_file.split("/")
            file_std = parts[-1]
            var_file_short = parts[-1]
            if self.file_loaded == False:
                if self.container_icpms["name"] != None:
                    var_skipheader = self.container_icpms["skipheader"]
                    var_skipfooter = self.container_icpms["skipfooter"]
                    df_data = DE(filename_long=var_file).get_measurements(
                        delimiter=",", skip_header=var_skipheader, skip_footer=var_skipfooter)
                else:
                    df_data = DE(filename_long=var_file).get_measurements(
                        delimiter=",", skip_header=3, skip_footer=1)
            else:
                df_data = self.container_measurements["Dataframe"][var_file_short]

            dataset_time = list(DE().get_times(dataframe=df_data))
            var_skipheader = self.container_icpms["skipheader"]
            var_skipfooter = self.container_icpms["skipfooter"]
            var_timestamp = self.container_icpms["timestamp"]
            var_icpms = self.container_icpms["name"]

            if self.file_loaded == False:
                dates, times = Data(filename=var_file).import_as_list(
                    skip_header=var_skipheader, skip_footer=var_skipfooter, timestamp=var_timestamp, icpms=var_icpms)
            else:
                var_time = self.container_var["acquisition times"]["STD"][var_file_short].get()
                parts_time = var_time.split(":")
                times = parts_time

            if self.file_loaded == False:
                entry_std = [file_std, len(list(df_data.keys())[1:]), dataset_time[0], dataset_time[-1],
                             times[0][0] + ":" + times[0][1] + ":" + times[0][2],
                             dates[0][0] + "/" + dates[0][1] + "/" + dates[0][2]]
            else:
                entry_std = [file_std, len(list(df_data.keys())[1:]), dataset_time[0], dataset_time[-1],
                             times[0] + ":" + times[1] + ":" + times[2]]

            tv_std.insert("", tk.END, values=entry_std)

            self.temp_axes_checkup["STD"][file_std] = fig_checkup_std.add_subplot(label=file_std)
            self.temp_lines_checkup["STD"][file_std] = 0

        tv_smpl = SE(
            parent=subwindow_imported_files, row_id=start_row + 17, column_id=start_column, n_rows=15, n_columns=26,
            fg=font_color_dark, bg=self.bg_colors["White"]).create_treeview(
            n_categories=6, text_n=[str_lbl_07, str_lbl_08, str_lbl_09, str_lbl_10, str_lbl_11, str_lbl_12],
            width_n=["120", "80", "80", "80", "80", "80"], individual=True)

        for index, var_file in enumerate(self.container_lists["SMPL"]["Long"]):
            parts = var_file.split("/")
            file_smpl = parts[-1]
            var_file_short = parts[-1]
            if self.file_loaded == False:
                if self.container_icpms["name"] != None:
                    var_skipheader = self.container_icpms["skipheader"]
                    var_skipfooter = self.container_icpms["skipfooter"]
                    df_data = DE(filename_long=var_file).get_measurements(
                        delimiter=",", skip_header=var_skipheader, skip_footer=var_skipfooter)
                else:
                    df_data = DE(filename_long=var_file).get_measurements(
                        delimiter=",", skip_header=3, skip_footer=1)
            else:
                df_data = self.container_measurements["Dataframe"][var_file_short]

            dataset_time = list(DE().get_times(dataframe=df_data))
            var_skipheader = self.container_icpms["skipheader"]
            var_skipfooter = self.container_icpms["skipfooter"]
            var_timestamp = self.container_icpms["timestamp"]
            var_icpms = self.container_icpms["name"]
            if self.file_loaded == False:
                dates, times = Data(filename=var_file).import_as_list(
                    skip_header=var_skipheader, skip_footer=var_skipfooter, timestamp=var_timestamp, icpms=var_icpms)
            else:
                var_time = self.container_var["acquisition times"]["SMPL"][var_file_short].get()
                parts_time = var_time.split(":")
                times = parts_time

            if self.file_loaded == False:
                entry_smpl = [file_smpl, len(list(df_data.keys())[1:]), dataset_time[0], dataset_time[-1],
                              times[0][0] + ":" + times[0][1] + ":" + times[0][2],
                              dates[0][0] + "/" + dates[0][1] + "/" + dates[0][2]]
            else:
                entry_smpl = [file_smpl, len(list(df_data.keys())[1:]), dataset_time[0], dataset_time[-1],
                              times[0] + ":" + times[1] + ":" + times[2]]

            tv_smpl.insert("", tk.END, values=entry_smpl)

            self.temp_axes_checkup["SMPL"][file_smpl] = fig_checkup_smpl.add_subplot(label=file_smpl)
            self.temp_lines_checkup["SMPL"][file_smpl] = 0

        ## INITIALIZATION
        self.change_graph_checkup(filetype="STD", initialization=True, var_fig=fig_checkup_std)
        self.change_graph_checkup(filetype="SMPL", initialization=True, var_fig=fig_checkup_smpl)

    def change_graph_checkup(self, filetype, var_fig, mode="Next", initialization=False):
        counter_key = "Counter " + str(filetype)
        #
        for key, axis_item in self.temp_axes_checkup[filetype].items():
            axis_item.axis("off")
            axis_item.set_visible(False)
        #
        self.temp_canvas_checkup[filetype].draw()
        #
        if initialization == False:
            n_files = len(self.container_lists[filetype]["Long"])
            #
            if mode == "Next":
                if self.container_checkup[counter_key] < n_files - 1:
                    self.container_checkup[counter_key] += 1
                else:
                    self.container_checkup[counter_key] = 0
            else:
                if self.container_checkup[counter_key] > 0:
                    self.container_checkup[counter_key] -= 1
                else:
                    self.container_checkup[counter_key] = n_files - 1
        #
        var_file = self.container_lists[filetype]["Long"][self.container_checkup[counter_key]]
        #
        parts = var_file.split("/")
        var_file_short = parts[-1]
        var_fig = self.temp_figure_checkup[filetype]
        #
        if filetype == "STD":
            self.fi_current_file_std = var_file
        elif filetype == "SMPL":
            self.fi_current_file_smpl = var_file
        #
        if self.temp_lines_checkup[filetype][var_file_short] == 0:
            if self.file_loaded == False:
                if self.container_icpms["name"] != None:
                    var_skipheader = self.container_icpms["skipheader"]
                    var_skipfooter = self.container_icpms["skipfooter"]
                    df_data = DE(filename_long=var_file).get_measurements(
                        delimiter=",", skip_header=var_skipheader, skip_footer=var_skipfooter)
                else:
                    df_data = DE(filename_long=var_file).get_measurements(
                        delimiter=",", skip_header=3, skip_footer=1)
            else:
                df_data = self.container_measurements["Dataframe"][var_file_short]

            dataset_time = list(DE().get_times(dataframe=df_data))
            x_max = max(dataset_time)
            icp_measurements = np.array([[df_data[isotope] for isotope in self.container_lists["ISOTOPES"]]])
            y_max = np.amax(icp_measurements)
            #
            ## DIAGRAMS
            ax = var_fig.add_subplot(label=np.random.uniform())
            self.temp_axes_checkup[filetype][var_file_short] = ax
            #
            for isotope in self.container_lists["ISOTOPES"]:
                ln = ax.plot(dataset_time, df_data[isotope], label=isotope, color=self.isotope_colors[isotope],
                             linewidth=1, visible=True)
            #
            if self.pysills_mode in ["FI", "MI", "INCL"]:
                var_check_bg = self.container_helper[filetype][var_file_short]["BG"]["Content"]
            else:
                var_check_bg = self.container_helper[filetype][var_file_short]["BG"]["Content"]
            #
            if len(var_check_bg) > 0:
                if self.pysills_mode in ["FI", "MI", "INCL"]:
                    for var_id, var_content in self.container_helper[filetype][var_file_short]["BG"]["Content"].items():
                        times_bg = var_content["Times"]
                        #
                        box_bg = ax.axvspan(times_bg[0], times_bg[1], alpha=0.35, color=self.colors_intervals["BG"])
                else:
                    for var_id, var_content in self.container_helper[filetype][var_file_short]["BG"]["Content"].items():
                        times_bg = var_content["Times"]
                        #
                        box_bg = ax.axvspan(times_bg[0], times_bg[1], alpha=0.35, color=self.colors_intervals["BG"])
                    # times_bg = self.container_helper[filetype][var_file_short]["BG"][1]["Times"]
                    # #
                    # box_bg = ax.axvspan(times_bg[0], times_bg[1], alpha=0.5, color=self.colors_intervals["BG"])
            #
            if self.pysills_mode == "MA":
                signal_key = "MAT"
                var_check_sig = self.container_helper[filetype][var_file_short][signal_key]["Content"]
                # signal_key = "SIG"
                # var_check_sig = self.container_helper[filetype][var_file_short][signal_key]
            else:
                signal_key = "MAT"
                var_check_sig = self.container_helper[filetype][var_file_short][signal_key]["Content"]
            #
            if len(var_check_sig) > 0:
                if self.pysills_mode in ["FI"]:
                    for var_id, var_content in self.container_helper[filetype][var_file_short][signal_key][
                        "Content"].items():
                        times_sig = var_content["Times"]
                        var_color = self.colors_intervals["MAT"]
                        #
                        box_sig = ax.axvspan(times_sig[0], times_sig[1], alpha=0.35, color=var_color)
                elif self.pysills_mode in ["MI"]:
                    for var_id, var_content in self.container_helper[filetype][var_file_short][signal_key][
                        "Content"].items():
                        times_sig = self.container_helper[filetype][var_file_short][signal_key]["Content"][1]["Times"]
                        var_color = self.colors_intervals["MAT"]
                        #
                        box_sig = ax.axvspan(times_sig[0], times_sig[1], alpha=0.35, color=var_color)
                else:
                    for var_id, var_content in self.container_helper[filetype][var_file_short][signal_key][
                        "Content"].items():
                        times_sig = var_content["Times"]
                        var_color = self.colors_intervals["MAT"]
                        #
                        box_sig = ax.axvspan(times_sig[0], times_sig[1], alpha=0.35, color=var_color)
                    # times_sig = times_sig = self.container_helper[filetype][var_file_short][signal_key][1]["Times"]
                    # var_color = self.colors_intervals["SIG"]
                    # #
                    # box_sig = ax.axvspan(times_sig[0], times_sig[1], alpha=0.35, color=var_color)
            #
            if self.pysills_mode == "MA":
                inclusion_key = "INCL"
                var_check_incl = []
            else:
                inclusion_key = "INCL"
                var_check_incl = self.container_helper[filetype][var_file_short][inclusion_key]["Content"]
            #
            if len(var_check_incl) > 0:
                if self.pysills_mode in ["FI"]:
                    for var_id, var_content in self.container_helper[filetype][var_file_short][inclusion_key][
                        "Content"].items():
                        times_sig = var_content["Times"]
                        #
                        box_sig = ax.axvspan(times_sig[0], times_sig[1], alpha=0.35,
                                             color=self.colors_intervals["INCL"])
                elif self.pysills_mode in ["MI"]:
                    for var_id, var_content in self.container_helper[filetype][var_file_short][inclusion_key][
                        "Content"].items():
                        times_sig = self.container_helper[filetype][var_file_short][inclusion_key]["Content"][1][
                            "Times"]
                        #
                        box_sig = ax.axvspan(times_sig[0], times_sig[1], alpha=0.35,
                                             color=self.colors_intervals["INCL"])
            #
            ax.grid(True)
            ax.set_yscale("log")
            ax.set_xlim(left=0, right=x_max)
            ax.set_xticks(np.arange(0, x_max, 20))
            ax.set_ylim(bottom=100, top=1.5*y_max)
            ax.grid(which="major", linestyle="-", linewidth=1)
            ax.minorticks_on()
            ax.grid(which="minor", linestyle=":", linewidth=0.5, alpha=0.75)
            ax.set_axisbelow(True)
            ax.set_title(var_file_short, fontsize=9)
            ax.set_xlabel("Time (s)", labelpad=0.5, fontsize=8)
            ax.set_ylabel("Signal (cps)", labelpad=0.5, fontsize=8)
            ax.xaxis.set_tick_params(labelsize=8)
            ax.yaxis.set_tick_params(labelsize=8)
            #
            var_fig.subplots_adjust(bottom=0.125, top=0.925, left=0.1125, right=0.95)
            #
            self.temp_canvas_checkup[filetype].draw()
            #
            self.temp_lines_checkup[filetype][var_file_short] = 1
            #
        else:
            self.temp_axes_checkup[filetype][var_file_short].axis("on")
            self.temp_axes_checkup[filetype][var_file_short].set_visible(True)
            #
            self.temp_canvas_checkup[filetype].draw()

    def change_value_acquisition(self, filetype, event):
        if filetype == "STD":
            selected_item = self.tv_std.selection()[0]
            index = int(str(selected_item)[-1]) - 1
            new_time = self.var_entr_std_time.get()
            var_file = self.container_lists[filetype]["Short"][index]
            #
            self.tv_std.item(item=selected_item, text=var_file, values=(var_file, new_time))
        else:
            selected_item = self.tv_smpl.selection()[0]
            index = int(str(selected_item)[-1]) - 1
            new_time = self.var_entr_smpl_time.get()
            var_file = self.container_lists[filetype]["Short"][index]
            #
            self.tv_smpl.item(item=selected_item, text=var_file, values=(var_file, new_time))
        #
        self.container_var["acquisition times"][filetype][var_file].set(new_time)

    ####################################################################################################################
    ### EXTRAS #########################################################################################################
    ####################################################################################################################
    def ma_extras(self):
        """Main window of additional analysis tools for a mineral analysis project."""
        str_title_01 = self.language_dict["Mineral Analysis"][self.var_language]
        str_title_02 = self.language_dict["Extras"][self.var_language]

        ## Window Settings
        row_min = self.row_height
        column_min = self.column_width
        n_rows = self.window_dimensions["MA Extras"][0]
        n_columns = self.window_dimensions["MA Extras"][1]

        window_width = int(n_columns*self.column_width)
        window_height = int(n_rows*self.row_height)

        var_geometry = str(window_width) + "x" + str(window_height) + "+" + str(0) + "+" + str(0)

        self.subwindow_ma_extras = tk.Toplevel(self.parent)
        self.subwindow_ma_extras.title(str_title_01 + " - " + str_title_02)
        self.subwindow_ma_extras.geometry(var_geometry)
        self.subwindow_ma_extras.resizable(False, False)
        self.subwindow_ma_extras["bg"] = self.bg_colors["BG Window"]

        for x in range(n_columns):
            tk.Grid.columnconfigure(self.subwindow_ma_extras, x, weight=1)
        for y in range(n_rows):
            tk.Grid.rowconfigure(self.subwindow_ma_extras, y, weight=1)

        # Rows
        for i in range(0, n_rows):
            self.subwindow_ma_extras.grid_rowconfigure(i, minsize=row_min)
        # Columns
        for i in range(0, n_columns):
            self.subwindow_ma_extras.grid_columnconfigure(i, minsize=column_min)

        var_row_start = 0
        var_column_start = 0
        var_header_n = 12
        int_category_n = 8

        # Initialization
        for var_filetype in ["STD", "SMPL"]:
            for var_file_short in self.container_lists[var_filetype]["Short"]:
                self.get_condensed_intervals_of_file(filetype=var_filetype, filename_short=var_file_short)

        var_filetype = "None"
        var_file_short = "None"
        var_file_long = "None"
        var_focus = "None"

        for var_datatype in ["RAW", "SMOOTHED"]:
            # Intensity Analysis
            self.get_intensity(
                var_filetype=var_filetype, var_datatype=var_datatype, var_file_short=var_file_short,
                var_focus=var_focus, mode="All")

            if self.container_var["General Settings"]["Desired Average"].get() == 1:
                str_averagetype = "arithmetic mean"
            else:
                str_averagetype = "median"

            IQ(dataframe=None, project_type=self.pysills_mode,
               results_container=self.container_intensity_corrected["STD"][var_datatype]).get_averaged_intensities(
                data_container=self.container_intensity_corrected["STD"][var_datatype],
                average_type=str_averagetype)

            self.ma_get_intensity_ratio(
                var_filetype=var_filetype, var_datatype=var_datatype, var_file_short=var_file_short,
                var_file_long=var_file_long, var_focus=var_focus, mode="All")
            # Sensitivity Analysis
            self.get_analytical_sensitivity(
                var_filetype=var_filetype, var_datatype=var_datatype, var_file_short=var_file_short,
                var_file_long=var_file_long, mode="All")
            results_is = self.determine_possible_is(filetype="ALL")
            IQ(dataframe=None, project_type=self.pysills_mode,
               results_container=self.container_intensity_ratio).get_intensity_ratio(
                data_container=self.container_intensity_corrected, dict_is=results_is, datatype=var_datatype)
            # Compositional Analysis
            self.ma_get_concentration(
                var_filetype=var_filetype, var_datatype=var_datatype, var_file_short=var_file_short,
                var_file_long=var_file_long, mode="All")
            SQ(dataframe_01=self.container_intensity_corrected, dataframe_02=self.container_concentration,
               results_container=self.container_normalized_sensitivity).get_normalized_sensitivity(
                datatype=var_datatype, data_sensitivity=self.container_analytical_sensitivity, dict_is=results_is)
            self.ma_get_rsf(
                var_filetype=var_filetype, var_datatype=var_datatype, var_file_short=var_file_short,
                var_file_long=var_file_long, mode="All")
            self.ma_get_concentration_ratio(
                var_filetype=var_filetype, var_datatype=var_datatype, var_file_short=var_file_short,
                var_file_long=var_file_long, mode="All")
            self.ma_get_lod(
                var_filetype=var_filetype, var_datatype=var_datatype, var_file_short=var_file_short,
                var_file_long=var_file_long, mode="All")

        # LABELS
        str_lbl_01 = self.language_dict["Mineral Analysis"][self.var_language]
        str_lbl_02 = self.language_dict["Further data analysis"][self.var_language]
        str_lbl_03 = self.language_dict["x-y diagram"][self.var_language]
        str_lbl_04 = self.language_dict["Ternary diagram"][self.var_language]
        str_lbl_05 = self.language_dict["Geothermometry"][self.var_language]
        str_lbl_06 = self.language_dict["elements"][self.var_language]
        str_lbl_07 = self.language_dict["element ratios"][self.var_language]
        str_lbl_08 = self.language_dict["oxides"][self.var_language]
        str_lbl_09 = self.language_dict["oxide ratios"][self.var_language]

        lbl_01 = SE(
            parent=self.subwindow_ma_extras, row_id=var_row_start, column_id=var_column_start, n_rows=1,
            n_columns=var_header_n, fg=self.bg_colors["Light Font"],
            bg=self.bg_colors["BG Window"]).create_simple_label(
            text=str_lbl_01 + " - " + str_lbl_02, relief=tk.FLAT, fontsize="sans 10 bold")
        lbl_001a = SE(
            parent=self.subwindow_ma_extras, row_id=var_row_start + 1, column_id=var_column_start, n_rows=1,
            n_columns=int_category_n, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_simple_label(
            text=str_lbl_03 + " (" + str_lbl_06 + ")", relief=tk.FLAT, fontsize="sans 10 bold", anchor=tk.W)
        lbl_001b = SE(
            parent=self.subwindow_ma_extras, row_id=var_row_start + 2, column_id=var_column_start, n_rows=1,
            n_columns=int_category_n, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_simple_label(
            text=str_lbl_03 + " (" + str_lbl_07 + ")", relief=tk.FLAT, fontsize="sans 10 bold", anchor=tk.W)
        lbl_001c = SE(
            parent=self.subwindow_ma_extras, row_id=var_row_start + 3, column_id=var_column_start, n_rows=1,
            n_columns=int_category_n, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_simple_label(
            text=str_lbl_03 + " (" + str_lbl_08 + ")", relief=tk.FLAT, fontsize="sans 10 bold", anchor=tk.W)
        lbl_001d = SE(
            parent=self.subwindow_ma_extras, row_id=var_row_start + 4, column_id=var_column_start, n_rows=1,
            n_columns=int_category_n, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_simple_label(
            text=str_lbl_03 + " (" + str_lbl_09 + ")", relief=tk.FLAT, fontsize="sans 10 bold", anchor=tk.W)
        lbl_001e = SE(
            parent=self.subwindow_ma_extras, row_id=var_row_start + 5, column_id=var_column_start, n_rows=1,
            n_columns=int_category_n, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_simple_label(
            text=str_lbl_04 + " (" + str_lbl_06 + ")", relief=tk.FLAT, fontsize="sans 10 bold", anchor=tk.W)
        lbl_001f = SE(
            parent=self.subwindow_ma_extras, row_id=var_row_start + 6, column_id=var_column_start, n_rows=1,
            n_columns=int_category_n, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_simple_label(
            text=str_lbl_04 + " (" + str_lbl_08 + ")", relief=tk.FLAT, fontsize="sans 10 bold", anchor=tk.W)
        lbl_02 = SE(
            parent=self.subwindow_ma_extras, row_id=var_row_start, column_id=var_header_n, n_rows=1,
            n_columns=var_header_n, fg=self.bg_colors["Light Font"],
            bg=self.bg_colors["BG Window"]).create_simple_label(
            text=str_lbl_05, relief=tk.FLAT, fontsize="sans 10 bold")

        lbl_002 = SE(
            parent=self.subwindow_ma_extras, row_id=var_row_start + 1, column_id=var_header_n, n_rows=1,
            n_columns=int_category_n, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_simple_label(
            text="Ti-in-Quartz", relief=tk.FLAT, fontsize="sans 10 bold")

        # BUTTONS
        str_btn_01 = self.language_dict["Setup"][self.var_language]

        btn_001a = SE(
            parent=self.subwindow_ma_extras, row_id=var_row_start + 1, column_id=int_category_n,
            n_rows=1, n_columns=var_header_n - int_category_n, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_simple_button(
            text=str_btn_01, bg_active=self.accent_color, fg_active=self.bg_colors["Dark Font"],
            command=lambda mode="Elements": self.diagram_xy(mode))
        btn_001b = SE(
            parent=self.subwindow_ma_extras, row_id=var_row_start + 2, column_id=int_category_n,
            n_rows=1, n_columns=var_header_n - int_category_n, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_simple_button(
            text=str_btn_01, bg_active=self.accent_color, fg_active=self.bg_colors["Dark Font"],
            command=lambda mode="Element ratios": self.diagram_xy(mode))
        btn_001c = SE(
            parent=self.subwindow_ma_extras, row_id=var_row_start + 3, column_id=int_category_n,
            n_rows=1, n_columns=var_header_n - int_category_n, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_simple_button(
            text=str_btn_01, bg_active=self.accent_color, fg_active=self.bg_colors["Dark Font"],
            command=lambda mode="Oxides": self.diagram_xy(mode))
        btn_001d = SE(
            parent=self.subwindow_ma_extras, row_id=var_row_start + 4, column_id=int_category_n,
            n_rows=1, n_columns=var_header_n - int_category_n, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_simple_button(
            text=str_btn_01, bg_active=self.accent_color, fg_active=self.bg_colors["Dark Font"],
            command=lambda mode="Oxide ratios": self.diagram_xy(mode))
        btn_001e = SE(
            parent=self.subwindow_ma_extras, row_id=var_row_start + 5, column_id=int_category_n,
            n_rows=1, n_columns=var_header_n - int_category_n, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_simple_button(
            text=str_btn_01, bg_active=self.accent_color, fg_active=self.bg_colors["Dark Font"])
        btn_001f = SE(
            parent=self.subwindow_ma_extras, row_id=var_row_start + 6, column_id=int_category_n,
            n_rows=1, n_columns=var_header_n - int_category_n, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_simple_button(
            text=str_btn_01, bg_active=self.accent_color, fg_active=self.bg_colors["Dark Font"])

        if len(self.container_lists["Selected Oxides"]["All"]) == 0:
            btn_001c.configure(state="disabled")
            btn_001d.configure(state="disabled")

        btn_001e.configure(state="disabled")
        btn_001f.configure(state="disabled")

        btn_002 = SE(
            parent=self.subwindow_ma_extras, row_id=var_row_start + 1, column_id=var_header_n + int_category_n,
            n_rows=1, n_columns=var_header_n - int_category_n, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_simple_button(
            text=str_btn_01, bg_active=self.accent_color, fg_active=self.bg_colors["Dark Font"],
            command=self.geothermometry_titanium_in_quartz)

    def geothermometry_titanium_in_quartz(self):
        """Window for the Titanium-in-Quartz thermometry."""
        str_title_01 = self.language_dict["Geothermometry"][self.var_language]

        ## Window Settings
        row_min = self.row_height
        column_min = self.column_width
        n_rows = self.window_dimensions["Geothermometry"][0]
        n_columns = self.window_dimensions["Geothermometry"][1]

        window_width = int(n_columns*self.column_width)
        window_height = int(n_rows*self.row_height)

        var_geometry = str(window_width) + "x" + str(window_height) + "+" + str(0) + "+" + str(0)

        self.subwindow_geothermometry_ti_in_qz = tk.Toplevel(self.parent)
        self.subwindow_geothermometry_ti_in_qz.title(str_title_01 + " - Titanium-in-Quartz")
        self.subwindow_geothermometry_ti_in_qz.geometry(var_geometry)
        self.subwindow_geothermometry_ti_in_qz.resizable(False, False)
        self.subwindow_geothermometry_ti_in_qz["bg"] = self.bg_colors["BG Window"]

        for x in range(n_columns):
            tk.Grid.columnconfigure(self.subwindow_geothermometry_ti_in_qz, x, weight=1)
        for y in range(n_rows):
            tk.Grid.rowconfigure(self.subwindow_geothermometry_ti_in_qz, y, weight=1)

        # Rows
        for i in range(0, n_rows):
            self.subwindow_geothermometry_ti_in_qz.grid_rowconfigure(i, minsize=row_min)
        # Columns
        for i in range(0, n_columns):
            self.subwindow_geothermometry_ti_in_qz.grid_columnconfigure(i, minsize=column_min)

        var_row_start = 0
        var_column_start = 0
        var_header_n = 12
        int_category_n = 12

        # LABELS
        str_lbl_01 = self.language_dict["Geothermometry"][self.var_language]
        str_lbl_02 = self.language_dict["Additional Settings"][self.var_language]
        str_lbl_03 = self.language_dict["Results"][self.var_language]

        lbl_01 = SE(
            parent=self.subwindow_geothermometry_ti_in_qz, row_id=var_row_start, column_id=var_column_start, n_rows=1,
            n_columns=var_header_n, fg=self.bg_colors["Light Font"],
            bg=self.bg_colors["BG Window"]).create_simple_label(
            text="Titanium-in-Quartz " + str_lbl_01, relief=tk.FLAT, fontsize="sans 10 bold")

        lbl_02 = SE(
            parent=self.subwindow_geothermometry_ti_in_qz, row_id=var_row_start + 6, column_id=var_column_start,
            n_rows=1, n_columns=var_header_n, fg=self.bg_colors["Light Font"],
            bg=self.bg_colors["BG Window"]).create_simple_label(
            text=str_lbl_02, relief=tk.FLAT, fontsize="sans 10 bold")
        lbl_02a = SE(
            parent=self.subwindow_geothermometry_ti_in_qz, row_id=var_row_start + 7, column_id=var_column_start,
            n_rows=1, n_columns=9, fg=self.bg_colors["Light Font"],
            bg=self.bg_colors["BG Window"]).create_simple_label(
            text="Activity of TiO2 in Rt", relief=tk.FLAT, fontsize="sans 10 bold", anchor=tk.W)

        lbl_03 = SE(
            parent=self.subwindow_geothermometry_ti_in_qz, row_id=var_row_start,
            column_id=var_column_start + var_header_n + 1, n_rows=1, n_columns=var_header_n + 14,
            fg=self.bg_colors["Light Font"], bg=self.bg_colors["BG Window"]).create_simple_label(
            text=str_lbl_03, relief=tk.FLAT, fontsize="sans 10 bold")

        # BUTTONS
        str_btn_01 = self.language_dict["Run"][self.var_language]
        str_btn_02 = self.language_dict["Export results"][self.var_language]

        btn_03 = SE(
            parent=self.subwindow_geothermometry_ti_in_qz, row_id=var_row_start + 5, column_id=var_row_start, n_rows=1,
            n_columns=6, fg=self.bg_colors["Dark Font"], bg=self.accent_color).create_simple_button(
            text=str_btn_01, bg_active=self.accent_color, fg_active=self.bg_colors["Dark Font"],
            command=self.run_geothermometer_titanium_in_quartz)
        btn_03 = SE(
            parent=self.subwindow_geothermometry_ti_in_qz, row_id=var_row_start + 5, column_id=6, n_rows=1,
            n_columns=6, fg=self.bg_colors["Dark Font"], bg=self.accent_color).create_simple_button(
            text=str_btn_02, bg_active=self.accent_color, fg_active=self.bg_colors["Dark Font"])

        # RADIOBUTTONS
        rb_001 = SE(
            parent=self.subwindow_geothermometry_ti_in_qz, row_id=var_row_start + 1, column_id=var_column_start,
            n_rows=1, n_columns=int_category_n, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_radiobutton(
            var_rb=self.container_var["Geothermometry"]["Titanium in Quartz"], value_rb=0,
            color_bg=self.bg_colors["Light"], fg=self.bg_colors["Dark Font"], text="TitaniQ - after Wark (2006)",
            sticky="nesw", relief=tk.FLAT, font="sans 10 bold")

        ## ENTRIES
        entr_02a = SE(
            parent=self.subwindow_geothermometry_ti_in_qz, row_id=var_row_start + 7, column_id=var_column_start + 9,
            n_rows=1, n_columns=3, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["White"]).create_simple_entry(
            var=self.container_var["a(TiO2,Rt)"], text_default=self.container_var["a(TiO2,Rt)"].get())

        ## TREEVIEW
        str_tv_01 = self.language_dict["Name"][self.var_language]
        str_tv_02 = self.language_dict["Category"][self.var_language]

        self.tv_results_ti_in_qz = SE(
            parent=self.subwindow_geothermometry_ti_in_qz, row_id=var_row_start + 1,
            column_id=var_column_start + var_header_n + 1, n_rows=14, n_columns=var_header_n + 14,
            fg=self.bg_colors["Dark Font"], bg=self.bg_colors["White"]).create_treeview(
            n_categories=5, text_n=[str_tv_01, str_tv_02, "C(ppm)", "T(°C)", "T(K)"],
            width_n=["150", "100", "90", "90", "90"], individual=True)

        for index, str_filename_short in enumerate(self.container_lists["SMPL"]["Short"]):
            entry_results = [str_filename_short, "-", "-"]
            self.tv_results_ti_in_qz.insert("", tk.END, values=entry_results)
            file_isotopes = self.container_lists["Measured Isotopes"][str_filename_short]
            self.list_ti_isotopes = []

            for isotope in file_isotopes:
                key_element = re.search(r"(\D+)(\d+)", isotope)
                element = key_element.group(1)
                if element == "Ti":
                    self.list_ti_isotopes.append(isotope)

            for isotope_ti in self.list_ti_isotopes:
                entry_results = [isotope_ti, "-", "-"]
                self.tv_results_ti_in_qz.insert("", tk.END, values=entry_results)

    def run_geothermometer_titanium_in_quartz(self):
        if self.container_var["Geothermometry"]["Titanium in Quartz"].get() == 0:
            # TitaniQ
            self.calculate_temperature_titaniq()

    def calculate_temperature_titaniq(self):
        if len(self.tv_results_ti_in_qz.get_children()) > 0:
            for item in self.tv_results_ti_in_qz.get_children():
                self.tv_results_ti_in_qz.delete(item)

        a_TiO2_Rt = float(self.container_var["a(TiO2,Rt)"].get())

        for index, str_filename_short in enumerate(self.container_lists["SMPL"]["Short"]):
            entry_results = [str_filename_short, "-", "-"]
            self.tv_results_ti_in_qz.insert("", tk.END, values=entry_results)
            for datatype in ["RAW", "SMOOTHED"]:
                for isotope_ti in self.list_ti_isotopes:
                    val_concentration_ti = round(self.container_concentration["SMPL"][datatype][str_filename_short][
                                                     "MAT"][isotope_ti], 4)
                    val_temperature_i_celsius = -3765/(np.log(val_concentration_ti/a_TiO2_Rt) - 5.69) - 273.15
                    val_temperature_i_kelvin = val_temperature_i_celsius + 273.15
                    entry_results = [isotope_ti, datatype, round(val_concentration_ti, 4),
                                     round(val_temperature_i_celsius, 2), round(val_temperature_i_kelvin, 2)]
                    self.tv_results_ti_in_qz.insert("", tk.END, values=entry_results)

    def fi_extras(self, init=False):
        """Main window of additional analysis tools for a fluid inclusion analysis project."""
        str_title_02 = self.language_dict["Extras"][self.var_language]

        if self.pysills_mode in ["FI", "INCL"]:
            str_title_01 = self.language_dict["Fluid Inclusions"][self.var_language]
        elif self.pysills_mode == "MI":
            str_title_01 = self.language_dict["Melt Inclusions"][self.var_language]

        ## Window Settings
        row_min = self.row_height
        column_min = self.column_width
        n_rows = self.window_dimensions["FI Extras"][0]
        n_columns = self.window_dimensions["FI Extras"][1]

        window_width = int(n_columns*self.column_width)
        window_height = int(n_rows*self.row_height)

        var_geometry = str(window_width) + "x" + str(window_height) + "+" + str(0) + "+" + str(0)

        self.subwindow_fi_extras = tk.Toplevel(self.parent)
        self.subwindow_fi_extras.title(str_title_01 + " - " + str_title_02)
        self.subwindow_fi_extras.geometry(var_geometry)
        self.subwindow_fi_extras.resizable(False, False)
        self.subwindow_fi_extras["bg"] = self.bg_colors["BG Window"]

        for x in range(n_columns):
            tk.Grid.columnconfigure(self.subwindow_fi_extras, x, weight=1)
        for y in range(n_rows):
            tk.Grid.rowconfigure(self.subwindow_fi_extras, y, weight=1)

        # Rows
        for i in range(0, n_rows):
            self.subwindow_fi_extras.grid_rowconfigure(i, minsize=row_min)
        # Columns
        for i in range(0, n_columns):
            self.subwindow_fi_extras.grid_columnconfigure(i, minsize=column_min)

        ## Initialization
        if init:
            for var_filetype in ["STD", "SMPL"]:
                for var_file_short in self.container_lists[var_filetype]["Short"]:
                    self.get_condensed_intervals_of_file(filetype=var_filetype, filename_short=var_file_short)

                if (self.container_var["Spike Elimination"]["STD"]["State"] == False and
                        self.container_var["Spike Elimination"]["SMPL"]["State"] == False):
                    list_datatype = ["RAW"]
                else:
                    list_datatype = ["RAW", "SMOOTHED"]

            var_filetype = "None"
            var_file_short = "None"
            var_file_long = "None"
            var_focus = "None"

            for var_datatype in list_datatype:
                if len(self.container_spikes) > 0:
                    # Intensity Results
                    self.get_intensity(
                        var_filetype=var_filetype, var_datatype=var_datatype, var_file_short=var_file_short,
                        var_focus=var_focus, mode="All")
                    self.fi_get_intensity_corrected(
                        var_filetype=var_filetype, var_datatype=var_datatype, var_file_short=var_file_short,
                        var_focus=var_focus, mode="All")
                    self.fi_get_intensity_mix(
                        var_filetype=var_filetype, var_datatype=var_datatype, var_file_short=var_file_short, mode="All")
                    # Sensitivity Results
                    self.get_analytical_sensitivity(
                        var_filetype=var_filetype, var_datatype=var_datatype, var_file_short=var_file_short,
                        var_file_long=var_file_long, mode="All")
                    results_is = self.determine_possible_is(filetype="ALL")
                    IQ(dataframe=None, project_type=self.pysills_mode,
                       results_container=self.container_intensity_ratio).get_intensity_ratio(
                        data_container=self.container_intensity_corrected, dict_is=results_is, datatype=var_datatype)
                    self.fi_get_rsf(
                        var_filetype=var_filetype, var_datatype=var_datatype, var_file_short=var_file_short,
                        var_file_long=var_file_long, var_focus=var_focus, mode="All")
                    # Concentration Results
                    self.fi_get_concentration2(
                        var_filetype=var_filetype, var_datatype=var_datatype, var_file_short=var_file_short,
                        var_file_long=var_file_long, var_focus=var_focus, mode="All")
                    self.fi_get_normalized_sensitivity(
                        var_filetype=var_filetype, var_datatype=var_datatype, var_file_short=var_file_short,
                        var_file_long=var_file_long, var_focus=var_focus, mode="All")
                    self.fi_get_concentration_ratio(
                        var_filetype=var_filetype, var_datatype=var_datatype, var_file_short=var_file_short,
                        var_file_long=var_file_long, var_focus=var_focus, mode="All")
                    self.get_lod(
                        var_filetype=var_filetype, var_datatype=var_datatype, var_file_short=var_file_short,
                        var_file_long=var_file_long, var_focus=None, mode="All")
                    self.fi_get_mixed_concentration_ratio(
                        var_datatype=var_datatype, var_file_short=var_file_short, var_file_long=var_file_long,
                        mode="All")
                    self.fi_get_mixing_ratio(
                        var_datatype=var_datatype, var_file_short=var_file_short, var_file_long=var_file_long,
                        mode="All")
                    self.fi_get_concentration_mixed(var_datatype=var_datatype, var_file_short=var_file_short,
                                                    mode="All")

        var_row_start = 0
        var_column_start = 0
        var_header_n = 16
        int_category_n = 12

        ## LABELS
        str_lbl_01 = self.language_dict["Matrix"][self.var_language]
        str_lbl_01b = self.language_dict["Inclusion"][self.var_language]
        str_lbl_02 = self.language_dict["Further data analysis"][self.var_language]
        str_lbl_03 = self.language_dict["x-y diagram"][self.var_language]
        str_lbl_04 = self.language_dict["Ternary diagram"][self.var_language]
        str_lbl_06 = self.language_dict["elements"][self.var_language]
        str_lbl_07 = self.language_dict["element ratios"][self.var_language]
        str_lbl_08 = self.language_dict["oxides"][self.var_language]
        str_lbl_09 = self.language_dict["oxide ratios"][self.var_language]

        lbl_01 = SE(
            parent=self.subwindow_fi_extras, row_id=var_row_start, column_id=var_column_start, n_rows=1,
            n_columns=var_header_n, fg=self.bg_colors["Light Font"],
            bg=self.bg_colors["BG Window"]).create_simple_label(
            text=str_lbl_01 + " - " + str_lbl_02, relief=tk.FLAT, fontsize="sans 10 bold", anchor=tk.W)
        lbl_02 = SE(
            parent=self.subwindow_fi_extras, row_id=var_row_start, column_id=var_header_n + 1, n_rows=1,
            n_columns=var_header_n, fg=self.bg_colors["Light Font"],
            bg=self.bg_colors["BG Window"]).create_simple_label(
            text=str_lbl_01b + " - " + str_lbl_02, relief=tk.FLAT, fontsize="sans 10 bold", anchor=tk.W)
        lbl_001a = SE(
            parent=self.subwindow_fi_extras, row_id=var_row_start + 1, column_id=var_column_start, n_rows=1,
            n_columns=int_category_n, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_simple_label(
            text=str_lbl_03 + " (" + str_lbl_06 + ")", relief=tk.FLAT, fontsize="sans 10 bold", anchor=tk.W)
        lbl_001b = SE(
            parent=self.subwindow_fi_extras, row_id=var_row_start + 2, column_id=var_column_start, n_rows=1,
            n_columns=int_category_n, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_simple_label(
            text=str_lbl_03 + " (" + str_lbl_07 + ")", relief=tk.FLAT, fontsize="sans 10 bold", anchor=tk.W)
        lbl_001c = SE(
            parent=self.subwindow_fi_extras, row_id=var_row_start + 3, column_id=var_column_start, n_rows=1,
            n_columns=int_category_n, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_simple_label(
            text=str_lbl_03 + " (" + str_lbl_08 + ")", relief=tk.FLAT, fontsize="sans 10 bold", anchor=tk.W)
        lbl_001d = SE(
            parent=self.subwindow_fi_extras, row_id=var_row_start + 4, column_id=var_column_start, n_rows=1,
            n_columns=int_category_n, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_simple_label(
            text=str_lbl_03 + " (" + str_lbl_09 + ")", relief=tk.FLAT, fontsize="sans 10 bold", anchor=tk.W)
        lbl_001e = SE(
            parent=self.subwindow_fi_extras, row_id=var_row_start + 5, column_id=var_column_start, n_rows=1,
            n_columns=int_category_n, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_simple_label(
            text=str_lbl_04 + " (" + str_lbl_06 + ")", relief=tk.FLAT, fontsize="sans 10 bold", anchor=tk.W)
        lbl_001f = SE(
            parent=self.subwindow_fi_extras, row_id=var_row_start + 6, column_id=var_column_start, n_rows=1,
            n_columns=int_category_n, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_simple_label(
            text=str_lbl_04 + " (" + str_lbl_08 + ")", relief=tk.FLAT, fontsize="sans 10 bold", anchor=tk.W)
        lbl_002a = SE(
            parent=self.subwindow_fi_extras, row_id=var_row_start + 1, column_id=var_header_n + 1, n_rows=1,
            n_columns=int_category_n, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_simple_label(
            text=str_lbl_03 + " (" + str_lbl_06 + ")", relief=tk.FLAT, fontsize="sans 10 bold", anchor=tk.W)
        lbl_002b = SE(
            parent=self.subwindow_fi_extras, row_id=var_row_start + 2, column_id=var_header_n + 1, n_rows=1,
            n_columns=int_category_n, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_simple_label(
            text=str_lbl_03 + " (" + str_lbl_07 + ")", relief=tk.FLAT, fontsize="sans 10 bold", anchor=tk.W)
        lbl_002c = SE(
            parent=self.subwindow_fi_extras, row_id=var_row_start + 3, column_id=var_header_n + 1, n_rows=1,
            n_columns=int_category_n, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_simple_label(
            text=str_lbl_03 + " (" + str_lbl_08 + ")", relief=tk.FLAT, fontsize="sans 10 bold", anchor=tk.W)
        lbl_002d = SE(
            parent=self.subwindow_fi_extras, row_id=var_row_start + 4, column_id=var_header_n + 1, n_rows=1,
            n_columns=int_category_n, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_simple_label(
            text=str_lbl_03 + " (" + str_lbl_09 + ")", relief=tk.FLAT, fontsize="sans 10 bold", anchor=tk.W)
        lbl_002e = SE(
            parent=self.subwindow_fi_extras, row_id=var_row_start + 5, column_id=var_header_n + 1, n_rows=1,
            n_columns=int_category_n, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_simple_label(
            text=str_lbl_04 + " (" + str_lbl_06 + ")", relief=tk.FLAT, fontsize="sans 10 bold", anchor=tk.W)
        lbl_002f = SE(
            parent=self.subwindow_fi_extras, row_id=var_row_start + 6, column_id=var_header_n + 1, n_rows=1,
            n_columns=int_category_n, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_simple_label(
            text=str_lbl_04 + " (" + str_lbl_08 + ")", relief=tk.FLAT, fontsize="sans 10 bold", anchor=tk.W)
        lbl_002g = SE(
            parent=self.subwindow_fi_extras, row_id=var_row_start + 7, column_id=var_header_n + 1, n_rows=1,
            n_columns=int_category_n, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_simple_label(
            text="Halogen ratios", relief=tk.FLAT, fontsize="sans 10 bold", anchor=tk.W)

        # BUTTONS
        str_btn_01 = self.language_dict["Setup"][self.var_language]

        btn_001a = SE(
            parent=self.subwindow_fi_extras, row_id=var_row_start + 1, column_id=int_category_n,
            n_rows=1, n_columns=var_header_n - int_category_n, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_simple_button(
            text=str_btn_01, bg_active=self.accent_color, fg_active=self.bg_colors["Dark Font"],
            command=lambda mode="Elements": self.diagram_xy(mode))
        btn_001b = SE(
            parent=self.subwindow_fi_extras, row_id=var_row_start + 2, column_id=int_category_n,
            n_rows=1, n_columns=var_header_n - int_category_n, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_simple_button(
            text=str_btn_01, bg_active=self.accent_color, fg_active=self.bg_colors["Dark Font"],
            command=lambda mode="Element ratios": self.diagram_xy(mode))
        btn_001c = SE(
            parent=self.subwindow_fi_extras, row_id=var_row_start + 3, column_id=int_category_n,
            n_rows=1, n_columns=var_header_n - int_category_n, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_simple_button(
            text=str_btn_01, bg_active=self.accent_color, fg_active=self.bg_colors["Dark Font"],
            command=lambda mode="Oxides": self.diagram_xy(mode))
        btn_001d = SE(
            parent=self.subwindow_fi_extras, row_id=var_row_start + 4, column_id=int_category_n,
            n_rows=1, n_columns=var_header_n - int_category_n, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_simple_button(
            text=str_btn_01, bg_active=self.accent_color, fg_active=self.bg_colors["Dark Font"],
            command=lambda mode="Oxide ratios": self.diagram_xy(mode))
        btn_001e = SE(
            parent=self.subwindow_fi_extras, row_id=var_row_start + 5, column_id=int_category_n,
            n_rows=1, n_columns=var_header_n - int_category_n, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_simple_button(
            text=str_btn_01, bg_active=self.accent_color, fg_active=self.bg_colors["Dark Font"])
        btn_001f = SE(
            parent=self.subwindow_fi_extras, row_id=var_row_start + 6, column_id=int_category_n,
            n_rows=1, n_columns=var_header_n - int_category_n, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_simple_button(
            text=str_btn_01, bg_active=self.accent_color, fg_active=self.bg_colors["Dark Font"])

        if len(self.container_lists["Selected Oxides"]["All"]) == 0:
            btn_001c.configure(state="disabled")
            btn_001d.configure(state="disabled")

        btn_001e.configure(state="disabled")
        btn_001f.configure(state="disabled")

        btn_002a = SE(
            parent=self.subwindow_fi_extras, row_id=var_row_start + 1, column_id=var_header_n + int_category_n + 1,
            n_rows=1, n_columns=var_header_n - int_category_n, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_simple_button(
            text=str_btn_01, bg_active=self.accent_color, fg_active=self.bg_colors["Dark Font"],
            command=lambda mode="Elements", focus="INCL": self.diagram_xy(mode, focus))
        btn_002b = SE(
            parent=self.subwindow_fi_extras, row_id=var_row_start + 2, column_id=var_header_n + int_category_n + 1,
            n_rows=1, n_columns=var_header_n - int_category_n, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_simple_button(
            text=str_btn_01, bg_active=self.accent_color, fg_active=self.bg_colors["Dark Font"],
            command=lambda mode="Element ratios", focus="INCL": self.diagram_xy(mode, focus))
        btn_002c = SE(
            parent=self.subwindow_fi_extras, row_id=var_row_start + 3, column_id=var_header_n + int_category_n + 1,
            n_rows=1, n_columns=var_header_n - int_category_n, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_simple_button(
            text=str_btn_01, bg_active=self.accent_color, fg_active=self.bg_colors["Dark Font"],
            command=lambda mode="Oxides", focus="INCL": self.diagram_xy(mode, focus))
        btn_002d = SE(
            parent=self.subwindow_fi_extras, row_id=var_row_start + 4, column_id=var_header_n + int_category_n + 1,
            n_rows=1, n_columns=var_header_n - int_category_n, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_simple_button(
            text=str_btn_01, bg_active=self.accent_color, fg_active=self.bg_colors["Dark Font"],
            command=lambda mode="Oxide ratios", focus="INCL": self.diagram_xy(mode, focus))
        btn_002e = SE(
            parent=self.subwindow_fi_extras, row_id=var_row_start + 5, column_id=var_header_n + int_category_n + 1,
            n_rows=1, n_columns=var_header_n - int_category_n, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_simple_button(
            text=str_btn_01, bg_active=self.accent_color, fg_active=self.bg_colors["Dark Font"])
        btn_002f = SE(
            parent=self.subwindow_fi_extras, row_id=var_row_start + 6, column_id=var_header_n + int_category_n + 1,
            n_rows=1, n_columns=var_header_n - int_category_n, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_simple_button(
            text=str_btn_01, bg_active=self.accent_color, fg_active=self.bg_colors["Dark Font"])
        btn_002g = SE(
            parent=self.subwindow_fi_extras, row_id=var_row_start + 7, column_id=var_header_n + int_category_n + 1,
            n_rows=1, n_columns=var_header_n - int_category_n, fg=self.bg_colors["Dark Font"],
            bg=self.bg_colors["Light"]).create_simple_button(
            text=str_btn_01, bg_active=self.accent_color, fg_active=self.bg_colors["Dark Font"],
            command=lambda init=True: self.halogen_ratios_diagram(init))

        if len(self.container_lists["Selected Oxides"]["All"]) == 0:
            btn_002c.configure(state="disabled")
            btn_002d.configure(state="disabled")

        btn_002e.configure(state="disabled")
        btn_002f.configure(state="disabled")

    def mi_extras(self, init=False):
        """Main window of additional analysis tools for a melt inclusion analysis project."""
        ## Window Settings
        row_min = self.row_height
        column_min = self.column_width
        n_rows = self.window_dimensions["MI Extras"][0]
        n_columns = self.window_dimensions["MI Extras"][1]

        window_width = int(n_columns*self.column_width)
        window_height = int(n_rows*self.row_height)

        var_geometry = str(window_width) + "x" + str(window_height) + "+" + str(0) + "+" + str(0)

        self.subwindow_mi_extras = tk.Toplevel(self.parent)
        self.subwindow_mi_extras.title("MELT INCLUSION ANALYSIS - Extras")
        self.subwindow_mi_extras.geometry(var_geometry)
        self.subwindow_mi_extras.resizable(False, False)
        self.subwindow_mi_extras["bg"] = self.bg_colors["BG Window"]

        for x in range(n_columns):
            tk.Grid.columnconfigure(self.subwindow_mi_extras, x, weight=1)
        for y in range(n_rows):
            tk.Grid.rowconfigure(self.subwindow_mi_extras, y, weight=1)

        # Rows
        for i in range(0, n_rows):
            self.subwindow_mi_extras.grid_rowconfigure(i, minsize=row_min)
        # Columns
        for i in range(0, n_columns):
            self.subwindow_mi_extras.grid_columnconfigure(i, minsize=column_min)

        ## Initialization
        if init:
            for var_filetype in ["STD", "SMPL"]:
                for var_file_short in self.container_lists[var_filetype]["Short"]:
                    self.get_condensed_intervals_of_file(filetype=var_filetype, filename_short=var_file_short)

                if (self.container_var["Spike Elimination"]["STD"]["State"] == False and
                        self.container_var["Spike Elimination"]["SMPL"]["State"] == False):
                    list_datatype = ["RAW"]
                else:
                    list_datatype = ["RAW", "SMOOTHED"]

            var_filetype = "None"
            var_file_short = "None"
            var_file_long = "None"
            var_focus = "None"

            for var_datatype in list_datatype:
                if len(self.container_spikes) > 0:
                    # Intensity Results
                    self.get_intensity(
                        var_filetype=var_filetype, var_datatype=var_datatype, var_file_short=var_file_short,
                        var_focus=var_focus, mode="All")
                    self.fi_get_intensity_corrected(
                        var_filetype=var_filetype, var_datatype=var_datatype, var_file_short=var_file_short,
                        var_focus=var_focus, mode="All")
                    self.fi_get_intensity_mix(
                        var_filetype=var_filetype, var_datatype=var_datatype, var_file_short=var_file_short, mode="All")
                    # Sensitivity Results
                    self.get_analytical_sensitivity(
                        var_filetype=var_filetype, var_datatype=var_datatype, var_file_short=var_file_short,
                        var_file_long=var_file_long, mode="All")
                    results_is = self.determine_possible_is(filetype="ALL")
                    IQ(dataframe=None, project_type=self.pysills_mode,
                       results_container=self.container_intensity_ratio).get_intensity_ratio(
                        data_container=self.container_intensity_corrected, dict_is=results_is, datatype=var_datatype)
                    self.fi_get_rsf(
                        var_filetype=var_filetype, var_datatype=var_datatype, var_file_short=var_file_short,
                        var_file_long=var_file_long, var_focus=var_focus, mode="All")
                    # Concentration Results
                    self.fi_get_concentration2(
                        var_filetype=var_filetype, var_datatype=var_datatype, var_file_short=var_file_short,
                        var_file_long=var_file_long, var_focus=var_foc