"""This file contains metadata describing the results from Gaussian
"""

metadata = {}

methods = {
    "DFT: Kohn-Sham density functional theory": {
        "method": "DFT",
        "calculation": ["energy", "gradients"],
        "level": "normal",
        "gradients": "analytic",
    },
    "HF: Hartree-Fock self consistent field (SCF)": {
        "method": "HF",
        "calculation": ["energy", "gradients"],
        "level": "normal",
        "gradients": "analytic",
    },
    "MP2: 2nd-order Møller–Plesset perturbation theory": {
        "method": "MP2",
        "calculation": ["energy", "gradients"],
        "level": "normal",
        "gradients": "analytic",
        "freeze core?": True,
    },
    "MP3: 3rd-order Møller–Plesset perturbation theory": {
        "method": "MP3",
        "calculation": ["energy", "gradients"],
        "level": "normal",
        "gradients": "analytic",
        "freeze core?": True,
    },
    "MP4(SDTQ): 4th-order Møller–Plesset perturbation theory": {
        "method": "MP4",
        "calculation": ["energy", "gradients"],
        "level": "normal",
        "gradients": "numeric",
        "freeze core?": True,
    },
    "MP4(SDQ): 4th-order Møller–Plesset perturbation theory": {
        "method": "MP4(SDQ)",
        "calculation": ["energy", "gradients"],
        "level": "normal",
        "gradients": "analytic",
        "freeze core?": True,
    },
    "MP5: 5th-order Møller–Plesset perturbation theory": {
        "method": "MP5",
        "calculation": ["energy", "gradients"],
        "level": "advanced",
        "gradients": "analytic",
        "freeze core?": True,
    },
    "CCD: coupled cluster doubles": {
        "method": "CCD",
        "calculation": ["energy", "gradients"],
        "level": "normal",
        "gradients": "analytic",
        "freeze core?": True,
    },
    "CCSD: coupled cluster singles & doubles": {
        "method": "CCSD",
        "calculation": ["energy", "gradients"],
        "level": "normal",
        "gradients": "numeric",
        "freeze core?": True,
    },
    "CCSD(T): coupled cluster singles & doubles plus triples": {
        "method": "CCSD(T)",
        "calculation": ["energy", "gradients"],
        "level": "normal",
        "gradients": "numeric",
        "freeze core?": True,
    },
    "CBS-4M: Complete Basis Set method of Petersson, et al.": {
        "method": "CBS-4M",
        "calculation": ["energy", "gradients"],
        "level": "advanced",
        "gradients": "none",
        "nobasis": True,
    },
    "CBS-QB3: Complete Basis Set method of Petersson, et al.": {
        "method": "CBS-QB3",
        "calculation": ["energy", "gradients"],
        "level": "normal",
        "gradients": "none",
        "nobasis": True,
    },
    "CBS-APNO: Complete Basis Set method of Petersson, et al.": {
        "method": "CBS-APNO",
        "calculation": ["energy", "gradients"],
        "level": "advanced",
        "gradients": "none",
        "nobasis": True,
    },
    "G1: Gaussian-1 composite method": {
        "method": "G1",
        "calculation": ["energy", "gradients"],
        "level": "advanced",
        "gradients": "none",
        "nobasis": True,
    },
    "G2: Gaussian-2 composite method": {
        "method": "G2",
        "calculation": ["energy", "gradients"],
        "level": "advanced",
        "gradients": "none",
        "nobasis": True,
    },
    "G2MP2: Gaussian-2 MP2 composite method": {
        "method": "G2MP2",
        "calculation": ["energy", "gradients"],
        "level": "advanced",
        "gradients": "none",
        "nobasis": True,
    },
    "G3: Gaussian-3 composite method": {
        "method": "G3",
        "calculation": ["energy", "gradients"],
        "level": "advanced",
        "gradients": "none",
        "nobasis": True,
    },
    "G3MP2: Gaussian-3 MP2 composite method": {
        "method": "G3MP2",
        "calculation": ["energy", "gradients"],
        "level": "advanced",
        "gradients": "none",
        "nobasis": True,
    },
    "G3B3: Gaussian-3 B3LYP composite method": {
        "method": "G3B3",
        "calculation": ["energy", "gradients"],
        "level": "advanced",
        "gradients": "none",
        "nobasis": True,
    },
    "G3MP2B3: Gaussian-3 MP2 B3LYP composite method": {
        "method": "G3MP2B3",
        "calculation": ["energy", "gradients"],
        "level": "advanced",
        "gradients": "none",
        "nobasis": True,
    },
    "G4: Gaussian-4 composite method": {
        "method": "G4",
        "calculation": ["energy", "gradients"],
        "level": "advanced",
        "gradients": "none",
        "nobasis": True,
    },
    "G4MP2: Gaussian-4 MP2 composite method": {
        "method": "G4MP2",
        "calculation": ["energy", "gradients"],
        "level": "advanced",
        "gradients": "none",
        "nobasis": True,
    },
}

dft_functionals = {
    "PBE : GGA of Perdew, Burke and Ernzerhof": {
        "name": "PBEPBE",
        "dispersion": ["none", "GD3BJ", "GD3", "GD2", "PFD"],
        "level": "normal",
    },
    "PBE-98 : GGA of Perdew, Burke and Ernzerhof, revised": {
        "name": "PBEhPBE",
        "dispersion": ["none", "PFD"],
        "level": "normal",
    },
    "B3LYP : hybrid functional of Becke and Lee, Yang, and Parr": {
        "name": "B3LYP",
        "dispersion": ["none", "GD3BJ", "GD3", "GD2", "PFD"],
        "level": "normal",
    },
    "HSE06 : hybrid functional of Heyd, Scuseria and Ernzerhof": {
        "name": "HSEH1PBE",
        "dispersion": ["none", "PFD"],
        "level": "normal",
    },
    "PBE0 : hybrid functional of Perdew, Burke and Ernzerhof, and Adamo": {
        "name": "PBE1PBE",
        "dispersion": ["none", "GD3BJ", "GD3", "PFD"],
        "level": "normal",
    },
}

optimization_convergence = {
    "default": "",
    "tight": "Tight",
    "very tight": "VeryTight",
    "loose": "Loose",
}

"""Properties that Gaussian produces.
`metadata["results"]` describes the results that this step can produce. It is a
dictionary where the keys are the internal names of the results within this step, and
the values are a dictionary describing the result. For example::

    metadata["results"] = {
        "total_energy": {
            "calculation": [
                "energy",
                "optimization",
            ],
            "description": "The total energy",
            "dimensionality": "scalar",
            "methods": [
                "ccsd",
                "ccsd(t)",
                "dft",
                "hf",
            ],
            "property": "total energy#QuickMin#{model}",
            "type": "float",
            "units": "E_h",
        },
    }

Fields
______

calculation : [str]
    Optional metadata describing what subtype of the step produces this result.
    The subtypes are completely arbitrary, but often they are types of calculations
    which is why this is name `calculation`. To use this, the step or a substep
    define `self._calculation` as a value. That value is used to select only the
    results with that value in this field.

description : str
    A human-readable description of the result.

dimensionality : str
    The dimensions of the data. The value can be "scalar" or an array definition
    of the form "[dim1, dim2,...]". Symmetric tringular matrices are denoted
    "triangular[n,n]". The dimensions can be integers, other scalar
    results, or standard parameters such as `n_atoms`. For example, '[3]',
    [3, n_atoms], or "triangular[n_aos, n_aos]".

methods : str
    Optional metadata like the `calculation` data. `methods` provides a second
    level of filtering, often used for the Hamiltionian for *ab initio* calculations
    where some properties may or may not be calculated depending on the type of
    theory.

property : str
    An optional definition of the property for storing this result. Must be one of
    the standard properties defined either in SEAMM or in this steps property
    metadata in `data/properties.csv`.

type : str
    The type of the data: string, integer, or float.

units : str
    Optional units for the result. If present, the value should be in these units.
"""
composite = (
    "CBS-4M",
    "CBS-QB3",
    "CBS-APNO",
    "G1",
    "G2",
    "G3",
    "G4",
    "G2MP2",
    "G3B3",
    "G3MP2",
    "G3MP2B3",
    "G4MP2",
)
metadata["results"] = {
    "energy": {
        "description": "energy",
        "dimensionality": "scalar",
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "gradients": {
        "calculation": ["energy", "optimization", "thermodynamics", "vibrations"],
        "description": "gradients",
        "dimensionality": [3, "natoms"],
        "type": "float",
        "units": "E_h/a0",
        "format": ".6f",
    },
    "nsteps": {
        "calculation": ["optimization"],
        "description": "optimization nsteps",
        "dimensionality": "scalar",
        "type": "integer",
    },
    "model": {
        "description": "The model string",
        "dimensionality": "scalar",
        "type": "string",
    },
    "Virial Ratio": {
        "calculation": ["energy", "optimization", "thermodynamics", "vibrations"],
        "description": "the virial ratio",
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "float",
        "format": ".4f",
    },
    "Total Energy": {
        "calculation": ["energy", "optimization", "thermodynamics", "vibrations"],
        "description": "total energy including all terms",
        "dimensionality": "scalar",
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "Cartesian Gradient": {
        "calculation": ["energy", "optimization", "thermodynamics", "vibrations"],
        "description": "the cartesian gradients",
        "dimensionality": ["natoms", 3],
        "type": "float",
        "units": "E_h/a0",
        "format": ".6f",
    },
    "RMS Density": {
        "calculation": ["energy", "optimization", "thermodynamics", "vibrations"],
        "description": "the RMS density difference in the SCF",
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "float",
        "format": ".2e",
    },
    "Geometry Optimization Converged": {
        "calculation": ["optimization"],
        "description": "whether the geometry optimzation converged",
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "boolean",
        "format": "s",
    },
    "RMS Force": {
        "calculation": ["optimization"],
        "description": "the RMS force on the atoms",
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "float",
        "units": "E_h/Å",
        "format": ".6f",
    },
    "Maximum Force": {
        "calculation": ["optimization"],
        "description": "the maximum force on an atom",
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "float",
        "units": "E_h/Å",
        "format": ".6f",
    },
    "RMS Displacement": {
        "calculation": ["optimization"],
        "description": "the RMS displacement of the atoms",
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "float",
        "units": "Å",
        "format": ".6f",
    },
    "Maximum Displacement": {
        "calculation": ["optimization"],
        "description": "the maximum displacement of an atom",
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "float",
        "units": "Å",
        "format": ".6f",
    },
    "RMS Force Threshold": {
        "calculation": ["optimization"],
        "description": "the RMS force threshold for the atoms",
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "float",
        "units": "E_h/Å",
        "format": ".6f",
    },
    "Maximum Force Threshold": {
        "calculation": ["optimization"],
        "description": "the maximum force threshold for an atom",
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "float",
        "units": "E_h/Å",
        "format": ".6f",
    },
    "RMS Displacement Threshold": {
        "calculation": ["optimization"],
        "description": "the RMS displacement threshold for the atoms",
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "float",
        "units": "Å",
        "format": ".6f",
    },
    "Maximum Displacement Threshold": {
        "calculation": ["optimization"],
        "description": "the maximum displacement threshold for an atom",
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "float",
        "units": "Å",
        "format": ".6f",
    },
    "Gaussian Version": {
        "description": "the version of Gaussian used",
        "dimensionality": "scalar",
        "type": "string",
        "format": "s",
    },
    "E(gap)": {
        "calculation": ["energy", "optimization"],
        "description": (
            "The gap between the LUMO and HOMO or conduction and valence bands"
        ),
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "float",
        "units": "eV",
        "format": ".3f",
    },
    "E(homo)": {
        "calculation": ["energy", "optimization"],
        "description": "The energy of highest occupied MO (HOMO)",
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "float",
        "units": "eV",
        "format": ".3f",
    },
    "E(homo-1)": {
        "calculation": ["energy", "optimization"],
        "description": "The energy of the next-to-highest occupied orbital (NHOMO)",
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "float",
        "units": "eV",
        "format": ".3f",
    },
    "E(lumo)": {
        "calculation": ["energy", "optimization"],
        "description": "The energy of lowest unoccupied MO (LUMO)",
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "float",
        "units": "eV",
        "format": ".3f",
    },
    "E(lumo+1)": {
        "calculation": ["energy", "optimization"],
        "description": "The energy of the second lowest unoccupied orbital (SLUMO)",
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "float",
        "units": "eV",
        "format": ".3f",
    },
    "N(homo)": {
        "calculation": ["energy", "optimization"],
        "description": "The orbital number of the HOMO (starting at 1)",
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "integer",
        "units": "",
        "format": "",
    },
    "Sym(homo)": {
        "calculation": ["symmetry", "optimization"],
        "description": "The symmetry of the highest occupied MO (HOMO)",
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "string",
        "units": "",
        "format": "",
    },
    "Sym(homo-1)": {
        "calculation": ["symmetry", "optimization"],
        "description": "The symmetry of the next-to-highest occupied orbital (NHOMO)",
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "string",
        "units": "",
        "format": "",
    },
    "Sym(lumo)": {
        "calculation": ["symmetry", "optimization"],
        "description": "The symmetry of the lowest unoccupied MO (LUMO)",
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "string",
        "units": "",
        "format": "",
    },
    "Sym(lumo+1)": {
        "calculation": ["symmetry", "optimization"],
        "description": "The symmetry of the second lowest unoccupied orbital (SLUMO)",
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "string",
        "units": "",
        "format": "",
    },
    "E(α-gap)": {
        "calculation": ["energy", "optimization"],
        "description": (
            "The gap between the LUMO and HOMO or conduction and valence bands"
        ),
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "float",
        "units": "eV",
        "format": ".3f",
    },
    "E(α-homo)": {
        "calculation": ["energy", "optimization"],
        "description": "The energy of highest occupied MO (α-HOMO)",
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "float",
        "units": "eV",
        "format": ".3f",
    },
    "E(α-homo-1)": {
        "calculation": ["energy", "optimization"],
        "description": "The energy of the next-to-highest occupied orbital (α-NHOMO)",
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "float",
        "units": "eV",
        "format": ".3f",
    },
    "E(α-lumo)": {
        "calculation": ["energy", "optimization"],
        "description": "The energy of lowest unoccupied MO (α-LUMO)",
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "float",
        "units": "eV",
        "format": ".3f",
    },
    "E(α-lumo+1)": {
        "calculation": ["energy", "optimization"],
        "description": "The energy of the second lowest unoccupied orbital (α-SLUMO)",
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "float",
        "units": "eV",
        "format": ".3f",
    },
    "N(α-homo)": {
        "calculation": ["energy", "optimization"],
        "description": "The orbital number of the HOMO (starting at 1)",
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "integer",
        "units": "",
        "format": "",
    },
    "Sym(α-homo)": {
        "calculation": ["symmetry", "optimization"],
        "description": "The symmetry of the highest occupied MO (α-HOMO)",
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "string",
        "units": "",
        "format": "",
    },
    "Sym(α-homo-1)": {
        "calculation": ["symmetry", "optimization"],
        "description": "The symmetry of the next-to-highest occupied orbital (α-NHOMO)",
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "string",
        "units": "",
        "format": "",
    },
    "Sym(α-lumo)": {
        "calculation": ["symmetry", "optimization"],
        "description": "The symmetry of the lowest unoccupied MO (α-LUMO)",
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "string",
        "units": "",
        "format": "",
    },
    "Sym(α-lumo+1)": {
        "calculation": ["symmetry", "optimization"],
        "description": "The symmetry of the second lowest unoccupied orbital (α-SLUMO)",
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "string",
        "units": "",
        "format": "",
    },
    "E(β-gap)": {
        "calculation": ["energy", "optimization"],
        "description": (
            "The gap between the LUMO and HOMO or conduction and valence bands"
        ),
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "float",
        "units": "eV",
        "format": ".3f",
    },
    "E(β-homo)": {
        "calculation": ["energy", "optimization"],
        "description": "The energy of highest occupied MO (β-HOMO)",
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "float",
        "units": "eV",
        "format": ".3f",
    },
    "E(β-homo-1)": {
        "calculation": ["energy", "optimization"],
        "description": "The energy of the next-to-highest occupied orbital (β-NHOMO)",
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "float",
        "units": "eV",
        "format": ".3f",
    },
    "E(β-lumo)": {
        "calculation": ["energy", "optimization"],
        "description": "The energy of lowest unoccupied MO (β-LUMO)",
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "float",
        "units": "eV",
        "format": ".3f",
    },
    "E(β-lumo+1)": {
        "calculation": ["energy", "optimization"],
        "description": "The energy of the second lowest unoccupied orbital (β-SLUMO)",
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "float",
        "units": "eV",
        "format": ".3f",
    },
    "N(β-homo)": {
        "calculation": ["energy", "optimization"],
        "description": "The orbital number of the HOMO (starting at 1)",
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "integer",
        "units": "",
        "format": "",
    },
    "Sym(β-homo)": {
        "calculation": ["symmetry", "optimization"],
        "description": "The symmetry of the highest occupied MO (β-HOMO)",
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "string",
        "units": "",
        "format": "",
    },
    "Sym(β-homo-1)": {
        "calculation": ["symmetry", "optimization"],
        "description": "The symmetry of the next-to-highest occupied orbital (β-NHOMO)",
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "string",
        "units": "",
        "format": "",
    },
    "Sym(β-lumo)": {
        "calculation": ["symmetry", "optimization"],
        "description": "The symmetry of the lowest unoccupied MO (β-LUMO)",
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "string",
        "units": "",
        "format": "",
    },
    "Sym(β-lumo+1)": {
        "calculation": ["symmetry", "optimization"],
        "description": "The symmetry of the second lowest unoccupied orbital (β-SLUMO)",
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "string",
        "units": "",
        "format": "",
    },
    "atomcharges/mulliken": {
        "calculation": ["energy", "optimization"],
        "description": "The Mulliken charges of the atoms",
        "dimensionality": "[natoms]",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "float",
        "units": "",
        "format": ".6f",
    },
    "coreelectrons": {
        "calculation": ["energy", "optimization"],
        "description": "The number of core electrons per atom",
        "dimensionality": "[natoms]",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "integer",
        "units": "",
        "format": "",
    },
    "metadata/basis_set": {
        "description": "The basis set used in the calculation",
        "dimensionality": "scalar",
        "type": "string",
        "units": "",
        "format": "",
    },
    "metadata/cpu_time": {
        "description": "The cpu time for the calculation",
        "dimensionality": "scalar",
        "type": "string",
        "units": "",
        "format": "",
    },
    "metadata/functional": {
        "description": "The density functional",
        "dimensionality": "scalar",
        "methods": ["DFT"],
        "type": "string",
        "units": "",
        "format": "",
    },
    "metadata/legacy_package_version": {
        "description": "The version of the Gaussian code (XXrevisionYYY)",
        "dimensionality": "scalar",
        "type": "string",
        "units": "",
        "format": "",
    },
    "metadata/methods": {
        "description": (
            "The method used in each optimization step or step of the calculation"
        ),
        "dimensionality": "[nsteps]",
        "type": "string",
        "units": "",
        "format": "",
    },
    "metadata/package": {
        "description": "The name of the code (Gaussian)",
        "dimensionality": "scalar",
        "type": "string",
        "units": "",
        "format": "",
    },
    "metadata/package_version": {
        "description": "The version of the Gaussian code (year+revision)",
        "dimensionality": "scalar",
        "type": "string",
        "units": "",
        "format": "",
    },
    "metadata/platform": {
        "description": "The platform used for the calculation",
        "dimensionality": "scalar",
        "type": "string",
        "units": "",
        "format": "",
    },
    "metadata/success": {
        "description": "Whether the calculation was successful",
        "dimensionality": "scalar",
        "type": "boolean",
        "units": "",
        "format": "",
    },
    "metadata/symmetry_detected": {
        "description": "The symmetry of the system",
        "dimensionality": "scalar",
        "type": "string",
        "units": "",
        "format": "",
    },
    "metadata/symmetry_used": {
        "description": "The symmetry used in the calculation",
        "dimensionality": "scalar",
        "type": "string",
        "units": "",
        "format": "",
    },
    "metadata/wall_time": {
        "description": "The wall-clock time for the calculation",
        "dimensionality": "scalar",
        "type": "string",
        "units": "",
        "format": "",
    },
    "moenergies": {
        "calculation": ["energy", "optimization"],
        "description": "The energy of the molecular orbitals (MOs)",
        "dimensionality": "[[nmo]*nspin]",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "float",
        "units": "eV",
        "format": ".3f",
    },
    "moments": {
        "calculation": ["energy", "optimization"],
        "description": "",
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "float",
        "units": "",
        "format": ".6f",
    },
    "multipole_reference": {
        "calculation": ["energy", "optimization"],
        "description": "The reference point for the multipole expansions",
        "dimensionality": "[3]",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "float",
        "units": "Å",
        "format": ".6f",
    },
    "dipole_moment": {
        "calculation": ["energy", "optimization"],
        "description": "The dipole moment",
        "dimensionality": "[3]",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "float",
        "units": "debye",
        "format": ".6f",
    },
    "dipole_moment_magnitude": {
        "calculation": ["energy", "optimization"],
        "description": "The magnitude of the dipole moment",
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "float",
        "units": "debye",
        "format": ".6f",
    },
    "quadrupole_moment": {
        "calculation": ["energy", "optimization"],
        "description": "The quadrupole moment",
        "dimensionality": "[6]",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "float",
        "units": "debye*Å",
        "format": ".6f",
    },
    "octapole_moment": {
        "calculation": ["energy", "optimization"],
        "description": "The octapole moment",
        "dimensionality": "[10]",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "float",
        "units": "debye*Å^2",
        "format": ".6f",
    },
    "hexadecapole_moment": {
        "calculation": ["energy", "optimization"],
        "description": "The hexadecapole moment",
        "dimensionality": "[15]",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "float",
        "units": "debye*Å^3",
        "format": ".6f",
    },
    "mosysm": {
        "calculation": ["energy", "optimization"],
        "description": "The symmetry of the MOs",
        "dimensionality": "[nmo]",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "string",
        "units": "",
        "format": "",
    },
    "nbasis": {
        "description": "Number of basis functions",
        "dimensionality": "scalar",
        "type": "integer",
        "units": "",
        "format": "",
    },
    "nmo": {
        "description": "Number of molecular orbitals",
        "dimensionality": "scalar",
        "type": "integer",
        "units": "",
        "format": "",
    },
    "rotconstants": {
        "calculation": ["energy", "optimization"],
        "description": "Rotational constants",
        "dimensionality": "[[3]*nsteps]",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "float",
        "units": "GHz",
        "format": ".3f",
    },
    "scfenergies": {
        "calculation": ["energy", "optimization"],
        "description": "The energies per step",
        "dimensionality": "[nsteps]",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "float",
        "units": "eV",
        "format": ".3f",
    },
    "MP2 Energy": {
        "calculation": ["energy", "gradient"],
        "description": "the MP2 energy",
        "dimensionality": "scalar",
        "methods": ["MP2", "MP3", "MP4", "MP4DQ", "MP4SQD", "MP5", "CCD", "CCSD"],
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "MP3 Energy": {
        "calculation": ["energy", "gradient"],
        "description": "the MP3 energy",
        "dimensionality": "scalar",
        "methods": ["MP3", "MP4", "MP4DQ", "MP4SQD", "MP5", "CCD", "CCSD"],
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "MP4 Energy": {
        "calculation": ["energy", "gradient"],
        "description": "the MP4 energy",
        "dimensionality": "scalar",
        "methods": ["MP4", "MP4DQ", "MP4SQD", "MP5", "CCD", "CCSD"],
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "MP4DQ Energy": {
        "calculation": ["energy", "gradient"],
        "description": "the MP4DQ energy",
        "dimensionality": "scalar",
        "methods": ["MP4", "MP4DQ", "MP4SQD", "MP5", "CCD", "CCSD"],
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "MP4SDQ Energy": {
        "calculation": ["energy", "gradient"],
        "description": "the MP4SDQ energy",
        "dimensionality": "scalar",
        "methods": ["MP4", "MP4SQD", "MP5", "CCSD"],
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "MP5 Energy": {
        "calculation": ["energy", "gradient"],
        "description": "the MP5 energy",
        "dimensionality": "scalar",
        "methods": ["MP5"],
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "Cluster Energy": {
        "calculation": ["energy", "gradient"],
        "description": "the CCD or CCSD energy",
        "dimensionality": "scalar",
        "methods": ["CCD", "CCSD", "CCSD(T)"],
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "Cluster Energy with triples": {
        "calculation": ["energy", "gradient"],
        "description": "the CCSD(T) energy",
        "dimensionality": "scalar",
        "methods": ["CCSD(T)"],
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "Composite/Temperature": {
        "calculation": ["energy", "optimization"],
        "description": "the temperature for the CBS/Gn extrapolation",
        "dimensionality": "scalar",
        "methods": composite,
        "type": "float",
        "units": "K",
        "format": ".2f",
    },
    "Composite/Pressure": {
        "calculation": ["energy", "optimization"],
        "description": "the pressure for the CBS/Gn extrapolation",
        "dimensionality": "scalar",
        "methods": composite,
        "type": "float",
        "units": "atm",
        "format": ".2f",
    },
    "Composite/E(ZPE)": {
        "calculation": ["energy", "optimization"],
        "description": "the ZPE from the CBS/Gn extrapolation",
        "dimensionality": "scalar",
        "methods": composite,
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "Composite/E(Thermal)": {
        "calculation": ["energy", "optimization"],
        "description": "the thermal energy from the CBS/Gn extrapolation",
        "dimensionality": "scalar",
        "methods": composite,
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "Composite/E(SCF)": {
        "calculation": ["energy", "optimization"],
        "description": "the SCF energy from the CBS/Gn extrapolation",
        "dimensionality": "scalar",
        "methods": composite,
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "Composite/DE(MP2)": {
        "calculation": ["energy", "optimization"],
        "description": "dE from MP2 from the CBS/Gn extrapolation",
        "dimensionality": "scalar",
        "methods": composite,
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "Composite/DE(CBS)": {
        "calculation": ["energy", "optimization"],
        "description": "dE from the basis extrapolation from the CBS/Gn extrapolation",
        "dimensionality": "scalar",
        "methods": composite,
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "Composite/DE(MP34)": {
        "calculation": ["energy", "optimization"],
        "description": "dE of MP3-4 from the CBS/Gn extrapolation",
        "dimensionality": "scalar",
        "methods": composite,
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "Composite/DE(CCSD)": {
        "calculation": ["energy", "optimization"],
        "description": "dE of CCSD from the CBS/Gn extrapolation",
        "dimensionality": "scalar",
        "methods": ["CBS-4M", "CBS-QB3", "CBS-APNO"],
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "Composite/DE(int)": {
        "calculation": ["energy", "optimization"],
        "description": "dE from internals from the CBS/Gn extrapolation",
        "dimensionality": "scalar",
        "methods": composite,
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "Composite/DE(empirical)": {
        "calculation": ["energy", "optimization"],
        "description": "Empirical correction to the CBS/Gn extrapolation",
        "dimensionality": "scalar",
        "methods": composite,
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "Composite/E(CCSD(T))": {
        "calculation": ["energy", "optimization"],
        "description": "CCSD(T) term for the Gn extrapolation",
        "dimensionality": "scalar",
        "methods": ["G1", "G2", "G3", "G4"],
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "Composite/DE(Plus)": {
        "calculation": ["energy", "optimization"],
        "description": "Plus term for the Gn extrapolation",
        "dimensionality": "scalar",
        "methods": ["G1", "G2", "G3", "G4"],
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "Composite/DE(2DF)": {
        "calculation": ["energy", "optimization"],
        "description": "The delta energy for 2D's and F term for the Gn extrapolation",
        "dimensionality": "scalar",
        "methods": ["G1", "G2", "G3", "G4"],
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "Composite/E(Delta-G3XP)": {
        "calculation": ["energy", "optimization"],
        "description": "Delta G3XP term for the Gn extrapolation",
        "dimensionality": "scalar",
        "methods": ["G1", "G2", "G3", "G4"],
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "Composite/DE(HF)": {
        "calculation": ["energy", "optimization"],
        "description": "delta E(HF) term for the Gn extrapolation",
        "dimensionality": "scalar",
        "methods": ["G1", "G2", "G3", "G4"],
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "Composite/E(0 K)": {
        "description": "the 0 K energy from the CBS/Gn extrapolation",
        "dimensionality": "scalar",
        "methods": composite,
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "Composite/Energy": {
        "calculation": ["energy", "optimization"],
        "description": "the energy from the CBS/Gn extrapolation",
        "dimensionality": "scalar",
        "methods": composite,
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "Composite/Enthalpy": {
        "calculation": ["energy", "optimization"],
        "description": "the enthalpy from the CBS/Gn extrapolation",
        "dimensionality": "scalar",
        "methods": composite,
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "Composite/Free Energy": {
        "calculation": ["energy", "optimization"],
        "description": "the free energy from the CBS/Gn extrapolation",
        "dimensionality": "scalar",
        "methods": composite,
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
}
