#include <iostream>

#include <pybind11/pybind11.h>
#include <pybind11/stl.h>
#include <memory>

#include "../libs/pcg/pcg_random.hpp"
#include "./Simulator.h"

namespace py = pybind11;


PYBIND11_MODULE(_Sailfish, m) {
    m.doc() = R"pbdoc(
        Sailfish simulator
        -----------------------

        .. currentmodule:: _Sailfish

        .. autosummary::
           :toctree: _generate

            DiscreteDistribution
            SimProtocol
            alphabetCode
            modelCode
            modelFactory
            Simulator
            Msa
            Tree
    )pbdoc";

    using SelectedRNG = pcg64_fast;

    py::class_<Block>(m, "Block")
        .def(py::init<size_t, size_t>());

    py::class_<BlockTree>(m, "BlockTree")
        .def(py::init<>())
        .def("print_tree", &BlockTree::printTree)
        .def("block_list", &BlockTree::getBlockList);

    py::enum_<event>(m, "event")
        .value("Insertion", event::INSERTION)
        .value("Deletion", event::DELETION)
        .export_values();

    py::class_<DiscreteDistribution>(m, "DiscreteDistribution")
        .def(py::init<std::vector<double>>());

    py::class_<tree>(m, "Tree")
        .def(py::init<const std::string&, bool>(), "Create Phylogenetic tree object from newick formatted file")
        .def_property_readonly("num_nodes", &tree::getNodesNum)
        .def_property_readonly("root", &tree::getRoot);

    py::class_<tree::TreeNode>(m, "node")
        .def_property_readonly("sons", &tree::TreeNode::getSons)
        .def_property_readonly("num_leaves", &tree::TreeNode::getNumberLeaves)
        .def_property_readonly("name", &tree::TreeNode::name)
        .def("distance_to_father", &tree::TreeNode::dis2father);

    py::class_<SimulationProtocol>(m, "SimProtocol")
        .def(py::init<tree*>())
        .def("set_seed", &SimulationProtocol::setSeed)
        .def("get_seed", &SimulationProtocol::getSeed)
        .def("set_sequence_size", &SimulationProtocol::setSequenceSize)
        .def("get_sequence_size", &SimulationProtocol::getSequenceSize)
        .def("set_insertion_rates", &SimulationProtocol::setInsertionRates)
        .def("get_insertion_rate", &SimulationProtocol::getInsertionRate)
        .def("set_deletion_rates", &SimulationProtocol::setDeletionRates)
        .def("get_deletion_rate", &SimulationProtocol::getDeletionRate)
        .def("set_insertion_length_distributions", &SimulationProtocol::setInsertionLengthDistributions)
        .def("get_insertion_length_distribution", &SimulationProtocol::getInsertionDistribution)
        .def("set_deletion_length_distributions", &SimulationProtocol::setDeletionLengthDistributions)
        .def("get_deletion_length_distribution", &SimulationProtocol::getDeletionDistribution)
        .def("set_minimum_sequence_size", &SimulationProtocol::setMinSequenceSize)
        .def("get_minimum_sequence_size", &SimulationProtocol::getMinSequenceSize);



    py::class_<sequenceContainer, std::shared_ptr<sequenceContainer>>(m, "sequenceContainer")
        .def(py::init<>());

    py::enum_<alphabetCode>(m, "alphabetCode")
        .value("NULLCODE", alphabetCode::NULLCODE)
        .value("NUCLEOTIDE", alphabetCode::NUCLEOTIDE)
        .value("AMINOACID", alphabetCode::AMINOACID)
        .export_values();


    py::enum_<modelCode>(m, "modelCode")
        .value("NUCJC", modelCode::NUCJC)
        .value("AAJC", modelCode::AAJC)
        .value("GTR", modelCode::GTR)
        .value("HKY", modelCode::HKY)
        .value("TAMURA92", modelCode::TAMURA92)
        // .value("WYANGMODEL", modelCode::WYANGMODEL)
        .value("CPREV45", modelCode::CPREV45)
        .value("DAYHOFF", modelCode::DAYHOFF)
        .value("JONES", modelCode::JONES)	// THIS IS JTT
        .value("MTREV24", modelCode::MTREV24)
        .value("WAG", modelCode::WAG)
        .value("HIVB", modelCode::HIVB)
        .value("HIVW", modelCode::HIVW)
        .value("LG", modelCode::LG)
        .value("EMPIRICODON", modelCode::EMPIRICODON)
        .value("EX_BURIED", modelCode::EX_BURIED) 
        .value("EX_EXPOSED", modelCode::EX_EXPOSED)
        .value("EHO_EXTENDED", modelCode::EHO_EXTENDED)
        .value("EHO_HELIX", modelCode::EHO_HELIX)
        .value("EHO_OTHER", modelCode::EHO_OTHER)
        .value("EX_EHO_BUR_EXT", modelCode::EX_EHO_BUR_EXT)
        .value("EX_EHO_BUR_HEL", modelCode::EX_EHO_BUR_HEL)
        .value("EX_EHO_BUR_OTH", modelCode::EX_EHO_BUR_OTH)
        .value("EX_EHO_EXP_EXT", modelCode::EX_EHO_EXP_EXT)
        .value("EX_EHO_EXP_HEL", modelCode::EX_EHO_EXP_HEL)
        .value("EX_EHO_EXP_OTH", modelCode::EX_EHO_EXP_OTH)
        .value("CUSTOM", modelCode::CUSTOM)
        .export_values();



    py::class_<modelFactory>(m, "modelFactory")
        .def(py::init<tree*>())
        .def("set_alphabet", &modelFactory::setAlphabet)
        .def("set_replacement_model" , &modelFactory::setReplacementModel)
        .def("set_amino_replacement_model_file" , &modelFactory::setCustomAAModelFile)
        .def("set_model_parameters" , &modelFactory::setModelParameters)
        .def("set_gamma_parameters" , &modelFactory::setGammaParameters)
        .def("set_invariant_sites_proportion", &modelFactory::setInvariantSitesProportion)
        .def("set_site_rate_correlation", &modelFactory::setSiteRateCorrelation)
        .def("reset", &modelFactory::resetFactory);


    py::class_<Simulator<SelectedRNG, 20>>(m, "AminoSimulator")
        .def(py::init<SimulationProtocol*>())
        .def("reset_sim", &Simulator<SelectedRNG, 20>::resetSimulator)
        .def("gen_indels", &Simulator<SelectedRNG, 20>::generateSimulation)
        .def("run_sim", &Simulator<SelectedRNG, 20>::runSimulator)
        .def("init_substitution_sim", &Simulator<SelectedRNG, 20>::initSubstitionSim)
        .def("gen_substitutions", &Simulator<SelectedRNG, 20>::simulateSubstitutions)
        .def("gen_substitutions_to_file", &Simulator<SelectedRNG, 20>::simulateAndWriteSubstitutions)
        .def("set_aligned_sequence_map", &Simulator<SelectedRNG, 20>::setAlignedSequenceMap)
        .def("save_site_rates", &Simulator<SelectedRNG, 20>::setSaveRates)
        .def("get_site_rates", &Simulator<SelectedRNG, 20>::getSiteRates)
        .def("save_all_nodes_sequences", &Simulator<SelectedRNG, 20>::setSaveAllNodes)
        .def("save_root_sequence", &Simulator<SelectedRNG, 20>::setSaveRoot)
        .def("get_saved_nodes_mask", &Simulator<SelectedRNG, 20>::getNodesSaveList);

    py::class_<Simulator<SelectedRNG, 4>>(m, "NucleotideSimulator")
        .def(py::init<SimulationProtocol*>())
        .def("reset_sim", &Simulator<SelectedRNG, 4>::resetSimulator)
        .def("gen_indels", &Simulator<SelectedRNG, 4>::generateSimulation)
        .def("run_sim", &Simulator<SelectedRNG, 4>::runSimulator)
        .def("init_substitution_sim", &Simulator<SelectedRNG, 4>::initSubstitionSim)
        .def("gen_substitutions", &Simulator<SelectedRNG, 4>::simulateSubstitutions)
        .def("gen_substitutions_to_file", &Simulator<SelectedRNG, 4>::simulateAndWriteSubstitutions)
        .def("set_aligned_sequence_map", &Simulator<SelectedRNG, 4>::setAlignedSequenceMap)
        .def("save_site_rates", &Simulator<SelectedRNG, 4>::setSaveRates)
        .def("get_site_rates", &Simulator<SelectedRNG, 4>::getSiteRates)
        .def("save_all_nodes_sequences", &Simulator<SelectedRNG, 4>::setSaveAllNodes)
        .def("save_root_sequence", &Simulator<SelectedRNG, 4>::setSaveRoot)
        .def("get_saved_nodes_mask", &Simulator<SelectedRNG, 4>::getNodesSaveList);


    py::class_<MSA>(m, "Msa")
        .def(py::init<size_t, size_t, const std::vector<bool>& >())
        .def(py::init<BlockMap&, tree::TreeNode*, const std::vector<bool>& >())
        .def("generate_msas", &MSA::generateMSAs)
        .def("length", &MSA::getMSAlength)
        .def("num_sequences", &MSA::getNumberOfSequences)
        .def("fill_substitutions", &MSA::fillSubstitutions)
        .def("print_msa", &MSA::printFullMsa)
        .def("print_indels", &MSA::printIndels)
        .def("write_msa", &MSA::writeFullMsa)
        .def("write_msa_from_dir", &MSA::writeMsaFromDir)
        .def("get_msa_string", &MSA::generateMsaString)
        .def("get_msa", &MSA::getMSAVec);

}
