# SPDX-License-Identifier: GPL-3.0-or-later
from typing import Any,Union

import numpy as np

from topoptlab.elements.trilinear_hexahedron import shape_functions,bmatrix
from topoptlab.fem import get_integrpoints

def _fk_heatexp_3d(xe: np.ndarray, c: np.ndarray,
                   a: np.ndarray, DeltaT: Union[None,np.ndarray] = None,
                   quadr_method: str = "gauss-legendre",
                   nquad: int = 2,
                   **kwargs: Any) -> np.ndarray:
    """
    Create force vector for 3D heat expansion with
    trilinear hexahedral Lagrangian elements. This amounts to

    int_Omega B_T @ C_v @ alpha_v @ N_T @ dOmega DeltaT

    Parameters
    ----------
    xe : np.ndarray, shape (nels,8,2)
        coordinates of element nodes. Please look at the
        definition/function of the shape function, then the node ordering is
        clear.
    c : np.ndarray, shape (nels,6,6) or
        stiffness tensor.
    a : np.ndarray, shape (nels,3,3) or
        linear heat expansion tensor.
    DeltaT : np.ndarray shape (nels,8) or None
        difference of nodal temperatures with respect to reference temperature.
    quadr_method: str or callable
        name of quadrature method or function/callable that returns coordinates of
        quadrature points and weights. Check function get_integrpoints for
        available options.
    nquad : int
        number of quadrature points

    Returns
    -------
    fe : np.ndarray, shape (nels,24) or shape (nels,24,8)
        nodal forces due to thermal expansion.

    """
    #
    if len(xe.shape) == 2:
        xe = xe[None,:,:]
    nel = xe.shape[0]
    # convert linear heat expansion tensor to Voigt notation (nel,3,1)
    if len(a.shape) == 2:
        a = a[None,:,:]
    a = a[:,[0,1,2,1,0,0],[0,1,2,2,2,1],None]
    #
    if len(c.shape) == 2:
        c = c[None,:,:]
    #
    if DeltaT is not None:
        if len(DeltaT.shape) == 1:
            DeltaT = DeltaT[None,:]
    #
    x,w=get_integrpoints(ndim=3,nq=nquad,method=quadr_method)
    nq =w.shape[0]
    #
    xi,eta,zeta = [_x[:,0] for _x in np.split(x, 3,axis=1)]
    # shape functions at integration points
    N = shape_functions(xi=xi,eta=eta,zeta=zeta)[None,:,:,None]
    #
    B,detJ = bmatrix(xi=xi, eta=eta, zeta=zeta,
                     xe=xe, all_elems=True,
                     return_detJ=True)
    detJ = detJ.reshape(nel,nq)
    B = B.reshape(nel, nq,  B.shape[-2], B.shape[-1])
    #
    fe = B.transpose([0,1,3,2])@c[:,None,:,:]
    fe = fe@a[:,None,:,:]@N.transpose([0,1,3,2])
    # calculate integral via quadrature
    fe = (w[:,None,None]*fe*detJ[:,:,None,None]).sum(axis=1)
    if DeltaT is None:
        return fe
    else:
        # this is basically a matrix product
        return np.sum(fe*DeltaT[:,None,:],axis=2)

def fk_heatexp_3d(E: float, nu: float,
                  a: float, DeltaT: Union[None,np.ndarray] = None,
                  l: np.ndarray = np.array([1.,1.,1.]), 
                  g: np.ndarray = np.array([0.,0.])) -> np.ndarray:
    """
    Create force vector for 3D heat expansion with
    trilinear hexahedral Lagrangian elements. This amounts to

    int_Omega B_T @ C_v @ alpha_v @ N_T @ dOmega DeltaT

    Parameters
    ----------
    E : float
        Young's modulus.
    nu : float
        Poisson's ratio.
    a : float
        linear heat expansion coefficient.
    DeltaT : np.ndarray shape (8) or None
        difference of nodal temperatures with respect to reference temperature.
    l : np.ndarray (3)
        side length of element
    g : np.ndarray (2)
        angles of parallelepiped.

    Returns
    -------
    fe : np.ndarray, shape (24) or (24,8)
        nodal forces due to thermal expansion or matrix that returns forces by
        fe@DeltaT.

    """
    #
    fe = np.array([[E*a*l[1]*l[2]/(18*(2*nu - 1)),
           E*a*l[1]*l[2]/(18*(2*nu - 1)),
           E*a*l[1]*l[2]/(36*(2*nu - 1)),
           E*a*l[1]*l[2]/(36*(2*nu - 1)),
           E*a*l[1]*l[2]/(36*(2*nu - 1)),
           E*a*l[1]*l[2]/(36*(2*nu - 1)),
           E*a*l[1]*l[2]/(72*(2*nu - 1)),
           E*a*l[1]*l[2]/(72*(2*nu - 1))],
          [E*a*l[2]*(l[0] - l[1]*np.tan(g[0]))/(18*(2*nu - 1)),
           E*a*l[2]*(l[0] - 2*l[1]*np.tan(g[0]))/(36*(2*nu - 1)),
           E*a*l[2]*(l[0] - l[1]*np.tan(g[0]))/(36*(2*nu - 1)),
           E*a*l[2]*(2*l[0] - l[1]*np.tan(g[0]))/(36*(2*nu - 1)),
           E*a*l[2]*(l[0] - l[1]*np.tan(g[0]))/(36*(2*nu - 1)),
           E*a*l[2]*(l[0] - 2*l[1]*np.tan(g[0]))/(72*(2*nu - 1)),
           E*a*l[2]*(l[0] - l[1]*np.tan(g[0]))/(72*(2*nu - 1)),
           E*a*l[2]*(2*l[0] - l[1]*np.tan(g[0]))/(72*(2*nu - 1))],
          [E*a*l[1]*(l[0] - l[2]*np.tan(g[1]))/(18*(2*nu - 1)),
           E*a*l[1]*(l[0] - 2*l[2]*np.tan(g[1]))/(36*(2*nu - 1)),
           E*a*l[1]*(l[0] - 2*l[2]*np.tan(g[1]))/(72*(2*nu - 1)),
           E*a*l[1]*(l[0] - l[2]*np.tan(g[1]))/(36*(2*nu - 1)),
           E*a*l[1]*(2*l[0] - l[2]*np.tan(g[1]))/(36*(2*nu - 1)),
           E*a*l[1]*(l[0] - l[2]*np.tan(g[1]))/(36*(2*nu - 1)),
           E*a*l[1]*(l[0] - l[2]*np.tan(g[1]))/(72*(2*nu - 1)),
           E*a*l[1]*(2*l[0] - l[2]*np.tan(g[1]))/(72*(2*nu - 1))],
          [-E*a*l[1]*l[2]/(36*nu - 18),
           -E*a*l[1]*l[2]/(36*nu - 18),
           -E*a*l[1]*l[2]/(72*nu - 36),
           -E*a*l[1]*l[2]/(72*nu - 36),
           -E*a*l[1]*l[2]/(72*nu - 36),
           -E*a*l[1]*l[2]/(72*nu - 36),
           -E*a*l[1]*l[2]/(144*nu - 72),
           -E*a*l[1]*l[2]/(144*nu - 72)],
          [E*a*l[2]*(l[0] + 2*l[1]*np.tan(g[0]))/(36*(2*nu - 1)),
           E*a*l[2]*(l[0] + l[1]*np.tan(g[0]))/(18*(2*nu - 1)),
           E*a*l[2]*(2*l[0] + l[1]*np.tan(g[0]))/(36*(2*nu - 1)),
           E*a*l[2]*(l[0] + l[1]*np.tan(g[0]))/(36*(2*nu - 1)),
           E*a*l[2]*(l[0] + 2*l[1]*np.tan(g[0]))/(72*(2*nu - 1)),
           E*a*l[2]*(l[0] + l[1]*np.tan(g[0]))/(36*(2*nu - 1)),
           E*a*l[2]*(2*l[0] + l[1]*np.tan(g[0]))/(72*(2*nu - 1)),
           E*a*l[2]*(l[0] + l[1]*np.tan(g[0]))/(72*(2*nu - 1))],
          [E*a*l[1]*(l[0] + 2*l[2]*np.tan(g[1]))/(36*(2*nu - 1)),
           E*a*l[1]*(l[0] + l[2]*np.tan(g[1]))/(18*(2*nu - 1)),
           E*a*l[1]*(l[0] + l[2]*np.tan(g[1]))/(36*(2*nu - 1)),
           E*a*l[1]*(l[0] + 2*l[2]*np.tan(g[1]))/(72*(2*nu - 1)),
           E*a*l[1]*(l[0] + l[2]*np.tan(g[1]))/(36*(2*nu - 1)),
           E*a*l[1]*(2*l[0] + l[2]*np.tan(g[1]))/(36*(2*nu - 1)),
           E*a*l[1]*(2*l[0] + l[2]*np.tan(g[1]))/(72*(2*nu - 1)),
           E*a*l[1]*(l[0] + l[2]*np.tan(g[1]))/(72*(2*nu - 1))],
          [-E*a*l[1]*l[2]/(72*nu - 36),
           -E*a*l[1]*l[2]/(72*nu - 36),
           -E*a*l[1]*l[2]/(36*nu - 18),
           -E*a*l[1]*l[2]/(36*nu - 18),
           -E*a*l[1]*l[2]/(144*nu - 72),
           -E*a*l[1]*l[2]/(144*nu - 72),
           -E*a*l[1]*l[2]/(72*nu - 36),
           -E*a*l[1]*l[2]/(72*nu - 36)],
          [E*a*l[2]*(-l[0] + l[1]*np.tan(g[0]))/(36*(2*nu - 1)),
           E*a*l[2]*(-2*l[0] + l[1]*np.tan(g[0]))/(36*(2*nu - 1)),
           E*a*l[2]*(-l[0] + l[1]*np.tan(g[0]))/(18*(2*nu - 1)),
           E*a*l[2]*(-l[0] + 2*l[1]*np.tan(g[0]))/(36*(2*nu - 1)),
           E*a*l[2]*(-l[0] + l[1]*np.tan(g[0]))/(72*(2*nu - 1)),
           E*a*l[2]*(-2*l[0] + l[1]*np.tan(g[0]))/(72*(2*nu - 1)),
           E*a*l[2]*(-l[0] + l[1]*np.tan(g[0]))/(36*(2*nu - 1)),
           E*a*l[2]*(-l[0] + 2*l[1]*np.tan(g[0]))/(72*(2*nu - 1))],
          [E*a*l[1]*(l[0] + 2*l[2]*np.tan(g[1]))/(72*(2*nu - 1)),
           E*a*l[1]*(l[0] + l[2]*np.tan(g[1]))/(36*(2*nu - 1)),
           E*a*l[1]*(l[0] + l[2]*np.tan(g[1]))/(18*(2*nu - 1)),
           E*a*l[1]*(l[0] + 2*l[2]*np.tan(g[1]))/(36*(2*nu - 1)),
           E*a*l[1]*(l[0] + l[2]*np.tan(g[1]))/(72*(2*nu - 1)),
           E*a*l[1]*(2*l[0] + l[2]*np.tan(g[1]))/(72*(2*nu - 1)),
           E*a*l[1]*(2*l[0] + l[2]*np.tan(g[1]))/(36*(2*nu - 1)),
           E*a*l[1]*(l[0] + l[2]*np.tan(g[1]))/(36*(2*nu - 1))],
          [E*a*l[1]*l[2]/(36*(2*nu - 1)),
           E*a*l[1]*l[2]/(36*(2*nu - 1)),
           E*a*l[1]*l[2]/(18*(2*nu - 1)),
           E*a*l[1]*l[2]/(18*(2*nu - 1)),
           E*a*l[1]*l[2]/(72*(2*nu - 1)),
           E*a*l[1]*l[2]/(72*(2*nu - 1)),
           E*a*l[1]*l[2]/(36*(2*nu - 1)),
           E*a*l[1]*l[2]/(36*(2*nu - 1))],
          [-E*a*l[2]*(2*l[0] + l[1]*np.tan(g[0]))/(72*nu - 36),
           -E*a*l[2]*(l[0] + l[1]*np.tan(g[0]))/(72*nu - 36),
           -E*a*l[2]*(l[0] + 2*l[1]*np.tan(g[0]))/(72*nu - 36),
           -E*a*l[2]*(l[0] + l[1]*np.tan(g[0]))/(36*nu - 18),
           -E*a*l[2]*(2*l[0] + l[1]*np.tan(g[0]))/(144*nu - 72),
           -E*a*l[2]*(l[0] + l[1]*np.tan(g[0]))/(144*nu - 72),
           -E*a*l[2]*(l[0] + 2*l[1]*np.tan(g[0]))/(144*nu - 72),
           -E*a*l[2]*(l[0] + l[1]*np.tan(g[0]))/(72*nu - 36)],
          [E*a*l[1]*(l[0] - l[2]*np.tan(g[1]))/(36*(2*nu - 1)),
           E*a*l[1]*(l[0] - 2*l[2]*np.tan(g[1]))/(72*(2*nu - 1)),
           E*a*l[1]*(l[0] - 2*l[2]*np.tan(g[1]))/(36*(2*nu - 1)),
           E*a*l[1]*(l[0] - l[2]*np.tan(g[1]))/(18*(2*nu - 1)),
           E*a*l[1]*(2*l[0] - l[2]*np.tan(g[1]))/(72*(2*nu - 1)),
           E*a*l[1]*(l[0] - l[2]*np.tan(g[1]))/(72*(2*nu - 1)),
           E*a*l[1]*(l[0] - l[2]*np.tan(g[1]))/(36*(2*nu - 1)),
           E*a*l[1]*(2*l[0] - l[2]*np.tan(g[1]))/(36*(2*nu - 1))],
          [E*a*l[1]*l[2]/(36*(2*nu - 1)),
           E*a*l[1]*l[2]/(36*(2*nu - 1)),
           E*a*l[1]*l[2]/(72*(2*nu - 1)),
           E*a*l[1]*l[2]/(72*(2*nu - 1)),
           E*a*l[1]*l[2]/(18*(2*nu - 1)),
           E*a*l[1]*l[2]/(18*(2*nu - 1)),
           E*a*l[1]*l[2]/(36*(2*nu - 1)),
           E*a*l[1]*l[2]/(36*(2*nu - 1))],
          [E*a*l[2]*(l[0] - l[1]*np.tan(g[0]))/(36*(2*nu - 1)),
           E*a*l[2]*(l[0] - 2*l[1]*np.tan(g[0]))/(72*(2*nu - 1)),
           E*a*l[2]*(l[0] - l[1]*np.tan(g[0]))/(72*(2*nu - 1)),
           E*a*l[2]*(2*l[0] - l[1]*np.tan(g[0]))/(72*(2*nu - 1)),
           E*a*l[2]*(l[0] - l[1]*np.tan(g[0]))/(18*(2*nu - 1)),
           E*a*l[2]*(l[0] - 2*l[1]*np.tan(g[0]))/(36*(2*nu - 1)),
           E*a*l[2]*(l[0] - l[1]*np.tan(g[0]))/(36*(2*nu - 1)),
           E*a*l[2]*(2*l[0] - l[1]*np.tan(g[0]))/(36*(2*nu - 1))],
          [-E*a*l[1]*(2*l[0] + l[2]*np.tan(g[1]))/(72*nu - 36),
           -E*a*l[1]*(l[0] + l[2]*np.tan(g[1]))/(72*nu - 36),
           -E*a*l[1]*(l[0] + l[2]*np.tan(g[1]))/(144*nu - 72),
           -E*a*l[1]*(2*l[0] + l[2]*np.tan(g[1]))/(144*nu - 72),
           -E*a*l[1]*(l[0] + l[2]*np.tan(g[1]))/(36*nu - 18),
           -E*a*l[1]*(l[0] + 2*l[2]*np.tan(g[1]))/(72*nu - 36),
           -E*a*l[1]*(l[0] + 2*l[2]*np.tan(g[1]))/(144*nu - 72),
           -E*a*l[1]*(l[0] + l[2]*np.tan(g[1]))/(72*nu - 36)],
          [-E*a*l[1]*l[2]/(72*nu - 36),
           -E*a*l[1]*l[2]/(72*nu - 36),
           -E*a*l[1]*l[2]/(144*nu - 72),
           -E*a*l[1]*l[2]/(144*nu - 72),
           -E*a*l[1]*l[2]/(36*nu - 18),
           -E*a*l[1]*l[2]/(36*nu - 18),
           -E*a*l[1]*l[2]/(72*nu - 36),
           -E*a*l[1]*l[2]/(72*nu - 36)],
          [E*a*l[2]*(l[0] + 2*l[1]*np.tan(g[0]))/(72*(2*nu - 1)),
           E*a*l[2]*(l[0] + l[1]*np.tan(g[0]))/(36*(2*nu - 1)),
           E*a*l[2]*(2*l[0] + l[1]*np.tan(g[0]))/(72*(2*nu - 1)),
           E*a*l[2]*(l[0] + l[1]*np.tan(g[0]))/(72*(2*nu - 1)),
           E*a*l[2]*(l[0] + 2*l[1]*np.tan(g[0]))/(36*(2*nu - 1)),
           E*a*l[2]*(l[0] + l[1]*np.tan(g[0]))/(18*(2*nu - 1)),
           E*a*l[2]*(2*l[0] + l[1]*np.tan(g[0]))/(36*(2*nu - 1)),
           E*a*l[2]*(l[0] + l[1]*np.tan(g[0]))/(36*(2*nu - 1))],
          [E*a*l[1]*(-l[0] + l[2]*np.tan(g[1]))/(36*(2*nu - 1)),
           E*a*l[1]*(-2*l[0] + l[2]*np.tan(g[1]))/(36*(2*nu - 1)),
           E*a*l[1]*(-2*l[0] + l[2]*np.tan(g[1]))/(72*(2*nu - 1)),
           E*a*l[1]*(-l[0] + l[2]*np.tan(g[1]))/(72*(2*nu - 1)),
           E*a*l[1]*(-l[0] + 2*l[2]*np.tan(g[1]))/(36*(2*nu - 1)),
           E*a*l[1]*(-l[0] + l[2]*np.tan(g[1]))/(18*(2*nu - 1)),
           E*a*l[1]*(-l[0] + l[2]*np.tan(g[1]))/(36*(2*nu - 1)),
           E*a*l[1]*(-l[0] + 2*l[2]*np.tan(g[1]))/(72*(2*nu - 1))],
          [-E*a*l[1]*l[2]/(144*nu - 72),
           -E*a*l[1]*l[2]/(144*nu - 72),
           -E*a*l[1]*l[2]/(72*nu - 36),
           -E*a*l[1]*l[2]/(72*nu - 36),
           -E*a*l[1]*l[2]/(72*nu - 36),
           -E*a*l[1]*l[2]/(72*nu - 36),
           -E*a*l[1]*l[2]/(36*nu - 18),
           -E*a*l[1]*l[2]/(36*nu - 18)],
          [E*a*l[2]*(-l[0] + l[1]*np.tan(g[0]))/(72*(2*nu - 1)),
           E*a*l[2]*(-2*l[0] + l[1]*np.tan(g[0]))/(72*(2*nu - 1)),
           E*a*l[2]*(-l[0] + l[1]*np.tan(g[0]))/(36*(2*nu - 1)),
           E*a*l[2]*(-l[0] + 2*l[1]*np.tan(g[0]))/(72*(2*nu - 1)),
           E*a*l[2]*(-l[0] + l[1]*np.tan(g[0]))/(36*(2*nu - 1)),
           E*a*l[2]*(-2*l[0] + l[1]*np.tan(g[0]))/(36*(2*nu - 1)),
           E*a*l[2]*(-l[0] + l[1]*np.tan(g[0]))/(18*(2*nu - 1)),
           E*a*l[2]*(-l[0] + 2*l[1]*np.tan(g[0]))/(36*(2*nu - 1))],
          [E*a*l[1]*(-l[0] + l[2]*np.tan(g[1]))/(72*(2*nu - 1)),
           E*a*l[1]*(-2*l[0] + l[2]*np.tan(g[1]))/(72*(2*nu - 1)),
           E*a*l[1]*(-2*l[0] + l[2]*np.tan(g[1]))/(36*(2*nu - 1)),
           E*a*l[1]*(-l[0] + l[2]*np.tan(g[1]))/(36*(2*nu - 1)),
           E*a*l[1]*(-l[0] + 2*l[2]*np.tan(g[1]))/(72*(2*nu - 1)),
           E*a*l[1]*(-l[0] + l[2]*np.tan(g[1]))/(36*(2*nu - 1)),
           E*a*l[1]*(-l[0] + l[2]*np.tan(g[1]))/(18*(2*nu - 1)),
           E*a*l[1]*(-l[0] + 2*l[2]*np.tan(g[1]))/(36*(2*nu - 1))],
          [E*a*l[1]*l[2]/(72*(2*nu - 1)),
           E*a*l[1]*l[2]/(72*(2*nu - 1)),
           E*a*l[1]*l[2]/(36*(2*nu - 1)),
           E*a*l[1]*l[2]/(36*(2*nu - 1)),
           E*a*l[1]*l[2]/(36*(2*nu - 1)),
           E*a*l[1]*l[2]/(36*(2*nu - 1)),
           E*a*l[1]*l[2]/(18*(2*nu - 1)),
           E*a*l[1]*l[2]/(18*(2*nu - 1))],
          [-E*a*l[2]*(2*l[0] + l[1]*np.tan(g[0]))/(144*nu - 72),
           -E*a*l[2]*(l[0] + l[1]*np.tan(g[0]))/(144*nu - 72),
           -E*a*l[2]*(l[0] + 2*l[1]*np.tan(g[0]))/(144*nu - 72),
           -E*a*l[2]*(l[0] + l[1]*np.tan(g[0]))/(72*nu - 36),
           -E*a*l[2]*(2*l[0] + l[1]*np.tan(g[0]))/(72*nu - 36),
           -E*a*l[2]*(l[0] + l[1]*np.tan(g[0]))/(72*nu - 36),
           -E*a*l[2]*(l[0] + 2*l[1]*np.tan(g[0]))/(72*nu - 36),
           -E*a*l[2]*(l[0] + l[1]*np.tan(g[0]))/(36*nu - 18)],
          [-E*a*l[1]*(2*l[0] + l[2]*np.tan(g[1]))/(144*nu - 72),
           -E*a*l[1]*(l[0] + l[2]*np.tan(g[1]))/(144*nu - 72),
           -E*a*l[1]*(l[0] + l[2]*np.tan(g[1]))/(72*nu - 36),
           -E*a*l[1]*(2*l[0] + l[2]*np.tan(g[1]))/(72*nu - 36),
           -E*a*l[1]*(l[0] + l[2]*np.tan(g[1]))/(72*nu - 36),
           -E*a*l[1]*(l[0] + 2*l[2]*np.tan(g[1]))/(144*nu - 72),
           -E*a*l[1]*(l[0] + 2*l[2]*np.tan(g[1]))/(72*nu - 36),
           -E*a*l[1]*(l[0] + l[2]*np.tan(g[1]))/(36*nu - 18)]])
    if DeltaT is None:
        return fe
    else:
        return fe@DeltaT

def fk_heatexp_aniso_3d(c: np.ndarray,
                        a: np.ndarray, DeltaT: Union[None,np.ndarray] = None,
                        l: np.ndarray = np.array([1.,1.,1.]), 
                        g: np.ndarray = np.array([0.,0.])) -> np.ndarray:
    """
    Create force vector for 3D heat expansion with
    trilinear hexahedral Lagrangian elements. This amounts to

    int_Omega B_T @ C_v @ alpha_v @ N_T @ dOmega DeltaT

    Parameters
    ----------
    c : np.ndarray, shape (6,6)
        stiffness tensor.
    a : np.ndarray, shape (3,3)
        anisotropic linear heat expansion coefficient. If isotropic k would be
        [[a,0,0],[0,a,0],[0,0,a]]
    DeltaT : np.ndarray shape (8) or None
        difference of nodal temperatures with respect to reference temperature.
    l : np.ndarray (3)
        side length of element
    g : np.ndarray (2)
        angles of parallelepiped.

    Returns
    -------
    fe : np.ndarray, shape (24) or (24,8)
        nodal forces due to thermal expansion or matrix that returns forces by
        fe@DeltaT.

    """
    # convert heat expans. coeff. to Voigt notation
    a = a[[0,1,2,1,0,0],[0,1,2,2,2,1]]
    #
    fe = np.array([[-a[0]*c[0,0]*l[1]*l[2]/18 - a[0]*c[4,0]*l[0]*l[1]/18 + a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/18 - a[0]*c[5,0]*l[0]*l[2]/18 + a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/18 - a[1]*c[0,1]*l[1]*l[2]/18 - a[1]*c[4,1]*l[0]*l[1]/18 + a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/18 - a[1]*c[5,1]*l[0]*l[2]/18 + a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/18 - a[2]*c[0,2]*l[1]*l[2]/18 - a[2]*c[4,2]*l[0]*l[1]/18 + a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/18 - a[2]*c[5,2]*l[0]*l[2]/18 + a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/18 - a[3]*c[0,3]*l[1]*l[2]/18 - a[3]*c[4,3]*l[0]*l[1]/18 + a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/18 - a[3]*c[5,3]*l[0]*l[2]/18 + a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/18 - a[4]*c[0,4]*l[1]*l[2]/18 - a[4]*c[4,4]*l[0]*l[1]/18 + a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/18 - a[4]*c[5,4]*l[0]*l[2]/18 + a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/18 - a[5]*c[0,5]*l[1]*l[2]/18 - a[5]*c[4,5]*l[0]*l[1]/18 + a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/18 - a[5]*c[5,5]*l[0]*l[2]/18 + a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/18,
           -a[0]*c[0,0]*l[1]*l[2]/18 - a[0]*c[4,0]*l[0]*l[1]/36 + a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/18 - a[0]*c[5,0]*l[0]*l[2]/36 + a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/18 - a[1]*c[0,1]*l[1]*l[2]/18 - a[1]*c[4,1]*l[0]*l[1]/36 + a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/18 - a[1]*c[5,1]*l[0]*l[2]/36 + a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/18 - a[2]*c[0,2]*l[1]*l[2]/18 - a[2]*c[4,2]*l[0]*l[1]/36 + a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/18 - a[2]*c[5,2]*l[0]*l[2]/36 + a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/18 - a[3]*c[0,3]*l[1]*l[2]/18 - a[3]*c[4,3]*l[0]*l[1]/36 + a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/18 - a[3]*c[5,3]*l[0]*l[2]/36 + a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/18 - a[4]*c[0,4]*l[1]*l[2]/18 - a[4]*c[4,4]*l[0]*l[1]/36 + a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/18 - a[4]*c[5,4]*l[0]*l[2]/36 + a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/18 - a[5]*c[0,5]*l[1]*l[2]/18 - a[5]*c[4,5]*l[0]*l[1]/36 + a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/18 - a[5]*c[5,5]*l[0]*l[2]/36 + a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/18,
           -a[0]*c[0,0]*l[1]*l[2]/36 - a[0]*c[4,0]*l[0]*l[1]/72 + a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[5,0]*l[0]*l[2]/36 + a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[0,1]*l[1]*l[2]/36 - a[1]*c[4,1]*l[0]*l[1]/72 + a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[5,1]*l[0]*l[2]/36 + a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[0,2]*l[1]*l[2]/36 - a[2]*c[4,2]*l[0]*l[1]/72 + a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[5,2]*l[0]*l[2]/36 + a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[0,3]*l[1]*l[2]/36 - a[3]*c[4,3]*l[0]*l[1]/72 + a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[5,3]*l[0]*l[2]/36 + a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[0,4]*l[1]*l[2]/36 - a[4]*c[4,4]*l[0]*l[1]/72 + a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[5,4]*l[0]*l[2]/36 + a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[0,5]*l[1]*l[2]/36 - a[5]*c[4,5]*l[0]*l[1]/72 + a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[5,5]*l[0]*l[2]/36 + a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/36,
           -a[0]*c[0,0]*l[1]*l[2]/36 - a[0]*c[4,0]*l[0]*l[1]/36 + a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[5,0]*l[0]*l[2]/18 + a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[0,1]*l[1]*l[2]/36 - a[1]*c[4,1]*l[0]*l[1]/36 + a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[5,1]*l[0]*l[2]/18 + a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[0,2]*l[1]*l[2]/36 - a[2]*c[4,2]*l[0]*l[1]/36 + a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[5,2]*l[0]*l[2]/18 + a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[0,3]*l[1]*l[2]/36 - a[3]*c[4,3]*l[0]*l[1]/36 + a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[5,3]*l[0]*l[2]/18 + a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[0,4]*l[1]*l[2]/36 - a[4]*c[4,4]*l[0]*l[1]/36 + a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[5,4]*l[0]*l[2]/18 + a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[0,5]*l[1]*l[2]/36 - a[5]*c[4,5]*l[0]*l[1]/36 + a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[5,5]*l[0]*l[2]/18 + a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/36,
           -a[0]*c[0,0]*l[1]*l[2]/36 - a[0]*c[4,0]*l[0]*l[1]/18 + a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[5,0]*l[0]*l[2]/36 + a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[0,1]*l[1]*l[2]/36 - a[1]*c[4,1]*l[0]*l[1]/18 + a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[5,1]*l[0]*l[2]/36 + a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[0,2]*l[1]*l[2]/36 - a[2]*c[4,2]*l[0]*l[1]/18 + a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[5,2]*l[0]*l[2]/36 + a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[0,3]*l[1]*l[2]/36 - a[3]*c[4,3]*l[0]*l[1]/18 + a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[5,3]*l[0]*l[2]/36 + a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[0,4]*l[1]*l[2]/36 - a[4]*c[4,4]*l[0]*l[1]/18 + a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[5,4]*l[0]*l[2]/36 + a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[0,5]*l[1]*l[2]/36 - a[5]*c[4,5]*l[0]*l[1]/18 + a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[5,5]*l[0]*l[2]/36 + a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/36,
           -a[0]*c[0,0]*l[1]*l[2]/36 - a[0]*c[4,0]*l[0]*l[1]/36 + a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[5,0]*l[0]*l[2]/72 + a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[0,1]*l[1]*l[2]/36 - a[1]*c[4,1]*l[0]*l[1]/36 + a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[5,1]*l[0]*l[2]/72 + a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[0,2]*l[1]*l[2]/36 - a[2]*c[4,2]*l[0]*l[1]/36 + a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[5,2]*l[0]*l[2]/72 + a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[0,3]*l[1]*l[2]/36 - a[3]*c[4,3]*l[0]*l[1]/36 + a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[5,3]*l[0]*l[2]/72 + a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[0,4]*l[1]*l[2]/36 - a[4]*c[4,4]*l[0]*l[1]/36 + a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[5,4]*l[0]*l[2]/72 + a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[0,5]*l[1]*l[2]/36 - a[5]*c[4,5]*l[0]*l[1]/36 + a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[5,5]*l[0]*l[2]/72 + a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/36,
           -a[0]*c[0,0]*l[1]*l[2]/72 - a[0]*c[4,0]*l[0]*l[1]/72 + a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/72 - a[0]*c[5,0]*l[0]*l[2]/72 + a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/72 - a[1]*c[0,1]*l[1]*l[2]/72 - a[1]*c[4,1]*l[0]*l[1]/72 + a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/72 - a[1]*c[5,1]*l[0]*l[2]/72 + a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/72 - a[2]*c[0,2]*l[1]*l[2]/72 - a[2]*c[4,2]*l[0]*l[1]/72 + a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/72 - a[2]*c[5,2]*l[0]*l[2]/72 + a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/72 - a[3]*c[0,3]*l[1]*l[2]/72 - a[3]*c[4,3]*l[0]*l[1]/72 + a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/72 - a[3]*c[5,3]*l[0]*l[2]/72 + a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/72 - a[4]*c[0,4]*l[1]*l[2]/72 - a[4]*c[4,4]*l[0]*l[1]/72 + a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/72 - a[4]*c[5,4]*l[0]*l[2]/72 + a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/72 - a[5]*c[0,5]*l[1]*l[2]/72 - a[5]*c[4,5]*l[0]*l[1]/72 + a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/72 - a[5]*c[5,5]*l[0]*l[2]/72 + a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/72,
           -a[0]*c[0,0]*l[1]*l[2]/72 - a[0]*c[4,0]*l[0]*l[1]/36 + a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/72 - a[0]*c[5,0]*l[0]*l[2]/36 + a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/72 - a[1]*c[0,1]*l[1]*l[2]/72 - a[1]*c[4,1]*l[0]*l[1]/36 + a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/72 - a[1]*c[5,1]*l[0]*l[2]/36 + a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/72 - a[2]*c[0,2]*l[1]*l[2]/72 - a[2]*c[4,2]*l[0]*l[1]/36 + a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/72 - a[2]*c[5,2]*l[0]*l[2]/36 + a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/72 - a[3]*c[0,3]*l[1]*l[2]/72 - a[3]*c[4,3]*l[0]*l[1]/36 + a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/72 - a[3]*c[5,3]*l[0]*l[2]/36 + a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/72 - a[4]*c[0,4]*l[1]*l[2]/72 - a[4]*c[4,4]*l[0]*l[1]/36 + a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/72 - a[4]*c[5,4]*l[0]*l[2]/36 + a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/72 - a[5]*c[0,5]*l[1]*l[2]/72 - a[5]*c[4,5]*l[0]*l[1]/36 + a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/72 - a[5]*c[5,5]*l[0]*l[2]/36 + a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/72],
          [-a[0]*c[1,0]*l[0]*l[2]/18 + a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/18 - a[0]*c[3,0]*l[0]*l[1]/18 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/18 - a[0]*c[5,0]*l[1]*l[2]/18 - a[1]*c[1,1]*l[0]*l[2]/18 + a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/18 - a[1]*c[3,1]*l[0]*l[1]/18 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/18 - a[1]*c[5,1]*l[1]*l[2]/18 - a[2]*c[1,2]*l[0]*l[2]/18 + a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/18 - a[2]*c[3,2]*l[0]*l[1]/18 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/18 - a[2]*c[5,2]*l[1]*l[2]/18 - a[3]*c[1,3]*l[0]*l[2]/18 + a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/18 - a[3]*c[3,3]*l[0]*l[1]/18 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/18 - a[3]*c[5,3]*l[1]*l[2]/18 - a[4]*c[1,4]*l[0]*l[2]/18 + a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/18 - a[4]*c[3,4]*l[0]*l[1]/18 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/18 - a[4]*c[5,4]*l[1]*l[2]/18 - a[5]*c[1,5]*l[0]*l[2]/18 + a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/18 - a[5]*c[3,5]*l[0]*l[1]/18 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/18 - a[5]*c[5,5]*l[1]*l[2]/18,
           -a[0]*c[1,0]*l[0]*l[2]/36 + a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/18 - a[0]*c[3,0]*l[0]*l[1]/36 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/18 - a[0]*c[5,0]*l[1]*l[2]/18 - a[1]*c[1,1]*l[0]*l[2]/36 + a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/18 - a[1]*c[3,1]*l[0]*l[1]/36 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/18 - a[1]*c[5,1]*l[1]*l[2]/18 - a[2]*c[1,2]*l[0]*l[2]/36 + a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/18 - a[2]*c[3,2]*l[0]*l[1]/36 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/18 - a[2]*c[5,2]*l[1]*l[2]/18 - a[3]*c[1,3]*l[0]*l[2]/36 + a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/18 - a[3]*c[3,3]*l[0]*l[1]/36 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/18 - a[3]*c[5,3]*l[1]*l[2]/18 - a[4]*c[1,4]*l[0]*l[2]/36 + a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/18 - a[4]*c[3,4]*l[0]*l[1]/36 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/18 - a[4]*c[5,4]*l[1]*l[2]/18 - a[5]*c[1,5]*l[0]*l[2]/36 + a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/18 - a[5]*c[3,5]*l[0]*l[1]/36 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/18 - a[5]*c[5,5]*l[1]*l[2]/18,
           -a[0]*c[1,0]*l[0]*l[2]/36 + a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/36 - a[0]*c[3,0]*l[0]*l[1]/72 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[5,0]*l[1]*l[2]/36 - a[1]*c[1,1]*l[0]*l[2]/36 + a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[3,1]*l[0]*l[1]/72 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[5,1]*l[1]*l[2]/36 - a[2]*c[1,2]*l[0]*l[2]/36 + a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[3,2]*l[0]*l[1]/72 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[5,2]*l[1]*l[2]/36 - a[3]*c[1,3]*l[0]*l[2]/36 + a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[3,3]*l[0]*l[1]/72 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[5,3]*l[1]*l[2]/36 - a[4]*c[1,4]*l[0]*l[2]/36 + a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[3,4]*l[0]*l[1]/72 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[5,4]*l[1]*l[2]/36 - a[5]*c[1,5]*l[0]*l[2]/36 + a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[3,5]*l[0]*l[1]/72 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[5,5]*l[1]*l[2]/36,
           -a[0]*c[1,0]*l[0]*l[2]/18 + a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/36 - a[0]*c[3,0]*l[0]*l[1]/36 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[5,0]*l[1]*l[2]/36 - a[1]*c[1,1]*l[0]*l[2]/18 + a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[3,1]*l[0]*l[1]/36 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[5,1]*l[1]*l[2]/36 - a[2]*c[1,2]*l[0]*l[2]/18 + a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[3,2]*l[0]*l[1]/36 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[5,2]*l[1]*l[2]/36 - a[3]*c[1,3]*l[0]*l[2]/18 + a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[3,3]*l[0]*l[1]/36 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[5,3]*l[1]*l[2]/36 - a[4]*c[1,4]*l[0]*l[2]/18 + a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[3,4]*l[0]*l[1]/36 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[5,4]*l[1]*l[2]/36 - a[5]*c[1,5]*l[0]*l[2]/18 + a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[3,5]*l[0]*l[1]/36 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[5,5]*l[1]*l[2]/36,
           -a[0]*c[1,0]*l[0]*l[2]/36 + a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/36 - a[0]*c[3,0]*l[0]*l[1]/18 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[5,0]*l[1]*l[2]/36 - a[1]*c[1,1]*l[0]*l[2]/36 + a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[3,1]*l[0]*l[1]/18 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[5,1]*l[1]*l[2]/36 - a[2]*c[1,2]*l[0]*l[2]/36 + a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[3,2]*l[0]*l[1]/18 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[5,2]*l[1]*l[2]/36 - a[3]*c[1,3]*l[0]*l[2]/36 + a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[3,3]*l[0]*l[1]/18 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[5,3]*l[1]*l[2]/36 - a[4]*c[1,4]*l[0]*l[2]/36 + a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[3,4]*l[0]*l[1]/18 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[5,4]*l[1]*l[2]/36 - a[5]*c[1,5]*l[0]*l[2]/36 + a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[3,5]*l[0]*l[1]/18 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[5,5]*l[1]*l[2]/36,
           -a[0]*c[1,0]*l[0]*l[2]/72 + a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/36 - a[0]*c[3,0]*l[0]*l[1]/36 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[5,0]*l[1]*l[2]/36 - a[1]*c[1,1]*l[0]*l[2]/72 + a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[3,1]*l[0]*l[1]/36 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[5,1]*l[1]*l[2]/36 - a[2]*c[1,2]*l[0]*l[2]/72 + a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[3,2]*l[0]*l[1]/36 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[5,2]*l[1]*l[2]/36 - a[3]*c[1,3]*l[0]*l[2]/72 + a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[3,3]*l[0]*l[1]/36 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[5,3]*l[1]*l[2]/36 - a[4]*c[1,4]*l[0]*l[2]/72 + a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[3,4]*l[0]*l[1]/36 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[5,4]*l[1]*l[2]/36 - a[5]*c[1,5]*l[0]*l[2]/72 + a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[3,5]*l[0]*l[1]/36 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[5,5]*l[1]*l[2]/36,
           -a[0]*c[1,0]*l[0]*l[2]/72 + a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/72 - a[0]*c[3,0]*l[0]*l[1]/72 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/72 - a[0]*c[5,0]*l[1]*l[2]/72 - a[1]*c[1,1]*l[0]*l[2]/72 + a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/72 - a[1]*c[3,1]*l[0]*l[1]/72 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/72 - a[1]*c[5,1]*l[1]*l[2]/72 - a[2]*c[1,2]*l[0]*l[2]/72 + a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/72 - a[2]*c[3,2]*l[0]*l[1]/72 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/72 - a[2]*c[5,2]*l[1]*l[2]/72 - a[3]*c[1,3]*l[0]*l[2]/72 + a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/72 - a[3]*c[3,3]*l[0]*l[1]/72 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/72 - a[3]*c[5,3]*l[1]*l[2]/72 - a[4]*c[1,4]*l[0]*l[2]/72 + a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/72 - a[4]*c[3,4]*l[0]*l[1]/72 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/72 - a[4]*c[5,4]*l[1]*l[2]/72 - a[5]*c[1,5]*l[0]*l[2]/72 + a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/72 - a[5]*c[3,5]*l[0]*l[1]/72 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/72 - a[5]*c[5,5]*l[1]*l[2]/72,
           -a[0]*c[1,0]*l[0]*l[2]/36 + a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/72 - a[0]*c[3,0]*l[0]*l[1]/36 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/72 - a[0]*c[5,0]*l[1]*l[2]/72 - a[1]*c[1,1]*l[0]*l[2]/36 + a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/72 - a[1]*c[3,1]*l[0]*l[1]/36 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/72 - a[1]*c[5,1]*l[1]*l[2]/72 - a[2]*c[1,2]*l[0]*l[2]/36 + a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/72 - a[2]*c[3,2]*l[0]*l[1]/36 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/72 - a[2]*c[5,2]*l[1]*l[2]/72 - a[3]*c[1,3]*l[0]*l[2]/36 + a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/72 - a[3]*c[3,3]*l[0]*l[1]/36 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/72 - a[3]*c[5,3]*l[1]*l[2]/72 - a[4]*c[1,4]*l[0]*l[2]/36 + a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/72 - a[4]*c[3,4]*l[0]*l[1]/36 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/72 - a[4]*c[5,4]*l[1]*l[2]/72 - a[5]*c[1,5]*l[0]*l[2]/36 + a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/72 - a[5]*c[3,5]*l[0]*l[1]/36 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/72 - a[5]*c[5,5]*l[1]*l[2]/72],
          [-a[0]*c[2,0]*l[0]*l[1]/18 + a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/18 - a[0]*c[3,0]*l[0]*l[2]/18 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/18 - a[0]*c[4,0]*l[1]*l[2]/18 - a[1]*c[2,1]*l[0]*l[1]/18 + a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/18 - a[1]*c[3,1]*l[0]*l[2]/18 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/18 - a[1]*c[4,1]*l[1]*l[2]/18 - a[2]*c[2,2]*l[0]*l[1]/18 + a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/18 - a[2]*c[3,2]*l[0]*l[2]/18 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/18 - a[2]*c[4,2]*l[1]*l[2]/18 - a[3]*c[2,3]*l[0]*l[1]/18 + a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/18 - a[3]*c[3,3]*l[0]*l[2]/18 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/18 - a[3]*c[4,3]*l[1]*l[2]/18 - a[4]*c[2,4]*l[0]*l[1]/18 + a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/18 - a[4]*c[3,4]*l[0]*l[2]/18 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/18 - a[4]*c[4,4]*l[1]*l[2]/18 - a[5]*c[2,5]*l[0]*l[1]/18 + a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/18 - a[5]*c[3,5]*l[0]*l[2]/18 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/18 - a[5]*c[4,5]*l[1]*l[2]/18,
           -a[0]*c[2,0]*l[0]*l[1]/36 + a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/18 - a[0]*c[3,0]*l[0]*l[2]/36 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/18 - a[0]*c[4,0]*l[1]*l[2]/18 - a[1]*c[2,1]*l[0]*l[1]/36 + a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/18 - a[1]*c[3,1]*l[0]*l[2]/36 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/18 - a[1]*c[4,1]*l[1]*l[2]/18 - a[2]*c[2,2]*l[0]*l[1]/36 + a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/18 - a[2]*c[3,2]*l[0]*l[2]/36 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/18 - a[2]*c[4,2]*l[1]*l[2]/18 - a[3]*c[2,3]*l[0]*l[1]/36 + a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/18 - a[3]*c[3,3]*l[0]*l[2]/36 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/18 - a[3]*c[4,3]*l[1]*l[2]/18 - a[4]*c[2,4]*l[0]*l[1]/36 + a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/18 - a[4]*c[3,4]*l[0]*l[2]/36 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/18 - a[4]*c[4,4]*l[1]*l[2]/18 - a[5]*c[2,5]*l[0]*l[1]/36 + a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/18 - a[5]*c[3,5]*l[0]*l[2]/36 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/18 - a[5]*c[4,5]*l[1]*l[2]/18,
           -a[0]*c[2,0]*l[0]*l[1]/72 + a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[3,0]*l[0]*l[2]/36 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/36 - a[0]*c[4,0]*l[1]*l[2]/36 - a[1]*c[2,1]*l[0]*l[1]/72 + a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[3,1]*l[0]*l[2]/36 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[4,1]*l[1]*l[2]/36 - a[2]*c[2,2]*l[0]*l[1]/72 + a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[3,2]*l[0]*l[2]/36 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[4,2]*l[1]*l[2]/36 - a[3]*c[2,3]*l[0]*l[1]/72 + a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[3,3]*l[0]*l[2]/36 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[4,3]*l[1]*l[2]/36 - a[4]*c[2,4]*l[0]*l[1]/72 + a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[3,4]*l[0]*l[2]/36 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[4,4]*l[1]*l[2]/36 - a[5]*c[2,5]*l[0]*l[1]/72 + a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[3,5]*l[0]*l[2]/36 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[4,5]*l[1]*l[2]/36,
           -a[0]*c[2,0]*l[0]*l[1]/36 + a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[3,0]*l[0]*l[2]/18 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/36 - a[0]*c[4,0]*l[1]*l[2]/36 - a[1]*c[2,1]*l[0]*l[1]/36 + a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[3,1]*l[0]*l[2]/18 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[4,1]*l[1]*l[2]/36 - a[2]*c[2,2]*l[0]*l[1]/36 + a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[3,2]*l[0]*l[2]/18 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[4,2]*l[1]*l[2]/36 - a[3]*c[2,3]*l[0]*l[1]/36 + a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[3,3]*l[0]*l[2]/18 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[4,3]*l[1]*l[2]/36 - a[4]*c[2,4]*l[0]*l[1]/36 + a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[3,4]*l[0]*l[2]/18 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[4,4]*l[1]*l[2]/36 - a[5]*c[2,5]*l[0]*l[1]/36 + a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[3,5]*l[0]*l[2]/18 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[4,5]*l[1]*l[2]/36,
           -a[0]*c[2,0]*l[0]*l[1]/18 + a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[3,0]*l[0]*l[2]/36 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/36 - a[0]*c[4,0]*l[1]*l[2]/36 - a[1]*c[2,1]*l[0]*l[1]/18 + a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[3,1]*l[0]*l[2]/36 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[4,1]*l[1]*l[2]/36 - a[2]*c[2,2]*l[0]*l[1]/18 + a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[3,2]*l[0]*l[2]/36 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[4,2]*l[1]*l[2]/36 - a[3]*c[2,3]*l[0]*l[1]/18 + a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[3,3]*l[0]*l[2]/36 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[4,3]*l[1]*l[2]/36 - a[4]*c[2,4]*l[0]*l[1]/18 + a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[3,4]*l[0]*l[2]/36 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[4,4]*l[1]*l[2]/36 - a[5]*c[2,5]*l[0]*l[1]/18 + a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[3,5]*l[0]*l[2]/36 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[4,5]*l[1]*l[2]/36,
           -a[0]*c[2,0]*l[0]*l[1]/36 + a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[3,0]*l[0]*l[2]/72 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/36 - a[0]*c[4,0]*l[1]*l[2]/36 - a[1]*c[2,1]*l[0]*l[1]/36 + a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[3,1]*l[0]*l[2]/72 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[4,1]*l[1]*l[2]/36 - a[2]*c[2,2]*l[0]*l[1]/36 + a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[3,2]*l[0]*l[2]/72 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[4,2]*l[1]*l[2]/36 - a[3]*c[2,3]*l[0]*l[1]/36 + a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[3,3]*l[0]*l[2]/72 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[4,3]*l[1]*l[2]/36 - a[4]*c[2,4]*l[0]*l[1]/36 + a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[3,4]*l[0]*l[2]/72 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[4,4]*l[1]*l[2]/36 - a[5]*c[2,5]*l[0]*l[1]/36 + a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[3,5]*l[0]*l[2]/72 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[4,5]*l[1]*l[2]/36,
           -a[0]*c[2,0]*l[0]*l[1]/72 + a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/72 - a[0]*c[3,0]*l[0]*l[2]/72 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/72 - a[0]*c[4,0]*l[1]*l[2]/72 - a[1]*c[2,1]*l[0]*l[1]/72 + a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/72 - a[1]*c[3,1]*l[0]*l[2]/72 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/72 - a[1]*c[4,1]*l[1]*l[2]/72 - a[2]*c[2,2]*l[0]*l[1]/72 + a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/72 - a[2]*c[3,2]*l[0]*l[2]/72 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/72 - a[2]*c[4,2]*l[1]*l[2]/72 - a[3]*c[2,3]*l[0]*l[1]/72 + a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/72 - a[3]*c[3,3]*l[0]*l[2]/72 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/72 - a[3]*c[4,3]*l[1]*l[2]/72 - a[4]*c[2,4]*l[0]*l[1]/72 + a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/72 - a[4]*c[3,4]*l[0]*l[2]/72 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/72 - a[4]*c[4,4]*l[1]*l[2]/72 - a[5]*c[2,5]*l[0]*l[1]/72 + a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/72 - a[5]*c[3,5]*l[0]*l[2]/72 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/72 - a[5]*c[4,5]*l[1]*l[2]/72,
           -a[0]*c[2,0]*l[0]*l[1]/36 + a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/72 - a[0]*c[3,0]*l[0]*l[2]/36 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/72 - a[0]*c[4,0]*l[1]*l[2]/72 - a[1]*c[2,1]*l[0]*l[1]/36 + a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/72 - a[1]*c[3,1]*l[0]*l[2]/36 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/72 - a[1]*c[4,1]*l[1]*l[2]/72 - a[2]*c[2,2]*l[0]*l[1]/36 + a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/72 - a[2]*c[3,2]*l[0]*l[2]/36 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/72 - a[2]*c[4,2]*l[1]*l[2]/72 - a[3]*c[2,3]*l[0]*l[1]/36 + a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/72 - a[3]*c[3,3]*l[0]*l[2]/36 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/72 - a[3]*c[4,3]*l[1]*l[2]/72 - a[4]*c[2,4]*l[0]*l[1]/36 + a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/72 - a[4]*c[3,4]*l[0]*l[2]/36 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/72 - a[4]*c[4,4]*l[1]*l[2]/72 - a[5]*c[2,5]*l[0]*l[1]/36 + a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/72 - a[5]*c[3,5]*l[0]*l[2]/36 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/72 - a[5]*c[4,5]*l[1]*l[2]/72],
          [a[0]*c[0,0]*l[1]*l[2]/18 - a[0]*c[4,0]*l[0]*l[1]/36 - a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/18 - a[0]*c[5,0]*l[0]*l[2]/36 - a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/18 + a[1]*c[0,1]*l[1]*l[2]/18 - a[1]*c[4,1]*l[0]*l[1]/36 - a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/18 - a[1]*c[5,1]*l[0]*l[2]/36 - a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/18 + a[2]*c[0,2]*l[1]*l[2]/18 - a[2]*c[4,2]*l[0]*l[1]/36 - a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/18 - a[2]*c[5,2]*l[0]*l[2]/36 - a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/18 + a[3]*c[0,3]*l[1]*l[2]/18 - a[3]*c[4,3]*l[0]*l[1]/36 - a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/18 - a[3]*c[5,3]*l[0]*l[2]/36 - a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/18 + a[4]*c[0,4]*l[1]*l[2]/18 - a[4]*c[4,4]*l[0]*l[1]/36 - a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/18 - a[4]*c[5,4]*l[0]*l[2]/36 - a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/18 + a[5]*c[0,5]*l[1]*l[2]/18 - a[5]*c[4,5]*l[0]*l[1]/36 - a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/18 - a[5]*c[5,5]*l[0]*l[2]/36 - a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/18,
           a[0]*c[0,0]*l[1]*l[2]/18 - a[0]*c[4,0]*l[0]*l[1]/18 - a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/18 - a[0]*c[5,0]*l[0]*l[2]/18 - a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/18 + a[1]*c[0,1]*l[1]*l[2]/18 - a[1]*c[4,1]*l[0]*l[1]/18 - a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/18 - a[1]*c[5,1]*l[0]*l[2]/18 - a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/18 + a[2]*c[0,2]*l[1]*l[2]/18 - a[2]*c[4,2]*l[0]*l[1]/18 - a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/18 - a[2]*c[5,2]*l[0]*l[2]/18 - a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/18 + a[3]*c[0,3]*l[1]*l[2]/18 - a[3]*c[4,3]*l[0]*l[1]/18 - a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/18 - a[3]*c[5,3]*l[0]*l[2]/18 - a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/18 + a[4]*c[0,4]*l[1]*l[2]/18 - a[4]*c[4,4]*l[0]*l[1]/18 - a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/18 - a[4]*c[5,4]*l[0]*l[2]/18 - a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/18 + a[5]*c[0,5]*l[1]*l[2]/18 - a[5]*c[4,5]*l[0]*l[1]/18 - a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/18 - a[5]*c[5,5]*l[0]*l[2]/18 - a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/18,
           a[0]*c[0,0]*l[1]*l[2]/36 - a[0]*c[4,0]*l[0]*l[1]/36 - a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[5,0]*l[0]*l[2]/18 - a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[0,1]*l[1]*l[2]/36 - a[1]*c[4,1]*l[0]*l[1]/36 - a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[5,1]*l[0]*l[2]/18 - a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[0,2]*l[1]*l[2]/36 - a[2]*c[4,2]*l[0]*l[1]/36 - a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[5,2]*l[0]*l[2]/18 - a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[0,3]*l[1]*l[2]/36 - a[3]*c[4,3]*l[0]*l[1]/36 - a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[5,3]*l[0]*l[2]/18 - a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[0,4]*l[1]*l[2]/36 - a[4]*c[4,4]*l[0]*l[1]/36 - a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[5,4]*l[0]*l[2]/18 - a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[0,5]*l[1]*l[2]/36 - a[5]*c[4,5]*l[0]*l[1]/36 - a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[5,5]*l[0]*l[2]/18 - a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/36,
           a[0]*c[0,0]*l[1]*l[2]/36 - a[0]*c[4,0]*l[0]*l[1]/72 - a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[5,0]*l[0]*l[2]/36 - a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[0,1]*l[1]*l[2]/36 - a[1]*c[4,1]*l[0]*l[1]/72 - a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[5,1]*l[0]*l[2]/36 - a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[0,2]*l[1]*l[2]/36 - a[2]*c[4,2]*l[0]*l[1]/72 - a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[5,2]*l[0]*l[2]/36 - a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[0,3]*l[1]*l[2]/36 - a[3]*c[4,3]*l[0]*l[1]/72 - a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[5,3]*l[0]*l[2]/36 - a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[0,4]*l[1]*l[2]/36 - a[4]*c[4,4]*l[0]*l[1]/72 - a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[5,4]*l[0]*l[2]/36 - a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[0,5]*l[1]*l[2]/36 - a[5]*c[4,5]*l[0]*l[1]/72 - a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[5,5]*l[0]*l[2]/36 - a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/36,
           a[0]*c[0,0]*l[1]*l[2]/36 - a[0]*c[4,0]*l[0]*l[1]/36 - a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[5,0]*l[0]*l[2]/72 - a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[0,1]*l[1]*l[2]/36 - a[1]*c[4,1]*l[0]*l[1]/36 - a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[5,1]*l[0]*l[2]/72 - a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[0,2]*l[1]*l[2]/36 - a[2]*c[4,2]*l[0]*l[1]/36 - a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[5,2]*l[0]*l[2]/72 - a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[0,3]*l[1]*l[2]/36 - a[3]*c[4,3]*l[0]*l[1]/36 - a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[5,3]*l[0]*l[2]/72 - a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[0,4]*l[1]*l[2]/36 - a[4]*c[4,4]*l[0]*l[1]/36 - a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[5,4]*l[0]*l[2]/72 - a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[0,5]*l[1]*l[2]/36 - a[5]*c[4,5]*l[0]*l[1]/36 - a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[5,5]*l[0]*l[2]/72 - a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/36,
           a[0]*c[0,0]*l[1]*l[2]/36 - a[0]*c[4,0]*l[0]*l[1]/18 - a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[5,0]*l[0]*l[2]/36 - a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[0,1]*l[1]*l[2]/36 - a[1]*c[4,1]*l[0]*l[1]/18 - a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[5,1]*l[0]*l[2]/36 - a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[0,2]*l[1]*l[2]/36 - a[2]*c[4,2]*l[0]*l[1]/18 - a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[5,2]*l[0]*l[2]/36 - a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[0,3]*l[1]*l[2]/36 - a[3]*c[4,3]*l[0]*l[1]/18 - a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[5,3]*l[0]*l[2]/36 - a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[0,4]*l[1]*l[2]/36 - a[4]*c[4,4]*l[0]*l[1]/18 - a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[5,4]*l[0]*l[2]/36 - a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[0,5]*l[1]*l[2]/36 - a[5]*c[4,5]*l[0]*l[1]/18 - a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[5,5]*l[0]*l[2]/36 - a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/36,
           a[0]*c[0,0]*l[1]*l[2]/72 - a[0]*c[4,0]*l[0]*l[1]/36 - a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/72 - a[0]*c[5,0]*l[0]*l[2]/36 - a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/72 + a[1]*c[0,1]*l[1]*l[2]/72 - a[1]*c[4,1]*l[0]*l[1]/36 - a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/72 - a[1]*c[5,1]*l[0]*l[2]/36 - a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/72 + a[2]*c[0,2]*l[1]*l[2]/72 - a[2]*c[4,2]*l[0]*l[1]/36 - a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/72 - a[2]*c[5,2]*l[0]*l[2]/36 - a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/72 + a[3]*c[0,3]*l[1]*l[2]/72 - a[3]*c[4,3]*l[0]*l[1]/36 - a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/72 - a[3]*c[5,3]*l[0]*l[2]/36 - a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/72 + a[4]*c[0,4]*l[1]*l[2]/72 - a[4]*c[4,4]*l[0]*l[1]/36 - a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/72 - a[4]*c[5,4]*l[0]*l[2]/36 - a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/72 + a[5]*c[0,5]*l[1]*l[2]/72 - a[5]*c[4,5]*l[0]*l[1]/36 - a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/72 - a[5]*c[5,5]*l[0]*l[2]/36 - a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/72,
           a[0]*c[0,0]*l[1]*l[2]/72 - a[0]*c[4,0]*l[0]*l[1]/72 - a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/72 - a[0]*c[5,0]*l[0]*l[2]/72 - a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/72 + a[1]*c[0,1]*l[1]*l[2]/72 - a[1]*c[4,1]*l[0]*l[1]/72 - a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/72 - a[1]*c[5,1]*l[0]*l[2]/72 - a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/72 + a[2]*c[0,2]*l[1]*l[2]/72 - a[2]*c[4,2]*l[0]*l[1]/72 - a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/72 - a[2]*c[5,2]*l[0]*l[2]/72 - a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/72 + a[3]*c[0,3]*l[1]*l[2]/72 - a[3]*c[4,3]*l[0]*l[1]/72 - a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/72 - a[3]*c[5,3]*l[0]*l[2]/72 - a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/72 + a[4]*c[0,4]*l[1]*l[2]/72 - a[4]*c[4,4]*l[0]*l[1]/72 - a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/72 - a[4]*c[5,4]*l[0]*l[2]/72 - a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/72 + a[5]*c[0,5]*l[1]*l[2]/72 - a[5]*c[4,5]*l[0]*l[1]/72 - a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/72 - a[5]*c[5,5]*l[0]*l[2]/72 - a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/72],
          [-a[0]*c[1,0]*l[0]*l[2]/36 - a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/18 - a[0]*c[3,0]*l[0]*l[1]/36 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/18 + a[0]*c[5,0]*l[1]*l[2]/18 - a[1]*c[1,1]*l[0]*l[2]/36 - a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/18 - a[1]*c[3,1]*l[0]*l[1]/36 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/18 + a[1]*c[5,1]*l[1]*l[2]/18 - a[2]*c[1,2]*l[0]*l[2]/36 - a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/18 - a[2]*c[3,2]*l[0]*l[1]/36 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/18 + a[2]*c[5,2]*l[1]*l[2]/18 - a[3]*c[1,3]*l[0]*l[2]/36 - a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/18 - a[3]*c[3,3]*l[0]*l[1]/36 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/18 + a[3]*c[5,3]*l[1]*l[2]/18 - a[4]*c[1,4]*l[0]*l[2]/36 - a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/18 - a[4]*c[3,4]*l[0]*l[1]/36 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/18 + a[4]*c[5,4]*l[1]*l[2]/18 - a[5]*c[1,5]*l[0]*l[2]/36 - a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/18 - a[5]*c[3,5]*l[0]*l[1]/36 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/18 + a[5]*c[5,5]*l[1]*l[2]/18,
           -a[0]*c[1,0]*l[0]*l[2]/18 - a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/18 - a[0]*c[3,0]*l[0]*l[1]/18 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/18 + a[0]*c[5,0]*l[1]*l[2]/18 - a[1]*c[1,1]*l[0]*l[2]/18 - a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/18 - a[1]*c[3,1]*l[0]*l[1]/18 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/18 + a[1]*c[5,1]*l[1]*l[2]/18 - a[2]*c[1,2]*l[0]*l[2]/18 - a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/18 - a[2]*c[3,2]*l[0]*l[1]/18 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/18 + a[2]*c[5,2]*l[1]*l[2]/18 - a[3]*c[1,3]*l[0]*l[2]/18 - a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/18 - a[3]*c[3,3]*l[0]*l[1]/18 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/18 + a[3]*c[5,3]*l[1]*l[2]/18 - a[4]*c[1,4]*l[0]*l[2]/18 - a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/18 - a[4]*c[3,4]*l[0]*l[1]/18 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/18 + a[4]*c[5,4]*l[1]*l[2]/18 - a[5]*c[1,5]*l[0]*l[2]/18 - a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/18 - a[5]*c[3,5]*l[0]*l[1]/18 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/18 + a[5]*c[5,5]*l[1]*l[2]/18,
           -a[0]*c[1,0]*l[0]*l[2]/18 - a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/36 - a[0]*c[3,0]*l[0]*l[1]/36 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[5,0]*l[1]*l[2]/36 - a[1]*c[1,1]*l[0]*l[2]/18 - a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[3,1]*l[0]*l[1]/36 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[5,1]*l[1]*l[2]/36 - a[2]*c[1,2]*l[0]*l[2]/18 - a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[3,2]*l[0]*l[1]/36 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[5,2]*l[1]*l[2]/36 - a[3]*c[1,3]*l[0]*l[2]/18 - a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[3,3]*l[0]*l[1]/36 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[5,3]*l[1]*l[2]/36 - a[4]*c[1,4]*l[0]*l[2]/18 - a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[3,4]*l[0]*l[1]/36 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[5,4]*l[1]*l[2]/36 - a[5]*c[1,5]*l[0]*l[2]/18 - a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[3,5]*l[0]*l[1]/36 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[5,5]*l[1]*l[2]/36,
           -a[0]*c[1,0]*l[0]*l[2]/36 - a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/36 - a[0]*c[3,0]*l[0]*l[1]/72 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[5,0]*l[1]*l[2]/36 - a[1]*c[1,1]*l[0]*l[2]/36 - a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[3,1]*l[0]*l[1]/72 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[5,1]*l[1]*l[2]/36 - a[2]*c[1,2]*l[0]*l[2]/36 - a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[3,2]*l[0]*l[1]/72 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[5,2]*l[1]*l[2]/36 - a[3]*c[1,3]*l[0]*l[2]/36 - a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[3,3]*l[0]*l[1]/72 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[5,3]*l[1]*l[2]/36 - a[4]*c[1,4]*l[0]*l[2]/36 - a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[3,4]*l[0]*l[1]/72 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[5,4]*l[1]*l[2]/36 - a[5]*c[1,5]*l[0]*l[2]/36 - a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[3,5]*l[0]*l[1]/72 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[5,5]*l[1]*l[2]/36,
           -a[0]*c[1,0]*l[0]*l[2]/72 - a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/36 - a[0]*c[3,0]*l[0]*l[1]/36 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[5,0]*l[1]*l[2]/36 - a[1]*c[1,1]*l[0]*l[2]/72 - a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[3,1]*l[0]*l[1]/36 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[5,1]*l[1]*l[2]/36 - a[2]*c[1,2]*l[0]*l[2]/72 - a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[3,2]*l[0]*l[1]/36 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[5,2]*l[1]*l[2]/36 - a[3]*c[1,3]*l[0]*l[2]/72 - a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[3,3]*l[0]*l[1]/36 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[5,3]*l[1]*l[2]/36 - a[4]*c[1,4]*l[0]*l[2]/72 - a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[3,4]*l[0]*l[1]/36 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[5,4]*l[1]*l[2]/36 - a[5]*c[1,5]*l[0]*l[2]/72 - a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[3,5]*l[0]*l[1]/36 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[5,5]*l[1]*l[2]/36,
           -a[0]*c[1,0]*l[0]*l[2]/36 - a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/36 - a[0]*c[3,0]*l[0]*l[1]/18 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[5,0]*l[1]*l[2]/36 - a[1]*c[1,1]*l[0]*l[2]/36 - a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[3,1]*l[0]*l[1]/18 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[5,1]*l[1]*l[2]/36 - a[2]*c[1,2]*l[0]*l[2]/36 - a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[3,2]*l[0]*l[1]/18 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[5,2]*l[1]*l[2]/36 - a[3]*c[1,3]*l[0]*l[2]/36 - a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[3,3]*l[0]*l[1]/18 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[5,3]*l[1]*l[2]/36 - a[4]*c[1,4]*l[0]*l[2]/36 - a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[3,4]*l[0]*l[1]/18 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[5,4]*l[1]*l[2]/36 - a[5]*c[1,5]*l[0]*l[2]/36 - a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[3,5]*l[0]*l[1]/18 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[5,5]*l[1]*l[2]/36,
           -a[0]*c[1,0]*l[0]*l[2]/36 - a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/72 - a[0]*c[3,0]*l[0]*l[1]/36 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/72 + a[0]*c[5,0]*l[1]*l[2]/72 - a[1]*c[1,1]*l[0]*l[2]/36 - a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/72 - a[1]*c[3,1]*l[0]*l[1]/36 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/72 + a[1]*c[5,1]*l[1]*l[2]/72 - a[2]*c[1,2]*l[0]*l[2]/36 - a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/72 - a[2]*c[3,2]*l[0]*l[1]/36 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/72 + a[2]*c[5,2]*l[1]*l[2]/72 - a[3]*c[1,3]*l[0]*l[2]/36 - a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/72 - a[3]*c[3,3]*l[0]*l[1]/36 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/72 + a[3]*c[5,3]*l[1]*l[2]/72 - a[4]*c[1,4]*l[0]*l[2]/36 - a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/72 - a[4]*c[3,4]*l[0]*l[1]/36 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/72 + a[4]*c[5,4]*l[1]*l[2]/72 - a[5]*c[1,5]*l[0]*l[2]/36 - a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/72 - a[5]*c[3,5]*l[0]*l[1]/36 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/72 + a[5]*c[5,5]*l[1]*l[2]/72,
           -a[0]*c[1,0]*l[0]*l[2]/72 - a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/72 - a[0]*c[3,0]*l[0]*l[1]/72 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/72 + a[0]*c[5,0]*l[1]*l[2]/72 - a[1]*c[1,1]*l[0]*l[2]/72 - a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/72 - a[1]*c[3,1]*l[0]*l[1]/72 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/72 + a[1]*c[5,1]*l[1]*l[2]/72 - a[2]*c[1,2]*l[0]*l[2]/72 - a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/72 - a[2]*c[3,2]*l[0]*l[1]/72 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/72 + a[2]*c[5,2]*l[1]*l[2]/72 - a[3]*c[1,3]*l[0]*l[2]/72 - a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/72 - a[3]*c[3,3]*l[0]*l[1]/72 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/72 + a[3]*c[5,3]*l[1]*l[2]/72 - a[4]*c[1,4]*l[0]*l[2]/72 - a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/72 - a[4]*c[3,4]*l[0]*l[1]/72 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/72 + a[4]*c[5,4]*l[1]*l[2]/72 - a[5]*c[1,5]*l[0]*l[2]/72 - a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/72 - a[5]*c[3,5]*l[0]*l[1]/72 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/72 + a[5]*c[5,5]*l[1]*l[2]/72],
          [-a[0]*c[2,0]*l[0]*l[1]/36 - a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/18 - a[0]*c[3,0]*l[0]*l[2]/36 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/18 + a[0]*c[4,0]*l[1]*l[2]/18 - a[1]*c[2,1]*l[0]*l[1]/36 - a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/18 - a[1]*c[3,1]*l[0]*l[2]/36 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/18 + a[1]*c[4,1]*l[1]*l[2]/18 - a[2]*c[2,2]*l[0]*l[1]/36 - a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/18 - a[2]*c[3,2]*l[0]*l[2]/36 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/18 + a[2]*c[4,2]*l[1]*l[2]/18 - a[3]*c[2,3]*l[0]*l[1]/36 - a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/18 - a[3]*c[3,3]*l[0]*l[2]/36 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/18 + a[3]*c[4,3]*l[1]*l[2]/18 - a[4]*c[2,4]*l[0]*l[1]/36 - a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/18 - a[4]*c[3,4]*l[0]*l[2]/36 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/18 + a[4]*c[4,4]*l[1]*l[2]/18 - a[5]*c[2,5]*l[0]*l[1]/36 - a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/18 - a[5]*c[3,5]*l[0]*l[2]/36 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/18 + a[5]*c[4,5]*l[1]*l[2]/18,
           -a[0]*c[2,0]*l[0]*l[1]/18 - a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/18 - a[0]*c[3,0]*l[0]*l[2]/18 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/18 + a[0]*c[4,0]*l[1]*l[2]/18 - a[1]*c[2,1]*l[0]*l[1]/18 - a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/18 - a[1]*c[3,1]*l[0]*l[2]/18 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/18 + a[1]*c[4,1]*l[1]*l[2]/18 - a[2]*c[2,2]*l[0]*l[1]/18 - a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/18 - a[2]*c[3,2]*l[0]*l[2]/18 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/18 + a[2]*c[4,2]*l[1]*l[2]/18 - a[3]*c[2,3]*l[0]*l[1]/18 - a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/18 - a[3]*c[3,3]*l[0]*l[2]/18 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/18 + a[3]*c[4,3]*l[1]*l[2]/18 - a[4]*c[2,4]*l[0]*l[1]/18 - a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/18 - a[4]*c[3,4]*l[0]*l[2]/18 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/18 + a[4]*c[4,4]*l[1]*l[2]/18 - a[5]*c[2,5]*l[0]*l[1]/18 - a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/18 - a[5]*c[3,5]*l[0]*l[2]/18 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/18 + a[5]*c[4,5]*l[1]*l[2]/18,
           -a[0]*c[2,0]*l[0]*l[1]/36 - a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[3,0]*l[0]*l[2]/18 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/36 + a[0]*c[4,0]*l[1]*l[2]/36 - a[1]*c[2,1]*l[0]*l[1]/36 - a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[3,1]*l[0]*l[2]/18 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[4,1]*l[1]*l[2]/36 - a[2]*c[2,2]*l[0]*l[1]/36 - a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[3,2]*l[0]*l[2]/18 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[4,2]*l[1]*l[2]/36 - a[3]*c[2,3]*l[0]*l[1]/36 - a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[3,3]*l[0]*l[2]/18 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[4,3]*l[1]*l[2]/36 - a[4]*c[2,4]*l[0]*l[1]/36 - a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[3,4]*l[0]*l[2]/18 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[4,4]*l[1]*l[2]/36 - a[5]*c[2,5]*l[0]*l[1]/36 - a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[3,5]*l[0]*l[2]/18 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[4,5]*l[1]*l[2]/36,
           -a[0]*c[2,0]*l[0]*l[1]/72 - a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[3,0]*l[0]*l[2]/36 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/36 + a[0]*c[4,0]*l[1]*l[2]/36 - a[1]*c[2,1]*l[0]*l[1]/72 - a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[3,1]*l[0]*l[2]/36 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[4,1]*l[1]*l[2]/36 - a[2]*c[2,2]*l[0]*l[1]/72 - a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[3,2]*l[0]*l[2]/36 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[4,2]*l[1]*l[2]/36 - a[3]*c[2,3]*l[0]*l[1]/72 - a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[3,3]*l[0]*l[2]/36 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[4,3]*l[1]*l[2]/36 - a[4]*c[2,4]*l[0]*l[1]/72 - a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[3,4]*l[0]*l[2]/36 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[4,4]*l[1]*l[2]/36 - a[5]*c[2,5]*l[0]*l[1]/72 - a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[3,5]*l[0]*l[2]/36 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[4,5]*l[1]*l[2]/36,
           -a[0]*c[2,0]*l[0]*l[1]/36 - a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[3,0]*l[0]*l[2]/72 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/36 + a[0]*c[4,0]*l[1]*l[2]/36 - a[1]*c[2,1]*l[0]*l[1]/36 - a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[3,1]*l[0]*l[2]/72 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[4,1]*l[1]*l[2]/36 - a[2]*c[2,2]*l[0]*l[1]/36 - a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[3,2]*l[0]*l[2]/72 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[4,2]*l[1]*l[2]/36 - a[3]*c[2,3]*l[0]*l[1]/36 - a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[3,3]*l[0]*l[2]/72 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[4,3]*l[1]*l[2]/36 - a[4]*c[2,4]*l[0]*l[1]/36 - a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[3,4]*l[0]*l[2]/72 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[4,4]*l[1]*l[2]/36 - a[5]*c[2,5]*l[0]*l[1]/36 - a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[3,5]*l[0]*l[2]/72 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[4,5]*l[1]*l[2]/36,
           -a[0]*c[2,0]*l[0]*l[1]/18 - a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[3,0]*l[0]*l[2]/36 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/36 + a[0]*c[4,0]*l[1]*l[2]/36 - a[1]*c[2,1]*l[0]*l[1]/18 - a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[3,1]*l[0]*l[2]/36 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[4,1]*l[1]*l[2]/36 - a[2]*c[2,2]*l[0]*l[1]/18 - a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[3,2]*l[0]*l[2]/36 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[4,2]*l[1]*l[2]/36 - a[3]*c[2,3]*l[0]*l[1]/18 - a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[3,3]*l[0]*l[2]/36 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[4,3]*l[1]*l[2]/36 - a[4]*c[2,4]*l[0]*l[1]/18 - a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[3,4]*l[0]*l[2]/36 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[4,4]*l[1]*l[2]/36 - a[5]*c[2,5]*l[0]*l[1]/18 - a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[3,5]*l[0]*l[2]/36 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[4,5]*l[1]*l[2]/36,
           -a[0]*c[2,0]*l[0]*l[1]/36 - a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/72 - a[0]*c[3,0]*l[0]*l[2]/36 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/72 + a[0]*c[4,0]*l[1]*l[2]/72 - a[1]*c[2,1]*l[0]*l[1]/36 - a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/72 - a[1]*c[3,1]*l[0]*l[2]/36 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/72 + a[1]*c[4,1]*l[1]*l[2]/72 - a[2]*c[2,2]*l[0]*l[1]/36 - a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/72 - a[2]*c[3,2]*l[0]*l[2]/36 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/72 + a[2]*c[4,2]*l[1]*l[2]/72 - a[3]*c[2,3]*l[0]*l[1]/36 - a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/72 - a[3]*c[3,3]*l[0]*l[2]/36 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/72 + a[3]*c[4,3]*l[1]*l[2]/72 - a[4]*c[2,4]*l[0]*l[1]/36 - a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/72 - a[4]*c[3,4]*l[0]*l[2]/36 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/72 + a[4]*c[4,4]*l[1]*l[2]/72 - a[5]*c[2,5]*l[0]*l[1]/36 - a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/72 - a[5]*c[3,5]*l[0]*l[2]/36 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/72 + a[5]*c[4,5]*l[1]*l[2]/72,
           -a[0]*c[2,0]*l[0]*l[1]/72 - a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/72 - a[0]*c[3,0]*l[0]*l[2]/72 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/72 + a[0]*c[4,0]*l[1]*l[2]/72 - a[1]*c[2,1]*l[0]*l[1]/72 - a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/72 - a[1]*c[3,1]*l[0]*l[2]/72 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/72 + a[1]*c[4,1]*l[1]*l[2]/72 - a[2]*c[2,2]*l[0]*l[1]/72 - a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/72 - a[2]*c[3,2]*l[0]*l[2]/72 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/72 + a[2]*c[4,2]*l[1]*l[2]/72 - a[3]*c[2,3]*l[0]*l[1]/72 - a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/72 - a[3]*c[3,3]*l[0]*l[2]/72 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/72 + a[3]*c[4,3]*l[1]*l[2]/72 - a[4]*c[2,4]*l[0]*l[1]/72 - a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/72 - a[4]*c[3,4]*l[0]*l[2]/72 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/72 + a[4]*c[4,4]*l[1]*l[2]/72 - a[5]*c[2,5]*l[0]*l[1]/72 - a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/72 - a[5]*c[3,5]*l[0]*l[2]/72 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/72 + a[5]*c[4,5]*l[1]*l[2]/72],
          [a[0]*c[0,0]*l[1]*l[2]/36 - a[0]*c[4,0]*l[0]*l[1]/72 - a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[5,0]*l[0]*l[2]/36 - a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[0,1]*l[1]*l[2]/36 - a[1]*c[4,1]*l[0]*l[1]/72 - a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[5,1]*l[0]*l[2]/36 - a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[0,2]*l[1]*l[2]/36 - a[2]*c[4,2]*l[0]*l[1]/72 - a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[5,2]*l[0]*l[2]/36 - a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[0,3]*l[1]*l[2]/36 - a[3]*c[4,3]*l[0]*l[1]/72 - a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[5,3]*l[0]*l[2]/36 - a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[0,4]*l[1]*l[2]/36 - a[4]*c[4,4]*l[0]*l[1]/72 - a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[5,4]*l[0]*l[2]/36 - a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[0,5]*l[1]*l[2]/36 - a[5]*c[4,5]*l[0]*l[1]/72 - a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[5,5]*l[0]*l[2]/36 - a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/36,
           a[0]*c[0,0]*l[1]*l[2]/36 - a[0]*c[4,0]*l[0]*l[1]/36 - a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[5,0]*l[0]*l[2]/18 - a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[0,1]*l[1]*l[2]/36 - a[1]*c[4,1]*l[0]*l[1]/36 - a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[5,1]*l[0]*l[2]/18 - a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[0,2]*l[1]*l[2]/36 - a[2]*c[4,2]*l[0]*l[1]/36 - a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[5,2]*l[0]*l[2]/18 - a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[0,3]*l[1]*l[2]/36 - a[3]*c[4,3]*l[0]*l[1]/36 - a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[5,3]*l[0]*l[2]/18 - a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[0,4]*l[1]*l[2]/36 - a[4]*c[4,4]*l[0]*l[1]/36 - a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[5,4]*l[0]*l[2]/18 - a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[0,5]*l[1]*l[2]/36 - a[5]*c[4,5]*l[0]*l[1]/36 - a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[5,5]*l[0]*l[2]/18 - a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/36,
           a[0]*c[0,0]*l[1]*l[2]/18 - a[0]*c[4,0]*l[0]*l[1]/18 - a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/18 + a[0]*c[5,0]*l[0]*l[2]/18 - a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/18 + a[1]*c[0,1]*l[1]*l[2]/18 - a[1]*c[4,1]*l[0]*l[1]/18 - a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/18 + a[1]*c[5,1]*l[0]*l[2]/18 - a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/18 + a[2]*c[0,2]*l[1]*l[2]/18 - a[2]*c[4,2]*l[0]*l[1]/18 - a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/18 + a[2]*c[5,2]*l[0]*l[2]/18 - a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/18 + a[3]*c[0,3]*l[1]*l[2]/18 - a[3]*c[4,3]*l[0]*l[1]/18 - a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/18 + a[3]*c[5,3]*l[0]*l[2]/18 - a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/18 + a[4]*c[0,4]*l[1]*l[2]/18 - a[4]*c[4,4]*l[0]*l[1]/18 - a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/18 + a[4]*c[5,4]*l[0]*l[2]/18 - a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/18 + a[5]*c[0,5]*l[1]*l[2]/18 - a[5]*c[4,5]*l[0]*l[1]/18 - a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/18 + a[5]*c[5,5]*l[0]*l[2]/18 - a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/18,
           a[0]*c[0,0]*l[1]*l[2]/18 - a[0]*c[4,0]*l[0]*l[1]/36 - a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/18 + a[0]*c[5,0]*l[0]*l[2]/36 - a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/18 + a[1]*c[0,1]*l[1]*l[2]/18 - a[1]*c[4,1]*l[0]*l[1]/36 - a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/18 + a[1]*c[5,1]*l[0]*l[2]/36 - a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/18 + a[2]*c[0,2]*l[1]*l[2]/18 - a[2]*c[4,2]*l[0]*l[1]/36 - a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/18 + a[2]*c[5,2]*l[0]*l[2]/36 - a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/18 + a[3]*c[0,3]*l[1]*l[2]/18 - a[3]*c[4,3]*l[0]*l[1]/36 - a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/18 + a[3]*c[5,3]*l[0]*l[2]/36 - a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/18 + a[4]*c[0,4]*l[1]*l[2]/18 - a[4]*c[4,4]*l[0]*l[1]/36 - a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/18 + a[4]*c[5,4]*l[0]*l[2]/36 - a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/18 + a[5]*c[0,5]*l[1]*l[2]/18 - a[5]*c[4,5]*l[0]*l[1]/36 - a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/18 + a[5]*c[5,5]*l[0]*l[2]/36 - a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/18,
           a[0]*c[0,0]*l[1]*l[2]/72 - a[0]*c[4,0]*l[0]*l[1]/72 - a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/72 + a[0]*c[5,0]*l[0]*l[2]/72 - a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/72 + a[1]*c[0,1]*l[1]*l[2]/72 - a[1]*c[4,1]*l[0]*l[1]/72 - a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/72 + a[1]*c[5,1]*l[0]*l[2]/72 - a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/72 + a[2]*c[0,2]*l[1]*l[2]/72 - a[2]*c[4,2]*l[0]*l[1]/72 - a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/72 + a[2]*c[5,2]*l[0]*l[2]/72 - a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/72 + a[3]*c[0,3]*l[1]*l[2]/72 - a[3]*c[4,3]*l[0]*l[1]/72 - a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/72 + a[3]*c[5,3]*l[0]*l[2]/72 - a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/72 + a[4]*c[0,4]*l[1]*l[2]/72 - a[4]*c[4,4]*l[0]*l[1]/72 - a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/72 + a[4]*c[5,4]*l[0]*l[2]/72 - a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/72 + a[5]*c[0,5]*l[1]*l[2]/72 - a[5]*c[4,5]*l[0]*l[1]/72 - a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/72 + a[5]*c[5,5]*l[0]*l[2]/72 - a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/72,
           a[0]*c[0,0]*l[1]*l[2]/72 - a[0]*c[4,0]*l[0]*l[1]/36 - a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/72 + a[0]*c[5,0]*l[0]*l[2]/36 - a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/72 + a[1]*c[0,1]*l[1]*l[2]/72 - a[1]*c[4,1]*l[0]*l[1]/36 - a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/72 + a[1]*c[5,1]*l[0]*l[2]/36 - a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/72 + a[2]*c[0,2]*l[1]*l[2]/72 - a[2]*c[4,2]*l[0]*l[1]/36 - a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/72 + a[2]*c[5,2]*l[0]*l[2]/36 - a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/72 + a[3]*c[0,3]*l[1]*l[2]/72 - a[3]*c[4,3]*l[0]*l[1]/36 - a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/72 + a[3]*c[5,3]*l[0]*l[2]/36 - a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/72 + a[4]*c[0,4]*l[1]*l[2]/72 - a[4]*c[4,4]*l[0]*l[1]/36 - a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/72 + a[4]*c[5,4]*l[0]*l[2]/36 - a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/72 + a[5]*c[0,5]*l[1]*l[2]/72 - a[5]*c[4,5]*l[0]*l[1]/36 - a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/72 + a[5]*c[5,5]*l[0]*l[2]/36 - a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/72,
           a[0]*c[0,0]*l[1]*l[2]/36 - a[0]*c[4,0]*l[0]*l[1]/18 - a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[5,0]*l[0]*l[2]/36 - a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[0,1]*l[1]*l[2]/36 - a[1]*c[4,1]*l[0]*l[1]/18 - a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[5,1]*l[0]*l[2]/36 - a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[0,2]*l[1]*l[2]/36 - a[2]*c[4,2]*l[0]*l[1]/18 - a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[5,2]*l[0]*l[2]/36 - a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[0,3]*l[1]*l[2]/36 - a[3]*c[4,3]*l[0]*l[1]/18 - a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[5,3]*l[0]*l[2]/36 - a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[0,4]*l[1]*l[2]/36 - a[4]*c[4,4]*l[0]*l[1]/18 - a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[5,4]*l[0]*l[2]/36 - a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[0,5]*l[1]*l[2]/36 - a[5]*c[4,5]*l[0]*l[1]/18 - a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[5,5]*l[0]*l[2]/36 - a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/36,
           a[0]*c[0,0]*l[1]*l[2]/36 - a[0]*c[4,0]*l[0]*l[1]/36 - a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[5,0]*l[0]*l[2]/72 - a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[0,1]*l[1]*l[2]/36 - a[1]*c[4,1]*l[0]*l[1]/36 - a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[5,1]*l[0]*l[2]/72 - a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[0,2]*l[1]*l[2]/36 - a[2]*c[4,2]*l[0]*l[1]/36 - a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[5,2]*l[0]*l[2]/72 - a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[0,3]*l[1]*l[2]/36 - a[3]*c[4,3]*l[0]*l[1]/36 - a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[5,3]*l[0]*l[2]/72 - a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[0,4]*l[1]*l[2]/36 - a[4]*c[4,4]*l[0]*l[1]/36 - a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[5,4]*l[0]*l[2]/72 - a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[0,5]*l[1]*l[2]/36 - a[5]*c[4,5]*l[0]*l[1]/36 - a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[5,5]*l[0]*l[2]/72 - a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/36],
          [a[0]*c[1,0]*l[0]*l[2]/36 - a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/36 - a[0]*c[3,0]*l[0]*l[1]/72 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[5,0]*l[1]*l[2]/36 + a[1]*c[1,1]*l[0]*l[2]/36 - a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[3,1]*l[0]*l[1]/72 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[5,1]*l[1]*l[2]/36 + a[2]*c[1,2]*l[0]*l[2]/36 - a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[3,2]*l[0]*l[1]/72 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[5,2]*l[1]*l[2]/36 + a[3]*c[1,3]*l[0]*l[2]/36 - a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[3,3]*l[0]*l[1]/72 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[5,3]*l[1]*l[2]/36 + a[4]*c[1,4]*l[0]*l[2]/36 - a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[3,4]*l[0]*l[1]/72 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[5,4]*l[1]*l[2]/36 + a[5]*c[1,5]*l[0]*l[2]/36 - a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[3,5]*l[0]*l[1]/72 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[5,5]*l[1]*l[2]/36,
           a[0]*c[1,0]*l[0]*l[2]/18 - a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/36 - a[0]*c[3,0]*l[0]*l[1]/36 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[5,0]*l[1]*l[2]/36 + a[1]*c[1,1]*l[0]*l[2]/18 - a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[3,1]*l[0]*l[1]/36 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[5,1]*l[1]*l[2]/36 + a[2]*c[1,2]*l[0]*l[2]/18 - a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[3,2]*l[0]*l[1]/36 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[5,2]*l[1]*l[2]/36 + a[3]*c[1,3]*l[0]*l[2]/18 - a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[3,3]*l[0]*l[1]/36 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[5,3]*l[1]*l[2]/36 + a[4]*c[1,4]*l[0]*l[2]/18 - a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[3,4]*l[0]*l[1]/36 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[5,4]*l[1]*l[2]/36 + a[5]*c[1,5]*l[0]*l[2]/18 - a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[3,5]*l[0]*l[1]/36 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[5,5]*l[1]*l[2]/36,
           a[0]*c[1,0]*l[0]*l[2]/18 - a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/18 - a[0]*c[3,0]*l[0]*l[1]/18 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/18 + a[0]*c[5,0]*l[1]*l[2]/18 + a[1]*c[1,1]*l[0]*l[2]/18 - a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/18 - a[1]*c[3,1]*l[0]*l[1]/18 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/18 + a[1]*c[5,1]*l[1]*l[2]/18 + a[2]*c[1,2]*l[0]*l[2]/18 - a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/18 - a[2]*c[3,2]*l[0]*l[1]/18 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/18 + a[2]*c[5,2]*l[1]*l[2]/18 + a[3]*c[1,3]*l[0]*l[2]/18 - a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/18 - a[3]*c[3,3]*l[0]*l[1]/18 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/18 + a[3]*c[5,3]*l[1]*l[2]/18 + a[4]*c[1,4]*l[0]*l[2]/18 - a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/18 - a[4]*c[3,4]*l[0]*l[1]/18 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/18 + a[4]*c[5,4]*l[1]*l[2]/18 + a[5]*c[1,5]*l[0]*l[2]/18 - a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/18 - a[5]*c[3,5]*l[0]*l[1]/18 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/18 + a[5]*c[5,5]*l[1]*l[2]/18,
           a[0]*c[1,0]*l[0]*l[2]/36 - a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/18 - a[0]*c[3,0]*l[0]*l[1]/36 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/18 + a[0]*c[5,0]*l[1]*l[2]/18 + a[1]*c[1,1]*l[0]*l[2]/36 - a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/18 - a[1]*c[3,1]*l[0]*l[1]/36 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/18 + a[1]*c[5,1]*l[1]*l[2]/18 + a[2]*c[1,2]*l[0]*l[2]/36 - a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/18 - a[2]*c[3,2]*l[0]*l[1]/36 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/18 + a[2]*c[5,2]*l[1]*l[2]/18 + a[3]*c[1,3]*l[0]*l[2]/36 - a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/18 - a[3]*c[3,3]*l[0]*l[1]/36 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/18 + a[3]*c[5,3]*l[1]*l[2]/18 + a[4]*c[1,4]*l[0]*l[2]/36 - a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/18 - a[4]*c[3,4]*l[0]*l[1]/36 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/18 + a[4]*c[5,4]*l[1]*l[2]/18 + a[5]*c[1,5]*l[0]*l[2]/36 - a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/18 - a[5]*c[3,5]*l[0]*l[1]/36 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/18 + a[5]*c[5,5]*l[1]*l[2]/18,
           a[0]*c[1,0]*l[0]*l[2]/72 - a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/72 - a[0]*c[3,0]*l[0]*l[1]/72 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/72 + a[0]*c[5,0]*l[1]*l[2]/72 + a[1]*c[1,1]*l[0]*l[2]/72 - a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/72 - a[1]*c[3,1]*l[0]*l[1]/72 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/72 + a[1]*c[5,1]*l[1]*l[2]/72 + a[2]*c[1,2]*l[0]*l[2]/72 - a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/72 - a[2]*c[3,2]*l[0]*l[1]/72 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/72 + a[2]*c[5,2]*l[1]*l[2]/72 + a[3]*c[1,3]*l[0]*l[2]/72 - a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/72 - a[3]*c[3,3]*l[0]*l[1]/72 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/72 + a[3]*c[5,3]*l[1]*l[2]/72 + a[4]*c[1,4]*l[0]*l[2]/72 - a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/72 - a[4]*c[3,4]*l[0]*l[1]/72 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/72 + a[4]*c[5,4]*l[1]*l[2]/72 + a[5]*c[1,5]*l[0]*l[2]/72 - a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/72 - a[5]*c[3,5]*l[0]*l[1]/72 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/72 + a[5]*c[5,5]*l[1]*l[2]/72,
           a[0]*c[1,0]*l[0]*l[2]/36 - a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/72 - a[0]*c[3,0]*l[0]*l[1]/36 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/72 + a[0]*c[5,0]*l[1]*l[2]/72 + a[1]*c[1,1]*l[0]*l[2]/36 - a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/72 - a[1]*c[3,1]*l[0]*l[1]/36 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/72 + a[1]*c[5,1]*l[1]*l[2]/72 + a[2]*c[1,2]*l[0]*l[2]/36 - a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/72 - a[2]*c[3,2]*l[0]*l[1]/36 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/72 + a[2]*c[5,2]*l[1]*l[2]/72 + a[3]*c[1,3]*l[0]*l[2]/36 - a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/72 - a[3]*c[3,3]*l[0]*l[1]/36 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/72 + a[3]*c[5,3]*l[1]*l[2]/72 + a[4]*c[1,4]*l[0]*l[2]/36 - a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/72 - a[4]*c[3,4]*l[0]*l[1]/36 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/72 + a[4]*c[5,4]*l[1]*l[2]/72 + a[5]*c[1,5]*l[0]*l[2]/36 - a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/72 - a[5]*c[3,5]*l[0]*l[1]/36 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/72 + a[5]*c[5,5]*l[1]*l[2]/72,
           a[0]*c[1,0]*l[0]*l[2]/36 - a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/36 - a[0]*c[3,0]*l[0]*l[1]/18 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[5,0]*l[1]*l[2]/36 + a[1]*c[1,1]*l[0]*l[2]/36 - a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[3,1]*l[0]*l[1]/18 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[5,1]*l[1]*l[2]/36 + a[2]*c[1,2]*l[0]*l[2]/36 - a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[3,2]*l[0]*l[1]/18 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[5,2]*l[1]*l[2]/36 + a[3]*c[1,3]*l[0]*l[2]/36 - a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[3,3]*l[0]*l[1]/18 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[5,3]*l[1]*l[2]/36 + a[4]*c[1,4]*l[0]*l[2]/36 - a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[3,4]*l[0]*l[1]/18 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[5,4]*l[1]*l[2]/36 + a[5]*c[1,5]*l[0]*l[2]/36 - a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[3,5]*l[0]*l[1]/18 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[5,5]*l[1]*l[2]/36,
           a[0]*c[1,0]*l[0]*l[2]/72 - a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/36 - a[0]*c[3,0]*l[0]*l[1]/36 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[5,0]*l[1]*l[2]/36 + a[1]*c[1,1]*l[0]*l[2]/72 - a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[3,1]*l[0]*l[1]/36 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[5,1]*l[1]*l[2]/36 + a[2]*c[1,2]*l[0]*l[2]/72 - a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[3,2]*l[0]*l[1]/36 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[5,2]*l[1]*l[2]/36 + a[3]*c[1,3]*l[0]*l[2]/72 - a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[3,3]*l[0]*l[1]/36 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[5,3]*l[1]*l[2]/36 + a[4]*c[1,4]*l[0]*l[2]/72 - a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[3,4]*l[0]*l[1]/36 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[5,4]*l[1]*l[2]/36 + a[5]*c[1,5]*l[0]*l[2]/72 - a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[3,5]*l[0]*l[1]/36 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[5,5]*l[1]*l[2]/36],
          [-a[0]*c[2,0]*l[0]*l[1]/72 - a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[3,0]*l[0]*l[2]/36 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/36 + a[0]*c[4,0]*l[1]*l[2]/36 - a[1]*c[2,1]*l[0]*l[1]/72 - a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[3,1]*l[0]*l[2]/36 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[4,1]*l[1]*l[2]/36 - a[2]*c[2,2]*l[0]*l[1]/72 - a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[3,2]*l[0]*l[2]/36 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[4,2]*l[1]*l[2]/36 - a[3]*c[2,3]*l[0]*l[1]/72 - a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[3,3]*l[0]*l[2]/36 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[4,3]*l[1]*l[2]/36 - a[4]*c[2,4]*l[0]*l[1]/72 - a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[3,4]*l[0]*l[2]/36 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[4,4]*l[1]*l[2]/36 - a[5]*c[2,5]*l[0]*l[1]/72 - a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[3,5]*l[0]*l[2]/36 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[4,5]*l[1]*l[2]/36,
           -a[0]*c[2,0]*l[0]*l[1]/36 - a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[3,0]*l[0]*l[2]/18 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/36 + a[0]*c[4,0]*l[1]*l[2]/36 - a[1]*c[2,1]*l[0]*l[1]/36 - a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[3,1]*l[0]*l[2]/18 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[4,1]*l[1]*l[2]/36 - a[2]*c[2,2]*l[0]*l[1]/36 - a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[3,2]*l[0]*l[2]/18 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[4,2]*l[1]*l[2]/36 - a[3]*c[2,3]*l[0]*l[1]/36 - a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[3,3]*l[0]*l[2]/18 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[4,3]*l[1]*l[2]/36 - a[4]*c[2,4]*l[0]*l[1]/36 - a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[3,4]*l[0]*l[2]/18 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[4,4]*l[1]*l[2]/36 - a[5]*c[2,5]*l[0]*l[1]/36 - a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[3,5]*l[0]*l[2]/18 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[4,5]*l[1]*l[2]/36,
           -a[0]*c[2,0]*l[0]*l[1]/18 - a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/18 + a[0]*c[3,0]*l[0]*l[2]/18 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/18 + a[0]*c[4,0]*l[1]*l[2]/18 - a[1]*c[2,1]*l[0]*l[1]/18 - a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/18 + a[1]*c[3,1]*l[0]*l[2]/18 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/18 + a[1]*c[4,1]*l[1]*l[2]/18 - a[2]*c[2,2]*l[0]*l[1]/18 - a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/18 + a[2]*c[3,2]*l[0]*l[2]/18 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/18 + a[2]*c[4,2]*l[1]*l[2]/18 - a[3]*c[2,3]*l[0]*l[1]/18 - a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/18 + a[3]*c[3,3]*l[0]*l[2]/18 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/18 + a[3]*c[4,3]*l[1]*l[2]/18 - a[4]*c[2,4]*l[0]*l[1]/18 - a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/18 + a[4]*c[3,4]*l[0]*l[2]/18 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/18 + a[4]*c[4,4]*l[1]*l[2]/18 - a[5]*c[2,5]*l[0]*l[1]/18 - a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/18 + a[5]*c[3,5]*l[0]*l[2]/18 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/18 + a[5]*c[4,5]*l[1]*l[2]/18,
           -a[0]*c[2,0]*l[0]*l[1]/36 - a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/18 + a[0]*c[3,0]*l[0]*l[2]/36 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/18 + a[0]*c[4,0]*l[1]*l[2]/18 - a[1]*c[2,1]*l[0]*l[1]/36 - a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/18 + a[1]*c[3,1]*l[0]*l[2]/36 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/18 + a[1]*c[4,1]*l[1]*l[2]/18 - a[2]*c[2,2]*l[0]*l[1]/36 - a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/18 + a[2]*c[3,2]*l[0]*l[2]/36 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/18 + a[2]*c[4,2]*l[1]*l[2]/18 - a[3]*c[2,3]*l[0]*l[1]/36 - a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/18 + a[3]*c[3,3]*l[0]*l[2]/36 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/18 + a[3]*c[4,3]*l[1]*l[2]/18 - a[4]*c[2,4]*l[0]*l[1]/36 - a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/18 + a[4]*c[3,4]*l[0]*l[2]/36 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/18 + a[4]*c[4,4]*l[1]*l[2]/18 - a[5]*c[2,5]*l[0]*l[1]/36 - a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/18 + a[5]*c[3,5]*l[0]*l[2]/36 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/18 + a[5]*c[4,5]*l[1]*l[2]/18,
           -a[0]*c[2,0]*l[0]*l[1]/72 - a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/72 + a[0]*c[3,0]*l[0]*l[2]/72 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/72 + a[0]*c[4,0]*l[1]*l[2]/72 - a[1]*c[2,1]*l[0]*l[1]/72 - a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/72 + a[1]*c[3,1]*l[0]*l[2]/72 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/72 + a[1]*c[4,1]*l[1]*l[2]/72 - a[2]*c[2,2]*l[0]*l[1]/72 - a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/72 + a[2]*c[3,2]*l[0]*l[2]/72 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/72 + a[2]*c[4,2]*l[1]*l[2]/72 - a[3]*c[2,3]*l[0]*l[1]/72 - a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/72 + a[3]*c[3,3]*l[0]*l[2]/72 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/72 + a[3]*c[4,3]*l[1]*l[2]/72 - a[4]*c[2,4]*l[0]*l[1]/72 - a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/72 + a[4]*c[3,4]*l[0]*l[2]/72 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/72 + a[4]*c[4,4]*l[1]*l[2]/72 - a[5]*c[2,5]*l[0]*l[1]/72 - a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/72 + a[5]*c[3,5]*l[0]*l[2]/72 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/72 + a[5]*c[4,5]*l[1]*l[2]/72,
           -a[0]*c[2,0]*l[0]*l[1]/36 - a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/72 + a[0]*c[3,0]*l[0]*l[2]/36 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/72 + a[0]*c[4,0]*l[1]*l[2]/72 - a[1]*c[2,1]*l[0]*l[1]/36 - a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/72 + a[1]*c[3,1]*l[0]*l[2]/36 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/72 + a[1]*c[4,1]*l[1]*l[2]/72 - a[2]*c[2,2]*l[0]*l[1]/36 - a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/72 + a[2]*c[3,2]*l[0]*l[2]/36 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/72 + a[2]*c[4,2]*l[1]*l[2]/72 - a[3]*c[2,3]*l[0]*l[1]/36 - a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/72 + a[3]*c[3,3]*l[0]*l[2]/36 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/72 + a[3]*c[4,3]*l[1]*l[2]/72 - a[4]*c[2,4]*l[0]*l[1]/36 - a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/72 + a[4]*c[3,4]*l[0]*l[2]/36 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/72 + a[4]*c[4,4]*l[1]*l[2]/72 - a[5]*c[2,5]*l[0]*l[1]/36 - a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/72 + a[5]*c[3,5]*l[0]*l[2]/36 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/72 + a[5]*c[4,5]*l[1]*l[2]/72,
           -a[0]*c[2,0]*l[0]*l[1]/18 - a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[3,0]*l[0]*l[2]/36 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/36 + a[0]*c[4,0]*l[1]*l[2]/36 - a[1]*c[2,1]*l[0]*l[1]/18 - a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[3,1]*l[0]*l[2]/36 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[4,1]*l[1]*l[2]/36 - a[2]*c[2,2]*l[0]*l[1]/18 - a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[3,2]*l[0]*l[2]/36 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[4,2]*l[1]*l[2]/36 - a[3]*c[2,3]*l[0]*l[1]/18 - a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[3,3]*l[0]*l[2]/36 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[4,3]*l[1]*l[2]/36 - a[4]*c[2,4]*l[0]*l[1]/18 - a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[3,4]*l[0]*l[2]/36 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[4,4]*l[1]*l[2]/36 - a[5]*c[2,5]*l[0]*l[1]/18 - a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[3,5]*l[0]*l[2]/36 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[4,5]*l[1]*l[2]/36,
           -a[0]*c[2,0]*l[0]*l[1]/36 - a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[3,0]*l[0]*l[2]/72 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/36 + a[0]*c[4,0]*l[1]*l[2]/36 - a[1]*c[2,1]*l[0]*l[1]/36 - a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[3,1]*l[0]*l[2]/72 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[4,1]*l[1]*l[2]/36 - a[2]*c[2,2]*l[0]*l[1]/36 - a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[3,2]*l[0]*l[2]/72 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[4,2]*l[1]*l[2]/36 - a[3]*c[2,3]*l[0]*l[1]/36 - a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[3,3]*l[0]*l[2]/72 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[4,3]*l[1]*l[2]/36 - a[4]*c[2,4]*l[0]*l[1]/36 - a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[3,4]*l[0]*l[2]/72 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[4,4]*l[1]*l[2]/36 - a[5]*c[2,5]*l[0]*l[1]/36 - a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[3,5]*l[0]*l[2]/72 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[4,5]*l[1]*l[2]/36],
          [-a[0]*c[0,0]*l[1]*l[2]/36 - a[0]*c[4,0]*l[0]*l[1]/36 + a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[5,0]*l[0]*l[2]/18 + a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[0,1]*l[1]*l[2]/36 - a[1]*c[4,1]*l[0]*l[1]/36 + a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[5,1]*l[0]*l[2]/18 + a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[0,2]*l[1]*l[2]/36 - a[2]*c[4,2]*l[0]*l[1]/36 + a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[5,2]*l[0]*l[2]/18 + a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[0,3]*l[1]*l[2]/36 - a[3]*c[4,3]*l[0]*l[1]/36 + a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[5,3]*l[0]*l[2]/18 + a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[0,4]*l[1]*l[2]/36 - a[4]*c[4,4]*l[0]*l[1]/36 + a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[5,4]*l[0]*l[2]/18 + a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[0,5]*l[1]*l[2]/36 - a[5]*c[4,5]*l[0]*l[1]/36 + a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[5,5]*l[0]*l[2]/18 + a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/36,
           -a[0]*c[0,0]*l[1]*l[2]/36 - a[0]*c[4,0]*l[0]*l[1]/72 + a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[5,0]*l[0]*l[2]/36 + a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[0,1]*l[1]*l[2]/36 - a[1]*c[4,1]*l[0]*l[1]/72 + a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[5,1]*l[0]*l[2]/36 + a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[0,2]*l[1]*l[2]/36 - a[2]*c[4,2]*l[0]*l[1]/72 + a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[5,2]*l[0]*l[2]/36 + a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[0,3]*l[1]*l[2]/36 - a[3]*c[4,3]*l[0]*l[1]/72 + a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[5,3]*l[0]*l[2]/36 + a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[0,4]*l[1]*l[2]/36 - a[4]*c[4,4]*l[0]*l[1]/72 + a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[5,4]*l[0]*l[2]/36 + a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[0,5]*l[1]*l[2]/36 - a[5]*c[4,5]*l[0]*l[1]/72 + a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[5,5]*l[0]*l[2]/36 + a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/36,
           -a[0]*c[0,0]*l[1]*l[2]/18 - a[0]*c[4,0]*l[0]*l[1]/36 + a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/18 + a[0]*c[5,0]*l[0]*l[2]/36 + a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/18 - a[1]*c[0,1]*l[1]*l[2]/18 - a[1]*c[4,1]*l[0]*l[1]/36 + a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/18 + a[1]*c[5,1]*l[0]*l[2]/36 + a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/18 - a[2]*c[0,2]*l[1]*l[2]/18 - a[2]*c[4,2]*l[0]*l[1]/36 + a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/18 + a[2]*c[5,2]*l[0]*l[2]/36 + a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/18 - a[3]*c[0,3]*l[1]*l[2]/18 - a[3]*c[4,3]*l[0]*l[1]/36 + a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/18 + a[3]*c[5,3]*l[0]*l[2]/36 + a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/18 - a[4]*c[0,4]*l[1]*l[2]/18 - a[4]*c[4,4]*l[0]*l[1]/36 + a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/18 + a[4]*c[5,4]*l[0]*l[2]/36 + a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/18 - a[5]*c[0,5]*l[1]*l[2]/18 - a[5]*c[4,5]*l[0]*l[1]/36 + a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/18 + a[5]*c[5,5]*l[0]*l[2]/36 + a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/18,
           -a[0]*c[0,0]*l[1]*l[2]/18 - a[0]*c[4,0]*l[0]*l[1]/18 + a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/18 + a[0]*c[5,0]*l[0]*l[2]/18 + a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/18 - a[1]*c[0,1]*l[1]*l[2]/18 - a[1]*c[4,1]*l[0]*l[1]/18 + a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/18 + a[1]*c[5,1]*l[0]*l[2]/18 + a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/18 - a[2]*c[0,2]*l[1]*l[2]/18 - a[2]*c[4,2]*l[0]*l[1]/18 + a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/18 + a[2]*c[5,2]*l[0]*l[2]/18 + a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/18 - a[3]*c[0,3]*l[1]*l[2]/18 - a[3]*c[4,3]*l[0]*l[1]/18 + a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/18 + a[3]*c[5,3]*l[0]*l[2]/18 + a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/18 - a[4]*c[0,4]*l[1]*l[2]/18 - a[4]*c[4,4]*l[0]*l[1]/18 + a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/18 + a[4]*c[5,4]*l[0]*l[2]/18 + a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/18 - a[5]*c[0,5]*l[1]*l[2]/18 - a[5]*c[4,5]*l[0]*l[1]/18 + a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/18 + a[5]*c[5,5]*l[0]*l[2]/18 + a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/18,
           -a[0]*c[0,0]*l[1]*l[2]/72 - a[0]*c[4,0]*l[0]*l[1]/36 + a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/72 + a[0]*c[5,0]*l[0]*l[2]/36 + a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/72 - a[1]*c[0,1]*l[1]*l[2]/72 - a[1]*c[4,1]*l[0]*l[1]/36 + a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/72 + a[1]*c[5,1]*l[0]*l[2]/36 + a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/72 - a[2]*c[0,2]*l[1]*l[2]/72 - a[2]*c[4,2]*l[0]*l[1]/36 + a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/72 + a[2]*c[5,2]*l[0]*l[2]/36 + a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/72 - a[3]*c[0,3]*l[1]*l[2]/72 - a[3]*c[4,3]*l[0]*l[1]/36 + a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/72 + a[3]*c[5,3]*l[0]*l[2]/36 + a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/72 - a[4]*c[0,4]*l[1]*l[2]/72 - a[4]*c[4,4]*l[0]*l[1]/36 + a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/72 + a[4]*c[5,4]*l[0]*l[2]/36 + a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/72 - a[5]*c[0,5]*l[1]*l[2]/72 - a[5]*c[4,5]*l[0]*l[1]/36 + a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/72 + a[5]*c[5,5]*l[0]*l[2]/36 + a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/72,
           -a[0]*c[0,0]*l[1]*l[2]/72 - a[0]*c[4,0]*l[0]*l[1]/72 + a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/72 + a[0]*c[5,0]*l[0]*l[2]/72 + a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/72 - a[1]*c[0,1]*l[1]*l[2]/72 - a[1]*c[4,1]*l[0]*l[1]/72 + a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/72 + a[1]*c[5,1]*l[0]*l[2]/72 + a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/72 - a[2]*c[0,2]*l[1]*l[2]/72 - a[2]*c[4,2]*l[0]*l[1]/72 + a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/72 + a[2]*c[5,2]*l[0]*l[2]/72 + a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/72 - a[3]*c[0,3]*l[1]*l[2]/72 - a[3]*c[4,3]*l[0]*l[1]/72 + a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/72 + a[3]*c[5,3]*l[0]*l[2]/72 + a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/72 - a[4]*c[0,4]*l[1]*l[2]/72 - a[4]*c[4,4]*l[0]*l[1]/72 + a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/72 + a[4]*c[5,4]*l[0]*l[2]/72 + a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/72 - a[5]*c[0,5]*l[1]*l[2]/72 - a[5]*c[4,5]*l[0]*l[1]/72 + a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/72 + a[5]*c[5,5]*l[0]*l[2]/72 + a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/72,
           -a[0]*c[0,0]*l[1]*l[2]/36 - a[0]*c[4,0]*l[0]*l[1]/36 + a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[5,0]*l[0]*l[2]/72 + a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[0,1]*l[1]*l[2]/36 - a[1]*c[4,1]*l[0]*l[1]/36 + a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[5,1]*l[0]*l[2]/72 + a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[0,2]*l[1]*l[2]/36 - a[2]*c[4,2]*l[0]*l[1]/36 + a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[5,2]*l[0]*l[2]/72 + a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[0,3]*l[1]*l[2]/36 - a[3]*c[4,3]*l[0]*l[1]/36 + a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[5,3]*l[0]*l[2]/72 + a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[0,4]*l[1]*l[2]/36 - a[4]*c[4,4]*l[0]*l[1]/36 + a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[5,4]*l[0]*l[2]/72 + a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[0,5]*l[1]*l[2]/36 - a[5]*c[4,5]*l[0]*l[1]/36 + a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[5,5]*l[0]*l[2]/72 + a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/36,
           -a[0]*c[0,0]*l[1]*l[2]/36 - a[0]*c[4,0]*l[0]*l[1]/18 + a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[5,0]*l[0]*l[2]/36 + a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[0,1]*l[1]*l[2]/36 - a[1]*c[4,1]*l[0]*l[1]/18 + a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[5,1]*l[0]*l[2]/36 + a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[0,2]*l[1]*l[2]/36 - a[2]*c[4,2]*l[0]*l[1]/18 + a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[5,2]*l[0]*l[2]/36 + a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[0,3]*l[1]*l[2]/36 - a[3]*c[4,3]*l[0]*l[1]/18 + a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[5,3]*l[0]*l[2]/36 + a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[0,4]*l[1]*l[2]/36 - a[4]*c[4,4]*l[0]*l[1]/18 + a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[5,4]*l[0]*l[2]/36 + a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[0,5]*l[1]*l[2]/36 - a[5]*c[4,5]*l[0]*l[1]/18 + a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[5,5]*l[0]*l[2]/36 + a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/36],
          [a[0]*c[1,0]*l[0]*l[2]/18 + a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/36 - a[0]*c[3,0]*l[0]*l[1]/36 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[5,0]*l[1]*l[2]/36 + a[1]*c[1,1]*l[0]*l[2]/18 + a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[3,1]*l[0]*l[1]/36 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[5,1]*l[1]*l[2]/36 + a[2]*c[1,2]*l[0]*l[2]/18 + a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[3,2]*l[0]*l[1]/36 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[5,2]*l[1]*l[2]/36 + a[3]*c[1,3]*l[0]*l[2]/18 + a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[3,3]*l[0]*l[1]/36 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[5,3]*l[1]*l[2]/36 + a[4]*c[1,4]*l[0]*l[2]/18 + a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[3,4]*l[0]*l[1]/36 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[5,4]*l[1]*l[2]/36 + a[5]*c[1,5]*l[0]*l[2]/18 + a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[3,5]*l[0]*l[1]/36 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[5,5]*l[1]*l[2]/36,
           a[0]*c[1,0]*l[0]*l[2]/36 + a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/36 - a[0]*c[3,0]*l[0]*l[1]/72 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[5,0]*l[1]*l[2]/36 + a[1]*c[1,1]*l[0]*l[2]/36 + a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[3,1]*l[0]*l[1]/72 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[5,1]*l[1]*l[2]/36 + a[2]*c[1,2]*l[0]*l[2]/36 + a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[3,2]*l[0]*l[1]/72 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[5,2]*l[1]*l[2]/36 + a[3]*c[1,3]*l[0]*l[2]/36 + a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[3,3]*l[0]*l[1]/72 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[5,3]*l[1]*l[2]/36 + a[4]*c[1,4]*l[0]*l[2]/36 + a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[3,4]*l[0]*l[1]/72 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[5,4]*l[1]*l[2]/36 + a[5]*c[1,5]*l[0]*l[2]/36 + a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[3,5]*l[0]*l[1]/72 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[5,5]*l[1]*l[2]/36,
           a[0]*c[1,0]*l[0]*l[2]/36 + a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/18 - a[0]*c[3,0]*l[0]*l[1]/36 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/18 - a[0]*c[5,0]*l[1]*l[2]/18 + a[1]*c[1,1]*l[0]*l[2]/36 + a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/18 - a[1]*c[3,1]*l[0]*l[1]/36 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/18 - a[1]*c[5,1]*l[1]*l[2]/18 + a[2]*c[1,2]*l[0]*l[2]/36 + a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/18 - a[2]*c[3,2]*l[0]*l[1]/36 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/18 - a[2]*c[5,2]*l[1]*l[2]/18 + a[3]*c[1,3]*l[0]*l[2]/36 + a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/18 - a[3]*c[3,3]*l[0]*l[1]/36 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/18 - a[3]*c[5,3]*l[1]*l[2]/18 + a[4]*c[1,4]*l[0]*l[2]/36 + a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/18 - a[4]*c[3,4]*l[0]*l[1]/36 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/18 - a[4]*c[5,4]*l[1]*l[2]/18 + a[5]*c[1,5]*l[0]*l[2]/36 + a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/18 - a[5]*c[3,5]*l[0]*l[1]/36 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/18 - a[5]*c[5,5]*l[1]*l[2]/18,
           a[0]*c[1,0]*l[0]*l[2]/18 + a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/18 - a[0]*c[3,0]*l[0]*l[1]/18 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/18 - a[0]*c[5,0]*l[1]*l[2]/18 + a[1]*c[1,1]*l[0]*l[2]/18 + a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/18 - a[1]*c[3,1]*l[0]*l[1]/18 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/18 - a[1]*c[5,1]*l[1]*l[2]/18 + a[2]*c[1,2]*l[0]*l[2]/18 + a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/18 - a[2]*c[3,2]*l[0]*l[1]/18 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/18 - a[2]*c[5,2]*l[1]*l[2]/18 + a[3]*c[1,3]*l[0]*l[2]/18 + a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/18 - a[3]*c[3,3]*l[0]*l[1]/18 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/18 - a[3]*c[5,3]*l[1]*l[2]/18 + a[4]*c[1,4]*l[0]*l[2]/18 + a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/18 - a[4]*c[3,4]*l[0]*l[1]/18 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/18 - a[4]*c[5,4]*l[1]*l[2]/18 + a[5]*c[1,5]*l[0]*l[2]/18 + a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/18 - a[5]*c[3,5]*l[0]*l[1]/18 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/18 - a[5]*c[5,5]*l[1]*l[2]/18,
           a[0]*c[1,0]*l[0]*l[2]/36 + a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/72 - a[0]*c[3,0]*l[0]*l[1]/36 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/72 - a[0]*c[5,0]*l[1]*l[2]/72 + a[1]*c[1,1]*l[0]*l[2]/36 + a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/72 - a[1]*c[3,1]*l[0]*l[1]/36 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/72 - a[1]*c[5,1]*l[1]*l[2]/72 + a[2]*c[1,2]*l[0]*l[2]/36 + a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/72 - a[2]*c[3,2]*l[0]*l[1]/36 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/72 - a[2]*c[5,2]*l[1]*l[2]/72 + a[3]*c[1,3]*l[0]*l[2]/36 + a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/72 - a[3]*c[3,3]*l[0]*l[1]/36 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/72 - a[3]*c[5,3]*l[1]*l[2]/72 + a[4]*c[1,4]*l[0]*l[2]/36 + a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/72 - a[4]*c[3,4]*l[0]*l[1]/36 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/72 - a[4]*c[5,4]*l[1]*l[2]/72 + a[5]*c[1,5]*l[0]*l[2]/36 + a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/72 - a[5]*c[3,5]*l[0]*l[1]/36 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/72 - a[5]*c[5,5]*l[1]*l[2]/72,
           a[0]*c[1,0]*l[0]*l[2]/72 + a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/72 - a[0]*c[3,0]*l[0]*l[1]/72 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/72 - a[0]*c[5,0]*l[1]*l[2]/72 + a[1]*c[1,1]*l[0]*l[2]/72 + a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/72 - a[1]*c[3,1]*l[0]*l[1]/72 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/72 - a[1]*c[5,1]*l[1]*l[2]/72 + a[2]*c[1,2]*l[0]*l[2]/72 + a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/72 - a[2]*c[3,2]*l[0]*l[1]/72 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/72 - a[2]*c[5,2]*l[1]*l[2]/72 + a[3]*c[1,3]*l[0]*l[2]/72 + a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/72 - a[3]*c[3,3]*l[0]*l[1]/72 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/72 - a[3]*c[5,3]*l[1]*l[2]/72 + a[4]*c[1,4]*l[0]*l[2]/72 + a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/72 - a[4]*c[3,4]*l[0]*l[1]/72 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/72 - a[4]*c[5,4]*l[1]*l[2]/72 + a[5]*c[1,5]*l[0]*l[2]/72 + a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/72 - a[5]*c[3,5]*l[0]*l[1]/72 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/72 - a[5]*c[5,5]*l[1]*l[2]/72,
           a[0]*c[1,0]*l[0]*l[2]/72 + a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/36 - a[0]*c[3,0]*l[0]*l[1]/36 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[5,0]*l[1]*l[2]/36 + a[1]*c[1,1]*l[0]*l[2]/72 + a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[3,1]*l[0]*l[1]/36 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[5,1]*l[1]*l[2]/36 + a[2]*c[1,2]*l[0]*l[2]/72 + a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[3,2]*l[0]*l[1]/36 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[5,2]*l[1]*l[2]/36 + a[3]*c[1,3]*l[0]*l[2]/72 + a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[3,3]*l[0]*l[1]/36 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[5,3]*l[1]*l[2]/36 + a[4]*c[1,4]*l[0]*l[2]/72 + a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[3,4]*l[0]*l[1]/36 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[5,4]*l[1]*l[2]/36 + a[5]*c[1,5]*l[0]*l[2]/72 + a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[3,5]*l[0]*l[1]/36 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[5,5]*l[1]*l[2]/36,
           a[0]*c[1,0]*l[0]*l[2]/36 + a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/36 - a[0]*c[3,0]*l[0]*l[1]/18 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[5,0]*l[1]*l[2]/36 + a[1]*c[1,1]*l[0]*l[2]/36 + a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[3,1]*l[0]*l[1]/18 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[5,1]*l[1]*l[2]/36 + a[2]*c[1,2]*l[0]*l[2]/36 + a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[3,2]*l[0]*l[1]/18 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[5,2]*l[1]*l[2]/36 + a[3]*c[1,3]*l[0]*l[2]/36 + a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[3,3]*l[0]*l[1]/18 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[5,3]*l[1]*l[2]/36 + a[4]*c[1,4]*l[0]*l[2]/36 + a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[3,4]*l[0]*l[1]/18 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[5,4]*l[1]*l[2]/36 + a[5]*c[1,5]*l[0]*l[2]/36 + a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[3,5]*l[0]*l[1]/18 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[5,5]*l[1]*l[2]/36],
          [-a[0]*c[2,0]*l[0]*l[1]/36 + a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[3,0]*l[0]*l[2]/18 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/36 - a[0]*c[4,0]*l[1]*l[2]/36 - a[1]*c[2,1]*l[0]*l[1]/36 + a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[3,1]*l[0]*l[2]/18 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[4,1]*l[1]*l[2]/36 - a[2]*c[2,2]*l[0]*l[1]/36 + a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[3,2]*l[0]*l[2]/18 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[4,2]*l[1]*l[2]/36 - a[3]*c[2,3]*l[0]*l[1]/36 + a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[3,3]*l[0]*l[2]/18 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[4,3]*l[1]*l[2]/36 - a[4]*c[2,4]*l[0]*l[1]/36 + a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[3,4]*l[0]*l[2]/18 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[4,4]*l[1]*l[2]/36 - a[5]*c[2,5]*l[0]*l[1]/36 + a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[3,5]*l[0]*l[2]/18 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[4,5]*l[1]*l[2]/36,
           -a[0]*c[2,0]*l[0]*l[1]/72 + a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[3,0]*l[0]*l[2]/36 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/36 - a[0]*c[4,0]*l[1]*l[2]/36 - a[1]*c[2,1]*l[0]*l[1]/72 + a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[3,1]*l[0]*l[2]/36 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[4,1]*l[1]*l[2]/36 - a[2]*c[2,2]*l[0]*l[1]/72 + a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[3,2]*l[0]*l[2]/36 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[4,2]*l[1]*l[2]/36 - a[3]*c[2,3]*l[0]*l[1]/72 + a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[3,3]*l[0]*l[2]/36 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[4,3]*l[1]*l[2]/36 - a[4]*c[2,4]*l[0]*l[1]/72 + a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[3,4]*l[0]*l[2]/36 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[4,4]*l[1]*l[2]/36 - a[5]*c[2,5]*l[0]*l[1]/72 + a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[3,5]*l[0]*l[2]/36 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[4,5]*l[1]*l[2]/36,
           -a[0]*c[2,0]*l[0]*l[1]/36 + a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/18 + a[0]*c[3,0]*l[0]*l[2]/36 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/18 - a[0]*c[4,0]*l[1]*l[2]/18 - a[1]*c[2,1]*l[0]*l[1]/36 + a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/18 + a[1]*c[3,1]*l[0]*l[2]/36 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/18 - a[1]*c[4,1]*l[1]*l[2]/18 - a[2]*c[2,2]*l[0]*l[1]/36 + a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/18 + a[2]*c[3,2]*l[0]*l[2]/36 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/18 - a[2]*c[4,2]*l[1]*l[2]/18 - a[3]*c[2,3]*l[0]*l[1]/36 + a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/18 + a[3]*c[3,3]*l[0]*l[2]/36 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/18 - a[3]*c[4,3]*l[1]*l[2]/18 - a[4]*c[2,4]*l[0]*l[1]/36 + a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/18 + a[4]*c[3,4]*l[0]*l[2]/36 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/18 - a[4]*c[4,4]*l[1]*l[2]/18 - a[5]*c[2,5]*l[0]*l[1]/36 + a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/18 + a[5]*c[3,5]*l[0]*l[2]/36 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/18 - a[5]*c[4,5]*l[1]*l[2]/18,
           -a[0]*c[2,0]*l[0]*l[1]/18 + a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/18 + a[0]*c[3,0]*l[0]*l[2]/18 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/18 - a[0]*c[4,0]*l[1]*l[2]/18 - a[1]*c[2,1]*l[0]*l[1]/18 + a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/18 + a[1]*c[3,1]*l[0]*l[2]/18 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/18 - a[1]*c[4,1]*l[1]*l[2]/18 - a[2]*c[2,2]*l[0]*l[1]/18 + a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/18 + a[2]*c[3,2]*l[0]*l[2]/18 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/18 - a[2]*c[4,2]*l[1]*l[2]/18 - a[3]*c[2,3]*l[0]*l[1]/18 + a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/18 + a[3]*c[3,3]*l[0]*l[2]/18 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/18 - a[3]*c[4,3]*l[1]*l[2]/18 - a[4]*c[2,4]*l[0]*l[1]/18 + a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/18 + a[4]*c[3,4]*l[0]*l[2]/18 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/18 - a[4]*c[4,4]*l[1]*l[2]/18 - a[5]*c[2,5]*l[0]*l[1]/18 + a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/18 + a[5]*c[3,5]*l[0]*l[2]/18 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/18 - a[5]*c[4,5]*l[1]*l[2]/18,
           -a[0]*c[2,0]*l[0]*l[1]/36 + a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/72 + a[0]*c[3,0]*l[0]*l[2]/36 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/72 - a[0]*c[4,0]*l[1]*l[2]/72 - a[1]*c[2,1]*l[0]*l[1]/36 + a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/72 + a[1]*c[3,1]*l[0]*l[2]/36 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/72 - a[1]*c[4,1]*l[1]*l[2]/72 - a[2]*c[2,2]*l[0]*l[1]/36 + a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/72 + a[2]*c[3,2]*l[0]*l[2]/36 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/72 - a[2]*c[4,2]*l[1]*l[2]/72 - a[3]*c[2,3]*l[0]*l[1]/36 + a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/72 + a[3]*c[3,3]*l[0]*l[2]/36 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/72 - a[3]*c[4,3]*l[1]*l[2]/72 - a[4]*c[2,4]*l[0]*l[1]/36 + a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/72 + a[4]*c[3,4]*l[0]*l[2]/36 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/72 - a[4]*c[4,4]*l[1]*l[2]/72 - a[5]*c[2,5]*l[0]*l[1]/36 + a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/72 + a[5]*c[3,5]*l[0]*l[2]/36 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/72 - a[5]*c[4,5]*l[1]*l[2]/72,
           -a[0]*c[2,0]*l[0]*l[1]/72 + a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/72 + a[0]*c[3,0]*l[0]*l[2]/72 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/72 - a[0]*c[4,0]*l[1]*l[2]/72 - a[1]*c[2,1]*l[0]*l[1]/72 + a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/72 + a[1]*c[3,1]*l[0]*l[2]/72 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/72 - a[1]*c[4,1]*l[1]*l[2]/72 - a[2]*c[2,2]*l[0]*l[1]/72 + a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/72 + a[2]*c[3,2]*l[0]*l[2]/72 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/72 - a[2]*c[4,2]*l[1]*l[2]/72 - a[3]*c[2,3]*l[0]*l[1]/72 + a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/72 + a[3]*c[3,3]*l[0]*l[2]/72 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/72 - a[3]*c[4,3]*l[1]*l[2]/72 - a[4]*c[2,4]*l[0]*l[1]/72 + a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/72 + a[4]*c[3,4]*l[0]*l[2]/72 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/72 - a[4]*c[4,4]*l[1]*l[2]/72 - a[5]*c[2,5]*l[0]*l[1]/72 + a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/72 + a[5]*c[3,5]*l[0]*l[2]/72 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/72 - a[5]*c[4,5]*l[1]*l[2]/72,
           -a[0]*c[2,0]*l[0]*l[1]/36 + a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[3,0]*l[0]*l[2]/72 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/36 - a[0]*c[4,0]*l[1]*l[2]/36 - a[1]*c[2,1]*l[0]*l[1]/36 + a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[3,1]*l[0]*l[2]/72 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[4,1]*l[1]*l[2]/36 - a[2]*c[2,2]*l[0]*l[1]/36 + a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[3,2]*l[0]*l[2]/72 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[4,2]*l[1]*l[2]/36 - a[3]*c[2,3]*l[0]*l[1]/36 + a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[3,3]*l[0]*l[2]/72 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[4,3]*l[1]*l[2]/36 - a[4]*c[2,4]*l[0]*l[1]/36 + a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[3,4]*l[0]*l[2]/72 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[4,4]*l[1]*l[2]/36 - a[5]*c[2,5]*l[0]*l[1]/36 + a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[3,5]*l[0]*l[2]/72 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[4,5]*l[1]*l[2]/36,
           -a[0]*c[2,0]*l[0]*l[1]/18 + a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[3,0]*l[0]*l[2]/36 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/36 - a[0]*c[4,0]*l[1]*l[2]/36 - a[1]*c[2,1]*l[0]*l[1]/18 + a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[3,1]*l[0]*l[2]/36 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[4,1]*l[1]*l[2]/36 - a[2]*c[2,2]*l[0]*l[1]/18 + a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[3,2]*l[0]*l[2]/36 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[4,2]*l[1]*l[2]/36 - a[3]*c[2,3]*l[0]*l[1]/18 + a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[3,3]*l[0]*l[2]/36 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[4,3]*l[1]*l[2]/36 - a[4]*c[2,4]*l[0]*l[1]/18 + a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[3,4]*l[0]*l[2]/36 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[4,4]*l[1]*l[2]/36 - a[5]*c[2,5]*l[0]*l[1]/18 + a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[3,5]*l[0]*l[2]/36 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[4,5]*l[1]*l[2]/36],
          [-a[0]*c[0,0]*l[1]*l[2]/36 + a[0]*c[4,0]*l[0]*l[1]/18 + a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[5,0]*l[0]*l[2]/36 + a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[0,1]*l[1]*l[2]/36 + a[1]*c[4,1]*l[0]*l[1]/18 + a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[5,1]*l[0]*l[2]/36 + a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[0,2]*l[1]*l[2]/36 + a[2]*c[4,2]*l[0]*l[1]/18 + a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[5,2]*l[0]*l[2]/36 + a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[0,3]*l[1]*l[2]/36 + a[3]*c[4,3]*l[0]*l[1]/18 + a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[5,3]*l[0]*l[2]/36 + a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[0,4]*l[1]*l[2]/36 + a[4]*c[4,4]*l[0]*l[1]/18 + a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[5,4]*l[0]*l[2]/36 + a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[0,5]*l[1]*l[2]/36 + a[5]*c[4,5]*l[0]*l[1]/18 + a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[5,5]*l[0]*l[2]/36 + a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/36,
           -a[0]*c[0,0]*l[1]*l[2]/36 + a[0]*c[4,0]*l[0]*l[1]/36 + a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[5,0]*l[0]*l[2]/72 + a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[0,1]*l[1]*l[2]/36 + a[1]*c[4,1]*l[0]*l[1]/36 + a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[5,1]*l[0]*l[2]/72 + a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[0,2]*l[1]*l[2]/36 + a[2]*c[4,2]*l[0]*l[1]/36 + a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[5,2]*l[0]*l[2]/72 + a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[0,3]*l[1]*l[2]/36 + a[3]*c[4,3]*l[0]*l[1]/36 + a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[5,3]*l[0]*l[2]/72 + a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[0,4]*l[1]*l[2]/36 + a[4]*c[4,4]*l[0]*l[1]/36 + a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[5,4]*l[0]*l[2]/72 + a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[0,5]*l[1]*l[2]/36 + a[5]*c[4,5]*l[0]*l[1]/36 + a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[5,5]*l[0]*l[2]/72 + a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/36,
           -a[0]*c[0,0]*l[1]*l[2]/72 + a[0]*c[4,0]*l[0]*l[1]/72 + a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/72 - a[0]*c[5,0]*l[0]*l[2]/72 + a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/72 - a[1]*c[0,1]*l[1]*l[2]/72 + a[1]*c[4,1]*l[0]*l[1]/72 + a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/72 - a[1]*c[5,1]*l[0]*l[2]/72 + a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/72 - a[2]*c[0,2]*l[1]*l[2]/72 + a[2]*c[4,2]*l[0]*l[1]/72 + a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/72 - a[2]*c[5,2]*l[0]*l[2]/72 + a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/72 - a[3]*c[0,3]*l[1]*l[2]/72 + a[3]*c[4,3]*l[0]*l[1]/72 + a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/72 - a[3]*c[5,3]*l[0]*l[2]/72 + a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/72 - a[4]*c[0,4]*l[1]*l[2]/72 + a[4]*c[4,4]*l[0]*l[1]/72 + a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/72 - a[4]*c[5,4]*l[0]*l[2]/72 + a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/72 - a[5]*c[0,5]*l[1]*l[2]/72 + a[5]*c[4,5]*l[0]*l[1]/72 + a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/72 - a[5]*c[5,5]*l[0]*l[2]/72 + a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/72,
           -a[0]*c[0,0]*l[1]*l[2]/72 + a[0]*c[4,0]*l[0]*l[1]/36 + a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/72 - a[0]*c[5,0]*l[0]*l[2]/36 + a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/72 - a[1]*c[0,1]*l[1]*l[2]/72 + a[1]*c[4,1]*l[0]*l[1]/36 + a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/72 - a[1]*c[5,1]*l[0]*l[2]/36 + a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/72 - a[2]*c[0,2]*l[1]*l[2]/72 + a[2]*c[4,2]*l[0]*l[1]/36 + a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/72 - a[2]*c[5,2]*l[0]*l[2]/36 + a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/72 - a[3]*c[0,3]*l[1]*l[2]/72 + a[3]*c[4,3]*l[0]*l[1]/36 + a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/72 - a[3]*c[5,3]*l[0]*l[2]/36 + a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/72 - a[4]*c[0,4]*l[1]*l[2]/72 + a[4]*c[4,4]*l[0]*l[1]/36 + a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/72 - a[4]*c[5,4]*l[0]*l[2]/36 + a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/72 - a[5]*c[0,5]*l[1]*l[2]/72 + a[5]*c[4,5]*l[0]*l[1]/36 + a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/72 - a[5]*c[5,5]*l[0]*l[2]/36 + a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/72,
           -a[0]*c[0,0]*l[1]*l[2]/18 + a[0]*c[4,0]*l[0]*l[1]/18 + a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/18 - a[0]*c[5,0]*l[0]*l[2]/18 + a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/18 - a[1]*c[0,1]*l[1]*l[2]/18 + a[1]*c[4,1]*l[0]*l[1]/18 + a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/18 - a[1]*c[5,1]*l[0]*l[2]/18 + a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/18 - a[2]*c[0,2]*l[1]*l[2]/18 + a[2]*c[4,2]*l[0]*l[1]/18 + a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/18 - a[2]*c[5,2]*l[0]*l[2]/18 + a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/18 - a[3]*c[0,3]*l[1]*l[2]/18 + a[3]*c[4,3]*l[0]*l[1]/18 + a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/18 - a[3]*c[5,3]*l[0]*l[2]/18 + a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/18 - a[4]*c[0,4]*l[1]*l[2]/18 + a[4]*c[4,4]*l[0]*l[1]/18 + a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/18 - a[4]*c[5,4]*l[0]*l[2]/18 + a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/18 - a[5]*c[0,5]*l[1]*l[2]/18 + a[5]*c[4,5]*l[0]*l[1]/18 + a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/18 - a[5]*c[5,5]*l[0]*l[2]/18 + a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/18,
           -a[0]*c[0,0]*l[1]*l[2]/18 + a[0]*c[4,0]*l[0]*l[1]/36 + a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/18 - a[0]*c[5,0]*l[0]*l[2]/36 + a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/18 - a[1]*c[0,1]*l[1]*l[2]/18 + a[1]*c[4,1]*l[0]*l[1]/36 + a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/18 - a[1]*c[5,1]*l[0]*l[2]/36 + a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/18 - a[2]*c[0,2]*l[1]*l[2]/18 + a[2]*c[4,2]*l[0]*l[1]/36 + a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/18 - a[2]*c[5,2]*l[0]*l[2]/36 + a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/18 - a[3]*c[0,3]*l[1]*l[2]/18 + a[3]*c[4,3]*l[0]*l[1]/36 + a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/18 - a[3]*c[5,3]*l[0]*l[2]/36 + a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/18 - a[4]*c[0,4]*l[1]*l[2]/18 + a[4]*c[4,4]*l[0]*l[1]/36 + a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/18 - a[4]*c[5,4]*l[0]*l[2]/36 + a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/18 - a[5]*c[0,5]*l[1]*l[2]/18 + a[5]*c[4,5]*l[0]*l[1]/36 + a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/18 - a[5]*c[5,5]*l[0]*l[2]/36 + a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/18,
           -a[0]*c[0,0]*l[1]*l[2]/36 + a[0]*c[4,0]*l[0]*l[1]/72 + a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[5,0]*l[0]*l[2]/36 + a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[0,1]*l[1]*l[2]/36 + a[1]*c[4,1]*l[0]*l[1]/72 + a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[5,1]*l[0]*l[2]/36 + a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[0,2]*l[1]*l[2]/36 + a[2]*c[4,2]*l[0]*l[1]/72 + a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[5,2]*l[0]*l[2]/36 + a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[0,3]*l[1]*l[2]/36 + a[3]*c[4,3]*l[0]*l[1]/72 + a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[5,3]*l[0]*l[2]/36 + a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[0,4]*l[1]*l[2]/36 + a[4]*c[4,4]*l[0]*l[1]/72 + a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[5,4]*l[0]*l[2]/36 + a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[0,5]*l[1]*l[2]/36 + a[5]*c[4,5]*l[0]*l[1]/72 + a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[5,5]*l[0]*l[2]/36 + a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/36,
           -a[0]*c[0,0]*l[1]*l[2]/36 + a[0]*c[4,0]*l[0]*l[1]/36 + a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[5,0]*l[0]*l[2]/18 + a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[0,1]*l[1]*l[2]/36 + a[1]*c[4,1]*l[0]*l[1]/36 + a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[5,1]*l[0]*l[2]/18 + a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[0,2]*l[1]*l[2]/36 + a[2]*c[4,2]*l[0]*l[1]/36 + a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[5,2]*l[0]*l[2]/18 + a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[0,3]*l[1]*l[2]/36 + a[3]*c[4,3]*l[0]*l[1]/36 + a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[5,3]*l[0]*l[2]/18 + a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[0,4]*l[1]*l[2]/36 + a[4]*c[4,4]*l[0]*l[1]/36 + a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[5,4]*l[0]*l[2]/18 + a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[0,5]*l[1]*l[2]/36 + a[5]*c[4,5]*l[0]*l[1]/36 + a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[5,5]*l[0]*l[2]/18 + a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/36],
          [-a[0]*c[1,0]*l[0]*l[2]/36 + a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/36 + a[0]*c[3,0]*l[0]*l[1]/18 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[5,0]*l[1]*l[2]/36 - a[1]*c[1,1]*l[0]*l[2]/36 + a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[3,1]*l[0]*l[1]/18 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[5,1]*l[1]*l[2]/36 - a[2]*c[1,2]*l[0]*l[2]/36 + a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[3,2]*l[0]*l[1]/18 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[5,2]*l[1]*l[2]/36 - a[3]*c[1,3]*l[0]*l[2]/36 + a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[3,3]*l[0]*l[1]/18 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[5,3]*l[1]*l[2]/36 - a[4]*c[1,4]*l[0]*l[2]/36 + a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[3,4]*l[0]*l[1]/18 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[5,4]*l[1]*l[2]/36 - a[5]*c[1,5]*l[0]*l[2]/36 + a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[3,5]*l[0]*l[1]/18 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[5,5]*l[1]*l[2]/36,
           -a[0]*c[1,0]*l[0]*l[2]/72 + a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/36 + a[0]*c[3,0]*l[0]*l[1]/36 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[5,0]*l[1]*l[2]/36 - a[1]*c[1,1]*l[0]*l[2]/72 + a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[3,1]*l[0]*l[1]/36 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[5,1]*l[1]*l[2]/36 - a[2]*c[1,2]*l[0]*l[2]/72 + a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[3,2]*l[0]*l[1]/36 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[5,2]*l[1]*l[2]/36 - a[3]*c[1,3]*l[0]*l[2]/72 + a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[3,3]*l[0]*l[1]/36 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[5,3]*l[1]*l[2]/36 - a[4]*c[1,4]*l[0]*l[2]/72 + a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[3,4]*l[0]*l[1]/36 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[5,4]*l[1]*l[2]/36 - a[5]*c[1,5]*l[0]*l[2]/72 + a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[3,5]*l[0]*l[1]/36 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[5,5]*l[1]*l[2]/36,
           -a[0]*c[1,0]*l[0]*l[2]/72 + a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/72 + a[0]*c[3,0]*l[0]*l[1]/72 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/72 - a[0]*c[5,0]*l[1]*l[2]/72 - a[1]*c[1,1]*l[0]*l[2]/72 + a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/72 + a[1]*c[3,1]*l[0]*l[1]/72 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/72 - a[1]*c[5,1]*l[1]*l[2]/72 - a[2]*c[1,2]*l[0]*l[2]/72 + a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/72 + a[2]*c[3,2]*l[0]*l[1]/72 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/72 - a[2]*c[5,2]*l[1]*l[2]/72 - a[3]*c[1,3]*l[0]*l[2]/72 + a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/72 + a[3]*c[3,3]*l[0]*l[1]/72 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/72 - a[3]*c[5,3]*l[1]*l[2]/72 - a[4]*c[1,4]*l[0]*l[2]/72 + a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/72 + a[4]*c[3,4]*l[0]*l[1]/72 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/72 - a[4]*c[5,4]*l[1]*l[2]/72 - a[5]*c[1,5]*l[0]*l[2]/72 + a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/72 + a[5]*c[3,5]*l[0]*l[1]/72 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/72 - a[5]*c[5,5]*l[1]*l[2]/72,
           -a[0]*c[1,0]*l[0]*l[2]/36 + a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/72 + a[0]*c[3,0]*l[0]*l[1]/36 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/72 - a[0]*c[5,0]*l[1]*l[2]/72 - a[1]*c[1,1]*l[0]*l[2]/36 + a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/72 + a[1]*c[3,1]*l[0]*l[1]/36 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/72 - a[1]*c[5,1]*l[1]*l[2]/72 - a[2]*c[1,2]*l[0]*l[2]/36 + a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/72 + a[2]*c[3,2]*l[0]*l[1]/36 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/72 - a[2]*c[5,2]*l[1]*l[2]/72 - a[3]*c[1,3]*l[0]*l[2]/36 + a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/72 + a[3]*c[3,3]*l[0]*l[1]/36 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/72 - a[3]*c[5,3]*l[1]*l[2]/72 - a[4]*c[1,4]*l[0]*l[2]/36 + a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/72 + a[4]*c[3,4]*l[0]*l[1]/36 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/72 - a[4]*c[5,4]*l[1]*l[2]/72 - a[5]*c[1,5]*l[0]*l[2]/36 + a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/72 + a[5]*c[3,5]*l[0]*l[1]/36 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/72 - a[5]*c[5,5]*l[1]*l[2]/72,
           -a[0]*c[1,0]*l[0]*l[2]/18 + a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/18 + a[0]*c[3,0]*l[0]*l[1]/18 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/18 - a[0]*c[5,0]*l[1]*l[2]/18 - a[1]*c[1,1]*l[0]*l[2]/18 + a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/18 + a[1]*c[3,1]*l[0]*l[1]/18 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/18 - a[1]*c[5,1]*l[1]*l[2]/18 - a[2]*c[1,2]*l[0]*l[2]/18 + a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/18 + a[2]*c[3,2]*l[0]*l[1]/18 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/18 - a[2]*c[5,2]*l[1]*l[2]/18 - a[3]*c[1,3]*l[0]*l[2]/18 + a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/18 + a[3]*c[3,3]*l[0]*l[1]/18 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/18 - a[3]*c[5,3]*l[1]*l[2]/18 - a[4]*c[1,4]*l[0]*l[2]/18 + a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/18 + a[4]*c[3,4]*l[0]*l[1]/18 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/18 - a[4]*c[5,4]*l[1]*l[2]/18 - a[5]*c[1,5]*l[0]*l[2]/18 + a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/18 + a[5]*c[3,5]*l[0]*l[1]/18 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/18 - a[5]*c[5,5]*l[1]*l[2]/18,
           -a[0]*c[1,0]*l[0]*l[2]/36 + a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/18 + a[0]*c[3,0]*l[0]*l[1]/36 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/18 - a[0]*c[5,0]*l[1]*l[2]/18 - a[1]*c[1,1]*l[0]*l[2]/36 + a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/18 + a[1]*c[3,1]*l[0]*l[1]/36 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/18 - a[1]*c[5,1]*l[1]*l[2]/18 - a[2]*c[1,2]*l[0]*l[2]/36 + a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/18 + a[2]*c[3,2]*l[0]*l[1]/36 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/18 - a[2]*c[5,2]*l[1]*l[2]/18 - a[3]*c[1,3]*l[0]*l[2]/36 + a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/18 + a[3]*c[3,3]*l[0]*l[1]/36 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/18 - a[3]*c[5,3]*l[1]*l[2]/18 - a[4]*c[1,4]*l[0]*l[2]/36 + a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/18 + a[4]*c[3,4]*l[0]*l[1]/36 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/18 - a[4]*c[5,4]*l[1]*l[2]/18 - a[5]*c[1,5]*l[0]*l[2]/36 + a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/18 + a[5]*c[3,5]*l[0]*l[1]/36 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/18 - a[5]*c[5,5]*l[1]*l[2]/18,
           -a[0]*c[1,0]*l[0]*l[2]/36 + a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/36 + a[0]*c[3,0]*l[0]*l[1]/72 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[5,0]*l[1]*l[2]/36 - a[1]*c[1,1]*l[0]*l[2]/36 + a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[3,1]*l[0]*l[1]/72 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[5,1]*l[1]*l[2]/36 - a[2]*c[1,2]*l[0]*l[2]/36 + a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[3,2]*l[0]*l[1]/72 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[5,2]*l[1]*l[2]/36 - a[3]*c[1,3]*l[0]*l[2]/36 + a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[3,3]*l[0]*l[1]/72 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[5,3]*l[1]*l[2]/36 - a[4]*c[1,4]*l[0]*l[2]/36 + a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[3,4]*l[0]*l[1]/72 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[5,4]*l[1]*l[2]/36 - a[5]*c[1,5]*l[0]*l[2]/36 + a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[3,5]*l[0]*l[1]/72 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[5,5]*l[1]*l[2]/36,
           -a[0]*c[1,0]*l[0]*l[2]/18 + a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/36 + a[0]*c[3,0]*l[0]*l[1]/36 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[5,0]*l[1]*l[2]/36 - a[1]*c[1,1]*l[0]*l[2]/18 + a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[3,1]*l[0]*l[1]/36 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[5,1]*l[1]*l[2]/36 - a[2]*c[1,2]*l[0]*l[2]/18 + a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[3,2]*l[0]*l[1]/36 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[5,2]*l[1]*l[2]/36 - a[3]*c[1,3]*l[0]*l[2]/18 + a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[3,3]*l[0]*l[1]/36 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[5,3]*l[1]*l[2]/36 - a[4]*c[1,4]*l[0]*l[2]/18 + a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[3,4]*l[0]*l[1]/36 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[5,4]*l[1]*l[2]/36 - a[5]*c[1,5]*l[0]*l[2]/18 + a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[3,5]*l[0]*l[1]/36 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[5,5]*l[1]*l[2]/36],
          [a[0]*c[2,0]*l[0]*l[1]/18 + a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[3,0]*l[0]*l[2]/36 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/36 - a[0]*c[4,0]*l[1]*l[2]/36 + a[1]*c[2,1]*l[0]*l[1]/18 + a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[3,1]*l[0]*l[2]/36 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[4,1]*l[1]*l[2]/36 + a[2]*c[2,2]*l[0]*l[1]/18 + a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[3,2]*l[0]*l[2]/36 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[4,2]*l[1]*l[2]/36 + a[3]*c[2,3]*l[0]*l[1]/18 + a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[3,3]*l[0]*l[2]/36 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[4,3]*l[1]*l[2]/36 + a[4]*c[2,4]*l[0]*l[1]/18 + a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[3,4]*l[0]*l[2]/36 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[4,4]*l[1]*l[2]/36 + a[5]*c[2,5]*l[0]*l[1]/18 + a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[3,5]*l[0]*l[2]/36 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[4,5]*l[1]*l[2]/36,
           a[0]*c[2,0]*l[0]*l[1]/36 + a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[3,0]*l[0]*l[2]/72 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/36 - a[0]*c[4,0]*l[1]*l[2]/36 + a[1]*c[2,1]*l[0]*l[1]/36 + a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[3,1]*l[0]*l[2]/72 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[4,1]*l[1]*l[2]/36 + a[2]*c[2,2]*l[0]*l[1]/36 + a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[3,2]*l[0]*l[2]/72 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[4,2]*l[1]*l[2]/36 + a[3]*c[2,3]*l[0]*l[1]/36 + a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[3,3]*l[0]*l[2]/72 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[4,3]*l[1]*l[2]/36 + a[4]*c[2,4]*l[0]*l[1]/36 + a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[3,4]*l[0]*l[2]/72 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[4,4]*l[1]*l[2]/36 + a[5]*c[2,5]*l[0]*l[1]/36 + a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[3,5]*l[0]*l[2]/72 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[4,5]*l[1]*l[2]/36,
           a[0]*c[2,0]*l[0]*l[1]/72 + a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/72 - a[0]*c[3,0]*l[0]*l[2]/72 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/72 - a[0]*c[4,0]*l[1]*l[2]/72 + a[1]*c[2,1]*l[0]*l[1]/72 + a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/72 - a[1]*c[3,1]*l[0]*l[2]/72 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/72 - a[1]*c[4,1]*l[1]*l[2]/72 + a[2]*c[2,2]*l[0]*l[1]/72 + a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/72 - a[2]*c[3,2]*l[0]*l[2]/72 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/72 - a[2]*c[4,2]*l[1]*l[2]/72 + a[3]*c[2,3]*l[0]*l[1]/72 + a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/72 - a[3]*c[3,3]*l[0]*l[2]/72 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/72 - a[3]*c[4,3]*l[1]*l[2]/72 + a[4]*c[2,4]*l[0]*l[1]/72 + a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/72 - a[4]*c[3,4]*l[0]*l[2]/72 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/72 - a[4]*c[4,4]*l[1]*l[2]/72 + a[5]*c[2,5]*l[0]*l[1]/72 + a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/72 - a[5]*c[3,5]*l[0]*l[2]/72 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/72 - a[5]*c[4,5]*l[1]*l[2]/72,
           a[0]*c[2,0]*l[0]*l[1]/36 + a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/72 - a[0]*c[3,0]*l[0]*l[2]/36 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/72 - a[0]*c[4,0]*l[1]*l[2]/72 + a[1]*c[2,1]*l[0]*l[1]/36 + a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/72 - a[1]*c[3,1]*l[0]*l[2]/36 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/72 - a[1]*c[4,1]*l[1]*l[2]/72 + a[2]*c[2,2]*l[0]*l[1]/36 + a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/72 - a[2]*c[3,2]*l[0]*l[2]/36 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/72 - a[2]*c[4,2]*l[1]*l[2]/72 + a[3]*c[2,3]*l[0]*l[1]/36 + a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/72 - a[3]*c[3,3]*l[0]*l[2]/36 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/72 - a[3]*c[4,3]*l[1]*l[2]/72 + a[4]*c[2,4]*l[0]*l[1]/36 + a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/72 - a[4]*c[3,4]*l[0]*l[2]/36 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/72 - a[4]*c[4,4]*l[1]*l[2]/72 + a[5]*c[2,5]*l[0]*l[1]/36 + a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/72 - a[5]*c[3,5]*l[0]*l[2]/36 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/72 - a[5]*c[4,5]*l[1]*l[2]/72,
           a[0]*c[2,0]*l[0]*l[1]/18 + a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/18 - a[0]*c[3,0]*l[0]*l[2]/18 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/18 - a[0]*c[4,0]*l[1]*l[2]/18 + a[1]*c[2,1]*l[0]*l[1]/18 + a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/18 - a[1]*c[3,1]*l[0]*l[2]/18 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/18 - a[1]*c[4,1]*l[1]*l[2]/18 + a[2]*c[2,2]*l[0]*l[1]/18 + a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/18 - a[2]*c[3,2]*l[0]*l[2]/18 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/18 - a[2]*c[4,2]*l[1]*l[2]/18 + a[3]*c[2,3]*l[0]*l[1]/18 + a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/18 - a[3]*c[3,3]*l[0]*l[2]/18 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/18 - a[3]*c[4,3]*l[1]*l[2]/18 + a[4]*c[2,4]*l[0]*l[1]/18 + a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/18 - a[4]*c[3,4]*l[0]*l[2]/18 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/18 - a[4]*c[4,4]*l[1]*l[2]/18 + a[5]*c[2,5]*l[0]*l[1]/18 + a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/18 - a[5]*c[3,5]*l[0]*l[2]/18 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/18 - a[5]*c[4,5]*l[1]*l[2]/18,
           a[0]*c[2,0]*l[0]*l[1]/36 + a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/18 - a[0]*c[3,0]*l[0]*l[2]/36 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/18 - a[0]*c[4,0]*l[1]*l[2]/18 + a[1]*c[2,1]*l[0]*l[1]/36 + a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/18 - a[1]*c[3,1]*l[0]*l[2]/36 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/18 - a[1]*c[4,1]*l[1]*l[2]/18 + a[2]*c[2,2]*l[0]*l[1]/36 + a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/18 - a[2]*c[3,2]*l[0]*l[2]/36 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/18 - a[2]*c[4,2]*l[1]*l[2]/18 + a[3]*c[2,3]*l[0]*l[1]/36 + a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/18 - a[3]*c[3,3]*l[0]*l[2]/36 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/18 - a[3]*c[4,3]*l[1]*l[2]/18 + a[4]*c[2,4]*l[0]*l[1]/36 + a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/18 - a[4]*c[3,4]*l[0]*l[2]/36 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/18 - a[4]*c[4,4]*l[1]*l[2]/18 + a[5]*c[2,5]*l[0]*l[1]/36 + a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/18 - a[5]*c[3,5]*l[0]*l[2]/36 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/18 - a[5]*c[4,5]*l[1]*l[2]/18,
           a[0]*c[2,0]*l[0]*l[1]/72 + a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[3,0]*l[0]*l[2]/36 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/36 - a[0]*c[4,0]*l[1]*l[2]/36 + a[1]*c[2,1]*l[0]*l[1]/72 + a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[3,1]*l[0]*l[2]/36 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[4,1]*l[1]*l[2]/36 + a[2]*c[2,2]*l[0]*l[1]/72 + a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[3,2]*l[0]*l[2]/36 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[4,2]*l[1]*l[2]/36 + a[3]*c[2,3]*l[0]*l[1]/72 + a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[3,3]*l[0]*l[2]/36 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[4,3]*l[1]*l[2]/36 + a[4]*c[2,4]*l[0]*l[1]/72 + a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[3,4]*l[0]*l[2]/36 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[4,4]*l[1]*l[2]/36 + a[5]*c[2,5]*l[0]*l[1]/72 + a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[3,5]*l[0]*l[2]/36 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[4,5]*l[1]*l[2]/36,
           a[0]*c[2,0]*l[0]*l[1]/36 + a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[3,0]*l[0]*l[2]/18 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/36 - a[0]*c[4,0]*l[1]*l[2]/36 + a[1]*c[2,1]*l[0]*l[1]/36 + a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[3,1]*l[0]*l[2]/18 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[4,1]*l[1]*l[2]/36 + a[2]*c[2,2]*l[0]*l[1]/36 + a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[3,2]*l[0]*l[2]/18 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[4,2]*l[1]*l[2]/36 + a[3]*c[2,3]*l[0]*l[1]/36 + a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[3,3]*l[0]*l[2]/18 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[4,3]*l[1]*l[2]/36 + a[4]*c[2,4]*l[0]*l[1]/36 + a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[3,4]*l[0]*l[2]/18 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[4,4]*l[1]*l[2]/36 + a[5]*c[2,5]*l[0]*l[1]/36 + a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[3,5]*l[0]*l[2]/18 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[4,5]*l[1]*l[2]/36],
          [a[0]*c[0,0]*l[1]*l[2]/36 + a[0]*c[4,0]*l[0]*l[1]/36 - a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[5,0]*l[0]*l[2]/72 - a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[0,1]*l[1]*l[2]/36 + a[1]*c[4,1]*l[0]*l[1]/36 - a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[5,1]*l[0]*l[2]/72 - a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[0,2]*l[1]*l[2]/36 + a[2]*c[4,2]*l[0]*l[1]/36 - a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[5,2]*l[0]*l[2]/72 - a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[0,3]*l[1]*l[2]/36 + a[3]*c[4,3]*l[0]*l[1]/36 - a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[5,3]*l[0]*l[2]/72 - a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[0,4]*l[1]*l[2]/36 + a[4]*c[4,4]*l[0]*l[1]/36 - a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[5,4]*l[0]*l[2]/72 - a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[0,5]*l[1]*l[2]/36 + a[5]*c[4,5]*l[0]*l[1]/36 - a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[5,5]*l[0]*l[2]/72 - a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/36,
           a[0]*c[0,0]*l[1]*l[2]/36 + a[0]*c[4,0]*l[0]*l[1]/18 - a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[5,0]*l[0]*l[2]/36 - a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[0,1]*l[1]*l[2]/36 + a[1]*c[4,1]*l[0]*l[1]/18 - a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[5,1]*l[0]*l[2]/36 - a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[0,2]*l[1]*l[2]/36 + a[2]*c[4,2]*l[0]*l[1]/18 - a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[5,2]*l[0]*l[2]/36 - a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[0,3]*l[1]*l[2]/36 + a[3]*c[4,3]*l[0]*l[1]/18 - a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[5,3]*l[0]*l[2]/36 - a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[0,4]*l[1]*l[2]/36 + a[4]*c[4,4]*l[0]*l[1]/18 - a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[5,4]*l[0]*l[2]/36 - a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[0,5]*l[1]*l[2]/36 + a[5]*c[4,5]*l[0]*l[1]/18 - a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[5,5]*l[0]*l[2]/36 - a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/36,
           a[0]*c[0,0]*l[1]*l[2]/72 + a[0]*c[4,0]*l[0]*l[1]/36 - a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/72 - a[0]*c[5,0]*l[0]*l[2]/36 - a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/72 + a[1]*c[0,1]*l[1]*l[2]/72 + a[1]*c[4,1]*l[0]*l[1]/36 - a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/72 - a[1]*c[5,1]*l[0]*l[2]/36 - a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/72 + a[2]*c[0,2]*l[1]*l[2]/72 + a[2]*c[4,2]*l[0]*l[1]/36 - a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/72 - a[2]*c[5,2]*l[0]*l[2]/36 - a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/72 + a[3]*c[0,3]*l[1]*l[2]/72 + a[3]*c[4,3]*l[0]*l[1]/36 - a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/72 - a[3]*c[5,3]*l[0]*l[2]/36 - a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/72 + a[4]*c[0,4]*l[1]*l[2]/72 + a[4]*c[4,4]*l[0]*l[1]/36 - a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/72 - a[4]*c[5,4]*l[0]*l[2]/36 - a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/72 + a[5]*c[0,5]*l[1]*l[2]/72 + a[5]*c[4,5]*l[0]*l[1]/36 - a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/72 - a[5]*c[5,5]*l[0]*l[2]/36 - a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/72,
           a[0]*c[0,0]*l[1]*l[2]/72 + a[0]*c[4,0]*l[0]*l[1]/72 - a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/72 - a[0]*c[5,0]*l[0]*l[2]/72 - a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/72 + a[1]*c[0,1]*l[1]*l[2]/72 + a[1]*c[4,1]*l[0]*l[1]/72 - a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/72 - a[1]*c[5,1]*l[0]*l[2]/72 - a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/72 + a[2]*c[0,2]*l[1]*l[2]/72 + a[2]*c[4,2]*l[0]*l[1]/72 - a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/72 - a[2]*c[5,2]*l[0]*l[2]/72 - a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/72 + a[3]*c[0,3]*l[1]*l[2]/72 + a[3]*c[4,3]*l[0]*l[1]/72 - a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/72 - a[3]*c[5,3]*l[0]*l[2]/72 - a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/72 + a[4]*c[0,4]*l[1]*l[2]/72 + a[4]*c[4,4]*l[0]*l[1]/72 - a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/72 - a[4]*c[5,4]*l[0]*l[2]/72 - a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/72 + a[5]*c[0,5]*l[1]*l[2]/72 + a[5]*c[4,5]*l[0]*l[1]/72 - a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/72 - a[5]*c[5,5]*l[0]*l[2]/72 - a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/72,
           a[0]*c[0,0]*l[1]*l[2]/18 + a[0]*c[4,0]*l[0]*l[1]/36 - a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/18 - a[0]*c[5,0]*l[0]*l[2]/36 - a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/18 + a[1]*c[0,1]*l[1]*l[2]/18 + a[1]*c[4,1]*l[0]*l[1]/36 - a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/18 - a[1]*c[5,1]*l[0]*l[2]/36 - a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/18 + a[2]*c[0,2]*l[1]*l[2]/18 + a[2]*c[4,2]*l[0]*l[1]/36 - a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/18 - a[2]*c[5,2]*l[0]*l[2]/36 - a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/18 + a[3]*c[0,3]*l[1]*l[2]/18 + a[3]*c[4,3]*l[0]*l[1]/36 - a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/18 - a[3]*c[5,3]*l[0]*l[2]/36 - a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/18 + a[4]*c[0,4]*l[1]*l[2]/18 + a[4]*c[4,4]*l[0]*l[1]/36 - a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/18 - a[4]*c[5,4]*l[0]*l[2]/36 - a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/18 + a[5]*c[0,5]*l[1]*l[2]/18 + a[5]*c[4,5]*l[0]*l[1]/36 - a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/18 - a[5]*c[5,5]*l[0]*l[2]/36 - a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/18,
           a[0]*c[0,0]*l[1]*l[2]/18 + a[0]*c[4,0]*l[0]*l[1]/18 - a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/18 - a[0]*c[5,0]*l[0]*l[2]/18 - a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/18 + a[1]*c[0,1]*l[1]*l[2]/18 + a[1]*c[4,1]*l[0]*l[1]/18 - a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/18 - a[1]*c[5,1]*l[0]*l[2]/18 - a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/18 + a[2]*c[0,2]*l[1]*l[2]/18 + a[2]*c[4,2]*l[0]*l[1]/18 - a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/18 - a[2]*c[5,2]*l[0]*l[2]/18 - a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/18 + a[3]*c[0,3]*l[1]*l[2]/18 + a[3]*c[4,3]*l[0]*l[1]/18 - a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/18 - a[3]*c[5,3]*l[0]*l[2]/18 - a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/18 + a[4]*c[0,4]*l[1]*l[2]/18 + a[4]*c[4,4]*l[0]*l[1]/18 - a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/18 - a[4]*c[5,4]*l[0]*l[2]/18 - a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/18 + a[5]*c[0,5]*l[1]*l[2]/18 + a[5]*c[4,5]*l[0]*l[1]/18 - a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/18 - a[5]*c[5,5]*l[0]*l[2]/18 - a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/18,
           a[0]*c[0,0]*l[1]*l[2]/36 + a[0]*c[4,0]*l[0]*l[1]/36 - a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[5,0]*l[0]*l[2]/18 - a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[0,1]*l[1]*l[2]/36 + a[1]*c[4,1]*l[0]*l[1]/36 - a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[5,1]*l[0]*l[2]/18 - a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[0,2]*l[1]*l[2]/36 + a[2]*c[4,2]*l[0]*l[1]/36 - a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[5,2]*l[0]*l[2]/18 - a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[0,3]*l[1]*l[2]/36 + a[3]*c[4,3]*l[0]*l[1]/36 - a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[5,3]*l[0]*l[2]/18 - a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[0,4]*l[1]*l[2]/36 + a[4]*c[4,4]*l[0]*l[1]/36 - a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[5,4]*l[0]*l[2]/18 - a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[0,5]*l[1]*l[2]/36 + a[5]*c[4,5]*l[0]*l[1]/36 - a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[5,5]*l[0]*l[2]/18 - a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/36,
           a[0]*c[0,0]*l[1]*l[2]/36 + a[0]*c[4,0]*l[0]*l[1]/72 - a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[5,0]*l[0]*l[2]/36 - a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[0,1]*l[1]*l[2]/36 + a[1]*c[4,1]*l[0]*l[1]/72 - a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[5,1]*l[0]*l[2]/36 - a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[0,2]*l[1]*l[2]/36 + a[2]*c[4,2]*l[0]*l[1]/72 - a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[5,2]*l[0]*l[2]/36 - a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[0,3]*l[1]*l[2]/36 + a[3]*c[4,3]*l[0]*l[1]/72 - a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[5,3]*l[0]*l[2]/36 - a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[0,4]*l[1]*l[2]/36 + a[4]*c[4,4]*l[0]*l[1]/72 - a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[5,4]*l[0]*l[2]/36 - a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[0,5]*l[1]*l[2]/36 + a[5]*c[4,5]*l[0]*l[1]/72 - a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[5,5]*l[0]*l[2]/36 - a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/36],
          [-a[0]*c[1,0]*l[0]*l[2]/72 - a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/36 + a[0]*c[3,0]*l[0]*l[1]/36 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[5,0]*l[1]*l[2]/36 - a[1]*c[1,1]*l[0]*l[2]/72 - a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[3,1]*l[0]*l[1]/36 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[5,1]*l[1]*l[2]/36 - a[2]*c[1,2]*l[0]*l[2]/72 - a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[3,2]*l[0]*l[1]/36 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[5,2]*l[1]*l[2]/36 - a[3]*c[1,3]*l[0]*l[2]/72 - a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[3,3]*l[0]*l[1]/36 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[5,3]*l[1]*l[2]/36 - a[4]*c[1,4]*l[0]*l[2]/72 - a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[3,4]*l[0]*l[1]/36 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[5,4]*l[1]*l[2]/36 - a[5]*c[1,5]*l[0]*l[2]/72 - a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[3,5]*l[0]*l[1]/36 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[5,5]*l[1]*l[2]/36,
           -a[0]*c[1,0]*l[0]*l[2]/36 - a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/36 + a[0]*c[3,0]*l[0]*l[1]/18 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[5,0]*l[1]*l[2]/36 - a[1]*c[1,1]*l[0]*l[2]/36 - a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[3,1]*l[0]*l[1]/18 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[5,1]*l[1]*l[2]/36 - a[2]*c[1,2]*l[0]*l[2]/36 - a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[3,2]*l[0]*l[1]/18 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[5,2]*l[1]*l[2]/36 - a[3]*c[1,3]*l[0]*l[2]/36 - a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[3,3]*l[0]*l[1]/18 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[5,3]*l[1]*l[2]/36 - a[4]*c[1,4]*l[0]*l[2]/36 - a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[3,4]*l[0]*l[1]/18 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[5,4]*l[1]*l[2]/36 - a[5]*c[1,5]*l[0]*l[2]/36 - a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[3,5]*l[0]*l[1]/18 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[5,5]*l[1]*l[2]/36,
           -a[0]*c[1,0]*l[0]*l[2]/36 - a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/72 + a[0]*c[3,0]*l[0]*l[1]/36 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/72 + a[0]*c[5,0]*l[1]*l[2]/72 - a[1]*c[1,1]*l[0]*l[2]/36 - a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/72 + a[1]*c[3,1]*l[0]*l[1]/36 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/72 + a[1]*c[5,1]*l[1]*l[2]/72 - a[2]*c[1,2]*l[0]*l[2]/36 - a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/72 + a[2]*c[3,2]*l[0]*l[1]/36 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/72 + a[2]*c[5,2]*l[1]*l[2]/72 - a[3]*c[1,3]*l[0]*l[2]/36 - a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/72 + a[3]*c[3,3]*l[0]*l[1]/36 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/72 + a[3]*c[5,3]*l[1]*l[2]/72 - a[4]*c[1,4]*l[0]*l[2]/36 - a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/72 + a[4]*c[3,4]*l[0]*l[1]/36 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/72 + a[4]*c[5,4]*l[1]*l[2]/72 - a[5]*c[1,5]*l[0]*l[2]/36 - a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/72 + a[5]*c[3,5]*l[0]*l[1]/36 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/72 + a[5]*c[5,5]*l[1]*l[2]/72,
           -a[0]*c[1,0]*l[0]*l[2]/72 - a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/72 + a[0]*c[3,0]*l[0]*l[1]/72 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/72 + a[0]*c[5,0]*l[1]*l[2]/72 - a[1]*c[1,1]*l[0]*l[2]/72 - a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/72 + a[1]*c[3,1]*l[0]*l[1]/72 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/72 + a[1]*c[5,1]*l[1]*l[2]/72 - a[2]*c[1,2]*l[0]*l[2]/72 - a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/72 + a[2]*c[3,2]*l[0]*l[1]/72 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/72 + a[2]*c[5,2]*l[1]*l[2]/72 - a[3]*c[1,3]*l[0]*l[2]/72 - a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/72 + a[3]*c[3,3]*l[0]*l[1]/72 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/72 + a[3]*c[5,3]*l[1]*l[2]/72 - a[4]*c[1,4]*l[0]*l[2]/72 - a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/72 + a[4]*c[3,4]*l[0]*l[1]/72 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/72 + a[4]*c[5,4]*l[1]*l[2]/72 - a[5]*c[1,5]*l[0]*l[2]/72 - a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/72 + a[5]*c[3,5]*l[0]*l[1]/72 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/72 + a[5]*c[5,5]*l[1]*l[2]/72,
           -a[0]*c[1,0]*l[0]*l[2]/36 - a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/18 + a[0]*c[3,0]*l[0]*l[1]/36 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/18 + a[0]*c[5,0]*l[1]*l[2]/18 - a[1]*c[1,1]*l[0]*l[2]/36 - a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/18 + a[1]*c[3,1]*l[0]*l[1]/36 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/18 + a[1]*c[5,1]*l[1]*l[2]/18 - a[2]*c[1,2]*l[0]*l[2]/36 - a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/18 + a[2]*c[3,2]*l[0]*l[1]/36 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/18 + a[2]*c[5,2]*l[1]*l[2]/18 - a[3]*c[1,3]*l[0]*l[2]/36 - a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/18 + a[3]*c[3,3]*l[0]*l[1]/36 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/18 + a[3]*c[5,3]*l[1]*l[2]/18 - a[4]*c[1,4]*l[0]*l[2]/36 - a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/18 + a[4]*c[3,4]*l[0]*l[1]/36 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/18 + a[4]*c[5,4]*l[1]*l[2]/18 - a[5]*c[1,5]*l[0]*l[2]/36 - a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/18 + a[5]*c[3,5]*l[0]*l[1]/36 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/18 + a[5]*c[5,5]*l[1]*l[2]/18,
           -a[0]*c[1,0]*l[0]*l[2]/18 - a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/18 + a[0]*c[3,0]*l[0]*l[1]/18 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/18 + a[0]*c[5,0]*l[1]*l[2]/18 - a[1]*c[1,1]*l[0]*l[2]/18 - a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/18 + a[1]*c[3,1]*l[0]*l[1]/18 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/18 + a[1]*c[5,1]*l[1]*l[2]/18 - a[2]*c[1,2]*l[0]*l[2]/18 - a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/18 + a[2]*c[3,2]*l[0]*l[1]/18 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/18 + a[2]*c[5,2]*l[1]*l[2]/18 - a[3]*c[1,3]*l[0]*l[2]/18 - a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/18 + a[3]*c[3,3]*l[0]*l[1]/18 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/18 + a[3]*c[5,3]*l[1]*l[2]/18 - a[4]*c[1,4]*l[0]*l[2]/18 - a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/18 + a[4]*c[3,4]*l[0]*l[1]/18 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/18 + a[4]*c[5,4]*l[1]*l[2]/18 - a[5]*c[1,5]*l[0]*l[2]/18 - a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/18 + a[5]*c[3,5]*l[0]*l[1]/18 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/18 + a[5]*c[5,5]*l[1]*l[2]/18,
           -a[0]*c[1,0]*l[0]*l[2]/18 - a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/36 + a[0]*c[3,0]*l[0]*l[1]/36 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[5,0]*l[1]*l[2]/36 - a[1]*c[1,1]*l[0]*l[2]/18 - a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[3,1]*l[0]*l[1]/36 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[5,1]*l[1]*l[2]/36 - a[2]*c[1,2]*l[0]*l[2]/18 - a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[3,2]*l[0]*l[1]/36 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[5,2]*l[1]*l[2]/36 - a[3]*c[1,3]*l[0]*l[2]/18 - a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[3,3]*l[0]*l[1]/36 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[5,3]*l[1]*l[2]/36 - a[4]*c[1,4]*l[0]*l[2]/18 - a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[3,4]*l[0]*l[1]/36 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[5,4]*l[1]*l[2]/36 - a[5]*c[1,5]*l[0]*l[2]/18 - a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[3,5]*l[0]*l[1]/36 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[5,5]*l[1]*l[2]/36,
           -a[0]*c[1,0]*l[0]*l[2]/36 - a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/36 + a[0]*c[3,0]*l[0]*l[1]/72 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[5,0]*l[1]*l[2]/36 - a[1]*c[1,1]*l[0]*l[2]/36 - a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[3,1]*l[0]*l[1]/72 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[5,1]*l[1]*l[2]/36 - a[2]*c[1,2]*l[0]*l[2]/36 - a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[3,2]*l[0]*l[1]/72 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[5,2]*l[1]*l[2]/36 - a[3]*c[1,3]*l[0]*l[2]/36 - a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[3,3]*l[0]*l[1]/72 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[5,3]*l[1]*l[2]/36 - a[4]*c[1,4]*l[0]*l[2]/36 - a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[3,4]*l[0]*l[1]/72 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[5,4]*l[1]*l[2]/36 - a[5]*c[1,5]*l[0]*l[2]/36 - a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[3,5]*l[0]*l[1]/72 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[5,5]*l[1]*l[2]/36],
          [a[0]*c[2,0]*l[0]*l[1]/36 - a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[3,0]*l[0]*l[2]/72 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/36 + a[0]*c[4,0]*l[1]*l[2]/36 + a[1]*c[2,1]*l[0]*l[1]/36 - a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[3,1]*l[0]*l[2]/72 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[4,1]*l[1]*l[2]/36 + a[2]*c[2,2]*l[0]*l[1]/36 - a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[3,2]*l[0]*l[2]/72 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[4,2]*l[1]*l[2]/36 + a[3]*c[2,3]*l[0]*l[1]/36 - a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[3,3]*l[0]*l[2]/72 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[4,3]*l[1]*l[2]/36 + a[4]*c[2,4]*l[0]*l[1]/36 - a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[3,4]*l[0]*l[2]/72 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[4,4]*l[1]*l[2]/36 + a[5]*c[2,5]*l[0]*l[1]/36 - a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[3,5]*l[0]*l[2]/72 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[4,5]*l[1]*l[2]/36,
           a[0]*c[2,0]*l[0]*l[1]/18 - a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[3,0]*l[0]*l[2]/36 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/36 + a[0]*c[4,0]*l[1]*l[2]/36 + a[1]*c[2,1]*l[0]*l[1]/18 - a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[3,1]*l[0]*l[2]/36 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[4,1]*l[1]*l[2]/36 + a[2]*c[2,2]*l[0]*l[1]/18 - a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[3,2]*l[0]*l[2]/36 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[4,2]*l[1]*l[2]/36 + a[3]*c[2,3]*l[0]*l[1]/18 - a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[3,3]*l[0]*l[2]/36 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[4,3]*l[1]*l[2]/36 + a[4]*c[2,4]*l[0]*l[1]/18 - a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[3,4]*l[0]*l[2]/36 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[4,4]*l[1]*l[2]/36 + a[5]*c[2,5]*l[0]*l[1]/18 - a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[3,5]*l[0]*l[2]/36 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[4,5]*l[1]*l[2]/36,
           a[0]*c[2,0]*l[0]*l[1]/36 - a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/72 - a[0]*c[3,0]*l[0]*l[2]/36 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/72 + a[0]*c[4,0]*l[1]*l[2]/72 + a[1]*c[2,1]*l[0]*l[1]/36 - a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/72 - a[1]*c[3,1]*l[0]*l[2]/36 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/72 + a[1]*c[4,1]*l[1]*l[2]/72 + a[2]*c[2,2]*l[0]*l[1]/36 - a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/72 - a[2]*c[3,2]*l[0]*l[2]/36 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/72 + a[2]*c[4,2]*l[1]*l[2]/72 + a[3]*c[2,3]*l[0]*l[1]/36 - a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/72 - a[3]*c[3,3]*l[0]*l[2]/36 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/72 + a[3]*c[4,3]*l[1]*l[2]/72 + a[4]*c[2,4]*l[0]*l[1]/36 - a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/72 - a[4]*c[3,4]*l[0]*l[2]/36 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/72 + a[4]*c[4,4]*l[1]*l[2]/72 + a[5]*c[2,5]*l[0]*l[1]/36 - a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/72 - a[5]*c[3,5]*l[0]*l[2]/36 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/72 + a[5]*c[4,5]*l[1]*l[2]/72,
           a[0]*c[2,0]*l[0]*l[1]/72 - a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/72 - a[0]*c[3,0]*l[0]*l[2]/72 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/72 + a[0]*c[4,0]*l[1]*l[2]/72 + a[1]*c[2,1]*l[0]*l[1]/72 - a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/72 - a[1]*c[3,1]*l[0]*l[2]/72 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/72 + a[1]*c[4,1]*l[1]*l[2]/72 + a[2]*c[2,2]*l[0]*l[1]/72 - a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/72 - a[2]*c[3,2]*l[0]*l[2]/72 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/72 + a[2]*c[4,2]*l[1]*l[2]/72 + a[3]*c[2,3]*l[0]*l[1]/72 - a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/72 - a[3]*c[3,3]*l[0]*l[2]/72 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/72 + a[3]*c[4,3]*l[1]*l[2]/72 + a[4]*c[2,4]*l[0]*l[1]/72 - a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/72 - a[4]*c[3,4]*l[0]*l[2]/72 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/72 + a[4]*c[4,4]*l[1]*l[2]/72 + a[5]*c[2,5]*l[0]*l[1]/72 - a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/72 - a[5]*c[3,5]*l[0]*l[2]/72 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/72 + a[5]*c[4,5]*l[1]*l[2]/72,
           a[0]*c[2,0]*l[0]*l[1]/36 - a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/18 - a[0]*c[3,0]*l[0]*l[2]/36 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/18 + a[0]*c[4,0]*l[1]*l[2]/18 + a[1]*c[2,1]*l[0]*l[1]/36 - a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/18 - a[1]*c[3,1]*l[0]*l[2]/36 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/18 + a[1]*c[4,1]*l[1]*l[2]/18 + a[2]*c[2,2]*l[0]*l[1]/36 - a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/18 - a[2]*c[3,2]*l[0]*l[2]/36 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/18 + a[2]*c[4,2]*l[1]*l[2]/18 + a[3]*c[2,3]*l[0]*l[1]/36 - a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/18 - a[3]*c[3,3]*l[0]*l[2]/36 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/18 + a[3]*c[4,3]*l[1]*l[2]/18 + a[4]*c[2,4]*l[0]*l[1]/36 - a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/18 - a[4]*c[3,4]*l[0]*l[2]/36 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/18 + a[4]*c[4,4]*l[1]*l[2]/18 + a[5]*c[2,5]*l[0]*l[1]/36 - a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/18 - a[5]*c[3,5]*l[0]*l[2]/36 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/18 + a[5]*c[4,5]*l[1]*l[2]/18,
           a[0]*c[2,0]*l[0]*l[1]/18 - a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/18 - a[0]*c[3,0]*l[0]*l[2]/18 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/18 + a[0]*c[4,0]*l[1]*l[2]/18 + a[1]*c[2,1]*l[0]*l[1]/18 - a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/18 - a[1]*c[3,1]*l[0]*l[2]/18 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/18 + a[1]*c[4,1]*l[1]*l[2]/18 + a[2]*c[2,2]*l[0]*l[1]/18 - a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/18 - a[2]*c[3,2]*l[0]*l[2]/18 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/18 + a[2]*c[4,2]*l[1]*l[2]/18 + a[3]*c[2,3]*l[0]*l[1]/18 - a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/18 - a[3]*c[3,3]*l[0]*l[2]/18 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/18 + a[3]*c[4,3]*l[1]*l[2]/18 + a[4]*c[2,4]*l[0]*l[1]/18 - a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/18 - a[4]*c[3,4]*l[0]*l[2]/18 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/18 + a[4]*c[4,4]*l[1]*l[2]/18 + a[5]*c[2,5]*l[0]*l[1]/18 - a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/18 - a[5]*c[3,5]*l[0]*l[2]/18 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/18 + a[5]*c[4,5]*l[1]*l[2]/18,
           a[0]*c[2,0]*l[0]*l[1]/36 - a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[3,0]*l[0]*l[2]/18 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/36 + a[0]*c[4,0]*l[1]*l[2]/36 + a[1]*c[2,1]*l[0]*l[1]/36 - a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[3,1]*l[0]*l[2]/18 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[4,1]*l[1]*l[2]/36 + a[2]*c[2,2]*l[0]*l[1]/36 - a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[3,2]*l[0]*l[2]/18 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[4,2]*l[1]*l[2]/36 + a[3]*c[2,3]*l[0]*l[1]/36 - a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[3,3]*l[0]*l[2]/18 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[4,3]*l[1]*l[2]/36 + a[4]*c[2,4]*l[0]*l[1]/36 - a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[3,4]*l[0]*l[2]/18 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[4,4]*l[1]*l[2]/36 + a[5]*c[2,5]*l[0]*l[1]/36 - a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[3,5]*l[0]*l[2]/18 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[4,5]*l[1]*l[2]/36,
           a[0]*c[2,0]*l[0]*l[1]/72 - a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[3,0]*l[0]*l[2]/36 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/36 + a[0]*c[4,0]*l[1]*l[2]/36 + a[1]*c[2,1]*l[0]*l[1]/72 - a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[3,1]*l[0]*l[2]/36 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[4,1]*l[1]*l[2]/36 + a[2]*c[2,2]*l[0]*l[1]/72 - a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[3,2]*l[0]*l[2]/36 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[4,2]*l[1]*l[2]/36 + a[3]*c[2,3]*l[0]*l[1]/72 - a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[3,3]*l[0]*l[2]/36 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[4,3]*l[1]*l[2]/36 + a[4]*c[2,4]*l[0]*l[1]/72 - a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[3,4]*l[0]*l[2]/36 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[4,4]*l[1]*l[2]/36 + a[5]*c[2,5]*l[0]*l[1]/72 - a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[3,5]*l[0]*l[2]/36 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[4,5]*l[1]*l[2]/36],
          [a[0]*c[0,0]*l[1]*l[2]/72 + a[0]*c[4,0]*l[0]*l[1]/72 - a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/72 + a[0]*c[5,0]*l[0]*l[2]/72 - a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/72 + a[1]*c[0,1]*l[1]*l[2]/72 + a[1]*c[4,1]*l[0]*l[1]/72 - a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/72 + a[1]*c[5,1]*l[0]*l[2]/72 - a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/72 + a[2]*c[0,2]*l[1]*l[2]/72 + a[2]*c[4,2]*l[0]*l[1]/72 - a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/72 + a[2]*c[5,2]*l[0]*l[2]/72 - a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/72 + a[3]*c[0,3]*l[1]*l[2]/72 + a[3]*c[4,3]*l[0]*l[1]/72 - a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/72 + a[3]*c[5,3]*l[0]*l[2]/72 - a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/72 + a[4]*c[0,4]*l[1]*l[2]/72 + a[4]*c[4,4]*l[0]*l[1]/72 - a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/72 + a[4]*c[5,4]*l[0]*l[2]/72 - a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/72 + a[5]*c[0,5]*l[1]*l[2]/72 + a[5]*c[4,5]*l[0]*l[1]/72 - a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/72 + a[5]*c[5,5]*l[0]*l[2]/72 - a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/72,
           a[0]*c[0,0]*l[1]*l[2]/72 + a[0]*c[4,0]*l[0]*l[1]/36 - a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/72 + a[0]*c[5,0]*l[0]*l[2]/36 - a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/72 + a[1]*c[0,1]*l[1]*l[2]/72 + a[1]*c[4,1]*l[0]*l[1]/36 - a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/72 + a[1]*c[5,1]*l[0]*l[2]/36 - a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/72 + a[2]*c[0,2]*l[1]*l[2]/72 + a[2]*c[4,2]*l[0]*l[1]/36 - a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/72 + a[2]*c[5,2]*l[0]*l[2]/36 - a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/72 + a[3]*c[0,3]*l[1]*l[2]/72 + a[3]*c[4,3]*l[0]*l[1]/36 - a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/72 + a[3]*c[5,3]*l[0]*l[2]/36 - a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/72 + a[4]*c[0,4]*l[1]*l[2]/72 + a[4]*c[4,4]*l[0]*l[1]/36 - a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/72 + a[4]*c[5,4]*l[0]*l[2]/36 - a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/72 + a[5]*c[0,5]*l[1]*l[2]/72 + a[5]*c[4,5]*l[0]*l[1]/36 - a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/72 + a[5]*c[5,5]*l[0]*l[2]/36 - a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/72,
           a[0]*c[0,0]*l[1]*l[2]/36 + a[0]*c[4,0]*l[0]*l[1]/18 - a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[5,0]*l[0]*l[2]/36 - a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[0,1]*l[1]*l[2]/36 + a[1]*c[4,1]*l[0]*l[1]/18 - a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[5,1]*l[0]*l[2]/36 - a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[0,2]*l[1]*l[2]/36 + a[2]*c[4,2]*l[0]*l[1]/18 - a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[5,2]*l[0]*l[2]/36 - a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[0,3]*l[1]*l[2]/36 + a[3]*c[4,3]*l[0]*l[1]/18 - a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[5,3]*l[0]*l[2]/36 - a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[0,4]*l[1]*l[2]/36 + a[4]*c[4,4]*l[0]*l[1]/18 - a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[5,4]*l[0]*l[2]/36 - a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[0,5]*l[1]*l[2]/36 + a[5]*c[4,5]*l[0]*l[1]/18 - a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[5,5]*l[0]*l[2]/36 - a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/36,
           a[0]*c[0,0]*l[1]*l[2]/36 + a[0]*c[4,0]*l[0]*l[1]/36 - a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[5,0]*l[0]*l[2]/72 - a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[0,1]*l[1]*l[2]/36 + a[1]*c[4,1]*l[0]*l[1]/36 - a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[5,1]*l[0]*l[2]/72 - a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[0,2]*l[1]*l[2]/36 + a[2]*c[4,2]*l[0]*l[1]/36 - a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[5,2]*l[0]*l[2]/72 - a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[0,3]*l[1]*l[2]/36 + a[3]*c[4,3]*l[0]*l[1]/36 - a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[5,3]*l[0]*l[2]/72 - a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[0,4]*l[1]*l[2]/36 + a[4]*c[4,4]*l[0]*l[1]/36 - a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[5,4]*l[0]*l[2]/72 - a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[0,5]*l[1]*l[2]/36 + a[5]*c[4,5]*l[0]*l[1]/36 - a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[5,5]*l[0]*l[2]/72 - a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/36,
           a[0]*c[0,0]*l[1]*l[2]/36 + a[0]*c[4,0]*l[0]*l[1]/72 - a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[5,0]*l[0]*l[2]/36 - a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[0,1]*l[1]*l[2]/36 + a[1]*c[4,1]*l[0]*l[1]/72 - a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[5,1]*l[0]*l[2]/36 - a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[0,2]*l[1]*l[2]/36 + a[2]*c[4,2]*l[0]*l[1]/72 - a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[5,2]*l[0]*l[2]/36 - a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[0,3]*l[1]*l[2]/36 + a[3]*c[4,3]*l[0]*l[1]/72 - a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[5,3]*l[0]*l[2]/36 - a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[0,4]*l[1]*l[2]/36 + a[4]*c[4,4]*l[0]*l[1]/72 - a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[5,4]*l[0]*l[2]/36 - a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[0,5]*l[1]*l[2]/36 + a[5]*c[4,5]*l[0]*l[1]/72 - a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[5,5]*l[0]*l[2]/36 - a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/36,
           a[0]*c[0,0]*l[1]*l[2]/36 + a[0]*c[4,0]*l[0]*l[1]/36 - a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[5,0]*l[0]*l[2]/18 - a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[0,1]*l[1]*l[2]/36 + a[1]*c[4,1]*l[0]*l[1]/36 - a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[5,1]*l[0]*l[2]/18 - a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[0,2]*l[1]*l[2]/36 + a[2]*c[4,2]*l[0]*l[1]/36 - a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[5,2]*l[0]*l[2]/18 - a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[0,3]*l[1]*l[2]/36 + a[3]*c[4,3]*l[0]*l[1]/36 - a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[5,3]*l[0]*l[2]/18 - a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[0,4]*l[1]*l[2]/36 + a[4]*c[4,4]*l[0]*l[1]/36 - a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[5,4]*l[0]*l[2]/18 - a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[0,5]*l[1]*l[2]/36 + a[5]*c[4,5]*l[0]*l[1]/36 - a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[5,5]*l[0]*l[2]/18 - a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/36,
           a[0]*c[0,0]*l[1]*l[2]/18 + a[0]*c[4,0]*l[0]*l[1]/18 - a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/18 + a[0]*c[5,0]*l[0]*l[2]/18 - a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/18 + a[1]*c[0,1]*l[1]*l[2]/18 + a[1]*c[4,1]*l[0]*l[1]/18 - a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/18 + a[1]*c[5,1]*l[0]*l[2]/18 - a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/18 + a[2]*c[0,2]*l[1]*l[2]/18 + a[2]*c[4,2]*l[0]*l[1]/18 - a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/18 + a[2]*c[5,2]*l[0]*l[2]/18 - a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/18 + a[3]*c[0,3]*l[1]*l[2]/18 + a[3]*c[4,3]*l[0]*l[1]/18 - a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/18 + a[3]*c[5,3]*l[0]*l[2]/18 - a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/18 + a[4]*c[0,4]*l[1]*l[2]/18 + a[4]*c[4,4]*l[0]*l[1]/18 - a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/18 + a[4]*c[5,4]*l[0]*l[2]/18 - a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/18 + a[5]*c[0,5]*l[1]*l[2]/18 + a[5]*c[4,5]*l[0]*l[1]/18 - a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/18 + a[5]*c[5,5]*l[0]*l[2]/18 - a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/18,
           a[0]*c[0,0]*l[1]*l[2]/18 + a[0]*c[4,0]*l[0]*l[1]/36 - a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/18 + a[0]*c[5,0]*l[0]*l[2]/36 - a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/18 + a[1]*c[0,1]*l[1]*l[2]/18 + a[1]*c[4,1]*l[0]*l[1]/36 - a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/18 + a[1]*c[5,1]*l[0]*l[2]/36 - a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/18 + a[2]*c[0,2]*l[1]*l[2]/18 + a[2]*c[4,2]*l[0]*l[1]/36 - a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/18 + a[2]*c[5,2]*l[0]*l[2]/36 - a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/18 + a[3]*c[0,3]*l[1]*l[2]/18 + a[3]*c[4,3]*l[0]*l[1]/36 - a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/18 + a[3]*c[5,3]*l[0]*l[2]/36 - a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/18 + a[4]*c[0,4]*l[1]*l[2]/18 + a[4]*c[4,4]*l[0]*l[1]/36 - a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/18 + a[4]*c[5,4]*l[0]*l[2]/36 - a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/18 + a[5]*c[0,5]*l[1]*l[2]/18 + a[5]*c[4,5]*l[0]*l[1]/36 - a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/18 + a[5]*c[5,5]*l[0]*l[2]/36 - a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/18],
          [a[0]*c[1,0]*l[0]*l[2]/72 - a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/72 + a[0]*c[3,0]*l[0]*l[1]/72 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/72 + a[0]*c[5,0]*l[1]*l[2]/72 + a[1]*c[1,1]*l[0]*l[2]/72 - a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/72 + a[1]*c[3,1]*l[0]*l[1]/72 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/72 + a[1]*c[5,1]*l[1]*l[2]/72 + a[2]*c[1,2]*l[0]*l[2]/72 - a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/72 + a[2]*c[3,2]*l[0]*l[1]/72 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/72 + a[2]*c[5,2]*l[1]*l[2]/72 + a[3]*c[1,3]*l[0]*l[2]/72 - a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/72 + a[3]*c[3,3]*l[0]*l[1]/72 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/72 + a[3]*c[5,3]*l[1]*l[2]/72 + a[4]*c[1,4]*l[0]*l[2]/72 - a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/72 + a[4]*c[3,4]*l[0]*l[1]/72 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/72 + a[4]*c[5,4]*l[1]*l[2]/72 + a[5]*c[1,5]*l[0]*l[2]/72 - a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/72 + a[5]*c[3,5]*l[0]*l[1]/72 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/72 + a[5]*c[5,5]*l[1]*l[2]/72,
           a[0]*c[1,0]*l[0]*l[2]/36 - a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/72 + a[0]*c[3,0]*l[0]*l[1]/36 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/72 + a[0]*c[5,0]*l[1]*l[2]/72 + a[1]*c[1,1]*l[0]*l[2]/36 - a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/72 + a[1]*c[3,1]*l[0]*l[1]/36 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/72 + a[1]*c[5,1]*l[1]*l[2]/72 + a[2]*c[1,2]*l[0]*l[2]/36 - a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/72 + a[2]*c[3,2]*l[0]*l[1]/36 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/72 + a[2]*c[5,2]*l[1]*l[2]/72 + a[3]*c[1,3]*l[0]*l[2]/36 - a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/72 + a[3]*c[3,3]*l[0]*l[1]/36 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/72 + a[3]*c[5,3]*l[1]*l[2]/72 + a[4]*c[1,4]*l[0]*l[2]/36 - a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/72 + a[4]*c[3,4]*l[0]*l[1]/36 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/72 + a[4]*c[5,4]*l[1]*l[2]/72 + a[5]*c[1,5]*l[0]*l[2]/36 - a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/72 + a[5]*c[3,5]*l[0]*l[1]/36 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/72 + a[5]*c[5,5]*l[1]*l[2]/72,
           a[0]*c[1,0]*l[0]*l[2]/36 - a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/36 + a[0]*c[3,0]*l[0]*l[1]/18 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[5,0]*l[1]*l[2]/36 + a[1]*c[1,1]*l[0]*l[2]/36 - a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[3,1]*l[0]*l[1]/18 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[5,1]*l[1]*l[2]/36 + a[2]*c[1,2]*l[0]*l[2]/36 - a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[3,2]*l[0]*l[1]/18 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[5,2]*l[1]*l[2]/36 + a[3]*c[1,3]*l[0]*l[2]/36 - a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[3,3]*l[0]*l[1]/18 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[5,3]*l[1]*l[2]/36 + a[4]*c[1,4]*l[0]*l[2]/36 - a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[3,4]*l[0]*l[1]/18 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[5,4]*l[1]*l[2]/36 + a[5]*c[1,5]*l[0]*l[2]/36 - a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[3,5]*l[0]*l[1]/18 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[5,5]*l[1]*l[2]/36,
           a[0]*c[1,0]*l[0]*l[2]/72 - a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/36 + a[0]*c[3,0]*l[0]*l[1]/36 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[5,0]*l[1]*l[2]/36 + a[1]*c[1,1]*l[0]*l[2]/72 - a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[3,1]*l[0]*l[1]/36 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[5,1]*l[1]*l[2]/36 + a[2]*c[1,2]*l[0]*l[2]/72 - a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[3,2]*l[0]*l[1]/36 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[5,2]*l[1]*l[2]/36 + a[3]*c[1,3]*l[0]*l[2]/72 - a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[3,3]*l[0]*l[1]/36 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[5,3]*l[1]*l[2]/36 + a[4]*c[1,4]*l[0]*l[2]/72 - a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[3,4]*l[0]*l[1]/36 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[5,4]*l[1]*l[2]/36 + a[5]*c[1,5]*l[0]*l[2]/72 - a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[3,5]*l[0]*l[1]/36 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[5,5]*l[1]*l[2]/36,
           a[0]*c[1,0]*l[0]*l[2]/36 - a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/36 + a[0]*c[3,0]*l[0]*l[1]/72 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[5,0]*l[1]*l[2]/36 + a[1]*c[1,1]*l[0]*l[2]/36 - a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[3,1]*l[0]*l[1]/72 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[5,1]*l[1]*l[2]/36 + a[2]*c[1,2]*l[0]*l[2]/36 - a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[3,2]*l[0]*l[1]/72 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[5,2]*l[1]*l[2]/36 + a[3]*c[1,3]*l[0]*l[2]/36 - a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[3,3]*l[0]*l[1]/72 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[5,3]*l[1]*l[2]/36 + a[4]*c[1,4]*l[0]*l[2]/36 - a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[3,4]*l[0]*l[1]/72 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[5,4]*l[1]*l[2]/36 + a[5]*c[1,5]*l[0]*l[2]/36 - a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[3,5]*l[0]*l[1]/72 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[5,5]*l[1]*l[2]/36,
           a[0]*c[1,0]*l[0]*l[2]/18 - a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/36 + a[0]*c[3,0]*l[0]*l[1]/36 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[5,0]*l[1]*l[2]/36 + a[1]*c[1,1]*l[0]*l[2]/18 - a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[3,1]*l[0]*l[1]/36 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[5,1]*l[1]*l[2]/36 + a[2]*c[1,2]*l[0]*l[2]/18 - a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[3,2]*l[0]*l[1]/36 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[5,2]*l[1]*l[2]/36 + a[3]*c[1,3]*l[0]*l[2]/18 - a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[3,3]*l[0]*l[1]/36 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[5,3]*l[1]*l[2]/36 + a[4]*c[1,4]*l[0]*l[2]/18 - a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[3,4]*l[0]*l[1]/36 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[5,4]*l[1]*l[2]/36 + a[5]*c[1,5]*l[0]*l[2]/18 - a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[3,5]*l[0]*l[1]/36 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[5,5]*l[1]*l[2]/36,
           a[0]*c[1,0]*l[0]*l[2]/18 - a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/18 + a[0]*c[3,0]*l[0]*l[1]/18 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/18 + a[0]*c[5,0]*l[1]*l[2]/18 + a[1]*c[1,1]*l[0]*l[2]/18 - a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/18 + a[1]*c[3,1]*l[0]*l[1]/18 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/18 + a[1]*c[5,1]*l[1]*l[2]/18 + a[2]*c[1,2]*l[0]*l[2]/18 - a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/18 + a[2]*c[3,2]*l[0]*l[1]/18 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/18 + a[2]*c[5,2]*l[1]*l[2]/18 + a[3]*c[1,3]*l[0]*l[2]/18 - a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/18 + a[3]*c[3,3]*l[0]*l[1]/18 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/18 + a[3]*c[5,3]*l[1]*l[2]/18 + a[4]*c[1,4]*l[0]*l[2]/18 - a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/18 + a[4]*c[3,4]*l[0]*l[1]/18 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/18 + a[4]*c[5,4]*l[1]*l[2]/18 + a[5]*c[1,5]*l[0]*l[2]/18 - a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/18 + a[5]*c[3,5]*l[0]*l[1]/18 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/18 + a[5]*c[5,5]*l[1]*l[2]/18,
           a[0]*c[1,0]*l[0]*l[2]/36 - a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/18 + a[0]*c[3,0]*l[0]*l[1]/36 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/18 + a[0]*c[5,0]*l[1]*l[2]/18 + a[1]*c[1,1]*l[0]*l[2]/36 - a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/18 + a[1]*c[3,1]*l[0]*l[1]/36 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/18 + a[1]*c[5,1]*l[1]*l[2]/18 + a[2]*c[1,2]*l[0]*l[2]/36 - a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/18 + a[2]*c[3,2]*l[0]*l[1]/36 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/18 + a[2]*c[5,2]*l[1]*l[2]/18 + a[3]*c[1,3]*l[0]*l[2]/36 - a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/18 + a[3]*c[3,3]*l[0]*l[1]/36 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/18 + a[3]*c[5,3]*l[1]*l[2]/18 + a[4]*c[1,4]*l[0]*l[2]/36 - a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/18 + a[4]*c[3,4]*l[0]*l[1]/36 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/18 + a[4]*c[5,4]*l[1]*l[2]/18 + a[5]*c[1,5]*l[0]*l[2]/36 - a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/18 + a[5]*c[3,5]*l[0]*l[1]/36 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/18 + a[5]*c[5,5]*l[1]*l[2]/18],
          [a[0]*c[2,0]*l[0]*l[1]/72 - a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/72 + a[0]*c[3,0]*l[0]*l[2]/72 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/72 + a[0]*c[4,0]*l[1]*l[2]/72 + a[1]*c[2,1]*l[0]*l[1]/72 - a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/72 + a[1]*c[3,1]*l[0]*l[2]/72 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/72 + a[1]*c[4,1]*l[1]*l[2]/72 + a[2]*c[2,2]*l[0]*l[1]/72 - a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/72 + a[2]*c[3,2]*l[0]*l[2]/72 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/72 + a[2]*c[4,2]*l[1]*l[2]/72 + a[3]*c[2,3]*l[0]*l[1]/72 - a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/72 + a[3]*c[3,3]*l[0]*l[2]/72 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/72 + a[3]*c[4,3]*l[1]*l[2]/72 + a[4]*c[2,4]*l[0]*l[1]/72 - a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/72 + a[4]*c[3,4]*l[0]*l[2]/72 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/72 + a[4]*c[4,4]*l[1]*l[2]/72 + a[5]*c[2,5]*l[0]*l[1]/72 - a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/72 + a[5]*c[3,5]*l[0]*l[2]/72 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/72 + a[5]*c[4,5]*l[1]*l[2]/72,
           a[0]*c[2,0]*l[0]*l[1]/36 - a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/72 + a[0]*c[3,0]*l[0]*l[2]/36 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/72 + a[0]*c[4,0]*l[1]*l[2]/72 + a[1]*c[2,1]*l[0]*l[1]/36 - a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/72 + a[1]*c[3,1]*l[0]*l[2]/36 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/72 + a[1]*c[4,1]*l[1]*l[2]/72 + a[2]*c[2,2]*l[0]*l[1]/36 - a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/72 + a[2]*c[3,2]*l[0]*l[2]/36 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/72 + a[2]*c[4,2]*l[1]*l[2]/72 + a[3]*c[2,3]*l[0]*l[1]/36 - a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/72 + a[3]*c[3,3]*l[0]*l[2]/36 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/72 + a[3]*c[4,3]*l[1]*l[2]/72 + a[4]*c[2,4]*l[0]*l[1]/36 - a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/72 + a[4]*c[3,4]*l[0]*l[2]/36 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/72 + a[4]*c[4,4]*l[1]*l[2]/72 + a[5]*c[2,5]*l[0]*l[1]/36 - a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/72 + a[5]*c[3,5]*l[0]*l[2]/36 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/72 + a[5]*c[4,5]*l[1]*l[2]/72,
           a[0]*c[2,0]*l[0]*l[1]/18 - a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[3,0]*l[0]*l[2]/36 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/36 + a[0]*c[4,0]*l[1]*l[2]/36 + a[1]*c[2,1]*l[0]*l[1]/18 - a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[3,1]*l[0]*l[2]/36 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[4,1]*l[1]*l[2]/36 + a[2]*c[2,2]*l[0]*l[1]/18 - a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[3,2]*l[0]*l[2]/36 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[4,2]*l[1]*l[2]/36 + a[3]*c[2,3]*l[0]*l[1]/18 - a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[3,3]*l[0]*l[2]/36 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[4,3]*l[1]*l[2]/36 + a[4]*c[2,4]*l[0]*l[1]/18 - a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[3,4]*l[0]*l[2]/36 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[4,4]*l[1]*l[2]/36 + a[5]*c[2,5]*l[0]*l[1]/18 - a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[3,5]*l[0]*l[2]/36 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[4,5]*l[1]*l[2]/36,
           a[0]*c[2,0]*l[0]*l[1]/36 - a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[3,0]*l[0]*l[2]/72 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/36 + a[0]*c[4,0]*l[1]*l[2]/36 + a[1]*c[2,1]*l[0]*l[1]/36 - a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[3,1]*l[0]*l[2]/72 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[4,1]*l[1]*l[2]/36 + a[2]*c[2,2]*l[0]*l[1]/36 - a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[3,2]*l[0]*l[2]/72 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[4,2]*l[1]*l[2]/36 + a[3]*c[2,3]*l[0]*l[1]/36 - a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[3,3]*l[0]*l[2]/72 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[4,3]*l[1]*l[2]/36 + a[4]*c[2,4]*l[0]*l[1]/36 - a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[3,4]*l[0]*l[2]/72 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[4,4]*l[1]*l[2]/36 + a[5]*c[2,5]*l[0]*l[1]/36 - a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[3,5]*l[0]*l[2]/72 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[4,5]*l[1]*l[2]/36,
           a[0]*c[2,0]*l[0]*l[1]/72 - a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[3,0]*l[0]*l[2]/36 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/36 + a[0]*c[4,0]*l[1]*l[2]/36 + a[1]*c[2,1]*l[0]*l[1]/72 - a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[3,1]*l[0]*l[2]/36 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[4,1]*l[1]*l[2]/36 + a[2]*c[2,2]*l[0]*l[1]/72 - a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[3,2]*l[0]*l[2]/36 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[4,2]*l[1]*l[2]/36 + a[3]*c[2,3]*l[0]*l[1]/72 - a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[3,3]*l[0]*l[2]/36 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[4,3]*l[1]*l[2]/36 + a[4]*c[2,4]*l[0]*l[1]/72 - a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[3,4]*l[0]*l[2]/36 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[4,4]*l[1]*l[2]/36 + a[5]*c[2,5]*l[0]*l[1]/72 - a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[3,5]*l[0]*l[2]/36 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[4,5]*l[1]*l[2]/36,
           a[0]*c[2,0]*l[0]*l[1]/36 - a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[3,0]*l[0]*l[2]/18 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/36 + a[0]*c[4,0]*l[1]*l[2]/36 + a[1]*c[2,1]*l[0]*l[1]/36 - a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[3,1]*l[0]*l[2]/18 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[4,1]*l[1]*l[2]/36 + a[2]*c[2,2]*l[0]*l[1]/36 - a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[3,2]*l[0]*l[2]/18 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[4,2]*l[1]*l[2]/36 + a[3]*c[2,3]*l[0]*l[1]/36 - a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[3,3]*l[0]*l[2]/18 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[4,3]*l[1]*l[2]/36 + a[4]*c[2,4]*l[0]*l[1]/36 - a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[3,4]*l[0]*l[2]/18 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[4,4]*l[1]*l[2]/36 + a[5]*c[2,5]*l[0]*l[1]/36 - a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[3,5]*l[0]*l[2]/18 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[4,5]*l[1]*l[2]/36,
           a[0]*c[2,0]*l[0]*l[1]/18 - a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/18 + a[0]*c[3,0]*l[0]*l[2]/18 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/18 + a[0]*c[4,0]*l[1]*l[2]/18 + a[1]*c[2,1]*l[0]*l[1]/18 - a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/18 + a[1]*c[3,1]*l[0]*l[2]/18 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/18 + a[1]*c[4,1]*l[1]*l[2]/18 + a[2]*c[2,2]*l[0]*l[1]/18 - a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/18 + a[2]*c[3,2]*l[0]*l[2]/18 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/18 + a[2]*c[4,2]*l[1]*l[2]/18 + a[3]*c[2,3]*l[0]*l[1]/18 - a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/18 + a[3]*c[3,3]*l[0]*l[2]/18 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/18 + a[3]*c[4,3]*l[1]*l[2]/18 + a[4]*c[2,4]*l[0]*l[1]/18 - a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/18 + a[4]*c[3,4]*l[0]*l[2]/18 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/18 + a[4]*c[4,4]*l[1]*l[2]/18 + a[5]*c[2,5]*l[0]*l[1]/18 - a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/18 + a[5]*c[3,5]*l[0]*l[2]/18 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/18 + a[5]*c[4,5]*l[1]*l[2]/18,
           a[0]*c[2,0]*l[0]*l[1]/36 - a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/18 + a[0]*c[3,0]*l[0]*l[2]/36 - a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/18 + a[0]*c[4,0]*l[1]*l[2]/18 + a[1]*c[2,1]*l[0]*l[1]/36 - a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/18 + a[1]*c[3,1]*l[0]*l[2]/36 - a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/18 + a[1]*c[4,1]*l[1]*l[2]/18 + a[2]*c[2,2]*l[0]*l[1]/36 - a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/18 + a[2]*c[3,2]*l[0]*l[2]/36 - a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/18 + a[2]*c[4,2]*l[1]*l[2]/18 + a[3]*c[2,3]*l[0]*l[1]/36 - a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/18 + a[3]*c[3,3]*l[0]*l[2]/36 - a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/18 + a[3]*c[4,3]*l[1]*l[2]/18 + a[4]*c[2,4]*l[0]*l[1]/36 - a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/18 + a[4]*c[3,4]*l[0]*l[2]/36 - a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/18 + a[4]*c[4,4]*l[1]*l[2]/18 + a[5]*c[2,5]*l[0]*l[1]/36 - a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/18 + a[5]*c[3,5]*l[0]*l[2]/36 - a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/18 + a[5]*c[4,5]*l[1]*l[2]/18],
          [-a[0]*c[0,0]*l[1]*l[2]/72 + a[0]*c[4,0]*l[0]*l[1]/36 + a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/72 + a[0]*c[5,0]*l[0]*l[2]/36 + a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/72 - a[1]*c[0,1]*l[1]*l[2]/72 + a[1]*c[4,1]*l[0]*l[1]/36 + a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/72 + a[1]*c[5,1]*l[0]*l[2]/36 + a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/72 - a[2]*c[0,2]*l[1]*l[2]/72 + a[2]*c[4,2]*l[0]*l[1]/36 + a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/72 + a[2]*c[5,2]*l[0]*l[2]/36 + a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/72 - a[3]*c[0,3]*l[1]*l[2]/72 + a[3]*c[4,3]*l[0]*l[1]/36 + a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/72 + a[3]*c[5,3]*l[0]*l[2]/36 + a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/72 - a[4]*c[0,4]*l[1]*l[2]/72 + a[4]*c[4,4]*l[0]*l[1]/36 + a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/72 + a[4]*c[5,4]*l[0]*l[2]/36 + a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/72 - a[5]*c[0,5]*l[1]*l[2]/72 + a[5]*c[4,5]*l[0]*l[1]/36 + a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/72 + a[5]*c[5,5]*l[0]*l[2]/36 + a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/72,
           -a[0]*c[0,0]*l[1]*l[2]/72 + a[0]*c[4,0]*l[0]*l[1]/72 + a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/72 + a[0]*c[5,0]*l[0]*l[2]/72 + a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/72 - a[1]*c[0,1]*l[1]*l[2]/72 + a[1]*c[4,1]*l[0]*l[1]/72 + a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/72 + a[1]*c[5,1]*l[0]*l[2]/72 + a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/72 - a[2]*c[0,2]*l[1]*l[2]/72 + a[2]*c[4,2]*l[0]*l[1]/72 + a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/72 + a[2]*c[5,2]*l[0]*l[2]/72 + a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/72 - a[3]*c[0,3]*l[1]*l[2]/72 + a[3]*c[4,3]*l[0]*l[1]/72 + a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/72 + a[3]*c[5,3]*l[0]*l[2]/72 + a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/72 - a[4]*c[0,4]*l[1]*l[2]/72 + a[4]*c[4,4]*l[0]*l[1]/72 + a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/72 + a[4]*c[5,4]*l[0]*l[2]/72 + a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/72 - a[5]*c[0,5]*l[1]*l[2]/72 + a[5]*c[4,5]*l[0]*l[1]/72 + a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/72 + a[5]*c[5,5]*l[0]*l[2]/72 + a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/72,
           -a[0]*c[0,0]*l[1]*l[2]/36 + a[0]*c[4,0]*l[0]*l[1]/36 + a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[5,0]*l[0]*l[2]/72 + a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[0,1]*l[1]*l[2]/36 + a[1]*c[4,1]*l[0]*l[1]/36 + a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[5,1]*l[0]*l[2]/72 + a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[0,2]*l[1]*l[2]/36 + a[2]*c[4,2]*l[0]*l[1]/36 + a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[5,2]*l[0]*l[2]/72 + a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[0,3]*l[1]*l[2]/36 + a[3]*c[4,3]*l[0]*l[1]/36 + a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[5,3]*l[0]*l[2]/72 + a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[0,4]*l[1]*l[2]/36 + a[4]*c[4,4]*l[0]*l[1]/36 + a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[5,4]*l[0]*l[2]/72 + a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[0,5]*l[1]*l[2]/36 + a[5]*c[4,5]*l[0]*l[1]/36 + a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[5,5]*l[0]*l[2]/72 + a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/36,
           -a[0]*c[0,0]*l[1]*l[2]/36 + a[0]*c[4,0]*l[0]*l[1]/18 + a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[5,0]*l[0]*l[2]/36 + a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[0,1]*l[1]*l[2]/36 + a[1]*c[4,1]*l[0]*l[1]/18 + a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[5,1]*l[0]*l[2]/36 + a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[0,2]*l[1]*l[2]/36 + a[2]*c[4,2]*l[0]*l[1]/18 + a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[5,2]*l[0]*l[2]/36 + a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[0,3]*l[1]*l[2]/36 + a[3]*c[4,3]*l[0]*l[1]/18 + a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[5,3]*l[0]*l[2]/36 + a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[0,4]*l[1]*l[2]/36 + a[4]*c[4,4]*l[0]*l[1]/18 + a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[5,4]*l[0]*l[2]/36 + a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[0,5]*l[1]*l[2]/36 + a[5]*c[4,5]*l[0]*l[1]/18 + a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[5,5]*l[0]*l[2]/36 + a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/36,
           -a[0]*c[0,0]*l[1]*l[2]/36 + a[0]*c[4,0]*l[0]*l[1]/36 + a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[5,0]*l[0]*l[2]/18 + a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[0,1]*l[1]*l[2]/36 + a[1]*c[4,1]*l[0]*l[1]/36 + a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[5,1]*l[0]*l[2]/18 + a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[0,2]*l[1]*l[2]/36 + a[2]*c[4,2]*l[0]*l[1]/36 + a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[5,2]*l[0]*l[2]/18 + a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[0,3]*l[1]*l[2]/36 + a[3]*c[4,3]*l[0]*l[1]/36 + a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[5,3]*l[0]*l[2]/18 + a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[0,4]*l[1]*l[2]/36 + a[4]*c[4,4]*l[0]*l[1]/36 + a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[5,4]*l[0]*l[2]/18 + a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[0,5]*l[1]*l[2]/36 + a[5]*c[4,5]*l[0]*l[1]/36 + a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[5,5]*l[0]*l[2]/18 + a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/36,
           -a[0]*c[0,0]*l[1]*l[2]/36 + a[0]*c[4,0]*l[0]*l[1]/72 + a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[5,0]*l[0]*l[2]/36 + a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[0,1]*l[1]*l[2]/36 + a[1]*c[4,1]*l[0]*l[1]/72 + a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[5,1]*l[0]*l[2]/36 + a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[0,2]*l[1]*l[2]/36 + a[2]*c[4,2]*l[0]*l[1]/72 + a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[5,2]*l[0]*l[2]/36 + a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[0,3]*l[1]*l[2]/36 + a[3]*c[4,3]*l[0]*l[1]/72 + a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[5,3]*l[0]*l[2]/36 + a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[0,4]*l[1]*l[2]/36 + a[4]*c[4,4]*l[0]*l[1]/72 + a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[5,4]*l[0]*l[2]/36 + a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[0,5]*l[1]*l[2]/36 + a[5]*c[4,5]*l[0]*l[1]/72 + a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[5,5]*l[0]*l[2]/36 + a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/36,
           -a[0]*c[0,0]*l[1]*l[2]/18 + a[0]*c[4,0]*l[0]*l[1]/36 + a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/18 + a[0]*c[5,0]*l[0]*l[2]/36 + a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/18 - a[1]*c[0,1]*l[1]*l[2]/18 + a[1]*c[4,1]*l[0]*l[1]/36 + a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/18 + a[1]*c[5,1]*l[0]*l[2]/36 + a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/18 - a[2]*c[0,2]*l[1]*l[2]/18 + a[2]*c[4,2]*l[0]*l[1]/36 + a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/18 + a[2]*c[5,2]*l[0]*l[2]/36 + a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/18 - a[3]*c[0,3]*l[1]*l[2]/18 + a[3]*c[4,3]*l[0]*l[1]/36 + a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/18 + a[3]*c[5,3]*l[0]*l[2]/36 + a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/18 - a[4]*c[0,4]*l[1]*l[2]/18 + a[4]*c[4,4]*l[0]*l[1]/36 + a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/18 + a[4]*c[5,4]*l[0]*l[2]/36 + a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/18 - a[5]*c[0,5]*l[1]*l[2]/18 + a[5]*c[4,5]*l[0]*l[1]/36 + a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/18 + a[5]*c[5,5]*l[0]*l[2]/36 + a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/18,
           -a[0]*c[0,0]*l[1]*l[2]/18 + a[0]*c[4,0]*l[0]*l[1]/18 + a[0]*c[4,0]*l[1]*l[2]*np.tan(g[1])/18 + a[0]*c[5,0]*l[0]*l[2]/18 + a[0]*c[5,0]*l[1]*l[2]*np.tan(g[0])/18 - a[1]*c[0,1]*l[1]*l[2]/18 + a[1]*c[4,1]*l[0]*l[1]/18 + a[1]*c[4,1]*l[1]*l[2]*np.tan(g[1])/18 + a[1]*c[5,1]*l[0]*l[2]/18 + a[1]*c[5,1]*l[1]*l[2]*np.tan(g[0])/18 - a[2]*c[0,2]*l[1]*l[2]/18 + a[2]*c[4,2]*l[0]*l[1]/18 + a[2]*c[4,2]*l[1]*l[2]*np.tan(g[1])/18 + a[2]*c[5,2]*l[0]*l[2]/18 + a[2]*c[5,2]*l[1]*l[2]*np.tan(g[0])/18 - a[3]*c[0,3]*l[1]*l[2]/18 + a[3]*c[4,3]*l[0]*l[1]/18 + a[3]*c[4,3]*l[1]*l[2]*np.tan(g[1])/18 + a[3]*c[5,3]*l[0]*l[2]/18 + a[3]*c[5,3]*l[1]*l[2]*np.tan(g[0])/18 - a[4]*c[0,4]*l[1]*l[2]/18 + a[4]*c[4,4]*l[0]*l[1]/18 + a[4]*c[4,4]*l[1]*l[2]*np.tan(g[1])/18 + a[4]*c[5,4]*l[0]*l[2]/18 + a[4]*c[5,4]*l[1]*l[2]*np.tan(g[0])/18 - a[5]*c[0,5]*l[1]*l[2]/18 + a[5]*c[4,5]*l[0]*l[1]/18 + a[5]*c[4,5]*l[1]*l[2]*np.tan(g[1])/18 + a[5]*c[5,5]*l[0]*l[2]/18 + a[5]*c[5,5]*l[1]*l[2]*np.tan(g[0])/18],
          [a[0]*c[1,0]*l[0]*l[2]/36 + a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/72 + a[0]*c[3,0]*l[0]*l[1]/36 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/72 - a[0]*c[5,0]*l[1]*l[2]/72 + a[1]*c[1,1]*l[0]*l[2]/36 + a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/72 + a[1]*c[3,1]*l[0]*l[1]/36 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/72 - a[1]*c[5,1]*l[1]*l[2]/72 + a[2]*c[1,2]*l[0]*l[2]/36 + a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/72 + a[2]*c[3,2]*l[0]*l[1]/36 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/72 - a[2]*c[5,2]*l[1]*l[2]/72 + a[3]*c[1,3]*l[0]*l[2]/36 + a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/72 + a[3]*c[3,3]*l[0]*l[1]/36 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/72 - a[3]*c[5,3]*l[1]*l[2]/72 + a[4]*c[1,4]*l[0]*l[2]/36 + a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/72 + a[4]*c[3,4]*l[0]*l[1]/36 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/72 - a[4]*c[5,4]*l[1]*l[2]/72 + a[5]*c[1,5]*l[0]*l[2]/36 + a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/72 + a[5]*c[3,5]*l[0]*l[1]/36 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/72 - a[5]*c[5,5]*l[1]*l[2]/72,
           a[0]*c[1,0]*l[0]*l[2]/72 + a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/72 + a[0]*c[3,0]*l[0]*l[1]/72 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/72 - a[0]*c[5,0]*l[1]*l[2]/72 + a[1]*c[1,1]*l[0]*l[2]/72 + a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/72 + a[1]*c[3,1]*l[0]*l[1]/72 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/72 - a[1]*c[5,1]*l[1]*l[2]/72 + a[2]*c[1,2]*l[0]*l[2]/72 + a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/72 + a[2]*c[3,2]*l[0]*l[1]/72 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/72 - a[2]*c[5,2]*l[1]*l[2]/72 + a[3]*c[1,3]*l[0]*l[2]/72 + a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/72 + a[3]*c[3,3]*l[0]*l[1]/72 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/72 - a[3]*c[5,3]*l[1]*l[2]/72 + a[4]*c[1,4]*l[0]*l[2]/72 + a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/72 + a[4]*c[3,4]*l[0]*l[1]/72 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/72 - a[4]*c[5,4]*l[1]*l[2]/72 + a[5]*c[1,5]*l[0]*l[2]/72 + a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/72 + a[5]*c[3,5]*l[0]*l[1]/72 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/72 - a[5]*c[5,5]*l[1]*l[2]/72,
           a[0]*c[1,0]*l[0]*l[2]/72 + a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/36 + a[0]*c[3,0]*l[0]*l[1]/36 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[5,0]*l[1]*l[2]/36 + a[1]*c[1,1]*l[0]*l[2]/72 + a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[3,1]*l[0]*l[1]/36 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[5,1]*l[1]*l[2]/36 + a[2]*c[1,2]*l[0]*l[2]/72 + a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[3,2]*l[0]*l[1]/36 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[5,2]*l[1]*l[2]/36 + a[3]*c[1,3]*l[0]*l[2]/72 + a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[3,3]*l[0]*l[1]/36 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[5,3]*l[1]*l[2]/36 + a[4]*c[1,4]*l[0]*l[2]/72 + a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[3,4]*l[0]*l[1]/36 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[5,4]*l[1]*l[2]/36 + a[5]*c[1,5]*l[0]*l[2]/72 + a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[3,5]*l[0]*l[1]/36 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[5,5]*l[1]*l[2]/36,
           a[0]*c[1,0]*l[0]*l[2]/36 + a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/36 + a[0]*c[3,0]*l[0]*l[1]/18 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[5,0]*l[1]*l[2]/36 + a[1]*c[1,1]*l[0]*l[2]/36 + a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[3,1]*l[0]*l[1]/18 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[5,1]*l[1]*l[2]/36 + a[2]*c[1,2]*l[0]*l[2]/36 + a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[3,2]*l[0]*l[1]/18 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[5,2]*l[1]*l[2]/36 + a[3]*c[1,3]*l[0]*l[2]/36 + a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[3,3]*l[0]*l[1]/18 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[5,3]*l[1]*l[2]/36 + a[4]*c[1,4]*l[0]*l[2]/36 + a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[3,4]*l[0]*l[1]/18 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[5,4]*l[1]*l[2]/36 + a[5]*c[1,5]*l[0]*l[2]/36 + a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[3,5]*l[0]*l[1]/18 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[5,5]*l[1]*l[2]/36,
           a[0]*c[1,0]*l[0]*l[2]/18 + a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/36 + a[0]*c[3,0]*l[0]*l[1]/36 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[5,0]*l[1]*l[2]/36 + a[1]*c[1,1]*l[0]*l[2]/18 + a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[3,1]*l[0]*l[1]/36 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[5,1]*l[1]*l[2]/36 + a[2]*c[1,2]*l[0]*l[2]/18 + a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[3,2]*l[0]*l[1]/36 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[5,2]*l[1]*l[2]/36 + a[3]*c[1,3]*l[0]*l[2]/18 + a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[3,3]*l[0]*l[1]/36 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[5,3]*l[1]*l[2]/36 + a[4]*c[1,4]*l[0]*l[2]/18 + a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[3,4]*l[0]*l[1]/36 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[5,4]*l[1]*l[2]/36 + a[5]*c[1,5]*l[0]*l[2]/18 + a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[3,5]*l[0]*l[1]/36 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[5,5]*l[1]*l[2]/36,
           a[0]*c[1,0]*l[0]*l[2]/36 + a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/36 + a[0]*c[3,0]*l[0]*l[1]/72 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/36 - a[0]*c[5,0]*l[1]*l[2]/36 + a[1]*c[1,1]*l[0]*l[2]/36 + a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/36 + a[1]*c[3,1]*l[0]*l[1]/72 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/36 - a[1]*c[5,1]*l[1]*l[2]/36 + a[2]*c[1,2]*l[0]*l[2]/36 + a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/36 + a[2]*c[3,2]*l[0]*l[1]/72 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/36 - a[2]*c[5,2]*l[1]*l[2]/36 + a[3]*c[1,3]*l[0]*l[2]/36 + a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/36 + a[3]*c[3,3]*l[0]*l[1]/72 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/36 - a[3]*c[5,3]*l[1]*l[2]/36 + a[4]*c[1,4]*l[0]*l[2]/36 + a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/36 + a[4]*c[3,4]*l[0]*l[1]/72 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/36 - a[4]*c[5,4]*l[1]*l[2]/36 + a[5]*c[1,5]*l[0]*l[2]/36 + a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/36 + a[5]*c[3,5]*l[0]*l[1]/72 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/36 - a[5]*c[5,5]*l[1]*l[2]/36,
           a[0]*c[1,0]*l[0]*l[2]/36 + a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/18 + a[0]*c[3,0]*l[0]*l[1]/36 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/18 - a[0]*c[5,0]*l[1]*l[2]/18 + a[1]*c[1,1]*l[0]*l[2]/36 + a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/18 + a[1]*c[3,1]*l[0]*l[1]/36 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/18 - a[1]*c[5,1]*l[1]*l[2]/18 + a[2]*c[1,2]*l[0]*l[2]/36 + a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/18 + a[2]*c[3,2]*l[0]*l[1]/36 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/18 - a[2]*c[5,2]*l[1]*l[2]/18 + a[3]*c[1,3]*l[0]*l[2]/36 + a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/18 + a[3]*c[3,3]*l[0]*l[1]/36 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/18 - a[3]*c[5,3]*l[1]*l[2]/18 + a[4]*c[1,4]*l[0]*l[2]/36 + a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/18 + a[4]*c[3,4]*l[0]*l[1]/36 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/18 - a[4]*c[5,4]*l[1]*l[2]/18 + a[5]*c[1,5]*l[0]*l[2]/36 + a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/18 + a[5]*c[3,5]*l[0]*l[1]/36 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/18 - a[5]*c[5,5]*l[1]*l[2]/18,
           a[0]*c[1,0]*l[0]*l[2]/18 + a[0]*c[1,0]*l[1]*l[2]*np.tan(g[0])/18 + a[0]*c[3,0]*l[0]*l[1]/18 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[1])/18 - a[0]*c[5,0]*l[1]*l[2]/18 + a[1]*c[1,1]*l[0]*l[2]/18 + a[1]*c[1,1]*l[1]*l[2]*np.tan(g[0])/18 + a[1]*c[3,1]*l[0]*l[1]/18 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[1])/18 - a[1]*c[5,1]*l[1]*l[2]/18 + a[2]*c[1,2]*l[0]*l[2]/18 + a[2]*c[1,2]*l[1]*l[2]*np.tan(g[0])/18 + a[2]*c[3,2]*l[0]*l[1]/18 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[1])/18 - a[2]*c[5,2]*l[1]*l[2]/18 + a[3]*c[1,3]*l[0]*l[2]/18 + a[3]*c[1,3]*l[1]*l[2]*np.tan(g[0])/18 + a[3]*c[3,3]*l[0]*l[1]/18 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[1])/18 - a[3]*c[5,3]*l[1]*l[2]/18 + a[4]*c[1,4]*l[0]*l[2]/18 + a[4]*c[1,4]*l[1]*l[2]*np.tan(g[0])/18 + a[4]*c[3,4]*l[0]*l[1]/18 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[1])/18 - a[4]*c[5,4]*l[1]*l[2]/18 + a[5]*c[1,5]*l[0]*l[2]/18 + a[5]*c[1,5]*l[1]*l[2]*np.tan(g[0])/18 + a[5]*c[3,5]*l[0]*l[1]/18 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[1])/18 - a[5]*c[5,5]*l[1]*l[2]/18],
          [a[0]*c[2,0]*l[0]*l[1]/36 + a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/72 + a[0]*c[3,0]*l[0]*l[2]/36 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/72 - a[0]*c[4,0]*l[1]*l[2]/72 + a[1]*c[2,1]*l[0]*l[1]/36 + a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/72 + a[1]*c[3,1]*l[0]*l[2]/36 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/72 - a[1]*c[4,1]*l[1]*l[2]/72 + a[2]*c[2,2]*l[0]*l[1]/36 + a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/72 + a[2]*c[3,2]*l[0]*l[2]/36 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/72 - a[2]*c[4,2]*l[1]*l[2]/72 + a[3]*c[2,3]*l[0]*l[1]/36 + a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/72 + a[3]*c[3,3]*l[0]*l[2]/36 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/72 - a[3]*c[4,3]*l[1]*l[2]/72 + a[4]*c[2,4]*l[0]*l[1]/36 + a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/72 + a[4]*c[3,4]*l[0]*l[2]/36 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/72 - a[4]*c[4,4]*l[1]*l[2]/72 + a[5]*c[2,5]*l[0]*l[1]/36 + a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/72 + a[5]*c[3,5]*l[0]*l[2]/36 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/72 - a[5]*c[4,5]*l[1]*l[2]/72,
           a[0]*c[2,0]*l[0]*l[1]/72 + a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/72 + a[0]*c[3,0]*l[0]*l[2]/72 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/72 - a[0]*c[4,0]*l[1]*l[2]/72 + a[1]*c[2,1]*l[0]*l[1]/72 + a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/72 + a[1]*c[3,1]*l[0]*l[2]/72 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/72 - a[1]*c[4,1]*l[1]*l[2]/72 + a[2]*c[2,2]*l[0]*l[1]/72 + a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/72 + a[2]*c[3,2]*l[0]*l[2]/72 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/72 - a[2]*c[4,2]*l[1]*l[2]/72 + a[3]*c[2,3]*l[0]*l[1]/72 + a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/72 + a[3]*c[3,3]*l[0]*l[2]/72 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/72 - a[3]*c[4,3]*l[1]*l[2]/72 + a[4]*c[2,4]*l[0]*l[1]/72 + a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/72 + a[4]*c[3,4]*l[0]*l[2]/72 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/72 - a[4]*c[4,4]*l[1]*l[2]/72 + a[5]*c[2,5]*l[0]*l[1]/72 + a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/72 + a[5]*c[3,5]*l[0]*l[2]/72 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/72 - a[5]*c[4,5]*l[1]*l[2]/72,
           a[0]*c[2,0]*l[0]*l[1]/36 + a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[3,0]*l[0]*l[2]/72 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/36 - a[0]*c[4,0]*l[1]*l[2]/36 + a[1]*c[2,1]*l[0]*l[1]/36 + a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[3,1]*l[0]*l[2]/72 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[4,1]*l[1]*l[2]/36 + a[2]*c[2,2]*l[0]*l[1]/36 + a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[3,2]*l[0]*l[2]/72 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[4,2]*l[1]*l[2]/36 + a[3]*c[2,3]*l[0]*l[1]/36 + a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[3,3]*l[0]*l[2]/72 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[4,3]*l[1]*l[2]/36 + a[4]*c[2,4]*l[0]*l[1]/36 + a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[3,4]*l[0]*l[2]/72 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[4,4]*l[1]*l[2]/36 + a[5]*c[2,5]*l[0]*l[1]/36 + a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[3,5]*l[0]*l[2]/72 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[4,5]*l[1]*l[2]/36,
           a[0]*c[2,0]*l[0]*l[1]/18 + a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[3,0]*l[0]*l[2]/36 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/36 - a[0]*c[4,0]*l[1]*l[2]/36 + a[1]*c[2,1]*l[0]*l[1]/18 + a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[3,1]*l[0]*l[2]/36 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[4,1]*l[1]*l[2]/36 + a[2]*c[2,2]*l[0]*l[1]/18 + a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[3,2]*l[0]*l[2]/36 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[4,2]*l[1]*l[2]/36 + a[3]*c[2,3]*l[0]*l[1]/18 + a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[3,3]*l[0]*l[2]/36 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[4,3]*l[1]*l[2]/36 + a[4]*c[2,4]*l[0]*l[1]/18 + a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[3,4]*l[0]*l[2]/36 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[4,4]*l[1]*l[2]/36 + a[5]*c[2,5]*l[0]*l[1]/18 + a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[3,5]*l[0]*l[2]/36 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[4,5]*l[1]*l[2]/36,
           a[0]*c[2,0]*l[0]*l[1]/36 + a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[3,0]*l[0]*l[2]/18 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/36 - a[0]*c[4,0]*l[1]*l[2]/36 + a[1]*c[2,1]*l[0]*l[1]/36 + a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[3,1]*l[0]*l[2]/18 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[4,1]*l[1]*l[2]/36 + a[2]*c[2,2]*l[0]*l[1]/36 + a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[3,2]*l[0]*l[2]/18 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[4,2]*l[1]*l[2]/36 + a[3]*c[2,3]*l[0]*l[1]/36 + a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[3,3]*l[0]*l[2]/18 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[4,3]*l[1]*l[2]/36 + a[4]*c[2,4]*l[0]*l[1]/36 + a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[3,4]*l[0]*l[2]/18 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[4,4]*l[1]*l[2]/36 + a[5]*c[2,5]*l[0]*l[1]/36 + a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[3,5]*l[0]*l[2]/18 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[4,5]*l[1]*l[2]/36,
           a[0]*c[2,0]*l[0]*l[1]/72 + a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/36 + a[0]*c[3,0]*l[0]*l[2]/36 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/36 - a[0]*c[4,0]*l[1]*l[2]/36 + a[1]*c[2,1]*l[0]*l[1]/72 + a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/36 + a[1]*c[3,1]*l[0]*l[2]/36 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/36 - a[1]*c[4,1]*l[1]*l[2]/36 + a[2]*c[2,2]*l[0]*l[1]/72 + a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/36 + a[2]*c[3,2]*l[0]*l[2]/36 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/36 - a[2]*c[4,2]*l[1]*l[2]/36 + a[3]*c[2,3]*l[0]*l[1]/72 + a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/36 + a[3]*c[3,3]*l[0]*l[2]/36 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/36 - a[3]*c[4,3]*l[1]*l[2]/36 + a[4]*c[2,4]*l[0]*l[1]/72 + a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/36 + a[4]*c[3,4]*l[0]*l[2]/36 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/36 - a[4]*c[4,4]*l[1]*l[2]/36 + a[5]*c[2,5]*l[0]*l[1]/72 + a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/36 + a[5]*c[3,5]*l[0]*l[2]/36 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/36 - a[5]*c[4,5]*l[1]*l[2]/36,
           a[0]*c[2,0]*l[0]*l[1]/36 + a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/18 + a[0]*c[3,0]*l[0]*l[2]/36 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/18 - a[0]*c[4,0]*l[1]*l[2]/18 + a[1]*c[2,1]*l[0]*l[1]/36 + a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/18 + a[1]*c[3,1]*l[0]*l[2]/36 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/18 - a[1]*c[4,1]*l[1]*l[2]/18 + a[2]*c[2,2]*l[0]*l[1]/36 + a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/18 + a[2]*c[3,2]*l[0]*l[2]/36 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/18 - a[2]*c[4,2]*l[1]*l[2]/18 + a[3]*c[2,3]*l[0]*l[1]/36 + a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/18 + a[3]*c[3,3]*l[0]*l[2]/36 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/18 - a[3]*c[4,3]*l[1]*l[2]/18 + a[4]*c[2,4]*l[0]*l[1]/36 + a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/18 + a[4]*c[3,4]*l[0]*l[2]/36 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/18 - a[4]*c[4,4]*l[1]*l[2]/18 + a[5]*c[2,5]*l[0]*l[1]/36 + a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/18 + a[5]*c[3,5]*l[0]*l[2]/36 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/18 - a[5]*c[4,5]*l[1]*l[2]/18,
           a[0]*c[2,0]*l[0]*l[1]/18 + a[0]*c[2,0]*l[1]*l[2]*np.tan(g[1])/18 + a[0]*c[3,0]*l[0]*l[2]/18 + a[0]*c[3,0]*l[1]*l[2]*np.tan(g[0])/18 - a[0]*c[4,0]*l[1]*l[2]/18 + a[1]*c[2,1]*l[0]*l[1]/18 + a[1]*c[2,1]*l[1]*l[2]*np.tan(g[1])/18 + a[1]*c[3,1]*l[0]*l[2]/18 + a[1]*c[3,1]*l[1]*l[2]*np.tan(g[0])/18 - a[1]*c[4,1]*l[1]*l[2]/18 + a[2]*c[2,2]*l[0]*l[1]/18 + a[2]*c[2,2]*l[1]*l[2]*np.tan(g[1])/18 + a[2]*c[3,2]*l[0]*l[2]/18 + a[2]*c[3,2]*l[1]*l[2]*np.tan(g[0])/18 - a[2]*c[4,2]*l[1]*l[2]/18 + a[3]*c[2,3]*l[0]*l[1]/18 + a[3]*c[2,3]*l[1]*l[2]*np.tan(g[1])/18 + a[3]*c[3,3]*l[0]*l[2]/18 + a[3]*c[3,3]*l[1]*l[2]*np.tan(g[0])/18 - a[3]*c[4,3]*l[1]*l[2]/18 + a[4]*c[2,4]*l[0]*l[1]/18 + a[4]*c[2,4]*l[1]*l[2]*np.tan(g[1])/18 + a[4]*c[3,4]*l[0]*l[2]/18 + a[4]*c[3,4]*l[1]*l[2]*np.tan(g[0])/18 - a[4]*c[4,4]*l[1]*l[2]/18 + a[5]*c[2,5]*l[0]*l[1]/18 + a[5]*c[2,5]*l[1]*l[2]*np.tan(g[1])/18 + a[5]*c[3,5]*l[0]*l[2]/18 + a[5]*c[3,5]*l[1]*l[2]*np.tan(g[0])/18 - a[5]*c[4,5]*l[1]*l[2]/18]])
    if DeltaT is None:
        return fe
    else:
        return fe@DeltaT
