# SPDX-License-Identifier: GPL-3.0-or-later
from typing import Any
from warnings import warn

import numpy as np

from topoptlab.fem import get_integrpoints
from topoptlab.elements.trilinear_hexahedron import bmatrix

def _lk_linear_elast_3d(xe: np.ndarray, c: np.ndarray,
                        quadr_method: str = "gauss-legendre",
                        nquad: int = 2,
                        **kwargs: Any) -> np.ndarray:
    """
    Create element stiffness matrix for 2D isotropic linear elasticity with
    bilinear quadrilateral Lagrangian elements in plane stress.

    Parameters
    ----------
    xe : np.ndarray, shape (nels,8,2)
        coordinates of element nodes. Please look at the
        definition/function of the shape function, then the node ordering is
        clear.
    c : np.ndarray, shape (nels,9,9) or
        stiffness tensor.
    quadr_method: str or callable
        name of quadrature method or function/callable that returns coordinates of
        quadrature points and weights. Check function get_integrpoints for
        available options.
    nquad : int
        number of quadrature points

    Returns
    -------
    Ke : np.ndarray, shape (nels,24,24)
        element stiffness matrix.

    """
    #
    if len(xe.shape) == 2:
        xe = xe[None,:,:]
    nel = xe.shape[0]
    #
    if len(c.shape) == 2:
        c = c[None,:,:]
    #
    x,w=get_integrpoints(ndim=3,nq=nquad,method=quadr_method)
    nq =w.shape[0]
    #
    xi,eta,zeta = [_x[:,0] for _x in np.split(x, 3,axis=1)]
    #
    nel = xe.shape[0]
    nq =w.shape[0]
    #
    B,detJ = bmatrix(xi=xi, eta=eta, zeta=zeta, xe=xe,
                     all_elems=True,
                     return_detJ=True)
    detJ = detJ.reshape(nel,nq)
    B = B.reshape(nel, nq,  B.shape[-2], B.shape[-1])
    #
    integral = B.transpose([0,1,3,2])@c[:,None,:,:]@B
    # multiply by determinant and quadrature
    return (w[None,:,None,None]*integral*detJ[:,:,None,None]).sum(axis=1)

def lk_linear_elast_3d(E: float = 1, nu:float = 0.3,
                       l: np.ndarray = np.array([1.,1.,1.]), 
                       g: np.ndarray = np.array([0.,0.]),
                       **kwargs: Any) -> np.ndarray:
    """
    Create element stiffness matrix for 3D isotropic linear elasticity with
    hexahedron elements. Taken and adjusted from the new generation 99 line
    code.

    Parameters
    ----------
    E : float
        Young's modulus.
    nu : float
        Poisson' ratio.
    l : np.ndarray (3)
        side length of element
    g : np.ndarray (2)
        angles of parallelogram.

    Returns
    -------
    Ke : np.ndarray, shape (24,24)
        element stiffness matrix.

    """
    return np.array([[E*(2*l[0]**2*l[1]**2*nu - 2*l[0]**2*l[1]**2 + 2*l[0]**2*l[2]**2*nu - 2*l[0]**2*l[2]**2 - 3*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) + 3*l[0]*l[1]**2*l[2]*np.tan(g[1]) - 3*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) + 3*l[0]*l[1]*l[2]**2*np.tan(g[0]) + 2*l[1]**2*l[2]**2*nu/np.cos(g[1])**2 + 2*l[1]**2*l[2]**2*nu/np.cos(g[0])**2 - 2*l[1]**2*l[2]**2/np.cos(g[1])**2 - 2*l[1]**2*l[2]**2/np.cos(g[0])**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(-3*l[0] + 4*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu - 1)),
           E*l[1]*(-3*l[0] + 4*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*(l[0]**2*l[1]**2*nu - l[0]**2*l[1]**2 + l[0]**2*l[2]**2*nu - l[0]**2*l[2]**2 - 2*l[1]**2*l[2]**2*nu/np.cos(g[1])**2 - 2*l[1]**2*l[2]**2*nu/np.cos(g[0])**2 + 2*l[1]**2*l[2]**2/np.cos(g[1])**2 + 2*l[1]**2*l[2]**2/np.cos(g[0])**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(-9*l[0]*nu + 3*l[0] - 4*l[1]*nu*np.tan(g[0]) - 4*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*l[1]*(-9*l[0]*nu + 3*l[0] - 4*l[2]*nu*np.tan(g[1]) - 4*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(l[0]**2*l[1]**2*nu - l[0]**2*l[1]**2 - 2*l[0]**2*l[2]**2*nu + 2*l[0]**2*l[2]**2 + 6*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) - 6*l[0]*l[1]*l[2]**2*np.tan(g[0]) - 2*l[1]**2*l[2]**2*nu/np.cos(g[1])**2 - 2*l[1]**2*l[2]**2*nu/np.cos(g[0])**2 + 2*l[1]**2*l[2]**2/np.cos(g[1])**2 + 2*l[1]**2*l[2]**2/np.cos(g[0])**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(3*l[0] - 2*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu - 1)),
           E*l[1]*(-9*l[0]*nu + 3*l[0] - 4*l[2]*nu*np.tan(g[1]) - 4*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*l[1]**2*nu - 2*l[0]**2*l[1]**2 - 4*l[0]**2*l[2]**2*nu + 4*l[0]**2*l[2]**2 - 3*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) + 3*l[0]*l[1]**2*l[2]*np.tan(g[1]) + 2*l[1]**2*l[2]**2*nu/np.cos(g[1])**2 + 2*l[1]**2*l[2]**2*nu/np.cos(g[0])**2 - 2*l[1]**2*l[2]**2/np.cos(g[1])**2 - 2*l[1]**2*l[2]**2/np.cos(g[0])**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(9*l[0]*nu - 3*l[0] + 2*l[1]*nu*np.tan(g[0]) + 2*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*l[1]*(-3*l[0] + 4*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*(-4*l[0]**2*l[1]**2*nu + 4*l[0]**2*l[1]**2 + 2*l[0]**2*l[2]**2*nu - 2*l[0]**2*l[2]**2 - 3*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) + 3*l[0]*l[1]*l[2]**2*np.tan(g[0]) + 2*l[1]**2*l[2]**2*nu/np.cos(g[1])**2 + 2*l[1]**2*l[2]**2*nu/np.cos(g[0])**2 - 2*l[1]**2*l[2]**2/np.cos(g[1])**2 - 2*l[1]**2*l[2]**2/np.cos(g[0])**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(-3*l[0] + 4*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu - 1)),
           E*l[1]*(9*l[0]*nu - 3*l[0] + 2*l[2]*nu*np.tan(g[1]) + 2*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(-2*l[0]**2*l[1]**2*nu + 2*l[0]**2*l[1]**2 + l[0]**2*l[2]**2*nu - l[0]**2*l[2]**2 + 6*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) - 6*l[0]*l[1]**2*l[2]*np.tan(g[1]) - 2*l[1]**2*l[2]**2*nu/np.cos(g[1])**2 - 2*l[1]**2*l[2]**2*nu/np.cos(g[0])**2 + 2*l[1]**2*l[2]**2/np.cos(g[1])**2 + 2*l[1]**2*l[2]**2/np.cos(g[0])**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(-9*l[0]*nu + 3*l[0] - 4*l[1]*nu*np.tan(g[0]) - 4*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[1]*(3*l[0] - 2*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*(3*l[0]**2*(-l[1]**2*nu + l[1]**2 - l[2]**2*nu + l[2]**2) + 2*l[0]**2*(l[1]**2*nu - l[1]**2 + l[2]**2*nu - l[2]**2) + l[1]*l[2]*(3*l[0]*l[1]*nu*np.tan(g[1]) - 3*l[0]*l[1]*np.tan(g[1]) + 3*l[0]*l[2]*nu*np.tan(g[0]) - 3*l[0]*l[2]*np.tan(g[0]) - l[1]*l[2]*nu/np.cos(g[1])**2 - l[1]*l[2]*nu/np.cos(g[0])**2 + l[1]*l[2]/np.cos(g[1])**2 + l[1]*l[2]/np.cos(g[0])**2))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(3*l[0] - 2*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu - 1)),
           E*l[1]*(3*l[0] - 2*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*(-2*l[0]**2*l[1]**2*nu + 2*l[0]**2*l[1]**2 - 2*l[0]**2*l[2]**2*nu + 2*l[0]**2*l[2]**2 + l[1]**2*l[2]**2*nu/np.cos(g[1])**2 + l[1]**2*l[2]**2*nu/np.cos(g[0])**2 - l[1]**2*l[2]**2/np.cos(g[1])**2 - l[1]**2*l[2]**2/np.cos(g[0])**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(9*l[0]*nu - 3*l[0] + 2*l[1]*nu*np.tan(g[0]) + 2*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[1]*(9*l[0]*nu - 3*l[0] + 2*l[2]*nu*np.tan(g[1]) + 2*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1))],
          [E*l[2]*(-3*l[0] + 4*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu - 1)),
           E*(2*l[0]**2*l[1]**2*nu - 2*l[0]**2*l[1]**2 + 4*l[0]**2*l[2]**2*nu - 4*l[0]**2*l[2]**2 - 3*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) + 3*l[0]*l[1]**2*l[2]*np.tan(g[1]) - 6*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) + 6*l[0]*l[1]*l[2]**2*np.tan(g[0]) + 4*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 + 2*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2*nu - 4*l[1]**2*l[2]**2*np.tan(g[0])**2 - 2*l[1]**2*l[2]**2*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(-3*l[0]**2 + 3*l[0]*l[1]*np.tan(g[0]) + 3*l[0]*l[2]*np.tan(g[1]) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*l[2]*(9*l[0]*nu - 3*l[0] - 4*l[1]*nu*np.tan(g[0]) - 4*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(l[0]**2*l[1]**2*nu - l[0]**2*l[1]**2 + 2*l[0]**2*l[2]**2*nu - 2*l[0]**2*l[2]**2 - 4*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - 2*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2*nu + 4*l[1]**2*l[2]**2*np.tan(g[0])**2 + 2*l[1]**2*l[2]**2*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(-3*l[0]**2*nu - 3*l[0]**2 + 18*l[0]*l[1]*nu*np.tan(g[0]) - 6*l[0]*l[1]*np.tan(g[0]) - 18*l[0]*l[2]*nu*np.tan(g[1]) + 6*l[0]*l[2]*np.tan(g[1]) + 8*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 8*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[2]*(3*l[0] - 2*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu - 1)),
           E*(l[0]**2*l[1]**2*nu - l[0]**2*l[1]**2 - 4*l[0]**2*l[2]**2*nu + 4*l[0]**2*l[2]**2 + 12*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) - 12*l[0]*l[1]*l[2]**2*np.tan(g[0]) - 4*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - 2*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2*nu + 4*l[1]**2*l[2]**2*np.tan(g[0])**2 + 2*l[1]**2*l[2]**2*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(-9*l[0]**2*nu + 3*l[0]**2 + 9*l[0]*l[1]*nu*np.tan(g[0]) - 3*l[0]*l[1]*np.tan(g[0]) - 6*l[0]*l[2]*nu*np.tan(g[1]) - 6*l[0]*l[2]*np.tan(g[1]) + 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[2]*(-9*l[0]*nu + 3*l[0] + 2*l[1]*nu*np.tan(g[0]) + 2*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*l[1]**2*nu - 2*l[0]**2*l[1]**2 - 8*l[0]**2*l[2]**2*nu + 8*l[0]**2*l[2]**2 - 3*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) + 3*l[0]*l[1]**2*l[2]*np.tan(g[1]) + 4*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 + 2*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2*nu - 4*l[1]**2*l[2]**2*np.tan(g[0])**2 - 2*l[1]**2*l[2]**2*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(-18*l[0]**2*nu + 6*l[0]**2 + 3*l[0]*l[1]*nu*np.tan(g[0]) + 3*l[0]*l[1]*np.tan(g[0]) + 18*l[0]*l[2]*nu*np.tan(g[1]) - 6*l[0]*l[2]*np.tan(g[1]) - 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[2]*(-3*l[0] + 4*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu - 1)),
           E*(-2*l[0]**2*l[1]**2*nu + 2*l[0]**2*l[1]**2 + 2*l[0]**2*l[2]**2*nu - 2*l[0]**2*l[2]**2 - 3*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) + 3*l[0]*l[1]*l[2]**2*np.tan(g[0]) + 2*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 + l[1]**2*l[2]**2*nu*np.tan(g[1])**2 + l[1]**2*l[2]**2*nu - 2*l[1]**2*l[2]**2*np.tan(g[0])**2 - l[1]**2*l[2]**2*np.tan(g[1])**2 - l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(18*l[0]**2*nu - 6*l[0]**2 - 18*l[0]*l[1]*nu*np.tan(g[0]) + 6*l[0]*l[1]*np.tan(g[0]) + 3*l[0]*l[2]*nu*np.tan(g[1]) + 3*l[0]*l[2]*np.tan(g[1]) - 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[2]*(9*l[0]*nu - 3*l[0] - 4*l[1]*nu*np.tan(g[0]) - 4*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(-l[0]**2*l[1]**2*nu + l[0]**2*l[1]**2 + l[0]**2*l[2]**2*nu - l[0]**2*l[2]**2 + 3*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) - 3*l[0]*l[1]**2*l[2]*np.tan(g[1]) - 2*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - l[1]**2*l[2]**2*nu + 2*l[1]**2*l[2]**2*np.tan(g[0])**2 + l[1]**2*l[2]**2*np.tan(g[1])**2 + l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(9*l[0]**2*nu - 3*l[0]**2 - 6*l[0]*l[1]*nu*np.tan(g[0]) - 6*l[0]*l[1]*np.tan(g[0]) - 9*l[0]*l[2]*nu*np.tan(g[1]) + 3*l[0]*l[2]*np.tan(g[1]) + 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[2]*(3*l[0] - 2*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu - 1)),
           E*(3*l[0]**2*(-l[1]**2*nu + l[1]**2 - 2*l[2]**2*nu + 2*l[2]**2) + 2*l[0]**2*(l[1]**2*nu - l[1]**2 + 2*l[2]**2*nu - 2*l[2]**2) + l[1]*l[2]*(3*l[0]*l[1]*nu*np.tan(g[1]) - 3*l[0]*l[1]*np.tan(g[1]) + 6*l[0]*l[2]*nu*np.tan(g[0]) - 6*l[0]*l[2]*np.tan(g[0]) - 2*l[1]*l[2]*nu*np.tan(g[0])**2 - l[1]*l[2]*nu*np.tan(g[1])**2 - l[1]*l[2]*nu + 2*l[1]*l[2]*np.tan(g[0])**2 + l[1]*l[2]*np.tan(g[1])**2 + l[1]*l[2]))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(3*l[0]**2 - 3*l[0]*l[1]*np.tan(g[0]) - 3*l[0]*l[2]*np.tan(g[1]) + 2*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*l[2]*(-9*l[0]*nu + 3*l[0] + 2*l[1]*nu*np.tan(g[0]) + 2*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(-2*l[0]**2*l[1]**2*nu + 2*l[0]**2*l[1]**2 - 4*l[0]**2*l[2]**2*nu + 4*l[0]**2*l[2]**2 + 2*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 + l[1]**2*l[2]**2*nu*np.tan(g[1])**2 + l[1]**2*l[2]**2*nu - 2*l[1]**2*l[2]**2*np.tan(g[0])**2 - l[1]**2*l[2]**2*np.tan(g[1])**2 - l[1]**2*l[2]**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(6*l[0]**2*nu + 6*l[0]**2 - 9*l[0]*l[1]*nu*np.tan(g[0]) + 3*l[0]*l[1]*np.tan(g[0]) + 9*l[0]*l[2]*nu*np.tan(g[1]) - 3*l[0]*l[2]*np.tan(g[1]) - 2*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) - 2*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1))],
          [E*l[1]*(-3*l[0] + 4*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*(-3*l[0]**2 + 3*l[0]*l[1]*np.tan(g[0]) + 3*l[0]*l[2]*np.tan(g[1]) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*(4*l[0]**2*l[1]**2*nu - 4*l[0]**2*l[1]**2 + 2*l[0]**2*l[2]**2*nu - 2*l[0]**2*l[2]**2 - 6*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) + 6*l[0]*l[1]**2*l[2]*np.tan(g[1]) - 3*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) + 3*l[0]*l[1]*l[2]**2*np.tan(g[0]) + 2*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 + 4*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2*nu - 2*l[1]**2*l[2]**2*np.tan(g[0])**2 - 4*l[1]**2*l[2]**2*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(9*l[0]*nu - 3*l[0] - 4*l[2]*nu*np.tan(g[1]) - 4*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(-3*l[0]**2*nu - 3*l[0]**2 - 18*l[0]*l[1]*nu*np.tan(g[0]) + 6*l[0]*l[1]*np.tan(g[0]) + 18*l[0]*l[2]*nu*np.tan(g[1]) - 6*l[0]*l[2]*np.tan(g[1]) + 8*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 8*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*l[1]**2*nu - 2*l[0]**2*l[1]**2 + l[0]**2*l[2]**2*nu - l[0]**2*l[2]**2 - 2*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - 4*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2*nu + 2*l[1]**2*l[2]**2*np.tan(g[0])**2 + 4*l[1]**2*l[2]**2*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(9*l[0]*nu - 3*l[0] - 4*l[2]*nu*np.tan(g[1]) - 4*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(9*l[0]**2*nu - 3*l[0]**2 - 9*l[0]*l[1]*nu*np.tan(g[0]) + 3*l[0]*l[1]*np.tan(g[0]) - 6*l[0]*l[2]*nu*np.tan(g[1]) - 6*l[0]*l[2]*np.tan(g[1]) + 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(l[0]**2*l[1]**2*nu - l[0]**2*l[1]**2 - l[0]**2*l[2]**2*nu + l[0]**2*l[2]**2 + 3*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) - 3*l[0]*l[1]*l[2]**2*np.tan(g[0]) - l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - 2*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - l[1]**2*l[2]**2*nu + l[1]**2*l[2]**2*np.tan(g[0])**2 + 2*l[1]**2*l[2]**2*np.tan(g[1])**2 + l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(-3*l[0] + 4*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*(18*l[0]**2*nu - 6*l[0]**2 + 3*l[0]*l[1]*nu*np.tan(g[0]) + 3*l[0]*l[1]*np.tan(g[0]) - 18*l[0]*l[2]*nu*np.tan(g[1]) + 6*l[0]*l[2]*np.tan(g[1]) - 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*l[1]**2*nu - 2*l[0]**2*l[1]**2 - 2*l[0]**2*l[2]**2*nu + 2*l[0]**2*l[2]**2 - 3*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) + 3*l[0]*l[1]**2*l[2]*np.tan(g[1]) + l[1]**2*l[2]**2*nu*np.tan(g[0])**2 + 2*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 + l[1]**2*l[2]**2*nu - l[1]**2*l[2]**2*np.tan(g[0])**2 - 2*l[1]**2*l[2]**2*np.tan(g[1])**2 - l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(-9*l[0]*nu + 3*l[0] + 2*l[2]*nu*np.tan(g[1]) + 2*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(-18*l[0]**2*nu + 6*l[0]**2 + 18*l[0]*l[1]*nu*np.tan(g[0]) - 6*l[0]*l[1]*np.tan(g[0]) + 3*l[0]*l[2]*nu*np.tan(g[1]) + 3*l[0]*l[2]*np.tan(g[1]) - 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(-8*l[0]**2*l[1]**2*nu + 8*l[0]**2*l[1]**2 + 2*l[0]**2*l[2]**2*nu - 2*l[0]**2*l[2]**2 - 3*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) + 3*l[0]*l[1]*l[2]**2*np.tan(g[0]) + 2*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 + 4*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2*nu - 2*l[1]**2*l[2]**2*np.tan(g[0])**2 - 4*l[1]**2*l[2]**2*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(3*l[0] - 2*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*(-9*l[0]**2*nu + 3*l[0]**2 - 6*l[0]*l[1]*nu*np.tan(g[0]) - 6*l[0]*l[1]*np.tan(g[0]) + 9*l[0]*l[2]*nu*np.tan(g[1]) - 3*l[0]*l[2]*np.tan(g[1]) + 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(-4*l[0]**2*l[1]**2*nu + 4*l[0]**2*l[1]**2 + l[0]**2*l[2]**2*nu - l[0]**2*l[2]**2 + 12*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) - 12*l[0]*l[1]**2*l[2]*np.tan(g[1]) - 2*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - 4*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2*nu + 2*l[1]**2*l[2]**2*np.tan(g[0])**2 + 4*l[1]**2*l[2]**2*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(3*l[0] - 2*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*(3*l[0]**2 - 3*l[0]*l[1]*np.tan(g[0]) - 3*l[0]*l[2]*np.tan(g[1]) + 2*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*(3*l[0]**2*(-2*l[1]**2*nu + 2*l[1]**2 - l[2]**2*nu + l[2]**2) + 2*l[0]**2*(2*l[1]**2*nu - 2*l[1]**2 + l[2]**2*nu - l[2]**2) + l[1]*l[2]*(6*l[0]*l[1]*nu*np.tan(g[1]) - 6*l[0]*l[1]*np.tan(g[1]) + 3*l[0]*l[2]*nu*np.tan(g[0]) - 3*l[0]*l[2]*np.tan(g[0]) - l[1]*l[2]*nu*np.tan(g[0])**2 - 2*l[1]*l[2]*nu*np.tan(g[1])**2 - l[1]*l[2]*nu + l[1]*l[2]*np.tan(g[0])**2 + 2*l[1]*l[2]*np.tan(g[1])**2 + l[1]*l[2]))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(-9*l[0]*nu + 3*l[0] + 2*l[2]*nu*np.tan(g[1]) + 2*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(6*l[0]**2*nu + 6*l[0]**2 + 9*l[0]*l[1]*nu*np.tan(g[0]) - 3*l[0]*l[1]*np.tan(g[0]) - 9*l[0]*l[2]*nu*np.tan(g[1]) + 3*l[0]*l[2]*np.tan(g[1]) - 2*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) - 2*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(-4*l[0]**2*l[1]**2*nu + 4*l[0]**2*l[1]**2 - 2*l[0]**2*l[2]**2*nu + 2*l[0]**2*l[2]**2 + l[1]**2*l[2]**2*nu*np.tan(g[0])**2 + 2*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 + l[1]**2*l[2]**2*nu - l[1]**2*l[2]**2*np.tan(g[0])**2 - 2*l[1]**2*l[2]**2*np.tan(g[1])**2 - l[1]**2*l[2]**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1))],
          [E*(l[0]**2*l[1]**2*nu - l[0]**2*l[1]**2 + l[0]**2*l[2]**2*nu - l[0]**2*l[2]**2 - 2*l[1]**2*l[2]**2*nu/np.cos(g[1])**2 - 2*l[1]**2*l[2]**2*nu/np.cos(g[0])**2 + 2*l[1]**2*l[2]**2/np.cos(g[1])**2 + 2*l[1]**2*l[2]**2/np.cos(g[0])**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(9*l[0]*nu - 3*l[0] - 4*l[1]*nu*np.tan(g[0]) - 4*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*l[1]*(9*l[0]*nu - 3*l[0] - 4*l[2]*nu*np.tan(g[1]) - 4*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*(l[1]**2*nu - l[1]**2 + l[2]**2*nu - l[2]**2) + 3*l[0]*l[1]*l[2]*(l[1]*nu*np.tan(g[1]) - l[1]*np.tan(g[1]) + l[2]*nu*np.tan(g[0]) - l[2]*np.tan(g[0])) + 2*l[1]**2*l[2]**2*(nu/np.cos(g[1])**2 + nu/np.cos(g[0])**2 - 1/np.cos(g[1])**2 - 1/np.cos(g[0])**2))/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(3*l[0] + 4*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu - 1)),
           E*l[1]*(3*l[0] + 4*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*(2*l[0]**2*(l[1]**2*nu - l[1]**2 - 2*l[2]**2*nu + 2*l[2]**2) + 3*l[0]*l[1]**2*l[2]*(nu - 1)*np.tan(g[1]) + 2*l[1]**2*l[2]**2*(nu/np.cos(g[1])**2 + nu/np.cos(g[0])**2 - 1/np.cos(g[1])**2 - 1/np.cos(g[0])**2))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(-9*l[0]*nu + 3*l[0] + 2*l[1]*nu*np.tan(g[0]) + 2*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*l[1]*(3*l[0] + 4*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*(l[0]**2*l[1]**2*nu - l[0]**2*l[1]**2 - 2*l[0]**2*l[2]**2*nu + 2*l[0]**2*l[2]**2 - 6*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) + 6*l[0]*l[1]*l[2]**2*np.tan(g[0]) - 2*l[1]**2*l[2]**2*nu/np.cos(g[1])**2 - 2*l[1]**2*l[2]**2*nu/np.cos(g[0])**2 + 2*l[1]**2*l[2]**2/np.cos(g[1])**2 + 2*l[1]**2*l[2]**2/np.cos(g[0])**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           -E*l[2]*(3*l[0] + 2*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu - 1)),
           E*l[1]*(9*l[0]*nu - 3*l[0] - 4*l[2]*nu*np.tan(g[1]) - 4*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(-2*l[0]**2*l[1]**2*nu + 2*l[0]**2*l[1]**2 + l[0]**2*l[2]**2*nu - l[0]**2*l[2]**2 - 6*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) + 6*l[0]*l[1]**2*l[2]*np.tan(g[1]) - 2*l[1]**2*l[2]**2*nu/np.cos(g[1])**2 - 2*l[1]**2*l[2]**2*nu/np.cos(g[0])**2 + 2*l[1]**2*l[2]**2/np.cos(g[1])**2 + 2*l[1]**2*l[2]**2/np.cos(g[0])**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(9*l[0]*nu - 3*l[0] - 4*l[1]*nu*np.tan(g[0]) - 4*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu**2 + nu - 1)),
           -E*l[1]*(3*l[0] + 2*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*(2*l[0]**2*(-2*l[1]**2*nu + 2*l[1]**2 + l[2]**2*nu - l[2]**2) + 3*l[0]*l[1]*l[2]**2*(nu - 1)*np.tan(g[0]) + 2*l[1]**2*l[2]**2*(nu/np.cos(g[1])**2 + nu/np.cos(g[0])**2 - 1/np.cos(g[1])**2 - 1/np.cos(g[0])**2))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(3*l[0] + 4*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu - 1)),
           E*l[1]*(-9*l[0]*nu + 3*l[0] + 2*l[2]*nu*np.tan(g[1]) + 2*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*(-l[1]**2*nu + l[1]**2 - l[2]**2*nu + l[2]**2) + l[1]**2*l[2]**2*(nu/np.cos(g[1])**2 + nu/np.cos(g[0])**2 - 1/np.cos(g[1])**2 - 1/np.cos(g[0])**2))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(-9*l[0]*nu + 3*l[0] + 2*l[1]*nu*np.tan(g[0]) + 2*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[1]*(-9*l[0]*nu + 3*l[0] + 2*l[2]*nu*np.tan(g[1]) + 2*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(3*l[0]**2*(-l[1]**2*nu + l[1]**2 - l[2]**2*nu + l[2]**2) + 2*l[0]**2*(l[1]**2*nu - l[1]**2 + l[2]**2*nu - l[2]**2) + l[1]*l[2]*(-3*l[0]*l[1]*nu*np.tan(g[1]) + 3*l[0]*l[1]*np.tan(g[1]) - 3*l[0]*l[2]*nu*np.tan(g[0]) + 3*l[0]*l[2]*np.tan(g[0]) - l[1]*l[2]*nu/np.cos(g[1])**2 - l[1]*l[2]*nu/np.cos(g[0])**2 + l[1]*l[2]/np.cos(g[1])**2 + l[1]*l[2]/np.cos(g[0])**2))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           -E*l[2]*(3*l[0] + 2*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu - 1)),
           -E*l[1]*(3*l[0] + 2*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu - 1))],
          [E*l[2]*(-9*l[0]*nu + 3*l[0] - 4*l[1]*nu*np.tan(g[0]) - 4*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(l[0]**2*l[1]**2*nu - l[0]**2*l[1]**2 + 2*l[0]**2*l[2]**2*nu - 2*l[0]**2*l[2]**2 - 4*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - 2*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2*nu + 4*l[1]**2*l[2]**2*np.tan(g[0])**2 + 2*l[1]**2*l[2]**2*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(-3*l[0]**2*nu - 3*l[0]**2 - 18*l[0]*l[1]*nu*np.tan(g[0]) + 6*l[0]*l[1]*np.tan(g[0]) + 18*l[0]*l[2]*nu*np.tan(g[1]) - 6*l[0]*l[2]*np.tan(g[1]) + 8*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 8*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[2]*(3*l[0] + 4*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu - 1)),
           E*(2*l[0]**2*(l[1]**2*nu - l[1]**2 + 2*l[2]**2*nu - 2*l[2]**2) + 3*l[0]*l[1]*l[2]*(l[1]*nu*np.tan(g[1]) - l[1]*np.tan(g[1]) + 2*l[2]*nu*np.tan(g[0]) - 2*l[2]*np.tan(g[0])) + 2*l[1]**2*l[2]**2*(2*nu*np.tan(g[0])**2 + nu*np.tan(g[1])**2 + nu - 2*np.tan(g[0])**2 - np.tan(g[1])**2 - 1))/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(-3*l[0]*(l[0] + l[1]*np.tan(g[0]) + l[2]*np.tan(g[1])) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*l[2]*(9*l[0]*nu - 3*l[0] + 2*l[1]*nu*np.tan(g[0]) + 2*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*(l[1]**2*nu - l[1]**2 - 4*l[2]**2*nu + 4*l[2]**2) + 3*l[0]*l[1]**2*l[2]*(nu - 1)*np.tan(g[1]) + 2*l[1]**2*l[2]**2*(2*nu*np.tan(g[0])**2 + nu*np.tan(g[1])**2 + nu - 2*np.tan(g[0])**2 - np.tan(g[1])**2 - 1))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(-18*l[0]**2*nu + 6*l[0]**2 - 3*l[0]*l[1]*nu*np.tan(g[0]) - 3*l[0]*l[1]*np.tan(g[0]) - 18*l[0]*l[2]*nu*np.tan(g[1]) + 6*l[0]*l[2]*np.tan(g[1]) - 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           -E*l[2]*(3*l[0] + 2*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu - 1)),
           E*(l[0]**2*l[1]**2*nu - l[0]**2*l[1]**2 - 4*l[0]**2*l[2]**2*nu + 4*l[0]**2*l[2]**2 - 12*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) + 12*l[0]*l[1]*l[2]**2*np.tan(g[0]) - 4*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - 2*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2*nu + 4*l[1]**2*l[2]**2*np.tan(g[0])**2 + 2*l[1]**2*l[2]**2*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(-9*l[0]**2*nu + 3*l[0]**2 - 9*l[0]*l[1]*nu*np.tan(g[0]) + 3*l[0]*l[1]*np.tan(g[0]) + 6*l[0]*l[2]*nu*np.tan(g[1]) + 6*l[0]*l[2]*np.tan(g[1]) + 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[2]*(-9*l[0]*nu + 3*l[0] - 4*l[1]*nu*np.tan(g[0]) - 4*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(-l[0]**2*l[1]**2*nu + l[0]**2*l[1]**2 + l[0]**2*l[2]**2*nu - l[0]**2*l[2]**2 - 3*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) + 3*l[0]*l[1]**2*l[2]*np.tan(g[1]) - 2*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - l[1]**2*l[2]**2*nu + 2*l[1]**2*l[2]**2*np.tan(g[0])**2 + l[1]**2*l[2]**2*np.tan(g[1])**2 + l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(9*l[0]**2*nu - 3*l[0]**2 + 6*l[0]*l[1]*nu*np.tan(g[0]) + 6*l[0]*l[1]*np.tan(g[0]) + 9*l[0]*l[2]*nu*np.tan(g[1]) - 3*l[0]*l[2]*np.tan(g[1]) + 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[2]*(3*l[0] + 4*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu - 1)),
           E*(2*l[0]**2*(-l[1]**2*nu + l[1]**2 + l[2]**2*nu - l[2]**2) + 3*l[0]*l[1]*l[2]**2*(nu - 1)*np.tan(g[0]) + l[1]**2*l[2]**2*(2*nu*np.tan(g[0])**2 + nu*np.tan(g[1])**2 + nu - 2*np.tan(g[0])**2 - np.tan(g[1])**2 - 1))/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(18*l[0]**2*nu - 6*l[0]**2 + 18*l[0]*l[1]*nu*np.tan(g[0]) - 6*l[0]*l[1]*np.tan(g[0]) - 3*l[0]*l[2]*nu*np.tan(g[1]) - 3*l[0]*l[2]*np.tan(g[1]) - 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[2]*(9*l[0]*nu - 3*l[0] + 2*l[1]*nu*np.tan(g[0]) + 2*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*(-l[1]**2*nu + l[1]**2 - 2*l[2]**2*nu + 2*l[2]**2) + l[1]**2*l[2]**2*(2*nu*np.tan(g[0])**2 + nu*np.tan(g[1])**2 + nu - 2*np.tan(g[0])**2 - np.tan(g[1])**2 - 1))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(6*l[0]**2*(2*nu**2 + nu - 1) + 3*l[0]*(2*nu - 1)*(3*l[1]*nu*np.tan(g[0]) - l[1]*np.tan(g[0]) - 3*l[2]*nu*np.tan(g[1]) + l[2]*np.tan(g[1])) - 2*l[1]*l[2]*(2*nu**2 + nu - 1)*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu - 1)*(2*nu**2 + nu - 1)),
           -E*l[2]*(3*l[0] + 2*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu - 1)),
           E*(3*l[0]**2*(-l[1]**2*nu + l[1]**2 - 2*l[2]**2*nu + 2*l[2]**2) + 2*l[0]**2*(l[1]**2*nu - l[1]**2 + 2*l[2]**2*nu - 2*l[2]**2) + l[1]*l[2]*(-3*l[0]*l[1]*nu*np.tan(g[1]) + 3*l[0]*l[1]*np.tan(g[1]) - 6*l[0]*l[2]*nu*np.tan(g[0]) + 6*l[0]*l[2]*np.tan(g[0]) - 2*l[1]*l[2]*nu*np.tan(g[0])**2 - l[1]*l[2]*nu*np.tan(g[1])**2 - l[1]*l[2]*nu + 2*l[1]*l[2]*np.tan(g[0])**2 + l[1]*l[2]*np.tan(g[1])**2 + l[1]*l[2]))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(3*l[0]**2 + 3*l[0]*l[1]*np.tan(g[0]) + 3*l[0]*l[2]*np.tan(g[1]) + 2*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu - 1))],
          [E*l[1]*(-9*l[0]*nu + 3*l[0] - 4*l[2]*nu*np.tan(g[1]) - 4*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(-3*l[0]**2*nu - 3*l[0]**2 + 18*l[0]*l[1]*nu*np.tan(g[0]) - 6*l[0]*l[1]*np.tan(g[0]) - 18*l[0]*l[2]*nu*np.tan(g[1]) + 6*l[0]*l[2]*np.tan(g[1]) + 8*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 8*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*l[1]**2*nu - 2*l[0]**2*l[1]**2 + l[0]**2*l[2]**2*nu - l[0]**2*l[2]**2 - 2*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - 4*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2*nu + 2*l[1]**2*l[2]**2*np.tan(g[0])**2 + 4*l[1]**2*l[2]**2*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(3*l[0] + 4*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*(-3*l[0]*(l[0] + l[1]*np.tan(g[0]) + l[2]*np.tan(g[1])) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*(2*l[0]**2*(2*l[1]**2*nu - 2*l[1]**2 + l[2]**2*nu - l[2]**2) + 3*l[0]*l[1]*l[2]*(2*l[1]*nu*np.tan(g[1]) - 2*l[1]*np.tan(g[1]) + l[2]*nu*np.tan(g[0]) - l[2]*np.tan(g[0])) + 2*l[1]**2*l[2]**2*(nu*np.tan(g[0])**2 + 2*nu*np.tan(g[1])**2 + nu - np.tan(g[0])**2 - 2*np.tan(g[1])**2 - 1))/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(3*l[0] + 4*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*(18*l[0]**2*nu - 6*l[0]**2 - 3*l[0]*l[1]*nu*np.tan(g[0]) - 3*l[0]*l[1]*np.tan(g[0]) + 18*l[0]*l[2]*nu*np.tan(g[1]) - 6*l[0]*l[2]*np.tan(g[1]) - 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*(l[1]**2*nu - l[1]**2 - l[2]**2*nu + l[2]**2) + 3*l[0]*l[1]**2*l[2]*(nu - 1)*np.tan(g[1]) + l[1]**2*l[2]**2*(nu*np.tan(g[0])**2 + 2*nu*np.tan(g[1])**2 + nu - np.tan(g[0])**2 - 2*np.tan(g[1])**2 - 1))/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(-9*l[0]*nu + 3*l[0] - 4*l[2]*nu*np.tan(g[1]) - 4*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(9*l[0]**2*nu - 3*l[0]**2 + 9*l[0]*l[1]*nu*np.tan(g[0]) - 3*l[0]*l[1]*np.tan(g[0]) + 6*l[0]*l[2]*nu*np.tan(g[1]) + 6*l[0]*l[2]*np.tan(g[1]) + 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(l[0]**2*l[1]**2*nu - l[0]**2*l[1]**2 - l[0]**2*l[2]**2*nu + l[0]**2*l[2]**2 - 3*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) + 3*l[0]*l[1]*l[2]**2*np.tan(g[0]) - l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - 2*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - l[1]**2*l[2]**2*nu + l[1]**2*l[2]**2*np.tan(g[0])**2 + 2*l[1]**2*l[2]**2*np.tan(g[1])**2 + l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           -E*l[1]*(3*l[0] + 2*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*(-9*l[0]**2*nu + 3*l[0]**2 + 6*l[0]*l[1]*nu*np.tan(g[0]) + 6*l[0]*l[1]*np.tan(g[0]) - 9*l[0]*l[2]*nu*np.tan(g[1]) + 3*l[0]*l[2]*np.tan(g[1]) + 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(-4*l[0]**2*l[1]**2*nu + 4*l[0]**2*l[1]**2 + l[0]**2*l[2]**2*nu - l[0]**2*l[2]**2 - 12*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) + 12*l[0]*l[1]**2*l[2]*np.tan(g[1]) - 2*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - 4*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2*nu + 2*l[1]**2*l[2]**2*np.tan(g[0])**2 + 4*l[1]**2*l[2]**2*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(9*l[0]*nu - 3*l[0] + 2*l[2]*nu*np.tan(g[1]) + 2*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(-18*l[0]**2*nu + 6*l[0]**2 - 18*l[0]*l[1]*nu*np.tan(g[0]) + 6*l[0]*l[1]*np.tan(g[0]) - 3*l[0]*l[2]*nu*np.tan(g[1]) - 3*l[0]*l[2]*np.tan(g[1]) - 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*(-4*l[1]**2*nu + 4*l[1]**2 + l[2]**2*nu - l[2]**2) + 3*l[0]*l[1]*l[2]**2*(nu - 1)*np.tan(g[0]) + 2*l[1]**2*l[2]**2*(nu*np.tan(g[0])**2 + 2*nu*np.tan(g[1])**2 + nu - np.tan(g[0])**2 - 2*np.tan(g[1])**2 - 1))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(9*l[0]*nu - 3*l[0] + 2*l[2]*nu*np.tan(g[1]) + 2*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(6*l[0]**2*(2*nu**2 + nu - 1) + 3*l[0]*(2*nu - 1)*(-3*l[1]*nu*np.tan(g[0]) + l[1]*np.tan(g[0]) + 3*l[2]*nu*np.tan(g[1]) - l[2]*np.tan(g[1])) - 2*l[1]*l[2]*(2*nu**2 + nu - 1)*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu - 1)*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*(-2*l[1]**2*nu + 2*l[1]**2 - l[2]**2*nu + l[2]**2) + l[1]**2*l[2]**2*(nu*np.tan(g[0])**2 + 2*nu*np.tan(g[1])**2 + nu - np.tan(g[0])**2 - 2*np.tan(g[1])**2 - 1))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           -E*l[1]*(3*l[0] + 2*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*(3*l[0]**2 + 3*l[0]*l[1]*np.tan(g[0]) + 3*l[0]*l[2]*np.tan(g[1]) + 2*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*(3*l[0]**2*(-2*l[1]**2*nu + 2*l[1]**2 - l[2]**2*nu + l[2]**2) + 2*l[0]**2*(2*l[1]**2*nu - 2*l[1]**2 + l[2]**2*nu - l[2]**2) + l[1]*l[2]*(-6*l[0]*l[1]*nu*np.tan(g[1]) + 6*l[0]*l[1]*np.tan(g[1]) - 3*l[0]*l[2]*nu*np.tan(g[0]) + 3*l[0]*l[2]*np.tan(g[0]) - l[1]*l[2]*nu*np.tan(g[0])**2 - 2*l[1]*l[2]*nu*np.tan(g[1])**2 - l[1]*l[2]*nu + l[1]*l[2]*np.tan(g[0])**2 + 2*l[1]*l[2]*np.tan(g[1])**2 + l[1]*l[2]))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1))],
          [E*(l[0]**2*l[1]**2*nu - l[0]**2*l[1]**2 - 2*l[0]**2*l[2]**2*nu + 2*l[0]**2*l[2]**2 + 6*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) - 6*l[0]*l[1]*l[2]**2*np.tan(g[0]) - 2*l[1]**2*l[2]**2*nu/np.cos(g[1])**2 - 2*l[1]**2*l[2]**2*nu/np.cos(g[0])**2 + 2*l[1]**2*l[2]**2/np.cos(g[1])**2 + 2*l[1]**2*l[2]**2/np.cos(g[0])**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(3*l[0] - 2*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu - 1)),
           E*l[1]*(9*l[0]*nu - 3*l[0] - 4*l[2]*nu*np.tan(g[1]) - 4*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*(l[1]**2*nu - l[1]**2 - 2*l[2]**2*nu + 2*l[2]**2) + 3*l[0]*l[1]**2*l[2]*(nu - 1)*np.tan(g[1]) + 2*l[1]**2*l[2]**2*(nu/np.cos(g[1])**2 + nu/np.cos(g[0])**2 - 1/np.cos(g[1])**2 - 1/np.cos(g[0])**2))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(9*l[0]*nu - 3*l[0] + 2*l[1]*nu*np.tan(g[0]) + 2*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*l[1]*(3*l[0] + 4*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*(2*l[0]**2*(l[1]**2*nu - l[1]**2 + l[2]**2*nu - l[2]**2) + 3*l[0]*l[1]*l[2]*(l[1]*nu*np.tan(g[1]) - l[1]*np.tan(g[1]) - l[2]*nu*np.tan(g[0]) + l[2]*np.tan(g[0])) + 2*l[1]**2*l[2]**2*(nu/np.cos(g[1])**2 + nu/np.cos(g[0])**2 - 1/np.cos(g[1])**2 - 1/np.cos(g[0])**2))/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(-3*l[0] + 4*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu - 1)),
           E*l[1]*(3*l[0] + 4*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*(l[0]**2*l[1]**2*nu - l[0]**2*l[1]**2 + l[0]**2*l[2]**2*nu - l[0]**2*l[2]**2 - 2*l[1]**2*l[2]**2*nu/np.cos(g[1])**2 - 2*l[1]**2*l[2]**2*nu/np.cos(g[0])**2 + 2*l[1]**2*l[2]**2/np.cos(g[1])**2 + 2*l[1]**2*l[2]**2/np.cos(g[0])**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(-9*l[0]*nu + 3*l[0] - 4*l[1]*nu*np.tan(g[0]) - 4*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*l[1]*(9*l[0]*nu - 3*l[0] - 4*l[2]*nu*np.tan(g[1]) - 4*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(3*l[0]**2*(-l[1]**2*nu + l[1]**2 - l[2]**2*nu + l[2]**2) + 2*l[0]**2*(l[1]**2*nu - l[1]**2 + l[2]**2*nu - l[2]**2) + l[1]*l[2]*(-3*l[0]*l[1]*nu*np.tan(g[1]) + 3*l[0]*l[1]*np.tan(g[1]) + 3*l[0]*l[2]*nu*np.tan(g[0]) - 3*l[0]*l[2]*np.tan(g[0]) - l[1]*l[2]*nu/np.cos(g[1])**2 - l[1]*l[2]*nu/np.cos(g[0])**2 + l[1]*l[2]/np.cos(g[1])**2 + l[1]*l[2]/np.cos(g[0])**2))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(3*l[0] - 2*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu - 1)),
           -E*l[1]*(3*l[0] + 2*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*(2*l[0]**2*(-l[1]**2*nu + l[1]**2 - l[2]**2*nu + l[2]**2) + l[1]**2*l[2]**2*(nu/np.cos(g[1])**2 + nu/np.cos(g[0])**2 - 1/np.cos(g[1])**2 - 1/np.cos(g[0])**2))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(9*l[0]*nu - 3*l[0] + 2*l[1]*nu*np.tan(g[0]) + 2*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[1]*(-9*l[0]*nu + 3*l[0] + 2*l[2]*nu*np.tan(g[1]) + 2*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*(-2*l[1]**2*nu + 2*l[1]**2 + l[2]**2*nu - l[2]**2) + 3*l[0]*l[1]*l[2]**2*(1 - nu)*np.tan(g[0]) + 2*l[1]**2*l[2]**2*(nu/np.cos(g[1])**2 + nu/np.cos(g[0])**2 - 1/np.cos(g[1])**2 - 1/np.cos(g[0])**2))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(-3*l[0] + 4*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu - 1)),
           E*l[1]*(-9*l[0]*nu + 3*l[0] + 2*l[2]*nu*np.tan(g[1]) + 2*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(-2*l[0]**2*l[1]**2*nu + 2*l[0]**2*l[1]**2 + l[0]**2*l[2]**2*nu - l[0]**2*l[2]**2 - 6*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) + 6*l[0]*l[1]**2*l[2]*np.tan(g[1]) - 2*l[1]**2*l[2]**2*nu/np.cos(g[1])**2 - 2*l[1]**2*l[2]**2*nu/np.cos(g[0])**2 + 2*l[1]**2*l[2]**2/np.cos(g[1])**2 + 2*l[1]**2*l[2]**2/np.cos(g[0])**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(-9*l[0]*nu + 3*l[0] - 4*l[1]*nu*np.tan(g[0]) - 4*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu**2 + nu - 1)),
           -E*l[1]*(3*l[0] + 2*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu - 1))],
          [E*l[2]*(3*l[0] - 2*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu - 1)),
           E*(l[0]**2*l[1]**2*nu - l[0]**2*l[1]**2 - 4*l[0]**2*l[2]**2*nu + 4*l[0]**2*l[2]**2 + 12*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) - 12*l[0]*l[1]*l[2]**2*np.tan(g[0]) - 4*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - 2*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2*nu + 4*l[1]**2*l[2]**2*np.tan(g[0])**2 + 2*l[1]**2*l[2]**2*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(9*l[0]**2*nu - 3*l[0]**2 - 9*l[0]*l[1]*nu*np.tan(g[0]) + 3*l[0]*l[1]*np.tan(g[0]) - 6*l[0]*l[2]*nu*np.tan(g[1]) - 6*l[0]*l[2]*np.tan(g[1]) + 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[2]*(-9*l[0]*nu + 3*l[0] + 2*l[1]*nu*np.tan(g[0]) + 2*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*(l[1]**2*nu - l[1]**2 - 4*l[2]**2*nu + 4*l[2]**2) + 3*l[0]*l[1]**2*l[2]*(nu - 1)*np.tan(g[1]) + 2*l[1]**2*l[2]**2*(2*nu*np.tan(g[0])**2 + nu*np.tan(g[1])**2 + nu - 2*np.tan(g[0])**2 - np.tan(g[1])**2 - 1))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(18*l[0]**2*nu - 6*l[0]**2 - 3*l[0]*l[1]*nu*np.tan(g[0]) - 3*l[0]*l[1]*np.tan(g[0]) + 18*l[0]*l[2]*nu*np.tan(g[1]) - 6*l[0]*l[2]*np.tan(g[1]) - 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[2]*(-3*l[0] + 4*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu - 1)),
           E*(2*l[0]**2*(l[1]**2*nu - l[1]**2 + 2*l[2]**2*nu - 2*l[2]**2) + 3*l[0]*l[1]*l[2]*(l[1]*nu*np.tan(g[1]) - l[1]*np.tan(g[1]) - 2*l[2]*nu*np.tan(g[0]) + 2*l[2]*np.tan(g[0])) + 2*l[1]**2*l[2]**2*(2*nu*np.tan(g[0])**2 + nu*np.tan(g[1])**2 + nu - 2*np.tan(g[0])**2 - np.tan(g[1])**2 - 1))/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(3*l[0]*(l[0] - l[1]*np.tan(g[0]) + l[2]*np.tan(g[1])) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*l[2]*(9*l[0]*nu - 3*l[0] - 4*l[1]*nu*np.tan(g[0]) - 4*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(l[0]**2*l[1]**2*nu - l[0]**2*l[1]**2 + 2*l[0]**2*l[2]**2*nu - 2*l[0]**2*l[2]**2 - 4*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - 2*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2*nu + 4*l[1]**2*l[2]**2*np.tan(g[0])**2 + 2*l[1]**2*l[2]**2*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(3*l[0]**2*nu + 3*l[0]**2 - 18*l[0]*l[1]*nu*np.tan(g[0]) + 6*l[0]*l[1]*np.tan(g[0]) - 18*l[0]*l[2]*nu*np.tan(g[1]) + 6*l[0]*l[2]*np.tan(g[1]) + 8*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 8*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[2]*(3*l[0] - 2*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu - 1)),
           E*(3*l[0]**2*(-l[1]**2*nu + l[1]**2 - 2*l[2]**2*nu + 2*l[2]**2) + 2*l[0]**2*(l[1]**2*nu - l[1]**2 + 2*l[2]**2*nu - 2*l[2]**2) + l[1]*l[2]*(-3*l[0]*l[1]*nu*np.tan(g[1]) + 3*l[0]*l[1]*np.tan(g[1]) + 6*l[0]*l[2]*nu*np.tan(g[0]) - 6*l[0]*l[2]*np.tan(g[0]) - 2*l[1]*l[2]*nu*np.tan(g[0])**2 - l[1]*l[2]*nu*np.tan(g[1])**2 - l[1]*l[2]*nu + 2*l[1]*l[2]*np.tan(g[0])**2 + l[1]*l[2]*np.tan(g[1])**2 + l[1]*l[2]))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(-3*l[0]**2 + 3*l[0]*l[1]*np.tan(g[0]) - 3*l[0]*l[2]*np.tan(g[1]) + 2*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*l[2]*(-9*l[0]*nu + 3*l[0] + 2*l[1]*nu*np.tan(g[0]) + 2*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*(-l[1]**2*nu + l[1]**2 - 2*l[2]**2*nu + 2*l[2]**2) + l[1]**2*l[2]**2*(2*nu*np.tan(g[0])**2 + nu*np.tan(g[1])**2 + nu - 2*np.tan(g[0])**2 - np.tan(g[1])**2 - 1))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(-6*l[0]**2*(2*nu**2 + nu - 1) + 3*l[0]*(2*nu - 1)*(3*l[1]*nu*np.tan(g[0]) - l[1]*np.tan(g[0]) + 3*l[2]*nu*np.tan(g[1]) - l[2]*np.tan(g[1])) - 2*l[1]*l[2]*(2*nu**2 + nu - 1)*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu - 1)*(2*nu**2 + nu - 1)),
           E*l[2]*(-3*l[0] + 4*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu - 1)),
           E*(2*l[0]**2*(-l[1]**2*nu + l[1]**2 + l[2]**2*nu - l[2]**2) + 3*l[0]*l[1]*l[2]**2*(1 - nu)*np.tan(g[0]) + l[1]**2*l[2]**2*(2*nu*np.tan(g[0])**2 + nu*np.tan(g[1])**2 + nu - 2*np.tan(g[0])**2 - np.tan(g[1])**2 - 1))/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(-18*l[0]**2*nu + 6*l[0]**2 + 18*l[0]*l[1]*nu*np.tan(g[0]) - 6*l[0]*l[1]*np.tan(g[0]) + 3*l[0]*l[2]*nu*np.tan(g[1]) + 3*l[0]*l[2]*np.tan(g[1]) - 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[2]*(9*l[0]*nu - 3*l[0] - 4*l[1]*nu*np.tan(g[0]) - 4*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(-l[0]**2*l[1]**2*nu + l[0]**2*l[1]**2 + l[0]**2*l[2]**2*nu - l[0]**2*l[2]**2 - 3*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) + 3*l[0]*l[1]**2*l[2]*np.tan(g[1]) - 2*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - l[1]**2*l[2]**2*nu + 2*l[1]**2*l[2]**2*np.tan(g[0])**2 + l[1]**2*l[2]**2*np.tan(g[1])**2 + l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(-9*l[0]**2*nu + 3*l[0]**2 + 6*l[0]*l[1]*nu*np.tan(g[0]) + 6*l[0]*l[1]*np.tan(g[0]) - 9*l[0]*l[2]*nu*np.tan(g[1]) + 3*l[0]*l[2]*np.tan(g[1]) + 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1))],
          [E*l[1]*(-9*l[0]*nu + 3*l[0] - 4*l[2]*nu*np.tan(g[1]) - 4*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(-9*l[0]**2*nu + 3*l[0]**2 + 9*l[0]*l[1]*nu*np.tan(g[0]) - 3*l[0]*l[1]*np.tan(g[0]) - 6*l[0]*l[2]*nu*np.tan(g[1]) - 6*l[0]*l[2]*np.tan(g[1]) + 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(l[0]**2*l[1]**2*nu - l[0]**2*l[1]**2 - l[0]**2*l[2]**2*nu + l[0]**2*l[2]**2 + 3*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) - 3*l[0]*l[1]*l[2]**2*np.tan(g[0]) - l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - 2*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - l[1]**2*l[2]**2*nu + l[1]**2*l[2]**2*np.tan(g[0])**2 + 2*l[1]**2*l[2]**2*np.tan(g[1])**2 + l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(3*l[0] + 4*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*(-18*l[0]**2*nu + 6*l[0]**2 - 3*l[0]*l[1]*nu*np.tan(g[0]) - 3*l[0]*l[1]*np.tan(g[0]) - 18*l[0]*l[2]*nu*np.tan(g[1]) + 6*l[0]*l[2]*np.tan(g[1]) - 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*(l[1]**2*nu - l[1]**2 - l[2]**2*nu + l[2]**2) + 3*l[0]*l[1]**2*l[2]*(nu - 1)*np.tan(g[1]) + l[1]**2*l[2]**2*(nu*np.tan(g[0])**2 + 2*nu*np.tan(g[1])**2 + nu - np.tan(g[0])**2 - 2*np.tan(g[1])**2 - 1))/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(3*l[0] + 4*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*(3*l[0]*(l[0] - l[1]*np.tan(g[0]) + l[2]*np.tan(g[1])) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*(2*l[0]**2*(2*l[1]**2*nu - 2*l[1]**2 + l[2]**2*nu - l[2]**2) + 3*l[0]*l[1]*l[2]*(2*l[1]*nu*np.tan(g[1]) - 2*l[1]*np.tan(g[1]) - l[2]*nu*np.tan(g[0]) + l[2]*np.tan(g[0])) + 2*l[1]**2*l[2]**2*(nu*np.tan(g[0])**2 + 2*nu*np.tan(g[1])**2 + nu - np.tan(g[0])**2 - 2*np.tan(g[1])**2 - 1))/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(-9*l[0]*nu + 3*l[0] - 4*l[2]*nu*np.tan(g[1]) - 4*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(3*l[0]**2*nu + 3*l[0]**2 + 18*l[0]*l[1]*nu*np.tan(g[0]) - 6*l[0]*l[1]*np.tan(g[0]) + 18*l[0]*l[2]*nu*np.tan(g[1]) - 6*l[0]*l[2]*np.tan(g[1]) + 8*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 8*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*l[1]**2*nu - 2*l[0]**2*l[1]**2 + l[0]**2*l[2]**2*nu - l[0]**2*l[2]**2 - 2*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - 4*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2*nu + 2*l[1]**2*l[2]**2*np.tan(g[0])**2 + 4*l[1]**2*l[2]**2*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           -E*l[1]*(3*l[0] + 2*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*(-3*l[0]**2 + 3*l[0]*l[1]*np.tan(g[0]) - 3*l[0]*l[2]*np.tan(g[1]) + 2*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*(3*l[0]**2*(-2*l[1]**2*nu + 2*l[1]**2 - l[2]**2*nu + l[2]**2) + 2*l[0]**2*(2*l[1]**2*nu - 2*l[1]**2 + l[2]**2*nu - l[2]**2) + l[1]*l[2]*(-6*l[0]*l[1]*nu*np.tan(g[1]) + 6*l[0]*l[1]*np.tan(g[1]) + 3*l[0]*l[2]*nu*np.tan(g[0]) - 3*l[0]*l[2]*np.tan(g[0]) - l[1]*l[2]*nu*np.tan(g[0])**2 - 2*l[1]*l[2]*nu*np.tan(g[1])**2 - l[1]*l[2]*nu + l[1]*l[2]*np.tan(g[0])**2 + 2*l[1]*l[2]*np.tan(g[1])**2 + l[1]*l[2]))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(9*l[0]*nu - 3*l[0] + 2*l[2]*nu*np.tan(g[1]) + 2*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(-6*l[0]**2*(2*nu**2 + nu - 1) + 3*l[0]*(2*nu - 1)*(-3*l[1]*nu*np.tan(g[0]) + l[1]*np.tan(g[0]) - 3*l[2]*nu*np.tan(g[1]) + l[2]*np.tan(g[1])) - 2*l[1]*l[2]*(2*nu**2 + nu - 1)*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu - 1)*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*(-2*l[1]**2*nu + 2*l[1]**2 - l[2]**2*nu + l[2]**2) + l[1]**2*l[2]**2*(nu*np.tan(g[0])**2 + 2*nu*np.tan(g[1])**2 + nu - np.tan(g[0])**2 - 2*np.tan(g[1])**2 - 1))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(9*l[0]*nu - 3*l[0] + 2*l[2]*nu*np.tan(g[1]) + 2*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(18*l[0]**2*nu - 6*l[0]**2 - 18*l[0]*l[1]*nu*np.tan(g[0]) + 6*l[0]*l[1]*np.tan(g[0]) + 3*l[0]*l[2]*nu*np.tan(g[1]) + 3*l[0]*l[2]*np.tan(g[1]) - 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*(-4*l[1]**2*nu + 4*l[1]**2 + l[2]**2*nu - l[2]**2) + 3*l[0]*l[1]*l[2]**2*(1 - nu)*np.tan(g[0]) + 2*l[1]**2*l[2]**2*(nu*np.tan(g[0])**2 + 2*nu*np.tan(g[1])**2 + nu - np.tan(g[0])**2 - 2*np.tan(g[1])**2 - 1))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           -E*l[1]*(3*l[0] + 2*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*(9*l[0]**2*nu - 3*l[0]**2 + 6*l[0]*l[1]*nu*np.tan(g[0]) + 6*l[0]*l[1]*np.tan(g[0]) + 9*l[0]*l[2]*nu*np.tan(g[1]) - 3*l[0]*l[2]*np.tan(g[1]) + 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(-4*l[0]**2*l[1]**2*nu + 4*l[0]**2*l[1]**2 + l[0]**2*l[2]**2*nu - l[0]**2*l[2]**2 - 12*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) + 12*l[0]*l[1]**2*l[2]*np.tan(g[1]) - 2*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - 4*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2*nu + 2*l[1]**2*l[2]**2*np.tan(g[0])**2 + 4*l[1]**2*l[2]**2*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1))],
          [E*(2*l[0]**2*l[1]**2*nu - 2*l[0]**2*l[1]**2 - 4*l[0]**2*l[2]**2*nu + 4*l[0]**2*l[2]**2 - 3*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) + 3*l[0]*l[1]**2*l[2]*np.tan(g[1]) + 2*l[1]**2*l[2]**2*nu/np.cos(g[1])**2 + 2*l[1]**2*l[2]**2*nu/np.cos(g[0])**2 - 2*l[1]**2*l[2]**2/np.cos(g[1])**2 - 2*l[1]**2*l[2]**2/np.cos(g[0])**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(-9*l[0]*nu + 3*l[0] + 2*l[1]*nu*np.tan(g[0]) + 2*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*l[1]*(-3*l[0] + 4*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*(l[0]**2*l[1]**2*nu - l[0]**2*l[1]**2 - 2*l[0]**2*l[2]**2*nu + 2*l[0]**2*l[2]**2 - 6*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) + 6*l[0]*l[1]*l[2]**2*np.tan(g[0]) - 2*l[1]**2*l[2]**2*nu/np.cos(g[1])**2 - 2*l[1]**2*l[2]**2*nu/np.cos(g[0])**2 + 2*l[1]**2*l[2]**2/np.cos(g[1])**2 + 2*l[1]**2*l[2]**2/np.cos(g[0])**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           -E*l[2]*(3*l[0] + 2*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu - 1)),
           E*l[1]*(-9*l[0]*nu + 3*l[0] - 4*l[2]*nu*np.tan(g[1]) - 4*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(l[0]**2*l[1]**2*nu - l[0]**2*l[1]**2 + l[0]**2*l[2]**2*nu - l[0]**2*l[2]**2 - 2*l[1]**2*l[2]**2*nu/np.cos(g[1])**2 - 2*l[1]**2*l[2]**2*nu/np.cos(g[0])**2 + 2*l[1]**2*l[2]**2/np.cos(g[1])**2 + 2*l[1]**2*l[2]**2/np.cos(g[0])**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(9*l[0]*nu - 3*l[0] - 4*l[1]*nu*np.tan(g[0]) - 4*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*l[1]*(-9*l[0]*nu + 3*l[0] - 4*l[2]*nu*np.tan(g[1]) - 4*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*l[1]**2*nu - 2*l[0]**2*l[1]**2 + 2*l[0]**2*l[2]**2*nu - 2*l[0]**2*l[2]**2 - 3*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) + 3*l[0]*l[1]**2*l[2]*np.tan(g[1]) + 3*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) - 3*l[0]*l[1]*l[2]**2*np.tan(g[0]) + 2*l[1]**2*l[2]**2*nu/np.cos(g[1])**2 + 2*l[1]**2*l[2]**2*nu/np.cos(g[0])**2 - 2*l[1]**2*l[2]**2/np.cos(g[1])**2 - 2*l[1]**2*l[2]**2/np.cos(g[0])**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(3*l[0] + 4*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu - 1)),
           E*l[1]*(-3*l[0] + 4*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*(-2*l[0]**2*l[1]**2*nu + 2*l[0]**2*l[1]**2 - 2*l[0]**2*l[2]**2*nu + 2*l[0]**2*l[2]**2 + l[1]**2*l[2]**2*nu/np.cos(g[1])**2 + l[1]**2*l[2]**2*nu/np.cos(g[0])**2 - l[1]**2*l[2]**2/np.cos(g[1])**2 - l[1]**2*l[2]**2/np.cos(g[0])**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(-9*l[0]*nu + 3*l[0] + 2*l[1]*nu*np.tan(g[0]) + 2*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[1]*(9*l[0]*nu - 3*l[0] + 2*l[2]*nu*np.tan(g[1]) + 2*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(3*l[0]**2*(-l[1]**2*nu + l[1]**2 - l[2]**2*nu + l[2]**2) + 2*l[0]**2*(l[1]**2*nu - l[1]**2 + l[2]**2*nu - l[2]**2) + l[1]*l[2]*(3*l[0]*l[1]*nu*np.tan(g[1]) - 3*l[0]*l[1]*np.tan(g[1]) - 3*l[0]*l[2]*nu*np.tan(g[0]) + 3*l[0]*l[2]*np.tan(g[0]) - l[1]*l[2]*nu/np.cos(g[1])**2 - l[1]*l[2]*nu/np.cos(g[0])**2 + l[1]*l[2]/np.cos(g[1])**2 + l[1]*l[2]/np.cos(g[0])**2))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           -E*l[2]*(3*l[0] + 2*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu - 1)),
           E*l[1]*(3*l[0] - 2*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*(-2*l[0]**2*l[1]**2*nu + 2*l[0]**2*l[1]**2 + l[0]**2*l[2]**2*nu - l[0]**2*l[2]**2 + 6*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) - 6*l[0]*l[1]**2*l[2]*np.tan(g[1]) - 2*l[1]**2*l[2]**2*nu/np.cos(g[1])**2 - 2*l[1]**2*l[2]**2*nu/np.cos(g[0])**2 + 2*l[1]**2*l[2]**2/np.cos(g[1])**2 + 2*l[1]**2*l[2]**2/np.cos(g[0])**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(9*l[0]*nu - 3*l[0] - 4*l[1]*nu*np.tan(g[0]) - 4*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[1]*(3*l[0] - 2*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*(-4*l[0]**2*l[1]**2*nu + 4*l[0]**2*l[1]**2 + 2*l[0]**2*l[2]**2*nu - 2*l[0]**2*l[2]**2 + 3*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) - 3*l[0]*l[1]*l[2]**2*np.tan(g[0]) + 2*l[1]**2*l[2]**2*nu/np.cos(g[1])**2 + 2*l[1]**2*l[2]**2*nu/np.cos(g[0])**2 - 2*l[1]**2*l[2]**2/np.cos(g[1])**2 - 2*l[1]**2*l[2]**2/np.cos(g[0])**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(3*l[0] + 4*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu - 1)),
           E*l[1]*(9*l[0]*nu - 3*l[0] + 2*l[2]*nu*np.tan(g[1]) + 2*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu**2 + nu - 1))],
          [E*l[2]*(9*l[0]*nu - 3*l[0] + 2*l[1]*nu*np.tan(g[0]) + 2*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*l[1]**2*nu - 2*l[0]**2*l[1]**2 - 8*l[0]**2*l[2]**2*nu + 8*l[0]**2*l[2]**2 - 3*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) + 3*l[0]*l[1]**2*l[2]*np.tan(g[1]) + 4*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 + 2*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2*nu - 4*l[1]**2*l[2]**2*np.tan(g[0])**2 - 2*l[1]**2*l[2]**2*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(18*l[0]**2*nu - 6*l[0]**2 + 3*l[0]*l[1]*nu*np.tan(g[0]) + 3*l[0]*l[1]*np.tan(g[0]) - 18*l[0]*l[2]*nu*np.tan(g[1]) + 6*l[0]*l[2]*np.tan(g[1]) - 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           -E*l[2]*(3*l[0] + 2*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu - 1)),
           E*(l[0]**2*l[1]**2*nu - l[0]**2*l[1]**2 - 4*l[0]**2*l[2]**2*nu + 4*l[0]**2*l[2]**2 - 12*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) + 12*l[0]*l[1]*l[2]**2*np.tan(g[0]) - 4*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - 2*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2*nu + 4*l[1]**2*l[2]**2*np.tan(g[0])**2 + 2*l[1]**2*l[2]**2*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(9*l[0]**2*nu - 3*l[0]**2 + 9*l[0]*l[1]*nu*np.tan(g[0]) - 3*l[0]*l[1]*np.tan(g[0]) + 6*l[0]*l[2]*nu*np.tan(g[1]) + 6*l[0]*l[2]*np.tan(g[1]) + 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[2]*(-9*l[0]*nu + 3*l[0] - 4*l[1]*nu*np.tan(g[0]) - 4*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(l[0]**2*l[1]**2*nu - l[0]**2*l[1]**2 + 2*l[0]**2*l[2]**2*nu - 2*l[0]**2*l[2]**2 - 4*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - 2*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2*nu + 4*l[1]**2*l[2]**2*np.tan(g[0])**2 + 2*l[1]**2*l[2]**2*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(3*l[0]**2*nu + 3*l[0]**2 + 18*l[0]*l[1]*nu*np.tan(g[0]) - 6*l[0]*l[1]*np.tan(g[0]) + 18*l[0]*l[2]*nu*np.tan(g[1]) - 6*l[0]*l[2]*np.tan(g[1]) + 8*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 8*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[2]*(3*l[0] + 4*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu - 1)),
           E*(2*l[0]**2*l[1]**2*nu - 2*l[0]**2*l[1]**2 + 4*l[0]**2*l[2]**2*nu - 4*l[0]**2*l[2]**2 - 3*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) + 3*l[0]*l[1]**2*l[2]*np.tan(g[1]) + 6*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) - 6*l[0]*l[1]*l[2]**2*np.tan(g[0]) + 4*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 + 2*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2*nu - 4*l[1]**2*l[2]**2*np.tan(g[0])**2 - 2*l[1]**2*l[2]**2*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(3*l[0]**2 + 3*l[0]*l[1]*np.tan(g[0]) - 3*l[0]*l[2]*np.tan(g[1]) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*l[2]*(9*l[0]*nu - 3*l[0] + 2*l[1]*nu*np.tan(g[0]) + 2*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(-2*l[0]**2*l[1]**2*nu + 2*l[0]**2*l[1]**2 - 4*l[0]**2*l[2]**2*nu + 4*l[0]**2*l[2]**2 + 2*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 + l[1]**2*l[2]**2*nu*np.tan(g[1])**2 + l[1]**2*l[2]**2*nu - 2*l[1]**2*l[2]**2*np.tan(g[0])**2 - l[1]**2*l[2]**2*np.tan(g[1])**2 - l[1]**2*l[2]**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(-6*l[0]**2*nu - 6*l[0]**2 - 9*l[0]*l[1]*nu*np.tan(g[0]) + 3*l[0]*l[1]*np.tan(g[0]) - 9*l[0]*l[2]*nu*np.tan(g[1]) + 3*l[0]*l[2]*np.tan(g[1]) - 2*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) - 2*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           -E*l[2]*(3*l[0] + 2*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu - 1)),
           E*(3*l[0]**2*(-l[1]**2*nu + l[1]**2 - 2*l[2]**2*nu + 2*l[2]**2) + 2*l[0]**2*(l[1]**2*nu - l[1]**2 + 2*l[2]**2*nu - 2*l[2]**2) + l[1]*l[2]*(3*l[0]*l[1]*nu*np.tan(g[1]) - 3*l[0]*l[1]*np.tan(g[1]) - 6*l[0]*l[2]*nu*np.tan(g[0]) + 6*l[0]*l[2]*np.tan(g[0]) - 2*l[1]*l[2]*nu*np.tan(g[0])**2 - l[1]*l[2]*nu*np.tan(g[1])**2 - l[1]*l[2]*nu + 2*l[1]*l[2]*np.tan(g[0])**2 + l[1]*l[2]*np.tan(g[1])**2 + l[1]*l[2]))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(-3*l[0]**2 - 3*l[0]*l[1]*np.tan(g[0]) + 3*l[0]*l[2]*np.tan(g[1]) + 2*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*l[2]*(-9*l[0]*nu + 3*l[0] - 4*l[1]*nu*np.tan(g[0]) - 4*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(-l[0]**2*l[1]**2*nu + l[0]**2*l[1]**2 + l[0]**2*l[2]**2*nu - l[0]**2*l[2]**2 + 3*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) - 3*l[0]*l[1]**2*l[2]*np.tan(g[1]) - 2*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - l[1]**2*l[2]**2*nu + 2*l[1]**2*l[2]**2*np.tan(g[0])**2 + l[1]**2*l[2]**2*np.tan(g[1])**2 + l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(-9*l[0]**2*nu + 3*l[0]**2 - 6*l[0]*l[1]*nu*np.tan(g[0]) - 6*l[0]*l[1]*np.tan(g[0]) + 9*l[0]*l[2]*nu*np.tan(g[1]) - 3*l[0]*l[2]*np.tan(g[1]) + 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[2]*(3*l[0] + 4*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu - 1)),
           E*(-2*l[0]**2*l[1]**2*nu + 2*l[0]**2*l[1]**2 + 2*l[0]**2*l[2]**2*nu - 2*l[0]**2*l[2]**2 + 3*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) - 3*l[0]*l[1]*l[2]**2*np.tan(g[0]) + 2*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 + l[1]**2*l[2]**2*nu*np.tan(g[1])**2 + l[1]**2*l[2]**2*nu - 2*l[1]**2*l[2]**2*np.tan(g[0])**2 - l[1]**2*l[2]**2*np.tan(g[1])**2 - l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(-18*l[0]**2*nu + 6*l[0]**2 - 18*l[0]*l[1]*nu*np.tan(g[0]) + 6*l[0]*l[1]*np.tan(g[0]) - 3*l[0]*l[2]*nu*np.tan(g[1]) - 3*l[0]*l[2]*np.tan(g[1]) - 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1))],
          [E*l[1]*(-3*l[0] + 4*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*(-18*l[0]**2*nu + 6*l[0]**2 + 3*l[0]*l[1]*nu*np.tan(g[0]) + 3*l[0]*l[1]*np.tan(g[0]) + 18*l[0]*l[2]*nu*np.tan(g[1]) - 6*l[0]*l[2]*np.tan(g[1]) - 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*l[1]**2*nu - 2*l[0]**2*l[1]**2 - 2*l[0]**2*l[2]**2*nu + 2*l[0]**2*l[2]**2 - 3*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) + 3*l[0]*l[1]**2*l[2]*np.tan(g[1]) + l[1]**2*l[2]**2*nu*np.tan(g[0])**2 + 2*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 + l[1]**2*l[2]**2*nu - l[1]**2*l[2]**2*np.tan(g[0])**2 - 2*l[1]**2*l[2]**2*np.tan(g[1])**2 - l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(9*l[0]*nu - 3*l[0] - 4*l[2]*nu*np.tan(g[1]) - 4*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(-9*l[0]**2*nu + 3*l[0]**2 - 9*l[0]*l[1]*nu*np.tan(g[0]) + 3*l[0]*l[1]*np.tan(g[0]) + 6*l[0]*l[2]*nu*np.tan(g[1]) + 6*l[0]*l[2]*np.tan(g[1]) + 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(l[0]**2*l[1]**2*nu - l[0]**2*l[1]**2 - l[0]**2*l[2]**2*nu + l[0]**2*l[2]**2 - 3*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) + 3*l[0]*l[1]*l[2]**2*np.tan(g[0]) - l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - 2*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - l[1]**2*l[2]**2*nu + l[1]**2*l[2]**2*np.tan(g[0])**2 + 2*l[1]**2*l[2]**2*np.tan(g[1])**2 + l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(9*l[0]*nu - 3*l[0] - 4*l[2]*nu*np.tan(g[1]) - 4*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(3*l[0]**2*nu + 3*l[0]**2 - 18*l[0]*l[1]*nu*np.tan(g[0]) + 6*l[0]*l[1]*np.tan(g[0]) - 18*l[0]*l[2]*nu*np.tan(g[1]) + 6*l[0]*l[2]*np.tan(g[1]) + 8*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 8*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*l[1]**2*nu - 2*l[0]**2*l[1]**2 + l[0]**2*l[2]**2*nu - l[0]**2*l[2]**2 - 2*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - 4*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2*nu + 2*l[1]**2*l[2]**2*np.tan(g[0])**2 + 4*l[1]**2*l[2]**2*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(-3*l[0] + 4*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*(3*l[0]**2 + 3*l[0]*l[1]*np.tan(g[0]) - 3*l[0]*l[2]*np.tan(g[1]) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*(4*l[0]**2*l[1]**2*nu - 4*l[0]**2*l[1]**2 + 2*l[0]**2*l[2]**2*nu - 2*l[0]**2*l[2]**2 - 6*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) + 6*l[0]*l[1]**2*l[2]*np.tan(g[1]) + 3*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) - 3*l[0]*l[1]*l[2]**2*np.tan(g[0]) + 2*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 + 4*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2*nu - 2*l[1]**2*l[2]**2*np.tan(g[0])**2 - 4*l[1]**2*l[2]**2*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(-9*l[0]*nu + 3*l[0] + 2*l[2]*nu*np.tan(g[1]) + 2*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(-6*l[0]**2*nu - 6*l[0]**2 + 9*l[0]*l[1]*nu*np.tan(g[0]) - 3*l[0]*l[1]*np.tan(g[0]) + 9*l[0]*l[2]*nu*np.tan(g[1]) - 3*l[0]*l[2]*np.tan(g[1]) - 2*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) - 2*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(-4*l[0]**2*l[1]**2*nu + 4*l[0]**2*l[1]**2 - 2*l[0]**2*l[2]**2*nu + 2*l[0]**2*l[2]**2 + l[1]**2*l[2]**2*nu*np.tan(g[0])**2 + 2*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 + l[1]**2*l[2]**2*nu - l[1]**2*l[2]**2*np.tan(g[0])**2 - 2*l[1]**2*l[2]**2*np.tan(g[1])**2 - l[1]**2*l[2]**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(3*l[0] - 2*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*(-3*l[0]**2 - 3*l[0]*l[1]*np.tan(g[0]) + 3*l[0]*l[2]*np.tan(g[1]) + 2*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*(3*l[0]**2*(-2*l[1]**2*nu + 2*l[1]**2 - l[2]**2*nu + l[2]**2) + 2*l[0]**2*(2*l[1]**2*nu - 2*l[1]**2 + l[2]**2*nu - l[2]**2) + l[1]*l[2]*(6*l[0]*l[1]*nu*np.tan(g[1]) - 6*l[0]*l[1]*np.tan(g[1]) - 3*l[0]*l[2]*nu*np.tan(g[0]) + 3*l[0]*l[2]*np.tan(g[0]) - l[1]*l[2]*nu*np.tan(g[0])**2 - 2*l[1]*l[2]*nu*np.tan(g[1])**2 - l[1]*l[2]*nu + l[1]*l[2]*np.tan(g[0])**2 + 2*l[1]*l[2]*np.tan(g[1])**2 + l[1]*l[2]))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(3*l[0] - 2*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*(9*l[0]**2*nu - 3*l[0]**2 - 6*l[0]*l[1]*nu*np.tan(g[0]) - 6*l[0]*l[1]*np.tan(g[0]) - 9*l[0]*l[2]*nu*np.tan(g[1]) + 3*l[0]*l[2]*np.tan(g[1]) + 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(-4*l[0]**2*l[1]**2*nu + 4*l[0]**2*l[1]**2 + l[0]**2*l[2]**2*nu - l[0]**2*l[2]**2 + 12*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) - 12*l[0]*l[1]**2*l[2]*np.tan(g[1]) - 2*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - 4*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2*nu + 2*l[1]**2*l[2]**2*np.tan(g[0])**2 + 4*l[1]**2*l[2]**2*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(-9*l[0]*nu + 3*l[0] + 2*l[2]*nu*np.tan(g[1]) + 2*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(18*l[0]**2*nu - 6*l[0]**2 + 18*l[0]*l[1]*nu*np.tan(g[0]) - 6*l[0]*l[1]*np.tan(g[0]) - 3*l[0]*l[2]*nu*np.tan(g[1]) - 3*l[0]*l[2]*np.tan(g[1]) - 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(-8*l[0]**2*l[1]**2*nu + 8*l[0]**2*l[1]**2 + 2*l[0]**2*l[2]**2*nu - 2*l[0]**2*l[2]**2 + 3*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) - 3*l[0]*l[1]*l[2]**2*np.tan(g[0]) + 2*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 + 4*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2*nu - 2*l[1]**2*l[2]**2*np.tan(g[0])**2 - 4*l[1]**2*l[2]**2*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1))],
          [E*(-4*l[0]**2*l[1]**2*nu + 4*l[0]**2*l[1]**2 + 2*l[0]**2*l[2]**2*nu - 2*l[0]**2*l[2]**2 - 3*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) + 3*l[0]*l[1]*l[2]**2*np.tan(g[0]) + 2*l[1]**2*l[2]**2*nu/np.cos(g[1])**2 + 2*l[1]**2*l[2]**2*nu/np.cos(g[0])**2 - 2*l[1]**2*l[2]**2/np.cos(g[1])**2 - 2*l[1]**2*l[2]**2/np.cos(g[0])**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(-3*l[0] + 4*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu - 1)),
           E*l[1]*(-9*l[0]*nu + 3*l[0] + 2*l[2]*nu*np.tan(g[1]) + 2*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(-2*l[0]**2*l[1]**2*nu + 2*l[0]**2*l[1]**2 + l[0]**2*l[2]**2*nu - l[0]**2*l[2]**2 - 6*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) + 6*l[0]*l[1]**2*l[2]*np.tan(g[1]) - 2*l[1]**2*l[2]**2*nu/np.cos(g[1])**2 - 2*l[1]**2*l[2]**2*nu/np.cos(g[0])**2 + 2*l[1]**2*l[2]**2/np.cos(g[1])**2 + 2*l[1]**2*l[2]**2/np.cos(g[0])**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(-9*l[0]*nu + 3*l[0] - 4*l[1]*nu*np.tan(g[0]) - 4*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu**2 + nu - 1)),
           -E*l[1]*(3*l[0] + 2*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*(3*l[0]**2*(-l[1]**2*nu + l[1]**2 - l[2]**2*nu + l[2]**2) + 2*l[0]**2*(l[1]**2*nu - l[1]**2 + l[2]**2*nu - l[2]**2) + l[1]*l[2]*(-3*l[0]*l[1]*nu*np.tan(g[1]) + 3*l[0]*l[1]*np.tan(g[1]) + 3*l[0]*l[2]*nu*np.tan(g[0]) - 3*l[0]*l[2]*np.tan(g[0]) - l[1]*l[2]*nu/np.cos(g[1])**2 - l[1]*l[2]*nu/np.cos(g[0])**2 + l[1]*l[2]/np.cos(g[1])**2 + l[1]*l[2]/np.cos(g[0])**2))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(3*l[0] - 2*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu - 1)),
           -E*l[1]*(3*l[0] + 2*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*(-2*l[0]**2*l[1]**2*nu + 2*l[0]**2*l[1]**2 - 2*l[0]**2*l[2]**2*nu + 2*l[0]**2*l[2]**2 + l[1]**2*l[2]**2*nu/np.cos(g[1])**2 + l[1]**2*l[2]**2*nu/np.cos(g[0])**2 - l[1]**2*l[2]**2/np.cos(g[1])**2 - l[1]**2*l[2]**2/np.cos(g[0])**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(9*l[0]*nu - 3*l[0] + 2*l[1]*nu*np.tan(g[0]) + 2*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[1]*(-9*l[0]*nu + 3*l[0] + 2*l[2]*nu*np.tan(g[1]) + 2*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*l[1]**2*nu - 2*l[0]**2*l[1]**2 + 2*l[0]**2*l[2]**2*nu - 2*l[0]**2*l[2]**2 + 3*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) - 3*l[0]*l[1]**2*l[2]*np.tan(g[1]) - 3*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) + 3*l[0]*l[1]*l[2]**2*np.tan(g[0]) + 2*l[1]**2*l[2]**2*nu/np.cos(g[1])**2 + 2*l[1]**2*l[2]**2*nu/np.cos(g[0])**2 - 2*l[1]**2*l[2]**2/np.cos(g[1])**2 - 2*l[1]**2*l[2]**2/np.cos(g[0])**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(-3*l[0] + 4*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu - 1)),
           E*l[1]*(3*l[0] + 4*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*(l[0]**2*l[1]**2*nu - l[0]**2*l[1]**2 + l[0]**2*l[2]**2*nu - l[0]**2*l[2]**2 - 2*l[1]**2*l[2]**2*nu/np.cos(g[1])**2 - 2*l[1]**2*l[2]**2*nu/np.cos(g[0])**2 + 2*l[1]**2*l[2]**2/np.cos(g[1])**2 + 2*l[1]**2*l[2]**2/np.cos(g[0])**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(-9*l[0]*nu + 3*l[0] - 4*l[1]*nu*np.tan(g[0]) - 4*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*l[1]*(9*l[0]*nu - 3*l[0] - 4*l[2]*nu*np.tan(g[1]) - 4*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(l[0]**2*l[1]**2*nu - l[0]**2*l[1]**2 - 2*l[0]**2*l[2]**2*nu + 2*l[0]**2*l[2]**2 + 6*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) - 6*l[0]*l[1]*l[2]**2*np.tan(g[0]) - 2*l[1]**2*l[2]**2*nu/np.cos(g[1])**2 - 2*l[1]**2*l[2]**2*nu/np.cos(g[0])**2 + 2*l[1]**2*l[2]**2/np.cos(g[1])**2 + 2*l[1]**2*l[2]**2/np.cos(g[0])**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(3*l[0] - 2*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu - 1)),
           E*l[1]*(9*l[0]*nu - 3*l[0] - 4*l[2]*nu*np.tan(g[1]) - 4*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*l[1]**2*nu - 2*l[0]**2*l[1]**2 - 4*l[0]**2*l[2]**2*nu + 4*l[0]**2*l[2]**2 + 3*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) - 3*l[0]*l[1]**2*l[2]*np.tan(g[1]) + 2*l[1]**2*l[2]**2*nu/np.cos(g[1])**2 + 2*l[1]**2*l[2]**2*nu/np.cos(g[0])**2 - 2*l[1]**2*l[2]**2/np.cos(g[1])**2 - 2*l[1]**2*l[2]**2/np.cos(g[0])**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(9*l[0]*nu - 3*l[0] + 2*l[1]*nu*np.tan(g[0]) + 2*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*l[1]*(3*l[0] + 4*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu - 1))],
          [E*l[2]*(-3*l[0] + 4*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu - 1)),
           E*(-2*l[0]**2*l[1]**2*nu + 2*l[0]**2*l[1]**2 + 2*l[0]**2*l[2]**2*nu - 2*l[0]**2*l[2]**2 - 3*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) + 3*l[0]*l[1]*l[2]**2*np.tan(g[0]) + 2*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 + l[1]**2*l[2]**2*nu*np.tan(g[1])**2 + l[1]**2*l[2]**2*nu - 2*l[1]**2*l[2]**2*np.tan(g[0])**2 - l[1]**2*l[2]**2*np.tan(g[1])**2 - l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(-18*l[0]**2*nu + 6*l[0]**2 + 18*l[0]*l[1]*nu*np.tan(g[0]) - 6*l[0]*l[1]*np.tan(g[0]) + 3*l[0]*l[2]*nu*np.tan(g[1]) + 3*l[0]*l[2]*np.tan(g[1]) - 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[2]*(9*l[0]*nu - 3*l[0] - 4*l[1]*nu*np.tan(g[0]) - 4*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(-l[0]**2*l[1]**2*nu + l[0]**2*l[1]**2 + l[0]**2*l[2]**2*nu - l[0]**2*l[2]**2 - 3*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) + 3*l[0]*l[1]**2*l[2]*np.tan(g[1]) - 2*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - l[1]**2*l[2]**2*nu + 2*l[1]**2*l[2]**2*np.tan(g[0])**2 + l[1]**2*l[2]**2*np.tan(g[1])**2 + l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(-9*l[0]**2*nu + 3*l[0]**2 + 6*l[0]*l[1]*nu*np.tan(g[0]) + 6*l[0]*l[1]*np.tan(g[0]) - 9*l[0]*l[2]*nu*np.tan(g[1]) + 3*l[0]*l[2]*np.tan(g[1]) + 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[2]*(3*l[0] - 2*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu - 1)),
           E*(3*l[0]**2*(-l[1]**2*nu + l[1]**2 - 2*l[2]**2*nu + 2*l[2]**2) + 2*l[0]**2*(l[1]**2*nu - l[1]**2 + 2*l[2]**2*nu - 2*l[2]**2) + l[1]*l[2]*(-3*l[0]*l[1]*nu*np.tan(g[1]) + 3*l[0]*l[1]*np.tan(g[1]) + 6*l[0]*l[2]*nu*np.tan(g[0]) - 6*l[0]*l[2]*np.tan(g[0]) - 2*l[1]*l[2]*nu*np.tan(g[0])**2 - l[1]*l[2]*nu*np.tan(g[1])**2 - l[1]*l[2]*nu + 2*l[1]*l[2]*np.tan(g[0])**2 + l[1]*l[2]*np.tan(g[1])**2 + l[1]*l[2]))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(-3*l[0]**2 + 3*l[0]*l[1]*np.tan(g[0]) - 3*l[0]*l[2]*np.tan(g[1]) + 2*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*l[2]*(-9*l[0]*nu + 3*l[0] + 2*l[1]*nu*np.tan(g[0]) + 2*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(-2*l[0]**2*l[1]**2*nu + 2*l[0]**2*l[1]**2 - 4*l[0]**2*l[2]**2*nu + 4*l[0]**2*l[2]**2 + 2*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 + l[1]**2*l[2]**2*nu*np.tan(g[1])**2 + l[1]**2*l[2]**2*nu - 2*l[1]**2*l[2]**2*np.tan(g[0])**2 - l[1]**2*l[2]**2*np.tan(g[1])**2 - l[1]**2*l[2]**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(-6*l[0]**2*nu - 6*l[0]**2 + 9*l[0]*l[1]*nu*np.tan(g[0]) - 3*l[0]*l[1]*np.tan(g[0]) + 9*l[0]*l[2]*nu*np.tan(g[1]) - 3*l[0]*l[2]*np.tan(g[1]) - 2*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) - 2*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[2]*(-3*l[0] + 4*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu - 1)),
           E*(2*l[0]**2*l[1]**2*nu - 2*l[0]**2*l[1]**2 + 4*l[0]**2*l[2]**2*nu - 4*l[0]**2*l[2]**2 + 3*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) - 3*l[0]*l[1]**2*l[2]*np.tan(g[1]) - 6*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) + 6*l[0]*l[1]*l[2]**2*np.tan(g[0]) + 4*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 + 2*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2*nu - 4*l[1]**2*l[2]**2*np.tan(g[0])**2 - 2*l[1]**2*l[2]**2*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(3*l[0]**2 - 3*l[0]*l[1]*np.tan(g[0]) + 3*l[0]*l[2]*np.tan(g[1]) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*l[2]*(9*l[0]*nu - 3*l[0] - 4*l[1]*nu*np.tan(g[0]) - 4*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(l[0]**2*l[1]**2*nu - l[0]**2*l[1]**2 + 2*l[0]**2*l[2]**2*nu - 2*l[0]**2*l[2]**2 - 4*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - 2*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2*nu + 4*l[1]**2*l[2]**2*np.tan(g[0])**2 + 2*l[1]**2*l[2]**2*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(3*l[0]**2*nu + 3*l[0]**2 - 18*l[0]*l[1]*nu*np.tan(g[0]) + 6*l[0]*l[1]*np.tan(g[0]) - 18*l[0]*l[2]*nu*np.tan(g[1]) + 6*l[0]*l[2]*np.tan(g[1]) + 8*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 8*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[2]*(3*l[0] - 2*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu - 1)),
           E*(l[0]**2*l[1]**2*nu - l[0]**2*l[1]**2 - 4*l[0]**2*l[2]**2*nu + 4*l[0]**2*l[2]**2 + 12*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) - 12*l[0]*l[1]*l[2]**2*np.tan(g[0]) - 4*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - 2*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2*nu + 4*l[1]**2*l[2]**2*np.tan(g[0])**2 + 2*l[1]**2*l[2]**2*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(9*l[0]**2*nu - 3*l[0]**2 - 9*l[0]*l[1]*nu*np.tan(g[0]) + 3*l[0]*l[1]*np.tan(g[0]) - 6*l[0]*l[2]*nu*np.tan(g[1]) - 6*l[0]*l[2]*np.tan(g[1]) + 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[2]*(-9*l[0]*nu + 3*l[0] + 2*l[1]*nu*np.tan(g[0]) + 2*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*l[1]**2*nu - 2*l[0]**2*l[1]**2 - 8*l[0]**2*l[2]**2*nu + 8*l[0]**2*l[2]**2 + 3*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) - 3*l[0]*l[1]**2*l[2]*np.tan(g[1]) + 4*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 + 2*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2*nu - 4*l[1]**2*l[2]**2*np.tan(g[0])**2 - 2*l[1]**2*l[2]**2*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(18*l[0]**2*nu - 6*l[0]**2 - 3*l[0]*l[1]*nu*np.tan(g[0]) - 3*l[0]*l[1]*np.tan(g[0]) + 18*l[0]*l[2]*nu*np.tan(g[1]) - 6*l[0]*l[2]*np.tan(g[1]) - 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1))],
          [E*l[1]*(9*l[0]*nu - 3*l[0] + 2*l[2]*nu*np.tan(g[1]) + 2*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(18*l[0]**2*nu - 6*l[0]**2 - 18*l[0]*l[1]*nu*np.tan(g[0]) + 6*l[0]*l[1]*np.tan(g[0]) + 3*l[0]*l[2]*nu*np.tan(g[1]) + 3*l[0]*l[2]*np.tan(g[1]) - 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(-8*l[0]**2*l[1]**2*nu + 8*l[0]**2*l[1]**2 + 2*l[0]**2*l[2]**2*nu - 2*l[0]**2*l[2]**2 - 3*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) + 3*l[0]*l[1]*l[2]**2*np.tan(g[0]) + 2*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 + 4*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2*nu - 2*l[1]**2*l[2]**2*np.tan(g[0])**2 - 4*l[1]**2*l[2]**2*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           -E*l[1]*(3*l[0] + 2*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*(9*l[0]**2*nu - 3*l[0]**2 + 6*l[0]*l[1]*nu*np.tan(g[0]) + 6*l[0]*l[1]*np.tan(g[0]) + 9*l[0]*l[2]*nu*np.tan(g[1]) - 3*l[0]*l[2]*np.tan(g[1]) + 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(-4*l[0]**2*l[1]**2*nu + 4*l[0]**2*l[1]**2 + l[0]**2*l[2]**2*nu - l[0]**2*l[2]**2 - 12*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) + 12*l[0]*l[1]**2*l[2]*np.tan(g[1]) - 2*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - 4*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2*nu + 2*l[1]**2*l[2]**2*np.tan(g[0])**2 + 4*l[1]**2*l[2]**2*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           -E*l[1]*(3*l[0] + 2*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*(-3*l[0]**2 + 3*l[0]*l[1]*np.tan(g[0]) - 3*l[0]*l[2]*np.tan(g[1]) + 2*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*(3*l[0]**2*(-2*l[1]**2*nu + 2*l[1]**2 - l[2]**2*nu + l[2]**2) + 2*l[0]**2*(2*l[1]**2*nu - 2*l[1]**2 + l[2]**2*nu - l[2]**2) + l[1]*l[2]*(-6*l[0]*l[1]*nu*np.tan(g[1]) + 6*l[0]*l[1]*np.tan(g[1]) + 3*l[0]*l[2]*nu*np.tan(g[0]) - 3*l[0]*l[2]*np.tan(g[0]) - l[1]*l[2]*nu*np.tan(g[0])**2 - 2*l[1]*l[2]*nu*np.tan(g[1])**2 - l[1]*l[2]*nu + l[1]*l[2]*np.tan(g[0])**2 + 2*l[1]*l[2]*np.tan(g[1])**2 + l[1]*l[2]))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(9*l[0]*nu - 3*l[0] + 2*l[2]*nu*np.tan(g[1]) + 2*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(-6*l[0]**2*nu - 6*l[0]**2 - 9*l[0]*l[1]*nu*np.tan(g[0]) + 3*l[0]*l[1]*np.tan(g[0]) - 9*l[0]*l[2]*nu*np.tan(g[1]) + 3*l[0]*l[2]*np.tan(g[1]) - 2*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) - 2*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(-4*l[0]**2*l[1]**2*nu + 4*l[0]**2*l[1]**2 - 2*l[0]**2*l[2]**2*nu + 2*l[0]**2*l[2]**2 + l[1]**2*l[2]**2*nu*np.tan(g[0])**2 + 2*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 + l[1]**2*l[2]**2*nu - l[1]**2*l[2]**2*np.tan(g[0])**2 - 2*l[1]**2*l[2]**2*np.tan(g[1])**2 - l[1]**2*l[2]**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(3*l[0] + 4*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*(3*l[0]**2 - 3*l[0]*l[1]*np.tan(g[0]) + 3*l[0]*l[2]*np.tan(g[1]) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*(4*l[0]**2*l[1]**2*nu - 4*l[0]**2*l[1]**2 + 2*l[0]**2*l[2]**2*nu - 2*l[0]**2*l[2]**2 + 6*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) - 6*l[0]*l[1]**2*l[2]*np.tan(g[1]) - 3*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) + 3*l[0]*l[1]*l[2]**2*np.tan(g[0]) + 2*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 + 4*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2*nu - 2*l[1]**2*l[2]**2*np.tan(g[0])**2 - 4*l[1]**2*l[2]**2*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(-9*l[0]*nu + 3*l[0] - 4*l[2]*nu*np.tan(g[1]) - 4*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(3*l[0]**2*nu + 3*l[0]**2 + 18*l[0]*l[1]*nu*np.tan(g[0]) - 6*l[0]*l[1]*np.tan(g[0]) + 18*l[0]*l[2]*nu*np.tan(g[1]) - 6*l[0]*l[2]*np.tan(g[1]) + 8*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 8*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*l[1]**2*nu - 2*l[0]**2*l[1]**2 + l[0]**2*l[2]**2*nu - l[0]**2*l[2]**2 - 2*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - 4*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2*nu + 2*l[1]**2*l[2]**2*np.tan(g[0])**2 + 4*l[1]**2*l[2]**2*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(-9*l[0]*nu + 3*l[0] - 4*l[2]*nu*np.tan(g[1]) - 4*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(-9*l[0]**2*nu + 3*l[0]**2 + 9*l[0]*l[1]*nu*np.tan(g[0]) - 3*l[0]*l[1]*np.tan(g[0]) - 6*l[0]*l[2]*nu*np.tan(g[1]) - 6*l[0]*l[2]*np.tan(g[1]) + 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(l[0]**2*l[1]**2*nu - l[0]**2*l[1]**2 - l[0]**2*l[2]**2*nu + l[0]**2*l[2]**2 + 3*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) - 3*l[0]*l[1]*l[2]**2*np.tan(g[0]) - l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - 2*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - l[1]**2*l[2]**2*nu + l[1]**2*l[2]**2*np.tan(g[0])**2 + 2*l[1]**2*l[2]**2*np.tan(g[1])**2 + l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(3*l[0] + 4*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*(-18*l[0]**2*nu + 6*l[0]**2 - 3*l[0]*l[1]*nu*np.tan(g[0]) - 3*l[0]*l[1]*np.tan(g[0]) - 18*l[0]*l[2]*nu*np.tan(g[1]) + 6*l[0]*l[2]*np.tan(g[1]) - 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*l[1]**2*nu - 2*l[0]**2*l[1]**2 - 2*l[0]**2*l[2]**2*nu + 2*l[0]**2*l[2]**2 + 3*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) - 3*l[0]*l[1]**2*l[2]*np.tan(g[1]) + l[1]**2*l[2]**2*nu*np.tan(g[0])**2 + 2*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 + l[1]**2*l[2]**2*nu - l[1]**2*l[2]**2*np.tan(g[0])**2 - 2*l[1]**2*l[2]**2*np.tan(g[1])**2 - l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1))],
          [E*(-2*l[0]**2*l[1]**2*nu + 2*l[0]**2*l[1]**2 + l[0]**2*l[2]**2*nu - l[0]**2*l[2]**2 + 6*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) - 6*l[0]*l[1]**2*l[2]*np.tan(g[1]) - 2*l[1]**2*l[2]**2*nu/np.cos(g[1])**2 - 2*l[1]**2*l[2]**2*nu/np.cos(g[0])**2 + 2*l[1]**2*l[2]**2/np.cos(g[1])**2 + 2*l[1]**2*l[2]**2/np.cos(g[0])**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(9*l[0]*nu - 3*l[0] - 4*l[1]*nu*np.tan(g[0]) - 4*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[1]*(3*l[0] - 2*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*(2*l[0]**2*(-2*l[1]**2*nu + 2*l[1]**2 + l[2]**2*nu - l[2]**2) + 3*l[0]*l[1]*l[2]**2*(nu - 1)*np.tan(g[0]) + 2*l[1]**2*l[2]**2*(nu/np.cos(g[1])**2 + nu/np.cos(g[0])**2 - 1/np.cos(g[1])**2 - 1/np.cos(g[0])**2))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(3*l[0] + 4*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu - 1)),
           E*l[1]*(9*l[0]*nu - 3*l[0] + 2*l[2]*nu*np.tan(g[1]) + 2*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*(-l[1]**2*nu + l[1]**2 - l[2]**2*nu + l[2]**2) + l[1]**2*l[2]**2*(nu/np.cos(g[1])**2 + nu/np.cos(g[0])**2 - 1/np.cos(g[1])**2 - 1/np.cos(g[0])**2))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(-9*l[0]*nu + 3*l[0] + 2*l[1]*nu*np.tan(g[0]) + 2*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[1]*(9*l[0]*nu - 3*l[0] + 2*l[2]*nu*np.tan(g[1]) + 2*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(3*l[0]**2*(-l[1]**2*nu + l[1]**2 - l[2]**2*nu + l[2]**2) + 2*l[0]**2*(l[1]**2*nu - l[1]**2 + l[2]**2*nu - l[2]**2) + l[1]*l[2]*(3*l[0]*l[1]*nu*np.tan(g[1]) - 3*l[0]*l[1]*np.tan(g[1]) - 3*l[0]*l[2]*nu*np.tan(g[0]) + 3*l[0]*l[2]*np.tan(g[0]) - l[1]*l[2]*nu/np.cos(g[1])**2 - l[1]*l[2]*nu/np.cos(g[0])**2 + l[1]*l[2]/np.cos(g[1])**2 + l[1]*l[2]/np.cos(g[0])**2))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           -E*l[2]*(3*l[0] + 2*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu - 1)),
           E*l[1]*(3*l[0] - 2*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*(l[0]**2*l[1]**2*nu - l[0]**2*l[1]**2 + l[0]**2*l[2]**2*nu - l[0]**2*l[2]**2 - 2*l[1]**2*l[2]**2*nu/np.cos(g[1])**2 - 2*l[1]**2*l[2]**2*nu/np.cos(g[0])**2 + 2*l[1]**2*l[2]**2/np.cos(g[1])**2 + 2*l[1]**2*l[2]**2/np.cos(g[0])**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(9*l[0]*nu - 3*l[0] - 4*l[1]*nu*np.tan(g[0]) - 4*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*l[1]*(-9*l[0]*nu + 3*l[0] - 4*l[2]*nu*np.tan(g[1]) - 4*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*(l[1]**2*nu - l[1]**2 + l[2]**2*nu - l[2]**2) + 3*l[0]*l[1]*l[2]*(-l[1]*nu*np.tan(g[1]) + l[1]*np.tan(g[1]) + l[2]*nu*np.tan(g[0]) - l[2]*np.tan(g[0])) + 2*l[1]**2*l[2]**2*(nu/np.cos(g[1])**2 + nu/np.cos(g[0])**2 - 1/np.cos(g[1])**2 - 1/np.cos(g[0])**2))/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(3*l[0] + 4*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu - 1)),
           E*l[1]*(-3*l[0] + 4*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*(2*l[0]**2*(l[1]**2*nu - l[1]**2 - 2*l[2]**2*nu + 2*l[2]**2) + 3*l[0]*l[1]**2*l[2]*(1 - nu)*np.tan(g[1]) + 2*l[1]**2*l[2]**2*(nu/np.cos(g[1])**2 + nu/np.cos(g[0])**2 - 1/np.cos(g[1])**2 - 1/np.cos(g[0])**2))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(-9*l[0]*nu + 3*l[0] + 2*l[1]*nu*np.tan(g[0]) + 2*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*l[1]*(-3*l[0] + 4*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*(l[0]**2*l[1]**2*nu - l[0]**2*l[1]**2 - 2*l[0]**2*l[2]**2*nu + 2*l[0]**2*l[2]**2 - 6*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) + 6*l[0]*l[1]*l[2]**2*np.tan(g[0]) - 2*l[1]**2*l[2]**2*nu/np.cos(g[1])**2 - 2*l[1]**2*l[2]**2*nu/np.cos(g[0])**2 + 2*l[1]**2*l[2]**2/np.cos(g[1])**2 + 2*l[1]**2*l[2]**2/np.cos(g[0])**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           -E*l[2]*(3*l[0] + 2*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu - 1)),
           E*l[1]*(-9*l[0]*nu + 3*l[0] - 4*l[2]*nu*np.tan(g[1]) - 4*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1))],
          [E*l[2]*(-9*l[0]*nu + 3*l[0] - 4*l[1]*nu*np.tan(g[0]) - 4*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(-l[0]**2*l[1]**2*nu + l[0]**2*l[1]**2 + l[0]**2*l[2]**2*nu - l[0]**2*l[2]**2 + 3*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) - 3*l[0]*l[1]**2*l[2]*np.tan(g[1]) - 2*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - l[1]**2*l[2]**2*nu + 2*l[1]**2*l[2]**2*np.tan(g[0])**2 + l[1]**2*l[2]**2*np.tan(g[1])**2 + l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(-9*l[0]**2*nu + 3*l[0]**2 - 6*l[0]*l[1]*nu*np.tan(g[0]) - 6*l[0]*l[1]*np.tan(g[0]) + 9*l[0]*l[2]*nu*np.tan(g[1]) - 3*l[0]*l[2]*np.tan(g[1]) + 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[2]*(3*l[0] + 4*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu - 1)),
           E*(2*l[0]**2*(-l[1]**2*nu + l[1]**2 + l[2]**2*nu - l[2]**2) + 3*l[0]*l[1]*l[2]**2*(nu - 1)*np.tan(g[0]) + l[1]**2*l[2]**2*(2*nu*np.tan(g[0])**2 + nu*np.tan(g[1])**2 + nu - 2*np.tan(g[0])**2 - np.tan(g[1])**2 - 1))/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(-18*l[0]**2*nu + 6*l[0]**2 - 18*l[0]*l[1]*nu*np.tan(g[0]) + 6*l[0]*l[1]*np.tan(g[0]) - 3*l[0]*l[2]*nu*np.tan(g[1]) - 3*l[0]*l[2]*np.tan(g[1]) - 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[2]*(9*l[0]*nu - 3*l[0] + 2*l[1]*nu*np.tan(g[0]) + 2*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*(-l[1]**2*nu + l[1]**2 - 2*l[2]**2*nu + 2*l[2]**2) + l[1]**2*l[2]**2*(2*nu*np.tan(g[0])**2 + nu*np.tan(g[1])**2 + nu - 2*np.tan(g[0])**2 - np.tan(g[1])**2 - 1))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(-6*l[0]**2*(2*nu**2 + nu - 1) + 3*l[0]*(2*nu - 1)*(-3*l[1]*nu*np.tan(g[0]) + l[1]*np.tan(g[0]) - 3*l[2]*nu*np.tan(g[1]) + l[2]*np.tan(g[1])) - 2*l[1]*l[2]*(2*nu**2 + nu - 1)*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu - 1)*(2*nu**2 + nu - 1)),
           -E*l[2]*(3*l[0] + 2*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu - 1)),
           E*(3*l[0]**2*(-l[1]**2*nu + l[1]**2 - 2*l[2]**2*nu + 2*l[2]**2) + 2*l[0]**2*(l[1]**2*nu - l[1]**2 + 2*l[2]**2*nu - 2*l[2]**2) + l[1]*l[2]*(3*l[0]*l[1]*nu*np.tan(g[1]) - 3*l[0]*l[1]*np.tan(g[1]) - 6*l[0]*l[2]*nu*np.tan(g[0]) + 6*l[0]*l[2]*np.tan(g[0]) - 2*l[1]*l[2]*nu*np.tan(g[0])**2 - l[1]*l[2]*nu*np.tan(g[1])**2 - l[1]*l[2]*nu + 2*l[1]*l[2]*np.tan(g[0])**2 + l[1]*l[2]*np.tan(g[1])**2 + l[1]*l[2]))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(-3*l[0]**2 - 3*l[0]*l[1]*np.tan(g[0]) + 3*l[0]*l[2]*np.tan(g[1]) + 2*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*l[2]*(-9*l[0]*nu + 3*l[0] - 4*l[1]*nu*np.tan(g[0]) - 4*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(l[0]**2*l[1]**2*nu - l[0]**2*l[1]**2 + 2*l[0]**2*l[2]**2*nu - 2*l[0]**2*l[2]**2 - 4*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - 2*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2*nu + 4*l[1]**2*l[2]**2*np.tan(g[0])**2 + 2*l[1]**2*l[2]**2*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(3*l[0]**2*nu + 3*l[0]**2 + 18*l[0]*l[1]*nu*np.tan(g[0]) - 6*l[0]*l[1]*np.tan(g[0]) + 18*l[0]*l[2]*nu*np.tan(g[1]) - 6*l[0]*l[2]*np.tan(g[1]) + 8*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 8*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[2]*(3*l[0] + 4*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu - 1)),
           E*(2*l[0]**2*(l[1]**2*nu - l[1]**2 + 2*l[2]**2*nu - 2*l[2]**2) + 3*l[0]*l[1]*l[2]*(-l[1]*nu*np.tan(g[1]) + l[1]*np.tan(g[1]) + 2*l[2]*nu*np.tan(g[0]) - 2*l[2]*np.tan(g[0])) + 2*l[1]**2*l[2]**2*(2*nu*np.tan(g[0])**2 + nu*np.tan(g[1])**2 + nu - 2*np.tan(g[0])**2 - np.tan(g[1])**2 - 1))/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(3*l[0]*(l[0] + l[1]*np.tan(g[0]) - l[2]*np.tan(g[1])) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*l[2]*(9*l[0]*nu - 3*l[0] + 2*l[1]*nu*np.tan(g[0]) + 2*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*(l[1]**2*nu - l[1]**2 - 4*l[2]**2*nu + 4*l[2]**2) + 3*l[0]*l[1]**2*l[2]*(1 - nu)*np.tan(g[1]) + 2*l[1]**2*l[2]**2*(2*nu*np.tan(g[0])**2 + nu*np.tan(g[1])**2 + nu - 2*np.tan(g[0])**2 - np.tan(g[1])**2 - 1))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(18*l[0]**2*nu - 6*l[0]**2 + 3*l[0]*l[1]*nu*np.tan(g[0]) + 3*l[0]*l[1]*np.tan(g[0]) - 18*l[0]*l[2]*nu*np.tan(g[1]) + 6*l[0]*l[2]*np.tan(g[1]) - 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           -E*l[2]*(3*l[0] + 2*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu - 1)),
           E*(l[0]**2*l[1]**2*nu - l[0]**2*l[1]**2 - 4*l[0]**2*l[2]**2*nu + 4*l[0]**2*l[2]**2 - 12*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) + 12*l[0]*l[1]*l[2]**2*np.tan(g[0]) - 4*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - 2*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2*nu + 4*l[1]**2*l[2]**2*np.tan(g[0])**2 + 2*l[1]**2*l[2]**2*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(9*l[0]**2*nu - 3*l[0]**2 + 9*l[0]*l[1]*nu*np.tan(g[0]) - 3*l[0]*l[1]*np.tan(g[0]) + 6*l[0]*l[2]*nu*np.tan(g[1]) + 6*l[0]*l[2]*np.tan(g[1]) + 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1))],
          [E*l[1]*(3*l[0] - 2*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*(9*l[0]**2*nu - 3*l[0]**2 - 6*l[0]*l[1]*nu*np.tan(g[0]) - 6*l[0]*l[1]*np.tan(g[0]) - 9*l[0]*l[2]*nu*np.tan(g[1]) + 3*l[0]*l[2]*np.tan(g[1]) + 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(-4*l[0]**2*l[1]**2*nu + 4*l[0]**2*l[1]**2 + l[0]**2*l[2]**2*nu - l[0]**2*l[2]**2 + 12*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) - 12*l[0]*l[1]**2*l[2]*np.tan(g[1]) - 2*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - 4*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2*nu + 2*l[1]**2*l[2]**2*np.tan(g[0])**2 + 4*l[1]**2*l[2]**2*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(-9*l[0]*nu + 3*l[0] + 2*l[2]*nu*np.tan(g[1]) + 2*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(18*l[0]**2*nu - 6*l[0]**2 + 18*l[0]*l[1]*nu*np.tan(g[0]) - 6*l[0]*l[1]*np.tan(g[0]) - 3*l[0]*l[2]*nu*np.tan(g[1]) - 3*l[0]*l[2]*np.tan(g[1]) - 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*(-4*l[1]**2*nu + 4*l[1]**2 + l[2]**2*nu - l[2]**2) + 3*l[0]*l[1]*l[2]**2*(nu - 1)*np.tan(g[0]) + 2*l[1]**2*l[2]**2*(nu*np.tan(g[0])**2 + 2*nu*np.tan(g[1])**2 + nu - np.tan(g[0])**2 - 2*np.tan(g[1])**2 - 1))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(-9*l[0]*nu + 3*l[0] + 2*l[2]*nu*np.tan(g[1]) + 2*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(-6*l[0]**2*(2*nu**2 + nu - 1) + 3*l[0]*(2*nu - 1)*(3*l[1]*nu*np.tan(g[0]) - l[1]*np.tan(g[0]) + 3*l[2]*nu*np.tan(g[1]) - l[2]*np.tan(g[1])) - 2*l[1]*l[2]*(2*nu**2 + nu - 1)*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu - 1)*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*(-2*l[1]**2*nu + 2*l[1]**2 - l[2]**2*nu + l[2]**2) + l[1]**2*l[2]**2*(nu*np.tan(g[0])**2 + 2*nu*np.tan(g[1])**2 + nu - np.tan(g[0])**2 - 2*np.tan(g[1])**2 - 1))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(3*l[0] - 2*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*(-3*l[0]**2 - 3*l[0]*l[1]*np.tan(g[0]) + 3*l[0]*l[2]*np.tan(g[1]) + 2*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*(3*l[0]**2*(-2*l[1]**2*nu + 2*l[1]**2 - l[2]**2*nu + l[2]**2) + 2*l[0]**2*(2*l[1]**2*nu - 2*l[1]**2 + l[2]**2*nu - l[2]**2) + l[1]*l[2]*(6*l[0]*l[1]*nu*np.tan(g[1]) - 6*l[0]*l[1]*np.tan(g[1]) - 3*l[0]*l[2]*nu*np.tan(g[0]) + 3*l[0]*l[2]*np.tan(g[0]) - l[1]*l[2]*nu*np.tan(g[0])**2 - 2*l[1]*l[2]*nu*np.tan(g[1])**2 - l[1]*l[2]*nu + l[1]*l[2]*np.tan(g[0])**2 + 2*l[1]*l[2]*np.tan(g[1])**2 + l[1]*l[2]))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(9*l[0]*nu - 3*l[0] - 4*l[2]*nu*np.tan(g[1]) - 4*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(3*l[0]**2*nu + 3*l[0]**2 - 18*l[0]*l[1]*nu*np.tan(g[0]) + 6*l[0]*l[1]*np.tan(g[0]) - 18*l[0]*l[2]*nu*np.tan(g[1]) + 6*l[0]*l[2]*np.tan(g[1]) + 8*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 8*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*l[1]**2*nu - 2*l[0]**2*l[1]**2 + l[0]**2*l[2]**2*nu - l[0]**2*l[2]**2 - 2*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - 4*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2*nu + 2*l[1]**2*l[2]**2*np.tan(g[0])**2 + 4*l[1]**2*l[2]**2*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(-3*l[0] + 4*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*(3*l[0]*(l[0] + l[1]*np.tan(g[0]) - l[2]*np.tan(g[1])) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*(2*l[0]**2*(2*l[1]**2*nu - 2*l[1]**2 + l[2]**2*nu - l[2]**2) + 3*l[0]*l[1]*l[2]*(-2*l[1]*nu*np.tan(g[1]) + 2*l[1]*np.tan(g[1]) + l[2]*nu*np.tan(g[0]) - l[2]*np.tan(g[0])) + 2*l[1]**2*l[2]**2*(nu*np.tan(g[0])**2 + 2*nu*np.tan(g[1])**2 + nu - np.tan(g[0])**2 - 2*np.tan(g[1])**2 - 1))/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(-3*l[0] + 4*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*(-18*l[0]**2*nu + 6*l[0]**2 + 3*l[0]*l[1]*nu*np.tan(g[0]) + 3*l[0]*l[1]*np.tan(g[0]) + 18*l[0]*l[2]*nu*np.tan(g[1]) - 6*l[0]*l[2]*np.tan(g[1]) - 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*(l[1]**2*nu - l[1]**2 - l[2]**2*nu + l[2]**2) + 3*l[0]*l[1]**2*l[2]*(1 - nu)*np.tan(g[1]) + l[1]**2*l[2]**2*(nu*np.tan(g[0])**2 + 2*nu*np.tan(g[1])**2 + nu - np.tan(g[0])**2 - 2*np.tan(g[1])**2 - 1))/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(9*l[0]*nu - 3*l[0] - 4*l[2]*nu*np.tan(g[1]) - 4*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(-9*l[0]**2*nu + 3*l[0]**2 - 9*l[0]*l[1]*nu*np.tan(g[0]) + 3*l[0]*l[1]*np.tan(g[0]) + 6*l[0]*l[2]*nu*np.tan(g[1]) + 6*l[0]*l[2]*np.tan(g[1]) + 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(l[0]**2*l[1]**2*nu - l[0]**2*l[1]**2 - l[0]**2*l[2]**2*nu + l[0]**2*l[2]**2 - 3*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) + 3*l[0]*l[1]*l[2]**2*np.tan(g[0]) - l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - 2*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - l[1]**2*l[2]**2*nu + l[1]**2*l[2]**2*np.tan(g[0])**2 + 2*l[1]**2*l[2]**2*np.tan(g[1])**2 + l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1))],
          [E*(3*l[0]**2*(-l[1]**2*nu + l[1]**2 - l[2]**2*nu + l[2]**2) + 2*l[0]**2*(l[1]**2*nu - l[1]**2 + l[2]**2*nu - l[2]**2) + l[1]*l[2]*(3*l[0]*l[1]*nu*np.tan(g[1]) - 3*l[0]*l[1]*np.tan(g[1]) + 3*l[0]*l[2]*nu*np.tan(g[0]) - 3*l[0]*l[2]*np.tan(g[0]) - l[1]*l[2]*nu/np.cos(g[1])**2 - l[1]*l[2]*nu/np.cos(g[0])**2 + l[1]*l[2]/np.cos(g[1])**2 + l[1]*l[2]/np.cos(g[0])**2))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(3*l[0] - 2*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu - 1)),
           E*l[1]*(3*l[0] - 2*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*(2*l[0]**2*(-l[1]**2*nu + l[1]**2 - l[2]**2*nu + l[2]**2) + l[1]**2*l[2]**2*(nu/np.cos(g[1])**2 + nu/np.cos(g[0])**2 - 1/np.cos(g[1])**2 - 1/np.cos(g[0])**2))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(9*l[0]*nu - 3*l[0] + 2*l[1]*nu*np.tan(g[0]) + 2*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[1]*(9*l[0]*nu - 3*l[0] + 2*l[2]*nu*np.tan(g[1]) + 2*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*(-2*l[1]**2*nu + 2*l[1]**2 + l[2]**2*nu - l[2]**2) + 3*l[0]*l[1]*l[2]**2*(1 - nu)*np.tan(g[0]) + 2*l[1]**2*l[2]**2*(nu/np.cos(g[1])**2 + nu/np.cos(g[0])**2 - 1/np.cos(g[1])**2 - 1/np.cos(g[0])**2))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(-3*l[0] + 4*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu - 1)),
           E*l[1]*(9*l[0]*nu - 3*l[0] + 2*l[2]*nu*np.tan(g[1]) + 2*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(-2*l[0]**2*l[1]**2*nu + 2*l[0]**2*l[1]**2 + l[0]**2*l[2]**2*nu - l[0]**2*l[2]**2 + 6*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) - 6*l[0]*l[1]**2*l[2]*np.tan(g[1]) - 2*l[1]**2*l[2]**2*nu/np.cos(g[1])**2 - 2*l[1]**2*l[2]**2*nu/np.cos(g[0])**2 + 2*l[1]**2*l[2]**2/np.cos(g[1])**2 + 2*l[1]**2*l[2]**2/np.cos(g[0])**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(-9*l[0]*nu + 3*l[0] - 4*l[1]*nu*np.tan(g[0]) - 4*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[1]*(3*l[0] - 2*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*(l[0]**2*l[1]**2*nu - l[0]**2*l[1]**2 - 2*l[0]**2*l[2]**2*nu + 2*l[0]**2*l[2]**2 + 6*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) - 6*l[0]*l[1]*l[2]**2*np.tan(g[0]) - 2*l[1]**2*l[2]**2*nu/np.cos(g[1])**2 - 2*l[1]**2*l[2]**2*nu/np.cos(g[0])**2 + 2*l[1]**2*l[2]**2/np.cos(g[1])**2 + 2*l[1]**2*l[2]**2/np.cos(g[0])**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(3*l[0] - 2*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu - 1)),
           E*l[1]*(-9*l[0]*nu + 3*l[0] - 4*l[2]*nu*np.tan(g[1]) - 4*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*(l[1]**2*nu - l[1]**2 - 2*l[2]**2*nu + 2*l[2]**2) + 3*l[0]*l[1]**2*l[2]*(1 - nu)*np.tan(g[1]) + 2*l[1]**2*l[2]**2*(nu/np.cos(g[1])**2 + nu/np.cos(g[0])**2 - 1/np.cos(g[1])**2 - 1/np.cos(g[0])**2))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(9*l[0]*nu - 3*l[0] + 2*l[1]*nu*np.tan(g[0]) + 2*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*l[1]*(-3*l[0] + 4*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*(2*l[0]**2*(l[1]**2*nu - l[1]**2 + l[2]**2*nu - l[2]**2) + 3*l[0]*l[1]*l[2]*(-l[1]*nu*np.tan(g[1]) + l[1]*np.tan(g[1]) - l[2]*nu*np.tan(g[0]) + l[2]*np.tan(g[0])) + 2*l[1]**2*l[2]**2*(nu/np.cos(g[1])**2 + nu/np.cos(g[0])**2 - 1/np.cos(g[1])**2 - 1/np.cos(g[0])**2))/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(-3*l[0] + 4*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu - 1)),
           E*l[1]*(-3*l[0] + 4*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*(l[0]**2*l[1]**2*nu - l[0]**2*l[1]**2 + l[0]**2*l[2]**2*nu - l[0]**2*l[2]**2 - 2*l[1]**2*l[2]**2*nu/np.cos(g[1])**2 - 2*l[1]**2*l[2]**2*nu/np.cos(g[0])**2 + 2*l[1]**2*l[2]**2/np.cos(g[1])**2 + 2*l[1]**2*l[2]**2/np.cos(g[0])**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(-9*l[0]*nu + 3*l[0] - 4*l[1]*nu*np.tan(g[0]) - 4*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*l[1]*(-9*l[0]*nu + 3*l[0] - 4*l[2]*nu*np.tan(g[1]) - 4*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu**2 + nu - 1))],
          [E*l[2]*(3*l[0] - 2*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu - 1)),
           E*(3*l[0]**2*(-l[1]**2*nu + l[1]**2 - 2*l[2]**2*nu + 2*l[2]**2) + 2*l[0]**2*(l[1]**2*nu - l[1]**2 + 2*l[2]**2*nu - 2*l[2]**2) + l[1]*l[2]*(3*l[0]*l[1]*nu*np.tan(g[1]) - 3*l[0]*l[1]*np.tan(g[1]) + 6*l[0]*l[2]*nu*np.tan(g[0]) - 6*l[0]*l[2]*np.tan(g[0]) - 2*l[1]*l[2]*nu*np.tan(g[0])**2 - l[1]*l[2]*nu*np.tan(g[1])**2 - l[1]*l[2]*nu + 2*l[1]*l[2]*np.tan(g[0])**2 + l[1]*l[2]*np.tan(g[1])**2 + l[1]*l[2]))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(3*l[0]**2 - 3*l[0]*l[1]*np.tan(g[0]) - 3*l[0]*l[2]*np.tan(g[1]) + 2*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*l[2]*(-9*l[0]*nu + 3*l[0] + 2*l[1]*nu*np.tan(g[0]) + 2*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*(-l[1]**2*nu + l[1]**2 - 2*l[2]**2*nu + 2*l[2]**2) + l[1]**2*l[2]**2*(2*nu*np.tan(g[0])**2 + nu*np.tan(g[1])**2 + nu - 2*np.tan(g[0])**2 - np.tan(g[1])**2 - 1))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(6*l[0]**2*(2*nu**2 + nu - 1) + 3*l[0]*(2*nu - 1)*(-3*l[1]*nu*np.tan(g[0]) + l[1]*np.tan(g[0]) + 3*l[2]*nu*np.tan(g[1]) - l[2]*np.tan(g[1])) - 2*l[1]*l[2]*(2*nu**2 + nu - 1)*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu - 1)*(2*nu**2 + nu - 1)),
           E*l[2]*(-3*l[0] + 4*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu - 1)),
           E*(2*l[0]**2*(-l[1]**2*nu + l[1]**2 + l[2]**2*nu - l[2]**2) + 3*l[0]*l[1]*l[2]**2*(1 - nu)*np.tan(g[0]) + l[1]**2*l[2]**2*(2*nu*np.tan(g[0])**2 + nu*np.tan(g[1])**2 + nu - 2*np.tan(g[0])**2 - np.tan(g[1])**2 - 1))/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(18*l[0]**2*nu - 6*l[0]**2 - 18*l[0]*l[1]*nu*np.tan(g[0]) + 6*l[0]*l[1]*np.tan(g[0]) + 3*l[0]*l[2]*nu*np.tan(g[1]) + 3*l[0]*l[2]*np.tan(g[1]) - 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[2]*(9*l[0]*nu - 3*l[0] - 4*l[1]*nu*np.tan(g[0]) - 4*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(-l[0]**2*l[1]**2*nu + l[0]**2*l[1]**2 + l[0]**2*l[2]**2*nu - l[0]**2*l[2]**2 + 3*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) - 3*l[0]*l[1]**2*l[2]*np.tan(g[1]) - 2*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - l[1]**2*l[2]**2*nu + 2*l[1]**2*l[2]**2*np.tan(g[0])**2 + l[1]**2*l[2]**2*np.tan(g[1])**2 + l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(9*l[0]**2*nu - 3*l[0]**2 - 6*l[0]*l[1]*nu*np.tan(g[0]) - 6*l[0]*l[1]*np.tan(g[0]) - 9*l[0]*l[2]*nu*np.tan(g[1]) + 3*l[0]*l[2]*np.tan(g[1]) + 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[2]*(3*l[0] - 2*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu - 1)),
           E*(l[0]**2*l[1]**2*nu - l[0]**2*l[1]**2 - 4*l[0]**2*l[2]**2*nu + 4*l[0]**2*l[2]**2 + 12*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) - 12*l[0]*l[1]*l[2]**2*np.tan(g[0]) - 4*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - 2*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2*nu + 4*l[1]**2*l[2]**2*np.tan(g[0])**2 + 2*l[1]**2*l[2]**2*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(-9*l[0]**2*nu + 3*l[0]**2 + 9*l[0]*l[1]*nu*np.tan(g[0]) - 3*l[0]*l[1]*np.tan(g[0]) - 6*l[0]*l[2]*nu*np.tan(g[1]) - 6*l[0]*l[2]*np.tan(g[1]) + 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[2]*(-9*l[0]*nu + 3*l[0] + 2*l[1]*nu*np.tan(g[0]) + 2*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*(l[1]**2*nu - l[1]**2 - 4*l[2]**2*nu + 4*l[2]**2) + 3*l[0]*l[1]**2*l[2]*(1 - nu)*np.tan(g[1]) + 2*l[1]**2*l[2]**2*(2*nu*np.tan(g[0])**2 + nu*np.tan(g[1])**2 + nu - 2*np.tan(g[0])**2 - np.tan(g[1])**2 - 1))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(-18*l[0]**2*nu + 6*l[0]**2 + 3*l[0]*l[1]*nu*np.tan(g[0]) + 3*l[0]*l[1]*np.tan(g[0]) + 18*l[0]*l[2]*nu*np.tan(g[1]) - 6*l[0]*l[2]*np.tan(g[1]) - 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[2]*(-3*l[0] + 4*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu - 1)),
           E*(2*l[0]**2*(l[1]**2*nu - l[1]**2 + 2*l[2]**2*nu - 2*l[2]**2) + 3*l[0]*l[1]*l[2]*(-l[1]*nu*np.tan(g[1]) + l[1]*np.tan(g[1]) - 2*l[2]*nu*np.tan(g[0]) + 2*l[2]*np.tan(g[0])) + 2*l[1]**2*l[2]**2*(2*nu*np.tan(g[0])**2 + nu*np.tan(g[1])**2 + nu - 2*np.tan(g[0])**2 - np.tan(g[1])**2 - 1))/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(3*l[0]*(-l[0] + l[1]*np.tan(g[0]) + l[2]*np.tan(g[1])) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*l[2]*(9*l[0]*nu - 3*l[0] - 4*l[1]*nu*np.tan(g[0]) - 4*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(l[0]**2*l[1]**2*nu - l[0]**2*l[1]**2 + 2*l[0]**2*l[2]**2*nu - 2*l[0]**2*l[2]**2 - 4*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - 2*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2*nu + 4*l[1]**2*l[2]**2*np.tan(g[0])**2 + 2*l[1]**2*l[2]**2*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(-3*l[0]**2*nu - 3*l[0]**2 + 18*l[0]*l[1]*nu*np.tan(g[0]) - 6*l[0]*l[1]*np.tan(g[0]) - 18*l[0]*l[2]*nu*np.tan(g[1]) + 6*l[0]*l[2]*np.tan(g[1]) + 8*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 8*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1))],
          [E*l[1]*(3*l[0] - 2*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*(3*l[0]**2 - 3*l[0]*l[1]*np.tan(g[0]) - 3*l[0]*l[2]*np.tan(g[1]) + 2*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*(3*l[0]**2*(-2*l[1]**2*nu + 2*l[1]**2 - l[2]**2*nu + l[2]**2) + 2*l[0]**2*(2*l[1]**2*nu - 2*l[1]**2 + l[2]**2*nu - l[2]**2) + l[1]*l[2]*(6*l[0]*l[1]*nu*np.tan(g[1]) - 6*l[0]*l[1]*np.tan(g[1]) + 3*l[0]*l[2]*nu*np.tan(g[0]) - 3*l[0]*l[2]*np.tan(g[0]) - l[1]*l[2]*nu*np.tan(g[0])**2 - 2*l[1]*l[2]*nu*np.tan(g[1])**2 - l[1]*l[2]*nu + l[1]*l[2]*np.tan(g[0])**2 + 2*l[1]*l[2]*np.tan(g[1])**2 + l[1]*l[2]))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(-9*l[0]*nu + 3*l[0] + 2*l[2]*nu*np.tan(g[1]) + 2*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(6*l[0]**2*(2*nu**2 + nu - 1) + 3*l[0]*(2*nu - 1)*(3*l[1]*nu*np.tan(g[0]) - l[1]*np.tan(g[0]) - 3*l[2]*nu*np.tan(g[1]) + l[2]*np.tan(g[1])) - 2*l[1]*l[2]*(2*nu**2 + nu - 1)*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu - 1)*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*(-2*l[1]**2*nu + 2*l[1]**2 - l[2]**2*nu + l[2]**2) + l[1]**2*l[2]**2*(nu*np.tan(g[0])**2 + 2*nu*np.tan(g[1])**2 + nu - np.tan(g[0])**2 - 2*np.tan(g[1])**2 - 1))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(-9*l[0]*nu + 3*l[0] + 2*l[2]*nu*np.tan(g[1]) + 2*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(-18*l[0]**2*nu + 6*l[0]**2 + 18*l[0]*l[1]*nu*np.tan(g[0]) - 6*l[0]*l[1]*np.tan(g[0]) + 3*l[0]*l[2]*nu*np.tan(g[1]) + 3*l[0]*l[2]*np.tan(g[1]) - 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*(-4*l[1]**2*nu + 4*l[1]**2 + l[2]**2*nu - l[2]**2) + 3*l[0]*l[1]*l[2]**2*(1 - nu)*np.tan(g[0]) + 2*l[1]**2*l[2]**2*(nu*np.tan(g[0])**2 + 2*nu*np.tan(g[1])**2 + nu - np.tan(g[0])**2 - 2*np.tan(g[1])**2 - 1))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(3*l[0] - 2*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*(-9*l[0]**2*nu + 3*l[0]**2 - 6*l[0]*l[1]*nu*np.tan(g[0]) - 6*l[0]*l[1]*np.tan(g[0]) + 9*l[0]*l[2]*nu*np.tan(g[1]) - 3*l[0]*l[2]*np.tan(g[1]) + 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(-4*l[0]**2*l[1]**2*nu + 4*l[0]**2*l[1]**2 + l[0]**2*l[2]**2*nu - l[0]**2*l[2]**2 + 12*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) - 12*l[0]*l[1]**2*l[2]*np.tan(g[1]) - 2*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - 4*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2*nu + 2*l[1]**2*l[2]**2*np.tan(g[0])**2 + 4*l[1]**2*l[2]**2*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(9*l[0]*nu - 3*l[0] - 4*l[2]*nu*np.tan(g[1]) - 4*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(9*l[0]**2*nu - 3*l[0]**2 - 9*l[0]*l[1]*nu*np.tan(g[0]) + 3*l[0]*l[1]*np.tan(g[0]) - 6*l[0]*l[2]*nu*np.tan(g[1]) - 6*l[0]*l[2]*np.tan(g[1]) + 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(l[0]**2*l[1]**2*nu - l[0]**2*l[1]**2 - l[0]**2*l[2]**2*nu + l[0]**2*l[2]**2 + 3*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) - 3*l[0]*l[1]*l[2]**2*np.tan(g[0]) - l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - 2*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - l[1]**2*l[2]**2*nu + l[1]**2*l[2]**2*np.tan(g[0])**2 + 2*l[1]**2*l[2]**2*np.tan(g[1])**2 + l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(-3*l[0] + 4*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*(18*l[0]**2*nu - 6*l[0]**2 + 3*l[0]*l[1]*nu*np.tan(g[0]) + 3*l[0]*l[1]*np.tan(g[0]) - 18*l[0]*l[2]*nu*np.tan(g[1]) + 6*l[0]*l[2]*np.tan(g[1]) - 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*(l[1]**2*nu - l[1]**2 - l[2]**2*nu + l[2]**2) + 3*l[0]*l[1]**2*l[2]*(1 - nu)*np.tan(g[1]) + l[1]**2*l[2]**2*(nu*np.tan(g[0])**2 + 2*nu*np.tan(g[1])**2 + nu - np.tan(g[0])**2 - 2*np.tan(g[1])**2 - 1))/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(-3*l[0] + 4*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*(3*l[0]*(-l[0] + l[1]*np.tan(g[0]) + l[2]*np.tan(g[1])) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*(2*l[0]**2*(2*l[1]**2*nu - 2*l[1]**2 + l[2]**2*nu - l[2]**2) + 3*l[0]*l[1]*l[2]*(-2*l[1]*nu*np.tan(g[1]) + 2*l[1]*np.tan(g[1]) - l[2]*nu*np.tan(g[0]) + l[2]*np.tan(g[0])) + 2*l[1]**2*l[2]**2*(nu*np.tan(g[0])**2 + 2*nu*np.tan(g[1])**2 + nu - np.tan(g[0])**2 - 2*np.tan(g[1])**2 - 1))/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(9*l[0]*nu - 3*l[0] - 4*l[2]*nu*np.tan(g[1]) - 4*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(-3*l[0]**2*nu - 3*l[0]**2 - 18*l[0]*l[1]*nu*np.tan(g[0]) + 6*l[0]*l[1]*np.tan(g[0]) + 18*l[0]*l[2]*nu*np.tan(g[1]) - 6*l[0]*l[2]*np.tan(g[1]) + 8*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 8*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*l[1]**2*nu - 2*l[0]**2*l[1]**2 + l[0]**2*l[2]**2*nu - l[0]**2*l[2]**2 - 2*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - 4*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2*nu + 2*l[1]**2*l[2]**2*np.tan(g[0])**2 + 4*l[1]**2*l[2]**2*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1))],
          [E*(-2*l[0]**2*l[1]**2*nu + 2*l[0]**2*l[1]**2 - 2*l[0]**2*l[2]**2*nu + 2*l[0]**2*l[2]**2 + l[1]**2*l[2]**2*nu/np.cos(g[1])**2 + l[1]**2*l[2]**2*nu/np.cos(g[0])**2 - l[1]**2*l[2]**2/np.cos(g[1])**2 - l[1]**2*l[2]**2/np.cos(g[0])**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(-9*l[0]*nu + 3*l[0] + 2*l[1]*nu*np.tan(g[0]) + 2*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[1]*(-9*l[0]*nu + 3*l[0] + 2*l[2]*nu*np.tan(g[1]) + 2*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(3*l[0]**2*(-l[1]**2*nu + l[1]**2 - l[2]**2*nu + l[2]**2) + 2*l[0]**2*(l[1]**2*nu - l[1]**2 + l[2]**2*nu - l[2]**2) + l[1]*l[2]*(-3*l[0]*l[1]*nu*np.tan(g[1]) + 3*l[0]*l[1]*np.tan(g[1]) - 3*l[0]*l[2]*nu*np.tan(g[0]) + 3*l[0]*l[2]*np.tan(g[0]) - l[1]*l[2]*nu/np.cos(g[1])**2 - l[1]*l[2]*nu/np.cos(g[0])**2 + l[1]*l[2]/np.cos(g[1])**2 + l[1]*l[2]/np.cos(g[0])**2))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           -E*l[2]*(3*l[0] + 2*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu - 1)),
           -E*l[1]*(3*l[0] + 2*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*(-2*l[0]**2*l[1]**2*nu + 2*l[0]**2*l[1]**2 + l[0]**2*l[2]**2*nu - l[0]**2*l[2]**2 - 6*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) + 6*l[0]*l[1]**2*l[2]*np.tan(g[1]) - 2*l[1]**2*l[2]**2*nu/np.cos(g[1])**2 - 2*l[1]**2*l[2]**2*nu/np.cos(g[0])**2 + 2*l[1]**2*l[2]**2/np.cos(g[1])**2 + 2*l[1]**2*l[2]**2/np.cos(g[0])**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(9*l[0]*nu - 3*l[0] - 4*l[1]*nu*np.tan(g[0]) - 4*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu**2 + nu - 1)),
           -E*l[1]*(3*l[0] + 2*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*(-4*l[0]**2*l[1]**2*nu + 4*l[0]**2*l[1]**2 + 2*l[0]**2*l[2]**2*nu - 2*l[0]**2*l[2]**2 + 3*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) - 3*l[0]*l[1]*l[2]**2*np.tan(g[0]) + 2*l[1]**2*l[2]**2*nu/np.cos(g[1])**2 + 2*l[1]**2*l[2]**2*nu/np.cos(g[0])**2 - 2*l[1]**2*l[2]**2/np.cos(g[1])**2 - 2*l[1]**2*l[2]**2/np.cos(g[0])**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(3*l[0] + 4*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu - 1)),
           E*l[1]*(-9*l[0]*nu + 3*l[0] + 2*l[2]*nu*np.tan(g[1]) + 2*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*l[1]**2*nu - 2*l[0]**2*l[1]**2 - 4*l[0]**2*l[2]**2*nu + 4*l[0]**2*l[2]**2 + 3*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) - 3*l[0]*l[1]**2*l[2]*np.tan(g[1]) + 2*l[1]**2*l[2]**2*nu/np.cos(g[1])**2 + 2*l[1]**2*l[2]**2*nu/np.cos(g[0])**2 - 2*l[1]**2*l[2]**2/np.cos(g[1])**2 - 2*l[1]**2*l[2]**2/np.cos(g[0])**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(-9*l[0]*nu + 3*l[0] + 2*l[1]*nu*np.tan(g[0]) + 2*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*l[1]*(3*l[0] + 4*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*(l[0]**2*l[1]**2*nu - l[0]**2*l[1]**2 - 2*l[0]**2*l[2]**2*nu + 2*l[0]**2*l[2]**2 - 6*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) + 6*l[0]*l[1]*l[2]**2*np.tan(g[0]) - 2*l[1]**2*l[2]**2*nu/np.cos(g[1])**2 - 2*l[1]**2*l[2]**2*nu/np.cos(g[0])**2 + 2*l[1]**2*l[2]**2/np.cos(g[1])**2 + 2*l[1]**2*l[2]**2/np.cos(g[0])**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           -E*l[2]*(3*l[0] + 2*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu - 1)),
           E*l[1]*(9*l[0]*nu - 3*l[0] - 4*l[2]*nu*np.tan(g[1]) - 4*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(l[0]**2*l[1]**2*nu - l[0]**2*l[1]**2 + l[0]**2*l[2]**2*nu - l[0]**2*l[2]**2 - 2*l[1]**2*l[2]**2*nu/np.cos(g[1])**2 - 2*l[1]**2*l[2]**2*nu/np.cos(g[0])**2 + 2*l[1]**2*l[2]**2/np.cos(g[1])**2 + 2*l[1]**2*l[2]**2/np.cos(g[0])**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(9*l[0]*nu - 3*l[0] - 4*l[1]*nu*np.tan(g[0]) - 4*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*l[1]*(9*l[0]*nu - 3*l[0] - 4*l[2]*nu*np.tan(g[1]) - 4*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*l[1]**2*nu - 2*l[0]**2*l[1]**2 + 2*l[0]**2*l[2]**2*nu - 2*l[0]**2*l[2]**2 + 3*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) - 3*l[0]*l[1]**2*l[2]*np.tan(g[1]) + 3*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) - 3*l[0]*l[1]*l[2]**2*np.tan(g[0]) + 2*l[1]**2*l[2]**2*nu/np.cos(g[1])**2 + 2*l[1]**2*l[2]**2*nu/np.cos(g[0])**2 - 2*l[1]**2*l[2]**2/np.cos(g[1])**2 - 2*l[1]**2*l[2]**2/np.cos(g[0])**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[2]*(3*l[0] + 4*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu - 1)),
           E*l[1]*(3*l[0] + 4*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu - 1))],
          [E*l[2]*(9*l[0]*nu - 3*l[0] + 2*l[1]*nu*np.tan(g[0]) + 2*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(-2*l[0]**2*l[1]**2*nu + 2*l[0]**2*l[1]**2 - 4*l[0]**2*l[2]**2*nu + 4*l[0]**2*l[2]**2 + 2*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 + l[1]**2*l[2]**2*nu*np.tan(g[1])**2 + l[1]**2*l[2]**2*nu - 2*l[1]**2*l[2]**2*np.tan(g[0])**2 - l[1]**2*l[2]**2*np.tan(g[1])**2 - l[1]**2*l[2]**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(6*l[0]**2*nu + 6*l[0]**2 + 9*l[0]*l[1]*nu*np.tan(g[0]) - 3*l[0]*l[1]*np.tan(g[0]) - 9*l[0]*l[2]*nu*np.tan(g[1]) + 3*l[0]*l[2]*np.tan(g[1]) - 2*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) - 2*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           -E*l[2]*(3*l[0] + 2*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu - 1)),
           E*(3*l[0]**2*(-l[1]**2*nu + l[1]**2 - 2*l[2]**2*nu + 2*l[2]**2) + 2*l[0]**2*(l[1]**2*nu - l[1]**2 + 2*l[2]**2*nu - 2*l[2]**2) + l[1]*l[2]*(-3*l[0]*l[1]*nu*np.tan(g[1]) + 3*l[0]*l[1]*np.tan(g[1]) - 6*l[0]*l[2]*nu*np.tan(g[0]) + 6*l[0]*l[2]*np.tan(g[0]) - 2*l[1]*l[2]*nu*np.tan(g[0])**2 - l[1]*l[2]*nu*np.tan(g[1])**2 - l[1]*l[2]*nu + 2*l[1]*l[2]*np.tan(g[0])**2 + l[1]*l[2]*np.tan(g[1])**2 + l[1]*l[2]))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(3*l[0]**2 + 3*l[0]*l[1]*np.tan(g[0]) + 3*l[0]*l[2]*np.tan(g[1]) + 2*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*l[2]*(-9*l[0]*nu + 3*l[0] - 4*l[1]*nu*np.tan(g[0]) - 4*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(-l[0]**2*l[1]**2*nu + l[0]**2*l[1]**2 + l[0]**2*l[2]**2*nu - l[0]**2*l[2]**2 - 3*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) + 3*l[0]*l[1]**2*l[2]*np.tan(g[1]) - 2*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - l[1]**2*l[2]**2*nu + 2*l[1]**2*l[2]**2*np.tan(g[0])**2 + l[1]**2*l[2]**2*np.tan(g[1])**2 + l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(9*l[0]**2*nu - 3*l[0]**2 + 6*l[0]*l[1]*nu*np.tan(g[0]) + 6*l[0]*l[1]*np.tan(g[0]) + 9*l[0]*l[2]*nu*np.tan(g[1]) - 3*l[0]*l[2]*np.tan(g[1]) + 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[2]*(3*l[0] + 4*l[1]*np.tan(g[0]))/(144*l[0]*(2*nu - 1)),
           E*(-2*l[0]**2*l[1]**2*nu + 2*l[0]**2*l[1]**2 + 2*l[0]**2*l[2]**2*nu - 2*l[0]**2*l[2]**2 + 3*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) - 3*l[0]*l[1]*l[2]**2*np.tan(g[0]) + 2*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 + l[1]**2*l[2]**2*nu*np.tan(g[1])**2 + l[1]**2*l[2]**2*nu - 2*l[1]**2*l[2]**2*np.tan(g[0])**2 - l[1]**2*l[2]**2*np.tan(g[1])**2 - l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(18*l[0]**2*nu - 6*l[0]**2 + 18*l[0]*l[1]*nu*np.tan(g[0]) - 6*l[0]*l[1]*np.tan(g[0]) - 3*l[0]*l[2]*nu*np.tan(g[1]) - 3*l[0]*l[2]*np.tan(g[1]) - 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[2]*(9*l[0]*nu - 3*l[0] + 2*l[1]*nu*np.tan(g[0]) + 2*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*l[1]**2*nu - 2*l[0]**2*l[1]**2 - 8*l[0]**2*l[2]**2*nu + 8*l[0]**2*l[2]**2 + 3*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) - 3*l[0]*l[1]**2*l[2]*np.tan(g[1]) + 4*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 + 2*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2*nu - 4*l[1]**2*l[2]**2*np.tan(g[0])**2 - 2*l[1]**2*l[2]**2*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(-18*l[0]**2*nu + 6*l[0]**2 - 3*l[0]*l[1]*nu*np.tan(g[0]) - 3*l[0]*l[1]*np.tan(g[0]) - 18*l[0]*l[2]*nu*np.tan(g[1]) + 6*l[0]*l[2]*np.tan(g[1]) - 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           -E*l[2]*(3*l[0] + 2*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu - 1)),
           E*(l[0]**2*l[1]**2*nu - l[0]**2*l[1]**2 - 4*l[0]**2*l[2]**2*nu + 4*l[0]**2*l[2]**2 - 12*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) + 12*l[0]*l[1]*l[2]**2*np.tan(g[0]) - 4*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - 2*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2*nu + 4*l[1]**2*l[2]**2*np.tan(g[0])**2 + 2*l[1]**2*l[2]**2*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(-9*l[0]**2*nu + 3*l[0]**2 - 9*l[0]*l[1]*nu*np.tan(g[0]) + 3*l[0]*l[1]*np.tan(g[0]) + 6*l[0]*l[2]*nu*np.tan(g[1]) + 6*l[0]*l[2]*np.tan(g[1]) + 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[2]*(-9*l[0]*nu + 3*l[0] - 4*l[1]*nu*np.tan(g[0]) - 4*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(l[0]**2*l[1]**2*nu - l[0]**2*l[1]**2 + 2*l[0]**2*l[2]**2*nu - 2*l[0]**2*l[2]**2 - 4*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - 2*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2*nu + 4*l[1]**2*l[2]**2*np.tan(g[0])**2 + 2*l[1]**2*l[2]**2*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*(-3*l[0]**2*nu - 3*l[0]**2 - 18*l[0]*l[1]*nu*np.tan(g[0]) + 6*l[0]*l[1]*np.tan(g[0]) + 18*l[0]*l[2]*nu*np.tan(g[1]) - 6*l[0]*l[2]*np.tan(g[1]) + 8*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 8*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*l[2]*(3*l[0] + 4*l[1]*np.tan(g[0]))/(72*l[0]*(2*nu - 1)),
           E*(2*l[0]**2*l[1]**2*nu - 2*l[0]**2*l[1]**2 + 4*l[0]**2*l[2]**2*nu - 4*l[0]**2*l[2]**2 + 3*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) - 3*l[0]*l[1]**2*l[2]*np.tan(g[1]) + 6*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) - 6*l[0]*l[1]*l[2]**2*np.tan(g[0]) + 4*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 + 2*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2*nu - 4*l[1]**2*l[2]**2*np.tan(g[0])**2 - 2*l[1]**2*l[2]**2*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           -E*(3*l[0]**2 + 3*l[0]*l[1]*np.tan(g[0]) + 3*l[0]*l[2]*np.tan(g[1]) + 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(72*l[0]*(2*nu - 1))],
          [E*l[1]*(9*l[0]*nu - 3*l[0] + 2*l[2]*nu*np.tan(g[1]) + 2*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(6*l[0]**2*nu + 6*l[0]**2 - 9*l[0]*l[1]*nu*np.tan(g[0]) + 3*l[0]*l[1]*np.tan(g[0]) + 9*l[0]*l[2]*nu*np.tan(g[1]) - 3*l[0]*l[2]*np.tan(g[1]) - 2*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) - 2*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(-4*l[0]**2*l[1]**2*nu + 4*l[0]**2*l[1]**2 - 2*l[0]**2*l[2]**2*nu + 2*l[0]**2*l[2]**2 + l[1]**2*l[2]**2*nu*np.tan(g[0])**2 + 2*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 + l[1]**2*l[2]**2*nu - l[1]**2*l[2]**2*np.tan(g[0])**2 - 2*l[1]**2*l[2]**2*np.tan(g[1])**2 - l[1]**2*l[2]**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           -E*l[1]*(3*l[0] + 2*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*(3*l[0]**2 + 3*l[0]*l[1]*np.tan(g[0]) + 3*l[0]*l[2]*np.tan(g[1]) + 2*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*(3*l[0]**2*(-2*l[1]**2*nu + 2*l[1]**2 - l[2]**2*nu + l[2]**2) + 2*l[0]**2*(2*l[1]**2*nu - 2*l[1]**2 + l[2]**2*nu - l[2]**2) + l[1]*l[2]*(-6*l[0]*l[1]*nu*np.tan(g[1]) + 6*l[0]*l[1]*np.tan(g[1]) - 3*l[0]*l[2]*nu*np.tan(g[0]) + 3*l[0]*l[2]*np.tan(g[0]) - l[1]*l[2]*nu*np.tan(g[0])**2 - 2*l[1]*l[2]*nu*np.tan(g[1])**2 - l[1]*l[2]*nu + l[1]*l[2]*np.tan(g[0])**2 + 2*l[1]*l[2]*np.tan(g[1])**2 + l[1]*l[2]))/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           -E*l[1]*(3*l[0] + 2*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*(-9*l[0]**2*nu + 3*l[0]**2 + 6*l[0]*l[1]*nu*np.tan(g[0]) + 6*l[0]*l[1]*np.tan(g[0]) - 9*l[0]*l[2]*nu*np.tan(g[1]) + 3*l[0]*l[2]*np.tan(g[1]) + 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(-4*l[0]**2*l[1]**2*nu + 4*l[0]**2*l[1]**2 + l[0]**2*l[2]**2*nu - l[0]**2*l[2]**2 - 12*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) + 12*l[0]*l[1]**2*l[2]*np.tan(g[1]) - 2*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - 4*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2*nu + 2*l[1]**2*l[2]**2*np.tan(g[0])**2 + 4*l[1]**2*l[2]**2*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(9*l[0]*nu - 3*l[0] + 2*l[2]*nu*np.tan(g[1]) + 2*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(-18*l[0]**2*nu + 6*l[0]**2 - 18*l[0]*l[1]*nu*np.tan(g[0]) + 6*l[0]*l[1]*np.tan(g[0]) - 3*l[0]*l[2]*nu*np.tan(g[1]) - 3*l[0]*l[2]*np.tan(g[1]) - 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(-8*l[0]**2*l[1]**2*nu + 8*l[0]**2*l[1]**2 + 2*l[0]**2*l[2]**2*nu - 2*l[0]**2*l[2]**2 + 3*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) - 3*l[0]*l[1]*l[2]**2*np.tan(g[0]) + 2*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 + 4*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2*nu - 2*l[1]**2*l[2]**2*np.tan(g[0])**2 - 4*l[1]**2*l[2]**2*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2)/(72*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(3*l[0] + 4*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu - 1)),
           E*(18*l[0]**2*nu - 6*l[0]**2 - 3*l[0]*l[1]*nu*np.tan(g[0]) - 3*l[0]*l[1]*np.tan(g[0]) + 18*l[0]*l[2]*nu*np.tan(g[1]) - 6*l[0]*l[2]*np.tan(g[1]) - 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) - 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*l[1]**2*nu - 2*l[0]**2*l[1]**2 - 2*l[0]**2*l[2]**2*nu + 2*l[0]**2*l[2]**2 + 3*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) - 3*l[0]*l[1]**2*l[2]*np.tan(g[1]) + l[1]**2*l[2]**2*nu*np.tan(g[0])**2 + 2*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 + l[1]**2*l[2]**2*nu - l[1]**2*l[2]**2*np.tan(g[0])**2 - 2*l[1]**2*l[2]**2*np.tan(g[1])**2 - l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(-9*l[0]*nu + 3*l[0] - 4*l[2]*nu*np.tan(g[1]) - 4*l[2]*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(9*l[0]**2*nu - 3*l[0]**2 + 9*l[0]*l[1]*nu*np.tan(g[0]) - 3*l[0]*l[1]*np.tan(g[0]) + 6*l[0]*l[2]*nu*np.tan(g[1]) + 6*l[0]*l[2]*np.tan(g[1]) + 4*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(l[0]**2*l[1]**2*nu - l[0]**2*l[1]**2 - l[0]**2*l[2]**2*nu + l[0]**2*l[2]**2 - 3*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) + 3*l[0]*l[1]*l[2]**2*np.tan(g[0]) - l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - 2*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - l[1]**2*l[2]**2*nu + l[1]**2*l[2]**2*np.tan(g[0])**2 + 2*l[1]**2*l[2]**2*np.tan(g[1])**2 + l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(-9*l[0]*nu + 3*l[0] - 4*l[2]*nu*np.tan(g[1]) - 4*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu**2 + nu - 1)),
           E*(-3*l[0]**2*nu - 3*l[0]**2 + 18*l[0]*l[1]*nu*np.tan(g[0]) - 6*l[0]*l[1]*np.tan(g[0]) - 18*l[0]*l[2]*nu*np.tan(g[1]) + 6*l[0]*l[2]*np.tan(g[1]) + 8*l[1]*l[2]*nu*np.tan(g[0])*np.tan(g[1]) + 8*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(144*l[0]*(2*nu**2 + nu - 1)),
           E*(2*l[0]**2*l[1]**2*nu - 2*l[0]**2*l[1]**2 + l[0]**2*l[2]**2*nu - l[0]**2*l[2]**2 - 2*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 - 4*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2*nu + 2*l[1]**2*l[2]**2*np.tan(g[0])**2 + 4*l[1]**2*l[2]**2*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1)),
           E*l[1]*(3*l[0] + 4*l[2]*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           -E*(3*l[0]**2 + 3*l[0]*l[1]*np.tan(g[0]) + 3*l[0]*l[2]*np.tan(g[1]) + 4*l[1]*l[2]*np.tan(g[0])*np.tan(g[1]))/(72*l[0]*(2*nu - 1)),
           E*(4*l[0]**2*l[1]**2*nu - 4*l[0]**2*l[1]**2 + 2*l[0]**2*l[2]**2*nu - 2*l[0]**2*l[2]**2 + 6*l[0]*l[1]**2*l[2]*nu*np.tan(g[1]) - 6*l[0]*l[1]**2*l[2]*np.tan(g[1]) + 3*l[0]*l[1]*l[2]**2*nu*np.tan(g[0]) - 3*l[0]*l[1]*l[2]**2*np.tan(g[0]) + 2*l[1]**2*l[2]**2*nu*np.tan(g[0])**2 + 4*l[1]**2*l[2]**2*nu*np.tan(g[1])**2 + 2*l[1]**2*l[2]**2*nu - 2*l[1]**2*l[2]**2*np.tan(g[0])**2 - 4*l[1]**2*l[2]**2*np.tan(g[1])**2 - 2*l[1]**2*l[2]**2)/(36*l[0]*l[1]*l[2]*(2*nu**2 + nu - 1))]])

def lk_linear_elast_aniso_3d(c: np.ndarray,
                             l: np.ndarray = np.array([1.,1.,1.]), 
                             g: np.ndarray = np.array([0.,0.]),
                             **kwargs: Any) -> np.ndarray:
    """
    Create element stiffness matrix for 3D anisotropic linear elasticity with
    hexahedron elements.

    Parameters
    ----------
    c : np.ndarray, shape (6,6)
    l : np.ndarray (3)
        side length of element.
    g : np.ndarray (2)
        angles of parallelogram.

    Returns
    -------
    Ke : np.ndarray, shape (24,24)
        element stiffness matrix.

    """
    return np.array([[c[0,0]*l[1]*l[2]/(9*l[0]) + c[0,4]*l[1]/12 - c[0,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[0,5]*l[2]/12 - c[0,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[4,0]*l[1]/12 - c[4,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[4,4]*l[0]*l[1]/(9*l[2]) - c[4,4]*l[1]*np.tan(g[1])/6 + c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[4,5]*l[0]/12 - c[4,5]*l[1]*np.tan(g[0])/12 - c[4,5]*l[2]*np.tan(g[1])/12 + c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[5,0]*l[2]/12 - c[5,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[5,4]*l[0]/12 - c[5,4]*l[1]*np.tan(g[0])/12 - c[5,4]*l[2]*np.tan(g[1])/12 + c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[5,5]*l[0]*l[2]/(9*l[1]) - c[5,5]*l[2]*np.tan(g[0])/6 + c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]),
           c[0,1]*l[2]/12 - c[0,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[0,3]*l[1]/12 - c[0,3]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[0,5]*l[1]*l[2]/(9*l[0]) + c[4,1]*l[0]/12 - c[4,1]*l[1]*np.tan(g[0])/12 - c[4,1]*l[2]*np.tan(g[1])/12 + c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[4,3]*l[0]*l[1]/(9*l[2]) - c[4,3]*l[1]*np.tan(g[1])/6 + c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[4,5]*l[1]/12 - c[4,5]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,1]*l[0]*l[2]/(9*l[1]) - c[5,1]*l[2]*np.tan(g[0])/6 + c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[5,3]*l[0]/12 - c[5,3]*l[1]*np.tan(g[0])/12 - c[5,3]*l[2]*np.tan(g[1])/12 + c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[5,5]*l[2]/12 - c[5,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           c[0,2]*l[1]/12 - c[0,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[0,3]*l[2]/12 - c[0,3]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[0,4]*l[1]*l[2]/(9*l[0]) + c[4,2]*l[0]*l[1]/(9*l[2]) - c[4,2]*l[1]*np.tan(g[1])/6 + c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[4,3]*l[0]/12 - c[4,3]*l[1]*np.tan(g[0])/12 - c[4,3]*l[2]*np.tan(g[1])/12 + c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[4,4]*l[1]/12 - c[4,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,2]*l[0]/12 - c[5,2]*l[1]*np.tan(g[0])/12 - c[5,2]*l[2]*np.tan(g[1])/12 + c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[5,3]*l[0]*l[2]/(9*l[1]) - c[5,3]*l[2]*np.tan(g[0])/6 + c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[5,4]*l[2]/12 - c[5,4]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           -c[0,0]*l[1]*l[2]/(9*l[0]) + c[0,4]*l[1]/12 + c[0,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[0,5]*l[2]/12 + c[0,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[4,0]*l[1]/12 + c[4,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[4,4]*l[0]*l[1]/(18*l[2]) - c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[4,5]*l[0]/24 + c[4,5]*l[1]*np.tan(g[0])/12 - c[4,5]*l[2]*np.tan(g[1])/12 - c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[5,0]*l[2]/12 + c[5,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[5,4]*l[0]/24 - c[5,4]*l[1]*np.tan(g[0])/12 + c[5,4]*l[2]*np.tan(g[1])/12 - c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[5,5]*l[0]*l[2]/(18*l[1]) - c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]),
           c[0,1]*l[2]/12 + c[0,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[0,3]*l[1]/12 + c[0,3]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[0,5]*l[1]*l[2]/(9*l[0]) + c[4,1]*l[0]/24 + c[4,1]*l[1]*np.tan(g[0])/12 - c[4,1]*l[2]*np.tan(g[1])/12 - c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[4,3]*l[0]*l[1]/(18*l[2]) - c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[4,5]*l[1]/12 + c[4,5]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,1]*l[0]*l[2]/(18*l[1]) - c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[5,3]*l[0]/24 - c[5,3]*l[1]*np.tan(g[0])/12 + c[5,3]*l[2]*np.tan(g[1])/12 - c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[5,5]*l[2]/12 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           c[0,2]*l[1]/12 + c[0,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[0,3]*l[2]/12 + c[0,3]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[0,4]*l[1]*l[2]/(9*l[0]) + c[4,2]*l[0]*l[1]/(18*l[2]) - c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[4,3]*l[0]/24 + c[4,3]*l[1]*np.tan(g[0])/12 - c[4,3]*l[2]*np.tan(g[1])/12 - c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[4,4]*l[1]/12 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,2]*l[0]/24 - c[5,2]*l[1]*np.tan(g[0])/12 + c[5,2]*l[2]*np.tan(g[1])/12 - c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[5,3]*l[0]*l[2]/(18*l[1]) - c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[5,4]*l[2]/12 + c[5,4]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           -c[0,0]*l[1]*l[2]/(18*l[0]) + c[0,4]*l[1]/24 + c[0,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[0,5]*l[2]/12 + c[0,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,0]*l[1]/24 + c[4,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,4]*l[0]*l[1]/(36*l[2]) - c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[4,5]*l[0]/24 + c[4,5]*l[1]*np.tan(g[0])/24 + c[4,5]*l[2]*np.tan(g[1])/12 - c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,0]*l[2]/12 + c[5,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[5,4]*l[0]/24 - c[5,4]*l[1]*np.tan(g[0])/24 + c[5,4]*l[2]*np.tan(g[1])/12 - c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,5]*l[0]*l[2]/(18*l[1]) + c[5,5]*l[2]*np.tan(g[0])/6 - c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]),
           -c[0,1]*l[2]/12 + c[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[0,3]*l[1]/24 + c[0,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[0,5]*l[1]*l[2]/(18*l[0]) - c[4,1]*l[0]/24 + c[4,1]*l[1]*np.tan(g[0])/24 + c[4,1]*l[2]*np.tan(g[1])/12 - c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[4,3]*l[0]*l[1]/(36*l[2]) - c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[4,5]*l[1]/24 + c[4,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,1]*l[0]*l[2]/(18*l[1]) + c[5,1]*l[2]*np.tan(g[0])/6 - c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[5,3]*l[0]/24 - c[5,3]*l[1]*np.tan(g[0])/24 + c[5,3]*l[2]*np.tan(g[1])/12 - c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,5]*l[2]/12 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[0,2]*l[1]/24 + c[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[0,3]*l[2]/12 + c[0,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[0,4]*l[1]*l[2]/(18*l[0]) + c[4,2]*l[0]*l[1]/(36*l[2]) - c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[4,3]*l[0]/24 + c[4,3]*l[1]*np.tan(g[0])/24 + c[4,3]*l[2]*np.tan(g[1])/12 - c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[4,4]*l[1]/24 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,2]*l[0]/24 - c[5,2]*l[1]*np.tan(g[0])/24 + c[5,2]*l[2]*np.tan(g[1])/12 - c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,3]*l[0]*l[2]/(18*l[1]) + c[5,3]*l[2]*np.tan(g[0])/6 - c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[5,4]*l[2]/12 + c[5,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[0,0]*l[1]*l[2]/(18*l[0]) + c[0,4]*l[1]/24 - c[0,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[0,5]*l[2]/12 - c[0,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,0]*l[1]/24 - c[4,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,4]*l[0]*l[1]/(18*l[2]) - c[4,4]*l[1]*np.tan(g[1])/12 + c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[4,5]*l[0]/12 - c[4,5]*l[1]*np.tan(g[0])/24 + c[4,5]*l[2]*np.tan(g[1])/12 + c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,0]*l[2]/12 - c[5,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[5,4]*l[0]/12 - c[5,4]*l[1]*np.tan(g[0])/24 - c[5,4]*l[2]*np.tan(g[1])/12 + c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,5]*l[0]*l[2]/(9*l[1]) + c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]),
           -c[0,1]*l[2]/12 - c[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[0,3]*l[1]/24 - c[0,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[0,5]*l[1]*l[2]/(18*l[0]) - c[4,1]*l[0]/12 - c[4,1]*l[1]*np.tan(g[0])/24 + c[4,1]*l[2]*np.tan(g[1])/12 + c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[4,3]*l[0]*l[1]/(18*l[2]) - c[4,3]*l[1]*np.tan(g[1])/12 + c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[4,5]*l[1]/24 - c[4,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,1]*l[0]*l[2]/(9*l[1]) + c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[5,3]*l[0]/12 - c[5,3]*l[1]*np.tan(g[0])/24 - c[5,3]*l[2]*np.tan(g[1])/12 + c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,5]*l[2]/12 - c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[0,2]*l[1]/24 - c[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[0,3]*l[2]/12 - c[0,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[0,4]*l[1]*l[2]/(18*l[0]) + c[4,2]*l[0]*l[1]/(18*l[2]) - c[4,2]*l[1]*np.tan(g[1])/12 + c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[4,3]*l[0]/12 - c[4,3]*l[1]*np.tan(g[0])/24 + c[4,3]*l[2]*np.tan(g[1])/12 + c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[4,4]*l[1]/24 - c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,2]*l[0]/12 - c[5,2]*l[1]*np.tan(g[0])/24 - c[5,2]*l[2]*np.tan(g[1])/12 + c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,3]*l[0]*l[2]/(9*l[1]) + c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[5,4]*l[2]/12 - c[5,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[0,0]*l[1]*l[2]/(18*l[0]) - c[0,4]*l[1]/12 - c[0,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[0,5]*l[2]/24 - c[0,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,0]*l[1]/12 - c[4,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,4]*l[0]*l[1]/(9*l[2]) + c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[4,5]*l[0]/12 - c[4,5]*l[1]*np.tan(g[0])/12 - c[4,5]*l[2]*np.tan(g[1])/24 + c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,0]*l[2]/24 - c[5,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[5,4]*l[0]/12 + c[5,4]*l[1]*np.tan(g[0])/12 - c[5,4]*l[2]*np.tan(g[1])/24 + c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,5]*l[0]*l[2]/(18*l[1]) - c[5,5]*l[2]*np.tan(g[0])/12 + c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]),
           c[0,1]*l[2]/24 - c[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[0,3]*l[1]/12 - c[0,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[0,5]*l[1]*l[2]/(18*l[0]) + c[4,1]*l[0]/12 - c[4,1]*l[1]*np.tan(g[0])/12 - c[4,1]*l[2]*np.tan(g[1])/24 + c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[4,3]*l[0]*l[1]/(9*l[2]) + c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[4,5]*l[1]/12 - c[4,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,1]*l[0]*l[2]/(18*l[1]) - c[5,1]*l[2]*np.tan(g[0])/12 + c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[5,3]*l[0]/12 + c[5,3]*l[1]*np.tan(g[0])/12 - c[5,3]*l[2]*np.tan(g[1])/24 + c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,5]*l[2]/24 - c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[0,2]*l[1]/12 - c[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[0,3]*l[2]/24 - c[0,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[0,4]*l[1]*l[2]/(18*l[0]) - c[4,2]*l[0]*l[1]/(9*l[2]) + c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[4,3]*l[0]/12 - c[4,3]*l[1]*np.tan(g[0])/12 - c[4,3]*l[2]*np.tan(g[1])/24 + c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[4,4]*l[1]/12 - c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,2]*l[0]/12 + c[5,2]*l[1]*np.tan(g[0])/12 - c[5,2]*l[2]*np.tan(g[1])/24 + c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,3]*l[0]*l[2]/(18*l[1]) - c[5,3]*l[2]*np.tan(g[0])/12 + c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[5,4]*l[2]/24 - c[5,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[0,0]*l[1]*l[2]/(18*l[0]) - c[0,4]*l[1]/12 + c[0,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[0,5]*l[2]/24 + c[0,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,0]*l[1]/12 + c[4,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,4]*l[0]*l[1]/(18*l[2]) + c[4,4]*l[1]*np.tan(g[1])/6 - c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[4,5]*l[0]/24 + c[4,5]*l[1]*np.tan(g[0])/12 - c[4,5]*l[2]*np.tan(g[1])/24 - c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,0]*l[2]/24 + c[5,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[5,4]*l[0]/24 + c[5,4]*l[1]*np.tan(g[0])/12 + c[5,4]*l[2]*np.tan(g[1])/24 - c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,5]*l[0]*l[2]/(36*l[1]) - c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]),
           c[0,1]*l[2]/24 + c[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[0,3]*l[1]/12 + c[0,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[0,5]*l[1]*l[2]/(18*l[0]) + c[4,1]*l[0]/24 + c[4,1]*l[1]*np.tan(g[0])/12 - c[4,1]*l[2]*np.tan(g[1])/24 - c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[4,3]*l[0]*l[1]/(18*l[2]) + c[4,3]*l[1]*np.tan(g[1])/6 - c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[4,5]*l[1]/12 + c[4,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,1]*l[0]*l[2]/(36*l[1]) - c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[5,3]*l[0]/24 + c[5,3]*l[1]*np.tan(g[0])/12 + c[5,3]*l[2]*np.tan(g[1])/24 - c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,5]*l[2]/24 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[0,2]*l[1]/12 + c[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[0,3]*l[2]/24 + c[0,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[0,4]*l[1]*l[2]/(18*l[0]) - c[4,2]*l[0]*l[1]/(18*l[2]) + c[4,2]*l[1]*np.tan(g[1])/6 - c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[4,3]*l[0]/24 + c[4,3]*l[1]*np.tan(g[0])/12 - c[4,3]*l[2]*np.tan(g[1])/24 - c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[4,4]*l[1]/12 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,2]*l[0]/24 + c[5,2]*l[1]*np.tan(g[0])/12 + c[5,2]*l[2]*np.tan(g[1])/24 - c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,3]*l[0]*l[2]/(36*l[1]) - c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[5,4]*l[2]/24 + c[5,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[0,0]*l[1]*l[2]/(36*l[0]) - c[0,4]*l[1]/24 + c[0,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[0,5]*l[2]/24 + c[0,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[4,0]*l[1]/24 + c[4,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[4,4]*l[0]*l[1]/(36*l[2]) + c[4,4]*l[1]*np.tan(g[1])/12 - c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[4,5]*l[0]/24 + c[4,5]*l[1]*np.tan(g[0])/24 + c[4,5]*l[2]*np.tan(g[1])/24 - c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,0]*l[2]/24 + c[5,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[5,4]*l[0]/24 + c[5,4]*l[1]*np.tan(g[0])/24 + c[5,4]*l[2]*np.tan(g[1])/24 - c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,5]*l[0]*l[2]/(36*l[1]) + c[5,5]*l[2]*np.tan(g[0])/12 - c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]),
           -c[0,1]*l[2]/24 + c[0,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[0,3]*l[1]/24 + c[0,3]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[0,5]*l[1]*l[2]/(36*l[0]) - c[4,1]*l[0]/24 + c[4,1]*l[1]*np.tan(g[0])/24 + c[4,1]*l[2]*np.tan(g[1])/24 - c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[4,3]*l[0]*l[1]/(36*l[2]) + c[4,3]*l[1]*np.tan(g[1])/12 - c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[4,5]*l[1]/24 + c[4,5]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,1]*l[0]*l[2]/(36*l[1]) + c[5,1]*l[2]*np.tan(g[0])/12 - c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[5,3]*l[0]/24 + c[5,3]*l[1]*np.tan(g[0])/24 + c[5,3]*l[2]*np.tan(g[1])/24 - c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,5]*l[2]/24 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           -c[0,2]*l[1]/24 + c[0,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[0,3]*l[2]/24 + c[0,3]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[0,4]*l[1]*l[2]/(36*l[0]) - c[4,2]*l[0]*l[1]/(36*l[2]) + c[4,2]*l[1]*np.tan(g[1])/12 - c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[4,3]*l[0]/24 + c[4,3]*l[1]*np.tan(g[0])/24 + c[4,3]*l[2]*np.tan(g[1])/24 - c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[4,4]*l[1]/24 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,2]*l[0]/24 + c[5,2]*l[1]*np.tan(g[0])/24 + c[5,2]*l[2]*np.tan(g[1])/24 - c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,3]*l[0]*l[2]/(36*l[1]) + c[5,3]*l[2]*np.tan(g[0])/12 - c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[5,4]*l[2]/24 + c[5,4]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           c[0,0]*l[1]*l[2]/(36*l[0]) - c[0,4]*l[1]/24 - c[0,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[0,5]*l[2]/24 - c[0,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[4,0]*l[1]/24 - c[4,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[4,4]*l[0]*l[1]/(18*l[2]) + c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[4,5]*l[0]/12 - c[4,5]*l[1]*np.tan(g[0])/24 + c[4,5]*l[2]*np.tan(g[1])/24 + c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[5,0]*l[2]/24 - c[5,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[5,4]*l[0]/12 + c[5,4]*l[1]*np.tan(g[0])/24 - c[5,4]*l[2]*np.tan(g[1])/24 + c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,5]*l[0]*l[2]/(18*l[1]) + c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]),
           -c[0,1]*l[2]/24 - c[0,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[0,3]*l[1]/24 - c[0,3]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[0,5]*l[1]*l[2]/(36*l[0]) - c[4,1]*l[0]/12 - c[4,1]*l[1]*np.tan(g[0])/24 + c[4,1]*l[2]*np.tan(g[1])/24 + c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[4,3]*l[0]*l[1]/(18*l[2]) + c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[4,5]*l[1]/24 - c[4,5]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,1]*l[0]*l[2]/(18*l[1]) + c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[5,3]*l[0]/12 + c[5,3]*l[1]*np.tan(g[0])/24 - c[5,3]*l[2]*np.tan(g[1])/24 + c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[5,5]*l[2]/24 - c[5,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           -c[0,2]*l[1]/24 - c[0,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[0,3]*l[2]/24 - c[0,3]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[0,4]*l[1]*l[2]/(36*l[0]) - c[4,2]*l[0]*l[1]/(18*l[2]) + c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[4,3]*l[0]/12 - c[4,3]*l[1]*np.tan(g[0])/24 + c[4,3]*l[2]*np.tan(g[1])/24 + c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[4,4]*l[1]/24 - c[4,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,2]*l[0]/12 + c[5,2]*l[1]*np.tan(g[0])/24 - c[5,2]*l[2]*np.tan(g[1])/24 + c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,3]*l[0]*l[2]/(18*l[1]) + c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[5,4]*l[2]/24 - c[5,4]*l[1]*l[2]*np.tan(g[0])/(36*l[0])],
          [c[1,0]*l[2]/12 - c[1,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[1,4]*l[0]/12 - c[1,4]*l[1]*np.tan(g[0])/12 - c[1,4]*l[2]*np.tan(g[1])/12 + c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[1,5]*l[0]*l[2]/(9*l[1]) - c[1,5]*l[2]*np.tan(g[0])/6 + c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[3,0]*l[1]/12 - c[3,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[3,4]*l[0]*l[1]/(9*l[2]) - c[3,4]*l[1]*np.tan(g[1])/6 + c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[3,5]*l[0]/12 - c[3,5]*l[1]*np.tan(g[0])/12 - c[3,5]*l[2]*np.tan(g[1])/12 + c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[5,0]*l[1]*l[2]/(9*l[0]) + c[5,4]*l[1]/12 - c[5,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,5]*l[2]/12 - c[5,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           c[1,1]*l[0]*l[2]/(9*l[1]) - c[1,1]*l[2]*np.tan(g[0])/6 + c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[1,3]*l[0]/12 - c[1,3]*l[1]*np.tan(g[0])/12 - c[1,3]*l[2]*np.tan(g[1])/12 + c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[1,5]*l[2]/12 - c[1,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[3,1]*l[0]/12 - c[3,1]*l[1]*np.tan(g[0])/12 - c[3,1]*l[2]*np.tan(g[1])/12 + c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,3]*l[0]*l[1]/(9*l[2]) - c[3,3]*l[1]*np.tan(g[1])/6 + c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[3,5]*l[1]/12 - c[3,5]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,1]*l[2]/12 - c[5,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[5,3]*l[1]/12 - c[5,3]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,5]*l[1]*l[2]/(9*l[0]),
           c[1,2]*l[0]/12 - c[1,2]*l[1]*np.tan(g[0])/12 - c[1,2]*l[2]*np.tan(g[1])/12 + c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[1,3]*l[0]*l[2]/(9*l[1]) - c[1,3]*l[2]*np.tan(g[0])/6 + c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[1,4]*l[2]/12 - c[1,4]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[3,2]*l[0]*l[1]/(9*l[2]) - c[3,2]*l[1]*np.tan(g[1])/6 + c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[3,3]*l[0]/12 - c[3,3]*l[1]*np.tan(g[0])/12 - c[3,3]*l[2]*np.tan(g[1])/12 + c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,4]*l[1]/12 - c[3,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,2]*l[1]/12 - c[5,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,3]*l[2]/12 - c[5,3]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[5,4]*l[1]*l[2]/(9*l[0]),
           -c[1,0]*l[2]/12 + c[1,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[1,4]*l[0]/24 - c[1,4]*l[1]*np.tan(g[0])/12 + c[1,4]*l[2]*np.tan(g[1])/12 - c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[1,5]*l[0]*l[2]/(18*l[1]) - c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[3,0]*l[1]/12 + c[3,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[3,4]*l[0]*l[1]/(18*l[2]) - c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[3,5]*l[0]/24 + c[3,5]*l[1]*np.tan(g[0])/12 - c[3,5]*l[2]*np.tan(g[1])/12 - c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[5,0]*l[1]*l[2]/(9*l[0]) + c[5,4]*l[1]/12 + c[5,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,5]*l[2]/12 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           c[1,1]*l[0]*l[2]/(18*l[1]) - c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[1,3]*l[0]/24 - c[1,3]*l[1]*np.tan(g[0])/12 + c[1,3]*l[2]*np.tan(g[1])/12 - c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[1,5]*l[2]/12 + c[1,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[3,1]*l[0]/24 + c[3,1]*l[1]*np.tan(g[0])/12 - c[3,1]*l[2]*np.tan(g[1])/12 - c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,3]*l[0]*l[1]/(18*l[2]) - c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[3,5]*l[1]/12 + c[3,5]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,1]*l[2]/12 + c[5,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[5,3]*l[1]/12 + c[5,3]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[5,5]*l[1]*l[2]/(9*l[0]),
           c[1,2]*l[0]/24 - c[1,2]*l[1]*np.tan(g[0])/12 + c[1,2]*l[2]*np.tan(g[1])/12 - c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[1,3]*l[0]*l[2]/(18*l[1]) - c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[1,4]*l[2]/12 + c[1,4]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[3,2]*l[0]*l[1]/(18*l[2]) - c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[3,3]*l[0]/24 + c[3,3]*l[1]*np.tan(g[0])/12 - c[3,3]*l[2]*np.tan(g[1])/12 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[3,4]*l[1]/12 + c[3,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,2]*l[1]/12 + c[5,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,3]*l[2]/12 + c[5,3]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[5,4]*l[1]*l[2]/(9*l[0]),
           -c[1,0]*l[2]/12 + c[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[1,4]*l[0]/24 - c[1,4]*l[1]*np.tan(g[0])/24 + c[1,4]*l[2]*np.tan(g[1])/12 - c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[1,5]*l[0]*l[2]/(18*l[1]) + c[1,5]*l[2]*np.tan(g[0])/6 - c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[3,0]*l[1]/24 + c[3,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[3,4]*l[0]*l[1]/(36*l[2]) - c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[3,5]*l[0]/24 + c[3,5]*l[1]*np.tan(g[0])/24 + c[3,5]*l[2]*np.tan(g[1])/12 - c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,0]*l[1]*l[2]/(18*l[0]) + c[5,4]*l[1]/24 + c[5,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,5]*l[2]/12 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[1,1]*l[0]*l[2]/(18*l[1]) + c[1,1]*l[2]*np.tan(g[0])/6 - c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[1,3]*l[0]/24 - c[1,3]*l[1]*np.tan(g[0])/24 + c[1,3]*l[2]*np.tan(g[1])/12 - c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[1,5]*l[2]/12 + c[1,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[3,1]*l[0]/24 + c[3,1]*l[1]*np.tan(g[0])/24 + c[3,1]*l[2]*np.tan(g[1])/12 - c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,3]*l[0]*l[1]/(36*l[2]) - c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[3,5]*l[1]/24 + c[3,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,1]*l[2]/12 + c[5,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[5,3]*l[1]/24 + c[5,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,5]*l[1]*l[2]/(18*l[0]),
           c[1,2]*l[0]/24 - c[1,2]*l[1]*np.tan(g[0])/24 + c[1,2]*l[2]*np.tan(g[1])/12 - c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[1,3]*l[0]*l[2]/(18*l[1]) + c[1,3]*l[2]*np.tan(g[0])/6 - c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[1,4]*l[2]/12 + c[1,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[3,2]*l[0]*l[1]/(36*l[2]) - c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[3,3]*l[0]/24 + c[3,3]*l[1]*np.tan(g[0])/24 + c[3,3]*l[2]*np.tan(g[1])/12 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,4]*l[1]/24 + c[3,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,2]*l[1]/24 + c[5,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,3]*l[2]/12 + c[5,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[5,4]*l[1]*l[2]/(18*l[0]),
           c[1,0]*l[2]/12 - c[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[1,4]*l[0]/12 - c[1,4]*l[1]*np.tan(g[0])/24 - c[1,4]*l[2]*np.tan(g[1])/12 + c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[1,5]*l[0]*l[2]/(9*l[1]) + c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[3,0]*l[1]/24 - c[3,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[3,4]*l[0]*l[1]/(18*l[2]) - c[3,4]*l[1]*np.tan(g[1])/12 + c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[3,5]*l[0]/12 - c[3,5]*l[1]*np.tan(g[0])/24 + c[3,5]*l[2]*np.tan(g[1])/12 + c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,0]*l[1]*l[2]/(18*l[0]) + c[5,4]*l[1]/24 - c[5,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,5]*l[2]/12 - c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[1,1]*l[0]*l[2]/(9*l[1]) + c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[1,3]*l[0]/12 - c[1,3]*l[1]*np.tan(g[0])/24 - c[1,3]*l[2]*np.tan(g[1])/12 + c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[1,5]*l[2]/12 - c[1,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[3,1]*l[0]/12 - c[3,1]*l[1]*np.tan(g[0])/24 + c[3,1]*l[2]*np.tan(g[1])/12 + c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,3]*l[0]*l[1]/(18*l[2]) - c[3,3]*l[1]*np.tan(g[1])/12 + c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[3,5]*l[1]/24 - c[3,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,1]*l[2]/12 - c[5,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[5,3]*l[1]/24 - c[5,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,5]*l[1]*l[2]/(18*l[0]),
           c[1,2]*l[0]/12 - c[1,2]*l[1]*np.tan(g[0])/24 - c[1,2]*l[2]*np.tan(g[1])/12 + c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[1,3]*l[0]*l[2]/(9*l[1]) + c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[1,4]*l[2]/12 - c[1,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[3,2]*l[0]*l[1]/(18*l[2]) - c[3,2]*l[1]*np.tan(g[1])/12 + c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[3,3]*l[0]/12 - c[3,3]*l[1]*np.tan(g[0])/24 + c[3,3]*l[2]*np.tan(g[1])/12 + c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,4]*l[1]/24 - c[3,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,2]*l[1]/24 - c[5,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,3]*l[2]/12 - c[5,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[5,4]*l[1]*l[2]/(18*l[0]),
           c[1,0]*l[2]/24 - c[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[1,4]*l[0]/12 + c[1,4]*l[1]*np.tan(g[0])/12 - c[1,4]*l[2]*np.tan(g[1])/24 + c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[1,5]*l[0]*l[2]/(18*l[1]) - c[1,5]*l[2]*np.tan(g[0])/12 + c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[3,0]*l[1]/12 - c[3,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[3,4]*l[0]*l[1]/(9*l[2]) + c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[3,5]*l[0]/12 - c[3,5]*l[1]*np.tan(g[0])/12 - c[3,5]*l[2]*np.tan(g[1])/24 + c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,0]*l[1]*l[2]/(18*l[0]) - c[5,4]*l[1]/12 - c[5,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,5]*l[2]/24 - c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[1,1]*l[0]*l[2]/(18*l[1]) - c[1,1]*l[2]*np.tan(g[0])/12 + c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[1,3]*l[0]/12 + c[1,3]*l[1]*np.tan(g[0])/12 - c[1,3]*l[2]*np.tan(g[1])/24 + c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[1,5]*l[2]/24 - c[1,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[3,1]*l[0]/12 - c[3,1]*l[1]*np.tan(g[0])/12 - c[3,1]*l[2]*np.tan(g[1])/24 + c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,3]*l[0]*l[1]/(9*l[2]) + c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[3,5]*l[1]/12 - c[3,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,1]*l[2]/24 - c[5,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[5,3]*l[1]/12 - c[5,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,5]*l[1]*l[2]/(18*l[0]),
           -c[1,2]*l[0]/12 + c[1,2]*l[1]*np.tan(g[0])/12 - c[1,2]*l[2]*np.tan(g[1])/24 + c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[1,3]*l[0]*l[2]/(18*l[1]) - c[1,3]*l[2]*np.tan(g[0])/12 + c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[1,4]*l[2]/24 - c[1,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[3,2]*l[0]*l[1]/(9*l[2]) + c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[3,3]*l[0]/12 - c[3,3]*l[1]*np.tan(g[0])/12 - c[3,3]*l[2]*np.tan(g[1])/24 + c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,4]*l[1]/12 - c[3,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,2]*l[1]/12 - c[5,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,3]*l[2]/24 - c[5,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[5,4]*l[1]*l[2]/(18*l[0]),
           -c[1,0]*l[2]/24 + c[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[1,4]*l[0]/24 + c[1,4]*l[1]*np.tan(g[0])/12 + c[1,4]*l[2]*np.tan(g[1])/24 - c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[1,5]*l[0]*l[2]/(36*l[1]) - c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[3,0]*l[1]/12 + c[3,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[3,4]*l[0]*l[1]/(18*l[2]) + c[3,4]*l[1]*np.tan(g[1])/6 - c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[3,5]*l[0]/24 + c[3,5]*l[1]*np.tan(g[0])/12 - c[3,5]*l[2]*np.tan(g[1])/24 - c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,0]*l[1]*l[2]/(18*l[0]) - c[5,4]*l[1]/12 + c[5,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,5]*l[2]/24 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[1,1]*l[0]*l[2]/(36*l[1]) - c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[1,3]*l[0]/24 + c[1,3]*l[1]*np.tan(g[0])/12 + c[1,3]*l[2]*np.tan(g[1])/24 - c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[1,5]*l[2]/24 + c[1,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[3,1]*l[0]/24 + c[3,1]*l[1]*np.tan(g[0])/12 - c[3,1]*l[2]*np.tan(g[1])/24 - c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,3]*l[0]*l[1]/(18*l[2]) + c[3,3]*l[1]*np.tan(g[1])/6 - c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[3,5]*l[1]/12 + c[3,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,1]*l[2]/24 + c[5,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[5,3]*l[1]/12 + c[5,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,5]*l[1]*l[2]/(18*l[0]),
           -c[1,2]*l[0]/24 + c[1,2]*l[1]*np.tan(g[0])/12 + c[1,2]*l[2]*np.tan(g[1])/24 - c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[1,3]*l[0]*l[2]/(36*l[1]) - c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[1,4]*l[2]/24 + c[1,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[3,2]*l[0]*l[1]/(18*l[2]) + c[3,2]*l[1]*np.tan(g[1])/6 - c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[3,3]*l[0]/24 + c[3,3]*l[1]*np.tan(g[0])/12 - c[3,3]*l[2]*np.tan(g[1])/24 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,4]*l[1]/12 + c[3,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,2]*l[1]/12 + c[5,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,3]*l[2]/24 + c[5,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[5,4]*l[1]*l[2]/(18*l[0]),
           -c[1,0]*l[2]/24 + c[1,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[1,4]*l[0]/24 + c[1,4]*l[1]*np.tan(g[0])/24 + c[1,4]*l[2]*np.tan(g[1])/24 - c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[1,5]*l[0]*l[2]/(36*l[1]) + c[1,5]*l[2]*np.tan(g[0])/12 - c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[3,0]*l[1]/24 + c[3,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[3,4]*l[0]*l[1]/(36*l[2]) + c[3,4]*l[1]*np.tan(g[1])/12 - c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[3,5]*l[0]/24 + c[3,5]*l[1]*np.tan(g[0])/24 + c[3,5]*l[2]*np.tan(g[1])/24 - c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,0]*l[1]*l[2]/(36*l[0]) - c[5,4]*l[1]/24 + c[5,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,5]*l[2]/24 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           -c[1,1]*l[0]*l[2]/(36*l[1]) + c[1,1]*l[2]*np.tan(g[0])/12 - c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[1,3]*l[0]/24 + c[1,3]*l[1]*np.tan(g[0])/24 + c[1,3]*l[2]*np.tan(g[1])/24 - c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[1,5]*l[2]/24 + c[1,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[3,1]*l[0]/24 + c[3,1]*l[1]*np.tan(g[0])/24 + c[3,1]*l[2]*np.tan(g[1])/24 - c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,3]*l[0]*l[1]/(36*l[2]) + c[3,3]*l[1]*np.tan(g[1])/12 - c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[3,5]*l[1]/24 + c[3,5]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,1]*l[2]/24 + c[5,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[5,3]*l[1]/24 + c[5,3]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,5]*l[1]*l[2]/(36*l[0]),
           -c[1,2]*l[0]/24 + c[1,2]*l[1]*np.tan(g[0])/24 + c[1,2]*l[2]*np.tan(g[1])/24 - c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[1,3]*l[0]*l[2]/(36*l[1]) + c[1,3]*l[2]*np.tan(g[0])/12 - c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[1,4]*l[2]/24 + c[1,4]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[3,2]*l[0]*l[1]/(36*l[2]) + c[3,2]*l[1]*np.tan(g[1])/12 - c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[3,3]*l[0]/24 + c[3,3]*l[1]*np.tan(g[0])/24 + c[3,3]*l[2]*np.tan(g[1])/24 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,4]*l[1]/24 + c[3,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,2]*l[1]/24 + c[5,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,3]*l[2]/24 + c[5,3]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[5,4]*l[1]*l[2]/(36*l[0]),
           c[1,0]*l[2]/24 - c[1,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[1,4]*l[0]/12 + c[1,4]*l[1]*np.tan(g[0])/24 - c[1,4]*l[2]*np.tan(g[1])/24 + c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[1,5]*l[0]*l[2]/(18*l[1]) + c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[3,0]*l[1]/24 - c[3,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[3,4]*l[0]*l[1]/(18*l[2]) + c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[3,5]*l[0]/12 - c[3,5]*l[1]*np.tan(g[0])/24 + c[3,5]*l[2]*np.tan(g[1])/24 + c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[5,0]*l[1]*l[2]/(36*l[0]) - c[5,4]*l[1]/24 - c[5,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,5]*l[2]/24 - c[5,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           -c[1,1]*l[0]*l[2]/(18*l[1]) + c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[1,3]*l[0]/12 + c[1,3]*l[1]*np.tan(g[0])/24 - c[1,3]*l[2]*np.tan(g[1])/24 + c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[1,5]*l[2]/24 - c[1,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[3,1]*l[0]/12 - c[3,1]*l[1]*np.tan(g[0])/24 + c[3,1]*l[2]*np.tan(g[1])/24 + c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,3]*l[0]*l[1]/(18*l[2]) + c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[3,5]*l[1]/24 - c[3,5]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,1]*l[2]/24 - c[5,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[5,3]*l[1]/24 - c[5,3]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[5,5]*l[1]*l[2]/(36*l[0]),
           -c[1,2]*l[0]/12 + c[1,2]*l[1]*np.tan(g[0])/24 - c[1,2]*l[2]*np.tan(g[1])/24 + c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[1,3]*l[0]*l[2]/(18*l[1]) + c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[1,4]*l[2]/24 - c[1,4]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[3,2]*l[0]*l[1]/(18*l[2]) + c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[3,3]*l[0]/12 - c[3,3]*l[1]*np.tan(g[0])/24 + c[3,3]*l[2]*np.tan(g[1])/24 + c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[3,4]*l[1]/24 - c[3,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,2]*l[1]/24 - c[5,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,3]*l[2]/24 - c[5,3]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[5,4]*l[1]*l[2]/(36*l[0])],
          [c[2,0]*l[1]/12 - c[2,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[2,4]*l[0]*l[1]/(9*l[2]) - c[2,4]*l[1]*np.tan(g[1])/6 + c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[2,5]*l[0]/12 - c[2,5]*l[1]*np.tan(g[0])/12 - c[2,5]*l[2]*np.tan(g[1])/12 + c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,0]*l[2]/12 - c[3,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[3,4]*l[0]/12 - c[3,4]*l[1]*np.tan(g[0])/12 - c[3,4]*l[2]*np.tan(g[1])/12 + c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,5]*l[0]*l[2]/(9*l[1]) - c[3,5]*l[2]*np.tan(g[0])/6 + c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[4,0]*l[1]*l[2]/(9*l[0]) + c[4,4]*l[1]/12 - c[4,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[4,5]*l[2]/12 - c[4,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           c[2,1]*l[0]/12 - c[2,1]*l[1]*np.tan(g[0])/12 - c[2,1]*l[2]*np.tan(g[1])/12 + c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[2,3]*l[0]*l[1]/(9*l[2]) - c[2,3]*l[1]*np.tan(g[1])/6 + c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[2,5]*l[1]/12 - c[2,5]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[3,1]*l[0]*l[2]/(9*l[1]) - c[3,1]*l[2]*np.tan(g[0])/6 + c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[3,3]*l[0]/12 - c[3,3]*l[1]*np.tan(g[0])/12 - c[3,3]*l[2]*np.tan(g[1])/12 + c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,5]*l[2]/12 - c[3,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[4,1]*l[2]/12 - c[4,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[4,3]*l[1]/12 - c[4,3]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[4,5]*l[1]*l[2]/(9*l[0]),
           c[2,2]*l[0]*l[1]/(9*l[2]) - c[2,2]*l[1]*np.tan(g[1])/6 + c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[2,3]*l[0]/12 - c[2,3]*l[1]*np.tan(g[0])/12 - c[2,3]*l[2]*np.tan(g[1])/12 + c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[2,4]*l[1]/12 - c[2,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[3,2]*l[0]/12 - c[3,2]*l[1]*np.tan(g[0])/12 - c[3,2]*l[2]*np.tan(g[1])/12 + c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,3]*l[0]*l[2]/(9*l[1]) - c[3,3]*l[2]*np.tan(g[0])/6 + c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[3,4]*l[2]/12 - c[3,4]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[4,2]*l[1]/12 - c[4,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[4,3]*l[2]/12 - c[4,3]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[4,4]*l[1]*l[2]/(9*l[0]),
           -c[2,0]*l[1]/12 + c[2,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[2,4]*l[0]*l[1]/(18*l[2]) - c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[2,5]*l[0]/24 + c[2,5]*l[1]*np.tan(g[0])/12 - c[2,5]*l[2]*np.tan(g[1])/12 - c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[3,0]*l[2]/12 + c[3,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[3,4]*l[0]/24 - c[3,4]*l[1]*np.tan(g[0])/12 + c[3,4]*l[2]*np.tan(g[1])/12 - c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,5]*l[0]*l[2]/(18*l[1]) - c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[4,0]*l[1]*l[2]/(9*l[0]) + c[4,4]*l[1]/12 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[4,5]*l[2]/12 + c[4,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           c[2,1]*l[0]/24 + c[2,1]*l[1]*np.tan(g[0])/12 - c[2,1]*l[2]*np.tan(g[1])/12 - c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[2,3]*l[0]*l[1]/(18*l[2]) - c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[2,5]*l[1]/12 + c[2,5]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[3,1]*l[0]*l[2]/(18*l[1]) - c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[3,3]*l[0]/24 - c[3,3]*l[1]*np.tan(g[0])/12 + c[3,3]*l[2]*np.tan(g[1])/12 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[3,5]*l[2]/12 + c[3,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[4,1]*l[2]/12 + c[4,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[4,3]*l[1]/12 + c[4,3]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[4,5]*l[1]*l[2]/(9*l[0]),
           c[2,2]*l[0]*l[1]/(18*l[2]) - c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[2,3]*l[0]/24 + c[2,3]*l[1]*np.tan(g[0])/12 - c[2,3]*l[2]*np.tan(g[1])/12 - c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[2,4]*l[1]/12 + c[2,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[3,2]*l[0]/24 - c[3,2]*l[1]*np.tan(g[0])/12 + c[3,2]*l[2]*np.tan(g[1])/12 - c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,3]*l[0]*l[2]/(18*l[1]) - c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[3,4]*l[2]/12 + c[3,4]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[4,2]*l[1]/12 + c[4,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[4,3]*l[2]/12 + c[4,3]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[4,4]*l[1]*l[2]/(9*l[0]),
           -c[2,0]*l[1]/24 + c[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[2,4]*l[0]*l[1]/(36*l[2]) - c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[2,5]*l[0]/24 + c[2,5]*l[1]*np.tan(g[0])/24 + c[2,5]*l[2]*np.tan(g[1])/12 - c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,0]*l[2]/12 + c[3,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[3,4]*l[0]/24 - c[3,4]*l[1]*np.tan(g[0])/24 + c[3,4]*l[2]*np.tan(g[1])/12 - c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,5]*l[0]*l[2]/(18*l[1]) + c[3,5]*l[2]*np.tan(g[0])/6 - c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[4,0]*l[1]*l[2]/(18*l[0]) + c[4,4]*l[1]/24 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,5]*l[2]/12 + c[4,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[2,1]*l[0]/24 + c[2,1]*l[1]*np.tan(g[0])/24 + c[2,1]*l[2]*np.tan(g[1])/12 - c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[2,3]*l[0]*l[1]/(36*l[2]) - c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[2,5]*l[1]/24 + c[2,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[3,1]*l[0]*l[2]/(18*l[1]) + c[3,1]*l[2]*np.tan(g[0])/6 - c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[3,3]*l[0]/24 - c[3,3]*l[1]*np.tan(g[0])/24 + c[3,3]*l[2]*np.tan(g[1])/12 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,5]*l[2]/12 + c[3,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,1]*l[2]/12 + c[4,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,3]*l[1]/24 + c[4,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,5]*l[1]*l[2]/(18*l[0]),
           c[2,2]*l[0]*l[1]/(36*l[2]) - c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[2,3]*l[0]/24 + c[2,3]*l[1]*np.tan(g[0])/24 + c[2,3]*l[2]*np.tan(g[1])/12 - c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[2,4]*l[1]/24 + c[2,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[3,2]*l[0]/24 - c[3,2]*l[1]*np.tan(g[0])/24 + c[3,2]*l[2]*np.tan(g[1])/12 - c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,3]*l[0]*l[2]/(18*l[1]) + c[3,3]*l[2]*np.tan(g[0])/6 - c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[3,4]*l[2]/12 + c[3,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,2]*l[1]/24 + c[4,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,3]*l[2]/12 + c[4,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,4]*l[1]*l[2]/(18*l[0]),
           c[2,0]*l[1]/24 - c[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[2,4]*l[0]*l[1]/(18*l[2]) - c[2,4]*l[1]*np.tan(g[1])/12 + c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[2,5]*l[0]/12 - c[2,5]*l[1]*np.tan(g[0])/24 + c[2,5]*l[2]*np.tan(g[1])/12 + c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,0]*l[2]/12 - c[3,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[3,4]*l[0]/12 - c[3,4]*l[1]*np.tan(g[0])/24 - c[3,4]*l[2]*np.tan(g[1])/12 + c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,5]*l[0]*l[2]/(9*l[1]) + c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[4,0]*l[1]*l[2]/(18*l[0]) + c[4,4]*l[1]/24 - c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,5]*l[2]/12 - c[4,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[2,1]*l[0]/12 - c[2,1]*l[1]*np.tan(g[0])/24 + c[2,1]*l[2]*np.tan(g[1])/12 + c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[2,3]*l[0]*l[1]/(18*l[2]) - c[2,3]*l[1]*np.tan(g[1])/12 + c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[2,5]*l[1]/24 - c[2,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[3,1]*l[0]*l[2]/(9*l[1]) + c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[3,3]*l[0]/12 - c[3,3]*l[1]*np.tan(g[0])/24 - c[3,3]*l[2]*np.tan(g[1])/12 + c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,5]*l[2]/12 - c[3,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,1]*l[2]/12 - c[4,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,3]*l[1]/24 - c[4,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,5]*l[1]*l[2]/(18*l[0]),
           c[2,2]*l[0]*l[1]/(18*l[2]) - c[2,2]*l[1]*np.tan(g[1])/12 + c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[2,3]*l[0]/12 - c[2,3]*l[1]*np.tan(g[0])/24 + c[2,3]*l[2]*np.tan(g[1])/12 + c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[2,4]*l[1]/24 - c[2,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[3,2]*l[0]/12 - c[3,2]*l[1]*np.tan(g[0])/24 - c[3,2]*l[2]*np.tan(g[1])/12 + c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,3]*l[0]*l[2]/(9*l[1]) + c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[3,4]*l[2]/12 - c[3,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,2]*l[1]/24 - c[4,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,3]*l[2]/12 - c[4,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,4]*l[1]*l[2]/(18*l[0]),
           c[2,0]*l[1]/12 - c[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[2,4]*l[0]*l[1]/(9*l[2]) + c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[2,5]*l[0]/12 - c[2,5]*l[1]*np.tan(g[0])/12 - c[2,5]*l[2]*np.tan(g[1])/24 + c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,0]*l[2]/24 - c[3,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[3,4]*l[0]/12 + c[3,4]*l[1]*np.tan(g[0])/12 - c[3,4]*l[2]*np.tan(g[1])/24 + c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,5]*l[0]*l[2]/(18*l[1]) - c[3,5]*l[2]*np.tan(g[0])/12 + c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[4,0]*l[1]*l[2]/(18*l[0]) - c[4,4]*l[1]/12 - c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,5]*l[2]/24 - c[4,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[2,1]*l[0]/12 - c[2,1]*l[1]*np.tan(g[0])/12 - c[2,1]*l[2]*np.tan(g[1])/24 + c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[2,3]*l[0]*l[1]/(9*l[2]) + c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[2,5]*l[1]/12 - c[2,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[3,1]*l[0]*l[2]/(18*l[1]) - c[3,1]*l[2]*np.tan(g[0])/12 + c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[3,3]*l[0]/12 + c[3,3]*l[1]*np.tan(g[0])/12 - c[3,3]*l[2]*np.tan(g[1])/24 + c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,5]*l[2]/24 - c[3,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,1]*l[2]/24 - c[4,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,3]*l[1]/12 - c[4,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,5]*l[1]*l[2]/(18*l[0]),
           -c[2,2]*l[0]*l[1]/(9*l[2]) + c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[2,3]*l[0]/12 - c[2,3]*l[1]*np.tan(g[0])/12 - c[2,3]*l[2]*np.tan(g[1])/24 + c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[2,4]*l[1]/12 - c[2,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[3,2]*l[0]/12 + c[3,2]*l[1]*np.tan(g[0])/12 - c[3,2]*l[2]*np.tan(g[1])/24 + c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,3]*l[0]*l[2]/(18*l[1]) - c[3,3]*l[2]*np.tan(g[0])/12 + c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[3,4]*l[2]/24 - c[3,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,2]*l[1]/12 - c[4,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,3]*l[2]/24 - c[4,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,4]*l[1]*l[2]/(18*l[0]),
           -c[2,0]*l[1]/12 + c[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[2,4]*l[0]*l[1]/(18*l[2]) + c[2,4]*l[1]*np.tan(g[1])/6 - c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[2,5]*l[0]/24 + c[2,5]*l[1]*np.tan(g[0])/12 - c[2,5]*l[2]*np.tan(g[1])/24 - c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,0]*l[2]/24 + c[3,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[3,4]*l[0]/24 + c[3,4]*l[1]*np.tan(g[0])/12 + c[3,4]*l[2]*np.tan(g[1])/24 - c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,5]*l[0]*l[2]/(36*l[1]) - c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[4,0]*l[1]*l[2]/(18*l[0]) - c[4,4]*l[1]/12 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,5]*l[2]/24 + c[4,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[2,1]*l[0]/24 + c[2,1]*l[1]*np.tan(g[0])/12 - c[2,1]*l[2]*np.tan(g[1])/24 - c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[2,3]*l[0]*l[1]/(18*l[2]) + c[2,3]*l[1]*np.tan(g[1])/6 - c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[2,5]*l[1]/12 + c[2,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[3,1]*l[0]*l[2]/(36*l[1]) - c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[3,3]*l[0]/24 + c[3,3]*l[1]*np.tan(g[0])/12 + c[3,3]*l[2]*np.tan(g[1])/24 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,5]*l[2]/24 + c[3,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,1]*l[2]/24 + c[4,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,3]*l[1]/12 + c[4,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,5]*l[1]*l[2]/(18*l[0]),
           -c[2,2]*l[0]*l[1]/(18*l[2]) + c[2,2]*l[1]*np.tan(g[1])/6 - c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[2,3]*l[0]/24 + c[2,3]*l[1]*np.tan(g[0])/12 - c[2,3]*l[2]*np.tan(g[1])/24 - c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[2,4]*l[1]/12 + c[2,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[3,2]*l[0]/24 + c[3,2]*l[1]*np.tan(g[0])/12 + c[3,2]*l[2]*np.tan(g[1])/24 - c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,3]*l[0]*l[2]/(36*l[1]) - c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[3,4]*l[2]/24 + c[3,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,2]*l[1]/12 + c[4,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,3]*l[2]/24 + c[4,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,4]*l[1]*l[2]/(18*l[0]),
           -c[2,0]*l[1]/24 + c[2,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[2,4]*l[0]*l[1]/(36*l[2]) + c[2,4]*l[1]*np.tan(g[1])/12 - c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[2,5]*l[0]/24 + c[2,5]*l[1]*np.tan(g[0])/24 + c[2,5]*l[2]*np.tan(g[1])/24 - c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,0]*l[2]/24 + c[3,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[3,4]*l[0]/24 + c[3,4]*l[1]*np.tan(g[0])/24 + c[3,4]*l[2]*np.tan(g[1])/24 - c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,5]*l[0]*l[2]/(36*l[1]) + c[3,5]*l[2]*np.tan(g[0])/12 - c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[4,0]*l[1]*l[2]/(36*l[0]) - c[4,4]*l[1]/24 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[4,5]*l[2]/24 + c[4,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           -c[2,1]*l[0]/24 + c[2,1]*l[1]*np.tan(g[0])/24 + c[2,1]*l[2]*np.tan(g[1])/24 - c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[2,3]*l[0]*l[1]/(36*l[2]) + c[2,3]*l[1]*np.tan(g[1])/12 - c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[2,5]*l[1]/24 + c[2,5]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[3,1]*l[0]*l[2]/(36*l[1]) + c[3,1]*l[2]*np.tan(g[0])/12 - c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[3,3]*l[0]/24 + c[3,3]*l[1]*np.tan(g[0])/24 + c[3,3]*l[2]*np.tan(g[1])/24 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,5]*l[2]/24 + c[3,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[4,1]*l[2]/24 + c[4,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[4,3]*l[1]/24 + c[4,3]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[4,5]*l[1]*l[2]/(36*l[0]),
           -c[2,2]*l[0]*l[1]/(36*l[2]) + c[2,2]*l[1]*np.tan(g[1])/12 - c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[2,3]*l[0]/24 + c[2,3]*l[1]*np.tan(g[0])/24 + c[2,3]*l[2]*np.tan(g[1])/24 - c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[2,4]*l[1]/24 + c[2,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[3,2]*l[0]/24 + c[3,2]*l[1]*np.tan(g[0])/24 + c[3,2]*l[2]*np.tan(g[1])/24 - c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,3]*l[0]*l[2]/(36*l[1]) + c[3,3]*l[2]*np.tan(g[0])/12 - c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[3,4]*l[2]/24 + c[3,4]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[4,2]*l[1]/24 + c[4,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[4,3]*l[2]/24 + c[4,3]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[4,4]*l[1]*l[2]/(36*l[0]),
           c[2,0]*l[1]/24 - c[2,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[2,4]*l[0]*l[1]/(18*l[2]) + c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[2,5]*l[0]/12 - c[2,5]*l[1]*np.tan(g[0])/24 + c[2,5]*l[2]*np.tan(g[1])/24 + c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[3,0]*l[2]/24 - c[3,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[3,4]*l[0]/12 + c[3,4]*l[1]*np.tan(g[0])/24 - c[3,4]*l[2]*np.tan(g[1])/24 + c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,5]*l[0]*l[2]/(18*l[1]) + c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[4,0]*l[1]*l[2]/(36*l[0]) - c[4,4]*l[1]/24 - c[4,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[4,5]*l[2]/24 - c[4,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           -c[2,1]*l[0]/12 - c[2,1]*l[1]*np.tan(g[0])/24 + c[2,1]*l[2]*np.tan(g[1])/24 + c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[2,3]*l[0]*l[1]/(18*l[2]) + c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[2,5]*l[1]/24 - c[2,5]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[3,1]*l[0]*l[2]/(18*l[1]) + c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[3,3]*l[0]/12 + c[3,3]*l[1]*np.tan(g[0])/24 - c[3,3]*l[2]*np.tan(g[1])/24 + c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[3,5]*l[2]/24 - c[3,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[4,1]*l[2]/24 - c[4,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[4,3]*l[1]/24 - c[4,3]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[4,5]*l[1]*l[2]/(36*l[0]),
           -c[2,2]*l[0]*l[1]/(18*l[2]) + c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[2,3]*l[0]/12 - c[2,3]*l[1]*np.tan(g[0])/24 + c[2,3]*l[2]*np.tan(g[1])/24 + c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[2,4]*l[1]/24 - c[2,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[3,2]*l[0]/12 + c[3,2]*l[1]*np.tan(g[0])/24 - c[3,2]*l[2]*np.tan(g[1])/24 + c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,3]*l[0]*l[2]/(18*l[1]) + c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[3,4]*l[2]/24 - c[3,4]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[4,2]*l[1]/24 - c[4,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[4,3]*l[2]/24 - c[4,3]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[4,4]*l[1]*l[2]/(36*l[0])],
          [-c[0,0]*l[1]*l[2]/(9*l[0]) - c[0,4]*l[1]/12 + c[0,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[0,5]*l[2]/12 + c[0,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[4,0]*l[1]/12 + c[4,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[4,4]*l[0]*l[1]/(18*l[2]) - c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[4,5]*l[0]/24 - c[4,5]*l[1]*np.tan(g[0])/12 + c[4,5]*l[2]*np.tan(g[1])/12 - c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[5,0]*l[2]/12 + c[5,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[5,4]*l[0]/24 + c[5,4]*l[1]*np.tan(g[0])/12 - c[5,4]*l[2]*np.tan(g[1])/12 - c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[5,5]*l[0]*l[2]/(18*l[1]) - c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]),
           -c[0,1]*l[2]/12 + c[0,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[0,3]*l[1]/12 + c[0,3]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[0,5]*l[1]*l[2]/(9*l[0]) + c[4,1]*l[0]/24 - c[4,1]*l[1]*np.tan(g[0])/12 + c[4,1]*l[2]*np.tan(g[1])/12 - c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[4,3]*l[0]*l[1]/(18*l[2]) - c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[4,5]*l[1]/12 + c[4,5]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,1]*l[0]*l[2]/(18*l[1]) - c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[5,3]*l[0]/24 + c[5,3]*l[1]*np.tan(g[0])/12 - c[5,3]*l[2]*np.tan(g[1])/12 - c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[5,5]*l[2]/12 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           -c[0,2]*l[1]/12 + c[0,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[0,3]*l[2]/12 + c[0,3]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[0,4]*l[1]*l[2]/(9*l[0]) + c[4,2]*l[0]*l[1]/(18*l[2]) - c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[4,3]*l[0]/24 - c[4,3]*l[1]*np.tan(g[0])/12 + c[4,3]*l[2]*np.tan(g[1])/12 - c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[4,4]*l[1]/12 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,2]*l[0]/24 + c[5,2]*l[1]*np.tan(g[0])/12 - c[5,2]*l[2]*np.tan(g[1])/12 - c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[5,3]*l[0]*l[2]/(18*l[1]) - c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[5,4]*l[2]/12 + c[5,4]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           (l[0]**2*(4*c[4,4]*l[1]**2 + 3*c[4,5]*l[1]*l[2] + 3*c[5,4]*l[1]*l[2] + 4*c[5,5]*l[2]**2) + 3*l[0]*l[1]*l[2]*(-c[0,4]*l[1] - c[0,5]*l[2] - c[4,0]*l[1] + 2*c[4,4]*l[1]*np.tan(g[1]) + c[4,5]*l[1]*np.tan(g[0]) + c[4,5]*l[2]*np.tan(g[1]) - c[5,0]*l[2] + c[5,4]*l[1]*np.tan(g[0]) + c[5,4]*l[2]*np.tan(g[1]) + 2*c[5,5]*l[2]*np.tan(g[0])) + 4*l[1]**2*l[2]**2*(c[0,0] - c[0,4]*np.tan(g[1]) - c[0,5]*np.tan(g[0]) - c[4,0]*np.tan(g[1]) + c[4,4]*np.tan(g[1])**2 + c[4,5]*np.tan(g[0])*np.tan(g[1]) - c[5,0]*np.tan(g[0]) + c[5,4]*np.tan(g[0])*np.tan(g[1]) + c[5,5]*np.tan(g[0])**2))/(36*l[0]*l[1]*l[2]),
           (l[0]**2*(3*c[4,1]*l[1]*l[2] + 4*c[4,3]*l[1]**2 + 4*c[5,1]*l[2]**2 + 3*c[5,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(-c[0,1]*l[2] - c[0,3]*l[1] + c[4,1]*l[1]*np.tan(g[0]) + c[4,1]*l[2]*np.tan(g[1]) + 2*c[4,3]*l[1]*np.tan(g[1]) - c[4,5]*l[1] + 2*c[5,1]*l[2]*np.tan(g[0]) + c[5,3]*l[1]*np.tan(g[0]) + c[5,3]*l[2]*np.tan(g[1]) - c[5,5]*l[2]) + 4*l[1]**2*l[2]**2*(-c[0,1]*np.tan(g[0]) - c[0,3]*np.tan(g[1]) + c[0,5] + c[4,1]*np.tan(g[0])*np.tan(g[1]) + c[4,3]*np.tan(g[1])**2 - c[4,5]*np.tan(g[1]) + c[5,1]*np.tan(g[0])**2 + c[5,3]*np.tan(g[0])*np.tan(g[1]) - c[5,5]*np.tan(g[0])))/(36*l[0]*l[1]*l[2]),
           (l[0]**2*(4*c[4,2]*l[1]**2 + 3*c[4,3]*l[1]*l[2] + 3*c[5,2]*l[1]*l[2] + 4*c[5,3]*l[2]**2) + 3*l[0]*l[1]*l[2]*(-c[0,2]*l[1] - c[0,3]*l[2] + 2*c[4,2]*l[1]*np.tan(g[1]) + c[4,3]*l[1]*np.tan(g[0]) + c[4,3]*l[2]*np.tan(g[1]) - c[4,4]*l[1] + c[5,2]*l[1]*np.tan(g[0]) + c[5,2]*l[2]*np.tan(g[1]) + 2*c[5,3]*l[2]*np.tan(g[0]) - c[5,4]*l[2]) + 4*l[1]**2*l[2]**2*(-c[0,2]*np.tan(g[1]) - c[0,3]*np.tan(g[0]) + c[0,4] + c[4,2]*np.tan(g[1])**2 + c[4,3]*np.tan(g[0])*np.tan(g[1]) - c[4,4]*np.tan(g[1]) + c[5,2]*np.tan(g[0])*np.tan(g[1]) + c[5,3]*np.tan(g[0])**2 - c[5,4]*np.tan(g[0])))/(36*l[0]*l[1]*l[2]),
           (2*l[0]**2*(2*c[4,4]*l[1]**2 - 3*c[4,5]*l[1]*l[2] + 3*c[5,4]*l[1]*l[2] - 4*c[5,5]*l[2]**2) + 3*l[0]*l[1]*l[2]*(-c[0,4]*l[1] + 2*c[0,5]*l[2] - c[4,0]*l[1] + 2*c[4,4]*l[1]*np.tan(g[1]) + c[4,5]*l[1]*np.tan(g[0]) - 2*c[4,5]*l[2]*np.tan(g[1]) - 2*c[5,0]*l[2] + c[5,4]*l[1]*np.tan(g[0]) + 2*c[5,4]*l[2]*np.tan(g[1])) + 4*l[1]**2*l[2]**2*(c[0,0] - c[0,4]*np.tan(g[1]) - c[0,5]*np.tan(g[0]) - c[4,0]*np.tan(g[1]) + c[4,4]*np.tan(g[1])**2 + c[4,5]*np.tan(g[0])*np.tan(g[1]) - c[5,0]*np.tan(g[0]) + c[5,4]*np.tan(g[0])*np.tan(g[1]) + c[5,5]*np.tan(g[0])**2))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(-3*c[4,1]*l[1]*l[2] + 2*c[4,3]*l[1]**2 - 4*c[5,1]*l[2]**2 + 3*c[5,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(2*c[0,1]*l[2] - c[0,3]*l[1] + c[4,1]*l[1]*np.tan(g[0]) - 2*c[4,1]*l[2]*np.tan(g[1]) + 2*c[4,3]*l[1]*np.tan(g[1]) - c[4,5]*l[1] + c[5,3]*l[1]*np.tan(g[0]) + 2*c[5,3]*l[2]*np.tan(g[1]) - 2*c[5,5]*l[2]) + 4*l[1]**2*l[2]**2*(-c[0,1]*np.tan(g[0]) - c[0,3]*np.tan(g[1]) + c[0,5] + c[4,1]*np.tan(g[0])*np.tan(g[1]) + c[4,3]*np.tan(g[1])**2 - c[4,5]*np.tan(g[1]) + c[5,1]*np.tan(g[0])**2 + c[5,3]*np.tan(g[0])*np.tan(g[1]) - c[5,5]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(2*c[4,2]*l[1]**2 - 3*c[4,3]*l[1]*l[2] + 3*c[5,2]*l[1]*l[2] - 4*c[5,3]*l[2]**2) + 3*l[0]*l[1]*l[2]*(-c[0,2]*l[1] + 2*c[0,3]*l[2] + 2*c[4,2]*l[1]*np.tan(g[1]) + c[4,3]*l[1]*np.tan(g[0]) - 2*c[4,3]*l[2]*np.tan(g[1]) - c[4,4]*l[1] + c[5,2]*l[1]*np.tan(g[0]) + 2*c[5,2]*l[2]*np.tan(g[1]) - 2*c[5,4]*l[2]) + 4*l[1]**2*l[2]**2*(-c[0,2]*np.tan(g[1]) - c[0,3]*np.tan(g[0]) + c[0,4] + c[4,2]*np.tan(g[1])**2 + c[4,3]*np.tan(g[0])*np.tan(g[1]) - c[4,4]*np.tan(g[1]) + c[5,2]*np.tan(g[0])*np.tan(g[1]) + c[5,3]*np.tan(g[0])**2 - c[5,4]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           -c[0,0]*l[1]*l[2]/(18*l[0]) - c[0,4]*l[1]/24 + c[0,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[0,5]*l[2]/12 + c[0,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,0]*l[1]/24 + c[4,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,4]*l[0]*l[1]/(36*l[2]) - c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[4,5]*l[0]/24 - c[4,5]*l[1]*np.tan(g[0])/24 - c[4,5]*l[2]*np.tan(g[1])/12 - c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,0]*l[2]/12 + c[5,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[5,4]*l[0]/24 + c[5,4]*l[1]*np.tan(g[0])/24 - c[5,4]*l[2]*np.tan(g[1])/12 - c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,5]*l[0]*l[2]/(18*l[1]) - c[5,5]*l[2]*np.tan(g[0])/6 - c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]),
           c[0,1]*l[2]/12 + c[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[0,3]*l[1]/24 + c[0,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[0,5]*l[1]*l[2]/(18*l[0]) - c[4,1]*l[0]/24 - c[4,1]*l[1]*np.tan(g[0])/24 - c[4,1]*l[2]*np.tan(g[1])/12 - c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[4,3]*l[0]*l[1]/(36*l[2]) - c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[4,5]*l[1]/24 + c[4,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,1]*l[0]*l[2]/(18*l[1]) - c[5,1]*l[2]*np.tan(g[0])/6 - c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[5,3]*l[0]/24 + c[5,3]*l[1]*np.tan(g[0])/24 - c[5,3]*l[2]*np.tan(g[1])/12 - c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,5]*l[2]/12 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[0,2]*l[1]/24 + c[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[0,3]*l[2]/12 + c[0,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[0,4]*l[1]*l[2]/(18*l[0]) + c[4,2]*l[0]*l[1]/(36*l[2]) - c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[4,3]*l[0]/24 - c[4,3]*l[1]*np.tan(g[0])/24 - c[4,3]*l[2]*np.tan(g[1])/12 - c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[4,4]*l[1]/24 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,2]*l[0]/24 + c[5,2]*l[1]*np.tan(g[0])/24 - c[5,2]*l[2]*np.tan(g[1])/12 - c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,3]*l[0]*l[2]/(18*l[1]) - c[5,3]*l[2]*np.tan(g[0])/6 - c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[5,4]*l[2]/12 + c[5,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[0,0]*l[1]*l[2]/(18*l[0]) + c[0,4]*l[1]/12 + c[0,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[0,5]*l[2]/24 + c[0,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,0]*l[1]/12 + c[4,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,4]*l[0]*l[1]/(18*l[2]) - c[4,4]*l[1]*np.tan(g[1])/6 - c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[4,5]*l[0]/24 - c[4,5]*l[1]*np.tan(g[0])/12 + c[4,5]*l[2]*np.tan(g[1])/24 - c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,0]*l[2]/24 + c[5,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[5,4]*l[0]/24 - c[5,4]*l[1]*np.tan(g[0])/12 - c[5,4]*l[2]*np.tan(g[1])/24 - c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,5]*l[0]*l[2]/(36*l[1]) - c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]),
           -c[0,1]*l[2]/24 + c[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[0,3]*l[1]/12 + c[0,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[0,5]*l[1]*l[2]/(18*l[0]) + c[4,1]*l[0]/24 - c[4,1]*l[1]*np.tan(g[0])/12 + c[4,1]*l[2]*np.tan(g[1])/24 - c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[4,3]*l[0]*l[1]/(18*l[2]) - c[4,3]*l[1]*np.tan(g[1])/6 - c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[4,5]*l[1]/12 + c[4,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,1]*l[0]*l[2]/(36*l[1]) - c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[5,3]*l[0]/24 - c[5,3]*l[1]*np.tan(g[0])/12 - c[5,3]*l[2]*np.tan(g[1])/24 - c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,5]*l[2]/24 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[0,2]*l[1]/12 + c[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[0,3]*l[2]/24 + c[0,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[0,4]*l[1]*l[2]/(18*l[0]) - c[4,2]*l[0]*l[1]/(18*l[2]) - c[4,2]*l[1]*np.tan(g[1])/6 - c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[4,3]*l[0]/24 - c[4,3]*l[1]*np.tan(g[0])/12 + c[4,3]*l[2]*np.tan(g[1])/24 - c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[4,4]*l[1]/12 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,2]*l[0]/24 - c[5,2]*l[1]*np.tan(g[0])/12 - c[5,2]*l[2]*np.tan(g[1])/24 - c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,3]*l[0]*l[2]/(36*l[1]) - c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[5,4]*l[2]/24 + c[5,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           (2*l[0]**2*(-4*c[4,4]*l[1]**2 + 3*c[4,5]*l[1]*l[2] - 3*c[5,4]*l[1]*l[2] + 2*c[5,5]*l[2]**2) + 3*l[0]*l[1]*l[2]*(2*c[0,4]*l[1] - c[0,5]*l[2] - 2*c[4,0]*l[1] + 2*c[4,5]*l[1]*np.tan(g[0]) + c[4,5]*l[2]*np.tan(g[1]) - c[5,0]*l[2] - 2*c[5,4]*l[1]*np.tan(g[0]) + c[5,4]*l[2]*np.tan(g[1]) + 2*c[5,5]*l[2]*np.tan(g[0])) + 4*l[1]**2*l[2]**2*(c[0,0] - c[0,4]*np.tan(g[1]) - c[0,5]*np.tan(g[0]) - c[4,0]*np.tan(g[1]) + c[4,4]*np.tan(g[1])**2 + c[4,5]*np.tan(g[0])*np.tan(g[1]) - c[5,0]*np.tan(g[0]) + c[5,4]*np.tan(g[0])*np.tan(g[1]) + c[5,5]*np.tan(g[0])**2))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(3*c[4,1]*l[1]*l[2] - 4*c[4,3]*l[1]**2 + 2*c[5,1]*l[2]**2 - 3*c[5,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(-c[0,1]*l[2] + 2*c[0,3]*l[1] + 2*c[4,1]*l[1]*np.tan(g[0]) + c[4,1]*l[2]*np.tan(g[1]) - 2*c[4,5]*l[1] + 2*c[5,1]*l[2]*np.tan(g[0]) - 2*c[5,3]*l[1]*np.tan(g[0]) + c[5,3]*l[2]*np.tan(g[1]) - c[5,5]*l[2]) + 4*l[1]**2*l[2]**2*(-c[0,1]*np.tan(g[0]) - c[0,3]*np.tan(g[1]) + c[0,5] + c[4,1]*np.tan(g[0])*np.tan(g[1]) + c[4,3]*np.tan(g[1])**2 - c[4,5]*np.tan(g[1]) + c[5,1]*np.tan(g[0])**2 + c[5,3]*np.tan(g[0])*np.tan(g[1]) - c[5,5]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(-4*c[4,2]*l[1]**2 + 3*c[4,3]*l[1]*l[2] - 3*c[5,2]*l[1]*l[2] + 2*c[5,3]*l[2]**2) + 3*l[0]*l[1]*l[2]*(2*c[0,2]*l[1] - c[0,3]*l[2] + 2*c[4,3]*l[1]*np.tan(g[0]) + c[4,3]*l[2]*np.tan(g[1]) - 2*c[4,4]*l[1] - 2*c[5,2]*l[1]*np.tan(g[0]) + c[5,2]*l[2]*np.tan(g[1]) + 2*c[5,3]*l[2]*np.tan(g[0]) - c[5,4]*l[2]) + 4*l[1]**2*l[2]**2*(-c[0,2]*np.tan(g[1]) - c[0,3]*np.tan(g[0]) + c[0,4] + c[4,2]*np.tan(g[1])**2 + c[4,3]*np.tan(g[0])*np.tan(g[1]) - c[4,4]*np.tan(g[1]) + c[5,2]*np.tan(g[0])*np.tan(g[1]) + c[5,3]*np.tan(g[0])**2 - c[5,4]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (-2*l[0]**2*(2*c[4,4]*l[1]**2 + 3*c[4,5]*l[1]*l[2] + 3*c[5,4]*l[1]*l[2] + 2*c[5,5]*l[2]**2) + 3*l[0]*l[1]*l[2]*(c[0,4]*l[1] + c[0,5]*l[2] - c[4,0]*l[1] + c[4,5]*l[1]*np.tan(g[0]) - c[4,5]*l[2]*np.tan(g[1]) - c[5,0]*l[2] - c[5,4]*l[1]*np.tan(g[0]) + c[5,4]*l[2]*np.tan(g[1])) + 2*l[1]**2*l[2]**2*(c[0,0] - c[0,4]*np.tan(g[1]) - c[0,5]*np.tan(g[0]) - c[4,0]*np.tan(g[1]) + c[4,4]*np.tan(g[1])**2 + c[4,5]*np.tan(g[0])*np.tan(g[1]) - c[5,0]*np.tan(g[0]) + c[5,4]*np.tan(g[0])*np.tan(g[1]) + c[5,5]*np.tan(g[0])**2))/(72*l[0]*l[1]*l[2]),
           (-2*l[0]**2*(3*c[4,1]*l[1]*l[2] + 2*c[4,3]*l[1]**2 + 2*c[5,1]*l[2]**2 + 3*c[5,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(c[0,1]*l[2] + c[0,3]*l[1] + c[4,1]*l[1]*np.tan(g[0]) - c[4,1]*l[2]*np.tan(g[1]) - c[4,5]*l[1] - c[5,3]*l[1]*np.tan(g[0]) + c[5,3]*l[2]*np.tan(g[1]) - c[5,5]*l[2]) + 2*l[1]**2*l[2]**2*(-c[0,1]*np.tan(g[0]) - c[0,3]*np.tan(g[1]) + c[0,5] + c[4,1]*np.tan(g[0])*np.tan(g[1]) + c[4,3]*np.tan(g[1])**2 - c[4,5]*np.tan(g[1]) + c[5,1]*np.tan(g[0])**2 + c[5,3]*np.tan(g[0])*np.tan(g[1]) - c[5,5]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (-2*l[0]**2*(2*c[4,2]*l[1]**2 + 3*c[4,3]*l[1]*l[2] + 3*c[5,2]*l[1]*l[2] + 2*c[5,3]*l[2]**2) + 3*l[0]*l[1]*l[2]*(c[0,2]*l[1] + c[0,3]*l[2] + c[4,3]*l[1]*np.tan(g[0]) - c[4,3]*l[2]*np.tan(g[1]) - c[4,4]*l[1] - c[5,2]*l[1]*np.tan(g[0]) + c[5,2]*l[2]*np.tan(g[1]) - c[5,4]*l[2]) + 2*l[1]**2*l[2]**2*(-c[0,2]*np.tan(g[1]) - c[0,3]*np.tan(g[0]) + c[0,4] + c[4,2]*np.tan(g[1])**2 + c[4,3]*np.tan(g[0])*np.tan(g[1]) - c[4,4]*np.tan(g[1]) + c[5,2]*np.tan(g[0])*np.tan(g[1]) + c[5,3]*np.tan(g[0])**2 - c[5,4]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           -c[0,0]*l[1]*l[2]/(36*l[0]) + c[0,4]*l[1]/24 + c[0,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[0,5]*l[2]/24 + c[0,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[4,0]*l[1]/24 + c[4,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[4,4]*l[0]*l[1]/(36*l[2]) - c[4,4]*l[1]*np.tan(g[1])/12 - c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[4,5]*l[0]/24 - c[4,5]*l[1]*np.tan(g[0])/24 - c[4,5]*l[2]*np.tan(g[1])/24 - c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[5,0]*l[2]/24 + c[5,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[5,4]*l[0]/24 - c[5,4]*l[1]*np.tan(g[0])/24 - c[5,4]*l[2]*np.tan(g[1])/24 - c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,5]*l[0]*l[2]/(36*l[1]) - c[5,5]*l[2]*np.tan(g[0])/12 - c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]),
           c[0,1]*l[2]/24 + c[0,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[0,3]*l[1]/24 + c[0,3]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[0,5]*l[1]*l[2]/(36*l[0]) - c[4,1]*l[0]/24 - c[4,1]*l[1]*np.tan(g[0])/24 - c[4,1]*l[2]*np.tan(g[1])/24 - c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[4,3]*l[0]*l[1]/(36*l[2]) - c[4,3]*l[1]*np.tan(g[1])/12 - c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[4,5]*l[1]/24 + c[4,5]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,1]*l[0]*l[2]/(36*l[1]) - c[5,1]*l[2]*np.tan(g[0])/12 - c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[5,3]*l[0]/24 - c[5,3]*l[1]*np.tan(g[0])/24 - c[5,3]*l[2]*np.tan(g[1])/24 - c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[5,5]*l[2]/24 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           c[0,2]*l[1]/24 + c[0,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[0,3]*l[2]/24 + c[0,3]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[0,4]*l[1]*l[2]/(36*l[0]) - c[4,2]*l[0]*l[1]/(36*l[2]) - c[4,2]*l[1]*np.tan(g[1])/12 - c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[4,3]*l[0]/24 - c[4,3]*l[1]*np.tan(g[0])/24 - c[4,3]*l[2]*np.tan(g[1])/24 - c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[4,4]*l[1]/24 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,2]*l[0]/24 - c[5,2]*l[1]*np.tan(g[0])/24 - c[5,2]*l[2]*np.tan(g[1])/24 - c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,3]*l[0]*l[2]/(36*l[1]) - c[5,3]*l[2]*np.tan(g[0])/12 - c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[5,4]*l[2]/24 + c[5,4]*l[1]*l[2]*np.tan(g[0])/(36*l[0])],
          [c[1,0]*l[2]/12 + c[1,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[1,4]*l[0]/24 + c[1,4]*l[1]*np.tan(g[0])/12 - c[1,4]*l[2]*np.tan(g[1])/12 - c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[1,5]*l[0]*l[2]/(18*l[1]) - c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[3,0]*l[1]/12 + c[3,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[3,4]*l[0]*l[1]/(18*l[2]) - c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[3,5]*l[0]/24 - c[3,5]*l[1]*np.tan(g[0])/12 + c[3,5]*l[2]*np.tan(g[1])/12 - c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[5,0]*l[1]*l[2]/(9*l[0]) - c[5,4]*l[1]/12 + c[5,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[5,5]*l[2]/12 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           c[1,1]*l[0]*l[2]/(18*l[1]) - c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[1,3]*l[0]/24 + c[1,3]*l[1]*np.tan(g[0])/12 - c[1,3]*l[2]*np.tan(g[1])/12 - c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[1,5]*l[2]/12 + c[1,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[3,1]*l[0]/24 - c[3,1]*l[1]*np.tan(g[0])/12 + c[3,1]*l[2]*np.tan(g[1])/12 - c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,3]*l[0]*l[1]/(18*l[2]) - c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[3,5]*l[1]/12 + c[3,5]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[5,1]*l[2]/12 + c[5,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[5,3]*l[1]/12 + c[5,3]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[5,5]*l[1]*l[2]/(9*l[0]),
           c[1,2]*l[0]/24 + c[1,2]*l[1]*np.tan(g[0])/12 - c[1,2]*l[2]*np.tan(g[1])/12 - c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[1,3]*l[0]*l[2]/(18*l[1]) - c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[1,4]*l[2]/12 + c[1,4]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[3,2]*l[0]*l[1]/(18*l[2]) - c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[3,3]*l[0]/24 - c[3,3]*l[1]*np.tan(g[0])/12 + c[3,3]*l[2]*np.tan(g[1])/12 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,4]*l[1]/12 + c[3,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[5,2]*l[1]/12 + c[5,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[5,3]*l[2]/12 + c[5,3]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[5,4]*l[1]*l[2]/(9*l[0]),
           (l[0]**2*(3*c[1,4]*l[1]*l[2] + 4*c[1,5]*l[2]**2 + 4*c[3,4]*l[1]**2 + 3*c[3,5]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(-c[1,0]*l[2] + c[1,4]*l[1]*np.tan(g[0]) + c[1,4]*l[2]*np.tan(g[1]) + 2*c[1,5]*l[2]*np.tan(g[0]) - c[3,0]*l[1] + 2*c[3,4]*l[1]*np.tan(g[1]) + c[3,5]*l[1]*np.tan(g[0]) + c[3,5]*l[2]*np.tan(g[1]) - c[5,4]*l[1] - c[5,5]*l[2]) + 4*l[1]**2*l[2]**2*(-c[1,0]*np.tan(g[0]) + c[1,4]*np.tan(g[0])*np.tan(g[1]) + c[1,5]*np.tan(g[0])**2 - c[3,0]*np.tan(g[1]) + c[3,4]*np.tan(g[1])**2 + c[3,5]*np.tan(g[0])*np.tan(g[1]) + c[5,0] - c[5,4]*np.tan(g[1]) - c[5,5]*np.tan(g[0])))/(36*l[0]*l[1]*l[2]),
           (l[0]**2*(4*c[1,1]*l[2]**2 + 3*c[1,3]*l[1]*l[2] + 3*c[3,1]*l[1]*l[2] + 4*c[3,3]*l[1]**2) + 3*l[0]*l[1]*l[2]*(2*c[1,1]*l[2]*np.tan(g[0]) + c[1,3]*l[1]*np.tan(g[0]) + c[1,3]*l[2]*np.tan(g[1]) - c[1,5]*l[2] + c[3,1]*l[1]*np.tan(g[0]) + c[3,1]*l[2]*np.tan(g[1]) + 2*c[3,3]*l[1]*np.tan(g[1]) - c[3,5]*l[1] - c[5,1]*l[2] - c[5,3]*l[1]) + 4*l[1]**2*l[2]**2*(c[1,1]*np.tan(g[0])**2 + c[1,3]*np.tan(g[0])*np.tan(g[1]) - c[1,5]*np.tan(g[0]) + c[3,1]*np.tan(g[0])*np.tan(g[1]) + c[3,3]*np.tan(g[1])**2 - c[3,5]*np.tan(g[1]) - c[5,1]*np.tan(g[0]) - c[5,3]*np.tan(g[1]) + c[5,5]))/(36*l[0]*l[1]*l[2]),
           (l[0]**2*(3*c[1,2]*l[1]*l[2] + 4*c[1,3]*l[2]**2 + 4*c[3,2]*l[1]**2 + 3*c[3,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(c[1,2]*l[1]*np.tan(g[0]) + c[1,2]*l[2]*np.tan(g[1]) + 2*c[1,3]*l[2]*np.tan(g[0]) - c[1,4]*l[2] + 2*c[3,2]*l[1]*np.tan(g[1]) + c[3,3]*l[1]*np.tan(g[0]) + c[3,3]*l[2]*np.tan(g[1]) - c[3,4]*l[1] - c[5,2]*l[1] - c[5,3]*l[2]) + 4*l[1]**2*l[2]**2*(c[1,2]*np.tan(g[0])*np.tan(g[1]) + c[1,3]*np.tan(g[0])**2 - c[1,4]*np.tan(g[0]) + c[3,2]*np.tan(g[1])**2 + c[3,3]*np.tan(g[0])*np.tan(g[1]) - c[3,4]*np.tan(g[1]) - c[5,2]*np.tan(g[1]) - c[5,3]*np.tan(g[0]) + c[5,4]))/(36*l[0]*l[1]*l[2]),
           (2*l[0]**2*(3*c[1,4]*l[1]*l[2] - 4*c[1,5]*l[2]**2 + 2*c[3,4]*l[1]**2 - 3*c[3,5]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(-2*c[1,0]*l[2] + c[1,4]*l[1]*np.tan(g[0]) + 2*c[1,4]*l[2]*np.tan(g[1]) - c[3,0]*l[1] + 2*c[3,4]*l[1]*np.tan(g[1]) + c[3,5]*l[1]*np.tan(g[0]) - 2*c[3,5]*l[2]*np.tan(g[1]) - c[5,4]*l[1] + 2*c[5,5]*l[2]) + 4*l[1]**2*l[2]**2*(-c[1,0]*np.tan(g[0]) + c[1,4]*np.tan(g[0])*np.tan(g[1]) + c[1,5]*np.tan(g[0])**2 - c[3,0]*np.tan(g[1]) + c[3,4]*np.tan(g[1])**2 + c[3,5]*np.tan(g[0])*np.tan(g[1]) + c[5,0] - c[5,4]*np.tan(g[1]) - c[5,5]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(-4*c[1,1]*l[2]**2 + 3*c[1,3]*l[1]*l[2] - 3*c[3,1]*l[1]*l[2] + 2*c[3,3]*l[1]**2) + 3*l[0]*l[1]*l[2]*(c[1,3]*l[1]*np.tan(g[0]) + 2*c[1,3]*l[2]*np.tan(g[1]) - 2*c[1,5]*l[2] + c[3,1]*l[1]*np.tan(g[0]) - 2*c[3,1]*l[2]*np.tan(g[1]) + 2*c[3,3]*l[1]*np.tan(g[1]) - c[3,5]*l[1] + 2*c[5,1]*l[2] - c[5,3]*l[1]) + 4*l[1]**2*l[2]**2*(c[1,1]*np.tan(g[0])**2 + c[1,3]*np.tan(g[0])*np.tan(g[1]) - c[1,5]*np.tan(g[0]) + c[3,1]*np.tan(g[0])*np.tan(g[1]) + c[3,3]*np.tan(g[1])**2 - c[3,5]*np.tan(g[1]) - c[5,1]*np.tan(g[0]) - c[5,3]*np.tan(g[1]) + c[5,5]))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(3*c[1,2]*l[1]*l[2] - 4*c[1,3]*l[2]**2 + 2*c[3,2]*l[1]**2 - 3*c[3,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(c[1,2]*l[1]*np.tan(g[0]) + 2*c[1,2]*l[2]*np.tan(g[1]) - 2*c[1,4]*l[2] + 2*c[3,2]*l[1]*np.tan(g[1]) + c[3,3]*l[1]*np.tan(g[0]) - 2*c[3,3]*l[2]*np.tan(g[1]) - c[3,4]*l[1] - c[5,2]*l[1] + 2*c[5,3]*l[2]) + 4*l[1]**2*l[2]**2*(c[1,2]*np.tan(g[0])*np.tan(g[1]) + c[1,3]*np.tan(g[0])**2 - c[1,4]*np.tan(g[0]) + c[3,2]*np.tan(g[1])**2 + c[3,3]*np.tan(g[0])*np.tan(g[1]) - c[3,4]*np.tan(g[1]) - c[5,2]*np.tan(g[1]) - c[5,3]*np.tan(g[0]) + c[5,4]))/(72*l[0]*l[1]*l[2]),
           c[1,0]*l[2]/12 + c[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[1,4]*l[0]/24 + c[1,4]*l[1]*np.tan(g[0])/24 - c[1,4]*l[2]*np.tan(g[1])/12 - c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[1,5]*l[0]*l[2]/(18*l[1]) - c[1,5]*l[2]*np.tan(g[0])/6 - c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[3,0]*l[1]/24 + c[3,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[3,4]*l[0]*l[1]/(36*l[2]) - c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[3,5]*l[0]/24 - c[3,5]*l[1]*np.tan(g[0])/24 - c[3,5]*l[2]*np.tan(g[1])/12 - c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,0]*l[1]*l[2]/(18*l[0]) - c[5,4]*l[1]/24 + c[5,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,5]*l[2]/12 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[1,1]*l[0]*l[2]/(18*l[1]) - c[1,1]*l[2]*np.tan(g[0])/6 - c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[1,3]*l[0]/24 + c[1,3]*l[1]*np.tan(g[0])/24 - c[1,3]*l[2]*np.tan(g[1])/12 - c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[1,5]*l[2]/12 + c[1,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[3,1]*l[0]/24 - c[3,1]*l[1]*np.tan(g[0])/24 - c[3,1]*l[2]*np.tan(g[1])/12 - c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,3]*l[0]*l[1]/(36*l[2]) - c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[3,5]*l[1]/24 + c[3,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,1]*l[2]/12 + c[5,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[5,3]*l[1]/24 + c[5,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,5]*l[1]*l[2]/(18*l[0]),
           c[1,2]*l[0]/24 + c[1,2]*l[1]*np.tan(g[0])/24 - c[1,2]*l[2]*np.tan(g[1])/12 - c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[1,3]*l[0]*l[2]/(18*l[1]) - c[1,3]*l[2]*np.tan(g[0])/6 - c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[1,4]*l[2]/12 + c[1,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[3,2]*l[0]*l[1]/(36*l[2]) - c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[3,3]*l[0]/24 - c[3,3]*l[1]*np.tan(g[0])/24 - c[3,3]*l[2]*np.tan(g[1])/12 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,4]*l[1]/24 + c[3,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,2]*l[1]/24 + c[5,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,3]*l[2]/12 + c[5,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[5,4]*l[1]*l[2]/(18*l[0]),
           c[1,0]*l[2]/24 + c[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[1,4]*l[0]/24 - c[1,4]*l[1]*np.tan(g[0])/12 - c[1,4]*l[2]*np.tan(g[1])/24 - c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[1,5]*l[0]*l[2]/(36*l[1]) - c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[3,0]*l[1]/12 + c[3,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[3,4]*l[0]*l[1]/(18*l[2]) - c[3,4]*l[1]*np.tan(g[1])/6 - c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[3,5]*l[0]/24 - c[3,5]*l[1]*np.tan(g[0])/12 + c[3,5]*l[2]*np.tan(g[1])/24 - c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,0]*l[1]*l[2]/(18*l[0]) + c[5,4]*l[1]/12 + c[5,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,5]*l[2]/24 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[1,1]*l[0]*l[2]/(36*l[1]) - c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[1,3]*l[0]/24 - c[1,3]*l[1]*np.tan(g[0])/12 - c[1,3]*l[2]*np.tan(g[1])/24 - c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[1,5]*l[2]/24 + c[1,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[3,1]*l[0]/24 - c[3,1]*l[1]*np.tan(g[0])/12 + c[3,1]*l[2]*np.tan(g[1])/24 - c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,3]*l[0]*l[1]/(18*l[2]) - c[3,3]*l[1]*np.tan(g[1])/6 - c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[3,5]*l[1]/12 + c[3,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,1]*l[2]/24 + c[5,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[5,3]*l[1]/12 + c[5,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,5]*l[1]*l[2]/(18*l[0]),
           -c[1,2]*l[0]/24 - c[1,2]*l[1]*np.tan(g[0])/12 - c[1,2]*l[2]*np.tan(g[1])/24 - c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[1,3]*l[0]*l[2]/(36*l[1]) - c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[1,4]*l[2]/24 + c[1,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[3,2]*l[0]*l[1]/(18*l[2]) - c[3,2]*l[1]*np.tan(g[1])/6 - c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[3,3]*l[0]/24 - c[3,3]*l[1]*np.tan(g[0])/12 + c[3,3]*l[2]*np.tan(g[1])/24 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,4]*l[1]/12 + c[3,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,2]*l[1]/12 + c[5,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,3]*l[2]/24 + c[5,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[5,4]*l[1]*l[2]/(18*l[0]),
           (2*l[0]**2*(-3*c[1,4]*l[1]*l[2] + 2*c[1,5]*l[2]**2 - 4*c[3,4]*l[1]**2 + 3*c[3,5]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(-c[1,0]*l[2] - 2*c[1,4]*l[1]*np.tan(g[0]) + c[1,4]*l[2]*np.tan(g[1]) + 2*c[1,5]*l[2]*np.tan(g[0]) - 2*c[3,0]*l[1] + 2*c[3,5]*l[1]*np.tan(g[0]) + c[3,5]*l[2]*np.tan(g[1]) + 2*c[5,4]*l[1] - c[5,5]*l[2]) + 4*l[1]**2*l[2]**2*(-c[1,0]*np.tan(g[0]) + c[1,4]*np.tan(g[0])*np.tan(g[1]) + c[1,5]*np.tan(g[0])**2 - c[3,0]*np.tan(g[1]) + c[3,4]*np.tan(g[1])**2 + c[3,5]*np.tan(g[0])*np.tan(g[1]) + c[5,0] - c[5,4]*np.tan(g[1]) - c[5,5]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(2*c[1,1]*l[2]**2 - 3*c[1,3]*l[1]*l[2] + 3*c[3,1]*l[1]*l[2] - 4*c[3,3]*l[1]**2) + 3*l[0]*l[1]*l[2]*(2*c[1,1]*l[2]*np.tan(g[0]) - 2*c[1,3]*l[1]*np.tan(g[0]) + c[1,3]*l[2]*np.tan(g[1]) - c[1,5]*l[2] + 2*c[3,1]*l[1]*np.tan(g[0]) + c[3,1]*l[2]*np.tan(g[1]) - 2*c[3,5]*l[1] - c[5,1]*l[2] + 2*c[5,3]*l[1]) + 4*l[1]**2*l[2]**2*(c[1,1]*np.tan(g[0])**2 + c[1,3]*np.tan(g[0])*np.tan(g[1]) - c[1,5]*np.tan(g[0]) + c[3,1]*np.tan(g[0])*np.tan(g[1]) + c[3,3]*np.tan(g[1])**2 - c[3,5]*np.tan(g[1]) - c[5,1]*np.tan(g[0]) - c[5,3]*np.tan(g[1]) + c[5,5]))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(-3*c[1,2]*l[1]*l[2] + 2*c[1,3]*l[2]**2 - 4*c[3,2]*l[1]**2 + 3*c[3,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(-2*c[1,2]*l[1]*np.tan(g[0]) + c[1,2]*l[2]*np.tan(g[1]) + 2*c[1,3]*l[2]*np.tan(g[0]) - c[1,4]*l[2] + 2*c[3,3]*l[1]*np.tan(g[0]) + c[3,3]*l[2]*np.tan(g[1]) - 2*c[3,4]*l[1] + 2*c[5,2]*l[1] - c[5,3]*l[2]) + 4*l[1]**2*l[2]**2*(c[1,2]*np.tan(g[0])*np.tan(g[1]) + c[1,3]*np.tan(g[0])**2 - c[1,4]*np.tan(g[0]) + c[3,2]*np.tan(g[1])**2 + c[3,3]*np.tan(g[0])*np.tan(g[1]) - c[3,4]*np.tan(g[1]) - c[5,2]*np.tan(g[1]) - c[5,3]*np.tan(g[0]) + c[5,4]))/(72*l[0]*l[1]*l[2]),
           (-2*l[0]**2*(3*c[1,4]*l[1]*l[2] + 2*c[1,5]*l[2]**2 + 2*c[3,4]*l[1]**2 + 3*c[3,5]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(-c[1,0]*l[2] - c[1,4]*l[1]*np.tan(g[0]) + c[1,4]*l[2]*np.tan(g[1]) - c[3,0]*l[1] + c[3,5]*l[1]*np.tan(g[0]) - c[3,5]*l[2]*np.tan(g[1]) + c[5,4]*l[1] + c[5,5]*l[2]) + 2*l[1]**2*l[2]**2*(-c[1,0]*np.tan(g[0]) + c[1,4]*np.tan(g[0])*np.tan(g[1]) + c[1,5]*np.tan(g[0])**2 - c[3,0]*np.tan(g[1]) + c[3,4]*np.tan(g[1])**2 + c[3,5]*np.tan(g[0])*np.tan(g[1]) + c[5,0] - c[5,4]*np.tan(g[1]) - c[5,5]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (-2*l[0]**2*(2*c[1,1]*l[2]**2 + 3*c[1,3]*l[1]*l[2] + 3*c[3,1]*l[1]*l[2] + 2*c[3,3]*l[1]**2) + 3*l[0]*l[1]*l[2]*(-c[1,3]*l[1]*np.tan(g[0]) + c[1,3]*l[2]*np.tan(g[1]) - c[1,5]*l[2] + c[3,1]*l[1]*np.tan(g[0]) - c[3,1]*l[2]*np.tan(g[1]) - c[3,5]*l[1] + c[5,1]*l[2] + c[5,3]*l[1]) + 2*l[1]**2*l[2]**2*(c[1,1]*np.tan(g[0])**2 + c[1,3]*np.tan(g[0])*np.tan(g[1]) - c[1,5]*np.tan(g[0]) + c[3,1]*np.tan(g[0])*np.tan(g[1]) + c[3,3]*np.tan(g[1])**2 - c[3,5]*np.tan(g[1]) - c[5,1]*np.tan(g[0]) - c[5,3]*np.tan(g[1]) + c[5,5]))/(72*l[0]*l[1]*l[2]),
           (-2*l[0]**2*(3*c[1,2]*l[1]*l[2] + 2*c[1,3]*l[2]**2 + 2*c[3,2]*l[1]**2 + 3*c[3,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(-c[1,2]*l[1]*np.tan(g[0]) + c[1,2]*l[2]*np.tan(g[1]) - c[1,4]*l[2] + c[3,3]*l[1]*np.tan(g[0]) - c[3,3]*l[2]*np.tan(g[1]) - c[3,4]*l[1] + c[5,2]*l[1] + c[5,3]*l[2]) + 2*l[1]**2*l[2]**2*(c[1,2]*np.tan(g[0])*np.tan(g[1]) + c[1,3]*np.tan(g[0])**2 - c[1,4]*np.tan(g[0]) + c[3,2]*np.tan(g[1])**2 + c[3,3]*np.tan(g[0])*np.tan(g[1]) - c[3,4]*np.tan(g[1]) - c[5,2]*np.tan(g[1]) - c[5,3]*np.tan(g[0]) + c[5,4]))/(72*l[0]*l[1]*l[2]),
           c[1,0]*l[2]/24 + c[1,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[1,4]*l[0]/24 - c[1,4]*l[1]*np.tan(g[0])/24 - c[1,4]*l[2]*np.tan(g[1])/24 - c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[1,5]*l[0]*l[2]/(36*l[1]) - c[1,5]*l[2]*np.tan(g[0])/12 - c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[3,0]*l[1]/24 + c[3,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[3,4]*l[0]*l[1]/(36*l[2]) - c[3,4]*l[1]*np.tan(g[1])/12 - c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[3,5]*l[0]/24 - c[3,5]*l[1]*np.tan(g[0])/24 - c[3,5]*l[2]*np.tan(g[1])/24 - c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,0]*l[1]*l[2]/(36*l[0]) + c[5,4]*l[1]/24 + c[5,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[5,5]*l[2]/24 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           -c[1,1]*l[0]*l[2]/(36*l[1]) - c[1,1]*l[2]*np.tan(g[0])/12 - c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[1,3]*l[0]/24 - c[1,3]*l[1]*np.tan(g[0])/24 - c[1,3]*l[2]*np.tan(g[1])/24 - c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[1,5]*l[2]/24 + c[1,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[3,1]*l[0]/24 - c[3,1]*l[1]*np.tan(g[0])/24 - c[3,1]*l[2]*np.tan(g[1])/24 - c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,3]*l[0]*l[1]/(36*l[2]) - c[3,3]*l[1]*np.tan(g[1])/12 - c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[3,5]*l[1]/24 + c[3,5]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[5,1]*l[2]/24 + c[5,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[5,3]*l[1]/24 + c[5,3]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,5]*l[1]*l[2]/(36*l[0]),
           -c[1,2]*l[0]/24 - c[1,2]*l[1]*np.tan(g[0])/24 - c[1,2]*l[2]*np.tan(g[1])/24 - c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[1,3]*l[0]*l[2]/(36*l[1]) - c[1,3]*l[2]*np.tan(g[0])/12 - c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[1,4]*l[2]/24 + c[1,4]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[3,2]*l[0]*l[1]/(36*l[2]) - c[3,2]*l[1]*np.tan(g[1])/12 - c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[3,3]*l[0]/24 - c[3,3]*l[1]*np.tan(g[0])/24 - c[3,3]*l[2]*np.tan(g[1])/24 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[3,4]*l[1]/24 + c[3,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[5,2]*l[1]/24 + c[5,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[5,3]*l[2]/24 + c[5,3]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[5,4]*l[1]*l[2]/(36*l[0])],
          [c[2,0]*l[1]/12 + c[2,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[2,4]*l[0]*l[1]/(18*l[2]) - c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[2,5]*l[0]/24 - c[2,5]*l[1]*np.tan(g[0])/12 + c[2,5]*l[2]*np.tan(g[1])/12 - c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,0]*l[2]/12 + c[3,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[3,4]*l[0]/24 + c[3,4]*l[1]*np.tan(g[0])/12 - c[3,4]*l[2]*np.tan(g[1])/12 - c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,5]*l[0]*l[2]/(18*l[1]) - c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[4,0]*l[1]*l[2]/(9*l[0]) - c[4,4]*l[1]/12 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[4,5]*l[2]/12 + c[4,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           c[2,1]*l[0]/24 - c[2,1]*l[1]*np.tan(g[0])/12 + c[2,1]*l[2]*np.tan(g[1])/12 - c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[2,3]*l[0]*l[1]/(18*l[2]) - c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[2,5]*l[1]/12 + c[2,5]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[3,1]*l[0]*l[2]/(18*l[1]) - c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[3,3]*l[0]/24 + c[3,3]*l[1]*np.tan(g[0])/12 - c[3,3]*l[2]*np.tan(g[1])/12 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,5]*l[2]/12 + c[3,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[4,1]*l[2]/12 + c[4,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[4,3]*l[1]/12 + c[4,3]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[4,5]*l[1]*l[2]/(9*l[0]),
           c[2,2]*l[0]*l[1]/(18*l[2]) - c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[2,3]*l[0]/24 - c[2,3]*l[1]*np.tan(g[0])/12 + c[2,3]*l[2]*np.tan(g[1])/12 - c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[2,4]*l[1]/12 + c[2,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[3,2]*l[0]/24 + c[3,2]*l[1]*np.tan(g[0])/12 - c[3,2]*l[2]*np.tan(g[1])/12 - c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,3]*l[0]*l[2]/(18*l[1]) - c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[3,4]*l[2]/12 + c[3,4]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[4,2]*l[1]/12 + c[4,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[4,3]*l[2]/12 + c[4,3]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[4,4]*l[1]*l[2]/(9*l[0]),
           (l[0]**2*(4*c[2,4]*l[1]**2 + 3*c[2,5]*l[1]*l[2] + 3*c[3,4]*l[1]*l[2] + 4*c[3,5]*l[2]**2) + 3*l[0]*l[1]*l[2]*(-c[2,0]*l[1] + 2*c[2,4]*l[1]*np.tan(g[1]) + c[2,5]*l[1]*np.tan(g[0]) + c[2,5]*l[2]*np.tan(g[1]) - c[3,0]*l[2] + c[3,4]*l[1]*np.tan(g[0]) + c[3,4]*l[2]*np.tan(g[1]) + 2*c[3,5]*l[2]*np.tan(g[0]) - c[4,4]*l[1] - c[4,5]*l[2]) + 4*l[1]**2*l[2]**2*(-c[2,0]*np.tan(g[1]) + c[2,4]*np.tan(g[1])**2 + c[2,5]*np.tan(g[0])*np.tan(g[1]) - c[3,0]*np.tan(g[0]) + c[3,4]*np.tan(g[0])*np.tan(g[1]) + c[3,5]*np.tan(g[0])**2 + c[4,0] - c[4,4]*np.tan(g[1]) - c[4,5]*np.tan(g[0])))/(36*l[0]*l[1]*l[2]),
           (l[0]**2*(3*c[2,1]*l[1]*l[2] + 4*c[2,3]*l[1]**2 + 4*c[3,1]*l[2]**2 + 3*c[3,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(c[2,1]*l[1]*np.tan(g[0]) + c[2,1]*l[2]*np.tan(g[1]) + 2*c[2,3]*l[1]*np.tan(g[1]) - c[2,5]*l[1] + 2*c[3,1]*l[2]*np.tan(g[0]) + c[3,3]*l[1]*np.tan(g[0]) + c[3,3]*l[2]*np.tan(g[1]) - c[3,5]*l[2] - c[4,1]*l[2] - c[4,3]*l[1]) + 4*l[1]**2*l[2]**2*(c[2,1]*np.tan(g[0])*np.tan(g[1]) + c[2,3]*np.tan(g[1])**2 - c[2,5]*np.tan(g[1]) + c[3,1]*np.tan(g[0])**2 + c[3,3]*np.tan(g[0])*np.tan(g[1]) - c[3,5]*np.tan(g[0]) - c[4,1]*np.tan(g[0]) - c[4,3]*np.tan(g[1]) + c[4,5]))/(36*l[0]*l[1]*l[2]),
           (l[0]**2*(4*c[2,2]*l[1]**2 + 3*c[2,3]*l[1]*l[2] + 3*c[3,2]*l[1]*l[2] + 4*c[3,3]*l[2]**2) + 3*l[0]*l[1]*l[2]*(2*c[2,2]*l[1]*np.tan(g[1]) + c[2,3]*l[1]*np.tan(g[0]) + c[2,3]*l[2]*np.tan(g[1]) - c[2,4]*l[1] + c[3,2]*l[1]*np.tan(g[0]) + c[3,2]*l[2]*np.tan(g[1]) + 2*c[3,3]*l[2]*np.tan(g[0]) - c[3,4]*l[2] - c[4,2]*l[1] - c[4,3]*l[2]) + 4*l[1]**2*l[2]**2*(c[2,2]*np.tan(g[1])**2 + c[2,3]*np.tan(g[0])*np.tan(g[1]) - c[2,4]*np.tan(g[1]) + c[3,2]*np.tan(g[0])*np.tan(g[1]) + c[3,3]*np.tan(g[0])**2 - c[3,4]*np.tan(g[0]) - c[4,2]*np.tan(g[1]) - c[4,3]*np.tan(g[0]) + c[4,4]))/(36*l[0]*l[1]*l[2]),
           (2*l[0]**2*(2*c[2,4]*l[1]**2 - 3*c[2,5]*l[1]*l[2] + 3*c[3,4]*l[1]*l[2] - 4*c[3,5]*l[2]**2) + 3*l[0]*l[1]*l[2]*(-c[2,0]*l[1] + 2*c[2,4]*l[1]*np.tan(g[1]) + c[2,5]*l[1]*np.tan(g[0]) - 2*c[2,5]*l[2]*np.tan(g[1]) - 2*c[3,0]*l[2] + c[3,4]*l[1]*np.tan(g[0]) + 2*c[3,4]*l[2]*np.tan(g[1]) - c[4,4]*l[1] + 2*c[4,5]*l[2]) + 4*l[1]**2*l[2]**2*(-c[2,0]*np.tan(g[1]) + c[2,4]*np.tan(g[1])**2 + c[2,5]*np.tan(g[0])*np.tan(g[1]) - c[3,0]*np.tan(g[0]) + c[3,4]*np.tan(g[0])*np.tan(g[1]) + c[3,5]*np.tan(g[0])**2 + c[4,0] - c[4,4]*np.tan(g[1]) - c[4,5]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(-3*c[2,1]*l[1]*l[2] + 2*c[2,3]*l[1]**2 - 4*c[3,1]*l[2]**2 + 3*c[3,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(c[2,1]*l[1]*np.tan(g[0]) - 2*c[2,1]*l[2]*np.tan(g[1]) + 2*c[2,3]*l[1]*np.tan(g[1]) - c[2,5]*l[1] + c[3,3]*l[1]*np.tan(g[0]) + 2*c[3,3]*l[2]*np.tan(g[1]) - 2*c[3,5]*l[2] + 2*c[4,1]*l[2] - c[4,3]*l[1]) + 4*l[1]**2*l[2]**2*(c[2,1]*np.tan(g[0])*np.tan(g[1]) + c[2,3]*np.tan(g[1])**2 - c[2,5]*np.tan(g[1]) + c[3,1]*np.tan(g[0])**2 + c[3,3]*np.tan(g[0])*np.tan(g[1]) - c[3,5]*np.tan(g[0]) - c[4,1]*np.tan(g[0]) - c[4,3]*np.tan(g[1]) + c[4,5]))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(2*c[2,2]*l[1]**2 - 3*c[2,3]*l[1]*l[2] + 3*c[3,2]*l[1]*l[2] - 4*c[3,3]*l[2]**2) + 3*l[0]*l[1]*l[2]*(2*c[2,2]*l[1]*np.tan(g[1]) + c[2,3]*l[1]*np.tan(g[0]) - 2*c[2,3]*l[2]*np.tan(g[1]) - c[2,4]*l[1] + c[3,2]*l[1]*np.tan(g[0]) + 2*c[3,2]*l[2]*np.tan(g[1]) - 2*c[3,4]*l[2] - c[4,2]*l[1] + 2*c[4,3]*l[2]) + 4*l[1]**2*l[2]**2*(c[2,2]*np.tan(g[1])**2 + c[2,3]*np.tan(g[0])*np.tan(g[1]) - c[2,4]*np.tan(g[1]) + c[3,2]*np.tan(g[0])*np.tan(g[1]) + c[3,3]*np.tan(g[0])**2 - c[3,4]*np.tan(g[0]) - c[4,2]*np.tan(g[1]) - c[4,3]*np.tan(g[0]) + c[4,4]))/(72*l[0]*l[1]*l[2]),
           c[2,0]*l[1]/24 + c[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[2,4]*l[0]*l[1]/(36*l[2]) - c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[2,5]*l[0]/24 - c[2,5]*l[1]*np.tan(g[0])/24 - c[2,5]*l[2]*np.tan(g[1])/12 - c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,0]*l[2]/12 + c[3,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[3,4]*l[0]/24 + c[3,4]*l[1]*np.tan(g[0])/24 - c[3,4]*l[2]*np.tan(g[1])/12 - c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,5]*l[0]*l[2]/(18*l[1]) - c[3,5]*l[2]*np.tan(g[0])/6 - c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[4,0]*l[1]*l[2]/(18*l[0]) - c[4,4]*l[1]/24 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,5]*l[2]/12 + c[4,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[2,1]*l[0]/24 - c[2,1]*l[1]*np.tan(g[0])/24 - c[2,1]*l[2]*np.tan(g[1])/12 - c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[2,3]*l[0]*l[1]/(36*l[2]) - c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[2,5]*l[1]/24 + c[2,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[3,1]*l[0]*l[2]/(18*l[1]) - c[3,1]*l[2]*np.tan(g[0])/6 - c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[3,3]*l[0]/24 + c[3,3]*l[1]*np.tan(g[0])/24 - c[3,3]*l[2]*np.tan(g[1])/12 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,5]*l[2]/12 + c[3,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,1]*l[2]/12 + c[4,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,3]*l[1]/24 + c[4,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,5]*l[1]*l[2]/(18*l[0]),
           c[2,2]*l[0]*l[1]/(36*l[2]) - c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[2,3]*l[0]/24 - c[2,3]*l[1]*np.tan(g[0])/24 - c[2,3]*l[2]*np.tan(g[1])/12 - c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[2,4]*l[1]/24 + c[2,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[3,2]*l[0]/24 + c[3,2]*l[1]*np.tan(g[0])/24 - c[3,2]*l[2]*np.tan(g[1])/12 - c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,3]*l[0]*l[2]/(18*l[1]) - c[3,3]*l[2]*np.tan(g[0])/6 - c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[3,4]*l[2]/12 + c[3,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,2]*l[1]/24 + c[4,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,3]*l[2]/12 + c[4,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,4]*l[1]*l[2]/(18*l[0]),
           c[2,0]*l[1]/12 + c[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[2,4]*l[0]*l[1]/(18*l[2]) - c[2,4]*l[1]*np.tan(g[1])/6 - c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[2,5]*l[0]/24 - c[2,5]*l[1]*np.tan(g[0])/12 + c[2,5]*l[2]*np.tan(g[1])/24 - c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,0]*l[2]/24 + c[3,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[3,4]*l[0]/24 - c[3,4]*l[1]*np.tan(g[0])/12 - c[3,4]*l[2]*np.tan(g[1])/24 - c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,5]*l[0]*l[2]/(36*l[1]) - c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[4,0]*l[1]*l[2]/(18*l[0]) + c[4,4]*l[1]/12 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,5]*l[2]/24 + c[4,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[2,1]*l[0]/24 - c[2,1]*l[1]*np.tan(g[0])/12 + c[2,1]*l[2]*np.tan(g[1])/24 - c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[2,3]*l[0]*l[1]/(18*l[2]) - c[2,3]*l[1]*np.tan(g[1])/6 - c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[2,5]*l[1]/12 + c[2,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[3,1]*l[0]*l[2]/(36*l[1]) - c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[3,3]*l[0]/24 - c[3,3]*l[1]*np.tan(g[0])/12 - c[3,3]*l[2]*np.tan(g[1])/24 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,5]*l[2]/24 + c[3,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,1]*l[2]/24 + c[4,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,3]*l[1]/12 + c[4,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,5]*l[1]*l[2]/(18*l[0]),
           -c[2,2]*l[0]*l[1]/(18*l[2]) - c[2,2]*l[1]*np.tan(g[1])/6 - c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[2,3]*l[0]/24 - c[2,3]*l[1]*np.tan(g[0])/12 + c[2,3]*l[2]*np.tan(g[1])/24 - c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[2,4]*l[1]/12 + c[2,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[3,2]*l[0]/24 - c[3,2]*l[1]*np.tan(g[0])/12 - c[3,2]*l[2]*np.tan(g[1])/24 - c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,3]*l[0]*l[2]/(36*l[1]) - c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[3,4]*l[2]/24 + c[3,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,2]*l[1]/12 + c[4,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,3]*l[2]/24 + c[4,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,4]*l[1]*l[2]/(18*l[0]),
           (2*l[0]**2*(-4*c[2,4]*l[1]**2 + 3*c[2,5]*l[1]*l[2] - 3*c[3,4]*l[1]*l[2] + 2*c[3,5]*l[2]**2) + 3*l[0]*l[1]*l[2]*(-2*c[2,0]*l[1] + 2*c[2,5]*l[1]*np.tan(g[0]) + c[2,5]*l[2]*np.tan(g[1]) - c[3,0]*l[2] - 2*c[3,4]*l[1]*np.tan(g[0]) + c[3,4]*l[2]*np.tan(g[1]) + 2*c[3,5]*l[2]*np.tan(g[0]) + 2*c[4,4]*l[1] - c[4,5]*l[2]) + 4*l[1]**2*l[2]**2*(-c[2,0]*np.tan(g[1]) + c[2,4]*np.tan(g[1])**2 + c[2,5]*np.tan(g[0])*np.tan(g[1]) - c[3,0]*np.tan(g[0]) + c[3,4]*np.tan(g[0])*np.tan(g[1]) + c[3,5]*np.tan(g[0])**2 + c[4,0] - c[4,4]*np.tan(g[1]) - c[4,5]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(3*c[2,1]*l[1]*l[2] - 4*c[2,3]*l[1]**2 + 2*c[3,1]*l[2]**2 - 3*c[3,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(2*c[2,1]*l[1]*np.tan(g[0]) + c[2,1]*l[2]*np.tan(g[1]) - 2*c[2,5]*l[1] + 2*c[3,1]*l[2]*np.tan(g[0]) - 2*c[3,3]*l[1]*np.tan(g[0]) + c[3,3]*l[2]*np.tan(g[1]) - c[3,5]*l[2] - c[4,1]*l[2] + 2*c[4,3]*l[1]) + 4*l[1]**2*l[2]**2*(c[2,1]*np.tan(g[0])*np.tan(g[1]) + c[2,3]*np.tan(g[1])**2 - c[2,5]*np.tan(g[1]) + c[3,1]*np.tan(g[0])**2 + c[3,3]*np.tan(g[0])*np.tan(g[1]) - c[3,5]*np.tan(g[0]) - c[4,1]*np.tan(g[0]) - c[4,3]*np.tan(g[1]) + c[4,5]))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(-4*c[2,2]*l[1]**2 + 3*c[2,3]*l[1]*l[2] - 3*c[3,2]*l[1]*l[2] + 2*c[3,3]*l[2]**2) + 3*l[0]*l[1]*l[2]*(2*c[2,3]*l[1]*np.tan(g[0]) + c[2,3]*l[2]*np.tan(g[1]) - 2*c[2,4]*l[1] - 2*c[3,2]*l[1]*np.tan(g[0]) + c[3,2]*l[2]*np.tan(g[1]) + 2*c[3,3]*l[2]*np.tan(g[0]) - c[3,4]*l[2] + 2*c[4,2]*l[1] - c[4,3]*l[2]) + 4*l[1]**2*l[2]**2*(c[2,2]*np.tan(g[1])**2 + c[2,3]*np.tan(g[0])*np.tan(g[1]) - c[2,4]*np.tan(g[1]) + c[3,2]*np.tan(g[0])*np.tan(g[1]) + c[3,3]*np.tan(g[0])**2 - c[3,4]*np.tan(g[0]) - c[4,2]*np.tan(g[1]) - c[4,3]*np.tan(g[0]) + c[4,4]))/(72*l[0]*l[1]*l[2]),
           (-2*l[0]**2*(2*c[2,4]*l[1]**2 + 3*c[2,5]*l[1]*l[2] + 3*c[3,4]*l[1]*l[2] + 2*c[3,5]*l[2]**2) + 3*l[0]*l[1]*l[2]*(-c[2,0]*l[1] + c[2,5]*l[1]*np.tan(g[0]) - c[2,5]*l[2]*np.tan(g[1]) - c[3,0]*l[2] - c[3,4]*l[1]*np.tan(g[0]) + c[3,4]*l[2]*np.tan(g[1]) + c[4,4]*l[1] + c[4,5]*l[2]) + 2*l[1]**2*l[2]**2*(-c[2,0]*np.tan(g[1]) + c[2,4]*np.tan(g[1])**2 + c[2,5]*np.tan(g[0])*np.tan(g[1]) - c[3,0]*np.tan(g[0]) + c[3,4]*np.tan(g[0])*np.tan(g[1]) + c[3,5]*np.tan(g[0])**2 + c[4,0] - c[4,4]*np.tan(g[1]) - c[4,5]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (-2*l[0]**2*(3*c[2,1]*l[1]*l[2] + 2*c[2,3]*l[1]**2 + 2*c[3,1]*l[2]**2 + 3*c[3,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(c[2,1]*l[1]*np.tan(g[0]) - c[2,1]*l[2]*np.tan(g[1]) - c[2,5]*l[1] - c[3,3]*l[1]*np.tan(g[0]) + c[3,3]*l[2]*np.tan(g[1]) - c[3,5]*l[2] + c[4,1]*l[2] + c[4,3]*l[1]) + 2*l[1]**2*l[2]**2*(c[2,1]*np.tan(g[0])*np.tan(g[1]) + c[2,3]*np.tan(g[1])**2 - c[2,5]*np.tan(g[1]) + c[3,1]*np.tan(g[0])**2 + c[3,3]*np.tan(g[0])*np.tan(g[1]) - c[3,5]*np.tan(g[0]) - c[4,1]*np.tan(g[0]) - c[4,3]*np.tan(g[1]) + c[4,5]))/(72*l[0]*l[1]*l[2]),
           (-2*l[0]**2*(2*c[2,2]*l[1]**2 + 3*c[2,3]*l[1]*l[2] + 3*c[3,2]*l[1]*l[2] + 2*c[3,3]*l[2]**2) + 3*l[0]*l[1]*l[2]*(c[2,3]*l[1]*np.tan(g[0]) - c[2,3]*l[2]*np.tan(g[1]) - c[2,4]*l[1] - c[3,2]*l[1]*np.tan(g[0]) + c[3,2]*l[2]*np.tan(g[1]) - c[3,4]*l[2] + c[4,2]*l[1] + c[4,3]*l[2]) + 2*l[1]**2*l[2]**2*(c[2,2]*np.tan(g[1])**2 + c[2,3]*np.tan(g[0])*np.tan(g[1]) - c[2,4]*np.tan(g[1]) + c[3,2]*np.tan(g[0])*np.tan(g[1]) + c[3,3]*np.tan(g[0])**2 - c[3,4]*np.tan(g[0]) - c[4,2]*np.tan(g[1]) - c[4,3]*np.tan(g[0]) + c[4,4]))/(72*l[0]*l[1]*l[2]),
           c[2,0]*l[1]/24 + c[2,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[2,4]*l[0]*l[1]/(36*l[2]) - c[2,4]*l[1]*np.tan(g[1])/12 - c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[2,5]*l[0]/24 - c[2,5]*l[1]*np.tan(g[0])/24 - c[2,5]*l[2]*np.tan(g[1])/24 - c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[3,0]*l[2]/24 + c[3,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[3,4]*l[0]/24 - c[3,4]*l[1]*np.tan(g[0])/24 - c[3,4]*l[2]*np.tan(g[1])/24 - c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,5]*l[0]*l[2]/(36*l[1]) - c[3,5]*l[2]*np.tan(g[0])/12 - c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[4,0]*l[1]*l[2]/(36*l[0]) + c[4,4]*l[1]/24 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[4,5]*l[2]/24 + c[4,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           -c[2,1]*l[0]/24 - c[2,1]*l[1]*np.tan(g[0])/24 - c[2,1]*l[2]*np.tan(g[1])/24 - c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[2,3]*l[0]*l[1]/(36*l[2]) - c[2,3]*l[1]*np.tan(g[1])/12 - c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[2,5]*l[1]/24 + c[2,5]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[3,1]*l[0]*l[2]/(36*l[1]) - c[3,1]*l[2]*np.tan(g[0])/12 - c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[3,3]*l[0]/24 - c[3,3]*l[1]*np.tan(g[0])/24 - c[3,3]*l[2]*np.tan(g[1])/24 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[3,5]*l[2]/24 + c[3,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[4,1]*l[2]/24 + c[4,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[4,3]*l[1]/24 + c[4,3]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[4,5]*l[1]*l[2]/(36*l[0]),
           -c[2,2]*l[0]*l[1]/(36*l[2]) - c[2,2]*l[1]*np.tan(g[1])/12 - c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[2,3]*l[0]/24 - c[2,3]*l[1]*np.tan(g[0])/24 - c[2,3]*l[2]*np.tan(g[1])/24 - c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[2,4]*l[1]/24 + c[2,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[3,2]*l[0]/24 - c[3,2]*l[1]*np.tan(g[0])/24 - c[3,2]*l[2]*np.tan(g[1])/24 - c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,3]*l[0]*l[2]/(36*l[1]) - c[3,3]*l[2]*np.tan(g[0])/12 - c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[3,4]*l[2]/24 + c[3,4]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[4,2]*l[1]/24 + c[4,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[4,3]*l[2]/24 + c[4,3]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[4,4]*l[1]*l[2]/(36*l[0])],
          [-c[0,0]*l[1]*l[2]/(18*l[0]) - c[0,4]*l[1]/24 + c[0,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[0,5]*l[2]/12 + c[0,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,0]*l[1]/24 + c[4,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,4]*l[0]*l[1]/(36*l[2]) - c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[4,5]*l[0]/24 - c[4,5]*l[1]*np.tan(g[0])/24 + c[4,5]*l[2]*np.tan(g[1])/12 - c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,0]*l[2]/12 + c[5,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[5,4]*l[0]/24 + c[5,4]*l[1]*np.tan(g[0])/24 + c[5,4]*l[2]*np.tan(g[1])/12 - c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,5]*l[0]*l[2]/(18*l[1]) + c[5,5]*l[2]*np.tan(g[0])/6 - c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]),
           -c[0,1]*l[2]/12 + c[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[0,3]*l[1]/24 + c[0,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[0,5]*l[1]*l[2]/(18*l[0]) + c[4,1]*l[0]/24 - c[4,1]*l[1]*np.tan(g[0])/24 + c[4,1]*l[2]*np.tan(g[1])/12 - c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[4,3]*l[0]*l[1]/(36*l[2]) - c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[4,5]*l[1]/24 + c[4,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,1]*l[0]*l[2]/(18*l[1]) + c[5,1]*l[2]*np.tan(g[0])/6 - c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[5,3]*l[0]/24 + c[5,3]*l[1]*np.tan(g[0])/24 + c[5,3]*l[2]*np.tan(g[1])/12 - c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,5]*l[2]/12 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[0,2]*l[1]/24 + c[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[0,3]*l[2]/12 + c[0,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[0,4]*l[1]*l[2]/(18*l[0]) + c[4,2]*l[0]*l[1]/(36*l[2]) - c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[4,3]*l[0]/24 - c[4,3]*l[1]*np.tan(g[0])/24 + c[4,3]*l[2]*np.tan(g[1])/12 - c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[4,4]*l[1]/24 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,2]*l[0]/24 + c[5,2]*l[1]*np.tan(g[0])/24 + c[5,2]*l[2]*np.tan(g[1])/12 - c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,3]*l[0]*l[2]/(18*l[1]) + c[5,3]*l[2]*np.tan(g[0])/6 - c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[5,4]*l[2]/12 + c[5,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           (2*l[0]**2*(2*c[4,4]*l[1]**2 + 3*c[4,5]*l[1]*l[2] - 3*c[5,4]*l[1]*l[2] - 4*c[5,5]*l[2]**2) + 3*l[0]*l[1]*l[2]*(-c[0,4]*l[1] - 2*c[0,5]*l[2] - c[4,0]*l[1] + 2*c[4,4]*l[1]*np.tan(g[1]) + c[4,5]*l[1]*np.tan(g[0]) + 2*c[4,5]*l[2]*np.tan(g[1]) + 2*c[5,0]*l[2] + c[5,4]*l[1]*np.tan(g[0]) - 2*c[5,4]*l[2]*np.tan(g[1])) + 4*l[1]**2*l[2]**2*(c[0,0] - c[0,4]*np.tan(g[1]) - c[0,5]*np.tan(g[0]) - c[4,0]*np.tan(g[1]) + c[4,4]*np.tan(g[1])**2 + c[4,5]*np.tan(g[0])*np.tan(g[1]) - c[5,0]*np.tan(g[0]) + c[5,4]*np.tan(g[0])*np.tan(g[1]) + c[5,5]*np.tan(g[0])**2))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(3*c[4,1]*l[1]*l[2] + 2*c[4,3]*l[1]**2 - 4*c[5,1]*l[2]**2 - 3*c[5,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(-2*c[0,1]*l[2] - c[0,3]*l[1] + c[4,1]*l[1]*np.tan(g[0]) + 2*c[4,1]*l[2]*np.tan(g[1]) + 2*c[4,3]*l[1]*np.tan(g[1]) - c[4,5]*l[1] + c[5,3]*l[1]*np.tan(g[0]) - 2*c[5,3]*l[2]*np.tan(g[1]) + 2*c[5,5]*l[2]) + 4*l[1]**2*l[2]**2*(-c[0,1]*np.tan(g[0]) - c[0,3]*np.tan(g[1]) + c[0,5] + c[4,1]*np.tan(g[0])*np.tan(g[1]) + c[4,3]*np.tan(g[1])**2 - c[4,5]*np.tan(g[1]) + c[5,1]*np.tan(g[0])**2 + c[5,3]*np.tan(g[0])*np.tan(g[1]) - c[5,5]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(2*c[4,2]*l[1]**2 + 3*c[4,3]*l[1]*l[2] - 3*c[5,2]*l[1]*l[2] - 4*c[5,3]*l[2]**2) + 3*l[0]*l[1]*l[2]*(-c[0,2]*l[1] - 2*c[0,3]*l[2] + 2*c[4,2]*l[1]*np.tan(g[1]) + c[4,3]*l[1]*np.tan(g[0]) + 2*c[4,3]*l[2]*np.tan(g[1]) - c[4,4]*l[1] + c[5,2]*l[1]*np.tan(g[0]) - 2*c[5,2]*l[2]*np.tan(g[1]) + 2*c[5,4]*l[2]) + 4*l[1]**2*l[2]**2*(-c[0,2]*np.tan(g[1]) - c[0,3]*np.tan(g[0]) + c[0,4] + c[4,2]*np.tan(g[1])**2 + c[4,3]*np.tan(g[0])*np.tan(g[1]) - c[4,4]*np.tan(g[1]) + c[5,2]*np.tan(g[0])*np.tan(g[1]) + c[5,3]*np.tan(g[0])**2 - c[5,4]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (l[0]**2*(4*c[4,4]*l[1]**2 - 3*c[4,5]*l[1]*l[2] - 3*c[5,4]*l[1]*l[2] + 4*c[5,5]*l[2]**2) + 3*l[0]*l[1]*l[2]*(-c[0,4]*l[1] + c[0,5]*l[2] - c[4,0]*l[1] + 2*c[4,4]*l[1]*np.tan(g[1]) + c[4,5]*l[1]*np.tan(g[0]) - c[4,5]*l[2]*np.tan(g[1]) + c[5,0]*l[2] + c[5,4]*l[1]*np.tan(g[0]) - c[5,4]*l[2]*np.tan(g[1]) - 2*c[5,5]*l[2]*np.tan(g[0])) + 4*l[1]**2*l[2]**2*(c[0,0] - c[0,4]*np.tan(g[1]) - c[0,5]*np.tan(g[0]) - c[4,0]*np.tan(g[1]) + c[4,4]*np.tan(g[1])**2 + c[4,5]*np.tan(g[0])*np.tan(g[1]) - c[5,0]*np.tan(g[0]) + c[5,4]*np.tan(g[0])*np.tan(g[1]) + c[5,5]*np.tan(g[0])**2))/(36*l[0]*l[1]*l[2]),
           (l[0]**2*(-3*c[4,1]*l[1]*l[2] + 4*c[4,3]*l[1]**2 + 4*c[5,1]*l[2]**2 - 3*c[5,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(c[0,1]*l[2] - c[0,3]*l[1] + c[4,1]*l[1]*np.tan(g[0]) - c[4,1]*l[2]*np.tan(g[1]) + 2*c[4,3]*l[1]*np.tan(g[1]) - c[4,5]*l[1] - 2*c[5,1]*l[2]*np.tan(g[0]) + c[5,3]*l[1]*np.tan(g[0]) - c[5,3]*l[2]*np.tan(g[1]) + c[5,5]*l[2]) + 4*l[1]**2*l[2]**2*(-c[0,1]*np.tan(g[0]) - c[0,3]*np.tan(g[1]) + c[0,5] + c[4,1]*np.tan(g[0])*np.tan(g[1]) + c[4,3]*np.tan(g[1])**2 - c[4,5]*np.tan(g[1]) + c[5,1]*np.tan(g[0])**2 + c[5,3]*np.tan(g[0])*np.tan(g[1]) - c[5,5]*np.tan(g[0])))/(36*l[0]*l[1]*l[2]),
           (l[0]**2*(4*c[4,2]*l[1]**2 - 3*c[4,3]*l[1]*l[2] - 3*c[5,2]*l[1]*l[2] + 4*c[5,3]*l[2]**2) + 3*l[0]*l[1]*l[2]*(-c[0,2]*l[1] + c[0,3]*l[2] + 2*c[4,2]*l[1]*np.tan(g[1]) + c[4,3]*l[1]*np.tan(g[0]) - c[4,3]*l[2]*np.tan(g[1]) - c[4,4]*l[1] + c[5,2]*l[1]*np.tan(g[0]) - c[5,2]*l[2]*np.tan(g[1]) - 2*c[5,3]*l[2]*np.tan(g[0]) + c[5,4]*l[2]) + 4*l[1]**2*l[2]**2*(-c[0,2]*np.tan(g[1]) - c[0,3]*np.tan(g[0]) + c[0,4] + c[4,2]*np.tan(g[1])**2 + c[4,3]*np.tan(g[0])*np.tan(g[1]) - c[4,4]*np.tan(g[1]) + c[5,2]*np.tan(g[0])*np.tan(g[1]) + c[5,3]*np.tan(g[0])**2 - c[5,4]*np.tan(g[0])))/(36*l[0]*l[1]*l[2]),
           -c[0,0]*l[1]*l[2]/(9*l[0]) - c[0,4]*l[1]/12 + c[0,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[0,5]*l[2]/12 + c[0,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[4,0]*l[1]/12 + c[4,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[4,4]*l[0]*l[1]/(18*l[2]) - c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[4,5]*l[0]/24 - c[4,5]*l[1]*np.tan(g[0])/12 - c[4,5]*l[2]*np.tan(g[1])/12 - c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[5,0]*l[2]/12 + c[5,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[5,4]*l[0]/24 + c[5,4]*l[1]*np.tan(g[0])/12 + c[5,4]*l[2]*np.tan(g[1])/12 - c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[5,5]*l[0]*l[2]/(18*l[1]) - c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]),
           c[0,1]*l[2]/12 + c[0,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[0,3]*l[1]/12 + c[0,3]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[0,5]*l[1]*l[2]/(9*l[0]) - c[4,1]*l[0]/24 - c[4,1]*l[1]*np.tan(g[0])/12 - c[4,1]*l[2]*np.tan(g[1])/12 - c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[4,3]*l[0]*l[1]/(18*l[2]) - c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[4,5]*l[1]/12 + c[4,5]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,1]*l[0]*l[2]/(18*l[1]) - c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[5,3]*l[0]/24 + c[5,3]*l[1]*np.tan(g[0])/12 + c[5,3]*l[2]*np.tan(g[1])/12 - c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[5,5]*l[2]/12 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           -c[0,2]*l[1]/12 + c[0,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[0,3]*l[2]/12 + c[0,3]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[0,4]*l[1]*l[2]/(9*l[0]) + c[4,2]*l[0]*l[1]/(18*l[2]) - c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[4,3]*l[0]/24 - c[4,3]*l[1]*np.tan(g[0])/12 - c[4,3]*l[2]*np.tan(g[1])/12 - c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[4,4]*l[1]/12 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[5,2]*l[0]/24 + c[5,2]*l[1]*np.tan(g[0])/12 + c[5,2]*l[2]*np.tan(g[1])/12 - c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[5,3]*l[0]*l[2]/(18*l[1]) - c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[5,4]*l[2]/12 + c[5,4]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           -c[0,0]*l[1]*l[2]/(36*l[0]) + c[0,4]*l[1]/24 + c[0,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[0,5]*l[2]/24 + c[0,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[4,0]*l[1]/24 + c[4,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[4,4]*l[0]*l[1]/(36*l[2]) - c[4,4]*l[1]*np.tan(g[1])/12 - c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[4,5]*l[0]/24 - c[4,5]*l[1]*np.tan(g[0])/24 + c[4,5]*l[2]*np.tan(g[1])/24 - c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,0]*l[2]/24 + c[5,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[5,4]*l[0]/24 - c[5,4]*l[1]*np.tan(g[0])/24 + c[5,4]*l[2]*np.tan(g[1])/24 - c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,5]*l[0]*l[2]/(36*l[1]) + c[5,5]*l[2]*np.tan(g[0])/12 - c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]),
           -c[0,1]*l[2]/24 + c[0,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[0,3]*l[1]/24 + c[0,3]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[0,5]*l[1]*l[2]/(36*l[0]) + c[4,1]*l[0]/24 - c[4,1]*l[1]*np.tan(g[0])/24 + c[4,1]*l[2]*np.tan(g[1])/24 - c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[4,3]*l[0]*l[1]/(36*l[2]) - c[4,3]*l[1]*np.tan(g[1])/12 - c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[4,5]*l[1]/24 + c[4,5]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,1]*l[0]*l[2]/(36*l[1]) + c[5,1]*l[2]*np.tan(g[0])/12 - c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[5,3]*l[0]/24 - c[5,3]*l[1]*np.tan(g[0])/24 + c[5,3]*l[2]*np.tan(g[1])/24 - c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,5]*l[2]/24 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           c[0,2]*l[1]/24 + c[0,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[0,3]*l[2]/24 + c[0,3]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[0,4]*l[1]*l[2]/(36*l[0]) - c[4,2]*l[0]*l[1]/(36*l[2]) - c[4,2]*l[1]*np.tan(g[1])/12 - c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[4,3]*l[0]/24 - c[4,3]*l[1]*np.tan(g[0])/24 + c[4,3]*l[2]*np.tan(g[1])/24 - c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[4,4]*l[1]/24 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[5,2]*l[0]/24 - c[5,2]*l[1]*np.tan(g[0])/24 + c[5,2]*l[2]*np.tan(g[1])/24 - c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,3]*l[0]*l[2]/(36*l[1]) + c[5,3]*l[2]*np.tan(g[0])/12 - c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[5,4]*l[2]/24 + c[5,4]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           (2*l[0]**2*(-2*c[4,4]*l[1]**2 + 3*c[4,5]*l[1]*l[2] + 3*c[5,4]*l[1]*l[2] - 2*c[5,5]*l[2]**2) + 3*l[0]*l[1]*l[2]*(c[0,4]*l[1] - c[0,5]*l[2] - c[4,0]*l[1] + c[4,5]*l[1]*np.tan(g[0]) + c[4,5]*l[2]*np.tan(g[1]) + c[5,0]*l[2] - c[5,4]*l[1]*np.tan(g[0]) - c[5,4]*l[2]*np.tan(g[1])) + 2*l[1]**2*l[2]**2*(c[0,0] - c[0,4]*np.tan(g[1]) - c[0,5]*np.tan(g[0]) - c[4,0]*np.tan(g[1]) + c[4,4]*np.tan(g[1])**2 + c[4,5]*np.tan(g[0])*np.tan(g[1]) - c[5,0]*np.tan(g[0]) + c[5,4]*np.tan(g[0])*np.tan(g[1]) + c[5,5]*np.tan(g[0])**2))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(3*c[4,1]*l[1]*l[2] - 2*c[4,3]*l[1]**2 - 2*c[5,1]*l[2]**2 + 3*c[5,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(-c[0,1]*l[2] + c[0,3]*l[1] + c[4,1]*l[1]*np.tan(g[0]) + c[4,1]*l[2]*np.tan(g[1]) - c[4,5]*l[1] - c[5,3]*l[1]*np.tan(g[0]) - c[5,3]*l[2]*np.tan(g[1]) + c[5,5]*l[2]) + 2*l[1]**2*l[2]**2*(-c[0,1]*np.tan(g[0]) - c[0,3]*np.tan(g[1]) + c[0,5] + c[4,1]*np.tan(g[0])*np.tan(g[1]) + c[4,3]*np.tan(g[1])**2 - c[4,5]*np.tan(g[1]) + c[5,1]*np.tan(g[0])**2 + c[5,3]*np.tan(g[0])*np.tan(g[1]) - c[5,5]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(-2*c[4,2]*l[1]**2 + 3*c[4,3]*l[1]*l[2] + 3*c[5,2]*l[1]*l[2] - 2*c[5,3]*l[2]**2) + 3*l[0]*l[1]*l[2]*(c[0,2]*l[1] - c[0,3]*l[2] + c[4,3]*l[1]*np.tan(g[0]) + c[4,3]*l[2]*np.tan(g[1]) - c[4,4]*l[1] - c[5,2]*l[1]*np.tan(g[0]) - c[5,2]*l[2]*np.tan(g[1]) + c[5,4]*l[2]) + 2*l[1]**2*l[2]**2*(-c[0,2]*np.tan(g[1]) - c[0,3]*np.tan(g[0]) + c[0,4] + c[4,2]*np.tan(g[1])**2 + c[4,3]*np.tan(g[0])*np.tan(g[1]) - c[4,4]*np.tan(g[1]) + c[5,2]*np.tan(g[0])*np.tan(g[1]) + c[5,3]*np.tan(g[0])**2 - c[5,4]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(-4*c[4,4]*l[1]**2 - 3*c[4,5]*l[1]*l[2] + 3*c[5,4]*l[1]*l[2] + 2*c[5,5]*l[2]**2) + 3*l[0]*l[1]*l[2]*(2*c[0,4]*l[1] + c[0,5]*l[2] - 2*c[4,0]*l[1] + 2*c[4,5]*l[1]*np.tan(g[0]) - c[4,5]*l[2]*np.tan(g[1]) + c[5,0]*l[2] - 2*c[5,4]*l[1]*np.tan(g[0]) - c[5,4]*l[2]*np.tan(g[1]) - 2*c[5,5]*l[2]*np.tan(g[0])) + 4*l[1]**2*l[2]**2*(c[0,0] - c[0,4]*np.tan(g[1]) - c[0,5]*np.tan(g[0]) - c[4,0]*np.tan(g[1]) + c[4,4]*np.tan(g[1])**2 + c[4,5]*np.tan(g[0])*np.tan(g[1]) - c[5,0]*np.tan(g[0]) + c[5,4]*np.tan(g[0])*np.tan(g[1]) + c[5,5]*np.tan(g[0])**2))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(-3*c[4,1]*l[1]*l[2] - 4*c[4,3]*l[1]**2 + 2*c[5,1]*l[2]**2 + 3*c[5,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(c[0,1]*l[2] + 2*c[0,3]*l[1] + 2*c[4,1]*l[1]*np.tan(g[0]) - c[4,1]*l[2]*np.tan(g[1]) - 2*c[4,5]*l[1] - 2*c[5,1]*l[2]*np.tan(g[0]) - 2*c[5,3]*l[1]*np.tan(g[0]) - c[5,3]*l[2]*np.tan(g[1]) + c[5,5]*l[2]) + 4*l[1]**2*l[2]**2*(-c[0,1]*np.tan(g[0]) - c[0,3]*np.tan(g[1]) + c[0,5] + c[4,1]*np.tan(g[0])*np.tan(g[1]) + c[4,3]*np.tan(g[1])**2 - c[4,5]*np.tan(g[1]) + c[5,1]*np.tan(g[0])**2 + c[5,3]*np.tan(g[0])*np.tan(g[1]) - c[5,5]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(-4*c[4,2]*l[1]**2 - 3*c[4,3]*l[1]*l[2] + 3*c[5,2]*l[1]*l[2] + 2*c[5,3]*l[2]**2) + 3*l[0]*l[1]*l[2]*(2*c[0,2]*l[1] + c[0,3]*l[2] + 2*c[4,3]*l[1]*np.tan(g[0]) - c[4,3]*l[2]*np.tan(g[1]) - 2*c[4,4]*l[1] - 2*c[5,2]*l[1]*np.tan(g[0]) - c[5,2]*l[2]*np.tan(g[1]) - 2*c[5,3]*l[2]*np.tan(g[0]) + c[5,4]*l[2]) + 4*l[1]**2*l[2]**2*(-c[0,2]*np.tan(g[1]) - c[0,3]*np.tan(g[0]) + c[0,4] + c[4,2]*np.tan(g[1])**2 + c[4,3]*np.tan(g[0])*np.tan(g[1]) - c[4,4]*np.tan(g[1]) + c[5,2]*np.tan(g[0])*np.tan(g[1]) + c[5,3]*np.tan(g[0])**2 - c[5,4]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           -c[0,0]*l[1]*l[2]/(18*l[0]) + c[0,4]*l[1]/12 + c[0,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[0,5]*l[2]/24 + c[0,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,0]*l[1]/12 + c[4,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,4]*l[0]*l[1]/(18*l[2]) - c[4,4]*l[1]*np.tan(g[1])/6 - c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[4,5]*l[0]/24 - c[4,5]*l[1]*np.tan(g[0])/12 - c[4,5]*l[2]*np.tan(g[1])/24 - c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,0]*l[2]/24 + c[5,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[5,4]*l[0]/24 - c[5,4]*l[1]*np.tan(g[0])/12 + c[5,4]*l[2]*np.tan(g[1])/24 - c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,5]*l[0]*l[2]/(36*l[1]) - c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]),
           c[0,1]*l[2]/24 + c[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[0,3]*l[1]/12 + c[0,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[0,5]*l[1]*l[2]/(18*l[0]) - c[4,1]*l[0]/24 - c[4,1]*l[1]*np.tan(g[0])/12 - c[4,1]*l[2]*np.tan(g[1])/24 - c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[4,3]*l[0]*l[1]/(18*l[2]) - c[4,3]*l[1]*np.tan(g[1])/6 - c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[4,5]*l[1]/12 + c[4,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,1]*l[0]*l[2]/(36*l[1]) - c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[5,3]*l[0]/24 - c[5,3]*l[1]*np.tan(g[0])/12 + c[5,3]*l[2]*np.tan(g[1])/24 - c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,5]*l[2]/24 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[0,2]*l[1]/12 + c[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[0,3]*l[2]/24 + c[0,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[0,4]*l[1]*l[2]/(18*l[0]) - c[4,2]*l[0]*l[1]/(18*l[2]) - c[4,2]*l[1]*np.tan(g[1])/6 - c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[4,3]*l[0]/24 - c[4,3]*l[1]*np.tan(g[0])/12 - c[4,3]*l[2]*np.tan(g[1])/24 - c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[4,4]*l[1]/12 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,2]*l[0]/24 - c[5,2]*l[1]*np.tan(g[0])/12 + c[5,2]*l[2]*np.tan(g[1])/24 - c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,3]*l[0]*l[2]/(36*l[1]) - c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[5,4]*l[2]/24 + c[5,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0])],
          [-c[1,0]*l[2]/12 + c[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[1,4]*l[0]/24 + c[1,4]*l[1]*np.tan(g[0])/24 + c[1,4]*l[2]*np.tan(g[1])/12 - c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[1,5]*l[0]*l[2]/(18*l[1]) + c[1,5]*l[2]*np.tan(g[0])/6 - c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[3,0]*l[1]/24 + c[3,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[3,4]*l[0]*l[1]/(36*l[2]) - c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[3,5]*l[0]/24 - c[3,5]*l[1]*np.tan(g[0])/24 + c[3,5]*l[2]*np.tan(g[1])/12 - c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,0]*l[1]*l[2]/(18*l[0]) - c[5,4]*l[1]/24 + c[5,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,5]*l[2]/12 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[1,1]*l[0]*l[2]/(18*l[1]) + c[1,1]*l[2]*np.tan(g[0])/6 - c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[1,3]*l[0]/24 + c[1,3]*l[1]*np.tan(g[0])/24 + c[1,3]*l[2]*np.tan(g[1])/12 - c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[1,5]*l[2]/12 + c[1,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[3,1]*l[0]/24 - c[3,1]*l[1]*np.tan(g[0])/24 + c[3,1]*l[2]*np.tan(g[1])/12 - c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,3]*l[0]*l[1]/(36*l[2]) - c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[3,5]*l[1]/24 + c[3,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,1]*l[2]/12 + c[5,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[5,3]*l[1]/24 + c[5,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,5]*l[1]*l[2]/(18*l[0]),
           -c[1,2]*l[0]/24 + c[1,2]*l[1]*np.tan(g[0])/24 + c[1,2]*l[2]*np.tan(g[1])/12 - c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[1,3]*l[0]*l[2]/(18*l[1]) + c[1,3]*l[2]*np.tan(g[0])/6 - c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[1,4]*l[2]/12 + c[1,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[3,2]*l[0]*l[1]/(36*l[2]) - c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[3,3]*l[0]/24 - c[3,3]*l[1]*np.tan(g[0])/24 + c[3,3]*l[2]*np.tan(g[1])/12 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,4]*l[1]/24 + c[3,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,2]*l[1]/24 + c[5,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,3]*l[2]/12 + c[5,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[5,4]*l[1]*l[2]/(18*l[0]),
           (2*l[0]**2*(-3*c[1,4]*l[1]*l[2] - 4*c[1,5]*l[2]**2 + 2*c[3,4]*l[1]**2 + 3*c[3,5]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(2*c[1,0]*l[2] + c[1,4]*l[1]*np.tan(g[0]) - 2*c[1,4]*l[2]*np.tan(g[1]) - c[3,0]*l[1] + 2*c[3,4]*l[1]*np.tan(g[1]) + c[3,5]*l[1]*np.tan(g[0]) + 2*c[3,5]*l[2]*np.tan(g[1]) - c[5,4]*l[1] - 2*c[5,5]*l[2]) + 4*l[1]**2*l[2]**2*(-c[1,0]*np.tan(g[0]) + c[1,4]*np.tan(g[0])*np.tan(g[1]) + c[1,5]*np.tan(g[0])**2 - c[3,0]*np.tan(g[1]) + c[3,4]*np.tan(g[1])**2 + c[3,5]*np.tan(g[0])*np.tan(g[1]) + c[5,0] - c[5,4]*np.tan(g[1]) - c[5,5]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(-4*c[1,1]*l[2]**2 - 3*c[1,3]*l[1]*l[2] + 3*c[3,1]*l[1]*l[2] + 2*c[3,3]*l[1]**2) + 3*l[0]*l[1]*l[2]*(c[1,3]*l[1]*np.tan(g[0]) - 2*c[1,3]*l[2]*np.tan(g[1]) + 2*c[1,5]*l[2] + c[3,1]*l[1]*np.tan(g[0]) + 2*c[3,1]*l[2]*np.tan(g[1]) + 2*c[3,3]*l[1]*np.tan(g[1]) - c[3,5]*l[1] - 2*c[5,1]*l[2] - c[5,3]*l[1]) + 4*l[1]**2*l[2]**2*(c[1,1]*np.tan(g[0])**2 + c[1,3]*np.tan(g[0])*np.tan(g[1]) - c[1,5]*np.tan(g[0]) + c[3,1]*np.tan(g[0])*np.tan(g[1]) + c[3,3]*np.tan(g[1])**2 - c[3,5]*np.tan(g[1]) - c[5,1]*np.tan(g[0]) - c[5,3]*np.tan(g[1]) + c[5,5]))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(-3*c[1,2]*l[1]*l[2] - 4*c[1,3]*l[2]**2 + 2*c[3,2]*l[1]**2 + 3*c[3,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(c[1,2]*l[1]*np.tan(g[0]) - 2*c[1,2]*l[2]*np.tan(g[1]) + 2*c[1,4]*l[2] + 2*c[3,2]*l[1]*np.tan(g[1]) + c[3,3]*l[1]*np.tan(g[0]) + 2*c[3,3]*l[2]*np.tan(g[1]) - c[3,4]*l[1] - c[5,2]*l[1] - 2*c[5,3]*l[2]) + 4*l[1]**2*l[2]**2*(c[1,2]*np.tan(g[0])*np.tan(g[1]) + c[1,3]*np.tan(g[0])**2 - c[1,4]*np.tan(g[0]) + c[3,2]*np.tan(g[1])**2 + c[3,3]*np.tan(g[0])*np.tan(g[1]) - c[3,4]*np.tan(g[1]) - c[5,2]*np.tan(g[1]) - c[5,3]*np.tan(g[0]) + c[5,4]))/(72*l[0]*l[1]*l[2]),
           (l[0]**2*(-3*c[1,4]*l[1]*l[2] + 4*c[1,5]*l[2]**2 + 4*c[3,4]*l[1]**2 - 3*c[3,5]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(c[1,0]*l[2] + c[1,4]*l[1]*np.tan(g[0]) - c[1,4]*l[2]*np.tan(g[1]) - 2*c[1,5]*l[2]*np.tan(g[0]) - c[3,0]*l[1] + 2*c[3,4]*l[1]*np.tan(g[1]) + c[3,5]*l[1]*np.tan(g[0]) - c[3,5]*l[2]*np.tan(g[1]) - c[5,4]*l[1] + c[5,5]*l[2]) + 4*l[1]**2*l[2]**2*(-c[1,0]*np.tan(g[0]) + c[1,4]*np.tan(g[0])*np.tan(g[1]) + c[1,5]*np.tan(g[0])**2 - c[3,0]*np.tan(g[1]) + c[3,4]*np.tan(g[1])**2 + c[3,5]*np.tan(g[0])*np.tan(g[1]) + c[5,0] - c[5,4]*np.tan(g[1]) - c[5,5]*np.tan(g[0])))/(36*l[0]*l[1]*l[2]),
           (l[0]**2*(4*c[1,1]*l[2]**2 - 3*c[1,3]*l[1]*l[2] - 3*c[3,1]*l[1]*l[2] + 4*c[3,3]*l[1]**2) + 3*l[0]*l[1]*l[2]*(-2*c[1,1]*l[2]*np.tan(g[0]) + c[1,3]*l[1]*np.tan(g[0]) - c[1,3]*l[2]*np.tan(g[1]) + c[1,5]*l[2] + c[3,1]*l[1]*np.tan(g[0]) - c[3,1]*l[2]*np.tan(g[1]) + 2*c[3,3]*l[1]*np.tan(g[1]) - c[3,5]*l[1] + c[5,1]*l[2] - c[5,3]*l[1]) + 4*l[1]**2*l[2]**2*(c[1,1]*np.tan(g[0])**2 + c[1,3]*np.tan(g[0])*np.tan(g[1]) - c[1,5]*np.tan(g[0]) + c[3,1]*np.tan(g[0])*np.tan(g[1]) + c[3,3]*np.tan(g[1])**2 - c[3,5]*np.tan(g[1]) - c[5,1]*np.tan(g[0]) - c[5,3]*np.tan(g[1]) + c[5,5]))/(36*l[0]*l[1]*l[2]),
           (l[0]**2*(-3*c[1,2]*l[1]*l[2] + 4*c[1,3]*l[2]**2 + 4*c[3,2]*l[1]**2 - 3*c[3,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(c[1,2]*l[1]*np.tan(g[0]) - c[1,2]*l[2]*np.tan(g[1]) - 2*c[1,3]*l[2]*np.tan(g[0]) + c[1,4]*l[2] + 2*c[3,2]*l[1]*np.tan(g[1]) + c[3,3]*l[1]*np.tan(g[0]) - c[3,3]*l[2]*np.tan(g[1]) - c[3,4]*l[1] - c[5,2]*l[1] + c[5,3]*l[2]) + 4*l[1]**2*l[2]**2*(c[1,2]*np.tan(g[0])*np.tan(g[1]) + c[1,3]*np.tan(g[0])**2 - c[1,4]*np.tan(g[0]) + c[3,2]*np.tan(g[1])**2 + c[3,3]*np.tan(g[0])*np.tan(g[1]) - c[3,4]*np.tan(g[1]) - c[5,2]*np.tan(g[1]) - c[5,3]*np.tan(g[0]) + c[5,4]))/(36*l[0]*l[1]*l[2]),
           -c[1,0]*l[2]/12 + c[1,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[1,4]*l[0]/24 + c[1,4]*l[1]*np.tan(g[0])/12 + c[1,4]*l[2]*np.tan(g[1])/12 - c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[1,5]*l[0]*l[2]/(18*l[1]) - c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[3,0]*l[1]/12 + c[3,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[3,4]*l[0]*l[1]/(18*l[2]) - c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[3,5]*l[0]/24 - c[3,5]*l[1]*np.tan(g[0])/12 - c[3,5]*l[2]*np.tan(g[1])/12 - c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[5,0]*l[1]*l[2]/(9*l[0]) - c[5,4]*l[1]/12 + c[5,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,5]*l[2]/12 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           c[1,1]*l[0]*l[2]/(18*l[1]) - c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[1,3]*l[0]/24 + c[1,3]*l[1]*np.tan(g[0])/12 + c[1,3]*l[2]*np.tan(g[1])/12 - c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[1,5]*l[2]/12 + c[1,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[3,1]*l[0]/24 - c[3,1]*l[1]*np.tan(g[0])/12 - c[3,1]*l[2]*np.tan(g[1])/12 - c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,3]*l[0]*l[1]/(18*l[2]) - c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[3,5]*l[1]/12 + c[3,5]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,1]*l[2]/12 + c[5,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[5,3]*l[1]/12 + c[5,3]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[5,5]*l[1]*l[2]/(9*l[0]),
           -c[1,2]*l[0]/24 + c[1,2]*l[1]*np.tan(g[0])/12 + c[1,2]*l[2]*np.tan(g[1])/12 - c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[1,3]*l[0]*l[2]/(18*l[1]) - c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[1,4]*l[2]/12 + c[1,4]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[3,2]*l[0]*l[1]/(18*l[2]) - c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[3,3]*l[0]/24 - c[3,3]*l[1]*np.tan(g[0])/12 - c[3,3]*l[2]*np.tan(g[1])/12 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,4]*l[1]/12 + c[3,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[5,2]*l[1]/12 + c[5,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,3]*l[2]/12 + c[5,3]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[5,4]*l[1]*l[2]/(9*l[0]),
           -c[1,0]*l[2]/24 + c[1,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[1,4]*l[0]/24 - c[1,4]*l[1]*np.tan(g[0])/24 + c[1,4]*l[2]*np.tan(g[1])/24 - c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[1,5]*l[0]*l[2]/(36*l[1]) + c[1,5]*l[2]*np.tan(g[0])/12 - c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[3,0]*l[1]/24 + c[3,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[3,4]*l[0]*l[1]/(36*l[2]) - c[3,4]*l[1]*np.tan(g[1])/12 - c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[3,5]*l[0]/24 - c[3,5]*l[1]*np.tan(g[0])/24 + c[3,5]*l[2]*np.tan(g[1])/24 - c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,0]*l[1]*l[2]/(36*l[0]) + c[5,4]*l[1]/24 + c[5,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,5]*l[2]/24 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           -c[1,1]*l[0]*l[2]/(36*l[1]) + c[1,1]*l[2]*np.tan(g[0])/12 - c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[1,3]*l[0]/24 - c[1,3]*l[1]*np.tan(g[0])/24 + c[1,3]*l[2]*np.tan(g[1])/24 - c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[1,5]*l[2]/24 + c[1,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[3,1]*l[0]/24 - c[3,1]*l[1]*np.tan(g[0])/24 + c[3,1]*l[2]*np.tan(g[1])/24 - c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,3]*l[0]*l[1]/(36*l[2]) - c[3,3]*l[1]*np.tan(g[1])/12 - c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[3,5]*l[1]/24 + c[3,5]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,1]*l[2]/24 + c[5,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[5,3]*l[1]/24 + c[5,3]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,5]*l[1]*l[2]/(36*l[0]),
           c[1,2]*l[0]/24 - c[1,2]*l[1]*np.tan(g[0])/24 + c[1,2]*l[2]*np.tan(g[1])/24 - c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[1,3]*l[0]*l[2]/(36*l[1]) + c[1,3]*l[2]*np.tan(g[0])/12 - c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[1,4]*l[2]/24 + c[1,4]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[3,2]*l[0]*l[1]/(36*l[2]) - c[3,2]*l[1]*np.tan(g[1])/12 - c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[3,3]*l[0]/24 - c[3,3]*l[1]*np.tan(g[0])/24 + c[3,3]*l[2]*np.tan(g[1])/24 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[3,4]*l[1]/24 + c[3,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[5,2]*l[1]/24 + c[5,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,3]*l[2]/24 + c[5,3]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[5,4]*l[1]*l[2]/(36*l[0]),
           (2*l[0]**2*(3*c[1,4]*l[1]*l[2] - 2*c[1,5]*l[2]**2 - 2*c[3,4]*l[1]**2 + 3*c[3,5]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(c[1,0]*l[2] - c[1,4]*l[1]*np.tan(g[0]) - c[1,4]*l[2]*np.tan(g[1]) - c[3,0]*l[1] + c[3,5]*l[1]*np.tan(g[0]) + c[3,5]*l[2]*np.tan(g[1]) + c[5,4]*l[1] - c[5,5]*l[2]) + 2*l[1]**2*l[2]**2*(-c[1,0]*np.tan(g[0]) + c[1,4]*np.tan(g[0])*np.tan(g[1]) + c[1,5]*np.tan(g[0])**2 - c[3,0]*np.tan(g[1]) + c[3,4]*np.tan(g[1])**2 + c[3,5]*np.tan(g[0])*np.tan(g[1]) + c[5,0] - c[5,4]*np.tan(g[1]) - c[5,5]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(-2*c[1,1]*l[2]**2 + 3*c[1,3]*l[1]*l[2] + 3*c[3,1]*l[1]*l[2] - 2*c[3,3]*l[1]**2) + 3*l[0]*l[1]*l[2]*(-c[1,3]*l[1]*np.tan(g[0]) - c[1,3]*l[2]*np.tan(g[1]) + c[1,5]*l[2] + c[3,1]*l[1]*np.tan(g[0]) + c[3,1]*l[2]*np.tan(g[1]) - c[3,5]*l[1] - c[5,1]*l[2] + c[5,3]*l[1]) + 2*l[1]**2*l[2]**2*(c[1,1]*np.tan(g[0])**2 + c[1,3]*np.tan(g[0])*np.tan(g[1]) - c[1,5]*np.tan(g[0]) + c[3,1]*np.tan(g[0])*np.tan(g[1]) + c[3,3]*np.tan(g[1])**2 - c[3,5]*np.tan(g[1]) - c[5,1]*np.tan(g[0]) - c[5,3]*np.tan(g[1]) + c[5,5]))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(3*c[1,2]*l[1]*l[2] - 2*c[1,3]*l[2]**2 - 2*c[3,2]*l[1]**2 + 3*c[3,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(-c[1,2]*l[1]*np.tan(g[0]) - c[1,2]*l[2]*np.tan(g[1]) + c[1,4]*l[2] + c[3,3]*l[1]*np.tan(g[0]) + c[3,3]*l[2]*np.tan(g[1]) - c[3,4]*l[1] + c[5,2]*l[1] - c[5,3]*l[2]) + 2*l[1]**2*l[2]**2*(c[1,2]*np.tan(g[0])*np.tan(g[1]) + c[1,3]*np.tan(g[0])**2 - c[1,4]*np.tan(g[0]) + c[3,2]*np.tan(g[1])**2 + c[3,3]*np.tan(g[0])*np.tan(g[1]) - c[3,4]*np.tan(g[1]) - c[5,2]*np.tan(g[1]) - c[5,3]*np.tan(g[0]) + c[5,4]))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(3*c[1,4]*l[1]*l[2] + 2*c[1,5]*l[2]**2 - 4*c[3,4]*l[1]**2 - 3*c[3,5]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(c[1,0]*l[2] - 2*c[1,4]*l[1]*np.tan(g[0]) - c[1,4]*l[2]*np.tan(g[1]) - 2*c[1,5]*l[2]*np.tan(g[0]) - 2*c[3,0]*l[1] + 2*c[3,5]*l[1]*np.tan(g[0]) - c[3,5]*l[2]*np.tan(g[1]) + 2*c[5,4]*l[1] + c[5,5]*l[2]) + 4*l[1]**2*l[2]**2*(-c[1,0]*np.tan(g[0]) + c[1,4]*np.tan(g[0])*np.tan(g[1]) + c[1,5]*np.tan(g[0])**2 - c[3,0]*np.tan(g[1]) + c[3,4]*np.tan(g[1])**2 + c[3,5]*np.tan(g[0])*np.tan(g[1]) + c[5,0] - c[5,4]*np.tan(g[1]) - c[5,5]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(2*c[1,1]*l[2]**2 + 3*c[1,3]*l[1]*l[2] - 3*c[3,1]*l[1]*l[2] - 4*c[3,3]*l[1]**2) + 3*l[0]*l[1]*l[2]*(-2*c[1,1]*l[2]*np.tan(g[0]) - 2*c[1,3]*l[1]*np.tan(g[0]) - c[1,3]*l[2]*np.tan(g[1]) + c[1,5]*l[2] + 2*c[3,1]*l[1]*np.tan(g[0]) - c[3,1]*l[2]*np.tan(g[1]) - 2*c[3,5]*l[1] + c[5,1]*l[2] + 2*c[5,3]*l[1]) + 4*l[1]**2*l[2]**2*(c[1,1]*np.tan(g[0])**2 + c[1,3]*np.tan(g[0])*np.tan(g[1]) - c[1,5]*np.tan(g[0]) + c[3,1]*np.tan(g[0])*np.tan(g[1]) + c[3,3]*np.tan(g[1])**2 - c[3,5]*np.tan(g[1]) - c[5,1]*np.tan(g[0]) - c[5,3]*np.tan(g[1]) + c[5,5]))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(3*c[1,2]*l[1]*l[2] + 2*c[1,3]*l[2]**2 - 4*c[3,2]*l[1]**2 - 3*c[3,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(-2*c[1,2]*l[1]*np.tan(g[0]) - c[1,2]*l[2]*np.tan(g[1]) - 2*c[1,3]*l[2]*np.tan(g[0]) + c[1,4]*l[2] + 2*c[3,3]*l[1]*np.tan(g[0]) - c[3,3]*l[2]*np.tan(g[1]) - 2*c[3,4]*l[1] + 2*c[5,2]*l[1] + c[5,3]*l[2]) + 4*l[1]**2*l[2]**2*(c[1,2]*np.tan(g[0])*np.tan(g[1]) + c[1,3]*np.tan(g[0])**2 - c[1,4]*np.tan(g[0]) + c[3,2]*np.tan(g[1])**2 + c[3,3]*np.tan(g[0])*np.tan(g[1]) - c[3,4]*np.tan(g[1]) - c[5,2]*np.tan(g[1]) - c[5,3]*np.tan(g[0]) + c[5,4]))/(72*l[0]*l[1]*l[2]),
           -c[1,0]*l[2]/24 + c[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[1,4]*l[0]/24 - c[1,4]*l[1]*np.tan(g[0])/12 + c[1,4]*l[2]*np.tan(g[1])/24 - c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[1,5]*l[0]*l[2]/(36*l[1]) - c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[3,0]*l[1]/12 + c[3,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[3,4]*l[0]*l[1]/(18*l[2]) - c[3,4]*l[1]*np.tan(g[1])/6 - c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[3,5]*l[0]/24 - c[3,5]*l[1]*np.tan(g[0])/12 - c[3,5]*l[2]*np.tan(g[1])/24 - c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,0]*l[1]*l[2]/(18*l[0]) + c[5,4]*l[1]/12 + c[5,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,5]*l[2]/24 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[1,1]*l[0]*l[2]/(36*l[1]) - c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[1,3]*l[0]/24 - c[1,3]*l[1]*np.tan(g[0])/12 + c[1,3]*l[2]*np.tan(g[1])/24 - c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[1,5]*l[2]/24 + c[1,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[3,1]*l[0]/24 - c[3,1]*l[1]*np.tan(g[0])/12 - c[3,1]*l[2]*np.tan(g[1])/24 - c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,3]*l[0]*l[1]/(18*l[2]) - c[3,3]*l[1]*np.tan(g[1])/6 - c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[3,5]*l[1]/12 + c[3,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,1]*l[2]/24 + c[5,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[5,3]*l[1]/12 + c[5,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,5]*l[1]*l[2]/(18*l[0]),
           c[1,2]*l[0]/24 - c[1,2]*l[1]*np.tan(g[0])/12 + c[1,2]*l[2]*np.tan(g[1])/24 - c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[1,3]*l[0]*l[2]/(36*l[1]) - c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[1,4]*l[2]/24 + c[1,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[3,2]*l[0]*l[1]/(18*l[2]) - c[3,2]*l[1]*np.tan(g[1])/6 - c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[3,3]*l[0]/24 - c[3,3]*l[1]*np.tan(g[0])/12 - c[3,3]*l[2]*np.tan(g[1])/24 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,4]*l[1]/12 + c[3,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,2]*l[1]/12 + c[5,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,3]*l[2]/24 + c[5,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[5,4]*l[1]*l[2]/(18*l[0])],
          [c[2,0]*l[1]/24 + c[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[2,4]*l[0]*l[1]/(36*l[2]) - c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[2,5]*l[0]/24 - c[2,5]*l[1]*np.tan(g[0])/24 + c[2,5]*l[2]*np.tan(g[1])/12 - c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,0]*l[2]/12 + c[3,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[3,4]*l[0]/24 + c[3,4]*l[1]*np.tan(g[0])/24 + c[3,4]*l[2]*np.tan(g[1])/12 - c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,5]*l[0]*l[2]/(18*l[1]) + c[3,5]*l[2]*np.tan(g[0])/6 - c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[4,0]*l[1]*l[2]/(18*l[0]) - c[4,4]*l[1]/24 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,5]*l[2]/12 + c[4,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[2,1]*l[0]/24 - c[2,1]*l[1]*np.tan(g[0])/24 + c[2,1]*l[2]*np.tan(g[1])/12 - c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[2,3]*l[0]*l[1]/(36*l[2]) - c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[2,5]*l[1]/24 + c[2,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[3,1]*l[0]*l[2]/(18*l[1]) + c[3,1]*l[2]*np.tan(g[0])/6 - c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[3,3]*l[0]/24 + c[3,3]*l[1]*np.tan(g[0])/24 + c[3,3]*l[2]*np.tan(g[1])/12 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,5]*l[2]/12 + c[3,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,1]*l[2]/12 + c[4,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,3]*l[1]/24 + c[4,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,5]*l[1]*l[2]/(18*l[0]),
           c[2,2]*l[0]*l[1]/(36*l[2]) - c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[2,3]*l[0]/24 - c[2,3]*l[1]*np.tan(g[0])/24 + c[2,3]*l[2]*np.tan(g[1])/12 - c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[2,4]*l[1]/24 + c[2,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[3,2]*l[0]/24 + c[3,2]*l[1]*np.tan(g[0])/24 + c[3,2]*l[2]*np.tan(g[1])/12 - c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,3]*l[0]*l[2]/(18*l[1]) + c[3,3]*l[2]*np.tan(g[0])/6 - c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[3,4]*l[2]/12 + c[3,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,2]*l[1]/24 + c[4,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,3]*l[2]/12 + c[4,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,4]*l[1]*l[2]/(18*l[0]),
           (2*l[0]**2*(2*c[2,4]*l[1]**2 + 3*c[2,5]*l[1]*l[2] - 3*c[3,4]*l[1]*l[2] - 4*c[3,5]*l[2]**2) + 3*l[0]*l[1]*l[2]*(-c[2,0]*l[1] + 2*c[2,4]*l[1]*np.tan(g[1]) + c[2,5]*l[1]*np.tan(g[0]) + 2*c[2,5]*l[2]*np.tan(g[1]) + 2*c[3,0]*l[2] + c[3,4]*l[1]*np.tan(g[0]) - 2*c[3,4]*l[2]*np.tan(g[1]) - c[4,4]*l[1] - 2*c[4,5]*l[2]) + 4*l[1]**2*l[2]**2*(-c[2,0]*np.tan(g[1]) + c[2,4]*np.tan(g[1])**2 + c[2,5]*np.tan(g[0])*np.tan(g[1]) - c[3,0]*np.tan(g[0]) + c[3,4]*np.tan(g[0])*np.tan(g[1]) + c[3,5]*np.tan(g[0])**2 + c[4,0] - c[4,4]*np.tan(g[1]) - c[4,5]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(3*c[2,1]*l[1]*l[2] + 2*c[2,3]*l[1]**2 - 4*c[3,1]*l[2]**2 - 3*c[3,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(c[2,1]*l[1]*np.tan(g[0]) + 2*c[2,1]*l[2]*np.tan(g[1]) + 2*c[2,3]*l[1]*np.tan(g[1]) - c[2,5]*l[1] + c[3,3]*l[1]*np.tan(g[0]) - 2*c[3,3]*l[2]*np.tan(g[1]) + 2*c[3,5]*l[2] - 2*c[4,1]*l[2] - c[4,3]*l[1]) + 4*l[1]**2*l[2]**2*(c[2,1]*np.tan(g[0])*np.tan(g[1]) + c[2,3]*np.tan(g[1])**2 - c[2,5]*np.tan(g[1]) + c[3,1]*np.tan(g[0])**2 + c[3,3]*np.tan(g[0])*np.tan(g[1]) - c[3,5]*np.tan(g[0]) - c[4,1]*np.tan(g[0]) - c[4,3]*np.tan(g[1]) + c[4,5]))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(2*c[2,2]*l[1]**2 + 3*c[2,3]*l[1]*l[2] - 3*c[3,2]*l[1]*l[2] - 4*c[3,3]*l[2]**2) + 3*l[0]*l[1]*l[2]*(2*c[2,2]*l[1]*np.tan(g[1]) + c[2,3]*l[1]*np.tan(g[0]) + 2*c[2,3]*l[2]*np.tan(g[1]) - c[2,4]*l[1] + c[3,2]*l[1]*np.tan(g[0]) - 2*c[3,2]*l[2]*np.tan(g[1]) + 2*c[3,4]*l[2] - c[4,2]*l[1] - 2*c[4,3]*l[2]) + 4*l[1]**2*l[2]**2*(c[2,2]*np.tan(g[1])**2 + c[2,3]*np.tan(g[0])*np.tan(g[1]) - c[2,4]*np.tan(g[1]) + c[3,2]*np.tan(g[0])*np.tan(g[1]) + c[3,3]*np.tan(g[0])**2 - c[3,4]*np.tan(g[0]) - c[4,2]*np.tan(g[1]) - c[4,3]*np.tan(g[0]) + c[4,4]))/(72*l[0]*l[1]*l[2]),
           (l[0]**2*(4*c[2,4]*l[1]**2 - 3*c[2,5]*l[1]*l[2] - 3*c[3,4]*l[1]*l[2] + 4*c[3,5]*l[2]**2) + 3*l[0]*l[1]*l[2]*(-c[2,0]*l[1] + 2*c[2,4]*l[1]*np.tan(g[1]) + c[2,5]*l[1]*np.tan(g[0]) - c[2,5]*l[2]*np.tan(g[1]) + c[3,0]*l[2] + c[3,4]*l[1]*np.tan(g[0]) - c[3,4]*l[2]*np.tan(g[1]) - 2*c[3,5]*l[2]*np.tan(g[0]) - c[4,4]*l[1] + c[4,5]*l[2]) + 4*l[1]**2*l[2]**2*(-c[2,0]*np.tan(g[1]) + c[2,4]*np.tan(g[1])**2 + c[2,5]*np.tan(g[0])*np.tan(g[1]) - c[3,0]*np.tan(g[0]) + c[3,4]*np.tan(g[0])*np.tan(g[1]) + c[3,5]*np.tan(g[0])**2 + c[4,0] - c[4,4]*np.tan(g[1]) - c[4,5]*np.tan(g[0])))/(36*l[0]*l[1]*l[2]),
           (l[0]**2*(-3*c[2,1]*l[1]*l[2] + 4*c[2,3]*l[1]**2 + 4*c[3,1]*l[2]**2 - 3*c[3,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(c[2,1]*l[1]*np.tan(g[0]) - c[2,1]*l[2]*np.tan(g[1]) + 2*c[2,3]*l[1]*np.tan(g[1]) - c[2,5]*l[1] - 2*c[3,1]*l[2]*np.tan(g[0]) + c[3,3]*l[1]*np.tan(g[0]) - c[3,3]*l[2]*np.tan(g[1]) + c[3,5]*l[2] + c[4,1]*l[2] - c[4,3]*l[1]) + 4*l[1]**2*l[2]**2*(c[2,1]*np.tan(g[0])*np.tan(g[1]) + c[2,3]*np.tan(g[1])**2 - c[2,5]*np.tan(g[1]) + c[3,1]*np.tan(g[0])**2 + c[3,3]*np.tan(g[0])*np.tan(g[1]) - c[3,5]*np.tan(g[0]) - c[4,1]*np.tan(g[0]) - c[4,3]*np.tan(g[1]) + c[4,5]))/(36*l[0]*l[1]*l[2]),
           (l[0]**2*(4*c[2,2]*l[1]**2 - 3*c[2,3]*l[1]*l[2] - 3*c[3,2]*l[1]*l[2] + 4*c[3,3]*l[2]**2) + 3*l[0]*l[1]*l[2]*(2*c[2,2]*l[1]*np.tan(g[1]) + c[2,3]*l[1]*np.tan(g[0]) - c[2,3]*l[2]*np.tan(g[1]) - c[2,4]*l[1] + c[3,2]*l[1]*np.tan(g[0]) - c[3,2]*l[2]*np.tan(g[1]) - 2*c[3,3]*l[2]*np.tan(g[0]) + c[3,4]*l[2] - c[4,2]*l[1] + c[4,3]*l[2]) + 4*l[1]**2*l[2]**2*(c[2,2]*np.tan(g[1])**2 + c[2,3]*np.tan(g[0])*np.tan(g[1]) - c[2,4]*np.tan(g[1]) + c[3,2]*np.tan(g[0])*np.tan(g[1]) + c[3,3]*np.tan(g[0])**2 - c[3,4]*np.tan(g[0]) - c[4,2]*np.tan(g[1]) - c[4,3]*np.tan(g[0]) + c[4,4]))/(36*l[0]*l[1]*l[2]),
           c[2,0]*l[1]/12 + c[2,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[2,4]*l[0]*l[1]/(18*l[2]) - c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[2,5]*l[0]/24 - c[2,5]*l[1]*np.tan(g[0])/12 - c[2,5]*l[2]*np.tan(g[1])/12 - c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[3,0]*l[2]/12 + c[3,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[3,4]*l[0]/24 + c[3,4]*l[1]*np.tan(g[0])/12 + c[3,4]*l[2]*np.tan(g[1])/12 - c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,5]*l[0]*l[2]/(18*l[1]) - c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[4,0]*l[1]*l[2]/(9*l[0]) - c[4,4]*l[1]/12 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[4,5]*l[2]/12 + c[4,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           -c[2,1]*l[0]/24 - c[2,1]*l[1]*np.tan(g[0])/12 - c[2,1]*l[2]*np.tan(g[1])/12 - c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[2,3]*l[0]*l[1]/(18*l[2]) - c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[2,5]*l[1]/12 + c[2,5]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[3,1]*l[0]*l[2]/(18*l[1]) - c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[3,3]*l[0]/24 + c[3,3]*l[1]*np.tan(g[0])/12 + c[3,3]*l[2]*np.tan(g[1])/12 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[3,5]*l[2]/12 + c[3,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[4,1]*l[2]/12 + c[4,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[4,3]*l[1]/12 + c[4,3]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[4,5]*l[1]*l[2]/(9*l[0]),
           c[2,2]*l[0]*l[1]/(18*l[2]) - c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[2,3]*l[0]/24 - c[2,3]*l[1]*np.tan(g[0])/12 - c[2,3]*l[2]*np.tan(g[1])/12 - c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[2,4]*l[1]/12 + c[2,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[3,2]*l[0]/24 + c[3,2]*l[1]*np.tan(g[0])/12 + c[3,2]*l[2]*np.tan(g[1])/12 - c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,3]*l[0]*l[2]/(18*l[1]) - c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[3,4]*l[2]/12 + c[3,4]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[4,2]*l[1]/12 + c[4,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[4,3]*l[2]/12 + c[4,3]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[4,4]*l[1]*l[2]/(9*l[0]),
           c[2,0]*l[1]/24 + c[2,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[2,4]*l[0]*l[1]/(36*l[2]) - c[2,4]*l[1]*np.tan(g[1])/12 - c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[2,5]*l[0]/24 - c[2,5]*l[1]*np.tan(g[0])/24 + c[2,5]*l[2]*np.tan(g[1])/24 - c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,0]*l[2]/24 + c[3,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[3,4]*l[0]/24 - c[3,4]*l[1]*np.tan(g[0])/24 + c[3,4]*l[2]*np.tan(g[1])/24 - c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,5]*l[0]*l[2]/(36*l[1]) + c[3,5]*l[2]*np.tan(g[0])/12 - c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[4,0]*l[1]*l[2]/(36*l[0]) + c[4,4]*l[1]/24 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[4,5]*l[2]/24 + c[4,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           c[2,1]*l[0]/24 - c[2,1]*l[1]*np.tan(g[0])/24 + c[2,1]*l[2]*np.tan(g[1])/24 - c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[2,3]*l[0]*l[1]/(36*l[2]) - c[2,3]*l[1]*np.tan(g[1])/12 - c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[2,5]*l[1]/24 + c[2,5]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[3,1]*l[0]*l[2]/(36*l[1]) + c[3,1]*l[2]*np.tan(g[0])/12 - c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[3,3]*l[0]/24 - c[3,3]*l[1]*np.tan(g[0])/24 + c[3,3]*l[2]*np.tan(g[1])/24 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,5]*l[2]/24 + c[3,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[4,1]*l[2]/24 + c[4,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[4,3]*l[1]/24 + c[4,3]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[4,5]*l[1]*l[2]/(36*l[0]),
           -c[2,2]*l[0]*l[1]/(36*l[2]) - c[2,2]*l[1]*np.tan(g[1])/12 - c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[2,3]*l[0]/24 - c[2,3]*l[1]*np.tan(g[0])/24 + c[2,3]*l[2]*np.tan(g[1])/24 - c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[2,4]*l[1]/24 + c[2,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[3,2]*l[0]/24 - c[3,2]*l[1]*np.tan(g[0])/24 + c[3,2]*l[2]*np.tan(g[1])/24 - c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,3]*l[0]*l[2]/(36*l[1]) + c[3,3]*l[2]*np.tan(g[0])/12 - c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[3,4]*l[2]/24 + c[3,4]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[4,2]*l[1]/24 + c[4,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[4,3]*l[2]/24 + c[4,3]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[4,4]*l[1]*l[2]/(36*l[0]),
           (2*l[0]**2*(-2*c[2,4]*l[1]**2 + 3*c[2,5]*l[1]*l[2] + 3*c[3,4]*l[1]*l[2] - 2*c[3,5]*l[2]**2) + 3*l[0]*l[1]*l[2]*(-c[2,0]*l[1] + c[2,5]*l[1]*np.tan(g[0]) + c[2,5]*l[2]*np.tan(g[1]) + c[3,0]*l[2] - c[3,4]*l[1]*np.tan(g[0]) - c[3,4]*l[2]*np.tan(g[1]) + c[4,4]*l[1] - c[4,5]*l[2]) + 2*l[1]**2*l[2]**2*(-c[2,0]*np.tan(g[1]) + c[2,4]*np.tan(g[1])**2 + c[2,5]*np.tan(g[0])*np.tan(g[1]) - c[3,0]*np.tan(g[0]) + c[3,4]*np.tan(g[0])*np.tan(g[1]) + c[3,5]*np.tan(g[0])**2 + c[4,0] - c[4,4]*np.tan(g[1]) - c[4,5]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(3*c[2,1]*l[1]*l[2] - 2*c[2,3]*l[1]**2 - 2*c[3,1]*l[2]**2 + 3*c[3,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(c[2,1]*l[1]*np.tan(g[0]) + c[2,1]*l[2]*np.tan(g[1]) - c[2,5]*l[1] - c[3,3]*l[1]*np.tan(g[0]) - c[3,3]*l[2]*np.tan(g[1]) + c[3,5]*l[2] - c[4,1]*l[2] + c[4,3]*l[1]) + 2*l[1]**2*l[2]**2*(c[2,1]*np.tan(g[0])*np.tan(g[1]) + c[2,3]*np.tan(g[1])**2 - c[2,5]*np.tan(g[1]) + c[3,1]*np.tan(g[0])**2 + c[3,3]*np.tan(g[0])*np.tan(g[1]) - c[3,5]*np.tan(g[0]) - c[4,1]*np.tan(g[0]) - c[4,3]*np.tan(g[1]) + c[4,5]))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(-2*c[2,2]*l[1]**2 + 3*c[2,3]*l[1]*l[2] + 3*c[3,2]*l[1]*l[2] - 2*c[3,3]*l[2]**2) + 3*l[0]*l[1]*l[2]*(c[2,3]*l[1]*np.tan(g[0]) + c[2,3]*l[2]*np.tan(g[1]) - c[2,4]*l[1] - c[3,2]*l[1]*np.tan(g[0]) - c[3,2]*l[2]*np.tan(g[1]) + c[3,4]*l[2] + c[4,2]*l[1] - c[4,3]*l[2]) + 2*l[1]**2*l[2]**2*(c[2,2]*np.tan(g[1])**2 + c[2,3]*np.tan(g[0])*np.tan(g[1]) - c[2,4]*np.tan(g[1]) + c[3,2]*np.tan(g[0])*np.tan(g[1]) + c[3,3]*np.tan(g[0])**2 - c[3,4]*np.tan(g[0]) - c[4,2]*np.tan(g[1]) - c[4,3]*np.tan(g[0]) + c[4,4]))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(-4*c[2,4]*l[1]**2 - 3*c[2,5]*l[1]*l[2] + 3*c[3,4]*l[1]*l[2] + 2*c[3,5]*l[2]**2) + 3*l[0]*l[1]*l[2]*(-2*c[2,0]*l[1] + 2*c[2,5]*l[1]*np.tan(g[0]) - c[2,5]*l[2]*np.tan(g[1]) + c[3,0]*l[2] - 2*c[3,4]*l[1]*np.tan(g[0]) - c[3,4]*l[2]*np.tan(g[1]) - 2*c[3,5]*l[2]*np.tan(g[0]) + 2*c[4,4]*l[1] + c[4,5]*l[2]) + 4*l[1]**2*l[2]**2*(-c[2,0]*np.tan(g[1]) + c[2,4]*np.tan(g[1])**2 + c[2,5]*np.tan(g[0])*np.tan(g[1]) - c[3,0]*np.tan(g[0]) + c[3,4]*np.tan(g[0])*np.tan(g[1]) + c[3,5]*np.tan(g[0])**2 + c[4,0] - c[4,4]*np.tan(g[1]) - c[4,5]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(-3*c[2,1]*l[1]*l[2] - 4*c[2,3]*l[1]**2 + 2*c[3,1]*l[2]**2 + 3*c[3,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(2*c[2,1]*l[1]*np.tan(g[0]) - c[2,1]*l[2]*np.tan(g[1]) - 2*c[2,5]*l[1] - 2*c[3,1]*l[2]*np.tan(g[0]) - 2*c[3,3]*l[1]*np.tan(g[0]) - c[3,3]*l[2]*np.tan(g[1]) + c[3,5]*l[2] + c[4,1]*l[2] + 2*c[4,3]*l[1]) + 4*l[1]**2*l[2]**2*(c[2,1]*np.tan(g[0])*np.tan(g[1]) + c[2,3]*np.tan(g[1])**2 - c[2,5]*np.tan(g[1]) + c[3,1]*np.tan(g[0])**2 + c[3,3]*np.tan(g[0])*np.tan(g[1]) - c[3,5]*np.tan(g[0]) - c[4,1]*np.tan(g[0]) - c[4,3]*np.tan(g[1]) + c[4,5]))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(-4*c[2,2]*l[1]**2 - 3*c[2,3]*l[1]*l[2] + 3*c[3,2]*l[1]*l[2] + 2*c[3,3]*l[2]**2) + 3*l[0]*l[1]*l[2]*(2*c[2,3]*l[1]*np.tan(g[0]) - c[2,3]*l[2]*np.tan(g[1]) - 2*c[2,4]*l[1] - 2*c[3,2]*l[1]*np.tan(g[0]) - c[3,2]*l[2]*np.tan(g[1]) - 2*c[3,3]*l[2]*np.tan(g[0]) + c[3,4]*l[2] + 2*c[4,2]*l[1] + c[4,3]*l[2]) + 4*l[1]**2*l[2]**2*(c[2,2]*np.tan(g[1])**2 + c[2,3]*np.tan(g[0])*np.tan(g[1]) - c[2,4]*np.tan(g[1]) + c[3,2]*np.tan(g[0])*np.tan(g[1]) + c[3,3]*np.tan(g[0])**2 - c[3,4]*np.tan(g[0]) - c[4,2]*np.tan(g[1]) - c[4,3]*np.tan(g[0]) + c[4,4]))/(72*l[0]*l[1]*l[2]),
           c[2,0]*l[1]/12 + c[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[2,4]*l[0]*l[1]/(18*l[2]) - c[2,4]*l[1]*np.tan(g[1])/6 - c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[2,5]*l[0]/24 - c[2,5]*l[1]*np.tan(g[0])/12 - c[2,5]*l[2]*np.tan(g[1])/24 - c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,0]*l[2]/24 + c[3,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[3,4]*l[0]/24 - c[3,4]*l[1]*np.tan(g[0])/12 + c[3,4]*l[2]*np.tan(g[1])/24 - c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,5]*l[0]*l[2]/(36*l[1]) - c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[4,0]*l[1]*l[2]/(18*l[0]) + c[4,4]*l[1]/12 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,5]*l[2]/24 + c[4,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[2,1]*l[0]/24 - c[2,1]*l[1]*np.tan(g[0])/12 - c[2,1]*l[2]*np.tan(g[1])/24 - c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[2,3]*l[0]*l[1]/(18*l[2]) - c[2,3]*l[1]*np.tan(g[1])/6 - c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[2,5]*l[1]/12 + c[2,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[3,1]*l[0]*l[2]/(36*l[1]) - c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[3,3]*l[0]/24 - c[3,3]*l[1]*np.tan(g[0])/12 + c[3,3]*l[2]*np.tan(g[1])/24 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,5]*l[2]/24 + c[3,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,1]*l[2]/24 + c[4,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,3]*l[1]/12 + c[4,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,5]*l[1]*l[2]/(18*l[0]),
           -c[2,2]*l[0]*l[1]/(18*l[2]) - c[2,2]*l[1]*np.tan(g[1])/6 - c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[2,3]*l[0]/24 - c[2,3]*l[1]*np.tan(g[0])/12 - c[2,3]*l[2]*np.tan(g[1])/24 - c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[2,4]*l[1]/12 + c[2,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[3,2]*l[0]/24 - c[3,2]*l[1]*np.tan(g[0])/12 + c[3,2]*l[2]*np.tan(g[1])/24 - c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,3]*l[0]*l[2]/(36*l[1]) - c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[3,4]*l[2]/24 + c[3,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,2]*l[1]/12 + c[4,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,3]*l[2]/24 + c[4,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,4]*l[1]*l[2]/(18*l[0])],
          [c[0,0]*l[1]*l[2]/(18*l[0]) + c[0,4]*l[1]/24 - c[0,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[0,5]*l[2]/12 - c[0,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,0]*l[1]/24 - c[4,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,4]*l[0]*l[1]/(18*l[2]) - c[4,4]*l[1]*np.tan(g[1])/12 + c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[4,5]*l[0]/12 - c[4,5]*l[1]*np.tan(g[0])/24 - c[4,5]*l[2]*np.tan(g[1])/12 + c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,0]*l[2]/12 - c[5,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[5,4]*l[0]/12 - c[5,4]*l[1]*np.tan(g[0])/24 + c[5,4]*l[2]*np.tan(g[1])/12 + c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,5]*l[0]*l[2]/(9*l[1]) + c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]),
           c[0,1]*l[2]/12 - c[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[0,3]*l[1]/24 - c[0,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[0,5]*l[1]*l[2]/(18*l[0]) + c[4,1]*l[0]/12 - c[4,1]*l[1]*np.tan(g[0])/24 - c[4,1]*l[2]*np.tan(g[1])/12 + c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[4,3]*l[0]*l[1]/(18*l[2]) - c[4,3]*l[1]*np.tan(g[1])/12 + c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[4,5]*l[1]/24 - c[4,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,1]*l[0]*l[2]/(9*l[1]) + c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[5,3]*l[0]/12 - c[5,3]*l[1]*np.tan(g[0])/24 + c[5,3]*l[2]*np.tan(g[1])/12 + c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,5]*l[2]/12 - c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[0,2]*l[1]/24 - c[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[0,3]*l[2]/12 - c[0,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[0,4]*l[1]*l[2]/(18*l[0]) + c[4,2]*l[0]*l[1]/(18*l[2]) - c[4,2]*l[1]*np.tan(g[1])/12 + c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[4,3]*l[0]/12 - c[4,3]*l[1]*np.tan(g[0])/24 - c[4,3]*l[2]*np.tan(g[1])/12 + c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[4,4]*l[1]/24 - c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,2]*l[0]/12 - c[5,2]*l[1]*np.tan(g[0])/24 + c[5,2]*l[2]*np.tan(g[1])/12 + c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,3]*l[0]*l[2]/(9*l[1]) + c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[5,4]*l[2]/12 - c[5,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[0,0]*l[1]*l[2]/(18*l[0]) + c[0,4]*l[1]/24 + c[0,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[0,5]*l[2]/12 + c[0,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,0]*l[1]/24 + c[4,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,4]*l[0]*l[1]/(36*l[2]) - c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[4,5]*l[0]/24 + c[4,5]*l[1]*np.tan(g[0])/24 - c[4,5]*l[2]*np.tan(g[1])/12 - c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,0]*l[2]/12 + c[5,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[5,4]*l[0]/24 - c[5,4]*l[1]*np.tan(g[0])/24 - c[5,4]*l[2]*np.tan(g[1])/12 - c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,5]*l[0]*l[2]/(18*l[1]) - c[5,5]*l[2]*np.tan(g[0])/6 - c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]),
           c[0,1]*l[2]/12 + c[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[0,3]*l[1]/24 + c[0,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[0,5]*l[1]*l[2]/(18*l[0]) + c[4,1]*l[0]/24 + c[4,1]*l[1]*np.tan(g[0])/24 - c[4,1]*l[2]*np.tan(g[1])/12 - c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[4,3]*l[0]*l[1]/(36*l[2]) - c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[4,5]*l[1]/24 + c[4,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,1]*l[0]*l[2]/(18*l[1]) - c[5,1]*l[2]*np.tan(g[0])/6 - c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[5,3]*l[0]/24 - c[5,3]*l[1]*np.tan(g[0])/24 - c[5,3]*l[2]*np.tan(g[1])/12 - c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,5]*l[2]/12 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[0,2]*l[1]/24 + c[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[0,3]*l[2]/12 + c[0,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[0,4]*l[1]*l[2]/(18*l[0]) + c[4,2]*l[0]*l[1]/(36*l[2]) - c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[4,3]*l[0]/24 + c[4,3]*l[1]*np.tan(g[0])/24 - c[4,3]*l[2]*np.tan(g[1])/12 - c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[4,4]*l[1]/24 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,2]*l[0]/24 - c[5,2]*l[1]*np.tan(g[0])/24 - c[5,2]*l[2]*np.tan(g[1])/12 - c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,3]*l[0]*l[2]/(18*l[1]) - c[5,3]*l[2]*np.tan(g[0])/6 - c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[5,4]*l[2]/12 + c[5,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[0,0]*l[1]*l[2]/(9*l[0]) + c[0,4]*l[1]/12 + c[0,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[0,5]*l[2]/12 + c[0,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[4,0]*l[1]/12 + c[4,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[4,4]*l[0]*l[1]/(18*l[2]) - c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[4,5]*l[0]/24 + c[4,5]*l[1]*np.tan(g[0])/12 + c[4,5]*l[2]*np.tan(g[1])/12 - c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[5,0]*l[2]/12 + c[5,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[5,4]*l[0]/24 - c[5,4]*l[1]*np.tan(g[0])/12 - c[5,4]*l[2]*np.tan(g[1])/12 - c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[5,5]*l[0]*l[2]/(18*l[1]) - c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]),
           -c[0,1]*l[2]/12 + c[0,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[0,3]*l[1]/12 + c[0,3]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[0,5]*l[1]*l[2]/(9*l[0]) - c[4,1]*l[0]/24 + c[4,1]*l[1]*np.tan(g[0])/12 + c[4,1]*l[2]*np.tan(g[1])/12 - c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[4,3]*l[0]*l[1]/(18*l[2]) - c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[4,5]*l[1]/12 + c[4,5]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,1]*l[0]*l[2]/(18*l[1]) - c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[5,3]*l[0]/24 - c[5,3]*l[1]*np.tan(g[0])/12 - c[5,3]*l[2]*np.tan(g[1])/12 - c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[5,5]*l[2]/12 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           c[0,2]*l[1]/12 + c[0,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[0,3]*l[2]/12 + c[0,3]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[0,4]*l[1]*l[2]/(9*l[0]) + c[4,2]*l[0]*l[1]/(18*l[2]) - c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[4,3]*l[0]/24 + c[4,3]*l[1]*np.tan(g[0])/12 + c[4,3]*l[2]*np.tan(g[1])/12 - c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[4,4]*l[1]/12 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[5,2]*l[0]/24 - c[5,2]*l[1]*np.tan(g[0])/12 - c[5,2]*l[2]*np.tan(g[1])/12 - c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[5,3]*l[0]*l[2]/(18*l[1]) - c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[5,4]*l[2]/12 + c[5,4]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           c[0,0]*l[1]*l[2]/(9*l[0]) + c[0,4]*l[1]/12 - c[0,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[0,5]*l[2]/12 - c[0,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[4,0]*l[1]/12 - c[4,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[4,4]*l[0]*l[1]/(9*l[2]) - c[4,4]*l[1]*np.tan(g[1])/6 + c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[4,5]*l[0]/12 - c[4,5]*l[1]*np.tan(g[0])/12 + c[4,5]*l[2]*np.tan(g[1])/12 + c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[5,0]*l[2]/12 - c[5,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[5,4]*l[0]/12 - c[5,4]*l[1]*np.tan(g[0])/12 + c[5,4]*l[2]*np.tan(g[1])/12 + c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[5,5]*l[0]*l[2]/(9*l[1]) + c[5,5]*l[2]*np.tan(g[0])/6 + c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]),
           -c[0,1]*l[2]/12 - c[0,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[0,3]*l[1]/12 - c[0,3]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[0,5]*l[1]*l[2]/(9*l[0]) - c[4,1]*l[0]/12 - c[4,1]*l[1]*np.tan(g[0])/12 + c[4,1]*l[2]*np.tan(g[1])/12 + c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[4,3]*l[0]*l[1]/(9*l[2]) - c[4,3]*l[1]*np.tan(g[1])/6 + c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[4,5]*l[1]/12 - c[4,5]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,1]*l[0]*l[2]/(9*l[1]) + c[5,1]*l[2]*np.tan(g[0])/6 + c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[5,3]*l[0]/12 - c[5,3]*l[1]*np.tan(g[0])/12 + c[5,3]*l[2]*np.tan(g[1])/12 + c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[5,5]*l[2]/12 - c[5,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           c[0,2]*l[1]/12 - c[0,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[0,3]*l[2]/12 - c[0,3]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[0,4]*l[1]*l[2]/(9*l[0]) + c[4,2]*l[0]*l[1]/(9*l[2]) - c[4,2]*l[1]*np.tan(g[1])/6 + c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[4,3]*l[0]/12 - c[4,3]*l[1]*np.tan(g[0])/12 + c[4,3]*l[2]*np.tan(g[1])/12 + c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[4,4]*l[1]/12 - c[4,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[5,2]*l[0]/12 - c[5,2]*l[1]*np.tan(g[0])/12 + c[5,2]*l[2]*np.tan(g[1])/12 + c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[5,3]*l[0]*l[2]/(9*l[1]) + c[5,3]*l[2]*np.tan(g[0])/6 + c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[5,4]*l[2]/12 - c[5,4]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           c[0,0]*l[1]*l[2]/(36*l[0]) - c[0,4]*l[1]/24 - c[0,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[0,5]*l[2]/24 - c[0,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[4,0]*l[1]/24 - c[4,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[4,4]*l[0]*l[1]/(18*l[2]) + c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[4,5]*l[0]/12 - c[4,5]*l[1]*np.tan(g[0])/24 - c[4,5]*l[2]*np.tan(g[1])/24 + c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,0]*l[2]/24 - c[5,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[5,4]*l[0]/12 + c[5,4]*l[1]*np.tan(g[0])/24 + c[5,4]*l[2]*np.tan(g[1])/24 + c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,5]*l[0]*l[2]/(18*l[1]) + c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]),
           c[0,1]*l[2]/24 - c[0,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[0,3]*l[1]/24 - c[0,3]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[0,5]*l[1]*l[2]/(36*l[0]) + c[4,1]*l[0]/12 - c[4,1]*l[1]*np.tan(g[0])/24 - c[4,1]*l[2]*np.tan(g[1])/24 + c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[4,3]*l[0]*l[1]/(18*l[2]) + c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[4,5]*l[1]/24 - c[4,5]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,1]*l[0]*l[2]/(18*l[1]) + c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[5,3]*l[0]/12 + c[5,3]*l[1]*np.tan(g[0])/24 + c[5,3]*l[2]*np.tan(g[1])/24 + c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,5]*l[2]/24 - c[5,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           -c[0,2]*l[1]/24 - c[0,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[0,3]*l[2]/24 - c[0,3]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[0,4]*l[1]*l[2]/(36*l[0]) - c[4,2]*l[0]*l[1]/(18*l[2]) + c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[4,3]*l[0]/12 - c[4,3]*l[1]*np.tan(g[0])/24 - c[4,3]*l[2]*np.tan(g[1])/24 + c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[4,4]*l[1]/24 - c[4,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[5,2]*l[0]/12 + c[5,2]*l[1]*np.tan(g[0])/24 + c[5,2]*l[2]*np.tan(g[1])/24 + c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,3]*l[0]*l[2]/(18*l[1]) + c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[5,4]*l[2]/24 - c[5,4]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           -c[0,0]*l[1]*l[2]/(36*l[0]) - c[0,4]*l[1]/24 + c[0,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[0,5]*l[2]/24 + c[0,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[4,0]*l[1]/24 + c[4,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[4,4]*l[0]*l[1]/(36*l[2]) + c[4,4]*l[1]*np.tan(g[1])/12 - c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[4,5]*l[0]/24 + c[4,5]*l[1]*np.tan(g[0])/24 - c[4,5]*l[2]*np.tan(g[1])/24 - c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[5,0]*l[2]/24 + c[5,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[5,4]*l[0]/24 + c[5,4]*l[1]*np.tan(g[0])/24 - c[5,4]*l[2]*np.tan(g[1])/24 - c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,5]*l[0]*l[2]/(36*l[1]) - c[5,5]*l[2]*np.tan(g[0])/12 - c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]),
           c[0,1]*l[2]/24 + c[0,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[0,3]*l[1]/24 + c[0,3]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[0,5]*l[1]*l[2]/(36*l[0]) + c[4,1]*l[0]/24 + c[4,1]*l[1]*np.tan(g[0])/24 - c[4,1]*l[2]*np.tan(g[1])/24 - c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[4,3]*l[0]*l[1]/(36*l[2]) + c[4,3]*l[1]*np.tan(g[1])/12 - c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[4,5]*l[1]/24 + c[4,5]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,1]*l[0]*l[2]/(36*l[1]) - c[5,1]*l[2]*np.tan(g[0])/12 - c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[5,3]*l[0]/24 + c[5,3]*l[1]*np.tan(g[0])/24 - c[5,3]*l[2]*np.tan(g[1])/24 - c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[5,5]*l[2]/24 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           -c[0,2]*l[1]/24 + c[0,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[0,3]*l[2]/24 + c[0,3]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[0,4]*l[1]*l[2]/(36*l[0]) - c[4,2]*l[0]*l[1]/(36*l[2]) + c[4,2]*l[1]*np.tan(g[1])/12 - c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[4,3]*l[0]/24 + c[4,3]*l[1]*np.tan(g[0])/24 - c[4,3]*l[2]*np.tan(g[1])/24 - c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[4,4]*l[1]/24 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[5,2]*l[0]/24 + c[5,2]*l[1]*np.tan(g[0])/24 - c[5,2]*l[2]*np.tan(g[1])/24 - c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,3]*l[0]*l[2]/(36*l[1]) - c[5,3]*l[2]*np.tan(g[0])/12 - c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[5,4]*l[2]/24 + c[5,4]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           -c[0,0]*l[1]*l[2]/(18*l[0]) - c[0,4]*l[1]/12 + c[0,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[0,5]*l[2]/24 + c[0,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,0]*l[1]/12 + c[4,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,4]*l[0]*l[1]/(18*l[2]) + c[4,4]*l[1]*np.tan(g[1])/6 - c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[4,5]*l[0]/24 + c[4,5]*l[1]*np.tan(g[0])/12 + c[4,5]*l[2]*np.tan(g[1])/24 - c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,0]*l[2]/24 + c[5,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[5,4]*l[0]/24 + c[5,4]*l[1]*np.tan(g[0])/12 - c[5,4]*l[2]*np.tan(g[1])/24 - c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,5]*l[0]*l[2]/(36*l[1]) - c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]),
           -c[0,1]*l[2]/24 + c[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[0,3]*l[1]/12 + c[0,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[0,5]*l[1]*l[2]/(18*l[0]) - c[4,1]*l[0]/24 + c[4,1]*l[1]*np.tan(g[0])/12 + c[4,1]*l[2]*np.tan(g[1])/24 - c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[4,3]*l[0]*l[1]/(18*l[2]) + c[4,3]*l[1]*np.tan(g[1])/6 - c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[4,5]*l[1]/12 + c[4,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,1]*l[0]*l[2]/(36*l[1]) - c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[5,3]*l[0]/24 + c[5,3]*l[1]*np.tan(g[0])/12 - c[5,3]*l[2]*np.tan(g[1])/24 - c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,5]*l[2]/24 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[0,2]*l[1]/12 + c[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[0,3]*l[2]/24 + c[0,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[0,4]*l[1]*l[2]/(18*l[0]) - c[4,2]*l[0]*l[1]/(18*l[2]) + c[4,2]*l[1]*np.tan(g[1])/6 - c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[4,3]*l[0]/24 + c[4,3]*l[1]*np.tan(g[0])/12 + c[4,3]*l[2]*np.tan(g[1])/24 - c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[4,4]*l[1]/12 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,2]*l[0]/24 + c[5,2]*l[1]*np.tan(g[0])/12 - c[5,2]*l[2]*np.tan(g[1])/24 - c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,3]*l[0]*l[2]/(36*l[1]) - c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[5,4]*l[2]/24 + c[5,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[0,0]*l[1]*l[2]/(18*l[0]) - c[0,4]*l[1]/12 - c[0,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[0,5]*l[2]/24 - c[0,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,0]*l[1]/12 - c[4,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,4]*l[0]*l[1]/(9*l[2]) + c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[4,5]*l[0]/12 - c[4,5]*l[1]*np.tan(g[0])/12 + c[4,5]*l[2]*np.tan(g[1])/24 + c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,0]*l[2]/24 - c[5,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[5,4]*l[0]/12 + c[5,4]*l[1]*np.tan(g[0])/12 + c[5,4]*l[2]*np.tan(g[1])/24 + c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,5]*l[0]*l[2]/(18*l[1]) + c[5,5]*l[2]*np.tan(g[0])/12 + c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]),
           -c[0,1]*l[2]/24 - c[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[0,3]*l[1]/12 - c[0,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[0,5]*l[1]*l[2]/(18*l[0]) - c[4,1]*l[0]/12 - c[4,1]*l[1]*np.tan(g[0])/12 + c[4,1]*l[2]*np.tan(g[1])/24 + c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[4,3]*l[0]*l[1]/(9*l[2]) + c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[4,5]*l[1]/12 - c[4,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,1]*l[0]*l[2]/(18*l[1]) + c[5,1]*l[2]*np.tan(g[0])/12 + c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[5,3]*l[0]/12 + c[5,3]*l[1]*np.tan(g[0])/12 + c[5,3]*l[2]*np.tan(g[1])/24 + c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,5]*l[2]/24 - c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[0,2]*l[1]/12 - c[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[0,3]*l[2]/24 - c[0,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[0,4]*l[1]*l[2]/(18*l[0]) - c[4,2]*l[0]*l[1]/(9*l[2]) + c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[4,3]*l[0]/12 - c[4,3]*l[1]*np.tan(g[0])/12 + c[4,3]*l[2]*np.tan(g[1])/24 + c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[4,4]*l[1]/12 - c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,2]*l[0]/12 + c[5,2]*l[1]*np.tan(g[0])/12 + c[5,2]*l[2]*np.tan(g[1])/24 + c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,3]*l[0]*l[2]/(18*l[1]) + c[5,3]*l[2]*np.tan(g[0])/12 + c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[5,4]*l[2]/24 - c[5,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0])],
          [-c[1,0]*l[2]/12 - c[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[1,4]*l[0]/12 - c[1,4]*l[1]*np.tan(g[0])/24 + c[1,4]*l[2]*np.tan(g[1])/12 + c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[1,5]*l[0]*l[2]/(9*l[1]) + c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[3,0]*l[1]/24 - c[3,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[3,4]*l[0]*l[1]/(18*l[2]) - c[3,4]*l[1]*np.tan(g[1])/12 + c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[3,5]*l[0]/12 - c[3,5]*l[1]*np.tan(g[0])/24 - c[3,5]*l[2]*np.tan(g[1])/12 + c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,0]*l[1]*l[2]/(18*l[0]) + c[5,4]*l[1]/24 - c[5,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,5]*l[2]/12 - c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[1,1]*l[0]*l[2]/(9*l[1]) + c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[1,3]*l[0]/12 - c[1,3]*l[1]*np.tan(g[0])/24 + c[1,3]*l[2]*np.tan(g[1])/12 + c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[1,5]*l[2]/12 - c[1,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[3,1]*l[0]/12 - c[3,1]*l[1]*np.tan(g[0])/24 - c[3,1]*l[2]*np.tan(g[1])/12 + c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,3]*l[0]*l[1]/(18*l[2]) - c[3,3]*l[1]*np.tan(g[1])/12 + c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[3,5]*l[1]/24 - c[3,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,1]*l[2]/12 - c[5,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[5,3]*l[1]/24 - c[5,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,5]*l[1]*l[2]/(18*l[0]),
           -c[1,2]*l[0]/12 - c[1,2]*l[1]*np.tan(g[0])/24 + c[1,2]*l[2]*np.tan(g[1])/12 + c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[1,3]*l[0]*l[2]/(9*l[1]) + c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[1,4]*l[2]/12 - c[1,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[3,2]*l[0]*l[1]/(18*l[2]) - c[3,2]*l[1]*np.tan(g[1])/12 + c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[3,3]*l[0]/12 - c[3,3]*l[1]*np.tan(g[0])/24 - c[3,3]*l[2]*np.tan(g[1])/12 + c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,4]*l[1]/24 - c[3,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,2]*l[1]/24 - c[5,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,3]*l[2]/12 - c[5,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[5,4]*l[1]*l[2]/(18*l[0]),
           c[1,0]*l[2]/12 + c[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[1,4]*l[0]/24 - c[1,4]*l[1]*np.tan(g[0])/24 - c[1,4]*l[2]*np.tan(g[1])/12 - c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[1,5]*l[0]*l[2]/(18*l[1]) - c[1,5]*l[2]*np.tan(g[0])/6 - c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[3,0]*l[1]/24 + c[3,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[3,4]*l[0]*l[1]/(36*l[2]) - c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[3,5]*l[0]/24 + c[3,5]*l[1]*np.tan(g[0])/24 - c[3,5]*l[2]*np.tan(g[1])/12 - c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,0]*l[1]*l[2]/(18*l[0]) + c[5,4]*l[1]/24 + c[5,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,5]*l[2]/12 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[1,1]*l[0]*l[2]/(18*l[1]) - c[1,1]*l[2]*np.tan(g[0])/6 - c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[1,3]*l[0]/24 - c[1,3]*l[1]*np.tan(g[0])/24 - c[1,3]*l[2]*np.tan(g[1])/12 - c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[1,5]*l[2]/12 + c[1,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[3,1]*l[0]/24 + c[3,1]*l[1]*np.tan(g[0])/24 - c[3,1]*l[2]*np.tan(g[1])/12 - c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,3]*l[0]*l[1]/(36*l[2]) - c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[3,5]*l[1]/24 + c[3,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,1]*l[2]/12 + c[5,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[5,3]*l[1]/24 + c[5,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,5]*l[1]*l[2]/(18*l[0]),
           -c[1,2]*l[0]/24 - c[1,2]*l[1]*np.tan(g[0])/24 - c[1,2]*l[2]*np.tan(g[1])/12 - c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[1,3]*l[0]*l[2]/(18*l[1]) - c[1,3]*l[2]*np.tan(g[0])/6 - c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[1,4]*l[2]/12 + c[1,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[3,2]*l[0]*l[1]/(36*l[2]) - c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[3,3]*l[0]/24 + c[3,3]*l[1]*np.tan(g[0])/24 - c[3,3]*l[2]*np.tan(g[1])/12 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,4]*l[1]/24 + c[3,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,2]*l[1]/24 + c[5,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,3]*l[2]/12 + c[5,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[5,4]*l[1]*l[2]/(18*l[0]),
           c[1,0]*l[2]/12 + c[1,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[1,4]*l[0]/24 - c[1,4]*l[1]*np.tan(g[0])/12 - c[1,4]*l[2]*np.tan(g[1])/12 - c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[1,5]*l[0]*l[2]/(18*l[1]) - c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[3,0]*l[1]/12 + c[3,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[3,4]*l[0]*l[1]/(18*l[2]) - c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[3,5]*l[0]/24 + c[3,5]*l[1]*np.tan(g[0])/12 + c[3,5]*l[2]*np.tan(g[1])/12 - c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[5,0]*l[1]*l[2]/(9*l[0]) + c[5,4]*l[1]/12 + c[5,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[5,5]*l[2]/12 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           c[1,1]*l[0]*l[2]/(18*l[1]) - c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[1,3]*l[0]/24 - c[1,3]*l[1]*np.tan(g[0])/12 - c[1,3]*l[2]*np.tan(g[1])/12 - c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[1,5]*l[2]/12 + c[1,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[3,1]*l[0]/24 + c[3,1]*l[1]*np.tan(g[0])/12 + c[3,1]*l[2]*np.tan(g[1])/12 - c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,3]*l[0]*l[1]/(18*l[2]) - c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[3,5]*l[1]/12 + c[3,5]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[5,1]*l[2]/12 + c[5,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[5,3]*l[1]/12 + c[5,3]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[5,5]*l[1]*l[2]/(9*l[0]),
           -c[1,2]*l[0]/24 - c[1,2]*l[1]*np.tan(g[0])/12 - c[1,2]*l[2]*np.tan(g[1])/12 - c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[1,3]*l[0]*l[2]/(18*l[1]) - c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[1,4]*l[2]/12 + c[1,4]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[3,2]*l[0]*l[1]/(18*l[2]) - c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[3,3]*l[0]/24 + c[3,3]*l[1]*np.tan(g[0])/12 + c[3,3]*l[2]*np.tan(g[1])/12 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[3,4]*l[1]/12 + c[3,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,2]*l[1]/12 + c[5,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[5,3]*l[2]/12 + c[5,3]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[5,4]*l[1]*l[2]/(9*l[0]),
           -c[1,0]*l[2]/12 - c[1,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[1,4]*l[0]/12 - c[1,4]*l[1]*np.tan(g[0])/12 + c[1,4]*l[2]*np.tan(g[1])/12 + c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[1,5]*l[0]*l[2]/(9*l[1]) + c[1,5]*l[2]*np.tan(g[0])/6 + c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[3,0]*l[1]/12 - c[3,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[3,4]*l[0]*l[1]/(9*l[2]) - c[3,4]*l[1]*np.tan(g[1])/6 + c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[3,5]*l[0]/12 - c[3,5]*l[1]*np.tan(g[0])/12 + c[3,5]*l[2]*np.tan(g[1])/12 + c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[5,0]*l[1]*l[2]/(9*l[0]) + c[5,4]*l[1]/12 - c[5,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[5,5]*l[2]/12 - c[5,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           c[1,1]*l[0]*l[2]/(9*l[1]) + c[1,1]*l[2]*np.tan(g[0])/6 + c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[1,3]*l[0]/12 - c[1,3]*l[1]*np.tan(g[0])/12 + c[1,3]*l[2]*np.tan(g[1])/12 + c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[1,5]*l[2]/12 - c[1,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[3,1]*l[0]/12 - c[3,1]*l[1]*np.tan(g[0])/12 + c[3,1]*l[2]*np.tan(g[1])/12 + c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,3]*l[0]*l[1]/(9*l[2]) - c[3,3]*l[1]*np.tan(g[1])/6 + c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[3,5]*l[1]/12 - c[3,5]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[5,1]*l[2]/12 - c[5,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[5,3]*l[1]/12 - c[5,3]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,5]*l[1]*l[2]/(9*l[0]),
           -c[1,2]*l[0]/12 - c[1,2]*l[1]*np.tan(g[0])/12 + c[1,2]*l[2]*np.tan(g[1])/12 + c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[1,3]*l[0]*l[2]/(9*l[1]) + c[1,3]*l[2]*np.tan(g[0])/6 + c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[1,4]*l[2]/12 - c[1,4]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[3,2]*l[0]*l[1]/(9*l[2]) - c[3,2]*l[1]*np.tan(g[1])/6 + c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[3,3]*l[0]/12 - c[3,3]*l[1]*np.tan(g[0])/12 + c[3,3]*l[2]*np.tan(g[1])/12 + c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,4]*l[1]/12 - c[3,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,2]*l[1]/12 - c[5,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[5,3]*l[2]/12 - c[5,3]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[5,4]*l[1]*l[2]/(9*l[0]),
           -c[1,0]*l[2]/24 - c[1,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[1,4]*l[0]/12 + c[1,4]*l[1]*np.tan(g[0])/24 + c[1,4]*l[2]*np.tan(g[1])/24 + c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[1,5]*l[0]*l[2]/(18*l[1]) + c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[3,0]*l[1]/24 - c[3,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[3,4]*l[0]*l[1]/(18*l[2]) + c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[3,5]*l[0]/12 - c[3,5]*l[1]*np.tan(g[0])/24 - c[3,5]*l[2]*np.tan(g[1])/24 + c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[5,0]*l[1]*l[2]/(36*l[0]) - c[5,4]*l[1]/24 - c[5,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[5,5]*l[2]/24 - c[5,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           -c[1,1]*l[0]*l[2]/(18*l[1]) + c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[1,3]*l[0]/12 + c[1,3]*l[1]*np.tan(g[0])/24 + c[1,3]*l[2]*np.tan(g[1])/24 + c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[1,5]*l[2]/24 - c[1,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[3,1]*l[0]/12 - c[3,1]*l[1]*np.tan(g[0])/24 - c[3,1]*l[2]*np.tan(g[1])/24 + c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,3]*l[0]*l[1]/(18*l[2]) + c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[3,5]*l[1]/24 - c[3,5]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[5,1]*l[2]/24 - c[5,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[5,3]*l[1]/24 - c[5,3]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[5,5]*l[1]*l[2]/(36*l[0]),
           c[1,2]*l[0]/12 + c[1,2]*l[1]*np.tan(g[0])/24 + c[1,2]*l[2]*np.tan(g[1])/24 + c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[1,3]*l[0]*l[2]/(18*l[1]) + c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[1,4]*l[2]/24 - c[1,4]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[3,2]*l[0]*l[1]/(18*l[2]) + c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[3,3]*l[0]/12 - c[3,3]*l[1]*np.tan(g[0])/24 - c[3,3]*l[2]*np.tan(g[1])/24 + c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[3,4]*l[1]/24 - c[3,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,2]*l[1]/24 - c[5,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[5,3]*l[2]/24 - c[5,3]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[5,4]*l[1]*l[2]/(36*l[0]),
           c[1,0]*l[2]/24 + c[1,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[1,4]*l[0]/24 + c[1,4]*l[1]*np.tan(g[0])/24 - c[1,4]*l[2]*np.tan(g[1])/24 - c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[1,5]*l[0]*l[2]/(36*l[1]) - c[1,5]*l[2]*np.tan(g[0])/12 - c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[3,0]*l[1]/24 + c[3,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[3,4]*l[0]*l[1]/(36*l[2]) + c[3,4]*l[1]*np.tan(g[1])/12 - c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[3,5]*l[0]/24 + c[3,5]*l[1]*np.tan(g[0])/24 - c[3,5]*l[2]*np.tan(g[1])/24 - c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,0]*l[1]*l[2]/(36*l[0]) - c[5,4]*l[1]/24 + c[5,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[5,5]*l[2]/24 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           -c[1,1]*l[0]*l[2]/(36*l[1]) - c[1,1]*l[2]*np.tan(g[0])/12 - c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[1,3]*l[0]/24 + c[1,3]*l[1]*np.tan(g[0])/24 - c[1,3]*l[2]*np.tan(g[1])/24 - c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[1,5]*l[2]/24 + c[1,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[3,1]*l[0]/24 + c[3,1]*l[1]*np.tan(g[0])/24 - c[3,1]*l[2]*np.tan(g[1])/24 - c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,3]*l[0]*l[1]/(36*l[2]) + c[3,3]*l[1]*np.tan(g[1])/12 - c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[3,5]*l[1]/24 + c[3,5]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[5,1]*l[2]/24 + c[5,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[5,3]*l[1]/24 + c[5,3]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,5]*l[1]*l[2]/(36*l[0]),
           c[1,2]*l[0]/24 + c[1,2]*l[1]*np.tan(g[0])/24 - c[1,2]*l[2]*np.tan(g[1])/24 - c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[1,3]*l[0]*l[2]/(36*l[1]) - c[1,3]*l[2]*np.tan(g[0])/12 - c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[1,4]*l[2]/24 + c[1,4]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[3,2]*l[0]*l[1]/(36*l[2]) + c[3,2]*l[1]*np.tan(g[1])/12 - c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[3,3]*l[0]/24 + c[3,3]*l[1]*np.tan(g[0])/24 - c[3,3]*l[2]*np.tan(g[1])/24 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,4]*l[1]/24 + c[3,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,2]*l[1]/24 + c[5,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[5,3]*l[2]/24 + c[5,3]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[5,4]*l[1]*l[2]/(36*l[0]),
           c[1,0]*l[2]/24 + c[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[1,4]*l[0]/24 + c[1,4]*l[1]*np.tan(g[0])/12 - c[1,4]*l[2]*np.tan(g[1])/24 - c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[1,5]*l[0]*l[2]/(36*l[1]) - c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[3,0]*l[1]/12 + c[3,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[3,4]*l[0]*l[1]/(18*l[2]) + c[3,4]*l[1]*np.tan(g[1])/6 - c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[3,5]*l[0]/24 + c[3,5]*l[1]*np.tan(g[0])/12 + c[3,5]*l[2]*np.tan(g[1])/24 - c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,0]*l[1]*l[2]/(18*l[0]) - c[5,4]*l[1]/12 + c[5,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,5]*l[2]/24 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[1,1]*l[0]*l[2]/(36*l[1]) - c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[1,3]*l[0]/24 + c[1,3]*l[1]*np.tan(g[0])/12 - c[1,3]*l[2]*np.tan(g[1])/24 - c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[1,5]*l[2]/24 + c[1,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[3,1]*l[0]/24 + c[3,1]*l[1]*np.tan(g[0])/12 + c[3,1]*l[2]*np.tan(g[1])/24 - c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,3]*l[0]*l[1]/(18*l[2]) + c[3,3]*l[1]*np.tan(g[1])/6 - c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[3,5]*l[1]/12 + c[3,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,1]*l[2]/24 + c[5,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[5,3]*l[1]/12 + c[5,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,5]*l[1]*l[2]/(18*l[0]),
           c[1,2]*l[0]/24 + c[1,2]*l[1]*np.tan(g[0])/12 - c[1,2]*l[2]*np.tan(g[1])/24 - c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[1,3]*l[0]*l[2]/(36*l[1]) - c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[1,4]*l[2]/24 + c[1,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[3,2]*l[0]*l[1]/(18*l[2]) + c[3,2]*l[1]*np.tan(g[1])/6 - c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[3,3]*l[0]/24 + c[3,3]*l[1]*np.tan(g[0])/12 + c[3,3]*l[2]*np.tan(g[1])/24 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,4]*l[1]/12 + c[3,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,2]*l[1]/12 + c[5,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,3]*l[2]/24 + c[5,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[5,4]*l[1]*l[2]/(18*l[0]),
           -c[1,0]*l[2]/24 - c[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[1,4]*l[0]/12 + c[1,4]*l[1]*np.tan(g[0])/12 + c[1,4]*l[2]*np.tan(g[1])/24 + c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[1,5]*l[0]*l[2]/(18*l[1]) + c[1,5]*l[2]*np.tan(g[0])/12 + c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[3,0]*l[1]/12 - c[3,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[3,4]*l[0]*l[1]/(9*l[2]) + c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[3,5]*l[0]/12 - c[3,5]*l[1]*np.tan(g[0])/12 + c[3,5]*l[2]*np.tan(g[1])/24 + c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,0]*l[1]*l[2]/(18*l[0]) - c[5,4]*l[1]/12 - c[5,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,5]*l[2]/24 - c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[1,1]*l[0]*l[2]/(18*l[1]) + c[1,1]*l[2]*np.tan(g[0])/12 + c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[1,3]*l[0]/12 + c[1,3]*l[1]*np.tan(g[0])/12 + c[1,3]*l[2]*np.tan(g[1])/24 + c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[1,5]*l[2]/24 - c[1,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[3,1]*l[0]/12 - c[3,1]*l[1]*np.tan(g[0])/12 + c[3,1]*l[2]*np.tan(g[1])/24 + c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,3]*l[0]*l[1]/(9*l[2]) + c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[3,5]*l[1]/12 - c[3,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,1]*l[2]/24 - c[5,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[5,3]*l[1]/12 - c[5,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,5]*l[1]*l[2]/(18*l[0]),
           c[1,2]*l[0]/12 + c[1,2]*l[1]*np.tan(g[0])/12 + c[1,2]*l[2]*np.tan(g[1])/24 + c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[1,3]*l[0]*l[2]/(18*l[1]) + c[1,3]*l[2]*np.tan(g[0])/12 + c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[1,4]*l[2]/24 - c[1,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[3,2]*l[0]*l[1]/(9*l[2]) + c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[3,3]*l[0]/12 - c[3,3]*l[1]*np.tan(g[0])/12 + c[3,3]*l[2]*np.tan(g[1])/24 + c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,4]*l[1]/12 - c[3,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,2]*l[1]/12 - c[5,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,3]*l[2]/24 - c[5,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[5,4]*l[1]*l[2]/(18*l[0])],
          [c[2,0]*l[1]/24 - c[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[2,4]*l[0]*l[1]/(18*l[2]) - c[2,4]*l[1]*np.tan(g[1])/12 + c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[2,5]*l[0]/12 - c[2,5]*l[1]*np.tan(g[0])/24 - c[2,5]*l[2]*np.tan(g[1])/12 + c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,0]*l[2]/12 - c[3,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[3,4]*l[0]/12 - c[3,4]*l[1]*np.tan(g[0])/24 + c[3,4]*l[2]*np.tan(g[1])/12 + c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,5]*l[0]*l[2]/(9*l[1]) + c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[4,0]*l[1]*l[2]/(18*l[0]) + c[4,4]*l[1]/24 - c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,5]*l[2]/12 - c[4,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[2,1]*l[0]/12 - c[2,1]*l[1]*np.tan(g[0])/24 - c[2,1]*l[2]*np.tan(g[1])/12 + c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[2,3]*l[0]*l[1]/(18*l[2]) - c[2,3]*l[1]*np.tan(g[1])/12 + c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[2,5]*l[1]/24 - c[2,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[3,1]*l[0]*l[2]/(9*l[1]) + c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[3,3]*l[0]/12 - c[3,3]*l[1]*np.tan(g[0])/24 + c[3,3]*l[2]*np.tan(g[1])/12 + c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,5]*l[2]/12 - c[3,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,1]*l[2]/12 - c[4,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,3]*l[1]/24 - c[4,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,5]*l[1]*l[2]/(18*l[0]),
           c[2,2]*l[0]*l[1]/(18*l[2]) - c[2,2]*l[1]*np.tan(g[1])/12 + c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[2,3]*l[0]/12 - c[2,3]*l[1]*np.tan(g[0])/24 - c[2,3]*l[2]*np.tan(g[1])/12 + c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[2,4]*l[1]/24 - c[2,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[3,2]*l[0]/12 - c[3,2]*l[1]*np.tan(g[0])/24 + c[3,2]*l[2]*np.tan(g[1])/12 + c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,3]*l[0]*l[2]/(9*l[1]) + c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[3,4]*l[2]/12 - c[3,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,2]*l[1]/24 - c[4,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,3]*l[2]/12 - c[4,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,4]*l[1]*l[2]/(18*l[0]),
           -c[2,0]*l[1]/24 + c[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[2,4]*l[0]*l[1]/(36*l[2]) - c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[2,5]*l[0]/24 + c[2,5]*l[1]*np.tan(g[0])/24 - c[2,5]*l[2]*np.tan(g[1])/12 - c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,0]*l[2]/12 + c[3,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[3,4]*l[0]/24 - c[3,4]*l[1]*np.tan(g[0])/24 - c[3,4]*l[2]*np.tan(g[1])/12 - c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,5]*l[0]*l[2]/(18*l[1]) - c[3,5]*l[2]*np.tan(g[0])/6 - c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[4,0]*l[1]*l[2]/(18*l[0]) + c[4,4]*l[1]/24 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,5]*l[2]/12 + c[4,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[2,1]*l[0]/24 + c[2,1]*l[1]*np.tan(g[0])/24 - c[2,1]*l[2]*np.tan(g[1])/12 - c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[2,3]*l[0]*l[1]/(36*l[2]) - c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[2,5]*l[1]/24 + c[2,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[3,1]*l[0]*l[2]/(18*l[1]) - c[3,1]*l[2]*np.tan(g[0])/6 - c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[3,3]*l[0]/24 - c[3,3]*l[1]*np.tan(g[0])/24 - c[3,3]*l[2]*np.tan(g[1])/12 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,5]*l[2]/12 + c[3,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,1]*l[2]/12 + c[4,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,3]*l[1]/24 + c[4,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,5]*l[1]*l[2]/(18*l[0]),
           c[2,2]*l[0]*l[1]/(36*l[2]) - c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[2,3]*l[0]/24 + c[2,3]*l[1]*np.tan(g[0])/24 - c[2,3]*l[2]*np.tan(g[1])/12 - c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[2,4]*l[1]/24 + c[2,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[3,2]*l[0]/24 - c[3,2]*l[1]*np.tan(g[0])/24 - c[3,2]*l[2]*np.tan(g[1])/12 - c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,3]*l[0]*l[2]/(18*l[1]) - c[3,3]*l[2]*np.tan(g[0])/6 - c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[3,4]*l[2]/12 + c[3,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,2]*l[1]/24 + c[4,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,3]*l[2]/12 + c[4,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,4]*l[1]*l[2]/(18*l[0]),
           -c[2,0]*l[1]/12 + c[2,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[2,4]*l[0]*l[1]/(18*l[2]) - c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[2,5]*l[0]/24 + c[2,5]*l[1]*np.tan(g[0])/12 + c[2,5]*l[2]*np.tan(g[1])/12 - c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,0]*l[2]/12 + c[3,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[3,4]*l[0]/24 - c[3,4]*l[1]*np.tan(g[0])/12 - c[3,4]*l[2]*np.tan(g[1])/12 - c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,5]*l[0]*l[2]/(18*l[1]) - c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[4,0]*l[1]*l[2]/(9*l[0]) + c[4,4]*l[1]/12 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[4,5]*l[2]/12 + c[4,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           -c[2,1]*l[0]/24 + c[2,1]*l[1]*np.tan(g[0])/12 + c[2,1]*l[2]*np.tan(g[1])/12 - c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[2,3]*l[0]*l[1]/(18*l[2]) - c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[2,5]*l[1]/12 + c[2,5]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[3,1]*l[0]*l[2]/(18*l[1]) - c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[3,3]*l[0]/24 - c[3,3]*l[1]*np.tan(g[0])/12 - c[3,3]*l[2]*np.tan(g[1])/12 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,5]*l[2]/12 + c[3,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[4,1]*l[2]/12 + c[4,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[4,3]*l[1]/12 + c[4,3]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[4,5]*l[1]*l[2]/(9*l[0]),
           c[2,2]*l[0]*l[1]/(18*l[2]) - c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[2,3]*l[0]/24 + c[2,3]*l[1]*np.tan(g[0])/12 + c[2,3]*l[2]*np.tan(g[1])/12 - c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[2,4]*l[1]/12 + c[2,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[3,2]*l[0]/24 - c[3,2]*l[1]*np.tan(g[0])/12 - c[3,2]*l[2]*np.tan(g[1])/12 - c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,3]*l[0]*l[2]/(18*l[1]) - c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[3,4]*l[2]/12 + c[3,4]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[4,2]*l[1]/12 + c[4,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[4,3]*l[2]/12 + c[4,3]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[4,4]*l[1]*l[2]/(9*l[0]),
           c[2,0]*l[1]/12 - c[2,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[2,4]*l[0]*l[1]/(9*l[2]) - c[2,4]*l[1]*np.tan(g[1])/6 + c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[2,5]*l[0]/12 - c[2,5]*l[1]*np.tan(g[0])/12 + c[2,5]*l[2]*np.tan(g[1])/12 + c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[3,0]*l[2]/12 - c[3,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[3,4]*l[0]/12 - c[3,4]*l[1]*np.tan(g[0])/12 + c[3,4]*l[2]*np.tan(g[1])/12 + c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,5]*l[0]*l[2]/(9*l[1]) + c[3,5]*l[2]*np.tan(g[0])/6 + c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[4,0]*l[1]*l[2]/(9*l[0]) + c[4,4]*l[1]/12 - c[4,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[4,5]*l[2]/12 - c[4,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           -c[2,1]*l[0]/12 - c[2,1]*l[1]*np.tan(g[0])/12 + c[2,1]*l[2]*np.tan(g[1])/12 + c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[2,3]*l[0]*l[1]/(9*l[2]) - c[2,3]*l[1]*np.tan(g[1])/6 + c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[2,5]*l[1]/12 - c[2,5]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[3,1]*l[0]*l[2]/(9*l[1]) + c[3,1]*l[2]*np.tan(g[0])/6 + c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[3,3]*l[0]/12 - c[3,3]*l[1]*np.tan(g[0])/12 + c[3,3]*l[2]*np.tan(g[1])/12 + c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[3,5]*l[2]/12 - c[3,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[4,1]*l[2]/12 - c[4,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[4,3]*l[1]/12 - c[4,3]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[4,5]*l[1]*l[2]/(9*l[0]),
           c[2,2]*l[0]*l[1]/(9*l[2]) - c[2,2]*l[1]*np.tan(g[1])/6 + c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[2,3]*l[0]/12 - c[2,3]*l[1]*np.tan(g[0])/12 + c[2,3]*l[2]*np.tan(g[1])/12 + c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[2,4]*l[1]/12 - c[2,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[3,2]*l[0]/12 - c[3,2]*l[1]*np.tan(g[0])/12 + c[3,2]*l[2]*np.tan(g[1])/12 + c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,3]*l[0]*l[2]/(9*l[1]) + c[3,3]*l[2]*np.tan(g[0])/6 + c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[3,4]*l[2]/12 - c[3,4]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[4,2]*l[1]/12 - c[4,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[4,3]*l[2]/12 - c[4,3]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[4,4]*l[1]*l[2]/(9*l[0]),
           c[2,0]*l[1]/24 - c[2,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[2,4]*l[0]*l[1]/(18*l[2]) + c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[2,5]*l[0]/12 - c[2,5]*l[1]*np.tan(g[0])/24 - c[2,5]*l[2]*np.tan(g[1])/24 + c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,0]*l[2]/24 - c[3,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[3,4]*l[0]/12 + c[3,4]*l[1]*np.tan(g[0])/24 + c[3,4]*l[2]*np.tan(g[1])/24 + c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,5]*l[0]*l[2]/(18*l[1]) + c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[4,0]*l[1]*l[2]/(36*l[0]) - c[4,4]*l[1]/24 - c[4,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[4,5]*l[2]/24 - c[4,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           c[2,1]*l[0]/12 - c[2,1]*l[1]*np.tan(g[0])/24 - c[2,1]*l[2]*np.tan(g[1])/24 + c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[2,3]*l[0]*l[1]/(18*l[2]) + c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[2,5]*l[1]/24 - c[2,5]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[3,1]*l[0]*l[2]/(18*l[1]) + c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[3,3]*l[0]/12 + c[3,3]*l[1]*np.tan(g[0])/24 + c[3,3]*l[2]*np.tan(g[1])/24 + c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,5]*l[2]/24 - c[3,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[4,1]*l[2]/24 - c[4,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[4,3]*l[1]/24 - c[4,3]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[4,5]*l[1]*l[2]/(36*l[0]),
           -c[2,2]*l[0]*l[1]/(18*l[2]) + c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[2,3]*l[0]/12 - c[2,3]*l[1]*np.tan(g[0])/24 - c[2,3]*l[2]*np.tan(g[1])/24 + c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[2,4]*l[1]/24 - c[2,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[3,2]*l[0]/12 + c[3,2]*l[1]*np.tan(g[0])/24 + c[3,2]*l[2]*np.tan(g[1])/24 + c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,3]*l[0]*l[2]/(18*l[1]) + c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[3,4]*l[2]/24 - c[3,4]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[4,2]*l[1]/24 - c[4,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[4,3]*l[2]/24 - c[4,3]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[4,4]*l[1]*l[2]/(36*l[0]),
           -c[2,0]*l[1]/24 + c[2,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[2,4]*l[0]*l[1]/(36*l[2]) + c[2,4]*l[1]*np.tan(g[1])/12 - c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[2,5]*l[0]/24 + c[2,5]*l[1]*np.tan(g[0])/24 - c[2,5]*l[2]*np.tan(g[1])/24 - c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[3,0]*l[2]/24 + c[3,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[3,4]*l[0]/24 + c[3,4]*l[1]*np.tan(g[0])/24 - c[3,4]*l[2]*np.tan(g[1])/24 - c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,5]*l[0]*l[2]/(36*l[1]) - c[3,5]*l[2]*np.tan(g[0])/12 - c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[4,0]*l[1]*l[2]/(36*l[0]) - c[4,4]*l[1]/24 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[4,5]*l[2]/24 + c[4,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           c[2,1]*l[0]/24 + c[2,1]*l[1]*np.tan(g[0])/24 - c[2,1]*l[2]*np.tan(g[1])/24 - c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[2,3]*l[0]*l[1]/(36*l[2]) + c[2,3]*l[1]*np.tan(g[1])/12 - c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[2,5]*l[1]/24 + c[2,5]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[3,1]*l[0]*l[2]/(36*l[1]) - c[3,1]*l[2]*np.tan(g[0])/12 - c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[3,3]*l[0]/24 + c[3,3]*l[1]*np.tan(g[0])/24 - c[3,3]*l[2]*np.tan(g[1])/24 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[3,5]*l[2]/24 + c[3,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[4,1]*l[2]/24 + c[4,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[4,3]*l[1]/24 + c[4,3]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[4,5]*l[1]*l[2]/(36*l[0]),
           -c[2,2]*l[0]*l[1]/(36*l[2]) + c[2,2]*l[1]*np.tan(g[1])/12 - c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[2,3]*l[0]/24 + c[2,3]*l[1]*np.tan(g[0])/24 - c[2,3]*l[2]*np.tan(g[1])/24 - c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[2,4]*l[1]/24 + c[2,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[3,2]*l[0]/24 + c[3,2]*l[1]*np.tan(g[0])/24 - c[3,2]*l[2]*np.tan(g[1])/24 - c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,3]*l[0]*l[2]/(36*l[1]) - c[3,3]*l[2]*np.tan(g[0])/12 - c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[3,4]*l[2]/24 + c[3,4]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[4,2]*l[1]/24 + c[4,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[4,3]*l[2]/24 + c[4,3]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[4,4]*l[1]*l[2]/(36*l[0]),
           -c[2,0]*l[1]/12 + c[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[2,4]*l[0]*l[1]/(18*l[2]) + c[2,4]*l[1]*np.tan(g[1])/6 - c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[2,5]*l[0]/24 + c[2,5]*l[1]*np.tan(g[0])/12 + c[2,5]*l[2]*np.tan(g[1])/24 - c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,0]*l[2]/24 + c[3,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[3,4]*l[0]/24 + c[3,4]*l[1]*np.tan(g[0])/12 - c[3,4]*l[2]*np.tan(g[1])/24 - c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,5]*l[0]*l[2]/(36*l[1]) - c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[4,0]*l[1]*l[2]/(18*l[0]) - c[4,4]*l[1]/12 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,5]*l[2]/24 + c[4,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[2,1]*l[0]/24 + c[2,1]*l[1]*np.tan(g[0])/12 + c[2,1]*l[2]*np.tan(g[1])/24 - c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[2,3]*l[0]*l[1]/(18*l[2]) + c[2,3]*l[1]*np.tan(g[1])/6 - c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[2,5]*l[1]/12 + c[2,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[3,1]*l[0]*l[2]/(36*l[1]) - c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[3,3]*l[0]/24 + c[3,3]*l[1]*np.tan(g[0])/12 - c[3,3]*l[2]*np.tan(g[1])/24 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,5]*l[2]/24 + c[3,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,1]*l[2]/24 + c[4,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,3]*l[1]/12 + c[4,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,5]*l[1]*l[2]/(18*l[0]),
           -c[2,2]*l[0]*l[1]/(18*l[2]) + c[2,2]*l[1]*np.tan(g[1])/6 - c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[2,3]*l[0]/24 + c[2,3]*l[1]*np.tan(g[0])/12 + c[2,3]*l[2]*np.tan(g[1])/24 - c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[2,4]*l[1]/12 + c[2,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[3,2]*l[0]/24 + c[3,2]*l[1]*np.tan(g[0])/12 - c[3,2]*l[2]*np.tan(g[1])/24 - c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,3]*l[0]*l[2]/(36*l[1]) - c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[3,4]*l[2]/24 + c[3,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,2]*l[1]/12 + c[4,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,3]*l[2]/24 + c[4,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,4]*l[1]*l[2]/(18*l[0]),
           c[2,0]*l[1]/12 - c[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[2,4]*l[0]*l[1]/(9*l[2]) + c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[2,5]*l[0]/12 - c[2,5]*l[1]*np.tan(g[0])/12 + c[2,5]*l[2]*np.tan(g[1])/24 + c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,0]*l[2]/24 - c[3,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[3,4]*l[0]/12 + c[3,4]*l[1]*np.tan(g[0])/12 + c[3,4]*l[2]*np.tan(g[1])/24 + c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,5]*l[0]*l[2]/(18*l[1]) + c[3,5]*l[2]*np.tan(g[0])/12 + c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[4,0]*l[1]*l[2]/(18*l[0]) - c[4,4]*l[1]/12 - c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,5]*l[2]/24 - c[4,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[2,1]*l[0]/12 - c[2,1]*l[1]*np.tan(g[0])/12 + c[2,1]*l[2]*np.tan(g[1])/24 + c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[2,3]*l[0]*l[1]/(9*l[2]) + c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[2,5]*l[1]/12 - c[2,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[3,1]*l[0]*l[2]/(18*l[1]) + c[3,1]*l[2]*np.tan(g[0])/12 + c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[3,3]*l[0]/12 + c[3,3]*l[1]*np.tan(g[0])/12 + c[3,3]*l[2]*np.tan(g[1])/24 + c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,5]*l[2]/24 - c[3,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,1]*l[2]/24 - c[4,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,3]*l[1]/12 - c[4,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,5]*l[1]*l[2]/(18*l[0]),
           -c[2,2]*l[0]*l[1]/(9*l[2]) + c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[2,3]*l[0]/12 - c[2,3]*l[1]*np.tan(g[0])/12 + c[2,3]*l[2]*np.tan(g[1])/24 + c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[2,4]*l[1]/12 - c[2,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[3,2]*l[0]/12 + c[3,2]*l[1]*np.tan(g[0])/12 + c[3,2]*l[2]*np.tan(g[1])/24 + c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,3]*l[0]*l[2]/(18*l[1]) + c[3,3]*l[2]*np.tan(g[0])/12 + c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[3,4]*l[2]/24 - c[3,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,2]*l[1]/12 - c[4,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,3]*l[2]/24 - c[4,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,4]*l[1]*l[2]/(18*l[0])],
          [c[0,0]*l[1]*l[2]/(18*l[0]) + c[0,4]*l[1]/12 - c[0,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[0,5]*l[2]/24 - c[0,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,0]*l[1]/12 - c[4,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,4]*l[0]*l[1]/(9*l[2]) + c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[4,5]*l[0]/12 + c[4,5]*l[1]*np.tan(g[0])/12 - c[4,5]*l[2]*np.tan(g[1])/24 + c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,0]*l[2]/24 - c[5,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[5,4]*l[0]/12 - c[5,4]*l[1]*np.tan(g[0])/12 - c[5,4]*l[2]*np.tan(g[1])/24 + c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,5]*l[0]*l[2]/(18*l[1]) - c[5,5]*l[2]*np.tan(g[0])/12 + c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]),
           c[0,1]*l[2]/24 - c[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[0,3]*l[1]/12 - c[0,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[0,5]*l[1]*l[2]/(18*l[0]) - c[4,1]*l[0]/12 + c[4,1]*l[1]*np.tan(g[0])/12 - c[4,1]*l[2]*np.tan(g[1])/24 + c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[4,3]*l[0]*l[1]/(9*l[2]) + c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[4,5]*l[1]/12 - c[4,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,1]*l[0]*l[2]/(18*l[1]) - c[5,1]*l[2]*np.tan(g[0])/12 + c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[5,3]*l[0]/12 - c[5,3]*l[1]*np.tan(g[0])/12 - c[5,3]*l[2]*np.tan(g[1])/24 + c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,5]*l[2]/24 - c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[0,2]*l[1]/12 - c[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[0,3]*l[2]/24 - c[0,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[0,4]*l[1]*l[2]/(18*l[0]) - c[4,2]*l[0]*l[1]/(9*l[2]) + c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[4,3]*l[0]/12 + c[4,3]*l[1]*np.tan(g[0])/12 - c[4,3]*l[2]*np.tan(g[1])/24 + c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[4,4]*l[1]/12 - c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,2]*l[0]/12 - c[5,2]*l[1]*np.tan(g[0])/12 - c[5,2]*l[2]*np.tan(g[1])/24 + c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,3]*l[0]*l[2]/(18*l[1]) - c[5,3]*l[2]*np.tan(g[0])/12 + c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[5,4]*l[2]/24 - c[5,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[0,0]*l[1]*l[2]/(18*l[0]) + c[0,4]*l[1]/12 + c[0,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[0,5]*l[2]/24 + c[0,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,0]*l[1]/12 + c[4,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,4]*l[0]*l[1]/(18*l[2]) - c[4,4]*l[1]*np.tan(g[1])/6 - c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[4,5]*l[0]/24 - c[4,5]*l[1]*np.tan(g[0])/12 - c[4,5]*l[2]*np.tan(g[1])/24 - c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,0]*l[2]/24 + c[5,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[5,4]*l[0]/24 - c[5,4]*l[1]*np.tan(g[0])/12 + c[5,4]*l[2]*np.tan(g[1])/24 - c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,5]*l[0]*l[2]/(36*l[1]) - c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]),
           c[0,1]*l[2]/24 + c[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[0,3]*l[1]/12 + c[0,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[0,5]*l[1]*l[2]/(18*l[0]) - c[4,1]*l[0]/24 - c[4,1]*l[1]*np.tan(g[0])/12 - c[4,1]*l[2]*np.tan(g[1])/24 - c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[4,3]*l[0]*l[1]/(18*l[2]) - c[4,3]*l[1]*np.tan(g[1])/6 - c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[4,5]*l[1]/12 + c[4,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,1]*l[0]*l[2]/(36*l[1]) - c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[5,3]*l[0]/24 - c[5,3]*l[1]*np.tan(g[0])/12 + c[5,3]*l[2]*np.tan(g[1])/24 - c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,5]*l[2]/24 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[0,2]*l[1]/12 + c[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[0,3]*l[2]/24 + c[0,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[0,4]*l[1]*l[2]/(18*l[0]) - c[4,2]*l[0]*l[1]/(18*l[2]) - c[4,2]*l[1]*np.tan(g[1])/6 - c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[4,3]*l[0]/24 - c[4,3]*l[1]*np.tan(g[0])/12 - c[4,3]*l[2]*np.tan(g[1])/24 - c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[4,4]*l[1]/12 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,2]*l[0]/24 - c[5,2]*l[1]*np.tan(g[0])/12 + c[5,2]*l[2]*np.tan(g[1])/24 - c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,3]*l[0]*l[2]/(36*l[1]) - c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[5,4]*l[2]/24 + c[5,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[0,0]*l[1]*l[2]/(36*l[0]) + c[0,4]*l[1]/24 + c[0,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[0,5]*l[2]/24 + c[0,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[4,0]*l[1]/24 + c[4,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[4,4]*l[0]*l[1]/(36*l[2]) - c[4,4]*l[1]*np.tan(g[1])/12 - c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[4,5]*l[0]/24 - c[4,5]*l[1]*np.tan(g[0])/24 + c[4,5]*l[2]*np.tan(g[1])/24 - c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,0]*l[2]/24 + c[5,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[5,4]*l[0]/24 - c[5,4]*l[1]*np.tan(g[0])/24 + c[5,4]*l[2]*np.tan(g[1])/24 - c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,5]*l[0]*l[2]/(36*l[1]) + c[5,5]*l[2]*np.tan(g[0])/12 - c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]),
           -c[0,1]*l[2]/24 + c[0,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[0,3]*l[1]/24 + c[0,3]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[0,5]*l[1]*l[2]/(36*l[0]) + c[4,1]*l[0]/24 - c[4,1]*l[1]*np.tan(g[0])/24 + c[4,1]*l[2]*np.tan(g[1])/24 - c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[4,3]*l[0]*l[1]/(36*l[2]) - c[4,3]*l[1]*np.tan(g[1])/12 - c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[4,5]*l[1]/24 + c[4,5]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,1]*l[0]*l[2]/(36*l[1]) + c[5,1]*l[2]*np.tan(g[0])/12 - c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[5,3]*l[0]/24 - c[5,3]*l[1]*np.tan(g[0])/24 + c[5,3]*l[2]*np.tan(g[1])/24 - c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,5]*l[2]/24 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           c[0,2]*l[1]/24 + c[0,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[0,3]*l[2]/24 + c[0,3]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[0,4]*l[1]*l[2]/(36*l[0]) - c[4,2]*l[0]*l[1]/(36*l[2]) - c[4,2]*l[1]*np.tan(g[1])/12 - c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[4,3]*l[0]/24 - c[4,3]*l[1]*np.tan(g[0])/24 + c[4,3]*l[2]*np.tan(g[1])/24 - c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[4,4]*l[1]/24 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[5,2]*l[0]/24 - c[5,2]*l[1]*np.tan(g[0])/24 + c[5,2]*l[2]*np.tan(g[1])/24 - c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,3]*l[0]*l[2]/(36*l[1]) + c[5,3]*l[2]*np.tan(g[0])/12 - c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[5,4]*l[2]/24 + c[5,4]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           c[0,0]*l[1]*l[2]/(36*l[0]) + c[0,4]*l[1]/24 - c[0,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[0,5]*l[2]/24 - c[0,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[4,0]*l[1]/24 - c[4,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[4,4]*l[0]*l[1]/(18*l[2]) + c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[4,5]*l[0]/12 + c[4,5]*l[1]*np.tan(g[0])/24 + c[4,5]*l[2]*np.tan(g[1])/24 + c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[5,0]*l[2]/24 - c[5,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[5,4]*l[0]/12 - c[5,4]*l[1]*np.tan(g[0])/24 - c[5,4]*l[2]*np.tan(g[1])/24 + c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,5]*l[0]*l[2]/(18*l[1]) + c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]),
           -c[0,1]*l[2]/24 - c[0,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[0,3]*l[1]/24 - c[0,3]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[0,5]*l[1]*l[2]/(36*l[0]) + c[4,1]*l[0]/12 + c[4,1]*l[1]*np.tan(g[0])/24 + c[4,1]*l[2]*np.tan(g[1])/24 + c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[4,3]*l[0]*l[1]/(18*l[2]) + c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[4,5]*l[1]/24 - c[4,5]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,1]*l[0]*l[2]/(18*l[1]) + c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[5,3]*l[0]/12 - c[5,3]*l[1]*np.tan(g[0])/24 - c[5,3]*l[2]*np.tan(g[1])/24 + c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[5,5]*l[2]/24 - c[5,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           c[0,2]*l[1]/24 - c[0,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[0,3]*l[2]/24 - c[0,3]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[0,4]*l[1]*l[2]/(36*l[0]) - c[4,2]*l[0]*l[1]/(18*l[2]) + c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[4,3]*l[0]/12 + c[4,3]*l[1]*np.tan(g[0])/24 + c[4,3]*l[2]*np.tan(g[1])/24 + c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[4,4]*l[1]/24 - c[4,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[5,2]*l[0]/12 - c[5,2]*l[1]*np.tan(g[0])/24 - c[5,2]*l[2]*np.tan(g[1])/24 + c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,3]*l[0]*l[2]/(18*l[1]) + c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[5,4]*l[2]/24 - c[5,4]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           c[0,0]*l[1]*l[2]/(9*l[0]) - c[0,4]*l[1]/12 - c[0,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[0,5]*l[2]/12 - c[0,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[4,0]*l[1]/12 - c[4,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[4,4]*l[0]*l[1]/(9*l[2]) + c[4,4]*l[1]*np.tan(g[1])/6 + c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[4,5]*l[0]/12 + c[4,5]*l[1]*np.tan(g[0])/12 - c[4,5]*l[2]*np.tan(g[1])/12 + c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[5,0]*l[2]/12 - c[5,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[5,4]*l[0]/12 + c[5,4]*l[1]*np.tan(g[0])/12 - c[5,4]*l[2]*np.tan(g[1])/12 + c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[5,5]*l[0]*l[2]/(9*l[1]) - c[5,5]*l[2]*np.tan(g[0])/6 + c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]),
           c[0,1]*l[2]/12 - c[0,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[0,3]*l[1]/12 - c[0,3]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[0,5]*l[1]*l[2]/(9*l[0]) - c[4,1]*l[0]/12 + c[4,1]*l[1]*np.tan(g[0])/12 - c[4,1]*l[2]*np.tan(g[1])/12 + c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[4,3]*l[0]*l[1]/(9*l[2]) + c[4,3]*l[1]*np.tan(g[1])/6 + c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[4,5]*l[1]/12 - c[4,5]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,1]*l[0]*l[2]/(9*l[1]) - c[5,1]*l[2]*np.tan(g[0])/6 + c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[5,3]*l[0]/12 + c[5,3]*l[1]*np.tan(g[0])/12 - c[5,3]*l[2]*np.tan(g[1])/12 + c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[5,5]*l[2]/12 - c[5,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           -c[0,2]*l[1]/12 - c[0,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[0,3]*l[2]/12 - c[0,3]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[0,4]*l[1]*l[2]/(9*l[0]) + c[4,2]*l[0]*l[1]/(9*l[2]) + c[4,2]*l[1]*np.tan(g[1])/6 + c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[4,3]*l[0]/12 + c[4,3]*l[1]*np.tan(g[0])/12 - c[4,3]*l[2]*np.tan(g[1])/12 + c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[4,4]*l[1]/12 - c[4,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[5,2]*l[0]/12 + c[5,2]*l[1]*np.tan(g[0])/12 - c[5,2]*l[2]*np.tan(g[1])/12 + c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[5,3]*l[0]*l[2]/(9*l[1]) - c[5,3]*l[2]*np.tan(g[0])/6 + c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[5,4]*l[2]/12 - c[5,4]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           -c[0,0]*l[1]*l[2]/(9*l[0]) - c[0,4]*l[1]/12 + c[0,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[0,5]*l[2]/12 + c[0,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[4,0]*l[1]/12 + c[4,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[4,4]*l[0]*l[1]/(18*l[2]) - c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[4,5]*l[0]/24 - c[4,5]*l[1]*np.tan(g[0])/12 - c[4,5]*l[2]*np.tan(g[1])/12 - c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[5,0]*l[2]/12 + c[5,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[5,4]*l[0]/24 + c[5,4]*l[1]*np.tan(g[0])/12 + c[5,4]*l[2]*np.tan(g[1])/12 - c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[5,5]*l[0]*l[2]/(18*l[1]) - c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]),
           c[0,1]*l[2]/12 + c[0,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[0,3]*l[1]/12 + c[0,3]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[0,5]*l[1]*l[2]/(9*l[0]) - c[4,1]*l[0]/24 - c[4,1]*l[1]*np.tan(g[0])/12 - c[4,1]*l[2]*np.tan(g[1])/12 - c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[4,3]*l[0]*l[1]/(18*l[2]) - c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[4,5]*l[1]/12 + c[4,5]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,1]*l[0]*l[2]/(18*l[1]) - c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[5,3]*l[0]/24 + c[5,3]*l[1]*np.tan(g[0])/12 + c[5,3]*l[2]*np.tan(g[1])/12 - c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[5,5]*l[2]/12 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           -c[0,2]*l[1]/12 + c[0,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[0,3]*l[2]/12 + c[0,3]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[0,4]*l[1]*l[2]/(9*l[0]) + c[4,2]*l[0]*l[1]/(18*l[2]) - c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[4,3]*l[0]/24 - c[4,3]*l[1]*np.tan(g[0])/12 - c[4,3]*l[2]*np.tan(g[1])/12 - c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[4,4]*l[1]/12 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[5,2]*l[0]/24 + c[5,2]*l[1]*np.tan(g[0])/12 + c[5,2]*l[2]*np.tan(g[1])/12 - c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[5,3]*l[0]*l[2]/(18*l[1]) - c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[5,4]*l[2]/12 + c[5,4]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           -c[0,0]*l[1]*l[2]/(18*l[0]) - c[0,4]*l[1]/24 + c[0,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[0,5]*l[2]/12 + c[0,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,0]*l[1]/24 + c[4,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,4]*l[0]*l[1]/(36*l[2]) - c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[4,5]*l[0]/24 - c[4,5]*l[1]*np.tan(g[0])/24 + c[4,5]*l[2]*np.tan(g[1])/12 - c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,0]*l[2]/12 + c[5,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[5,4]*l[0]/24 + c[5,4]*l[1]*np.tan(g[0])/24 + c[5,4]*l[2]*np.tan(g[1])/12 - c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,5]*l[0]*l[2]/(18*l[1]) + c[5,5]*l[2]*np.tan(g[0])/6 - c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]),
           -c[0,1]*l[2]/12 + c[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[0,3]*l[1]/24 + c[0,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[0,5]*l[1]*l[2]/(18*l[0]) + c[4,1]*l[0]/24 - c[4,1]*l[1]*np.tan(g[0])/24 + c[4,1]*l[2]*np.tan(g[1])/12 - c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[4,3]*l[0]*l[1]/(36*l[2]) - c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[4,5]*l[1]/24 + c[4,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,1]*l[0]*l[2]/(18*l[1]) + c[5,1]*l[2]*np.tan(g[0])/6 - c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[5,3]*l[0]/24 + c[5,3]*l[1]*np.tan(g[0])/24 + c[5,3]*l[2]*np.tan(g[1])/12 - c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,5]*l[2]/12 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[0,2]*l[1]/24 + c[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[0,3]*l[2]/12 + c[0,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[0,4]*l[1]*l[2]/(18*l[0]) + c[4,2]*l[0]*l[1]/(36*l[2]) - c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[4,3]*l[0]/24 - c[4,3]*l[1]*np.tan(g[0])/24 + c[4,3]*l[2]*np.tan(g[1])/12 - c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[4,4]*l[1]/24 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,2]*l[0]/24 + c[5,2]*l[1]*np.tan(g[0])/24 + c[5,2]*l[2]*np.tan(g[1])/12 - c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,3]*l[0]*l[2]/(18*l[1]) + c[5,3]*l[2]*np.tan(g[0])/6 - c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[5,4]*l[2]/12 + c[5,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[0,0]*l[1]*l[2]/(18*l[0]) - c[0,4]*l[1]/24 - c[0,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[0,5]*l[2]/12 - c[0,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,0]*l[1]/24 - c[4,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,4]*l[0]*l[1]/(18*l[2]) + c[4,4]*l[1]*np.tan(g[1])/12 + c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[4,5]*l[0]/12 + c[4,5]*l[1]*np.tan(g[0])/24 + c[4,5]*l[2]*np.tan(g[1])/12 + c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,0]*l[2]/12 - c[5,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[5,4]*l[0]/12 + c[5,4]*l[1]*np.tan(g[0])/24 - c[5,4]*l[2]*np.tan(g[1])/12 + c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,5]*l[0]*l[2]/(9*l[1]) + c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]),
           -c[0,1]*l[2]/12 - c[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[0,3]*l[1]/24 - c[0,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[0,5]*l[1]*l[2]/(18*l[0]) + c[4,1]*l[0]/12 + c[4,1]*l[1]*np.tan(g[0])/24 + c[4,1]*l[2]*np.tan(g[1])/12 + c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[4,3]*l[0]*l[1]/(18*l[2]) + c[4,3]*l[1]*np.tan(g[1])/12 + c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[4,5]*l[1]/24 - c[4,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,1]*l[0]*l[2]/(9*l[1]) + c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[5,3]*l[0]/12 + c[5,3]*l[1]*np.tan(g[0])/24 - c[5,3]*l[2]*np.tan(g[1])/12 + c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,5]*l[2]/12 - c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[0,2]*l[1]/24 - c[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[0,3]*l[2]/12 - c[0,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[0,4]*l[1]*l[2]/(18*l[0]) + c[4,2]*l[0]*l[1]/(18*l[2]) + c[4,2]*l[1]*np.tan(g[1])/12 + c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[4,3]*l[0]/12 + c[4,3]*l[1]*np.tan(g[0])/24 + c[4,3]*l[2]*np.tan(g[1])/12 + c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[4,4]*l[1]/24 - c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,2]*l[0]/12 + c[5,2]*l[1]*np.tan(g[0])/24 - c[5,2]*l[2]*np.tan(g[1])/12 + c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,3]*l[0]*l[2]/(9*l[1]) + c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[5,4]*l[2]/12 - c[5,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0])],
          [c[1,0]*l[2]/24 - c[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[1,4]*l[0]/12 - c[1,4]*l[1]*np.tan(g[0])/12 - c[1,4]*l[2]*np.tan(g[1])/24 + c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[1,5]*l[0]*l[2]/(18*l[1]) - c[1,5]*l[2]*np.tan(g[0])/12 + c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[3,0]*l[1]/12 - c[3,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[3,4]*l[0]*l[1]/(9*l[2]) + c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[3,5]*l[0]/12 + c[3,5]*l[1]*np.tan(g[0])/12 - c[3,5]*l[2]*np.tan(g[1])/24 + c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,0]*l[1]*l[2]/(18*l[0]) + c[5,4]*l[1]/12 - c[5,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,5]*l[2]/24 - c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[1,1]*l[0]*l[2]/(18*l[1]) - c[1,1]*l[2]*np.tan(g[0])/12 + c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[1,3]*l[0]/12 - c[1,3]*l[1]*np.tan(g[0])/12 - c[1,3]*l[2]*np.tan(g[1])/24 + c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[1,5]*l[2]/24 - c[1,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[3,1]*l[0]/12 + c[3,1]*l[1]*np.tan(g[0])/12 - c[3,1]*l[2]*np.tan(g[1])/24 + c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,3]*l[0]*l[1]/(9*l[2]) + c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[3,5]*l[1]/12 - c[3,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,1]*l[2]/24 - c[5,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[5,3]*l[1]/12 - c[5,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,5]*l[1]*l[2]/(18*l[0]),
           c[1,2]*l[0]/12 - c[1,2]*l[1]*np.tan(g[0])/12 - c[1,2]*l[2]*np.tan(g[1])/24 + c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[1,3]*l[0]*l[2]/(18*l[1]) - c[1,3]*l[2]*np.tan(g[0])/12 + c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[1,4]*l[2]/24 - c[1,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[3,2]*l[0]*l[1]/(9*l[2]) + c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[3,3]*l[0]/12 + c[3,3]*l[1]*np.tan(g[0])/12 - c[3,3]*l[2]*np.tan(g[1])/24 + c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,4]*l[1]/12 - c[3,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,2]*l[1]/12 - c[5,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,3]*l[2]/24 - c[5,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[5,4]*l[1]*l[2]/(18*l[0]),
           -c[1,0]*l[2]/24 + c[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[1,4]*l[0]/24 - c[1,4]*l[1]*np.tan(g[0])/12 + c[1,4]*l[2]*np.tan(g[1])/24 - c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[1,5]*l[0]*l[2]/(36*l[1]) - c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[3,0]*l[1]/12 + c[3,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[3,4]*l[0]*l[1]/(18*l[2]) - c[3,4]*l[1]*np.tan(g[1])/6 - c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[3,5]*l[0]/24 - c[3,5]*l[1]*np.tan(g[0])/12 - c[3,5]*l[2]*np.tan(g[1])/24 - c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,0]*l[1]*l[2]/(18*l[0]) + c[5,4]*l[1]/12 + c[5,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,5]*l[2]/24 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[1,1]*l[0]*l[2]/(36*l[1]) - c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[1,3]*l[0]/24 - c[1,3]*l[1]*np.tan(g[0])/12 + c[1,3]*l[2]*np.tan(g[1])/24 - c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[1,5]*l[2]/24 + c[1,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[3,1]*l[0]/24 - c[3,1]*l[1]*np.tan(g[0])/12 - c[3,1]*l[2]*np.tan(g[1])/24 - c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,3]*l[0]*l[1]/(18*l[2]) - c[3,3]*l[1]*np.tan(g[1])/6 - c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[3,5]*l[1]/12 + c[3,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,1]*l[2]/24 + c[5,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[5,3]*l[1]/12 + c[5,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,5]*l[1]*l[2]/(18*l[0]),
           c[1,2]*l[0]/24 - c[1,2]*l[1]*np.tan(g[0])/12 + c[1,2]*l[2]*np.tan(g[1])/24 - c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[1,3]*l[0]*l[2]/(36*l[1]) - c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[1,4]*l[2]/24 + c[1,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[3,2]*l[0]*l[1]/(18*l[2]) - c[3,2]*l[1]*np.tan(g[1])/6 - c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[3,3]*l[0]/24 - c[3,3]*l[1]*np.tan(g[0])/12 - c[3,3]*l[2]*np.tan(g[1])/24 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,4]*l[1]/12 + c[3,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,2]*l[1]/12 + c[5,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,3]*l[2]/24 + c[5,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[5,4]*l[1]*l[2]/(18*l[0]),
           -c[1,0]*l[2]/24 + c[1,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[1,4]*l[0]/24 - c[1,4]*l[1]*np.tan(g[0])/24 + c[1,4]*l[2]*np.tan(g[1])/24 - c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[1,5]*l[0]*l[2]/(36*l[1]) + c[1,5]*l[2]*np.tan(g[0])/12 - c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[3,0]*l[1]/24 + c[3,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[3,4]*l[0]*l[1]/(36*l[2]) - c[3,4]*l[1]*np.tan(g[1])/12 - c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[3,5]*l[0]/24 - c[3,5]*l[1]*np.tan(g[0])/24 + c[3,5]*l[2]*np.tan(g[1])/24 - c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,0]*l[1]*l[2]/(36*l[0]) + c[5,4]*l[1]/24 + c[5,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,5]*l[2]/24 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           -c[1,1]*l[0]*l[2]/(36*l[1]) + c[1,1]*l[2]*np.tan(g[0])/12 - c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[1,3]*l[0]/24 - c[1,3]*l[1]*np.tan(g[0])/24 + c[1,3]*l[2]*np.tan(g[1])/24 - c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[1,5]*l[2]/24 + c[1,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[3,1]*l[0]/24 - c[3,1]*l[1]*np.tan(g[0])/24 + c[3,1]*l[2]*np.tan(g[1])/24 - c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,3]*l[0]*l[1]/(36*l[2]) - c[3,3]*l[1]*np.tan(g[1])/12 - c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[3,5]*l[1]/24 + c[3,5]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,1]*l[2]/24 + c[5,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[5,3]*l[1]/24 + c[5,3]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,5]*l[1]*l[2]/(36*l[0]),
           c[1,2]*l[0]/24 - c[1,2]*l[1]*np.tan(g[0])/24 + c[1,2]*l[2]*np.tan(g[1])/24 - c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[1,3]*l[0]*l[2]/(36*l[1]) + c[1,3]*l[2]*np.tan(g[0])/12 - c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[1,4]*l[2]/24 + c[1,4]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[3,2]*l[0]*l[1]/(36*l[2]) - c[3,2]*l[1]*np.tan(g[1])/12 - c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[3,3]*l[0]/24 - c[3,3]*l[1]*np.tan(g[0])/24 + c[3,3]*l[2]*np.tan(g[1])/24 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[3,4]*l[1]/24 + c[3,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[5,2]*l[1]/24 + c[5,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,3]*l[2]/24 + c[5,3]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[5,4]*l[1]*l[2]/(36*l[0]),
           c[1,0]*l[2]/24 - c[1,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[1,4]*l[0]/12 - c[1,4]*l[1]*np.tan(g[0])/24 - c[1,4]*l[2]*np.tan(g[1])/24 + c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[1,5]*l[0]*l[2]/(18*l[1]) + c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[3,0]*l[1]/24 - c[3,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[3,4]*l[0]*l[1]/(18*l[2]) + c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[3,5]*l[0]/12 + c[3,5]*l[1]*np.tan(g[0])/24 + c[3,5]*l[2]*np.tan(g[1])/24 + c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[5,0]*l[1]*l[2]/(36*l[0]) + c[5,4]*l[1]/24 - c[5,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,5]*l[2]/24 - c[5,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           -c[1,1]*l[0]*l[2]/(18*l[1]) + c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[1,3]*l[0]/12 - c[1,3]*l[1]*np.tan(g[0])/24 - c[1,3]*l[2]*np.tan(g[1])/24 + c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[1,5]*l[2]/24 - c[1,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[3,1]*l[0]/12 + c[3,1]*l[1]*np.tan(g[0])/24 + c[3,1]*l[2]*np.tan(g[1])/24 + c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,3]*l[0]*l[1]/(18*l[2]) + c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[3,5]*l[1]/24 - c[3,5]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,1]*l[2]/24 - c[5,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[5,3]*l[1]/24 - c[5,3]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[5,5]*l[1]*l[2]/(36*l[0]),
           c[1,2]*l[0]/12 - c[1,2]*l[1]*np.tan(g[0])/24 - c[1,2]*l[2]*np.tan(g[1])/24 + c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[1,3]*l[0]*l[2]/(18*l[1]) + c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[1,4]*l[2]/24 - c[1,4]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[3,2]*l[0]*l[1]/(18*l[2]) + c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[3,3]*l[0]/12 + c[3,3]*l[1]*np.tan(g[0])/24 + c[3,3]*l[2]*np.tan(g[1])/24 + c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,4]*l[1]/24 - c[3,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[5,2]*l[1]/24 - c[5,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,3]*l[2]/24 - c[5,3]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[5,4]*l[1]*l[2]/(36*l[0]),
           c[1,0]*l[2]/12 - c[1,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[1,4]*l[0]/12 + c[1,4]*l[1]*np.tan(g[0])/12 - c[1,4]*l[2]*np.tan(g[1])/12 + c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[1,5]*l[0]*l[2]/(9*l[1]) - c[1,5]*l[2]*np.tan(g[0])/6 + c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[3,0]*l[1]/12 - c[3,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[3,4]*l[0]*l[1]/(9*l[2]) + c[3,4]*l[1]*np.tan(g[1])/6 + c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[3,5]*l[0]/12 + c[3,5]*l[1]*np.tan(g[0])/12 - c[3,5]*l[2]*np.tan(g[1])/12 + c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[5,0]*l[1]*l[2]/(9*l[0]) - c[5,4]*l[1]/12 - c[5,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,5]*l[2]/12 - c[5,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           c[1,1]*l[0]*l[2]/(9*l[1]) - c[1,1]*l[2]*np.tan(g[0])/6 + c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[1,3]*l[0]/12 + c[1,3]*l[1]*np.tan(g[0])/12 - c[1,3]*l[2]*np.tan(g[1])/12 + c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[1,5]*l[2]/12 - c[1,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[3,1]*l[0]/12 + c[3,1]*l[1]*np.tan(g[0])/12 - c[3,1]*l[2]*np.tan(g[1])/12 + c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,3]*l[0]*l[1]/(9*l[2]) + c[3,3]*l[1]*np.tan(g[1])/6 + c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[3,5]*l[1]/12 - c[3,5]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,1]*l[2]/12 - c[5,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[5,3]*l[1]/12 - c[5,3]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,5]*l[1]*l[2]/(9*l[0]),
           -c[1,2]*l[0]/12 + c[1,2]*l[1]*np.tan(g[0])/12 - c[1,2]*l[2]*np.tan(g[1])/12 + c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[1,3]*l[0]*l[2]/(9*l[1]) - c[1,3]*l[2]*np.tan(g[0])/6 + c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[1,4]*l[2]/12 - c[1,4]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[3,2]*l[0]*l[1]/(9*l[2]) + c[3,2]*l[1]*np.tan(g[1])/6 + c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[3,3]*l[0]/12 + c[3,3]*l[1]*np.tan(g[0])/12 - c[3,3]*l[2]*np.tan(g[1])/12 + c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[3,4]*l[1]/12 - c[3,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[5,2]*l[1]/12 - c[5,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,3]*l[2]/12 - c[5,3]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[5,4]*l[1]*l[2]/(9*l[0]),
           -c[1,0]*l[2]/12 + c[1,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[1,4]*l[0]/24 + c[1,4]*l[1]*np.tan(g[0])/12 + c[1,4]*l[2]*np.tan(g[1])/12 - c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[1,5]*l[0]*l[2]/(18*l[1]) - c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[3,0]*l[1]/12 + c[3,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[3,4]*l[0]*l[1]/(18*l[2]) - c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[3,5]*l[0]/24 - c[3,5]*l[1]*np.tan(g[0])/12 - c[3,5]*l[2]*np.tan(g[1])/12 - c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[5,0]*l[1]*l[2]/(9*l[0]) - c[5,4]*l[1]/12 + c[5,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,5]*l[2]/12 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           c[1,1]*l[0]*l[2]/(18*l[1]) - c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[1,3]*l[0]/24 + c[1,3]*l[1]*np.tan(g[0])/12 + c[1,3]*l[2]*np.tan(g[1])/12 - c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[1,5]*l[2]/12 + c[1,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[3,1]*l[0]/24 - c[3,1]*l[1]*np.tan(g[0])/12 - c[3,1]*l[2]*np.tan(g[1])/12 - c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,3]*l[0]*l[1]/(18*l[2]) - c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[3,5]*l[1]/12 + c[3,5]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,1]*l[2]/12 + c[5,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[5,3]*l[1]/12 + c[5,3]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[5,5]*l[1]*l[2]/(9*l[0]),
           -c[1,2]*l[0]/24 + c[1,2]*l[1]*np.tan(g[0])/12 + c[1,2]*l[2]*np.tan(g[1])/12 - c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[1,3]*l[0]*l[2]/(18*l[1]) - c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[1,4]*l[2]/12 + c[1,4]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[3,2]*l[0]*l[1]/(18*l[2]) - c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[3,3]*l[0]/24 - c[3,3]*l[1]*np.tan(g[0])/12 - c[3,3]*l[2]*np.tan(g[1])/12 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,4]*l[1]/12 + c[3,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[5,2]*l[1]/12 + c[5,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,3]*l[2]/12 + c[5,3]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[5,4]*l[1]*l[2]/(9*l[0]),
           -c[1,0]*l[2]/12 + c[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[1,4]*l[0]/24 + c[1,4]*l[1]*np.tan(g[0])/24 + c[1,4]*l[2]*np.tan(g[1])/12 - c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[1,5]*l[0]*l[2]/(18*l[1]) + c[1,5]*l[2]*np.tan(g[0])/6 - c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[3,0]*l[1]/24 + c[3,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[3,4]*l[0]*l[1]/(36*l[2]) - c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[3,5]*l[0]/24 - c[3,5]*l[1]*np.tan(g[0])/24 + c[3,5]*l[2]*np.tan(g[1])/12 - c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,0]*l[1]*l[2]/(18*l[0]) - c[5,4]*l[1]/24 + c[5,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,5]*l[2]/12 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[1,1]*l[0]*l[2]/(18*l[1]) + c[1,1]*l[2]*np.tan(g[0])/6 - c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[1,3]*l[0]/24 + c[1,3]*l[1]*np.tan(g[0])/24 + c[1,3]*l[2]*np.tan(g[1])/12 - c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[1,5]*l[2]/12 + c[1,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[3,1]*l[0]/24 - c[3,1]*l[1]*np.tan(g[0])/24 + c[3,1]*l[2]*np.tan(g[1])/12 - c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,3]*l[0]*l[1]/(36*l[2]) - c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[3,5]*l[1]/24 + c[3,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,1]*l[2]/12 + c[5,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[5,3]*l[1]/24 + c[5,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,5]*l[1]*l[2]/(18*l[0]),
           -c[1,2]*l[0]/24 + c[1,2]*l[1]*np.tan(g[0])/24 + c[1,2]*l[2]*np.tan(g[1])/12 - c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[1,3]*l[0]*l[2]/(18*l[1]) + c[1,3]*l[2]*np.tan(g[0])/6 - c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[1,4]*l[2]/12 + c[1,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[3,2]*l[0]*l[1]/(36*l[2]) - c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[3,3]*l[0]/24 - c[3,3]*l[1]*np.tan(g[0])/24 + c[3,3]*l[2]*np.tan(g[1])/12 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,4]*l[1]/24 + c[3,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,2]*l[1]/24 + c[5,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,3]*l[2]/12 + c[5,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[5,4]*l[1]*l[2]/(18*l[0]),
           c[1,0]*l[2]/12 - c[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[1,4]*l[0]/12 + c[1,4]*l[1]*np.tan(g[0])/24 - c[1,4]*l[2]*np.tan(g[1])/12 + c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[1,5]*l[0]*l[2]/(9*l[1]) + c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[3,0]*l[1]/24 - c[3,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[3,4]*l[0]*l[1]/(18*l[2]) + c[3,4]*l[1]*np.tan(g[1])/12 + c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[3,5]*l[0]/12 + c[3,5]*l[1]*np.tan(g[0])/24 + c[3,5]*l[2]*np.tan(g[1])/12 + c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,0]*l[1]*l[2]/(18*l[0]) - c[5,4]*l[1]/24 - c[5,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,5]*l[2]/12 - c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[1,1]*l[0]*l[2]/(9*l[1]) + c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[1,3]*l[0]/12 + c[1,3]*l[1]*np.tan(g[0])/24 - c[1,3]*l[2]*np.tan(g[1])/12 + c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[1,5]*l[2]/12 - c[1,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[3,1]*l[0]/12 + c[3,1]*l[1]*np.tan(g[0])/24 + c[3,1]*l[2]*np.tan(g[1])/12 + c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,3]*l[0]*l[1]/(18*l[2]) + c[3,3]*l[1]*np.tan(g[1])/12 + c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[3,5]*l[1]/24 - c[3,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,1]*l[2]/12 - c[5,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[5,3]*l[1]/24 - c[5,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,5]*l[1]*l[2]/(18*l[0]),
           -c[1,2]*l[0]/12 + c[1,2]*l[1]*np.tan(g[0])/24 - c[1,2]*l[2]*np.tan(g[1])/12 + c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[1,3]*l[0]*l[2]/(9*l[1]) + c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[1,4]*l[2]/12 - c[1,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[3,2]*l[0]*l[1]/(18*l[2]) + c[3,2]*l[1]*np.tan(g[1])/12 + c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[3,3]*l[0]/12 + c[3,3]*l[1]*np.tan(g[0])/24 + c[3,3]*l[2]*np.tan(g[1])/12 + c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,4]*l[1]/24 - c[3,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,2]*l[1]/24 - c[5,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,3]*l[2]/12 - c[5,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[5,4]*l[1]*l[2]/(18*l[0])],
          [-c[2,0]*l[1]/12 - c[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[2,4]*l[0]*l[1]/(9*l[2]) + c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[2,5]*l[0]/12 + c[2,5]*l[1]*np.tan(g[0])/12 - c[2,5]*l[2]*np.tan(g[1])/24 + c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,0]*l[2]/24 - c[3,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[3,4]*l[0]/12 - c[3,4]*l[1]*np.tan(g[0])/12 - c[3,4]*l[2]*np.tan(g[1])/24 + c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,5]*l[0]*l[2]/(18*l[1]) - c[3,5]*l[2]*np.tan(g[0])/12 + c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[4,0]*l[1]*l[2]/(18*l[0]) + c[4,4]*l[1]/12 - c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,5]*l[2]/24 - c[4,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[2,1]*l[0]/12 + c[2,1]*l[1]*np.tan(g[0])/12 - c[2,1]*l[2]*np.tan(g[1])/24 + c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[2,3]*l[0]*l[1]/(9*l[2]) + c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[2,5]*l[1]/12 - c[2,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[3,1]*l[0]*l[2]/(18*l[1]) - c[3,1]*l[2]*np.tan(g[0])/12 + c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[3,3]*l[0]/12 - c[3,3]*l[1]*np.tan(g[0])/12 - c[3,3]*l[2]*np.tan(g[1])/24 + c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,5]*l[2]/24 - c[3,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,1]*l[2]/24 - c[4,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,3]*l[1]/12 - c[4,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,5]*l[1]*l[2]/(18*l[0]),
           -c[2,2]*l[0]*l[1]/(9*l[2]) + c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[2,3]*l[0]/12 + c[2,3]*l[1]*np.tan(g[0])/12 - c[2,3]*l[2]*np.tan(g[1])/24 + c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[2,4]*l[1]/12 - c[2,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[3,2]*l[0]/12 - c[3,2]*l[1]*np.tan(g[0])/12 - c[3,2]*l[2]*np.tan(g[1])/24 + c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,3]*l[0]*l[2]/(18*l[1]) - c[3,3]*l[2]*np.tan(g[0])/12 + c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[3,4]*l[2]/24 - c[3,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,2]*l[1]/12 - c[4,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,3]*l[2]/24 - c[4,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,4]*l[1]*l[2]/(18*l[0]),
           c[2,0]*l[1]/12 + c[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[2,4]*l[0]*l[1]/(18*l[2]) - c[2,4]*l[1]*np.tan(g[1])/6 - c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[2,5]*l[0]/24 - c[2,5]*l[1]*np.tan(g[0])/12 - c[2,5]*l[2]*np.tan(g[1])/24 - c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,0]*l[2]/24 + c[3,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[3,4]*l[0]/24 - c[3,4]*l[1]*np.tan(g[0])/12 + c[3,4]*l[2]*np.tan(g[1])/24 - c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,5]*l[0]*l[2]/(36*l[1]) - c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[4,0]*l[1]*l[2]/(18*l[0]) + c[4,4]*l[1]/12 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,5]*l[2]/24 + c[4,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[2,1]*l[0]/24 - c[2,1]*l[1]*np.tan(g[0])/12 - c[2,1]*l[2]*np.tan(g[1])/24 - c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[2,3]*l[0]*l[1]/(18*l[2]) - c[2,3]*l[1]*np.tan(g[1])/6 - c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[2,5]*l[1]/12 + c[2,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[3,1]*l[0]*l[2]/(36*l[1]) - c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[3,3]*l[0]/24 - c[3,3]*l[1]*np.tan(g[0])/12 + c[3,3]*l[2]*np.tan(g[1])/24 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,5]*l[2]/24 + c[3,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,1]*l[2]/24 + c[4,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,3]*l[1]/12 + c[4,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,5]*l[1]*l[2]/(18*l[0]),
           -c[2,2]*l[0]*l[1]/(18*l[2]) - c[2,2]*l[1]*np.tan(g[1])/6 - c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[2,3]*l[0]/24 - c[2,3]*l[1]*np.tan(g[0])/12 - c[2,3]*l[2]*np.tan(g[1])/24 - c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[2,4]*l[1]/12 + c[2,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[3,2]*l[0]/24 - c[3,2]*l[1]*np.tan(g[0])/12 + c[3,2]*l[2]*np.tan(g[1])/24 - c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,3]*l[0]*l[2]/(36*l[1]) - c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[3,4]*l[2]/24 + c[3,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,2]*l[1]/12 + c[4,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,3]*l[2]/24 + c[4,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,4]*l[1]*l[2]/(18*l[0]),
           c[2,0]*l[1]/24 + c[2,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[2,4]*l[0]*l[1]/(36*l[2]) - c[2,4]*l[1]*np.tan(g[1])/12 - c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[2,5]*l[0]/24 - c[2,5]*l[1]*np.tan(g[0])/24 + c[2,5]*l[2]*np.tan(g[1])/24 - c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,0]*l[2]/24 + c[3,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[3,4]*l[0]/24 - c[3,4]*l[1]*np.tan(g[0])/24 + c[3,4]*l[2]*np.tan(g[1])/24 - c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,5]*l[0]*l[2]/(36*l[1]) + c[3,5]*l[2]*np.tan(g[0])/12 - c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[4,0]*l[1]*l[2]/(36*l[0]) + c[4,4]*l[1]/24 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[4,5]*l[2]/24 + c[4,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           c[2,1]*l[0]/24 - c[2,1]*l[1]*np.tan(g[0])/24 + c[2,1]*l[2]*np.tan(g[1])/24 - c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[2,3]*l[0]*l[1]/(36*l[2]) - c[2,3]*l[1]*np.tan(g[1])/12 - c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[2,5]*l[1]/24 + c[2,5]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[3,1]*l[0]*l[2]/(36*l[1]) + c[3,1]*l[2]*np.tan(g[0])/12 - c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[3,3]*l[0]/24 - c[3,3]*l[1]*np.tan(g[0])/24 + c[3,3]*l[2]*np.tan(g[1])/24 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,5]*l[2]/24 + c[3,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[4,1]*l[2]/24 + c[4,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[4,3]*l[1]/24 + c[4,3]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[4,5]*l[1]*l[2]/(36*l[0]),
           -c[2,2]*l[0]*l[1]/(36*l[2]) - c[2,2]*l[1]*np.tan(g[1])/12 - c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[2,3]*l[0]/24 - c[2,3]*l[1]*np.tan(g[0])/24 + c[2,3]*l[2]*np.tan(g[1])/24 - c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[2,4]*l[1]/24 + c[2,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[3,2]*l[0]/24 - c[3,2]*l[1]*np.tan(g[0])/24 + c[3,2]*l[2]*np.tan(g[1])/24 - c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,3]*l[0]*l[2]/(36*l[1]) + c[3,3]*l[2]*np.tan(g[0])/12 - c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[3,4]*l[2]/24 + c[3,4]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[4,2]*l[1]/24 + c[4,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[4,3]*l[2]/24 + c[4,3]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[4,4]*l[1]*l[2]/(36*l[0]),
           -c[2,0]*l[1]/24 - c[2,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[2,4]*l[0]*l[1]/(18*l[2]) + c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[2,5]*l[0]/12 + c[2,5]*l[1]*np.tan(g[0])/24 + c[2,5]*l[2]*np.tan(g[1])/24 + c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[3,0]*l[2]/24 - c[3,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[3,4]*l[0]/12 - c[3,4]*l[1]*np.tan(g[0])/24 - c[3,4]*l[2]*np.tan(g[1])/24 + c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,5]*l[0]*l[2]/(18*l[1]) + c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[4,0]*l[1]*l[2]/(36*l[0]) + c[4,4]*l[1]/24 - c[4,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[4,5]*l[2]/24 - c[4,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           c[2,1]*l[0]/12 + c[2,1]*l[1]*np.tan(g[0])/24 + c[2,1]*l[2]*np.tan(g[1])/24 + c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[2,3]*l[0]*l[1]/(18*l[2]) + c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[2,5]*l[1]/24 - c[2,5]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[3,1]*l[0]*l[2]/(18*l[1]) + c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[3,3]*l[0]/12 - c[3,3]*l[1]*np.tan(g[0])/24 - c[3,3]*l[2]*np.tan(g[1])/24 + c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[3,5]*l[2]/24 - c[3,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[4,1]*l[2]/24 - c[4,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[4,3]*l[1]/24 - c[4,3]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[4,5]*l[1]*l[2]/(36*l[0]),
           -c[2,2]*l[0]*l[1]/(18*l[2]) + c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[2,3]*l[0]/12 + c[2,3]*l[1]*np.tan(g[0])/24 + c[2,3]*l[2]*np.tan(g[1])/24 + c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[2,4]*l[1]/24 - c[2,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[3,2]*l[0]/12 - c[3,2]*l[1]*np.tan(g[0])/24 - c[3,2]*l[2]*np.tan(g[1])/24 + c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,3]*l[0]*l[2]/(18*l[1]) + c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[3,4]*l[2]/24 - c[3,4]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[4,2]*l[1]/24 - c[4,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[4,3]*l[2]/24 - c[4,3]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[4,4]*l[1]*l[2]/(36*l[0]),
           -c[2,0]*l[1]/12 - c[2,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[2,4]*l[0]*l[1]/(9*l[2]) + c[2,4]*l[1]*np.tan(g[1])/6 + c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[2,5]*l[0]/12 + c[2,5]*l[1]*np.tan(g[0])/12 - c[2,5]*l[2]*np.tan(g[1])/12 + c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,0]*l[2]/12 - c[3,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[3,4]*l[0]/12 + c[3,4]*l[1]*np.tan(g[0])/12 - c[3,4]*l[2]*np.tan(g[1])/12 + c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,5]*l[0]*l[2]/(9*l[1]) - c[3,5]*l[2]*np.tan(g[0])/6 + c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[4,0]*l[1]*l[2]/(9*l[0]) - c[4,4]*l[1]/12 - c[4,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[4,5]*l[2]/12 - c[4,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           -c[2,1]*l[0]/12 + c[2,1]*l[1]*np.tan(g[0])/12 - c[2,1]*l[2]*np.tan(g[1])/12 + c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[2,3]*l[0]*l[1]/(9*l[2]) + c[2,3]*l[1]*np.tan(g[1])/6 + c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[2,5]*l[1]/12 - c[2,5]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[3,1]*l[0]*l[2]/(9*l[1]) - c[3,1]*l[2]*np.tan(g[0])/6 + c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[3,3]*l[0]/12 + c[3,3]*l[1]*np.tan(g[0])/12 - c[3,3]*l[2]*np.tan(g[1])/12 + c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,5]*l[2]/12 - c[3,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[4,1]*l[2]/12 - c[4,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[4,3]*l[1]/12 - c[4,3]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[4,5]*l[1]*l[2]/(9*l[0]),
           c[2,2]*l[0]*l[1]/(9*l[2]) + c[2,2]*l[1]*np.tan(g[1])/6 + c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[2,3]*l[0]/12 + c[2,3]*l[1]*np.tan(g[0])/12 - c[2,3]*l[2]*np.tan(g[1])/12 + c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[2,4]*l[1]/12 - c[2,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[3,2]*l[0]/12 + c[3,2]*l[1]*np.tan(g[0])/12 - c[3,2]*l[2]*np.tan(g[1])/12 + c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,3]*l[0]*l[2]/(9*l[1]) - c[3,3]*l[2]*np.tan(g[0])/6 + c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[3,4]*l[2]/12 - c[3,4]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[4,2]*l[1]/12 - c[4,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[4,3]*l[2]/12 - c[4,3]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[4,4]*l[1]*l[2]/(9*l[0]),
           c[2,0]*l[1]/12 + c[2,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[2,4]*l[0]*l[1]/(18*l[2]) - c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[2,5]*l[0]/24 - c[2,5]*l[1]*np.tan(g[0])/12 - c[2,5]*l[2]*np.tan(g[1])/12 - c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[3,0]*l[2]/12 + c[3,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[3,4]*l[0]/24 + c[3,4]*l[1]*np.tan(g[0])/12 + c[3,4]*l[2]*np.tan(g[1])/12 - c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,5]*l[0]*l[2]/(18*l[1]) - c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[4,0]*l[1]*l[2]/(9*l[0]) - c[4,4]*l[1]/12 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[4,5]*l[2]/12 + c[4,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           -c[2,1]*l[0]/24 - c[2,1]*l[1]*np.tan(g[0])/12 - c[2,1]*l[2]*np.tan(g[1])/12 - c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[2,3]*l[0]*l[1]/(18*l[2]) - c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[2,5]*l[1]/12 + c[2,5]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[3,1]*l[0]*l[2]/(18*l[1]) - c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[3,3]*l[0]/24 + c[3,3]*l[1]*np.tan(g[0])/12 + c[3,3]*l[2]*np.tan(g[1])/12 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[3,5]*l[2]/12 + c[3,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[4,1]*l[2]/12 + c[4,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[4,3]*l[1]/12 + c[4,3]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[4,5]*l[1]*l[2]/(9*l[0]),
           c[2,2]*l[0]*l[1]/(18*l[2]) - c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[2,3]*l[0]/24 - c[2,3]*l[1]*np.tan(g[0])/12 - c[2,3]*l[2]*np.tan(g[1])/12 - c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[2,4]*l[1]/12 + c[2,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[3,2]*l[0]/24 + c[3,2]*l[1]*np.tan(g[0])/12 + c[3,2]*l[2]*np.tan(g[1])/12 - c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,3]*l[0]*l[2]/(18*l[1]) - c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[3,4]*l[2]/12 + c[3,4]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[4,2]*l[1]/12 + c[4,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[4,3]*l[2]/12 + c[4,3]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[4,4]*l[1]*l[2]/(9*l[0]),
           c[2,0]*l[1]/24 + c[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[2,4]*l[0]*l[1]/(36*l[2]) - c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[2,5]*l[0]/24 - c[2,5]*l[1]*np.tan(g[0])/24 + c[2,5]*l[2]*np.tan(g[1])/12 - c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,0]*l[2]/12 + c[3,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[3,4]*l[0]/24 + c[3,4]*l[1]*np.tan(g[0])/24 + c[3,4]*l[2]*np.tan(g[1])/12 - c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,5]*l[0]*l[2]/(18*l[1]) + c[3,5]*l[2]*np.tan(g[0])/6 - c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[4,0]*l[1]*l[2]/(18*l[0]) - c[4,4]*l[1]/24 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,5]*l[2]/12 + c[4,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[2,1]*l[0]/24 - c[2,1]*l[1]*np.tan(g[0])/24 + c[2,1]*l[2]*np.tan(g[1])/12 - c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[2,3]*l[0]*l[1]/(36*l[2]) - c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[2,5]*l[1]/24 + c[2,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[3,1]*l[0]*l[2]/(18*l[1]) + c[3,1]*l[2]*np.tan(g[0])/6 - c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[3,3]*l[0]/24 + c[3,3]*l[1]*np.tan(g[0])/24 + c[3,3]*l[2]*np.tan(g[1])/12 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,5]*l[2]/12 + c[3,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,1]*l[2]/12 + c[4,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,3]*l[1]/24 + c[4,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,5]*l[1]*l[2]/(18*l[0]),
           c[2,2]*l[0]*l[1]/(36*l[2]) - c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[2,3]*l[0]/24 - c[2,3]*l[1]*np.tan(g[0])/24 + c[2,3]*l[2]*np.tan(g[1])/12 - c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[2,4]*l[1]/24 + c[2,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[3,2]*l[0]/24 + c[3,2]*l[1]*np.tan(g[0])/24 + c[3,2]*l[2]*np.tan(g[1])/12 - c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,3]*l[0]*l[2]/(18*l[1]) + c[3,3]*l[2]*np.tan(g[0])/6 - c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[3,4]*l[2]/12 + c[3,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,2]*l[1]/24 + c[4,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,3]*l[2]/12 + c[4,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,4]*l[1]*l[2]/(18*l[0]),
           -c[2,0]*l[1]/24 - c[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[2,4]*l[0]*l[1]/(18*l[2]) + c[2,4]*l[1]*np.tan(g[1])/12 + c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[2,5]*l[0]/12 + c[2,5]*l[1]*np.tan(g[0])/24 + c[2,5]*l[2]*np.tan(g[1])/12 + c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,0]*l[2]/12 - c[3,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[3,4]*l[0]/12 + c[3,4]*l[1]*np.tan(g[0])/24 - c[3,4]*l[2]*np.tan(g[1])/12 + c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,5]*l[0]*l[2]/(9*l[1]) + c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[4,0]*l[1]*l[2]/(18*l[0]) - c[4,4]*l[1]/24 - c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,5]*l[2]/12 - c[4,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[2,1]*l[0]/12 + c[2,1]*l[1]*np.tan(g[0])/24 + c[2,1]*l[2]*np.tan(g[1])/12 + c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[2,3]*l[0]*l[1]/(18*l[2]) + c[2,3]*l[1]*np.tan(g[1])/12 + c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[2,5]*l[1]/24 - c[2,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[3,1]*l[0]*l[2]/(9*l[1]) + c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[3,3]*l[0]/12 + c[3,3]*l[1]*np.tan(g[0])/24 - c[3,3]*l[2]*np.tan(g[1])/12 + c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,5]*l[2]/12 - c[3,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,1]*l[2]/12 - c[4,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,3]*l[1]/24 - c[4,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,5]*l[1]*l[2]/(18*l[0]),
           c[2,2]*l[0]*l[1]/(18*l[2]) + c[2,2]*l[1]*np.tan(g[1])/12 + c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[2,3]*l[0]/12 + c[2,3]*l[1]*np.tan(g[0])/24 + c[2,3]*l[2]*np.tan(g[1])/12 + c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[2,4]*l[1]/24 - c[2,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[3,2]*l[0]/12 + c[3,2]*l[1]*np.tan(g[0])/24 - c[3,2]*l[2]*np.tan(g[1])/12 + c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,3]*l[0]*l[2]/(9*l[1]) + c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[3,4]*l[2]/12 - c[3,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,2]*l[1]/24 - c[4,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,3]*l[2]/12 - c[4,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,4]*l[1]*l[2]/(18*l[0])],
          [-c[0,0]*l[1]*l[2]/(18*l[0]) - c[0,4]*l[1]/12 + c[0,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[0,5]*l[2]/24 + c[0,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,0]*l[1]/12 + c[4,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,4]*l[0]*l[1]/(18*l[2]) + c[4,4]*l[1]*np.tan(g[1])/6 - c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[4,5]*l[0]/24 + c[4,5]*l[1]*np.tan(g[0])/12 + c[4,5]*l[2]*np.tan(g[1])/24 - c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,0]*l[2]/24 + c[5,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[5,4]*l[0]/24 + c[5,4]*l[1]*np.tan(g[0])/12 - c[5,4]*l[2]*np.tan(g[1])/24 - c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,5]*l[0]*l[2]/(36*l[1]) - c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]),
           -c[0,1]*l[2]/24 + c[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[0,3]*l[1]/12 + c[0,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[0,5]*l[1]*l[2]/(18*l[0]) - c[4,1]*l[0]/24 + c[4,1]*l[1]*np.tan(g[0])/12 + c[4,1]*l[2]*np.tan(g[1])/24 - c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[4,3]*l[0]*l[1]/(18*l[2]) + c[4,3]*l[1]*np.tan(g[1])/6 - c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[4,5]*l[1]/12 + c[4,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,1]*l[0]*l[2]/(36*l[1]) - c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[5,3]*l[0]/24 + c[5,3]*l[1]*np.tan(g[0])/12 - c[5,3]*l[2]*np.tan(g[1])/24 - c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,5]*l[2]/24 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[0,2]*l[1]/12 + c[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[0,3]*l[2]/24 + c[0,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[0,4]*l[1]*l[2]/(18*l[0]) - c[4,2]*l[0]*l[1]/(18*l[2]) + c[4,2]*l[1]*np.tan(g[1])/6 - c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[4,3]*l[0]/24 + c[4,3]*l[1]*np.tan(g[0])/12 + c[4,3]*l[2]*np.tan(g[1])/24 - c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[4,4]*l[1]/12 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,2]*l[0]/24 + c[5,2]*l[1]*np.tan(g[0])/12 - c[5,2]*l[2]*np.tan(g[1])/24 - c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,3]*l[0]*l[2]/(36*l[1]) - c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[5,4]*l[2]/24 + c[5,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           (2*l[0]**2*(-4*c[4,4]*l[1]**2 - 3*c[4,5]*l[1]*l[2] + 3*c[5,4]*l[1]*l[2] + 2*c[5,5]*l[2]**2) + 3*l[0]*l[1]*l[2]*(-2*c[0,4]*l[1] - c[0,5]*l[2] + 2*c[4,0]*l[1] - 2*c[4,5]*l[1]*np.tan(g[0]) + c[4,5]*l[2]*np.tan(g[1]) - c[5,0]*l[2] + 2*c[5,4]*l[1]*np.tan(g[0]) + c[5,4]*l[2]*np.tan(g[1]) + 2*c[5,5]*l[2]*np.tan(g[0])) + 4*l[1]**2*l[2]**2*(c[0,0] - c[0,4]*np.tan(g[1]) - c[0,5]*np.tan(g[0]) - c[4,0]*np.tan(g[1]) + c[4,4]*np.tan(g[1])**2 + c[4,5]*np.tan(g[0])*np.tan(g[1]) - c[5,0]*np.tan(g[0]) + c[5,4]*np.tan(g[0])*np.tan(g[1]) + c[5,5]*np.tan(g[0])**2))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(-3*c[4,1]*l[1]*l[2] - 4*c[4,3]*l[1]**2 + 2*c[5,1]*l[2]**2 + 3*c[5,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(-c[0,1]*l[2] - 2*c[0,3]*l[1] - 2*c[4,1]*l[1]*np.tan(g[0]) + c[4,1]*l[2]*np.tan(g[1]) + 2*c[4,5]*l[1] + 2*c[5,1]*l[2]*np.tan(g[0]) + 2*c[5,3]*l[1]*np.tan(g[0]) + c[5,3]*l[2]*np.tan(g[1]) - c[5,5]*l[2]) + 4*l[1]**2*l[2]**2*(-c[0,1]*np.tan(g[0]) - c[0,3]*np.tan(g[1]) + c[0,5] + c[4,1]*np.tan(g[0])*np.tan(g[1]) + c[4,3]*np.tan(g[1])**2 - c[4,5]*np.tan(g[1]) + c[5,1]*np.tan(g[0])**2 + c[5,3]*np.tan(g[0])*np.tan(g[1]) - c[5,5]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(-4*c[4,2]*l[1]**2 - 3*c[4,3]*l[1]*l[2] + 3*c[5,2]*l[1]*l[2] + 2*c[5,3]*l[2]**2) + 3*l[0]*l[1]*l[2]*(-2*c[0,2]*l[1] - c[0,3]*l[2] - 2*c[4,3]*l[1]*np.tan(g[0]) + c[4,3]*l[2]*np.tan(g[1]) + 2*c[4,4]*l[1] + 2*c[5,2]*l[1]*np.tan(g[0]) + c[5,2]*l[2]*np.tan(g[1]) + 2*c[5,3]*l[2]*np.tan(g[0]) - c[5,4]*l[2]) + 4*l[1]**2*l[2]**2*(-c[0,2]*np.tan(g[1]) - c[0,3]*np.tan(g[0]) + c[0,4] + c[4,2]*np.tan(g[1])**2 + c[4,3]*np.tan(g[0])*np.tan(g[1]) - c[4,4]*np.tan(g[1]) + c[5,2]*np.tan(g[0])*np.tan(g[1]) + c[5,3]*np.tan(g[0])**2 - c[5,4]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(-2*c[4,4]*l[1]**2 + 3*c[4,5]*l[1]*l[2] + 3*c[5,4]*l[1]*l[2] - 2*c[5,5]*l[2]**2) + 3*l[0]*l[1]*l[2]*(-c[0,4]*l[1] + c[0,5]*l[2] + c[4,0]*l[1] - c[4,5]*l[1]*np.tan(g[0]) - c[4,5]*l[2]*np.tan(g[1]) - c[5,0]*l[2] + c[5,4]*l[1]*np.tan(g[0]) + c[5,4]*l[2]*np.tan(g[1])) + 2*l[1]**2*l[2]**2*(c[0,0] - c[0,4]*np.tan(g[1]) - c[0,5]*np.tan(g[0]) - c[4,0]*np.tan(g[1]) + c[4,4]*np.tan(g[1])**2 + c[4,5]*np.tan(g[0])*np.tan(g[1]) - c[5,0]*np.tan(g[0]) + c[5,4]*np.tan(g[0])*np.tan(g[1]) + c[5,5]*np.tan(g[0])**2))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(3*c[4,1]*l[1]*l[2] - 2*c[4,3]*l[1]**2 - 2*c[5,1]*l[2]**2 + 3*c[5,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(c[0,1]*l[2] - c[0,3]*l[1] - c[4,1]*l[1]*np.tan(g[0]) - c[4,1]*l[2]*np.tan(g[1]) + c[4,5]*l[1] + c[5,3]*l[1]*np.tan(g[0]) + c[5,3]*l[2]*np.tan(g[1]) - c[5,5]*l[2]) + 2*l[1]**2*l[2]**2*(-c[0,1]*np.tan(g[0]) - c[0,3]*np.tan(g[1]) + c[0,5] + c[4,1]*np.tan(g[0])*np.tan(g[1]) + c[4,3]*np.tan(g[1])**2 - c[4,5]*np.tan(g[1]) + c[5,1]*np.tan(g[0])**2 + c[5,3]*np.tan(g[0])*np.tan(g[1]) - c[5,5]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(-2*c[4,2]*l[1]**2 + 3*c[4,3]*l[1]*l[2] + 3*c[5,2]*l[1]*l[2] - 2*c[5,3]*l[2]**2) + 3*l[0]*l[1]*l[2]*(-c[0,2]*l[1] + c[0,3]*l[2] - c[4,3]*l[1]*np.tan(g[0]) - c[4,3]*l[2]*np.tan(g[1]) + c[4,4]*l[1] + c[5,2]*l[1]*np.tan(g[0]) + c[5,2]*l[2]*np.tan(g[1]) - c[5,4]*l[2]) + 2*l[1]**2*l[2]**2*(-c[0,2]*np.tan(g[1]) - c[0,3]*np.tan(g[0]) + c[0,4] + c[4,2]*np.tan(g[1])**2 + c[4,3]*np.tan(g[0])*np.tan(g[1]) - c[4,4]*np.tan(g[1]) + c[5,2]*np.tan(g[0])*np.tan(g[1]) + c[5,3]*np.tan(g[0])**2 - c[5,4]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           -c[0,0]*l[1]*l[2]/(36*l[0]) - c[0,4]*l[1]/24 + c[0,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[0,5]*l[2]/24 + c[0,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[4,0]*l[1]/24 + c[4,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[4,4]*l[0]*l[1]/(36*l[2]) + c[4,4]*l[1]*np.tan(g[1])/12 - c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[4,5]*l[0]/24 + c[4,5]*l[1]*np.tan(g[0])/24 - c[4,5]*l[2]*np.tan(g[1])/24 - c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[5,0]*l[2]/24 + c[5,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[5,4]*l[0]/24 + c[5,4]*l[1]*np.tan(g[0])/24 - c[5,4]*l[2]*np.tan(g[1])/24 - c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,5]*l[0]*l[2]/(36*l[1]) - c[5,5]*l[2]*np.tan(g[0])/12 - c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]),
           c[0,1]*l[2]/24 + c[0,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[0,3]*l[1]/24 + c[0,3]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[0,5]*l[1]*l[2]/(36*l[0]) + c[4,1]*l[0]/24 + c[4,1]*l[1]*np.tan(g[0])/24 - c[4,1]*l[2]*np.tan(g[1])/24 - c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[4,3]*l[0]*l[1]/(36*l[2]) + c[4,3]*l[1]*np.tan(g[1])/12 - c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[4,5]*l[1]/24 + c[4,5]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,1]*l[0]*l[2]/(36*l[1]) - c[5,1]*l[2]*np.tan(g[0])/12 - c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[5,3]*l[0]/24 + c[5,3]*l[1]*np.tan(g[0])/24 - c[5,3]*l[2]*np.tan(g[1])/24 - c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[5,5]*l[2]/24 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           -c[0,2]*l[1]/24 + c[0,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[0,3]*l[2]/24 + c[0,3]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[0,4]*l[1]*l[2]/(36*l[0]) - c[4,2]*l[0]*l[1]/(36*l[2]) + c[4,2]*l[1]*np.tan(g[1])/12 - c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[4,3]*l[0]/24 + c[4,3]*l[1]*np.tan(g[0])/24 - c[4,3]*l[2]*np.tan(g[1])/24 - c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[4,4]*l[1]/24 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[5,2]*l[0]/24 + c[5,2]*l[1]*np.tan(g[0])/24 - c[5,2]*l[2]*np.tan(g[1])/24 - c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,3]*l[0]*l[2]/(36*l[1]) - c[5,3]*l[2]*np.tan(g[0])/12 - c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[5,4]*l[2]/24 + c[5,4]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           -c[0,0]*l[1]*l[2]/(9*l[0]) + c[0,4]*l[1]/12 + c[0,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[0,5]*l[2]/12 + c[0,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[4,0]*l[1]/12 + c[4,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[4,4]*l[0]*l[1]/(18*l[2]) - c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[4,5]*l[0]/24 + c[4,5]*l[1]*np.tan(g[0])/12 + c[4,5]*l[2]*np.tan(g[1])/12 - c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[5,0]*l[2]/12 + c[5,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[5,4]*l[0]/24 - c[5,4]*l[1]*np.tan(g[0])/12 - c[5,4]*l[2]*np.tan(g[1])/12 - c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[5,5]*l[0]*l[2]/(18*l[1]) - c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]),
           -c[0,1]*l[2]/12 + c[0,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[0,3]*l[1]/12 + c[0,3]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[0,5]*l[1]*l[2]/(9*l[0]) - c[4,1]*l[0]/24 + c[4,1]*l[1]*np.tan(g[0])/12 + c[4,1]*l[2]*np.tan(g[1])/12 - c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[4,3]*l[0]*l[1]/(18*l[2]) - c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[4,5]*l[1]/12 + c[4,5]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,1]*l[0]*l[2]/(18*l[1]) - c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[5,3]*l[0]/24 - c[5,3]*l[1]*np.tan(g[0])/12 - c[5,3]*l[2]*np.tan(g[1])/12 - c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[5,5]*l[2]/12 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           c[0,2]*l[1]/12 + c[0,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[0,3]*l[2]/12 + c[0,3]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[0,4]*l[1]*l[2]/(9*l[0]) + c[4,2]*l[0]*l[1]/(18*l[2]) - c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[4,3]*l[0]/24 + c[4,3]*l[1]*np.tan(g[0])/12 + c[4,3]*l[2]*np.tan(g[1])/12 - c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[4,4]*l[1]/12 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[5,2]*l[0]/24 - c[5,2]*l[1]*np.tan(g[0])/12 - c[5,2]*l[2]*np.tan(g[1])/12 - c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[5,3]*l[0]*l[2]/(18*l[1]) - c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[5,4]*l[2]/12 + c[5,4]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           (l[0]**2*(4*c[4,4]*l[1]**2 - 3*c[4,5]*l[1]*l[2] - 3*c[5,4]*l[1]*l[2] + 4*c[5,5]*l[2]**2) + 3*l[0]*l[1]*l[2]*(c[0,4]*l[1] - c[0,5]*l[2] + c[4,0]*l[1] - 2*c[4,4]*l[1]*np.tan(g[1]) - c[4,5]*l[1]*np.tan(g[0]) + c[4,5]*l[2]*np.tan(g[1]) - c[5,0]*l[2] - c[5,4]*l[1]*np.tan(g[0]) + c[5,4]*l[2]*np.tan(g[1]) + 2*c[5,5]*l[2]*np.tan(g[0])) + 4*l[1]**2*l[2]**2*(c[0,0] - c[0,4]*np.tan(g[1]) - c[0,5]*np.tan(g[0]) - c[4,0]*np.tan(g[1]) + c[4,4]*np.tan(g[1])**2 + c[4,5]*np.tan(g[0])*np.tan(g[1]) - c[5,0]*np.tan(g[0]) + c[5,4]*np.tan(g[0])*np.tan(g[1]) + c[5,5]*np.tan(g[0])**2))/(36*l[0]*l[1]*l[2]),
           (l[0]**2*(-3*c[4,1]*l[1]*l[2] + 4*c[4,3]*l[1]**2 + 4*c[5,1]*l[2]**2 - 3*c[5,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(-c[0,1]*l[2] + c[0,3]*l[1] - c[4,1]*l[1]*np.tan(g[0]) + c[4,1]*l[2]*np.tan(g[1]) - 2*c[4,3]*l[1]*np.tan(g[1]) + c[4,5]*l[1] + 2*c[5,1]*l[2]*np.tan(g[0]) - c[5,3]*l[1]*np.tan(g[0]) + c[5,3]*l[2]*np.tan(g[1]) - c[5,5]*l[2]) + 4*l[1]**2*l[2]**2*(-c[0,1]*np.tan(g[0]) - c[0,3]*np.tan(g[1]) + c[0,5] + c[4,1]*np.tan(g[0])*np.tan(g[1]) + c[4,3]*np.tan(g[1])**2 - c[4,5]*np.tan(g[1]) + c[5,1]*np.tan(g[0])**2 + c[5,3]*np.tan(g[0])*np.tan(g[1]) - c[5,5]*np.tan(g[0])))/(36*l[0]*l[1]*l[2]),
           (l[0]**2*(4*c[4,2]*l[1]**2 - 3*c[4,3]*l[1]*l[2] - 3*c[5,2]*l[1]*l[2] + 4*c[5,3]*l[2]**2) + 3*l[0]*l[1]*l[2]*(c[0,2]*l[1] - c[0,3]*l[2] - 2*c[4,2]*l[1]*np.tan(g[1]) - c[4,3]*l[1]*np.tan(g[0]) + c[4,3]*l[2]*np.tan(g[1]) + c[4,4]*l[1] - c[5,2]*l[1]*np.tan(g[0]) + c[5,2]*l[2]*np.tan(g[1]) + 2*c[5,3]*l[2]*np.tan(g[0]) - c[5,4]*l[2]) + 4*l[1]**2*l[2]**2*(-c[0,2]*np.tan(g[1]) - c[0,3]*np.tan(g[0]) + c[0,4] + c[4,2]*np.tan(g[1])**2 + c[4,3]*np.tan(g[0])*np.tan(g[1]) - c[4,4]*np.tan(g[1]) + c[5,2]*np.tan(g[0])*np.tan(g[1]) + c[5,3]*np.tan(g[0])**2 - c[5,4]*np.tan(g[0])))/(36*l[0]*l[1]*l[2]),
           (2*l[0]**2*(2*c[4,4]*l[1]**2 + 3*c[4,5]*l[1]*l[2] - 3*c[5,4]*l[1]*l[2] - 4*c[5,5]*l[2]**2) + 3*l[0]*l[1]*l[2]*(c[0,4]*l[1] + 2*c[0,5]*l[2] + c[4,0]*l[1] - 2*c[4,4]*l[1]*np.tan(g[1]) - c[4,5]*l[1]*np.tan(g[0]) - 2*c[4,5]*l[2]*np.tan(g[1]) - 2*c[5,0]*l[2] - c[5,4]*l[1]*np.tan(g[0]) + 2*c[5,4]*l[2]*np.tan(g[1])) + 4*l[1]**2*l[2]**2*(c[0,0] - c[0,4]*np.tan(g[1]) - c[0,5]*np.tan(g[0]) - c[4,0]*np.tan(g[1]) + c[4,4]*np.tan(g[1])**2 + c[4,5]*np.tan(g[0])*np.tan(g[1]) - c[5,0]*np.tan(g[0]) + c[5,4]*np.tan(g[0])*np.tan(g[1]) + c[5,5]*np.tan(g[0])**2))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(3*c[4,1]*l[1]*l[2] + 2*c[4,3]*l[1]**2 - 4*c[5,1]*l[2]**2 - 3*c[5,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(2*c[0,1]*l[2] + c[0,3]*l[1] - c[4,1]*l[1]*np.tan(g[0]) - 2*c[4,1]*l[2]*np.tan(g[1]) - 2*c[4,3]*l[1]*np.tan(g[1]) + c[4,5]*l[1] - c[5,3]*l[1]*np.tan(g[0]) + 2*c[5,3]*l[2]*np.tan(g[1]) - 2*c[5,5]*l[2]) + 4*l[1]**2*l[2]**2*(-c[0,1]*np.tan(g[0]) - c[0,3]*np.tan(g[1]) + c[0,5] + c[4,1]*np.tan(g[0])*np.tan(g[1]) + c[4,3]*np.tan(g[1])**2 - c[4,5]*np.tan(g[1]) + c[5,1]*np.tan(g[0])**2 + c[5,3]*np.tan(g[0])*np.tan(g[1]) - c[5,5]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(2*c[4,2]*l[1]**2 + 3*c[4,3]*l[1]*l[2] - 3*c[5,2]*l[1]*l[2] - 4*c[5,3]*l[2]**2) + 3*l[0]*l[1]*l[2]*(c[0,2]*l[1] + 2*c[0,3]*l[2] - 2*c[4,2]*l[1]*np.tan(g[1]) - c[4,3]*l[1]*np.tan(g[0]) - 2*c[4,3]*l[2]*np.tan(g[1]) + c[4,4]*l[1] - c[5,2]*l[1]*np.tan(g[0]) + 2*c[5,2]*l[2]*np.tan(g[1]) - 2*c[5,4]*l[2]) + 4*l[1]**2*l[2]**2*(-c[0,2]*np.tan(g[1]) - c[0,3]*np.tan(g[0]) + c[0,4] + c[4,2]*np.tan(g[1])**2 + c[4,3]*np.tan(g[0])*np.tan(g[1]) - c[4,4]*np.tan(g[1]) + c[5,2]*np.tan(g[0])*np.tan(g[1]) + c[5,3]*np.tan(g[0])**2 - c[5,4]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           -c[0,0]*l[1]*l[2]/(18*l[0]) + c[0,4]*l[1]/24 + c[0,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[0,5]*l[2]/12 + c[0,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,0]*l[1]/24 + c[4,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,4]*l[0]*l[1]/(36*l[2]) - c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[4,5]*l[0]/24 + c[4,5]*l[1]*np.tan(g[0])/24 - c[4,5]*l[2]*np.tan(g[1])/12 - c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,0]*l[2]/12 + c[5,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[5,4]*l[0]/24 - c[5,4]*l[1]*np.tan(g[0])/24 - c[5,4]*l[2]*np.tan(g[1])/12 - c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,5]*l[0]*l[2]/(18*l[1]) - c[5,5]*l[2]*np.tan(g[0])/6 - c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]),
           c[0,1]*l[2]/12 + c[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[0,3]*l[1]/24 + c[0,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[0,5]*l[1]*l[2]/(18*l[0]) + c[4,1]*l[0]/24 + c[4,1]*l[1]*np.tan(g[0])/24 - c[4,1]*l[2]*np.tan(g[1])/12 - c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[4,3]*l[0]*l[1]/(36*l[2]) - c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[4,5]*l[1]/24 + c[4,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,1]*l[0]*l[2]/(18*l[1]) - c[5,1]*l[2]*np.tan(g[0])/6 - c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[5,3]*l[0]/24 - c[5,3]*l[1]*np.tan(g[0])/24 - c[5,3]*l[2]*np.tan(g[1])/12 - c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,5]*l[2]/12 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[0,2]*l[1]/24 + c[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[0,3]*l[2]/12 + c[0,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[0,4]*l[1]*l[2]/(18*l[0]) + c[4,2]*l[0]*l[1]/(36*l[2]) - c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[4,3]*l[0]/24 + c[4,3]*l[1]*np.tan(g[0])/24 - c[4,3]*l[2]*np.tan(g[1])/12 - c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[4,4]*l[1]/24 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,2]*l[0]/24 - c[5,2]*l[1]*np.tan(g[0])/24 - c[5,2]*l[2]*np.tan(g[1])/12 - c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,3]*l[0]*l[2]/(18*l[1]) - c[5,3]*l[2]*np.tan(g[0])/6 - c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[5,4]*l[2]/12 + c[5,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0])],
          [c[1,0]*l[2]/24 + c[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[1,4]*l[0]/24 + c[1,4]*l[1]*np.tan(g[0])/12 - c[1,4]*l[2]*np.tan(g[1])/24 - c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[1,5]*l[0]*l[2]/(36*l[1]) - c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[3,0]*l[1]/12 + c[3,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[3,4]*l[0]*l[1]/(18*l[2]) + c[3,4]*l[1]*np.tan(g[1])/6 - c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[3,5]*l[0]/24 + c[3,5]*l[1]*np.tan(g[0])/12 + c[3,5]*l[2]*np.tan(g[1])/24 - c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,0]*l[1]*l[2]/(18*l[0]) - c[5,4]*l[1]/12 + c[5,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,5]*l[2]/24 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[1,1]*l[0]*l[2]/(36*l[1]) - c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[1,3]*l[0]/24 + c[1,3]*l[1]*np.tan(g[0])/12 - c[1,3]*l[2]*np.tan(g[1])/24 - c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[1,5]*l[2]/24 + c[1,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[3,1]*l[0]/24 + c[3,1]*l[1]*np.tan(g[0])/12 + c[3,1]*l[2]*np.tan(g[1])/24 - c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,3]*l[0]*l[1]/(18*l[2]) + c[3,3]*l[1]*np.tan(g[1])/6 - c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[3,5]*l[1]/12 + c[3,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,1]*l[2]/24 + c[5,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[5,3]*l[1]/12 + c[5,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,5]*l[1]*l[2]/(18*l[0]),
           c[1,2]*l[0]/24 + c[1,2]*l[1]*np.tan(g[0])/12 - c[1,2]*l[2]*np.tan(g[1])/24 - c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[1,3]*l[0]*l[2]/(36*l[1]) - c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[1,4]*l[2]/24 + c[1,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[3,2]*l[0]*l[1]/(18*l[2]) + c[3,2]*l[1]*np.tan(g[1])/6 - c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[3,3]*l[0]/24 + c[3,3]*l[1]*np.tan(g[0])/12 + c[3,3]*l[2]*np.tan(g[1])/24 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,4]*l[1]/12 + c[3,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,2]*l[1]/12 + c[5,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,3]*l[2]/24 + c[5,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[5,4]*l[1]*l[2]/(18*l[0]),
           (2*l[0]**2*(3*c[1,4]*l[1]*l[2] + 2*c[1,5]*l[2]**2 - 4*c[3,4]*l[1]**2 - 3*c[3,5]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(-c[1,0]*l[2] + 2*c[1,4]*l[1]*np.tan(g[0]) + c[1,4]*l[2]*np.tan(g[1]) + 2*c[1,5]*l[2]*np.tan(g[0]) + 2*c[3,0]*l[1] - 2*c[3,5]*l[1]*np.tan(g[0]) + c[3,5]*l[2]*np.tan(g[1]) - 2*c[5,4]*l[1] - c[5,5]*l[2]) + 4*l[1]**2*l[2]**2*(-c[1,0]*np.tan(g[0]) + c[1,4]*np.tan(g[0])*np.tan(g[1]) + c[1,5]*np.tan(g[0])**2 - c[3,0]*np.tan(g[1]) + c[3,4]*np.tan(g[1])**2 + c[3,5]*np.tan(g[0])*np.tan(g[1]) + c[5,0] - c[5,4]*np.tan(g[1]) - c[5,5]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(2*c[1,1]*l[2]**2 + 3*c[1,3]*l[1]*l[2] - 3*c[3,1]*l[1]*l[2] - 4*c[3,3]*l[1]**2) + 3*l[0]*l[1]*l[2]*(2*c[1,1]*l[2]*np.tan(g[0]) + 2*c[1,3]*l[1]*np.tan(g[0]) + c[1,3]*l[2]*np.tan(g[1]) - c[1,5]*l[2] - 2*c[3,1]*l[1]*np.tan(g[0]) + c[3,1]*l[2]*np.tan(g[1]) + 2*c[3,5]*l[1] - c[5,1]*l[2] - 2*c[5,3]*l[1]) + 4*l[1]**2*l[2]**2*(c[1,1]*np.tan(g[0])**2 + c[1,3]*np.tan(g[0])*np.tan(g[1]) - c[1,5]*np.tan(g[0]) + c[3,1]*np.tan(g[0])*np.tan(g[1]) + c[3,3]*np.tan(g[1])**2 - c[3,5]*np.tan(g[1]) - c[5,1]*np.tan(g[0]) - c[5,3]*np.tan(g[1]) + c[5,5]))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(3*c[1,2]*l[1]*l[2] + 2*c[1,3]*l[2]**2 - 4*c[3,2]*l[1]**2 - 3*c[3,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(2*c[1,2]*l[1]*np.tan(g[0]) + c[1,2]*l[2]*np.tan(g[1]) + 2*c[1,3]*l[2]*np.tan(g[0]) - c[1,4]*l[2] - 2*c[3,3]*l[1]*np.tan(g[0]) + c[3,3]*l[2]*np.tan(g[1]) + 2*c[3,4]*l[1] - 2*c[5,2]*l[1] - c[5,3]*l[2]) + 4*l[1]**2*l[2]**2*(c[1,2]*np.tan(g[0])*np.tan(g[1]) + c[1,3]*np.tan(g[0])**2 - c[1,4]*np.tan(g[0]) + c[3,2]*np.tan(g[1])**2 + c[3,3]*np.tan(g[0])*np.tan(g[1]) - c[3,4]*np.tan(g[1]) - c[5,2]*np.tan(g[1]) - c[5,3]*np.tan(g[0]) + c[5,4]))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(3*c[1,4]*l[1]*l[2] - 2*c[1,5]*l[2]**2 - 2*c[3,4]*l[1]**2 + 3*c[3,5]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(-c[1,0]*l[2] + c[1,4]*l[1]*np.tan(g[0]) + c[1,4]*l[2]*np.tan(g[1]) + c[3,0]*l[1] - c[3,5]*l[1]*np.tan(g[0]) - c[3,5]*l[2]*np.tan(g[1]) - c[5,4]*l[1] + c[5,5]*l[2]) + 2*l[1]**2*l[2]**2*(-c[1,0]*np.tan(g[0]) + c[1,4]*np.tan(g[0])*np.tan(g[1]) + c[1,5]*np.tan(g[0])**2 - c[3,0]*np.tan(g[1]) + c[3,4]*np.tan(g[1])**2 + c[3,5]*np.tan(g[0])*np.tan(g[1]) + c[5,0] - c[5,4]*np.tan(g[1]) - c[5,5]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(-2*c[1,1]*l[2]**2 + 3*c[1,3]*l[1]*l[2] + 3*c[3,1]*l[1]*l[2] - 2*c[3,3]*l[1]**2) + 3*l[0]*l[1]*l[2]*(c[1,3]*l[1]*np.tan(g[0]) + c[1,3]*l[2]*np.tan(g[1]) - c[1,5]*l[2] - c[3,1]*l[1]*np.tan(g[0]) - c[3,1]*l[2]*np.tan(g[1]) + c[3,5]*l[1] + c[5,1]*l[2] - c[5,3]*l[1]) + 2*l[1]**2*l[2]**2*(c[1,1]*np.tan(g[0])**2 + c[1,3]*np.tan(g[0])*np.tan(g[1]) - c[1,5]*np.tan(g[0]) + c[3,1]*np.tan(g[0])*np.tan(g[1]) + c[3,3]*np.tan(g[1])**2 - c[3,5]*np.tan(g[1]) - c[5,1]*np.tan(g[0]) - c[5,3]*np.tan(g[1]) + c[5,5]))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(3*c[1,2]*l[1]*l[2] - 2*c[1,3]*l[2]**2 - 2*c[3,2]*l[1]**2 + 3*c[3,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(c[1,2]*l[1]*np.tan(g[0]) + c[1,2]*l[2]*np.tan(g[1]) - c[1,4]*l[2] - c[3,3]*l[1]*np.tan(g[0]) - c[3,3]*l[2]*np.tan(g[1]) + c[3,4]*l[1] - c[5,2]*l[1] + c[5,3]*l[2]) + 2*l[1]**2*l[2]**2*(c[1,2]*np.tan(g[0])*np.tan(g[1]) + c[1,3]*np.tan(g[0])**2 - c[1,4]*np.tan(g[0]) + c[3,2]*np.tan(g[1])**2 + c[3,3]*np.tan(g[0])*np.tan(g[1]) - c[3,4]*np.tan(g[1]) - c[5,2]*np.tan(g[1]) - c[5,3]*np.tan(g[0]) + c[5,4]))/(72*l[0]*l[1]*l[2]),
           c[1,0]*l[2]/24 + c[1,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[1,4]*l[0]/24 + c[1,4]*l[1]*np.tan(g[0])/24 - c[1,4]*l[2]*np.tan(g[1])/24 - c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[1,5]*l[0]*l[2]/(36*l[1]) - c[1,5]*l[2]*np.tan(g[0])/12 - c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[3,0]*l[1]/24 + c[3,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[3,4]*l[0]*l[1]/(36*l[2]) + c[3,4]*l[1]*np.tan(g[1])/12 - c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[3,5]*l[0]/24 + c[3,5]*l[1]*np.tan(g[0])/24 - c[3,5]*l[2]*np.tan(g[1])/24 - c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,0]*l[1]*l[2]/(36*l[0]) - c[5,4]*l[1]/24 + c[5,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[5,5]*l[2]/24 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           -c[1,1]*l[0]*l[2]/(36*l[1]) - c[1,1]*l[2]*np.tan(g[0])/12 - c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[1,3]*l[0]/24 + c[1,3]*l[1]*np.tan(g[0])/24 - c[1,3]*l[2]*np.tan(g[1])/24 - c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[1,5]*l[2]/24 + c[1,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[3,1]*l[0]/24 + c[3,1]*l[1]*np.tan(g[0])/24 - c[3,1]*l[2]*np.tan(g[1])/24 - c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,3]*l[0]*l[1]/(36*l[2]) + c[3,3]*l[1]*np.tan(g[1])/12 - c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[3,5]*l[1]/24 + c[3,5]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[5,1]*l[2]/24 + c[5,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[5,3]*l[1]/24 + c[5,3]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,5]*l[1]*l[2]/(36*l[0]),
           c[1,2]*l[0]/24 + c[1,2]*l[1]*np.tan(g[0])/24 - c[1,2]*l[2]*np.tan(g[1])/24 - c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[1,3]*l[0]*l[2]/(36*l[1]) - c[1,3]*l[2]*np.tan(g[0])/12 - c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[1,4]*l[2]/24 + c[1,4]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[3,2]*l[0]*l[1]/(36*l[2]) + c[3,2]*l[1]*np.tan(g[1])/12 - c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[3,3]*l[0]/24 + c[3,3]*l[1]*np.tan(g[0])/24 - c[3,3]*l[2]*np.tan(g[1])/24 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,4]*l[1]/24 + c[3,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,2]*l[1]/24 + c[5,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[5,3]*l[2]/24 + c[5,3]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[5,4]*l[1]*l[2]/(36*l[0]),
           c[1,0]*l[2]/12 + c[1,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[1,4]*l[0]/24 - c[1,4]*l[1]*np.tan(g[0])/12 - c[1,4]*l[2]*np.tan(g[1])/12 - c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[1,5]*l[0]*l[2]/(18*l[1]) - c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[3,0]*l[1]/12 + c[3,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[3,4]*l[0]*l[1]/(18*l[2]) - c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[3,5]*l[0]/24 + c[3,5]*l[1]*np.tan(g[0])/12 + c[3,5]*l[2]*np.tan(g[1])/12 - c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[5,0]*l[1]*l[2]/(9*l[0]) + c[5,4]*l[1]/12 + c[5,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[5,5]*l[2]/12 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           c[1,1]*l[0]*l[2]/(18*l[1]) - c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[1,3]*l[0]/24 - c[1,3]*l[1]*np.tan(g[0])/12 - c[1,3]*l[2]*np.tan(g[1])/12 - c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[1,5]*l[2]/12 + c[1,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[3,1]*l[0]/24 + c[3,1]*l[1]*np.tan(g[0])/12 + c[3,1]*l[2]*np.tan(g[1])/12 - c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,3]*l[0]*l[1]/(18*l[2]) - c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[3,5]*l[1]/12 + c[3,5]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[5,1]*l[2]/12 + c[5,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[5,3]*l[1]/12 + c[5,3]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[5,5]*l[1]*l[2]/(9*l[0]),
           -c[1,2]*l[0]/24 - c[1,2]*l[1]*np.tan(g[0])/12 - c[1,2]*l[2]*np.tan(g[1])/12 - c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[1,3]*l[0]*l[2]/(18*l[1]) - c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[1,4]*l[2]/12 + c[1,4]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[3,2]*l[0]*l[1]/(18*l[2]) - c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[3,3]*l[0]/24 + c[3,3]*l[1]*np.tan(g[0])/12 + c[3,3]*l[2]*np.tan(g[1])/12 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[3,4]*l[1]/12 + c[3,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,2]*l[1]/12 + c[5,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[5,3]*l[2]/12 + c[5,3]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[5,4]*l[1]*l[2]/(9*l[0]),
           (l[0]**2*(-3*c[1,4]*l[1]*l[2] + 4*c[1,5]*l[2]**2 + 4*c[3,4]*l[1]**2 - 3*c[3,5]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(-c[1,0]*l[2] - c[1,4]*l[1]*np.tan(g[0]) + c[1,4]*l[2]*np.tan(g[1]) + 2*c[1,5]*l[2]*np.tan(g[0]) + c[3,0]*l[1] - 2*c[3,4]*l[1]*np.tan(g[1]) - c[3,5]*l[1]*np.tan(g[0]) + c[3,5]*l[2]*np.tan(g[1]) + c[5,4]*l[1] - c[5,5]*l[2]) + 4*l[1]**2*l[2]**2*(-c[1,0]*np.tan(g[0]) + c[1,4]*np.tan(g[0])*np.tan(g[1]) + c[1,5]*np.tan(g[0])**2 - c[3,0]*np.tan(g[1]) + c[3,4]*np.tan(g[1])**2 + c[3,5]*np.tan(g[0])*np.tan(g[1]) + c[5,0] - c[5,4]*np.tan(g[1]) - c[5,5]*np.tan(g[0])))/(36*l[0]*l[1]*l[2]),
           (l[0]**2*(4*c[1,1]*l[2]**2 - 3*c[1,3]*l[1]*l[2] - 3*c[3,1]*l[1]*l[2] + 4*c[3,3]*l[1]**2) + 3*l[0]*l[1]*l[2]*(2*c[1,1]*l[2]*np.tan(g[0]) - c[1,3]*l[1]*np.tan(g[0]) + c[1,3]*l[2]*np.tan(g[1]) - c[1,5]*l[2] - c[3,1]*l[1]*np.tan(g[0]) + c[3,1]*l[2]*np.tan(g[1]) - 2*c[3,3]*l[1]*np.tan(g[1]) + c[3,5]*l[1] - c[5,1]*l[2] + c[5,3]*l[1]) + 4*l[1]**2*l[2]**2*(c[1,1]*np.tan(g[0])**2 + c[1,3]*np.tan(g[0])*np.tan(g[1]) - c[1,5]*np.tan(g[0]) + c[3,1]*np.tan(g[0])*np.tan(g[1]) + c[3,3]*np.tan(g[1])**2 - c[3,5]*np.tan(g[1]) - c[5,1]*np.tan(g[0]) - c[5,3]*np.tan(g[1]) + c[5,5]))/(36*l[0]*l[1]*l[2]),
           (l[0]**2*(-3*c[1,2]*l[1]*l[2] + 4*c[1,3]*l[2]**2 + 4*c[3,2]*l[1]**2 - 3*c[3,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(-c[1,2]*l[1]*np.tan(g[0]) + c[1,2]*l[2]*np.tan(g[1]) + 2*c[1,3]*l[2]*np.tan(g[0]) - c[1,4]*l[2] - 2*c[3,2]*l[1]*np.tan(g[1]) - c[3,3]*l[1]*np.tan(g[0]) + c[3,3]*l[2]*np.tan(g[1]) + c[3,4]*l[1] + c[5,2]*l[1] - c[5,3]*l[2]) + 4*l[1]**2*l[2]**2*(c[1,2]*np.tan(g[0])*np.tan(g[1]) + c[1,3]*np.tan(g[0])**2 - c[1,4]*np.tan(g[0]) + c[3,2]*np.tan(g[1])**2 + c[3,3]*np.tan(g[0])*np.tan(g[1]) - c[3,4]*np.tan(g[1]) - c[5,2]*np.tan(g[1]) - c[5,3]*np.tan(g[0]) + c[5,4]))/(36*l[0]*l[1]*l[2]),
           (2*l[0]**2*(-3*c[1,4]*l[1]*l[2] - 4*c[1,5]*l[2]**2 + 2*c[3,4]*l[1]**2 + 3*c[3,5]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(-2*c[1,0]*l[2] - c[1,4]*l[1]*np.tan(g[0]) + 2*c[1,4]*l[2]*np.tan(g[1]) + c[3,0]*l[1] - 2*c[3,4]*l[1]*np.tan(g[1]) - c[3,5]*l[1]*np.tan(g[0]) - 2*c[3,5]*l[2]*np.tan(g[1]) + c[5,4]*l[1] + 2*c[5,5]*l[2]) + 4*l[1]**2*l[2]**2*(-c[1,0]*np.tan(g[0]) + c[1,4]*np.tan(g[0])*np.tan(g[1]) + c[1,5]*np.tan(g[0])**2 - c[3,0]*np.tan(g[1]) + c[3,4]*np.tan(g[1])**2 + c[3,5]*np.tan(g[0])*np.tan(g[1]) + c[5,0] - c[5,4]*np.tan(g[1]) - c[5,5]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(-4*c[1,1]*l[2]**2 - 3*c[1,3]*l[1]*l[2] + 3*c[3,1]*l[1]*l[2] + 2*c[3,3]*l[1]**2) + 3*l[0]*l[1]*l[2]*(-c[1,3]*l[1]*np.tan(g[0]) + 2*c[1,3]*l[2]*np.tan(g[1]) - 2*c[1,5]*l[2] - c[3,1]*l[1]*np.tan(g[0]) - 2*c[3,1]*l[2]*np.tan(g[1]) - 2*c[3,3]*l[1]*np.tan(g[1]) + c[3,5]*l[1] + 2*c[5,1]*l[2] + c[5,3]*l[1]) + 4*l[1]**2*l[2]**2*(c[1,1]*np.tan(g[0])**2 + c[1,3]*np.tan(g[0])*np.tan(g[1]) - c[1,5]*np.tan(g[0]) + c[3,1]*np.tan(g[0])*np.tan(g[1]) + c[3,3]*np.tan(g[1])**2 - c[3,5]*np.tan(g[1]) - c[5,1]*np.tan(g[0]) - c[5,3]*np.tan(g[1]) + c[5,5]))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(-3*c[1,2]*l[1]*l[2] - 4*c[1,3]*l[2]**2 + 2*c[3,2]*l[1]**2 + 3*c[3,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(-c[1,2]*l[1]*np.tan(g[0]) + 2*c[1,2]*l[2]*np.tan(g[1]) - 2*c[1,4]*l[2] - 2*c[3,2]*l[1]*np.tan(g[1]) - c[3,3]*l[1]*np.tan(g[0]) - 2*c[3,3]*l[2]*np.tan(g[1]) + c[3,4]*l[1] + c[5,2]*l[1] + 2*c[5,3]*l[2]) + 4*l[1]**2*l[2]**2*(c[1,2]*np.tan(g[0])*np.tan(g[1]) + c[1,3]*np.tan(g[0])**2 - c[1,4]*np.tan(g[0]) + c[3,2]*np.tan(g[1])**2 + c[3,3]*np.tan(g[0])*np.tan(g[1]) - c[3,4]*np.tan(g[1]) - c[5,2]*np.tan(g[1]) - c[5,3]*np.tan(g[0]) + c[5,4]))/(72*l[0]*l[1]*l[2]),
           c[1,0]*l[2]/12 + c[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[1,4]*l[0]/24 - c[1,4]*l[1]*np.tan(g[0])/24 - c[1,4]*l[2]*np.tan(g[1])/12 - c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[1,5]*l[0]*l[2]/(18*l[1]) - c[1,5]*l[2]*np.tan(g[0])/6 - c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[3,0]*l[1]/24 + c[3,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[3,4]*l[0]*l[1]/(36*l[2]) - c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[3,5]*l[0]/24 + c[3,5]*l[1]*np.tan(g[0])/24 - c[3,5]*l[2]*np.tan(g[1])/12 - c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,0]*l[1]*l[2]/(18*l[0]) + c[5,4]*l[1]/24 + c[5,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,5]*l[2]/12 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[1,1]*l[0]*l[2]/(18*l[1]) - c[1,1]*l[2]*np.tan(g[0])/6 - c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[1,3]*l[0]/24 - c[1,3]*l[1]*np.tan(g[0])/24 - c[1,3]*l[2]*np.tan(g[1])/12 - c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[1,5]*l[2]/12 + c[1,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[3,1]*l[0]/24 + c[3,1]*l[1]*np.tan(g[0])/24 - c[3,1]*l[2]*np.tan(g[1])/12 - c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,3]*l[0]*l[1]/(36*l[2]) - c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[3,5]*l[1]/24 + c[3,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,1]*l[2]/12 + c[5,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[5,3]*l[1]/24 + c[5,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,5]*l[1]*l[2]/(18*l[0]),
           -c[1,2]*l[0]/24 - c[1,2]*l[1]*np.tan(g[0])/24 - c[1,2]*l[2]*np.tan(g[1])/12 - c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[1,3]*l[0]*l[2]/(18*l[1]) - c[1,3]*l[2]*np.tan(g[0])/6 - c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[1,4]*l[2]/12 + c[1,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[3,2]*l[0]*l[1]/(36*l[2]) - c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[3,3]*l[0]/24 + c[3,3]*l[1]*np.tan(g[0])/24 - c[3,3]*l[2]*np.tan(g[1])/12 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,4]*l[1]/24 + c[3,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,2]*l[1]/24 + c[5,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,3]*l[2]/12 + c[5,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[5,4]*l[1]*l[2]/(18*l[0])],
          [-c[2,0]*l[1]/12 + c[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[2,4]*l[0]*l[1]/(18*l[2]) + c[2,4]*l[1]*np.tan(g[1])/6 - c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[2,5]*l[0]/24 + c[2,5]*l[1]*np.tan(g[0])/12 + c[2,5]*l[2]*np.tan(g[1])/24 - c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,0]*l[2]/24 + c[3,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[3,4]*l[0]/24 + c[3,4]*l[1]*np.tan(g[0])/12 - c[3,4]*l[2]*np.tan(g[1])/24 - c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,5]*l[0]*l[2]/(36*l[1]) - c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[4,0]*l[1]*l[2]/(18*l[0]) - c[4,4]*l[1]/12 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,5]*l[2]/24 + c[4,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[2,1]*l[0]/24 + c[2,1]*l[1]*np.tan(g[0])/12 + c[2,1]*l[2]*np.tan(g[1])/24 - c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[2,3]*l[0]*l[1]/(18*l[2]) + c[2,3]*l[1]*np.tan(g[1])/6 - c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[2,5]*l[1]/12 + c[2,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[3,1]*l[0]*l[2]/(36*l[1]) - c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[3,3]*l[0]/24 + c[3,3]*l[1]*np.tan(g[0])/12 - c[3,3]*l[2]*np.tan(g[1])/24 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,5]*l[2]/24 + c[3,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,1]*l[2]/24 + c[4,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,3]*l[1]/12 + c[4,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,5]*l[1]*l[2]/(18*l[0]),
           -c[2,2]*l[0]*l[1]/(18*l[2]) + c[2,2]*l[1]*np.tan(g[1])/6 - c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[2,3]*l[0]/24 + c[2,3]*l[1]*np.tan(g[0])/12 + c[2,3]*l[2]*np.tan(g[1])/24 - c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[2,4]*l[1]/12 + c[2,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[3,2]*l[0]/24 + c[3,2]*l[1]*np.tan(g[0])/12 - c[3,2]*l[2]*np.tan(g[1])/24 - c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,3]*l[0]*l[2]/(36*l[1]) - c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[3,4]*l[2]/24 + c[3,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,2]*l[1]/12 + c[4,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,3]*l[2]/24 + c[4,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,4]*l[1]*l[2]/(18*l[0]),
           (2*l[0]**2*(-4*c[2,4]*l[1]**2 - 3*c[2,5]*l[1]*l[2] + 3*c[3,4]*l[1]*l[2] + 2*c[3,5]*l[2]**2) + 3*l[0]*l[1]*l[2]*(2*c[2,0]*l[1] - 2*c[2,5]*l[1]*np.tan(g[0]) + c[2,5]*l[2]*np.tan(g[1]) - c[3,0]*l[2] + 2*c[3,4]*l[1]*np.tan(g[0]) + c[3,4]*l[2]*np.tan(g[1]) + 2*c[3,5]*l[2]*np.tan(g[0]) - 2*c[4,4]*l[1] - c[4,5]*l[2]) + 4*l[1]**2*l[2]**2*(-c[2,0]*np.tan(g[1]) + c[2,4]*np.tan(g[1])**2 + c[2,5]*np.tan(g[0])*np.tan(g[1]) - c[3,0]*np.tan(g[0]) + c[3,4]*np.tan(g[0])*np.tan(g[1]) + c[3,5]*np.tan(g[0])**2 + c[4,0] - c[4,4]*np.tan(g[1]) - c[4,5]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(-3*c[2,1]*l[1]*l[2] - 4*c[2,3]*l[1]**2 + 2*c[3,1]*l[2]**2 + 3*c[3,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(-2*c[2,1]*l[1]*np.tan(g[0]) + c[2,1]*l[2]*np.tan(g[1]) + 2*c[2,5]*l[1] + 2*c[3,1]*l[2]*np.tan(g[0]) + 2*c[3,3]*l[1]*np.tan(g[0]) + c[3,3]*l[2]*np.tan(g[1]) - c[3,5]*l[2] - c[4,1]*l[2] - 2*c[4,3]*l[1]) + 4*l[1]**2*l[2]**2*(c[2,1]*np.tan(g[0])*np.tan(g[1]) + c[2,3]*np.tan(g[1])**2 - c[2,5]*np.tan(g[1]) + c[3,1]*np.tan(g[0])**2 + c[3,3]*np.tan(g[0])*np.tan(g[1]) - c[3,5]*np.tan(g[0]) - c[4,1]*np.tan(g[0]) - c[4,3]*np.tan(g[1]) + c[4,5]))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(-4*c[2,2]*l[1]**2 - 3*c[2,3]*l[1]*l[2] + 3*c[3,2]*l[1]*l[2] + 2*c[3,3]*l[2]**2) + 3*l[0]*l[1]*l[2]*(-2*c[2,3]*l[1]*np.tan(g[0]) + c[2,3]*l[2]*np.tan(g[1]) + 2*c[2,4]*l[1] + 2*c[3,2]*l[1]*np.tan(g[0]) + c[3,2]*l[2]*np.tan(g[1]) + 2*c[3,3]*l[2]*np.tan(g[0]) - c[3,4]*l[2] - 2*c[4,2]*l[1] - c[4,3]*l[2]) + 4*l[1]**2*l[2]**2*(c[2,2]*np.tan(g[1])**2 + c[2,3]*np.tan(g[0])*np.tan(g[1]) - c[2,4]*np.tan(g[1]) + c[3,2]*np.tan(g[0])*np.tan(g[1]) + c[3,3]*np.tan(g[0])**2 - c[3,4]*np.tan(g[0]) - c[4,2]*np.tan(g[1]) - c[4,3]*np.tan(g[0]) + c[4,4]))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(-2*c[2,4]*l[1]**2 + 3*c[2,5]*l[1]*l[2] + 3*c[3,4]*l[1]*l[2] - 2*c[3,5]*l[2]**2) + 3*l[0]*l[1]*l[2]*(c[2,0]*l[1] - c[2,5]*l[1]*np.tan(g[0]) - c[2,5]*l[2]*np.tan(g[1]) - c[3,0]*l[2] + c[3,4]*l[1]*np.tan(g[0]) + c[3,4]*l[2]*np.tan(g[1]) - c[4,4]*l[1] + c[4,5]*l[2]) + 2*l[1]**2*l[2]**2*(-c[2,0]*np.tan(g[1]) + c[2,4]*np.tan(g[1])**2 + c[2,5]*np.tan(g[0])*np.tan(g[1]) - c[3,0]*np.tan(g[0]) + c[3,4]*np.tan(g[0])*np.tan(g[1]) + c[3,5]*np.tan(g[0])**2 + c[4,0] - c[4,4]*np.tan(g[1]) - c[4,5]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(3*c[2,1]*l[1]*l[2] - 2*c[2,3]*l[1]**2 - 2*c[3,1]*l[2]**2 + 3*c[3,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(-c[2,1]*l[1]*np.tan(g[0]) - c[2,1]*l[2]*np.tan(g[1]) + c[2,5]*l[1] + c[3,3]*l[1]*np.tan(g[0]) + c[3,3]*l[2]*np.tan(g[1]) - c[3,5]*l[2] + c[4,1]*l[2] - c[4,3]*l[1]) + 2*l[1]**2*l[2]**2*(c[2,1]*np.tan(g[0])*np.tan(g[1]) + c[2,3]*np.tan(g[1])**2 - c[2,5]*np.tan(g[1]) + c[3,1]*np.tan(g[0])**2 + c[3,3]*np.tan(g[0])*np.tan(g[1]) - c[3,5]*np.tan(g[0]) - c[4,1]*np.tan(g[0]) - c[4,3]*np.tan(g[1]) + c[4,5]))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(-2*c[2,2]*l[1]**2 + 3*c[2,3]*l[1]*l[2] + 3*c[3,2]*l[1]*l[2] - 2*c[3,3]*l[2]**2) + 3*l[0]*l[1]*l[2]*(-c[2,3]*l[1]*np.tan(g[0]) - c[2,3]*l[2]*np.tan(g[1]) + c[2,4]*l[1] + c[3,2]*l[1]*np.tan(g[0]) + c[3,2]*l[2]*np.tan(g[1]) - c[3,4]*l[2] - c[4,2]*l[1] + c[4,3]*l[2]) + 2*l[1]**2*l[2]**2*(c[2,2]*np.tan(g[1])**2 + c[2,3]*np.tan(g[0])*np.tan(g[1]) - c[2,4]*np.tan(g[1]) + c[3,2]*np.tan(g[0])*np.tan(g[1]) + c[3,3]*np.tan(g[0])**2 - c[3,4]*np.tan(g[0]) - c[4,2]*np.tan(g[1]) - c[4,3]*np.tan(g[0]) + c[4,4]))/(72*l[0]*l[1]*l[2]),
           -c[2,0]*l[1]/24 + c[2,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[2,4]*l[0]*l[1]/(36*l[2]) + c[2,4]*l[1]*np.tan(g[1])/12 - c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[2,5]*l[0]/24 + c[2,5]*l[1]*np.tan(g[0])/24 - c[2,5]*l[2]*np.tan(g[1])/24 - c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[3,0]*l[2]/24 + c[3,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[3,4]*l[0]/24 + c[3,4]*l[1]*np.tan(g[0])/24 - c[3,4]*l[2]*np.tan(g[1])/24 - c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,5]*l[0]*l[2]/(36*l[1]) - c[3,5]*l[2]*np.tan(g[0])/12 - c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[4,0]*l[1]*l[2]/(36*l[0]) - c[4,4]*l[1]/24 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[4,5]*l[2]/24 + c[4,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           c[2,1]*l[0]/24 + c[2,1]*l[1]*np.tan(g[0])/24 - c[2,1]*l[2]*np.tan(g[1])/24 - c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[2,3]*l[0]*l[1]/(36*l[2]) + c[2,3]*l[1]*np.tan(g[1])/12 - c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[2,5]*l[1]/24 + c[2,5]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[3,1]*l[0]*l[2]/(36*l[1]) - c[3,1]*l[2]*np.tan(g[0])/12 - c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[3,3]*l[0]/24 + c[3,3]*l[1]*np.tan(g[0])/24 - c[3,3]*l[2]*np.tan(g[1])/24 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[3,5]*l[2]/24 + c[3,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[4,1]*l[2]/24 + c[4,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[4,3]*l[1]/24 + c[4,3]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[4,5]*l[1]*l[2]/(36*l[0]),
           -c[2,2]*l[0]*l[1]/(36*l[2]) + c[2,2]*l[1]*np.tan(g[1])/12 - c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[2,3]*l[0]/24 + c[2,3]*l[1]*np.tan(g[0])/24 - c[2,3]*l[2]*np.tan(g[1])/24 - c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[2,4]*l[1]/24 + c[2,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[3,2]*l[0]/24 + c[3,2]*l[1]*np.tan(g[0])/24 - c[3,2]*l[2]*np.tan(g[1])/24 - c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,3]*l[0]*l[2]/(36*l[1]) - c[3,3]*l[2]*np.tan(g[0])/12 - c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[3,4]*l[2]/24 + c[3,4]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[4,2]*l[1]/24 + c[4,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[4,3]*l[2]/24 + c[4,3]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[4,4]*l[1]*l[2]/(36*l[0]),
           -c[2,0]*l[1]/12 + c[2,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[2,4]*l[0]*l[1]/(18*l[2]) - c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[2,5]*l[0]/24 + c[2,5]*l[1]*np.tan(g[0])/12 + c[2,5]*l[2]*np.tan(g[1])/12 - c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,0]*l[2]/12 + c[3,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[3,4]*l[0]/24 - c[3,4]*l[1]*np.tan(g[0])/12 - c[3,4]*l[2]*np.tan(g[1])/12 - c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,5]*l[0]*l[2]/(18*l[1]) - c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[4,0]*l[1]*l[2]/(9*l[0]) + c[4,4]*l[1]/12 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[4,5]*l[2]/12 + c[4,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           -c[2,1]*l[0]/24 + c[2,1]*l[1]*np.tan(g[0])/12 + c[2,1]*l[2]*np.tan(g[1])/12 - c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[2,3]*l[0]*l[1]/(18*l[2]) - c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[2,5]*l[1]/12 + c[2,5]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[3,1]*l[0]*l[2]/(18*l[1]) - c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[3,3]*l[0]/24 - c[3,3]*l[1]*np.tan(g[0])/12 - c[3,3]*l[2]*np.tan(g[1])/12 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,5]*l[2]/12 + c[3,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[4,1]*l[2]/12 + c[4,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[4,3]*l[1]/12 + c[4,3]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[4,5]*l[1]*l[2]/(9*l[0]),
           c[2,2]*l[0]*l[1]/(18*l[2]) - c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[2,3]*l[0]/24 + c[2,3]*l[1]*np.tan(g[0])/12 + c[2,3]*l[2]*np.tan(g[1])/12 - c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[2,4]*l[1]/12 + c[2,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[3,2]*l[0]/24 - c[3,2]*l[1]*np.tan(g[0])/12 - c[3,2]*l[2]*np.tan(g[1])/12 - c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,3]*l[0]*l[2]/(18*l[1]) - c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[3,4]*l[2]/12 + c[3,4]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[4,2]*l[1]/12 + c[4,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[4,3]*l[2]/12 + c[4,3]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[4,4]*l[1]*l[2]/(9*l[0]),
           (l[0]**2*(4*c[2,4]*l[1]**2 - 3*c[2,5]*l[1]*l[2] - 3*c[3,4]*l[1]*l[2] + 4*c[3,5]*l[2]**2) + 3*l[0]*l[1]*l[2]*(c[2,0]*l[1] - 2*c[2,4]*l[1]*np.tan(g[1]) - c[2,5]*l[1]*np.tan(g[0]) + c[2,5]*l[2]*np.tan(g[1]) - c[3,0]*l[2] - c[3,4]*l[1]*np.tan(g[0]) + c[3,4]*l[2]*np.tan(g[1]) + 2*c[3,5]*l[2]*np.tan(g[0]) + c[4,4]*l[1] - c[4,5]*l[2]) + 4*l[1]**2*l[2]**2*(-c[2,0]*np.tan(g[1]) + c[2,4]*np.tan(g[1])**2 + c[2,5]*np.tan(g[0])*np.tan(g[1]) - c[3,0]*np.tan(g[0]) + c[3,4]*np.tan(g[0])*np.tan(g[1]) + c[3,5]*np.tan(g[0])**2 + c[4,0] - c[4,4]*np.tan(g[1]) - c[4,5]*np.tan(g[0])))/(36*l[0]*l[1]*l[2]),
           (l[0]**2*(-3*c[2,1]*l[1]*l[2] + 4*c[2,3]*l[1]**2 + 4*c[3,1]*l[2]**2 - 3*c[3,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(-c[2,1]*l[1]*np.tan(g[0]) + c[2,1]*l[2]*np.tan(g[1]) - 2*c[2,3]*l[1]*np.tan(g[1]) + c[2,5]*l[1] + 2*c[3,1]*l[2]*np.tan(g[0]) - c[3,3]*l[1]*np.tan(g[0]) + c[3,3]*l[2]*np.tan(g[1]) - c[3,5]*l[2] - c[4,1]*l[2] + c[4,3]*l[1]) + 4*l[1]**2*l[2]**2*(c[2,1]*np.tan(g[0])*np.tan(g[1]) + c[2,3]*np.tan(g[1])**2 - c[2,5]*np.tan(g[1]) + c[3,1]*np.tan(g[0])**2 + c[3,3]*np.tan(g[0])*np.tan(g[1]) - c[3,5]*np.tan(g[0]) - c[4,1]*np.tan(g[0]) - c[4,3]*np.tan(g[1]) + c[4,5]))/(36*l[0]*l[1]*l[2]),
           (l[0]**2*(4*c[2,2]*l[1]**2 - 3*c[2,3]*l[1]*l[2] - 3*c[3,2]*l[1]*l[2] + 4*c[3,3]*l[2]**2) + 3*l[0]*l[1]*l[2]*(-2*c[2,2]*l[1]*np.tan(g[1]) - c[2,3]*l[1]*np.tan(g[0]) + c[2,3]*l[2]*np.tan(g[1]) + c[2,4]*l[1] - c[3,2]*l[1]*np.tan(g[0]) + c[3,2]*l[2]*np.tan(g[1]) + 2*c[3,3]*l[2]*np.tan(g[0]) - c[3,4]*l[2] + c[4,2]*l[1] - c[4,3]*l[2]) + 4*l[1]**2*l[2]**2*(c[2,2]*np.tan(g[1])**2 + c[2,3]*np.tan(g[0])*np.tan(g[1]) - c[2,4]*np.tan(g[1]) + c[3,2]*np.tan(g[0])*np.tan(g[1]) + c[3,3]*np.tan(g[0])**2 - c[3,4]*np.tan(g[0]) - c[4,2]*np.tan(g[1]) - c[4,3]*np.tan(g[0]) + c[4,4]))/(36*l[0]*l[1]*l[2]),
           (2*l[0]**2*(2*c[2,4]*l[1]**2 + 3*c[2,5]*l[1]*l[2] - 3*c[3,4]*l[1]*l[2] - 4*c[3,5]*l[2]**2) + 3*l[0]*l[1]*l[2]*(c[2,0]*l[1] - 2*c[2,4]*l[1]*np.tan(g[1]) - c[2,5]*l[1]*np.tan(g[0]) - 2*c[2,5]*l[2]*np.tan(g[1]) - 2*c[3,0]*l[2] - c[3,4]*l[1]*np.tan(g[0]) + 2*c[3,4]*l[2]*np.tan(g[1]) + c[4,4]*l[1] + 2*c[4,5]*l[2]) + 4*l[1]**2*l[2]**2*(-c[2,0]*np.tan(g[1]) + c[2,4]*np.tan(g[1])**2 + c[2,5]*np.tan(g[0])*np.tan(g[1]) - c[3,0]*np.tan(g[0]) + c[3,4]*np.tan(g[0])*np.tan(g[1]) + c[3,5]*np.tan(g[0])**2 + c[4,0] - c[4,4]*np.tan(g[1]) - c[4,5]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(3*c[2,1]*l[1]*l[2] + 2*c[2,3]*l[1]**2 - 4*c[3,1]*l[2]**2 - 3*c[3,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(-c[2,1]*l[1]*np.tan(g[0]) - 2*c[2,1]*l[2]*np.tan(g[1]) - 2*c[2,3]*l[1]*np.tan(g[1]) + c[2,5]*l[1] - c[3,3]*l[1]*np.tan(g[0]) + 2*c[3,3]*l[2]*np.tan(g[1]) - 2*c[3,5]*l[2] + 2*c[4,1]*l[2] + c[4,3]*l[1]) + 4*l[1]**2*l[2]**2*(c[2,1]*np.tan(g[0])*np.tan(g[1]) + c[2,3]*np.tan(g[1])**2 - c[2,5]*np.tan(g[1]) + c[3,1]*np.tan(g[0])**2 + c[3,3]*np.tan(g[0])*np.tan(g[1]) - c[3,5]*np.tan(g[0]) - c[4,1]*np.tan(g[0]) - c[4,3]*np.tan(g[1]) + c[4,5]))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(2*c[2,2]*l[1]**2 + 3*c[2,3]*l[1]*l[2] - 3*c[3,2]*l[1]*l[2] - 4*c[3,3]*l[2]**2) + 3*l[0]*l[1]*l[2]*(-2*c[2,2]*l[1]*np.tan(g[1]) - c[2,3]*l[1]*np.tan(g[0]) - 2*c[2,3]*l[2]*np.tan(g[1]) + c[2,4]*l[1] - c[3,2]*l[1]*np.tan(g[0]) + 2*c[3,2]*l[2]*np.tan(g[1]) - 2*c[3,4]*l[2] + c[4,2]*l[1] + 2*c[4,3]*l[2]) + 4*l[1]**2*l[2]**2*(c[2,2]*np.tan(g[1])**2 + c[2,3]*np.tan(g[0])*np.tan(g[1]) - c[2,4]*np.tan(g[1]) + c[3,2]*np.tan(g[0])*np.tan(g[1]) + c[3,3]*np.tan(g[0])**2 - c[3,4]*np.tan(g[0]) - c[4,2]*np.tan(g[1]) - c[4,3]*np.tan(g[0]) + c[4,4]))/(72*l[0]*l[1]*l[2]),
           -c[2,0]*l[1]/24 + c[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[2,4]*l[0]*l[1]/(36*l[2]) - c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[2,5]*l[0]/24 + c[2,5]*l[1]*np.tan(g[0])/24 - c[2,5]*l[2]*np.tan(g[1])/12 - c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,0]*l[2]/12 + c[3,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[3,4]*l[0]/24 - c[3,4]*l[1]*np.tan(g[0])/24 - c[3,4]*l[2]*np.tan(g[1])/12 - c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,5]*l[0]*l[2]/(18*l[1]) - c[3,5]*l[2]*np.tan(g[0])/6 - c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[4,0]*l[1]*l[2]/(18*l[0]) + c[4,4]*l[1]/24 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,5]*l[2]/12 + c[4,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[2,1]*l[0]/24 + c[2,1]*l[1]*np.tan(g[0])/24 - c[2,1]*l[2]*np.tan(g[1])/12 - c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[2,3]*l[0]*l[1]/(36*l[2]) - c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[2,5]*l[1]/24 + c[2,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[3,1]*l[0]*l[2]/(18*l[1]) - c[3,1]*l[2]*np.tan(g[0])/6 - c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[3,3]*l[0]/24 - c[3,3]*l[1]*np.tan(g[0])/24 - c[3,3]*l[2]*np.tan(g[1])/12 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,5]*l[2]/12 + c[3,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,1]*l[2]/12 + c[4,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,3]*l[1]/24 + c[4,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,5]*l[1]*l[2]/(18*l[0]),
           c[2,2]*l[0]*l[1]/(36*l[2]) - c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[2,3]*l[0]/24 + c[2,3]*l[1]*np.tan(g[0])/24 - c[2,3]*l[2]*np.tan(g[1])/12 - c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[2,4]*l[1]/24 + c[2,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[3,2]*l[0]/24 - c[3,2]*l[1]*np.tan(g[0])/24 - c[3,2]*l[2]*np.tan(g[1])/12 - c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,3]*l[0]*l[2]/(18*l[1]) - c[3,3]*l[2]*np.tan(g[0])/6 - c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[3,4]*l[2]/12 + c[3,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,2]*l[1]/24 + c[4,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,3]*l[2]/12 + c[4,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,4]*l[1]*l[2]/(18*l[0])],
          [-c[0,0]*l[1]*l[2]/(36*l[0]) - c[0,4]*l[1]/24 + c[0,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[0,5]*l[2]/24 + c[0,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[4,0]*l[1]/24 + c[4,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[4,4]*l[0]*l[1]/(36*l[2]) + c[4,4]*l[1]*np.tan(g[1])/12 - c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[4,5]*l[0]/24 + c[4,5]*l[1]*np.tan(g[0])/24 + c[4,5]*l[2]*np.tan(g[1])/24 - c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,0]*l[2]/24 + c[5,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[5,4]*l[0]/24 + c[5,4]*l[1]*np.tan(g[0])/24 + c[5,4]*l[2]*np.tan(g[1])/24 - c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,5]*l[0]*l[2]/(36*l[1]) + c[5,5]*l[2]*np.tan(g[0])/12 - c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]),
           -c[0,1]*l[2]/24 + c[0,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[0,3]*l[1]/24 + c[0,3]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[0,5]*l[1]*l[2]/(36*l[0]) - c[4,1]*l[0]/24 + c[4,1]*l[1]*np.tan(g[0])/24 + c[4,1]*l[2]*np.tan(g[1])/24 - c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[4,3]*l[0]*l[1]/(36*l[2]) + c[4,3]*l[1]*np.tan(g[1])/12 - c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[4,5]*l[1]/24 + c[4,5]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,1]*l[0]*l[2]/(36*l[1]) + c[5,1]*l[2]*np.tan(g[0])/12 - c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[5,3]*l[0]/24 + c[5,3]*l[1]*np.tan(g[0])/24 + c[5,3]*l[2]*np.tan(g[1])/24 - c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,5]*l[2]/24 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           -c[0,2]*l[1]/24 + c[0,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[0,3]*l[2]/24 + c[0,3]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[0,4]*l[1]*l[2]/(36*l[0]) - c[4,2]*l[0]*l[1]/(36*l[2]) + c[4,2]*l[1]*np.tan(g[1])/12 - c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[4,3]*l[0]/24 + c[4,3]*l[1]*np.tan(g[0])/24 + c[4,3]*l[2]*np.tan(g[1])/24 - c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[4,4]*l[1]/24 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,2]*l[0]/24 + c[5,2]*l[1]*np.tan(g[0])/24 + c[5,2]*l[2]*np.tan(g[1])/24 - c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,3]*l[0]*l[2]/(36*l[1]) + c[5,3]*l[2]*np.tan(g[0])/12 - c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[5,4]*l[2]/24 + c[5,4]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           (-2*l[0]**2*(2*c[4,4]*l[1]**2 + 3*c[4,5]*l[1]*l[2] + 3*c[5,4]*l[1]*l[2] + 2*c[5,5]*l[2]**2) + 3*l[0]*l[1]*l[2]*(-c[0,4]*l[1] - c[0,5]*l[2] + c[4,0]*l[1] - c[4,5]*l[1]*np.tan(g[0]) + c[4,5]*l[2]*np.tan(g[1]) + c[5,0]*l[2] + c[5,4]*l[1]*np.tan(g[0]) - c[5,4]*l[2]*np.tan(g[1])) + 2*l[1]**2*l[2]**2*(c[0,0] - c[0,4]*np.tan(g[1]) - c[0,5]*np.tan(g[0]) - c[4,0]*np.tan(g[1]) + c[4,4]*np.tan(g[1])**2 + c[4,5]*np.tan(g[0])*np.tan(g[1]) - c[5,0]*np.tan(g[0]) + c[5,4]*np.tan(g[0])*np.tan(g[1]) + c[5,5]*np.tan(g[0])**2))/(72*l[0]*l[1]*l[2]),
           (-2*l[0]**2*(3*c[4,1]*l[1]*l[2] + 2*c[4,3]*l[1]**2 + 2*c[5,1]*l[2]**2 + 3*c[5,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(-c[0,1]*l[2] - c[0,3]*l[1] - c[4,1]*l[1]*np.tan(g[0]) + c[4,1]*l[2]*np.tan(g[1]) + c[4,5]*l[1] + c[5,3]*l[1]*np.tan(g[0]) - c[5,3]*l[2]*np.tan(g[1]) + c[5,5]*l[2]) + 2*l[1]**2*l[2]**2*(-c[0,1]*np.tan(g[0]) - c[0,3]*np.tan(g[1]) + c[0,5] + c[4,1]*np.tan(g[0])*np.tan(g[1]) + c[4,3]*np.tan(g[1])**2 - c[4,5]*np.tan(g[1]) + c[5,1]*np.tan(g[0])**2 + c[5,3]*np.tan(g[0])*np.tan(g[1]) - c[5,5]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (-2*l[0]**2*(2*c[4,2]*l[1]**2 + 3*c[4,3]*l[1]*l[2] + 3*c[5,2]*l[1]*l[2] + 2*c[5,3]*l[2]**2) + 3*l[0]*l[1]*l[2]*(-c[0,2]*l[1] - c[0,3]*l[2] - c[4,3]*l[1]*np.tan(g[0]) + c[4,3]*l[2]*np.tan(g[1]) + c[4,4]*l[1] + c[5,2]*l[1]*np.tan(g[0]) - c[5,2]*l[2]*np.tan(g[1]) + c[5,4]*l[2]) + 2*l[1]**2*l[2]**2*(-c[0,2]*np.tan(g[1]) - c[0,3]*np.tan(g[0]) + c[0,4] + c[4,2]*np.tan(g[1])**2 + c[4,3]*np.tan(g[0])*np.tan(g[1]) - c[4,4]*np.tan(g[1]) + c[5,2]*np.tan(g[0])*np.tan(g[1]) + c[5,3]*np.tan(g[0])**2 - c[5,4]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(-4*c[4,4]*l[1]**2 + 3*c[4,5]*l[1]*l[2] - 3*c[5,4]*l[1]*l[2] + 2*c[5,5]*l[2]**2) + 3*l[0]*l[1]*l[2]*(-2*c[0,4]*l[1] + c[0,5]*l[2] + 2*c[4,0]*l[1] - 2*c[4,5]*l[1]*np.tan(g[0]) - c[4,5]*l[2]*np.tan(g[1]) + c[5,0]*l[2] + 2*c[5,4]*l[1]*np.tan(g[0]) - c[5,4]*l[2]*np.tan(g[1]) - 2*c[5,5]*l[2]*np.tan(g[0])) + 4*l[1]**2*l[2]**2*(c[0,0] - c[0,4]*np.tan(g[1]) - c[0,5]*np.tan(g[0]) - c[4,0]*np.tan(g[1]) + c[4,4]*np.tan(g[1])**2 + c[4,5]*np.tan(g[0])*np.tan(g[1]) - c[5,0]*np.tan(g[0]) + c[5,4]*np.tan(g[0])*np.tan(g[1]) + c[5,5]*np.tan(g[0])**2))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(3*c[4,1]*l[1]*l[2] - 4*c[4,3]*l[1]**2 + 2*c[5,1]*l[2]**2 - 3*c[5,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(c[0,1]*l[2] - 2*c[0,3]*l[1] - 2*c[4,1]*l[1]*np.tan(g[0]) - c[4,1]*l[2]*np.tan(g[1]) + 2*c[4,5]*l[1] - 2*c[5,1]*l[2]*np.tan(g[0]) + 2*c[5,3]*l[1]*np.tan(g[0]) - c[5,3]*l[2]*np.tan(g[1]) + c[5,5]*l[2]) + 4*l[1]**2*l[2]**2*(-c[0,1]*np.tan(g[0]) - c[0,3]*np.tan(g[1]) + c[0,5] + c[4,1]*np.tan(g[0])*np.tan(g[1]) + c[4,3]*np.tan(g[1])**2 - c[4,5]*np.tan(g[1]) + c[5,1]*np.tan(g[0])**2 + c[5,3]*np.tan(g[0])*np.tan(g[1]) - c[5,5]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(-4*c[4,2]*l[1]**2 + 3*c[4,3]*l[1]*l[2] - 3*c[5,2]*l[1]*l[2] + 2*c[5,3]*l[2]**2) + 3*l[0]*l[1]*l[2]*(-2*c[0,2]*l[1] + c[0,3]*l[2] - 2*c[4,3]*l[1]*np.tan(g[0]) - c[4,3]*l[2]*np.tan(g[1]) + 2*c[4,4]*l[1] + 2*c[5,2]*l[1]*np.tan(g[0]) - c[5,2]*l[2]*np.tan(g[1]) - 2*c[5,3]*l[2]*np.tan(g[0]) + c[5,4]*l[2]) + 4*l[1]**2*l[2]**2*(-c[0,2]*np.tan(g[1]) - c[0,3]*np.tan(g[0]) + c[0,4] + c[4,2]*np.tan(g[1])**2 + c[4,3]*np.tan(g[0])*np.tan(g[1]) - c[4,4]*np.tan(g[1]) + c[5,2]*np.tan(g[0])*np.tan(g[1]) + c[5,3]*np.tan(g[0])**2 - c[5,4]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           -c[0,0]*l[1]*l[2]/(18*l[0]) - c[0,4]*l[1]/12 + c[0,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[0,5]*l[2]/24 + c[0,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,0]*l[1]/12 + c[4,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,4]*l[0]*l[1]/(18*l[2]) + c[4,4]*l[1]*np.tan(g[1])/6 - c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[4,5]*l[0]/24 + c[4,5]*l[1]*np.tan(g[0])/12 - c[4,5]*l[2]*np.tan(g[1])/24 - c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,0]*l[2]/24 + c[5,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[5,4]*l[0]/24 + c[5,4]*l[1]*np.tan(g[0])/12 + c[5,4]*l[2]*np.tan(g[1])/24 - c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,5]*l[0]*l[2]/(36*l[1]) - c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]),
           c[0,1]*l[2]/24 + c[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[0,3]*l[1]/12 + c[0,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[0,5]*l[1]*l[2]/(18*l[0]) + c[4,1]*l[0]/24 + c[4,1]*l[1]*np.tan(g[0])/12 - c[4,1]*l[2]*np.tan(g[1])/24 - c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[4,3]*l[0]*l[1]/(18*l[2]) + c[4,3]*l[1]*np.tan(g[1])/6 - c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[4,5]*l[1]/12 + c[4,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,1]*l[0]*l[2]/(36*l[1]) - c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[5,3]*l[0]/24 + c[5,3]*l[1]*np.tan(g[0])/12 + c[5,3]*l[2]*np.tan(g[1])/24 - c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,5]*l[2]/24 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[0,2]*l[1]/12 + c[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[0,3]*l[2]/24 + c[0,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[0,4]*l[1]*l[2]/(18*l[0]) - c[4,2]*l[0]*l[1]/(18*l[2]) + c[4,2]*l[1]*np.tan(g[1])/6 - c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[4,3]*l[0]/24 + c[4,3]*l[1]*np.tan(g[0])/12 - c[4,3]*l[2]*np.tan(g[1])/24 - c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[4,4]*l[1]/12 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,2]*l[0]/24 + c[5,2]*l[1]*np.tan(g[0])/12 + c[5,2]*l[2]*np.tan(g[1])/24 - c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,3]*l[0]*l[2]/(36*l[1]) - c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[5,4]*l[2]/24 + c[5,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[0,0]*l[1]*l[2]/(18*l[0]) + c[0,4]*l[1]/24 + c[0,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[0,5]*l[2]/12 + c[0,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,0]*l[1]/24 + c[4,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,4]*l[0]*l[1]/(36*l[2]) - c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[4,5]*l[0]/24 + c[4,5]*l[1]*np.tan(g[0])/24 + c[4,5]*l[2]*np.tan(g[1])/12 - c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,0]*l[2]/12 + c[5,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[5,4]*l[0]/24 - c[5,4]*l[1]*np.tan(g[0])/24 + c[5,4]*l[2]*np.tan(g[1])/12 - c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,5]*l[0]*l[2]/(18*l[1]) + c[5,5]*l[2]*np.tan(g[0])/6 - c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]),
           -c[0,1]*l[2]/12 + c[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[0,3]*l[1]/24 + c[0,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[0,5]*l[1]*l[2]/(18*l[0]) - c[4,1]*l[0]/24 + c[4,1]*l[1]*np.tan(g[0])/24 + c[4,1]*l[2]*np.tan(g[1])/12 - c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[4,3]*l[0]*l[1]/(36*l[2]) - c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[4,5]*l[1]/24 + c[4,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,1]*l[0]*l[2]/(18*l[1]) + c[5,1]*l[2]*np.tan(g[0])/6 - c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[5,3]*l[0]/24 - c[5,3]*l[1]*np.tan(g[0])/24 + c[5,3]*l[2]*np.tan(g[1])/12 - c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,5]*l[2]/12 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[0,2]*l[1]/24 + c[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[0,3]*l[2]/12 + c[0,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[0,4]*l[1]*l[2]/(18*l[0]) + c[4,2]*l[0]*l[1]/(36*l[2]) - c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[4,3]*l[0]/24 + c[4,3]*l[1]*np.tan(g[0])/24 + c[4,3]*l[2]*np.tan(g[1])/12 - c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[4,4]*l[1]/24 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,2]*l[0]/24 - c[5,2]*l[1]*np.tan(g[0])/24 + c[5,2]*l[2]*np.tan(g[1])/12 - c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,3]*l[0]*l[2]/(18*l[1]) + c[5,3]*l[2]*np.tan(g[0])/6 - c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[5,4]*l[2]/12 + c[5,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           (2*l[0]**2*(2*c[4,4]*l[1]**2 - 3*c[4,5]*l[1]*l[2] + 3*c[5,4]*l[1]*l[2] - 4*c[5,5]*l[2]**2) + 3*l[0]*l[1]*l[2]*(c[0,4]*l[1] - 2*c[0,5]*l[2] + c[4,0]*l[1] - 2*c[4,4]*l[1]*np.tan(g[1]) - c[4,5]*l[1]*np.tan(g[0]) + 2*c[4,5]*l[2]*np.tan(g[1]) + 2*c[5,0]*l[2] - c[5,4]*l[1]*np.tan(g[0]) - 2*c[5,4]*l[2]*np.tan(g[1])) + 4*l[1]**2*l[2]**2*(c[0,0] - c[0,4]*np.tan(g[1]) - c[0,5]*np.tan(g[0]) - c[4,0]*np.tan(g[1]) + c[4,4]*np.tan(g[1])**2 + c[4,5]*np.tan(g[0])*np.tan(g[1]) - c[5,0]*np.tan(g[0]) + c[5,4]*np.tan(g[0])*np.tan(g[1]) + c[5,5]*np.tan(g[0])**2))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(-3*c[4,1]*l[1]*l[2] + 2*c[4,3]*l[1]**2 - 4*c[5,1]*l[2]**2 + 3*c[5,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(-2*c[0,1]*l[2] + c[0,3]*l[1] - c[4,1]*l[1]*np.tan(g[0]) + 2*c[4,1]*l[2]*np.tan(g[1]) - 2*c[4,3]*l[1]*np.tan(g[1]) + c[4,5]*l[1] - c[5,3]*l[1]*np.tan(g[0]) - 2*c[5,3]*l[2]*np.tan(g[1]) + 2*c[5,5]*l[2]) + 4*l[1]**2*l[2]**2*(-c[0,1]*np.tan(g[0]) - c[0,3]*np.tan(g[1]) + c[0,5] + c[4,1]*np.tan(g[0])*np.tan(g[1]) + c[4,3]*np.tan(g[1])**2 - c[4,5]*np.tan(g[1]) + c[5,1]*np.tan(g[0])**2 + c[5,3]*np.tan(g[0])*np.tan(g[1]) - c[5,5]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(2*c[4,2]*l[1]**2 - 3*c[4,3]*l[1]*l[2] + 3*c[5,2]*l[1]*l[2] - 4*c[5,3]*l[2]**2) + 3*l[0]*l[1]*l[2]*(c[0,2]*l[1] - 2*c[0,3]*l[2] - 2*c[4,2]*l[1]*np.tan(g[1]) - c[4,3]*l[1]*np.tan(g[0]) + 2*c[4,3]*l[2]*np.tan(g[1]) + c[4,4]*l[1] - c[5,2]*l[1]*np.tan(g[0]) - 2*c[5,2]*l[2]*np.tan(g[1]) + 2*c[5,4]*l[2]) + 4*l[1]**2*l[2]**2*(-c[0,2]*np.tan(g[1]) - c[0,3]*np.tan(g[0]) + c[0,4] + c[4,2]*np.tan(g[1])**2 + c[4,3]*np.tan(g[0])*np.tan(g[1]) - c[4,4]*np.tan(g[1]) + c[5,2]*np.tan(g[0])*np.tan(g[1]) + c[5,3]*np.tan(g[0])**2 - c[5,4]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (l[0]**2*(4*c[4,4]*l[1]**2 + 3*c[4,5]*l[1]*l[2] + 3*c[5,4]*l[1]*l[2] + 4*c[5,5]*l[2]**2) + 3*l[0]*l[1]*l[2]*(c[0,4]*l[1] + c[0,5]*l[2] + c[4,0]*l[1] - 2*c[4,4]*l[1]*np.tan(g[1]) - c[4,5]*l[1]*np.tan(g[0]) - c[4,5]*l[2]*np.tan(g[1]) + c[5,0]*l[2] - c[5,4]*l[1]*np.tan(g[0]) - c[5,4]*l[2]*np.tan(g[1]) - 2*c[5,5]*l[2]*np.tan(g[0])) + 4*l[1]**2*l[2]**2*(c[0,0] - c[0,4]*np.tan(g[1]) - c[0,5]*np.tan(g[0]) - c[4,0]*np.tan(g[1]) + c[4,4]*np.tan(g[1])**2 + c[4,5]*np.tan(g[0])*np.tan(g[1]) - c[5,0]*np.tan(g[0]) + c[5,4]*np.tan(g[0])*np.tan(g[1]) + c[5,5]*np.tan(g[0])**2))/(36*l[0]*l[1]*l[2]),
           (l[0]**2*(3*c[4,1]*l[1]*l[2] + 4*c[4,3]*l[1]**2 + 4*c[5,1]*l[2]**2 + 3*c[5,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(c[0,1]*l[2] + c[0,3]*l[1] - c[4,1]*l[1]*np.tan(g[0]) - c[4,1]*l[2]*np.tan(g[1]) - 2*c[4,3]*l[1]*np.tan(g[1]) + c[4,5]*l[1] - 2*c[5,1]*l[2]*np.tan(g[0]) - c[5,3]*l[1]*np.tan(g[0]) - c[5,3]*l[2]*np.tan(g[1]) + c[5,5]*l[2]) + 4*l[1]**2*l[2]**2*(-c[0,1]*np.tan(g[0]) - c[0,3]*np.tan(g[1]) + c[0,5] + c[4,1]*np.tan(g[0])*np.tan(g[1]) + c[4,3]*np.tan(g[1])**2 - c[4,5]*np.tan(g[1]) + c[5,1]*np.tan(g[0])**2 + c[5,3]*np.tan(g[0])*np.tan(g[1]) - c[5,5]*np.tan(g[0])))/(36*l[0]*l[1]*l[2]),
           (l[0]**2*(4*c[4,2]*l[1]**2 + 3*c[4,3]*l[1]*l[2] + 3*c[5,2]*l[1]*l[2] + 4*c[5,3]*l[2]**2) + 3*l[0]*l[1]*l[2]*(c[0,2]*l[1] + c[0,3]*l[2] - 2*c[4,2]*l[1]*np.tan(g[1]) - c[4,3]*l[1]*np.tan(g[0]) - c[4,3]*l[2]*np.tan(g[1]) + c[4,4]*l[1] - c[5,2]*l[1]*np.tan(g[0]) - c[5,2]*l[2]*np.tan(g[1]) - 2*c[5,3]*l[2]*np.tan(g[0]) + c[5,4]*l[2]) + 4*l[1]**2*l[2]**2*(-c[0,2]*np.tan(g[1]) - c[0,3]*np.tan(g[0]) + c[0,4] + c[4,2]*np.tan(g[1])**2 + c[4,3]*np.tan(g[0])*np.tan(g[1]) - c[4,4]*np.tan(g[1]) + c[5,2]*np.tan(g[0])*np.tan(g[1]) + c[5,3]*np.tan(g[0])**2 - c[5,4]*np.tan(g[0])))/(36*l[0]*l[1]*l[2]),
           -c[0,0]*l[1]*l[2]/(9*l[0]) + c[0,4]*l[1]/12 + c[0,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[0,5]*l[2]/12 + c[0,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[4,0]*l[1]/12 + c[4,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[4,4]*l[0]*l[1]/(18*l[2]) - c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[4,5]*l[0]/24 + c[4,5]*l[1]*np.tan(g[0])/12 - c[4,5]*l[2]*np.tan(g[1])/12 - c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[5,0]*l[2]/12 + c[5,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[5,4]*l[0]/24 - c[5,4]*l[1]*np.tan(g[0])/12 + c[5,4]*l[2]*np.tan(g[1])/12 - c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[5,5]*l[0]*l[2]/(18*l[1]) - c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]),
           c[0,1]*l[2]/12 + c[0,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[0,3]*l[1]/12 + c[0,3]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[0,5]*l[1]*l[2]/(9*l[0]) + c[4,1]*l[0]/24 + c[4,1]*l[1]*np.tan(g[0])/12 - c[4,1]*l[2]*np.tan(g[1])/12 - c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[4,3]*l[0]*l[1]/(18*l[2]) - c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[4,5]*l[1]/12 + c[4,5]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,1]*l[0]*l[2]/(18*l[1]) - c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[5,3]*l[0]/24 - c[5,3]*l[1]*np.tan(g[0])/12 + c[5,3]*l[2]*np.tan(g[1])/12 - c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[5,5]*l[2]/12 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           c[0,2]*l[1]/12 + c[0,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[0,3]*l[2]/12 + c[0,3]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[0,4]*l[1]*l[2]/(9*l[0]) + c[4,2]*l[0]*l[1]/(18*l[2]) - c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[4,3]*l[0]/24 + c[4,3]*l[1]*np.tan(g[0])/12 - c[4,3]*l[2]*np.tan(g[1])/12 - c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[4,4]*l[1]/12 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,2]*l[0]/24 - c[5,2]*l[1]*np.tan(g[0])/12 + c[5,2]*l[2]*np.tan(g[1])/12 - c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[5,3]*l[0]*l[2]/(18*l[1]) - c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[5,4]*l[2]/12 + c[5,4]*l[1]*l[2]*np.tan(g[0])/(9*l[0])],
          [-c[1,0]*l[2]/24 + c[1,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[1,4]*l[0]/24 + c[1,4]*l[1]*np.tan(g[0])/24 + c[1,4]*l[2]*np.tan(g[1])/24 - c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[1,5]*l[0]*l[2]/(36*l[1]) + c[1,5]*l[2]*np.tan(g[0])/12 - c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[3,0]*l[1]/24 + c[3,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[3,4]*l[0]*l[1]/(36*l[2]) + c[3,4]*l[1]*np.tan(g[1])/12 - c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[3,5]*l[0]/24 + c[3,5]*l[1]*np.tan(g[0])/24 + c[3,5]*l[2]*np.tan(g[1])/24 - c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,0]*l[1]*l[2]/(36*l[0]) - c[5,4]*l[1]/24 + c[5,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,5]*l[2]/24 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           -c[1,1]*l[0]*l[2]/(36*l[1]) + c[1,1]*l[2]*np.tan(g[0])/12 - c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[1,3]*l[0]/24 + c[1,3]*l[1]*np.tan(g[0])/24 + c[1,3]*l[2]*np.tan(g[1])/24 - c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[1,5]*l[2]/24 + c[1,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[3,1]*l[0]/24 + c[3,1]*l[1]*np.tan(g[0])/24 + c[3,1]*l[2]*np.tan(g[1])/24 - c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,3]*l[0]*l[1]/(36*l[2]) + c[3,3]*l[1]*np.tan(g[1])/12 - c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[3,5]*l[1]/24 + c[3,5]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,1]*l[2]/24 + c[5,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[5,3]*l[1]/24 + c[5,3]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,5]*l[1]*l[2]/(36*l[0]),
           -c[1,2]*l[0]/24 + c[1,2]*l[1]*np.tan(g[0])/24 + c[1,2]*l[2]*np.tan(g[1])/24 - c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[1,3]*l[0]*l[2]/(36*l[1]) + c[1,3]*l[2]*np.tan(g[0])/12 - c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[1,4]*l[2]/24 + c[1,4]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[3,2]*l[0]*l[1]/(36*l[2]) + c[3,2]*l[1]*np.tan(g[1])/12 - c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[3,3]*l[0]/24 + c[3,3]*l[1]*np.tan(g[0])/24 + c[3,3]*l[2]*np.tan(g[1])/24 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,4]*l[1]/24 + c[3,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,2]*l[1]/24 + c[5,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,3]*l[2]/24 + c[5,3]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[5,4]*l[1]*l[2]/(36*l[0]),
           (-2*l[0]**2*(3*c[1,4]*l[1]*l[2] + 2*c[1,5]*l[2]**2 + 2*c[3,4]*l[1]**2 + 3*c[3,5]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(c[1,0]*l[2] + c[1,4]*l[1]*np.tan(g[0]) - c[1,4]*l[2]*np.tan(g[1]) + c[3,0]*l[1] - c[3,5]*l[1]*np.tan(g[0]) + c[3,5]*l[2]*np.tan(g[1]) - c[5,4]*l[1] - c[5,5]*l[2]) + 2*l[1]**2*l[2]**2*(-c[1,0]*np.tan(g[0]) + c[1,4]*np.tan(g[0])*np.tan(g[1]) + c[1,5]*np.tan(g[0])**2 - c[3,0]*np.tan(g[1]) + c[3,4]*np.tan(g[1])**2 + c[3,5]*np.tan(g[0])*np.tan(g[1]) + c[5,0] - c[5,4]*np.tan(g[1]) - c[5,5]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (-2*l[0]**2*(2*c[1,1]*l[2]**2 + 3*c[1,3]*l[1]*l[2] + 3*c[3,1]*l[1]*l[2] + 2*c[3,3]*l[1]**2) + 3*l[0]*l[1]*l[2]*(c[1,3]*l[1]*np.tan(g[0]) - c[1,3]*l[2]*np.tan(g[1]) + c[1,5]*l[2] - c[3,1]*l[1]*np.tan(g[0]) + c[3,1]*l[2]*np.tan(g[1]) + c[3,5]*l[1] - c[5,1]*l[2] - c[5,3]*l[1]) + 2*l[1]**2*l[2]**2*(c[1,1]*np.tan(g[0])**2 + c[1,3]*np.tan(g[0])*np.tan(g[1]) - c[1,5]*np.tan(g[0]) + c[3,1]*np.tan(g[0])*np.tan(g[1]) + c[3,3]*np.tan(g[1])**2 - c[3,5]*np.tan(g[1]) - c[5,1]*np.tan(g[0]) - c[5,3]*np.tan(g[1]) + c[5,5]))/(72*l[0]*l[1]*l[2]),
           (-2*l[0]**2*(3*c[1,2]*l[1]*l[2] + 2*c[1,3]*l[2]**2 + 2*c[3,2]*l[1]**2 + 3*c[3,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(c[1,2]*l[1]*np.tan(g[0]) - c[1,2]*l[2]*np.tan(g[1]) + c[1,4]*l[2] - c[3,3]*l[1]*np.tan(g[0]) + c[3,3]*l[2]*np.tan(g[1]) + c[3,4]*l[1] - c[5,2]*l[1] - c[5,3]*l[2]) + 2*l[1]**2*l[2]**2*(c[1,2]*np.tan(g[0])*np.tan(g[1]) + c[1,3]*np.tan(g[0])**2 - c[1,4]*np.tan(g[0]) + c[3,2]*np.tan(g[1])**2 + c[3,3]*np.tan(g[0])*np.tan(g[1]) - c[3,4]*np.tan(g[1]) - c[5,2]*np.tan(g[1]) - c[5,3]*np.tan(g[0]) + c[5,4]))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(-3*c[1,4]*l[1]*l[2] + 2*c[1,5]*l[2]**2 - 4*c[3,4]*l[1]**2 + 3*c[3,5]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(c[1,0]*l[2] + 2*c[1,4]*l[1]*np.tan(g[0]) - c[1,4]*l[2]*np.tan(g[1]) - 2*c[1,5]*l[2]*np.tan(g[0]) + 2*c[3,0]*l[1] - 2*c[3,5]*l[1]*np.tan(g[0]) - c[3,5]*l[2]*np.tan(g[1]) - 2*c[5,4]*l[1] + c[5,5]*l[2]) + 4*l[1]**2*l[2]**2*(-c[1,0]*np.tan(g[0]) + c[1,4]*np.tan(g[0])*np.tan(g[1]) + c[1,5]*np.tan(g[0])**2 - c[3,0]*np.tan(g[1]) + c[3,4]*np.tan(g[1])**2 + c[3,5]*np.tan(g[0])*np.tan(g[1]) + c[5,0] - c[5,4]*np.tan(g[1]) - c[5,5]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(2*c[1,1]*l[2]**2 - 3*c[1,3]*l[1]*l[2] + 3*c[3,1]*l[1]*l[2] - 4*c[3,3]*l[1]**2) + 3*l[0]*l[1]*l[2]*(-2*c[1,1]*l[2]*np.tan(g[0]) + 2*c[1,3]*l[1]*np.tan(g[0]) - c[1,3]*l[2]*np.tan(g[1]) + c[1,5]*l[2] - 2*c[3,1]*l[1]*np.tan(g[0]) - c[3,1]*l[2]*np.tan(g[1]) + 2*c[3,5]*l[1] + c[5,1]*l[2] - 2*c[5,3]*l[1]) + 4*l[1]**2*l[2]**2*(c[1,1]*np.tan(g[0])**2 + c[1,3]*np.tan(g[0])*np.tan(g[1]) - c[1,5]*np.tan(g[0]) + c[3,1]*np.tan(g[0])*np.tan(g[1]) + c[3,3]*np.tan(g[1])**2 - c[3,5]*np.tan(g[1]) - c[5,1]*np.tan(g[0]) - c[5,3]*np.tan(g[1]) + c[5,5]))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(-3*c[1,2]*l[1]*l[2] + 2*c[1,3]*l[2]**2 - 4*c[3,2]*l[1]**2 + 3*c[3,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(2*c[1,2]*l[1]*np.tan(g[0]) - c[1,2]*l[2]*np.tan(g[1]) - 2*c[1,3]*l[2]*np.tan(g[0]) + c[1,4]*l[2] - 2*c[3,3]*l[1]*np.tan(g[0]) - c[3,3]*l[2]*np.tan(g[1]) + 2*c[3,4]*l[1] - 2*c[5,2]*l[1] + c[5,3]*l[2]) + 4*l[1]**2*l[2]**2*(c[1,2]*np.tan(g[0])*np.tan(g[1]) + c[1,3]*np.tan(g[0])**2 - c[1,4]*np.tan(g[0]) + c[3,2]*np.tan(g[1])**2 + c[3,3]*np.tan(g[0])*np.tan(g[1]) - c[3,4]*np.tan(g[1]) - c[5,2]*np.tan(g[1]) - c[5,3]*np.tan(g[0]) + c[5,4]))/(72*l[0]*l[1]*l[2]),
           -c[1,0]*l[2]/24 + c[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[1,4]*l[0]/24 + c[1,4]*l[1]*np.tan(g[0])/12 + c[1,4]*l[2]*np.tan(g[1])/24 - c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[1,5]*l[0]*l[2]/(36*l[1]) - c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[3,0]*l[1]/12 + c[3,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[3,4]*l[0]*l[1]/(18*l[2]) + c[3,4]*l[1]*np.tan(g[1])/6 - c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[3,5]*l[0]/24 + c[3,5]*l[1]*np.tan(g[0])/12 - c[3,5]*l[2]*np.tan(g[1])/24 - c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,0]*l[1]*l[2]/(18*l[0]) - c[5,4]*l[1]/12 + c[5,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,5]*l[2]/24 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[1,1]*l[0]*l[2]/(36*l[1]) - c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[1,3]*l[0]/24 + c[1,3]*l[1]*np.tan(g[0])/12 + c[1,3]*l[2]*np.tan(g[1])/24 - c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[1,5]*l[2]/24 + c[1,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[3,1]*l[0]/24 + c[3,1]*l[1]*np.tan(g[0])/12 - c[3,1]*l[2]*np.tan(g[1])/24 - c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,3]*l[0]*l[1]/(18*l[2]) + c[3,3]*l[1]*np.tan(g[1])/6 - c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[3,5]*l[1]/12 + c[3,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,1]*l[2]/24 + c[5,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[5,3]*l[1]/12 + c[5,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,5]*l[1]*l[2]/(18*l[0]),
           -c[1,2]*l[0]/24 + c[1,2]*l[1]*np.tan(g[0])/12 + c[1,2]*l[2]*np.tan(g[1])/24 - c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[1,3]*l[0]*l[2]/(36*l[1]) - c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[1,4]*l[2]/24 + c[1,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[3,2]*l[0]*l[1]/(18*l[2]) + c[3,2]*l[1]*np.tan(g[1])/6 - c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[3,3]*l[0]/24 + c[3,3]*l[1]*np.tan(g[0])/12 - c[3,3]*l[2]*np.tan(g[1])/24 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,4]*l[1]/12 + c[3,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,2]*l[1]/12 + c[5,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,3]*l[2]/24 + c[5,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[5,4]*l[1]*l[2]/(18*l[0]),
           -c[1,0]*l[2]/12 + c[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[1,4]*l[0]/24 - c[1,4]*l[1]*np.tan(g[0])/24 + c[1,4]*l[2]*np.tan(g[1])/12 - c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[1,5]*l[0]*l[2]/(18*l[1]) + c[1,5]*l[2]*np.tan(g[0])/6 - c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[3,0]*l[1]/24 + c[3,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[3,4]*l[0]*l[1]/(36*l[2]) - c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[3,5]*l[0]/24 + c[3,5]*l[1]*np.tan(g[0])/24 + c[3,5]*l[2]*np.tan(g[1])/12 - c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,0]*l[1]*l[2]/(18*l[0]) + c[5,4]*l[1]/24 + c[5,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,5]*l[2]/12 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[1,1]*l[0]*l[2]/(18*l[1]) + c[1,1]*l[2]*np.tan(g[0])/6 - c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[1,3]*l[0]/24 - c[1,3]*l[1]*np.tan(g[0])/24 + c[1,3]*l[2]*np.tan(g[1])/12 - c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[1,5]*l[2]/12 + c[1,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[3,1]*l[0]/24 + c[3,1]*l[1]*np.tan(g[0])/24 + c[3,1]*l[2]*np.tan(g[1])/12 - c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,3]*l[0]*l[1]/(36*l[2]) - c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[3,5]*l[1]/24 + c[3,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,1]*l[2]/12 + c[5,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[5,3]*l[1]/24 + c[5,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,5]*l[1]*l[2]/(18*l[0]),
           c[1,2]*l[0]/24 - c[1,2]*l[1]*np.tan(g[0])/24 + c[1,2]*l[2]*np.tan(g[1])/12 - c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[1,3]*l[0]*l[2]/(18*l[1]) + c[1,3]*l[2]*np.tan(g[0])/6 - c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[1,4]*l[2]/12 + c[1,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[3,2]*l[0]*l[1]/(36*l[2]) - c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[3,3]*l[0]/24 + c[3,3]*l[1]*np.tan(g[0])/24 + c[3,3]*l[2]*np.tan(g[1])/12 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,4]*l[1]/24 + c[3,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,2]*l[1]/24 + c[5,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,3]*l[2]/12 + c[5,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[5,4]*l[1]*l[2]/(18*l[0]),
           (2*l[0]**2*(3*c[1,4]*l[1]*l[2] - 4*c[1,5]*l[2]**2 + 2*c[3,4]*l[1]**2 - 3*c[3,5]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(2*c[1,0]*l[2] - c[1,4]*l[1]*np.tan(g[0]) - 2*c[1,4]*l[2]*np.tan(g[1]) + c[3,0]*l[1] - 2*c[3,4]*l[1]*np.tan(g[1]) - c[3,5]*l[1]*np.tan(g[0]) + 2*c[3,5]*l[2]*np.tan(g[1]) + c[5,4]*l[1] - 2*c[5,5]*l[2]) + 4*l[1]**2*l[2]**2*(-c[1,0]*np.tan(g[0]) + c[1,4]*np.tan(g[0])*np.tan(g[1]) + c[1,5]*np.tan(g[0])**2 - c[3,0]*np.tan(g[1]) + c[3,4]*np.tan(g[1])**2 + c[3,5]*np.tan(g[0])*np.tan(g[1]) + c[5,0] - c[5,4]*np.tan(g[1]) - c[5,5]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(-4*c[1,1]*l[2]**2 + 3*c[1,3]*l[1]*l[2] - 3*c[3,1]*l[1]*l[2] + 2*c[3,3]*l[1]**2) + 3*l[0]*l[1]*l[2]*(-c[1,3]*l[1]*np.tan(g[0]) - 2*c[1,3]*l[2]*np.tan(g[1]) + 2*c[1,5]*l[2] - c[3,1]*l[1]*np.tan(g[0]) + 2*c[3,1]*l[2]*np.tan(g[1]) - 2*c[3,3]*l[1]*np.tan(g[1]) + c[3,5]*l[1] - 2*c[5,1]*l[2] + c[5,3]*l[1]) + 4*l[1]**2*l[2]**2*(c[1,1]*np.tan(g[0])**2 + c[1,3]*np.tan(g[0])*np.tan(g[1]) - c[1,5]*np.tan(g[0]) + c[3,1]*np.tan(g[0])*np.tan(g[1]) + c[3,3]*np.tan(g[1])**2 - c[3,5]*np.tan(g[1]) - c[5,1]*np.tan(g[0]) - c[5,3]*np.tan(g[1]) + c[5,5]))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(3*c[1,2]*l[1]*l[2] - 4*c[1,3]*l[2]**2 + 2*c[3,2]*l[1]**2 - 3*c[3,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(-c[1,2]*l[1]*np.tan(g[0]) - 2*c[1,2]*l[2]*np.tan(g[1]) + 2*c[1,4]*l[2] - 2*c[3,2]*l[1]*np.tan(g[1]) - c[3,3]*l[1]*np.tan(g[0]) + 2*c[3,3]*l[2]*np.tan(g[1]) + c[3,4]*l[1] + c[5,2]*l[1] - 2*c[5,3]*l[2]) + 4*l[1]**2*l[2]**2*(c[1,2]*np.tan(g[0])*np.tan(g[1]) + c[1,3]*np.tan(g[0])**2 - c[1,4]*np.tan(g[0]) + c[3,2]*np.tan(g[1])**2 + c[3,3]*np.tan(g[0])*np.tan(g[1]) - c[3,4]*np.tan(g[1]) - c[5,2]*np.tan(g[1]) - c[5,3]*np.tan(g[0]) + c[5,4]))/(72*l[0]*l[1]*l[2]),
           (l[0]**2*(3*c[1,4]*l[1]*l[2] + 4*c[1,5]*l[2]**2 + 4*c[3,4]*l[1]**2 + 3*c[3,5]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(c[1,0]*l[2] - c[1,4]*l[1]*np.tan(g[0]) - c[1,4]*l[2]*np.tan(g[1]) - 2*c[1,5]*l[2]*np.tan(g[0]) + c[3,0]*l[1] - 2*c[3,4]*l[1]*np.tan(g[1]) - c[3,5]*l[1]*np.tan(g[0]) - c[3,5]*l[2]*np.tan(g[1]) + c[5,4]*l[1] + c[5,5]*l[2]) + 4*l[1]**2*l[2]**2*(-c[1,0]*np.tan(g[0]) + c[1,4]*np.tan(g[0])*np.tan(g[1]) + c[1,5]*np.tan(g[0])**2 - c[3,0]*np.tan(g[1]) + c[3,4]*np.tan(g[1])**2 + c[3,5]*np.tan(g[0])*np.tan(g[1]) + c[5,0] - c[5,4]*np.tan(g[1]) - c[5,5]*np.tan(g[0])))/(36*l[0]*l[1]*l[2]),
           (l[0]**2*(4*c[1,1]*l[2]**2 + 3*c[1,3]*l[1]*l[2] + 3*c[3,1]*l[1]*l[2] + 4*c[3,3]*l[1]**2) + 3*l[0]*l[1]*l[2]*(-2*c[1,1]*l[2]*np.tan(g[0]) - c[1,3]*l[1]*np.tan(g[0]) - c[1,3]*l[2]*np.tan(g[1]) + c[1,5]*l[2] - c[3,1]*l[1]*np.tan(g[0]) - c[3,1]*l[2]*np.tan(g[1]) - 2*c[3,3]*l[1]*np.tan(g[1]) + c[3,5]*l[1] + c[5,1]*l[2] + c[5,3]*l[1]) + 4*l[1]**2*l[2]**2*(c[1,1]*np.tan(g[0])**2 + c[1,3]*np.tan(g[0])*np.tan(g[1]) - c[1,5]*np.tan(g[0]) + c[3,1]*np.tan(g[0])*np.tan(g[1]) + c[3,3]*np.tan(g[1])**2 - c[3,5]*np.tan(g[1]) - c[5,1]*np.tan(g[0]) - c[5,3]*np.tan(g[1]) + c[5,5]))/(36*l[0]*l[1]*l[2]),
           (l[0]**2*(3*c[1,2]*l[1]*l[2] + 4*c[1,3]*l[2]**2 + 4*c[3,2]*l[1]**2 + 3*c[3,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(-c[1,2]*l[1]*np.tan(g[0]) - c[1,2]*l[2]*np.tan(g[1]) - 2*c[1,3]*l[2]*np.tan(g[0]) + c[1,4]*l[2] - 2*c[3,2]*l[1]*np.tan(g[1]) - c[3,3]*l[1]*np.tan(g[0]) - c[3,3]*l[2]*np.tan(g[1]) + c[3,4]*l[1] + c[5,2]*l[1] + c[5,3]*l[2]) + 4*l[1]**2*l[2]**2*(c[1,2]*np.tan(g[0])*np.tan(g[1]) + c[1,3]*np.tan(g[0])**2 - c[1,4]*np.tan(g[0]) + c[3,2]*np.tan(g[1])**2 + c[3,3]*np.tan(g[0])*np.tan(g[1]) - c[3,4]*np.tan(g[1]) - c[5,2]*np.tan(g[1]) - c[5,3]*np.tan(g[0]) + c[5,4]))/(36*l[0]*l[1]*l[2]),
           -c[1,0]*l[2]/12 + c[1,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[1,4]*l[0]/24 - c[1,4]*l[1]*np.tan(g[0])/12 + c[1,4]*l[2]*np.tan(g[1])/12 - c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[1,5]*l[0]*l[2]/(18*l[1]) - c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[3,0]*l[1]/12 + c[3,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[3,4]*l[0]*l[1]/(18*l[2]) - c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[3,5]*l[0]/24 + c[3,5]*l[1]*np.tan(g[0])/12 - c[3,5]*l[2]*np.tan(g[1])/12 - c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[5,0]*l[1]*l[2]/(9*l[0]) + c[5,4]*l[1]/12 + c[5,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,5]*l[2]/12 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           c[1,1]*l[0]*l[2]/(18*l[1]) - c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[1,3]*l[0]/24 - c[1,3]*l[1]*np.tan(g[0])/12 + c[1,3]*l[2]*np.tan(g[1])/12 - c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[1,5]*l[2]/12 + c[1,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[3,1]*l[0]/24 + c[3,1]*l[1]*np.tan(g[0])/12 - c[3,1]*l[2]*np.tan(g[1])/12 - c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,3]*l[0]*l[1]/(18*l[2]) - c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[3,5]*l[1]/12 + c[3,5]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,1]*l[2]/12 + c[5,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[5,3]*l[1]/12 + c[5,3]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[5,5]*l[1]*l[2]/(9*l[0]),
           c[1,2]*l[0]/24 - c[1,2]*l[1]*np.tan(g[0])/12 + c[1,2]*l[2]*np.tan(g[1])/12 - c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[1,3]*l[0]*l[2]/(18*l[1]) - c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[1,4]*l[2]/12 + c[1,4]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[3,2]*l[0]*l[1]/(18*l[2]) - c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[3,3]*l[0]/24 + c[3,3]*l[1]*np.tan(g[0])/12 - c[3,3]*l[2]*np.tan(g[1])/12 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[3,4]*l[1]/12 + c[3,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,2]*l[1]/12 + c[5,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,3]*l[2]/12 + c[5,3]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[5,4]*l[1]*l[2]/(9*l[0])],
          [-c[2,0]*l[1]/24 + c[2,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[2,4]*l[0]*l[1]/(36*l[2]) + c[2,4]*l[1]*np.tan(g[1])/12 - c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[2,5]*l[0]/24 + c[2,5]*l[1]*np.tan(g[0])/24 + c[2,5]*l[2]*np.tan(g[1])/24 - c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,0]*l[2]/24 + c[3,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[3,4]*l[0]/24 + c[3,4]*l[1]*np.tan(g[0])/24 + c[3,4]*l[2]*np.tan(g[1])/24 - c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,5]*l[0]*l[2]/(36*l[1]) + c[3,5]*l[2]*np.tan(g[0])/12 - c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[4,0]*l[1]*l[2]/(36*l[0]) - c[4,4]*l[1]/24 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[4,5]*l[2]/24 + c[4,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           -c[2,1]*l[0]/24 + c[2,1]*l[1]*np.tan(g[0])/24 + c[2,1]*l[2]*np.tan(g[1])/24 - c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[2,3]*l[0]*l[1]/(36*l[2]) + c[2,3]*l[1]*np.tan(g[1])/12 - c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[2,5]*l[1]/24 + c[2,5]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[3,1]*l[0]*l[2]/(36*l[1]) + c[3,1]*l[2]*np.tan(g[0])/12 - c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[3,3]*l[0]/24 + c[3,3]*l[1]*np.tan(g[0])/24 + c[3,3]*l[2]*np.tan(g[1])/24 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,5]*l[2]/24 + c[3,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[4,1]*l[2]/24 + c[4,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[4,3]*l[1]/24 + c[4,3]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[4,5]*l[1]*l[2]/(36*l[0]),
           -c[2,2]*l[0]*l[1]/(36*l[2]) + c[2,2]*l[1]*np.tan(g[1])/12 - c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[2,3]*l[0]/24 + c[2,3]*l[1]*np.tan(g[0])/24 + c[2,3]*l[2]*np.tan(g[1])/24 - c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[2,4]*l[1]/24 + c[2,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[3,2]*l[0]/24 + c[3,2]*l[1]*np.tan(g[0])/24 + c[3,2]*l[2]*np.tan(g[1])/24 - c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,3]*l[0]*l[2]/(36*l[1]) + c[3,3]*l[2]*np.tan(g[0])/12 - c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[3,4]*l[2]/24 + c[3,4]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[4,2]*l[1]/24 + c[4,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[4,3]*l[2]/24 + c[4,3]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[4,4]*l[1]*l[2]/(36*l[0]),
           (-2*l[0]**2*(2*c[2,4]*l[1]**2 + 3*c[2,5]*l[1]*l[2] + 3*c[3,4]*l[1]*l[2] + 2*c[3,5]*l[2]**2) + 3*l[0]*l[1]*l[2]*(c[2,0]*l[1] - c[2,5]*l[1]*np.tan(g[0]) + c[2,5]*l[2]*np.tan(g[1]) + c[3,0]*l[2] + c[3,4]*l[1]*np.tan(g[0]) - c[3,4]*l[2]*np.tan(g[1]) - c[4,4]*l[1] - c[4,5]*l[2]) + 2*l[1]**2*l[2]**2*(-c[2,0]*np.tan(g[1]) + c[2,4]*np.tan(g[1])**2 + c[2,5]*np.tan(g[0])*np.tan(g[1]) - c[3,0]*np.tan(g[0]) + c[3,4]*np.tan(g[0])*np.tan(g[1]) + c[3,5]*np.tan(g[0])**2 + c[4,0] - c[4,4]*np.tan(g[1]) - c[4,5]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (-2*l[0]**2*(3*c[2,1]*l[1]*l[2] + 2*c[2,3]*l[1]**2 + 2*c[3,1]*l[2]**2 + 3*c[3,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(-c[2,1]*l[1]*np.tan(g[0]) + c[2,1]*l[2]*np.tan(g[1]) + c[2,5]*l[1] + c[3,3]*l[1]*np.tan(g[0]) - c[3,3]*l[2]*np.tan(g[1]) + c[3,5]*l[2] - c[4,1]*l[2] - c[4,3]*l[1]) + 2*l[1]**2*l[2]**2*(c[2,1]*np.tan(g[0])*np.tan(g[1]) + c[2,3]*np.tan(g[1])**2 - c[2,5]*np.tan(g[1]) + c[3,1]*np.tan(g[0])**2 + c[3,3]*np.tan(g[0])*np.tan(g[1]) - c[3,5]*np.tan(g[0]) - c[4,1]*np.tan(g[0]) - c[4,3]*np.tan(g[1]) + c[4,5]))/(72*l[0]*l[1]*l[2]),
           (-2*l[0]**2*(2*c[2,2]*l[1]**2 + 3*c[2,3]*l[1]*l[2] + 3*c[3,2]*l[1]*l[2] + 2*c[3,3]*l[2]**2) + 3*l[0]*l[1]*l[2]*(-c[2,3]*l[1]*np.tan(g[0]) + c[2,3]*l[2]*np.tan(g[1]) + c[2,4]*l[1] + c[3,2]*l[1]*np.tan(g[0]) - c[3,2]*l[2]*np.tan(g[1]) + c[3,4]*l[2] - c[4,2]*l[1] - c[4,3]*l[2]) + 2*l[1]**2*l[2]**2*(c[2,2]*np.tan(g[1])**2 + c[2,3]*np.tan(g[0])*np.tan(g[1]) - c[2,4]*np.tan(g[1]) + c[3,2]*np.tan(g[0])*np.tan(g[1]) + c[3,3]*np.tan(g[0])**2 - c[3,4]*np.tan(g[0]) - c[4,2]*np.tan(g[1]) - c[4,3]*np.tan(g[0]) + c[4,4]))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(-4*c[2,4]*l[1]**2 + 3*c[2,5]*l[1]*l[2] - 3*c[3,4]*l[1]*l[2] + 2*c[3,5]*l[2]**2) + 3*l[0]*l[1]*l[2]*(2*c[2,0]*l[1] - 2*c[2,5]*l[1]*np.tan(g[0]) - c[2,5]*l[2]*np.tan(g[1]) + c[3,0]*l[2] + 2*c[3,4]*l[1]*np.tan(g[0]) - c[3,4]*l[2]*np.tan(g[1]) - 2*c[3,5]*l[2]*np.tan(g[0]) - 2*c[4,4]*l[1] + c[4,5]*l[2]) + 4*l[1]**2*l[2]**2*(-c[2,0]*np.tan(g[1]) + c[2,4]*np.tan(g[1])**2 + c[2,5]*np.tan(g[0])*np.tan(g[1]) - c[3,0]*np.tan(g[0]) + c[3,4]*np.tan(g[0])*np.tan(g[1]) + c[3,5]*np.tan(g[0])**2 + c[4,0] - c[4,4]*np.tan(g[1]) - c[4,5]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(3*c[2,1]*l[1]*l[2] - 4*c[2,3]*l[1]**2 + 2*c[3,1]*l[2]**2 - 3*c[3,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(-2*c[2,1]*l[1]*np.tan(g[0]) - c[2,1]*l[2]*np.tan(g[1]) + 2*c[2,5]*l[1] - 2*c[3,1]*l[2]*np.tan(g[0]) + 2*c[3,3]*l[1]*np.tan(g[0]) - c[3,3]*l[2]*np.tan(g[1]) + c[3,5]*l[2] + c[4,1]*l[2] - 2*c[4,3]*l[1]) + 4*l[1]**2*l[2]**2*(c[2,1]*np.tan(g[0])*np.tan(g[1]) + c[2,3]*np.tan(g[1])**2 - c[2,5]*np.tan(g[1]) + c[3,1]*np.tan(g[0])**2 + c[3,3]*np.tan(g[0])*np.tan(g[1]) - c[3,5]*np.tan(g[0]) - c[4,1]*np.tan(g[0]) - c[4,3]*np.tan(g[1]) + c[4,5]))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(-4*c[2,2]*l[1]**2 + 3*c[2,3]*l[1]*l[2] - 3*c[3,2]*l[1]*l[2] + 2*c[3,3]*l[2]**2) + 3*l[0]*l[1]*l[2]*(-2*c[2,3]*l[1]*np.tan(g[0]) - c[2,3]*l[2]*np.tan(g[1]) + 2*c[2,4]*l[1] + 2*c[3,2]*l[1]*np.tan(g[0]) - c[3,2]*l[2]*np.tan(g[1]) - 2*c[3,3]*l[2]*np.tan(g[0]) + c[3,4]*l[2] - 2*c[4,2]*l[1] + c[4,3]*l[2]) + 4*l[1]**2*l[2]**2*(c[2,2]*np.tan(g[1])**2 + c[2,3]*np.tan(g[0])*np.tan(g[1]) - c[2,4]*np.tan(g[1]) + c[3,2]*np.tan(g[0])*np.tan(g[1]) + c[3,3]*np.tan(g[0])**2 - c[3,4]*np.tan(g[0]) - c[4,2]*np.tan(g[1]) - c[4,3]*np.tan(g[0]) + c[4,4]))/(72*l[0]*l[1]*l[2]),
           -c[2,0]*l[1]/12 + c[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[2,4]*l[0]*l[1]/(18*l[2]) + c[2,4]*l[1]*np.tan(g[1])/6 - c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[2,5]*l[0]/24 + c[2,5]*l[1]*np.tan(g[0])/12 - c[2,5]*l[2]*np.tan(g[1])/24 - c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,0]*l[2]/24 + c[3,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[3,4]*l[0]/24 + c[3,4]*l[1]*np.tan(g[0])/12 + c[3,4]*l[2]*np.tan(g[1])/24 - c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,5]*l[0]*l[2]/(36*l[1]) - c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[4,0]*l[1]*l[2]/(18*l[0]) - c[4,4]*l[1]/12 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,5]*l[2]/24 + c[4,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[2,1]*l[0]/24 + c[2,1]*l[1]*np.tan(g[0])/12 - c[2,1]*l[2]*np.tan(g[1])/24 - c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[2,3]*l[0]*l[1]/(18*l[2]) + c[2,3]*l[1]*np.tan(g[1])/6 - c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[2,5]*l[1]/12 + c[2,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[3,1]*l[0]*l[2]/(36*l[1]) - c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[3,3]*l[0]/24 + c[3,3]*l[1]*np.tan(g[0])/12 + c[3,3]*l[2]*np.tan(g[1])/24 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,5]*l[2]/24 + c[3,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,1]*l[2]/24 + c[4,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,3]*l[1]/12 + c[4,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,5]*l[1]*l[2]/(18*l[0]),
           -c[2,2]*l[0]*l[1]/(18*l[2]) + c[2,2]*l[1]*np.tan(g[1])/6 - c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[2,3]*l[0]/24 + c[2,3]*l[1]*np.tan(g[0])/12 - c[2,3]*l[2]*np.tan(g[1])/24 - c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[2,4]*l[1]/12 + c[2,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[3,2]*l[0]/24 + c[3,2]*l[1]*np.tan(g[0])/12 + c[3,2]*l[2]*np.tan(g[1])/24 - c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,3]*l[0]*l[2]/(36*l[1]) - c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[3,4]*l[2]/24 + c[3,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,2]*l[1]/12 + c[4,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,3]*l[2]/24 + c[4,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,4]*l[1]*l[2]/(18*l[0]),
           -c[2,0]*l[1]/24 + c[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[2,4]*l[0]*l[1]/(36*l[2]) - c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[2,5]*l[0]/24 + c[2,5]*l[1]*np.tan(g[0])/24 + c[2,5]*l[2]*np.tan(g[1])/12 - c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,0]*l[2]/12 + c[3,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[3,4]*l[0]/24 - c[3,4]*l[1]*np.tan(g[0])/24 + c[3,4]*l[2]*np.tan(g[1])/12 - c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,5]*l[0]*l[2]/(18*l[1]) + c[3,5]*l[2]*np.tan(g[0])/6 - c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[4,0]*l[1]*l[2]/(18*l[0]) + c[4,4]*l[1]/24 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,5]*l[2]/12 + c[4,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[2,1]*l[0]/24 + c[2,1]*l[1]*np.tan(g[0])/24 + c[2,1]*l[2]*np.tan(g[1])/12 - c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[2,3]*l[0]*l[1]/(36*l[2]) - c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[2,5]*l[1]/24 + c[2,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[3,1]*l[0]*l[2]/(18*l[1]) + c[3,1]*l[2]*np.tan(g[0])/6 - c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[3,3]*l[0]/24 - c[3,3]*l[1]*np.tan(g[0])/24 + c[3,3]*l[2]*np.tan(g[1])/12 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,5]*l[2]/12 + c[3,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,1]*l[2]/12 + c[4,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,3]*l[1]/24 + c[4,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,5]*l[1]*l[2]/(18*l[0]),
           c[2,2]*l[0]*l[1]/(36*l[2]) - c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[2,3]*l[0]/24 + c[2,3]*l[1]*np.tan(g[0])/24 + c[2,3]*l[2]*np.tan(g[1])/12 - c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[2,4]*l[1]/24 + c[2,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[3,2]*l[0]/24 - c[3,2]*l[1]*np.tan(g[0])/24 + c[3,2]*l[2]*np.tan(g[1])/12 - c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,3]*l[0]*l[2]/(18*l[1]) + c[3,3]*l[2]*np.tan(g[0])/6 - c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[3,4]*l[2]/12 + c[3,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,2]*l[1]/24 + c[4,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,3]*l[2]/12 + c[4,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,4]*l[1]*l[2]/(18*l[0]),
           (2*l[0]**2*(2*c[2,4]*l[1]**2 - 3*c[2,5]*l[1]*l[2] + 3*c[3,4]*l[1]*l[2] - 4*c[3,5]*l[2]**2) + 3*l[0]*l[1]*l[2]*(c[2,0]*l[1] - 2*c[2,4]*l[1]*np.tan(g[1]) - c[2,5]*l[1]*np.tan(g[0]) + 2*c[2,5]*l[2]*np.tan(g[1]) + 2*c[3,0]*l[2] - c[3,4]*l[1]*np.tan(g[0]) - 2*c[3,4]*l[2]*np.tan(g[1]) + c[4,4]*l[1] - 2*c[4,5]*l[2]) + 4*l[1]**2*l[2]**2*(-c[2,0]*np.tan(g[1]) + c[2,4]*np.tan(g[1])**2 + c[2,5]*np.tan(g[0])*np.tan(g[1]) - c[3,0]*np.tan(g[0]) + c[3,4]*np.tan(g[0])*np.tan(g[1]) + c[3,5]*np.tan(g[0])**2 + c[4,0] - c[4,4]*np.tan(g[1]) - c[4,5]*np.tan(g[0])))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(-3*c[2,1]*l[1]*l[2] + 2*c[2,3]*l[1]**2 - 4*c[3,1]*l[2]**2 + 3*c[3,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(-c[2,1]*l[1]*np.tan(g[0]) + 2*c[2,1]*l[2]*np.tan(g[1]) - 2*c[2,3]*l[1]*np.tan(g[1]) + c[2,5]*l[1] - c[3,3]*l[1]*np.tan(g[0]) - 2*c[3,3]*l[2]*np.tan(g[1]) + 2*c[3,5]*l[2] - 2*c[4,1]*l[2] + c[4,3]*l[1]) + 4*l[1]**2*l[2]**2*(c[2,1]*np.tan(g[0])*np.tan(g[1]) + c[2,3]*np.tan(g[1])**2 - c[2,5]*np.tan(g[1]) + c[3,1]*np.tan(g[0])**2 + c[3,3]*np.tan(g[0])*np.tan(g[1]) - c[3,5]*np.tan(g[0]) - c[4,1]*np.tan(g[0]) - c[4,3]*np.tan(g[1]) + c[4,5]))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(2*c[2,2]*l[1]**2 - 3*c[2,3]*l[1]*l[2] + 3*c[3,2]*l[1]*l[2] - 4*c[3,3]*l[2]**2) + 3*l[0]*l[1]*l[2]*(-2*c[2,2]*l[1]*np.tan(g[1]) - c[2,3]*l[1]*np.tan(g[0]) + 2*c[2,3]*l[2]*np.tan(g[1]) + c[2,4]*l[1] - c[3,2]*l[1]*np.tan(g[0]) - 2*c[3,2]*l[2]*np.tan(g[1]) + 2*c[3,4]*l[2] + c[4,2]*l[1] - 2*c[4,3]*l[2]) + 4*l[1]**2*l[2]**2*(c[2,2]*np.tan(g[1])**2 + c[2,3]*np.tan(g[0])*np.tan(g[1]) - c[2,4]*np.tan(g[1]) + c[3,2]*np.tan(g[0])*np.tan(g[1]) + c[3,3]*np.tan(g[0])**2 - c[3,4]*np.tan(g[0]) - c[4,2]*np.tan(g[1]) - c[4,3]*np.tan(g[0]) + c[4,4]))/(72*l[0]*l[1]*l[2]),
           (l[0]**2*(4*c[2,4]*l[1]**2 + 3*c[2,5]*l[1]*l[2] + 3*c[3,4]*l[1]*l[2] + 4*c[3,5]*l[2]**2) + 3*l[0]*l[1]*l[2]*(c[2,0]*l[1] - 2*c[2,4]*l[1]*np.tan(g[1]) - c[2,5]*l[1]*np.tan(g[0]) - c[2,5]*l[2]*np.tan(g[1]) + c[3,0]*l[2] - c[3,4]*l[1]*np.tan(g[0]) - c[3,4]*l[2]*np.tan(g[1]) - 2*c[3,5]*l[2]*np.tan(g[0]) + c[4,4]*l[1] + c[4,5]*l[2]) + 4*l[1]**2*l[2]**2*(-c[2,0]*np.tan(g[1]) + c[2,4]*np.tan(g[1])**2 + c[2,5]*np.tan(g[0])*np.tan(g[1]) - c[3,0]*np.tan(g[0]) + c[3,4]*np.tan(g[0])*np.tan(g[1]) + c[3,5]*np.tan(g[0])**2 + c[4,0] - c[4,4]*np.tan(g[1]) - c[4,5]*np.tan(g[0])))/(36*l[0]*l[1]*l[2]),
           (l[0]**2*(3*c[2,1]*l[1]*l[2] + 4*c[2,3]*l[1]**2 + 4*c[3,1]*l[2]**2 + 3*c[3,3]*l[1]*l[2]) + 3*l[0]*l[1]*l[2]*(-c[2,1]*l[1]*np.tan(g[0]) - c[2,1]*l[2]*np.tan(g[1]) - 2*c[2,3]*l[1]*np.tan(g[1]) + c[2,5]*l[1] - 2*c[3,1]*l[2]*np.tan(g[0]) - c[3,3]*l[1]*np.tan(g[0]) - c[3,3]*l[2]*np.tan(g[1]) + c[3,5]*l[2] + c[4,1]*l[2] + c[4,3]*l[1]) + 4*l[1]**2*l[2]**2*(c[2,1]*np.tan(g[0])*np.tan(g[1]) + c[2,3]*np.tan(g[1])**2 - c[2,5]*np.tan(g[1]) + c[3,1]*np.tan(g[0])**2 + c[3,3]*np.tan(g[0])*np.tan(g[1]) - c[3,5]*np.tan(g[0]) - c[4,1]*np.tan(g[0]) - c[4,3]*np.tan(g[1]) + c[4,5]))/(36*l[0]*l[1]*l[2]),
           (l[0]**2*(4*c[2,2]*l[1]**2 + 3*c[2,3]*l[1]*l[2] + 3*c[3,2]*l[1]*l[2] + 4*c[3,3]*l[2]**2) + 3*l[0]*l[1]*l[2]*(-2*c[2,2]*l[1]*np.tan(g[1]) - c[2,3]*l[1]*np.tan(g[0]) - c[2,3]*l[2]*np.tan(g[1]) + c[2,4]*l[1] - c[3,2]*l[1]*np.tan(g[0]) - c[3,2]*l[2]*np.tan(g[1]) - 2*c[3,3]*l[2]*np.tan(g[0]) + c[3,4]*l[2] + c[4,2]*l[1] + c[4,3]*l[2]) + 4*l[1]**2*l[2]**2*(c[2,2]*np.tan(g[1])**2 + c[2,3]*np.tan(g[0])*np.tan(g[1]) - c[2,4]*np.tan(g[1]) + c[3,2]*np.tan(g[0])*np.tan(g[1]) + c[3,3]*np.tan(g[0])**2 - c[3,4]*np.tan(g[0]) - c[4,2]*np.tan(g[1]) - c[4,3]*np.tan(g[0]) + c[4,4]))/(36*l[0]*l[1]*l[2]),
           -c[2,0]*l[1]/12 + c[2,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[2,4]*l[0]*l[1]/(18*l[2]) - c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[2,5]*l[0]/24 + c[2,5]*l[1]*np.tan(g[0])/12 - c[2,5]*l[2]*np.tan(g[1])/12 - c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[3,0]*l[2]/12 + c[3,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[3,4]*l[0]/24 - c[3,4]*l[1]*np.tan(g[0])/12 + c[3,4]*l[2]*np.tan(g[1])/12 - c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,5]*l[0]*l[2]/(18*l[1]) - c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[4,0]*l[1]*l[2]/(9*l[0]) + c[4,4]*l[1]/12 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[4,5]*l[2]/12 + c[4,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           c[2,1]*l[0]/24 + c[2,1]*l[1]*np.tan(g[0])/12 - c[2,1]*l[2]*np.tan(g[1])/12 - c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[2,3]*l[0]*l[1]/(18*l[2]) - c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[2,5]*l[1]/12 + c[2,5]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[3,1]*l[0]*l[2]/(18*l[1]) - c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[3,3]*l[0]/24 - c[3,3]*l[1]*np.tan(g[0])/12 + c[3,3]*l[2]*np.tan(g[1])/12 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[3,5]*l[2]/12 + c[3,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[4,1]*l[2]/12 + c[4,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[4,3]*l[1]/12 + c[4,3]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[4,5]*l[1]*l[2]/(9*l[0]),
           c[2,2]*l[0]*l[1]/(18*l[2]) - c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[2,3]*l[0]/24 + c[2,3]*l[1]*np.tan(g[0])/12 - c[2,3]*l[2]*np.tan(g[1])/12 - c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[2,4]*l[1]/12 + c[2,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[3,2]*l[0]/24 - c[3,2]*l[1]*np.tan(g[0])/12 + c[3,2]*l[2]*np.tan(g[1])/12 - c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,3]*l[0]*l[2]/(18*l[1]) - c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[3,4]*l[2]/12 + c[3,4]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[4,2]*l[1]/12 + c[4,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[4,3]*l[2]/12 + c[4,3]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[4,4]*l[1]*l[2]/(9*l[0])],
          [c[0,0]*l[1]*l[2]/(36*l[0]) + c[0,4]*l[1]/24 - c[0,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[0,5]*l[2]/24 - c[0,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[4,0]*l[1]/24 - c[4,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[4,4]*l[0]*l[1]/(18*l[2]) + c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[4,5]*l[0]/12 + c[4,5]*l[1]*np.tan(g[0])/24 - c[4,5]*l[2]*np.tan(g[1])/24 + c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,0]*l[2]/24 - c[5,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[5,4]*l[0]/12 - c[5,4]*l[1]*np.tan(g[0])/24 + c[5,4]*l[2]*np.tan(g[1])/24 + c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,5]*l[0]*l[2]/(18*l[1]) + c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]),
           c[0,1]*l[2]/24 - c[0,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[0,3]*l[1]/24 - c[0,3]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[0,5]*l[1]*l[2]/(36*l[0]) - c[4,1]*l[0]/12 + c[4,1]*l[1]*np.tan(g[0])/24 - c[4,1]*l[2]*np.tan(g[1])/24 + c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[4,3]*l[0]*l[1]/(18*l[2]) + c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[4,5]*l[1]/24 - c[4,5]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,1]*l[0]*l[2]/(18*l[1]) + c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[5,3]*l[0]/12 - c[5,3]*l[1]*np.tan(g[0])/24 + c[5,3]*l[2]*np.tan(g[1])/24 + c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,5]*l[2]/24 - c[5,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           c[0,2]*l[1]/24 - c[0,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[0,3]*l[2]/24 - c[0,3]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[0,4]*l[1]*l[2]/(36*l[0]) - c[4,2]*l[0]*l[1]/(18*l[2]) + c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[4,3]*l[0]/12 + c[4,3]*l[1]*np.tan(g[0])/24 - c[4,3]*l[2]*np.tan(g[1])/24 + c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[4,4]*l[1]/24 - c[4,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,2]*l[0]/12 - c[5,2]*l[1]*np.tan(g[0])/24 + c[5,2]*l[2]*np.tan(g[1])/24 + c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,3]*l[0]*l[2]/(18*l[1]) + c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[5,4]*l[2]/24 - c[5,4]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           -c[0,0]*l[1]*l[2]/(36*l[0]) + c[0,4]*l[1]/24 + c[0,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[0,5]*l[2]/24 + c[0,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[4,0]*l[1]/24 + c[4,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[4,4]*l[0]*l[1]/(36*l[2]) - c[4,4]*l[1]*np.tan(g[1])/12 - c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[4,5]*l[0]/24 - c[4,5]*l[1]*np.tan(g[0])/24 - c[4,5]*l[2]*np.tan(g[1])/24 - c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[5,0]*l[2]/24 + c[5,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[5,4]*l[0]/24 - c[5,4]*l[1]*np.tan(g[0])/24 - c[5,4]*l[2]*np.tan(g[1])/24 - c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,5]*l[0]*l[2]/(36*l[1]) - c[5,5]*l[2]*np.tan(g[0])/12 - c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]),
           c[0,1]*l[2]/24 + c[0,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[0,3]*l[1]/24 + c[0,3]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[0,5]*l[1]*l[2]/(36*l[0]) - c[4,1]*l[0]/24 - c[4,1]*l[1]*np.tan(g[0])/24 - c[4,1]*l[2]*np.tan(g[1])/24 - c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[4,3]*l[0]*l[1]/(36*l[2]) - c[4,3]*l[1]*np.tan(g[1])/12 - c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[4,5]*l[1]/24 + c[4,5]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,1]*l[0]*l[2]/(36*l[1]) - c[5,1]*l[2]*np.tan(g[0])/12 - c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[5,3]*l[0]/24 - c[5,3]*l[1]*np.tan(g[0])/24 - c[5,3]*l[2]*np.tan(g[1])/24 - c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[5,5]*l[2]/24 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           c[0,2]*l[1]/24 + c[0,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[0,3]*l[2]/24 + c[0,3]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[0,4]*l[1]*l[2]/(36*l[0]) - c[4,2]*l[0]*l[1]/(36*l[2]) - c[4,2]*l[1]*np.tan(g[1])/12 - c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[4,3]*l[0]/24 - c[4,3]*l[1]*np.tan(g[0])/24 - c[4,3]*l[2]*np.tan(g[1])/24 - c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[4,4]*l[1]/24 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,2]*l[0]/24 - c[5,2]*l[1]*np.tan(g[0])/24 - c[5,2]*l[2]*np.tan(g[1])/24 - c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,3]*l[0]*l[2]/(36*l[1]) - c[5,3]*l[2]*np.tan(g[0])/12 - c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[5,4]*l[2]/24 + c[5,4]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           -c[0,0]*l[1]*l[2]/(18*l[0]) + c[0,4]*l[1]/12 + c[0,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[0,5]*l[2]/24 + c[0,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,0]*l[1]/12 + c[4,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,4]*l[0]*l[1]/(18*l[2]) - c[4,4]*l[1]*np.tan(g[1])/6 - c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[4,5]*l[0]/24 - c[4,5]*l[1]*np.tan(g[0])/12 + c[4,5]*l[2]*np.tan(g[1])/24 - c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,0]*l[2]/24 + c[5,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[5,4]*l[0]/24 - c[5,4]*l[1]*np.tan(g[0])/12 - c[5,4]*l[2]*np.tan(g[1])/24 - c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,5]*l[0]*l[2]/(36*l[1]) - c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]),
           -c[0,1]*l[2]/24 + c[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[0,3]*l[1]/12 + c[0,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[0,5]*l[1]*l[2]/(18*l[0]) + c[4,1]*l[0]/24 - c[4,1]*l[1]*np.tan(g[0])/12 + c[4,1]*l[2]*np.tan(g[1])/24 - c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[4,3]*l[0]*l[1]/(18*l[2]) - c[4,3]*l[1]*np.tan(g[1])/6 - c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[4,5]*l[1]/12 + c[4,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,1]*l[0]*l[2]/(36*l[1]) - c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[5,3]*l[0]/24 - c[5,3]*l[1]*np.tan(g[0])/12 - c[5,3]*l[2]*np.tan(g[1])/24 - c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,5]*l[2]/24 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[0,2]*l[1]/12 + c[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[0,3]*l[2]/24 + c[0,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[0,4]*l[1]*l[2]/(18*l[0]) - c[4,2]*l[0]*l[1]/(18*l[2]) - c[4,2]*l[1]*np.tan(g[1])/6 - c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[4,3]*l[0]/24 - c[4,3]*l[1]*np.tan(g[0])/12 + c[4,3]*l[2]*np.tan(g[1])/24 - c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[4,4]*l[1]/12 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,2]*l[0]/24 - c[5,2]*l[1]*np.tan(g[0])/12 - c[5,2]*l[2]*np.tan(g[1])/24 - c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,3]*l[0]*l[2]/(36*l[1]) - c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[5,4]*l[2]/24 + c[5,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[0,0]*l[1]*l[2]/(18*l[0]) + c[0,4]*l[1]/12 - c[0,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[0,5]*l[2]/24 - c[0,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,0]*l[1]/12 - c[4,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,4]*l[0]*l[1]/(9*l[2]) + c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[4,5]*l[0]/12 + c[4,5]*l[1]*np.tan(g[0])/12 + c[4,5]*l[2]*np.tan(g[1])/24 + c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,0]*l[2]/24 - c[5,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[5,4]*l[0]/12 - c[5,4]*l[1]*np.tan(g[0])/12 + c[5,4]*l[2]*np.tan(g[1])/24 + c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,5]*l[0]*l[2]/(18*l[1]) + c[5,5]*l[2]*np.tan(g[0])/12 + c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]),
           -c[0,1]*l[2]/24 - c[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[0,3]*l[1]/12 - c[0,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[0,5]*l[1]*l[2]/(18*l[0]) + c[4,1]*l[0]/12 + c[4,1]*l[1]*np.tan(g[0])/12 + c[4,1]*l[2]*np.tan(g[1])/24 + c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[4,3]*l[0]*l[1]/(9*l[2]) + c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[4,5]*l[1]/12 - c[4,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,1]*l[0]*l[2]/(18*l[1]) + c[5,1]*l[2]*np.tan(g[0])/12 + c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[5,3]*l[0]/12 - c[5,3]*l[1]*np.tan(g[0])/12 + c[5,3]*l[2]*np.tan(g[1])/24 + c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,5]*l[2]/24 - c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[0,2]*l[1]/12 - c[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[0,3]*l[2]/24 - c[0,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[0,4]*l[1]*l[2]/(18*l[0]) - c[4,2]*l[0]*l[1]/(9*l[2]) + c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[4,3]*l[0]/12 + c[4,3]*l[1]*np.tan(g[0])/12 + c[4,3]*l[2]*np.tan(g[1])/24 + c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[4,4]*l[1]/12 - c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,2]*l[0]/12 - c[5,2]*l[1]*np.tan(g[0])/12 + c[5,2]*l[2]*np.tan(g[1])/24 + c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,3]*l[0]*l[2]/(18*l[1]) + c[5,3]*l[2]*np.tan(g[0])/12 + c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[5,4]*l[2]/24 - c[5,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[0,0]*l[1]*l[2]/(18*l[0]) - c[0,4]*l[1]/24 - c[0,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[0,5]*l[2]/12 - c[0,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,0]*l[1]/24 - c[4,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,4]*l[0]*l[1]/(18*l[2]) + c[4,4]*l[1]*np.tan(g[1])/12 + c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[4,5]*l[0]/12 + c[4,5]*l[1]*np.tan(g[0])/24 - c[4,5]*l[2]*np.tan(g[1])/12 + c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,0]*l[2]/12 - c[5,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[5,4]*l[0]/12 + c[5,4]*l[1]*np.tan(g[0])/24 + c[5,4]*l[2]*np.tan(g[1])/12 + c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,5]*l[0]*l[2]/(9*l[1]) + c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]),
           c[0,1]*l[2]/12 - c[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[0,3]*l[1]/24 - c[0,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[0,5]*l[1]*l[2]/(18*l[0]) - c[4,1]*l[0]/12 + c[4,1]*l[1]*np.tan(g[0])/24 - c[4,1]*l[2]*np.tan(g[1])/12 + c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[4,3]*l[0]*l[1]/(18*l[2]) + c[4,3]*l[1]*np.tan(g[1])/12 + c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[4,5]*l[1]/24 - c[4,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,1]*l[0]*l[2]/(9*l[1]) + c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[5,3]*l[0]/12 + c[5,3]*l[1]*np.tan(g[0])/24 + c[5,3]*l[2]*np.tan(g[1])/12 + c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,5]*l[2]/12 - c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[0,2]*l[1]/24 - c[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[0,3]*l[2]/12 - c[0,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[0,4]*l[1]*l[2]/(18*l[0]) + c[4,2]*l[0]*l[1]/(18*l[2]) + c[4,2]*l[1]*np.tan(g[1])/12 + c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[4,3]*l[0]/12 + c[4,3]*l[1]*np.tan(g[0])/24 - c[4,3]*l[2]*np.tan(g[1])/12 + c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[4,4]*l[1]/24 - c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,2]*l[0]/12 + c[5,2]*l[1]*np.tan(g[0])/24 + c[5,2]*l[2]*np.tan(g[1])/12 + c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,3]*l[0]*l[2]/(9*l[1]) + c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[5,4]*l[2]/12 - c[5,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[0,0]*l[1]*l[2]/(18*l[0]) - c[0,4]*l[1]/24 + c[0,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[0,5]*l[2]/12 + c[0,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,0]*l[1]/24 + c[4,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,4]*l[0]*l[1]/(36*l[2]) - c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[4,5]*l[0]/24 - c[4,5]*l[1]*np.tan(g[0])/24 - c[4,5]*l[2]*np.tan(g[1])/12 - c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,0]*l[2]/12 + c[5,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[5,4]*l[0]/24 + c[5,4]*l[1]*np.tan(g[0])/24 - c[5,4]*l[2]*np.tan(g[1])/12 - c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,5]*l[0]*l[2]/(18*l[1]) - c[5,5]*l[2]*np.tan(g[0])/6 - c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]),
           c[0,1]*l[2]/12 + c[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[0,3]*l[1]/24 + c[0,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[0,5]*l[1]*l[2]/(18*l[0]) - c[4,1]*l[0]/24 - c[4,1]*l[1]*np.tan(g[0])/24 - c[4,1]*l[2]*np.tan(g[1])/12 - c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[4,3]*l[0]*l[1]/(36*l[2]) - c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[4,5]*l[1]/24 + c[4,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,1]*l[0]*l[2]/(18*l[1]) - c[5,1]*l[2]*np.tan(g[0])/6 - c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[5,3]*l[0]/24 + c[5,3]*l[1]*np.tan(g[0])/24 - c[5,3]*l[2]*np.tan(g[1])/12 - c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,5]*l[2]/12 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[0,2]*l[1]/24 + c[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[0,3]*l[2]/12 + c[0,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[0,4]*l[1]*l[2]/(18*l[0]) + c[4,2]*l[0]*l[1]/(36*l[2]) - c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[4,3]*l[0]/24 - c[4,3]*l[1]*np.tan(g[0])/24 - c[4,3]*l[2]*np.tan(g[1])/12 - c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[4,4]*l[1]/24 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,2]*l[0]/24 + c[5,2]*l[1]*np.tan(g[0])/24 - c[5,2]*l[2]*np.tan(g[1])/12 - c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,3]*l[0]*l[2]/(18*l[1]) - c[5,3]*l[2]*np.tan(g[0])/6 - c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[5,4]*l[2]/12 + c[5,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[0,0]*l[1]*l[2]/(9*l[0]) - c[0,4]*l[1]/12 + c[0,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[0,5]*l[2]/12 + c[0,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[4,0]*l[1]/12 + c[4,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[4,4]*l[0]*l[1]/(18*l[2]) - c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[4,5]*l[0]/24 - c[4,5]*l[1]*np.tan(g[0])/12 + c[4,5]*l[2]*np.tan(g[1])/12 - c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[5,0]*l[2]/12 + c[5,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[5,4]*l[0]/24 + c[5,4]*l[1]*np.tan(g[0])/12 - c[5,4]*l[2]*np.tan(g[1])/12 - c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[5,5]*l[0]*l[2]/(18*l[1]) - c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]),
           -c[0,1]*l[2]/12 + c[0,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[0,3]*l[1]/12 + c[0,3]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[0,5]*l[1]*l[2]/(9*l[0]) + c[4,1]*l[0]/24 - c[4,1]*l[1]*np.tan(g[0])/12 + c[4,1]*l[2]*np.tan(g[1])/12 - c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[4,3]*l[0]*l[1]/(18*l[2]) - c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[4,5]*l[1]/12 + c[4,5]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,1]*l[0]*l[2]/(18*l[1]) - c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[5,3]*l[0]/24 + c[5,3]*l[1]*np.tan(g[0])/12 - c[5,3]*l[2]*np.tan(g[1])/12 - c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[5,5]*l[2]/12 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           -c[0,2]*l[1]/12 + c[0,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[0,3]*l[2]/12 + c[0,3]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[0,4]*l[1]*l[2]/(9*l[0]) + c[4,2]*l[0]*l[1]/(18*l[2]) - c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[4,3]*l[0]/24 - c[4,3]*l[1]*np.tan(g[0])/12 + c[4,3]*l[2]*np.tan(g[1])/12 - c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[4,4]*l[1]/12 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,2]*l[0]/24 + c[5,2]*l[1]*np.tan(g[0])/12 - c[5,2]*l[2]*np.tan(g[1])/12 - c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[5,3]*l[0]*l[2]/(18*l[1]) - c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[5,4]*l[2]/12 + c[5,4]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           c[0,0]*l[1]*l[2]/(9*l[0]) - c[0,4]*l[1]/12 - c[0,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[0,5]*l[2]/12 - c[0,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[4,0]*l[1]/12 - c[4,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[4,4]*l[0]*l[1]/(9*l[2]) + c[4,4]*l[1]*np.tan(g[1])/6 + c[4,4]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[4,5]*l[0]/12 + c[4,5]*l[1]*np.tan(g[0])/12 + c[4,5]*l[2]*np.tan(g[1])/12 + c[4,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[5,0]*l[2]/12 - c[5,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[5,4]*l[0]/12 + c[5,4]*l[1]*np.tan(g[0])/12 + c[5,4]*l[2]*np.tan(g[1])/12 + c[5,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[5,5]*l[0]*l[2]/(9*l[1]) + c[5,5]*l[2]*np.tan(g[0])/6 + c[5,5]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]),
           -c[0,1]*l[2]/12 - c[0,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[0,3]*l[1]/12 - c[0,3]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[0,5]*l[1]*l[2]/(9*l[0]) + c[4,1]*l[0]/12 + c[4,1]*l[1]*np.tan(g[0])/12 + c[4,1]*l[2]*np.tan(g[1])/12 + c[4,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[4,3]*l[0]*l[1]/(9*l[2]) + c[4,3]*l[1]*np.tan(g[1])/6 + c[4,3]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[4,5]*l[1]/12 - c[4,5]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,1]*l[0]*l[2]/(9*l[1]) + c[5,1]*l[2]*np.tan(g[0])/6 + c[5,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[5,3]*l[0]/12 + c[5,3]*l[1]*np.tan(g[0])/12 + c[5,3]*l[2]*np.tan(g[1])/12 + c[5,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[5,5]*l[2]/12 - c[5,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           -c[0,2]*l[1]/12 - c[0,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[0,3]*l[2]/12 - c[0,3]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[0,4]*l[1]*l[2]/(9*l[0]) + c[4,2]*l[0]*l[1]/(9*l[2]) + c[4,2]*l[1]*np.tan(g[1])/6 + c[4,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[4,3]*l[0]/12 + c[4,3]*l[1]*np.tan(g[0])/12 + c[4,3]*l[2]*np.tan(g[1])/12 + c[4,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[4,4]*l[1]/12 - c[4,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,2]*l[0]/12 + c[5,2]*l[1]*np.tan(g[0])/12 + c[5,2]*l[2]*np.tan(g[1])/12 + c[5,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[5,3]*l[0]*l[2]/(9*l[1]) + c[5,3]*l[2]*np.tan(g[0])/6 + c[5,3]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[5,4]*l[2]/12 - c[5,4]*l[1]*l[2]*np.tan(g[0])/(9*l[0])],
          [-c[1,0]*l[2]/24 - c[1,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[1,4]*l[0]/12 - c[1,4]*l[1]*np.tan(g[0])/24 + c[1,4]*l[2]*np.tan(g[1])/24 + c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[1,5]*l[0]*l[2]/(18*l[1]) + c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[3,0]*l[1]/24 - c[3,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[3,4]*l[0]*l[1]/(18*l[2]) + c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[3,5]*l[0]/12 + c[3,5]*l[1]*np.tan(g[0])/24 - c[3,5]*l[2]*np.tan(g[1])/24 + c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[5,0]*l[1]*l[2]/(36*l[0]) + c[5,4]*l[1]/24 - c[5,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[5,5]*l[2]/24 - c[5,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           -c[1,1]*l[0]*l[2]/(18*l[1]) + c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[1,3]*l[0]/12 - c[1,3]*l[1]*np.tan(g[0])/24 + c[1,3]*l[2]*np.tan(g[1])/24 + c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[1,5]*l[2]/24 - c[1,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[3,1]*l[0]/12 + c[3,1]*l[1]*np.tan(g[0])/24 - c[3,1]*l[2]*np.tan(g[1])/24 + c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,3]*l[0]*l[1]/(18*l[2]) + c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[3,5]*l[1]/24 - c[3,5]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[5,1]*l[2]/24 - c[5,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[5,3]*l[1]/24 - c[5,3]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[5,5]*l[1]*l[2]/(36*l[0]),
           -c[1,2]*l[0]/12 - c[1,2]*l[1]*np.tan(g[0])/24 + c[1,2]*l[2]*np.tan(g[1])/24 + c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[1,3]*l[0]*l[2]/(18*l[1]) + c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[1,4]*l[2]/24 - c[1,4]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[3,2]*l[0]*l[1]/(18*l[2]) + c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[3,3]*l[0]/12 + c[3,3]*l[1]*np.tan(g[0])/24 - c[3,3]*l[2]*np.tan(g[1])/24 + c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,4]*l[1]/24 - c[3,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[5,2]*l[1]/24 - c[5,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[5,3]*l[2]/24 - c[5,3]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[5,4]*l[1]*l[2]/(36*l[0]),
           c[1,0]*l[2]/24 + c[1,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[1,4]*l[0]/24 - c[1,4]*l[1]*np.tan(g[0])/24 - c[1,4]*l[2]*np.tan(g[1])/24 - c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[1,5]*l[0]*l[2]/(36*l[1]) - c[1,5]*l[2]*np.tan(g[0])/12 - c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[3,0]*l[1]/24 + c[3,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[3,4]*l[0]*l[1]/(36*l[2]) - c[3,4]*l[1]*np.tan(g[1])/12 - c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[3,5]*l[0]/24 - c[3,5]*l[1]*np.tan(g[0])/24 - c[3,5]*l[2]*np.tan(g[1])/24 - c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[5,0]*l[1]*l[2]/(36*l[0]) + c[5,4]*l[1]/24 + c[5,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[5,5]*l[2]/24 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           -c[1,1]*l[0]*l[2]/(36*l[1]) - c[1,1]*l[2]*np.tan(g[0])/12 - c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[1,3]*l[0]/24 - c[1,3]*l[1]*np.tan(g[0])/24 - c[1,3]*l[2]*np.tan(g[1])/24 - c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[1,5]*l[2]/24 + c[1,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[3,1]*l[0]/24 - c[3,1]*l[1]*np.tan(g[0])/24 - c[3,1]*l[2]*np.tan(g[1])/24 - c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,3]*l[0]*l[1]/(36*l[2]) - c[3,3]*l[1]*np.tan(g[1])/12 - c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[3,5]*l[1]/24 + c[3,5]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[5,1]*l[2]/24 + c[5,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[5,3]*l[1]/24 + c[5,3]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[5,5]*l[1]*l[2]/(36*l[0]),
           -c[1,2]*l[0]/24 - c[1,2]*l[1]*np.tan(g[0])/24 - c[1,2]*l[2]*np.tan(g[1])/24 - c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[1,3]*l[0]*l[2]/(36*l[1]) - c[1,3]*l[2]*np.tan(g[0])/12 - c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[1,4]*l[2]/24 + c[1,4]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[3,2]*l[0]*l[1]/(36*l[2]) - c[3,2]*l[1]*np.tan(g[1])/12 - c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[3,3]*l[0]/24 - c[3,3]*l[1]*np.tan(g[0])/24 - c[3,3]*l[2]*np.tan(g[1])/24 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[3,4]*l[1]/24 + c[3,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[5,2]*l[1]/24 + c[5,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[5,3]*l[2]/24 + c[5,3]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[5,4]*l[1]*l[2]/(36*l[0]),
           c[1,0]*l[2]/24 + c[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[1,4]*l[0]/24 - c[1,4]*l[1]*np.tan(g[0])/12 - c[1,4]*l[2]*np.tan(g[1])/24 - c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[1,5]*l[0]*l[2]/(36*l[1]) - c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[3,0]*l[1]/12 + c[3,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[3,4]*l[0]*l[1]/(18*l[2]) - c[3,4]*l[1]*np.tan(g[1])/6 - c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[3,5]*l[0]/24 - c[3,5]*l[1]*np.tan(g[0])/12 + c[3,5]*l[2]*np.tan(g[1])/24 - c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,0]*l[1]*l[2]/(18*l[0]) + c[5,4]*l[1]/12 + c[5,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,5]*l[2]/24 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[1,1]*l[0]*l[2]/(36*l[1]) - c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[1,3]*l[0]/24 - c[1,3]*l[1]*np.tan(g[0])/12 - c[1,3]*l[2]*np.tan(g[1])/24 - c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[1,5]*l[2]/24 + c[1,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[3,1]*l[0]/24 - c[3,1]*l[1]*np.tan(g[0])/12 + c[3,1]*l[2]*np.tan(g[1])/24 - c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,3]*l[0]*l[1]/(18*l[2]) - c[3,3]*l[1]*np.tan(g[1])/6 - c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[3,5]*l[1]/12 + c[3,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,1]*l[2]/24 + c[5,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[5,3]*l[1]/12 + c[5,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,5]*l[1]*l[2]/(18*l[0]),
           -c[1,2]*l[0]/24 - c[1,2]*l[1]*np.tan(g[0])/12 - c[1,2]*l[2]*np.tan(g[1])/24 - c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[1,3]*l[0]*l[2]/(36*l[1]) - c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[1,4]*l[2]/24 + c[1,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[3,2]*l[0]*l[1]/(18*l[2]) - c[3,2]*l[1]*np.tan(g[1])/6 - c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[3,3]*l[0]/24 - c[3,3]*l[1]*np.tan(g[0])/12 + c[3,3]*l[2]*np.tan(g[1])/24 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,4]*l[1]/12 + c[3,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,2]*l[1]/12 + c[5,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,3]*l[2]/24 + c[5,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[5,4]*l[1]*l[2]/(18*l[0]),
           -c[1,0]*l[2]/24 - c[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[1,4]*l[0]/12 - c[1,4]*l[1]*np.tan(g[0])/12 + c[1,4]*l[2]*np.tan(g[1])/24 + c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[1,5]*l[0]*l[2]/(18*l[1]) + c[1,5]*l[2]*np.tan(g[0])/12 + c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[3,0]*l[1]/12 - c[3,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[3,4]*l[0]*l[1]/(9*l[2]) + c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[3,5]*l[0]/12 + c[3,5]*l[1]*np.tan(g[0])/12 + c[3,5]*l[2]*np.tan(g[1])/24 + c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,0]*l[1]*l[2]/(18*l[0]) + c[5,4]*l[1]/12 - c[5,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,5]*l[2]/24 - c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[1,1]*l[0]*l[2]/(18*l[1]) + c[1,1]*l[2]*np.tan(g[0])/12 + c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[1,3]*l[0]/12 - c[1,3]*l[1]*np.tan(g[0])/12 + c[1,3]*l[2]*np.tan(g[1])/24 + c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[1,5]*l[2]/24 - c[1,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[3,1]*l[0]/12 + c[3,1]*l[1]*np.tan(g[0])/12 + c[3,1]*l[2]*np.tan(g[1])/24 + c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,3]*l[0]*l[1]/(9*l[2]) + c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[3,5]*l[1]/12 - c[3,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,1]*l[2]/24 - c[5,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[5,3]*l[1]/12 - c[5,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,5]*l[1]*l[2]/(18*l[0]),
           -c[1,2]*l[0]/12 - c[1,2]*l[1]*np.tan(g[0])/12 + c[1,2]*l[2]*np.tan(g[1])/24 + c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[1,3]*l[0]*l[2]/(18*l[1]) + c[1,3]*l[2]*np.tan(g[0])/12 + c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[1,4]*l[2]/24 - c[1,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[3,2]*l[0]*l[1]/(9*l[2]) + c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[3,3]*l[0]/12 + c[3,3]*l[1]*np.tan(g[0])/12 + c[3,3]*l[2]*np.tan(g[1])/24 + c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,4]*l[1]/12 - c[3,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,2]*l[1]/12 - c[5,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,3]*l[2]/24 - c[5,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[5,4]*l[1]*l[2]/(18*l[0]),
           -c[1,0]*l[2]/12 - c[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[1,4]*l[0]/12 + c[1,4]*l[1]*np.tan(g[0])/24 + c[1,4]*l[2]*np.tan(g[1])/12 + c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[1,5]*l[0]*l[2]/(9*l[1]) + c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[3,0]*l[1]/24 - c[3,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[3,4]*l[0]*l[1]/(18*l[2]) + c[3,4]*l[1]*np.tan(g[1])/12 + c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[3,5]*l[0]/12 + c[3,5]*l[1]*np.tan(g[0])/24 - c[3,5]*l[2]*np.tan(g[1])/12 + c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[5,0]*l[1]*l[2]/(18*l[0]) - c[5,4]*l[1]/24 - c[5,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,5]*l[2]/12 - c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[1,1]*l[0]*l[2]/(9*l[1]) + c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[1,3]*l[0]/12 + c[1,3]*l[1]*np.tan(g[0])/24 + c[1,3]*l[2]*np.tan(g[1])/12 + c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[1,5]*l[2]/12 - c[1,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[3,1]*l[0]/12 + c[3,1]*l[1]*np.tan(g[0])/24 - c[3,1]*l[2]*np.tan(g[1])/12 + c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,3]*l[0]*l[1]/(18*l[2]) + c[3,3]*l[1]*np.tan(g[1])/12 + c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[3,5]*l[1]/24 - c[3,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,1]*l[2]/12 - c[5,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[5,3]*l[1]/24 - c[5,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,5]*l[1]*l[2]/(18*l[0]),
           c[1,2]*l[0]/12 + c[1,2]*l[1]*np.tan(g[0])/24 + c[1,2]*l[2]*np.tan(g[1])/12 + c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[1,3]*l[0]*l[2]/(9*l[1]) + c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[1,4]*l[2]/12 - c[1,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[3,2]*l[0]*l[1]/(18*l[2]) + c[3,2]*l[1]*np.tan(g[1])/12 + c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[3,3]*l[0]/12 + c[3,3]*l[1]*np.tan(g[0])/24 - c[3,3]*l[2]*np.tan(g[1])/12 + c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,4]*l[1]/24 - c[3,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,2]*l[1]/24 - c[5,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,3]*l[2]/12 - c[5,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[5,4]*l[1]*l[2]/(18*l[0]),
           c[1,0]*l[2]/12 + c[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[1,4]*l[0]/24 + c[1,4]*l[1]*np.tan(g[0])/24 - c[1,4]*l[2]*np.tan(g[1])/12 - c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[1,5]*l[0]*l[2]/(18*l[1]) - c[1,5]*l[2]*np.tan(g[0])/6 - c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[3,0]*l[1]/24 + c[3,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[3,4]*l[0]*l[1]/(36*l[2]) - c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[3,5]*l[0]/24 - c[3,5]*l[1]*np.tan(g[0])/24 - c[3,5]*l[2]*np.tan(g[1])/12 - c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[5,0]*l[1]*l[2]/(18*l[0]) - c[5,4]*l[1]/24 + c[5,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,5]*l[2]/12 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[1,1]*l[0]*l[2]/(18*l[1]) - c[1,1]*l[2]*np.tan(g[0])/6 - c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[1,3]*l[0]/24 + c[1,3]*l[1]*np.tan(g[0])/24 - c[1,3]*l[2]*np.tan(g[1])/12 - c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[1,5]*l[2]/12 + c[1,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[3,1]*l[0]/24 - c[3,1]*l[1]*np.tan(g[0])/24 - c[3,1]*l[2]*np.tan(g[1])/12 - c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,3]*l[0]*l[1]/(36*l[2]) - c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[3,5]*l[1]/24 + c[3,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,1]*l[2]/12 + c[5,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[5,3]*l[1]/24 + c[5,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,5]*l[1]*l[2]/(18*l[0]),
           c[1,2]*l[0]/24 + c[1,2]*l[1]*np.tan(g[0])/24 - c[1,2]*l[2]*np.tan(g[1])/12 - c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[1,3]*l[0]*l[2]/(18*l[1]) - c[1,3]*l[2]*np.tan(g[0])/6 - c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[1,4]*l[2]/12 + c[1,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[3,2]*l[0]*l[1]/(36*l[2]) - c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[3,3]*l[0]/24 - c[3,3]*l[1]*np.tan(g[0])/24 - c[3,3]*l[2]*np.tan(g[1])/12 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,4]*l[1]/24 + c[3,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[5,2]*l[1]/24 + c[5,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[5,3]*l[2]/12 + c[5,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[5,4]*l[1]*l[2]/(18*l[0]),
           c[1,0]*l[2]/12 + c[1,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[1,4]*l[0]/24 + c[1,4]*l[1]*np.tan(g[0])/12 - c[1,4]*l[2]*np.tan(g[1])/12 - c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[1,5]*l[0]*l[2]/(18*l[1]) - c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[3,0]*l[1]/12 + c[3,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[3,4]*l[0]*l[1]/(18*l[2]) - c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[3,5]*l[0]/24 - c[3,5]*l[1]*np.tan(g[0])/12 + c[3,5]*l[2]*np.tan(g[1])/12 - c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[5,0]*l[1]*l[2]/(9*l[0]) - c[5,4]*l[1]/12 + c[5,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[5,5]*l[2]/12 + c[5,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           c[1,1]*l[0]*l[2]/(18*l[1]) - c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[1,3]*l[0]/24 + c[1,3]*l[1]*np.tan(g[0])/12 - c[1,3]*l[2]*np.tan(g[1])/12 - c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[1,5]*l[2]/12 + c[1,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[3,1]*l[0]/24 - c[3,1]*l[1]*np.tan(g[0])/12 + c[3,1]*l[2]*np.tan(g[1])/12 - c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,3]*l[0]*l[1]/(18*l[2]) - c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[3,5]*l[1]/12 + c[3,5]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[5,1]*l[2]/12 + c[5,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[5,3]*l[1]/12 + c[5,3]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[5,5]*l[1]*l[2]/(9*l[0]),
           c[1,2]*l[0]/24 + c[1,2]*l[1]*np.tan(g[0])/12 - c[1,2]*l[2]*np.tan(g[1])/12 - c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[1,3]*l[0]*l[2]/(18*l[1]) - c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[1,4]*l[2]/12 + c[1,4]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[3,2]*l[0]*l[1]/(18*l[2]) - c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[3,3]*l[0]/24 - c[3,3]*l[1]*np.tan(g[0])/12 + c[3,3]*l[2]*np.tan(g[1])/12 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,4]*l[1]/12 + c[3,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[5,2]*l[1]/12 + c[5,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[5,3]*l[2]/12 + c[5,3]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[5,4]*l[1]*l[2]/(9*l[0]),
           -c[1,0]*l[2]/12 - c[1,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[1,4]*l[0]/12 + c[1,4]*l[1]*np.tan(g[0])/12 + c[1,4]*l[2]*np.tan(g[1])/12 + c[1,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[1,5]*l[0]*l[2]/(9*l[1]) + c[1,5]*l[2]*np.tan(g[0])/6 + c[1,5]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[3,0]*l[1]/12 - c[3,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[3,4]*l[0]*l[1]/(9*l[2]) + c[3,4]*l[1]*np.tan(g[1])/6 + c[3,4]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[3,5]*l[0]/12 + c[3,5]*l[1]*np.tan(g[0])/12 + c[3,5]*l[2]*np.tan(g[1])/12 + c[3,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[5,0]*l[1]*l[2]/(9*l[0]) - c[5,4]*l[1]/12 - c[5,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[5,5]*l[2]/12 - c[5,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           c[1,1]*l[0]*l[2]/(9*l[1]) + c[1,1]*l[2]*np.tan(g[0])/6 + c[1,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[1,3]*l[0]/12 + c[1,3]*l[1]*np.tan(g[0])/12 + c[1,3]*l[2]*np.tan(g[1])/12 + c[1,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[1,5]*l[2]/12 - c[1,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[3,1]*l[0]/12 + c[3,1]*l[1]*np.tan(g[0])/12 + c[3,1]*l[2]*np.tan(g[1])/12 + c[3,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,3]*l[0]*l[1]/(9*l[2]) + c[3,3]*l[1]*np.tan(g[1])/6 + c[3,3]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[3,5]*l[1]/12 - c[3,5]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[5,1]*l[2]/12 - c[5,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[5,3]*l[1]/12 - c[5,3]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[5,5]*l[1]*l[2]/(9*l[0]),
           c[1,2]*l[0]/12 + c[1,2]*l[1]*np.tan(g[0])/12 + c[1,2]*l[2]*np.tan(g[1])/12 + c[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[1,3]*l[0]*l[2]/(9*l[1]) + c[1,3]*l[2]*np.tan(g[0])/6 + c[1,3]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[1,4]*l[2]/12 - c[1,4]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[3,2]*l[0]*l[1]/(9*l[2]) + c[3,2]*l[1]*np.tan(g[1])/6 + c[3,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[3,3]*l[0]/12 + c[3,3]*l[1]*np.tan(g[0])/12 + c[3,3]*l[2]*np.tan(g[1])/12 + c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[3,4]*l[1]/12 - c[3,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[5,2]*l[1]/12 - c[5,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[5,3]*l[2]/12 - c[5,3]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[5,4]*l[1]*l[2]/(9*l[0])],
          [-c[2,0]*l[1]/24 - c[2,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[2,4]*l[0]*l[1]/(18*l[2]) + c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[2,5]*l[0]/12 + c[2,5]*l[1]*np.tan(g[0])/24 - c[2,5]*l[2]*np.tan(g[1])/24 + c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,0]*l[2]/24 - c[3,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[3,4]*l[0]/12 - c[3,4]*l[1]*np.tan(g[0])/24 + c[3,4]*l[2]*np.tan(g[1])/24 + c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,5]*l[0]*l[2]/(18*l[1]) + c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[4,0]*l[1]*l[2]/(36*l[0]) + c[4,4]*l[1]/24 - c[4,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[4,5]*l[2]/24 - c[4,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           -c[2,1]*l[0]/12 + c[2,1]*l[1]*np.tan(g[0])/24 - c[2,1]*l[2]*np.tan(g[1])/24 + c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[2,3]*l[0]*l[1]/(18*l[2]) + c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[2,5]*l[1]/24 - c[2,5]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[3,1]*l[0]*l[2]/(18*l[1]) + c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[3,3]*l[0]/12 - c[3,3]*l[1]*np.tan(g[0])/24 + c[3,3]*l[2]*np.tan(g[1])/24 + c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,5]*l[2]/24 - c[3,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[4,1]*l[2]/24 - c[4,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[4,3]*l[1]/24 - c[4,3]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[4,5]*l[1]*l[2]/(36*l[0]),
           -c[2,2]*l[0]*l[1]/(18*l[2]) + c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[2,3]*l[0]/12 + c[2,3]*l[1]*np.tan(g[0])/24 - c[2,3]*l[2]*np.tan(g[1])/24 + c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[2,4]*l[1]/24 - c[2,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[3,2]*l[0]/12 - c[3,2]*l[1]*np.tan(g[0])/24 + c[3,2]*l[2]*np.tan(g[1])/24 + c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,3]*l[0]*l[2]/(18*l[1]) + c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[3,4]*l[2]/24 - c[3,4]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[4,2]*l[1]/24 - c[4,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[4,3]*l[2]/24 - c[4,3]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[4,4]*l[1]*l[2]/(36*l[0]),
           c[2,0]*l[1]/24 + c[2,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[2,4]*l[0]*l[1]/(36*l[2]) - c[2,4]*l[1]*np.tan(g[1])/12 - c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[2,5]*l[0]/24 - c[2,5]*l[1]*np.tan(g[0])/24 - c[2,5]*l[2]*np.tan(g[1])/24 - c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[3,0]*l[2]/24 + c[3,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[3,4]*l[0]/24 - c[3,4]*l[1]*np.tan(g[0])/24 - c[3,4]*l[2]*np.tan(g[1])/24 - c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,5]*l[0]*l[2]/(36*l[1]) - c[3,5]*l[2]*np.tan(g[0])/12 - c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[4,0]*l[1]*l[2]/(36*l[0]) + c[4,4]*l[1]/24 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[4,5]*l[2]/24 + c[4,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]),
           -c[2,1]*l[0]/24 - c[2,1]*l[1]*np.tan(g[0])/24 - c[2,1]*l[2]*np.tan(g[1])/24 - c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[2,3]*l[0]*l[1]/(36*l[2]) - c[2,3]*l[1]*np.tan(g[1])/12 - c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + c[2,5]*l[1]/24 + c[2,5]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[3,1]*l[0]*l[2]/(36*l[1]) - c[3,1]*l[2]*np.tan(g[0])/12 - c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - c[3,3]*l[0]/24 - c[3,3]*l[1]*np.tan(g[0])/24 - c[3,3]*l[2]*np.tan(g[1])/24 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[3,5]*l[2]/24 + c[3,5]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[4,1]*l[2]/24 + c[4,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[4,3]*l[1]/24 + c[4,3]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[4,5]*l[1]*l[2]/(36*l[0]),
           -c[2,2]*l[0]*l[1]/(36*l[2]) - c[2,2]*l[1]*np.tan(g[1])/12 - c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) - c[2,3]*l[0]/24 - c[2,3]*l[1]*np.tan(g[0])/24 - c[2,3]*l[2]*np.tan(g[1])/24 - c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + c[2,4]*l[1]/24 + c[2,4]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - c[3,2]*l[0]/24 - c[3,2]*l[1]*np.tan(g[0])/24 - c[3,2]*l[2]*np.tan(g[1])/24 - c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - c[3,3]*l[0]*l[2]/(36*l[1]) - c[3,3]*l[2]*np.tan(g[0])/12 - c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + c[3,4]*l[2]/24 + c[3,4]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + c[4,2]*l[1]/24 + c[4,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + c[4,3]*l[2]/24 + c[4,3]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - c[4,4]*l[1]*l[2]/(36*l[0]),
           c[2,0]*l[1]/12 + c[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[2,4]*l[0]*l[1]/(18*l[2]) - c[2,4]*l[1]*np.tan(g[1])/6 - c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[2,5]*l[0]/24 - c[2,5]*l[1]*np.tan(g[0])/12 + c[2,5]*l[2]*np.tan(g[1])/24 - c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,0]*l[2]/24 + c[3,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[3,4]*l[0]/24 - c[3,4]*l[1]*np.tan(g[0])/12 - c[3,4]*l[2]*np.tan(g[1])/24 - c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,5]*l[0]*l[2]/(36*l[1]) - c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[4,0]*l[1]*l[2]/(18*l[0]) + c[4,4]*l[1]/12 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,5]*l[2]/24 + c[4,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[2,1]*l[0]/24 - c[2,1]*l[1]*np.tan(g[0])/12 + c[2,1]*l[2]*np.tan(g[1])/24 - c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[2,3]*l[0]*l[1]/(18*l[2]) - c[2,3]*l[1]*np.tan(g[1])/6 - c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[2,5]*l[1]/12 + c[2,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[3,1]*l[0]*l[2]/(36*l[1]) - c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[3,3]*l[0]/24 - c[3,3]*l[1]*np.tan(g[0])/12 - c[3,3]*l[2]*np.tan(g[1])/24 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,5]*l[2]/24 + c[3,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,1]*l[2]/24 + c[4,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,3]*l[1]/12 + c[4,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,5]*l[1]*l[2]/(18*l[0]),
           -c[2,2]*l[0]*l[1]/(18*l[2]) - c[2,2]*l[1]*np.tan(g[1])/6 - c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[2,3]*l[0]/24 - c[2,3]*l[1]*np.tan(g[0])/12 + c[2,3]*l[2]*np.tan(g[1])/24 - c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[2,4]*l[1]/12 + c[2,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[3,2]*l[0]/24 - c[3,2]*l[1]*np.tan(g[0])/12 - c[3,2]*l[2]*np.tan(g[1])/24 - c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,3]*l[0]*l[2]/(36*l[1]) - c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[3,4]*l[2]/24 + c[3,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,2]*l[1]/12 + c[4,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,3]*l[2]/24 + c[4,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,4]*l[1]*l[2]/(18*l[0]),
           -c[2,0]*l[1]/12 - c[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[2,4]*l[0]*l[1]/(9*l[2]) + c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[2,5]*l[0]/12 + c[2,5]*l[1]*np.tan(g[0])/12 + c[2,5]*l[2]*np.tan(g[1])/24 + c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,0]*l[2]/24 - c[3,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[3,4]*l[0]/12 - c[3,4]*l[1]*np.tan(g[0])/12 + c[3,4]*l[2]*np.tan(g[1])/24 + c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,5]*l[0]*l[2]/(18*l[1]) + c[3,5]*l[2]*np.tan(g[0])/12 + c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[4,0]*l[1]*l[2]/(18*l[0]) + c[4,4]*l[1]/12 - c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,5]*l[2]/24 - c[4,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           c[2,1]*l[0]/12 + c[2,1]*l[1]*np.tan(g[0])/12 + c[2,1]*l[2]*np.tan(g[1])/24 + c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[2,3]*l[0]*l[1]/(9*l[2]) + c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[2,5]*l[1]/12 - c[2,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[3,1]*l[0]*l[2]/(18*l[1]) + c[3,1]*l[2]*np.tan(g[0])/12 + c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[3,3]*l[0]/12 - c[3,3]*l[1]*np.tan(g[0])/12 + c[3,3]*l[2]*np.tan(g[1])/24 + c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,5]*l[2]/24 - c[3,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,1]*l[2]/24 - c[4,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,3]*l[1]/12 - c[4,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,5]*l[1]*l[2]/(18*l[0]),
           -c[2,2]*l[0]*l[1]/(9*l[2]) + c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[2,3]*l[0]/12 + c[2,3]*l[1]*np.tan(g[0])/12 + c[2,3]*l[2]*np.tan(g[1])/24 + c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[2,4]*l[1]/12 - c[2,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[3,2]*l[0]/12 - c[3,2]*l[1]*np.tan(g[0])/12 + c[3,2]*l[2]*np.tan(g[1])/24 + c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,3]*l[0]*l[2]/(18*l[1]) + c[3,3]*l[2]*np.tan(g[0])/12 + c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[3,4]*l[2]/24 - c[3,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,2]*l[1]/12 - c[4,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,3]*l[2]/24 - c[4,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,4]*l[1]*l[2]/(18*l[0]),
           -c[2,0]*l[1]/24 - c[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[2,4]*l[0]*l[1]/(18*l[2]) + c[2,4]*l[1]*np.tan(g[1])/12 + c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[2,5]*l[0]/12 + c[2,5]*l[1]*np.tan(g[0])/24 - c[2,5]*l[2]*np.tan(g[1])/12 + c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,0]*l[2]/12 - c[3,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[3,4]*l[0]/12 + c[3,4]*l[1]*np.tan(g[0])/24 + c[3,4]*l[2]*np.tan(g[1])/12 + c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,5]*l[0]*l[2]/(9*l[1]) + c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[4,0]*l[1]*l[2]/(18*l[0]) - c[4,4]*l[1]/24 - c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,5]*l[2]/12 - c[4,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[2,1]*l[0]/12 + c[2,1]*l[1]*np.tan(g[0])/24 - c[2,1]*l[2]*np.tan(g[1])/12 + c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[2,3]*l[0]*l[1]/(18*l[2]) + c[2,3]*l[1]*np.tan(g[1])/12 + c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[2,5]*l[1]/24 - c[2,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[3,1]*l[0]*l[2]/(9*l[1]) + c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[3,3]*l[0]/12 + c[3,3]*l[1]*np.tan(g[0])/24 + c[3,3]*l[2]*np.tan(g[1])/12 + c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,5]*l[2]/12 - c[3,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,1]*l[2]/12 - c[4,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,3]*l[1]/24 - c[4,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,5]*l[1]*l[2]/(18*l[0]),
           c[2,2]*l[0]*l[1]/(18*l[2]) + c[2,2]*l[1]*np.tan(g[1])/12 + c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[2,3]*l[0]/12 + c[2,3]*l[1]*np.tan(g[0])/24 - c[2,3]*l[2]*np.tan(g[1])/12 + c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[2,4]*l[1]/24 - c[2,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[3,2]*l[0]/12 + c[3,2]*l[1]*np.tan(g[0])/24 + c[3,2]*l[2]*np.tan(g[1])/12 + c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,3]*l[0]*l[2]/(9*l[1]) + c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[3,4]*l[2]/12 - c[3,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,2]*l[1]/24 - c[4,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,3]*l[2]/12 - c[4,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,4]*l[1]*l[2]/(18*l[0]),
           c[2,0]*l[1]/24 + c[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[2,4]*l[0]*l[1]/(36*l[2]) - c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[2,5]*l[0]/24 - c[2,5]*l[1]*np.tan(g[0])/24 - c[2,5]*l[2]*np.tan(g[1])/12 - c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,0]*l[2]/12 + c[3,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[3,4]*l[0]/24 + c[3,4]*l[1]*np.tan(g[0])/24 - c[3,4]*l[2]*np.tan(g[1])/12 - c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,5]*l[0]*l[2]/(18*l[1]) - c[3,5]*l[2]*np.tan(g[0])/6 - c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - c[4,0]*l[1]*l[2]/(18*l[0]) - c[4,4]*l[1]/24 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,5]*l[2]/12 + c[4,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]),
           -c[2,1]*l[0]/24 - c[2,1]*l[1]*np.tan(g[0])/24 - c[2,1]*l[2]*np.tan(g[1])/12 - c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[2,3]*l[0]*l[1]/(36*l[2]) - c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + c[2,5]*l[1]/24 + c[2,5]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[3,1]*l[0]*l[2]/(18*l[1]) - c[3,1]*l[2]*np.tan(g[0])/6 - c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[3,3]*l[0]/24 + c[3,3]*l[1]*np.tan(g[0])/24 - c[3,3]*l[2]*np.tan(g[1])/12 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[3,5]*l[2]/12 + c[3,5]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + c[4,1]*l[2]/12 + c[4,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,3]*l[1]/24 + c[4,3]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - c[4,5]*l[1]*l[2]/(18*l[0]),
           c[2,2]*l[0]*l[1]/(36*l[2]) - c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - c[2,3]*l[0]/24 - c[2,3]*l[1]*np.tan(g[0])/24 - c[2,3]*l[2]*np.tan(g[1])/12 - c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + c[2,4]*l[1]/24 + c[2,4]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[3,2]*l[0]/24 + c[3,2]*l[1]*np.tan(g[0])/24 - c[3,2]*l[2]*np.tan(g[1])/12 - c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - c[3,3]*l[0]*l[2]/(18*l[1]) - c[3,3]*l[2]*np.tan(g[0])/6 - c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + c[3,4]*l[2]/12 + c[3,4]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,2]*l[1]/24 + c[4,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + c[4,3]*l[2]/12 + c[4,3]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - c[4,4]*l[1]*l[2]/(18*l[0]),
           c[2,0]*l[1]/12 + c[2,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[2,4]*l[0]*l[1]/(18*l[2]) - c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[2,5]*l[0]/24 - c[2,5]*l[1]*np.tan(g[0])/12 + c[2,5]*l[2]*np.tan(g[1])/12 - c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,0]*l[2]/12 + c[3,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[3,4]*l[0]/24 + c[3,4]*l[1]*np.tan(g[0])/12 - c[3,4]*l[2]*np.tan(g[1])/12 - c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,5]*l[0]*l[2]/(18*l[1]) - c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[4,0]*l[1]*l[2]/(9*l[0]) - c[4,4]*l[1]/12 + c[4,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[4,5]*l[2]/12 + c[4,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           c[2,1]*l[0]/24 - c[2,1]*l[1]*np.tan(g[0])/12 + c[2,1]*l[2]*np.tan(g[1])/12 - c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[2,3]*l[0]*l[1]/(18*l[2]) - c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[2,5]*l[1]/12 + c[2,5]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[3,1]*l[0]*l[2]/(18*l[1]) - c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[3,3]*l[0]/24 + c[3,3]*l[1]*np.tan(g[0])/12 - c[3,3]*l[2]*np.tan(g[1])/12 - c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,5]*l[2]/12 + c[3,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[4,1]*l[2]/12 + c[4,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[4,3]*l[1]/12 + c[4,3]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[4,5]*l[1]*l[2]/(9*l[0]),
           c[2,2]*l[0]*l[1]/(18*l[2]) - c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[2,3]*l[0]/24 - c[2,3]*l[1]*np.tan(g[0])/12 + c[2,3]*l[2]*np.tan(g[1])/12 - c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[2,4]*l[1]/12 + c[2,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[3,2]*l[0]/24 + c[3,2]*l[1]*np.tan(g[0])/12 - c[3,2]*l[2]*np.tan(g[1])/12 - c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,3]*l[0]*l[2]/(18*l[1]) - c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[3,4]*l[2]/12 + c[3,4]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[4,2]*l[1]/12 + c[4,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[4,3]*l[2]/12 + c[4,3]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[4,4]*l[1]*l[2]/(9*l[0]),
           -c[2,0]*l[1]/12 - c[2,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[2,4]*l[0]*l[1]/(9*l[2]) + c[2,4]*l[1]*np.tan(g[1])/6 + c[2,4]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[2,5]*l[0]/12 + c[2,5]*l[1]*np.tan(g[0])/12 + c[2,5]*l[2]*np.tan(g[1])/12 + c[2,5]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[3,0]*l[2]/12 - c[3,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[3,4]*l[0]/12 + c[3,4]*l[1]*np.tan(g[0])/12 + c[3,4]*l[2]*np.tan(g[1])/12 + c[3,4]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,5]*l[0]*l[2]/(9*l[1]) + c[3,5]*l[2]*np.tan(g[0])/6 + c[3,5]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[4,0]*l[1]*l[2]/(9*l[0]) - c[4,4]*l[1]/12 - c[4,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[4,5]*l[2]/12 - c[4,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]),
           c[2,1]*l[0]/12 + c[2,1]*l[1]*np.tan(g[0])/12 + c[2,1]*l[2]*np.tan(g[1])/12 + c[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[2,3]*l[0]*l[1]/(9*l[2]) + c[2,3]*l[1]*np.tan(g[1])/6 + c[2,3]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - c[2,5]*l[1]/12 - c[2,5]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[3,1]*l[0]*l[2]/(9*l[1]) + c[3,1]*l[2]*np.tan(g[0])/6 + c[3,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + c[3,3]*l[0]/12 + c[3,3]*l[1]*np.tan(g[0])/12 + c[3,3]*l[2]*np.tan(g[1])/12 + c[3,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[3,5]*l[2]/12 - c[3,5]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[4,1]*l[2]/12 - c[4,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[4,3]*l[1]/12 - c[4,3]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[4,5]*l[1]*l[2]/(9*l[0]),
           c[2,2]*l[0]*l[1]/(9*l[2]) + c[2,2]*l[1]*np.tan(g[1])/6 + c[2,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + c[2,3]*l[0]/12 + c[2,3]*l[1]*np.tan(g[0])/12 + c[2,3]*l[2]*np.tan(g[1])/12 + c[2,3]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - c[2,4]*l[1]/12 - c[2,4]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + c[3,2]*l[0]/12 + c[3,2]*l[1]*np.tan(g[0])/12 + c[3,2]*l[2]*np.tan(g[1])/12 + c[3,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + c[3,3]*l[0]*l[2]/(9*l[1]) + c[3,3]*l[2]*np.tan(g[0])/6 + c[3,3]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - c[3,4]*l[2]/12 - c[3,4]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - c[4,2]*l[1]/12 - c[4,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - c[4,3]*l[2]/12 - c[4,3]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + c[4,4]*l[1]*l[2]/(9*l[0])]])

def _lf_strain_3d(xe: np.ndarray, eps: np.ndarray, c: np.ndarray,
                  quadr_method: str = "gauss-legendre",
                  nquad: int = 2,
                  **kwargs: Any) -> np.ndarray:
    """
    Compute nodal forces on trilinear hexahedral Lagrangian element
    (1st order) due to a uniform strain via numerical integration.
    We assume anisotropic elasticity.

    Parameters
    ----------
    xe : np.ndarray, shape (nels,8,3)
        coordinates of element nodes. Please look at the
        definition/function of the shape function, then the node ordering is
        clear.
    eps : np.ndarray shape (nels,6) or (6)
        uniform strain in Voigt notation.
    c : np.ndarray, shape (6,6)
        stiffness tensor in Voigt notation.
    quadr_method: str or callable
        name of quadrature method or function/callable that returns coordinates of
        quadrature points and weights. Check function get_integrpoints for
        available options.
    nquad : int
        number of quadrature points

    Returns
    -------
    fe : np.ndarray, shape (nels,24,1)
        nodal forces.

    """
    #
    if len(xe.shape) == 3:
        xe = xe[None,:,:]
    nel = xe.shape[0]
    #
    if (len(eps.shape) == 1) or (eps.shape[0] == 1):
        eps = np.full((xe.shape[0],6), eps)
    #
    if len(c.shape) == 2:
        c = c[None,:,:]
    #
    x,w=get_integrpoints(ndim=3,nq=nquad,method=quadr_method)
    nq =w.shape[0]
    #
    xi,eta = [_x[:,0] for _x in np.split(x, 3,axis=1)]
    #
    B,detJ = bmatrix(xi=xi, eta=eta, xe=xe,
                     all_elems=True,
                     return_detJ=True)
    detJ = detJ.reshape(nel,nq)
    B = B.reshape(nel, nq,  B.shape[-2], B.shape[-1])
    #
    integral = B.transpose([0,1,3,2])@c[:,None,:,:]@eps[:,None,None,:].transpose(0,1,3,2)
    # multiply by determinant and quadrature
    return (w[None,:,None,None]*integral*detJ[:,:,None,None]).sum(axis=1)

def lf_strain_3d(eps: np.ndarray, E: float = 1., nu: float = 0.3,
                 l: np.ndarray = np.array([1.,1.,1.]), 
                 g: np.ndarray = np.array([0.,0.]),
                 **kwargs: Any) -> np.ndarray:
    """
    Compute nodal forces on trilinear hexahedral Lagrangian element
    (1st order) due to a uniform strain via analytical integration.
    Element shape is a parallelepiped and we assume isotropic elasticity.

    Parameters
    ----------
    eps : np.ndarray shape (6)
        uniform strain in Voigt notation.
    E : float
        Young's modulus.
    nu : float
        Poisson' ratio.
    l : np.ndarray (3)
        side lengths of element
    g : np.ndarray (2)
        angles of parallelepiped.

    Returns
    -------
    fe : np.ndarray, shape (24,1)
        nodal forces.

    """
    return np.array([[E*(-2*eps[0]*l[1]*l[2]*nu + 2*eps[0]*l[1]*l[2] + 2*eps[1]*l[1]*l[2]*nu + 2*eps[2]*l[1]*l[2]*nu - eps[4]*l[0]*l[1]*nu + eps[4]*l[0]*l[1] + eps[4]*l[1]*l[2]*nu*np.tan(g[1]) - eps[4]*l[1]*l[2]*np.tan(g[1]) - eps[5]*l[0]*l[2]*nu + eps[5]*l[0]*l[2] + eps[5]*l[1]*l[2]*nu*np.tan(g[0]) - eps[5]*l[1]*l[2]*np.tan(g[0]))/(8*(2*nu**2 + nu - 1))],
          [E*(2*eps[0]*l[0]*l[2]*nu - 2*eps[0]*l[1]*l[2]*nu*np.tan(g[0]) - 2*eps[1]*l[0]*l[2]*nu + 2*eps[1]*l[0]*l[2] + 2*eps[1]*l[1]*l[2]*nu*np.tan(g[0]) - 2*eps[1]*l[1]*l[2]*np.tan(g[0]) + 2*eps[2]*l[0]*l[2]*nu - 2*eps[2]*l[1]*l[2]*nu*np.tan(g[0]) - eps[3]*l[0]*l[1]*nu + eps[3]*l[0]*l[1] + eps[3]*l[1]*l[2]*nu*np.tan(g[1]) - eps[3]*l[1]*l[2]*np.tan(g[1]) - eps[5]*l[1]*l[2]*nu + eps[5]*l[1]*l[2])/(8*(2*nu**2 + nu - 1))],
          [E*(2*eps[0]*l[0]*l[1]*nu - 2*eps[0]*l[1]*l[2]*nu*np.tan(g[1]) + 2*eps[1]*l[0]*l[1]*nu - 2*eps[1]*l[1]*l[2]*nu*np.tan(g[1]) - 2*eps[2]*l[0]*l[1]*nu + 2*eps[2]*l[0]*l[1] + 2*eps[2]*l[1]*l[2]*nu*np.tan(g[1]) - 2*eps[2]*l[1]*l[2]*np.tan(g[1]) - eps[3]*l[0]*l[2]*nu + eps[3]*l[0]*l[2] + eps[3]*l[1]*l[2]*nu*np.tan(g[0]) - eps[3]*l[1]*l[2]*np.tan(g[0]) - eps[4]*l[1]*l[2]*nu + eps[4]*l[1]*l[2])/(8*(2*nu**2 + nu - 1))],
          [E*(2*eps[0]*l[1]*l[2]*nu - 2*eps[0]*l[1]*l[2] - 2*eps[1]*l[1]*l[2]*nu - 2*eps[2]*l[1]*l[2]*nu - eps[4]*l[0]*l[1]*nu + eps[4]*l[0]*l[1] - eps[4]*l[1]*l[2]*nu*np.tan(g[1]) + eps[4]*l[1]*l[2]*np.tan(g[1]) - eps[5]*l[0]*l[2]*nu + eps[5]*l[0]*l[2] - eps[5]*l[1]*l[2]*nu*np.tan(g[0]) + eps[5]*l[1]*l[2]*np.tan(g[0]))/(8*(2*nu**2 + nu - 1))],
          [E*(2*eps[0]*l[0]*l[2]*nu + 2*eps[0]*l[1]*l[2]*nu*np.tan(g[0]) - 2*eps[1]*l[0]*l[2]*nu + 2*eps[1]*l[0]*l[2] - 2*eps[1]*l[1]*l[2]*nu*np.tan(g[0]) + 2*eps[1]*l[1]*l[2]*np.tan(g[0]) + 2*eps[2]*l[0]*l[2]*nu + 2*eps[2]*l[1]*l[2]*nu*np.tan(g[0]) - eps[3]*l[0]*l[1]*nu + eps[3]*l[0]*l[1] - eps[3]*l[1]*l[2]*nu*np.tan(g[1]) + eps[3]*l[1]*l[2]*np.tan(g[1]) + eps[5]*l[1]*l[2]*nu - eps[5]*l[1]*l[2])/(8*(2*nu**2 + nu - 1))],
          [E*(2*eps[0]*l[0]*l[1]*nu + 2*eps[0]*l[1]*l[2]*nu*np.tan(g[1]) + 2*eps[1]*l[0]*l[1]*nu + 2*eps[1]*l[1]*l[2]*nu*np.tan(g[1]) - 2*eps[2]*l[0]*l[1]*nu + 2*eps[2]*l[0]*l[1] - 2*eps[2]*l[1]*l[2]*nu*np.tan(g[1]) + 2*eps[2]*l[1]*l[2]*np.tan(g[1]) - eps[3]*l[0]*l[2]*nu + eps[3]*l[0]*l[2] - eps[3]*l[1]*l[2]*nu*np.tan(g[0]) + eps[3]*l[1]*l[2]*np.tan(g[0]) + eps[4]*l[1]*l[2]*nu - eps[4]*l[1]*l[2])/(8*(2*nu**2 + nu - 1))],
          [E*(2*eps[0]*l[1]*l[2]*nu - 2*eps[0]*l[1]*l[2] - 2*eps[1]*l[1]*l[2]*nu - 2*eps[2]*l[1]*l[2]*nu - eps[4]*l[0]*l[1]*nu + eps[4]*l[0]*l[1] - eps[4]*l[1]*l[2]*nu*np.tan(g[1]) + eps[4]*l[1]*l[2]*np.tan(g[1]) + eps[5]*l[0]*l[2]*nu - eps[5]*l[0]*l[2] - eps[5]*l[1]*l[2]*nu*np.tan(g[0]) + eps[5]*l[1]*l[2]*np.tan(g[0]))/(8*(2*nu**2 + nu - 1))],
          [E*(-2*eps[0]*l[0]*l[2]*nu + 2*eps[0]*l[1]*l[2]*nu*np.tan(g[0]) + 2*eps[1]*l[0]*l[2]*nu - 2*eps[1]*l[0]*l[2] - 2*eps[1]*l[1]*l[2]*nu*np.tan(g[0]) + 2*eps[1]*l[1]*l[2]*np.tan(g[0]) - 2*eps[2]*l[0]*l[2]*nu + 2*eps[2]*l[1]*l[2]*nu*np.tan(g[0]) - eps[3]*l[0]*l[1]*nu + eps[3]*l[0]*l[1] - eps[3]*l[1]*l[2]*nu*np.tan(g[1]) + eps[3]*l[1]*l[2]*np.tan(g[1]) + eps[5]*l[1]*l[2]*nu - eps[5]*l[1]*l[2])/(8*(2*nu**2 + nu - 1))],
          [E*(2*eps[0]*l[0]*l[1]*nu + 2*eps[0]*l[1]*l[2]*nu*np.tan(g[1]) + 2*eps[1]*l[0]*l[1]*nu + 2*eps[1]*l[1]*l[2]*nu*np.tan(g[1]) - 2*eps[2]*l[0]*l[1]*nu + 2*eps[2]*l[0]*l[1] - 2*eps[2]*l[1]*l[2]*nu*np.tan(g[1]) + 2*eps[2]*l[1]*l[2]*np.tan(g[1]) + eps[3]*l[0]*l[2]*nu - eps[3]*l[0]*l[2] - eps[3]*l[1]*l[2]*nu*np.tan(g[0]) + eps[3]*l[1]*l[2]*np.tan(g[0]) + eps[4]*l[1]*l[2]*nu - eps[4]*l[1]*l[2])/(8*(2*nu**2 + nu - 1))],
          [E*(-2*eps[0]*l[1]*l[2]*nu + 2*eps[0]*l[1]*l[2] + 2*eps[1]*l[1]*l[2]*nu + 2*eps[2]*l[1]*l[2]*nu - eps[4]*l[0]*l[1]*nu + eps[4]*l[0]*l[1] + eps[4]*l[1]*l[2]*nu*np.tan(g[1]) - eps[4]*l[1]*l[2]*np.tan(g[1]) + eps[5]*l[0]*l[2]*nu - eps[5]*l[0]*l[2] + eps[5]*l[1]*l[2]*nu*np.tan(g[0]) - eps[5]*l[1]*l[2]*np.tan(g[0]))/(8*(2*nu**2 + nu - 1))],
          [E*(-2*eps[0]*l[0]*l[2]*nu - 2*eps[0]*l[1]*l[2]*nu*np.tan(g[0]) + 2*eps[1]*l[0]*l[2]*nu - 2*eps[1]*l[0]*l[2] + 2*eps[1]*l[1]*l[2]*nu*np.tan(g[0]) - 2*eps[1]*l[1]*l[2]*np.tan(g[0]) - 2*eps[2]*l[0]*l[2]*nu - 2*eps[2]*l[1]*l[2]*nu*np.tan(g[0]) - eps[3]*l[0]*l[1]*nu + eps[3]*l[0]*l[1] + eps[3]*l[1]*l[2]*nu*np.tan(g[1]) - eps[3]*l[1]*l[2]*np.tan(g[1]) - eps[5]*l[1]*l[2]*nu + eps[5]*l[1]*l[2])/(8*(2*nu**2 + nu - 1))],
          [E*(2*eps[0]*l[0]*l[1]*nu - 2*eps[0]*l[1]*l[2]*nu*np.tan(g[1]) + 2*eps[1]*l[0]*l[1]*nu - 2*eps[1]*l[1]*l[2]*nu*np.tan(g[1]) - 2*eps[2]*l[0]*l[1]*nu + 2*eps[2]*l[0]*l[1] + 2*eps[2]*l[1]*l[2]*nu*np.tan(g[1]) - 2*eps[2]*l[1]*l[2]*np.tan(g[1]) + eps[3]*l[0]*l[2]*nu - eps[3]*l[0]*l[2] + eps[3]*l[1]*l[2]*nu*np.tan(g[0]) - eps[3]*l[1]*l[2]*np.tan(g[0]) - eps[4]*l[1]*l[2]*nu + eps[4]*l[1]*l[2])/(8*(2*nu**2 + nu - 1))],
          [E*(-2*eps[0]*l[1]*l[2]*nu + 2*eps[0]*l[1]*l[2] + 2*eps[1]*l[1]*l[2]*nu + 2*eps[2]*l[1]*l[2]*nu + eps[4]*l[0]*l[1]*nu - eps[4]*l[0]*l[1] + eps[4]*l[1]*l[2]*nu*np.tan(g[1]) - eps[4]*l[1]*l[2]*np.tan(g[1]) - eps[5]*l[0]*l[2]*nu + eps[5]*l[0]*l[2] + eps[5]*l[1]*l[2]*nu*np.tan(g[0]) - eps[5]*l[1]*l[2]*np.tan(g[0]))/(8*(2*nu**2 + nu - 1))],
          [E*(2*eps[0]*l[0]*l[2]*nu - 2*eps[0]*l[1]*l[2]*nu*np.tan(g[0]) - 2*eps[1]*l[0]*l[2]*nu + 2*eps[1]*l[0]*l[2] + 2*eps[1]*l[1]*l[2]*nu*np.tan(g[0]) - 2*eps[1]*l[1]*l[2]*np.tan(g[0]) + 2*eps[2]*l[0]*l[2]*nu - 2*eps[2]*l[1]*l[2]*nu*np.tan(g[0]) + eps[3]*l[0]*l[1]*nu - eps[3]*l[0]*l[1] + eps[3]*l[1]*l[2]*nu*np.tan(g[1]) - eps[3]*l[1]*l[2]*np.tan(g[1]) - eps[5]*l[1]*l[2]*nu + eps[5]*l[1]*l[2])/(8*(2*nu**2 + nu - 1))],
          [E*(-2*eps[0]*l[0]*l[1]*nu - 2*eps[0]*l[1]*l[2]*nu*np.tan(g[1]) - 2*eps[1]*l[0]*l[1]*nu - 2*eps[1]*l[1]*l[2]*nu*np.tan(g[1]) + 2*eps[2]*l[0]*l[1]*nu - 2*eps[2]*l[0]*l[1] + 2*eps[2]*l[1]*l[2]*nu*np.tan(g[1]) - 2*eps[2]*l[1]*l[2]*np.tan(g[1]) - eps[3]*l[0]*l[2]*nu + eps[3]*l[0]*l[2] + eps[3]*l[1]*l[2]*nu*np.tan(g[0]) - eps[3]*l[1]*l[2]*np.tan(g[0]) - eps[4]*l[1]*l[2]*nu + eps[4]*l[1]*l[2])/(8*(2*nu**2 + nu - 1))],
          [E*(2*eps[0]*l[1]*l[2]*nu - 2*eps[0]*l[1]*l[2] - 2*eps[1]*l[1]*l[2]*nu - 2*eps[2]*l[1]*l[2]*nu + eps[4]*l[0]*l[1]*nu - eps[4]*l[0]*l[1] - eps[4]*l[1]*l[2]*nu*np.tan(g[1]) + eps[4]*l[1]*l[2]*np.tan(g[1]) - eps[5]*l[0]*l[2]*nu + eps[5]*l[0]*l[2] - eps[5]*l[1]*l[2]*nu*np.tan(g[0]) + eps[5]*l[1]*l[2]*np.tan(g[0]))/(8*(2*nu**2 + nu - 1))],
          [E*(2*eps[0]*l[0]*l[2]*nu + 2*eps[0]*l[1]*l[2]*nu*np.tan(g[0]) - 2*eps[1]*l[0]*l[2]*nu + 2*eps[1]*l[0]*l[2] - 2*eps[1]*l[1]*l[2]*nu*np.tan(g[0]) + 2*eps[1]*l[1]*l[2]*np.tan(g[0]) + 2*eps[2]*l[0]*l[2]*nu + 2*eps[2]*l[1]*l[2]*nu*np.tan(g[0]) + eps[3]*l[0]*l[1]*nu - eps[3]*l[0]*l[1] - eps[3]*l[1]*l[2]*nu*np.tan(g[1]) + eps[3]*l[1]*l[2]*np.tan(g[1]) + eps[5]*l[1]*l[2]*nu - eps[5]*l[1]*l[2])/(8*(2*nu**2 + nu - 1))],
          [E*(-2*eps[0]*l[0]*l[1]*nu + 2*eps[0]*l[1]*l[2]*nu*np.tan(g[1]) - 2*eps[1]*l[0]*l[1]*nu + 2*eps[1]*l[1]*l[2]*nu*np.tan(g[1]) + 2*eps[2]*l[0]*l[1]*nu - 2*eps[2]*l[0]*l[1] - 2*eps[2]*l[1]*l[2]*nu*np.tan(g[1]) + 2*eps[2]*l[1]*l[2]*np.tan(g[1]) - eps[3]*l[0]*l[2]*nu + eps[3]*l[0]*l[2] - eps[3]*l[1]*l[2]*nu*np.tan(g[0]) + eps[3]*l[1]*l[2]*np.tan(g[0]) + eps[4]*l[1]*l[2]*nu - eps[4]*l[1]*l[2])/(8*(2*nu**2 + nu - 1))],
          [E*(2*eps[0]*l[1]*l[2]*nu - 2*eps[0]*l[1]*l[2] - 2*eps[1]*l[1]*l[2]*nu - 2*eps[2]*l[1]*l[2]*nu + eps[4]*l[0]*l[1]*nu - eps[4]*l[0]*l[1] - eps[4]*l[1]*l[2]*nu*np.tan(g[1]) + eps[4]*l[1]*l[2]*np.tan(g[1]) + eps[5]*l[0]*l[2]*nu - eps[5]*l[0]*l[2] - eps[5]*l[1]*l[2]*nu*np.tan(g[0]) + eps[5]*l[1]*l[2]*np.tan(g[0]))/(8*(2*nu**2 + nu - 1))],
          [E*(-2*eps[0]*l[0]*l[2]*nu + 2*eps[0]*l[1]*l[2]*nu*np.tan(g[0]) + 2*eps[1]*l[0]*l[2]*nu - 2*eps[1]*l[0]*l[2] - 2*eps[1]*l[1]*l[2]*nu*np.tan(g[0]) + 2*eps[1]*l[1]*l[2]*np.tan(g[0]) - 2*eps[2]*l[0]*l[2]*nu + 2*eps[2]*l[1]*l[2]*nu*np.tan(g[0]) + eps[3]*l[0]*l[1]*nu - eps[3]*l[0]*l[1] - eps[3]*l[1]*l[2]*nu*np.tan(g[1]) + eps[3]*l[1]*l[2]*np.tan(g[1]) + eps[5]*l[1]*l[2]*nu - eps[5]*l[1]*l[2])/(8*(2*nu**2 + nu - 1))],
          [E*(-2*eps[0]*l[0]*l[1]*nu + 2*eps[0]*l[1]*l[2]*nu*np.tan(g[1]) - 2*eps[1]*l[0]*l[1]*nu + 2*eps[1]*l[1]*l[2]*nu*np.tan(g[1]) + 2*eps[2]*l[0]*l[1]*nu - 2*eps[2]*l[0]*l[1] - 2*eps[2]*l[1]*l[2]*nu*np.tan(g[1]) + 2*eps[2]*l[1]*l[2]*np.tan(g[1]) + eps[3]*l[0]*l[2]*nu - eps[3]*l[0]*l[2] - eps[3]*l[1]*l[2]*nu*np.tan(g[0]) + eps[3]*l[1]*l[2]*np.tan(g[0]) + eps[4]*l[1]*l[2]*nu - eps[4]*l[1]*l[2])/(8*(2*nu**2 + nu - 1))],
          [E*(-2*eps[0]*l[1]*l[2]*nu + 2*eps[0]*l[1]*l[2] + 2*eps[1]*l[1]*l[2]*nu + 2*eps[2]*l[1]*l[2]*nu + eps[4]*l[0]*l[1]*nu - eps[4]*l[0]*l[1] + eps[4]*l[1]*l[2]*nu*np.tan(g[1]) - eps[4]*l[1]*l[2]*np.tan(g[1]) + eps[5]*l[0]*l[2]*nu - eps[5]*l[0]*l[2] + eps[5]*l[1]*l[2]*nu*np.tan(g[0]) - eps[5]*l[1]*l[2]*np.tan(g[0]))/(8*(2*nu**2 + nu - 1))],
          [E*(-2*eps[0]*l[0]*l[2]*nu - 2*eps[0]*l[1]*l[2]*nu*np.tan(g[0]) + 2*eps[1]*l[0]*l[2]*nu - 2*eps[1]*l[0]*l[2] + 2*eps[1]*l[1]*l[2]*nu*np.tan(g[0]) - 2*eps[1]*l[1]*l[2]*np.tan(g[0]) - 2*eps[2]*l[0]*l[2]*nu - 2*eps[2]*l[1]*l[2]*nu*np.tan(g[0]) + eps[3]*l[0]*l[1]*nu - eps[3]*l[0]*l[1] + eps[3]*l[1]*l[2]*nu*np.tan(g[1]) - eps[3]*l[1]*l[2]*np.tan(g[1]) - eps[5]*l[1]*l[2]*nu + eps[5]*l[1]*l[2])/(8*(2*nu**2 + nu - 1))],
          [E*(-2*eps[0]*l[0]*l[1]*nu - 2*eps[0]*l[1]*l[2]*nu*np.tan(g[1]) - 2*eps[1]*l[0]*l[1]*nu - 2*eps[1]*l[1]*l[2]*nu*np.tan(g[1]) + 2*eps[2]*l[0]*l[1]*nu - 2*eps[2]*l[0]*l[1] + 2*eps[2]*l[1]*l[2]*nu*np.tan(g[1]) - 2*eps[2]*l[1]*l[2]*np.tan(g[1]) + eps[3]*l[0]*l[2]*nu - eps[3]*l[0]*l[2] + eps[3]*l[1]*l[2]*nu*np.tan(g[0]) - eps[3]*l[1]*l[2]*np.tan(g[0]) - eps[4]*l[1]*l[2]*nu + eps[4]*l[1]*l[2])/(8*(2*nu**2 + nu - 1))]])

def lf_strain_aniso_3d(eps: np.ndarray, c: np.ndarray,
                       l: np.ndarray = np.array([1.,1.,1.]), 
                       g: np.ndarray = np.array([0.,0.]),
                       **kwargs: Any) -> np.ndarray:
    """
    Compute nodal forces on trilinear hexahedral Lagrangian element
    (1st order) due to a uniform strain via analytical integration.
    Element shape is a parallelepiped and we assume anisotropic elasticity.

    Parameters
    ----------
    eps : np.ndarray shape (6)
        uniform strain in Voigt notation.
    c : np.ndarray, shape (6,6)
        stiffness tensor in Voigt notation.
    l : np.ndarray (3)
        side lengths of element
    g : np.ndarray (2)
        angles of parallelepiped.

    Returns
    -------
    fe : np.ndarray, shape (24,1)
        nodal forces.

    """
    return np.array([[-c[0,0]*eps[0]*l[1]*l[2]/4 - c[0,1]*eps[1]*l[1]*l[2]/4 - c[0,2]*eps[2]*l[1]*l[2]/4 - c[0,3]*eps[3]*l[1]*l[2]/4 - c[0,4]*eps[4]*l[1]*l[2]/4 - c[0,5]*eps[5]*l[1]*l[2]/4 - c[4,0]*eps[0]*l[0]*l[1]/4 + c[4,0]*eps[0]*l[1]*l[2]*np.tan(g[1])/4 - c[4,1]*eps[1]*l[0]*l[1]/4 + c[4,1]*eps[1]*l[1]*l[2]*np.tan(g[1])/4 - c[4,2]*eps[2]*l[0]*l[1]/4 + c[4,2]*eps[2]*l[1]*l[2]*np.tan(g[1])/4 - c[4,3]*eps[3]*l[0]*l[1]/4 + c[4,3]*eps[3]*l[1]*l[2]*np.tan(g[1])/4 - c[4,4]*eps[4]*l[0]*l[1]/4 + c[4,4]*eps[4]*l[1]*l[2]*np.tan(g[1])/4 - c[4,5]*eps[5]*l[0]*l[1]/4 + c[4,5]*eps[5]*l[1]*l[2]*np.tan(g[1])/4 - c[5,0]*eps[0]*l[0]*l[2]/4 + c[5,0]*eps[0]*l[1]*l[2]*np.tan(g[0])/4 - c[5,1]*eps[1]*l[0]*l[2]/4 + c[5,1]*eps[1]*l[1]*l[2]*np.tan(g[0])/4 - c[5,2]*eps[2]*l[0]*l[2]/4 + c[5,2]*eps[2]*l[1]*l[2]*np.tan(g[0])/4 - c[5,3]*eps[3]*l[0]*l[2]/4 + c[5,3]*eps[3]*l[1]*l[2]*np.tan(g[0])/4 - c[5,4]*eps[4]*l[0]*l[2]/4 + c[5,4]*eps[4]*l[1]*l[2]*np.tan(g[0])/4 - c[5,5]*eps[5]*l[0]*l[2]/4 + c[5,5]*eps[5]*l[1]*l[2]*np.tan(g[0])/4],
          [-c[1,0]*eps[0]*l[0]*l[2]/4 + c[1,0]*eps[0]*l[1]*l[2]*np.tan(g[0])/4 - c[1,1]*eps[1]*l[0]*l[2]/4 + c[1,1]*eps[1]*l[1]*l[2]*np.tan(g[0])/4 - c[1,2]*eps[2]*l[0]*l[2]/4 + c[1,2]*eps[2]*l[1]*l[2]*np.tan(g[0])/4 - c[1,3]*eps[3]*l[0]*l[2]/4 + c[1,3]*eps[3]*l[1]*l[2]*np.tan(g[0])/4 - c[1,4]*eps[4]*l[0]*l[2]/4 + c[1,4]*eps[4]*l[1]*l[2]*np.tan(g[0])/4 - c[1,5]*eps[5]*l[0]*l[2]/4 + c[1,5]*eps[5]*l[1]*l[2]*np.tan(g[0])/4 - c[3,0]*eps[0]*l[0]*l[1]/4 + c[3,0]*eps[0]*l[1]*l[2]*np.tan(g[1])/4 - c[3,1]*eps[1]*l[0]*l[1]/4 + c[3,1]*eps[1]*l[1]*l[2]*np.tan(g[1])/4 - c[3,2]*eps[2]*l[0]*l[1]/4 + c[3,2]*eps[2]*l[1]*l[2]*np.tan(g[1])/4 - c[3,3]*eps[3]*l[0]*l[1]/4 + c[3,3]*eps[3]*l[1]*l[2]*np.tan(g[1])/4 - c[3,4]*eps[4]*l[0]*l[1]/4 + c[3,4]*eps[4]*l[1]*l[2]*np.tan(g[1])/4 - c[3,5]*eps[5]*l[0]*l[1]/4 + c[3,5]*eps[5]*l[1]*l[2]*np.tan(g[1])/4 - c[5,0]*eps[0]*l[1]*l[2]/4 - c[5,1]*eps[1]*l[1]*l[2]/4 - c[5,2]*eps[2]*l[1]*l[2]/4 - c[5,3]*eps[3]*l[1]*l[2]/4 - c[5,4]*eps[4]*l[1]*l[2]/4 - c[5,5]*eps[5]*l[1]*l[2]/4],
          [-c[2,0]*eps[0]*l[0]*l[1]/4 + c[2,0]*eps[0]*l[1]*l[2]*np.tan(g[1])/4 - c[2,1]*eps[1]*l[0]*l[1]/4 + c[2,1]*eps[1]*l[1]*l[2]*np.tan(g[1])/4 - c[2,2]*eps[2]*l[0]*l[1]/4 + c[2,2]*eps[2]*l[1]*l[2]*np.tan(g[1])/4 - c[2,3]*eps[3]*l[0]*l[1]/4 + c[2,3]*eps[3]*l[1]*l[2]*np.tan(g[1])/4 - c[2,4]*eps[4]*l[0]*l[1]/4 + c[2,4]*eps[4]*l[1]*l[2]*np.tan(g[1])/4 - c[2,5]*eps[5]*l[0]*l[1]/4 + c[2,5]*eps[5]*l[1]*l[2]*np.tan(g[1])/4 - c[3,0]*eps[0]*l[0]*l[2]/4 + c[3,0]*eps[0]*l[1]*l[2]*np.tan(g[0])/4 - c[3,1]*eps[1]*l[0]*l[2]/4 + c[3,1]*eps[1]*l[1]*l[2]*np.tan(g[0])/4 - c[3,2]*eps[2]*l[0]*l[2]/4 + c[3,2]*eps[2]*l[1]*l[2]*np.tan(g[0])/4 - c[3,3]*eps[3]*l[0]*l[2]/4 + c[3,3]*eps[3]*l[1]*l[2]*np.tan(g[0])/4 - c[3,4]*eps[4]*l[0]*l[2]/4 + c[3,4]*eps[4]*l[1]*l[2]*np.tan(g[0])/4 - c[3,5]*eps[5]*l[0]*l[2]/4 + c[3,5]*eps[5]*l[1]*l[2]*np.tan(g[0])/4 - c[4,0]*eps[0]*l[1]*l[2]/4 - c[4,1]*eps[1]*l[1]*l[2]/4 - c[4,2]*eps[2]*l[1]*l[2]/4 - c[4,3]*eps[3]*l[1]*l[2]/4 - c[4,4]*eps[4]*l[1]*l[2]/4 - c[4,5]*eps[5]*l[1]*l[2]/4],
          [c[0,0]*eps[0]*l[1]*l[2]/4 + c[0,1]*eps[1]*l[1]*l[2]/4 + c[0,2]*eps[2]*l[1]*l[2]/4 + c[0,3]*eps[3]*l[1]*l[2]/4 + c[0,4]*eps[4]*l[1]*l[2]/4 + c[0,5]*eps[5]*l[1]*l[2]/4 - c[4,0]*eps[0]*l[0]*l[1]/4 - c[4,0]*eps[0]*l[1]*l[2]*np.tan(g[1])/4 - c[4,1]*eps[1]*l[0]*l[1]/4 - c[4,1]*eps[1]*l[1]*l[2]*np.tan(g[1])/4 - c[4,2]*eps[2]*l[0]*l[1]/4 - c[4,2]*eps[2]*l[1]*l[2]*np.tan(g[1])/4 - c[4,3]*eps[3]*l[0]*l[1]/4 - c[4,3]*eps[3]*l[1]*l[2]*np.tan(g[1])/4 - c[4,4]*eps[4]*l[0]*l[1]/4 - c[4,4]*eps[4]*l[1]*l[2]*np.tan(g[1])/4 - c[4,5]*eps[5]*l[0]*l[1]/4 - c[4,5]*eps[5]*l[1]*l[2]*np.tan(g[1])/4 - c[5,0]*eps[0]*l[0]*l[2]/4 - c[5,0]*eps[0]*l[1]*l[2]*np.tan(g[0])/4 - c[5,1]*eps[1]*l[0]*l[2]/4 - c[5,1]*eps[1]*l[1]*l[2]*np.tan(g[0])/4 - c[5,2]*eps[2]*l[0]*l[2]/4 - c[5,2]*eps[2]*l[1]*l[2]*np.tan(g[0])/4 - c[5,3]*eps[3]*l[0]*l[2]/4 - c[5,3]*eps[3]*l[1]*l[2]*np.tan(g[0])/4 - c[5,4]*eps[4]*l[0]*l[2]/4 - c[5,4]*eps[4]*l[1]*l[2]*np.tan(g[0])/4 - c[5,5]*eps[5]*l[0]*l[2]/4 - c[5,5]*eps[5]*l[1]*l[2]*np.tan(g[0])/4],
          [-c[1,0]*eps[0]*l[0]*l[2]/4 - c[1,0]*eps[0]*l[1]*l[2]*np.tan(g[0])/4 - c[1,1]*eps[1]*l[0]*l[2]/4 - c[1,1]*eps[1]*l[1]*l[2]*np.tan(g[0])/4 - c[1,2]*eps[2]*l[0]*l[2]/4 - c[1,2]*eps[2]*l[1]*l[2]*np.tan(g[0])/4 - c[1,3]*eps[3]*l[0]*l[2]/4 - c[1,3]*eps[3]*l[1]*l[2]*np.tan(g[0])/4 - c[1,4]*eps[4]*l[0]*l[2]/4 - c[1,4]*eps[4]*l[1]*l[2]*np.tan(g[0])/4 - c[1,5]*eps[5]*l[0]*l[2]/4 - c[1,5]*eps[5]*l[1]*l[2]*np.tan(g[0])/4 - c[3,0]*eps[0]*l[0]*l[1]/4 - c[3,0]*eps[0]*l[1]*l[2]*np.tan(g[1])/4 - c[3,1]*eps[1]*l[0]*l[1]/4 - c[3,1]*eps[1]*l[1]*l[2]*np.tan(g[1])/4 - c[3,2]*eps[2]*l[0]*l[1]/4 - c[3,2]*eps[2]*l[1]*l[2]*np.tan(g[1])/4 - c[3,3]*eps[3]*l[0]*l[1]/4 - c[3,3]*eps[3]*l[1]*l[2]*np.tan(g[1])/4 - c[3,4]*eps[4]*l[0]*l[1]/4 - c[3,4]*eps[4]*l[1]*l[2]*np.tan(g[1])/4 - c[3,5]*eps[5]*l[0]*l[1]/4 - c[3,5]*eps[5]*l[1]*l[2]*np.tan(g[1])/4 + c[5,0]*eps[0]*l[1]*l[2]/4 + c[5,1]*eps[1]*l[1]*l[2]/4 + c[5,2]*eps[2]*l[1]*l[2]/4 + c[5,3]*eps[3]*l[1]*l[2]/4 + c[5,4]*eps[4]*l[1]*l[2]/4 + c[5,5]*eps[5]*l[1]*l[2]/4],
          [-c[2,0]*eps[0]*l[0]*l[1]/4 - c[2,0]*eps[0]*l[1]*l[2]*np.tan(g[1])/4 - c[2,1]*eps[1]*l[0]*l[1]/4 - c[2,1]*eps[1]*l[1]*l[2]*np.tan(g[1])/4 - c[2,2]*eps[2]*l[0]*l[1]/4 - c[2,2]*eps[2]*l[1]*l[2]*np.tan(g[1])/4 - c[2,3]*eps[3]*l[0]*l[1]/4 - c[2,3]*eps[3]*l[1]*l[2]*np.tan(g[1])/4 - c[2,4]*eps[4]*l[0]*l[1]/4 - c[2,4]*eps[4]*l[1]*l[2]*np.tan(g[1])/4 - c[2,5]*eps[5]*l[0]*l[1]/4 - c[2,5]*eps[5]*l[1]*l[2]*np.tan(g[1])/4 - c[3,0]*eps[0]*l[0]*l[2]/4 - c[3,0]*eps[0]*l[1]*l[2]*np.tan(g[0])/4 - c[3,1]*eps[1]*l[0]*l[2]/4 - c[3,1]*eps[1]*l[1]*l[2]*np.tan(g[0])/4 - c[3,2]*eps[2]*l[0]*l[2]/4 - c[3,2]*eps[2]*l[1]*l[2]*np.tan(g[0])/4 - c[3,3]*eps[3]*l[0]*l[2]/4 - c[3,3]*eps[3]*l[1]*l[2]*np.tan(g[0])/4 - c[3,4]*eps[4]*l[0]*l[2]/4 - c[3,4]*eps[4]*l[1]*l[2]*np.tan(g[0])/4 - c[3,5]*eps[5]*l[0]*l[2]/4 - c[3,5]*eps[5]*l[1]*l[2]*np.tan(g[0])/4 + c[4,0]*eps[0]*l[1]*l[2]/4 + c[4,1]*eps[1]*l[1]*l[2]/4 + c[4,2]*eps[2]*l[1]*l[2]/4 + c[4,3]*eps[3]*l[1]*l[2]/4 + c[4,4]*eps[4]*l[1]*l[2]/4 + c[4,5]*eps[5]*l[1]*l[2]/4],
          [c[0,0]*eps[0]*l[1]*l[2]/4 + c[0,1]*eps[1]*l[1]*l[2]/4 + c[0,2]*eps[2]*l[1]*l[2]/4 + c[0,3]*eps[3]*l[1]*l[2]/4 + c[0,4]*eps[4]*l[1]*l[2]/4 + c[0,5]*eps[5]*l[1]*l[2]/4 - c[4,0]*eps[0]*l[0]*l[1]/4 - c[4,0]*eps[0]*l[1]*l[2]*np.tan(g[1])/4 - c[4,1]*eps[1]*l[0]*l[1]/4 - c[4,1]*eps[1]*l[1]*l[2]*np.tan(g[1])/4 - c[4,2]*eps[2]*l[0]*l[1]/4 - c[4,2]*eps[2]*l[1]*l[2]*np.tan(g[1])/4 - c[4,3]*eps[3]*l[0]*l[1]/4 - c[4,3]*eps[3]*l[1]*l[2]*np.tan(g[1])/4 - c[4,4]*eps[4]*l[0]*l[1]/4 - c[4,4]*eps[4]*l[1]*l[2]*np.tan(g[1])/4 - c[4,5]*eps[5]*l[0]*l[1]/4 - c[4,5]*eps[5]*l[1]*l[2]*np.tan(g[1])/4 + c[5,0]*eps[0]*l[0]*l[2]/4 - c[5,0]*eps[0]*l[1]*l[2]*np.tan(g[0])/4 + c[5,1]*eps[1]*l[0]*l[2]/4 - c[5,1]*eps[1]*l[1]*l[2]*np.tan(g[0])/4 + c[5,2]*eps[2]*l[0]*l[2]/4 - c[5,2]*eps[2]*l[1]*l[2]*np.tan(g[0])/4 + c[5,3]*eps[3]*l[0]*l[2]/4 - c[5,3]*eps[3]*l[1]*l[2]*np.tan(g[0])/4 + c[5,4]*eps[4]*l[0]*l[2]/4 - c[5,4]*eps[4]*l[1]*l[2]*np.tan(g[0])/4 + c[5,5]*eps[5]*l[0]*l[2]/4 - c[5,5]*eps[5]*l[1]*l[2]*np.tan(g[0])/4],
          [c[1,0]*eps[0]*l[0]*l[2]/4 - c[1,0]*eps[0]*l[1]*l[2]*np.tan(g[0])/4 + c[1,1]*eps[1]*l[0]*l[2]/4 - c[1,1]*eps[1]*l[1]*l[2]*np.tan(g[0])/4 + c[1,2]*eps[2]*l[0]*l[2]/4 - c[1,2]*eps[2]*l[1]*l[2]*np.tan(g[0])/4 + c[1,3]*eps[3]*l[0]*l[2]/4 - c[1,3]*eps[3]*l[1]*l[2]*np.tan(g[0])/4 + c[1,4]*eps[4]*l[0]*l[2]/4 - c[1,4]*eps[4]*l[1]*l[2]*np.tan(g[0])/4 + c[1,5]*eps[5]*l[0]*l[2]/4 - c[1,5]*eps[5]*l[1]*l[2]*np.tan(g[0])/4 - c[3,0]*eps[0]*l[0]*l[1]/4 - c[3,0]*eps[0]*l[1]*l[2]*np.tan(g[1])/4 - c[3,1]*eps[1]*l[0]*l[1]/4 - c[3,1]*eps[1]*l[1]*l[2]*np.tan(g[1])/4 - c[3,2]*eps[2]*l[0]*l[1]/4 - c[3,2]*eps[2]*l[1]*l[2]*np.tan(g[1])/4 - c[3,3]*eps[3]*l[0]*l[1]/4 - c[3,3]*eps[3]*l[1]*l[2]*np.tan(g[1])/4 - c[3,4]*eps[4]*l[0]*l[1]/4 - c[3,4]*eps[4]*l[1]*l[2]*np.tan(g[1])/4 - c[3,5]*eps[5]*l[0]*l[1]/4 - c[3,5]*eps[5]*l[1]*l[2]*np.tan(g[1])/4 + c[5,0]*eps[0]*l[1]*l[2]/4 + c[5,1]*eps[1]*l[1]*l[2]/4 + c[5,2]*eps[2]*l[1]*l[2]/4 + c[5,3]*eps[3]*l[1]*l[2]/4 + c[5,4]*eps[4]*l[1]*l[2]/4 + c[5,5]*eps[5]*l[1]*l[2]/4],
          [-c[2,0]*eps[0]*l[0]*l[1]/4 - c[2,0]*eps[0]*l[1]*l[2]*np.tan(g[1])/4 - c[2,1]*eps[1]*l[0]*l[1]/4 - c[2,1]*eps[1]*l[1]*l[2]*np.tan(g[1])/4 - c[2,2]*eps[2]*l[0]*l[1]/4 - c[2,2]*eps[2]*l[1]*l[2]*np.tan(g[1])/4 - c[2,3]*eps[3]*l[0]*l[1]/4 - c[2,3]*eps[3]*l[1]*l[2]*np.tan(g[1])/4 - c[2,4]*eps[4]*l[0]*l[1]/4 - c[2,4]*eps[4]*l[1]*l[2]*np.tan(g[1])/4 - c[2,5]*eps[5]*l[0]*l[1]/4 - c[2,5]*eps[5]*l[1]*l[2]*np.tan(g[1])/4 + c[3,0]*eps[0]*l[0]*l[2]/4 - c[3,0]*eps[0]*l[1]*l[2]*np.tan(g[0])/4 + c[3,1]*eps[1]*l[0]*l[2]/4 - c[3,1]*eps[1]*l[1]*l[2]*np.tan(g[0])/4 + c[3,2]*eps[2]*l[0]*l[2]/4 - c[3,2]*eps[2]*l[1]*l[2]*np.tan(g[0])/4 + c[3,3]*eps[3]*l[0]*l[2]/4 - c[3,3]*eps[3]*l[1]*l[2]*np.tan(g[0])/4 + c[3,4]*eps[4]*l[0]*l[2]/4 - c[3,4]*eps[4]*l[1]*l[2]*np.tan(g[0])/4 + c[3,5]*eps[5]*l[0]*l[2]/4 - c[3,5]*eps[5]*l[1]*l[2]*np.tan(g[0])/4 + c[4,0]*eps[0]*l[1]*l[2]/4 + c[4,1]*eps[1]*l[1]*l[2]/4 + c[4,2]*eps[2]*l[1]*l[2]/4 + c[4,3]*eps[3]*l[1]*l[2]/4 + c[4,4]*eps[4]*l[1]*l[2]/4 + c[4,5]*eps[5]*l[1]*l[2]/4],
          [-c[0,0]*eps[0]*l[1]*l[2]/4 - c[0,1]*eps[1]*l[1]*l[2]/4 - c[0,2]*eps[2]*l[1]*l[2]/4 - c[0,3]*eps[3]*l[1]*l[2]/4 - c[0,4]*eps[4]*l[1]*l[2]/4 - c[0,5]*eps[5]*l[1]*l[2]/4 - c[4,0]*eps[0]*l[0]*l[1]/4 + c[4,0]*eps[0]*l[1]*l[2]*np.tan(g[1])/4 - c[4,1]*eps[1]*l[0]*l[1]/4 + c[4,1]*eps[1]*l[1]*l[2]*np.tan(g[1])/4 - c[4,2]*eps[2]*l[0]*l[1]/4 + c[4,2]*eps[2]*l[1]*l[2]*np.tan(g[1])/4 - c[4,3]*eps[3]*l[0]*l[1]/4 + c[4,3]*eps[3]*l[1]*l[2]*np.tan(g[1])/4 - c[4,4]*eps[4]*l[0]*l[1]/4 + c[4,4]*eps[4]*l[1]*l[2]*np.tan(g[1])/4 - c[4,5]*eps[5]*l[0]*l[1]/4 + c[4,5]*eps[5]*l[1]*l[2]*np.tan(g[1])/4 + c[5,0]*eps[0]*l[0]*l[2]/4 + c[5,0]*eps[0]*l[1]*l[2]*np.tan(g[0])/4 + c[5,1]*eps[1]*l[0]*l[2]/4 + c[5,1]*eps[1]*l[1]*l[2]*np.tan(g[0])/4 + c[5,2]*eps[2]*l[0]*l[2]/4 + c[5,2]*eps[2]*l[1]*l[2]*np.tan(g[0])/4 + c[5,3]*eps[3]*l[0]*l[2]/4 + c[5,3]*eps[3]*l[1]*l[2]*np.tan(g[0])/4 + c[5,4]*eps[4]*l[0]*l[2]/4 + c[5,4]*eps[4]*l[1]*l[2]*np.tan(g[0])/4 + c[5,5]*eps[5]*l[0]*l[2]/4 + c[5,5]*eps[5]*l[1]*l[2]*np.tan(g[0])/4],
          [c[1,0]*eps[0]*l[0]*l[2]/4 + c[1,0]*eps[0]*l[1]*l[2]*np.tan(g[0])/4 + c[1,1]*eps[1]*l[0]*l[2]/4 + c[1,1]*eps[1]*l[1]*l[2]*np.tan(g[0])/4 + c[1,2]*eps[2]*l[0]*l[2]/4 + c[1,2]*eps[2]*l[1]*l[2]*np.tan(g[0])/4 + c[1,3]*eps[3]*l[0]*l[2]/4 + c[1,3]*eps[3]*l[1]*l[2]*np.tan(g[0])/4 + c[1,4]*eps[4]*l[0]*l[2]/4 + c[1,4]*eps[4]*l[1]*l[2]*np.tan(g[0])/4 + c[1,5]*eps[5]*l[0]*l[2]/4 + c[1,5]*eps[5]*l[1]*l[2]*np.tan(g[0])/4 - c[3,0]*eps[0]*l[0]*l[1]/4 + c[3,0]*eps[0]*l[1]*l[2]*np.tan(g[1])/4 - c[3,1]*eps[1]*l[0]*l[1]/4 + c[3,1]*eps[1]*l[1]*l[2]*np.tan(g[1])/4 - c[3,2]*eps[2]*l[0]*l[1]/4 + c[3,2]*eps[2]*l[1]*l[2]*np.tan(g[1])/4 - c[3,3]*eps[3]*l[0]*l[1]/4 + c[3,3]*eps[3]*l[1]*l[2]*np.tan(g[1])/4 - c[3,4]*eps[4]*l[0]*l[1]/4 + c[3,4]*eps[4]*l[1]*l[2]*np.tan(g[1])/4 - c[3,5]*eps[5]*l[0]*l[1]/4 + c[3,5]*eps[5]*l[1]*l[2]*np.tan(g[1])/4 - c[5,0]*eps[0]*l[1]*l[2]/4 - c[5,1]*eps[1]*l[1]*l[2]/4 - c[5,2]*eps[2]*l[1]*l[2]/4 - c[5,3]*eps[3]*l[1]*l[2]/4 - c[5,4]*eps[4]*l[1]*l[2]/4 - c[5,5]*eps[5]*l[1]*l[2]/4],
          [-c[2,0]*eps[0]*l[0]*l[1]/4 + c[2,0]*eps[0]*l[1]*l[2]*np.tan(g[1])/4 - c[2,1]*eps[1]*l[0]*l[1]/4 + c[2,1]*eps[1]*l[1]*l[2]*np.tan(g[1])/4 - c[2,2]*eps[2]*l[0]*l[1]/4 + c[2,2]*eps[2]*l[1]*l[2]*np.tan(g[1])/4 - c[2,3]*eps[3]*l[0]*l[1]/4 + c[2,3]*eps[3]*l[1]*l[2]*np.tan(g[1])/4 - c[2,4]*eps[4]*l[0]*l[1]/4 + c[2,4]*eps[4]*l[1]*l[2]*np.tan(g[1])/4 - c[2,5]*eps[5]*l[0]*l[1]/4 + c[2,5]*eps[5]*l[1]*l[2]*np.tan(g[1])/4 + c[3,0]*eps[0]*l[0]*l[2]/4 + c[3,0]*eps[0]*l[1]*l[2]*np.tan(g[0])/4 + c[3,1]*eps[1]*l[0]*l[2]/4 + c[3,1]*eps[1]*l[1]*l[2]*np.tan(g[0])/4 + c[3,2]*eps[2]*l[0]*l[2]/4 + c[3,2]*eps[2]*l[1]*l[2]*np.tan(g[0])/4 + c[3,3]*eps[3]*l[0]*l[2]/4 + c[3,3]*eps[3]*l[1]*l[2]*np.tan(g[0])/4 + c[3,4]*eps[4]*l[0]*l[2]/4 + c[3,4]*eps[4]*l[1]*l[2]*np.tan(g[0])/4 + c[3,5]*eps[5]*l[0]*l[2]/4 + c[3,5]*eps[5]*l[1]*l[2]*np.tan(g[0])/4 - c[4,0]*eps[0]*l[1]*l[2]/4 - c[4,1]*eps[1]*l[1]*l[2]/4 - c[4,2]*eps[2]*l[1]*l[2]/4 - c[4,3]*eps[3]*l[1]*l[2]/4 - c[4,4]*eps[4]*l[1]*l[2]/4 - c[4,5]*eps[5]*l[1]*l[2]/4],
          [-c[0,0]*eps[0]*l[1]*l[2]/4 - c[0,1]*eps[1]*l[1]*l[2]/4 - c[0,2]*eps[2]*l[1]*l[2]/4 - c[0,3]*eps[3]*l[1]*l[2]/4 - c[0,4]*eps[4]*l[1]*l[2]/4 - c[0,5]*eps[5]*l[1]*l[2]/4 + c[4,0]*eps[0]*l[0]*l[1]/4 + c[4,0]*eps[0]*l[1]*l[2]*np.tan(g[1])/4 + c[4,1]*eps[1]*l[0]*l[1]/4 + c[4,1]*eps[1]*l[1]*l[2]*np.tan(g[1])/4 + c[4,2]*eps[2]*l[0]*l[1]/4 + c[4,2]*eps[2]*l[1]*l[2]*np.tan(g[1])/4 + c[4,3]*eps[3]*l[0]*l[1]/4 + c[4,3]*eps[3]*l[1]*l[2]*np.tan(g[1])/4 + c[4,4]*eps[4]*l[0]*l[1]/4 + c[4,4]*eps[4]*l[1]*l[2]*np.tan(g[1])/4 + c[4,5]*eps[5]*l[0]*l[1]/4 + c[4,5]*eps[5]*l[1]*l[2]*np.tan(g[1])/4 - c[5,0]*eps[0]*l[0]*l[2]/4 + c[5,0]*eps[0]*l[1]*l[2]*np.tan(g[0])/4 - c[5,1]*eps[1]*l[0]*l[2]/4 + c[5,1]*eps[1]*l[1]*l[2]*np.tan(g[0])/4 - c[5,2]*eps[2]*l[0]*l[2]/4 + c[5,2]*eps[2]*l[1]*l[2]*np.tan(g[0])/4 - c[5,3]*eps[3]*l[0]*l[2]/4 + c[5,3]*eps[3]*l[1]*l[2]*np.tan(g[0])/4 - c[5,4]*eps[4]*l[0]*l[2]/4 + c[5,4]*eps[4]*l[1]*l[2]*np.tan(g[0])/4 - c[5,5]*eps[5]*l[0]*l[2]/4 + c[5,5]*eps[5]*l[1]*l[2]*np.tan(g[0])/4],
          [-c[1,0]*eps[0]*l[0]*l[2]/4 + c[1,0]*eps[0]*l[1]*l[2]*np.tan(g[0])/4 - c[1,1]*eps[1]*l[0]*l[2]/4 + c[1,1]*eps[1]*l[1]*l[2]*np.tan(g[0])/4 - c[1,2]*eps[2]*l[0]*l[2]/4 + c[1,2]*eps[2]*l[1]*l[2]*np.tan(g[0])/4 - c[1,3]*eps[3]*l[0]*l[2]/4 + c[1,3]*eps[3]*l[1]*l[2]*np.tan(g[0])/4 - c[1,4]*eps[4]*l[0]*l[2]/4 + c[1,4]*eps[4]*l[1]*l[2]*np.tan(g[0])/4 - c[1,5]*eps[5]*l[0]*l[2]/4 + c[1,5]*eps[5]*l[1]*l[2]*np.tan(g[0])/4 + c[3,0]*eps[0]*l[0]*l[1]/4 + c[3,0]*eps[0]*l[1]*l[2]*np.tan(g[1])/4 + c[3,1]*eps[1]*l[0]*l[1]/4 + c[3,1]*eps[1]*l[1]*l[2]*np.tan(g[1])/4 + c[3,2]*eps[2]*l[0]*l[1]/4 + c[3,2]*eps[2]*l[1]*l[2]*np.tan(g[1])/4 + c[3,3]*eps[3]*l[0]*l[1]/4 + c[3,3]*eps[3]*l[1]*l[2]*np.tan(g[1])/4 + c[3,4]*eps[4]*l[0]*l[1]/4 + c[3,4]*eps[4]*l[1]*l[2]*np.tan(g[1])/4 + c[3,5]*eps[5]*l[0]*l[1]/4 + c[3,5]*eps[5]*l[1]*l[2]*np.tan(g[1])/4 - c[5,0]*eps[0]*l[1]*l[2]/4 - c[5,1]*eps[1]*l[1]*l[2]/4 - c[5,2]*eps[2]*l[1]*l[2]/4 - c[5,3]*eps[3]*l[1]*l[2]/4 - c[5,4]*eps[4]*l[1]*l[2]/4 - c[5,5]*eps[5]*l[1]*l[2]/4],
          [c[2,0]*eps[0]*l[0]*l[1]/4 + c[2,0]*eps[0]*l[1]*l[2]*np.tan(g[1])/4 + c[2,1]*eps[1]*l[0]*l[1]/4 + c[2,1]*eps[1]*l[1]*l[2]*np.tan(g[1])/4 + c[2,2]*eps[2]*l[0]*l[1]/4 + c[2,2]*eps[2]*l[1]*l[2]*np.tan(g[1])/4 + c[2,3]*eps[3]*l[0]*l[1]/4 + c[2,3]*eps[3]*l[1]*l[2]*np.tan(g[1])/4 + c[2,4]*eps[4]*l[0]*l[1]/4 + c[2,4]*eps[4]*l[1]*l[2]*np.tan(g[1])/4 + c[2,5]*eps[5]*l[0]*l[1]/4 + c[2,5]*eps[5]*l[1]*l[2]*np.tan(g[1])/4 - c[3,0]*eps[0]*l[0]*l[2]/4 + c[3,0]*eps[0]*l[1]*l[2]*np.tan(g[0])/4 - c[3,1]*eps[1]*l[0]*l[2]/4 + c[3,1]*eps[1]*l[1]*l[2]*np.tan(g[0])/4 - c[3,2]*eps[2]*l[0]*l[2]/4 + c[3,2]*eps[2]*l[1]*l[2]*np.tan(g[0])/4 - c[3,3]*eps[3]*l[0]*l[2]/4 + c[3,3]*eps[3]*l[1]*l[2]*np.tan(g[0])/4 - c[3,4]*eps[4]*l[0]*l[2]/4 + c[3,4]*eps[4]*l[1]*l[2]*np.tan(g[0])/4 - c[3,5]*eps[5]*l[0]*l[2]/4 + c[3,5]*eps[5]*l[1]*l[2]*np.tan(g[0])/4 - c[4,0]*eps[0]*l[1]*l[2]/4 - c[4,1]*eps[1]*l[1]*l[2]/4 - c[4,2]*eps[2]*l[1]*l[2]/4 - c[4,3]*eps[3]*l[1]*l[2]/4 - c[4,4]*eps[4]*l[1]*l[2]/4 - c[4,5]*eps[5]*l[1]*l[2]/4],
          [c[0,0]*eps[0]*l[1]*l[2]/4 + c[0,1]*eps[1]*l[1]*l[2]/4 + c[0,2]*eps[2]*l[1]*l[2]/4 + c[0,3]*eps[3]*l[1]*l[2]/4 + c[0,4]*eps[4]*l[1]*l[2]/4 + c[0,5]*eps[5]*l[1]*l[2]/4 + c[4,0]*eps[0]*l[0]*l[1]/4 - c[4,0]*eps[0]*l[1]*l[2]*np.tan(g[1])/4 + c[4,1]*eps[1]*l[0]*l[1]/4 - c[4,1]*eps[1]*l[1]*l[2]*np.tan(g[1])/4 + c[4,2]*eps[2]*l[0]*l[1]/4 - c[4,2]*eps[2]*l[1]*l[2]*np.tan(g[1])/4 + c[4,3]*eps[3]*l[0]*l[1]/4 - c[4,3]*eps[3]*l[1]*l[2]*np.tan(g[1])/4 + c[4,4]*eps[4]*l[0]*l[1]/4 - c[4,4]*eps[4]*l[1]*l[2]*np.tan(g[1])/4 + c[4,5]*eps[5]*l[0]*l[1]/4 - c[4,5]*eps[5]*l[1]*l[2]*np.tan(g[1])/4 - c[5,0]*eps[0]*l[0]*l[2]/4 - c[5,0]*eps[0]*l[1]*l[2]*np.tan(g[0])/4 - c[5,1]*eps[1]*l[0]*l[2]/4 - c[5,1]*eps[1]*l[1]*l[2]*np.tan(g[0])/4 - c[5,2]*eps[2]*l[0]*l[2]/4 - c[5,2]*eps[2]*l[1]*l[2]*np.tan(g[0])/4 - c[5,3]*eps[3]*l[0]*l[2]/4 - c[5,3]*eps[3]*l[1]*l[2]*np.tan(g[0])/4 - c[5,4]*eps[4]*l[0]*l[2]/4 - c[5,4]*eps[4]*l[1]*l[2]*np.tan(g[0])/4 - c[5,5]*eps[5]*l[0]*l[2]/4 - c[5,5]*eps[5]*l[1]*l[2]*np.tan(g[0])/4],
          [-c[1,0]*eps[0]*l[0]*l[2]/4 - c[1,0]*eps[0]*l[1]*l[2]*np.tan(g[0])/4 - c[1,1]*eps[1]*l[0]*l[2]/4 - c[1,1]*eps[1]*l[1]*l[2]*np.tan(g[0])/4 - c[1,2]*eps[2]*l[0]*l[2]/4 - c[1,2]*eps[2]*l[1]*l[2]*np.tan(g[0])/4 - c[1,3]*eps[3]*l[0]*l[2]/4 - c[1,3]*eps[3]*l[1]*l[2]*np.tan(g[0])/4 - c[1,4]*eps[4]*l[0]*l[2]/4 - c[1,4]*eps[4]*l[1]*l[2]*np.tan(g[0])/4 - c[1,5]*eps[5]*l[0]*l[2]/4 - c[1,5]*eps[5]*l[1]*l[2]*np.tan(g[0])/4 + c[3,0]*eps[0]*l[0]*l[1]/4 - c[3,0]*eps[0]*l[1]*l[2]*np.tan(g[1])/4 + c[3,1]*eps[1]*l[0]*l[1]/4 - c[3,1]*eps[1]*l[1]*l[2]*np.tan(g[1])/4 + c[3,2]*eps[2]*l[0]*l[1]/4 - c[3,2]*eps[2]*l[1]*l[2]*np.tan(g[1])/4 + c[3,3]*eps[3]*l[0]*l[1]/4 - c[3,3]*eps[3]*l[1]*l[2]*np.tan(g[1])/4 + c[3,4]*eps[4]*l[0]*l[1]/4 - c[3,4]*eps[4]*l[1]*l[2]*np.tan(g[1])/4 + c[3,5]*eps[5]*l[0]*l[1]/4 - c[3,5]*eps[5]*l[1]*l[2]*np.tan(g[1])/4 + c[5,0]*eps[0]*l[1]*l[2]/4 + c[5,1]*eps[1]*l[1]*l[2]/4 + c[5,2]*eps[2]*l[1]*l[2]/4 + c[5,3]*eps[3]*l[1]*l[2]/4 + c[5,4]*eps[4]*l[1]*l[2]/4 + c[5,5]*eps[5]*l[1]*l[2]/4],
          [c[2,0]*eps[0]*l[0]*l[1]/4 - c[2,0]*eps[0]*l[1]*l[2]*np.tan(g[1])/4 + c[2,1]*eps[1]*l[0]*l[1]/4 - c[2,1]*eps[1]*l[1]*l[2]*np.tan(g[1])/4 + c[2,2]*eps[2]*l[0]*l[1]/4 - c[2,2]*eps[2]*l[1]*l[2]*np.tan(g[1])/4 + c[2,3]*eps[3]*l[0]*l[1]/4 - c[2,3]*eps[3]*l[1]*l[2]*np.tan(g[1])/4 + c[2,4]*eps[4]*l[0]*l[1]/4 - c[2,4]*eps[4]*l[1]*l[2]*np.tan(g[1])/4 + c[2,5]*eps[5]*l[0]*l[1]/4 - c[2,5]*eps[5]*l[1]*l[2]*np.tan(g[1])/4 - c[3,0]*eps[0]*l[0]*l[2]/4 - c[3,0]*eps[0]*l[1]*l[2]*np.tan(g[0])/4 - c[3,1]*eps[1]*l[0]*l[2]/4 - c[3,1]*eps[1]*l[1]*l[2]*np.tan(g[0])/4 - c[3,2]*eps[2]*l[0]*l[2]/4 - c[3,2]*eps[2]*l[1]*l[2]*np.tan(g[0])/4 - c[3,3]*eps[3]*l[0]*l[2]/4 - c[3,3]*eps[3]*l[1]*l[2]*np.tan(g[0])/4 - c[3,4]*eps[4]*l[0]*l[2]/4 - c[3,4]*eps[4]*l[1]*l[2]*np.tan(g[0])/4 - c[3,5]*eps[5]*l[0]*l[2]/4 - c[3,5]*eps[5]*l[1]*l[2]*np.tan(g[0])/4 + c[4,0]*eps[0]*l[1]*l[2]/4 + c[4,1]*eps[1]*l[1]*l[2]/4 + c[4,2]*eps[2]*l[1]*l[2]/4 + c[4,3]*eps[3]*l[1]*l[2]/4 + c[4,4]*eps[4]*l[1]*l[2]/4 + c[4,5]*eps[5]*l[1]*l[2]/4],
          [c[0,0]*eps[0]*l[1]*l[2]/4 + c[0,1]*eps[1]*l[1]*l[2]/4 + c[0,2]*eps[2]*l[1]*l[2]/4 + c[0,3]*eps[3]*l[1]*l[2]/4 + c[0,4]*eps[4]*l[1]*l[2]/4 + c[0,5]*eps[5]*l[1]*l[2]/4 + c[4,0]*eps[0]*l[0]*l[1]/4 - c[4,0]*eps[0]*l[1]*l[2]*np.tan(g[1])/4 + c[4,1]*eps[1]*l[0]*l[1]/4 - c[4,1]*eps[1]*l[1]*l[2]*np.tan(g[1])/4 + c[4,2]*eps[2]*l[0]*l[1]/4 - c[4,2]*eps[2]*l[1]*l[2]*np.tan(g[1])/4 + c[4,3]*eps[3]*l[0]*l[1]/4 - c[4,3]*eps[3]*l[1]*l[2]*np.tan(g[1])/4 + c[4,4]*eps[4]*l[0]*l[1]/4 - c[4,4]*eps[4]*l[1]*l[2]*np.tan(g[1])/4 + c[4,5]*eps[5]*l[0]*l[1]/4 - c[4,5]*eps[5]*l[1]*l[2]*np.tan(g[1])/4 + c[5,0]*eps[0]*l[0]*l[2]/4 - c[5,0]*eps[0]*l[1]*l[2]*np.tan(g[0])/4 + c[5,1]*eps[1]*l[0]*l[2]/4 - c[5,1]*eps[1]*l[1]*l[2]*np.tan(g[0])/4 + c[5,2]*eps[2]*l[0]*l[2]/4 - c[5,2]*eps[2]*l[1]*l[2]*np.tan(g[0])/4 + c[5,3]*eps[3]*l[0]*l[2]/4 - c[5,3]*eps[3]*l[1]*l[2]*np.tan(g[0])/4 + c[5,4]*eps[4]*l[0]*l[2]/4 - c[5,4]*eps[4]*l[1]*l[2]*np.tan(g[0])/4 + c[5,5]*eps[5]*l[0]*l[2]/4 - c[5,5]*eps[5]*l[1]*l[2]*np.tan(g[0])/4],
          [c[1,0]*eps[0]*l[0]*l[2]/4 - c[1,0]*eps[0]*l[1]*l[2]*np.tan(g[0])/4 + c[1,1]*eps[1]*l[0]*l[2]/4 - c[1,1]*eps[1]*l[1]*l[2]*np.tan(g[0])/4 + c[1,2]*eps[2]*l[0]*l[2]/4 - c[1,2]*eps[2]*l[1]*l[2]*np.tan(g[0])/4 + c[1,3]*eps[3]*l[0]*l[2]/4 - c[1,3]*eps[3]*l[1]*l[2]*np.tan(g[0])/4 + c[1,4]*eps[4]*l[0]*l[2]/4 - c[1,4]*eps[4]*l[1]*l[2]*np.tan(g[0])/4 + c[1,5]*eps[5]*l[0]*l[2]/4 - c[1,5]*eps[5]*l[1]*l[2]*np.tan(g[0])/4 + c[3,0]*eps[0]*l[0]*l[1]/4 - c[3,0]*eps[0]*l[1]*l[2]*np.tan(g[1])/4 + c[3,1]*eps[1]*l[0]*l[1]/4 - c[3,1]*eps[1]*l[1]*l[2]*np.tan(g[1])/4 + c[3,2]*eps[2]*l[0]*l[1]/4 - c[3,2]*eps[2]*l[1]*l[2]*np.tan(g[1])/4 + c[3,3]*eps[3]*l[0]*l[1]/4 - c[3,3]*eps[3]*l[1]*l[2]*np.tan(g[1])/4 + c[3,4]*eps[4]*l[0]*l[1]/4 - c[3,4]*eps[4]*l[1]*l[2]*np.tan(g[1])/4 + c[3,5]*eps[5]*l[0]*l[1]/4 - c[3,5]*eps[5]*l[1]*l[2]*np.tan(g[1])/4 + c[5,0]*eps[0]*l[1]*l[2]/4 + c[5,1]*eps[1]*l[1]*l[2]/4 + c[5,2]*eps[2]*l[1]*l[2]/4 + c[5,3]*eps[3]*l[1]*l[2]/4 + c[5,4]*eps[4]*l[1]*l[2]/4 + c[5,5]*eps[5]*l[1]*l[2]/4],
          [c[2,0]*eps[0]*l[0]*l[1]/4 - c[2,0]*eps[0]*l[1]*l[2]*np.tan(g[1])/4 + c[2,1]*eps[1]*l[0]*l[1]/4 - c[2,1]*eps[1]*l[1]*l[2]*np.tan(g[1])/4 + c[2,2]*eps[2]*l[0]*l[1]/4 - c[2,2]*eps[2]*l[1]*l[2]*np.tan(g[1])/4 + c[2,3]*eps[3]*l[0]*l[1]/4 - c[2,3]*eps[3]*l[1]*l[2]*np.tan(g[1])/4 + c[2,4]*eps[4]*l[0]*l[1]/4 - c[2,4]*eps[4]*l[1]*l[2]*np.tan(g[1])/4 + c[2,5]*eps[5]*l[0]*l[1]/4 - c[2,5]*eps[5]*l[1]*l[2]*np.tan(g[1])/4 + c[3,0]*eps[0]*l[0]*l[2]/4 - c[3,0]*eps[0]*l[1]*l[2]*np.tan(g[0])/4 + c[3,1]*eps[1]*l[0]*l[2]/4 - c[3,1]*eps[1]*l[1]*l[2]*np.tan(g[0])/4 + c[3,2]*eps[2]*l[0]*l[2]/4 - c[3,2]*eps[2]*l[1]*l[2]*np.tan(g[0])/4 + c[3,3]*eps[3]*l[0]*l[2]/4 - c[3,3]*eps[3]*l[1]*l[2]*np.tan(g[0])/4 + c[3,4]*eps[4]*l[0]*l[2]/4 - c[3,4]*eps[4]*l[1]*l[2]*np.tan(g[0])/4 + c[3,5]*eps[5]*l[0]*l[2]/4 - c[3,5]*eps[5]*l[1]*l[2]*np.tan(g[0])/4 + c[4,0]*eps[0]*l[1]*l[2]/4 + c[4,1]*eps[1]*l[1]*l[2]/4 + c[4,2]*eps[2]*l[1]*l[2]/4 + c[4,3]*eps[3]*l[1]*l[2]/4 + c[4,4]*eps[4]*l[1]*l[2]/4 + c[4,5]*eps[5]*l[1]*l[2]/4],
          [-c[0,0]*eps[0]*l[1]*l[2]/4 - c[0,1]*eps[1]*l[1]*l[2]/4 - c[0,2]*eps[2]*l[1]*l[2]/4 - c[0,3]*eps[3]*l[1]*l[2]/4 - c[0,4]*eps[4]*l[1]*l[2]/4 - c[0,5]*eps[5]*l[1]*l[2]/4 + c[4,0]*eps[0]*l[0]*l[1]/4 + c[4,0]*eps[0]*l[1]*l[2]*np.tan(g[1])/4 + c[4,1]*eps[1]*l[0]*l[1]/4 + c[4,1]*eps[1]*l[1]*l[2]*np.tan(g[1])/4 + c[4,2]*eps[2]*l[0]*l[1]/4 + c[4,2]*eps[2]*l[1]*l[2]*np.tan(g[1])/4 + c[4,3]*eps[3]*l[0]*l[1]/4 + c[4,3]*eps[3]*l[1]*l[2]*np.tan(g[1])/4 + c[4,4]*eps[4]*l[0]*l[1]/4 + c[4,4]*eps[4]*l[1]*l[2]*np.tan(g[1])/4 + c[4,5]*eps[5]*l[0]*l[1]/4 + c[4,5]*eps[5]*l[1]*l[2]*np.tan(g[1])/4 + c[5,0]*eps[0]*l[0]*l[2]/4 + c[5,0]*eps[0]*l[1]*l[2]*np.tan(g[0])/4 + c[5,1]*eps[1]*l[0]*l[2]/4 + c[5,1]*eps[1]*l[1]*l[2]*np.tan(g[0])/4 + c[5,2]*eps[2]*l[0]*l[2]/4 + c[5,2]*eps[2]*l[1]*l[2]*np.tan(g[0])/4 + c[5,3]*eps[3]*l[0]*l[2]/4 + c[5,3]*eps[3]*l[1]*l[2]*np.tan(g[0])/4 + c[5,4]*eps[4]*l[0]*l[2]/4 + c[5,4]*eps[4]*l[1]*l[2]*np.tan(g[0])/4 + c[5,5]*eps[5]*l[0]*l[2]/4 + c[5,5]*eps[5]*l[1]*l[2]*np.tan(g[0])/4],
          [c[1,0]*eps[0]*l[0]*l[2]/4 + c[1,0]*eps[0]*l[1]*l[2]*np.tan(g[0])/4 + c[1,1]*eps[1]*l[0]*l[2]/4 + c[1,1]*eps[1]*l[1]*l[2]*np.tan(g[0])/4 + c[1,2]*eps[2]*l[0]*l[2]/4 + c[1,2]*eps[2]*l[1]*l[2]*np.tan(g[0])/4 + c[1,3]*eps[3]*l[0]*l[2]/4 + c[1,3]*eps[3]*l[1]*l[2]*np.tan(g[0])/4 + c[1,4]*eps[4]*l[0]*l[2]/4 + c[1,4]*eps[4]*l[1]*l[2]*np.tan(g[0])/4 + c[1,5]*eps[5]*l[0]*l[2]/4 + c[1,5]*eps[5]*l[1]*l[2]*np.tan(g[0])/4 + c[3,0]*eps[0]*l[0]*l[1]/4 + c[3,0]*eps[0]*l[1]*l[2]*np.tan(g[1])/4 + c[3,1]*eps[1]*l[0]*l[1]/4 + c[3,1]*eps[1]*l[1]*l[2]*np.tan(g[1])/4 + c[3,2]*eps[2]*l[0]*l[1]/4 + c[3,2]*eps[2]*l[1]*l[2]*np.tan(g[1])/4 + c[3,3]*eps[3]*l[0]*l[1]/4 + c[3,3]*eps[3]*l[1]*l[2]*np.tan(g[1])/4 + c[3,4]*eps[4]*l[0]*l[1]/4 + c[3,4]*eps[4]*l[1]*l[2]*np.tan(g[1])/4 + c[3,5]*eps[5]*l[0]*l[1]/4 + c[3,5]*eps[5]*l[1]*l[2]*np.tan(g[1])/4 - c[5,0]*eps[0]*l[1]*l[2]/4 - c[5,1]*eps[1]*l[1]*l[2]/4 - c[5,2]*eps[2]*l[1]*l[2]/4 - c[5,3]*eps[3]*l[1]*l[2]/4 - c[5,4]*eps[4]*l[1]*l[2]/4 - c[5,5]*eps[5]*l[1]*l[2]/4],
          [c[2,0]*eps[0]*l[0]*l[1]/4 + c[2,0]*eps[0]*l[1]*l[2]*np.tan(g[1])/4 + c[2,1]*eps[1]*l[0]*l[1]/4 + c[2,1]*eps[1]*l[1]*l[2]*np.tan(g[1])/4 + c[2,2]*eps[2]*l[0]*l[1]/4 + c[2,2]*eps[2]*l[1]*l[2]*np.tan(g[1])/4 + c[2,3]*eps[3]*l[0]*l[1]/4 + c[2,3]*eps[3]*l[1]*l[2]*np.tan(g[1])/4 + c[2,4]*eps[4]*l[0]*l[1]/4 + c[2,4]*eps[4]*l[1]*l[2]*np.tan(g[1])/4 + c[2,5]*eps[5]*l[0]*l[1]/4 + c[2,5]*eps[5]*l[1]*l[2]*np.tan(g[1])/4 + c[3,0]*eps[0]*l[0]*l[2]/4 + c[3,0]*eps[0]*l[1]*l[2]*np.tan(g[0])/4 + c[3,1]*eps[1]*l[0]*l[2]/4 + c[3,1]*eps[1]*l[1]*l[2]*np.tan(g[0])/4 + c[3,2]*eps[2]*l[0]*l[2]/4 + c[3,2]*eps[2]*l[1]*l[2]*np.tan(g[0])/4 + c[3,3]*eps[3]*l[0]*l[2]/4 + c[3,3]*eps[3]*l[1]*l[2]*np.tan(g[0])/4 + c[3,4]*eps[4]*l[0]*l[2]/4 + c[3,4]*eps[4]*l[1]*l[2]*np.tan(g[0])/4 + c[3,5]*eps[5]*l[0]*l[2]/4 + c[3,5]*eps[5]*l[1]*l[2]*np.tan(g[0])/4 - c[4,0]*eps[0]*l[1]*l[2]/4 - c[4,1]*eps[1]*l[1]*l[2]/4 - c[4,2]*eps[2]*l[1]*l[2]/4 - c[4,3]*eps[3]*l[1]*l[2]/4 - c[4,4]*eps[4]*l[1]*l[2]/4 - c[4,5]*eps[5]*l[1]*l[2]/4]])
