# SPDX-License-Identifier: GPL-3.0-or-later
from typing import Any

import numpy as np

from topoptlab.fem import get_integrpoints
from topoptlab.elements.bilinear_quadrilateral import shape_functions, jacobian

def _lm_monomial_2d(xe: np.ndarray, u: np.ndarray, n: int,
                    p: np.ndarray = np.array([1.]),
                    quadr_method: str = "gauss-legendre",
                    nquad: int = 3,
                    **kwargs: Any) -> np.ndarray:
    """
    Create element matrix for a monomial of a scalar field in 3D with trilinear
    hexahedral elements. The special case for polynomial of order 1 is the
    mass matrix

    Parameters
    ----------
    xe : np.ndarray, shape (nels,8,3)
        coordinates of element nodes. Please look at the
        definition/function of the shape function, then the node ordering is
        clear.
    u : np.ndarray, shape (nels,8)
        nodal scalar field variable.
    n : int
        polynomial order.
    p : np.ndarray of shape (nels) or (1)
        scalar prefactor for each element.
    quadr_method: str or callable
        name of quadrature method or function/callable that returns coordinates of
        quadrature points and weights. Check function get_integrpoints for
        available options.
    nquad : int
        number of quadrature points

    Returns
    -------
    Ke : np.ndarray, shape (nels,8,8)
        element polynomial matrix.

    """
    #
    if len(xe.shape) == 2 and len(u.shape)==1:
        xe,u = xe[None,:,:],u[None,:]
    if len(u.shape) == 1:
        u = u[None,:]
    if xe.shape[0]-1 >= u.shape[0]:
        nel = xe.shape[0]
    else:
        nel = u.shape[0]
    #
    if isinstance(p,float) or (p.shape[0] == 1 and xe.shape[0] !=1):
        p = np.full(nel, p)
    # shape (ncoords,4*)
    x,w=get_integrpoints(ndim=3,nq=nquad,method=quadr_method)
    nq =w.shape[0]
    #
    xi,eta,zeta = [_x[:,0] for _x in np.split(x, 3,axis=1)]
    # (nq,4)
    N = shape_functions(xi=xi,eta=eta)
    #
    integral = N[None,:,:,None]@N[None,:,:,None].transpose([0,1,3,2])
    # calculate determinant of jacobian
    J = jacobian(xi=xi,eta=eta,xe=xe,all_elems=True)
    detJ = (J[:,0,0]*(J[:,1,1]*J[:,2,2] - J[:,1,2]*J[:,2,1])-
            J[:,0,1]*(J[:,1,0]*J[:,2,2] - J[:,1,2]*J[:,2,0])+
            J[:,0,2]*(J[:,1,0]*J[:,2,1] - J[:,1,1]*J[:,2,0])).reshape(nel,nq)
    # multiply by determinant and quadrature
    factor = ((u@N.transpose())**(n-1))[:,:,None,None]
    Ke = (w[None,:,None,None]*integral*factor*detJ[:,:,None,None]).sum(axis=1)
    #
    return p[:,None,None] * Ke

def lm_cubic_3d(u: np.ndarray,
                p: float = 1.,
                l: np.ndarray = np.array([1.,1.,1.]),
                **kwargs: Any) -> np.ndarray:
    """
    Create element matrix for a cubic monomial for scalar field in 3D with
    trilinear hexahedral Lagrangian elements.

    Parameters
    ----------
    p : float
        scalar prefactor
    l : np.ndarray (2)
        side length of element

    Returns
    -------
    Ke : np.ndarray, shape (8,8)
        element stiffness matrix.

    """
    v = l[0] * l[1] * l[2]
    return p*v*np.array([[(864*u[0]**2 + 432*u[0]*u[1] + 108*u[0]*u[2] + 432*u[0]*u[3] + 432*u[0]*u[4] + 108*u[0]*u[5] + 27*u[0]*u[6] + 108*u[0]*u[7] + 144*u[1]**2 + 72*u[1]*u[2] + 108*u[1]*u[3] + 108*u[1]*u[4] + 72*u[1]*u[5] + 18*u[1]*u[6] + 27*u[1]*u[7] + 24*u[2]**2 + 72*u[2]*u[3] + 27*u[2]*u[4] + 18*u[2]*u[5] + 12*u[2]*u[6] + 18*u[2]*u[7] + 144*u[3]**2 + 108*u[3]*u[4] + 27*u[3]*u[5] + 18*u[3]*u[6] + 72*u[3]*u[7] + 144*u[4]**2 + 72*u[4]*u[5] + 18*u[4]*u[6] + 72*u[4]*u[7] + 24*u[5]**2 +  12*u[5]*u[6] + 18*u[5]*u[7] + 4*u[6]**2 + 12*u[6]*u[7] + 24*u[7]**2)/108000,
           (216*u[0]**2 + 288*u[0]*u[1] + 72*u[0]*u[2] + 108*u[0]*u[3] + 108*u[0]*u[4] + 72*u[0]*u[5] + 18*u[0]*u[6] + 27*u[0]*u[7] + 216*u[1]**2 + 108*u[1]*u[2] + 72*u[1]*u[3] + 72*u[1]*u[4] + 108*u[1]*u[5] + 27*u[1]*u[6] + 18*u[1]*u[7] + 36*u[2]**2 + 48*u[2]*u[3] + 18*u[2]*u[4] + 27*u[2]*u[5] + 18*u[2]*u[6] + 12*u[2]*u[7] + 36*u[3]**2 + 27*u[3]*u[4] + 18*u[3]*u[5] + 12*u[3]*u[6] + 18*u[3]*u[7] + 36*u[4]**2 + 48*u[4]*u[5] + 12*u[4]*u[6] + 18*u[4]*u[7] + 36*u[5]**2 + 18*u[5]*u[6] + 12*u[5]*u[7] + 6*u[6]**2 + 8*u[6]*u[7] + 6*u[7]**2)/108000,
           (54*u[0]**2 + 72*u[0]*u[1] + 48*u[0]*u[2] + 72*u[0]*u[3] + 27*u[0]*u[4] + 18*u[0]*u[5] + 12*u[0]*u[6] + 18*u[0]*u[7] + 54*u[1]**2 + 72*u[1]*u[2] + 48*u[1]*u[3] + 18*u[1]*u[4] + 27*u[1]*u[5] + 18*u[1]*u[6] + 12*u[1]*u[7] + 54*u[2]**2 + 72*u[2]*u[3] + 12*u[2]*u[4] + 18*u[2]*u[5] + 27*u[2]*u[6] + 18*u[2]*u[7] + 54*u[3]**2 + 18*u[3]*u[4] + 12*u[3]*u[5] + 18*u[3]*u[6] + 27*u[3]*u[7] + 9*u[4]**2 + 12*u[4]*u[5] + 8*u[4]*u[6] + 12*u[4]*u[7] + 9*u[5]**2 + 12*u[5]*u[6] + 8*u[5]*u[7] + 9*u[6]**2 + 12*u[6]*u[7] + 9*u[7]**2)/108000,
           (216*u[0]**2 + 108*u[0]*u[1] + 72*u[0]*u[2] + 288*u[0]*u[3] + 108*u[0]*u[4] + 27*u[0]*u[5] + 18*u[0]*u[6] + 72*u[0]*u[7] + 36*u[1]**2 + 48*u[1]*u[2] + 72*u[1]*u[3] + 27*u[1]*u[4] + 18*u[1]*u[5] + 12*u[1]*u[6] + 18*u[1]*u[7] + 36*u[2]**2 + 108*u[2]*u[3] + 18*u[2]*u[4] + 12*u[2]*u[5] + 18*u[2]*u[6] + 27*u[2]*u[7] + 216*u[3]**2 + 72*u[3]*u[4] + 18*u[3]*u[5] + 27*u[3]*u[6] + 108*u[3]*u[7] + 36*u[4]**2 + 18*u[4]*u[5] + 12*u[4]*u[6] + 48*u[4]*u[7] + 6*u[5]**2 + 8*u[5]*u[6] + 12*u[5]*u[7] + 6*u[6]**2 + 18*u[6]*u[7] + 36*u[7]**2)/108000,
           (216*u[0]**2 + 108*u[0]*u[1] + 27*u[0]*u[2] + 108*u[0]*u[3] + 288*u[0]*u[4] + 72*u[0]*u[5] + 18*u[0]*u[6] + 72*u[0]*u[7] + 36*u[1]**2 + 18*u[1]*u[2] + 27*u[1]*u[3] + 72*u[1]*u[4] + 48*u[1]*u[5] + 12*u[1]*u[6] + 18*u[1]*u[7] + 6*u[2]**2 + 18*u[2]*u[3] + 18*u[2]*u[4] + 12*u[2]*u[5] + 8*u[2]*u[6] + 12*u[2]*u[7] + 36*u[3]**2 + 72*u[3]*u[4] + 18*u[3]*u[5] + 12*u[3]*u[6] + 48*u[3]*u[7] + 216*u[4]**2 + 108*u[4]*u[5] + 27*u[4]*u[6] + 108*u[4]*u[7] + 36*u[5]**2 + 18*u[5]*u[6] + 27*u[5]*u[7] + 6*u[6]**2 + 18*u[6]*u[7] + 36*u[7]**2)/108000,
           (54*u[0]**2 + 72*u[0]*u[1] + 18*u[0]*u[2] + 27*u[0]*u[3] + 72*u[0]*u[4] + 48*u[0]*u[5] + 12*u[0]*u[6] + 18*u[0]*u[7] + 54*u[1]**2 + 27*u[1]*u[2] + 18*u[1]*u[3] + 48*u[1]*u[4] + 72*u[1]*u[5] + 18*u[1]*u[6] + 12*u[1]*u[7] + 9*u[2]**2 + 12*u[2]*u[3] + 12*u[2]*u[4] + 18*u[2]*u[5] + 12*u[2]*u[6] + 8*u[2]*u[7] + 9*u[3]**2 + 18*u[3]*u[4] + 12*u[3]*u[5] + 8*u[3]*u[6] + 12*u[3]*u[7] + 54*u[4]**2 + 72*u[4]*u[5] + 18*u[4]*u[6] + 27*u[4]*u[7] + 54*u[5]**2 + 27*u[5]*u[6] + 18*u[5]*u[7] + 9*u[6]**2 + 12*u[6]*u[7] + 9*u[7]**2)/108000,
           (27*u[0]**2 + 36*u[0]*u[1] + 24*u[0]*u[2] + 36*u[0]*u[3] + 36*u[0]*u[4] + 24*u[0]*u[5] + 16*u[0]*u[6] + 24*u[0]*u[7] + 27*u[1]**2 + 36*u[1]*u[2] + 24*u[1]*u[3] + 24*u[1]*u[4] + 36*u[1]*u[5] + 24*u[1]*u[6] + 16*u[1]*u[7] + 27*u[2]**2 + 36*u[2]*u[3] + 16*u[2]*u[4] + 24*u[2]*u[5] + 36*u[2]*u[6] + 24*u[2]*u[7] + 27*u[3]**2 + 24*u[3]*u[4] + 16*u[3]*u[5] + 24*u[3]*u[6] + 36*u[3]*u[7] + 27*u[4]**2 + 36*u[4]*u[5] + 24*u[4]*u[6] + 36*u[4]*u[7] + 27*u[5]**2 + 36*u[5]*u[6] + 24*u[5]*u[7] + 27*u[6]**2 + 36*u[6]*u[7] + 27*u[7]**2)/216000,
           (54*u[0]**2 + 27*u[0]*u[1] + 18*u[0]*u[2] + 72*u[0]*u[3] + 72*u[0]*u[4] + 18*u[0]*u[5] + 12*u[0]*u[6] + 48*u[0]*u[7] + 9*u[1]**2 + 12*u[1]*u[2] + 18*u[1]*u[3] + 18*u[1]*u[4] + 12*u[1]*u[5] + 8*u[1]*u[6] + 12*u[1]*u[7] + 9*u[2]**2 + 27*u[2]*u[3] + 12*u[2]*u[4] + 8*u[2]*u[5] + 12*u[2]*u[6] + 18*u[2]*u[7] + 54*u[3]**2 + 48*u[3]*u[4] + 12*u[3]*u[5] + 18*u[3]*u[6] + 72*u[3]*u[7] + 54*u[4]**2 + 27*u[4]*u[5] + 18*u[4]*u[6] + 72*u[4]*u[7] + 9*u[5]**2 + 12*u[5]*u[6] + 18*u[5]*u[7] + 9*u[6]**2 + 27*u[6]*u[7] + 54*u[7]**2)/108000],
          [(216*u[0]**2 + 288*u[0]*u[1] + 72*u[0]*u[2] + 108*u[0]*u[3] + 108*u[0]*u[4] + 72*u[0]*u[5] + 18*u[0]*u[6] + 27*u[0]*u[7] + 216*u[1]**2 + 108*u[1]*u[2] + 72*u[1]*u[3] + 72*u[1]*u[4] + 108*u[1]*u[5] + 27*u[1]*u[6] + 18*u[1]*u[7] + 36*u[2]**2 + 48*u[2]*u[3] + 18*u[2]*u[4] + 27*u[2]*u[5] + 18*u[2]*u[6] + 12*u[2]*u[7] + 36*u[3]**2 + 27*u[3]*u[4] + 18*u[3]*u[5] + 12*u[3]*u[6] + 18*u[3]*u[7] + 36*u[4]**2 + 48*u[4]*u[5] + 12*u[4]*u[6] + 18*u[4]*u[7] + 36*u[5]**2 + 18*u[5]*u[6] + 12*u[5]*u[7] + 6*u[6]**2 + 8*u[6]*u[7] + 6*u[7]**2)/108000,
           (144*u[0]**2 + 432*u[0]*u[1] + 108*u[0]*u[2] + 72*u[0]*u[3] + 72*u[0]*u[4] + 108*u[0]*u[5] + 27*u[0]*u[6] + 18*u[0]*u[7] + 864*u[1]**2 + 432*u[1]*u[2] + 108*u[1]*u[3] + 108*u[1]*u[4] + 432*u[1]*u[5] + 108*u[1]*u[6] + 27*u[1]*u[7] + 144*u[2]**2 + 72*u[2]*u[3] + 27*u[2]*u[4] + 108*u[2]*u[5] + 72*u[2]*u[6] + 18*u[2]*u[7] + 24*u[3]**2 + 18*u[3]*u[4] + 27*u[3]*u[5] + 18*u[3]*u[6] + 12*u[3]*u[7] + 24*u[4]**2 + 72*u[4]*u[5] + 18*u[4]*u[6] + 12*u[4]*u[7] + 144*u[5]**2 + 72*u[5]*u[6] + 18*u[5]*u[7] + 24*u[6]**2 + 12*u[6]*u[7] + 4*u[7]**2)/108000,
           (36*u[0]**2 + 108*u[0]*u[1] + 72*u[0]*u[2] + 48*u[0]*u[3] + 18*u[0]*u[4] + 27*u[0]*u[5] + 18*u[0]*u[6] + 12*u[0]*u[7] + 216*u[1]**2 + 288*u[1]*u[2] + 72*u[1]*u[3] + 27*u[1]*u[4] + 108*u[1]*u[5] + 72*u[1]*u[6] + 18*u[1]*u[7] + 216*u[2]**2 + 108*u[2]*u[3] + 18*u[2]*u[4] + 72*u[2]*u[5] + 108*u[2]*u[6] + 27*u[2]*u[7] + 36*u[3]**2 + 12*u[3]*u[4] + 18*u[3]*u[5] + 27*u[3]*u[6] + 18*u[3]*u[7] + 6*u[4]**2 + 18*u[4]*u[5] + 12*u[4]*u[6] + 8*u[4]*u[7] + 36*u[5]**2 + 48*u[5]*u[6] + 12*u[5]*u[7] + 36*u[6]**2 + 18*u[6]*u[7] + 6*u[7]**2)/108000,
           (54*u[0]**2 + 72*u[0]*u[1] + 48*u[0]*u[2] + 72*u[0]*u[3] + 27*u[0]*u[4] + 18*u[0]*u[5] + 12*u[0]*u[6] + 18*u[0]*u[7] + 54*u[1]**2 + 72*u[1]*u[2] + 48*u[1]*u[3] + 18*u[1]*u[4] + 27*u[1]*u[5] + 18*u[1]*u[6] + 12*u[1]*u[7] + 54*u[2]**2 + 72*u[2]*u[3] + 12*u[2]*u[4] + 18*u[2]*u[5] + 27*u[2]*u[6] + 18*u[2]*u[7] + 54*u[3]**2 + 18*u[3]*u[4] + 12*u[3]*u[5] + 18*u[3]*u[6] + 27*u[3]*u[7] + 9*u[4]**2 + 12*u[4]*u[5] + 8*u[4]*u[6] + 12*u[4]*u[7] + 9*u[5]**2 + 12*u[5]*u[6] + 8*u[5]*u[7] + 9*u[6]**2 + 12*u[6]*u[7] + 9*u[7]**2)/108000,
           (54*u[0]**2 + 72*u[0]*u[1] + 18*u[0]*u[2] + 27*u[0]*u[3] + 72*u[0]*u[4] + 48*u[0]*u[5] + 12*u[0]*u[6] + 18*u[0]*u[7] + 54*u[1]**2 + 27*u[1]*u[2] + 18*u[1]*u[3] + 48*u[1]*u[4] + 72*u[1]*u[5] + 18*u[1]*u[6] + 12*u[1]*u[7] + 9*u[2]**2 + 12*u[2]*u[3] + 12*u[2]*u[4] + 18*u[2]*u[5] + 12*u[2]*u[6] + 8*u[2]*u[7] + 9*u[3]**2 + 18*u[3]*u[4] + 12*u[3]*u[5] + 8*u[3]*u[6] + 12*u[3]*u[7] + 54*u[4]**2 + 72*u[4]*u[5] + 18*u[4]*u[6] + 27*u[4]*u[7] + 54*u[5]**2 + 27*u[5]*u[6] + 18*u[5]*u[7] + 9*u[6]**2 + 12*u[6]*u[7] + 9*u[7]**2)/108000,
           (36*u[0]**2 + 108*u[0]*u[1] + 27*u[0]*u[2] + 18*u[0]*u[3] + 48*u[0]*u[4] + 72*u[0]*u[5] + 18*u[0]*u[6] + 12*u[0]*u[7] + 216*u[1]**2 + 108*u[1]*u[2] + 27*u[1]*u[3] + 72*u[1]*u[4] + 288*u[1]*u[5] + 72*u[1]*u[6] + 18*u[1]*u[7] + 36*u[2]**2 + 18*u[2]*u[3] + 18*u[2]*u[4] + 72*u[2]*u[5] + 48*u[2]*u[6] + 12*u[2]*u[7] + 6*u[3]**2 + 12*u[3]*u[4] + 18*u[3]*u[5] + 12*u[3]*u[6] + 8*u[3]*u[7] + 36*u[4]**2 + 108*u[4]*u[5] + 27*u[4]*u[6] + 18*u[4]*u[7] + 216*u[5]**2 + 108*u[5]*u[6] + 27*u[5]*u[7] + 36*u[6]**2 + 18*u[6]*u[7] + 6*u[7]**2)/108000,
           (9*u[0]**2 + 27*u[0]*u[1] + 18*u[0]*u[2] + 12*u[0]*u[3] + 12*u[0]*u[4] + 18*u[0]*u[5] + 12*u[0]*u[6] + 8*u[0]*u[7] + 54*u[1]**2 + 72*u[1]*u[2] + 18*u[1]*u[3] + 18*u[1]*u[4] + 72*u[1]*u[5] + 48*u[1]*u[6] + 12*u[1]*u[7] + 54*u[2]**2 + 27*u[2]*u[3] + 12*u[2]*u[4] + 48*u[2]*u[5] + 72*u[2]*u[6] + 18*u[2]*u[7] + 9*u[3]**2 + 8*u[3]*u[4] + 12*u[3]*u[5] + 18*u[3]*u[6] + 12*u[3]*u[7] + 9*u[4]**2 + 27*u[4]*u[5] + 18*u[4]*u[6] + 12*u[4]*u[7] + 54*u[5]**2 + 72*u[5]*u[6] + 18*u[5]*u[7] + 54*u[6]**2 + 27*u[6]*u[7] + 9*u[7]**2)/108000,
           (27*u[0]**2 + 36*u[0]*u[1] + 24*u[0]*u[2] + 36*u[0]*u[3] + 36*u[0]*u[4] + 24*u[0]*u[5] + 16*u[0]*u[6] + 24*u[0]*u[7] + 27*u[1]**2 + 36*u[1]*u[2] + 24*u[1]*u[3] + 24*u[1]*u[4] + 36*u[1]*u[5] + 24*u[1]*u[6] + 16*u[1]*u[7] + 27*u[2]**2 + 36*u[2]*u[3] + 16*u[2]*u[4] + 24*u[2]*u[5] + 36*u[2]*u[6] + 24*u[2]*u[7] + 27*u[3]**2 + 24*u[3]*u[4] + 16*u[3]*u[5] + 24*u[3]*u[6] + 36*u[3]*u[7] + 27*u[4]**2 + 36*u[4]*u[5] + 24*u[4]*u[6] + 36*u[4]*u[7] + 27*u[5]**2 + 36*u[5]*u[6] + 24*u[5]*u[7] + 27*u[6]**2 + 36*u[6]*u[7] + 27*u[7]**2)/216000],
          [(54*u[0]**2 + 72*u[0]*u[1] + 48*u[0]*u[2] + 72*u[0]*u[3] + 27*u[0]*u[4] + 18*u[0]*u[5] + 12*u[0]*u[6] + 18*u[0]*u[7] + 54*u[1]**2 + 72*u[1]*u[2] + 48*u[1]*u[3] + 18*u[1]*u[4] + 27*u[1]*u[5] + 18*u[1]*u[6] + 12*u[1]*u[7] + 54*u[2]**2 + 72*u[2]*u[3] + 12*u[2]*u[4] + 18*u[2]*u[5] + 27*u[2]*u[6] + 18*u[2]*u[7] + 54*u[3]**2 + 18*u[3]*u[4] + 12*u[3]*u[5] + 18*u[3]*u[6] + 27*u[3]*u[7] + 9*u[4]**2 + 12*u[4]*u[5] + 8*u[4]*u[6] + 12*u[4]*u[7] + 9*u[5]**2 + 12*u[5]*u[6] + 8*u[5]*u[7] + 9*u[6]**2 + 12*u[6]*u[7] + 9*u[7]**2)/108000,
           (36*u[0]**2 + 108*u[0]*u[1] + 72*u[0]*u[2] + 48*u[0]*u[3] + 18*u[0]*u[4] + 27*u[0]*u[5] + 18*u[0]*u[6] + 12*u[0]*u[7] + 216*u[1]**2 + 288*u[1]*u[2] + 72*u[1]*u[3] + 27*u[1]*u[4] + 108*u[1]*u[5] + 72*u[1]*u[6] + 18*u[1]*u[7] + 216*u[2]**2 + 108*u[2]*u[3] + 18*u[2]*u[4] + 72*u[2]*u[5] + 108*u[2]*u[6] + 27*u[2]*u[7] + 36*u[3]**2 + 12*u[3]*u[4] + 18*u[3]*u[5] + 27*u[3]*u[6] + 18*u[3]*u[7] + 6*u[4]**2 + 18*u[4]*u[5] + 12*u[4]*u[6] + 8*u[4]*u[7] + 36*u[5]**2 + 48*u[5]*u[6] + 12*u[5]*u[7] + 36*u[6]**2 + 18*u[6]*u[7] + 6*u[7]**2)/108000,
           (24*u[0]**2 + 72*u[0]*u[1] + 108*u[0]*u[2] + 72*u[0]*u[3] + 12*u[0]*u[4] + 18*u[0]*u[5] + 27*u[0]*u[6] + 18*u[0]*u[7] + 144*u[1]**2 + 432*u[1]*u[2] + 108*u[1]*u[3] + 18*u[1]*u[4] + 72*u[1]*u[5] + 108*u[1]*u[6] + 27*u[1]*u[7] + 864*u[2]**2 + 432*u[2]*u[3] + 27*u[2]*u[4] + 108*u[2]*u[5] + 432*u[2]*u[6] + 108*u[2]*u[7] + 144*u[3]**2 + 18*u[3]*u[4] + 27*u[3]*u[5] + 108*u[3]*u[6] + 72*u[3]*u[7] + 4*u[4]**2 + 12*u[4]*u[5] + 18*u[4]*u[6] + 12*u[4]*u[7] + 24*u[5]**2 + 72*u[5]*u[6] + 18*u[5]*u[7] + 144*u[6]**2 + 72*u[6]*u[7] + 24*u[7]**2)/108000,
           (36*u[0]**2 + 48*u[0]*u[1] + 72*u[0]*u[2] + 108*u[0]*u[3] + 18*u[0]*u[4] + 12*u[0]*u[5] + 18*u[0]*u[6] + 27*u[0]*u[7] + 36*u[1]**2 + 108*u[1]*u[2] + 72*u[1]*u[3] + 12*u[1]*u[4] + 18*u[1]*u[5] + 27*u[1]*u[6] + 18*u[1]*u[7] + 216*u[2]**2 + 288*u[2]*u[3] + 18*u[2]*u[4] + 27*u[2]*u[5] + 108*u[2]*u[6] + 72*u[2]*u[7] + 216*u[3]**2 + 27*u[3]*u[4] + 18*u[3]*u[5] + 72*u[3]*u[6] + 108*u[3]*u[7] + 6*u[4]**2 + 8*u[4]*u[5] + 12*u[4]*u[6] + 18*u[4]*u[7] + 6*u[5]**2 + 18*u[5]*u[6] + 12*u[5]*u[7] + 36*u[6]**2 + 48*u[6]*u[7] + 36*u[7]**2)/108000,
           (27*u[0]**2 + 36*u[0]*u[1] + 24*u[0]*u[2] + 36*u[0]*u[3] + 36*u[0]*u[4] + 24*u[0]*u[5] + 16*u[0]*u[6] + 24*u[0]*u[7] + 27*u[1]**2 + 36*u[1]*u[2] + 24*u[1]*u[3] + 24*u[1]*u[4] + 36*u[1]*u[5] + 24*u[1]*u[6] + 16*u[1]*u[7] + 27*u[2]**2 + 36*u[2]*u[3] + 16*u[2]*u[4] + 24*u[2]*u[5] + 36*u[2]*u[6] + 24*u[2]*u[7] + 27*u[3]**2 + 24*u[3]*u[4] + 16*u[3]*u[5] + 24*u[3]*u[6] + 36*u[3]*u[7] + 27*u[4]**2 + 36*u[4]*u[5] + 24*u[4]*u[6] + 36*u[4]*u[7] + 27*u[5]**2 + 36*u[5]*u[6] + 24*u[5]*u[7] + 27*u[6]**2 + 36*u[6]*u[7] + 27*u[7]**2)/216000,
           (9*u[0]**2 + 27*u[0]*u[1] + 18*u[0]*u[2] + 12*u[0]*u[3] + 12*u[0]*u[4] + 18*u[0]*u[5] + 12*u[0]*u[6] + 8*u[0]*u[7] + 54*u[1]**2 + 72*u[1]*u[2] + 18*u[1]*u[3] + 18*u[1]*u[4] + 72*u[1]*u[5] + 48*u[1]*u[6] + 12*u[1]*u[7] + 54*u[2]**2 + 27*u[2]*u[3] + 12*u[2]*u[4] + 48*u[2]*u[5] + 72*u[2]*u[6] + 18*u[2]*u[7] + 9*u[3]**2 + 8*u[3]*u[4] + 12*u[3]*u[5] + 18*u[3]*u[6] + 12*u[3]*u[7] + 9*u[4]**2 + 27*u[4]*u[5] + 18*u[4]*u[6] + 12*u[4]*u[7] + 54*u[5]**2 + 72*u[5]*u[6] + 18*u[5]*u[7] + 54*u[6]**2 + 27*u[6]*u[7] + 9*u[7]**2)/108000,
           (6*u[0]**2 + 18*u[0]*u[1] + 27*u[0]*u[2] + 18*u[0]*u[3] + 8*u[0]*u[4] + 12*u[0]*u[5] + 18*u[0]*u[6] + 12*u[0]*u[7] + 36*u[1]**2 + 108*u[1]*u[2] + 27*u[1]*u[3] + 12*u[1]*u[4] + 48*u[1]*u[5] + 72*u[1]*u[6] + 18*u[1]*u[7] + 216*u[2]**2 + 108*u[2]*u[3] + 18*u[2]*u[4] + 72*u[2]*u[5] + 288*u[2]*u[6] + 72*u[2]*u[7] + 36*u[3]**2 + 12*u[3]*u[4] + 18*u[3]*u[5] + 72*u[3]*u[6] + 48*u[3]*u[7] + 6*u[4]**2 + 18*u[4]*u[5] + 27*u[4]*u[6] + 18*u[4]*u[7] + 36*u[5]**2 + 108*u[5]*u[6] + 27*u[5]*u[7] + 216*u[6]**2 + 108*u[6]*u[7] + 36*u[7]**2)/108000,
           (9*u[0]**2 + 12*u[0]*u[1] + 18*u[0]*u[2] + 27*u[0]*u[3] + 12*u[0]*u[4] + 8*u[0]*u[5] + 12*u[0]*u[6] + 18*u[0]*u[7] + 9*u[1]**2 + 27*u[1]*u[2] + 18*u[1]*u[3] + 8*u[1]*u[4] + 12*u[1]*u[5] + 18*u[1]*u[6] + 12*u[1]*u[7] + 54*u[2]**2 + 72*u[2]*u[3] + 12*u[2]*u[4] + 18*u[2]*u[5] + 72*u[2]*u[6] + 48*u[2]*u[7] + 54*u[3]**2 + 18*u[3]*u[4] + 12*u[3]*u[5] + 48*u[3]*u[6] + 72*u[3]*u[7] + 9*u[4]**2 + 12*u[4]*u[5] + 18*u[4]*u[6] + 27*u[4]*u[7] + 9*u[5]**2 + 27*u[5]*u[6] + 18*u[5]*u[7] + 54*u[6]**2 + 72*u[6]*u[7] + 54*u[7]**2)/108000],
          [(216*u[0]**2 + 108*u[0]*u[1] + 72*u[0]*u[2] + 288*u[0]*u[3] + 108*u[0]*u[4] + 27*u[0]*u[5] + 18*u[0]*u[6] + 72*u[0]*u[7] + 36*u[1]**2 + 48*u[1]*u[2] + 72*u[1]*u[3] + 27*u[1]*u[4] + 18*u[1]*u[5] + 12*u[1]*u[6] + 18*u[1]*u[7] + 36*u[2]**2 + 108*u[2]*u[3] + 18*u[2]*u[4] + 12*u[2]*u[5] + 18*u[2]*u[6] + 27*u[2]*u[7] + 216*u[3]**2 + 72*u[3]*u[4] + 18*u[3]*u[5] + 27*u[3]*u[6] + 108*u[3]*u[7] + 36*u[4]**2 + 18*u[4]*u[5] + 12*u[4]*u[6] + 48*u[4]*u[7] + 6*u[5]**2 + 8*u[5]*u[6] + 12*u[5]*u[7] + 6*u[6]**2 + 18*u[6]*u[7] + 36*u[7]**2)/108000,
           (54*u[0]**2 + 72*u[0]*u[1] + 48*u[0]*u[2] + 72*u[0]*u[3] + 27*u[0]*u[4] + 18*u[0]*u[5] + 12*u[0]*u[6] + 18*u[0]*u[7] + 54*u[1]**2 + 72*u[1]*u[2] + 48*u[1]*u[3] + 18*u[1]*u[4] + 27*u[1]*u[5] + 18*u[1]*u[6] + 12*u[1]*u[7] + 54*u[2]**2 + 72*u[2]*u[3] + 12*u[2]*u[4] + 18*u[2]*u[5] + 27*u[2]*u[6] + 18*u[2]*u[7] + 54*u[3]**2 + 18*u[3]*u[4] + 12*u[3]*u[5] + 18*u[3]*u[6] + 27*u[3]*u[7] + 9*u[4]**2 + 12*u[4]*u[5] + 8*u[4]*u[6] + 12*u[4]*u[7] + 9*u[5]**2 + 12*u[5]*u[6] + 8*u[5]*u[7] + 9*u[6]**2 + 12*u[6]*u[7] + 9*u[7]**2)/108000,
           (36*u[0]**2 + 48*u[0]*u[1] + 72*u[0]*u[2] + 108*u[0]*u[3] + 18*u[0]*u[4] + 12*u[0]*u[5] + 18*u[0]*u[6] + 27*u[0]*u[7] + 36*u[1]**2 + 108*u[1]*u[2] + 72*u[1]*u[3] + 12*u[1]*u[4] + 18*u[1]*u[5] + 27*u[1]*u[6] + 18*u[1]*u[7] + 216*u[2]**2 + 288*u[2]*u[3] + 18*u[2]*u[4] + 27*u[2]*u[5] + 108*u[2]*u[6] + 72*u[2]*u[7] + 216*u[3]**2 + 27*u[3]*u[4] + 18*u[3]*u[5] + 72*u[3]*u[6] + 108*u[3]*u[7] + 6*u[4]**2 + 8*u[4]*u[5] + 12*u[4]*u[6] + 18*u[4]*u[7] + 6*u[5]**2 + 18*u[5]*u[6] + 12*u[5]*u[7] + 36*u[6]**2 + 48*u[6]*u[7] + 36*u[7]**2)/108000,
           (144*u[0]**2 + 72*u[0]*u[1] + 108*u[0]*u[2] + 432*u[0]*u[3] + 72*u[0]*u[4] + 18*u[0]*u[5] + 27*u[0]*u[6] + 108*u[0]*u[7] + 24*u[1]**2 + 72*u[1]*u[2] + 108*u[1]*u[3] + 18*u[1]*u[4] + 12*u[1]*u[5] + 18*u[1]*u[6] + 27*u[1]*u[7] + 144*u[2]**2 + 432*u[2]*u[3] + 27*u[2]*u[4] + 18*u[2]*u[5] + 72*u[2]*u[6] + 108*u[2]*u[7] + 864*u[3]**2 + 108*u[3]*u[4] + 27*u[3]*u[5] + 108*u[3]*u[6] + 432*u[3]*u[7] + 24*u[4]**2 + 12*u[4]*u[5] + 18*u[4]*u[6] + 72*u[4]*u[7] + 4*u[5]**2 + 12*u[5]*u[6] + 18*u[5]*u[7] + 24*u[6]**2 + 72*u[6]*u[7] + 144*u[7]**2)/108000,
           (54*u[0]**2 + 27*u[0]*u[1] + 18*u[0]*u[2] + 72*u[0]*u[3] + 72*u[0]*u[4] + 18*u[0]*u[5] + 12*u[0]*u[6] + 48*u[0]*u[7] + 9*u[1]**2 + 12*u[1]*u[2] + 18*u[1]*u[3] + 18*u[1]*u[4] + 12*u[1]*u[5] + 8*u[1]*u[6] + 12*u[1]*u[7] + 9*u[2]**2 + 27*u[2]*u[3] + 12*u[2]*u[4] + 8*u[2]*u[5] + 12*u[2]*u[6] + 18*u[2]*u[7] + 54*u[3]**2 + 48*u[3]*u[4] + 12*u[3]*u[5] + 18*u[3]*u[6] + 72*u[3]*u[7] + 54*u[4]**2 + 27*u[4]*u[5] + 18*u[4]*u[6] + 72*u[4]*u[7] + 9*u[5]**2 + 12*u[5]*u[6] + 18*u[5]*u[7] + 9*u[6]**2 + 27*u[6]*u[7] + 54*u[7]**2)/108000,
           (27*u[0]**2 + 36*u[0]*u[1] + 24*u[0]*u[2] + 36*u[0]*u[3] + 36*u[0]*u[4] + 24*u[0]*u[5] + 16*u[0]*u[6] + 24*u[0]*u[7] + 27*u[1]**2 + 36*u[1]*u[2] + 24*u[1]*u[3] + 24*u[1]*u[4] + 36*u[1]*u[5] + 24*u[1]*u[6] + 16*u[1]*u[7] + 27*u[2]**2 + 36*u[2]*u[3] + 16*u[2]*u[4] + 24*u[2]*u[5] + 36*u[2]*u[6] + 24*u[2]*u[7] + 27*u[3]**2 + 24*u[3]*u[4] + 16*u[3]*u[5] + 24*u[3]*u[6] + 36*u[3]*u[7] + 27*u[4]**2 + 36*u[4]*u[5] + 24*u[4]*u[6] + 36*u[4]*u[7] + 27*u[5]**2 + 36*u[5]*u[6] + 24*u[5]*u[7] + 27*u[6]**2 + 36*u[6]*u[7] + 27*u[7]**2)/216000,
           (9*u[0]**2 + 12*u[0]*u[1] + 18*u[0]*u[2] + 27*u[0]*u[3] + 12*u[0]*u[4] + 8*u[0]*u[5] + 12*u[0]*u[6] + 18*u[0]*u[7] + 9*u[1]**2 + 27*u[1]*u[2] + 18*u[1]*u[3] + 8*u[1]*u[4] + 12*u[1]*u[5] + 18*u[1]*u[6] + 12*u[1]*u[7] + 54*u[2]**2 + 72*u[2]*u[3] + 12*u[2]*u[4] + 18*u[2]*u[5] + 72*u[2]*u[6] + 48*u[2]*u[7] + 54*u[3]**2 + 18*u[3]*u[4] + 12*u[3]*u[5] + 48*u[3]*u[6] + 72*u[3]*u[7] + 9*u[4]**2 + 12*u[4]*u[5] + 18*u[4]*u[6] + 27*u[4]*u[7] + 9*u[5]**2 + 27*u[5]*u[6] + 18*u[5]*u[7] + 54*u[6]**2 + 72*u[6]*u[7] + 54*u[7]**2)/108000,
           (36*u[0]**2 + 18*u[0]*u[1] + 27*u[0]*u[2] + 108*u[0]*u[3] + 48*u[0]*u[4] + 12*u[0]*u[5] + 18*u[0]*u[6] + 72*u[0]*u[7] + 6*u[1]**2 + 18*u[1]*u[2] + 27*u[1]*u[3] + 12*u[1]*u[4] + 8*u[1]*u[5] + 12*u[1]*u[6] + 18*u[1]*u[7] + 36*u[2]**2 + 108*u[2]*u[3] + 18*u[2]*u[4] + 12*u[2]*u[5] + 48*u[2]*u[6] + 72*u[2]*u[7] + 216*u[3]**2 + 72*u[3]*u[4] + 18*u[3]*u[5] + 72*u[3]*u[6] + 288*u[3]*u[7] + 36*u[4]**2 + 18*u[4]*u[5] + 27*u[4]*u[6] + 108*u[4]*u[7] + 6*u[5]**2 + 18*u[5]*u[6] + 27*u[5]*u[7] + 36*u[6]**2 + 108*u[6]*u[7] + 216*u[7]**2)/108000],
          [(216*u[0]**2 + 108*u[0]*u[1] + 27*u[0]*u[2] + 108*u[0]*u[3] + 288*u[0]*u[4] + 72*u[0]*u[5] + 18*u[0]*u[6] + 72*u[0]*u[7] + 36*u[1]**2 + 18*u[1]*u[2] + 27*u[1]*u[3] + 72*u[1]*u[4] + 48*u[1]*u[5] + 12*u[1]*u[6] + 18*u[1]*u[7] + 6*u[2]**2 + 18*u[2]*u[3] + 18*u[2]*u[4] + 12*u[2]*u[5] + 8*u[2]*u[6] + 12*u[2]*u[7] + 36*u[3]**2 + 72*u[3]*u[4] + 18*u[3]*u[5] + 12*u[3]*u[6] + 48*u[3]*u[7] + 216*u[4]**2 + 108*u[4]*u[5] + 27*u[4]*u[6] + 108*u[4]*u[7] + 36*u[5]**2 + 18*u[5]*u[6] + 27*u[5]*u[7] + 6*u[6]**2 + 18*u[6]*u[7] + 36*u[7]**2)/108000,
           (54*u[0]**2 + 72*u[0]*u[1] + 18*u[0]*u[2] + 27*u[0]*u[3] + 72*u[0]*u[4] + 48*u[0]*u[5] + 12*u[0]*u[6] + 18*u[0]*u[7] + 54*u[1]**2 + 27*u[1]*u[2] + 18*u[1]*u[3] + 48*u[1]*u[4] + 72*u[1]*u[5] + 18*u[1]*u[6] + 12*u[1]*u[7] + 9*u[2]**2 + 12*u[2]*u[3] + 12*u[2]*u[4] + 18*u[2]*u[5] + 12*u[2]*u[6] + 8*u[2]*u[7] + 9*u[3]**2 + 18*u[3]*u[4] + 12*u[3]*u[5] + 8*u[3]*u[6] + 12*u[3]*u[7] + 54*u[4]**2 + 72*u[4]*u[5] + 18*u[4]*u[6] + 27*u[4]*u[7] + 54*u[5]**2 + 27*u[5]*u[6] + 18*u[5]*u[7] + 9*u[6]**2 + 12*u[6]*u[7] + 9*u[7]**2)/108000,
           (27*u[0]**2 + 36*u[0]*u[1] + 24*u[0]*u[2] + 36*u[0]*u[3] + 36*u[0]*u[4] + 24*u[0]*u[5] + 16*u[0]*u[6] + 24*u[0]*u[7] + 27*u[1]**2 + 36*u[1]*u[2] + 24*u[1]*u[3] + 24*u[1]*u[4] + 36*u[1]*u[5] + 24*u[1]*u[6] + 16*u[1]*u[7] + 27*u[2]**2 + 36*u[2]*u[3] + 16*u[2]*u[4] + 24*u[2]*u[5] + 36*u[2]*u[6] + 24*u[2]*u[7] + 27*u[3]**2 + 24*u[3]*u[4] + 16*u[3]*u[5] + 24*u[3]*u[6] + 36*u[3]*u[7] + 27*u[4]**2 + 36*u[4]*u[5] + 24*u[4]*u[6] + 36*u[4]*u[7] + 27*u[5]**2 + 36*u[5]*u[6] + 24*u[5]*u[7] + 27*u[6]**2 + 36*u[6]*u[7] + 27*u[7]**2)/216000,
           (54*u[0]**2 + 27*u[0]*u[1] + 18*u[0]*u[2] + 72*u[0]*u[3] + 72*u[0]*u[4] + 18*u[0]*u[5] + 12*u[0]*u[6] + 48*u[0]*u[7] + 9*u[1]**2 + 12*u[1]*u[2] + 18*u[1]*u[3] + 18*u[1]*u[4] + 12*u[1]*u[5] + 8*u[1]*u[6] + 12*u[1]*u[7] + 9*u[2]**2 + 27*u[2]*u[3] + 12*u[2]*u[4] + 8*u[2]*u[5] + 12*u[2]*u[6] + 18*u[2]*u[7] + 54*u[3]**2 + 48*u[3]*u[4] + 12*u[3]*u[5] + 18*u[3]*u[6] + 72*u[3]*u[7] + 54*u[4]**2 + 27*u[4]*u[5] + 18*u[4]*u[6] + 72*u[4]*u[7] + 9*u[5]**2 + 12*u[5]*u[6] + 18*u[5]*u[7] + 9*u[6]**2 + 27*u[6]*u[7] + 54*u[7]**2)/108000,
           (144*u[0]**2 + 72*u[0]*u[1] + 18*u[0]*u[2] + 72*u[0]*u[3] + 432*u[0]*u[4] + 108*u[0]*u[5] + 27*u[0]*u[6] + 108*u[0]*u[7] + 24*u[1]**2 + 12*u[1]*u[2] + 18*u[1]*u[3] + 108*u[1]*u[4] + 72*u[1]*u[5] + 18*u[1]*u[6] + 27*u[1]*u[7] + 4*u[2]**2 + 12*u[2]*u[3] + 27*u[2]*u[4] + 18*u[2]*u[5] + 12*u[2]*u[6] + 18*u[2]*u[7] + 24*u[3]**2 + 108*u[3]*u[4] + 27*u[3]*u[5] + 18*u[3]*u[6] + 72*u[3]*u[7] + 864*u[4]**2 + 432*u[4]*u[5] + 108*u[4]*u[6] + 432*u[4]*u[7] + 144*u[5]**2 + 72*u[5]*u[6] + 108*u[5]*u[7] + 24*u[6]**2 + 72*u[6]*u[7] + 144*u[7]**2)/108000,
           (36*u[0]**2 + 48*u[0]*u[1] + 12*u[0]*u[2] + 18*u[0]*u[3] + 108*u[0]*u[4] + 72*u[0]*u[5] + 18*u[0]*u[6] + 27*u[0]*u[7] + 36*u[1]**2 + 18*u[1]*u[2] + 12*u[1]*u[3] + 72*u[1]*u[4] + 108*u[1]*u[5] + 27*u[1]*u[6] + 18*u[1]*u[7] + 6*u[2]**2 + 8*u[2]*u[3] + 18*u[2]*u[4] + 27*u[2]*u[5] + 18*u[2]*u[6] + 12*u[2]*u[7] + 6*u[3]**2 + 27*u[3]*u[4] + 18*u[3]*u[5] + 12*u[3]*u[6] + 18*u[3]*u[7] + 216*u[4]**2 + 288*u[4]*u[5] + 72*u[4]*u[6] + 108*u[4]*u[7] + 216*u[5]**2 + 108*u[5]*u[6] + 72*u[5]*u[7] + 36*u[6]**2 + 48*u[6]*u[7] + 36*u[7]**2)/108000,
           (9*u[0]**2 + 12*u[0]*u[1] + 8*u[0]*u[2] + 12*u[0]*u[3] + 27*u[0]*u[4] + 18*u[0]*u[5] + 12*u[0]*u[6] + 18*u[0]*u[7] + 9*u[1]**2 + 12*u[1]*u[2] + 8*u[1]*u[3] + 18*u[1]*u[4] + 27*u[1]*u[5] + 18*u[1]*u[6] + 12*u[1]*u[7] + 9*u[2]**2 + 12*u[2]*u[3] + 12*u[2]*u[4] + 18*u[2]*u[5] + 27*u[2]*u[6] + 18*u[2]*u[7] + 9*u[3]**2 + 18*u[3]*u[4] + 12*u[3]*u[5] + 18*u[3]*u[6] + 27*u[3]*u[7] + 54*u[4]**2 + 72*u[4]*u[5] + 48*u[4]*u[6] + 72*u[4]*u[7] + 54*u[5]**2 + 72*u[5]*u[6] + 48*u[5]*u[7] + 54*u[6]**2 + 72*u[6]*u[7] + 54*u[7]**2)/108000,
           (36*u[0]**2 + 18*u[0]*u[1] + 12*u[0]*u[2] + 48*u[0]*u[3] + 108*u[0]*u[4] + 27*u[0]*u[5] + 18*u[0]*u[6] + 72*u[0]*u[7] + 6*u[1]**2 + 8*u[1]*u[2] + 12*u[1]*u[3] + 27*u[1]*u[4] + 18*u[1]*u[5] + 12*u[1]*u[6] + 18*u[1]*u[7] + 6*u[2]**2 + 18*u[2]*u[3] + 18*u[2]*u[4] + 12*u[2]*u[5] + 18*u[2]*u[6] + 27*u[2]*u[7] + 36*u[3]**2 + 72*u[3]*u[4] + 18*u[3]*u[5] + 27*u[3]*u[6] + 108*u[3]*u[7] + 216*u[4]**2 + 108*u[4]*u[5] + 72*u[4]*u[6] + 288*u[4]*u[7] + 36*u[5]**2 + 48*u[5]*u[6] + 72*u[5]*u[7] + 36*u[6]**2 + 108*u[6]*u[7] + 216*u[7]**2)/108000],
          [(54*u[0]**2 + 72*u[0]*u[1] + 18*u[0]*u[2] + 27*u[0]*u[3] + 72*u[0]*u[4] + 48*u[0]*u[5] + 12*u[0]*u[6] + 18*u[0]*u[7] + 54*u[1]**2 + 27*u[1]*u[2] + 18*u[1]*u[3] + 48*u[1]*u[4] + 72*u[1]*u[5] + 18*u[1]*u[6] + 12*u[1]*u[7] + 9*u[2]**2 + 12*u[2]*u[3] + 12*u[2]*u[4] + 18*u[2]*u[5] + 12*u[2]*u[6] + 8*u[2]*u[7] + 9*u[3]**2 + 18*u[3]*u[4] + 12*u[3]*u[5] + 8*u[3]*u[6] + 12*u[3]*u[7] + 54*u[4]**2 + 72*u[4]*u[5] + 18*u[4]*u[6] + 27*u[4]*u[7] + 54*u[5]**2 + 27*u[5]*u[6] + 18*u[5]*u[7] + 9*u[6]**2 + 12*u[6]*u[7] + 9*u[7]**2)/108000,
           (36*u[0]**2 + 108*u[0]*u[1] + 27*u[0]*u[2] + 18*u[0]*u[3] + 48*u[0]*u[4] + 72*u[0]*u[5] + 18*u[0]*u[6] + 12*u[0]*u[7] + 216*u[1]**2 + 108*u[1]*u[2] + 27*u[1]*u[3] + 72*u[1]*u[4] + 288*u[1]*u[5] + 72*u[1]*u[6] + 18*u[1]*u[7] + 36*u[2]**2 + 18*u[2]*u[3] + 18*u[2]*u[4] + 72*u[2]*u[5] + 48*u[2]*u[6] + 12*u[2]*u[7] + 6*u[3]**2 + 12*u[3]*u[4] + 18*u[3]*u[5] + 12*u[3]*u[6] + 8*u[3]*u[7] + 36*u[4]**2 + 108*u[4]*u[5] + 27*u[4]*u[6] + 18*u[4]*u[7] + 216*u[5]**2 + 108*u[5]*u[6] + 27*u[5]*u[7] + 36*u[6]**2 + 18*u[6]*u[7] + 6*u[7]**2)/108000,
           (9*u[0]**2 + 27*u[0]*u[1] + 18*u[0]*u[2] + 12*u[0]*u[3] + 12*u[0]*u[4] + 18*u[0]*u[5] + 12*u[0]*u[6] + 8*u[0]*u[7] + 54*u[1]**2 + 72*u[1]*u[2] + 18*u[1]*u[3] + 18*u[1]*u[4] + 72*u[1]*u[5] + 48*u[1]*u[6] + 12*u[1]*u[7] + 54*u[2]**2 + 27*u[2]*u[3] + 12*u[2]*u[4] + 48*u[2]*u[5] + 72*u[2]*u[6] + 18*u[2]*u[7] + 9*u[3]**2 + 8*u[3]*u[4] + 12*u[3]*u[5] + 18*u[3]*u[6] + 12*u[3]*u[7] + 9*u[4]**2 + 27*u[4]*u[5] + 18*u[4]*u[6] + 12*u[4]*u[7] + 54*u[5]**2 + 72*u[5]*u[6] + 18*u[5]*u[7] + 54*u[6]**2 + 27*u[6]*u[7] + 9*u[7]**2)/108000,
           (27*u[0]**2 + 36*u[0]*u[1] + 24*u[0]*u[2] + 36*u[0]*u[3] + 36*u[0]*u[4] + 24*u[0]*u[5] + 16*u[0]*u[6] + 24*u[0]*u[7] + 27*u[1]**2 + 36*u[1]*u[2] + 24*u[1]*u[3] + 24*u[1]*u[4] + 36*u[1]*u[5] + 24*u[1]*u[6] + 16*u[1]*u[7] + 27*u[2]**2 + 36*u[2]*u[3] + 16*u[2]*u[4] + 24*u[2]*u[5] + 36*u[2]*u[6] + 24*u[2]*u[7] + 27*u[3]**2 + 24*u[3]*u[4] + 16*u[3]*u[5] + 24*u[3]*u[6] + 36*u[3]*u[7] + 27*u[4]**2 + 36*u[4]*u[5] + 24*u[4]*u[6] + 36*u[4]*u[7] + 27*u[5]**2 + 36*u[5]*u[6] + 24*u[5]*u[7] + 27*u[6]**2 + 36*u[6]*u[7] + 27*u[7]**2)/216000,
           (36*u[0]**2 + 48*u[0]*u[1] + 12*u[0]*u[2] + 18*u[0]*u[3] + 108*u[0]*u[4] + 72*u[0]*u[5] + 18*u[0]*u[6] + 27*u[0]*u[7] + 36*u[1]**2 + 18*u[1]*u[2] + 12*u[1]*u[3] + 72*u[1]*u[4] + 108*u[1]*u[5] + 27*u[1]*u[6] + 18*u[1]*u[7] + 6*u[2]**2 + 8*u[2]*u[3] + 18*u[2]*u[4] + 27*u[2]*u[5] + 18*u[2]*u[6] + 12*u[2]*u[7] + 6*u[3]**2 + 27*u[3]*u[4] + 18*u[3]*u[5] + 12*u[3]*u[6] + 18*u[3]*u[7] + 216*u[4]**2 + 288*u[4]*u[5] + 72*u[4]*u[6] + 108*u[4]*u[7] + 216*u[5]**2 + 108*u[5]*u[6] + 72*u[5]*u[7] + 36*u[6]**2 + 48*u[6]*u[7] + 36*u[7]**2)/108000,
           (24*u[0]**2 + 72*u[0]*u[1] + 18*u[0]*u[2] + 12*u[0]*u[3] + 72*u[0]*u[4] + 108*u[0]*u[5] + 27*u[0]*u[6] + 18*u[0]*u[7] + 144*u[1]**2 + 72*u[1]*u[2] + 18*u[1]*u[3] + 108*u[1]*u[4] + 432*u[1]*u[5] + 108*u[1]*u[6] + 27*u[1]*u[7] + 24*u[2]**2 + 12*u[2]*u[3] + 27*u[2]*u[4] + 108*u[2]*u[5] + 72*u[2]*u[6] + 18*u[2]*u[7] + 4*u[3]**2 + 18*u[3]*u[4] + 27*u[3]*u[5] + 18*u[3]*u[6] + 12*u[3]*u[7] + 144*u[4]**2 + 432*u[4]*u[5] + 108*u[4]*u[6] + 72*u[4]*u[7] + 864*u[5]**2 + 432*u[5]*u[6] + 108*u[5]*u[7] + 144*u[6]**2 + 72*u[6]*u[7] + 24*u[7]**2)/108000,
           (6*u[0]**2 + 18*u[0]*u[1] + 12*u[0]*u[2] + 8*u[0]*u[3] + 18*u[0]*u[4] + 27*u[0]*u[5] + 18*u[0]*u[6] + 12*u[0]*u[7] + 36*u[1]**2 + 48*u[1]*u[2] + 12*u[1]*u[3] + 27*u[1]*u[4] + 108*u[1]*u[5] + 72*u[1]*u[6] + 18*u[1]*u[7] + 36*u[2]**2 + 18*u[2]*u[3] + 18*u[2]*u[4] + 72*u[2]*u[5] + 108*u[2]*u[6] + 27*u[2]*u[7] + 6*u[3]**2 + 12*u[3]*u[4] + 18*u[3]*u[5] + 27*u[3]*u[6] + 18*u[3]*u[7] + 36*u[4]**2 + 108*u[4]*u[5] + 72*u[4]*u[6] + 48*u[4]*u[7] + 216*u[5]**2 + 288*u[5]*u[6] + 72*u[5]*u[7] + 216*u[6]**2 + 108*u[6]*u[7] + 36*u[7]**2)/108000,
           (9*u[0]**2 + 12*u[0]*u[1] + 8*u[0]*u[2] + 12*u[0]*u[3] + 27*u[0]*u[4] + 18*u[0]*u[5] + 12*u[0]*u[6] + 18*u[0]*u[7] + 9*u[1]**2 + 12*u[1]*u[2] + 8*u[1]*u[3] + 18*u[1]*u[4] + 27*u[1]*u[5] + 18*u[1]*u[6] + 12*u[1]*u[7] + 9*u[2]**2 + 12*u[2]*u[3] + 12*u[2]*u[4] + 18*u[2]*u[5] + 27*u[2]*u[6] + 18*u[2]*u[7] + 9*u[3]**2 + 18*u[3]*u[4] + 12*u[3]*u[5] + 18*u[3]*u[6] + 27*u[3]*u[7] + 54*u[4]**2 + 72*u[4]*u[5] + 48*u[4]*u[6] + 72*u[4]*u[7] + 54*u[5]**2 + 72*u[5]*u[6] + 48*u[5]*u[7] + 54*u[6]**2 + 72*u[6]*u[7] + 54*u[7]**2)/108000],
          [(27*u[0]**2 + 36*u[0]*u[1] + 24*u[0]*u[2] + 36*u[0]*u[3] + 36*u[0]*u[4] + 24*u[0]*u[5] + 16*u[0]*u[6] + 24*u[0]*u[7] + 27*u[1]**2 + 36*u[1]*u[2] + 24*u[1]*u[3] + 24*u[1]*u[4] + 36*u[1]*u[5] + 24*u[1]*u[6] + 16*u[1]*u[7] + 27*u[2]**2 + 36*u[2]*u[3] + 16*u[2]*u[4] + 24*u[2]*u[5] + 36*u[2]*u[6] + 24*u[2]*u[7] + 27*u[3]**2 + 24*u[3]*u[4] + 16*u[3]*u[5] + 24*u[3]*u[6] + 36*u[3]*u[7] + 27*u[4]**2 + 36*u[4]*u[5] + 24*u[4]*u[6] + 36*u[4]*u[7] + 27*u[5]**2 + 36*u[5]*u[6] + 24*u[5]*u[7] + 27*u[6]**2 + 36*u[6]*u[7] + 27*u[7]**2)/216000,
           (9*u[0]**2 + 27*u[0]*u[1] + 18*u[0]*u[2] + 12*u[0]*u[3] + 12*u[0]*u[4] + 18*u[0]*u[5] + 12*u[0]*u[6] + 8*u[0]*u[7] + 54*u[1]**2 + 72*u[1]*u[2] + 18*u[1]*u[3] + 18*u[1]*u[4] + 72*u[1]*u[5] + 48*u[1]*u[6] + 12*u[1]*u[7] + 54*u[2]**2 + 27*u[2]*u[3] + 12*u[2]*u[4] + 48*u[2]*u[5] + 72*u[2]*u[6] + 18*u[2]*u[7] + 9*u[3]**2 + 8*u[3]*u[4] + 12*u[3]*u[5] + 18*u[3]*u[6] + 12*u[3]*u[7] + 9*u[4]**2 + 27*u[4]*u[5] + 18*u[4]*u[6] + 12*u[4]*u[7] + 54*u[5]**2 + 72*u[5]*u[6] + 18*u[5]*u[7] + 54*u[6]**2 + 27*u[6]*u[7] + 9*u[7]**2)/108000,
           (6*u[0]**2 + 18*u[0]*u[1] + 27*u[0]*u[2] + 18*u[0]*u[3] + 8*u[0]*u[4] + 12*u[0]*u[5] + 18*u[0]*u[6] + 12*u[0]*u[7] + 36*u[1]**2 + 108*u[1]*u[2] + 27*u[1]*u[3] + 12*u[1]*u[4] + 48*u[1]*u[5] + 72*u[1]*u[6] + 18*u[1]*u[7] + 216*u[2]**2 + 108*u[2]*u[3] + 18*u[2]*u[4] + 72*u[2]*u[5] + 288*u[2]*u[6] + 72*u[2]*u[7] + 36*u[3]**2 + 12*u[3]*u[4] + 18*u[3]*u[5] + 72*u[3]*u[6] + 48*u[3]*u[7] + 6*u[4]**2 + 18*u[4]*u[5] + 27*u[4]*u[6] + 18*u[4]*u[7] + 36*u[5]**2 + 108*u[5]*u[6] + 27*u[5]*u[7] + 216*u[6]**2 + 108*u[6]*u[7] + 36*u[7]**2)/108000,
           (9*u[0]**2 + 12*u[0]*u[1] + 18*u[0]*u[2] + 27*u[0]*u[3] + 12*u[0]*u[4] + 8*u[0]*u[5] + 12*u[0]*u[6] + 18*u[0]*u[7] + 9*u[1]**2 + 27*u[1]*u[2] + 18*u[1]*u[3] + 8*u[1]*u[4] + 12*u[1]*u[5] + 18*u[1]*u[6] + 12*u[1]*u[7] + 54*u[2]**2 + 72*u[2]*u[3] + 12*u[2]*u[4] + 18*u[2]*u[5] + 72*u[2]*u[6] + 48*u[2]*u[7] + 54*u[3]**2 + 18*u[3]*u[4] + 12*u[3]*u[5] + 48*u[3]*u[6] + 72*u[3]*u[7] + 9*u[4]**2 + 12*u[4]*u[5] + 18*u[4]*u[6] + 27*u[4]*u[7] + 9*u[5]**2 + 27*u[5]*u[6] + 18*u[5]*u[7] + 54*u[6]**2 + 72*u[6]*u[7] + 54*u[7]**2)/108000,
           (9*u[0]**2 + 12*u[0]*u[1] + 8*u[0]*u[2] + 12*u[0]*u[3] + 27*u[0]*u[4] + 18*u[0]*u[5] + 12*u[0]*u[6] + 18*u[0]*u[7] + 9*u[1]**2 + 12*u[1]*u[2] + 8*u[1]*u[3] + 18*u[1]*u[4] + 27*u[1]*u[5] + 18*u[1]*u[6] + 12*u[1]*u[7] + 9*u[2]**2 + 12*u[2]*u[3] + 12*u[2]*u[4] + 18*u[2]*u[5] + 27*u[2]*u[6] + 18*u[2]*u[7] + 9*u[3]**2 + 18*u[3]*u[4] + 12*u[3]*u[5] + 18*u[3]*u[6] + 27*u[3]*u[7] + 54*u[4]**2 + 72*u[4]*u[5] + 48*u[4]*u[6] + 72*u[4]*u[7] + 54*u[5]**2 + 72*u[5]*u[6] + 48*u[5]*u[7] + 54*u[6]**2 + 72*u[6]*u[7] + 54*u[7]**2)/108000,
           (6*u[0]**2 + 18*u[0]*u[1] + 12*u[0]*u[2] + 8*u[0]*u[3] + 18*u[0]*u[4] + 27*u[0]*u[5] + 18*u[0]*u[6] + 12*u[0]*u[7] + 36*u[1]**2 + 48*u[1]*u[2] + 12*u[1]*u[3] + 27*u[1]*u[4] + 108*u[1]*u[5] + 72*u[1]*u[6] + 18*u[1]*u[7] + 36*u[2]**2 + 18*u[2]*u[3] + 18*u[2]*u[4] + 72*u[2]*u[5] + 108*u[2]*u[6] + 27*u[2]*u[7] + 6*u[3]**2 + 12*u[3]*u[4] + 18*u[3]*u[5] + 27*u[3]*u[6] + 18*u[3]*u[7] + 36*u[4]**2 + 108*u[4]*u[5] + 72*u[4]*u[6] + 48*u[4]*u[7] + 216*u[5]**2 + 288*u[5]*u[6] + 72*u[5]*u[7] + 216*u[6]**2 + 108*u[6]*u[7] + 36*u[7]**2)/108000,
           (4*u[0]**2 + 12*u[0]*u[1] + 18*u[0]*u[2] + 12*u[0]*u[3] + 12*u[0]*u[4] + 18*u[0]*u[5] + 27*u[0]*u[6] + 18*u[0]*u[7] + 24*u[1]**2 + 72*u[1]*u[2] + 18*u[1]*u[3] + 18*u[1]*u[4] + 72*u[1]*u[5] + 108*u[1]*u[6] + 27*u[1]*u[7] + 144*u[2]**2 + 72*u[2]*u[3] + 27*u[2]*u[4] + 108*u[2]*u[5] + 432*u[2]*u[6] + 108*u[2]*u[7] + 24*u[3]**2 + 18*u[3]*u[4] + 27*u[3]*u[5] + 108*u[3]*u[6] + 72*u[3]*u[7] + 24*u[4]**2 + 72*u[4]*u[5] + 108*u[4]*u[6] + 72*u[4]*u[7] + 144*u[5]**2 + 432*u[5]*u[6] + 108*u[5]*u[7] + 864*u[6]**2 + 432*u[6]*u[7] + 144*u[7]**2)/108000,
           (6*u[0]**2 + 8*u[0]*u[1] + 12*u[0]*u[2] + 18*u[0]*u[3] + 18*u[0]*u[4] + 12*u[0]*u[5] + 18*u[0]*u[6] + 27*u[0]*u[7] + 6*u[1]**2 + 18*u[1]*u[2] + 12*u[1]*u[3] + 12*u[1]*u[4] + 18*u[1]*u[5] + 27*u[1]*u[6] + 18*u[1]*u[7] + 36*u[2]**2 + 48*u[2]*u[3] + 18*u[2]*u[4] + 27*u[2]*u[5] + 108*u[2]*u[6] + 72*u[2]*u[7] + 36*u[3]**2 + 27*u[3]*u[4] + 18*u[3]*u[5] + 72*u[3]*u[6] + 108*u[3]*u[7] + 36*u[4]**2 + 48*u[4]*u[5] + 72*u[4]*u[6] + 108*u[4]*u[7] + 36*u[5]**2 + 108*u[5]*u[6] + 72*u[5]*u[7] + 216*u[6]**2 + 288*u[6]*u[7] + 216*u[7]**2)/108000],
          [(54*u[0]**2 + 27*u[0]*u[1] + 18*u[0]*u[2] + 72*u[0]*u[3] + 72*u[0]*u[4] + 18*u[0]*u[5] + 12*u[0]*u[6] + 48*u[0]*u[7] + 9*u[1]**2 + 12*u[1]*u[2] + 18*u[1]*u[3] + 18*u[1]*u[4] + 12*u[1]*u[5] + 8*u[1]*u[6] + 12*u[1]*u[7] + 9*u[2]**2 + 27*u[2]*u[3] + 12*u[2]*u[4] + 8*u[2]*u[5] + 12*u[2]*u[6] + 18*u[2]*u[7] + 54*u[3]**2 + 48*u[3]*u[4] + 12*u[3]*u[5] + 18*u[3]*u[6] + 72*u[3]*u[7] + 54*u[4]**2 + 27*u[4]*u[5] + 18*u[4]*u[6] + 72*u[4]*u[7] + 9*u[5]**2 + 12*u[5]*u[6] + 18*u[5]*u[7] + 9*u[6]**2 + 27*u[6]*u[7] + 54*u[7]**2)/108000,
           (27*u[0]**2 + 36*u[0]*u[1] + 24*u[0]*u[2] + 36*u[0]*u[3] + 36*u[0]*u[4] + 24*u[0]*u[5] + 16*u[0]*u[6] + 24*u[0]*u[7] + 27*u[1]**2 + 36*u[1]*u[2] + 24*u[1]*u[3] + 24*u[1]*u[4] + 36*u[1]*u[5] + 24*u[1]*u[6] + 16*u[1]*u[7] + 27*u[2]**2 + 36*u[2]*u[3] + 16*u[2]*u[4] + 24*u[2]*u[5] + 36*u[2]*u[6] + 24*u[2]*u[7] + 27*u[3]**2 + 24*u[3]*u[4] + 16*u[3]*u[5] + 24*u[3]*u[6] + 36*u[3]*u[7] + 27*u[4]**2 + 36*u[4]*u[5] + 24*u[4]*u[6] + 36*u[4]*u[7] + 27*u[5]**2 + 36*u[5]*u[6] + 24*u[5]*u[7] + 27*u[6]**2 + 36*u[6]*u[7] + 27*u[7]**2)/216000,
           (9*u[0]**2 + 12*u[0]*u[1] + 18*u[0]*u[2] + 27*u[0]*u[3] + 12*u[0]*u[4] + 8*u[0]*u[5] + 12*u[0]*u[6] + 18*u[0]*u[7] + 9*u[1]**2 + 27*u[1]*u[2] + 18*u[1]*u[3] + 8*u[1]*u[4] + 12*u[1]*u[5] + 18*u[1]*u[6] + 12*u[1]*u[7] + 54*u[2]**2 + 72*u[2]*u[3] + 12*u[2]*u[4] + 18*u[2]*u[5] + 72*u[2]*u[6] + 48*u[2]*u[7] + 54*u[3]**2 + 18*u[3]*u[4] + 12*u[3]*u[5] + 48*u[3]*u[6] + 72*u[3]*u[7] + 9*u[4]**2 + 12*u[4]*u[5] + 18*u[4]*u[6] + 27*u[4]*u[7] + 9*u[5]**2 + 27*u[5]*u[6] + 18*u[5]*u[7] + 54*u[6]**2 + 72*u[6]*u[7] + 54*u[7]**2)/108000,
           (36*u[0]**2 + 18*u[0]*u[1] + 27*u[0]*u[2] + 108*u[0]*u[3] + 48*u[0]*u[4] + 12*u[0]*u[5] + 18*u[0]*u[6] + 72*u[0]*u[7] + 6*u[1]**2 + 18*u[1]*u[2] + 27*u[1]*u[3] + 12*u[1]*u[4] + 8*u[1]*u[5] + 12*u[1]*u[6] + 18*u[1]*u[7] + 36*u[2]**2 + 108*u[2]*u[3] + 18*u[2]*u[4] + 12*u[2]*u[5] + 48*u[2]*u[6] + 72*u[2]*u[7] + 216*u[3]**2 + 72*u[3]*u[4] + 18*u[3]*u[5] + 72*u[3]*u[6] + 288*u[3]*u[7] + 36*u[4]**2 + 18*u[4]*u[5] + 27*u[4]*u[6] + 108*u[4]*u[7] + 6*u[5]**2 + 18*u[5]*u[6] + 27*u[5]*u[7] + 36*u[6]**2 + 108*u[6]*u[7] + 216*u[7]**2)/108000,
           (36*u[0]**2 + 18*u[0]*u[1] + 12*u[0]*u[2] + 48*u[0]*u[3] + 108*u[0]*u[4] + 27*u[0]*u[5] + 18*u[0]*u[6] + 72*u[0]*u[7] + 6*u[1]**2 + 8*u[1]*u[2] + 12*u[1]*u[3] + 27*u[1]*u[4] + 18*u[1]*u[5] + 12*u[1]*u[6] + 18*u[1]*u[7] + 6*u[2]**2 + 18*u[2]*u[3] + 18*u[2]*u[4] + 12*u[2]*u[5] + 18*u[2]*u[6] + 27*u[2]*u[7] + 36*u[3]**2 + 72*u[3]*u[4] + 18*u[3]*u[5] + 27*u[3]*u[6] + 108*u[3]*u[7] + 216*u[4]**2 + 108*u[4]*u[5] + 72*u[4]*u[6] + 288*u[4]*u[7] + 36*u[5]**2 + 48*u[5]*u[6] + 72*u[5]*u[7] + 36*u[6]**2 + 108*u[6]*u[7] + 216*u[7]**2)/108000,
           (9*u[0]**2 + 12*u[0]*u[1] + 8*u[0]*u[2] + 12*u[0]*u[3] + 27*u[0]*u[4] + 18*u[0]*u[5] + 12*u[0]*u[6] + 18*u[0]*u[7] + 9*u[1]**2 + 12*u[1]*u[2] + 8*u[1]*u[3] + 18*u[1]*u[4] + 27*u[1]*u[5] + 18*u[1]*u[6] + 12*u[1]*u[7] + 9*u[2]**2 + 12*u[2]*u[3] + 12*u[2]*u[4] + 18*u[2]*u[5] + 27*u[2]*u[6] + 18*u[2]*u[7] + 9*u[3]**2 + 18*u[3]*u[4] + 12*u[3]*u[5] + 18*u[3]*u[6] + 27*u[3]*u[7] + 54*u[4]**2 + 72*u[4]*u[5] + 48*u[4]*u[6] + 72*u[4]*u[7] + 54*u[5]**2 + 72*u[5]*u[6] + 48*u[5]*u[7] + 54*u[6]**2 + 72*u[6]*u[7] + 54*u[7]**2)/108000,
           (6*u[0]**2 + 8*u[0]*u[1] + 12*u[0]*u[2] + 18*u[0]*u[3] + 18*u[0]*u[4] + 12*u[0]*u[5] + 18*u[0]*u[6] + 27*u[0]*u[7] + 6*u[1]**2 + 18*u[1]*u[2] + 12*u[1]*u[3] + 12*u[1]*u[4] + 18*u[1]*u[5] + 27*u[1]*u[6] + 18*u[1]*u[7] + 36*u[2]**2 + 48*u[2]*u[3] + 18*u[2]*u[4] + 27*u[2]*u[5] + 108*u[2]*u[6] + 72*u[2]*u[7] + 36*u[3]**2 + 27*u[3]*u[4] + 18*u[3]*u[5] + 72*u[3]*u[6] + 108*u[3]*u[7] + 36*u[4]**2 + 48*u[4]*u[5] + 72*u[4]*u[6] + 108*u[4]*u[7] + 36*u[5]**2 + 108*u[5]*u[6] + 72*u[5]*u[7] + 216*u[6]**2 + 288*u[6]*u[7] + 216*u[7]**2)/108000,
           (24*u[0]**2 + 12*u[0]*u[1] + 18*u[0]*u[2] + 72*u[0]*u[3] + 72*u[0]*u[4] + 18*u[0]*u[5] + 27*u[0]*u[6] + 108*u[0]*u[7] + 4*u[1]**2 + 12*u[1]*u[2] + 18*u[1]*u[3] + 18*u[1]*u[4] + 12*u[1]*u[5] + 18*u[1]*u[6] + 27*u[1]*u[7] + 24*u[2]**2 + 72*u[2]*u[3] + 27*u[2]*u[4] + 18*u[2]*u[5] + 72*u[2]*u[6] + 108*u[2]*u[7] + 144*u[3]**2 + 108*u[3]*u[4] + 27*u[3]*u[5] + 108*u[3]*u[6] + 432*u[3]*u[7] + 144*u[4]**2 + 72*u[4]*u[5] + 108*u[4]*u[6] + 432*u[4]*u[7] + 24*u[5]**2 + 72*u[5]*u[6] + 108*u[5]*u[7] + 144*u[6]**2 + 432*u[6]*u[7] + 864*u[7]**2)/108000]])
