# SPDX-License-Identifier: GPL-3.0-or-later
from typing import Any

import numpy as np

from topoptlab.fem import get_integrpoints
from topoptlab.elements.trilinear_hexahedron import invjacobian,shape_functions_dxi

def _lk_poisson_3d(xe: np.ndarray, k: np.ndarray,
                   quadr_method: str = "gauss-legendre",
                   nquad: int = 2,
                   **kwargs: Any) -> np.ndarray:
    """
    Create element stiffness matrix for 3D Laplacian operator with bilinear
    quadrilateral elements. 
    
    Parameters
    ----------
    xe : np.ndarray, shape (nels,4,2)
        coordinates of element nodes. Please look at the 
        definition/function of the shape function, then the node ordering is 
        clear.
    k : np.ndarray, shape (nels,3,3) or 
        conductivity tensor or something equivalent.
    quadr_method: str or callable
        name of quadrature method or function/callable that returns coordinates of 
        quadrature points and weights. Check function get_integrpoints for 
        available options. 
    nquad : int
        number of quadrature points
        
    Returns
    -------
    Ke : np.ndarray, shape (nels,8,8)
        element stiffness matrix.
        
    """
    #
    if len(xe.shape) == 2:
        xe = xe[None,:,:]
    nel = xe.shape[0]
    #
    if len(k.shape) == 2:
        k = k[None,:,:]
    #
    x,w=get_integrpoints(ndim=3,nq=nquad,method=quadr_method)
    nq =w.shape[0]
    #
    xi,eta,zeta = [_x[:,0] for _x in np.split(x, 3,axis=1)]
    #
    Jinv,detJ = invjacobian(xi=xi,eta=eta,zeta=zeta,xe=xe,
                            all_elems=True,return_det=True)
    Jinv = Jinv.reshape(nel,nq,3,3)
    detJ = detJ.reshape(nel,nq)
    gradN = shape_functions_dxi(xi=xi,eta=eta,zeta=zeta)[None,:,:,:]@\
            Jinv.transpose((0,1,3,2))
    #
    integral = gradN@k[:,None,:,:]@gradN.transpose([0,1,3,2])
    # multiply by determinant and quadrature
    return (w[None,:,None,None]*integral*detJ[:,:,None,None]).sum(axis=1)

def lk_poisson_3d(k: float = 1.,
                  l: np.ndarray= np.array([1.,1.,1.]), 
                  g: np.ndarray = np.array([0.,0.]) ) -> np.ndarray:
    """
    Create element stiffness matrix for 3D Poisson with trilinear
    hexahedral elements. 
    
    Parameters
    ----------
    k : float
        heat conductivity.
    l : np.ndarray (3)
        side length of element.
    g : np.ndarray (2)
        angles of parallelogram.
    
    Returns
    -------
    Ke : np.ndarray, shape (8,8)
        element stiffness matrix.
        
    """
    return k*np.array([[l[0]*l[1]/(9*l[2]) + l[0]*l[2]/(9*l[1]) - l[1]*np.tan(g[1])/6 - l[2]*np.tan(g[0])/6 + l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + l[1]*l[2]/(9*l[0]),
           l[0]*l[1]/(18*l[2]) + l[0]*l[2]/(18*l[1]) - l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - l[1]*l[2]/(9*l[0]),
           l[0]*l[1]/(36*l[2]) - l[0]*l[2]/(18*l[1]) + l[2]*np.tan(g[0])/6 - l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - l[1]*l[2]/(18*l[0]),
           l[0]*l[1]/(18*l[2]) - l[0]*l[2]/(9*l[1]) - l[1]*np.tan(g[1])/12 + l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + l[1]*l[2]/(18*l[0]),
           -l[0]*l[1]/(9*l[2]) + l[0]*l[2]/(18*l[1]) - l[2]*np.tan(g[0])/12 + l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + l[1]*l[2]/(18*l[0]),
           -l[0]*l[1]/(18*l[2]) + l[0]*l[2]/(36*l[1]) + l[1]*np.tan(g[1])/6 - l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - l[1]*l[2]/(18*l[0]),
           (-l[0]**2*(l[1]**2 + l[2]**2) + l[1]*l[2]*(3*l[0]*l[1]*np.tan(g[1]) + 3*l[0]*l[2]*np.tan(g[0]) + l[1]*l[2] - l[1]*l[2]/np.cos(g[1])**2 - l[1]*l[2]/np.cos(g[0])**2))/(36*l[0]*l[1]*l[2]),
           -l[0]*l[1]/(18*l[2]) - l[0]*l[2]/(18*l[1]) + l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + l[1]*l[2]/(36*l[0])],
          [l[0]*l[1]/(18*l[2]) + l[0]*l[2]/(18*l[1]) - l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - l[1]*l[2]/(9*l[0]),
           (2*l[0]**2*(l[1]**2 + l[2]**2) + 3*l[0]*l[1]*l[2]*(l[1]*np.tan(g[1]) + l[2]*np.tan(g[0])) + 2*l[1]**2*l[2]**2*(np.tan(g[0])**2 + np.tan(g[1])**2 + 1))/(18*l[0]*l[1]*l[2]),
           l[0]*l[1]/(18*l[2]) - l[0]*l[2]/(9*l[1]) + l[1]*np.tan(g[1])/12 + l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + l[1]*l[2]/(18*l[0]),
           l[0]*l[1]/(36*l[2]) - l[0]*l[2]/(18*l[1]) - l[2]*np.tan(g[0])/6 - l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - l[1]*l[2]/(18*l[0]),
           -l[0]*l[1]/(18*l[2]) + l[0]*l[2]/(36*l[1]) - l[1]*np.tan(g[1])/6 - l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - l[1]*l[2]/(18*l[0]),
           -l[0]*l[1]/(9*l[2]) + l[0]*l[2]/(18*l[1]) + l[2]*np.tan(g[0])/12 + l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + l[1]*l[2]/(18*l[0]),
           (-2*l[0]**2*(l[1]**2 + l[2]**2) + l[1]**2*l[2]**2*(np.tan(g[0])**2 + np.tan(g[1])**2 + 1))/(36*l[0]*l[1]*l[2]),
           (-l[0]**2*(l[1]**2 + l[2]**2) + l[1]*l[2]*(-3*l[0]*l[1]*np.tan(g[1]) - 3*l[0]*l[2]*np.tan(g[0]) + l[1]*l[2] - l[1]*l[2]/np.cos(g[1])**2 - l[1]*l[2]/np.cos(g[0])**2))/(36*l[0]*l[1]*l[2])],
          [l[0]*l[1]/(36*l[2]) - l[0]*l[2]/(18*l[1]) + l[2]*np.tan(g[0])/6 - l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - l[1]*l[2]/(18*l[0]),
           l[0]*l[1]/(18*l[2]) - l[0]*l[2]/(9*l[1]) + l[1]*np.tan(g[1])/12 + l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + l[1]*l[2]/(18*l[0]),
           (2*l[0]**2*(l[1]**2 + l[2]**2) + 3*l[0]*l[1]*l[2]*(l[1]*np.tan(g[1]) - l[2]*np.tan(g[0])) + 2*l[1]**2*l[2]**2*(np.tan(g[0])**2 + np.tan(g[1])**2 + 1))/(18*l[0]*l[1]*l[2]),
           l[0]*l[1]/(18*l[2]) + l[0]*l[2]/(18*l[1]) - l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - l[1]*l[2]/(9*l[0]),
           (-l[0]**2*(l[1]**2 + l[2]**2) + l[1]*l[2]*(-3*l[0]*l[1]*np.tan(g[1]) + 3*l[0]*l[2]*np.tan(g[0]) + l[1]*l[2] - l[1]*l[2]/np.cos(g[1])**2 - l[1]*l[2]/np.cos(g[0])**2))/(36*l[0]*l[1]*l[2]),
           (-2*l[0]**2*(l[1]**2 + l[2]**2) + l[1]**2*l[2]**2*(np.tan(g[0])**2 + np.tan(g[1])**2 + 1))/(36*l[0]*l[1]*l[2]),
           -l[0]*l[1]/(9*l[2]) + l[0]*l[2]/(18*l[1]) - l[2]*np.tan(g[0])/12 + l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + l[1]*l[2]/(18*l[0]),
           -l[0]*l[1]/(18*l[2]) + l[0]*l[2]/(36*l[1]) - l[1]*np.tan(g[1])/6 - l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - l[1]*l[2]/(18*l[0])],
          [l[0]*l[1]/(18*l[2]) - l[0]*l[2]/(9*l[1]) - l[1]*np.tan(g[1])/12 + l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + l[1]*l[2]/(18*l[0]),
           l[0]*l[1]/(36*l[2]) - l[0]*l[2]/(18*l[1]) - l[2]*np.tan(g[0])/6 - l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - l[1]*l[2]/(18*l[0]),
           l[0]*l[1]/(18*l[2]) + l[0]*l[2]/(18*l[1]) - l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - l[1]*l[2]/(9*l[0]),
           l[0]*l[1]/(9*l[2]) + l[0]*l[2]/(9*l[1]) - l[1]*np.tan(g[1])/6 + l[2]*np.tan(g[0])/6 + l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + l[1]*l[2]/(9*l[0]),
           -l[0]*l[1]/(18*l[2]) - l[0]*l[2]/(18*l[1]) + l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + l[1]*l[2]/(36*l[0]),
           (-l[0]**2*(l[1]**2 + l[2]**2) + l[1]*l[2]*(3*l[0]*l[1]*np.tan(g[1]) - 3*l[0]*l[2]*np.tan(g[0]) + l[1]*l[2] - l[1]*l[2]/np.cos(g[1])**2 - l[1]*l[2]/np.cos(g[0])**2))/(36*l[0]*l[1]*l[2]),
           -l[0]*l[1]/(18*l[2]) + l[0]*l[2]/(36*l[1]) + l[1]*np.tan(g[1])/6 - l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - l[1]*l[2]/(18*l[0]),
           -l[0]*l[1]/(9*l[2]) + l[0]*l[2]/(18*l[1]) + l[2]*np.tan(g[0])/12 + l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + l[1]*l[2]/(18*l[0])],
          [-l[0]*l[1]/(9*l[2]) + l[0]*l[2]/(18*l[1]) - l[2]*np.tan(g[0])/12 + l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + l[1]*l[2]/(18*l[0]),
           -l[0]*l[1]/(18*l[2]) + l[0]*l[2]/(36*l[1]) - l[1]*np.tan(g[1])/6 - l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - l[1]*l[2]/(18*l[0]),
           (-l[0]**2*(l[1]**2 + l[2]**2) + l[1]*l[2]*(-3*l[0]*l[1]*np.tan(g[1]) + 3*l[0]*l[2]*np.tan(g[0]) + l[1]*l[2] - l[1]*l[2]/np.cos(g[1])**2 - l[1]*l[2]/np.cos(g[0])**2))/(36*l[0]*l[1]*l[2]),
           -l[0]*l[1]/(18*l[2]) - l[0]*l[2]/(18*l[1]) + l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + l[1]*l[2]/(36*l[0]),
           l[0]*l[1]/(9*l[2]) + l[0]*l[2]/(9*l[1]) + l[1]*np.tan(g[1])/6 - l[2]*np.tan(g[0])/6 + l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + l[1]*l[2]/(9*l[0]),
           l[0]*l[1]/(18*l[2]) + l[0]*l[2]/(18*l[1]) - l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - l[1]*l[2]/(9*l[0]),
           l[0]*l[1]/(36*l[2]) - l[0]*l[2]/(18*l[1]) + l[2]*np.tan(g[0])/6 - l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - l[1]*l[2]/(18*l[0]),
           l[0]*l[1]/(18*l[2]) - l[0]*l[2]/(9*l[1]) + l[1]*np.tan(g[1])/12 + l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + l[1]*l[2]/(18*l[0])],
          [-l[0]*l[1]/(18*l[2]) + l[0]*l[2]/(36*l[1]) + l[1]*np.tan(g[1])/6 - l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - l[1]*l[2]/(18*l[0]),
           -l[0]*l[1]/(9*l[2]) + l[0]*l[2]/(18*l[1]) + l[2]*np.tan(g[0])/12 + l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + l[1]*l[2]/(18*l[0]),
           (-2*l[0]**2*(l[1]**2 + l[2]**2) + l[1]**2*l[2]**2*(np.tan(g[0])**2 + np.tan(g[1])**2 + 1))/(36*l[0]*l[1]*l[2]),
           (-l[0]**2*(l[1]**2 + l[2]**2) + l[1]*l[2]*(3*l[0]*l[1]*np.tan(g[1]) - 3*l[0]*l[2]*np.tan(g[0]) + l[1]*l[2] - l[1]*l[2]/np.cos(g[1])**2 - l[1]*l[2]/np.cos(g[0])**2))/(36*l[0]*l[1]*l[2]),
           l[0]*l[1]/(18*l[2]) + l[0]*l[2]/(18*l[1]) - l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - l[1]*l[2]/(9*l[0]),
           (2*l[0]**2*(l[1]**2 + l[2]**2) + 3*l[0]*l[1]*l[2]*(-l[1]*np.tan(g[1]) + l[2]*np.tan(g[0])) + 2*l[1]**2*l[2]**2*(np.tan(g[0])**2 + np.tan(g[1])**2 + 1))/(18*l[0]*l[1]*l[2]),
           l[0]*l[1]/(18*l[2]) - l[0]*l[2]/(9*l[1]) - l[1]*np.tan(g[1])/12 + l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + l[1]*l[2]/(18*l[0]),
           l[0]*l[1]/(36*l[2]) - l[0]*l[2]/(18*l[1]) - l[2]*np.tan(g[0])/6 - l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - l[1]*l[2]/(18*l[0])],
          [(-l[0]**2*(l[1]**2 + l[2]**2) + l[1]*l[2]*(3*l[0]*l[1]*np.tan(g[1]) + 3*l[0]*l[2]*np.tan(g[0]) + l[1]*l[2] - l[1]*l[2]/np.cos(g[1])**2 - l[1]*l[2]/np.cos(g[0])**2))/(36*l[0]*l[1]*l[2]),
           (-2*l[0]**2*(l[1]**2 + l[2]**2) + l[1]**2*l[2]**2*(np.tan(g[0])**2 + np.tan(g[1])**2 + 1))/(36*l[0]*l[1]*l[2]),
           -l[0]*l[1]/(9*l[2]) + l[0]*l[2]/(18*l[1]) - l[2]*np.tan(g[0])/12 + l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + l[1]*l[2]/(18*l[0]),
           -l[0]*l[1]/(18*l[2]) + l[0]*l[2]/(36*l[1]) + l[1]*np.tan(g[1])/6 - l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - l[1]*l[2]/(18*l[0]),
           l[0]*l[1]/(36*l[2]) - l[0]*l[2]/(18*l[1]) + l[2]*np.tan(g[0])/6 - l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - l[1]*l[2]/(18*l[0]),
           l[0]*l[1]/(18*l[2]) - l[0]*l[2]/(9*l[1]) - l[1]*np.tan(g[1])/12 + l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + l[1]*l[2]/(18*l[0]),
           (2*l[0]**2*(l[1]**2 + l[2]**2) - 3*l[0]*l[1]*l[2]*(l[1]*np.tan(g[1]) + l[2]*np.tan(g[0])) + 2*l[1]**2*l[2]**2*(np.tan(g[0])**2 + np.tan(g[1])**2 + 1))/(18*l[0]*l[1]*l[2]),
           l[0]*l[1]/(18*l[2]) + l[0]*l[2]/(18*l[1]) - l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - l[1]*l[2]/(9*l[0])],
          [-l[0]*l[1]/(18*l[2]) - l[0]*l[2]/(18*l[1]) + l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + l[1]*l[2]*np.tan(g[1])**2/(36*l[0]) + l[1]*l[2]/(36*l[0]),
           (-l[0]**2*(l[1]**2 + l[2]**2) + l[1]*l[2]*(-3*l[0]*l[1]*np.tan(g[1]) - 3*l[0]*l[2]*np.tan(g[0]) + l[1]*l[2] - l[1]*l[2]/np.cos(g[1])**2 - l[1]*l[2]/np.cos(g[0])**2))/(36*l[0]*l[1]*l[2]),
           -l[0]*l[1]/(18*l[2]) + l[0]*l[2]/(36*l[1]) - l[1]*np.tan(g[1])/6 - l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - l[1]*l[2]/(18*l[0]),
           -l[0]*l[1]/(9*l[2]) + l[0]*l[2]/(18*l[1]) + l[2]*np.tan(g[0])/12 + l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + l[1]*l[2]/(18*l[0]),
           l[0]*l[1]/(18*l[2]) - l[0]*l[2]/(9*l[1]) + l[1]*np.tan(g[1])/12 + l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) + l[1]*l[2]/(18*l[0]),
           l[0]*l[1]/(36*l[2]) - l[0]*l[2]/(18*l[1]) - l[2]*np.tan(g[0])/6 - l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - l[1]*l[2]*np.tan(g[1])**2/(18*l[0]) - l[1]*l[2]/(18*l[0]),
           l[0]*l[1]/(18*l[2]) + l[0]*l[2]/(18*l[1]) - l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) - l[1]*l[2]/(9*l[0]),
           l[0]*l[1]/(9*l[2]) + l[0]*l[2]/(9*l[1]) + l[1]*np.tan(g[1])/6 + l[2]*np.tan(g[0])/6 + l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + l[1]*l[2]*np.tan(g[1])**2/(9*l[0]) + l[1]*l[2]/(9*l[0])]])

def lk_poisson_aniso_3d(k: np.ndarray,
                        l: np.ndarray = np.array([1.,1.,1.]), 
                        g: np.ndarray = np.array([0.,0.])) -> np.ndarray:
    """
    Create element stiffness matrix for anisotropic 3D Poisson with trilinear
    hexahedral elements. 
    
    Parameters
    ----------
    k : np.ndarray, shape (3,3)
        anisotropic heat conductivity. If isotropic k would be 
        [[k,0,0],[0,k,0],[0,0,k]].
    l : np.ndarray (3)
        side length of element.
    g : np.ndarray (2)
        angles of parallelogram.
        
    Returns
    -------
    Ke : np.ndarray, shape (8,8)
        element stiffness matrix.
        
    """
    return np.array([[k[0,0]*l[1]*l[2]/(9*l[0]) + k[0,1]*l[2]/12 - k[0,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + k[0,2]*l[1]/12 - k[0,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + k[1,0]*l[2]/12 - k[1,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + k[1,1]*l[0]*l[2]/(9*l[1]) - k[1,1]*l[2]*np.tan(g[0])/6 + k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + k[1,2]*l[0]/12 - k[1,2]*l[1]*np.tan(g[0])/12 - k[1,2]*l[2]*np.tan(g[1])/12 + k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + k[2,0]*l[1]/12 - k[2,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + k[2,1]*l[0]/12 - k[2,1]*l[1]*np.tan(g[0])/12 - k[2,1]*l[2]*np.tan(g[1])/12 + k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + k[2,2]*l[0]*l[1]/(9*l[2]) - k[2,2]*l[1]*np.tan(g[1])/6 + k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]),
           -k[0,0]*l[1]*l[2]/(9*l[0]) + k[0,1]*l[2]/12 + k[0,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + k[0,2]*l[1]/12 + k[0,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - k[1,0]*l[2]/12 + k[1,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + k[1,1]*l[0]*l[2]/(18*l[1]) - k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + k[1,2]*l[0]/24 - k[1,2]*l[1]*np.tan(g[0])/12 + k[1,2]*l[2]*np.tan(g[1])/12 - k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - k[2,0]*l[1]/12 + k[2,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + k[2,1]*l[0]/24 + k[2,1]*l[1]*np.tan(g[0])/12 - k[2,1]*l[2]*np.tan(g[1])/12 - k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + k[2,2]*l[0]*l[1]/(18*l[2]) - k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]),
           -k[0,0]*l[1]*l[2]/(18*l[0]) - k[0,1]*l[2]/12 + k[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + k[0,2]*l[1]/24 + k[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - k[1,0]*l[2]/12 + k[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - k[1,1]*l[0]*l[2]/(18*l[1]) + k[1,1]*l[2]*np.tan(g[0])/6 - k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + k[1,2]*l[0]/24 - k[1,2]*l[1]*np.tan(g[0])/24 + k[1,2]*l[2]*np.tan(g[1])/12 - k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - k[2,0]*l[1]/24 + k[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - k[2,1]*l[0]/24 + k[2,1]*l[1]*np.tan(g[0])/24 + k[2,1]*l[2]*np.tan(g[1])/12 - k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + k[2,2]*l[0]*l[1]/(36*l[2]) - k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]),
           k[0,0]*l[1]*l[2]/(18*l[0]) - k[0,1]*l[2]/12 - k[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + k[0,2]*l[1]/24 - k[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + k[1,0]*l[2]/12 - k[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - k[1,1]*l[0]*l[2]/(9*l[1]) + k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + k[1,2]*l[0]/12 - k[1,2]*l[1]*np.tan(g[0])/24 - k[1,2]*l[2]*np.tan(g[1])/12 + k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + k[2,0]*l[1]/24 - k[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - k[2,1]*l[0]/12 - k[2,1]*l[1]*np.tan(g[0])/24 + k[2,1]*l[2]*np.tan(g[1])/12 + k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + k[2,2]*l[0]*l[1]/(18*l[2]) - k[2,2]*l[1]*np.tan(g[1])/12 + k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]),
           k[0,0]*l[1]*l[2]/(18*l[0]) + k[0,1]*l[2]/24 - k[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - k[0,2]*l[1]/12 - k[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + k[1,0]*l[2]/24 - k[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + k[1,1]*l[0]*l[2]/(18*l[1]) - k[1,1]*l[2]*np.tan(g[0])/12 + k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - k[1,2]*l[0]/12 + k[1,2]*l[1]*np.tan(g[0])/12 - k[1,2]*l[2]*np.tan(g[1])/24 + k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + k[2,0]*l[1]/12 - k[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + k[2,1]*l[0]/12 - k[2,1]*l[1]*np.tan(g[0])/12 - k[2,1]*l[2]*np.tan(g[1])/24 + k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - k[2,2]*l[0]*l[1]/(9*l[2]) + k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]),
           -k[0,0]*l[1]*l[2]/(18*l[0]) + k[0,1]*l[2]/24 + k[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - k[0,2]*l[1]/12 + k[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - k[1,0]*l[2]/24 + k[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + k[1,1]*l[0]*l[2]/(36*l[1]) - k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - k[1,2]*l[0]/24 + k[1,2]*l[1]*np.tan(g[0])/12 + k[1,2]*l[2]*np.tan(g[1])/24 - k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - k[2,0]*l[1]/12 + k[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + k[2,1]*l[0]/24 + k[2,1]*l[1]*np.tan(g[0])/12 - k[2,1]*l[2]*np.tan(g[1])/24 - k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - k[2,2]*l[0]*l[1]/(18*l[2]) + k[2,2]*l[1]*np.tan(g[1])/6 - k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]),
           -k[0,0]*l[1]*l[2]/(36*l[0]) - k[0,1]*l[2]/24 + k[0,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - k[0,2]*l[1]/24 + k[0,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - k[1,0]*l[2]/24 + k[1,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - k[1,1]*l[0]*l[2]/(36*l[1]) + k[1,1]*l[2]*np.tan(g[0])/12 - k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - k[1,2]*l[0]/24 + k[1,2]*l[1]*np.tan(g[0])/24 + k[1,2]*l[2]*np.tan(g[1])/24 - k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - k[2,0]*l[1]/24 + k[2,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - k[2,1]*l[0]/24 + k[2,1]*l[1]*np.tan(g[0])/24 + k[2,1]*l[2]*np.tan(g[1])/24 - k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - k[2,2]*l[0]*l[1]/(36*l[2]) + k[2,2]*l[1]*np.tan(g[1])/12 - k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]),
           k[0,0]*l[1]*l[2]/(36*l[0]) - k[0,1]*l[2]/24 - k[0,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - k[0,2]*l[1]/24 - k[0,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + k[1,0]*l[2]/24 - k[1,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - k[1,1]*l[0]*l[2]/(18*l[1]) + k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - k[1,2]*l[0]/12 + k[1,2]*l[1]*np.tan(g[0])/24 - k[1,2]*l[2]*np.tan(g[1])/24 + k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + k[2,0]*l[1]/24 - k[2,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - k[2,1]*l[0]/12 - k[2,1]*l[1]*np.tan(g[0])/24 + k[2,1]*l[2]*np.tan(g[1])/24 + k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - k[2,2]*l[0]*l[1]/(18*l[2]) + k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0])],
          [-k[0,0]*l[1]*l[2]/(9*l[0]) - k[0,1]*l[2]/12 + k[0,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - k[0,2]*l[1]/12 + k[0,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + k[1,0]*l[2]/12 + k[1,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + k[1,1]*l[0]*l[2]/(18*l[1]) - k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + k[1,2]*l[0]/24 + k[1,2]*l[1]*np.tan(g[0])/12 - k[1,2]*l[2]*np.tan(g[1])/12 - k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + k[2,0]*l[1]/12 + k[2,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + k[2,1]*l[0]/24 - k[2,1]*l[1]*np.tan(g[0])/12 + k[2,1]*l[2]*np.tan(g[1])/12 - k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + k[2,2]*l[0]*l[1]/(18*l[2]) - k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]),
           (l[0]**2*(4*k[1,1]*l[2]**2 + 3*k[1,2]*l[1]*l[2] + 3*k[2,1]*l[1]*l[2] + 4*k[2,2]*l[1]**2) + 3*l[0]*l[1]*l[2]*(-k[0,1]*l[2] - k[0,2]*l[1] - k[1,0]*l[2] + 2*k[1,1]*l[2]*np.tan(g[0]) + k[1,2]*l[1]*np.tan(g[0]) + k[1,2]*l[2]*np.tan(g[1]) - k[2,0]*l[1] + k[2,1]*l[1]*np.tan(g[0]) + k[2,1]*l[2]*np.tan(g[1]) + 2*k[2,2]*l[1]*np.tan(g[1])) + 4*l[1]**2*l[2]**2*(k[0,0] - k[0,1]*np.tan(g[0]) - k[0,2]*np.tan(g[1]) - k[1,0]*np.tan(g[0]) + k[1,1]*np.tan(g[0])**2 + k[1,2]*np.tan(g[0])*np.tan(g[1]) - k[2,0]*np.tan(g[1]) + k[2,1]*np.tan(g[0])*np.tan(g[1]) + k[2,2]*np.tan(g[1])**2))/(36*l[0]*l[1]*l[2]),
           (2*l[0]**2*(-4*k[1,1]*l[2]**2 + 3*k[1,2]*l[1]*l[2] - 3*k[2,1]*l[1]*l[2] + 2*k[2,2]*l[1]**2) + 3*l[0]*l[1]*l[2]*(2*k[0,1]*l[2] - k[0,2]*l[1] - 2*k[1,0]*l[2] + k[1,2]*l[1]*np.tan(g[0]) + 2*k[1,2]*l[2]*np.tan(g[1]) - k[2,0]*l[1] + k[2,1]*l[1]*np.tan(g[0]) - 2*k[2,1]*l[2]*np.tan(g[1]) + 2*k[2,2]*l[1]*np.tan(g[1])) + 4*l[1]**2*l[2]**2*(k[0,0] - k[0,1]*np.tan(g[0]) - k[0,2]*np.tan(g[1]) - k[1,0]*np.tan(g[0]) + k[1,1]*np.tan(g[0])**2 + k[1,2]*np.tan(g[0])*np.tan(g[1]) - k[2,0]*np.tan(g[1]) + k[2,1]*np.tan(g[0])*np.tan(g[1]) + k[2,2]*np.tan(g[1])**2))/(72*l[0]*l[1]*l[2]),
           -k[0,0]*l[1]*l[2]/(18*l[0]) + k[0,1]*l[2]/12 + k[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - k[0,2]*l[1]/24 + k[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + k[1,0]*l[2]/12 + k[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - k[1,1]*l[0]*l[2]/(18*l[1]) - k[1,1]*l[2]*np.tan(g[0])/6 - k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + k[1,2]*l[0]/24 + k[1,2]*l[1]*np.tan(g[0])/24 - k[1,2]*l[2]*np.tan(g[1])/12 - k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + k[2,0]*l[1]/24 + k[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - k[2,1]*l[0]/24 - k[2,1]*l[1]*np.tan(g[0])/24 - k[2,1]*l[2]*np.tan(g[1])/12 - k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + k[2,2]*l[0]*l[1]/(36*l[2]) - k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]),
           -k[0,0]*l[1]*l[2]/(18*l[0]) - k[0,1]*l[2]/24 + k[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + k[0,2]*l[1]/12 + k[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + k[1,0]*l[2]/24 + k[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + k[1,1]*l[0]*l[2]/(36*l[1]) - k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - k[1,2]*l[0]/24 - k[1,2]*l[1]*np.tan(g[0])/12 - k[1,2]*l[2]*np.tan(g[1])/24 - k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + k[2,0]*l[1]/12 + k[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + k[2,1]*l[0]/24 - k[2,1]*l[1]*np.tan(g[0])/12 + k[2,1]*l[2]*np.tan(g[1])/24 - k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - k[2,2]*l[0]*l[1]/(18*l[2]) - k[2,2]*l[1]*np.tan(g[1])/6 - k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]),
           (2*l[0]**2*(2*k[1,1]*l[2]**2 - 3*k[1,2]*l[1]*l[2] + 3*k[2,1]*l[1]*l[2] - 4*k[2,2]*l[1]**2) + 3*l[0]*l[1]*l[2]*(-k[0,1]*l[2] + 2*k[0,2]*l[1] - k[1,0]*l[2] + 2*k[1,1]*l[2]*np.tan(g[0]) - 2*k[1,2]*l[1]*np.tan(g[0]) + k[1,2]*l[2]*np.tan(g[1]) - 2*k[2,0]*l[1] + 2*k[2,1]*l[1]*np.tan(g[0]) + k[2,1]*l[2]*np.tan(g[1])) + 4*l[1]**2*l[2]**2*(k[0,0] - k[0,1]*np.tan(g[0]) - k[0,2]*np.tan(g[1]) - k[1,0]*np.tan(g[0]) + k[1,1]*np.tan(g[0])**2 + k[1,2]*np.tan(g[0])*np.tan(g[1]) - k[2,0]*np.tan(g[1]) + k[2,1]*np.tan(g[0])*np.tan(g[1]) + k[2,2]*np.tan(g[1])**2))/(72*l[0]*l[1]*l[2]),
           (-2*l[0]**2*(2*k[1,1]*l[2]**2 + 3*k[1,2]*l[1]*l[2] + 3*k[2,1]*l[1]*l[2] + 2*k[2,2]*l[1]**2) + 3*l[0]*l[1]*l[2]*(k[0,1]*l[2] + k[0,2]*l[1] - k[1,0]*l[2] - k[1,2]*l[1]*np.tan(g[0]) + k[1,2]*l[2]*np.tan(g[1]) - k[2,0]*l[1] + k[2,1]*l[1]*np.tan(g[0]) - k[2,1]*l[2]*np.tan(g[1])) + 2*l[1]**2*l[2]**2*(k[0,0] - k[0,1]*np.tan(g[0]) - k[0,2]*np.tan(g[1]) - k[1,0]*np.tan(g[0]) + k[1,1]*np.tan(g[0])**2 + k[1,2]*np.tan(g[0])*np.tan(g[1]) - k[2,0]*np.tan(g[1]) + k[2,1]*np.tan(g[0])*np.tan(g[1]) + k[2,2]*np.tan(g[1])**2))/(72*l[0]*l[1]*l[2]),
           -k[0,0]*l[1]*l[2]/(36*l[0]) + k[0,1]*l[2]/24 + k[0,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + k[0,2]*l[1]/24 + k[0,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + k[1,0]*l[2]/24 + k[1,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - k[1,1]*l[0]*l[2]/(36*l[1]) - k[1,1]*l[2]*np.tan(g[0])/12 - k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - k[1,2]*l[0]/24 - k[1,2]*l[1]*np.tan(g[0])/24 - k[1,2]*l[2]*np.tan(g[1])/24 - k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + k[2,0]*l[1]/24 + k[2,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - k[2,1]*l[0]/24 - k[2,1]*l[1]*np.tan(g[0])/24 - k[2,1]*l[2]*np.tan(g[1])/24 - k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - k[2,2]*l[0]*l[1]/(36*l[2]) - k[2,2]*l[1]*np.tan(g[1])/12 - k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0])],
          [-k[0,0]*l[1]*l[2]/(18*l[0]) - k[0,1]*l[2]/12 + k[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - k[0,2]*l[1]/24 + k[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - k[1,0]*l[2]/12 + k[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - k[1,1]*l[0]*l[2]/(18*l[1]) + k[1,1]*l[2]*np.tan(g[0])/6 - k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - k[1,2]*l[0]/24 + k[1,2]*l[1]*np.tan(g[0])/24 + k[1,2]*l[2]*np.tan(g[1])/12 - k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + k[2,0]*l[1]/24 + k[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + k[2,1]*l[0]/24 - k[2,1]*l[1]*np.tan(g[0])/24 + k[2,1]*l[2]*np.tan(g[1])/12 - k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + k[2,2]*l[0]*l[1]/(36*l[2]) - k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]),
           (2*l[0]**2*(-4*k[1,1]*l[2]**2 - 3*k[1,2]*l[1]*l[2] + 3*k[2,1]*l[1]*l[2] + 2*k[2,2]*l[1]**2) + 3*l[0]*l[1]*l[2]*(-2*k[0,1]*l[2] - k[0,2]*l[1] + 2*k[1,0]*l[2] + k[1,2]*l[1]*np.tan(g[0]) - 2*k[1,2]*l[2]*np.tan(g[1]) - k[2,0]*l[1] + k[2,1]*l[1]*np.tan(g[0]) + 2*k[2,1]*l[2]*np.tan(g[1]) + 2*k[2,2]*l[1]*np.tan(g[1])) + 4*l[1]**2*l[2]**2*(k[0,0] - k[0,1]*np.tan(g[0]) - k[0,2]*np.tan(g[1]) - k[1,0]*np.tan(g[0]) + k[1,1]*np.tan(g[0])**2 + k[1,2]*np.tan(g[0])*np.tan(g[1]) - k[2,0]*np.tan(g[1]) + k[2,1]*np.tan(g[0])*np.tan(g[1]) + k[2,2]*np.tan(g[1])**2))/(72*l[0]*l[1]*l[2]),
           (l[0]**2*(4*k[1,1]*l[2]**2 - 3*k[1,2]*l[1]*l[2] - 3*k[2,1]*l[1]*l[2] + 4*k[2,2]*l[1]**2) + 3*l[0]*l[1]*l[2]*(k[0,1]*l[2] - k[0,2]*l[1] + k[1,0]*l[2] - 2*k[1,1]*l[2]*np.tan(g[0]) + k[1,2]*l[1]*np.tan(g[0]) - k[1,2]*l[2]*np.tan(g[1]) - k[2,0]*l[1] + k[2,1]*l[1]*np.tan(g[0]) - k[2,1]*l[2]*np.tan(g[1]) + 2*k[2,2]*l[1]*np.tan(g[1])) + 4*l[1]**2*l[2]**2*(k[0,0] - k[0,1]*np.tan(g[0]) - k[0,2]*np.tan(g[1]) - k[1,0]*np.tan(g[0]) + k[1,1]*np.tan(g[0])**2 + k[1,2]*np.tan(g[0])*np.tan(g[1]) - k[2,0]*np.tan(g[1]) + k[2,1]*np.tan(g[0])*np.tan(g[1]) + k[2,2]*np.tan(g[1])**2))/(36*l[0]*l[1]*l[2]),
           -k[0,0]*l[1]*l[2]/(9*l[0]) + k[0,1]*l[2]/12 + k[0,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - k[0,2]*l[1]/12 + k[0,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - k[1,0]*l[2]/12 + k[1,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + k[1,1]*l[0]*l[2]/(18*l[1]) - k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - k[1,2]*l[0]/24 + k[1,2]*l[1]*np.tan(g[0])/12 + k[1,2]*l[2]*np.tan(g[1])/12 - k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + k[2,0]*l[1]/12 + k[2,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - k[2,1]*l[0]/24 - k[2,1]*l[1]*np.tan(g[0])/12 - k[2,1]*l[2]*np.tan(g[1])/12 - k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + k[2,2]*l[0]*l[1]/(18*l[2]) - k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]),
           -k[0,0]*l[1]*l[2]/(36*l[0]) - k[0,1]*l[2]/24 + k[0,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + k[0,2]*l[1]/24 + k[0,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - k[1,0]*l[2]/24 + k[1,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - k[1,1]*l[0]*l[2]/(36*l[1]) + k[1,1]*l[2]*np.tan(g[0])/12 - k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + k[1,2]*l[0]/24 - k[1,2]*l[1]*np.tan(g[0])/24 + k[1,2]*l[2]*np.tan(g[1])/24 - k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + k[2,0]*l[1]/24 + k[2,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + k[2,1]*l[0]/24 - k[2,1]*l[1]*np.tan(g[0])/24 + k[2,1]*l[2]*np.tan(g[1])/24 - k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - k[2,2]*l[0]*l[1]/(36*l[2]) - k[2,2]*l[1]*np.tan(g[1])/12 - k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]),
           (2*l[0]**2*(-2*k[1,1]*l[2]**2 + 3*k[1,2]*l[1]*l[2] + 3*k[2,1]*l[1]*l[2] - 2*k[2,2]*l[1]**2) + 3*l[0]*l[1]*l[2]*(-k[0,1]*l[2] + k[0,2]*l[1] + k[1,0]*l[2] - k[1,2]*l[1]*np.tan(g[0]) - k[1,2]*l[2]*np.tan(g[1]) - k[2,0]*l[1] + k[2,1]*l[1]*np.tan(g[0]) + k[2,1]*l[2]*np.tan(g[1])) + 2*l[1]**2*l[2]**2*(k[0,0] - k[0,1]*np.tan(g[0]) - k[0,2]*np.tan(g[1]) - k[1,0]*np.tan(g[0]) + k[1,1]*np.tan(g[0])**2 + k[1,2]*np.tan(g[0])*np.tan(g[1]) - k[2,0]*np.tan(g[1]) + k[2,1]*np.tan(g[0])*np.tan(g[1]) + k[2,2]*np.tan(g[1])**2))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(2*k[1,1]*l[2]**2 + 3*k[1,2]*l[1]*l[2] - 3*k[2,1]*l[1]*l[2] - 4*k[2,2]*l[1]**2) + 3*l[0]*l[1]*l[2]*(k[0,1]*l[2] + 2*k[0,2]*l[1] + k[1,0]*l[2] - 2*k[1,1]*l[2]*np.tan(g[0]) - 2*k[1,2]*l[1]*np.tan(g[0]) - k[1,2]*l[2]*np.tan(g[1]) - 2*k[2,0]*l[1] + 2*k[2,1]*l[1]*np.tan(g[0]) - k[2,1]*l[2]*np.tan(g[1])) + 4*l[1]**2*l[2]**2*(k[0,0] - k[0,1]*np.tan(g[0]) - k[0,2]*np.tan(g[1]) - k[1,0]*np.tan(g[0]) + k[1,1]*np.tan(g[0])**2 + k[1,2]*np.tan(g[0])*np.tan(g[1]) - k[2,0]*np.tan(g[1]) + k[2,1]*np.tan(g[0])*np.tan(g[1]) + k[2,2]*np.tan(g[1])**2))/(72*l[0]*l[1]*l[2]),
           -k[0,0]*l[1]*l[2]/(18*l[0]) + k[0,1]*l[2]/24 + k[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + k[0,2]*l[1]/12 + k[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - k[1,0]*l[2]/24 + k[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + k[1,1]*l[0]*l[2]/(36*l[1]) - k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + k[1,2]*l[0]/24 - k[1,2]*l[1]*np.tan(g[0])/12 + k[1,2]*l[2]*np.tan(g[1])/24 - k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + k[2,0]*l[1]/12 + k[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - k[2,1]*l[0]/24 - k[2,1]*l[1]*np.tan(g[0])/12 - k[2,1]*l[2]*np.tan(g[1])/24 - k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - k[2,2]*l[0]*l[1]/(18*l[2]) - k[2,2]*l[1]*np.tan(g[1])/6 - k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0])],
          [k[0,0]*l[1]*l[2]/(18*l[0]) + k[0,1]*l[2]/12 - k[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + k[0,2]*l[1]/24 - k[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - k[1,0]*l[2]/12 - k[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - k[1,1]*l[0]*l[2]/(9*l[1]) + k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - k[1,2]*l[0]/12 - k[1,2]*l[1]*np.tan(g[0])/24 + k[1,2]*l[2]*np.tan(g[1])/12 + k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + k[2,0]*l[1]/24 - k[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + k[2,1]*l[0]/12 - k[2,1]*l[1]*np.tan(g[0])/24 - k[2,1]*l[2]*np.tan(g[1])/12 + k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + k[2,2]*l[0]*l[1]/(18*l[2]) - k[2,2]*l[1]*np.tan(g[1])/12 + k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]),
           -k[0,0]*l[1]*l[2]/(18*l[0]) + k[0,1]*l[2]/12 + k[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + k[0,2]*l[1]/24 + k[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + k[1,0]*l[2]/12 + k[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - k[1,1]*l[0]*l[2]/(18*l[1]) - k[1,1]*l[2]*np.tan(g[0])/6 - k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - k[1,2]*l[0]/24 - k[1,2]*l[1]*np.tan(g[0])/24 - k[1,2]*l[2]*np.tan(g[1])/12 - k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - k[2,0]*l[1]/24 + k[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + k[2,1]*l[0]/24 + k[2,1]*l[1]*np.tan(g[0])/24 - k[2,1]*l[2]*np.tan(g[1])/12 - k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + k[2,2]*l[0]*l[1]/(36*l[2]) - k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]),
           -k[0,0]*l[1]*l[2]/(9*l[0]) - k[0,1]*l[2]/12 + k[0,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + k[0,2]*l[1]/12 + k[0,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + k[1,0]*l[2]/12 + k[1,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + k[1,1]*l[0]*l[2]/(18*l[1]) - k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - k[1,2]*l[0]/24 - k[1,2]*l[1]*np.tan(g[0])/12 - k[1,2]*l[2]*np.tan(g[1])/12 - k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - k[2,0]*l[1]/12 + k[2,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - k[2,1]*l[0]/24 + k[2,1]*l[1]*np.tan(g[0])/12 + k[2,1]*l[2]*np.tan(g[1])/12 - k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + k[2,2]*l[0]*l[1]/(18*l[2]) - k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]),
           k[0,0]*l[1]*l[2]/(9*l[0]) - k[0,1]*l[2]/12 - k[0,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + k[0,2]*l[1]/12 - k[0,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - k[1,0]*l[2]/12 - k[1,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + k[1,1]*l[0]*l[2]/(9*l[1]) + k[1,1]*l[2]*np.tan(g[0])/6 + k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - k[1,2]*l[0]/12 - k[1,2]*l[1]*np.tan(g[0])/12 + k[1,2]*l[2]*np.tan(g[1])/12 + k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + k[2,0]*l[1]/12 - k[2,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - k[2,1]*l[0]/12 - k[2,1]*l[1]*np.tan(g[0])/12 + k[2,1]*l[2]*np.tan(g[1])/12 + k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + k[2,2]*l[0]*l[1]/(9*l[2]) - k[2,2]*l[1]*np.tan(g[1])/6 + k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]),
           k[0,0]*l[1]*l[2]/(36*l[0]) + k[0,1]*l[2]/24 - k[0,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - k[0,2]*l[1]/24 - k[0,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - k[1,0]*l[2]/24 - k[1,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - k[1,1]*l[0]*l[2]/(18*l[1]) + k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + k[1,2]*l[0]/12 + k[1,2]*l[1]*np.tan(g[0])/24 + k[1,2]*l[2]*np.tan(g[1])/24 + k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + k[2,0]*l[1]/24 - k[2,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + k[2,1]*l[0]/12 - k[2,1]*l[1]*np.tan(g[0])/24 - k[2,1]*l[2]*np.tan(g[1])/24 + k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - k[2,2]*l[0]*l[1]/(18*l[2]) + k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]),
           -k[0,0]*l[1]*l[2]/(36*l[0]) + k[0,1]*l[2]/24 + k[0,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - k[0,2]*l[1]/24 + k[0,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + k[1,0]*l[2]/24 + k[1,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - k[1,1]*l[0]*l[2]/(36*l[1]) - k[1,1]*l[2]*np.tan(g[0])/12 - k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + k[1,2]*l[0]/24 + k[1,2]*l[1]*np.tan(g[0])/24 - k[1,2]*l[2]*np.tan(g[1])/24 - k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - k[2,0]*l[1]/24 + k[2,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + k[2,1]*l[0]/24 + k[2,1]*l[1]*np.tan(g[0])/24 - k[2,1]*l[2]*np.tan(g[1])/24 - k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - k[2,2]*l[0]*l[1]/(36*l[2]) + k[2,2]*l[1]*np.tan(g[1])/12 - k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]),
           -k[0,0]*l[1]*l[2]/(18*l[0]) - k[0,1]*l[2]/24 + k[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - k[0,2]*l[1]/12 + k[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + k[1,0]*l[2]/24 + k[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + k[1,1]*l[0]*l[2]/(36*l[1]) - k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + k[1,2]*l[0]/24 + k[1,2]*l[1]*np.tan(g[0])/12 - k[1,2]*l[2]*np.tan(g[1])/24 - k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - k[2,0]*l[1]/12 + k[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - k[2,1]*l[0]/24 + k[2,1]*l[1]*np.tan(g[0])/12 + k[2,1]*l[2]*np.tan(g[1])/24 - k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - k[2,2]*l[0]*l[1]/(18*l[2]) + k[2,2]*l[1]*np.tan(g[1])/6 - k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]),
           k[0,0]*l[1]*l[2]/(18*l[0]) - k[0,1]*l[2]/24 - k[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - k[0,2]*l[1]/12 - k[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - k[1,0]*l[2]/24 - k[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + k[1,1]*l[0]*l[2]/(18*l[1]) + k[1,1]*l[2]*np.tan(g[0])/12 + k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + k[1,2]*l[0]/12 + k[1,2]*l[1]*np.tan(g[0])/12 + k[1,2]*l[2]*np.tan(g[1])/24 + k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + k[2,0]*l[1]/12 - k[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - k[2,1]*l[0]/12 - k[2,1]*l[1]*np.tan(g[0])/12 + k[2,1]*l[2]*np.tan(g[1])/24 + k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - k[2,2]*l[0]*l[1]/(9*l[2]) + k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0])],
          [k[0,0]*l[1]*l[2]/(18*l[0]) + k[0,1]*l[2]/24 - k[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + k[0,2]*l[1]/12 - k[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + k[1,0]*l[2]/24 - k[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + k[1,1]*l[0]*l[2]/(18*l[1]) - k[1,1]*l[2]*np.tan(g[0])/12 + k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + k[1,2]*l[0]/12 - k[1,2]*l[1]*np.tan(g[0])/12 - k[1,2]*l[2]*np.tan(g[1])/24 + k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - k[2,0]*l[1]/12 - k[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - k[2,1]*l[0]/12 + k[2,1]*l[1]*np.tan(g[0])/12 - k[2,1]*l[2]*np.tan(g[1])/24 + k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - k[2,2]*l[0]*l[1]/(9*l[2]) + k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]),
           -k[0,0]*l[1]*l[2]/(18*l[0]) + k[0,1]*l[2]/24 + k[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + k[0,2]*l[1]/12 + k[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - k[1,0]*l[2]/24 + k[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + k[1,1]*l[0]*l[2]/(36*l[1]) - k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + k[1,2]*l[0]/24 - k[1,2]*l[1]*np.tan(g[0])/12 + k[1,2]*l[2]*np.tan(g[1])/24 - k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + k[2,0]*l[1]/12 + k[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - k[2,1]*l[0]/24 - k[2,1]*l[1]*np.tan(g[0])/12 - k[2,1]*l[2]*np.tan(g[1])/24 - k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - k[2,2]*l[0]*l[1]/(18*l[2]) - k[2,2]*l[1]*np.tan(g[1])/6 - k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]),
           -k[0,0]*l[1]*l[2]/(36*l[0]) - k[0,1]*l[2]/24 + k[0,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + k[0,2]*l[1]/24 + k[0,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - k[1,0]*l[2]/24 + k[1,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - k[1,1]*l[0]*l[2]/(36*l[1]) + k[1,1]*l[2]*np.tan(g[0])/12 - k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + k[1,2]*l[0]/24 - k[1,2]*l[1]*np.tan(g[0])/24 + k[1,2]*l[2]*np.tan(g[1])/24 - k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + k[2,0]*l[1]/24 + k[2,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + k[2,1]*l[0]/24 - k[2,1]*l[1]*np.tan(g[0])/24 + k[2,1]*l[2]*np.tan(g[1])/24 - k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - k[2,2]*l[0]*l[1]/(36*l[2]) - k[2,2]*l[1]*np.tan(g[1])/12 - k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]),
           k[0,0]*l[1]*l[2]/(36*l[0]) - k[0,1]*l[2]/24 - k[0,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + k[0,2]*l[1]/24 - k[0,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + k[1,0]*l[2]/24 - k[1,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - k[1,1]*l[0]*l[2]/(18*l[1]) + k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + k[1,2]*l[0]/12 - k[1,2]*l[1]*np.tan(g[0])/24 - k[1,2]*l[2]*np.tan(g[1])/24 + k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - k[2,0]*l[1]/24 - k[2,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + k[2,1]*l[0]/12 + k[2,1]*l[1]*np.tan(g[0])/24 + k[2,1]*l[2]*np.tan(g[1])/24 + k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - k[2,2]*l[0]*l[1]/(18*l[2]) + k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]),
           k[0,0]*l[1]*l[2]/(9*l[0]) + k[0,1]*l[2]/12 - k[0,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - k[0,2]*l[1]/12 - k[0,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + k[1,0]*l[2]/12 - k[1,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + k[1,1]*l[0]*l[2]/(9*l[1]) - k[1,1]*l[2]*np.tan(g[0])/6 + k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - k[1,2]*l[0]/12 + k[1,2]*l[1]*np.tan(g[0])/12 - k[1,2]*l[2]*np.tan(g[1])/12 + k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - k[2,0]*l[1]/12 - k[2,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - k[2,1]*l[0]/12 + k[2,1]*l[1]*np.tan(g[0])/12 - k[2,1]*l[2]*np.tan(g[1])/12 + k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + k[2,2]*l[0]*l[1]/(9*l[2]) + k[2,2]*l[1]*np.tan(g[1])/6 + k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]),
           -k[0,0]*l[1]*l[2]/(9*l[0]) + k[0,1]*l[2]/12 + k[0,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - k[0,2]*l[1]/12 + k[0,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - k[1,0]*l[2]/12 + k[1,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + k[1,1]*l[0]*l[2]/(18*l[1]) - k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - k[1,2]*l[0]/24 + k[1,2]*l[1]*np.tan(g[0])/12 + k[1,2]*l[2]*np.tan(g[1])/12 - k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + k[2,0]*l[1]/12 + k[2,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - k[2,1]*l[0]/24 - k[2,1]*l[1]*np.tan(g[0])/12 - k[2,1]*l[2]*np.tan(g[1])/12 - k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + k[2,2]*l[0]*l[1]/(18*l[2]) - k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]),
           -k[0,0]*l[1]*l[2]/(18*l[0]) - k[0,1]*l[2]/12 + k[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - k[0,2]*l[1]/24 + k[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - k[1,0]*l[2]/12 + k[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - k[1,1]*l[0]*l[2]/(18*l[1]) + k[1,1]*l[2]*np.tan(g[0])/6 - k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - k[1,2]*l[0]/24 + k[1,2]*l[1]*np.tan(g[0])/24 + k[1,2]*l[2]*np.tan(g[1])/12 - k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + k[2,0]*l[1]/24 + k[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + k[2,1]*l[0]/24 - k[2,1]*l[1]*np.tan(g[0])/24 + k[2,1]*l[2]*np.tan(g[1])/12 - k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + k[2,2]*l[0]*l[1]/(36*l[2]) - k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]),
           k[0,0]*l[1]*l[2]/(18*l[0]) - k[0,1]*l[2]/12 - k[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - k[0,2]*l[1]/24 - k[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + k[1,0]*l[2]/12 - k[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - k[1,1]*l[0]*l[2]/(9*l[1]) + k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - k[1,2]*l[0]/12 + k[1,2]*l[1]*np.tan(g[0])/24 - k[1,2]*l[2]*np.tan(g[1])/12 + k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - k[2,0]*l[1]/24 - k[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + k[2,1]*l[0]/12 + k[2,1]*l[1]*np.tan(g[0])/24 + k[2,1]*l[2]*np.tan(g[1])/12 + k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + k[2,2]*l[0]*l[1]/(18*l[2]) + k[2,2]*l[1]*np.tan(g[1])/12 + k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0])],
          [-k[0,0]*l[1]*l[2]/(18*l[0]) - k[0,1]*l[2]/24 + k[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - k[0,2]*l[1]/12 + k[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + k[1,0]*l[2]/24 + k[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + k[1,1]*l[0]*l[2]/(36*l[1]) - k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + k[1,2]*l[0]/24 + k[1,2]*l[1]*np.tan(g[0])/12 - k[1,2]*l[2]*np.tan(g[1])/24 - k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - k[2,0]*l[1]/12 + k[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - k[2,1]*l[0]/24 + k[2,1]*l[1]*np.tan(g[0])/12 + k[2,1]*l[2]*np.tan(g[1])/24 - k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - k[2,2]*l[0]*l[1]/(18*l[2]) + k[2,2]*l[1]*np.tan(g[1])/6 - k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]),
           (2*l[0]**2*(2*k[1,1]*l[2]**2 + 3*k[1,2]*l[1]*l[2] - 3*k[2,1]*l[1]*l[2] - 4*k[2,2]*l[1]**2) + 3*l[0]*l[1]*l[2]*(-k[0,1]*l[2] - 2*k[0,2]*l[1] - k[1,0]*l[2] + 2*k[1,1]*l[2]*np.tan(g[0]) + 2*k[1,2]*l[1]*np.tan(g[0]) + k[1,2]*l[2]*np.tan(g[1]) + 2*k[2,0]*l[1] - 2*k[2,1]*l[1]*np.tan(g[0]) + k[2,1]*l[2]*np.tan(g[1])) + 4*l[1]**2*l[2]**2*(k[0,0] - k[0,1]*np.tan(g[0]) - k[0,2]*np.tan(g[1]) - k[1,0]*np.tan(g[0]) + k[1,1]*np.tan(g[0])**2 + k[1,2]*np.tan(g[0])*np.tan(g[1]) - k[2,0]*np.tan(g[1]) + k[2,1]*np.tan(g[0])*np.tan(g[1]) + k[2,2]*np.tan(g[1])**2))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(-2*k[1,1]*l[2]**2 + 3*k[1,2]*l[1]*l[2] + 3*k[2,1]*l[1]*l[2] - 2*k[2,2]*l[1]**2) + 3*l[0]*l[1]*l[2]*(k[0,1]*l[2] - k[0,2]*l[1] - k[1,0]*l[2] + k[1,2]*l[1]*np.tan(g[0]) + k[1,2]*l[2]*np.tan(g[1]) + k[2,0]*l[1] - k[2,1]*l[1]*np.tan(g[0]) - k[2,1]*l[2]*np.tan(g[1])) + 2*l[1]**2*l[2]**2*(k[0,0] - k[0,1]*np.tan(g[0]) - k[0,2]*np.tan(g[1]) - k[1,0]*np.tan(g[0]) + k[1,1]*np.tan(g[0])**2 + k[1,2]*np.tan(g[0])*np.tan(g[1]) - k[2,0]*np.tan(g[1]) + k[2,1]*np.tan(g[0])*np.tan(g[1]) + k[2,2]*np.tan(g[1])**2))/(72*l[0]*l[1]*l[2]),
           -k[0,0]*l[1]*l[2]/(36*l[0]) + k[0,1]*l[2]/24 + k[0,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - k[0,2]*l[1]/24 + k[0,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + k[1,0]*l[2]/24 + k[1,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - k[1,1]*l[0]*l[2]/(36*l[1]) - k[1,1]*l[2]*np.tan(g[0])/12 - k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) + k[1,2]*l[0]/24 + k[1,2]*l[1]*np.tan(g[0])/24 - k[1,2]*l[2]*np.tan(g[1])/24 - k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - k[2,0]*l[1]/24 + k[2,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + k[2,1]*l[0]/24 + k[2,1]*l[1]*np.tan(g[0])/24 - k[2,1]*l[2]*np.tan(g[1])/24 - k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - k[2,2]*l[0]*l[1]/(36*l[2]) + k[2,2]*l[1]*np.tan(g[1])/12 - k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]),
           -k[0,0]*l[1]*l[2]/(9*l[0]) - k[0,1]*l[2]/12 + k[0,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + k[0,2]*l[1]/12 + k[0,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + k[1,0]*l[2]/12 + k[1,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + k[1,1]*l[0]*l[2]/(18*l[1]) - k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) - k[1,2]*l[0]/24 - k[1,2]*l[1]*np.tan(g[0])/12 - k[1,2]*l[2]*np.tan(g[1])/12 - k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - k[2,0]*l[1]/12 + k[2,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - k[2,1]*l[0]/24 + k[2,1]*l[1]*np.tan(g[0])/12 + k[2,1]*l[2]*np.tan(g[1])/12 - k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + k[2,2]*l[0]*l[1]/(18*l[2]) - k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]),
           (l[0]**2*(4*k[1,1]*l[2]**2 - 3*k[1,2]*l[1]*l[2] - 3*k[2,1]*l[1]*l[2] + 4*k[2,2]*l[1]**2) + 3*l[0]*l[1]*l[2]*(-k[0,1]*l[2] + k[0,2]*l[1] - k[1,0]*l[2] + 2*k[1,1]*l[2]*np.tan(g[0]) - k[1,2]*l[1]*np.tan(g[0]) + k[1,2]*l[2]*np.tan(g[1]) + k[2,0]*l[1] - k[2,1]*l[1]*np.tan(g[0]) + k[2,1]*l[2]*np.tan(g[1]) - 2*k[2,2]*l[1]*np.tan(g[1])) + 4*l[1]**2*l[2]**2*(k[0,0] - k[0,1]*np.tan(g[0]) - k[0,2]*np.tan(g[1]) - k[1,0]*np.tan(g[0]) + k[1,1]*np.tan(g[0])**2 + k[1,2]*np.tan(g[0])*np.tan(g[1]) - k[2,0]*np.tan(g[1]) + k[2,1]*np.tan(g[0])*np.tan(g[1]) + k[2,2]*np.tan(g[1])**2))/(36*l[0]*l[1]*l[2]),
           (2*l[0]**2*(-4*k[1,1]*l[2]**2 - 3*k[1,2]*l[1]*l[2] + 3*k[2,1]*l[1]*l[2] + 2*k[2,2]*l[1]**2) + 3*l[0]*l[1]*l[2]*(2*k[0,1]*l[2] + k[0,2]*l[1] - 2*k[1,0]*l[2] - k[1,2]*l[1]*np.tan(g[0]) + 2*k[1,2]*l[2]*np.tan(g[1]) + k[2,0]*l[1] - k[2,1]*l[1]*np.tan(g[0]) - 2*k[2,1]*l[2]*np.tan(g[1]) - 2*k[2,2]*l[1]*np.tan(g[1])) + 4*l[1]**2*l[2]**2*(k[0,0] - k[0,1]*np.tan(g[0]) - k[0,2]*np.tan(g[1]) - k[1,0]*np.tan(g[0]) + k[1,1]*np.tan(g[0])**2 + k[1,2]*np.tan(g[0])*np.tan(g[1]) - k[2,0]*np.tan(g[1]) + k[2,1]*np.tan(g[0])*np.tan(g[1]) + k[2,2]*np.tan(g[1])**2))/(72*l[0]*l[1]*l[2]),
           -k[0,0]*l[1]*l[2]/(18*l[0]) + k[0,1]*l[2]/12 + k[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + k[0,2]*l[1]/24 + k[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + k[1,0]*l[2]/12 + k[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - k[1,1]*l[0]*l[2]/(18*l[1]) - k[1,1]*l[2]*np.tan(g[0])/6 - k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - k[1,2]*l[0]/24 - k[1,2]*l[1]*np.tan(g[0])/24 - k[1,2]*l[2]*np.tan(g[1])/12 - k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - k[2,0]*l[1]/24 + k[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + k[2,1]*l[0]/24 + k[2,1]*l[1]*np.tan(g[0])/24 - k[2,1]*l[2]*np.tan(g[1])/12 - k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + k[2,2]*l[0]*l[1]/(36*l[2]) - k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0])],
          [-k[0,0]*l[1]*l[2]/(36*l[0]) - k[0,1]*l[2]/24 + k[0,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - k[0,2]*l[1]/24 + k[0,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - k[1,0]*l[2]/24 + k[1,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - k[1,1]*l[0]*l[2]/(36*l[1]) + k[1,1]*l[2]*np.tan(g[0])/12 - k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - k[1,2]*l[0]/24 + k[1,2]*l[1]*np.tan(g[0])/24 + k[1,2]*l[2]*np.tan(g[1])/24 - k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - k[2,0]*l[1]/24 + k[2,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - k[2,1]*l[0]/24 + k[2,1]*l[1]*np.tan(g[0])/24 + k[2,1]*l[2]*np.tan(g[1])/24 - k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - k[2,2]*l[0]*l[1]/(36*l[2]) + k[2,2]*l[1]*np.tan(g[1])/12 - k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]),
           (-2*l[0]**2*(2*k[1,1]*l[2]**2 + 3*k[1,2]*l[1]*l[2] + 3*k[2,1]*l[1]*l[2] + 2*k[2,2]*l[1]**2) + 3*l[0]*l[1]*l[2]*(-k[0,1]*l[2] - k[0,2]*l[1] + k[1,0]*l[2] + k[1,2]*l[1]*np.tan(g[0]) - k[1,2]*l[2]*np.tan(g[1]) + k[2,0]*l[1] - k[2,1]*l[1]*np.tan(g[0]) + k[2,1]*l[2]*np.tan(g[1])) + 2*l[1]**2*l[2]**2*(k[0,0] - k[0,1]*np.tan(g[0]) - k[0,2]*np.tan(g[1]) - k[1,0]*np.tan(g[0]) + k[1,1]*np.tan(g[0])**2 + k[1,2]*np.tan(g[0])*np.tan(g[1]) - k[2,0]*np.tan(g[1]) + k[2,1]*np.tan(g[0])*np.tan(g[1]) + k[2,2]*np.tan(g[1])**2))/(72*l[0]*l[1]*l[2]),
           (2*l[0]**2*(2*k[1,1]*l[2]**2 - 3*k[1,2]*l[1]*l[2] + 3*k[2,1]*l[1]*l[2] - 4*k[2,2]*l[1]**2) + 3*l[0]*l[1]*l[2]*(k[0,1]*l[2] - 2*k[0,2]*l[1] + k[1,0]*l[2] - 2*k[1,1]*l[2]*np.tan(g[0]) + 2*k[1,2]*l[1]*np.tan(g[0]) - k[1,2]*l[2]*np.tan(g[1]) + 2*k[2,0]*l[1] - 2*k[2,1]*l[1]*np.tan(g[0]) - k[2,1]*l[2]*np.tan(g[1])) + 4*l[1]**2*l[2]**2*(k[0,0] - k[0,1]*np.tan(g[0]) - k[0,2]*np.tan(g[1]) - k[1,0]*np.tan(g[0]) + k[1,1]*np.tan(g[0])**2 + k[1,2]*np.tan(g[0])*np.tan(g[1]) - k[2,0]*np.tan(g[1]) + k[2,1]*np.tan(g[0])*np.tan(g[1]) + k[2,2]*np.tan(g[1])**2))/(72*l[0]*l[1]*l[2]),
           -k[0,0]*l[1]*l[2]/(18*l[0]) + k[0,1]*l[2]/24 + k[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - k[0,2]*l[1]/12 + k[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - k[1,0]*l[2]/24 + k[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + k[1,1]*l[0]*l[2]/(36*l[1]) - k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - k[1,2]*l[0]/24 + k[1,2]*l[1]*np.tan(g[0])/12 + k[1,2]*l[2]*np.tan(g[1])/24 - k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - k[2,0]*l[1]/12 + k[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + k[2,1]*l[0]/24 + k[2,1]*l[1]*np.tan(g[0])/12 - k[2,1]*l[2]*np.tan(g[1])/24 - k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - k[2,2]*l[0]*l[1]/(18*l[2]) + k[2,2]*l[1]*np.tan(g[1])/6 - k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]),
           -k[0,0]*l[1]*l[2]/(18*l[0]) - k[0,1]*l[2]/12 + k[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + k[0,2]*l[1]/24 + k[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - k[1,0]*l[2]/12 + k[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - k[1,1]*l[0]*l[2]/(18*l[1]) + k[1,1]*l[2]*np.tan(g[0])/6 - k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + k[1,2]*l[0]/24 - k[1,2]*l[1]*np.tan(g[0])/24 + k[1,2]*l[2]*np.tan(g[1])/12 - k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - k[2,0]*l[1]/24 + k[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - k[2,1]*l[0]/24 + k[2,1]*l[1]*np.tan(g[0])/24 + k[2,1]*l[2]*np.tan(g[1])/12 - k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + k[2,2]*l[0]*l[1]/(36*l[2]) - k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]),
           (2*l[0]**2*(-4*k[1,1]*l[2]**2 + 3*k[1,2]*l[1]*l[2] - 3*k[2,1]*l[1]*l[2] + 2*k[2,2]*l[1]**2) + 3*l[0]*l[1]*l[2]*(-2*k[0,1]*l[2] + k[0,2]*l[1] + 2*k[1,0]*l[2] - k[1,2]*l[1]*np.tan(g[0]) - 2*k[1,2]*l[2]*np.tan(g[1]) + k[2,0]*l[1] - k[2,1]*l[1]*np.tan(g[0]) + 2*k[2,1]*l[2]*np.tan(g[1]) - 2*k[2,2]*l[1]*np.tan(g[1])) + 4*l[1]**2*l[2]**2*(k[0,0] - k[0,1]*np.tan(g[0]) - k[0,2]*np.tan(g[1]) - k[1,0]*np.tan(g[0]) + k[1,1]*np.tan(g[0])**2 + k[1,2]*np.tan(g[0])*np.tan(g[1]) - k[2,0]*np.tan(g[1]) + k[2,1]*np.tan(g[0])*np.tan(g[1]) + k[2,2]*np.tan(g[1])**2))/(72*l[0]*l[1]*l[2]),
           (l[0]**2*(4*k[1,1]*l[2]**2 + 3*k[1,2]*l[1]*l[2] + 3*k[2,1]*l[1]*l[2] + 4*k[2,2]*l[1]**2) + 3*l[0]*l[1]*l[2]*(k[0,1]*l[2] + k[0,2]*l[1] + k[1,0]*l[2] - 2*k[1,1]*l[2]*np.tan(g[0]) - k[1,2]*l[1]*np.tan(g[0]) - k[1,2]*l[2]*np.tan(g[1]) + k[2,0]*l[1] - k[2,1]*l[1]*np.tan(g[0]) - k[2,1]*l[2]*np.tan(g[1]) - 2*k[2,2]*l[1]*np.tan(g[1])) + 4*l[1]**2*l[2]**2*(k[0,0] - k[0,1]*np.tan(g[0]) - k[0,2]*np.tan(g[1]) - k[1,0]*np.tan(g[0]) + k[1,1]*np.tan(g[0])**2 + k[1,2]*np.tan(g[0])*np.tan(g[1]) - k[2,0]*np.tan(g[1]) + k[2,1]*np.tan(g[0])*np.tan(g[1]) + k[2,2]*np.tan(g[1])**2))/(36*l[0]*l[1]*l[2]),
           -k[0,0]*l[1]*l[2]/(9*l[0]) + k[0,1]*l[2]/12 + k[0,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + k[0,2]*l[1]/12 + k[0,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - k[1,0]*l[2]/12 + k[1,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + k[1,1]*l[0]*l[2]/(18*l[1]) - k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + k[1,2]*l[0]/24 - k[1,2]*l[1]*np.tan(g[0])/12 + k[1,2]*l[2]*np.tan(g[1])/12 - k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - k[2,0]*l[1]/12 + k[2,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + k[2,1]*l[0]/24 + k[2,1]*l[1]*np.tan(g[0])/12 - k[2,1]*l[2]*np.tan(g[1])/12 - k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + k[2,2]*l[0]*l[1]/(18*l[2]) - k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0])],
          [k[0,0]*l[1]*l[2]/(36*l[0]) + k[0,1]*l[2]/24 - k[0,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + k[0,2]*l[1]/24 - k[0,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - k[1,0]*l[2]/24 - k[1,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - k[1,1]*l[0]*l[2]/(18*l[1]) + k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - k[1,2]*l[0]/12 - k[1,2]*l[1]*np.tan(g[0])/24 + k[1,2]*l[2]*np.tan(g[1])/24 + k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - k[2,0]*l[1]/24 - k[2,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - k[2,1]*l[0]/12 + k[2,1]*l[1]*np.tan(g[0])/24 - k[2,1]*l[2]*np.tan(g[1])/24 + k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - k[2,2]*l[0]*l[1]/(18*l[2]) + k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]),
           -k[0,0]*l[1]*l[2]/(36*l[0]) + k[0,1]*l[2]/24 + k[0,1]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) + k[0,2]*l[1]/24 + k[0,2]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) + k[1,0]*l[2]/24 + k[1,0]*l[1]*l[2]*np.tan(g[0])/(36*l[0]) - k[1,1]*l[0]*l[2]/(36*l[1]) - k[1,1]*l[2]*np.tan(g[0])/12 - k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(36*l[0]) - k[1,2]*l[0]/24 - k[1,2]*l[1]*np.tan(g[0])/24 - k[1,2]*l[2]*np.tan(g[1])/24 - k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) + k[2,0]*l[1]/24 + k[2,0]*l[1]*l[2]*np.tan(g[1])/(36*l[0]) - k[2,1]*l[0]/24 - k[2,1]*l[1]*np.tan(g[0])/24 - k[2,1]*l[2]*np.tan(g[1])/24 - k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(36*l[0]) - k[2,2]*l[0]*l[1]/(36*l[2]) - k[2,2]*l[1]*np.tan(g[1])/12 - k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(36*l[0]),
           -k[0,0]*l[1]*l[2]/(18*l[0]) - k[0,1]*l[2]/24 + k[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + k[0,2]*l[1]/12 + k[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + k[1,0]*l[2]/24 + k[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + k[1,1]*l[0]*l[2]/(36*l[1]) - k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - k[1,2]*l[0]/24 - k[1,2]*l[1]*np.tan(g[0])/12 - k[1,2]*l[2]*np.tan(g[1])/24 - k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + k[2,0]*l[1]/12 + k[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + k[2,1]*l[0]/24 - k[2,1]*l[1]*np.tan(g[0])/12 + k[2,1]*l[2]*np.tan(g[1])/24 - k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - k[2,2]*l[0]*l[1]/(18*l[2]) - k[2,2]*l[1]*np.tan(g[1])/6 - k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]),
           k[0,0]*l[1]*l[2]/(18*l[0]) - k[0,1]*l[2]/24 - k[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + k[0,2]*l[1]/12 - k[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - k[1,0]*l[2]/24 - k[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) + k[1,1]*l[0]*l[2]/(18*l[1]) + k[1,1]*l[2]*np.tan(g[0])/12 + k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) - k[1,2]*l[0]/12 - k[1,2]*l[1]*np.tan(g[0])/12 + k[1,2]*l[2]*np.tan(g[1])/24 + k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - k[2,0]*l[1]/12 - k[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + k[2,1]*l[0]/12 + k[2,1]*l[1]*np.tan(g[0])/12 + k[2,1]*l[2]*np.tan(g[1])/24 + k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - k[2,2]*l[0]*l[1]/(9*l[2]) + k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]),
           k[0,0]*l[1]*l[2]/(18*l[0]) + k[0,1]*l[2]/12 - k[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - k[0,2]*l[1]/24 - k[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - k[1,0]*l[2]/12 - k[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - k[1,1]*l[0]*l[2]/(9*l[1]) + k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + k[1,2]*l[0]/12 + k[1,2]*l[1]*np.tan(g[0])/24 + k[1,2]*l[2]*np.tan(g[1])/12 + k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) - k[2,0]*l[1]/24 - k[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - k[2,1]*l[0]/12 + k[2,1]*l[1]*np.tan(g[0])/24 - k[2,1]*l[2]*np.tan(g[1])/12 + k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + k[2,2]*l[0]*l[1]/(18*l[2]) + k[2,2]*l[1]*np.tan(g[1])/12 + k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]),
           -k[0,0]*l[1]*l[2]/(18*l[0]) + k[0,1]*l[2]/12 + k[0,1]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - k[0,2]*l[1]/24 + k[0,2]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) + k[1,0]*l[2]/12 + k[1,0]*l[1]*l[2]*np.tan(g[0])/(18*l[0]) - k[1,1]*l[0]*l[2]/(18*l[1]) - k[1,1]*l[2]*np.tan(g[0])/6 - k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(18*l[0]) + k[1,2]*l[0]/24 + k[1,2]*l[1]*np.tan(g[0])/24 - k[1,2]*l[2]*np.tan(g[1])/12 - k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + k[2,0]*l[1]/24 + k[2,0]*l[1]*l[2]*np.tan(g[1])/(18*l[0]) - k[2,1]*l[0]/24 - k[2,1]*l[1]*np.tan(g[0])/24 - k[2,1]*l[2]*np.tan(g[1])/12 - k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(18*l[0]) + k[2,2]*l[0]*l[1]/(36*l[2]) - k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(18*l[0]),
           -k[0,0]*l[1]*l[2]/(9*l[0]) - k[0,1]*l[2]/12 + k[0,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - k[0,2]*l[1]/12 + k[0,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + k[1,0]*l[2]/12 + k[1,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + k[1,1]*l[0]*l[2]/(18*l[1]) - k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + k[1,2]*l[0]/24 + k[1,2]*l[1]*np.tan(g[0])/12 - k[1,2]*l[2]*np.tan(g[1])/12 - k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + k[2,0]*l[1]/12 + k[2,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + k[2,1]*l[0]/24 - k[2,1]*l[1]*np.tan(g[0])/12 + k[2,1]*l[2]*np.tan(g[1])/12 - k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + k[2,2]*l[0]*l[1]/(18*l[2]) - k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0]),
           k[0,0]*l[1]*l[2]/(9*l[0]) - k[0,1]*l[2]/12 - k[0,1]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) - k[0,2]*l[1]/12 - k[0,2]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) - k[1,0]*l[2]/12 - k[1,0]*l[1]*l[2]*np.tan(g[0])/(9*l[0]) + k[1,1]*l[0]*l[2]/(9*l[1]) + k[1,1]*l[2]*np.tan(g[0])/6 + k[1,1]*l[1]*l[2]*np.tan(g[0])**2/(9*l[0]) + k[1,2]*l[0]/12 + k[1,2]*l[1]*np.tan(g[0])/12 + k[1,2]*l[2]*np.tan(g[1])/12 + k[1,2]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) - k[2,0]*l[1]/12 - k[2,0]*l[1]*l[2]*np.tan(g[1])/(9*l[0]) + k[2,1]*l[0]/12 + k[2,1]*l[1]*np.tan(g[0])/12 + k[2,1]*l[2]*np.tan(g[1])/12 + k[2,1]*l[1]*l[2]*np.tan(g[0])*np.tan(g[1])/(9*l[0]) + k[2,2]*l[0]*l[1]/(9*l[2]) + k[2,2]*l[1]*np.tan(g[1])/6 + k[2,2]*l[1]*l[2]*np.tan(g[1])**2/(9*l[0])]]) 