# SPDX-License-Identifier: GPL-3.0-or-later
from typing import List

from symfem.functions import VectorFunction, MatrixFunction

def shape_function_matrix(basis: List, nedof: int,
                          mode: str = "col") -> MatrixFunction:
    """
    Generate the shape function matrix.

    Parameters
    ----------
    basis : list
        list of basis functions as generated by base_cell
    nedof : int
        number of nodal degrees of freedom.
    mode : str
        either "row" or "col" which results in shape (nedof,n_nodes) or
        (n_nodes,nedof)

    Returns
    -------
    shape_function_matrix : symfem.functions.MatrixFunction
        shape function matrix either of shape
    """
    #
    if isinstance(basis, list):
        n_nodes = len(basis)
    elif isinstance(basis, (VectorFunction,MatrixFunction)):
        n_nodes = basis.shape[0]
    if mode in ["row","col"]:
        #
        shpfc_matr = [[0 for j in range(nedof*n_nodes)] for i in range(nedof)]
        #
        for i in range(nedof):
            shpfc_matr[i][i::nedof] = basis
    else:
        raise ValueError("Unknown construction mode.")
    if mode == "col":
        return MatrixFunction(shpfc_matr).transpose()
    else:
        return MatrixFunction(shpfc_matr)