"""
Auto-detect Python environment (version, packages) for Docker generation.
Users don't need to configure anything - we snapshot their current working environment.
"""

import subprocess
import sys
import logging
from pathlib import Path
from typing import Optional


class EnvironmentDetector:
    """Detect Python version and dependencies from user's current environment."""
    
    def __init__(self, working_directory: str):
        self.working_directory = Path(working_directory)
    
    def detect_python_version(self) -> str:
        """
        Detect Python version from current environment.
        
        Returns:
            Python version string like "3.10", "3.11", etc.
        """
        try:
            # Use the Python that's currently running (user's environment)
            version_info = sys.version_info
            # Return major.minor (e.g., "3.10")
            python_version = f"{version_info.major}.{version_info.minor}"
            logging.info(f"Detected Python version: {python_version}")
            return python_version
        except Exception as e:
            logging.warning(f"Failed to detect Python version: {e}, defaulting to 3.10")
            return "3.10"
    
    def detect_requirements(self) -> list[str]:
        """
        Detect requirements from user's CURRENT ACTIVE environment.
        
        Strategy:
        1. Check if requirements.txt exists in user's project → use it
        2. Otherwise, use pip freeze from the currently active environment
        3. Filter out problematic packages (conda internals, editable installs)
        
        Returns:
            List of requirement strings (e.g., ["pandas==1.5.0", "numpy>=1.20.0"])
        """
        # Strategy 1: Use existing requirements.txt if present
        req_file = self.working_directory / "requirements.txt"
        if req_file.exists():
            logging.info("Found existing requirements.txt in project")
            try:
                requirements = req_file.read_text().strip().split('\n')
                # Filter out empty lines and comments
                requirements = [
                    req.strip() 
                    for req in requirements 
                    if req.strip() and not req.strip().startswith('#')
                ]
                if requirements:
                    logging.info(f"Using {len(requirements)} packages from requirements.txt")
                    return requirements
            except Exception as e:
                logging.warning(f"Failed to read requirements.txt: {e}")
        
        # Strategy 2: Snapshot pip freeze from user's active environment
        logging.info("No requirements.txt found. Snapshotting current environment with pip freeze...")
        try:
            result = subprocess.run(
                [sys.executable, "-m", "pip", "freeze"],
                capture_output=True,
                text=True,
                check=True
            )
            
            requirements = result.stdout.strip().split('\n')
            
            # Filter out problematic entries
            filtered_requirements = []
            for req in requirements:
                req = req.strip()
                if not req:
                    continue
                
                # Skip editable installs
                if req.startswith('-e'):
                    logging.debug(f"   Skipping editable install: {req}")
                    continue
                
                # Skip conda build artifacts
                if '/croot/' in req or '@ file://' in req:
                    logging.debug(f"   Skipping conda artifact: {req}")
                    continue
                
                # Skip local paths
                if req.startswith('/') or req.startswith('./') or req.startswith('file://'):
                    logging.debug(f"   Skipping local path: {req}")
                    continue
                
                filtered_requirements.append(req)
            
            logging.info(f"Captured {len(filtered_requirements)} packages from active environment")
            logging.info(f"   (filtered from {len(requirements)} total)")
            return filtered_requirements
            
        except subprocess.CalledProcessError as e:
            logging.error(f"Failed to run pip freeze: {e}")
            return []
        except Exception as e:
            logging.error(f"Error detecting requirements: {e}")
            return []
    
    def detect_data_path(self) -> Optional[str]:
        """
        Try to detect if user is loading data from a specific directory.
        This is optional and helps with auto-mounting volumes.
        
        Returns:
            Path to data directory if detected, None otherwise
        """
        # Common data directory names
        data_dirs = ['data', 'dataset', 'datasets', 'input', 'Data']
        
        for dirname in data_dirs:
            data_path = self.working_directory / dirname
            if data_path.exists() and data_path.is_dir():
                # Check if it has files
                if list(data_path.iterdir()):
                    logging.info(f"Detected data directory: {dirname}/")
                    return str(data_path.absolute())
        
        return None
    
    def get_environment_snapshot(self) -> dict:
        """
        Get complete snapshot of user's environment for Docker generation.
        
        Returns:
            Dictionary with python_version, requirements, and optional data_path
        """
        snapshot = {
            'python_version': self.detect_python_version(),
            'requirements': self.detect_requirements(),
            'data_path': self.detect_data_path(),
        }
        
        logging.info("Environment snapshot:")
        logging.info(f"   Python: {snapshot['python_version']}")
        logging.info(f"   Packages: {len(snapshot['requirements'])} detected")
        if snapshot['data_path']:
            logging.info(f"   Data: {snapshot['data_path']}")
        
        return snapshot

