# Release Process

This document outlines the release process for DECOYABLE, including versioning strategy, release preparation, and deployment procedures.

## Versioning Strategy

DECOYABLE follows [Semantic Versioning](https://semver.org/) (SemVer):

```
MAJOR.MINOR.PATCH[-PRERELEASE][+BUILD]
```

### Version Components

- **MAJOR**: Breaking changes that are not backward compatible
- **MINOR**: New features that are backward compatible
- **PATCH**: Bug fixes and minor changes that are backward compatible
- **PRERELEASE**: Pre-release identifiers (alpha, beta, rc)
- **BUILD**: Build metadata (timestamps, commit hashes)

### Examples

- `1.0.0` - First stable release
- `1.1.0` - New features added
- `1.1.1` - Bug fixes
- `2.0.0` - Breaking changes
- `1.2.0-alpha.1` - Pre-release
- `1.2.0-rc.1+20231201` - Release candidate with build metadata

## Release Types

### Major Releases (X.0.0)

- Breaking API changes
- Significant architectural changes
- Major feature additions
- Released quarterly or when major milestones are reached

### Minor Releases (X.Y.0)

- New features and enhancements
- Backward compatible API additions
- Released monthly or when feature milestones are completed

### Patch Releases (X.Y.Z)

- Bug fixes and security patches
- Documentation updates
- Performance improvements
- Released as needed, typically weekly

### Pre-releases

- **Alpha**: Early testing, APIs may change
- **Beta**: Feature complete, minor changes possible
- **RC (Release Candidate)**: Ready for production, only critical fixes

## Release Workflow

### 1. Preparation Phase

#### Create Release Branch

```bash
git checkout -b release/v1.2.0
```

#### Update Version Numbers

Update version in key files:

- `pyproject.toml`
- `decoyable/__init__.py`
- `README.md` (if version is mentioned)
- `docker-compose.yml` (if applicable)

#### Update Changelog

Update `CHANGELOG.md` with all changes since last release:

```markdown
## [1.2.0] - 2025-09-21

### Added
- New LLM provider support for Anthropic Claude
- Enhanced security scanning capabilities

### Changed
- Improved error handling in API endpoints
- Updated dependencies for better performance

### Fixed
- Memory leak in long-running analysis tasks
- Incorrect validation in honeypot configuration

### Security
- Fixed potential information disclosure in logs
```

#### Run Final Tests

```bash
# Run full test suite
python -m pytest tests/ -v

# Run security scans
python -m bandit -r decoyable/
python -m safety check

# Run linting
python -m ruff check decoyable/ tests/
python -m mypy decoyable/
```

### 2. Pre-release Phase

#### Create GitHub Pre-release

1. Go to GitHub Releases
2. Click "Create a new release"
3. Use tag format: `v1.2.0-rc.1`
4. Mark as pre-release
5. Include changelog excerpt

#### Test Pre-release

- Deploy to staging environment
- Run integration tests
- Gather feedback from beta testers

### 3. Release Phase

#### Final Release Preparation

```bash
# Tag the release
git tag -a v1.2.0 -m "Release version 1.2.0"

# Push tags
git push origin v1.2.0
```

#### Create GitHub Release

1. Go to GitHub Releases
2. Click "Create a new release"
3. Use tag: `v1.2.0`
4. Title: "DECOYABLE v1.2.0"
5. Copy full changelog
6. **Do not** mark as pre-release

#### Publish to PyPI

```bash
# Build distribution
python -m build

# Upload to PyPI (requires API token)
python -m twine upload dist/*
```

#### Update Docker Images

```bash
# Build and push Docker image
docker build -t kolerr/decoyable:v1.2.0 .
docker push kolerr/decoyable:v1.2.0

# Update latest tag
docker tag kolerr/decoyable:v1.2.0 kolerr/decoyable:latest
docker push kolerr/decoyable:latest
```

### 4. Post-release Phase

#### Update Documentation

- Update version badges in README
- Update API documentation if needed
- Update installation instructions

#### Announce Release

- Create discussion post on GitHub
- Send announcement to mailing list (if applicable)
- Update social media channels

#### Monitor and Support

- Watch for issues and bug reports
- Provide support for upgrade issues
- Plan next release cycle

## Release Checklist

### Pre-release

- [ ] Release branch created
- [ ] Version numbers updated
- [ ] Changelog updated
- [ ] All tests pass
- [ ] Security scans pass
- [ ] Code quality checks pass
- [ ] Documentation updated

### Release

- [ ] Git tag created and pushed
- [ ] GitHub release created
- [ ] PyPI package published
- [ ] Docker images built and pushed
- [ ] Release announced

### Post-release

- [ ] Documentation updated with new version
- [ ] Release communicated to community
- [ ] Next release cycle planned

## Hotfix Releases

For critical security fixes or important bug fixes:

1. Create hotfix branch from the release tag: `git checkout -b hotfix/v1.2.1 v1.2.0`
2. Apply minimal fixes only
3. Follow patch release process
4. Update version to `1.2.1`

## Deprecation Policy

- Features are deprecated with a **minor release**
- Deprecated features are removed in the **next major release**
- Deprecated features show warnings for at least one minor release cycle

## Support Policy

- **Latest major version**: Full support
- **Previous major version**: Security fixes only (6 months)
- **Older versions**: No support

## Release Automation

Future releases may be automated using:

- **GitHub Actions** for automated releases
- **Release Drafter** for changelog generation
- **Semantic PRs** for automatic versioning
- **Dependabot** for dependency updates

## Contact

For release-related questions, contact the maintainers or create an issue with the `release` label.
 
 
