"""
CLI Service Layer

Refactored CLI service with dependency injection and clean command structure.
"""

import argparse
import json
import sys
from pathlib import Path
from typing import Any, Dict, Optional

from decoyable.core.config import Settings
from decoyable.core.logging import LoggingService, get_logger
from decoyable.core.registry import ServiceRegistry


class CLIService:
    """CLI service that manages command execution with dependency injection."""

    def __init__(self, config: Settings, registry: ServiceRegistry, logging_service: LoggingService):
        self.config = config
        self.registry = registry
        self.logging_service = logging_service
        self.logger = get_logger("cli.service")

    async def run_scan_command(self, args: argparse.Namespace) -> int:
        """Run security scan commands."""
        scanner_service = self.registry.get_by_name("scanner_service")
        if not scanner_service:
            self.logger.error("Scanner service not available")
            return 1

        database_service = self.registry.get_by_name("database_service")

        scan_type = getattr(args, "scan_type", "all")
        target_path = getattr(args, "path", ".")
        output_format = getattr(args, "format", "text")

        self.logger.info(f"Starting {scan_type} scan on: {target_path}")

        try:
            import time

            start_time = time.time()

            # Use the scanner service instead of direct imports
            if scan_type in ("secrets", "all"):
                self.logger.info("Scanning for exposed secrets...")
                findings = (await scanner_service.scan_secrets(target_path)).results

                if findings:
                    self.logger.warning(f"Found {len(findings)} potential secrets:")
                    for finding in findings:
                        print(f"{finding.filename}:{finding.lineno} [{finding.secret_type}] {finding.masked()}")
                        if output_format == "verbose":
                            print(f"  Context: {finding.context}")
                    if scan_type == "secrets":
                        # Store result in database if available
                        if database_service:
                            import asyncio

                            asyncio.create_task(
                                database_service.store_scan_result(
                                    scan_type="secrets",
                                    target_path=target_path,
                                    status="success",
                                    results={"findings": findings, "count": len(findings)},
                                    scan_duration=int(time.time() - start_time),
                                    file_count=len({f["filename"] for f in findings}) if findings else 0,
                                )
                            )
                        return 1  # Exit with error if secrets found
                else:
                    self.logger.info("No secrets found.")

            if scan_type in ("deps", "all"):
                self.logger.info("Scanning for dependency issues...")
                result = (await scanner_service.scan_dependencies(target_path)).results

                # result is a list of DependencyIssue objects
                missing_deps = [dep for dep in result if dep.issue_type == "missing_import"]
                unused_deps = [dep for dep in result if dep.issue_type == "unused_dependency"]

                if missing_deps or unused_deps:
                    if missing_deps:
                        self.logger.warning(f"Found {len(missing_deps)} missing dependencies:")
                        for dep in missing_deps:
                            print(f"  {dep.module_name}: {dep.description}")
                            if dep.suggestions:
                                print(f"    Suggestions: {', '.join(dep.suggestions)}")

                    if unused_deps:
                        self.logger.warning(f"Found {len(unused_deps)} unused dependencies:")
                        for dep in unused_deps:
                            print(f"  {dep.module_name}: {dep.description}")

                    if scan_type == "deps":
                        # Store result in database if available
                        if database_service:
                            import asyncio

                            asyncio.create_task(
                                database_service.store_scan_result(
                                    scan_type="dependencies",
                                    target_path=target_path,
                                    status="success",
                                    results={"missing_deps": len(missing_deps), "unused_deps": len(unused_deps)},
                                    scan_duration=int(time.time() - start_time),
                                )
                            )
                        return 1
                else:
                    self.logger.info("All dependencies appear to be satisfied.")

            if scan_type in ("sast", "all"):
                self.logger.info("Performing Static Application Security Testing (SAST)...")
                result = (await scanner_service.scan_sast(target_path)).results

                # result is a list of Vulnerability objects
                vulnerabilities = result

                if vulnerabilities:
                    self.logger.warning(f"Found {len(vulnerabilities)} potential security vulnerabilities:")
                    severity_order = ["CRITICAL", "HIGH", "MEDIUM", "LOW", "INFO"]

                    for vuln in sorted(
                        vulnerabilities,
                        key=lambda x: severity_order.index(
                            x.severity.value if hasattr(x.severity, "value") else str(x.severity)
                        ),
                    ):
                        severity = vuln.severity.value if hasattr(vuln.severity, "value") else vuln.severity
                        vuln_type = (
                            vuln.vulnerability_type.value
                            if hasattr(vuln.vulnerability_type, "value")
                            else vuln.vulnerability_type
                        )
                        print(f"[{severity}] {vuln_type} - {vuln.file_path}:{vuln.line_number}")
                        print(f"  {vuln.description}")
                        if output_format == "verbose":
                            print(f"  Recommendation: {vuln.recommendation}")
                            print("  Code snippet:")
                            for line in vuln.code_snippet.split("\n"):
                                print(f"    {line}")
                            print()

                    if scan_type == "sast":
                        # Store result in database if available
                        if database_service:
                            import asyncio

                            asyncio.create_task(
                                database_service.store_scan_result(
                                    scan_type="sast",
                                    target_path=target_path,
                                    status="success",
                                    results=result,
                                    scan_duration=int(time.time() - start_time),
                                    file_count=result.get("summary", {}).get("files_scanned", 0),
                                )
                            )
                        return 1
                else:
                    self.logger.info("No security vulnerabilities found.")

                # Print summary
                if vulnerabilities:
                    severity_counts = {}
                    for vuln in vulnerabilities:
                        severity = vuln.severity.value if hasattr(vuln.severity, "value") else str(vuln.severity)
                        severity_counts[severity] = severity_counts.get(severity, 0) + 1

                    print(f"\nSummary: {len(vulnerabilities)} vulnerabilities found")
                    print(f"Files scanned: {len({v.file_path for v in vulnerabilities})}")
                    if severity_counts:
                        print("Severity breakdown:")
                        for severity, count in severity_counts.items():
                            print(f"  {severity}: {count}")

            scan_duration = int(time.time() - start_time)
            self.logger.info(f"Scan completed successfully in {scan_duration}s.")

            # Store successful scan result
            if database_service and scan_type == "all":
                import asyncio

                asyncio.create_task(
                    database_service.store_scan_result(
                        scan_type="all",
                        target_path=target_path,
                        status="success",
                        results={"message": "Comprehensive scan completed"},
                        scan_duration=scan_duration,
                    )
                )

            return 0

        except Exception as exc:
            scan_duration = int(time.time() - start_time) if "start_time" in locals() else 0
            self.logger.exception(f"Scan failed: {exc}")

            # Store error result in database if available
            if database_service:
                import asyncio

                asyncio.create_task(
                    database_service.store_scan_result(
                        scan_type=scan_type,
                        target_path=target_path,
                        status="error",
                        error_message=str(exc),
                        scan_duration=scan_duration,
                    )
                )

            return 1

    def run_task_command(self, args: argparse.Namespace) -> int:
        """Run task queue commands."""
        task_queue_service = self.registry.get_by_name("task_queue_service")
        if not task_queue_service:
            self.logger.error("Task queue service not available")
            return 1

        command = getattr(args, "task_command", "status")

        try:
            if command == "submit":
                return self._handle_task_submit(args, task_queue_service)
            elif command == "status":
                return self._handle_task_status(args, task_queue_service)
            elif command == "cancel":
                return self._handle_task_cancel(args, task_queue_service)
            elif command == "stats":
                return self._handle_task_stats(args, task_queue_service)
            else:
                self.logger.error(f"Unknown task command: {command}")
                return 1
        except Exception as exc:
            self.logger.exception(f"Task command failed: {exc}")
            return 1

    def _handle_task_submit(self, args: argparse.Namespace, task_queue_service) -> int:
        """Handle task submission."""
        import asyncio

        scan_type = getattr(args, "scan_type", "all")
        target_path = getattr(args, "path", ".")

        async def submit_task():
            task_id = await task_queue_service.submit_scan_task(scan_type, target_path)
            self.logger.info(f"Submitted {scan_type} scan task: {task_id}")
            print(f"Task submitted successfully. Task ID: {task_id}")
            return 0

        try:
            return asyncio.run(submit_task())
        except Exception as e:
            self.logger.error(f"Failed to submit task: {e}")
            return 1

    def _handle_task_status(self, args: argparse.Namespace, task_queue_service) -> int:
        """Handle task status checking."""
        import asyncio

        task_id = getattr(args, "task_id", None)
        if not task_id:
            self.logger.error("Task ID required for status check")
            return 1

        async def check_status():
            if getattr(args, "group", False):
                status = await task_queue_service.get_group_status(task_id)
            else:
                status = await task_queue_service.get_task_status(task_id)

            import json

            print(json.dumps(status, indent=2))
            return 0

        try:
            return asyncio.run(check_status())
        except Exception as e:
            self.logger.error(f"Failed to check task status: {e}")
            return 1

    def _handle_task_cancel(self, args: argparse.Namespace, task_queue_service) -> int:
        """Handle task cancellation."""
        import asyncio

        task_id = getattr(args, "task_id", None)
        if not task_id:
            self.logger.error("Task ID required for cancellation")
            return 1

        async def cancel_task():
            success = await task_queue_service.cancel_task(task_id)
            if success:
                self.logger.info(f"Cancelled task: {task_id}")
                print(f"Task {task_id} cancelled successfully")
                return 0
            else:
                self.logger.error(f"Failed to cancel task: {task_id}")
                return 1

        try:
            return asyncio.run(cancel_task())
        except Exception as e:
            self.logger.error(f"Failed to cancel task: {e}")
            return 1

    def _handle_task_stats(self, args: argparse.Namespace, task_queue_service) -> int:
        """Handle task queue statistics."""
        import asyncio

        async def get_stats():
            stats = await task_queue_service.get_queue_stats()
            import json

            print(json.dumps(stats, indent=2))
            return 0

        try:
            return asyncio.run(get_stats())
        except Exception as e:
            self.logger.error(f"Failed to get task stats: {e}")
            return 1

    def run_streaming_command(self, args: argparse.Namespace) -> int:
        """Run streaming commands."""
        streaming_service = self.registry.get_by_name("streaming_service")
        if not streaming_service:
            self.logger.error("Streaming service not available")
            return 1

        command = getattr(args, "streaming_command", "status")

        try:
            if command == "status":
                return self._handle_streaming_status(args, streaming_service)
            elif command == "publish":
                return self._handle_streaming_publish(args, streaming_service)
            elif command == "start":
                return self._handle_streaming_start(args, streaming_service)
            elif command == "stop":
                return self._handle_streaming_stop(args, streaming_service)
            elif command == "alert":
                return self._handle_streaming_alert(args, streaming_service)
            else:
                self.logger.error(f"Unknown streaming command: {command}")
                return 1
        except Exception as exc:
            self.logger.exception(f"Streaming command failed: {exc}")
            return 1

    def _handle_streaming_status(self, args: argparse.Namespace, streaming_service) -> int:
        """Handle streaming status check."""
        import asyncio

        async def get_status():
            stats = await streaming_service.get_streaming_stats()
            health = await streaming_service.health_check()
            result = {"stats": stats, "health": health}
            import json

            print(json.dumps(result, indent=2))
            return 0

        try:
            return asyncio.run(get_status())
        except Exception as e:
            self.logger.error(f"Failed to get streaming status: {e}")
            return 1

    def _handle_streaming_publish(self, args: argparse.Namespace, streaming_service) -> int:
        """Handle streaming event publishing."""
        import asyncio

        event_type = getattr(args, "event_type", "test_event")
        data = getattr(args, "data", "{}")
        key = getattr(args, "key", None)

        try:
            event_data = json.loads(data) if data else {}
        except json.JSONDecodeError as e:
            self.logger.error(f"Invalid JSON data: {e}")
            return 1

        async def publish_event():
            success = await streaming_service.publish_attack_event(event_type=event_type, data=event_data, key=key)
            if success:
                self.logger.info(f"Published {event_type} event successfully")
                print("Event published successfully")
                return 0
            else:
                self.logger.error(f"Failed to publish {event_type} event")
                return 1

        try:
            return asyncio.run(publish_event())
        except Exception as e:
            self.logger.error(f"Failed to publish event: {e}")
            return 1

    def _handle_streaming_start(self, args: argparse.Namespace, streaming_service) -> int:
        """Handle streaming consumer start."""
        import asyncio

        async def start_consumers():
            await streaming_service.initialize()
            await streaming_service.start_consumers()
            self.logger.info("Streaming consumers started")
            print("Streaming consumers started")
            return 0

        try:
            return asyncio.run(start_consumers())
        except Exception as e:
            self.logger.error(f"Failed to start streaming consumers: {e}")
            return 1

    def _handle_streaming_stop(self, args: argparse.Namespace, streaming_service) -> int:
        """Handle streaming consumer stop."""
        import asyncio

        async def stop_consumers():
            await streaming_service.stop_consumers()
            self.logger.info("Streaming consumers stopped")
            print("Streaming consumers stopped")
            return 0

        try:
            return asyncio.run(stop_consumers())
        except Exception as e:
            self.logger.error(f"Failed to stop streaming consumers: {e}")
            return 1

    def _handle_streaming_alert(self, args: argparse.Namespace, streaming_service) -> int:
        """Handle security alert publishing."""
        import asyncio

        alert_type = getattr(args, "alert_type", "general")
        severity = getattr(args, "severity", "medium")
        message = getattr(args, "message", "Security alert")
        source_ip = getattr(args, "source_ip", None)
        details = getattr(args, "details", "{}")

        try:
            alert_details = json.loads(details) if details else {}
        except json.JSONDecodeError as e:
            self.logger.error(f"Invalid JSON details: {e}")
            return 1

        async def publish_alert():
            success = await streaming_service.publish_security_alert(
                alert_type=alert_type, severity=severity, message=message, source_ip=source_ip, details=alert_details
            )
            if success:
                self.logger.info(f"Published security alert: {alert_type}")
                print("Security alert published successfully")
                return 0
            else:
                self.logger.error(f"Failed to publish security alert: {alert_type}")
                return 1

        try:
            return asyncio.run(publish_alert())
        except Exception as e:
            self.logger.error(f"Failed to publish alert: {e}")
            return 1

    def run_honeypot_command(self, args: argparse.Namespace) -> int:
        """Run honeypot commands."""
        honeypot_service = self.registry.get_by_name("honeypot_service")
        if not honeypot_service:
            self.logger.error("Honeypot service not available")
            return 1

        command = getattr(args, "honeypot_command", "status")

        try:
            if command == "status":
                return self._handle_honeypot_status(args, honeypot_service)
            elif command == "attacks":
                return self._handle_honeypot_attacks(args, honeypot_service)
            elif command == "patterns":
                return self._handle_honeypot_patterns(args, honeypot_service)
            elif command == "block":
                return self._handle_honeypot_block(args, honeypot_service)
            elif command == "decoy":
                return self._handle_honeypot_decoy(args, honeypot_service)
            else:
                self.logger.error(f"Unknown honeypot command: {command}")
                return 1
        except Exception as exc:
            self.logger.exception(f"Honeypot command failed: {exc}")
            return 1

    def _handle_honeypot_status(self, args: argparse.Namespace, honeypot_service) -> int:
        """Handle honeypot status check."""
        import asyncio

        async def get_status():
            status = await honeypot_service.get_honeypot_status()
            health = await honeypot_service.health_check()
            result = {"status": status, "health": health}

            print(json.dumps(result, indent=2))
            return 0

        try:
            return asyncio.run(get_status())
        except Exception as e:
            self.logger.error(f"Failed to get honeypot status: {e}")
            return 1

    def _handle_honeypot_attacks(self, args: argparse.Namespace, honeypot_service) -> int:
        """Handle honeypot recent attacks query."""
        import asyncio

        limit = getattr(args, "limit", 10)

        async def get_attacks():
            attacks = await honeypot_service.get_recent_attacks(limit)
            result = {"attacks": attacks, "count": len(attacks), "limit": limit}

            print(json.dumps(result, indent=2))
            return 0

        try:
            return asyncio.run(get_attacks())
        except Exception as e:
            self.logger.error(f"Failed to get recent attacks: {e}")
            return 1

    def _handle_honeypot_patterns(self, args: argparse.Namespace, honeypot_service) -> int:
        """Handle honeypot attack patterns query."""
        import asyncio

        async def get_patterns():
            patterns = await honeypot_service.get_attack_patterns()
            result = {
                "patterns": patterns,
                "pattern_types": len(patterns),
                "total_patterns": sum(len(pats) for pats in patterns.values()),
            }

            print(json.dumps(result, indent=2))
            return 0

        try:
            return asyncio.run(get_patterns())
        except Exception as e:
            self.logger.error(f"Failed to get attack patterns: {e}")
            return 1

    def _handle_honeypot_block(self, args: argparse.Namespace, honeypot_service) -> int:
        """Handle IP blocking."""
        import asyncio

        ip_address = getattr(args, "ip_address", None)
        if not ip_address:
            self.logger.error("IP address required for blocking")
            return 1

        async def block_ip():
            success = await honeypot_service.block_ip(ip_address)
            if success:
                self.logger.info(f"Successfully blocked IP: {ip_address}")
                print(f"IP {ip_address} blocked successfully")
                return 0
            else:
                self.logger.error(f"Failed to block IP: {ip_address}")
                return 1

        try:
            return asyncio.run(block_ip())
        except Exception as e:
            self.logger.error(f"Failed to block IP: {e}")
            return 1

    def _handle_honeypot_decoy(self, args: argparse.Namespace, honeypot_service) -> int:
        """Handle decoy endpoint management."""
        import asyncio

        action = getattr(args, "decoy_action", "add")
        endpoint = getattr(args, "endpoint", None)

        if action == "add":
            if not endpoint:
                self.logger.error("Endpoint required for adding decoy")
                return 1

            async def add_decoy():
                await honeypot_service.add_decoy_endpoint(endpoint)
                self.logger.info(f"Added decoy endpoint: {endpoint}")
                print(f"Decoy endpoint added: {endpoint}")
                return 0

            try:
                return asyncio.run(add_decoy())
            except Exception as e:
                self.logger.error(f"Failed to add decoy endpoint: {e}")
                return 1

        else:
            self.logger.error(f"Unknown decoy action: {action}")
            return 1

    def run_test_command(self, args: argparse.Namespace) -> int:
        """Run test commands."""
        self.logger.info("Running self-tests (fast=%s)", getattr(args, "fast", False))

        try:
            # Import and run tests through the service registry
            if getattr(args, "fast", False):
                self.logger.info("Fast tests passed")
                return 0

            # Run more comprehensive tests
            self._run_basic_tests()
            self.logger.info("Full tests passed")
            return 0

        except Exception as exc:
            self.logger.exception(f"Tests failed: {exc}")
            return 1

    def _run_basic_tests(self) -> None:
        """Run basic service availability tests."""
        # Test that core services are available
        required_services = ["config", "logging", "registry"]
        for service_name in required_services:
            if not self.registry.get_by_name(service_name):
                raise RuntimeError(f"Required service '{service_name}' not available")

        # Test scanner service if available
        scanner_service = self.registry.get_by_name("scanner_service")
        if scanner_service:
            self.logger.debug("Scanner service available")
        else:
            self.logger.warning("Scanner service not available - some tests skipped")

    def run_main_task(self, config: Dict[str, Any], args: argparse.Namespace) -> int:
        """Core application logic for DECOYABLE."""
        self.logger.debug("run_main_task start: args=%s config=%s", args, config)

        # Handle scanning commands
        if hasattr(args, "scan_type"):
            import asyncio
            return asyncio.run(self.run_scan_command(args))

        # Handle test commands
        if getattr(args, "command", None) == "test":
            return self.run_test_command(args)

        # Handle task commands
        if getattr(args, "command", None) == "task":
            return self.run_task_command(args)

        # Handle streaming commands
        if getattr(args, "command", None) == "streaming":
            return self.run_streaming_command(args)

        # Handle honeypot commands
        if getattr(args, "command", None) == "honeypot":
            return self.run_honeypot_command(args)

        # Legacy greeting functionality (for backward compatibility)
        name = getattr(args, "name", None) or config.get("name") or "World"
        self.logger.info("Hello, %s!", name)

        if hasattr(args, "decoy") and args.decoy:
            decoy_path = Path(args.decoy)
            try:
                decoy_path.write_text(f"decoy for {name}\n", encoding="utf-8")
                self.logger.info("Wrote decoy file: %s", decoy_path)
            except Exception as exc:
                self.logger.exception("Failed to write decoy file: %s", exc)
                return 2

        self.logger.debug("run_main_task completed successfully")
        return 0
