#!/usr/bin/env python3
"""
Generate self-signed SSL certificates for DECOYABLE development.
Run this script to create certificates for HTTPS testing.
"""
import ipaddress
import os
from datetime import datetime, timedelta, timezone

from cryptography import x509
from cryptography.hazmat.primitives import hashes, serialization
from cryptography.hazmat.primitives.asymmetric import rsa
from cryptography.x509.oid import NameOID


def generate_ssl_certificates(cert_dir: str = "ssl"):
    """Generate self-signed SSL certificates for development."""
    # Create certificate directory if it doesn't exist
    os.makedirs(cert_dir, exist_ok=True)
    # Generate private key
    private_key = rsa.generate_private_key(
        public_exponent=65537,
        key_size=2048,
    )
    # Create certificate
    subject = issuer = x509.Name(
        [
            x509.NameAttribute(NameOID.COUNTRY_NAME, "US"),
            x509.NameAttribute(NameOID.STATE_OR_PROVINCE_NAME, "CA"),
            x509.NameAttribute(NameOID.LOCALITY_NAME, "San Francisco"),
            x509.NameAttribute(NameOID.ORGANIZATION_NAME, "DECOYABLE"),
            x509.NameAttribute(NameOID.COMMON_NAME, "localhost"),
        ]
    )
    cert = (
        x509.CertificateBuilder()
        .subject_name(subject)
        .issuer_name(issuer)
        .public_key(private_key.public_key())
        .serial_number(x509.random_serial_number())
        .not_valid_before(datetime.now(timezone.utc))
        .not_valid_after(datetime.now(timezone.utc) + timedelta(days=365))
        .add_extension(
            x509.SubjectAlternativeName(
                [
                    x509.DNSName("localhost"),
                    x509.IPAddress(ipaddress.IPv4Address("127.0.0.1")),
                ]
            ),
            critical=False,
        )
        .sign(private_key, hashes.SHA256())
    )
    # Write private key
    with open(os.path.join(cert_dir, "key.pem"), "wb") as f:
        f.write(
            private_key.private_bytes(
                encoding=serialization.Encoding.PEM,
                format=serialization.PrivateFormat.PKCS8,
                encryption_algorithm=serialization.NoEncryption(),
            )
        )
    # Write certificate
    with open(os.path.join(cert_dir, "cert.pem"), "wb") as f:
        f.write(cert.public_bytes(serialization.Encoding.PEM))
    print(f"SSL certificates generated in {cert_dir}/")
    print("  - cert.pem: SSL certificate")
    print("  - key.pem: Private key")
    print()
    print("To run the API with SSL:")
    print("  python -m decoyable.api.app --ssl-cert ssl/cert.pem --ssl-key ssl/key.pem")


if __name__ == "__main__":
    generate_ssl_certificates()
