"""Type definitions for language models."""

from typing import Any, Dict, Generic, List, TypedDict, TypeVar

import torch
from typing_extensions import NamedTuple, NotRequired

from eva.language.data.messages import MessageSeries

TargetType = TypeVar("TargetType")
"""The target data type."""


class TextBatch(NamedTuple, Generic[TargetType]):
    """Text sample with target and metadata."""

    text: List[MessageSeries]
    """Text content."""

    target: TargetType | None
    """Target data."""

    metadata: Dict[str, Any] | None
    """Additional metadata."""


class PredictionBatch(NamedTuple, Generic[TargetType]):
    """Text sample with target and metadata."""

    prediction: TargetType
    """Prediction data."""

    target: TargetType
    """Target data."""

    text: List[MessageSeries] | None
    """Conversation messages that were used as input."""

    metadata: Dict[str, Any] | None
    """Additional metadata."""


class ModelOutput(TypedDict):
    """The output batch produced by the model forward pass."""

    generated_text: List[str]
    """The text generated by the model."""

    input_ids: NotRequired[torch.Tensor | None]
    """The token ids of the input text."""

    output_ids: NotRequired[torch.Tensor | None]
    """The token ids of the model output (usually containing both input and prediction)."""

    attention_mask: NotRequired[torch.Tensor | None]
    """The attention mask for the input tokens."""
