from convokit import Corpus, Conversation, Utterance, Transformer, Speaker

from typing import Callable, Optional, Union, Any, List, Iterator
from collections import namedtuple
import numpy as np
import matplotlib.pyplot as plt

from convokit.forecaster.forecasterModel import ForecasterModel
from convokit.forecaster.forecaster import Forecaster

try:
    from convokit.utterance_simulator.utteranceSimulatorModel import UtteranceSimulatorModel
except NotImplementedError as e:
    raise ImportError("Unsloth GPU requirement not met") from e
from convokit.utterance_simulator.utteranceSimulator import UtteranceSimulator
from .util import ContextTuple, DEFAULT_LABELER


class PivotalMomentMeasure(Transformer):
    """
    ConvoKit transformer to compute pivotal scores. The pivotal framework
    consists of two main components: (1) `simulator_model` to simulate potential
    next responses and (2) `forecaster_model` to predict the likelihood of the outcome
    based on these potential responses.

    :param simulator_model: UtteranceSimulatorModel to simulate next utterances
        in the conversation given a conversation context, ConvoKit provides
        implementation for models supported by Unsloth via unslothUtteranceSimulatorModel
    :param forecaster_model: ForecasterModel to forecast the likelihood of the
        conversation's outcome given a conversation context
    :param piv_attribute_name: Name of metadata field to save pivotal scores
    :param simulated_reply_attribute_name: Name of metadata field to save
        simulated replies generated by the simulator_model
    :param simulated_reply_forecast_attribute_name: Name of metadata field to save
        the forecasts of the contexts with simulated replies
    :param simulated_reply_forecast_prob_attribute_name: Name of metadata field
        to save the forecast probabilities of the contexts with simulated replies
    :param forecast_attribute_name: Name of metadata field to save the forecasts
        of the contexts
    :param forecast_attribute_name: Name of metadata field to save the forecast
        probabilties of the contexts
    :param labeler: Name of metadata field to indicate the outcome or label
        of the conversation
    """

    def __init__(
        self,
        simulator_model: UtteranceSimulatorModel,
        forecaster_model: ForecasterModel,
        piv_attribute_name: str = "PIV",
        simulated_reply_attribute_name: str = "sim_replies",
        simulated_reply_forecast_attribute_name: str = "sim_replies_forecasts",
        simulated_reply_forecast_prob_attribute_name: str = "sim_replies_forecast_probs",
        forecast_attribute_name: str = "forecast",
        forecast_prob_attribute_name: str = "forecast_prob",
        labeler: str = DEFAULT_LABELER,
    ):
        self.simulator_model = simulator_model
        self.simulator = UtteranceSimulator(
            simulator_model=self.simulator_model,
            simulated_reply_attribute_name=simulated_reply_attribute_name,
        )

        self.forecaster_model = forecaster_model
        self.forecaster = Forecaster(
            forecaster_model=self.forecaster_model,
            labeler=labeler,
            forecast_attribute_name=forecast_attribute_name,
            forecast_prob_attribute_name=forecast_prob_attribute_name,
        )

        self.piv_attribute_name = piv_attribute_name
        self.simulated_reply_attribute_name = simulated_reply_attribute_name
        self.simulated_reply_forecast_attribute_name = simulated_reply_forecast_attribute_name
        self.simulated_reply_forecast_prob_attribute_name = (
            simulated_reply_forecast_prob_attribute_name
        )

        self.forecast_attribute_name = forecast_attribute_name
        self.forecast_prob_attribute_name = forecast_prob_attribute_name

        self.labeler = labeler

    def fit_forecaster(
        self,
        corpus: Corpus,
        train_context_selector: Callable[[ContextTuple], bool] = lambda context: True,
        val_context_selector: Optional[Callable[[ContextTuple], bool]] = None,
        test_context_selector: Callable[[ContextTuple], bool] = lambda context: True,
    ):
        """
        Wrapper method for fitting and transforming the underlying forecaster model.

        :param corpus: Corpus containing the data to train and test on
        :param train_context_selector: Function to select context tuples for training
        :param val_context_selector: Function to select context tuples for validation
        :param test_context_selector: Function to select context tuples for testing
        """
        self.forecaster.fit(
            corpus=corpus,
            context_selector=train_context_selector,
            val_context_selector=val_context_selector,
        )
        corpus = self.forecaster.transform(
            corpus=corpus,
            context_selector=test_context_selector,
        )

    def fit_simulator(
        self,
        corpus: Corpus,
        train_context_selector: Callable[[ContextTuple], bool] = lambda context: True,
        val_context_selector: Optional[Callable[[ContextTuple], bool]] = None,
    ):
        """
        Wrapper method for fitting the underlying utterance simulator model.

        :param corpus: Corpus containing the data to train and validate on
        :param train_context_selector: Function to select context tuples for training
        :param val_context_selector: Function to select context tuples for validation
        """
        self.simulator.fit(
            corpus=corpus,
            context_selector=train_context_selector,
            val_context_selector=val_context_selector,
        )

    def fit(
        self,
        corpus: Corpus,
        forecaster_train_context_selector: Callable[[ContextTuple], bool] = lambda context: True,
        forecaster_val_context_selector: Optional[Callable[[ContextTuple], bool]] = None,
        forecaster_test_context_selector: Callable[[ContextTuple], bool] = lambda context: True,
        simulator_train_context_selector: Callable[[ContextTuple], bool] = None,
        simulator_val_context_selector: Optional[Callable[[ContextTuple], bool]] = None,
    ):
        """
        Fit the PivotalMomentMeasure transformer by fitting the underlying Forecaster
        and UtteranceSimulator components using the selected contexts for each
        component.

        :param corpus: Corpus containing the data
        :param forecaster_train_context_selector: Function to select context
            tuples for forecaster training
        :param forecaster_val_context_selector: Function to select context
            tuples for forecaster validation
        :param forecaster_test_context_selector: Function to select context
            tuples for forecaster testing
        :param simulator_train_context_selector: Function to select context
            tuples for simulator training
        :param simulator_val_context_selector: Function to select context
            tuples for simulator validation

        :return: fitted PivotalMomentMeasure Transformer
        """
        self.fit_forecaster(
            corpus=corpus,
            train_context_selector=forecaster_train_context_selector,
            val_context_selector=forecaster_val_context_selector,
            test_context_selector=forecaster_test_context_selector,
        )

        if simulator_train_context_selector is not None:
            self.simulator = self.simulator.fit(
                corpus=corpus,
                train_selector=simulator_train_context_selector,
                val_selector=simulator_val_context_selector,
            )

    def transform(
        self,
        corpus: Corpus,
        context_selector: Callable[[ContextTuple], bool] = lambda convo: True,
        annotate_high_low_scores: bool = False,
    ):
        """
        Apply the PivotalMomentMeasure transformer to the selected contexts
        and annotates the corpus with the pivotal scores.

        :param corpus: Corpus containing the data
        :param context_selector: Function to select context tuples to run on
        :param annotate_high_low_scores: Whether to annotate utterances with
            high and low pivotal scores based on top/bottom percentile

        :return: annotated Corpus
        """
        self.simulator.transform(
            corpus=corpus,
            context_selector=context_selector,
        )

        contexts = self._create_context_iterator_simulations(
            corpus=corpus,
            context_selector=context_selector,
        )

        corpus = self._transform_simulations_forecasts(
            corpus=corpus,
            contexts=contexts,
        )

        if annotate_high_low_scores:
            self._annotate_high_low_pivotal_scores()

        return corpus

    def fit_transform(
        self,
        corpus: Corpus,
        forecaster_train_context_selector: Callable[[ContextTuple], bool] = lambda context: True,
        forecaster_val_context_selector: Optional[Callable[[ContextTuple], bool]] = None,
        forecaster_test_context_selector: Callable[[ContextTuple], bool] = lambda context: True,
        simulator_train_context_selector: Callable[[ContextTuple], bool] = lambda context: True,
        simulator_val_context_selector: Optional[Callable[[ContextTuple], bool]] = None,
        transform_context_selector: Callable[[ContextTuple], bool] = lambda context: True,
    ):
        """
        Fit and transform the PivotalMomentMeasure transformer.

        :param corpus: Corpus containing the data
        :param forecaster_train_context_selector: Function to select context
            tuples for forecaster training
        :param forecaster_val_context_selector: Function to select context
            tuples for forecaster validation
        :param forecaster_test_context_selector: Function to select context
            tuples for forecaster testing
        :param simulator_train_context_selector: Function to select context
            tuples for simulator training
        :param simulator_val_context_selector: Function to select context
            tuples for simulator validation
        :param transform_context_selector: Function to select context tuples
            to run over

        :return: annotated Corpus
        """
        self.fit(
            corpus=corpus,
            forecaster_train_context_selector=forecaster_train_context_selector,
            forecaster_val_context_selector=forecaster_val_context_selector,
            forecaster_test_context_selector=forecaster_test_context_selector,
            simulator_train_context_selector=simulator_train_context_selector,
            simulator_val_context_selector=simulator_val_context_selector,
        )
        corpus = self.transform(
            corpus=corpus,
            context_selector=transform_context_selector,
        )
        return corpus

    def _transform_simulations_forecasts(
        self,
        corpus: Corpus,
        contexts: Iterator[ContextTuple],
    ):
        """
        Helper function to obtain the forecasts of the conversational contexts with
        and without the simulated responses. Annotates the corpus with the
        forecasts, forecast probabilities, and pivotal scores.
        """
        forecast_df = self.forecaster_model.transform(
            contexts=contexts,
            forecast_attribute_name=self.forecast_attribute_name,
            forecast_prob_attribute_name=self.forecast_prob_attribute_name,
        )

        for utt in corpus.iter_utterances():
            index = utt.id
            if index in forecast_df.index:
                utt.add_meta(
                    self.forecast_attribute_name,
                    forecast_df.loc[index][self.forecast_attribute_name],
                )
                utt.add_meta(
                    self.forecast_prob_attribute_name,
                    forecast_df.loc[index][self.forecast_prob_attribute_name],
                )

                simulated_forecasts = []
                simulated_forecast_probs = []
                num_simulations = self.simulator_model.get_num_simulations()
                for i in range(num_simulations):
                    index = f"{utt.id}_{i}"
                    f = forecast_df.loc[index][self.forecast_attribute_name]
                    simulated_forecasts.append(f)
                    f_prob = forecast_df.loc[index][self.forecast_prob_attribute_name]
                    simulated_forecast_probs.append(f_prob)

                utt.add_meta(
                    self.simulated_reply_forecast_attribute_name,
                    simulated_forecasts,
                )
                utt.add_meta(
                    self.simulated_reply_forecast_prob_attribute_name, simulated_forecast_probs
                )
                utt.add_meta(self.piv_attribute_name, np.var(simulated_forecast_probs))
            else:
                utt.add_meta(self.forecast_attribute_name, None)
                utt.add_meta(self.forecast_prob_attribute_name, None)
                utt.add_meta(self.simulated_reply_forecast_attribute_name, None)
                utt.add_meta(self.simulated_reply_forecast_prob_attribute_name, None)
                utt.add_meta(self.piv_attribute_name, None)

    def _create_context_iterator_simulations(
        self,
        corpus: Corpus,
        context_selector: Callable[[ContextTuple], bool],
    ):
        """
        Helper function that generates an iterator over conversational contexts
        with their corresponding simulated responses, satisfying the provided context
        selector, across the entire corpus.
        """
        sim_corpus = Corpus(utterances=[])
        for convo in corpus.iter_conversations():
            chronological_utts = convo.get_chronological_utterance_list()
            for i in range(len(chronological_utts)):
                current_utt = chronological_utts[i]
                context = chronological_utts[: (i + 1)]
                current_convo = current_utt.get_conversation()

                context_tuple = ContextTuple(context, current_utt, None, convo.id)
                if not context_selector(context_tuple):
                    continue
                yield context_tuple

                simulated_replies = current_utt.meta[self.simulated_reply_attribute_name]
                for simulated_idx, simulated_reply in enumerate(simulated_replies):
                    simulated_utt = Utterance(
                        owner=sim_corpus,
                        id=f"{current_utt.id}_{simulated_idx}",
                        speaker=Speaker(),
                        text=simulated_reply,
                    )
                    simulated_convo = Conversation(
                        owner=sim_corpus,
                        id=f"{convo.id}_{i}_{simulated_idx}",
                        utterances=[simulated_utt],
                        meta={self.labeler: current_convo.retrieve_meta(self.labeler)},
                    )
                    simulated_utt.conversation_id = simulated_convo.id
                    sim_corpus.add_utterances(utterances=[simulated_utt])

                    context_tuple = ContextTuple(
                        context + [simulated_utt], simulated_utt, None, convo.id
                    )
                    yield context_tuple

    def _annotate_high_low_pivotal_scores(
        self,
        percentile: float = 10,
    ):
        """
        Annotates utterances with high and low pivotal scores based on
        top/bottom percentile.
        """
        utts = [
            utt
            for utt in corpus.iter_utterances()
            if self.piv_attribute_name in utt.meta and utt.meta[self.piv_attribute_name] is not None
        ]
        pivotal_scores = [utt.meta[self.piv_attribute_name] for utt in utts]

        low_threshold = np.percentile(pivotal_scores, percentile)
        high_threshold = np.percentile(pivotal_scores, 100 - percentile)

        for utt in utts:
            if utt.meta[self.piv_attribute_name] >= high_threshold:
                utt.add_meta("high_pivotal", True)
            elif utt.meta[self.piv_attribute_name] <= low_threshold:
                utt.add_meta("low_pivotal", True)

    def summarize(
        self,
        corpus: Corpus,
    ):
        """
        Summarizes PivotalMomentMeasure transformer with distribution of pivotal
        scores.

        :param corpus: Corpus to analyze
        """
        utts = [
            utt
            for utt in corpus.iter_utterances()
            if self.piv_attribute_name in utt.meta and utt.meta[self.piv_attribute_name] is not None
        ]
        pivotal_scores = [utt.meta[self.piv_attribute_name] for utt in utts]

        plt.hist(pivotal_scores)
        plt.title("Distribution of Pivotal Scores")
        plt.xlabel("Pivotal Score")
        plt.ylabel("Frequency")
        plt.grid(True)
        plt.show()

        return self
