# AgentSystems SDK & CLI

[![GitHub stars](https://img.shields.io/github/stars/agentsystems/agentsystems?style=flat-square&logo=github)](https://github.com/agentsystems/agentsystems/stargazers)
[![CI](https://github.com/agentsystems/agentsystems-sdk/actions/workflows/ci.yml/badge.svg?branch=main)](https://github.com/agentsystems/agentsystems-sdk/actions/workflows/ci.yml) [![codecov](https://codecov.io/gh/agentsystems/agentsystems-sdk/branch/main/graph/badge.svg?token=25Mahs4a96)](https://codecov.io/gh/agentsystems/agentsystems-sdk)

> [!NOTE]
> **Public Beta** - Part of the AgentSystems platform. Official public launch September 15, 2025.
> ⭐ [**Star the main repository**](https://github.com/agentsystems/agentsystems) to show your support!

> This is the **CLI and deployment tool** for AgentSystems. See the [main repository](https://github.com/agentsystems/agentsystems) for platform overview and documentation.

The AgentSystems **SDK** is a single-install Python package that provides several components:

* `agentsystems` — a polished command-line interface for bootstrapping and operating an AgentSystems deployment.
* A small helper library so you can embed AgentSystems clients in your own code.

The CLI is designed to work **both interactively** (laptops) and **non-interactively** (CI, cloud VMs).


## Quick install

### Automatic installation (Recommended)

```bash
# Linux & macOS - installs Docker, pipx, and the SDK
curl -fsSL https://raw.githubusercontent.com/agentsystems/agentsystems/main/install.sh | sh
```

### Manual installation

```bash
# Install pipx if you don't have it
python3 -m pip install --upgrade pipx
pipx ensurepath

# Install the SDK
pipx install agentsystems-sdk

# Verify installation
agentsystems --version
```



### Install from source (contributors)

```bash
# in the repository root
pipx uninstall agentsystems-sdk  # if previously installed
pipx install --editable .        # live-reloads on file changes
```

---
## Continuous Integration (GitHub Actions)

Every push and pull request runs `ci.yml` which now goes beyond linting:

1. Installs dev dependencies & runs pre-commit hooks (ruff, black, shellcheck, hadolint).
2. Builds the wheel (`python -m build`).
3. Installs the built wheel into a fresh venv.
4. Runs smoke tests (`agentsystems --version`, `agentsystems info`).

A failing build or test blocks the merge, ensuring every released version installs cleanly.

---

## CLI commands

All commands are available through `agentsystems` (or the shorter alias `agntsys`).

| Command | Description |
|---------|-------------|
| `agentsystems init [TARGET_DIR]` | Interactive bootstrap: prompts for Langfuse organization & admin details, generates keys, creates `.env` automatically, then copies the built-in deployment template and pulls the required Docker images into `TARGET_DIR`. No Docker authentication required - uses public images. |
| `agentsystems up [PROJECT_DIR]` | Start the platform **plus Langfuse tracing stack** (`docker compose up`). **Waits for the gateway _and all agent containers_ to become _healthy_ by default (spinner)**. Pass `--no-wait` to skip readiness wait or `--no-langfuse` to disable tracing. Uses the `.env` generated by `init` (or pass `--env-file PATH`). |
| `agentsystems down [PROJECT_DIR]` | Stop containers. Volumes are preserved by default; add `--delete-volumes` (or `--delete-all`) to wipe data. |
| `agentsystems logs [PROJECT_DIR]` | Stream or view recent logs (`docker compose logs`). |
| `agentsystems status [PROJECT_DIR]` | List running containers and state (`docker compose ps`). |
| `agentsystems restart [PROJECT_DIR]` | Quick bounce (`down` → `up`). **Waits for readiness by default**. Pass `--no-wait` to skip. **Requires `.env`**. |
| `agentsystems version` | Show the installed SDK version. |
| `agentsystems versions` | Show version information for all AgentSystems components. |
| `agentsystems clean` | Clean up deployment artifacts and containers. |
| `agentsystems artifacts-path THREAD_ID [REL_PATH]` | Resolve a path inside the shared artifacts volume using thread-centric structure. |
| `agentsystems run AGENT PAYLOAD` | Invoke an agent with JSON payload and optional file uploads, stream progress, and return results. |
| `agentsystems update [PROJECT_DIR]` | Update core platform images (agent-control-plane, agentsystems-ui) to latest versions. Faster than re-running `up` when you only need to update platform components. |

### Index Commands

| Command | Description |
|---------|-------------|
| `agentsystems index login` | Login to AgentSystems Index - opens browser for authentication and stores API key locally. |
| `agentsystems index logout` | Logout from AgentSystems Index - removes stored API key from local machine. |
| `agentsystems index list` | List all your published agents on the index. |
| `agentsystems index publish [PATH]` | Publish agent to the index. Use `--listed` for public agents. Agents are unlisted (private) by default. |
| `agentsystems index delete DEVELOPER/AGENT` | Permanently remove an agent from the index. |
| `agentsystems index allow-listed --enable\|--disable` | Enable or disable listed (public) agents. When disabling, use `--preserve` to keep existing agents or `--cascade` to unlist all. |

### `up` options

```
--detach / --foreground   Run containers in background (default) or stream logs
--fresh                   docker compose down -v before starting
--env-file PATH           Pass a custom .env file to Compose
--wait / --no-wait        Wait for gateway readiness (default: --wait)
--no-langfuse             Skip the Langfuse tracing stack (core services only)
--agents MODE             Agent startup mode: none, create, or all (default: create)
```

Run `agentsystems up --help` for the authoritative list.

---
### Tracing & observability (Langfuse)

By default the CLI starts the [Langfuse](https://langfuse.com/) tracing stack alongside the core services and exposes its UI at <http://localhost:3000>. You can explore request traces and performance metrics there while developing.

If you prefer to run only the core platform (for example on a small CI runner) pass `--no-langfuse` to any stack command (`up`, `down`, `restart`, `logs`, `status`).


### Example: start and watch logs

```bash
agentsystems up --foreground    # run inside the deployment dir
```

### Example: fresh restart in CI

```bash
agentsystems up /opt/agent-platform-deployments --fresh --detach
```

---

## File Uploads & Artifacts

The AgentSystems platform supports file uploads to agents and provides tools for managing artifacts.

### Uploading Files to Agents

Use the `run` command to invoke agents with file uploads:

```bash
# Upload single file with JSON payload
agentsystems run agent-name '{"format": "csv"}' --input-file data.csv

# Upload multiple files
agentsystems run agent-name '{"sync": true}' \
  --input-file input1.txt \
  --input-file input2.txt

# Use file path for JSON payload
echo '{"date": "July 4"}' > payload.json
agentsystems run agent-name payload.json --input-file date.txt
```

### Artifacts Management

The platform uses a thread-centric artifacts structure where each request gets a unique directory:

```
/artifacts/
├── {thread-id}/
│   ├── in/          # Files uploaded by client
│   └── out/         # Files created by agent
```

#### Using artifacts-path Command

Get paths for reading/writing artifacts:

```bash
# Get path to thread's output directory
agentsystems artifacts-path abc123-def456

# Get path to specific output file
agentsystems artifacts-path abc123-def456 result.json

# Get path to input directory
agentsystems artifacts-path abc123-def456 --input

# Get path to specific input file
agentsystems artifacts-path abc123-def456 data.csv --input
```

#### Manual Artifact Access

You can also access artifacts directly through any container with the volume mounted:

```bash
# List all active threads
docker exec local-gateway-1 ls -la /artifacts/

# Read agent output files
docker exec local-gateway-1 cat /artifacts/{thread-id}/out/result.txt

# Check uploaded input files
docker exec local-gateway-1 ls -la /artifacts/{thread-id}/in/
```

### Volume Mounting for Agents

When deploying agents through `agentsystems-config.yml`, the CLI automatically:

1. **Mounts artifacts volume**: All agents get `/artifacts` mounted with proper permissions
2. **Creates thread directories**: Gateway creates `/artifacts/{thread-id}/{in,out}/` as needed
3. **Handles file uploads**: Gateway saves uploaded files to `/artifacts/{thread-id}/in/`
4. **Manages permissions**: Attempts to configure agents (UID 1001) for artifact access

### Agent Development

When building agents that handle file uploads:

```python
# In your agent's invoke() function
import pathlib
from fastapi import Request

def invoke(request: Request, req: InvokeRequest):
    thread_id = request.headers.get("X-Thread-Id", "")

    # Read uploaded files
    in_dir = pathlib.Path("/artifacts") / thread_id / "in"
    if (in_dir / "data.txt").exists():
        content = (in_dir / "data.txt").read_text()

    # Write output files
    out_dir = pathlib.Path("/artifacts") / thread_id / "out"
    out_dir.mkdir(parents=True, exist_ok=True)
    (out_dir / "result.json").write_text('{"status": "complete"}')
```

See the [agent-template](https://github.com/agentsystems/agent-template) for a complete example.

---

### `init` details

The init command:
1. Prompts for Langfuse organization and admin credentials (interactive mode)
2. Generates keys automatically
3. Creates `.env` file with all configuration
4. Pulls required Docker images from public registries (no authentication needed)

The core platform images are publicly available on GitHub Container Registry:
- `ghcr.io/agentsystems/agent-control-plane:latest`

### Example: interactive laptop

```bash
agentsystems init
# prompts for directory and Langfuse setup
```

### Example: scripted cloud VM / CI

```bash
agentsystems init /opt/agentsystems/engine
# Non-interactive mode - uses defaults for Langfuse
```

---
## Multi-registry agent marketplace

Agent images and credentials are now configured via a single **`agentsystems-config.yml`** file found in the deployment repo. Use the new top-level `registry_connections:` key (replacing the legacy `registries:`) to define one or more logins—multiple Docker Hub accounts, Harbor, ECR, etc.

```yaml
registry_connections:
  dockerhub_main:
    url: docker.io
    enabled: true
    auth:
      method: basic
      username_env: DOCKERHUB_USER
      password_env: DOCKERHUB_TOKEN

agents:
  - name: hello-world
    registry_connection: dockerhub_main
    repo: agentsystems/hello-world-agent
    tag: latest
```

When you run `agentsystems up` the CLI logs in to each **enabled** connection using the referenced env vars, pulls the images, starts the containers with your `.env`, and waits until Docker marks them `healthy` (see `HEALTHCHECK` below). The CLI uses an isolated Docker config directory per registry, so multiple connections targeting the same hostname (e.g. several Docker Hub organisations) work seamlessly.

---
## Environment variables & `.env`

> Note: `LANGFUSE_HOST`, `LANGFUSE_PUBLIC_KEY`, and `LANGFUSE_SECRET_KEY` are written *without* quotes so they are passed correctly into Docker containers. If you edit `.env` manually, ensure these three remain unquoted.

The `.env` file is generated automatically by `agentsystems init`. It contains both runtime vars and a temporary set of `LANGFUSE_INIT_*` variables used on the very first startup. On the first successful `agentsystems up` these init variables are commented out and moved to the bottom of the file so they don’t confuse future edits.
You can still keep them for reference or delete them entirely.

> Note: `LANGFUSE_HOST`, `LANGFUSE_PUBLIC_KEY`, and `LANGFUSE_SECRET_KEY` are written *without* quotes so they are passed correctly into Docker containers. If you edit `.env` manually, ensure these three remain unquoted.


---
## Updating / upgrading

```bash
pipx upgrade agentsystems-sdk           # upgrade from PyPI
# or, from source repo:
pipx reinstall --editable .
```

---
## Security notes

* The CLI is designed not to print secret values. GitHub PATs are masked and Docker login uses `--password-stdin`.
* Delete tokens / `.env` after the resources become public.

---
## Support & feedback

Open an issue or discussion in the private GitHub repository.
Contributions welcome—see [CONTRIBUTING.md](CONTRIBUTING.md).

## License

Licensed under the [Apache-2.0 license](./LICENSE).
