# 👁️ EyeTestUtils — Eye Test & Contact Lens Utilities

**EyeTestUtils** is a Python library that helps you validate, convert, and format eye test and contact lens data easily.  
It’s designed for developers building optical systems, clinics, or vision-related applications.

---

## 🚀 Features
- ✅ Validate eye test values (SPH, CYL, AXIS, PD, ADD)
- 🔄 Convert prescription data to contact lens parameters (Spheric / Toric)
- 🧮 Calculate and validate vertex distance
- 🧠 Normalize numeric formats (±00.00)
- ⚙️ Ready to integrate with Django, FastAPI, or desktop apps

---

## 📦 Installation

From PyPI:
```bash
pip install eyeTestUtils
```

From source:
```bash
pip install -e .
```

## 🧰 Basic Usage
### 1️⃣ Validate eye test values
```python
    from eyetools.validators import EyeTestValidator

    validator = EyeTestValidator()

    print(validator.is_multiple_of_quarter(1.25))  # ✅ True
    print(validator.is_multiple_of_quarter(1.33))  # ❌ False

    # Axis validation
    print(validator.check_axis(180))  # ✅ True
    print(validator.check_axis(-1))    # ❌ False
    print(validator.check_axis(190))    # ❌ False

    # Add validation
    print(validator.check_add(2.25))  # ✅ True
    print(validator.check_add(-2.25))  # ❌ False
```

### 2️⃣ Convert to contact lens values
```python
    from eyetools.validators import ContactLensValidator

    validator = ContactLensValidator()
    data = {
    "SPH": "3.00",
    "CY": "-1.25",
    "AX": "90",
    "BV": "12", # optional
    "ADD": "0.00" # optional
    }

    toric_lens = validator.convert_to_toric(data)
    print(toric_lens)
    # Expected output:
    # {
    #   "Exact SPH": "+03.11",
    #   "Exact CY": "-1.32",
    #   "SPH": "+03.00",
    #   "CY": "-1.25",
    #   "AX": "090",
    #   "BV": "12",
    #   "ADD": "0.00" 
    # }
    
    spheric_lens = validator.convert_to_spheric(data)
    print(spheric_lens)
    # Expected output:
    # {
    #   "Exact SPH": "+02.38",
    #   "SPH": "+02.50",
    #   "BV": "12",
    #   "ADD": "0.00" 
    # }
```

### Format powers
```python
    from eyetools.validators import EyeTestValidator

    validator = EyeTestValidator()

    print(validator.is_valida_eye_test_power(-1.25))   # "-01.25"
    print(validator.remove_sign("-2.50"))              # "2.50"
```

### Vertex distance check
```python
    from eyetools.validators import EyeTestValidator

    validator = EyeTestValidator()

    print(validator.Check_vertex_distance(12))  # ✅ True
    print(validator.Check_vertex_distance(20))  # ❌ False
```

## 🧪 Testing
Run all tests using pytest:
```bash
pytest -v
```



eyetools/
├── eyetools/
│   ├── __init__.py
│   └── validators.py
├── tests/
│   ├── test_eye_validator.py
│   └── test_contact_validator.py
├── LICENSE  
├── pyproject.toml
└── README.md




## 💡 Contributing

We welcome contributions! If you have suggestions for improving validation rules, finding bugs, or adding new features, please open an issue or submit a Pull Request.

**GitHub Repository:** [https://github.com/Elhussin/eye-test-utils-python.git](https://github.com/Elhussin/eye-test-utils-python.git)

## 👨‍💻 Author

Elhussini Taha
Optical systems & Web developer 💡

[GitHub](https://github.com/Elhussin)

[LinkedIn](https://linkedin.com/in/elhusseini2023)


## 💬 Support & Contact

If you have any questions, need technical support, or want to discuss features, feel free to reach out:

* **Email:** [hasin3112@gmail.com](mailto:hasin3112@gmail.com)
* **GitHub Issues:** [Open an Issue here](https://github.com/Elhussin/eye-test-utils-python/issues) (The preferred way to report bugs)

## 📜 License
This project is licensed under the **MIT License**.

