from collections import namedtuple
from enum import Enum

CONTINUATION_CODE = '88'


class RecordCode(Enum):
    file_header = '01'
    group_header = '02'
    account_identifier = '03'
    transaction_detail = '16'
    account_trailer = '49'
    continuation = '88'
    group_trailer = '98'
    file_trailer = '99'


class GroupStatus(Enum):
    update = '1'
    deletion = '2'
    correction = '3'
    test_only = '4'


class AsOfDateModifier(Enum):
    interim_previous_day = '1'
    final_previous_day = '2'
    interim_same_day = '3'
    final_same_day = '4'


class FundsType(Enum):
    unknown_availability = 'Z'
    immediate_availability = '0'
    one_day_availability = '1'
    two_day_availability = '2'
    distributed_availability_simple = 'S'
    value_dated = 'V'
    distributed_availability = 'D'


class TypeCodeTransaction(Enum):
    credit = 'credit'
    debit = 'debit'
    misc = 'misc'


class TypeCodeLevel(Enum):
    status = 'status'
    detail = 'detail'
    summary = 'summary'


TypeCode = namedtuple('TypeCode', ['code', 'transaction', 'level', 'description'])
TypeCodes = [
    TypeCode('010', None, TypeCodeLevel.status, 'Opening Ledger'),
    TypeCode('011', None, TypeCodeLevel.status, 'Average Opening Ledger MTD'),
    TypeCode('012', None, TypeCodeLevel.status, 'Average Opening Ledger YTD'),
    TypeCode('015', None, TypeCodeLevel.status, 'Closing Ledger'),
    TypeCode('020', None, TypeCodeLevel.status, 'Average Closing Ledger MTD'),
    TypeCode('021', None, TypeCodeLevel.status, 'Average Closing Ledger – Previous Month'),
    TypeCode('022', None, TypeCodeLevel.status, 'Aggregate Balance Adjustments'),
    TypeCode('024', None, TypeCodeLevel.status, 'Average Closing Ledger YTD – Previous Month'),
    TypeCode('025', None, TypeCodeLevel.status, 'Average Closing Ledger YTD'),
    TypeCode('030', None, TypeCodeLevel.status, 'Current Ledger'),
    TypeCode('037', None, TypeCodeLevel.status, 'ACH Net Position'),
    TypeCode('039', None, TypeCodeLevel.status, 'Opening Available + Total Same-Day ACH DTC Deposit'),
    TypeCode('040', None, TypeCodeLevel.status, 'Opening Available'),
    TypeCode('041', None, TypeCodeLevel.status, 'Average Opening Available MTD'),
    TypeCode('042', None, TypeCodeLevel.status, 'Average Opening Available YTD'),
    TypeCode('043', None, TypeCodeLevel.status, 'Average Available – Previous Month'),
    TypeCode('044', None, TypeCodeLevel.status, 'Disbursing Opening Available Balance'),
    TypeCode('045', None, TypeCodeLevel.status, 'Closing Available'),
    TypeCode('050', None, TypeCodeLevel.status, 'Average Closing Available MTD'),
    TypeCode('051', None, TypeCodeLevel.status, 'Average Closing Available – Last Month'),
    TypeCode('054', None, TypeCodeLevel.status, 'Average Closing Available YTD – Last Month'),
    TypeCode('055', None, TypeCodeLevel.status, 'Average Closing Available YTD'),
    TypeCode('056', None, TypeCodeLevel.status, 'Loan Balance'),
    TypeCode('057', None, TypeCodeLevel.status, 'Total Investment Position'),
    TypeCode('059', None, TypeCodeLevel.status, 'Current Available (CRS Supressed)'),
    TypeCode('060', None, TypeCodeLevel.status, 'Current Available'),
    TypeCode('061', None, TypeCodeLevel.status, 'Average Current Available MTD'),
    TypeCode('062', None, TypeCodeLevel.status, 'Average Current Available YTD'),
    TypeCode('063', None, TypeCodeLevel.status, 'Total Float'),
    TypeCode('065', None, TypeCodeLevel.status, 'Target Balance'),
    TypeCode('066', None, TypeCodeLevel.status, 'Adjusted Balance'),
    TypeCode('067', None, TypeCodeLevel.status, 'Adjusted Balance MTD'),
    TypeCode('068', None, TypeCodeLevel.status, 'Adjusted Balance YTD'),
    TypeCode('070', None, TypeCodeLevel.status, '0-Day Float'),
    TypeCode('072', None, TypeCodeLevel.status, '1-Day Float'),
    TypeCode('073', None, TypeCodeLevel.status, 'Float Adjustment'),
    TypeCode('074', None, TypeCodeLevel.status, '2 or More Days Float'),
    TypeCode('075', None, TypeCodeLevel.status, '3 or More Days Float'),
    TypeCode('076', None, TypeCodeLevel.status, 'Adjustment to Balances'),
    TypeCode('077', None, TypeCodeLevel.status, 'Average Adjustment to Balances MTD'),
    TypeCode('078', None, TypeCodeLevel.status, 'Average Adjustment to Balances YTD'),
    TypeCode('079', None, TypeCodeLevel.status, '4-Day Float'),
    TypeCode('080', None, TypeCodeLevel.status, '5-Day Float'),
    TypeCode('081', None, TypeCodeLevel.status, '6-Day Float'),
    TypeCode('082', None, TypeCodeLevel.status, 'Average 1-Day Float MTD'),
    TypeCode('083', None, TypeCodeLevel.status, 'Average 1-Day Float YTD'),
    TypeCode('084', None, TypeCodeLevel.status, 'Average 2-Day Float MTD'),
    TypeCode('085', None, TypeCodeLevel.status, 'Average 2-Day Float YTD'),
    TypeCode('086', None, TypeCodeLevel.status, 'Transfer Calculation'),
    TypeCode('087', None, TypeCodeLevel.status, 'MTD Ave Collected'),
    TypeCode('088', None, TypeCodeLevel.status, 'MTD Ave Neg Collected'),
    TypeCode('100', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Credits'),
    TypeCode('101', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Credit Amount MTD'),
    TypeCode('102', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Number of Credits'),
    TypeCode('105', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Credits Not Detailed'),
    TypeCode('106', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Deposits Subject to Float'),
    TypeCode('107', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Adjustment Credits YTD'),
    TypeCode('108', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Credit (Any Type)'),
    TypeCode('109', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Current Day Total Lockbox Deposits'),
    TypeCode('110', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Lockbox Deposits'),
    TypeCode('115', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Lockbox Deposit'),
    TypeCode('116', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Item in Lockbox Deposit'),
    TypeCode('118', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Lockbox Adjustment Credit'),
    TypeCode('120', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'EDI* Transaction Credit'),
    TypeCode('121', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'EDI Transaction Credit'),
    TypeCode('122', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'EDIBANX Credit Received'),
    TypeCode('123', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'EDIBANX Credit Return'),
    TypeCode('130', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Concentration Credits'),
    TypeCode('131', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total DTC Credits'),
    TypeCode('135', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'DTC Concentration Credit'),
    TypeCode('136', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Item in DTC Deposit'),
    TypeCode('140', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total ACH Credits'),
    TypeCode('142', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'ACH Credit Received'),
    TypeCode('143', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Item in ACH Deposit'),
    TypeCode('145', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'ACH Concentration Credit'),
    TypeCode('146', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Bank Card Deposits'),
    TypeCode('147', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Individual Bank Card Deposit'),
    TypeCode('150', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Preauthorized Payment Credits'),
    TypeCode('155', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Preauthorized Draft Credit'),
    TypeCode('156', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Item in PAC Deposit'),
    TypeCode('158', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Real Time Payment Credit'),
    TypeCode('159', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Real Time Payment Credits'),
    TypeCode('160', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total ACH Disbursing Funding Credits'),
    TypeCode('162', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Corporate Trade Payment Settlement'),
    TypeCode('163', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Corporate Trade Payment Credits'),
    TypeCode('164', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Corporate Trade Payment Credit'),
    TypeCode('165', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Preauthorized ACH Credit'),
    TypeCode('166', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'ACH Settlement'),
    TypeCode('167', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'ACH Settlement Credits'),
    TypeCode('168', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'ACH Return Item or Adjustment Settlement'),
    TypeCode('169', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Miscellaneous ACH Credit'),
    TypeCode('170', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Other Check Deposits'),
    TypeCode('171', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Individual Loan Deposit'),
    TypeCode('172', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Deposit Correction'),
    TypeCode('173', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Bank-Prepared Deposit'),
    TypeCode('174', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Other Deposit'),
    TypeCode('175', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Check Deposit Package'),
    TypeCode('176', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Re-presented Check Deposit'),
    TypeCode('178', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'List Post Credits'),
    TypeCode('180', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Loan Proceeds'),
    TypeCode('182', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Bank-Prepared Deposits'),
    TypeCode('184', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Draft Deposit'),
    TypeCode('185', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Miscellaneous Deposits'),
    TypeCode('186', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Cash Letter Credits'),
    TypeCode('187', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Cash Letter Credit'),
    TypeCode('188', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Cash Letter Adjustments'),
    TypeCode('189', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Cash Letter Adjustment'),
    TypeCode('190', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Incoming Money Transfers'),
    TypeCode('191', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Individual Incoming Internal Money Transfer'),
    TypeCode('195', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Incoming Money Transfer'),
    TypeCode('196', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Money Transfer Adjustment'),
    TypeCode('198', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Compensation'),
    TypeCode('200', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Automatic Transfer Credits'),
    TypeCode('201', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Individual Automatic Transfer Credit'),
    TypeCode('202', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Bond Operations Credit'),
    TypeCode('205', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Book Transfer Credits'),
    TypeCode('206', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Book Transfer Credit'),
    TypeCode('207', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total International Money Transfer Credits'),
    TypeCode('208', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Individual International Money Transfer Credit'),
    TypeCode('210', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total International Credits'),
    TypeCode('212', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Foreign Letter of Credit'),
    TypeCode('213', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Letter of Credit'),
    TypeCode('214', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Foreign Exchange of Credit'),
    TypeCode('215', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Letters of Credit'),
    TypeCode('216', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Foreign Remittance Credit'),
    TypeCode('218', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Foreign Collection Credit'),
    TypeCode('221', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Foreign Check Purchase'),
    TypeCode('222', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Foreign Checks Deposited'),
    TypeCode('224', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Commission'),
    TypeCode('226', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'International Money Market Trading'),
    TypeCode('227', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Standing Order'),
    TypeCode('229', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Miscellaneous International Credit'),
    TypeCode('230', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Security Credits'),
    TypeCode('231', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Collection Credits'),
    TypeCode('232', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Sale of Debt Security'),
    TypeCode('233', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Securities Sold'),
    TypeCode('234', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Sale of Equity Security'),
    TypeCode('235', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Matured Reverse Repurchase Order'),
    TypeCode('236', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Maturity of Debt Security'),
    TypeCode('237', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Individual Collection Credit'),
    TypeCode('238', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Collection of Dividends'),
    TypeCode('239', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Bankers’ Acceptance Credits'),
    TypeCode('240', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Coupon Collections – Banks'),
    TypeCode('241', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Bankers’ Acceptances'),
    TypeCode('242', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Collection of Interest Income'),
    TypeCode('243', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Matured Fed Funds Purchased'),
    TypeCode('244', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Interest/Matured Principal Payment'),
    TypeCode('245', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Monthly Dividends'),
    TypeCode('246', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Commercial Paper'),
    TypeCode('247', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Capital Change'),
    TypeCode('248', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Savings Bonds Sales Adjustment'),
    TypeCode('249', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Miscellaneous Security Credit'),
    TypeCode('250', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Checks Posted and Returned'),
    TypeCode('251', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Debit Reversals'),
    TypeCode('252', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Debit Reversal'),
    TypeCode('254', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Posting Error Correction Credit'),
    TypeCode('255', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Check Posted and Returned'),
    TypeCode('256', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total ACH Return Items'),
    TypeCode('257', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Individual ACH Return Item'),
    TypeCode('258', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'ACH Reversal Credit'),
    TypeCode('260', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Rejected Credits'),
    TypeCode('261', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Individual Rejected Credit'),
    TypeCode('263', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Overdraft'),
    TypeCode('266', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Return Item'),
    TypeCode('268', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Return Item Adjustment'),
    TypeCode('270', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total ZBA Credits'),
    TypeCode('271', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Net Zero-Balance Amount'),
    TypeCode('274', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Cumulative** ZBA or Disbursement Credits'),
    TypeCode('275', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'ZBA Credit'),
    TypeCode('276', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'ZBA Float Adjustment'),
    TypeCode('277', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'ZBA Credit Transfer'),
    TypeCode('278', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'ZBA Credit Adjustment'),
    TypeCode('280', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Controlled Disbursing Credits'),
    TypeCode('281', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Individual Controlled Disbursing Credit'),
    TypeCode('285', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total DTC Disbursing Credits'),
    TypeCode('286', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Individual DTC Disbursing Credit'),
    TypeCode('294', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total ATM Credits'),
    TypeCode('295', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'ATM Credit'),
    TypeCode('301', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Commercial Deposit'),
    TypeCode('302', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Correspondent Bank Deposit'),
    TypeCode('303', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Wire Transfers In – FF'),
    TypeCode('304', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Wire Transfers In – CHF'),
    TypeCode('305', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Fed Funds Sold'),
    TypeCode('306', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Fed Funds Sold'),
    TypeCode('307', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Trust Credits'),
    TypeCode('308', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Trust Credit'),
    TypeCode('309', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Value - Dated Funds'),
    TypeCode('310', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Commercial Deposits'),
    TypeCode('315', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total International Credits – FF'),
    TypeCode('316', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total International Credits – CHF'),
    TypeCode('318', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Foreign Check Purchased'),
    TypeCode('319', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Late Deposit'),
    TypeCode('320', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Securities Sold – FF'),
    TypeCode('321', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Securities Sold – CHF'),
    TypeCode('324', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Securities Matured – FF'),
    TypeCode('325', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Securities Matured – CHF'),
    TypeCode('326', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Securities Interest'),
    TypeCode('327', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Securities Matured'),
    TypeCode('328', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Securities Interest – FF'),
    TypeCode('329', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Securities Interest – CHF'),
    TypeCode('330', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Escrow Credits'),
    TypeCode('331', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Individual Escrow Credit'),
    TypeCode('332', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Miscellaneous Securities Credits – FF'),
    TypeCode('336', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Miscellaneous Securities Credits – CHF'),
    TypeCode('338', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Securities Sold'),
    TypeCode('340', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Broker Deposits'),
    TypeCode('341', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Broker Deposits – FF'),
    TypeCode('342', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Broker Deposit'),
    TypeCode('343', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Broker Deposits – CHF'),
    TypeCode('344', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Individual Back Value Credit'),
    TypeCode('345', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Item in Brokers Deposit'),
    TypeCode('346', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Sweep Interest Income'),
    TypeCode('347', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Sweep Principal Sell'),
    TypeCode('348', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Futures Credit'),
    TypeCode('349', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Principal Payments Credit'),
    TypeCode('350', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Investment Sold'),
    TypeCode('351', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Individual Investment Sold'),
    TypeCode('352', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Cash Center Credits'),
    TypeCode('353', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Cash Center Credit'),
    TypeCode('354', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Interest Credit'),
    TypeCode('355', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Investment Interest'),
    TypeCode('356', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Credit Adjustment'),
    TypeCode('357', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Credit Adjustment'),
    TypeCode('358', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'YTD Adjustment Credit'),
    TypeCode('359', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Interest Adjustment Credit'),
    TypeCode('360', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Credits Less Wire Transfer and '
                                                                       'Returned Checks'),
    TypeCode('361', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Grand Total Credits Less Grand Total Debits'),
    TypeCode('362', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Correspondent Collection'),
    TypeCode('363', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Correspondent Collection Adjustment'),
    TypeCode('364', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Loan Participation'),
    TypeCode('366', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Currency and Coin Deposited'),
    TypeCode('367', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Food Stamp Letter'),
    TypeCode('368', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Food Stamp Adjustment'),
    TypeCode('369', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Clearing Settlement Credit'),
    TypeCode('370', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Back Value Credits'),
    TypeCode('372', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Back Value Adjustment'),
    TypeCode('373', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Customer Payroll'),
    TypeCode('374', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'FRB Statement Recap'),
    TypeCode('376', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Savings Bond Letter or Adjustment'),
    TypeCode('377', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Treasury Tax and Loan Credit'),
    TypeCode('378', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Transfer of Treasury Credit'),
    TypeCode('379', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'FRB Government Checks Cash Letter Credit'),
    TypeCode('381', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'FRB Government Check Adjustment'),
    TypeCode('382', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'FRB Postal Money Order Credit'),
    TypeCode('383', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'FRB Postal Money Order Adjustment'),
    TypeCode('384', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'FRB Cash Letter Auto Charge Credit'),
    TypeCode('385', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Universal Credits'),
    TypeCode('386', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'FRB Cash Letter Auto Charge Adjustment'),
    TypeCode('387', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'FRB Fine-Sort Cash Letter Credit'),
    TypeCode('388', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'FRB Fine-Sort Adjustment'),
    TypeCode('389', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Freight Payment Credits'),
    TypeCode('390', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Miscellaneous Credits'),
    TypeCode('391', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Universal Credit'),
    TypeCode('392', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Freight Payment Credit'),
    TypeCode('393', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Itemized Credit Over $10,000'),
    TypeCode('394', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Cumulative** Credits'),
    TypeCode('395', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Check Reversal'),
    TypeCode('397', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Float Adjustment'),
    TypeCode('398', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Miscellaneous Fee Refund'),
    TypeCode('399', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Miscellaneous Credit'),
    TypeCode('400', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Debits'),
    TypeCode('401', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Debit Amount MTD'),
    TypeCode('402', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Number of Debits'),
    TypeCode('403', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Today’s Total Debits'),
    TypeCode('405', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Debit Less Wire Transfers and '
                                                                      'ChargeBacks'),
    TypeCode('406', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Debits not Detailed'),
    TypeCode('408', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Float Adjustment'),
    TypeCode('409', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Debit (Any Type)'),
    TypeCode('410', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total YTD Adjustment'),
    TypeCode('412', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Debits (Excluding Returned Items)'),
    TypeCode('415', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Lockbox Debit'),
    TypeCode('416', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Lockbox Debits'),
    TypeCode('420', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'EDI Transaction Debits'),
    TypeCode('421', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'EDI Transaction Debit'),
    TypeCode('422', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'EDIBANX Settlement Debit'),
    TypeCode('423', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'EDIBANX Return Item Debit'),
    TypeCode('430', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Payable–Through Drafts'),
    TypeCode('435', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Payable–Through Draft'),
    TypeCode('445', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'ACH Concentration Debit'),
    TypeCode('446', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total ACH Disbursement Funding Debits'),
    TypeCode('447', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'ACH Disbursement Funding Debit'),
    TypeCode('450', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total ACH Debits'),
    TypeCode('451', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'ACH Debit Received'),
    TypeCode('452', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Item in ACH Disbursement or Debit'),
    TypeCode('455', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Preauthorized ACH Debit'),
    TypeCode('458', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Real Time Payment Debit'),
    TypeCode('459', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Real Time Payment Debits'),
    TypeCode('462', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Account Holder Initiated ACH Debit'),
    TypeCode('463', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Corporate Trade Payment Debits'),
    TypeCode('464', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Corporate Trade Payment Debit'),
    TypeCode('465', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Corporate Trade Payment Settlement'),
    TypeCode('466', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'ACH Settlement'),
    TypeCode('467', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'ACH Settlement Debits'),
    TypeCode('468', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'ACH Return Item or Adjustment Settlement'),
    TypeCode('469', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Miscellaneous ACH Debit'),
    TypeCode('470', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Check Paid'),
    TypeCode('471', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Check Paid – Cumulative MTD'),
    TypeCode('472', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Cumulative** Checks Paid'),
    TypeCode('474', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Certified Check Debit'),
    TypeCode('475', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Check Paid'),
    TypeCode('476', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Federal Reserve Bank Letter Debit'),
    TypeCode('477', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Bank Originated Debit'),
    TypeCode('478', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'List Post Debits'),
    TypeCode('479', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'List Post Debit'),
    TypeCode('480', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Loan Payments'),
    TypeCode('481', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Individual Loan Payment'),
    TypeCode('482', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Bank-Originated Debits'),
    TypeCode('484', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Draft'),
    TypeCode('485', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'DTC Debit'),
    TypeCode('486', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Cash Letter Debits'),
    TypeCode('487', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Cash Letter Debit'),
    TypeCode('489', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Cash Letter Adjustment'),
    TypeCode('490', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Outgoing Money Transfers'),
    TypeCode('491', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Individual Outgoing Internal Money Transfer'),
    TypeCode('493', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Customer Terminal Initiated Money Transfer'),
    TypeCode('495', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Outgoing Money Transfer'),
    TypeCode('496', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Money Transfer Adjustment'),
    TypeCode('498', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Compensation'),
    TypeCode('500', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Automatic Transfer Debits'),
    TypeCode('501', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Individual Automatic Transfer Debit'),
    TypeCode('502', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Bond Operations Debit'),
    TypeCode('505', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Book Transfer Debits'),
    TypeCode('506', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Book Transfer Debit'),
    TypeCode('507', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total International Money Transfer Debits'),
    TypeCode('508', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Individual International Money Transfer Debits'),
    TypeCode('510', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total International Debits'),
    TypeCode('512', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Letter of Credit Debit'),
    TypeCode('513', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Letter of Credit'),
    TypeCode('514', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Foreign Exchange Debit'),
    TypeCode('515', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Letters of Credit'),
    TypeCode('516', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Foreign Remittance Debit'),
    TypeCode('518', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Foreign Collection Debit'),
    TypeCode('522', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Foreign Checks Paid'),
    TypeCode('524', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Commission'),
    TypeCode('526', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'International Money Market Trading'),
    TypeCode('527', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Standing Order'),
    TypeCode('529', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Miscellaneous International Debit'),
    TypeCode('530', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Security Debits'),
    TypeCode('531', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Securities Purchased'),
    TypeCode('532', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Amount of Securities Purchased'),
    TypeCode('533', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Security Collection Debit'),
    TypeCode('534', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Miscellaneous Securities DB – FF'),
    TypeCode('535', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Purchase of Equity Securities'),
    TypeCode('536', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Miscellaneous Securities Debit – CHF'),
    TypeCode('537', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Collection Debit'),
    TypeCode('538', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Matured Repurchase Order'),
    TypeCode('539', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Bankers’ Acceptances Debit'),
    TypeCode('540', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Coupon Collection Debit'),
    TypeCode('541', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Bankers’ Acceptances'),
    TypeCode('542', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Purchase of Debt Securities'),
    TypeCode('543', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Domestic Collection'),
    TypeCode('544', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Interest/Matured Principal Payment'),
    TypeCode('546', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Commercial paper'),
    TypeCode('547', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Capital Change'),
    TypeCode('548', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Savings Bonds Sales Adjustment'),
    TypeCode('549', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Miscellaneous Security Debit'),
    TypeCode('550', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Deposited Items Returned'),
    TypeCode('551', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Credit Reversals'),
    TypeCode('552', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Credit Reversal'),
    TypeCode('554', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Posting Error Correction Debit'),
    TypeCode('555', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Deposited Item Returned'),
    TypeCode('556', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total ACH Return Items'),
    TypeCode('557', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Individual ACH Return Item'),
    TypeCode('558', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'ACH Reversal Debit'),
    TypeCode('560', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Rejected Debits'),
    TypeCode('561', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Individual Rejected Debit'),
    TypeCode('563', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Overdraft'),
    TypeCode('564', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Overdraft Fee'),
    TypeCode('566', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Return Item'),
    TypeCode('567', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Return Item Fee'),
    TypeCode('568', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Return Item Adjustment'),
    TypeCode('570', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total ZBA Debits'),
    TypeCode('574', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Cumulative ZBA Debits'),
    TypeCode('575', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'ZBA Debit'),
    TypeCode('577', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'ZBA Debit Transfer'),
    TypeCode('578', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'ZBA Debit Adjustment'),
    TypeCode('580', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Controlled Disbursing Debits'),
    TypeCode('581', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Individual Controlled Disbursing Debit'),
    TypeCode('583', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Disbursing Checks Paid – Early Amount'),
    TypeCode('584', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Disbursing Checks Paid – Later Amount'),
    TypeCode('585', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Disbursing Funding Requirement'),
    TypeCode('586', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'FRB Presentment Estimate (Fed Estimate)'),
    TypeCode('587', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Late Debits (After Notification)'),
    TypeCode('588', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Disbursing Checks Paid-Last Amount'),
    TypeCode('590', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total DTC Debits'),
    TypeCode('594', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total ATM Debits'),
    TypeCode('595', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'ATM Debit'),
    TypeCode('596', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total APR Debits'),
    TypeCode('597', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'ARP Debit'),
    TypeCode('601', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Estimated Total Disbursement'),
    TypeCode('602', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Adjusted Total Disbursement'),
    TypeCode('610', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Funds Required'),
    TypeCode('611', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Wire Transfers Out- CHF'),
    TypeCode('612', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Wire Transfers Out – FF'),
    TypeCode('613', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total International Debit – CHF'),
    TypeCode('614', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total International Debit – FF'),
    TypeCode('615', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Federal Reserve Bank – '
                                                                      'Commercial Bank Debit'),
    TypeCode('616', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Federal Reserve Bank – Commercial Bank Debit'),
    TypeCode('617', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Securities Purchased – CHF'),
    TypeCode('618', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Securities Purchased – FF'),
    TypeCode('621', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Broker Debits – CHF'),
    TypeCode('622', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Broker Debit'),
    TypeCode('623', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Broker Debits – FF'),
    TypeCode('625', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Broker Debits'),
    TypeCode('626', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Fed Funds Purchased'),
    TypeCode('627', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Fed Funds Purchased'),
    TypeCode('628', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Cash Center Debits'),
    TypeCode('629', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Cash Center Debit'),
    TypeCode('630', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Debit Adjustments'),
    TypeCode('631', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Debit Adjustment'),
    TypeCode('632', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Trust Debits'),
    TypeCode('633', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Trust Debit'),
    TypeCode('634', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'YTD Adjustment Debit'),
    TypeCode('640', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Escrow Debits'),
    TypeCode('641', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Individual Escrow Debit'),
    TypeCode('644', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Individual Back Value Debit'),
    TypeCode('646', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Transfer Calculation Debit'),
    TypeCode('650', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Investments Purchased'),
    TypeCode('651', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Individual Investment purchased'),
    TypeCode('654', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Interest Debit'),
    TypeCode('655', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Investment Interest Debits'),
    TypeCode('656', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Sweep Principal Buy'),
    TypeCode('657', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Futures Debit'),
    TypeCode('658', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Principal Payments Debit'),
    TypeCode('659', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Interest Adjustment Debit'),
    TypeCode('661', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Account Analysis Fee'),
    TypeCode('662', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Correspondent Collection Debit'),
    TypeCode('663', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Correspondent Collection Adjustment'),
    TypeCode('664', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Loan Participation'),
    TypeCode('665', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Intercept Debits'),
    TypeCode('666', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Currency and Coin Shipped'),
    TypeCode('667', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Food Stamp Letter'),
    TypeCode('668', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Food Stamp Adjustment'),
    TypeCode('669', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Clearing Settlement Debit'),
    TypeCode('670', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Back Value Debits'),
    TypeCode('672', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Back Value Adjustment'),
    TypeCode('673', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Customer Payroll'),
    TypeCode('674', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'FRB Statement Recap'),
    TypeCode('676', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Savings Bond Letter or Adjustment'),
    TypeCode('677', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Treasury Tax and Loan Debit'),
    TypeCode('678', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Transfer of Treasury Debit'),
    TypeCode('679', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'FRB Government Checks Cash Letter Debit'),
    TypeCode('681', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'FRB Government Check Adjustment'),
    TypeCode('682', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'FRB Postal Money Order Debit'),
    TypeCode('683', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'FRB Postal Money Order Adjustment'),
    TypeCode('684', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'FRB Cash Letter Auto Charge Debit'),
    TypeCode('685', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Universal Debits'),
    TypeCode('686', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'FRB Cash Letter Auto Charge Adjustment'),
    TypeCode('687', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'FRB Fine-Sort Cash Letter Debit'),
    TypeCode('688', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'FRB Fine-Sort Adjustment'),
    TypeCode('689', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'FRB Freight Payment Debits'),
    TypeCode('690', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Total Miscellaneous Debits'),
    TypeCode('691', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Universal Debit'),
    TypeCode('692', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Freight Payment Debit'),
    TypeCode('693', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Itemized Debit Over $10,000'),
    TypeCode('694', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Deposit Reversal'),
    TypeCode('695', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Deposit Correction Debit'),
    TypeCode('696', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Regular Collection Debit'),
    TypeCode('697', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Cumulative** Debits'),
    TypeCode('698', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Miscellaneous Fees'),
    TypeCode('699', TypeCodeTransaction.debit, TypeCodeLevel.detail, 'Miscellaneous Debit'),
    TypeCode('701', None, TypeCodeLevel.status, 'Principal Loan Balance'),
    TypeCode('703', None, TypeCodeLevel.status, 'Available Commitment Amount'),
    TypeCode('705', None, TypeCodeLevel.status, 'Payment Amount Due'),
    TypeCode('706', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Loan Participation'),
    TypeCode('707', None, TypeCodeLevel.status, 'Principal Amount Past Due'),
    TypeCode('708', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Value Dated Funds'),
    TypeCode('709', None, TypeCodeLevel.status, 'Interest Amount Past Due'),
    TypeCode('720', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Total Loan Payment'),
    TypeCode('721', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Amount Applied to Interest'),
    TypeCode('722', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Amount Applied to Principal'),
    TypeCode('723', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Amount Applied to Escrow'),
    TypeCode('724', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Amount Applied to Late Charges'),
    TypeCode('725', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Amount Applied to Buydown'),
    TypeCode('726', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Amount Applied to Misc. Fees'),
    TypeCode('727', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Amount Applied to Deferred Interest Detail'),
    TypeCode('728', TypeCodeTransaction.credit, TypeCodeLevel.detail, 'Amount Applied to Service Charge'),
    TypeCode('760', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Loan Disbursement'),
    TypeCode('827', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Incoming SEPA Payments'),
    TypeCode('828', TypeCodeTransaction.debit, TypeCodeLevel.summary, 'Outgoing SEPA Payments'),
    TypeCode('829', TypeCodeTransaction.credit, TypeCodeLevel.summary, 'Incoming BACS Payment'),
    TypeCode('890', TypeCodeTransaction.misc, TypeCodeLevel.detail, 'Contains Non-monetary Information'),
    TypeCode('906', None, TypeCodeLevel.detail, 'Today’s Opening 1 Day Float'),
    TypeCode('907', None, TypeCodeLevel.detail, 'Today’s Opening 2+ Day Float'),
]
TypeCodes = {
    type_code.code: type_code
    for type_code in TypeCodes
}
