"""Module for ablation-based Hyperparameter Importance analysis.

This module provides the `AblationGame` class for analyzing hyperparameter
interactions through ablation-based experiments.  It leverages a surrogate model
to estimate the impact of individual hyperparameters by systematically combining
baseline and optimized configuration values.  The `AblationGame` allows for
understanding which hyperparameters contribute most to performance gains and
identifying potential areas for further tuning.

The module is designed to work in conjunction with the `hypershap.games.abstract`
module and relies on the `hypershap.task` module for defining the
hyperparameter search space and surrogate model.

Key components include:

*   **`AblationGame`**: A game class that evaluates the performance of
    configurations generated by blending baseline and optimized hyperparameter
    values, providing insights into individual hyperparameter contributions.

This module facilitates a deeper understanding of hyperparameter dependencies
and guides optimization efforts by revealing the impact of each hyperparameter
in a systematic and interpretable manner.
"""

from __future__ import annotations

import numpy as np

from hypershap.games.abstract import AbstractHPIGame
from hypershap.task import AblationExplanationTask, MultiBaselineAblationExplanationTask
from hypershap.utils import Aggregation, evaluate_aggregation


class AblationGame(AbstractHPIGame):
    """The ablation game generates local explanations for hyperparameter configurations.

    It does so by evaluating all potential ablation paths, switching from a baseline configuration
    to an optimized configuration value by value. It leverages a surrogate model to estimate
    performance based on blended configurations.
    """

    def __init__(
        self,
        explanation_task: AblationExplanationTask,
        n_workers: int | None = None,
        verbose: bool | None = None,
    ) -> None:
        """Initialize an AblationGame.

        Args:
        explanation_task (AblationExplanationTask): An instance of `AblationExplanationTask`
            providing access to the baseline configuration, configuration of interest,
            and the surrogate model for evaluation.
        n_workers (int | None): The number of worker threads to use for parallel
            evaluation of coalitions. Defaults to None, which disables parallelization.
            Using more workers can significantly speed up the computation of Shapley values.
            The maximum number of workers is capped by the number of coalitions.
        verbose (bool | None): A boolean indicating whether to print verbose messages
            during computation. Defaults to None. When set to True, the method prints
            debugging information and progress updates.

        """
        super().__init__(explanation_task, n_workers=n_workers, verbose=verbose)

    def evaluate_single_coalition(self, coalition: np.ndarray) -> float:
        """Evaluate a single coalition (combination of baseline and optimized hyperparameters).

        This method blends the baseline and optimized configurations based on the provided coalition,
        then uses the surrogate model to estimate the performance of the blended configuration.

        Args:
            coalition (np.ndarray): A Boolean array indicating which hyperparameters should be taken from the
                baseline configuration (False) and which from the configuration of interest (True).

        Returns:
            float: The performance score of the blended configuration as estimated by the surrogate model.

        """
        baseline_cfg = self._get_explanation_task().baseline_config.get_array()
        cfg_of_interest = self._get_explanation_task().config_of_interest.get_array()
        blend = np.where(coalition == 0, baseline_cfg, cfg_of_interest)

        res = self._get_explanation_task().get_single_surrogate_model().evaluate(blend)

        # validate that we do not get a list of floats by accident
        if isinstance(res, list):  # pragma: no cover
            raise TypeError  # pragma: no cover

        return res

    def _get_explanation_task(self) -> AblationExplanationTask:
        """Retrieve the explanation task associated with this ablation game.

        This method simply returns the `explanation_task` attribute, which provides
        access to the baseline configuration, configuration of interest, and surrogate model.

        Returns:
            AblationExplanationTask: The explanation task associated with this ablation game.

        """
        if isinstance(self.explanation_task, AblationExplanationTask):
            return self.explanation_task

        raise ValueError  # pragma: no cover


class MultiBaselineAblationGame(AbstractHPIGame):
    """The multi-baseline ablation game generates local explanations for hyperparameter configurations.

    It does so by considering multiple baseline configurations as starting points.
    """

    def __init__(
        self,
        explanation_task: MultiBaselineAblationExplanationTask,
        aggregation: Aggregation = Aggregation.AVG,
        n_workers: int | None = None,
        verbose: bool | None = None,
    ) -> None:
        """Initialize a MultiBaselineAblationGame.

        n_workers (int | None): The number of worker threads to use for parallel
            evaluation of coalitions. Defaults to None, which disables parallelization.
            Using more workers can significantly speed up the computation of Shapley values.
            The maximum number of workers is capped by the number of coalitions.
        verbose (bool | None): A boolean indicating whether to print verbose messages
            during computation. Defaults to None. When set to True, the method prints
            debugging information and progress updates.
        """
        self.aggregation = aggregation
        self.ablation_games = [
            AblationGame(
                AblationExplanationTask(
                    config_space=explanation_task.config_space,
                    surrogate_model=explanation_task.surrogate_model,
                    baseline_config=baseline_config,
                    config_of_interest=explanation_task.config_of_interest,
                ),
                n_workers=n_workers,
                verbose=verbose,
            )
            for baseline_config in explanation_task.baseline_configs
        ]
        super().__init__(explanation_task, n_workers, verbose)

    def evaluate_single_coalition(self, coalition: np.ndarray) -> float:
        """Evaluate a single coalition (combination of baseline and optimized hyperparameters)."""
        vals = np.array([ag.evaluate_single_coalition(coalition) for ag in self.ablation_games])
        return evaluate_aggregation(self.aggregation, vals)
