from typing import List

from cidc_schemas.prism.constants import SUPPORTED_MANIFESTS

from .utils import (
    copy_dict_with_branch,
    get_prismify_args,
    get_test_trial,
    LocalFileUploadEntry,
    PrismTestData,
)

manifest_data_generators = []


def manifest_data_generator(f):
    manifest_data_generators.append(f)
    return f


@manifest_data_generator
def plasma() -> PrismTestData:
    upload_type = "plasma"
    prismify_args = get_prismify_args(upload_type)
    prismify_patch = {
        "protocol_identifier": "test_prism_trial_id",
        "shipments": [
            {
                "manifest_id": "test_prism_trial_id_PLASMA",
                "assay_priority": "4",
                "assay_type": "Olink",
                "receiving_party": "MSSM_Gnjatic",
                "courier": "USPS",
                "tracking_number": "TrackN",
                "account_number": "AccN",
                "shipping_condition": "Frozen_Dry_Ice",
                "date_shipped": "2001-10-10 00:00:00",
                "date_received": "2002-10-10 00:00:00",
                "quality_of_shipment": "Specimen shipment received in good condition",
                "ship_from": "ship from",
                "ship_to": "ship to",
            }
        ],
        "participants": [
            {
                "samples": [
                    {
                        "shipping_entry_number": 1,
                        "cimac_id": "CTTTP01A1.00",
                        "surgical_pathology_report_id": "Surgical pathology report 1",
                        "clinical_report_id": "clinical report 1",
                        "collection_event_name": "Baseline",
                        "diagnosis_verification": "Local pathology review was not consistent",
                        "site_description": "ADRENAL GLANDS",
                        "topography_code": "C00.2",
                        "topography_description": "External lower lip",
                        "histology_behavior": "8003/3",
                        "histology_behavior_description": "Neoplasm, malignant",
                        "parent_sample_id": "encrypt('TRIALGROUP 1')",
                        "processed_sample_id": "encrypt('BIOBANK 1')",
                        "box_number": "1",
                        "sample_location": "A1",
                        "type_of_sample": "Blood",
                        "sample_collection_procedure": "Core Biopsy",
                        "type_of_primary_container": "Stool collection container with DNA stabilizer",
                        "processed_sample_type": "Plasma",
                        "processed_sample_volume": 1.0,
                        "processed_sample_volume_units": "Other",
                        "material_used": 1.0,
                        "material_used_units": "Milliliters",
                        "material_remaining": 0.0,
                        "material_remaining_units": "Milliliters",
                        "material_storage_condition": "(-20)oC",
                        "quality_of_sample": "Pass",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Sample Returned",
                        "comments": "Comment",
                    },
                    {
                        "shipping_entry_number": 2,
                        "collection_event_name": "Baseline",
                        "parent_sample_id": "encrypt('TRIALGROUP 2')",
                        "processed_sample_id": "encrypt('BIOBANK 1')",
                        "cimac_id": "CTTTP01A2.00",
                        "box_number": "1",
                        "sample_location": "A2",
                        "type_of_sample": "Blood",
                        "sample_collection_procedure": "Core Biopsy",
                        "type_of_primary_container": "Stool collection container with DNA stabilizer",
                        "processed_sample_type": "Plasma",
                        "processed_sample_volume": 2.0,
                        "processed_sample_volume_units": "Other",
                        "material_used": 1.0,
                        "material_used_units": "Milliliters",
                        "material_remaining": 0.0,
                        "material_remaining_units": "Milliliters",
                        "material_storage_condition": "(-20)oC",
                        "quality_of_sample": "Pass",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Sample Returned",
                        "comments": "No comment",
                    },
                    {
                        "shipping_entry_number": 3,
                        "collection_event_name": "Baseline",
                        "parent_sample_id": "encrypt('TRIALGROUP 3')",
                        "processed_sample_id": "encrypt('BIOBANK 1')",
                        "cimac_id": "CTTTP01A3.00",
                        "box_number": "1",
                        "sample_location": "A3",
                        "type_of_sample": "Blood",
                        "sample_collection_procedure": "Core Biopsy",
                        "type_of_primary_container": "Stool collection container with DNA stabilizer",
                        "processed_sample_type": "Plasma",
                        "processed_sample_volume": 3.0,
                        "processed_sample_volume_units": "Other",
                        "material_used": 1.0,
                        "material_used_units": "Milliliters",
                        "material_remaining": 0.0,
                        "material_remaining_units": "Milliliters",
                        "material_storage_condition": "(-20)oC",
                        "quality_of_sample": "Pass",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Sample Returned",
                    },
                ],
                "participant_id": "TTTPP103",
                "cimac_participant_id": "CTTTP01",
                "gender": "Female",
                "race": "Black/African American",
                "ethnicity": "Not Hispanic or Latino",
                "cohort_name": "Arm_A",
            },
            {
                "samples": [
                    {
                        "shipping_entry_number": 4,
                        "collection_event_name": "Baseline",
                        "parent_sample_id": "encrypt('TRIALGROUP 4')",
                        "processed_sample_id": "encrypt('BIOBANK 1')",
                        "cimac_id": "CTTTP02A1.00",
                        "box_number": "1",
                        "sample_location": "A4",
                        "type_of_sample": "Blood",
                        "sample_collection_procedure": "Core Biopsy",
                        "type_of_primary_container": "Stool collection container with DNA stabilizer",
                        "processed_sample_type": "Plasma",
                        "processed_sample_volume": 4.0,
                        "processed_sample_volume_units": "Other",
                        "material_used": 1.0,
                        "material_used_units": "Milliliters",
                        "material_remaining": 0.0,
                        "material_remaining_units": "Milliliters",
                        "material_storage_condition": "(-20)oC",
                        "quality_of_sample": "Pass",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Sample Returned",
                    },
                    {
                        "shipping_entry_number": 5,
                        "collection_event_name": "Baseline",
                        "parent_sample_id": "encrypt('TRIALGROUP 5')",
                        "processed_sample_id": "encrypt('BIOBANK 1')",
                        "cimac_id": "CTTTP02A2.00",
                        "box_number": "1",
                        "sample_location": "A5",
                        "type_of_sample": "Blood",
                        "sample_collection_procedure": "Core Biopsy",
                        "type_of_primary_container": "Stool collection container with DNA stabilizer",
                        "processed_sample_type": "Plasma",
                        "processed_sample_volume": 5.0,
                        "processed_sample_volume_units": "Other",
                        "material_used": 1.0,
                        "material_used_units": "Milliliters",
                        "material_remaining": 0.0,
                        "material_remaining_units": "Milliliters",
                        "material_storage_condition": "(-20)oC",
                        "quality_of_sample": "Pass",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Sample Returned",
                    },
                    {
                        "shipping_entry_number": 6,
                        "collection_event_name": "Baseline",
                        "parent_sample_id": "encrypt('TRIALGROUP 6')",
                        "processed_sample_id": "encrypt('BIOBANK 1')",
                        "cimac_id": "CTTTP02A3.00",
                        "box_number": "1",
                        "sample_location": "A6",
                        "type_of_sample": "Blood",
                        "sample_collection_procedure": "Core Biopsy",
                        "type_of_primary_container": "Stool collection container with DNA stabilizer",
                        "processed_sample_type": "Plasma",
                        "processed_sample_volume": 6.0,
                        "processed_sample_volume_units": "Other",
                        "material_used": 1.0,
                        "material_used_units": "Milliliters",
                        "material_remaining": 0.0,
                        "material_remaining_units": "Milliliters",
                        "material_storage_condition": "(-20)oC",
                        "quality_of_sample": "Pass",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Sample Returned",
                    },
                ],
                "cohort_name": "Arm_Z",
                "participant_id": "TTTPP203",
                "cimac_participant_id": "CTTTP02",
            },
        ],
    }
    upload_entries: List[LocalFileUploadEntry] = []
    base_trial = get_test_trial(
        allowed_collection_event_names=["Baseline"],
        allowed_cohort_names=["Arm_A", "Arm_Z"],
    )

    target_trial = copy_dict_with_branch(
        base_trial, prismify_patch, ["participants", "shipments"]
    )

    return PrismTestData(
        upload_type,
        prismify_args,
        prismify_patch,
        upload_entries,
        base_trial,
        target_trial,
    )


@manifest_data_generator
def pbmc() -> PrismTestData:
    upload_type = "pbmc"
    prismify_args = get_prismify_args(upload_type)
    prismify_patch = {
        "protocol_identifier": "test_prism_trial_id",
        "shipments": [
            {
                "manifest_id": "test_prism_trial_id_PBMC",
                "assay_priority": "4",
                "assay_type": "Olink",
                "receiving_party": "MSSM_Rahman",
                "courier": "USPS",
                "tracking_number": "TrackN",
                "account_number": "AccN",
                "shipping_condition": "Frozen_Dry_Ice",
                "date_shipped": "2001-10-10 00:00:00",
                "date_received": "2002-10-10 00:00:00",
                "quality_of_shipment": "Specimen shipment received in good condition",
                "ship_from": "ship from",
                "ship_to": "ship to",
            }
        ],
        "participants": [
            {
                "samples": [
                    {
                        "shipping_entry_number": 1,
                        "cimac_id": "CTTTP01A1.00",
                        "surgical_pathology_report_id": "Surgical pathology report 1",
                        "clinical_report_id": "clinical report 1",
                        "collection_event_name": "Baseline",
                        "diagnosis_verification": "Local pathology review was not consistent",
                        "site_description": "ANAL CANAL & ANUS",
                        "topography_code": "C00.1",
                        "topography_description": "LIP",
                        "histology_behavior": "8004/3",
                        "histology_behavior_description": "Neoplasm, malignant",
                        "parent_sample_id": "TRIALGROUP 1",
                        "processed_sample_id": "BIOBANK 1",
                        "box_number": "1",
                        "sample_location": "123",
                        "type_of_sample": "Blood",
                        "sample_collection_procedure": "Core Biopsy",
                        "type_of_primary_container": "Stool collection container with DNA stabilizer",
                        "processed_sample_type": "Plasma",
                        "processed_sample_volume": 1.0,
                        "processed_sample_volume_units": "Other",
                        "processed_sample_concentration": 0.2,
                        "processed_sample_concentration_units": "Not Reported",
                        "pbmc_viability": 1.0,
                        "pbmc_recovery": 1.0,
                        "pbmc_resting_period_used": "Yes",
                        "material_used": 1.0,
                        "material_remaining": 0.0,
                        "material_storage_condition": "Other",
                        "quality_of_sample": "Pass",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Sample Returned",
                        "comments": "Comment",
                    },
                    {
                        "shipping_entry_number": 2,
                        "cimac_id": "CTTTP01A2.00",
                        "surgical_pathology_report_id": "Surgical pathology report 2",
                        "clinical_report_id": "clinical report 2",
                        "collection_event_name": "Pre_Day_1_Cycle_2",
                        "diagnosis_verification": "Local pathology review was not consistent",
                        "site_description": "ANAL CANAL & ANUS",
                        "topography_code": "C00.1",
                        "topography_description": "LIP",
                        "histology_behavior": "8004/3",
                        "histology_behavior_description": "Neoplasm, malignant",
                        "parent_sample_id": "TRIALGROUP 2",
                        "processed_sample_id": "BIOBANK 1",
                        "box_number": "1",
                        "sample_location": "123",
                        "type_of_sample": "Blood",
                        "sample_collection_procedure": "Core Biopsy",
                        "type_of_primary_container": "Stool collection container with DNA stabilizer",
                        "processed_sample_type": "Plasma",
                        "processed_sample_volume": 2.0,
                        "processed_sample_volume_units": "Other",
                        "processed_sample_concentration": 0.3,
                        "processed_sample_concentration_units": "Not Reported",
                        "pbmc_viability": 1.0,
                        "pbmc_recovery": 1.0,
                        "pbmc_resting_period_used": "No",
                        "material_used": 1.0,
                        "material_remaining": 0.0,
                        "material_storage_condition": "Other",
                        "quality_of_sample": "Fail",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Sample Returned",
                    },
                    {
                        "shipping_entry_number": 3,
                        "cimac_id": "CTTTP01A3.00",
                        "surgical_pathology_report_id": "Surgical pathology report 3",
                        "clinical_report_id": "clinical report 3",
                        "collection_event_name": "Baseline",
                        "diagnosis_verification": "Local pathology review was not consistent",
                        "site_description": "ANAL CANAL & ANUS",
                        "topography_code": "C00.1",
                        "topography_description": "LIP",
                        "histology_behavior": "8004/3",
                        "histology_behavior_description": "Neoplasm, malignant",
                        "parent_sample_id": "TRIALGROUP 3",
                        "processed_sample_id": "BIOBANK 1",
                        "box_number": "1",
                        "sample_location": "123",
                        "type_of_sample": "Blood",
                        "sample_collection_procedure": "Core Biopsy",
                        "type_of_primary_container": "Stool collection container with DNA stabilizer",
                        "processed_sample_type": "Plasma",
                        "processed_sample_volume": 3.0,
                        "processed_sample_volume_units": "Other",
                        "processed_sample_concentration": 0.2,
                        "processed_sample_concentration_units": "Not Reported",
                        "pbmc_viability": 1.0,
                        "pbmc_recovery": 1.0,
                        "pbmc_resting_period_used": "Not Reported",
                        "material_used": 1.0,
                        "material_remaining": 0.0,
                        "material_storage_condition": "Other",
                        "quality_of_sample": "Pass",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Sample Returned",
                    },
                ],
                "participant_id": "TTTP01A3",
                "cimac_participant_id": "CTTTP01",
                "gender": "Female",
                "race": "Asian",
                "ethnicity": "Hispanic or Latino",
                "cohort_name": "Arm_Z",
            },
            {
                "samples": [
                    {
                        "shipping_entry_number": 4,
                        "cimac_id": "CTTTP02A1.00",
                        "surgical_pathology_report_id": "Surgical pathology report 4",
                        "clinical_report_id": "clinical report 4",
                        "collection_event_name": "Baseline",
                        "diagnosis_verification": "Local pathology review was not consistent",
                        "site_description": "ANAL CANAL & ANUS",
                        "topography_code": "C00.1",
                        "topography_description": "LIP",
                        "histology_behavior": "8004/3",
                        "histology_behavior_description": "Neoplasm, malignant",
                        "parent_sample_id": "TRIALGROUP 4",
                        "processed_sample_id": "BIOBANK 1",
                        "box_number": "1",
                        "sample_location": "123",
                        "type_of_sample": "Blood",
                        "sample_collection_procedure": "Core Biopsy",
                        "type_of_primary_container": "Stool collection container with DNA stabilizer",
                        "processed_sample_type": "Plasma",
                        "processed_sample_volume": 4.0,
                        "processed_sample_volume_units": "Other",
                        "processed_sample_concentration": 0.2,
                        "processed_sample_concentration_units": "Not Reported",
                        "pbmc_viability": 1.0,
                        "pbmc_recovery": 1.0,
                        "pbmc_resting_period_used": "Other",
                        "material_used": 1.0,
                        "material_remaining": 0.0,
                        "material_storage_condition": "Not Reported",
                        "quality_of_sample": "Fail",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Sample Returned",
                    },
                    {
                        "shipping_entry_number": 5,
                        "cimac_id": "CTTTP02A2.00",
                        "surgical_pathology_report_id": "Surgical pathology report 5",
                        "clinical_report_id": "clinical report 5",
                        "collection_event_name": "Pre_Day_1_Cycle_2",
                        "diagnosis_verification": "Local pathology review was not consistent",
                        "site_description": "ANAL CANAL & ANUS",
                        "topography_code": "C00.1",
                        "topography_description": "LIP",
                        "histology_behavior": "8004/3",
                        "histology_behavior_description": "Neoplasm, malignant",
                        "parent_sample_id": "TRIALGROUP 5",
                        "processed_sample_id": "BIOBANK 1",
                        "box_number": "1",
                        "sample_location": "123",
                        "type_of_sample": "Blood",
                        "sample_collection_procedure": "Core Biopsy",
                        "type_of_primary_container": "Stool collection container with DNA stabilizer",
                        "processed_sample_type": "Plasma",
                        "processed_sample_volume": 5.0,
                        "processed_sample_volume_units": "Other",
                        "processed_sample_concentration": 0.2,
                        "processed_sample_concentration_units": "Not Reported",
                        "pbmc_viability": 1.0,
                        "pbmc_recovery": 1.0,
                        "pbmc_resting_period_used": "Other",
                        "material_used": 1.0,
                        "material_remaining": 0.0,
                        "material_storage_condition": "Not Reported",
                        "quality_of_sample": "Pass",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Sample Returned",
                    },
                    {
                        "shipping_entry_number": 6,
                        "cimac_id": "CTTTP02A3.00",
                        "surgical_pathology_report_id": "Surgical pathology report 6",
                        "clinical_report_id": "clinical report 6",
                        "collection_event_name": "Baseline",
                        "diagnosis_verification": "Local pathology review was not consistent",
                        "site_description": "ANAL CANAL & ANUS",
                        "topography_code": "C00.1",
                        "topography_description": "LIP",
                        "histology_behavior": "8004/3",
                        "histology_behavior_description": "Neoplasm, malignant",
                        "parent_sample_id": "TRIALGROUP 6",
                        "processed_sample_id": "BIOBANK 1",
                        "box_number": "1",
                        "sample_location": "123",
                        "type_of_sample": "Blood",
                        "sample_collection_procedure": "Core Biopsy",
                        "type_of_primary_container": "Stool collection container with DNA stabilizer",
                        "processed_sample_type": "Plasma",
                        "processed_sample_volume": 6.0,
                        "processed_sample_volume_units": "Other",
                        "processed_sample_concentration": 0.3,
                        "processed_sample_concentration_units": "Not Reported",
                        "pbmc_viability": 1.0,
                        "pbmc_recovery": 1.0,
                        "pbmc_resting_period_used": "Other",
                        "material_used": 1.0,
                        "material_remaining": 0.0,
                        "material_storage_condition": "Not Reported",
                        "quality_of_sample": "Fail",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Sample Returned",
                    },
                ],
                "participant_id": "TTTP02A3",
                "cimac_participant_id": "CTTTP02",
                "gender": "Male",
                "race": "Native Hawaiian/Pacific Islander",
                "ethnicity": "Unknown",
                "cohort_name": "Arm_A",
            },
        ],
    }
    upload_entries = []
    base_trial = get_test_trial(
        allowed_cohort_names=["Arm_A", "Arm_Z"],
        allowed_collection_event_names=["Baseline", "Pre_Day_1_Cycle_2"],
    )

    target_trial = copy_dict_with_branch(
        base_trial, prismify_patch, ["participants", "shipments"]
    )

    return PrismTestData(
        upload_type,
        prismify_args,
        prismify_patch,
        upload_entries,
        base_trial,
        target_trial,
    )


@manifest_data_generator
def tissue_slide() -> PrismTestData:
    upload_type = "tissue_slide"
    prismify_args = get_prismify_args(upload_type)
    prismify_patch = {
        "protocol_identifier": "test_prism_trial_id",
        "shipments": [
            {
                "manifest_id": "test_prism_trial_id_slide",
                "assay_priority": "3",
                "assay_type": "IHC",
                "receiving_party": "DFCI_Severgnini",
                "courier": "USPS",
                "tracking_number": "TrackN",
                "account_number": "AccN",
                "shipping_condition": "Not Reported",
                "date_shipped": "2001-10-10 00:00:00",
                "date_received": "2002-10-10 00:00:00",
                "quality_of_shipment": "Specimen shipment received in poor condition",
                "ship_from": "ship from",
                "ship_to": "ship to",
            }
        ],
        "participants": [
            {
                "samples": [
                    {
                        "shipping_entry_number": 1,
                        "collection_event_name": "Baseline",
                        "parent_sample_id": "TRIALGROUP 1",
                        "processed_sample_id": "BIOBANK 1",
                        "cimac_id": "CTTTP08T1.00",
                        "box_number": "2",
                        "sample_location": "A1",
                        "type_of_sample": "Tumor Tissue",
                        "type_of_tumor_sample": "Metastatic Tumor",
                        "sample_collection_procedure": "Core Biopsy",
                        "core_number": 1.0,
                        "fixation_stabilization_type": "Archival FFPE",
                        "processed_sample_type": "Fixed Slide",
                        "processed_sample_quantity": 4.0,
                        "material_used": 3.0,
                        "material_used_units": "Slides",
                        "material_remaining": 1.0,
                        "material_remaining_units": "Slides",
                        "material_storage_condition": "RT",
                        "quality_of_sample": "Pass",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Not Reported",
                    },
                    {
                        "shipping_entry_number": 2,
                        "collection_event_name": "Pre_Day_1_Cycle_2",
                        "parent_sample_id": "TRIALGROUP 2",
                        "processed_sample_id": "BIOBANK 1",
                        "cimac_id": "CTTTP08T2.00",
                        "box_number": "2",
                        "sample_location": "A2",
                        "type_of_sample": "Tumor Tissue",
                        "type_of_tumor_sample": "Metastatic Tumor",
                        "sample_collection_procedure": "Core Biopsy",
                        "core_number": 1.0,
                        "fixation_stabilization_type": "Formalin-Fixed Paraffin-Embedded",
                        "processed_sample_type": "Fixed Slide",
                        "processed_sample_quantity": 4.0,
                        "material_used": 3.0,
                        "material_used_units": "Slides",
                        "material_remaining": 1.0,
                        "material_remaining_units": "Slides",
                        "material_storage_condition": "RT",
                        "quality_of_sample": "Pass",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Not Reported",
                    },
                    {
                        "shipping_entry_number": 3,
                        "collection_event_name": "Pre_Day_1_Cycle_2",
                        "parent_sample_id": "TRIALGROUP 3",
                        "processed_sample_id": "BIOBANK 1",
                        "cimac_id": "CTTTP08T3.00",
                        "box_number": "2",
                        "sample_location": "A3",
                        "type_of_sample": "Tumor Tissue",
                        "type_of_tumor_sample": "Primary Tumor",
                        "sample_collection_procedure": "Core Biopsy",
                        "core_number": 1.0,
                        "fixation_stabilization_type": "Formalin-Fixed Paraffin-Embedded",
                        "processed_sample_type": "H&E-Stained Fixed Tissue Slide Specimen",
                        "processed_sample_quantity": 1.0,
                        "material_used": 1.0,
                        "material_used_units": "Slides",
                        "material_remaining": 0.0,
                        "material_remaining_units": "Slides",
                        "material_storage_condition": "RT",
                        "quality_of_sample": "Pass",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Not Reported",
                    },
                ],
                "cohort_name": "Arm_A",
                "participant_id": "TTTPP803",
                "cimac_participant_id": "CTTTP08",
            },
            {
                "samples": [
                    {
                        "shipping_entry_number": 4,
                        "collection_event_name": "Baseline",
                        "parent_sample_id": "TRIALGROUP 4",
                        "processed_sample_id": "BIOBANK 1",
                        "cimac_id": "CTTTP09T1.00",
                        "box_number": "2",
                        "sample_location": "A4",
                        "type_of_sample": "Tumor Tissue",
                        "type_of_tumor_sample": "Primary Tumor",
                        "sample_collection_procedure": "Core Biopsy",
                        "core_number": 1.0,
                        "fixation_stabilization_type": "Not Reported",
                        "processed_sample_type": "H&E-Stained Fixed Tissue Slide Specimen",
                        "processed_sample_quantity": 1.0,
                        "material_used": 1.0,
                        "material_used_units": "Slides",
                        "material_remaining": 0.0,
                        "material_remaining_units": "Slides",
                        "material_storage_condition": "RT",
                        "quality_of_sample": "Pass",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Not Reported",
                    }
                ],
                "cohort_name": "Arm_Z",
                "participant_id": "TTTPP901",
                "cimac_participant_id": "CTTTP09",
            },
        ],
    }
    upload_entries = []
    base_trial = get_test_trial(
        allowed_cohort_names=["Arm_A", "Arm_Z"],
        allowed_collection_event_names=["Baseline", "Pre_Day_1_Cycle_2"],
    )

    target_trial = copy_dict_with_branch(
        base_trial, prismify_patch, ["participants", "shipments"]
    )

    return PrismTestData(
        upload_type,
        prismify_args,
        prismify_patch,
        upload_entries,
        base_trial,
        target_trial,
    )


@manifest_data_generator
def normal_blood_dna() -> PrismTestData:
    upload_type = "normal_blood_dna"
    prismify_args = get_prismify_args(upload_type)
    prismify_patch = {
        "protocol_identifier": "test_prism_trial_id",
        "shipments": [
            {
                "manifest_id": "test_prism_trial_id_DNA",
                "assay_priority": "4",
                "assay_type": "WES",
                "receiving_party": "MDA_Bernatchez",
                "courier": "USPS",
                "tracking_number": "TrackN",
                "account_number": "AccN",
                "shipping_condition": "Ice_Pack",
                "date_shipped": "2001-10-10 00:00:00",
                "date_received": "2002-10-10 00:00:00",
                "quality_of_shipment": "Specimen shipment received in good condition",
                "ship_from": "ship from",
                "ship_to": "ship to",
            }
        ],
        "participants": [
            {
                "samples": [
                    {
                        "shipping_entry_number": 1,
                        "collection_event_name": "Baseline",
                        "parent_sample_id": "TRIALGROUP 1",
                        "processed_sample_id": "BIOBANK 1",
                        "cimac_id": "CTTTP01N1.00",
                        "box_number": "1",
                        "sample_location": "A1",
                        "type_of_sample": "Blood",
                        "sample_collection_procedure": "Blood Draw",
                        "type_of_primary_container": "Blood specimen container with EDTA",
                        "processed_sample_type": "Plasma",
                        "processed_sample_volume": 1.0,
                        "processed_sample_volume_units": "Other",
                        "processed_sample_quantity": 1.0,
                        "processed_sample_derivative": "Tumor DNA",
                        "sample_derivative_volume": 1.0,
                        "sample_derivative_volume_units": "Microliters",
                        "sample_derivative_concentration": 1.0,
                        "sample_derivative_concentration_units": "Nanogram per Microliter",
                        "din": 9.0,
                        "a260_a280": 1.8,
                        "a260_a230": 2.0,
                        "material_used": 1.0,
                        "material_used_units": "Milliliters",
                        "material_remaining": 0.0,
                        "material_remaining_units": "Milliliters",
                        "material_storage_condition": "(-20)oC",
                        "quality_of_sample": "Pass",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Not Reported",
                        "comments": "Comment",
                    },
                    {
                        "shipping_entry_number": 2,
                        "collection_event_name": "Baseline",
                        "parent_sample_id": "TRIALGROUP 2",
                        "processed_sample_id": "BIOBANK 1",
                        "cimac_id": "CTTTP01N2.00",
                        "box_number": "1",
                        "sample_location": "A2",
                        "type_of_sample": "Blood",
                        "sample_collection_procedure": "Blood Draw",
                        "type_of_primary_container": "Blood specimen container with EDTA",
                        "processed_sample_type": "Plasma",
                        "processed_sample_volume": 2.0,
                        "processed_sample_volume_units": "Other",
                        "processed_sample_quantity": 2.0,
                        "processed_sample_derivative": "Tumor RNA",
                        "sample_derivative_volume": 2.0,
                        "sample_derivative_volume_units": "Milliliters",
                        "sample_derivative_concentration": 2.0,
                        "sample_derivative_concentration_units": "Milligram per Milliliter",
                        "din": 9.0,
                        "a260_a280": 1.7,
                        "a260_a230": 2.1,
                        "material_used": 1.0,
                        "material_used_units": "Milliliters",
                        "material_remaining": 0.0,
                        "material_remaining_units": "Milliliters",
                        "material_storage_condition": "(-20)oC",
                        "quality_of_sample": "Pass",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Not Reported",
                        "comments": "No comment",
                    },
                    {
                        "shipping_entry_number": 3,
                        "collection_event_name": "Baseline",
                        "parent_sample_id": "TRIALGROUP 3",
                        "processed_sample_id": "BIOBANK 1",
                        "cimac_id": "CTTTP01N3.00",
                        "box_number": "1",
                        "sample_location": "A3",
                        "type_of_sample": "Blood",
                        "sample_collection_procedure": "Blood Draw",
                        "type_of_primary_container": "Blood specimen container with EDTA",
                        "processed_sample_type": "Plasma",
                        "processed_sample_volume": 3.0,
                        "processed_sample_volume_units": "Other",
                        "processed_sample_quantity": 3.0,
                        "processed_sample_derivative": "Germline DNA",
                        "sample_derivative_volume": 3.0,
                        "sample_derivative_volume_units": "Microliters",
                        "sample_derivative_concentration": 3.0,
                        "sample_derivative_concentration_units": "Micrograms per Microliter",
                        "din": 8.0,
                        "a260_a280": 1.9,
                        "a260_a230": 2.2,
                        "material_used": 1.0,
                        "material_used_units": "Milliliters",
                        "material_remaining": 0.0,
                        "material_remaining_units": "Milliliters",
                        "material_storage_condition": "(-20)oC",
                        "quality_of_sample": "Pass",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Not Reported",
                    },
                ],
                "cohort_name": "Arm_A",
                "participant_id": "TTTPP103",
                "cimac_participant_id": "CTTTP01",
            },
            {
                "samples": [
                    {
                        "shipping_entry_number": 4,
                        "collection_event_name": "Baseline",
                        "parent_sample_id": "TRIALGROUP 4",
                        "processed_sample_id": "BIOBANK 1",
                        "cimac_id": "CTTTP02N1.00",
                        "box_number": "1",
                        "sample_location": "A4",
                        "type_of_sample": "Blood",
                        "sample_collection_procedure": "Blood Draw",
                        "type_of_primary_container": "Blood specimen container with EDTA",
                        "processed_sample_type": "Plasma",
                        "processed_sample_volume": 4.0,
                        "processed_sample_volume_units": "Other",
                        "processed_sample_quantity": 4.0,
                        "processed_sample_derivative": "Circulating Tumor-Derived DNA",
                        "sample_derivative_volume": 4.0,
                        "sample_derivative_volume_units": "Milliliters",
                        "sample_derivative_concentration": 4.0,
                        "sample_derivative_concentration_units": "Cells per Vial",
                        "din": 9.0,
                        "a260_a280": 1.8,
                        "a260_a230": 2.0,
                        "material_used": 1.0,
                        "material_used_units": "Milliliters",
                        "material_remaining": 0.0,
                        "material_remaining_units": "Milliliters",
                        "material_storage_condition": "(-20)oC",
                        "quality_of_sample": "Pass",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Not Reported",
                    },
                    {
                        "shipping_entry_number": 5,
                        "collection_event_name": "Baseline",
                        "parent_sample_id": "TRIALGROUP 5",
                        "processed_sample_id": "BIOBANK 1",
                        "cimac_id": "CTTTP02N2.00",
                        "box_number": "1",
                        "sample_location": "A5",
                        "type_of_sample": "Blood",
                        "sample_collection_procedure": "Blood Draw",
                        "type_of_primary_container": "Blood specimen container with EDTA",
                        "processed_sample_type": "Plasma",
                        "processed_sample_volume": 5.0,
                        "processed_sample_volume_units": "Other",
                        "processed_sample_quantity": 5.0,
                        "processed_sample_derivative": "Not Reported",
                        "sample_derivative_volume": 5.0,
                        "sample_derivative_volume_units": "Not Reported",
                        "sample_derivative_concentration": 5.0,
                        "sample_derivative_concentration_units": "Not Reported",
                        "din": 9.0,
                        "a260_a280": 1.7,
                        "a260_a230": 2.1,
                        "material_used": 1.0,
                        "material_used_units": "Milliliters",
                        "material_remaining": 0.0,
                        "material_remaining_units": "Milliliters",
                        "material_storage_condition": "(-20)oC",
                        "quality_of_sample": "Pass",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Not Reported",
                    },
                    {
                        "shipping_entry_number": 6,
                        "collection_event_name": "Baseline",
                        "parent_sample_id": "TRIALGROUP 6",
                        "processed_sample_id": "BIOBANK 1",
                        "cimac_id": "CTTTP02N3.00",
                        "box_number": "1",
                        "sample_location": "A6",
                        "type_of_sample": "Blood",
                        "sample_collection_procedure": "Blood Draw",
                        "type_of_primary_container": "Blood specimen container with EDTA",
                        "processed_sample_type": "Plasma",
                        "processed_sample_volume": 6.0,
                        "processed_sample_volume_units": "Other",
                        "processed_sample_quantity": 6.0,
                        "processed_sample_derivative": "Other",
                        "sample_derivative_volume": 6.0,
                        "sample_derivative_volume_units": "Other",
                        "sample_derivative_concentration": 6.0,
                        "sample_derivative_concentration_units": "Other",
                        "din": 8.0,
                        "a260_a280": 1.9,
                        "a260_a230": 2.2,
                        "material_used": 1.0,
                        "material_used_units": "Milliliters",
                        "material_remaining": 0.0,
                        "material_remaining_units": "Milliliters",
                        "material_storage_condition": "(-20)oC",
                        "quality_of_sample": "Pass",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Not Reported",
                    },
                ],
                "cohort_name": "Arm_Z",
                "participant_id": "TTTPP203",
                "cimac_participant_id": "CTTTP02",
            },
        ],
    }
    upload_entries = []
    base_trial = get_test_trial(
        allowed_cohort_names=["Arm_A", "Arm_Z"],
        allowed_collection_event_names=["Baseline", "Pre_Day_1_Cycle_2"],
    )

    target_trial = copy_dict_with_branch(
        base_trial, prismify_patch, ["participants", "shipments"]
    )

    return PrismTestData(
        upload_type,
        prismify_args,
        prismify_patch,
        upload_entries,
        base_trial,
        target_trial,
    )


@manifest_data_generator
def normal_tissue_dna() -> PrismTestData:
    upload_type = "normal_tissue_dna"
    prismify_args = get_prismify_args(upload_type)
    prismify_patch = {
        "protocol_identifier": "test_prism_trial_id",
        "shipments": [
            {
                "manifest_id": "test_prism_trial_id_DNA",
                "assay_priority": "4",
                "assay_type": "WES",
                "receiving_party": "Broad_Cibulskis",
                "courier": "USPS",
                "tracking_number": "TrackN",
                "account_number": "AccN",
                "shipping_condition": "Ice_Pack",
                "date_shipped": "2001-10-10 00:00:00",
                "date_received": "2002-10-10 00:00:00",
                "quality_of_shipment": "Specimen shipment received in good condition",
                "ship_from": "ship from",
                "ship_to": "ship to",
            }
        ],
        "participants": [
            {
                "samples": [
                    {
                        "shipping_entry_number": 1,
                        "collection_event_name": "Baseline",
                        "parent_sample_id": "TRIALGROUP 1",
                        "processed_sample_id": "BIOBANK 1",
                        "cimac_id": "CTTTP03N1.00",
                        "box_number": "1",
                        "sample_location": "A1",
                        "type_of_sample": "Skin Tissue",
                        "sample_collection_procedure": "Core Biopsy",
                        "core_number": 1.0,
                        "fixation_stabilization_type": "Archival FFPE",
                        "processed_sample_type": "Fixed Slide",
                        "processed_sample_quantity": 1.0,
                        "processed_sample_derivative": "Germline DNA",
                        "sample_derivative_volume": 1.0,
                        "sample_derivative_volume_units": "Microliters",
                        "sample_derivative_concentration": 1.0,
                        "sample_derivative_concentration_units": "Nanogram per Microliter",
                        "din": 9.0,
                        "a260_a280": 1.8,
                        "a260_a230": 2.0,
                        "material_used": 1.0,
                        "material_used_units": "Milliliters",
                        "material_remaining": 0.0,
                        "material_remaining_units": "Milliliters",
                        "material_storage_condition": "(-20)oC",
                        "quality_of_sample": "Pass",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Not Reported",
                        "comments": "Comment",
                    },
                    {
                        "shipping_entry_number": 2,
                        "collection_event_name": "Baseline",
                        "parent_sample_id": "TRIALGROUP 2",
                        "processed_sample_id": "BIOBANK 1",
                        "cimac_id": "CTTTP03N2.00",
                        "box_number": "1",
                        "sample_location": "A2",
                        "type_of_sample": "Skin Tissue",
                        "sample_collection_procedure": "Core Biopsy",
                        "core_number": 1.0,
                        "fixation_stabilization_type": "Fresh Specimen",
                        "processed_sample_type": "Fixed Slide",
                        "processed_sample_quantity": 1.0,
                        "processed_sample_derivative": "Germline DNA",
                        "sample_derivative_volume": 2.0,
                        "sample_derivative_volume_units": "Milliliters",
                        "sample_derivative_concentration": 2.0,
                        "sample_derivative_concentration_units": "Milligram per Milliliter",
                        "din": 9.0,
                        "a260_a280": 1.7,
                        "a260_a230": 2.1,
                        "material_used": 1.0,
                        "material_used_units": "Milliliters",
                        "material_remaining": 0.0,
                        "material_remaining_units": "Milliliters",
                        "material_storage_condition": "(-20)oC",
                        "quality_of_sample": "Pass",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Not Reported",
                        "comments": "No comment",
                    },
                    {
                        "shipping_entry_number": 3,
                        "collection_event_name": "Baseline",
                        "parent_sample_id": "TRIALGROUP 3",
                        "processed_sample_id": "BIOBANK 1",
                        "cimac_id": "CTTTP03N3.00",
                        "box_number": "1",
                        "sample_location": "A3",
                        "type_of_sample": "Skin Tissue",
                        "sample_collection_procedure": "Core Biopsy",
                        "core_number": 1.0,
                        "fixation_stabilization_type": "Frozen Specimen",
                        "processed_sample_type": "Fixed Slide",
                        "processed_sample_quantity": 2.0,
                        "processed_sample_derivative": "Germline DNA",
                        "sample_derivative_volume": 3.0,
                        "sample_derivative_volume_units": "Microliters",
                        "sample_derivative_concentration": 3.0,
                        "sample_derivative_concentration_units": "Micrograms per Microliter",
                        "din": 8.0,
                        "a260_a280": 1.9,
                        "a260_a230": 2.2,
                        "material_used": 1.0,
                        "material_used_units": "Milliliters",
                        "material_remaining": 0.0,
                        "material_remaining_units": "Milliliters",
                        "material_storage_condition": "(-20)oC",
                        "quality_of_sample": "Pass",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Not Reported",
                    },
                ],
                "cohort_name": "Arm_A",
                "participant_id": "TTTPP303",
                "cimac_participant_id": "CTTTP03",
            },
            {
                "samples": [
                    {
                        "shipping_entry_number": 4,
                        "collection_event_name": "Baseline",
                        "parent_sample_id": "TRIALGROUP 4",
                        "processed_sample_id": "BIOBANK 1",
                        "cimac_id": "CTTTP04N1.00",
                        "box_number": "1",
                        "sample_location": "A4",
                        "type_of_sample": "Skin Tissue",
                        "sample_collection_procedure": "Core Biopsy",
                        "core_number": 1.0,
                        "fixation_stabilization_type": "Formalin-Fixed Paraffin-Embedded",
                        "processed_sample_type": "Fixed Slide",
                        "processed_sample_quantity": 1.0,
                        "processed_sample_derivative": "Germline DNA",
                        "sample_derivative_volume": 4.0,
                        "sample_derivative_volume_units": "Milliliters",
                        "sample_derivative_concentration": 4.0,
                        "sample_derivative_concentration_units": "Cells per Vial",
                        "din": 9.0,
                        "a260_a280": 1.8,
                        "a260_a230": 2.0,
                        "material_used": 1.0,
                        "material_used_units": "Milliliters",
                        "material_remaining": 0.0,
                        "material_remaining_units": "Milliliters",
                        "material_storage_condition": "(-20)oC",
                        "quality_of_sample": "Pass",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Not Reported",
                    },
                    {
                        "shipping_entry_number": 5,
                        "collection_event_name": "Baseline",
                        "parent_sample_id": "TRIALGROUP 5",
                        "processed_sample_id": "BIOBANK 1",
                        "cimac_id": "CTTTP04N2.00",
                        "box_number": "1",
                        "sample_location": "A5",
                        "type_of_sample": "Skin Tissue",
                        "sample_collection_procedure": "Core Biopsy",
                        "core_number": 1.0,
                        "fixation_stabilization_type": "Optimum cutting temperature medium",
                        "processed_sample_type": "Fixed Slide",
                        "processed_sample_quantity": 1.0,
                        "processed_sample_derivative": "Germline DNA",
                        "sample_derivative_volume": 5.0,
                        "sample_derivative_volume_units": "Not Reported",
                        "sample_derivative_concentration": 5.0,
                        "sample_derivative_concentration_units": "Not Reported",
                        "din": 9.0,
                        "a260_a280": 1.7,
                        "a260_a230": 2.1,
                        "material_used": 1.0,
                        "material_used_units": "Milliliters",
                        "material_remaining": 0.0,
                        "material_remaining_units": "Milliliters",
                        "material_storage_condition": "(-20)oC",
                        "quality_of_sample": "Pass",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Not Reported",
                    },
                    {
                        "shipping_entry_number": 6,
                        "collection_event_name": "Baseline",
                        "parent_sample_id": "TRIALGROUP 6",
                        "processed_sample_id": "BIOBANK 1",
                        "cimac_id": "CTTTP04N3.00",
                        "box_number": "1",
                        "sample_location": "A6",
                        "type_of_sample": "Skin Tissue",
                        "sample_collection_procedure": "Core Biopsy",
                        "core_number": 1.0,
                        "fixation_stabilization_type": "Thaw-Lyse",
                        "processed_sample_type": "Fixed Slide",
                        "processed_sample_quantity": 2.0,
                        "processed_sample_derivative": "Germline DNA",
                        "sample_derivative_volume": 6.0,
                        "sample_derivative_volume_units": "Other",
                        "sample_derivative_concentration": 6.0,
                        "sample_derivative_concentration_units": "Other",
                        "din": 8.0,
                        "a260_a280": 1.9,
                        "a260_a230": 2.2,
                        "material_used": 1.0,
                        "material_used_units": "Milliliters",
                        "material_remaining": 0.0,
                        "material_remaining_units": "Milliliters",
                        "material_storage_condition": "(-20)oC",
                        "quality_of_sample": "Pass",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Not Reported",
                    },
                ],
                "cohort_name": "Arm_Z",
                "participant_id": "TTTPP403",
                "cimac_participant_id": "CTTTP04",
            },
        ],
    }
    upload_entries = []
    base_trial = get_test_trial(
        allowed_cohort_names=["Arm_A", "Arm_Z"],
        allowed_collection_event_names=["Baseline", "Pre_Day_1_Cycle_2"],
    )

    target_trial = copy_dict_with_branch(
        base_trial, prismify_patch, ["participants", "shipments"]
    )

    return PrismTestData(
        upload_type,
        prismify_args,
        prismify_patch,
        upload_entries,
        base_trial,
        target_trial,
    )


@manifest_data_generator
def tumor_tissue_dna() -> PrismTestData:
    upload_type = "tumor_tissue_dna"
    prismify_args = get_prismify_args(upload_type)
    prismify_patch = {
        "protocol_identifier": "test_prism_trial_id",
        "shipments": [
            {
                "manifest_id": "test_prism_trial_id_DNA",
                "assay_priority": "4",
                "assay_type": "WES",
                "receiving_party": "MDA_Wistuba",
                "courier": "USPS",
                "tracking_number": "TrackN",
                "account_number": "AccN",
                "shipping_condition": "Ice_Pack",
                "date_shipped": "2001-10-10 00:00:00",
                "date_received": "2002-10-10 00:00:00",
                "quality_of_shipment": "Specimen shipment received in good condition",
                "ship_from": "ship from",
                "ship_to": "ship to",
            }
        ],
        "participants": [
            {
                "samples": [
                    {
                        "shipping_entry_number": 1,
                        "collection_event_name": "Baseline",
                        "parent_sample_id": "TRIALGROUP 1",
                        "processed_sample_id": "BIOBANK 1",
                        "cimac_id": "CTTTP05T1.00",
                        "box_number": "1",
                        "sample_location": "A1",
                        "type_of_sample": "Skin Tissue",
                        "type_of_tumor_sample": "Metastatic Tumor",
                        "sample_collection_procedure": "Core Biopsy",
                        "core_number": 1.0,
                        "fixation_stabilization_type": "Archival FFPE",
                        "processed_sample_type": "Fixed Slide",
                        "processed_sample_quantity": 1.0,
                        "processed_sample_derivative": "Germline DNA",
                        "sample_derivative_volume": 1.0,
                        "sample_derivative_volume_units": "Microliters",
                        "sample_derivative_concentration": 1.0,
                        "sample_derivative_concentration_units": "Nanogram per Microliter",
                        "din": 9.0,
                        "a260_a280": 1.8,
                        "a260_a230": 2.0,
                        "material_used": 1.0,
                        "material_used_units": "Milliliters",
                        "material_remaining": 0.0,
                        "material_remaining_units": "Milliliters",
                        "material_storage_condition": "(-20)oC",
                        "quality_of_sample": "Pass",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Not Reported",
                        "comments": "Comment",
                    },
                    {
                        "shipping_entry_number": 2,
                        "collection_event_name": "Baseline",
                        "parent_sample_id": "TRIALGROUP 2",
                        "processed_sample_id": "BIOBANK 1",
                        "cimac_id": "CTTTP05T2.00",
                        "box_number": "1",
                        "sample_location": "A2",
                        "type_of_sample": "Skin Tissue",
                        "type_of_tumor_sample": "Metastatic Tumor",
                        "sample_collection_procedure": "Core Biopsy",
                        "core_number": 1.0,
                        "fixation_stabilization_type": "Fresh Specimen",
                        "processed_sample_type": "Fixed Slide",
                        "processed_sample_quantity": 1.0,
                        "processed_sample_derivative": "Germline DNA",
                        "sample_derivative_volume": 2.0,
                        "sample_derivative_volume_units": "Milliliters",
                        "sample_derivative_concentration": 2.0,
                        "sample_derivative_concentration_units": "Milligram per Milliliter",
                        "din": 9.0,
                        "a260_a280": 1.7,
                        "a260_a230": 2.1,
                        "material_used": 1.0,
                        "material_used_units": "Milliliters",
                        "material_remaining": 0.0,
                        "material_remaining_units": "Milliliters",
                        "material_storage_condition": "(-20)oC",
                        "quality_of_sample": "Pass",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Not Reported",
                        "comments": "No comment",
                    },
                    {
                        "shipping_entry_number": 3,
                        "collection_event_name": "Baseline",
                        "parent_sample_id": "TRIALGROUP 3",
                        "processed_sample_id": "BIOBANK 1",
                        "cimac_id": "CTTTP05T3.00",
                        "box_number": "1",
                        "sample_location": "A3",
                        "type_of_sample": "Skin Tissue",
                        "type_of_tumor_sample": "Primary Tumor",
                        "sample_collection_procedure": "Core Biopsy",
                        "core_number": 1.0,
                        "fixation_stabilization_type": "Frozen Specimen",
                        "processed_sample_type": "Fixed Slide",
                        "processed_sample_quantity": 2.0,
                        "processed_sample_derivative": "Germline DNA",
                        "sample_derivative_volume": 3.0,
                        "sample_derivative_volume_units": "Microliters",
                        "sample_derivative_concentration": 3.0,
                        "sample_derivative_concentration_units": "Micrograms per Microliter",
                        "din": 8.0,
                        "a260_a280": 1.9,
                        "a260_a230": 2.2,
                        "material_used": 1.0,
                        "material_used_units": "Milliliters",
                        "material_remaining": 0.0,
                        "material_remaining_units": "Milliliters",
                        "material_storage_condition": "(-20)oC",
                        "quality_of_sample": "Pass",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Not Reported",
                    },
                ],
                "cohort_name": "Arm_A",
                "participant_id": "TTTPP503",
                "cimac_participant_id": "CTTTP05",
            },
            {
                "samples": [
                    {
                        "shipping_entry_number": 4,
                        "collection_event_name": "Baseline",
                        "parent_sample_id": "TRIALGROUP 4",
                        "processed_sample_id": "BIOBANK 1",
                        "cimac_id": "CTTTP06T1.00",
                        "box_number": "1",
                        "sample_location": "A4",
                        "type_of_sample": "Skin Tissue",
                        "type_of_tumor_sample": "Primary Tumor",
                        "sample_collection_procedure": "Core Biopsy",
                        "core_number": 1.0,
                        "fixation_stabilization_type": "Formalin-Fixed Paraffin-Embedded",
                        "processed_sample_type": "Fixed Slide",
                        "processed_sample_quantity": 1.0,
                        "processed_sample_derivative": "Germline DNA",
                        "sample_derivative_volume": 4.0,
                        "sample_derivative_volume_units": "Milliliters",
                        "sample_derivative_concentration": 4.0,
                        "sample_derivative_concentration_units": "Cells per Vial",
                        "din": 9.0,
                        "a260_a280": 1.8,
                        "a260_a230": 2.0,
                        "material_used": 1.0,
                        "material_used_units": "Milliliters",
                        "material_remaining": 0.0,
                        "material_remaining_units": "Milliliters",
                        "material_storage_condition": "(-20)oC",
                        "quality_of_sample": "Pass",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Not Reported",
                    },
                    {
                        "shipping_entry_number": 5,
                        "collection_event_name": "Baseline",
                        "parent_sample_id": "TRIALGROUP 5",
                        "processed_sample_id": "BIOBANK 1",
                        "cimac_id": "CTTTP06T2.00",
                        "box_number": "1",
                        "sample_location": "A5",
                        "type_of_sample": "Skin Tissue",
                        "type_of_tumor_sample": "Primary Tumor",
                        "sample_collection_procedure": "Core Biopsy",
                        "core_number": 1.0,
                        "fixation_stabilization_type": "Optimum cutting temperature medium",
                        "processed_sample_type": "Fixed Slide",
                        "processed_sample_quantity": 1.0,
                        "processed_sample_derivative": "Germline DNA",
                        "sample_derivative_volume": 5.0,
                        "sample_derivative_volume_units": "Not Reported",
                        "sample_derivative_concentration": 5.0,
                        "sample_derivative_concentration_units": "Not Reported",
                        "din": 9.0,
                        "a260_a280": 1.7,
                        "a260_a230": 2.1,
                        "material_used": 1.0,
                        "material_used_units": "Milliliters",
                        "material_remaining": 0.0,
                        "material_remaining_units": "Milliliters",
                        "material_storage_condition": "(-20)oC",
                        "quality_of_sample": "Pass",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Not Reported",
                    },
                    {
                        "shipping_entry_number": 6,
                        "collection_event_name": "Baseline",
                        "parent_sample_id": "TRIALGROUP 6",
                        "processed_sample_id": "BIOBANK 1",
                        "cimac_id": "CTTTP06T3.00",
                        "box_number": "1",
                        "sample_location": "A6",
                        "type_of_sample": "Skin Tissue",
                        "type_of_tumor_sample": "Primary Tumor",
                        "sample_collection_procedure": "Core Biopsy",
                        "core_number": 1.0,
                        "fixation_stabilization_type": "Thaw-Lyse",
                        "processed_sample_type": "Fixed Slide",
                        "processed_sample_quantity": 2.0,
                        "processed_sample_derivative": "Germline DNA",
                        "sample_derivative_volume": 6.0,
                        "sample_derivative_volume_units": "Other",
                        "sample_derivative_concentration": 6.0,
                        "sample_derivative_concentration_units": "Other",
                        "din": 8.0,
                        "a260_a280": 1.9,
                        "a260_a230": 2.2,
                        "material_used": 1.0,
                        "material_used_units": "Milliliters",
                        "material_remaining": 0.0,
                        "material_remaining_units": "Milliliters",
                        "material_storage_condition": "(-20)oC",
                        "quality_of_sample": "Pass",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Not Reported",
                    },
                ],
                "cohort_name": "Arm_Z",
                "participant_id": "TTTPP603",
                "cimac_participant_id": "CTTTP06",
            },
        ],
    }
    upload_entries = []
    base_trial = get_test_trial(
        allowed_cohort_names=["Arm_A", "Arm_Z"],
        allowed_collection_event_names=["Baseline", "Pre_Day_1_Cycle_2"],
    )

    target_trial = copy_dict_with_branch(
        base_trial, prismify_patch, ["participants", "shipments"]
    )

    return PrismTestData(
        upload_type,
        prismify_args,
        prismify_patch,
        upload_entries,
        base_trial,
        target_trial,
    )


@manifest_data_generator
def h_and_e() -> PrismTestData:
    upload_type = "h_and_e"
    prismify_args = get_prismify_args(upload_type)
    prismify_patch = {
        "protocol_identifier": "test_prism_trial_id",
        "shipments": [
            {
                "manifest_id": "test_prism_trial_id_slide",
                "assay_priority": "3",
                "assay_type": "IHC",
                "receiving_party": "Broad_Cibulskis",
                "courier": "USPS",
                "tracking_number": "TrackN",
                "account_number": "AccN",
                "shipping_condition": "Not Reported",
                "date_shipped": "2001-10-10 00:00:00",
                "date_received": "2002-10-10 00:00:00",
                "quality_of_shipment": "Specimen shipment received in poor condition",
                "ship_from": "ship from",
                "ship_to": "ship to",
            }
        ],
        "participants": [
            {
                "samples": [
                    {
                        "shipping_entry_number": 1,
                        "collection_event_name": "Baseline",
                        "parent_sample_id": "TRIALGROUP 1",
                        "processed_sample_id": "BIOBANK 1",
                        "cimac_id": "CTTTP08T1.00",
                        "box_number": "2",
                        "sample_location": "A1",
                        "type_of_sample": "Tumor Tissue",
                        "type_of_tumor_sample": "Primary Tumor",
                        "sample_collection_procedure": "Core Biopsy",
                        "core_number": 1.0,
                        "fixation_stabilization_type": "Formalin-Fixed Paraffin-Embedded",
                        "processed_sample_type": "H&E-Stained Fixed Tissue Slide Specimen",
                        "processed_sample_quantity": 1.0,
                        "tumor_tissue_total_area_percentage": 0.0,
                        "viable_tumor_area_percentage": 0.0,
                        "viable_stroma_area_percentage": 0.0,
                        "necrosis_area_percentage": 0.0,
                        "fibrosis_area_percentage": 0.0,
                        "material_used": 1.0,
                        "material_used_units": "Slides",
                        "material_remaining": 1.0,
                        "material_remaining_units": "Slides",
                        "material_storage_condition": "4oC",
                        "quality_of_sample": "Pass",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Not Reported",
                    },
                    {
                        "shipping_entry_number": 2,
                        "collection_event_name": "Pre_Day_1_Cycle_2",
                        "parent_sample_id": "TRIALGROUP 2",
                        "processed_sample_id": "BIOBANK 1",
                        "cimac_id": "CTTTP08T2.00",
                        "box_number": "2",
                        "sample_location": "A2",
                        "type_of_sample": "Tumor Tissue",
                        "type_of_tumor_sample": "Primary Tumor",
                        "sample_collection_procedure": "Core Biopsy",
                        "core_number": 1.0,
                        "fixation_stabilization_type": "Formalin-Fixed Paraffin-Embedded",
                        "processed_sample_type": "H&E-Stained Fixed Tissue Slide Specimen",
                        "processed_sample_quantity": 1.0,
                        "tumor_tissue_total_area_percentage": 100.0,
                        "viable_tumor_area_percentage": 100.0,
                        "viable_stroma_area_percentage": 100.0,
                        "necrosis_area_percentage": 100.0,
                        "fibrosis_area_percentage": 100.0,
                        "material_used": 1.0,
                        "material_used_units": "Slides",
                        "material_remaining": 1.0,
                        "material_remaining_units": "Slides",
                        "material_storage_condition": "4oC",
                        "quality_of_sample": "Pass",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Not Reported",
                    },
                ],
                "cohort_name": "Arm_A",
                "participant_id": "TTTPP802",
                "cimac_participant_id": "CTTTP08",
            }
        ],
    }
    upload_entries = []
    base_trial = get_test_trial(
        allowed_cohort_names=["Arm_A", "Arm_Z"],
        allowed_collection_event_names=["Baseline", "Pre_Day_1_Cycle_2"],
    )

    target_trial = copy_dict_with_branch(
        base_trial, prismify_patch, ["participants", "shipments"]
    )

    return PrismTestData(
        upload_type,
        prismify_args,
        prismify_patch,
        upload_entries,
        base_trial,
        target_trial,
    )


@manifest_data_generator
def tumor_tissue_rna() -> PrismTestData:
    upload_type = "tumor_tissue_rna"
    prismify_args = get_prismify_args(upload_type)
    prismify_patch = {
        "protocol_identifier": "test_prism_trial_id",
        "shipments": [
            {
                "manifest_id": "test_prism_trial_tumor_tissue_RNA",
                "assay_priority": "4",
                "assay_type": "RNAseq",
                "receiving_party": "MDA_Wistuba",
                "courier": "USPS",
                "tracking_number": "TrackN",
                "account_number": "AccN",
                "shipping_condition": "Ice_Pack",
                "date_shipped": "2001-10-10 00:00:00",
                "date_received": "2002-10-10 00:00:00",
                "quality_of_shipment": "Specimen shipment received in good condition",
                "ship_from": "ship from",
                "ship_to": "ship to",
            }
        ],
        "participants": [
            {
                "samples": [
                    {
                        "shipping_entry_number": 1,
                        "collection_event_name": "Baseline",
                        "parent_sample_id": "TRIALGROUP 1",
                        "processed_sample_id": "BIOBANK 1",
                        "cimac_id": "CTTTP05T1.00",
                        "box_number": "1",
                        "sample_location": "A1",
                        "type_of_sample": "Skin Tissue",
                        "type_of_tumor_sample": "Metastatic Tumor",
                        "sample_collection_procedure": "Core Biopsy",
                        "core_number": 1.0,
                        "fixation_stabilization_type": "Archival FFPE",
                        "processed_sample_type": "Fixed Slide",
                        "processed_sample_quantity": 1.0,
                        "processed_sample_derivative": "Tumor RNA",
                        "sample_derivative_volume": 1.0,
                        "sample_derivative_volume_units": "Microliters",
                        "sample_derivative_concentration": 1.0,
                        "sample_derivative_concentration_units": "Nanogram per Microliter",
                        "material_used": 1.0,
                        "material_used_units": "Milliliters",
                        "material_remaining": 0.0,
                        "material_remaining_units": "Milliliters",
                        "material_storage_condition": "(-20)oC",
                        "quality_of_sample": "Pass",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Not Reported",
                        "comments": "Comment",
                    },
                    {
                        "shipping_entry_number": 2,
                        "collection_event_name": "Baseline",
                        "parent_sample_id": "TRIALGROUP 2",
                        "processed_sample_id": "BIOBANK 2",
                        "cimac_id": "CTTTP05T2.00",
                        "box_number": "1",
                        "sample_location": "A2",
                        "type_of_sample": "Skin Tissue",
                        "type_of_tumor_sample": "Metastatic Tumor",
                        "sample_collection_procedure": "Core Biopsy",
                        "core_number": 1.0,
                        "fixation_stabilization_type": "Fresh Specimen",
                        "processed_sample_type": "Fixed Slide",
                        "processed_sample_quantity": 1.0,
                        "processed_sample_derivative": "Tumor RNA",
                        "sample_derivative_volume": 2.0,
                        "sample_derivative_volume_units": "Milliliters",
                        "sample_derivative_concentration": 2.0,
                        "sample_derivative_concentration_units": "Milligram per Milliliter",
                        "material_used": 1.0,
                        "material_used_units": "Milliliters",
                        "material_remaining": 0.0,
                        "material_remaining_units": "Milliliters",
                        "material_storage_condition": "(-20)oC",
                        "quality_of_sample": "Pass",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Not Reported",
                        "comments": "No comment",
                    },
                    {
                        "shipping_entry_number": 3,
                        "collection_event_name": "Baseline",
                        "parent_sample_id": "TRIALGROUP 3",
                        "processed_sample_id": "BIOBANK 3",
                        "cimac_id": "CTTTP05T3.00",
                        "box_number": "1",
                        "sample_location": "A3",
                        "type_of_sample": "Skin Tissue",
                        "type_of_tumor_sample": "Primary Tumor",
                        "sample_collection_procedure": "Core Biopsy",
                        "core_number": 1.0,
                        "fixation_stabilization_type": "Frozen Specimen",
                        "processed_sample_type": "Fixed Slide",
                        "processed_sample_quantity": 2.0,
                        "processed_sample_derivative": "Tumor RNA",
                        "sample_derivative_volume": 3.0,
                        "sample_derivative_volume_units": "Microliters",
                        "sample_derivative_concentration": 3.0,
                        "sample_derivative_concentration_units": "Micrograms per Microliter",
                        "material_used": 1.0,
                        "material_used_units": "Milliliters",
                        "material_remaining": 0.0,
                        "material_remaining_units": "Milliliters",
                        "material_storage_condition": "(-20)oC",
                        "quality_of_sample": "Pass",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Not Reported",
                    },
                ],
                "cohort_name": "Arm_A",
                "participant_id": "TTTPP503",
                "cimac_participant_id": "CTTTP05",
            },
            {
                "samples": [
                    {
                        "shipping_entry_number": 4,
                        "collection_event_name": "Baseline",
                        "parent_sample_id": "TRIALGROUP 4",
                        "processed_sample_id": "BIOBANK 4",
                        "cimac_id": "CTTTP06T1.00",
                        "box_number": "1",
                        "sample_location": "A4",
                        "type_of_sample": "Skin Tissue",
                        "type_of_tumor_sample": "Primary Tumor",
                        "sample_collection_procedure": "Core Biopsy",
                        "core_number": 1.0,
                        "fixation_stabilization_type": "Formalin-Fixed Paraffin-Embedded",
                        "processed_sample_type": "Fixed Slide",
                        "processed_sample_quantity": 1.0,
                        "processed_sample_derivative": "Tumor RNA",
                        "sample_derivative_volume": 4.0,
                        "sample_derivative_volume_units": "Milliliters",
                        "sample_derivative_concentration": 4.0,
                        "sample_derivative_concentration_units": "Cells per Vial",
                        "material_used": 1.0,
                        "material_used_units": "Milliliters",
                        "material_remaining": 0.0,
                        "material_remaining_units": "Milliliters",
                        "material_storage_condition": "(-20)oC",
                        "quality_of_sample": "Pass",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Not Reported",
                    },
                    {
                        "shipping_entry_number": 5,
                        "collection_event_name": "Baseline",
                        "parent_sample_id": "TRIALGROUP 5",
                        "processed_sample_id": "BIOBANK 5",
                        "cimac_id": "CTTTP06T2.00",
                        "box_number": "1",
                        "sample_location": "A5",
                        "type_of_sample": "Skin Tissue",
                        "type_of_tumor_sample": "Primary Tumor",
                        "sample_collection_procedure": "Core Biopsy",
                        "core_number": 1.0,
                        "fixation_stabilization_type": "Optimum cutting temperature medium",
                        "processed_sample_type": "Fixed Slide",
                        "processed_sample_quantity": 1.0,
                        "processed_sample_derivative": "Tumor RNA",
                        "sample_derivative_volume": 5.0,
                        "sample_derivative_volume_units": "Not Reported",
                        "sample_derivative_concentration": 5.0,
                        "sample_derivative_concentration_units": "Not Reported",
                        "material_used": 1.0,
                        "material_used_units": "Milliliters",
                        "material_remaining": 0.0,
                        "material_remaining_units": "Milliliters",
                        "material_storage_condition": "(-20)oC",
                        "quality_of_sample": "Pass",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Not Reported",
                    },
                    {
                        "shipping_entry_number": 6,
                        "collection_event_name": "Baseline",
                        "parent_sample_id": "TRIALGROUP 6",
                        "processed_sample_id": "BIOBANK 6",
                        "cimac_id": "CTTTP06T3.00",
                        "box_number": "1",
                        "sample_location": "A6",
                        "type_of_sample": "Skin Tissue",
                        "type_of_tumor_sample": "Primary Tumor",
                        "sample_collection_procedure": "Core Biopsy",
                        "core_number": 1.0,
                        "fixation_stabilization_type": "Thaw-Lyse",
                        "processed_sample_type": "Fixed Slide",
                        "processed_sample_quantity": 2.0,
                        "processed_sample_derivative": "Tumor RNA",
                        "sample_derivative_volume": 6.0,
                        "sample_derivative_volume_units": "Other",
                        "sample_derivative_concentration": 6.0,
                        "sample_derivative_concentration_units": "Other",
                        "material_used": 1.0,
                        "material_used_units": "Milliliters",
                        "material_remaining": 0.0,
                        "material_remaining_units": "Milliliters",
                        "material_storage_condition": "(-20)oC",
                        "quality_of_sample": "Pass",
                        "sample_replacement": "Replacement Not Requested",
                        "residual_sample_use": "Not Reported",
                    },
                ],
                "cohort_name": "Arm_Z",
                "participant_id": "TTTPP603",
                "cimac_participant_id": "CTTTP06",
            },
        ],
    }
    upload_entries = []
    base_trial = get_test_trial(
        allowed_cohort_names=["Arm_A", "Arm_Z"],
        allowed_collection_event_names=["Baseline", "Pre_Day_1_Cycle_2"],
    )

    target_trial = copy_dict_with_branch(
        base_trial, prismify_patch, ["participants", "shipments"]
    )

    return PrismTestData(
        upload_type,
        prismify_args,
        prismify_patch,
        upload_entries,
        base_trial,
        target_trial,
    )


@manifest_data_generator
def tumor_normal_pairing():
    upload_type = "tumor_normal_pairing"
    prismify_args = get_prismify_args(upload_type)
    prismify_patch = {
        "protocol_identifier": "test_prism_trial_id",
        "analysis": {
            "wes_analysis": {
                "pair_runs": [
                    {
                        "tumor": {"cimac_id": "CTTTPP111.00"},
                        "run_id": "CTTTPP111.00",
                        "normal": {"cimac_id": "CTTTPP121.00"},
                    }
                ]
            }
        },
    }
    upload_entries = []

    cimac_ids = [
        sample["cimac_id"]
        for pair_run in prismify_patch["analysis"]["wes_analysis"]["pair_runs"]
        for sample in [pair_run["tumor"], pair_run["normal"]]
    ]
    base_trial = get_test_trial(cimac_ids)

    target_trial = copy_dict_with_branch(base_trial, prismify_patch, "analysis")

    return PrismTestData(
        upload_type,
        prismify_args,
        prismify_patch,
        upload_entries,
        base_trial,
        target_trial,
    )


missing = set(SUPPORTED_MANIFESTS).difference(
    [f.__name__ for f in manifest_data_generators]
)
assert not missing, f"Missing manifest test data generators for {missing}"
