"""patientfields client for Open Dental SDK."""

from typing import List, Optional, Union
from ...base.resource import BaseResource
from .models import (
    PatientField,
    CreatePatientFieldRequest,
    UpdatePatientFieldRequest,
    PatientFieldListResponse,
    PatientFieldSearchRequest
)


class PatientFieldsClient(BaseResource):
    """Client for managing patient fields in Open Dental."""
    
    def __init__(self, client):
        """Initialize the patient fields client."""
        super().__init__(client, "patient_fields")
    
    def get(self, item_id: Union[int, str]) -> PatientField:
        """Get a patient field by ID."""
        item_id = self._validate_id(item_id)
        endpoint = self._build_endpoint(item_id)
        response = self._get(endpoint)
        return self._handle_response(response, PatientField)
    
    def list(self, page: int = 1, per_page: int = 50) -> PatientFieldListResponse:
        """List all patient fields."""
        params = {"page": page, "per_page": per_page}
        endpoint = self._build_endpoint()
        response = self._get(endpoint, params=params)
        
        if isinstance(response, dict):
            return PatientFieldListResponse(**response)
        elif isinstance(response, list):
            return PatientFieldListResponse(
                patient_fields=[PatientField(**item) for item in response],
                total=len(response), page=page, per_page=per_page
            )
        return PatientFieldListResponse(patient_fields=[], total=0, page=page, per_page=per_page)
    
    def create(self, item_data: CreatePatientFieldRequest) -> PatientField:
        """Create a new patient field."""
        endpoint = self._build_endpoint()
        data = item_data.model_dump()
        response = self._post(endpoint, json_data=data)
        return self._handle_response(response, PatientField)
    
    def update(self, item_id: Union[int, str], item_data: UpdatePatientFieldRequest) -> PatientField:
        """Update an existing patient field."""
        item_id = self._validate_id(item_id)
        endpoint = self._build_endpoint(item_id)
        data = item_data.model_dump()
        response = self._put(endpoint, json_data=data)
        return self._handle_response(response, PatientField)
    
    def delete(self, item_id: Union[int, str]) -> bool:
        """Delete a patient field."""
        item_id = self._validate_id(item_id)
        endpoint = self._build_endpoint(item_id)
        response = self._delete(endpoint)
        return response is None or response.get("success", True)
    
    def search(self, search_params: PatientFieldSearchRequest) -> PatientFieldListResponse:
        """Search for patient fields."""
        endpoint = self._build_endpoint("search")
        params = search_params.model_dump()
        response = self._get(endpoint, params=params)
        
        if isinstance(response, dict):
            return PatientFieldListResponse(**response)
        elif isinstance(response, list):
            return PatientFieldListResponse(
                patient_fields=[PatientField(**item) for item in response],
                total=len(response), page=search_params.page, per_page=search_params.per_page
            )
        return PatientFieldListResponse(
            patient_fields=[], total=0, page=search_params.page, per_page=search_params.per_page
        )
