# Copyright (c) 2025, DjaoDjin inc.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# 1. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
# "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
# TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
# PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
# CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
# EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
# PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
# OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
# WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
# OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
# ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
from __future__ import unicode_literals

from datetime import datetime, timedelta
import re

import markdown
from django import template
from django.shortcuts import get_object_or_404
from django.template.defaultfilters import stringfilter
from django.utils.safestring import mark_safe
from django.utils import formats
from django.utils.dateparse import parse_date, parse_datetime

from .. import settings
from ..compat import gettext_lazy as _, is_authenticated, six, timezone_or_utc
from ..decorators import fail_direct, _valid_manager
from ..humanize import as_money, as_percentage
from ..mixins import as_html_description
from ..models import Plan, Price, Subscription, get_broker
from ..utils import get_organization_model

register = template.Library()

@register.filter
def active_with_provider(organization, provider):
    """
    Returns a list of active subscriptions for organization for which provider
    is the owner of the plan.
    """
    return Subscription.objects.active_with(provider).filter(
        organization=organization)


@register.filter()
def date_in_future(value, arg=None):
    if value:
        if arg:
            base = arg
        else:
            base = datetime.utcnow().replace(tzinfo=timezone_or_utc())
        if isinstance(value, six.integer_types):
            value = datetime.fromtimestamp(value).replace(
                tzinfo=timezone_or_utc())
        if value > base:
            return True
    return False


@register.filter(needs_autoescape=False)
def describe(transaction):
    try:
        descr = transaction['description']  # Serializers have already run
                                            # `as_html_description`.
    except (TypeError, KeyError):
        # We would need the request here but we don;t have it
        # so we set `force_links` to `True` in order to generate
        # URL paths anyway.
        descr = as_html_description(transaction, force_links=True)
    return mark_safe(descr)


# `describe_no_links` is used on the checkout page to show plans
# that will be subscribed to.
@register.filter(needs_autoescape=False)
def describe_no_links(transaction):
    return mark_safe(as_html_description(transaction))


@register.filter()
def price(item):
    try:
        return item.price
    except AttributeError:
        pass
    return Price(item['amount'], item['unit'])


@register.filter()
def dest_price(item):
    try:
        return item.dest_price
    except AttributeError:
        pass
    return Price(item['dest_amount'], item['dest_unit'])


@register.filter()
def orig_price(item):
    try:
        return item.orig_price
    except AttributeError:
        pass
    return Price(item['orig_amount'], item['orig_unit'])


@register.filter()
def isoformat(val):
    if isinstance(val, datetime):
        return val.isoformat()
    return val


@register.filter()
def iteritems(val):
    return six.iteritems(val)


@register.filter()
def htmlize_money(amount_unit_tuple):
    text = as_money(amount_unit_tuple.amount, amount_unit_tuple.unit)
    look = re.match(r'(\$|€|£)?(\d(\d|,)*)(\.\d+)?(.*)', text)
    if look:
        unit_prefix = '<span class="unit-prefix">%s</span>' % look.group(1)
        int_amount = look.group(2)
        frac_amount = look.group(4)
        unit_suffx = '<span class="unit-suffix">%s</span>' % look.group(5)
        if frac_amount == '.00':
            frac_amount = (
                '<span class="frac-digits zero-frac-digits">%s</span>'
                % frac_amount)
        else:
            frac_amount = ('<span class="frac-digits">%s</span>' % frac_amount)
        html = unit_prefix + int_amount + frac_amount + unit_suffx
        return  mark_safe(html)
    return text


@register.filter()
def humanize_money(amount_unit_tuple):
    return as_money(amount_unit_tuple.amount, amount_unit_tuple.unit)

@register.filter()
def humanize_percent(percentage):
    return as_percentage(percentage)

@register.filter()
def humanize_period(period):
    result = _("per ?")
    if period == Plan.INTERVAL_CHOICES[0][0]:
        result = _("per hour")
    elif period == Plan.INTERVAL_CHOICES[1][0]:
        result = _("per day")
    elif period == Plan.INTERVAL_CHOICES[2][0]:
        result = _("per week")
    elif period == Plan.INTERVAL_CHOICES[3][0]:
        result = _("per month")
    elif period == Plan.INTERVAL_CHOICES[4][0]:
        result = _("per year")
    return result


@register.filter()
def is_debit(transaction, organization):
    """
    True if the transaction can be tagged as a debit. That is
    it is either payable by the organization or the transaction
    moves from a Funds account to the organization's Expenses account.
    """
    return transaction.is_debit(organization)


@register.filter()
def is_incomplete_month(date):
    return ((isinstance(date, six.string_types) and not date.endswith('01'))
        or (isinstance(date, datetime) and date.day != 1))


@register.filter
def is_direct(request, organization=None):
    if organization is None:
        organization = get_broker()
    return not fail_direct(request, profile=organization)


@register.filter
def is_manager(request, organization):
    organization_model = get_organization_model()
    if not isinstance(organization, organization_model):
        organization = get_object_or_404(organization_model, slug=organization)
    return _valid_manager(request.user if is_authenticated(request) else None,
        [organization])


@register.filter(needs_autoescape=False)
@stringfilter
def md(text): #pylint: disable=invalid-name
    # XXX safe_mode is deprecated. Should we use bleach? As shown in example:
    # https://pythonhosted.org/Markdown/reference.html#markdown
    return mark_safe(markdown.markdown(text, enable_attributes=False))


@register.filter()
def monthly_caption(last_date):
    """returns a formatted caption describing the period whose end
    date is *last_date*."""
    if last_date.day == 1:
        prev = last_date - timedelta(days=2) # more than one day to make sure
        return datetime.strftime(prev, "%b'%y")
    return datetime.strftime(last_date, "%b'%y") + "*"


@register.filter()
def attached_organization(user):
    """
    Returns the person ``Organization`` associated to the user or None
    in none can be reliably found.
    """
    return get_organization_model().objects.attached(user)


@register.filter()
def products(subscriptions):
    """
    Returns a list of distinct providers (i.e. ``Organization``) from
    the plans the *organization* is subscribed to.
    """
    if subscriptions:
        # We don't use QuerySet.distinct('organization') because SQLite
        # does not support DISTINCT ON queries.
        return subscriptions.values(
            'organization__slug', 'organization__full_name').distinct()
    return []


@register.filter()
def humanize_short_date(dtime_at):
    as_datetime = dtime_at
    if isinstance(dtime_at, six.string_types):
        as_datetime = parse_datetime(dtime_at)
        if not as_datetime:
            as_date = parse_date(dtime_at)
            if as_date:
                as_datetime = datetime.datetime.combine(
                    as_date, datetime.time.min)
    if isinstance(as_datetime, datetime):
        return formats.date_format(
            as_datetime, format=formats.get_format("SHORT_DATE_FORMAT"))
    return as_datetime


@register.filter()
def privacy_settings(request):
    # Note: "Starting in Firefox version 135, the “Do Not Track” setting
    # has been removed."
    # https://support.mozilla.org/en-US/kb/how-do-i-turn-do-not-track-feature
    do_not_track = bool(request.META.get('HTTP_DNT') == '1')
    gpc = bool(request.META.get('HTTP_SEC_GPC') == '1')
    if do_not_track or gpc:
        privacy_settings = {
            key:False for key in settings.PRIVACY_COOKIES_ENABLED}
    else:
        privacy_settings = request.session.get('privacy', {}) if request else {}
    return privacy_settings
