"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Normalize the given interval set to cover the complete number line and make sure it has at most one gap
 */
function normalizeIntervals(intervals, changesAreAbsolute) {
    // Make intervals a complete numberline
    const full = orderAndCompleteIntervals(intervals);
    // Add 'undefined's in uncovered areas of the number line
    makeGapsUndefined(full);
    // In case of relative changes, treat 0-change also as 'undefined' (= no change action)
    if (!changesAreAbsolute) {
        makeZerosUndefined(full);
    }
    // Combine adjacent undefines and make sure there's at most one of them
    combineUndefineds(full);
    validateAtMostOneUndefined(full);
    return full;
}
exports.normalizeIntervals = normalizeIntervals;
/**
 * Completely order scaling intervals, making their lower and upper bounds concrete.
 */
function orderAndCompleteIntervals(intervals) {
    if (intervals.length < 2) {
        throw new Error('Require at least 2 intervals');
    }
    for (const interval of intervals) {
        if (interval.lower === undefined && interval.upper === undefined) {
            throw new Error(`Must supply at least one of 'upper' or 'lower', got: ${JSON.stringify(interval)}`);
        }
    }
    // Make a copy
    intervals = intervals.map(x => ({ ...x }));
    // Sort by whatever number we have for each interval
    intervals.sort(comparatorFromKey((x) => x.lower !== undefined ? x.lower : x.upper));
    // Propagate boundaries until no more change
    while (propagateBounds(intervals)) { /* Repeat */ }
    const lastIndex = intervals.length - 1;
    // Validate that no intervals have undefined bounds now, which must mean they're complete.
    if (intervals[0].lower === undefined) {
        intervals[0] = { ...intervals[0], lower: 0 };
    }
    if (intervals[lastIndex].upper === undefined) {
        intervals[lastIndex] = { ...intervals[lastIndex], upper: Infinity };
    }
    for (const interval of intervals) {
        if (interval.lower === undefined || interval.upper === undefined) {
            throw new Error(`Could not determine the lower and upper bounds for ${JSON.stringify(interval)}`);
        }
    }
    const completeIntervals = intervals;
    // Validate that we have nonoverlapping intervals now.
    for (let i = 0; i < completeIntervals.length - 1; i++) {
        if (overlap(completeIntervals[i], completeIntervals[i + 1])) {
            throw new Error(`Two intervals overlap: ${JSON.stringify(completeIntervals[i])} and ${JSON.stringify(completeIntervals[i + 1])}`);
        }
    }
    // Fill up the gaps
    return completeIntervals;
}
/**
 * Make the intervals cover the complete number line
 *
 * This entails adding intervals with an 'undefined' change to fill up the gaps.
 *
 * Since metrics have a halfopen interval, the first one will get a lower bound
 * of 0, the last one will get an upper bound of +Infinity.
 *
 * In case of absolute adjustments, the lower number of the adjacent bound will
 * be used, which means conservative change. In case of relative adjustments,
 * we'll use relative adjusment 0 (which means no change).
 */
function makeGapsUndefined(intervals) {
    // Add edge intervals if necessary, but only for relative adjustments. Since we're
    // going to make scaling intervals extend all the way out to infinity on either side,
    // the result is the same for absolute adjustments anyway.
    if (intervals[0].lower !== 0) {
        intervals.splice(0, 1, {
            lower: 0,
            upper: intervals[0].lower,
            change: undefined,
        });
    }
    if (last(intervals).upper !== Infinity) {
        intervals.push({
            lower: last(intervals).upper,
            upper: Infinity,
            change: undefined
        });
    }
    let i = 1;
    while (i < intervals.length) {
        if (intervals[i - 1].upper < intervals[i].lower) {
            intervals.splice(i, 0, {
                lower: intervals[i - 1].upper,
                upper: intervals[i].lower,
                change: undefined
            });
        }
        else {
            i++;
        }
    }
}
/**
 * Turn zero changes into undefined, in-place
 */
function makeZerosUndefined(intervals) {
    for (let i = 0; i < intervals.length; ++i) {
        const interval = intervals[i];
        if (interval.change === 0) {
            intervals[i] = { ...interval, change: undefined };
        }
    }
}
/**
 * If there are adjacent "undefined" intervals, combine them
 */
function combineUndefineds(intervals) {
    let i = 0;
    while (i < intervals.length - 1) {
        if (intervals[i].change === undefined && intervals[i + 1].change === undefined) {
            intervals[i] = { ...intervals[i], upper: intervals[i + 1].upper };
            intervals.splice(i + 1, 1);
        }
        else {
            i++;
        }
    }
}
function validateAtMostOneUndefined(intervals) {
    const undef = intervals.filter(x => x.change === undefined);
    if (undef.length > 1) {
        throw new Error(`Can have at most one no-change interval, got ${JSON.stringify(undef)}`);
    }
}
function comparatorFromKey(keyFn) {
    return (a, b) => {
        const keyA = keyFn(a);
        const keyB = keyFn(b);
        if (keyA < keyB) {
            return -1;
        }
        if (keyA === keyB) {
            return 0;
        }
        return 1;
    };
}
function propagateBounds(intervals) {
    let ret = false;
    // Propagate upper bounds upwards
    for (let i = 0; i < intervals.length - 1; i++) {
        if (intervals[i].upper !== undefined && intervals[i + 1].lower === undefined) {
            intervals[i + 1] = { ...intervals[i + 1], lower: intervals[i].upper };
            ret = true;
        }
    }
    // Propagate lower bounds downwards
    for (let i = intervals.length - 1; i >= 1; i--) {
        if (intervals[i].lower !== undefined && intervals[i - 1].upper === undefined) {
            intervals[i - 1] = { ...intervals[i - 1], upper: intervals[i].lower };
            ret = true;
        }
    }
    return ret;
}
/**
 * Whether two intervals overlap
 */
function overlap(a, b) {
    return a.lower < b.upper && a.upper > b.lower;
}
function last(xs) {
    return xs[xs.length - 1];
}
/**
 * Locate the intervals that should have the alarm thresholds, by index.
 *
 * Pick the intervals on either side of the singleton "undefined" interval, or
 * pick the middle interval if there's no such interval.
 */
function findAlarmThresholds(intervals) {
    const gapIndex = intervals.findIndex(x => x.change === undefined);
    if (gapIndex !== -1) {
        return {
            lowerAlarmIntervalIndex: gapIndex > 0 ? gapIndex - 1 : undefined,
            upperAlarmIntervalIndex: gapIndex < intervals.length - 1 ? gapIndex + 1 : undefined,
        };
    }
    if (intervals.length === 1) {
        return { upperAlarmIntervalIndex: 0 };
    }
    const middleIndex = Math.floor(intervals.length / 2);
    return {
        lowerAlarmIntervalIndex: middleIndex - 1,
        upperAlarmIntervalIndex: middleIndex
    };
}
exports.findAlarmThresholds = findAlarmThresholds;
//# sourceMappingURL=data:application/json;base64,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