# ZeroProof Python SDK

[![PyPI version](https://badge.fury.io/py/zeroproof.svg)](https://badge.fury.io/py/zeroproof)
[![Python 3.8+](https://img.shields.io/badge/python-3.8+-blue.svg)](https://www.python.org/downloads/)
[![License: MIT](https://img.shields.io/badge/License-MIT-yellow.svg)](https://opensource.org/licenses/MIT)

Python SDK for the ZeroProof AI verification API. Secure your agentic e-commerce ecosystem with zero-knowledge proofs.

## Installation

```bash
pip install zeroproof
```

## Quick Start

```python
from zeroproof import ZeroProof

# Initialize the client with your API key
client = ZeroProof(api_key="zkp_your_api_key_here")

# Create a verification challenge
challenge = client.create_challenge(
    agent_id="shopping-assistant-v1",
    action="add_to_cart",
    context={"item_id": "laptop-123", "price": 999.99}
)

print(f"Challenge ID: {challenge.challenge_id}")
print(f"Nonce: {challenge.nonce}")
print(f"Expires in: {challenge.expires_in} seconds")

# Verify the proof (in production, this would be generated by your agent)
result = client.verify_proof(
    challenge_id=challenge.challenge_id,
    proof="your_cryptographic_proof_here",
    agent_signature="optional_signature_here"
)

if result.verified:
    print(f"✅ Verification successful!")
    print(f"   Agent: {result.agent_id}")
    print(f"   Action: {result.action}")
    print(f"   Confidence: {result.confidence * 100}%")
    # Now execute the actual e-commerce action
else:
    print("❌ Verification failed")
```

## Use Cases

### E-Commerce Agent Authorization

```python
from zeroproof import ZeroProof

client = ZeroProof(api_key="zkp_...")

# Before allowing an agent to make a purchase
challenge = client.create_challenge(
    agent_id="checkout-agent-v2",
    action="initiate_purchase",
    context={
        "amount": 1499.99,
        "merchant": "TechStore",
        "items": ["laptop", "mouse", "keyboard"]
    }
)

# Agent generates proof
proof = your_agent.generate_proof(challenge)

# Verify before processing payment
result = client.verify_proof(challenge.challenge_id, proof)

if result.verified and result.confidence > 0.95:
    process_payment(1499.99)
else:
    log_security_incident(result)
```

### Refund Authorization

```python
# Verify agent before approving refund
challenge = client.create_challenge(
    agent_id="refund-processor",
    action="approve_refund",
    context={"order_id": "ORD-12345", "amount": 299.99}
)

result = client.verify_proof(challenge.challenge_id, proof)

if result.verified:
    approve_refund(order_id="ORD-12345")
```

### Inventory Management

```python
# Verify agent before updating inventory
challenge = client.create_challenge(
    agent_id="inventory-manager",
    action="update_stock",
    context={"product_id": "PROD-789", "quantity_change": -50}
)

result = client.verify_proof(challenge.challenge_id, proof)

if result.verified:
    update_inventory("PROD-789", quantity=-50)
```

## API Reference

### ZeroProof

The main client class for interacting with the API.

#### `__init__(api_key: str, base_url: Optional[str] = None)`

Initialize the ZeroProof client.

**Parameters:**
- `api_key` (str): Your ZeroProof API key (starts with 'zkp_')
- `base_url` (str, optional): Custom API base URL (defaults to production)

**Raises:**
- `ValueError`: If API key is invalid or missing

#### `create_challenge(agent_id: str, action: str, context: Optional[Dict] = None) -> Challenge`

Create a new verification challenge.

**Parameters:**
- `agent_id` (str): Unique identifier for the AI agent
- `action` (str): The action the agent wants to perform
- `context` (dict, optional): Additional context for the action

**Returns:**
- `Challenge`: Object containing challenge_id, nonce, expires_in, timestamp

**Raises:**
- `ZeroProofError`: If the request fails

#### `verify_proof(challenge_id: str, proof: str, agent_signature: Optional[str] = None) -> VerificationResult`

Verify a proof for a given challenge.

**Parameters:**
- `challenge_id` (str): The challenge ID from create_challenge()
- `proof` (str): The cryptographic proof data
- `agent_signature` (str, optional): Optional agent signature

**Returns:**
- `VerificationResult`: Object containing verification details

**Raises:**
- `ZeroProofError`: If verification fails

#### `get_status(session_id: str) -> Dict`

Get the status of a verification session.

**Parameters:**
- `session_id` (str): The session/challenge ID to check

**Returns:**
- `dict`: Session status details

**Raises:**
- `ZeroProofError`: If the request fails

### Data Classes

#### Challenge

Represents a verification challenge.

**Attributes:**
- `challenge_id` (str): Unique challenge identifier
- `nonce` (str): Random nonce for this challenge
- `expires_in` (int): Time until expiration in seconds
- `timestamp` (int): Challenge creation timestamp

#### VerificationResult

Represents the result of proof verification.

**Attributes:**
- `verified` (bool): Whether the proof was verified
- `agent_id` (str): The agent that was verified
- `action` (str): The action that was authorized
- `confidence` (float): Confidence score (0.0 to 1.0)
- `timestamp` (str): Verification timestamp
- `session_id` (str): Session identifier

### Exceptions

#### ZeroProofError

Base exception for SDK errors.

**Attributes:**
- `message` (str): Error message
- `status_code` (int, optional): HTTP status code
- `response` (dict, optional): Full API response

## Context Manager Support

The SDK supports context managers for automatic resource cleanup:

```python
with ZeroProof(api_key="zkp_...") as client:
    challenge = client.create_challenge("agent-1", "purchase")
    result = client.verify_proof(challenge.challenge_id, proof)
# Session is automatically closed
```

## Error Handling

```python
from zeroproof import ZeroProof, ZeroProofError

client = ZeroProof(api_key="zkp_...")

try:
    challenge = client.create_challenge("agent", "action")
    result = client.verify_proof(challenge.challenge_id, "invalid_proof")
except ZeroProofError as e:
    print(f"Error: {e.message}")
    print(f"Status Code: {e.status_code}")
    print(f"Response: {e.response}")
```

## Configuration

### Custom API Endpoint

For testing or self-hosted deployments:

```python
client = ZeroProof(
    api_key="zkp_...",
    base_url="https://api.custom-domain.com/v1"
)
```

### Environment Variables

You can set your API key via environment variable:

```python
import os
from zeroproof import ZeroProof

api_key = os.getenv("ZEROPROOF_API_KEY")
client = ZeroProof(api_key=api_key)
```

## Requirements

- Python 3.8+
- requests >= 2.25.0

## Getting an API Key

1. Sign up at [https://zeroproofai.com](https://zeroproofai.com)
2. Navigate to your dashboard
3. Generate a new API key
4. Copy the key (starts with `zkp_`)

## Development

### Installing for Development

```bash
git clone https://github.com/jacobweiss2305/zeroproof.git
cd zeroproof/python-sdk
pip install -e ".[dev]"
```

### Running Tests

```bash
pytest tests/
```

### Code Formatting

```bash
black zeroproof/
flake8 zeroproof/
mypy zeroproof/
```

## Support

- **Documentation**: [https://docs.zeroproofai.com](https://docs.zeroproofai.com)
- **Issues**: [GitHub Issues](https://github.com/jacobweiss2305/zeroproof/issues)
- **Email**: support@zeroproofai.com

## License

MIT License - see [LICENSE](LICENSE) file for details.

## Contributing

Contributions are welcome! Please feel free to submit a Pull Request.

## Changelog

### 0.1.0 (2025-01-10)

- Initial release
- Basic challenge/proof verification flow
- Context manager support
- Comprehensive error handling
- Full type hints
