# Copyright © Michal Čihař <michal@weblate.org>
#
# SPDX-License-Identifier: GPL-3.0-or-later
from __future__ import annotations

import warnings
from typing import TYPE_CHECKING, ClassVar

from django.utils.translation import gettext_lazy

from weblate.addons.base import BaseAddon
from weblate.addons.events import AddonEvent
from weblate.addons.tasks import language_consistency
from weblate.lang.models import Language

if TYPE_CHECKING:
    from weblate.addons.models import Addon
    from weblate.trans.models import Component, Translation


class LanguageConsistencyAddon(BaseAddon):
    events: ClassVar[set[AddonEvent]] = {
        AddonEvent.EVENT_DAILY,
        AddonEvent.EVENT_POST_ADD,
    }
    name = "weblate.consistency.languages"
    verbose = gettext_lazy("Add missing languages")
    description = gettext_lazy(
        "Ensures a consistent set of languages is used for all components "
        "within a project."
    )
    icon = "language.svg"
    project_scope = True
    user_name = "languages"
    user_verbose = "Languages add-on"

    def daily(self, component: Component, activity_log_id: int | None = None) -> None:
        # The languages list is built here because we want to exclude shared
        # component's languages that are included in Project.languages
        language_consistency.delay_on_commit(
            self.instance.id,
            list(
                Language.objects.filter(
                    translation__component__project=component.project
                ).values_list("id", flat=True)
            ),
            component.project_id,
            activity_log_id=activity_log_id,
        )

    def post_add(
        self, translation: Translation, activity_log_id: int | None = None, **kwargs
    ) -> None:
        language_consistency.delay_on_commit(
            self.instance.id,
            [translation.language_id],
            translation.component.project_id,
            activity_log_id=activity_log_id,
        )


class LangaugeConsistencyAddon(LanguageConsistencyAddon):
    def __init__(self, storage: Addon) -> None:
        super().__init__(storage)
        # TODO: Remove in Weblate 5.14
        warnings.warn(
            "LangaugeConsistencyAddon is deprecated, use LanguageConsistencyAddon",
            DeprecationWarning,
            stacklevel=1,
        )
