# Copyright © Michal Čihař <michal@weblate.org>
#
# SPDX-License-Identifier: GPL-3.0-or-later

# Generated by Django 4.2.5 on 2023-09-18 07:26

import django.core.validators
import django.db.models.deletion
import django.db.models.functions.text
from django.conf import settings
from django.core.exceptions import ImproperlyConfigured
from django.db import migrations, models

import weblate.lang.models
import weblate.trans.fields
import weblate.trans.mixins
import weblate.trans.models.unit
import weblate.trans.validators
import weblate.utils.fields
import weblate.utils.render
import weblate.utils.validators
from weblate.formats.models import FILE_FORMATS
from weblate.utils.db import MY_DROP, MY_FTX, PG_DROP, PG_TRGM
from weblate.utils.licenses import get_license_choices
from weblate.vcs.models import VCS_REGISTRY

FIELDS = (
    ("unit", "source", ", translation_id"),
    ("unit", "target", ", translation_id"),
    ("unit", "context", ", translation_id"),
    ("unit", "note", ", translation_id"),
    ("unit", "location", ", translation_id"),
    ("unit", "explanation", ", translation_id"),
    ("suggestion", "target", ", unit_id"),
    ("comment", "comment", ", unit_id"),
)


def create_index(apps, schema_editor) -> None:
    vendor = schema_editor.connection.vendor
    if vendor == "postgresql":
        cur = schema_editor.connection.cursor()

        # Install pg_trgm for trigram search and index
        cur.execute("SELECT * FROM pg_extension WHERE extname = 'pg_trgm'")
        if not cur.fetchone():
            schema_editor.execute("CREATE EXTENSION IF NOT EXISTS pg_trgm")

        # Install btree_gin for gin btree search and index
        cur.execute("SELECT * FROM pg_extension WHERE extname = 'btree_gin'")
        if not cur.fetchone():
            schema_editor.execute("CREATE EXTENSION IF NOT EXISTS btree_gin")

        # Create GIN trigram index on searched fields
        for table, field, extra in FIELDS:
            schema_editor.execute(PG_TRGM.format(table, field, extra))
    elif vendor == "mysql":
        for table, field, _extra in FIELDS:
            schema_editor.execute(MY_FTX.format(table, field))
        schema_editor.execute(
            "CREATE INDEX unit_source_index ON trans_unit(source(255))"
        )
        schema_editor.execute(
            "CREATE INDEX unit_context_index ON trans_unit(context(255))"
        )
    else:
        msg = f"Unsupported database: {vendor}"
        raise ImproperlyConfigured(msg)


def drop_index(apps, schema_editor) -> None:
    vendor = schema_editor.connection.vendor
    if vendor == "postgresql":
        for table, field, _extra in FIELDS:
            schema_editor.execute(PG_DROP.format(table, field))
    elif vendor == "mysql":
        for table, field, _extra in FIELDS:
            schema_editor.execute(MY_DROP.format(table, field))
        schema_editor.execute("ALTER TABLE trans_unit DROP INDEX unit_source_index")
        schema_editor.execute("ALTER TABLE trans_unit DROP INDEX unit_context_index")
    else:
        msg = f"Unsupported database: {vendor}"
        raise ImproperlyConfigured(msg)


class Migration(migrations.Migration):
    replaces = [
        ("trans", "0001_squashed_0074_fix_broken_browser_alert"),
        ("trans", "0075_auto_20200501_1428"),
        ("trans", "0076_comment_userdetails"),
        ("trans", "0077_auto_20200513_1512"),
        ("trans", "0078_auto_20200515_0729"),
        ("trans", "0079_auto_20200519_1436"),
        ("trans", "0080_auto_20200522_0909"),
        ("trans", "0081_announcement_notify"),
        ("trans", "0082_component_push_branch"),
        ("trans", "0083_component_restricted"),
        ("trans", "0084_auto_20200605_0648"),
        ("trans", "0085_change_glossary_term"),
        ("trans", "0086_auto_20200609_1134"),
        ("trans", "0087_auto_20200615_0747"),
        ("trans", "0088_component_auto_lock_error"),
        ("trans", "0089_auto_20200630_1321"),
        ("trans", "0090_alert_updated"),
        ("trans", "0091_json_key"),
        ("trans", "0092_alert_dismissed"),
        ("trans", "0093_auto_20200730_1432"),
        ("trans", "0094_project_language_aliases"),
        ("trans", "0095_fix_json_units"),
        ("trans", "0096_fix_enforced_checks"),
        ("trans", "0097_component_source_language"),
        ("trans", "0098_move_source_language"),
        ("trans", "0099_remove_project_source_language"),
        ("trans", "0100_auto_20200912_1131"),
        ("trans", "0101_fix_rename_changes"),
        ("trans", "0102_unit_source_unit"),
        ("trans", "0103_update_source_unit"),
        ("trans", "0104_update_source_unit_source"),
        ("trans", "0105_auto_20201006_1254"),
        ("trans", "0106_remove_unit_content_hash"),
        ("trans", "0107_component_links"),
        ("trans", "0108_move_mailing"),
        ("trans", "0109_remove_project_mail"),
        ("trans", "0110_auto_20210120_0845"),
        ("trans", "0111_index_explanation"),
        ("trans", "0112_component_new_unit"),
        ("trans", "0113_component_new_unit"),
        ("trans", "0114_auto_20210129_1239"),
        ("trans", "0115_auto_20210201_1305"),
        ("trans", "0116_migrate_glossaries"),
        ("trans", "0117_remove_change_glossary_term"),
        ("trans", "0118_fixup_glossary_changes"),
        ("trans", "0119_auto_20210206_1141"),
        ("trans", "0120_glossary_name"),
        ("trans", "0121_remove_component_glossary_name"),
        ("trans", "0122_auto_20210228_1846"),
        ("trans", "0123_fix_enforced_checks"),
        ("trans", "0124_glossary_new_lang"),
        ("trans", "0125_unit_details"),
        ("trans", "0126_auto_20210312_1348"),
        ("trans", "0127_fix_source_glossary"),
        ("trans", "0128_fix_pending_read_only"),
        ("trans", "0130_glossary_target"),
        ("trans", "0131_alter_unit_index_together"),
        ("trans", "0132_alter_unit_state"),
        ("trans", "0133_glossary_missing_files"),
        ("trans", "0134_component_remote_revision"),
        ("trans", "0135_component_local_revision"),
        ("trans", "0136_auto_20210611_0731"),
        ("trans", "0137_alter_project_language_aliases"),
        ("trans", "0138_alter_component_report_source_bugs"),
        ("trans", "0139_alter_component_repoweb"),
        ("trans", "0140_fix_commit_age"),
        ("trans", "0141_alter_component_commit_pending_age"),
        ("trans", "0142_projecttoken"),
        ("trans", "0143_alter_unit_labels"),
        ("trans", "0144_projecttoken_user"),
        ("trans", "0145_alter_change_action"),
        ("trans", "0146_alter_component_merge_style"),
        ("trans", "0147_delete_projecttoken"),
        ("trans", "0148_alter_component_language_code_style"),
        ("trans", "0149_component_pull_message"),
        ("trans", "0150_convert_context"),
        ("trans", "0151_project_machinery_settings"),
        ("trans", "0152_alter_change_action"),
        ("trans", "0153_rename_java_language"),
        ("trans", "0154_alter_component_language_code_style"),
        ("trans", "0155_java_format"),
        ("trans", "0156_alter_change_action"),
        ("trans", "0157_alter_alert_details_alter_component_enforced_checks"),
        ("trans", "0158_alter_change_action"),
        ("trans", "0159_alter_change_index_together_alter_change_timestamp_and_more"),
        ("trans", "0160_alter_change_index_together"),
        ("trans", "0161_alter_project_name_alter_project_web"),
        ("trans", "0162_alter_component_language_code_style"),
        ("trans", "0163_update_indexes"),
        ("trans", "0164_alter_component_push_alter_component_repo"),
        (
            "trans",
            "0165_rename_change_timestamp_project_component_language_action_trans_chang_timesta_33178f_idx_and_more",
        ),
        ("trans", "0166_alter_change_action"),
        ("trans", "0167_pending_explanation"),
        ("trans", "0168_unit_last_updated_unit_target_hash"),
        ("trans", "0169_unit_trans_unit_source_md5_index_and_more"),
        ("trans", "0170_component_screenshot_filemask"),
        ("trans", "0171_alter_component_language_code_style"),
        ("trans", "0172_alter_change_action"),
        ("trans", "0173_change_duplicate_string"),
        ("trans", "0174_adjust_fluent_unit_flags"),
        ("trans", "0175_alert_details_new_change_details_new_and_more"),
        ("trans", "0176_jsonfield"),
        ("trans", "0177_remove_alert_details_remove_change_details_and_more"),
        ("trans", "0178_rename_details_new_alert_details_and_more"),
        ("trans", "0179_alter_change_action"),
        ("trans", "0180_change_duplicate_language"),
        ("trans", "0181_change_trans_chang_user_id_b1b554_idx"),
        ("trans", "0182_category_component_category"),
        ("trans", "0183_alter_component_unique_together"),
        ("trans", "0184_alter_change_action"),
        ("trans", "0185_alter_component_allow_translation_propagation_and_more"),
        ("trans", "0186_alter_unit_translation"),
        ("trans", "0187_alter_variant_unique_together_alter_alert_component_and_more"),
        ("trans", "0188_remove_change_trans_chang_timesta_33178f_idx_and_more"),
        ("trans", "0189_remove_unit_trans_unit_source_md5_index_and_more"),
    ]

    initial = True

    dependencies = [
        ("lang", "0001_squashed_weblate_5"),
        migrations.swappable_dependency(settings.AUTH_USER_MODEL),
    ]

    operations = [
        migrations.CreateModel(
            name="Project",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                (
                    "name",
                    models.CharField(
                        help_text="Display name",
                        max_length=60,
                        unique=True,
                        validators=[weblate.utils.validators.validate_project_name],
                        verbose_name="Project name",
                    ),
                ),
                (
                    "slug",
                    models.SlugField(
                        help_text="Name used in URLs and filenames.",
                        max_length=60,
                        unique=True,
                        validators=[weblate.utils.validators.validate_slug],
                        verbose_name="URL slug",
                    ),
                ),
                (
                    "web",
                    models.URLField(
                        help_text="Main website of translated project.",
                        validators=[weblate.utils.validators.validate_project_web],
                        verbose_name="Project website",
                        blank=not settings.WEBSITE_REQUIRED,
                    ),
                ),
                (
                    "instructions",
                    models.TextField(
                        blank=True,
                        help_text="You can use Markdown and mention users by @username.",
                        verbose_name="Translation instructions",
                    ),
                ),
                (
                    "set_language_team",
                    models.BooleanField(
                        default=True,
                        help_text='Lets Weblate update the "Language-Team" file header of your project.',
                        verbose_name='Set "Language-Team" header',
                    ),
                ),
                (
                    "enable_hooks",
                    models.BooleanField(
                        default=True,
                        help_text="Whether to allow updating this repository by remote hooks.",
                        verbose_name="Enable hooks",
                    ),
                ),
                (
                    "access_control",
                    models.IntegerField(
                        choices=[
                            (0, "Public"),
                            (1, "Protected"),
                            (100, "Private"),
                            (200, "Custom"),
                        ],
                        default=settings.DEFAULT_ACCESS_CONTROL,
                        help_text="How to restrict access to this project is detailed in the documentation.",
                        verbose_name="Access control",
                    ),
                ),
                (
                    "translation_review",
                    models.BooleanField(
                        default=False,
                        help_text="Requires dedicated reviewers to approve translations.",
                        verbose_name="Enable reviews",
                    ),
                ),
                (
                    "source_review",
                    models.BooleanField(
                        default=False,
                        help_text="Requires dedicated reviewers to approve source strings.",
                        verbose_name="Enable source reviews",
                    ),
                ),
                (
                    "use_shared_tm",
                    models.BooleanField(
                        default=settings.DEFAULT_SHARED_TM,
                        help_text="Uses the pool of shared translations between projects.",
                        verbose_name="Use shared translation memory",
                    ),
                ),
                (
                    "contribute_shared_tm",
                    models.BooleanField(
                        default=settings.DEFAULT_SHARED_TM,
                        help_text="Contributes to the pool of shared translations between projects.",
                        verbose_name="Contribute to shared translation memory",
                    ),
                ),
                (
                    "language_aliases",
                    models.TextField(
                        blank=True,
                        default="",
                        help_text="Comma-separated list of language code mappings, for example: en_GB:en,en_US:en",
                        validators=[weblate.utils.validators.validate_language_aliases],
                        verbose_name="Language aliases",
                    ),
                ),
                ("machinery_settings", models.JSONField(blank=True, default=dict)),
            ],
            options={
                "verbose_name": "Project",
                "verbose_name_plural": "Projects",
            },
            bases=(models.Model, weblate.trans.mixins.PathMixin),
        ),
        migrations.CreateModel(
            name="Component",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                (
                    "name",
                    models.CharField(
                        help_text="Display name",
                        max_length=100,
                        verbose_name="Component name",
                    ),
                ),
                (
                    "slug",
                    models.SlugField(
                        help_text="Name used in URLs and filenames.",
                        max_length=100,
                        validators=[weblate.utils.validators.validate_slug],
                        verbose_name="URL slug",
                    ),
                ),
                (
                    "repo",
                    models.CharField(
                        help_text="URL of a repository, use weblate://project/component to share it with other component.",
                        max_length=300,
                        verbose_name="Source code repository",
                    ),
                ),
                (
                    "push",
                    models.CharField(
                        blank=True,
                        help_text="URL of a push repository, pushing is turned off if empty.",
                        max_length=300,
                        verbose_name="Repository push URL",
                    ),
                ),
                (
                    "repoweb",
                    models.URLField(
                        blank=True,
                        help_text="Link to repository browser, use {{branch}} for branch, {{filename}} and {{line}} as filename and line placeholders. You might want to strip leading directory by using {{filename|parentdir}}.",
                        validators=[weblate.utils.render.validate_repoweb],
                        verbose_name="Repository browser",
                    ),
                ),
                (
                    "git_export",
                    models.CharField(
                        blank=True,
                        help_text="URL of repository where users can fetch changes from Weblate",
                        max_length=220,
                        verbose_name="Exported repository URL",
                    ),
                ),
                (
                    "report_source_bugs",
                    weblate.utils.fields.EmailField(
                        blank=True,
                        help_text="E-mail address for reports on errors in source strings. Leave empty for no e-mails.",
                        max_length=190,
                        verbose_name="Source string bug reporting address",
                    ),
                ),
                (
                    "branch",
                    models.CharField(
                        blank=True,
                        default="",
                        help_text="Repository branch to translate",
                        max_length=200,
                        verbose_name="Repository branch",
                    ),
                ),
                (
                    "filemask",
                    models.CharField(
                        help_text="Path of files to translate relative to repository root, use * instead of language code, for example: po/*.po or locale/*/LC_MESSAGES/django.po.",
                        max_length=400,
                        validators=[
                            weblate.trans.validators.validate_filemask,
                            weblate.utils.validators.validate_filename,
                        ],
                        verbose_name="File mask",
                    ),
                ),
                (
                    "template",
                    models.CharField(
                        blank=True,
                        help_text="Filename of translation base file, containing all strings and their source for monolingual translations.",
                        max_length=400,
                        validators=[weblate.utils.validators.validate_filename],
                        verbose_name="Monolingual base language file",
                    ),
                ),
                (
                    "new_base",
                    models.CharField(
                        blank=True,
                        help_text="Filename of file used for creating new translations. For gettext choose .pot file.",
                        max_length=400,
                        validators=[weblate.utils.validators.validate_filename],
                        verbose_name="Template for new translations",
                    ),
                ),
                (
                    "intermediate",
                    models.CharField(
                        blank=True,
                        help_text="Filename of intermediate translation file. In most cases this is a translation file provided by developers and is used when creating actual source strings.",
                        max_length=400,
                        validators=[weblate.utils.validators.validate_filename],
                        verbose_name="Intermediate language file",
                    ),
                ),
                (
                    "file_format",
                    models.CharField(
                        choices=FILE_FORMATS.get_choices(),
                        max_length=50,
                        verbose_name="File format",
                    ),
                ),
                (
                    "locked",
                    models.BooleanField(
                        default=False,
                        help_text="Locked component will not get any translation updates.",
                        verbose_name="Locked",
                    ),
                ),
                (
                    "allow_translation_propagation",
                    models.BooleanField(
                        default=settings.DEFAULT_TRANSLATION_PROPAGATION,
                        help_text="Whether translation updates in other components will cause automatic translation in this one",
                        verbose_name="Allow translation propagation",
                    ),
                ),
                (
                    "enable_suggestions",
                    models.BooleanField(
                        default=True,
                        help_text="Whether to allow translation suggestions at all.",
                        verbose_name="Turn on suggestions",
                    ),
                ),
                (
                    "suggestion_voting",
                    models.BooleanField(
                        default=False,
                        help_text="Users can only vote for suggestions and can’t make direct translations.",
                        verbose_name="Suggestion voting",
                    ),
                ),
                (
                    "suggestion_autoaccept",
                    models.PositiveSmallIntegerField(
                        default=0,
                        help_text="Automatically accept suggestions with this number of votes, use 0 to turn it off.",
                        validators=[weblate.trans.validators.validate_autoaccept],
                        verbose_name="Autoaccept suggestions",
                    ),
                ),
                (
                    "check_flags",
                    models.TextField(
                        blank=True,
                        default="",
                        help_text="Additional comma-separated flags to influence Weblate behavior.",
                        validators=[weblate.trans.validators.validate_check_flags],
                        verbose_name="Translation flags",
                    ),
                ),
                (
                    "project",
                    models.ForeignKey(
                        db_index=False,
                        on_delete=django.db.models.deletion.CASCADE,
                        to="trans.project",
                        verbose_name="Project",
                    ),
                ),
                (
                    "commit_message",
                    models.TextField(
                        default=settings.DEFAULT_COMMIT_MESSAGE,
                        help_text="You can use template language for various info, please consult the documentation for more details.",
                        validators=[weblate.utils.render.validate_render_commit],
                        verbose_name="Commit message when translating",
                    ),
                ),
                (
                    "license",
                    models.CharField(
                        blank=not settings.LICENSE_REQUIRED,
                        choices=get_license_choices(),
                        default="",
                        max_length=150,
                        verbose_name="Translation license",
                    ),
                ),
                (
                    "merge_style",
                    models.CharField(
                        choices=[
                            ("merge", "Merge"),
                            ("rebase", "Rebase"),
                            ("merge_noff", "Merge without fast-forward"),
                        ],
                        default=settings.DEFAULT_MERGE_STYLE,
                        help_text="Define whether Weblate should merge the upstream repository or rebase changes onto it.",
                        max_length=10,
                        verbose_name="Merge style",
                    ),
                ),
                (
                    "new_lang",
                    models.CharField(
                        choices=[
                            ("contact", "Contact maintainers"),
                            ("url", "Point to translation instructions URL"),
                            ("add", "Create new language file"),
                            ("none", "Disable adding new translations"),
                        ],
                        default="add",
                        help_text="How to handle requests for creating new translations.",
                        max_length=10,
                        verbose_name="Adding new translation",
                    ),
                ),
                (
                    "vcs",
                    models.CharField(
                        choices=VCS_REGISTRY.get_choices(),
                        default="git",
                        help_text="Version control system to use to access your repository containing translations. You can also choose additional integration with third party providers to submit merge requests.",
                        max_length=20,
                        verbose_name="Version control system",
                    ),
                ),
                (
                    "edit_template",
                    models.BooleanField(
                        default=True,
                        help_text="Whether users will be able to edit the base file for monolingual translations.",
                        verbose_name="Edit base file",
                    ),
                ),
                (
                    "agreement",
                    models.TextField(
                        blank=True,
                        default="",
                        help_text="User agreement which needs to be approved before a user can translate this component.",
                        verbose_name="Contributor agreement",
                    ),
                ),
                (
                    "language_regex",
                    weblate.trans.fields.RegexField(
                        default="^[^.]+$",
                        help_text="Regular expression used to filter translation files when scanning for file mask.",
                        max_length=500,
                        verbose_name="Language filter",
                    ),
                ),
                (
                    "add_message",
                    models.TextField(
                        default=settings.DEFAULT_ADD_MESSAGE,
                        help_text="You can use template language for various info, please consult the documentation for more details.",
                        validators=[weblate.utils.render.validate_render_commit],
                        verbose_name="Commit message when adding translation",
                    ),
                ),
                (
                    "delete_message",
                    models.TextField(
                        default=settings.DEFAULT_DELETE_MESSAGE,
                        help_text="You can use template language for various info, please consult the documentation for more details.",
                        validators=[weblate.utils.render.validate_render_commit],
                        verbose_name="Commit message when removing translation",
                    ),
                ),
                (
                    "priority",
                    models.IntegerField(
                        choices=[
                            (60, "Very high"),
                            (80, "High"),
                            (100, "Medium"),
                            (120, "Low"),
                            (140, "Very low"),
                        ],
                        default=100,
                        help_text="Components with higher priority are offered first to translators.",
                        verbose_name="Priority",
                    ),
                ),
                (
                    "commit_pending_age",
                    models.SmallIntegerField(
                        default=settings.COMMIT_PENDING_HOURS,
                        help_text="Time in hours after which any pending changes will be committed to the VCS.",
                        validators=[django.core.validators.MaxValueValidator(2160)],
                        verbose_name="Age of changes to commit",
                    ),
                ),
                (
                    "push_on_commit",
                    models.BooleanField(
                        default=settings.DEFAULT_PUSH_ON_COMMIT,
                        help_text="Whether the repository should be pushed upstream on every commit.",
                        verbose_name="Push on commit",
                    ),
                ),
                (
                    "linked_component",
                    models.ForeignKey(
                        editable=False,
                        null=True,
                        on_delete=django.db.models.deletion.CASCADE,
                        to="trans.component",
                        verbose_name="Project",
                    ),
                ),
                (
                    "merge_message",
                    models.TextField(
                        default=settings.DEFAULT_MERGE_MESSAGE,
                        help_text="You can use template language for various info, please consult the documentation for more details.",
                        validators=[weblate.utils.render.validate_render_component],
                        verbose_name="Commit message when merging translation",
                    ),
                ),
                (
                    "addon_message",
                    models.TextField(
                        default=settings.DEFAULT_ADDON_MESSAGE,
                        help_text="You can use template language for various info, please consult the documentation for more details.",
                        validators=[weblate.utils.render.validate_render_addon],
                        verbose_name="Commit message when add-on makes a change",
                    ),
                ),
                (
                    "language_code_style",
                    models.CharField(
                        blank=True,
                        choices=[
                            ("", "Default based on the file format"),
                            ("posix", "POSIX style using underscore as a separator"),
                            ("bcp", "BCP style using hyphen as a separator"),
                            (
                                "posix_long",
                                "POSIX style using underscore as a separator, including country code",
                            ),
                            (
                                "posix_long_lowercase",
                                "POSIX style using underscore as a separator, including country code (lowercase)",
                            ),
                            (
                                "bcp_long",
                                "BCP style using hyphen as a separator, including country code",
                            ),
                            (
                                "bcp_legacy",
                                "BCP style using hyphen as a separator, legacy language codes",
                            ),
                            (
                                "bcp_lower",
                                "BCP style using hyphen as a separator, lower cased",
                            ),
                            ("android", "Android style"),
                            ("appstore", "Apple App Store metadata style"),
                            ("googleplay", "Google Play metadata style"),
                            ("linux", "Linux style"),
                        ],
                        default="",
                        help_text="Customize language code used to generate the filename for translations created by Weblate.",
                        max_length=20,
                        verbose_name="Language code style",
                    ),
                ),
                (
                    "variant_regex",
                    weblate.trans.fields.RegexField(
                        blank=True,
                        default="",
                        help_text="Regular expression used to determine variants of a string.",
                        max_length=190,
                        validators=[weblate.utils.validators.validate_re_nonempty],
                        verbose_name="Variants regular expression",
                    ),
                ),
                (
                    "push_branch",
                    models.CharField(
                        blank=True,
                        default="",
                        help_text="Branch for pushing changes, leave empty to use repository branch",
                        max_length=200,
                        verbose_name="Push branch",
                    ),
                ),
                (
                    "restricted",
                    models.BooleanField(
                        db_index=True,
                        default=settings.DEFAULT_RESTRICTED_COMPONENT,
                        help_text="Restrict access to the component to only those explicitly given permission.",
                        verbose_name="Restricted component",
                    ),
                ),
                (
                    "auto_lock_error",
                    models.BooleanField(
                        default=settings.DEFAULT_AUTO_LOCK_ERROR,
                        help_text="Whether the component should be locked on repository errors.",
                        verbose_name="Lock on error",
                    ),
                ),
                (
                    "source_language",
                    models.ForeignKey(
                        default=weblate.lang.models.get_default_lang,
                        help_text="Language used for source strings in all components",
                        on_delete=django.db.models.deletion.CASCADE,
                        to="lang.language",
                        verbose_name="Source language",
                    ),
                ),
                (
                    "links",
                    models.ManyToManyField(
                        blank=True,
                        help_text="Choose additional projects where this component will be listed.",
                        related_name="shared_components",
                        to="trans.project",
                        verbose_name="Share in projects",
                    ),
                ),
                (
                    "manage_units",
                    models.BooleanField(
                        default=False,
                        help_text="Enables adding and removing strings straight from Weblate. If your strings are extracted from the source code or managed externally you probably want to keep it disabled.",
                        verbose_name="Manage strings",
                    ),
                ),
                (
                    "glossary_color",
                    models.CharField(
                        choices=[
                            ("navy", "Navy"),
                            ("blue", "Blue"),
                            ("aqua", "Aqua"),
                            ("teal", "Teal"),
                            ("olive", "Olive"),
                            ("green", "Green"),
                            ("lime", "Lime"),
                            ("yellow", "Yellow"),
                            ("orange", "Orange"),
                            ("red", "Red"),
                            ("maroon", "Maroon"),
                            ("fuchsia", "Fuchsia"),
                            ("purple", "Purple"),
                            ("black", "Black"),
                            ("gray", "Gray"),
                            ("silver", "Silver"),
                        ],
                        default="silver",
                        max_length=30,
                        verbose_name="Glossary color",
                    ),
                ),
                (
                    "is_glossary",
                    models.BooleanField(
                        db_index=True, default=False, verbose_name="Use as a glossary"
                    ),
                ),
                (
                    "remote_revision",
                    models.CharField(blank=True, default="", max_length=200),
                ),
                (
                    "local_revision",
                    models.CharField(blank=True, default="", max_length=200),
                ),
                (
                    "pull_message",
                    models.TextField(
                        default=settings.DEFAULT_PULL_MESSAGE,
                        help_text="You can use template language for various info, please consult the documentation for more details.",
                        validators=[weblate.utils.render.validate_render_addon],
                        verbose_name="Merge request message",
                    ),
                ),
                (
                    "screenshot_filemask",
                    models.CharField(
                        blank=True,
                        help_text="Path of screenshots relative to repository root, for example: docs/screenshots/*.png.",
                        max_length=400,
                        validators=[
                            weblate.trans.validators.validate_filemask,
                            weblate.utils.validators.validate_filename,
                        ],
                        verbose_name="Screenshot file mask",
                    ),
                ),
                (
                    "enforced_checks",
                    models.JSONField(
                        blank=True,
                        default=list,
                        help_text="List of checks which can not be dismissed.",
                        verbose_name="Enforced checks",
                    ),
                ),
            ],
            options={
                "verbose_name": "Component",
                "verbose_name_plural": "Components",
            },
            bases=(models.Model, weblate.trans.mixins.PathMixin),
        ),
        migrations.CreateModel(
            name="Translation",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                ("revision", models.CharField(blank=True, default="", max_length=200)),
                ("filename", models.CharField(max_length=400)),
                (
                    "language_code",
                    models.CharField(blank=True, default="", max_length=50),
                ),
                (
                    "language",
                    models.ForeignKey(
                        on_delete=django.db.models.deletion.CASCADE, to="lang.language"
                    ),
                ),
                (
                    "component",
                    models.ForeignKey(
                        db_index=False,
                        on_delete=django.db.models.deletion.CASCADE,
                        to="trans.component",
                    ),
                ),
                (
                    "plural",
                    models.ForeignKey(
                        on_delete=django.db.models.deletion.CASCADE, to="lang.plural"
                    ),
                ),
                (
                    "check_flags",
                    models.TextField(
                        blank=True,
                        default="",
                        validators=[weblate.trans.validators.validate_check_flags],
                        verbose_name="Translation flags",
                    ),
                ),
            ],
            options={
                "unique_together": {("component", "language")},
                "verbose_name": "translation",
                "verbose_name_plural": "translations",
            },
            bases=(models.Model, weblate.trans.mixins.URLMixin),
        ),
        migrations.CreateModel(
            name="Label",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                ("name", models.CharField(max_length=190, verbose_name="Label name")),
                (
                    "color",
                    models.CharField(
                        choices=[
                            ("navy", "Navy"),
                            ("blue", "Blue"),
                            ("aqua", "Aqua"),
                            ("teal", "Teal"),
                            ("olive", "Olive"),
                            ("green", "Green"),
                            ("lime", "Lime"),
                            ("yellow", "Yellow"),
                            ("orange", "Orange"),
                            ("red", "Red"),
                            ("maroon", "Maroon"),
                            ("fuchsia", "Fuchsia"),
                            ("purple", "Purple"),
                            ("black", "Black"),
                            ("gray", "Gray"),
                            ("silver", "Silver"),
                        ],
                        default=None,
                        max_length=30,
                        verbose_name="Color",
                    ),
                ),
                (
                    "project",
                    models.ForeignKey(
                        db_index=False,
                        on_delete=django.db.models.deletion.CASCADE,
                        to="trans.project",
                    ),
                ),
            ],
            options={
                "unique_together": {("project", "name")},
                "verbose_name": "label",
                "verbose_name_plural": "label",
            },
        ),
        migrations.CreateModel(
            name="Unit",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                ("location", models.TextField(blank=True, default="")),
                ("context", models.TextField(blank=True, default="")),
                ("note", models.TextField(blank=True, default="")),
                ("flags", models.TextField(blank=True, default="")),
                ("source", models.TextField()),
                ("previous_source", models.TextField(blank=True, default="")),
                ("target", models.TextField(blank=True, default="")),
                ("position", models.IntegerField()),
                ("num_words", models.IntegerField(default=0)),
                ("priority", models.IntegerField(default=100)),
                (
                    "translation",
                    models.ForeignKey(
                        db_index=False,
                        on_delete=django.db.models.deletion.CASCADE,
                        to="trans.translation",
                    ),
                ),
                ("id_hash", models.BigIntegerField()),
                ("pending", models.BooleanField(default=False)),
                (
                    "state",
                    models.IntegerField(
                        choices=[
                            (0, "Empty"),
                            (10, "Needs editing"),
                            (20, "Translated"),
                            (30, "Approved"),
                            (100, "Read-only"),
                        ],
                        default=0,
                    ),
                ),
                (
                    "original_state",
                    models.IntegerField(
                        choices=[
                            (0, "Empty"),
                            (10, "Needs editing"),
                            (20, "Translated"),
                            (30, "Approved"),
                            (100, "Read-only"),
                        ],
                        default=0,
                    ),
                ),
                (
                    "explanation",
                    models.TextField(
                        blank=True,
                        default="",
                        help_text="Additional explanation to clarify meaning or usage of the string.",
                        verbose_name="Explanation",
                    ),
                ),
                (
                    "extra_flags",
                    models.TextField(
                        blank=True,
                        default="",
                        help_text="Additional comma-separated flags to influence Weblate behavior.",
                        validators=[weblate.trans.validators.validate_check_flags],
                        verbose_name="Translation flags",
                    ),
                ),
                ("timestamp", models.DateTimeField(auto_now_add=True)),
                (
                    "labels",
                    weblate.trans.models.unit.LabelsField(
                        blank=True, to="trans.label", verbose_name="Labels"
                    ),
                ),
                (
                    "source_unit",
                    models.ForeignKey(
                        blank=True,
                        null=True,
                        on_delete=django.db.models.deletion.CASCADE,
                        to="trans.unit",
                    ),
                ),
                ("details", models.JSONField(default=dict)),
                ("last_updated", models.DateTimeField(auto_now=True)),
            ],
            options={
                "verbose_name": "string",
                "verbose_name_plural": "strings",
                "unique_together": {("translation", "id_hash")},
            },
        ),
        migrations.CreateModel(
            name="Suggestion",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                ("target", models.TextField()),
                (
                    "unit",
                    models.ForeignKey(
                        on_delete=django.db.models.deletion.CASCADE, to="trans.unit"
                    ),
                ),
                (
                    "user",
                    models.ForeignKey(
                        blank=True,
                        null=True,
                        on_delete=django.db.models.deletion.CASCADE,
                        to=settings.AUTH_USER_MODEL,
                    ),
                ),
                ("timestamp", models.DateTimeField(auto_now_add=True)),
                ("userdetails", models.JSONField(default=dict)),
            ],
        ),
        migrations.CreateModel(
            name="Vote",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                (
                    "suggestion",
                    models.ForeignKey(
                        db_index=False,
                        on_delete=django.db.models.deletion.CASCADE,
                        to="trans.suggestion",
                    ),
                ),
                (
                    "user",
                    models.ForeignKey(
                        on_delete=django.db.models.deletion.CASCADE,
                        to=settings.AUTH_USER_MODEL,
                    ),
                ),
                ("value", models.SmallIntegerField(default=0)),
            ],
            options={
                "unique_together": {("suggestion", "user")},
                "verbose_name": "suggestion vote",
                "verbose_name_plural": "suggestion votes",
            },
        ),
        migrations.AddField(
            model_name="suggestion",
            name="votes",
            field=models.ManyToManyField(
                related_name="user_votes",
                through="trans.Vote",
                to=settings.AUTH_USER_MODEL,
            ),
        ),
        migrations.CreateModel(
            name="ComponentList",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                (
                    "name",
                    models.CharField(
                        help_text="Display name",
                        max_length=100,
                        unique=True,
                        verbose_name="Component list name",
                    ),
                ),
                (
                    "slug",
                    models.SlugField(
                        help_text="Name used in URLs and filenames.",
                        max_length=100,
                        unique=True,
                        verbose_name="URL slug",
                    ),
                ),
                (
                    "components",
                    models.ManyToManyField(blank=True, to="trans.component"),
                ),
                (
                    "show_dashboard",
                    models.BooleanField(
                        db_index=True,
                        default=True,
                        help_text="When enabled this component list will be shown as a tab on the dashboard",
                        verbose_name="Show on dashboard",
                    ),
                ),
            ],
            options={
                "verbose_name": "Component list",
                "verbose_name_plural": "Component lists",
            },
        ),
        migrations.CreateModel(
            name="Alert",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                ("timestamp", models.DateTimeField(auto_now_add=True)),
                ("name", models.CharField(max_length=150)),
                (
                    "component",
                    models.ForeignKey(
                        db_index=False,
                        on_delete=django.db.models.deletion.CASCADE,
                        to="trans.component",
                    ),
                ),
                ("updated", models.DateTimeField(auto_now=True)),
                ("dismissed", models.BooleanField(db_index=True, default=False)),
                ("details", models.JSONField(default=dict)),
            ],
            options={
                "verbose_name": "component alert",
                "verbose_name_plural": "component alerts",
                "unique_together": {("component", "name")},
            },
        ),
        migrations.CreateModel(
            name="Announcement",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                (
                    "message",
                    models.TextField(
                        help_text="You can use Markdown and mention users by @username.",
                        verbose_name="Message",
                    ),
                ),
                (
                    "language",
                    models.ForeignKey(
                        blank=True,
                        null=True,
                        on_delete=django.db.models.deletion.CASCADE,
                        to="lang.language",
                        verbose_name="Language",
                    ),
                ),
                (
                    "project",
                    models.ForeignKey(
                        blank=True,
                        null=True,
                        on_delete=django.db.models.deletion.CASCADE,
                        to="trans.project",
                        verbose_name="Project",
                    ),
                ),
                (
                    "component",
                    models.ForeignKey(
                        blank=True,
                        null=True,
                        on_delete=django.db.models.deletion.CASCADE,
                        to="trans.component",
                        verbose_name="Component",
                    ),
                ),
                (
                    "category",
                    models.CharField(
                        choices=[
                            ("info", "Info (light blue)"),
                            ("warning", "Warning (yellow)"),
                            ("danger", "Danger (red)"),
                            ("success", "Success (green)"),
                        ],
                        default="info",
                        help_text="Category defines color used for the message.",
                        max_length=25,
                        verbose_name="Category",
                    ),
                ),
                (
                    "expiry",
                    models.DateField(
                        blank=True,
                        db_index=True,
                        help_text="The message will be not shown after this date. Use it to announce string freeze and translation deadline for next release.",
                        null=True,
                        verbose_name="Expiry date",
                    ),
                ),
                (
                    "notify",
                    models.BooleanField(
                        blank=True, default=True, verbose_name="Notify users"
                    ),
                ),
            ],
            options={
                "verbose_name": "Announcement",
                "verbose_name_plural": "Announcements",
            },
        ),
        migrations.CreateModel(
            name="Comment",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                ("comment", models.TextField()),
                ("timestamp", models.DateTimeField(auto_now_add=True, db_index=True)),
                (
                    "user",
                    models.ForeignKey(
                        blank=True,
                        null=True,
                        on_delete=django.db.models.deletion.CASCADE,
                        to=settings.AUTH_USER_MODEL,
                    ),
                ),
                (
                    "unit",
                    models.ForeignKey(
                        on_delete=django.db.models.deletion.CASCADE, to="trans.unit"
                    ),
                ),
                ("resolved", models.BooleanField(db_index=True, default=False)),
                ("userdetails", models.JSONField(default=dict)),
            ],
            options={
                "verbose_name": "string comment",
                "verbose_name_plural": "string comments",
            },
        ),
        migrations.AlterModelOptions(
            name="suggestion",
            options={
                "verbose_name": "string suggestion",
                "verbose_name_plural": "string suggestions",
            },
        ),
        migrations.CreateModel(
            name="ContributorAgreement",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                ("timestamp", models.DateTimeField(auto_now=True)),
                (
                    "component",
                    models.ForeignKey(
                        on_delete=django.db.models.deletion.CASCADE,
                        to="trans.component",
                    ),
                ),
                (
                    "user",
                    models.ForeignKey(
                        db_index=False,
                        on_delete=django.db.models.deletion.CASCADE,
                        to=settings.AUTH_USER_MODEL,
                    ),
                ),
            ],
            options={
                "unique_together": {("user", "component")},
                "verbose_name": "contributor agreement",
                "verbose_name_plural": "contributor agreements",
            },
        ),
        migrations.CreateModel(
            name="AutoComponentList",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                (
                    "project_match",
                    weblate.trans.fields.RegexField(
                        default="^$",
                        help_text="Regular expression which is used to match project slug.",
                        max_length=200,
                        verbose_name="Project regular expression",
                    ),
                ),
                (
                    "component_match",
                    weblate.trans.fields.RegexField(
                        default="^$",
                        help_text="Regular expression which is used to match component slug.",
                        max_length=200,
                        verbose_name="Component regular expression",
                    ),
                ),
                (
                    "componentlist",
                    models.ForeignKey(
                        on_delete=django.db.models.deletion.CASCADE,
                        to="trans.componentlist",
                        verbose_name="Component list to assign",
                    ),
                ),
            ],
            options={
                "verbose_name": "Automatic component list assignment",
                "verbose_name_plural": "Automatic component list assignments",
            },
        ),
        migrations.CreateModel(
            name="Variant",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                (
                    "variant_regex",
                    weblate.trans.fields.RegexField(blank=True, max_length=190),
                ),
                ("key", models.CharField(max_length=576)),
                (
                    "component",
                    models.ForeignKey(
                        db_index=False,
                        on_delete=django.db.models.deletion.CASCADE,
                        to="trans.component",
                    ),
                ),
                (
                    "defining_units",
                    models.ManyToManyField(
                        related_name="defined_variants", to="trans.unit"
                    ),
                ),
            ],
            options={
                "unique_together": {("component", "key", "variant_regex")},
                "verbose_name": "variant definition",
                "verbose_name_plural": "variant definitions",
            },
        ),
        migrations.AddField(
            model_name="unit",
            name="variant",
            field=models.ForeignKey(
                blank=True,
                default=None,
                null=True,
                on_delete=django.db.models.deletion.SET_NULL,
                to="trans.variant",
            ),
        ),
        migrations.CreateModel(
            name="Category",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                (
                    "name",
                    models.CharField(
                        help_text="Display name",
                        max_length=100,
                        verbose_name="Category name",
                    ),
                ),
                (
                    "slug",
                    models.SlugField(
                        help_text="Name used in URLs and filenames.",
                        max_length=100,
                        validators=[weblate.utils.validators.validate_slug],
                        verbose_name="URL slug",
                    ),
                ),
                (
                    "category",
                    models.ForeignKey(
                        blank=True,
                        null=True,
                        on_delete=django.db.models.deletion.CASCADE,
                        related_name="category_set",
                        to="trans.category",
                        verbose_name="Category",
                    ),
                ),
                (
                    "project",
                    models.ForeignKey(
                        on_delete=django.db.models.deletion.CASCADE,
                        to="trans.project",
                        verbose_name="Project",
                    ),
                ),
            ],
            bases=(
                models.Model,
                weblate.trans.mixins.PathMixin,
                weblate.trans.mixins.CacheKeyMixin,
            ),
        ),
        migrations.AddField(
            model_name="component",
            name="category",
            field=models.ForeignKey(
                blank=True,
                null=True,
                on_delete=django.db.models.deletion.CASCADE,
                to="trans.category",
                verbose_name="Category",
            ),
        ),
        migrations.CreateModel(
            name="Change",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                ("timestamp", models.DateTimeField(auto_now_add=True)),
                (
                    "action",
                    models.IntegerField(
                        choices=[
                            (0, "Resource update"),
                            (1, "Translation completed"),
                            (2, "Translation changed"),
                            (5, "New translation"),
                            (3, "Comment added"),
                            (4, "Suggestion added"),
                            (6, "Automatic translation"),
                            (7, "Suggestion accepted"),
                            (8, "Translation reverted"),
                            (9, "Translation uploaded"),
                            (13, "New source string"),
                            (14, "Component locked"),
                            (15, "Component unlocked"),
                            (17, "Committed changes"),
                            (18, "Pushed changes"),
                            (19, "Reset repository"),
                            (20, "Merged repository"),
                            (21, "Rebased repository"),
                            (22, "Failed merge on repository"),
                            (23, "Failed rebase on repository"),
                            (28, "Failed push on repository"),
                            (24, "Parse error"),
                            (25, "Removed translation"),
                            (26, "Suggestion removed"),
                            (27, "Search and replace"),
                            (29, "Suggestion removed during cleanup"),
                            (30, "Source string changed"),
                            (31, "New string added"),
                            (32, "Bulk status change"),
                            (33, "Changed visibility"),
                            (34, "Added user"),
                            (35, "Removed user"),
                            (36, "Translation approved"),
                            (37, "Marked for edit"),
                            (38, "Removed component"),
                            (39, "Removed project"),
                            (41, "Renamed project"),
                            (42, "Renamed component"),
                            (43, "Moved component"),
                            (44, "New string to translate"),
                            (45, "New contributor"),
                            (46, "New announcement"),
                            (47, "New alert"),
                            (48, "Added new language"),
                            (49, "Requested new language"),
                            (50, "Created project"),
                            (51, "Created component"),
                            (52, "Invited user"),
                            (53, "Received repository notification"),
                            (54, "Replaced file by upload"),
                            (55, "License changed"),
                            (56, "Contributor agreement changed"),
                            (57, "Screenshot added"),
                            (58, "Screenshot uploaded"),
                            (59, "String updated in the repository"),
                            (60, "Add-on installed"),
                            (61, "Add-on configuration changed"),
                            (62, "Add-on uninstalled"),
                            (63, "Removed string"),
                            (64, "Removed comment"),
                            (65, "Resolved comment"),
                            (66, "Explanation updated"),
                            (67, "Removed category"),
                            (68, "Renamed category"),
                            (69, "Moved category"),
                        ],
                        default=2,
                    ),
                ),
                ("target", models.TextField(blank=True, default="")),
                (
                    "author",
                    models.ForeignKey(
                        null=True,
                        db_index=False,
                        on_delete=django.db.models.deletion.CASCADE,
                        related_name="+",
                        to=settings.AUTH_USER_MODEL,
                    ),
                ),
                (
                    "translation",
                    models.ForeignKey(
                        db_index=False,
                        null=True,
                        on_delete=django.db.models.deletion.CASCADE,
                        to="trans.translation",
                    ),
                ),
                (
                    "unit",
                    models.ForeignKey(
                        db_index=False,
                        null=True,
                        on_delete=django.db.models.deletion.CASCADE,
                        to="trans.unit",
                    ),
                ),
                (
                    "user",
                    models.ForeignKey(
                        null=True,
                        on_delete=django.db.models.deletion.CASCADE,
                        to=settings.AUTH_USER_MODEL,
                    ),
                ),
                (
                    "component",
                    models.ForeignKey(
                        null=True,
                        db_index=False,
                        on_delete=django.db.models.deletion.CASCADE,
                        to="trans.component",
                    ),
                ),
                (
                    "language",
                    models.ForeignKey(
                        null=True,
                        db_index=False,
                        on_delete=django.db.models.deletion.CASCADE,
                        to="lang.language",
                    ),
                ),
                ("old", models.TextField(blank=True, default="")),
                (
                    "project",
                    models.ForeignKey(
                        db_index=False,
                        null=True,
                        on_delete=django.db.models.deletion.CASCADE,
                        to="trans.project",
                    ),
                ),
                (
                    "comment",
                    models.ForeignKey(
                        db_index=False,
                        null=True,
                        on_delete=django.db.models.deletion.SET_NULL,
                        to="trans.comment",
                    ),
                ),
                (
                    "suggestion",
                    models.ForeignKey(
                        db_index=False,
                        null=True,
                        on_delete=django.db.models.deletion.SET_NULL,
                        to="trans.suggestion",
                    ),
                ),
                (
                    "alert",
                    models.ForeignKey(
                        db_index=False,
                        null=True,
                        on_delete=django.db.models.deletion.SET_NULL,
                        to="trans.alert",
                    ),
                ),
                (
                    "announcement",
                    models.ForeignKey(
                        db_index=False,
                        null=True,
                        on_delete=django.db.models.deletion.SET_NULL,
                        to="trans.announcement",
                    ),
                ),
                ("details", models.JSONField(default=dict)),
            ],
            options={
                "verbose_name": "history event",
                "verbose_name_plural": "history events",
            },
        ),
        migrations.AddIndex(
            model_name="component",
            index=models.Index(
                fields=["project", "allow_translation_propagation"],
                name="trans_compo_project_d044e1_idx",
            ),
        ),
        migrations.AddIndex(
            model_name="change",
            index=models.Index(
                fields=["timestamp", "action"], name="trans_chang_timesta_2565db_idx"
            ),
        ),
        migrations.AddIndex(
            model_name="change",
            index=models.Index(
                fields=["project", "action", "timestamp"],
                name="trans_chang_project_6d802b_idx",
            ),
        ),
        migrations.AddIndex(
            model_name="change",
            index=models.Index(
                fields=["language", "action", "timestamp"],
                name="trans_chang_languag_33816c_idx",  # codespell:ignore
            ),
        ),
        migrations.AddIndex(
            model_name="change",
            index=models.Index(
                fields=["project", "language", "action", "timestamp"],
                name="trans_chang_project_d37cd3_idx",
            ),
        ),
        migrations.AddIndex(
            model_name="change",
            index=models.Index(
                fields=["component", "action", "timestamp"],
                name="trans_chang_compone_5e4964_idx",
            ),
        ),
        migrations.AddIndex(
            model_name="change",
            index=models.Index(
                fields=["translation", "action", "timestamp"],
                name="trans_chang_transla_36a8a9_idx",
            ),
        ),
        migrations.AddIndex(
            model_name="change",
            index=models.Index(
                fields=["unit", "action", "timestamp"],
                name="trans_chang_unit_id_6d86c1_idx",
            ),
        ),
        migrations.AddIndex(
            model_name="change",
            index=models.Index(
                fields=["user", "action", "timestamp"],
                name="trans_chang_user_id_88ba38_idx",  # codespell:ignore
            ),
        ),
        migrations.AddIndex(
            model_name="unit",
            index=models.Index(
                django.db.models.functions.text.MD5(
                    django.db.models.functions.text.Lower("source")
                ),
                models.F("translation"),
                name="trans_unit_source_md5",
            ),
        ),
        migrations.AddIndex(
            model_name="unit",
            index=models.Index(
                django.db.models.functions.text.MD5(
                    django.db.models.functions.text.Lower("target")
                ),
                models.F("translation"),
                name="trans_unit_target_md5",
            ),
        ),
        migrations.AddIndex(
            model_name="unit",
            index=models.Index(
                django.db.models.functions.text.MD5(
                    django.db.models.functions.text.Lower("context")
                ),
                models.F("translation"),
                name="trans_unit_context_md5",
            ),
        ),
        migrations.RunPython(
            code=create_index,
            reverse_code=drop_index,
            atomic=False,
            elidable=True,
        ),
    ]
