# Copyright © Michal Čihař <michal@weblate.org>
#
# SPDX-License-Identifier: GPL-3.0-or-later
# Generated by Django 5.2.1 on 2025-06-13 18:13

from django.conf import settings
from django.db import migrations

from weblate.trans.actions import ACTIONS_CONTENT


def migrate_changes(apps, schema_editor) -> None:
    PendingUnitChange = apps.get_model("trans", "PendingUnitChange")
    Unit = apps.get_model("trans", "Unit")

    User = apps.get_model(*settings.AUTH_USER_MODEL.split("."))  # noqa: N806
    anonymous_user = None

    pending_units = Unit.objects.filter(pending=True)

    pending_changes = []
    for unit in pending_units:
        try:
            # can be None in some cases, for instance unit updated in the
            # repository outside of weblate
            author = (
                unit.change_set.filter(action__in=ACTIONS_CONTENT)
                .select_related("author")
                .order_by("-timestamp")[0]
                .author
            )
        except IndexError:
            author = None

        if author is None:
            # load anonymous user lazily as it doesn't exist when a new database
            # is created, such a database would not have any pending units either,
            # and this code block would never be executed.
            if anonymous_user is None:
                anonymous_user = User.objects.get(username=settings.ANONYMOUS_USER_NAME)
            author = anonymous_user

        pending_changes.append(
            PendingUnitChange(
                unit=unit,
                author=author,
                target=unit.target,
                explanation=unit.explanation,
                state=unit.state,
                source_unit_explanation=unit.source_unit.explanation,
                add_unit=unit.details.get("add_unit", False),
            )
        )
    PendingUnitChange.objects.bulk_create(pending_changes)

    pending_units.update(pending=False)


def reverse_migrate_changes(apps, schema_editor) -> None:
    PendingUnitChange = apps.get_model("trans", "PendingUnitChange")
    Unit = apps.get_model("trans", "Unit")
    pending_changes = PendingUnitChange.objects.select_related("unit").all()
    all_units_to_update = []
    add_unit_updates = []
    for change in PendingUnitChange.objects.select_related("unit").iterator():
        pk = change.unit.pk
        all_units_to_update.append(pk)
        if change.add_unit:
            add_unit_updates.append(pk)
    Unit.objects.filter(pk__in=all_units_to_update).update(pending=True)
    Unit.objects.filter(pk__in=add_unit_updates).update(details={"add_unit": True})
    pending_changes.delete()


class Migration(migrations.Migration):
    dependencies = [
        ("weblate_auth", "0006_group_enforced_2fa"),
        ("trans", "0039_pendingunitchange"),
    ]

    operations = [
        migrations.RunPython(migrate_changes, reverse_migrate_changes),
    ]
