# Copyright © Michal Čihař <michal@weblate.org>
#
# SPDX-License-Identifier: GPL-3.0-or-later

# Generated by Django 5.2.3 on 2025-07-07 06:44

from collections import defaultdict
from typing import Any

from django.db import migrations, models

ADDON_CONFIG_TO_FILE_FORMAT_PARAMS: dict[str, dict[str, Any]] = {
    "weblate.json.customize": {
        "file_formats": (
            "json",
            "json-nested",
            "webextension",
            "i18next",
            "i18nextv4",
            "arb",
            "go-i18n-json",
            "go-i18n-json-v2",
            "formatjs",
            "gotext",
        ),
        "format_params": {
            "sort_keys": "json_sort_keys",
            "use_compact_separators": "json_use_compact_separators",
            "indent": "json_indent",
            "style": "json_indent_style",
        },
    },
    "weblate.gettext.msgmerge": {
        "file_formats": ("po",),
        "format_params": {
            "previous": "po_keep_previous",
            "no_location": "po_no_location",
            "fuzzy": "po_fuzzy_matching",
        },
    },
    "weblate.gettext.customize": {
        "file_formats": (
            "po",
            "po-mono",
        ),
        "format_params": {
            "width": "po_line_wrap",
        },
    },
    "weblate.yaml.customize": {
        "file_formats": (
            "yaml",
            "ruby-yaml",
        ),
        "format_params": {
            "indent": "yaml_indent",
            "width": "yaml_line_wrap",
            "line_break": "yaml_line_break",
        },
    },
    "weblate.xml.customize": {
        "file_formats": (
            "ts",
            "plainxliff",
            "xliff",
            "poxliff",
            "resx",
            "aresource",
            "moko-resource",
            "cmp-resource",
            "tbx",
        ),
        "format_params": {
            "closing_tags": "xml_closing_tags",
        },
    },
}


def migrate_addons_config(apps, schema_editor) -> None:
    Component = apps.get_model("trans", "Component")
    Addon = apps.get_model("addons", "Addon")
    to_update = []

    addon_configs = collect_addon_configurations(apps)

    for component in Component.objects.all():
        file_format_params = merge_file_format_params(component, addon_configs)
        if file_format_params:
            component.file_format_params = file_format_params
            to_update.append(component)
    if to_update:
        Component.objects.bulk_update(to_update, ["file_format_params"])

    # delete stale addons
    # weblate.xml.customize was missed here, it is handled in
    # weblate/addons/migrations/0009_remove_migrated_customize_addons.py
    to_delete = [
        "weblate.json.customize",
        "weblate.gettext.customize",
        "weblate.yaml.customize",
    ]
    Addon.objects.filter(name__in=to_delete).delete()

    # remove configurations from msgmerge addon
    Addon.objects.filter(name="weblate.gettext.msgmerge").update(configuration={})


def convert_addon_config_to_file_format_params(addon) -> tuple[tuple, dict[str, Any]]:
    """Convert addon configuration to file format parameters."""
    attrs = ADDON_CONFIG_TO_FILE_FORMAT_PARAMS[addon.name]
    params = {}
    for addon_config, param_name in attrs["format_params"].items():
        if addon_config in addon.configuration:
            params[param_name] = addon.configuration[addon_config]
    return attrs["file_formats"], params


def collect_addon_configurations(apps) -> dict[str, list | dict]:
    """Collect and organize addon configurations by scope."""
    Addon = apps.get_model("addons", "Addon")

    addons = Addon.objects.filter(
        name__in=ADDON_CONFIG_TO_FILE_FORMAT_PARAMS.keys()
    ).select_related("component", "project")

    site_wide = []
    project_wide = defaultdict(list)
    component_wide = defaultdict(list)

    for addon in addons:
        file_formats, params = convert_addon_config_to_file_format_params(addon)
        config_entry = (file_formats, params)

        if not addon.component and not addon.project:
            site_wide.append(config_entry)
        elif addon.project and not addon.component:
            project_wide[addon.project_id].append(config_entry)
        elif addon.component:
            component_wide[addon.component_id].append(config_entry)
    return {
        "site_wide": site_wide,
        "project_wide": project_wide,
        "component_wide": component_wide,
    }


def merge_file_format_params(component, configs) -> dict[str, bool | int | str]:
    """Merge file format parameters for a component based on priority."""
    file_format_params = {}
    for file_formats, params in configs["site_wide"]:
        if component.file_format in file_formats:
            file_format_params.update(params)

    for file_formats, params in configs["project_wide"].get(component.project_id, []):
        if component.file_format in file_formats:
            file_format_params.update(params)

    for file_formats, params in configs["component_wide"].get(component.id, []):
        if component.file_format in file_formats:
            file_format_params.update(params)

    return file_format_params


class Migration(migrations.Migration):
    dependencies = [
        ("trans", "0045_alter_change_action"),
        ("addons", "0006_alter_addonactivitylog_component"),
    ]

    operations = [
        migrations.AddField(
            model_name="component",
            name="file_format_params",
            field=models.JSONField(
                blank=True, default=dict, verbose_name="File format parameters"
            ),
        ),
        migrations.RunPython(
            migrate_addons_config, migrations.RunPython.noop, elidable=True
        ),
    ]
