"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseProxy = exports.ProxyTarget = exports.SessionPinningFilter = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const cdk = require("@aws-cdk/core");
const util_1 = require("./private/util");
const rds_generated_1 = require("./rds.generated");
/**
 * SessionPinningFilter
 *
 * @see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-proxy.html#rds-proxy-pinning
 */
class SessionPinningFilter {
    constructor(
    /**
     * Filter name
     */
    filterName) {
        this.filterName = filterName;
    }
    /**
     * custom filter
     */
    static of(filterName) {
        return new SessionPinningFilter(filterName);
    }
}
exports.SessionPinningFilter = SessionPinningFilter;
_a = JSII_RTTI_SYMBOL_1;
SessionPinningFilter[_a] = { fqn: "@aws-cdk/aws-rds.SessionPinningFilter", version: "1.178.0" };
/**
 * You can opt out of session pinning for the following kinds of application statements:
 *
 * - Setting session variables and configuration settings.
 */
SessionPinningFilter.EXCLUDE_VARIABLE_SETS = new SessionPinningFilter('EXCLUDE_VARIABLE_SETS');
/**
 * Proxy target: Instance or Cluster
 *
 * A target group is a collection of databases that the proxy can connect to.
 * Currently, you can specify only one RDS DB instance or Aurora DB cluster.
 */
class ProxyTarget {
    constructor(dbInstance, dbCluster) {
        this.dbInstance = dbInstance;
        this.dbCluster = dbCluster;
    }
    /**
     * From instance
     *
     * @param instance RDS database instance
     */
    static fromInstance(instance) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_rds_IDatabaseInstance(instance);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromInstance);
            }
            throw error;
        }
        return new ProxyTarget(instance, undefined);
    }
    /**
     * From cluster
     *
     * @param cluster RDS database cluster
     */
    static fromCluster(cluster) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_rds_IDatabaseCluster(cluster);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromCluster);
            }
            throw error;
        }
        return new ProxyTarget(undefined, cluster);
    }
    /**
     * Bind this target to the specified database proxy.
     */
    bind(proxy) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_rds_DatabaseProxy(proxy);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bind);
            }
            throw error;
        }
        const engine = this.dbInstance?.engine ?? this.dbCluster?.engine;
        if (!engine) {
            const errorResource = this.dbCluster ?? this.dbInstance;
            throw new Error(`Could not determine engine for proxy target '${errorResource?.node.path}'. ` +
                'Please provide it explicitly when importing the resource');
        }
        const engineFamily = engine.engineFamily;
        if (!engineFamily) {
            throw new Error(`Engine '${util_1.engineDescription(engine)}' does not support proxies`);
        }
        // allow connecting to the Cluster/Instance from the Proxy
        this.dbCluster?.connections.allowDefaultPortFrom(proxy, 'Allow connections to the database Cluster from the Proxy');
        this.dbInstance?.connections.allowDefaultPortFrom(proxy, 'Allow connections to the database Instance from the Proxy');
        return {
            engineFamily,
            dbClusters: this.dbCluster ? [this.dbCluster] : undefined,
            dbInstances: this.dbInstance ? [this.dbInstance] : undefined,
        };
    }
}
exports.ProxyTarget = ProxyTarget;
_b = JSII_RTTI_SYMBOL_1;
ProxyTarget[_b] = { fqn: "@aws-cdk/aws-rds.ProxyTarget", version: "1.178.0" };
/**
 * Represents an RDS Database Proxy.
 *
 */
class DatabaseProxyBase extends cdk.Resource {
    grantConnect(grantee, dbUser) {
        if (!dbUser) {
            throw new Error('For imported Database Proxies, the dbUser is required in grantConnect()');
        }
        const scopeStack = cdk.Stack.of(this);
        const proxyGeneratedId = scopeStack.splitArn(this.dbProxyArn, cdk.ArnFormat.COLON_RESOURCE_NAME).resourceName;
        const userArn = scopeStack.formatArn({
            service: 'rds-db',
            resource: 'dbuser',
            resourceName: `${proxyGeneratedId}/${dbUser}`,
            arnFormat: cdk.ArnFormat.COLON_RESOURCE_NAME,
        });
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['rds-db:connect'],
            resourceArns: [userArn],
        });
    }
}
/**
 * RDS Database Proxy
 *
 * @resource AWS::RDS::DBProxy
 */
class DatabaseProxy extends DatabaseProxyBase {
    constructor(scope, id, props) {
        super(scope, id, { physicalName: props.dbProxyName || id });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_rds_DatabaseProxyProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, DatabaseProxy);
            }
            throw error;
        }
        const role = props.role || new iam.Role(this, 'IAMRole', {
            assumedBy: new iam.ServicePrincipal('rds.amazonaws.com'),
        });
        for (const secret of props.secrets) {
            secret.grantRead(role);
        }
        const securityGroups = props.securityGroups ?? [
            new ec2.SecurityGroup(this, 'ProxySecurityGroup', {
                description: 'SecurityGroup for Database Proxy',
                vpc: props.vpc,
            }),
        ];
        this.connections = new ec2.Connections({ securityGroups });
        const bindResult = props.proxyTarget.bind(this);
        if (props.secrets.length < 1) {
            throw new Error('One or more secrets are required.');
        }
        this.secrets = props.secrets;
        this.resource = new rds_generated_1.CfnDBProxy(this, 'Resource', {
            auth: props.secrets.map(_ => {
                return {
                    authScheme: 'SECRETS',
                    iamAuth: props.iamAuth ? 'REQUIRED' : 'DISABLED',
                    secretArn: _.secretArn,
                };
            }),
            dbProxyName: this.physicalName,
            debugLogging: props.debugLogging,
            engineFamily: bindResult.engineFamily,
            idleClientTimeout: props.idleClientTimeout?.toSeconds(),
            requireTls: props.requireTLS ?? true,
            roleArn: role.roleArn,
            vpcSecurityGroupIds: cdk.Lazy.list({ produce: () => this.connections.securityGroups.map(_ => _.securityGroupId) }),
            vpcSubnetIds: props.vpc.selectSubnets(props.vpcSubnets).subnetIds,
        });
        this.dbProxyName = this.resource.ref;
        this.dbProxyArn = this.resource.attrDbProxyArn;
        this.endpoint = this.resource.attrEndpoint;
        let dbInstanceIdentifiers;
        if (bindResult.dbInstances) {
            // support for only single instance
            dbInstanceIdentifiers = [bindResult.dbInstances[0].instanceIdentifier];
        }
        let dbClusterIdentifiers;
        if (bindResult.dbClusters) {
            dbClusterIdentifiers = bindResult.dbClusters.map((c) => c.clusterIdentifier);
        }
        if (!!dbInstanceIdentifiers && !!dbClusterIdentifiers) {
            throw new Error('Cannot specify both dbInstanceIdentifiers and dbClusterIdentifiers');
        }
        const proxyTargetGroup = new rds_generated_1.CfnDBProxyTargetGroup(this, 'ProxyTargetGroup', {
            targetGroupName: 'default',
            dbProxyName: this.dbProxyName,
            dbInstanceIdentifiers,
            dbClusterIdentifiers,
            connectionPoolConfigurationInfo: toConnectionPoolConfigurationInfo(props),
        });
        bindResult.dbClusters?.forEach((c) => proxyTargetGroup.node.addDependency(c));
    }
    /**
     * Import an existing database proxy.
     */
    static fromDatabaseProxyAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_rds_DatabaseProxyAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromDatabaseProxyAttributes);
            }
            throw error;
        }
        class Import extends DatabaseProxyBase {
            constructor() {
                super(...arguments);
                this.dbProxyName = attrs.dbProxyName;
                this.dbProxyArn = attrs.dbProxyArn;
                this.endpoint = attrs.endpoint;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.dbProxyName,
            targetType: secretsmanager.AttachmentTargetType.RDS_DB_PROXY,
        };
    }
    grantConnect(grantee, dbUser) {
        if (!dbUser) {
            if (this.secrets.length > 1) {
                throw new Error('When the Proxy contains multiple Secrets, you must pass a dbUser explicitly to grantConnect()');
            }
            // 'username' is the field RDS uses here,
            // see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-proxy.html#rds-proxy-secrets-arns
            dbUser = this.secrets[0].secretValueFromJson('username').unsafeUnwrap();
        }
        return super.grantConnect(grantee, dbUser);
    }
}
exports.DatabaseProxy = DatabaseProxy;
_c = JSII_RTTI_SYMBOL_1;
DatabaseProxy[_c] = { fqn: "@aws-cdk/aws-rds.DatabaseProxy", version: "1.178.0" };
/**
 * ConnectionPoolConfiguration (L2 => L1)
 */
function toConnectionPoolConfigurationInfo(props) {
    return {
        connectionBorrowTimeout: props.borrowTimeout?.toSeconds(),
        initQuery: props.initQuery,
        maxConnectionsPercent: props.maxConnectionsPercent,
        maxIdleConnectionsPercent: props.maxIdleConnectionsPercent,
        sessionPinningFilters: props.sessionPinningFilters?.map(_ => _.filterName),
    };
}
//# sourceMappingURL=data:application/json;base64,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