# UniConv: Универсальный конвертер инженерных единиц

[![PyPI version](https://badge.fury.io/py/uniconv.svg)](https://badge.fury.io/py/uniconv)
[![License: MIT](https://img.shields.io/badge/License-MIT-yellow.svg)](https://opensource.org/licenses/MIT)

Универсальный и расширяемый конвертер инженерных единиц измерения, ориентированный на термодинамические и теплогидравлические расчеты.

## Особенности

- **Гибкость**: Легко добавляйте новые физические параметры (вязкость, скорость) и единицы измерения "на лету".
- **Специализация**: "Из коробки" содержит все основные единицы, используемые в теплотехнике (кгс/см², ккал/кг, т/ч и др.).
- **Соответствие ГОСТ**: Обозначения единиц приведены в соответствие с ГОСТ 8.417.
- **Надежность**: Код покрыт набором автоматизированных тестов.
- **Чистая архитектура**: Логика конвертации отделена от физических констант для удобства поддержки.

## Установка

Библиотека может быть установлена из PyPI с помощью pip:

```bash
pip install uniconv
```

## Быстрый старт

Использование конвертера интуитивно понятно.

```python
from uniconv import UnitConverter

# 1. Создаем экземпляр конвертера
uc = UnitConverter()

# 2. Конвертируем значение
# Пример: перевести давление 10 бар в технические атмосферы (кгс/см²)
p_kgf = uc.convert(10, from_unit="бар", to_unit="кгс/см²", parameter_type="pressure")
print(f"10 бар = {p_kgf:.3f} кгс/см²")
# >> 10 бар = 10.197 кгс/см²

# 3. Получаем список доступных единиц для параметра
pressure_units = uc.get_available_units("pressure")
print("Доступные единицы давления:", pressure_units)
# >> Доступные единицы давления: ['кгс/см²', 'ат', 'Па', 'кПа', 'МПа', 'бар', 'атм', 'мм рт. ст.']

# 4. Расширение "на лету"
# Добавляем параметр "скорость" и новую единицу "км/ч"
uc.add_parameter(
    parameter_type="speed",
    parameter_name="Скорость",
    base_unit_symbol="м/с",
    base_unit_name="Метр в секунду"
)
uc.add_unit(
    "speed",
    unit_symbol="км/ч",
    unit_name="Километр в час",
    to_base=1/3.6
)

# Используем новую единицу
speed_ms = uc.convert(100, from_unit="км/ч", to_unit="м/с", parameter_type="speed")
print(f"100 км/ч = {speed_ms:.2f} м/с")
# >> 100 км/ч = 27.78 м/с
```

## Лицензия

Этот проект распространяется под лицензией MIT. Детали см. в файле [LICENSE](LICENSE).
