# /// script
# dependencies = ["pymhf[gui]>=0.1.15"]
#
# [tool.pymhf]
# exe = "NMS.exe"
# steam_gameid = 275850
# start_paused = false
#
# [tool.pymhf.gui]
# always_on_top = true
#
# [tool.pymhf.logging]
# log_dir = "."
# log_level = "info"
# window_name_override = "NMS Example gravity manipulator"
# ///

import logging
import ctypes
from dataclasses import dataclass, field

from pymhf import Mod, ModState
from pymhf.core.hooking import on_key_pressed
from pymhf.gui import FLOAT
import nmspy.data.types as nms

# A quick mod used to change the gravity multiplier on all planets simultaneously, utilizing pyMHF's auto-gui.

logger = logging.getLogger("GravMod")


@dataclass
class gravModState(ModState):
    # A special class inheriting from ModState which persists between mod Hot Reloads, allowing mod developers
    # to cache pointers, values etc.
    gravity: float = 1
    planets: list[nms.cGcPlanet] = field(default_factory=list)


class gravityManipulator(Mod):
    # General "Nice To Have"s
    __author__ = ["ThatBomberBoi", "monkeyman192"]
    __description__ = "Gravity Manipulator"
    __version__ = "0.3"

    # Create an instance of the persistant ModState Class in your mod class.
    state = gravModState()

    def __init__(self):
        super().__init__()
        self.should_print = False

    # Used to define a Float Type with a label in the Mod GUI, autogenerated by pyMHF.
    @property
    @FLOAT("Gravity Multiplier:")
    def gravMult(self):
        return self.state.gravity

    # Used to actually update the persisted value with the one input by the user in the GUI.
    @gravMult.setter
    def gravMult(self, value):
        self.state.gravity = value

    # Functions are hooked by specifying the function to be hooked from the list of in game functions
    # available. You can specify whether you want your detour to run either before or after the original
    # function as shown below.
    @nms.cGcPlanet.SetupRegionMap.after
    def onRegionMap(self, this: ctypes._Pointer[nms.cGcPlanet]):
        # Include each in-game function's arguments seperately in the function, or use *args to get all
        # arguments without knowing them prior.
        # In this case we know that the argument is `this` which is a pointer to the instance of `cGcPlanet`
        # which is automatically passed into this function by the game.
        planet = this.contents
        self.state.planets.append(this.contents)
        biome = planet.mPlanetGenerationInputData.Biome
        logger.info(f"Generated a {biome.name} Planet!")
        logger.debug(f"Found {len(self.state.planets)} Planets So Far")

    @on_key_pressed("o")
    def modifyGravity(self):
        for planet in self.state.planets:
            # Call an in-game function directly from your mod code.
            # You will need to provide the arguments for the in-game function
            planet.UpdateGravity(self.state.gravity)
        logger.info(
            f"Set Planetary Gravity Multiplier To {self.state.gravity} For All Planets"
        )
