# asimtote.command
#
# Copyright (C) Robert Franklin <rcf34@cam.ac.uk>



"""Configuration command parser module.

This module contains the abstract base class for parsing configuration
file commands into configuration dictionaries.
"""



# --- imports ---



import re



# --- constants ---



# different debugging levels and what they mean (they're used in lots of
# places so it seems sensible to avoid hard coding the values, in case
# anything changes)

DEBUG_COMMAND_MATCH = 1     # display match()ed commands (or 'no matches')
DEBUG_COMMAND_NOMATCH = 2   # display commands which did not match
DEBUG_COMMAND_ARGS = 3      # display arguments to match()ed commands

# maximum debug level
DEBUG_COMMAND_MAX = DEBUG_COMMAND_ARGS



# --- classes ---



class IndentedContextualCommand:
    """This is an abstract class that represents a command from a
    configuration file using indented contexts.

    It includes the context in which it occurs (the indented block), a
    regular expression to match it (which will include groups to extract
    parameters from), if it enters a subcontext and an parse() method
    to run, if there is a match (which might store configuration
    information).

    Actual commands will inherit from this and override or implement
    some of the constants:

    context -- the context (indented section name) in which the command
    applies; the default is '.', which represents the top level (no
    indent, i.e. no subcontext)

    match -- an uncompiled regular expression string which fully matches
    (with re.fullmatch()) the command, case insensitively

    enter_context -- if not Null, it will be a string specifying the
    subcontext to enter, if the command matches

    In addtion, the parse() method is likely to need to be implemented.
    """


    context = '.'
    match = None
    enter_context = None


    def __init__(self):
        """The constructor simply compiles the string-based regular
        expression and stores it once, for performance.
        """

        # if there's a mistake in a regular expression, the trace just
        # displays the commands and not which particular command was at
        # fault, so we catch that exception and display the command
        self._match_re = re.compile(self.match + r"$", re.IGNORECASE)


    def parse(self, cfg):
        """This method is called when the command is matched.  It will
        typically be used to update the configuration dictionary.

        The default is not to do anything - child classes will likely
        need to implement this.

        The configuration dictionary supplied to this function is not
        necessarily the entire configuration for the device but what is
        returned by the parse() method of the parent context.  This
        allows a parent context to pass on just the configuration
        dictionary for a portion of the device, such as an individual
        interface.  This makes it easier to update the correct portion
        of the configuration, without having to index through the entire
        structure.

        Keyword arguments:

        cfg -- the dictionary for this part of the configuration, or the
        the full configuration, as returned by the parent context's
        parse() method

        *groups -- the re.Match.groupdict() returned after matching the
        command regular expression will be expanded into keyword
        arguments as '**groupdict()', so there'll be one arguemnt for
        each named groups, with the corresponding name (e.g. if the
        regular expression contains '(<?P<addr>...)' an argument will
        be provided called 'addr' with the matching pattern)

        The return value is the dictionary to use as the 'cfg' argument
        of any subcontext parse() calls, or None if the same
        configuration dictionary as the parent context is to be used.
        """

        pass


    def ismatch(self, line):
        """This method returns if this command matches the supplied
        command line.  The regular expression must be matched
        in its entirity (it uses re.fullmatch()) and does not include
        any leading indenting due to the context.
        """

        return self._match_re.fullmatch(line)
