# asimtote.convert
#
# Copyright (C) Robert Franklin <rcf34@cam.ac.uk>



"""Configuration element converters module.

This module contains abstract classes and functions to convert elements
of configurations.
"""



# --- imports ---



from deepops import deepget



# --- classes ---



class Convert:
    """This abstract class handles converting the difference between an
    old ('from') configuration item to the corresponding new ('to')
    configuration item.

    The main difference process will use deepops.deepdiff() to work out
    what has been removed and what updated (added/changed) between the
    two configurations.

    Individual differences are checked using child classes, which
    specify the part of the configuration directionaries where they
    occur and the remove() and update() methods called.  For example, if
    the hostname is changed, a 'hostname' converter would specify the
    part of the configuration dictionary where the hostname is stored
    and the update() method would return the commands required to change
    it to the new value.

    The methods called to do the conversions (documented in the methods
    themselves) are:

    * add(new, *args) -- appears in the new configuration but was not
      present in the old configuration; calls update(None, new, new,
      *args) by default.

    * delete(old, rem, new, *args) -- appears in the old configuration
      but is not present in the new configuration; calls remove(old,
      *args) by default)

    * remove(old, *args) -- appears in the old configuration but is not
      present in the new configuration; calls truncate(old, old, None,
      *args) by default.

    * truncate(old, rem, new, *args) -- appears in the remove
      differences but still exists in the new configuration - i.e. is
      being partially removed.

    * update(old, upd, new, *args) -- appers in the update differences
      and existed in the old configuration - i.e. is being changed or
      added to.

    * trigger(new, *args) -- called when a trigger causes a converter to
      fire, and would not otherwise, e.g. with a remove() or update(),
      as a remove() or update() has not been called, nothing has been
      changed in the configuration, so only a 'new' argument is
      supplied; calls update(new, None, new, *args) by default.

    In addition, there are some filter methods which are called with the
    same wildcard arguments as the conversion action methods, above.  If
    this returns a false value, this particular conversion will be
    skipped:

    * filter_delete() - called before a delete(), remove() or
      truncate().

    * filter_update() - called before an add() or update().

    * filter_trigger() - called before a trigger().

    * filter() - by default, the above method calls the same filter()
      method allowing all conversion actions to be filtered in one
      place.

    In most cases, only the remove() and update() methods need defining
    as the add() and trigger() methods call update(), and truncate() is
    rarely required, but there some odd cases where they are, hence
    their inclusion.

    The 'context', 'cmd' and 'ext' parameters, below, are all tuples
    giving parts of the path through the keys of the configuration
    dictionary to match for the specific converter.  These can be
    literal keys, the value None, or a set.  The value None or a set
    specify a wildcard match to any value or limited to the specified
    set of values.  For example, '("interface", None)' to match all
    interfaces; ("redistribute", { "connected", "static" }) will match
    only connected or static redistributions.

    The part of the configuration dictionary that matches the 'context +
    cmd' parts of the path will be passed as the 'old/new' and 'rem/upd'
    parameters to the action methods.

    The child classes can override the following values:

    context -- the path to the context containing the commands used in
    this conversion: if this context is removed in its entirety, the
    remove() action for this converter will not be called as the
    removal of the context will remove it; this variable defaults to
    the empty tuple, for a command at the top level

    cmd -- the path for this command in the configuration, following
    the context; this part of the dictionary will be passed in the
    'old/new' and 'rem/upd' parameters to the action methods; this
    variable defaults to None and must be defined by a child class

    ext -- an extension to path which must be matched, but this part is
    not used to further specify the part of the dictionaries passed to
    the action functions for the 'old/new' and 'rem/upd' parameters:
    this is useful when a higher level in the configuration dictionary
    is required in the action function

    block -- the block of configuration in which this converter is to be
    applied (see DiffConfig)

    trigger_blocks -- a set of block names to be triggered when a match
    is made against this converter and the conversion methods return
    some output (i.e. commands to do the conversion)

    empty_trigger -- if this is set to True, the converter will always
    fire the named triggers, even if the converter generated no output

    context_offset -- this is used by the get_args() method to adjust
    the boundary between the context and cmd+ext parts when fetching
    wildcard argument values to be passed to converter methods: it
    defaults to 0, causing it to use the natural boundary between the
    context and cmd/ext) but can be adjusted to (typically) negative
    values and include part of the context as the local arguments
    """


    context = tuple()
    cmd = None    # leave as None to cause an error if not defined in child
    ext = tuple()
    block = None
    sort_key = None
    trigger_blocks = set()
    empty_trigger = False
    context_offset = 0


    def __init__(self):
        """The constructor just precalculates some details to optimise
        the repeated processing of the converter matches.
        """

        # to avoid confusing error messages, just check all the
        # definitions for this converter class are all tuples and not
        # something like simple strings
        if not (isinstance(self.context, tuple)):
            raise TypeError("%s: definition not tuple: context"
                                % type(self).__name__)
        if not (isinstance(self.cmd, tuple)):
            raise TypeError("%s: definition not tuple: cmd"
                                % type(self).__name__)
        if not (isinstance(self.ext, tuple)):
            raise TypeError("%s: definition not tuple: ext"
                                % type(self).__name__)

        # calculate and store a few things, for efficiency
        self._path_full = self.context + self.cmd + self.ext
        self._context_len = len(self.context)
        self._path_len = len(self.context + self.cmd)

        # store the set of indices of wildcard elements of the path (we
        # need these to get the argument list to pass to the converter
        # action methods)
        self.wildcard_indices = [
            i for i, v in enumerate(self._path_full)
                if (v is None) or isinstance(v, set) ]

        # set _context_args value for this converter, specifies the
        # number of wildcard arguments in the context (vs the cmd+ext),
        # and used by get_args() method
        #
        # by default, this counts the number of wildcard arguments in
        # the context but can be adjusted with context_offset
        #
        # this is useful when writing converters which can work in
        # several different contexts (such as those for commands in BGP
        # configuration)
        #
        # if set to negative values, additional wildcard arguments from
        # the context will be included in the cmd/ext side; positive
        # values will shift the other way
        self._context_args = (
            len([ i for i, v in enumerate(self.context)
                     if (v is None) or isinstance(v, set) ])
            + self.context_offset)


    def _path_matches(self, d, path):
        """This method is used to recursively step along the paths.  It
        is initially called by full_matches() from the top of the path -
        see that for more information.
        """

        # if the path is finished, return a single result with an
        # empty list, as there are no more path elements
        #
        # note that this is different from an empty list (which
        # would mean no matches
        if not path:
            return [ [] ]

        # get the path head and tail to make things easier to read
        path_head, path_tail = path[0], path[1:]

        # if this element is not a type we can iterate or search
        # through (we've perhaps reached a value or a None), or the
        # dictionary is empty, there are no matches, so just return
        # an empty list (which will cause all higher levels in the
        # path that matched to be discarded, giving no resulsts)
        if not isinstance(d, (dict, list, set)) or (not d):
            return []

        # if the path element is None, we're doing a wildcard match
        if (path_head is None) or (isinstance(path_head, set)):
            # initialise an empty list for all returned matching
            # results
            results = []

            # go through all the keys at this level in the dictonary
            for d_key in d:
                if (path_head is None) or (d_key in path_head):
                    # are there levels below this one?
                    if path_tail:
                        # yes - recursively get the matches from under
                        # this key
                        for matches in self._path_matches(d[d_key], path_tail):
                            # add this match to the list of matches,
                            # prepending this key onto the start of the
                            # path
                            results.append([d_key] + matches)
                    else:
                        # no - just add this result (this is a minor
                        # optimisation, as well as avoiding an error by
                        # trying to index into a non-dict type), above
                        results.append([d_key])

            return results

        # we have a literal key to match - if it's found,
        # recursively get the matches under this level
        if path_head in d:
            return [ ([path_head] + matches)
                            for matches
                            in self._path_matches(d[path_head], path_tail) ]

        # we have no matches, return the empty list
        return []


    def full_matches(self, d):
        """This method takes a dictionary and returns any matches for
        the full path in it, as a list of paths; each path is, itself, a
        list of keys.

        If any of the elements of the path are None, this is treated as
        a wildcard and will match all keys at that level.

        If there are no matching entries, the returned list will be
        empty.

        The returned list of matches is not guaranteed to be in any
        particular order.
        """

        return self._path_matches(d, self._path_full)


    def explicit_context_matches(self, d, trigger_context):
        """This method is used when checking for trigger matches.

        It is similar to full_matches() except the explicit context
        (without wildcards) is supplied and the command and extended
        paths appended; the matches are returned as a list.

        The context will have been taken from the converter which set up
        the trigger (with the wildcarded fields completed) so we only
        want to match parts of the configuration which have that
        explicit context.
        """

        # check the supplied context to confirm that it matches that of
        # this converter (allowing for wildcards) - if it does not,
        # return an empty list of matches
        #
        # this is because it may have been triggered in a completely
        # different context and we need to make sure that this converter
        # applies to the same context as the trigger

        # if the length of the trigger context differs from ours, this
        # is definitely not the same context
        if len(trigger_context) != len(self.context):
            return []

        # check the items in our context and the trigger context
        for s, t in zip(self.context, trigger_context):
            # if our item is a set, and the trigger is not in it, no match
            if isinstance(s, set) and (t not in s):
                return []

            # if our item is a literal, and the trigger is not the same,
            # no match
            if (s is not None) and (s != t):
                return []

        # we know the contexts are the same, so find all the matches in
        # the supplied dictionary
        return self._path_matches(d, tuple(trigger_context) + self.cmd + self.ext)


    def context_removed(self, d, match):
        """This method takes a remove dictionary, as returned by
        deepdiff(), and a specific match into it (which must be from
        the same Convert object as the method is called on) and returns
        True iff the context for this converter is either:

        1.  In the remove dictionary entirely and exactly (i.e. it is
        empty at the end of the match path), or

        2.  The dictionary does not contain the match path but the match
        path runs out at a point in the dictionary where it is empty
        (indicating everything below this is removed).
        """

        # if this converter has no containing context, we definitely
        # can't be removing it, so will need to remove the item
        if not self.context:
            return False

        # get the part of the match for this item which covers the
        # context of this converter
        match_context = match[0 : self._context_len]

        # loop whilst there is more match remaining and the remove
        # dictionary still has items in it
        while match_context and d:
            # get the head of the match path and remove it from the path
            # if we iterate after matching
            match_head = match_context.pop(0)

            # if the the head of the match path is not in the
            # dictionary, we're not its context so return no match
            if match_head not in d:
                return False

            # move down to the next level in the remove dictionary
            d = d[match_head]


        # if the remove dictionary is not empty, we've either reached
        # the end of the match path and something remains, so we won't
        # be removing the entire context, or the match path was not
        # completely traversed but something remains
        #
        # either way, this converter will not be removing the context of
        # this item so return no match
        if d:
            return False

        # the remove dictionary is empty, so we've either reached the
        # end of the match path and are removing everything, or we ran
        # out of remove dictionary, traversing the path, but everything
        # at this point is going
        #
        # regardless, this converter will already be removing the
        # context of this match so this item doesn't need to be
        # explicitly removed, itself
        return True


    def get_path(self):
        """Get the non-extended path for this converter (i.e. the
        'context' and 'cmd' paths joined, excluding the 'ext').
        """

        return self.context + self.cmd


    def _get_sort_key(self):
        """Get the key used to order converters within a block.

        This consists of the context with the sort_key appended (which
        is a tuple of elements).  If sort_key is None, cmd+ext will be
        assumed as a default.
        """

        key = self.context + (self.sort_key or (self.cmd + self.ext))
        return [ "" if i is None else i for i in key ]


    def get_match_context(self, match):
        """Return the context portion of a particular match for this
        converter.

        This is used when a trigger is set, to extract the context of a
        converter.
        """

        return match[0 : self._context_len]


    def get_cfg(self, cfg, match):
        """This method returns the configuration to be passed to the
        converter's action methods [remove() and update()].

        By default, it indexes through the configuration using the path
        in the converter.  Converters may wish to override this, in some
        cases, for performance (perhaps if the entire configuration is
        to be returned).

        If the specified match path is not found, or there was a problem
        indexing along the path, None is returned, rather than an
        exception raised.
        """

        # try to get the specified matching path
        try:
            return deepget(cfg, *match[0 : self._path_len])

        except TypeError:
            # we got a TypeError so make the assumption that we've hit
            # a non-indexable element (such as a set) as the final
            # element of the path, so just return None
            return None


    def get_ext(self, cfg, *args):
        """This method gets the extension part of the path, given a
        configuration dictionary starting at the path (i.e. what is
        passed as 'old/new' and 'rem/upd' in the action methods).

        An action method [remove() or update()] can use this to get the
        extension portion without needing to explicitly index through
        it.

        If the extension contains any wildcard elements (= None), these
        will be filled from the supplied 'args'.  The number of args
        must be the same (or larger) than the number of wildcard fields,
        else an error will occur.
        """

        # get the extension path, filling in the wildcards (= None) with
        # the supplied matched arguments
        ext_match = []
        for i in self.ext:
            if i is None:
                i = args[0]
                args = args[1:]
            ext_match.append(i)

        return deepget(cfg, *ext_match)


    def get_args(self, match):
        """This method returns a the wildcarded parts of the specified
        match as a 2-tuple with each element a list.  The first element
        is the list of wildcard arguments from the context and the
        second is from the cmd+ext part.

        The boundary between the context and the cmd+ext parts is
        normally determined automatically but can be adjusted by setting
        context_offset in the converter (typically to a negative value,
        to include part of the context in the local arguments to the
        converter).
        """

        wildcard_args = [ match[i] for i in self.wildcard_indices ]
        return (wildcard_args[:self._context_args],
                wildcard_args[self._context_args:])


    def filter(self, context_args, *local_args):
        """Specifies if a particular conversion is to be filtered out
        (return = False) or executed (True), given the wildcard match
        arguments.

        This filtering could be done by the action methods delete(),
        remove(), truncate(), add(), update() and trigger() but
        sometimes it is useful to separate this out to operate
        independently, if a filter method is to be shared across a set
        of subclasses.

        A side benefit is it's also a bit quicker as the filtering
        happens before all the differences are calculated.

        This method is not called directly by convert() but provides a
        single method that the individual action filtering methods
        (filter_delete(), filter_update() and filter_trigger()) will
        call by default, if all types of action are to be equally
        filtered.

        This method (and its associated action-specific methods) differs
        from the trigger_set_filter() method (and its methods) as this
        is called on the converter which has been triggered, rather than
        on the converter which sets up the trigger.

        Keyword arguments:

        context_args -- [only] the wildcard arguments from the context
        part of path, supplied as an iterable (typically a list).  For
        example, if the context matches ["interface", None], this will
        be the the value of None (as the list [a]).

        *local_args -- [only] the wildcard arguments in the command and
        extensions parts of the path, supplied as a number of discrete
        arguments.  For example, if the command and extension matches
        ["standby", "group", None, "ip-secondary", None], this will be
        the value of the two None fields (as the list [a, b]).
        """

        return True


    def filter_delete(self, context_args, *local_args):
        """See filter() - this method is called by convert() before a
        delete()/remove()/truncate() action is called.
        """

        return self.filter(context_args, *local_args)


    def filter_update(self, context_args, *local_args):
        """See filter() - this method is called by convert() before an
        add()/update() action is called.
        """

        return self.filter(context_args, *local_args)


    def filter_trigger(self, context_args, *local_args):
        """See filter() - this method is called by convert() before a
        trigger() action is called.
        """

        return self.filter(context_args, *local_args)


    def trigger_set_filter(self, trigger, old, new, context_args, *local_args):
        """If a converter action has fired and trigger_blocks is
        specified, this method will be called for each trigger, along
        with the same parameters supplied to the action method.  If
        True is returned, the trigger will be set up; if False is
        returned, the trigger will NOT be set up.

        This method can be used to prevent a trigger being set up in
        certain situations.

        By default, the method returns True, which performs no
        filtering (so triggers are always set up).

        Note that this differs from the filter() method (and its
        associated methods for each type of action) as this is called on
        the converter that sets up the trigger, rather than on the
        converter which is triggered later.

        Keyword arguments:

        old, new, context_args, *local_args -- see the update() method
        """

        return True


    def trigger_set_filter_delete(
            self, trigger, old, rem, new, context_args, *local_args):

        """See trigger_set_filter() - this method is called before a
        trigger is set up following a delete()/remove()/truncate()
        action method.
        """

        return self.trigger_set_filter(
                   trigger, old, new, context_args, *local_args)


    def trigger_set_filter_update(
            self, trigger, old, upd, new, context_args, *local_args):

        """See trigger_set_filter() - this method is called before a
        trigger is set up following a add()/update() action method.
        """

        return self.trigger_set_filter(
                   trigger, old, new, context_args, *local_args)


    def delete(self, old, rem, new, context_args, *local_args):
        """The delete() method is called when the specified path is in
        the remove differences (i.e. it's in the old configuration but
        but not in the new configuration), unless the containing context
        is being removed in its entirety.

        The default behaviour of this method is to call remove() with
        the old value for 'old'.

        This method may be required in odd situations where there is a
        full removal of a configuration element matched by the 'cmd' and
        'ext' attributes but there still something in the context (at a
        higher level) and this is required to do the removal.  Normally
        the remove() method should be sufficient, however.

        The difference between delete() and truncate() is that the
        latter is called when something remains in the configuration
        value identified by the complete match (i.e. context + cmd +
        ext), so only some items are being removed.  delete(), on the
        other hand, is called when the configuration item identified by
        the match is empty; 'rem' and 'new' are identify the
        configuration identified by just the context and not the
        complete match and may be required when deleting items.

        Keyword arguments:

        old -- the value of the dictionary item at the matching path in
        the old configuration dictionary (sometimes, the full details of
        the old configuration may be required to remove it)

        rem -- the value of the dictionary item at the matching path in
        the remove differences dictionary (sometimes, it's necessary to
        know only what is removed from the new configuration)

        new -- the value of the dictionary item at the matching path in
        the new configuration dictionary (sometimes, parts of the old
        configuration cannot be removed and the entire new configuration
        is required to replace it)

        context_args -- [only] the wildcard arguments from the context
        part of path, supplied as an iterable (typically a list).  For
        example, if the context matches ["interface", None], this will
        be the the value of None (as the list [a]).

        *local_args -- [only] the wildcard arguments in the command and
        extensions parts of the path, supplied as a number of discrete
        arguments.  For example, if the command and extension matches
        ["standby", "group", None, "ip-secondary", None], this will be
        the value of the two None fields (as the list [a, b]).
        """

        return self.remove(old, context_args, *local_args)


    def remove(self, old, context_args, *local_args):
        """The remove() method is called when the specified path is in
        the remove differences (i.e. it's in the old configuration but
        but not in the new configuration), unless the containing context
        is being removed in its entirety.

        The default behaviour of this method is to call truncate() with
        the old value for 'old' and 'remove' as, if there is no specific
        behaviour for removing an entire object, we should remove the
        individual elements.

        The 'new' value passed to truncate() is None as nothing exists
        in the new configuration - if the truncate() needs this, a
        separate remove() method will normally be needed.

        Keyword arguments:

        old -- the value of the dictionary item at the matching path in
        the old configuration dictionary (sometimes, the full details of
        the old configuration may be required to remove it)

        context_args -- [only] the wildcard arguments from the context
        part of path, supplied as an iterable (typically a list).  For
        example, if the context matches ["interface", None], this will
        be the the value of None (as the list [a]).

        *local_args -- [only] the wildcard arguments in the command and
        extensions parts of the path, supplied as a number of discrete
        arguments.  For example, if the command and extension matches
        ["standby", "group", None, "ip-secondary", None], this will be
        the value of the two None fields (as the list [a, b]).

        The return value is the commands to insert into the
        configuration to convert it.  This can either be a simple
        string, in the case of only one line being required, or an
        iterable containing one string per line.  If the return value is
        None, it is an indication nothing needed to be done.  An empty
        string or iterable indicates the update did something, which did
        not change the configuration (this may have semantic
        differences).
        """

        return self.truncate(old, old, None, context_args, *local_args)


    def truncate(self, old, rem, new, context_args, *local_args):
        """The truncate() method is identical to remove() except that it
        is called when an item is partially removed (i.e. something
        remains in the new configuration - it is 'truncated').

        It is useful when amending lists or sets and matching at the
        containing object level.

        Keyword arguments:

        old -- the value of the dictionary item at the matching path in
        the old configuration dictionary (sometimes, the full details of
        the old configuration may be required to remove it)

        rem -- the value of the dictionary item at the matching path in
        the remove differences dictionary (sometimes, it's necessary to
        know only what is removed from the new configuration)

        new -- the value of the dictionary item at the matching path in
        the new configuration dictionary (sometimes, parts of the old
        configuration cannot be removed and the entire new configuration
        is required to replace it)

        context_args -- [only] the wildcard arguments from the context
        part of path, supplied as an iterable (typically a list).  For
        example, if the context matches ["interface", None], this will
        be the the value of None (as the list [a]).

        *local_args -- [only] the wildcard arguments in the command and
        extensions parts of the path, supplied as a number of discrete
        arguments.  For example, if the command and extension matches
        ["standby", "group", None, "ip-secondary", None], this will be
        the value of the two None fields (as the list [a, b]).

        The return value is the commands to insert into the
        configuration to convert it.  This can either be a simple
        string, in the case of only one line being required, or an
        iterable containing one string per line.  If the return value is
        None, it is an indication nothing needed to be done.  An empty
        string or iterable indicates the update did something, which did
        not change the configuration (this may have semantic
        differences).
        """

        pass


    def add(self, new, context_args, *local_args):
        """The add() method is called when the specified path is in the
        update differences but did not exist in the old configuration
        (i.e. it's something new that is being added to the
        configuration).

        By default, this calls update() with the new configuration as
        the updated and new configuration arguments (and 'args') as, in
        many cases, the process for adding something is the same as
        updating it (e.g. updating an interface description).  It can,
        however, be overridden to do something different or, more
        commonly, implement add() but not update() for a particular
        change (as the updates will be picked up by more specific
        paths in other objects).

        Keyword arguments:

        new -- the value of the dictionary item at the matching path in
        the new configuration dictionary

        context_args -- [only] the wildcard arguments from the context
        part of path, supplied as an iterable (typically a list).  For
        example, if the context matches ["interface", None], this will
        be the the value of None (as the list [a]).

        *local_args -- [only] the wildcard arguments in the command and
        extensions parts of the path, supplied as a number of discrete
        arguments.  For example, if the command and extension matches
        ["standby", "group", None, "ip-secondary", None], this will be
        the value of the two None fields (as the list [a, b]).

        The return value is the commands to insert into the
        configuration to convert it.  This can either be a simple
        string, in the case of only one line being required, or an
        iterable containing one string per line.  If the return value is
        None, it is an indication nothing needed to be done.  An empty
        string or iterable indicates the update did something, which did
        not change the configuration (this may have semantic
        differences).
        """

        return self.update(None, new, new, context_args, *local_args)


    def update(self, old, upd, new, context_args, *local_args):
        """The update() method is called when the specified path is in
        the update differences and also in the old configuration (i.e.
        something is being updated in the configuration).

        Keyword arguments:

        old -- the value of the dictionary item at the matching path in
        the old configuration dictionary (sometimes, the old
        configuration must be removed first, before the new
        configuration can be updated, so the details are required)

        upd -- the value of the dictionary item at the matching path in
        the update differences dictionary

        new -- the value of the dictionary item at the matching path in
        the new configuration dictionary (sometimes, the full details of
        the new configuration may be required to update it)

        context_args -- [only] the wildcard arguments from the context
        part of path, supplied as an iterable (typically a list).  For
        example, if the context matches ["interface", None], this will
        be the the value of None (as the list [a]).

        *local_args -- [only] the wildcard arguments in the command and
        extensions parts of the path, supplied as a number of discrete
        arguments.  For example, if the command and extension matches
        ["standby", "group", None, "ip-secondary", None], this will be
        the value of the two None fields (as the list [a, b]).

        The return value is the commands to insert into the
        configuration to convert it.  This can either be a simple
        string, in the case of only one line being required, or an
        iterable containing one string per line.  If the return value is
        None, it is an indication nothing needed to be done.  An empty
        string or iterable indicates the update did something, which did
        not change the configuration (this may have semantic
        differences).
        """

        pass


    def trigger(self, new, context_args, *local_args):
        """The trigger() method is the same as the update() method
        except it is called when a trigger match occurs.

        By default, this method just calls update(new, None, old,
        context_args, *local_args) but can be overridden by subclasses
        if a special action needs to be taken in this situation (i.e.
        where this is no change but an item of configuration is already
        present).

        Note that as the upd argument passed to update() is None, which
        may need to be carefully handled to avoid checking things like
        dictionary key membership.

        Note that a trigger() method will not be called if a remove or
        update match has already been called on a particular match.
        """

        return self.update(new, None, new, context_args, *local_args)
