# asimtote.diff
#
# Copyright (C) Robert Franklin <rcf34@cam.ac.uk>



"""Configuration differences processing module.

This module contains the abstract base class for determing the
differences between configuration files, processing rules and triggering
converters.
"""



# --- imports ---



from itertools import chain
import sys

from deepops import (
    deepdiff, deepfilter, deepget, deepremoveitems, deepsetdefault)

import yaml

from .misc import deepselect



# --- constants ---



# different debugging levels and what they mean (they're used in lots of
# places so it seems sensible to avoid hard coding the values, in case
# anything changes)

DEBUG_CONVERT_MATCH = 1         # matching class and action type only
DEBUG_CONVERT_STEPS = 2         # include working steps
DEBUG_CONVERT_PARAMS = 3        # include old/new/remove/update arguments
DEBUG_CONVERT_NODIFF = 4        # include skipped converters or no action
DEBUG_CONVERT_NOMATCH = 5       # include non-matching converters

# maximum debug level
DEBUG_CONVERT_MAX = DEBUG_CONVERT_NOMATCH



# --- functions ---



def pathstr(path, wildcard_indices=set()):
    """This function converts a path, which is a list of items of
    various types (typically strings and integers), into a string,
    useful for debugging messages.

    The items are separated by commas and elements which are None are
    converted to a '*' since they're wildcards.

    If the index of the item in the path is in the wildcard_indices
    list, that item is prefixed with "*=" to show it matched a wildcard.
    The default for this parameter is an empty list, which causes no
    index to match.

    If we just printed the list, it would by in the more verbose Python
    format, with surrounding square brackets, quotes around strings and
    None for a wildcard, which is a little less readable.
    """

    return ':'.join([ ("*=" if i in wildcard_indices else "") + str(v)
                            for i, v in enumerate(path) ])



# --- classes ---



class DiffConfig:
    """This abstract class is used to represent a configuration
    difference processor that can convert a configuration from one to
    another, using a method (which can be called once for each pair of
    configuration files).

    It encapsulates the rules for exluding items from the comparison.

    The list of converters is organised into 'blocks', each with a list
    of converters.  The sequence of blocks is set in _init_blocks() and
    each converter has an optional 'block' attribute which specifies
    which block it is to be added to.  The converters are added with
    _add_converters().  Both of these methods will likely need to be
    overridden in concrete classes.
    """


    def __init__(self, init_explain=False, init_dump_config=False,
                 init_dump_diff=False, init_debug_convert=0,
                 init_subtree_dump_filter=[]):

        """The constructor initialises the exclude list to empty and
        adds the converter block sequence with _add_blocks() and
        individual converters using _add_converters().  It also stores
        some settings controlling the level of information describing
        the conversion process, based on the command line arguments:

        init_explain=False -- include comments in the output
        configuration changes that explain the differences being matched
        by the Convert objects (if available).

        init_dump_config=False -- dump the old and new configurations
        (after excludes).

        init_dump_diff=False -- dump the differences (remove and update
        configurations).

        init_debug_convert=0 -- level of debugging information for the
        conversion process (see DEBUG_CONVERT_xxx constants).
        """

        # store the [initial] settings for the conversion process
        self._explain = init_explain
        self._dump_config = init_dump_config
        self._dump_diff = init_dump_diff
        self._debug_convert = init_debug_convert
        self._subtree_dump_filter = init_subtree_dump_filter

        # initialise the dictionary of excludes, which will be passed
        # to deepremoveitems()
        self.init_rules()

        # initialise the converter block sequence
        self._init_blocks()

        # the converters are stored in a dictionary, keyed on the block
        # name, with each value a list of converters in that block, to
        # be applied in the specified order
        self._cvts = {}
        for block in self._blocks:
            self._cvts[block] = []

        # add and sort the converters
        self._add_converters()
        self._sort_converters()


    def _add_blocks(self):
        """This method adds the sequence of converter blocks.  Blocks
        group together converters which must be run before others (in
        other blocks).

        The base class assumes a single block with no name ('None'),
        which is the default under a converter specifies a different
        one.
        """

        self._blocks = [None]


    def _add_converters(self):
        """This method adds the converters for a particular object to
        the list used by the convert() method, usually by calling
        _add_converter() for each (see that method).

        The base class does nothing but child classes will implement it
        as they require.
        """

        pass


    def _add_converter(self, cvt):
        """Add an individual converter object, a child of the Convert
        class, to the list of converters for its block.

        If the block used in the converter does not exist, a KeyError
        will be raised.
        """

        block = cvt.block

        if block not in self._cvts:
            raise KeyError("converter: %s block: %s not found"
                                % (type(cvt).__name__, block))

        self._cvts[block].append(cvt)


    def _sort_converters(self):
        """This method sorts the converters within each block.  The
        default ordering is the the complete path (context + cmd + ext)
        but can be overridden, if required.

        This method is called by the constructor, after the converters
        have been added to ensure a consist order.
        """

        for block in self._cvts:
            self._cvts[block].sort(key=lambda c: c._get_sort_key())


    def _explain_comment(self, path):
        """This method returns a comment or other configuration item
        explaining the path supplied (which will typically be a match
        against a converter).  The path is supplied a list of levels and
        converted to a string using pathstr().

        Child classes can override this to provide a comment appropriate
        for their platform.

        If the function returns None, no comment is inserted.
        """

        return None


    def _diffs_begin(self):
        """This method returns a head (beginning) for a generated
        changes configuration file as an iterable of strings.

        In the abstract class, it returns None (which does not add
        anything) but, in child classes it may return a beginning line
        or similar.

        Note that if there are no changes, this will not be included in
        an otherwise empty file.
        """

        return []


    def _diffs_end(self):
        """This method returns a tail (ending) for a generated changes
        configuration file as an iterable of strings.

        In the abstract class, it returns None (which does not add
        anything) but, in child classes it may return an 'end' line or
        similar.

        Note that if there are no changes, this will not be included in
        an otherwise empty file.
        """

        return []


    def init_rules(self):
        """This method initialises the rules s.  In the base
        class, it the resets it to an empty dictionary, but child
        classes can extend this to add some default exclusions for
        standard system configuration entries which should not be
        changed.

        It is normally only called by the constructor but can be called
        later, to clear the excludes list before adding more (if
        required).
        """

        self.init_rules_tree()
        self.init_rules_active()


    def init_rules_tree(self):
        """This method initialises the rules tree (the dictionary of
        rules typically read from a file.

        In the base class, it just sets it to an empty dictionary but
        some platform-specific classes may wish to extend this to set up
        a default tree (along with init_rules_active()).
        """

        self._rules_tree = {}


    def init_rules_active(self):
        """This method initialises the active rules list (the list of
        rules specifying what should be used from the rules tree).

        In the base class, it just sets it to an empty list but some
        platform-specific classes may wish to extend this to set up a
        default list.
        """

        self._rules_active = []


    def add_rules_tree_file(self, filename):
        """Read a tree of rules items from a YAML file.  This is
        typically done once but then different portions of the rules
        dictionary selected with set_rules_active().

        The contents of the file are added to the current tree.  To
        clear the current tree first, use init_rules_tree().
        """

        try:
            file_rules_tree = yaml.safe_load(open(filename))

        except yaml.parser.ParserError as exception:
            raise ValueError("failed parsing rules file: %s: %s"
                                 % (filename, exception))

        self._rules_tree.update(file_rules_tree)


    def add_rules_active(self, rule_specs, devicename):
        """Add a list of rules to the current rule list, which specifies
        what parts of the rules tree should be used and how (include or
        exclude these items), to control the comparison done by the
        convert() method.

        The rules are specified as a list of strings in the format
        '[!]<path>' where '!' means 'exclude' (if omitted, it means
        'include') and 'path' is a colon-separated list of keys giving
        the path into the rules tree.  A particular element can be
        given as '%', in which case the 'devicename' parameter will be
        used but, if the devicename argument is None/empty (False),
        then a warning is printed and the rule skipped.

        For example, if '!device-excludes:%' is given, and the
        devicename is 'router1', the part of the rules tree indexed by
        ["device-excludes"]["router1"] will be excluded from the
        comparison.

        The rules given will be added to the current rules list; if the
        list is to be cleared first, use init_rules_list().
        """

        for rule_spec in rule_specs:
            # find if this is an include or exclude rule and get the
            # path part

            include = not rule_spec.startswith("!")

            if include:
                rule_spec_path = rule_spec
            else:
                rule_spec_path = rule_spec[1:]


            path_elements = rule_spec_path.split(":")

            if ("%" in path_elements) and (not devicename):
                print("warning: rule specification: %s contains '%%' but no "
                      "device name - ignoring" % rule_spec_path,
                      file=sys.stderr)

                continue


            path = [ devicename if i == '%' else i for i in path_elements ]

            self._rules_active.append( (include, path) )


    def get_rules_tree(self):
        """The method returns the current rules tree (as initialised by
        init_rules_tree() and extended with read_rules_tree()).  This
        should not be modified.
        """

        return self._rules_tree


    def get_rules(self):
        """This method just returns the active rules list and the
        portion of the rules tree it applies to.

        The return value is a list, one entry per rule, containing
        2-tuples: the first entry is the rule specification (a string,
        in the format described by add_rules_active()), and the second
        the portion of the rules tree that the path references (or None,
        if that part of the tree does not exist).

        This method is mainly used for debugging messages.
        """

        r = []
        for include, path in self._rules_active:
            r.append(
                (("" if include else '!') + ':'.join(path),
                 deepget(self._rules_tree, *path) ) )

        return r


    def apply_rules(self, d):
        """This method applies the current rules (tree and active list)
        to the supplied configuration dictionary, either only including
        what's specified (using deepfilter()) or excluding (using
        deepremoveitems()) the specified portions.

        The configuration dictionary is modified in place.

        The method can be used on a configuration dictionary, or the
        remove/update dictionaries returned by deepdiff().
        """

        for include, path in self._rules_active:
            # get the porttion of the rules tree specified by the path
            # in this rule
            path_dict = deepget(self._rules_tree, *path)

            # skip to the next entry, if this part was not found
            if path_dict is None:
                continue

            # do either the include or exclude on the configuration
            # dictionary, in place)
            if include:
                d = deepfilter(d, path_dict)
            else:
                deepremoveitems(d, path_dict)


    def update_dump_config(self, new_dump_config):
        "Updates the dump configuration setting."
        self._dump_config = new_dump_config


    def update_dump_diff(self, new_dump_diff):
        "Updates the dump differences setting."
        self._dump_diff = new_dump_diff


    def update_debug_convert(self, new_debug_convert):
        "Updates the debug conversion level."
        self._debug_convert = new_debug_convert


    def _print_debug(self, msg, blank_if_single=True):
        """Print the supplied debugging message followed by a blank
        line, if it's not empty (in which case, do nothing), unless
        blank_if_single is True and the message is only one line long.

        Prior to printing the message, _print_debug_block() and
        _print_debug_converter() will be called, to print those
        messages, if they haven't already been printed.
        """

        if msg:
            self._print_debug_block()
            self._print_debug_converter()

            for line in msg:
                print(line, file=sys.stderr)

            if blank_if_single or (len(msg) > 1):
                print(file=sys.stderr)


    def _print_debug_block(self):
            """Print the block debug message explicitly, if the debug
            level is above DEBUG_CONVERT_MATCH (the bottom level).

            Normally this is called by _print_debug() when the first
            debug message is displayed but, if more detailed debugging
            is requireed, it can be called explicitly.

            After printing, the message is cleared to not be printed
            again.
            """

            if self._debug_block_msg:
                for line in self._debug_block_msg + [""]:
                    print(line, file=sys.stderr)

                self._debug_block_msg = []


    def _print_debug_converter(self):
            """Print the converter debug message explicitly, if the debug
            level is above DEBUG_CONVERT_MATCH (the bottom level).

            Normally this is called by _print_debug() when the first
            debug message is displayed but, if more detailed debugging
            is requireed, it can be called explicitly.

            After printing, the message is cleared to not be printed
            again.
            """

            if self._debug_converter_msg:
                for line in self._debug_converter_msg + [""]:
                    print(line, file=sys.stderr)

                self._debug_converter_msg = []


    def convert(self, old_cfg, new_cfg):
        """This method processes the conversion from the old
        configuration to the new configuration, removing excluded parts
        of each and calling the applicable converters' action methods.

        Note that, if excludes are used, the configurations will be
        modified in place by a deepremoveitems().  They will need to be
        copy.deepcopy()ed before passing them in, if this is
        undesirable.

        The returned value is a 2-tuple:

        - the first element is a big string of all the configuration
          changes that need to be made, sandwiched between
          _diffs_begin() and _diffs_end(), or None, if there were no
          differences

        - the second element is a dictionary giving the tree of
          differences (i.e. the elements where a difference was
          encountered - either a remove or an update)
        """


        self.apply_rules(old_cfg)

        if self._dump_config:
            print(">>"
                  + (" old" if new_cfg else "")
                  + " configuration (after rules, if specified):",
                  yaml.dump(deepselect(dict(old_cfg),
                                       *self._subtree_dump_filter),
                            default_flow_style=False),
                  sep="\n")


        # initialise the list of diffs (the returned configuration
        # conversions) and the tree of differences to empty

        diffs = []
        diffs_tree = {}


        # initialise the dictionary of activated triggers and their
        # configuration points

        active_triggers = {}


        # if no new config was specified, stop here (we assume we're
        # just testing, parsing and excluding items from the old
        # configuration and stopping
        #
        # we check for None explicitly for the difference between no
        # configuration and an empty configuration

        if new_cfg is None:
            return None, diffs_tree


        self.apply_rules(new_cfg)

        if self._dump_config:
            print(">> new configuration (after rules, if specified):",
                  yaml.dump(deepselect(dict(new_cfg),
                                       *self._subtree_dump_filter),
                            default_flow_style=False),
                  sep="\n")


        # use deepdiff() to work out the differences between the two
        # configuration dictionaries - what must be removed and what
        # needs to be added or updated
        #
        # then use deepfilter() to get the full details of each item
        # being removed, rather than just the top of a subtree being
        # removed

        remove_cfg, update_cfg = deepdiff(old_cfg, new_cfg)
        remove_cfg_full = deepfilter(old_cfg, remove_cfg)


        if self._dump_diff:
            print("=> differences - remove:",
                  yaml.dump(deepselect(dict(remove_cfg),
                                       *self._subtree_dump_filter),
                            default_flow_style=False),

                  "=> differences - remove full:",
                  yaml.dump(deepselect(dict(remove_cfg_full),
                                       *self._subtree_dump_filter),
                            default_flow_style=False),

                  "=> differences - update (add/change):",
                  yaml.dump(deepselect(dict(update_cfg),
                                       *self._subtree_dump_filter),
                            default_flow_style=False),

                  sep="\n")


        # set the current block name (used to print when a new block is
        # entered) - we use an empty string as something that won't be
        # used as a block name (we can't use 'None' as that's the
        # default block)

        current_block = ""

        self._debug_block_msg = []


        # go through the list of blocks and then the converter objects
        # within them, in order

        for cvt in chain.from_iterable(
            [ self._cvts[block] for block in self._blocks ]):

            # if we're entering a new block, set a 'new block' debug
            # message and print it immediately, if we are debugging to
            # that level

            if cvt.block != current_block:
                current_block = cvt.block

                if self._debug_convert > DEBUG_CONVERT_MATCH:
                    self._debug_block_msg = [">> [block: %s]" % current_block]

                    if self._debug_convert >= DEBUG_CONVERT_NOMATCH:
                        self._print_debug_block()


            # get all the remove and update matches for this converter
            # and combine them into one list, discarding any duplicates
            #
            # we do this rather than processing each list one after the
            # other so we combine removes and updates on the same part
            # of the configuration together

            remove_matches = cvt.full_matches(remove_cfg_full)
            update_matches = cvt.full_matches(update_cfg)


            # if this converter is in a block which has been triggered,
            # find all the matches in the new configuration which match
            # the context in which the trigger was set

            triggered_matches = []
            if cvt.block in active_triggers:
                for t in active_triggers[cvt.block]:
                    triggered_matches.extend(
                        cvt.explicit_context_matches(new_cfg, t))


            # combine all the matches for this converter into a single
            # list
            #
            # we sort this list so changes are processed in a consistent
            # and predictable order - there can, however, be elements
            # where the corresponding element in another path is of a
            # different type (e.g. one is a string and the other None),
            # so they can't be sorted directly
            #
            # to resolve this, the sorting key for a match is trans-
            # formed by converting all the path elements to strings,
            # with the value None being transformed to an empty string
            # ('')

            sorted_matches = (
                sorted(remove_matches + update_matches + triggered_matches,
                       key=lambda k: [ ('' if i is None else str(i))
                                          for i in k ]))

            all_matches = []
            for match in sorted_matches:
                if match not in all_matches:
                    all_matches.append(match)


            # print the name of the converter if either: 1. we have the
            # debugging messages always enabled for this, or 2. we are
            # only debugging converters with matches and there are some
            #
            # the matching converters might be skipped or not generate
            # any differences, but we can't avoid doing that unless we
            # store this message to be printed later, if so - it doesn't
            # currently seem worth doing that for debugging messages

            self._debug_converter_msg = []

            if self._debug_convert > DEBUG_CONVERT_MATCH:
                self._debug_converter_msg = [
                    ">> " + pathstr([ '*' if i is None else i
                                        for i in cvt.get_path() ]),
                    "-> " + type(cvt).__name__]

                if self._debug_convert >= DEBUG_CONVERT_NOMATCH:
                    self._print_debug_converter()


            for match in all_matches:
                # handle REMOVE conversions, if matching

                if match in remove_matches:
                    debug_msg = []


                    # check if anything remains at this level or below
                    # in the new configuration - if it does, we're doing
                    # a partial removal

                    remove_is_truncate = False

                    try:
                        # we don't want the result here, just to find
                        # out if the path exists (if not, KeyError will
                        # be raised)
                        deepget(new_cfg, *match, default_error=True)

                    except KeyError:
                        # nothing remains so we're doing a full remove
                        pass

                    else:
                        # something remains so this is partial
                        if self._debug_convert >= DEBUG_CONVERT_STEPS:
                            debug_msg.append(
                                "-> subconfiguration not empty - truncate")
                        remove_is_truncate = True


                    if self._debug_convert >= DEBUG_CONVERT_MATCH:
                        debug_msg.append(
                            "=> "
                            + ("truncate" if remove_is_truncate else "remove")
                            + ": "
                            + pathstr(match, cvt.wildcard_indices))


                    # get elements in the path matching wildcards

                    context_args, local_args = cvt.get_args(match)


                    # skip this conversion if it is filtered out

                    try:
                        filtered = cvt.filter_delete(context_args, *local_args)

                    except:
                        print("Exception in %s.filter_delete() with:"
                                  % type(cvt),
                              "  context_args=" + repr(context_args),
                              "  *local_args=" + repr(local_args),
                              "",
                              sep="\n", file=sys.stderr)

                        raise

                    if not filtered:
                        if self._debug_convert >= DEBUG_CONVERT_NODIFF:
                            debug_msg.append("-> filtered - skip")
                            self._print_debug(debug_msg)
                        continue


                    # if we're removing the entire context for this
                    # match, we don't need to do it, as that will
                    # remove this, while it's at it

                    if cvt.context_removed(remove_cfg, match):
                        if self._debug_convert >= DEBUG_CONVERT_NODIFF:
                            debug_msg.append(
                                "-> containing context being removed - skip")

                            self._print_debug(
                                debug_msg,
                                blank_if_single=
                                    self._debug_convert > DEBUG_CONVERT_MATCH)

                        continue


                    # get the old, remove and new parts of the
                    # configuration and remove difference dictionaries,
                    # for the path specified in the converter (ignoring
                    # the extension 'ext')

                    cvt_old = cvt.get_cfg(old_cfg, match)
                    cvt_rem = cvt.get_cfg(remove_cfg_full, match)
                    cvt_new = cvt.get_cfg(new_cfg, match)

                    if self._debug_convert >= DEBUG_CONVERT_PARAMS:
                        debug_msg.extend([
                            "-> old configuration:",
                            yaml.dump(cvt_old, default_flow_style=False)])

                        if remove_is_truncate:
                            debug_msg.extend([
                                "-> remove configuration:",
                                yaml.dump(cvt_rem, default_flow_style=False)])

                        debug_msg.extend([
                            "-> new configuration:",
                            yaml.dump(cvt_new, default_flow_style=False)])


                    # call the truncate or delete converter action method

                    try:
                        if remove_is_truncate:
                            diff = cvt.truncate(cvt_old, cvt_rem, cvt_new,
                                                context_args, *local_args)
                        else:
                            diff = cvt.delete(cvt_old, cvt_rem, cvt_new,
                                              context_args, *local_args)

                    except:
                        # builld a list of arguments for passing to the
                        # converter action method for the exception
                        # message
                        action_args_str = ["  old=" + repr(cvt_old)]
                        if remove_is_truncate:
                            action_args_str.append("  rem=" + repr(cvt_rem))
                        action_args_str.append("  new=" + repr(cvt_new))

                        print("Exception in %s.%s() with:"
                                  % (type(cvt).__name__,
                                     "truncate" if remove_is_truncate
                                         else "delete"),
                              *action_args_str,
                              "  context_args=" + repr(context_args),
                              "  *local_args=" + repr(local_args),
                              "",
                              sep="\n", file=sys.stderr)

                        raise


                    # if some diffs were returned by the action, add
                    # them

                    if diff is not None:
                        # the return can be either a simple string or a
                        # list of strings - if it's a string, make it a
                        # list of one so we can do the rest the same way

                        if isinstance(diff, str):
                            diff = [diff]

                        if self._debug_convert >= DEBUG_CONVERT_STEPS:
                            debug_msg += diff


                        # add a comment, explaining the match, if
                        # enabled

                        if self._explain:
                            comment = self._explain_comment(match)
                            if comment:
                                diffs.append(comment)


                        # store this diff on the end of the list of
                        # diffs so far

                        diffs += diff
                        diffs.append("")


                        # add this match to the differences tree

                        deepsetdefault(diffs_tree, *match)

                    else:
                        if self._debug_convert >= DEBUG_CONVERT_NODIFF:
                            debug_msg.append("-> no action")
                        else:
                            # no differences were returned by the
                            # conversion method and we're not debugging
                            # 'no diff's so blank any debugging message
                            # we've built up so we don't print anything
                            # for this match

                            debug_msg = []


                    # if this converter sets off some triggers, add
                    # those to the active trigger dictionary, along with
                    # the matching patch
                    #
                    # we only set triggers if the converter actually
                    # generated some conversion commands, or if the
                    # empty_trigger option is set, and they're not
                    # filtered out with trigger_set_filter_delete()

                    if (diff is not None) or cvt.empty_trigger:
                        for t in cvt.trigger_blocks:
                            match_context = cvt.get_match_context(match)

                            if t not in self._blocks:
                                raise KeyError(
                                    "converter: %s triggers non-existent"
                                    " block: %s" % (type(cvt).__name__, t))

                            if not cvt.trigger_set_filter_delete(
                                       t, cvt_old, cvt_rem, cvt_new,
                                       context_args, *local_args):

                                if self._debug_convert >= DEBUG_CONVERT_STEPS:
                                    debug_msg.append(
                                        "-> trigger filtered out: %s @ %s"
                                            % (t,
                                               pathstr(match_context, set())))

                                # skip this trigger without setting it up
                                continue

                            if self._debug_convert >= DEBUG_CONVERT_STEPS:
                                debug_msg.append(
                                    "-> trigger set: %s @ %s"
                                        % (t, pathstr(match_context, set())))

                            active_triggers.setdefault(t, []).append(
                                match_context)


                    self._print_debug(
                        debug_msg,
                        blank_if_single=
                            self._debug_convert > DEBUG_CONVERT_MATCH)


                # handle UPDATE conversions, if matching

                if match in update_matches:
                    debug_msg = []


                    # check if there is anything for this level in the
                    # old configuration - if not, we're actually adding
                    # this, rather than updating it, so record that

                    update_is_add = False

                    try:
                        deepget(old_cfg, *match, default_error=True)

                    except KeyError:
                        if self._debug_convert >= DEBUG_CONVERT_STEPS:
                            debug_msg.append("-> no old configuration - add")
                        update_is_add = True


                    if self._debug_convert >= DEBUG_CONVERT_MATCH:
                        debug_msg.append(
                            "=> "
                            + ("add" if update_is_add else "update")
                            + ": "
                            + pathstr(match, cvt.wildcard_indices))


                    # (same as in remove, above)

                    context_args, local_args = cvt.get_args(match)

                    try:
                        filtered = cvt.filter_update(context_args, *local_args)

                    except:
                        print("Exception in %s.filter_update() with:"
                                  % type(cvt),
                              "  context_args=" + repr(context_args),
                              "  *local_args=" + repr(local_args),
                              "",
                              sep="\n", file=sys.stderr)

                        raise

                    if not filtered:
                        if self._debug_convert >= DEBUG_CONVERT_NODIFF:
                            debug_msg.append("-> filtered - skip")
                            self._print_debug(debug_msg)
                        continue


                    # get the old, update and new parts of the
                    # configuration difference dictionaries, for the
                    # path specified in the converter (ignoring the
                    # extension 'ext')

                    cvt_old = cvt.get_cfg(old_cfg, match)
                    cvt_upd = cvt.get_cfg(update_cfg, match)
                    cvt_new = cvt.get_cfg(new_cfg, match)

                    if self._debug_convert >= DEBUG_CONVERT_PARAMS:
                        if not update_is_add:
                            debug_msg.extend([
                                "-> old configuration:",
                                yaml.dump(cvt_old, default_flow_style=False),
                                "-> update configuration:",
                                yaml.dump(cvt_upd, default_flow_style=False)])

                        debug_msg.extend([
                            "-> new configuration:",
                            yaml.dump(cvt_new, default_flow_style=False)])


                    # call the update or add converter action method

                    try:
                        # if we're adding this, call the add() method,
                        # else update()
                        if update_is_add:
                            diff = cvt.add(cvt_new, context_args, *local_args)
                        else:
                            diff = cvt.update(cvt_old, cvt_upd, cvt_new,
                                              context_args, *local_args)

                    except:
                        # builld a list of arguments for passing to the
                        # converter action method for the exception
                        # message
                        action_args_str = []
                        if not update_is_add:
                            action_args_str.extend([
                                "  old=" + repr(cvt_old),
                                "  upd=" + repr(cvt_upd)])
                        action_args_str.append("  new=" + repr(cvt_new))

                        print("Exception in %s.%s() with:"
                                  % (type(cvt).__name__,
                                     "add" if update_is_add else "update"),
                              *action_args_str,
                              "  context_args=" + repr(context_args),
                              "  *local_args=" + repr(local_args),
                              "",
                              sep="\n", file=sys.stderr)

                        raise


                    # (same as in remove, above)

                    if diff is not None:
                        if isinstance(diff, str):
                            diff = [diff]

                        if self._debug_convert >= DEBUG_CONVERT_STEPS:
                            debug_msg += diff


                        if self._explain:
                            comment = self._explain_comment(match)
                            if comment:
                                diffs.append(comment)


                        diffs.extend(diff)
                        diffs.append("")

                        deepsetdefault(diffs_tree, *match)

                    else:
                        if self._debug_convert >= DEBUG_CONVERT_NODIFF:
                            debug_msg.append("-> no action")
                        else:
                            debug_msg = []


                    if (diff is not None) or cvt.empty_trigger:
                        for t in cvt.trigger_blocks:
                            match_context = cvt.get_match_context(match)

                            if t not in self._blocks:
                                raise KeyError(
                                    "converter: %s triggers non-existent"
                                    " block: %s" % (type(cvt).__name__, t))

                            if not cvt.trigger_set_filter_update(
                                       t, cvt_old, cvt_upd, cvt_new,
                                       context_args, *local_args):

                                if self._debug_convert >= DEBUG_CONVERT_STEPS:
                                    debug_msg.append(
                                        "-> trigger filtered out: %s @ %s"
                                            % (t,
                                               pathstr(match_context, set())))

                                # skip this trigger without setting it up
                                continue

                            if self._debug_convert >= DEBUG_CONVERT_STEPS:
                                debug_msg.append(
                                    "-> trigger set: %s @ %s"
                                        % (t, pathstr(match_context, set())))

                            active_triggers.setdefault(t, []).append(
                                match_context)


                    self._print_debug(
                        debug_msg,
                        blank_if_single=
                            self._debug_convert > DEBUG_CONVERT_MATCH)


                # handle TRIGGERED conversions, if matching, and this
                # wasn't already a remove or update conversion

                if ((match in triggered_matches)
                    and (match not in remove_matches)
                    and (match not in update_matches)):

                    debug_msg = []

                    if self._debug_convert >= DEBUG_CONVERT_MATCH:
                        debug_msg.append(
                            "=> trigger: %s @ %s"
                                % (cvt.block,
                                   pathstr(match, cvt.wildcard_indices)))


                    # (same as in remove, above)

                    context_args, local_args = cvt.get_args(match)

                    try:
                        filtered = cvt.filter_trigger(context_args, *local_args)

                    except:
                        print("Exception in %s.filter_trigger() with:"
                                  % type(cvt),
                              "  context_args=" + repr(context_args),
                              "  *local_args=" + repr(local_args),
                              "",
                              sep="\n", file=sys.stderr)

                        raise

                    if not filtered:
                        if self._debug_convert >= DEBUG_CONVERT_NODIFF:
                            debug_msg.append("-> filtered - skip")
                            self._print_debug(debug_msg)
                        continue


                    # get the old, update and new parts of the
                    # configuration difference dictionaries for the path
                    # specified in the converter (ignoring the extension
                    # 'ext')

                    cvt_new = cvt.get_cfg(new_cfg, match)

                    if self._debug_convert >= DEBUG_CONVERT_PARAMS:
                        debug_msg.extend([
                            "-> new configuration:",
                            yaml.dump(cvt_new, default_flow_style=False)])


                    # call the trigger converter action method

                    try:
                        diff = cvt.trigger(cvt_new, context_args, *local_args)

                    except:
                        print("%s.trigger() with:" % type(cvt).__name__,
                              "  new=" + repr(cvt_new),
                              "  context_args=" + repr(context_args),
                              "  *local_args=" + repr(local_args),
                              sep="\n", file=sys.stderr)

                        raise


                    # (same as in remove, above)

                    if diff is not None:
                        if isinstance(diff, str):
                            diff = [diff]

                        if self._debug_convert >= DEBUG_CONVERT_STEPS:
                            debug_msg += diff


                        if self._explain:
                            comment = self._explain_comment(match)
                            if comment:
                                diffs.append(comment)


                        diffs.extend(diff)
                        diffs.append("")

                        deepsetdefault(diffs_tree, *match)

                        # except one trigger cannot fire another

                    else:
                        if self._debug_convert >= DEBUG_CONVERT_STEPS:
                            debug_msg.append("-> no action")
                        else:
                            debug_msg = []


                    self._print_debug(
                        debug_msg,
                        blank_if_single=
                            self._debug_convert > DEBUG_CONVERT_MATCH)


        # if nothing was generated, just return nothing

        if not diffs:
            return None, diffs_tree


        # return the diffs concatenated into a big, multiline string,
        # along with the begin and end blocks

        return ("\n".join(self._diffs_begin() + diffs + self._diffs_end()),
                diffs_tree)
