# Expiration Date Feature für FlowEngine

## Übersicht

Das Expiration Date Feature ermöglicht es, beim Erstellen von neuen Trades über die FlowEngine ein spezifisches Verfallsdatum zu verwenden, anstatt sich auf die DTE-Berechnung des Templates zu verlassen.

## Motivation

**Anwendungsfall**: Wenn ein 0DTE Iron Condor vorzeitig geschlossen wird (Early Exit), soll ein 1DTE Iron Condor mit **demselben Verfallsdatum** geöffnet werden, nicht mit dem Verfallsdatum, das sich aus "1 DTE ab heute" ergibt.

**Beispiel**:
- 0DTE Trade wird am 22.10.2025 um 14:00 Uhr eröffnet (Verfallsdatum: 22.10.2025)
- Trade wird um 15:30 Uhr vorzeitig geschlossen (Early Exit)
- Neuer 1DTE Trade soll eröffnet werden **mit Verfallsdatum 22.10.2025** (nicht 23.10.2025)

## Implementierung

### 1. Event-Daten erweitert

**Datei**: `optrabot/flowengine/flowevent.py`

Alle Event-Datenklassen wurden um das Feld `trade_expiration: date = None` erweitert:
- `TradeOpenedEventData`
- `EarlyExitEventData`
- `StopLossHitEventData`
- `TakeProfitHitEventData`

Die Variable `$EVENT_TRADE_EXPIRATION` steht nun in allen Flow-Ausdrücken zur Verfügung.

### 2. Action-Konfiguration erweitert

**Datei**: `optrabot/flowengine/flowconfig.py`

Die `ProcessTemplateAction` wurde um den optionalen Parameter `expiration: Any = None` erweitert.

### 3. Config-Parsing erweitert

**Datei**: `optrabot/config.py`

Die `_parse_flow()` Methode parst nun auch den optionalen `expiration` Parameter aus der YAML-Konfiguration.

### 4. FlowEngine-Logik erweitert

**Datei**: `optrabot/flowengine/flowengine.py`

Die `_execute_process_template()` Methode:
1. Evaluiert den `expiration` Ausdruck (falls vorhanden)
2. Validiert, dass das Ergebnis ein `datetime.date` Objekt ist
3. Setzt das Expiration-Datum explizit auf dem Template über `set_expiration_date()`

### 5. Template Basisklasse erweitert

**Datei**: `optrabot/tradetemplate/templatefactory.py`

Die `Template` Klasse wurde erweitert um:
1. **Attribut `expiration_date`**: Speichert das explizite Verfallsdatum (hat Vorrang vor `dte`)
2. **Methode `set_expiration_date(expiration_date)`**: Setzt das explizite Verfallsdatum

### 6. TemplateProcessorBase erweitert

**Datei**: `optrabot/tradetemplate/processor/templateprocessorbase.py`

Die `composeEntryOrder()` Methode:
1. Prüft zuerst, ob `template.expiration_date` gesetzt ist
2. Verwendet `expiration_date` wenn vorhanden (höchste Priorität)
3. Verwendet die DTE-Berechnung nur, wenn kein `expiration_date` gesetzt wurde

**Vorteile dieser Implementierung**:
- Keine Fehleranfälligkeit durch DTE-Umrechnungen
- Das exakte Verfallsdatum wird direkt verwendet
- Keine Rundungsfehler oder Probleme mit Handelstagen

### 7. TradeManager erweitert

**Datei**: `optrabot/trademanager.py`

#### Event-Emission mit Expiration:
- **Trade Opened**: Extrahiert `expiration` aus `managedTrade.entryOrder.legs[0].expiration`
- **Early Exit**: Extrahiert `expiration` aus dem Entry-Order beim Schließen
- **Stop Loss Hit**: Extrahiert `expiration` aus dem Entry-Order
- **Take Profit Hit**: Extrahiert `expiration` aus dem Entry-Order

#### Helper-Methoden hinzugefügt:
- `_calculate_trade_premium()`: Berechnet die Entry-Prämie inkl. Gebühren
- `_calculate_trade_net_result()`: Berechnet das Netto-Ergebnis des Trades
- `_emit_trade_exit_event()`: Emittiert Exit-Events mit allen Daten

## Verwendung

### Konfiguration in config.yaml

```yaml
flows:
  iic_rollover2:
    name: "Roll of 0DTE IC to 1DTE IC with same expiration"
    enabled: true
    event:
      type: early_exit
      template: 0DTEIC100Income
    actions:
      - send_notification:
          message: "Rollover mit gleicher Expiration ausgelöst."
      - process_template:
          template: 1DTEIC100Income
          amount: $EVENT_TRADE_AMOUNT * 2
          premium: ($EVENT_TRADE_NET_RESULT + $EVENT_TRADE_PREMIUM) / ($EVENT_TRADE_AMOUNT * 2)
          expiration: $EVENT_TRADE_EXPIRATION  # Verwendet das Verfallsdatum des geschlossenen Trades
```

## Verfügbare Variablen

In allen Event-Typen steht nun die Variable zur Verfügung:

- `$EVENT_TRADE_EXPIRATION`: `datetime.date` - Verfallsdatum der Optionen des auslösenden Trades

## Priorisierung

Beim Erstellen eines neuen Trades wird das Verfallsdatum in folgender Reihenfolge bestimmt:

1. **Expiration-Parameter** (falls in Flow-Action angegeben) ← **Höchste Priorität**
2. **Template DTE** (Berechnung basierend auf DTE-Wert des Templates)
3. **Standard** (0 DTE, falls keine Angabe vorhanden)

## Logging

Die FlowEngine loggt die Verwendung des Expiration-Parameters:

```
INFO: Executing process_template action: template=1DTEIC100Income, amount=2, premium=0.65, expiration=2025-10-22
DEBUG: Using provided expiration date: 2025-10-22
```

## Getestete Szenarien

✅ Trade Opened Event mit Expiration
✅ Early Exit Event mit Expiration
✅ Stop Loss Hit Event mit Expiration
✅ Take Profit Hit Event mit Expiration
✅ Process Template mit Expiration-Parameter
✅ Expiration-Priorisierung über DTE
✅ Fehlende Expiration (Fallback auf DTE)

## Kompatibilität

- **Abwärtskompatibel**: Flows ohne `expiration`-Parameter funktionieren weiterhin wie bisher
- **Optional**: Der `expiration`-Parameter ist in allen Actions optional
- **Validierung**: Wenn ein `expiration`-Parameter angegeben wird, muss dieser zu einem `datetime.date` Objekt evaluiert werden

## Dependencies

Keine neuen Dependencies erforderlich. Das Feature nutzt die bereits vorhandene `simpleeval`-Bibliothek für die Expression-Evaluierung.
