# FlowEngine Implementation - Zusammenfassung

## Implementierte Features

### 1. Kernkomponenten

#### FlowEngine (`optrabot/flowengine/flowengine.py`)
- Singleton-Pattern für zentrale Event-Verwaltung
- Event-basierte Architektur mit `eventkit`
- Asynchrone Flow-Ausführung über `APScheduler`
- Sichere Expression-Evaluation mit `simpleeval`
- Sequentielle Action-Ausführung mit Error-Handling

#### Flow-Konfiguration (`optrabot/flowengine/flowconfig.py`)
- `Flow`: Basis-Konfiguration eines Flows
- `FlowEventConfig`: Event-Trigger-Konfiguration
- `SendNotificationAction`: Notification-Action-Konfiguration
- `ProcessTemplateAction`: Template-Processing-Action-Konfiguration

#### Event-Daten (`optrabot/flowengine/flowevent.py`)
- `FlowEventType`: Enum für Event-Typen
- `TradeOpenedEventData`: Daten für trade_opened Events
- `EarlyExitEventData`: Daten für early_exit Events
- `StopLossHitEventData`: Daten für stop_loss_hit Events
- `TakeProfitHitEventData`: Daten für take_profit_hit Events

### 2. Event-Typen

#### trade_opened
- Ausgelöst: Wenn Entry Order gefüllt wurde und Trade auf OPEN gesetzt wird
- Variablen: ID, Amount, Entry Price, Symbol, Strategy
- Location: `TradeManager._onOrderStatusChanged()` nach Status-Update auf OPEN

#### early_exit
- Ausgelöst: Wenn Closing Order gefüllt wurde (z.B. bei Breakeven)
- Variablen: ID, Amount, Entry Price, Exit Price, Net Result, Premium, Symbol, Strategy
- Location: `TradeManager._onOrderStatusChanged()` für closing_order mit Status FILLED

#### stop_loss_hit
- Ausgelöst: Wenn Stop-Loss Order gefüllt wurde
- Variablen: Wie early_exit
- Location: `TradeManager._onOrderStatusChanged()` für stopLossOrder mit Status FILLED

#### take_profit_hit
- Ausgelöst: Wenn Take-Profit Order gefüllt wurde
- Variablen: Wie early_exit
- Location: `TradeManager._onOrderStatusChanged()` für takeProfitOrder mit Status FILLED

### 3. Actions

#### send_notification
- Sendet Notification über TradingHub
- Unterstützt Variable-Interpolation in Message
- Konfigurierbarer Notification-Typ (INFO, WARN, ERROR)

#### process_template
- Verarbeitet ein Trade-Template
- Unterstützt statische Werte und Berechnungsformeln
- Expression-Evaluation für amount und premium

### 4. Integration

#### config.py
- Neue Methode `readFlows()` zum Laden der Flow-Konfigurationen
- Neue Methode `_parse_flow()` zum Parsen einzelner Flows
- Neue Methode `getFlows()` zum Abrufen der konfigurierten Flows
- Unterstützt beide YAML-Formate (List und Dict)

#### optrabot.py
- FlowEngine-Initialisierung im `startup()`
- FlowEngine-Shutdown im `shutdown()`
- Übergabe des Background-Schedulers an FlowEngine

#### trademanager.py
- Neue Methode `_calculate_trade_premium()` zur Prämienberechnung
- Neue Methode `_calculate_trade_net_result()` zur Ergebnisberechnung
- Neue Methode `_emit_trade_exit_event()` zur Event-Emission
- Event-Emission an 4 Stellen in `_onOrderStatusChanged()`

### 5. Konfiguration

#### config.yaml Format
```yaml
flows:
  flow_id:
    name: "Flow Name"
    enabled: true
    event:
      type: early_exit
      template: TemplateName
    actions:
      - send_notification:
          message: "Message text"
          type: INFO
      - process_template:
          template: TargetTemplate
          amount: $EVENT_TRADE_AMOUNT * 2
          premium: ($EVENT_TRADE_NET_RESULT + $EVENT_TRADE_PREMIUM) / 4
```

### 6. Abhängigkeiten

Neue Abhängigkeit hinzugefügt:
- `simpleeval = "^0.9.13"` in pyproject.toml

### 7. Tests

Unit-Tests erstellt in `tests/flowengine_test.py`:
- Singleton-Pattern Test
- Expression-Evaluation Tests (einfach, komplex, statisch, Fehlerfall)
- Send-Notification Action Test
- Event-Data Variables Tests

## Verwendete Design-Patterns

1. **Singleton**: FlowEngine ist ein Singleton für zentrale Event-Verwaltung
2. **Observer**: Event-basierte Architektur mit eventkit
3. **Strategy**: Verschiedene Action-Typen mit einheitlichem Interface
4. **Factory**: Flow-Erstellung aus Konfigurationsdaten

## Sicherheitsaspekte

- **Expression Evaluation**: Verwendung von `simpleeval` statt `eval()` verhindert Code-Injection
- **Error Handling**: Umfassendes Try-Catch in allen kritischen Bereichen
- **Validierung**: Template-Namen und Konfigurationswerte werden validiert
- **Logging**: Alle wichtigen Operationen werden geloggt

## Erweiterbarkeit

Die Architektur ist designed für einfache Erweiterungen:

1. **Neue Event-Typen**: Neue Event-Klasse in `flowevent.py` und Enum-Eintrag
2. **Neue Actions**: Neue Action-Klasse und Handler-Methode in FlowEngine
3. **Neue Variablen**: Einfach zur `get_variables()` Methode der Event-Klasse hinzufügen

## Bekannte Limitationen

1. Expressions können nur einfache mathematische Operationen (+, -, *, /, Klammern)
2. Flows werden sequenziell ausgeführt (Actions innerhalb eines Flows)
3. Keine Retry-Logik bei fehlgeschlagenen Actions
4. Keine Zeitplanung für Flows (nur Event-basiert)

## Nächste Schritte

Mögliche zukünftige Erweiterungen:
1. Conditional Actions (if-then-else Logik)
2. Retry-Mechanismus für fehlgeschlagene Actions
3. Flow-Scheduling (zeitbasierte Ausführung)
4. Mehr komplexe Expressions (z.B. min/max Funktionen)
5. Flow-Logging in Datenbank für Audit-Trail
6. Flow-Statistiken und Monitoring
