# OTB-253: Trade Recovery

**Feature-Branch:** `feature/OTB-253`  
**Version:** 0.18.0a10  
**Datum:** 27. Oktober 2025  
**Status:** Phase 1 & Phase 2 implementiert

## Übersicht

Dieses Feature implementiert Trade Recovery beim OptraBot-Start. Beim Neustart kann es vorkommen, dass Trades in der Datenbank noch als OPEN markiert sind, obwohl:
- Ihre Optionen bereits abgelaufen sind (Phase 1)
- OptraBot über Nacht/mehrere Tage gestoppt war (Phase 2)
- Das reguläre EOD Settlement verpasst wurde (Phase 1)

**Phase 1** (implementiert) behandelt das Settlement von abgelaufenen Trades.
**Phase 2** (implementiert) behandelt die Wiederaufnahme aktiver Trades in TradeManager.

**Zukünftige Phasen** werden TP/SL Order Recovery und Position-Reconciliation implementieren.

## Problem

### Szenario

Ein Trader:
1. Öffnet einen Trade mit Expiration in 30 Tagen
2. Stoppt OptraBot über Nacht
3. Startet OptraBot am nächsten Tag
4. **Problem:** Der Trade ist noch OPEN in der DB, aber OptraBot "weiß nichts" davon

Noch kritischer:
1. Trade wurde vor 10 Tagen eröffnet (Expiration in 30 Tagen)
2. OptraBot läuft 5 Tage, wird dann gestoppt
3. Bleibt 40 Tage gestoppt
4. **Problem beim Start:** Trade ist längst expired, aber noch als OPEN in DB

### Konsequenzen

- Falsche P&L-Berichte (expired Positionen nicht abgerechnet)
- Datenbank-Inkonsistenzen
- Unmögliche Analyse der Trading-Historie
- Compliance-Probleme (fehlende Settlement-Records)

## Lösung - Phase 1: Expired Trade Settlement

### Konzept

Beim OptraBot-Start, **vor** der TradeManager-Initialisierung:

1. **Datenbank durchsuchen** nach allen OPEN Trades
2. **Expiration-Datum prüfen** anhand der Transaktionen
3. **Abgelaufene Trades identifizieren** (Expiration < heute)
4. **Settlement durchführen** mit gleicher Logik wie EOD Settlement

### Warum vor TradeManager?

- TradeManager verwaltet nur **aktive** ManagedTrades
- Expired Trades müssen vorher aus DB "aufgeräumt" werden
- Keine Broker-Verbindung nötig (Settlement basiert auf DB-Daten)

## Implementation

### Neue Komponente: `TradeRecoveryService`

**Datei:** `optrabot/traderecovery.py`

Hauptmethode: `settle_expired_trades()`
- Lädt alle OPEN Trades aus DB
- Prüft Expiration-Datum
- Settled abgelaufene Trades mit EOD Settlement Logik

### Integration in Startup

```python
# optrabot/optrabot.py

await BrokerFactory().createBrokerConnectors()

# OTB-253: Trade Recovery
from optrabot.traderecovery import TradeRecoveryService
recovery = TradeRecoveryService()
await recovery.recover_all()  # Phase 1 + Phase 2

TradeManager()
```

## Phase 2: Active Trade Recovery

### Konzept

Beim OptraBot-Start, **vor** der TradeManager-Initialisierung:

1. **Datenbank durchsuchen** nach allen OPEN (nicht-expired) Trades
2. **Template laden** anhand des gespeicherten `template_name`
3. **Entry Order rekonstruieren** aus Transaction-Historie
4. **ManagedTrade erstellen** mit vollständigem State
5. **In TradeManager registrieren** für Monitoring

### Voraussetzung: template_name in Datenbank

Ab OTB-253 Phase 2 wird bei jedem neuen Trade der `template_name` gespeichert:

```python
# Beim Trade-Erstellen (TradeManager)
newTradeSchema = schemas.TradeCreate(
    account=template.account,
    symbol=entryOrder.symbol,
    strategy=template.strategy,
    template_name=template.name  # NEU: Template-Name speichern
)
```

**Migration:** Datenbankschema wurde erweitert:
- Neue Spalte `template_name` (nullable) in `trades` Tabelle
- Alte Trades haben `template_name = NULL` (können nicht recovered werden)
- Neue Trades haben den Template-Namen gespeichert

### Implementation Details

#### 1. Template Lookup

Neue Methode in `config.py`:

```python
def get_template_by_name(self, template_name: str) -> Template | None:
    """Find a template by its name."""
    for template in self._templates:
        if template.name == template_name:
            return template
    return None
```

#### 2. Entry Order Rekonstruktion

```python
def _reconstruct_entry_order(self, trade: Trade) -> Order:
    """
    Rekonstruiert die Entry Order aus Transaktionen.
    
    - Nur opening transactions (SELL/BUY mit positiven Kontrakten)
    - Berechnet Entry Price aus Transaction-Preisen
    - Bestimmt PriceEffect (CREDIT/DEBIT)
    """
```

#### 3. ManagedTrade Erstellung

```python
def _create_managed_trade(self, trade, template, entry_order) -> ManagedTrade:
    """
    Erstellt ManagedTrade-Objekt wie beim normalen Trade-Opening.
    
    - Setzt entry_price
    - Kopiert current_legs vom Entry Order
    - Setup StopLoss Adjusters
    - Setup Delta Adjusters
    """
```

#### 4. TradeManager Registration

```python
def _register_with_trade_manager(self, managed_trade):
    """
    Fügt den recovered Trade zu TradeManager._trades[] hinzu.
    Ab jetzt wird der Trade normal überwacht (TP, SL, Delta).
    """
```

### Fehlerbehandlung

**Fall 1: Template nicht gefunden**
```
ERROR: Cannot recover trade 123: Template 'IC_SPX_0DTE' not found in config
```
→ Recovery für diesen Trade überspringen, mit nächstem fortfahren

**Fall 2: Kein template_name (alte Trades)**
```
WARNING: Trade 456 has no template_name - created before OTB-253, cannot recover
```
→ Trade kann nicht recovered werden, wird übersprungen

**Fall 3: Keine Transaktionen**
```
ERROR: Trade 789 has no transactions - cannot recover
```
→ Recovery fehlgeschlagen, Trade übersprungen

### Log-Ausgabe

```
======================================================================
Starting Trade Recovery
======================================================================
Phase 1: Settling expired trades
Found 2 expired trade(s) requiring settlement
✅ Trade 101 settled successfully (Realized P&L: $245.50)
✅ Trade 102 settled successfully (Realized P&L: $-120.00)

Phase 2: Recovering active trades
Found 3 active trade(s) requiring recovery
Recovering trade 201 (Account: DU123456, Symbol: SPX, Template: IC_SPX_0DTE)
Reconstructed entry order for trade 201: 4 legs, entry price: $1.50
✅ Trade 201 recovered successfully (Template: IC_SPX_0DTE, Entry: $1.50)
✅ Trade 202 recovered successfully (Template: IC_SPX_1DTE, Entry: $2.10)
✅ Trade 203 recovered successfully (Template: IC_SPX_2DTE, Entry: $2.75)

======================================================================
Trade Recovery Summary:
  Expired trades settled: 2
  Active trades recovered: 3
  Failed recoveries: 0
======================================================================
```

## Verwendung

Trade Recovery läuft automatisch bei jedem OptraBot-Start und loggt Ergebnisse im INFO-Level.

Nach der Recovery:
- Expired Trades sind auf Status EXPIRED gesetzt
- Active Trades sind in TradeManager._trades[] registriert
- Monitoring läuft normal (Take Profit, Stop Loss, Delta Adjustments)

## Zukünftige Phasen

- **Phase 2:** Active Trade Recovery (template-basiert)
- **Phase 3:** Order Recovery (TP/SL Orders)
- **Phase 4:** Position Reconciliation mit Broker

## Abhängigkeiten

- OTB-254: PNL Correction Logic
- OTB-259: SQLAlchemy Session Fix
