# OTB-258: NTP Time Synchronization

**Feature-Branch:** `feature/OTB-258`  
**Version:** 0.18.0a8  
**Datum:** 25. Oktober 2025

## Übersicht

Dieses Feature implementiert eine automatische Zeitüberwachung, die die Systemzeit des OptraBot mit NTP-Servern vergleicht. Bei Abweichungen über 10 Sekunden wird eine Warnung geloggt und eine Telegram-Nachricht versendet.

## Problem

Trading-Systeme sind auf präzise Zeitstempel angewiesen:
- Ordereingabe muss zeitgenau erfolgen
- Marktdaten-Timestamps müssen korrekt sein
- Trading-Fenster (z.B. "erste 30 Minuten nach Börsenöffnung") müssen exakt sein
- Audit-Logs und Compliance erfordern korrekte Zeitstempel

Wenn die Systemzeit des Computers von der tatsächlichen Zeit abweicht, kann dies zu:
- Verpassten Trades (falsche Timing-Berechnung)
- Falschen Marktdaten-Interpretationen
- Compliance-Problemen
- Fehlerhaften Log-Analysen

führen.

## Lösung

### Architektur

Das Feature besteht aus einer neuen Komponente `TimeSync`, die:

1. **NTP-Server abfragt** um die korrekte Zeit zu ermitteln
2. **Zeitdifferenz berechnet** zwischen Systemzeit und NTP-Zeit
3. **Bei Abweichung > 30 Sekunden:**
   - Warnung im Log ausgibt
   - Telegram-Nachricht an den Nutzer sendet

### Integration mit APScheduler

Die Zeitprüfung wird über den vorhandenen `AsyncIOScheduler` gesteuert:

- **Beim Start:** 5 Sekunden nach OptraBot-Start
- **Täglich:** Um 6:00 Uhr Eastern Time

### NTP-Server

Das System verwendet mehrere NTP-Server als Fallback:

1. `time.nist.gov` - US Government time server (Primär)
2. `pool.ntp.org` - Global NTP pool (Backup 1)
3. `time.google.com` - Google's NTP server (Backup 2)
4. `time.apple.com` - Apple's NTP server (Backup 3)

Wenn der erste Server nicht antwortet, wird automatisch der nächste versucht.

## Implementation

### Neue Dateien

#### `optrabot/timesync.py`

```python
class TimeSync:
    """Manages time synchronization monitoring with NTP servers."""
    
    # NTP servers in order of preference
    NTP_SERVERS = [
        'pool.ntp.org',
        'time.google.com',
        'time.apple.com'
    ]
    
    # Maximum allowed time difference in seconds
    MAX_DRIFT_SECONDS = 30
    
    async def check_and_sync_time(self):
        """Main time synchronization check - called by APScheduler"""
```

Hauptmethoden:
- `get_ntp_time()` - Fragt NTP-Server ab
- `get_system_time()` - Holt Systemzeit
- `calculate_time_drift()` - Berechnet Differenz
- `send_telegram_warning()` - Sendet Warnung bei Abweichung
- `check_and_sync_time()` - Hauptmethode (von Scheduler aufgerufen)

### Geänderte Dateien

#### `optrabot/optrabot.py`

**Import:**
```python
from .timesync import TimeSync
```

**Initialisierung in `__init__`:**
```python
self._timeSync : TimeSync = None
```

**Scheduler-Integration in `startup()`:**
```python
# OTB-258: Initialize time synchronization
self._timeSync = TimeSync(telegram_client=self.thc)

# Run immediately at startup
self._backgroundScheduler.add_job(
    self._timeSync.check_and_sync_time,
    'date',
    run_date=dt.datetime.now() + dt.timedelta(seconds=5),
    id='time_sync_startup',
    misfire_grace_time=None
)

# Run daily at 6:00 AM Eastern Time
self._backgroundScheduler.add_job(
    self._timeSync.check_and_sync_time,
    'cron',
    hour=6,
    minute=0,
    second=0,
    timezone=pytz.timezone('US/Eastern'),
    id='time_sync_daily',
    misfire_grace_time=None
)
```

#### `pyproject.toml`

**Version erhöht:**
```toml
version = "0.18.0a8"
```

**Neue Dependency:**
```toml
ntplib = "^0.4.0"
```

## Verwendung

### Automatischer Betrieb

Nach dem Start des OptraBot wird die Zeitprüfung automatisch ausgeführt:

1. **Startup-Check** (5 Sekunden nach Start):
   ```
   ======================================================================
   Starting time synchronization check (OTB-258)
   ======================================================================
   Time drift: -0.23 seconds
   ✅ Time drift within acceptable range: -0.23s (max: 30s)
   ======================================================================
   Time synchronization check completed
   ======================================================================
   ```

2. **Täglicher Check** (6:00 AM Eastern Time):
   - Gleiche Prüfung wie beim Start
   - Bei Abweichung > 30s: Telegram-Warnung

### Telegram-Warnung

Bei Zeitabweichung > 30 Sekunden wird folgende Nachricht gesendet:

```
⚠️ TIME SYNC WARNING ⚠️

System time is off by 45.3 seconds

System time: 2025-10-25 13:45:23 UTC
NTP time: 2025-10-25 13:44:38 UTC

Please check and adjust system time manually.
```

### Log-Ausgaben

**Normale Zeitabweichung (<30s):**
```
INFO  | Querying NTP server: time.nist.gov
INFO  | Time drift: 2.34 seconds
INFO  | ✅ Time drift within acceptable range: 2.34s (max: 30s)
```

**Kritische Zeitabweichung (>30s):**
```
WARNING | ⚠️ Time drift exceeds threshold: 45.3s (max: 30s)
INFO    | Telegram warning sent
```

**NTP-Server nicht erreichbar:**
```
WARNING | Failed to get time from time.nist.gov: [Errno 8] nodename nor servname provided
WARNING | Failed to get time from pool.ntp.org: timed out
ERROR   | All NTP servers failed
ERROR   | Cannot check time drift - NTP servers unavailable
```

## Konfiguration

### Anpassbare Parameter in `TimeSync`

```python
# NTP servers in order of preference
NTP_SERVERS = [
    'time.nist.gov',
    'pool.ntp.org',
    'time.google.com',
    'time.apple.com'
]

# Maximum allowed time difference in seconds
MAX_DRIFT_SECONDS = 30
```

### Anpassung der Prüfzeiten

In `optrabot.py` können die Scheduler-Jobs angepasst werden:

```python
# Startup-Check: Verzögerung ändern (aktuell 5 Sekunden)
run_date=dt.datetime.now() + dt.timedelta(seconds=5)

# Täglicher Check: Zeit ändern (aktuell 6:00 AM Eastern)
hour=6, minute=0, second=0
```

## Testing

### Manueller Test

```python
from optrabot.timesync import TimeSync

# Ohne Telegram-Client (nur Logging)
ts = TimeSync()
await ts.check_and_sync_time()
```

### Test mit abweichender Zeit

Um das Feature zu testen, kann die Systemzeit manuell verstellt werden:

**macOS/Linux:**
```bash
# Zeit 1 Minute zurückstellen
sudo date -s "-1 minute"

# OptraBot starten - sollte Warnung ausgeben

# Zeit wieder korrigieren
sudo sntp -sS time.nist.gov
```

**Windows:**
```powershell
# Zeit manuell in Systemeinstellungen verstellen
# OptraBot starten - sollte Warnung ausgeben
```

## Einschränkungen

1. **Keine automatische Korrektur:**
   - Der OptraBot korrigiert die Systemzeit NICHT automatisch
   - Dies würde Admin-Rechte erfordern
   - Nur Warnung + Telegram-Nachricht

2. **Netzwerkabhängigkeit:**
   - Benötigt Internetverbindung zu NTP-Servern
   - Bei Netzwerkproblemen: Fallback auf mehrere Server
   - Bei totalem Ausfall: Fehler wird geloggt, aber OptraBot läuft weiter

3. **Einmalige Prüfung:**
   - Nur beim Start und 1x täglich
   - Nicht kontinuierlich
   - Für kontinuierliche Überwachung OS-Level NTP-Sync empfohlen

## Best Practices

### Betriebssystem-Level NTP

Es wird empfohlen, zusätzlich OS-Level NTP-Synchronisation zu aktivieren:

**macOS:**
```bash
sudo systemsetup -setusingnetworktime on
sudo systemsetup -setnetworktimeserver time.apple.com
```

**Windows:**
```powershell
w32tm /config /manualpeerlist:"time.windows.com" /syncfromflags:manual /update
net stop w32time && net start w32time
w32tm /resync
```

**Linux:**
```bash
sudo timedatectl set-ntp true
sudo systemctl enable systemd-timesyncd
sudo systemctl start systemd-timesyncd
```

### Monitoring

- Telegram-Nachrichten ernst nehmen
- Bei wiederholten Warnungen: Hardware-Uhr (CMOS) prüfen
- Log-Datei auf NTP-Fehler überwachen

## Zukünftige Erweiterungen

Mögliche Verbesserungen für zukünftige Versionen:

1. **Konfigurierbare Schwellwerte** in `config.yaml`
2. **Mehrfache Messungen** mit Durchschnittsbildung
3. **Trendanalyse** - Clock-Drift-Rate berechnen
4. **Erweiterte Statistiken** - Min/Max/Durchschnitt über Zeit
5. **Grafische Darstellung** im Frontend

## Zusammenfassung

OTB-258 implementiert eine robuste Zeitüberwachung für den OptraBot:

✅ **Einfache Integration** - Nutzt vorhandenen APScheduler  
✅ **Mehrere NTP-Server** - Hohe Verfügbarkeit  
✅ **Automatische Prüfung** - Beim Start und täglich  
✅ **Telegram-Benachrichtigung** - Bei kritischer Abweichung  
✅ **Keine Admin-Rechte** - Nur Monitoring, keine Änderungen  
✅ **Robuste Fehlerbehandlung** - Läuft auch bei NTP-Ausfall weiter

Das Feature erhöht die Zuverlässigkeit des Trading-Systems durch frühzeitige Erkennung von Zeitproblemen.
