"""
Test script for OTB-256: TWS Version Check

This script tests the TWS version check logic without connecting to actual TWS.
"""
from datetime import date

def test_tws_version_check():
    """
    Test the version check logic
    """
    ENFORCEMENT_DATE = date(2025, 11, 3)
    MIN_REQUIRED_VERSION = 176  # TWS 10.37
    
    # Test scenarios
    test_cases = [
        {
            'name': 'Before enforcement, old version',
            'today': date(2025, 10, 24),
            'server_version': 175,
            'expected': 'WARNING'
        },
        {
            'name': 'Before enforcement, new version',
            'today': date(2025, 10, 24),
            'server_version': 176,
            'expected': 'OK'
        },
        {
            'name': 'After enforcement, old version',
            'today': date(2025, 11, 3),
            'server_version': 175,
            'expected': 'ERROR'
        },
        {
            'name': 'After enforcement, new version',
            'today': date(2025, 11, 3),
            'server_version': 176,
            'expected': 'OK'
        },
        {
            'name': 'Way after enforcement, old version',
            'today': date(2025, 12, 1),
            'server_version': 170,
            'expected': 'ERROR'
        },
        {
            'name': 'Way after enforcement, newer version',
            'today': date(2025, 12, 1),
            'server_version': 180,
            'expected': 'OK'
        }
    ]
    
    print("=" * 80)
    print("TWS VERSION CHECK TEST (OTB-256)")
    print("=" * 80)
    print(f"Enforcement Date: {ENFORCEMENT_DATE}")
    print(f"Minimum Required Version: {MIN_REQUIRED_VERSION} (TWS 10.37)")
    print("=" * 80)
    
    for test_case in test_cases:
        print(f"\n📋 Test: {test_case['name']}")
        print(f"   Today: {test_case['today']}")
        print(f"   TWS Version: {test_case['server_version']}")
        
        today = test_case['today']
        server_version = test_case['server_version']
        
        if server_version < MIN_REQUIRED_VERSION:
            if today >= ENFORCEMENT_DATE:
                result = 'ERROR'
                days_until = 0
            else:
                result = 'WARNING'
                days_until = (ENFORCEMENT_DATE - today).days
        else:
            result = 'OK'
            days_until = 0
        
        status_emoji = '✅' if result == 'OK' else ('⚠️' if result == 'WARNING' else '❌')
        print(f"   Result: {status_emoji} {result}")
        
        if result == 'WARNING':
            print(f"   Days until enforcement: {days_until}")
        
        # Verify expected result
        if result == test_case['expected']:
            print(f"   ✓ Test passed")
        else:
            print(f"   ✗ Test FAILED - Expected: {test_case['expected']}, Got: {result}")
    
    print("\n" + "=" * 80)
    print("Test completed!")
    print("=" * 80)

if __name__ == '__main__':
    test_tws_version_check()
