/** vim: et:ts=4:sw=4:sts=4
 * @license RequireJS 2.0.5 Copyright (c) 2010-2012, The Dojo Foundation All Rights Reserved.
 * Available via the MIT or new BSD license.
 * see: http://github.com/jrburke/requirejs for details
 */
//Not using strict: uneven strict support in browsers, #392, and causes
//problems with requirejs.exec()/transpiler plugins that may not be strict.
/*jslint regexp: true, nomen: true, sloppy: true */
/*global window, navigator, document, importScripts, jQuery, setTimeout, opera */

var requirejs, require, define;
(function (global) {
    var req, s, head, baseElement, dataMain, src,
        interactiveScript, currentlyAddingScript, mainScript, subPath,
        version = '2.0.5',
        commentRegExp = /(\/\*([\s\S]*?)\*\/|([^:]|^)\/\/(.*)$)/mg,
        cjsRequireRegExp = /[^.]\s*require\s*\(\s*["']([^'"\s]+)["']\s*\)/g,
        jsSuffixRegExp = /\.js$/,
        currDirRegExp = /^\.\//,
        op = Object.prototype,
        ostring = op.toString,
        hasOwn = op.hasOwnProperty,
        ap = Array.prototype,
        aps = ap.slice,
        apsp = ap.splice,
        isBrowser = !!(typeof window !== 'undefined' && navigator && document),
        isWebWorker = !isBrowser && typeof importScripts !== 'undefined',
        //PS3 indicates loaded and complete, but need to wait for complete
        //specifically. Sequence is 'loading', 'loaded', execution,
        // then 'complete'. The UA check is unfortunate, but not sure how
        //to feature test w/o causing perf issues.
        readyRegExp = isBrowser && navigator.platform === 'PLAYSTATION 3' ?
                      /^complete$/ : /^(complete|loaded)$/,
        defContextName = '_',
        //Oh the tragedy, detecting opera. See the usage of isOpera for reason.
        isOpera = typeof opera !== 'undefined' && opera.toString() === '[object Opera]',
        contexts = {},
        cfg = {},
        globalDefQueue = [],
        useInteractive = false;

    function isFunction(it) {
        return ostring.call(it) === '[object Function]';
    }

    function isArray(it) {
        return ostring.call(it) === '[object Array]';
    }

    /**
     * Helper function for iterating over an array. If the func returns
     * a true value, it will break out of the loop.
     */
    function each(ary, func) {
        if (ary) {
            var i;
            for (i = 0; i < ary.length; i += 1) {
                if (ary[i] && func(ary[i], i, ary)) {
                    break;
                }
            }
        }
    }

    /**
     * Helper function for iterating over an array backwards. If the func
     * returns a true value, it will break out of the loop.
     */
    function eachReverse(ary, func) {
        if (ary) {
            var i;
            for (i = ary.length - 1; i > -1; i -= 1) {
                if (ary[i] && func(ary[i], i, ary)) {
                    break;
                }
            }
        }
    }

    function hasProp(obj, prop) {
        return hasOwn.call(obj, prop);
    }

    /**
     * Cycles over properties in an object and calls a function for each
     * property value. If the function returns a truthy value, then the
     * iteration is stopped.
     */
    function eachProp(obj, func) {
        var prop;
        for (prop in obj) {
            if (obj.hasOwnProperty(prop)) {
                if (func(obj[prop], prop)) {
                    break;
                }
            }
        }
    }

    /**
     * Simple function to mix in properties from source into target,
     * but only if target does not already have a property of the same name.
     * This is not robust in IE for transferring methods that match
     * Object.prototype names, but the uses of mixin here seem unlikely to
     * trigger a problem related to that.
     */
    function mixin(target, source, force, deepStringMixin) {
        if (source) {
            eachProp(source, function (value, prop) {
                if (force || !hasProp(target, prop)) {
                    if (deepStringMixin && typeof value !== 'string') {
                        if (!target[prop]) {
                            target[prop] = {};
                        }
                        mixin(target[prop], value, force, deepStringMixin);
                    } else {
                        target[prop] = value;
                    }
                }
            });
        }
        return target;
    }

    //Similar to Function.prototype.bind, but the 'this' object is specified
    //first, since it is easier to read/figure out what 'this' will be.
    function bind(obj, fn) {
        return function () {
            return fn.apply(obj, arguments);
        };
    }

    function scripts() {
        return document.getElementsByTagName('script');
    }

    //Allow getting a global that expressed in
    //dot notation, like 'a.b.c'.
    function getGlobal(value) {
        if (!value) {
            return value;
        }
        var g = global;
        each(value.split('.'), function (part) {
            g = g[part];
        });
        return g;
    }

    function makeContextModuleFunc(func, relMap, enableBuildCallback) {
        return function () {
            //A version of a require function that passes a moduleName
            //value for items that may need to
            //look up paths relative to the moduleName
            var args = aps.call(arguments, 0), lastArg;
            if (enableBuildCallback &&
                    isFunction((lastArg = args[args.length - 1]))) {
                lastArg.__requireJsBuild = true;
            }
            args.push(relMap);
            return func.apply(null, args);
        };
    }

    function addRequireMethods(req, context, relMap) {
        each([
            ['toUrl'],
            ['undef'],
            ['defined', 'requireDefined'],
            ['specified', 'requireSpecified']
        ], function (item) {
            var prop = item[1] || item[0];
            req[item[0]] = context ? makeContextModuleFunc(context[prop], relMap) :
                    //If no context, then use default context. Reference from
                    //contexts instead of early binding to default context, so
                    //that during builds, the latest instance of the default
                    //context with its config gets used.
                    function () {
                        var ctx = contexts[defContextName];
                        return ctx[prop].apply(ctx, arguments);
                    };
        });
    }

    /**
     * Constructs an error with a pointer to an URL with more information.
     * @param {String} id the error ID that maps to an ID on a web page.
     * @param {String} message human readable error.
     * @param {Error} [err] the original error, if there is one.
     *
     * @returns {Error}
     */
    function makeError(id, msg, err, requireModules) {
        var e = new Error(msg + '\nhttp://requirejs.org/docs/errors.html#' + id);
        e.requireType = id;
        e.requireModules = requireModules;
        if (err) {
            e.originalError = err;
        }
        return e;
    }

    if (typeof define !== 'undefined') {
        //If a define is already in play via another AMD loader,
        //do not overwrite.
        return;
    }

    if (typeof requirejs !== 'undefined') {
        if (isFunction(requirejs)) {
            //Do not overwrite and existing requirejs instance.
            return;
        }
        cfg = requirejs;
        requirejs = undefined;
    }

    //Allow for a require config object
    if (typeof require !== 'undefined' && !isFunction(require)) {
        //assume it is a config object.
        cfg = require;
        require = undefined;
    }

    function newContext(contextName) {
        var inCheckLoaded, Module, context, handlers,
            checkLoadedTimeoutId,
            config = {
                waitSeconds: 7,
                baseUrl: './',
                paths: {},
                pkgs: {},
                shim: {}
            },
            registry = {},
            undefEvents = {},
            defQueue = [],
            defined = {},
            urlFetched = {},
            requireCounter = 1,
            unnormalizedCounter = 1,
            //Used to track the order in which modules
            //should be executed, by the order they
            //load. Important for consistent cycle resolution
            //behavior.
            waitAry = [];

        /**
         * Trims the . and .. from an array of path segments.
         * It will keep a leading path segment if a .. will become
         * the first path segment, to help with module name lookups,
         * which act like paths, but can be remapped. But the end result,
         * all paths that use this function should look normalized.
         * NOTE: this method MODIFIES the input array.
         * @param {Array} ary the array of path segments.
         */
        function trimDots(ary) {
            var i, part;
            for (i = 0; ary[i]; i += 1) {
                part = ary[i];
                if (part === '.') {
                    ary.splice(i, 1);
                    i -= 1;
                } else if (part === '..') {
                    if (i === 1 && (ary[2] === '..' || ary[0] === '..')) {
                        //End of the line. Keep at least one non-dot
                        //path segment at the front so it can be mapped
                        //correctly to disk. Otherwise, there is likely
                        //no path mapping for a path starting with '..'.
                        //This can still fail, but catches the most reasonable
                        //uses of ..
                        break;
                    } else if (i > 0) {
                        ary.splice(i - 1, 2);
                        i -= 2;
                    }
                }
            }
        }

        /**
         * Given a relative module name, like ./something, normalize it to
         * a real name that can be mapped to a path.
         * @param {String} name the relative name
         * @param {String} baseName a real name that the name arg is relative
         * to.
         * @param {Boolean} applyMap apply the map config to the value. Should
         * only be done if this normalization is for a dependency ID.
         * @returns {String} normalized name
         */
        function normalize(name, baseName, applyMap) {
            var pkgName, pkgConfig, mapValue, nameParts, i, j, nameSegment,
                foundMap, foundI, foundStarMap, starI,
                baseParts = baseName && baseName.split('/'),
                normalizedBaseParts = baseParts,
                map = config.map,
                starMap = map && map['*'];

            //Adjust any relative paths.
            if (name && name.charAt(0) === '.') {
                //If have a base name, try to normalize against it,
                //otherwise, assume it is a top-level require that will
                //be relative to baseUrl in the end.
                if (baseName) {
                    if (config.pkgs[baseName]) {
                        //If the baseName is a package name, then just treat it as one
                        //name to concat the name with.
                        normalizedBaseParts = baseParts = [baseName];
                    } else {
                        //Convert baseName to array, and lop off the last part,
                        //so that . matches that 'directory' and not name of the baseName's
                        //module. For instance, baseName of 'one/two/three', maps to
                        //'one/two/three.js', but we want the directory, 'one/two' for
                        //this normalization.
                        normalizedBaseParts = baseParts.slice(0, baseParts.length - 1);
                    }

                    name = normalizedBaseParts.concat(name.split('/'));
                    trimDots(name);

                    //Some use of packages may use a . path to reference the
                    //'main' module name, so normalize for that.
                    pkgConfig = config.pkgs[(pkgName = name[0])];
                    name = name.join('/');
                    if (pkgConfig && name === pkgName + '/' + pkgConfig.main) {
                        name = pkgName;
                    }
                } else if (name.indexOf('./') === 0) {
                    // No baseName, so this is ID is resolved relative
                    // to baseUrl, pull off the leading dot.
                    name = name.substring(2);
                }
            }

            //Apply map config if available.
            if (applyMap && (baseParts || starMap) && map) {
                nameParts = name.split('/');

                for (i = nameParts.length; i > 0; i -= 1) {
                    nameSegment = nameParts.slice(0, i).join('/');

                    if (baseParts) {
                        //Find the longest baseName segment match in the config.
                        //So, do joins on the biggest to smallest lengths of baseParts.
                        for (j = baseParts.length; j > 0; j -= 1) {
                            mapValue = map[baseParts.slice(0, j).join('/')];

                            //baseName segment has config, find if it has one for
                            //this name.
                            if (mapValue) {
                                mapValue = mapValue[nameSegment];
                                if (mapValue) {
                                    //Match, update name to the new value.
                                    foundMap = mapValue;
                                    foundI = i;
                                    break;
                                }
                            }
                        }
                    }

                    if (foundMap) {
                        break;
                    }

                    //Check for a star map match, but just hold on to it,
                    //if there is a shorter segment match later in a matching
                    //config, then favor over this star map.
                    if (!foundStarMap && starMap && starMap[nameSegment]) {
                        foundStarMap = starMap[nameSegment];
                        starI = i;
                    }
                }

                if (!foundMap && foundStarMap) {
                    foundMap = foundStarMap;
                    foundI = starI;
                }

                if (foundMap) {
                    nameParts.splice(0, foundI, foundMap);
                    name = nameParts.join('/');
                }
            }

            return name;
        }

        function removeScript(name) {
            if (isBrowser) {
                each(scripts(), function (scriptNode) {
                    if (scriptNode.getAttribute('data-requiremodule') === name &&
                            scriptNode.getAttribute('data-requirecontext') === context.contextName) {
                        scriptNode.parentNode.removeChild(scriptNode);
                        return true;
                    }
                });
            }
        }

        function hasPathFallback(id) {
            var pathConfig = config.paths[id];
            if (pathConfig && isArray(pathConfig) && pathConfig.length > 1) {
                removeScript(id);
                //Pop off the first array value, since it failed, and
                //retry
                pathConfig.shift();
                context.undef(id);
                context.require([id]);
                return true;
            }
        }

        /**
         * Creates a module mapping that includes plugin prefix, module
         * name, and path. If parentModuleMap is provided it will
         * also normalize the name via require.normalize()
         *
         * @param {String} name the module name
         * @param {String} [parentModuleMap] parent module map
         * for the module name, used to resolve relative names.
         * @param {Boolean} isNormalized: is the ID already normalized.
         * This is true if this call is done for a define() module ID.
         * @param {Boolean} applyMap: apply the map config to the ID.
         * Should only be true if this map is for a dependency.
         *
         * @returns {Object}
         */
        function makeModuleMap(name, parentModuleMap, isNormalized, applyMap) {
            var url, pluginModule, suffix,
                index = name ? name.indexOf('!') : -1,
                prefix = null,
                parentName = parentModuleMap ? parentModuleMap.name : null,
                originalName = name,
                isDefine = true,
                normalizedName = '';

            //If no name, then it means it is a require call, generate an
            //internal name.
            if (!name) {
                isDefine = false;
                name = '_@r' + (requireCounter += 1);
            }

            if (index !== -1) {
                prefix = name.substring(0, index);
                name = name.substring(index + 1, name.length);
            }

            if (prefix) {
                prefix = normalize(prefix, parentName, applyMap);
                pluginModule = defined[prefix];
            }

            //Account for relative paths if there is a base name.
            if (name) {
                if (prefix) {
                    if (pluginModule && pluginModule.normalize) {
                        //Plugin is loaded, use its normalize method.
                        normalizedName = pluginModule.normalize(name, function (name) {
                            return normalize(name, parentName, applyMap);
                        });
                    } else {
                        normalizedName = normalize(name, parentName, applyMap);
                    }
                } else {
                    //A regular module.
                    normalizedName = normalize(name, parentName, applyMap);
                    url = context.nameToUrl(normalizedName);
                }
            }

            //If the id is a plugin id that cannot be determined if it needs
            //normalization, stamp it with a unique ID so two matching relative
            //ids that may conflict can be separate.
            suffix = prefix && !pluginModule && !isNormalized ?
                     '_unnormalized' + (unnormalizedCounter += 1) :
                     '';

            return {
                prefix: prefix,
                name: normalizedName,
                parentMap: parentModuleMap,
                unnormalized: !!suffix,
                url: url,
                originalName: originalName,
                isDefine: isDefine,
                id: (prefix ?
                        prefix + '!' + normalizedName :
                        normalizedName) + suffix
            };
        }

        function getModule(depMap) {
            var id = depMap.id,
                mod = registry[id];

            if (!mod) {
                mod = registry[id] = new context.Module(depMap);
            }

            return mod;
        }

        function on(depMap, name, fn) {
            var id = depMap.id,
                mod = registry[id];

            if (hasProp(defined, id) &&
                    (!mod || mod.defineEmitComplete)) {
                if (name === 'defined') {
                    fn(defined[id]);
                }
            } else {
                getModule(depMap).on(name, fn);
            }
        }

        function onError(err, errback) {
            var ids = err.requireModules,
                notified = false;

            if (errback) {
                errback(err);
            } else {
                each(ids, function (id) {
                    var mod = registry[id];
                    if (mod) {
                        //Set error on module, so it skips timeout checks.
                        mod.error = err;
                        if (mod.events.error) {
                            notified = true;
                            mod.emit('error', err);
                        }
                    }
                });

                if (!notified) {
                    req.onError(err);
                }
            }
        }

        /**
         * Internal method to transfer globalQueue items to this context's
         * defQueue.
         */
        function takeGlobalQueue() {
            //Push all the globalDefQueue items into the context's defQueue
            if (globalDefQueue.length) {
                //Array splice in the values since the context code has a
                //local var ref to defQueue, so cannot just reassign the one
                //on context.
                apsp.apply(defQueue,
                           [defQueue.length - 1, 0].concat(globalDefQueue));
                globalDefQueue = [];
            }
        }

        /**
         * Helper function that creates a require function object to give to
         * modules that ask for it as a dependency. It needs to be specific
         * per module because of the implication of path mappings that may
         * need to be relative to the module name.
         */
        function makeRequire(mod, enableBuildCallback, altRequire) {
            var relMap = mod && mod.map,
                modRequire = makeContextModuleFunc(altRequire || context.require,
                                                   relMap,
                                                   enableBuildCallback);

            addRequireMethods(modRequire, context, relMap);
            modRequire.isBrowser = isBrowser;

            return modRequire;
        }

        handlers = {
            'require': function (mod) {
                return makeRequire(mod);
            },
            'exports': function (mod) {
                mod.usingExports = true;
                if (mod.map.isDefine) {
                    return (mod.exports = defined[mod.map.id] = {});
                }
            },
            'module': function (mod) {
                return (mod.module = {
                    id: mod.map.id,
                    uri: mod.map.url,
                    config: function () {
                        return (config.config && config.config[mod.map.id]) || {};
                    },
                    exports: defined[mod.map.id]
                });
            }
        };

        function removeWaiting(id) {
            //Clean up machinery used for waiting modules.
            delete registry[id];

            each(waitAry, function (mod, i) {
                if (mod.map.id === id) {
                    waitAry.splice(i, 1);
                    if (!mod.defined) {
                        context.waitCount -= 1;
                    }
                    return true;
                }
            });
        }

        function findCycle(mod, traced) {
            var id = mod.map.id,
                depArray = mod.depMaps,
                foundModule;

            //Do not bother with unitialized modules or not yet enabled
            //modules.
            if (!mod.inited) {
                return;
            }

            //Found the cycle.
            if (traced[id]) {
                return mod;
            }

            traced[id] = true;

            //Trace through the dependencies.
            each(depArray, function (depMap) {
                var depId = depMap.id,
                    depMod = registry[depId];

                if (!depMod) {
                    return;
                }

                if (!depMod.inited || !depMod.enabled) {
                    //Dependency is not inited, so this cannot
                    //be used to determine a cycle.
                    foundModule = null;
                    delete traced[id];
                    return true;
                }

                //mixin traced to a new object for each dependency, so that
                //sibling dependencies in this object to not generate a
                //false positive match on a cycle. Ideally an Object.create
                //type of prototype delegation would be used here, but
                //optimizing for file size vs. execution speed since hopefully
                //the trees are small for circular dependency scans relative
                //to the full app perf.
                return (foundModule = findCycle(depMod, mixin({}, traced)));
            });

            return foundModule;
        }

        function forceExec(mod, traced, uninited) {
            var id = mod.map.id,
                depArray = mod.depMaps;

            if (!mod.inited || !mod.map.isDefine) {
                return;
            }

            if (traced[id]) {
                return defined[id];
            }

            traced[id] = mod;

            each(depArray, function (depMap) {
                var depId = depMap.id,
                    depMod = registry[depId],
                    value;

                if (handlers[depId]) {
                    return;
                }

                if (depMod) {
                    if (!depMod.inited || !depMod.enabled) {
                        //Dependency is not inited,
                        //so this module cannot be
                        //given a forced value yet.
                        uninited[id] = true;
                        return;
                    }

                    //Get the value for the current dependency
                    value = forceExec(depMod, traced, uninited);

                    //Even with forcing it may not be done,
                    //in particular if the module is waiting
                    //on a plugin resource.
                    if (!uninited[depId]) {
                        mod.defineDepById(depId, value);
                    }
                }
            });

            mod.check(true);

            return defined[id];
        }

        function modCheck(mod) {
            mod.check();
        }

        function checkLoaded() {
            var map, modId, err, usingPathFallback,
                waitInterval = config.waitSeconds * 1000,
                //It is possible to disable the wait interval by using waitSeconds of 0.
                expired = waitInterval && (context.startTime + waitInterval) < new Date().getTime(),
                noLoads = [],
                stillLoading = false,
                needCycleCheck = true;

            //Do not bother if this call was a result of a cycle break.
            if (inCheckLoaded) {
                return;
            }

            inCheckLoaded = true;

            //Figure out the state of all the modules.
            eachProp(registry, function (mod) {
                map = mod.map;
                modId = map.id;

                //Skip things that are not enabled or in error state.
                if (!mod.enabled) {
                    return;
                }

                if (!mod.error) {
                    //If the module should be executed, and it has not
                    //been inited and time is up, remember it.
                    if (!mod.inited && expired) {
                        if (hasPathFallback(modId)) {
                            usingPathFallback = true;
                            stillLoading = true;
                        } else {
                            noLoads.push(modId);
                            removeScript(modId);
                        }
                    } else if (!mod.inited && mod.fetched && map.isDefine) {
                        stillLoading = true;
                        if (!map.prefix) {
                            //No reason to keep looking for unfinished
                            //loading. If the only stillLoading is a
                            //plugin resource though, keep going,
                            //because it may be that a plugin resource
                            //is waiting on a non-plugin cycle.
                            return (needCycleCheck = false);
                        }
                    }
                }
            });

            if (expired && noLoads.length) {
                //If wait time expired, throw error of unloaded modules.
                err = makeError('timeout', 'Load timeout for modules: ' + noLoads, null, noLoads);
                err.contextName = context.contextName;
                return onError(err);
            }

            //Not expired, check for a cycle.
            if (needCycleCheck) {

                each(waitAry, function (mod) {
                    if (mod.defined) {
                        return;
                    }

                    var cycleMod = findCycle(mod, {}),
                        traced = {};

                    if (cycleMod) {
                        forceExec(cycleMod, traced, {});

                        //traced modules may have been
                        //removed from the registry, but
                        //their listeners still need to
                        //be called.
                        eachProp(traced, modCheck);
                    }
                });

                //Now that dependencies have
                //been satisfied, trigger the
                //completion check that then
                //notifies listeners.
                eachProp(registry, modCheck);
            }

            //If still waiting on loads, and the waiting load is something
            //other than a plugin resource, or there are still outstanding
            //scripts, then just try back later.
            if ((!expired || usingPathFallback) && stillLoading) {
                //Something is still waiting to load. Wait for it, but only
                //if a timeout is not already in effect.
                if ((isBrowser || isWebWorker) && !checkLoadedTimeoutId) {
                    checkLoadedTimeoutId = setTimeout(function () {
                        checkLoadedTimeoutId = 0;
                        checkLoaded();
                    }, 50);
                }
            }

            inCheckLoaded = false;
        }

        Module = function (map) {
            this.events = undefEvents[map.id] || {};
            this.map = map;
            this.shim = config.shim[map.id];
            this.depExports = [];
            this.depMaps = [];
            this.depMatched = [];
            this.pluginMaps = {};
            this.depCount = 0;

            /* this.exports this.factory
               this.depMaps = [],
               this.enabled, this.fetched
            */
        };

        Module.prototype = {
            init: function (depMaps, factory, errback, options) {
                options = options || {};

                //Do not do more inits if already done. Can happen if there
                //are multiple define calls for the same module. That is not
                //a normal, common case, but it is also not unexpected.
                if (this.inited) {
                    return;
                }

                this.factory = factory;

                if (errback) {
                    //Register for errors on this module.
                    this.on('error', errback);
                } else if (this.events.error) {
                    //If no errback already, but there are error listeners
                    //on this module, set up an errback to pass to the deps.
                    errback = bind(this, function (err) {
                        this.emit('error', err);
                    });
                }

                //Do a copy of the dependency array, so that
                //source inputs are not modified. For example
                //"shim" deps are passed in here directly, and
                //doing a direct modification of the depMaps array
                //would affect that config.
                this.depMaps = depMaps && depMaps.slice(0);
                this.depMaps.rjsSkipMap = depMaps.rjsSkipMap;

                this.errback = errback;

                //Indicate this module has be initialized
                this.inited = true;

                this.ignore = options.ignore;

                //Could have option to init this module in enabled mode,
                //or could have been previously marked as enabled. However,
                //the dependencies are not known until init is called. So
                //if enabled previously, now trigger dependencies as enabled.
                if (options.enabled || this.enabled) {
                    //Enable this module and dependencies.
                    //Will call this.check()
                    this.enable();
                } else {
                    this.check();
                }
            },

            defineDepById: function (id, depExports) {
                var i;

                //Find the index for this dependency.
                each(this.depMaps, function (map, index) {
                    if (map.id === id) {
                        i = index;
                        return true;
                    }
                });

                return this.defineDep(i, depExports);
            },

            defineDep: function (i, depExports) {
                //Because of cycles, defined callback for a given
                //export can be called more than once.
                if (!this.depMatched[i]) {
                    this.depMatched[i] = true;
                    this.depCount -= 1;
                    this.depExports[i] = depExports;
                }
            },

            fetch: function () {
                if (this.fetched) {
                    return;
                }
                this.fetched = true;

                context.startTime = (new Date()).getTime();

                var map = this.map;

                //If the manager is for a plugin managed resource,
                //ask the plugin to load it now.
                if (this.shim) {
                    makeRequire(this, true)(this.shim.deps || [], bind(this, function () {
                        return map.prefix ? this.callPlugin() : this.load();
                    }));
                } else {
                    //Regular dependency.
                    return map.prefix ? this.callPlugin() : this.load();
                }
            },

            load: function () {
                var url = this.map.url;

                //Regular dependency.
                if (!urlFetched[url]) {
                    urlFetched[url] = true;
                    context.load(this.map.id, url);
                }
            },

            /**
             * Checks is the module is ready to define itself, and if so,
             * define it. If the silent argument is true, then it will just
             * define, but not notify listeners, and not ask for a context-wide
             * check of all loaded modules. That is useful for cycle breaking.
             */
            check: function (silent) {
                if (!this.enabled || this.enabling) {
                    return;
                }

                var err, cjsModule,
                    id = this.map.id,
                    depExports = this.depExports,
                    exports = this.exports,
                    factory = this.factory;

                if (!this.inited) {
                    this.fetch();
                } else if (this.error) {
                    this.emit('error', this.error);
                } else if (!this.defining) {
                    //The factory could trigger another require call
                    //that would result in checking this module to
                    //define itself again. If already in the process
                    //of doing that, skip this work.
                    this.defining = true;

                    if (this.depCount < 1 && !this.defined) {
                        if (isFunction(factory)) {
                            //If there is an error listener, favor passing
                            //to that instead of throwing an error.
                            if (this.events.error) {
                                try {
                                    exports = context.execCb(id, factory, depExports, exports);
                                } catch (e) {
                                    err = e;
                                }
                            } else {
                                exports = context.execCb(id, factory, depExports, exports);
                            }

                            if (this.map.isDefine) {
                                //If setting exports via 'module' is in play,
                                //favor that over return value and exports. After that,
                                //favor a non-undefined return value over exports use.
                                cjsModule = this.module;
                                if (cjsModule &&
                                        cjsModule.exports !== undefined &&
                                        //Make sure it is not already the exports value
                                        cjsModule.exports !== this.exports) {
                                    exports = cjsModule.exports;
                                } else if (exports === undefined && this.usingExports) {
                                    //exports already set the defined value.
                                    exports = this.exports;
                                }
                            }

                            if (err) {
                                err.requireMap = this.map;
                                err.requireModules = [this.map.id];
                                err.requireType = 'define';
                                return onError((this.error = err));
                            }

                        } else {
                            //Just a literal value
                            exports = factory;
                        }

                        this.exports = exports;

                        if (this.map.isDefine && !this.ignore) {
                            defined[id] = exports;

                            if (req.onResourceLoad) {
                                req.onResourceLoad(context, this.map, this.depMaps);
                            }
                        }

                        //Clean up
                        delete registry[id];

                        this.defined = true;
                        context.waitCount -= 1;
                        if (context.waitCount === 0) {
                            //Clear the wait array used for cycles.
                            waitAry = [];
                        }
                    }

                    //Finished the define stage. Allow calling check again
                    //to allow define notifications below in the case of a
                    //cycle.
                    this.defining = false;

                    if (!silent) {
                        if (this.defined && !this.defineEmitted) {
                            this.defineEmitted = true;
                            this.emit('defined', this.exports);
                            this.defineEmitComplete = true;
                        }
                    }
                }
            },

            callPlugin: function () {
                var map = this.map,
                    id = map.id,
                    pluginMap = makeModuleMap(map.prefix, null, false, true);

                on(pluginMap, 'defined', bind(this, function (plugin) {
                    var load, normalizedMap, normalizedMod,
                        name = this.map.name,
                        parentName = this.map.parentMap ? this.map.parentMap.name : null;

                    //If current map is not normalized, wait for that
                    //normalized name to load instead of continuing.
                    if (this.map.unnormalized) {
                        //Normalize the ID if the plugin allows it.
                        if (plugin.normalize) {
                            name = plugin.normalize(name, function (name) {
                                return normalize(name, parentName, true);
                            }) || '';
                        }

                        normalizedMap = makeModuleMap(map.prefix + '!' + name,
                                                      this.map.parentMap,
                                                      false,
                                                      true);
                        on(normalizedMap,
                            'defined', bind(this, function (value) {
                                this.init([], function () { return value; }, null, {
                                    enabled: true,
                                    ignore: true
                                });
                            }));
                        normalizedMod = registry[normalizedMap.id];
                        if (normalizedMod) {
                            if (this.events.error) {
                                normalizedMod.on('error', bind(this, function (err) {
                                    this.emit('error', err);
                                }));
                            }
                            normalizedMod.enable();
                        }

                        return;
                    }

                    load = bind(this, function (value) {
                        this.init([], function () { return value; }, null, {
                            enabled: true
                        });
                    });

                    load.error = bind(this, function (err) {
                        this.inited = true;
                        this.error = err;
                        err.requireModules = [id];

                        //Remove temp unnormalized modules for this module,
                        //since they will never be resolved otherwise now.
                        eachProp(registry, function (mod) {
                            if (mod.map.id.indexOf(id + '_unnormalized') === 0) {
                                removeWaiting(mod.map.id);
                            }
                        });

                        onError(err);
                    });

                    //Allow plugins to load other code without having to know the
                    //context or how to 'complete' the load.
                    load.fromText = function (moduleName, text) {
                        /*jslint evil: true */
                        var hasInteractive = useInteractive;

                        //Turn off interactive script matching for IE for any define
                        //calls in the text, then turn it back on at the end.
                        if (hasInteractive) {
                            useInteractive = false;
                        }

                        //Prime the system by creating a module instance for
                        //it.
                        getModule(makeModuleMap(moduleName));

                        req.exec(text);

                        if (hasInteractive) {
                            useInteractive = true;
                        }

                        //Support anonymous modules.
                        context.completeLoad(moduleName);
                    };

                    //Use parentName here since the plugin's name is not reliable,
                    //could be some weird string with no path that actually wants to
                    //reference the parentName's path.
                    plugin.load(map.name, makeRequire(map.parentMap, true, function (deps, cb, er) {
                        deps.rjsSkipMap = true;
                        return context.require(deps, cb, er);
                    }), load, config);
                }));

                context.enable(pluginMap, this);
                this.pluginMaps[pluginMap.id] = pluginMap;
            },

            enable: function () {
                this.enabled = true;

                if (!this.waitPushed) {
                    waitAry.push(this);
                    context.waitCount += 1;
                    this.waitPushed = true;
                }

                //Set flag mentioning that the module is enabling,
                //so that immediate calls to the defined callbacks
                //for dependencies do not trigger inadvertent load
                //with the depCount still being zero.
                this.enabling = true;

                //Enable each dependency
                each(this.depMaps, bind(this, function (depMap, i) {
                    var id, mod, handler;

                    if (typeof depMap === 'string') {
                        //Dependency needs to be converted to a depMap
                        //and wired up to this module.
                        depMap = makeModuleMap(depMap,
                                               (this.map.isDefine ? this.map : this.map.parentMap),
                                               false,
                                               !this.depMaps.rjsSkipMap);
                        this.depMaps[i] = depMap;

                        handler = handlers[depMap.id];

                        if (handler) {
                            this.depExports[i] = handler(this);
                            return;
                        }

                        this.depCount += 1;

                        on(depMap, 'defined', bind(this, function (depExports) {
                            this.defineDep(i, depExports);
                            this.check();
                        }));

                        if (this.errback) {
                            on(depMap, 'error', this.errback);
                        }
                    }

                    id = depMap.id;
                    mod = registry[id];

                    //Skip special modules like 'require', 'exports', 'module'
                    //Also, don't call enable if it is already enabled,
                    //important in circular dependency cases.
                    if (!handlers[id] && mod && !mod.enabled) {
                        context.enable(depMap, this);
                    }
                }));

                //Enable each plugin that is used in
                //a dependency
                eachProp(this.pluginMaps, bind(this, function (pluginMap) {
                    var mod = registry[pluginMap.id];
                    if (mod && !mod.enabled) {
                        context.enable(pluginMap, this);
                    }
                }));

                this.enabling = false;

                this.check();
            },

            on: function (name, cb) {
                var cbs = this.events[name];
                if (!cbs) {
                    cbs = this.events[name] = [];
                }
                cbs.push(cb);
            },

            emit: function (name, evt) {
                each(this.events[name], function (cb) {
                    cb(evt);
                });
                if (name === 'error') {
                    //Now that the error handler was triggered, remove
                    //the listeners, since this broken Module instance
                    //can stay around for a while in the registry/waitAry.
                    delete this.events[name];
                }
            }
        };

        function callGetModule(args) {
            getModule(makeModuleMap(args[0], null, true)).init(args[1], args[2]);
        }

        function removeListener(node, func, name, ieName) {
            //Favor detachEvent because of IE9
            //issue, see attachEvent/addEventListener comment elsewhere
            //in this file.
            if (node.detachEvent && !isOpera) {
                //Probably IE. If not it will throw an error, which will be
                //useful to know.
                if (ieName) {
                    node.detachEvent(ieName, func);
                }
            } else {
                node.removeEventListener(name, func, false);
            }
        }

        /**
         * Given an event from a script node, get the requirejs info from it,
         * and then removes the event listeners on the node.
         * @param {Event} evt
         * @returns {Object}
         */
        function getScriptData(evt) {
            //Using currentTarget instead of target for Firefox 2.0's sake. Not
            //all old browsers will be supported, but this one was easy enough
            //to support and still makes sense.
            var node = evt.currentTarget || evt.srcElement;

            //Remove the listeners once here.
            removeListener(node, context.onScriptLoad, 'load', 'onreadystatechange');
            removeListener(node, context.onScriptError, 'error');

            return {
                node: node,
                id: node && node.getAttribute('data-requiremodule')
            };
        }

        return (context = {
            config: config,
            contextName: contextName,
            registry: registry,
            defined: defined,
            urlFetched: urlFetched,
            waitCount: 0,
            defQueue: defQueue,
            Module: Module,
            makeModuleMap: makeModuleMap,

            /**
             * Set a configuration for the context.
             * @param {Object} cfg config object to integrate.
             */
            configure: function (cfg) {
                //Make sure the baseUrl ends in a slash.
                if (cfg.baseUrl) {
                    if (cfg.baseUrl.charAt(cfg.baseUrl.length - 1) !== '/') {
                        cfg.baseUrl += '/';
                    }
                }

                //Save off the paths and packages since they require special processing,
                //they are additive.
                var pkgs = config.pkgs,
                    shim = config.shim,
                    paths = config.paths,
                    map = config.map;

                //Mix in the config values, favoring the new values over
                //existing ones in context.config.
                mixin(config, cfg, true);

                //Merge paths.
                config.paths = mixin(paths, cfg.paths, true);

                //Merge map
                if (cfg.map) {
                    config.map = mixin(map || {}, cfg.map, true, true);
                }

                //Merge shim
                if (cfg.shim) {
                    eachProp(cfg.shim, function (value, id) {
                        //Normalize the structure
                        if (isArray(value)) {
                            value = {
                                deps: value
                            };
                        }
                        if (value.exports && !value.exports.__buildReady) {
                            value.exports = context.makeShimExports(value.exports);
                        }
                        shim[id] = value;
                    });
                    config.shim = shim;
                }

                //Adjust packages if necessary.
                if (cfg.packages) {
                    each(cfg.packages, function (pkgObj) {
                        var location;

                        pkgObj = typeof pkgObj === 'string' ? { name: pkgObj } : pkgObj;
                        location = pkgObj.location;

                        //Create a brand new object on pkgs, since currentPackages can
                        //be passed in again, and config.pkgs is the internal transformed
                        //state for all package configs.
                        pkgs[pkgObj.name] = {
                            name: pkgObj.name,
                            location: location || pkgObj.name,
                            //Remove leading dot in main, so main paths are normalized,
                            //and remove any trailing .js, since different package
                            //envs have different conventions: some use a module name,
                            //some use a file name.
                            main: (pkgObj.main || 'main')
                                  .replace(currDirRegExp, '')
                                  .replace(jsSuffixRegExp, '')
                        };
                    });

                    //Done with modifications, assing packages back to context config
                    config.pkgs = pkgs;
                }

                //If there are any "waiting to execute" modules in the registry,
                //update the maps for them, since their info, like URLs to load,
                //may have changed.
                eachProp(registry, function (mod, id) {
                    //If module already has init called, since it is too
                    //late to modify them, and ignore unnormalized ones
                    //since they are transient.
                    if (!mod.inited && !mod.map.unnormalized) {
                        mod.map = makeModuleMap(id);
                    }
                });

                //If a deps array or a config callback is specified, then call
                //require with those args. This is useful when require is defined as a
                //config object before require.js is loaded.
                if (cfg.deps || cfg.callback) {
                    context.require(cfg.deps || [], cfg.callback);
                }
            },

            makeShimExports: function (exports) {
                var func;
                if (typeof exports === 'string') {
                    func = function () {
                        return getGlobal(exports);
                    };
                    //Save the exports for use in nodefine checking.
                    func.exports = exports;
                    return func;
                } else {
                    return function () {
                        return exports.apply(global, arguments);
                    };
                }
            },

            requireDefined: function (id, relMap) {
                return hasProp(defined, makeModuleMap(id, relMap, false, true).id);
            },

            requireSpecified: function (id, relMap) {
                id = makeModuleMap(id, relMap, false, true).id;
                return hasProp(defined, id) || hasProp(registry, id);
            },

            require: function (deps, callback, errback, relMap) {
                var moduleName, id, map, requireMod, args;
                if (typeof deps === 'string') {
                    if (isFunction(callback)) {
                        //Invalid call
                        return onError(makeError('requireargs', 'Invalid require call'), errback);
                    }

                    //Synchronous access to one module. If require.get is
                    //available (as in the Node adapter), prefer that.
                    //In this case deps is the moduleName and callback is
                    //the relMap
                    if (req.get) {
                        return req.get(context, deps, callback);
                    }

                    //Just return the module wanted. In this scenario, the
                    //second arg (if passed) is just the relMap.
                    moduleName = deps;
                    relMap = callback;

                    //Normalize module name, if it contains . or ..
                    map = makeModuleMap(moduleName, relMap, false, true);
                    id = map.id;

                    if (!hasProp(defined, id)) {
                        return onError(makeError('notloaded', 'Module name "' +
                                    id +
                                    '" has not been loaded yet for context: ' +
                                    contextName));
                    }
                    return defined[id];
                }

                //Callback require. Normalize args. if callback or errback is
                //not a function, it means it is a relMap. Test errback first.
                if (errback && !isFunction(errback)) {
                    relMap = errback;
                    errback = undefined;
                }
                if (callback && !isFunction(callback)) {
                    relMap = callback;
                    callback = undefined;
                }

                //Any defined modules in the global queue, intake them now.
                takeGlobalQueue();

                //Make sure any remaining defQueue items get properly processed.
                while (defQueue.length) {
                    args = defQueue.shift();
                    if (args[0] === null) {
                        return onError(makeError('mismatch', 'Mismatched anonymous define() module: ' + args[args.length - 1]));
                    } else {
                        //args are id, deps, factory. Should be normalized by the
                        //define() function.
                        callGetModule(args);
                    }
                }

                //Mark all the dependencies as needing to be loaded.
                requireMod = getModule(makeModuleMap(null, relMap));

                requireMod.init(deps, callback, errback, {
                    enabled: true
                });

                checkLoaded();

                return context.require;
            },

            undef: function (id) {
                //Bind any waiting define() calls to this context,
                //fix for #408
                takeGlobalQueue();

                var map = makeModuleMap(id, null, true),
                    mod = registry[id];

                delete defined[id];
                delete urlFetched[map.url];
                delete undefEvents[id];

                if (mod) {
                    //Hold on to listeners in case the
                    //module will be attempted to be reloaded
                    //using a different config.
                    if (mod.events.defined) {
                        undefEvents[id] = mod.events;
                    }

                    removeWaiting(id);
                }
            },

            /**
             * Called to enable a module if it is still in the registry
             * awaiting enablement. parent module is passed in for context,
             * used by the optimizer.
             */
            enable: function (depMap, parent) {
                var mod = registry[depMap.id];
                if (mod) {
                    getModule(depMap).enable();
                }
            },

            /**
             * Internal method used by environment adapters to complete a load event.
             * A load event could be a script load or just a load pass from a synchronous
             * load call.
             * @param {String} moduleName the name of the module to potentially complete.
             */
            completeLoad: function (moduleName) {
                var found, args, mod,
                    shim = config.shim[moduleName] || {},
                    shExports = shim.exports && shim.exports.exports;

                takeGlobalQueue();

                while (defQueue.length) {
                    args = defQueue.shift();
                    if (args[0] === null) {
                        args[0] = moduleName;
                        //If already found an anonymous module and bound it
                        //to this name, then this is some other anon module
                        //waiting for its completeLoad to fire.
                        if (found) {
                            break;
                        }
                        found = true;
                    } else if (args[0] === moduleName) {
                        //Found matching define call for this script!
                        found = true;
                    }

                    callGetModule(args);
                }

                //Do this after the cycle of callGetModule in case the result
                //of those calls/init calls changes the registry.
                mod = registry[moduleName];

                if (!found && !defined[moduleName] && mod && !mod.inited) {
                    if (config.enforceDefine && (!shExports || !getGlobal(shExports))) {
                        if (hasPathFallback(moduleName)) {
                            return;
                        } else {
                            return onError(makeError('nodefine',
                                             'No define call for ' + moduleName,
                                             null,
                                             [moduleName]));
                        }
                    } else {
                        //A script that does not call define(), so just simulate
                        //the call for it.
                        callGetModule([moduleName, (shim.deps || []), shim.exports]);
                    }
                }

                checkLoaded();
            },

            /**
             * Converts a module name + .extension into an URL path.
             * *Requires* the use of a module name. It does not support using
             * plain URLs like nameToUrl.
             */
            toUrl: function (moduleNamePlusExt, relModuleMap) {
                var index = moduleNamePlusExt.lastIndexOf('.'),
                    ext = null;

                if (index !== -1) {
                    ext = moduleNamePlusExt.substring(index, moduleNamePlusExt.length);
                    moduleNamePlusExt = moduleNamePlusExt.substring(0, index);
                }

                return context.nameToUrl(normalize(moduleNamePlusExt, relModuleMap && relModuleMap.id, true),
                                         ext);
            },

            /**
             * Converts a module name to a file path. Supports cases where
             * moduleName may actually be just an URL.
             * Note that it **does not** call normalize on the moduleName,
             * it is assumed to have already been normalized. This is an
             * internal API, not a public one. Use toUrl for the public API.
             */
            nameToUrl: function (moduleName, ext) {
                var paths, pkgs, pkg, pkgPath, syms, i, parentModule, url,
                    parentPath;

                //If a colon is in the URL, it indicates a protocol is used and it is just
                //an URL to a file, or if it starts with a slash, contains a query arg (i.e. ?)
                //or ends with .js, then assume the user meant to use an url and not a module id.
                //The slash is important for protocol-less URLs as well as full paths.
                if (req.jsExtRegExp.test(moduleName)) {
                    //Just a plain path, not module name lookup, so just return it.
                    //Add extension if it is included. This is a bit wonky, only non-.js things pass
                    //an extension, this method probably needs to be reworked.
                    url = moduleName + (ext || '');
                } else {
                    //A module that needs to be converted to a path.
                    paths = config.paths;
                    pkgs = config.pkgs;

                    syms = moduleName.split('/');
                    //For each module name segment, see if there is a path
                    //registered for it. Start with most specific name
                    //and work up from it.
                    for (i = syms.length; i > 0; i -= 1) {
                        parentModule = syms.slice(0, i).join('/');
                        pkg = pkgs[parentModule];
                        parentPath = paths[parentModule];
                        if (parentPath) {
                            //If an array, it means there are a few choices,
                            //Choose the one that is desired
                            if (isArray(parentPath)) {
                                parentPath = parentPath[0];
                            }
                            syms.splice(0, i, parentPath);
                            break;
                        } else if (pkg) {
                            //If module name is just the package name, then looking
                            //for the main module.
                            if (moduleName === pkg.name) {
                                pkgPath = pkg.location + '/' + pkg.main;
                            } else {
                                pkgPath = pkg.location;
                            }
                            syms.splice(0, i, pkgPath);
                            break;
                        }
                    }

                    //Join the path parts together, then figure out if baseUrl is needed.
                    url = syms.join('/');
                    url += (ext || (/\?/.test(url) ? '' : '.js'));
                    url = (url.charAt(0) === '/' || url.match(/^[\w\+\.\-]+:/) ? '' : config.baseUrl) + url;
                }

                return config.urlArgs ? url +
                                        ((url.indexOf('?') === -1 ? '?' : '&') +
                                         config.urlArgs) : url;
            },

            //Delegates to req.load. Broken out as a separate function to
            //allow overriding in the optimizer.
            load: function (id, url) {
                req.load(context, id, url);
            },

            /**
             * Executes a module callack function. Broken out as a separate function
             * solely to allow the build system to sequence the files in the built
             * layer in the right sequence.
             *
             * @private
             */
            execCb: function (name, callback, args, exports) {
                return callback.apply(exports, args);
            },

            /**
             * callback for script loads, used to check status of loading.
             *
             * @param {Event} evt the event from the browser for the script
             * that was loaded.
             */
            onScriptLoad: function (evt) {
                //Using currentTarget instead of target for Firefox 2.0's sake. Not
                //all old browsers will be supported, but this one was easy enough
                //to support and still makes sense.
                if (evt.type === 'load' ||
                        (readyRegExp.test((evt.currentTarget || evt.srcElement).readyState))) {
                    //Reset interactive script so a script node is not held onto for
                    //to long.
                    interactiveScript = null;

                    //Pull out the name of the module and the context.
                    var data = getScriptData(evt);
                    context.completeLoad(data.id);
                }
            },

            /**
             * Callback for script errors.
             */
            onScriptError: function (evt) {
                var data = getScriptData(evt);
                if (!hasPathFallback(data.id)) {
                    return onError(makeError('scripterror', 'Script error', evt, [data.id]));
                }
            }
        });
    }

    /**
     * Main entry point.
     *
     * If the only argument to require is a string, then the module that
     * is represented by that string is fetched for the appropriate context.
     *
     * If the first argument is an array, then it will be treated as an array
     * of dependency string names to fetch. An optional function callback can
     * be specified to execute when all of those dependencies are available.
     *
     * Make a local req variable to help Caja compliance (it assumes things
     * on a require that are not standardized), and to give a short
     * name for minification/local scope use.
     */
    req = requirejs = function (deps, callback, errback, optional) {

        //Find the right context, use default
        var context, config,
            contextName = defContextName;

        // Determine if have config object in the call.
        if (!isArray(deps) && typeof deps !== 'string') {
            // deps is a config object
            config = deps;
            if (isArray(callback)) {
                // Adjust args if there are dependencies
                deps = callback;
                callback = errback;
                errback = optional;
            } else {
                deps = [];
            }
        }

        if (config && config.context) {
            contextName = config.context;
        }

        context = contexts[contextName];
        if (!context) {
            context = contexts[contextName] = req.s.newContext(contextName);
        }

        if (config) {
            context.configure(config);
        }

        return context.require(deps, callback, errback);
    };

    /**
     * Support require.config() to make it easier to cooperate with other
     * AMD loaders on globally agreed names.
     */
    req.config = function (config) {
        return req(config);
    };

    /**
     * Export require as a global, but only if it does not already exist.
     */
    if (!require) {
        require = req;
    }

    req.version = version;

    //Used to filter out dependencies that are already paths.
    req.jsExtRegExp = /^\/|:|\?|\.js$/;
    req.isBrowser = isBrowser;
    s = req.s = {
        contexts: contexts,
        newContext: newContext
    };

    //Create default context.
    req({});

    //Exports some context-sensitive methods on global require, using
    //default context if no context specified.
    addRequireMethods(req);

    if (isBrowser) {
        head = s.head = document.getElementsByTagName('head')[0];
        //If BASE tag is in play, using appendChild is a problem for IE6.
        //When that browser dies, this can be removed. Details in this jQuery bug:
        //http://dev.jquery.com/ticket/2709
        baseElement = document.getElementsByTagName('base')[0];
        if (baseElement) {
            head = s.head = baseElement.parentNode;
        }
    }

    /**
     * Any errors that require explicitly generates will be passed to this
     * function. Intercept/override it if you want custom error handling.
     * @param {Error} err the error object.
     */
    req.onError = function (err) {
        throw err;
    };

    /**
     * Does the request to load a module for the browser case.
     * Make this a separate function to allow other environments
     * to override it.
     *
     * @param {Object} context the require context to find state.
     * @param {String} moduleName the name of the module.
     * @param {Object} url the URL to the module.
     */
    req.load = function (context, moduleName, url) {
        var config = (context && context.config) || {},
            node;
        if (isBrowser) {
            //In the browser so use a script tag
            node = config.xhtml ?
                    document.createElementNS('http://www.w3.org/1999/xhtml', 'html:script') :
                    document.createElement('script');
            node.type = config.scriptType || 'text/javascript';
            node.charset = 'utf-8';
            node.async = true;

            node.setAttribute('data-requirecontext', context.contextName);
            node.setAttribute('data-requiremodule', moduleName);

            //Set up load listener. Test attachEvent first because IE9 has
            //a subtle issue in its addEventListener and script onload firings
            //that do not match the behavior of all other browsers with
            //addEventListener support, which fire the onload event for a
            //script right after the script execution. See:
            //https://connect.microsoft.com/IE/feedback/details/648057/script-onload-event-is-not-fired-immediately-after-script-execution
            //UNFORTUNATELY Opera implements attachEvent but does not follow the script
            //script execution mode.
            if (node.attachEvent &&
                    //Check if node.attachEvent is artificially added by custom script or
                    //natively supported by browser
                    //read https://github.com/jrburke/requirejs/issues/187
                    //if we can NOT find [native code] then it must NOT natively supported.
                    //in IE8, node.attachEvent does not have toString()
                    //Note the test for "[native code" with no closing brace, see:
                    //https://github.com/jrburke/requirejs/issues/273
                    !(node.attachEvent.toString && node.attachEvent.toString().indexOf('[native code') < 0) &&
                    !isOpera) {
                //Probably IE. IE (at least 6-8) do not fire
                //script onload right after executing the script, so
                //we cannot tie the anonymous define call to a name.
                //However, IE reports the script as being in 'interactive'
                //readyState at the time of the define call.
                useInteractive = true;

                node.attachEvent('onreadystatechange', context.onScriptLoad);
                //It would be great to add an error handler here to catch
                //404s in IE9+. However, onreadystatechange will fire before
                //the error handler, so that does not help. If addEvenListener
                //is used, then IE will fire error before load, but we cannot
                //use that pathway given the connect.microsoft.com issue
                //mentioned above about not doing the 'script execute,
                //then fire the script load event listener before execute
                //next script' that other browsers do.
                //Best hope: IE10 fixes the issues,
                //and then destroys all installs of IE 6-9.
                //node.attachEvent('onerror', context.onScriptError);
            } else {
                node.addEventListener('load', context.onScriptLoad, false);
                node.addEventListener('error', context.onScriptError, false);
            }
            node.src = url;

            //For some cache cases in IE 6-8, the script executes before the end
            //of the appendChild execution, so to tie an anonymous define
            //call to the module name (which is stored on the node), hold on
            //to a reference to this node, but clear after the DOM insertion.
            currentlyAddingScript = node;
            if (baseElement) {
                head.insertBefore(node, baseElement);
            } else {
                head.appendChild(node);
            }
            currentlyAddingScript = null;

            return node;
        } else if (isWebWorker) {
            //In a web worker, use importScripts. This is not a very
            //efficient use of importScripts, importScripts will block until
            //its script is downloaded and evaluated. However, if web workers
            //are in play, the expectation that a build has been done so that
            //only one script needs to be loaded anyway. This may need to be
            //reevaluated if other use cases become common.
            importScripts(url);

            //Account for anonymous modules
            context.completeLoad(moduleName);
        }
    };

    function getInteractiveScript() {
        if (interactiveScript && interactiveScript.readyState === 'interactive') {
            return interactiveScript;
        }

        eachReverse(scripts(), function (script) {
            if (script.readyState === 'interactive') {
                return (interactiveScript = script);
            }
        });
        return interactiveScript;
    }

    //Look for a data-main script attribute, which could also adjust the baseUrl.
    if (isBrowser) {
        //Figure out baseUrl. Get it from the script tag with require.js in it.
        eachReverse(scripts(), function (script) {
            //Set the 'head' where we can append children by
            //using the script's parent.
            if (!head) {
                head = script.parentNode;
            }

            //Look for a data-main attribute to set main script for the page
            //to load. If it is there, the path to data main becomes the
            //baseUrl, if it is not already set.
            dataMain = script.getAttribute('data-main');
            if (dataMain) {
                //Set final baseUrl if there is not already an explicit one.
                if (!cfg.baseUrl) {
                    //Pull off the directory of data-main for use as the
                    //baseUrl.
                    src = dataMain.split('/');
                    mainScript = src.pop();
                    subPath = src.length ? src.join('/')  + '/' : './';

                    cfg.baseUrl = subPath;
                    dataMain = mainScript;
                }

                //Strip off any trailing .js since dataMain is now
                //like a module name.
                dataMain = dataMain.replace(jsSuffixRegExp, '');

                //Put the data-main script in the files to load.
                cfg.deps = cfg.deps ? cfg.deps.concat(dataMain) : [dataMain];

                return true;
            }
        });
    }

    /**
     * The function that handles definitions of modules. Differs from
     * require() in that a string for the module should be the first argument,
     * and the function to execute after dependencies are loaded should
     * return a value to define the module corresponding to the first argument's
     * name.
     */
    define = function (name, deps, callback) {
        var node, context;

        //Allow for anonymous functions
        if (typeof name !== 'string') {
            //Adjust args appropriately
            callback = deps;
            deps = name;
            name = null;
        }

        //This module may not have dependencies
        if (!isArray(deps)) {
            callback = deps;
            deps = [];
        }

        //If no name, and callback is a function, then figure out if it a
        //CommonJS thing with dependencies.
        if (!deps.length && isFunction(callback)) {
            //Remove comments from the callback string,
            //look for require calls, and pull them into the dependencies,
            //but only if there are function args.
            if (callback.length) {
                callback
                    .toString()
                    .replace(commentRegExp, '')
                    .replace(cjsRequireRegExp, function (match, dep) {
                        deps.push(dep);
                    });

                //May be a CommonJS thing even without require calls, but still
                //could use exports, and module. Avoid doing exports and module
                //work though if it just needs require.
                //REQUIRES the function to expect the CommonJS variables in the
                //order listed below.
                deps = (callback.length === 1 ? ['require'] : ['require', 'exports', 'module']).concat(deps);
            }
        }

        //If in IE 6-8 and hit an anonymous define() call, do the interactive
        //work.
        if (useInteractive) {
            node = currentlyAddingScript || getInteractiveScript();
            if (node) {
                if (!name) {
                    name = node.getAttribute('data-requiremodule');
                }
                context = contexts[node.getAttribute('data-requirecontext')];
            }
        }

        //Always save off evaluating the def call until the script onload handler.
        //This allows multiple modules to be in a file without prematurely
        //tracing dependencies, and allows for anonymous module support,
        //where the module name is not known until the script onload event
        //occurs. If no context, use the global queue, and get it processed
        //in the onscript load callback.
        (context ? context.defQueue : globalDefQueue).push([name, deps, callback]);
    };

    define.amd = {
        jQuery: true
    };


    /**
     * Executes the text. Normally just uses eval, but can be modified
     * to use a better, environment-specific call. Only used for transpiling
     * loader plugins, not for plain JS modules.
     * @param {String} text the text to execute/evaluate.
     */
    req.exec = function (text) {
        /*jslint evil: true */
        return eval(text);
    };

    //Set up with config info.
    req(cfg);
}(this));
/*!
 * jQuery JavaScript Library v1.7.2
 * http://jquery.com/
 *
 * Copyright 2011, John Resig
 * Dual licensed under the MIT or GPL Version 2 licenses.
 * http://jquery.org/license
 *
 * Includes Sizzle.js
 * http://sizzlejs.com/
 * Copyright 2011, The Dojo Foundation
 * Released under the MIT, BSD, and GPL Licenses.
 *
 * Date: Wed Mar 21 12:46:34 2012 -0700
 */
(function( window, undefined ) {

// Use the correct document accordingly with window argument (sandbox)
var document = window.document,
	navigator = window.navigator,
	location = window.location;
var jQuery = (function() {

// Define a local copy of jQuery
var jQuery = function( selector, context ) {
		// The jQuery object is actually just the init constructor 'enhanced'
		return new jQuery.fn.init( selector, context, rootjQuery );
	},

	// Map over jQuery in case of overwrite
	_jQuery = window.jQuery,

	// Map over the $ in case of overwrite
	_$ = window.$,

	// A central reference to the root jQuery(document)
	rootjQuery,

	// A simple way to check for HTML strings or ID strings
	// Prioritize #id over <tag> to avoid XSS via location.hash (#9521)
	quickExpr = /^(?:[^#<]*(<[\w\W]+>)[^>]*$|#([\w\-]*)$)/,

	// Check if a string has a non-whitespace character in it
	rnotwhite = /\S/,

	// Used for trimming whitespace
	trimLeft = /^\s+/,
	trimRight = /\s+$/,

	// Match a standalone tag
	rsingleTag = /^<(\w+)\s*\/?>(?:<\/\1>)?$/,

	// JSON RegExp
	rvalidchars = /^[\],:{}\s]*$/,
	rvalidescape = /\\(?:["\\\/bfnrt]|u[0-9a-fA-F]{4})/g,
	rvalidtokens = /"[^"\\\n\r]*"|true|false|null|-?\d+(?:\.\d*)?(?:[eE][+\-]?\d+)?/g,
	rvalidbraces = /(?:^|:|,)(?:\s*\[)+/g,

	// Useragent RegExp
	rwebkit = /(webkit)[ \/]([\w.]+)/,
	ropera = /(opera)(?:.*version)?[ \/]([\w.]+)/,
	rmsie = /(msie) ([\w.]+)/,
	rmozilla = /(mozilla)(?:.*? rv:([\w.]+))?/,

	// Matches dashed string for camelizing
	rdashAlpha = /-([a-z]|[0-9])/ig,
	rmsPrefix = /^-ms-/,

	// Used by jQuery.camelCase as callback to replace()
	fcamelCase = function( all, letter ) {
		return ( letter + "" ).toUpperCase();
	},

	// Keep a UserAgent string for use with jQuery.browser
	userAgent = navigator.userAgent,

	// For matching the engine and version of the browser
	browserMatch,

	// The deferred used on DOM ready
	readyList,

	// The ready event handler
	DOMContentLoaded,

	// Save a reference to some core methods
	toString = Object.prototype.toString,
	hasOwn = Object.prototype.hasOwnProperty,
	push = Array.prototype.push,
	slice = Array.prototype.slice,
	trim = String.prototype.trim,
	indexOf = Array.prototype.indexOf,

	// [[Class]] -> type pairs
	class2type = {};

jQuery.fn = jQuery.prototype = {
	constructor: jQuery,
	init: function( selector, context, rootjQuery ) {
		var match, elem, ret, doc;

		// Handle $(""), $(null), or $(undefined)
		if ( !selector ) {
			return this;
		}

		// Handle $(DOMElement)
		if ( selector.nodeType ) {
			this.context = this[0] = selector;
			this.length = 1;
			return this;
		}

		// The body element only exists once, optimize finding it
		if ( selector === "body" && !context && document.body ) {
			this.context = document;
			this[0] = document.body;
			this.selector = selector;
			this.length = 1;
			return this;
		}

		// Handle HTML strings
		if ( typeof selector === "string" ) {
			// Are we dealing with HTML string or an ID?
			if ( selector.charAt(0) === "<" && selector.charAt( selector.length - 1 ) === ">" && selector.length >= 3 ) {
				// Assume that strings that start and end with <> are HTML and skip the regex check
				match = [ null, selector, null ];

			} else {
				match = quickExpr.exec( selector );
			}

			// Verify a match, and that no context was specified for #id
			if ( match && (match[1] || !context) ) {

				// HANDLE: $(html) -> $(array)
				if ( match[1] ) {
					context = context instanceof jQuery ? context[0] : context;
					doc = ( context ? context.ownerDocument || context : document );

					// If a single string is passed in and it's a single tag
					// just do a createElement and skip the rest
					ret = rsingleTag.exec( selector );

					if ( ret ) {
						if ( jQuery.isPlainObject( context ) ) {
							selector = [ document.createElement( ret[1] ) ];
							jQuery.fn.attr.call( selector, context, true );

						} else {
							selector = [ doc.createElement( ret[1] ) ];
						}

					} else {
						ret = jQuery.buildFragment( [ match[1] ], [ doc ] );
						selector = ( ret.cacheable ? jQuery.clone(ret.fragment) : ret.fragment ).childNodes;
					}

					return jQuery.merge( this, selector );

				// HANDLE: $("#id")
				} else {
					elem = document.getElementById( match[2] );

					// Check parentNode to catch when Blackberry 4.6 returns
					// nodes that are no longer in the document #6963
					if ( elem && elem.parentNode ) {
						// Handle the case where IE and Opera return items
						// by name instead of ID
						if ( elem.id !== match[2] ) {
							return rootjQuery.find( selector );
						}

						// Otherwise, we inject the element directly into the jQuery object
						this.length = 1;
						this[0] = elem;
					}

					this.context = document;
					this.selector = selector;
					return this;
				}

			// HANDLE: $(expr, $(...))
			} else if ( !context || context.jquery ) {
				return ( context || rootjQuery ).find( selector );

			// HANDLE: $(expr, context)
			// (which is just equivalent to: $(context).find(expr)
			} else {
				return this.constructor( context ).find( selector );
			}

		// HANDLE: $(function)
		// Shortcut for document ready
		} else if ( jQuery.isFunction( selector ) ) {
			return rootjQuery.ready( selector );
		}

		if ( selector.selector !== undefined ) {
			this.selector = selector.selector;
			this.context = selector.context;
		}

		return jQuery.makeArray( selector, this );
	},

	// Start with an empty selector
	selector: "",

	// The current version of jQuery being used
	jquery: "1.7.2",

	// The default length of a jQuery object is 0
	length: 0,

	// The number of elements contained in the matched element set
	size: function() {
		return this.length;
	},

	toArray: function() {
		return slice.call( this, 0 );
	},

	// Get the Nth element in the matched element set OR
	// Get the whole matched element set as a clean array
	get: function( num ) {
		return num == null ?

			// Return a 'clean' array
			this.toArray() :

			// Return just the object
			( num < 0 ? this[ this.length + num ] : this[ num ] );
	},

	// Take an array of elements and push it onto the stack
	// (returning the new matched element set)
	pushStack: function( elems, name, selector ) {
		// Build a new jQuery matched element set
		var ret = this.constructor();

		if ( jQuery.isArray( elems ) ) {
			push.apply( ret, elems );

		} else {
			jQuery.merge( ret, elems );
		}

		// Add the old object onto the stack (as a reference)
		ret.prevObject = this;

		ret.context = this.context;

		if ( name === "find" ) {
			ret.selector = this.selector + ( this.selector ? " " : "" ) + selector;
		} else if ( name ) {
			ret.selector = this.selector + "." + name + "(" + selector + ")";
		}

		// Return the newly-formed element set
		return ret;
	},

	// Execute a callback for every element in the matched set.
	// (You can seed the arguments with an array of args, but this is
	// only used internally.)
	each: function( callback, args ) {
		return jQuery.each( this, callback, args );
	},

	ready: function( fn ) {
		// Attach the listeners
		jQuery.bindReady();

		// Add the callback
		readyList.add( fn );

		return this;
	},

	eq: function( i ) {
		i = +i;
		return i === -1 ?
			this.slice( i ) :
			this.slice( i, i + 1 );
	},

	first: function() {
		return this.eq( 0 );
	},

	last: function() {
		return this.eq( -1 );
	},

	slice: function() {
		return this.pushStack( slice.apply( this, arguments ),
			"slice", slice.call(arguments).join(",") );
	},

	map: function( callback ) {
		return this.pushStack( jQuery.map(this, function( elem, i ) {
			return callback.call( elem, i, elem );
		}));
	},

	end: function() {
		return this.prevObject || this.constructor(null);
	},

	// For internal use only.
	// Behaves like an Array's method, not like a jQuery method.
	push: push,
	sort: [].sort,
	splice: [].splice
};

// Give the init function the jQuery prototype for later instantiation
jQuery.fn.init.prototype = jQuery.fn;

jQuery.extend = jQuery.fn.extend = function() {
	var options, name, src, copy, copyIsArray, clone,
		target = arguments[0] || {},
		i = 1,
		length = arguments.length,
		deep = false;

	// Handle a deep copy situation
	if ( typeof target === "boolean" ) {
		deep = target;
		target = arguments[1] || {};
		// skip the boolean and the target
		i = 2;
	}

	// Handle case when target is a string or something (possible in deep copy)
	if ( typeof target !== "object" && !jQuery.isFunction(target) ) {
		target = {};
	}

	// extend jQuery itself if only one argument is passed
	if ( length === i ) {
		target = this;
		--i;
	}

	for ( ; i < length; i++ ) {
		// Only deal with non-null/undefined values
		if ( (options = arguments[ i ]) != null ) {
			// Extend the base object
			for ( name in options ) {
				src = target[ name ];
				copy = options[ name ];

				// Prevent never-ending loop
				if ( target === copy ) {
					continue;
				}

				// Recurse if we're merging plain objects or arrays
				if ( deep && copy && ( jQuery.isPlainObject(copy) || (copyIsArray = jQuery.isArray(copy)) ) ) {
					if ( copyIsArray ) {
						copyIsArray = false;
						clone = src && jQuery.isArray(src) ? src : [];

					} else {
						clone = src && jQuery.isPlainObject(src) ? src : {};
					}

					// Never move original objects, clone them
					target[ name ] = jQuery.extend( deep, clone, copy );

				// Don't bring in undefined values
				} else if ( copy !== undefined ) {
					target[ name ] = copy;
				}
			}
		}
	}

	// Return the modified object
	return target;
};

jQuery.extend({
	noConflict: function( deep ) {
		if ( window.$ === jQuery ) {
			window.$ = _$;
		}

		if ( deep && window.jQuery === jQuery ) {
			window.jQuery = _jQuery;
		}

		return jQuery;
	},

	// Is the DOM ready to be used? Set to true once it occurs.
	isReady: false,

	// A counter to track how many items to wait for before
	// the ready event fires. See #6781
	readyWait: 1,

	// Hold (or release) the ready event
	holdReady: function( hold ) {
		if ( hold ) {
			jQuery.readyWait++;
		} else {
			jQuery.ready( true );
		}
	},

	// Handle when the DOM is ready
	ready: function( wait ) {
		// Either a released hold or an DOMready/load event and not yet ready
		if ( (wait === true && !--jQuery.readyWait) || (wait !== true && !jQuery.isReady) ) {
			// Make sure body exists, at least, in case IE gets a little overzealous (ticket #5443).
			if ( !document.body ) {
				return setTimeout( jQuery.ready, 1 );
			}

			// Remember that the DOM is ready
			jQuery.isReady = true;

			// If a normal DOM Ready event fired, decrement, and wait if need be
			if ( wait !== true && --jQuery.readyWait > 0 ) {
				return;
			}

			// If there are functions bound, to execute
			readyList.fireWith( document, [ jQuery ] );

			// Trigger any bound ready events
			if ( jQuery.fn.trigger ) {
				jQuery( document ).trigger( "ready" ).off( "ready" );
			}
		}
	},

	bindReady: function() {
		if ( readyList ) {
			return;
		}

		readyList = jQuery.Callbacks( "once memory" );

		// Catch cases where $(document).ready() is called after the
		// browser event has already occurred.
		if ( document.readyState === "complete" ) {
			// Handle it asynchronously to allow scripts the opportunity to delay ready
			return setTimeout( jQuery.ready, 1 );
		}

		// Mozilla, Opera and webkit nightlies currently support this event
		if ( document.addEventListener ) {
			// Use the handy event callback
			document.addEventListener( "DOMContentLoaded", DOMContentLoaded, false );

			// A fallback to window.onload, that will always work
			window.addEventListener( "load", jQuery.ready, false );

		// If IE event model is used
		} else if ( document.attachEvent ) {
			// ensure firing before onload,
			// maybe late but safe also for iframes
			document.attachEvent( "onreadystatechange", DOMContentLoaded );

			// A fallback to window.onload, that will always work
			window.attachEvent( "onload", jQuery.ready );

			// If IE and not a frame
			// continually check to see if the document is ready
			var toplevel = false;

			try {
				toplevel = window.frameElement == null;
			} catch(e) {}

			if ( document.documentElement.doScroll && toplevel ) {
				doScrollCheck();
			}
		}
	},

	// See test/unit/core.js for details concerning isFunction.
	// Since version 1.3, DOM methods and functions like alert
	// aren't supported. They return false on IE (#2968).
	isFunction: function( obj ) {
		return jQuery.type(obj) === "function";
	},

	isArray: Array.isArray || function( obj ) {
		return jQuery.type(obj) === "array";
	},

	isWindow: function( obj ) {
		return obj != null && obj == obj.window;
	},

	isNumeric: function( obj ) {
		return !isNaN( parseFloat(obj) ) && isFinite( obj );
	},

	type: function( obj ) {
		return obj == null ?
			String( obj ) :
			class2type[ toString.call(obj) ] || "object";
	},

	isPlainObject: function( obj ) {
		// Must be an Object.
		// Because of IE, we also have to check the presence of the constructor property.
		// Make sure that DOM nodes and window objects don't pass through, as well
		if ( !obj || jQuery.type(obj) !== "object" || obj.nodeType || jQuery.isWindow( obj ) ) {
			return false;
		}

		try {
			// Not own constructor property must be Object
			if ( obj.constructor &&
				!hasOwn.call(obj, "constructor") &&
				!hasOwn.call(obj.constructor.prototype, "isPrototypeOf") ) {
				return false;
			}
		} catch ( e ) {
			// IE8,9 Will throw exceptions on certain host objects #9897
			return false;
		}

		// Own properties are enumerated firstly, so to speed up,
		// if last one is own, then all properties are own.

		var key;
		for ( key in obj ) {}

		return key === undefined || hasOwn.call( obj, key );
	},

	isEmptyObject: function( obj ) {
		for ( var name in obj ) {
			return false;
		}
		return true;
	},

	error: function( msg ) {
		throw new Error( msg );
	},

	parseJSON: function( data ) {
		if ( typeof data !== "string" || !data ) {
			return null;
		}

		// Make sure leading/trailing whitespace is removed (IE can't handle it)
		data = jQuery.trim( data );

		// Attempt to parse using the native JSON parser first
		if ( window.JSON && window.JSON.parse ) {
			return window.JSON.parse( data );
		}

		// Make sure the incoming data is actual JSON
		// Logic borrowed from http://json.org/json2.js
		if ( rvalidchars.test( data.replace( rvalidescape, "@" )
			.replace( rvalidtokens, "]" )
			.replace( rvalidbraces, "")) ) {

			return ( new Function( "return " + data ) )();

		}
		jQuery.error( "Invalid JSON: " + data );
	},

	// Cross-browser xml parsing
	parseXML: function( data ) {
		if ( typeof data !== "string" || !data ) {
			return null;
		}
		var xml, tmp;
		try {
			if ( window.DOMParser ) { // Standard
				tmp = new DOMParser();
				xml = tmp.parseFromString( data , "text/xml" );
			} else { // IE
				xml = new ActiveXObject( "Microsoft.XMLDOM" );
				xml.async = "false";
				xml.loadXML( data );
			}
		} catch( e ) {
			xml = undefined;
		}
		if ( !xml || !xml.documentElement || xml.getElementsByTagName( "parsererror" ).length ) {
			jQuery.error( "Invalid XML: " + data );
		}
		return xml;
	},

	noop: function() {},

	// Evaluates a script in a global context
	// Workarounds based on findings by Jim Driscoll
	// http://weblogs.java.net/blog/driscoll/archive/2009/09/08/eval-javascript-global-context
	globalEval: function( data ) {
		if ( data && rnotwhite.test( data ) ) {
			// We use execScript on Internet Explorer
			// We use an anonymous function so that context is window
			// rather than jQuery in Firefox
			( window.execScript || function( data ) {
				window[ "eval" ].call( window, data );
			} )( data );
		}
	},

	// Convert dashed to camelCase; used by the css and data modules
	// Microsoft forgot to hump their vendor prefix (#9572)
	camelCase: function( string ) {
		return string.replace( rmsPrefix, "ms-" ).replace( rdashAlpha, fcamelCase );
	},

	nodeName: function( elem, name ) {
		return elem.nodeName && elem.nodeName.toUpperCase() === name.toUpperCase();
	},

	// args is for internal usage only
	each: function( object, callback, args ) {
		var name, i = 0,
			length = object.length,
			isObj = length === undefined || jQuery.isFunction( object );

		if ( args ) {
			if ( isObj ) {
				for ( name in object ) {
					if ( callback.apply( object[ name ], args ) === false ) {
						break;
					}
				}
			} else {
				for ( ; i < length; ) {
					if ( callback.apply( object[ i++ ], args ) === false ) {
						break;
					}
				}
			}

		// A special, fast, case for the most common use of each
		} else {
			if ( isObj ) {
				for ( name in object ) {
					if ( callback.call( object[ name ], name, object[ name ] ) === false ) {
						break;
					}
				}
			} else {
				for ( ; i < length; ) {
					if ( callback.call( object[ i ], i, object[ i++ ] ) === false ) {
						break;
					}
				}
			}
		}

		return object;
	},

	// Use native String.trim function wherever possible
	trim: trim ?
		function( text ) {
			return text == null ?
				"" :
				trim.call( text );
		} :

		// Otherwise use our own trimming functionality
		function( text ) {
			return text == null ?
				"" :
				text.toString().replace( trimLeft, "" ).replace( trimRight, "" );
		},

	// results is for internal usage only
	makeArray: function( array, results ) {
		var ret = results || [];

		if ( array != null ) {
			// The window, strings (and functions) also have 'length'
			// Tweaked logic slightly to handle Blackberry 4.7 RegExp issues #6930
			var type = jQuery.type( array );

			if ( array.length == null || type === "string" || type === "function" || type === "regexp" || jQuery.isWindow( array ) ) {
				push.call( ret, array );
			} else {
				jQuery.merge( ret, array );
			}
		}

		return ret;
	},

	inArray: function( elem, array, i ) {
		var len;

		if ( array ) {
			if ( indexOf ) {
				return indexOf.call( array, elem, i );
			}

			len = array.length;
			i = i ? i < 0 ? Math.max( 0, len + i ) : i : 0;

			for ( ; i < len; i++ ) {
				// Skip accessing in sparse arrays
				if ( i in array && array[ i ] === elem ) {
					return i;
				}
			}
		}

		return -1;
	},

	merge: function( first, second ) {
		var i = first.length,
			j = 0;

		if ( typeof second.length === "number" ) {
			for ( var l = second.length; j < l; j++ ) {
				first[ i++ ] = second[ j ];
			}

		} else {
			while ( second[j] !== undefined ) {
				first[ i++ ] = second[ j++ ];
			}
		}

		first.length = i;

		return first;
	},

	grep: function( elems, callback, inv ) {
		var ret = [], retVal;
		inv = !!inv;

		// Go through the array, only saving the items
		// that pass the validator function
		for ( var i = 0, length = elems.length; i < length; i++ ) {
			retVal = !!callback( elems[ i ], i );
			if ( inv !== retVal ) {
				ret.push( elems[ i ] );
			}
		}

		return ret;
	},

	// arg is for internal usage only
	map: function( elems, callback, arg ) {
		var value, key, ret = [],
			i = 0,
			length = elems.length,
			// jquery objects are treated as arrays
			isArray = elems instanceof jQuery || length !== undefined && typeof length === "number" && ( ( length > 0 && elems[ 0 ] && elems[ length -1 ] ) || length === 0 || jQuery.isArray( elems ) ) ;

		// Go through the array, translating each of the items to their
		if ( isArray ) {
			for ( ; i < length; i++ ) {
				value = callback( elems[ i ], i, arg );

				if ( value != null ) {
					ret[ ret.length ] = value;
				}
			}

		// Go through every key on the object,
		} else {
			for ( key in elems ) {
				value = callback( elems[ key ], key, arg );

				if ( value != null ) {
					ret[ ret.length ] = value;
				}
			}
		}

		// Flatten any nested arrays
		return ret.concat.apply( [], ret );
	},

	// A global GUID counter for objects
	guid: 1,

	// Bind a function to a context, optionally partially applying any
	// arguments.
	proxy: function( fn, context ) {
		if ( typeof context === "string" ) {
			var tmp = fn[ context ];
			context = fn;
			fn = tmp;
		}

		// Quick check to determine if target is callable, in the spec
		// this throws a TypeError, but we will just return undefined.
		if ( !jQuery.isFunction( fn ) ) {
			return undefined;
		}

		// Simulated bind
		var args = slice.call( arguments, 2 ),
			proxy = function() {
				return fn.apply( context, args.concat( slice.call( arguments ) ) );
			};

		// Set the guid of unique handler to the same of original handler, so it can be removed
		proxy.guid = fn.guid = fn.guid || proxy.guid || jQuery.guid++;

		return proxy;
	},

	// Mutifunctional method to get and set values to a collection
	// The value/s can optionally be executed if it's a function
	access: function( elems, fn, key, value, chainable, emptyGet, pass ) {
		var exec,
			bulk = key == null,
			i = 0,
			length = elems.length;

		// Sets many values
		if ( key && typeof key === "object" ) {
			for ( i in key ) {
				jQuery.access( elems, fn, i, key[i], 1, emptyGet, value );
			}
			chainable = 1;

		// Sets one value
		} else if ( value !== undefined ) {
			// Optionally, function values get executed if exec is true
			exec = pass === undefined && jQuery.isFunction( value );

			if ( bulk ) {
				// Bulk operations only iterate when executing function values
				if ( exec ) {
					exec = fn;
					fn = function( elem, key, value ) {
						return exec.call( jQuery( elem ), value );
					};

				// Otherwise they run against the entire set
				} else {
					fn.call( elems, value );
					fn = null;
				}
			}

			if ( fn ) {
				for (; i < length; i++ ) {
					fn( elems[i], key, exec ? value.call( elems[i], i, fn( elems[i], key ) ) : value, pass );
				}
			}

			chainable = 1;
		}

		return chainable ?
			elems :

			// Gets
			bulk ?
				fn.call( elems ) :
				length ? fn( elems[0], key ) : emptyGet;
	},

	now: function() {
		return ( new Date() ).getTime();
	},

	// Use of jQuery.browser is frowned upon.
	// More details: http://docs.jquery.com/Utilities/jQuery.browser
	uaMatch: function( ua ) {
		ua = ua.toLowerCase();

		var match = rwebkit.exec( ua ) ||
			ropera.exec( ua ) ||
			rmsie.exec( ua ) ||
			ua.indexOf("compatible") < 0 && rmozilla.exec( ua ) ||
			[];

		return { browser: match[1] || "", version: match[2] || "0" };
	},

	sub: function() {
		function jQuerySub( selector, context ) {
			return new jQuerySub.fn.init( selector, context );
		}
		jQuery.extend( true, jQuerySub, this );
		jQuerySub.superclass = this;
		jQuerySub.fn = jQuerySub.prototype = this();
		jQuerySub.fn.constructor = jQuerySub;
		jQuerySub.sub = this.sub;
		jQuerySub.fn.init = function init( selector, context ) {
			if ( context && context instanceof jQuery && !(context instanceof jQuerySub) ) {
				context = jQuerySub( context );
			}

			return jQuery.fn.init.call( this, selector, context, rootjQuerySub );
		};
		jQuerySub.fn.init.prototype = jQuerySub.fn;
		var rootjQuerySub = jQuerySub(document);
		return jQuerySub;
	},

	browser: {}
});

// Populate the class2type map
jQuery.each("Boolean Number String Function Array Date RegExp Object".split(" "), function(i, name) {
	class2type[ "[object " + name + "]" ] = name.toLowerCase();
});

browserMatch = jQuery.uaMatch( userAgent );
if ( browserMatch.browser ) {
	jQuery.browser[ browserMatch.browser ] = true;
	jQuery.browser.version = browserMatch.version;
}

// Deprecated, use jQuery.browser.webkit instead
if ( jQuery.browser.webkit ) {
	jQuery.browser.safari = true;
}

// IE doesn't match non-breaking spaces with \s
if ( rnotwhite.test( "\xA0" ) ) {
	trimLeft = /^[\s\xA0]+/;
	trimRight = /[\s\xA0]+$/;
}

// All jQuery objects should point back to these
rootjQuery = jQuery(document);

// Cleanup functions for the document ready method
if ( document.addEventListener ) {
	DOMContentLoaded = function() {
		document.removeEventListener( "DOMContentLoaded", DOMContentLoaded, false );
		jQuery.ready();
	};

} else if ( document.attachEvent ) {
	DOMContentLoaded = function() {
		// Make sure body exists, at least, in case IE gets a little overzealous (ticket #5443).
		if ( document.readyState === "complete" ) {
			document.detachEvent( "onreadystatechange", DOMContentLoaded );
			jQuery.ready();
		}
	};
}

// The DOM ready check for Internet Explorer
function doScrollCheck() {
	if ( jQuery.isReady ) {
		return;
	}

	try {
		// If IE is used, use the trick by Diego Perini
		// http://javascript.nwbox.com/IEContentLoaded/
		document.documentElement.doScroll("left");
	} catch(e) {
		setTimeout( doScrollCheck, 1 );
		return;
	}

	// and execute any waiting functions
	jQuery.ready();
}

return jQuery;

})();


// String to Object flags format cache
var flagsCache = {};

// Convert String-formatted flags into Object-formatted ones and store in cache
function createFlags( flags ) {
	var object = flagsCache[ flags ] = {},
		i, length;
	flags = flags.split( /\s+/ );
	for ( i = 0, length = flags.length; i < length; i++ ) {
		object[ flags[i] ] = true;
	}
	return object;
}

/*
 * Create a callback list using the following parameters:
 *
 *	flags:	an optional list of space-separated flags that will change how
 *			the callback list behaves
 *
 * By default a callback list will act like an event callback list and can be
 * "fired" multiple times.
 *
 * Possible flags:
 *
 *	once:			will ensure the callback list can only be fired once (like a Deferred)
 *
 *	memory:			will keep track of previous values and will call any callback added
 *					after the list has been fired right away with the latest "memorized"
 *					values (like a Deferred)
 *
 *	unique:			will ensure a callback can only be added once (no duplicate in the list)
 *
 *	stopOnFalse:	interrupt callings when a callback returns false
 *
 */
jQuery.Callbacks = function( flags ) {

	// Convert flags from String-formatted to Object-formatted
	// (we check in cache first)
	flags = flags ? ( flagsCache[ flags ] || createFlags( flags ) ) : {};

	var // Actual callback list
		list = [],
		// Stack of fire calls for repeatable lists
		stack = [],
		// Last fire value (for non-forgettable lists)
		memory,
		// Flag to know if list was already fired
		fired,
		// Flag to know if list is currently firing
		firing,
		// First callback to fire (used internally by add and fireWith)
		firingStart,
		// End of the loop when firing
		firingLength,
		// Index of currently firing callback (modified by remove if needed)
		firingIndex,
		// Add one or several callbacks to the list
		add = function( args ) {
			var i,
				length,
				elem,
				type,
				actual;
			for ( i = 0, length = args.length; i < length; i++ ) {
				elem = args[ i ];
				type = jQuery.type( elem );
				if ( type === "array" ) {
					// Inspect recursively
					add( elem );
				} else if ( type === "function" ) {
					// Add if not in unique mode and callback is not in
					if ( !flags.unique || !self.has( elem ) ) {
						list.push( elem );
					}
				}
			}
		},
		// Fire callbacks
		fire = function( context, args ) {
			args = args || [];
			memory = !flags.memory || [ context, args ];
			fired = true;
			firing = true;
			firingIndex = firingStart || 0;
			firingStart = 0;
			firingLength = list.length;
			for ( ; list && firingIndex < firingLength; firingIndex++ ) {
				if ( list[ firingIndex ].apply( context, args ) === false && flags.stopOnFalse ) {
					memory = true; // Mark as halted
					break;
				}
			}
			firing = false;
			if ( list ) {
				if ( !flags.once ) {
					if ( stack && stack.length ) {
						memory = stack.shift();
						self.fireWith( memory[ 0 ], memory[ 1 ] );
					}
				} else if ( memory === true ) {
					self.disable();
				} else {
					list = [];
				}
			}
		},
		// Actual Callbacks object
		self = {
			// Add a callback or a collection of callbacks to the list
			add: function() {
				if ( list ) {
					var length = list.length;
					add( arguments );
					// Do we need to add the callbacks to the
					// current firing batch?
					if ( firing ) {
						firingLength = list.length;
					// With memory, if we're not firing then
					// we should call right away, unless previous
					// firing was halted (stopOnFalse)
					} else if ( memory && memory !== true ) {
						firingStart = length;
						fire( memory[ 0 ], memory[ 1 ] );
					}
				}
				return this;
			},
			// Remove a callback from the list
			remove: function() {
				if ( list ) {
					var args = arguments,
						argIndex = 0,
						argLength = args.length;
					for ( ; argIndex < argLength ; argIndex++ ) {
						for ( var i = 0; i < list.length; i++ ) {
							if ( args[ argIndex ] === list[ i ] ) {
								// Handle firingIndex and firingLength
								if ( firing ) {
									if ( i <= firingLength ) {
										firingLength--;
										if ( i <= firingIndex ) {
											firingIndex--;
										}
									}
								}
								// Remove the element
								list.splice( i--, 1 );
								// If we have some unicity property then
								// we only need to do this once
								if ( flags.unique ) {
									break;
								}
							}
						}
					}
				}
				return this;
			},
			// Control if a given callback is in the list
			has: function( fn ) {
				if ( list ) {
					var i = 0,
						length = list.length;
					for ( ; i < length; i++ ) {
						if ( fn === list[ i ] ) {
							return true;
						}
					}
				}
				return false;
			},
			// Remove all callbacks from the list
			empty: function() {
				list = [];
				return this;
			},
			// Have the list do nothing anymore
			disable: function() {
				list = stack = memory = undefined;
				return this;
			},
			// Is it disabled?
			disabled: function() {
				return !list;
			},
			// Lock the list in its current state
			lock: function() {
				stack = undefined;
				if ( !memory || memory === true ) {
					self.disable();
				}
				return this;
			},
			// Is it locked?
			locked: function() {
				return !stack;
			},
			// Call all callbacks with the given context and arguments
			fireWith: function( context, args ) {
				if ( stack ) {
					if ( firing ) {
						if ( !flags.once ) {
							stack.push( [ context, args ] );
						}
					} else if ( !( flags.once && memory ) ) {
						fire( context, args );
					}
				}
				return this;
			},
			// Call all the callbacks with the given arguments
			fire: function() {
				self.fireWith( this, arguments );
				return this;
			},
			// To know if the callbacks have already been called at least once
			fired: function() {
				return !!fired;
			}
		};

	return self;
};




var // Static reference to slice
	sliceDeferred = [].slice;

jQuery.extend({

	Deferred: function( func ) {
		var doneList = jQuery.Callbacks( "once memory" ),
			failList = jQuery.Callbacks( "once memory" ),
			progressList = jQuery.Callbacks( "memory" ),
			state = "pending",
			lists = {
				resolve: doneList,
				reject: failList,
				notify: progressList
			},
			promise = {
				done: doneList.add,
				fail: failList.add,
				progress: progressList.add,

				state: function() {
					return state;
				},

				// Deprecated
				isResolved: doneList.fired,
				isRejected: failList.fired,

				then: function( doneCallbacks, failCallbacks, progressCallbacks ) {
					deferred.done( doneCallbacks ).fail( failCallbacks ).progress( progressCallbacks );
					return this;
				},
				always: function() {
					deferred.done.apply( deferred, arguments ).fail.apply( deferred, arguments );
					return this;
				},
				pipe: function( fnDone, fnFail, fnProgress ) {
					return jQuery.Deferred(function( newDefer ) {
						jQuery.each( {
							done: [ fnDone, "resolve" ],
							fail: [ fnFail, "reject" ],
							progress: [ fnProgress, "notify" ]
						}, function( handler, data ) {
							var fn = data[ 0 ],
								action = data[ 1 ],
								returned;
							if ( jQuery.isFunction( fn ) ) {
								deferred[ handler ](function() {
									returned = fn.apply( this, arguments );
									if ( returned && jQuery.isFunction( returned.promise ) ) {
										returned.promise().then( newDefer.resolve, newDefer.reject, newDefer.notify );
									} else {
										newDefer[ action + "With" ]( this === deferred ? newDefer : this, [ returned ] );
									}
								});
							} else {
								deferred[ handler ]( newDefer[ action ] );
							}
						});
					}).promise();
				},
				// Get a promise for this deferred
				// If obj is provided, the promise aspect is added to the object
				promise: function( obj ) {
					if ( obj == null ) {
						obj = promise;
					} else {
						for ( var key in promise ) {
							obj[ key ] = promise[ key ];
						}
					}
					return obj;
				}
			},
			deferred = promise.promise({}),
			key;

		for ( key in lists ) {
			deferred[ key ] = lists[ key ].fire;
			deferred[ key + "With" ] = lists[ key ].fireWith;
		}

		// Handle state
		deferred.done( function() {
			state = "resolved";
		}, failList.disable, progressList.lock ).fail( function() {
			state = "rejected";
		}, doneList.disable, progressList.lock );

		// Call given func if any
		if ( func ) {
			func.call( deferred, deferred );
		}

		// All done!
		return deferred;
	},

	// Deferred helper
	when: function( firstParam ) {
		var args = sliceDeferred.call( arguments, 0 ),
			i = 0,
			length = args.length,
			pValues = new Array( length ),
			count = length,
			pCount = length,
			deferred = length <= 1 && firstParam && jQuery.isFunction( firstParam.promise ) ?
				firstParam :
				jQuery.Deferred(),
			promise = deferred.promise();
		function resolveFunc( i ) {
			return function( value ) {
				args[ i ] = arguments.length > 1 ? sliceDeferred.call( arguments, 0 ) : value;
				if ( !( --count ) ) {
					deferred.resolveWith( deferred, args );
				}
			};
		}
		function progressFunc( i ) {
			return function( value ) {
				pValues[ i ] = arguments.length > 1 ? sliceDeferred.call( arguments, 0 ) : value;
				deferred.notifyWith( promise, pValues );
			};
		}
		if ( length > 1 ) {
			for ( ; i < length; i++ ) {
				if ( args[ i ] && args[ i ].promise && jQuery.isFunction( args[ i ].promise ) ) {
					args[ i ].promise().then( resolveFunc(i), deferred.reject, progressFunc(i) );
				} else {
					--count;
				}
			}
			if ( !count ) {
				deferred.resolveWith( deferred, args );
			}
		} else if ( deferred !== firstParam ) {
			deferred.resolveWith( deferred, length ? [ firstParam ] : [] );
		}
		return promise;
	}
});




jQuery.support = (function() {

	var support,
		all,
		a,
		select,
		opt,
		input,
		fragment,
		tds,
		events,
		eventName,
		i,
		isSupported,
		div = document.createElement( "div" ),
		documentElement = document.documentElement;

	// Preliminary tests
	div.setAttribute("className", "t");
	div.innerHTML = "   <link/><table></table><a href='/a' style='top:1px;float:left;opacity:.55;'>a</a><input type='checkbox'/>";

	all = div.getElementsByTagName( "*" );
	a = div.getElementsByTagName( "a" )[ 0 ];

	// Can't get basic test support
	if ( !all || !all.length || !a ) {
		return {};
	}

	// First batch of supports tests
	select = document.createElement( "select" );
	opt = select.appendChild( document.createElement("option") );
	input = div.getElementsByTagName( "input" )[ 0 ];

	support = {
		// IE strips leading whitespace when .innerHTML is used
		leadingWhitespace: ( div.firstChild.nodeType === 3 ),

		// Make sure that tbody elements aren't automatically inserted
		// IE will insert them into empty tables
		tbody: !div.getElementsByTagName("tbody").length,

		// Make sure that link elements get serialized correctly by innerHTML
		// This requires a wrapper element in IE
		htmlSerialize: !!div.getElementsByTagName("link").length,

		// Get the style information from getAttribute
		// (IE uses .cssText instead)
		style: /top/.test( a.getAttribute("style") ),

		// Make sure that URLs aren't manipulated
		// (IE normalizes it by default)
		hrefNormalized: ( a.getAttribute("href") === "/a" ),

		// Make sure that element opacity exists
		// (IE uses filter instead)
		// Use a regex to work around a WebKit issue. See #5145
		opacity: /^0.55/.test( a.style.opacity ),

		// Verify style float existence
		// (IE uses styleFloat instead of cssFloat)
		cssFloat: !!a.style.cssFloat,

		// Make sure that if no value is specified for a checkbox
		// that it defaults to "on".
		// (WebKit defaults to "" instead)
		checkOn: ( input.value === "on" ),

		// Make sure that a selected-by-default option has a working selected property.
		// (WebKit defaults to false instead of true, IE too, if it's in an optgroup)
		optSelected: opt.selected,

		// Test setAttribute on camelCase class. If it works, we need attrFixes when doing get/setAttribute (ie6/7)
		getSetAttribute: div.className !== "t",

		// Tests for enctype support on a form(#6743)
		enctype: !!document.createElement("form").enctype,

		// Makes sure cloning an html5 element does not cause problems
		// Where outerHTML is undefined, this still works
		html5Clone: document.createElement("nav").cloneNode( true ).outerHTML !== "<:nav></:nav>",

		// Will be defined later
		submitBubbles: true,
		changeBubbles: true,
		focusinBubbles: false,
		deleteExpando: true,
		noCloneEvent: true,
		inlineBlockNeedsLayout: false,
		shrinkWrapBlocks: false,
		reliableMarginRight: true,
		pixelMargin: true
	};

	// jQuery.boxModel DEPRECATED in 1.3, use jQuery.support.boxModel instead
	jQuery.boxModel = support.boxModel = (document.compatMode === "CSS1Compat");

	// Make sure checked status is properly cloned
	input.checked = true;
	support.noCloneChecked = input.cloneNode( true ).checked;

	// Make sure that the options inside disabled selects aren't marked as disabled
	// (WebKit marks them as disabled)
	select.disabled = true;
	support.optDisabled = !opt.disabled;

	// Test to see if it's possible to delete an expando from an element
	// Fails in Internet Explorer
	try {
		delete div.test;
	} catch( e ) {
		support.deleteExpando = false;
	}

	if ( !div.addEventListener && div.attachEvent && div.fireEvent ) {
		div.attachEvent( "onclick", function() {
			// Cloning a node shouldn't copy over any
			// bound event handlers (IE does this)
			support.noCloneEvent = false;
		});
		div.cloneNode( true ).fireEvent( "onclick" );
		// Clear event handler that was attached just now.
		// Fix for IE8 memory leak across frame reloads when jQuery is
		// loaded inside a frame. Memory leak wouldn't go away until
		// window.top was unloaded.
		if (div.clearAttributes) {
			div.clearAttributes();
		}
	}

	// Check if a radio maintains its value
	// after being appended to the DOM
	input = document.createElement("input");
	input.value = "t";
	input.setAttribute("type", "radio");
	support.radioValue = input.value === "t";

	input.setAttribute("checked", "checked");

	// #11217 - WebKit loses check when the name is after the checked attribute
	input.setAttribute( "name", "t" );

	div.appendChild( input );
	fragment = document.createDocumentFragment();
	fragment.appendChild( div.lastChild );

	// WebKit doesn't clone checked state correctly in fragments
	support.checkClone = fragment.cloneNode( true ).cloneNode( true ).lastChild.checked;

	// Check if a disconnected checkbox will retain its checked
	// value of true after appended to the DOM (IE6/7)
	support.appendChecked = input.checked;

	fragment.removeChild( input );
	fragment.appendChild( div );

	// Technique from Juriy Zaytsev
	// http://perfectionkills.com/detecting-event-support-without-browser-sniffing/
	// We only care about the case where non-standard event systems
	// are used, namely in IE. Short-circuiting here helps us to
	// avoid an eval call (in setAttribute) which can cause CSP
	// to go haywire. See: https://developer.mozilla.org/en/Security/CSP
	if ( div.attachEvent ) {
		for ( i in {
			submit: 1,
			change: 1,
			focusin: 1
		}) {
			eventName = "on" + i;
			isSupported = ( eventName in div );
			if ( !isSupported ) {
				div.setAttribute( eventName, "return;" );
				isSupported = ( typeof div[ eventName ] === "function" );
			}
			support[ i + "Bubbles" ] = isSupported;
		}
	}

	fragment.removeChild( div );

	// Null elements to avoid leaks in IE
	fragment = select = opt = div = input = null;

	// Run tests that need a body at doc ready
	jQuery(function() {
		var container, outer, inner, table, td, offsetSupport,
			marginDiv, conMarginTop, style, html, positionTopLeftWidthHeight,
			paddingMarginBorderVisibility, paddingMarginBorder,
			body = document.getElementsByTagName("body")[0];

		if ( !body ) {
			// Return for frameset docs that don't have a body
			return;
		}

		conMarginTop = 1;
		paddingMarginBorder = "padding:0;margin:0;border:";
		positionTopLeftWidthHeight = "position:absolute;top:0;left:0;width:1px;height:1px;";
		paddingMarginBorderVisibility = paddingMarginBorder + "0;visibility:hidden;";
		style = "style='" + positionTopLeftWidthHeight + paddingMarginBorder + "5px solid #000;";
		html = "<div " + style + "display:block;'><div style='" + paddingMarginBorder + "0;display:block;overflow:hidden;'></div></div>" +
			"<table " + style + "' cellpadding='0' cellspacing='0'>" +
			"<tr><td></td></tr></table>";

		container = document.createElement("div");
		container.style.cssText = paddingMarginBorderVisibility + "width:0;height:0;position:static;top:0;margin-top:" + conMarginTop + "px";
		body.insertBefore( container, body.firstChild );

		// Construct the test element
		div = document.createElement("div");
		container.appendChild( div );

		// Check if table cells still have offsetWidth/Height when they are set
		// to display:none and there are still other visible table cells in a
		// table row; if so, offsetWidth/Height are not reliable for use when
		// determining if an element has been hidden directly using
		// display:none (it is still safe to use offsets if a parent element is
		// hidden; don safety goggles and see bug #4512 for more information).
		// (only IE 8 fails this test)
		div.innerHTML = "<table><tr><td style='" + paddingMarginBorder + "0;display:none'></td><td>t</td></tr></table>";
		tds = div.getElementsByTagName( "td" );
		isSupported = ( tds[ 0 ].offsetHeight === 0 );

		tds[ 0 ].style.display = "";
		tds[ 1 ].style.display = "none";

		// Check if empty table cells still have offsetWidth/Height
		// (IE <= 8 fail this test)
		support.reliableHiddenOffsets = isSupported && ( tds[ 0 ].offsetHeight === 0 );

		// Check if div with explicit width and no margin-right incorrectly
		// gets computed margin-right based on width of container. For more
		// info see bug #3333
		// Fails in WebKit before Feb 2011 nightlies
		// WebKit Bug 13343 - getComputedStyle returns wrong value for margin-right
		if ( window.getComputedStyle ) {
			div.innerHTML = "";
			marginDiv = document.createElement( "div" );
			marginDiv.style.width = "0";
			marginDiv.style.marginRight = "0";
			div.style.width = "2px";
			div.appendChild( marginDiv );
			support.reliableMarginRight =
				( parseInt( ( window.getComputedStyle( marginDiv, null ) || { marginRight: 0 } ).marginRight, 10 ) || 0 ) === 0;
		}

		if ( typeof div.style.zoom !== "undefined" ) {
			// Check if natively block-level elements act like inline-block
			// elements when setting their display to 'inline' and giving
			// them layout
			// (IE < 8 does this)
			div.innerHTML = "";
			div.style.width = div.style.padding = "1px";
			div.style.border = 0;
			div.style.overflow = "hidden";
			div.style.display = "inline";
			div.style.zoom = 1;
			support.inlineBlockNeedsLayout = ( div.offsetWidth === 3 );

			// Check if elements with layout shrink-wrap their children
			// (IE 6 does this)
			div.style.display = "block";
			div.style.overflow = "visible";
			div.innerHTML = "<div style='width:5px;'></div>";
			support.shrinkWrapBlocks = ( div.offsetWidth !== 3 );
		}

		div.style.cssText = positionTopLeftWidthHeight + paddingMarginBorderVisibility;
		div.innerHTML = html;

		outer = div.firstChild;
		inner = outer.firstChild;
		td = outer.nextSibling.firstChild.firstChild;

		offsetSupport = {
			doesNotAddBorder: ( inner.offsetTop !== 5 ),
			doesAddBorderForTableAndCells: ( td.offsetTop === 5 )
		};

		inner.style.position = "fixed";
		inner.style.top = "20px";

		// safari subtracts parent border width here which is 5px
		offsetSupport.fixedPosition = ( inner.offsetTop === 20 || inner.offsetTop === 15 );
		inner.style.position = inner.style.top = "";

		outer.style.overflow = "hidden";
		outer.style.position = "relative";

		offsetSupport.subtractsBorderForOverflowNotVisible = ( inner.offsetTop === -5 );
		offsetSupport.doesNotIncludeMarginInBodyOffset = ( body.offsetTop !== conMarginTop );

		if ( window.getComputedStyle ) {
			div.style.marginTop = "1%";
			support.pixelMargin = ( window.getComputedStyle( div, null ) || { marginTop: 0 } ).marginTop !== "1%";
		}

		if ( typeof container.style.zoom !== "undefined" ) {
			container.style.zoom = 1;
		}

		body.removeChild( container );
		marginDiv = div = container = null;

		jQuery.extend( support, offsetSupport );
	});

	return support;
})();




var rbrace = /^(?:\{.*\}|\[.*\])$/,
	rmultiDash = /([A-Z])/g;

jQuery.extend({
	cache: {},

	// Please use with caution
	uuid: 0,

	// Unique for each copy of jQuery on the page
	// Non-digits removed to match rinlinejQuery
	expando: "jQuery" + ( jQuery.fn.jquery + Math.random() ).replace( /\D/g, "" ),

	// The following elements throw uncatchable exceptions if you
	// attempt to add expando properties to them.
	noData: {
		"embed": true,
		// Ban all objects except for Flash (which handle expandos)
		"object": "clsid:D27CDB6E-AE6D-11cf-96B8-444553540000",
		"applet": true
	},

	hasData: function( elem ) {
		elem = elem.nodeType ? jQuery.cache[ elem[jQuery.expando] ] : elem[ jQuery.expando ];
		return !!elem && !isEmptyDataObject( elem );
	},

	data: function( elem, name, data, pvt /* Internal Use Only */ ) {
		if ( !jQuery.acceptData( elem ) ) {
			return;
		}

		var privateCache, thisCache, ret,
			internalKey = jQuery.expando,
			getByName = typeof name === "string",

			// We have to handle DOM nodes and JS objects differently because IE6-7
			// can't GC object references properly across the DOM-JS boundary
			isNode = elem.nodeType,

			// Only DOM nodes need the global jQuery cache; JS object data is
			// attached directly to the object so GC can occur automatically
			cache = isNode ? jQuery.cache : elem,

			// Only defining an ID for JS objects if its cache already exists allows
			// the code to shortcut on the same path as a DOM node with no cache
			id = isNode ? elem[ internalKey ] : elem[ internalKey ] && internalKey,
			isEvents = name === "events";

		// Avoid doing any more work than we need to when trying to get data on an
		// object that has no data at all
		if ( (!id || !cache[id] || (!isEvents && !pvt && !cache[id].data)) && getByName && data === undefined ) {
			return;
		}

		if ( !id ) {
			// Only DOM nodes need a new unique ID for each element since their data
			// ends up in the global cache
			if ( isNode ) {
				elem[ internalKey ] = id = ++jQuery.uuid;
			} else {
				id = internalKey;
			}
		}

		if ( !cache[ id ] ) {
			cache[ id ] = {};

			// Avoids exposing jQuery metadata on plain JS objects when the object
			// is serialized using JSON.stringify
			if ( !isNode ) {
				cache[ id ].toJSON = jQuery.noop;
			}
		}

		// An object can be passed to jQuery.data instead of a key/value pair; this gets
		// shallow copied over onto the existing cache
		if ( typeof name === "object" || typeof name === "function" ) {
			if ( pvt ) {
				cache[ id ] = jQuery.extend( cache[ id ], name );
			} else {
				cache[ id ].data = jQuery.extend( cache[ id ].data, name );
			}
		}

		privateCache = thisCache = cache[ id ];

		// jQuery data() is stored in a separate object inside the object's internal data
		// cache in order to avoid key collisions between internal data and user-defined
		// data.
		if ( !pvt ) {
			if ( !thisCache.data ) {
				thisCache.data = {};
			}

			thisCache = thisCache.data;
		}

		if ( data !== undefined ) {
			thisCache[ jQuery.camelCase( name ) ] = data;
		}

		// Users should not attempt to inspect the internal events object using jQuery.data,
		// it is undocumented and subject to change. But does anyone listen? No.
		if ( isEvents && !thisCache[ name ] ) {
			return privateCache.events;
		}

		// Check for both converted-to-camel and non-converted data property names
		// If a data property was specified
		if ( getByName ) {

			// First Try to find as-is property data
			ret = thisCache[ name ];

			// Test for null|undefined property data
			if ( ret == null ) {

				// Try to find the camelCased property
				ret = thisCache[ jQuery.camelCase( name ) ];
			}
		} else {
			ret = thisCache;
		}

		return ret;
	},

	removeData: function( elem, name, pvt /* Internal Use Only */ ) {
		if ( !jQuery.acceptData( elem ) ) {
			return;
		}

		var thisCache, i, l,

			// Reference to internal data cache key
			internalKey = jQuery.expando,

			isNode = elem.nodeType,

			// See jQuery.data for more information
			cache = isNode ? jQuery.cache : elem,

			// See jQuery.data for more information
			id = isNode ? elem[ internalKey ] : internalKey;

		// If there is already no cache entry for this object, there is no
		// purpose in continuing
		if ( !cache[ id ] ) {
			return;
		}

		if ( name ) {

			thisCache = pvt ? cache[ id ] : cache[ id ].data;

			if ( thisCache ) {

				// Support array or space separated string names for data keys
				if ( !jQuery.isArray( name ) ) {

					// try the string as a key before any manipulation
					if ( name in thisCache ) {
						name = [ name ];
					} else {

						// split the camel cased version by spaces unless a key with the spaces exists
						name = jQuery.camelCase( name );
						if ( name in thisCache ) {
							name = [ name ];
						} else {
							name = name.split( " " );
						}
					}
				}

				for ( i = 0, l = name.length; i < l; i++ ) {
					delete thisCache[ name[i] ];
				}

				// If there is no data left in the cache, we want to continue
				// and let the cache object itself get destroyed
				if ( !( pvt ? isEmptyDataObject : jQuery.isEmptyObject )( thisCache ) ) {
					return;
				}
			}
		}

		// See jQuery.data for more information
		if ( !pvt ) {
			delete cache[ id ].data;

			// Don't destroy the parent cache unless the internal data object
			// had been the only thing left in it
			if ( !isEmptyDataObject(cache[ id ]) ) {
				return;
			}
		}

		// Browsers that fail expando deletion also refuse to delete expandos on
		// the window, but it will allow it on all other JS objects; other browsers
		// don't care
		// Ensure that `cache` is not a window object #10080
		if ( jQuery.support.deleteExpando || !cache.setInterval ) {
			delete cache[ id ];
		} else {
			cache[ id ] = null;
		}

		// We destroyed the cache and need to eliminate the expando on the node to avoid
		// false lookups in the cache for entries that no longer exist
		if ( isNode ) {
			// IE does not allow us to delete expando properties from nodes,
			// nor does it have a removeAttribute function on Document nodes;
			// we must handle all of these cases
			if ( jQuery.support.deleteExpando ) {
				delete elem[ internalKey ];
			} else if ( elem.removeAttribute ) {
				elem.removeAttribute( internalKey );
			} else {
				elem[ internalKey ] = null;
			}
		}
	},

	// For internal use only.
	_data: function( elem, name, data ) {
		return jQuery.data( elem, name, data, true );
	},

	// A method for determining if a DOM node can handle the data expando
	acceptData: function( elem ) {
		if ( elem.nodeName ) {
			var match = jQuery.noData[ elem.nodeName.toLowerCase() ];

			if ( match ) {
				return !(match === true || elem.getAttribute("classid") !== match);
			}
		}

		return true;
	}
});

jQuery.fn.extend({
	data: function( key, value ) {
		var parts, part, attr, name, l,
			elem = this[0],
			i = 0,
			data = null;

		// Gets all values
		if ( key === undefined ) {
			if ( this.length ) {
				data = jQuery.data( elem );

				if ( elem.nodeType === 1 && !jQuery._data( elem, "parsedAttrs" ) ) {
					attr = elem.attributes;
					for ( l = attr.length; i < l; i++ ) {
						name = attr[i].name;

						if ( name.indexOf( "data-" ) === 0 ) {
							name = jQuery.camelCase( name.substring(5) );

							dataAttr( elem, name, data[ name ] );
						}
					}
					jQuery._data( elem, "parsedAttrs", true );
				}
			}

			return data;
		}

		// Sets multiple values
		if ( typeof key === "object" ) {
			return this.each(function() {
				jQuery.data( this, key );
			});
		}

		parts = key.split( ".", 2 );
		parts[1] = parts[1] ? "." + parts[1] : "";
		part = parts[1] + "!";

		return jQuery.access( this, function( value ) {

			if ( value === undefined ) {
				data = this.triggerHandler( "getData" + part, [ parts[0] ] );

				// Try to fetch any internally stored data first
				if ( data === undefined && elem ) {
					data = jQuery.data( elem, key );
					data = dataAttr( elem, key, data );
				}

				return data === undefined && parts[1] ?
					this.data( parts[0] ) :
					data;
			}

			parts[1] = value;
			this.each(function() {
				var self = jQuery( this );

				self.triggerHandler( "setData" + part, parts );
				jQuery.data( this, key, value );
				self.triggerHandler( "changeData" + part, parts );
			});
		}, null, value, arguments.length > 1, null, false );
	},

	removeData: function( key ) {
		return this.each(function() {
			jQuery.removeData( this, key );
		});
	}
});

function dataAttr( elem, key, data ) {
	// If nothing was found internally, try to fetch any
	// data from the HTML5 data-* attribute
	if ( data === undefined && elem.nodeType === 1 ) {

		var name = "data-" + key.replace( rmultiDash, "-$1" ).toLowerCase();

		data = elem.getAttribute( name );

		if ( typeof data === "string" ) {
			try {
				data = data === "true" ? true :
				data === "false" ? false :
				data === "null" ? null :
				jQuery.isNumeric( data ) ? +data :
					rbrace.test( data ) ? jQuery.parseJSON( data ) :
					data;
			} catch( e ) {}

			// Make sure we set the data so it isn't changed later
			jQuery.data( elem, key, data );

		} else {
			data = undefined;
		}
	}

	return data;
}

// checks a cache object for emptiness
function isEmptyDataObject( obj ) {
	for ( var name in obj ) {

		// if the public data object is empty, the private is still empty
		if ( name === "data" && jQuery.isEmptyObject( obj[name] ) ) {
			continue;
		}
		if ( name !== "toJSON" ) {
			return false;
		}
	}

	return true;
}




function handleQueueMarkDefer( elem, type, src ) {
	var deferDataKey = type + "defer",
		queueDataKey = type + "queue",
		markDataKey = type + "mark",
		defer = jQuery._data( elem, deferDataKey );
	if ( defer &&
		( src === "queue" || !jQuery._data(elem, queueDataKey) ) &&
		( src === "mark" || !jQuery._data(elem, markDataKey) ) ) {
		// Give room for hard-coded callbacks to fire first
		// and eventually mark/queue something else on the element
		setTimeout( function() {
			if ( !jQuery._data( elem, queueDataKey ) &&
				!jQuery._data( elem, markDataKey ) ) {
				jQuery.removeData( elem, deferDataKey, true );
				defer.fire();
			}
		}, 0 );
	}
}

jQuery.extend({

	_mark: function( elem, type ) {
		if ( elem ) {
			type = ( type || "fx" ) + "mark";
			jQuery._data( elem, type, (jQuery._data( elem, type ) || 0) + 1 );
		}
	},

	_unmark: function( force, elem, type ) {
		if ( force !== true ) {
			type = elem;
			elem = force;
			force = false;
		}
		if ( elem ) {
			type = type || "fx";
			var key = type + "mark",
				count = force ? 0 : ( (jQuery._data( elem, key ) || 1) - 1 );
			if ( count ) {
				jQuery._data( elem, key, count );
			} else {
				jQuery.removeData( elem, key, true );
				handleQueueMarkDefer( elem, type, "mark" );
			}
		}
	},

	queue: function( elem, type, data ) {
		var q;
		if ( elem ) {
			type = ( type || "fx" ) + "queue";
			q = jQuery._data( elem, type );

			// Speed up dequeue by getting out quickly if this is just a lookup
			if ( data ) {
				if ( !q || jQuery.isArray(data) ) {
					q = jQuery._data( elem, type, jQuery.makeArray(data) );
				} else {
					q.push( data );
				}
			}
			return q || [];
		}
	},

	dequeue: function( elem, type ) {
		type = type || "fx";

		var queue = jQuery.queue( elem, type ),
			fn = queue.shift(),
			hooks = {};

		// If the fx queue is dequeued, always remove the progress sentinel
		if ( fn === "inprogress" ) {
			fn = queue.shift();
		}

		if ( fn ) {
			// Add a progress sentinel to prevent the fx queue from being
			// automatically dequeued
			if ( type === "fx" ) {
				queue.unshift( "inprogress" );
			}

			jQuery._data( elem, type + ".run", hooks );
			fn.call( elem, function() {
				jQuery.dequeue( elem, type );
			}, hooks );
		}

		if ( !queue.length ) {
			jQuery.removeData( elem, type + "queue " + type + ".run", true );
			handleQueueMarkDefer( elem, type, "queue" );
		}
	}
});

jQuery.fn.extend({
	queue: function( type, data ) {
		var setter = 2;

		if ( typeof type !== "string" ) {
			data = type;
			type = "fx";
			setter--;
		}

		if ( arguments.length < setter ) {
			return jQuery.queue( this[0], type );
		}

		return data === undefined ?
			this :
			this.each(function() {
				var queue = jQuery.queue( this, type, data );

				if ( type === "fx" && queue[0] !== "inprogress" ) {
					jQuery.dequeue( this, type );
				}
			});
	},
	dequeue: function( type ) {
		return this.each(function() {
			jQuery.dequeue( this, type );
		});
	},
	// Based off of the plugin by Clint Helfers, with permission.
	// http://blindsignals.com/index.php/2009/07/jquery-delay/
	delay: function( time, type ) {
		time = jQuery.fx ? jQuery.fx.speeds[ time ] || time : time;
		type = type || "fx";

		return this.queue( type, function( next, hooks ) {
			var timeout = setTimeout( next, time );
			hooks.stop = function() {
				clearTimeout( timeout );
			};
		});
	},
	clearQueue: function( type ) {
		return this.queue( type || "fx", [] );
	},
	// Get a promise resolved when queues of a certain type
	// are emptied (fx is the type by default)
	promise: function( type, object ) {
		if ( typeof type !== "string" ) {
			object = type;
			type = undefined;
		}
		type = type || "fx";
		var defer = jQuery.Deferred(),
			elements = this,
			i = elements.length,
			count = 1,
			deferDataKey = type + "defer",
			queueDataKey = type + "queue",
			markDataKey = type + "mark",
			tmp;
		function resolve() {
			if ( !( --count ) ) {
				defer.resolveWith( elements, [ elements ] );
			}
		}
		while( i-- ) {
			if (( tmp = jQuery.data( elements[ i ], deferDataKey, undefined, true ) ||
					( jQuery.data( elements[ i ], queueDataKey, undefined, true ) ||
						jQuery.data( elements[ i ], markDataKey, undefined, true ) ) &&
					jQuery.data( elements[ i ], deferDataKey, jQuery.Callbacks( "once memory" ), true ) )) {
				count++;
				tmp.add( resolve );
			}
		}
		resolve();
		return defer.promise( object );
	}
});




var rclass = /[\n\t\r]/g,
	rspace = /\s+/,
	rreturn = /\r/g,
	rtype = /^(?:button|input)$/i,
	rfocusable = /^(?:button|input|object|select|textarea)$/i,
	rclickable = /^a(?:rea)?$/i,
	rboolean = /^(?:autofocus|autoplay|async|checked|controls|defer|disabled|hidden|loop|multiple|open|readonly|required|scoped|selected)$/i,
	getSetAttribute = jQuery.support.getSetAttribute,
	nodeHook, boolHook, fixSpecified;

jQuery.fn.extend({
	attr: function( name, value ) {
		return jQuery.access( this, jQuery.attr, name, value, arguments.length > 1 );
	},

	removeAttr: function( name ) {
		return this.each(function() {
			jQuery.removeAttr( this, name );
		});
	},

	prop: function( name, value ) {
		return jQuery.access( this, jQuery.prop, name, value, arguments.length > 1 );
	},

	removeProp: function( name ) {
		name = jQuery.propFix[ name ] || name;
		return this.each(function() {
			// try/catch handles cases where IE balks (such as removing a property on window)
			try {
				this[ name ] = undefined;
				delete this[ name ];
			} catch( e ) {}
		});
	},

	addClass: function( value ) {
		var classNames, i, l, elem,
			setClass, c, cl;

		if ( jQuery.isFunction( value ) ) {
			return this.each(function( j ) {
				jQuery( this ).addClass( value.call(this, j, this.className) );
			});
		}

		if ( value && typeof value === "string" ) {
			classNames = value.split( rspace );

			for ( i = 0, l = this.length; i < l; i++ ) {
				elem = this[ i ];

				if ( elem.nodeType === 1 ) {
					if ( !elem.className && classNames.length === 1 ) {
						elem.className = value;

					} else {
						setClass = " " + elem.className + " ";

						for ( c = 0, cl = classNames.length; c < cl; c++ ) {
							if ( !~setClass.indexOf( " " + classNames[ c ] + " " ) ) {
								setClass += classNames[ c ] + " ";
							}
						}
						elem.className = jQuery.trim( setClass );
					}
				}
			}
		}

		return this;
	},

	removeClass: function( value ) {
		var classNames, i, l, elem, className, c, cl;

		if ( jQuery.isFunction( value ) ) {
			return this.each(function( j ) {
				jQuery( this ).removeClass( value.call(this, j, this.className) );
			});
		}

		if ( (value && typeof value === "string") || value === undefined ) {
			classNames = ( value || "" ).split( rspace );

			for ( i = 0, l = this.length; i < l; i++ ) {
				elem = this[ i ];

				if ( elem.nodeType === 1 && elem.className ) {
					if ( value ) {
						className = (" " + elem.className + " ").replace( rclass, " " );
						for ( c = 0, cl = classNames.length; c < cl; c++ ) {
							className = className.replace(" " + classNames[ c ] + " ", " ");
						}
						elem.className = jQuery.trim( className );

					} else {
						elem.className = "";
					}
				}
			}
		}

		return this;
	},

	toggleClass: function( value, stateVal ) {
		var type = typeof value,
			isBool = typeof stateVal === "boolean";

		if ( jQuery.isFunction( value ) ) {
			return this.each(function( i ) {
				jQuery( this ).toggleClass( value.call(this, i, this.className, stateVal), stateVal );
			});
		}

		return this.each(function() {
			if ( type === "string" ) {
				// toggle individual class names
				var className,
					i = 0,
					self = jQuery( this ),
					state = stateVal,
					classNames = value.split( rspace );

				while ( (className = classNames[ i++ ]) ) {
					// check each className given, space seperated list
					state = isBool ? state : !self.hasClass( className );
					self[ state ? "addClass" : "removeClass" ]( className );
				}

			} else if ( type === "undefined" || type === "boolean" ) {
				if ( this.className ) {
					// store className if set
					jQuery._data( this, "__className__", this.className );
				}

				// toggle whole className
				this.className = this.className || value === false ? "" : jQuery._data( this, "__className__" ) || "";
			}
		});
	},

	hasClass: function( selector ) {
		var className = " " + selector + " ",
			i = 0,
			l = this.length;
		for ( ; i < l; i++ ) {
			if ( this[i].nodeType === 1 && (" " + this[i].className + " ").replace(rclass, " ").indexOf( className ) > -1 ) {
				return true;
			}
		}

		return false;
	},

	val: function( value ) {
		var hooks, ret, isFunction,
			elem = this[0];

		if ( !arguments.length ) {
			if ( elem ) {
				hooks = jQuery.valHooks[ elem.type ] || jQuery.valHooks[ elem.nodeName.toLowerCase() ];

				if ( hooks && "get" in hooks && (ret = hooks.get( elem, "value" )) !== undefined ) {
					return ret;
				}

				ret = elem.value;

				return typeof ret === "string" ?
					// handle most common string cases
					ret.replace(rreturn, "") :
					// handle cases where value is null/undef or number
					ret == null ? "" : ret;
			}

			return;
		}

		isFunction = jQuery.isFunction( value );

		return this.each(function( i ) {
			var self = jQuery(this), val;

			if ( this.nodeType !== 1 ) {
				return;
			}

			if ( isFunction ) {
				val = value.call( this, i, self.val() );
			} else {
				val = value;
			}

			// Treat null/undefined as ""; convert numbers to string
			if ( val == null ) {
				val = "";
			} else if ( typeof val === "number" ) {
				val += "";
			} else if ( jQuery.isArray( val ) ) {
				val = jQuery.map(val, function ( value ) {
					return value == null ? "" : value + "";
				});
			}

			hooks = jQuery.valHooks[ this.type ] || jQuery.valHooks[ this.nodeName.toLowerCase() ];

			// If set returns undefined, fall back to normal setting
			if ( !hooks || !("set" in hooks) || hooks.set( this, val, "value" ) === undefined ) {
				this.value = val;
			}
		});
	}
});

jQuery.extend({
	valHooks: {
		option: {
			get: function( elem ) {
				// attributes.value is undefined in Blackberry 4.7 but
				// uses .value. See #6932
				var val = elem.attributes.value;
				return !val || val.specified ? elem.value : elem.text;
			}
		},
		select: {
			get: function( elem ) {
				var value, i, max, option,
					index = elem.selectedIndex,
					values = [],
					options = elem.options,
					one = elem.type === "select-one";

				// Nothing was selected
				if ( index < 0 ) {
					return null;
				}

				// Loop through all the selected options
				i = one ? index : 0;
				max = one ? index + 1 : options.length;
				for ( ; i < max; i++ ) {
					option = options[ i ];

					// Don't return options that are disabled or in a disabled optgroup
					if ( option.selected && (jQuery.support.optDisabled ? !option.disabled : option.getAttribute("disabled") === null) &&
							(!option.parentNode.disabled || !jQuery.nodeName( option.parentNode, "optgroup" )) ) {

						// Get the specific value for the option
						value = jQuery( option ).val();

						// We don't need an array for one selects
						if ( one ) {
							return value;
						}

						// Multi-Selects return an array
						values.push( value );
					}
				}

				// Fixes Bug #2551 -- select.val() broken in IE after form.reset()
				if ( one && !values.length && options.length ) {
					return jQuery( options[ index ] ).val();
				}

				return values;
			},

			set: function( elem, value ) {
				var values = jQuery.makeArray( value );

				jQuery(elem).find("option").each(function() {
					this.selected = jQuery.inArray( jQuery(this).val(), values ) >= 0;
				});

				if ( !values.length ) {
					elem.selectedIndex = -1;
				}
				return values;
			}
		}
	},

	attrFn: {
		val: true,
		css: true,
		html: true,
		text: true,
		data: true,
		width: true,
		height: true,
		offset: true
	},

	attr: function( elem, name, value, pass ) {
		var ret, hooks, notxml,
			nType = elem.nodeType;

		// don't get/set attributes on text, comment and attribute nodes
		if ( !elem || nType === 3 || nType === 8 || nType === 2 ) {
			return;
		}

		if ( pass && name in jQuery.attrFn ) {
			return jQuery( elem )[ name ]( value );
		}

		// Fallback to prop when attributes are not supported
		if ( typeof elem.getAttribute === "undefined" ) {
			return jQuery.prop( elem, name, value );
		}

		notxml = nType !== 1 || !jQuery.isXMLDoc( elem );

		// All attributes are lowercase
		// Grab necessary hook if one is defined
		if ( notxml ) {
			name = name.toLowerCase();
			hooks = jQuery.attrHooks[ name ] || ( rboolean.test( name ) ? boolHook : nodeHook );
		}

		if ( value !== undefined ) {

			if ( value === null ) {
				jQuery.removeAttr( elem, name );
				return;

			} else if ( hooks && "set" in hooks && notxml && (ret = hooks.set( elem, value, name )) !== undefined ) {
				return ret;

			} else {
				elem.setAttribute( name, "" + value );
				return value;
			}

		} else if ( hooks && "get" in hooks && notxml && (ret = hooks.get( elem, name )) !== null ) {
			return ret;

		} else {

			ret = elem.getAttribute( name );

			// Non-existent attributes return null, we normalize to undefined
			return ret === null ?
				undefined :
				ret;
		}
	},

	removeAttr: function( elem, value ) {
		var propName, attrNames, name, l, isBool,
			i = 0;

		if ( value && elem.nodeType === 1 ) {
			attrNames = value.toLowerCase().split( rspace );
			l = attrNames.length;

			for ( ; i < l; i++ ) {
				name = attrNames[ i ];

				if ( name ) {
					propName = jQuery.propFix[ name ] || name;
					isBool = rboolean.test( name );

					// See #9699 for explanation of this approach (setting first, then removal)
					// Do not do this for boolean attributes (see #10870)
					if ( !isBool ) {
						jQuery.attr( elem, name, "" );
					}
					elem.removeAttribute( getSetAttribute ? name : propName );

					// Set corresponding property to false for boolean attributes
					if ( isBool && propName in elem ) {
						elem[ propName ] = false;
					}
				}
			}
		}
	},

	attrHooks: {
		type: {
			set: function( elem, value ) {
				// We can't allow the type property to be changed (since it causes problems in IE)
				if ( rtype.test( elem.nodeName ) && elem.parentNode ) {
					jQuery.error( "type property can't be changed" );
				} else if ( !jQuery.support.radioValue && value === "radio" && jQuery.nodeName(elem, "input") ) {
					// Setting the type on a radio button after the value resets the value in IE6-9
					// Reset value to it's default in case type is set after value
					// This is for element creation
					var val = elem.value;
					elem.setAttribute( "type", value );
					if ( val ) {
						elem.value = val;
					}
					return value;
				}
			}
		},
		// Use the value property for back compat
		// Use the nodeHook for button elements in IE6/7 (#1954)
		value: {
			get: function( elem, name ) {
				if ( nodeHook && jQuery.nodeName( elem, "button" ) ) {
					return nodeHook.get( elem, name );
				}
				return name in elem ?
					elem.value :
					null;
			},
			set: function( elem, value, name ) {
				if ( nodeHook && jQuery.nodeName( elem, "button" ) ) {
					return nodeHook.set( elem, value, name );
				}
				// Does not return so that setAttribute is also used
				elem.value = value;
			}
		}
	},

	propFix: {
		tabindex: "tabIndex",
		readonly: "readOnly",
		"for": "htmlFor",
		"class": "className",
		maxlength: "maxLength",
		cellspacing: "cellSpacing",
		cellpadding: "cellPadding",
		rowspan: "rowSpan",
		colspan: "colSpan",
		usemap: "useMap",
		frameborder: "frameBorder",
		contenteditable: "contentEditable"
	},

	prop: function( elem, name, value ) {
		var ret, hooks, notxml,
			nType = elem.nodeType;

		// don't get/set properties on text, comment and attribute nodes
		if ( !elem || nType === 3 || nType === 8 || nType === 2 ) {
			return;
		}

		notxml = nType !== 1 || !jQuery.isXMLDoc( elem );

		if ( notxml ) {
			// Fix name and attach hooks
			name = jQuery.propFix[ name ] || name;
			hooks = jQuery.propHooks[ name ];
		}

		if ( value !== undefined ) {
			if ( hooks && "set" in hooks && (ret = hooks.set( elem, value, name )) !== undefined ) {
				return ret;

			} else {
				return ( elem[ name ] = value );
			}

		} else {
			if ( hooks && "get" in hooks && (ret = hooks.get( elem, name )) !== null ) {
				return ret;

			} else {
				return elem[ name ];
			}
		}
	},

	propHooks: {
		tabIndex: {
			get: function( elem ) {
				// elem.tabIndex doesn't always return the correct value when it hasn't been explicitly set
				// http://fluidproject.org/blog/2008/01/09/getting-setting-and-removing-tabindex-values-with-javascript/
				var attributeNode = elem.getAttributeNode("tabindex");

				return attributeNode && attributeNode.specified ?
					parseInt( attributeNode.value, 10 ) :
					rfocusable.test( elem.nodeName ) || rclickable.test( elem.nodeName ) && elem.href ?
						0 :
						undefined;
			}
		}
	}
});

// Add the tabIndex propHook to attrHooks for back-compat (different case is intentional)
jQuery.attrHooks.tabindex = jQuery.propHooks.tabIndex;

// Hook for boolean attributes
boolHook = {
	get: function( elem, name ) {
		// Align boolean attributes with corresponding properties
		// Fall back to attribute presence where some booleans are not supported
		var attrNode,
			property = jQuery.prop( elem, name );
		return property === true || typeof property !== "boolean" && ( attrNode = elem.getAttributeNode(name) ) && attrNode.nodeValue !== false ?
			name.toLowerCase() :
			undefined;
	},
	set: function( elem, value, name ) {
		var propName;
		if ( value === false ) {
			// Remove boolean attributes when set to false
			jQuery.removeAttr( elem, name );
		} else {
			// value is true since we know at this point it's type boolean and not false
			// Set boolean attributes to the same name and set the DOM property
			propName = jQuery.propFix[ name ] || name;
			if ( propName in elem ) {
				// Only set the IDL specifically if it already exists on the element
				elem[ propName ] = true;
			}

			elem.setAttribute( name, name.toLowerCase() );
		}
		return name;
	}
};

// IE6/7 do not support getting/setting some attributes with get/setAttribute
if ( !getSetAttribute ) {

	fixSpecified = {
		name: true,
		id: true,
		coords: true
	};

	// Use this for any attribute in IE6/7
	// This fixes almost every IE6/7 issue
	nodeHook = jQuery.valHooks.button = {
		get: function( elem, name ) {
			var ret;
			ret = elem.getAttributeNode( name );
			return ret && ( fixSpecified[ name ] ? ret.nodeValue !== "" : ret.specified ) ?
				ret.nodeValue :
				undefined;
		},
		set: function( elem, value, name ) {
			// Set the existing or create a new attribute node
			var ret = elem.getAttributeNode( name );
			if ( !ret ) {
				ret = document.createAttribute( name );
				elem.setAttributeNode( ret );
			}
			return ( ret.nodeValue = value + "" );
		}
	};

	// Apply the nodeHook to tabindex
	jQuery.attrHooks.tabindex.set = nodeHook.set;

	// Set width and height to auto instead of 0 on empty string( Bug #8150 )
	// This is for removals
	jQuery.each([ "width", "height" ], function( i, name ) {
		jQuery.attrHooks[ name ] = jQuery.extend( jQuery.attrHooks[ name ], {
			set: function( elem, value ) {
				if ( value === "" ) {
					elem.setAttribute( name, "auto" );
					return value;
				}
			}
		});
	});

	// Set contenteditable to false on removals(#10429)
	// Setting to empty string throws an error as an invalid value
	jQuery.attrHooks.contenteditable = {
		get: nodeHook.get,
		set: function( elem, value, name ) {
			if ( value === "" ) {
				value = "false";
			}
			nodeHook.set( elem, value, name );
		}
	};
}


// Some attributes require a special call on IE
if ( !jQuery.support.hrefNormalized ) {
	jQuery.each([ "href", "src", "width", "height" ], function( i, name ) {
		jQuery.attrHooks[ name ] = jQuery.extend( jQuery.attrHooks[ name ], {
			get: function( elem ) {
				var ret = elem.getAttribute( name, 2 );
				return ret === null ? undefined : ret;
			}
		});
	});
}

if ( !jQuery.support.style ) {
	jQuery.attrHooks.style = {
		get: function( elem ) {
			// Return undefined in the case of empty string
			// Normalize to lowercase since IE uppercases css property names
			return elem.style.cssText.toLowerCase() || undefined;
		},
		set: function( elem, value ) {
			return ( elem.style.cssText = "" + value );
		}
	};
}

// Safari mis-reports the default selected property of an option
// Accessing the parent's selectedIndex property fixes it
if ( !jQuery.support.optSelected ) {
	jQuery.propHooks.selected = jQuery.extend( jQuery.propHooks.selected, {
		get: function( elem ) {
			var parent = elem.parentNode;

			if ( parent ) {
				parent.selectedIndex;

				// Make sure that it also works with optgroups, see #5701
				if ( parent.parentNode ) {
					parent.parentNode.selectedIndex;
				}
			}
			return null;
		}
	});
}

// IE6/7 call enctype encoding
if ( !jQuery.support.enctype ) {
	jQuery.propFix.enctype = "encoding";
}

// Radios and checkboxes getter/setter
if ( !jQuery.support.checkOn ) {
	jQuery.each([ "radio", "checkbox" ], function() {
		jQuery.valHooks[ this ] = {
			get: function( elem ) {
				// Handle the case where in Webkit "" is returned instead of "on" if a value isn't specified
				return elem.getAttribute("value") === null ? "on" : elem.value;
			}
		};
	});
}
jQuery.each([ "radio", "checkbox" ], function() {
	jQuery.valHooks[ this ] = jQuery.extend( jQuery.valHooks[ this ], {
		set: function( elem, value ) {
			if ( jQuery.isArray( value ) ) {
				return ( elem.checked = jQuery.inArray( jQuery(elem).val(), value ) >= 0 );
			}
		}
	});
});




var rformElems = /^(?:textarea|input|select)$/i,
	rtypenamespace = /^([^\.]*)?(?:\.(.+))?$/,
	rhoverHack = /(?:^|\s)hover(\.\S+)?\b/,
	rkeyEvent = /^key/,
	rmouseEvent = /^(?:mouse|contextmenu)|click/,
	rfocusMorph = /^(?:focusinfocus|focusoutblur)$/,
	rquickIs = /^(\w*)(?:#([\w\-]+))?(?:\.([\w\-]+))?$/,
	quickParse = function( selector ) {
		var quick = rquickIs.exec( selector );
		if ( quick ) {
			//   0  1    2   3
			// [ _, tag, id, class ]
			quick[1] = ( quick[1] || "" ).toLowerCase();
			quick[3] = quick[3] && new RegExp( "(?:^|\\s)" + quick[3] + "(?:\\s|$)" );
		}
		return quick;
	},
	quickIs = function( elem, m ) {
		var attrs = elem.attributes || {};
		return (
			(!m[1] || elem.nodeName.toLowerCase() === m[1]) &&
			(!m[2] || (attrs.id || {}).value === m[2]) &&
			(!m[3] || m[3].test( (attrs[ "class" ] || {}).value ))
		);
	},
	hoverHack = function( events ) {
		return jQuery.event.special.hover ? events : events.replace( rhoverHack, "mouseenter$1 mouseleave$1" );
	};

/*
 * Helper functions for managing events -- not part of the public interface.
 * Props to Dean Edwards' addEvent library for many of the ideas.
 */
jQuery.event = {

	add: function( elem, types, handler, data, selector ) {

		var elemData, eventHandle, events,
			t, tns, type, namespaces, handleObj,
			handleObjIn, quick, handlers, special;

		// Don't attach events to noData or text/comment nodes (allow plain objects tho)
		if ( elem.nodeType === 3 || elem.nodeType === 8 || !types || !handler || !(elemData = jQuery._data( elem )) ) {
			return;
		}

		// Caller can pass in an object of custom data in lieu of the handler
		if ( handler.handler ) {
			handleObjIn = handler;
			handler = handleObjIn.handler;
			selector = handleObjIn.selector;
		}

		// Make sure that the handler has a unique ID, used to find/remove it later
		if ( !handler.guid ) {
			handler.guid = jQuery.guid++;
		}

		// Init the element's event structure and main handler, if this is the first
		events = elemData.events;
		if ( !events ) {
			elemData.events = events = {};
		}
		eventHandle = elemData.handle;
		if ( !eventHandle ) {
			elemData.handle = eventHandle = function( e ) {
				// Discard the second event of a jQuery.event.trigger() and
				// when an event is called after a page has unloaded
				return typeof jQuery !== "undefined" && (!e || jQuery.event.triggered !== e.type) ?
					jQuery.event.dispatch.apply( eventHandle.elem, arguments ) :
					undefined;
			};
			// Add elem as a property of the handle fn to prevent a memory leak with IE non-native events
			eventHandle.elem = elem;
		}

		// Handle multiple events separated by a space
		// jQuery(...).bind("mouseover mouseout", fn);
		types = jQuery.trim( hoverHack(types) ).split( " " );
		for ( t = 0; t < types.length; t++ ) {

			tns = rtypenamespace.exec( types[t] ) || [];
			type = tns[1];
			namespaces = ( tns[2] || "" ).split( "." ).sort();

			// If event changes its type, use the special event handlers for the changed type
			special = jQuery.event.special[ type ] || {};

			// If selector defined, determine special event api type, otherwise given type
			type = ( selector ? special.delegateType : special.bindType ) || type;

			// Update special based on newly reset type
			special = jQuery.event.special[ type ] || {};

			// handleObj is passed to all event handlers
			handleObj = jQuery.extend({
				type: type,
				origType: tns[1],
				data: data,
				handler: handler,
				guid: handler.guid,
				selector: selector,
				quick: selector && quickParse( selector ),
				namespace: namespaces.join(".")
			}, handleObjIn );

			// Init the event handler queue if we're the first
			handlers = events[ type ];
			if ( !handlers ) {
				handlers = events[ type ] = [];
				handlers.delegateCount = 0;

				// Only use addEventListener/attachEvent if the special events handler returns false
				if ( !special.setup || special.setup.call( elem, data, namespaces, eventHandle ) === false ) {
					// Bind the global event handler to the element
					if ( elem.addEventListener ) {
						elem.addEventListener( type, eventHandle, false );

					} else if ( elem.attachEvent ) {
						elem.attachEvent( "on" + type, eventHandle );
					}
				}
			}

			if ( special.add ) {
				special.add.call( elem, handleObj );

				if ( !handleObj.handler.guid ) {
					handleObj.handler.guid = handler.guid;
				}
			}

			// Add to the element's handler list, delegates in front
			if ( selector ) {
				handlers.splice( handlers.delegateCount++, 0, handleObj );
			} else {
				handlers.push( handleObj );
			}

			// Keep track of which events have ever been used, for event optimization
			jQuery.event.global[ type ] = true;
		}

		// Nullify elem to prevent memory leaks in IE
		elem = null;
	},

	global: {},

	// Detach an event or set of events from an element
	remove: function( elem, types, handler, selector, mappedTypes ) {

		var elemData = jQuery.hasData( elem ) && jQuery._data( elem ),
			t, tns, type, origType, namespaces, origCount,
			j, events, special, handle, eventType, handleObj;

		if ( !elemData || !(events = elemData.events) ) {
			return;
		}

		// Once for each type.namespace in types; type may be omitted
		types = jQuery.trim( hoverHack( types || "" ) ).split(" ");
		for ( t = 0; t < types.length; t++ ) {
			tns = rtypenamespace.exec( types[t] ) || [];
			type = origType = tns[1];
			namespaces = tns[2];

			// Unbind all events (on this namespace, if provided) for the element
			if ( !type ) {
				for ( type in events ) {
					jQuery.event.remove( elem, type + types[ t ], handler, selector, true );
				}
				continue;
			}

			special = jQuery.event.special[ type ] || {};
			type = ( selector? special.delegateType : special.bindType ) || type;
			eventType = events[ type ] || [];
			origCount = eventType.length;
			namespaces = namespaces ? new RegExp("(^|\\.)" + namespaces.split(".").sort().join("\\.(?:.*\\.)?") + "(\\.|$)") : null;

			// Remove matching events
			for ( j = 0; j < eventType.length; j++ ) {
				handleObj = eventType[ j ];

				if ( ( mappedTypes || origType === handleObj.origType ) &&
					 ( !handler || handler.guid === handleObj.guid ) &&
					 ( !namespaces || namespaces.test( handleObj.namespace ) ) &&
					 ( !selector || selector === handleObj.selector || selector === "**" && handleObj.selector ) ) {
					eventType.splice( j--, 1 );

					if ( handleObj.selector ) {
						eventType.delegateCount--;
					}
					if ( special.remove ) {
						special.remove.call( elem, handleObj );
					}
				}
			}

			// Remove generic event handler if we removed something and no more handlers exist
			// (avoids potential for endless recursion during removal of special event handlers)
			if ( eventType.length === 0 && origCount !== eventType.length ) {
				if ( !special.teardown || special.teardown.call( elem, namespaces ) === false ) {
					jQuery.removeEvent( elem, type, elemData.handle );
				}

				delete events[ type ];
			}
		}

		// Remove the expando if it's no longer used
		if ( jQuery.isEmptyObject( events ) ) {
			handle = elemData.handle;
			if ( handle ) {
				handle.elem = null;
			}

			// removeData also checks for emptiness and clears the expando if empty
			// so use it instead of delete
			jQuery.removeData( elem, [ "events", "handle" ], true );
		}
	},

	// Events that are safe to short-circuit if no handlers are attached.
	// Native DOM events should not be added, they may have inline handlers.
	customEvent: {
		"getData": true,
		"setData": true,
		"changeData": true
	},

	trigger: function( event, data, elem, onlyHandlers ) {
		// Don't do events on text and comment nodes
		if ( elem && (elem.nodeType === 3 || elem.nodeType === 8) ) {
			return;
		}

		// Event object or event type
		var type = event.type || event,
			namespaces = [],
			cache, exclusive, i, cur, old, ontype, special, handle, eventPath, bubbleType;

		// focus/blur morphs to focusin/out; ensure we're not firing them right now
		if ( rfocusMorph.test( type + jQuery.event.triggered ) ) {
			return;
		}

		if ( type.indexOf( "!" ) >= 0 ) {
			// Exclusive events trigger only for the exact event (no namespaces)
			type = type.slice(0, -1);
			exclusive = true;
		}

		if ( type.indexOf( "." ) >= 0 ) {
			// Namespaced trigger; create a regexp to match event type in handle()
			namespaces = type.split(".");
			type = namespaces.shift();
			namespaces.sort();
		}

		if ( (!elem || jQuery.event.customEvent[ type ]) && !jQuery.event.global[ type ] ) {
			// No jQuery handlers for this event type, and it can't have inline handlers
			return;
		}

		// Caller can pass in an Event, Object, or just an event type string
		event = typeof event === "object" ?
			// jQuery.Event object
			event[ jQuery.expando ] ? event :
			// Object literal
			new jQuery.Event( type, event ) :
			// Just the event type (string)
			new jQuery.Event( type );

		event.type = type;
		event.isTrigger = true;
		event.exclusive = exclusive;
		event.namespace = namespaces.join( "." );
		event.namespace_re = event.namespace? new RegExp("(^|\\.)" + namespaces.join("\\.(?:.*\\.)?") + "(\\.|$)") : null;
		ontype = type.indexOf( ":" ) < 0 ? "on" + type : "";

		// Handle a global trigger
		if ( !elem ) {

			// TODO: Stop taunting the data cache; remove global events and always attach to document
			cache = jQuery.cache;
			for ( i in cache ) {
				if ( cache[ i ].events && cache[ i ].events[ type ] ) {
					jQuery.event.trigger( event, data, cache[ i ].handle.elem, true );
				}
			}
			return;
		}

		// Clean up the event in case it is being reused
		event.result = undefined;
		if ( !event.target ) {
			event.target = elem;
		}

		// Clone any incoming data and prepend the event, creating the handler arg list
		data = data != null ? jQuery.makeArray( data ) : [];
		data.unshift( event );

		// Allow special events to draw outside the lines
		special = jQuery.event.special[ type ] || {};
		if ( special.trigger && special.trigger.apply( elem, data ) === false ) {
			return;
		}

		// Determine event propagation path in advance, per W3C events spec (#9951)
		// Bubble up to document, then to window; watch for a global ownerDocument var (#9724)
		eventPath = [[ elem, special.bindType || type ]];
		if ( !onlyHandlers && !special.noBubble && !jQuery.isWindow( elem ) ) {

			bubbleType = special.delegateType || type;
			cur = rfocusMorph.test( bubbleType + type ) ? elem : elem.parentNode;
			old = null;
			for ( ; cur; cur = cur.parentNode ) {
				eventPath.push([ cur, bubbleType ]);
				old = cur;
			}

			// Only add window if we got to document (e.g., not plain obj or detached DOM)
			if ( old && old === elem.ownerDocument ) {
				eventPath.push([ old.defaultView || old.parentWindow || window, bubbleType ]);
			}
		}

		// Fire handlers on the event path
		for ( i = 0; i < eventPath.length && !event.isPropagationStopped(); i++ ) {

			cur = eventPath[i][0];
			event.type = eventPath[i][1];

			handle = ( jQuery._data( cur, "events" ) || {} )[ event.type ] && jQuery._data( cur, "handle" );
			if ( handle ) {
				handle.apply( cur, data );
			}
			// Note that this is a bare JS function and not a jQuery handler
			handle = ontype && cur[ ontype ];
			if ( handle && jQuery.acceptData( cur ) && handle.apply( cur, data ) === false ) {
				event.preventDefault();
			}
		}
		event.type = type;

		// If nobody prevented the default action, do it now
		if ( !onlyHandlers && !event.isDefaultPrevented() ) {

			if ( (!special._default || special._default.apply( elem.ownerDocument, data ) === false) &&
				!(type === "click" && jQuery.nodeName( elem, "a" )) && jQuery.acceptData( elem ) ) {

				// Call a native DOM method on the target with the same name name as the event.
				// Can't use an .isFunction() check here because IE6/7 fails that test.
				// Don't do default actions on window, that's where global variables be (#6170)
				// IE<9 dies on focus/blur to hidden element (#1486)
				if ( ontype && elem[ type ] && ((type !== "focus" && type !== "blur") || event.target.offsetWidth !== 0) && !jQuery.isWindow( elem ) ) {

					// Don't re-trigger an onFOO event when we call its FOO() method
					old = elem[ ontype ];

					if ( old ) {
						elem[ ontype ] = null;
					}

					// Prevent re-triggering of the same event, since we already bubbled it above
					jQuery.event.triggered = type;
					elem[ type ]();
					jQuery.event.triggered = undefined;

					if ( old ) {
						elem[ ontype ] = old;
					}
				}
			}
		}

		return event.result;
	},

	dispatch: function( event ) {

		// Make a writable jQuery.Event from the native event object
		event = jQuery.event.fix( event || window.event );

		var handlers = ( (jQuery._data( this, "events" ) || {} )[ event.type ] || []),
			delegateCount = handlers.delegateCount,
			args = [].slice.call( arguments, 0 ),
			run_all = !event.exclusive && !event.namespace,
			special = jQuery.event.special[ event.type ] || {},
			handlerQueue = [],
			i, j, cur, jqcur, ret, selMatch, matched, matches, handleObj, sel, related;

		// Use the fix-ed jQuery.Event rather than the (read-only) native event
		args[0] = event;
		event.delegateTarget = this;

		// Call the preDispatch hook for the mapped type, and let it bail if desired
		if ( special.preDispatch && special.preDispatch.call( this, event ) === false ) {
			return;
		}

		// Determine handlers that should run if there are delegated events
		// Avoid non-left-click bubbling in Firefox (#3861)
		if ( delegateCount && !(event.button && event.type === "click") ) {

			// Pregenerate a single jQuery object for reuse with .is()
			jqcur = jQuery(this);
			jqcur.context = this.ownerDocument || this;

			for ( cur = event.target; cur != this; cur = cur.parentNode || this ) {

				// Don't process events on disabled elements (#6911, #8165)
				if ( cur.disabled !== true ) {
					selMatch = {};
					matches = [];
					jqcur[0] = cur;
					for ( i = 0; i < delegateCount; i++ ) {
						handleObj = handlers[ i ];
						sel = handleObj.selector;

						if ( selMatch[ sel ] === undefined ) {
							selMatch[ sel ] = (
								handleObj.quick ? quickIs( cur, handleObj.quick ) : jqcur.is( sel )
							);
						}
						if ( selMatch[ sel ] ) {
							matches.push( handleObj );
						}
					}
					if ( matches.length ) {
						handlerQueue.push({ elem: cur, matches: matches });
					}
				}
			}
		}

		// Add the remaining (directly-bound) handlers
		if ( handlers.length > delegateCount ) {
			handlerQueue.push({ elem: this, matches: handlers.slice( delegateCount ) });
		}

		// Run delegates first; they may want to stop propagation beneath us
		for ( i = 0; i < handlerQueue.length && !event.isPropagationStopped(); i++ ) {
			matched = handlerQueue[ i ];
			event.currentTarget = matched.elem;

			for ( j = 0; j < matched.matches.length && !event.isImmediatePropagationStopped(); j++ ) {
				handleObj = matched.matches[ j ];

				// Triggered event must either 1) be non-exclusive and have no namespace, or
				// 2) have namespace(s) a subset or equal to those in the bound event (both can have no namespace).
				if ( run_all || (!event.namespace && !handleObj.namespace) || event.namespace_re && event.namespace_re.test( handleObj.namespace ) ) {

					event.data = handleObj.data;
					event.handleObj = handleObj;

					ret = ( (jQuery.event.special[ handleObj.origType ] || {}).handle || handleObj.handler )
							.apply( matched.elem, args );

					if ( ret !== undefined ) {
						event.result = ret;
						if ( ret === false ) {
							event.preventDefault();
							event.stopPropagation();
						}
					}
				}
			}
		}

		// Call the postDispatch hook for the mapped type
		if ( special.postDispatch ) {
			special.postDispatch.call( this, event );
		}

		return event.result;
	},

	// Includes some event props shared by KeyEvent and MouseEvent
	// *** attrChange attrName relatedNode srcElement  are not normalized, non-W3C, deprecated, will be removed in 1.8 ***
	props: "attrChange attrName relatedNode srcElement altKey bubbles cancelable ctrlKey currentTarget eventPhase metaKey relatedTarget shiftKey target timeStamp view which".split(" "),

	fixHooks: {},

	keyHooks: {
		props: "char charCode key keyCode".split(" "),
		filter: function( event, original ) {

			// Add which for key events
			if ( event.which == null ) {
				event.which = original.charCode != null ? original.charCode : original.keyCode;
			}

			return event;
		}
	},

	mouseHooks: {
		props: "button buttons clientX clientY fromElement offsetX offsetY pageX pageY screenX screenY toElement".split(" "),
		filter: function( event, original ) {
			var eventDoc, doc, body,
				button = original.button,
				fromElement = original.fromElement;

			// Calculate pageX/Y if missing and clientX/Y available
			if ( event.pageX == null && original.clientX != null ) {
				eventDoc = event.target.ownerDocument || document;
				doc = eventDoc.documentElement;
				body = eventDoc.body;

				event.pageX = original.clientX + ( doc && doc.scrollLeft || body && body.scrollLeft || 0 ) - ( doc && doc.clientLeft || body && body.clientLeft || 0 );
				event.pageY = original.clientY + ( doc && doc.scrollTop  || body && body.scrollTop  || 0 ) - ( doc && doc.clientTop  || body && body.clientTop  || 0 );
			}

			// Add relatedTarget, if necessary
			if ( !event.relatedTarget && fromElement ) {
				event.relatedTarget = fromElement === event.target ? original.toElement : fromElement;
			}

			// Add which for click: 1 === left; 2 === middle; 3 === right
			// Note: button is not normalized, so don't use it
			if ( !event.which && button !== undefined ) {
				event.which = ( button & 1 ? 1 : ( button & 2 ? 3 : ( button & 4 ? 2 : 0 ) ) );
			}

			return event;
		}
	},

	fix: function( event ) {
		if ( event[ jQuery.expando ] ) {
			return event;
		}

		// Create a writable copy of the event object and normalize some properties
		var i, prop,
			originalEvent = event,
			fixHook = jQuery.event.fixHooks[ event.type ] || {},
			copy = fixHook.props ? this.props.concat( fixHook.props ) : this.props;

		event = jQuery.Event( originalEvent );

		for ( i = copy.length; i; ) {
			prop = copy[ --i ];
			event[ prop ] = originalEvent[ prop ];
		}

		// Fix target property, if necessary (#1925, IE 6/7/8 & Safari2)
		if ( !event.target ) {
			event.target = originalEvent.srcElement || document;
		}

		// Target should not be a text node (#504, Safari)
		if ( event.target.nodeType === 3 ) {
			event.target = event.target.parentNode;
		}

		// For mouse/key events; add metaKey if it's not there (#3368, IE6/7/8)
		if ( event.metaKey === undefined ) {
			event.metaKey = event.ctrlKey;
		}

		return fixHook.filter? fixHook.filter( event, originalEvent ) : event;
	},

	special: {
		ready: {
			// Make sure the ready event is setup
			setup: jQuery.bindReady
		},

		load: {
			// Prevent triggered image.load events from bubbling to window.load
			noBubble: true
		},

		focus: {
			delegateType: "focusin"
		},
		blur: {
			delegateType: "focusout"
		},

		beforeunload: {
			setup: function( data, namespaces, eventHandle ) {
				// We only want to do this special case on windows
				if ( jQuery.isWindow( this ) ) {
					this.onbeforeunload = eventHandle;
				}
			},

			teardown: function( namespaces, eventHandle ) {
				if ( this.onbeforeunload === eventHandle ) {
					this.onbeforeunload = null;
				}
			}
		}
	},

	simulate: function( type, elem, event, bubble ) {
		// Piggyback on a donor event to simulate a different one.
		// Fake originalEvent to avoid donor's stopPropagation, but if the
		// simulated event prevents default then we do the same on the donor.
		var e = jQuery.extend(
			new jQuery.Event(),
			event,
			{ type: type,
				isSimulated: true,
				originalEvent: {}
			}
		);
		if ( bubble ) {
			jQuery.event.trigger( e, null, elem );
		} else {
			jQuery.event.dispatch.call( elem, e );
		}
		if ( e.isDefaultPrevented() ) {
			event.preventDefault();
		}
	}
};

// Some plugins are using, but it's undocumented/deprecated and will be removed.
// The 1.7 special event interface should provide all the hooks needed now.
jQuery.event.handle = jQuery.event.dispatch;

jQuery.removeEvent = document.removeEventListener ?
	function( elem, type, handle ) {
		if ( elem.removeEventListener ) {
			elem.removeEventListener( type, handle, false );
		}
	} :
	function( elem, type, handle ) {
		if ( elem.detachEvent ) {
			elem.detachEvent( "on" + type, handle );
		}
	};

jQuery.Event = function( src, props ) {
	// Allow instantiation without the 'new' keyword
	if ( !(this instanceof jQuery.Event) ) {
		return new jQuery.Event( src, props );
	}

	// Event object
	if ( src && src.type ) {
		this.originalEvent = src;
		this.type = src.type;

		// Events bubbling up the document may have been marked as prevented
		// by a handler lower down the tree; reflect the correct value.
		this.isDefaultPrevented = ( src.defaultPrevented || src.returnValue === false ||
			src.getPreventDefault && src.getPreventDefault() ) ? returnTrue : returnFalse;

	// Event type
	} else {
		this.type = src;
	}

	// Put explicitly provided properties onto the event object
	if ( props ) {
		jQuery.extend( this, props );
	}

	// Create a timestamp if incoming event doesn't have one
	this.timeStamp = src && src.timeStamp || jQuery.now();

	// Mark it as fixed
	this[ jQuery.expando ] = true;
};

function returnFalse() {
	return false;
}
function returnTrue() {
	return true;
}

// jQuery.Event is based on DOM3 Events as specified by the ECMAScript Language Binding
// http://www.w3.org/TR/2003/WD-DOM-Level-3-Events-20030331/ecma-script-binding.html
jQuery.Event.prototype = {
	preventDefault: function() {
		this.isDefaultPrevented = returnTrue;

		var e = this.originalEvent;
		if ( !e ) {
			return;
		}

		// if preventDefault exists run it on the original event
		if ( e.preventDefault ) {
			e.preventDefault();

		// otherwise set the returnValue property of the original event to false (IE)
		} else {
			e.returnValue = false;
		}
	},
	stopPropagation: function() {
		this.isPropagationStopped = returnTrue;

		var e = this.originalEvent;
		if ( !e ) {
			return;
		}
		// if stopPropagation exists run it on the original event
		if ( e.stopPropagation ) {
			e.stopPropagation();
		}
		// otherwise set the cancelBubble property of the original event to true (IE)
		e.cancelBubble = true;
	},
	stopImmediatePropagation: function() {
		this.isImmediatePropagationStopped = returnTrue;
		this.stopPropagation();
	},
	isDefaultPrevented: returnFalse,
	isPropagationStopped: returnFalse,
	isImmediatePropagationStopped: returnFalse
};

// Create mouseenter/leave events using mouseover/out and event-time checks
jQuery.each({
	mouseenter: "mouseover",
	mouseleave: "mouseout"
}, function( orig, fix ) {
	jQuery.event.special[ orig ] = {
		delegateType: fix,
		bindType: fix,

		handle: function( event ) {
			var target = this,
				related = event.relatedTarget,
				handleObj = event.handleObj,
				selector = handleObj.selector,
				ret;

			// For mousenter/leave call the handler if related is outside the target.
			// NB: No relatedTarget if the mouse left/entered the browser window
			if ( !related || (related !== target && !jQuery.contains( target, related )) ) {
				event.type = handleObj.origType;
				ret = handleObj.handler.apply( this, arguments );
				event.type = fix;
			}
			return ret;
		}
	};
});

// IE submit delegation
if ( !jQuery.support.submitBubbles ) {

	jQuery.event.special.submit = {
		setup: function() {
			// Only need this for delegated form submit events
			if ( jQuery.nodeName( this, "form" ) ) {
				return false;
			}

			// Lazy-add a submit handler when a descendant form may potentially be submitted
			jQuery.event.add( this, "click._submit keypress._submit", function( e ) {
				// Node name check avoids a VML-related crash in IE (#9807)
				var elem = e.target,
					form = jQuery.nodeName( elem, "input" ) || jQuery.nodeName( elem, "button" ) ? elem.form : undefined;
				if ( form && !form._submit_attached ) {
					jQuery.event.add( form, "submit._submit", function( event ) {
						event._submit_bubble = true;
					});
					form._submit_attached = true;
				}
			});
			// return undefined since we don't need an event listener
		},
		
		postDispatch: function( event ) {
			// If form was submitted by the user, bubble the event up the tree
			if ( event._submit_bubble ) {
				delete event._submit_bubble;
				if ( this.parentNode && !event.isTrigger ) {
					jQuery.event.simulate( "submit", this.parentNode, event, true );
				}
			}
		},

		teardown: function() {
			// Only need this for delegated form submit events
			if ( jQuery.nodeName( this, "form" ) ) {
				return false;
			}

			// Remove delegated handlers; cleanData eventually reaps submit handlers attached above
			jQuery.event.remove( this, "._submit" );
		}
	};
}

// IE change delegation and checkbox/radio fix
if ( !jQuery.support.changeBubbles ) {

	jQuery.event.special.change = {

		setup: function() {

			if ( rformElems.test( this.nodeName ) ) {
				// IE doesn't fire change on a check/radio until blur; trigger it on click
				// after a propertychange. Eat the blur-change in special.change.handle.
				// This still fires onchange a second time for check/radio after blur.
				if ( this.type === "checkbox" || this.type === "radio" ) {
					jQuery.event.add( this, "propertychange._change", function( event ) {
						if ( event.originalEvent.propertyName === "checked" ) {
							this._just_changed = true;
						}
					});
					jQuery.event.add( this, "click._change", function( event ) {
						if ( this._just_changed && !event.isTrigger ) {
							this._just_changed = false;
							jQuery.event.simulate( "change", this, event, true );
						}
					});
				}
				return false;
			}
			// Delegated event; lazy-add a change handler on descendant inputs
			jQuery.event.add( this, "beforeactivate._change", function( e ) {
				var elem = e.target;

				if ( rformElems.test( elem.nodeName ) && !elem._change_attached ) {
					jQuery.event.add( elem, "change._change", function( event ) {
						if ( this.parentNode && !event.isSimulated && !event.isTrigger ) {
							jQuery.event.simulate( "change", this.parentNode, event, true );
						}
					});
					elem._change_attached = true;
				}
			});
		},

		handle: function( event ) {
			var elem = event.target;

			// Swallow native change events from checkbox/radio, we already triggered them above
			if ( this !== elem || event.isSimulated || event.isTrigger || (elem.type !== "radio" && elem.type !== "checkbox") ) {
				return event.handleObj.handler.apply( this, arguments );
			}
		},

		teardown: function() {
			jQuery.event.remove( this, "._change" );

			return rformElems.test( this.nodeName );
		}
	};
}

// Create "bubbling" focus and blur events
if ( !jQuery.support.focusinBubbles ) {
	jQuery.each({ focus: "focusin", blur: "focusout" }, function( orig, fix ) {

		// Attach a single capturing handler while someone wants focusin/focusout
		var attaches = 0,
			handler = function( event ) {
				jQuery.event.simulate( fix, event.target, jQuery.event.fix( event ), true );
			};

		jQuery.event.special[ fix ] = {
			setup: function() {
				if ( attaches++ === 0 ) {
					document.addEventListener( orig, handler, true );
				}
			},
			teardown: function() {
				if ( --attaches === 0 ) {
					document.removeEventListener( orig, handler, true );
				}
			}
		};
	});
}

jQuery.fn.extend({

	on: function( types, selector, data, fn, /*INTERNAL*/ one ) {
		var origFn, type;

		// Types can be a map of types/handlers
		if ( typeof types === "object" ) {
			// ( types-Object, selector, data )
			if ( typeof selector !== "string" ) { // && selector != null
				// ( types-Object, data )
				data = data || selector;
				selector = undefined;
			}
			for ( type in types ) {
				this.on( type, selector, data, types[ type ], one );
			}
			return this;
		}

		if ( data == null && fn == null ) {
			// ( types, fn )
			fn = selector;
			data = selector = undefined;
		} else if ( fn == null ) {
			if ( typeof selector === "string" ) {
				// ( types, selector, fn )
				fn = data;
				data = undefined;
			} else {
				// ( types, data, fn )
				fn = data;
				data = selector;
				selector = undefined;
			}
		}
		if ( fn === false ) {
			fn = returnFalse;
		} else if ( !fn ) {
			return this;
		}

		if ( one === 1 ) {
			origFn = fn;
			fn = function( event ) {
				// Can use an empty set, since event contains the info
				jQuery().off( event );
				return origFn.apply( this, arguments );
			};
			// Use same guid so caller can remove using origFn
			fn.guid = origFn.guid || ( origFn.guid = jQuery.guid++ );
		}
		return this.each( function() {
			jQuery.event.add( this, types, fn, data, selector );
		});
	},
	one: function( types, selector, data, fn ) {
		return this.on( types, selector, data, fn, 1 );
	},
	off: function( types, selector, fn ) {
		if ( types && types.preventDefault && types.handleObj ) {
			// ( event )  dispatched jQuery.Event
			var handleObj = types.handleObj;
			jQuery( types.delegateTarget ).off(
				handleObj.namespace ? handleObj.origType + "." + handleObj.namespace : handleObj.origType,
				handleObj.selector,
				handleObj.handler
			);
			return this;
		}
		if ( typeof types === "object" ) {
			// ( types-object [, selector] )
			for ( var type in types ) {
				this.off( type, selector, types[ type ] );
			}
			return this;
		}
		if ( selector === false || typeof selector === "function" ) {
			// ( types [, fn] )
			fn = selector;
			selector = undefined;
		}
		if ( fn === false ) {
			fn = returnFalse;
		}
		return this.each(function() {
			jQuery.event.remove( this, types, fn, selector );
		});
	},

	bind: function( types, data, fn ) {
		return this.on( types, null, data, fn );
	},
	unbind: function( types, fn ) {
		return this.off( types, null, fn );
	},

	live: function( types, data, fn ) {
		jQuery( this.context ).on( types, this.selector, data, fn );
		return this;
	},
	die: function( types, fn ) {
		jQuery( this.context ).off( types, this.selector || "**", fn );
		return this;
	},

	delegate: function( selector, types, data, fn ) {
		return this.on( types, selector, data, fn );
	},
	undelegate: function( selector, types, fn ) {
		// ( namespace ) or ( selector, types [, fn] )
		return arguments.length == 1? this.off( selector, "**" ) : this.off( types, selector, fn );
	},

	trigger: function( type, data ) {
		return this.each(function() {
			jQuery.event.trigger( type, data, this );
		});
	},
	triggerHandler: function( type, data ) {
		if ( this[0] ) {
			return jQuery.event.trigger( type, data, this[0], true );
		}
	},

	toggle: function( fn ) {
		// Save reference to arguments for access in closure
		var args = arguments,
			guid = fn.guid || jQuery.guid++,
			i = 0,
			toggler = function( event ) {
				// Figure out which function to execute
				var lastToggle = ( jQuery._data( this, "lastToggle" + fn.guid ) || 0 ) % i;
				jQuery._data( this, "lastToggle" + fn.guid, lastToggle + 1 );

				// Make sure that clicks stop
				event.preventDefault();

				// and execute the function
				return args[ lastToggle ].apply( this, arguments ) || false;
			};

		// link all the functions, so any of them can unbind this click handler
		toggler.guid = guid;
		while ( i < args.length ) {
			args[ i++ ].guid = guid;
		}

		return this.click( toggler );
	},

	hover: function( fnOver, fnOut ) {
		return this.mouseenter( fnOver ).mouseleave( fnOut || fnOver );
	}
});

jQuery.each( ("blur focus focusin focusout load resize scroll unload click dblclick " +
	"mousedown mouseup mousemove mouseover mouseout mouseenter mouseleave " +
	"change select submit keydown keypress keyup error contextmenu").split(" "), function( i, name ) {

	// Handle event binding
	jQuery.fn[ name ] = function( data, fn ) {
		if ( fn == null ) {
			fn = data;
			data = null;
		}

		return arguments.length > 0 ?
			this.on( name, null, data, fn ) :
			this.trigger( name );
	};

	if ( jQuery.attrFn ) {
		jQuery.attrFn[ name ] = true;
	}

	if ( rkeyEvent.test( name ) ) {
		jQuery.event.fixHooks[ name ] = jQuery.event.keyHooks;
	}

	if ( rmouseEvent.test( name ) ) {
		jQuery.event.fixHooks[ name ] = jQuery.event.mouseHooks;
	}
});



/*!
 * Sizzle CSS Selector Engine
 *  Copyright 2011, The Dojo Foundation
 *  Released under the MIT, BSD, and GPL Licenses.
 *  More information: http://sizzlejs.com/
 */
(function(){

var chunker = /((?:\((?:\([^()]+\)|[^()]+)+\)|\[(?:\[[^\[\]]*\]|['"][^'"]*['"]|[^\[\]'"]+)+\]|\\.|[^ >+~,(\[\\]+)+|[>+~])(\s*,\s*)?((?:.|\r|\n)*)/g,
	expando = "sizcache" + (Math.random() + '').replace('.', ''),
	done = 0,
	toString = Object.prototype.toString,
	hasDuplicate = false,
	baseHasDuplicate = true,
	rBackslash = /\\/g,
	rReturn = /\r\n/g,
	rNonWord = /\W/;

// Here we check if the JavaScript engine is using some sort of
// optimization where it does not always call our comparision
// function. If that is the case, discard the hasDuplicate value.
//   Thus far that includes Google Chrome.
[0, 0].sort(function() {
	baseHasDuplicate = false;
	return 0;
});

var Sizzle = function( selector, context, results, seed ) {
	results = results || [];
	context = context || document;

	var origContext = context;

	if ( context.nodeType !== 1 && context.nodeType !== 9 ) {
		return [];
	}

	if ( !selector || typeof selector !== "string" ) {
		return results;
	}

	var m, set, checkSet, extra, ret, cur, pop, i,
		prune = true,
		contextXML = Sizzle.isXML( context ),
		parts = [],
		soFar = selector;

	// Reset the position of the chunker regexp (start from head)
	do {
		chunker.exec( "" );
		m = chunker.exec( soFar );

		if ( m ) {
			soFar = m[3];

			parts.push( m[1] );

			if ( m[2] ) {
				extra = m[3];
				break;
			}
		}
	} while ( m );

	if ( parts.length > 1 && origPOS.exec( selector ) ) {

		if ( parts.length === 2 && Expr.relative[ parts[0] ] ) {
			set = posProcess( parts[0] + parts[1], context, seed );

		} else {
			set = Expr.relative[ parts[0] ] ?
				[ context ] :
				Sizzle( parts.shift(), context );

			while ( parts.length ) {
				selector = parts.shift();

				if ( Expr.relative[ selector ] ) {
					selector += parts.shift();
				}

				set = posProcess( selector, set, seed );
			}
		}

	} else {
		// Take a shortcut and set the context if the root selector is an ID
		// (but not if it'll be faster if the inner selector is an ID)
		if ( !seed && parts.length > 1 && context.nodeType === 9 && !contextXML &&
				Expr.match.ID.test(parts[0]) && !Expr.match.ID.test(parts[parts.length - 1]) ) {

			ret = Sizzle.find( parts.shift(), context, contextXML );
			context = ret.expr ?
				Sizzle.filter( ret.expr, ret.set )[0] :
				ret.set[0];
		}

		if ( context ) {
			ret = seed ?
				{ expr: parts.pop(), set: makeArray(seed) } :
				Sizzle.find( parts.pop(), parts.length === 1 && (parts[0] === "~" || parts[0] === "+") && context.parentNode ? context.parentNode : context, contextXML );

			set = ret.expr ?
				Sizzle.filter( ret.expr, ret.set ) :
				ret.set;

			if ( parts.length > 0 ) {
				checkSet = makeArray( set );

			} else {
				prune = false;
			}

			while ( parts.length ) {
				cur = parts.pop();
				pop = cur;

				if ( !Expr.relative[ cur ] ) {
					cur = "";
				} else {
					pop = parts.pop();
				}

				if ( pop == null ) {
					pop = context;
				}

				Expr.relative[ cur ]( checkSet, pop, contextXML );
			}

		} else {
			checkSet = parts = [];
		}
	}

	if ( !checkSet ) {
		checkSet = set;
	}

	if ( !checkSet ) {
		Sizzle.error( cur || selector );
	}

	if ( toString.call(checkSet) === "[object Array]" ) {
		if ( !prune ) {
			results.push.apply( results, checkSet );

		} else if ( context && context.nodeType === 1 ) {
			for ( i = 0; checkSet[i] != null; i++ ) {
				if ( checkSet[i] && (checkSet[i] === true || checkSet[i].nodeType === 1 && Sizzle.contains(context, checkSet[i])) ) {
					results.push( set[i] );
				}
			}

		} else {
			for ( i = 0; checkSet[i] != null; i++ ) {
				if ( checkSet[i] && checkSet[i].nodeType === 1 ) {
					results.push( set[i] );
				}
			}
		}

	} else {
		makeArray( checkSet, results );
	}

	if ( extra ) {
		Sizzle( extra, origContext, results, seed );
		Sizzle.uniqueSort( results );
	}

	return results;
};

Sizzle.uniqueSort = function( results ) {
	if ( sortOrder ) {
		hasDuplicate = baseHasDuplicate;
		results.sort( sortOrder );

		if ( hasDuplicate ) {
			for ( var i = 1; i < results.length; i++ ) {
				if ( results[i] === results[ i - 1 ] ) {
					results.splice( i--, 1 );
				}
			}
		}
	}

	return results;
};

Sizzle.matches = function( expr, set ) {
	return Sizzle( expr, null, null, set );
};

Sizzle.matchesSelector = function( node, expr ) {
	return Sizzle( expr, null, null, [node] ).length > 0;
};

Sizzle.find = function( expr, context, isXML ) {
	var set, i, len, match, type, left;

	if ( !expr ) {
		return [];
	}

	for ( i = 0, len = Expr.order.length; i < len; i++ ) {
		type = Expr.order[i];

		if ( (match = Expr.leftMatch[ type ].exec( expr )) ) {
			left = match[1];
			match.splice( 1, 1 );

			if ( left.substr( left.length - 1 ) !== "\\" ) {
				match[1] = (match[1] || "").replace( rBackslash, "" );
				set = Expr.find[ type ]( match, context, isXML );

				if ( set != null ) {
					expr = expr.replace( Expr.match[ type ], "" );
					break;
				}
			}
		}
	}

	if ( !set ) {
		set = typeof context.getElementsByTagName !== "undefined" ?
			context.getElementsByTagName( "*" ) :
			[];
	}

	return { set: set, expr: expr };
};

Sizzle.filter = function( expr, set, inplace, not ) {
	var match, anyFound,
		type, found, item, filter, left,
		i, pass,
		old = expr,
		result = [],
		curLoop = set,
		isXMLFilter = set && set[0] && Sizzle.isXML( set[0] );

	while ( expr && set.length ) {
		for ( type in Expr.filter ) {
			if ( (match = Expr.leftMatch[ type ].exec( expr )) != null && match[2] ) {
				filter = Expr.filter[ type ];
				left = match[1];

				anyFound = false;

				match.splice(1,1);

				if ( left.substr( left.length - 1 ) === "\\" ) {
					continue;
				}

				if ( curLoop === result ) {
					result = [];
				}

				if ( Expr.preFilter[ type ] ) {
					match = Expr.preFilter[ type ]( match, curLoop, inplace, result, not, isXMLFilter );

					if ( !match ) {
						anyFound = found = true;

					} else if ( match === true ) {
						continue;
					}
				}

				if ( match ) {
					for ( i = 0; (item = curLoop[i]) != null; i++ ) {
						if ( item ) {
							found = filter( item, match, i, curLoop );
							pass = not ^ found;

							if ( inplace && found != null ) {
								if ( pass ) {
									anyFound = true;

								} else {
									curLoop[i] = false;
								}

							} else if ( pass ) {
								result.push( item );
								anyFound = true;
							}
						}
					}
				}

				if ( found !== undefined ) {
					if ( !inplace ) {
						curLoop = result;
					}

					expr = expr.replace( Expr.match[ type ], "" );

					if ( !anyFound ) {
						return [];
					}

					break;
				}
			}
		}

		// Improper expression
		if ( expr === old ) {
			if ( anyFound == null ) {
				Sizzle.error( expr );

			} else {
				break;
			}
		}

		old = expr;
	}

	return curLoop;
};

Sizzle.error = function( msg ) {
	throw new Error( "Syntax error, unrecognized expression: " + msg );
};

/**
 * Utility function for retreiving the text value of an array of DOM nodes
 * @param {Array|Element} elem
 */
var getText = Sizzle.getText = function( elem ) {
    var i, node,
		nodeType = elem.nodeType,
		ret = "";

	if ( nodeType ) {
		if ( nodeType === 1 || nodeType === 9 || nodeType === 11 ) {
			// Use textContent || innerText for elements
			if ( typeof elem.textContent === 'string' ) {
				return elem.textContent;
			} else if ( typeof elem.innerText === 'string' ) {
				// Replace IE's carriage returns
				return elem.innerText.replace( rReturn, '' );
			} else {
				// Traverse it's children
				for ( elem = elem.firstChild; elem; elem = elem.nextSibling) {
					ret += getText( elem );
				}
			}
		} else if ( nodeType === 3 || nodeType === 4 ) {
			return elem.nodeValue;
		}
	} else {

		// If no nodeType, this is expected to be an array
		for ( i = 0; (node = elem[i]); i++ ) {
			// Do not traverse comment nodes
			if ( node.nodeType !== 8 ) {
				ret += getText( node );
			}
		}
	}
	return ret;
};

var Expr = Sizzle.selectors = {
	order: [ "ID", "NAME", "TAG" ],

	match: {
		ID: /#((?:[\w\u00c0-\uFFFF\-]|\\.)+)/,
		CLASS: /\.((?:[\w\u00c0-\uFFFF\-]|\\.)+)/,
		NAME: /\[name=['"]*((?:[\w\u00c0-\uFFFF\-]|\\.)+)['"]*\]/,
		ATTR: /\[\s*((?:[\w\u00c0-\uFFFF\-]|\\.)+)\s*(?:(\S?=)\s*(?:(['"])(.*?)\3|(#?(?:[\w\u00c0-\uFFFF\-]|\\.)*)|)|)\s*\]/,
		TAG: /^((?:[\w\u00c0-\uFFFF\*\-]|\\.)+)/,
		CHILD: /:(only|nth|last|first)-child(?:\(\s*(even|odd|(?:[+\-]?\d+|(?:[+\-]?\d*)?n\s*(?:[+\-]\s*\d+)?))\s*\))?/,
		POS: /:(nth|eq|gt|lt|first|last|even|odd)(?:\((\d*)\))?(?=[^\-]|$)/,
		PSEUDO: /:((?:[\w\u00c0-\uFFFF\-]|\\.)+)(?:\((['"]?)((?:\([^\)]+\)|[^\(\)]*)+)\2\))?/
	},

	leftMatch: {},

	attrMap: {
		"class": "className",
		"for": "htmlFor"
	},

	attrHandle: {
		href: function( elem ) {
			return elem.getAttribute( "href" );
		},
		type: function( elem ) {
			return elem.getAttribute( "type" );
		}
	},

	relative: {
		"+": function(checkSet, part){
			var isPartStr = typeof part === "string",
				isTag = isPartStr && !rNonWord.test( part ),
				isPartStrNotTag = isPartStr && !isTag;

			if ( isTag ) {
				part = part.toLowerCase();
			}

			for ( var i = 0, l = checkSet.length, elem; i < l; i++ ) {
				if ( (elem = checkSet[i]) ) {
					while ( (elem = elem.previousSibling) && elem.nodeType !== 1 ) {}

					checkSet[i] = isPartStrNotTag || elem && elem.nodeName.toLowerCase() === part ?
						elem || false :
						elem === part;
				}
			}

			if ( isPartStrNotTag ) {
				Sizzle.filter( part, checkSet, true );
			}
		},

		">": function( checkSet, part ) {
			var elem,
				isPartStr = typeof part === "string",
				i = 0,
				l = checkSet.length;

			if ( isPartStr && !rNonWord.test( part ) ) {
				part = part.toLowerCase();

				for ( ; i < l; i++ ) {
					elem = checkSet[i];

					if ( elem ) {
						var parent = elem.parentNode;
						checkSet[i] = parent.nodeName.toLowerCase() === part ? parent : false;
					}
				}

			} else {
				for ( ; i < l; i++ ) {
					elem = checkSet[i];

					if ( elem ) {
						checkSet[i] = isPartStr ?
							elem.parentNode :
							elem.parentNode === part;
					}
				}

				if ( isPartStr ) {
					Sizzle.filter( part, checkSet, true );
				}
			}
		},

		"": function(checkSet, part, isXML){
			var nodeCheck,
				doneName = done++,
				checkFn = dirCheck;

			if ( typeof part === "string" && !rNonWord.test( part ) ) {
				part = part.toLowerCase();
				nodeCheck = part;
				checkFn = dirNodeCheck;
			}

			checkFn( "parentNode", part, doneName, checkSet, nodeCheck, isXML );
		},

		"~": function( checkSet, part, isXML ) {
			var nodeCheck,
				doneName = done++,
				checkFn = dirCheck;

			if ( typeof part === "string" && !rNonWord.test( part ) ) {
				part = part.toLowerCase();
				nodeCheck = part;
				checkFn = dirNodeCheck;
			}

			checkFn( "previousSibling", part, doneName, checkSet, nodeCheck, isXML );
		}
	},

	find: {
		ID: function( match, context, isXML ) {
			if ( typeof context.getElementById !== "undefined" && !isXML ) {
				var m = context.getElementById(match[1]);
				// Check parentNode to catch when Blackberry 4.6 returns
				// nodes that are no longer in the document #6963
				return m && m.parentNode ? [m] : [];
			}
		},

		NAME: function( match, context ) {
			if ( typeof context.getElementsByName !== "undefined" ) {
				var ret = [],
					results = context.getElementsByName( match[1] );

				for ( var i = 0, l = results.length; i < l; i++ ) {
					if ( results[i].getAttribute("name") === match[1] ) {
						ret.push( results[i] );
					}
				}

				return ret.length === 0 ? null : ret;
			}
		},

		TAG: function( match, context ) {
			if ( typeof context.getElementsByTagName !== "undefined" ) {
				return context.getElementsByTagName( match[1] );
			}
		}
	},
	preFilter: {
		CLASS: function( match, curLoop, inplace, result, not, isXML ) {
			match = " " + match[1].replace( rBackslash, "" ) + " ";

			if ( isXML ) {
				return match;
			}

			for ( var i = 0, elem; (elem = curLoop[i]) != null; i++ ) {
				if ( elem ) {
					if ( not ^ (elem.className && (" " + elem.className + " ").replace(/[\t\n\r]/g, " ").indexOf(match) >= 0) ) {
						if ( !inplace ) {
							result.push( elem );
						}

					} else if ( inplace ) {
						curLoop[i] = false;
					}
				}
			}

			return false;
		},

		ID: function( match ) {
			return match[1].replace( rBackslash, "" );
		},

		TAG: function( match, curLoop ) {
			return match[1].replace( rBackslash, "" ).toLowerCase();
		},

		CHILD: function( match ) {
			if ( match[1] === "nth" ) {
				if ( !match[2] ) {
					Sizzle.error( match[0] );
				}

				match[2] = match[2].replace(/^\+|\s*/g, '');

				// parse equations like 'even', 'odd', '5', '2n', '3n+2', '4n-1', '-n+6'
				var test = /(-?)(\d*)(?:n([+\-]?\d*))?/.exec(
					match[2] === "even" && "2n" || match[2] === "odd" && "2n+1" ||
					!/\D/.test( match[2] ) && "0n+" + match[2] || match[2]);

				// calculate the numbers (first)n+(last) including if they are negative
				match[2] = (test[1] + (test[2] || 1)) - 0;
				match[3] = test[3] - 0;
			}
			else if ( match[2] ) {
				Sizzle.error( match[0] );
			}

			// TODO: Move to normal caching system
			match[0] = done++;

			return match;
		},

		ATTR: function( match, curLoop, inplace, result, not, isXML ) {
			var name = match[1] = match[1].replace( rBackslash, "" );

			if ( !isXML && Expr.attrMap[name] ) {
				match[1] = Expr.attrMap[name];
			}

			// Handle if an un-quoted value was used
			match[4] = ( match[4] || match[5] || "" ).replace( rBackslash, "" );

			if ( match[2] === "~=" ) {
				match[4] = " " + match[4] + " ";
			}

			return match;
		},

		PSEUDO: function( match, curLoop, inplace, result, not ) {
			if ( match[1] === "not" ) {
				// If we're dealing with a complex expression, or a simple one
				if ( ( chunker.exec(match[3]) || "" ).length > 1 || /^\w/.test(match[3]) ) {
					match[3] = Sizzle(match[3], null, null, curLoop);

				} else {
					var ret = Sizzle.filter(match[3], curLoop, inplace, true ^ not);

					if ( !inplace ) {
						result.push.apply( result, ret );
					}

					return false;
				}

			} else if ( Expr.match.POS.test( match[0] ) || Expr.match.CHILD.test( match[0] ) ) {
				return true;
			}

			return match;
		},

		POS: function( match ) {
			match.unshift( true );

			return match;
		}
	},

	filters: {
		enabled: function( elem ) {
			return elem.disabled === false && elem.type !== "hidden";
		},

		disabled: function( elem ) {
			return elem.disabled === true;
		},

		checked: function( elem ) {
			return elem.checked === true;
		},

		selected: function( elem ) {
			// Accessing this property makes selected-by-default
			// options in Safari work properly
			if ( elem.parentNode ) {
				elem.parentNode.selectedIndex;
			}

			return elem.selected === true;
		},

		parent: function( elem ) {
			return !!elem.firstChild;
		},

		empty: function( elem ) {
			return !elem.firstChild;
		},

		has: function( elem, i, match ) {
			return !!Sizzle( match[3], elem ).length;
		},

		header: function( elem ) {
			return (/h\d/i).test( elem.nodeName );
		},

		text: function( elem ) {
			var attr = elem.getAttribute( "type" ), type = elem.type;
			// IE6 and 7 will map elem.type to 'text' for new HTML5 types (search, etc)
			// use getAttribute instead to test this case
			return elem.nodeName.toLowerCase() === "input" && "text" === type && ( attr === type || attr === null );
		},

		radio: function( elem ) {
			return elem.nodeName.toLowerCase() === "input" && "radio" === elem.type;
		},

		checkbox: function( elem ) {
			return elem.nodeName.toLowerCase() === "input" && "checkbox" === elem.type;
		},

		file: function( elem ) {
			return elem.nodeName.toLowerCase() === "input" && "file" === elem.type;
		},

		password: function( elem ) {
			return elem.nodeName.toLowerCase() === "input" && "password" === elem.type;
		},

		submit: function( elem ) {
			var name = elem.nodeName.toLowerCase();
			return (name === "input" || name === "button") && "submit" === elem.type;
		},

		image: function( elem ) {
			return elem.nodeName.toLowerCase() === "input" && "image" === elem.type;
		},

		reset: function( elem ) {
			var name = elem.nodeName.toLowerCase();
			return (name === "input" || name === "button") && "reset" === elem.type;
		},

		button: function( elem ) {
			var name = elem.nodeName.toLowerCase();
			return name === "input" && "button" === elem.type || name === "button";
		},

		input: function( elem ) {
			return (/input|select|textarea|button/i).test( elem.nodeName );
		},

		focus: function( elem ) {
			return elem === elem.ownerDocument.activeElement;
		}
	},
	setFilters: {
		first: function( elem, i ) {
			return i === 0;
		},

		last: function( elem, i, match, array ) {
			return i === array.length - 1;
		},

		even: function( elem, i ) {
			return i % 2 === 0;
		},

		odd: function( elem, i ) {
			return i % 2 === 1;
		},

		lt: function( elem, i, match ) {
			return i < match[3] - 0;
		},

		gt: function( elem, i, match ) {
			return i > match[3] - 0;
		},

		nth: function( elem, i, match ) {
			return match[3] - 0 === i;
		},

		eq: function( elem, i, match ) {
			return match[3] - 0 === i;
		}
	},
	filter: {
		PSEUDO: function( elem, match, i, array ) {
			var name = match[1],
				filter = Expr.filters[ name ];

			if ( filter ) {
				return filter( elem, i, match, array );

			} else if ( name === "contains" ) {
				return (elem.textContent || elem.innerText || getText([ elem ]) || "").indexOf(match[3]) >= 0;

			} else if ( name === "not" ) {
				var not = match[3];

				for ( var j = 0, l = not.length; j < l; j++ ) {
					if ( not[j] === elem ) {
						return false;
					}
				}

				return true;

			} else {
				Sizzle.error( name );
			}
		},

		CHILD: function( elem, match ) {
			var first, last,
				doneName, parent, cache,
				count, diff,
				type = match[1],
				node = elem;

			switch ( type ) {
				case "only":
				case "first":
					while ( (node = node.previousSibling) ) {
						if ( node.nodeType === 1 ) {
							return false;
						}
					}

					if ( type === "first" ) {
						return true;
					}

					node = elem;

					/* falls through */
				case "last":
					while ( (node = node.nextSibling) ) {
						if ( node.nodeType === 1 ) {
							return false;
						}
					}

					return true;

				case "nth":
					first = match[2];
					last = match[3];

					if ( first === 1 && last === 0 ) {
						return true;
					}

					doneName = match[0];
					parent = elem.parentNode;

					if ( parent && (parent[ expando ] !== doneName || !elem.nodeIndex) ) {
						count = 0;

						for ( node = parent.firstChild; node; node = node.nextSibling ) {
							if ( node.nodeType === 1 ) {
								node.nodeIndex = ++count;
							}
						}

						parent[ expando ] = doneName;
					}

					diff = elem.nodeIndex - last;

					if ( first === 0 ) {
						return diff === 0;

					} else {
						return ( diff % first === 0 && diff / first >= 0 );
					}
			}
		},

		ID: function( elem, match ) {
			return elem.nodeType === 1 && elem.getAttribute("id") === match;
		},

		TAG: function( elem, match ) {
			return (match === "*" && elem.nodeType === 1) || !!elem.nodeName && elem.nodeName.toLowerCase() === match;
		},

		CLASS: function( elem, match ) {
			return (" " + (elem.className || elem.getAttribute("class")) + " ")
				.indexOf( match ) > -1;
		},

		ATTR: function( elem, match ) {
			var name = match[1],
				result = Sizzle.attr ?
					Sizzle.attr( elem, name ) :
					Expr.attrHandle[ name ] ?
					Expr.attrHandle[ name ]( elem ) :
					elem[ name ] != null ?
						elem[ name ] :
						elem.getAttribute( name ),
				value = result + "",
				type = match[2],
				check = match[4];

			return result == null ?
				type === "!=" :
				!type && Sizzle.attr ?
				result != null :
				type === "=" ?
				value === check :
				type === "*=" ?
				value.indexOf(check) >= 0 :
				type === "~=" ?
				(" " + value + " ").indexOf(check) >= 0 :
				!check ?
				value && result !== false :
				type === "!=" ?
				value !== check :
				type === "^=" ?
				value.indexOf(check) === 0 :
				type === "$=" ?
				value.substr(value.length - check.length) === check :
				type === "|=" ?
				value === check || value.substr(0, check.length + 1) === check + "-" :
				false;
		},

		POS: function( elem, match, i, array ) {
			var name = match[2],
				filter = Expr.setFilters[ name ];

			if ( filter ) {
				return filter( elem, i, match, array );
			}
		}
	}
};

var origPOS = Expr.match.POS,
	fescape = function(all, num){
		return "\\" + (num - 0 + 1);
	};

for ( var type in Expr.match ) {
	Expr.match[ type ] = new RegExp( Expr.match[ type ].source + (/(?![^\[]*\])(?![^\(]*\))/.source) );
	Expr.leftMatch[ type ] = new RegExp( /(^(?:.|\r|\n)*?)/.source + Expr.match[ type ].source.replace(/\\(\d+)/g, fescape) );
}
// Expose origPOS
// "global" as in regardless of relation to brackets/parens
Expr.match.globalPOS = origPOS;

var makeArray = function( array, results ) {
	array = Array.prototype.slice.call( array, 0 );

	if ( results ) {
		results.push.apply( results, array );
		return results;
	}

	return array;
};

// Perform a simple check to determine if the browser is capable of
// converting a NodeList to an array using builtin methods.
// Also verifies that the returned array holds DOM nodes
// (which is not the case in the Blackberry browser)
try {
	Array.prototype.slice.call( document.documentElement.childNodes, 0 )[0].nodeType;

// Provide a fallback method if it does not work
} catch( e ) {
	makeArray = function( array, results ) {
		var i = 0,
			ret = results || [];

		if ( toString.call(array) === "[object Array]" ) {
			Array.prototype.push.apply( ret, array );

		} else {
			if ( typeof array.length === "number" ) {
				for ( var l = array.length; i < l; i++ ) {
					ret.push( array[i] );
				}

			} else {
				for ( ; array[i]; i++ ) {
					ret.push( array[i] );
				}
			}
		}

		return ret;
	};
}

var sortOrder, siblingCheck;

if ( document.documentElement.compareDocumentPosition ) {
	sortOrder = function( a, b ) {
		if ( a === b ) {
			hasDuplicate = true;
			return 0;
		}

		if ( !a.compareDocumentPosition || !b.compareDocumentPosition ) {
			return a.compareDocumentPosition ? -1 : 1;
		}

		return a.compareDocumentPosition(b) & 4 ? -1 : 1;
	};

} else {
	sortOrder = function( a, b ) {
		// The nodes are identical, we can exit early
		if ( a === b ) {
			hasDuplicate = true;
			return 0;

		// Fallback to using sourceIndex (in IE) if it's available on both nodes
		} else if ( a.sourceIndex && b.sourceIndex ) {
			return a.sourceIndex - b.sourceIndex;
		}

		var al, bl,
			ap = [],
			bp = [],
			aup = a.parentNode,
			bup = b.parentNode,
			cur = aup;

		// If the nodes are siblings (or identical) we can do a quick check
		if ( aup === bup ) {
			return siblingCheck( a, b );

		// If no parents were found then the nodes are disconnected
		} else if ( !aup ) {
			return -1;

		} else if ( !bup ) {
			return 1;
		}

		// Otherwise they're somewhere else in the tree so we need
		// to build up a full list of the parentNodes for comparison
		while ( cur ) {
			ap.unshift( cur );
			cur = cur.parentNode;
		}

		cur = bup;

		while ( cur ) {
			bp.unshift( cur );
			cur = cur.parentNode;
		}

		al = ap.length;
		bl = bp.length;

		// Start walking down the tree looking for a discrepancy
		for ( var i = 0; i < al && i < bl; i++ ) {
			if ( ap[i] !== bp[i] ) {
				return siblingCheck( ap[i], bp[i] );
			}
		}

		// We ended someplace up the tree so do a sibling check
		return i === al ?
			siblingCheck( a, bp[i], -1 ) :
			siblingCheck( ap[i], b, 1 );
	};

	siblingCheck = function( a, b, ret ) {
		if ( a === b ) {
			return ret;
		}

		var cur = a.nextSibling;

		while ( cur ) {
			if ( cur === b ) {
				return -1;
			}

			cur = cur.nextSibling;
		}

		return 1;
	};
}

// Check to see if the browser returns elements by name when
// querying by getElementById (and provide a workaround)
(function(){
	// We're going to inject a fake input element with a specified name
	var form = document.createElement("div"),
		id = "script" + (new Date()).getTime(),
		root = document.documentElement;

	form.innerHTML = "<a name='" + id + "'/>";

	// Inject it into the root element, check its status, and remove it quickly
	root.insertBefore( form, root.firstChild );

	// The workaround has to do additional checks after a getElementById
	// Which slows things down for other browsers (hence the branching)
	if ( document.getElementById( id ) ) {
		Expr.find.ID = function( match, context, isXML ) {
			if ( typeof context.getElementById !== "undefined" && !isXML ) {
				var m = context.getElementById(match[1]);

				return m ?
					m.id === match[1] || typeof m.getAttributeNode !== "undefined" && m.getAttributeNode("id").nodeValue === match[1] ?
						[m] :
						undefined :
					[];
			}
		};

		Expr.filter.ID = function( elem, match ) {
			var node = typeof elem.getAttributeNode !== "undefined" && elem.getAttributeNode("id");

			return elem.nodeType === 1 && node && node.nodeValue === match;
		};
	}

	root.removeChild( form );

	// release memory in IE
	root = form = null;
})();

(function(){
	// Check to see if the browser returns only elements
	// when doing getElementsByTagName("*")

	// Create a fake element
	var div = document.createElement("div");
	div.appendChild( document.createComment("") );

	// Make sure no comments are found
	if ( div.getElementsByTagName("*").length > 0 ) {
		Expr.find.TAG = function( match, context ) {
			var results = context.getElementsByTagName( match[1] );

			// Filter out possible comments
			if ( match[1] === "*" ) {
				var tmp = [];

				for ( var i = 0; results[i]; i++ ) {
					if ( results[i].nodeType === 1 ) {
						tmp.push( results[i] );
					}
				}

				results = tmp;
			}

			return results;
		};
	}

	// Check to see if an attribute returns normalized href attributes
	div.innerHTML = "<a href='#'></a>";

	if ( div.firstChild && typeof div.firstChild.getAttribute !== "undefined" &&
			div.firstChild.getAttribute("href") !== "#" ) {

		Expr.attrHandle.href = function( elem ) {
			return elem.getAttribute( "href", 2 );
		};
	}

	// release memory in IE
	div = null;
})();

if ( document.querySelectorAll ) {
	(function(){
		var oldSizzle = Sizzle,
			div = document.createElement("div"),
			id = "__sizzle__";

		div.innerHTML = "<p class='TEST'></p>";

		// Safari can't handle uppercase or unicode characters when
		// in quirks mode.
		if ( div.querySelectorAll && div.querySelectorAll(".TEST").length === 0 ) {
			return;
		}

		Sizzle = function( query, context, extra, seed ) {
			context = context || document;

			// Only use querySelectorAll on non-XML documents
			// (ID selectors don't work in non-HTML documents)
			if ( !seed && !Sizzle.isXML(context) ) {
				// See if we find a selector to speed up
				var match = /^(\w+$)|^\.([\w\-]+$)|^#([\w\-]+$)/.exec( query );

				if ( match && (context.nodeType === 1 || context.nodeType === 9) ) {
					// Speed-up: Sizzle("TAG")
					if ( match[1] ) {
						return makeArray( context.getElementsByTagName( query ), extra );

					// Speed-up: Sizzle(".CLASS")
					} else if ( match[2] && Expr.find.CLASS && context.getElementsByClassName ) {
						return makeArray( context.getElementsByClassName( match[2] ), extra );
					}
				}

				if ( context.nodeType === 9 ) {
					// Speed-up: Sizzle("body")
					// The body element only exists once, optimize finding it
					if ( query === "body" && context.body ) {
						return makeArray( [ context.body ], extra );

					// Speed-up: Sizzle("#ID")
					} else if ( match && match[3] ) {
						var elem = context.getElementById( match[3] );

						// Check parentNode to catch when Blackberry 4.6 returns
						// nodes that are no longer in the document #6963
						if ( elem && elem.parentNode ) {
							// Handle the case where IE and Opera return items
							// by name instead of ID
							if ( elem.id === match[3] ) {
								return makeArray( [ elem ], extra );
							}

						} else {
							return makeArray( [], extra );
						}
					}

					try {
						return makeArray( context.querySelectorAll(query), extra );
					} catch(qsaError) {}

				// qSA works strangely on Element-rooted queries
				// We can work around this by specifying an extra ID on the root
				// and working up from there (Thanks to Andrew Dupont for the technique)
				// IE 8 doesn't work on object elements
				} else if ( context.nodeType === 1 && context.nodeName.toLowerCase() !== "object" ) {
					var oldContext = context,
						old = context.getAttribute( "id" ),
						nid = old || id,
						hasParent = context.parentNode,
						relativeHierarchySelector = /^\s*[+~]/.test( query );

					if ( !old ) {
						context.setAttribute( "id", nid );
					} else {
						nid = nid.replace( /'/g, "\\$&" );
					}
					if ( relativeHierarchySelector && hasParent ) {
						context = context.parentNode;
					}

					try {
						if ( !relativeHierarchySelector || hasParent ) {
							return makeArray( context.querySelectorAll( "[id='" + nid + "'] " + query ), extra );
						}

					} catch(pseudoError) {
					} finally {
						if ( !old ) {
							oldContext.removeAttribute( "id" );
						}
					}
				}
			}

			return oldSizzle(query, context, extra, seed);
		};

		for ( var prop in oldSizzle ) {
			Sizzle[ prop ] = oldSizzle[ prop ];
		}

		// release memory in IE
		div = null;
	})();
}

(function(){
	var html = document.documentElement,
		matches = html.matchesSelector || html.mozMatchesSelector || html.webkitMatchesSelector || html.msMatchesSelector;

	if ( matches ) {
		// Check to see if it's possible to do matchesSelector
		// on a disconnected node (IE 9 fails this)
		var disconnectedMatch = !matches.call( document.createElement( "div" ), "div" ),
			pseudoWorks = false;

		try {
			// This should fail with an exception
			// Gecko does not error, returns false instead
			matches.call( document.documentElement, "[test!='']:sizzle" );

		} catch( pseudoError ) {
			pseudoWorks = true;
		}

		Sizzle.matchesSelector = function( node, expr ) {
			// Make sure that attribute selectors are quoted
			expr = expr.replace(/\=\s*([^'"\]]*)\s*\]/g, "='$1']");

			if ( !Sizzle.isXML( node ) ) {
				try {
					if ( pseudoWorks || !Expr.match.PSEUDO.test( expr ) && !/!=/.test( expr ) ) {
						var ret = matches.call( node, expr );

						// IE 9's matchesSelector returns false on disconnected nodes
						if ( ret || !disconnectedMatch ||
								// As well, disconnected nodes are said to be in a document
								// fragment in IE 9, so check for that
								node.document && node.document.nodeType !== 11 ) {
							return ret;
						}
					}
				} catch(e) {}
			}

			return Sizzle(expr, null, null, [node]).length > 0;
		};
	}
})();

(function(){
	var div = document.createElement("div");

	div.innerHTML = "<div class='test e'></div><div class='test'></div>";

	// Opera can't find a second classname (in 9.6)
	// Also, make sure that getElementsByClassName actually exists
	if ( !div.getElementsByClassName || div.getElementsByClassName("e").length === 0 ) {
		return;
	}

	// Safari caches class attributes, doesn't catch changes (in 3.2)
	div.lastChild.className = "e";

	if ( div.getElementsByClassName("e").length === 1 ) {
		return;
	}

	Expr.order.splice(1, 0, "CLASS");
	Expr.find.CLASS = function( match, context, isXML ) {
		if ( typeof context.getElementsByClassName !== "undefined" && !isXML ) {
			return context.getElementsByClassName(match[1]);
		}
	};

	// release memory in IE
	div = null;
})();

function dirNodeCheck( dir, cur, doneName, checkSet, nodeCheck, isXML ) {
	for ( var i = 0, l = checkSet.length; i < l; i++ ) {
		var elem = checkSet[i];

		if ( elem ) {
			var match = false;

			elem = elem[dir];

			while ( elem ) {
				if ( elem[ expando ] === doneName ) {
					match = checkSet[elem.sizset];
					break;
				}

				if ( elem.nodeType === 1 && !isXML ){
					elem[ expando ] = doneName;
					elem.sizset = i;
				}

				if ( elem.nodeName.toLowerCase() === cur ) {
					match = elem;
					break;
				}

				elem = elem[dir];
			}

			checkSet[i] = match;
		}
	}
}

function dirCheck( dir, cur, doneName, checkSet, nodeCheck, isXML ) {
	for ( var i = 0, l = checkSet.length; i < l; i++ ) {
		var elem = checkSet[i];

		if ( elem ) {
			var match = false;

			elem = elem[dir];

			while ( elem ) {
				if ( elem[ expando ] === doneName ) {
					match = checkSet[elem.sizset];
					break;
				}

				if ( elem.nodeType === 1 ) {
					if ( !isXML ) {
						elem[ expando ] = doneName;
						elem.sizset = i;
					}

					if ( typeof cur !== "string" ) {
						if ( elem === cur ) {
							match = true;
							break;
						}

					} else if ( Sizzle.filter( cur, [elem] ).length > 0 ) {
						match = elem;
						break;
					}
				}

				elem = elem[dir];
			}

			checkSet[i] = match;
		}
	}
}

if ( document.documentElement.contains ) {
	Sizzle.contains = function( a, b ) {
		return a !== b && (a.contains ? a.contains(b) : true);
	};

} else if ( document.documentElement.compareDocumentPosition ) {
	Sizzle.contains = function( a, b ) {
		return !!(a.compareDocumentPosition(b) & 16);
	};

} else {
	Sizzle.contains = function() {
		return false;
	};
}

Sizzle.isXML = function( elem ) {
	// documentElement is verified for cases where it doesn't yet exist
	// (such as loading iframes in IE - #4833)
	var documentElement = (elem ? elem.ownerDocument || elem : 0).documentElement;

	return documentElement ? documentElement.nodeName !== "HTML" : false;
};

var posProcess = function( selector, context, seed ) {
	var match,
		tmpSet = [],
		later = "",
		root = context.nodeType ? [context] : context;

	// Position selectors must be done after the filter
	// And so must :not(positional) so we move all PSEUDOs to the end
	while ( (match = Expr.match.PSEUDO.exec( selector )) ) {
		later += match[0];
		selector = selector.replace( Expr.match.PSEUDO, "" );
	}

	selector = Expr.relative[selector] ? selector + "*" : selector;

	for ( var i = 0, l = root.length; i < l; i++ ) {
		Sizzle( selector, root[i], tmpSet, seed );
	}

	return Sizzle.filter( later, tmpSet );
};

// EXPOSE
// Override sizzle attribute retrieval
Sizzle.attr = jQuery.attr;
Sizzle.selectors.attrMap = {};
jQuery.find = Sizzle;
jQuery.expr = Sizzle.selectors;
jQuery.expr[":"] = jQuery.expr.filters;
jQuery.unique = Sizzle.uniqueSort;
jQuery.text = Sizzle.getText;
jQuery.isXMLDoc = Sizzle.isXML;
jQuery.contains = Sizzle.contains;


})();


var runtil = /Until$/,
	rparentsprev = /^(?:parents|prevUntil|prevAll)/,
	// Note: This RegExp should be improved, or likely pulled from Sizzle
	rmultiselector = /,/,
	isSimple = /^.[^:#\[\.,]*$/,
	slice = Array.prototype.slice,
	POS = jQuery.expr.match.globalPOS,
	// methods guaranteed to produce a unique set when starting from a unique set
	guaranteedUnique = {
		children: true,
		contents: true,
		next: true,
		prev: true
	};

jQuery.fn.extend({
	find: function( selector ) {
		var self = this,
			i, l;

		if ( typeof selector !== "string" ) {
			return jQuery( selector ).filter(function() {
				for ( i = 0, l = self.length; i < l; i++ ) {
					if ( jQuery.contains( self[ i ], this ) ) {
						return true;
					}
				}
			});
		}

		var ret = this.pushStack( "", "find", selector ),
			length, n, r;

		for ( i = 0, l = this.length; i < l; i++ ) {
			length = ret.length;
			jQuery.find( selector, this[i], ret );

			if ( i > 0 ) {
				// Make sure that the results are unique
				for ( n = length; n < ret.length; n++ ) {
					for ( r = 0; r < length; r++ ) {
						if ( ret[r] === ret[n] ) {
							ret.splice(n--, 1);
							break;
						}
					}
				}
			}
		}

		return ret;
	},

	has: function( target ) {
		var targets = jQuery( target );
		return this.filter(function() {
			for ( var i = 0, l = targets.length; i < l; i++ ) {
				if ( jQuery.contains( this, targets[i] ) ) {
					return true;
				}
			}
		});
	},

	not: function( selector ) {
		return this.pushStack( winnow(this, selector, false), "not", selector);
	},

	filter: function( selector ) {
		return this.pushStack( winnow(this, selector, true), "filter", selector );
	},

	is: function( selector ) {
		return !!selector && (
			typeof selector === "string" ?
				// If this is a positional selector, check membership in the returned set
				// so $("p:first").is("p:last") won't return true for a doc with two "p".
				POS.test( selector ) ?
					jQuery( selector, this.context ).index( this[0] ) >= 0 :
					jQuery.filter( selector, this ).length > 0 :
				this.filter( selector ).length > 0 );
	},

	closest: function( selectors, context ) {
		var ret = [], i, l, cur = this[0];

		// Array (deprecated as of jQuery 1.7)
		if ( jQuery.isArray( selectors ) ) {
			var level = 1;

			while ( cur && cur.ownerDocument && cur !== context ) {
				for ( i = 0; i < selectors.length; i++ ) {

					if ( jQuery( cur ).is( selectors[ i ] ) ) {
						ret.push({ selector: selectors[ i ], elem: cur, level: level });
					}
				}

				cur = cur.parentNode;
				level++;
			}

			return ret;
		}

		// String
		var pos = POS.test( selectors ) || typeof selectors !== "string" ?
				jQuery( selectors, context || this.context ) :
				0;

		for ( i = 0, l = this.length; i < l; i++ ) {
			cur = this[i];

			while ( cur ) {
				if ( pos ? pos.index(cur) > -1 : jQuery.find.matchesSelector(cur, selectors) ) {
					ret.push( cur );
					break;

				} else {
					cur = cur.parentNode;
					if ( !cur || !cur.ownerDocument || cur === context || cur.nodeType === 11 ) {
						break;
					}
				}
			}
		}

		ret = ret.length > 1 ? jQuery.unique( ret ) : ret;

		return this.pushStack( ret, "closest", selectors );
	},

	// Determine the position of an element within
	// the matched set of elements
	index: function( elem ) {

		// No argument, return index in parent
		if ( !elem ) {
			return ( this[0] && this[0].parentNode ) ? this.prevAll().length : -1;
		}

		// index in selector
		if ( typeof elem === "string" ) {
			return jQuery.inArray( this[0], jQuery( elem ) );
		}

		// Locate the position of the desired element
		return jQuery.inArray(
			// If it receives a jQuery object, the first element is used
			elem.jquery ? elem[0] : elem, this );
	},

	add: function( selector, context ) {
		var set = typeof selector === "string" ?
				jQuery( selector, context ) :
				jQuery.makeArray( selector && selector.nodeType ? [ selector ] : selector ),
			all = jQuery.merge( this.get(), set );

		return this.pushStack( isDisconnected( set[0] ) || isDisconnected( all[0] ) ?
			all :
			jQuery.unique( all ) );
	},

	andSelf: function() {
		return this.add( this.prevObject );
	}
});

// A painfully simple check to see if an element is disconnected
// from a document (should be improved, where feasible).
function isDisconnected( node ) {
	return !node || !node.parentNode || node.parentNode.nodeType === 11;
}

jQuery.each({
	parent: function( elem ) {
		var parent = elem.parentNode;
		return parent && parent.nodeType !== 11 ? parent : null;
	},
	parents: function( elem ) {
		return jQuery.dir( elem, "parentNode" );
	},
	parentsUntil: function( elem, i, until ) {
		return jQuery.dir( elem, "parentNode", until );
	},
	next: function( elem ) {
		return jQuery.nth( elem, 2, "nextSibling" );
	},
	prev: function( elem ) {
		return jQuery.nth( elem, 2, "previousSibling" );
	},
	nextAll: function( elem ) {
		return jQuery.dir( elem, "nextSibling" );
	},
	prevAll: function( elem ) {
		return jQuery.dir( elem, "previousSibling" );
	},
	nextUntil: function( elem, i, until ) {
		return jQuery.dir( elem, "nextSibling", until );
	},
	prevUntil: function( elem, i, until ) {
		return jQuery.dir( elem, "previousSibling", until );
	},
	siblings: function( elem ) {
		return jQuery.sibling( ( elem.parentNode || {} ).firstChild, elem );
	},
	children: function( elem ) {
		return jQuery.sibling( elem.firstChild );
	},
	contents: function( elem ) {
		return jQuery.nodeName( elem, "iframe" ) ?
			elem.contentDocument || elem.contentWindow.document :
			jQuery.makeArray( elem.childNodes );
	}
}, function( name, fn ) {
	jQuery.fn[ name ] = function( until, selector ) {
		var ret = jQuery.map( this, fn, until );

		if ( !runtil.test( name ) ) {
			selector = until;
		}

		if ( selector && typeof selector === "string" ) {
			ret = jQuery.filter( selector, ret );
		}

		ret = this.length > 1 && !guaranteedUnique[ name ] ? jQuery.unique( ret ) : ret;

		if ( (this.length > 1 || rmultiselector.test( selector )) && rparentsprev.test( name ) ) {
			ret = ret.reverse();
		}

		return this.pushStack( ret, name, slice.call( arguments ).join(",") );
	};
});

jQuery.extend({
	filter: function( expr, elems, not ) {
		if ( not ) {
			expr = ":not(" + expr + ")";
		}

		return elems.length === 1 ?
			jQuery.find.matchesSelector(elems[0], expr) ? [ elems[0] ] : [] :
			jQuery.find.matches(expr, elems);
	},

	dir: function( elem, dir, until ) {
		var matched = [],
			cur = elem[ dir ];

		while ( cur && cur.nodeType !== 9 && (until === undefined || cur.nodeType !== 1 || !jQuery( cur ).is( until )) ) {
			if ( cur.nodeType === 1 ) {
				matched.push( cur );
			}
			cur = cur[dir];
		}
		return matched;
	},

	nth: function( cur, result, dir, elem ) {
		result = result || 1;
		var num = 0;

		for ( ; cur; cur = cur[dir] ) {
			if ( cur.nodeType === 1 && ++num === result ) {
				break;
			}
		}

		return cur;
	},

	sibling: function( n, elem ) {
		var r = [];

		for ( ; n; n = n.nextSibling ) {
			if ( n.nodeType === 1 && n !== elem ) {
				r.push( n );
			}
		}

		return r;
	}
});

// Implement the identical functionality for filter and not
function winnow( elements, qualifier, keep ) {

	// Can't pass null or undefined to indexOf in Firefox 4
	// Set to 0 to skip string check
	qualifier = qualifier || 0;

	if ( jQuery.isFunction( qualifier ) ) {
		return jQuery.grep(elements, function( elem, i ) {
			var retVal = !!qualifier.call( elem, i, elem );
			return retVal === keep;
		});

	} else if ( qualifier.nodeType ) {
		return jQuery.grep(elements, function( elem, i ) {
			return ( elem === qualifier ) === keep;
		});

	} else if ( typeof qualifier === "string" ) {
		var filtered = jQuery.grep(elements, function( elem ) {
			return elem.nodeType === 1;
		});

		if ( isSimple.test( qualifier ) ) {
			return jQuery.filter(qualifier, filtered, !keep);
		} else {
			qualifier = jQuery.filter( qualifier, filtered );
		}
	}

	return jQuery.grep(elements, function( elem, i ) {
		return ( jQuery.inArray( elem, qualifier ) >= 0 ) === keep;
	});
}




function createSafeFragment( document ) {
	var list = nodeNames.split( "|" ),
	safeFrag = document.createDocumentFragment();

	if ( safeFrag.createElement ) {
		while ( list.length ) {
			safeFrag.createElement(
				list.pop()
			);
		}
	}
	return safeFrag;
}

var nodeNames = "abbr|article|aside|audio|bdi|canvas|data|datalist|details|figcaption|figure|footer|" +
		"header|hgroup|mark|meter|nav|output|progress|section|summary|time|video",
	rinlinejQuery = / jQuery\d+="(?:\d+|null)"/g,
	rleadingWhitespace = /^\s+/,
	rxhtmlTag = /<(?!area|br|col|embed|hr|img|input|link|meta|param)(([\w:]+)[^>]*)\/>/ig,
	rtagName = /<([\w:]+)/,
	rtbody = /<tbody/i,
	rhtml = /<|&#?\w+;/,
	rnoInnerhtml = /<(?:script|style)/i,
	rnocache = /<(?:script|object|embed|option|style)/i,
	rnoshimcache = new RegExp("<(?:" + nodeNames + ")[\\s/>]", "i"),
	// checked="checked" or checked
	rchecked = /checked\s*(?:[^=]|=\s*.checked.)/i,
	rscriptType = /\/(java|ecma)script/i,
	rcleanScript = /^\s*<!(?:\[CDATA\[|\-\-)/,
	wrapMap = {
		option: [ 1, "<select multiple='multiple'>", "</select>" ],
		legend: [ 1, "<fieldset>", "</fieldset>" ],
		thead: [ 1, "<table>", "</table>" ],
		tr: [ 2, "<table><tbody>", "</tbody></table>" ],
		td: [ 3, "<table><tbody><tr>", "</tr></tbody></table>" ],
		col: [ 2, "<table><tbody></tbody><colgroup>", "</colgroup></table>" ],
		area: [ 1, "<map>", "</map>" ],
		_default: [ 0, "", "" ]
	},
	safeFragment = createSafeFragment( document );

wrapMap.optgroup = wrapMap.option;
wrapMap.tbody = wrapMap.tfoot = wrapMap.colgroup = wrapMap.caption = wrapMap.thead;
wrapMap.th = wrapMap.td;

// IE can't serialize <link> and <script> tags normally
if ( !jQuery.support.htmlSerialize ) {
	wrapMap._default = [ 1, "div<div>", "</div>" ];
}

jQuery.fn.extend({
	text: function( value ) {
		return jQuery.access( this, function( value ) {
			return value === undefined ?
				jQuery.text( this ) :
				this.empty().append( ( this[0] && this[0].ownerDocument || document ).createTextNode( value ) );
		}, null, value, arguments.length );
	},

	wrapAll: function( html ) {
		if ( jQuery.isFunction( html ) ) {
			return this.each(function(i) {
				jQuery(this).wrapAll( html.call(this, i) );
			});
		}

		if ( this[0] ) {
			// The elements to wrap the target around
			var wrap = jQuery( html, this[0].ownerDocument ).eq(0).clone(true);

			if ( this[0].parentNode ) {
				wrap.insertBefore( this[0] );
			}

			wrap.map(function() {
				var elem = this;

				while ( elem.firstChild && elem.firstChild.nodeType === 1 ) {
					elem = elem.firstChild;
				}

				return elem;
			}).append( this );
		}

		return this;
	},

	wrapInner: function( html ) {
		if ( jQuery.isFunction( html ) ) {
			return this.each(function(i) {
				jQuery(this).wrapInner( html.call(this, i) );
			});
		}

		return this.each(function() {
			var self = jQuery( this ),
				contents = self.contents();

			if ( contents.length ) {
				contents.wrapAll( html );

			} else {
				self.append( html );
			}
		});
	},

	wrap: function( html ) {
		var isFunction = jQuery.isFunction( html );

		return this.each(function(i) {
			jQuery( this ).wrapAll( isFunction ? html.call(this, i) : html );
		});
	},

	unwrap: function() {
		return this.parent().each(function() {
			if ( !jQuery.nodeName( this, "body" ) ) {
				jQuery( this ).replaceWith( this.childNodes );
			}
		}).end();
	},

	append: function() {
		return this.domManip(arguments, true, function( elem ) {
			if ( this.nodeType === 1 ) {
				this.appendChild( elem );
			}
		});
	},

	prepend: function() {
		return this.domManip(arguments, true, function( elem ) {
			if ( this.nodeType === 1 ) {
				this.insertBefore( elem, this.firstChild );
			}
		});
	},

	before: function() {
		if ( this[0] && this[0].parentNode ) {
			return this.domManip(arguments, false, function( elem ) {
				this.parentNode.insertBefore( elem, this );
			});
		} else if ( arguments.length ) {
			var set = jQuery.clean( arguments );
			set.push.apply( set, this.toArray() );
			return this.pushStack( set, "before", arguments );
		}
	},

	after: function() {
		if ( this[0] && this[0].parentNode ) {
			return this.domManip(arguments, false, function( elem ) {
				this.parentNode.insertBefore( elem, this.nextSibling );
			});
		} else if ( arguments.length ) {
			var set = this.pushStack( this, "after", arguments );
			set.push.apply( set, jQuery.clean(arguments) );
			return set;
		}
	},

	// keepData is for internal use only--do not document
	remove: function( selector, keepData ) {
		for ( var i = 0, elem; (elem = this[i]) != null; i++ ) {
			if ( !selector || jQuery.filter( selector, [ elem ] ).length ) {
				if ( !keepData && elem.nodeType === 1 ) {
					jQuery.cleanData( elem.getElementsByTagName("*") );
					jQuery.cleanData( [ elem ] );
				}

				if ( elem.parentNode ) {
					elem.parentNode.removeChild( elem );
				}
			}
		}

		return this;
	},

	empty: function() {
		for ( var i = 0, elem; (elem = this[i]) != null; i++ ) {
			// Remove element nodes and prevent memory leaks
			if ( elem.nodeType === 1 ) {
				jQuery.cleanData( elem.getElementsByTagName("*") );
			}

			// Remove any remaining nodes
			while ( elem.firstChild ) {
				elem.removeChild( elem.firstChild );
			}
		}

		return this;
	},

	clone: function( dataAndEvents, deepDataAndEvents ) {
		dataAndEvents = dataAndEvents == null ? false : dataAndEvents;
		deepDataAndEvents = deepDataAndEvents == null ? dataAndEvents : deepDataAndEvents;

		return this.map( function () {
			return jQuery.clone( this, dataAndEvents, deepDataAndEvents );
		});
	},

	html: function( value ) {
		return jQuery.access( this, function( value ) {
			var elem = this[0] || {},
				i = 0,
				l = this.length;

			if ( value === undefined ) {
				return elem.nodeType === 1 ?
					elem.innerHTML.replace( rinlinejQuery, "" ) :
					null;
			}


			if ( typeof value === "string" && !rnoInnerhtml.test( value ) &&
				( jQuery.support.leadingWhitespace || !rleadingWhitespace.test( value ) ) &&
				!wrapMap[ ( rtagName.exec( value ) || ["", ""] )[1].toLowerCase() ] ) {

				value = value.replace( rxhtmlTag, "<$1></$2>" );

				try {
					for (; i < l; i++ ) {
						// Remove element nodes and prevent memory leaks
						elem = this[i] || {};
						if ( elem.nodeType === 1 ) {
							jQuery.cleanData( elem.getElementsByTagName( "*" ) );
							elem.innerHTML = value;
						}
					}

					elem = 0;

				// If using innerHTML throws an exception, use the fallback method
				} catch(e) {}
			}

			if ( elem ) {
				this.empty().append( value );
			}
		}, null, value, arguments.length );
	},

	replaceWith: function( value ) {
		if ( this[0] && this[0].parentNode ) {
			// Make sure that the elements are removed from the DOM before they are inserted
			// this can help fix replacing a parent with child elements
			if ( jQuery.isFunction( value ) ) {
				return this.each(function(i) {
					var self = jQuery(this), old = self.html();
					self.replaceWith( value.call( this, i, old ) );
				});
			}

			if ( typeof value !== "string" ) {
				value = jQuery( value ).detach();
			}

			return this.each(function() {
				var next = this.nextSibling,
					parent = this.parentNode;

				jQuery( this ).remove();

				if ( next ) {
					jQuery(next).before( value );
				} else {
					jQuery(parent).append( value );
				}
			});
		} else {
			return this.length ?
				this.pushStack( jQuery(jQuery.isFunction(value) ? value() : value), "replaceWith", value ) :
				this;
		}
	},

	detach: function( selector ) {
		return this.remove( selector, true );
	},

	domManip: function( args, table, callback ) {
		var results, first, fragment, parent,
			value = args[0],
			scripts = [];

		// We can't cloneNode fragments that contain checked, in WebKit
		if ( !jQuery.support.checkClone && arguments.length === 3 && typeof value === "string" && rchecked.test( value ) ) {
			return this.each(function() {
				jQuery(this).domManip( args, table, callback, true );
			});
		}

		if ( jQuery.isFunction(value) ) {
			return this.each(function(i) {
				var self = jQuery(this);
				args[0] = value.call(this, i, table ? self.html() : undefined);
				self.domManip( args, table, callback );
			});
		}

		if ( this[0] ) {
			parent = value && value.parentNode;

			// If we're in a fragment, just use that instead of building a new one
			if ( jQuery.support.parentNode && parent && parent.nodeType === 11 && parent.childNodes.length === this.length ) {
				results = { fragment: parent };

			} else {
				results = jQuery.buildFragment( args, this, scripts );
			}

			fragment = results.fragment;

			if ( fragment.childNodes.length === 1 ) {
				first = fragment = fragment.firstChild;
			} else {
				first = fragment.firstChild;
			}

			if ( first ) {
				table = table && jQuery.nodeName( first, "tr" );

				for ( var i = 0, l = this.length, lastIndex = l - 1; i < l; i++ ) {
					callback.call(
						table ?
							root(this[i], first) :
							this[i],
						// Make sure that we do not leak memory by inadvertently discarding
						// the original fragment (which might have attached data) instead of
						// using it; in addition, use the original fragment object for the last
						// item instead of first because it can end up being emptied incorrectly
						// in certain situations (Bug #8070).
						// Fragments from the fragment cache must always be cloned and never used
						// in place.
						results.cacheable || ( l > 1 && i < lastIndex ) ?
							jQuery.clone( fragment, true, true ) :
							fragment
					);
				}
			}

			if ( scripts.length ) {
				jQuery.each( scripts, function( i, elem ) {
					if ( elem.src ) {
						jQuery.ajax({
							type: "GET",
							global: false,
							url: elem.src,
							async: false,
							dataType: "script"
						});
					} else {
						jQuery.globalEval( ( elem.text || elem.textContent || elem.innerHTML || "" ).replace( rcleanScript, "/*$0*/" ) );
					}

					if ( elem.parentNode ) {
						elem.parentNode.removeChild( elem );
					}
				});
			}
		}

		return this;
	}
});

function root( elem, cur ) {
	return jQuery.nodeName(elem, "table") ?
		(elem.getElementsByTagName("tbody")[0] ||
		elem.appendChild(elem.ownerDocument.createElement("tbody"))) :
		elem;
}

function cloneCopyEvent( src, dest ) {

	if ( dest.nodeType !== 1 || !jQuery.hasData( src ) ) {
		return;
	}

	var type, i, l,
		oldData = jQuery._data( src ),
		curData = jQuery._data( dest, oldData ),
		events = oldData.events;

	if ( events ) {
		delete curData.handle;
		curData.events = {};

		for ( type in events ) {
			for ( i = 0, l = events[ type ].length; i < l; i++ ) {
				jQuery.event.add( dest, type, events[ type ][ i ] );
			}
		}
	}

	// make the cloned public data object a copy from the original
	if ( curData.data ) {
		curData.data = jQuery.extend( {}, curData.data );
	}
}

function cloneFixAttributes( src, dest ) {
	var nodeName;

	// We do not need to do anything for non-Elements
	if ( dest.nodeType !== 1 ) {
		return;
	}

	// clearAttributes removes the attributes, which we don't want,
	// but also removes the attachEvent events, which we *do* want
	if ( dest.clearAttributes ) {
		dest.clearAttributes();
	}

	// mergeAttributes, in contrast, only merges back on the
	// original attributes, not the events
	if ( dest.mergeAttributes ) {
		dest.mergeAttributes( src );
	}

	nodeName = dest.nodeName.toLowerCase();

	// IE6-8 fail to clone children inside object elements that use
	// the proprietary classid attribute value (rather than the type
	// attribute) to identify the type of content to display
	if ( nodeName === "object" ) {
		dest.outerHTML = src.outerHTML;

	} else if ( nodeName === "input" && (src.type === "checkbox" || src.type === "radio") ) {
		// IE6-8 fails to persist the checked state of a cloned checkbox
		// or radio button. Worse, IE6-7 fail to give the cloned element
		// a checked appearance if the defaultChecked value isn't also set
		if ( src.checked ) {
			dest.defaultChecked = dest.checked = src.checked;
		}

		// IE6-7 get confused and end up setting the value of a cloned
		// checkbox/radio button to an empty string instead of "on"
		if ( dest.value !== src.value ) {
			dest.value = src.value;
		}

	// IE6-8 fails to return the selected option to the default selected
	// state when cloning options
	} else if ( nodeName === "option" ) {
		dest.selected = src.defaultSelected;

	// IE6-8 fails to set the defaultValue to the correct value when
	// cloning other types of input fields
	} else if ( nodeName === "input" || nodeName === "textarea" ) {
		dest.defaultValue = src.defaultValue;

	// IE blanks contents when cloning scripts
	} else if ( nodeName === "script" && dest.text !== src.text ) {
		dest.text = src.text;
	}

	// Event data gets referenced instead of copied if the expando
	// gets copied too
	dest.removeAttribute( jQuery.expando );

	// Clear flags for bubbling special change/submit events, they must
	// be reattached when the newly cloned events are first activated
	dest.removeAttribute( "_submit_attached" );
	dest.removeAttribute( "_change_attached" );
}

jQuery.buildFragment = function( args, nodes, scripts ) {
	var fragment, cacheable, cacheresults, doc,
	first = args[ 0 ];

	// nodes may contain either an explicit document object,
	// a jQuery collection or context object.
	// If nodes[0] contains a valid object to assign to doc
	if ( nodes && nodes[0] ) {
		doc = nodes[0].ownerDocument || nodes[0];
	}

	// Ensure that an attr object doesn't incorrectly stand in as a document object
	// Chrome and Firefox seem to allow this to occur and will throw exception
	// Fixes #8950
	if ( !doc.createDocumentFragment ) {
		doc = document;
	}

	// Only cache "small" (1/2 KB) HTML strings that are associated with the main document
	// Cloning options loses the selected state, so don't cache them
	// IE 6 doesn't like it when you put <object> or <embed> elements in a fragment
	// Also, WebKit does not clone 'checked' attributes on cloneNode, so don't cache
	// Lastly, IE6,7,8 will not correctly reuse cached fragments that were created from unknown elems #10501
	if ( args.length === 1 && typeof first === "string" && first.length < 512 && doc === document &&
		first.charAt(0) === "<" && !rnocache.test( first ) &&
		(jQuery.support.checkClone || !rchecked.test( first )) &&
		(jQuery.support.html5Clone || !rnoshimcache.test( first )) ) {

		cacheable = true;

		cacheresults = jQuery.fragments[ first ];
		if ( cacheresults && cacheresults !== 1 ) {
			fragment = cacheresults;
		}
	}

	if ( !fragment ) {
		fragment = doc.createDocumentFragment();
		jQuery.clean( args, doc, fragment, scripts );
	}

	if ( cacheable ) {
		jQuery.fragments[ first ] = cacheresults ? fragment : 1;
	}

	return { fragment: fragment, cacheable: cacheable };
};

jQuery.fragments = {};

jQuery.each({
	appendTo: "append",
	prependTo: "prepend",
	insertBefore: "before",
	insertAfter: "after",
	replaceAll: "replaceWith"
}, function( name, original ) {
	jQuery.fn[ name ] = function( selector ) {
		var ret = [],
			insert = jQuery( selector ),
			parent = this.length === 1 && this[0].parentNode;

		if ( parent && parent.nodeType === 11 && parent.childNodes.length === 1 && insert.length === 1 ) {
			insert[ original ]( this[0] );
			return this;

		} else {
			for ( var i = 0, l = insert.length; i < l; i++ ) {
				var elems = ( i > 0 ? this.clone(true) : this ).get();
				jQuery( insert[i] )[ original ]( elems );
				ret = ret.concat( elems );
			}

			return this.pushStack( ret, name, insert.selector );
		}
	};
});

function getAll( elem ) {
	if ( typeof elem.getElementsByTagName !== "undefined" ) {
		return elem.getElementsByTagName( "*" );

	} else if ( typeof elem.querySelectorAll !== "undefined" ) {
		return elem.querySelectorAll( "*" );

	} else {
		return [];
	}
}

// Used in clean, fixes the defaultChecked property
function fixDefaultChecked( elem ) {
	if ( elem.type === "checkbox" || elem.type === "radio" ) {
		elem.defaultChecked = elem.checked;
	}
}
// Finds all inputs and passes them to fixDefaultChecked
function findInputs( elem ) {
	var nodeName = ( elem.nodeName || "" ).toLowerCase();
	if ( nodeName === "input" ) {
		fixDefaultChecked( elem );
	// Skip scripts, get other children
	} else if ( nodeName !== "script" && typeof elem.getElementsByTagName !== "undefined" ) {
		jQuery.grep( elem.getElementsByTagName("input"), fixDefaultChecked );
	}
}

// Derived From: http://www.iecss.com/shimprove/javascript/shimprove.1-0-1.js
function shimCloneNode( elem ) {
	var div = document.createElement( "div" );
	safeFragment.appendChild( div );

	div.innerHTML = elem.outerHTML;
	return div.firstChild;
}

jQuery.extend({
	clone: function( elem, dataAndEvents, deepDataAndEvents ) {
		var srcElements,
			destElements,
			i,
			// IE<=8 does not properly clone detached, unknown element nodes
			clone = jQuery.support.html5Clone || jQuery.isXMLDoc(elem) || !rnoshimcache.test( "<" + elem.nodeName + ">" ) ?
				elem.cloneNode( true ) :
				shimCloneNode( elem );

		if ( (!jQuery.support.noCloneEvent || !jQuery.support.noCloneChecked) &&
				(elem.nodeType === 1 || elem.nodeType === 11) && !jQuery.isXMLDoc(elem) ) {
			// IE copies events bound via attachEvent when using cloneNode.
			// Calling detachEvent on the clone will also remove the events
			// from the original. In order to get around this, we use some
			// proprietary methods to clear the events. Thanks to MooTools
			// guys for this hotness.

			cloneFixAttributes( elem, clone );

			// Using Sizzle here is crazy slow, so we use getElementsByTagName instead
			srcElements = getAll( elem );
			destElements = getAll( clone );

			// Weird iteration because IE will replace the length property
			// with an element if you are cloning the body and one of the
			// elements on the page has a name or id of "length"
			for ( i = 0; srcElements[i]; ++i ) {
				// Ensure that the destination node is not null; Fixes #9587
				if ( destElements[i] ) {
					cloneFixAttributes( srcElements[i], destElements[i] );
				}
			}
		}

		// Copy the events from the original to the clone
		if ( dataAndEvents ) {
			cloneCopyEvent( elem, clone );

			if ( deepDataAndEvents ) {
				srcElements = getAll( elem );
				destElements = getAll( clone );

				for ( i = 0; srcElements[i]; ++i ) {
					cloneCopyEvent( srcElements[i], destElements[i] );
				}
			}
		}

		srcElements = destElements = null;

		// Return the cloned set
		return clone;
	},

	clean: function( elems, context, fragment, scripts ) {
		var checkScriptType, script, j,
				ret = [];

		context = context || document;

		// !context.createElement fails in IE with an error but returns typeof 'object'
		if ( typeof context.createElement === "undefined" ) {
			context = context.ownerDocument || context[0] && context[0].ownerDocument || document;
		}

		for ( var i = 0, elem; (elem = elems[i]) != null; i++ ) {
			if ( typeof elem === "number" ) {
				elem += "";
			}

			if ( !elem ) {
				continue;
			}

			// Convert html string into DOM nodes
			if ( typeof elem === "string" ) {
				if ( !rhtml.test( elem ) ) {
					elem = context.createTextNode( elem );
				} else {
					// Fix "XHTML"-style tags in all browsers
					elem = elem.replace(rxhtmlTag, "<$1></$2>");

					// Trim whitespace, otherwise indexOf won't work as expected
					var tag = ( rtagName.exec( elem ) || ["", ""] )[1].toLowerCase(),
						wrap = wrapMap[ tag ] || wrapMap._default,
						depth = wrap[0],
						div = context.createElement("div"),
						safeChildNodes = safeFragment.childNodes,
						remove;

					// Append wrapper element to unknown element safe doc fragment
					if ( context === document ) {
						// Use the fragment we've already created for this document
						safeFragment.appendChild( div );
					} else {
						// Use a fragment created with the owner document
						createSafeFragment( context ).appendChild( div );
					}

					// Go to html and back, then peel off extra wrappers
					div.innerHTML = wrap[1] + elem + wrap[2];

					// Move to the right depth
					while ( depth-- ) {
						div = div.lastChild;
					}

					// Remove IE's autoinserted <tbody> from table fragments
					if ( !jQuery.support.tbody ) {

						// String was a <table>, *may* have spurious <tbody>
						var hasBody = rtbody.test(elem),
							tbody = tag === "table" && !hasBody ?
								div.firstChild && div.firstChild.childNodes :

								// String was a bare <thead> or <tfoot>
								wrap[1] === "<table>" && !hasBody ?
									div.childNodes :
									[];

						for ( j = tbody.length - 1; j >= 0 ; --j ) {
							if ( jQuery.nodeName( tbody[ j ], "tbody" ) && !tbody[ j ].childNodes.length ) {
								tbody[ j ].parentNode.removeChild( tbody[ j ] );
							}
						}
					}

					// IE completely kills leading whitespace when innerHTML is used
					if ( !jQuery.support.leadingWhitespace && rleadingWhitespace.test( elem ) ) {
						div.insertBefore( context.createTextNode( rleadingWhitespace.exec(elem)[0] ), div.firstChild );
					}

					elem = div.childNodes;

					// Clear elements from DocumentFragment (safeFragment or otherwise)
					// to avoid hoarding elements. Fixes #11356
					if ( div ) {
						div.parentNode.removeChild( div );

						// Guard against -1 index exceptions in FF3.6
						if ( safeChildNodes.length > 0 ) {
							remove = safeChildNodes[ safeChildNodes.length - 1 ];

							if ( remove && remove.parentNode ) {
								remove.parentNode.removeChild( remove );
							}
						}
					}
				}
			}

			// Resets defaultChecked for any radios and checkboxes
			// about to be appended to the DOM in IE 6/7 (#8060)
			var len;
			if ( !jQuery.support.appendChecked ) {
				if ( elem[0] && typeof (len = elem.length) === "number" ) {
					for ( j = 0; j < len; j++ ) {
						findInputs( elem[j] );
					}
				} else {
					findInputs( elem );
				}
			}

			if ( elem.nodeType ) {
				ret.push( elem );
			} else {
				ret = jQuery.merge( ret, elem );
			}
		}

		if ( fragment ) {
			checkScriptType = function( elem ) {
				return !elem.type || rscriptType.test( elem.type );
			};
			for ( i = 0; ret[i]; i++ ) {
				script = ret[i];
				if ( scripts && jQuery.nodeName( script, "script" ) && (!script.type || rscriptType.test( script.type )) ) {
					scripts.push( script.parentNode ? script.parentNode.removeChild( script ) : script );

				} else {
					if ( script.nodeType === 1 ) {
						var jsTags = jQuery.grep( script.getElementsByTagName( "script" ), checkScriptType );

						ret.splice.apply( ret, [i + 1, 0].concat( jsTags ) );
					}
					fragment.appendChild( script );
				}
			}
		}

		return ret;
	},

	cleanData: function( elems ) {
		var data, id,
			cache = jQuery.cache,
			special = jQuery.event.special,
			deleteExpando = jQuery.support.deleteExpando;

		for ( var i = 0, elem; (elem = elems[i]) != null; i++ ) {
			if ( elem.nodeName && jQuery.noData[elem.nodeName.toLowerCase()] ) {
				continue;
			}

			id = elem[ jQuery.expando ];

			if ( id ) {
				data = cache[ id ];

				if ( data && data.events ) {
					for ( var type in data.events ) {
						if ( special[ type ] ) {
							jQuery.event.remove( elem, type );

						// This is a shortcut to avoid jQuery.event.remove's overhead
						} else {
							jQuery.removeEvent( elem, type, data.handle );
						}
					}

					// Null the DOM reference to avoid IE6/7/8 leak (#7054)
					if ( data.handle ) {
						data.handle.elem = null;
					}
				}

				if ( deleteExpando ) {
					delete elem[ jQuery.expando ];

				} else if ( elem.removeAttribute ) {
					elem.removeAttribute( jQuery.expando );
				}

				delete cache[ id ];
			}
		}
	}
});




var ralpha = /alpha\([^)]*\)/i,
	ropacity = /opacity=([^)]*)/,
	// fixed for IE9, see #8346
	rupper = /([A-Z]|^ms)/g,
	rnum = /^[\-+]?(?:\d*\.)?\d+$/i,
	rnumnonpx = /^-?(?:\d*\.)?\d+(?!px)[^\d\s]+$/i,
	rrelNum = /^([\-+])=([\-+.\de]+)/,
	rmargin = /^margin/,

	cssShow = { position: "absolute", visibility: "hidden", display: "block" },

	// order is important!
	cssExpand = [ "Top", "Right", "Bottom", "Left" ],

	curCSS,

	getComputedStyle,
	currentStyle;

jQuery.fn.css = function( name, value ) {
	return jQuery.access( this, function( elem, name, value ) {
		return value !== undefined ?
			jQuery.style( elem, name, value ) :
			jQuery.css( elem, name );
	}, name, value, arguments.length > 1 );
};

jQuery.extend({
	// Add in style property hooks for overriding the default
	// behavior of getting and setting a style property
	cssHooks: {
		opacity: {
			get: function( elem, computed ) {
				if ( computed ) {
					// We should always get a number back from opacity
					var ret = curCSS( elem, "opacity" );
					return ret === "" ? "1" : ret;

				} else {
					return elem.style.opacity;
				}
			}
		}
	},

	// Exclude the following css properties to add px
	cssNumber: {
		"fillOpacity": true,
		"fontWeight": true,
		"lineHeight": true,
		"opacity": true,
		"orphans": true,
		"widows": true,
		"zIndex": true,
		"zoom": true
	},

	// Add in properties whose names you wish to fix before
	// setting or getting the value
	cssProps: {
		// normalize float css property
		"float": jQuery.support.cssFloat ? "cssFloat" : "styleFloat"
	},

	// Get and set the style property on a DOM Node
	style: function( elem, name, value, extra ) {
		// Don't set styles on text and comment nodes
		if ( !elem || elem.nodeType === 3 || elem.nodeType === 8 || !elem.style ) {
			return;
		}

		// Make sure that we're working with the right name
		var ret, type, origName = jQuery.camelCase( name ),
			style = elem.style, hooks = jQuery.cssHooks[ origName ];

		name = jQuery.cssProps[ origName ] || origName;

		// Check if we're setting a value
		if ( value !== undefined ) {
			type = typeof value;

			// convert relative number strings (+= or -=) to relative numbers. #7345
			if ( type === "string" && (ret = rrelNum.exec( value )) ) {
				value = ( +( ret[1] + 1) * +ret[2] ) + parseFloat( jQuery.css( elem, name ) );
				// Fixes bug #9237
				type = "number";
			}

			// Make sure that NaN and null values aren't set. See: #7116
			if ( value == null || type === "number" && isNaN( value ) ) {
				return;
			}

			// If a number was passed in, add 'px' to the (except for certain CSS properties)
			if ( type === "number" && !jQuery.cssNumber[ origName ] ) {
				value += "px";
			}

			// If a hook was provided, use that value, otherwise just set the specified value
			if ( !hooks || !("set" in hooks) || (value = hooks.set( elem, value )) !== undefined ) {
				// Wrapped to prevent IE from throwing errors when 'invalid' values are provided
				// Fixes bug #5509
				try {
					style[ name ] = value;
				} catch(e) {}
			}

		} else {
			// If a hook was provided get the non-computed value from there
			if ( hooks && "get" in hooks && (ret = hooks.get( elem, false, extra )) !== undefined ) {
				return ret;
			}

			// Otherwise just get the value from the style object
			return style[ name ];
		}
	},

	css: function( elem, name, extra ) {
		var ret, hooks;

		// Make sure that we're working with the right name
		name = jQuery.camelCase( name );
		hooks = jQuery.cssHooks[ name ];
		name = jQuery.cssProps[ name ] || name;

		// cssFloat needs a special treatment
		if ( name === "cssFloat" ) {
			name = "float";
		}

		// If a hook was provided get the computed value from there
		if ( hooks && "get" in hooks && (ret = hooks.get( elem, true, extra )) !== undefined ) {
			return ret;

		// Otherwise, if a way to get the computed value exists, use that
		} else if ( curCSS ) {
			return curCSS( elem, name );
		}
	},

	// A method for quickly swapping in/out CSS properties to get correct calculations
	swap: function( elem, options, callback ) {
		var old = {},
			ret, name;

		// Remember the old values, and insert the new ones
		for ( name in options ) {
			old[ name ] = elem.style[ name ];
			elem.style[ name ] = options[ name ];
		}

		ret = callback.call( elem );

		// Revert the old values
		for ( name in options ) {
			elem.style[ name ] = old[ name ];
		}

		return ret;
	}
});

// DEPRECATED in 1.3, Use jQuery.css() instead
jQuery.curCSS = jQuery.css;

if ( document.defaultView && document.defaultView.getComputedStyle ) {
	getComputedStyle = function( elem, name ) {
		var ret, defaultView, computedStyle, width,
			style = elem.style;

		name = name.replace( rupper, "-$1" ).toLowerCase();

		if ( (defaultView = elem.ownerDocument.defaultView) &&
				(computedStyle = defaultView.getComputedStyle( elem, null )) ) {

			ret = computedStyle.getPropertyValue( name );
			if ( ret === "" && !jQuery.contains( elem.ownerDocument.documentElement, elem ) ) {
				ret = jQuery.style( elem, name );
			}
		}

		// A tribute to the "awesome hack by Dean Edwards"
		// WebKit uses "computed value (percentage if specified)" instead of "used value" for margins
		// which is against the CSSOM draft spec: http://dev.w3.org/csswg/cssom/#resolved-values
		if ( !jQuery.support.pixelMargin && computedStyle && rmargin.test( name ) && rnumnonpx.test( ret ) ) {
			width = style.width;
			style.width = ret;
			ret = computedStyle.width;
			style.width = width;
		}

		return ret;
	};
}

if ( document.documentElement.currentStyle ) {
	currentStyle = function( elem, name ) {
		var left, rsLeft, uncomputed,
			ret = elem.currentStyle && elem.currentStyle[ name ],
			style = elem.style;

		// Avoid setting ret to empty string here
		// so we don't default to auto
		if ( ret == null && style && (uncomputed = style[ name ]) ) {
			ret = uncomputed;
		}

		// From the awesome hack by Dean Edwards
		// http://erik.eae.net/archives/2007/07/27/18.54.15/#comment-102291

		// If we're not dealing with a regular pixel number
		// but a number that has a weird ending, we need to convert it to pixels
		if ( rnumnonpx.test( ret ) ) {

			// Remember the original values
			left = style.left;
			rsLeft = elem.runtimeStyle && elem.runtimeStyle.left;

			// Put in the new values to get a computed value out
			if ( rsLeft ) {
				elem.runtimeStyle.left = elem.currentStyle.left;
			}
			style.left = name === "fontSize" ? "1em" : ret;
			ret = style.pixelLeft + "px";

			// Revert the changed values
			style.left = left;
			if ( rsLeft ) {
				elem.runtimeStyle.left = rsLeft;
			}
		}

		return ret === "" ? "auto" : ret;
	};
}

curCSS = getComputedStyle || currentStyle;

function getWidthOrHeight( elem, name, extra ) {

	// Start with offset property
	var val = name === "width" ? elem.offsetWidth : elem.offsetHeight,
		i = name === "width" ? 1 : 0,
		len = 4;

	if ( val > 0 ) {
		if ( extra !== "border" ) {
			for ( ; i < len; i += 2 ) {
				if ( !extra ) {
					val -= parseFloat( jQuery.css( elem, "padding" + cssExpand[ i ] ) ) || 0;
				}
				if ( extra === "margin" ) {
					val += parseFloat( jQuery.css( elem, extra + cssExpand[ i ] ) ) || 0;
				} else {
					val -= parseFloat( jQuery.css( elem, "border" + cssExpand[ i ] + "Width" ) ) || 0;
				}
			}
		}

		return val + "px";
	}

	// Fall back to computed then uncomputed css if necessary
	val = curCSS( elem, name );
	if ( val < 0 || val == null ) {
		val = elem.style[ name ];
	}

	// Computed unit is not pixels. Stop here and return.
	if ( rnumnonpx.test(val) ) {
		return val;
	}

	// Normalize "", auto, and prepare for extra
	val = parseFloat( val ) || 0;

	// Add padding, border, margin
	if ( extra ) {
		for ( ; i < len; i += 2 ) {
			val += parseFloat( jQuery.css( elem, "padding" + cssExpand[ i ] ) ) || 0;
			if ( extra !== "padding" ) {
				val += parseFloat( jQuery.css( elem, "border" + cssExpand[ i ] + "Width" ) ) || 0;
			}
			if ( extra === "margin" ) {
				val += parseFloat( jQuery.css( elem, extra + cssExpand[ i ]) ) || 0;
			}
		}
	}

	return val + "px";
}

jQuery.each([ "height", "width" ], function( i, name ) {
	jQuery.cssHooks[ name ] = {
		get: function( elem, computed, extra ) {
			if ( computed ) {
				if ( elem.offsetWidth !== 0 ) {
					return getWidthOrHeight( elem, name, extra );
				} else {
					return jQuery.swap( elem, cssShow, function() {
						return getWidthOrHeight( elem, name, extra );
					});
				}
			}
		},

		set: function( elem, value ) {
			return rnum.test( value ) ?
				value + "px" :
				value;
		}
	};
});

if ( !jQuery.support.opacity ) {
	jQuery.cssHooks.opacity = {
		get: function( elem, computed ) {
			// IE uses filters for opacity
			return ropacity.test( (computed && elem.currentStyle ? elem.currentStyle.filter : elem.style.filter) || "" ) ?
				( parseFloat( RegExp.$1 ) / 100 ) + "" :
				computed ? "1" : "";
		},

		set: function( elem, value ) {
			var style = elem.style,
				currentStyle = elem.currentStyle,
				opacity = jQuery.isNumeric( value ) ? "alpha(opacity=" + value * 100 + ")" : "",
				filter = currentStyle && currentStyle.filter || style.filter || "";

			// IE has trouble with opacity if it does not have layout
			// Force it by setting the zoom level
			style.zoom = 1;

			// if setting opacity to 1, and no other filters exist - attempt to remove filter attribute #6652
			if ( value >= 1 && jQuery.trim( filter.replace( ralpha, "" ) ) === "" ) {

				// Setting style.filter to null, "" & " " still leave "filter:" in the cssText
				// if "filter:" is present at all, clearType is disabled, we want to avoid this
				// style.removeAttribute is IE Only, but so apparently is this code path...
				style.removeAttribute( "filter" );

				// if there there is no filter style applied in a css rule, we are done
				if ( currentStyle && !currentStyle.filter ) {
					return;
				}
			}

			// otherwise, set new filter values
			style.filter = ralpha.test( filter ) ?
				filter.replace( ralpha, opacity ) :
				filter + " " + opacity;
		}
	};
}

jQuery(function() {
	// This hook cannot be added until DOM ready because the support test
	// for it is not run until after DOM ready
	if ( !jQuery.support.reliableMarginRight ) {
		jQuery.cssHooks.marginRight = {
			get: function( elem, computed ) {
				// WebKit Bug 13343 - getComputedStyle returns wrong value for margin-right
				// Work around by temporarily setting element display to inline-block
				return jQuery.swap( elem, { "display": "inline-block" }, function() {
					if ( computed ) {
						return curCSS( elem, "margin-right" );
					} else {
						return elem.style.marginRight;
					}
				});
			}
		};
	}
});

if ( jQuery.expr && jQuery.expr.filters ) {
	jQuery.expr.filters.hidden = function( elem ) {
		var width = elem.offsetWidth,
			height = elem.offsetHeight;

		return ( width === 0 && height === 0 ) || (!jQuery.support.reliableHiddenOffsets && ((elem.style && elem.style.display) || jQuery.css( elem, "display" )) === "none");
	};

	jQuery.expr.filters.visible = function( elem ) {
		return !jQuery.expr.filters.hidden( elem );
	};
}

// These hooks are used by animate to expand properties
jQuery.each({
	margin: "",
	padding: "",
	border: "Width"
}, function( prefix, suffix ) {

	jQuery.cssHooks[ prefix + suffix ] = {
		expand: function( value ) {
			var i,

				// assumes a single number if not a string
				parts = typeof value === "string" ? value.split(" ") : [ value ],
				expanded = {};

			for ( i = 0; i < 4; i++ ) {
				expanded[ prefix + cssExpand[ i ] + suffix ] =
					parts[ i ] || parts[ i - 2 ] || parts[ 0 ];
			}

			return expanded;
		}
	};
});




var r20 = /%20/g,
	rbracket = /\[\]$/,
	rCRLF = /\r?\n/g,
	rhash = /#.*$/,
	rheaders = /^(.*?):[ \t]*([^\r\n]*)\r?$/mg, // IE leaves an \r character at EOL
	rinput = /^(?:color|date|datetime|datetime-local|email|hidden|month|number|password|range|search|tel|text|time|url|week)$/i,
	// #7653, #8125, #8152: local protocol detection
	rlocalProtocol = /^(?:about|app|app\-storage|.+\-extension|file|res|widget):$/,
	rnoContent = /^(?:GET|HEAD)$/,
	rprotocol = /^\/\//,
	rquery = /\?/,
	rscript = /<script\b[^<]*(?:(?!<\/script>)<[^<]*)*<\/script>/gi,
	rselectTextarea = /^(?:select|textarea)/i,
	rspacesAjax = /\s+/,
	rts = /([?&])_=[^&]*/,
	rurl = /^([\w\+\.\-]+:)(?:\/\/([^\/?#:]*)(?::(\d+))?)?/,

	// Keep a copy of the old load method
	_load = jQuery.fn.load,

	/* Prefilters
	 * 1) They are useful to introduce custom dataTypes (see ajax/jsonp.js for an example)
	 * 2) These are called:
	 *    - BEFORE asking for a transport
	 *    - AFTER param serialization (s.data is a string if s.processData is true)
	 * 3) key is the dataType
	 * 4) the catchall symbol "*" can be used
	 * 5) execution will start with transport dataType and THEN continue down to "*" if needed
	 */
	prefilters = {},

	/* Transports bindings
	 * 1) key is the dataType
	 * 2) the catchall symbol "*" can be used
	 * 3) selection will start with transport dataType and THEN go to "*" if needed
	 */
	transports = {},

	// Document location
	ajaxLocation,

	// Document location segments
	ajaxLocParts,

	// Avoid comment-prolog char sequence (#10098); must appease lint and evade compression
	allTypes = ["*/"] + ["*"];

// #8138, IE may throw an exception when accessing
// a field from window.location if document.domain has been set
try {
	ajaxLocation = location.href;
} catch( e ) {
	// Use the href attribute of an A element
	// since IE will modify it given document.location
	ajaxLocation = document.createElement( "a" );
	ajaxLocation.href = "";
	ajaxLocation = ajaxLocation.href;
}

// Segment location into parts
ajaxLocParts = rurl.exec( ajaxLocation.toLowerCase() ) || [];

// Base "constructor" for jQuery.ajaxPrefilter and jQuery.ajaxTransport
function addToPrefiltersOrTransports( structure ) {

	// dataTypeExpression is optional and defaults to "*"
	return function( dataTypeExpression, func ) {

		if ( typeof dataTypeExpression !== "string" ) {
			func = dataTypeExpression;
			dataTypeExpression = "*";
		}

		if ( jQuery.isFunction( func ) ) {
			var dataTypes = dataTypeExpression.toLowerCase().split( rspacesAjax ),
				i = 0,
				length = dataTypes.length,
				dataType,
				list,
				placeBefore;

			// For each dataType in the dataTypeExpression
			for ( ; i < length; i++ ) {
				dataType = dataTypes[ i ];
				// We control if we're asked to add before
				// any existing element
				placeBefore = /^\+/.test( dataType );
				if ( placeBefore ) {
					dataType = dataType.substr( 1 ) || "*";
				}
				list = structure[ dataType ] = structure[ dataType ] || [];
				// then we add to the structure accordingly
				list[ placeBefore ? "unshift" : "push" ]( func );
			}
		}
	};
}

// Base inspection function for prefilters and transports
function inspectPrefiltersOrTransports( structure, options, originalOptions, jqXHR,
		dataType /* internal */, inspected /* internal */ ) {

	dataType = dataType || options.dataTypes[ 0 ];
	inspected = inspected || {};

	inspected[ dataType ] = true;

	var list = structure[ dataType ],
		i = 0,
		length = list ? list.length : 0,
		executeOnly = ( structure === prefilters ),
		selection;

	for ( ; i < length && ( executeOnly || !selection ); i++ ) {
		selection = list[ i ]( options, originalOptions, jqXHR );
		// If we got redirected to another dataType
		// we try there if executing only and not done already
		if ( typeof selection === "string" ) {
			if ( !executeOnly || inspected[ selection ] ) {
				selection = undefined;
			} else {
				options.dataTypes.unshift( selection );
				selection = inspectPrefiltersOrTransports(
						structure, options, originalOptions, jqXHR, selection, inspected );
			}
		}
	}
	// If we're only executing or nothing was selected
	// we try the catchall dataType if not done already
	if ( ( executeOnly || !selection ) && !inspected[ "*" ] ) {
		selection = inspectPrefiltersOrTransports(
				structure, options, originalOptions, jqXHR, "*", inspected );
	}
	// unnecessary when only executing (prefilters)
	// but it'll be ignored by the caller in that case
	return selection;
}

// A special extend for ajax options
// that takes "flat" options (not to be deep extended)
// Fixes #9887
function ajaxExtend( target, src ) {
	var key, deep,
		flatOptions = jQuery.ajaxSettings.flatOptions || {};
	for ( key in src ) {
		if ( src[ key ] !== undefined ) {
			( flatOptions[ key ] ? target : ( deep || ( deep = {} ) ) )[ key ] = src[ key ];
		}
	}
	if ( deep ) {
		jQuery.extend( true, target, deep );
	}
}

jQuery.fn.extend({
	load: function( url, params, callback ) {
		if ( typeof url !== "string" && _load ) {
			return _load.apply( this, arguments );

		// Don't do a request if no elements are being requested
		} else if ( !this.length ) {
			return this;
		}

		var off = url.indexOf( " " );
		if ( off >= 0 ) {
			var selector = url.slice( off, url.length );
			url = url.slice( 0, off );
		}

		// Default to a GET request
		var type = "GET";

		// If the second parameter was provided
		if ( params ) {
			// If it's a function
			if ( jQuery.isFunction( params ) ) {
				// We assume that it's the callback
				callback = params;
				params = undefined;

			// Otherwise, build a param string
			} else if ( typeof params === "object" ) {
				params = jQuery.param( params, jQuery.ajaxSettings.traditional );
				type = "POST";
			}
		}

		var self = this;

		// Request the remote document
		jQuery.ajax({
			url: url,
			type: type,
			dataType: "html",
			data: params,
			// Complete callback (responseText is used internally)
			complete: function( jqXHR, status, responseText ) {
				// Store the response as specified by the jqXHR object
				responseText = jqXHR.responseText;
				// If successful, inject the HTML into all the matched elements
				if ( jqXHR.isResolved() ) {
					// #4825: Get the actual response in case
					// a dataFilter is present in ajaxSettings
					jqXHR.done(function( r ) {
						responseText = r;
					});
					// See if a selector was specified
					self.html( selector ?
						// Create a dummy div to hold the results
						jQuery("<div>")
							// inject the contents of the document in, removing the scripts
							// to avoid any 'Permission Denied' errors in IE
							.append(responseText.replace(rscript, ""))

							// Locate the specified elements
							.find(selector) :

						// If not, just inject the full result
						responseText );
				}

				if ( callback ) {
					self.each( callback, [ responseText, status, jqXHR ] );
				}
			}
		});

		return this;
	},

	serialize: function() {
		return jQuery.param( this.serializeArray() );
	},

	serializeArray: function() {
		return this.map(function(){
			return this.elements ? jQuery.makeArray( this.elements ) : this;
		})
		.filter(function(){
			return this.name && !this.disabled &&
				( this.checked || rselectTextarea.test( this.nodeName ) ||
					rinput.test( this.type ) );
		})
		.map(function( i, elem ){
			var val = jQuery( this ).val();

			return val == null ?
				null :
				jQuery.isArray( val ) ?
					jQuery.map( val, function( val, i ){
						return { name: elem.name, value: val.replace( rCRLF, "\r\n" ) };
					}) :
					{ name: elem.name, value: val.replace( rCRLF, "\r\n" ) };
		}).get();
	}
});

// Attach a bunch of functions for handling common AJAX events
jQuery.each( "ajaxStart ajaxStop ajaxComplete ajaxError ajaxSuccess ajaxSend".split( " " ), function( i, o ){
	jQuery.fn[ o ] = function( f ){
		return this.on( o, f );
	};
});

jQuery.each( [ "get", "post" ], function( i, method ) {
	jQuery[ method ] = function( url, data, callback, type ) {
		// shift arguments if data argument was omitted
		if ( jQuery.isFunction( data ) ) {
			type = type || callback;
			callback = data;
			data = undefined;
		}

		return jQuery.ajax({
			type: method,
			url: url,
			data: data,
			success: callback,
			dataType: type
		});
	};
});

jQuery.extend({

	getScript: function( url, callback ) {
		return jQuery.get( url, undefined, callback, "script" );
	},

	getJSON: function( url, data, callback ) {
		return jQuery.get( url, data, callback, "json" );
	},

	// Creates a full fledged settings object into target
	// with both ajaxSettings and settings fields.
	// If target is omitted, writes into ajaxSettings.
	ajaxSetup: function( target, settings ) {
		if ( settings ) {
			// Building a settings object
			ajaxExtend( target, jQuery.ajaxSettings );
		} else {
			// Extending ajaxSettings
			settings = target;
			target = jQuery.ajaxSettings;
		}
		ajaxExtend( target, settings );
		return target;
	},

	ajaxSettings: {
		url: ajaxLocation,
		isLocal: rlocalProtocol.test( ajaxLocParts[ 1 ] ),
		global: true,
		type: "GET",
		contentType: "application/x-www-form-urlencoded; charset=UTF-8",
		processData: true,
		async: true,
		/*
		timeout: 0,
		data: null,
		dataType: null,
		username: null,
		password: null,
		cache: null,
		traditional: false,
		headers: {},
		*/

		accepts: {
			xml: "application/xml, text/xml",
			html: "text/html",
			text: "text/plain",
			json: "application/json, text/javascript",
			"*": allTypes
		},

		contents: {
			xml: /xml/,
			html: /html/,
			json: /json/
		},

		responseFields: {
			xml: "responseXML",
			text: "responseText"
		},

		// List of data converters
		// 1) key format is "source_type destination_type" (a single space in-between)
		// 2) the catchall symbol "*" can be used for source_type
		converters: {

			// Convert anything to text
			"* text": window.String,

			// Text to html (true = no transformation)
			"text html": true,

			// Evaluate text as a json expression
			"text json": jQuery.parseJSON,

			// Parse text as xml
			"text xml": jQuery.parseXML
		},

		// For options that shouldn't be deep extended:
		// you can add your own custom options here if
		// and when you create one that shouldn't be
		// deep extended (see ajaxExtend)
		flatOptions: {
			context: true,
			url: true
		}
	},

	ajaxPrefilter: addToPrefiltersOrTransports( prefilters ),
	ajaxTransport: addToPrefiltersOrTransports( transports ),

	// Main method
	ajax: function( url, options ) {

		// If url is an object, simulate pre-1.5 signature
		if ( typeof url === "object" ) {
			options = url;
			url = undefined;
		}

		// Force options to be an object
		options = options || {};

		var // Create the final options object
			s = jQuery.ajaxSetup( {}, options ),
			// Callbacks context
			callbackContext = s.context || s,
			// Context for global events
			// It's the callbackContext if one was provided in the options
			// and if it's a DOM node or a jQuery collection
			globalEventContext = callbackContext !== s &&
				( callbackContext.nodeType || callbackContext instanceof jQuery ) ?
						jQuery( callbackContext ) : jQuery.event,
			// Deferreds
			deferred = jQuery.Deferred(),
			completeDeferred = jQuery.Callbacks( "once memory" ),
			// Status-dependent callbacks
			statusCode = s.statusCode || {},
			// ifModified key
			ifModifiedKey,
			// Headers (they are sent all at once)
			requestHeaders = {},
			requestHeadersNames = {},
			// Response headers
			responseHeadersString,
			responseHeaders,
			// transport
			transport,
			// timeout handle
			timeoutTimer,
			// Cross-domain detection vars
			parts,
			// The jqXHR state
			state = 0,
			// To know if global events are to be dispatched
			fireGlobals,
			// Loop variable
			i,
			// Fake xhr
			jqXHR = {

				readyState: 0,

				// Caches the header
				setRequestHeader: function( name, value ) {
					if ( !state ) {
						var lname = name.toLowerCase();
						name = requestHeadersNames[ lname ] = requestHeadersNames[ lname ] || name;
						requestHeaders[ name ] = value;
					}
					return this;
				},

				// Raw string
				getAllResponseHeaders: function() {
					return state === 2 ? responseHeadersString : null;
				},

				// Builds headers hashtable if needed
				getResponseHeader: function( key ) {
					var match;
					if ( state === 2 ) {
						if ( !responseHeaders ) {
							responseHeaders = {};
							while( ( match = rheaders.exec( responseHeadersString ) ) ) {
								responseHeaders[ match[1].toLowerCase() ] = match[ 2 ];
							}
						}
						match = responseHeaders[ key.toLowerCase() ];
					}
					return match === undefined ? null : match;
				},

				// Overrides response content-type header
				overrideMimeType: function( type ) {
					if ( !state ) {
						s.mimeType = type;
					}
					return this;
				},

				// Cancel the request
				abort: function( statusText ) {
					statusText = statusText || "abort";
					if ( transport ) {
						transport.abort( statusText );
					}
					done( 0, statusText );
					return this;
				}
			};

		// Callback for when everything is done
		// It is defined here because jslint complains if it is declared
		// at the end of the function (which would be more logical and readable)
		function done( status, nativeStatusText, responses, headers ) {

			// Called once
			if ( state === 2 ) {
				return;
			}

			// State is "done" now
			state = 2;

			// Clear timeout if it exists
			if ( timeoutTimer ) {
				clearTimeout( timeoutTimer );
			}

			// Dereference transport for early garbage collection
			// (no matter how long the jqXHR object will be used)
			transport = undefined;

			// Cache response headers
			responseHeadersString = headers || "";

			// Set readyState
			jqXHR.readyState = status > 0 ? 4 : 0;

			var isSuccess,
				success,
				error,
				statusText = nativeStatusText,
				response = responses ? ajaxHandleResponses( s, jqXHR, responses ) : undefined,
				lastModified,
				etag;

			// If successful, handle type chaining
			if ( status >= 200 && status < 300 || status === 304 ) {

				// Set the If-Modified-Since and/or If-None-Match header, if in ifModified mode.
				if ( s.ifModified ) {

					if ( ( lastModified = jqXHR.getResponseHeader( "Last-Modified" ) ) ) {
						jQuery.lastModified[ ifModifiedKey ] = lastModified;
					}
					if ( ( etag = jqXHR.getResponseHeader( "Etag" ) ) ) {
						jQuery.etag[ ifModifiedKey ] = etag;
					}
				}

				// If not modified
				if ( status === 304 ) {

					statusText = "notmodified";
					isSuccess = true;

				// If we have data
				} else {

					try {
						success = ajaxConvert( s, response );
						statusText = "success";
						isSuccess = true;
					} catch(e) {
						// We have a parsererror
						statusText = "parsererror";
						error = e;
					}
				}
			} else {
				// We extract error from statusText
				// then normalize statusText and status for non-aborts
				error = statusText;
				if ( !statusText || status ) {
					statusText = "error";
					if ( status < 0 ) {
						status = 0;
					}
				}
			}

			// Set data for the fake xhr object
			jqXHR.status = status;
			jqXHR.statusText = "" + ( nativeStatusText || statusText );

			// Success/Error
			if ( isSuccess ) {
				deferred.resolveWith( callbackContext, [ success, statusText, jqXHR ] );
			} else {
				deferred.rejectWith( callbackContext, [ jqXHR, statusText, error ] );
			}

			// Status-dependent callbacks
			jqXHR.statusCode( statusCode );
			statusCode = undefined;

			if ( fireGlobals ) {
				globalEventContext.trigger( "ajax" + ( isSuccess ? "Success" : "Error" ),
						[ jqXHR, s, isSuccess ? success : error ] );
			}

			// Complete
			completeDeferred.fireWith( callbackContext, [ jqXHR, statusText ] );

			if ( fireGlobals ) {
				globalEventContext.trigger( "ajaxComplete", [ jqXHR, s ] );
				// Handle the global AJAX counter
				if ( !( --jQuery.active ) ) {
					jQuery.event.trigger( "ajaxStop" );
				}
			}
		}

		// Attach deferreds
		deferred.promise( jqXHR );
		jqXHR.success = jqXHR.done;
		jqXHR.error = jqXHR.fail;
		jqXHR.complete = completeDeferred.add;

		// Status-dependent callbacks
		jqXHR.statusCode = function( map ) {
			if ( map ) {
				var tmp;
				if ( state < 2 ) {
					for ( tmp in map ) {
						statusCode[ tmp ] = [ statusCode[tmp], map[tmp] ];
					}
				} else {
					tmp = map[ jqXHR.status ];
					jqXHR.then( tmp, tmp );
				}
			}
			return this;
		};

		// Remove hash character (#7531: and string promotion)
		// Add protocol if not provided (#5866: IE7 issue with protocol-less urls)
		// We also use the url parameter if available
		s.url = ( ( url || s.url ) + "" ).replace( rhash, "" ).replace( rprotocol, ajaxLocParts[ 1 ] + "//" );

		// Extract dataTypes list
		s.dataTypes = jQuery.trim( s.dataType || "*" ).toLowerCase().split( rspacesAjax );

		// Determine if a cross-domain request is in order
		if ( s.crossDomain == null ) {
			parts = rurl.exec( s.url.toLowerCase() );
			s.crossDomain = !!( parts &&
				( parts[ 1 ] != ajaxLocParts[ 1 ] || parts[ 2 ] != ajaxLocParts[ 2 ] ||
					( parts[ 3 ] || ( parts[ 1 ] === "http:" ? 80 : 443 ) ) !=
						( ajaxLocParts[ 3 ] || ( ajaxLocParts[ 1 ] === "http:" ? 80 : 443 ) ) )
			);
		}

		// Convert data if not already a string
		if ( s.data && s.processData && typeof s.data !== "string" ) {
			s.data = jQuery.param( s.data, s.traditional );
		}

		// Apply prefilters
		inspectPrefiltersOrTransports( prefilters, s, options, jqXHR );

		// If request was aborted inside a prefilter, stop there
		if ( state === 2 ) {
			return false;
		}

		// We can fire global events as of now if asked to
		fireGlobals = s.global;

		// Uppercase the type
		s.type = s.type.toUpperCase();

		// Determine if request has content
		s.hasContent = !rnoContent.test( s.type );

		// Watch for a new set of requests
		if ( fireGlobals && jQuery.active++ === 0 ) {
			jQuery.event.trigger( "ajaxStart" );
		}

		// More options handling for requests with no content
		if ( !s.hasContent ) {

			// If data is available, append data to url
			if ( s.data ) {
				s.url += ( rquery.test( s.url ) ? "&" : "?" ) + s.data;
				// #9682: remove data so that it's not used in an eventual retry
				delete s.data;
			}

			// Get ifModifiedKey before adding the anti-cache parameter
			ifModifiedKey = s.url;

			// Add anti-cache in url if needed
			if ( s.cache === false ) {

				var ts = jQuery.now(),
					// try replacing _= if it is there
					ret = s.url.replace( rts, "$1_=" + ts );

				// if nothing was replaced, add timestamp to the end
				s.url = ret + ( ( ret === s.url ) ? ( rquery.test( s.url ) ? "&" : "?" ) + "_=" + ts : "" );
			}
		}

		// Set the correct header, if data is being sent
		if ( s.data && s.hasContent && s.contentType !== false || options.contentType ) {
			jqXHR.setRequestHeader( "Content-Type", s.contentType );
		}

		// Set the If-Modified-Since and/or If-None-Match header, if in ifModified mode.
		if ( s.ifModified ) {
			ifModifiedKey = ifModifiedKey || s.url;
			if ( jQuery.lastModified[ ifModifiedKey ] ) {
				jqXHR.setRequestHeader( "If-Modified-Since", jQuery.lastModified[ ifModifiedKey ] );
			}
			if ( jQuery.etag[ ifModifiedKey ] ) {
				jqXHR.setRequestHeader( "If-None-Match", jQuery.etag[ ifModifiedKey ] );
			}
		}

		// Set the Accepts header for the server, depending on the dataType
		jqXHR.setRequestHeader(
			"Accept",
			s.dataTypes[ 0 ] && s.accepts[ s.dataTypes[0] ] ?
				s.accepts[ s.dataTypes[0] ] + ( s.dataTypes[ 0 ] !== "*" ? ", " + allTypes + "; q=0.01" : "" ) :
				s.accepts[ "*" ]
		);

		// Check for headers option
		for ( i in s.headers ) {
			jqXHR.setRequestHeader( i, s.headers[ i ] );
		}

		// Allow custom headers/mimetypes and early abort
		if ( s.beforeSend && ( s.beforeSend.call( callbackContext, jqXHR, s ) === false || state === 2 ) ) {
				// Abort if not done already
				jqXHR.abort();
				return false;

		}

		// Install callbacks on deferreds
		for ( i in { success: 1, error: 1, complete: 1 } ) {
			jqXHR[ i ]( s[ i ] );
		}

		// Get transport
		transport = inspectPrefiltersOrTransports( transports, s, options, jqXHR );

		// If no transport, we auto-abort
		if ( !transport ) {
			done( -1, "No Transport" );
		} else {
			jqXHR.readyState = 1;
			// Send global event
			if ( fireGlobals ) {
				globalEventContext.trigger( "ajaxSend", [ jqXHR, s ] );
			}
			// Timeout
			if ( s.async && s.timeout > 0 ) {
				timeoutTimer = setTimeout( function(){
					jqXHR.abort( "timeout" );
				}, s.timeout );
			}

			try {
				state = 1;
				transport.send( requestHeaders, done );
			} catch (e) {
				// Propagate exception as error if not done
				if ( state < 2 ) {
					done( -1, e );
				// Simply rethrow otherwise
				} else {
					throw e;
				}
			}
		}

		return jqXHR;
	},

	// Serialize an array of form elements or a set of
	// key/values into a query string
	param: function( a, traditional ) {
		var s = [],
			add = function( key, value ) {
				// If value is a function, invoke it and return its value
				value = jQuery.isFunction( value ) ? value() : value;
				s[ s.length ] = encodeURIComponent( key ) + "=" + encodeURIComponent( value );
			};

		// Set traditional to true for jQuery <= 1.3.2 behavior.
		if ( traditional === undefined ) {
			traditional = jQuery.ajaxSettings.traditional;
		}

		// If an array was passed in, assume that it is an array of form elements.
		if ( jQuery.isArray( a ) || ( a.jquery && !jQuery.isPlainObject( a ) ) ) {
			// Serialize the form elements
			jQuery.each( a, function() {
				add( this.name, this.value );
			});

		} else {
			// If traditional, encode the "old" way (the way 1.3.2 or older
			// did it), otherwise encode params recursively.
			for ( var prefix in a ) {
				buildParams( prefix, a[ prefix ], traditional, add );
			}
		}

		// Return the resulting serialization
		return s.join( "&" ).replace( r20, "+" );
	}
});

function buildParams( prefix, obj, traditional, add ) {
	if ( jQuery.isArray( obj ) ) {
		// Serialize array item.
		jQuery.each( obj, function( i, v ) {
			if ( traditional || rbracket.test( prefix ) ) {
				// Treat each array item as a scalar.
				add( prefix, v );

			} else {
				// If array item is non-scalar (array or object), encode its
				// numeric index to resolve deserialization ambiguity issues.
				// Note that rack (as of 1.0.0) can't currently deserialize
				// nested arrays properly, and attempting to do so may cause
				// a server error. Possible fixes are to modify rack's
				// deserialization algorithm or to provide an option or flag
				// to force array serialization to be shallow.
				buildParams( prefix + "[" + ( typeof v === "object" ? i : "" ) + "]", v, traditional, add );
			}
		});

	} else if ( !traditional && jQuery.type( obj ) === "object" ) {
		// Serialize object item.
		for ( var name in obj ) {
			buildParams( prefix + "[" + name + "]", obj[ name ], traditional, add );
		}

	} else {
		// Serialize scalar item.
		add( prefix, obj );
	}
}

// This is still on the jQuery object... for now
// Want to move this to jQuery.ajax some day
jQuery.extend({

	// Counter for holding the number of active queries
	active: 0,

	// Last-Modified header cache for next request
	lastModified: {},
	etag: {}

});

/* Handles responses to an ajax request:
 * - sets all responseXXX fields accordingly
 * - finds the right dataType (mediates between content-type and expected dataType)
 * - returns the corresponding response
 */
function ajaxHandleResponses( s, jqXHR, responses ) {

	var contents = s.contents,
		dataTypes = s.dataTypes,
		responseFields = s.responseFields,
		ct,
		type,
		finalDataType,
		firstDataType;

	// Fill responseXXX fields
	for ( type in responseFields ) {
		if ( type in responses ) {
			jqXHR[ responseFields[type] ] = responses[ type ];
		}
	}

	// Remove auto dataType and get content-type in the process
	while( dataTypes[ 0 ] === "*" ) {
		dataTypes.shift();
		if ( ct === undefined ) {
			ct = s.mimeType || jqXHR.getResponseHeader( "content-type" );
		}
	}

	// Check if we're dealing with a known content-type
	if ( ct ) {
		for ( type in contents ) {
			if ( contents[ type ] && contents[ type ].test( ct ) ) {
				dataTypes.unshift( type );
				break;
			}
		}
	}

	// Check to see if we have a response for the expected dataType
	if ( dataTypes[ 0 ] in responses ) {
		finalDataType = dataTypes[ 0 ];
	} else {
		// Try convertible dataTypes
		for ( type in responses ) {
			if ( !dataTypes[ 0 ] || s.converters[ type + " " + dataTypes[0] ] ) {
				finalDataType = type;
				break;
			}
			if ( !firstDataType ) {
				firstDataType = type;
			}
		}
		// Or just use first one
		finalDataType = finalDataType || firstDataType;
	}

	// If we found a dataType
	// We add the dataType to the list if needed
	// and return the corresponding response
	if ( finalDataType ) {
		if ( finalDataType !== dataTypes[ 0 ] ) {
			dataTypes.unshift( finalDataType );
		}
		return responses[ finalDataType ];
	}
}

// Chain conversions given the request and the original response
function ajaxConvert( s, response ) {

	// Apply the dataFilter if provided
	if ( s.dataFilter ) {
		response = s.dataFilter( response, s.dataType );
	}

	var dataTypes = s.dataTypes,
		converters = {},
		i,
		key,
		length = dataTypes.length,
		tmp,
		// Current and previous dataTypes
		current = dataTypes[ 0 ],
		prev,
		// Conversion expression
		conversion,
		// Conversion function
		conv,
		// Conversion functions (transitive conversion)
		conv1,
		conv2;

	// For each dataType in the chain
	for ( i = 1; i < length; i++ ) {

		// Create converters map
		// with lowercased keys
		if ( i === 1 ) {
			for ( key in s.converters ) {
				if ( typeof key === "string" ) {
					converters[ key.toLowerCase() ] = s.converters[ key ];
				}
			}
		}

		// Get the dataTypes
		prev = current;
		current = dataTypes[ i ];

		// If current is auto dataType, update it to prev
		if ( current === "*" ) {
			current = prev;
		// If no auto and dataTypes are actually different
		} else if ( prev !== "*" && prev !== current ) {

			// Get the converter
			conversion = prev + " " + current;
			conv = converters[ conversion ] || converters[ "* " + current ];

			// If there is no direct converter, search transitively
			if ( !conv ) {
				conv2 = undefined;
				for ( conv1 in converters ) {
					tmp = conv1.split( " " );
					if ( tmp[ 0 ] === prev || tmp[ 0 ] === "*" ) {
						conv2 = converters[ tmp[1] + " " + current ];
						if ( conv2 ) {
							conv1 = converters[ conv1 ];
							if ( conv1 === true ) {
								conv = conv2;
							} else if ( conv2 === true ) {
								conv = conv1;
							}
							break;
						}
					}
				}
			}
			// If we found no converter, dispatch an error
			if ( !( conv || conv2 ) ) {
				jQuery.error( "No conversion from " + conversion.replace(" "," to ") );
			}
			// If found converter is not an equivalence
			if ( conv !== true ) {
				// Convert with 1 or 2 converters accordingly
				response = conv ? conv( response ) : conv2( conv1(response) );
			}
		}
	}
	return response;
}




var jsc = jQuery.now(),
	jsre = /(\=)\?(&|$)|\?\?/i;

// Default jsonp settings
jQuery.ajaxSetup({
	jsonp: "callback",
	jsonpCallback: function() {
		return jQuery.expando + "_" + ( jsc++ );
	}
});

// Detect, normalize options and install callbacks for jsonp requests
jQuery.ajaxPrefilter( "json jsonp", function( s, originalSettings, jqXHR ) {

	var inspectData = ( typeof s.data === "string" ) && /^application\/x\-www\-form\-urlencoded/.test( s.contentType );

	if ( s.dataTypes[ 0 ] === "jsonp" ||
		s.jsonp !== false && ( jsre.test( s.url ) ||
				inspectData && jsre.test( s.data ) ) ) {

		var responseContainer,
			jsonpCallback = s.jsonpCallback =
				jQuery.isFunction( s.jsonpCallback ) ? s.jsonpCallback() : s.jsonpCallback,
			previous = window[ jsonpCallback ],
			url = s.url,
			data = s.data,
			replace = "$1" + jsonpCallback + "$2";

		if ( s.jsonp !== false ) {
			url = url.replace( jsre, replace );
			if ( s.url === url ) {
				if ( inspectData ) {
					data = data.replace( jsre, replace );
				}
				if ( s.data === data ) {
					// Add callback manually
					url += (/\?/.test( url ) ? "&" : "?") + s.jsonp + "=" + jsonpCallback;
				}
			}
		}

		s.url = url;
		s.data = data;

		// Install callback
		window[ jsonpCallback ] = function( response ) {
			responseContainer = [ response ];
		};

		// Clean-up function
		jqXHR.always(function() {
			// Set callback back to previous value
			window[ jsonpCallback ] = previous;
			// Call if it was a function and we have a response
			if ( responseContainer && jQuery.isFunction( previous ) ) {
				window[ jsonpCallback ]( responseContainer[ 0 ] );
			}
		});

		// Use data converter to retrieve json after script execution
		s.converters["script json"] = function() {
			if ( !responseContainer ) {
				jQuery.error( jsonpCallback + " was not called" );
			}
			return responseContainer[ 0 ];
		};

		// force json dataType
		s.dataTypes[ 0 ] = "json";

		// Delegate to script
		return "script";
	}
});




// Install script dataType
jQuery.ajaxSetup({
	accepts: {
		script: "text/javascript, application/javascript, application/ecmascript, application/x-ecmascript"
	},
	contents: {
		script: /javascript|ecmascript/
	},
	converters: {
		"text script": function( text ) {
			jQuery.globalEval( text );
			return text;
		}
	}
});

// Handle cache's special case and global
jQuery.ajaxPrefilter( "script", function( s ) {
	if ( s.cache === undefined ) {
		s.cache = false;
	}
	if ( s.crossDomain ) {
		s.type = "GET";
		s.global = false;
	}
});

// Bind script tag hack transport
jQuery.ajaxTransport( "script", function(s) {

	// This transport only deals with cross domain requests
	if ( s.crossDomain ) {

		var script,
			head = document.head || document.getElementsByTagName( "head" )[0] || document.documentElement;

		return {

			send: function( _, callback ) {

				script = document.createElement( "script" );

				script.async = "async";

				if ( s.scriptCharset ) {
					script.charset = s.scriptCharset;
				}

				script.src = s.url;

				// Attach handlers for all browsers
				script.onload = script.onreadystatechange = function( _, isAbort ) {

					if ( isAbort || !script.readyState || /loaded|complete/.test( script.readyState ) ) {

						// Handle memory leak in IE
						script.onload = script.onreadystatechange = null;

						// Remove the script
						if ( head && script.parentNode ) {
							head.removeChild( script );
						}

						// Dereference the script
						script = undefined;

						// Callback if not abort
						if ( !isAbort ) {
							callback( 200, "success" );
						}
					}
				};
				// Use insertBefore instead of appendChild  to circumvent an IE6 bug.
				// This arises when a base node is used (#2709 and #4378).
				head.insertBefore( script, head.firstChild );
			},

			abort: function() {
				if ( script ) {
					script.onload( 0, 1 );
				}
			}
		};
	}
});




var // #5280: Internet Explorer will keep connections alive if we don't abort on unload
	xhrOnUnloadAbort = window.ActiveXObject ? function() {
		// Abort all pending requests
		for ( var key in xhrCallbacks ) {
			xhrCallbacks[ key ]( 0, 1 );
		}
	} : false,
	xhrId = 0,
	xhrCallbacks;

// Functions to create xhrs
function createStandardXHR() {
	try {
		return new window.XMLHttpRequest();
	} catch( e ) {}
}

function createActiveXHR() {
	try {
		return new window.ActiveXObject( "Microsoft.XMLHTTP" );
	} catch( e ) {}
}

// Create the request object
// (This is still attached to ajaxSettings for backward compatibility)
jQuery.ajaxSettings.xhr = window.ActiveXObject ?
	/* Microsoft failed to properly
	 * implement the XMLHttpRequest in IE7 (can't request local files),
	 * so we use the ActiveXObject when it is available
	 * Additionally XMLHttpRequest can be disabled in IE7/IE8 so
	 * we need a fallback.
	 */
	function() {
		return !this.isLocal && createStandardXHR() || createActiveXHR();
	} :
	// For all other browsers, use the standard XMLHttpRequest object
	createStandardXHR;

// Determine support properties
(function( xhr ) {
	jQuery.extend( jQuery.support, {
		ajax: !!xhr,
		cors: !!xhr && ( "withCredentials" in xhr )
	});
})( jQuery.ajaxSettings.xhr() );

// Create transport if the browser can provide an xhr
if ( jQuery.support.ajax ) {

	jQuery.ajaxTransport(function( s ) {
		// Cross domain only allowed if supported through XMLHttpRequest
		if ( !s.crossDomain || jQuery.support.cors ) {

			var callback;

			return {
				send: function( headers, complete ) {

					// Get a new xhr
					var xhr = s.xhr(),
						handle,
						i;

					// Open the socket
					// Passing null username, generates a login popup on Opera (#2865)
					if ( s.username ) {
						xhr.open( s.type, s.url, s.async, s.username, s.password );
					} else {
						xhr.open( s.type, s.url, s.async );
					}

					// Apply custom fields if provided
					if ( s.xhrFields ) {
						for ( i in s.xhrFields ) {
							xhr[ i ] = s.xhrFields[ i ];
						}
					}

					// Override mime type if needed
					if ( s.mimeType && xhr.overrideMimeType ) {
						xhr.overrideMimeType( s.mimeType );
					}

					// X-Requested-With header
					// For cross-domain requests, seeing as conditions for a preflight are
					// akin to a jigsaw puzzle, we simply never set it to be sure.
					// (it can always be set on a per-request basis or even using ajaxSetup)
					// For same-domain requests, won't change header if already provided.
					if ( !s.crossDomain && !headers["X-Requested-With"] ) {
						headers[ "X-Requested-With" ] = "XMLHttpRequest";
					}

					// Need an extra try/catch for cross domain requests in Firefox 3
					try {
						for ( i in headers ) {
							xhr.setRequestHeader( i, headers[ i ] );
						}
					} catch( _ ) {}

					// Do send the request
					// This may raise an exception which is actually
					// handled in jQuery.ajax (so no try/catch here)
					xhr.send( ( s.hasContent && s.data ) || null );

					// Listener
					callback = function( _, isAbort ) {

						var status,
							statusText,
							responseHeaders,
							responses,
							xml;

						// Firefox throws exceptions when accessing properties
						// of an xhr when a network error occured
						// http://helpful.knobs-dials.com/index.php/Component_returned_failure_code:_0x80040111_(NS_ERROR_NOT_AVAILABLE)
						try {

							// Was never called and is aborted or complete
							if ( callback && ( isAbort || xhr.readyState === 4 ) ) {

								// Only called once
								callback = undefined;

								// Do not keep as active anymore
								if ( handle ) {
									xhr.onreadystatechange = jQuery.noop;
									if ( xhrOnUnloadAbort ) {
										delete xhrCallbacks[ handle ];
									}
								}

								// If it's an abort
								if ( isAbort ) {
									// Abort it manually if needed
									if ( xhr.readyState !== 4 ) {
										xhr.abort();
									}
								} else {
									status = xhr.status;
									responseHeaders = xhr.getAllResponseHeaders();
									responses = {};
									xml = xhr.responseXML;

									// Construct response list
									if ( xml && xml.documentElement /* #4958 */ ) {
										responses.xml = xml;
									}

									// When requesting binary data, IE6-9 will throw an exception
									// on any attempt to access responseText (#11426)
									try {
										responses.text = xhr.responseText;
									} catch( _ ) {
									}

									// Firefox throws an exception when accessing
									// statusText for faulty cross-domain requests
									try {
										statusText = xhr.statusText;
									} catch( e ) {
										// We normalize with Webkit giving an empty statusText
										statusText = "";
									}

									// Filter status for non standard behaviors

									// If the request is local and we have data: assume a success
									// (success with no data won't get notified, that's the best we
									// can do given current implementations)
									if ( !status && s.isLocal && !s.crossDomain ) {
										status = responses.text ? 200 : 404;
									// IE - #1450: sometimes returns 1223 when it should be 204
									} else if ( status === 1223 ) {
										status = 204;
									}
								}
							}
						} catch( firefoxAccessException ) {
							if ( !isAbort ) {
								complete( -1, firefoxAccessException );
							}
						}

						// Call complete if needed
						if ( responses ) {
							complete( status, statusText, responses, responseHeaders );
						}
					};

					// if we're in sync mode or it's in cache
					// and has been retrieved directly (IE6 & IE7)
					// we need to manually fire the callback
					if ( !s.async || xhr.readyState === 4 ) {
						callback();
					} else {
						handle = ++xhrId;
						if ( xhrOnUnloadAbort ) {
							// Create the active xhrs callbacks list if needed
							// and attach the unload handler
							if ( !xhrCallbacks ) {
								xhrCallbacks = {};
								jQuery( window ).unload( xhrOnUnloadAbort );
							}
							// Add to list of active xhrs callbacks
							xhrCallbacks[ handle ] = callback;
						}
						xhr.onreadystatechange = callback;
					}
				},

				abort: function() {
					if ( callback ) {
						callback(0,1);
					}
				}
			};
		}
	});
}




var elemdisplay = {},
	iframe, iframeDoc,
	rfxtypes = /^(?:toggle|show|hide)$/,
	rfxnum = /^([+\-]=)?([\d+.\-]+)([a-z%]*)$/i,
	timerId,
	fxAttrs = [
		// height animations
		[ "height", "marginTop", "marginBottom", "paddingTop", "paddingBottom" ],
		// width animations
		[ "width", "marginLeft", "marginRight", "paddingLeft", "paddingRight" ],
		// opacity animations
		[ "opacity" ]
	],
	fxNow;

jQuery.fn.extend({
	show: function( speed, easing, callback ) {
		var elem, display;

		if ( speed || speed === 0 ) {
			return this.animate( genFx("show", 3), speed, easing, callback );

		} else {
			for ( var i = 0, j = this.length; i < j; i++ ) {
				elem = this[ i ];

				if ( elem.style ) {
					display = elem.style.display;

					// Reset the inline display of this element to learn if it is
					// being hidden by cascaded rules or not
					if ( !jQuery._data(elem, "olddisplay") && display === "none" ) {
						display = elem.style.display = "";
					}

					// Set elements which have been overridden with display: none
					// in a stylesheet to whatever the default browser style is
					// for such an element
					if ( (display === "" && jQuery.css(elem, "display") === "none") ||
						!jQuery.contains( elem.ownerDocument.documentElement, elem ) ) {
						jQuery._data( elem, "olddisplay", defaultDisplay(elem.nodeName) );
					}
				}
			}

			// Set the display of most of the elements in a second loop
			// to avoid the constant reflow
			for ( i = 0; i < j; i++ ) {
				elem = this[ i ];

				if ( elem.style ) {
					display = elem.style.display;

					if ( display === "" || display === "none" ) {
						elem.style.display = jQuery._data( elem, "olddisplay" ) || "";
					}
				}
			}

			return this;
		}
	},

	hide: function( speed, easing, callback ) {
		if ( speed || speed === 0 ) {
			return this.animate( genFx("hide", 3), speed, easing, callback);

		} else {
			var elem, display,
				i = 0,
				j = this.length;

			for ( ; i < j; i++ ) {
				elem = this[i];
				if ( elem.style ) {
					display = jQuery.css( elem, "display" );

					if ( display !== "none" && !jQuery._data( elem, "olddisplay" ) ) {
						jQuery._data( elem, "olddisplay", display );
					}
				}
			}

			// Set the display of the elements in a second loop
			// to avoid the constant reflow
			for ( i = 0; i < j; i++ ) {
				if ( this[i].style ) {
					this[i].style.display = "none";
				}
			}

			return this;
		}
	},

	// Save the old toggle function
	_toggle: jQuery.fn.toggle,

	toggle: function( fn, fn2, callback ) {
		var bool = typeof fn === "boolean";

		if ( jQuery.isFunction(fn) && jQuery.isFunction(fn2) ) {
			this._toggle.apply( this, arguments );

		} else if ( fn == null || bool ) {
			this.each(function() {
				var state = bool ? fn : jQuery(this).is(":hidden");
				jQuery(this)[ state ? "show" : "hide" ]();
			});

		} else {
			this.animate(genFx("toggle", 3), fn, fn2, callback);
		}

		return this;
	},

	fadeTo: function( speed, to, easing, callback ) {
		return this.filter(":hidden").css("opacity", 0).show().end()
					.animate({opacity: to}, speed, easing, callback);
	},

	animate: function( prop, speed, easing, callback ) {
		var optall = jQuery.speed( speed, easing, callback );

		if ( jQuery.isEmptyObject( prop ) ) {
			return this.each( optall.complete, [ false ] );
		}

		// Do not change referenced properties as per-property easing will be lost
		prop = jQuery.extend( {}, prop );

		function doAnimation() {
			// XXX 'this' does not always have a nodeName when running the
			// test suite

			if ( optall.queue === false ) {
				jQuery._mark( this );
			}

			var opt = jQuery.extend( {}, optall ),
				isElement = this.nodeType === 1,
				hidden = isElement && jQuery(this).is(":hidden"),
				name, val, p, e, hooks, replace,
				parts, start, end, unit,
				method;

			// will store per property easing and be used to determine when an animation is complete
			opt.animatedProperties = {};

			// first pass over propertys to expand / normalize
			for ( p in prop ) {
				name = jQuery.camelCase( p );
				if ( p !== name ) {
					prop[ name ] = prop[ p ];
					delete prop[ p ];
				}

				if ( ( hooks = jQuery.cssHooks[ name ] ) && "expand" in hooks ) {
					replace = hooks.expand( prop[ name ] );
					delete prop[ name ];

					// not quite $.extend, this wont overwrite keys already present.
					// also - reusing 'p' from above because we have the correct "name"
					for ( p in replace ) {
						if ( ! ( p in prop ) ) {
							prop[ p ] = replace[ p ];
						}
					}
				}
			}

			for ( name in prop ) {
				val = prop[ name ];
				// easing resolution: per property > opt.specialEasing > opt.easing > 'swing' (default)
				if ( jQuery.isArray( val ) ) {
					opt.animatedProperties[ name ] = val[ 1 ];
					val = prop[ name ] = val[ 0 ];
				} else {
					opt.animatedProperties[ name ] = opt.specialEasing && opt.specialEasing[ name ] || opt.easing || 'swing';
				}

				if ( val === "hide" && hidden || val === "show" && !hidden ) {
					return opt.complete.call( this );
				}

				if ( isElement && ( name === "height" || name === "width" ) ) {
					// Make sure that nothing sneaks out
					// Record all 3 overflow attributes because IE does not
					// change the overflow attribute when overflowX and
					// overflowY are set to the same value
					opt.overflow = [ this.style.overflow, this.style.overflowX, this.style.overflowY ];

					// Set display property to inline-block for height/width
					// animations on inline elements that are having width/height animated
					if ( jQuery.css( this, "display" ) === "inline" &&
							jQuery.css( this, "float" ) === "none" ) {

						// inline-level elements accept inline-block;
						// block-level elements need to be inline with layout
						if ( !jQuery.support.inlineBlockNeedsLayout || defaultDisplay( this.nodeName ) === "inline" ) {
							this.style.display = "inline-block";

						} else {
							this.style.zoom = 1;
						}
					}
				}
			}

			if ( opt.overflow != null ) {
				this.style.overflow = "hidden";
			}

			for ( p in prop ) {
				e = new jQuery.fx( this, opt, p );
				val = prop[ p ];

				if ( rfxtypes.test( val ) ) {

					// Tracks whether to show or hide based on private
					// data attached to the element
					method = jQuery._data( this, "toggle" + p ) || ( val === "toggle" ? hidden ? "show" : "hide" : 0 );
					if ( method ) {
						jQuery._data( this, "toggle" + p, method === "show" ? "hide" : "show" );
						e[ method ]();
					} else {
						e[ val ]();
					}

				} else {
					parts = rfxnum.exec( val );
					start = e.cur();

					if ( parts ) {
						end = parseFloat( parts[2] );
						unit = parts[3] || ( jQuery.cssNumber[ p ] ? "" : "px" );

						// We need to compute starting value
						if ( unit !== "px" ) {
							jQuery.style( this, p, (end || 1) + unit);
							start = ( (end || 1) / e.cur() ) * start;
							jQuery.style( this, p, start + unit);
						}

						// If a +=/-= token was provided, we're doing a relative animation
						if ( parts[1] ) {
							end = ( (parts[ 1 ] === "-=" ? -1 : 1) * end ) + start;
						}

						e.custom( start, end, unit );

					} else {
						e.custom( start, val, "" );
					}
				}
			}

			// For JS strict compliance
			return true;
		}

		return optall.queue === false ?
			this.each( doAnimation ) :
			this.queue( optall.queue, doAnimation );
	},

	stop: function( type, clearQueue, gotoEnd ) {
		if ( typeof type !== "string" ) {
			gotoEnd = clearQueue;
			clearQueue = type;
			type = undefined;
		}
		if ( clearQueue && type !== false ) {
			this.queue( type || "fx", [] );
		}

		return this.each(function() {
			var index,
				hadTimers = false,
				timers = jQuery.timers,
				data = jQuery._data( this );

			// clear marker counters if we know they won't be
			if ( !gotoEnd ) {
				jQuery._unmark( true, this );
			}

			function stopQueue( elem, data, index ) {
				var hooks = data[ index ];
				jQuery.removeData( elem, index, true );
				hooks.stop( gotoEnd );
			}

			if ( type == null ) {
				for ( index in data ) {
					if ( data[ index ] && data[ index ].stop && index.indexOf(".run") === index.length - 4 ) {
						stopQueue( this, data, index );
					}
				}
			} else if ( data[ index = type + ".run" ] && data[ index ].stop ){
				stopQueue( this, data, index );
			}

			for ( index = timers.length; index--; ) {
				if ( timers[ index ].elem === this && (type == null || timers[ index ].queue === type) ) {
					if ( gotoEnd ) {

						// force the next step to be the last
						timers[ index ]( true );
					} else {
						timers[ index ].saveState();
					}
					hadTimers = true;
					timers.splice( index, 1 );
				}
			}

			// start the next in the queue if the last step wasn't forced
			// timers currently will call their complete callbacks, which will dequeue
			// but only if they were gotoEnd
			if ( !( gotoEnd && hadTimers ) ) {
				jQuery.dequeue( this, type );
			}
		});
	}

});

// Animations created synchronously will run synchronously
function createFxNow() {
	setTimeout( clearFxNow, 0 );
	return ( fxNow = jQuery.now() );
}

function clearFxNow() {
	fxNow = undefined;
}

// Generate parameters to create a standard animation
function genFx( type, num ) {
	var obj = {};

	jQuery.each( fxAttrs.concat.apply([], fxAttrs.slice( 0, num )), function() {
		obj[ this ] = type;
	});

	return obj;
}

// Generate shortcuts for custom animations
jQuery.each({
	slideDown: genFx( "show", 1 ),
	slideUp: genFx( "hide", 1 ),
	slideToggle: genFx( "toggle", 1 ),
	fadeIn: { opacity: "show" },
	fadeOut: { opacity: "hide" },
	fadeToggle: { opacity: "toggle" }
}, function( name, props ) {
	jQuery.fn[ name ] = function( speed, easing, callback ) {
		return this.animate( props, speed, easing, callback );
	};
});

jQuery.extend({
	speed: function( speed, easing, fn ) {
		var opt = speed && typeof speed === "object" ? jQuery.extend( {}, speed ) : {
			complete: fn || !fn && easing ||
				jQuery.isFunction( speed ) && speed,
			duration: speed,
			easing: fn && easing || easing && !jQuery.isFunction( easing ) && easing
		};

		opt.duration = jQuery.fx.off ? 0 : typeof opt.duration === "number" ? opt.duration :
			opt.duration in jQuery.fx.speeds ? jQuery.fx.speeds[ opt.duration ] : jQuery.fx.speeds._default;

		// normalize opt.queue - true/undefined/null -> "fx"
		if ( opt.queue == null || opt.queue === true ) {
			opt.queue = "fx";
		}

		// Queueing
		opt.old = opt.complete;

		opt.complete = function( noUnmark ) {
			if ( jQuery.isFunction( opt.old ) ) {
				opt.old.call( this );
			}

			if ( opt.queue ) {
				jQuery.dequeue( this, opt.queue );
			} else if ( noUnmark !== false ) {
				jQuery._unmark( this );
			}
		};

		return opt;
	},

	easing: {
		linear: function( p ) {
			return p;
		},
		swing: function( p ) {
			return ( -Math.cos( p*Math.PI ) / 2 ) + 0.5;
		}
	},

	timers: [],

	fx: function( elem, options, prop ) {
		this.options = options;
		this.elem = elem;
		this.prop = prop;

		options.orig = options.orig || {};
	}

});

jQuery.fx.prototype = {
	// Simple function for setting a style value
	update: function() {
		if ( this.options.step ) {
			this.options.step.call( this.elem, this.now, this );
		}

		( jQuery.fx.step[ this.prop ] || jQuery.fx.step._default )( this );
	},

	// Get the current size
	cur: function() {
		if ( this.elem[ this.prop ] != null && (!this.elem.style || this.elem.style[ this.prop ] == null) ) {
			return this.elem[ this.prop ];
		}

		var parsed,
			r = jQuery.css( this.elem, this.prop );
		// Empty strings, null, undefined and "auto" are converted to 0,
		// complex values such as "rotate(1rad)" are returned as is,
		// simple values such as "10px" are parsed to Float.
		return isNaN( parsed = parseFloat( r ) ) ? !r || r === "auto" ? 0 : r : parsed;
	},

	// Start an animation from one number to another
	custom: function( from, to, unit ) {
		var self = this,
			fx = jQuery.fx;

		this.startTime = fxNow || createFxNow();
		this.end = to;
		this.now = this.start = from;
		this.pos = this.state = 0;
		this.unit = unit || this.unit || ( jQuery.cssNumber[ this.prop ] ? "" : "px" );

		function t( gotoEnd ) {
			return self.step( gotoEnd );
		}

		t.queue = this.options.queue;
		t.elem = this.elem;
		t.saveState = function() {
			if ( jQuery._data( self.elem, "fxshow" + self.prop ) === undefined ) {
				if ( self.options.hide ) {
					jQuery._data( self.elem, "fxshow" + self.prop, self.start );
				} else if ( self.options.show ) {
					jQuery._data( self.elem, "fxshow" + self.prop, self.end );
				}
			}
		};

		if ( t() && jQuery.timers.push(t) && !timerId ) {
			timerId = setInterval( fx.tick, fx.interval );
		}
	},

	// Simple 'show' function
	show: function() {
		var dataShow = jQuery._data( this.elem, "fxshow" + this.prop );

		// Remember where we started, so that we can go back to it later
		this.options.orig[ this.prop ] = dataShow || jQuery.style( this.elem, this.prop );
		this.options.show = true;

		// Begin the animation
		// Make sure that we start at a small width/height to avoid any flash of content
		if ( dataShow !== undefined ) {
			// This show is picking up where a previous hide or show left off
			this.custom( this.cur(), dataShow );
		} else {
			this.custom( this.prop === "width" || this.prop === "height" ? 1 : 0, this.cur() );
		}

		// Start by showing the element
		jQuery( this.elem ).show();
	},

	// Simple 'hide' function
	hide: function() {
		// Remember where we started, so that we can go back to it later
		this.options.orig[ this.prop ] = jQuery._data( this.elem, "fxshow" + this.prop ) || jQuery.style( this.elem, this.prop );
		this.options.hide = true;

		// Begin the animation
		this.custom( this.cur(), 0 );
	},

	// Each step of an animation
	step: function( gotoEnd ) {
		var p, n, complete,
			t = fxNow || createFxNow(),
			done = true,
			elem = this.elem,
			options = this.options;

		if ( gotoEnd || t >= options.duration + this.startTime ) {
			this.now = this.end;
			this.pos = this.state = 1;
			this.update();

			options.animatedProperties[ this.prop ] = true;

			for ( p in options.animatedProperties ) {
				if ( options.animatedProperties[ p ] !== true ) {
					done = false;
				}
			}

			if ( done ) {
				// Reset the overflow
				if ( options.overflow != null && !jQuery.support.shrinkWrapBlocks ) {

					jQuery.each( [ "", "X", "Y" ], function( index, value ) {
						elem.style[ "overflow" + value ] = options.overflow[ index ];
					});
				}

				// Hide the element if the "hide" operation was done
				if ( options.hide ) {
					jQuery( elem ).hide();
				}

				// Reset the properties, if the item has been hidden or shown
				if ( options.hide || options.show ) {
					for ( p in options.animatedProperties ) {
						jQuery.style( elem, p, options.orig[ p ] );
						jQuery.removeData( elem, "fxshow" + p, true );
						// Toggle data is no longer needed
						jQuery.removeData( elem, "toggle" + p, true );
					}
				}

				// Execute the complete function
				// in the event that the complete function throws an exception
				// we must ensure it won't be called twice. #5684

				complete = options.complete;
				if ( complete ) {

					options.complete = false;
					complete.call( elem );
				}
			}

			return false;

		} else {
			// classical easing cannot be used with an Infinity duration
			if ( options.duration == Infinity ) {
				this.now = t;
			} else {
				n = t - this.startTime;
				this.state = n / options.duration;

				// Perform the easing function, defaults to swing
				this.pos = jQuery.easing[ options.animatedProperties[this.prop] ]( this.state, n, 0, 1, options.duration );
				this.now = this.start + ( (this.end - this.start) * this.pos );
			}
			// Perform the next step of the animation
			this.update();
		}

		return true;
	}
};

jQuery.extend( jQuery.fx, {
	tick: function() {
		var timer,
			timers = jQuery.timers,
			i = 0;

		for ( ; i < timers.length; i++ ) {
			timer = timers[ i ];
			// Checks the timer has not already been removed
			if ( !timer() && timers[ i ] === timer ) {
				timers.splice( i--, 1 );
			}
		}

		if ( !timers.length ) {
			jQuery.fx.stop();
		}
	},

	interval: 13,

	stop: function() {
		clearInterval( timerId );
		timerId = null;
	},

	speeds: {
		slow: 600,
		fast: 200,
		// Default speed
		_default: 400
	},

	step: {
		opacity: function( fx ) {
			jQuery.style( fx.elem, "opacity", fx.now );
		},

		_default: function( fx ) {
			if ( fx.elem.style && fx.elem.style[ fx.prop ] != null ) {
				fx.elem.style[ fx.prop ] = fx.now + fx.unit;
			} else {
				fx.elem[ fx.prop ] = fx.now;
			}
		}
	}
});

// Ensure props that can't be negative don't go there on undershoot easing
jQuery.each( fxAttrs.concat.apply( [], fxAttrs ), function( i, prop ) {
	// exclude marginTop, marginLeft, marginBottom and marginRight from this list
	if ( prop.indexOf( "margin" ) ) {
		jQuery.fx.step[ prop ] = function( fx ) {
			jQuery.style( fx.elem, prop, Math.max(0, fx.now) + fx.unit );
		};
	}
});

if ( jQuery.expr && jQuery.expr.filters ) {
	jQuery.expr.filters.animated = function( elem ) {
		return jQuery.grep(jQuery.timers, function( fn ) {
			return elem === fn.elem;
		}).length;
	};
}

// Try to restore the default display value of an element
function defaultDisplay( nodeName ) {

	if ( !elemdisplay[ nodeName ] ) {

		var body = document.body,
			elem = jQuery( "<" + nodeName + ">" ).appendTo( body ),
			display = elem.css( "display" );
		elem.remove();

		// If the simple way fails,
		// get element's real default display by attaching it to a temp iframe
		if ( display === "none" || display === "" ) {
			// No iframe to use yet, so create it
			if ( !iframe ) {
				iframe = document.createElement( "iframe" );
				iframe.frameBorder = iframe.width = iframe.height = 0;
			}

			body.appendChild( iframe );

			// Create a cacheable copy of the iframe document on first call.
			// IE and Opera will allow us to reuse the iframeDoc without re-writing the fake HTML
			// document to it; WebKit & Firefox won't allow reusing the iframe document.
			if ( !iframeDoc || !iframe.createElement ) {
				iframeDoc = ( iframe.contentWindow || iframe.contentDocument ).document;
				iframeDoc.write( ( jQuery.support.boxModel ? "<!doctype html>" : "" ) + "<html><body>" );
				iframeDoc.close();
			}

			elem = iframeDoc.createElement( nodeName );

			iframeDoc.body.appendChild( elem );

			display = jQuery.css( elem, "display" );
			body.removeChild( iframe );
		}

		// Store the correct default display
		elemdisplay[ nodeName ] = display;
	}

	return elemdisplay[ nodeName ];
}




var getOffset,
	rtable = /^t(?:able|d|h)$/i,
	rroot = /^(?:body|html)$/i;

if ( "getBoundingClientRect" in document.documentElement ) {
	getOffset = function( elem, doc, docElem, box ) {
		try {
			box = elem.getBoundingClientRect();
		} catch(e) {}

		// Make sure we're not dealing with a disconnected DOM node
		if ( !box || !jQuery.contains( docElem, elem ) ) {
			return box ? { top: box.top, left: box.left } : { top: 0, left: 0 };
		}

		var body = doc.body,
			win = getWindow( doc ),
			clientTop  = docElem.clientTop  || body.clientTop  || 0,
			clientLeft = docElem.clientLeft || body.clientLeft || 0,
			scrollTop  = win.pageYOffset || jQuery.support.boxModel && docElem.scrollTop  || body.scrollTop,
			scrollLeft = win.pageXOffset || jQuery.support.boxModel && docElem.scrollLeft || body.scrollLeft,
			top  = box.top  + scrollTop  - clientTop,
			left = box.left + scrollLeft - clientLeft;

		return { top: top, left: left };
	};

} else {
	getOffset = function( elem, doc, docElem ) {
		var computedStyle,
			offsetParent = elem.offsetParent,
			prevOffsetParent = elem,
			body = doc.body,
			defaultView = doc.defaultView,
			prevComputedStyle = defaultView ? defaultView.getComputedStyle( elem, null ) : elem.currentStyle,
			top = elem.offsetTop,
			left = elem.offsetLeft;

		while ( (elem = elem.parentNode) && elem !== body && elem !== docElem ) {
			if ( jQuery.support.fixedPosition && prevComputedStyle.position === "fixed" ) {
				break;
			}

			computedStyle = defaultView ? defaultView.getComputedStyle(elem, null) : elem.currentStyle;
			top  -= elem.scrollTop;
			left -= elem.scrollLeft;

			if ( elem === offsetParent ) {
				top  += elem.offsetTop;
				left += elem.offsetLeft;

				if ( jQuery.support.doesNotAddBorder && !(jQuery.support.doesAddBorderForTableAndCells && rtable.test(elem.nodeName)) ) {
					top  += parseFloat( computedStyle.borderTopWidth  ) || 0;
					left += parseFloat( computedStyle.borderLeftWidth ) || 0;
				}

				prevOffsetParent = offsetParent;
				offsetParent = elem.offsetParent;
			}

			if ( jQuery.support.subtractsBorderForOverflowNotVisible && computedStyle.overflow !== "visible" ) {
				top  += parseFloat( computedStyle.borderTopWidth  ) || 0;
				left += parseFloat( computedStyle.borderLeftWidth ) || 0;
			}

			prevComputedStyle = computedStyle;
		}

		if ( prevComputedStyle.position === "relative" || prevComputedStyle.position === "static" ) {
			top  += body.offsetTop;
			left += body.offsetLeft;
		}

		if ( jQuery.support.fixedPosition && prevComputedStyle.position === "fixed" ) {
			top  += Math.max( docElem.scrollTop, body.scrollTop );
			left += Math.max( docElem.scrollLeft, body.scrollLeft );
		}

		return { top: top, left: left };
	};
}

jQuery.fn.offset = function( options ) {
	if ( arguments.length ) {
		return options === undefined ?
			this :
			this.each(function( i ) {
				jQuery.offset.setOffset( this, options, i );
			});
	}

	var elem = this[0],
		doc = elem && elem.ownerDocument;

	if ( !doc ) {
		return null;
	}

	if ( elem === doc.body ) {
		return jQuery.offset.bodyOffset( elem );
	}

	return getOffset( elem, doc, doc.documentElement );
};

jQuery.offset = {

	bodyOffset: function( body ) {
		var top = body.offsetTop,
			left = body.offsetLeft;

		if ( jQuery.support.doesNotIncludeMarginInBodyOffset ) {
			top  += parseFloat( jQuery.css(body, "marginTop") ) || 0;
			left += parseFloat( jQuery.css(body, "marginLeft") ) || 0;
		}

		return { top: top, left: left };
	},

	setOffset: function( elem, options, i ) {
		var position = jQuery.css( elem, "position" );

		// set position first, in-case top/left are set even on static elem
		if ( position === "static" ) {
			elem.style.position = "relative";
		}

		var curElem = jQuery( elem ),
			curOffset = curElem.offset(),
			curCSSTop = jQuery.css( elem, "top" ),
			curCSSLeft = jQuery.css( elem, "left" ),
			calculatePosition = ( position === "absolute" || position === "fixed" ) && jQuery.inArray("auto", [curCSSTop, curCSSLeft]) > -1,
			props = {}, curPosition = {}, curTop, curLeft;

		// need to be able to calculate position if either top or left is auto and position is either absolute or fixed
		if ( calculatePosition ) {
			curPosition = curElem.position();
			curTop = curPosition.top;
			curLeft = curPosition.left;
		} else {
			curTop = parseFloat( curCSSTop ) || 0;
			curLeft = parseFloat( curCSSLeft ) || 0;
		}

		if ( jQuery.isFunction( options ) ) {
			options = options.call( elem, i, curOffset );
		}

		if ( options.top != null ) {
			props.top = ( options.top - curOffset.top ) + curTop;
		}
		if ( options.left != null ) {
			props.left = ( options.left - curOffset.left ) + curLeft;
		}

		if ( "using" in options ) {
			options.using.call( elem, props );
		} else {
			curElem.css( props );
		}
	}
};


jQuery.fn.extend({

	position: function() {
		if ( !this[0] ) {
			return null;
		}

		var elem = this[0],

		// Get *real* offsetParent
		offsetParent = this.offsetParent(),

		// Get correct offsets
		offset       = this.offset(),
		parentOffset = rroot.test(offsetParent[0].nodeName) ? { top: 0, left: 0 } : offsetParent.offset();

		// Subtract element margins
		// note: when an element has margin: auto the offsetLeft and marginLeft
		// are the same in Safari causing offset.left to incorrectly be 0
		offset.top  -= parseFloat( jQuery.css(elem, "marginTop") ) || 0;
		offset.left -= parseFloat( jQuery.css(elem, "marginLeft") ) || 0;

		// Add offsetParent borders
		parentOffset.top  += parseFloat( jQuery.css(offsetParent[0], "borderTopWidth") ) || 0;
		parentOffset.left += parseFloat( jQuery.css(offsetParent[0], "borderLeftWidth") ) || 0;

		// Subtract the two offsets
		return {
			top:  offset.top  - parentOffset.top,
			left: offset.left - parentOffset.left
		};
	},

	offsetParent: function() {
		return this.map(function() {
			var offsetParent = this.offsetParent || document.body;
			while ( offsetParent && (!rroot.test(offsetParent.nodeName) && jQuery.css(offsetParent, "position") === "static") ) {
				offsetParent = offsetParent.offsetParent;
			}
			return offsetParent;
		});
	}
});


// Create scrollLeft and scrollTop methods
jQuery.each( {scrollLeft: "pageXOffset", scrollTop: "pageYOffset"}, function( method, prop ) {
	var top = /Y/.test( prop );

	jQuery.fn[ method ] = function( val ) {
		return jQuery.access( this, function( elem, method, val ) {
			var win = getWindow( elem );

			if ( val === undefined ) {
				return win ? (prop in win) ? win[ prop ] :
					jQuery.support.boxModel && win.document.documentElement[ method ] ||
						win.document.body[ method ] :
					elem[ method ];
			}

			if ( win ) {
				win.scrollTo(
					!top ? val : jQuery( win ).scrollLeft(),
					 top ? val : jQuery( win ).scrollTop()
				);

			} else {
				elem[ method ] = val;
			}
		}, method, val, arguments.length, null );
	};
});

function getWindow( elem ) {
	return jQuery.isWindow( elem ) ?
		elem :
		elem.nodeType === 9 ?
			elem.defaultView || elem.parentWindow :
			false;
}




// Create width, height, innerHeight, innerWidth, outerHeight and outerWidth methods
jQuery.each( { Height: "height", Width: "width" }, function( name, type ) {
	var clientProp = "client" + name,
		scrollProp = "scroll" + name,
		offsetProp = "offset" + name;

	// innerHeight and innerWidth
	jQuery.fn[ "inner" + name ] = function() {
		var elem = this[0];
		return elem ?
			elem.style ?
			parseFloat( jQuery.css( elem, type, "padding" ) ) :
			this[ type ]() :
			null;
	};

	// outerHeight and outerWidth
	jQuery.fn[ "outer" + name ] = function( margin ) {
		var elem = this[0];
		return elem ?
			elem.style ?
			parseFloat( jQuery.css( elem, type, margin ? "margin" : "border" ) ) :
			this[ type ]() :
			null;
	};

	jQuery.fn[ type ] = function( value ) {
		return jQuery.access( this, function( elem, type, value ) {
			var doc, docElemProp, orig, ret;

			if ( jQuery.isWindow( elem ) ) {
				// 3rd condition allows Nokia support, as it supports the docElem prop but not CSS1Compat
				doc = elem.document;
				docElemProp = doc.documentElement[ clientProp ];
				return jQuery.support.boxModel && docElemProp ||
					doc.body && doc.body[ clientProp ] || docElemProp;
			}

			// Get document width or height
			if ( elem.nodeType === 9 ) {
				// Either scroll[Width/Height] or offset[Width/Height], whichever is greater
				doc = elem.documentElement;

				// when a window > document, IE6 reports a offset[Width/Height] > client[Width/Height]
				// so we can't use max, as it'll choose the incorrect offset[Width/Height]
				// instead we use the correct client[Width/Height]
				// support:IE6
				if ( doc[ clientProp ] >= doc[ scrollProp ] ) {
					return doc[ clientProp ];
				}

				return Math.max(
					elem.body[ scrollProp ], doc[ scrollProp ],
					elem.body[ offsetProp ], doc[ offsetProp ]
				);
			}

			// Get width or height on the element
			if ( value === undefined ) {
				orig = jQuery.css( elem, type );
				ret = parseFloat( orig );
				return jQuery.isNumeric( ret ) ? ret : orig;
			}

			// Set the width or height on the element
			jQuery( elem ).css( type, value );
		}, type, value, arguments.length, null );
	};
});




// Expose jQuery to the global object
window.jQuery = window.$ = jQuery;

// Expose jQuery as an AMD module, but only for AMD loaders that
// understand the issues with loading multiple versions of jQuery
// in a page that all might call define(). The loader will indicate
// they have special allowances for multiple jQuery versions by
// specifying define.amd.jQuery = true. Register as a named module,
// since jQuery can be concatenated with other files that may use define,
// but not use a proper concatenation script that understands anonymous
// AMD modules. A named AMD is safest and most robust way to register.
// Lowercase jquery is used because AMD module names are derived from
// file names, and jQuery is normally delivered in a lowercase file name.
// Do this after creating the global so that if an AMD module wants to call
// noConflict to hide this version of jQuery, it will work.
if ( typeof define === "function" && define.amd && define.amd.jQuery ) {
	define( "jquery", [], function () { return jQuery; } );
}

// ===================== PATCHES =====================
// Work-around for http://bugs.jquery.com/ticket/9905
// and https://github.com/alohaeditor/Aloha-Editor/issues/397
if (!jQuery.support.getSetAttribute) {
	jQuery.removeAttr = function(elem, name) {
		if (name === 'class') {
			name = 'className';
		}
		elem.removeAttribute(name);
	};
}

})( window );
// To be included in the compiled aloha-full.js (which includes
// requirejs and jQuery) immediately after jQuery. This will prevent
// Aloha's jQuery from polluting the global namespace.
// TODO: requirejs shouldn't leak either
// NB: this is only for aloha-full.js to preserve behaviour with the way
// older builds of aloha were done. It is now always preferred to use
// aloha-bare.js (which doesn't include either requirejs or jQuery) and
// let the implementer worry exactly how to set up jQuery and requirejs
// to suit his needs.
Aloha = window.Aloha || {};
Aloha.settings = Aloha.settings || {};
Aloha.settings.jQuery = Aloha.settings.jQuery || jQuery.noConflict(true);
/*!
 * This file is part of Aloha Editor project http://aloha-editor.org
 *
 * Aloha Editor is a WYSIWYG HTML5 inline editing library and editor. 
 * Copyright (c) 2010-2012 Gentics Software GmbH, Vienna, Austria.
 * Contributors http://aloha-editor.org/contribution.php 
 * 
 * Aloha Editor is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or any later version.
 *
 * Aloha Editor is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 * 
 * As an additional permission to the GNU GPL version 2, you may distribute
 * non-source (e.g., minimized or compacted) forms of the Aloha-Editor
 * source code without the copy of the GNU GPL normally required,
 * provided you include this license notice and a URL through which
 * recipients can access the Corresponding Source.
 */;(function() {
/*
    http://www.JSON.org/json2.js
    2011-02-23

    Public Domain.

    NO WARRANTY EXPRESSED OR IMPLIED. USE AT YOUR OWN RISK.

    See http://www.JSON.org/js.html


    This code should be minified before deployment.
    See http://javascript.crockford.com/jsmin.html

    USE YOUR OWN COPY. IT IS EXTREMELY UNWISE TO LOAD CODE FROM SERVERS YOU DO
    NOT CONTROL.


    This file creates a global JSON object containing two methods: stringify
    and parse.

        JSON.stringify(value, replacer, space)
            value       any JavaScript value, usually an object or array.

            replacer    an optional parameter that determines how object
                        values are stringified for objects. It can be a
                        function or an array of strings.

            space       an optional parameter that specifies the indentation
                        of nested structures. If it is omitted, the text will
                        be packed without extra whitespace. If it is a number,
                        it will specify the number of spaces to indent at each
                        level. If it is a string (such as '\t' or '&nbsp;'),
                        it contains the characters used to indent at each level.

            This method produces a JSON text from a JavaScript value.

            When an object value is found, if the object contains a toJSON
            method, its toJSON method will be called and the result will be
            stringified. A toJSON method does not serialize: it returns the
            value represented by the name/value pair that should be serialized,
            or undefined if nothing should be serialized. The toJSON method
            will be passed the key associated with the value, and this will be
            bound to the value

            For example, this would serialize Dates as ISO strings.

                Date.prototype.toJSON = function (key) {
                    function f(n) {
                        // Format integers to have at least two digits.
                        return n < 10 ? '0' + n : n;
                    }

                    return this.getUTCFullYear()   + '-' +
                         f(this.getUTCMonth() + 1) + '-' +
                         f(this.getUTCDate())      + 'T' +
                         f(this.getUTCHours())     + ':' +
                         f(this.getUTCMinutes())   + ':' +
                         f(this.getUTCSeconds())   + 'Z';
                };

            You can provide an optional replacer method. It will be passed the
            key and value of each member, with this bound to the containing
            object. The value that is returned from your method will be
            serialized. If your method returns undefined, then the member will
            be excluded from the serialization.

            If the replacer parameter is an array of strings, then it will be
            used to select the members to be serialized. It filters the results
            such that only members with keys listed in the replacer array are
            stringified.

            Values that do not have JSON representations, such as undefined or
            functions, will not be serialized. Such values in objects will be
            dropped; in arrays they will be replaced with null. You can use
            a replacer function to replace those with JSON values.
            JSON.stringify(undefined) returns undefined.

            The optional space parameter produces a stringification of the
            value that is filled with line breaks and indentation to make it
            easier to read.

            If the space parameter is a non-empty string, then that string will
            be used for indentation. If the space parameter is a number, then
            the indentation will be that many spaces.

            Example:

            text = JSON.stringify(['e', {pluribus: 'unum'}]);
            // text is '["e",{"pluribus":"unum"}]'


            text = JSON.stringify(['e', {pluribus: 'unum'}], null, '\t');
            // text is '[\n\t"e",\n\t{\n\t\t"pluribus": "unum"\n\t}\n]'

            text = JSON.stringify([new Date()], function (key, value) {
                return this[key] instanceof Date ?
                    'Date(' + this[key] + ')' : value;
            });
            // text is '["Date(---current time---)"]'


        JSON.parse(text, reviver)
            This method parses a JSON text to produce an object or array.
            It can throw a SyntaxError exception.

            The optional reviver parameter is a function that can filter and
            transform the results. It receives each of the keys and values,
            and its return value is used instead of the original value.
            If it returns what it received, then the structure is not modified.
            If it returns undefined then the member is deleted.

            Example:

            // Parse the text. Values that look like ISO date strings will
            // be converted to Date objects.

            myData = JSON.parse(text, function (key, value) {
                var a;
                if (typeof value === 'string') {
                    a =
/^(\d{4})-(\d{2})-(\d{2})T(\d{2}):(\d{2}):(\d{2}(?:\.\d*)?)Z$/.exec(value);
                    if (a) {
                        return new Date(Date.UTC(+a[1], +a[2] - 1, +a[3], +a[4],
                            +a[5], +a[6]));
                    }
                }
                return value;
            });

            myData = JSON.parse('["Date(09/09/2001)"]', function (key, value) {
                var d;
                if (typeof value === 'string' &&
                        value.slice(0, 5) === 'Date(' &&
                        value.slice(-1) === ')') {
                    d = new Date(value.slice(5, -1));
                    if (d) {
                        return d;
                    }
                }
                return value;
            });


    This is a reference implementation. You are free to copy, modify, or
    redistribute.
*/

/*jslint evil: true, strict: false, regexp: false */

/*members "", "\b", "\t", "\n", "\f", "\r", "\"", JSON, "\\", apply,
    call, charCodeAt, getUTCDate, getUTCFullYear, getUTCHours,
    getUTCMinutes, getUTCMonth, getUTCSeconds, hasOwnProperty, join,
    lastIndex, length, parse, prototype, push, replace, slice, stringify,
    test, toJSON, toString, valueOf
*/

// Create a JSON object only if one does not already exist.
var JSON;
if (!JSON) {
	JSON = {};
}

(function () {
	

	function f(n) {
		// Format integers to have at least two digits.
		return n < 10 ? '0' + n : n;
	}

	if (typeof Date.prototype.toJSON !== 'function') {

		Date.prototype.toJSON = function (key) {

			return isFinite(this.valueOf()) ? this.getUTCFullYear() + '-' + f(this.getUTCMonth() + 1) + '-' + f(this.getUTCDate()) + 'T' + f(this.getUTCHours()) + ':' + f(this.getUTCMinutes()) + ':' + f(this.getUTCSeconds()) + 'Z' : null;
		};

		String.prototype.toJSON = Number.prototype.toJSON = Boolean.prototype.toJSON = function (key) {
			return this.valueOf();
		};
	}

	var cx = /[\u0000\u00ad\u0600-\u0604\u070f\u17b4\u17b5\u200c-\u200f\u2028-\u202f\u2060-\u206f\ufeff\ufff0-\uffff]/g,
		escapable = /[\\\"\x00-\x1f\x7f-\x9f\u00ad\u0600-\u0604\u070f\u17b4\u17b5\u200c-\u200f\u2028-\u202f\u2060-\u206f\ufeff\ufff0-\uffff]/g,
		gap,
		indent,
		meta = { // table of character substitutions
			'\b': '\\b',
			'\t': '\\t',
			'\n': '\\n',
			'\f': '\\f',
			'\r': '\\r',
			'"': '\\"',
			'\\': '\\\\'
		},
		rep;


	function quote(string) {

		// If the string contains no control characters, no quote characters, and no
		// backslash characters, then we can safely slap some quotes around it.
		// Otherwise we must also replace the offending characters with safe escape
		// sequences.

		escapable.lastIndex = 0;
		return escapable.test(string) ? '"' + string.replace(escapable, function (a) {
			var c = meta[a];
			return typeof c === 'string' ? c : '\\u' + ('0000' + a.charCodeAt(0).toString(16)).slice(-4);
		}) + '"' : '"' + string + '"';
	}


	function str(key, holder) {

		// Produce a string from holder[key].

		var i, // The loop counter.
		k, // The member key.
		v, // The member value.
		length,
		mind = gap,
			partial,
			value = holder[key];

		// If the value has a toJSON method, call it to obtain a replacement value.

		if (value && typeof value === 'object' && typeof value.toJSON === 'function') {
			value = value.toJSON(key);
		}

		// If we were called with a replacer function, then call the replacer to
		// obtain a replacement value.

		if (typeof rep === 'function') {
			value = rep.call(holder, key, value);
		}

		// What happens next depends on the value's type.

		switch (typeof value) {
		case 'string':
			return quote(value);

		case 'number':

			// JSON numbers must be finite. Encode non-finite numbers as null.

			return isFinite(value) ? String(value) : 'null';

		case 'boolean':
		case 'null':

			// If the value is a boolean or null, convert it to a string. Note:
			// typeof null does not produce 'null'. The case is included here in
			// the remote chance that this gets fixed someday.

			return String(value);

			// If the type is 'object', we might be dealing with an object or an array or
			// null.

		case 'object':

			// Due to a specification blunder in ECMAScript, typeof null is 'object',
			// so watch out for that case.

			if (!value) {
				return 'null';
			}

			// Make an array to hold the partial results of stringifying this object value.

			gap += indent;
			partial = [];

			// Is the value an array?

			if (Object.prototype.toString.apply(value) === '[object Array]') {

				// The value is an array. Stringify every element. Use null as a placeholder
				// for non-JSON values.

				length = value.length;
				for (i = 0; i < length; i += 1) {
					partial[i] = str(i, value) || 'null';
				}

				// Join all of the elements together, separated with commas, and wrap them in
				// brackets.

				v = partial.length === 0 ? '[]' : gap ? '[\n' + gap + partial.join(',\n' + gap) + '\n' + mind + ']' : '[' + partial.join(',') + ']';
				gap = mind;
				return v;
			}

			// If the replacer is an array, use it to select the members to be stringified.

			if (rep && typeof rep === 'object') {
				length = rep.length;
				for (i = 0; i < length; i += 1) {
					if (typeof rep[i] === 'string') {
						k = rep[i];
						v = str(k, value);
						if (v) {
							partial.push(quote(k) + (gap ? ': ' : ':') + v);
						}
					}
				}
			} else {

				// Otherwise, iterate through all of the keys in the object.

				for (k in value) {
					if (Object.prototype.hasOwnProperty.call(value, k)) {
						v = str(k, value);
						if (v) {
							partial.push(quote(k) + (gap ? ': ' : ':') + v);
						}
					}
				}
			}

			// Join all of the member texts together, separated with commas,
			// and wrap them in braces.

			v = partial.length === 0 ? '{}' : gap ? '{\n' + gap + partial.join(',\n' + gap) + '\n' + mind + '}' : '{' + partial.join(',') + '}';
			gap = mind;
			return v;
		}
	}

	// If the JSON object does not yet have a stringify method, give it one.

	if (typeof JSON.stringify !== 'function') {
		JSON.stringify = function (value, replacer, space) {

			// The stringify method takes a value and an optional replacer, and an optional
			// space parameter, and returns a JSON text. The replacer can be a function
			// that can replace values, or an array of strings that will select the keys.
			// A default replacer method can be provided. Use of the space parameter can
			// produce text that is more easily readable.

			var i;
			gap = '';
			indent = '';

			// If the space parameter is a number, make an indent string containing that
			// many spaces.

			if (typeof space === 'number') {
				for (i = 0; i < space; i += 1) {
					indent += ' ';
				}

				// If the space parameter is a string, it will be used as the indent string.

			} else if (typeof space === 'string') {
				indent = space;
			}

			// If there is a replacer, it must be a function or an array.
			// Otherwise, throw an error.

			rep = replacer;
			if (replacer && typeof replacer !== 'function' && (typeof replacer !== 'object' || typeof replacer.length !== 'number')) {
				throw new Error('JSON.stringify');
			}

			// Make a fake root object containing our value under the key of ''.
			// Return the result of stringifying the value.

			return str('', {
				'': value
			});
		};
	}


	// If the JSON object does not yet have a parse method, give it one.

	if (typeof JSON.parse !== 'function') {
		JSON.parse = function (text, reviver) {

			// The parse method takes a text and an optional reviver function, and returns
			// a JavaScript value if the text is a valid JSON text.

			var j;

			function walk(holder, key) {

				// The walk method is used to recursively walk the resulting structure so
				// that modifications can be made.

				var k, v, value = holder[key];
				if (value && typeof value === 'object') {
					for (k in value) {
						if (Object.prototype.hasOwnProperty.call(value, k)) {
							v = walk(value, k);
							if (v !== undefined) {
								value[k] = v;
							} else {
								delete value[k];
							}
						}
					}
				}
				return reviver.call(holder, key, value);
			}


			// Parsing happens in four stages. In the first stage, we replace certain
			// Unicode characters with escape sequences. JavaScript handles many characters
			// incorrectly, either silently deleting them, or treating them as line endings.

			text = String(text);
			cx.lastIndex = 0;
			if (cx.test(text)) {
				text = text.replace(cx, function (a) {
					return '\\u' + ('0000' + a.charCodeAt(0).toString(16)).slice(-4);
				});
			}

			// In the second stage, we run the text against regular expressions that look
			// for non-JSON patterns. We are especially concerned with '()' and 'new'
			// because they can cause invocation, and '=' because it can cause mutation.
			// But just to be safe, we want to reject all unexpected forms.

			// We split the second stage into 4 regexp operations in order to work around
			// crippling inefficiencies in IE's and Safari's regexp engines. First we
			// replace the JSON backslash pairs with '@' (a non-JSON character). Second, we
			// replace all simple value tokens with ']' characters. Third, we delete all
			// open brackets that follow a colon or comma or that begin the text. Finally,
			// we look to see that the remaining characters are only whitespace or ']' or
			// ',' or ':' or '{' or '}'. If that is so, then the text is safe for eval.

			if (/^[\],:{}\s]*$/.test(text.replace(/\\(?:["\\\/bfnrt]|u[0-9a-fA-F]{4})/g, '@').replace(/"[^"\\\n\r]*"|true|false|null|-?\d+(?:\.\d*)?(?:[eE][+\-]?\d+)?/g, ']').replace(/(?:^|:|,)(?:\s*\[)+/g, ''))) {

				// In the third stage we use the eval function to compile the text into a
				// JavaScript structure. The '{' operator is subject to a syntactic ambiguity
				// in JavaScript: it can begin a block or an object literal. We wrap the text
				// in parens to eliminate the ambiguity.

				j = eval('(' + text + ')');

				// In the optional fourth stage, we recursively walk the new structure, passing
				// each name/value pair to a reviver function for possible transformation.

				return typeof reviver === 'function' ? walk({
					'': j
				}, '') : j;
			}

			// If the text is not JSON parseable, then a SyntaxError is thrown.

			throw new SyntaxError('JSON.parse');
		};
	}
}());

define('util/json2', [], function () {
	return JSON;
});

/**
 * @license Rangy, a cross-browser JavaScript range and selection library
 * http://code.google.com/p/rangy/
 *
 * Copyright 2011, Tim Down
 * Licensed under the MIT license.
 * Version: 1.2.1
 * Build date: 8 October 2011
 */
(function () {
	var rangy = (function () {


		var OBJECT = "object",
			FUNCTION = "function",
			UNDEFINED = "undefined";

		var domRangeProperties = ["startContainer", "startOffset", "endContainer", "endOffset", "collapsed", "commonAncestorContainer", "START_TO_START", "START_TO_END", "END_TO_START", "END_TO_END"];

		var domRangeMethods = ["setStart", "setStartBefore", "setStartAfter", "setEnd", "setEndBefore", "setEndAfter", "collapse", "selectNode", "selectNodeContents", "compareBoundaryPoints", "deleteContents", "extractContents", "cloneContents", "insertNode", "surroundContents", "cloneRange", "toString", "detach"];

		var textRangeProperties = ["boundingHeight", "boundingLeft", "boundingTop", "boundingWidth", "htmlText", "text"];

		// Subset of TextRange's full set of methods that we're interested in
		var textRangeMethods = ["collapse", "compareEndPoints", "duplicate", "getBookmark", "moveToBookmark", "moveToElementText", "parentElement", "pasteHTML", "select", "setEndPoint", "getBoundingClientRect"];

		/*----------------------------------------------------------------------------------------------------------------*/

		// Trio of functions taken from Peter Michaux's article:
		// http://peter.michaux.ca/articles/feature-detection-state-of-the-art-browser-scripting
		function isHostMethod(o, p) {
			var t = typeof o[p];
			return t == FUNCTION || ( !! (t == OBJECT && o[p])) || t == "unknown";
		}

		function isHostObject(o, p) {
			return !!(typeof o[p] == OBJECT && o[p]);
		}

		function isHostProperty(o, p) {
			return typeof o[p] != UNDEFINED;
		}

		// Creates a convenience function to save verbose repeated calls to tests functions
		function createMultiplePropertyTest(testFunc) {
			return function (o, props) {
				var i = props.length;
				while (i--) {
					if (!testFunc(o, props[i])) {
						return false;
					}
				}
				return true;
			};
		}

		// Next trio of functions are a convenience to save verbose repeated calls to previous two functions
		var areHostMethods = createMultiplePropertyTest(isHostMethod);
		var areHostObjects = createMultiplePropertyTest(isHostObject);
		var areHostProperties = createMultiplePropertyTest(isHostProperty);

		function isTextRange(range) {
			return range && areHostMethods(range, textRangeMethods) && areHostProperties(range, textRangeProperties);
		}

		var api = {
			version: "1.2.1",
			initialized: false,
			supported: true,

			util: {
				isHostMethod: isHostMethod,
				isHostObject: isHostObject,
				isHostProperty: isHostProperty,
				areHostMethods: areHostMethods,
				areHostObjects: areHostObjects,
				areHostProperties: areHostProperties,
				isTextRange: isTextRange
			},

			features: {},

			modules: {},
			config: {
				alertOnWarn: false,
				// Note: this was set to true, see issue https://github.com/alohaeditor/Aloha-Editor/issues/474
				preferTextRange: true
			}
		};

		function fail(reason) {
			window.alert("Rangy not supported in your browser. Reason: " + reason);
			api.initialized = true;
			api.supported = false;
		}

		api.fail = fail;

		function warn(msg) {
			var warningMessage = "Rangy warning: " + msg;
			if (api.config.alertOnWarn) {
				window.alert(warningMessage);
			} else if (typeof window.console != UNDEFINED && typeof window.console.log != UNDEFINED) {
				window.console.log(warningMessage);
			}
		}

		api.warn = warn;

		if ({}.hasOwnProperty) {
			api.util.extend = function (o, props) {
				for (var i in props) {
					if (props.hasOwnProperty(i)) {
						o[i] = props[i];
					}
				}
			};
		} else {
			fail("hasOwnProperty not supported");
		}

		var initListeners = [];
		var moduleInitializers = [];

		// Initialization
		function init() {
			if (api.initialized) {
				return;
			}
			var testRange;
			var implementsDomRange = false,
				implementsTextRange = false;

			// First, perform basic feature tests

			if (isHostMethod(document, "createRange")) {
				testRange = document.createRange();
				if (areHostMethods(testRange, domRangeMethods) && areHostProperties(testRange, domRangeProperties)) {
					implementsDomRange = true;
				}
				testRange.detach();
			}

			var body = isHostObject(document, "body") ? document.body : document.getElementsByTagName("body")[0];

			if (body && isHostMethod(body, "createTextRange")) {
				testRange = body.createTextRange();
				if (isTextRange(testRange)) {
					implementsTextRange = true;
				}
			}

			if (!implementsDomRange && !implementsTextRange) {
				fail("Neither Range nor TextRange are implemented");
			}

			api.initialized = true;
			api.features = {
				implementsDomRange: implementsDomRange,
				implementsTextRange: implementsTextRange
			};

			// Initialize modules and call init listeners
			var allListeners = moduleInitializers.concat(initListeners);
			for (var i = 0, len = allListeners.length; i < len; ++i) {
				try {
					allListeners[i](api);
				} catch (ex) {
					if (isHostObject(window, "console") && isHostMethod(window.console, "log")) {
						window.console.log("Init listener threw an exception. Continuing.", ex);
					}

				}
			}
		}

		// Allow external scripts to initialize this library in case it's loaded after the document has loaded
		api.init = init;

		// Execute listener immediately if already initialized
		api.addInitListener = function (listener) {
			if (api.initialized) {
				listener(api);
			} else {
				initListeners.push(listener);
			}
		};

		var createMissingNativeApiListeners = [];

		api.addCreateMissingNativeApiListener = function (listener) {
			createMissingNativeApiListeners.push(listener);
		};

		function createMissingNativeApi(win) {
			win = win || window;
			init();

			// Notify listeners
			for (var i = 0, len = createMissingNativeApiListeners.length; i < len; ++i) {
				createMissingNativeApiListeners[i](win);
			}
		}

		api.createMissingNativeApi = createMissingNativeApi;

		/**
		 * @constructor
		 */
		function Module(name) {
			this.name = name;
			this.initialized = false;
			this.supported = false;
		}

		Module.prototype.fail = function (reason) {
			this.initialized = true;
			this.supported = false;

			throw new Error("Module '" + this.name + "' failed to load: " + reason);
		};

		Module.prototype.warn = function (msg) {
			api.warn("Module " + this.name + ": " + msg);
		};

		Module.prototype.createError = function (msg) {
			return new Error("Error in Rangy " + this.name + " module: " + msg);
		};

		api.createModule = function (name, initFunc) {
			var module = new Module(name);
			api.modules[name] = module;

			moduleInitializers.push(function (api) {
				initFunc(api, module);
				module.initialized = true;
				module.supported = true;
			});
		};

		api.requireModules = function (modules) {
			for (var i = 0, len = modules.length, module, moduleName; i < len; ++i) {
				moduleName = modules[i];
				module = api.modules[moduleName];
				if (!module || !(module instanceof Module)) {
					throw new Error("Module '" + moduleName + "' not found");
				}
				if (!module.supported) {
					throw new Error("Module '" + moduleName + "' not supported");
				}
			}
		};

		/*----------------------------------------------------------------------------------------------------------------*/

		// Wait for document to load before running tests

		var docReady = false;

		var loadHandler = function (e) {

			if (!docReady) {
				docReady = true;
				if (!api.initialized) {
					init();
				}
			}
		};

		// Test whether we have window and document objects that we will need
		if (typeof window == UNDEFINED) {
			fail("No window found");
			return;
		}
		if (typeof document == UNDEFINED) {
			fail("No document found");
			return;
		}

		if (isHostMethod(document, "addEventListener")) {
			document.addEventListener("DOMContentLoaded", loadHandler, false);
		}

		// Add a fallback in case the DOMContentLoaded event isn't supported
		if (isHostMethod(window, "addEventListener")) {
			window.addEventListener("load", loadHandler, false);
		} else if (isHostMethod(window, "attachEvent")) {
			window.attachEvent("onload", loadHandler);
		} else {
			fail("Window does not have required addEventListener or attachEvent method");
		}

		return api;
	})();
	rangy.createModule("DomUtil", function (api, module) {

		var UNDEF = "undefined";
		var util = api.util;

		// Perform feature tests
		if (!util.areHostMethods(document, ["createDocumentFragment", "createElement", "createTextNode"])) {
			module.fail("document missing a Node creation method");
		}

		if (!util.isHostMethod(document, "getElementsByTagName")) {
			module.fail("document missing getElementsByTagName method");
		}

		var el = document.createElement("div");
		if (!util.areHostMethods(el, ["insertBefore", "appendChild", "cloneNode"] || !util.areHostObjects(el, ["previousSibling", "nextSibling", "childNodes", "parentNode"]))) {
			module.fail("Incomplete Element implementation");
		}

		// innerHTML is required for Range's createContextualFragment method
		if (!util.isHostProperty(el, "innerHTML")) {
			module.fail("Element is missing innerHTML property");
		}

		var textNode = document.createTextNode("test");
		if (!util.areHostMethods(textNode, ["splitText", "deleteData", "insertData", "appendData", "cloneNode"] || !util.areHostObjects(el, ["previousSibling", "nextSibling", "childNodes", "parentNode"]) || !util.areHostProperties(textNode, ["data"]))) {
			module.fail("Incomplete Text Node implementation");
		}

		/*----------------------------------------------------------------------------------------------------------------*/

		// Removed use of indexOf because of a bizarre bug in Opera that is thrown in one of the Acid3 tests. I haven't been
		// able to replicate it outside of the test. The bug is that indexOf returns -1 when called on an Array that
		// contains just the document as a single element and the value searched for is the document.
		var arrayContains =
		/*Array.prototype.indexOf ?
        function(arr, val) {
            return arr.indexOf(val) > -1;
        }:*/

			function (arr, val) {
				var i = arr.length;
				while (i--) {
					if (arr[i] === val) {
						return true;
					}
				}
				return false;
			};

		// Opera 11 puts HTML elements in the null namespace, it seems, and IE 7 has undefined namespaceURI
		function isHtmlNamespace(node) {
			var ns;
			return typeof node.namespaceURI == UNDEF || ((ns = node.namespaceURI) === null || ns == "http://www.w3.org/1999/xhtml");
		}

		function parentElement(node) {
			var parent = node.parentNode;
			return (parent.nodeType == 1) ? parent : null;
		}

		function getNodeIndex(node) {
			var i = 0;
			while ((node = node.previousSibling)) {
				i++;
			}
			return i;
		}

		function getNodeLength(node) {
			var childNodes;
			return isCharacterDataNode(node) ? node.length : ((childNodes = node.childNodes) ? childNodes.length : 0);
		}

		function getCommonAncestor(node1, node2) {
			var ancestors = [],
				n;
			for (n = node1; n; n = n.parentNode) {
				ancestors.push(n);
			}

			for (n = node2; n; n = n.parentNode) {
				if (arrayContains(ancestors, n)) {
					return n;
				}
			}

			return null;
		}

		function isAncestorOf(ancestor, descendant, selfIsAncestor) {
			var n = selfIsAncestor ? descendant : descendant.parentNode;
			while (n) {
				if (n === ancestor) {
					return true;
				} else {
					n = n.parentNode;
				}
			}
			return false;
		}

		function getClosestAncestorIn(node, ancestor, selfIsAncestor) {
			var p, n = selfIsAncestor ? node : node.parentNode;
			while (n) {
				p = n.parentNode;
				if (p === ancestor) {
					return n;
				}
				n = p;
			}
			return null;
		}

		function isCharacterDataNode(node) {
			var t = node.nodeType;
			return t == 3 || t == 4 || t == 8; // Text, CDataSection or Comment
		}

		function insertAfter(node, precedingNode) {
			var nextNode = precedingNode.nextSibling,
				parent = precedingNode.parentNode;
			if (nextNode) {
				parent.insertBefore(node, nextNode);
			} else {
				parent.appendChild(node);
			}
			return node;
		}

		// Note that we cannot use splitText() because it is bugridden in IE 9.
		function splitDataNode(node, index) {
			var newNode = node.cloneNode(false);
			newNode.deleteData(0, index);
			node.deleteData(index, node.length - index);
			insertAfter(newNode, node);
			return newNode;
		}

		function getDocument(node) {
			if (node.nodeType == 9) {
				return node;
			} else if (typeof node.ownerDocument != UNDEF) {
				return node.ownerDocument;
			} else if (typeof node.document != UNDEF) {
				return node.document;
			} else if (node.parentNode) {
				return getDocument(node.parentNode);
			} else {
				throw new Error("getDocument: no document found for node");
			}
		}

		function getWindow(node) {
			var doc = getDocument(node);
			if (typeof doc.defaultView != UNDEF) {
				return doc.defaultView;
			} else if (typeof doc.parentWindow != UNDEF) {
				return doc.parentWindow;
			} else {
				throw new Error("Cannot get a window object for node");
			}
		}

		function getIframeDocument(iframeEl) {
			if (typeof iframeEl.contentDocument != UNDEF) {
				return iframeEl.contentDocument;
			} else if (typeof iframeEl.contentWindow != UNDEF) {
				return iframeEl.contentWindow.document;
			} else {
				throw new Error("getIframeWindow: No Document object found for iframe element");
			}
		}

		function getIframeWindow(iframeEl) {
			if (typeof iframeEl.contentWindow != UNDEF) {
				return iframeEl.contentWindow;
			} else if (typeof iframeEl.contentDocument != UNDEF) {
				return iframeEl.contentDocument.defaultView;
			} else {
				throw new Error("getIframeWindow: No Window object found for iframe element");
			}
		}

		function getBody(doc) {
			return util.isHostObject(doc, "body") ? doc.body : doc.getElementsByTagName("body")[0];
		}

		function getRootContainer(node) {
			var parent;
			while ((parent = node.parentNode)) {
				node = parent;
			}
			return node;
		}

		/**
		 * This is a very ugly workaround for an IE9 issue Before comparing DOM
		 * elements "normalize" them. There are cases, where anchorNode and
		 * focusNode in a nativeselection point to DOM elements with same
		 * parentNode, same previousSibling and same nextSibling, but the nodes
		 * themselves are not the same
		 * If such nodes are compared in the comparePoints method, an error occurs.
		 * To fix this, we move to the previousSibling/nextSibling/parentNode and back, to hopefully get
		 * the "correct" node in the DOM
		 * @param node node to fix
		 * @return normalized node
		 */
		function fixNode(node) {
			if (!node) {
				return;
			}
			if (node.previousSibling) {
				return node.previousSibling.nextSibling;
			} else if (node.nextSibling) {
				return node.nextSibling.previousSibling;
			} else if (node.parentNode) {
				return node.parentNode.firstChild;
			} else {
				return node;
			}
		}

		function comparePoints(nodeA, offsetA, nodeB, offsetB) {
			// fix the nodes before comparing them
			nodeA = fixNode(nodeA);
			nodeB = fixNode(nodeB);
			// See http://www.w3.org/TR/DOM-Level-2-Traversal-Range/ranges.html#Level-2-Range-Comparing
			var nodeC, root, childA, childB, n;
			if (nodeA == nodeB) {

				// Case 1: nodes are the same
				return offsetA === offsetB ? 0 : (offsetA < offsetB) ? -1 : 1;
			} else if ((nodeC = getClosestAncestorIn(nodeB, nodeA, true))) {

				// Case 2: node C (container B or an ancestor) is a child node of A
				return offsetA <= getNodeIndex(nodeC) ? -1 : 1;
			} else if ((nodeC = getClosestAncestorIn(nodeA, nodeB, true))) {

				// Case 3: node C (container A or an ancestor) is a child node of B
				return getNodeIndex(nodeC) < offsetB ? -1 : 1;
			} else {

				// Case 4: containers are siblings or descendants of siblings
				root = getCommonAncestor(nodeA, nodeB);
				childA = (nodeA === root) ? root : getClosestAncestorIn(nodeA, root, true);
				childB = (nodeB === root) ? root : getClosestAncestorIn(nodeB, root, true);

				if (childA === childB) {
					// This shouldn't be possible

					throw new Error("comparePoints got to case 4 and childA and childB are the same!");
				} else {
					n = root.firstChild;
					while (n) {
						if (n === childA) {
							return -1;
						} else if (n === childB) {
							return 1;
						}
						n = n.nextSibling;
					}
					throw new Error("Should not be here!");
				}
			}
		}

		function fragmentFromNodeChildren(node) {
			var fragment = getDocument(node).createDocumentFragment(),
				child;
			while ((child = node.firstChild)) {
				fragment.appendChild(child);
			}
			return fragment;
		}

		function inspectNode(node) {
			if (!node) {
				return "[No node]";
			}
			if (isCharacterDataNode(node)) {
				return '"' + node.data + '"';
			} else if (node.nodeType == 1) {
				var idAttr = node.id ? ' id="' + node.id + '"' : "";
				return "<" + node.nodeName + idAttr + ">[" + node.childNodes.length + "]";
			} else {
				return node.nodeName;
			}
		}

		/**
		 * @constructor
		 */
		function NodeIterator(root) {
			this.root = root;
			this._next = root;
		}

		NodeIterator.prototype = {
			_current: null,

			hasNext: function () {
				return !!this._next;
			},

			next: function () {
				var n = this._current = this._next;
				var child, next;
				if (this._current) {
					child = n.firstChild;
					if (child) {
						this._next = child;
					} else {
						next = null;
						while ((n !== this.root) && !(next = n.nextSibling)) {
							n = n.parentNode;
						}
						this._next = next;
					}
				}
				return this._current;
			},

			detach: function () {
				this._current = this._next = this.root = null;
			}
		};

		function createIterator(root) {
			return new NodeIterator(root);
		}

		/**
		 * @constructor
		 */
		function DomPosition(node, offset) {
			this.node = node;
			this.offset = offset;
		}

		DomPosition.prototype = {
			equals: function (pos) {
				return this.node === pos.node & this.offset == pos.offset;
			},

			inspect: function () {
				return "[DomPosition(" + inspectNode(this.node) + ":" + this.offset + ")]";
			}
		};

		/**
		 * @constructor
		 */
		function DOMException(codeName) {
			this.code = this[codeName];
			this.codeName = codeName;
			this.message = "DOMException: " + this.codeName;
		}

		DOMException.prototype = {
			INDEX_SIZE_ERR: 1,
			HIERARCHY_REQUEST_ERR: 3,
			WRONG_DOCUMENT_ERR: 4,
			NO_MODIFICATION_ALLOWED_ERR: 7,
			NOT_FOUND_ERR: 8,
			NOT_SUPPORTED_ERR: 9,
			INVALID_STATE_ERR: 11
		};

		DOMException.prototype.toString = function () {
			return this.message;
		};

		api.dom = {
			arrayContains: arrayContains,
			isHtmlNamespace: isHtmlNamespace,
			parentElement: parentElement,
			getNodeIndex: getNodeIndex,
			getNodeLength: getNodeLength,
			getCommonAncestor: getCommonAncestor,
			isAncestorOf: isAncestorOf,
			getClosestAncestorIn: getClosestAncestorIn,
			isCharacterDataNode: isCharacterDataNode,
			insertAfter: insertAfter,
			splitDataNode: splitDataNode,
			getDocument: getDocument,
			getWindow: getWindow,
			getIframeWindow: getIframeWindow,
			getIframeDocument: getIframeDocument,
			getBody: getBody,
			getRootContainer: getRootContainer,
			comparePoints: comparePoints,
			inspectNode: inspectNode,
			fragmentFromNodeChildren: fragmentFromNodeChildren,
			createIterator: createIterator,
			DomPosition: DomPosition
		};

		api.DOMException = DOMException;
	});
	rangy.createModule("DomRange", function (api, module) {
		api.requireModules(["DomUtil"]);


		var dom = api.dom;
		var DomPosition = dom.DomPosition;
		var DOMException = api.DOMException;

		/*----------------------------------------------------------------------------------------------------------------*/

		// Utility functions

		function isNonTextPartiallySelected(node, range) {
			return (node.nodeType != 3) && (dom.isAncestorOf(node, range.startContainer, true) || dom.isAncestorOf(node, range.endContainer, true));
		}

		function getRangeDocument(range) {
			return dom.getDocument(range.startContainer);
		}

		function dispatchEvent(range, type, args) {
			var listeners = range._listeners[type];
			if (listeners) {
				for (var i = 0, len = listeners.length; i < len; ++i) {
					listeners[i].call(range, {
						target: range,
						args: args
					});
				}
			}
		}

		function getBoundaryBeforeNode(node) {
			return new DomPosition(node.parentNode, dom.getNodeIndex(node));
		}

		function getBoundaryAfterNode(node) {
			return new DomPosition(node.parentNode, dom.getNodeIndex(node) + 1);
		}

		function insertNodeAtPosition(node, n, o) {
			var firstNodeInserted = node.nodeType == 11 ? node.firstChild : node;
			if (dom.isCharacterDataNode(n)) {
				if (o == n.length) {
					dom.insertAfter(node, n);
				} else {
					n.parentNode.insertBefore(node, o == 0 ? n : dom.splitDataNode(n, o));
				}
			} else if (o >= n.childNodes.length) {
				n.appendChild(node);
			} else {
				n.insertBefore(node, n.childNodes[o]);
			}
			return firstNodeInserted;
		}

		function cloneSubtree(iterator) {
			var partiallySelected;
			for (var node, frag = getRangeDocument(iterator.range).createDocumentFragment(), subIterator; node = iterator.next();) {
				partiallySelected = iterator.isPartiallySelectedSubtree();

				node = node.cloneNode(!partiallySelected);
				if (partiallySelected) {
					subIterator = iterator.getSubtreeIterator();
					node.appendChild(cloneSubtree(subIterator));
					subIterator.detach(true);
				}

				if (node.nodeType == 10) { // DocumentType
					throw new DOMException("HIERARCHY_REQUEST_ERR");
				}
				frag.appendChild(node);
			}
			return frag;
		}

		function iterateSubtree(rangeIterator, func, iteratorState) {
			var it, n;
			iteratorState = iteratorState || {
				stop: false
			};
			for (var node, subRangeIterator; node = rangeIterator.next();) {
				//log.debug("iterateSubtree, partially selected: " + rangeIterator.isPartiallySelectedSubtree(), nodeToString(node));
				if (rangeIterator.isPartiallySelectedSubtree()) {
					// The node is partially selected by the Range, so we can use a new RangeIterator on the portion of the
					// node selected by the Range.
					if (func(node) === false) {
						iteratorState.stop = true;
						return;
					} else {
						subRangeIterator = rangeIterator.getSubtreeIterator();
						iterateSubtree(subRangeIterator, func, iteratorState);
						subRangeIterator.detach(true);
						if (iteratorState.stop) {
							return;
						}
					}
				} else {
					// The whole node is selected, so we can use efficient DOM iteration to iterate over the node and its
					// descendant
					it = dom.createIterator(node);
					while ((n = it.next())) {
						if (func(n) === false) {
							iteratorState.stop = true;
							return;
						}
					}
				}
			}
		}

		function deleteSubtree(iterator) {
			var subIterator;
			while (iterator.next()) {
				if (iterator.isPartiallySelectedSubtree()) {
					subIterator = iterator.getSubtreeIterator();
					deleteSubtree(subIterator);
					subIterator.detach(true);
				} else {
					iterator.remove();
				}
			}
		}

		function extractSubtree(iterator) {

			for (var node, frag = getRangeDocument(iterator.range).createDocumentFragment(), subIterator; node = iterator.next();) {


				if (iterator.isPartiallySelectedSubtree()) {
					node = node.cloneNode(false);
					subIterator = iterator.getSubtreeIterator();
					node.appendChild(extractSubtree(subIterator));
					subIterator.detach(true);
				} else {
					iterator.remove();
				}
				if (node.nodeType == 10) { // DocumentType
					throw new DOMException("HIERARCHY_REQUEST_ERR");
				}
				frag.appendChild(node);
			}
			return frag;
		}

		function getNodesInRange(range, nodeTypes, filter) {
			//log.info("getNodesInRange, " + nodeTypes.join(","));
			var filterNodeTypes = !! (nodeTypes && nodeTypes.length),
				regex;
			var filterExists = !! filter;
			if (filterNodeTypes) {
				regex = new RegExp("^(" + nodeTypes.join("|") + ")$");
			}

			var nodes = [];
			iterateSubtree(new RangeIterator(range, false), function (node) {
				if ((!filterNodeTypes || regex.test(node.nodeType)) && (!filterExists || filter(node))) {
					nodes.push(node);
				}
			});
			return nodes;
		}

		function inspect(range) {
			var name = (typeof range.getName == "undefined") ? "Range" : range.getName();
			return "[" + name + "(" + dom.inspectNode(range.startContainer) + ":" + range.startOffset + ", " + dom.inspectNode(range.endContainer) + ":" + range.endOffset + ")]";
		}

		/*----------------------------------------------------------------------------------------------------------------*/

		// RangeIterator code partially borrows from IERange by Tim Ryan (http://github.com/timcameronryan/IERange)

		/**
		 * @constructor
		 */
		function RangeIterator(range, clonePartiallySelectedTextNodes) {
			this.range = range;
			this.clonePartiallySelectedTextNodes = clonePartiallySelectedTextNodes;



			if (!range.collapsed) {
				this.sc = range.startContainer;
				this.so = range.startOffset;
				this.ec = range.endContainer;
				this.eo = range.endOffset;
				var root = range.commonAncestorContainer;

				if (this.sc === this.ec && dom.isCharacterDataNode(this.sc)) {
					this.isSingleCharacterDataNode = true;
					this._first = this._last = this._next = this.sc;
				} else {
					this._first = this._next = (this.sc === root && !dom.isCharacterDataNode(this.sc)) ? this.sc.childNodes[this.so] : dom.getClosestAncestorIn(this.sc, root, true);
					this._last = (this.ec === root && !dom.isCharacterDataNode(this.ec)) ? this.ec.childNodes[this.eo - 1] : dom.getClosestAncestorIn(this.ec, root, true);
				}

			}
		}

		RangeIterator.prototype = {
			_current: null,
			_next: null,
			_first: null,
			_last: null,
			isSingleCharacterDataNode: false,

			reset: function () {
				this._current = null;
				this._next = this._first;
			},

			hasNext: function () {
				return !!this._next;
			},

			next: function () {
				// Move to next node
				var current = this._current = this._next;
				if (current) {
					this._next = (current !== this._last) ? current.nextSibling : null;

					// Check for partially selected text nodes
					if (dom.isCharacterDataNode(current) && this.clonePartiallySelectedTextNodes) {
						if (current === this.ec) {

							(current = current.cloneNode(true)).deleteData(this.eo, current.length - this.eo);
						}
						if (this._current === this.sc) {

							(current = current.cloneNode(true)).deleteData(0, this.so);
						}
					}
				}

				return current;
			},

			remove: function () {
				var current = this._current,
					start, end;

				if (dom.isCharacterDataNode(current) && (current === this.sc || current === this.ec)) {
					start = (current === this.sc) ? this.so : 0;
					end = (current === this.ec) ? this.eo : current.length;
					if (start != end) {
						current.deleteData(start, end - start);
					}
				} else {
					if (current.parentNode) {
						current.parentNode.removeChild(current);
					} else {

					}
				}
			},

			// Checks if the current node is partially selected
			isPartiallySelectedSubtree: function () {
				var current = this._current;
				return isNonTextPartiallySelected(current, this.range);
			},

			getSubtreeIterator: function () {
				var subRange;
				if (this.isSingleCharacterDataNode) {
					subRange = this.range.cloneRange();
					subRange.collapse();
				} else {
					subRange = new Range(getRangeDocument(this.range));
					var current = this._current;
					var startContainer = current,
						startOffset = 0,
						endContainer = current,
						endOffset = dom.getNodeLength(current);

					if (dom.isAncestorOf(current, this.sc, true)) {
						startContainer = this.sc;
						startOffset = this.so;
					}
					if (dom.isAncestorOf(current, this.ec, true)) {
						endContainer = this.ec;
						endOffset = this.eo;
					}

					updateBoundaries(subRange, startContainer, startOffset, endContainer, endOffset);
				}
				return new RangeIterator(subRange, this.clonePartiallySelectedTextNodes);
			},

			detach: function (detachRange) {
				if (detachRange) {
					this.range.detach();
				}
				this.range = this._current = this._next = this._first = this._last = this.sc = this.so = this.ec = this.eo = null;
			}
		};

		/*----------------------------------------------------------------------------------------------------------------*/

		// Exceptions

		/**
		 * @constructor
		 */
		function RangeException(codeName) {
			this.code = this[codeName];
			this.codeName = codeName;
			this.message = "RangeException: " + this.codeName;
		}

		RangeException.prototype = {
			BAD_BOUNDARYPOINTS_ERR: 1,
			INVALID_NODE_TYPE_ERR: 2
		};

		RangeException.prototype.toString = function () {
			return this.message;
		};

		/*----------------------------------------------------------------------------------------------------------------*/

		/**
		 * Currently iterates through all nodes in the range on creation until I think of a decent way to do it
		 * TODO: Look into making this a proper iterator, not requiring preloading everything first
		 * @constructor
		 */
		function RangeNodeIterator(range, nodeTypes, filter) {
			this.nodes = getNodesInRange(range, nodeTypes, filter);
			this._next = this.nodes[0];
			this._position = 0;
		}

		RangeNodeIterator.prototype = {
			_current: null,

			hasNext: function () {
				return !!this._next;
			},

			next: function () {
				this._current = this._next;
				this._next = this.nodes[++this._position];
				return this._current;
			},

			detach: function () {
				this._current = this._next = this.nodes = null;
			}
		};

		var beforeAfterNodeTypes = [1, 3, 4, 5, 7, 8, 10];
		var rootContainerNodeTypes = [2, 9, 11];
		var readonlyNodeTypes = [5, 6, 10, 12];
		var insertableNodeTypes = [1, 3, 4, 5, 7, 8, 10, 11];
		var surroundNodeTypes = [1, 3, 4, 5, 7, 8];

		function createAncestorFinder(nodeTypes) {
			return function (node, selfIsAncestor) {
				var t, n = selfIsAncestor ? node : node.parentNode;
				while (n) {
					t = n.nodeType;
					if (dom.arrayContains(nodeTypes, t)) {
						return n;
					}
					n = n.parentNode;
				}
				return null;
			};
		}

		var getRootContainer = dom.getRootContainer;
		var getDocumentOrFragmentContainer = createAncestorFinder([9, 11]);
		var getReadonlyAncestor = createAncestorFinder(readonlyNodeTypes);
		var getDocTypeNotationEntityAncestor = createAncestorFinder([6, 10, 12]);

		function assertNoDocTypeNotationEntityAncestor(node, allowSelf) {
			if (getDocTypeNotationEntityAncestor(node, allowSelf)) {
				throw new RangeException("INVALID_NODE_TYPE_ERR");
			}
		}

		function assertNotDetached(range) {
			if (!range.startContainer) {
				throw new DOMException("INVALID_STATE_ERR");
			}
		}

		function assertValidNodeType(node, invalidTypes) {
			if (!dom.arrayContains(invalidTypes, node.nodeType)) {
				throw new RangeException("INVALID_NODE_TYPE_ERR");
			}
		}

		function assertValidOffset(node, offset) {
			if (offset < 0 || offset > (dom.isCharacterDataNode(node) ? node.length : node.childNodes.length)) {
				throw new DOMException("INDEX_SIZE_ERR");
			}
		}

		function assertSameDocumentOrFragment(node1, node2) {
			if (getDocumentOrFragmentContainer(node1, true) !== getDocumentOrFragmentContainer(node2, true)) {
				throw new DOMException("WRONG_DOCUMENT_ERR");
			}
		}

		function assertNodeNotReadOnly(node) {
			if (getReadonlyAncestor(node, true)) {
				throw new DOMException("NO_MODIFICATION_ALLOWED_ERR");
			}
		}

		function assertNode(node, codeName) {
			if (!node) {
				throw new DOMException(codeName);
			}
		}

		function isOrphan(node) {
			return !dom.arrayContains(rootContainerNodeTypes, node.nodeType) && !getDocumentOrFragmentContainer(node, true);
		}

		function isValidOffset(node, offset) {
			return offset <= (dom.isCharacterDataNode(node) ? node.length : node.childNodes.length);
		}

		function assertRangeValid(range) {
			assertNotDetached(range);
			if (isOrphan(range.startContainer) || isOrphan(range.endContainer) || !isValidOffset(range.startContainer, range.startOffset) || !isValidOffset(range.endContainer, range.endOffset)) {
				throw new Error("Range error: Range is no longer valid after DOM mutation (" + range.inspect() + ")");
			}
		}

		/*----------------------------------------------------------------------------------------------------------------*/

		// Test the browser's innerHTML support to decide how to implement createContextualFragment
		var styleEl = document.createElement("style");
		var htmlParsingConforms = false;
		try {
			styleEl.innerHTML = "<b>x</b>";
			htmlParsingConforms = (styleEl.firstChild.nodeType == 3); // Opera incorrectly creates an element node
		} catch (e) {
			// IE 6 and 7 throw
		}

		api.features.htmlParsingConforms = htmlParsingConforms;

		var createContextualFragment = htmlParsingConforms ?

		// Implementation as per HTML parsing spec, trusting in the browser's implementation of innerHTML. See
		// discussion and base code for this implementation at issue 67.
		// Spec: http://html5.org/specs/dom-parsing.html#extensions-to-the-range-interface
		// Thanks to Aleks Williams.
			function (fragmentStr) {
				// "Let node the context object's start's node."
				var node = this.startContainer;
				var doc = dom.getDocument(node);

				// "If the context object's start's node is null, raise an INVALID_STATE_ERR
				// exception and abort these steps."
				if (!node) {
					throw new DOMException("INVALID_STATE_ERR");
				}

				// "Let element be as follows, depending on node's interface:"
				// Document, Document Fragment: null
				var el = null;

				// "Element: node"
				if (node.nodeType == 1) {
					el = node;

					// "Text, Comment: node's parentElement"
				} else if (dom.isCharacterDataNode(node)) {
					el = dom.parentElement(node);
				}

				// "If either element is null or element's ownerDocument is an HTML document
				// and element's local name is "html" and element's namespace is the HTML
				// namespace"
				if (el === null || (
				el.nodeName == "HTML" && dom.isHtmlNamespace(dom.getDocument(el).documentElement) && dom.isHtmlNamespace(el))) {

					// "let element be a new Element with "body" as its local name and the HTML
					// namespace as its namespace.""
					el = doc.createElement("body");
				} else {
					el = el.cloneNode(false);
				}

				// "If the node's document is an HTML document: Invoke the HTML fragment parsing algorithm."
				// "If the node's document is an XML document: Invoke the XML fragment parsing algorithm."
				// "In either case, the algorithm must be invoked with fragment as the input
				// and element as the context element."
				el.innerHTML = fragmentStr;

				// "If this raises an exception, then abort these steps. Otherwise, let new
				// children be the nodes returned."

				// "Let fragment be a new DocumentFragment."
				// "Append all new children to fragment."
				// "Return fragment."
				return dom.fragmentFromNodeChildren(el);
			} :

			// In this case, innerHTML cannot be trusted, so fall back to a simpler, non-conformant implementation that
			// previous versions of Rangy used (with the exception of using a body element rather than a div)
			function (fragmentStr) {
				assertNotDetached(this);
				var doc = getRangeDocument(this);
				var el = doc.createElement("body");
				el.innerHTML = fragmentStr;

				return dom.fragmentFromNodeChildren(el);
			};

		/*----------------------------------------------------------------------------------------------------------------*/

		var rangeProperties = ["startContainer", "startOffset", "endContainer", "endOffset", "collapsed", "commonAncestorContainer"];

		var s2s = 0,
			s2e = 1,
			e2e = 2,
			e2s = 3;
		var n_b = 0,
			n_a = 1,
			n_b_a = 2,
			n_i = 3;

		function RangePrototype() {}

		RangePrototype.prototype = {
			attachListener: function (type, listener) {
				this._listeners[type].push(listener);
			},

			compareBoundaryPoints: function (how, range) {
				assertRangeValid(this);
				assertSameDocumentOrFragment(this.startContainer, range.startContainer);

				var nodeA, offsetA, nodeB, offsetB;
				var prefixA = (how == e2s || how == s2s) ? "start" : "end";
				var prefixB = (how == s2e || how == s2s) ? "start" : "end";
				nodeA = this[prefixA + "Container"];
				offsetA = this[prefixA + "Offset"];
				nodeB = range[prefixB + "Container"];
				offsetB = range[prefixB + "Offset"];
				return dom.comparePoints(nodeA, offsetA, nodeB, offsetB);
			},

			insertNode: function (node) {
				assertRangeValid(this);
				assertValidNodeType(node, insertableNodeTypes);
				assertNodeNotReadOnly(this.startContainer);

				if (dom.isAncestorOf(node, this.startContainer, true)) {
					throw new DOMException("HIERARCHY_REQUEST_ERR");
				}

				// No check for whether the container of the start of the Range is of a type that does not allow
				// children of the type of node: the browser's DOM implementation should do this for us when we attempt
				// to add the node

				var firstNodeInserted = insertNodeAtPosition(node, this.startContainer, this.startOffset);
				this.setStartBefore(firstNodeInserted);
			},

			cloneContents: function () {
				assertRangeValid(this);

				var clone, frag;
				if (this.collapsed) {
					return getRangeDocument(this).createDocumentFragment();
				} else {
					if (this.startContainer === this.endContainer && dom.isCharacterDataNode(this.startContainer)) {
						clone = this.startContainer.cloneNode(true);
						clone.data = clone.data.slice(this.startOffset, this.endOffset);
						frag = getRangeDocument(this).createDocumentFragment();
						frag.appendChild(clone);
						return frag;
					} else {
						var iterator = new RangeIterator(this, true);
						clone = cloneSubtree(iterator);
						iterator.detach();
					}
					return clone;
				}
			},

			canSurroundContents: function () {
				assertRangeValid(this);
				assertNodeNotReadOnly(this.startContainer);
				assertNodeNotReadOnly(this.endContainer);

				// Check if the contents can be surrounded. Specifically, this means whether the range partially selects
				// no non-text nodes.
				var iterator = new RangeIterator(this, true);
				var boundariesInvalid = (iterator._first && (isNonTextPartiallySelected(iterator._first, this)) || (iterator._last && isNonTextPartiallySelected(iterator._last, this)));
				iterator.detach();
				return !boundariesInvalid;
			},

			surroundContents: function (node) {
				assertValidNodeType(node, surroundNodeTypes);

				if (!this.canSurroundContents()) {
					throw new RangeException("BAD_BOUNDARYPOINTS_ERR");
				}

				// Extract the contents
				var content = this.extractContents();

				// Clear the children of the node
				if (node.hasChildNodes()) {
					while (node.lastChild) {
						node.removeChild(node.lastChild);
					}
				}

				// Insert the new node and add the extracted contents
				insertNodeAtPosition(node, this.startContainer, this.startOffset);
				node.appendChild(content);

				this.selectNode(node);
			},

			cloneRange: function () {
				assertRangeValid(this);
				var range = new Range(getRangeDocument(this));
				var i = rangeProperties.length,
					prop;
				while (i--) {
					prop = rangeProperties[i];
					range[prop] = this[prop];
				}
				return range;
			},

			toString: function () {
				assertRangeValid(this);
				var sc = this.startContainer;
				if (sc === this.endContainer && dom.isCharacterDataNode(sc)) {
					return (sc.nodeType == 3 || sc.nodeType == 4) ? sc.data.slice(this.startOffset, this.endOffset) : "";
				} else {
					var textBits = [],
						iterator = new RangeIterator(this, true);

					iterateSubtree(iterator, function (node) {
						// Accept only text or CDATA nodes, not comments

						if (node.nodeType == 3 || node.nodeType == 4) {
							textBits.push(node.data);
						}
					});
					iterator.detach();
					return textBits.join("");
				}
			},

			// The methods below are all non-standard. The following batch were introduced by Mozilla but have since
			// been removed from Mozilla.

			compareNode: function (node) {
				assertRangeValid(this);

				var parent = node.parentNode;
				var nodeIndex = dom.getNodeIndex(node);

				if (!parent) {
					throw new DOMException("NOT_FOUND_ERR");
				}

				var startComparison = this.comparePoint(parent, nodeIndex),
					endComparison = this.comparePoint(parent, nodeIndex + 1);

				if (startComparison < 0) { // Node starts before
					return (endComparison > 0) ? n_b_a : n_b;
				} else {
					return (endComparison > 0) ? n_a : n_i;
				}
			},

			comparePoint: function (node, offset) {
				assertRangeValid(this);
				assertNode(node, "HIERARCHY_REQUEST_ERR");
				assertSameDocumentOrFragment(node, this.startContainer);

				if (dom.comparePoints(node, offset, this.startContainer, this.startOffset) < 0) {
					return -1;
				} else if (dom.comparePoints(node, offset, this.endContainer, this.endOffset) > 0) {
					return 1;
				}
				return 0;
			},

			createContextualFragment: createContextualFragment,

			toHtml: function () {
				assertRangeValid(this);
				var container = getRangeDocument(this).createElement("div");
				container.appendChild(this.cloneContents());
				return container.innerHTML;
			},

			// touchingIsIntersecting determines whether this method considers a node that borders a range intersects
			// with it (as in WebKit) or not (as in Gecko pre-1.9, and the default)
			intersectsNode: function (node, touchingIsIntersecting) {
				assertRangeValid(this);
				assertNode(node, "NOT_FOUND_ERR");
				if (dom.getDocument(node) !== getRangeDocument(this)) {
					return false;
				}

				var parent = node.parentNode,
					offset = dom.getNodeIndex(node);
				assertNode(parent, "NOT_FOUND_ERR");

				var startComparison = dom.comparePoints(parent, offset, this.endContainer, this.endOffset),
					endComparison = dom.comparePoints(parent, offset + 1, this.startContainer, this.startOffset);

				return touchingIsIntersecting ? startComparison <= 0 && endComparison >= 0 : startComparison < 0 && endComparison > 0;
			},


			isPointInRange: function (node, offset) {
				assertRangeValid(this);
				assertNode(node, "HIERARCHY_REQUEST_ERR");
				assertSameDocumentOrFragment(node, this.startContainer);

				return (dom.comparePoints(node, offset, this.startContainer, this.startOffset) >= 0) && (dom.comparePoints(node, offset, this.endContainer, this.endOffset) <= 0);
			},

			// The methods below are non-standard and invented by me.

			// Sharing a boundary start-to-end or end-to-start does not count as intersection.
			intersectsRange: function (range, touchingIsIntersecting) {
				assertRangeValid(this);

				if (getRangeDocument(range) != getRangeDocument(this)) {
					throw new DOMException("WRONG_DOCUMENT_ERR");
				}

				var startComparison = dom.comparePoints(this.startContainer, this.startOffset, range.endContainer, range.endOffset),
					endComparison = dom.comparePoints(this.endContainer, this.endOffset, range.startContainer, range.startOffset);

				return touchingIsIntersecting ? startComparison <= 0 && endComparison >= 0 : startComparison < 0 && endComparison > 0;
			},

			intersection: function (range) {
				if (this.intersectsRange(range)) {
					var startComparison = dom.comparePoints(this.startContainer, this.startOffset, range.startContainer, range.startOffset),
						endComparison = dom.comparePoints(this.endContainer, this.endOffset, range.endContainer, range.endOffset);

					var intersectionRange = this.cloneRange();

					if (startComparison == -1) {
						intersectionRange.setStart(range.startContainer, range.startOffset);
					}
					if (endComparison == 1) {
						intersectionRange.setEnd(range.endContainer, range.endOffset);
					}
					return intersectionRange;
				}
				return null;
			},

			union: function (range) {
				if (this.intersectsRange(range, true)) {
					var unionRange = this.cloneRange();
					if (dom.comparePoints(range.startContainer, range.startOffset, this.startContainer, this.startOffset) == -1) {
						unionRange.setStart(range.startContainer, range.startOffset);
					}
					if (dom.comparePoints(range.endContainer, range.endOffset, this.endContainer, this.endOffset) == 1) {
						unionRange.setEnd(range.endContainer, range.endOffset);
					}
					return unionRange;
				} else {
					throw new RangeException("Ranges do not intersect");
				}
			},

			containsNode: function (node, allowPartial) {
				if (allowPartial) {
					return this.intersectsNode(node, false);
				} else {
					return this.compareNode(node) == n_i;
				}
			},

			containsNodeContents: function (node) {
				return this.comparePoint(node, 0) >= 0 && this.comparePoint(node, dom.getNodeLength(node)) <= 0;
			},

			containsRange: function (range) {
				return this.intersection(range).equals(range);
			},

			containsNodeText: function (node) {
				var nodeRange = this.cloneRange();
				nodeRange.selectNode(node);
				var textNodes = nodeRange.getNodes([3]);
				if (textNodes.length > 0) {
					nodeRange.setStart(textNodes[0], 0);
					var lastTextNode = textNodes.pop();
					nodeRange.setEnd(lastTextNode, lastTextNode.length);
					var contains = this.containsRange(nodeRange);
					nodeRange.detach();
					return contains;
				} else {
					return this.containsNodeContents(node);
				}
			},

			createNodeIterator: function (nodeTypes, filter) {
				assertRangeValid(this);
				return new RangeNodeIterator(this, nodeTypes, filter);
			},

			getNodes: function (nodeTypes, filter) {
				assertRangeValid(this);
				return getNodesInRange(this, nodeTypes, filter);
			},

			getDocument: function () {
				return getRangeDocument(this);
			},

			collapseBefore: function (node) {
				assertNotDetached(this);

				this.setEndBefore(node);
				this.collapse(false);
			},

			collapseAfter: function (node) {
				assertNotDetached(this);

				this.setStartAfter(node);
				this.collapse(true);
			},

			getName: function () {
				return "DomRange";
			},

			equals: function (range) {
				return Range.rangesEqual(this, range);
			},

			inspect: function () {
				return inspect(this);
			}
		};

		function copyComparisonConstantsToObject(obj) {
			obj.START_TO_START = s2s;
			obj.START_TO_END = s2e;
			obj.END_TO_END = e2e;
			obj.END_TO_START = e2s;

			obj.NODE_BEFORE = n_b;
			obj.NODE_AFTER = n_a;
			obj.NODE_BEFORE_AND_AFTER = n_b_a;
			obj.NODE_INSIDE = n_i;
		}

		function copyComparisonConstants(constructor) {
			copyComparisonConstantsToObject(constructor);
			copyComparisonConstantsToObject(constructor.prototype);
		}

		function createRangeContentRemover(remover, boundaryUpdater) {
			return function () {
				assertRangeValid(this);

				var sc = this.startContainer,
					so = this.startOffset,
					root = this.commonAncestorContainer;

				var iterator = new RangeIterator(this, true);

				// Work out where to position the range after content removal
				var node, boundary;
				if (sc !== root) {
					node = dom.getClosestAncestorIn(sc, root, true);
					boundary = getBoundaryAfterNode(node);
					sc = boundary.node;
					so = boundary.offset;
				}

				// Check none of the range is read-only
				iterateSubtree(iterator, assertNodeNotReadOnly);

				iterator.reset();

				// Remove the content
				var returnValue = remover(iterator);
				iterator.detach();

				// Move to the new position
				boundaryUpdater(this, sc, so, sc, so);

				return returnValue;
			};
		}

		function createPrototypeRange(constructor, boundaryUpdater, detacher) {
			function createBeforeAfterNodeSetter(isBefore, isStart) {
				return function (node) {
					assertNotDetached(this);
					assertValidNodeType(node, beforeAfterNodeTypes);
					assertValidNodeType(getRootContainer(node), rootContainerNodeTypes);

					var boundary = (isBefore ? getBoundaryBeforeNode : getBoundaryAfterNode)(node);
					(isStart ? setRangeStart : setRangeEnd)(this, boundary.node, boundary.offset);
				};
			}

			function setRangeStart(range, node, offset) {
				var ec = range.endContainer,
					eo = range.endOffset;
				if (node !== range.startContainer || offset !== this.startOffset) {
					// Check the root containers of the range and the new boundary, and also check whether the new boundary
					// is after the current end. In either case, collapse the range to the new position
					if (getRootContainer(node) != getRootContainer(ec) || dom.comparePoints(node, offset, ec, eo) == 1) {
						ec = node;
						eo = offset;
					}
					boundaryUpdater(range, node, offset, ec, eo);
				}
			}

			function setRangeEnd(range, node, offset) {
				var sc = range.startContainer,
					so = range.startOffset;
				if (node !== range.endContainer || offset !== this.endOffset) {
					// Check the root containers of the range and the new boundary, and also check whether the new boundary
					// is after the current end. In either case, collapse the range to the new position
					if (getRootContainer(node) != getRootContainer(sc) || dom.comparePoints(node, offset, sc, so) == -1) {
						sc = node;
						so = offset;
					}
					boundaryUpdater(range, sc, so, node, offset);
				}
			}

			function setRangeStartAndEnd(range, node, offset) {
				if (node !== range.startContainer || offset !== this.startOffset || node !== range.endContainer || offset !== this.endOffset) {
					boundaryUpdater(range, node, offset, node, offset);
				}
			}

			constructor.prototype = new RangePrototype();

			api.util.extend(constructor.prototype, {
				setStart: function (node, offset) {
					assertNotDetached(this);
					assertNoDocTypeNotationEntityAncestor(node, true);
					assertValidOffset(node, offset);

					setRangeStart(this, node, offset);
				},

				setEnd: function (node, offset) {
					assertNotDetached(this);
					assertNoDocTypeNotationEntityAncestor(node, true);
					assertValidOffset(node, offset);

					setRangeEnd(this, node, offset);
				},

				setStartBefore: createBeforeAfterNodeSetter(true, true),
				setStartAfter: createBeforeAfterNodeSetter(false, true),
				setEndBefore: createBeforeAfterNodeSetter(true, false),
				setEndAfter: createBeforeAfterNodeSetter(false, false),

				collapse: function (isStart) {
					assertRangeValid(this);
					if (isStart) {
						boundaryUpdater(this, this.startContainer, this.startOffset, this.startContainer, this.startOffset);
					} else {
						boundaryUpdater(this, this.endContainer, this.endOffset, this.endContainer, this.endOffset);
					}
				},

				selectNodeContents: function (node) {
					// This doesn't seem well specified: the spec talks only about selecting the node's contents, which
					// could be taken to mean only its children. However, browsers implement this the same as selectNode for
					// text nodes, so I shall do likewise
					assertNotDetached(this);
					assertNoDocTypeNotationEntityAncestor(node, true);

					boundaryUpdater(this, node, 0, node, dom.getNodeLength(node));
				},

				selectNode: function (node) {
					assertNotDetached(this);
					assertNoDocTypeNotationEntityAncestor(node, false);
					assertValidNodeType(node, beforeAfterNodeTypes);

					var start = getBoundaryBeforeNode(node),
						end = getBoundaryAfterNode(node);
					boundaryUpdater(this, start.node, start.offset, end.node, end.offset);
				},

				extractContents: createRangeContentRemover(extractSubtree, boundaryUpdater),

				deleteContents: createRangeContentRemover(deleteSubtree, boundaryUpdater),

				canSurroundContents: function () {
					assertRangeValid(this);
					assertNodeNotReadOnly(this.startContainer);
					assertNodeNotReadOnly(this.endContainer);

					// Check if the contents can be surrounded. Specifically, this means whether the range partially selects
					// no non-text nodes.
					var iterator = new RangeIterator(this, true);
					var boundariesInvalid = (iterator._first && (isNonTextPartiallySelected(iterator._first, this)) || (iterator._last && isNonTextPartiallySelected(iterator._last, this)));
					iterator.detach();
					return !boundariesInvalid;
				},

				detach: function () {
					detacher(this);
				},

				splitBoundaries: function () {
					assertRangeValid(this);


					var sc = this.startContainer,
						so = this.startOffset,
						ec = this.endContainer,
						eo = this.endOffset;
					var startEndSame = (sc === ec);

					if (dom.isCharacterDataNode(ec) && eo > 0 && eo < ec.length) {
						dom.splitDataNode(ec, eo);

					}

					if (dom.isCharacterDataNode(sc) && so > 0 && so < sc.length) {

						sc = dom.splitDataNode(sc, so);
						if (startEndSame) {
							eo -= so;
							ec = sc;
						} else if (ec == sc.parentNode && eo >= dom.getNodeIndex(sc)) {
							eo++;
						}
						so = 0;

					}
					boundaryUpdater(this, sc, so, ec, eo);
				},

				normalizeBoundaries: function () {
					assertRangeValid(this);

					var sc = this.startContainer,
						so = this.startOffset,
						ec = this.endContainer,
						eo = this.endOffset;

					var mergeForward = function (node) {
						var sibling = node.nextSibling;
						if (sibling && sibling.nodeType == node.nodeType) {
							ec = node;
							eo = node.length;
							node.appendData(sibling.data);
							sibling.parentNode.removeChild(sibling);
						}
					};

					var mergeBackward = function (node) {
						var sibling = node.previousSibling;
						if (sibling && sibling.nodeType == node.nodeType) {
							sc = node;
							var nodeLength = node.length;
							so = sibling.length;
							node.insertData(0, sibling.data);
							sibling.parentNode.removeChild(sibling);
							if (sc == ec) {
								eo += so;
								ec = sc;
							} else if (ec == node.parentNode) {
								var nodeIndex = dom.getNodeIndex(node);
								if (eo == nodeIndex) {
									ec = node;
									eo = nodeLength;
								} else if (eo > nodeIndex) {
									eo--;
								}
							}
						}
					};

					var normalizeStart = true;

					if (dom.isCharacterDataNode(ec)) {
						if (ec.length == eo) {
							mergeForward(ec);
						}
					} else {
						if (eo > 0) {
							var endNode = ec.childNodes[eo - 1];
							if (endNode && dom.isCharacterDataNode(endNode)) {
								mergeForward(endNode);
							}
						}
						normalizeStart = !this.collapsed;
					}

					if (normalizeStart) {
						if (dom.isCharacterDataNode(sc)) {
							if (so == 0) {
								mergeBackward(sc);
							}
						} else {
							if (so < sc.childNodes.length) {
								var startNode = sc.childNodes[so];
								if (startNode && dom.isCharacterDataNode(startNode)) {
									mergeBackward(startNode);
								}
							}
						}
					} else {
						sc = ec;
						so = eo;
					}

					boundaryUpdater(this, sc, so, ec, eo);
				},

				collapseToPoint: function (node, offset) {
					assertNotDetached(this);

					assertNoDocTypeNotationEntityAncestor(node, true);
					assertValidOffset(node, offset);

					setRangeStartAndEnd(this, node, offset);
				}
			});

			copyComparisonConstants(constructor);
		}

		/*----------------------------------------------------------------------------------------------------------------*/

		// Updates commonAncestorContainer and collapsed after boundary change
		function updateCollapsedAndCommonAncestor(range) {
			range.collapsed = (range.startContainer === range.endContainer && range.startOffset === range.endOffset);
			range.commonAncestorContainer = range.collapsed ? range.startContainer : dom.getCommonAncestor(range.startContainer, range.endContainer);
		}

		function updateBoundaries(range, startContainer, startOffset, endContainer, endOffset) {
			var startMoved = (range.startContainer !== startContainer || range.startOffset !== startOffset);
			var endMoved = (range.endContainer !== endContainer || range.endOffset !== endOffset);

			range.startContainer = startContainer;
			range.startOffset = startOffset;
			range.endContainer = endContainer;
			range.endOffset = endOffset;

			updateCollapsedAndCommonAncestor(range);
			dispatchEvent(range, "boundarychange", {
				startMoved: startMoved,
				endMoved: endMoved
			});
		}

		function detach(range) {
			assertNotDetached(range);
			range.startContainer = range.startOffset = range.endContainer = range.endOffset = null;
			range.collapsed = range.commonAncestorContainer = null;
			dispatchEvent(range, "detach", null);
			range._listeners = null;
		}

		/**
		 * @constructor
		 */
		function Range(doc) {
			this.startContainer = doc;
			this.startOffset = 0;
			this.endContainer = doc;
			this.endOffset = 0;
			this._listeners = {
				boundarychange: [],
				detach: []
			};
			updateCollapsedAndCommonAncestor(this);
		}

		createPrototypeRange(Range, updateBoundaries, detach);

		api.rangePrototype = RangePrototype.prototype;

		Range.rangeProperties = rangeProperties;
		Range.RangeIterator = RangeIterator;
		Range.copyComparisonConstants = copyComparisonConstants;
		Range.createPrototypeRange = createPrototypeRange;
		Range.inspect = inspect;
		Range.getRangeDocument = getRangeDocument;
		Range.rangesEqual = function (r1, r2) {
			return r1.startContainer === r2.startContainer && r1.startOffset === r2.startOffset && r1.endContainer === r2.endContainer && r1.endOffset === r2.endOffset;
		};

		api.DomRange = Range;
		api.RangeException = RangeException;
	});
	rangy.createModule("WrappedRange", function (api, module) {
		api.requireModules(["DomUtil", "DomRange"]);

		/**
		 * @constructor
		 */
		var WrappedRange;
		var dom = api.dom;
		var DomPosition = dom.DomPosition;
		var DomRange = api.DomRange;



		/*----------------------------------------------------------------------------------------------------------------*/

		/*
    This is a workaround for a bug where IE returns the wrong container element from the TextRange's parentElement()
    method. For example, in the following (where pipes denote the selection boundaries):

    <ul id="ul"><li id="a">| a </li><li id="b"> b |</li></ul>

    var range = document.selection.createRange();
    alert(range.parentElement().id); // Should alert "ul" but alerts "b"

    This method returns the common ancestor node of the following:
    - the parentElement() of the textRange
    - the parentElement() of the textRange after calling collapse(true)
    - the parentElement() of the textRange after calling collapse(false)
     */
		function getTextRangeContainerElement(textRange) {
			var parentEl = textRange.parentElement();

			var range = textRange.duplicate();
			range.collapse(true);
			var startEl = range.parentElement();
			range = textRange.duplicate();
			range.collapse(false);
			var endEl = range.parentElement();
			var startEndContainer = (startEl == endEl) ? startEl : dom.getCommonAncestor(startEl, endEl);

			return startEndContainer == parentEl ? startEndContainer : dom.getCommonAncestor(parentEl, startEndContainer);
		}

		function textRangeIsCollapsed(textRange) {
			return textRange.compareEndPoints("StartToEnd", textRange) == 0;
		}

		// Gets the boundary of a TextRange expressed as a node and an offset within that node. This function started out as
		// an improved version of code found in Tim Cameron Ryan's IERange (http://code.google.com/p/ierange/) but has
		// grown, fixing problems with line breaks in preformatted text, adding workaround for IE TextRange bugs, handling
		// for inputs and images, plus optimizations.
		function getTextRangeBoundaryPosition(textRange, wholeRangeContainerElement, isStart, isCollapsed) {
			var workingRange = textRange.duplicate();

			workingRange.collapse(isStart);
			var containerElement = workingRange.parentElement();

			// Sometimes collapsing a TextRange that's at the start of a text node can move it into the previous node, so
			// check for that
			// TODO: Find out when. Workaround for wholeRangeContainerElement may break this
			if (!dom.isAncestorOf(wholeRangeContainerElement, containerElement, true)) {
				containerElement = wholeRangeContainerElement;

			}



			// Deal with nodes that cannot "contain rich HTML markup". In practice, this means form inputs, images and
			// similar. See http://msdn.microsoft.com/en-us/library/aa703950%28VS.85%29.aspx
			if (!containerElement.canHaveHTML) {
				return new DomPosition(containerElement.parentNode, dom.getNodeIndex(containerElement));
			}

			var workingNode = dom.getDocument(containerElement).createElement("span");

			// Workaround for HTML5 Shiv's insane violation of
			// document.createElement(). See Rangy issue 104 and HTML 5 Shiv issue
			// 64: https://github.com/aFarkas/html5shiv/issues/64
			if (workingNode.parentNode) {
				workingNode.parentNode.removeChild(workingNode);
			}

			var comparison, workingComparisonType = isStart ? "StartToStart" : "StartToEnd";
			var previousNode, nextNode, boundaryPosition, boundaryNode;

			// Move the working range through the container's children, starting at the end and working backwards, until the
			// working range reaches or goes past the boundary we're interested in
			do {
				containerElement.insertBefore(workingNode, workingNode.previousSibling);
				workingRange.moveToElementText(workingNode);
			} while ((comparison = workingRange.compareEndPoints(workingComparisonType, textRange)) > 0 && workingNode.previousSibling);

			// We've now reached or gone past the boundary of the text range we're interested in
			// so have identified the node we want
			boundaryNode = workingNode.nextSibling;

			if (comparison == -1 && boundaryNode && dom.isCharacterDataNode(boundaryNode)) {
				// This is a character data node (text, comment, cdata). The working range is collapsed at the start of the
				// node containing the text range's boundary, so we move the end of the working range to the boundary point
				// and measure the length of its text to get the boundary's offset within the node.
				workingRange.setEndPoint(isStart ? "EndToStart" : "EndToEnd", textRange);


				var offset;

				if (/[\r\n]/.test(boundaryNode.data)) {
					/*
                For the particular case of a boundary within a text node containing line breaks (within a <pre> element,
                for example), we need a slightly complicated approach to get the boundary's offset in IE. The facts:

                - Each line break is represented as \r in the text node's data/nodeValue properties
                - Each line break is represented as \r\n in the TextRange's 'text' property
                - The 'text' property of the TextRange does not contain trailing line breaks

                To get round the problem presented by the final fact above, we can use the fact that TextRange's
                moveStart() and moveEnd() methods return the actual number of characters moved, which is not necessarily
                the same as the number of characters it was instructed to move. The simplest approach is to use this to
                store the characters moved when moving both the start and end of the range to the start of the document
                body and subtracting the start offset from the end offset (the "move-negative-gazillion" method).
                However, this is extremely slow when the document is large and the range is near the end of it. Clearly
                doing the mirror image (i.e. moving the range boundaries to the end of the document) has the same
                problem.

                Another approach that works is to use moveStart() to move the start boundary of the range up to the end
                boundary one character at a time and incrementing a counter with the value returned by the moveStart()
                call. However, the check for whether the start boundary has reached the end boundary is expensive, so
                this method is slow (although unlike "move-negative-gazillion" is largely unaffected by the location of
                the range within the document).

                The method below is a hybrid of the two methods above. It uses the fact that a string containing the
                TextRange's 'text' property with each \r\n converted to a single \r character cannot be longer than the
                text of the TextRange, so the start of the range is moved that length initially and then a character at
                a time to make up for any trailing line breaks not contained in the 'text' property. This has good
                performance in most situations compared to the previous two methods.
                */
					var tempRange = workingRange.duplicate();
					var rangeLength = tempRange.text.replace(/\r\n/g, "\r").length;

					offset = tempRange.moveStart("character", rangeLength);
					while ((comparison = tempRange.compareEndPoints("StartToEnd", tempRange)) == -1) {
						offset++;
						tempRange.moveStart("character", 1);
					}
				} else {
					// IE7 sometimes has weird workingranges that apparently do not start in the workingNode any more, but in
					// some kind of phantom paragraph, that cannot be found in the DOM.
					// in such situations, the workingRange.text no longer is a substring at the start of the boundaryNode.data
					// If we find such a situation, we skip all characters at the start of the workingRange.data, that are not
					// at the start of the boundaryNode.data.
					// Before comparing, we have to replace all nbsp with normal spaces
					var wrText = workingRange.text.replace(/\u00a0/g, " ");
					var bnText = boundaryNode.data.replace(/\u00a0/g, " ");
					if (bnText.indexOf(wrText) !== 0) {
						while (wrText.length > 0 && bnText.indexOf(wrText) !== 0) {
							wrText = wrText.substr(1);
						}
						offset = wrText.length;
					} else {
						offset = workingRange.text.length;
					}
				}
				boundaryPosition = new DomPosition(boundaryNode, offset);
			} else {


				// If the boundary immediately follows a character data node and this is the end boundary, we should favour
				// a position within that, and likewise for a start boundary preceding a character data node
				previousNode = (isCollapsed || !isStart) && workingNode.previousSibling;
				nextNode = (isCollapsed || isStart) && workingNode.nextSibling;



				if (nextNode && dom.isCharacterDataNode(nextNode)) {
					boundaryPosition = new DomPosition(nextNode, 0);
				} else if (previousNode && dom.isCharacterDataNode(previousNode)) {
					boundaryPosition = new DomPosition(previousNode, previousNode.length);
				} else {
					boundaryPosition = new DomPosition(containerElement, dom.getNodeIndex(workingNode));
				}
			}

			// Clean up
			workingNode.parentNode.removeChild(workingNode);

			return boundaryPosition;
		}

		// Returns a TextRange representing the boundary of a TextRange expressed as a node and an offset within that node.
		// This function started out as an optimized version of code found in Tim Cameron Ryan's IERange
		// (http://code.google.com/p/ierange/)
		function createBoundaryTextRange(boundaryPosition, isStart) {
			var boundaryNode, boundaryParent, boundaryOffset = boundaryPosition.offset;
			var doc = dom.getDocument(boundaryPosition.node);
			var workingNode, childNodes, workingRange = doc.body.createTextRange();
			var nodeIsDataNode = dom.isCharacterDataNode(boundaryPosition.node);

			if (nodeIsDataNode) {
				boundaryNode = boundaryPosition.node;
				boundaryParent = boundaryNode.parentNode;
			} else {
				childNodes = boundaryPosition.node.childNodes;
				boundaryNode = (boundaryOffset < childNodes.length) ? childNodes[boundaryOffset] : null;
				boundaryParent = boundaryPosition.node;
			}

			// Position the range immediately before the node containing the boundary
			workingNode = doc.createElement("span");

			// Making the working element non-empty element persuades IE to consider the TextRange boundary to be within the
			// element rather than immediately before or after it, which is what we want
			workingNode.innerHTML = "&#feff;";

			// insertBefore is supposed to work like appendChild if the second parameter is null. However, a bug report
			// for IERange suggests that it can crash the browser: http://code.google.com/p/ierange/issues/detail?id=12
			if (boundaryNode) {
				boundaryParent.insertBefore(workingNode, boundaryNode);
			} else {
				boundaryParent.appendChild(workingNode);
			}

			try {
				workingRange.moveToElementText(workingNode);
				workingRange.collapse(!isStart);
			} catch (err) {
				// @todo window.console.log('problem with moveToElementText');
				//return false;
			}

			// Clean up
			boundaryParent.removeChild(workingNode);

			// Move the working range to the text offset, if required
			if (nodeIsDataNode) {
				workingRange[isStart ? "moveStart" : "moveEnd"]("character", boundaryOffset);
			}

			return workingRange;
		}

		/*----------------------------------------------------------------------------------------------------------------*/

		if (api.features.implementsDomRange && (!api.features.implementsTextRange || !api.config.preferTextRange)) {
			// This is a wrapper around the browser's native DOM Range. It has two aims:
			// - Provide workarounds for specific browser bugs
			// - provide convenient extensions, which are inherited from Rangy's DomRange

			(function () {
				var rangeProto;
				var rangeProperties = DomRange.rangeProperties;
				var canSetRangeStartAfterEnd;

				function updateRangeProperties(range) {
					var i = rangeProperties.length,
						prop;
					while (i--) {
						prop = rangeProperties[i];
						range[prop] = range.nativeRange[prop];
					}
				}

				function updateNativeRange(range, startContainer, startOffset, endContainer, endOffset) {
					var startMoved = (range.startContainer !== startContainer || range.startOffset != startOffset);
					var endMoved = (range.endContainer !== endContainer || range.endOffset != endOffset);

					// Always set both boundaries for the benefit of IE9 (see issue 35)
					if (startMoved || endMoved) {
						range.setEnd(endContainer, endOffset);
						range.setStart(startContainer, startOffset);
					}
				}

				function detach(range) {
					range.nativeRange.detach();
					range.detached = true;
					var i = rangeProperties.length,
						prop;
					while (i--) {
						prop = rangeProperties[i];
						range[prop] = null;
					}
				}

				var createBeforeAfterNodeSetter;

				WrappedRange = function (range) {
					if (!range) {
						throw new Error("Range must be specified");
					}
					this.nativeRange = range;
					updateRangeProperties(this);
				};

				DomRange.createPrototypeRange(WrappedRange, updateNativeRange, detach);

				rangeProto = WrappedRange.prototype;

				rangeProto.selectNode = function (node) {
					this.nativeRange.selectNode(node);
					updateRangeProperties(this);
				};

				rangeProto.deleteContents = function () {
					this.nativeRange.deleteContents();
					updateRangeProperties(this);
				};

				rangeProto.extractContents = function () {
					var frag = this.nativeRange.extractContents();
					updateRangeProperties(this);
					return frag;
				};

				rangeProto.cloneContents = function () {
					return this.nativeRange.cloneContents();
				};

				// TODO: Until I can find a way to programmatically trigger the Firefox bug (apparently long-standing, still
				// present in 3.6.8) that throws "Index or size is negative or greater than the allowed amount" for
				// insertNode in some circumstances, all browsers will have to use the Rangy's own implementation of
				// insertNode, which works but is almost certainly slower than the native implementation.
				/*
            rangeProto.insertNode = function(node) {
                this.nativeRange.insertNode(node);
                updateRangeProperties(this);
            };
*/

				rangeProto.surroundContents = function (node) {
					this.nativeRange.surroundContents(node);
					updateRangeProperties(this);
				};

				rangeProto.collapse = function (isStart) {
					this.nativeRange.collapse(isStart);
					updateRangeProperties(this);
				};

				rangeProto.cloneRange = function () {
					return new WrappedRange(this.nativeRange.cloneRange());
				};

				rangeProto.refresh = function () {
					updateRangeProperties(this);
				};

				rangeProto.toString = function () {
					return this.nativeRange.toString();
				};

				// Create test range and node for feature detection

				var testTextNode = document.createTextNode("test");
				dom.getBody(document).appendChild(testTextNode);
				var range = document.createRange();

				/*--------------------------------------------------------------------------------------------------------*/

				// Test for Firefox 2 bug that prevents moving the start of a Range to a point after its current end and
				// correct for it

				range.setStart(testTextNode, 0);
				range.setEnd(testTextNode, 0);

				try {
					range.setStart(testTextNode, 1);
					canSetRangeStartAfterEnd = true;

					rangeProto.setStart = function (node, offset) {
						this.nativeRange.setStart(node, offset);
						updateRangeProperties(this);
					};

					rangeProto.setEnd = function (node, offset) {
						this.nativeRange.setEnd(node, offset);
						updateRangeProperties(this);
					};

					createBeforeAfterNodeSetter = function (name) {
						return function (node) {
							this.nativeRange[name](node);
							updateRangeProperties(this);
						};
					};

				} catch (ex) {


					canSetRangeStartAfterEnd = false;

					rangeProto.setStart = function (node, offset) {
						try {
							this.nativeRange.setStart(node, offset);
						} catch (ex) {
							this.nativeRange.setEnd(node, offset);
							this.nativeRange.setStart(node, offset);
						}
						updateRangeProperties(this);
					};

					rangeProto.setEnd = function (node, offset) {
						try {
							this.nativeRange.setEnd(node, offset);
						} catch (ex) {
							this.nativeRange.setStart(node, offset);
							this.nativeRange.setEnd(node, offset);
						}
						updateRangeProperties(this);
					};

					createBeforeAfterNodeSetter = function (name, oppositeName) {
						return function (node) {
							try {
								this.nativeRange[name](node);
							} catch (ex) {
								this.nativeRange[oppositeName](node);
								this.nativeRange[name](node);
							}
							updateRangeProperties(this);
						};
					};
				}

				rangeProto.setStartBefore = createBeforeAfterNodeSetter("setStartBefore", "setEndBefore");
				rangeProto.setStartAfter = createBeforeAfterNodeSetter("setStartAfter", "setEndAfter");
				rangeProto.setEndBefore = createBeforeAfterNodeSetter("setEndBefore", "setStartBefore");
				rangeProto.setEndAfter = createBeforeAfterNodeSetter("setEndAfter", "setStartAfter");

				/*--------------------------------------------------------------------------------------------------------*/

				// Test for and correct Firefox 2 behaviour with selectNodeContents on text nodes: it collapses the range to
				// the 0th character of the text node
				range.selectNodeContents(testTextNode);
				if (range.startContainer == testTextNode && range.endContainer == testTextNode && range.startOffset == 0 && range.endOffset == testTextNode.length) {
					rangeProto.selectNodeContents = function (node) {
						this.nativeRange.selectNodeContents(node);
						updateRangeProperties(this);
					};
				} else {
					rangeProto.selectNodeContents = function (node) {
						this.setStart(node, 0);
						this.setEnd(node, DomRange.getEndOffset(node));
					};
				}

				/*--------------------------------------------------------------------------------------------------------*/

				// Test for WebKit bug that has the beahviour of compareBoundaryPoints round the wrong way for constants
				// START_TO_END and END_TO_START: https://bugs.webkit.org/show_bug.cgi?id=20738

				range.selectNodeContents(testTextNode);
				range.setEnd(testTextNode, 3);

				var range2 = document.createRange();
				range2.selectNodeContents(testTextNode);
				range2.setEnd(testTextNode, 4);
				range2.setStart(testTextNode, 2);

				if (range.compareBoundaryPoints(range.START_TO_END, range2) == -1 & range.compareBoundaryPoints(range.END_TO_START, range2) == 1) {
					// This is the wrong way round, so correct for it


					rangeProto.compareBoundaryPoints = function (type, range) {
						range = range.nativeRange || range;
						if (type == range.START_TO_END) {
							type = range.END_TO_START;
						} else if (type == range.END_TO_START) {
							type = range.START_TO_END;
						}
						return this.nativeRange.compareBoundaryPoints(type, range);
					};
				} else {
					rangeProto.compareBoundaryPoints = function (type, range) {
						return this.nativeRange.compareBoundaryPoints(type, range.nativeRange || range);
					};
				}

				/*--------------------------------------------------------------------------------------------------------*/

				// Test for existence of createContextualFragment and delegate to it if it exists
				if (api.util.isHostMethod(range, "createContextualFragment")) {
					rangeProto.createContextualFragment = function (fragmentStr) {
						return this.nativeRange.createContextualFragment(fragmentStr);
					};
				}

				/*--------------------------------------------------------------------------------------------------------*/

				// Clean up
				dom.getBody(document).removeChild(testTextNode);
				range.detach();
				range2.detach();
			})();

			api.createNativeRange = function (doc) {
				doc = doc || document;
				return doc.createRange();
			};
		} else if (api.features.implementsTextRange) {
			// This is a wrapper around a TextRange, providing full DOM Range functionality using rangy's DomRange as a
			// prototype

			WrappedRange = function (textRange) {
				this.textRange = textRange;
				this.refresh();
			};

			WrappedRange.prototype = new DomRange(document);

			WrappedRange.prototype.refresh = function () {
				var start, end;

				// TextRange's parentElement() method cannot be trusted. getTextRangeContainerElement() works around that.
				var rangeContainerElement = getTextRangeContainerElement(this.textRange);

				if (textRangeIsCollapsed(this.textRange)) {
					end = start = getTextRangeBoundaryPosition(this.textRange, rangeContainerElement, true, true);
				} else {

					start = getTextRangeBoundaryPosition(this.textRange, rangeContainerElement, true, false);
					end = getTextRangeBoundaryPosition(this.textRange, rangeContainerElement, false, false);
				}

				this.setStart(start.node, start.offset);
				this.setEnd(end.node, end.offset);
			};

			DomRange.copyComparisonConstants(WrappedRange);

			// Add WrappedRange as the Range property of the global object to allow expression like Range.END_TO_END to work
			var globalObj = (function () {
				return this;
			})();
			if (typeof globalObj.Range == "undefined") {
				globalObj.Range = WrappedRange;
			}

			api.createNativeRange = function (doc) {
				doc = doc || document;
				return doc.body.createTextRange();
			};
		}

		if (api.features.implementsTextRange) {
			WrappedRange.rangeToTextRange = function (range) {
				if (range.collapsed) {
					var tr = createBoundaryTextRange(new DomPosition(range.startContainer, range.startOffset), true);
					return tr;
					//return createBoundaryTextRange(new DomPosition(range.startContainer, range.startOffset), true);
				} else {
					var startRange = createBoundaryTextRange(new DomPosition(range.startContainer, range.startOffset), true);
					var endRange = createBoundaryTextRange(new DomPosition(range.endContainer, range.endOffset), false);
					var textRange = dom.getDocument(range.startContainer).body.createTextRange();
					textRange.setEndPoint("StartToStart", startRange);
					textRange.setEndPoint("EndToEnd", endRange);
					return textRange;
				}
			};
		}

		WrappedRange.prototype.getName = function () {
			return "WrappedRange";
		};

		api.WrappedRange = WrappedRange;

		api.createRange = function (doc) {
			doc = doc || document;
			return new WrappedRange(api.createNativeRange(doc));
		};

		api.createRangyRange = function (doc) {
			doc = doc || document;
			return new DomRange(doc);
		};

		api.createIframeRange = function (iframeEl) {
			return api.createRange(dom.getIframeDocument(iframeEl));
		};

		api.createIframeRangyRange = function (iframeEl) {
			return api.createRangyRange(dom.getIframeDocument(iframeEl));
		};

		api.addCreateMissingNativeApiListener(function (win) {
			var doc = win.document;
			if (typeof doc.createRange == "undefined") {
				doc.createRange = function () {
					return api.createRange(this);
				};
			}
			doc = win = null;
		});
	});
	rangy.createModule("WrappedSelection", function (api, module) {
		// This will create a selection object wrapper that follows the Selection object found in the WHATWG draft DOM Range
		// spec (http://html5.org/specs/dom-range.html)

		api.requireModules(["DomUtil", "DomRange", "WrappedRange"]);

		api.config.checkSelectionRanges = true;

		var BOOLEAN = "boolean",
			windowPropertyName = "_rangySelection",
			dom = api.dom,
			util = api.util,
			DomRange = api.DomRange,
			WrappedRange = api.WrappedRange,
			DOMException = api.DOMException,
			DomPosition = dom.DomPosition,
			getSelection,
			selectionIsCollapsed,
			CONTROL = "Control";



		function getWinSelection(winParam) {
			return (winParam || window).getSelection();
		}

		function getDocSelection(winParam) {
			return (winParam || window).document.selection;
		}

		// Test for the Range/TextRange and Selection features required
		// Test for ability to retrieve selection
		var implementsWinGetSelection = api.util.isHostMethod(window, "getSelection"),
			implementsDocSelection = api.util.isHostObject(document, "selection");

		var useDocumentSelection = implementsDocSelection && (!implementsWinGetSelection || api.config.preferTextRange);

		if (useDocumentSelection) {
			getSelection = getDocSelection;
			api.isSelectionValid = function (winParam) {
				var doc = (winParam || window).document,
					nativeSel = doc.selection;

				// Check whether the selection TextRange is actually contained within the correct document
				return (nativeSel.type != "None" || dom.getDocument(nativeSel.createRange().parentElement()) == doc);
			};
		} else if (implementsWinGetSelection) {
			getSelection = getWinSelection;
			api.isSelectionValid = function () {
				return true;
			};
		} else {
			module.fail("Neither document.selection or window.getSelection() detected.");
		}

		api.getNativeSelection = getSelection;

		var testSelection = getSelection();
		var testRange = api.createNativeRange(document);
		var body = dom.getBody(document);

		// Obtaining a range from a selection
		var selectionHasAnchorAndFocus = util.areHostObjects(testSelection, ["anchorNode", "focusNode"] && util.areHostProperties(testSelection, ["anchorOffset", "focusOffset"]));
		api.features.selectionHasAnchorAndFocus = selectionHasAnchorAndFocus;

		// Test for existence of native selection extend() method
		var selectionHasExtend = util.isHostMethod(testSelection, "extend");
		api.features.selectionHasExtend = selectionHasExtend;

		// Test if rangeCount exists
		var selectionHasRangeCount = (typeof testSelection.rangeCount == "number");
		api.features.selectionHasRangeCount = selectionHasRangeCount;

		var selectionSupportsMultipleRanges = false;
		var collapsedNonEditableSelectionsSupported = true;

		if (util.areHostMethods(testSelection, ["addRange", "getRangeAt", "removeAllRanges"]) && typeof testSelection.rangeCount == "number" && api.features.implementsDomRange) {

			(function () {
				var iframe = document.createElement("iframe");
				body.appendChild(iframe);

				var iframeDoc = dom.getIframeDocument(iframe);
				iframeDoc.open();
				iframeDoc.write("<html><head></head><body>12</body></html>");
				iframeDoc.close();

				var sel = dom.getIframeWindow(iframe).getSelection();
				var docEl = iframeDoc.documentElement;
				var iframeBody = docEl.lastChild,
					textNode = iframeBody.firstChild;

				// Test whether the native selection will allow a collapsed selection within a non-editable element
				var r1 = iframeDoc.createRange();
				r1.setStart(textNode, 1);
				r1.collapse(true);
				sel.addRange(r1);
				collapsedNonEditableSelectionsSupported = (sel.rangeCount == 1);
				sel.removeAllRanges();

				// Test whether the native selection is capable of supporting multiple ranges
				var r2 = r1.cloneRange();
				r1.setStart(textNode, 0);
				r2.setEnd(textNode, 2);
				sel.addRange(r1);
				sel.addRange(r2);

				selectionSupportsMultipleRanges = (sel.rangeCount == 2);

				// Clean up
				r1.detach();
				r2.detach();

				body.removeChild(iframe);
			})();
		}

		api.features.selectionSupportsMultipleRanges = selectionSupportsMultipleRanges;
		api.features.collapsedNonEditableSelectionsSupported = collapsedNonEditableSelectionsSupported;

		// ControlRanges
		var implementsControlRange = false,
			testControlRange;

		if (body && util.isHostMethod(body, "createControlRange")) {
			testControlRange = body.createControlRange();
			if (util.areHostProperties(testControlRange, ["item", "add"])) {
				implementsControlRange = true;
			}
		}
		api.features.implementsControlRange = implementsControlRange;

		// Selection collapsedness
		if (selectionHasAnchorAndFocus) {
			selectionIsCollapsed = function (sel) {
				return sel.anchorNode === sel.focusNode && sel.anchorOffset === sel.focusOffset;
			};
		} else {
			selectionIsCollapsed = function (sel) {
				return sel.rangeCount ? sel.getRangeAt(sel.rangeCount - 1).collapsed : false;
			};
		}

		function updateAnchorAndFocusFromRange(sel, range, backwards) {
			var anchorPrefix = backwards ? "end" : "start",
				focusPrefix = backwards ? "start" : "end";
			sel.anchorNode = range[anchorPrefix + "Container"];
			sel.anchorOffset = range[anchorPrefix + "Offset"];
			sel.focusNode = range[focusPrefix + "Container"];
			sel.focusOffset = range[focusPrefix + "Offset"];
		}

		function updateAnchorAndFocusFromNativeSelection(sel) {
			var nativeSel = sel.nativeSelection;
			sel.anchorNode = nativeSel.anchorNode;
			sel.anchorOffset = nativeSel.anchorOffset;
			sel.focusNode = nativeSel.focusNode;
			sel.focusOffset = nativeSel.focusOffset;
		}

		function updateEmptySelection(sel) {
			sel.anchorNode = sel.focusNode = null;
			sel.anchorOffset = sel.focusOffset = 0;
			sel.rangeCount = 0;
			sel.isCollapsed = true;
			sel._ranges.length = 0;
		}

		function getNativeRange(range) {
			var nativeRange;
			if (range instanceof DomRange) {
				nativeRange = range._selectionNativeRange;
				if (!nativeRange) {
					nativeRange = api.createNativeRange(dom.getDocument(range.startContainer));
					nativeRange.setEnd(range.endContainer, range.endOffset);
					nativeRange.setStart(range.startContainer, range.startOffset);
					range._selectionNativeRange = nativeRange;
					range.attachListener("detach", function () {

						this._selectionNativeRange = null;
					});
				}
			} else if (range instanceof WrappedRange) {
				nativeRange = range.nativeRange;
			} else if (api.features.implementsDomRange && (range instanceof dom.getWindow(range.startContainer).Range)) {
				nativeRange = range;
			}
			return nativeRange;
		}

		function rangeContainsSingleElement(rangeNodes) {
			if (!rangeNodes.length || rangeNodes[0].nodeType != 1) {
				return false;
			}
			for (var i = 1, len = rangeNodes.length; i < len; ++i) {
				if (!dom.isAncestorOf(rangeNodes[0], rangeNodes[i])) {
					return false;
				}
			}
			return true;
		}

		function getSingleElementFromRange(range) {
			var nodes = range.getNodes();
			if (!rangeContainsSingleElement(nodes)) {
				throw new Error("getSingleElementFromRange: range " + range.inspect() + " did not consist of a single element");
			}
			return nodes[0];
		}

		function isTextRange(range) {
			return !!range && typeof range.text != "undefined";
		}

		function updateFromTextRange(sel, range) {
			// Create a Range from the selected TextRange
			var wrappedRange = new WrappedRange(range);
			sel._ranges = [wrappedRange];

			updateAnchorAndFocusFromRange(sel, wrappedRange, false);
			sel.rangeCount = 1;
			sel.isCollapsed = wrappedRange.collapsed;
		}

		function updateControlSelection(sel) {
			// Update the wrapped selection based on what's now in the native selection
			sel._ranges.length = 0;
			if (sel.docSelection.type == "None") {
				updateEmptySelection(sel);
			} else {
				var controlRange = sel.docSelection.createRange();
				if (isTextRange(controlRange)) {
					// This case (where the selection type is "Control" and calling createRange() on the selection returns
					// a TextRange) can happen in IE 9. It happens, for example, when all elements in the selected
					// ControlRange have been removed from the ControlRange and removed from the document.
					updateFromTextRange(sel, controlRange);
				} else {
					sel.rangeCount = controlRange.length;
					var range, doc = dom.getDocument(controlRange.item(0));
					for (var i = 0; i < sel.rangeCount; ++i) {
						range = api.createRange(doc);
						range.selectNode(controlRange.item(i));
						sel._ranges.push(range);
					}
					sel.isCollapsed = sel.rangeCount == 1 && sel._ranges[0].collapsed;
					updateAnchorAndFocusFromRange(sel, sel._ranges[sel.rangeCount - 1], false);
				}
			}
		}

		function addRangeToControlSelection(sel, range) {
			var controlRange = sel.docSelection.createRange();
			var rangeElement = getSingleElementFromRange(range);

			// Create a new ControlRange containing all the elements in the selected ControlRange plus the element
			// contained by the supplied range
			var doc = dom.getDocument(controlRange.item(0));
			var newControlRange = dom.getBody(doc).createControlRange();
			for (var i = 0, len = controlRange.length; i < len; ++i) {
				newControlRange.add(controlRange.item(i));
			}
			try {
				newControlRange.add(rangeElement);
			} catch (ex) {
				throw new Error("addRange(): Element within the specified Range could not be added to control selection (does it have layout?)");
			}
			newControlRange.select();

			// Update the wrapped selection based on what's now in the native selection
			updateControlSelection(sel);
		}

		var getSelectionRangeAt;

		if (util.isHostMethod(testSelection, "getRangeAt")) {
			getSelectionRangeAt = function (sel, index) {
				try {
					return sel.getRangeAt(index);
				} catch (ex) {
					return null;
				}
			};
		} else if (selectionHasAnchorAndFocus) {
			getSelectionRangeAt = function (sel) {
				var doc = dom.getDocument(sel.anchorNode);
				var range = api.createRange(doc);
				range.setStart(sel.anchorNode, sel.anchorOffset);
				range.setEnd(sel.focusNode, sel.focusOffset);

				// Handle the case when the selection was selected backwards (from the end to the start in the
				// document)
				if (range.collapsed !== this.isCollapsed) {
					range.setStart(sel.focusNode, sel.focusOffset);
					range.setEnd(sel.anchorNode, sel.anchorOffset);
				}

				return range;
			};
		}

		/**
		 * @constructor
		 */
		function WrappedSelection(selection, docSelection, win) {
			this.nativeSelection = selection;
			this.docSelection = docSelection;
			this._ranges = [];
			this.win = win;
			this.refresh();
		}

		api.getSelection = function (win) {
			win = win || window;
			var sel = win[windowPropertyName];
			var nativeSel = getSelection(win),
				docSel = implementsDocSelection ? getDocSelection(win) : null;
			if (sel) {
				sel.nativeSelection = nativeSel;
				sel.docSelection = docSel;
				sel.refresh(win);
			} else {
				sel = new WrappedSelection(nativeSel, docSel, win);
				win[windowPropertyName] = sel;
			}
			return sel;
		};

		api.getIframeSelection = function (iframeEl) {
			return api.getSelection(dom.getIframeWindow(iframeEl));
		};

		var selProto = WrappedSelection.prototype;

		function createControlSelection(sel, ranges) {
			// Ensure that the selection becomes of type "Control"
			var doc = dom.getDocument(ranges[0].startContainer);
			var controlRange = dom.getBody(doc).createControlRange();
			for (var i = 0, el; i < rangeCount; ++i) {
				el = getSingleElementFromRange(ranges[i]);
				try {
					controlRange.add(el);
				} catch (ex) {
					throw new Error("setRanges(): Element within the one of the specified Ranges could not be added to control selection (does it have layout?)");
				}
			}
			controlRange.select();

			// Update the wrapped selection based on what's now in the native selection
			updateControlSelection(sel);
		}

		// Selecting a range
		if (!useDocumentSelection && selectionHasAnchorAndFocus && util.areHostMethods(testSelection, ["removeAllRanges", "addRange"])) {
			selProto.removeAllRanges = function () {
				this.nativeSelection.removeAllRanges();
				updateEmptySelection(this);
			};

			var addRangeBackwards = function (sel, range) {
				var doc = DomRange.getRangeDocument(range);
				var endRange = api.createRange(doc);
				endRange.collapseToPoint(range.endContainer, range.endOffset);
				sel.nativeSelection.addRange(getNativeRange(endRange));
				sel.nativeSelection.extend(range.startContainer, range.startOffset);
				sel.refresh();
			};

			if (selectionHasRangeCount) {
				selProto.addRange = function (range, backwards) {
					if (implementsControlRange && implementsDocSelection && this.docSelection.type == CONTROL) {
						addRangeToControlSelection(this, range);
					} else {
						if (backwards && selectionHasExtend) {
							addRangeBackwards(this, range);
						} else {
							var previousRangeCount;
							if (selectionSupportsMultipleRanges) {
								previousRangeCount = this.rangeCount;
							} else {
								this.removeAllRanges();
								previousRangeCount = 0;
							}
							this.nativeSelection.addRange(getNativeRange(range));

							// Check whether adding the range was successful
							this.rangeCount = this.nativeSelection.rangeCount;

							if (this.rangeCount == previousRangeCount + 1) {
								// The range was added successfully

								// Check whether the range that we added to the selection is reflected in the last range extracted from
								// the selection
								if (api.config.checkSelectionRanges) {
									var nativeRange = getSelectionRangeAt(this.nativeSelection, this.rangeCount - 1);
									if (nativeRange && !DomRange.rangesEqual(nativeRange, range)) {
										// Happens in WebKit with, for example, a selection placed at the start of a text node
										range = new WrappedRange(nativeRange);
									}
								}
								this._ranges[this.rangeCount - 1] = range;
								updateAnchorAndFocusFromRange(this, range, selectionIsBackwards(this.nativeSelection));
								this.isCollapsed = selectionIsCollapsed(this);
							} else {
								// The range was not added successfully. The simplest thing is to refresh
								this.refresh();
							}
						}
					}
				};
			} else {
				selProto.addRange = function (range, backwards) {
					if (backwards && selectionHasExtend) {
						addRangeBackwards(this, range);
					} else {
						this.nativeSelection.addRange(getNativeRange(range));
						this.refresh();
					}
				};
			}

			selProto.setRanges = function (ranges) {
				if (implementsControlRange && ranges.length > 1) {
					createControlSelection(this, ranges);
				} else {
					this.removeAllRanges();
					for (var i = 0, len = ranges.length; i < len; ++i) {
						this.addRange(ranges[i]);
					}
				}
			};
		} else if (util.isHostMethod(testSelection, "empty") && util.isHostMethod(testRange, "select") && implementsControlRange && useDocumentSelection) {

			selProto.removeAllRanges = function () {
				// Added try/catch as fix for issue #21
				try {

					var isNativeIE7 = (jQuery.browser.msie && jQuery.browser.version < 8 && (typeof document.documentMode === 'undefined'));
					if (!isNativeIE7) {
						this.docSelection.empty();
					}

					// Check for empty() not working (issue #24)
					if (this.docSelection.type != "None") {

						if (isNativeIE7) {
							this.docSelection.empty();
						}

						// removed workaround of rangy-core implementation
						// for IE to fix issue with strange selection of
						// hole body in some selection change cases
					}

				} catch (ex) {}
				updateEmptySelection(this);
			};

			selProto.addRange = function (range) {
				if (this.docSelection.type == CONTROL) {
					addRangeToControlSelection(this, range);
				} else {
					try {
						WrappedRange.rangeToTextRange(range).select();
						this._ranges[0] = range;
						this.rangeCount = 1;
						this.isCollapsed = this._ranges[0].collapsed;
						updateAnchorAndFocusFromRange(this, range, false);
					} catch (e) {
						// @todo
						// window.console.log('problem at addRange');
					}
				}
			};

			selProto.setRanges = function (ranges) {
				this.removeAllRanges();
				var rangeCount = ranges.length;
				if (rangeCount > 1) {
					createControlSelection(this, ranges);
				} else if (rangeCount) {
					this.addRange(ranges[0]);
				}
			};
		} else {
			module.fail("No means of selecting a Range or TextRange was found");
			return false;
		}

		selProto.getRangeAt = function (index) {
			if (index < 0 || index >= this.rangeCount) {
				throw new DOMException("INDEX_SIZE_ERR");
			} else {
				return this._ranges[index];
			}
		};

		var refreshSelection;

		if (useDocumentSelection) {
			refreshSelection = function (sel) {
				var range;
				if (api.isSelectionValid(sel.win)) {
					range = sel.docSelection.createRange();
				} else {
					range = dom.getBody(sel.win.document).createTextRange();
					range.collapse(true);
				}


				if (sel.docSelection.type == CONTROL) {
					updateControlSelection(sel);
				} else if (isTextRange(range)) {
					updateFromTextRange(sel, range);
				} else {
					updateEmptySelection(sel);
				}
			};
		} else if (util.isHostMethod(testSelection, "getRangeAt") && typeof testSelection.rangeCount == "number") {
			refreshSelection = function (sel) {
				if (implementsControlRange && implementsDocSelection && sel.docSelection.type == CONTROL) {
					updateControlSelection(sel);
				} else {
					sel._ranges.length = sel.rangeCount = sel.nativeSelection.rangeCount;
					if (sel.rangeCount) {
						for (var i = 0, len = sel.rangeCount; i < len; ++i) {
							sel._ranges[i] = new api.WrappedRange(sel.nativeSelection.getRangeAt(i));
						}
						updateAnchorAndFocusFromRange(sel, sel._ranges[sel.rangeCount - 1], selectionIsBackwards(sel.nativeSelection));
						sel.isCollapsed = selectionIsCollapsed(sel);
					} else {
						updateEmptySelection(sel);
					}
				}
			};
		} else if (selectionHasAnchorAndFocus && typeof testSelection.isCollapsed == BOOLEAN && typeof testRange.collapsed == BOOLEAN && api.features.implementsDomRange) {
			refreshSelection = function (sel) {
				var range, nativeSel = sel.nativeSelection;
				if (nativeSel.anchorNode) {
					range = getSelectionRangeAt(nativeSel, 0);
					sel._ranges = [range];
					sel.rangeCount = 1;
					updateAnchorAndFocusFromNativeSelection(sel);
					sel.isCollapsed = selectionIsCollapsed(sel);
				} else {
					updateEmptySelection(sel);
				}
			};
		} else {
			module.fail("No means of obtaining a Range or TextRange from the user's selection was found");
			return false;
		}

		selProto.refresh = function (checkForChanges) {
			var oldRanges = checkForChanges ? this._ranges.slice(0) : null;
			refreshSelection(this);
			if (checkForChanges) {
				var i = oldRanges.length;
				if (i != this._ranges.length) {
					return false;
				}
				while (i--) {
					if (!DomRange.rangesEqual(oldRanges[i], this._ranges[i])) {
						return false;
					}
				}
				return true;
			}
		};

		// Removal of a single range
		var removeRangeManually = function (sel, range) {
			var ranges = sel.getAllRanges(),
				removed = false;
			sel.removeAllRanges();
			for (var i = 0, len = ranges.length; i < len; ++i) {
				if (removed || range !== ranges[i]) {
					sel.addRange(ranges[i]);
				} else {
					// According to the draft WHATWG Range spec, the same range may be added to the selection multiple
					// times. removeRange should only remove the first instance, so the following ensures only the first
					// instance is removed
					removed = true;
				}
			}
			if (!sel.rangeCount) {
				updateEmptySelection(sel);
			}
		};

		if (implementsControlRange) {
			selProto.removeRange = function (range) {
				if (this.docSelection.type == CONTROL) {
					var controlRange = this.docSelection.createRange();
					var rangeElement = getSingleElementFromRange(range);

					// Create a new ControlRange containing all the elements in the selected ControlRange minus the
					// element contained by the supplied range
					var doc = dom.getDocument(controlRange.item(0));
					var newControlRange = dom.getBody(doc).createControlRange();
					var el, removed = false;
					for (var i = 0, len = controlRange.length; i < len; ++i) {
						el = controlRange.item(i);
						if (el !== rangeElement || removed) {
							newControlRange.add(controlRange.item(i));
						} else {
							removed = true;
						}
					}
					newControlRange.select();

					// Update the wrapped selection based on what's now in the native selection
					updateControlSelection(this);
				} else {
					removeRangeManually(this, range);
				}
			};
		} else {
			selProto.removeRange = function (range) {
				removeRangeManually(this, range);
			};
		}

		// Detecting if a selection is backwards
		var selectionIsBackwards;
		if (!useDocumentSelection && selectionHasAnchorAndFocus && api.features.implementsDomRange) {
			selectionIsBackwards = function (sel) {
				var backwards = false;
				if (sel.anchorNode) {
					backwards = (dom.comparePoints(sel.anchorNode, sel.anchorOffset, sel.focusNode, sel.focusOffset) == 1);
				}
				return backwards;
			};

			selProto.isBackwards = function () {
				return selectionIsBackwards(this);
			};
		} else {
			selectionIsBackwards = selProto.isBackwards = function () {
				return false;
			};
		}

		// Selection text
		// This is conformant to the new WHATWG DOM Range draft spec but differs from WebKit and Mozilla's implementation
		selProto.toString = function () {

			var rangeTexts = [];
			for (var i = 0, len = this.rangeCount; i < len; ++i) {
				rangeTexts[i] = "" + this._ranges[i];
			}
			return rangeTexts.join("");
		};

		function assertNodeInSameDocument(sel, node) {
			if (sel.anchorNode && (dom.getDocument(sel.anchorNode) !== dom.getDocument(node))) {
				throw new DOMException("WRONG_DOCUMENT_ERR");
			}
		}

		// No current browsers conform fully to the HTML 5 draft spec for this method, so Rangy's own method is always used
		selProto.collapse = function (node, offset) {
			assertNodeInSameDocument(this, node);
			var range = api.createRange(dom.getDocument(node));
			range.collapseToPoint(node, offset);
			this.removeAllRanges();
			this.addRange(range);
			this.isCollapsed = true;
		};

		selProto.collapseToStart = function () {
			if (this.rangeCount) {
				var range = this._ranges[0];
				this.collapse(range.startContainer, range.startOffset);
			} else {
				throw new DOMException("INVALID_STATE_ERR");
			}
		};

		selProto.collapseToEnd = function () {
			if (this.rangeCount) {
				var range = this._ranges[this.rangeCount - 1];
				this.collapse(range.endContainer, range.endOffset);
			} else {
				throw new DOMException("INVALID_STATE_ERR");
			}
		};

		// The HTML 5 spec is very specific on how selectAllChildren should be implemented so the native implementation is
		// never used by Rangy.
		selProto.selectAllChildren = function (node) {
			assertNodeInSameDocument(this, node);
			var range = api.createRange(dom.getDocument(node));
			range.selectNodeContents(node);
			this.removeAllRanges();
			this.addRange(range);
		};

		selProto.deleteFromDocument = function () {
			// Sepcial behaviour required for Control selections
			if (implementsControlRange && implementsDocSelection && this.docSelection.type == CONTROL) {
				var controlRange = this.docSelection.createRange();
				var element;
				while (controlRange.length) {
					element = controlRange.item(0);
					controlRange.remove(element);
					element.parentNode.removeChild(element);
				}
				this.refresh();
			} else if (this.rangeCount) {
				var ranges = this.getAllRanges();
				this.removeAllRanges();
				for (var i = 0, len = ranges.length; i < len; ++i) {
					ranges[i].deleteContents();
				}
				// The HTML5 spec says nothing about what the selection should contain after calling deleteContents on each
				// range. Firefox moves the selection to where the final selected range was, so we emulate that
				this.addRange(ranges[len - 1]);
			}
		};

		// The following are non-standard extensions
		selProto.getAllRanges = function () {
			return this._ranges.slice(0);
		};

		selProto.setSingleRange = function (range) {
			this.setRanges([range]);
		};

		selProto.containsNode = function (node, allowPartial) {
			for (var i = 0, len = this._ranges.length; i < len; ++i) {
				if (this._ranges[i].containsNode(node, allowPartial)) {
					return true;
				}
			}
			return false;
		};

		selProto.toHtml = function () {
			var html = "";
			if (this.rangeCount) {
				var container = DomRange.getRangeDocument(this._ranges[0]).createElement("div");
				for (var i = 0, len = this._ranges.length; i < len; ++i) {
					container.appendChild(this._ranges[i].cloneContents());
				}
				html = container.innerHTML;
			}
			return html;
		};

		function inspect(sel) {
			var rangeInspects = [];
			var anchor = new DomPosition(sel.anchorNode, sel.anchorOffset);
			var focus = new DomPosition(sel.focusNode, sel.focusOffset);
			var name = (typeof sel.getName == "function") ? sel.getName() : "Selection";

			if (typeof sel.rangeCount != "undefined") {
				for (var i = 0, len = sel.rangeCount; i < len; ++i) {
					rangeInspects[i] = DomRange.inspect(sel.getRangeAt(i));
				}
			}
			return "[" + name + "(Ranges: " + rangeInspects.join(", ") + ")(anchor: " + anchor.inspect() + ", focus: " + focus.inspect() + "]";

		}

		selProto.getName = function () {
			return "WrappedSelection";
		};

		selProto.inspect = function () {
			return inspect(this);
		};

		selProto.detach = function () {
			this.win[windowPropertyName] = null;
			this.win = this.anchorNode = this.focusNode = null;
		};

		WrappedSelection.inspect = inspect;

		api.Selection = WrappedSelection;

		api.selectionPrototype = selProto;

		api.addCreateMissingNativeApiListener(function (win) {
			if (typeof win.getSelection == "undefined") {
				win.getSelection = function () {
					return api.getSelection(this);
				};
			}
			win = null;
		});
	});

	// TODO we should avoid populating the global namespace
	window.rangy = rangy;
}());

define("vendor/rangy-core", function(){});

/* rangy-core.js is part of Aloha Editor project http://aloha-editor.org
 *
 * Aloha Editor is a WYSIWYG HTML5 inline editing library and editor. 
 * Copyright (c) 2010-2012 Gentics Software GmbH, Vienna, Austria.
 * Contributors http://aloha-editor.org/contribution.php 
 * 
 * Aloha Editor is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or any later version.
 *
 * Aloha Editor is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 * 
 * As an additional permission to the GNU GPL version 2, you may distribute
 * non-source (e.g., minimized or compacted) forms of the Aloha-Editor
 * source code without the copy of the GNU GPL normally required,
 * provided you include this license notice and a URL through which
 * recipients can access the Corresponding Source.
 */
define('aloha/rangy-core', ['jquery', 'vendor/rangy-core'], function (jQuery) {
	
	return window.rangy;
});

/* Simple JavaScript Inheritance
 * By John Resig http://ejohn.org/
 * MIT Licensed.
 */
// Inspired by base2 and Prototype
/*
 * MODIFICATIONS: 
 * * The name of the "constructor" method was changed from "init" to "_constructor"
 * * Mixin Support using https://gist.github.com/1006243
 * * Modified to be a require.js module
 */
define('util/class',[], function () {
	var initializing = false,
		fnTest = /xyz/.test(function () {
			xyz;
		}) ? /\b_super\b/ : /.*/;

	// The base Class implementation (does nothing)
	// with doing that Class is available in the global namespace.
	this.Class = function () {};

	// Create a new Class that inherits from this class
	Class.extend = function () {
		var _super = this.prototype;

		// Instantiate a base class (but only create the instance,
		// don't run the init constructor)
		initializing = true;
		var prototype = new this();
		initializing = false;

		// Copy the properties over onto the new prototype
		for (var i = 0; i < arguments.length; i++) {
			var prop = arguments[i];
			for (var name in prop) {
				// Check if we're overwriting an existing function
				prototype[name] = typeof prop[name] == "function" && typeof _super[name] == "function" && fnTest.test(prop[name]) ? (function (name, fn) {
					return function () {

						var tmp = this._super;

						// Add a new ._super() method that is the same method
						// but on the super-class
						this._super = _super[name];

						// The method only need to be bound temporarily, so we
						// remove it when we're done executing
						var ret = fn.apply(this, arguments);
						this._super = tmp;

						return ret;
					};
				})(name, prop[name]) : prop[name];
			}
		}

		// The dummy class constructor
		function Class() {
			// All construction is actually done in the _constructor method
			if (!initializing && this._constructor) this._constructor.apply(this, arguments);
		}

		// Populate our constructed prototype object
		Class.prototype = prototype;

		// Enforce the constructor to be what we expect
		Class.constructor = Class;

		// And make this class extendable
		Class.extend = arguments.callee;

		return Class;

	};

	return this.Class;

});

/* lang.js is part of Aloha Editor project http://aloha-editor.org
 *
 * Aloha Editor is a WYSIWYG HTML5 inline editing library and editor. 
 * Copyright (c) 2010-2012 Gentics Software GmbH, Vienna, Austria.
 * Contributors http://aloha-editor.org/contribution.php 
 * 
 * Aloha Editor is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or any later version.
 *
 * Aloha Editor is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 * 
 * As an additional permission to the GNU GPL version 2, you may distribute
 * non-source (e.g., minimized or compacted) forms of the Aloha-Editor
 * source code without the copy of the GNU GPL normally required,
 * provided you include this license notice and a URL through which
 * recipients can access the Corresponding Source.
 */
// Ensure GENTICS Namespace
window.GENTICS = window.GENTICS || {};
window.GENTICS.Utils = window.GENTICS.Utils || {};

define('util/lang', [], function () {
	
});

// Start Closure
(function (window, undefined) {
	
	var jQuery = window.alohaQuery || window.jQuery,
		$ = jQuery,
		GENTICS = window.GENTICS,
		Class = window.Class,
		console = window.console;

	/**
	 * Takes over all properties from the 'properties' object to the target object.
	 * If a property in 'target' with the same name as a property in 'properties' is already defined it is overridden.
	 *
	 * Example:
	 *
	 * var o1 = {a : 1, b : 'hello'};
	 * var o2 = {a : 3, c : 'world'};
	 *
	 * GENTICS.Utils.applyProperties(o1, o2);
	 *
	 * Will result in an o1 object like this:
	 *
	 * {a : 3, b: 'hello', c: 'world'}
	 *
	 * @static
	 * @return void
	 */
	GENTICS.Utils.applyProperties = function (target, properties) {
		var name;
		for (name in properties) {
			if (properties.hasOwnProperty(name)) {
				target[name] = properties[name];
			}
		}
	};

	/**
	 * Generate a unique hexadecimal string with 4 charachters
	 * @return {string}
	 */
	GENTICS.Utils.uniqeString4 = function () {
		return (((1 + Math.random()) * 0x10000) | 0).toString(16).substring(1);
	};

	/**
	 * Generate a unique value represented as a 32 character hexadecimal string,
	 * such as 21EC2020-3AEA-1069-A2DD-08002B30309D
	 * @return {string}
	 */
	GENTICS.Utils.guid = function () {
		var S4 = GENTICS.Utils.uniqeString4;
		return (S4() + S4() + '-' + S4() + '-' + S4() + '-' + S4() + '-' + S4() + S4() + S4());
	};

}(window));

/**
 * ecma5schims.js - Shim for ECMA5 compatibility
 * (http://en.wikipedia.org/wiki/Shim_%28computing%29)
 *
 * A shim library that implements common functions that are missing on some
 * environments in order to complete ECMA5 compatibility across all major
 * browsers.
 *
 * TODO: This code needs to be refactored so as to conform to Aloha coding
 *       standards.  It is also severly lacking in documentation.  Please take
 *       note of: https://github.com/alohaeditor/Aloha-Editor/wiki/Commit-Checklist .
 */

define('aloha/ecma5shims',[], function () {
	
	var $_;

	var shims = {
		// Function bind
		bind: function (owner) {
			var obj = this.obj || this;
			var native_method = Function.prototype.bind;
			var args = Array.prototype.slice.call(arguments, 1);

			if (native_method) {
				return native_method.apply(obj, arguments);
			}
			return function () {
				return obj.apply(owner, arguments.length === 0 ? args : args.concat(Array.prototype.slice.call(arguments)));
			};
		},

		// String trim
		trim: function () {
			var obj = this.obj || this;
			var native_method = String.prototype.trim;

			if (native_method) {
				return native_method.call(obj);
			}
			return obj.replace(/^\s+/, '').replace(/\s+$/, '');
		},

		// Array methods 
		// i optional
		indexOf: function (find, i) {
			var obj = this.obj || this;
			var native_method = Array.prototype.indexOf;

			if (native_method) {
				return native_method.call(obj, find, i);
			}
			if (i === undefined) {
				i = 0;
			}
			if (i < 0) {
				i += obj.length;
			}
			if (i < 0) {
				i = 0;
			}
			var n;
			for (n = obj.length; i < n; i++) {
				if (undefined !== obj[i] && obj[i] === find) {
					return i;
				}
			}
			return -1;
		},

		// that optional
		forEach: function (action, that) {
			var obj = this.obj || this;
			var native_method = Array.prototype.forEach;

			if (native_method) {
				return native_method.call(obj, action, that);
			}
			var i, n;
			for (i = 0, n = obj.length; i < n; i++) {
				if (undefined !== obj[i]) {
					action.call(that, obj[i], i, obj);
				}
			}
		},

		// that optional
		// chain optional
		map: function (mapper, that, chain) {
			var obj = this.obj || this;
			var native_method = Array.prototype.map;
			var returnWrapper = (typeof arguments[arguments.length - 1] == "boolean") ? Array.prototype.pop.call(arguments) : false;
			var result = [];

			if (native_method) {
				result = native_method.call(obj, mapper, that);
			} else {
				var other = [];
				var i, n;
				for (i = 0, n = obj.length; i < n; i++) {
					if (undefined !== obj[i]) {
						other[i] = mapper.call(that, obj[i], i, obj);
					}
				}
				result = other;
			}

			return returnWrapper ? $_(result) : result;
		},

		// that optional
		// chain optional
		filter: function (filterFunc, that, chain) {
			var obj = this.obj || this;
			var native_method = Array.prototype.filter;
			var returnWrapper = (typeof arguments[arguments.length - 1] == "boolean") ? Array.prototype.pop.call(arguments) : false;
			var result = [];

			if (native_method) {
				result = native_method.call(obj, filterFunc, that);
			} else {
				var other = [],
				    v,
				    i,
				    n;
				for (i = 0, n = obj.length; i < n; i++) {
					if (undefined !== obj[i] && filterFunc.call(that, v = obj[i], i, obj)) {
						other.push(v);
					}
				}
				result = other;
			}

			return returnWrapper ? $_(result) : result;
		},

		// that optional
		every: function (tester, that) {
			var obj = this.obj || this;
			var native_method = Array.prototype.every;

			if (native_method) {
				return native_method.call(obj, tester, that);
			}
			var i, n;
			for (i = 0, n = obj.length; i < n; i++) {
				if (undefined !== obj[i] && !tester.call(that, obj[i], i, obj)) {
					return false;
				}
			}
			return true;
		},

		// that optional
		some: function (tester, that) {
			var obj = this.obj || this;
			var native_method = Array.prototype.some;

			if (native_method) {
				return native_method.call(obj, tester, that);
			}
			var i, n;
			for (i = 0, n = obj.length; i < n; i++) {
				if (undefined !== obj[i] && tester.call(that, obj[i], i, obj)) {
					return true;
				}
			}
			return false;
		},

		// Since IE7 doesn't support 'hasAttribute' method on nodes
		// TODO: raise an exception if the object is not an node
		hasAttribute: function (attr) {
			var obj = this.obj || this;
			var native_method = obj.hasAttribute;

			if (native_method) {
				return obj.hasAttribute(attr);
			}
			return !!obj.getAttribute(attr);
		}

	};

	$_ = function (obj) {
		var Wrapper = function () {};
		Wrapper.prototype = shims;

		var wrapper_instance = new Wrapper();
		wrapper_instance.obj = obj;
		return wrapper_instance;
	};

	var shim;
	for (shim in shims) {
		if (shims.hasOwnProperty(shim)) {
			$_[shim] = shims[shim];
		}
	}


	// Node constants
	// http://www.w3.org/TR/DOM-Level-3-Core/core.html#ID-1841493061
	if (typeof window.Node != 'undefined') {
		$_.Node = window.Node;
	} else {
		$_.Node = {
			'ELEMENT_NODE': 1,
			'ATTRIBUTE_NODE': 2,
			'TEXT_NODE': 3,
			'CDATA_SECTION_NODE': 4,
			'ENTITY_REFERENCE_NODE': 5,
			'ENTITY_NODE': 6,
			'PROCESSING_INSTRUCTION_NODE': 7,
			'COMMENT_NODE': 8,
			'DOCUMENT_NODE': 9,
			'DOCUMENT_TYPE_NODE': 10,
			'DOCUMENT_FRAGMENT_NODE': 11,
			'NOTATION_NODE': 12,
			//The two nodes are disconnected. Order between disconnected nodes is always implementation-specific.
			'DOCUMENT_POSITION_DISCONNECTED': 0x01,
			//The second node precedes the reference node.
			'DOCUMENT_POSITION_PRECEDING': 0x02,
			//The node follows the reference node.
			'DOCUMENT_POSITION_FOLLOWING': 0x04,
			//The node contains the reference node. A node which contains is always preceding, too.
			'DOCUMENT_POSITION_CONTAINS': 0x08,
			//The node is contained by the reference node. A node which is contained is always following, too.
			'DOCUMENT_POSITION_CONTAINED_BY': 0x10,
			//The determination of preceding versus following is implementation-specific.
			'DOCUMENT_POSITION_IMPLEMENTATION_SPECIFIC': 0x20
		};
	}

	//node.ownerDocument gives the document object, which isn't the right info for a disconnect
	function getRootParent(node) {
		var parent = null;

		if (node) {
			do {
				parent = node;
			} while (null != (node = node.parentNode));
		}

		return parent;
	}

	//Compare Position - MIT Licensed, John Resig; http://ejohn.org/blog/comparing-document-position/
	//Already checked for equality and disconnect
	function comparePosition(node1, node2) {
		return (node1.contains(node2) && 16) + (node2.contains(node1) && 8) + (node1.sourceIndex >= 0 && node2.sourceIndex >= 0 ? (node1.sourceIndex < node2.sourceIndex && 4) + (node1.sourceIndex > node2.sourceIndex && 2) : 1);
	}

	//get a node with a sourceIndex to use
	function getUseNode(node) {
		//if the node already has a sourceIndex, use that node
		if (null != node.sourceIndex) {
			return node;
		}
		//otherwise, insert a comment (which has a sourceIndex but minimal DOM impact) before the node and use that
		return node.parentNode.insertBefore(document.createComment(""), node);
	}

	// http://www.w3.org/TR/DOM-Level-3-Core/core.html#Node3-compareDocumentPosition
	// FIXME: Check if the DOMNode prototype can be set.
	$_.compareDocumentPosition = function (node1, node2) {

		if (document.documentElement.compareDocumentPosition) {
			return node1.compareDocumentPosition(node2);
		}

		if (!document.documentElement.contains) {
			throw 'neither compareDocumentPosition nor contains is supported by this browser.';
		}

		if (node1 == node2) {
			return 0;
		}

		//if they don't have the same parent, there's a disconnect
		if (getRootParent(node1) != getRootParent(node2)) {
			return 1;
		}

		//use this if both nodes have a sourceIndex (text nodes don't)
		if (null != node1.sourceIndex && null != node2.sourceIndex) {
			return comparePosition(node1, node2);
		}

		//document will definitely contain the other node
		if (node1 == document) {
			return 20;
		}
		if (node2 == document) {
			return 10;
		}

		//get sourceIndexes to use for both nodes
		var useNode1 = getUseNode(node1),
			useNode2 = getUseNode(node2);

		//call this function again to get the result
		var result = comparePosition(useNode1, useNode2);

		//clean up if needed
		if (node1 != useNode1) {
			useNode1.parentNode.removeChild(useNode1);
		}
		if (node2 != useNode2) {
			useNode2.parentNode.removeChild(useNode2);
		}
		return result;
	};

	$_.getComputedStyle = function (node, style) {
		if (window.getComputedStyle) {
			return window.getComputedStyle(node, style);
		}
		if (node.currentStyle) {
			return node.currentStyle;
		}
		return null;
	};

	return $_;
});

/* dom.js is part of Aloha Editor project http://aloha-editor.org
 *
 * Aloha Editor is a WYSIWYG HTML5 inline editing library and editor. 
 * Copyright (c) 2010-2012 Gentics Software GmbH, Vienna, Austria.
 * Contributors http://aloha-editor.org/contribution.php 
 * 
 * Aloha Editor is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or any later version.
 *
 * Aloha Editor is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 * 
 * As an additional permission to the GNU GPL version 2, you may distribute
 * non-source (e.g., minimized or compacted) forms of the Aloha-Editor
 * source code without the copy of the GNU GPL normally required,
 * provided you include this license notice and a URL through which
 * recipients can access the Corresponding Source.
 */
// Ensure GENTICS Namespace
window.GENTICS = window.GENTICS || {};
window.GENTICS.Utils = window.GENTICS.Utils || {};

define('util/dom',['jquery', 'util/class', 'aloha/ecma5shims'], function (jQuery, Class, $_) {
	

	var	GENTICS = window.GENTICS,
		//		Class = window.Class,
		// http://www.w3.org/TR/DOM-Level-3-Core/core.html#ID-1841493061
		Node = {
			'ELEMENT_NODE': 1,
			'ATTRIBUTE_NODE': 2,
			'TEXT_NODE': 3,
			'CDATA_SECTION_NODE': 4,
			'ENTITY_REFERENCE_NODE': 5,
			'ENTITY_NODE': 6,
			'PROCESSING_INSTRUCTION_NODE': 7,
			'COMMENT_NODE': 8,
			'DOCUMENT_NODE': 9,
			'DOCUMENT_TYPE_NODE': 10,
			'DOCUMENT_FRAGMENT_NODE': 11,
			'NOTATION_NODE': 12,
			//The two nodes are disconnected. Order between disconnected nodes is always implementation-specific.
			'DOCUMENT_POSITION_DISCONNECTED': 0x01,
			//The second node precedes the reference node.
			'DOCUMENT_POSITION_PRECEDING': 0x02,
			//The node follows the reference node.
			'DOCUMENT_POSITION_FOLLOWING': 0x04,
			//The node contains the reference node. A node which contains is always preceding, too.
			'DOCUMENT_POSITION_CONTAINS': 0x08,
			//The node is contained by the reference node. A node which is contained is always following, too.
			'DOCUMENT_POSITION_CONTAINED_BY': 0x10,
			//The determination of preceding versus following is implementation-specific.
			'DOCUMENT_POSITION_IMPLEMENTATION_SPECIFIC': 0x20
		},
		blockElementNames = {
			'P': true,
			'H1': true,
			'H2': true,
			'H3': true,
			'H4': true,
			'H5': true,
			'H6': true,
			'LI': true
		};

	/**
	 * @namespace GENTICS.Utils
	 * @class Dom provides methods to get information about the DOM and to manipulate it
	 * @singleton
	 */
	var Dom = Class.extend({
		/**
		 * Regex to find word characters.
		 */
		wordRegex: /[\u0041-\u005A\u0061-\u007A\u00AA\u00B5\u00BA\u00C0-\u00D6\u00D8-\u00F6\u00F8-\u02C1\u02C6-\u02D1\u02E0-\u02E4\u02EC\u02EE\u0370-\u0374\u0376\u0377\u037A-\u037D\u0386\u0388-\u038A\u038C\u038E-\u03A1\u03A3-\u03F5\u03F7-\u0481\u048A-\u0525\u0531-\u0556\u0559\u0561-\u0587\u05D0-\u05EA\u05F0-\u05F2\u0621-\u064A\u066E\u066F\u0671-\u06D3\u06D5\u06E5\u06E6\u06EE\u06EF\u06FA-\u06FC\u06FF\u0710\u0712-\u072F\u074D-\u07A5\u07B1\u07CA-\u07EA\u07F4\u07F5\u07FA\u0800-\u0815\u081A\u0824\u0828\u0904-\u0939\u093D\u0950\u0958-\u0961\u0971\u0972\u0979-\u097F\u0985-\u098C\u098F\u0990\u0993-\u09A8\u09AA-\u09B0\u09B2\u09B6-\u09B9\u09BD\u09CE\u09DC\u09DD\u09DF-\u09E1\u09F0\u09F1\u0A05-\u0A0A\u0A0F\u0A10\u0A13-\u0A28\u0A2A-\u0A30\u0A32\u0A33\u0A35\u0A36\u0A38\u0A39\u0A59-\u0A5C\u0A5E\u0A72-\u0A74\u0A85-\u0A8D\u0A8F-\u0A91\u0A93-\u0AA8\u0AAA-\u0AB0\u0AB2\u0AB3\u0AB5-\u0AB9\u0ABD\u0AD0\u0AE0\u0AE1\u0B05-\u0B0C\u0B0F\u0B10\u0B13-\u0B28\u0B2A-\u0B30\u0B32\u0B33\u0B35-\u0B39\u0B3D\u0B5C\u0B5D\u0B5F-\u0B61\u0B71\u0B83\u0B85-\u0B8A\u0B8E-\u0B90\u0B92-\u0B95\u0B99\u0B9A\u0B9C\u0B9E\u0B9F\u0BA3\u0BA4\u0BA8-\u0BAA\u0BAE-\u0BB9\u0BD0\u0C05-\u0C0C\u0C0E-\u0C10\u0C12-\u0C28\u0C2A-\u0C33\u0C35-\u0C39\u0C3D\u0C58\u0C59\u0C60\u0C61\u0C85-\u0C8C\u0C8E-\u0C90\u0C92-\u0CA8\u0CAA-\u0CB3\u0CB5-\u0CB9\u0CBD\u0CDE\u0CE0\u0CE1\u0D05-\u0D0C\u0D0E-\u0D10\u0D12-\u0D28\u0D2A-\u0D39\u0D3D\u0D60\u0D61\u0D7A-\u0D7F\u0D85-\u0D96\u0D9A-\u0DB1\u0DB3-\u0DBB\u0DBD\u0DC0-\u0DC6\u0E01-\u0E30\u0E32\u0E33\u0E40-\u0E46\u0E81\u0E82\u0E84\u0E87\u0E88\u0E8A\u0E8D\u0E94-\u0E97\u0E99-\u0E9F\u0EA1-\u0EA3\u0EA5\u0EA7\u0EAA\u0EAB\u0EAD-\u0EB0\u0EB2\u0EB3\u0EBD\u0EC0-\u0EC4\u0EC6\u0EDC\u0EDD\u0F00\u0F40-\u0F47\u0F49-\u0F6C\u0F88-\u0F8B\u1000-\u102A\u103F\u1050-\u1055\u105A-\u105D\u1061\u1065\u1066\u106E-\u1070\u1075-\u1081\u108E\u10A0-\u10C5\u10D0-\u10FA\u10FC\u1100-\u1248\u124A-\u124D\u1250-\u1256\u1258\u125A-\u125D\u1260-\u1288\u128A-\u128D\u1290-\u12B0\u12B2-\u12B5\u12B8-\u12BE\u12C0\u12C2-\u12C5\u12C8-\u12D6\u12D8-\u1310\u1312-\u1315\u1318-\u135A\u1380-\u138F\u13A0-\u13F4\u1401-\u166C\u166F-\u167F\u1681-\u169A\u16A0-\u16EA\u1700-\u170C\u170E-\u1711\u1720-\u1731\u1740-\u1751\u1760-\u176C\u176E-\u1770\u1780-\u17B3\u17D7\u17DC\u1820-\u1877\u1880-\u18A8\u18AA\u18B0-\u18F5\u1900-\u191C\u1950-\u196D\u1970-\u1974\u1980-\u19AB\u19C1-\u19C7\u1A00-\u1A16\u1A20-\u1A54\u1AA7\u1B05-\u1B33\u1B45-\u1B4B\u1B83-\u1BA0\u1BAE\u1BAF\u1C00-\u1C23\u1C4D-\u1C4F\u1C5A-\u1C7D\u1CE9-\u1CEC\u1CEE-\u1CF1\u1D00-\u1DBF\u1E00-\u1F15\u1F18-\u1F1D\u1F20-\u1F45\u1F48-\u1F4D\u1F50-\u1F57\u1F59\u1F5B\u1F5D\u1F5F-\u1F7D\u1F80-\u1FB4\u1FB6-\u1FBC\u1FBE\u1FC2-\u1FC4\u1FC6-\u1FCC\u1FD0-\u1FD3\u1FD6-\u1FDB\u1FE0-\u1FEC\u1FF2-\u1FF4\u1FF6-\u1FFC\u2071\u207F\u2090-\u2094\u2102\u2107\u210A-\u2113\u2115\u2119-\u211D\u2124\u2126\u2128\u212A-\u212D\u212F-\u2139\u213C-\u213F\u2145-\u2149\u214E\u2183\u2184\u2C00-\u2C2E\u2C30-\u2C5E\u2C60-\u2CE4\u2CEB-\u2CEE\u2D00-\u2D25\u2D30-\u2D65\u2D6F\u2D80-\u2D96\u2DA0-\u2DA6\u2DA8-\u2DAE\u2DB0-\u2DB6\u2DB8-\u2DBE\u2DC0-\u2DC6\u2DC8-\u2DCE\u2DD0-\u2DD6\u2DD8-\u2DDE\u2E2F\u3005\u3006\u3031-\u3035\u303B\u303C\u3041-\u3096\u309D-\u309F\u30A1-\u30FA\u30FC-\u30FF\u3105-\u312D\u3131-\u318E\u31A0-\u31B7\u31F0-\u31FF\u3400-\u4DB5\u4E00-\u9FCB\uA000-\uA48C\uA4D0-\uA4FD\uA500-\uA60C\uA610-\uA61F\uA62A\uA62B\uA640-\uA65F\uA662-\uA66E\uA67F-\uA697\uA6A0-\uA6E5\uA717-\uA71F\uA722-\uA788\uA78B\uA78C\uA7FB-\uA801\uA803-\uA805\uA807-\uA80A\uA80C-\uA822\uA840-\uA873\uA882-\uA8B3\uA8F2-\uA8F7\uA8FB\uA90A-\uA925\uA930-\uA946\uA960-\uA97C\uA984-\uA9B2\uA9CF\uAA00-\uAA28\uAA40-\uAA42\uAA44-\uAA4B\uAA60-\uAA76\uAA7A\uAA80-\uAAAF\uAAB1\uAAB5\uAAB6\uAAB9-\uAABD\uAAC0\uAAC2\uAADB-\uAADD\uABC0-\uABE2\uAC00-\uD7A3\uD7B0-\uD7C6\uD7CB-\uD7FB\uF900-\uFA2D\uFA30-\uFA6D\uFA70-\uFAD9\uFB00-\uFB06\uFB13-\uFB17\uFB1D\uFB1F-\uFB28\uFB2A-\uFB36\uFB38-\uFB3C\uFB3E\uFB40\uFB41\uFB43\uFB44\uFB46-\uFBB1\uFBD3-\uFD3D\uFD50-\uFD8F\uFD92-\uFDC7\uFDF0-\uFDFB\uFE70-\uFE74\uFE76-\uFEFC\uFF21-\uFF3A\uFF41-\uFF5A\uFF66-\uFFBE\uFFC2-\uFFC7\uFFCA-\uFFCF\uFFD2-\uFFD7\uFFDA-\uFFDC]/,

		/**
		 * Regex to find non-word characters.
		 */
		nonWordRegex: /[^\u0041-\u005A\u0061-\u007A\u00AA\u00B5\u00BA\u00C0-\u00D6\u00D8-\u00F6\u00F8-\u02C1\u02C6-\u02D1\u02E0-\u02E4\u02EC\u02EE\u0370-\u0374\u0376\u0377\u037A-\u037D\u0386\u0388-\u038A\u038C\u038E-\u03A1\u03A3-\u03F5\u03F7-\u0481\u048A-\u0525\u0531-\u0556\u0559\u0561-\u0587\u05D0-\u05EA\u05F0-\u05F2\u0621-\u064A\u066E\u066F\u0671-\u06D3\u06D5\u06E5\u06E6\u06EE\u06EF\u06FA-\u06FC\u06FF\u0710\u0712-\u072F\u074D-\u07A5\u07B1\u07CA-\u07EA\u07F4\u07F5\u07FA\u0800-\u0815\u081A\u0824\u0828\u0904-\u0939\u093D\u0950\u0958-\u0961\u0971\u0972\u0979-\u097F\u0985-\u098C\u098F\u0990\u0993-\u09A8\u09AA-\u09B0\u09B2\u09B6-\u09B9\u09BD\u09CE\u09DC\u09DD\u09DF-\u09E1\u09F0\u09F1\u0A05-\u0A0A\u0A0F\u0A10\u0A13-\u0A28\u0A2A-\u0A30\u0A32\u0A33\u0A35\u0A36\u0A38\u0A39\u0A59-\u0A5C\u0A5E\u0A72-\u0A74\u0A85-\u0A8D\u0A8F-\u0A91\u0A93-\u0AA8\u0AAA-\u0AB0\u0AB2\u0AB3\u0AB5-\u0AB9\u0ABD\u0AD0\u0AE0\u0AE1\u0B05-\u0B0C\u0B0F\u0B10\u0B13-\u0B28\u0B2A-\u0B30\u0B32\u0B33\u0B35-\u0B39\u0B3D\u0B5C\u0B5D\u0B5F-\u0B61\u0B71\u0B83\u0B85-\u0B8A\u0B8E-\u0B90\u0B92-\u0B95\u0B99\u0B9A\u0B9C\u0B9E\u0B9F\u0BA3\u0BA4\u0BA8-\u0BAA\u0BAE-\u0BB9\u0BD0\u0C05-\u0C0C\u0C0E-\u0C10\u0C12-\u0C28\u0C2A-\u0C33\u0C35-\u0C39\u0C3D\u0C58\u0C59\u0C60\u0C61\u0C85-\u0C8C\u0C8E-\u0C90\u0C92-\u0CA8\u0CAA-\u0CB3\u0CB5-\u0CB9\u0CBD\u0CDE\u0CE0\u0CE1\u0D05-\u0D0C\u0D0E-\u0D10\u0D12-\u0D28\u0D2A-\u0D39\u0D3D\u0D60\u0D61\u0D7A-\u0D7F\u0D85-\u0D96\u0D9A-\u0DB1\u0DB3-\u0DBB\u0DBD\u0DC0-\u0DC6\u0E01-\u0E30\u0E32\u0E33\u0E40-\u0E46\u0E81\u0E82\u0E84\u0E87\u0E88\u0E8A\u0E8D\u0E94-\u0E97\u0E99-\u0E9F\u0EA1-\u0EA3\u0EA5\u0EA7\u0EAA\u0EAB\u0EAD-\u0EB0\u0EB2\u0EB3\u0EBD\u0EC0-\u0EC4\u0EC6\u0EDC\u0EDD\u0F00\u0F40-\u0F47\u0F49-\u0F6C\u0F88-\u0F8B\u1000-\u102A\u103F\u1050-\u1055\u105A-\u105D\u1061\u1065\u1066\u106E-\u1070\u1075-\u1081\u108E\u10A0-\u10C5\u10D0-\u10FA\u10FC\u1100-\u1248\u124A-\u124D\u1250-\u1256\u1258\u125A-\u125D\u1260-\u1288\u128A-\u128D\u1290-\u12B0\u12B2-\u12B5\u12B8-\u12BE\u12C0\u12C2-\u12C5\u12C8-\u12D6\u12D8-\u1310\u1312-\u1315\u1318-\u135A\u1380-\u138F\u13A0-\u13F4\u1401-\u166C\u166F-\u167F\u1681-\u169A\u16A0-\u16EA\u1700-\u170C\u170E-\u1711\u1720-\u1731\u1740-\u1751\u1760-\u176C\u176E-\u1770\u1780-\u17B3\u17D7\u17DC\u1820-\u1877\u1880-\u18A8\u18AA\u18B0-\u18F5\u1900-\u191C\u1950-\u196D\u1970-\u1974\u1980-\u19AB\u19C1-\u19C7\u1A00-\u1A16\u1A20-\u1A54\u1AA7\u1B05-\u1B33\u1B45-\u1B4B\u1B83-\u1BA0\u1BAE\u1BAF\u1C00-\u1C23\u1C4D-\u1C4F\u1C5A-\u1C7D\u1CE9-\u1CEC\u1CEE-\u1CF1\u1D00-\u1DBF\u1E00-\u1F15\u1F18-\u1F1D\u1F20-\u1F45\u1F48-\u1F4D\u1F50-\u1F57\u1F59\u1F5B\u1F5D\u1F5F-\u1F7D\u1F80-\u1FB4\u1FB6-\u1FBC\u1FBE\u1FC2-\u1FC4\u1FC6-\u1FCC\u1FD0-\u1FD3\u1FD6-\u1FDB\u1FE0-\u1FEC\u1FF2-\u1FF4\u1FF6-\u1FFC\u2071\u207F\u2090-\u2094\u2102\u2107\u210A-\u2113\u2115\u2119-\u211D\u2124\u2126\u2128\u212A-\u212D\u212F-\u2139\u213C-\u213F\u2145-\u2149\u214E\u2183\u2184\u2C00-\u2C2E\u2C30-\u2C5E\u2C60-\u2CE4\u2CEB-\u2CEE\u2D00-\u2D25\u2D30-\u2D65\u2D6F\u2D80-\u2D96\u2DA0-\u2DA6\u2DA8-\u2DAE\u2DB0-\u2DB6\u2DB8-\u2DBE\u2DC0-\u2DC6\u2DC8-\u2DCE\u2DD0-\u2DD6\u2DD8-\u2DDE\u2E2F\u3005\u3006\u3031-\u3035\u303B\u303C\u3041-\u3096\u309D-\u309F\u30A1-\u30FA\u30FC-\u30FF\u3105-\u312D\u3131-\u318E\u31A0-\u31B7\u31F0-\u31FF\u3400-\u4DB5\u4E00-\u9FCB\uA000-\uA48C\uA4D0-\uA4FD\uA500-\uA60C\uA610-\uA61F\uA62A\uA62B\uA640-\uA65F\uA662-\uA66E\uA67F-\uA697\uA6A0-\uA6E5\uA717-\uA71F\uA722-\uA788\uA78B\uA78C\uA7FB-\uA801\uA803-\uA805\uA807-\uA80A\uA80C-\uA822\uA840-\uA873\uA882-\uA8B3\uA8F2-\uA8F7\uA8FB\uA90A-\uA925\uA930-\uA946\uA960-\uA97C\uA984-\uA9B2\uA9CF\uAA00-\uAA28\uAA40-\uAA42\uAA44-\uAA4B\uAA60-\uAA76\uAA7A\uAA80-\uAAAF\uAAB1\uAAB5\uAAB6\uAAB9-\uAABD\uAAC0\uAAC2\uAADB-\uAADD\uABC0-\uABE2\uAC00-\uD7A3\uD7B0-\uD7C6\uD7CB-\uD7FB\uF900-\uFA2D\uFA30-\uFA6D\uFA70-\uFAD9\uFB00-\uFB06\uFB13-\uFB17\uFB1D\uFB1F-\uFB28\uFB2A-\uFB36\uFB38-\uFB3C\uFB3E\uFB40\uFB41\uFB43\uFB44\uFB46-\uFBB1\uFBD3-\uFD3D\uFD50-\uFD8F\uFD92-\uFDC7\uFDF0-\uFDFB\uFE70-\uFE74\uFE76-\uFEFC\uFF21-\uFF3A\uFF41-\uFF5A\uFF66-\uFFBE\uFFC2-\uFFC7\uFFCA-\uFFCF\uFFD2-\uFFD7\uFFDA-\uFFDC]/,

		/**
		 * Tags which can safely be merged
		 * @hide
		 */
		mergeableTags: ['b', 'code', 'del', 'em', 'i', 'ins', 'strong', 'sub', 'sup', '#text'],

		/**
		 * Tags which do not mark word boundaries
		 * @hide
		 */
		nonWordBoundaryTags: ['a', 'b', 'code', 'del', 'em', 'i', 'ins', 'span', 'strong', 'sub', 'sup', '#text'],

		/**
		 * Tags which are considered 'nonempty', even if they have no children (or not data)
		 * TODO: finish this list
		 * @hide
		 */
		nonEmptyTags: ['br'],

		/**
		 * Tags which make up Flow Content or Phrasing Content, according to the HTML 5 specification,
		 * @see http://dev.w3.org/html5/spec/Overview.html#flow-content
		 * @see http://dev.w3.org/html5/spec/Overview.html#phrasing-content
		 * @hide
		 */
		tags: {
			'flow': ['a', 'abbr', 'address', 'area', 'article', 'aside', 'audio', 'b', 'bdi', 'bdo', 'blockquote', 'br', 'button', 'canvas', 'cite', 'code', 'command', 'datalist', 'del', 'details', 'dfn', 'div', 'dl', 'em', 'embed', 'fieldset', 'figure', 'footer', 'form', 'h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'header', 'hgroup', 'hr', 'i', 'iframe', 'img', 'input', 'ins', 'kbd', 'keygen', 'label', 'map', 'mark', 'math', 'menu', 'meter', 'nav', 'noscript', 'object', 'ol', 'output', 'p', 'pre', 'progress', 'q', 'ruby', 's', 'samp', 'script', 'section', 'select', 'small', 'span', 'strong', 'style', 'sub', 'sup', 'svg', 'table', 'textarea', 'time', 'u', 'ul', 'var', 'video', 'wbr', '#text'],
			'phrasing': ['a', 'abbr', 'area', 'audio', 'b', 'bdi', 'bdo', 'br', 'button', 'canvas', 'cite', 'code', 'command', 'datalist', 'del', 'dfn', 'em', 'embed', 'i', 'iframe', 'img', 'input', 'ins', 'kbd', 'keygen', 'label', 'map', 'mark', 'math', 'meter', 'noscript', 'object', 'output', 'progress', 'q', 'ruby', 'samp', 'script', 'select', 'small', 'span', 'strong', 'sub', 'sup', 'svg', 'textarea', 'time', 'u', 'var', 'video', 'wbr', '#text']
		},

		/**
		 * Possible children of tags, according to the HTML 5
		 * specification.
		 * See http://dev.w3.org/html5/spec/Overview.html#elements-1
		 * Moved to http://www.whatwg.org/specs/web-apps/current-work/#elements-1
		 * @hide
		 */
		children: {
			'a': 'phrasing', // transparent
			'abbr': 'phrasing',
			'address': 'flow',
			'area': 'empty',
			'article': 'flow',
			'aside': 'flow',
			'audio': 'source', // transparent
			'b': 'phrasing',
			'base': 'empty',
			'bdo': 'phrasing',
			'blockquote': 'phrasing',
			'body': 'flow',
			'br': 'empty',
			'button': 'phrasing',
			'canvas': 'phrasing', // transparent
			'caption': 'flow',
			'cite': 'phrasing',
			'code': 'phrasing',
			'col': 'empty',
			'colgroup': 'col',
			'command': 'empty',
			'datalist': ['phrasing', 'option'],
			'dd': 'flow',
			'del': 'phrasing',
			'div': 'flow',
			'details': ['summary', 'flow'],
			'dfn': 'flow',
			'dl': ['dt', 'dd'],
			'dt': 'phrasing', // varies
			'em': 'phrasing',
			'embed': 'empty',
			'fieldset': ['legend', 'flow'],
			'figcaption': 'flow',
			'figure': ['figcaption', 'flow'],
			'footer': 'flow',
			'form': 'flow',
			'h1': 'phrasing',
			'h2': 'phrasing',
			'h3': 'phrasing',
			'h4': 'phrasing',
			'h5': 'phrasing',
			'h6': 'phrasing',
			//head
			'header': 'flow',
			'hgroup': ['h1', 'h2', 'h3', 'h4', 'h5', 'h6'],
			'hr': 'empty',
			//html :)
			'i': 'phrasing',
			'iframe': '#text',
			'img': 'empty',
			'input': 'empty',
			'ins': 'phrasing', // transparent
			'kbd': 'phrasing',
			'keygen': 'empty',
			'label': 'phrasing',
			'legend': 'phrasing',
			'li': 'flow',
			'link': 'empty',
			'map': 'area', // transparent
			'mark': 'phrasing',
			'menu': ['li', 'flow'],
			'meta': 'empty',
			'meter': 'phrasing',
			'nav': 'flow',
			'noscript': 'phrasing', // varies
			'object': 'param', // transparent
			'ol': 'li',
			'optgroup': 'option',
			'option': '#text',
			'output': 'phrasing',
			'p': 'phrasing',
			'param': 'empty',
			'pre': 'phrasing',
			'progress': 'phrasing',
			'q': 'phrasing',
			'rp': 'phrasing',
			'rt': 'phrasing',
			'ruby': ['phrasing', 'rt', 'rp'],
			's': 'phrasing',
			'samp': 'pharsing',
			'script': '#script', //script
			'section': 'flow',
			'select': ['option', 'optgroup'],
			'small': 'phrasing',
			'source': 'empty',
			'span': 'phrasing',
			'strong': 'phrasing',
			'style': 'phrasing', // varies
			'sub': 'phrasing',
			'summary': 'phrasing',
			'sup': 'phrasing',
			'table': ['caption', 'colgroup', 'thead', 'tbody', 'tfoot', 'tr'],
			'tbody': 'tr',
			'td': 'flow',
			'textarea': '#text',
			'tfoot': 'tr',
			'th': 'phrasing',
			'thead': 'tr',
			'time': 'phrasing',
			'title': '#text',
			'tr': ['th', 'td'],
			'track': 'empty',
			'u': 'phrasing',
			'ul': 'li',
			'var': 'phrasing',
			'video': 'source', // transparent
			'wbr': 'empty'
		},

		/**
		 * List of nodenames of blocklevel elements
		 * TODO: finish this list
		 * @hide
		 */
		blockLevelElements: ['p', 'h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'blockquote', 'div', 'pre'],

		/**
		 * List of nodenames of list elements
		 * @hide
		 */
		listElements: ['li', 'ol', 'ul'],

		/**
		 * Splits a DOM element at the given position up until the limiting object(s), so that it is valid HTML again afterwards.
		 * @param {RangeObject} range Range object that indicates the position of the splitting.
		 *				This range will be updated, so that it represents the same range as before the split.
		 * @param {jQuery} limit Limiting node(s) for the split.
		 *				The limiting node will not be included in the split itself.
		 *				If no limiting object is set, the document body will be the limiting object.
		 * @param {boolean} atEnd If set to true, the DOM will be splitted at the end of the range otherwise at the start.
		 * @return {object} jQuery object containing the two root DOM objects of the split, true if the DOM did not need to be split or false if the DOM could not be split
		 * @method
		 */
		split: function (range, limit, atEnd) {
			var splitElement = jQuery(range.startContainer),
				splitPosition = range.startOffset,
				updateRange,
			    path,
			    parents,
				newDom,
			    insertElement,
			    secondPart,
				i,
			    pathLength,
			    element,
			    jqelement,
			    children,
			    newElement,
				next,
			    prev,
			    offset;

			if (atEnd) {
				splitElement = jQuery(range.endContainer);
				splitPosition = range.endOffset;
			}

			if (limit.length < 1) {
				limit = jQuery(document.body);
			}

			// we may have to update the range if it is not collapsed and we are splitting at the start
			updateRange = (!range.isCollapsed() && !atEnd);

			// find the path up to the highest object that will be splitted
			parents = splitElement.parents().get();
			parents.unshift(splitElement.get(0));

			jQuery.each(parents, function (index, element) {
				var isLimit = limit.filter(
					function () {
						return this == element;
					}
				).length;
				if (isLimit) {
					if (index > 0) {
						path = parents.slice(0, index);
					}
					return false;
				}
			});

			// nothing found to split -> return here
			if (!path) {
				return true;
			}

			path = path.reverse();

			// iterate over the path, create new dom nodes for every element and move
			// the contents right of the split to the new element
			for (i = 0, pathLength = path.length; i < pathLength; ++i) {
				element = path[i];
				if (i === pathLength - 1) {
					// last element in the path -> we have to split it

					// split the last part into two parts
					if (element.nodeType === 3) {
						// text node
						secondPart = document.createTextNode(element.data.substring(splitPosition, element.data.length));
						element.data = element.data.substring(0, splitPosition);
					} else {
						// other nodes
						jqelement = jQuery(element);
						children = jqelement.contents();
						newElement = jqelement.clone(false).empty();
						secondPart = newElement.append(children.slice(splitPosition, children.length)).get(0);
					}

					// update the range if necessary
					if (updateRange && range.endContainer === element) {
						range.endContainer = secondPart;
						range.endOffset -= splitPosition;
						range.clearCaches();
					}

					// add the second part
					if (insertElement) {
						insertElement.prepend(secondPart);
					} else {
						jQuery(element).after(secondPart);
					}
				} else {
					// create the new element of the same type and prepend it to the previously created element
					newElement = jQuery(element).clone(false).empty();

					if (!newDom) {
						newDom = newElement;
					} else {
						insertElement.prepend(newElement);
					}
					insertElement = newElement;

					// move all contents right of the split to the new element
					while (true) {
						next = path[i + 1].nextSibling;
						if (!next) {
							break;
						}
						insertElement.append(next);
					}

					// update the range if necessary
					if (updateRange && range.endContainer === element) {
						range.endContainer = newElement.get(0);
						prev = path[i + 1];
						offset = 0;
						while (true) {
							prev = prev.previousSibling;
							if (!prev) {
								break;
							}
							offset++;
						}
						range.endOffset -= offset;
						range.clearCaches();
					}
				}
			}

			// append the new dom
			jQuery(path[0]).after(newDom);

			return jQuery([path[0], newDom ? newDom.get(0) : secondPart]);
		},

		/**
		 * Check whether the HTML 5 specification allows direct nesting of the given DOM
		 * objects.
		 * @param {object} outerDOMObject
		 *            outer (nesting) DOM Object
		 * @param {object} innerDOMObject
		 *            inner (nested) DOM Object
		 * @return {boolean} true when the nesting is allowed, false if not
		 * @method
		 */
		allowsNesting: function (outerDOMObject, innerDOMObject) {
			if (!outerDOMObject || !outerDOMObject.nodeName || !innerDOMObject || !innerDOMObject.nodeName) {
				return false;
			}

			var outerNodeName = outerDOMObject.nodeName.toLowerCase(),
				innerNodeName = innerDOMObject.nodeName.toLowerCase();

			if (!this.children[outerNodeName]) {
				return false;
			}

			// check whether the nesting is configured by node names (like for table)
			if (this.children[outerNodeName] == innerNodeName) {
				return true;
			}
			if (jQuery.isArray(this.children[outerNodeName]) && jQuery.inArray(innerNodeName, this.children[outerNodeName]) >= 0) {
				return true;
			}

			if (jQuery.isArray(this.tags[this.children[outerNodeName]])
				    && jQuery.inArray(innerNodeName, this.tags[this.children[outerNodeName]]) >= 0) {
				return true;
			}

			return false;
		},

		/**
		 * Apply the given markup additively to the given range. The given rangeObject will be modified if necessary
		 * @param {GENTICS.Utils.RangeObject} rangeObject range to which the markup shall be added
		 * @param {jQuery} markup markup to be applied as jQuery object
		 * @param {boolean} allownesting true when nesting of the added markup is allowed, false if not (default: false)
		 * @method
		 */
		addMarkup: function (rangeObject, markup, nesting) {
			// split partially contained text nodes at the start and end of the range
			if (rangeObject.startContainer.nodeType === 3
				    && rangeObject.startOffset > 0
				    && rangeObject.startOffset < rangeObject.startContainer.data.length) {
				this.split(rangeObject, jQuery(rangeObject.startContainer).parent(), false);
			}
			if (rangeObject.endContainer.nodeType === 3 && rangeObject.endOffset > 0 && rangeObject.endOffset < rangeObject.endContainer.data.length) {
				this.split(rangeObject, jQuery(rangeObject.endContainer).parent(), true);
			}

			// get the range tree
			var rangeTree = rangeObject.getRangeTree();
			this.recursiveAddMarkup(rangeTree, markup, rangeObject, nesting);

			// cleanup DOM
			this.doCleanup({
				'merge': true,
				'removeempty': true
			}, rangeObject);
		},

		/**
		 * Recursive helper method to add the given markup to the range
		 * @param rangeTree rangetree at the current level
		 * @param markup markup to be applied
		 * @param rangeObject range object, which eventually is updated
		 * @param nesting true when nesting of the added markup is allowed, false if not
		 * @hide
		 */
		recursiveAddMarkup: function (rangeTree, markup, rangeObject, nesting) {
			var i, innerRange, rangeLength;

			// iterate through all rangetree objects of that level
			for (i = 0, rangeLength = rangeTree.length; i < rangeLength; ++i) {
				// check whether the rangetree object is fully contained and the markup may be wrapped around the object
				if (rangeTree[i].type == 'full' && this.allowsNesting(markup.get(0), rangeTree[i].domobj)) {
					// we wrap the object, when
					// 1. nesting of markup is allowed or the node is not of the markup to be added
					// 2. the node an element node or a non-empty text node
					if ((nesting || rangeTree[i].domobj.nodeName != markup.get(0).nodeName) && (rangeTree[i].domobj.nodeType !== 3 || jQuery.trim(rangeTree[i].domobj.data).length !== 0)) {
						// wrap the object
						jQuery(rangeTree[i].domobj).wrap(markup);

						// TODO eventually update the range (if it changed)

						// when nesting is not allowed, we remove the markup from the inner element
						if (!nesting && rangeTree[i].domobj.nodeType !== 3) {
							innerRange = new GENTICS.Utils.RangeObject();
							innerRange.startContainer = innerRange.endContainer = rangeTree[i].domobj.parentNode;
							innerRange.startOffset = 0;
							innerRange.endOffset = innerRange.endContainer.childNodes.length;
							this.removeMarkup(innerRange, markup, jQuery(rangeTree[i].domobj.parentNode));
						}
					}
				} else {
					// TODO check whether the object may be replaced by the given markup
					//if (false) {
					// TODO replace
					//} else {
					// recurse into the children (if any), but not if nesting is not
					// allowed and the object is of the markup to be added
					if ((nesting || (rangeTree[i].domobj && rangeTree[i].domobj.nodeName !== markup.get(0).nodeName)) && rangeTree[i].children && rangeTree[i].children.length > 0) {
						this.recursiveAddMarkup(rangeTree[i].children, markup);
					}
				}
			}
		},

		/**
		 * Find the highest occurrence of a node with given nodename within the parents
		 * of the start. When limit objects are given, the search stops there.
		 * The limiting object is of the found type, it won't be considered
		 * @param {DOMObject} start start object
		 * @param {String} nodeName name of the node to search for (case-insensitive)
		 * @param {jQuery} limit Limiting node(s) as jQuery object (if none given, the search will stop when there are no more parents)
		 * @return {DOMObject} the found DOM object or undefined
		 * @method
		 */
		findHighestElement: function (start, nodeName, limit) {
			nodeName = nodeName.toLowerCase();

			// this will be the highest found markup object (up to a limit object)
			var highestObject,
			// blah
			    testObject = start,
				// helper function to stop when we reach a limit object
				isLimit = limit ? function () {
					return limit.filter(
						function () {
							return testObject == this;
						}
					).length;
				} : function () {
					return false;
				};

			// now get the highest parent that has the given markup (until we reached
			// one of the limit objects or there are no more parent nodes)
			while (!isLimit() && testObject) {
				if (testObject.nodeName.toLowerCase() === nodeName) {
					highestObject = testObject;
				}
				testObject = testObject.parentNode;
			}

			return highestObject;
		},

		/**
		 * Remove the given markup from the given range. The given rangeObject will be modified if necessary
		 * TODO: add parameter deep/shallow
		 * @param {GENTICS.Utils.RangeObject} rangeObject range from which the markup shall be removed
		 * @param {jQuery} markup markup to be removed as jQuery object
		 * @param {jQuery} limit Limiting node(s) as jQuery object
		 * @method
		 */
		removeMarkup: function (rangeObject, markup, limit) {
			var nodeName = markup.get(0).nodeName,
				startSplitLimit = this.findHighestElement(rangeObject.startContainer, nodeName, limit),
				endSplitLimit = this.findHighestElement(rangeObject.endContainer, nodeName, limit),
				didSplit = false,
				highestObject,
			    root,
			    rangeTree;

			if (startSplitLimit && rangeObject.startOffset > 0) {
				// when the start is in the start of its container, we don't split
				this.split(rangeObject, jQuery(startSplitLimit).parent(), false);
				didSplit = true;
			}

			if (endSplitLimit) {
				// when the end is in the end of its container, we don't split
				if (rangeObject.endContainer.nodeType === 3 && rangeObject.endOffset < rangeObject.endContainer.data.length) {
					this.split(rangeObject, jQuery(endSplitLimit).parent(), true);
					didSplit = true;
				}
				if (rangeObject.endContainer.nodeType === 1 && rangeObject.endOffset < rangeObject.endContainer.childNodes.length) {
					this.split(rangeObject, jQuery(endSplitLimit).parent(), true);
					didSplit = true;
				}
			}

			// when we split the DOM, we maybe need to correct the range
			if (didSplit) {
				rangeObject.correctRange();
			}

			// find the highest occurrence of the markup
			highestObject = this.findHighestElement(rangeObject.getCommonAncestorContainer(), nodeName, limit);
			root = highestObject ? highestObject.parentNode : rangeObject.getCommonAncestorContainer();

			if (root) {
				// construct the range tree
				rangeTree = rangeObject.getRangeTree(root);

				// remove the markup from the range tree
				this.recursiveRemoveMarkup(rangeTree, markup);

				// cleanup DOM
				this.doCleanup({
					'merge': true,
					'removeempty': true
				}, rangeObject, root);
			}
		},

		/**
		 * TODO: pass the range itself and eventually update it if necessary
		 * Recursive helper method to remove the given markup from the range
		 * @param rangeTree rangetree at the current level
		 * @param markup markup to be applied
		 * @hide
		 */
		recursiveRemoveMarkup: function (rangeTree, markup) {
			var i, rangeLength, content;
			// iterate over the rangetree objects of this level
			for (i = 0, rangeLength = rangeTree.length; i < rangeLength; ++i) {
				// check whether the object is the markup to be removed and is fully into the range
				if (rangeTree[i].type == 'full' && rangeTree[i].domobj.nodeName == markup.get(0).nodeName) {
					// found the markup, so remove it
					content = jQuery(rangeTree[i].domobj).contents();
					if (content.length > 0) {
						// when the object has children, we unwrap them
						content.first().unwrap();
					} else {
						// obj has no children, so just remove it
						jQuery(rangeTree[i].domobj).remove();
					}
				}

				// if the object has children, we do the recursion now
				if (rangeTree[i].children) {
					this.recursiveRemoveMarkup(rangeTree[i].children, markup);
				}
			}
		},

		/**
		 * Cleanup the DOM, starting with the given startobject (or the common ancestor container of the given range)
		 * ATTENTION: If range is a selection you need to update the selection after doCleanup
		 * Cleanup modes (given as properties in 'cleanup'):
		 * <pre>
		 * - merge: merges multiple successive nodes of same type, if this is allowed, starting at the children of the given node (defaults to false)
		 * - removeempty: removes empty element nodes (defaults to false)
		 * </pre>
		 * Example for calling this method:<br/>
		 * <code>GENTICS.Utils.Dom.doCleanup({merge:true,removeempty:false}, range)</code>
		 * @param {object} cleanup type of cleanup to be done
		 * @param {GENTICS.Utils.RangeObject} rangeObject range which is eventually updated
		 * @param {DOMObject} start start object, if not given, the commonancestorcontainer is used as startobject insted
		 * @return {boolean} true when the range (startContainer/startOffset/endContainer/endOffset) was modified, false if not
		 * @method
		 */
		doCleanup: function (cleanup, rangeObject, start) {
			var that = this,
				prevNode,
			    modifiedRange,
			    startObject,
			    startOffset,
			    endOffset;

			if (typeof cleanup === 'undefined') {
				cleanup = {};
			}
			if (typeof cleanup.merge === 'undefined') {
				cleanup.merge = false;
			}
			if (typeof cleanup.removeempty === 'undefined') {
				cleanup.removeempty = false;
			}

			if (typeof start === 'undefined' && rangeObject) {
				start = rangeObject.getCommonAncestorContainer();
			}
			// remember the previous node here (successive nodes of same type will be merged into this)
			prevNode = false;
			// check whether the range needed to be modified during merging
			modifiedRange = false;
			// get the start object
			startObject = jQuery(start);
			startOffset = rangeObject.startOffset;
			endOffset = rangeObject.endOffset;

			// iterate through all sub nodes
			startObject.contents().each(function () {
				var index;

				// Try to read the nodeType property and return if we do not have permission
				// ie.: frame document to an external URL
				var nodeType;
				try {
					nodeType = this.nodeType;
					index = that.getIndexInParent(this);
				} catch (e) {
					return;
				}

				// decide further actions by node type
				switch (nodeType) {
					// found a non-text node
				case 1:
					if (prevNode && prevNode.nodeName == this.nodeName) {
						// found a successive node of same type

						// now we check whether the selection starts or ends in the mother node after the current node
						if (rangeObject.startContainer === startObject && startOffset > index) {
							// there will be one less object, so reduce the startOffset by one
							rangeObject.startOffset -= 1;
							// set the flag for range modification
							modifiedRange = true;
						}
						if (rangeObject.endContainer === startObject && endOffset > index) {
							// there will be one less object, so reduce the endOffset by one
							rangeObject.endOffset -= 1;
							// set the flag for range modification
							modifiedRange = true;
						}

						// merge the contents of this node into the previous one
						jQuery(prevNode).append(jQuery(this).contents());

						// after merging, we eventually need to cleanup the prevNode again
						modifiedRange |= that.doCleanup(cleanup, rangeObject, prevNode);

						// remove this node
						jQuery(this).remove();

					} else {

						// do the recursion step here
						modifiedRange |= that.doCleanup(cleanup, rangeObject, this);

						// eventually remove empty elements
						var removed = false;
						if (cleanup.removeempty) {
							if (GENTICS.Utils.Dom.isBlockLevelElement(this) && this.childNodes.length === 0) {
								//							jQuery(this).remove();
								removed = true;
							}
							if (jQuery.inArray(this.nodeName.toLowerCase(), that.mergeableTags) >= 0 && jQuery(this).text().length === 0 && this.childNodes.length === 0) {
								//							jQuery(this).remove();
								removed = true;
							}
						}

						// when the current node was not removed, we eventually store it as previous (mergeable) tag
						if (!removed) {
							if (jQuery.inArray(this.nodeName.toLowerCase(), that.mergeableTags) >= 0) {
								prevNode = this;
							} else {
								prevNode = false;
							}
						} else {
							// now we check whether the selection starts or ends in the mother node of this
							if (rangeObject.startContainer === this.parentNode && startOffset > index) {
								// there will be one less object, so reduce the startOffset by one
								rangeObject.startOffset = rangeObject.startOffset - 1;
								// set the flag for range modification
								modifiedRange = true;
							}
							if (rangeObject.endContainer === this.parentNode && endOffset > index) {
								// there will be one less object, so reduce the endOffset by one
								rangeObject.endOffset = rangeObject.endOffset - 1;
								// set the flag for range modification
								modifiedRange = true;
							}

							// remove this text node
							jQuery(this).remove();

						}
					}

					break;
					// found a text node
				case 3:
					// found a text node
					if (prevNode && prevNode.nodeType === 3 && cleanup.merge) {
						// the current text node will be merged into the last one, so
						// check whether the selection starts or ends in the current
						// text node
						if (rangeObject.startContainer === this) {
							// selection starts in the current text node

							// update the start container to the last node
							rangeObject.startContainer = prevNode;

							// update the start offset
							rangeObject.startOffset += prevNode.nodeValue.length;

							// set the flag for range modification
							modifiedRange = true;

						} else if (rangeObject.startContainer === prevNode.parentNode && rangeObject.startOffset === that.getIndexInParent(prevNode) + 1) {
							// selection starts right between the previous and current text nodes (which will be merged)

							// update the start container to the previous node
							rangeObject.startContainer = prevNode;

							// set the start offset
							rangeObject.startOffset = prevNode.nodeValue.length;

							// set the flag for range modification
							modifiedRange = true;
						}

						if (rangeObject.endContainer === this) {
							// selection ends in the current text node

							// update the end container to be the last node
							rangeObject.endContainer = prevNode;

							// update the end offset
							rangeObject.endOffset += prevNode.nodeValue.length;

							// set the flag for range modification
							modifiedRange = true;

						} else if (rangeObject.endContainer === prevNode.parentNode && rangeObject.endOffset === that.getIndexInParent(prevNode) + 1) {
							// selection ends right between the previous and current text nodes (which will be merged)

							// update the end container to the previous node
							rangeObject.endContainer = prevNode;

							// set the end offset
							rangeObject.endOffset = prevNode.nodeValue.length;

							// set the flag for range modification
							modifiedRange = true;
						}

						// now append the contents of the current text node into the previous
						prevNode.data += this.data;

						// remove empty text nodes	
					} else if (!(this.nodeValue === '' && cleanup.removeempty)) {
						prevNode = this;
						// we are finish here don't delete this node
						break;
					}

					// now we check whether the selection starts or ends in the mother node of this
					if (rangeObject.startContainer === this.parentNode && rangeObject.startOffset > index) {
						// there will be one less object, so reduce the startOffset by one
						rangeObject.startOffset = rangeObject.startOffset - 1;
						// set the flag for range modification
						modifiedRange = true;
					}
					if (rangeObject.endContainer === this.parentNode && rangeObject.endOffset > index) {
						// there will be one less object, so reduce the endOffset by one
						rangeObject.endOffset = rangeObject.endOffset - 1;
						// set the flag for range modification
						modifiedRange = true;
					}

					// remove this text node
					jQuery(this).remove();

					// if this is the last text node in a sequence, we remove any zero-width spaces in the text node,
					// unless it is the only character
					if (prevNode && (!prevNode.nextSibling || prevNode.nextSibling.nodeType !== 3)) {
						var pos;
						for (pos = prevNode.data.length - 1; pos >= 0 && prevNode.data.length > 1; pos--) {
							if (prevNode.data.charAt(pos) === '\u200b') {
								prevNode.deleteData(pos, 1);
								if (rangeObject.startContainer === prevNode && rangeObject.startOffset > pos) {
									rangeObject.startOffset--;
									modifiedRange = true;
								}
								if (rangeObject.endContainer === prevNode && rangeObject.endOffset > pos) {
									rangeObject.endOffset--;
									modifiedRange = true;
								}
							}
						}
					}

					break;
				}
			});

			// eventually remove the startnode itself
			//		if (cleanup.removeempty
			//				&& GENTICS.Utils.Dom.isBlockLevelElement(start)
			//				&& (!start.childNodes || start.childNodes.length === 0)) {
			//			if (rangeObject.startContainer == start) {
			//				rangeObject.startContainer = start.parentNode;
			//				rangeObject.startOffset = GENTICS.Utils.Dom.getIndexInParent(start);
			//			}
			//			if (rangeObject.endContainer == start) {
			//				rangeObject.endContainer = start.parentNode;
			//				rangeObject.endOffset = GENTICS.Utils.Dom.getIndexInParent(start);
			//			}
			//			startObject.remove();
			//			modifiedRange = true;
			//		}

			if (modifiedRange) {
				rangeObject.clearCaches();
			}

			return modifiedRange;
		},

		/**
		 * Get the index of the given node within its parent node
		 * @param {DOMObject} node node to check
		 * @return {Integer} index in the parent node or false if no node given or node has no parent
		 * @method
		 */
		getIndexInParent: function (node) {
			if (!node) {
				return false;
			}

			var index = 0,
				check = node.previousSibling;

			while (check) {
				index++;
				check = check.previousSibling;
			}

			return index;
		},

		/**
		 * Check whether the given node is a blocklevel element
		 * @param {DOMObject} node node to check
		 * @return {boolean} true if yes, false if not (or null)
		 * @method
		 */
		isBlockLevelElement: function (node) {
			if (!node) {
				return false;
			}
			if (node.nodeType === 1 && jQuery.inArray(node.nodeName.toLowerCase(), this.blockLevelElements) >= 0) {
				return true;
			}
			return false;
		},

		/**
		 * Check whether the given node is a linebreak element
		 * @param {DOMObject} node node to check
		 * @return {boolean} true for linebreak elements, false for everything else
		 * @method
		 */
		isLineBreakElement: function (node) {
			if (!node) {
				return false;
			}
			return node.nodeType === 1 && node.nodeName.toLowerCase() == 'br';
		},

		/**
		 * Check whether the given node is a list element
		 * @param {DOMObject} node node to check
		 * @return {boolean} true for list elements (li, ul, ol), false for everything else
		 * @method
		 */
		isListElement: function (node) {
			if (!node) {
				return false;
			}
			return node.nodeType === 1 && jQuery.inArray(node.nodeName.toLowerCase(), this.listElements) >= 0;
		},

		/**
		 * This method checks, whether the passed dom object is a dom object, that would
		 * be split in cases of pressing enter. This currently is true for paragraphs
		 * and headings
		 * @param {DOMObject} el
		 *            dom object to check
		 * @return {boolean} true for split objects, false for other
		 * @method
		 */
		isSplitObject: function (el) {
			return el.nodeType === 1 && blockElementNames.hasOwnProperty(el.nodeName);
		},

		/**
		 * Starting with the given position (between nodes), search in the given direction to an adjacent notempty text node
		 * @param {DOMObject} parent parent node containing the position
		 * @param {Integer} index index of the position within the parent node
		 * @param {boolean} searchleft true when search direction is 'left' (default), false for 'right'
		 * @param {object} stopat define at which types of element we shall stop, may contain the following properties
		 * <pre>
		 * - blocklevel (default: true)
		 * - list (default: true)
		 * - linebreak (default: true)
		 * </pre>
		 * @return {DOMObject} the found text node or false if none found
		 * @method
		 */
		searchAdjacentTextNode: function (parent, index, searchleft, stopat) {
			if (!parent || parent.nodeType !== 1 || index < 0 || index > parent.childNodes.length) {
				return false;
			}

			if (typeof stopat === 'undefined') {
				stopat = {
					'blocklevel': true,
					'list': true,
					'linebreak': true
				};
			}

			if (typeof stopat.blocklevel === 'undefined') {
				stopat.blocklevel = true;
			}
			if (typeof stopat.list === 'undefined') {
				stopat.list = true;
			}
			if (typeof stopat.linebreak === 'undefined') {
				stopat.linebreak = true;
			}

			if (typeof searchleft === 'undefined') {
				searchleft = true;
			}

			var nextNode,
			    currentParent = parent;

			// start at the node left/right of the given position
			if (searchleft && index > 0) {
				nextNode = parent.childNodes[index - 1];
			}
			if (!searchleft && index < parent.childNodes.length) {
				nextNode = parent.childNodes[index];
			}

			//currentParent is not a number therefore it is sufficient to directly test for it with while(currentParent)
			//otherwise there would be an error if the object is null
			while (currentParent) {
				//while (typeof currentParent !== 'undefined') {
				if (!nextNode) {
					// no next node found, check whether the parent is a blocklevel element
					if (stopat.blocklevel && this.isBlockLevelElement(currentParent)) {
						// do not leave block level elements
						return false;
					}
					if (stopat.list && this.isListElement(currentParent)) {
						// do not leave list elements
						return false;
					}
					// continue with the parent
					nextNode = searchleft ? currentParent.previousSibling : currentParent.nextSibling;
					currentParent = currentParent.parentNode;
					continue;
				} else if (nextNode.nodeType === 3 && jQuery.trim(nextNode.data).length > 0) {
					// we are lucky and found a notempty text node
					return nextNode;
				}
				if (stopat.blocklevel && this.isBlockLevelElement(nextNode)) {
					// we found a blocklevel element, stop here
					return false;
				}
				if (stopat.linebreak && this.isLineBreakElement(nextNode)) {
					// we found a linebreak, stop here
					return false;
				}
				if (stopat.list && this.isListElement(nextNode)) {
					// we found a linebreak, stop here
					return false;
				}
				if (nextNode.nodeType === 3) {
					// we found an empty text node, so step to the next
					nextNode = searchleft ? nextNode.previousSibling : nextNode.nextSibling;
				} else {
					// we found a non-blocklevel element, step into
					currentParent = nextNode;
					nextNode = searchleft ? nextNode.lastChild : nextNode.firstChild;
				}
			}
		},

		/**
		 * Insert the given DOM Object into the start/end of the given range. The method
		 * will find the appropriate place in the DOM tree for inserting the given
		 * object, and will eventually split elements in between. The given range will
		 * be updated if necessary. The updated range will NOT embrace the inserted
		 * object, which means that the object is actually inserted before or after the
		 * given range (depending on the atEnd parameter)
		 *
		 * @param {jQuery}
		 *				object object to insert into the DOM
		 * @param {GENTICS.Utils.RangeObject}
		 *				range range where to insert the object (at start or end)
		 * @param {jQuery}
		 *				limit limiting object(s) of the DOM modification
		 * @param {boolean}
		 *				atEnd true when the object shall be inserted at the end, false for
		 *				insertion at the start (default)
		 * @param {boolean}
		 *				true when the insertion shall be done, even if inserting the element
		 *				would not be allowed, false to deny inserting unallowed elements (default)
		 * @return true if the object could be inserted, false if not.
		 * @method
		 */
		insertIntoDOM: function (object, range, limit, atEnd, force) {
			// first find the appropriate place to insert the given object
			var parentElements = range.getContainerParents(limit, atEnd),
				that = this,
				newParent,
				container,
			    offset,
			    splitParts,
			    contents;

			if (!limit) {
				limit = jQuery(document.body);
			}

			// if no parent elements exist (up to the limit), the new parent will be the
			// limiter itself
			if (parentElements.length === 0) {
				newParent = limit.get(0);
			} else {
				jQuery.each(parentElements, function (index, parent) {
					if (that.allowsNesting(parent, object.get(0))) {
						newParent = parent;
						return false;
					}
				});
			}

			if (typeof newParent === 'undefined' && limit.length > 0) {
				// found no possible new parent, so split up to the limit object
				newParent = limit.get(0);
			}

			// check whether it is allowed to insert the element at all
			if (!this.allowsNesting(newParent, object.get(0)) && !force) {
				return false;
			}

			if (typeof newParent !== 'undefined') {
				// we found a possible new parent, so we split the DOM up to the new parent
				splitParts = this.split(range, jQuery(newParent), atEnd);
				if (splitParts === true) {
					// DOM was not split (there was no need to split it), insert the new object anyway
					container = range.startContainer;
					offset = range.startOffset;
					if (atEnd) {
						container = range.endContainer;
						offset = range.endOffset;
					}
					if (offset === 0) {
						// insert right before the first element in the container
						contents = jQuery(container).contents();
						if (contents.length > 0) {
							contents.eq(0).before(object);
						} else {
							jQuery(container).append(object);
						}
						return true;
					}
					// insert right after the element at offset-1
					jQuery(container).contents().eq(offset - 1).after(object);
					return true;
				}
				if (splitParts) {
					// if the DOM could be split, we insert the new object in between the split parts
					splitParts.eq(0).after(object);
					return true;
				}
				// could not split, so could not insert
				return false;
			}
			// found no possible new parent, so we shall not insert
			return false;
		},

		/**
		 * Remove the given DOM object from the DOM and modify the given range to reflect the user expected range after the object was removed
		 * TODO: finish this
		 * @param {DOMObject} object DOM object to remove
		 * @param {GENTICS.Utils.RangeObject} range range which eventually be modified
		 * @param {boolean} preserveContent true if the contents of the removed DOM object shall be preserved, false if not (default: false)
		 * @return true if the DOM object could be removed, false if not
		 * @hide
		 */
		removeFromDOM: function (object, range, preserveContent) {
			if (preserveContent) {
				// check whether the range will need modification
				var indexInParent = this.getIndexInParent(object),
					numChildren = jQuery(object).contents().length,
					parent = object.parentNode;

				if (range.startContainer == parent && range.startOffset > indexInParent) {
					range.startOffset += numChildren - 1;
				} else if (range.startContainer == object) {
					range.startContainer = parent;
					range.startOffset = indexInParent + range.startOffset;
				}

				if (range.endContainer == parent && range.endOffset > indexInParent) {
					range.endOffset += numChildren - 1;
				} else if (range.endContainer == object) {
					range.endContainer = parent;
					range.endOffset = indexInParent + range.endOffset;
				}

				// we simply unwrap the children of the object
				jQuery(object).contents().unwrap();

				// optionally do cleanup
				this.doCleanup({
					'merge': true
				}, range, parent);
			}
		},

		/**
		 * Remove the content defined by the given range from the DOM. Update the given
		 * range object to be a collapsed selection at the place of the previous
		 * selection.
		 * @param rangeObject range object
		 * @return true if the range could be removed, false if not
		 */
		removeRange: function (rangeObject) {
			if (!rangeObject) {
				// no range given
				return false;
			}
			if (rangeObject.isCollapsed()) {
				// the range is collapsed, nothing to delete
				return false;
			}

			// split partially contained text nodes at the start and end of the range
			if (rangeObject.startContainer.nodeType == 3
				    && rangeObject.startOffset > 0
				    && rangeObject.startOffset < rangeObject.startContainer.data.length) {
				this.split(rangeObject, jQuery(rangeObject.startContainer).parent(), false);
			}
			if (rangeObject.endContainer.nodeType == 3 && rangeObject.endOffset > 0 && rangeObject.endOffset < rangeObject.endContainer.data.length) {
				this.split(rangeObject, jQuery(rangeObject.endContainer).parent(), true);
			}

			// construct the range tree
			var rangeTree = rangeObject.getRangeTree();

			// collapse the range
			rangeObject.endContainer = rangeObject.startContainer;
			rangeObject.endOffset = rangeObject.startOffset;

			// remove the markup from the range tree
			this.recursiveRemoveRange(rangeTree, rangeObject);

			// do some cleanup
			this.doCleanup({
				'merge': true
			}, rangeObject);
			//		this.doCleanup({'merge' : true, 'removeempty' : true}, rangeObject);

			// clear the caches of the range object
			rangeObject.clearCaches();
		},

		recursiveRemoveRange: function (rangeTree, rangeObject) {
			// iterate over the rangetree objects of this level
			var i;
			for (i = 0; i < rangeTree.length; ++i) {
				// check for nodes fully in the range
				if (rangeTree[i].type == 'full') {
					// if the domobj is the startcontainer, or the startcontainer is inside the domobj, we need to update the rangeObject
					if (jQuery(rangeObject.startContainer).parents().andSelf().filter(rangeTree[i].domobj).length > 0) {
						rangeObject.startContainer = rangeObject.endContainer = rangeTree[i].domobj.parentNode;
						rangeObject.startOffset = rangeObject.endOffset = this.getIndexInParent(rangeTree[i].domobj);
					}

					// remove the object from the DOM
					jQuery(rangeTree[i].domobj).remove();
				} else if (rangeTree[i].type == 'partial' && rangeTree[i].children) {
					// node partially selected and has children, so do recursion
					this.recursiveRemoveRange(rangeTree[i].children, rangeObject);
				}
			}
		},

		/**
		 * Extend the given range to have start and end at the nearest word boundaries to the left (start) and right (end)
		 * @param {GENTICS.Utils.RangeObject} range range to be extended
		 * @param {boolean} fromBoundaries true if extending will also be done, if one or both ends of the range already are at a word boundary, false if not, default: false
		 * @method
		 */
		extendToWord: function (range, fromBoundaries) {
			// search the word boundaries to the left and right
			var leftBoundary = this.searchWordBoundary(range.startContainer, range.startOffset, true),
				rightBoundary = this.searchWordBoundary(range.endContainer, range.endOffset, false);

			// check whether we must not extend the range from word boundaries
			if (!fromBoundaries) {
				// we only extend the range if both ends would be different
				if (range.startContainer == leftBoundary.container && range.startOffset == leftBoundary.offset) {
					return;
				}
				if (range.endContainer == rightBoundary.container && range.endOffset == rightBoundary.offset) {
					return;
				}
			}

			// set the new boundaries
			range.startContainer = leftBoundary.container;
			range.startOffset = leftBoundary.offset;
			range.endContainer = rightBoundary.container;
			range.endOffset = rightBoundary.offset;

			// correct the range
			range.correctRange();

			// clear caches
			range.clearCaches();
		},

		/**
		 * Helper method to check whether the given DOM object is a word boundary.
		 * @param {DOMObject} object DOM object in question
		 * @return {boolean} true when the DOM object is a word boundary, false if not
		 * @hide
		 */
		isWordBoundaryElement: function (object) {
			if (!object || !object.nodeName) {
				return false;
			}
			return jQuery.inArray(object.nodeName.toLowerCase(), this.nonWordBoundaryTags) == -1;
		},

		/**
		 * Search for the next word boundary, starting at the given position
		 * @param {DOMObject} container container of the start position
		 * @param {Integer} offset offset of the start position
		 * @param {boolean} searchleft true for searching to the left, false for searching to the right (default: true)
		 * @return {object} object with properties 'container' and 'offset' marking the found word boundary
		 * @method
		 */
		searchWordBoundary: function (container, offset, searchleft) {
			if (typeof searchleft === 'undefined') {
				searchleft = true;
			}
			var boundaryFound = false,
				wordBoundaryPos,
			    tempWordBoundaryPos,
			    textNode;
			while (!boundaryFound) {
				// check the node type
				if (container.nodeType === 3) {
					// we are currently in a text node

					// find the nearest word boundary character
					if (!searchleft) {
						// search right
						wordBoundaryPos = container.data.substring(offset).search(this.nonWordRegex);
						if (wordBoundaryPos != -1) {
							// found a word boundary
							offset = offset + wordBoundaryPos;
							boundaryFound = true;
						} else {
							// found no word boundary, so we set the position after the container
							offset = this.getIndexInParent(container) + 1;
							container = container.parentNode;
						}
					} else {
						// search left
						wordBoundaryPos = container.data.substring(0, offset).search(this.nonWordRegex);
						tempWordBoundaryPos = wordBoundaryPos;
						while (tempWordBoundaryPos != -1) {
							wordBoundaryPos = tempWordBoundaryPos;
							tempWordBoundaryPos = container.data.substring(wordBoundaryPos + 1, offset).search(this.nonWordRegex);
							if (tempWordBoundaryPos != -1) {
								tempWordBoundaryPos = tempWordBoundaryPos + wordBoundaryPos + 1;
							}
						}

						if (wordBoundaryPos != -1) {
							// found a word boundary
							offset = wordBoundaryPos + 1;
							boundaryFound = true;
						} else {
							// found no word boundary, so we set the position before the container
							offset = this.getIndexInParent(container);
							container = container.parentNode;
						}
					}
				} else if (container.nodeType === 1) {
					// we are currently in an element node (between nodes)

					if (!searchleft) {
						// check whether there is an element to the right
						if (offset < container.childNodes.length) {
							// there is an element to the right, check whether it is a word boundary element
							if (this.isWordBoundaryElement(container.childNodes[offset])) {
								// we are done
								boundaryFound = true;
							} else {
								// element to the right is no word boundary, so enter it
								container = container.childNodes[offset];
								offset = 0;
							}
						} else {
							// no element to the right, check whether the element itself is a boundary element
							if (this.isWordBoundaryElement(container)) {
								// we are done
								boundaryFound = true;
							} else {
								// element itself is no boundary element, so go to parent
								offset = this.getIndexInParent(container) + 1;
								container = container.parentNode;
							}
						}
					} else {
						// check whether there is an element to the left
						if (offset > 0) {
							// there is an element to the left, check whether it is a word boundary element
							if (this.isWordBoundaryElement(container.childNodes[offset - 1])) {
								// we are done
								boundaryFound = true;
							} else {
								// element to the left is no word boundary, so enter it
								container = container.childNodes[offset - 1];
								offset = container.nodeType === 3 ? container.data.length : container.childNodes.length;
							}
						} else {
							// no element to the left, check whether the element itself is a boundary element
							if (this.isWordBoundaryElement(container)) {
								// we are done
								boundaryFound = true;
							} else {
								// element itself is no boundary element, so go to parent
								offset = this.getIndexInParent(container);
								container = container.parentNode;
							}
						}
					}
				}
			}

			if (container.nodeType !== 3) {
				textNode = this.searchAdjacentTextNode(container, offset, !searchleft);
				if (textNode) {
					container = textNode;
					offset = searchleft ? 0 : container.data.length;
				}
			}

			return {
				'container': container,
				'offset': offset
			};
		},

		/**
		 * Check whether the given dom object is empty
		 * @param {DOMObject} domObject object to check
		 * @return {boolean} true when the object is empty, false if not
		 * @method
		 */
		isEmpty: function (domObject) {
			// a non dom object is considered empty
			if (!domObject) {
				return true;
			}

			// some tags are considered to be non-empty
			if (jQuery.inArray(domObject.nodeName.toLowerCase(), this.nonEmptyTags) != -1) {
				return false;
			}

			// text nodes are not empty, if they contain non-whitespace characters
			if (domObject.nodeType === 3) {
				return domObject.data.search(/\S/) == -1;
			}

			// all other nodes are not empty if they contain at least one child which is not empty
			var i, childNodes;
			for (i = 0, childNodes = domObject.childNodes.length; i < childNodes; ++i) {
				if (!this.isEmpty(domObject.childNodes[i])) {
					return false;
				}
			}

			// found no contents, so the element is empty
			return true;
		},

		/**
		 * Set the cursor (collapsed selection) right after the given DOM object
		 * @param domObject DOM object
		 * @method
		 */
		setCursorAfter: function (domObject) {
			var newRange = new GENTICS.Utils.RangeObject(),
				index = this.getIndexInParent(domObject),
				targetNode,
				offset;

			// selection cannot be set between to TEXT_NODEs
			// if domOject is a Text node set selection at last position in that node
			if (domObject.nodeType == 3) {
				targetNode = domObject;
				offset = targetNode.nodeValue.length;

				// if domOject is a Text node set selection at last position in that node
			} else if (domObject.nextSibling && domObject.nextSibling.nodeType == 3) {
				targetNode = domObject.nextSibling;
				offset = 0;
			} else {
				targetNode = domObject.parentNode;
				offset = this.getIndexInParent(domObject) + 1;
			}

			newRange.startContainer = newRange.endContainer = targetNode;
			newRange.startOffset = newRange.endOffset = offset;

			// select the range
			newRange.select();

			return newRange;
		},

		/**
		 * Select a DOM node
		 * will create a new range which spans the provided dom node and selects it afterwards
		 * @param domObject DOM object
		 * @method
		 */
		selectDomNode: function (domObject) {
			var newRange = new GENTICS.Utils.RangeObject();
			newRange.startContainer = newRange.endContainer = domObject.parentNode;
			newRange.startOffset = this.getIndexInParent(domObject);
			newRange.endOffset = newRange.startOffset + 1;
			newRange.select();
		},

		/**
		 * Set the cursor (collapsed selection) at the start into the given DOM object
		 * @param domObject DOM object
		 * @method
		 */
		setCursorInto: function (domObject) {
			// set a new range into the given dom object
			var newRange = new GENTICS.Utils.RangeObject();
			newRange.startContainer = newRange.endContainer = domObject;
			newRange.startOffset = newRange.endOffset = 0;

			// select the range
			newRange.select();
		},


		/**
		 * "An editing host is a node that is either an Element with a contenteditable
		 * attribute set to the true state, or the Element child of a Document whose
		 * designMode is enabled."
		 * @param domObject DOM object
		 * @method
		 */
		isEditingHost: function (node) {
			return node
				&& node.nodeType == 1 //ELEMENT_NODE
				&& (node.contentEditable == "true" || (node.parentNode && node.parentNode.nodeType == 9 //DOCUEMENT_NODE
													   && node.parentNode.designMode == "on"));
		},

		/**
		 * "Something is editable if it is a node which is not an editing host, does
		 * not have a contenteditable attribute set to the false state, and whose
		 * parent is an editing host or editable."
		 * @param domObject DOM object
		 * @method
		 */
		isEditable: function (node) {
			// This is slightly a lie, because we're excluding non-HTML elements with
			// contentEditable attributes.
			return node
				&& !this.isEditingHost(node)
				&& (node.nodeType != 1 || node.contentEditable != "false") // ELEMENT_NODE
				&& (this.isEditingHost(node.parentNode) || this.isEditable(node.parentNode));
		},

		/**
		 * "The editing host of node is null if node is neither editable nor an editing
		 * host; node itself, if node is an editing host; or the nearest ancestor of
		 * node that is an editing host, if node is editable."
		 * @param domObject DOM object
		 * @method
		 */
		getEditingHostOf: function (node) {
			if (this.isEditingHost(node)) {
				return node;
			}
			if (this.isEditable(node)) {
				var ancestor = node.parentNode;
				while (!this.isEditingHost(ancestor)) {
					ancestor = ancestor.parentNode;
				}
				return ancestor;
			}
			return null;
		},

		/**
		 * 
		 * "Two nodes are in the same editing host if the editing host of the first is
		 * non-null and the same as the editing host of the second."
		 * @param node1 DOM object
		 * @param node2 DOM object
		 * @method
		 */
		inSameEditingHost: function (node1, node2) {
			return this.getEditingHostOf(node1) && this.getEditingHostOf(node1) == this.getEditingHostOf(node2);
		},

		// "A block node is either an Element whose "display" property does not have
		// resolved value "inline" or "inline-block" or "inline-table" or "none", or a
		// Document, or a DocumentFragment."
		isBlockNode: function (node) {
			return node && ((node.nodeType == $_.Node.ELEMENT_NODE && $_(["inline", "inline-block", "inline-table", "none"]).indexOf($_.getComputedStyle(node).display) == -1) || node.nodeType == $_.Node.DOCUMENT_NODE || node.nodeType == $_.Node.DOCUMENT_FRAGMENT_NODE);
		},

		/**
		 * Get the first visible child of the given node.
		 * @param node node
		 * @param includeNode when set to true, the node itself may be returned, otherwise only children are allowed
		 * @return first visible child or null if none found
		 */
		getFirstVisibleChild: function (node, includeNode) {
			// no node -> no child
			if (!node) {
				return null;
			}

			// check whether the node itself is visible
			if ((node.nodeType == $_.Node.TEXT_NODE && this.isEmpty(node)) || (node.nodeType == $_.Node.ELEMENT_NODE && node.offsetHeight == 0 && jQuery.inArray(node.nodeName.toLowerCase(), this.nonEmptyTags) === -1)) {
				return null;
			}

			// if the node is a text node, or does not have children, or is not editable, it is the first visible child
			if (node.nodeType == $_.Node.TEXT_NODE || (node.nodeType == $_.Node.ELEMENT_NODE && node.childNodes.length == 0) || !jQuery(node).contentEditable()) {
				return includeNode ? node : null;
			}

			// otherwise traverse through the children
			var i;
			for (i = 0; i < node.childNodes.length; ++i) {
				var visibleChild = this.getFirstVisibleChild(node.childNodes[i], true);
				if (visibleChild != null) {
					return visibleChild;
				}
			}

			return null;
		},

		/**
		 * Get the last visible child of the given node.
		 * @param node node
		 * @param includeNode when set to true, the node itself may be returned, otherwise only children are allowed
		 * @return last visible child or null if none found
		 */
		getLastVisibleChild: function (node, includeNode) {
			// no node -> no child
			if (!node) {
				return null;
			}

			// check whether the node itself is visible
			if ((node.nodeType == $_.Node.TEXT_NODE && this.isEmpty(node)) || (node.nodeType == $_.Node.ELEMENT_NODE && node.offsetHeight == 0 && jQuery.inArray(node.nodeName.toLowerCase(), this.nonEmptyTags) === -1)) {
				return null;
			}

			// if the node is a text node, or does not have children, or is not editable, it is the first visible child
			if (node.nodeType == $_.Node.TEXT_NODE || (node.nodeType == $_.Node.ELEMENT_NODE && node.childNodes.length == 0) || !jQuery(node).contentEditable()) {
				return includeNode ? node : null;
			}

			// otherwise traverse through the children
			var i;
			for (i = node.childNodes.length - 1; i >= 0; --i) {
				var visibleChild = this.getLastVisibleChild(node.childNodes[i], true);
				if (visibleChild != null) {
					return visibleChild;
				}
			}

			return null;
		}
	});


	/**
	 * Create the singleton object
	 * @hide
	 */
	GENTICS.Utils.Dom = new Dom();

	return GENTICS.Utils.Dom;

});

/* pluginmanager.js is part of Aloha Editor project http://aloha-editor.org
 *
 * Aloha Editor is a WYSIWYG HTML5 inline editing library and editor.
 * Copyright (c) 2010-2012 Gentics Software GmbH, Vienna, Austria.
 * Contributors http://aloha-editor.org/contribution.php
 *
 * Aloha Editor is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or any later version.
 *
 * Aloha Editor is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * As an additional permission to the GNU GPL version 2, you may distribute
 * non-source (e.g., minimized or compacted) forms of the Aloha-Editor
 * source code without the copy of the GNU GPL normally required,
 * provided you include this license notice and a URL through which
 * recipients can access the Corresponding Source.
 */
// Do not add dependencies that require depend on aloha/core
define('aloha/pluginmanager',[
	'jquery',
	'util/class'
], function (
	$,
	Class
) {
	

	var Aloha = window.Aloha;

	/**
	 * For each plugin setting, assigns it into its corresponding plugin.
	 *
	 * @param {Array.<Plugin>} plugins
	 * @param {object<string, object>} settings
	 */
	function mapSettingsIntoPlugins(plugins, settings) {
		var plugin;
		for (plugin in settings) {
			if (settings.hasOwnProperty(plugin) && plugins[plugin]) {
				plugins[plugin].settings = settings[plugin] || {};
			}
		}
	}

	/**
	 * Retrieves a set of plugins or the given `names' list, from among those
	 * specified in `plugins'.
	 *
	 * @param {object<string, object>} plugins
	 * @param {Array.<string>} names List of plugins names.
	 * @return {Array.<Plugins>} List of available plugins.
	 */
	function getPlugins(plugins, names) {
		var available = [];
		var plugin;
		var i;
		for (i = 0; i < names.length; i++) {
			plugin = plugins[names[i]];
			if (plugin) {
				available.push(plugin);
			}
		}
		return available;
	}

	/**
	 * Initializes the plugins in the given list.
	 *
	 * @param {Array.<Plugins>} plugins Plugins to initialize.
	 * @param {function} callback Function to invoke once all plugins have been
	 *                            successfully initialized.
	 */
	function initializePlugins(plugins, callback) {
		var numToEnable = plugins.length;
		var onInit = function () {
			if (0 === --numToEnable && callback) {
				callback();
			}
		};
		var i;
		var ret;
		var plugin;
		for (i = 0; i < plugins.length; i++) {
			plugin = plugins[i];
			plugin.settings = plugin.settings || {};
			if (typeof plugin.settings.enabled === 'undefined') {
				plugin.settings.enabled = true;
			}
			if (plugin.settings.enabled && plugin.checkDependencies()) {
				ret = plugin.init();
				if (ret && typeof ret.done === 'function') {
					ret.done(onInit);
				} else {
					onInit();
				}
			}
		}
	}

	/**
	 * The Plugin Manager controls the lifecycle of all Aloha Plugins.
	 *
	 * @namespace Aloha
	 * @class PluginManager
	 * @singleton
	 */
	return new (Class.extend({

		plugins: {},

		/**
		 * Initialize all registered plugins.
		 *
		 * @param {function} next Callback to invoke after plugins have
		 *                        succefully initialized.
		 * @param {Array.<string>} enabled A list of plugin names which are to
		 *                                 be enable.
		 */
		init: function (next, enabled) {
			var manager = this;
			var plugins = manager.plugins;

			mapSettingsIntoPlugins(plugins,
					Aloha && Aloha.settings && Aloha.settings.plugins);

			// Because all plugins are enabled by default if specific plugins
			// are not specified.
			var plugin;
			if (!plugins || 0 === enabled.length) {
				enabled = [];
				for (plugin in plugins) {
					if (plugins.hasOwnProperty(plugin)) {
						enabled.push(plugin);
					}
				}
			}

			initializePlugins(getPlugins(plugins, enabled), next);
		},

		/**
		 * Register a plugin
		 * @param {Plugin} plugin plugin to register
		 */
		register: function (plugin) {

			if (!plugin.name) {
				throw new Error('Plugin does not have an name.');
			}

			if (this.plugins[plugin.name]) {
				throw new Error('Already registered the plugin "' + plugin.name + '"!');
			}

			this.plugins[plugin.name] = plugin;
		},

		/**
		 * Pass the given jQuery object, which represents an editable to all plugins, so that they can make the content clean (prepare for saving)
		 * @param obj jQuery object representing an editable
		 * @return void
		 * @hide
		 */
		makeClean: function (obj) {
			var i, plugin;
			// iterate through all registered plugins
			for (plugin in this.plugins) {
				if (this.plugins.hasOwnProperty(plugin)) {
					if (Aloha.Log.isDebugEnabled()) {
						Aloha.Log.debug(this, 'Passing contents of HTML Element with id { ' + obj.attr('id') + ' } for cleaning to plugin { ' + plugin + ' }');
					}
					this.plugins[plugin].makeClean(obj);
				}
			}
		},

		/**
		 * Expose a nice name for the Plugin Manager
		 * @hide
		 */
		toString: function () {
			return 'pluginmanager';
		}

	}))();
});

/*core.js is part of Aloha Editor project http://aloha-editor.org
 *
 * Aloha Editor is a WYSIWYG HTML5 inline editing library and editor.
 * Copyright (c) 2010-2012 Gentics Software GmbH, Vienna, Austria.
 * Contributors http://aloha-editor.org/contribution.php
 *
 * Aloha Editor is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or any later version.
 *
 * Aloha Editor is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * As an additional permission to the GNU GPL version 2, you may distribute
 * non-source (e.g., minimized or compacted) forms of the Aloha-Editor
 * source code without the copy of the GNU GPL normally required,
 * provided you include this license notice and a URL through which
 * recipients can access the Corresponding Source.
 */
define('aloha/core',[
	'jquery',
	'aloha/pluginmanager'
], function (
	$,
	PluginManager
) {
	

	var Aloha = window.Aloha;

	/**
	 * Checks whether the current user agent is supported.
	 *
	 * @return {boolean} True if Aloha supports the current browser.
	 */
	function isBrowserSupported() {
		var browser = $.browser;
		var version = browser.version;
		return !(
			// Chrome/Safari 4
			(browser.webkit && parseFloat(version) < 532.5) ||
			// FF 3.5
			(browser.mozilla && parseFloat(version) < 1.9) ||
			// IE 7
			(browser.msie && version < 7) ||
			// Right now Opera needs some work
			(browser.opera && version < 11)
		);
	}

	/**
	 * Checks whether the given jQuery event originates from an Aloha dialog
	 * element.
	 *
	 * This is used to facilitate a hackish way of preventing blurring
	 * editables when interacting with Aloha UI modals.
	 *
	 * @param {jQuery<Event>} $event
	 * @return {boolean} True if $event is initiated from within an Aloha
	 *                   dialog element.
	 */
	function originatesFromDialog($event) {
		var $target = $($event.target);
		return $target.is('.aloha-dialog') ||
		       0 < $target.closest('.aloha-dialog').length;
	}

	/**
	 * Registers events on the documents to cause editables to be blurred when
	 * clicking outside of editables.
	 *
	 * Hack: Except when the click originates from a modal dialog.
	 */
	function registerEvents() {
		$('html').mousedown(function ($event) {
			if (Aloha.activeEditable && !Aloha.eventHandled
					&& !originatesFromDialog($event)) {
				Aloha.deactivateEditable();
			}
		}).mouseup(function () {
			Aloha.eventHandled = false;
		});
	}

	/**
	 * Initialize Aloha.
	 *
	 * @param {function} next Function to call after initialization.
	 */
	function initAloha(event, next) {
		if (!isBrowserSupported()) {
			var console = window.console;
			if (console) {
				var fn = console.error ? 'error' : console.log ? 'log' : null;
				if (fn) {
					console[fn]('This browser is not supported');
				}
			}
			return;
		}

		// Because different css is to be applied based on what the user-agent
		// supports.  For example: outlines do not render in IE7.
		if ($.browser.webkit) {
			$('html').addClass('aloha-webkit');
		} else if ($.browser.opera) {
			$('html').addClass('aloha-opera');
		} else if ($.browser.msie) {
			$('html').addClass('aloha-ie' + parseInt($.browser.version, 10));
		} else if ($.browser.mozilla) {
			$('html').addClass('aloha-mozilla');
		}

		if (navigator.appVersion.indexOf('Win') !== -1) {
			Aloha.OSName = 'Win';
		} else if (navigator.appVersion.indexOf('Mac') !== -1) {
			Aloha.OSName = 'Mac';
		} else if (navigator.appVersion.indexOf('X11') !== -1) {
			Aloha.OSName = 'Unix';
		} else if (navigator.appVersion.indexOf('Linux') !== -1) {
			Aloha.OSName = 'Linux';
		}

		registerEvents();
		Aloha.settings.base = Aloha.getAlohaUrl();
		Aloha.Log.init();

		// Initialize error handler for general javascript errors.
		if (Aloha.settings.errorhandling) {
			window.onerror = function (msg, url, line) {
				Aloha.Log.error(Aloha, 'Error message: ' + msg + '\nURL: ' +
				                       url + '\nLine Number: ' + line);
				return true;
			};
		}

		event();
		next();
	}

	/**
	 * Initialize managers
	 */
	function initRepositoryManager(event, next) {
		Aloha.RepositoryManager.init();
		event();
		next();
	}

	/**
	 * Initialize Aloha plugins.
	 *
	 * @param {function} onPluginsInitialized Callback that will be invoked
	 *                                        after all plugins have been
	 *                                        initialized.  Whereas plugins are
	 *                                        loaded synchronously, plugins may
	 *                                        initialize asynchronously.
	 */
	function initPluginManager(event, next) {
		// Because if there are no loadedPlugins specified, then the default is
		// to initialized all available plugins.
		if (0 === Aloha.loadedPlugins.length) {
			var plugins = PluginManager.plugins;
			var plugin;
			for (plugin in plugins) {
				if (plugins.hasOwnProperty(plugin)) {
					Aloha.loadedPlugins.push(plugin);
				}
			}
		}

		var fired = false;

		PluginManager.init(function () {
			if (!fired) {
				event();
				fired = true;
			}
			next();
		}, Aloha.loadedPlugins);

		if (!fired) {
			event();
			fired = true;
		}
	}

	/**
	 * Begin initialize editables.
	 */
	function initEditables(event, next) {
		var i;
		for (i = 0; i < Aloha.editables.length; i++) {
			if (!Aloha.editables[i].ready) {
				Aloha.editables[i].init();
			}
		}
		event();
		next();
	}

	/**
	 * Initialization phases.
	 *
	 * These stages denote the initialization states which Aloha will go
	 * through from loading to ready.
	 *
	 * Each phase object contains the following properties:
	 *        fn : The process that is to be invoked during this phase.
	 *             Will take two functions: event() and next().
	 *     event : The event name, which if provided, will be fired after the
	 *             phase has been started (optional).
	 *  deferred : A $.Deferred() object to hold event handlers until that
	 *             initialization phase has been done (optional).
	 *
	 * @type {Array.<phase>}
	 */
	var phases = [
		// Phase 0: Waiting for initialization to begin.  This is the state
		//          before at load-time.
		{
			fn: null,
			event: null,
			deferred: null
		},

		// Phase 1: DOM is ready; performing compatibility checks, registering
		//          basic events, and initialize logging.
		{
			fn: initAloha,
			event: null,
			deferred: null
		},

		// Phase 2: Initial checks have passed; Initialize repository manger.
		{
			fn: initRepositoryManager,
			event: null,
			deferred: null
		},

		// Phase 3: Repository manager is ready for use; commence
		//          initialization of all configured (or default) plugins.
		{
			fn: initPluginManager,
			event: 'aloha-plugins-loaded',
			deferred: null
		},

		// Phase 4: Plugins have all begun their initialization process, but it
		//          is not necessary that their have completed.  Start editable
		//          initializing editable, along with their scaffolding UI.
		//          Editables will not be fully initialized however, until
		//          plugins have fully finished initialization.
		{
			fn: initEditables,
			event: null,
			deferred: null
		},

		// Phase 5: The "ready" state.  Notify the system that Aloha is fully
		//          loaded, and ready for use.
		{
			fn: null,
			event: 'aloha-ready',
			deferred: null
		}
	];


	/**
	 * Base Aloha Object
	 * @namespace Aloha
	 * @class Aloha The Aloha base object, which contains all the core functionality
	 * @singleton
	 */
	$.extend(true, Aloha, {

		/**
		 * The Aloha Editor Version we are using
		 * It should be set by us and updated for the particular branch
		 * @property
		 */
		version: '0.22.6',

		/**
		 * Array of editables that are managed by Aloha
		 * @property
		 * @type Array
		 */
		editables: [],

		/**
		 * The currently active editable is referenced here
		 * @property
		 * @type Aloha.Editable
		 */
		activeEditable: null,

		/**
		 * settings object, which will contain all Aloha settings
		 * @cfg {Object} object Aloha's settings
		 */
		settings: {},

		/**
		 * defaults object, which will contain all Aloha defaults
		 * @cfg {Object} object Aloha's settings
		 */
		defaults: {},

		/**
		 * Namespace for ui components
		 */
		ui: {},

		/**
		 * This represents the name of the users OS. Could be:
		 * 'Mac', 'Linux', 'Win', 'Unix', 'Unknown'
		 * @property
		 * @type string
		 */
		OSName: 'Unknown',

		/**
		 * A list of loaded plugin names, available after the STAGES.PLUGINS
		 * initialization phase.
		 *
		 * @type {Array.<string>}
		 * @internal
		 */
		loadedPlugins: [],

		/**
		 * Maps names of plugins (link) to the base URL (../plugins/common/link).
		 */
		_pluginBaseUrlByName: {},

		/**
		 * Start the initialization process.
		 */
		init: function () {
			Aloha.initialize(phases);
		},

		/**
		 * Returns list of loaded plugins (without Bundle name)
		 *
		 * @return array
		 */
		getLoadedPlugins: function () {
			return this.loadedPlugins;
		},

		/**
		 * Returns true if a certain plugin is loaded, false otherwise.
		 *
		 * @param {string} plugin Name of plugin
		 * @return {boolean} True if plugin with given name is load.
		 */
		isPluginLoaded: function (name) {
			var loaded = false;
			$.each(this.loadedPlugins, function (i, plugin) {
				if (name === plugin.toString()) {
					loaded = true;
					return false;
				}
			});
			return loaded;
		},

		/**
		 * Activates editable and deactivates all other Editables.
		 *
		 * @param {Editable} editable the Editable to be activated
		 */
		activateEditable: function (editable) {
			// Because editables may be removed on blur, Aloha.editables.length
			// is not cached.
			var editables = Aloha.editables;
			var i;
			for (i = 0; i < editables.length; i++) {
				if (editables[i] !== editable && editables[i].isActive) {
					editables[i].blur();
				}
			}
			Aloha.activeEditable = editable;
		},

		/**
		 * Returns the current Editable
		 * @return {Editable} returns the active Editable
		 */
		getActiveEditable: function () {
			return Aloha.activeEditable;
		},

		/**
		 * Deactivates the active Editable.
		 *
		 * TODO: Would be better named "deactivateActiveEditable".
		 */
		deactivateEditable: function () {
			if (Aloha.activeEditable) {
				Aloha.activeEditable.blur();
				Aloha.activeEditable = null;
			}
		},

		/**
		 * Gets an editable by an ID or null if no Editable with that ID
		 * registered.
		 *
		 * @param {string} id The element id to look for.
		 * @return {Aloha.Editable|null} An editable, or null if none if found
		 *                               for the given id.
		 */
		getEditableById: function (id) {
			// Because if the element is a textarea, then it's necessary to
			// route to the editable div.
			var $editable = $('#' + id);
			if ($editable.length
					&& 'textarea' === $editable[0].nodeName.toLowerCase()) {
				id = id + '-aloha';
			}
			var i;
			for (i = 0; i < Aloha.editables.length; i++) {
				if (Aloha.editables[i].getId() === id) {
					return Aloha.editables[i];
				}
			}
			return null;
		},

		/**
		 * Checks whether an object is a registered Aloha Editable.
		 * @param {jQuery} obj the jQuery object to be checked.
		 * @return {boolean}
		 */
		isEditable: function (obj) {
			var i, editablesLength;

			for (i = 0, editablesLength = Aloha.editables.length; i < editablesLength; i++) {
				if (Aloha.editables[i].originalObj.get(0) === obj) {
					return true;
				}
			}
			return false;
		},

		/**
		 * Get the nearest editable parent of the DOM element contained in the
		 * given jQuery object.
		 *
		 * @param {jQuery} $obj jQuery unit set containing DOM element.
		 * @return {Aloha.Editable} Editable, or null if none found.
		 */
		getEditableHost: function ($obj) {
			if (!$obj) {
				return null;
			}
			var $editable;
			$obj.parents().andSelf().each(function () {
				var i;
				for (i = 0; i < Aloha.editables.length; i++) {
					if (Aloha.editables[i].originalObj.get(0) === this) {
						$editable = Aloha.editables[i];
						return false;
					}
				}
			});
			return $editable;
		},

		/**
		 * Logs a message to the console.
		 *
		 * @param {string} level Level of the log
		 *                       ("error", "warn" or "info", "debug").
		 * @param {object} component Component that calls the log.
		 * @param {string} message Log message.
		 * @hide
		 */
		log: function (level, component, message) {
			if (typeof Aloha.Log !== 'undefined') {
				Aloha.Log.log(level, component, message);
			}
		},

		/**
		 * Register the given editable.
		 *
		 * @param {Editable} editable to register.
		 * @hide
		 */
		registerEditable: function (editable) {
			Aloha.editables.push(editable);
		},

		/**
		 * Unregister the given editable. It will be deactivated and removed
		 * from editables.
		 *
		 * @param {Editable} editable The editable to unregister.
		 * @hide
		 */
		unregisterEditable: function (editable) {
			var index = $.inArray(editable, Aloha.editables);
			if (index !== -1) {
				Aloha.editables.splice(index, 1);
			}
		},

		/**
		 * Check whether at least one editable was modified.
		 *
		 * @return {boolean} True when at least one editable was modified,
		 *                   false otherwise.
		 */
		isModified: function () {
			var i;
			for (i = 0; i < Aloha.editables.length; i++) {
				if (Aloha.editables[i].isModified
						&& Aloha.editables[i].isModified()) {
					return true;
				}
			}
			return false;
		},

		/**
		 * Determines the Aloha Url.
		 *
		 * @return {String} Aloha's baseUrl setting or "" if not set.
		 */
		getAlohaUrl: function (suffix) {
			return Aloha.settings.baseUrl || '';
		},

		/**
		 * Gets the plugin's url.
		 *
		 * @param {string} name The name with which the plugin was registered
		 *                      with.
		 * @return {string} The fully qualified url of this plugin.
		 */
		getPluginUrl: function (name) {
			if (name) {
				return null;
			}
			var url = Aloha.settings._pluginBaseUrlByName[name];
			if (url) {
				// Check if url is absolute and attach base url if it is not.
				if (!url.match("^(\/|http[s]?:).*")) {
					url = Aloha.getAlohaUrl() + '/' + url;
				}
			}
			return url;
		},

		/**
		 * Disable object resizing by executing command 'enableObjectResizing',
		 * if the browser supports this.
		 */
		disableObjectResizing: function () {
			try {
				// This will disable browsers image resizing facilities in
				// order disable resize handles.
				var supported;
				try {
					supported = document.queryCommandSupported('enableObjectResizing');
				} catch (e) {
					supported = false;
					Aloha.Log.log('enableObjectResizing is not supported.');
				}
				if (supported) {
					document.execCommand('enableObjectResizing', false, false);
					Aloha.Log.log('enableObjectResizing disabled.');
				}
			} catch (e2) {
				Aloha.Log.error(e2, 'Could not disable enableObjectResizing');
				// this is just for others, who will not support disabling enableObjectResizing
			}
		},

		/**
		 * Human-readable string representation of this.
		 *
		 * @hide
		 */
		toString: function () {
			return 'Aloha';
		}
	});

	return Aloha;
});

/* console.js is part of Aloha Editor project http://aloha-editor.org
 *
 * Aloha Editor is a WYSIWYG HTML5 inline editing library and editor. 
 * Copyright (c) 2010-2012 Gentics Software GmbH, Vienna, Austria.
 * Contributors http://aloha-editor.org/contribution.php 
 * 
 * Aloha Editor is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or any later version.
 *
 * Aloha Editor is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 * 
 * As an additional permission to the GNU GPL version 2, you may distribute
 * non-source (e.g., minimized or compacted) forms of the Aloha-Editor
 * source code without the copy of the GNU GPL normally required,
 * provided you include this license notice and a URL through which
 * recipients can access the Corresponding Source.
 */
define('aloha/console',[
	'aloha/core',
	'util/class',
	'jquery'
], function (
	Aloha,
	Class,
	jQuery
) {
	

	/**
	 * This is the aloha Log
	 * @namespace Aloha
	 * @class Log
	 * @singleton
	 */
	var AlohaConsole = Class.extend({
		/**
		 * Initialize the logging
		 * @hide
		 */
		init: function () {

			// initialize the logging settings (if not present)
			if (typeof Aloha.settings.logLevels === 'undefined' || !Aloha.settings.logLevels) {
				Aloha.settings.logLevels = {
					'error': true,
					'warn': true
				};
			}

			// initialize the logHistory settings (if not present)
			if (typeof Aloha.settings.logHistory === 'undefined' || !Aloha.settings.logHistory) {
				Aloha.settings.logHistory = {};
			}
			// set the default values for the loghistory
			if (!Aloha.settings.logHistory.maxEntries) {
				Aloha.settings.logHistory.maxEntries = 100;
			}
			if (!Aloha.settings.logHistory.highWaterMark) {
				Aloha.settings.logHistory.highWaterMark = 90;
			}
			if (!Aloha.settings.logHistory.levels) {
				Aloha.settings.logHistory.levels = {
					'error': true,
					'warn': true
				};
			}
			this.flushLogHistory();

			Aloha.trigger('aloha-logger-ready');
		},

		/**
		 * Log History as array of Message Objects. Every object has the properties
		 * 'level', 'component' and 'message'
		 * @property
		 * @type Array
		 * @hide
		 */
		logHistory: [],

		/**
		 * Flag, which is set as soon as the highWaterMark for the log history is reached.
		 * This flag is reset on every call of flushLogHistory()
		 * @hide
		 */
		highWaterMarkReached: false,

		/**
		 * Logs a message to the console
		 * @method
		 * @param {String} level Level of the log ('error', 'warn' or 'info', 'debug')
		 * @param {String} component Component that calls the log
		 * @param {String} message log message
		 */
		log: function (level, component, message) {


			// log ('Logging message');
			if (typeof component === 'undefined') {
				message = level;
			}
			if (typeof component !== 'string' && component && component.toString) {
				component = component.toString();
			}

			// log ('warn', 'Warning message');
			if (typeof message === 'undefined') {
				message = component;
				component = undefined;
			}

			if (typeof level === 'undefined' || !level) {
				level = 'log';
			}

			level = level.toLowerCase();

			if (typeof Aloha.settings.logLevels === "undefined") {
				return;
			}

			// now check whether the log level is activated
			if (!Aloha.settings.logLevels[level]) {
				return;
			}

			component = component || "Unkown Aloha Component";

			this.addToLogHistory({
				'level': level,
				'component': component,
				'message': message,
				'date': new Date()
			});

			var console = window.console;

			switch (level) {
			case 'error':
				if (window.console && console.error) {
					// FIXME:
					// Using console.error rather than throwing an error is very
					// problematic because we get not stack.
					// We ought to consider doing the following:
					// throw component + ': ' + message;
					if (!component && !message) {
						console.error("Error occured without message and component");
					} else {
						console.error(component + ': ' + message);
					}
				}
				break;
			case 'warn':
				if (window.console && console.warn) {
					console.warn(component + ': ' + message);
				}
				break;
			case 'info':
				if (window.console && console.info) {
					console.info(component + ': ' + message);
				}
				break;
			case 'debug':
				if (window.console && console.log) {
					console.log(component + ' [' + level + ']: ' + message);
				}
				break;
			default:
				if (window.console && console.log) {
					console.log(component + ' [' + level + ']: ' + message);
				}
				break;
			}
		},

		/**
		 * Log a message of log level 'error'
		 * @method
		 * @param {String} component Component that calls the log
		 * @param {String} message log message
		 */
		error: function (component, message) {
			this.log('error', component, message);
		},

		/**
		 * Log a message of log level 'warn'
		 * @method
		 * @param {String} component Component that calls the log
		 * @param {String} message log message
		 */
		warn: function (component, message) {
			this.log('warn', component, message);
		},

		/**
		 * Log a message of log level 'info'
		 * @method
		 * @param {String} component Component that calls the log
		 * @param {String} message log message
		 */
		info: function (component, message) {
			this.log('info', component, message);
		},

		/**
		 * Log a message of log level 'debug'
		 * @param {String} component Component that calls the log
		 * @param {String} message log message
		 */
		debug: function (component, message) {
			this.log('debug', component, message);
		},

		/**
		 * Methods to mark function as deprecated for developers.
		 * @param {String} component String that calls the log
		 * @param {String} message log message
		 */
		deprecated: function (component, message) {
			this.log('warn', component, message);
			// help the developer to locate the call.
			if (Aloha.settings.logLevels.deprecated) {
				throw new Error(message);
			}
		},

		/**
		 * Check whether the given log level is currently enabled
		 * @param {String} level
		 * @return true when log level is enabled, false if not
		 */
		isLogLevelEnabled: function (level) {
			return Aloha.settings && Aloha.settings.logLevels && Aloha.settings.logLevels[level];
		},

		/**
		 * Check whether error logging is enabled
		 * @return true if error logging is enabled, false if not
		 */
		isErrorEnabled: function () {
			return this.isLogLevelEnabled('error');
		},

		/**
		 * Check whether warn logging is enabled
		 * @return true if warn logging is enabled, false if not
		 */
		isWarnEnabled: function () {
			return this.isLogLevelEnabled('warn');
		},

		/**
		 * Check whether info logging is enabled
		 * @return true if info logging is enabled, false if not
		 */
		isInfoEnabled: function () {
			return this.isLogLevelEnabled('info');
		},

		/**
		 * Check whether debug logging is enabled
		 * @return true if debug logging is enabled, false if not
		 */
		isDebugEnabled: function () {
			return this.isLogLevelEnabled('debug');
		},

		/**
		 * Add the given entry to the log history. Check whether the highWaterMark has been reached, and fire an event if yes.
		 * @param {Object} entry entry to be added to the log history
		 * @hide
		 */
		addToLogHistory: function (entry) {

			if (!Aloha.settings.logHistory) {
				this.init();
			}

			// when maxEntries is set to something illegal, we do nothing (log history is disabled)
			// check whether the level is one we like to have logged
			if (Aloha.settings.logHistory.maxEntries <= 0 || !Aloha.settings.logHistory.levels[entry.level]) {

				return;
			}

			// first add the entry as last element to the history array
			this.logHistory.push(entry);

			// check whether the highWaterMark was reached, if so, fire an event
			if (!this.highWaterMarkReached) {

				if (this.logHistory.length >= Aloha.settings.logHistory.maxEntries * Aloha.settings.logHistory.highWaterMark / 100) {

					// fire the event
					Aloha.trigger('aloha-log-full');
					// set the flag (so we will not fire the event again until the logHistory is flushed)
					this.highWaterMarkReached = true;
				}
			}

			// check whether the log is full and eventually remove the oldest entries
			// @todo remove old entries when aloha-log-full event is triggered
			while (this.logHistory.length > Aloha.settings.logHistory.maxEntries) {
				this.logHistory.shift();
			}
		},

		/**
		 * Get the log history
		 * @return log history as array of objects
		 * @hide
		 */
		getLogHistory: function () {
			return this.logHistory;
		},

		/**
		 * Flush the log history. Remove all log entries and reset the flag for the highWaterMark
		 * @return void
		 * @hide
		 */
		flushLogHistory: function () {
			this.logHistory = [];
			this.highWaterMarkReached = false;
		}
	});

	/**
	 * Create the Log object
	 * @hide
	 */
	AlohaConsole = new AlohaConsole();

	// add to log namespace for compatiblility.
	Aloha.Log = Aloha.Console = AlohaConsole;
	return AlohaConsole;
});

/* range.js is part of Aloha Editor project http://aloha-editor.org
 *
 * Aloha Editor is a WYSIWYG HTML5 inline editing library and editor. 
 * Copyright (c) 2010-2012 Gentics Software GmbH, Vienna, Austria.
 * Contributors http://aloha-editor.org/contribution.php 
 * 
 * Aloha Editor is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or any later version.
 *
 * Aloha Editor is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 * 
 * As an additional permission to the GNU GPL version 2, you may distribute
 * non-source (e.g., minimized or compacted) forms of the Aloha-Editor
 * source code without the copy of the GNU GPL normally required,
 * provided you include this license notice and a URL through which
 * recipients can access the Corresponding Source.
 */
// Ensure GENTICS Namespace
window.GENTICS = window.GENTICS || {};
window.GENTICS.Utils = window.GENTICS.Utils || {};

define('util/range',[
	'jquery',
	'util/dom',
	'util/class',
	'aloha/console',
	'aloha/rangy-core'
], function (jQuery, Dom, Class, console, rangy) {
	

	var GENTICS = window.GENTICS;
	var Aloha = window.Aloha;

	function selfAndParentsUntil(container, limit) {
		var parents = [],
			cur;
		if (1 === container.nodeType) {
			cur = container;
		} else {
			cur = container.parentNode;
		}
		for (;;) {
			if (!cur || cur === limit || 9 === cur.nodeType) {
				break;
			}
			if (1 === cur.nodeType) {
				parents.push(cur);
			}
			cur = cur.parentNode;
		}
		return parents;
	}

	/**
	 * @namespace GENTICS.Utils
	 * @class RangeObject
	 * Represents a selection range in the browser that
	 * has some advanced features like selecting the range.
	 * @param {object} param if boolean true is passed, the range will be deducted from the current browser selection.
	 * If another rangeObject is passed, it will be cloned.
	 * If nothing is passed, the rangeObject will be empty.
	 * @constructor
	 */
	GENTICS.Utils.RangeObject = Class.extend({
		_constructor: function (param) {
			// Take the values from the passed object
			if (typeof param === 'object') {
				if (typeof param.startContainer !== 'undefined') {
					this.startContainer = param.startContainer;
				}
				if (typeof param.startOffset !== 'undefined') {
					this.startOffset = param.startOffset;
				}
				if (typeof param.endContainer !== 'undefined') {
					this.endContainer = param.endContainer;
				}
				if (typeof param.endOffset !== 'undefined') {
					this.endOffset = param.endOffset;
				}
			} else if (param === true) {
				this.initializeFromUserSelection();
			}
		},

		/**
		 * DOM object of the start container of the selection.
		 * This is always has to be a DOM text node.
		 * @property startContainer
		 * @type {DOMObject}
		 */
		startContainer: undefined,

		/**
		 * Offset of the selection in the start container
		 * @property startOffset
		 * @type {Integer}
		 */
		startOffset: undefined,

		/**
		 * DOM object of the end container of the selection.
		 * This is always has to be a DOM text node.
		 * @property endContainer
		 * @type {DOMObject}
		 */
		endContainer: undefined,

		/**
		 * Offset of the selection in the end container
		 * @property endOffset
		 * @type {Integer}
		 */
		endOffset: undefined,

		/**
		 * Delete all contents selected by the current range
		 * @param rangeTree a GENTICS.Utils.RangeTree object may be provided to start from. This parameter is optional
		 */
		deleteContents: function () {

			Dom.removeRange(this);

		},

		/**
		 * Output some log
		 * TODO: move this to Aloha.Log
		 * @param message log message to output
		 * @return void
		 * @deprecated
		 * @hide
		 */
		log: function (message) {
			console.deprecated('Utils.RangeObject', 'log() is deprecated. use ' + 'console.log() from module "aloha/console" instead: ' + message);
		},

		/**
		 * Method to test if a range object is collapsed.
		 * A range is considered collapsed if either no endContainer exists or the endContainer/Offset equal startContainer/Offset
		 * @return {boolean} true if collapsed, false otherwise
		 * @method
		 */
		isCollapsed: function () {
			return (!this.endContainer || (this.startContainer === this.endContainer && this.startOffset === this.endOffset));
		},

		/**
		 * Method to (re-)calculate the common ancestor container and to get it.
		 * The common ancestor container is the DOM Object which encloses the
		 * whole range and is nearest to the start and end container objects.
		 * @return {DOMObject} get the common ancestor container
		 * @method
		 */
		getCommonAncestorContainer: function () {
			if (this.commonAncestorContainer) {
				// sometimes it's cached (or was set)
				return this.commonAncestorContainer;
			}
			// if it's not cached, calculate and then cache it
			this.updateCommonAncestorContainer();

			// now return it anyway
			return this.commonAncestorContainer;
		},

		/**
		 * Get the parent elements of the startContainer/endContainer up to the given limit. When the startContainer/endContainer
		 * is no text element, but a node, the node itself is returned as first element.
		 * @param {jQuery} limit limit object (default: body)
		 * @param {boolean} fromStart true to fetch the parents from the startContainer, false for the endContainer
		 * @return {jQuery} parent elements of the startContainer/endContainer as jQuery objects
		 * @method
		 */
		getContainerParents: function (limit, fromEnd) {
			// TODO cache the calculated parents
			var container = fromEnd ? this.endContainer : this.startContainer;
			if (!container) {
				return false;
			}
			return jQuery(selfAndParentsUntil(container, limit ? limit[0] : null));
		},

		/**
		 * Get the parent elements of the startContainer up to the given limit. When the startContainer
		 * is no text element, but a node, the node itself is returned as first element.
		 * @param {jQuery} limit limit object (default: body)
		 * @return {jQuery} parent elements of the startContainer as jQuery objects
		 * @method
		 */
		getStartContainerParents: function (limit) {
			return this.getContainerParents(limit, false);
		},

		/**
		 * Get the parent elements of the endContainer up to the given limit. When the endContainer is
		 * no text element, but a node, the node itself is returned as first element.
		 * @param {jQuery} limit limit object (default: body)
		 * @return {jQuery} parent elements of the endContainer as jQuery objects
		 * @method
		 */
		getEndContainerParents: function (limit) {
			return this.getContainerParents(limit, true);
		},

		/**
		 * TODO: the commonAncestorContainer is not calculated correctly, if either the start or
		 * the endContainer would be the cac itself (e.g. when the startContainer is a textNode
		 * and the endContainer is the startContainer's parent <p>). in this case the cac will be set
		 * to the parent div
		 * Method to update a range object internally
		 * @param commonAncestorContainer (DOM Object); optional Parameter; if set, the parameter
		 * will be used instead of the automatically calculated CAC
		 * @return void
		 * @hide
		 */
		updateCommonAncestorContainer: function (commonAncestorContainer) {
			// if no parameter was passed, calculate it
			if (!commonAncestorContainer) {
				// this will be needed either right now for finding the CAC or later for the crossing index
				var parentsStartContainer = this.getStartContainerParents(),
					parentsEndContainer = this.getEndContainerParents(),
					i;

				// find the crossing between startContainer and endContainer parents (=commonAncestorContainer)
				if (!(parentsStartContainer.length > 0 && parentsEndContainer.length > 0)) {
					console.warn('aloha/range', 'could not find commonAncestorContainer');
					return false;
				}

				for (i = 0; i < parentsStartContainer.length; i++) {
					if (parentsEndContainer.index(parentsStartContainer[i]) != -1) {
						this.commonAncestorContainer = parentsStartContainer[i];
						break;
					}
				}
			} else {
				this.commonAncestorContainer = commonAncestorContainer;
			}

			// if everything went well, return true :-)
			console.debug(commonAncestorContainer ? 'commonAncestorContainer was set successfully' : 'commonAncestorContainer was calculated successfully');
			return true;
		},

		/**
		 * Helper function for selection in IE. Creates a collapsed text range at the given position
		 * @param container container
		 * @param offset offset
		 * @return collapsed text range at that position
		 * @hide
		 */
		getCollapsedIERange: function (container, offset) {
			// create a text range
			var ieRange = document.body.createTextRange(),
				tmpRange,
			    right,
			    parent,
			    left;

			// search to the left for the next element
			left = this.searchElementToLeft(container, offset);
			if (left.element) {
				// found an element, set the start to the end of that element
				tmpRange = document.body.createTextRange();
				tmpRange.moveToElementText(left.element);
				ieRange.setEndPoint('StartToEnd', tmpRange);

				// and correct the start
				if (left.characters !== 0) {
					ieRange.moveStart('character', left.characters);
				} else {
					// this is a hack, when we are at the start of a text node, move the range anyway
					ieRange.moveStart('character', 1);
					ieRange.moveStart('character', -1);
				}
			} else {
				// found nothing to the left, so search right
				right = this.searchElementToRight(container, offset);
				// also found no element to the right, use the container itself
				parent = container.nodeType == 3 ? container.parentNode : container;
				tmpRange = document.body.createTextRange();
				tmpRange.moveToElementText(parent);
				ieRange.setEndPoint('StartToStart', tmpRange);

				// and correct the start
				if (left.characters !== 0) {
					ieRange.moveStart('character', left.characters);
				}
			}
			ieRange.collapse();

			return ieRange;
		},

		/**
		 * Sets the visible selection in the Browser based on the range object.
		 * If the selection is collapsed, this will result in a blinking cursor,
		 * otherwise in a text selection.
		 * @method
		 */
		select: function () {
			var ieRange, endRange, startRange, range, sel;

			if (typeof this.startContainer === 'undefined' || typeof this.endContainer === 'undefined') {
				console.warn('can not select an empty range');
				return false;
			}

			// create a range
			range = rangy.createRange();
			// set start and endContainer
			range.setStart(this.startContainer, this.startOffset);
			range.setEnd(this.endContainer, this.endOffset);

			// update the selection
			sel = rangy.getSelection();
			sel.setSingleRange(range);
		},

		/**
		 * Starting at the given position, search for the next element to the left and count the number of characters are in between
		 * @param container container of the startpoint
		 * @param offset offset of the startpoint in the container
		 * @return object with 'element' (null if no element found) and 'characters'
		 * @hide
		 */
		searchElementToLeft: function (container, offset) {
			var checkElement,
			    characters = 0;

			if (container.nodeType === 3) {
				// start is in a text node
				characters = offset;
				// begin check at the element to the left (if any)
				checkElement = container.previousSibling;
			} else {
				// start is between nodes, begin check at the element to the left (if any)
				if (offset > 0) {
					checkElement = container.childNodes[offset - 1];
				}
			}

			// move to the right until we find an element
			while (checkElement && checkElement.nodeType === 3) {
				characters += checkElement.data.length;
				checkElement = checkElement.previousSibling;
			}

			return {
				'element': checkElement,
				'characters': characters
			};
		},

		/**
		 * Starting at the given position, search for the next element to the right and count the number of characters that are in between
		 * @param container container of the startpoint
		 * @param offset offset of the startpoint in the container
		 * @return object with 'element' (null if no element found) and 'characters'
		 * @hide
		 */
		searchElementToRight: function (container, offset) {
			var checkElement,
			    characters = 0;

			if (container.nodeType === 3) {
				// start is in a text node
				characters = container.data.length - offset;

				// begin check at the element to the right (if any)
				checkElement = container.nextSibling;
			} else {
				// start is between nodes, begin check at the element to the right (if any)
				if (offset < container.childNodes.length) {
					checkElement = container.childNodes[offset];
				}
			}

			// move to the right until we find an element
			while (checkElement && checkElement.nodeType === 3) {
				characters += checkElement.data.length;
				checkElement = checkElement.nextSibling;
			}

			return {
				'element': checkElement,
				'characters': characters
			};
		},

		/**
		 * Method which updates the rangeObject including all extending properties like commonAncestorContainer etc...
		 * TODO: is this method needed here? or should it contain the same code as Aloha.Selection.prototype.SelectionRange.prototype.update?
		 * @return void
		 * @hide
		 */
		update: function (event) {
			console.debug('now updating rangeObject');

			this.initializeFromUserSelection(event);
			this.updateCommonAncestorContainer();
		},

		/**
		 * Initialize the current range object from the user selection of the browser.
		 * @param event which calls the method
		 * @return void
		 * @hide
		 */
		initializeFromUserSelection: function (event) {
			var selection = rangy.getSelection(),
				browserRange;

			if (!selection) {
				return false;
			}

			// check if a ragne exists
			if (!selection.rangeCount) {
				return false;
			}

			// getBrowserRange
			browserRange = selection.getRangeAt(0);
			if (!browserRange) {
				return false;
			}

			// initially set the range to what the browser tells us
			this.startContainer = browserRange.startContainer;
			this.endContainer = browserRange.endContainer;
			this.startOffset = browserRange.startOffset;
			this.endOffset = browserRange.endOffset;

			// now try to correct the range
			this.correctRange();
			return;
		},

		/**
		 * Correct the current range. The general goal of the algorithm is to have start
		 * and end of the range in text nodes if possible and the end of the range never
		 * at the beginning of an element or text node. Details of the algorithm can be
		 * found in the code comments
		 * @method
		 */
		correctRange: function () {
			var adjacentTextNode,
			    textNode,
			    checkedElement,
			    parentNode,
			    offset;

			this.clearCaches();
			if (this.isCollapsed()) {
				// collapsed ranges are treated specially

				// first check if the range is not in a text node
				if (this.startContainer.nodeType === 1) {
					if (this.startOffset > 0 && this.startContainer.childNodes[this.startOffset - 1].nodeType === 3) {
						// when the range is between nodes (container is an element
						// node) and there is a text node to the left -> move into this text
						// node (at the end)
						this.startContainer = this.startContainer.childNodes[this.startOffset - 1];
						this.startOffset = this.startContainer.data.length;
						this.endContainer = this.startContainer;
						this.endOffset = this.startOffset;
						return;
					}

					if (this.startOffset > 0 && this.startContainer.childNodes[this.startOffset - 1].nodeType === 1) {
						// search for the next text node to the left
						adjacentTextNode = GENTICS.Utils.Dom.searchAdjacentTextNode(this.startContainer, this.startOffset, true);
						if (adjacentTextNode) {
							this.startContainer = this.endContainer = adjacentTextNode;
							this.startOffset = this.endOffset = adjacentTextNode.data.length;
							return;
						}
						// search for the next text node to the right
						adjacentTextNode = GENTICS.Utils.Dom.searchAdjacentTextNode(this.startContainer, this.startOffset, false);
						if (adjacentTextNode) {
							this.startContainer = this.endContainer = adjacentTextNode;
							this.startOffset = this.endOffset = 0;
							return;
						}
					}

					if (this.startOffset < this.startContainer.childNodes.length && this.startContainer.childNodes[this.startOffset].nodeType === 3) {
						// when the range is between nodes and there is a text node
						// to the right -> move into this text node (at the start)
						this.startContainer = this.startContainer.childNodes[this.startOffset];
						this.startOffset = 0;
						this.endContainer = this.startContainer;
						this.endOffset = 0;
						return;
					}
				}

				// when the selection is in a text node at the start, look for an adjacent text node and if one found, move into that at the end
				if (this.startContainer.nodeType === 3 && this.startOffset === 0) {
					adjacentTextNode = GENTICS.Utils.Dom.searchAdjacentTextNode(this.startContainer.parentNode, GENTICS.Utils.Dom.getIndexInParent(this.startContainer), true);
					//only move the selection if the adjacentTextNode is inside the current editable
					//the cursor should not be outside the editable
					if (adjacentTextNode && jQuery(adjacentTextNode).closest(Aloha.activeEditable.obj).length > 0) {
						this.startContainer = this.endContainer = adjacentTextNode;
						this.startOffset = this.endOffset = adjacentTextNode.data.length;
					}
				}
			} else {
				// expanded range found

				// correct the start, but only if between nodes
				if (this.startContainer.nodeType === 1) {
					// if there is a text node to the right, move into this
					if (this.startOffset < this.startContainer.childNodes.length && this.startContainer.childNodes[this.startOffset].nodeType === 3) {
						this.startContainer = this.startContainer.childNodes[this.startOffset];
						this.startOffset = 0;
					} else if (this.startOffset < this.startContainer.childNodes.length && this.startContainer.childNodes[this.startOffset].nodeType === 1) {
						// there is an element node to the right, so recursively check all first child nodes until we find a text node
						textNode = false;
						checkedElement = this.startContainer.childNodes[this.startOffset];
						while (textNode === false && checkedElement.childNodes && checkedElement.childNodes.length > 0) {
							// go to the first child of the checked element
							checkedElement = checkedElement.childNodes[0];
							// when this element is a text node, we are done
							if (checkedElement.nodeType === 3) {
								textNode = checkedElement;
							}
						}

						// found a text node, so move into it
						if (textNode !== false) {
							this.startContainer = textNode;
							this.startOffset = 0;
						}
					}
				}

				// check whether the start is inside a text node at the end
				if (this.startContainer.nodeType === 3 && this.startOffset === this.startContainer.data.length) {
					// check whether there is an adjacent text node to the right and if
					// yes, move into it
					adjacentTextNode = GENTICS.Utils.Dom.searchAdjacentTextNode(this.startContainer.parentNode, GENTICS.Utils.Dom.getIndexInParent(this.startContainer) + 1, false);
					if (adjacentTextNode) {
						this.startContainer = adjacentTextNode;
						this.startOffset = 0;
					}
				}

				// now correct the end
				if (this.endContainer.nodeType === 3 && this.endOffset === 0) {
					// we are in a text node at the start
					if (this.endContainer.previousSibling && this.endContainer.previousSibling.nodeType === 3) {
						// found a text node to the left -> move into it (at the end)
						this.endContainer = this.endContainer.previousSibling;
						this.endOffset = this.endContainer.data.length;
					} else if (this.endContainer.previousSibling && this.endContainer.previousSibling.nodeType === 1 && this.endContainer.parentNode) {
						// found an element node to the left -> move in between
						parentNode = this.endContainer.parentNode;
						for (offset = 0; offset < parentNode.childNodes.length; ++offset) {
							if (parentNode.childNodes[offset] == this.endContainer) {
								this.endOffset = offset;
								break;
							}
						}
						this.endContainer = parentNode;
					}
				}

				if (this.endContainer.nodeType == 1 && this.endOffset === 0) {
					// we are in an element node at the start, possibly move to the previous sibling at the end
					if (this.endContainer.previousSibling) {
						if (this.endContainer.previousSibling.nodeType === 3) {
							// previous sibling is a text node, move end into here (at the end)
							this.endContainer = this.endContainer.previousSibling;
							this.endOffset = this.endContainer.data.length;
						} else if (this.endContainer.previousSibling.nodeType === 1 && this.endContainer.previousSibling.childNodes && this.endContainer.previousSibling.childNodes.length > 0) {
							// previous sibling is another element node with children,
							// move end into here (at the end)
							this.endContainer = this.endContainer.previousSibling;
							this.endOffset = this.endContainer.childNodes.length;
						}
					}
				}

				// correct the end, but only if between nodes
				if (this.endContainer.nodeType == 1) {
					// if there is a text node to the left, move into this
					if (this.endOffset > 0 && this.endContainer.childNodes[this.endOffset - 1].nodeType === 3) {
						this.endContainer = this.endContainer.childNodes[this.endOffset - 1];
						this.endOffset = this.endContainer.data.length;
					} else if (this.endOffset > 0 && this.endContainer.childNodes[this.endOffset - 1].nodeType === 1) {
						// there is an element node to the left, so recursively check all last child nodes until we find a text node
						textNode = false;
						checkedElement = this.endContainer.childNodes[this.endOffset - 1];
						while (textNode === false && checkedElement.childNodes && checkedElement.childNodes.length > 0) {
							// go to the last child of the checked element
							checkedElement = checkedElement.childNodes[checkedElement.childNodes.length - 1];
							// when this element is a text node, we are done
							if (checkedElement.nodeType === 3) {
								textNode = checkedElement;
							}
						}

						// found a text node, so move into it
						if (textNode !== false) {
							this.endContainer = textNode;
							this.endOffset = this.endContainer.data.length;
						}
					}
				}
			}
		},

		/**
		 * Clear the caches for this range. This method must be called when the range itself (start-/endContainer or start-/endOffset) is modified.
		 * @method
		 */
		clearCaches: function () {
			this.commonAncestorContainer = undefined;
		},

		/**
		 * Get the range tree of this range.
		 * The range tree will be cached for every root object. When the range itself is modified, the cache should be cleared by calling GENTICS.Utils.RangeObject.clearCaches
		 * @param {DOMObject} root root object of the range tree, if non given, the common ancestor container of the start and end containers will be used
		 * @return {RangeTree} array of RangeTree object for the given root object
		 * @method
		 */
		getRangeTree: function (root) {
			// TODO cache rangeTrees
			if (typeof root === 'undefined') {
				root = this.getCommonAncestorContainer();
			}

			this.inselection = false;
			return this.recursiveGetRangeTree(root);
		},

		/**
		 * Recursive inner function for generating the range tree.
		 * @param currentObject current DOM object for which the range tree shall be generated
		 * @return array of Tree objects for the children of the current DOM object
		 * @hide
		 */
		recursiveGetRangeTree: function (currentObject) {
			// get all direct children of the given object
			var jQueryCurrentObject = jQuery(currentObject),
				childCount = 0,
				that = this,
				currentElements = [];

			jQueryCurrentObject.contents().each(function (index) {
				var type = 'none',
					startOffset = false,
					endOffset = false,
					collapsedFound = false,
					noneFound = false,
					partialFound = false,
					fullFound = false,
					i;

				// check for collapsed selections between nodes
				if (that.isCollapsed() && currentObject === that.startContainer && that.startOffset === index) {
					// insert an extra rangetree object for the collapsed range here
					currentElements[childCount] = new GENTICS.Utils.RangeTree();
					currentElements[childCount].type = 'collapsed';
					currentElements[childCount].domobj = undefined;
					that.inselection = false;
					collapsedFound = true;
					childCount++;
				}

				if (!that.inselection && !collapsedFound) {
					// the start of the selection was not yet found, so look for it now
					// check whether the start of the selection is found here

					// check is dependent on the node type
					switch (this.nodeType) {
					case 3:
						// text node
						if (this === that.startContainer) {
							// the selection starts here
							that.inselection = true;

							// when the startoffset is > 0, the selection type is only partial
							type = that.startOffset > 0 ? 'partial' : 'full';
							startOffset = that.startOffset;
							endOffset = this.length;
						}
						break;
					case 1:
						// element node
						if (this === that.startContainer && that.startOffset === 0) {
							// the selection starts here
							that.inselection = true;
							type = 'full';
						}
						if (currentObject === that.startContainer && that.startOffset === index) {
							// the selection starts here
							that.inselection = true;
							type = 'full';
						}
						break;
					}
				}

				if (that.inselection && !collapsedFound) {
					if (type == 'none') {
						type = 'full';
					}
					// we already found the start of the selection, so look for the end of the selection now
					// check whether the end of the selection is found here

					switch (this.nodeType) {
					case 3:
						// text node
						if (this === that.endContainer) {
							// the selection ends here
							that.inselection = false;

							// check for partial selection here
							if (that.endOffset < this.length) {
								type = 'partial';
							}
							if (startOffset === false) {
								startOffset = 0;
							}
							endOffset = that.endOffset;
						}
						break;
					case 1:
						// element node
						if (this === that.endContainer && that.endOffset === 0) {
							that.inselection = false;
						}
						break;
					}
					if (currentObject === that.endContainer && that.endOffset <= index) {
						that.inselection = false;
						type = 'none';
					}
				}

				// create the current selection tree entry
				currentElements[childCount] = new GENTICS.Utils.RangeTree();
				currentElements[childCount].domobj = this;
				currentElements[childCount].type = type;
				if (type == 'partial') {
					currentElements[childCount].startOffset = startOffset;
					currentElements[childCount].endOffset = endOffset;
				}

				// now do the recursion step into the current object
				currentElements[childCount].children = that.recursiveGetRangeTree(this);

				// check whether a selection was found within the children
				if (currentElements[childCount].children.length > 0) {
					for (i = 0; i < currentElements[childCount].children.length; ++i) {
						switch (currentElements[childCount].children[i].type) {
						case 'none':
							noneFound = true;
							break;
						case 'full':
							fullFound = true;
							break;
						case 'partial':
							partialFound = true;
							break;
						}
					}

					if (partialFound || (fullFound && noneFound)) {
						// found at least one 'partial' DOM object in the children, or both 'full' and 'none', so this element is also 'partial' contained
						currentElements[childCount].type = 'partial';
					} else if (fullFound && !partialFound && !noneFound) {
						// only found 'full' contained children, so this element is also 'full' contained
						currentElements[childCount].type = 'full';
					}
				}

				childCount++;
			});

			// extra check for collapsed selections at the end of the current element
			if (this.isCollapsed() && currentObject === this.startContainer && this.startOffset == currentObject.childNodes.length) {
				currentElements[childCount] = new GENTICS.Utils.RangeTree();
				currentElements[childCount].type = 'collapsed';
				currentElements[childCount].domobj = undefined;
			}

			return currentElements;
		},

		/**
		 * Find certain the first occurrence of some markup within the parents of either the start or the end of this range.
		 * The markup can be identified by means of a given comparator function. The function will be passed every parent (up to the eventually given limit object, which itself is not considered) to the comparator function as this.
		 * When the comparator function returns boolean true, the markup found and finally returned from this function as dom object.<br/>
		 * Example for finding an anchor tag at the start of the range up to the active editable object:<br/>
		 * <pre>
		 * range.findMarkup(
		 *   function() {
		 *     return this.nodeName.toLowerCase() == 'a';
		 *   },
		 *   jQuery(Aloha.activeEditable.obj)
		 * );
		 * </pre>
		 * @param {function} comparator comparator function to find certain markup
		 * @param {jQuery} limit limit objects for limit the parents taken into consideration
		 * @param {boolean} atEnd true for searching at the end of the range, false for the start (default: false)
		 * @return {DOMObject} the found dom object or false if nothing found.
		 * @method
		 */
		findMarkup: function (comparator, limit, atEnd) {
			var container = atEnd ? this.endContainer : this.startContainer,
				parents,
				i,
				len;
			limit = limit ? limit[0] : null;
			if (!container) {
				return;
			}
			parents = selfAndParentsUntil(container, limit);
			for (i = 0, len = parents.length; i < len; i++) {
				if (comparator.apply(parents[i])) {
					return parents[i];
				}
			}
			return false;
		},

		/**
		 * Get the text enclosed by this range
		 * @return {String} the text of the range
		 * @method
		 */
		getText: function () {
			if (this.isCollapsed()) {
				return '';
			}
			return this.recursiveGetText(this.getRangeTree());
		},

		recursiveGetText: function (tree) {
			if (!tree) {
				return '';
			}
			var that = this,
			    text = '';
			jQuery.each(tree, function () {
				if (this.type == 'full') {
					// fully selected element/text node
					text += jQuery(this.domobj).text();
				} else if (this.type == 'partial' && this.domobj.nodeType === 3) {
					// partially selected text node
					text += jQuery(this.domobj).text().substring(this.startOffset, this.endOffset);
				} else if (this.type == 'partial' && this.domobj.nodeType === 1 && this.children) {
					// partially selected element node
					text += that.recursiveGetText(this.children);
				}
			});
			return text;
		}
	});

	/**
	 * @namespace GENTICS.Utils
	 * @class RangeTree
	 * Class definition of a RangeTree, which gives a tree view of the DOM objects included in this range
	 * Structure:
	 * <pre>
	 * +
	 * |-domobj: <reference to the DOM Object> (NOT jQuery)
	 * |-type: defines if this node is marked by user [none|partial|full|collapsed]
	 * |-children: recursive structure like this
	 * </pre>
	 */
	GENTICS.Utils.RangeTree = Class.extend({
		/**
		 * DOMObject, if the type is one of [none|partial|full], undefined if the type is [collapsed]
		 * @property domobj
		 * @type {DOMObject}
		 */
		domobj: {},

		/**
		 * type of the participation of the dom object in the range. Is one of:
		 * <pre>
		 * - none the DOMObject is outside of the range
		 * - partial the DOMObject partially in the range
		 * - full the DOMObject is completely in the range
		 * - collapsed the current RangeTree element marks the position of a collapsed range between DOM nodes
		 * </pre>
		 * @property type
		 * @type {String}
		 */
		type: null,

		/**
		 * Array of RangeTree objects which reflect the status of the child elements of the current DOMObject
		 * @property children
		 * @type {Array}
		 */
		children: []
	});

	return GENTICS.Utils.RangeObject;
});

/* arrays.js is part of Aloha Editor project http://aloha-editor.org
 *
 * Aloha Editor is a WYSIWYG HTML5 inline editing library and editor. 
 * Copyright (c) 2010-2012 Gentics Software GmbH, Vienna, Austria.
 * Contributors http://aloha-editor.org/contribution.php 
 * 
 * Aloha Editor is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or any later version.
 *
 * Aloha Editor is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 * 
 * As an additional permission to the GNU GPL version 2, you may distribute
 * non-source (e.g., minimized or compacted) forms of the Aloha-Editor
 * source code without the copy of the GNU GPL normally required,
 * provided you include this license notice and a URL through which
 * recipients can access the Corresponding Source.
 */
define('util/arrays',[], function () {
	

	/**
	 * Implements unique() using the browser's sort().
	 *
	 * @param a
	 *        The array to sort and strip of duplicate values.
	 *        Warning: this array will be modified in-place.
	 * @param compFn
	 *        A custom comparison function that accepts two values a and
	 *        b from the given array and returns -1, 0, 1 depending on
	 *        whether a < b, a == b, a > b respectively.
	 *
	 *        If no compFn is provided, the algorithm will use the
	 *        browsers default sort behaviour and loose comparison to
	 *        detect duplicates.
	 * @return
	 *        The given array.
	 */
	function sortUnique(a, compFn) {
		var i;
		if (compFn) {
			a.sort(compFn);
			for (i = 1; i < a.length; i++) {
				if (0 === compFn(a[i], a[i - 1])) {
					a.splice(i--, 1);
				}
			}
		} else {
			a.sort();
			for (i = 1; i < a.length; i++) {
				// Use loosely typed comparsion if no compFn is given
				// to avoid sortUnique( [6, "6", 6] ) => [6, "6", 6]
				if (a[i] == a[i - 1]) {
					a.splice(i--, 1);
				}
			}
		}
		return a;
	}

	/**
	 * Shallow comparison of two arrays.
	 *
	 * @param a, b
	 *        The arrays to compare.
	 * @param equalFn
	 *        A custom comparison function that accepts two values a and
	 *        b from the given arrays and returns true or false for
	 *        equal and not equal respectively.
	 *
	 *        If no equalFn is provided, the algorithm will use the strict
	 *        equals operator.
	 * @return
	 *        True if all items in a and b are equal, false if not.
	 */
	function equal(a, b, equalFn) {
		var i,
			len = a.length;
		if (len !== b.length) {
			return false;
		}
		if (equalFn) {
			for (i = 0; i < len; i++) {
				if (!equalFn(a[i], b[i])) {
					return false;
				}
			}
		} else {
			for (i = 0; i < len; i++) {
				if (a[i] !== b[i]) {
					return false;
				}
			}
		}
		return true;
	}

	/**
	 * ECMAScript map replacement
	 * See https://developer.mozilla.org/en/JavaScript/Reference/Global_Objects/Array/map
	 * And http://es5.github.com/#x15.4.4.19
	 * It's not exactly according to standard, but it does exactly what one expects.
	 */
	function map(a, fn) {
		var i, len, result = [];
		for (i = 0, len = a.length; i < len; i++) {
			result.push(fn(a[i]));
		}
		return result;
	}

	function mapNative(a, fn) {
		// Call map directly on the object instead of going through
		// Array.prototype.map. This avoids the problem that we may get
		// passed an array-like object (NodeList) which may cause an
		// error if the implementation of Array.prototype.map can only
		// deal with arrays (Array.prototype.map may be native or
		// provided by a javscript framework).
		return a.map(fn);
	}

	/**
	 * Returns a new array that contains all values in the given a for
	 * which pred returns true.
	 */
	function filter(a, pred) {
		var i,
		    len,
		    value,
		    result = [];
		for (i = 0, len = a.length; i < len; i++) {
			value = a[i];
			if (pred(value)) {
				result.push(value);
			}
		}
		return result;
	}

	/**
	 * Finds a value in the given array.
	 * Strict comparison is used to find the value.
	 * Returns the index of the first occurrence of the given value in
	 * the given a, or -1 if a contains no such value.
	 */
	function indexOf(a, value) {
		var i,
		    len;
		for (i = 0, len = a.length; i < len; i++) {
			if (value === a[i]) {
				return i;
			}
		}
		return -1;
	}

	/**
	 * Reduces an array of values to a single value.
	 *
	 * For example:
	 * Arrays.reduce([2, 3, 4], 1, function (a, b) { return a + b; });
	 * returns the result of (((1 + 2) + 3) + 4)
	 *
	 * @param a
	 *        An array of values.
	 * @param init
	 *        An initial value.
	 * @param fn
	 *        A function that takes two values and returns the reduction
	 *        of both.
	 */
	function reduce(a, init, fn) {
		var i,
		    len;
		for (i = 0, len = a.length; i < len; i++) {
			init = fn(init, a[i]);
		}
		return init;
	}

	/**
	 * Applies the given value to the given function unless the value is
	 * null, in which case just returns null.
	 *
	 * This is a utility function to be used with reduce().
	 */
	function applyNotNull(value, fn) {
		return value == null ? null : fn(value);
	}

	return {
		filter: filter,
		indexOf: indexOf,
		reduce: reduce,
		applyNotNull: applyNotNull,
		sortUnique: sortUnique,
		equal: equal,
		map: Array.prototype.map ? mapNative : map
	};
});

/* strings.js is part of Aloha Editor project http://aloha-editor.org
 *
 * Aloha Editor is a WYSIWYG HTML5 inline editing library and editor. 
 * Copyright (c) 2010-2012 Gentics Software GmbH, Vienna, Austria.
 * Contributors http://aloha-editor.org/contribution.php 
 * 
 * Aloha Editor is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or any later version.
 *
 * Aloha Editor is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 * 
 * As an additional permission to the GNU GPL version 2, you may distribute
 * non-source (e.g., minimized or compacted) forms of the Aloha-Editor
 * source code without the copy of the GNU GPL normally required,
 * provided you include this license notice and a URL through which
 * recipients can access the Corresponding Source.
 */
define('util/strings',['jquery'], function ($) {
	

	var spacesRx = /\s+/;

	/**
	 * Splits a string into individual words.
	 *
	 * Words are non-empty sequences of non-space characaters.
	 */
	function words(str) {
		var list = str.split(spacesRx);
		// "  x  ".split(/\s+/) => ["", "x", ""] (Chrome)
		// "".split(/\s+/) => [""] (Chrome)
		if (list.length && list[0] === "") {
			list.shift();
		}
		if (list.length && list[list.length - 1] === "") {
			list.pop();
		}
		return list;
	}

	/**
	 * Converst a dashes form into camel cased form.
	 *
	 * For example 'data-my-attr' becomes 'dataMyAttr'.
	 *
	 * @param {string} s
	 *        Should be all lowercase and should not begin with a dash
	 */
	function dashesToCamelCase(s) {
		return s.replace(/[\-]([a-z])/gi, function (all, upper) {
			return upper.toUpperCase();
		});
	}

	/**
	 * Converts a camel cased form into dashes form.
	 *
	 * For example
	 * 'dataMyAttr' becomes 'data-my-attr',
	 * 'dataAB'     becomes 'data-a-b'.
	 *
	 * @param {string} s
	 *        Should begin with a lowercase letter and should not contain dashes.
	 */
	function camelCaseToDashes(s) {
		return s.replace(/[A-Z]/g, function (match) {
			return '-' + match.toLowerCase();
		});
	}

	return {
		'words': words,
		'dashesToCamelCase': dashesToCamelCase,
		'camelCaseToDashes': camelCaseToDashes
	};
});

/*!
 * Aloha Editor
 * Author & Copyright (c) 2012 Gentics Software GmbH
 * aloha-sales@gentics.com
 * Licensed under the terms of http://www.aloha-editor.com/license.html
 *
 * @overview Provides methods to broker publish/subscribe facilities.
 */
define('PubSub', [], function () {
	

	/**
	 * A hash of channel names mapped to an array of ids of subscriptions that
	 * are listening on that channel.
	 *
	 * @type {Object<String, Array.<Number>>}
	 */
	var channels = {};

	/**
	 * A hash of subscription tuples (channel, callback), mapped against unique
	 * ids assigned to each subscription.
	 * As subscriptions are removed from this object via `unsub()' this object
	 * will become a sparse array.
	 *
	 * @type {Object<Number, Object>}
	 */
	var subscriptions = {};

	/**
	 * The last used subscription id.  This values is only used and modified in
	 * `sub().'
	 *
	 * @type {number}
	 */
	var sid = 0;

	/**
	 * Returns the channel to which a subscription matching the given sid is
	 * listening on.
	 *
	 * @param {Number} sid Id of subscription.
	 * @return {Array.<Object>} sid Id of subscription.
	 */
	function getSubscriptionChannel(sid) {
		return subscriptions[sid] && channels[subscriptions[sid].channel];
	}

	/**
	 * Publishes a message `message' on the given channel.
	 * All callbacks that have sub()scribed to listen on this channel will be
	 * invoked and receive `message' as their only argument.
	 *
	 * @private
	 * @param {String} channel Name of channel to publish the message on.
	 * @param {*} message Variable to pass to all callbacks listening on the
	 *                    given channel.
	 * @return {Number} The number of subscribed callbacks that were invoked.
	 */
	function pub(channel, message) {
		if (!channels[channel]) {
			return 0;
		}

		if (!message) {
			message = {};
		} else if (typeof message !== 'object') {
			message = {
				data: message
			};
		}

		message.channel = channel;

		// Clone a immutable snapshot of the subscription ids that we can
		// safetly iterate over.
		var sids = channels[channel].slice();

		// NB: It is necessary to read the size of the `sids' array on each
		// iteration, in case the size changes (via unsubscription) between
		// iterations.
		var i;
		for (i = 0; i < sids.length; ++i) {
			subscriptions[sids[i]].callback(message);
		}

		return i;
	}

	var PubSub = {

		/**
		 * Subscribes a callback function to a channel.  Whenever this channel
		 * publishes, this function will be invoked.  The return value is an id
		 * which identifies this subscription (a channel, and callback tuple).
		 * This id can be used to unsubscribe this subscription from the given
		 * channel.
		 *
		 * @param {String} channel Name of channel to listen on.
		 * @param {Function(Object)} callback Function to be invoked when
		 *                                    messages are published on the
		 *                                    given channel.
		 * @return {Number} Positive integer representing the sid of this
		 *                  subscription, that can be used with unsub() if
		 *                  subscription succeeds.  Otherwise the return value
		 *                  is -1;
		 */
		sub: function (channel, callback) {
			if (typeof callback !== 'function') {
				return -1;
			}

			var subscriptionIds = channels[channel];

			if (!subscriptionIds) {
				subscriptionIds = channels[channel] = [];
			}

			subscriptionIds.push(++sid);
			subscriptions[sid] = {
				channel  : channel,
				callback : callback
			};

			return sid;
		},

		/**
		 * Unsubscribes callback using an sid which was returned by sub() when
		 * the callback was subscribed.  Returns true if a subscription for
		 * this sid was found and removed, otherwise returns false.
		 *
		 * @param {Number} sid Id of subscription.
		 * @return {Boolean} True if a a subscription matching this sid was
		 *                   removed.
		 */
		unsub: function (sid) {
			if (-1 === sid || !subscriptions[sid]) {
				return false;
			}

			var subscriptionIds = getSubscriptionChannel(sid);

			// assert(typeof subscriptionIds === 'array')

			delete subscriptions[sid];
			var j = subscriptionIds.length;

			while (j) {
				if (subscriptionIds[--j] === sid) {
					subscriptionIds.splice(j, 1);
					return true;
				}
			}

			return false;
		},

		/**
		 * Publishes a message `message' on all channels that can be derived
		 * from the given channel name.
		 *
		 * @param {String} channel Name of channel to publish the message on.
		 * @param {*} message Variable to pass to all callbacks listening on
		 *                    the given channel.
		 * @return {Number} The number of subscribed callbacks that were
		 *                  invoked.
		 */
		pub: function (channel, message) {
			var segments = channel.split('.');
			var i;
			var len = segments.length;
			var channelName = '';
			var tally = 0;

			for (i = 0; i < len; ++i) {
				channelName += (0 === i ? '' : '.') + segments[i];
				tally += pub(channelName, message);
			}

			return tally;
		}

	};

	return PubSub;
});

define('aloha/engine',['aloha/core', 'aloha/ecma5shims', 'jquery'], function (Aloha, $_, jQuery) {
	

	function hasAttribute(obj, attr) {
		var native_method = obj.hasAttribute;
		if (native_method) {
			return obj.hasAttribute(attr);
		}
		return (typeof obj.attributes[attr] != "undefined");
	}

	var htmlNamespace = "http://www.w3.org/1999/xhtml";

	var cssStylingFlag = false;

	// This is bad :(
	var globalRange = null;

	// Commands are stored in a dictionary where we call their actions and such
	var commands = {};

	///////////////////////////////////////////////////////////////////////////////
	////////////////////////////// Utility functions //////////////////////////////
	///////////////////////////////////////////////////////////////////////////////
	//@{

	// Opera 11 puts HTML elements in the null namespace, it seems.
	function isHtmlNamespace(ns) {
		return ns === null || !ns || ns === htmlNamespace;
	}

	// "An HTML element is an Element whose namespace is the HTML namespace."
	//
	// I allow an extra argument to more easily check whether something is a
	// particular HTML element, like isNamedHtmlElement(node, 'OL').  It accepts arrays
	// too, like isHtmlElementInArray(node, ["OL", "UL"]) to check if it's an ol or ul.
	// TODO This function was prominent during profiling. Remove it
	//      and replace with calls to isAnyHtmlElement, isNamedHtmlElement
	//      and is isMappedHtmlElement.
	function isHtmlElement_obsolete(node, tags) {
		if (typeof tags == "string") {
			tags = [tags];
		}
		if (typeof tags == "object") {
			tags = $_(tags).map(function (tag) {
				return tag.toUpperCase();
			});
		}
		return node && node.nodeType == 1 && isHtmlNamespace(node.namespaceURI) && (typeof tags == "undefined" || $_(tags).indexOf(node.tagName) != -1);
	}

	function isAnyHtmlElement(node) {
		return node && node.nodeType == 1 && isHtmlNamespace(node.namespaceURI);
	}

	// name should be uppercase
	function isNamedHtmlElement(node, name) {
		return node && node.nodeType == 1 && isHtmlNamespace(node.namespaceURI)
		// This function is passed in a mix of upper and lower case names
			&& name.toUpperCase() === node.nodeName;
	}

	// TODO remove when isHtmlElementInArray is removed
	function arrayContainsInsensitive(array, str) {
		var i, len;
		str = str.toUpperCase();
		for (i = 0, len = array.length; i < len; i++) {
			if (array[i].toUpperCase() === str) {
				return true;
			}
		}
		return false;
	}
	// TODO replace calls to this function with calls to isMappedHtmlElement()
	function isHtmlElementInArray(node, array) {
		return node && node.nodeType == 1 && isHtmlNamespace(node.namespaceURI)
		// This function is passed in a mix of upper and lower case names
			&& arrayContainsInsensitive(array, node.nodeName);
	}

	// map must have all-uppercase keys
	function isMappedHtmlElement(node, map) {
		return node && node.nodeType == 1 && isHtmlNamespace(node.namespaceURI) && map[node.nodeName];
	}

	/**
	 * Method to count the number of styles in the given style
	 */
	function getStyleLength(node) {
		var s;
		var styleLength = 0;

		if (!node) {
			return 0;
		}

		if (!node.style) {
			return 0;
		}

		// some browsers support .length on styles
		if (typeof node.style.length !== 'undefined') {
			return node.style.length;
		}

		/*jslint forin: true*/ //not sure whether node.style.hasOwnProperty is valid
		for (s in node.style) {
			if (node.style[s] && node.style[s] !== 0 && node.style[s] !== 'false') {
				styleLength++;
			}
		}
		/*jslint forin: false*/

		return styleLength;
	}

	function toArray(obj) {
		if (!obj) {
			return null;
		}
		var array = [],
			i,
		    l = obj.length;
		// iterate backwards ensuring that length is an UInt32
		i = l >>> 0;
		while (i--) {
			array[i] = obj[i];
		}
		return array;
	}

	function nextNodeDescendants(node) {
		while (node && !node.nextSibling) {
			node = node.parentNode;
		}
		if (!node) {
			return null;
		}
		return node.nextSibling;
	}

	function nextNode(node) {
		if (node.hasChildNodes()) {
			return node.firstChild;
		}
		return nextNodeDescendants(node);
	}

	function previousNode(node) {
		if (node.previousSibling) {
			node = node.previousSibling;
			while (node.hasChildNodes()) {
				node = node.lastChild;
			}
			return node;
		}
		if (node.parentNode && node.parentNode.nodeType == $_.Node.ELEMENT_NODE) {
			return node.parentNode;
		}
		return null;
	}

	/**
	 * Returns true if ancestor is an ancestor of descendant, false otherwise.
	 */
	function isAncestor(ancestor, descendant) {
		return ancestor && descendant && Boolean($_.compareDocumentPosition(ancestor, descendant) & $_.Node.DOCUMENT_POSITION_CONTAINED_BY);
	}

	/**
	 * Returns true if ancestor is an ancestor of or equal to descendant, false
	 * otherwise.
	 */
	function isAncestorContainer(ancestor, descendant) {
		return (ancestor || descendant) && (ancestor == descendant || isAncestor(ancestor, descendant));
	}

	/**
	 * Returns true if descendant is a descendant of ancestor, false otherwise.
	 */
	function isDescendant(descendant, ancestor) {
		return ancestor && descendant && Boolean($_.compareDocumentPosition(ancestor, descendant) & $_.Node.DOCUMENT_POSITION_CONTAINED_BY);
	}

	/**
	 * Returns true if node1 is before node2 in tree order, false otherwise.
	 */
	function isBefore(node1, node2) {
		return Boolean($_.compareDocumentPosition(node1, node2) & $_.Node.DOCUMENT_POSITION_FOLLOWING);
	}

	/**
	 * Returns true if node1 is after node2 in tree order, false otherwise.
	 */
	function isAfter(node1, node2) {
		return Boolean($_.compareDocumentPosition(node1, node2) & $_.Node.DOCUMENT_POSITION_PRECEDING);
	}

	function getAncestors(node) {
		var ancestors = [];
		while (node.parentNode) {
			ancestors.unshift(node.parentNode);
			node = node.parentNode;
		}
		return ancestors;
	}

	function getDescendants(node) {
		var descendants = [];
		var stop = nextNodeDescendants(node);
		while (null != (node = nextNode(node)) && node != stop) {
			descendants.push(node);
		}
		return descendants;
	}

	function convertProperty(property) {
		// Special-case for now
		var map = {
			"fontFamily": "font-family",
			"fontSize": "font-size",
			"fontStyle": "font-style",
			"fontWeight": "font-weight",
			"textDecoration": "text-decoration"
		};
		if (typeof map[property] != "undefined") {
			return map[property];
		}

		return property;
	}

	// Return the <font size=X> value for the given CSS size, or undefined if there
	// is none.
	function cssSizeToLegacy(cssVal) {
		return {
			"xx-small": 1,
			"small": 2,
			"medium": 3,
			"large": 4,
			"x-large": 5,
			"xx-large": 6,
			"xxx-large": 7
		}[cssVal];
	}

	// Return the CSS size given a legacy size.
	function legacySizeToCss(legacyVal) {
		return {
			1: "xx-small",
			2: "small",
			3: "medium",
			4: "large",
			5: "x-large",
			6: "xx-large",
			7: "xxx-large"
		}[legacyVal];
	}

	// "the directionality" from HTML.  I don't bother caring about non-HTML
	// elements.
	//
	// "The directionality of an element is either 'ltr' or 'rtl', and is
	// determined as per the first appropriate set of steps from the following
	// list:"
	function getDirectionality(element) {
		// "If the element's dir attribute is in the ltr state
		//     The directionality of the element is 'ltr'."
		if (element.dir == "ltr") {
			return "ltr";
		}

		// "If the element's dir attribute is in the rtl state
		//     The directionality of the element is 'rtl'."
		if (element.dir == "rtl") {
			return "rtl";
		}

		// "If the element's dir attribute is in the auto state
		// "If the element is a bdi element and the dir attribute is not in a
		// defined state (i.e. it is not present or has an invalid value)
		//     [lots of complicated stuff]
		//
		// Skip this, since no browser implements it anyway.

		// "If the element is a root element and the dir attribute is not in a
		// defined state (i.e. it is not present or has an invalid value)
		//     The directionality of the element is 'ltr'."
		if (!isAnyHtmlElement(element.parentNode)) {
			return "ltr";
		}

		// "If the element has a parent element and the dir attribute is not in a
		// defined state (i.e. it is not present or has an invalid value)
		//     The directionality of the element is the same as the element's
		//     parent element's directionality."
		return getDirectionality(element.parentNode);
	}

	//@}

	///////////////////////////////////////////////////////////////////////////////
	///////////////////////////// DOM Range functions /////////////////////////////
	///////////////////////////////////////////////////////////////////////////////
	//@{

	function getNodeIndex(node) {
		var ret = 0;
		while (node.previousSibling) {
			ret++;
			node = node.previousSibling;
		}
		return ret;
	}

	// "The length of a Node node is the following, depending on node:
	//
	// ProcessingInstruction
	// DocumentType
	//   Always 0.
	// Text
	// Comment
	//   node's length.
	// Any other node
	//   node's childNodes's length."
	function getNodeLength(node) {
		switch (node.nodeType) {
		case $_.Node.PROCESSING_INSTRUCTION_NODE:
		case $_.Node.DOCUMENT_TYPE_NODE:
			return 0;

		case $_.Node.TEXT_NODE:
		case $_.Node.COMMENT_NODE:
			return node.length;

		default:
			return node.childNodes.length;
		}
	}

	/**
	 * The position of two boundary points relative to one another, as defined by
	 * DOM Range.
	 */
	function getPosition(nodeA, offsetA, nodeB, offsetB) {
		// "If node A is the same as node B, return equal if offset A equals offset
		// B, before if offset A is less than offset B, and after if offset A is
		// greater than offset B."
		if (nodeA == nodeB) {
			if (offsetA == offsetB) {
				return "equal";
			}
			if (offsetA < offsetB) {
				return "before";
			}
			if (offsetA > offsetB) {
				return "after";
			}
		}

		var documentPosition = $_.compareDocumentPosition(nodeB, nodeA);
		// "If node A is after node B in tree order, compute the position of (node
		// B, offset B) relative to (node A, offset A). If it is before, return
		// after. If it is after, return before."
		if (documentPosition & $_.Node.DOCUMENT_POSITION_FOLLOWING) {
			var pos = getPosition(nodeB, offsetB, nodeA, offsetA);
			if (pos == "before") {
				return "after";
			}
			if (pos == "after") {
				return "before";
			}
		}

		// "If node A is an ancestor of node B:"
		if (documentPosition & $_.Node.DOCUMENT_POSITION_CONTAINS) {
			// "Let child equal node B."
			var child = nodeB;

			// "While child is not a child of node A, set child to its parent."
			while (child.parentNode != nodeA) {
				child = child.parentNode;
			}

			// "If the index of child is less than offset A, return after."
			if (getNodeIndex(child) < offsetA) {
				return "after";
			}
		}

		// "Return before."
		return "before";
	}

	/**
	 * Returns the furthest ancestor of a Node as defined by DOM Range.
	 */
	function getFurthestAncestor(node) {
		var root = node;
		while (root.parentNode != null) {
			root = root.parentNode;
		}
		return root;
	}

	/**
	 * "contained" as defined by DOM Range: "A Node node is contained in a range
	 * range if node's furthest ancestor is the same as range's root, and (node, 0)
	 * is after range's start, and (node, length of node) is before range's end."
	 */
	function isContained(node, range) {
		var pos1 = getPosition(node, 0, range.startContainer, range.startOffset);
		if (pos1 !== "after") {
			return false;
		}
		var pos2 = getPosition(node, getNodeLength(node), range.endContainer, range.endOffset);
		if (pos2 !== "before") {
			return false;
		}
		return getFurthestAncestor(node) == getFurthestAncestor(range.startContainer);
	}

	/**
	 * Return all nodes contained in range that the provided function returns true
	 * for, omitting any with an ancestor already being returned.
	 */
	function getContainedNodes(range, condition) {
		if (typeof condition == "undefined") {
			condition = function () {
				return true;
			};
		}
		var node = range.startContainer;
		if (node.hasChildNodes() && range.startOffset < node.childNodes.length) {
			// A child is contained
			node = node.childNodes[range.startOffset];
		} else if (range.startOffset == getNodeLength(node)) {
			// No descendant can be contained
			node = nextNodeDescendants(node);
		} else {
			// No children; this node at least can't be contained
			node = nextNode(node);
		}

		var stop = range.endContainer;
		if (stop.hasChildNodes() && range.endOffset < stop.childNodes.length) {
			// The node after the last contained node is a child
			stop = stop.childNodes[range.endOffset];
		} else {
			// This node and/or some of its children might be contained
			stop = nextNodeDescendants(stop);
		}

		var nodeList = [];
		while (isBefore(node, stop)) {
			if (isContained(node, range) && condition(node)) {
				nodeList.push(node);
				node = nextNodeDescendants(node);
				continue;
			}
			node = nextNode(node);
		}
		return nodeList;
	}

	/**
	 * As above, but includes nodes with an ancestor that's already been returned.
	 */
	function getAllContainedNodes(range, condition) {
		if (typeof condition == "undefined") {
			condition = function () {
				return true;
			};
		}
		var node = range.startContainer;
		if (node.hasChildNodes() && range.startOffset < node.childNodes.length) {
			// A child is contained
			node = node.childNodes[range.startOffset];
		} else if (range.startOffset == getNodeLength(node)) {
			// No descendant can be contained
			node = nextNodeDescendants(node);
		} else {
			// No children; this node at least can't be contained
			node = nextNode(node);
		}

		var stop = range.endContainer;
		if (stop.hasChildNodes() && range.endOffset < stop.childNodes.length) {
			// The node after the last contained node is a child
			stop = stop.childNodes[range.endOffset];
		} else {
			// This node and/or some of its children might be contained
			stop = nextNodeDescendants(stop);
		}

		var nodeList = [];
		while (isBefore(node, stop)) {
			if (isContained(node, range) && condition(node)) {
				nodeList.push(node);
			}
			node = nextNode(node);
		}
		return nodeList;
	}

	// Returns either null, or something of the form rgb(x, y, z), or something of
	// the form rgb(x, y, z, w) with w != 0.
	function normalizeColor(color) {
		if (color.toLowerCase() == "currentcolor") {
			return null;
		}

		var outerSpan = document.createElement("span");
		document.body.appendChild(outerSpan);
		outerSpan.style.color = "black";

		var innerSpan = document.createElement("span");
		outerSpan.appendChild(innerSpan);
		innerSpan.style.color = color;
		color = $_.getComputedStyle(innerSpan).color;

		if (color == "rgb(0, 0, 0)") {
			// Maybe it's really black, maybe it's invalid.
			outerSpan.color = "white";
			color = $_.getComputedStyle(innerSpan).color;
			if (color != "rgb(0, 0, 0)") {
				return null;
			}
		}

		document.body.removeChild(outerSpan);

		// I rely on the fact that browsers generally provide consistent syntax for
		// getComputedStyle(), although it's not standardized.  There are only two
		// exceptions I found:
		if (/^rgba\([0-9]+, [0-9]+, [0-9]+, 1\)$/.test(color)) {
			// IE10PP2 seems to do this sometimes.
			return color.replace("rgba", "rgb").replace(", 1)", ")");
		}
		if (color == "transparent") {
			// IE10PP2, Firefox 7.0a2, and Opera 11.50 all return "transparent" if
			// the specified value is "transparent".
			return "rgba(0, 0, 0, 0)";
		}
		return color;
	}

	// Returns either null, or something of the form #xxxxxx, or the color itself
	// if it's a valid keyword.
	function parseSimpleColor(color) {
		color = color.toLowerCase();
		if ($_(["aliceblue", "antiquewhite", "aqua", "aquamarine", "azure", "beige", "bisque", "black", "blanchedalmond", "blue", "blueviolet", "brown", "burlywood", "cadetblue", "chartreuse", "chocolate", "coral", "cornflowerblue", "cornsilk", "crimson", "cyan", "darkblue", "darkcyan", "darkgoldenrod", "darkgray", "darkgreen", "darkgrey", "darkkhaki", "darkmagenta", "darkolivegreen", "darkorange", "darkorchid", "darkred", "darksalmon", "darkseagreen", "darkslateblue", "darkslategray", "darkslategrey", "darkturquoise", "darkviolet", "deeppink", "deepskyblue", "dimgray", "dimgrey", "dodgerblue", "firebrick", "floralwhite", "forestgreen", "fuchsia", "gainsboro", "ghostwhite", "gold", "goldenrod", "gray", "green", "greenyellow", "grey", "honeydew", "hotpink", "indianred", "indigo", "ivory", "khaki", "lavender", "lavenderblush", "lawngreen", "lemonchiffon", "lightblue", "lightcoral", "lightcyan", "lightgoldenrodyellow", "lightgray", "lightgreen", "lightgrey", "lightpink", "lightsalmon", "lightseagreen", "lightskyblue", "lightslategray", "lightslategrey", "lightsteelblue", "lightyellow", "lime", "limegreen", "linen", "magenta", "maroon", "mediumaquamarine", "mediumblue", "mediumorchid", "mediumpurple", "mediumseagreen", "mediumslateblue", "mediumspringgreen", "mediumturquoise", "mediumvioletred", "midnightblue", "mintcream", "mistyrose", "moccasin", "navajowhite", "navy", "oldlace", "olive", "olivedrab", "orange", "orangered", "orchid", "palegoldenrod", "palegreen", "paleturquoise", "palevioletred", "papayawhip", "peachpuff", "peru", "pink", "plum", "powderblue", "purple", "red", "rosybrown", "royalblue", "saddlebrown", "salmon", "sandybrown", "seagreen", "seashell", "sienna", "silver", "skyblue", "slateblue", "slategray", "slategrey", "snow", "springgreen", "steelblue", "tan", "teal", "thistle", "tomato", "turquoise", "violet", "wheat", "white", "whitesmoke", "yellow", "yellowgreen"]).indexOf(color) != -1) {
			return color;
		}

		color = normalizeColor(color);
		var matches = /^rgb\(([0-9]+), ([0-9]+), ([0-9]+)\)$/.exec(color);
		if (matches) {
			return "#" + parseInt(matches[1], 10).toString(16).replace(/^.$/, "0$&") + parseInt(matches[2], 10).toString(16).replace(/^.$/, "0$&") + parseInt(matches[3], 10).toString(16).replace(/^.$/, "0$&");
		}
		return null;
	}

	//@}

	//////////////////////////////////////////////////////////////////////////////
	/////////////////////////// Edit command functions ///////////////////////////
	//////////////////////////////////////////////////////////////////////////////

	/////////////////////////////////////////////////
	///// Methods of the HTMLDocument interface /////
	/////////////////////////////////////////////////
	//@{

	var getStateOverride,
	    setStateOverride,
	    unsetStateOverride,
	    getValueOverride,
	    setValueOverride,
	    unsetValueOverride;

	var executionStackDepth = 0;

	// Helper function for fontSize's action plus queryOutputHelper.  It's just the
	// middle of fontSize's action, ripped out into its own function.
	function normalizeFontSize(value) {
		// "Strip leading and trailing whitespace from value."
		//
		// Cheap hack, not following the actual algorithm.
		value = $_(value).trim();

		// "If value is a valid floating point number, or would be a valid
		// floating point number if a single leading "+" character were
		// stripped:"
		if (/^[\-+]?[0-9]+(\.[0-9]+)?([eE][\-+]?[0-9]+)?$/.test(value)) {
			var mode;

			// "If the first character of value is "+", delete the character
			// and let mode be "relative-plus"."
			if (value[0] == "+") {
				value = value.slice(1);
				mode = "relative-plus";
				// "Otherwise, if the first character of value is "-", delete the
				// character and let mode be "relative-minus"."
			} else if (value[0] == "-") {
				value = value.slice(1);
				mode = "relative-minus";
				// "Otherwise, let mode be "absolute"."
			} else {
				mode = "absolute";
			}

			// "Apply the rules for parsing non-negative integers to value, and
			// let number be the result."
			//
			// Another cheap hack.
			var num = parseInt(value, 10);

			// "If mode is "relative-plus", add three to number."
			if (mode == "relative-plus") {
				num += 3;
			}

			// "If mode is "relative-minus", negate number, then add three to
			// it."
			if (mode == "relative-minus") {
				num = 3 - num;
			}

			// "If number is less than one, let number equal 1."
			if (num < 1) {
				num = 1;
			}

			// "If number is greater than seven, let number equal 7."
			if (num > 7) {
				num = 7;
			}

			// "Set value to the string here corresponding to number:" [table
			// omitted]
			value = {
				1: "xx-small",
				2: "small",
				3: "medium",
				4: "large",
				5: "x-large",
				6: "xx-large",
				7: "xxx-large"
			}[num];
		}

		return value;
	}

	function getLegacyFontSize(size) {
		// For convenience in other places in my code, I handle all sizes, not just
		// pixel sizes as the spec says.  This means pixel sizes have to be passed
		// in suffixed with "px", not as plain numbers.
		size = normalizeFontSize(size);

		if (jQuery.inArray(size, ["xx-small", "x-small", "small", "medium", "large", "x-large", "xx-large", "xxx-large"]) == -1 && !/^[0-9]+(\.[0-9]+)?(cm|mm|in|pt|pc|px)$/.test(size)) {
			// There is no sensible legacy size for things like "2em".
			return null;
		}

		var font = document.createElement("font");
		document.body.appendChild(font);
		if (size == "xxx-large") {
			font.size = 7;
		} else {
			font.style.fontSize = size;
		}
		var pixelSize = parseInt($_.getComputedStyle(font).fontSize, 10);
		document.body.removeChild(font);

		// "Let returned size be 1."
		var returnedSize = 1;

		// "While returned size is less than 7:"
		while (returnedSize < 7) {
			// "Let lower bound be the resolved value of "font-size" in pixels
			// of a font element whose size attribute is set to returned size."
			font = document.createElement("font");
			font.size = returnedSize;
			document.body.appendChild(font);
			var lowerBound = parseInt($_.getComputedStyle(font).fontSize, 10);

			// "Let upper bound be the resolved value of "font-size" in pixels
			// of a font element whose size attribute is set to one plus
			// returned size."
			font.size = 1 + returnedSize;
			var upperBound = parseInt($_.getComputedStyle(font).fontSize, 10);
			document.body.removeChild(font);

			// "Let average be the average of upper bound and lower bound."
			var average = (upperBound + lowerBound) / 2;

			// "If pixel size is less than average, return the one-element
			// string consisting of the digit returned size."
			if (pixelSize < average) {
				return String(returnedSize);
			}

			// "Add one to returned size."
			returnedSize++;
		}

		// "Return "7"."
		return "7";
	}

	// Helper function for common behavior.
	function editCommandMethod(command, prop, range, callback) {
		var ret;

		// Set up our global range magic, but only if we're the outermost function
		if (executionStackDepth == 0 && typeof range != "undefined") {
			globalRange = range;
		} else if (executionStackDepth == 0) {
			globalRange = null;
			globalRange = range;
		}

		// "If command is not supported, raise a NOT_SUPPORTED_ERR exception."
		//
		// We can't throw a real one, but a string will do for our purposes.
		if (!commands.hasOwnProperty(command)) {
			throw "NOT_SUPPORTED_ERR";
		}

		// "If command has no action, raise an INVALID_ACCESS_ERR exception."
		// "If command has no indeterminacy, raise an INVALID_ACCESS_ERR
		// exception."
		// "If command has no state, raise an INVALID_ACCESS_ERR exception."
		// "If command has no value, raise an INVALID_ACCESS_ERR exception."
		if (prop != "enabled" && !commands[command].hasOwnProperty(prop)) {
			throw "INVALID_ACCESS_ERR";
		}

		executionStackDepth++;
		try {
			ret = callback();
		} catch (e) {
			executionStackDepth--;
			throw e;
		}
		executionStackDepth--;
		return ret;
	}

	function myQueryCommandEnabled(command, range) {
		// "All of these methods must treat their command argument ASCII
		// case-insensitively."
		command = command.toLowerCase();

		// "If command is not supported, raise a NOT_SUPPORTED_ERR exception."
		return editCommandMethod(command, "action", range, (function (command) {
			return function () {
				// "Among commands defined in this specification, those listed in
				// Miscellaneous commands are always enabled. The other commands defined
				// here are enabled if the active range is not null, and disabled
				// otherwise."
				return jQuery.inArray(command, ["copy", "cut", "paste", "selectall", "stylewithcss", "usecss"]) != -1 || range !== null;
			};
		}(command)));
	}

	function setActiveRange(range) {
		var rangeObject = new window.GENTICS.Utils.RangeObject();

		rangeObject.startContainer = range.startContainer;
		rangeObject.startOffset = range.startOffset;
		rangeObject.endContainer = range.endContainer;
		rangeObject.endOffset = range.endOffset;

		rangeObject.select();
	}

	function myExecCommand(commandArg, showUiArg, valueArg, range) {
		// "All of these methods must treat their command argument ASCII
		// case-insensitively."
		var command = commandArg.toLowerCase();
		var showUi = showUiArg;
		var value = valueArg;

		// "If only one argument was provided, let show UI be false."
		//
		// If range was passed, I can't actually detect how many args were passed
		// . . .
		if (arguments.length == 1 || (arguments.length >= 4 && typeof showUi == "undefined")) {
			showUi = false;
		}

		// "If only one or two arguments were provided, let value be the empty
		// string."
		if (arguments.length <= 2 || (arguments.length >= 4 && typeof value == "undefined")) {
			value = "";
		}

		// "If command is not supported, raise a NOT_SUPPORTED_ERR exception."
		//
		// "If command has no action, raise an INVALID_ACCESS_ERR exception."
		return editCommandMethod(command, "action", range, (function (command, showUi, value) {
			return function () {
				// "If command is not enabled, return false."
				if (!myQueryCommandEnabled(command)) {
					return false;
				}

				// "Take the action for command, passing value to the instructions as an
				// argument."
				commands[command].action(value, range);

				// always fix the range after the command is complete
				setActiveRange(range);

				// "Return true."
				return true;
			};
		}(command, showUi, value)));
	}

	function myQueryCommandIndeterm(command, range) {
		// "All of these methods must treat their command argument ASCII
		// case-insensitively."
		command = command.toLowerCase();

		// "If command is not supported, raise a NOT_SUPPORTED_ERR exception."
		//
		// "If command has no indeterminacy, raise an INVALID_ACCESS_ERR
		// exception."
		return editCommandMethod(command, "indeterm", range, (function (command) {
			return function () {
				// "If command is not enabled, return false."
				if (!myQueryCommandEnabled(command, range)) {
					return false;
				}

				// "Return true if command is indeterminate, otherwise false."
				return commands[command].indeterm(range);
			};
		}(command)));
	}

	function myQueryCommandState(command, range) {
		// "All of these methods must treat their command argument ASCII
		// case-insensitively."
		command = command.toLowerCase();

		// "If command is not supported, raise a NOT_SUPPORTED_ERR exception."
		//
		// "If command has no state, raise an INVALID_ACCESS_ERR exception."
		return editCommandMethod(command, "state", range, (function (command) {
			return function () {
				// "If command is not enabled, return false."
				if (!myQueryCommandEnabled(command, range)) {
					return false;
				}

				// "If the state override for command is set, return it."
				if (typeof getStateOverride(command, range) != "undefined") {
					return getStateOverride(command, range);
				}

				// "Return true if command's state is true, otherwise false."
				return commands[command].state(range);
			};
		}(command)));
	}

	// "When the queryCommandSupported(command) method on the HTMLDocument
	// interface is invoked, the user agent must return true if command is
	// supported, and false otherwise."
	function myQueryCommandSupported(command) {
		// "All of these methods must treat their command argument ASCII
		// case-insensitively."
		command = command.toLowerCase();

		return commands.hasOwnProperty(command);
	}

	function myQueryCommandValue(command, range) {
		// "All of these methods must treat their command argument ASCII
		// case-insensitively."
		command = command.toLowerCase();

		// "If command is not supported, raise a NOT_SUPPORTED_ERR exception."
		//
		// "If command has no value, raise an INVALID_ACCESS_ERR exception."
		return editCommandMethod(command, "value", range, function () {
			// "If command is not enabled, return the empty string."
			if (!myQueryCommandEnabled(command, range)) {
				return "";
			}

			// "If command is "fontSize" and its value override is set, convert the
			// value override to an integer number of pixels and return the legacy
			// font size for the result."
			if (command == "fontsize" && getValueOverride("fontsize", range) !== undefined) {
				return getLegacyFontSize(getValueOverride("fontsize", range));
			}

			// "If the value override for command is set, return it."
			if (typeof getValueOverride(command, range) != "undefined") {
				return getValueOverride(command, range);
			}

			// "Return command's value."
			return commands[command].value(range);
		});
	}
	//@}

	//////////////////////////////
	///// Common definitions /////
	//////////////////////////////
	//@{

	// "A prohibited paragraph child name is "address", "article", "aside",
	// "blockquote", "caption", "center", "col", "colgroup", "dd", "details",
	// "dir", "div", "dl", "dt", "fieldset", "figcaption", "figure", "footer",
	// "form", "h1", "h2", "h3", "h4", "h5", "h6", "header", "hgroup", "hr", "li",
	// "listing", "menu", "nav", "ol", "p", "plaintext", "pre", "section",
	// "summary", "table", "tbody", "td", "tfoot", "th", "thead", "tr", "ul", or
	// "xmp"."
	var prohibitedParagraphChildNamesMap = {
		"ADDRESS": true,
		"ARTICLE": true,
		"ASIDE": true,
		"BLOCKQUOTE": true,
		"CAPTION": true,
		"CENTER": true,
		"COL": true,
		"COLGROUP": true,
		"DD": true,
		"DETAILS": true,
		"DIR": true,
		"DIV": true,
		"DL": true,
		"DT": true,
		"FIELDSET": true,
		"FIGCAPTION": true,
		"FIGURE": true,
		"FOOTER": true,
		"FORM": true,
		"H1": true,
		"H2": true,
		"H3": true,
		"H4": true,
		"H5": true,
		"H6": true,
		"HEADER": true,
		"HGROUP": true,
		"HR": true,
		"LI": true,
		"LISTING": true,
		"MENU": true,
		"NAV": true,
		"OL": true,
		"P": true,
		"PLAINTEXT": true,
		"PRE": true,
		"SECTION": true,
		"SUMMARY": true,
		"TABLE": true,
		"TBODY": true,
		"TD": true,
		"TFOOT": true,
		"TH": true,
		"THEAD": true,
		"TR": true,
		"UL": true,
		"XMP": true
	};

	// "A prohibited paragraph child is an HTML element whose local name is a
	// prohibited paragraph child name."
	function isProhibitedParagraphChild(node) {
		return isMappedHtmlElement(node, prohibitedParagraphChildNamesMap);
	}

	var nonBlockDisplayValuesMap = {
		"inline": true,
		"inline-block": true,
		"inline-table": true,
		"none": true
	};

	// "A block node is either an Element whose "display" property does not have
	// resolved value "inline" or "inline-block" or "inline-table" or "none", or a
	// Document, or a DocumentFragment."
	function isBlockNode(node) {
		return node && ((node.nodeType == $_.Node.ELEMENT_NODE && !nonBlockDisplayValuesMap[$_.getComputedStyle(node).display]) || node.nodeType == $_.Node.DOCUMENT_NODE || node.nodeType == $_.Node.DOCUMENT_FRAGMENT_NODE);
	}

	// "An inline node is a node that is not a block node."
	function isInlineNode(node) {
		return node && !isBlockNode(node);
	}

	// "An editing host is a node that is either an Element with a contenteditable
	// attribute set to the true state, or the Element child of a Document whose
	// designMode is enabled."
	function isEditingHost(node) {
		return node && node.nodeType == $_.Node.ELEMENT_NODE && (node.contentEditable == "true" || (node.parentNode && node.parentNode.nodeType == $_.Node.DOCUMENT_NODE && node.parentNode.designMode == "on"));
	}

	// "Something is editable if it is a node which is not an editing host, does
	// not have a contenteditable attribute set to the false state, and whose
	// parent is an editing host or editable."
	function isEditable(node) {
		// This is slightly a lie, because we're excluding non-HTML elements with
		// contentEditable attributes.
		return node && !isEditingHost(node) && (node.nodeType != $_.Node.ELEMENT_NODE || node.contentEditable != "false" || jQuery(node).hasClass('aloha-table-wrapper')) && (isEditingHost(node.parentNode) || isEditable(node.parentNode));
	}

	// Helper function, not defined in the spec
	function hasEditableDescendants(node) {
		var i;
		for (i = 0; i < node.childNodes.length; i++) {
			if (isEditable(node.childNodes[i]) || hasEditableDescendants(node.childNodes[i])) {
				return true;
			}
		}
		return false;
	}

	// "The editing host of node is null if node is neither editable nor an editing
	// host; node itself, if node is an editing host; or the nearest ancestor of
	// node that is an editing host, if node is editable."
	function getEditingHostOf(node) {
		if (isEditingHost(node)) {
			return node;
		}
		if (isEditable(node)) {
			var ancestor = node.parentNode;
			while (!isEditingHost(ancestor)) {
				ancestor = ancestor.parentNode;
			}
			return ancestor;
		}
		return null;
	}

	// "Two nodes are in the same editing host if the editing host of the first is
	// non-null and the same as the editing host of the second."
	function inSameEditingHost(node1, node2) {
		return getEditingHostOf(node1) && getEditingHostOf(node1) == getEditingHostOf(node2);
	}

	// "A collapsed line break is a br that begins a line box which has nothing
	// else in it, and therefore has zero height."
	function isCollapsedLineBreak(br) {
		if (!isNamedHtmlElement(br, 'br')) {
			return false;
		}

		// Add a zwsp after it and see if that changes the height of the nearest
		// non-inline parent.  Note: this is not actually reliable, because the
		// parent might have a fixed height or something.
		var ref = br.parentNode;
		while ($_.getComputedStyle(ref).display == "inline") {
			ref = ref.parentNode;
		}

		var origStyle = {
			height: ref.style.height,
			maxHeight: ref.style.maxHeight,
			minHeight: ref.style.minHeight
		};

		ref.style.height = 'auto';
		ref.style.maxHeight = 'none';
		if (!(jQuery.browser.msie && jQuery.browser.version < 8)) {
			ref.style.minHeight = '0';
		}
		var space = document.createTextNode('\u200b');
		var origHeight = ref.offsetHeight;
		if (origHeight == 0) {
			throw 'isCollapsedLineBreak: original height is zero, bug?';
		}
		br.parentNode.insertBefore(space, br.nextSibling);
		var finalHeight = ref.offsetHeight;
		space.parentNode.removeChild(space);

		ref.style.height = origStyle.height;
		ref.style.maxHeight = origStyle.maxHeight;
		if (!(jQuery.browser.msie && jQuery.browser.version < 8)) {
			ref.style.minHeight = origStyle.minHeight;
		}

		// Allow some leeway in case the zwsp didn't create a whole new line, but
		// only made an existing line slightly higher.  Firefox 6.0a2 shows this
		// behavior when the first line is bold.
		return origHeight < finalHeight - 5;
	}

	// "An extraneous line break is a br that has no visual effect, in that
	// removing it from the DOM would not change layout, except that a br that is
	// the sole child of an li is not extraneous."
	function isExtraneousLineBreak(br) {

		if (!isNamedHtmlElement(br, 'br')) {
			return false;
		}

		if (isNamedHtmlElement(br.parentNode, "li") && br.parentNode.childNodes.length == 1) {
			return false;
		}

		// Make the line break disappear and see if that changes the block's
		// height.  Yes, this is an absurd hack.  We have to reset height etc. on
		// the reference node because otherwise its height won't change if it's not
		// auto.
		var ref = br.parentNode;
		while ($_.getComputedStyle(ref).display == "inline") {
			ref = ref.parentNode;
		}

		var origStyle = {
			height: ref.style.height,
			maxHeight: ref.style.maxHeight,
			minHeight: ref.style.minHeight,
			contentEditable: ref.contentEditable
		};

		ref.style.height = 'auto';
		ref.style.maxHeight = 'none';
		ref.style.minHeight = '0';
		// IE7 would ignore display:none in contentEditable, so we temporarily set it to false
		if (jQuery.browser.msie && jQuery.browser.version <= 7) {
			ref.contentEditable = 'false';
		}

		var origHeight = ref.offsetHeight;
		if (origHeight == 0) {
			throw "isExtraneousLineBreak: original height is zero, bug?";
		}

		var origBrDisplay = br.style.display;
		br.style.display = 'none';
		var finalHeight = ref.offsetHeight;

		// Restore original styles to the touched elements.
		ref.style.height = origStyle.height;
		ref.style.maxHeight = origStyle.maxHeight;
		ref.style.minHeight = origStyle.minHeight;
		// reset contentEditable for IE7
		if (jQuery.browser.msie && jQuery.browser.version <= 7) {
			ref.contentEditable = origStyle.contentEditable;
		}
		br.style.display = origBrDisplay;

		// https://github.com/alohaeditor/Aloha-Editor/issues/516
		// look like it works in msie > 7
		/* if (jQuery.browser.msie && jQuery.browser.version < 8) {
		br.removeAttribute("style");
		ref.removeAttribute("style");
	} */

		return origHeight == finalHeight;
	}

	// "A whitespace node is either a Text node whose data is the empty string; or
	// a Text node whose data consists only of one or more tabs (0x0009), line
	// feeds (0x000A), carriage returns (0x000D), and/or spaces (0x0020), and whose
	// parent is an Element whose resolved value for "white-space" is "normal" or
	// "nowrap"; or a Text node whose data consists only of one or more tabs
	// (0x0009), carriage returns (0x000D), and/or spaces (0x0020), and whose
	// parent is an Element whose resolved value for "white-space" is "pre-line"."
	function isWhitespaceNode(node) {
		return node && node.nodeType == $_.Node.TEXT_NODE && (node.data == "" || (/^[\t\n\r ]+$/.test(node.data) && node.parentNode && node.parentNode.nodeType == $_.Node.ELEMENT_NODE && jQuery.inArray($_.getComputedStyle(node.parentNode).whiteSpace, ["normal", "nowrap"]) != -1) || (/^[\t\r ]+$/.test(node.data) && node.parentNode && node.parentNode.nodeType == $_.Node.ELEMENT_NODE && $_.getComputedStyle(node.parentNode).whiteSpace == "pre-line") || (/^[\t\n\r ]+$/.test(node.data) && node.parentNode && node.parentNode.nodeType == $_.Node.DOCUMENT_FRAGMENT_NODE));
	}

	/**
	 * Collapse sequences of ignorable whitespace (tab (0x0009), line feed (0x000A), carriage return (0x000D), space (0x0020)) to only one space.
	 * Preserve the given range if necessary.
	 * @param node text node
	 * @param range range
	 */
	function collapseWhitespace(node, range) {
		// "If node is neither editable nor an editing host, abort these steps."
		if (!isEditable(node) && !isEditingHost(node)) {
			return;
		}

		// if the given node is not a text node, return
		if (!node || node.nodeType !== $_.Node.TEXT_NODE) {
			return;
		}

		// if the node is in a pre or pre-wrap node, return
		if (jQuery.inArray($_.getComputedStyle(node.parentNode).whiteSpace, ["pre", "pre-wrap"]) != -1) {
			return;
		}

		// if the given node does not contain sequences of at least two consecutive ignorable whitespace characters, return
		if (!/[\t\n\r ]{2,}/.test(node.data)) {
			return;
		}

		var newData = '';
		var correctStart = range.startContainer == node;
		var correctEnd = range.endContainer == node;
		var wsFound = false;
		var i;

		// iterate through the node data
		for (i = 0; i < node.data.length; ++i) {
			if (/[\t\n\r ]/.test(node.data.substr(i, 1))) {
				// found a whitespace
				if (!wsFound) {
					// this is the first whitespace in the current sequence
					// add a whitespace to the new data sequence
					newData += ' ';
					// remember that we found a whitespace
					wsFound = true;
				} else {
					// this is not the first whitespace in the sequence, so omit this character
					if (correctStart && newData.length < range.startOffset) {
						range.startOffset--;
					}
					if (correctEnd && newData.length < range.endOffset) {
						range.endOffset--;
					}
				}
			} else {
				newData += node.data.substr(i, 1);
				wsFound = false;
			}
		}

		// set the new data
		node.data = newData;
	}

	// "node is a collapsed whitespace node if the following algorithm returns
	// true:"
	function isCollapsedWhitespaceNode(node) {
		// "If node is not a whitespace node, return false."
		if (!isWhitespaceNode(node)) {
			return false;
		}

		// "If node's data is the empty string, return true."
		if (node.data == "") {
			return true;
		}

		// "Let ancestor be node's parent."
		var ancestor = node.parentNode;

		// "If ancestor is null, return true."
		if (!ancestor) {
			return true;
		}

		// "If the "display" property of some ancestor of node has resolved value
		// "none", return true."
		if ($_(getAncestors(node)).some(function (ancestor) { return ancestor.nodeType == $_.Node.ELEMENT_NODE && $_.getComputedStyle(ancestor).display == "none"; })) {
			return true;
		}

		// "While ancestor is not a block node and its parent is not null, set
		// ancestor to its parent."
		while (!isBlockNode(ancestor) && ancestor.parentNode) {
			ancestor = ancestor.parentNode;
		}

		// "Let reference be node."
		var reference = node;

		// "While reference is a descendant of ancestor:"
		while (reference != ancestor) {
			// "Let reference be the node before it in tree order."
			reference = previousNode(reference);

			// "If reference is a block node or a br, return true."
			if (isBlockNode(reference) || isNamedHtmlElement(reference, 'br')) {
				return true;
			}

			// "If reference is a Text node that is not a whitespace node, or is an
			// img, break from this loop."
			if ((reference.nodeType == $_.Node.TEXT_NODE && !isWhitespaceNode(reference)) || isNamedHtmlElement(reference, 'img')) {
				break;
			}
		}

		// "Let reference be node."
		reference = node;

		// "While reference is a descendant of ancestor:"
		var stop = nextNodeDescendants(ancestor);
		while (reference != stop) {
			// "Let reference be the node after it in tree order, or null if there
			// is no such node."
			reference = nextNode(reference);

			// "If reference is a block node or a br, return true."
			if (isBlockNode(reference) || isNamedHtmlElement(reference, 'br')) {
				return true;
			}

			// "If reference is a Text node that is not a whitespace node, or is an
			// img, break from this loop."
			if ((reference && reference.nodeType == $_.Node.TEXT_NODE && !isWhitespaceNode(reference)) || isNamedHtmlElement(reference, 'img')) {
				break;
			}
		}

		// "Return false."
		return false;
	}

	// "Something is visible if it is a node that either is a block node, or a Text
	// node that is not a collapsed whitespace node, or an img, or a br that is not
	// an extraneous line break, or any node with a visible descendant; excluding
	// any node with an ancestor container Element whose "display" property has
	// resolved value "none"."
	function isVisible(node) {
		var i;

		if (!node) {
			return false;
		}

		if ($_(getAncestors(node).concat(node))
			    .filter(function (node) { return node.nodeType == $_.Node.ELEMENT_NODE; }, true)
			    .some(function (node) { return $_.getComputedStyle(node).display == "none"; })) {
			return false;
		}

		if (isBlockNode(node) || (node.nodeType == $_.Node.TEXT_NODE && !isCollapsedWhitespaceNode(node)) || isNamedHtmlElement(node, 'img') || (isNamedHtmlElement(node, 'br') && !isExtraneousLineBreak(node))) {
			return true;
		}

		for (i = 0; i < node.childNodes.length; i++) {
			if (isVisible(node.childNodes[i])) {
				return true;
			}
		}

		return false;
	}

	// "Something is invisible if it is a node that is not visible."
	function isInvisible(node) {
		return node && !isVisible(node);
	}

	// "A collapsed block prop is either a collapsed line break that is not an
	// extraneous line break, or an Element that is an inline node and whose
	// children are all either invisible or collapsed block props and that has at
	// least one child that is a collapsed block prop."
	function isCollapsedBlockProp(node) {
		var i;

		if (isCollapsedLineBreak(node) && !isExtraneousLineBreak(node)) {
			return true;
		}

		if (!isInlineNode(node) || node.nodeType != $_.Node.ELEMENT_NODE) {
			return false;
		}

		var hasCollapsedBlockPropChild = false;
		for (i = 0; i < node.childNodes.length; i++) {
			if (!isInvisible(node.childNodes[i]) && !isCollapsedBlockProp(node.childNodes[i])) {
				return false;
			}
			if (isCollapsedBlockProp(node.childNodes[i])) {
				hasCollapsedBlockPropChild = true;
			}
		}

		return hasCollapsedBlockPropChild;
	}

	// Please note: This method is deprecated and will be removed. 
	// Every command should use the value and range parameter. 
	// 
	// "The active range is the first range in the Selection given by calling
	// getSelection() on the context object, or null if there is no such range."
	//
	// We cheat and return globalRange if that's defined.  We also ensure that the
	// active range meets the requirements that selection boundary points are
	// supposed to meet, i.e., that the nodes are both Text or Element nodes that
	// descend from a Document.
	function getActiveRange() {
		var ret;
		if (globalRange) {
			ret = globalRange;
		} else if (Aloha.getSelection().rangeCount) {
			ret = Aloha.getSelection().getRangeAt(0);
		} else {
			return null;
		}
		if (jQuery.inArray(ret.startContainer.nodeType, [$_.Node.TEXT_NODE, $_.Node.ELEMENT_NODE]) == -1 || jQuery.inArray(ret.endContainer.nodeType, [$_.Node.TEXT_NODE, $_.Node.ELEMENT_NODE]) == -1 || !ret.startContainer.ownerDocument || !ret.endContainer.ownerDocument || !isDescendant(ret.startContainer, ret.startContainer.ownerDocument) || !isDescendant(ret.endContainer, ret.endContainer.ownerDocument)) {
			throw "Invalid active range; test bug?";
		}
		return ret;
	}

	// "For some commands, each HTMLDocument must have a boolean state override
	// and/or a string value override. These do not change the command's state or
	// value, but change the way some algorithms behave, as specified in those
	// algorithms' definitions. Initially, both must be unset for every command.
	// Whenever the number of ranges in the Selection changes to something
	// different, and whenever a boundary point of the range at a given index in
	// the Selection changes to something different, the state override and value
	// override must be unset for every command."
	//
	// We implement this crudely by using setters and getters.  To verify that the
	// selection hasn't changed, we copy the active range and just check the
	// endpoints match.  This isn't really correct, but it's good enough for us.
	// Unset state/value overrides are undefined.  We put everything in a function
	// so no one can access anything except via the provided functions, since
	// otherwise callers might mistakenly use outdated overrides (if the selection
	// has changed).
	(function () {
		var stateOverrides = {};
		var valueOverrides = {};
		var storedRange = null;

		function resetOverrides(range) {
			if (!storedRange || storedRange.startContainer != range.startContainer || storedRange.endContainer != range.endContainer || storedRange.startOffset != range.startOffset || storedRange.endOffset != range.endOffset) {
				stateOverrides = {};
				valueOverrides = {};
				storedRange = range.cloneRange();
			}
		}

		getStateOverride = function (command, range) {
			resetOverrides(range);
			return stateOverrides[command];
		};

		setStateOverride = function (command, newState, range) {
			resetOverrides(range);
			stateOverrides[command] = newState;
		};

		unsetStateOverride = function (command, range) {
			resetOverrides(range);
			delete stateOverrides[command];
		};

		getValueOverride = function (command, range) {
			resetOverrides(range);
			return valueOverrides[command];
		};

		// "The value override for the backColor command must be the same as the
		// value override for the hiliteColor command, such that setting one sets
		// the other to the same thing and unsetting one unsets the other."
		setValueOverride = function (command, newValue, range) {
			resetOverrides(range);
			valueOverrides[command] = newValue;
			if (command == "backcolor") {
				valueOverrides.hilitecolor = newValue;
			} else if (command == "hilitecolor") {
				valueOverrides.backcolor = newValue;
			}
		};

		unsetValueOverride = function (command, range) {
			resetOverrides(range);
			delete valueOverrides[command];
			if (command == "backcolor") {
				delete valueOverrides.hilitecolor;
			} else if (command == "hilitecolor") {
				delete valueOverrides.backcolor;
			}
		};
	}());

	//@}

	/////////////////////////////
	///// Common algorithms /////
	/////////////////////////////

	///// Assorted common algorithms /////
	//@{

	function movePreservingRanges(node, newParent, newIndex, range) {
		// For convenience, I allow newIndex to be -1 to mean "insert at the end".
		if (newIndex == -1) {
			newIndex = newParent.childNodes.length;
		}

		// "When the user agent is to move a Node to a new location, preserving
		// ranges, it must remove the Node from its original parent (if any), then
		// insert it in the new location. In doing so, however, it must ignore the
		// regular range mutation rules, and instead follow these rules:"

		// "Let node be the moved Node, old parent and old index be the old parent
		// (which may be null) and index, and new parent and new index be the new
		// parent and index."
		var oldParent = node.parentNode;
		var oldIndex = getNodeIndex(node);
		var i;

		// We only even attempt to preserve the global range object and the ranges
		// in the selection, not every range out there (the latter is probably
		// impossible).
		var ranges = [range];
		for (i = 0; i < Aloha.getSelection().rangeCount; i++) {
			ranges.push(Aloha.getSelection().getRangeAt(i));
		}
		var boundaryPoints = [];
		$_(ranges).forEach(function (range) {
			boundaryPoints.push([range.startContainer, range.startOffset]);
			boundaryPoints.push([range.endContainer, range.endOffset]);
		});

		$_(boundaryPoints).forEach(function (boundaryPoint) {
			// "If a boundary point's node is the same as or a descendant of node,
			// leave it unchanged, so it moves to the new location."
			//
			// No modifications necessary.

			// "If a boundary point's node is new parent and its offset is greater
			// than new index, add one to its offset."
			if (boundaryPoint[0] == newParent && boundaryPoint[1] > newIndex) {
				boundaryPoint[1]++;
			}

			// "If a boundary point's node is old parent and its offset is old index or
			// old index + 1, set its node to new parent and add new index − old index
			// to its offset."
			if (boundaryPoint[0] == oldParent && (boundaryPoint[1] == oldIndex || boundaryPoint[1] == oldIndex + 1)) {
				boundaryPoint[0] = newParent;
				boundaryPoint[1] += newIndex - oldIndex;
			}

			// "If a boundary point's node is old parent and its offset is greater than
			// old index + 1, subtract one from its offset."
			if (boundaryPoint[0] == oldParent && boundaryPoint[1] > oldIndex + 1) {
				boundaryPoint[1]--;
			}
		});

		// Now actually move it and preserve the ranges.
		if (newParent.childNodes.length == newIndex) {
			newParent.appendChild(node);
		} else {
			newParent.insertBefore(node, newParent.childNodes[newIndex]);
		}

		// if we're off actual node boundaries this implies that the move was
		// part of a deletion process (backspace). If that's the case we 
		// attempt to fix this by restoring the range to the first index of
		// the node that has been moved
		var newRange = null;
		if (boundaryPoints[0][1] > boundaryPoints[0][0].childNodes.length && boundaryPoints[1][1] > boundaryPoints[1][0].childNodes.length) {
			range.setStart(node, 0);
			range.setEnd(node, 0);
		} else {
			range.setStart(boundaryPoints[0][0], boundaryPoints[0][1]);
			range.setEnd(boundaryPoints[1][0], boundaryPoints[1][1]);

			Aloha.getSelection().removeAllRanges();
			for (i = 1; i < ranges.length; i++) {
				newRange = Aloha.createRange();
				newRange.setStart(boundaryPoints[2 * i][0], boundaryPoints[2 * i][1]);
				newRange.setEnd(boundaryPoints[2 * i + 1][0], boundaryPoints[2 * i + 1][1]);
				Aloha.getSelection().addRange(newRange);
			}
			if (newRange) {
				range = newRange;
			}
		}
	}

	/**
	 * Copy all non empty attributes from an existing to a new element
	 * 
	 * @param {dom} element The source DOM element
	 * @param {dom} newElement The new DOM element which will get the attributes of the source DOM element
	 * @return void
	 */
	function copyAttributes(element, newElement) {

		// This is an IE7 workaround. We identified three places that were connected 
		// to the mysterious ie7 crash:
		// 1. Add attribute to dom element (Initialization of jquery-ui sortable)
		// 2. Access the jquery expando attribute. Just reading the name is 
		//    sufficient to make the browser vulnerable for the crash (Press enter)
		// 3. On editable blur the Aloha.editables[0].getContents(); gets invoked.
		//    This invokation somehow crashes the ie7. We assume that the access of 
		//    shared expando attribute updates internal references which are not 
		//    correclty handled during clone(); 
		if (jQuery.browser.msie && jQuery.browser.version >= 7 && typeof element.attributes[jQuery.expando] !== 'undefined') {
			jQuery(element).removeAttr(jQuery.expando);
		}

		var attrs = element.attributes;
		var i;
		for (i = 0; i < attrs.length; i++) {
			var attr = attrs[i];
			// attr.specified is an IE specific check to exclude attributes that were never really set.
			if (typeof attr.specified === "undefined" || attr.specified) {
				if (typeof newElement.setAttributeNS === 'function') {
					newElement.setAttributeNS(attr.namespaceURI, attr.name, attr.value);
				} else {
					// fixes https://github.com/alohaeditor/Aloha-Editor/issues/515 
					newElement.setAttribute(attr.name, attr.value);
				}
			}
		}
	}

	function setTagName(element, newName, range) {
		// "If element is an HTML element with local name equal to new name, return
		// element."
		if (isNamedHtmlElement(element, newName)) {
			return element;
		}

		// "If element's parent is null, return element."
		if (!element.parentNode) {
			return element;
		}

		// "Let replacement element be the result of calling createElement(new
		// name) on the ownerDocument of element."
		var replacementElement = element.ownerDocument.createElement(newName);

		// "Insert replacement element into element's parent immediately before
		// element."
		element.parentNode.insertBefore(replacementElement, element);

		// "Copy all attributes of element to replacement element, in order."
		copyAttributes(element, replacementElement);

		// "While element has children, append the first child of element as the
		// last child of replacement element, preserving ranges."
		while (element.childNodes.length) {
			movePreservingRanges(element.firstChild, replacementElement, replacementElement.childNodes.length, range);
		}

		// "Remove element from its parent."
		element.parentNode.removeChild(element);

		// if the range still uses the old element, we modify it to the new one
		if (range.startContainer === element) {
			range.startContainer = replacementElement;
		}
		if (range.endContainer === element) {
			range.endContainer = replacementElement;
		}

		// "Return replacement element."
		return replacementElement;
	}

	function removeExtraneousLineBreaksBefore(node) {
		// "Let ref be the previousSibling of node."
		var ref = node.previousSibling;

		// "If ref is null, abort these steps."
		if (!ref) {
			return;
		}

		// "While ref has children, set ref to its lastChild."
		while (ref.hasChildNodes()) {
			ref = ref.lastChild;
		}

		// "While ref is invisible but not an extraneous line break, and ref does
		// not equal node's parent, set ref to the node before it in tree order."
		while (isInvisible(ref) && !isExtraneousLineBreak(ref) && ref != node.parentNode) {
			ref = previousNode(ref);
		}

		// "If ref is an editable extraneous line break, remove it from its
		// parent."
		if (isEditable(ref) && isExtraneousLineBreak(ref)) {
			ref.parentNode.removeChild(ref);
		}
	}

	function removeExtraneousLineBreaksAtTheEndOf(node) {
		// "Let ref be node."
		var ref = node;

		// "While ref has children, set ref to its lastChild."
		while (ref.hasChildNodes()) {
			ref = ref.lastChild;
		}

		// "While ref is invisible but not an extraneous line break, and ref does
		// not equal node, set ref to the node before it in tree order."
		while (isInvisible(ref) && !isExtraneousLineBreak(ref) && ref != node) {
			ref = previousNode(ref);
		}

		// "If ref is an editable extraneous line break, remove it from its
		// parent."
		if (isEditable(ref) && isExtraneousLineBreak(ref)) {
			ref.parentNode.removeChild(ref);
		}
	}

	// "To remove extraneous line breaks from a node, first remove extraneous line
	// breaks before it, then remove extraneous line breaks at the end of it."
	function removeExtraneousLineBreaksFrom(node) {
		removeExtraneousLineBreaksBefore(node);
		removeExtraneousLineBreaksAtTheEndOf(node);
	}

	//@}
	///// Wrapping a list of nodes /////
	//@{

	function wrap(nodeList, siblingCriteria, newParentInstructions, range) {
		var i;

		// "If not provided, sibling criteria returns false and new parent
		// instructions returns null."
		if (typeof siblingCriteria == "undefined") {
			siblingCriteria = function () {
				return false;
			};
		}
		if (typeof newParentInstructions == "undefined") {
			newParentInstructions = function () {
				return null;
			};
		}

		// "If node list is empty, or the first member of node list is not
		// editable, return null and abort these steps."
		if (!nodeList.length || !isEditable(nodeList[0])) {
			return null;
		}

		// "If node list's last member is an inline node that's not a br, and node
		// list's last member's nextSibling is a br, append that br to node list."
		if (isInlineNode(nodeList[nodeList.length - 1]) && !isNamedHtmlElement(nodeList[nodeList.length - 1], "br") && isNamedHtmlElement(nodeList[nodeList.length - 1].nextSibling, "br")) {
			nodeList.push(nodeList[nodeList.length - 1].nextSibling);
		}

		// "If the previousSibling of the first member of node list is editable and
		// running sibling criteria on it returns true, let new parent be the
		// previousSibling of the first member of node list."
		var newParent;
		if (isEditable(nodeList[0].previousSibling) && siblingCriteria(nodeList[0].previousSibling)) {
			newParent = nodeList[0].previousSibling;

			// "Otherwise, if the nextSibling of the last member of node list is
			// editable and running sibling criteria on it returns true, let new parent
			// be the nextSibling of the last member of node list."
		} else if (isEditable(nodeList[nodeList.length - 1].nextSibling) && siblingCriteria(nodeList[nodeList.length - 1].nextSibling)) {
			newParent = nodeList[nodeList.length - 1].nextSibling;

			// "Otherwise, run new parent instructions, and let new parent be the
			// result."
		} else {
			newParent = newParentInstructions();
		}

		// "If new parent is null, abort these steps and return null."
		if (!newParent) {
			return null;
		}

		// "If new parent's parent is null:"
		if (!newParent.parentNode) {
			// "Insert new parent into the parent of the first member of node list
			// immediately before the first member of node list."
			nodeList[0].parentNode.insertBefore(newParent, nodeList[0]);

			// "If any range has a boundary point with node equal to the parent of
			// new parent and offset equal to the index of new parent, add one to
			// that boundary point's offset."
			//
			// Try to fix range
			var startContainer = range.startContainer,
				startOffset = range.startOffset,
				endContainer = range.endContainer,
				endOffset = range.endOffset;
			if (startContainer == newParent.parentNode && startOffset >= getNodeIndex(newParent)) {
				range.setStart(startContainer, startOffset + 1);
			}
			if (endContainer == newParent.parentNode && endOffset >= getNodeIndex(newParent)) {
				range.setEnd(endContainer, endOffset + 1);
			}

			// Only try to fix the global range. TODO remove globalRange here
			if (globalRange && globalRange !== range) {
				startContainer = globalRange.startContainer;
				startOffset = globalRange.startOffset;
				endContainer = globalRange.endContainer;
				endOffset = globalRange.endOffset;
				if (startContainer == newParent.parentNode && startOffset >= getNodeIndex(newParent)) {
					globalRange.setStart(startContainer, startOffset + 1);
				}
				if (endContainer == newParent.parentNode && endOffset >= getNodeIndex(newParent)) {
					globalRange.setEnd(endContainer, endOffset + 1);
				}
			}
		}

		// "Let original parent be the parent of the first member of node list."
		var originalParent = nodeList[0].parentNode;

		// "If new parent is before the first member of node list in tree order:"
		if (isBefore(newParent, nodeList[0])) {
			// "If new parent is not an inline node, but the last child of new
			// parent and the first member of node list are both inline nodes, and
			// the last child of new parent is not a br, call createElement("br")
			// on the ownerDocument of new parent and append the result as the last
			// child of new parent."
			if (!isInlineNode(newParent) && isInlineNode(newParent.lastChild) && isInlineNode(nodeList[0]) && !isNamedHtmlElement(newParent.lastChild, "BR")) {
				newParent.appendChild(newParent.ownerDocument.createElement("br"));
			}

			// "For each node in node list, append node as the last child of new
			// parent, preserving ranges."
			for (i = 0; i < nodeList.length; i++) {
				movePreservingRanges(nodeList[i], newParent, -1, range);
			}

			// "Otherwise:"
		} else {
			// "If new parent is not an inline node, but the first child of new
			// parent and the last member of node list are both inline nodes, and
			// the last member of node list is not a br, call createElement("br")
			// on the ownerDocument of new parent and insert the result as the
			// first child of new parent."
			if (!isInlineNode(newParent) && isInlineNode(newParent.firstChild) && isInlineNode(nodeList[nodeList.length - 1]) && !isNamedHtmlElement(nodeList[nodeList.length - 1], "BR")) {
				newParent.insertBefore(newParent.ownerDocument.createElement("br"), newParent.firstChild);
			}

			// "For each node in node list, in reverse order, insert node as the
			// first child of new parent, preserving ranges."
			for (i = nodeList.length - 1; i >= 0; i--) {
				movePreservingRanges(nodeList[i], newParent, 0, range);
			}
		}

		// "If original parent is editable and has no children, remove it from its
		// parent."
		if (isEditable(originalParent) && !originalParent.hasChildNodes()) {
			originalParent.parentNode.removeChild(originalParent);
		}

		// "If new parent's nextSibling is editable and running sibling criteria on
		// it returns true:"
		if (isEditable(newParent.nextSibling) && siblingCriteria(newParent.nextSibling)) {
			// "If new parent is not an inline node, but new parent's last child
			// and new parent's nextSibling's first child are both inline nodes,
			// and new parent's last child is not a br, call createElement("br") on
			// the ownerDocument of new parent and append the result as the last
			// child of new parent."
			if (!isInlineNode(newParent) && isInlineNode(newParent.lastChild) && isInlineNode(newParent.nextSibling.firstChild) && !isNamedHtmlElement(newParent.lastChild, "BR")) {
				newParent.appendChild(newParent.ownerDocument.createElement("br"));
			}

			// "While new parent's nextSibling has children, append its first child
			// as the last child of new parent, preserving ranges."
			while (newParent.nextSibling.hasChildNodes()) {
				movePreservingRanges(newParent.nextSibling.firstChild, newParent, -1, range);
			}

			// "Remove new parent's nextSibling from its parent."
			newParent.parentNode.removeChild(newParent.nextSibling);
		}

		// "Remove extraneous line breaks from new parent."
		removeExtraneousLineBreaksFrom(newParent);

		// "Return new parent."
		return newParent;
	}


	//@}
	///// Allowed children /////
	//@{

	// "A name of an element with inline contents is "a", "abbr", "b", "bdi",
	// "bdo", "cite", "code", "dfn", "em", "h1", "h2", "h3", "h4", "h5", "h6", "i",
	// "kbd", "mark", "p", "pre", "q", "rp", "rt", "ruby", "s", "samp", "small",
	// "span", "strong", "sub", "sup", "u", "var", "acronym", "listing", "strike",
	// "xmp", "big", "blink", "font", "marquee", "nobr", or "tt"."
	var namesOfElementsWithInlineContentsMap = {
		"A": true,
		"ABBR": true,
		"B": true,
		"BDI": true,
		"BDO": true,
		"CITE": true,
		"CODE": true,
		"DFN": true,
		"EM": true,
		"H1": true,
		"H2": true,
		"H3": true,
		"H4": true,
		"H5": true,
		"H6": true,
		"I": true,
		"KBD": true,
		"MARK": true,
		"P": true,
		"PRE": true,
		"Q": true,
		"RP": true,
		"RT": true,
		"RUBY": true,
		"S": true,
		"SAMP": true,
		"SMALL": true,
		"SPAN": true,
		"STRONG": true,
		"SUB": true,
		"SUP": true,
		"U": true,
		"VAR": true,
		"ACRONYM": true,
		"LISTING": true,
		"STRIKE": true,
		"XMP": true,
		"BIG": true,
		"BLINK": true,
		"FONT": true,
		"MARQUEE": true,
		"NOBR": true,
		"TT": true
	};


	var tableRelatedElements = {
		"colgroup": true,
		"table": true,
		"tbody": true,
		"tfoot": true,
		"thead": true,
		"tr": true
	};

	var scriptRelatedElements = {
		"script": true,
		"style": true,
		"plaintext": true,
		"xmp": true
	};

	var prohibitedHeadingNestingMap = jQuery.extend({
		"H1": true,
		"H2": true,
		"H3": true,
		"H4": true,
		"H5": true,
		"H6": true
	}, prohibitedParagraphChildNamesMap);
	var prohibitedTableNestingMap = {
		"CAPTION": true,
		"COL": true,
		"COLGROUP": true,
		"TBODY": true,
		"TD": true,
		"TFOOT": true,
		"TH": true,
		"THEAD": true,
		"TR": true
	};
	var prohibitedDefNestingMap = {
		"DD": true,
		"DT": true
	};
	var prohibitedNestingCombinationsMap = {
		"A": jQuery.extend({
			"A": true
		}, prohibitedParagraphChildNamesMap),
		"DD": prohibitedDefNestingMap,
		"DT": prohibitedDefNestingMap,
		"LI": {
			"LI": true
		},
		"NOBR": jQuery.extend({
			"NOBR": true
		}, prohibitedParagraphChildNamesMap),
		"H1": prohibitedHeadingNestingMap,
		"H2": prohibitedHeadingNestingMap,
		"H3": prohibitedHeadingNestingMap,
		"H4": prohibitedHeadingNestingMap,
		"H5": prohibitedHeadingNestingMap,
		"H6": prohibitedHeadingNestingMap,
		"TD": prohibitedTableNestingMap,
		"TH": prohibitedTableNestingMap,
		// this is the same as namesOfElementsWithInlineContentsMap excluding a and h1-h6 elements above
		"ABBR": prohibitedParagraphChildNamesMap,
		"B": prohibitedParagraphChildNamesMap,
		"BDI": prohibitedParagraphChildNamesMap,
		"BDO": prohibitedParagraphChildNamesMap,
		"CITE": prohibitedParagraphChildNamesMap,
		"CODE": prohibitedParagraphChildNamesMap,
		"DFN": prohibitedParagraphChildNamesMap,
		"EM": prohibitedParagraphChildNamesMap,
		"I": prohibitedParagraphChildNamesMap,
		"KBD": prohibitedParagraphChildNamesMap,
		"MARK": prohibitedParagraphChildNamesMap,
		"P": prohibitedParagraphChildNamesMap,
		"PRE": prohibitedParagraphChildNamesMap,
		"Q": prohibitedParagraphChildNamesMap,
		"RP": prohibitedParagraphChildNamesMap,
		"RT": prohibitedParagraphChildNamesMap,
		"RUBY": prohibitedParagraphChildNamesMap,
		"S": prohibitedParagraphChildNamesMap,
		"SAMP": prohibitedParagraphChildNamesMap,
		"SMALL": prohibitedParagraphChildNamesMap,
		"SPAN": prohibitedParagraphChildNamesMap,
		"STRONG": prohibitedParagraphChildNamesMap,
		"SUB": prohibitedParagraphChildNamesMap,
		"SUP": prohibitedParagraphChildNamesMap,
		"U": prohibitedParagraphChildNamesMap,
		"VAR": prohibitedParagraphChildNamesMap,
		"ACRONYM": prohibitedParagraphChildNamesMap,
		"LISTING": prohibitedParagraphChildNamesMap,
		"STRIKE": prohibitedParagraphChildNamesMap,
		"XMP": prohibitedParagraphChildNamesMap,
		"BIG": prohibitedParagraphChildNamesMap,
		"BLINK": prohibitedParagraphChildNamesMap,
		"FONT": prohibitedParagraphChildNamesMap,
		"MARQUEE": prohibitedParagraphChildNamesMap,
		"TT": prohibitedParagraphChildNamesMap
	};

	// "An element with inline contents is an HTML element whose local name is a
	// name of an element with inline contents."
	function isElementWithInlineContents(node) {
		return isMappedHtmlElement(node, namesOfElementsWithInlineContentsMap);
	}

	function isAllowedChild(child, parent_) {
		// "If parent is "colgroup", "table", "tbody", "tfoot", "thead", "tr", or
		// an HTML element with local name equal to one of those, and child is a
		// Text node whose data does not consist solely of space characters, return
		// false."
		if ((tableRelatedElements[parent_] || isHtmlElementInArray(parent_, ["colgroup", "table", "tbody", "tfoot", "thead", "tr"])) && typeof child == "object" && child.nodeType == $_.Node.TEXT_NODE && !/^[ \t\n\f\r]*$/.test(child.data)) {
			return false;
		}

		// "If parent is "script", "style", "plaintext", or "xmp", or an HTML
		// element with local name equal to one of those, and child is not a Text
		// node, return false."
		if ((scriptRelatedElements[parent_] || isHtmlElementInArray(parent_, ["script", "style", "plaintext", "xmp"])) && (typeof child != "object" || child.nodeType != $_.Node.TEXT_NODE)) {
			return false;
		}

		// "If child is a Document, DocumentFragment, or DocumentType, return
		// false."
		if (typeof child == "object" && (child.nodeType == $_.Node.DOCUMENT_NODE || child.nodeType == $_.Node.DOCUMENT_FRAGMENT_NODE || child.nodeType == $_.Node.DOCUMENT_TYPE_NODE)) {
			return false;
		}

		// "If child is an HTML element, set child to the local name of child."
		if (isAnyHtmlElement(child)) {
			child = child.tagName.toLowerCase();
		}

		// "If child is not a string, return true."
		if (typeof child != "string") {
			return true;
		}

		// "If parent is an HTML element:"
		if (isAnyHtmlElement(parent_)) {
			// "If child is "a", and parent or some ancestor of parent is an a,
			// return false."
			//
			// "If child is a prohibited paragraph child name and parent or some
			// ancestor of parent is an element with inline contents, return
			// false."
			//
			// "If child is "h1", "h2", "h3", "h4", "h5", or "h6", and parent or
			// some ancestor of parent is an HTML element with local name "h1",
			// "h2", "h3", "h4", "h5", or "h6", return false."
			var ancestor = parent_;
			while (ancestor) {
				if (child == "a" && isNamedHtmlElement(ancestor, 'a')) {
					return false;
				}
				if (prohibitedParagraphChildNamesMap[child.toUpperCase()] && isElementWithInlineContents(ancestor)) {
					return false;
				}
				if (/^h[1-6]$/.test(child) && isAnyHtmlElement(ancestor) && /^H[1-6]$/.test(ancestor.tagName)) {
					return false;
				}
				ancestor = ancestor.parentNode;
			}

			// "Let parent be the local name of parent."
			parent_ = parent_.tagName.toLowerCase();
		}

		// "If parent is an Element or DocumentFragment, return true."
		if (typeof parent_ == "object" && (parent_.nodeType == $_.Node.ELEMENT_NODE || parent_.nodeType == $_.Node.DOCUMENT_FRAGMENT_NODE)) {
			return true;
		}

		// "If parent is not a string, return false."
		if (typeof parent_ != "string") {
			return false;
		}

		// "If parent is on the left-hand side of an entry on the following list,
		// then return true if child is listed on the right-hand side of that
		// entry, and false otherwise."
		switch (parent_) {
		case "colgroup":
			return child == "col";
		case "table":
			return jQuery.inArray(child, ["caption", "col", "colgroup", "tbody", "td", "tfoot", "th", "thead", "tr"]) != -1;
		case "tbody":
		case "thead":
		case "tfoot":
			return jQuery.inArray(child, ["td", "th", "tr"]) != -1;
		case "tr":
			return jQuery.inArray(child, ["td", "th"]) != -1;
		case "dl":
			return jQuery.inArray(child, ["dt", "dd"]) != -1;
		case "dir":
		case "ol":
		case "ul":
			return jQuery.inArray(child, ["dir", "li", "ol", "ul"]) != -1;
		case "hgroup":
			return (/^h[1-6]$/).test(child);
		}

		// "If child is "body", "caption", "col", "colgroup", "frame", "frameset",
		// "head", "html", "tbody", "td", "tfoot", "th", "thead", or "tr", return
		// false."
		if (jQuery.inArray(child, ["body", "caption", "col", "colgroup", "frame", "frameset", "head", "html", "tbody", "td", "tfoot", "th", "thead", "tr"]) != -1) {
			return false;
		}

		// "If child is "dd" or "dt" and parent is not "dl", return false."
		if (jQuery.inArray(child, ["dd", "dt"]) != -1 && parent_ != "dl") {
			return false;
		}

		// "If child is "li" and parent is not "ol" or "ul", return false."
		if (child == "li" && parent_ != "ol" && parent_ != "ul") {
			return false;
		}

		// "If parent is on the left-hand side of an entry on the following list
		// and child is listed on the right-hand side of that entry, return false."
		var leftSide = prohibitedNestingCombinationsMap[parent_.toUpperCase()];
		if (leftSide) {
			var rightSide = leftSide[child.toUpperCase()];
			if (rightSide) {
				return false;
			}
		}

		// "Return true."
		return true;
	}


	//@}

	//////////////////////////////////////
	///// Inline formatting commands /////
	//////////////////////////////////////

	///// Inline formatting command definitions /////
	//@{

	// "A node node is effectively contained in a range range if range is not
	// collapsed, and at least one of the following holds:"
	function isEffectivelyContained(node, range) {
		if (range.collapsed) {
			return false;
		}

		// "node is contained in range."
		if (isContained(node, range)) {
			return true;
		}

		// "node is range's start node, it is a Text node, and its length is
		// different from range's start offset."
		if (node == range.startContainer && node.nodeType == $_.Node.TEXT_NODE && getNodeLength(node) != range.startOffset) {
			return true;
		}

		// "node is range's end node, it is a Text node, and range's end offset is
		// not 0."
		if (node == range.endContainer && node.nodeType == $_.Node.TEXT_NODE && range.endOffset != 0) {
			return true;
		}

		// "node has at least one child; and all its children are effectively
		// contained in range; and either range's start node is not a descendant of
		// node or is not a Text node or range's start offset is zero; and either
		// range's end node is not a descendant of node or is not a Text node or
		// range's end offset is its end node's length."
		if (node.hasChildNodes() && $_(node.childNodes).every(function (child) { return isEffectivelyContained(child, range); })
			    && (!isDescendant(range.startContainer, node)
					|| range.startContainer.nodeType != $_.Node.TEXT_NODE
					|| range.startOffset == 0)
			    && (!isDescendant(range.endContainer, node)
					|| range.endContainer.nodeType != $_.Node.TEXT_NODE
					|| range.endOffset == getNodeLength(range.endContainer))) {
			return true;
		}

		return false;
	}

	// Like get(All)ContainedNodes(), but for effectively contained nodes.
	function getEffectivelyContainedNodes(range, condition) {
		if (typeof condition == "undefined") {
			condition = function () {
				return true;
			};
		}
		var node = range.startContainer;
		while (isEffectivelyContained(node.parentNode, range)) {
			node = node.parentNode;
		}

		var stop = nextNodeDescendants(range.endContainer);

		var nodeList = [];
		while (isBefore(node, stop)) {
			if (isEffectivelyContained(node, range) && condition(node)) {
				nodeList.push(node);
				node = nextNodeDescendants(node);
				continue;
			}
			node = nextNode(node);
		}
		return nodeList;
	}

	function getAllEffectivelyContainedNodes(range, condition) {
		if (typeof condition == "undefined") {
			condition = function () {
				return true;
			};
		}
		var node = range.startContainer;
		while (isEffectivelyContained(node.parentNode, range)) {
			node = node.parentNode;
		}

		var stop = nextNodeDescendants(range.endContainer);

		var nodeList = [];
		while (isBefore(node, stop)) {
			if (isEffectivelyContained(node, range) && condition(node)) {
				nodeList.push(node);
			}
			node = nextNode(node);
		}
		return nodeList;
	}

	// "A modifiable element is a b, em, i, s, span, strong, sub, sup, or u element
	// with no attributes except possibly style; or a font element with no
	// attributes except possibly style, color, face, and/or size; or an a element
	// with no attributes except possibly style and/or href."
	function isModifiableElement(node) {
		if (!isAnyHtmlElement(node)) {
			return false;
		}

		if (jQuery.inArray(node.tagName, ["B", "EM", "I", "S", "SPAN", "STRIKE", "STRONG", "SUB", "SUP", "U"]) != -1) {
			if (node.attributes.length == 0) {
				return true;
			}

			if (node.attributes.length == 1 && hasAttribute(node, "style")) {
				return true;
			}
		}

		if (node.tagName == "FONT" || node.tagName == "A") {
			var numAttrs = node.attributes.length;

			if (hasAttribute(node, "style")) {
				numAttrs--;
			}

			if (node.tagName == "FONT") {
				if (hasAttribute(node, "color")) {
					numAttrs--;
				}

				if (hasAttribute(node, "face")) {
					numAttrs--;
				}

				if (hasAttribute(node, "size")) {
					numAttrs--;
				}
			}

			if (node.tagName == "A" && hasAttribute(node, "href")) {
				numAttrs--;
			}

			if (numAttrs == 0) {
				return true;
			}
		}

		return false;
	}

	function isSimpleModifiableElement(node) {
		// "A simple modifiable element is an HTML element for which at least one
		// of the following holds:"
		if (!isAnyHtmlElement(node)) {
			return false;
		}

		// Only these elements can possibly be a simple modifiable element.
		if (jQuery.inArray(node.tagName, ["A", "B", "EM", "FONT", "I", "S", "SPAN", "STRIKE", "STRONG", "SUB", "SUP", "U"]) == -1) {
			return false;
		}

		// "It is an a, b, em, font, i, s, span, strike, strong, sub, sup, or u
		// element with no attributes."
		if (node.attributes.length == 0) {
			return true;
		}

		// If it's got more than one attribute, everything after this fails.
		if (node.attributes.length > 1) {
			return false;
		}

		// "It is an a, b, em, font, i, s, span, strike, strong, sub, sup, or u
		// element with exactly one attribute, which is style, which sets no CSS
		// properties (including invalid or unrecognized properties)."
		//
		// Not gonna try for invalid or unrecognized.
		if (hasAttribute(node, "style") && getStyleLength(node) == 0) {
			return true;
		}

		// "It is an a element with exactly one attribute, which is href."
		if (node.tagName == "A" && hasAttribute(node, "href")) {
			return true;
		}

		// "It is a font element with exactly one attribute, which is either color,
		// face, or size."
		if (node.tagName == "FONT" && (hasAttribute(node, "color") || hasAttribute(node, "face") || hasAttribute(node, "size"))) {
			return true;
		}

		// "It is a b or strong element with exactly one attribute, which is style,
		// and the style attribute sets exactly one CSS property (including invalid
		// or unrecognized properties), which is "font-weight"."
		if ((node.tagName == "B" || node.tagName == "STRONG") && hasAttribute(node, "style") && getStyleLength(node) == 1 && node.style.fontWeight != "") {
			return true;
		}

		// "It is an i or em element with exactly one attribute, which is style,
		// and the style attribute sets exactly one CSS property (including invalid
		// or unrecognized properties), which is "font-style"."
		if ((node.tagName == "I" || node.tagName == "EM") && hasAttribute(node, "style") && getStyleLength(node) == 1 && node.style.fontStyle != "") {
			return true;
		}

		// "It is an a, font, or span element with exactly one attribute, which is
		// style, and the style attribute sets exactly one CSS property (including
		// invalid or unrecognized properties), and that property is not
		// "text-decoration"."
		if ((node.tagName == "A" || node.tagName == "FONT" || node.tagName == "SPAN") && hasAttribute(node, "style") && getStyleLength(node) == 1 && node.style.textDecoration == "") {
			return true;
		}

		// "It is an a, font, s, span, strike, or u element with exactly one
		// attribute, which is style, and the style attribute sets exactly one CSS
		// property (including invalid or unrecognized properties), which is
		// "text-decoration", which is set to "line-through" or "underline" or
		// "overline" or "none"."
		if (jQuery.inArray(node.tagName, ["A", "FONT", "S", "SPAN", "STRIKE", "U"]) != -1 && hasAttribute(node, "style") && getStyleLength(node) == 1 && (node.style.textDecoration == "line-through" || node.style.textDecoration == "underline" || node.style.textDecoration == "overline" || node.style.textDecoration == "none")) {
			return true;
		}

		return false;
	}

	// "Two quantities are equivalent values for a command if either both are null,
	// or both are strings and they're equal and the command does not define any
	// equivalent values, or both are strings and the command defines equivalent
	// values and they match the definition."
	function areEquivalentValues(command, val1, val2) {
		if (val1 === null && val2 === null) {
			return true;
		}

		if (typeof val1 == "string" && typeof val2 == "string" && val1 == val2 && !(commands[command].hasOwnProperty("equivalentValues"))) {
			return true;
		}

		if (typeof val1 == "string" && typeof val2 == "string" && commands[command].hasOwnProperty("equivalentValues") && commands[command].equivalentValues(val1, val2)) {
			return true;
		}

		return false;
	}

	// "Two quantities are loosely equivalent values for a command if either they
	// are equivalent values for the command, or if the command is the fontSize
	// command; one of the quantities is one of "xx-small", "small", "medium",
	// "large", "x-large", "xx-large", or "xxx-large"; and the other quantity is
	// the resolved value of "font-size" on a font element whose size attribute has
	// the corresponding value set ("1" through "7" respectively)."
	function areLooselyEquivalentValues(command, val1, val2) {
		if (areEquivalentValues(command, val1, val2)) {
			return true;
		}

		if (command != "fontsize" || typeof val1 != "string" || typeof val2 != "string") {
			return false;
		}

		// Static variables in JavaScript?
		var callee = areLooselyEquivalentValues;
		if (callee.sizeMap === undefined) {
			callee.sizeMap = {};
			var font = document.createElement("font");
			document.body.appendChild(font);
			$_(["xx-small", "small", "medium", "large", "x-large", "xx-large", "xxx-large"]).forEach(function (keyword) {
				font.size = cssSizeToLegacy(keyword);
				callee.sizeMap[keyword] = $_.getComputedStyle(font).fontSize;
			});
			document.body.removeChild(font);
		}

		return val1 === callee.sizeMap[val2] || val2 === callee.sizeMap[val1];
	}

	//@}
	///// Assorted inline formatting command algorithms /////
	//@{

	function getEffectiveCommandValue(node, command) {
		// "If neither node nor its parent is an Element, return null."
		if (node.nodeType != $_.Node.ELEMENT_NODE && (!node.parentNode || node.parentNode.nodeType != $_.Node.ELEMENT_NODE)) {
			return null;
		}

		// "If node is not an Element, return the effective command value of its
		// parent for command."
		if (node.nodeType != $_.Node.ELEMENT_NODE) {
			return getEffectiveCommandValue(node.parentNode, command);
		}

		// "If command is "createLink" or "unlink":"
		if (command == "createlink" || command == "unlink") {
			// "While node is not null, and is not an a element that has an href
			// attribute, set node to its parent."
			while (node && (!isAnyHtmlElement(node) || node.tagName != "A" || !hasAttribute(node, "href"))) {
				node = node.parentNode;
			}

			// "If node is null, return null."
			if (!node) {
				return null;
			}

			// "Return the value of node's href attribute."
			return node.getAttribute("href");
		}

		// "If command is "backColor" or "hiliteColor":"
		if (command == "backcolor" || command == "hilitecolor") {
			// "While the resolved value of "background-color" on node is any
			// fully transparent value, and node's parent is an Element, set
			// node to its parent."
			//
			// Another lame hack to avoid flawed APIs.
			while (($_.getComputedStyle(node).backgroundColor == "rgba(0, 0, 0, 0)" || $_.getComputedStyle(node).backgroundColor === "" || $_.getComputedStyle(node).backgroundColor == "transparent") && node.parentNode && node.parentNode.nodeType == $_.Node.ELEMENT_NODE) {
				node = node.parentNode;
			}

			// "If the resolved value of "background-color" on node is a fully
			// transparent value, return "rgb(255, 255, 255)"."
			if ($_.getComputedStyle(node).backgroundColor == "rgba(0, 0, 0, 0)" || $_.getComputedStyle(node).backgroundColor === "" || $_.getComputedStyle(node).backgroundColor == "transparent") {
				return "rgb(255, 255, 255)";
			}

			// "Otherwise, return the resolved value of "background-color" for
			// node."
			return $_.getComputedStyle(node).backgroundColor;
		}

		// "If command is "subscript" or "superscript":"
		if (command == "subscript" || command == "superscript") {
			// "Let affected by subscript and affected by superscript be two
			// boolean variables, both initially false."
			var affectedBySubscript = false;
			var affectedBySuperscript = false;

			// "While node is an inline node:"
			while (isInlineNode(node)) {
				var verticalAlign = $_.getComputedStyle(node).verticalAlign;

				// "If node is a sub, set affected by subscript to true."
				if (isNamedHtmlElement(node, 'sub')) {
					affectedBySubscript = true;
					// "Otherwise, if node is a sup, set affected by superscript to
					// true."
				} else if (isNamedHtmlElement(node, 'sup')) {
					affectedBySuperscript = true;
				}

				// "Set node to its parent."
				node = node.parentNode;
			}

			// "If affected by subscript and affected by superscript are both true,
			// return the string "mixed"."
			if (affectedBySubscript && affectedBySuperscript) {
				return "mixed";
			}

			// "If affected by subscript is true, return "subscript"."
			if (affectedBySubscript) {
				return "subscript";
			}

			// "If affected by superscript is true, return "superscript"."
			if (affectedBySuperscript) {
				return "superscript";
			}

			// "Return null."
			return null;
		}

		// "If command is "strikethrough", and the "text-decoration" property of
		// node or any of its ancestors has resolved value containing
		// "line-through", return "line-through". Otherwise, return null."
		if (command == "strikethrough") {
			do {
				if ($_.getComputedStyle(node).textDecoration.indexOf("line-through") != -1) {
					return "line-through";
				}
				node = node.parentNode;
			} while (node && node.nodeType == $_.Node.ELEMENT_NODE);
			return null;
		}

		// "If command is "underline", and the "text-decoration" property of node
		// or any of its ancestors has resolved value containing "underline",
		// return "underline". Otherwise, return null."
		if (command == "underline") {
			do {
				if ($_.getComputedStyle(node).textDecoration.indexOf("underline") != -1) {
					return "underline";
				}
				node = node.parentNode;
			} while (node && node.nodeType == $_.Node.ELEMENT_NODE);
			return null;
		}

		if (!commands[command].hasOwnProperty("relevantCssProperty")) {
			throw "Bug: no relevantCssProperty for " + command + " in getEffectiveCommandValue";
		}

		// "Return the resolved value for node of the relevant CSS property for
		// command."
		return $_.getComputedStyle(node)[commands[command].relevantCssProperty].toString();
	}

	function getSpecifiedCommandValue(element, command) {
		// "If command is "backColor" or "hiliteColor" and element's display
		// property does not have resolved value "inline", return null."
		if ((command == "backcolor" || command == "hilitecolor") && $_.getComputedStyle(element).display != "inline") {
			return null;
		}

		// "If command is "createLink" or "unlink":"
		if (command == "createlink" || command == "unlink") {
			// "If element is an a element and has an href attribute, return the
			// value of that attribute."
			if (isAnyHtmlElement(element) && element.tagName == "A" && hasAttribute(element, "href")) {
				return element.getAttribute("href");
			}

			// "Return null."
			return null;
		}

		// "If command is "subscript" or "superscript":"
		if (command == "subscript" || command == "superscript") {
			// "If element is a sup, return "superscript"."
			if (isNamedHtmlElement(element, 'sup')) {
				return "superscript";
			}

			// "If element is a sub, return "subscript"."
			if (isNamedHtmlElement(element, 'sub')) {
				return "subscript";
			}

			// "Return null."
			return null;
		}

		// "If command is "strikethrough", and element has a style attribute set,
		// and that attribute sets "text-decoration":"
		if (command == "strikethrough" && element.style.textDecoration != "") {
			// "If element's style attribute sets "text-decoration" to a value
			// containing "line-through", return "line-through"."
			if (element.style.textDecoration.indexOf("line-through") != -1) {
				return "line-through";
			}

			// "Return null."
			return null;
		}

		// "If command is "strikethrough" and element is a s or strike element,
		// return "line-through"."
		if (command == "strikethrough" && isHtmlElementInArray(element, ["S", "STRIKE"])) {
			return "line-through";
		}

		// "If command is "underline", and element has a style attribute set, and
		// that attribute sets "text-decoration":"
		if (command == "underline" && element.style.textDecoration != "") {
			// "If element's style attribute sets "text-decoration" to a value
			// containing "underline", return "underline"."
			if (element.style.textDecoration.indexOf("underline") != -1) {
				return "underline";
			}

			// "Return null."
			return null;
		}

		// "If command is "underline" and element is a u element, return
		// "underline"."
		if (command == "underline" && isNamedHtmlElement(element, 'U')) {
			return "underline";
		}

		// "Let property be the relevant CSS property for command."
		var property = commands[command].relevantCssProperty;

		// "If property is null, return null."
		if (property === null) {
			return null;
		}

		// "If element has a style attribute set, and that attribute has the
		// effect of setting property, return the value that it sets property to."
		if (element.style[property] != "") {
			return element.style[property];
		}

		// "If element is a font element that has an attribute whose effect is
		// to create a presentational hint for property, return the value that the
		// hint sets property to.  (For a size of 7, this will be the non-CSS value
		// "xxx-large".)"
		if (isHtmlNamespace(element.namespaceURI) && element.tagName == "FONT") {
			if (property == "color" && hasAttribute(element, "color")) {
				return element.color;
			}
			if (property == "fontFamily" && hasAttribute(element, "face")) {
				return element.face;
			}
			if (property == "fontSize" && hasAttribute(element, "size")) {
				// This is not even close to correct in general.
				var size = parseInt(element.size, 10);
				if (size < 1) {
					size = 1;
				}
				if (size > 7) {
					size = 7;
				}
				return {
					1: "xx-small",
					2: "small",
					3: "medium",
					4: "large",
					5: "x-large",
					6: "xx-large",
					7: "xxx-large"
				}[size];
			}
		}

		// "If element is in the following list, and property is equal to the
		// CSS property name listed for it, return the string listed for it."
		//
		// A list follows, whose meaning is copied here.
		if (property == "fontWeight" && (element.tagName == "B" || element.tagName == "STRONG")) {
			return "bold";
		}
		if (property == "fontStyle" && (element.tagName == "I" || element.tagName == "EM")) {
			return "italic";
		}

		// "Return null."
		return null;
	}

	function reorderModifiableDescendants(node, command, newValue, range) {
		// "Let candidate equal node."
		var candidate = node;

		// "While candidate is a modifiable element, and candidate has exactly one
		// child, and that child is also a modifiable element, and candidate is not
		// a simple modifiable element or candidate's specified command value for
		// command is not equivalent to new value, set candidate to its child."
		while (isModifiableElement(candidate) && candidate.childNodes.length == 1 && isModifiableElement(candidate.firstChild) && (!isSimpleModifiableElement(candidate) || !areEquivalentValues(command, getSpecifiedCommandValue(candidate, command), newValue))) {
			candidate = candidate.firstChild;
		}

		// "If candidate is node, or is not a simple modifiable element, or its
		// specified command value is not equivalent to new value, or its effective
		// command value is not loosely equivalent to new value, abort these
		// steps."
		if (candidate == node || !isSimpleModifiableElement(candidate) || !areEquivalentValues(command, getSpecifiedCommandValue(candidate, command), newValue) || !areLooselyEquivalentValues(command, getEffectiveCommandValue(candidate, command), newValue)) {
			return;
		}

		// "While candidate has children, insert the first child of candidate into
		// candidate's parent immediately before candidate, preserving ranges."
		while (candidate.hasChildNodes()) {
			movePreservingRanges(candidate.firstChild, candidate.parentNode, getNodeIndex(candidate), range);
		}

		// "Insert candidate into node's parent immediately after node."
		node.parentNode.insertBefore(candidate, node.nextSibling);

		// "Append the node as the last child of candidate, preserving ranges."
		movePreservingRanges(node, candidate, -1, range);
	}

	var recordValuesCommands = ["subscript", "bold", "fontname", "fontsize", "forecolor", "hilitecolor", "italic", "strikethrough", "underline"];

	function recordValues(nodeList) {
		// "Let values be a list of (node, command, specified command value)
		// triples, initially empty."
		var values = [];

		// "For each node in node list, for each command in the list "subscript",
		// "bold", "fontName", "fontSize", "foreColor", "hiliteColor", "italic",
		// "strikethrough", and "underline" in that order:"

		// Ensure we have a plain array to avoid the potential performance
		// overhead of a NodeList
		var nodes = jQuery.makeArray(nodeList);
		var i, j;
		var node;
		var command;
		var ancestor;
		var specifiedCommandValue;
		for (i = 0; i < nodes.length; i++) {
			node = nodes[i];
			for (j = 0; j < recordValuesCommands.length; j++) {
				command = recordValuesCommands[j];

				// "Let ancestor equal node."
				ancestor = node;

				// "If ancestor is not an Element, set it to its parent."
				if (ancestor.nodeType != 1) {
					ancestor = ancestor.parentNode;
				}

				// "While ancestor is an Element and its specified command value
				// for command is null, set it to its parent."
				specifiedCommandValue = null;
				while (ancestor && ancestor.nodeType == 1 && (specifiedCommandValue = getSpecifiedCommandValue(ancestor, command)) === null) {
					ancestor = ancestor.parentNode;
				}

				// "If ancestor is an Element, add (node, command, ancestor's
				// specified command value for command) to values. Otherwise add
				// (node, command, null) to values."
				values.push([node, command, specifiedCommandValue]);
			}
		}

		// "Return values."
		return values;
	}

	//@}
	///// Clearing an element's value /////
	//@{

	function clearValue(element, command, range) {
		// "If element is not editable, return the empty list."
		if (!isEditable(element)) {
			return [];
		}

		// "If element's specified command value for command is null, return the
		// empty list."
		if (getSpecifiedCommandValue(element, command) === null) {
			return [];
		}

		// "If element is a simple modifiable element:"
		if (isSimpleModifiableElement(element)) {
			// "Let children be the children of element."
			var children = Array.prototype.slice.call(toArray(element.childNodes));

			// "For each child in children, insert child into element's parent
			// immediately before element, preserving ranges."
			var i;
			for (i = 0; i < children.length; i++) {
				movePreservingRanges(children[i], element.parentNode, getNodeIndex(element), range);
			}

			// "Remove element from its parent."
			element.parentNode.removeChild(element);

			// "Return children."
			return children;
		}

		// "If command is "strikethrough", and element has a style attribute that
		// sets "text-decoration" to some value containing "line-through", delete
		// "line-through" from the value."
		if (command == "strikethrough" && element.style.textDecoration.indexOf("line-through") != -1) {
			if (element.style.textDecoration == "line-through") {
				element.style.textDecoration = "";
			} else {
				element.style.textDecoration = element.style.textDecoration.replace("line-through", "");
			}
			if (element.getAttribute("style") == "") {
				element.removeAttribute("style");
			}
		}

		// "If command is "underline", and element has a style attribute that sets
		// "text-decoration" to some value containing "underline", delete
		// "underline" from the value."
		if (command == "underline" && element.style.textDecoration.indexOf("underline") != -1) {
			if (element.style.textDecoration == "underline") {
				element.style.textDecoration = "";
			} else {
				element.style.textDecoration = element.style.textDecoration.replace("underline", "");
			}
			if (element.getAttribute("style") == "") {
				element.removeAttribute("style");
			}
		}

		// "If the relevant CSS property for command is not null, unset the CSS
		// property property of element."
		if (commands[command].relevantCssProperty !== null) {
			element.style[commands[command].relevantCssProperty] = '';
			if (element.getAttribute("style") == "") {
				element.removeAttribute("style");
			}
		}

		// "If element is a font element:"
		if (isHtmlNamespace(element.namespaceURI) && element.tagName == "FONT") {
			// "If command is "foreColor", unset element's color attribute, if set."
			if (command == "forecolor") {
				element.removeAttribute("color");
			}

			// "If command is "fontName", unset element's face attribute, if set."
			if (command == "fontname") {
				element.removeAttribute("face");
			}

			// "If command is "fontSize", unset element's size attribute, if set."
			if (command == "fontsize") {
				element.removeAttribute("size");
			}
		}

		// "If element is an a element and command is "createLink" or "unlink",
		// unset the href property of element."
		if (isNamedHtmlElement(element, 'A') && (command == "createlink" || command == "unlink")) {
			element.removeAttribute("href");
		}

		// "If element's specified command value for command is null, return the
		// empty list."
		if (getSpecifiedCommandValue(element, command) === null) {
			return [];
		}

		// "Set the tag name of element to "span", and return the one-node list
		// consisting of the result."
		return [setTagName(element, "span", range)];
	}

	//@}
	///// Forcing the value of a node /////
	//@{

	function forceValue(node, command, newValue, range) {
		var children = [];
		var i;
		var specifiedValue;

		// "If node's parent is null, abort this algorithm."
		if (!node.parentNode) {
			return;
		}

		// "If new value is null, abort this algorithm."
		if (newValue === null) {
			return;
		}

		// "If node is an allowed child of "span":"
		if (isAllowedChild(node, "span")) {
			// "Reorder modifiable descendants of node's previousSibling."
			reorderModifiableDescendants(node.previousSibling, command, newValue, range);

			// "Reorder modifiable descendants of node's nextSibling."
			reorderModifiableDescendants(node.nextSibling, command, newValue, range);

			// "Wrap the one-node list consisting of node, with sibling criteria
			// returning true for a simple modifiable element whose specified
			// command value is equivalent to new value and whose effective command
			// value is loosely equivalent to new value and false otherwise, and
			// with new parent instructions returning null."
			wrap(
				[node],
				function (node) {
					return isSimpleModifiableElement(node) && areEquivalentValues(command, getSpecifiedCommandValue(node, command), newValue) && areLooselyEquivalentValues(command, getEffectiveCommandValue(node, command), newValue);
				},
				function () {
					return null;
				},
				range
			);
		}

		// "If the effective command value of command is loosely equivalent to new
		// value on node, abort this algorithm."
		if (areLooselyEquivalentValues(command, getEffectiveCommandValue(node, command), newValue)) {
			return;
		}

		// "If node is not an allowed child of "span":"
		if (!isAllowedChild(node, "span")) {
			// "Let children be all children of node, omitting any that are
			// Elements whose specified command value for command is neither null
			// nor equivalent to new value."
			for (i = 0; i < node.childNodes.length; i++) {
				if (node.childNodes[i].nodeType == $_.Node.ELEMENT_NODE) {
					specifiedValue = getSpecifiedCommandValue(node.childNodes[i], command);

					if (specifiedValue !== null && !areEquivalentValues(command, newValue, specifiedValue)) {
						continue;
					}
				}
				children.push(node.childNodes[i]);
			}

			// "Force the value of each Node in children, with command and new
			// value as in this invocation of the algorithm."
			for (i = 0; i < children.length; i++) {
				forceValue(children[i], command, newValue, range);
			}

			// "Abort this algorithm."
			return;
		}

		// "If the effective command value of command is loosely equivalent to new
		// value on node, abort this algorithm."
		if (areLooselyEquivalentValues(command, getEffectiveCommandValue(node, command), newValue)) {
			return;
		}

		// "Let new parent be null."
		var newParent = null;

		// "If the CSS styling flag is false:"
		if (!cssStylingFlag) {
			// "If command is "bold" and new value is "bold", let new parent be the
			// result of calling createElement("b") on the ownerDocument of node."
			if (command == "bold" && (newValue == "bold" || newValue == "700")) {
				newParent = node.ownerDocument.createElement("b");
			}

			// "If command is "italic" and new value is "italic", let new parent be
			// the result of calling createElement("i") on the ownerDocument of
			// node."
			if (command == "italic" && newValue == "italic") {
				newParent = node.ownerDocument.createElement("i");
			}

			// "If command is "strikethrough" and new value is "line-through", let
			// new parent be the result of calling createElement("s") on the
			// ownerDocument of node."
			if (command == "strikethrough" && newValue == "line-through") {
				newParent = node.ownerDocument.createElement("s");
			}

			// "If command is "underline" and new value is "underline", let new
			// parent be the result of calling createElement("u") on the
			// ownerDocument of node."
			if (command == "underline" && newValue == "underline") {
				newParent = node.ownerDocument.createElement("u");
			}

			// "If command is "foreColor", and new value is fully opaque with red,
			// green, and blue components in the range 0 to 255:"
			if (command == "forecolor" && parseSimpleColor(newValue)) {
				// "Let new parent be the result of calling createElement("span")
				// on the ownerDocument of node."
				// NOTE: modified this process to create span elements with style attributes
				// instead of oldschool font tags with color attributes
				newParent = node.ownerDocument.createElement("span");

				// "If new value is an extended color keyword, set the color
				// attribute of new parent to new value."
				//
				// "Otherwise, set the color attribute of new parent to the result
				// of applying the rules for serializing simple color values to new
				// value (interpreted as a simple color)."
				jQuery(newParent).css('color', parseSimpleColor(newValue));
			}

			// "If command is "fontName", let new parent be the result of calling
			// createElement("font") on the ownerDocument of node, then set the
			// face attribute of new parent to new value."
			if (command == "fontname") {
				newParent = node.ownerDocument.createElement("font");
				newParent.face = newValue;
			}
		}

		// "If command is "createLink" or "unlink":"
		if (command == "createlink" || command == "unlink") {
			// "Let new parent be the result of calling createElement("a") on the
			// ownerDocument of node."
			newParent = node.ownerDocument.createElement("a");

			// "Set the href attribute of new parent to new value."
			newParent.setAttribute("href", newValue);

			// "Let ancestor be node's parent."
			var ancestor = node.parentNode;

			// "While ancestor is not null:"
			while (ancestor) {
				// "If ancestor is an a, set the tag name of ancestor to "span",
				// and let ancestor be the result."
				if (isNamedHtmlElement(ancestor, 'A')) {
					ancestor = setTagName(ancestor, "span", range);
				}

				// "Set ancestor to its parent."
				ancestor = ancestor.parentNode;
			}
		}

		// "If command is "fontSize"; and new value is one of "xx-small", "small",
		// "medium", "large", "x-large", "xx-large", or "xxx-large"; and either the
		// CSS styling flag is false, or new value is "xxx-large": let new parent
		// be the result of calling createElement("font") on the ownerDocument of
		// node, then set the size attribute of new parent to the number from the
		// following table based on new value: [table omitted]"
		if (command == "fontsize" && jQuery.inArray(newValue, ["xx-small", "small", "medium", "large", "x-large", "xx-large", "xxx-large"]) != -1 && (!cssStylingFlag || newValue == "xxx-large")) {
			newParent = node.ownerDocument.createElement("font");
			newParent.size = cssSizeToLegacy(newValue);
		}

		// "If command is "subscript" or "superscript" and new value is
		// "subscript", let new parent be the result of calling
		// createElement("sub") on the ownerDocument of node."
		if ((command == "subscript" || command == "superscript") && newValue == "subscript") {
			newParent = node.ownerDocument.createElement("sub");
		}

		// "If command is "subscript" or "superscript" and new value is
		// "superscript", let new parent be the result of calling
		// createElement("sup") on the ownerDocument of node."
		if ((command == "subscript" || command == "superscript") && newValue == "superscript") {
			newParent = node.ownerDocument.createElement("sup");
		}

		// "If new parent is null, let new parent be the result of calling
		// createElement("span") on the ownerDocument of node."
		if (!newParent) {
			newParent = node.ownerDocument.createElement("span");
		}

		// "Insert new parent in node's parent before node."
		node.parentNode.insertBefore(newParent, node);

		// "If the effective command value of command for new parent is not loosely
		// equivalent to new value, and the relevant CSS property for command is
		// not null, set that CSS property of new parent to new value (if the new
		// value would be valid)."
		var property = commands[command].relevantCssProperty;
		if (property !== null && !areLooselyEquivalentValues(command, getEffectiveCommandValue(newParent, command), newValue)) {
			newParent.style[property] = newValue;
		}

		// "If command is "strikethrough", and new value is "line-through", and the
		// effective command value of "strikethrough" for new parent is not
		// "line-through", set the "text-decoration" property of new parent to
		// "line-through"."
		if (command == "strikethrough" && newValue == "line-through" && getEffectiveCommandValue(newParent, "strikethrough") != "line-through") {
			newParent.style.textDecoration = "line-through";
		}

		// "If command is "underline", and new value is "underline", and the
		// effective command value of "underline" for new parent is not
		// "underline", set the "text-decoration" property of new parent to
		// "underline"."
		if (command == "underline" && newValue == "underline" && getEffectiveCommandValue(newParent, "underline") != "underline") {
			newParent.style.textDecoration = "underline";
		}

		// "Append node to new parent as its last child, preserving ranges."
		movePreservingRanges(node, newParent, newParent.childNodes.length, range);

		// "If node is an Element and the effective command value of command for
		// node is not loosely equivalent to new value:"
		if (node.nodeType == $_.Node.ELEMENT_NODE && !areEquivalentValues(command, getEffectiveCommandValue(node, command), newValue)) {
			// "Insert node into the parent of new parent before new parent,
			// preserving ranges."
			movePreservingRanges(node, newParent.parentNode, getNodeIndex(newParent), range);

			// "Remove new parent from its parent."
			newParent.parentNode.removeChild(newParent);

			// "Let children be all children of node, omitting any that are
			// Elements whose specified command value for command is neither null
			// nor equivalent to new value."
			children = [];
			for (i = 0; i < node.childNodes.length; i++) {
				if (node.childNodes[i].nodeType == $_.Node.ELEMENT_NODE) {
					specifiedValue = getSpecifiedCommandValue(node.childNodes[i], command);

					if (specifiedValue !== null && !areEquivalentValues(command, newValue, specifiedValue)) {
						continue;
					}
				}
				children.push(node.childNodes[i]);
			}

			// "Force the value of each Node in children, with command and new
			// value as in this invocation of the algorithm."
			for (i = 0; i < children.length; i++) {
				forceValue(children[i], command, newValue, range);
			}
		}
	}

	//@}
	///// Pushing down values /////
	//@{

	function pushDownValues(node, command, newValue, range) {
		// "If node's parent is not an Element, abort this algorithm."
		if (!node.parentNode || node.parentNode.nodeType != $_.Node.ELEMENT_NODE) {
			return;
		}

		// "If the effective command value of command is loosely equivalent to new
		// value on node, abort this algorithm."
		if (areLooselyEquivalentValues(command, getEffectiveCommandValue(node, command), newValue)) {
			return;
		}

		// "Let current ancestor be node's parent."
		var currentAncestor = node.parentNode;

		// "Let ancestor list be a list of Nodes, initially empty."
		var ancestorList = [];

		// "While current ancestor is an editable Element and the effective command
		// value of command is not loosely equivalent to new value on it, append
		// current ancestor to ancestor list, then set current ancestor to its
		// parent."
		while (isEditable(currentAncestor) && currentAncestor.nodeType == $_.Node.ELEMENT_NODE && !areLooselyEquivalentValues(command, getEffectiveCommandValue(currentAncestor, command), newValue)) {
			ancestorList.push(currentAncestor);
			currentAncestor = currentAncestor.parentNode;
		}

		// "If ancestor list is empty, abort this algorithm."
		if (!ancestorList.length) {
			return;
		}

		// "Let propagated value be the specified command value of command on the
		// last member of ancestor list."
		var propagatedValue = getSpecifiedCommandValue(ancestorList[ancestorList.length - 1], command);

		// "If propagated value is null and is not equal to new value, abort this
		// algorithm."
		if (propagatedValue === null && propagatedValue != newValue) {
			return;
		}

		// "If the effective command value for the parent of the last member of
		// ancestor list is not loosely equivalent to new value, and new value is
		// not null, abort this algorithm."
		if (newValue !== null && !areLooselyEquivalentValues(command, getEffectiveCommandValue(ancestorList[ancestorList.length - 1].parentNode, command), newValue)) {
			return;
		}

		// "While ancestor list is not empty:"
		while (ancestorList.length) {
			// "Let current ancestor be the last member of ancestor list."
			// "Remove the last member from ancestor list."
			currentAncestor = ancestorList.pop();

			// "If the specified command value of current ancestor for command is
			// not null, set propagated value to that value."
			if (getSpecifiedCommandValue(currentAncestor, command) !== null) {
				propagatedValue = getSpecifiedCommandValue(currentAncestor, command);
			}

			// "Let children be the children of current ancestor."
			var children = Array.prototype.slice.call(toArray(currentAncestor.childNodes));

			// "If the specified command value of current ancestor for command is
			// not null, clear the value of current ancestor."
			if (getSpecifiedCommandValue(currentAncestor, command) !== null) {
				clearValue(currentAncestor, command, range);
			}

			// "For every child in children:"
			var i;
			for (i = 0; i < children.length; i++) {
				var child = children[i];

				// "If child is node, continue with the next child."
				if (child == node) {
					continue;
				}

				// "If child is an Element whose specified command value for
				// command is neither null nor equivalent to propagated value,
				// continue with the next child."
				if (child.nodeType == $_.Node.ELEMENT_NODE && getSpecifiedCommandValue(child, command) !== null && !areEquivalentValues(command, propagatedValue, getSpecifiedCommandValue(child, command))) {
					continue;
				}

				// "If child is the last member of ancestor list, continue with the
				// next child."
				if (child == ancestorList[ancestorList.length - 1]) {
					continue;
				}

				// "Force the value of child, with command as in this algorithm
				// and new value equal to propagated value."
				forceValue(child, command, propagatedValue, range);
			}
		}
	}

	function restoreValues(values, range) {
		// "For each (node, command, value) triple in values:"
		$_(values).forEach(function (triple) {
			var node = triple[0];
			var command = triple[1];
			var value = triple[2];

			// "Let ancestor equal node."
			var ancestor = node;

			// "If ancestor is not an Element, set it to its parent."
			if (!ancestor || ancestor.nodeType != $_.Node.ELEMENT_NODE) {
				ancestor = ancestor.parentNode;
			}

			// "While ancestor is an Element and its specified command value for
			// command is null, set it to its parent."
			while (ancestor && ancestor.nodeType == $_.Node.ELEMENT_NODE && getSpecifiedCommandValue(ancestor, command) === null) {
				ancestor = ancestor.parentNode;
			}

			// "If value is null and ancestor is an Element, push down values on
			// node for command, with new value null."
			if (value === null && ancestor && ancestor.nodeType == $_.Node.ELEMENT_NODE) {
				pushDownValues(node, command, null, range);

				// "Otherwise, if ancestor is an Element and its specified command
				// value for command is not equivalent to value, or if ancestor is not
				// an Element and value is not null, force the value of command to
				// value on node."
			} else if ((ancestor && ancestor.nodeType == $_.Node.ELEMENT_NODE && !areEquivalentValues(command, getSpecifiedCommandValue(ancestor, command), value)) || ((!ancestor || ancestor.nodeType != $_.Node.ELEMENT_NODE) && value !== null)) {
				forceValue(node, command, value, range);
			}
		});
	}

	//@}
	///// Setting the selection's value /////
	//@{

	function setSelectionValue(command, newValue, range) {

		// Use current selected range if no range passed
		range = range || getActiveRange();

		// "If there is no editable text node effectively contained in the active
		// range:"
		if (!$_(getAllEffectivelyContainedNodes(range)).filter(function (node) { return node.nodeType == $_.Node.TEXT_NODE; }, true).some(isEditable)) {
			// "If command has inline command activated values, set the state
			// override to true if new value is among them and false if it's not."
			if (commands[command].hasOwnProperty("inlineCommandActivatedValues")) {
				setStateOverride(
					command,
					$_(commands[command].inlineCommandActivatedValues).indexOf(newValue) != -1,
					range
				);
			}

			// "If command is "subscript", unset the state override for
			// "superscript"."
			if (command == "subscript") {
				unsetStateOverride("superscript", range);
			}

			// "If command is "superscript", unset the state override for
			// "subscript"."
			if (command == "superscript") {
				unsetStateOverride("subscript", range);
			}

			// "If new value is null, unset the value override (if any)."
			if (newValue === null) {
				unsetValueOverride(command, range);

				// "Otherwise, if command has a value specified, set the value override
				// to new value."
			} else if (commands[command].hasOwnProperty("value")) {
				setValueOverride(command, newValue, range);
			}

			// "Abort these steps."
			return;
		}

		// "If the active range's start node is an editable Text node, and its
		// start offset is neither zero nor its start node's length, call
		// splitText() on the active range's start node, with argument equal to the
		// active range's start offset. Then set the active range's start node to
		// the result, and its start offset to zero."
		if (isEditable(range.startContainer) && range.startContainer.nodeType == $_.Node.TEXT_NODE && range.startOffset != 0 && range.startOffset != getNodeLength(range.startContainer)) {
			// Account for browsers not following range mutation rules
			var newNode = range.startContainer.splitText(range.startOffset);
			var newActiveRange = Aloha.createRange();
			if (range.startContainer == range.endContainer) {
				var newEndOffset = range.endOffset - range.startOffset;
				newActiveRange.setEnd(newNode, newEndOffset);
				range.setEnd(newNode, newEndOffset);
			}
			newActiveRange.setStart(newNode, 0);
			Aloha.getSelection().removeAllRanges();
			Aloha.getSelection().addRange(newActiveRange);

			range.setStart(newNode, 0);
		}

		// "If the active range's end node is an editable Text node, and its end
		// offset is neither zero nor its end node's length, call splitText() on
		// the active range's end node, with argument equal to the active range's
		// end offset."
		if (isEditable(range.endContainer) && range.endContainer.nodeType == $_.Node.TEXT_NODE && range.endOffset != 0 && range.endOffset != getNodeLength(range.endContainer)) {
			// IE seems to mutate the range incorrectly here, so we need correction
			// here as well.  The active range will be temporarily in orphaned
			// nodes, so calling getActiveRange() after splitText() but before
			// fixing the range will throw an exception.
			// TODO: check if this is still neccessary 
			var activeRange = range;
			var newStart = [activeRange.startContainer, activeRange.startOffset];
			var newEnd = [activeRange.endContainer, activeRange.endOffset];
			activeRange.endContainer.splitText(activeRange.endOffset);
			activeRange.setStart(newStart[0], newStart[1]);
			activeRange.setEnd(newEnd[0], newEnd[1]);

			Aloha.getSelection().removeAllRanges();
			Aloha.getSelection().addRange(activeRange);
		}

		// "Let element list be all editable Elements effectively contained in the
		// active range.
		//
		// "For each element in element list, clear the value of element."
		$_(getAllEffectivelyContainedNodes(getActiveRange(), function (node) {
			return isEditable(node) && node.nodeType == $_.Node.ELEMENT_NODE;
		})).forEach(function (element) {
			clearValue(element, command, range);
		});

		// "Let node list be all editable nodes effectively contained in the active
		// range.
		//
		// "For each node in node list:"
		$_(getAllEffectivelyContainedNodes(range, isEditable)).forEach(function (node) {
			// "Push down values on node."
			pushDownValues(node, command, newValue, range);

			// "Force the value of node."
			forceValue(node, command, newValue, range);
		});
	}

	/**
	 * attempt to retrieve a block like a table or an Aloha Block
	 * which is located one step right of the current caret position.
	 * If an appropriate element is found it will be returned or
	 * false otherwise
	 * 
	 * @param {element} node current node we're in
	 * @param {number} offset current offset within that node
	 * 
	 * @return the dom node if found or false if no appropriate
	 * element was found
	 */
	function getBlockAtNextPosition(node, offset) {
		var i;

		// if we're inside a text node we first have to check
		// if there is nothing but tabs, newlines or the like
		// after our current cursor position
		if (node.nodeType === $_.Node.TEXT_NODE && offset < node.length) {
			for (i = offset; i < node.length; i++) {
				if ((node.data.charAt(i) !== '\t' && node.data.charAt(i) !== '\r' && node.data.charAt(i) !== '\n') || node.data.charCodeAt(i) === 160) { // &nbsp;
					// this is a character that has to be deleted first
					return false;
				}
			}
		}

		// try the most simple approach first: the next sibling
		// is a table
		if (node.nextSibling && node.nextSibling.className && node.nextSibling.className.indexOf("aloha-table-wrapper") >= 0) {
			return node.nextSibling;
		}

		// since we got only ignorable whitespace here determine if
		// our nodes parents next sibling is a table
		if (node.parentNode && node.parentNode.nextSibling && node.parentNode.nextSibling.className && node.parentNode.nextSibling.className.indexOf("aloha-table-wrapper") >= 0) {
			return node.parentNode.nextSibling;
		}

		// our parents nextsibling is a pure whitespace node such as
		// generated by sourcecode indentation so we'll check for
		// the next next sibling
		if (node.parentNode && node.parentNode.nextSibling && isWhitespaceNode(node.parentNode.nextSibling) && node.parentNode.nextSibling.nextSibling && node.parentNode.nextSibling.nextSibling.className && node.parentNode.nextSibling.nextSibling.className.indexOf("aloha-table-wrapper") >= 0) {
			return node.parentNode.nextSibling.nextSibling;
		}

		// Note: the search above works for tables, since they cannot be
		// nested deeply in paragraphs and other formatting tags. If this code
		// is extended to work also for other blocks, the search probably needs to be adapted
	}

	/**
	 * Attempt to retrieve a block like a table or an Aloha Block
	 * which is located right before the current position.
	 * If an appropriate element is found, it will be returned or
	 * false otherwise
	 * 
	 * @param {element} node current node
	 * @param {offset} offset current offset
	 * 
	 * @return dom node of found or false if no appropriate
	 * element was found
	 */
	function getBlockAtPreviousPosition(node, offset) {
		var i;

		if (node.nodeType === $_.Node.TEXT_NODE && offset > 0) {
			for (i = offset - 1; i >= 0; i--) {
				if ((node.data.charAt(i) !== '\t' && node.data.charAt(i) !== '\r' && node.data.charAt(i) !== '\n') || node.data.charCodeAt(i) === 160) { // &nbsp;
					// this is a character that has to be deleted first
					return false;
				}
			}
		}

		// try the previous sibling
		if (node.previousSibling && node.previousSibling.className && node.previousSibling.className.indexOf("aloha-table-wrapper") >= 0) {
			return node.previousSibling;
		}

		// try the parent's previous sibling
		if (node.parentNode && node.parentNode.previousSibling && node.parentNode.previousSibling.className && node.parentNode.previousSibling.className.indexOf("aloha-table-wrapper") >= 0) {
			return node.parentNode.previousSibling;
		}

		// the parent's previous sibling might be a whitespace node
		if (node.parentNode && node.parentNode.previousSibling && isWhitespaceNode(node.parentNode.previousSibling) && node.parentNode.previousSibling.previousSibling && node.parentNode.previousSibling.previousSibling.className && node.parentNode.previousSibling.previousSibling.className.indexOf('aloha-table-wrapper') >= 0) {
			return node.parentNode.previousSibling.previousSibling;
		}

		// Note: the search above works for tables, since they cannot be
		// nested deeply in paragraphs and other formatting tags. If this code
		// is extended to work also for other blocks, the search probably needs to be adapted

		return false;
	}

	// "A boundary point (node, offset) is a block start point if either node's
	// parent is null and offset is zero; or node has a child with index offset −
	// 1, and that child is either a visible block node or a visible br."
	function isBlockStartPoint(node, offset) {
		return (!node.parentNode && offset == 0) || (0 <= offset - 1 && offset - 1 < node.childNodes.length && isVisible(node.childNodes[offset - 1]) && (isBlockNode(node.childNodes[offset - 1]) || isNamedHtmlElement(node.childNodes[offset - 1], "br")));
	}

	// "A boundary point (node, offset) is a block end point if either node's
	// parent is null and offset is node's length; or node has a child with index
	// offset, and that child is a visible block node."
	function isBlockEndPoint(node, offset) {
		return (!node.parentNode && offset == getNodeLength(node)) || (offset < node.childNodes.length && isVisible(node.childNodes[offset]) && isBlockNode(node.childNodes[offset]));
	}

	// "A boundary point is a block boundary point if it is either a block start
	// point or a block end point."
	function isBlockBoundaryPoint(node, offset) {
		return isBlockStartPoint(node, offset) || isBlockEndPoint(node, offset);
	}

	function followsLineBreak(node) {
		// "Let offset be zero."
		var offset = 0;

		// "While (node, offset) is not a block boundary point:"
		while (!isBlockBoundaryPoint(node, offset)) {
			// "If node has a visible child with index offset minus one, return
			// false."
			if (0 <= offset - 1 && offset - 1 < node.childNodes.length && isVisible(node.childNodes[offset - 1])) {
				return false;
			}

			// "If offset is zero or node has no children, set offset to node's
			// index, then set node to its parent."
			if (offset == 0 || !node.hasChildNodes()) {
				offset = getNodeIndex(node);
				node = node.parentNode;

				// "Otherwise, set node to its child with index offset minus one, then
				// set offset to node's length."
			} else {
				node = node.childNodes[offset - 1];
				offset = getNodeLength(node);
			}
		}

		// "Return true."
		return true;
	}

	function precedesLineBreak(node) {
		// "Let offset be node's length."
		var offset = getNodeLength(node);

		// "While (node, offset) is not a block boundary point:"
		while (!isBlockBoundaryPoint(node, offset)) {
			// "If node has a visible child with index offset, return false."
			if (offset < node.childNodes.length && isVisible(node.childNodes[offset])) {
				return false;
			}

			// "If offset is node's length or node has no children, set offset to
			// one plus node's index, then set node to its parent."
			if (offset == getNodeLength(node) || !node.hasChildNodes()) {
				offset = 1 + getNodeIndex(node);
				node = node.parentNode;

				// "Otherwise, set node to its child with index offset and set offset
				// to zero."
			} else {
				node = node.childNodes[offset];
				offset = 0;
			}
		}

		// "Return true."
		return true;
	}

	//@}
	///// Splitting a node list's parent /////
	//@{

	function splitParent(nodeList, range) {
		var i;

		// "Let original parent be the parent of the first member of node list."
		var originalParent = nodeList[0].parentNode;

		// "If original parent is not editable or its parent is null, do nothing
		// and abort these steps."
		if (!isEditable(originalParent) || !originalParent.parentNode) {
			return;
		}

		// "If the first child of original parent is in node list, remove
		// extraneous line breaks before original parent."
		if (jQuery.inArray(originalParent.firstChild, nodeList) != -1) {
			removeExtraneousLineBreaksBefore(originalParent);
		}

		var firstChildInNodeList = jQuery.inArray(originalParent.firstChild, nodeList) != -1;
		var lastChildInNodeList = jQuery.inArray(originalParent.lastChild, nodeList) != -1;

		// "If the first child of original parent is in node list, and original
		// parent follows a line break, set follows line break to true. Otherwise,
		// set follows line break to false."
		var followsLineBreak_ = firstChildInNodeList && followsLineBreak(originalParent);

		// "If the last child of original parent is in node list, and original
		// parent precedes a line break, set precedes line break to true.
		// Otherwise, set precedes line break to false."
		var precedesLineBreak_ = lastChildInNodeList && precedesLineBreak(originalParent);

		// "If the first child of original parent is not in node list, but its last
		// child is:"
		if (!firstChildInNodeList && lastChildInNodeList) {
			// "For each node in node list, in reverse order, insert node into the
			// parent of original parent immediately after original parent,
			// preserving ranges."
			for (i = nodeList.length - 1; i >= 0; i--) {
				movePreservingRanges(nodeList[i], originalParent.parentNode, 1 + getNodeIndex(originalParent), range);
			}

			// "If precedes line break is true, and the last member of node list
			// does not precede a line break, call createElement("br") on the
			// context object and insert the result immediately after the last
			// member of node list."
			if (precedesLineBreak_ && !precedesLineBreak(nodeList[nodeList.length - 1])) {
				nodeList[nodeList.length - 1].parentNode.insertBefore(document.createElement("br"), nodeList[nodeList.length - 1].nextSibling);
			}

			// "Remove extraneous line breaks at the end of original parent."
			removeExtraneousLineBreaksAtTheEndOf(originalParent);

			// "Abort these steps."
			return;
		}

		// "If the first child of original parent is not in node list:"
		if (!firstChildInNodeList) {
			// "Let cloned parent be the result of calling cloneNode(false) on
			// original parent."
			var clonedParent = originalParent.cloneNode(false);

			// "If original parent has an id attribute, unset it."
			originalParent.removeAttribute("id");

			// "Insert cloned parent into the parent of original parent immediately
			// before original parent."
			originalParent.parentNode.insertBefore(clonedParent, originalParent);

			// "While the previousSibling of the first member of node list is not
			// null, append the first child of original parent as the last child of
			// cloned parent, preserving ranges."
			while (nodeList[0].previousSibling) {
				movePreservingRanges(originalParent.firstChild, clonedParent, clonedParent.childNodes.length, range);
			}
		}

		// "For each node in node list, insert node into the parent of original
		// parent immediately before original parent, preserving ranges."
		for (i = 0; i < nodeList.length; i++) {
			movePreservingRanges(nodeList[i], originalParent.parentNode, getNodeIndex(originalParent), range);
		}

		// "If follows line break is true, and the first member of node list does
		// not follow a line break, call createElement("br") on the context object
		// and insert the result immediately before the first member of node list."
		if (followsLineBreak_ && !followsLineBreak(nodeList[0])) {
			nodeList[0].parentNode.insertBefore(document.createElement("br"), nodeList[0]);
		}

		// "If the last member of node list is an inline node other than a br, and
		// the first child of original parent is a br, and original parent is not
		// an inline node, remove the first child of original parent from original
		// parent."
		if (isInlineNode(nodeList[nodeList.length - 1]) && !isNamedHtmlElement(nodeList[nodeList.length - 1], "br") && isNamedHtmlElement(originalParent.firstChild, "br") && !isInlineNode(originalParent)) {
			originalParent.removeChild(originalParent.firstChild);
		}

		// "If original parent has no children:"
		if (!originalParent.hasChildNodes()) {
			// if the current range is collapsed and at the end of the originalParent.parentNode
			// the offset will not be available anymore after the next step (remove child)
			// that's why we need to fix the range to prevent a bogus offset
			if (originalParent.parentNode === range.startContainer && originalParent.parentNode === range.endContainer && range.startContainer === range.endContainer && range.startOffset === range.endOffset && originalParent.parentNode.childNodes.length === range.startOffset) {
				range.startOffset = originalParent.parentNode.childNodes.length - 1;
				range.endOffset = range.startOffset;
			}

			// "Remove original parent from its parent."
			originalParent.parentNode.removeChild(originalParent);

			// "If precedes line break is true, and the last member of node list
			// does not precede a line break, call createElement("br") on the
			// context object and insert the result immediately after the last
			// member of node list."
			if (precedesLineBreak_ && !precedesLineBreak(nodeList[nodeList.length - 1])) {
				nodeList[nodeList.length - 1].parentNode.insertBefore(document.createElement("br"), nodeList[nodeList.length - 1].nextSibling);
			}

			// "Otherwise, remove extraneous line breaks before original parent."
		} else {
			removeExtraneousLineBreaksBefore(originalParent);
		}

		// "If node list's last member's nextSibling is null, but its parent is not
		// null, remove extraneous line breaks at the end of node list's last
		// member's parent."
		if (!nodeList[nodeList.length - 1].nextSibling && nodeList[nodeList.length - 1].parentNode) {
			removeExtraneousLineBreaksAtTheEndOf(nodeList[nodeList.length - 1].parentNode);
		}
	}

	//@}
	///// The backColor command /////
	//@{
	commands.backcolor = {
		// Copy-pasted, same as hiliteColor
		action: function (value, range) {
			// Action is further copy-pasted, same as foreColor

			// "If value is not a valid CSS color, prepend "#" to it."
			//
			// "If value is still not a valid CSS color, or if it is currentColor,
			// abort these steps and do nothing."
			//
			// Cheap hack for testing, no attempt to be comprehensive.
			if (/^([0-9a-fA-F]{3}){1,2}$/.test(value)) {
				value = "#" + value;
			}
			if (!/^(rgba?|hsla?)\(.*\)$/.test(value) && !parseSimpleColor(value) && value.toLowerCase() != "transparent") {
				return;
			}

			// "Set the selection's value to value."
			setSelectionValue("backcolor", value, range);
		},
		standardInlineValueCommand: true,
		relevantCssProperty: "backgroundColor",
		equivalentValues: function (val1, val2) {
			// "Either both strings are valid CSS colors and have the same red,
			// green, blue, and alpha components, or neither string is a valid CSS
			// color."
			return normalizeColor(val1) === normalizeColor(val2);
		}
	};

	//@}
	///// The bold command /////
	//@{
	commands.bold = {
		action: function (value, range) {
			// "If queryCommandState("bold") returns true, set the selection's
			// value to "normal". Otherwise set the selection's value to "bold"."
			if (myQueryCommandState("bold", range)) {
				setSelectionValue("bold", "normal", range);
			} else {
				setSelectionValue("bold", "bold", range);
			}
		},
		inlineCommandActivatedValues: ["bold", "600", "700", "800", "900"],
		relevantCssProperty: "fontWeight",
		equivalentValues: function (val1, val2) {
			// "Either the two strings are equal, or one is "bold" and the other is
			// "700", or one is "normal" and the other is "400"."
			return val1 == val2 || (val1 == "bold" && val2 == "700") || (val1 == "700" && val2 == "bold") || (val1 == "normal" && val2 == "400") || (val1 == "400" && val2 == "normal");
		}
	};

	//@}
	///// The createLink command /////
	//@{
	commands.createlink = {
		action: function (value, range) {
			// "If value is the empty string, abort these steps and do nothing."
			if (value === "") {
				return;
			}

			// "For each editable a element that has an href attribute and is an
			// ancestor of some node effectively contained in the active range, set
			// that a element's href attribute to value."
			//
			// TODO: We don't actually do this in tree order, not that it matters
			// unless you're spying with mutation events.
			$_(getAllEffectivelyContainedNodes(getActiveRange())).forEach(function (node) {
				$_(getAncestors(node)).forEach(function (ancestor) {
					if (isEditable(ancestor) && isNamedHtmlElement(ancestor, 'a') && hasAttribute(ancestor, "href")) {
						ancestor.setAttribute("href", value);
					}
				});
			});

			// "Set the selection's value to value."
			setSelectionValue("createlink", value, range);
		},
		standardInlineValueCommand: true
	};

	//@}
	///// The fontName command /////
	//@{
	commands.fontname = {
		action: function (value, range) {
			// "Set the selection's value to value."
			setSelectionValue("fontname", value, range);
		},
		standardInlineValueCommand: true,
		relevantCssProperty: "fontFamily"
	};

	//@}
	///// The fontSize command /////
	//@{

	commands.fontsize = {
		action: function (value, range) {
			// "If value is the empty string, abort these steps and do nothing."
			if (value === "") {
				return;
			}

			value = normalizeFontSize(value);

			// "If value is not one of the strings "xx-small", "x-small", "small",
			// "medium", "large", "x-large", "xx-large", "xxx-large", and is not a
			// valid CSS absolute length, then abort these steps and do nothing."
			//
			// More cheap hacks to skip valid CSS absolute length checks.
			if (jQuery.inArray(value, ["xx-small", "x-small", "small", "medium", "large", "x-large", "xx-large", "xxx-large"]) == -1 && !/^[0-9]+(\.[0-9]+)?(cm|mm|in|pt|pc)$/.test(value)) {
				return;
			}

			// "Set the selection's value to value."
			setSelectionValue("fontsize", value, range);
		},
		indeterm: function () {
			// "True if among editable Text nodes that are effectively contained in
			// the active range, there are two that have distinct effective command
			// values.  Otherwise false."
			return $_(getAllEffectivelyContainedNodes(getActiveRange(), function (node) {
				return isEditable(node) && node.nodeType == $_.Node.TEXT_NODE;
			})).map(function (node) {
				return getEffectiveCommandValue(node, "fontsize");
			}, true).filter(function (value, i, arr) {
				return $_(arr.slice(0, i)).indexOf(value) == -1;
			}).length >= 2;
		},
		value: function (range) {
			// "Let pixel size be the effective command value of the first editable
			// Text node that is effectively contained in the active range, or if
			// there is no such node, the effective command value of the active
			// range's start node, in either case interpreted as a number of
			// pixels."
			var node = getAllEffectivelyContainedNodes(range, function (node) {
				return isEditable(node) && node.nodeType == $_.Node.TEXT_NODE;
			})[0];
			if (node === undefined) {
				node = range.startContainer;
			}
			var pixelSize = getEffectiveCommandValue(node, "fontsize");

			// "Return the legacy font size for pixel size."
			return getLegacyFontSize(pixelSize);
		},
		relevantCssProperty: "fontSize"
	};

	//@}
	///// The foreColor command /////
	//@{
	commands.forecolor = {
		action: function (value, range) {
			// Copy-pasted, same as backColor and hiliteColor

			// "If value is not a valid CSS color, prepend "#" to it."
			//
			// "If value is still not a valid CSS color, or if it is currentColor,
			// abort these steps and do nothing."
			//
			// Cheap hack for testing, no attempt to be comprehensive.
			if (/^([0-9a-fA-F]{3}){1,2}$/.test(value)) {
				value = "#" + value;
			}
			if (!/^(rgba?|hsla?)\(.*\)$/.test(value) && !parseSimpleColor(value) && value.toLowerCase() != "transparent") {
				return;
			}

			// "Set the selection's value to value."
			setSelectionValue("forecolor", value, range);
		},
		standardInlineValueCommand: true,
		relevantCssProperty: "color",
		equivalentValues: function (val1, val2) {
			// "Either both strings are valid CSS colors and have the same red,
			// green, blue, and alpha components, or neither string is a valid CSS
			// color."
			return normalizeColor(val1) === normalizeColor(val2);
		}
	};

	//@}
	///// The hiliteColor command /////
	//@{
	commands.hilitecolor = {
		// Copy-pasted, same as backColor
		action: function (value, range) {
			// Action is further copy-pasted, same as foreColor

			// "If value is not a valid CSS color, prepend "#" to it."
			//
			// "If value is still not a valid CSS color, or if it is currentColor,
			// abort these steps and do nothing."
			//
			// Cheap hack for testing, no attempt to be comprehensive.
			if (/^([0-9a-fA-F]{3}){1,2}$/.test(value)) {
				value = "#" + value;
			}
			if (!/^(rgba?|hsla?)\(.*\)$/.test(value) && !parseSimpleColor(value) && value.toLowerCase() != "transparent") {
				return;
			}

			// "Set the selection's value to value."
			setSelectionValue("hilitecolor", value, range);
		},
		indeterm: function () {
			// "True if among editable Text nodes that are effectively contained in
			// the active range, there are two that have distinct effective command
			// values.  Otherwise false."
			return $_(getAllEffectivelyContainedNodes(getActiveRange(), function (node) {
				return isEditable(node) && node.nodeType == $_.Node.TEXT_NODE;
			})).map(function (node) {
				return getEffectiveCommandValue(node, "hilitecolor");
			}, true).filter(function (value, i, arr) {
				return $_(arr.slice(0, i)).indexOf(value) == -1;
			}).length >= 2;
		},
		standardInlineValueCommand: true,
		relevantCssProperty: "backgroundColor",
		equivalentValues: function (val1, val2) {
			// "Either both strings are valid CSS colors and have the same red,
			// green, blue, and alpha components, or neither string is a valid CSS
			// color."
			return normalizeColor(val1) === normalizeColor(val2);
		}
	};

	//@}
	///// The italic command /////
	//@{
	commands.italic = {
		action: function (value, range) {
			// "If queryCommandState("italic") returns true, set the selection's
			// value to "normal". Otherwise set the selection's value to "italic"."
			if (myQueryCommandState("italic", range)) {
				setSelectionValue("italic", "normal", range);
			} else {
				setSelectionValue("italic", "italic", range);
			}
		},
		inlineCommandActivatedValues: ["italic", "oblique"],
		relevantCssProperty: "fontStyle"
	};

	//@}
	///// The removeFormat command /////
	//@{
	commands.removeformat = {
		action: function (value, range) {
			var newEnd, newStart, newNode;

			// "A removeFormat candidate is an editable HTML element with local
			// name "abbr", "acronym", "b", "bdi", "bdo", "big", "blink", "cite",
			// "code", "dfn", "em", "font", "i", "ins", "kbd", "mark", "nobr", "q",
			// "s", "samp", "small", "span", "strike", "strong", "sub", "sup",
			// "tt", "u", or "var"."
			function isRemoveFormatCandidate(node) {
				return isEditable(node) && isHtmlElementInArray(node, ["abbr", "acronym", "b", "bdi", "bdo", "big", "blink", "cite", "code", "dfn", "em", "font", "i", "ins", "kbd", "mark", "nobr", "q", "s", "samp", "small", "span", "strike", "strong", "sub", "sup", "tt", "u", "var"]);
			}

			// "Let elements to remove be a list of every removeFormat candidate
			// effectively contained in the active range."
			var elementsToRemove = getAllEffectivelyContainedNodes(getActiveRange(), isRemoveFormatCandidate);

			// "For each element in elements to remove:"
			$_(elementsToRemove).forEach(function (element) {
				// "While element has children, insert the first child of element
				// into the parent of element immediately before element,
				// preserving ranges."
				while (element.hasChildNodes()) {
					movePreservingRanges(element.firstChild, element.parentNode, getNodeIndex(element), getActiveRange());
				}

				// "Remove element from its parent."
				element.parentNode.removeChild(element);
			});

			// "If the active range's start node is an editable Text node, and its
			// start offset is neither zero nor its start node's length, call
			// splitText() on the active range's start node, with argument equal to
			// the active range's start offset. Then set the active range's start
			// node to the result, and its start offset to zero."
			if (isEditable(getActiveRange().startContainer) && getActiveRange().startContainer.nodeType == $_.Node.TEXT_NODE && getActiveRange().startOffset != 0 && getActiveRange().startOffset != getNodeLength(getActiveRange().startContainer)) {
				// Account for browsers not following range mutation rules
				if (getActiveRange().startContainer == getActiveRange().endContainer) {
					newEnd = getActiveRange().endOffset - getActiveRange().startOffset;
					newNode = getActiveRange().startContainer.splitText(getActiveRange().startOffset);
					getActiveRange().setStart(newNode, 0);
					getActiveRange().setEnd(newNode, newEnd);
				} else {
					getActiveRange().setStart(getActiveRange().startContainer.splitText(getActiveRange().startOffset), 0);
				}
			}

			// "If the active range's end node is an editable Text node, and its
			// end offset is neither zero nor its end node's length, call
			// splitText() on the active range's end node, with argument equal to
			// the active range's end offset."
			if (isEditable(getActiveRange().endContainer) && getActiveRange().endContainer.nodeType == $_.Node.TEXT_NODE && getActiveRange().endOffset != 0 && getActiveRange().endOffset != getNodeLength(getActiveRange().endContainer)) {
				// IE seems to mutate the range incorrectly here, so we need
				// correction here as well.  Have to be careful to set the range to
				// something not including the text node so that getActiveRange()
				// doesn't throw an exception due to a temporarily detached
				// endpoint.
				newStart = [getActiveRange().startContainer, getActiveRange().startOffset];
				newEnd = [getActiveRange().endContainer, getActiveRange().endOffset];
				getActiveRange().setEnd(document.documentElement, 0);
				newEnd[0].splitText(newEnd[1]);
				getActiveRange().setStart(newStart[0], newStart[1]);
				getActiveRange().setEnd(newEnd[0], newEnd[1]);
			}

			// "Let node list consist of all editable nodes effectively contained
			// in the active range."
			//
			// "For each node in node list, while node's parent is a removeFormat
			// candidate in the same editing host as node, split the parent of the
			// one-node list consisting of node."
			$_(getAllEffectivelyContainedNodes(getActiveRange(), isEditable)).forEach(function (node) {
				while (isRemoveFormatCandidate(node.parentNode) && inSameEditingHost(node.parentNode, node)) {
					splitParent([node], getActiveRange());
				}
			});

			// "For each of the entries in the following list, in the given order,
			// set the selection's value to null, with command as given."
			$_(["subscript", "bold", "fontname", "fontsize", "forecolor", "hilitecolor", "italic", "strikethrough", "underline"]).forEach(function (command) {
				setSelectionValue(command, null, range);
			});
		}
	};

	//@}
	///// The strikethrough command /////
	//@{
	commands.strikethrough = {
		action: function (value, range) {
			// "If queryCommandState("strikethrough") returns true, set the
			// selection's value to null. Otherwise set the selection's value to
			// "line-through"."
			if (myQueryCommandState("strikethrough", range)) {
				setSelectionValue("strikethrough", null, range);
			} else {
				setSelectionValue("strikethrough", "line-through", range);
			}
		},
		inlineCommandActivatedValues: ["line-through"]
	};

	//@}
	///// The subscript command /////
	//@{
	commands.subscript = {
		action: function (value, range) {
			// "Call queryCommandState("subscript"), and let state be the result."
			var state = myQueryCommandState("subscript", range);

			// "Set the selection's value to null."
			setSelectionValue("subscript", null, range);

			// "If state is false, set the selection's value to "subscript"."
			if (!state) {
				setSelectionValue("subscript", "subscript", range);
			}
		},
		indeterm: function () {
			// "True if either among editable Text nodes that are effectively
			// contained in the active range, there is at least one with effective
			// command value "subscript" and at least one with some other effective
			// command value; or if there is some editable Text node effectively
			// contained in the active range with effective command value "mixed".
			// Otherwise false."
			var nodes = getAllEffectivelyContainedNodes(getActiveRange(), function (node) {
				return isEditable(node) && node.nodeType == $_.Node.TEXT_NODE;
			});
			return (($_(nodes).some(function (node) { return getEffectiveCommandValue(node, "subscript") == "subscript"; })
					 && $_(nodes).some(function (node) { return getEffectiveCommandValue(node, "subscript") != "subscript"; }))
					|| $_(nodes).some(function (node) { return getEffectiveCommandValue(node, "subscript") == "mixed"; }));
		},
		inlineCommandActivatedValues: ["subscript"]
	};

	//@}
	///// The superscript command /////
	//@{
	commands.superscript = {
		action: function (value, range) {
			// "Call queryCommandState("superscript"), and let state be the
			// result."
			var state = myQueryCommandState("superscript", range);

			// "Set the selection's value to null."
			setSelectionValue("superscript", null, range);

			// "If state is false, set the selection's value to "superscript"."
			if (!state) {
				setSelectionValue("superscript", "superscript", range);
			}
		},
		indeterm: function () {
			// "True if either among editable Text nodes that are effectively
			// contained in the active range, there is at least one with effective
			// command value "superscript" and at least one with some other
			// effective command value; or if there is some editable Text node
			// effectively contained in the active range with effective command
			// value "mixed".  Otherwise false."
			var nodes = getAllEffectivelyContainedNodes(
				getActiveRange(),
				function (node) {
					return isEditable(node) && node.nodeType == $_.Node.TEXT_NODE;
				}
			);
			return (($_(nodes).some(function (node) { return getEffectiveCommandValue(node, "superscript") == "superscript"; })
					 && $_(nodes).some(function (node) { return getEffectiveCommandValue(node, "superscript") != "superscript"; }))
					|| $_(nodes).some(function (node) { return getEffectiveCommandValue(node, "superscript") == "mixed"; }));
		},
		inlineCommandActivatedValues: ["superscript"]
	};

	//@}
	///// The underline command /////
	//@{
	commands.underline = {
		action: function (value, range) {
			// "If queryCommandState("underline") returns true, set the selection's
			// value to null. Otherwise set the selection's value to "underline"."
			if (myQueryCommandState("underline", range)) {
				setSelectionValue("underline", null, range);
			} else {
				setSelectionValue("underline", "underline", range);
			}
		},
		inlineCommandActivatedValues: ["underline"]
	};

	//@}
	///// The unlink command /////
	//@{
	commands.unlink = {
		action: function () {
			// "Let hyperlinks be a list of every a element that has an href
			// attribute and is contained in the active range or is an ancestor of
			// one of its boundary points."
			//
			// As usual, take care to ensure it's tree order.  The correctness of
			// the following is left as an exercise for the reader.
			var range = getActiveRange();
			var hyperlinks = [];
			var node;
			for (node = range.startContainer; node; node = node.parentNode) {
				if (isNamedHtmlElement(node, 'A') && hasAttribute(node, "href")) {
					hyperlinks.unshift(node);
				}
			}
			for (node = range.startContainer; node != nextNodeDescendants(range.endContainer); node = nextNode(node)) {
				if (isNamedHtmlElement(node, 'A') && hasAttribute(node, "href") && (isContained(node, range) || isAncestor(node, range.endContainer) || node == range.endContainer)) {
					hyperlinks.push(node);
				}
			}

			// "Clear the value of each member of hyperlinks."
			var i;
			for (i = 0; i < hyperlinks.length; i++) {
				clearValue(hyperlinks[i], "unlink", range);
			}
		},
		standardInlineValueCommand: true
	};

	//@}

	/////////////////////////////////////
	///// Block formatting commands /////
	/////////////////////////////////////

	///// Block formatting command definitions /////
	//@{

	// "An indentation element is either a blockquote, or a div that has a style
	// attribute that sets "margin" or some subproperty of it."
	function isIndentationElement(node) {
		if (!isAnyHtmlElement(node)) {
			return false;
		}

		if (node.tagName == "BLOCKQUOTE") {
			return true;
		}

		if (node.tagName != "DIV") {
			return false;
		}

		if (typeof node.style.length !== 'undefined') {
			var i;
			for (i = 0; i < node.style.length; i++) {
				// Approximate check
				if (/^(-[a-z]+-)?margin/.test(node.style[i])) {
					return true;
				}
			}
		} else {
			var s;
			/*jslint forin: true*/ //not sure whether node.style.hasOwnProperty is valid
			for (s in node.style) {
				if (/^(-[a-z]+-)?margin/.test(s) && node.style[s] && node.style[s] !== 0) {
					return true;
				}
			}
			/*jslint forin: false*/
		}

		return false;
	}

	// "A simple indentation element is an indentation element that has no
	// attributes other than one or more of
	//
	//   * "a style attribute that sets no properties other than "margin", "border",
	//     "padding", or subproperties of those;
	//   * "a class attribute;
	//   * "a dir attribute."
	function isSimpleIndentationElement(node) {
		if (!isIndentationElement(node)) {
			return false;
		}

		if (node.tagName != "BLOCKQUOTE" && node.tagName != "DIV") {
			return false;
		}

		var i;
		for (i = 0; i < node.attributes.length; i++) {
			if (!isHtmlNamespace(node.attributes[i].namespaceURI) || jQuery.inArray(node.attributes[i].name, ["style", "class", "dir"]) == -1) {
				return false;
			}
		}

		if (typeof node.style.length !== 'undefined') {
			for (i = 0; i < node.style.length; i++) {
				// This is approximate, but it works well enough for my purposes.
				if (!/^(-[a-z]+-)?(margin|border|padding)/.test(node.style[i])) {
					return false;
				}
			}
		} else {
			var s;
			/*jslint forin: true*/ //not sure whether node.style.hasOwnProperty is valid
			for (s in node.style) {
				// This is approximate, but it works well enough for my purposes.
				if (!/^(-[a-z]+-)?(margin|border|padding)/.test(s) && node.style[s] && node.style[s] !== 0 && node.style[s] !== 'false') {
					return false;
				}
			}
			/*jslint forin: false*/
		}

		return true;
	}

	// "A non-list single-line container is an HTML element with local name
	// "address", "div", "h1", "h2", "h3", "h4", "h5", "h6", "listing", "p", "pre",
	// or "xmp"."
	function isNonListSingleLineContainer(node) {
		return isHtmlElementInArray(node, ["address", "div", "h1", "h2", "h3", "h4", "h5", "h6", "listing", "p", "pre", "xmp"]);
	}

	// "A single-line container is either a non-list single-line container, or an
	// HTML element with local name "li", "dt", or "dd"."
	function isSingleLineContainer(node) {
		return isNonListSingleLineContainer(node) || isHtmlElementInArray(node, ["li", "dt", "dd"]);
	}

	// "The default single-line container name is "p"."
	var defaultSingleLineContainerName = "p";

	//@}
	///// Check whether the given element is an end break /////
	//@{
	function isEndBreak(element) {
		return (isNamedHtmlElement(element, 'br') && element.parentNode.lastChild === element);
	}

	//@}
	///// Create an end break /////
	//@{
	function createEndBreak() {
		return document.createElement("br");
	}

	/**
	 * Ensure the container is editable
	 * E.g. when called for an empty paragraph or header, and the browser is not IE,
	 * we need to append a br (marked with class aloha-end-br)
	 * For IE7, there is a special behaviour that will append zero-width whitespace
	 * @param {DOMNode} container
	 */
	function ensureContainerEditable(container) {
		if (!container) {
			return;
		}

		if (isNamedHtmlElement(container.lastChild, "br")) {
			return;
		}

		if ($_(container.childNodes).some(isVisible)) {
			return;
		}

		if (!jQuery.browser.msie) {
			// for normal browsers, the end-br will do
			container.appendChild(createEndBreak());
		} else if (jQuery.browser.msie && jQuery.browser.version <= 7 && isHtmlElementInArray(container, ["p", "h1", "h2", "h3", "h4", "h5", "h6", "pre", "blockquote"])) {
			// for IE7, we need to insert a text node containing a single zero-width whitespace character
			if (!container.firstChild) {
				container.appendChild(document.createTextNode('\u200b'));
			}
		}
	}

	//@}
	///// Assorted block formatting command algorithms /////
	//@{

	function fixDisallowedAncestors(node, range) {
		var i;

		// "If node is not editable, abort these steps."
		if (!isEditable(node)) {
			return;
		}

		// "If node is not an allowed child of any of its ancestors in the same
		// editing host, and is not an HTML element with local name equal to the
		// default single-line container name:"
		if ($_(getAncestors(node)).every(function (ancestor) { return !inSameEditingHost(node, ancestor) || !isAllowedChild(node, ancestor); })
			    && !isHtmlElement_obsolete(node, defaultSingleLineContainerName)) {
			// "If node is a dd or dt, wrap the one-node list consisting of node,
			// with sibling criteria returning true for any dl with no attributes
			// and false otherwise, and new parent instructions returning the
			// result of calling createElement("dl") on the context object. Then
			// abort these steps."
			if (isHtmlElementInArray(node, ["dd", "dt"])) {
				wrap(
					[node],
					function (sibling) {
						return isNamedHtmlElement(sibling, 'dl') && !sibling.attributes.length;
					},
					function () {
						return document.createElement("dl");
					},
					range
				);
				return;
			}

			// "If node is not a prohibited paragraph child, abort these steps."
			if (!isProhibitedParagraphChild(node)) {
				return;
			}

			// "Set the tag name of node to the default single-line container name,
			// and let node be the result."
			node = setTagName(node, defaultSingleLineContainerName, range);

			ensureContainerEditable(node);

			// "Fix disallowed ancestors of node."
			fixDisallowedAncestors(node, range);

			// "Let descendants be all descendants of node."
			var descendants = getDescendants(node);

			// "Fix disallowed ancestors of each member of descendants."
			for (i = 0; i < descendants.length; i++) {
				fixDisallowedAncestors(descendants[i], range);
			}

			// "Abort these steps."
			return;
		}

		// "Record the values of the one-node list consisting of node, and let
		// values be the result."
		var values = recordValues([node]);
		var newStartOffset, newEndOffset;

		// "While node is not an allowed child of its parent, split the parent of
		// the one-node list consisting of node."
		while (!isAllowedChild(node, node.parentNode)) {
			// If the parent contains only this node and possibly empty text nodes, we rather want to unwrap the node, instead of splitting.
			// With splitting, we would get empty nodes, like:
			// split: <p><p>foo</p></p> -> <p></p><p>foo</p> (bad)
			// unwrap: <p><p>foo</p></p> -> <p>foo</p> (good)

			// First remove empty text nodes that are children of the parent and correct the range if necessary
			// we do this to have the node being the only child of its parent, so that we can replace the parent with the node
			for (i = node.parentNode.childNodes.length - 1; i >= 0; --i) {
				if (node.parentNode.childNodes[i].nodeType == 3 && node.parentNode.childNodes[i].data.length == 0) {
					// we remove the empty text node
					node.parentNode.removeChild(node.parentNode.childNodes[i]);

					// if the range points to somewhere behind the removed text node, we reduce the offset
					if (range.startContainer == node.parentNode && range.startOffset > i) {
						range.startOffset--;
					}
					if (range.endContainer == node.parentNode && range.endOffset > i) {
						range.endOffset--;
					}
				}
			}

			// now that the parent has only the node as child (because we
			// removed any existing empty text nodes), we can safely unwrap the
			// node's contents, and correct the range if necessary
			if (node.parentNode.childNodes.length == 1) {
				newStartOffset = range.startOffset;
				newEndOffset = range.endOffset;

				if (range.startContainer === node.parentNode && range.startOffset > getNodeIndex(node)) {
					// the node (1 element) will be replaced by its contents (contents().length elements)
					newStartOffset = range.startOffset + (jQuery(node).contents().length - 1);
				}
				if (range.endContainer === node.parentNode && range.endOffset > getNodeIndex(node)) {
					// the node (1 element) will be replaced by its contents (contents().length elements)
					newEndOffset = range.endOffset + (jQuery(node).contents().length - 1);
				}
				jQuery(node).contents().unwrap();
				range.startOffset = newStartOffset;
				range.endOffset = newEndOffset;
				// after unwrapping, we are done
				break;
			} else {
				// store the original parent
				var originalParent = node.parentNode;
				splitParent([node], range);
				// check whether the parent did not change, so the split did not work, e.g.
				// because we already reached the editing host itself.
				// this situation can occur, e.g. when we insert a paragraph into an contenteditable span
				// in such cases, we just unwrap the contents of the paragraph
				if (originalParent === node.parentNode) {
					// so we unwrap now
					newStartOffset = range.startOffset;
					newEndOffset = range.endOffset;

					if (range.startContainer === node.parentNode && range.startOffset > getNodeIndex(node)) {
						// the node (1 element) will be replaced by its contents (contents().length elements)
						newStartOffset = range.startOffset + (jQuery(node).contents().length - 1);
					}
					if (range.endContainer === node.parentNode && range.endOffset > getNodeIndex(node)) {
						// the node (1 element) will be replaced by its contents (contents().length elements)
						newEndOffset = range.endOffset + (jQuery(node).contents().length - 1);
					}
					jQuery(node).contents().unwrap();
					range.startOffset = newStartOffset;
					range.endOffset = newEndOffset;
					// after unwrapping, we are done
					break;
				}
			}
		}

		// "Restore the values from values."
		restoreValues(values, range);
	}

	/**
	 * This method "normalizes" sublists of the given item (which is supposed to be a LI):
	 * If sublists are found in the LI element, they are moved directly into the outer list.
	 * @param item item
	 * @param range range, which will be modified if necessary
	 */
	function normalizeSublists(item, range) {
		// "If item is not an li or it is not editable or its parent is not
		// editable, abort these steps."
		if (!isNamedHtmlElement(item, 'LI') || !isEditable(item) || !isEditable(item.parentNode)) {
			return;
		}

		// "Let new item be null."
		var newItem = null;

		function isOlUl(node) {
			return isHtmlElementInArray(node, ["OL", "UL"]);
		}

		// "While item has an ol or ul child:"
		while ($_(item.childNodes).some(isOlUl)) {
			// "Let child be the last child of item."
			var child = item.lastChild;

			// "If child is an ol or ul, or new item is null and child is a Text
			// node whose data consists of zero of more space characters:"
			if (isHtmlElementInArray(child, ["OL", "UL"]) || (!newItem && child.nodeType == $_.Node.TEXT_NODE && /^[ \t\n\f\r]*$/.test(child.data))) {
				// "Set new item to null."
				newItem = null;

				// "Insert child into the parent of item immediately following
				// item, preserving ranges."
				movePreservingRanges(child, item.parentNode, 1 + getNodeIndex(item), range);

				// "Otherwise:"
			} else {
				// "If new item is null, let new item be the result of calling
				// createElement("li") on the ownerDocument of item, then insert
				// new item into the parent of item immediately after item."
				if (!newItem) {
					newItem = item.ownerDocument.createElement("li");
					item.parentNode.insertBefore(newItem, item.nextSibling);
				}

				// "Insert child into new item as its first child, preserving
				// ranges."
				movePreservingRanges(child, newItem, 0, range);
			}
		}
	}

	/**
	 * This method is the exact opposite of normalizeSublists.
	 * List nodes directly nested into each other are corrected to be nested in li elements (so that the resulting lists conform the html5 specification)
	 * @param item list node
	 * @param range range, which is preserved when modifying the list
	 */
	function unNormalizeSublists(item, range) {
		// "If item is not an ol or ol or it is not editable or its parent is not
		// editable, abort these steps."
		if (!isHtmlElementInArray(item, ["OL", "UL"]) || !isEditable(item)) {
			return;
		}

		var $list = jQuery(item);
		$list.children("ol,ul").each(function (index, sublist) {
			if (isNamedHtmlElement(sublist.previousSibling, "LI")) {
				// move the sublist into the LI
				movePreservingRanges(sublist, sublist.previousSibling, sublist.previousSibling.childNodes.length, range);
			}
		});
	}

	//@}
	///// Block-extending a range /////
	//@{

	function blockExtend(range) {
		// "Let start node, start offset, end node, and end offset be the start
		// and end nodes and offsets of the range."
		var startNode = range.startContainer;
		var startOffset = range.startOffset;
		var endNode = range.endContainer;
		var endOffset = range.endOffset;

		// "If some ancestor container of start node is an li, set start offset to
		// the index of the last such li in tree order, and set start node to that
		// li's parent."
		var liAncestors = $_(getAncestors(startNode).concat(startNode)).filter(function (ancestor) { return isNamedHtmlElement(ancestor, 'li'); }).slice(-1);
		if (liAncestors.length) {
			startOffset = getNodeIndex(liAncestors[0]);
			startNode = liAncestors[0].parentNode;
		}

		// "If (start node, start offset) is not a block start point, repeat the
		// following steps:"
		if (!isBlockStartPoint(startNode, startOffset)) {
			do {
				// "If start offset is zero, set it to start node's index, then set
				// start node to its parent."
				if (startOffset == 0) {
					startOffset = getNodeIndex(startNode);
					startNode = startNode.parentNode;

					// "Otherwise, subtract one from start offset."
				} else {
					startOffset--;
				}

				// "If (start node, start offset) is a block boundary point, break from
				// this loop."
			} while (!isBlockBoundaryPoint(startNode, startOffset));
		}

		// "While start offset is zero and start node's parent is not null, set
		// start offset to start node's index, then set start node to its parent."
		while (startOffset == 0 && startNode.parentNode) {
			startOffset = getNodeIndex(startNode);
			startNode = startNode.parentNode;
		}

		// "If some ancestor container of end node is an li, set end offset to one
		// plus the index of the last such li in tree order, and set end node to
		// that li's parent."
		liAncestors = $_(getAncestors(endNode).concat(endNode)).filter(function (ancestor) { return isNamedHtmlElement(ancestor, 'li'); }).slice(-1);
		if (liAncestors.length) {
			endOffset = 1 + getNodeIndex(liAncestors[0]);
			endNode = liAncestors[0].parentNode;
		}

		// "If (end node, end offset) is not a block end point, repeat the
		// following steps:"
		if (!isBlockEndPoint(endNode, endOffset)) {
			do {
				// "If end offset is end node's length, set it to one plus end node's
				// index, then set end node to its parent."
				if (endOffset == getNodeLength(endNode)) {
					endOffset = 1 + getNodeIndex(endNode);
					endNode = endNode.parentNode;

					// "Otherwise, add one to end offset.
				} else {
					endOffset++;
				}

				// "If (end node, end offset) is a block boundary point, break from
				// this loop."
			} while (!isBlockBoundaryPoint(endNode, endOffset));
		}

		// "While end offset is end node's length and end node's parent is not
		// null, set end offset to one plus end node's index, then set end node to
		// its parent."
		while (endOffset == getNodeLength(endNode) && endNode.parentNode) {
			endOffset = 1 + getNodeIndex(endNode);
			endNode = endNode.parentNode;
		}

		// "Let new range be a new range whose start and end nodes and offsets
		// are start node, start offset, end node, and end offset."
		var newRange = Aloha.createRange();
		newRange.setStart(startNode, startOffset);
		newRange.setEnd(endNode, endOffset);

		// "Return new range."
		return newRange;
	}

	function getSelectionListState() {
		// "Block-extend the active range, and let new range be the result."
		var newRange = blockExtend(getActiveRange());

		// "Let node list be a list of nodes, initially empty."
		//
		// "For each node contained in new range, append node to node list if the
		// last member of node list (if any) is not an ancestor of node; node is
		// editable; node is not an indentation element; and node is either an ol
		// or ul, or the child of an ol or ul, or an allowed child of "li"."
		var nodeList = getContainedNodes(newRange, function (node) {
			return isEditable(node) && !isIndentationElement(node) && (isHtmlElementInArray(node, ["ol", "ul"]) || isHtmlElementInArray(node.parentNode, ["ol", "ul"]) || isAllowedChild(node, "li"));
		});

		// "If node list is empty, return "none"."
		if (!nodeList.length) {
			return "none";
		}

		// "If every member of node list is either an ol or the child of an ol or
		// the child of an li child of an ol, and none is a ul or an ancestor of a
		// ul, return "ol"."
		if ($_(nodeList).every(function (node) { return (isNamedHtmlElement(node, 'ol')
														 || isNamedHtmlElement(node.parentNode, "ol")
														 || (isNamedHtmlElement(node.parentNode, "li")
															 && isNamedHtmlElement(node.parentNode.parentNode, "ol"))); })
			    && !$_(nodeList).some(function (node) { return isNamedHtmlElement(node, 'ul') || (node.querySelector && node.querySelector("ul")); })) {
			return "ol";
		}

		// "If every member of node list is either a ul or the child of a ul or the
		// child of an li child of a ul, and none is an ol or an ancestor of an ol,
		// return "ul"."
		if ($_(nodeList).every(function (node) { return (isNamedHtmlElement(node, 'ul')
														 || isNamedHtmlElement(node.parentNode, "ul")
														 || (isNamedHtmlElement(node.parentNode, "li")
															 && isNamedHtmlElement(node.parentNode.parentNode, "ul"))); })
			    && !$_(nodeList).some(function (node) { return isNamedHtmlElement(node, 'ol') || (node.querySelector && node.querySelector("ol")); })) {
			return "ul";
		}

		var hasOl = $_(nodeList).some(function (node) {
			return (isNamedHtmlElement(node, 'ol')
					|| isNamedHtmlElement(node.parentNode, "ol")
					|| (node.querySelector && node.querySelector("ol"))
					|| (isNamedHtmlElement(node.parentNode, "li")
						&& isNamedHtmlElement(node.parentNode.parentNode, "ol")));
		});
		var hasUl = $_(nodeList).some(function (node) {
			return (isNamedHtmlElement(node, 'ul')
					|| isNamedHtmlElement(node.parentNode, "ul")
					|| (node.querySelector && node.querySelector("ul"))
					|| (isNamedHtmlElement(node.parentNode, "li")
						&& isNamedHtmlElement(node.parentNode.parentNode, "ul")));
		});
		// "If some member of node list is either an ol or the child or ancestor of
		// an ol or the child of an li child of an ol, and some member of node list
		// is either a ul or the child or ancestor of a ul or the child of an li
		// child of a ul, return "mixed"."
		if (hasOl && hasUl) {
			return "mixed";
		}

		// "If some member of node list is either an ol or the child or ancestor of
		// an ol or the child of an li child of an ol, return "mixed ol"."
		if (hasOl) {
			return "mixed ol";
		}

		// "If some member of node list is either a ul or the child or ancestor of
		// a ul or the child of an li child of a ul, return "mixed ul"."
		if (hasUl) {
			return "mixed ul";
		}

		// "Return "none"."
		return "none";
	}

	function getAlignmentValue(node) {
		// "While node is neither null nor an Element, or it is an Element but its
		// "display" property has resolved value "inline" or "none", set node to
		// its parent."
		while ((node && node.nodeType != $_.Node.ELEMENT_NODE) || (node.nodeType == $_.Node.ELEMENT_NODE && jQuery.inArray($_.getComputedStyle(node).display, ["inline", "none"]) != -1)) {
			node = node.parentNode;
		}

		// "If node is not an Element, return "left"."
		if (!node || node.nodeType != $_.Node.ELEMENT_NODE) {
			return "left";
		}

		var resolvedValue = $_.getComputedStyle(node).textAlign
		// Hack around browser non-standardness
			.replace(/^-(moz|webkit)-/, "").replace(/^auto$/, "start");

		// "If node's "text-align" property has resolved value "start", return
		// "left" if the directionality of node is "ltr", "right" if it is "rtl"."
		if (resolvedValue == "start") {
			return getDirectionality(node) == "ltr" ? "left" : "right";
		}

		// "If node's "text-align" property has resolved value "end", return
		// "right" if the directionality of node is "ltr", "left" if it is "rtl"."
		if (resolvedValue == "end") {
			return getDirectionality(node) == "ltr" ? "right" : "left";
		}

		// "If node's "text-align" property has resolved value "center", "justify",
		// "left", or "right", return that value."
		if (jQuery.inArray(resolvedValue, ["center", "justify", "left", "right"]) != -1) {
			return resolvedValue;
		}

		// "Return "left"."
		return "left";
	}

	//@}
	///// Recording and restoring overrides /////
	//@{

	function recordCurrentOverrides(range) {
		// "Let overrides be a list of (string, string or boolean) ordered pairs,
		// initially empty."
		var overrides = [];

		// "If there is a value override for "createLink", add ("createLink", value
		// override for "createLink") to overrides."
		if (getValueOverride("createlink", range) !== undefined) {
			overrides.push(["createlink", getValueOverride("createlink", range)]);
		}

		// "For each command in the list "bold", "italic", "strikethrough",
		// "subscript", "superscript", "underline", in order: if there is a state
		// override for command, add (command, command's state override) to
		// overrides."
		$_(["bold", "italic", "strikethrough", "subscript", "superscript", "underline"]).forEach(function (command) {
			if (getStateOverride(command, range) !== undefined) {
				overrides.push([command, getStateOverride(command, range)]);
			}
		});

		// "For each command in the list "fontName", "fontSize", "foreColor",
		// "hiliteColor", in order: if there is a value override for command, add
		// (command, command's value override) to overrides."
		$_(["fontname", "fontsize", "forecolor", "hilitecolor"]).forEach(function (command) {
			if (getValueOverride(command, range) !== undefined) {
				overrides.push([command, getValueOverride(command, range)]);
			}
		});

		// "Return overrides."
		return overrides;
	}

	function recordCurrentStatesAndValues(range) {
		// "Let overrides be a list of (string, string or boolean) ordered pairs,
		// initially empty."
		var overrides = [];

		// "Let node be the first editable Text node effectively contained in the
		// active range, or null if there is none."
		var node = $_(getAllEffectivelyContainedNodes(range)).filter(function (node) {
			return isEditable(node) && node.nodeType == $_.Node.TEXT_NODE;
		})[0];

		// "If node is null, return overrides."
		if (!node) {
			return overrides;
		}

		// "Add ("createLink", value for "createLink") to overrides."
		overrides.push(["createlink", commands.createlink.value(range)]);

		// "For each command in the list "bold", "italic", "strikethrough",
		// "subscript", "superscript", "underline", in order: if node's effective
		// command value for command is one of its inline command activated values,
		// add (command, true) to overrides, and otherwise add (command, false) to
		// overrides."
		$_(["bold", "italic", "strikethrough", "subscript", "superscript", "underline"]).forEach(function (command) {
			if ($_(commands[command].inlineCommandActivatedValues).indexOf(getEffectiveCommandValue(node, command)) != -1) {
				overrides.push([command, true]);
			} else {
				overrides.push([command, false]);
			}
		});

		// "For each command in the list "fontName", "foreColor", "hiliteColor", in
		// order: add (command, command's value) to overrides."

		$_(["fontname", "fontsize", "forecolor", "hilitecolor"]).forEach(function (command) {
			overrides.push([command, commands[command].value(range)]);
		});

		// "Add ("fontSize", node's effective command value for "fontSize") to
		// overrides."
		overrides.push(["fontsize", getEffectiveCommandValue(node, "fontsize")]);

		// "Return overrides."
		return overrides;
	}

	function restoreStatesAndValues(overrides, range) {
		var i;
		var command;
		var override;
		// "Let node be the first editable Text node effectively contained in the
		// active range, or null if there is none."
		var node = $_(getAllEffectivelyContainedNodes(range)).filter(function (node) {
			return isEditable(node) && node.nodeType == $_.Node.TEXT_NODE;
		})[0];

		function isEditableTextNode(node) {
			return isEditable(node) && node.nodeType == $_.Node.TEXT_NODE;
		}

		// "If node is not null, then for each (command, override) pair in
		// overrides, in order:"
		if (node) {

			for (i = 0; i < overrides.length; i++) {
				command = overrides[i][0];
				override = overrides[i][1];

				// "If override is a boolean, and queryCommandState(command)
				// returns something different from override, call
				// execCommand(command)."
				if (typeof override == "boolean" && myQueryCommandState(command, range) != override) {
					myExecCommand(command);

					// "Otherwise, if override is a string, and command is not
					// "fontSize", and queryCommandValue(command) returns something not
					// equivalent to override, call execCommand(command, false,
					// override)."
				} else if (typeof override == "string" && command != "fontsize" && !areEquivalentValues(command, myQueryCommandValue(command, range), override)) {
					myExecCommand(command, false, override, range);

					// "Otherwise, if override is a string; and command is "fontSize";
					// and either there is a value override for "fontSize" that is not
					// equal to override, or there is no value override for "fontSize"
					// and node's effective command value for "fontSize" is not loosely
					// equivalent to override: call execCommand("fontSize", false,
					// override)."
				} else if (typeof override == "string"
						   && command == "fontsize"
						   && ((getValueOverride("fontsize", range) !== undefined
								&& getValueOverride("fontsize", range) !== override)
							   || (getValueOverride("fontsize", range) === undefined
								   && !areLooselyEquivalentValues(command, getEffectiveCommandValue(node, "fontsize"), override)))) {
					myExecCommand("fontsize", false, override, range);

					// "Otherwise, continue this loop from the beginning."
				} else {
					continue;
				}

				// "Set node to the first editable Text node effectively contained
				// in the active range, if there is one."
				node = $_(getAllEffectivelyContainedNodes(range)).filter(isEditableTextNode)[0] || node;
			}

			// "Otherwise, for each (command, override) pair in overrides, in order:"
		} else {
			for (i = 0; i < overrides.length; i++) {
				command = overrides[i][0];
				override = overrides[i][1];

				// "If override is a boolean, set the state override for command to
				// override."
				if (typeof override == "boolean") {
					setStateOverride(command, override, range);
				}

				// "If override is a string, set the value override for command to
				// override."
				if (typeof override == "string") {
					setValueOverride(command, override, range);
				}
			}
		}
	}

	//@}
	///// Canonical space sequences /////
	//@{

	function canonicalSpaceSequence(n, nonBreakingStart, nonBreakingEnd) {
		// "If n is zero, return the empty string."
		if (n == 0) {
			return "";
		}

		// "If n is one and both non-breaking start and non-breaking end are false,
		// return a single space (U+0020)."
		if (n == 1 && !nonBreakingStart && !nonBreakingEnd) {
			return " ";
		}

		// "If n is one, return a single non-breaking space (U+00A0)."
		if (n == 1) {
			return "\xa0";
		}

		// "Let buffer be the empty string."
		var buffer = "";

		// "If non-breaking start is true, let repeated pair be U+00A0 U+0020.
		// Otherwise, let it be U+0020 U+00A0."
		var repeatedPair;
		if (nonBreakingStart) {
			repeatedPair = "\xa0 ";
		} else {
			repeatedPair = " \xa0";
		}

		// "While n is greater than three, append repeated pair to buffer and
		// subtract two from n."
		while (n > 3) {
			buffer += repeatedPair;
			n -= 2;
		}

		// "If n is three, append a three-element string to buffer depending on
		// non-breaking start and non-breaking end:"
		if (n == 3) {
			buffer += !nonBreakingStart && !nonBreakingEnd ? " \xa0 " : nonBreakingStart && !nonBreakingEnd ? "\xa0\xa0 " : !nonBreakingStart && nonBreakingEnd ? " \xa0\xa0" : nonBreakingStart && nonBreakingEnd ? "\xa0 \xa0" : "impossible";

			// "Otherwise, append a two-element string to buffer depending on
			// non-breaking start and non-breaking end:"
		} else {
			buffer += !nonBreakingStart && !nonBreakingEnd ? "\xa0 " : nonBreakingStart && !nonBreakingEnd ? "\xa0 " : !nonBreakingStart && nonBreakingEnd ? " \xa0" : nonBreakingStart && nonBreakingEnd ? "\xa0\xa0" : "impossible";
		}

		// "Return buffer."
		return buffer;
	}

	function canonicalizeWhitespace(node, offset) {
		// "If node is neither editable nor an editing host, abort these steps."
		if (!isEditable(node) && !isEditingHost(node)) {
			return;
		}

		// "Let start node equal node and let start offset equal offset."
		var startNode = node;
		var startOffset = offset;

		// "Repeat the following steps:"
		while (true) {
			// "If start node has a child in the same editing host with index start
			// offset minus one, set start node to that child, then set start
			// offset to start node's length."
			if (0 <= startOffset - 1 && inSameEditingHost(startNode, startNode.childNodes[startOffset - 1])) {
				startNode = startNode.childNodes[startOffset - 1];
				startOffset = getNodeLength(startNode);

				// "Otherwise, if start offset is zero and start node does not follow a
				// line break and start node's parent is in the same editing host, set
				// start offset to start node's index, then set start node to its
				// parent."
			} else if (startOffset == 0 && !followsLineBreak(startNode) && inSameEditingHost(startNode, startNode.parentNode)) {
				startOffset = getNodeIndex(startNode);
				startNode = startNode.parentNode;

				// "Otherwise, if start node is a Text node and its parent's resolved
				// value for "white-space" is neither "pre" nor "pre-wrap" and start
				// offset is not zero and the (start offset − 1)st element of start
				// node's data is a space (0x0020) or non-breaking space (0x00A0),
				// subtract one from start offset."
			} else if (startNode.nodeType == $_.Node.TEXT_NODE && jQuery.inArray($_.getComputedStyle(startNode.parentNode).whiteSpace, ["pre", "pre-wrap"]) == -1 && startOffset != 0 && /[ \xa0]/.test(startNode.data[startOffset - 1])) {
				startOffset--;

				// "Otherwise, break from this loop."
			} else {
				break;
			}
		}

		// "Let end node equal start node and end offset equal start offset."
		var endNode = startNode;
		var endOffset = startOffset;

		// "Let length equal zero."
		var length = 0;

		// "Let follows space be false."
		var followsSpace = false;

		// "Repeat the following steps:"
		while (true) {
			// "If end node has a child in the same editing host with index end
			// offset, set end node to that child, then set end offset to zero."
			if (endOffset < endNode.childNodes.length && inSameEditingHost(endNode, endNode.childNodes[endOffset])) {
				endNode = endNode.childNodes[endOffset];
				endOffset = 0;

				// "Otherwise, if end offset is end node's length and end node does not
				// precede a line break and end node's parent is in the same editing
				// host, set end offset to one plus end node's index, then set end node
				// to its parent."
			} else if (endOffset == getNodeLength(endNode) && !precedesLineBreak(endNode) && inSameEditingHost(endNode, endNode.parentNode)) {
				endOffset = 1 + getNodeIndex(endNode);
				endNode = endNode.parentNode;

				// "Otherwise, if end node is a Text node and its parent's resolved
				// value for "white-space" is neither "pre" nor "pre-wrap" and end
				// offset is not end node's length and the end offsetth element of
				// end node's data is a space (0x0020) or non-breaking space (0x00A0):"
			} else if (endNode.nodeType == $_.Node.TEXT_NODE && jQuery.inArray($_.getComputedStyle(endNode.parentNode).whiteSpace, ["pre", "pre-wrap"]) == -1 && endOffset != getNodeLength(endNode) && /[ \xa0]/.test(endNode.data[endOffset])) {
				// "If follows space is true and the end offsetth element of end
				// node's data is a space (0x0020), call deleteData(end offset, 1)
				// on end node, then continue this loop from the beginning."
				if (followsSpace && " " == endNode.data[endOffset]) {
					endNode.deleteData(endOffset, 1);
					continue;
				}

				// "Set follows space to true if the end offsetth element of end
				// node's data is a space (0x0020), false otherwise."
				followsSpace = " " == endNode.data[endOffset];

				// "Add one to end offset."
				endOffset++;

				// "Add one to length."
				length++;

				// "Otherwise, break from this loop."
			} else {
				break;
			}
		}

		// "Let replacement whitespace be the canonical space sequence of length
		// length. non-breaking start is true if start offset is zero and start
		// node follows a line break, and false otherwise. non-breaking end is true
		// if end offset is end node's length and end node precedes a line break,
		// and false otherwise."
		var replacementWhitespace = canonicalSpaceSequence(length, startOffset == 0 && followsLineBreak(startNode), endOffset == getNodeLength(endNode) && precedesLineBreak(endNode));

		// "While (start node, start offset) is before (end node, end offset):"
		while (getPosition(startNode, startOffset, endNode, endOffset) == "before") {
			// "If start node has a child with index start offset, set start node
			// to that child, then set start offset to zero."
			if (startOffset < startNode.childNodes.length) {
				startNode = startNode.childNodes[startOffset];
				startOffset = 0;

				// "Otherwise, if start node is not a Text node or if start offset is
				// start node's length, set start offset to one plus start node's
				// index, then set start node to its parent."
			} else if (startNode.nodeType != $_.Node.TEXT_NODE || startOffset == getNodeLength(startNode)) {
				startOffset = 1 + getNodeIndex(startNode);
				startNode = startNode.parentNode;

				// "Otherwise:"
			} else {
				// "Remove the first element from replacement whitespace, and let
				// element be that element."
				var element = replacementWhitespace[0];
				replacementWhitespace = replacementWhitespace.slice(1);

				// "If element is not the same as the start offsetth element of
				// start node's data:"
				if (element != startNode.data[startOffset]) {
					// "Call insertData(start offset, element) on start node."
					startNode.insertData(startOffset, element);

					// "Call deleteData(start offset + 1, 1) on start node."
					startNode.deleteData(startOffset + 1, 1);
				}

				// "Add one to start offset."
				startOffset++;
			}
		}
	}

	//@}
	///// Deleting the contents of a range /////
	//@{

	function deleteContents(arg1, arg2, arg3, arg4, arg5) {
		// We accept several different calling conventions:
		//
		// 1) A single argument, which is a range.
		//
		// 2) Two arguments, the first being a range and the second flags.
		//
		// 3) Four arguments, the start and end of a range.
		//
		// 4) Five arguments, the start and end of a range plus flags.
		//
		// The flags argument is a dictionary that can have up to two keys,
		// blockMerging and stripWrappers, whose corresponding values are
		// interpreted as boolean.  E.g., {stripWrappers: false}.
		var range;
		var flags = {};
		var i;

		if (arguments.length < 3) {
			range = arg1;
		} else {
			range = Aloha.createRange();
			range.setStart(arg1, arg2);
			range.setEnd(arg3, arg4);
		}
		if (arguments.length == 2) {
			flags = arg2;
		}
		if (arguments.length == 5) {
			flags = arg5;
		}

		var blockMerging = null != flags.blockMerging ? !!flags.blockMerging : true;
		var stripWrappers = null != flags.stripWrappers ? !!flags.stripWrappers : true;

		// "If range is null, abort these steps and do nothing."
		if (!range) {
			return;
		}

		// "Let start node, start offset, end node, and end offset be range's start
		// and end nodes and offsets."
		var startNode = range.startContainer;
		var startOffset = range.startOffset;
		var endNode = range.endContainer;
		var endOffset = range.endOffset;
		var referenceNode;

		// "While start node has at least one child:"
		while (startNode.hasChildNodes()) {
			// "If start offset is start node's length, and start node's parent is
			// in the same editing host, and start node is an inline node, set
			// start offset to one plus the index of start node, then set start
			// node to its parent and continue this loop from the beginning."
			if (startOffset == getNodeLength(startNode) && inSameEditingHost(startNode, startNode.parentNode) && isInlineNode(startNode)) {
				startOffset = 1 + getNodeIndex(startNode);
				startNode = startNode.parentNode;
				continue;
			}

			// "If start offset is start node's length, break from this loop."
			if (startOffset == getNodeLength(startNode)) {
				break;
			}

			// "Let reference node be the child of start node with index equal to
			// start offset."
			referenceNode = startNode.childNodes[startOffset];

			// "If reference node is a block node or an Element with no children,
			// or is neither an Element nor a Text node, break from this loop."
			if (isBlockNode(referenceNode) || (referenceNode.nodeType == $_.Node.ELEMENT_NODE && !referenceNode.hasChildNodes()) || (referenceNode.nodeType != $_.Node.ELEMENT_NODE && referenceNode.nodeType != $_.Node.TEXT_NODE)) {
				break;
			}

			// "Set start node to reference node and start offset to 0."
			startNode = referenceNode;
			startOffset = 0;
		}

		// "While end node has at least one child:"
		while (endNode.hasChildNodes()) {
			// "If end offset is 0, and end node's parent is in the same editing
			// host, and end node is an inline node, set end offset to the index of
			// end node, then set end node to its parent and continue this loop
			// from the beginning."
			if (endOffset == 0 && inSameEditingHost(endNode, endNode.parentNode) && isInlineNode(endNode)) {
				endOffset = getNodeIndex(endNode);
				endNode = endNode.parentNode;
				continue;
			}

			// "If end offset is 0, break from this loop."
			if (endOffset == 0) {
				break;
			}

			// "Let reference node be the child of end node with index equal to end
			// offset minus one."
			referenceNode = endNode.childNodes[endOffset - 1];

			// "If reference node is a block node or an Element with no children,
			// or is neither an Element nor a Text node, break from this loop."
			if (isBlockNode(referenceNode) || (referenceNode.nodeType == $_.Node.ELEMENT_NODE && !referenceNode.hasChildNodes()) || (referenceNode.nodeType != $_.Node.ELEMENT_NODE && referenceNode.nodeType != $_.Node.TEXT_NODE)) {
				break;
			}

			// "Set end node to reference node and end offset to the length of
			// reference node."
			endNode = referenceNode;
			endOffset = getNodeLength(referenceNode);
		}

		// "If (end node, end offset) is not after (start node, start offset), set
		// range's end to its start and abort these steps."
		if (getPosition(endNode, endOffset, startNode, startOffset) !== "after") {
			range.setEnd(range.startContainer, range.startOffset);
			return;
		}

		// "If start node is a Text node and start offset is 0, set start offset to
		// the index of start node, then set start node to its parent."
		// Commented out for unknown reason
		//if (startNode.nodeType == $_.Node.TEXT_NODE && startOffset == 0 && startNode != endNode) {
		//		startOffset = getNodeIndex(startNode);
		//		startNode = startNode.parentNode;
		//}

		// "If end node is a Text node and end offset is its length, set end offset
		// to one plus the index of end node, then set end node to its parent."
		if (endNode.nodeType == $_.Node.TEXT_NODE && endOffset == getNodeLength(endNode) && startNode != endNode) {
			endOffset = 1 + getNodeIndex(endNode);
			endNode = endNode.parentNode;
		}

		// "Set range's start to (start node, start offset) and its end to (end
		// node, end offset)."
		range.setStart(startNode, startOffset);
		range.setEnd(endNode, endOffset);

		// "Let start block be the start node of range."
		var startBlock = range.startContainer;

		// "While start block's parent is in the same editing host and start block
		// is an inline node, set start block to its parent."
		while (inSameEditingHost(startBlock, startBlock.parentNode) && isInlineNode(startBlock)) {
			startBlock = startBlock.parentNode;
		}

		// "If start block is neither a block node nor an editing host, or "span"
		// is not an allowed child of start block, or start block is a td or th,
		// set start block to null."
		if ((!isBlockNode(startBlock) && !isEditingHost(startBlock)) || !isAllowedChild("span", startBlock) || isHtmlElementInArray(startBlock, ["td", "th"])) {
			startBlock = null;
		}

		// "Let end block be the end node of range."
		var endBlock = range.endContainer;

		// "While end block's parent is in the same editing host and end block is
		// an inline node, set end block to its parent."
		while (inSameEditingHost(endBlock, endBlock.parentNode) && isInlineNode(endBlock)) {
			endBlock = endBlock.parentNode;
		}

		// "If end block is neither a block node nor an editing host, or "span" is
		// not an allowed child of end block, or end block is a td or th, set end
		// block to null."
		if ((!isBlockNode(endBlock) && !isEditingHost(endBlock)) || !isAllowedChild("span", endBlock) || isHtmlElementInArray(endBlock, ["td", "th"])) {
			endBlock = null;
		}

		// "Record current states and values, and let overrides be the result."
		var overrides = recordCurrentStatesAndValues(range);
		var parent_;
		// "If start node and end node are the same, and start node is an editable
		// Text node:"
		if (startNode == endNode && isEditable(startNode) && startNode.nodeType == $_.Node.TEXT_NODE) {
			// "Let parent be the parent of node."
			parent_ = startNode.parentNode;

			// "Call deleteData(start offset, end offset − start offset) on start
			// node."
			startNode.deleteData(startOffset, endOffset - startOffset);

			// if deleting the text moved two spaces together, we replace the left one by a &nbsp;, which makes the two spaces a visible
			// two space sequence
			if (startOffset > 0 && startNode.data.substr(startOffset - 1, 1) === ' ' && startOffset < startNode.data.length && startNode.data.substr(startOffset, 1) === ' ') {
				startNode.replaceData(startOffset - 1, 1, '\xa0');
			}

			// "Canonicalize whitespace at (start node, start offset)."
			canonicalizeWhitespace(startNode, startOffset);

			// "Set range's end to its start."
			// Ok, also set the range's start to its start, because modifying the text 
			// might have somehow corrupted the range
			range.setStart(range.startContainer, range.startOffset);
			range.setEnd(range.startContainer, range.startOffset);

			// "Restore states and values from overrides."
			restoreStatesAndValues(overrides, range);

			// "If parent is editable or an editing host, is not an inline node,
			// and has no children, call createElement("br") on the context object
			// and append the result as the last child of parent."
			// only do this, if the offsetHeight is 0
			if ((isEditable(parent_) || isEditingHost(parent_)) && !isInlineNode(parent_)) {
				// TODO is this always correct?
				ensureContainerEditable(parent_);
			}

			// "Abort these steps."
			return;
		}

		// "If start node is an editable Text node, call deleteData() on it, with
		// start offset as the first argument and (length of start node − start
		// offset) as the second argument."
		if (isEditable(startNode) && startNode.nodeType == $_.Node.TEXT_NODE) {
			startNode.deleteData(startOffset, getNodeLength(startNode) - startOffset);
		}

		// "Let node list be a list of nodes, initially empty."
		//
		// "For each node contained in range, append node to node list if the last
		// member of node list (if any) is not an ancestor of node; node is
		// editable; and node is not a thead, tbody, tfoot, tr, th, or td."
		var nodeList = getContainedNodes(
			range,
			function (node) {
				return isEditable(node) && !isHtmlElementInArray(node, ["thead", "tbody", "tfoot", "tr", "th", "td"]);
			}
		);

		// "For each node in node list:"
		for (i = 0; i < nodeList.length; i++) {
			var node = nodeList[i];

			// "Let parent be the parent of node."
			parent_ = node.parentNode;

			// "Remove node from parent."
			parent_.removeChild(node);

			// "If strip wrappers is true or parent is not an ancestor container of
			// start node, while parent is an editable inline node with length 0,
			// let grandparent be the parent of parent, then remove parent from
			// grandparent, then set parent to grandparent."
			if (stripWrappers || (!isAncestor(parent_, startNode) && parent_ != startNode)) {
				while (isEditable(parent_) && isInlineNode(parent_) && getNodeLength(parent_) == 0) {
					var grandparent = parent_.parentNode;
					grandparent.removeChild(parent_);
					parent_ = grandparent;
				}
			}

			// "If parent is editable or an editing host, is not an inline node,
			// and has no children, call createElement("br") on the context object
			// and append the result as the last child of parent."
			// only do this, if the offsetHeight is 0
			if ((isEditable(parent_) || isEditingHost(parent_)) && !isInlineNode(parent_)) {
				ensureContainerEditable(parent_);
			}
		}

		// "If end node is an editable Text node, call deleteData(0, end offset) on
		// it."
		if (isEditable(endNode) && endNode.nodeType == $_.Node.TEXT_NODE) {
			endNode.deleteData(0, endOffset);
		}

		// "Canonicalize whitespace at range's start."
		canonicalizeWhitespace(range.startContainer, range.startOffset);

		// "Canonicalize whitespace at range's end."
		canonicalizeWhitespace(range.endContainer, range.endOffset);

		// "If block merging is false, or start block or end block is null, or
		// start block is not in the same editing host as end block, or start block
		// and end block are the same:"
		if (!blockMerging || !startBlock || !endBlock || !inSameEditingHost(startBlock, endBlock) || startBlock == endBlock) {
			// "Set range's end to its start."
			range.setEnd(range.startContainer, range.startOffset);

			// "Restore states and values from overrides."
			restoreStatesAndValues(overrides, range);

			// "Abort these steps."
			return;
		}

		// "If start block has one child, which is a collapsed block prop, remove
		// its child from it."
		if (startBlock.children.length == 1 && isCollapsedBlockProp(startBlock.firstChild)) {
			startBlock.removeChild(startBlock.firstChild);
		}

		// "If end block has one child, which is a collapsed block prop, remove its
		// child from it."
		if (endBlock.children.length == 1 && isCollapsedBlockProp(endBlock.firstChild)) {
			endBlock.removeChild(endBlock.firstChild);
		}

		var values;
		// "If start block is an ancestor of end block:"
		if (isAncestor(startBlock, endBlock)) {
			// "Let reference node be end block."
			referenceNode = endBlock;

			// "While reference node is not a child of start block, set reference
			// node to its parent."
			while (referenceNode.parentNode != startBlock) {
				referenceNode = referenceNode.parentNode;
			}

			// "Set the start and end of range to (start block, index of reference
			// node)."
			range.setStart(startBlock, getNodeIndex(referenceNode));
			range.setEnd(startBlock, getNodeIndex(referenceNode));

			// "If end block has no children:"
			if (!endBlock.hasChildNodes()) {
				// "While end block is editable and is the only child of its parent
				// and is not a child of start block, let parent equal end block,
				// then remove end block from parent, then set end block to
				// parent."
				while (isEditable(endBlock) && endBlock.parentNode.childNodes.length == 1 && endBlock.parentNode != startBlock) {
					parent_ = endBlock;
					parent_.removeChild(endBlock);
					endBlock = parent_;
				}

				// "If end block is editable and is not an inline node, and its
				// previousSibling and nextSibling are both inline nodes, call
				// createElement("br") on the context object and insert it into end
				// block's parent immediately after end block."

				if (isEditable(endBlock) && !isInlineNode(endBlock) && isInlineNode(endBlock.previousSibling) && isInlineNode(endBlock.nextSibling)) {
					endBlock.parentNode.insertBefore(document.createElement("br"), endBlock.nextSibling);
				}

				// "If end block is editable, remove it from its parent."
				if (isEditable(endBlock)) {
					endBlock.parentNode.removeChild(endBlock);
				}

				// "Restore states and values from overrides."
				restoreStatesAndValues(overrides, range);

				// "Abort these steps."
				return;
			}

			// "If end block's firstChild is not an inline node, restore states and
			// values from overrides, then abort these steps."
			if (!isInlineNode(endBlock.firstChild)) {
				restoreStatesAndValues(overrides, range);
				return;
			}

			// "Let children be a list of nodes, initially empty."
			var children = [];

			// "Append the first child of end block to children."
			children.push(endBlock.firstChild);

			// "While children's last member is not a br, and children's last
			// member's nextSibling is an inline node, append children's last
			// member's nextSibling to children."
			while (!isNamedHtmlElement(children[children.length - 1], "br") && isInlineNode(children[children.length - 1].nextSibling)) {
				children.push(children[children.length - 1].nextSibling);
			}

			// "Record the values of children, and let values be the result."
			values = recordValues(children);

			// "While children's first member's parent is not start block, split
			// the parent of children."
			while (children[0].parentNode != startBlock) {
				splitParent(children, range);
			}

			// "If children's first member's previousSibling is an editable br,
			// remove that br from its parent."
			if (isEditable(children[0].previousSibling) && isNamedHtmlElement(children[0].previousSibling, "br")) {
				children[0].parentNode.removeChild(children[0].previousSibling);
			}

			// "Otherwise, if start block is a descendant of end block:"
		} else if (isDescendant(startBlock, endBlock)) {
			// "Set the start and end of range to (start block, length of start
			// block)."
			range.setStart(startBlock, getNodeLength(startBlock));
			range.setEnd(startBlock, getNodeLength(startBlock));

			// "Let reference node be start block."
			referenceNode = startBlock;

			// "While reference node is not a child of end block, set reference
			// node to its parent."
			while (referenceNode.parentNode != endBlock) {
				referenceNode = referenceNode.parentNode;
			}

			// "If reference node's nextSibling is an inline node and start block's
			// lastChild is a br, remove start block's lastChild from it."
			if (isInlineNode(referenceNode.nextSibling) && isNamedHtmlElement(startBlock.lastChild, "br")) {
				startBlock.removeChild(startBlock.lastChild);
			}

			// "Let nodes to move be a list of nodes, initially empty."
			var nodesToMove = [];

			// "If reference node's nextSibling is neither null nor a br nor a
			// block node, append it to nodes to move."
			if (referenceNode.nextSibling && !isNamedHtmlElement(referenceNode.nextSibling, "br") && !isBlockNode(referenceNode.nextSibling)) {
				nodesToMove.push(referenceNode.nextSibling);
			}

			// "While nodes to move is nonempty and its last member's nextSibling
			// is neither null nor a br nor a block node, append it to nodes to
			// move."
			if (nodesToMove.length && nodesToMove[nodesToMove.length - 1].nextSibling && !isNamedHtmlElement(nodesToMove[nodesToMove.length - 1].nextSibling, "br") && !isBlockNode(nodesToMove[nodesToMove.length - 1].nextSibling)) {
				nodesToMove.push(nodesToMove[nodesToMove.length - 1].nextSibling);
			}

			// "Record the values of nodes to move, and let values be the result."
			values = recordValues(nodesToMove);

			// "For each node in nodes to move, append node as the last child of
			// start block, preserving ranges."
			$_(nodesToMove).forEach(function (node) {
				movePreservingRanges(node, startBlock, -1, range);
			});

			// "If the nextSibling of reference node is a br, remove it from its
			// parent."
			if (isNamedHtmlElement(referenceNode.nextSibling, "br")) {
				referenceNode.parentNode.removeChild(referenceNode.nextSibling);
			}

			// "Otherwise:"
		} else {
			// "Set the start and end of range to (start block, length of start
			// block)."
			range.setStart(startBlock, getNodeLength(startBlock));
			range.setEnd(startBlock, getNodeLength(startBlock));

			// "If end block's firstChild is an inline node and start block's
			// lastChild is a br, remove start block's lastChild from it."
			if (isInlineNode(endBlock.firstChild) && isNamedHtmlElement(startBlock.lastChild, "br")) {
				startBlock.removeChild(startBlock.lastChild);
			}

			// "Record the values of end block's children, and let values be the
			// result."
			values = recordValues([].slice.call(toArray(endBlock.childNodes)));

			// "While end block has children, append the first child of end block
			// to start block, preserving ranges."
			while (endBlock.hasChildNodes()) {
				movePreservingRanges(endBlock.firstChild, startBlock, -1, range);
			}

			// "While end block has no children, let parent be the parent of end
			// block, then remove end block from parent, then set end block to
			// parent."
			while (!endBlock.hasChildNodes()) {
				parent_ = endBlock.parentNode;
				parent_.removeChild(endBlock);
				endBlock = parent_;
			}
		}

		// "Restore the values from values."
		restoreValues(values, range);

		// "If start block has no children, call createElement("br") on the context
		// object and append the result as the last child of start block."
		ensureContainerEditable(startBlock);

		// "Restore states and values from overrides."
		restoreStatesAndValues(overrides, range);
	}

	// "To remove a node node while preserving its descendants, split the parent of
	// node's children if it has any. If it has no children, instead remove it from
	// its parent."
	function removePreservingDescendants(node, range) {
		if (node.hasChildNodes()) {
			splitParent([].slice.call(toArray(node.childNodes)), range);
		} else {
			node.parentNode.removeChild(node);
		}
	}

	//@}
	///// Indenting and outdenting /////
	//@{

	function cleanLists(node, range) {
		// remove any whitespace nodes around list nodes
		if (node) {
			jQuery(node).find('ul,ol,li').each(function () {
				jQuery(this).contents().each(function () {
					if (isWhitespaceNode(this)) {
						var index = getNodeIndex(this);

						// if the range points to somewhere behind the removed text node, we reduce the offset
						if (range.startContainer === this.parentNode && range.startOffset > index) {
							range.startOffset--;
						} else if (range.startContainer === this) {
							// the range starts in the removed text node, let it start right before
							range.startContainer = this.parentNode;
							range.startOffset = index;
						}
						// same thing for end of the range
						if (range.endContainer === this.parentNode && range.endOffset > index) {
							range.endOffset--;
						} else if (range.endContainer === this) {
							range.endContainer = this.parentNode;
							range.endOffset = index;
						}
						// finally remove the whitespace node
						jQuery(this).remove();
					}
				});
			});
		}
	}


	//@}
	///// Indenting and outdenting /////
	//@{

	function indentNodes(nodeList, range) {
		// "If node list is empty, do nothing and abort these steps."
		if (!nodeList.length) {
			return;
		}

		// "Let first node be the first member of node list."
		var firstNode = nodeList[0];

		// "If first node's parent is an ol or ul:"
		if (isHtmlElementInArray(firstNode.parentNode, ["OL", "UL"])) {
			// "Let tag be the local name of the parent of first node."
			var tag = firstNode.parentNode.tagName;

			// "Wrap node list, with sibling criteria returning true for an HTML
			// element with local name tag and false otherwise, and new parent
			// instructions returning the result of calling createElement(tag) on
			// the ownerDocument of first node."
			wrap(
				nodeList,
				function (node) {
					return isHtmlElement_obsolete(node, tag);
				},
				function () {
					return firstNode.ownerDocument.createElement(tag);
				},
				range
			);

			// "Abort these steps."
			return;
		}

		// "Wrap node list, with sibling criteria returning true for a simple
		// indentation element and false otherwise, and new parent instructions
		// returning the result of calling createElement("blockquote") on the
		// ownerDocument of first node. Let new parent be the result."
		var newParent = wrap(
			nodeList,
			function (node) {
				return isSimpleIndentationElement(node);
			},
			function () {
				return firstNode.ownerDocument.createElement("blockquote");
			},
			range
		);

		// "Fix disallowed ancestors of new parent."
		fixDisallowedAncestors(newParent, range);
	}

	function outdentNode(node, range) {
		// "If node is not editable, abort these steps."
		if (!isEditable(node)) {
			return;
		}

		// "If node is a simple indentation element, remove node, preserving its
		// descendants.  Then abort these steps."
		if (isSimpleIndentationElement(node)) {
			removePreservingDescendants(node, range);
			return;
		}

		// "If node is an indentation element:"
		if (isIndentationElement(node)) {
			// "Unset the class and dir attributes of node, if any."
			node.removeAttribute("class");
			node.removeAttribute("dir");

			// "Unset the margin, padding, and border CSS properties of node."
			node.style.margin = "";
			node.style.padding = "";
			node.style.border = "";
			if (node.getAttribute("style") == "") {
				node.removeAttribute("style");
			}

			// "Set the tag name of node to "div"."
			setTagName(node, "div", range);

			// "Abort these steps."
			return;
		}

		// "Let current ancestor be node's parent."
		var currentAncestor = node.parentNode;

		// "Let ancestor list be a list of nodes, initially empty."
		var ancestorList = [];

		// "While current ancestor is an editable Element that is neither a simple
		// indentation element nor an ol nor a ul, append current ancestor to
		// ancestor list and then set current ancestor to its parent."
		while (isEditable(currentAncestor) && currentAncestor.nodeType == $_.Node.ELEMENT_NODE && !isSimpleIndentationElement(currentAncestor) && !isHtmlElementInArray(currentAncestor, ["ol", "ul"])) {
			ancestorList.push(currentAncestor);
			currentAncestor = currentAncestor.parentNode;
		}

		// "If current ancestor is not an editable simple indentation element:"
		if (!isEditable(currentAncestor) || !isSimpleIndentationElement(currentAncestor)) {
			// "Let current ancestor be node's parent."
			currentAncestor = node.parentNode;

			// "Let ancestor list be the empty list."
			ancestorList = [];

			// "While current ancestor is an editable Element that is neither an
			// indentation element nor an ol nor a ul, append current ancestor to
			// ancestor list and then set current ancestor to its parent."
			while (isEditable(currentAncestor) && currentAncestor.nodeType == $_.Node.ELEMENT_NODE && !isIndentationElement(currentAncestor) && !isHtmlElementInArray(currentAncestor, ["ol", "ul"])) {
				ancestorList.push(currentAncestor);
				currentAncestor = currentAncestor.parentNode;
			}
		}

		// "If node is an ol or ul and current ancestor is not an editable
		// indentation element:"
		if (isHtmlElementInArray(node, ["OL", "UL"]) && (!isEditable(currentAncestor) || !isIndentationElement(currentAncestor))) {
			// "Unset the reversed, start, and type attributes of node, if any are
			// set."
			node.removeAttribute("reversed");
			node.removeAttribute("start");
			node.removeAttribute("type");

			// "Let children be the children of node."
			var children = [].slice.call(toArray(node.childNodes));

			// "If node has attributes, and its parent is not an ol or ul, set the
			// tag name of node to "div"."
			if (node.attributes.length && !isHtmlElementInArray(node.parentNode, ["OL", "UL"])) {
				setTagName(node, "div", range);

				// "Otherwise:"
			} else {
				// "Record the values of node's children, and let values be the
				// result."
				var values = recordValues([].slice.call(toArray(node.childNodes)));

				// "Remove node, preserving its descendants."
				removePreservingDescendants(node, range);

				// "Restore the values from values."
				restoreValues(values, range);
			}

			// "Fix disallowed ancestors of each member of children."
			var i;
			for (i = 0; i < children.length; i++) {
				fixDisallowedAncestors(children[i], range);
			}

			// "Abort these steps."
			return;
		}

		// "If current ancestor is not an editable indentation element, abort these
		// steps."
		if (!isEditable(currentAncestor) || !isIndentationElement(currentAncestor)) {
			return;
		}

		// "Append current ancestor to ancestor list."
		ancestorList.push(currentAncestor);

		// "Let original ancestor be current ancestor."
		var originalAncestor = currentAncestor;

		// "While ancestor list is not empty:"
		while (ancestorList.length) {
			// "Let current ancestor be the last member of ancestor list."
			//
			// "Remove the last member of ancestor list."
			currentAncestor = ancestorList.pop();

			// "Let target be the child of current ancestor that is equal to either
			// node or the last member of ancestor list."
			var target = node.parentNode == currentAncestor ? node : ancestorList[ancestorList.length - 1];

			// "If target is an inline node that is not a br, and its nextSibling
			// is a br, remove target's nextSibling from its parent."
			if (isInlineNode(target) && !isNamedHtmlElement(target, 'BR') && isNamedHtmlElement(target.nextSibling, "BR")) {
				target.parentNode.removeChild(target.nextSibling);
			}

			// "Let preceding siblings be the preceding siblings of target, and let
			// following siblings be the following siblings of target."
			var precedingSiblings = [].slice.call(toArray(currentAncestor.childNodes), 0, getNodeIndex(target));
			var followingSiblings = [].slice.call(toArray(currentAncestor.childNodes), 1 + getNodeIndex(target));

			// "Indent preceding siblings."
			indentNodes(precedingSiblings, range);

			// "Indent following siblings."
			indentNodes(followingSiblings, range);
		}

		// "Outdent original ancestor."
		outdentNode(originalAncestor, range);
	}


	//@}
	///// Toggling lists /////
	//@{

	function toggleLists(tagName, range) {
		// "Let mode be "disable" if the selection's list state is tag name, and
		// "enable" otherwise."
		var mode = getSelectionListState() == tagName ? "disable" : "enable";

		tagName = tagName.toUpperCase();

		// "Let other tag name be "ol" if tag name is "ul", and "ul" if tag name is
		// "ol"."
		var otherTagName = tagName == "OL" ? "UL" : "OL";

		// "Let items be a list of all lis that are ancestor containers of the
		// range's start and/or end node."
		//
		// It's annoying to get this in tree order using functional stuff without
		// doing getDescendants(document), which is slow, so I do it imperatively.
		var items = [];
		(function () {
			var ancestorContainer;
			for (ancestorContainer = range.endContainer;
				     ancestorContainer != range.commonAncestorContainer;
				     ancestorContainer = ancestorContainer.parentNode) {
				if (isNamedHtmlElement(ancestorContainer, "li")) {
					items.unshift(ancestorContainer);
				}
			}
			for (ancestorContainer = range.startContainer;
				     ancestorContainer;
				     ancestorContainer = ancestorContainer.parentNode) {
				if (isNamedHtmlElement(ancestorContainer, "li")) {
					items.unshift(ancestorContainer);
				}
			}
		}());

		// "For each item in items, normalize sublists of item."
		$_(items).forEach(function (thisArg) {
			normalizeSublists(thisArg, range);
		});

		// "Block-extend the range, and let new range be the result."
		var newRange = blockExtend(range);

		// "If mode is "enable", then let lists to convert consist of every
		// editable HTML element with local name other tag name that is contained
		// in new range, and for every list in lists to convert:"
		if (mode == "enable") {
			$_(getAllContainedNodes(newRange, function (node) {
				return isEditable(node) && isHtmlElement_obsolete(node, otherTagName);
			})).forEach(function (list) {
				// "If list's previousSibling or nextSibling is an editable HTML
				// element with local name tag name:"
				if ((isEditable(list.previousSibling) && isHtmlElement_obsolete(list.previousSibling, tagName)) || (isEditable(list.nextSibling) && isHtmlElement_obsolete(list.nextSibling, tagName))) {
					// "Let children be list's children."
					var children = [].slice.call(toArray(list.childNodes));

					// "Record the values of children, and let values be the
					// result."
					var values = recordValues(children);

					// "Split the parent of children."
					splitParent(children, range);

					// "Wrap children, with sibling criteria returning true for an
					// HTML element with local name tag name and false otherwise."
					wrap(
						children,
						function (node) {
							return isHtmlElement_obsolete(node, tagName);
						},
						function () {
							return null;
						},
						range
					);

					// "Restore the values from values."
					restoreValues(values, range);

					// "Otherwise, set the tag name of list to tag name."
				} else {
					setTagName(list, tagName, range);
				}
			});
		}

		// "Let node list be a list of nodes, initially empty."
		//
		// "For each node node contained in new range, if node is editable; the
		// last member of node list (if any) is not an ancestor of node; node
		// is not an indentation element; and either node is an ol or ul, or its
		// parent is an ol or ul, or it is an allowed child of "li"; then append
		// node to node list."
		var nodeList = getContainedNodes(newRange, function (node) {
			return isEditable(node) && !isIndentationElement(node) && (isHtmlElementInArray(node, ["OL", "UL"]) || isHtmlElementInArray(node.parentNode, ["OL", "UL"]) || isAllowedChild(node, "li"));
		});

		// "If mode is "enable", remove from node list any ol or ul whose parent is
		// not also an ol or ul."
		if (mode == "enable") {
			nodeList = $_(nodeList).filter(function (node) {
				return !isHtmlElementInArray(node, ["ol", "ul"]) || isHtmlElementInArray(node.parentNode, ["ol", "ul"]);
			});
		}

		// "If mode is "disable", then while node list is not empty:"
		var sublist, values;

		function createLi() {
			return document.createElement("li");
		}

		function isOlUl(node) {
			return isHtmlElementInArray(node, ["ol", "ul"]);
		}

		function makeIsElementPred(tagName) {
			return function (node) {
				return isHtmlElement_obsolete(node, tagName);
			};
		}

		function makeCreateElement(tagName) {
			return function () {
				return document.createElement(tagName);
			};
		}

		function makeCreateElementSublist(tagName, sublist, range) {
			return function () {
				// "If sublist's first member's parent is not an editable
				// simple indentation element, or sublist's first member's
				// parent's previousSibling is not an editable HTML element
				// with local name tag name, call createElement(tag name)
				// on the context object and return the result."
				if (!isEditable(sublist[0].parentNode) || !isSimpleIndentationElement(sublist[0].parentNode) || !isEditable(sublist[0].parentNode.previousSibling) || !isHtmlElement_obsolete(sublist[0].parentNode.previousSibling, tagName)) {
					return document.createElement(tagName);
				}

				// "Let list be sublist's first member's parent's
				// previousSibling."
				var list = sublist[0].parentNode.previousSibling;

				// "Normalize sublists of list's lastChild."
				normalizeSublists(list.lastChild, range);

				// "If list's lastChild is not an editable HTML element
				// with local name tag name, call createElement(tag name)
				// on the context object, and append the result as the last
				// child of list."
				if (!isEditable(list.lastChild) || !isHtmlElement_obsolete(list.lastChild, tagName)) {
					list.appendChild(document.createElement(tagName));
				}

				// "Return the last child of list."
				return list.lastChild;
			};
		}

		if (mode == "disable") {
			while (nodeList.length) {
				// "Let sublist be an empty list of nodes."
				sublist = [];

				// "Remove the first member from node list and append it to
				// sublist."
				sublist.push(nodeList.shift());

				// "If the first member of sublist is an HTML element with local
				// name tag name, outdent it and continue this loop from the
				// beginning."
				if (isHtmlElement_obsolete(sublist[0], tagName)) {
					outdentNode(sublist[0], range);
					continue;
				}

				// "While node list is not empty, and the first member of node list
				// is the nextSibling of the last member of sublist and is not an
				// HTML element with local name tag name, remove the first member
				// from node list and append it to sublist."
				while (nodeList.length && nodeList[0] == sublist[sublist.length - 1].nextSibling && !isHtmlElement_obsolete(nodeList[0], tagName)) {
					sublist.push(nodeList.shift());
				}

				// "Record the values of sublist, and let values be the result."
				values = recordValues(sublist);

				// "Split the parent of sublist."
				splitParent(sublist, range);

				// "Fix disallowed ancestors of each member of sublist."
				var i;
				for (i = 0; i < sublist.length; i++) {
					fixDisallowedAncestors(sublist[i], range);
				}

				// "Restore the values from values."
				restoreValues(values, range);
			}

			// "Otherwise, while node list is not empty:"
		} else {
			while (nodeList.length) {
				// "Let sublist be an empty list of nodes."
				sublist = [];

				// "While either sublist is empty, or node list is not empty and
				// its first member is the nextSibling of sublist's last member:"
				while (!sublist.length || (nodeList.length && nodeList[0] == sublist[sublist.length - 1].nextSibling)) {
					// "If node list's first member is a p or div, set the tag name
					// of node list's first member to "li", and append the result
					// to sublist. Remove the first member from node list."
					if (isHtmlElementInArray(nodeList[0], ["p", "div"])) {
						sublist.push(setTagName(nodeList[0], "li", range));
						nodeList.shift();

						// "Otherwise, if the first member of node list is an li or ol
						// or ul, remove it from node list and append it to sublist."
					} else if (isHtmlElementInArray(nodeList[0], ["li", "ol", "ul"])) {
						sublist.push(nodeList.shift());

						// "Otherwise:"
					} else {
						// "Let nodes to wrap be a list of nodes, initially empty."
						var nodesToWrap = [];

						// "While nodes to wrap is empty, or node list is not empty
						// and its first member is the nextSibling of nodes to
						// wrap's last member and the first member of node list is
						// an inline node and the last member of nodes to wrap is
						// an inline node other than a br, remove the first member
						// from node list and append it to nodes to wrap."
						while (!nodesToWrap.length || (nodeList.length && nodeList[0] == nodesToWrap[nodesToWrap.length - 1].nextSibling && isInlineNode(nodeList[0]) && isInlineNode(nodesToWrap[nodesToWrap.length - 1]) && !isNamedHtmlElement(nodesToWrap[nodesToWrap.length - 1], "br"))) {
							nodesToWrap.push(nodeList.shift());
						}

						// "Wrap nodes to wrap, with new parent instructions
						// returning the result of calling createElement("li") on
						// the context object. Append the result to sublist."
						sublist.push(wrap(
							nodesToWrap,
							undefined,
							createLi,
							range
						));
					}
				}

				// "If sublist's first member's parent is an HTML element with
				// local name tag name, or if every member of sublist is an ol or
				// ul, continue this loop from the beginning."
				if (isHtmlElement_obsolete(sublist[0].parentNode, tagName) || $_(sublist).every(isOlUl)) {
					continue;
				}

				// "If sublist's first member's parent is an HTML element with
				// local name other tag name:"
				if (isHtmlElement_obsolete(sublist[0].parentNode, otherTagName)) {
					// "Record the values of sublist, and let values be the
					// result."
					values = recordValues(sublist);

					// "Split the parent of sublist."
					splitParent(sublist, range);

					// "Wrap sublist, with sibling criteria returning true for an
					// HTML element with local name tag name and false otherwise,
					// and new parent instructions returning the result of calling
					// createElement(tag name) on the context object."
					wrap(
						sublist,
						makeIsElementPred(tagName),
						makeCreateElement(tagName),
						range
					);

					// "Restore the values from values."
					restoreValues(values, range);

					// "Continue this loop from the beginning."
					continue;
				}

				// "Wrap sublist, with sibling criteria returning true for an HTML
				// element with local name tag name and false otherwise, and new
				// parent instructions being the following:"
				// . . .
				// "Fix disallowed ancestors of the previous step's result."
				fixDisallowedAncestors(wrap(
					sublist,
					makeIsElementPred(tagName),
					makeCreateElementSublist(tagName, sublist, range),
					range
				), range);
			}
		}
	}


	//@}
	///// Justifying the selection /////
	//@{

	function justifySelection(alignment, range) {

		// "Block-extend the active range, and let new range be the result."
		var newRange = blockExtend(range);

		// "Let element list be a list of all editable Elements contained in new
		// range that either has an attribute in the HTML namespace whose local
		// name is "align", or has a style attribute that sets "text-align", or is
		// a center."
		var elementList = getAllContainedNodes(newRange, function (node) {
			return node.nodeType == $_.Node.ELEMENT_NODE && isEditable(node)
			// Ignoring namespaces here
				&& (hasAttribute(node, "align") || node.style.textAlign != "" || isNamedHtmlElement(node, 'center'));
		});

		// "For each element in element list:"
		var i;
		for (i = 0; i < elementList.length; i++) {
			var element = elementList[i];

			// "If element has an attribute in the HTML namespace whose local name
			// is "align", remove that attribute."
			element.removeAttribute("align");

			// "Unset the CSS property "text-align" on element, if it's set by a
			// style attribute."
			element.style.textAlign = "";
			if (element.getAttribute("style") == "") {
				element.removeAttribute("style");
			}

			// "If element is a div or span or center with no attributes, remove
			// it, preserving its descendants."
			if (isHtmlElementInArray(element, ["div", "span", "center"]) && !element.attributes.length) {
				removePreservingDescendants(element, range);
			}

			// "If element is a center with one or more attributes, set the tag
			// name of element to "div"."
			if (isNamedHtmlElement(element, 'center') && element.attributes.length) {
				setTagName(element, "div", range);
			}
		}

		// "Block-extend the active range, and let new range be the result."
		newRange = blockExtend(globalRange);

		// "Let node list be a list of nodes, initially empty."
		var nodeList = [];

		// "For each node node contained in new range, append node to node list if
		// the last member of node list (if any) is not an ancestor of node; node
		// is editable; node is an allowed child of "div"; and node's alignment
		// value is not alignment."
		nodeList = getContainedNodes(newRange, function (node) {
			return isEditable(node) && isAllowedChild(node, "div") && getAlignmentValue(node) != alignment;
		});

		function makeIsAlignedDiv(alignment) {
			return function (node) {
				return isNamedHtmlElement(node, 'div') && $_(node.attributes).every(function (attr) {
					return (attr.name == "align" && attr.value.toLowerCase() == alignment) || (attr.name == "style" && getStyleLength(node) == 1 && node.style.textAlign == alignment);
				});
			};
		}

		function makeCreateAlignedDiv(alignment) {
			return function () {
				var newParent = document.createElement("div");
				newParent.setAttribute("style", "text-align: " + alignment);
				return newParent;
			};
		}

		// "While node list is not empty:"
		while (nodeList.length) {
			// "Let sublist be a list of nodes, initially empty."
			var sublist = [];

			// "Remove the first member of node list and append it to sublist."
			sublist.push(nodeList.shift());

			// "While node list is not empty, and the first member of node list is
			// the nextSibling of the last member of sublist, remove the first
			// member of node list and append it to sublist."
			while (nodeList.length && nodeList[0] == sublist[sublist.length - 1].nextSibling) {
				sublist.push(nodeList.shift());
			}

			// "Wrap sublist. Sibling criteria returns true for any div that has
			// one or both of the following two attributes and no other attributes,
			// and false otherwise:"
			//
			//   * "An align attribute whose value is an ASCII case-insensitive
			//     match for alignment.
			//   * "A style attribute which sets exactly one CSS property
			//     (including unrecognized or invalid attributes), which is
			//     "text-align", which is set to alignment.
			//
			// "New parent instructions are to call createElement("div") on the
			// context object, then set its CSS property "text-align" to alignment
			// and return the result."
			wrap(
				sublist,
				makeIsAlignedDiv(alignment),
				makeCreateAlignedDiv(alignment),
				range
			);
		}
	}

	//@}
	///// Move the given collapsed range over adjacent zero-width whitespace characters.
	///// The range is 
	//@{
	/**
	 * Move the given collapsed range over adjacent zero-width whitespace characters.
	 * If the range is not collapsed or is not contained in a text node, it is not modified
	 * @param range range to modify
	 * @param forward {Boolean} true to move forward, false to move backward
	 */
	function moveOverZWSP(range, forward) {
		var offset;
		if (!range.collapsed) {
			return;
		}

		offset = range.startOffset;

		if (forward) {
			// check whether the range starts in a text node
			if (range.startContainer && range.startContainer.nodeType === $_.Node.TEXT_NODE) {
				// move forward (i.e. increase offset) as long as we stay in the text node and have zwsp characters to the right
				while (offset < range.startContainer.data.length && range.startContainer.data.charAt(offset) === '\u200b') {
					offset++;
				}
			}
		} else {
			// check whether the range starts in a text node
			if (range.startContainer && range.startContainer.nodeType === $_.Node.TEXT_NODE) {
				// move backward (i.e. decrease offset) as long as we stay in the text node and have zwsp characters to the left
				while (offset > 0 && range.startContainer.data.charAt(offset - 1) === '\u200b') {
					offset--;
				}
			}
		}

		// if the offset was changed, set it back to the collapsed range
		if (offset !== range.startOffset) {
			range.setStart(range.startContainer, offset);
			range.setEnd(range.startContainer, offset);
		}
	}

	/**
	 * implementation of the delete command
	 * will attempt to delete contents within range if non-collapsed
	 * or delete the character left of the cursor position if range
	 * is collapsed. Is used to define the behaviour of the backspace
	 * button.
	 *
	 * @param      value   is just there for compatibility with the commands api. parameter is ignored.
	 * @param      range   the range to execute the delete command for
	 * @return     void
	 */
	commands["delete"] = {
		action: function (value, range) {
			var i;

			// special behaviour for skipping zero-width whitespaces in IE7
			if (jQuery.browser.msie && jQuery.browser.version <= 7) {
				moveOverZWSP(range, false);
			}

			// "If the active range is not collapsed, delete the contents of the
			// active range and abort these steps."
			if (!range.collapsed) {
				deleteContents(range);
				return;
			}

			// "Canonicalize whitespace at (active range's start node, active
			// range's start offset)."
			canonicalizeWhitespace(range.startContainer, range.startOffset);

			// "Let node and offset be the active range's start node and offset."
			var node = range.startContainer;
			var offset = range.startOffset;
			var isBr = false;
			var isHr = false;

			// "Repeat the following steps:"
			while (true) {
				// we need to reset isBr and isHr on every interation of the loop
				if (offset > 0) {
					isBr = isNamedHtmlElement(node.childNodes[offset - 1], "br") || false;
					isHr = isNamedHtmlElement(node.childNodes[offset - 1], "hr") || false;
				}
				// "If offset is zero and node's previousSibling is an editable
				// invisible node, remove node's previousSibling from its parent."
				if (offset == 0 && isEditable(node.previousSibling) && isInvisible(node.previousSibling)) {
					node.parentNode.removeChild(node.previousSibling);
					continue;
				}
				// "Otherwise, if node has a child with index offset − 1 and that
				// child is an editable invisible node, remove that child from
				// node, then subtract one from offset."
				if (0 <= offset - 1 && offset - 1 < node.childNodes.length && isEditable(node.childNodes[offset - 1]) && (isInvisible(node.childNodes[offset - 1]) || isBr || isHr)) {
					node.removeChild(node.childNodes[offset - 1]);
					offset--;
					if (isBr || isHr) {
						range.setStart(node, offset);
						range.setEnd(node, offset);
						return;
					}
					continue;

				}
				// "Otherwise, if offset is zero and node is an inline node, or if
				// node is an invisible node, set offset to the index of node, then
				// set node to its parent."
				if ((offset == 0 && isInlineNode(node)) || isInvisible(node)) {
					offset = getNodeIndex(node);
					node = node.parentNode;
					continue;
				}
				// "Otherwise, if node has a child with index offset − 1 and that
				// child is an editable a, remove that child from node, preserving
				// its descendants. Then abort these steps."
				if (0 <= offset - 1 && offset - 1 < node.childNodes.length && isEditable(node.childNodes[offset - 1]) && isNamedHtmlElement(node.childNodes[offset - 1], "a")) {
					removePreservingDescendants(node.childNodes[offset - 1], range);
					return;

				}
				// "Otherwise, if node has a child with index offset − 1 and that
				// child is not a block node or a br or an img, set node to that
				// child, then set offset to the length of node."
				if (0 <= offset - 1 && offset - 1 < node.childNodes.length && !isBlockNode(node.childNodes[offset - 1]) && !isHtmlElementInArray(node.childNodes[offset - 1], ["br", "img"])) {
					node = node.childNodes[offset - 1];
					offset = getNodeLength(node);
					continue;
				}
				// "Otherwise, break from this loop."
				// brk is a quick and dirty jslint workaround since I don't want to rewrite this loop
				var brk = true;
				if (brk) {
					break;
				}
			}

			// if the previous node is an aloha-table we want to delete it
			var delBlock = getBlockAtPreviousPosition(node, offset);
			if (delBlock) {
				delBlock.parentNode.removeChild(delBlock);
				return;
			}

			// "If node is a Text node and offset is not zero, call collapse(node,
			// offset) on the Selection. Then delete the contents of the range with
			// start (node, offset − 1) and end (node, offset) and abort these
			// steps."
			if (node.nodeType == $_.Node.TEXT_NODE && offset != 0) {
				range.setStart(node, offset - 1);
				range.setEnd(node, offset - 1);
				deleteContents(node, offset - 1, node, offset);
				return;
			}

			// @iebug
			// when inserting a special char via the plugin 
			// there where problems deleting them again with backspace after insertation
			// see https://github.com/alohaeditor/Aloha-Editor/issues/517
			if (node.nodeType == $_.Node.TEXT_NODE && offset == 0 && jQuery.browser.msie) {
				offset = 1;
				range.setStart(node, offset);
				range.setEnd(node, offset);
				range.startOffset = 0;
				deleteContents(range);
				return;
			}

			// "If node is an inline node, abort these steps."
			if (isInlineNode(node)) {
				return;
			}

			// "If node has a child with index offset − 1 and that child is a br or
			// hr or img, call collapse(node, offset) on the Selection. Then delete
			// the contents of the range with start (node, offset − 1) and end
			// (node, offset) and abort these steps."
			if (0 <= offset - 1 && offset - 1 < node.childNodes.length && isHtmlElementInArray(node.childNodes[offset - 1], ["br", "hr", "img"])) {
				range.setStart(node, offset);
				range.setEnd(node, offset);
				deleteContents(range);
				return;
			}

			// "If node is an li or dt or dd and is the first child of its parent,
			// and offset is zero:"
			if (isHtmlElementInArray(node, ["li", "dt", "dd"]) && node == node.parentNode.firstChild && offset == 0) {
				// "Let items be a list of all lis that are ancestors of node."
				//
				// Remember, must be in tree order.
				var items = [];
				var ancestor;
				for (ancestor = node.parentNode; ancestor; ancestor = ancestor.parentNode) {
					if (isNamedHtmlElement(ancestor, 'li')) {
						items.unshift(ancestor);
					}
				}

				// "Normalize sublists of each item in items."
				for (i = 0; i < items.length; i++) {
					normalizeSublists(items[i], range);
				}

				// "Record the values of the one-node list consisting of node, and
				// let values be the result."
				var values = recordValues([node]);

				// "Split the parent of the one-node list consisting of node."
				splitParent([node], range);

				// "Restore the values from values."
				restoreValues(values, range);

				// "If node is a dd or dt, and it is not an allowed child of any of
				// its ancestors in the same editing host, set the tag name of node
				// to the default single-line container name and let node be the
				// result."
				if (isHtmlElementInArray(node, ["dd", "dt"]) && $_(getAncestors(node)).every(function (ancestor) { return !inSameEditingHost(node, ancestor) || !isAllowedChild(node, ancestor); })) {
					node = setTagName(node, defaultSingleLineContainerName, range);
				}

				// "Fix disallowed ancestors of node."
				fixDisallowedAncestors(node, range);

				// fix the lists to be html5 conformant
				for (i = 0; i < items.length; i++) {
					unNormalizeSublists(items[i].parentNode, range);
				}

				// "Abort these steps."
				return;
			}

			// "Let start node equal node and let start offset equal offset."
			var startNode = node;
			var startOffset = offset;

			// "Repeat the following steps:"
			while (true) {
				// "If start offset is zero, set start offset to the index of start
				// node and then set start node to its parent."
				if (startOffset == 0) {
					startOffset = getNodeIndex(startNode);
					startNode = startNode.parentNode;

					// "Otherwise, if start node has an editable invisible child with
					// index start offset minus one, remove it from start node and
					// subtract one from start offset."
				} else if (0 <= startOffset - 1 && startOffset - 1 < startNode.childNodes.length && isEditable(startNode.childNodes[startOffset - 1]) && isInvisible(startNode.childNodes[startOffset - 1])) {
					startNode.removeChild(startNode.childNodes[startOffset - 1]);
					startOffset--;

					// "Otherwise, break from this loop."
				} else {
					break;
				}
			}

			// "If offset is zero, and node has an editable ancestor container in
			// the same editing host that's an indentation element:"
			if (offset == 0 && $_(getAncestors(node).concat(node)).filter(function (ancestor) { return isEditable(ancestor) && inSameEditingHost(ancestor, node) && isIndentationElement(ancestor); }).length) {
				// "Block-extend the range whose start and end are both (node, 0),
				// and let new range be the result."
				var newRange = Aloha.createRange();
				newRange.setStart(node, 0);
				newRange.setEnd(node, 0);
				newRange = blockExtend(newRange);

				// "Let node list be a list of nodes, initially empty."
				//
				// "For each node current node contained in new range, append
				// current node to node list if the last member of node list (if
				// any) is not an ancestor of current node, and current node is
				// editable but has no editable descendants."
				var nodeList = getContainedNodes(newRange, function (currentNode) {
					return isEditable(currentNode) && !hasEditableDescendants(currentNode);
				});

				// "Outdent each node in node list."
				for (i = 0; i < nodeList.length; i++) {
					outdentNode(nodeList[i], range);
				}

				// "Abort these steps."
				return;
			}

			// "If the child of start node with index start offset is a table,
			// abort these steps."
			if (isNamedHtmlElement(startNode.childNodes[startOffset], "table")) {
				return;
			}

			// "If start node has a child with index start offset − 1, and that
			// child is a table:"
			if (0 <= startOffset - 1 && startOffset - 1 < startNode.childNodes.length && isNamedHtmlElement(startNode.childNodes[startOffset - 1], "table")) {
				// "Call collapse(start node, start offset − 1) on the context
				// object's Selection."
				range.setStart(startNode, startOffset - 1);

				// "Call extend(start node, start offset) on the context object's
				// Selection."
				range.setEnd(startNode, startOffset);

				// "Abort these steps."
				return;
			}

			// "If offset is zero; and either the child of start node with index
			// start offset minus one is an hr, or the child is a br whose
			// previousSibling is either a br or not an inline node:"
			if (offset == 0
				    && (isNamedHtmlElement(startNode.childNodes[startOffset - 1], "hr")
						|| (isNamedHtmlElement(startNode.childNodes[startOffset - 1], "br")
							&& (isNamedHtmlElement(startNode.childNodes[startOffset - 1].previousSibling, "br")
								|| !isInlineNode(startNode.childNodes[startOffset - 1].previousSibling))))) {
				// "Call collapse(node, offset) on the Selection."
				range.setStart(node, offset);
				range.setEnd(node, offset);

				// "Delete the contents of the range with start (start node, start
				// offset − 1) and end (start node, start offset)."
				deleteContents(startNode, startOffset - 1, startNode, startOffset);

				// "Abort these steps."
				return;
			}

			// "If the child of start node with index start offset is an li or dt
			// or dd, and that child's firstChild is an inline node, and start
			// offset is not zero:"
			if (isHtmlElementInArray(startNode.childNodes[startOffset], ["li", "dt", "dd"]) && isInlineNode(startNode.childNodes[startOffset].firstChild) && startOffset != 0) {
				// "Let previous item be the child of start node with index start
				// offset minus one."
				var previousItem = startNode.childNodes[startOffset - 1];

				// "If previous item's lastChild is an inline node other than a br,
				// call createElement("br") on the context object and append the
				// result as the last child of previous item."
				if (isInlineNode(previousItem.lastChild) && !isNamedHtmlElement(previousItem.lastChild, "br")) {
					previousItem.appendChild(document.createElement("br"));
				}

				// "If previous item's lastChild is an inline node, call
				// createElement("br") on the context object and append the result
				// as the last child of previous item."
				if (isInlineNode(previousItem.lastChild)) {
					previousItem.appendChild(document.createElement("br"));
				}
			}

			// "If the child of start node with index start offset is an li or dt
			// or dd, and its previousSibling is also an li or dt or dd, set start
			// node to its child with index start offset − 1, then set start offset
			// to start node's length, then set node to start node's nextSibling,
			// then set offset to 0."
			if (isHtmlElementInArray(startNode.childNodes[startOffset], ["li", "dt", "dd"]) && isHtmlElementInArray(startNode.childNodes[startOffset - 1], ["li", "dt", "dd"])) {
				startNode = startNode.childNodes[startOffset - 1];
				startOffset = getNodeLength(startNode);
				node = startNode.nextSibling;
				offset = 0;

				// "Otherwise, while start node has a child with index start offset
				// minus one:"
			} else {
				while (0 <= startOffset - 1 && startOffset - 1 < startNode.childNodes.length) {
					// "If start node's child with index start offset minus one is
					// editable and invisible, remove it from start node, then
					// subtract one from start offset."
					if (isEditable(startNode.childNodes[startOffset - 1]) && isInvisible(startNode.childNodes[startOffset - 1])) {
						startNode.removeChild(startNode.childNodes[startOffset - 1]);
						startOffset--;

						// "Otherwise, set start node to its child with index start
						// offset minus one, then set start offset to the length of
						// start node."
					} else {
						startNode = startNode.childNodes[startOffset - 1];
						startOffset = getNodeLength(startNode);
					}
				}
			}

			// "Delete the contents of the range with start (start node, start
			// offset) and end (node, offset)."
			var delRange = Aloha.createRange();
			delRange.setStart(startNode, startOffset);
			delRange.setEnd(node, offset);
			deleteContents(delRange);

			if (!isAncestorContainer(document.body, range.startContainer)) {
				if (delRange.startContainer.hasChildNodes() || delRange.startContainer.nodeType == $_.Node.TEXT_NODE) {
					range.setStart(delRange.startContainer, delRange.startOffset);
					range.setEnd(delRange.startContainer, delRange.startOffset);
				} else {
					range.setStart(delRange.startContainer.parentNode, getNodeIndex(delRange.startContainer));
					range.setEnd(delRange.startContainer.parentNode, getNodeIndex(delRange.startContainer));
				}
			}
		}
	};

	//@}
	///// The formatBlock command /////
	//@{
	// "A formattable block name is "address", "dd", "div", "dt", "h1", "h2", "h3",
	// "h4", "h5", "h6", "p", or "pre"."
	var formattableBlockNames = ["address", "dd", "div", "dt", "h1", "h2", "h3", "h4", "h5", "h6", "p", "pre"];

	commands.formatblock = {
		action: function (value) {
			var i;

			// "If value begins with a "<" character and ends with a ">" character,
			// remove the first and last characters from it."
			if (/^<.*>$/.test(value)) {
				value = value.slice(1, -1);
			}

			// "Let value be converted to ASCII lowercase."
			value = value.toLowerCase();

			// "If value is not a formattable block name, abort these steps and do
			// nothing."
			if ($_(formattableBlockNames).indexOf(value) == -1) {
				return;
			}

			// "Block-extend the active range, and let new range be the result."
			var newRange = blockExtend(getActiveRange());

			// "Let node list be an empty list of nodes."
			//
			// "For each node node contained in new range, append node to node list
			// if it is editable, the last member of original node list (if any) is
			// not an ancestor of node, node is either a non-list single-line
			// container or an allowed child of "p" or a dd or dt, and node is not
			// the ancestor of a prohibited paragraph child."
			var nodeList = getContainedNodes(newRange, function (node) {
				return isEditable(node) && (isNonListSingleLineContainer(node) || isAllowedChild(node, "p") || isHtmlElementInArray(node, ["dd", "dt"])) && !$_(getDescendants(node)).some(isProhibitedParagraphChild);
			});

			// "Record the values of node list, and let values be the result."
			var values = recordValues(nodeList);

			function makeIsEditableElementInSameEditingHostDoesNotContainProhibitedParagraphChildren(node) {
				return function (ancestor) {
					return (isEditable(ancestor)
							&& inSameEditingHost(ancestor, node)
							&& isHtmlElement_obsolete(ancestor, formattableBlockNames)
							&& !$_(getDescendants(ancestor)).some(isProhibitedParagraphChild));
				};
			}

			function makeIsElementWithoutAttributes(value) {
				return function (node) {
					return isHtmlElement_obsolete(node, value) && !node.attributes.length;
				};
			}

			function returnFalse() {
				return false;
			}

			function makeCreateElement(value) {
				return function () {
					return document.createElement(value);
				};
			}

			// "For each node in node list, while node is the descendant of an
			// editable HTML element in the same editing host, whose local name is
			// a formattable block name, and which is not the ancestor of a
			// prohibited paragraph child, split the parent of the one-node list
			// consisting of node."
			for (i = 0; i < nodeList.length; i++) {
				var node = nodeList[i];
				while ($_(getAncestors(node)).some(makeIsEditableElementInSameEditingHostDoesNotContainProhibitedParagraphChildren(node))) {
					splitParent([node], newRange);
				}
			}

			// "Restore the values from values."
			restoreValues(values, newRange);

			// "While node list is not empty:"
			while (nodeList.length) {
				var sublist;

				// "If the first member of node list is a single-line
				// container:"
				if (isSingleLineContainer(nodeList[0])) {
					// "Let sublist be the children of the first member of node
					// list."
					sublist = [].slice.call(toArray(nodeList[0].childNodes));

					// "Record the values of sublist, and let values be the
					// result."
					values = recordValues(sublist);

					// "Remove the first member of node list from its parent,
					// preserving its descendants."
					removePreservingDescendants(nodeList[0], newRange);

					// "Restore the values from values."
					restoreValues(values, newRange);

					// "Remove the first member from node list."
					nodeList.shift();

					// "Otherwise:"
				} else {
					// "Let sublist be an empty list of nodes."
					sublist = [];

					// "Remove the first member of node list and append it to
					// sublist."
					sublist.push(nodeList.shift());

					// "While node list is not empty, and the first member of
					// node list is the nextSibling of the last member of
					// sublist, and the first member of node list is not a
					// single-line container, and the last member of sublist is
					// not a br, remove the first member of node list and
					// append it to sublist."
					while (nodeList.length && nodeList[0] == sublist[sublist.length - 1].nextSibling && !isSingleLineContainer(nodeList[0]) && !isNamedHtmlElement(sublist[sublist.length - 1], "BR")) {
						sublist.push(nodeList.shift());
					}
				}

				// "Wrap sublist. If value is "div" or "p", sibling criteria
				// returns false; otherwise it returns true for an HTML element
				// with local name value and no attributes, and false otherwise.
				// New parent instructions return the result of running
				// createElement(value) on the context object. Then fix disallowed
				// ancestors of the result."
				fixDisallowedAncestors(wrap(
					sublist,
					jQuery.inArray(value, ["div", "p"]) == -1 ? makeIsElementWithoutAttributes(value) : returnFalse,
					makeCreateElement(value),
					newRange
				), newRange);
			}
		},
		indeterm: function () {
			// "Block-extend the active range, and let new range be the result."
			var newRange = blockExtend(getActiveRange());

			// "Let node list be all visible editable nodes that are contained in
			// new range and have no children."
			var nodeList = getAllContainedNodes(newRange, function (node) {
				return isVisible(node) && isEditable(node) && !node.hasChildNodes();
			});

			// "If node list is empty, return false."
			if (!nodeList.length) {
				return false;
			}

			// "Let type be null."
			var type = null;

			// "For each node in node list:"
			var i;
			for (i = 0; i < nodeList.length; i++) {
				var node = nodeList[i];

				// "While node's parent is editable and in the same editing host as
				// node, and node is not an HTML element whose local name is a
				// formattable block name, set node to its parent."
				while (isEditable(node.parentNode) && inSameEditingHost(node, node.parentNode) && !isHtmlElement_obsolete(node, formattableBlockNames)) {
					node = node.parentNode;
				}

				// "Let current type be the empty string."
				var currentType = "";

				// "If node is an editable HTML element whose local name is a
				// formattable block name, and node is not the ancestor of a
				// prohibited paragraph child, set current type to node's local
				// name."
				if (isEditable(node) && isHtmlElement_obsolete(node, formattableBlockNames) && !$_(getDescendants(node)).some(isProhibitedParagraphChild)) {
					currentType = node.tagName;
				}

				// "If type is null, set type to current type."
				if (type === null) {
					type = currentType;

					// "Otherwise, if type does not equal current type, return true."
				} else if (type != currentType) {
					return true;
				}
			}

			// "Return false."
			return false;
		},
		value: function () {
			// "Block-extend the active range, and let new range be the result."
			var newRange = blockExtend(getActiveRange());

			// "Let node be the first visible editable node that is contained in
			// new range and has no children. If there is no such node, return the
			// empty string."
			var nodes = getAllContainedNodes(newRange, function (node) {
				return isVisible(node) && isEditable(node) && !node.hasChildNodes();
			});
			if (!nodes.length) {
				return "";
			}
			var node = nodes[0];

			// "While node's parent is editable and in the same editing host as
			// node, and node is not an HTML element whose local name is a
			// formattable block name, set node to its parent."
			while (isEditable(node.parentNode) && inSameEditingHost(node, node.parentNode) && !isHtmlElement_obsolete(node, formattableBlockNames)) {
				node = node.parentNode;
			}

			// "If node is an editable HTML element whose local name is a
			// formattable block name, and node is not the ancestor of a prohibited
			// paragraph child, return node's local name, converted to ASCII
			// lowercase."
			if (isEditable(node) && isHtmlElement_obsolete(node, formattableBlockNames) && !$_(getDescendants(node)).some(isProhibitedParagraphChild)) {
				return node.tagName.toLowerCase();
			}

			// "Return the empty string."
			return "";
		}
	};

	//@}
	///// The forwardDelete command /////
	//@{
	commands.forwarddelete = {
		action: function (value, range) {
			// special behaviour for skipping zero-width whitespaces in IE7
			if (jQuery.browser.msie && jQuery.browser.version <= 7) {
				moveOverZWSP(range, true);
			}

			// "If the active range is not collapsed, delete the contents of the
			// active range and abort these steps."
			if (!range.collapsed) {
				deleteContents(range);
				return;
			}

			// "Canonicalize whitespace at (active range's start node, active
			// range's start offset)."
			canonicalizeWhitespace(range.startContainer, range.startOffset);

			// "Let node and offset be the active range's start node and offset."
			var node = range.startContainer;
			var offset = range.startOffset;
			var isBr = false;
			var isHr = false;

			// "Repeat the following steps:"
			while (true) {
				// check whether the next element is a br or hr
				// Commented out for unknown reason.
				//if (offset < node.childNodes.length) {
				//				isBr = isHtmlElement_obsolete(node.childNodes[offset], "br") || false;
				//				isHr = isHtmlElement_obsolete(node.childNodes[offset], "hr") || false;
				//}

				// "If offset is the length of node and node's nextSibling is an
				// editable invisible node, remove node's nextSibling from its
				// parent."
				if (offset == getNodeLength(node) && isEditable(node.nextSibling) && isInvisible(node.nextSibling)) {
					node.parentNode.removeChild(node.nextSibling);

					// "Otherwise, if node has a child with index offset and that child
					// is an editable invisible node, remove that child from node."
				} else if (offset < node.childNodes.length && isEditable(node.childNodes[offset]) && (isInvisible(node.childNodes[offset]) || isBr || isHr)) {
					node.removeChild(node.childNodes[offset]);
					if (isBr || isHr) {
						ensureContainerEditable(node);
						range.setStart(node, offset);
						range.setEnd(node, offset);
						return;
					}

					// "Otherwise, if node has a child with index offset and that child
					// is a collapsed block prop, add one to offset."
				} else if (offset < node.childNodes.length && isCollapsedBlockProp(node.childNodes[offset])) {
					offset++;

					// "Otherwise, if offset is the length of node and node is an
					// inline node, or if node is invisible, set offset to one plus the
					// index of node, then set node to its parent."
				} else if ((offset == getNodeLength(node) && isInlineNode(node)) || isInvisible(node)) {
					offset = 1 + getNodeIndex(node);
					node = node.parentNode;

					// "Otherwise, if node has a child with index offset and that child
					// is not a block node or a br or an img, set node to that child,
					// then set offset to zero."
				} else if (offset < node.childNodes.length && !isBlockNode(node.childNodes[offset]) && !isHtmlElementInArray(node.childNodes[offset], ["br", "img"])) {
					node = node.childNodes[offset];
					offset = 0;

					// "Otherwise, break from this loop."
				} else {
					break;
				}
			}

			// collapse whitespace in the node, if it is a text node
			canonicalizeWhitespace(range.startContainer, range.startOffset);

			// if the next node is an aloha-table we want to delete it
			var delBlock = getBlockAtNextPosition(node, offset);
			if (delBlock) {
				delBlock.parentNode.removeChild(delBlock);
				return;
			}

			var endOffset;
			// "If node is a Text node and offset is not node's length:"
			if (node.nodeType == $_.Node.TEXT_NODE && offset != getNodeLength(node)) {
				// "Call collapse(node, offset) on the Selection."
				range.setStart(node, offset);
				range.setEnd(node, offset);

				// "Let end offset be offset plus one."
				endOffset = offset + 1;

				// "While end offset is not node's length and the end offsetth
				// element of node's data has general category M when interpreted
				// as a Unicode code point, add one to end offset."
				//
				// TODO: Not even going to try handling anything beyond the most
				// basic combining marks, since I couldn't find a good list.  I
				// special-case a few Hebrew diacritics too to test basic coverage
				// of non-Latin stuff.
				while (endOffset != node.length && /^[\u0300-\u036f\u0591-\u05bd\u05c1\u05c2]$/.test(node.data[endOffset])) {
					endOffset++;
				}

				// "Delete the contents of the range with start (node, offset) and
				// end (node, end offset)."
				deleteContents(node, offset, node, endOffset);

				// "Abort these steps."
				return;
			}

			// "If node is an inline node, abort these steps."
			if (isInlineNode(node)) {
				return;
			}

			// "If node has a child with index offset and that child is a br or hr
			// or img, call collapse(node, offset) on the Selection. Then delete
			// the contents of the range with start (node, offset) and end (node,
			// offset + 1) and abort these steps."
			if (offset < node.childNodes.length && isHtmlElementInArray(node.childNodes[offset], ["br", "hr", "img"])) {
				range.setStart(node, offset);
				range.setEnd(node, offset);
				deleteContents(node, offset, node, offset + 1);
				return;
			}

			// "Let end node equal node and let end offset equal offset."
			var endNode = node;
			endOffset = offset;

			// "Repeat the following steps:"
			while (true) {
				// "If end offset is the length of end node, set end offset to one
				// plus the index of end node and then set end node to its parent."
				if (endOffset == getNodeLength(endNode)) {
					endOffset = 1 + getNodeIndex(endNode);
					endNode = endNode.parentNode;

					// "Otherwise, if end node has a an editable invisible child with
					// index end offset, remove it from end node."
				} else if (endOffset < endNode.childNodes.length && isEditable(endNode.childNodes[endOffset]) && isInvisible(endNode.childNodes[endOffset])) {
					endNode.removeChild(endNode.childNodes[endOffset]);

					// "Otherwise, break from this loop."
				} else {
					break;
				}
			}

			// "If the child of end node with index end offset minus one is a
			// table, abort these steps."
			if (isNamedHtmlElement(endNode.childNodes[endOffset - 1], "table")) {
				return;
			}

			// "If the child of end node with index end offset is a table:"
			if (isNamedHtmlElement(endNode.childNodes[endOffset], "table")) {
				// "Call collapse(end node, end offset) on the context object's
				// Selection."
				range.setStart(endNode, endOffset);

				// "Call extend(end node, end offset + 1) on the context object's
				// Selection."
				range.setEnd(endNode, endOffset + 1);

				// "Abort these steps."
				return;
			}

			// "If offset is the length of node, and the child of end node with
			// index end offset is an hr or br:"
			if (offset == getNodeLength(node) && isHtmlElementInArray(endNode.childNodes[endOffset], ["br", "hr"])) {
				// "Call collapse(node, offset) on the Selection."
				range.setStart(node, offset);
				range.setEnd(node, offset);

				// "Delete the contents of the range with end (end node, end
				// offset) and end (end node, end offset + 1)."
				deleteContents(endNode, endOffset, endNode, endOffset + 1);

				// "Abort these steps."
				return;
			}

			// "While end node has a child with index end offset:"
			while (endOffset < endNode.childNodes.length) {
				// "If end node's child with index end offset is editable and
				// invisible, remove it from end node."
				if (isEditable(endNode.childNodes[endOffset]) && isInvisible(endNode.childNodes[endOffset])) {
					endNode.removeChild(endNode.childNodes[endOffset]);

					// "Otherwise, set end node to its child with index end offset and
					// set end offset to zero."
				} else {
					endNode = endNode.childNodes[endOffset];
					endOffset = 0;
				}
			}

			// "Delete the contents of the range with start (node, offset) and end
			// (end node, end offset)."
			deleteContents(node, offset, endNode, endOffset);
		}
	};

	//@}
	///// The indent command /////
	//@{
	commands.indent = {
		action: function () {
			// "Let items be a list of all lis that are ancestor containers of the
			// active range's start and/or end node."
			//
			// Has to be in tree order, remember!
			var items = [];
			var node;
			for (node = getActiveRange().endContainer; node != getActiveRange().commonAncestorContainer; node = node.parentNode) {
				if (isNamedHtmlElement(node, "LI")) {
					items.unshift(node);
				}
			}
			for (node = getActiveRange().startContainer; node != getActiveRange().commonAncestorContainer; node = node.parentNode) {
				if (isNamedHtmlElement(node, "LI")) {
					items.unshift(node);
				}
			}
			for (node = getActiveRange().commonAncestorContainer; node; node = node.parentNode) {
				if (isNamedHtmlElement(node, "LI")) {
					items.unshift(node);
				}
			}

			// "For each item in items, normalize sublists of item."
			var i;
			for (i = 0; i < items.length; i++) {
				normalizeSublists(items[i], getActiveRange());
			}

			// "Block-extend the active range, and let new range be the result."
			var newRange = blockExtend(getActiveRange());

			// "Let node list be a list of nodes, initially empty."
			var nodeList = [];

			// "For each node node contained in new range, if node is editable and
			// is an allowed child of "div" or "ol" and if the last member of node
			// list (if any) is not an ancestor of node, append node to node list."
			nodeList = getContainedNodes(newRange, function (node) {
				return isEditable(node) && (isAllowedChild(node, "div") || isAllowedChild(node, "ol"));
			});

			// "If the first member of node list is an li whose parent is an ol or
			// ul, and its previousSibling is an li as well, normalize sublists of
			// its previousSibling."
			if (nodeList.length && isNamedHtmlElement(nodeList[0], "LI") && isHtmlElementInArray(nodeList[0].parentNode, ["OL", "UL"]) && isNamedHtmlElement(nodeList[0].previousSibling, "LI")) {
				normalizeSublists(nodeList[0].previousSibling, newRange);
			}

			// "While node list is not empty:"
			while (nodeList.length) {
				// "Let sublist be a list of nodes, initially empty."
				var sublist = [];

				// "Remove the first member of node list and append it to sublist."
				sublist.push(nodeList.shift());

				// "While the first member of node list is the nextSibling of the
				// last member of sublist, remove the first member of node list and
				// append it to sublist."
				while (nodeList.length && nodeList[0] == sublist[sublist.length - 1].nextSibling) {
					sublist.push(nodeList.shift());
				}

				// "Indent sublist."
				indentNodes(sublist, newRange);
			}
		}
	};

	//@}
	///// The insertHorizontalRule command /////
	//@{
	commands.inserthorizontalrule = {
		action: function (value, range) {

			// "While range's start offset is 0 and its start node's parent is not
			// null, set range's start to (parent of start node, index of start
			// node)."
			while (range.startOffset == 0 && range.startContainer.parentNode) {
				range.setStart(range.startContainer.parentNode, getNodeIndex(range.startContainer));
			}

			// "While range's end offset is the length of its end node, and its end
			// node's parent is not null, set range's end to (parent of end node, 1
			// + index of start node)."
			while (range.endOffset == getNodeLength(range.endContainer) && range.endContainer.parentNode) {
				range.setEnd(range.endContainer.parentNode, 1 + getNodeIndex(range.endContainer));
			}

			// "Delete the contents of range, with block merging false."
			deleteContents(range, {
				blockMerging: false
			});

			// "If the active range's start node is neither editable nor an editing
			// host, abort these steps."
			if (!isEditable(getActiveRange().startContainer) && !isEditingHost(getActiveRange().startContainer)) {
				return;
			}

			// "If the active range's start node is a Text node and its start
			// offset is zero, set the active range's start and end to (parent of
			// start node, index of start node)."
			if (getActiveRange().startContainer.nodeType == $_.Node.TEXT_NODE && getActiveRange().startOffset == 0) {
				getActiveRange().setStart(getActiveRange().startContainer.parentNode, getNodeIndex(getActiveRange().startContainer));
				getActiveRange().collapse(true);
			}

			// "If the active range's start node is a Text node and its start
			// offset is the length of its start node, set the active range's start
			// and end to (parent of start node, 1 + index of start node)."
			if (getActiveRange().startContainer.nodeType == $_.Node.TEXT_NODE && getActiveRange().startOffset == getNodeLength(getActiveRange().startContainer)) {
				getActiveRange().setStart(getActiveRange().startContainer.parentNode, 1 + getNodeIndex(getActiveRange().startContainer));
				getActiveRange().collapse(true);
			}

			// "Let hr be the result of calling createElement("hr") on the
			// context object."
			var hr = document.createElement("hr");

			// "Run insertNode(hr) on the range."
			range.insertNode(hr);

			// "Fix disallowed ancestors of hr."
			fixDisallowedAncestors(hr, range);

			// "Run collapse() on the Selection, with first argument equal to the
			// parent of hr and the second argument equal to one plus the index of
			// hr."
			//
			// Not everyone actually supports collapse(), so we do it manually
			// instead.  Also, we need to modify the actual range we're given as
			// well, for the sake of autoimplementation.html's range-filling-in.
			range.setStart(hr.parentNode, 1 + getNodeIndex(hr));
			range.setEnd(hr.parentNode, 1 + getNodeIndex(hr));
			Aloha.getSelection().removeAllRanges();
			Aloha.getSelection().addRange(range);
		}
	};

	//@}
	///// The insertHTML command /////
	//@{
	commands.inserthtml = {
		action: function (value, range) {


			// "Delete the contents of the active range."
			deleteContents(range);

			// "If the active range's start node is neither editable nor an editing
			// host, abort these steps."
			if (!isEditable(range.startContainer) && !isEditingHost(range.startContainer)) {
				return;
			}

			// "Let frag be the result of calling createContextualFragment(value)
			// on the active range."
			var frag = range.createContextualFragment(value);

			// "Let last child be the lastChild of frag."
			var lastChild = frag.lastChild;

			// "If last child is null, abort these steps."
			if (!lastChild) {
				return;
			}

			// "Let descendants be all descendants of frag."
			var descendants = getDescendants(frag);

			// "If the active range's start node is a block node:"
			if (isBlockNode(range.startContainer)) {
				// "Let collapsed block props be all editable collapsed block prop
				// children of the active range's start node that have index
				// greater than or equal to the active range's start offset."
				//
				// "For each node in collapsed block props, remove node from its
				// parent."
				$_(range.startContainer.childNodes).filter(function (node, range) {
					return isEditable(node) && isCollapsedBlockProp(node) && getNodeIndex(node) >= range.startOffset;
				}, true).forEach(function (node) {
					node.parentNode.removeChild(node);
				});
			}

			// "Call insertNode(frag) on the active range."
			range.insertNode(frag);

			// "If the active range's start node is a block node with no visible
			// children, call createElement("br") on the context object and append
			// the result as the last child of the active range's start node."
			if (isBlockNode(range.startContainer)) {
				ensureContainerEditable(range.startContainer);
			}

			// "Call collapse() on the context object's Selection, with last
			// child's parent as the first argument and one plus its index as the
			// second."
			range.setStart(lastChild.parentNode, 1 + getNodeIndex(lastChild));
			range.setEnd(lastChild.parentNode, 1 + getNodeIndex(lastChild));

			// "Fix disallowed ancestors of each member of descendants."
			var i;
			for (i = 0; i < descendants.length; i++) {
				fixDisallowedAncestors(descendants[i], range);
			}

			setActiveRange(range);
		}
	};

	//@}
	///// The insertImage command /////
	//@{
	commands.insertimage = {
		action: function (value) {
			// "If value is the empty string, abort these steps and do nothing."
			if (value === "") {
				return;
			}

			// "Let range be the active range."
			var range = getActiveRange();

			// "Delete the contents of range, with strip wrappers false."
			deleteContents(range, {
				stripWrappers: false
			});

			// "If the active range's start node is neither editable nor an editing
			// host, abort these steps."
			if (!isEditable(getActiveRange().startContainer) && !isEditingHost(getActiveRange().startContainer)) {
				return;
			}

			// "If range's start node is a block node whose sole child is a br, and
			// its start offset is 0, remove its start node's child from it."
			if (isBlockNode(range.startContainer) && range.startContainer.childNodes.length == 1 && isNamedHtmlElement(range.startContainer.firstChild, "br") && range.startOffset == 0) {
				range.startContainer.removeChild(range.startContainer.firstChild);
			}

			// "Let img be the result of calling createElement("img") on the
			// context object."
			var img = document.createElement("img");

			// "Run setAttribute("src", value) on img."
			img.setAttribute("src", value);

			// "Run insertNode(img) on the range."
			range.insertNode(img);

			// "Run collapse() on the Selection, with first argument equal to the
			// parent of img and the second argument equal to one plus the index of
			// img."
			//
			// Not everyone actually supports collapse(), so we do it manually
			// instead.  Also, we need to modify the actual range we're given as
			// well, for the sake of autoimplementation.html's range-filling-in.
			range.setStart(img.parentNode, 1 + getNodeIndex(img));
			range.setEnd(img.parentNode, 1 + getNodeIndex(img));
			Aloha.getSelection().removeAllRanges();
			Aloha.getSelection().addRange(range);

			// IE adds width and height attributes for some reason, so remove those
			// to actually do what the spec says.
			img.removeAttribute("width");
			img.removeAttribute("height");
		}
	};

	//@}
	///// The insertLineBreak command /////
	//@{
	commands.insertlinebreak = {
		action: function (value, range) {
			// "Delete the contents of the active range, with strip wrappers false."
			deleteContents(range, {
				stripWrappers: false
			});

			// "If the active range's start node is neither editable nor an editing
			// host, abort these steps."
			if (!isEditable(range.startContainer) && !isEditingHost(range.startContainer)) {
				return;
			}

			// "If the active range's start node is an Element, and "br" is not an
			// allowed child of it, abort these steps."
			if (range.startContainer.nodeType == $_.Node.ELEMENT_NODE && !isAllowedChild("br", range.startContainer)) {
				return;
			}

			// "If the active range's start node is not an Element, and "br" is not
			// an allowed child of the active range's start node's parent, abort
			// these steps."
			if (range.startContainer.nodeType != $_.Node.ELEMENT_NODE && !isAllowedChild("br", range.startContainer.parentNode)) {
				return;
			}

			// "If the active range's start node is a Text node and its start
			// offset is zero, call collapse() on the context object's Selection,
			// with first argument equal to the active range's start node's parent
			// and second argument equal to the active range's start node's index."
			var newNode, newOffset;
			if (range.startContainer.nodeType == $_.Node.TEXT_NODE && range.startOffset == 0) {
				newNode = range.startContainer.parentNode;
				newOffset = getNodeIndex(range.startContainer);
				Aloha.getSelection().collapse(newNode, newOffset);
				range.setStart(newNode, newOffset);
				range.setEnd(newNode, newOffset);
			}

			// "If the active range's start node is a Text node and its start
			// offset is the length of its start node, call collapse() on the
			// context object's Selection, with first argument equal to the active
			// range's start node's parent and second argument equal to one plus
			// the active range's start node's index."
			if (range.startContainer.nodeType == $_.Node.TEXT_NODE && range.startOffset == getNodeLength(range.startContainer)) {
				newNode = range.startContainer.parentNode;
				newOffset = 1 + getNodeIndex(range.startContainer);
				Aloha.getSelection().collapse(newNode, newOffset);
				range.setStart(newNode, newOffset);
				range.setEnd(newNode, newOffset);
			}

			// "Let br be the result of calling createElement("br") on the context
			// object."
			var br = document.createElement("br");

			// "Call insertNode(br) on the active range."
			range.insertNode(br);

			// "Call collapse() on the context object's Selection, with br's parent
			// as the first argument and one plus br's index as the second
			// argument."
			Aloha.getSelection().collapse(br.parentNode, 1 + getNodeIndex(br));
			range.setStart(br.parentNode, 1 + getNodeIndex(br));
			range.setEnd(br.parentNode, 1 + getNodeIndex(br));

			// "If br is a collapsed line break, call createElement("br") on the
			// context object and let extra br be the result, then call
			// insertNode(extra br) on the active range."
			if (isCollapsedLineBreak(br)) {
				// TODO
				range.insertNode(createEndBreak());

				// Compensate for nonstandard implementations of insertNode
				Aloha.getSelection().collapse(br.parentNode, 1 + getNodeIndex(br));
				range.setStart(br.parentNode, 1 + getNodeIndex(br));
				range.setEnd(br.parentNode, 1 + getNodeIndex(br));
			}

			// IE7 is adding this styles: height: auto; min-height: 0px; max-height: none;
			// with that there is the ugly "IE-editable-outline"
			if (jQuery.browser.msie && jQuery.browser.version < 8) {
				br.parentNode.removeAttribute("style");
			}
		}
	};

	//@}
	///// The insertOrderedList command /////
	//@{
	commands.insertorderedlist = {
		// "Toggle lists with tag name "ol"."
		action: function (value, range) {
			toggleLists("ol", range);
		},
		// "True if the selection's list state is "mixed" or "mixed ol", false
		// otherwise."
		indeterm: function () {
			return (/^mixed( ol)?$/).test(getSelectionListState());
		},
		// "True if the selection's list state is "ol", false otherwise."
		state: function () {
			return getSelectionListState() == "ol";
		}
	};

	var listRelatedElements = {
		"LI": true,
		"DT": true,
		"DD": true
	};

	//@}
	///// The insertParagraph command /////
	//@{
	commands.insertparagraph = {
		action: function (value, range) {
			var i;

			// "Delete the contents of the active range."
			deleteContents(range);

			// clean lists in the editing host, this will remove any whitespace nodes around lists
			// because the following algorithm is not prepared to deal with them
			cleanLists(getEditingHostOf(range.startContainer), range);

			// "If the active range's start node is neither editable nor an editing
			// host, abort these steps."
			if (!isEditable(range.startContainer) && !isEditingHost(range.startContainer)) {
				return;
			}

			// "Let node and offset be the active range's start node and offset."
			var node = range.startContainer;
			var offset = range.startOffset;

			// "If node is a Text node, and offset is neither 0 nor the length of
			// node, call splitText(offset) on node."
			if (node.nodeType == $_.Node.TEXT_NODE && offset != 0 && offset != getNodeLength(node)) {
				node.splitText(offset);
			}

			// "If node is a Text node and offset is its length, set offset to one
			// plus the index of node, then set node to its parent."
			if (node.nodeType == $_.Node.TEXT_NODE && offset == getNodeLength(node)) {
				offset = 1 + getNodeIndex(node);
				node = node.parentNode;
			}

			// "If node is a Text or Comment node, set offset to the index of node,
			// then set node to its parent."
			if (node.nodeType == $_.Node.TEXT_NODE || node.nodeType == $_.Node.COMMENT_NODE) {
				offset = getNodeIndex(node);
				node = node.parentNode;
			}

			// "Call collapse(node, offset) on the context object's Selection."
			Aloha.getSelection().collapse(node, offset);
			range.setStart(node, offset);
			range.setEnd(node, offset);

			// "Let container equal node."
			var container = node;

			// "While container is not a single-line container, and container's
			// parent is editable and in the same editing host as node, set
			// container to its parent."
			while (!isSingleLineContainer(container) && isEditable(container.parentNode) && inSameEditingHost(node, container.parentNode)) {
				container = container.parentNode;
			}

			// "If container is not editable or not in the same editing host as
			// node or is not a single-line container:"
			if (!isEditable(container) || !inSameEditingHost(container, node) || !isSingleLineContainer(container)) {
				// "Let tag be the default single-line container name."
				var tag = defaultSingleLineContainerName;

				// "Block-extend the active range, and let new range be the
				// result."
				var newRange = blockExtend(range);

				// "Let node list be a list of nodes, initially empty."
				//
				// "Append to node list the first node in tree order that is
				// contained in new range and is an allowed child of "p", if any."
				var nodeList = getContainedNodes(newRange, function (node) {
					return isAllowedChild(node, "p");
				}).slice(0, 1);

				// "If node list is empty:"
				if (!nodeList.length) {
					// "If tag is not an allowed child of the active range's start
					// node, abort these steps."
					if (!isAllowedChild(tag, range.startContainer)) {
						return;
					}

					// "Set container to the result of calling createElement(tag)
					// on the context object."
					container = document.createElement(tag);

					// "Call insertNode(container) on the active range."
					range.insertNode(container);

					// "Call createElement("br") on the context object, and append
					// the result as the last child of container."
					// TODO not always
					container.appendChild(createEndBreak());

					// "Call collapse(container, 0) on the context object's
					// Selection."
					// TODO: remove selection from command
					Aloha.getSelection().collapse(container, 0);
					range.setStart(container, 0);
					range.setEnd(container, 0);

					// "Abort these steps."
					return;
				}

				// "While the nextSibling of the last member of node list is not
				// null and is an allowed child of "p", append it to node list."
				while (nodeList[nodeList.length - 1].nextSibling && isAllowedChild(nodeList[nodeList.length - 1].nextSibling, "p")) {
					nodeList.push(nodeList[nodeList.length - 1].nextSibling);
				}

				// "Wrap node list, with sibling criteria returning false and new
				// parent instructions returning the result of calling
				// createElement(tag) on the context object. Set container to the
				// result."
				container = wrap(
					nodeList,
					function () {
						return false;
					},
					function () {
						return document.createElement(tag);
					},
					range
				);
			}

			// "If container's local name is "address", "listing", or "pre":"
			var oldHeight, newHeight;
			if (container.tagName == "ADDRESS" || container.tagName == "LISTING" || container.tagName == "PRE") {
				// "Let br be the result of calling createElement("br") on the
				// context object."
				var br = document.createElement("br");

				// remember the old height
				oldHeight = container.offsetHeight;

				// "Call insertNode(br) on the active range."
				range.insertNode(br);

				// determine the new height
				newHeight = container.offsetHeight;

				// "Call collapse(node, offset + 1) on the context object's
				// Selection."
				Aloha.getSelection().collapse(node, offset + 1);
				range.setStart(node, offset + 1);
				range.setEnd(node, offset + 1);

				// "If br is the last descendant of container, let br be the result
				// of calling createElement("br") on the context object, then call
				// insertNode(br) on the active range." (Fix: only do this, if the container height did not change by inserting a single <br/>)
				//
				// Work around browser bugs: some browsers select the
				// newly-inserted node, not per spec.
				if (oldHeight == newHeight && !isDescendant(nextNode(br), container)) {
					// TODO check
					range.insertNode(createEndBreak());
					Aloha.getSelection().collapse(node, offset + 1);
					range.setEnd(node, offset + 1);
				}

				// "Abort these steps."
				return;
			}

			// "If container's local name is "li", "dt", or "dd"; and either it has
			// no children or it has a single child and that child is a br:"
			if (listRelatedElements[container.tagName] && (!container.hasChildNodes() || (container.childNodes.length == 1 && isNamedHtmlElement(container.firstChild, "br")))) {
				// "Split the parent of the one-node list consisting of container."
				splitParent([container], range);

				// "If container has no children, call createElement("br") on the
				// context object and append the result as the last child of
				// container."
				// only do this, if inserting the br does NOT modify the offset height of the container
				//			if (!container.hasChildNodes()) {
				//				var oldHeight = container.offsetHeight, endBr = createEndBreak();
				//				container.appendChild(endBr);
				//				if (container.offsetHeight !== oldHeight) {
				//					container.removeChild(endBr);
				//				}
				//			}

				// "If container is a dd or dt, and it is not an allowed child of
				// any of its ancestors in the same editing host, set the tag name
				// of container to the default single-line container name and let
				// container be the result."
				if (isHtmlElementInArray(container, ["dd", "dt"]) && $_(getAncestors(container)).every(function (ancestor) { return !inSameEditingHost(container, ancestor) || !isAllowedChild(container, ancestor); })) {
					container = setTagName(container, defaultSingleLineContainerName, range);
				}

				// "Fix disallowed ancestors of container."
				fixDisallowedAncestors(container, range);

				// fix invalid nested lists
				if (isNamedHtmlElement(container, 'li') && isNamedHtmlElement(container.nextSibling, "li") && isHtmlElementInArray(container.nextSibling.firstChild, ["ol", "ul"])) {
					// we found a li containing only a br followed by a li containing a list as first element: merge the two li's
					var listParent = container.nextSibling,
						length = container.nextSibling.childNodes.length;
					for (i = 0; i < length; i++) {
						// we always move the first child into the container
						container.appendChild(listParent.childNodes[0]);
					}
					listParent.parentNode.removeChild(listParent);
				}

				// "Abort these steps."
				return;
			}

			// "Let new line range be a new range whose start is the same as
			// the active range's, and whose end is (container, length of
			// container)."
			var newLineRange = Aloha.createRange();
			newLineRange.setStart(range.startContainer, range.startOffset);
			newLineRange.setEnd(container, getNodeLength(container));

			// "While new line range's start offset is zero and its start node is
			// not container, set its start to (parent of start node, index of
			// start node)."
			while (newLineRange.startOffset == 0 && newLineRange.startContainer != container) {
				newLineRange.setStart(newLineRange.startContainer.parentNode, getNodeIndex(newLineRange.startContainer));
			}

			// "While new line range's start offset is the length of its start node
			// and its start node is not container, set its start to (parent of
			// start node, 1 + index of start node)."
			while (newLineRange.startOffset == getNodeLength(newLineRange.startContainer) && newLineRange.startContainer != container) {
				newLineRange.setStart(newLineRange.startContainer.parentNode, 1 + getNodeIndex(newLineRange.startContainer));
			}

			// "Let end of line be true if new line range contains either nothing
			// or a single br, and false otherwise."
			var containedInNewLineRange = getContainedNodes(newLineRange);
			var endOfLine = !containedInNewLineRange.length || (containedInNewLineRange.length == 1 && isNamedHtmlElement(containedInNewLineRange[0], "br"));

			// "If the local name of container is "h1", "h2", "h3", "h4", "h5", or
			// "h6", and end of line is true, let new container name be the default
			// single-line container name."
			var newContainerName;
			if (/^H[1-6]$/.test(container.tagName) && endOfLine) {
				newContainerName = defaultSingleLineContainerName;

				// "Otherwise, if the local name of container is "dt" and end of line
				// is true, let new container name be "dd"."
			} else if (container.tagName == "DT" && endOfLine) {
				newContainerName = "dd";

				// "Otherwise, if the local name of container is "dd" and end of line
				// is true, let new container name be "dt"."
			} else if (container.tagName == "DD" && endOfLine) {
				newContainerName = "dt";

				// "Otherwise, let new container name be the local name of container."
			} else {
				newContainerName = container.tagName.toLowerCase();
			}

			// "Let new container be the result of calling createElement(new
			// container name) on the context object."
			var newContainer = document.createElement(newContainerName);

			// "Copy all non empty attributes of the container to new container."
			copyAttributes(container, newContainer);

			// "If new container has an id attribute, unset it."
			newContainer.removeAttribute("id");

			// "Insert new container into the parent of container immediately after
			// container."
			container.parentNode.insertBefore(newContainer, container.nextSibling);

			// "Let contained nodes be all nodes contained in new line range."
			var containedNodes = getAllContainedNodes(newLineRange);

			// "Let frag be the result of calling extractContents() on new line
			// range."
			var frag = newLineRange.extractContents();

			// "Unset the id attribute (if any) of each Element descendant of frag
			// that is not in contained nodes."
			var descendants = getDescendants(frag);
			for (i = 0; i < descendants.length; i++) {
				if (descendants[i].nodeType == $_.Node.ELEMENT_NODE && $_(containedNodes).indexOf(descendants[i]) == -1) {
					descendants[i].removeAttribute("id");
				}
			}

			var fragChildren = [],
				fragChild = frag.firstChild;
			if (fragChild) {
				do {
					if (!isWhitespaceNode(fragChild)) {
						fragChildren.push(fragChild);
					}
				} while (null != (fragChild = fragChild.nextSibling));
			}

			// if newContainer is a li and frag contains only a list, we add a br in the li (but only if the height would not change)
			if (isNamedHtmlElement(newContainer, 'li') && fragChildren.length && isHtmlElementInArray(fragChildren[0], ["ul", "ol"])) {
				oldHeight = newContainer.offsetHeight;
				var endBr = createEndBreak();
				newContainer.appendChild(endBr);
				newHeight = newContainer.offsetHeight;
				if (oldHeight !== newHeight) {
					newContainer.removeChild(endBr);
				}
			}

			// "Call appendChild(frag) on new container."
			newContainer.appendChild(frag);

			// "If container has no visible children, call createElement("br") on
			// the context object, and append the result as the last child of
			// container."
			ensureContainerEditable(container);

			// "If new container has no visible children, call createElement("br")
			// on the context object, and append the result as the last child of
			// new container."
			ensureContainerEditable(newContainer);

			// "Call collapse(new container, 0) on the context object's Selection."
			Aloha.getSelection().collapse(newContainer, 0);
			range.setStart(newContainer, 0);
			range.setEnd(newContainer, 0);
		}
	};

	//@}
	///// The insertText command /////
	//@{
	commands.inserttext = {
		action: function (value, range) {
			var i;

			// "Delete the contents of the active range, with strip wrappers
			// false."
			deleteContents(range, {
				stripWrappers: false
			});

			// "If the active range's start node is neither editable nor an editing
			// host, abort these steps."
			if (!isEditable(range.startContainer) && !isEditingHost(range.startContainer)) {
				return;
			}

			// "If value's length is greater than one:"
			if (value.length > 1) {
				// "For each element el in value, take the action for the
				// insertText command, with value equal to el."
				for (i = 0; i < value.length; i++) {
					commands.inserttext.action(value[i], range);
				}

				// "Abort these steps."
				return;
			}

			// "If value is the empty string, abort these steps."
			if (value == "") {
				return;
			}

			// "If value is a newline (U+00A0), take the action for the
			// insertParagraph command and abort these steps."
			if (value == "\n") {
				commands.insertparagraph.action('', range);
				return;
			}

			// "Let node and offset be the active range's start node and offset."
			var node = range.startContainer;
			var offset = range.startOffset;

			// "If node has a child whose index is offset − 1, and that child is a
			// Text node, set node to that child, then set offset to node's
			// length."
			if (0 <= offset - 1 && offset - 1 < node.childNodes.length && node.childNodes[offset - 1].nodeType == $_.Node.TEXT_NODE) {
				node = node.childNodes[offset - 1];
				offset = getNodeLength(node);
			}

			// "If node has a child whose index is offset, and that child is a Text
			// node, set node to that child, then set offset to zero."
			if (0 <= offset && offset < node.childNodes.length && node.childNodes[offset].nodeType == $_.Node.TEXT_NODE) {
				node = node.childNodes[offset];
				offset = 0;
			}

			// "If value is a space (U+0020), and either node is an Element whose
			// resolved value for "white-space" is neither "pre" nor "pre-wrap" or
			// node is not an Element but its parent is an Element whose resolved
			// value for "white-space" is neither "pre" nor "pre-wrap", set value
			// to a non-breaking space (U+00A0)."
			var refElement = node.nodeType == $_.Node.ELEMENT_NODE ? node : node.parentNode;
			if (value == " " && refElement.nodeType == $_.Node.ELEMENT_NODE && jQuery.inArray($_.getComputedStyle(refElement).whiteSpace, ["pre", "pre-wrap"]) == -1) {
				value = "\xa0";
			}

			// "Record current overrides, and let overrides be the result."
			var overrides = recordCurrentOverrides(range);

			// "If node is a Text node:"
			if (node.nodeType == $_.Node.TEXT_NODE) {
				// "Call insertData(offset, value) on node."
				node.insertData(offset, value);

				// "Call collapse(node, offset) on the context object's Selection."
				Aloha.getSelection().collapse(node, offset);
				range.setStart(node, offset);

				// "Call extend(node, offset + 1) on the context object's
				// Selection."
				Aloha.getSelection().extend(node, offset + 1);
				range.setEnd(node, offset + 1);

				// "Otherwise:"
			} else {
				// "If node has only one child, which is a collapsed line break,
				// remove its child from it."
				//
				// FIXME: IE incorrectly returns false here instead of true
				// sometimes?
				if (node.childNodes.length == 1 && isCollapsedLineBreak(node.firstChild)) {
					node.removeChild(node.firstChild);
				}

				// "Let text be the result of calling createTextNode(value) on the
				// context object."
				var text = document.createTextNode(value);

				// "Call insertNode(text) on the active range."
				range.insertNode(text);

				// "Call collapse(text, 0) on the context object's Selection."
				Aloha.getSelection().collapse(text, 0);
				range.setStart(text, 0);

				// "Call extend(text, 1) on the context object's Selection."
				Aloha.getSelection().extend(text, 1);
				range.setEnd(text, 1);
			}

			// "Restore states and values from overrides."
			restoreStatesAndValues(overrides);

			// "Canonicalize whitespace at the active range's start."
			canonicalizeWhitespace(range.startContainer, range.startOffset);

			// "Canonicalize whitespace at the active range's end."
			canonicalizeWhitespace(range.endContainer, range.endOffset);

			// "Call collapseToEnd() on the context object's Selection."
			Aloha.getSelection().collapseToEnd();
			range.collapse(false);
		}
	};

	//@}
	///// The insertUnorderedList command /////
	//@{
	commands.insertunorderedlist = {
		// "Toggle lists with tag name "ul"."
		action: function (value, range) {
			toggleLists("ul", range);
		},
		// "True if the selection's list state is "mixed" or "mixed ul", false
		// otherwise."
		indeterm: function () {
			return (/^mixed( ul)?$/).test(getSelectionListState());
		},
		// "True if the selection's list state is "ul", false otherwise."
		state: function () {
			return getSelectionListState() == "ul";
		}
	};

	//@}
	///// The justifyCenter command /////
	//@{
	commands.justifycenter = {
		// "Justify the selection with alignment "center"."
		action: function (value, range) {
			justifySelection("center", range);
		},
		indeterm: function () {
			// "Block-extend the active range. Return true if among visible
			// editable nodes that are contained in the result and have no
			// children, at least one has alignment value "center" and at least one
			// does not. Otherwise return false."
			var nodes = getAllContainedNodes(blockExtend(getActiveRange()), function (node) {
				return isEditable(node) && isVisible(node) && !node.hasChildNodes();
			});
			return $_(nodes).some(function (node) { return getAlignmentValue(node) == "center"; })
				&& $_(nodes).some(function (node) { return getAlignmentValue(node) != "center"; });
		},
		state: function () {
			// "Block-extend the active range. Return true if there is at least one
			// visible editable node that is contained in the result and has no
			// children, and all such nodes have alignment value "center".
			// Otherwise return false."
			var nodes = getAllContainedNodes(blockExtend(getActiveRange()), function (node) {
				return isEditable(node) && isVisible(node) && !node.hasChildNodes();
			});
			return nodes.length && $_(nodes).every(function (node) {
				return getAlignmentValue(node) == "center";
			});
		},
		value: function () {
			// "Block-extend the active range, and return the alignment value of
			// the first visible editable node that is contained in the result and
			// has no children. If there is no such node, return "left"."
			var nodes = getAllContainedNodes(blockExtend(getActiveRange()), function (node) {
				return isEditable(node) && isVisible(node) && !node.hasChildNodes();
			});
			if (nodes.length) {
				return getAlignmentValue(nodes[0]);
			}
			return "left";
		}
	};

	//@}
	///// The justifyFull command /////
	//@{
	commands.justifyfull = {
		// "Justify the selection with alignment "justify"."
		action: function (value, range) {
			justifySelection("justify", range);
		},
		indeterm: function () {
			// "Block-extend the active range. Return true if among visible
			// editable nodes that are contained in the result and have no
			// children, at least one has alignment value "justify" and at least
			// one does not. Otherwise return false."
			var nodes = getAllContainedNodes(blockExtend(getActiveRange()), function (node) {
				return isEditable(node) && isVisible(node) && !node.hasChildNodes();
			});
			return $_(nodes).some(function (node) { return getAlignmentValue(node) == "justify"; })
				&& $_(nodes).some(function (node) { return getAlignmentValue(node) != "justify"; });
		},
		state: function () {
			// "Block-extend the active range. Return true if there is at least one
			// visible editable node that is contained in the result and has no
			// children, and all such nodes have alignment value "justify".
			// Otherwise return false."
			var nodes = getAllContainedNodes(blockExtend(getActiveRange()), function (node) {
				return isEditable(node) && isVisible(node) && !node.hasChildNodes();
			});
			return nodes.length && $_(nodes).every(function (node) {
				return getAlignmentValue(node) == "justify";
			});
		},
		value: function () {
			// "Block-extend the active range, and return the alignment value of
			// the first visible editable node that is contained in the result and
			// has no children. If there is no such node, return "left"."
			var nodes = getAllContainedNodes(blockExtend(getActiveRange()), function (node) {
				return isEditable(node) && isVisible(node) && !node.hasChildNodes();
			});
			if (nodes.length) {
				return getAlignmentValue(nodes[0]);
			}
			return "left";
		}
	};

	//@}
	///// The justifyLeft command /////
	//@{
	commands.justifyleft = {
		// "Justify the selection with alignment "left"."
		action: function (value, range) {
			justifySelection("left", range);
		},
		indeterm: function () {
			// "Block-extend the active range. Return true if among visible
			// editable nodes that are contained in the result and have no
			// children, at least one has alignment value "left" and at least one
			// does not. Otherwise return false."
			var nodes = getAllContainedNodes(blockExtend(getActiveRange()), function (node) {
				return isEditable(node) && isVisible(node) && !node.hasChildNodes();
			});
			return $_(nodes).some(function (node) { return getAlignmentValue(node) == "left"; })
				&& $_(nodes).some(function (node) { return getAlignmentValue(node) != "left"; });
		},
		state: function () {
			// "Block-extend the active range. Return true if there is at least one
			// visible editable node that is contained in the result and has no
			// children, and all such nodes have alignment value "left".  Otherwise
			// return false."
			var nodes = getAllContainedNodes(blockExtend(getActiveRange()), function (node) {
				return isEditable(node) && isVisible(node) && !node.hasChildNodes();
			});
			return nodes.length && $_(nodes).every(function (node) {
				return getAlignmentValue(node) == "left";
			});
		},
		value: function () {
			// "Block-extend the active range, and return the alignment value of
			// the first visible editable node that is contained in the result and
			// has no children. If there is no such node, return "left"."
			var nodes = getAllContainedNodes(blockExtend(getActiveRange()), function (node) {
				return isEditable(node) && isVisible(node) && !node.hasChildNodes();
			});
			if (nodes.length) {
				return getAlignmentValue(nodes[0]);
			}
			return "left";
		}
	};

	//@}
	///// The justifyRight command /////
	//@{
	commands.justifyright = {
		// "Justify the selection with alignment "right"."
		action: function (value, range) {
			justifySelection("right", range);
		},
		indeterm: function () {
			// "Block-extend the active range. Return true if among visible
			// editable nodes that are contained in the result and have no
			// children, at least one has alignment value "right" and at least one
			// does not. Otherwise return false."
			var nodes = getAllContainedNodes(blockExtend(getActiveRange()), function (node) {
				return isEditable(node) && isVisible(node) && !node.hasChildNodes();
			});
			return $_(nodes).some(function (node) { return getAlignmentValue(node) == "right"; })
				&& $_(nodes).some(function (node) { return getAlignmentValue(node) != "right"; });
		},
		state: function () {
			// "Block-extend the active range. Return true if there is at least one
			// visible editable node that is contained in the result and has no
			// children, and all such nodes have alignment value "right".
			// Otherwise return false."
			var nodes = getAllContainedNodes(blockExtend(getActiveRange()), function (node) {
				return isEditable(node) && isVisible(node) && !node.hasChildNodes();
			});
			return nodes.length && $_(nodes).every(function (node) {
				return getAlignmentValue(node) == "right";
			});
		},
		value: function () {
			// "Block-extend the active range, and return the alignment value of
			// the first visible editable node that is contained in the result and
			// has no children. If there is no such node, return "left"."
			var nodes = getAllContainedNodes(blockExtend(getActiveRange()), function (node) {
				return isEditable(node) && isVisible(node) && !node.hasChildNodes();
			});
			if (nodes.length) {
				return getAlignmentValue(nodes[0]);
			}
			return "left";
		}
	};

	//@}
	///// The outdent command /////
	//@{
	commands.outdent = {
		action: function () {
			// "Let items be a list of all lis that are ancestor containers of the
			// range's start and/or end node."
			//
			// It's annoying to get this in tree order using functional stuff
			// without doing getDescendants(document), which is slow, so I do it
			// imperatively.
			var items = [];
			(function () {
				var ancestorContainer;
				for (ancestorContainer = getActiveRange().endContainer;
					     ancestorContainer != getActiveRange().commonAncestorContainer;
					     ancestorContainer = ancestorContainer.parentNode) {
					if (isNamedHtmlElement(ancestorContainer, "li")) {
						items.unshift(ancestorContainer);
					}
				}
				for (ancestorContainer = getActiveRange().startContainer;
					     ancestorContainer;
					     ancestorContainer = ancestorContainer.parentNode) {
					if (isNamedHtmlElement(ancestorContainer, "li")) {
						items.unshift(ancestorContainer);
					}
				}
			}());

			// "For each item in items, normalize sublists of item."
			$_(items).forEach(function (thisArg) {
				normalizeSublists(thisArg, getActiveRange());
			});

			// "Block-extend the active range, and let new range be the result."
			var newRange = blockExtend(getActiveRange());

			// "Let node list be a list of nodes, initially empty."
			//
			// "For each node node contained in new range, append node to node list
			// if the last member of node list (if any) is not an ancestor of node;
			// node is editable; and either node has no editable descendants, or is
			// an ol or ul, or is an li whose parent is an ol or ul."
			var nodeList = getContainedNodes(newRange, function (node) {
				return isEditable(node) && (!$_(getDescendants(node)).some(isEditable) || isHtmlElementInArray(node, ["ol", "ul"]) || (isNamedHtmlElement(node, 'li') && isHtmlElementInArray(node.parentNode, ["ol", "ul"])));
			});

			// "While node list is not empty:"
			while (nodeList.length) {
				// "While the first member of node list is an ol or ul or is not
				// the child of an ol or ul, outdent it and remove it from node
				// list."
				while (nodeList.length && (isHtmlElementInArray(nodeList[0], ["OL", "UL"]) || !isHtmlElementInArray(nodeList[0].parentNode, ["OL", "UL"]))) {
					outdentNode(nodeList.shift(), newRange);
				}

				// "If node list is empty, break from these substeps."
				if (!nodeList.length) {
					break;
				}

				// "Let sublist be a list of nodes, initially empty."
				var sublist = [];

				// "Remove the first member of node list and append it to sublist."
				sublist.push(nodeList.shift());

				// "While the first member of node list is the nextSibling of the
				// last member of sublist, and the first member of node list is not
				// an ol or ul, remove the first member of node list and append it
				// to sublist."
				while (nodeList.length && nodeList[0] == sublist[sublist.length - 1].nextSibling && !isHtmlElementInArray(nodeList[0], ["OL", "UL"])) {
					sublist.push(nodeList.shift());
				}

				// "Record the values of sublist, and let values be the result."
				var values = recordValues(sublist);

				// "Split the parent of sublist, with new parent null."
				splitParent(sublist, newRange);

				// "Fix disallowed ancestors of each member of sublist."
				$_(sublist).forEach(fixDisallowedAncestors);

				// "Restore the values from values."
				restoreValues(values, newRange);
			}
		}
	};

	//@}

	//////////////////////////////////
	///// Miscellaneous commands /////
	//////////////////////////////////

	///// The selectAll command /////
	//@{
	commands.selectall = {
		// Note, this ignores the whole globalRange/getActiveRange() thing and
		// works with actual selections.  Not suitable for autoimplementation.html.
		action: function () {
			// "Let target be the body element of the context object."
			var target = document.body;

			// "If target is null, let target be the context object's
			// documentElement."
			if (!target) {
				target = document.documentElement;
			}

			// "If target is null, call getSelection() on the context object, and
			// call removeAllRanges() on the result."
			if (!target) {
				Aloha.getSelection().removeAllRanges();

				// "Otherwise, call getSelection() on the context object, and call
				// selectAllChildren(target) on the result."
			} else {
				Aloha.getSelection().selectAllChildren(target);
			}
		}
	};

	//@}
	///// The styleWithCSS command /////
	//@{
	commands.stylewithcss = {
		action: function (value) {
			// "If value is an ASCII case-insensitive match for the string
			// "false", set the CSS styling flag to false. Otherwise, set the
			// CSS styling flag to true."
			cssStylingFlag = String(value).toLowerCase() != "false";
		},
		state: function () {
			return cssStylingFlag;
		}
	};

	//@}
	///// The useCSS command /////
	//@{
	commands.usecss = {
		action: function (value) {
			// "If value is an ASCII case-insensitive match for the string "false",
			// set the CSS styling flag to true. Otherwise, set the CSS styling
			// flag to false."
			cssStylingFlag = String(value).toLowerCase() == "false";
		}
	};
	//@}

	// Some final setup
	//@{
	(function () {
		// Opera 11.50 doesn't implement Object.keys, so I have to make an explicit
		// temporary, which means I need an extra closure to not leak the temporaries
		// into the global namespace.  >:(
		var commandNames = [];
		var command;
		for (command in commands) {
			if (commands.hasOwnProperty(command)) {
				commandNames.push(command);
			}
		}
		$_(commandNames).forEach(function (command) {
			// "If a command does not have a relevant CSS property specified, it
			// defaults to null."
			if (null == commands[command].relevantCssProperty) {
				commands[command].relevantCssProperty = null;
			}

			// "If a command has inline command activated values defined but
			// nothing else defines when it is indeterminate, it is indeterminate
			// if among editable Text nodes effectively contained in the active
			// range, there is at least one whose effective command value is one of
			// the given values and at least one whose effective command value is
			// not one of the given values."
			if (null != commands[command].inlineCommandActivatedValues && null == commands[command].indeterm) {
				commands[command].indeterm = function (range) {
					var values = $_(getAllEffectivelyContainedNodes(range, function (node) { return isEditable(node) && node.nodeType == $_.Node.TEXT_NODE; }))
						.map(function (node) { return getEffectiveCommandValue(node, command); });

					var matchingValues = $_(values).filter(function (value) {
						return $_(commands[command].inlineCommandActivatedValues).indexOf(value) != -1;
					});

					return matchingValues.length >= 1 && values.length - matchingValues.length >= 1;
				};
			}

			// "If a command has inline command activated values defined, its state
			// is true if either no editable Text node is effectively contained in
			// the active range, and the active range's start node's effective
			// command value is one of the given values; or if there is at least
			// one editable Text node effectively contained in the active range,
			// and all of them have an effective command value equal to one of the
			// given values."
			if (null != commands[command].inlineCommandActivatedValues) {
				commands[command].state = function (range) {
					var nodes = getAllEffectivelyContainedNodes(range, function (node) {
						return isEditable(node) && node.nodeType == $_.Node.TEXT_NODE;
					});

					if (nodes.length == 0) {
						return $_(commands[command].inlineCommandActivatedValues).indexOf(getEffectiveCommandValue(range.startContainer, command)) != -1;
					}
					return $_(nodes).every(function (node) {
						return $_(commands[command].inlineCommandActivatedValues).indexOf(getEffectiveCommandValue(node, command)) != -1;
					});
				};
			}

			// "If a command is a standard inline value command, it is
			// indeterminate if among editable Text nodes that are effectively
			// contained in the active range, there are two that have distinct
			// effective command values. Its value is the effective command value
			// of the first editable Text node that is effectively contained in the
			// active range, or if there is no such node, the effective command
			// value of the active range's start node."
			if (null != commands[command].standardInlineValueCommand) {
				commands[command].indeterm = function () {
					var values = $_(getAllEffectivelyContainedNodes(getActiveRange())).filter(function (node) { return isEditable(node) && node.nodeType == $_.Node.TEXT_NODE; }, true)
						.map(function (node) { return getEffectiveCommandValue(node, command); });
					var i;
					for (i = 1; i < values.length; i++) {
						if (values[i] != values[i - 1]) {
							return true;
						}
					}
					return false;
				};

				commands[command].value = function (range) {
					var refNode = getAllEffectivelyContainedNodes(range, function (node) {
						return isEditable(node) && node.nodeType == $_.Node.TEXT_NODE;
					})[0];

					if (typeof refNode == "undefined") {
						refNode = range.startContainer;
					}

					return getEffectiveCommandValue(refNode, command);
				};
			}
		});
	}());
	//@}
	return {
		commands: commands,
		execCommand: myExecCommand,
		queryCommandIndeterm: myQueryCommandIndeterm,
		queryCommandState: myQueryCommandState,
		queryCommandValue: myQueryCommandValue,
		queryCommandEnabled: myQueryCommandEnabled,
		queryCommandSupported: myQueryCommandSupported,
		copyAttributes: copyAttributes,
		createEndBreak: createEndBreak,
		isEndBreak: isEndBreak,
		ensureContainerEditable: ensureContainerEditable,
		isEditingHost: isEditingHost,
		isEditable: isEditable
	};
}); // end define
// vim: foldmarker=@{,@} foldmethod=marker
;
/* selection.js is part of Aloha Editor project http://aloha-editor.org
 *
 * Aloha Editor is a WYSIWYG HTML5 inline editing library and editor. 
 * Copyright (c) 2010-2012 Gentics Software GmbH, Vienna, Austria.
 * Contributors http://aloha-editor.org/contribution.php 
 * 
 * Aloha Editor is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or any later version.
 *
 * Aloha Editor is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 * 
 * As an additional permission to the GNU GPL version 2, you may distribute
 * non-source (e.g., minimized or compacted) forms of the Aloha-Editor
 * source code without the copy of the GNU GPL normally required,
 * provided you include this license notice and a URL through which
 * recipients can access the Corresponding Source.
 */
define('aloha/selection',[
	'aloha/core',
	'jquery',
	'util/class',
	'util/range',
	'util/arrays',
	'util/strings',
	'aloha/console',
	'PubSub',
	'aloha/engine',
	'aloha/ecma5shims',
	'aloha/rangy-core'
], function (
	Aloha,
	jQuery,
	Class,
	Range,
	Arrays,
	Strings,
	console,
	PubSub,
	Engine,
	e5s
) {
	

	var GENTICS = window.GENTICS;

	function isCollapsedAndEmptyOrEndBr(rangeObject) {
		var firstChild;
		if (rangeObject.startContainer !== rangeObject.endContainer) {
			return false;
		}
		// check whether the container starts in an element node
		if (rangeObject.startContainer.nodeType != 1) {
			return false;
		}
		firstChild = rangeObject.startContainer.firstChild;
		return (!firstChild || (!firstChild.nextSibling && firstChild.nodeName == 'BR'));
	}

	function isCollapsedAndEndBr(rangeObject) {
		if (rangeObject.startContainer !== rangeObject.endContainer) {
			return false;
		}
		if (rangeObject.startContainer.nodeType != 1) {
			return false;
		}
		return Engine.isEndBreak(rangeObject.startContainer);
	}

	var prevStartContext = null;
	var prevEndContext = null;

	function makeContextHtml(node, parents) {
		var result = [],
			parent,
			len,
			i;
		if (1 === node.nodeType && node.nodeName !== 'BODY' && node.nodeName !== 'HTML') {
			result.push(node.cloneNode(false).outerHTML);
		} else {
			result.push('#' + node.nodeType);
		}
		for (i = 0, len = parents.length; i < len; i++) {
			parent = parents[i];
			if (parent.nodeName === 'BODY' || parent.nodeName === 'HTML') {
				// Although we limit the ancestors in most cases to the
				// active editable, in some cases (copy&paste) the
				// parent may be outside.
				// On IE7 this means the following code may clone the
				// HTML node too, which causes the browser to crash.
				// On other browsers, this is just an optimization
				// because the body and html elements should probably
				// not be considered part of the context of an edit
				// operation.
				break;
			}
			result.push(parent.cloneNode(false).outerHTML);
		}
		return result.join('');
	}

	function getChangedContext(node, context) {
		var until = Aloha.activeEditable ? Aloha.activeEditable.obj.parent()[0] : null;
		var parents = jQuery(node).parentsUntil(until).get();
		var html = makeContextHtml(node, parents);
		var equal = (context && node === context.node && Arrays.equal(context.parents, parents) && html === context.html);
		return equal ? null : {
			node: node,
			parents: parents,
			html: html
		};
	}

	function triggerSelectionContextChanged(rangeObject, event) {
		var startContainer = rangeObject.startContainer;
		var endContainer = rangeObject.endContainer;
		if (!startContainer || !endContainer) {
			console.warn("aloha/selection", "encountered range object without start or end container");
			return;
		}
		var startContext = getChangedContext(startContainer, prevStartContext);
		var endContext = getChangedContext(endContainer, prevEndContext);
		if (!startContext && !endContext) {
			return;
		}
		prevStartContext = startContext;
		prevEndContext = endContext;

		/**
		 * @api documented in the guides
		 */
		PubSub.pub('aloha.selection.context-change', {
			range: rangeObject,
			event: event
		});
	}

	/**
	 * @namespace Aloha
	 * @class Selection
	 * This singleton class always represents the current user selection
	 * @singleton
	 */
	var Selection = Class.extend({
		_constructor: function () {
			// Pseudo Range Clone being cleaned up for better HTML wrapping support
			this.rangeObject = {};

			this.preventSelectionChangedFlag = false; // will remember if someone urged us to skip the next aloha-selection-changed event

			// define basics first
			this.tagHierarchy = {
				'textNode': {},
				'abbr': {
					'textNode': true
				},
				'b': {
					'textNode': true,
					'b': true,
					'i': true,
					'em': true,
					'sup': true,
					'sub': true,
					'br': true,
					'span': true,
					'img': true,
					'a': true,
					'del': true,
					'ins': true,
					'u': true,
					'cite': true,
					'q': true,
					'code': true,
					'abbr': true,
					'strong': true
				},
				'pre': {
					'textNode': true,
					'b': true,
					'i': true,
					'em': true,
					'sup': true,
					'sub': true,
					'br': true,
					'span': true,
					'img': true,
					'a': true,
					'del': true,
					'ins': true,
					'u': true,
					'cite': true,
					'q': true,
					'code': true,
					'abbr': true
				},
				'blockquote': {
					'textNode': true,
					'b': true,
					'i': true,
					'em': true,
					'sup': true,
					'sub': true,
					'br': true,
					'span': true,
					'img': true,
					'a': true,
					'del': true,
					'ins': true,
					'u': true,
					'cite': true,
					'q': true,
					'code': true,
					'abbr': true,
					'p': true,
					'h1': true,
					'h2': true,
					'h3': true,
					'h4': true,
					'h5': true,
					'h6': true
				},
				'ins': {
					'textNode': true,
					'b': true,
					'i': true,
					'em': true,
					'sup': true,
					'sub': true,
					'br': true,
					'span': true,
					'img': true,
					'a': true,
					'u': true,
					'p': true,
					'h1': true,
					'h2': true,
					'h3': true,
					'h4': true,
					'h5': true,
					'h6': true
				},
				'ul': {
					'li': true
				},
				'ol': {
					'li': true
				},
				'li': {
					'textNode': true,
					'b': true,
					'i': true,
					'em': true,
					'sup': true,
					'sub': true,
					'br': true,
					'span': true,
					'img': true,
					'ul': true,
					'ol': true,
					'h1': true,
					'h2': true,
					'h3': true,
					'h4': true,
					'h5': true,
					'h6': true,
					'del': true,
					'ins': true,
					'u': true,
					'a': true
				},
				'tr': {
					'td': true,
					'th': true
				},
				'table': {
					'tr': true
				},
				'div': {
					'textNode': true,
					'b': true,
					'i': true,
					'em': true,
					'sup': true,
					'sub': true,
					'br': true,
					'span': true,
					'img': true,
					'ul': true,
					'ol': true,
					'table': true,
					'h1': true,
					'h2': true,
					'h3': true,
					'h4': true,
					'h5': true,
					'h6': true,
					'del': true,
					'ins': true,
					'u': true,
					'p': true,
					'div': true,
					'pre': true,
					'blockquote': true,
					'a': true
				},
				'h1': {
					'textNode': true,
					'b': true,
					'i': true,
					'em': true,
					'sup': true,
					'sub': true,
					'br': true,
					'span': true,
					'img': true,
					'a': true,
					'del': true,
					'ins': true,
					'u': true
				}
			};

			// now reference the basics for all other equal tags (important: don't forget to include
			// the basics itself as reference: 'b' : this.tagHierarchy.b
			this.tagHierarchy = {
				'textNode': this.tagHierarchy.textNode,
				'abbr': this.tagHierarchy.abbr,
				'br': this.tagHierarchy.textNode,
				'img': this.tagHierarchy.textNode,
				'b': this.tagHierarchy.b,
				'strong': this.tagHierarchy.b,
				'code': this.tagHierarchy.b,
				'q': this.tagHierarchy.b,
				'blockquote': this.tagHierarchy.blockquote,
				'cite': this.tagHierarchy.b,
				'i': this.tagHierarchy.b,
				'em': this.tagHierarchy.b,
				'sup': this.tagHierarchy.b,
				'sub': this.tagHierarchy.b,
				'span': this.tagHierarchy.b,
				'del': this.tagHierarchy.del,
				'ins': this.tagHierarchy.ins,
				'u': this.tagHierarchy.b,
				'p': this.tagHierarchy.b,
				'pre': this.tagHierarchy.pre,
				'a': this.tagHierarchy.b,
				'ul': this.tagHierarchy.ul,
				'ol': this.tagHierarchy.ol,
				'li': this.tagHierarchy.li,
				'td': this.tagHierarchy.li,
				'div': this.tagHierarchy.div,
				'h1': this.tagHierarchy.h1,
				'h2': this.tagHierarchy.h1,
				'h3': this.tagHierarchy.h1,
				'h4': this.tagHierarchy.h1,
				'h5': this.tagHierarchy.h1,
				'h6': this.tagHierarchy.h1,
				'table': this.tagHierarchy.table
			};

			// When applying this elements to selection they will replace the assigned elements
			this.replacingElements = {
				'h1': {
					'p': true,
					'h1': true,
					'h2': true,
					'h3': true,
					'h4': true,
					'h5': true,
					'h6': true,
					'pre': true,
					'blockquote': true
				}
			};
			this.replacingElements = {
				'h1': this.replacingElements.h1,
				'h2': this.replacingElements.h1,
				'h3': this.replacingElements.h1,
				'h4': this.replacingElements.h1,
				'h5': this.replacingElements.h1,
				'h6': this.replacingElements.h1,
				'pre': this.replacingElements.h1,
				'p': this.replacingElements.h1,
				'blockquote': this.replacingElements.h1
			};
			this.allowedToStealElements = {
				'h1': {
					'textNode': true
				}
			};
			this.allowedToStealElements = {
				'h1': this.allowedToStealElements.h1,
				'h2': this.allowedToStealElements.h1,
				'h3': this.allowedToStealElements.h1,
				'h4': this.allowedToStealElements.h1,
				'h5': this.allowedToStealElements.h1,
				'h6': this.allowedToStealElements.h1,
				'p': this.tagHierarchy.b
			};
		},

		/**
		 * Class definition of a SelectionTree (relevant for all formatting / markup changes)
		 * TODO: remove this (was moved to range.js)
		 * Structure:
		 * +
		 * |-domobj: <reference to the DOM Object> (NOT jQuery)
		 * |-selection: defines if this node is marked by user [none|partial|full]
		 * |-children: recursive structure like this
		 * @hide
		 */
		SelectionTree: function () {
			this.domobj = {};
			this.selection = undefined;
			this.children = [];
		},

		/**
		 * INFO: Method is used for integration with Gentics Aloha, has no use otherwise
		 * Updates the rangeObject according to the current user selection
		 * Method is always called on selection change
		 * @param objectClicked Object that triggered the selectionChange event
		 * @return true when rangeObject was modified, false otherwise
		 * @hide
		 */
		onChange: function (objectClicked, event, timeout) {
			if (this.updateSelectionTimeout) {
				window.clearTimeout(this.updateSelectionTimeout);
			}

			// We have to update the selection in a timeout due to an IE
			// bug that is is caused by selecting some text and then
			// clicking once inside the selection (which collapses the
			// selection inside the previous selection).
			var selection = this;
			this.updateSelectionTimeout = window.setTimeout(function () {
				var range = new Aloha.Selection.SelectionRange(true);
				// We have to work around an IE bug that causes the user
				// selection to be incorrectly set on the body element
				// when the updateSelectionTimeout triggers. The
				// selection corrects itself after waiting a while.
				if (!range.startContainer || 'HTML' === range.startContainer.nodeName || 'BODY' === range.startContainer.nodeName) {
					if (!this.updateSelectionTimeout) {
						// First wait 5 millis, then 20 millis, 50 millis, 110 millis etc.
						selection.onChange(objectClicked, event, 10 + (timeout || 5) * 2);
					}
					return;
				}
				Aloha.Selection._updateSelection(event, range);
			}, timeout || 5);
		},

		/**
		 * prevents the next aloha-selection-changed event from being triggered
		 */
		preventSelectionChanged: function () {
			this.preventSelectionChangedFlag = true;
		},

		/**
		 * will return wheter selection change event was prevented or not, and reset the preventSelectionChangedFlag
		 * @return {Boolean} true if aloha-selection-change event was prevented
		 */
		isSelectionChangedPrevented: function () {
			var prevented = this.preventSelectionChangedFlag;
			this.preventSelectionChangedFlag = false;
			return prevented;
		},

		/**
		 * Checks if the current rangeObject common ancector container is edtiable
		 * @return {Boolean} true if current common ancestor is editable
		 */
		isSelectionEditable: function () {
			return (this.rangeObject.commonAncestorContainer && jQuery(this.rangeObject.commonAncestorContainer).contentEditable());
		},

		/**
		 * This method checks, if the current rangeObject common ancestor container has a 'data-aloha-floatingmenu-visible' Attribute.
		 * Needed in Floating Menu for exceptional display of floatingmenu.
		 */
		isFloatingMenuVisible: function () {
			var visible = jQuery(Aloha.Selection.rangeObject.commonAncestorContainer).attr('data-aloha-floatingmenu-visible');
			if (visible !== 'undefined') {
				if (visible === 'true') {
					return true;
				}
				return false;
			}
			return false;
		},

		/**
		 * INFO: Method is used for integration with Gentics Aloha, has no use otherwise
		 * Updates the rangeObject according to the current user selection
		 * Method is always called on selection change
		 * @param event jQuery browser event object
		 * @return true when rangeObject was modified, false otherwise
		 * @hide
		 */
		updateSelection: function (event) {
			return this._updateSelection(event, null);
		},

		/**
		 * Internal version of updateSelection that adds the range parameter to be
		 * able to work around an IE bug that caused the current user selection
		 * sometimes to be on the body element.
		 * @param {Object} event
		 * @param {Object} range a substitute for the current user selection. if not provided,
		 *   the current user selection will be used.
		 * @hide
		 */
		_updateSelection: function (event, range) {
			if (event && event.originalEvent &&
					true === event.originalEvent.stopSelectionUpdate) {
				return false;
			}

			if (typeof range === 'undefined') {
				return false;
			}

			this.rangeObject = range =
					range || new Aloha.Selection.SelectionRange(true);

			// Determine the common ancestor container and update the selection
			// tree.
			range.update();

			// Workaround for nasty IE bug that allows the user to select
			// text nodes inside areas with contenteditable "false"
			if (range && range.startContainer && range.endContainer) {
				var inEditable =
						jQuery(range.commonAncestorContainer)
							.closest('.aloha-editable').length > 0;

				if (inEditable) {
					var validStartPosition = !(3 === range.startContainer.nodeType &&
							!jQuery(range.startContainer.parentNode).contentEditable());

					var validEndPosition = !(3 === range.endContainer.nodeType &&
							!jQuery(range.endContainer.parentNode).contentEditable());

					if (!validStartPosition || !validEndPosition) {
						Aloha.getSelection().removeAllRanges();
						return true;
					}
				}
			}

			// check if aloha-selection-changed event has been prevented
			if (this.isSelectionChangedPrevented()) {
				return true;
			}

			Aloha.trigger('aloha-selection-changed-before', [this.rangeObject, event]);

			// throw the event that the selection has changed. Plugins now have the
			// chance to react on the currentElements[childCount].children.lengthged selection
			Aloha.trigger('aloha-selection-changed', [this.rangeObject, event]);

			triggerSelectionContextChanged(this.rangeObject, event);

			Aloha.trigger('aloha-selection-changed-after', [this.rangeObject, event]);

			return true;
		},

		/**
		 * creates an object with x items containing all relevant dom objects.
		 * Structure:
		 * +
		 * |-domobj: <reference to the DOM Object> (NOT jQuery)
		 * |-selection: defines if this node is marked by user [none|partial|full]
		 * |-children: recursive structure like this ("x.." because it's then shown last in DOM Browsers...)
		 * TODO: remove this (was moved to range.js)
		 *
		 * @param rangeObject "Aloha clean" range object including a commonAncestorContainer
		 * @return obj selection
		 * @hide
		 */
		getSelectionTree: function (rangeObject) {
			if (!rangeObject) { // if called without any parameters, the method acts as getter for this.selectionTree
				return this.rangeObject.getSelectionTree();
			}
			if (!rangeObject.commonAncestorContainer) {
				Aloha.Log.error(this, 'the rangeObject is missing the commonAncestorContainer');
				return false;
			}

			this.inselection = false;

			// before getting the selection tree, we do a cleanup
			if (GENTICS.Utils.Dom.doCleanup({ 'merge': true }, rangeObject)) {
				rangeObject.update();
				rangeObject.select();
			}

			return this.recursiveGetSelectionTree(rangeObject, rangeObject.commonAncestorContainer);
		},

		/**
		 * Recursive inner function for generating the selection tree.
		 * TODO: remove this (was moved to range.js)
		 * @param rangeObject range object
		 * @param currentObject current DOM object for which the selection tree shall be generated
		 * @return array of SelectionTree objects for the children of the current DOM object
		 * @hide
		 */
		recursiveGetSelectionTree: function (rangeObject, currentObject) {
			// get all direct children of the given object
			var jQueryCurrentObject = jQuery(currentObject),
				childCount = 0,
				that = this,
				currentElements = [];

			jQueryCurrentObject.contents().each(function (index) {
				var selectionType = 'none',
					startOffset = false,
					endOffset = false,
					collapsedFound = false,
					i,
				    elementsLength,
					noneFound = false,
					partialFound = false,
					fullFound = false;

				// check for collapsed selections between nodes
				if (rangeObject.isCollapsed() && currentObject === rangeObject.startContainer && rangeObject.startOffset == index) {
					// insert an extra selectiontree object for the collapsed selection here
					currentElements[childCount] = new Aloha.Selection.SelectionTree();
					currentElements[childCount].selection = 'collapsed';
					currentElements[childCount].domobj = undefined;
					that.inselection = false;
					collapsedFound = true;
					childCount++;
				}

				if (!that.inselection && !collapsedFound) {
					// the start of the selection was not yet found, so look for it now
					// check whether the start of the selection is found here

					// Try to read the nodeType property and return if we do not have permission
					// ie.: frame document to an external URL
					var nodeType;
					try {
						nodeType = this.nodeType;
					} catch (e) {
						return;
					}

					// check is dependent on the node type
					switch (nodeType) {
					case 3:
						// text node
						if (this === rangeObject.startContainer) {
							// the selection starts here
							that.inselection = true;

							// when the startoffset is > 0, the selection type is only partial
							selectionType = rangeObject.startOffset > 0 ? 'partial' : 'full';
							startOffset = rangeObject.startOffset;
							endOffset = this.length;
						}
						break;
					case 1:
						// element node
						if (this === rangeObject.startContainer && rangeObject.startOffset === 0) {
							// the selection starts here
							that.inselection = true;
							selectionType = 'full';
						}
						if (currentObject === rangeObject.startContainer && rangeObject.startOffset === index) {
							// the selection starts here
							that.inselection = true;
							selectionType = 'full';
						}
						break;
					}
				}

				if (that.inselection && !collapsedFound) {
					if (selectionType == 'none') {
						selectionType = 'full';
					}
					// we already found the start of the selection, so look for the end of the selection now
					// check whether the end of the selection is found here

					switch (this.nodeType) {
					case 3:
						// text node
						if (this === rangeObject.endContainer) {
							// the selection ends here
							that.inselection = false;

							// check for partial selection here
							if (rangeObject.endOffset < this.length) {
								selectionType = 'partial';
							}
							if (startOffset === false) {
								startOffset = 0;
							}
							endOffset = rangeObject.endOffset;
						}
						break;
					case 1:
						// element node
						if (this === rangeObject.endContainer && rangeObject.endOffset === 0) {
							that.inselection = false;
						}
						break;
					}
					if (currentObject === rangeObject.endContainer && rangeObject.endOffset <= index) {
						that.inselection = false;
						selectionType = 'none';
					}
				}

				// create the current selection tree entry
				currentElements[childCount] = new Aloha.Selection.SelectionTree();
				currentElements[childCount].domobj = this;
				currentElements[childCount].selection = selectionType;
				if (selectionType == 'partial') {
					currentElements[childCount].startOffset = startOffset;
					currentElements[childCount].endOffset = endOffset;
				}

				// now do the recursion step into the current object
				currentElements[childCount].children = that.recursiveGetSelectionTree(rangeObject, this);
				elementsLength = currentElements[childCount].children.length;

				// check whether a selection was found within the children
				if (elementsLength > 0) {
					for (i = 0; i < elementsLength; ++i) {
						switch (currentElements[childCount].children[i].selection) {
						case 'none':
							noneFound = true;
							break;
						case 'full':
							fullFound = true;
							break;
						case 'partial':
							partialFound = true;
							break;
						}
					}

					if (partialFound || (fullFound && noneFound)) {
						// found at least one 'partial' selection in the children, or both 'full' and 'none', so this element is also 'partial' selected
						currentElements[childCount].selection = 'partial';
					} else if (fullFound && !partialFound && !noneFound) {
						// only found 'full' selected children, so this element is also 'full' selected
						currentElements[childCount].selection = 'full';
					}
				}

				childCount++;
			});

			// extra check for collapsed selections at the end of the current element
			if (rangeObject.isCollapsed() && currentObject === rangeObject.startContainer && rangeObject.startOffset == currentObject.childNodes.length) {
				currentElements[childCount] = new Aloha.Selection.SelectionTree();
				currentElements[childCount].selection = 'collapsed';
				currentElements[childCount].domobj = undefined;
			}

			return currentElements;
		},

		/**
		 * Get the currently selected range
		 * @return {Aloha.Selection.SelectionRange} currently selected range
		 * @method
		 */
		getRangeObject: function () {
			return this.rangeObject;
		},

		/**
		 * method finds out, if a node is within a certain markup or not
		 * @param rangeObj Aloha rangeObject
		 * @param startOrEnd boolean; defines, if start or endContainer should be used: false for start, true for end
		 * @param markupObject jQuery object of the markup to look for
		 * @param tagComparator method, which is used to compare the dom object and the jQuery markup object. the method must accept 2 parameters, the first is the domobj, the second is the jquery object. if no method is specified, the method this.standardTextLevelSemanticsComparator is used
		 * @param limitObject dom object which limits the search are within the dom. normally this will be the active Editable
		 * @return true, if the markup is effective on the range objects start or end node
		 * @hide
		 */
		isRangeObjectWithinMarkup: function (rangeObject, startOrEnd, markupObject, tagComparator, limitObject) {
			var domObj = !startOrEnd ? rangeObject.startContainer : rangeObject.endContainer,
				that = this,
				parents = jQuery(domObj).parents(),
				returnVal = false,
				i = -1;

			// check if a comparison method was passed as parameter ...
			if (typeof tagComparator !== 'undefined' && typeof tagComparator !== 'function') {
				Aloha.Log.error(this, 'parameter tagComparator is not a function');
			}
			// ... if not use this as standard tag comparison method
			if (typeof tagComparator === 'undefined') {
				tagComparator = function (domobj, markupObject) {
					return that.standardTextLevelSemanticsComparator(domobj, markupObject); // TODO should actually be this.getStandardTagComparator(markupObject)
				};
			}

			if (parents.length > 0) {
				parents.each(function () {
					// the limit object was reached (normally the Editable Element)
					if (this === limitObject) {
						Aloha.Log.debug(that, 'reached limit dom obj');
						return false; // break() of jQuery .each(); THIS IS NOT THE FUNCTION RETURN VALUE
					}
					if (tagComparator(this, markupObject)) {
						if (returnVal === false) {
							returnVal = [];
						}
						Aloha.Log.debug(that, 'reached object equal to markup');
						i++;
						returnVal[i] = this;
						return true; // continue() of jQuery .each(); THIS IS NOT THE FUNCTION RETURN VALUE
					}
				});
			}
			return returnVal;
		},

		/**
		 * standard method, to compare a domobj and a jquery object for sections and grouping content (e.g. p, h1, h2, ul, ....).
		 * is always used when no other tag comparator is passed as parameter
		 * @param domobj domobject to compare with markup
		 * @param markupObject jQuery object of the markup to compare with domobj
		 * @return true if objects are equal and false if not
		 * @hide
		 */
		standardSectionsAndGroupingContentComparator: function (domobj, markupObject) {
			if (domobj.nodeType !== 1) {
				Aloha.Log.debug(this, 'only element nodes (nodeType == 1) can be compared');
				return false;
			}
			if (!markupObject[0].nodeName) {
				return false;
			}
			var elemMap = Aloha.Selection.replacingElements[domobj.nodeName.toLowerCase()];
			return elemMap && elemMap[markupObject[0].nodeName.toLowerCase()];
		},

		/**
		 * standard method, to compare a domobj and a jquery object for their tagName (aka span elements, e.g. b, i, sup, span, ...).
		 * is always used when no other tag comparator is passed as parameter
		 * @param domobj domobject to compare with markup
		 * @param markupObject jQuery object of the markup to compare with domobj
		 * @return true if objects are equal and false if not
		 * @hide
		 */
		standardTagNameComparator: function (domobj, markupObject) {
			if (domobj.nodeType === 1) {
				if (domobj.nodeName != markupObject[0].nodeName) {
					return false;
				}
				return true;
			}
			Aloha.Log.debug(this, 'only element nodes (nodeType == 1) can be compared');
			return false;
		},

		/**
		 * standard method, to compare a domobj and a jquery object for text level semantics (aka span elements, e.g. b, i, sup, span, ...).
		 * is always used when no other tag comparator is passed as parameter
		 * @param domobj domobject to compare with markup
		 * @param markupObject jQuery object of the markup to compare with domobj
		 * @return true if objects are equal and false if not
		 * @hide
		 */
		standardTextLevelSemanticsComparator: function (domobj, markupObject) {
			// only element nodes can be compared
			if (domobj.nodeType === 1) {
				if (domobj.nodeName != markupObject[0].nodeName) {
					return false;
				}
				if (!this.standardAttributesComparator(domobj, markupObject)) {
					return false;
				}
				return true;
			}
			Aloha.Log.debug(this, 'only element nodes (nodeType == 1) can be compared');
			return false;
		},


		/**
		 * standard method, to compare attributes of one dom obj and one markup obj (jQuery)
		 * @param domobj domobject to compare with markup
		 * @param markupObject jQuery object of the markup to compare with domobj
		 * @return true if objects are equal and false if not
		 * @hide
		 */
		standardAttributesComparator: function (domobj, markupObject) {
			var classesA = Strings.words((domobj && domobj.className) || '');
			var classesB = Strings.words((markupObject.length && markupObject[0].className) || '');
			Arrays.sortUnique(classesA);
			Arrays.sortUnique(classesB);
			return Arrays.equal(classesA, classesB);
		},

		/**
		 * method finds out, if a node is within a certain markup or not
		 * @param rangeObj Aloha rangeObject
		 * @param markupObject jQuery object of the markup to be applied (e.g. created with obj = jQuery('<b></b>'); )
		 * @param tagComparator method, which is used to compare the dom object and the jQuery markup object. the method must accept 2 parameters, the first is the domobj, the second is the jquery object. if no method is specified, the method this.standardTextLevelSemanticsComparator is used
		 * @return void; TODO: should return true if the markup applied successfully and false if not
		 * @hide
		 */
		changeMarkup: function (rangeObject, markupObject, tagComparator) {
			var tagName = markupObject[0].tagName.toLowerCase(),
				newCAC,
			    limitObject,
				backupRangeObject,
				relevantMarkupObjectsAtSelectionStart = this.isRangeObjectWithinMarkup(rangeObject, false, markupObject, tagComparator, limitObject),
				relevantMarkupObjectsAtSelectionEnd = this.isRangeObjectWithinMarkup(rangeObject, true, markupObject, tagComparator, limitObject),
				nextSibling,
			    relevantMarkupObjectAfterSelection,
				prevSibling,
			    relevantMarkupObjectBeforeSelection,
				extendedRangeObject;
			var parentElement;

			// if the element is a replacing element (like p/h1/h2/h3/h4/h5/h6...), which must not wrap each other
			// use a clone of rangeObject
			if (this.replacingElements[tagName]) {
				// backup rangeObject for later selection;
				backupRangeObject = rangeObject;

				// create a new range object to not modify the orginal
				rangeObject = new this.SelectionRange(rangeObject);

				// either select the active Editable as new commonAncestorContainer (CAC) or use the body
				if (Aloha.activeEditable) {
					newCAC = Aloha.activeEditable.obj.get(0);
				} else {
					newCAC = jQuery('body');
				}
				// update rangeObject by setting the newCAC and automatically recalculating the selectionTree
				rangeObject.update(newCAC);

				// store the information, that the markupObject can be replaced (not must be!!) inside the jQuery markup object
				markupObject.isReplacingElement = true;
			} else {
				// if the element is NOT a replacing element, then something needs to be selected, otherwise it can not be wrapped
				// therefor the method can return false, if nothing is selected ( = rangeObject is collapsed)
				if (rangeObject.isCollapsed()) {
					Aloha.Log.debug(this, 'early returning from applying markup because nothing is currently selected');
					return false;
				}
			}

			// is Start/End DOM Obj inside the markup to change
			if (Aloha.activeEditable) {
				limitObject = Aloha.activeEditable.obj[0];
			} else {
				limitObject = jQuery('body');
			}

			if (!markupObject.isReplacingElement && rangeObject.startOffset === 0) { // don't care about replacers, because they never extend
				if (null != (prevSibling = this.getTextNodeSibling(false, rangeObject.commonAncestorContainer.parentNode, rangeObject.startContainer))) {
					relevantMarkupObjectBeforeSelection = this.isRangeObjectWithinMarkup({
						startContainer: prevSibling,
						startOffset: 0
					}, false, markupObject, tagComparator, limitObject);
				}
			}
			if (!markupObject.isReplacingElement && (rangeObject.endOffset === rangeObject.endContainer.length)) { // don't care about replacers, because they never extend
				if (null != (nextSibling = this.getTextNodeSibling(true, rangeObject.commonAncestorContainer.parentNode, rangeObject.endContainer))) {
					relevantMarkupObjectAfterSelection = this.isRangeObjectWithinMarkup({
						startContainer: nextSibling,
						startOffset: 0
					}, false, markupObject, tagComparator, limitObject);
				}
			}

			// decide what to do (expand or reduce markup)
			// Alternative A: from markup to no-markup: markup will be removed in selection;
			// reapplied from original markup start to selection start
			if (!markupObject.isReplacingElement && (relevantMarkupObjectsAtSelectionStart && !relevantMarkupObjectsAtSelectionEnd)) {
				Aloha.Log.info(this, 'markup 2 non-markup');
				this.prepareForRemoval(rangeObject.getSelectionTree(), markupObject, tagComparator);
				jQuery(relevantMarkupObjectsAtSelectionStart).addClass('preparedForRemoval');
				this.insertCroppedMarkups(relevantMarkupObjectsAtSelectionStart, rangeObject, false, tagComparator);
			} else if (!markupObject.isReplacingElement && relevantMarkupObjectsAtSelectionStart && relevantMarkupObjectsAtSelectionEnd) {
				// Alternative B: from markup to markup:
				// remove selected markup (=split existing markup if single, shrink if two different)
				Aloha.Log.info(this, 'markup 2 markup');
				this.prepareForRemoval(rangeObject.getSelectionTree(), markupObject, tagComparator);
				this.splitRelevantMarkupObject(relevantMarkupObjectsAtSelectionStart, relevantMarkupObjectsAtSelectionEnd, rangeObject, tagComparator);
			} else if (!markupObject.isReplacingElement && ((!relevantMarkupObjectsAtSelectionStart && relevantMarkupObjectsAtSelectionEnd) || relevantMarkupObjectAfterSelection || relevantMarkupObjectBeforeSelection)) { //
				// Alternative C: from no-markup to markup OR with next2markup:
				// new markup is wrapped from selection start to end of originalmarkup, original is remove afterwards
				Aloha.Log.info(this, 'non-markup 2 markup OR with next2markup');
				// move end of rangeObject to end of relevant markups
				if (relevantMarkupObjectBeforeSelection && relevantMarkupObjectAfterSelection) {
					extendedRangeObject = new Aloha.Selection.SelectionRange(rangeObject);
					extendedRangeObject.startContainer = jQuery(relevantMarkupObjectBeforeSelection[relevantMarkupObjectBeforeSelection.length - 1]).textNodes()[0];
					extendedRangeObject.startOffset = 0;
					extendedRangeObject.endContainer = jQuery(relevantMarkupObjectAfterSelection[relevantMarkupObjectAfterSelection.length - 1]).textNodes().last()[0];
					extendedRangeObject.endOffset = extendedRangeObject.endContainer.length;
					extendedRangeObject.update();
					this.applyMarkup(extendedRangeObject.getSelectionTree(), rangeObject, markupObject, tagComparator);
					Aloha.Log.info(this, 'double extending previous markup(previous and after selection), actually wrapping it ...');

				} else if (relevantMarkupObjectBeforeSelection && !relevantMarkupObjectAfterSelection && !relevantMarkupObjectsAtSelectionEnd) {
					this.extendExistingMarkupWithSelection(relevantMarkupObjectBeforeSelection, rangeObject, false, tagComparator);
					Aloha.Log.info(this, 'extending previous markup');

				} else if (relevantMarkupObjectBeforeSelection && !relevantMarkupObjectAfterSelection && relevantMarkupObjectsAtSelectionEnd) {
					extendedRangeObject = new Aloha.Selection.SelectionRange(rangeObject);
					extendedRangeObject.startContainer = jQuery(relevantMarkupObjectBeforeSelection[relevantMarkupObjectBeforeSelection.length - 1]).textNodes()[0];
					extendedRangeObject.startOffset = 0;
					extendedRangeObject.endContainer = jQuery(relevantMarkupObjectsAtSelectionEnd[relevantMarkupObjectsAtSelectionEnd.length - 1]).textNodes().last()[0];
					extendedRangeObject.endOffset = extendedRangeObject.endContainer.length;
					extendedRangeObject.update();
					this.applyMarkup(extendedRangeObject.getSelectionTree(), rangeObject, markupObject, tagComparator);
					Aloha.Log.info(this, 'double extending previous markup(previous and relevant at the end), actually wrapping it ...');

				} else if (!relevantMarkupObjectBeforeSelection && relevantMarkupObjectAfterSelection) {
					this.extendExistingMarkupWithSelection(relevantMarkupObjectAfterSelection, rangeObject, true, tagComparator);
					Aloha.Log.info(this, 'extending following markup backwards');

				} else {
					this.extendExistingMarkupWithSelection(relevantMarkupObjectsAtSelectionEnd, rangeObject, true, tagComparator);
				}
			} else if (markupObject.isReplacingElement || (!relevantMarkupObjectsAtSelectionStart && !relevantMarkupObjectsAtSelectionEnd && !relevantMarkupObjectBeforeSelection && !relevantMarkupObjectAfterSelection)) {
				// Alternative D: no-markup to no-markup: easy
				Aloha.Log.info(this, 'non-markup 2 non-markup');

				// workaround to keep the caret at the right position if it's an empty element
				// applyMarkup was not working correctly and has a lot of overhead we don't need in that case
				if (isCollapsedAndEmptyOrEndBr(rangeObject)) {
					var newMarkup = markupObject.clone();

					if (isCollapsedAndEndBr(rangeObject)) {
						newMarkup[0].appendChild(Engine.createEndBreak());
					}

					// setting the focus is needed for mozilla and IE 7 to have a working rangeObject.select()
					if (Aloha.activeEditable && jQuery.browser.mozilla) {
						Aloha.activeEditable.obj.focus();
					}

					if (Engine.isEditable(rangeObject.startContainer)) {
						Engine.copyAttributes(rangeObject.startContainer, newMarkup[0]);
						jQuery(rangeObject.startContainer).after(newMarkup[0]).remove();
					} else if (Engine.isEditingHost(rangeObject.startContainer)) {
						jQuery(rangeObject.startContainer).append(newMarkup[0]);
						Engine.ensureContainerEditable(newMarkup[0]);
					}

					backupRangeObject.startContainer = newMarkup[0];
					backupRangeObject.endContainer = newMarkup[0];
					backupRangeObject.startOffset = 0;
					backupRangeObject.endOffset = 0;
					return;
				}
				this.applyMarkup(rangeObject.getSelectionTree(), rangeObject, markupObject, tagComparator, {
					setRangeObject2NewMarkup: true
				});
				backupRangeObject.startContainer = rangeObject.startContainer;
				backupRangeObject.endContainer = rangeObject.endContainer;
				backupRangeObject.startOffset = rangeObject.startOffset;
				backupRangeObject.endOffset = rangeObject.endOffset;
			}

			if (markupObject.isReplacingElement) {
				//Check if the startContainer is one of the zapped elements
				if (backupRangeObject && backupRangeObject.startContainer.className && backupRangeObject.startContainer.className.indexOf('preparedForRemoval') > -1) {
					//var parentElement = jQuery(backupRangeObject.startContainer).closest(markupObject[0].tagName).get(0);
					parentElement = jQuery(backupRangeObject.startContainer).parents(markupObject[0].tagName).get(0);
					backupRangeObject.startContainer = parentElement;
					rangeObject.startContainer = parentElement;
				}
				//check if the endContainer is one of the zapped elements
				if (backupRangeObject && backupRangeObject.endContainer.className && backupRangeObject.endContainer.className.indexOf('preparedForRemoval') > -1) {
					//var parentElement = jQuery(backupRangeObject.endContainer).closest(markupObject[0].tagName).get(0);
					parentElement = jQuery(backupRangeObject.endContainer).parents(markupObject[0].tagName).get(0);
					backupRangeObject.endContainer = parentElement;
					rangeObject.endContainer = parentElement;
				}
			}
			// remove all marked items
			jQuery('.preparedForRemoval').zap();

			// recalculate cac and selectionTree

			// update selection
			if (markupObject.isReplacingElement) {
				//After the zapping we have to check for wrong offsets
				if (e5s.Node.ELEMENT_NODE === backupRangeObject.startContainer.nodeType && backupRangeObject.startContainer.childNodes && backupRangeObject.startContainer.childNodes.length < backupRangeObject.startOffset) {
					backupRangeObject.startOffset = backupRangeObject.startContainer.childNodes.length;
					rangeObject.startOffset = backupRangeObject.startContainer.childNodes.length;
				}
				if (e5s.Node.ELEMENT_NODE === backupRangeObject.endContainer.nodeType && backupRangeObject.endContainer.childNodes && backupRangeObject.endContainer.childNodes.length < backupRangeObject.endOffset) {
					backupRangeObject.endOffset = backupRangeObject.endContainer.childNodes.length;
					rangeObject.endOffset = backupRangeObject.endContainer.childNodes.length;
				}
				rangeObject.endContainer = backupRangeObject.endContainer;
				rangeObject.endOffset = backupRangeObject.endOffset;
				rangeObject.startContainer = backupRangeObject.startContainer;
				rangeObject.startOffset = backupRangeObject.startOffset;
				backupRangeObject.update();
				backupRangeObject.select();
			} else {
				rangeObject.update();
				rangeObject.select();
			}
		},

		/**
		 * method compares a JS array of domobjects with a range object and decides, if the rangeObject spans the whole markup objects. method is used to decide if a markup2markup selection can be completely remove or if it must be splitted into 2 separate markups
		 * @param relevantMarkupObjectsAtSelectionStart JS Array of dom objects, which are parents to the rangeObject.startContainer
		 * @param relevantMarkupObjectsAtSelectionEnd JS Array of dom objects, which are parents to the rangeObject.endContainer
		 * @param rangeObj Aloha rangeObject
		 * @return true, if rangeObjects and markup objects are identical, false otherwise
		 * @hide
		 */
		areMarkupObjectsAsLongAsRangeObject: function (relevantMarkupObjectsAtSelectionStart, relevantMarkupObjectsAtSelectionEnd, rangeObject) {
			var i, el, textNode, relMarkupEnd, relMarkupStart;

			if (rangeObject.startOffset !== 0) {
				return false;
			}

			for (i = 0, relMarkupStart = relevantMarkupObjectsAtSelectionStart.length; i < relMarkupStart; i++) {
				el = jQuery(relevantMarkupObjectsAtSelectionStart[i]);
				if (el.textNodes().first()[0] !== rangeObject.startContainer) {
					return false;
				}
			}

			for (i = 0, relMarkupEnd = relevantMarkupObjectsAtSelectionEnd.length; i < relMarkupEnd; i++) {
				el = jQuery(relevantMarkupObjectsAtSelectionEnd[i]);
				textNode = el.textNodes().last()[0];
				if (textNode !== rangeObject.endContainer || textNode.length != rangeObject.endOffset) {
					return false;
				}
			}

			return true;
		},

		/**
		 * method used to remove/split markup from a "markup2markup" selection
		 * @param relevantMarkupObjectsAtSelectionStart JS Array of dom objects, which are parents to the rangeObject.startContainer
		 * @param relevantMarkupObjectsAtSelectionEnd JS Array of dom objects, which are parents to the rangeObject.endContainer
		 * @param rangeObj Aloha rangeObject
		 * @param tagComparator method, which is used to compare the dom object and the jQuery markup object. the method must accept 2 parameters, the first is the domobj, the second is the jquery object. if no method is specified, the method this.standardTextLevelSemanticsComparator is used
		 * @return true (always, since no "false" case is currently known...but might be added)
		 * @hide
		 */
		splitRelevantMarkupObject: function (relevantMarkupObjectsAtSelectionStart, relevantMarkupObjectsAtSelectionEnd, rangeObject, tagComparator) {
			// mark them to be deleted
			jQuery(relevantMarkupObjectsAtSelectionStart).addClass('preparedForRemoval');
			jQuery(relevantMarkupObjectsAtSelectionEnd).addClass('preparedForRemoval');

			// check if the rangeObject is identical with the relevantMarkupObjects (in this case the markup can simply be removed)
			if (this.areMarkupObjectsAsLongAsRangeObject(relevantMarkupObjectsAtSelectionStart, relevantMarkupObjectsAtSelectionEnd, rangeObject)) {
				return true;
			}

			// find intersection (this can always only be one dom element (namely the highest) because all others will be removed
			var relevantMarkupObjectAtSelectionStartAndEnd = this.intersectRelevantMarkupObjects(relevantMarkupObjectsAtSelectionStart, relevantMarkupObjectsAtSelectionEnd);

			if (relevantMarkupObjectAtSelectionStartAndEnd) {
				this.insertCroppedMarkups([relevantMarkupObjectAtSelectionStartAndEnd], rangeObject, false, tagComparator);
				this.insertCroppedMarkups([relevantMarkupObjectAtSelectionStartAndEnd], rangeObject, true, tagComparator);
			} else {
				this.insertCroppedMarkups(relevantMarkupObjectsAtSelectionStart, rangeObject, false, tagComparator);
				this.insertCroppedMarkups(relevantMarkupObjectsAtSelectionEnd, rangeObject, true, tagComparator);
			}
			return true;
		},

		/**
		 * method takes two arrays of bottom up dom objects, compares them and returns either the object closest to the root or false
		 * @param relevantMarkupObjectsAtSelectionStart JS Array of dom objects
		 * @param relevantMarkupObjectsAtSelectionEnd JS Array of dom objects
		 * @return dom object closest to the root or false
		 * @hide
		 */
		intersectRelevantMarkupObjects: function (relevantMarkupObjectsAtSelectionStart, relevantMarkupObjectsAtSelectionEnd) {
			var intersection = false, i, elStart, j, elEnd, relMarkupStart, relMarkupEnd;
			if (!relevantMarkupObjectsAtSelectionStart || !relevantMarkupObjectsAtSelectionEnd) {
				return intersection; // we can only intersect, if we have to arrays!
			}
			relMarkupStart = relevantMarkupObjectsAtSelectionStart.length;
			relMarkupEnd = relevantMarkupObjectsAtSelectionEnd.length;
			for (i = 0; i < relMarkupStart; i++) {
				elStart = relevantMarkupObjectsAtSelectionStart[i];
				for (j = 0; j < relMarkupEnd; j++) {
					elEnd = relevantMarkupObjectsAtSelectionEnd[j];
					if (elStart === elEnd) {
						intersection = elStart;
					}
				}
			}
			return intersection;
		},

		/**
		 * method used to add markup to a nonmarkup2markup selection
		 * @param relevantMarkupObjects JS Array of dom objects effecting either the start or endContainer of a selection (which should be extended)
		 * @param rangeObject Aloha rangeObject the markups should be extended to
		 * @param startOrEnd boolean; defines, if the existing markups should be extended forwards or backwards (is propably redundant and could be found out by comparing start or end container with the markup array dom objects)
		 * @param tagComparator method, which is used to compare the dom object and the jQuery markup object. the method must accept 2 parameters, the first is the domobj, the second is the jquery object. if no method is specified, the method this.standardTextLevelSemanticsComparator is used
		 * @return true
		 * @hide
		 */
		extendExistingMarkupWithSelection: function (relevantMarkupObjects, rangeObject, startOrEnd, tagComparator) {
			var extendMarkupsAtStart, extendMarkupsAtEnd, objects, i, relMarkupLength, el, textnodes, nodeNr;
			if (!startOrEnd) { // = Start
				// start part of rangeObject should be used, therefor existing markups are cropped at the end
				extendMarkupsAtStart = true;
			}
			if (startOrEnd) { // = End
				// end part of rangeObject should be used, therefor existing markups are cropped at start (beginning)
				extendMarkupsAtEnd = true;
			}
			objects = [];
			for (i = 0, relMarkupLength = relevantMarkupObjects.length; i < relMarkupLength; i++) {
				objects[i] = new this.SelectionRange();
				el = relevantMarkupObjects[i];
				if (extendMarkupsAtEnd && !extendMarkupsAtStart) {
					objects[i].startContainer = rangeObject.startContainer; // jQuery(el).contents()[0];
					objects[i].startOffset = rangeObject.startOffset;
					textnodes = jQuery(el).textNodes(true);

					nodeNr = textnodes.length - 1;
					objects[i].endContainer = textnodes[nodeNr];
					objects[i].endOffset = textnodes[nodeNr].length;
					objects[i].update();
					this.applyMarkup(objects[i].getSelectionTree(), rangeObject, this.getClonedMarkup4Wrapping(el), tagComparator, {
						setRangeObject2NewMarkup: true
					});
				}
				if (!extendMarkupsAtEnd && extendMarkupsAtStart) {
					textnodes = jQuery(el).textNodes(true);
					objects[i].startContainer = textnodes[0]; // jQuery(el).contents()[0];
					objects[i].startOffset = 0;
					objects[i].endContainer = rangeObject.endContainer;
					objects[i].endOffset = rangeObject.endOffset;
					objects[i].update();
					this.applyMarkup(objects[i].getSelectionTree(), rangeObject, this.getClonedMarkup4Wrapping(el), tagComparator, {
						setRangeObject2NewMarkup: true
					});
				}
			}
			return true;
		},

		/**
		 * method creates an empty markup jQuery object from a dom object passed as paramter
		 * @param domobj domobject to be cloned, cleaned and emptied
		 * @param tagComparator method, which is used to compare the dom object and the jQuery markup object. the method must accept 2 parameters, the first is the domobj, the second is the jquery object. if no method is specified, the method this.standardTextLevelSemanticsComparator is used
		 * @return jQuery wrapper object to be passed to e.g. this.applyMarkup(...)
		 * @hide
		 */
		getClonedMarkup4Wrapping: function (domobj) {
			var wrapper = jQuery(domobj.outerHTML).removeClass('preparedForRemoval').empty();
			if (wrapper.attr('class').length === 0) {
				wrapper.removeAttr('class');
			}
			return wrapper;
		},

		/**
		 * method used to subtract the range object from existing markup. in other words: certain markup is removed from the selections defined by the rangeObject
		 * @param relevantMarkupObjects JS Array of dom objects effecting either the start or endContainer of a selection (which should be extended)
		 * @param rangeObject Aloha rangeObject the markups should be removed from
		 * @param startOrEnd boolean; defines, if the existing markups should be reduced at the beginning of the tag or at the end (is propably redundant and could be found out by comparing start or end container with the markup array dom objects)
		 * @param tagComparator method, which is used to compare the dom object and the jQuery markup object. the method must accept 2 parameters, the first is the domobj, the second is the jquery object. if no method is specified, the method this.standardTextLevelSemanticsComparator is used
		 * @return true
		 * @hide
		 */
		insertCroppedMarkups: function (relevantMarkupObjects, rangeObject, startOrEnd, tagComparator) {
			var cropMarkupsAtEnd, cropMarkupsAtStart, textnodes, objects, i, el, textNodes;
			if (!startOrEnd) { // = Start
				// start part of rangeObject should be used, therefor existing markups are cropped at the end
				cropMarkupsAtEnd = true;
			} else { // = End
				// end part of rangeObject should be used, therefor existing markups are cropped at start (beginning)
				cropMarkupsAtStart = true;
			}
			objects = [];
			for (i = 0; i < relevantMarkupObjects.length; i++) {
				objects[i] = new this.SelectionRange();
				el = relevantMarkupObjects[i];
				if (cropMarkupsAtEnd && !cropMarkupsAtStart) {
					textNodes = jQuery(el).textNodes(true);
					objects[i].startContainer = textNodes[0];
					objects[i].startOffset = 0;
					// if the existing markup startContainer & startOffset are equal to the rangeObject startContainer and startOffset,
					// then markupobject does not have to be added again, because it would have no content (zero-length)
					if (objects[i].startContainer === rangeObject.startContainer && objects[i].startOffset === rangeObject.startOffset) {
						continue;
					}
					if (rangeObject.startOffset === 0) {
						objects[i].endContainer = this.getTextNodeSibling(false, el, rangeObject.startContainer);
						objects[i].endOffset = objects[i].endContainer.length;
					} else {
						objects[i].endContainer = rangeObject.startContainer;
						objects[i].endOffset = rangeObject.startOffset;
					}

					objects[i].update();

					this.applyMarkup(objects[i].getSelectionTree(), rangeObject, this.getClonedMarkup4Wrapping(el), tagComparator, {
						setRangeObject2NextSibling: true
					});
				}

				if (!cropMarkupsAtEnd && cropMarkupsAtStart) {
					objects[i].startContainer = rangeObject.endContainer; // jQuery(el).contents()[0];
					objects[i].startOffset = rangeObject.endOffset;
					textnodes = jQuery(el).textNodes(true);
					objects[i].endContainer = textnodes[textnodes.length - 1];
					objects[i].endOffset = textnodes[textnodes.length - 1].length;
					objects[i].update();
					this.applyMarkup(objects[i].getSelectionTree(), rangeObject, this.getClonedMarkup4Wrapping(el), tagComparator, {
						setRangeObject2PreviousSibling: true
					});
				}
			}
			return true;
		},

		/**
		 * apply a certain markup to the current selection
		 * @param markupObject jQuery object of the markup to be applied (e.g. created with obj = jQuery('<b></b>'); )
		 * @return void
		 * @hide
		 */
		changeMarkupOnSelection: function (markupObject) {
			var rangeObject = this.getRangeObject();

			// change the markup
			this.changeMarkup(rangeObject, markupObject, this.getStandardTagComparator(markupObject));

			// merge text nodes
			GENTICS.Utils.Dom.doCleanup({
				'merge': true
			}, rangeObject);

			// update the range and select it
			rangeObject.update();
			rangeObject.select();
			this.rangeObject = rangeObject;
		},

		/**
		 * apply a certain markup to the selection Tree
		 * @param selectionTree SelectionTree Object markup should be applied to
		 * @param rangeObject Aloha rangeObject which will be modified to reflect the dom changes, after the markup was applied (only if activated via options)
		 * @param markupObject jQuery object of the markup to be applied (e.g. created with obj = jQuery('<b></b>'); )
		 * @param tagComparator method, which is used to compare the dom object and the jQuery markup object. the method must accept 2 parameters, the first is the domobj, the second is the jquery object. if no method is specified, the method this.standardTextLevelSemanticsComparator is used
		 * @param options JS object, with the following boolean properties: setRangeObject2NewMarkup, setRangeObject2NextSibling, setRangeObject2PreviousSibling
		 * @return void
		 * @hide
		 */
		applyMarkup: function (selectionTree, rangeObject, markupObject, tagComparator, options) {
			var optimizedSelectionTree, i, el, breakpoint;
			options = options || {};
			// first same tags from within fully selected nodes for removal
			this.prepareForRemoval(selectionTree, markupObject, tagComparator);

			// first let's optimize the selection Tree in useful groups which can be wrapped together
			optimizedSelectionTree = this.optimizeSelectionTree4Markup(selectionTree, markupObject, tagComparator);
			breakpoint = true;

			// now iterate over grouped elements and either recursively dive into object or wrap it as a whole
			for (i = 0; i < optimizedSelectionTree.length; i++) {
				el = optimizedSelectionTree[i];
				if (el.wrappable) {
					this.wrapMarkupAroundSelectionTree(el.elements, rangeObject, markupObject, tagComparator, options);
				} else {
					Aloha.Log.debug(this, 'dive further into non-wrappable object');
					this.applyMarkup(el.element.children, rangeObject, markupObject, tagComparator, options);
				}
			}
		},

		/**
		 * returns the type of the given markup (trying to match HTML5)
		 * @param markupObject jQuery object of the markup to be applied (e.g. created with obj = jQuery('<b></b>'); )
		 * @return string name of the markup type
		 * @hide
		 */
		getMarkupType: function (markupObject) {
			var nn = jQuery(markupObject)[0].nodeName.toLowerCase();
			if (markupObject.outerHtml) {
				Aloha.Log.debug(this, 'Node name detected: ' + nn + ' for: ' + markupObject.outerHtml());
			}
			if (nn == '#text') {
				return 'textNode';
			}
			if (this.replacingElements[nn]) {
				return 'sectionOrGroupingContent';
			}
			if (this.tagHierarchy[nn]) {
				return 'textLevelSemantics';
			}
			Aloha.Log.warn(this, 'unknown markup passed to this.getMarkupType(...): ' + markupObject.outerHtml());
		},

		/**
		 * returns the standard tag comparator for the given markup object
		 * @param markupObject jQuery object of the markup to be applied (e.g. created with obj = jQuery('<b></b>'); )
		 * @return function tagComparator method, which is used to compare the dom object and the jQuery markup object. the method must accept 2 parameters, the first is the domobj, the second is the jquery object. if no method is specified, the method this.standardTextLevelSemanticsComparator is used
		 * @hide
		 */
		getStandardTagComparator: function (markupObject) {
			var that = this,
				result;
			switch (this.getMarkupType(markupObject)) {
			case 'textNode':
				result = function (p1, p2) {
					return false;
				};
				break;

			case 'sectionOrGroupingContent':
				result = function (domobj, markupObject) {
					return that.standardSectionsAndGroupingContentComparator(domobj, markupObject);
				};
				break;

			//case 'textLevelSemantics' covered by default
			default:
				result = function (domobj, markupObject) {
					return that.standardTextLevelSemanticsComparator(domobj, markupObject);
				};
				break;
			}
			return result;
		},

		/**
		 * searches for fully selected equal markup tags
		 * @param selectionTree SelectionTree Object markup should be applied to
		 * @param markupObject jQuery object of the markup to be applied (e.g. created with obj = jQuery('<b></b>'); )
		 * @param tagComparator method, which is used to compare the dom object and the jQuery markup object. the method must accept 2 parameters, the first is the domobj, the second is the jquery object. if no method is specified, the method this.standardTextLevelSemanticsComparator is used
		 * @return void
		 * @hide
		 */
		prepareForRemoval: function (selectionTree, markupObject, tagComparator) {
			var that = this, i, el;

			// check if a comparison method was passed as parameter ...
			if (typeof tagComparator !== 'undefined' && typeof tagComparator !== 'function') {
				Aloha.Log.error(this, 'parameter tagComparator is not a function');
			}
			// ... if not use this as standard tag comparison method
			if (typeof tagComparator === 'undefined') {
				tagComparator = this.getStandardTagComparator(markupObject);
			}
			for (i = 0; i < selectionTree.length; i++) {
				el = selectionTree[i];
				if (el.domobj && (el.selection == 'full' || (el.selection == 'partial' && markupObject.isReplacingElement))) {
					// mark for removal
					if (el.domobj.nodeType === 1 && tagComparator(el.domobj, markupObject)) {
						Aloha.Log.debug(this, 'Marking for removal: ' + el.domobj.nodeName);
						jQuery(el.domobj).addClass('preparedForRemoval');
					}
				}
				if (el.selection != 'none' && el.children.length > 0) {
					this.prepareForRemoval(el.children, markupObject, tagComparator);
				}

			}
		},

		/**
		 * searches for fully selected equal markup tags
		 * @param selectionTree SelectionTree Object markup should be applied to
		 * @param rangeObject Aloha rangeObject the markup will be applied to
		 * @param markupObject jQuery object of the markup to be applied (e.g. created with obj = jQuery('<b></b>'); )
		 * @param tagComparator method, which is used to compare the dom object and the jQuery markup object. the method must accept 2 parameters, the first is the domobj, the second is the jquery object. if no method is specified, the method this.standardTextLevelSemanticsComparator is used
		 * @param options JS object, with the following boolean properties: setRangeObject2NewMarkup, setRangeObject2NextSibling, setRangeObject2PreviousSibling
		 * @return void
		 * @hide
		 */
		wrapMarkupAroundSelectionTree: function (selectionTree, rangeObject, markupObject, tagComparator, options) {
			// first let's find out if theoretically the whole selection can be wrapped with one tag and save it for later use
			var objects2wrap = [], // // this will be used later to collect objects
				j = -1, // internal counter,
				breakpoint = true,
				preText = '',
				postText = '',
				prevOrNext,
				textNode2Start,
				textnodes,
				newMarkup,
				i,
			    el,
			    middleText;

			Aloha.Log.debug(this, 'The formatting <' + markupObject[0].tagName + '> will be wrapped around the selection');

			// now lets iterate over the elements
			for (i = 0; i < selectionTree.length; i++) {
				el = selectionTree[i];

				// check if markup is allowed inside the elements parent
				if (el.domobj && !this.canTag1WrapTag2(el.domobj.parentNode.tagName.toLowerCase(), markupObject[0].tagName.toLowerCase())) {
					Aloha.Log.info(this, 'Skipping the wrapping of <' + markupObject[0].tagName.toLowerCase() + '> because this tag is not allowed inside <' + el.domobj.parentNode.tagName.toLowerCase() + '>');
					continue;
				}

				// skip empty text nodes
				if (el.domobj && el.domobj.nodeType === 3 && jQuery.trim(el.domobj.nodeValue).length === 0) {
					continue;
				}

				// partial element, can either be a textnode and therefore be wrapped (at least partially)
				// or can be a nodeType == 1 (tag) which must be dived into
				if (el.domobj && el.selection == 'partial' && !markupObject.isReplacingElement) {
					if (el.startOffset !== undefined && el.endOffset === undefined) {
						j++;
						preText += el.domobj.data.substr(0, el.startOffset);
						el.domobj.data = el.domobj.data.substr(el.startOffset, el.domobj.data.length - el.startOffset);
						objects2wrap[j] = el.domobj;
					} else if (el.endOffset !== undefined && el.startOffset === undefined) {
						j++;
						postText += el.domobj.data.substr(el.endOffset, el.domobj.data.length - el.endOffset);
						el.domobj.data = el.domobj.data.substr(0, el.endOffset);
						objects2wrap[j] = el.domobj;
					} else if (el.endOffset !== undefined && el.startOffset !== undefined) {
						if (el.startOffset == el.endOffset) { // do not wrap empty selections
							Aloha.Log.debug(this, 'skipping empty selection');
							continue;
						}
						j++;
						preText += el.domobj.data.substr(0, el.startOffset);
						middleText = el.domobj.data.substr(el.startOffset, el.endOffset - el.startOffset);
						postText += el.domobj.data.substr(el.endOffset, el.domobj.data.length - el.endOffset);
						el.domobj.data = middleText;
						objects2wrap[j] = el.domobj;
					} else {
						// a partially selected item without selectionStart/EndOffset is a nodeType 1 Element on the way to the textnode
						Aloha.Log.debug(this, 'diving into object');
						this.applyMarkup(el.children, rangeObject, markupObject, tagComparator, options);
					}
				}
				// fully selected dom elements can be wrapped as whole element
				if (el.domobj && (el.selection == 'full' || (el.selection == 'partial' && markupObject.isReplacingElement))) {
					j++;
					objects2wrap[j] = el.domobj;
				}
			}

			if (objects2wrap.length > 0) {
				// wrap collected DOM object with markupObject
				objects2wrap = jQuery(objects2wrap);

				// make a fix for text nodes in <li>'s in ie
				jQuery.each(objects2wrap, function (index, element) {
					if (jQuery.browser.msie && element.nodeType == 3 && !element.nextSibling && !element.previousSibling && element.parentNode && element.parentNode.nodeName.toLowerCase() == 'li') {
						element.data = jQuery.trim(element.data);
					}
				});

				newMarkup = objects2wrap.wrapAll(markupObject).parent();
				newMarkup.before(preText).after(postText);

				if (options.setRangeObject2NewMarkup) { // this is used, when markup is added to normal/normal Text
					textnodes = objects2wrap.textNodes();

					if (textnodes.index(rangeObject.startContainer) != -1) {
						rangeObject.startOffset = 0;
					}
					if (textnodes.index(rangeObject.endContainer) != -1) {
						rangeObject.endOffset = rangeObject.endContainer.length;
					}
					breakpoint = true;
				}
				if (options.setRangeObject2NextSibling) {
					prevOrNext = true;
					textNode2Start = newMarkup.textNodes(true).last()[0];
					if (objects2wrap.index(rangeObject.startContainer) != -1) {
						rangeObject.startContainer = this.getTextNodeSibling(prevOrNext, newMarkup.parent(), textNode2Start);
						rangeObject.startOffset = 0;
					}
					if (objects2wrap.index(rangeObject.endContainer) != -1) {
						rangeObject.endContainer = this.getTextNodeSibling(prevOrNext, newMarkup.parent(), textNode2Start);
						rangeObject.endOffset = rangeObject.endOffset - textNode2Start.length;
					}
				}
				if (options.setRangeObject2PreviousSibling) {
					prevOrNext = false;
					textNode2Start = newMarkup.textNodes(true).first()[0];
					if (objects2wrap.index(rangeObject.startContainer) != -1) {
						rangeObject.startContainer = this.getTextNodeSibling(prevOrNext, newMarkup.parent(), textNode2Start);
						rangeObject.startOffset = 0;
					}
					if (objects2wrap.index(rangeObject.endContainer) != -1) {
						rangeObject.endContainer = this.getTextNodeSibling(prevOrNext, newMarkup.parent(), textNode2Start);
						rangeObject.endOffset = rangeObject.endContainer.length;
					}
				}
			}
		},

		/**
		 * takes a text node and return either the next recursive text node sibling or the previous
		 * @param previousOrNext boolean, false for previous, true for next sibling
		 * @param commonAncestorContainer dom object to be used as root for the sibling search
		 * @param currentTextNode dom object of the originating text node
		 * @return dom object of the sibling text node
		 * @hide
		 */
		getTextNodeSibling: function (previousOrNext, commonAncestorContainer, currentTextNode) {
			var textNodes = jQuery(commonAncestorContainer).textNodes(true), newIndex, index;

			index = textNodes.index(currentTextNode);
			if (index == -1) { // currentTextNode was not found
				return false;
			}
			newIndex = index + (!previousOrNext ? -1 : 1);
			return textNodes[newIndex] || false;
		},

		/**
		 * takes a selection tree and groups it into markup wrappable selection trees
		 * @param selectionTree rangeObject selection tree
		 * @param markupObject jQuery object of the markup to be applied (e.g. created with obj = jQuery('<b></b>'); )
		 * @return JS array of wrappable selection trees
		 * @hide
		 */
		optimizeSelectionTree4Markup: function (selectionTree, markupObject, tagComparator) {
			var groupMap = [],
				outerGroupIndex = 0,
				innerGroupIndex = 0,
				that = this,
				i,
			    j,
				endPosition,
			    startPosition;

			if (typeof tagComparator === 'undefined') {
				tagComparator = function (domobj, markupObject) {
					return that.standardTextLevelSemanticsComparator(markupObject);
				};
			}
			for (i = 0; i < selectionTree.length; i++) {
				// we are just interested in selected item, but not in non-selected items
				if (selectionTree[i].domobj && selectionTree[i].selection != 'none') {
					if (markupObject.isReplacingElement && tagComparator(markupObject[0], jQuery(selectionTree[i].domobj))) {
						if (groupMap[outerGroupIndex] !== undefined) {
							outerGroupIndex++;
						}
						groupMap[outerGroupIndex] = {};
						groupMap[outerGroupIndex].wrappable = true;
						groupMap[outerGroupIndex].elements = [];
						groupMap[outerGroupIndex].elements[innerGroupIndex] = selectionTree[i];
						outerGroupIndex++;

					} else if (this.canMarkupBeApplied2ElementAsWhole([selectionTree[i]], markupObject)) {
						// now check, if the children of our item could be wrapped all together by the markup object
						// if yes, add it to the current group
						if (groupMap[outerGroupIndex] === undefined) {
							groupMap[outerGroupIndex] = {};
							groupMap[outerGroupIndex].wrappable = true;
							groupMap[outerGroupIndex].elements = [];
						}
						if (markupObject.isReplacingElement) { //  && selectionTree[i].domobj.nodeType === 3
							/* we found the node to wrap for a replacing element. however there might
							 * be siblings which should be included as well
							 * although they are actually not selected. example:
							 * li
							 * |-textNode ( .selection = 'none')
							 * |-textNode (cursor inside, therefor .selection = 'partial')
							 * |-textNode ( .selection = 'none')
							 *
							 * in this case it would be useful to select the previous and following textNodes as well (they might result from a previous DOM manipulation)
							 * Think about other cases, where the parent is the Editable. In this case we propably only want to select from and until the next <br /> ??
							 * .... many possibilities, here I realize the two described cases
							 */

							// first find start element starting from the current element going backwards until sibling 0
							startPosition = i;
							for (j = i - 1; j >= 0; j--) {
								if (this.canMarkupBeApplied2ElementAsWhole([selectionTree[j]], markupObject) && this.isMarkupAllowedToStealSelectionTreeElement(selectionTree[j], markupObject)) {
									startPosition = j;
								} else {
									break;
								}
							}

							// now find the end element starting from the current element going forward until the last sibling
							endPosition = i;
							for (j = i + 1; j < selectionTree.length; j++) {
								if (this.canMarkupBeApplied2ElementAsWhole([selectionTree[j]], markupObject) && this.isMarkupAllowedToStealSelectionTreeElement(selectionTree[j], markupObject)) {
									endPosition = j;
								} else {
									break;
								}
							}

							// now add the elements to the groupMap
							innerGroupIndex = 0;
							for (j = startPosition; j <= endPosition; j++) {
								groupMap[outerGroupIndex].elements[innerGroupIndex] = selectionTree[j];
								groupMap[outerGroupIndex].elements[innerGroupIndex].selection = 'full';
								innerGroupIndex++;
							}
							innerGroupIndex = 0;
						} else {
							// normal text level semantics object, no siblings need to be selected
							groupMap[outerGroupIndex].elements[innerGroupIndex] = selectionTree[i];
							innerGroupIndex++;
						}
			