"""
Tests for dypybench problems. The 'x' in the module name is to make it run
late after other tests (since it is slow)
"""

import dataclasses
import os
from pathlib import Path

import pytest
from lmwrapper.openai_wrapper import get_open_ai_lm, OpenAiModelNames

from synthegrator.code_problems import DiscoveredTestsuite
from synthegrator.code_solver import DummyCodeSolverAutoRegressive, LmCodeSolverAutoRegressive
from synthegrator.problem_rendering import render_single_edit_codeproblem_autoregressive
from synthegrator.sandboxing import run_on_docker
from synthegrator.solution_eval import (
    apply_solution,
    evaluate_code_problem_execution,
    exact_match_metric,
    is_solution_syntax_likely_valid,
    parse_junit_test_cases,
)
from synthegrator.synthdatasets.dypybench import (
    _default_projects_root,
    _project_docker_envs,
    produce_dy_py_bench_problem,
    yield_dypybench,
)
from synthegrator.transformation_spec import StsEditable, StsSelectorLineSpan
from synthegrator.util import MultilineString

IN_GITHUB_ACTIONS = os.getenv("GITHUB_ACTIONS") == "true"


@pytest.fixture
def example_dypy_prob():
    fixture = Path(__file__).parent / "fixtures/document.py.fixture"

    function_w_doc = """def process_encoding(self, encoding: None | str = None) -> str:
    \"\"\"Process explicitly defined encoding or auto-detect it.

    If encoding is explicitly defined, ensure it is a valid encoding the python
    can deal with. If encoding is not specified, auto-detect it.

    Raises unicodec.InvalidEncodingName if explicitly set encoding is invalid.
    \"\"\"
    if encoding:
        return unicodec.normalize_encoding_name(encoding)
    return unicodec.detect_content_encoding(
        self.get_body_chunk() or b"",
        content_type_header=(
            self.headers.get("Content-Type", None) if self.headers else None
        ),
        markup="xml" if self.document_type == "xml" else "html",
    )"""

    import tempfile

    with tempfile.TemporaryDirectory() as tmpdir:
        # Make a project1/grab/document.py file in the tmpdir with contents of fixture
        Path(tmpdir).joinpath("project1/grab").mkdir(parents=True, exist_ok=True)
        Path(tmpdir).joinpath("project1/grab/document.py").write_text(
            fixture.read_text(),
        )

        problem_w_doc = produce_dy_py_bench_problem(
            task_id=str(1),
            path="grab/document.py",
            identifier="Document.process_encoding",
            func_begin=134,
            func_end=150,
            function=function_w_doc,
            project_id=str(1),
            projects_root=tmpdir,
            random_line_seed=2,
        )

        yield problem_w_doc, function_w_doc


@pytest.fixture
def example_dypy_prob_stripdoc():
    fixture = Path(__file__).parent / "fixtures/document.py.fixture"

    function_w_doc = """def process_encoding(self, encoding: None | str = None) -> str:
    \"\"\"Process explicitly defined encoding or auto-detect it.

    If encoding is explicitly defined, ensure it is a valid encoding the python
    can deal with. If encoding is not specified, auto-detect it.

    Raises unicodec.InvalidEncodingName if explicitly set encoding is invalid.
    \"\"\"
    if encoding:
        return unicodec.normalize_encoding_name(encoding)
    return unicodec.detect_content_encoding(
        self.get_body_chunk() or b"",
        content_type_header=(
            self.headers.get("Content-Type", None) if self.headers else None
        ),
        markup="xml" if self.document_type == "xml" else "html",
    )"""

    import tempfile

    with tempfile.TemporaryDirectory() as tmpdir:
        # Make a project1/grab/document.py file in the tmpdir with contents of fixture
        Path(tmpdir).joinpath("project1/grab").mkdir(parents=True, exist_ok=True)
        Path(tmpdir).joinpath("project1/grab/document.py").write_text(
            fixture.read_text(),
        )

        problem_w_doc = produce_dy_py_bench_problem(
            task_id=str(1),
            path="grab/document.py",
            identifier="Document.process_encoding",
            func_begin=134,
            func_end=150,
            function=function_w_doc,
            project_id=str(1),
            projects_root=tmpdir,
            strip_docstring=True,
            random_line_seed=2,
        )

        yield problem_w_doc, function_w_doc


@pytest.fixture
def example_dypy_prob_stripdoc_fixedline():
    fixture = Path(__file__).parent / "fixtures/document.py.fixture"

    function_w_doc = """def process_encoding(self, encoding: None | str = None) -> str:
    \"\"\"Process explicitly defined encoding or auto-detect it.

    If encoding is explicitly defined, ensure it is a valid encoding the python
    can deal with. If encoding is not specified, auto-detect it.

    Raises unicodec.InvalidEncodingName if explicitly set encoding is invalid.
    \"\"\"
    if encoding:
        return unicodec.normalize_encoding_name(encoding)
    return unicodec.detect_content_encoding(
        self.get_body_chunk() or b"",
        content_type_header=(
            self.headers.get("Content-Type", None) if self.headers else None
        ),
        markup="xml" if self.document_type == "xml" else "html",
    )"""

    import tempfile

    with tempfile.TemporaryDirectory() as tmpdir:
        # Make a project1/grab/document.py file in the tmpdir with contents of fixture
        Path(tmpdir).joinpath("project1/grab").mkdir(parents=True, exist_ok=True)
        Path(tmpdir).joinpath("project1/grab/document.py").write_text(
            fixture.read_text(),
        )

        problem_w_doc = produce_dy_py_bench_problem(
            task_id=str(1),
            path="grab/document.py",
            identifier="Document.process_encoding",
            func_begin=134,
            func_end=150,
            function=function_w_doc,
            project_id=str(1),
            projects_root=tmpdir,
            strip_docstring=True,
            random_line_seed=3,
            line_to_replace=141,
        )

        yield problem_w_doc, function_w_doc


@pytest.fixture
def example_dypy_prob_fixedline():
    fixture = Path(__file__).parent / "fixtures/document.py.fixture"

    function_w_doc = """def process_encoding(self, encoding: None | str = None) -> str:
    \"\"\"Process explicitly defined encoding or auto-detect it.

    If encoding is explicitly defined, ensure it is a valid encoding the python
    can deal with. If encoding is not specified, auto-detect it.

    Raises unicodec.InvalidEncodingName if explicitly set encoding is invalid.
    \"\"\"
    if encoding:
        return unicodec.normalize_encoding_name(encoding)
    return unicodec.detect_content_encoding(
        self.get_body_chunk() or b"",
        content_type_header=(
            self.headers.get("Content-Type", None) if self.headers else None
        ),
        markup="xml" if self.document_type == "xml" else "html",
    )"""

    import tempfile

    with tempfile.TemporaryDirectory() as tmpdir:
        # Make a project1/grab/document.py file in the tmpdir with contents of fixture
        Path(tmpdir).joinpath("project1/grab").mkdir(parents=True, exist_ok=True)
        Path(tmpdir).joinpath("project1/grab/document.py").write_text(
            fixture.read_text(),
        )

        problem_w_doc = produce_dy_py_bench_problem(
            task_id=str(1),
            path="grab/document.py",
            identifier="Document.process_encoding",
            func_begin=134,
            func_end=150,
            function=function_w_doc,
            project_id=str(1),
            projects_root=tmpdir,
            strip_docstring=False,
            random_line_seed=3,
            line_to_replace=141,
        )

        yield problem_w_doc, function_w_doc


def test_new_sample_no_eval(example_dypy_prob):
    problem_w_doc, _function_w_doc = example_dypy_prob
    solver = DummyCodeSolverAutoRegressive(None)
    solution = solver.solve(problem_w_doc)
    assert len(solution.solve_steps) == 1
    orig_text = MultilineString(
        problem_w_doc.working_directory.files.get_only_file().content_str,
    )
    text_selector = problem_w_doc.transformation_spec.statements[2].text_selector
    assert isinstance(text_selector, StsSelectorLineSpan)
    verb = problem_w_doc.transformation_spec.statements[2].verb
    assert isinstance(verb, StsEditable)
    assert (
        orig_text[text_selector.start_line_idx]
        == "            return unicodec.normalize_encoding_name(encoding)\n"
    )
    assert (
        solution.solve_steps[0].value
        == "            return unicodec.normalize_encoding_name(encoding)\n"
    )
    solve_text = MultilineString(apply_solution(solution).get_only_file().content_str)
    assert (
        solve_text[134 - 1 : 150]
        == """    def process_encoding(self, encoding: None | str = None) -> str:
        \"\"\"Process explicitly defined encoding or auto-detect it.

        If encoding is explicitly defined, ensure it is a valid encoding the python
        can deal with. If encoding is not specified, auto-detect it.

        Raises unicodec.InvalidEncodingName if explicitly set encoding is invalid.
        \"\"\"
        if encoding:
            return unicodec.normalize_encoding_name(encoding)
        return unicodec.detect_content_encoding(
            self.get_body_chunk() or b"",
            content_type_header=(
                self.headers.get("Content-Type", None) if self.headers else None
            ),
            markup="xml" if self.document_type == "xml" else "html",
        )\n"""
    )
    assert exact_match_metric(solution).is_success


def test_new_sample_no_eval_nodoc(example_dypy_prob_stripdoc):
    problem_wo_doc, _function_wo_doc = example_dypy_prob_stripdoc
    solver = DummyCodeSolverAutoRegressive(None)
    solution = solver.solve(problem_wo_doc)
    render = render_single_edit_codeproblem_autoregressive(problem_wo_doc)
    assert (
        render.prompt.text
        == "    def process_encoding(self, encoding: None | str = None) -> str:\n      "
        "  if encoding:\n"
    )
    assert len(solution.solve_steps) == 1
    orig_text = MultilineString(
        problem_wo_doc.working_directory.files.get_only_file().content_str,
    )
    text_selector = problem_wo_doc.transformation_spec.statements[-1].text_selector
    assert isinstance(text_selector, StsSelectorLineSpan)
    verb = problem_wo_doc.transformation_spec.statements[-1].verb
    assert isinstance(verb, StsEditable)
    assert (
        orig_text[text_selector.start_line_idx]
        == "            return unicodec.normalize_encoding_name(encoding)\n"
    )
    assert (
        solution.solve_steps[0].value
        == "            return unicodec.normalize_encoding_name(encoding)\n"
    )
    solve_text = MultilineString(apply_solution(solution).get_only_file().content_str)
    assert (
        solve_text[134 - 1 : 150]
        == """    def process_encoding(self, encoding: None | str = None) -> str:
        \"\"\"Process explicitly defined encoding or auto-detect it.

        If encoding is explicitly defined, ensure it is a valid encoding the python
        can deal with. If encoding is not specified, auto-detect it.

        Raises unicodec.InvalidEncodingName if explicitly set encoding is invalid.
        \"\"\"
        if encoding:
            return unicodec.normalize_encoding_name(encoding)
        return unicodec.detect_content_encoding(
            self.get_body_chunk() or b"",
            content_type_header=(
                self.headers.get("Content-Type", None) if self.headers else None
            ),
            markup="xml" if self.document_type == "xml" else "html",
        )\n"""
    )
    assert exact_match_metric(solution).is_success


def test_new_sample_no_eval_doc_fixed(example_dypy_prob_fixedline):
    problem_doc, _function_doc = example_dypy_prob_fixedline
    solver = DummyCodeSolverAutoRegressive(None)
    solution = solver.solve(problem_doc)
    render = render_single_edit_codeproblem_autoregressive(problem_doc)
    assert "if encoding:" not in render.prompt.text
    assert len(solution.solve_steps) == 1
    orig_text = MultilineString(
        problem_doc.working_directory.files.get_only_file().content_str,
    )
    text_selector = problem_doc.transformation_spec.statements[-1].text_selector
    assert isinstance(text_selector, StsSelectorLineSpan)
    verb = problem_doc.transformation_spec.statements[-1].verb
    assert text_selector.start_line_idx == 141
    assert orig_text[141] == "        if encoding:\n"
    assert isinstance(verb, StsEditable)
    assert orig_text[text_selector.start_line_idx] == "        if encoding:\n"
    assert solution.solve_steps[0].value == "        if encoding:\n"
    solve_text = MultilineString(apply_solution(solution).get_only_file().content_str)
    assert (
        solve_text[134 - 1 : 150]
        == """    def process_encoding(self, encoding: None | str = None) -> str:
        \"\"\"Process explicitly defined encoding or auto-detect it.

        If encoding is explicitly defined, ensure it is a valid encoding the python
        can deal with. If encoding is not specified, auto-detect it.

        Raises unicodec.InvalidEncodingName if explicitly set encoding is invalid.
        \"\"\"
        if encoding:
            return unicodec.normalize_encoding_name(encoding)
        return unicodec.detect_content_encoding(
            self.get_body_chunk() or b"",
            content_type_header=(
                self.headers.get("Content-Type", None) if self.headers else None
            ),
            markup="xml" if self.document_type == "xml" else "html",
        )\n"""
    )


@pytest.mark.skipif(IN_GITHUB_ACTIONS, reason="No openai key")
def test_new_sample_no_eval_chat(example_dypy_prob_fixedline):
    problem_doc, _function_doc = example_dypy_prob_fixedline
    lm = get_open_ai_lm(OpenAiModelNames.gpt_4o_mini)
    solver = LmCodeSolverAutoRegressive(lm)
    render = solver.prompt_renderer.render(problem_doc, lm)
    print("Prompt")
    print(render.prompt)
    assert render.prompt.is_text_a_chat()
    assert len(render.prompt.text) == 1
    assert render.prompt.text[0].content.startswith("Give a completion to the following code")
    solution = solver.solve(problem_doc)
    print("solution")
    print(solution)
    print("solve step:")
    print(solution.solve_steps[0])
    print("known solution:")
    print(problem_doc.known_solutions[0].solve_steps[0])
    assert solution.solve_steps[0].mark_id == problem_doc.known_solutions[0].solve_steps[0].mark_id
    assert solution.solve_steps[0].path == problem_doc.known_solutions[0].solve_steps[0].path
    val = solution.solve_steps[0].value
    assert len(val.strip().split("\n")) == 1  # just a single line
    assert val.strip() != ""
    applied = apply_solution(solution)
    text = applied.get_only_file().content_str
    # Ok we don't know exactly what the model will give us.
    # so we are just going to check if the output has been integrated
    # into the right part of the file.
    clean_v = val.strip("\n")
    expected_substr = f'''
    def select(self, *args: Any, **kwargs: Any) -> SelectorList[_Element]:
        return XpathSelector(self.tree).select(*args, **kwargs)

    def process_encoding(self, encoding: None | str = None) -> str:
        """Process explicitly defined encoding or auto-detect it.

        If encoding is explicitly defined, ensure it is a valid encoding the python
        can deal with. If encoding is not specified, auto-detect it.

        Raises unicodec.InvalidEncodingName if explicitly set encoding is invalid.
        """
{clean_v}
            return unicodec.normalize_encoding_name(encoding)
        return unicodec.detect_content_encoding(
'''.strip()
    print("Text")
    print(text)
    print("Expected")
    print(expected_substr)
    assert expected_substr in text


def test_new_sample_no_eval_nodoc_fixed(example_dypy_prob_stripdoc_fixedline):
    problem_wo_doc, _function_wo_doc = example_dypy_prob_stripdoc_fixedline
    solver = DummyCodeSolverAutoRegressive(None)
    solution = solver.solve(problem_wo_doc)
    render = render_single_edit_codeproblem_autoregressive(problem_wo_doc)
    assert (
        render.prompt.text
        == "    def process_encoding(self, encoding: None | str = None) -> str:\n"
    )
    assert len(solution.solve_steps) == 1
    orig_text = MultilineString(
        problem_wo_doc.working_directory.files.get_only_file().content_str,
    )
    text_selector = problem_wo_doc.transformation_spec.statements[-1].text_selector
    assert isinstance(text_selector, StsSelectorLineSpan)
    verb = problem_wo_doc.transformation_spec.statements[-1].verb
    assert text_selector.start_line_idx == 141
    assert orig_text[141] == "        if encoding:\n"
    assert isinstance(verb, StsEditable)
    assert orig_text[text_selector.start_line_idx] == "        if encoding:\n"
    assert solution.solve_steps[0].value == "        if encoding:\n"
    solve_text = MultilineString(apply_solution(solution).get_only_file().content_str)
    assert (
        solve_text[134 - 1 : 150]
        == """    def process_encoding(self, encoding: None | str = None) -> str:
        \"\"\"Process explicitly defined encoding or auto-detect it.

        If encoding is explicitly defined, ensure it is a valid encoding the python
        can deal with. If encoding is not specified, auto-detect it.

        Raises unicodec.InvalidEncodingName if explicitly set encoding is invalid.
        \"\"\"
        if encoding:
            return unicodec.normalize_encoding_name(encoding)
        return unicodec.detect_content_encoding(
            self.get_body_chunk() or b"",
            content_type_header=(
                self.headers.get("Content-Type", None) if self.headers else None
            ),
            markup="xml" if self.document_type == "xml" else "html",
        )\n"""
    )
    assert exact_match_metric(solution).is_success


def test_new_sample_eval_same_files(example_dypy_prob):
    problem_w_doc, _function_w_doc = example_dypy_prob
    solver = DummyCodeSolverAutoRegressive(None)
    solution = solver.solve(problem_w_doc)
    applied_files = apply_solution(solution)
    assert (
        applied_files.get_only_file().content_str
        == problem_w_doc.working_directory.files.get_only_file().content_str
    )
    assert applied_files == problem_w_doc.working_directory.files


@pytest.mark.skipif(IN_GITHUB_ACTIONS, reason="Dypybench downloads large docker images")
def test_new_sample_eval(example_dypy_prob):
    problem_w_doc, _function_w_doc = example_dypy_prob
    solver = DummyCodeSolverAutoRegressive(None)
    solution = solver.solve(problem_w_doc)
    print(solution)
    print(solution.solve_steps)
    assert apply_solution(solution) == problem_w_doc.working_directory.files
    result = evaluate_code_problem_execution(problem_w_doc, solution)
    assert result.summary.tests > 0
    assert result.summary.errors == 0
    assert result.all_passed()


@pytest.mark.skipif(IN_GITHUB_ACTIONS, reason="Dypybench downloads large docker images")
@pytest.mark.skip(
    reason=(
        "save time on execution. Uncomment if need to debug other more end-to-end tests"
    ),
)
def test_run_no_content(example_dypy_prob):
    problem_w_doc, _ = example_dypy_prob
    assert len(problem_w_doc.test_cases) == 1
    test_case = problem_w_doc.test_cases[0]
    print(test_case)
    assert isinstance(test_case, DiscoveredTestsuite)
    cmd_outputs = run_on_docker(
        docker_context=problem_w_doc.environment.docker_env,
        cmds=test_case.cmds,
    )
    exec_result = test_case.cmds_results_to_suite(cmd_outputs)
    parsed_junit = parse_junit_test_cases(exec_result.xml_result, problem_w_doc)
    assert parsed_junit.all_passed()


@pytest.mark.skipif(IN_GITHUB_ACTIONS, reason="Dypybench downloads large docker images")
@pytest.mark.skip(
    reason=(
        "save time on execution. Uncomment if need to debug other more end-to-end tests"
    ),
)
def test_run_same_content(example_dypy_prob):
    problem_w_doc, _ = example_dypy_prob
    assert len(problem_w_doc.test_cases) == 1
    test_case = problem_w_doc.test_cases[0]
    print(test_case)
    assert isinstance(test_case, DiscoveredTestsuite)
    cmd_outputs = run_on_docker(
        docker_context=problem_w_doc.environment.docker_env,
        cmds=test_case.cmds,
        files=problem_w_doc.working_directory.files,
    )
    exec_result = test_case.cmds_results_to_suite(cmd_outputs)
    parsed_junit = parse_junit_test_cases(exec_result.xml_result, problem_w_doc)
    assert parsed_junit.all_passed()
    assert (
        apply_solution(problem_w_doc.known_solutions[0])
        == problem_w_doc.working_directory.files
    )


@pytest.mark.skipif(IN_GITHUB_ACTIONS, reason="Dypybench downloads large docker images")
def test_new_sample_eval_bad_solve(example_dypy_prob):
    problem_w_doc, _function_w_doc = example_dypy_prob
    prob_files = problem_w_doc.working_directory.files
    assert len(list(prob_files.walk(include_dirs=False))) == 1
    assert "project1/grab/document.py" in prob_files
    solver = DummyCodeSolverAutoRegressive(None)
    solution = solver.solve(problem_w_doc)
    assert len(solution.solve_steps) == 1
    print(solution.solve_steps[0].value)
    orig_text = MultilineString(
        problem_w_doc.working_directory.files.get_only_file().content_str,
    )
    text_selector = problem_w_doc.transformation_spec.statements[2].text_selector
    assert isinstance(text_selector, StsSelectorLineSpan)
    verb = problem_w_doc.transformation_spec.statements[2].verb
    assert isinstance(verb, StsEditable)
    assert (
        orig_text[text_selector.start_line_idx]
        == "            return unicodec.normalize_encoding_name(encoding)\n"
    )
    assert (
        solution.solve_steps[0].value
        == "            return unicodec.normalize_encoding_name(encoding)\n"
    )
    solution.solve_steps = (
        dataclasses.replace(solution.solve_steps[0], value=" " * 12 + "assert False\n"),
    )
    solve_text = MultilineString(apply_solution(solution).get_only_file().content_str)
    assert (
        solve_text[134 - 1 : 150]
        == """    def process_encoding(self, encoding: None | str = None) -> str:
        \"\"\"Process explicitly defined encoding or auto-detect it.

        If encoding is explicitly defined, ensure it is a valid encoding the python
        can deal with. If encoding is not specified, auto-detect it.

        Raises unicodec.InvalidEncodingName if explicitly set encoding is invalid.
        \"\"\"
        if encoding:
            assert False
        return unicodec.detect_content_encoding(
            self.get_body_chunk() or b"",
            content_type_header=(
                self.headers.get("Content-Type", None) if self.headers else None
            ),
            markup="xml" if self.document_type == "xml" else "html",
        )\n"""
    )

    # Make sure gets on the docker
    r = run_on_docker(
        docker_context=_project_docker_envs[1],
        files=apply_solution(solution),
        cmds=[
            "cat project1/grab/document.py",
        ],
    )
    assert r[-1].completed
    assert r[-1].stdout.decode() == str(solve_text)

    assert is_solution_syntax_likely_valid(solution)
    result = evaluate_code_problem_execution(problem_w_doc, solution)
    assert result.summary.tests > 0
    assert result.summary.failures > 0
    assert not result.all_passed()


@pytest.mark.skipif(IN_GITHUB_ACTIONS, reason="Dypybench downloads large docker images")
def test_new_sample_eval_bad_solve_syntax_errors(example_dypy_prob):
    problem_w_doc, _function_w_doc = example_dypy_prob
    solver = DummyCodeSolverAutoRegressive(None)
    solution = solver.solve(problem_w_doc)
    assert len(solution.solve_steps) == 1
    print(solution.solve_steps[0].value)
    orig_text = MultilineString(
        problem_w_doc.working_directory.files.get_only_file().content_str,
    )
    text_selector = problem_w_doc.transformation_spec.statements[2].text_selector
    assert isinstance(text_selector, StsSelectorLineSpan)
    verb = problem_w_doc.transformation_spec.statements[2].verb
    assert isinstance(verb, StsEditable)
    assert (
        orig_text[text_selector.start_line_idx]
        == "            return unicodec.normalize_encoding_name(encoding)\n"
    )
    assert (
        solution.solve_steps[0].value
        == "            return unicodec.normalize_encoding_name(encoding)\n"
    )
    solution.solve_steps = (
        dataclasses.replace(
            solution.solve_steps[0],
            value=" " * 12 + "assert False + :: sfsd if\n",
        ),
    )
    assert not is_solution_syntax_likely_valid(solution)
    result = evaluate_code_problem_execution(problem_w_doc, solution)
    assert not result.all_passed()


def test_yield_dypy_single():
    try:
        root = _default_projects_root()
    except FileNotFoundError:
        pytest.skip("Dypybench projects root not found")
    problems = yield_dypybench()
    list(problems)
    print("got problem")


# @pytest.mark.skipif(
#    IN_GITHUB_ACTIONS,
# def test_dypybench_pytest_no_steps():
#    ).read_text()
#
#        ),
#        1,
#        ).read_text(),


# @pytest.mark.skipif(
#    IN_GITHUB_ACTIONS,
# def test_dypybench_broken_pytest():
#    assert False
#        1,
#        ).read_text(),


# def test_indent_fix():
#    assert False
#
#    function_w_doc = """def process_encoding(self, encoding: None | str = None) -> str:
#    \"\"\"Process explicitly defined encoding or auto-detect it.
#
#    If encoding is explicitly defined, ensure it is a valid encoding the python
#    can deal with. If encoding is not specified, auto-detect it.
#
#    Raises unicodec.InvalidEncodingName if explicitly set encoding is invalid.
#    \"\"\"
#    if encoding:
#    return unicodec.detect_content_encoding(
#        ),
#    )"""
#
#
#        problem_w_doc.format_with_answer(
#            indented_answer,
#        ),
#    assert (
#
#    assert (
#        indented_answer
#        == "            return unicodec.normalize_encoding_name(encoding)"
#
#        "\t",
#
#        "\t",
#
#        "\t",
#
#        "\t",
#
