from pathlib import Path
from plsconvert.converters.abstract import Converter
from plsconvert.converters.registry import addMethodData, registerConverter
from plsconvert.utils.graph import PairList
from plsconvert.utils.files import runCommand, fileType
from plsconvert.utils.dependency import Dependencies, ToolDependency as Tool


@registerConverter
class ffmpeg(Converter):
    """
    Video and audio converter using ffmpeg.
    """

    @property
    def name(self) -> str:
        return "FFmpeg Converter"

    @property
    def dependencies(self) -> Dependencies:
        return Dependencies([Tool("ffmpeg")])
    
    _SUPPORTED_PAIRS_VIDEO = PairList.all2all(
        [
            "mp4",
            "mkv",
            "mov",
            "avi",
            "webm",
            "flv",
            "mpg",
            "ts",
            "wmv",
            "hls",
            "rtsp",
            "rtmp",
            "mp3",
            "aac",
            "flac",
            "wav",
            "ogg",
            "wma",
            "m4a",
            "opus",
        ],
        [
            "mp4",
            "mkv",
            "mov",
            "avi",
            "webm",
            "flv",
            "mp3",
            "aac",
            "flac",
            "wav",
            "ogg",
            "opus",
            "hls",
            "dash",
            "null",
        ],
    )
    _SUPPORTED_PAIRS_IMAGES = PairList.all2all(
        ["png", "jpg", "jpeg", "gif", "bmp", "tiff"],
        [
            "mp4",
            "mkv",
            "mov",
            "avi",
            "webm",
            "flv",
            "gif",
            "png",
            "jpg",
            "hls",
            "dash",
        ],
    )

    @addMethodData(_SUPPORTED_PAIRS_VIDEO + _SUPPORTED_PAIRS_IMAGES, False)
    def media_to_media(
        self, input: Path, output: Path, input_extension: str, output_extension: str
    ) -> None:
        command = ["ffmpeg", "-y", "-i", str(input), str(output)]
        if (
            fileType(input_extension) == "video"
            and fileType(output_extension) == "image"
        ):
            command = ["ffmpeg", "-y", "-i", str(input), "-vframes", "1", str(output)]

        runCommand(command)


@registerConverter
class imagemagick(Converter):
    """
    Image converter using imagemagick.
    """

    @property
    def name(self) -> str:
        return "ImageMagick Converter"

    @property
    def dependencies(self) -> Dependencies:
        return Dependencies([Tool("magick")])

    _SUPPORTED_PAIRS_IMAGES = PairList.all2all(
            [
                "3fr",
                "3g2",
                "3gp",
                "aai",
                "ai",
                "apng",
                "art",
                "arw",
                "avi",
                "avs",
                "bayer",
                "bayera",
                "bgr",
                "bgra",
                "bgro",
                "bie",
                "bmp",
                "bmp2",
                "bmp3",
                "cal",
                "cals",
                "canvas",
                "caption",
                "cin",
                "clip",
                "cmyk",
                "cmyka",
                "cr2",
                "cr3",
                "crw",
                "cube",
                "cur",
                "cut",
                "data",
                "dcm",
                "dcr",
                "dcraw",
                "dcx",
                "dds",
                "dfont",
                "dng",
                "dot",
                "dpx",
                "dxt1",
                "dxt5",
                "epdf",
                "epi",
                "eps",
                "epsf",
                "epsi",
                "ept",
                "ept2",
                "ept3",
                "erf",
                "exr",
                "farbfeld",
                "fax",
                "ff",
                "fff",
                "file",
                "fits",
                "fl32",
                "flv",
                "fractal",
                "ftp",
                "fts",
                "ftxt",
                "g3",
                "g4",
                "gif",
                "gif87",
                "gradient",
                "gray",
                "graya",
                "group4",
                "gv",
                "hald",
                "hdr",
                "hrz",
                "htm",
                "html",
                "http",
                "https",
                "icb",
                "ico",
                "icon",
                "iiq",
                "inline",
                "ipl",
                "j2c",
                "j2k",
                "jbg",
                "jbig",
                "jng",
                "jnx",
                "jp2",
                "jpc",
                "jpe",
                "jpeg",
                "jpg",
                "jpm",
                "jps",
                "jpt",
                "jxl",
                "k25",
                "kdc",
                "label",
                "m2v",
                "m4v",
                "mac",
                "map",
                "mask",
                "mat",
                "mdc",
                "mef",
                "miff",
                "mkv",
                "mng",
                "mono",
                "mos",
                "mov",
                "mp4",
                "mpc",
                "mpeg",
                "mpg",
                "mpo",
                "mrw",
                "msl",
                "msvg",
                "mtv",
                "mvg",
                "nef",
                "nrw",
                "ora",
                "orf",
                "otb",
                "otf",
                "pal",
                "palm",
                "pam",
                "pango",
                "pattern",
                "pbm",
                "pcd",
                "pcds",
                "pcl",
                "pct",
                "pcx",
                "pdb",
                "pdf",
                "pdfa",
                "pef",
                "pes",
                "pfa",
                "pfb",
                "pfm",
                "pgm",
                "pgx",
                "phm",
                "picon",
                "pict",
                "pix",
                "pjpeg",
                "plasma",
                "png",
                "png00",
                "png24",
                "png32",
                "png48",
                "png64",
                "png8",
                "pnm",
                "ppm",
                "ps",
                "psb",
                "psd",
                "ptif",
                "pwp",
                "qoi",
                "radial-gradient",
                "raf",
                "ras",
                "raw",
                "rgb",
                "rgb565",
                "rgba",
                "rgbo",
                "rgf",
                "rla",
                "rle",
                "rmf",
                "rsvg",
                "rw2",
                "rwl",
                "scr",
                "sct",
                "sfw",
                "sgi",
                "shtml",
                "six",
                "sixel",
                "sr2",
                "srf",
                "srw",
                "stegano",
                "sti",
                "strimg",
                "sun",
                "svg",
                "svgz",
                "text",
                "tga",
                "tiff",
                "tiff64",
                "tile",
                "tim",
                "tm2",
                "ttc",
                "ttf",
                "uil",
                "uyvy",
                "vda",
                "vicar",
                "vid",
                "viff",
                "vips",
                "vst",
                "wbmp",
                "webm",
                "webp",
                "wmf",
                "wmv",
                "wmz",
                "wpg",
                "x",
                "x3f",
                "xbm",
                "xc",
                "xcf",
                "xpm",
                "xps",
                "xv",
                "xwd",
                "ycbcr",
                "ycbcra",
                "yuv",
            ],
            [
                "dng",
                "video",
                "aai",
                "pdf",
                "apng",
                "art",
                "ashlar",
                "avs",
                "bayer",
                "bgr",
                "bie",
                "bmp",
                "cals",
                "xc",
                "cin",
                "cip",
                "clip",
                "cmyk",
                "cube",
                "icon",
                "ico",
                "cut",
                "inline",
                "dcm",
                "pcx",
                "dds",
                "ttf",
                "dot",
                "dpx",
                "ps",
                "ps2",
                "ps3",
                "ept",
                "exr",
                "farbfeld",
                "fax",
                "fits",
                "fl32",
                "ftxt",
                "gif",
                "gradient",
                "gray",
                "tiff",
                "hald",
                "hdr",
                "histogram",
                "hrz",
                "html",
                "url",
                "ipl",
                "jp2",
                "jbig",
                "png",
                "jnx",
                "jpeg",
                "json",
                "jxl",
                "kernel",
                "label",
                "mac",
                "map",
                "mask",
                "mat",
                "matte",
                "miff",
                "mkv",
                "mono",
                "mpc",
                "msl",
                "svg",
                "mtv",
                "mvg",
                "ora",
                "otb",
                "uyvy",
                "palm",
                "pnm",
                "pango",
                "pattern",
                "pcd",
                "pcl",
                "pict",
                "pdb",
                "pes",
                "pgx",
                "pix",
                "psd",
                "pwp",
                "qoi",
                "sun",
                "rgb",
                "rgf",
                "rla",
                "rle",
                "scr",
                "sct",
                "sfw",
                "sgi",
                "sixel",
                "stegano",
                "strimg",
                "tga",
                "thumbnail",
                "tile",
                "tim",
                "tim2",
                "ubrl",
                "vicar",
                "vid",
                "viff",
                "vips",
                "wbmp",
                "webp",
                "wmf",
                "wpg",
                "x",
                "xbm",
                "xcf",
                "xpm",
                "xps",
                "xwd",
                "ycbcr",
                "yuv",
            ],
        )

    @addMethodData(_SUPPORTED_PAIRS_IMAGES, False)
    def image_to_image(
        self, input: Path, output: Path, input_extension: str, output_extension: str
    ) -> None:
        command = ["magick", str(input), str(output)]
        if output_extension == "ico":
            command.insert(2, "-define")
            command.insert(3, 'icon:auto-resize=256,128,96,64,48,32,16')
        runCommand(command)
