"""Weekly report generator for architect continuous improvement loop.

This module generates human-readable weekly improvement reports that are synthetic
(1-2 pages) with actionable insights.
"""

from datetime import datetime, timedelta
from pathlib import Path
from typing import Dict

from coffee_maker.autonomous.architect_metrics import ArchitectMetrics


class WeeklyReportGenerator:
    """Generate weekly spec review reports.

    Reports are synthetic (1-2 pages) with actionable insights.
    """

    def __init__(self, metrics: ArchitectMetrics, output_dir: Path = Path("docs/architecture")):
        """Initialize report generator.

        Args:
            metrics: ArchitectMetrics instance
            output_dir: Directory for report files
        """
        self.metrics = metrics
        self.output_dir = output_dir
        self.output_dir.mkdir(parents=True, exist_ok=True)

    def generate_report(self, review_findings: Dict) -> Path:
        """Generate weekly review report.

        Args:
            review_findings: Dict with review results:
                - specs_reviewed: List[str]
                - simplifications_made: List[Dict]
                - reuse_opportunities: List[Dict]
                - recommendations: List[str]

        Returns:
            Path to generated report file
        """
        date_str = datetime.now().strftime("%Y-%m-%d")
        report_file = self.output_dir / f"WEEKLY_SPEC_REVIEW_{date_str}.md"

        summary = self.metrics.get_summary()

        report_content = self._build_report_content(review_findings, summary)

        report_file.write_text(report_content)
        return report_file

    def _build_report_content(self, findings: Dict, summary: Dict) -> str:
        """Build markdown report content.

        Args:
            findings: Review findings dict
            summary: Summary metrics dict

        Returns:
            Markdown report content
        """
        date_str = datetime.now().strftime("%Y-%m-%d")

        content = f"""# Weekly Spec Review - {date_str}

## Summary

Reviewed {len(findings['specs_reviewed'])} specs, identified {len(findings['reuse_opportunities'])} reuse opportunities, simplified {len(findings['simplifications_made'])} implementations.

## Metrics

- **Simplification Rate**: {summary['avg_reduction_percent']:.1f}% average complexity reduction
- **Reuse Rate**: {summary['total_reuse_opportunities']} components reused across specs
- **Effort Saved**: {summary['total_effort_saved']:.1f} hours (cumulative)
- **Specs Reviewed**: {summary['specs_reviewed']} total

## Improvements Made This Week

"""

        # Add simplifications
        for simp in findings["simplifications_made"]:
            content += f"""### {simp['spec_id']}: {simp['title']}

**Reduction**: {simp['reduction_percent']:.1f}% ({simp['original_hours']}h → {simp['simplified_hours']}h)

**Changes**: {simp['description']}

**Impact**: Saved {simp['effort_saved']:.1f} hours implementation time

"""

        # Add reuse opportunities
        if findings["reuse_opportunities"]:
            content += "\n## Reuse Opportunities Identified\n\n"

            for reuse in findings["reuse_opportunities"]:
                components_str = ", ".join(reuse["components"])
                content += f"- **{reuse['spec_id']}**: Reuses {components_str}\n"

        # Add recommendations
        if findings["recommendations"]:
            content += "\n## Recommendations\n\n"

            for i, rec in enumerate(findings["recommendations"], 1):
                content += f"{i}. {rec}\n"

        # Add next week focus
        next_review_date = (datetime.now() + timedelta(days=7)).strftime("%Y-%m-%d")
        content += f"""

## Next Week Focus

- Review new priorities added to ROADMAP
- Continue identifying shared patterns
- Update older specs with new best practices
- Monitor implementation feedback for spec improvements

---

**Generated by**: architect agent (continuous improvement loop)
**Next Review**: {next_review_date}
"""

        return content
