"""
Delta – key-aware DataFrame diff
Copyright (c) 2025  MIT License
"""
from __future__ import annotations

from .diff import trim_whitespace

import pandas as pd


class Delta:  # pylint: disable=too-few-public-methods
    """
    Compare two DataFrames on one or many key columns.

    Parameters
    ----------
    df_a, df_b : pandas.DataFrame
        DataFrames to compare.
    keys : str | list[str]
        Column(s) that uniquely identify each row.
    abs_tol, rel_tol : float, default 0
        Absolute / relative numeric tolerance when checking for equality.
    """

    def __init__(
        self,
        df_a: pd.DataFrame,
        df_b: pd.DataFrame,
        keys: str | list[str],
        *,
        abs_tol: float = 0.0,
        rel_tol: float = 0.0,
    ) -> None:
        self.df_a = trim_whitespace(df_a.copy())
        self.df_b = trim_whitespace(df_b.copy())
        self.keys = [keys] if isinstance(keys, str) else list(keys)
        self.abs_tol = abs_tol
        self.rel_tol = rel_tol

        self._build_diff_frames()

    # --------------------------------------------------------------------- #
    # Public attributes filled by _build_diff_frames()
    #
    unmatched_a: pd.DataFrame  # rows in A but not in B
    unmatched_b: pd.DataFrame  # rows in B but not in A
    mismatches: pd.DataFrame   # aligned rows whose non-key columns differ
    # --------------------------------------------------------------------- #

    # ------------------------ public helpers ----------------------------- #
    def changed(self, column: str) -> pd.DataFrame:
        """Return rows where *column* differs between A and B."""
        col_a = f"{column}_a"
        col_b = f"{column}_b"
        return self.mismatches[self.keys + [col_a, col_b]]

    def to_html(self, path: str | None = None) -> str | None:
        """
        Generate an HTML report of the comparison.

        Parameters
        ----------
        path : str, optional
            File path to save the HTML report. If None, returns the HTML string.
        """
        from jinja2 import Template

        template_str = """
        <!DOCTYPE html>
        <html lang="en">
        <head>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <title>Analysta Delta Report</title>
            <link rel="preconnect" href="https://fonts.googleapis.com">
            <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
            <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600&display=swap" rel="stylesheet">
            <style>
                :root {
                    --bg-color: #f8fafc;
                    --card-bg: #ffffff;
                    --text-primary: #1e293b;
                    --text-secondary: #64748b;
                    --accent-color: #3b82f6;
                    --border-color: #e2e8f0;
                    --success-color: #10b981;
                    --warning-color: #f59e0b;
                    --danger-color: #ef4444;
                    --header-bg: #f1f5f9;
                }

                body {
                    font-family: 'Inter', sans-serif;
                    background-color: var(--bg-color);
                    color: var(--text-primary);
                    margin: 0;
                    padding: 2rem;
                    line-height: 1.5;
                }

                .container {
                    max-width: 1200px;
                    margin: 0 auto;
                }

                header {
                    margin-bottom: 3rem;
                    text-align: center;
                }

                h1 {
                    font-weight: 600;
                    color: var(--text-primary);
                    margin-bottom: 0.5rem;
                    font-size: 2.5rem;
                    letter-spacing: -0.025em;
                }

                .subtitle {
                    color: var(--text-secondary);
                    font-size: 1.1rem;
                }

                .stats-grid {
                    display: grid;
                    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
                    gap: 1.5rem;
                    margin-bottom: 3rem;
                }

                .stat-card {
                    background: var(--card-bg);
                    padding: 1.5rem;
                    border-radius: 12px;
                    box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
                    border: 1px solid var(--border-color);
                    transition: transform 0.2s;
                }

                .stat-card:hover {
                    transform: translateY(-2px);
                }

                .stat-value {
                    font-size: 2.5rem;
                    font-weight: 700;
                    margin-bottom: 0.5rem;
                }

                .stat-label {
                    color: var(--text-secondary);
                    font-weight: 500;
                    text-transform: uppercase;
                    font-size: 0.875rem;
                    letter-spacing: 0.05em;
                }

                .stat-card.danger .stat-value { color: var(--danger-color); }
                .stat-card.warning .stat-value { color: var(--warning-color); }
                .stat-card.info .stat-value { color: var(--accent-color); }

                .section {
                    background: var(--card-bg);
                    border-radius: 16px;
                    box-shadow: 0 10px 15px -3px rgba(0, 0, 0, 0.1);
                    margin-bottom: 3rem;
                    overflow: hidden;
                    border: 1px solid var(--border-color);
                }

                .section-header {
                    padding: 1.5rem 2rem;
                    border-bottom: 1px solid var(--border-color);
                    background: #fff;
                    display: flex;
                    align-items: center;
                    justify_content: space-between;
                }

                .section-title {
                    font-size: 1.25rem;
                    font-weight: 600;
                    color: var(--text-primary);
                    margin: 0;
                }

                .table-container {
                    overflow-x: auto;
                    max-height: 600px;
                }

                table {
                    width: 100%;
                    border-collapse: collapse;
                    font-size: 0.95rem;
                }

                th {
                    background-color: var(--header-bg);
                    color: var(--text-secondary);
                    font-weight: 600;
                    text-align: left;
                    padding: 1rem 1.5rem;
                    position: sticky;
                    top: 0;
                    z-index: 10;
                    border-bottom: 1px solid var(--border-color);
                }

                td {
                    padding: 1rem 1.5rem;
                    border-bottom: 1px solid var(--border-color);
                    color: var(--text-primary);
                }

                tr:last-child td {
                    border-bottom: none;
                }

                tr:hover td {
                    background-color: #f8fafc;
                }

                .empty-state {
                    padding: 3rem;
                    text-align: center;
                    color: var(--text-secondary);
                    font-style: italic;
                }

                /* Scrollbar styling */
                ::-webkit-scrollbar {
                    width: 8px;
                    height: 8px;
                }
                ::-webkit-scrollbar-track {
                    background: #f1f1f1;
                }
                ::-webkit-scrollbar-thumb {
                    background: #cbd5e1;
                    border-radius: 4px;
                }
                ::-webkit-scrollbar-thumb:hover {
                    background: #94a3b8;
                }
            </style>
        </head>
        <body>
            <div class="container">
                <header>
                    <h1>Analysta Delta Report</h1>
                    <div class="subtitle">Comparison Summary & Detailed Differences</div>
                </header>
                
                <div class="stats-grid">
                    <div class="stat-card info">
                        <div class="stat-value">{{ unmatched_a_count }}</div>
                        <div class="stat-label">Rows only in A</div>
                    </div>
                    <div class="stat-card info">
                        <div class="stat-value">{{ unmatched_b_count }}</div>
                        <div class="stat-label">Rows only in B</div>
                    </div>
                    <div class="stat-card {% if mismatches_count > 0 %}danger{% else %}info{% endif %}">
                        <div class="stat-value">{{ mismatches_count }}</div>
                        <div class="stat-label">Mismatches</div>
                    </div>
                </div>

                {% if unmatched_a_count > 0 %}
                <div class="section">
                    <div class="section-header">
                        <h2 class="section-title">Rows in A only <span style="color: var(--text-secondary); font-size: 0.9em; font-weight: normal;">(Top 50)</span></h2>
                    </div>
                    <div class="table-container">
                        {{ unmatched_a_table }}
                    </div>
                </div>
                {% endif %}

                {% if unmatched_b_count > 0 %}
                <div class="section">
                    <div class="section-header">
                        <h2 class="section-title">Rows in B only <span style="color: var(--text-secondary); font-size: 0.9em; font-weight: normal;">(Top 50)</span></h2>
                    </div>
                    <div class="table-container">
                        {{ unmatched_b_table }}
                    </div>
                </div>
                {% endif %}

                {% if mismatches_count > 0 %}
                <div class="section">
                    <div class="section-header">
                        <h2 class="section-title">Mismatches <span style="color: var(--text-secondary); font-size: 0.9em; font-weight: normal;">(Top 50)</span></h2>
                    </div>
                    <div class="table-container">
                        {{ mismatches_table }}
                    </div>
                </div>
                {% endif %}
                
                {% if unmatched_a_count == 0 and unmatched_b_count == 0 and mismatches_count == 0 %}
                <div class="section">
                    <div class="empty-state">
                        ✨ Perfect match! No differences found between the datasets.
                    </div>
                </div>
                {% endif %}
            </div>
        </body>
        </html>
        """

        template = Template(template_str)
        html = template.render(
            unmatched_a_count=len(self.unmatched_a),
            unmatched_b_count=len(self.unmatched_b),
            mismatches_count=len(self.mismatches),
            unmatched_a_table=self.unmatched_a.head(50).to_html(index=False, border=0),
            unmatched_b_table=self.unmatched_b.head(50).to_html(index=False, border=0),
            mismatches_table=self.mismatches.head(50).to_html(index=False, border=0),
        )

        if path:
            with open(path, "w", encoding="utf-8") as f:
                f.write(html)
            return None
        return html


    # ------------------------ internal logic ----------------------------- #
    def _build_diff_frames(self) -> None:
        # Outer merge tells us where rows exist
        merged = self.df_a.merge(
            self.df_b,
            on=self.keys,
            how="outer",
            suffixes=("_a", "_b"),
            indicator=True,
        )

        # A ⧵ B  and  B ⧵ A
        self.unmatched_a = merged.query("_merge == 'left_only'").drop(columns="_merge")
        self.unmatched_b = merged.query("_merge == 'right_only'").drop(columns="_merge")

        # Rows present in both → look for cell differences
        both = merged.query("_merge == 'both'").drop(columns="_merge").copy()

        diff_mask = pd.Series(False, index=both.index)
        for base in (c[:-2] for c in both.columns if c.endswith("_a")):
            col_a, col_b = f"{base}_a", f"{base}_b"
            if pd.api.types.is_numeric_dtype(both[col_a]):
                equal_num = (
                    (both[col_a] - both[col_b]).abs()
                    <= self.abs_tol + self.rel_tol * both[col_b].abs()
                )
                diff_mask |= ~equal_num
            else:
                diff_mask |= both[col_a] != both[col_b]

        self.mismatches = both.loc[diff_mask].reset_index(drop=True)

    # ------------------------ dunder sugar ------------------------------- #
    def __repr__(self) -> str:  # pragma: no cover
        return (
            f"<Delta unmatched_a={len(self.unmatched_a)}, "
            f"unmatched_b={len(self.unmatched_b)}, "
            f"mismatches={len(self.mismatches)}>"
        )

