import logging

import numpy as np
import scipy.fftpack
from numpy.typing import NDArray

from AlgoTuneTasks.base import register_task, Task


@register_task("dct_type_I_scipy_fftpack")
class DCTTypeIScipyFFTpack(Task):
    def __init__(self, **kwargs):
        super().__init__(**kwargs)

    def generate_problem(self, n: int, random_seed: int = 1) -> NDArray:
        """
        Generate an (n+1) x (n+1) real-valued array for DCT Type I.
        """
        logging.debug(
            f"Generating DCT Type I problem with size {n + 1}x{n + 1} using scipy.fftpack, random_seed={random_seed}."
        )
        np.random.seed(random_seed)
        x = np.random.rand(n + 1, n + 1)
        return x

    def solve(self, problem: NDArray) -> NDArray:
        """
        Compute the N-dimensional DCT Type I using scipy.fftpack.
        """
        result = scipy.fftpack.dctn(problem, type=1)
        return result

    def is_solution(self, problem: NDArray, solution: NDArray) -> bool:
        """
        Check if the DCT Type I solution is valid and optimal.

        A valid solution must match the reference implementation (scipy.fftpack.dctn)
        within a small tolerance.

        :param problem: Input array.
        :param solution: Computed DCT result.
        :return: True if the solution is valid and optimal, False otherwise.
        """
        tol = 1e-6
        reference = scipy.fftpack.dctn(problem, type=1)
        error = np.linalg.norm(solution - reference) / (np.linalg.norm(reference) + 1e-12)
        if error > tol:
            logging.error(f"DCT solution error {error} exceeds tolerance {tol}.")
            return False
        return True
