import itertools
import logging
from typing import Any

import numpy as np
from scipy.spatial import Delaunay as SciPyDelaunay

from AlgoTuneTasks.base import register_task, Task


ABS_TOL = 1e-12


@register_task("delaunay")
class Delaunay(Task):
    def __init__(self, **kwargs):
        """
        Benchmark task: 2‑D Delaunay triangulation using SciPy’s Qhull wrapper.
        Public interface (generate_problem, solve, is_solution) is **unchanged**.
        """
        super().__init__(**kwargs)

    @staticmethod
    def _rng(seed: int) -> np.random.Generator:
        return np.random.default_rng(seed)

    @staticmethod
    def _non_degenerate_points(n: int, rng: np.random.Generator) -> np.ndarray:
        """
        Draw random points until they span 2‑D (area > 0).
        """
        while True:
            pts = rng.random((n, 2))
            # Compute area of convex hull triangle fan; degenerate if area ≈ 0.
            if np.abs(np.cross(pts[1] - pts[0], pts[2] - pts[0])) <= 1e-4:
                continue

            try:
                tri_test = SciPyDelaunay(pts)
                # Return pts only if it has its Delaunay Triangulation.
                if tri_test.simplices is not None:
                    return pts
            except Exception:
                continue

    def generate_problem(self, n: int, random_seed: int = 0) -> dict[str, Any]:
        n_points = max(3, n)
        rng = self._rng(random_seed)
        points = self._non_degenerate_points(n_points, rng)

        problem = {
            "points": points,
        }
        logging.debug("Generated Delaunay problem (n=%d, seed=%d).", n_points, random_seed)
        return problem

    def solve(self, problem: dict[str, Any]) -> dict[str, Any]:
        pts = np.asarray(problem["points"])

        tri = SciPyDelaunay(pts)
        simplices = tri.simplices
        convex_hull = tri.convex_hull
        result = {
            "simplices": self._canonical_simplices(simplices.tolist()),
            "convex_hull": self._canonical_edges(convex_hull.tolist()),
        }
        return result

    def _canonical_simplices(self, simplices: np.ndarray) -> list[tuple[int, ...]]:
        """
        Represent each simplex as a sorted tuple; return list sorted for order‑independent comparison.
        """
        return sorted(map(sorted, simplices))

    def _canonical_edges(self, edges: np.ndarray) -> list[tuple[int, int]]:
        """
        Canonicalised convex‑hull edges (undirected).
        """
        return sorted(map(sorted, edges))

    def is_solution(self, problem: dict[str, Any], solution: dict[str, Any]) -> bool:
        # quick key / shape / type checks
        for k in ("simplices", "convex_hull"):
            if k not in solution:
                logging.error("Key '%s' missing from solution.", k)
                return False

        # generate reference solution
        pts = problem["points"]
        ref = self.solve(problem=problem)

        # canonicalise simplices & hull for order‑independent comparison
        ref_hull = self._canonical_edges(np.asarray(ref["convex_hull"]))
        sol_hull = self._canonical_edges(np.asarray(solution["convex_hull"]))

        if ref_hull != sol_hull:
            logging.error("Convex‑hull edge set mismatch.")
            return False

        # sort out list where two simplices form rectangle: [0, 1, 3], [1, 2, 3] => [0, 1, 2, 3]
        ref_simp = self._canonical_simplices(np.asarray(ref["simplices"]))
        sol_simp = self._canonical_simplices(np.asarray(solution["simplices"]))
        ref_simp_unique = np.setdiff1d(ref_simp, sol_simp)
        sol_simp_unique = np.setdiff1d(sol_simp, ref_simp)

        ref_simp2_joined = [
            np.unique(union)
            for union in itertools.combinations(ref_simp_unique, 2)
            if len(np.unique(union)) == 4
        ]
        sol_simp2_joined = [
            np.unique(union)
            for union in itertools.combinations(sol_simp_unique, 2)
            if len(np.unique(union)) == 4
        ]
        if len(ref_simp2_joined) != len(sol_simp2_joined):
            return False

        common_simp2_joined = np.intersect1d(ref_simp2_joined, sol_simp2_joined)
        if len(ref_simp_unique) != 2 * len(common_simp2_joined):
            return False

        # check whether chosen 4 vertices are on the same circle
        for simp2_joined in common_simp2_joined:
            mat = np.hstack(
                [pts[simp2_joined], np.sum(pts[simp2_joined] ** 2, axis=1), np.ones((4, 1))]
            )
            if np.abs(np.linalg.det(mat)) > ABS_TOL:
                return False

        if ref_simp_unique.shape != sol_simp_unique.shape:
            logging.error("Simplices set mismatch.")
            return False

        return True
