import pytest
import asyncio
import uuid
import logging

from synth_env.examples.red.environment import (
    PokemonRedEnvironment,
    PokemonRedPublicState,
    PokemonRedPrivateState,
)
from synth_env.environment.shared_engine import (
    GetObservationCallable,
    InternalObservation,
)
from synth_env.examples.red.taskset import PokemonRedTaskInstance
from synth_env.tasks.core import Impetus, Intent, TaskInstanceMetadata
from synth_env.environment.tools import EnvToolCall

# Set up logging to see debug messages from the engine
logging.basicConfig(level=logging.DEBUG)


class PressButtonCall(EnvToolCall):
    """Helper class for creating button press calls"""

    def __init__(self, button: str, frames: int = 1):
        super().__init__(tool="press_button", args={"button": button, "frames": frames})


class RetryTestObservationCallable(GetObservationCallable):
    """Simple observation callable for retry testing"""

    def __init__(self):
        self.screen_buffer = None

    async def get_observation(
        self, pub: PokemonRedPublicState, priv: PokemonRedPrivateState
    ) -> InternalObservation:
        if pub is None or priv is None:
            raise RuntimeError("Missing public or private state in get_observation")

        formatted_obs = (
            f"=== RETRY TEST STATE ===\n"
            f"Step: {pub.step_count}\n"
            f"Position: ({pub.player_x}, {pub.player_y})\n"
            f"Map ID: {pub.map_id}\n"
            f"=== END RETRY TEST STATE ==="
        )

        return {
            "public": pub,
            "private": priv,
            "formatted_obs": formatted_obs,
            "screen_buffer": self.screen_buffer,
        }


@pytest.mark.asyncio
async def test_movement_with_retry():
    """
    Test that the new retry mechanism makes movement reliable.
    """
    print("\n" + "=" * 60)
    print("TESTING ENGINE RETRY MECHANISM FOR MOVEMENT")
    print("=" * 60)

    # Create a task instance
    task_metadata = TaskInstanceMetadata()
    inst = PokemonRedTaskInstance(
        id=uuid.uuid4(),
        impetus=Impetus(instructions="Test retry mechanism with left movement."),
        intent=Intent(
            rubric={"goal": "Move left reliably"},
            gold_trajectories=None,
            gold_state_diff={},
        ),
        metadata=task_metadata,
        is_reproducible=True,
        initial_engine_snapshot=None,
    )

    # Create environment with retry test observation callable
    retry_obs = RetryTestObservationCallable()
    env = PokemonRedEnvironment(inst, custom_step_obs=retry_obs)

    try:
        # Initialize environment
        print("\n[DEBUG] Initializing environment...")
        obs_payload = await env.initialize()

        if "error" in obs_payload:
            pytest.fail(f"Environment initialization failed: {obs_payload['error']}")

        print("[DEBUG] Environment initialized successfully")

        # Get initial state
        initial_pub = obs_payload["public"]
        initial_position = (initial_pub.player_x, initial_pub.player_y)
        initial_map_id = initial_pub.map_id

        print(f"[DEBUG] Initial position: {initial_position}")
        print(f"[DEBUG] Initial map ID: {initial_map_id}")

        # Test movement commands that should now work reliably
        movement_tests = [
            ("LEFT", "should move left"),
            ("RIGHT", "should move right"),
            ("UP", "should move up"),
            ("DOWN", "should move down"),
        ]

        successful_movements = 0

        for button, expected_behavior in movement_tests:
            print(f"\n--- Testing {button} button ({expected_behavior}) ---")

            # Get position before movement
            before_pub = obs_payload["public"]
            before_position = (before_pub.player_x, before_pub.player_y)
            before_map = before_pub.map_id

            print(f"Position before {button}: {before_position}")

            # Execute movement command (engine will retry automatically)
            step_result = await env.step([[PressButtonCall(button)]])

            if "error" in step_result:
                print(f"[ERROR] {button} movement failed: {step_result['error']}")
                continue

            # Check position after movement
            after_pub = step_result["public"]
            after_position = (after_pub.player_x, after_pub.player_y)
            after_map = after_pub.map_id

            print(f"Position after {button}: {after_position}")

            # Check if movement occurred
            position_changed = after_position != before_position
            map_changed = after_map != before_map
            movement_occurred = position_changed or map_changed

            if movement_occurred:
                print(
                    f"[SUCCESS] {button} movement worked! Position: {before_position} -> {after_position}"
                )
                if map_changed:
                    print(f"[NOTICE] Map also changed: {before_map} -> {after_map}")
                successful_movements += 1
            else:
                print(
                    f"[WARNING] {button} movement had no effect. Position stayed: {after_position}"
                )

            # Update obs_payload for next test
            obs_payload = step_result

        # Test non-movement buttons (should work without retry)
        print("\n--- Testing non-movement buttons (A, B) ---")

        for button in ["A", "B"]:
            print(f"Testing {button} button...")

            step_result = await env.step([[PressButtonCall(button)]])

            if "error" in step_result:
                print(f"[ERROR] {button} button failed: {step_result['error']}")
            else:
                print(f"[SUCCESS] {button} button executed successfully")

        # Analysis
        print("\n" + "=" * 60)
        print("RETRY MECHANISM TEST RESULTS")
        print("=" * 60)

        print(f"Successful movements: {successful_movements}/{len(movement_tests)}")

        if successful_movements > 0:
            print(
                "[SUCCESS] Engine retry mechanism is working - at least some movements succeeded!"
            )
        else:
            print("[WARNING] No movements succeeded - may need to investigate further")

        # The test passes if we can execute without errors
        assert True, (
            "Retry mechanism test completed - check logs for movement success details"
        )

    except Exception as e:
        print(f"[ERROR] Test failed with exception: {e}")
        raise


if __name__ == "__main__":
    # Run the test directly
    asyncio.run(test_movement_with_retry())
