"""Stripe Connect account management routes."""

from fastapi import APIRouter, HTTPException, Header
from pydantic import BaseModel, EmailStr
from typing import Optional, Dict, Any, Literal
from agenticlypay.connect import ConnectManager
from agenticlypay.stripe_client import StripeClient

router = APIRouter()
connect_manager = ConnectManager()
stripe_client = StripeClient()


class CreateAccountRequest(BaseModel):
    """Request model for creating a developer account."""

    email: EmailStr
    country: str = "US"
    metadata: Optional[Dict[str, str]] = None


class CreateOnboardingLinkRequest(BaseModel):
    """Request model for creating an onboarding link."""

    account_id: str
    refresh_url: str
    return_url: str


class UpdateMetadataRequest(BaseModel):
    """Request model for updating account metadata."""

    account_id: str
    metadata: Dict[str, str]


class TaxInfoRequest(BaseModel):
    """Request model for tax information."""

    account_id: str
    name: str
    ssn: Optional[str] = None
    ein: Optional[str] = None
    address: str


class PayoutScheduleRequest(BaseModel):
    """Request model for configuring payout schedule."""

    account_id: str
    interval: Literal["manual", "daily", "weekly", "monthly"]
    monthly_anchor: Optional[int] = None
    weekly_anchor: Optional[Literal["monday", "tuesday", "wednesday", "thursday", "friday", "saturday", "sunday"]] = None


@router.post("/accounts")
async def create_account(request: CreateAccountRequest):
    """Create a new Stripe Connect account for a developer."""
    try:
        account = connect_manager.create_developer_account(
            email=request.email,
            country=request.country,
            metadata=request.metadata,
        )
        return {"success": True, "account": account}
    except Exception as e:
        raise HTTPException(status_code=400, detail=str(e))


@router.get("/accounts/{account_id}")
async def get_account(account_id: str, email: Optional[str] = None, x_developer_email: Optional[str] = None):
    """Get account status."""
    try:
        # Get email from query param or header
        email = email or x_developer_email
        if not email:
            raise HTTPException(status_code=400, detail="email is required (query param or X-Developer-Email header)")
        status = connect_manager.get_account_status(account_id, email)
        return {"success": True, "account": status}
    except Exception as e:
        raise HTTPException(status_code=404, detail=str(e))


@router.post("/onboarding")
async def create_onboarding_link(request: CreateOnboardingLinkRequest, x_developer_email: Optional[str] = Header(None, alias="X-Developer-Email")):
    """Create an onboarding link for a developer account."""
    try:
        # Get email from header
        email = x_developer_email
        if not email:
            raise HTTPException(status_code=400, detail="email is required in X-Developer-Email header")
        link = connect_manager.create_onboarding_link(
            account_id=request.account_id,
            refresh_url=request.refresh_url,
            return_url=request.return_url,
            email=email,
        )
        return {"success": True, "onboarding_link": link}
    except Exception as e:
        raise HTTPException(status_code=400, detail=str(e))


@router.put("/metadata")
async def update_metadata(request: UpdateMetadataRequest):
    """Update account metadata."""
    try:
        result = connect_manager.update_account_metadata(
            account_id=request.account_id, metadata=request.metadata
        )
        return {"success": True, "account": result}
    except Exception as e:
        raise HTTPException(status_code=400, detail=str(e))


@router.post("/tax-info")
async def collect_tax_info(request: TaxInfoRequest):
    """Collect tax information for a developer."""
    try:
        tax_info = {
            "name": request.name,
            "ssn": request.ssn,
            "ein": request.ein,
            "address": request.address,
        }
        result = connect_manager.collect_tax_information(
            account_id=request.account_id, tax_info=tax_info
        )
        return {"success": True, "result": result}
    except Exception as e:
        raise HTTPException(status_code=400, detail=str(e))


@router.post("/payout-schedule")
async def set_payout_schedule(request: PayoutScheduleRequest, x_developer_email: Optional[str] = Header(None, alias="X-Developer-Email")):
    """Set payout schedule for a connected account."""
    try:
        # Get email from header
        email = x_developer_email
        if not email:
            raise HTTPException(status_code=400, detail="email is required in X-Developer-Email header")
        account = stripe_client.update_payout_schedule(
            account_id=request.account_id,
            interval=request.interval,
            monthly_anchor=request.monthly_anchor,
            weekly_anchor=request.weekly_anchor,
        )
        schedule = account.settings.payouts.schedule if hasattr(account.settings, "payouts") else None
        return {
            "success": True,
            "account_id": account.id,
            "payout_schedule": schedule,
        }
    except Exception as e:
        raise HTTPException(status_code=400, detail=str(e))
